//******************************************************************************
//* File       : TermSize.cpp                                                  *
//* Author     : Mahlon R. Smith                                               *
//*              Copyright (c) 2021-2022 Mahlon R. Smith, The Software Samurai *
//*                  GNU GPL copyright notice located in FileMangler.hpp       *
//* Date       : 20-Oct-2021                                                   *
//* Version    : (see appVersion string)                                       *
//*                                                                            *
//* Description: TermSize is a simple utility which helps to compensate for    *
//* the serious design flaw in the GNOME terminal under Wayland which no longer*
//* reports the terminal-window size as it is being resized. In addition, there*
//* is a serious bug in the GNOME terminal program which calculates the initial*
//* size of the window using a hard-coded 12-point font as the reference.      *
//* If your terminal is set for a font other than 12-point, you may need to    *
//* play some tricks to get the dimensions you want.                           *
//* In addition, the gnome-terminal invocation accepts only a limited range    *
//* for row/column dimensions juxtaposed with the dimensions of the desktop.   *
//* Again, this seems to be a calculation based on a hard-coded 12-point font. *
//* (Wayland may someday be a reliable platform, but today is not that day.)   *
//*                                                                            *
//* Example ".desktop" file:                                                   *
//* ------------------------                                                   *
//* [Desktop Entry]                                                            *
//* Comment=custom geometry for terminal (Note: Wayland ignores positioning.)  *
//* Terminal=false                                                             *
//* Name=GnomeTerm_Custom                                                      *
//* Exec=gnome-terminal --profile='Sam' --geometry=141x37                      *
//*                               --working-directory=/home/sam/SoftwareDesign *
//*                               -- /bin/sh -c 'termsize ; clear ; exec bash' *
//* Type=Application                                                           *
//* Icon=/home/sam/Apps/GnomeTerm/SamLogo03.png                                *
//*                                                                            *
//*                ---  ---  ---  ---  ---  ---  ---  ---                      *
//*                                                                            *
//* This application is built on the NcDialog API, but for simplicity, only    *
//* the low-level NCurses class methods are used. Thread safety, sophisticated *
//* mouse support and dialog controls are implemented at a higher level of     *
//* NcDialog abstraction, and are not used by this application.                *
//*                                                                            *
//* Help and version/copyright info are reported using standard output: wcout. *
//*                                                                            *
//* Various debugging message are under the TS_DEBUG and DEBUG_CSCHEME         *
//* conditional compile flags. (see below)                                     *
//*                                                                            *
//*                                                                            *
//* Development Tools:                  Current:                               *
//* ----------------------------------  -------------------------------------- *
//*  GNU G++ v:4.8.0 or higher (C++11)  G++ v:11.2.1                           *
//*  Fedora Linux v:16 or higher        Fedora v:34                            *
//*  GNOME Terminal v:3.2.1 or higher   GNOME Term v:3.40.3                    *
//******************************************************************************
//* Copyright Notice:                                                          *
//* This program is free software: you can redistribute it and/or modify it    *
//* under the terms of the GNU General Public License as published by the Free *
//* Software Foundation, either version 3 of the License, or (at your option)  *
//* any later version.                                                         *
//*                                                                            *
//* This program is distributed in the hope that it will be useful, but        *
//* WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY *
//* or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License   *
//* for more details.                                                          *
//*                                                                            *
//* You should have received a copy of the GNU General Public License along    *
//* with this program.  If not, see <http://www.gnu.org/licenses/>.            *
//*                                                                            *
//******************************************************************************
//* Version History (most recent first):                                       *
//*                                                                            *
//* v: 0.00.01 12-Oct-2021                                                     *
//*    -- First effort. Adapt NCurses startup/shutdown sequence from the       *
//*       FileMangler application.                                             *
//*    -- This application is extremely simple _except_ for parsing the user's *
//*       selection of color attributes. Any of the first 64 NcDialog API      *
//*       foreground/background color combinations may be selected. While this *
//*       is clearly over-the-top design, it was an enjoyable way to spend a   *
//*       rainy afternoon. :-)                                                 *
//*       -- Note on color: a "brown" foreground will be rendered as yellow    *
//*          (bold brown) for most terminal color palettes. This provides      *
//*          higher contrast.                                                  *
//*                                                                            *
//******************************************************************************

//****************
//* Header Files *
//****************
#include "GlobalDef.hpp"   //* NcDialog family of classes

//***************
//* Definitions *
//***************
#define TS_DEBUG (0)       //* For debugging only, enable debug messages.
#define DEBUG_CSCHEME (0)  //* For debugging only, debug color-scheme
#if DEBUG_CSCHEME != 0
gString gsdebug ;
winPos wpdebug( 4, 0 ) ;
#endif   // DEBUG_CSCHEME


//* Return value for GetCommandLineArgs() *
enum clArgs : short { clHelp, clVersion, clExecute } ;

//* Codes for parsing the user's selection of color scheme. *
//* See gclaColorScheme() method for details.               *
enum textAttr : short
{
   taTERM = ZERO, // terminal default (usually black-on-white)
   taTERMr,       // terminal default with reversed foreground/background

   taBLACK,    // black text   (dflt bkgnd)
   taRED,      // red text     (dflt bkgnd)
   taGREEN,    // green text   (dflt bkgnd)
   taBROWN,    // brown text   (dflt bkgnd)
   taBLUE,     // blue text    (dflt bkgnd)
   taMAGENTA,  // magenta text (dflt bkgnd)
   taCYAN,     // cyan text    (dflt bkgnd)
   taGREY,     // grey text    (dflt bkgnd)

   taBLACKr,   // term bkgnd text on black bkgnd
   taREDr,     // term bkgnd text on red bkgnd
   taGREENr,   // term bkgnd text on green bkgnd
   taBROWNr,   // term bkgnd text on brown bkgnd
   taBLUEr,    // term bkgnd text on blue bkgnd
   taMAGENTAr, // term bkgnd text on magenta bkgnd
   taCYANr,    // term bkgnd text on cyan bkgnd
   taGREYr,    // term bkgnd text on grey bkgnd

   taREBK,     // red-on-black
   taGRBK,     // green-on-black
   taBRBK,     // brown-on-black
   taBLBK,     // blue-on-black
   taMABK,     // magenta-on-black
   taCYBK,     // cyan-on-black
   taGYBK,     // grey-on-black

   taBKRE,     // black-on-red
   taGRRE,     // green-on-red
   taBRRE,     // brown-on-red
   taBLRE,     // blue-on-red
   taMARE,     // magenta-on-red
   taCYRE,     // cyan-on-red
   taGYRE,     // grey-on-red

   taBKGR,     // black-on-green
   taREGR,     // red-on-green
   taBRGR,     // brown-on-green
   taBLGR,     // blue-on-green
   taMAGR,     // magenta-on-green
   taCYGR,     // cyan-on-green
   taGYGR,     // grey-on-green

   taBKBR,     // black-on-brown
   taREBR,     // red-on-brown
   taGRBR,     // green-on-brown
   taBLBR,     // blue-on-brown
   taMABR,     // magenta-on-brown
   taCYBR,     // cyan-on-brown
   taGYBR,     // grey-on-brown

   taBKBL,     // black-on-blue
   taREBL,     // red-on-blue
   taGRBL,     // green-on-blue
   taBRBL,     // brown-on-blue
   taMABL,     // magenta-on-blue
   taCYBL,     // cyan-on-blue
   taGYBL,     // grey-on-blue

   taBKMA,     // black-on-magenta
   taREMA,     // red-on-magenta
   taGRMA,     // green-on-magenta
   taBRMA,     // brown-on-magenta
   taBLMA,     // blue-on-magenta
   taCYMA,     // cyan-on-magenta
   taGYMA,     // grey-on-magenta

   taBKCY,     // black-on-cyan
   taRECY,     // red-on-cyan
   taGRCY,     // green-on-cyan
   taBRCY,     // brown-on-cyan
   taBLCY,     // blue-on-cyan
   taMACY,     // magenta-on-cyan
   taGYCY,     // grey-on-cyan

   taBKGY,     // black-on-grey
   taREGY,     // red-on-grey
   taGRGY,     // green-on-grey
   taBRGY,     // brown-on-grey
   taBLGY,     // blue-on-grey
   taMAGY,     // magenta-on-grey
   taCYGY,     // cyan-on-grey

   taITEMS     // number of attribute codes
} ;

//**************
//* Local data *
//**************
static const char* const appVersion = "0.0.01" ;
static const char* const crYears    = "2021-2022" ;
static const char* const appName1   = "TermSize" ;
static const char* const appName2   = "termsize" ;
static const char* const titleTemplate =
      "%s (%s) v:%s - %s\n"
      "========================================" ;

static const textAttr dfltAttr = taGRBK ;    // default text color attribute

//********************
//* Local prototypes *
//********************
static attr_t SetColorScheme ( textAttr tattr ) ;
static clArgs GetCommandLineArgs ( int argc, const char* argv[], const char* argenv[], 
                                   textAttr& txtAttr, bool& centerMsg ) ;
static textAttr gclaColorScheme ( const gString& gsColor ) ;
static void DisplayHelp ( void ) ;
static void DisplayVersion ( void ) ;


//*************************
//*        main           *
//*************************
//******************************************************************************
//* Program entry point.                                                       *
//*                                                                            *
//******************************************************************************

int main ( int argc, char* argv[], char* argenv[] )
{
   gString gsOut, gsHdr ;        // text formatting
   short    exitCode = ZERO,     // application exit code (0 == no error)
            termRows = ZERO,     // dimensions of terminal window
            termCols = ZERO,
            msgWidth ;           // width of report message (columns)
   textAttr attrCode = dfltAttr ;// enum code indicating user's color-scheme selection
   attr_t   txtColor = ZERO ;    // text foreground/background color (ZERO==terminal default)
   wkeyCode wk ;                 // user key/mouse input
   winPos txtPos( ZERO, ZERO ) ; // text display position
   mmask_t  mEvents  = BUTTON1_RELEASED ; // mouse-enable mask
   bool     cntrMsg  = false,    // 'true' if output is to be centered in terminal window
            done     = false ;   // loop control

   //* Get user's command-line argument *
   clArgs userOption = GetCommandLineArgs ( argc, (const char**)argv, 
                                            (const char**)argenv, attrCode, cntrMsg ) ;

   //* If user indicates to execute the program,         *
   //* initialize the ncurses engine and complete setup. *
   if ( (userOption == clExecute) && (nc.StartNCursesEngine ()) == OK )
   {
      #if TS_DEBUG != 0
      bool  colorMode  = false,    // 'true' if color engine started
            utf8Locale = false,    // 'true' if locale supports UTF-8 data
            mEnabled   = false ;   // 'true' if mouse support enabled
      #endif   // TS_DEBUG

      //* Determine whether the hardware supports display of   *
      //* color text, and if available, start the color engine.*
      if ( (nc.ColorText_Available ()) &&
           (nc.ColorText_Initialized ()) )
      {
         //* Convert user's color-scheme specification to a color attribute *
         txtColor = SetColorScheme ( attrCode ) ;

         #if TS_DEBUG != 0
         colorMode = true ;
         #endif   // TS_DEBUG
      }

      //* Enable the NCurses mouse *
      if ( (nc.EnableMouseEvents ( mEvents )) == OK )
      {
         #if TS_DEBUG != 0
         mEnabled = true ;
         #endif   // TS_DEBUG
      }

      //* Set the locale from the environment and *
      //* verify that locale supports UTF-8 data. *
      gString envLocale( nc.GetLocale() ) ;     // name of locale from environment
      #if TS_DEBUG != 0
      utf8Locale = bool((nc.VerifyLocale ()) == OK) ;
      #endif   // TS_DEBUG

      nc.SetCursorState ( nccvINVISIBLE ) ;  // hide the cursor
      nc.SetKeyProcState ( nckpRAW ) ;       // allow CTRL keys through unprocessed

      system ( "clear" ) ;                   // clear the terminal window

      #if DEBUG_CSCHEME != 0
      //* Report parsing of user's color-scheme selection *
      if ( gsdebug.gschars() > 1 )
         nc.WriteString ( wpdebug, gsdebug.gstr(), nc.bw ) ;
      #endif   // DEBUG_CSCHEME

      //* Get the dimensions of the terminal window *
      nc.ScreenDimensions ( termRows, termCols ) ;

      //* Write the header line and report initial terminal window size.*
      gsHdr.compose( "  %s  ", appName2 ) ;
      msgWidth = gsHdr.gscols() ;      // save width of output
      if ( cntrMsg )    // calculate output position
      {
         txtPos.ypos = (termRows / 2) - 1 ;
         txtPos.xpos = (termCols / 2) - (msgWidth / 2) ;
      }
      nc.WriteString ( txtPos, gsHdr.gstr(), txtColor ) ;
      const char* const dimTemplate = "  %2hd x %-3hd  " ;
      gsOut.compose( dimTemplate, &termRows, &termCols ) ;
      nc.WriteString ( txtPos.ypos + 1, txtPos.xpos, gsOut.gstr(), txtColor ) ;

      #if TS_DEBUG != 0
      //* Warn user if locale does not support UTF-8 character set,*
      //* OR if color engine not started OR if mouse not enabled.  *
      if ( ! utf8Locale )
         nc.WriteString ( ++txtPos.ypos, txtPos.xpos,
                          L"Locale may not support UTF-8 character set.", nc.bw ) ;
      if ( ! colorMode )
         nc.WriteString ( ++txtPos.ypos, txtPos.xpos,
                          L"Color text not available.", nc.bw ) ;
      if ( ! mEnabled )
         nc.WriteString ( ++txtPos.ypos, txtPos.xpos,
                          L"NCurses mouse support not available.", nc.bw ) ;

      txtPos.ypos = 1 ;
      #endif   // TS_DEBUG

      //* Loop until user gets tired *
      while ( ! done )
      {
         //* Get user's response. If it is a window-resize signal, *
         //* update the display, else exit the application.        *
         nc.GetKeyInput ( wk ) ;       // get user's response
         if ( (wk.type == wktFUNKEY) && (wk.key == nckRESIZE) )
         {
            nc.ScreenDimensions ( termRows, termCols ) ;
            if ( cntrMsg )    // calculate output position
            {
               txtPos.ypos = (termRows / 2) - 1 ;
               txtPos.xpos = (termCols / 2) - (msgWidth / 2) ;
               nc.ClearScreen () ;
            }
            nc.WriteString ( txtPos, gsHdr.gstr(), txtColor ) ;
            gsOut.compose( dimTemplate, &termRows, &termCols ) ;
            nc.WriteString ( txtPos.ypos + 1, txtPos.xpos, gsOut.gstr(), txtColor ) ;
         }
         else
            done = true ;
      }

      //* Shut down the ncurses engine *
      nc.DisableMouseEvents () ;             // disable mouse interface
      nc.RestoreCursorState () ;             // make cursor visible
      nc.StopNCursesEngine () ;              // Deactivate the NCurses engine
   }

   //* Ncurses engine not started. Test for Help or Version request. *
   else
   {
      if ( userOption == clVersion )
         DisplayVersion () ;
      else if ( userOption == clHelp )
         DisplayHelp () ;
      else
         wcout << L"Error! Unable to start the ncurses engine.\n" ;
   }

   wcout << endl ;
   exit ( exitCode ) ;

}  //* End main() *

//*************************
//*    SetColorScheme     *
//*************************
//******************************************************************************
//* Convert the specified member of enum textAttr to an NcDialog API color     *
//* attribute.                                                                 *
//*                                                                            *
//* Input  : attrCode : user's color-scheme selection                          *
//*                                                                            *
//* Returns: text/background color attribute                                   *
//******************************************************************************

static attr_t SetColorScheme ( textAttr attrCode )
{
   attr_t txtAttr ;           // return value

   switch ( attrCode )
   {
      //* Single-color and reversed single-color attributes.*
      //* These are the basic eight(8) color attributes.    *
      case taTERM:      txtAttr = nc.bw ;       break ;
      case taTERMr:     txtAttr = nc.bwR ;      break ;
      case taBLACK:     txtAttr = nc.bk ;       break ;
      case taBLACKr:    txtAttr = nc.bkR ;      break ;
      case taRED:       txtAttr = nc.re ;       break ;
      case taREDr:      txtAttr = nc.reR ;      break ;
      case taGREEN:     txtAttr = nc.gr ;       break ;
      case taGREENr:    txtAttr = nc.grR ;      break ;
      case taBROWN:     txtAttr = nc.br ;       break ;
      case taBROWNr:    txtAttr = nc.brR ;      break ;
      case taBLUE:      txtAttr = nc.bl ;       break ;
      case taBLUEr:     txtAttr = nc.blR ;      break ;
      case taMAGENTA:   txtAttr = nc.ma ;       break ;
      case taMAGENTAr:  txtAttr = nc.maR ;      break ;
      case taCYAN:      txtAttr = nc.cy ;       break ;
      case taCYANr:     txtAttr = nc.cyR ;      break ;
      case taGREY:      txtAttr = nc.gy ;       break ;
      case taGREYr:     txtAttr = nc.gyR ;      break ;

      //* Black background *
      case taREBK:      txtAttr = nc.rebk ;     break ;
      case taGRBK:      txtAttr = nc.grbk ;     break ;
      case taBRBK:      txtAttr = nc.brbk ;     break ;
      case taBLBK:      txtAttr = nc.blbk ;     break ;
      case taMABK:      txtAttr = nc.mabk ;     break ;
      case taCYBK:      txtAttr = nc.cybk ;     break ;
      case taGYBK:      txtAttr = nc.gybk ;     break ;

      //* Red background *
      case taBKRE:      txtAttr = nc.bkre ;     break ;
      case taGRRE:      txtAttr = nc.grre ;     break ;
      case taBRRE:      txtAttr = nc.brre ;     break ;
      case taBLRE:      txtAttr = nc.blre ;     break ;
      case taMARE:      txtAttr = nc.mare ;     break ;
      case taCYRE:      txtAttr = nc.cyre ;     break ;
      case taGYRE:      txtAttr = nc.gyre ;     break ;

      //* Green background *
      case taBKGR:      txtAttr = nc.bkgr ;     break ;
      case taREGR:      txtAttr = nc.regr ;     break ;
      case taBRGR:      txtAttr = nc.brgr ;     break ;
      case taBLGR:      txtAttr = nc.blgr ;     break ;
      case taMAGR:      txtAttr = nc.magr ;     break ;
      case taCYGR:      txtAttr = nc.cygr ;     break ;
      case taGYGR:      txtAttr = nc.gygr ;     break ;

      //* Brown background *
      case taBKBR:      txtAttr = nc.bkbr ;     break ;
      case taREBR:      txtAttr = nc.rebr ;     break ;
      case taGRBR:      txtAttr = nc.grbr ;     break ;
      case taBLBR:      txtAttr = nc.blbr ;     break ;
      case taMABR:      txtAttr = nc.mabr ;     break ;
      case taCYBR:      txtAttr = nc.cybr ;     break ;
      case taGYBR:      txtAttr = nc.gybr ;     break ;

      //* Blue background *
      case taBKBL:      txtAttr = nc.bkbl ;     break ;
      case taREBL:      txtAttr = nc.rebl ;     break ;
      case taGRBL:      txtAttr = nc.grbl ;     break ;
      case taBRBL:      txtAttr = nc.brbl ;     break ;
      case taMABL:      txtAttr = nc.mabl ;     break ;
      case taCYBL:      txtAttr = nc.cybl ;     break ;
      case taGYBL:      txtAttr = nc.gybl ;     break ;

      //* Magenta background *
      case taBKMA:      txtAttr = nc.bkma ;     break ;
      case taREMA:      txtAttr = nc.rema ;     break ;
      case taGRMA:      txtAttr = nc.grma ;     break ;
      case taBRMA:      txtAttr = nc.brma ;     break ;
      case taBLMA:      txtAttr = nc.blma ;     break ;
      case taCYMA:      txtAttr = nc.cyma ;     break ;
      case taGYMA:      txtAttr = nc.gyma ;     break ;

      //* Cyan background *
      case taBKCY:      txtAttr = nc.bkcy ;     break ;
      case taRECY:      txtAttr = nc.recy ;     break ;
      case taGRCY:      txtAttr = nc.grcy ;     break ;
      case taBRCY:      txtAttr = nc.brcy ;     break ;
      case taBLCY:      txtAttr = nc.blcy ;     break ;
      case taMACY:      txtAttr = nc.macy ;     break ;
      case taGYCY:      txtAttr = nc.gycy ;     break ;

      //* Grey background *
      case taBKGY:      txtAttr = nc.bkgy ;     break ;
      case taREGY:      txtAttr = nc.regy ;     break ;
      case taGRGY:      txtAttr = nc.grgy ;     break ;
      case taBRGY:      txtAttr = nc.brgy ;     break ;
      case taBLGY:      txtAttr = nc.blgy ;     break ;
      case taMAGY:      txtAttr = nc.magy ;     break ;
      case taCYGY:      txtAttr = nc.cygy ;     break ;

      default:          txtAttr = dfltAttr ;    break ; // (unlikely)
   } ;

   return txtAttr ;

}  //* End SetColorScheme() *

//*************************
//*    DisplayVersion     *
//*************************
//******************************************************************************
//* Display the application version number and copyright information.          *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

static void DisplayVersion ( void )
{
   static const char* const freeSoftware = 
   "=============================\n"
   "License GPLv3+: GNU GPL version 3 <http://gnu.org/licenses/gpl.html>\n"
   "This is free software: you are free to modify and/or redistribute it\n"
   "under the terms set out in the license.\n"
   "There is NO WARRANTY, to the extent permitted by law." ;

   gString gs( titleTemplate, appName1, appName2, appVersion, crYears ) ;
   wcout << L'\n' << gs.gstr() << freeSoftware << endl ;

}  //* End DisplayVersion() *

static void DisplayHelp ( void )
{
   const char* Help = 
   "\nUsage: termsize [OPTIONS]  (all arguments are optional)"
   "\n"
   "\nInteractively report the size of the terminal window (rows and columns)."
   "\nUse the mouse to drag the window borders to the desired size,"
   "\nthen press any key (or click inside the window) to exit."
   "\n"
   "\n -c   Center the report in the terminal window."
   "\n      (default is to place report in upper left corner of window)"
   "\n --color=FOREGROUND[,BACKGROUND]"
   "\n     Select foreground and background color attributes for display."
   "\n     Foreground and background may each be one of the following:"
   "\n      black, red, green, brown, blue, magenta, cyan, grey"
   "\n      (or \"reverse\" to indicate reversed foreground/background)."
   "\n     For foreground only, \"term\" specifies use of default terminal colors"
   "\n     Examples: --color=brown         (brown text on terminal background)"
   "\n               --color=cyan,magenta  (cyan text on magenta)"
   "\n               --color=blue,reverse  (terminal background on blue)"
   "\n               --color=term          (terminal foreground/background colors)"
   "\n               --color=reverse       (terminal colors with fg/bg reversed)"
   "\n     Default Color: green foreground on black background."
   "\n --help (-h)    Display application help, then exit."
   "\n --version (-v) Display application version and copyright information." ;

   system ( "clear" ) ;    // clear the terminal window

   gString gs( titleTemplate, appName1, appName2, appVersion, crYears ) ;
   wcout << gs.gstr() << Help << endl ;

}  //* End DisplayHelp() *

//*************************
//*  GetCommandLineArgs   *
//*************************
//******************************************************************************
//* Capture user's command-line arguments.                                     *
//* Valid Arguments: see DisplayHelp() method.                                 *
//*                                                                            *
//*                                                                            *
//* Input  : argCount : number of elements in 'argList' array                  *
//*          argList  : array of command-line arguments                        *
//*          argEnv   : array of environment variables                         *
//*          txtAttr  : (by reference) receives a member of enum textAttr      *
//*          centerMsg: (by reference) set if '-c' option specified            *
//*                                                                            *
//* Returns: member of enum clArgs                                             *
//******************************************************************************
//* Programmer's Note: Even though it is not documented, we accept both        *
//* lowercase and uppercase option and sub-option characters where possible.   *
//* This allows us to add new, case-sensitive options at a later date.         *
//******************************************************************************

static clArgs GetCommandLineArgs ( int argCount, const char* argList[], 
                                   const char* argEnv[], textAttr& txtAttr, 
                                   bool& centerMsg )
{
   clArgs userOption = clExecute ;  // return value

   //* If user provided command-line arguments, parse them. *
   //* Note that first argument is application name.        *
   if ( argCount > 1 )
   {
      gString gs ;               // text formatting
      //const char* argPtr ;       // pointer to option argument
      short j = ZERO ;           // for multiple arguments in same token
      char  argLetter ;          // short-form arguments
      bool  multiarg = false ;   // 'true' if concatenated arguments
      txtAttr = dfltAttr ;       // initialize caller's parameters
      centerMsg = false ;

      for ( short i = 1 ; (i < argCount) || (multiarg != false) ; i++ )
      {
         if ( multiarg != false )   // if additional switches in same argument
            --i ;
         else
            j = ZERO ;

         //* If a command-line switch OR continuing previous switch argument *
         if ( argList[i][j] == DASH || multiarg != false )
         {
            multiarg = false ;
            ++j ;

            if ( argList[i][j] == DASH ) // (double dash)
            {  //* Long-form command switches *
               gs = argList[i] ;

               //* Color scheme *
               if ( (gs.compare( L"--color", true, 7 )) == ZERO )
               {
                  txtAttr = gclaColorScheme ( gs ) ;
               }

               //* Request for application version number *
               else if ( (gs.compare( L"--version" )) == ZERO )
                  userOption = clVersion ;

               //* Long-form request for Help *
               else if ( (gs.compare( L"--help" )) == ZERO )
                  userOption = clHelp ;

               else  // invalid argument
               { userOption = clHelp ; break ; }

               continue ;     // finished with this argument
            }  // (double dash)

            //** Short-form Arguments **
            argLetter = argList[i][j] ;

            //* Request for application version *
            if ( (argLetter == 'c') || (argLetter == 'C') )
            {
               centerMsg = true ;
            }

            //* Request for application version *
            else if ( (argLetter == 'v') || (argLetter == 'V') )
            {
               userOption = clVersion ;
            }

            //* Else, is either 'h', 'H' or an invalid argument.  *
            //* Either way, invoke help, and if invalid argument, *
            //* discard any remaining arguments.                  *
            else
            {
               userOption = clHelp ;
               if ( (argLetter != 'h') && (argLetter != 'H') )
                  break ;
            }

            //* If separate tokens have been concatenated *
            if ( argList[i][j + 1] != nckNULLCHAR )
               multiarg = true ;
         }

         //* Token does not begin with a DASH character.      *
         //* This is an error, but give the poor user a break.*
         //* Accept: 'h', 'H', 'v', 'V' as valid arguments.   *
         else
         {
            argLetter = argList[i][j] ;

            if ( (argLetter == 'v') || (argLetter == 'V') )
               userOption = clVersion ;
            else
            {
               userOption = clHelp ;
               if ( (argLetter != 'h') && (argLetter != 'H') )
                  break ;
            }
         }
      }     // for(;;)
   }
   return userOption ;

}  //* End GetCommandLineArgs() *

//*************************
//*    gclaColorScheme     *
//*************************
//******************************************************************************
//* Decode user's color-scheme selection.                                      *
//*                                                                            *
//* User's color scheme takes the form:                                        *
//*    --color=FG/BG                                                           *
//* where FG == foreground color and BG== background color.                    *
//*                                                                            *
//* FG is required, and must be one of:                                        *
//* "black", "red", "green", "brown", "blue", "magenta", "cyan", "grey"(gray), *
//* or "term" or "reverse"                                                     *
//*                                                                            *
//* BG is optional, but if present must be separated from FG by a comma ( ',' )*
//* delimiter character. BG must be one of:                                    *
//* "black", "red", "green", "brown", "blue", "magenta", "cyan", "grey"(gray), *
//* or "reverse". ("term" is invalid for BG)                                   *
//* BG must be different from FG. If FG == BG, then BG will be ignored.        *
//*                                                                            *
//* Parsing the arguments:                                                     *
//* ----------------------                                                     *
//* The argument parsing examines only the first four(4) characters of the     *
//* argument, so arguments need not be fully spelled out. Actually only the    *
//* 1st, 2nd and 4th characters are referenced.                                *
//*                                                                            *
//* Special tests:                                                             *
//* 1) If foreground is specified as "reverse", ignore the background argument.*
//* 2) If foreground is specified as "term", then all background arguments     *
//*    _except_ "reverse" are ignored.                                         *
//*                                                                            *
//*                       ------------------                                   *
//*                                                                            *
//* Input  : gsColor: (by reference) text of command-line switch which         *
//*                   specifies color scheme                                   *
//*                                                                            *
//* Returns: member of enum textAttr                                           *
//******************************************************************************
//* Programmer's Note: The debugging code for this method may go away in a     *
//* subsequent release. It is functional, but it tends to clutter up the code  *
//* unnecessarily.                                                             *
//******************************************************************************

static textAttr gclaColorScheme ( const gString& gsColor )
{
   #if DEBUG_CSCHEME != 0
   const char* const dcsName[taITEMS + 1] =
   {
    "taTERM",
    "taTERMr",
    "taBLACK",
    "taRED",
    "taGREEN",
    "taBROWN",
    "taBLUE",
    "taMAGENTA",
    "taCYAN",
    "taGREY",
    "taBLACKr",
    "taREDr",
    "taGREENr",
    "taBROWNr",
    "taBLUEr",
    "taMAGENTAr",
    "taCYANr",
    "taGREYr",
    "taREBK", "taGRBK", "taBRBK", "taBLBK", "taMABK", "taCYBK", "taGYBK",
    "taBKRE", "taGRRE", "taBRRE", "taBLRE", "taMARE", "taCYRE", "taGYRE",
    "taBKGR", "taREGR", "taBRGR", "taBLGR", "taMAGR", "taCYGR", "taGYGR",
    "taBKBR", "taREBR", "taGRBR", "taBLBR", "taMABR", "taCYBR", "taGYBR",
    "taBKBL", "taREBL", "taGRBL", "taBRBL", "taMABL", "taCYBL", "taGYBL",
    "taBKMA", "taREMA", "taGRMA", "taBRMA", "taBLMA", "taCYMA", "taGYMA",
    "taBKCY", "taRECY", "taGRCY", "taBRCY", "taBLCY", "taMACY", "taGYCY",
    "taBKGY", "taREGY", "taGRGY", "taBRGY", "taBLGY", "taMAGY", "taCYGY",
    "taITEMS"
   } ;
   #endif   // DEBUG_CSCHEME

   textAttr color = taBRBL,         // return value
            cfgnd = taITEMS,        // temp color codes
            cbgnd = taITEMS ;
   short    indx ;                  // index into source string

   //* Find the '=' delimiter *
   if ( (indx = gsColor.find( L'=' )) > ZERO )
   {
      gString gs( &gsColor.gstr()[++indx] ) ;  // extract switch argument(s)
      wchar_t wchar1 = tolower( gs.gstr()[0] ),
              wchar2 = tolower( gs.gstr()[1] ),
              wchar4 = tolower( gs.gstr()[3] ) ;

      #if DEBUG_CSCHEME != 0
      if ( (wchar4 == L'\0') || (wchar4 == L',') )  wchar4 = L'_' ;
      gsdebug.compose( "gclaCS: \"%S\" %C%C_%C ", 
                       gs.gstr(), &wchar1, &wchar2, &wchar4 ) ;
      //wcout << gsdebug.gstr() ;
      #endif   // DEBUG_CSCHEME

      //* Parse the foreground argument *
      if ( (wchar1 == L'b') && (wchar2 == L'l') )
      {
         if ( wchar4 == L'e' ) cfgnd = taBLUE ; // blue
         else cfgnd = taBLACK ;                 // black
      }
      else if ( (wchar1 == L'r') && (wchar2 == L'e') )
      {
         if ( wchar4 == L'e' ) cfgnd = taTERMr ;   // 'reverse'
         else cfgnd = taRED ;                      // 'red'
      }
      else if ( (wchar1 == L'g') && (wchar2 == L'r') )
      {
         if ( wchar4 == L'y' )   cfgnd = taGREY ;  // grey
         else                    cfgnd = taGREEN ; // green
      }
      else if ( (wchar1 == L'b') && (wchar2 == L'r') )
         cfgnd = taBROWN ;
      else if ( (wchar1 == L'm') && (wchar2 == L'a') )
         cfgnd = taMAGENTA ;
      else if ( (wchar1 == L'c') && (wchar2 == L'y') )
         cfgnd = taCYAN ;
      else if ( (wchar1 == L't') && (wchar2 == L'e') )
         cfgnd = taTERM ;
      else  // invalid argument, assume terminal default (reversed)
         cfgnd = taTERMr ;

      #if DEBUG_CSCHEME != 0
      gsdebug.append( " fg:%s ", dcsName[cfgnd] ) ;
      //wcout << gsdebug.gstr() ;
      #endif   // DEBUG_CSCHEME

      //* Parse the background argument (if provided) *
      if ( (indx = gs.after( L',' )) > ZERO )
      {
         gs.shiftChars( -(indx) ) ;
         wchar1 = tolower( gs.gstr()[0] ),
         wchar2 = tolower( gs.gstr()[1] ),
         wchar4 = tolower( gs.gstr()[3] ) ;

         #if DEBUG_CSCHEME != 0
         if ( wchar4 == L'\0' )  wchar4 = L'_' ;
         gsdebug.append( " %C%C_%C ", &wchar1, &wchar2, &wchar4 ) ;
         //wcout << gsdebug.gstr() ;
         #endif   // DEBUG_CSCHEME

         if ( (wchar1 == L'b') && (wchar2 == L'l') )
         {
            if ( wchar4 == L'e' ) cbgnd = taBLUE ;    // blue
            else cbgnd = taBLACK ;                    // black
         }
         else if ( (wchar1 == L'r') && (wchar2 == L'e') )
         {
            if ( wchar4 == L'e' )                     // 'reverse'
               cbgnd = taTERMr ;
            else cbgnd = taRED ;                      // 'red'
         }
         else if ( (wchar1 == L'g') && (wchar2 == L'r') )
         {
            if ( wchar4 == L'y' )   cbgnd = taGREY ;  // grey
            else                    cbgnd = taGREEN ; // green
         }
         else if ( (wchar1 == L'b') && (wchar2 == L'r') )
            cbgnd = taBROWN ;
         else if ( (wchar1 == L'm') && (wchar2 == L'a') )
            cbgnd = taMAGENTA ;
         else if ( (wchar1 == L'c') && (wchar2 == L'y') )
            cbgnd = taCYAN ;
         else  // if taTERM or invalid argument, ignore it
            cbgnd = taITEMS ;
      }

      #if DEBUG_CSCHEME != 0
      gsdebug.append( " bg:%s ", dcsName[cbgnd] ) ;
      //wcout << gsdebug.gstr() ;
      #endif   // DEBUG_CSCHEME

      //* Convert user input to member of enum textAttr.      *
      //* -- If background color not specified, (or invalid)  *
      //*    use only specified foreground.                   *
      //* -- If both foreground and background specified,     *
      //*    combine them.                                    *
      if ( cbgnd == taITEMS )
         color = cfgnd ;
      else
      {
         //* If 'cfgnd' indicates terminal default OR *
         //* reversed terminal foreground/background. *
         if ( (cfgnd == taTERM) || (cfgnd == taTERMr) )
         {
            if ( (cfgnd == taTERMr) || ((cfgnd == taTERM) && (cbgnd == taTERMr)) )
               color = taTERMr ;
            else
               color = taTERM ;
         }

         //* If 'cbgnd' indicates reversed foreground/background, *
         //* then color becomes the reverse of 'cfgnd'.           *
         else if ( cbgnd == taTERMr )
         {
            switch ( cfgnd )
            {
               case taRED:     color = taREDr ;    break ;
               case taGREEN:   color = taGREENr ;  break ;
               case taBROWN:   color = taBROWNr ;  break ;
               case taBLUE:    color = taBLUEr ;   break ;
               case taMAGENTA: color = taMAGENTAr; break ;
               case taCYAN:    color = taCYANr ;   break ;
               case taGREY:    color = taGREYr ;   break ;
               case taBLACK:
               default:        // (unlikely)
                  color = dfltAttr ;
                  break ;
            } ;
         }

         //* If foreground == background, there is *
         //* no such color, so ignore background   *
         else if ( cfgnd == cbgnd )
         {
            color = cfgnd ;
         }

         else if ( cfgnd == taBLACK )  // for black foreground
         {
            switch ( cbgnd )           // specified background
            {
               case taRED:     color = taBKRE ;    break ;
               case taGREEN:   color = taBKGR ;    break ;
               case taBROWN:   color = taBKBR ;    break ;
               case taBLUE:    color = taBKBL ;    break ;
               case taMAGENTA: color = taBKMA ;    break ;
               case taCYAN:    color = taBKCY ;    break ;
               case taGREY:    color = taBKGY ;    break ;
               default:        // silence the compiler warning
                  ;
            } ;
         }
         else if ( cfgnd == taRED )    // for red foreground
         {
            switch ( cbgnd )           // specified background
            {
               case taBLACK:   color = taREBK ;    break ;
               case taGREEN:   color = taREGR ;    break ;
               case taBROWN:   color = taREBR ;    break ;
               case taBLUE:    color = taREBL ;    break ;
               case taMAGENTA: color = taREMA ;    break ;
               case taCYAN:    color = taRECY ;    break ;
               case taGREY:    color = taREGY ;    break ;
               default:        // silence the compiler warning
                  ;
            } ;
         }
         else if ( cfgnd == taGREEN )  // for green foreground
         {
            switch ( cbgnd )           // specified background
            {
               case taBLACK:   color = taGRBK ;    break ;
               case taRED:     color = taGRRE ;    break ;
               case taBROWN:   color = taGRBR ;    break ;
               case taBLUE:    color = taGRBL ;    break ;
               case taMAGENTA: color = taGRMA ;    break ;
               case taCYAN:    color = taGRCY ;    break ;
               case taGREY:    color = taGRGY ;    break ;
               default:        // silence the compiler warning
                  ;
            } ;
         }
         else if ( cfgnd == taBROWN )  // for brown foreground (often bolded to yellow)
         {
            switch ( cbgnd )           // specified background
            {
               case taBLACK:   color = taBRBK ;    break ;
               case taRED:     color = taBRRE ;    break ;
               case taGREEN:   color = taBRGR ;    break ;
               case taBLUE:    color = taBRBL ;    break ;
               case taMAGENTA: color = taBRMA ;    break ;
               case taCYAN:    color = taBRCY ;    break ;
               case taGREY:    color = taBRGY ;    break ;
               default:        // silence the compiler warning
                  ;
            } ;
         }
         else if ( cfgnd == taBLUE )   // for blue foreground
         {
            switch ( cbgnd )           // specified background
            {
               case taBLACK:   color = taBLBK ;    break ;
               case taRED:     color = taBLRE ;    break ;
               case taGREEN:   color = taBLGR ;    break ;
               case taBROWN:   color = taBLBR ;    break ;
               case taMAGENTA: color = taBLMA ;    break ;
               case taCYAN:    color = taBLCY ;    break ;
               case taGREY:    color = taBLGY ;    break ;
               default:        // silence the compiler warning
                  ;
            } ;
         }
         else if ( cfgnd == taMAGENTA )// for magenta foreground
         {
            switch ( cbgnd )           // specified background
            {
               case taBLACK:   color = taMABK ;    break ;
               case taRED:     color = taMARE ;    break ;
               case taGREEN:   color = taMAGR ;    break ;
               case taBROWN:   color = taMABR ;    break ;
               case taBLUE:    color = taMABL ;    break ;
               case taCYAN:    color = taMACY ;    break ;
               case taGREY:    color = taMAGY ;    break ;
               default:        // silence the compiler warning
                  ;
            } ;
         }
         else if ( cfgnd == taCYAN )   // for cyan foreground
         {
            switch ( cbgnd )           // specified background
            {
               case taBLACK:   color = taCYBK ;    break ;
               case taRED:     color = taCYRE ;    break ;
               case taGREEN:   color = taCYGR ;    break ;
               case taBROWN:   color = taCYBR ;    break ;
               case taBLUE:    color = taCYBL ;    break ;
               case taMAGENTA: color = taCYMA ;    break ;
               case taGREY:    color = taCYGY ;    break ;
               default:        // silence the compiler warning
                  ;
            } ;
         }
         else if ( cfgnd == taGREY )   // for grey foreground
         {
            switch ( cbgnd )           // specified background
            {
               case taBLACK:   color = taGYBK ;    break ;
               case taRED:     color = taGYRE ;    break ;
               case taGREEN:   color = taGYGR ;    break ;
               case taBROWN:   color = taGYBR ;    break ;
               case taBLUE:    color = taGYBL ;    break ;
               case taMAGENTA: color = taGYMA ;    break ;
               case taCYAN:    color = taGYCY ;    break ;
               default:        // silence the compiler warning
                  ;
            } ;
         }
         else     // invalid background (should never happen)
            ;
      }

      #if DEBUG_CSCHEME != 0
      gsdebug.append( " color:%s", dcsName[color] ) ;
      gsdebug.padCols( 80, L'-' ) ;
      //wcout << gsdebug.gstr() << endl ;
      //sleep ( 6 ) ;
      #endif   // DEBUG_CSCHEME
   }
   return color ;

}  //* End gclaColorScheme() *

//*************************
//*                       *
//*************************
//******************************************************************************
//*                                                                            *
//*                                                                            *
//*                                                                            *
//* Input  :                                                                   *
//*                                                                            *
//* Returns:                                                                   *
//******************************************************************************

