//******************************************************************************
//* File       : Taggit.hpp                                                    *
//* Author     : Mahlon R. Smith                                               *
//*              Copyright (c) 2016-2020 Mahlon R. Smith, The Software Samurai *
//*                 GNU GPL copyright notice below                             *
//* Date       : 06-Oct-2020                                                   *
//* Version    : (see AppVersion string below)                                 *
//*                                                                            *
//* Description: Class definition and miscellaneous constant definitions       *
//*              for the Taggit application.                                   *
//*                                                                            *
//*                                                                            *
//******************************************************************************
//* Copyright Notice:                                                          *
//* This program is free software: you can redistribute it and/or modify it    *
//* under the terms of the GNU General Public License as published by the Free *
//* Software Foundation, either version 3 of the License, or (at your option)  *
//* any later version.                                                         *
//*                                                                            *
//* This program is distributed in the hope that it will be useful, but        *
//* WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY *
//* or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License   *
//* for more details.                                                          *
//*                                                                            *
//* You should have received a copy of the GNU General Public License along    *
//* with this program.  If not, see <http://www.gnu.org/licenses/>.            *
//*                                                                            *
//*         Full text of the GPL License may be found in the Texinfo           *
//*         documentation for this program under 'Copyright Notice'.           *
//******************************************************************************

//****************
//* Header Files *
//****************
//* Externally-defined class definitions *
#include "GlobalDef.hpp"      //* NcDialog family of classes

#include <unistd.h>           //* For Linux/UNIX file access
#include <fstream>            //* For C++ file I/O
#include <thread>             //* std::thread definition (and POSIX thread defs)
#include <condition_variable> //* 
#include <chrono>             //* timers for temporarily putting threads to sleep
#include <mutex>              //* for access locks on critical data
#include <errno.h>
#include <sys/stat.h>
#include <dirent.h>
#include <fcntl.h>
#include <pwd.h>
#include <grp.h>
#include <sys/types.h>
#include <wordexp.h>          //* For environment-variable substitution
#include <limits.h>           //* for realpath() and PATH_MAX
#include <stdlib.h>

#include "TagMp3.hpp"         //* Metadata definitions and access methods for MPEG-3 audio
#include "TagOgg.hpp"         //* OGG (OGG/Vorbis I) specification
#include "TagAsf.hpp"         //* Advanced System Format (ASF/WMA) specification
#include "TagData.hpp"        //* Application data control structures

#ifndef TAGGIT_INCLUDED
#define TAGGIT_INCLUDED

//***************
//* Definitions *
//***************

//* This is a debugging flag which allows selective toggling of the cfgOpt.rtl *
//* flag for development of RTL-language support. The effect is that display   *
//* data for the specified user-interface language will be positioned and      *
//* written as if it were an RTL language. This provides visual confirmation   *
//* that when we DO implement an RTL language (probably Hebrew), it will be    *
//* displayed properly. When the RTL languages are fully supported, then this  *
//* definition and all code under its control will go away.                    *
#define FAKE_RTL (1)

//const short MAX_PATH  = gsMAXBYTES ;      // POSIX definition for path/filename length
//const short MAX_FNAME = MAX_PATH / 16 ;   // POSIX definition for filename length

//* The following c-type structures defined in the standard header files 
//* need to be C++ data types: stat{} dirent{}, etc.
typedef struct stat64 FileStats ;
typedef struct dirent64 deStats ;
typedef struct tm Tm ;
typedef struct passwd PwdInfo ;
typedef struct group GrpInfo ;
typedef struct utimbuf uTime ;

//* Local definitions for file type encoded in the "st_mode" *
//* element of the stat64{} structure                        *
enum fmFType : short
{ fmDIR_TYPE,   fmREG_TYPE,  fmLINK_TYPE, fmCHDEV_TYPE, 
  fmBKDEV_TYPE, fmFIFO_TYPE, fmSOCK_TYPE, fmUNKNOWN_TYPE, fmTYPES } ;

//* Class for reporting the system local date/time.                            *
//* ---------------------------------------------------------------------------*
const int64_t maxETIME32 = 0x07FFFFFFF,      // largest 32-bit time_t
              minETIME32 = -maxETIME32 - 1 ; // smallest 32-bit time_t
const short ltFMT_LEN = 16 ;                 // length of time string buffers
//* Important note about the year 2038 bug:                                    *
//* Because 'time_t' is defined as a signed, 32-bit value on 32-bit systems,   *
//* time reporting will break down for date/time values                        *
//*                     >= 19 Jan 2038 at 11:14:08                             *
//* For this reason and for others, we have defined our own local-time class.  *
//* This class uses a signed, 64-bit value for storing the epoch time.         *
//* It is hoped that all Linux systems will have been updated long before this *
//* becomes an issue; however, even now (2013), some applications are using    *
//* date/time manipulations that exceed the 32-bit limit. BEWARE!!             *
//* NOTE: time arithmetic should always be performed on signed 64-bit values.  *
//*                                                                            *
class localTime
{
   public:
   localTime ( void ) { this->reset() ; }
   void reset ( void )
   {  //* Set to epoch date: Thursday, 01 Jan. 1970 @ 08:00:00
      this->sysepoch = ZERO ;
      this->epoch = ZERO ;
      this->day = 4 ;
      this->year = 1970 ;
      this->month = this->date = 1 ;
      this->hours = 8 ;
      this->minutes = this->seconds = ZERO ;
      this->julian = ZERO ;
      #if 0    // not currently used
      *this->timezone = NULLCHAR ;
      *this->utc_zone  = NULLCHAR ;
      this->gmtoffset = ZERO ;
      this->dst       = false ;
      #endif   // not currently used
   }
   int64_t  epoch ;        //* Seconds since the epoch (01-Jan-1970)
   time_t   sysepoch ;     //* system epoch time (see note)
   USHORT   day ;          //* Day of the week (0 == Sun, ..., 6 == Sat)
   USHORT   date ;         //* Day of the month (1-[28|29|30|31])
   USHORT   month ;        //* Month of the year (1 == Jan, ... 12 == Dec)
   USHORT   year ;         //* Year (four digit year)
   USHORT   hours ;        //* Time of day, hour (0-23)
   USHORT   minutes ;      //* Time of day, minutes (0-59)
   USHORT   seconds ;      //* Time of day, seconds (0-59)
   USHORT   julian ;       //* Completed days since beginning of year

   #if 0    // additional fields not currently used
   // See DecodeEpochTime method for more information.
   char     timezone[ltFMT_LEN] ; //* string representing the local time zone (useless)
   char     utc_zone[ltFMT_LEN] ; //* string representing the UTC offset for local time zone
   short    gmtoffset ;    //* Expressed as (signed) number of seconds offset from UTC+0:00
   bool     dst ;          //* 'true' if daylight savings time in effect,
                           //* 'false if not DST or info not available
   #endif   // not currently used
} ;

//* Class definition for human-readable file stats.                            *
class tnFName
{
   public:
   ~tnFName ( void ) {}       //* Destructor
   tnFName ( void )           //* Constructor
   { this->ReInit () ; }
   void ReInit(void)
   {
      fName[0] = NULLCHAR ;
      fType    = fmTYPES ;
      fBytes   = ZERO ;
      modTime.day = modTime.date = modTime.month = modTime.year 
                  = modTime.hours = modTime.minutes = modTime.seconds = ZERO ;
      readAcc  = false ;
      writeAcc = false ;
      // NOTE: data member, rawStats is left unitialized
   }
   char        fName[MAX_FNAME] ; //* Filename string
   fmFType     fType ;     //* File type (enum fmFType)
   UINT64      fBytes ;    //* File size (in bytes)
   localTime   modTime ;   //* Date/Time file last modified (human readable)
   FileStats   rawStats ;  //* Copy of system's "stat" structure (all file info)
   bool        readAcc ;   //* 'true' if user has read access to file
   bool        writeAcc ;  //* 'true' if user has write access to file 
} ;


//* Interface language options. (see also alStrings[]) *
enum AppLang : short
{ 
  enLang = ZERO,  // English
  esLang,         // Español (Spanish)
  zhLang,         // Zhōngwén (中文) (Chinese, simplified)
  viLang,         // Tiếng Việt (Vietnamese)
  locLang         // language taken from locale if supported, else default language
} ;
//* For LTR (left-to-right) languages, reset. *
//* For RTL (right-to-left) languages, set.   *
const bool AppDirection[] =
{
   false,         // English,   LTR
   false,         // Espanol,   LTR
   false,         // Zhongwen,  LTR
   false,         // TiengViet, LTR
} ;

//* Color Scheme. Provides standardized colors for various display objects, so *
//* user can select desired background color on startup, and the dialog's      *
//* foreground objects and sub-dialogs will display in complementary colors.   *
class ColorScheme
{
   public:
   NcBaseColors scheme ;   // color scheme indicator
   attr_t bb ;             // dialog backgrounds and borders
   attr_t sd ;             // sub-dialog color
   attr_t sb ;             // sub-dialog Bold color
   attr_t em ;             // sub-dialog Emphasis color
   attr_t dm ;             // sub-dialog Dim color
   attr_t mn ;             // menu color (non-focus)
   attr_t mf ;             // menu color (with focus)
   attr_t pn ;             // pushbutton color (non-focus)
   attr_t pf ;             // pushbutton color (with focus)
   attr_t tn ;             // text box color (non-focus)
   attr_t tf ;             // text box color (with focus)
   attr_t title ;          // application's title color
   attr_t saved ;          // data-saved indicator color
   attr_t unsaved ;        // data-NOT-saved indicator color
   attr_t menu[2] ;        // menu interior color
} ;

//* Sort options. Sort by contents of specified field. *
enum SortBy : short
{
   sbFNAME,                // Filename
   sbTITLE,                // Title field
   sbTRACK,                // Track field
   sbALBUM,                // Album field
   sbARTIST,               // Artist field
   sbNONE,                 // No sort (in the order read from commline or disc)
} ;

//* Used by the WriteTargetFile() group to specify the metadata output filter. *
enum WriteFilter : short
{
   wfALL_DATA,             // Save all tag fields which contain data (default)
                           // (both active AND inactive fields)
   wfALL_ACTDATA,          // Save all active tag fields which contain data
                           // (empty active fields and all inactive fields ignored)
   wfALL_ACTIVE,           // Save all active tag fields, whether or not they 
                           // contain data. (all inactive fields ignored)
   wfAUDIO_ONLY,           // Save the audio data only.
                           // (all metadata discarded)
} ;

//* Warning/Error codes passed to the OpenDialog() method. *
enum odWarn : short
{
   odGOOD_DATA,   // no start-up errors
   odFILE_ACCESS, // (Fatal) user has limited access to one or more source files,
                  //         or specified file is usupported type or does not exist
   odNO_SOURCE,   // (Fatal) no source files identified
   odNO_ACCESS,   // (Fatal) user does not have read access to source directory
   odNOT_DIR,     // (Fatal) specified source directory is not a directory
   odSRC_FORMAT,  // (Fatal) source file(s) format does not match filename extension
   odREAD_ONLY,   // user has read-only access to source directory
   odDUPLICATES,  // user has specified duplicate source filenames
   odPRESERVE,    // Source file backup before modifications
   odSTRIP,       // Any existing source file metadata will be discarded
   odPRES_STRIP,  // both odPRESERVE and odSTRIP
} ;

//* Gather and process command-line arguments *
class commArgs
{
   public:
   commArgs ( int argc, char** argv, char** argenv ) :
               argCount(argc), argList(argv), envList(argenv)
   {
      *this->appPath    = NULLCHAR ;
      *this->cfgPath    = NULLCHAR ;
      *this->startDir   = NULLCHAR ;
      *this->altLocale  = NULLCHAR ;
      *this->fdArtist   = NULLCHAR ;
      *this->fdAlbum    = NULLCHAR ;
      this->colorScheme = ncbcCOLORS ; // default color scheme
      this->language    = locLang ;    // default language (from locale OR English)
      this->diagPause   = ZERO ;       // no pause on startup
      this->sortOption  = sbFNAME ;    // sort by filename
      this->enableMouse = true ;       // mouse enabled by default
      this->fieldFlag   =              // no field values specified
      this->presSrc     =              // reset special-processing flags
      this->stripMeta   = 
      this->insImage    =              // no image specified for insertion
      this->dumpMeta    =              // no dump to stdout
      this->mouseFlag   =              // no user-specified mouse setting
      this->sortFlag    =              // no sort criterion specified
      this->allFields   =              // let config file specify which fields to display
      this->verFlag     =              // reset 'help' and 'version' flags
      this->helpFlag    = false ;
   }

   //** Data Members **
   short    argCount ;           // application's 'argc'
   char**   argList ;            // application's 'argv'
   char**   envList ;            // application's 'argenv'
   char     appPath[MAX_PATH] ;  // directory of application executable
   char     cfgPath[MAX_PATH] ;  // -C option: alternate configuration file
   char     startDir[MAX_PATH] ; // -d option: startup directory
   char     altLocale[MAX_FNAME];// -L option: alternate character encoding locale name
   char     fdArtist[MAX_FNAME] ;// --artist option: data to be duplicated to all files
   char     fdAlbum[MAX_FNAME] ; // --album option:  data to be duplicated to all files
   id3v2_image image ;           // -I option: data for importing an external image
   NcBaseColors colorScheme ;    // -c option: border color scheme
   AppLang  language ;           // -l option: user-interface language
   SortBy   sortOption ;         // -s option: sort by criterion
   short    diagPause ;          // -p option: >0 == pause after start-up diagnostics
                                 // (undocumented) >1 == display verbose start-up diagnostics
   bool     enableMouse ;        // --mouse option: 'true' if enable mouse support
   bool     fieldFlag ;          // field options: 'true' if one or more field values specified
   bool     mouseFlag ;          // 'true' if user specified a mouse setting
   bool     sortFlag ;           // 'true' if user specified a sort option
   bool     allFields ;          // 'true' if user specified to enable display of all tag fields
   bool     presSrc ;            // -P option: preserve (do not modify) original source files
   bool     stripMeta ;          // -i option: strip (ignore) all existing metadata on startup
   bool     insImage ;           // -I option: insert an image into all files
   bool     dumpMeta ;           // -D option: write metadata to stdout and exit
   bool     verFlag ;            // --version: 'true' if display command-line version
   bool     helpFlag ;           // --help option: 'true' if display command-line help
} ;

//* Configuration Options structure *
class ConfigOptions
{
   public:

   //* Default constructor *
   ConfigOptions ( void )
   {
      this->reset () ;
   }

   void reset ( void )
   {
      *this->appPath    = NULLCHAR ;   // will be initialized during startup
      *this->srcPath    = NULLCHAR ;   // will be initialized during startup
      *this->tfPath     = NULLCHAR ;   // will be initialized during startup
      *this->appLocale  = NULLCHAR ;   // receives value (if any) from config file
      this->extImage.reset() ;         // receives data for external image
      this->appLanguage = locLang ;    // default language from environment (OR English)
      this->sortBy      = sbFNAME ;    // sort by filename
      this->txtEncode   = ENCODE_ASCII;// preferred text encoding
      this->enableMouse = true ;       // (enabled by default)
      this->rtl         = false ;      // (assume an LTR language)
      this->rtlf        = false ;
      this->preserve    = false ;      // (false by default)
      this->ignore      = false ;      // (false by default)
   }

   //* Assignment operator: copy one instance to another *
   void operator = ( const ConfigOptions& coSrc )
   {
      enableMouse    = coSrc.enableMouse ;
      for ( short i = ZERO ; i < MAX_PATH ; i++ )
      {
         appPath[i] = coSrc.appPath[i] ;
         if ( appPath[i] == NULLCHAR )
            break ;
      }
      for ( short i = ZERO ; i < MAX_PATH ; i++ )
      {
         tfPath[i] = coSrc.tfPath[i] ;
         if ( tfPath[i] == NULLCHAR )
            break ;
      }
   }

   //* DATA MEMBERS (public) *
   char appPath[MAX_PATH] ;   //* Directory of application executable
                              //* default config file, and Help live here
   char srcPath[MAX_PATH] ;   //* Path spec for CWD _or_ specified alt source directory
   char tfPath[MAX_PATH] ;    //* Path specification for creating temporary files
   char appLocale[MAX_FNAME] ;//* Locale name from environment or user-supplied
   id3v2_image extImage ;     //* Data for importing an external image
   AppLang appLanguage ;      //* Language used for application interface
   SortBy  sortBy ;           //* Record sort option
   TextEncode txtEncode ;     //* Preferred text encoding
   bool enableMouse ;         //* If true, mouse support enabled
   bool rtl ;                 //* Set (true) if user-interface language is RTL
   bool rtlf ;                //* Set (true) when user-interface language is RTL
                              //*   _AND_ RTL display of field contents
   bool preserve ;            //* If 'true' write to copy, if 'false' write to original file
   bool ignore ;              //* If 'true' ignore existing metadata, if 'false' retain metadata
} ;   // ConfigOptions

//* This is a dummy declaration. The actual declaration *
//* of this class is found in TagOgg.hpp.               *
class Ogg_Comment ;



//***********************************************
//* Constant Data:                              *
//***********************************************

const char* const AppTitle1 = "  Taggit - v:" ;   // constant strings for application title
const char* const AppTitle2 = "The Software Samurai" ;
const char* const AppVersion = "0.0.08" ;
const char* const copyrightYears = "2016-2020" ;

//* Name of default configuration file and ID string *
const char* const ConfigTaggit = "Taggit.cfg" ;
const char* const ConfigFileID = "TGCFG" ;

//* Name of file containing Taggit's Help Text (.info format) *
const char* const HelpFilename = "taggit.info" ;

//* Name of Metadata-summary output file. *
const char* const mdDumpName = "Metadata_Summary.txt" ;

//* Technical Support form. See DisplaySupportInfo(). *
const char* const tsFName = "Taggit_SupportRequest.txt" ;

//* Display names for members of enum AppLang. Keep them synchronized. *
const wchar_t* const alStrings[] = 
{
   L"English",
   L"Español",
   L"Zhōngwén",
   L"TiếngViệt",
   L"(from locale)",
} ;

//* 'OK' button text. Used for instantiating the genDialog class.*
const char* const okText[4] = 
{
   "  ^OK  ",        // English
   "  BUEN^O  ",     // Espanol
   "   好   ",       // Zhongwen
   "   Được   ",      // TiengViet
} ;

const short MIN_APP_ROWS = 30 ;        // minimum terminal dimensions for application
const short MIN_APP_COLS = 132 ;
const short FNAME_WIN_WIDTH = 39 ;     // interior width of filename window
const wchar_t openCIRCLE = 0x025CB ;   // circle character ( ○ unfilled)
const wchar_t fillCIRCLE = 0x025CF ;   // circle character ( ● filled)
const wchar_t FISHEYE    = 0x025C9 ;   // circle character ( ◉ )
const wchar_t BULLSEYE   = 0x025CE ;   // circle character ( ◎ )
//         wchar_t big_circle = 0x020DD ;
//         wchar_t big_square = 0x020DE ;
//         wchar_t big_nono   = 0x020E0 ;
//         wchar_t large_circle = 0x025EF ;

//* Dummy class declaration. This class is defined in TagOgg.hpp. *
class Ogg_Segments ;

class Taggit
{
   public:
   virtual ~Taggit ( void ) ;          // destructor
   Taggit ( commArgs& clArgs ) ;       // constructor

   private:

   //* Gather and process command-line arguments *
   bool  GetCommandLineArgs ( commArgs& ca ) ;
   void  gcaFindAppPath ( commArgs& ca ) ;
   void  gcaCaptureName ( char* trg, const char* src, short maxBytes ) ;
   bool  gcaParseFilelist ( const char* argPtr ) ;
   bool  gcaReadFilelist ( const char* argPtr, char* startDir ) ;
   bool  gcaScanSourceDir ( const char* argPtr, char* startDir ) ;
   bool  Realpath ( gString& rpath ) ;

   //***********************
   //* Configuration group *
   //***********************
   //* Read setup parameters from configuration file.                          *
   short ReadConfigFile ( const char* altConfig, attr_t cfgAttr, 
                          attr_t goodAttr, attr_t badAttr, attr_t nameAttr ) ;
   short rcfReadSourceLine ( ifstream& ifs, short& lineNum, gString& gs ) ;
   //* Set language for user interface                                         *
   short Set_UI_Language ( AppLang clLang ) ;
   //* Verify that user-provided locale is on list of supported locales.       *
   bool  IsSupportedLocale ( const char* localeName ) ;
   //* Test the NCurses Engine's color capabilities.                           *
   bool  IsColorTerm ( attr_t goodAttr, attr_t badAttr ) ;
   //* Create temp-file target directory                                       *
   bool  CreateTemppath ( gString& basePath ) ;
   //* Create a temp-file path/filename                                        *
   bool  CreateTempname ( gString& tmpPath ) ;
   //* Delete (unlink) the specified source file.                              *
   short DeleteTempname ( const gString& srcPath ) ;
   //* Delete temp-file target directory and its contents                      *
   void  DeleteTemppath ( void ) ;
   //* Stub for C++17 function fs::temp_directory_path().                      *
   bool  GetTempdirPath ( gString& tdPath ) ;

   //* Print command-line version info to console                              *
   void  DisplayVersion ( void ) ;
   //* Print command-line help to console                                      *
   void  DisplayHelp ( void ) ;

   //************************
   //* File Utilities group *
   //************************
   //* Determine whether specified file exists and return its filetype.        *
   bool  TargetExists ( const gString& trgPath, fmFType& fType ) ;
   //* Perform a 'stat' (lstat) on the specified file.                         *
   short GetFileStats ( const gString& trgPath, tnFName& fStats ) ;
   //* Stat the source directory to verify file type + r/w/x access.           *
   short SourceDirAccess ( fmFType& sdType, bool& sdRead, bool& sdWrite ) ;
   //* Copy a file preserving all attributes.                                  *
   bool  CopyFile ( const gString& srcPath, const gString& trgPath ) ;
   bool  CopyFile ( const wchar_t* srcPath, const wchar_t* trgPath ) ;
   bool  CopyFile ( const char*    srcPath, const char*    trgPath ) ;
   //* Delete (unlink) the specified source file.                              *
   short DeleteFile ( const gString& srcPath ) ;
   //* Rename the specified file.                                              *
   short RenameFile ( const gString& srcPath, const gString& trgPath ) ;
   //* "Escape" any double quotes in filespec strings.                         *
   bool  DQuoteEscape ( gString& src ) ;
   //* Launch the system default application for the specified file.           *
   int LaunchDefaultApplication ( const char* targFile, bool redirect = true ) ;
   //* Launch a specific external application along with desired arguments.    *
   int LaunchExternalApplication ( const char* appSpec, const gString& argList, 
                                   bool redirect = true ) ;

   //********************
   //* Media File group *
   //********************
   //* Read the media files and extract the metadata tag fields.               *
   short LoadMetadata ( void ) ;
   bool  LoadMetadata ( short fIndex ) ;
   bool  ExtractMetadata_MP3 ( short si ) ;
   bool  ExtractMetadata_OGG ( short si ) ;
   bool  ExtractMetadata_ASF ( short si ) ;
   bool  asfReadBinaryDescriptor ( short si, char* ibuff, asfConDesc& asfMeta, 
                                   short descIndx, ofstream& dbg ) ;
   bool  asfPrescanMetadata ( short si, WriteFilter filter, asfConDesc& asfMeta, 
                              char* ibuff, ofstream& dbg ) ;
   uint64_t asfFormatMetadata ( short si, asfConDesc& asfMeta, 
                                ofstream& ofs, ofstream& dbg ) ;
   int   oggReadCommentVectors ( short si, ifstream& ifs, int cvCount, 
                                 ofstream& dbg, Ogg_Comment* ocPtr = NULL ) ;
   bool  oggReadImageVector ( short si, char* ibuff, UINT32 ibLen, UINT32 veLen, 
                              ifstream& ifs, UINT32& srcCount, ofstream& dbg ) ;
   //* Write edited metadata to target files.                                  *
   bool  WriteMetadata_MP3 ( short si, WriteFilter filter, bool preserve = false ) ;
   int   mptEncodeTextFrames ( short si, WriteFilter filter, ofstream& tofs, ofstream& ofslog ) ;
   int   mptEncodeImageFrames ( short si, ofstream& tofs, ofstream& dbg ) ;
   short mptEncodePopFrame ( short si, ofstream& tofs, ofstream& dbg ) ;
   bool  WriteMetadata_OGG ( short si, WriteFilter filter, bool preserve = false ) ;
   const wchar_t* oggUnmapField ( short fldCode ) const ;
   //* Format the edited metadata in preparation for writing to target file.   *
   short oggPrescanMetadata ( short si, WriteFilter filter, Ogg_Segments& seg, ofstream& dbg ) ;
   //* Verify OGG/Vorbis CRC error-detection value (checksum).                 *
   bool  oggVerifyCRC ( short si ) ;
   bool  WriteMetadata_ASF ( short si, WriteFilter filter, bool preserve = false ) ;

   //* Compare the filename extension of the specified file against the list   *
   //* of supported media files from which we can extract the file's metadata. *
   MediafileType MediafileTarget ( const gString& trgPath ) ;
   //* Convert raw filename data to full filespecs and verify file exists AND  *
   //* that file is of a supported audio format.                               *
   short VerifyRawFilenames ( void ) ;
   //* Remove any source records for duplicated filespecs.                     *
   short RemoveDuplicates ( void ) ;

   //* For specified fields, duplicate those fields through all source files.  *
   void  AutoDuplicate ( const commArgs& ca ) ;

   //* Get the system timecode and convert it to localTime format.             *
   bool  GetLocalTime ( localTime& ft ) ;

   //* Reverse the order of wchar_t characters in a gString object.            *
   short gsRevChars ( gString& gs ) ;

   //*****************************************
   //* Dialog definition and interface group *
   //*****************************************
   void  InitColorScheme ( void ) ;          // Initialization
   void  InitAppControls ( void ) ;          // Initialization
   void  InitStaticDisplay ( void ) ;        // Initialization
   void  InitIndices ( void ) ;              // Display formatting
   void  InitSimulatedData ( short findex = -1 ) ; // DEBUGGING ONLY
   bool  OpenDialog ( odWarn odCode, bool& abort ) ;
   bool  ResizeDialog ( bool forceUpdate = false ) ;
   void  CloseDialog ( void ) ;
   short EditsPending ( void ) ;             // scan for pending edits
   void  Warning ( odWarn odCode ) ;
   void  UpdateFilenameWindow ( void ) ;     // Re-draw display data
   void  UpdateMetadataWindow ( void ) ;
   void  UpdateDataWindows ( void ) ;
   short NextActiveField ( void ) ;          // Navigation
   short PrevActiveField ( void ) ;
   short LastActiveField ( void ) ;

   short UserInterface ( void ) ;            // Talk with the animals
   bool  uiControls ( wkeyCode& wk ) ;
   bool  uiEditControl ( wkeyCode& wk, short ctrlObj ) ;
   short uiItemSelected ( const wkeyCode& wk ) ;
   short uiFieldSelected ( const wkeyCode& wk ) ;
   bool  uiEditFilename ( void ) ;
   void  uiefDuplicate ( bool titFlag, bool t2aFlag, bool a2tFlag, bool trkFlag ) ;
   void  uiefConstruct ( gString& gsOut, short fIndex, bool titFlag, 
                         bool t2aFlag, bool a2tFlag, bool trkFlag ) ;
   bool  uiEditMetadata ( short fldIndex ) ;
   bool  uiemFormatTrack ( gString& trkField ) ;
   bool  uiemFormatYear  ( gString& trkField ) ;
   void  uiemTransfer ( tagFields& eBuff, short fIndex ) ;
   void  uiemTransfer ( short fIndex, tagFields& eBuff ) ;
   short uiemLoadNext ( NcDialog* dp, const winPos& wp, tagFields& eBuff, 
                        short fIndex, short fldIndex ) ;
   short uiemLoadPrev ( NcDialog* dp, const winPos& wp, tagFields& eBuff, 
                        short fIndex, short fldIndex ) ;
   short uiemLoadField ( NcDialog* dp, const winPos& wp, 
                         tagFields& eBuff, short fIndex, short fldIndex ) ;
   bool  uiemFxSelect ( short fldIndex, gString& newValue,
                        short dRows, short dCols, short dY, short dX ) ;
   bool  uiemFieldModified ( NcDialog* dp, const winPos& wp, 
                             const tagFields& eBuff, short fldIndex, bool refresh = false ) ;
   bool  uiemCompareField ( const tagFields& eBuff, short fldIndex = tfCOUNT ) ;
   short uiEditsPending ( short modFiles ) ;
   void  uiDefineReservedKeys ( NcDialog* dp ) ;

   void  DebugMsg ( const gString& gsMsg, bool clear = false, 
                    short dbLine = ZERO, attr_t cAttr = nc.blR ) ;  // DEBUGGING ONLY
   void  DebugMsg ( const char* csMsg, bool clear = false, 
                    short dbLine = ZERO, attr_t cAttr = nc.blR ) ;  // DEBUGGING ONLY


   //**********************
   //* Menu Command Group *
   //**********************
   //* Sort the records according to the cfgOpt.sortBy option.                 *
   void  Cmd_SortList ( void ) ;
   //* Move an item up or down through the list.                               *
   bool  Cmd_SwapItems ( short aIndex, short bIndex ) ;
   //* Report target filesystem free space.                                    *
   void  Cmd_ReportTargetStats ( void ) ;
   //* Select active display fields.                                           *
   void  Cmd_ActiveFields ( bool enable, bool redraw ) ;
   //* Change the application's color scheme.                                  *
   void  Cmd_SetColorScheme ( void ) ;
   //* Put application dialog to sleep and go to command line.                 *
   void  Cmd_ShellOut ( bool prompt = false, const char* extCmd = NULL ) ;
   //* Write the edited metadata to the target file(s)                         *
   bool  Cmd_WriteTargetFile ( short fIndex = (-1), bool audioOnly = false ) ;
   bool  WriteTargetFile ( short fIndex, WriteFilter filter ) ;
   //* Scan the list of source files to verify that they have not been modified*
   //* since they were originally read.                                        *
   short wtfReverifySource ( short fIndex ) ;
   //* Write metadata summary to a file.                                       *
   bool  Cmd_TagSummary ( bool viewData, bool saveData, 
                          bool allData = true, bool prompt = false ) ;
   bool  ctsPrompt ( bool& allFiles, bool& allData, bool &fldRTL ) ;
   bool  ctsSummarize ( const gString& tmpPath, bool allFiles, bool allData, bool fldRTL ) ;
   void  ctsVerifySave ( const gString& trgPath ) ;
   //* Re-read the metadata for specified/all source file(s).                  *
   void  Cmd_RefreshMetadata ( void ) ;
   bool  crmRefreshMetadata ( short fIndex, bool refresh = true ) ;
   //* Clear (erase) display data from all metadata fields.                    *
   void  Cmd_ClearMetadata ( void ) ;
   void  ccmClearMetadata ( short fIndex, bool refresh = true ) ;
   //* Duplicate specified field's contents to all source files.               *
   void  Cmd_DuplicateField ( void ) ;
   void  cdfDuplicateField ( short fldIndex, const gString& newValue, bool redraw = true ) ;
   //* Set the 'Title' field based on the filename.                            *
   void  Cmd_SetSongTitle ( void ) ;
   void  csstSetSongTitle ( short fIndex ) ;
   //* Edit or reset Popularimeter and/or Play Counter.                        *
   void  Cmd_Popularimeter ( void ) ;
   //* Automatically calculate the Track field for all files.                  *
   void  Cmd_SequenceTracks ( void ) ;
   //* For RTL user interface languages, toggle direction of text in fields.   *
   void  Cmd_RTL_ToggleFields ( void ) ;
   //* Display setup and text info for embedded images.                        *
   void  Cmd_ReportImage ( short fIndex ) ;
   void  criTransform ( const char* Template, gString& gsOut, const id3v2_image* epPtr, 
                        const char* noDesc, bool rtlDesc = false ) ;
   //* Embed an image into the target file.                                    *
   void  Cmd_InsertImage ( short fIndex, bool editImg = false ) ;
   bool  ciiMime ( const gString& imgPath, id3v2_image& extImage ) ;
   //* Invoke the default media player for the specified file.                 *
   void  Cmd_Playback ( short fIndex ) ;
   //* Shell out and invoke the 'info' documentation reader.                   *
   void  Cmd_InfoHelp ( bool htmlFormat ) ;
   //* Display 'Quick Help', a list of navigation and shortcut keycodes.       *
   void  Cmd_QuickHelp ( void ) ;
   //* Help-About dialog and tech support info.                                *
   void  Cmd_HelpAbout ( void ) ;
   bool  haSupportInfo ( short ulY, short ulX, short dLines, short dCols ) ;

   //* Temporary application title and diagnostic messages in console window   *
   void  DrawTitle ( void ) ;
   void  DiagMsg ( const char* msg, attr_t color, bool newLine = true ) ;
   void  DiagMsg ( void ) ;


   //****************
   //* Data Members *
   //****************
   //* Pointer Taggit application dialog                                  *
   NcDialog* dPtr ;

   //* Dialog control definitions                                         *
   InitCtrl* ic ;

   //* Size of terminal window (set when NCurses Engine starts)           *
   short termRows,
         termCols ;

   //* Interior dimensions of field-display window. Determines how much   *
   //* field data can be simultaneously displayed.                        *
   short fwinRows,
         fwinCols ;

   //* Configuration Options                                              *
   ConfigOptions cfgOpt ;

   //* Dialog color scheme (borders, interiors, controls) *
   ColorScheme cs ;

   //* List of source files and contents for metadata fields              *
   tagData tData ;

   //* Cursor position for writing start-up messages to console           *
   winPos suPos ;
   bool   suVerbose ;   // 'true' if verbose startup diagnostics
} ;

#endif   // TAGGIT_INCLUDED
