//******************************************************************************
//* File       : Taggit.cpp                                                    *
//* Author     : Mahlon R. Smith                                               *
//*              Copyright (c) 2016-2020 Mahlon R. Smith, The Software Samurai *
//*                  GNU GPL copyright notice located in Taggit.hpp            *
//* Date       : 19-Oct-2020                                                   *
//* Version    : (see AppVersion string)                                       *
//*                                                                            *
//* Description: Taggit is a simple media-file tag editor.                     *
//* Taggit DOES NOT access any of the online databases which are dedicated to  *
//* the collection and distribution of media metadata. It simply allows view   *
//* and edit of all text metadata for supported media formats as well as       *
//* embedded-image metadata for audio formats that support it.                 *
//*                                                                            *
//* Development Tools:                  Current:                               *
//* ----------------------------------  -------------------------------------- *
//*  GNU G++ v:4.8.3 or higher          G++ v:10.2.1                           *
//*  Fedora Linux v:20 or higher        Fedora v:30                            *
//*  GNOME Terminal v:3.10.2 or higher  GNOME Term v:3.32.2                    *
//******************************************************************************
//* Version History (most recent first):                                       *
//*                                                                            *
//* v: 0.00.08 17-Sep-2020                                                     *
//*   -- Implement support for Windows(tm) Media Audio (WMA) files.            *
//*   -- Integrate Wayland system clipboard access for all Textbox controls.   *
//*      This functionality requires the external wlClipboard utilities.       *
//*   -- Documentation update.                                                 *
//*   -- Released 20-Oct-2020                                                  *
//*                                                                            *
//* v: 0.00.07 29-Sep-2019                                                     *
//*   -- The compiler upgrade to GCC (G++) v:9.0.1 introduced some new         *
//*      warning messages.                                                     *
//*      -- The C function snprintf() now causes complaints about potential    *
//*         buffer overrun.                                                    *
//*             -Wformat-truncation, pragma -Wno-format-truncation             *
//*         We now test for output truncation in snprintf() calls which use    *
//*         an "%s" in the formatting template. This silences the compiler     *
//*         warning.                                                           *
//*      -- The C function readdir64_r() is deprecated.                        *
//*             -Wdeprecated-declarations                                      *
//*         Switch to readdir64() which is now thread-safe under the GNU       *
//*         compiler. No functionality change.                                 *
//*   -- Bug Fix: Sorting by track number formerly assumed a single digit      *
//*      for the track number.                                                 *
//*   -- Add a "record-up" pushbutton to the Edit-Metadata dialog.             *
//*      Pushbuttons for all directions are now available:                     *
//*       Left : previous field       Up   : previous file                     *
//*       Right: next field           Down : next file                         *
//*   -- Documentation update.                                                 *
//*   -- Released 18-Dec-2019                                                  *
//*                                                                            *
//* v: 0.00.06 27-Nov-2018                                                     *
//*   - Bug Fix: Command-line options '--artist' and '--album' were not being  *
//*     properly inserted into the data stream for display on application      *
//*     startup.                                                               *
//*   - Bug Fix: If application binary, link or shell script used to invoke    *
//*     the application is on the path, then the configuration file was not    *
//*     being found. See gcaFindAppPath() method.                              *
//*   - Correct logical error: Previously, using the '-d' option to specify a  *
//*     source directory which contained no supported audio files would invoke *
//*     command-line help. Now the application dialog will open and display a  *
//*     "no source files" warning.                                             *
//*   - Documentation update.                                                  *
//*   - Released 29-Nov-2018                                                   *
//*                                                                            *
//* v: 0.00.05 24-Oct-2017                                                     *
//*   - A redundant temporary filename was being created. No change in         *
//*     functionality.                                                         *
//*   - Bug Fix: If target filename was specified, AND source preservation     *
//*     WAS NOT specified, the old source file was not being deleted. Both     *
//*     MP3 and OGG writes contained this error. Output file was not affected. *
//*   - Updates required by switch to G++ v:5.4.0:                             *
//*     - New warning that library functions 'tempnam' and 'tmpnam_r are       *
//*       dangerous. While this is technically true, this application uses     *
//*       them in such a way that negates the danger. In any case, silence the *
//*       compiler warning by updating methods which create temporary files.   *
//*   - Add a Pushbutton control to the uiEditMetadata() method which enables  *
//*     the user to save the edited data (if any) for the current file and open*
//*     the corresponding field in the next audio file for editing.            *
//*     We have found this useful primarily for sequentially editing the       *
//*     'Title' fields, but may be useful for other non-duplicated fields      *
//*     as well.                                                               *
//*   - Documentation update.                                                  *
//*   - Released 05-Mar-2018.                                                  *
//*                                                                            *
//* v: 0.00.04 17-Oct-2017                                                     *
//*   - Bug Fix: Corrected sizing error in encoding an OGG embedded image.     *
//*     This had caused the image to be truncated under some circumstances,    *
//*     which in turn caused media players to incorrectly parse the OGG        *
//*     comment header.                                                        *
//*   - Temporary Fix: The id3v2.3 tag specification (MP3 files) defines an    *
//*     Extended Header which has two functions:                               *
//*        1) If the seldom used CRC is enabled, it contains the CRC value.    *
//*        2) Indicates the number of padding bytes which follow the tag data. *
//*     - The Extended Header is specified as "optional," which all media      *
//*       players we have tested interpret as "it will never be there, even if *
//*       the tag header says that it is, so don't bother to look for it."     *
//*       We are seriously offended by this unprofessional approach to the     *
//*       standard, but in practical terms, this means that if the Extended    *
//*       Header is present, the media players choke and discard ALL the tag   *
//*       data. The playback is fully functional, but all tag fields will be   *
//*       displayed as "unknown."                                              *
//*     - The id3v2.4 specification makes more extensive use of the Extended   *
//*       Header, so id3v2.4-compliant players should parse it correctly;      *
//*       however, the media player industry has been slow to adopt id3v2.4.   *
//*     - Our temporary fix is to disable creation of the extended header, and *
//*       only _optionally_ add padding to the end of the MP3 tag data.        *
//*       This is done using conditional-compile flags, ENABLE_EXTHDR and      *
//*       ENABLE_PADDING, which are defined in the WriteMetadataMP3() method.  *
//*     - With the Extended Header disabled, the tag data become visible in    *
//*       all our media players, whether or not the padding block is present.  *
//*   - Begin implementation of support for references to external images      *
//*     (URL). New data member id3v2_image::url indicates that MIME-type string*
//*     contains a URL instead of an actual MIME type. (NOT YET FUNCTIONAL)    *
//*   - Documentation update.                                                  *
//*                                                                            *
//* v: 0.00.03 11-Oct-2017                                                     *
//*   - Bug Fix: Customized target filename was being ignored during write to  *
//*     target, and without the '-Preserve' option, the target filename was    *
//*     always the same as source filename. This error had been masked by      *
//*     active debugging code.                                                 *
//*   - Bug Fix: If custom target filename, do not rename the source file as   *
//*     a backup file. Formerly the source file was always renamed as backup,  *
//*     but if source will not be modified, it is more user friendly to retain *
//*     its original name. Tip o'the hat to creative muse and best friend, 楚瑶.*
//*   - Bug Fix: If source file's internal format does not correspond to the   *
//*     source filename extension, alert user and declare a fatal error.       *
//*     This was a rookie mistake and your author has egg-on-face: It is ALWAYS*
//*     a basic logical error to assume that the user has a brain or that      *
//*     he/she/it will never name a file as something it is not.               *
//*   - Bug Fix: Corrected a logical error in identification of an MP3 source  *
//*     file which contains no tag header. This error caused the application   *
//*     to see the source file as having an invalid format.                    *
//*     See the id3v2_audioframe class.                                        *
//*   - Bug Fix: For OGG file writes, the edits-pending indicators were not    *
//*     being reset after write to target was complete.                        *
//*   - Bug Fix: For Popularimeter dialog, the data fields were not being      *
//*     initialized to existing values when the dialog opened.                 *
//*   - For validation of path/filename strings:                               *
//*     Filenames for music files often contain odd characters in keeping with *
//*     the tone or mood of the music itself. For example, Ridin'.oga by       *
//*     Rascal Flatts. Some of these characters can cause problems with various*
//*     system functions and utilities. To reduce the chance of these errors   *
//*     we have enhanced filespec parsing to handle special characters:        *
//*       '('  ')'  '&'  '|'  ';'  '<'  '>'  '?'  '"' (double-quote) and       *
//*       ''' (singlequote) as normal filename characters.                     *
//*       See the Realpath() method for more information.                      *
//*   - Documentation update.                                                  *
//*                                                                            *
//* v: 0.00.02 13-Sep-2017                                                     *
//*   - Update user interface translations.                                    *
//*   - Finish implementation of the 'Set All Fields Active' menu item.        *
//*   - Implement decoding and encoding of popularity meter and play count.    *
//*     Implement Cmd_Popularimeter() method so user can edit values.          *
//*   - Implement a method for deleting a single image from the source file.   *
//*     See Cmd_InsertImage() method.                                          *
//*   - Begin implementation of OGG/Vorbis embedded image support. INCOMPLETE. *
//*   - Disable debugging code.                                                *
//*   - Documentation update.                                                  *
//*   - First public release.                                                  *
//*                                                                            *
//* v: 0.00.01 05-Nov-2016                                                     *
//*   - First pass. Source file parsing for MP3 and OGG media files based on   *
//*     the FileMangler View-File code, but with enhanced efficiency and error *
//*     checking.                                                              *
//*   - Startup code, config file format and command-line parsing also adapted *
//*     from FileMangler.                                                      *
//*   - Dynamic resizing of application dialog is a new design based on        *
//*     encapsulation of display data formatting.                              *
//*   - Translations are rudimentary and need review by native speakers of     *
//*     those languages.                                                       *
//*   - ConfigOptions::rtl indicates whether the user-interface language is    *
//*     LTR or RTL. Currently, there is no RTL user-interface language defined,*
//*     but we have tested the application window and most sub-dialogs using   *
//*     simulated data. Note that ConfigOptions::rtlf indicates whether        *
//*     field contents will be displayed and edited as RTL.                    *
//*                                                                            *
//******************************************************************************
//* Programmer's Notes:                                                        *
//*                                                                            *
//* Note on dynamic load library dependencies:                                 *
//*  Taggit requires certain system 'dynamic link library' files to be         *
//*  installed. If one or more needed libraries cannot be located, then the    *
//*  application will not start, and some messages about missing routines will *
//*  be displayed.                                                             *
//*                                                                            *
//*  For a list of the necessary dynamic link libraries, run the 'ldd' command.*
//*    ldd taggit                                                              *
//*                                                                            *
//*                                                                            *
//*     IMPORTANT NOTE   IMPORTANT NOTE   IMPORTANT NOTE   IMPORTANT NOTE      *
//* -------------------------------------------------------------------------- *
//* This application is data driven. What this means is that the order of      *
//* data in various arrays must remain synchronized. The controlling array     *
//* is 'enum TagFields'. All other field-data arrays must conform to the       *
//* order specified in 'TagFields'. THEREFORE: If 'TagFields' changes, then    *
//* all other arrays which depend on it must also be updated.                  *
//* These include, but are not limited to:                                     *
//*  1) tagFields::field[]                                                     *
//*  2) tagFields::prex[]                                                      *
//*  3) tagData::sfDisp[]                                                      *
//*  4) umwHeaders[][tfCount]                                                  *
//*  5) fldHelp[][]                                                            *
//*  6) sextText[][], sextData[], sextAttr[]                                   *
//*  7) OggFieldMap ofma[]                                                     *
//*  8) TextFrameID[]                                                          *
//*                                                                            *
//*                                                                            *
//* Basic application layout:                                                  *
//* -- Three main dialog windows constitute the main application display.      *
//*    1) Top window is full window width:                                     *
//*       a) title                                                             *
//*       b) menu                                                              *
//*       c) status                                                            *
//*       d) application-level controls                                        *
//*       e) context help                                                      *
//*    2) Filename window on the left side (LTR):                              *
//*       a) Filename-edits-pending indicator for each item                    *
//*       b) Field-edits-pending indicator for each item                       *
//*       c) Filename column header                                            *
//*       d) List of filenames                                                 *
//*    3) Fieldnames window on the right side (LTR):                           *
//*       a) Fieldname headers for all active fields                           *
//*       b) List of active fields for each item                               *
//*          (fields may, or may not contain data)                             *
//*                                                                            *
//* -- Secondary window is the field-editing window.                           *
//*    a) Size of edit window is 16-18 rows and width equal to the minimum     *
//*       number of columns for the field-display window.                      *
//*    b) Position of the edit window is in the upper-left corner of the       *
//*       field-display window (LTR).                                          *
//*    a) Textbox for editing the field with focus                             *
//*    b) Pushbuttons for Next/Previous field                                  *
//*    c) Context help in specified language                                   *
//*    d) 'DONE' pushbutton: Write edits into main application display object  *
//*              for the file and return to the application dialog with        *
//*              edits-pending indicator set.                                  *
//*    f) 'CANCEL' pushbutton: Discard recent edits and return to the          *
//*              application dialog with edits-pending indicator unchanged.    *
//*    g) 'CLEAR" pushbutton: Undo all recent field edits, i.e. return field   *
//*              contents to previous state.                                   *
//*    h) For RTL UI languages only, 'RTL' toggle pushbutton.                  *
//*    i) For select fields, offer special options:                            *
//*       -- 'Title' field, manual entry, OR                                   *
//*          a) set according to the '--title' command-line option             *
//*          b) copy from filename (without extension)                         *
//*          c) Optionally apply 'b' option to all items.                      *
//*             (It is difficult to do this smoothly during edit. )            *
//*             (Make it a menu item: Edit/Set Title from Filename)            *
//*       -- 'Album' field, manual entry, OR                                   *
//*          a) set according to the '--album' command-line option             *
//*          b) Optionally copy contents of this field to all items.           *
//*       -- 'Track' field                                                     *
//*          a) specify track number only (" 3") (note two-column format)      *
//*          b) specify both track number AND total tracks (" 4/12")           *
//*       -- 'Genre' field: manual entry, OR                                   *
//*          a) pick one or more from a list of pre-defined genres             *
//*       -- 'Language' field: manual entry, OR                                *
//*          a) pick from a list of pre-defined language codes (ISO-639-2)     *
//*                                                                            *
//* -- Filename edit window, manual entry, OR                                  *
//*    a) copy from 'Title' field                                              *
//*    b) combine Title and Artist ("Tush - ZZ Top")                           *
//*    c) combine Artist and Title ("ZZ Top - Tush")                           *
//*    d) prepend track number (%02hd) to either 'b' or 'c' above              *
//*       (This allows songs to play in the performer's intended order)        *
//*       (because they will be loaded in alphabetical order.         )        *
//*    e) Option to duplicate the pattern described in a-d above, to           *
//*       all filenames.                                                       *
//*                                                                            *
//* -- Miscellaneous dialogs for specific functionality.                       *
//*                                                                            *
//* -------------------------------------------------------------------------- *
//* Translations of text for UI Languages:                                     *
//* ================================================                           *
//*   METHOD or ARRAY   EN ES ZH VI                                            *
//* ------------------- -- -- -- -- -- -- -- --                                *
//* -- enum appLang     EN ES ZH VI                         // language        *
//* -- AppDirection[]   EN ES ZH VI                         // LTR or RTL      *
//* -- alStrings[]      EN ES ZH VI                         // language names  *
//* -- okText           EN ES ZH VI                         // OK pushbuttons  *
//* -- File Menu items  EN ES ZH VI                                            *
//* -- Edit Menu items  EN ES ZH VI                                            *
//* -- View Menu items  EN ES ZH VI                                            *
//* -- Help Menu items  EN ES ZH VI                                            *
//* -- Sort Menu items  EN ES ZH VI                                            *
//* -- Tag Fld Headers  EN ES ZH VI                                            *
//* -- InitAppControls  EN ES ZH VI                                            *
//* -- InitStaticDispla EN ES ZH VI                                            *
//* -- LoadMetadata     EN ES ZH VI                                            *
//* -- Warning          EN ES ZH VI                                            *
//* -- UpdateFilenameWi EN ES ZH VI                                            *
//* -- uiEditFilename   EN ES ZH VI                                            *
//* -- UpdateMetadataWi EN ES ZH VI                                            *
//* -- uiEditMetadata   EN ES ZH VI                                            *
//* -- uiemLoadField    EN                                  // field help desc *
//* -- uiemFxSelect     EN ES ZH VI                                            *
//* -- uiEditsPending   EN ES ZH VI                                            *
//* -- Cmd_ReportTarget EN ES ZH VI                                            *
//* -- Cmd_SetColorSche EN ES ZH VI                                            *
//* -- Cmd_WriteTargetF EN ES ZH VI                                            *
//* -- wtfReverifySourc EN ES ZH VI                                            *
//* -- Cmd_InfoHelp     EN ES ZH VI                                            *
//* -- Cmd_QuickHelp    EN ES ZH VI                                            *
//* -- Cmd_HelpAbout    EN ES ZH VI                                            *
//* -- haSuportInfo     EN ES ZH VI                                            *
//* -- ctsPrompt        EN ES ZH VI                                            *
//* -- ctsSummarize     EN ES ZH VI                                            *
//* -- ctsVerifySave    EN ES ZH VI                                            *
//* -- Cmd_RefreshMetad EN ES ZH VI                                            *
//* -- ccmClearMetadata EN ES ZH VI                                            *
//* -- Cmd_DuplicateFie EN ES ZH VI                                            *
//* -- Cmd_SetSongTitle EN ES ZH VI                                            *
//* -- Cmd_Popularimit  EN ES ZH VI                                            *
//* -- Cmd_ReportImage  EN ES ZH VI                                            *
//* -- Cmd_InsertImage  EN ES ZH VI                                            *
//* -- mptDecodeImageFr EN ES ZH VI                                            *
//*                                                                            *
//*    Note: When adding a new UI language, be sure to add translations for    *
//*          the functionality in all the above methods.                       *
//*                                                                            *
//*                                                                            *
//* -------------------------------------------------------------------------- *
//* Notes on support for RTL (right-to-left) languages:                        *
//* ===================================================                        *
//* 1) For LTR languages, static text and all data fields are always displayed *
//*    as LTR text.                                                            *
//* 2) For RTL languages, things get a bit more complex.                       *
//*    a) Display of static data as RTL is controlled by the cfgOpt.rtl flag.  *
//*       This flag should be set when the user-interface language is selected *
//*       and requires an array (AppDirection[]) parallel to the AppLang[]     *
//*       array indicating whether each language is LTR or RTL.                *
//*       (See Set_UI_Language().)                                             *
//*       -- Because we do not yet have the UI translated into an RTL language,*
//*          we have defined the "FAKE_RTL" flag to simulate RTL data for      *
//*          development purposes.                                             *
//*       -- Debug command CTRL+K, 5 toggles cfgOpt.rtl.                       *
//*       -- Most sub-dialogs and methods have been updated to support RTL     *
//*          languages. The exceptions are: CmdHelpAbout(), haSupportInfo()    *
//*          and the technical support request form. These contain significant *
//*          mixed LTR/RTL text and therefore will require customization for   *
//*          the RTL target language.                                          *
//*       -- Cmd_InfoHelp() is a special case in that it displays its message  *
//*          in all languages simultaneously. We may need to rethink this.     *
//*    b) Filenames are by default displayed as LTR text, both in the static   *
//*       data and in edit fields.                                             *
//*       This is because filename extensions e.g. ".mp3" are inherently LTR.  *
//*       However, we want to give the user the option of displaying and       *
//*       editing filenames as RTL text. This option is enabled by toggling    *
//*       the cfgOpt.rtlf flag or _dynamically_ (for the field under edit)     *
//*       via the CTRL+R hotkey.                                               *
//*       -- The user's toggle between LTR and RTL for field-edit Textboxes    *
//*          is implemented in UserInterface() and both uiEditMetadata() and   *
//*          uiEditFilename().                                                 *
//*       -- The cfgOpt.rtlf flag is toggled by the 'RTL Invert Field Data'    *
//*          menu item.                                                        *
//*       -- Sub-dialogs which contain Textbox controls have an RTL Pushbutton *
//*          which is used to toggle these fields between LTR and RTL.         *
//******************************************************************************
//* To Do List:                                                                *
//*                                                                            *
//* -- Verify cut-and-paste for all Textbox edits.                             *
//*    a) internal NcDialog cut-and-paste                       [DONE]         *
//*    b) system clipboard cut-and-paste (requires API update)                 *
//*                                                                            *
//* -- Implement embedded images (and external links?) for OGG/Vorbis audio    *
//*    files. See notes in TagOgg.cpp and the Ogg_Image class in TagOgg.hpp.   *
//*    Also oggReadCommentVectors(), oggReadImageVector().                     *
//*                                                                            *
//* -- Decoding and encoding image data (MP3 only) appears to be functional,   *
//*    but display of these images by media players has not yet been verified. *
//*                                                                            *
//* -- Disable all write-to-target debugging code and verify clean writes:     *
//*    a) OGG/Vorbis                                                           *
//*    b) MP3                                                                  *
//*    c)                                                                      *
//*                                                                            *
//* -- If a source file's embedded image MIME type references and external     *
//*    image (via URL), then the size of the image will be ZERO.               *
//*    If MIME type is an absolute or relative link to an external image file, *
//*    we _could_ test for its existence and copy its contents to our temp file*
//*    for later embedding. However, the file would need to be on the local    *
//*    system because we are definitely not accessing a network address from   *
//*    this application.                                                       *
//*    -- Check the image capture sequences to verify that a zero-length       *
//*       image does not cause problems. [MP3 ?  ] [OGA ?  ]                   *
//*    -- Check image embedding sequence. What happens if the temp file is     *
//*       ZERO bytes?                                                          *
//*    -- If a zero-length image is found, alert user that we are discarding   *
//*       the reference to the external image.                                 *
//*    -- Either discard the image reference after alerting the user OR make a *
//*       special case for view/edit of image setup data to indicate that the  *
//*       source file contains an external image reference.                    *
//*                                                                            *
//* -- Enhance tfTxxx field edit to allow for heading AND text. (MP3 only)     *
//*    Example:  this is the heading^this is the body text.                    *
//*    If heading not present, then for output use:                            *
//*              User-defined Text Field:this is the body text.                *
//*    -- It seems that no encoders do this, AND we think it's stupid,         *
//*       so we resist full implementation of this stupid, unnecessary feature.*
//*                                                                            *
//* -- Use of the 'modified' indicator in the field edit dialog is not well    *
//*    designed. It should be set if field contents are different from         *
//*    that field in the source file. Instead, it indicates whether the field  *
//*    has been changed during the current file-edit sequence.                 *
//*                                                                            *
//* -- Copyright character in summary file looks like: Â©                      *
//*    This could be a problem with UTF-16 encoding. (See configuration        *
//*    option for _preferred_ text encoding.)                                  *
//*                                                                            *
//* -- Combine EnvExpansion() and Realpath() as Realspec().                    *
//*    -- Update all calls to these methods.                                   *
//*    -- Regexp characters in EnvExpansion() source must be escaped.          *
//*       parentheses () and others...                                         *
//*                                                                            *
//* -- Enable system clipboard interface (when available).                     *
//*                                                                            *
//* --                                                                         *
//*                                                                            *
//* --                                                                         *
//*                                                                            *
//* --                                                                         *
//*                                                                            *
//******************************************************************************

//****************
//* Header Files *
//****************
#include "Taggit.hpp"         // Taggit-class definitions and data
                              // plus general definitions and NcDialogAPI definition

//**************
//* Local data *
//**************

//* Defines a diagnostic message display during the startup sequence *
//* by writing to the basic NCurses screen. See DiagMsg() method.    *
static const short ddmLENGTH = 132 ;
class Diagnostics
{
   public:

   ~Diagnostics ( void ) { }  // destructor
   Diagnostics ( void )       // constructor
   { this->reset () ; }

   void reset ( void )
   {
      this->msgText[0] = NULLCHAR ;
      this->msgAttr    = nc.bw ;
      this->nl         = true ;
   }

   char   msgText[ddmLENGTH] ;// message text
   attr_t msgAttr ;           // color attribute for message
   bool   nl ;                // 'true' if newline after message
} ;

//* Pointer to an array of temporary objects *
//* for displaying diagnostic messages.      *
static Diagnostics* diagData = NULL ;
static const short diagdataCOUNT = 100 ;  // max diag messages


//*************************
//*        main           *
//*************************
//******************************************************************************
//* Program entry point.                                                       *
//*                                                                            *
//* Command-line Usage: run taggit --help                                      *
//*                                                                            *
//* Command Line Arguments: See DisplayHelp()                                  *
//*                                                                            *
//******************************************************************************

int main ( int argc, char* argv[], char* argenv[] )
{
   int exitCode = ZERO ;

   //* User may have specified one or more command-line arguments.           *
   commArgs clArgs( argc, argv, argenv ) ;

   //* Create the application class object and interact with the user.       *
   Taggit* tgptr = new Taggit( clArgs ) ;

   //* Before returning to the system, delete the Taggit object.             *
   //* This forces execution of the class destructor for an orderly exit.    *
   delete tgptr ;

   exit ( exitCode ) ;

}  //* End main() *

//*************************
//*        ~Taggit        *
//*************************
//******************************************************************************
//* Destructor. Return all resources to the system.                            *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

Taggit::~Taggit ( void )
{
   //* Delete the array of dialog controls *
   if ( this->ic != NULL )
      delete [] this->ic ;

   //* If temporary image files are attached to the  *
   //* EmbeddedImage list, delete them now.          *
   EmbeddedImage* eiPtr ;
   gString gs ;
   for ( short si = ZERO ; si < this->tData.sfCount ; ++si )
   {
      if ( this->tData.sf[si].sfTag.ePic.inUse )
      {
         eiPtr = &this->tData.sf[si].sfTag.ePic ;
         while ( eiPtr->next != NULL )
            eiPtr = eiPtr->next ;
         do
         {
            gs = eiPtr->mp3img.picPath ;
            if ( (gs.compare( this->cfgOpt.tfPath, true, 8 )) == ZERO )
               this->DeleteTempname ( gs ) ;
            eiPtr = eiPtr->prev ;
         }
         while ( eiPtr != NULL ) ;
         //* Delete any dynamically-allocated objects *
         //* in the linked list.                      *
         this->tData.sf[si].sfTag.delPic() ;
      }
   }

   //* Delete any temporary files created by the application.*
   this->DeleteTemppath () ;

}  //* End ~Taggit() *

//*************************
//*        Taggit         *
//*************************
//******************************************************************************
//* Constructor.                                                               *
//*                                                                            *
//* Input  : commArgs class object (by reference)                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************
//* Notes:                                                                     *
//* ------                                                                     *
//*                                                                            *
//* There are three(3) levels of initialization:                               *
//*  1) Instantiation of the 'cfgOpt' member.                                  *
//*  2) Configuration file options (from 'Taggit.cfg' or alternate config file)*
//*  3) Command-line options (collected in the 'clArgs' parameter).            *
//* VALID options from each level override the settings of previous level.     *
//*                                                                            *
//* Item/processes initialized:                                                *
//*  1) cfgOpt.appPath : taken from inherited environment                      *
//*  2) cfgOpt.tfPath  : temporary-file path constructed in system's '/tmp'    *
//*                      directory.                                            *
//*  3) Config-file path : located on 'appPath' by default or specified in     *
//*                      'clArgs.cfgPath'                                      *
//*  4) Verbose diagnostics and pause to view diagnostics : clArgs.diagPause   *
//*                      'suVerbose' is initialized from this value.           *
//*  5) User request for 'help' or 'version' come from clArgs.helpFlag and     *
//*                      clArgs.verFlag, respectively.                         *
//*  6) cfgOpt.enableMouse :                                                   *
//*                      a) set during instantiation to enable mouse by default*
//*                      b) modifed from config file                           *
//*                      c) modified _again_ by clArgs.enableMouse             *
//*                         (see also 'mouseFlag')                             *
//*  7) Application Language:                                                  *
//*                      a) taken from environment by default                  *
//*                      b) modified from config file                          *
//*                      c) modified _again_ by clArgs.language                *
//*  8) Input/Output locale:                                                   *
//*                      a) taken from environment by default                  *
//*                      b) modified from config file 'cfgOpt.appLocale'       *
//*                      c) modified _again_ by clArgs.altLocale[]             *
//*  9) Color Scheme :   a) default scheme is set during instantiation         *
//*                         'cfgOpt.cs.scheme'                                 *
//*                      b) modified from config file                          *
//*                      c) modified _again_ by clArgs.colorScheme             *
//* 10) Source filenames: a) by default all valid source files in current      *
//*                          directory will be scanned.                        *
//                        b) may be specified individually by '-f' option      *
//*                       c) may be listed in a specified file '-F' option     *
//*                       d) will be scanned from the directory specified by   *
//*                          the '-d' option.                                  *
//*                       Note that '-f' option may be combined with '-d' or   *
//*                       '-F' option, but that '-d' and '-F' are mutually     *
//*                       exclusive.                                           *
//* 11) Start directory: cfgOpt.srcPath[] : This is where source files are     *
//*                      located. If filename(s) are specified on the command  *
//*                      line ('-f' option), then those files are searched for *
//*                      in (or relative to) this directory. If no filenames   *
//*                      specified then filenames are scanned from:            *
//*                      a) current-working-directory by default               *
//*                      b) OR directory specified by clArgs.startDir[]        *
//* 12) Tag-field contents: As a convenience, certain fields may be specified  *
//*                      on the command line. Indicated by clArgs.fieldFlag.   *
//*                      a) If specified, then the values are written to the   *
//*                         corresponding fields of the FIRST filename.        *
//*                      b) Some of these fields will be applicable to all     *
//*                         source files, and so will be duplicated to the     *
//*                         corresponding fields for all source files.         *
//* 13) Preserve-source flag : 'cfgOpt.preserve'                               *
//*                      a) reset on instantiation                             *
//*                      b) modified from config file                          *
//*                      c) modified _again_ by clArgs.presSrc                 *
//* 14) Ignore-metadata flag : 'cfgOpt.ignore'                                 *
//*                      a) reset on instantiation                             *
//*                      b) modified from config file                          *
//*                      c) modified _again_ by clArgs.stripMeta               *
//* 15) Insert-external-image flag : 'cfgOpt.extImage'                         *
//*                      a) reset on instantiation                             *
//*                      b) no config-file option modifies this object         *
//*                      c) receives clArgs.image if specified                 *
//* 16) RTL language flags: 'cfgOpt.rtl' and 'cfgOpt.rtlf'                     *
//*                      a) both reset on instantiation                        *
//*                      b) 'rtl' set if RTL UI language (see AppDirection[])  *
//*                      c) 'rtlf' set by View Menu "RTL InvertFieldData"      *
//*                      d) See also the CTRL+R dynamic RTL field toggle.      *
//*                                                                            *
//*                                                                            *
//*                                                                            *
//******************************************************************************

Taggit::Taggit ( commArgs& clArgs )
{
   //* Initialize our data members *
   this->dPtr      = NULL ;            // dialog window not yet defined
   this->ic        = NULL ;
   this->termRows  = ZERO ;            // window size unknown
   this->termCols  = ZERO ;
   this->fwinRows  = ZERO ;
   this->fwinCols  = ZERO ;
   this->cs.scheme = ncbcCOLORS ;      // application default scheme
   this->suPos = { ZERO, ZERO } ;
   this->suVerbose = false ;
   //* this->cfgOpt: configuration options defaults set during instantiation.  *
   //* this->tData : source data reset during instantiation.                   *

   gString gsOut ;               // messages to user
   short configOk   = ERR ;      // 'OK' if configuration file read successfully
   bool tmpdirOk    = false,     // 'true' if temp-file directory verified
        colortermOk = false,     // 'true' if terminal supports color output
        termsizeOk  = false,     // 'true' if terminal window is large enough for application
        languageOk  = true ;     // 'false' if specified user-interface language not loaded

   //* Create a place to keep our temporary files *
   if ( (tmpdirOk = this->CreateTemppath ( gsOut )) != false )
      gsOut.copy( this->cfgOpt.tfPath, MAX_PATH ) ;   // save tempfile path

   //* Gather user's command-line arguments *
   this->GetCommandLineArgs ( clArgs ) ;

   //* Save a copy of application's path *
   gsOut = clArgs.appPath ;
   gsOut.copy( this->cfgOpt.appPath, MAX_PATH ) ;

   //* Save path of source directory *
   gsOut = clArgs.startDir ;
   gsOut.copy( this->cfgOpt.srcPath, MAX_PATH ) ;

   //* If an external image specified *
   if ( clArgs.insImage )
      this->cfgOpt.extImage = clArgs.image ;

   //* Initialize verbose-diagnostics flag *
   this->suVerbose = (clArgs.diagPause > 1) ? true : false ;

   //** If not 'Help' or 'Version' request, initialize the NCurses Engine.   **
   if ( (!(clArgs.helpFlag || clArgs.verFlag)) &&
        ((nc.StartNCursesEngine ()) == OK) )
   {
      //* Array of objects to hold diagnostic messages.          *
      //* Note that 'diagData' is a module-scope global variable.*
      diagData = new Diagnostics[diagdataCOUNT] ;
      chrono::duration<short>aWhile( 4 ) ;   // pause interval for user to read diag messages
      attr_t goodAttr  = nc.grB,             // color attributes for diagnostic messages
             badAttr   = nc.reB,
             nameAttr  = nc.br,
             cfgAttr   = nc.bl,
             pauseAttr = nc.grR,
             exitAttr  = nc.reR ;

      nc.SetCursorState ( nccvINVISIBLE ) ;  // hide the cursor
      nc.SetKeyProcState ( nckpRAW ) ;       // allow CTRL keys through unprocessed

      this->DrawTitle () ;                   // draw app title line to console
      this->DiagMsg ( "NCurses Engine initialized.", goodAttr ) ;

      //* Test whether color output engine was properly initialized *
      colortermOk = this->IsColorTerm ( goodAttr, badAttr ) ;

      //* Extended (verbose) diagnostic messages *
      if ( this->suVerbose )
      {
         this->DiagMsg ( "Application Path  : ", cfgAttr, false ) ;
         this->DiagMsg ( this->cfgOpt.appPath, cfgAttr ) ;
         this->DiagMsg ( "Tempfile Path     : ", cfgAttr, false ) ;
         this->DiagMsg ( this->cfgOpt.tfPath, cfgAttr ) ;
         this->DiagMsg ( "Source Path       : ", cfgAttr, false ) ;
         this->DiagMsg ( this->cfgOpt.srcPath, cfgAttr ) ;
         if ( *clArgs.fdArtist != NULLCHAR )
         {
            this->DiagMsg ( "Artist (comm args): ", cfgAttr, false ) ;
            this->DiagMsg ( clArgs.fdArtist, cfgAttr ) ;
         }
         if ( *clArgs.fdAlbum != NULLCHAR )
         {
            this->DiagMsg ( "Album (comm args) : ", cfgAttr, false ) ;
            this->DiagMsg ( clArgs.fdAlbum, cfgAttr ) ;
         }
      }

      //* If error establishing path for temp files *
      if ( ! tmpdirOk )
         this->DiagMsg ( "SYSTEM ERROR: Unable to establish "
                         "path for temporary files.", badAttr ) ;

      //* Read the configuration file and initialize 'cfgOpt' members *
      configOk = this->ReadConfigFile ( clArgs.cfgPath, cfgAttr, 
                                        goodAttr, badAttr, nameAttr ) ;

      //* Verify that the current locale supports UTF-8 character encoding.    *
      gString envLocale( nc.GetLocale() ),   // name of locale from environment
              usrLocale ;                    // user-specified locale
      short localeSetOk = nc.VerifyLocale (),// does env locale support UTF-8?
            altLocaleOk = OK ; // alt locale: support UTF-8?, included in system list?

      //* If a non-default locale was specified, either in the configuration   *
      //* file or on the command line, validate it now.                        *
      //*   a) Is the specified locale in the system list of supported locales?*
      //*   b) Does alt locale support UTF-8 encoding?                         *
      //* IMPORTANT NOTE: The alt locale must be set _BEFORE_ any I/O occurs.  *
      if ( (*this->cfgOpt.appLocale != NULLCHAR) || (*clArgs.altLocale != NULLCHAR) )
      {
         if ( *clArgs.altLocale != NULLCHAR ) // command-line option overrides config file
            usrLocale = clArgs.altLocale ;
         else
            usrLocale = this->cfgOpt.appLocale ;
         if ( (this->IsSupportedLocale ( usrLocale.ustr() )) &&
              ((nc.VerifyLocale ( usrLocale.ustr() )) == OK) )
         {
            usrLocale.copy( this->cfgOpt.appLocale, MAX_FNAME ) ;
            altLocaleOk = nc.SetLocale ( this->cfgOpt.appLocale ) ;
         }
         else     // specified locale is not valid, retain locale from environment
         {
            envLocale.copy( this->cfgOpt.appLocale, MAX_FNAME ) ;
            altLocaleOk = ERR ;
         }
      }
      else     // save name of locale from environment
         envLocale.copy( this->cfgOpt.appLocale, MAX_FNAME ) ;

      //* Report the results of locale settings for character-encoding *
      this->DiagMsg ( "Locale from environment: ", goodAttr, false ) ;
      gsOut.compose( "\"%S\"", envLocale.gstr() ) ;
      this->DiagMsg ( gsOut.ustr(), nameAttr, false ) ;
      if ( localeSetOk == OK )
         this->DiagMsg ( "  UTF-8 encoding support verified.", goodAttr ) ;
      else
         this->DiagMsg ( "  may not support UTF-8 encoding.", badAttr ) ;
      if ( usrLocale.gschars() > 1 ) // if user specified an alternate locale
      {
         this->DiagMsg ( "Alternate Locale       : ", goodAttr, false ) ;
         gsOut.compose( "\"%S\"", usrLocale.gstr() ) ;
         this->DiagMsg ( gsOut.ustr(), nameAttr, false ) ;
         if ( altLocaleOk == OK )
            this->DiagMsg ( "  UTF-8 encoding support verified.", goodAttr ) ;
         else
            this->DiagMsg ( "  may not support UTF-8 encoding. (not loaded)", badAttr ) ;
      }

      //* Set the user interface language.                                     *
      //* Note that an error condition here means that the language itself is  *
      //* valid, but that it doesn't match the previously-established locale.  *
      languageOk = ((this->Set_UI_Language ( clArgs.language )) == OK) ;
      this->DiagMsg ( "User interface language: ", goodAttr, false ) ;
      gsOut = alStrings[this->cfgOpt.appLanguage] ;
      this->DiagMsg ( gsOut.ustr(), nameAttr, (languageOk ? true : false) ) ;
      if ( ! languageOk )
         this->DiagMsg ( "  (mismatch with locale setting)", badAttr ) ;

      //* Get the size of our playground.*
      //* Complain if window too small.  *
      nc.ScreenDimensions ( this->termRows, this->termCols ) ;
      if ( (this->termRows >= MIN_APP_ROWS) && (this->termCols >= MIN_APP_COLS) )
         termsizeOk = true ;
      gsOut.compose( "Terminal window dimensions: %hd rows by %hd columns.", 
                     &this->termRows, &this->termCols ) ;
      this->DiagMsg ( gsOut.ustr(), termsizeOk ? goodAttr : badAttr ) ;
      if ( ! termsizeOk )
      {
         gsOut.compose( "Taggit requires a terminal window of at least %hd rows by %hd columns.", 
                        &MIN_APP_ROWS, &MIN_APP_COLS ) ; 
         this->DiagMsg ( gsOut.ustr(), badAttr ) ;
         this->DiagMsg ( "Please re-size the window and try again.", badAttr ) ;
      }

      //* Terminal dimensions verified, continue *
      else
      {
         //* Process any command-line args (overrides config file settings).*
         //*    ('locale' and 'language' have already been established.)    *
         //*    (source directory has also been established.)               *

         //* Initialize color scheme *
         if ( clArgs.colorScheme != ncbcCOLORS )
            this->cs.scheme = clArgs.colorScheme ;
         this->InitColorScheme () ;

         //* If command-line option specified for mouse setting *
         if ( clArgs.mouseFlag )
            this->cfgOpt.enableMouse = clArgs.enableMouse ;

         //* Sort option for item display *
         if ( clArgs.sortFlag )
            this->cfgOpt.sortBy = clArgs.sortOption ;

         //* Special-processing flags: *
         if ( clArgs.presSrc )
            this->cfgOpt.preserve = clArgs.presSrc ;
         if ( clArgs.stripMeta || !clArgs.dumpMeta )
            this->cfgOpt.ignore = clArgs.stripMeta ;
         if ( clArgs.allFields )
            Cmd_ActiveFields ( true, false ) ;
      }

      //* For major errors, wait for user to press a key.   *
      //* For minor errors, pause so user can read messages.*
      if ( ! termsizeOk || ! tmpdirOk || (configOk != OK) )
      {
         this->DiagMsg ( " Press Any Key To Exit. ", exitAttr ) ;
         this->DiagMsg () ;   // display the array of diagnostic messages
         nckPause() ;
      }
      else     // Ready, set, GO!
      {
         if ( this->suVerbose )
         {
            this->DiagMsg ( " Press any key to continue... ", pauseAttr ) ;
            this->DiagMsg () ;   // display the array of diagnostic messages
            nckPause() ;
         }

         this->DiagMsg ( "  Opening application dialog  ", pauseAttr ) ;
         this->DiagMsg () ;   // display the array of diagnostic messages
         if ( ! this->suVerbose && (! colortermOk || 
                (localeSetOk != OK) || (altLocaleOk != OK) || 
                (clArgs.diagPause > ZERO)) )
            this_thread::sleep_for( aWhile ) ;

         //* Delete the array of diagnostic messages.       *
         //* The only additional diag message that MIGHT be *
         //* displayed is if the dialog window doesn't open.*
         if ( diagData != NULL )
         { delete [] diagData ; diagData = NULL ; }


         odWarn odCode = odGOOD_DATA ;    // warning code
         short vCount = ZERO ;            // number of verified source files

         //* Source directory: current-working directory *
         //* OR user-specified directory. Verify that:   *
         //* a) is a directory                           *
         //* b) user has read/execute and write access.  *
         fmFType sdType = fmUNKNOWN_TYPE; // source dir file type
         bool sdRead = false,             // source dir access flags
              sdWrite = false ;
         this->SourceDirAccess ( sdType, sdRead, sdWrite ) ;

         //* If source filenames have been specified,        *
         //* convert filenames to full file specs.           *
         //* vCount should equal sfCount.                    *
         //* We must at least have read access to directory. *
         if ( (sdType == fmDIR_TYPE) && (sdRead != false) && 
              (this->tData.sfCount > ZERO) )
         {
            if ( (vCount = this->VerifyRawFilenames ()) != this->tData.sfCount )
               odCode = odFILE_ACCESS ;         // (fatal error)
            //* If directory is read-only, user *
            //* can view but not edit the data  *
            else if ( ! sdWrite )
               odCode = odREAD_ONLY ;           // (warning)
         }
         else
         {
            if ( sdType != fmDIR_TYPE )
               odCode = odNOT_DIR ;          // (fatal error)
            else if ( ! sdRead )
               odCode = odNO_ACCESS ;        // (fatal error)
            else
               odCode = odNO_SOURCE ;        // (fatal error)
         }

         //* Read metadata from source files. For each file which contains *
         //* metadata, initialize the corresponding display fields.        *
         if ( odCode == odGOOD_DATA || odCode == odREAD_ONLY )
         {
            if ( (this->LoadMetadata ()) < this->tData.sfCount )
               odCode = odSRC_FORMAT ;       // (fatal error)
         }

         //* Remove any duplicate source file records *
         if ( (this->RemoveDuplicates ()) > ZERO )
            odCode = odDUPLICATES ;          // (warning)

         //* Sort the data by initial sort criterion *
         this->Cmd_SortList () ;

         if ( ! clArgs.dumpMeta )
         {
            //* Interact with the user. *
            //* (Watch out, they byte!) *
            bool abortDialog = false ;
            if ( this->OpenDialog ( odCode, abortDialog ) )
            {
               //* If tag field values were specified, duplicate the *
               //* fields to all files in list. This will overwrite  *
               //* any existing metadata for those fields, and will  *
               //* set the edits-pending flag for all files.         *
               if ( odCode == odGOOD_DATA && clArgs.fieldFlag )
                  this->AutoDuplicate ( clArgs ) ;

               if ( ! abortDialog )
               {

                  //* Launch the main application threads. *
                  //* (Note that currently, Taggit is a)   *
                  //* (single-threaded application.    )   *

                  //* Interact with the user *
                  this->UserInterface () ;

                  //* Close the application dialog *
                  this->CloseDialog () ;

                  //* Recall all active threads *
                  //* (Nothing to do.)          *
               }
            }
         }
      }

      //* If not already done, delete the array of diagnostic messages *
      if ( diagData != NULL )
      { delete [] diagData ; diagData = NULL ; }

      nc.RestoreCursorState () ;             // make cursor visible
      nc.StopNCursesEngine () ;              // Deactivate the NCurses engine

      //* If command was to dump metadata to the stdout stream *
      if ( clArgs.dumpMeta )
      {
#if 0    // TEMP TEMP TEMP - TEST Ogg_Image
#if 1    // full
// This paragraph is taken from the Wikipedia description of "Base64 encoding. *
const UCHAR* testText =  (const UCHAR*)  // 269 bytes (excluding NULLCHAR)
   "Man is distinguished, not only by his reason, but by this singular passion from "
   "other animals, which is a lust of the mind, that by a perseverance of delight "
   "in the continued and indefatigable generation of knowledge, exceeds the short "
   "vehemence of any carnal pleasure." ; 
#elif 1  // 3 chars
const UCHAR* testText =  (const UCHAR*)"Man" ;
#elif 1  // 2 chars
const UCHAR* testText =  (const UCHAR*)"Ma" ;
#elif 1  // 1 char
const UCHAR* testText =  (const UCHAR*)"M" ;
#elif 1  // 20 chars
const UCHAR* testText =  (const UCHAR*)"any carnal pleasure." ;
#elif 1  // 19 chars
const UCHAR* testText =  (const UCHAR*)"any carnal pleasure" ;
#elif 1  // 18 chars
const UCHAR* testText =  (const UCHAR*)"any carnal pleasur" ;
#elif 1  // 17 chars
const UCHAR* testText =  (const UCHAR*)"any carnal pleasu" ;
#else    // 16 chars
const UCHAR* testText =  (const UCHAR*)"any carnal pleas" ;
#endif
gString gx((char*)testText) ;
short srcBytes = gx.utfbytes() - 1 ;
UCHAR outBuff[gsMAXBYTES] ;
UCHAR inBuff[gsMAXBYTES] ;
Ogg_Image oimg ;
//* TEMP */ --srcBytes ;
UINT32 encBytes = oimg.encodeImage( outBuff, testText, srcBytes ) ;
outBuff[encBytes] = NULLCHAR ;
wcout << L"ENCODED STREAM: (" << srcBytes << L")\n=====================\n" 
      << (char*)outBuff << endl ;

//* TEMP */ outBuff[--encBytes] = '?' ; outBuff[--encBytes] = '?' ; 
UINT32 decBytes = oimg.decodeImage( inBuff, outBuff, encBytes ) ;
inBuff[decBytes] = NULLCHAR ;
wcout << L"DECODED STREAM: (" << decBytes << L")\n=====================\n" 
      << (char*)inBuff << endl ;
#else    // PRODUCTION
         if ( (this->Cmd_TagSummary ( false, true )) != false )
         {
            gString gs( "%s/%s", this->cfgOpt.srcPath, mdDumpName ) ;
            wcout << L"\nTag summary written to: " << gs.gstr() << L'\n' << endl ;
         }
         else
            wcout << L"\nError writing metadata summary to file.\n" << endl ;
#endif   // PRODUCTION
      }
   }
   else
   {  //* Command-line help and app version request are written to stdout.*
      if ( clArgs.verFlag )
         this->DisplayVersion () ;
      else if ( clArgs.helpFlag )
         this->DisplayHelp () ;
      else
         wcout << "\n ERROR! Unable to initialize NCurses engine.\n" << endl ;
   }

}  //* End Taggit() *

//*************************
//*  GetCommandLineArgs   *
//*************************
//******************************************************************************
//* Capture user's command-line arguments.                                     *
//*                                                                            *
//* See 'DisplayHelp() method for a list of documented command-line options    *
//* and arguments.                                                             *
//*                                                                            *
//* Input  : commArgs class object (by reference)                              *
//*                                                                            *
//* Returns: 'false' if normal startup                                         *
//*          'true'  if request for help or invalid command-line argument      *
//******************************************************************************
//* Notes:                                                                     *
//* ------                                                                     *
//* If the user specifies the same parameter multiple times, it could lead to  *
//* incorrect startup parameters. For this reason, we allow the user to specify*
//* duplicates ONLY if the consequences produce valid, unambiguous data.       *
//* The following list details how a duplication of each parameter will be     *
//* handled. Note that it is not practical to test for all forms of user       *
//* stupidity (e.g. multiple instances of the same source file); however, we   *
//* cover the most critical problems and the documentation warns about the     *
//* issues regarding ambiguous and/or invalid input.                           *
//*                                                                            *
//*            DUPLICATES                                                      *
//* PARAMETER   ALLOWED                                                        *
//* =========  ==========                                                      *
//* --album     NO         subsequent instances ignored                        *
//* --artist    NO         subsequent instances ignored                        *
//* --version   YES        enable only                                         *
//* --help      YES        enable only                                         *
//* -f          YES        instances are additive                              *
//* -F          NO         subsequent instances ignored                        *
//* -a          YES        enable only                                         *
//* -d          NO         subsequent instances ignored                        *
//* -D          YES        enable only                                         *
//* -C          NO         subsequent instances ignored                        *
//* -i          YES        enable only                                         *
//* -I          NO         subsequent instances ignored                        *
//* -l          YES        clean overwrite                                     *
//* -L          YES        clean overwrite                                     *
//* -c          YES        clean overwrite                                     *
//* -m          YES        boolean                                             *
//* -s          NO         subsequent instances ignored                        *
//* -P          YES        enable only                                         *
//* -p          YES        enable only                                         *
//* -h, -H      YES        enable only                                         *
//*                                                                            *
//* Note that the source directory may only be specified once, so the          *
//* '-F' and '-d' options are mutually exclusive.                              *
//* Programmer's Note: Although technically both '-F' and '-d' options could   *
//* be specified _if_ file specified by '-F' did not include a directory name  *
//* ("PATH: dirname"); however, the test for this is too complex and the       *
//* chance of subsequent source-file validatation error too great.             *
//******************************************************************************

bool Taggit::GetCommandLineArgs ( commArgs& ca )
{
   #define DEBUG_GCA  (0)
   #define DEBUG_TAGS (0)

   const wchar_t EQUAL = L'=' ;
   bool startDirSpecified = false,     // prevents multiple instances of "-d" 
                                       //  or "-d"/"-F" combinaton
        configFileSpecified = false,   // prevents multiple instances of "-C"
        imageFileSpecified = false ;   // prevents multiple instances of "-I"

   //* Get the application executable's directory path. *
   //* Path string will be stored in ca.appPath[]       *
   this->gcaFindAppPath ( ca ) ;

   //* Get path of CWD for source files. Will be replaced *
   //* if alternate source directory specified.           *
   if ( (getcwd ( ca.startDir, MAX_PATH )) == NULL )
      ;  // (unlikely, it would mean that user lacks permission)


   //* If user provided command-line arguments *
   if ( ca.argCount > 1 )
   {
      gString gs ;               // text formatting
      const char* argPtr ;       // pointer to option argument
      short j = ZERO ;           // for multiple arguments in same token
      bool multiarg = false ;

      for ( short i = 1 ; (i < ca.argCount) || (multiarg != false) ; i++ )
      {
         if ( multiarg != false )   // if additional switches in same argument
            --i ;
         else
            j = ZERO ;

         //* If a command-line switch OR continuing previous switch argument *
         if ( ca.argList[i][j] == DASH || multiarg != false )
         {
            #if DEBUG_GCA != 0  // debugging only
            wcout << L"TOKEN:" << i << L" '" << &ca.argList[i][j] << L"'" << endl ;
            #endif   // DEBUG_GCA

            multiarg = false ;
            ++j ;

            if ( ca.argList[i][j] == DASH ) // (double dash)
            {  //* Long-form command switches *
               gs = ca.argList[i] ;

               //*********************
               //* Tag field options *
               //*********************
               if ( (gs.compare( L"--album", true, 7 )) == ZERO )
               {
                  if ( *ca.fdAlbum == NULLCHAR )
                  {
                     if ( ca.argList[i][7] == EQUAL )
                        argPtr = &ca.argList[i][8] ;
                     else if ( i < (ca.argCount - 1) )
                        argPtr = ca.argList[++i] ;
                     else
                     { ca.helpFlag = true ; break ; }    // invalid syntax for argument

                     gs = argPtr ;
                     gs.copy( ca.fdAlbum, MAX_PATH ) ;
                     ca.fieldFlag = true ;
                  }
                  continue ;     // finished with this argument
               }

               else if ( (gs.compare( L"--artist", true, 8 )) == ZERO )
               {
                  if ( *ca.fdArtist == NULLCHAR )
                  {
                     if ( ca.argList[i][8] == EQUAL )
                        argPtr = &ca.argList[i][9] ;
                     else if ( i < (ca.argCount - 1) )
                        argPtr = ca.argList[++i] ;
                     else
                     { ca.helpFlag = true ; break ; }    // invalid syntax for argument

                     gs = argPtr ;
                     gs.copy( ca.fdArtist, MAX_PATH ) ;
                     ca.fieldFlag = true ;
                  }
                  continue ;     // finished with this argument
               }

               //* Request for application version number *
               else if ( (gs.compare( L"--version" )) == ZERO )
                  ca.helpFlag = ca.verFlag = true ;

               //* Long-form request for Help *
               else if ( (gs.compare( L"--help" )) == ZERO )
                  ca.helpFlag = true ;

               else  // invalid argument
               { ca.helpFlag = true ; break ; }

               continue ;     // finished with this argument
            }  // (double dash)

            //** Short-form Arguments **
            char argLetter = ca.argList[i][j] ;

            //* If user specified source files to be processed *
            if ( argLetter == 'f' )
            {
               if ( ca.argList[i][++j] == EQUAL )
                  argPtr = &ca.argList[i][++j] ;
               else if ( i < (ca.argCount - 1) )
                  argPtr = ca.argList[++i] ;
               else
               { ca.helpFlag = true ; break ; }    // invalid syntax for argument

               //* Parse the list of source files and    *
               //* store them in our 'tData' data member *
               this->gcaParseFilelist ( argPtr ) ;
               continue ;     // finished with this argument
            }

            //** File containing a list of source files **
            else if ( argLetter == 'F' )
            {
               if ( ! startDirSpecified )
               {
                  if ( ca.argList[i][++j] == EQUAL )
                     argPtr = &ca.argList[i][++j] ;
                  else if ( i < (ca.argCount - 1) )
                     argPtr = ca.argList[++i] ;
                  else
                  { ca.helpFlag = true ; break ; }    // invalid syntax for argument

                  //* Read the specified file and    *
                  //* extract the list of filenames. *
                  if ( (this->gcaReadFilelist ( argPtr, ca.startDir )) )
                  {
                     startDirSpecified = true ;
                  }
                  else
                  { ca.helpFlag = true ; break ; }    // file not found
               }
               continue ;     // finished with this argument
            }

            //* Enable all tag display fields (overrides config-file settings) *
            else if ( argLetter == 'a' )
            {
               ca.allFields = true ;
            }

            //* If user has specified a source directory other than the CWD *
            else if ( argLetter == 'd' )
            {
               if ( ! startDirSpecified )
               {
                  if ( ca.argList[i][++j] == EQUAL )
                     argPtr = &ca.argList[i][++j] ;
                  else if ( i < (ca.argCount - 1) )
                     argPtr = ca.argList[++i] ;
                  else
                  { ca.helpFlag = true ; break ; }    // invalid syntax for argument

                  //* Read the specified directory and *
                  //* extract the list of filenames.   *
                  if ( (this->gcaScanSourceDir ( argPtr, ca.startDir )) != false )
                     startDirSpecified = true ;
               }
               continue ;     // finished with this argument
            }

            //* If user has requested that metadata be written to a file. *
            //* (This will have side-effects at a higher level.)          *
            else if ( argLetter == 'D' )
            {
               ca.dumpMeta = true ;
            }

            //* Alternate configuration file *
            else if ( argLetter == 'C' )
            {
               if ( ! configFileSpecified )
               {
                  if ( ca.argList[i][++j] == EQUAL )
                     argPtr = &ca.argList[i][++j] ;
                  else if ( i < (ca.argCount - 1) )
                     argPtr = ca.argList[++i] ;
                  else
                  { ca.helpFlag = true ; break ; }    // invalid syntax for argument

                  //* Capture the path and expand any environment variables, *
                  //* translate symbolic links and expand to full filespec.  *
                  gs = argPtr ;
                  if ( (this->Realpath ( gs )) != false )
                  {
                     gs.copy( ca.cfgPath, MAX_PATH ) ;
                     configFileSpecified = true ;
                  }
                  else
                  { ca.helpFlag = true ; break ; }    // target file not found
               }
               continue ;     // finished with this argument
            }

            //* Ignore (discard) all existing text metadata (and image) *
            else if ( argLetter == 'i' )
            {
               ca.stripMeta = true ;
            }

            //* Insert image into each audio file *
            else if ( argLetter == 'I' )
            {
               if ( ! imageFileSpecified )
               {
                  if ( ca.argList[i][++j] == EQUAL )
                     argPtr = &ca.argList[i][++j] ;
                  else if ( i < (ca.argCount - 1) )
                     argPtr = ca.argList[++i] ;
                  else
                  { ca.helpFlag = true ; break ; }    // invalid syntax for argument

                  //* Capture the path and expand any environment variables, *
                  //* translate symbolic links and expand to full filespec.  *
                  gs = argPtr ;
                  if ( (this->Realpath ( gs )) != false )
                  {
                     ca.image.reset() ;            // reset the receiving object
                     gs.copy( ca.image.picPath, MAX_PATH ) ;   // save filespec
                     ca.insImage = true ;
                     imageFileSpecified = true ;   // disallow duplicates
                  }
                  else
                  { ca.helpFlag = true ; break ; }    // target file not found
               }
               continue ;     // finished with this argument
            }

            //* User interface language *
            else if ( argLetter == 'l' )
            {
               if ( ca.argList[i][++j] == EQUAL )
                  argPtr = &ca.argList[i][++j] ;
               else if ( i < (ca.argCount - 1) )
                  argPtr = ca.argList[++i] ;
               else
               { ca.helpFlag = true ; break ; }    // invalid syntax for argument

               gs = argPtr ;
               if ( (gs.compare( "es", false, 2 )) == ZERO )      // "Español"
                  ca.language = esLang ;
               else if ( (gs.compare( "zh", false, 2 )) == ZERO ) // "Zhōngwén"
                  ca.language = zhLang ;
               else if ( ((gs.compare( "ti", false, 2 )) == ZERO) // "Tiếng Việt"
                      || ((gs.compare( "vi", false, 2 )) == ZERO) ) // (or "Viet")
                  ca.language = viLang ;
               else if ( (gs.compare( "en", false, 2 )) == ZERO ) // "English"
                  ca.language = enLang ;
               else
               {
                  ca.language = locLang ;
                  if ( (gs.compare( "loc", false, 3 )) != ZERO )  // "locale"
                     ca.helpFlag = true ;
                  break ;                 // invalid syntax for argument
               }
               continue ;     // finished with this argument
            }

            //* Locale for Input/Output *
            else if ( argLetter == 'L' )
            {
               if ( ca.argList[i][++j] == EQUAL )
                  argPtr = &ca.argList[i][++j] ;
               else if ( i < (ca.argCount - 1) )
                  argPtr = ca.argList[++i] ;
               else
               { ca.helpFlag = true ; break ; }    // invalid syntax for argument

               gs = argPtr ;
               gs.copy( ca.altLocale, MAX_FNAME ) ;
               continue ;     // finished with this argument
            }

            //* Color Scheme (window border color, etc.) *
            else if ( argLetter == 'c' )
            {
               if ( ca.argList[i][++j] == EQUAL )
                  argPtr = &ca.argList[i][++j] ;
               else if ( i < (ca.argCount - 1) )
                  argPtr = ca.argList[++i] ;
               else
               { ca.helpFlag = true ; break ; }    // invalid syntax for argument

               gs = argPtr ;
               if ( (gs.compare( "bla", false, 3 )) == ZERO )
                  ca.colorScheme = ncbcBK ;        // black
               else if ( (gs.compare( "red", false, 3 )) == ZERO )
                  ca.colorScheme = ncbcRE ;        // red
               else if ( (gs.compare( "gree", false, 4 )) == ZERO )
                  ca.colorScheme = ncbcGR ;        // green
               else if ( (gs.compare( "bro", false, 3 )) == ZERO )
                  ca.colorScheme = ncbcBR ;        // brown
               else if ( (gs.compare( "blu", false, 3 )) == ZERO )
                  ca.colorScheme = ncbcBL ;        // blue
               else if ( (gs.compare( "mag", false, 3 )) == ZERO )
                  ca.colorScheme = ncbcMA ;        // magenta
               else if ( (gs.compare( "cya", false, 3 )) == ZERO )
                  ca.colorScheme = ncbcCY ;        //cyan
               else if ( (gs.compare( "gray", false, 4 )) == ZERO )
                  ca.colorScheme = ncbcGY ;        // grey
               else if ( (gs.compare( "def", false, 3 )) == ZERO )
                  ca.colorScheme = ncbcCOLORS ;    // application default
               else
               { ca.helpFlag = true ; break ; }    // invalid syntax for argument
               continue ;     // finished with this argument
            }

            //* Enable or disable mouse support *
            else if ( argLetter == 'm' )
            {
               if ( ! ca.mouseFlag )
               {
                  if ( ca.argList[i][++j] == EQUAL )
                     argPtr = &ca.argList[i][++j] ;
                  else if ( i < (ca.argCount - 1) )
                     argPtr = ca.argList[++i] ;
                  else
                  { ca.helpFlag = true ; break ; }    // invalid syntax for argument
   
                  gs = argPtr ;
                  if ( (gs.compare( "di", false, 2 )) == ZERO )
                  { ca.enableMouse = false ; ca.mouseFlag = true ; }
                  else if ( (gs.compare( "en", false, 2 )) == ZERO )
                     ca.mouseFlag = ca.enableMouse = true ;
                  else
                  { ca.helpFlag = true ; break ; }    // invalid syntax for argument
               }
               continue ;     // finished with this argument
            }

            //* Select a sort option *
            else if ( argLetter == 's' )
            {
               if ( ! ca.sortFlag )
               {
                  if ( ca.argList[i][++j] == EQUAL )
                     argPtr = &ca.argList[i][++j] ;
                  else if ( i < (ca.argCount - 1) )
                     argPtr = ca.argList[++i] ;
                  else
                  { ca.helpFlag = true ; break ; }    // invalid syntax for argument

                  gs = argPtr ;
                  if ( (gs.compare( "filename", false, 4 )) == ZERO )
                  { ca.sortOption = sbFNAME ; ca.sortFlag = true ; }
                  else if ( (gs.compare( "title", false, 4 )) == ZERO )
                  { ca.sortOption = sbTITLE ; ca.sortFlag = true ; }
                  else if ( (gs.compare( "track", false, 4 )) == ZERO )
                  { ca.sortOption = sbTRACK ; ca.sortFlag = true ; }
                  else if ( (gs.compare( "album", false, 4 )) == ZERO )
                  { ca.sortOption = sbALBUM ; ca.sortFlag = true ; }
                  else if ( (gs.compare( "artist", false, 4 )) == ZERO )
                  { ca.sortOption = sbARTIST ; ca.sortFlag = true ; }
                  else if ( (gs.compare( "none", false, 4 )) == ZERO )
                  { ca.sortOption = sbNONE ; ca.sortFlag = true ; }
                  else
                  { ca.helpFlag = true ; break ; }    // invalid syntax for argument
               }
               continue ;     // finished with this argument
            }

            //* Preserve original source files. (write changes to copy of file)*
            else if ( argLetter == 'P' )
            {
               ca.presSrc = true ;
            }

            //* Pause after start-up sequence so user can see diagnostics *
            else if ( argLetter == 'p' )
            {
               char subchar = tolower ( ca.argList[i][j + 1] ) ;
               if ( subchar == 'v' )
               {
                  ca.diagPause = 2 ;
                  ++j ;
               }
               else
                  ca.diagPause = 1 ;
            }

            //* Else, is either 'h', 'H' or an invalid argument.*
            //* Either way, invoke help.                        *
            else
            {
               ca.helpFlag = true ;
               if ( argLetter != 'h' && argLetter != 'H' )
                  break ;
            }

            //* If separate tokens have been concatenated *
            if ( ca.argList[i][j + 1] != nckNULLCHAR )
               multiarg = true ;
         }
         else  // invalid argument, token does not begin with a DASH character
         { ca.helpFlag = true ; break ; }

      }     // for(;;)

      #if DEBUG_GCA != 0  // debugging only
      wcout << L"commArgs Settings" << endl ;
      gs.compose( L"%hd", &ca.argCount ) ;
       wcout << L"argCount   : " << gs.gstr() << endl ;
      for ( short i = ZERO ; i < ca.argCount ; i++ )
      {
         gs.compose( L"argList[%hd] : '%s'", &i, ca.argList[i] ) ;
         wcout << gs.gstr() << endl ;
      }
      gs.compose( L"appPath    : '%s'", ca.appPath ) ;
       wcout << gs.gstr() << endl ;
      gs.compose( L"cfgPath    : '%s'", ca.cfgPath ) ;
       wcout << gs.gstr() << endl ;
      gs.compose( L"startDir   : '%s'", ca.startDir ) ;
       wcout << gs.gstr() << endl ;
      gs.compose( L"colorScheme: %hd", &ca.colorScheme ) ;
       wcout << gs.gstr() << endl ;
      gs.compose( L"language   : %hd", &ca.language ) ;
       wcout << gs.gstr() << endl ;
      gs.compose( L"altLocale  : %s", &ca.altLocale ) ;
       wcout << gs.gstr() << endl ;
      gs.compose( L"fdArtist   : %s", ca.fdArtist ) ;
       wcout << gs.gstr() << endl ;
      gs.compose( L"fdAlbum    : %s", ca.fdAlbum ) ;
       wcout << gs.gstr() << endl ;
      gs.compose( L"fieldFlag  : %hhd", &ca.fieldFlag ) ;
       wcout << gs.gstr() << endl ;
      gs.compose( L"diagPause  : %hd", &ca.diagPause ) ;
       wcout << gs.gstr() << endl ;
      gs.compose( L"enableMouse: %hhd (%hhd)", &ca.enableMouse, &ca.mouseFlag ) ;
       wcout << gs.gstr() << endl ;
      gs.compose( L"sortOption : %hd (%hhd)", &ca.sortOption, &ca.sortFlag ) ;
       wcout << gs.gstr() << endl ;
      gs.compose( L"allFields  : %hhd", &ca.allFields ) ;
       wcout << gs.gstr() << endl ;
      gs.compose( L"presSrc    : %hhd", &ca.presSrc ) ;
       wcout << gs.gstr() << endl ;
      gs.compose( L"stripMeta  : %hhd", &ca.stripMeta ) ;
       wcout << gs.gstr() << endl ;
      gs.compose( L"insImage   : %hhd '%s'", &ca.insImage, ca.image.picPath ) ;
       wcout << gs.gstr() << endl ;
      gs.compose( L"verFlag    : %hhd", &ca.verFlag ) ;
       wcout << gs.gstr() << endl ;
      gs.compose( L"helpFlag   : %hhd", &ca.helpFlag ) ;
       wcout << gs.gstr() << endl ;
      wcout << L"\n Press Enter" << endl ;
      getchar () ;
      #endif   // DEBUG_GCA

      #if DEBUG_TAGS != 0  // debugging only
      if ( this->tData.sfCount > ZERO )
      {
         wcout << L"Source Files Specified:" << endl ;
         for ( short i = ZERO ; i < this->tData.sfCount ; ++i )
            wcout << L"  '" << this->tData.sf[i].sfPath << "'" << endl ;
      }
      if ( ca.fieldFlag )
      {
         wcout << L"Tag Fields Specified:" << endl ;

         for ( short i = ZERO ; i < (tfCOUNT - 1) ; ++i )
         {
            if ( *this->tData.sf[ZERO].sfTag.field[i] != NULLCHAR )
            {
               gs.compose( "  %s) '%S'", TextFrameID[i], this->tData.sf[ZERO].sfTag.field[i] ) ;
               wcout << gs.gstr() << endl ;
            }
         }
      }
      if ( this->tData.sfCount > ZERO || ca.fieldFlag )
      {
         wcout << L"\n Press Enter" << endl ;
         getchar () ;
      }
      #endif   // DEBUG_TAGS
   }

   //* If no individual source files specified, AND if a start *
   //* directory was not specified, then scan the current      *
   //* working directory for source audio files.               *
   if ( ! startDirSpecified && (this->tData.sfCount == ZERO) )
   {
      this->gcaScanSourceDir ( ca.startDir, ca.startDir ) ;
   }

   return ca.helpFlag ;

   #undef DEBUG_GCA
   #undef DEBUG_TAGS
}  //* End GetCommandLineArgs() *

//*************************
//*    gcaFindAppPath     *
//*************************
//******************************************************************************
//* Get the path to the application's executable file and store the string     *
//* in the commArgs class object.                                              *
//*                                                                            *
//* There are two ways to do this:                                             *
//* 1) The path/filename of the executable is passed to the application as     *
//*    argv[0].                                                                *
//* 2) The environment variable "_" (underscore) holds the most recently       *
//*    executed command (but see note below).                                  *
//*                                                                            *
//* The application path is used to locate the default configuration file      *
//* used during startup as well as the application Help files.                 *
//*                                                                            *
//*                                                                            *
//* Input  : commArgs class object (by reference)                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************
//* Programmer's Note: The path string that is received is not necessarily     *
//* the one that was executed.                                                 *
//* 1) If an environment variable such as $HOME is used in the invocation,     *
//*    that value is replaced by the VALUE of $HOME before it is passed to     *
//*    the application AND before it is stored in the "_" environment          *
//*    variable.                                                               *
//*    Example:  $HOME/Apps/Taggit/taggit                                      *
//*              is translated to:                                             *
//*              /home/sam/Apps/Taggit/taggit                                  *
//*                                                                            *
//* 2) Relative paths ARE NOT translated before being passed to the            *
//*    application and being stored at "_". Therefore, we need to convert      *
//*    the path we receive into an absolute path.                              *
//*                                                                            *
//* 3) We are not entirely sure what would happen if this application were     *
//*    invoked from within another program using the 'exec' family of          *
//*    functions, but we hope that argv[0] would still give us something we    *
//*    can work with. (In this case, the "_" would probably have the calling   *
//*    application's path/filename, so we can't use that.                      *
//*                                                                            *
//*                                                                            *
//******************************************************************************

void Taggit::gcaFindAppPath ( commArgs& ca )
{
   //* Save the actual, absolute path specification, replacing        *
   //* symbolic links and relative path specifications.               *
   gString rPath( ca.argList[0] ) ;
   if ( (this->Realpath ( rPath )) == false )
   {
      //* Unable to find the "real" path. This is likely   *
      //* because the executed binary or script was on the *
      //* path. Get the program that was actually executed.*
      char lbuff[MAX_PATH] ;
      short j = readlink ( "/proc/self/exe", lbuff, MAX_PATH ) ;
      lbuff[j] = NULLCHAR ;
      rPath = lbuff ;
   }

   //* Strip the application name from the end of the filespec.*
   short i = rPath.findlast( L'/' ) ;
   if ( i > ZERO )
      rPath.limitChars( i ) ;
   rPath.copy( ca.appPath, MAX_PATH ) ;

}  //* End gcaFindAppPath() *

//*************************
//*   gcaParseFilelist    *
//*************************
//******************************************************************************
//* Parse the concatenated filename list, perform environment substitions,     *
//* and store the results in the 'tdata' member.                               *
//*                                                                            *
//* Input  : argPtr  : text string containing concatenated list of filenames   *
//*                                                                            *
//* Returns: 'true'  if all specified filenames successfully captured          *
//*          'false' if argument could not be fully parsed                     *
//******************************************************************************
//* Notes:                                                                     *
//* -- This parsing algorithm is complex and slow, but fairly robust           *
//*    considering the kinds of garbage a user may throw at us.                *
//* -- The input is a single string which specifies one or more source         *
//*    filenames, separated by commas ( ',' ).                                 *
//* -- Filenames must be terminated with a valid filename extension followed   *
//*    either by a comma ',' the null terminator, OR if the user can't follow  *
//*    instructions, by a space ' '.                                           *
//* -- Extra whitespace between filenames is silently discarded.               *
//* -- User may have specified:                                                *
//*      a) a filename only, or                                                *
//*      b) a path/filename (with or without env variable substitution) or     *
//*      c) a symlink to the actual file.                                      *
//*    Caller will have to deal with decoding.                                 *
//*                                                                            *
//* Programmer's Note: There is a small logical hole here. If an extension is  *
//* longer than 4 characters, BUT is a match for the first 4, then we will get *
//* a false match on the filename extension.                                   *
//*  Example: 'Buggerface.ogaxyz' would be a false match with the              *
//*           '.oga' extension, and the filename will be saved as              *
//*           'Buggerface.oga' (trailing garbage discarded).                   *
//* Fortunately the chance of this is small since even users are smart enough  *
//* to know what an audio filename extension looks like.                       *
//******************************************************************************

bool Taggit::gcaParseFilelist ( const char* argPtr )
{
   #define DEBUG_PF (0)             // For debugging only

   gString gsf( argPtr ), gt ;
   wchar_t wPath[gsMAXCHARS] ;      // work buffer
   short   wi ;                     // work-buffer index
   MediafileType mft ;              // media format
   bool    status = true ;          // return value

   //* Convert to wide-character string *
   short wCnt ;
   const wchar_t* wPtr = gsf.gstr( wCnt ) ;

   #if DEBUG_PF != 0    // DEBUGGING ONLY
   wcout << L"PF BASE: '" << wPtr << L"'" << endl ;
   #endif               // DEBUG_PF

   for ( short indx = ZERO ; indx < wCnt ; )
   {
      //* Step over whitespace and commas to beginning of filename *
      while ( (wPtr[indx] == nckSPACE) || (wPtr[indx] == COMMA) )
         ++indx ;

      wi = ZERO ;
      while ( indx < wCnt )
      {
         if ( (wPtr[indx] != PERIOD) )       // base filename character
            wPath[wi++] = wPtr[indx++] ;
         else                                // possible extension deliminator
         {
            #if DEBUG_PF != 0    // DEBUGGING ONLY
            wcout << L"PF PERI: '" << &wPtr[indx] << L"'" << endl ;
            #endif               // DEBUG_PF

            //* Scan for the filename delimiter *
            bool isExt = true ;
            for ( short i = (indx + 1) ; (wPtr[i] != COMMA) && (wPtr[i] != NULLCHAR) ; ++i )
            {
               if ( wPtr[i] == PERIOD )
               { isExt = false ; break ; }
            }
            //* Test for valid, supported media filename extension *
            //* ( ".mp3" ".oda" ".ogg" ".wma" etc.)                *
            gt = &wPtr[indx] ;
            gt.limitChars( 4 ) ;

            #if DEBUG_PF != 0    // DEBUGGING ONLY
            wcout << L"    EXT: '" << gt.gstr() << L"'" << endl ;
            #endif               // DEBUG_PF

            if ( isExt && ((mft = this->MediafileTarget ( gt )) != mftNONE) )
            {
               gt.copy( &wPath[wi], 5 ) ;
               while ( (indx < wCnt) && (wPtr[indx] != COMMA) )
                  ++indx ;
               gt = wPath ;
               //* Note that any trailing spaces user may have inserted *
               //* will be stripped by this call. Note also that if the *
               //* normalization fails or file does not exist, then this*
               //* will be caught by caller's validation routine.       *
               this->Realpath ( gt ) ;
               gt.copy( this->tData.sf[this->tData.sfCount++].sfPath, gsMAXBYTES ) ;

               #if DEBUG_PF != 0    // DEBUGGING ONLY
               wcout << L"   NAME: '" << this->tData.sf[this->tData.sfCount - 1].sfPath
                                      << L"'" << endl ;
               #endif               // DEBUG_PF

               break ;     // filename extracted successfully
            }
            else        // not yet at extension OR unsupported filename extension
            {
               #if DEBUG_PF != 0    // DEBUGGING ONLY
               wcout << L"  isExt:" << isExt <<  L" '" << &wPtr[indx] << L"'" << endl ;
               #endif               // DEBUG_PF

               //* A period('.') found but it is not the beginning of a *
               //* filename extension, so continue the scan.            *
               if ( ! isExt )
                  wPath[wi++] = wPtr[indx++] ;

               //* Otherwise, this is an unsupported filename extension. This  *
               //* will be caught during caller's verification, so let it pass.*
               else
               {
                  gt.clear() ;
                  while ( (indx < wCnt) && (wPtr[indx] != COMMA) )
                     gt.append( wPtr[indx++] ) ;
                  gt.copy( &wPath[wi], gt.gschars() ) ;
                  gt = wPath ;
                  this->Realpath ( gt ) ;
                  gt.copy( this->tData.sf[this->tData.sfCount++].sfPath, gsMAXBYTES ) ;
   
                  #if DEBUG_PF != 0    // DEBUGGING ONLY
                  wcout << L"  UNSUP: '" << this->tData.sf[this->tData.sfCount - 1].sfPath
                                         << L"'" << endl ;
                  #endif               // DEBUG_PF
   
                  break ;     // filename extracted successfully
               }
            }
         }
      }
   }
   #if DEBUG_PF != 0    // DEBUGGING ONLY
   wcout << L"PF DONE, PRESS ENTER..." << endl ;
   getchar();
   #endif               // DEBUG_PF

   return status ;

}  //* End gcaParseFilelist() *

//*************************
//*    gcaReadFilelist    *
//*************************
//******************************************************************************
//* Read the specified file which contains the list of source files to be      *
//* scanned.                                                                   *
//*  -- Store the results in the 'tData.srcPath[]' array.                      *
//*  -- It is the caller's responsibility to convert the raw filename data     *
//*     to full filespecs.                                                     *
//*                                                                            *
//* Input  : argPtr  : name or path/name of file containing the list           *
//*          startDir: if file specifies a source directory, then this buffer  *
//*                    receives the fully-normalized path                      *
//*                    (an invalid path will be silently ignored)              *
//*                                                                            *
//* Returns: 'true'  if specified file found and contains at least one filename*
//*          'false' if file not found                                         *
//******************************************************************************

bool Taggit::gcaReadFilelist ( const char* argPtr, char* startDir )
{
   gString gsf( argPtr ), gt ;
   char    lineData[gsMAXBYTES] ;   // input buffer
   bool    status = true ;          // return value

   //* Normalize the filename to full path/filename spec *
   gsf = argPtr ;
   if ( (this->Realpath ( gsf )) != false )
   {
      ifstream ifs( gsf.ustr(), ifstream::in ) ;
      if ( ifs.is_open() )
      {
         bool  firstItem = true,    // path keyword test
               done = false ;       // loop control

         while ( ! done )
         {
            ifs.getline( lineData, gsMAXBYTES, NEWLINE ) ;
            if ( ifs.good() || (ifs.gcount() > ZERO) )
            {
               //* Skip comments and blank lines *
               if ( (lineData[ZERO] == '#') || (lineData[ZERO] == NULLCHAR) )
                  continue ;
               gt = lineData ;
               //* Step over leading whitespace *
               gt.strip( true, false ) ;
               if ( (gt.gschars()) == 1 )    // only whitespace found
                  continue ;

               //* If this is the first non-comment line, *
               //* test for path specification.           *
               if ( firstItem )
               {
                  firstItem = false ;
                  if ( (gt.find( L"PATH:", ZERO, true, 5 )) == ZERO )
                  {
                     gt.shiftChars( -5 ) ;                  // discard keyword
                     gt.strip() ;                           // discard whitespace
                     if ( gt.gschars() > 1 )                // if non-empty string
                     {
                        gString gsPath = gt ;
                        if ( (this->Realpath ( gsPath )) != false )  // success
                           gsPath.copy( startDir, gsMAXBYTES ) ;
                     }
                     continue ;
                  }
               }

               //* Normalize the filespec                                   *
               //* Note that if normalization fails or file does not exist, *
               //* this will be caught by caller's validation routine.      *
               this->Realpath ( gt ) ;
               gt.copy( this->tData.sf[this->tData.sfCount++].sfPath, MAX_FNAME ) ;
            }
            else     // end-of-file
               done = true ;
         }
         ifs.close() ;
      }
   }
   else     // specified file not found
   {
      status = false ;
   }

   return status ;

}  //* End gcaReadFilelist() *

//*************************
//*   gcaScanSourceDir    *
//*************************
//******************************************************************************
//* Scan the specified source directory for a list of source audio files.      *
//*                                                                            *
//*  -- Store the results in the 'tData.srcPath[]' array.                      *
//*  -- It is the caller's responsibility to convert the raw filename data     *
//*     to full filespecs.                                                     *
//*                                                                            *
//* Input  : argPtr  : name or path/directory name to scan                     *
//*          startDir: if source directory verified, then this buffer          *
//*                    receives the fully-normalized path                      *
//*                    (an invalid path will be silently ignored)              *
//*                                                                            *
//* Returns: 'true'  if specified directory found and if if contains at least  *
//*                  one source filename                                       *
//*          'false' if directory not found or if dir contains no source files *
//******************************************************************************
//* -- Read directory contents                                                 *
//* -- Extract filenames                                                       *
//* -- Test for supported audio format file extensions                         *
//*                                                                            *
//* Programmer's Note:                                                         *
//* There is a bug in readdir_r(). The documentation says that when there are  *
//* no more entries, the function will return a non-zero value. This is not    *
//* true currently (GCC 4.4.4 20100630, lib source version unknown).           *
//* readdir_r() returns a zero value even when param 3 returns NULL. The       *
//* work-around is to test parm 3 (sptr in this module) for NULL, indicating   *
//* that there are no more entries in the file.                                *
//*                                                                            *
//******************************************************************************

bool Taggit::gcaScanSourceDir ( const char* argPtr, char* startDir )
{
   gString gsf( argPtr ), gt ;      // text formatting
   bool status = false ;

   //* Normalize the directory name to full path/filename spec *
   gsf = argPtr ;
   if ( (this->Realpath ( gsf )) != false )
   {  //* Save the validated directory path *
      gsf.copy( startDir, gsMAXBYTES ) ;

      //* Open the directory *
      DIR*     dirPtr ;
      if ( (dirPtr = opendir ( startDir )) != NULL )
      {
         deStats  *destat ;
         bool     showFile ;

         while ( (destat = readdir64 ( dirPtr )) != NULL )
         {
            //* DO NOT include 'current dir' and 'parent dir' names but DO     *
            //* include 'hidden' files, i.e. filenames whose first character   *
            //* is a '.' (PERIOD).                                             *
            showFile = true ;
            if ( destat->d_name[ZERO] == PERIOD )
            {
               if (   destat->d_name[1] == NULLCHAR 
                   || (destat->d_name[1] == PERIOD && destat->d_name[2] == NULLCHAR) )
                  showFile = false ;
            }

            //* If a real file *
            if ( showFile )
            {
               //* Test filename extension for supported media format *
               gsf = destat->d_name ;
               if ( (this->MediafileTarget ( gsf )) != mftNONE )
               {
                  gsf.copy( this->tData.sf[this->tData.sfCount++].sfPath, gsMAXCHARS ) ;
                  status = true ;   // valid source file found
               }
            }
         }  // while()

         //* Close the directory file *
         closedir ( dirPtr ) ;
      }
   }
   return status ;

}  //* End gcaScanSourceDir() *

//*************************
//*       Realpath        *
//*************************
//******************************************************************************
//* Normalize a file specification string.                                     *
//*                                                                            *
//* Given a filename, relative path/filename or full path/filename, with or    *
//* without environmental variables or symbolic links, resolve to an absolute  *
//* path/filename string.                                                      *
//*                                                                            *
//* Programmer's Note: If user specifies a path or filename that contains      *
//* spaces, then he/she/it must enclose the string with quotations marks       *
//* (either single or double). Otherwise part of the string will be            *
//* interpreted as a separate, probably invalid argument.                      *
//*                                                                            *
//* Input  : rpath   : receives full filespec string                           *
//*                                                                            *
//* Returns: 'true'  if successful, rpath contains converted filespec          *
//*          'false' if parsing error, rpath is unchanged                      *
//******************************************************************************
//* NOTES:                                                                     *
//* ------                                                                     *
//* 'realpath' will follow symbolic links; however, it WILL NOT do tilde ('~') *
//*            or environment-variable expansion, so do these substitutions    *
//*            before calling.                                                 *
//*                                                                            *
//* 'wordexp'                                                                  *
//* The 'wordexp' function is a pretty cool, but watch out:                    *
//*  a) wordexp returns ZERO on success or WRDE_BADCHAR (2) if an invalid      *
//*     character is detected in the stream.                                   *
//*     - Note that an empty string will pass the scan, but then               *
//*       'wexp.we_wordc' will be ZERO.                                        *
//*  b) Dynamic memory allocation happens, so remember to free it.             *
//*     - If a bad character in the stream, then freeing the dynamic           *
//*       allocation will cause a segmentation fault. This is a Standard       *
//*       Library bug, so the work-around is to call 'wordfree' ONLY if        *
//*      'wordexp' returns success.                                            *
//*  c) SPACE characters delimit the parsing, so if the path contains spaces,  *
//*     then we must concatenate the resulting substrings, reinserting the     *
//*     space characters. Leading and trailing spaces are ignored.             *
//*     (We assume that a path will never contain a TAB or NEWLINE character.) *
//*  d) wordexp will choke on the following characters in the stream:          *
//*             & | ; < >  \n     (unless they are quoted)                     *
//*     - Parentheses and braces should appear ONLY as part of a token to be   *
//*       expanded (or if they are quoted).                                    *
//*     - single-quotes ''' are seen as delimiters unless they are quoted.     *
//*     - double-quotes '"' are also seen as delimiters unless they are        *
//*       quoted; however, the use of double-quotes is further complicated by  *
//*       the fact that string literals are always delimited by double-quotes, *
//*       so a double-quote within a set of double-quotes must be SENT to us   *
//*       as quoted in order for it to arrive as the double-quote character.   *
//*  e) Wildcard characters '*' and '?' are tricky, but '?' especially is a    *
//*     common filename character.                                             *
//*  f) The tokens we are most likely to see are '${HOME}' and '~'.            *
//*     These are both expanded as '/home/sam' or the equivalent.              *
//*  g) The most likely special characters (especially for song titles, book   *
//*     title and similar) that represent themselves are:                      *
//*          single-quote '''  question mark '?' and parentheses '(' ')'       *
//*  h) Note that the characters: '$'  '{'  '}'  '`'  are reserved by the      *
//*     shell for expansion of environment variables, and therefore should     *
//*     not be used as filename characters; therefore, we do not quote them    *
//*     here.                                                                  *
//******************************************************************************

bool Taggit::Realpath ( gString& rpath )
{
   gString gstmp = rpath,        // working copy of source
           relpath ;
   short wchars, index = ZERO ;
   const wchar_t* wptr = gstmp.gstr( wchars ) ;
   bool status = true ;          // return value

   //* Quote (escape) the set of special characters so     *
   //* wordexp() will process them without special meaning.*
   while ( index < wchars )
   {
      if ( (wptr[index] == L'(') || (wptr[index] == L')') ||
           (wptr[index] == L'"') || (wptr[index] == L'\'') ||
           (wptr[index] == L'?') || (wptr[index] == L'&') ||
           (wptr[index] == L'<') || (wptr[index] == L'>') ||
           (wptr[index] == L';') || (wptr[index] == L'|')
         )
      {
         if ( (index == ZERO) || (wptr[index - 1] != L'\\') )
         {
            gstmp.insert( L'\\', index ) ;
            ++wchars ;
            index += 2 ;
         }
         else
            ++index ;
      }
      else
         ++index ;
   }

   //* Perform environment-variable and tilde substitutions. (see notes above) *
   wordexp_t wexp ;              // target structure
   if ( (wordexp ( gstmp.ustr(), &wexp, ZERO )) == ZERO )
   {
      if ( wexp.we_wordc > ZERO )   // if we have at least one element
      {
         relpath.clear() ;
         for ( UINT i = ZERO ; i < wexp.we_wordc ; )
         {
            relpath.append( wexp.we_wordv[i++] ) ;
            if ( i < wexp.we_wordc )      // re-insert stripped spaces (see note)
               relpath.append( L' ' ) ;
         }
      }
      wordfree ( &wexp ) ;
   }
   else
      status = false ;

   //* Expand relative paths and paths containing symbolic links.*
   if ( status != false )
   {
      char rtmp[MAX_PATH] ;      // temp buffer

      if ( (realpath ( relpath.ustr(), rtmp )) != NULL )
         rpath = rtmp ;
      else
         status = false ;
   }

   return status ;

}  //* End Realpath() *

//*************************
//*     TargetExists      *
//*************************
//******************************************************************************
//* Determine whether specified file exists and return its filetype.           *
//*                                                                            *
//* Input  : trgPath : filespec of target file                                 *
//*          fType   : (by reference)                                          *
//*                  : if target exists, initialized to target fileType        *
//*                    else fmUNKNOWN_TYPE                                     *
//*                                                                            *
//* Returns: 'true' if target exists, else 'false'                             *
//******************************************************************************
//* Programmer's Note: This method is grossly inefficient, but this            *
//* application requires simplicity of design over speed.                      *
//******************************************************************************

bool Taggit::TargetExists ( const gString& trgPath, fmFType& fType )
{
   tnFName fs ;               // file stats
   bool exists = false ;      // return value

   if ( (this->GetFileStats ( trgPath, fs )) == OK )
   {
      fType = fs.fType ;
      exists = true ;
   }
   return exists ;

}  //* End TargetExists() *

//*************************
//*     GetFileStats      *
//*************************
//******************************************************************************
//* Perform a 'stat' ('lstat') on the file specified by the full path string.  *
//*                                                                            *
//* Input  : trgPath: full path/filename specification                         *
//*          fStats : (by reference) receives the stat data for the file       *
//*                                                                            *
//* Returns: OK if successful, all data members of fStats will be initialized  *
//*          Else returns system 'errno' value                                 *
//******************************************************************************

short Taggit::GetFileStats ( const gString& trgPath, tnFName& fStats )
{
   short success ;         // return value

   //* Provides an orderly crash in case of failure *
   fStats.modTime.reset() ;
   fStats.fBytes = ZERO ;
   fStats.fType = fmUNKNOWN_TYPE ;
   fStats.readAcc = fStats.writeAcc = false ;

   if ( (success = lstat64 ( trgPath.ustr(), &fStats.rawStats )) == OK )
   {
      //* Extract filename from caller's string *
      short indx = trgPath.findlast( L'/' ) ;
      if ( indx < ZERO )   indx = ZERO ;
      else                 ++indx ;
      gString fName( &trgPath.gstr()[indx] ) ;
      fName.copy( fStats.fName, MAX_FNAME ) ;

      fStats.fBytes = fStats.rawStats.st_size ;    // number of bytes in file

      //* Decode the filetype *
      UINT mode = fStats.rawStats.st_mode ;
      if ( (S_ISREG(mode)) != false )        fStats.fType = fmREG_TYPE ;
      else if ( (S_ISDIR(mode)) != false )   fStats.fType = fmDIR_TYPE ;
      else if ( (S_ISLNK(mode)) != false )   fStats.fType = fmLINK_TYPE ;
      else if ( (S_ISCHR(mode)) != false )   fStats.fType = fmCHDEV_TYPE ;
      else if ( (S_ISBLK(mode)) != false )   fStats.fType = fmBKDEV_TYPE ;
      else if ( (S_ISFIFO(mode)) != false )  fStats.fType = fmFIFO_TYPE ;
      else if ( (S_ISSOCK(mode)) != false )  fStats.fType = fmSOCK_TYPE ;

      //* Decode the mod time *
      fStats.modTime.epoch = (int64_t)fStats.rawStats.st_mtime ;
      fStats.modTime.sysepoch = (time_t)fStats.rawStats.st_mtime ; // (possible narrowing)
      Tm bdt ;       // receives broken-down time
      if ( (localtime_r ( &fStats.modTime.sysepoch, &bdt )) != NULL )
      {
         //* Translate to localTime format (timezone data not decoded) *
         fStats.modTime.date    = bdt.tm_mday ;         // today's date
         fStats.modTime.month   = bdt.tm_mon + 1 ;      // month
         fStats.modTime.year    = bdt.tm_year + 1900 ;  // year
         fStats.modTime.hours   = bdt.tm_hour ;         // hour
         fStats.modTime.minutes = bdt.tm_min ;          // minutes
         fStats.modTime.seconds = bdt.tm_sec ;          // seconds
         fStats.modTime.day     = bdt.tm_wday ;         // day-of-week (0 == Sunday)
         fStats.modTime.julian  = bdt.tm_yday ;         // Julian date (0 == Jan.01)
      }

      //* Initialize the read-access and write-access flags *
      fStats.readAcc = bool((access ( trgPath.ustr(), R_OK )) == ZERO) ;
      if ( fStats.readAcc != false && fStats.fType == fmDIR_TYPE )
         fStats.readAcc = bool((access ( trgPath.ustr(), X_OK )) == ZERO) ;
      fStats.writeAcc = bool((access ( trgPath.ustr(), W_OK )) == ZERO) ;
   }
   else
      success = errno ;

   return success ;

}  //* End GetFileStats() *

//*************************
//*    SourceDirAccess    *
//*************************
//******************************************************************************
//* Stat the source directory to verify file type + r/w/x access.              *
//*                                                                            *
//* Input  : sdType : (by reference) receives file type                        *
//*          sdRead : (by reference) receives read/exec access flag            *
//*          sdWrite: (by reference) receives write access flag                *
//*                                                                            *
//* Returns: OK  if user has at least the ability to scan and read directory   *
//*          ERR if user lacks read/execute access OR target is not a directory*
//******************************************************************************
//* Notes:                                                                     *
//* -- For 'directory' files, read access is ACTUALLY read/execute access      *
//*    because BOTH are needed in order to scan the directory contents.        *
//*                                                                            *
//******************************************************************************

short Taggit::SourceDirAccess ( fmFType& sdType, bool& sdRead, bool& sdWrite )
{
   tnFName fs ;                     // file stats
   short status = ERR ;             // return value

   //* Initialize caller's variables *
   sdType = fmUNKNOWN_TYPE ;
   sdRead = sdWrite = false ;

   //* 'lstat' the target to see if it exists *
   if ( (this->GetFileStats ( this->cfgOpt.srcPath, fs )) == OK )
   {
      sdType  = fs.fType ;
      sdRead  = fs.readAcc ;
      sdWrite = fs.writeAcc ;
   }
   return status ;

}  //* End SourceDirAccess() *

//*************************
//*     LoadMetadata      *
//*************************
//******************************************************************************
//* Read each source file and store the metadata in our data members.          *
//*                                                                            *
//* If the 'ignore' flag is set, then clear all the display fields.            *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: number of source files successfully scanned and loaded            *
//******************************************************************************

short Taggit::LoadMetadata ( void )
{
   short validFiles = ZERO ;

   for ( short fIndex = ZERO ; fIndex < this->tData.sfCount ; ++fIndex )
   {
      if ( (this->LoadMetadata ( fIndex )) != false )
      {
         ++validFiles ;

         if ( this->cfgOpt.ignore && (this->tData.sf[fIndex].sfTag.contains_data()) )
         {
            this->tData.sf[fIndex].sfTag.clear_data() ;
            this->tData.sf[fIndex].sfTag.tfMod = true ;
         }
      }
   }
   return validFiles ;

}  //* End LoadMetadata() *

//*************************
//*     LoadMetadata      *
//*************************
//******************************************************************************
//* Read the specified source file and store its metadata in the corresponding *
//* data members.                                                              *
//*                                                                            *
//* Input  : fIndex : index at which to store the extracted data.              *
//*                                                                            *
//* Returns: 'true'  if data loaded successfully                               *
//*          'false' if file-not-found, invalid-format, access or other error  *
//******************************************************************************

bool Taggit::LoadMetadata ( short fIndex )
{
   gString gsfmt ;
   bool status = false ;

   if ( this->tData.sf[fIndex].sfType == mftMP3 )
      status = this->ExtractMetadata_MP3 ( fIndex ) ;
   else if ( this->tData.sf[fIndex].sfType == mftOGG )
      status = this->ExtractMetadata_OGG ( fIndex ) ;
   else if ( this->tData.sf[fIndex].sfType == mftASF )
      status = this->ExtractMetadata_ASF ( fIndex ) ;
   else     // invalid media type (unlikely)
      status = false ;

   //* Special processing for certain fields *
   if ( status != false )
   {
      //* Track field: format the entry consistently according to the MP3 *
      //* specification. This is necessary because the user may want to   *
      //* sort on this field. If field contains garbage, sort will fail.  *
      gsfmt = this->tData.sf[fIndex].sfTag.field[tfTrck] ;
      if ( (this->uiemFormatTrack ( gsfmt )) != false )
         gsfmt.copy( this->tData.sf[fIndex].sfTag.field[tfTrck], gsMAXCHARS ) ;
      gsfmt = this->tData.sf[fIndex].sfTag.field[tfTyer] ;
      if ( (this->uiemFormatYear ( gsfmt )) != false )
         gsfmt.copy( this->tData.sf[fIndex].sfTag.field[tfTyer], gsMAXCHARS ) ;

   }

   //* Invalid audio format, access error or other major error.  *
   //* Display an error message in the 'Title' field and clear   *
   //* the 'sfName' field which will cause the filename to be    *
   //* displayed in a contrasting color indicating a major error.*
   else
   {
      const char* msgTemplate[4] = 
      {
         "Not valid %s format!",
         "¡No es un formato %s válido!",
         "不是有效的 %s 格式！",
         "Không phải là một định dạng %s hợp lệ!",
      } ;
      gsfmt.compose( msgTemplate[this->cfgOpt.appLanguage], 
                     this->tData.sf[fIndex].sfType == mftMP3 ? "MP3" : 
                     this->tData.sf[fIndex].sfType == mftOGG ? "OGG" : "WMA" ) ;
      gsfmt.copy( this->tData.sf[fIndex].sfTag.field[tfTit2], gsMAXCHARS ) ;
      this->tData.sf[fIndex].sfName[0] = NULLCHAR ;
   }

   return status ;

}  //* End LoadMetadata() *

//*************************
//*    MediafileTarget    *
//*************************
//******************************************************************************
//* Compare the filename extension of the specified file against the list of   *
//* supported media files from which we can extract the file's metadata.       *
//*                                                                            *
//* Input  : trgPath  : filespec of file to be tested.                         *
//*                                                                            *
//* Returns: member of enum MediafileType                                      *
//******************************************************************************

MediafileType Taggit::MediafileTarget ( const gString& trgPath )
{
   MediafileType mftype = mftNONE ;    // return value
   short eIndex = trgPath.findlast( L'.' ) ;
   if ( eIndex >= ZERO )
   {
      gString trgExt( &trgPath.gstr()[eIndex + 1] ) ;

      if ( (trgExt.compare( L"mp3" )) == ZERO )
         mftype = mftMP3 ;
      else if (    ((trgExt.compare( L"oga" )) == ZERO) 
                || ((trgExt.compare( L"ogg" )) == ZERO) )
         mftype = mftOGG ;
      else if ( (trgExt.compare( L"wma" )) == ZERO )
         mftype = mftASF ;
   }
   return mftype ;

}  //* End MediafileTarget() *

//*************************
//*  VerifyRawFilenames   *
//*************************
//******************************************************************************
//* Convert raw filename data in tData.srcPath[] to:                           *
//*    a) full filespecs in tData.srcPath[] array, and                         *
//*    b) filenames in tData.srcName[] array                                   *
//*                                                                            *
//* A valid source file is one which:                                          *
//*    a) actually exists                                                      *
//*    b) is a 'regular' file                                                  *
//*    c) is a supported audio format (according to filename extension)        *
//*    d) user has read/write access                                           *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: number of verified filenames                                      *
//*          (if != sfCount, then caller should abort)                         *
//******************************************************************************
//* Notes:                                                                     *
//* 1) Environment-variable substitution has already been performed            *
//* 2) Symlinks will be followed to the actual target file.                    *
//* 3) For actually existing filenames, if we limit the input names to the     *
//*    filename only, then prepending the 'cfgOpt.srcPath' to the filename     *
//*    will generate an accurate filespec.                                     *
//* 4) If the input names are expressed as a _relative path_ to cfgOpt.srcPath,*
//*    then prepending the 'cfgOpt.srcPath' will also generate an accurate     *
//*    filespec.                                                               *
//* 5) If the input names are full filespecs (string begins with a '/'), then  *
//*    'cfgOpt.srcPath' is not prepended, and we hope that the user is an      *
//*    accurate typist.                                                        *
//* 6) If the input names are expressed as relative to the CWD, _BUT_ an       *
//*    alternate 'cfgOpt.srcPath' has been specified as a command-line option, *
//*    then the files will not be found. This is a user error, but users are   *
//*    simple creatures, so we compensate for this error by testing relative   *
//*    path specifications against BOTH the CWD and the specified target       *
//*    directory. Only if source is found in neither place do we declare the   *
//*    file as not found.                                                      *
//* 7) At this level, we allow the list to contain duplicate filespecs;        *
//*    however, caller should elimiate the duplicates before allowing user     *
//*    access to the data. See RemoveDuplicates().                             *
//*                                                                            *
//******************************************************************************

short Taggit::VerifyRawFilenames ( void )
{
   #define DEBUG_VRF (0)   // for debugging only (NCurses Engine IS running)
   #if DEBUG_VRF != 0   // DEBUGGING ONLY
   gString gs( "** VerifyRawFilenames() **" ) ;
   winPos wp( 0, 0 ) ;
   nc.ClearScreen () ;
   nc.WriteString ( wp.ypos++, wp.xpos, gs.gstr(), nc.bw | ncuATTR ) ;
   #endif               // DEBUG_VRF

   gString rawPath, gsPath, gsName ;
   MediafileType mft ;              // media format
   tnFName fStats ;                 // file stats
   short fIndex,                    // index to filename
         fConv = ZERO ;             // return value, count of verified filenames
   bool goodSpec = false ;          // 'true' if file exists

   for ( short i = ZERO ; i < this->tData.sfCount ; ++i )
   {
      //* Test for full filespec or path relative to CWD *
      gsPath = this->tData.sf[i].sfPath ;
      if ( !(goodSpec = this->Realpath ( gsPath )) )
      {  //* Test for path relative to specified source path *
         gsPath.compose( "%s/%s", this->cfgOpt.srcPath, this->tData.sf[i].sfPath ) ;
         goodSpec = this->Realpath ( gsPath ) ;
      }
      if ( goodSpec )
      {
         //* Save full filespec *
         gsPath.copy( this->tData.sf[i].sfPath, gsMAXBYTES ) ;

         #if DEBUG_VRF != 0   // DEBUGGING ONLY
         gs.compose( "srcPath[%02hd] '%s'", &i, this->tData.sf[i].sfPath ) ;
         nc.WriteString ( wp.ypos++, wp.xpos, gs.gstr(), nc.bw ) ;
         #endif               // DEBUG_VRF

         //* Extract filename *
         if ( (fIndex = ((gsPath.findlast( L'/' )) + 1)) > ZERO )
         {
            gsName = &gsPath.gstr()[fIndex] ;

            #if DEBUG_VRF != 0   // DEBUGGING ONLY
            gs.compose( "   Name: '%S'", gsName.gstr() ) ;
            wp = nc.WriteString ( wp, gs.gstr(), nc.bw ) ;
            #endif               // DEBUG_VRF

            //* Get filetype and user access *
            this->GetFileStats ( gsPath, fStats ) ;
            if ( (fStats.fType == fmREG_TYPE) && fStats.readAcc && fStats.writeAcc )
            {
               //* Validate as supported audio-file format *
               if ( (mft = this->MediafileTarget ( gsName )) != mftNONE )
               {
                  gsName.copy( this->tData.sf[i].sfName, MAX_FNAME ) ;
                  this->tData.sf[i].sfType = mft ;
                  ++fConv ;

                  #if DEBUG_VRF != 0   // DEBUGGING ONLY
                  nc.WriteString ( wp.ypos++, wp.xpos, L"   OK", nc.gr ) ;
                  #endif               // DEBUG_VRF
               }

               #if DEBUG_VRF != 0   // DEBUGGING ONLY
               else
               {
                  gs.compose( "   UNSUPP:%hd", &mft ) ;
                  nc.WriteString ( wp.ypos++, wp.xpos, gs.gstr(), nc.re ) ;
               }
               #endif               // DEBUG_VRF
            }

            #if DEBUG_VRF != 0   // DEBUGGING ONLY
            else
            {
               gs.compose( "   ACCESS: type:%hd read:%hhd write:%hhd", 
                           &fStats.fType, &fStats.readAcc, &fStats.writeAcc ) ;
               nc.WriteString ( wp.ypos++, wp.xpos, gs.gstr(), nc.re ) ;
            }
            wp.xpos = ZERO ;
            #endif               // DEBUG_VRF
         }
      }

      #if DEBUG_VRF != 0   // DEBUGGING ONLY
      else
      {
         gs.compose( "NOT FOUND : '%S'", gsPath.gstr() ) ;
         nc.WriteString ( wp.ypos++, wp.xpos, gs.gstr(), nc.re ) ;
      }
      #endif               // DEBUG_VRF
   }

   #if DEBUG_VRF != 0   // DEBUGGING ONLY
   nc.WriteString ( wp, L" PRESS A KEY... ", nc.reR ) ;
   nckPause();
   #endif               // DEBUG_VRF

   return fConv ;

}  //* End VerifyRawFilenames() *

//*************************
//*   RemoveDuplicates    *
//*************************
//******************************************************************************
//* Remove any source records for duplicated filespecs.                        *
//* (Duplicate file _names_ are acceptable.)                                   *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: number of duplicate records removed                               *
//******************************************************************************
//* Notes:                                                                     *
//* ======                                                                     *
//* We allow the user to specify duplicate references to the same source file; *
//* however, for reasons of data integrity we cannot allow these duplicates    *
//* during write to target. For this reason, we remove any duplicate records   *
//* and warn the user.                                                         *
//*                                                                            *
//* The problem with duplicate records is that if data for multiple source     *
//* files are written to the same target filename, the secondary reference     *
//* to the same source/target file will destroy the metadata from the first    *
//* instance. There is also the question of saving one edited record under a   *
//* different target name which will cause the source file of the secondary    *
//* record to be missing, which would very likely cause a processing error and *
//* possibly crash the application.                                            *
//*                                                                            *
//* The only way duplicate records would work is if the user takes great care  *
//* in designating different targets for the duplicated source records.        *
//* However, we all know that users have neither the intelligence, the         *
//* imagination, nor frankly the mental focus necessary to handle this         *
//* situation on their own.                                                    *
//*   (And don't even get us started on a user explicitly directing )          *
//*   (data for different source files to a single target.          )          *
//*                                                                            *
//* While in general, we fully support a user's right to shoot him/her/itself  *
//* in the foot, the downsize is that in this case, we will probably get       *
//* blamed for it.                                                             *
//* Transparency wins out here: duplicate source files will almost certainly   *
//* cause hidden (or obvious) errors. Eliminating the cause of such errors     *
//* AND then reporting what we have done is the better choice.                 *
//******************************************************************************

short Taggit::RemoveDuplicates ( void )
{
   gString gs ;                           // text analysis
   short sfCount = this->tData.sfCount,   // initial record count
         dCount = ZERO ;                  // return value

   for ( short i = ZERO ; i < sfCount ; ++i )
   {
      gs = this->tData.sf[i].sfPath ;     // filespec to be tested

      //* Compare the current filespec to previously stored filespecs. *
      for ( short j = ZERO ; j < i ; ++j )
      {
         if ( (gs.compare( this->tData.sf[j].sfPath )) == ZERO )
         {
            ++dCount ;                    // duplicate found

            //* Shift the remaining records upward by one.*
            for ( short k = j + 1 ; k < sfCount ; ++k, ++j )
               this->tData.sf[j] = this->tData.sf[k] ;
            --sfCount ;                   // array is shorter by one
            break ;
         }
      }
   }
   this->tData.sfCount = sfCount ;        // update the source-record counter
   return dCount ;

}  //* End RemoveDuplicates() *

//*************************
//*     AutoDuplicate     *
//*************************
//******************************************************************************
//* For field data specified as command-line options, duplicate those fields   *
//* through all source files.                                                  *
//*                                                                            *
//* Because metadata have not yet been read from the source files, only the    *
//* fields to be duplicated will contain data. Currently these are:            *
//*    --album  (tfTalb)                                                       *
//*    --artist (tfTpe1)                                                       *
//*                                                                            *
//*                                                                            *
//* Input  : ca      : object containing command-line arguments                *
//*                    clArtist (tfTpe1)                                       *
//*                    clAlbum  (tfTalb)                                       *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void Taggit::AutoDuplicate ( const commArgs& ca )
{
   gString gs ;

   for ( short i = ZERO ; i < this->tData.sfCount ; ++i )
   {
      if ( *ca.fdArtist != NULLCHAR )
      {
         gs = ca.fdArtist ;
         gs.copy( this->tData.sf[i].sfTag.field[tfTpe1], gsMAXCHARS ) ;
         this->tData.sf[i].sfTag.tfMod = true ;
      }
      if ( *ca.fdAlbum != NULLCHAR )
      {
         gs = ca.fdAlbum ;
         gs.copy( this->tData.sf[i].sfTag.field[tfTalb], gsMAXCHARS ) ;
         this->tData.sf[i].sfTag.tfMod = true ;
      }
   }

}  //* End AutoDuplicate() *

//*************************
//*     DrawTitle         *
//*************************
//******************************************************************************
//* Draw the application title in the main terminal window.                    *
//* This is a temporary title for display of startup diagnostics, or in case   *
//* NcDialog interface cannot open.                                            *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void Taggit::DrawTitle ( void )
{
   //* Construct and display the main application title line *
   gString  gs( "%s%s (c)%s %s  (diagnostic information) ", 
                AppTitle1, AppVersion, copyrightYears, AppTitle2 ) ;
   this->DiagMsg ( gs.ustr(), (nc.bw | ncuATTR) ) ;

}  //* End DrawTitle()

//*************************
//*       DiagMsg         *
//*************************
//******************************************************************************
//* Store start-up diagnostic messages for later display in the console window.*
//* DO NOT call this method after dialog window has opened.                    *
//*                                                                            *
//* If a storage array has been defined, store the message for later display;  *
//* otherwise, discard the message.                                            *
//*                                                                            *
//* This is done because at the time when the diagnostic messages are received,*
//* the NCurses Engine may, or may not have set the locale for input/output,   *
//* and to avoid a case of the uglies, no I/O may be done until AFTER the      *
//* locale has been established.                                               *
//*                                                                            *
//* Input  : msg    : message to be displayed                                  *
//*          color  : color attribute for message                              *
//*          newLine: (optional, 'true' by default)                            *
//*                   if 'true'  move cursor to next message position          *
//*                   if 'false' leave cursor at end of message                *
//* Returns: nothing                                                           *
//******************************************************************************

void Taggit::DiagMsg ( const char* msg, attr_t color, bool newLine )
{

   if ( diagData != NULL )
   {
      gString gs ;

      for ( short i = ZERO ; i < diagdataCOUNT ; ++i )
      {
         if ( *diagData[i].msgText == NULLCHAR )
         {
            gs = msg ;
            gs.copy( diagData[i].msgText, ddmLENGTH ) ;
            diagData[i].msgAttr = color ;
            diagData[i].nl = newLine ;
            break ;
         }
      }
   }

}  //* End DiagMsg() *

//*************************
//*       DiagMsg         *
//*************************
//******************************************************************************
//* Display in the console window any previously-stored start-up diagnostic    *
//* messages. Each record is erased after it is written to the display.        *
//* DO NOT call this method after dialog window has opened.                    *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void Taggit::DiagMsg ( void )
{
   if ( diagData != NULL )
   {
      //* Create output columns for verbose diagnostics *
      // Programmer's Note: These calculations assume that we have a terminal window
      // of at least 30 rows by 120 columns which falls within the minimum application size.
      short xCol     = ZERO,    // output column for each message
            colBreak = 25,      // number of output rows at which to start new column
            colRow   = 8,       // row at top of columns
            colWidth = 40 ;     // column width
      // 'true' if end of config file reached OR if not verbose output
      bool  cfgComplete = (this->suVerbose ? false : true) ;

      for ( short i = ZERO ; i < diagdataCOUNT ; ++i )
      {
         if ( *diagData[i].msgText != NULLCHAR )
         {
            this->suPos = nc.WriteString ( this->suPos, 
                                           diagData[i].msgText,
                                           diagData[i].msgAttr ) ;
            if ( diagData[i].nl != false )
            {
               if ( (++this->suPos.ypos > colBreak) && !cfgComplete )
               {
                  this->suPos.ypos = colRow ;
                  this->suPos.xpos = xCol += colWidth ;
               }
               else if ( this->suPos.ypos >= this->termRows ) // prevent wrap-around
                  this->suPos.ypos = (this->termRows - 1) ;
               this->suPos.xpos = xCol ;
            }

            if ( this->suVerbose && !cfgComplete )
            {
               gString gs( diagData[i].msgText ) ;
               if ( (gs.find( "Read configuration file:" )) == ZERO )
               {  //* This adjustment is made in case caller adds more *
                  //* messages before starting to read the config file.*
                  colRow = this->suPos.ypos + 1 ;  // adjust row for top of column
                  colBreak = colRow + 18 ;         // each column has 19 rows
               }
               else if ( (gs.find( "EndConfigFile" )) >= ZERO )
               {  //* End of config file found. Return to bottom of left column. *
                  //* Note that if scan of config file is aborted, execution     *
                  //* will not reach this point.                                 *
                  this->suPos.ypos = colBreak + 1 ;
                  this->suPos.xpos = xCol = ZERO ;
                  cfgComplete = true ;
               }
            }
            diagData[i].reset() ;   // erase the message
         }
         else           // no more messages
         {
            //* Keep cursor position from going off into the weeds *
            this->suPos.xpos = ZERO ;
            break ;
         }
      }
   }

}  //* End DiagMsg() *

//*************************
//*    DisplayVersion     *
//*************************
//******************************************************************************
//* Print application version number and copyright notice to tty (bypassing    *
//* NCurses output).                                                           *
//* The NCurses engine has been shut down (or didn't start) so we use simple   *
//* console I/O to display the version/copyright text.                         *
//*                                                                            *
//* NOTE: This method relies on the application title string being of a certain*
//*       format. If the string changes, the header output may break.          *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void Taggit::DisplayVersion ( void )
{
   static const char* const freeSoftware = 
    "License GPLv3+: GNU GPL version 3 <http://gnu.org/licenses/gpl.html>\n"
    "This is free software: you are free to modify and/or redistribute it\n"
    "under the terms set out in the license.\n"
    "There is NO WARRANTY, to the extent permitted by law.\n" ;

   gString gsOut( "\n%s", &AppTitle1[2] ) ;
   gsOut.append( "%s Copyright(c) %s  %s      \n", 
                 AppVersion, copyrightYears, AppTitle2 ) ;
   short tCols = gsOut.gscols() ;
   for ( short i = ZERO ; i < tCols ; ++i )
      gsOut.append ( L'=' ) ;
   wcout << gsOut.gstr() << L'\n' << freeSoftware << endl ;

}  //* End DisplayVersion() *

//*************************
//*    DisplayHelp        *
//*************************
//******************************************************************************
//* Print command-line help to tty (bypassing NCurses output).                 *
//* The NCurses engine has been shut down (or didn't start) so we use simple   *
//* console I/O to display the help text.                                      *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************
//* Command-line Options:                                                      *
//* -- This is an interactive application, so we keep the command-line options *
//*    simple, targeting configuration and source-file specification only.     *
//* -- The first implementation included several metadata (tag) fields as      *
//*    command-line arguments, assuming that the files had just been ripped    *
//*    and contained no metadata. However, over time we realized that the      *
//*    typical audio file already contains SOME metadata, and blindly          *
//*    specifying data which overwrites it before the user had a chance to see *
//*    it was inappropriate, even if the user thinks it knows what it's doing. *
//* -- We allow '--artist' and '--album' tags as command-line options because  *
//*    these are likely to be the same for all specified source files.         *
//*                                                                            *
//* Programmer's Note: Multilingual Help cannot be written to stdout unless we *
//* replace the 'C' language (ASCII) locale with a true UTF-8 locale.          *
//* -- Since this application is an exercise in constructing a multilingual    *
//*    user interface, a first step is to dynamically translate the quick help *
//*    based on information from the terminal environment.                     *
//* -- Unfortunately, this doesn't work correctly _unless_ the text is first   *
//*    converted to 'wide' characters. We _could have_ simply defined all the  *
//*    text as wchar_t data, but that would bloat the binary with little       *
//*    benefit. Instead, we define the text as char data and convert it before *
//*    writing to stdout. This should not be necessary, but is a quirk of the  *
//*    console design.                                                         *
//* -- Once we are inside the ncurses environment, the entire character set    *
//*    becomes available because the NcDialog API is specifically designed for *
//*    a multilingual user interface.                                          *
//*                                                                            *
//* Adding a new UI language to this method:                                   *
//* ========================================                                   *
//* 1) Enable the two debugging definitions.                                   *
//* 2) Explicitly set the system locale if it is not already the one you want. *
//* 3) Add a test to match the language to the locale name.                    *
//* 4) Translate the descriptions for the command-line options.                *
//* 5) Divide the text among the four Help[] arrays, verifying that the total  *
//*    character count for each array fits comfortably within a gString object *
//*    i.e. 1024 (wchar_t) characters.                                         *
//* 6) Disable the debugging definitions.                                      *
//******************************************************************************

void Taggit::DisplayHelp ( void )
{
   #define DEBUG_QHELP (0)
   #define DEBUG_QHELP_COUNT (0)

static const char* Help1[] =
{
//** English **
 //12345678901234567890123456789012345678901234567890123456789012345678901234567890
"\nUSAGE: taggit [OPTIONS]"
"\n Options may be specified in any order. All options are case sensitive."
"\n "
"\n -f=a[,b...] One or more individual source filenames may be specified"
"\n             (comma separated). If the filename(s) contain spaces, then the"
"\n             argument must be enclosed in quotes as shown."
"\n                Example: taggit -f='Rok On!.mp3,Phoning It In.mp3'"
"\n             If no filenames are specified, then source is assumed to be all"
"\n             files in current directory."
"\n -F=NAMELIST Specify a file which contains a list of audio source files."
"\n             Example: -F=music_filenames.txt"
"\n -a          Enable all display fields.",

//** Espanol **
 //12345678901234567890123456789012345678901234567890123456789012345678901234567890
"\nUSO: taggit [OPCIONES]"
"\n Las opciones se pueden especificar en cualquier orden. Todas las opciones"
"\n distinguen entre mayúsculas y minúsculas."
"\n "
"\n -f=a[,b...] Se pueden especificar uno o más nombres de archivo fuente"
"\n             individuales (separados por comas). Si los nombres de archivo"
"\n             contiene espacios, entonces el argumento debe estar encerrado"
"\n             entre comillas como se muestra."
"\n                Ejemplo: taggit -f='Rok On!.mp3,Phoning It In.mp3'"
"\n             Si no se especifica ningún nombre de archivo, se supone que el"
"\n             origen es todos los archivos del directorio actual."
"\n -F=LISTA    Especifique un archivo que contenga una lista de archivos de"
"\n             fuentes de audio.   Ejemplo: -F=music_filenames.txt"
"\n -a          Habilitar todos los campos de visualización.",

//** Zhongwen **
 //12345678901234567890123456789012345678901234567890123456789012345678901234567890
"\n用法: taggit [选择]"
"\n可以以任何顺序指定选项。 所有选项都区分大小写。"
"\n"
"\n -f=a[,b...] 可以指定一个或多个单独的源文件名 (用逗号分隔). 如果文件名包含空格，"
"\n             那么参数必须用引号括起来，如图所示。"
"\n                例： taggit -f='Rok On!.mp3,Phoning It In.mp3'"
"\n             如果没有指定文件名，那么源将是当前目录中的所有文件。"
"\n -F=NAMELIST 指定一个包含音频源文件列表的文件。"
"\n             例： -F=music_filenames.txt"
"\n -a          启用所有显示字段。",

//** TiengViet **
 //12345678901234567890123456789012345678901234567890123456789012345678901234567890
"\nSỬ DỤNG: taggit [LỰA CHỌN]"
"\n Tùy chọn có thể được chỉ định theo thứ tự bất kỳ. Tất cả tùy chọn đều phân"
"\n biệt chữ hoa chữ thường."
"\n "
"\n -f=a[,b...] Có thể chỉ định một hoặc nhiều tên tệp nguồn riêng lẻ (được phân"
"\n             tách bằng dấu phẩy). Nếu (các) tên tệp tin có chứa khoảng trống, sau"
"\n             đó đối số phải được bao gồm trong dấu ngoặc kép như được hiển thị."
"\n                Thí dụ: taggit -f='Rok On!.mp3,Phoning It In.mp3'"
"\n             Nếu không có tên tập tin được chỉ định, thì mã nguồn được giả định"
"\n             là tất cả các tệp trong thư mục hiện tại."
"\n -F=LIST     Chỉ định tệp chứa danh sách các tệp nguồn âm thanh."
"\n                Thí dụ: -F=Âm_nhạc_tên_tệp.txt"
"\n -a          Bật tất cả các trường hiển thị.",
} ;

static const char* Help2[] = 
{
 //12345678901234567890123456789012345678901234567890123456789012345678901234567890
"\n -c=SCHEME   Color scheme for application.  Example: -c=blue"
"\n             Colors: [black|red|green|brown|blue|magenta|cyan|gray|default]"
"\n -C=CFGFILE File from which to read configuration data on startup."
"\n            (default: 'Taggit.cfg')    Example: -C=custom.cfg"
"\n -d=DIRNAME  Directory containing files to be processed (default: current dir)"
"\n                Examples: -d=Beatles/Revolver"
"\n                          -d '~/Music/Foreigner/Jukebox Hero'"
"\n -D          Dump (copy) metadata for specified file(s) to a plain text file,"
"\n             and exit. Name of output file will be written to stdout stream.",

//** Espanol **
 //12345678901234567890123456789012345678901234567890123456789012345678901234567890
"\n -c=ESQUEMA  Esquema de color para la aplicación. Ejemplo: -c=blue"
"\n             Colores: [black|red|green|brown|blue|magenta|cyan|gray|default]"
"\n                      (negro rojo verde marrón azul magenta cian gris defecto)"
"\n -C=CFGFILE  Archivo desde el que leer los datos de configuración al inicio."
"\n             (defecto: 'Taggit.cfg')    Ejemplo: -C=custom.cfg"
"\n -d=DIRNAME  Directorio que contiene los archivos a procesar."
"\n             (defecto: directorio actual)"
"\n                Ejemplos: -d=Beatles/Revolver"
"\n                          -d '~/Music/Foreigner/Jukebox Hero'"
"\n -D          Escanee los archivos de origen, copie los metadatos en un archivo"
"\n             de texto sin formato y salga. El nombre del archivo de salida"
"\n             se escribirá en el flujo stdout.",

//** Zhongwen **
 //12345678901234567890123456789012345678901234567890123456789012345678901234567890
"\n -c=SCHEME   应用颜色方案。  例子: -c=blue"
"\n             颜色: [black|red|green|brown|blue|magenta|cyan|gray|default]"
"\n                   (黑色, 朱红,绿色, 棕色,蓝色,  品红, 青色,灰色,  默认)"
"\n -C=CFGFILE 文件包含启动的配置数据。"
"\n            (默认值: 'Taggit.cfg')    例子: -C=custom.cfg"
"\n -d=DIRNAME  指示要处理的文件的目录。 (默认值: 当前目录)"
"\n                例子: -d=Beatles/Revolver"
"\n                     -d '~/Music/Foreigner/Jukebox Hero'"
"\n -D          扫描源文件，和 将指定文件的元数据复制到纯文本文件，然后退出。"
"\n             输出文件的名称将被写入标准输出流。",

//** TiengViet **
 //12345678901234567890123456789012345678901234567890123456789012345678901234567890
"\n -c=SCHEME   Các kế hoạch màu cho ứng dụng.  Thí dụ: -c=blue"
"\n             Màu sắc: [black|red|green|brown|blue|magenta|cyan|gray|default]"
"\n                      (đen, màu đỏ), màu xanh lá, nâu, màu xanh da trời,"
"\n                       màu đỏ tươi, màu lục lam, màu xám, màu mặc định)"
"\n -C=CFGFILE Tập tin để đọc dữ liệu cấu hình từ lúc khởi động."
"\n            (mặc định: 'Taggit.cfg')    Thí dụ: -C=custom.cfg"
"\n -d=DIRNAME Thư mục chứa các tệp tin cần được xử lý. (mặc định:thư mục hiện tại)"
"\n                Thí dụ: -d=Beatles/Revolver"
"\n                        -d '~/Music/Foreigner/Jukebox Hero'"
"\n -D          Danh sách (sao chép) siêu dữ liệu cho các tệp tin cụ thể vào một"
"\n             tệp văn bản thuần túy và thoát."
"\n             Tên của tập tin đầu ra sẽ được ghi vào dòng đầu ra tiêu chuẩn.",
} ;

static const char* Help3[] = 
{
//** English **
 //12345678901234567890123456789012345678901234567890123456789012345678901234567890
"\n -i          Ignore (discard) any existing metadata in source files."
"\n             All tag fields will initially be empty."
"\n -I=PATH     Insert an external image into one or more source files."
"\n             (specify JPG or PNG files only)"
"\n -l=LANGUAGE User interface language. (set from system's locale by default)"
"\n             Languages: Español, Zhōngwén (中文), TiếngViệt, English"
"\n                Example:  -l=Zh"
"\n -L=LOCALE   Specify an alternate locale (must support UTF-8 encoding)"
"\n                Example: -L=zh_CN.utf8"
"\n -m=[enable|disable]  Enable/disable mouse support.   Example: -m=disable"
"\n            Enabled by default if mouse hardware supports a scroll-wheel.",

//** Espanol **
 //12345678901234567890123456789012345678901234567890123456789012345678901234567890
"\n -i          Ignorar (descartar) cualquier metadato existente en los archivos de"
"\n             origen. Todos los campos de etiquetas estarán inicialmente vacíos."
"\n -I=PATH     Inserte una imagen externa en uno o más archivos de origen."
"\n             (especificar sólo archivos JPG o PNG)"
"\n -l=LANGUAGE Idioma de la interfaz de usuario. (Por defecto, el idioma se"
"\n             establece en función de la configuración regional del sistema.)"
"\n             Idiomas: Español, Zhōngwén (中文), TiếngViệt, English"
"\n                Ejemplo:  -l=Es"
"\n -L=LOCALE   Especificar una configuración regional alternativa (debe admitir"
"\n             la codificación UTF-8).   Ejemplo: -L=zh_CN.utf8"
"\n -m=[enable|disable]  Activar o desactivar el soporte del ratón."
"\n            Ejemplo: -m=disable"
"\n            Se habilita de forma predeterminada si el hardware del ratón"
"\n            admite una rueda de desplazamiento.",

//** Zhongwen **
 //12345678901234567890123456789012345678901234567890123456789012345678901234567890
"\n -i          忽略（丢弃）源文件中的任何现有元数据。最初所有的标签字段都将为空。"
"\n -I=PATH     将外部图像插入一个或多个源文件。"
"\n             (指定格式为 JPG 或 PNG 的文件。)"
"\n -l=LANGUAGE 用户界面语言。 (默认情况下，语言取自本地系统设置。)"
"\n             语言： Español, Zhōngwén (中文), TiếngViệt, English"
"\n                例子:  -l=Zh"
"\n -L=LOCALE   指定备用语言环境。 (必须支持UTF-8编码)"
"\n                例子: -L=zh_CN.utf8"
"\n -m=[enable|disable]  启用或禁用鼠标支持。   例子: -m=disable"
"\n            如果鼠标有滚动轮，默认情况下启用鼠标。",

//** TiengViet **
 //12345678901234567890123456789012345678901234567890123456789012345678901234567890
"\n -i          Bỏ qua (loại bỏ) bất kỳ siêu dữ liệu hiện có trong các tệp nguồn."
"\n             Tất cả các trường nhãn sẽ trống rỗng ngay từ đầu."
"\n -I=PATH     Chèn một hình ảnh bên ngoài vào một hoặc nhiều tệp nguồn."
"\n             (Xác định một tệp với định dạng JPG hoặc PNG.)"
"\n -l=LANGUAGE Ngôn ngữ giao diện người dùng."
"\n             (Thiết lập từ địa bàn trong cài đặt hệ thống theo mặc định.)"
"\n             Ngôn ngữ: Español, Zhōngwén (中文), TiếngViệt, English"
"\n                Thí dụ:  -l=Vi"
"\n -L=LOCALE   Chỉ định một địa bàn thay thế (phải hỗ trợ mã hoá UTF-8)."
"\n                Thí dụ: -L=vi_VN.utf8"
"\n -m=[enable|disable]  Bật hoặc tắt hỗ trợ chuột.   Thí dụ: -m=disable"
"\n            Bật theo mặc định nếu phần cứng chuột hỗ trợ một bánh xe cuộn.",
} ;

static const char* Help4[] = 
{
//** English **
 //12345678901234567890123456789012345678901234567890123456789012345678901234567890
"\n -p[v]      Pause to display startup diagnostics before entering main program."
"\n            The 'v' sub-option presents verbose diagnostics."
"\n -P         Preserve source files. Write metadata to a COPY of the source file."
"\n            (original file renamed as a backup: 'Rain.mp3' ==>> 'Rain.mp3~')"
"\n -s=[Filename|Title|Track|Album|Artist|None]  Sort option. (default: Filename)"
"\n "
"\n --album=ALBUM     Name of album in which the work was released."             // TALB
"\n --artist=ARTISTS  Lead performer(s) or group"                                // TPE1
"\n Long-form option examples: (If text contains spaces, enclose in quotes.)"
"\n          taggit --artist='Bon Jovi' --album 'Slippery When Wet'"
"\n "
"\n --version        Display application version number and copyright information."
"\n --help, -h, -?   Help for command line options."
"\n",

//** Espanol **
 //12345678901234567890123456789012345678901234567890123456789012345678901234567890
"\n -p[v]      Haga una pausa para mostrar los diagnósticos de inicio antes"
"\n            de entrar en el programa principal."
"\n            La sub-opción 'v' presenta diagnósticos detallados."
"\n -P         Conservar archivos de origen. Escribir metadatos en una COPIA"
"\n            del archivo fuente. (archivo original renombrado como una copia"
"\n            de seguridad: 'Rain.mp3' ==>> 'Rain.mp3~')"
"\n -s=[Filename|Title|Track|Album|Artist|None]  Clasificación. (defecto:Filename)"
"\n    (archivo, título, pista, álbum, artista, ninguna)"
"\n "
"\n --album=ALBUM     Nombre del álbum en el que se publicó la canción."         // TALB
"\n --artist=ARTISTS  Artista(s) principal o grupo."                             // TPE1
"\n Opciones de formato largo ejemplo: (Si el texto contiene espacios,"
"\n colóquelos entre comillas.)"
"\n          taggit --artist='Bon Jovi' --album 'Slippery When Wet'"
"\n "
"\n --version        Mostrar el número de versión de la aplicación y la"
"\n                  información de copyright."
"\n --help, -h, -?   Ayuda para las opciones de la línea de comandos."
"\n",

//** Zhongwen **
 //12345678901234567890123456789012345678901234567890123456789012345678901234567890
"\n -p[v]      暂停显示启动诊断之前进入主要应用。"
"\n            'v'子选项显示详细诊断。"
"\n -P         保留源文件。将元数据写入源文件的副本。"
"\n            (原始文件重命名为备份： 'Rain.mp3' ==>> 'Rain.mp3~')"
"\n -s=[Filename|Title|Track|Album|Artist|None]  排序选项。 (默认值: Filename)"
"\n    (文件名， 标题，曲目，专辑，艺术家，没有)"
"\n "
"\n --album=ALBUM     作品发行的专辑名称 歌被拍了。"                                  // TALB
"\n --artist=ARTISTS  主要表演者或团体。"                                           // TPE1
"\n 长格式选项示例： (如果文本包含空格，请用引号括起来。)"
"\n          taggit --artist='Bon Jovi' --album 'Slippery When Wet'"
"\n "
"\n --version        显示应用程序的版本号和版权信息。"
"\n --help, -h, -?   帮助命令行选项。"
"\n",

//** TiengViet **
 //12345678901234567890123456789012345678901234567890123456789012345678901234567890
"\n -p[v]      Tạm dừng để hiển thị chẩn đoán khởi động trước khi nhập chương trình"
"\n            chính. Tùy chọn phụ 'v' hiển thị chẩn đoán chi tiết."
"\n -P         Bảo tồn tệp nguồn. Viết siêu dữ liệu vào một bản sao của tệp nguồn."
"\n            (tập tin gốc được đổi tên thành một bản sao lưu:"
"\n            'Rain.mp3' ==>> 'Rain.mp3~')"
"\n -s=[Filename|Title|Track|Album|Artist|None] Sắp xếp tùy chọn(mặc định:Filename)"
"\n    (tên tập tin, tiêu đề, chữ số, tập ảnh, nghệ sĩ, gì hết)"
"\n "
"\n --album=ALBUM     Tên album trong đó bài hát đã được phát hành."             // TALB
"\n --artist=ARTISTS  Người biểu diễn hoặc nhóm."                                // TPE1
"\n Các ví dụ tùy chọn dạng dài: (Nếu văn bản có chứa dấu cách, hãy đóng dấu"
"\n ngoặc kép.)   taggit --artist='Bon Jovi' --album 'Slippery When Wet'"
"\n "
"\n --version        Hiển thị số phiên bản ứng dụng và thông tin bản quyền."
"\n --help, -h, -?   Trợ giúp cho các tùy chọn dòng lệnh."
"\n",
} ;


   //* Set locale from 'C' to (it is hoped) the system's UTF-8 locale.*
   #if DEBUG_QHELP == 0    // PRODUCTION
   locale loco( "" ) ;
   #else    // DEBUG_QHELP - SET LANGUAGE IN ENVIRONMENT FOR TESTING TRANSLATIONS
     locale loco( "en_US.utf8" ) ;
   //locale loco( "es_MX.utf8" ) ;
   //locale loco( "zh_CN.utf8" ) ;
   //locale loco( "vi_VN.utf8" ) ;
   #if DEBUG_QHELP_COUNT != 0
   short cCnt ;      // character count
   #endif   // DEBUG_QHELP_COUNT
   #endif   // DEBUG_QHELP
   wcout.imbue ( loco.global( loco ) ) ;

   //* Try to determine language from the locale.*
   //* If we fail, then no harm done.            *
   AppLang lang = enLang ;
   std::string st = loco.name() ;
   gString gs( st.c_str() ) ;
   if ( (gs.find( "es" )) == ZERO )
      lang = esLang ;
   else if ( (gs.find( "zh" )) == ZERO )
      lang = zhLang ;
   else if ( (gs.find( "vi" )) == ZERO )
      lang = viLang ;

   //* Display application title *
   gString gsOut( "\n%s", &AppTitle1[2] ) ;
   gsOut.append( "%s Copyright(c) %s  %s                  \n", 
                 AppVersion, copyrightYears, AppTitle2 ) ;
   short tCols = gsOut.gscols() ;
   for ( short i = ZERO ; i < tCols ; ++i )
      gsOut.append ( L'=' ) ;
   wcout << gsOut.gstr() ;

   //* Display quick help in designated language *
   gsOut = Help1[lang] ;
   wcout << gsOut.gstr() ;

   #if DEBUG_QHELP != 0 && DEBUG_QHELP_COUNT != 0
   cCnt = gsOut.gschars() ; gsOut.compose( "\n** cCnt:%hd", &cCnt ) ;
   wcout << gsOut.gstr() ;
   #endif   // DEBUG_QHELP_COUNT

   gsOut = Help2[lang] ;
   wcout << gsOut.gstr() ;

   #if DEBUG_QHELP != 0 && DEBUG_QHELP_COUNT != 0
   cCnt = gsOut.gschars() ; gsOut.compose( "\n** cCnt:%hd", &cCnt ) ;
   wcout << gsOut.gstr() ;
   #endif   // DEBUG_QHELP_COUNT

   gsOut = Help3[lang] ;
   wcout << gsOut.gstr() ;

   #if DEBUG_QHELP != 0 && DEBUG_QHELP_COUNT != 0
   cCnt = gsOut.gschars() ; gsOut.compose( "\n** cCnt:%hd", &cCnt ) ;
   wcout << gsOut.gstr() ;
   #endif   // DEBUG_QHELP_COUNT

   gsOut = Help4[lang] ;
   wcout << gsOut.gstr() ;

   #if DEBUG_QHELP != 0 && DEBUG_QHELP_COUNT != 0
   cCnt = gsOut.gschars() ; gsOut.compose( "\n** cCnt:%hd", &cCnt ) ;
   wcout << gsOut.gstr() ;
   #endif   // DEBUG_QHELP_COUNT

   wcout << endl ;


   #undef DEBUG_QHELP
   #undef DEBUG_QHELP_COUNT


#if 0    // NOT CURRENTLY SUPPORTED AS COMMAND-LINE OPTIONS
   "\n Long-form option examples: (If text contains spaces, enclose in quotes.)"
   "\n          taggit --artist='Bon Jovi' --album 'Slippery When Wet'"
   "\n "
   "\n --title             The name of the work."                                // TIT2
   "\n --subtitle          Supplimentary or additional name of work."            // TIT3
   "\n --album             Name of album in which the work was released."        // TALB
   "\n --artist            Lead performer(s) or group"                           // TPE1
   "\n --guest_artist      Guest artist/band, orchestra or accompaniment."       // TPE2
   "\n --copyright         Copyright notice."                                    // TCOP
   "\n --publisher         Name of record label or publisher"                    // TPUB
   "\n "
   "\n --bpm               Beats-per-minute."                                    // TBPM
   "\n --composer          Author(s) of the music."                              // TCOM
   "\n --content           Content type. remix, cover, re-release, ect."         // TCON
   "\n --date              Day and month of recording in the format: DDMM"       // TDAT
   "\n --delay             Playlist delay between songs (milliseconds)"          // TDLY
   "\n --encodeby          Person or organization that encoded the file."        // TENC
   "\n --lyricist          Author(s) of the lyrics."                             // TEXT
   "\n --filetype          Specifies the flavor of MPEG encoding."               // TFLT
   "\n --time              Runtime of recording in the format: HHMM"             // TIME
   "\n --contentgrp        Content group specifies the category of music."       // TIT1
   "\n --key               Initial musical key signature of work."               // TKEY
   "\n --language          Spoken/sung language (three-character ISO-639-2 code)"// TLAN
   "\n --runtime           Actual audio file runtime in milliseconds"            // TLEN
   "\n --media             (this will always be \"DIG\" digital)"                // TMED
   "\n --o_album           Recording's original album."                          // TOAL
   "\n --o_title           Recording's original filename (includes suffix)."     // TOFN
   "\n --o_lyric           Recording's original lyricist."                       // TOLY
   "\n --o_artist          Recording's original artist/performer."               // TOPE
   "\n --owner             Name of the owner/licensee of the file and contents." // TOWN
   "\n --orig_year         Recording's original year of release."                // TORY
   "\n --conductor         Name of the conductor."                               // TPE3
   "\n --remix             Original piece interpreted, remixed or modified by."  // TPE4
   "\n --set_member        Item number in a set (e.g. item 1 of a 2-disc set)."  // TPOS
   "\n --track             Sequence number of set, and optionally items in set." // TRCK
   "\n                     --track=n[/m]   Examples: --track=4   --track=02/14"  
   "\n --rec_date          Day/month of recording (see TYER) Ex:July-October"    // TRDA
   "\n --web_radio         Internet radio station."                              // TRSN
   "\n --web_owner         Internet radio station owner."                        // TRSO
   "\n --size              Number of bytes of audio data (excludes tag info)"    // TSIZ
                           // (this can be calculated and entered automagically)                  
   "\n --isrc_code         International Standard Recording Code (12-char code)" // TSRC
   "\n --settings          Software/hardware settings used for encoding."        // TSSE
   "\n --year              Year of recording in the format: YYYY"                // TYER
   "\n --comment           Free-form text field."                                // TXXX
   "\n --cover_art         Name for embedded image file (.PNG or .JPG only)"     // APIC
//   "\n||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||"
#endif   // NOT CURRENTLY SUPPORTED AS COMMAND-LINE OPTIONS

}  //* End DisplayHelp() *

//*************************
//*                       *
//*************************
//******************************************************************************
//*                                                                            *
//*                                                                            *
//*                                                                            *
//* Input  :                                                                   *
//*                                                                            *
//* Returns:                                                                   *
//******************************************************************************

