//******************************************************************************
//* File       : TagOgg.hpp                                                    *
//* Author     : Mahlon R. Smith                                               *
//*              Copyright (c) 2016-2020 Mahlon R. Smith, The Software Samurai *
//*                  GNU GPL copyright notice located in Taggit.hpp            *
//* Date       : 03-Oct-2020                                                   *
//* Version    : (see AppVersion string)                                       *
//*                                                                            *
//* Description: This header defines the metadata fields for OGG/Vorbis        *
//*              audio files.                                                  *
//*                                                                            *
//******************************************************************************

//****************
//* Header Files *
//****************


const char* const Ogg_Tag = "OggS" ;// Tag indicating an Ogg-format file
const int OGG_TAG_BYTES = 4 ;       // Length of Ogg_Tag
const int OGG_PAGE1_BYTES = 58 ;    // Length of first Ogg page
const int OGG_PAGEHDR_BYTES = 27 ;  // Number of bytes in Page 2 page header
                                    // 0-255 'segments' follow
const char* const Ogg_PktHdr = "vorbis" ; // Packet Header ID
const int OGG_PKTHDR_BYTES = 7 ;    // Length of packet header (see Ogg_Packet)
const int OGG_SEGSIZE = 0x0FF ;     // Maximum byte count in segment table entry (255)
                                       // AND maximum number of segments
const int OGG_MAX_PAGESEGS = 255 ;  // Maximum number of segments on a single page
const uint8_t FRAMING_BIT = 0x01 ;  // The framing bit (byte) at the end of a page
typedef u_int32_t UINT32 ;          // alias a 32-bit integer type

//* This is the default OGG comment name and is associated *
//* with the 'tfTxxx' member of enum TagFields.            *
//* See also: OggFieldMap ofma[] in TagOgg.cpp.            *
const wchar_t* const OGG_DFLT_COMMENT = L"DESCRIPTION" ;
//* Un-map MP3 field codes to ofmb[] comment names.        *
const wchar_t* const OGG_BLIST_01 = L"ENCODER" ;   // tfTenc
const wchar_t* const OGG_BLIST_02 = L"DATE" ;      // tfTyer
const wchar_t* const OGG_BLIST_03 = L"COMPOSER" ;  // tfTcom

//* Each OGG page begins with a page header.                *
//* For reading the metadata, we don't care about the page  *
//* header contents except to verify that it is valid data. *
class Ogg_PageHdr
{
   public:

   Ogg_PageHdr ( void )
   { this->reset () ; }
   ~Ogg_PageHdr ( void ) {}
   void reset ( void )
   {
      *this->tag = NULLCHAR ;
      this->version = this->ptype = this->segcount = ZERO ;
      this->granpos = ZERO ;
      this->streamser = this->pageseq = this->cksum = this->segBytes = ZERO ;
   }

   //* Convert a 4-byte sequence (LSB at offset 0) to an integer value. *
   UINT32 intConv ( const uint8_t* ucp ) const
   {
      int u =    (UINT32)ucp[0] 
              | ((UINT32)ucp[1] << 8)
              | ((UINT32)ucp[2] << 16)
              | ((UINT32)ucp[3] << 24) ;
      return u ;
   }

   //* Convert a 32-bit integer into an 4-byte, little-endian binary stream *
   void intConv ( int val, uint8_t* ucp ) const
   {
      ucp[0] = (uint8_t)(val & 0x000000FF) ;
      ucp[1] = (uint8_t)((val >>  8) & 0x000000FF) ;
      ucp[2] = (uint8_t)((val >> 16) & 0x000000FF) ;
      ucp[3] = (uint8_t)((val >> 24) & 0x000000FF) ;
   }

   //* Convert an 8-byte sequence (LSB at offset 0) to an int64 value. *
   long long int intConv64 ( const uint8_t* ucp ) const
   {
      long long int i =
              (UINT64)ucp[0] 
            | ((UINT64)ucp[1] << 8)
            | ((UINT64)ucp[2] << 16)
            | ((UINT64)ucp[3] << 24)
            | ((UINT64)ucp[4] << 32)
            | ((UINT64)ucp[5] << 40)
            | ((UINT64)ucp[6] << 48)
            | ((UINT64)ucp[7] << 56) ;
      return i ;
   }

   //* Convert a 64-bit integer into an 8-byte, little-endian binary stream *
   void intConv64 ( long long int val64, uint8_t* ucp ) const
   {
      ucp[0] = (uint8_t)(val64 & 0x00000000000000FF) ;
      ucp[1] = (uint8_t)((val64 >>  8) & 0x00000000000000FF) ;
      ucp[2] = (uint8_t)((val64 >> 16) & 0x00000000000000FF) ;
      ucp[3] = (uint8_t)((val64 >> 24) & 0x00000000000000FF) ;
      ucp[4] = (uint8_t)((val64 >> 32) & 0x00000000000000FF) ;
      ucp[5] = (uint8_t)((val64 >> 40) & 0x00000000000000FF) ;
      ucp[6] = (uint8_t)((val64 >> 48) & 0x00000000000000FF) ;
      ucp[7] = (uint8_t)((val64 >> 56) & 0x00000000000000FF) ;
   }

   char   tag[5] ;         // should be 'OggS'
   uint8_t  version ;      // stream structure revision (should be 0x00)
   uint8_t  ptype ;        // may be 0x01, 0x02, 0x04 or an OR value from these
   uint8_t  segcount ;     // number of segments in page (range: 0-255)
   UINT64 granpos ;        // encoded data used by decoder (8 bytes)
   UINT32 streamser ;      // uniquely identifies the stream (4 bytes)
   UINT32 pageseq ;        // page sequence number (4 bytes)
   UINT32 cksum ;          // CRC checksum (4 bytes)
   UINT32 segBytes ;       // total bytes in page segments
} ;   // Ogg_PageHdr

//* Each of the three headers begins with the sequence:  *
//*             nn 'v' 'o' 'r' 'b' 'i' 's'               *
//* where 'nn' is the packet type:                       *
//*             01h==Identification header               *
//*             03h==Comment header                      *
//*             05h==Setup header                        *
class Ogg_Packet
{
   public:

   Ogg_Packet ( void )
   { this->reset () ; }
   ~Ogg_Packet ( void ) {}
   void reset ( void )
   {
      this->ptype = ZERO ;
      this->pid[0] = this->pid[1] = this->pid[2] = this->pid[3] = 
      this->pid[4] = this->pid[5] = this->pid[6] = NULLCHAR ;
   }
   void populate ( UINT32 page )
   {
      this->reset () ;
      this->ptype = page == 1 ? 1 : page == 2 ? 3 : 5 ;
      this->pid[0] = 'v' ; this->pid[1] = 'o' ; this->pid[2] = 'r' ; 
      this->pid[3] = 'b' ; this->pid[4] = 'i' ; this->pid[5] = 's' ; 
   }

   uint8_t ptype ;   // packet type
   char  pid[7] ;    // packet identifier
} ;

class Ogg_ID
{
   public:

   Ogg_ID ( void )
   { this->reset () ; }
   ~Ogg_ID ( void ) {}
   void reset ( void )
   {
      this->version = this->blksize0 = this->blksize1 = ZERO ;
      this->samprate = this->bitratemax = this->bitratenom = this->bitratemin = ZERO ;
      this->channels = ZERO ;
      this->frameflag = false ;
   }

   //* Convert a 4-byte sequence (LSB at offset 0) to an integer value. *
   //* Programmer's Note: This clunky construct avoids the C library's  *
   //* "helpful" automatic sign extension.                              *
   UINT32 intConv ( const uint8_t* ucp )
   {
      UINT32 u =    (UINT32)ucp[0] 
                 | ((UINT32)ucp[1] << 8)
                 | ((UINT32)ucp[2] << 16)
                 | ((UINT32)ucp[3] << 24) ;
      return u ;
   }

   UINT32 version ;      // vorbis I version (must be zero)
   int     samprate ;    // audio sample rate (must be > zero)
   int     bitratemax ;  // bitrate maximum
   int     bitratenom ;  // bitrate nominal
   int     bitratemin ;  // bitrate minimum
   UINT32  blksize0 ;    // exponent for power of 2 (4 bits only)
   UINT32  blksize1 ;    // exponent for power of 2 (4 bits only)
   uint8_t channels ;    // number of audio channels (must be > zero)
   bool    frameflag ;   // framing flag (true==GOOD, false==ERROR)
} ;
const int OGG_ID_BYTES = sizeof(Ogg_ID) ;

//* Individual comment *
class Ogg_Comment
{
   public:

   Ogg_Comment ( void )       // constructor
   { this->reset () ; }
   ~Ogg_Comment ( void ) {}   // destructor

   void reset ( void )
   {
      this->clen = ZERO ;
      for ( short i = ZERO ; i < gsMAXBYTES ; ++i )
         this->ctxt[i] = NULLCHAR ;
   }

   //* Convert a 4-byte sequence (LSB at offset 0) to an integer value. *
   //* Programmer's Note: This clunky construct avoids the C library's  *
   //* "helpful" automatic sign extension.                              *
   UINT32 intConv ( const uint8_t* ucp ) const
   {
      UINT32 u =    (UINT32)ucp[0] 
                 | ((UINT32)ucp[1] << 8)
                 | ((UINT32)ucp[2] << 16)
                 | ((UINT32)ucp[3] << 24) ;
      return u ;
   }

   //* Convert a 32-bit unsigned integer to a little-endian byte stream.      *
   //* NOTE: In a 32-bit or 64-bit little-endian system, 'u' and 'ucp'  will  *
   //*       be the same, but we can't make assumptions about user's hardware.*
   void intConv ( UINT32 u, uint8_t* ucp ) const
   {
      ucp[0] = (uint8_t)(u & 0x000000FF) ;
      ucp[1] = (uint8_t)((u >>  8) & 0x000000FF) ;
      ucp[2] = (uint8_t)((u >> 16) & 0x000000FF) ;
      ucp[3] = (uint8_t)((u >> 24) & 0x000000FF) ;
   }

   UINT32 clen ;              // number of bytes in comment (excl. NULLCHAR)
   char   ctxt[gsMAXBYTES] ;  // null terminated text comment
   // NOTE: See note about Page 2 overflow in oggAdjustSegmentTable() header
} ;

//* Capture existing Page 2 segment data, and *
//* adjust the segment data for output.       *
class Ogg_Segments
{
   public:
   Ogg_Segments ( void )
   {
      for ( int i = ZERO ; i < OGG_SEGSIZE ; ++i )
         this->segTable[i] = 0 ;
      this->segCount = this->segBytes = 0 ;
      this->oldCVectors = this->oldCBytes = this->oldCSegs = 0 ;
      this->newCVectors = this->newCBytes = this->newCSegs = 0 ;
      this->newComm = NULL ;
      this->pageBuff = NULL ;
      this->pbAlloc = this->pbIndex = 0 ;
   }
   ~Ogg_Segments ( void )
   {
      if ( this->pageBuff != NULL )
         delete [] this->pageBuff ;
      if ( this->newComm != NULL )
         delete [] this->newComm ;
   }

   //* Allocate an array of Ogg_Comment objects *
   //* to contain edited comment vectors.       *
   void cvAlloc ( short cvCount )
   {
      if ( this->newComm != NULL )
         delete [] this->newComm ;
      this->newCVectors = this->newCBytes = this->newCSegs = 0 ;
      if ( cvCount > ZERO )
         this->newComm = new Ogg_Comment[this->newCVectors = cvCount] ;
   }

   //* Allocate a buffer for formatting the output stream.    *
   //* Allocation is an arbitrary, larger-than necessary size.*
   int pbAllocate ( void )
   {
      // size of 'pageBuff' allocation
      this->pbAlloc = (OGG_PAGEHDR_BYTES + OGG_PKTHDR_BYTES 
                     + this->segCount
                     + (this->segBytes - this->oldCBytes) + this->newCBytes
                     + OGG_SEGSIZE) ;
      this->pbAlloc += (this->pbAlloc % OGG_SEGSIZE) ;
      this->pageBuff = new uint8_t[this->pbAlloc] ;

      return this->pbAlloc ;
   }

   //* Technical Note: According to the specification, the integers used to    *
   //* handle comments _should be_ 32-bit 'unsigned int' (2 to 32nd -1 bytes). *
   //* However, 'int' is more useful for our internal math, and in any case,   *
   //* except for images, our user interface limits the size of a comment      *
   //* to four(4) KB. Note that a "Page" is 255 * 255 == 65,025 bytes.         *

   char segTable[OGG_SEGSIZE] ;  // segment table (byte array)
   int  segCount ;               // number of active elements in 'table'
   int  segBytes ;               // total bytes in all segments
   int  oldCVectors ;            // original comment-vector count
   int  oldCBytes ;              // original comment-vector total size
   int  oldCSegs ;               // segments occupied by original comments
   int  newCVectors ;            // edited comment-vector count
   int  newCBytes ;              // edited comment-vector total size
   int  newCSegs ;               // segments occupied by edited comments
   Ogg_PageHdr oph ;             // Page header data
   Ogg_Packet  opkt ;            // Packet header data
   Ogg_Comment vendor ;          // 'Vendor' record
   Ogg_Comment *newComm ;        // Dynamic allocation for array of edited comment vectors
   uint8_t *pageBuff ;           // Dynamic allocation for output buffer
   int  pbAlloc ;                // size of 'pageBuff' allocation
   int  pbIndex ;                // insertion point into 'pageBuff'
} ;   // Ogg_Segments

//* This is the MIME "Base64" lookup table which matches six(6) bits of binary *
//* source data with a 7-bit ASCII character. This allows a binary image (or   *
//* other binary data) to be encoded as text for easy transmission using 7-bit *
//* transport protocols.                                                       *
//* It is called "Base64" because the lookup table has 64 elements.            *
//*     Editorial: We believe that the logic for using this to encode images   *
//*     embedded in OGG/Vorbis audio is faulty since the audio itself is       *
//*     trasported in its native binary format, so converting an image to      *
//*     text is, in our view, a huge waste of time. and storage space.         *
const uint32_t bt64BYTES = 64 ;
const uint8_t base64Table[bt64BYTES] = 
{//CHAR    INDX  SRC BITS    
   'A', // 0x00  0000 0000
   'B', // 0x01  0000 0001
   'C', // 0x02  0000 0010
   'D', // 0x03  0000 0011
   'E', // 0x04  0000 0100
   'F', // 0x05  0000 0101
   'G', // 0x06  0000 0110
   'H', // 0x07  0000 0111
   'I', // 0x08  0000 1000
   'J', // 0x09  0000 1001
   'K', // 0x0A  0000 1010
   'L', // 0x0B  0000 1011
   'M', // 0x0C  0000 1100
   'N', // 0x0D  0000 1101
   'O', // 0x0E  0000 1110
   'P', // 0x0F  0000 1111 (Hash_P)
                              
   'Q', // 0x10  0001 0000
   'R', // 0x11  0001 0001
   'S', // 0x12  0001 0010
   'T', // 0x13  0001 0011
   'U', // 0x14  0001 0100
   'V', // 0x15  0001 0101
   'W', // 0x16  0001 0110
   'X', // 0x17  0001 0111
   'Y', // 0x18  0001 1000
   'Z', // 0x19  0001 1001
   'a', // 0x1A  0001 1010
   'b', // 0x1B  0001 1011
   'c', // 0x1C  0001 1100
   'd', // 0x1D  0001 1101
   'e', // 0x1E  0001 1110
   'f', // 0x1F  0001 1111 (Hash_f)
   
   'g', // 0x20  0010 0000
   'h', // 0x21  0010 0001
   'i', // 0x22  0010 0010
   'j', // 0x23  0010 0011
   'k', // 0x24  0010 0100
   'l', // 0x25  0010 0101
   'm', // 0x26  0010 0110
   'n', // 0x27  0010 0111
   'o', // 0x28  0010 1000
   'p', // 0x29  0010 1001
   'q', // 0x2A  0010 1010
   'r', // 0x2B  0010 1011
   's', // 0x2C  0010 1100
   't', // 0x2D  0010 1101
   'u', // 0x2E  0010 1110
   'v', // 0x2F  0010 1111
   
   'w', // 0x30  0011 0000
   'x', // 0x31  0011 0001
   'y', // 0x32  0011 0010
   'z', // 0x33  0011 0011 (Hash_z)
   '0', // 0x34  0011 0100
   '1', // 0x35  0011 0101
   '2', // 0x36  0011 0110
   '3', // 0x37  0011 0111
   '4', // 0x38  0011 1000
   '5', // 0x39  0011 1001
   '6', // 0x3A  0011 1010
   '7', // 0x3B  0011 1011
   '8', // 0x3C  0011 1100
   '9', // 0x3D  0011 1101
   '+', // 0x3E  0011 1110
   '/', // 0x3F  0011 1111 (HashEnd)
} ;

//* The base64 lookup table (above) is based on groups of three(3) source      *
//* bytes. If the final group is incomplete i.e. 1 or 2 bytes, then this       *
//* character is used to pad the output to a multiple of four(4) characters.   *
const uint8_t base64PADCHAR = '=' ;   // (0x3D)

//* Decode and encode OGG/Vorbis images which are encoded in the               *
//* MIME-specified "base64" binary-to-text encoding. This encoding was created *
//* to prevent external programs from choking on binary data during            *
//* transmission. Unfortunately, is is outrageously inefficient in terms of    *
//* storage space, requiring approximately 1.37 times as much space as the     *
//* original binary data.                                                      *
//* The binary data in this case are JPG or PNG images which are embedded into *
//* the OGG/Vorbis Comment Header as text comments. We have serious            *
//* philosophical objections to this mechanism, and the OGG/Vorbis standard    *
//* itself says that any large text item stream should be encoded in an XML    *
//* stream, NOT in the comment header. That being said, it is really very      *
//* simple to decode and encode the image data. Unfortunately, media players   *
//* will see a group of integer data following the comment name, and may very  *
//* choke, or at best discard the entire vector.                               *
//*                                                                            *
//* The total size of an Image Vector is 4 bytes for vector length PLUS:       *
//* 22 bytes  "METADATA_BLOCK_PICTURE"                                         *
//*  1 byte   '=' character                                                    *
//*  4 bytes  picType code                                                     *
//*  4 bytes  MIME length                                                      *
//*  n bytes  MIME text (3 <= string <= 9  (no NULLCHAR))                      *
//*  4 bytes  description length                                               *
//*  n bytes  description text (0<= string <= gsMAXBYTES)                      *
//*  4 bytes  image width                                                      *
//*  4 bytes  image height                                                     *
//*  4 bytes  image color depth                                                *
//*  4 bytes  image color indexing                                             *
//*  4 bytes  length of encoded binary image data                              *
//*  n bytes  encoded binary data (4 <= data <= 2^32-1)                        *
//* -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  *
class Ogg_Image
{
   public:
   Ogg_Image ( void )               // default constructor
   { this->reset () ; }
   ~Ogg_Image ( void ) {}           // destructor

   void reset ( void )
   {
      //* NOTE: If 'picPath' points to a temp file, *
      //*       the reset orphans the file.         *
      this->picPath[0] = this->txtDesc[0] = 
      this->mimType[0] = this->picExpl[0] = NULLCHAR ;
      this->picType  = 0x00 ;          // default to "Other"
      this->mimBytes = this->txtBytes = 
      this->iWidth   = this->iHeight  = this->iCDepth = this->iCIndex = 
      this->picBytes = this->picEBytes = this->totBytes = ZERO ;
   }

   //* Convert a 4-byte sequence (MSB at offset 0) to an integer value. *
   //* Programmer's Note: This clunky construct avoids the C library's  *
   //* "helpful" automatic sign extension.                              *
   UINT32 intConv ( const uint8_t* ucp )
   {
      UINT32 u =    (UINT32)ucp[3] 
                 | ((UINT32)ucp[2] << 8)
                 | ((UINT32)ucp[1] << 16)
                 | ((UINT32)ucp[0] << 24) ;
      return u ;
   }

   //* Convert a 32-bit integer into a big-endian byte stream.    *
   short intConv ( UINT32 uval, uint8_t* obuff )
   {
      const UINT32 bMASK = 0x000000FF ;   // low-byte mask
      short indx = ZERO ;                 // return value

      obuff[indx++] = (uint8_t)((uval >> 24) & bMASK) ;
      obuff[indx++] = (uint8_t)((uval >> 16) & bMASK) ;
      obuff[indx++] = (uint8_t)((uval >>  8) & bMASK) ;
      obuff[indx++] = (uint8_t)(uval & bMASK) ;

      return indx ;
   }

   //* Convert the data in our members to a binary stream and return the     *
   //* number of binary bytes written to caller's buffer.                    *
   //* Note that the first four(4) bytes are not included in the vector size.*
   UINT32 encodeHeader ( uint8_t* ucp )
   {
      gString gs ;                  // text manipulation
      UINT32 trgCount = ZERO ;      // number of bytes written to target
      trgCount = this->intConv( this->totBytes, &ucp[3] ) ;
      ucp[0] = ucp[6] ;    // this is actually a little-endian value
      ucp[1] = ucp[5] ;
      ucp[2] = ucp[4] ;
      gs = this->ivNAME ;
      gs.copy( (char*)&ucp[trgCount], gsMAXBYTES ) ;
      trgCount += gs.utfbytes() - 1 ;
      ucp[trgCount++] = '=' ;
      trgCount += this->intConv( this->picType, &ucp[trgCount] ) ;
      trgCount += this->intConv( this->mimBytes, &ucp[trgCount] ) ;
      for ( UINT32 i = ZERO ; i < this->mimBytes ; ++i )
         ucp[trgCount++] = this->mimType[i] ;

      //* Prevent overflow of caller's buffer by truncating the description *
      //* text if necessary. This is rather unlikely ever to happen.        *
      if ( this->txtBytes >= (gsMAXBYTES - this->maxTHRESHOLD) )
      {
         gs = this->txtDesc ;
         while ( (UINT32)(gs.utfbytes()) >= (gsMAXBYTES - this->maxTHRESHOLD) )
            gs.limitChars( (gs.gschars()) - 2 ) ;
         gs.copy( this->txtDesc, gs.utfbytes() - 1 ) ;
         this->txtBytes = (gs.utfbytes()) - 1 ;
      }
         
      trgCount += this->intConv( this->txtBytes, &ucp[trgCount] ) ;
      for ( UINT32 i = ZERO ; i < this->txtBytes ; ++i )
         ucp[trgCount++] = this->txtDesc[i] ;
      trgCount += this->intConv( this->iWidth,    &ucp[trgCount] ) ;
      trgCount += this->intConv( this->iHeight,   &ucp[trgCount] ) ;
      trgCount += this->intConv( this->iCDepth,   &ucp[trgCount] ) ;
      trgCount += this->intConv( this->iCIndex,   &ucp[trgCount] ) ;
      trgCount += this->intConv( this->picEBytes, &ucp[trgCount] ) ;
      return trgCount ;
   }

   //* Convert the header data from the source binary input stream into    *
   //* formatted setup data and store it in our data members.              *
   //* - Caller has verified that the source data begins with either:      *
   //*   - the standard "METADATA_BLOCK_PICTURE" comment name or the       *
   //*   - the obsolete "COVERART" comment name.                           *
   //*   AND that the equals character ('=') delimeter is present.         *
   //* - Header source data continues through the integer representing the *
   //*   (encoded) image size.                                             *
   //* Returns the number of setup bytes scanned and decoded.              *
   //*   - The return values should index the beginning of the image data. *
   UINT32 decodeHeader ( const uint8_t* src, UINT32 vectorSize )
   {
      UINT32 i = ZERO,              // data index
             srcCount = ZERO ;      // return value
      
      bool stdFormat = true ;       // 'false' if obsolete encoding format

      this->reset () ;              // initialize our data members

      //* Determine whether we have the standard or the obsolete format.*
      while ( (src[srcCount] != '=') && (srcCount < vectorSize) )
         ++srcCount ;
      ++srcCount ;                  // step over the '=' character
      if ( srcCount < ivSIZE )  // number of characters in standard comment name
         stdFormat = false ;

      //* Standard Ogg_Image formatting:                        *
      //* 1) 'picBytes' is unknown at this time so set to ZERO. *
      //* 2) 'picExpl' is left uninitialized.                   *
      //* 3) All other fields are read from the input stream.   *
      //* Note that the 'mimType' field may contains a URL      *
      //* instead of a MIME type; however, this application does*
      //* not support references to externally-located images.  *
      if ( stdFormat )
      {
         //* Store the overall size of the vector *
         this->totBytes = vectorSize ;
         this->picType = this->intConv( &src[srcCount] ) ;
         srcCount += 4 ;
         this->mimBytes = this->intConv( &src[srcCount] ) ;
         srcCount += 4 ;
         for ( i = ZERO ; (i < this->mimBytes) && (i < (gsMAXBYTES - 1)) ; ++i )
            this->mimType[i] = src[srcCount++] ;
         this->mimType[i] = NULLCHAR ;    // terminate the string for display
         this->txtBytes = this->intConv( &src[srcCount] ) ;
         srcCount += 4 ;
         for ( i = ZERO ; i < this->txtBytes ; ++i )
            this->txtDesc[i] = src[srcCount++] ;
         this->txtDesc[i] = NULLCHAR ;    // terminate the string for display
         this->iWidth   = this->intConv( &src[srcCount] ) ;
         srcCount += 4 ;
         this->iHeight  = this->intConv( &src[srcCount] ) ;
         srcCount += 4 ;
         this->iCDepth  = this->intConv( &src[srcCount] ) ;
         srcCount += 4 ;
         this->iCIndex  = this->intConv( &src[srcCount] ) ;
         srcCount += 4 ;
         this->picEBytes = this->intConv( &src[srcCount] ) ;
         srcCount += 4 ;
         // srcCount should now be indexing the encoded binary picture data
      }

      //* Obsolete image formatting. Convert to standard format.*
      //* 1) 'picEBytes' Encoded image size is                  *
      //*     vectorSize - commentName("COVERART=" == 9 bytes)  *
      //* 2) Overall vector size == encoded image size +        *
      //*    minHEADERSIZE + default description.               *
      //* 3) picType == ZERO (unknown)                          *
      //* 4) mimType                                            *
      //*    a) length == 6                                     *
      //*    b) string == "image/"                              *
      //*    c) Determining MIME type (for future release):     *
      //*       Because the image data are Base64-encoded, it   *
      //*       would be inconvenient to determine the actual   *
      //*       binary sequence for an image. However, we can   *
      //*       empirically determe what the encoded beginning  *
      //*       sequence would be and compare against it.       *
      //*        i) JPEG == "image/jpeg"                        *
      //*           A JPEG file begins with:                    *
      //*                   FF D8 FF E0 nn nn J F I F           *
      //*           Base64: /9j/....SkZJ....                    *
      //*       ii) PNG == "image/png"                          *
      //*           A PNG file begins with:                     *
      //*                   89 P N G 0D 0A 1A 0A                *
      //*           Base64: iVBORw0K....                        *
      //* 5) description length == length of default description*
      //* 6) description (see contents of 'gs' below.           *
      //* 7) iWidth, iHeight, iCDepth and iCIndex == ZERO       *
      //* 8) 'picBytes' is unknown at this time so set to ZERO. *
      //* 9) 'picExpl' is left uninitialized.                   *
      else
      {
// UNDER CONSTRUCTION - NOT VERIFIED
         this->picEBytes = vectorSize - (oldivSIZE + 1) ;  // size of encoded image

         //* Calculate overall size of the converted vector.*
         gString gs( "Converted from obsolete \"COVERART\" vector." ) ;
         this->txtBytes = gs.utfbytes() - 1 ;
         this->totBytes = this->picEBytes + minHEADERSIZE + this->txtBytes ;
         gs.copy( this->txtDesc, gsMAXBYTES ) ;
         gs = mimeUnk ;
         gs.copy( this->mimType, gsMAXBYTES ) ;
      }
      return srcCount ;
   }  // decodeHeader

   //* Convert an ASCII text stream encoded as "Base64" to a binary byte *
   //* stream according to the MIME "Base64" encoding specification.     *
   //* Important Note:                                                   *
   //* ===============                                                   *
   //* If called multiple times to decode a single image, BE CERTAIN that*
   //* 'srcCount' is an even multiple of four(4) for all calls except the*
   //* final call. This will ensure that each encoded quartet of four    *
   //* bytes will be decoded as three complete decoded binary bytes.     *
   //* Failure to follow this pattern will result in corruption of the   *
   //* decoded image data.                                               *
   UINT32 decodeImage ( uint8_t* trg, const uint8_t* src, UINT32 srcCount )
   {
      uint8_t  b1, b2, b3, b4 ;        // 6-bit to 8-bit conversion
      uint32_t trgCount = ZERO,        // number of bytes written to target
             loopy    = srcCount / 4,  // loop counter for full, 4-byte units
             w = 0, x = 1, y = 2, z = 3 ;  // indices into source array
      bool   writeB2 = true, writeB3 = true ;
      //* Hash indices for reverse lookup *
//      uint8_t Hash_P = 0x0F, Hash_f = 0x1F, Hash_z = 0x33, HashEnd = (bt64BYTES - 1) ;

      while ( loopy > ZERO )
      {
         //* Get the 6-bit values corresponding to the source characters.*
         // Programmer's Note: This is very slow. We may institute hash 
         // indices after the algorithm is stable.
         for ( b1 = ZERO ; b1 < bt64BYTES ; ++b1 )
            if ( base64Table[b1] == src[w] )    break ;
         for ( b2 = ZERO ; b2 < bt64BYTES ; ++b2 )
            if ( base64Table[b2] == src[x] )    break ;
         for ( b3 = ZERO ; b3 < bt64BYTES ; ++b3 )
            if ( base64Table[b3] == src[y] )    break ;
         for ( b4 = ZERO ; b4 < bt64BYTES ; ++b4 )
            if ( base64Table[b4] == src[z] )    break ;

         //* Test for invalid characters (or padding) in the stream.*
         //* 1) If padding characters, just set value to 0x00.      *
         //*    This will result in correct output IF the padding   *
         //*    character is in a valid placeholder position.       *
         //* 2) If an invalid character, truncate the stream.       *
         //*    This _may_ discard one or two valid binary bytes,   *
         //*    but in a corrupted binary stream, this won't matter.*
         writeB2 = writeB3 = true ;
         if ( b1 == bt64BYTES )
            break ;     // invalid character code
         if ( b2 == bt64BYTES )
            break ;     // invalid character code
         if ( b3 == bt64BYTES)
         {
            if ( src[y] == base64PADCHAR )   // padding character(s)
            { b3 = b4 = ZERO ; writeB2 = writeB3 = false ; }
            else  break ;                    // invalid character code
         }
         else if ( b4 == bt64BYTES )
         {
            if ( src[z] == base64PADCHAR )   // padding character
            { b4 = ZERO ; writeB3 = false ; }
            else  break ;                    // invalid character code
         }

         trg[trgCount++] = (uint8_t)((b1 << 2) | (b2 >> 4)) ;
         if ( writeB2 )
            trg[trgCount++] = (uint8_t)((b2 << 4) | ((b3 & 0x3F) >> 2)) ;
         if ( writeB3 )
            trg[trgCount++] = (uint8_t)((b3 << 6) | b4) ;

         //* Increment the indices *
         w = z + 1 ;
         x = w + 1 ;
         y = x + 1 ;
         z = y + 1 ;
         --loopy ;
      }
      //* If source does not include correct padding bytes, then the last      *
      //* quartet of characters will be 1, 2, or 3 characters only.            *
      short remainder = srcCount % 4 ;
      if ( remainder > ZERO )
      {
         b2 = b3 = b4 = ZERO ;
         for ( b1 = ZERO ; b1 < bt64BYTES ; ++b1 )
            if ( base64Table[b1] == src[w] )    break ;
         if ( remainder > 1 )
         {
            for ( b2 = ZERO ; b2 < bt64BYTES ; ++b2 )
               if ( base64Table[b2] == src[x] )    break ;

            if ( remainder > 2 )
            {
               for ( b3 = ZERO ; b3 < bt64BYTES ; ++b3 )
                  if ( base64Table[b3] == src[y] )    break ;
            }
         }
         trg[trgCount++] = (uint8_t)((b1 << 2) | (b2 >> 4)) ;
         if ( remainder > 1 )
         {
            trg[trgCount++] = (uint8_t)((b2 << 4) | ((b3 & 0x3F) >> 2)) ;
            if ( remainder > 2 )
               trg[trgCount++] = (uint8_t)((b3 << 6) | b4) ;
         }
      }
      return trgCount ;
   }  // decodeImage()

   //* Convert a binary byte stream to ASCII text according to the *
   //* MIME "Base64" encoding specification.                       *
   UINT32 encodeImage ( uint8_t* trg, const uint8_t* src, UINT32 srcCount )
   {
      uint8_t  b1, b2, b3, b4 ;        // 8-bit to 6-bit conversion
      uint32_t trgCount = ZERO,        // number of bytes written to target
             loopy    = srcCount / 3,  // loop counter for full, 3-byte units
             x = 0, y = 1, z = 2 ;  // indices into source array

      while ( loopy > ZERO )
      {
         //* Convert three(3) 8-bit source bytes to four(4) 6-bit indices *
         b1 = (src[x] >> 2) ;
         b2 = ((src[x] & 0x03) << 4) | (src[y] >> 4) ;
         b3 = ((src[y] & 0x0F) << 2) | (src[z] >> 6) ;
         b4 = (src[z] & 0x3F) ;

         //* Do the table lookup and write results to target *
         trg[trgCount++] = base64Table[b1] ;
         trg[trgCount++] = base64Table[b2] ;
         trg[trgCount++] = base64Table[b3] ;
         trg[trgCount++] = base64Table[b4] ;

         //* Increment the indices *
         x = z + 1 ;
         y = x + 1 ;
         z = y + 1 ;
         --loopy ;
      }

      //* If there is a partial source triad, convert and pad *
      //* to create a full four bytes for output to target.   *
      if ( srcCount % 3 )
      {
         //* First character of incomplete source triad *
         b1 = (src[x] >> 2) ;
         trg[trgCount++] = base64Table[b1] ;
         b2 = ((src[x] & 0x03) << 4) ;
         //* Second character of triad? *
         if ( (srcCount % 3) > 1 )
         {
            b2 |= (src[y] >> 4) ;
            b3 = ((src[y] & 0x0F) << 2) ;
            trg[trgCount++] = base64Table[b2] ;
            trg[trgCount++] = base64Table[b3] ;
         }
         else
         {
            trg[trgCount++] = base64Table[b2] ;
            trg[trgCount++] = base64PADCHAR ;
         }
         trg[trgCount++] = base64PADCHAR ;
      }
      return trgCount ;
   }  // encodeImage

   //*** Data Members ***
   char picPath[gsMAXBYTES] ;       // path/filename of external image file
   char txtDesc[gsMAXBYTES] ;       // text description of image (optional)
   char mimType[gsMAXBYTES] ;       // MIME type: generally "image/jpg" or 
                                    //   "image/png" or "image/" (default)
                                    //   or possibly "-->url..."
   char picExpl[imgMAX_DESC] ;      // explanation of 'picType'. Note: Array of
                                    // pre-defined description strings: pType[][x]
   UINT32 picType ;                 // picture type code (see pType[][PIC_TYPES])
   UINT32 mimBytes ;                // length of 'mimType' in bytes (not incl null)
   UINT32 txtBytes ;                // length of 'txtDesc' in bytes (not incl null)
   UINT32 iWidth ;                  // width of image in pixels
   UINT32 iHeight ;                 // height of image in pixels
   UINT32 iCDepth ;                 // color depth of image
   UINT32 iCIndex ;                 // color indexing of image
   UINT32 picBytes ;                // image size in bytes (binary)
   UINT32 picEBytes ;               // image size in bytes (Base64 encoded)
   UINT32 totBytes ;                // total size of Image Vector (not incl size integer)

   private:
   //* Comment Vector name for embedded images *
   const char*  ivNAME = "METADATA_BLOCK_PICTURE" ;
   const UINT32 ivSIZE = 22 ;    // characters in vector name
   const UINT32 oldivSIZE = 9 ;  // characters in obsolete "COVERART" vector name

   //* Maximum size of header without contents of 'txtDesc'.                   *
   //* == 4 * 9 integers + max length of MIME string plus 4 safety bytes.      *
   //* This is used to prevent the encoded header from exceeding gsMAXBYTES.   *
   const UINT32 maxTHRESHOLD = (4 * 9 + 10 + 4) ;
   //* Minimum size of header (no txtDesc, unknown MIME type).                 *
   //* comment name + '=' + 4 * 9 integers + "image/"
   const UINT32 minHEADERSIZE = (22 + 1 + 4 * 9 + 6) ;

} ;   // Ogg_Image

