//******************************************************************************
//* File       : TagMenu.cpp                                                   *
//* Author     : Mahlon R. Smith                                               *
//*              Copyright (c) 2016-2020 Mahlon R. Smith, The Software Samurai *
//*                  GNU GPL copyright notice located in Taggit.hpp            *
//* Date       : 03-Oct-2020                                                   *
//* Version    : (see AppVersion string)                                       *
//*                                                                            *
//* Description: This module contains the methods supporting the Taggit dialog,*
//* primarily (but not exclusively) related to implementation of actions which *
//* can be selected from the Menu Bar.                                         *
//*                                                                            *
//******************************************************************************
//* Version History (see Taggit.cpp).                                          *
//******************************************************************************

//****************
//* Header Files *
//****************
#include "Taggit.hpp"         // Taggit-class definitions and data
                              // plus general definitions and NcDialogAPI definition
#include "TagDialog.hpp"      // Text data displayed in the dialog windows


//**************
//* Local data *
//**************

//* Contains filesystem statistics  *
class fileSystemStats
{
   public:
   fileSystemStats()
   {
      seLinuxCode[0] = fsType[0] = 'u' ;
      seLinuxCode[1] = fsType[1] = 'n' ;
      seLinuxCode[2] = fsType[2] = 'k' ;
      seLinuxCode[3] = fsType[3] = NULLCHAR ;
      freeBytes = usedBytes = systemID = ZERO ;
      blockTotal = blockFree = blockAvail = inodeTotal = inodeAvail = ZERO ;
      fsTypeCode = ZERO ;
      nameLen = blockSize = fblockSize = ZERO ;
   }                          // 'stat' format code
   char     seLinuxCode[64] ; // %C
   char     fsType[64] ;      // %T
   UINT64   systemID ;        // %i (hex)
   UINT64   freeBytes ;       // (calculated: blockAvail * fblockSize)
   UINT64   usedBytes ;       // (calculated: (blockTotal - blockFree) * fblockSize)
   ULONG    blockTotal ;      // %b
   ULONG    blockFree ;       // %f
   ULONG    blockAvail ;      // %a
   ULONG    inodeTotal ;      // %c
   ULONG    inodeAvail ;      // %d
   UINT     fsTypeCode ;      // %t (hex)
   USHORT   nameLen ;         // %l
   USHORT   blockSize ;       // %s
   USHORT   fblockSize ;      // %S
} ;


//********************
//* Local prototypes *
//********************

//* Cmd_DuplicateField() callback *
static short cdfCallback ( const short currIndex, const wkeyCode wkey, 
                           bool firstTime = false ) ;
       static NcDialog* cdfDialog = NULL ;    // callback access to dialog
       static const tagData* cdfTData = NULL ;// callback access to member data
       static AppLang cdfLang = enLang ;      // callback access to UI language
//* Cmd_SetColorScheme() callback *
static short scsCallback ( const short currIndex, const wkeyCode wkey, 
                           bool firstTime = false ) ;
       static NcDialog* scsDialog = NULL ;   // callback access to dialog
//* Cmd_Popularimeter() callback *
static short cpCallback ( const short currIndex, const wkeyCode wkey, 
                          bool firstTime = false ) ;
       static NcDialog* cpDialog = NULL ;    // callback access to dialog
       static attr_t cpColor = ZERO ;        // message color attribute
       static winPos cpWp ;                  // message position
#if 0    // CURRENTLY UNUSED
//* Generic CTRL+R toggle callback *
short togCallback ( const short currIndex, const wkeyCode wkey, 
                    bool firstTime = false ) ;
       NcDialog* togDp = NULL ;              // callback access to dialog
       short togCtrl[MAX_DIALOG_CONTROLS] ;  // target control indices
#endif   // CURRENTLY UNUSED
static bool crtsReportTargetStats ( const gString& tmpPath, const char* trgPath,
                                    fileSystemStats& fsStats ) ;
//static void ceiMime ( EmbeddedImage* eiPtr, gString& gsmt, bool rtl ) ;
static void ceiMime ( id3v2_image& img, gString& gsmt, bool rtl ) ;

//******************************************************************************
//**                      'CMD' GROUP OF METHODS.                             **
//******************************************************************************

//*************************
//*     Cmd_SortList      *
//*************************
//******************************************************************************
//* Sort the records according to the cfgOpt.sortBy option.                    *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void Taggit::Cmd_SortList ( void )
{
   if ( (this->tData.sfCount > 1) && (this->cfgOpt.sortBy != sbNONE) )
   {
      gString  gsi, gsl, gsg, gss ;
      short lIndex, gIndex ;           // low and high limits search indices

      for ( lIndex = 0, gIndex = (this->tData.sfCount - 1) ; 
                               (gIndex-lIndex) > ZERO ; lIndex++, gIndex-- )
      {
         for ( short i = lIndex+1 ; i <= gIndex ; ++i )
         {
            if ( this->cfgOpt.sortBy == sbTITLE )
            {
               gsi = this->tData.sf[i].sfTag.field[tfTit2] ;
               gsl = this->tData.sf[lIndex].sfTag.field[tfTit2] ;
            }
            else if ( this->cfgOpt.sortBy == sbTRACK )
            {
               //* If single-digit track number, insert leading '0'. *
               gsi = this->tData.sf[i].sfTag.field[tfTrck] ;
               if ( ((gsi.gschars()) == 2) || (gsi.find( L'/' )) == 1 )
                  gsi.insert( L'0' ) ;
               gsl = this->tData.sf[lIndex].sfTag.field[tfTrck] ;
               if ( ((gsl.gschars()) == 2) || (gsl.find( L'/' )) == 1 )
                  gsl.insert( L'0' ) ;
            }
            else if ( this->cfgOpt.sortBy == sbALBUM )
            {
               gsi = this->tData.sf[i].sfTag.field[tfTalb] ;
               gsl = this->tData.sf[lIndex].sfTag.field[tfTalb] ;
            }
            else if ( this->cfgOpt.sortBy == sbARTIST )
            {
               gsi = this->tData.sf[i].sfTag.field[tfTpe1] ;
               gsl = this->tData.sf[lIndex].sfTag.field[tfTpe1] ;
            }
            else  // ( this->cfgOpt.sortBy == sbFNAME ) (default)
            {
               gsi = this->tData.sf[i].sfName ;
               gsl = this->tData.sf[lIndex].sfName ;
            }
            if ( (gsi.compare( gsl )) < ZERO )
            {
               this->Cmd_SwapItems ( i, lIndex ) ;
               gss = gsl ;    // swap the test data
               gsl = gsi ;
               gsi = gss ;
            }

            if ( i < gIndex )
            {
               if ( this->cfgOpt.sortBy == sbTITLE )
               {
                  gsg = this->tData.sf[gIndex].sfTag.field[tfTit2] ;
               }
               else if ( this->cfgOpt.sortBy == sbTRACK )
               {
                  gsg = this->tData.sf[gIndex].sfTag.field[tfTrck] ;
                  if ( ((gsg.gschars()) == 2) || (gsg.find( L'/' )) == 1 )
                     gsg.insert( L'0' ) ;
               }
               else if ( this->cfgOpt.sortBy == sbALBUM )
               {
                  gsg = this->tData.sf[gIndex].sfTag.field[tfTalb] ;
               }
               else if ( this->cfgOpt.sortBy == sbARTIST )
               {
                  gsg = this->tData.sf[gIndex].sfTag.field[tfTpe1] ;
               }
               else  // ( this->cfgOpt.sortBy == sbFNAME ) (default)
               {
                  gsg = this->tData.sf[gIndex].sfName ;
               }
               if ( (gsg.compare( gsi )) < ZERO )
               {
                  this->Cmd_SwapItems ( gIndex, i ) ;
                  gsi = gsg ;    // swap test data

                  if ( (gsi.compare( gsl )) < ZERO )
                  {
                     this->Cmd_SwapItems ( i, lIndex ) ;
                  }
               }
            }
         }
      }
   }

}  //* End Cmd_SortList() *

//*************************
//*     Cmd_SwapItems     *
//*************************
//******************************************************************************
//* For the filename and metadata arrays, exchange item contents.              *
//* User may move an item up or down through the list, OR                      *
//* method is called by the sorting algorithm.                                 *
//*                                                                            *
//* NOTE: Does not redraw the display data.                                    *
//*                                                                            *
//* Input  : aIndex : index of first item                                      *
//*          bIndex : index of second item                                     *
//*                                                                            *
//* Returns: 'true'  if successful                                             *
//*          'false' if index out-of-range OR if a == b                        *
//******************************************************************************

bool Taggit::Cmd_SwapItems ( short aIndex, short bIndex )
{
   bool status = false ;

   if ( (aIndex >= ZERO && aIndex < this->tData.sfCount) &&
        (bIndex >= ZERO && bIndex < this->tData.sfCount) &&
        (aIndex != bIndex) )
   {
      SrcFile sfSwap = this->tData.sf[aIndex] ;
      this->tData.sf[aIndex] = this->tData.sf[bIndex] ;
      this->tData.sf[bIndex] = sfSwap ;
      status = true ;      // return success
   }
   return status ;

}  //* End Cmd_SwapItems() *

//*************************
//* Cmd_ReportTargetStats *
//*************************
//******************************************************************************
//* Report target filesystem free space.                                       *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void Taggit::Cmd_ReportTargetStats ( void )
{
   const short dlgROWS  = 16,             // dialog lines
               dlgCOLS  = 57,             // dialog columns
               dlgY     = fieldwinY + 1,  // dialog position
               dlgX     = fieldwinX + 1 ;
   attr_t dColor = this->cs.sd,           // dialog interior color
          hColor = this->cs.em ;          // bold dialog text
   AppLang lang = this->cfgOpt.appLanguage ; // language index
   gString gs, gsfsType, gsfsId, gstBlk,  // text formatting
           gsfBlk, gsaBlk, gsuBytes, gsfBytes,
           tmpPath ;                      // temp filespec
   fileSystemStats fsStats ;

   const char* Labels[][10] = 
   {
      {  // English
         "  File System Information  ",   // (0) dialog title
         "Target filesystem information:",// (1) header
         "File System Type:",             // (2) fs type
         "  File System ID:",             // (3) fs ID
         "Total Blocks:",                 // (4) total blocks
         " Free Blocks:",                 // (5) free blocks
         "Avail Blocks:",                 // (6) avail blocks
         "  Used Bytes:",                 // (7) used bytes
         "  Free Bytes:",                 // (8) free bytes
         "Error, data unavailable.",      // (9) system error message
      },
      {  // Espanol
         "  Información del Sistema de Archivos  ",         // (0) dialog title
         "Información del sistema de archivos de destino:", // (1) header
         "      Tipo de sistema de archivos:",              // (2) fs type
         "Identidad del sistema de archivos:",              // (3) fs ID
         "      Bloques de memoria total:",                 // (4) total blocks
         "     Bloques de memoria libres:",                 // (5) free blocks
         "Bloques de memoria disponibles:",                 // (6) avail blocks
         "    Número de bytes utilizados:",                 // (7) used bytes
         "   Número de bytes disponibles:",                 // (8) free bytes
         "Falta, los datos no están disponibles.",          // (9) system error message
      },
      {  // Zhongwen
         "       文件系统信息        ",     // (0) dialog title
         "目标文件系统信息：",               // (1) header
         "文件系统类型  ：",                // (2) fs type
         "文件系统标识符：",                // (3) fs ID
         "      总块数：",                 // (4) total blocks
         "  自由块计数：",                 // (5) free blocks
         "    可用块数：",                 // (6) avail blocks
         "使用的字节数：",                  // (7) used bytes
         "  自由字节数：",                 // (8) free bytes
         "错误，数据不可用。",              // (9) system error message
      },
      {  // TiengView
         "  Thông tin hệ thống tệp  ",          // (0) dialog title
         "Thông tin về hệ thống tập tin đích:", // (1) header
         "Loại hệ thống tập tin:",              // (2) fs type
         "ID hệ thống tệp:",                    // (3) fs ID
         "      Tổng số khối bộ nhớ:",          // (4) total blocks
         "       Khối không sử dụng:",          // (5) free blocks
         "          Các khối có sẵn:",          // (6) avail blocks
         "Byte dữ liệu đang sử dụng:",          // (7) used bytes
         "      Byte dữ liệu sẵn có:",          // (8) free bytes
         "Lỗi, dữ liệu không có sẵn.",          // (9) system error message
      },
   } ;
   const char* dlgText[] = 
   {
      Labels[lang][0],
      Labels[lang][1],
      "",
      gsfsType.ustr(),
      gsfsId.ustr(),
      "",
      gstBlk.ustr(),
      gsfBlk.ustr(),
      gsaBlk.ustr(),
      gsuBytes.ustr(),
      gsfBytes.ustr(),
      NULL
   } ;
   attr_t dlgAttr[] = 
   { hColor, dColor, dColor, dColor, dColor, dColor, dColor, 
     dColor, dColor, dColor, dColor } ;

   //* Get the filesystem stats *
   this->CreateTempname ( tmpPath ) ;  // create a temp file filespec
   if ( (crtsReportTargetStats ( tmpPath, this->cfgOpt.appPath, fsStats )) != false )
   {
      wchar_t revbuff[32] ;
      const wchar_t *wPtr ;
      short wCount, i, j ;

      //* Format the data *
      // Programmer's Note: For RTL output, we write the field labels as 
      // plain RTL text; however, we reverse the order of the numeric data 
      // characters so the numbers will be displayed as LTR.
      gs.compose( "%0X - %s", &fsStats.fsTypeCode, fsStats.fsType ) ;
      if ( this->cfgOpt.rtl )
      {
         wPtr = gs.gstr( wCount ) ;
         for ( i = wCount - 2, j = ZERO ; i >= ZERO ; --i, ++j )
            revbuff[j] = wPtr[i] ;
         revbuff[j] = NULLCHAR ;
         gs = revbuff ;
      }
      gsfsType.compose( "%s %S", Labels[lang][2], gs.gstr() ) ;
      gs.compose( "%llX", &fsStats.systemID ) ;
      if ( this->cfgOpt.rtl )
      {
         wPtr = gs.gstr( wCount ) ;
         for ( i = wCount - 2, j = ZERO ; i >= ZERO ; --i, ++j )
            revbuff[j] = wPtr[i] ;
         revbuff[j] = NULLCHAR ;
         gs = revbuff ;
      }
      gsfsId.compose( "%s %S", Labels[lang][3], gs.gstr() ) ;
      gs.formatInt( fsStats.blockTotal, 13, true ) ;
      if ( this->cfgOpt.rtl )
      {
         wPtr = gs.gstr( wCount ) ;
         for ( i = wCount - 2, j = ZERO ; i >= ZERO ; --i, ++j )
            revbuff[j] = wPtr[i] ;
         revbuff[j] = NULLCHAR ;
         gs = revbuff ;
      }
      gstBlk.compose( "%s %S", Labels[lang][4], gs.gstr() ) ;
      gs.formatInt( fsStats.blockFree, 13, true ) ;
      if ( this->cfgOpt.rtl )
      {
         wPtr = gs.gstr( wCount ) ;
         for ( i = wCount - 2, j = ZERO ; i >= ZERO ; --i, ++j )
            revbuff[j] = wPtr[i] ;
         revbuff[j] = NULLCHAR ;
         gs = revbuff ;
      }
      gsfBlk.compose( "%s %S", Labels[lang][5], gs.gstr() ) ;
      gs.formatInt( fsStats.blockAvail, 13, true ) ;
      if ( this->cfgOpt.rtl )
      {
         wPtr = gs.gstr( wCount ) ;
         for ( i = wCount - 2, j = ZERO ; i >= ZERO ; --i, ++j )
            revbuff[j] = wPtr[i] ;
         revbuff[j] = NULLCHAR ;
         gs = revbuff ;
      }
      gsaBlk.compose( "%s %S", Labels[lang][6], gs.gstr() ) ;
      gs.formatInt( fsStats.usedBytes, 8 ) ;
      if ( this->cfgOpt.rtl )
      {
         wPtr = gs.gstr( wCount ) ;
         for ( i = wCount - 2, j = ZERO ; i >= ZERO ; --i, ++j )
            revbuff[j] = wPtr[i] ;
         revbuff[j] = NULLCHAR ;
         gs = revbuff ;
      }
      gsuBytes.compose( "%s %S", Labels[lang][7], gs.gstr() ) ;
      gs.formatInt( fsStats.freeBytes, 8 ) ;
      if ( this->cfgOpt.rtl )
      {
         wPtr = gs.gstr( wCount ) ;
         for ( i = wCount - 2, j = ZERO ; i >= ZERO ; --i, ++j )
            revbuff[j] = wPtr[i] ;
         revbuff[j] = NULLCHAR ;
         gs = revbuff ;
      }
      gsfBytes.compose( "%s %S", Labels[lang][8], gs.gstr() ) ;
   }
   else
      gsfsType = Labels[lang][9] ;
   this->DeleteTempname ( tmpPath ) ;     // delete the temp file

   genDialog gd( dlgText, dColor, dlgROWS, dlgCOLS, dlgY, dlgX, dlgAttr, 
                 this->cfgOpt.rtl, attrDFLT, attrDFLT, okText[lang] ) ;

   this->dPtr->InfoDialog ( gd ) ;     // alert user

}  //* End Cmd_ReportTargetStats() *

//*************************
//* crtsReportTargetStats *
//*************************
//******************************************************************************
//* Non-member Method                                                          *
//* -----------------                                                          *
//* Report target filesystem free space.                                       *
//*                                                                            *
//* Input  : trgPath  : target filespec                                        *
//*          tmpPath  : filespec for temporary file                            *
//*          fsStats  : (by reference) receives the stat data                  *
//*                                                                            *
//* Returns: 'true' if successful, 'false' if access or other error            *
//******************************************************************************

static bool crtsReportTargetStats ( const gString& tmpPath, const char* trgPath,
                                    fileSystemStats& fsStats )
{
   static const char fssCmd[] = 
         "stat -f --printf=\"%b %f %a %c %d %s %S %l %i %t %T %C \n\"" ;
   char    cmdBuff[MAX_PATH*3] ;
   bool status = false ;

   snprintf ( cmdBuff, (MAX_PATH*3), "%s \"%s\" 1>\"%s\" 2>/dev/null", 
              fssCmd, trgPath, tmpPath.ustr() );

   system ( cmdBuff ) ;

   //* Read the only line in the file *
   ifstream ifsIn ( tmpPath.ustr(), ifstream::in ) ;
   if ( ifsIn.is_open() )
   {
      ifsIn.getline ( cmdBuff, (MAX_PATH*3) ) ;
      ifsIn.close () ;     // close the temp file

      gString fsData( cmdBuff ) ;
      swscanf ( fsData.gstr(), L"%lu %lu %lu %lu %lu %hu %hu %hu %llX %x %63s %63s", 
         &fsStats.blockTotal, &fsStats.blockFree,  &fsStats.blockAvail, 
         &fsStats.inodeTotal, &fsStats.inodeAvail,
         &fsStats.blockSize,  &fsStats.fblockSize, &fsStats.nameLen,
         &fsStats.systemID,   &fsStats.fsTypeCode,
         fsStats.fsType, fsStats.seLinuxCode ) ;

      fsStats.freeBytes = (UINT64)fsStats.fblockSize * fsStats.blockAvail ;
      fsStats.usedBytes = (UINT64)fsStats.fblockSize * 
                                  (fsStats.blockTotal - fsStats.blockFree) ;

      status = true ;      // returning good news
   }
   return status ;

}  //* End crtsReportTargetStats() *

//*************************
//*   Cmd_ActiveFields    *
//*************************
//******************************************************************************
//* Set all tag fields as active. This allows display and edit of all tags     *
//* defined by the id3v2 standard.                                             *
//*                                                                            *
//* This affects only the text fields. The embedded-image flags and data are   *
//* not modified.                                                              *
//*                                                                            *
//* Input  : enable : 'true'  enable all tag fields                            *
//*                   'false' disable all but the 'Title' field                *
//*                           (Note that while this is a valid option, it is)  *
//*                           (not currently used by this application.      )  *
//*          redraw : 'true'  after enabling the tag fields, redraw the        *
//*                           contents of the metadata window                  *
//*                   'false' enable the tag fields, but do not update display *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void Taggit::Cmd_ActiveFields ( bool enable, bool redraw )
{
   //* Set all tag fields as enabled *
   this->tData.enable_all ( enable ) ;

   //* If specified, redraw the display *
   if ( redraw )
   {
      this->tData.sffOffset = ZERO ;         // move to first field
      this->UpdateMetadataWindow () ;        // update the window
   }

}  //* End Cmd_ActiveFields()

//*************************
//*  Cmd_SetColorScheme   *
//*************************
//******************************************************************************
//* Change the application's color scheme.                                     *
//*  1) Ask the user which color scheme to use.                                *
//*  2) Reinitialize the color-scheme data.                                    *
//*  3) Call ResizeDialog() to close the dialog and open it with the new       *
//*     color scheme.                                                          *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void Taggit::Cmd_SetColorScheme ( void )
{
   const short dlgROWS = 21,              // dialog size
               dlgCOLS = 80,
               dlgY    = fieldwinY + 1,   // dialog position
               dlgX    = fieldwinX + 1 ;
   attr_t dColor = this->cs.sd,           // dialog interior color
          hColor = this->cs.em ;          // bold dialog text
   AppLang lang = this->cfgOpt.appLanguage ; // language index
   gString gs ;                           // text formatting

   enum scsControls : short 
   { savePB = ZERO, cancelPB, selSB, scsCTRLS } ;

   //* Control labels *
   const char* Labels[][scsCTRLS + 1] = 
   {
      {  // English
         "     SAVE     ",             // 'save' pushbutton
         "    CANCEL    ",             // 'cancel' pushbutton
         "Select Color",               // 'select' scrollbox
         "  Set Application Color Scheme  ",  // Dialog title
      },
      {  // Espanol
         "   GUARDAR    ",             // 'save' pushbutton
         "   CANCELAR   ",             // 'cancel' pushbutton
         "Selecciona el Color",        // 'select' scrollbox
         "  Establecer el Esquema de Color de la Aplicación  ",  // Dialog title
      },
      {  // Zhongwen
         "     保存     ",              // 'save' pushbutton
         "     取消     ",              // 'cancel' pushbutton
         "选择颜色",                    // 'select' scrollbox
         "  设置应用颜色方案  ",         // Dialog title
      },
      {  // TiengViet
         "    Để Dành   ",             // 'save' pushbutton
         "    Hủy Bỏ    ",             // 'cancel' pushbutton
         " Chọn Màu",                 // 'select' scrollbox
         "  Đặt Bảng màu ứng Dụng  ",  // Dialog title
      },
   } ;

   //* Color selection scrollbox items *
   const short selColorsWIDTH = 18 ;
   const short selColorsITEMS = 9 ;
   const char selColors[][selColorsITEMS][selColorsWIDTH + 6] = 
   {
      {  // English
         " Black           ",
         " Red             ",
         " Green           ",
         " Brown           ",
         " Blue            ",
         " Magenta         ",
         " Cyan            ",
         " Gray            ",
         " Default         ",
      },
      {  // Espanol
         " Negro           ",
         " Rojo            ",
         " Verde           ",
         " Marrón          ",
         " Azul            ",
         " Rosa Oscuro     ",
         " Ciánico         ",
         " Gris            ",
         " Predeterminado  ",
      },
      {  // Zhongwen
         " 黑色            ",
         " 红色            ",
         " 绿色            ",
         " 棕色            ",
         " 蓝色            ",
         " 品红色          ",
         " 青色            ",
         " 灰色            ",
         " 默认颜色        ",
      },
      {  // TiengViet
         " Màu đen         ",
         " Màu đỏ          ",
         " Màu xanh lá cây ",
         " Màu nâu         ",
         " Màu xanh da trời",
         " Màu đỏ tươi     ",
         " Màu lục lam     ",
         " Màu xám         ",
         " Mặc định        ",
      }
   } ;
   //* Control definitions *
   InitCtrl ic[scsCTRLS] =             // array of dialog control info
   {
   {  //* 'Save' pushbutton - - - - - - - - - - - - - - - - - - - - -   savePB *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(dlgROWS - 2),           // ulY:       upper left corner in Y
      short(dlgCOLS / 2 - 15),      // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      14,                           // cols:      control columns
      Labels[lang][savePB],         // dispText:  
      this->cs.pn,                  // nColor:    non-focus color
      this->cs.pf,                  // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "",                           // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[cancelPB],                // nextCtrl:  link in next structure
   },
   {  //* 'Cancel' pushbutton - - - - - - - - - - - - - - - - - - -   cancelPB *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      ic[savePB].ulY,               // ulY:       upper left corner in Y
      short(ic[savePB].ulX + ic[savePB].cols + 2), // ulX: upper left corner in X
      1,                            // lines:     (n/a)
      14,                           // cols:      control columns
      Labels[lang][cancelPB],       // dispText:  
      this->cs.pn,                  // nColor:    non-focus color
      this->cs.pf,                  // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "",                           // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
   &ic[selSB]                    // nextCtrl:  link in next structure
   },
   { //* 'Select Color' Scrollbox - - - - - - - - - - - - - - - - -      selSB *
      dctSCROLLBOX,                 // type:      define a scrolling-data control
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      3,                            // ulY:       upper left corner in Y
      2,                            // ulX:       upper left corner in X
      (selColorsITEMS + 2),         // lines:     control lines
      (selColorsWIDTH + 1),         // cols:      control columns
      (char*)&selColors[lang],      // dispText:  text-data array
      this->cs.sd,                  // nColor:    non-focus border color
      this->cs.pf,                  // fColor:    focus border color
      tbPrint,                      // filter:    (n/a)
      Labels[lang][selSB],          // label:     label text
      -1,                           // labY:      label offset
      1,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      selColorsITEMS,               // scrItems:  number of elements in text/color arrays
      ZERO,                         // scrSel:    index of initial highlighted element
      this->cs.menu,                // scrColor:  color-attribute list
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      NULL                          // nextCtrl:  link in next structure
   },
   } ;

   //* Dialog definition *
   InitNcDialog dInit( dlgROWS,         // number of display lines
                       dlgCOLS,         // number of display columns
                       dlgY,            // Y offset from upper-left of terminal 
                       dlgX,            // X offset from upper-left of terminal 
                       NULL,            // dialog title
                       ncltDUAL,        // border line-style
                       dColor,          // border color attribute
                       dColor,          // interior color attribute
                       ic               // pointer to list of control definitions
                     ) ;

   this->dPtr->SetDialogObscured () ;  // save parent dialog


   //* Open the dialog *
   NcDialog* dp = new NcDialog ( dInit ) ;
   scsDialog = dp ;        // give callback method access to the dialog
   if ( (dp->OpenWindow()) == OK )
   {
      //* Set the dialog title *
      dp->SetDialogTitle ( Labels[lang][scsCTRLS], hColor ) ;

      dp->RefreshWin () ;

      //* Establish a call-back method for updating Textbox data.*
      dp->EstablishCallback ( &scsCallback ) ;

      //* Interact with the user *
      uiInfo   Info ;                     // user interface data returned here
      short    icIndex = ZERO ;           // index of control with focus
      bool     done = false ;             // loop control
      while ( ! done )
      {
         if ( ic[icIndex].type == dctPUSHBUTTON )
         {
            if ( Info.viaHotkey )
               Info.HotData2Primary () ;
            else
               icIndex = dp->EditPushbutton ( Info ) ;
            if ( Info.dataMod != false )
            {
               if ( Info.ctrlIndex == savePB )
               {
                  short selItem = scsDialog->GetScrollboxSelect ( selSB ) ;
                  switch ( selItem )
                  {
                     case 0: this->cs.scheme = ncbcBK ; break ;   // Black   color scheme
                     case 1: this->cs.scheme = ncbcRE ; break ;   // Red     color scheme
                     case 2: this->cs.scheme = ncbcGR ; break ;   // Green   color scheme
                     case 3: this->cs.scheme = ncbcBR ; break ;   // Brown   color scheme
                     case 4: this->cs.scheme = ncbcBL ; break ;   // Blue    color scheme
                     case 5: this->cs.scheme = ncbcMA ; break ;   // Magenta color scheme
                     case 6: this->cs.scheme = ncbcCY ; break ;   // Cyan    color scheme
                     case 7: this->cs.scheme = ncbcGY ; break ;   // Gray    color scheme
                     case 8: this->cs.scheme = ncbcBK ; break ;   // Default color scheme
                  } ;
                  this->InitColorScheme () ;
                  this->ResizeDialog ( true ) ;
               }
               done = true ;
            }
         }

         else if ( ic[icIndex].type == dctSCROLLBOX )
         {
            //* Allow user to modify the selected member of the scroll box *
            //* Returns when edit is complete.                             *
            icIndex = dp->EditScrollbox ( Info ) ;

         }

         //* Move focus to appropriate control *
         if ( ! done && ! Info.viaHotkey )
         {
            if ( Info.keyIn == nckSTAB )
               icIndex = dp->PrevControl () ; 
            else
               icIndex = dp->NextControl () ;
         }
      }  // while()
   }
   if ( dp != NULL )                         // close the window
      delete ( dp ) ;

   this->dPtr->RefreshWin () ;               // restore parent dialog

}  //* End Cmd_SetColorScheme() *

//*************************
//*      scsCallback      *
//*************************
//******************************************************************************
//* This is a callback method for display of a color-scheme sample for the     *
//* Cmd_SetColorScheme() method.                                               *
//* Input  : currIndex: index of control that currently has focus              *
//*          wkey     : user's key input data                                  *
//*          firstTime: the EstablishCallback() method calls this method once  *
//*                     with firstTime==true, to perform any required          *
//*                     initialization. Subsequently, the NcDialog class       *
//*                     always calls with firstTime==false.                    *
//* Returns: OK                                                                *
//******************************************************************************

static short scsCallback ( const short currIndex, const wkeyCode wkey, bool firstTime )
{
   const char* txtItem[] = 
   {
      "  Taggit - v:0.1.00  "
   } ;
   const short BOX_ROWS = 14 ;
   const short BOX_COLS = 55 ;
   winPos wp( 3, 23 ), wpx ;
   gString gs ;
   short selItem = scsDialog->GetScrollboxSelect ( 2 ) ;

   //* Set sample colors - keep in synch with settings in InitColorScheme() *
   attr_t bb, pf, tn, title, saved, unsaved ;
   switch ( selItem )
   {
      case 0:              // Black color scheme
         bb = nc.bk ;
         pf = nc.reG ;
         tn = nc.bw ;
         title = nc.brbk ;
         saved = nc.grbk ;
         unsaved = nc.rebk ;
         break ;
      case 1:              // Red color scheme
         bb = nc.reR ;
         pf = nc.maG ;
         tn = nc.bw ;
         title = nc.brre ;
         saved = nc.grre ;
         unsaved = nc.mare ;
         break ;
      case 2:              // Green color scheme
         bb = nc.grR ;
         pf = nc.reG ;
         tn = nc.bw ;
         title = nc.brgr ;
         saved = nc.brgr ;
         unsaved = nc.regr ;
         break ;
      case 3:              // Brown color scheme
         bb = nc.brR ;
         pf = nc.maG ;
         tn = nc.bw ;
         title = nc.cybr ;
         saved = nc.grbr ;
         unsaved = nc.rebr ;
         break ;
      case 4:              // Blue color scheme
         bb = nc.blR ;
         pf = nc.reG ;
         tn = nc.bw ;
         title = nc.brbl ;
         saved = nc.grbl ;
         unsaved = nc.rebl ;
         break ;
      case 5:              // Magenta color scheme
         bb = nc.maR ;
         pf = nc.reG ;
         tn = nc.bw ;
         title = nc.brma ;
         saved = nc.grma ;
         unsaved = nc.rema ;
         break ;
      case 6:              // Cyan color scheme
         bb = nc.cyR ;
         pf = nc.reG ;
         tn = nc.bw ;
         title = nc.brcy ;
         saved = nc.grcy ;
         unsaved = nc.recy ;
         break ;
      case 7:              // Gray color scheme
         bb = nc.gyR ;
         pf = nc.reG ;
         tn = nc.bw ;
         title = nc.blgy ;
         saved = nc.grgy ;
         unsaved = nc.regy ;
         break ;
      case 8:              // Default color scheme
         bb = nc.cyR ;
         pf = nc.reG ;
         tn = nc.bw ;
         title = nc.brbl ;
         saved = nc.grcy ;
         unsaved = nc.recy ;
         break ;
   } ;

   //* Be safe... *
   if ( scsDialog != NULL )
   {
      //*************************
      //* First-time processing *
      //*************************
      if ( firstTime != false )
      {
         /* Nothing to do. */
      }

      //***********************
      //* Standard processing *
      //***********************
      wpx.xpos = scsDialog->DrawBox ( wp.ypos, wp.xpos, BOX_ROWS, BOX_COLS, 
                                 bb, txtItem[0], ncltDUAL ) ;
      scsDialog->WriteString ( wp.ypos, wpx.xpos, txtItem[0], title ) ;
      ++wp.ypos ; ++wp.xpos ;
      wpx = wp ;
      while ( (gs.gscols()) < (BOX_COLS - 2) )
         gs.append( L' ' ) ;
      gs.append( L'\n' ) ;
      while ( wpx.ypos < (wp.ypos + BOX_ROWS - 2) )
         wpx = scsDialog->WriteParagraph ( wpx, gs, bb ) ;
      LineDef lDef( ncltHORIZ, ncltDUAL, wp.ypos + 3, wp.xpos - 1, 
                    BOX_COLS, bb ) ;
      scsDialog->DrawLine ( lDef ) ;
      lDef.type = ncltVERT ;
      lDef.startX = wp.xpos + 24 ;
      lDef.length = BOX_ROWS - 4 ;

      scsDialog->WriteString ( wp.ypos, wp.xpos + 16, 
         "\u2584\u2584\u2584\u2584\u2584\u2584\u2584\u2584\u2584\u2584\u2584"
         "\u2584\u2584\u2584\u2584\u2584\u2584\u2584\u2584\u2584\u2584\u2584"
         "\u2584\u2584\u2584\u2584\u2584\u2584\u2584\u2584\u2584\u2584\u2584"
         "\u2584\u2584\u2584\u2584", saved ) ;
      ++wp.ypos ;
      wpx = scsDialog->WriteChar ( wp.ypos, wp.xpos + 16, 0x2588, saved ) ;
      wpx = scsDialog->WriteString ( wpx, "    File    ", bb ) ;
      wpx = scsDialog->WriteChar ( wpx, 0x2588, saved ) ;
      wpx = scsDialog->WriteString ( wpx, "    Edit    ", bb ) ;
      wpx = scsDialog->WriteChar ( wpx, 0x2588, saved ) ;
      wpx = scsDialog->WriteString ( wpx, "    View ", bb ) ;
      ++wp.ypos ;
      wpx = scsDialog->WriteString ( wp, "  Save All  ", pf ) ;
      scsDialog->WriteString ( wpx.ypos, wpx.xpos + 4, 
         "\u2580\u2580\u2580\u2580\u2580\u2580\u2580\u2580\u2580\u2580\u2580"
         "\u2580\u2580\u2580\u2580\u2580\u2580\u2580\u2580\u2580\u2580\u2580"
         "\u2580\u2580\u2580\u2580\u2580\u2580\u2580\u2580\u2580\u2580\u2580"
         "\u2580\u2580\u2580\u2580", saved ) ;
      wp = scsDialog->WriteParagraph ( wp.ypos + 2, wp.xpos,
         " F M  File Name \u25BC         Artist             Track\n", bb ) ;

      scsDialog->WriteParagraph ( wp.ypos, wp.xpos + 1,
                                  "\u25CF\n\n\u25CF\n\n\u25CF\n\n\u25CF", saved ) ;
      scsDialog->WriteParagraph ( wp.ypos, wp.xpos + 3,
                                  "\u25CF\n\n\u25CF\n\n\u25CF\n\n\u25CF", unsaved ) ;

      scsDialog->WriteParagraph ( wp.ypos, wp.xpos + 6,
         " School's Out     \n\n"
         " Slow Ride        \n\n"
         " Rock and Roll Hoo\n\n"
         " Love Hurts       \n\n", tn ) ;

      scsDialog->WriteParagraph ( wp.ypos, wp.xpos + 26,
         " Alice Cooper     \n\n"
         " Foghat           \n\n"
         " Rick Derringer   \n\n"
         " Nazareth         \n\n", tn ) ;

      scsDialog->WriteParagraph ( wp.ypos, wp.xpos + 45,
         " 1 / 14 \n\n"
         " 2 / 14 \n\n"
         " 3 / 14 \n\n"
         " 4 / 14 \n\n", tn ) ;
      scsDialog->WriteParagraph ( wp.ypos, wp.xpos + 48,
         "/\n\n"
         "/\n\n"
         "/\n\n"
         "/\n\n", bb ) ;
      gs.clear() ;
      while ( (gs.gscols()) < 18 )
         gs.append( L'\u2580' ) ;
      wpx = scsDialog->WriteString ( wp.ypos + 5, wp.xpos + 6, gs, unsaved ) ;
      wpx.xpos += 2 ;
      wpx = scsDialog->WriteString ( wpx, gs, unsaved ) ;
      ++wpx.xpos ;
      gs.limitCols( 8 ) ;
      scsDialog->WriteString ( wpx, gs, unsaved ) ;

      scsDialog->DrawLine ( lDef ) ;

      scsDialog->RefreshWin () ;
   }
   return OK ;

}  //* End scsCallback() *

//*************************
//*  Cmd_WriteTargetFile  *
//*************************
//******************************************************************************
//* Write the edited data (if any) along with the audio data. The low-level    *
//* formatting and output are handled by the specific method associated with   *
//* the audio format of the source file.                                       *
//* Typically this includes:                                                   *
//*   a) audio-format header(s)                                                *
//*   b) metadata header                                                       *
//*   c) metadata tags                                                         *
//*   d) audio data                                                            *
//*                                                                            *
//* Input  : fIndex    :  (optional, -1 by default)                            *
//*                       if -1, then save all source data to targets          *
//*                       if >- ZERO, then save only the indexed file          *
//*          audioOnly :  (optional, 'false' by default)                       *
//*                       if 'false' initialize to default choice              *
//*                       if 'true'  initialize to audio-only choice           *
//*                                                                            *
//* Returns: 'true'  if all files successfully written                         *
//*          'false' if user abort, access violation, insufficient space on    *
//*                  media, etc. This is rather meaningless since caller       *
//*                  can't do anything about it.                               *
//******************************************************************************
//* Notes:                                                                     *
//* 1) The configuration option 'preserve' affects the initial setting of the  *
//*    radio button.                                                           *
//*    a) 'preserve' indicates that source files will be saved unmodified as   *
//*       backup files, and that changes will be written to a COPY of the      *
//*       original source file.                                                *
//* 2) The 'audioOnly' parameter is simply a way of giving comfort to the      *
//*    user. If user chooses the 'Save Audio Only' menu item, we set up the    *
//*    radio buttons for that option. Nothing else changes.                    *
//* 3) For RTL user interface languages, we force the filename to LTR.         *
//*    As stated elsewhere, the ".oga" ".mp3" and other filename extensions    *
//*    are inherently LTR, and filenames are nearly always written in LTR      *
//*    languages.                                                              *
//*                                                                            *
//******************************************************************************

bool Taggit::Cmd_WriteTargetFile ( short fIndex, bool audioOnly )
{
   const short dlgROWS = 24,              // dialog size
               dlgCOLS = 80,
               dlgY    = fieldwinY + 1,   // dialog position
               dlgX    = fieldwinX + 1 ;
   short       rbX     = 2,               // radiobutton X offset
               rblX    = 7 ;              // radiobutton label X offset
   attr_t dColor = this->cs.sd,           // dialog interior color
          hColor = this->cs.em ;          // bold dialog text
   AppLang lang = this->cfgOpt.appLanguage ; // language index
   gString gs, gsrev ;                    // text formatting
   WriteFilter filter = wfALL_DATA ;      // output filtering option
   bool editsPending = false,             // 'true' if edits pending
        status = false ;                  // return value

   enum crtfControls : short 
   { savePB = ZERO, cancelPB, allDataRB, actPopRB, 
     actFldRB, audioRB, bakRB, ctCTRLS } ;

   //* Control labels *
   const char* Labels[][ctCTRLS + 5] = 
   {
      {  // English
         "     SAVE     ",             // 'save' pushbutton
         "    CANCEL    ",             // 'cancel' pushbutton
         "Save all fields which contain data. (default)\n"  // (save-all RB)
         "(both active and inactive fields)",
         "Save active fields which contain data, "          // (save active-pop RB)
         "ignoring empty fields.\n"
         "(all inactive fields ignored)",
         "Save all active fields, regardless of "           // (save-all-active RB)
         "whether they contain data.\n"
         "(all inactive fields ignored)",
         "Write audio data only.\n"                         // (save audio only RB)
         "(exclude all text and image metadata)",
         "Retain original source file.\n"                   // (make backup file RB)
         "(original file renamed as a backup file)\n"
         "Example: 'Shout.mp3'  becomes  'Shout.mp3~'",
         "  Save Metadata to Target File(s)  ",  // Dialog title
         //* Misc. Messages *
         "Save changes for all %S files.",                     // save all changes   (8)
         "There are currently no files with changes pending.", // none with pending  (9)
         "Save changes for highlighted file: '%S'",            // save changes for x (10)
         "No changes pending for: '%S'",                       // no changes for x   (11)
      },
      {  // Espanol
         "   GUARDAR    ",             // 'save' pushbutton
         "   CANCELAR   ",             // 'cancel' pushbutton
         "Guarde todos los campos que contengan datos. (valor predeterminado)\n" // save-all RB
         "(campos activos e inactivos)",
         "Guardar los campos activos que contienen datos, "    // save active-pop RB
         "ignorando los campos\n"
         "vacíos. (todos los campos inactivos ignorados)",
         "Guarde todos los campos activos, independientemente de si contienen\n" // save-all-active RB
         "datos. (todos los campos inactivos ignorados)",
         "Escribir sólo datos de audio.\n"                     // save audio only RB
         "(excluir todos los metadatos de texto e imagen)",
         "Conserve el archivo fuente original.\n"              // make backup file RB
         "(Archivo original renombrado como un archivo de copia de seguridad)\n"
         "Ejemplo: 'Gritar.mp3' se convierte en 'Gritar.mp3 ~'",
         "  Guardar Metadatos en los Archivos de Destino  ",   // Dialog title
         //* Misc. Messages *
         "Guardar cambios para todos %S los archivos.",        // save all changes
         "Actualmente no hay archivos con cambios pendientes.",// none with pending
         "Guardar cambios para el archivo resaltado: '%S'",    // save changes for x
         "No hay cambios pendientes para: '%S'",               // no changes for x
      },
      {  // Zhongwen
         "     保存     ",                  // 'save' pushbutton
         "     取消     ",                  // 'cancel' pushbutton
         "保存包含数据的所有字段。 （默认）\n"   // save-all RB
         "（主动和非活动字段）",
         "保存包含数据的活动字段，忽略空字段。\n" // (save active-pop RB)
         "（所有非活动字段都被忽略）",
         "保存所有活动字段，无论它们是否包含数据。\n" // (save-all-active RB)
         "（所有非活动字段都被忽略）",
         "仅写入音频数据。\n"                 // (save audio only RB)
         "（排除所有文字和图片元数据）",
         "保留原始源文件。（原文件重命名为备份文件）\n" // (make backup file RB)
         "例：'大喊.mp3'  成为  '大喊.mp3~'",
         "   将元数据保存到目标文件   ",       // Dialog title
         //* Misc. Messages *
         "保存所有 %S 个文件与更改。",         // save all changes
         "目前没有正在进行更改的文件。",        // none with pending
         "保存更改对于突出显示的文件： '%S'",   // save changes for x
         "没有更改等待： '%S'",               // no changes for x
      },
      {  // TiengViet
         "    Để Dành   ",             // 'save' pushbutton
         "    Hủy Bỏ    ",             // 'cancel' pushbutton
         "Lưu tất cả các trường có chứa dữ liệu. (hành động mặc định)\n" // (save-all RB)
         "(Đối với trường dữ liệu cả hoạt động và không hoạt động)",
         "Lưu các trường đang hoạt động chứa dữ liệu, "     // (save active-pop RB)
         "bỏ qua các trường trống.\n"
         "(tất cả các trường không hoạt động được bỏ qua)",
         "Lưu tất cả các trường đang hoạt động, "           // (save-all-active RB)
         "bất kể chúng chứa dữ liệu.\n"
         "(tất cả các trường không hoạt động được bỏ qua)",
         "Chỉ ghi dữ liệu âm thanh.\n"                      // (save audio only RB)
         "(loại trừ tất cả siêu dữ liệu văn bản và hình ảnh)",
         "Giữ nguyên tập tin gốc.\n"   // (make backup file RB)
         "(Tập tin gốc được đổi tên thành một tập tin sao lưu)\n"
         "Thí dụ: 'Kêu La.mp3'  trở thành  'Kêu La.mp3~'",
         "  Lưu siêu dữ liệu vào (các) tệp đích  ",  // Dialog title
         //* Misc. Messages *
         "Lưu thay đổi cho tất cả %S tệp.",                     // save all changes
         "Hiện tại không có tệp nào có thay đổi đang chờ xử lý.", // none with pending
         "Lưu thay đổi cho tập tin với nổi bật: '%S'",         // save changes for x
         "Không có thay đổi nào đang chờ xử lý cho: '%S'",     // no changes for x
      },
   } ;

   //* If user interface language is an RTL language, *
   //* adjust control and label positions.            *
   if ( this->cfgOpt.rtl )
   {
      rbX = (dlgCOLS - 7) ;
      rblX = -3 ;
   }

   //* Control definitions *
   InitCtrl ic[ctCTRLS] =              // array of dialog control info
   {
   {  //* 'Save' pushbutton - - - - - - - - - - - - - - - - - - - - -   savePB *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(dlgROWS - 2),           // ulY:       upper left corner in Y
      short(dlgCOLS / 2 - 15),      // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      14,                           // cols:      control columns
      Labels[lang][savePB],         // dispText:  
      this->cs.pn,                  // nColor:    non-focus color
      this->cs.pf,                  // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "",                           // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[cancelPB],                // nextCtrl:  link in next structure
   },
   {  //* 'Cancel' pushbutton - - - - - - - - - - - - - - - - - - -   cancelPB *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      ic[savePB].ulY,               // ulY:       upper left corner in Y
      short(ic[savePB].ulX + ic[savePB].cols + 2), // ulX: upper left corner in X
      1,                            // lines:     (n/a)
      14,                           // cols:      control columns
      Labels[lang][cancelPB],       // dispText:  
      this->cs.pn,                  // nColor:    non-focus color
      this->cs.pf,                  // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "",                           // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[allDataRB]                // nextCtrl:  link in next structure
   },
   {  //* 'All Data' radiobutton   - - - - - - - - - - - - - - - -   allDataRB *
      dctRADIOBUTTON,               // type:      
      rbtS5s,                       // sbSubtype: alt font, 5-wide
      (audioOnly ? false : true),   // rbSelect:  initially set
      5,                            // ulY:       upper left corner in Y
      rbX,                          // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      1,                            // cols:      (n/a)
      NULL,                         // dispText:  
      dColor,                       // nColor:    non-focus color
      this->cs.pf,                  // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      Labels[lang][allDataRB],      // label:     
      0,                            // labY:      
      rblX,                         // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[actPopRB]                 // nextCtrl:  link in next structure
   },
   {  //* 'All Populated' radiobutton   - - - - - - - - - - - - - -   allPopRB *
      dctRADIOBUTTON,               // type:      
      rbtS5s,                       // sbSubtype: alt font, 5-wide
      false,                        // rbSelect:  initially reset
      short(ic[allDataRB].ulY + 3), // ulY:       upper left corner in Y
      rbX,                          // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      1,                            // cols:      (n/a)
      NULL,                         // dispText:  
      dColor,                       // nColor:    non-focus color
      this->cs.pf,                  // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      Labels[lang][actPopRB],       // label:     
      0,                            // labY:      
      rblX,                         // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[actFldRB]                 // nextCtrl:  link in next structure
   },
   {  //* 'All Fields' radiobutton  - - - - - - - - - - - - - - - -   actFldRB *
      dctRADIOBUTTON,               // type:      
      rbtS5s,                       // sbSubtype: alt font, 5-wide
      false,                        // rbSelect:  initially reset
      short(ic[actPopRB].ulY + 3),  // ulY:       upper left corner in Y
      rbX,                          // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      1,                            // cols:      (n/a)
      NULL,                         // dispText:  
      dColor,                       // nColor:    non-focus color
      this->cs.pf,                  // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      Labels[lang][actFldRB],       // label:     
      0,                            // labY:      
      rblX,                         // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[audioRB]                  // nextCtrl:  link in next structure
   },
   {  //* 'Audio Only' radiobutton  - - - - - - - - - - - - - - - - -  audioRB *
      dctRADIOBUTTON,               // type:      
      rbtS5s,                       // sbSubtype: alt font, 5-wide
      audioOnly,                    // rbSelect:  initially reset
      short(ic[actFldRB].ulY + 3),  // ulY:       upper left corner in Y
      rbX,                          // ulX: upper left corner in X
      1,                            // lines:     (n/a)
      1,                            // cols:      (n/a)
      NULL,                         // dispText:  
      dColor,                       // nColor:    non-focus color
      this->cs.pf,                  // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      Labels[lang][audioRB],        // label:     
      0,                            // labY:      
      rblX,                         // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[bakRB]                    // nextCtrl:  link in next structure
   },
   {  //* 'Retain Source' radiobutton - - - - - - - - - - - - - - - -    bakRB *
      dctRADIOBUTTON,               // type:      
      rbtS5s,                       // sbSubtype: alt font, 5-wide
      this->cfgOpt.preserve,        // rbSelect:  initially reset
      short(ic[audioRB].ulY + 3),   // ulY:       upper left corner in Y
      rbX,                          // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      1,                            // cols:      (n/a)
      NULL,                         // dispText:  
      dColor,                       // nColor:    non-focus color
      this->cs.pf,                  // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      Labels[lang][bakRB],          // label:     
      0,                            // labY:      
      rblX,                         // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      NULL                          // nextCtrl:  link in next structure
   },
   } ;

   //* If the source data are no longer accessible, we continue *
   //* only with explicit permission from the user. This is     *
   //* because a missing or otherwise messed up source file will*
   //* cause the write to fail.                                 *
   //*             "It's not my fault!" -- Han Solo             *
   if ( (this->wtfReverifySource ( fIndex )) > ZERO )
      return false ;


   //* Dialog definition *
   InitNcDialog dInit( dlgROWS,         // number of display lines
                       dlgCOLS,         // number of display columns
                       dlgY,            // Y offset from upper-left of terminal 
                       dlgX,            // X offset from upper-left of terminal 
                       NULL,            // dialog title
                       ncltDUAL,        // border line-style
                       dColor,          // border color attribute
                       dColor,          // interior color attribute
                       ic               // pointer to list of control definitions
                     ) ;

   this->dPtr->SetDialogObscured () ;  // save parent dialog

   //* Open the dialog *
   NcDialog* dp = new NcDialog ( dInit ) ;

   //* If user interface language is an RTL language, *
   //* set the internal NcDialog flag for RTL output. *
   if ( this->cfgOpt.rtl )
      dp->DrawLabelsAsRTL () ;

   if ( (dp->OpenWindow()) == OK )
   {
      //* Set the dialog title *
      dp->SetDialogTitle ( Labels[lang][ctCTRLS], hColor ) ;

      LineDef  lDefH(ncltHORIZ, ncltDUAL, 3, ZERO, dlgCOLS, dColor ) ;
      dp->DrawLine ( lDefH ) ;

      //* Indicate which target(s) are to be written *
      winPos wp( 2, this->cfgOpt.rtl ? (dlgCOLS - 3) : 2 ) ;
      if ( (fIndex == -1) && (this->tData.sfCount > 1) ) // all files
      {
         for ( short i = ZERO ; i < this->tData.sfCount ; ++i )
         {
            if ( this->tData.sf[i].sfMod || this->tData.sf[i].sfTag.tfMod )
            {
               editsPending = true ;
               break ;
            }
         }
         if ( editsPending )
         {
            gsrev.compose( "%hd", &this->tData.sfCount ) ;
            if ( this->cfgOpt.rtl && (gsrev.gschars() > 2) )
               this->gsRevChars ( gsrev ) ;
            gs.compose( Labels[lang][8], gsrev.gstr() ) ;
         }
         else
            gs = Labels[lang][9] ;
      }
      else                       // highlighted file only
      {
         //* If there is only one file, no need to loop *
         if ( this->tData.sfCount == 1 )
            fIndex = ZERO ;

         //* Are edits pending? *
         gsrev = this->tData.sf[fIndex].sfName ;
         if ( this->cfgOpt.rtl )
            this->gsRevChars ( gsrev ) ;
         if ( this->tData.sf[fIndex].sfMod || this->tData.sf[fIndex].sfTag.tfMod )
         {
            editsPending = true ;
            gs.compose( Labels[lang][10], gsrev.gstr() ) ;
         }
         else
            gs.compose( Labels[lang][11], gsrev.gstr() ) ;
      }
      wp = dp->WriteString ( wp, gs, hColor, false, this->cfgOpt.rtl ) ;

      //* Create radiobutton XOR groups *
      short xorGroup[5] { allDataRB, actPopRB, actFldRB, audioRB, -1 } ;
      dp->GroupRadiobuttons ( xorGroup ) ;

      short    icIndex = savePB ;         // index of control with focus

      //* If no edits pending, disable everything except 'CANCEL' *
      if ( ! editsPending )
      {
         while ( (icIndex = dp->NextControl ()) != cancelPB ) ;
         dp->ControlActive ( savePB, false ) ;
         dp->ControlActive ( allDataRB, false ) ;
         dp->ControlActive ( actPopRB, false ) ;
         dp->ControlActive ( actFldRB, false ) ;
         dp->ControlActive ( audioRB, false ) ;
         dp->ControlActive ( bakRB, false ) ;
         winPos wp( short(ic[allDataRB].ulY + ic[allDataRB].labY), 
                    short(ic[allDataRB].ulX + ic[allDataRB].labX) ) ;
         dp->WriteParagraph ( wp, Labels[lang][allDataRB], this->cs.dm ) ;
         wp.ypos += 3 ;
         dp->WriteParagraph ( wp, Labels[lang][actPopRB], this->cs.dm ) ;
         wp.ypos += 3 ;
         dp->WriteParagraph ( wp, Labels[lang][actFldRB], this->cs.dm ) ;
         wp.ypos += 3 ;
         dp->WriteParagraph ( wp, Labels[lang][audioRB], this->cs.dm ) ;
         wp.ypos += 3 ;
         dp->WriteParagraph ( wp, Labels[lang][bakRB], this->cs.dm ) ;
      }

      dp->RefreshWin () ;

      //* Interact with the user *
      uiInfo   Info ;                     // user interface data returned here
      bool     done = false ;             // loop control
      while ( ! done )
      {
         if ( ic[icIndex].type == dctPUSHBUTTON )
         {
            if ( Info.viaHotkey )
               Info.HotData2Primary () ;
            else
               icIndex = dp->EditPushbutton ( Info ) ;
            if ( Info.dataMod != false )
            {
               if ( Info.ctrlIndex == savePB )
               {  //* Get the index of the selected control *
                  short selected = dp->GetRbGroupSelection ( allDataRB ) ;
                  switch ( selected )
                  {
                     case actPopRB:       // all active AND populated fields
                        filter = wfALL_ACTDATA ;
                        break ;
                     case actFldRB:       // all active fields
                        filter = wfALL_ACTIVE ;
                        break ;
                     case audioRB:        // audio data only (discard tag data)
                        filter = wfAUDIO_ONLY ;
                        break ;
                     case allDataRB:      // all populated fields
                     default:
                        filter = wfALL_DATA ;
                        break ;
                  }
                  //* Indicate whether to create a backup of source data.      *
                  //* NOTE: This changes the top-level configuration parameter.*
                  dp->GetRadiobuttonState ( bakRB, this->cfgOpt.preserve ) ;
                  status = true ;
               }
               done = true ;
            }
         }

         else if ( ic[icIndex].type == dctRADIOBUTTON )
         {
            if ( Info.viaHotkey )
               Info.HotData2Primary () ;
            else
               icIndex = dp->EditRadiobutton ( Info ) ;
            if ( Info.dataMod != false )
            { /* Nothing to do */ }
         }

         //* Move focus to appropriate control *
         if ( ! done && ! Info.viaHotkey )
         {
            if ( Info.keyIn == nckSTAB )
               icIndex = dp->PrevControl () ; 
            else
               icIndex = dp->NextControl () ;
         }
      }  // while()
   }
   if ( dp != NULL )                         // close the window
      delete ( dp ) ;

   this->dPtr->RefreshWin () ;               // restore parent dialog

   //* If user approved the operation *
   if ( status != false )
   {
      if ( fIndex < ZERO )                   // all source files
      {
         for ( fIndex = ZERO ; fIndex < this->tData.sfCount ; ++fIndex )
         {
            if ( (this->WriteTargetFile ( fIndex, filter )) == false )
               status = false ;
         }
      }
      else                                   // highlighted file only
         status = this->WriteTargetFile ( fIndex, filter ) ;

      this->UpdateDataWindows () ;  // update the display
   }
   return status ;

}  //* End Cmd_WriteTargetFile() *

//*************************
//*    WriteTargetFile    *
//*************************
//******************************************************************************
//* Write the edited metadata to the target file.                              *
//* 1) The target file will be in the same directory as the source file.       *
//* 2) The name of the target file will be the name in:                        *
//*          this->tData.sf[fIndex].sfName                                     *
//* 3) 'this->cfgOpt.preserve' indicates whether the original file will be     *
//*    saved as a backup file.                                                 *
//* 4) The edits-pending flags for filename and tag data will be reset if      *
//*    operation is successful, else unchanged.                                *
//*                                                                            *
//* Input  : fIndex  : index of file to be written                             *
//*          filter  : (member of enum WriteFilter)                            *
//*                    determines which metadata will be written to target.    *
//*                                                                            *
//* Returns: 'true'  if file successfully written                              *
//*          'false' if access violation, insufficient media space, etc.       *
//*          NOTE: We may want to return the type of error                     *
//******************************************************************************

bool Taggit::WriteTargetFile ( short fIndex, WriteFilter filter )
{
   bool success = false ;

   if ( this->tData.sf[fIndex].sfType == mftMP3 )
   {
      success = this->WriteMetadata_MP3 ( fIndex, filter, this->cfgOpt.preserve ) ;
   }

   else if ( this->tData.sf[fIndex].sfType == mftOGG )
   {
      success = this->WriteMetadata_OGG ( fIndex, filter, this->cfgOpt.preserve ) ;
   }

   else if ( this->tData.sf[fIndex].sfType == mftASF )
   {
      success = this->WriteMetadata_ASF ( fIndex, filter, this->cfgOpt.preserve ) ;
   }

   else
   { /* this should not happen */ }

   if ( success )    // reset the edits-pending flags
      this->tData.sf[fIndex].sfMod       = 
      this->tData.sf[fIndex].sfTag.tfMod = false ;

   return success ;

}  //* End WriteTargetFile() *

//*************************
//*   wtfReverifySource   *
//*************************
//******************************************************************************
//* Scan the list of source files to verify that:                              *
//*   1) Files exist                                                           *
//*   2) Files are of 'regular' type                                           *
//*   3) User has both read and write access to the files.                     *
//*   4) Note that if the media format (filename extension) has changed, then  *
//*      the original filename is no longer valid. Duh!                        *
//* This method is used to verify our data before beginning to write the       *
//* target files in case user (or forces of evil) has changed the source files *
//* since they were originally scanned and verified.                           *
//*                                                                            *
//* Input  : fIndex    :  (optional, -1 by default)                            *
//*                       if -1, then save all source data to targets          *
//*                       if >= ZERO, then save only the indexed file          *
//*                                                                            *
//* Returns: number of source files which have become inaccessible.            *
//******************************************************************************
//* Notes:                                                                     *
//* -- It must be assumed that the user may dink with one or more of the files *
//*    we are currently working with, so before starting the loop of writing   *
//*    to targets, we must re-verify the existence of all our source files.    *
//* -- One possibility we DO NOT manage is that a source file may be locked    *
//*    by another application. This could happen if for example, a media       *
//*    player is currently using the file. In this case, we could _read_ it    *
//*    but not _overwrite_ it. This means that if "preserve" is set OR if user *
//*    has specified an alternate target-file name, then the operation will    *
//*    complete successfully because we don't need to modify the source file.  *
//*    However, if "preserve" is reset then we may not be able to modify the   *
//*    source file.                                                            *
//******************************************************************************

short Taggit::wtfReverifySource ( short fIndex )
{
   gString gs ;
   tnFName fStats ;
   short modCount = ZERO ;          // return value

   short indx = fIndex < ZERO ? ZERO : fIndex ;
   for ( ; indx < this->tData.sfCount ; ++indx )
   {
      gs = this->tData.sf[indx].sfPath ;
      if (    ((this->GetFileStats ( gs, fStats )) != OK)
           || (fStats.fType != fmREG_TYPE)
           || (!fStats.readAcc || !fStats.writeAcc)
         )
         ++modCount ;

      if ( fIndex >= ZERO )   // if checking a single source file, we're done
         break ;
   }

   //* If user has been screwing with the source files *
   if ( modCount > ZERO )
   {
      const short dlgCOLS  = 60,
                  dlgLINES = 9 ;
      const char* dlgText[][dlgLINES + 2] = 
      {
         {  // English
            "   Warning!  Warning!   ",
            " ",
            "One or more source files in the list have been moved,",
            "renamed, deleted or are no longer read/write accessible.",
            "This may cause processing errors.",
            " ",
            "                Do you want to proceed?",
            " ",
            NULL,
            "  PROCEED  ",          // YES Pushbutton text
            "   ABORT   "           // NO  Pushbutton text
         },
         {  // Espanol
            "   ¡Advertencia!  ¡Advertencia!   ",
            " ",
            "Uno o más archivos de origen de la lista se han movido,",
            "renombrado, eliminado o ya no es accesible de ",
            "lectura/escritura.",
            "Esto puede causar errores de procesamiento.",
            " ",
            "                   Quieres proceder?",
            NULL,
            "  PROCEDER  ",         // YES Pushbutton text
            "   ANULAR   "          // NO  Pushbutton text
         },
         {  // Zhongwen
            "    警报!   警报!    ",
            " ",
            "列表中的一个或多个源文件已被移动，重命名，",
            "删除或不再读/写可访问。",
            "这可能会导致处理错误。",
            " ",
            "                      你要继续吗？",
            " ",
            NULL,
            "    继续    ",          // YES Pushbutton text
            "    舍弃    "           // NO  Pushbutton text
         },
         {  // Tieng View
            "   Cảnh báo!  Cảnh báo!   ",
            " ",
            "Một hoặc nhiều tệp nguồn trong danh sách đã được di",
            "chuyển, đổi tên, xóa hoặc không còn khả năng đọc/ghi nữa.",
            "Điều này có thể gây ra lỗi chế biến.",
            " ",
            "                  Bạn có muốn tiếp tục?",
            " ",
            NULL,
            "  Tiến Hành  ",        // YES Pushbutton text
            "   Hủy Bỏ   "          // NO  Pushbutton text
         },
      } ;

      AppLang lang = this->cfgOpt.appLanguage ;
      genDialog gd( dlgText[lang], this->cs.sd, 
                    (dlgLINES + 2), dlgCOLS, -1, -1, NULL, 
                    this->cfgOpt.rtl, attrDFLT, attrDFLT, 
                    dlgText[lang][dlgLINES], dlgText[lang][dlgLINES + 1] ) ;

      if ( (this->dPtr->DecisionDialog ( gd )) != false )
         modCount = ZERO ;       // user has given permission to proceed
   }
   return modCount ;

}  //* End wtfReverifySource() *

//*************************
//*       CopyFile        *
//*************************
//******************************************************************************
//* Copy a file preserving all attributes.                                     *
//* It is assumed that source is a 'regular' file.                             *
//*                                                                            *
//* Input  : srcPath : full filespec for source                                *
//*          trgPath : full filespec for target                                *
//*                                                                            *
//* Returns: 'true' if successful, else false                                  *
//******************************************************************************

bool Taggit::CopyFile ( const char* srcPath, const char* trgPath )
{
   const short bSIZE = MAX_PATH * 2 ;
   char cmd[bSIZE] ;
   bool status = true ;

   //* Escape any double-quote characters in the paths *
   gString sp( srcPath ), tp( trgPath ) ;
   this->DQuoteEscape ( sp ) ;
   this->DQuoteEscape ( tp ) ;

   snprintf ( cmd, bSIZE, "cp --preserve=all -d \"%s\" \"%s\" 1>/dev/null 2>/dev/null", 
                          sp.ustr(), tp.ustr() );
   system ( cmd ) ;

   return status ;

}  //* End CopyFile() *

bool Taggit::CopyFile ( const wchar_t* srcPath, const wchar_t* trgPath )
{

   gString sp( srcPath ), tp( trgPath ) ;
   return ( (this->CopyFile ( sp.ustr(), tp.ustr() )) ) ;

}  //* End CopyFile() *

bool Taggit::CopyFile ( const gString& srcPath, const gString& trgPath )
{

   return ( (this->CopyFile ( srcPath.ustr(), trgPath.ustr() )) ) ;

}  //* End CopyFile() *

//*************************
//*      DeleteFile       *
//*************************
//******************************************************************************
//* Delete (unlink) the specified source file.                                 *
//*                                                                            *
//* Input  : srcPath : full path/filename specification of file to be deleted  *
//*                                                                            *
//* Returns: OK if successful, else returns errno value                        *
//******************************************************************************

short Taggit::DeleteFile ( const gString& srcPath )
{
   short    status = OK ;
   if ( (unlink ( srcPath.ustr() )) != OK )
      status = errno ;         // return 'errno' value
   return status ;

}  //* End DeleteFile() *

//*************************
//*      RenameFile       *
//*************************
//******************************************************************************
//* Rename the specified file.                                                 *
//*                                                                            *
//* Input  : srcPath : full path/filename specification of source              *
//*          trgPath : full path/filename specification of target              *
//*                                                                            *
//* Returns: OK if successful, else returns errno value                        *
//******************************************************************************

short Taggit::RenameFile ( const gString& srcPath, const gString& trgPath )
{
   short    status = OK ;
   if ( (status = rename ( srcPath.ustr(), trgPath.ustr() )) != OK )
      status = errno ;         // return 'errno' value
   return status ;

}  //* End RenameFile() *

//*************************
//*     DQuoteEscape      *
//*************************
//******************************************************************************
//* Scan the source string (typically a filespec). If the string contains a    *
//* double quote character, 'escape' it with a backslash character.            *
//*                                                                            *
//*                                                                            *
//* Input  : src     : data to be scanned and modified                         *
//*                                                                            *
//* Returns: 'true' if either source or target path modified, else 'false'     *
//******************************************************************************
//* Notes:                                                                     *
//* ------                                                                     *
//*                                                                            *
//* 1) All calls to the 'system()' group of C library functions will be        *
//*    interpreted by the shell.                                               *
//*                                                                            *
//* 2) The shell assigns special meaning to certain characters UNLESS they     *
//*    are 'escaped', that is, the special meaning is explicitly removed.      *
//*                                                                            *
//* 3) The special characters are:                                             *
//*       |  &  ;  <  >  (  )  $  `  \  "  '  <space>  <tab>  <newline>        *
//*    The following are _sometimes_ special characters:                       *
//*       *   ?   [   #   ~   =   %                                            *
//*                                                                            *
//* 4) Escapes can be indicated in three ways:                                 *
//*    a) Place a backslash character before each special character in the     *
//*       string.                                                              *
//*    b) Enclose the entire string in single or double quotes.                *
//*       i.  Enclosing in single quotes will escape everything but an         *
//*           embedded single quote. Note that a backslash will not escape     *
//*           a single quote enclosed in single quotes.                        *
//*       ii. Enclosing in double quotes will escape everything but the        *
//*           special characters:  $  `  \  which are used by the shell for    *
//*           text substitution.                                               *
//*    c) Enclose the entire string in backtick characters. This is a special  *
//*       case for shell scripts, and we do not consider it here.              *
//*                                                                            *
//* 5) When sending strings to the shell parser, this application uses double  *
//*    quotes to enclose the strings. Because the double-quote character is    *
//*    technically a valid filename character (although only an idiot would    *
//*    use it that way), we must scan the outgoing strings for embedded        *
//*    double quote characters and 'escape' them with a backslash.             *
//*    Example: Review of "My House in Umbria" - starring Maggie Smith         *
//*               becomes:                                                     *
//*             Review of \"My House in Umbria\" - starring Maggie Smith       *
//*    See the CopyFile() method for a practical example.                      *
//*                                                                            *
//* 6) Note that if caller specified the filespec of a symbolic link to a file *
//*    with special characters in its name, then we will never see it here,    *
//*    and the external program must handle the link-target filename.          *
//*                                                                            *
//* -   -   -   -   -   -   -   -   -   -   -   -   -   -   -   -   -   -   -  *
//* From the 'Shell Command Language' page see:                                *
//*       "Quoting Special Characters in BASH.odt"                             *
//* a) A single quote may not be escaped within a string delimited by          *
//*    single quotes.                                                          *
//* b) Delimiting with double-quotes preserves all literal values EXCEPT:      *
//*               $   `   \                                                    *
//*    '$' is used in environment variable substitution                        *
//*    '`' is used to substitute stdout for whatever is delimited by           *
//*        the backticks.                                                      *
//*    '\' is the escape characters, so '\\' gives its literal value.          *
//*                                                                            *
//*                  --- AT THE COMMAND LINE ---                               *
//* This works:                                                                *
//*       cp --preserve=all "Tom's_Toothpaste.o" 'Tom"s_Toothpaste.o'          *
//* This works:                                                                *
//*       ls -l Tom"'"s_Toothpaste.o                                           *
//* This works:                                                                *
//*       ls -l Tom\"s_Toothpaste.o                                            *
//* This works:                                                                *
//*       ls -l Tom\'s_Toothpaste.o                                            *
//* This works:                                                                *
//*       ls -l "Tom's_Toothpaste.o"                                           *
//* This DOES NOT work:                                                        *
//*       ls -l 'Tom\'s_Toothpaste.o'                                          *
//*                   --- CALLING THE SYSTEM ---                               *
//* This works (but is clunky):                                                *
//*       gsSrc = L"Tom\\\"s_Toothpaste.o" ;                                   *
//*       snprintf ( cmd, bSIZE, "ls -l \"%s\"", gsSrc.ustr() );               *
//*       status = system ( cmd ) ;                                            *
//*                                                                            *
//* These work, and are the recommended syntax:                                *
//*       gsSrc = L"Tom's_Toothpaste.o" ;                                      *
//*       snprintf ( cmd, bSIZE, "ls -l \"%s\"", gsSrc.ustr() );               *
//*       status = system ( cmd ) ;                                            *
//*         OR                                                                 *
//*       gsSrc = L"I \"win\" in the game of life."                            *
//*       this->DQuoteEscape ( gsSrc ) ;
//*       snprintf ( cmd, bSIZE, "ls -l \"%s\"", gsSrc.ustr() );               *
//*       status = system ( cmd ) ;                                            *
//*                                                                            *
//******************************************************************************

bool Taggit::DQuoteEscape ( gString& src )
{
   const wchar_t wDQ = L'"',
                 wBS = L'\\' ;

   bool status = false ;

   short sqIndex = ZERO ;
   if ( (sqIndex = src.find( wDQ )) >= ZERO )
   {
      do
      {
         src.insert( wBS, sqIndex ) ;
         sqIndex += 2 ;
      }
      while ( (sqIndex = src.find( wDQ, sqIndex )) >= ZERO ) ;
      status = true ;
   }
   return status ;

}  //* End DQuoteEscape() *

//*************************
//*     Cmd_InfoHelp      *
//*************************
//******************************************************************************
//* Open the application Help file.                                            *
//* a) Shell out and invoke the 'info' documentation reader.                   *
//* b) Invoke the default browser with help in HTML format.                    *
//*                                                                            *
//* The 'taggit.info' or 'taggit.html' document _or_ a symlink to that         *
//* document must be in the same directory as the application binary.          *
//*                                                                            *
//* Input  : htmlFormat : if 'false', invoke help through info reader          *
//*                       if 'true',  load help into default browser           *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void Taggit::Cmd_InfoHelp ( bool htmlFormat )
{
   fmFType ft ;
   gString hPath( "%s/taggit.%s", 
                  this->cfgOpt.appPath, (htmlFormat ? "html" : "info") ) ;
   if ( this->TargetExists ( hPath, ft ) )
   {
      if ( htmlFormat )    // help (HTML format)
      {
         this->LaunchDefaultApplication ( hPath.ustr() ) ;
      }
      else                 // help (info format)
      {
         gString gs( "info -f \"%S\"", hPath.gstr() ) ;
         this->dPtr->ShellOut ( soX, gs.ustr() ) ;
      }
   }
   else                    // Error: help file not found
   {
      const char* dlgText_i[] = 
      {
         "  ALERT! - ¡Notificación! - 警报! - Thông báo!  ",
         "",
         "Help file 'taggit.info' was not found.",
         "No se encontró el archivo de ayuda 'taggit.info'.",
         "无法找到帮助文件 'taggit.info'。",
         "Không tìm thấy tệp trợ giúp 'taggit.info'.",
         "",
         NULL
      } ;
      const char* dlgText_h[] = 
      {
         "  ALERT! - ¡Notificación! - 警报! - Thông báo!  ",
         "",
         "Help file 'taggit.html' was not found.",
         "No se encontró el archivo de ayuda 'taggit.html'.",
         "无法找到帮助文件 'taggit.html'。",
         "Không tìm thấy tệp trợ giúp 'taggit.html'.",
         "",
         NULL
      } ;
      attr_t dColor = this->cs.sd,
             hColor = this->cs.em ;
      attr_t dlgAttr[] = 
      { hColor, dColor, dColor, dColor, dColor, dColor, dColor, dColor } ;

      genDialog gd( (htmlFormat ? dlgText_h : dlgText_i), 
                    dColor, 9, 52, -1, -1, dlgAttr, 
                    this->cfgOpt.rtl, attrDFLT, attrDFLT, 
                    okText[this->cfgOpt.appLanguage] ) ;

      this->dPtr->InfoDialog ( gd ) ;     // alert user
   }

}  //* End Cmd_InfoHelp() *

//*************************
//*     Cmd_QuickHelp     *
//*************************
//******************************************************************************
//* Display 'Quick Help', a list of navigation and shortcut keycodes.          *
//*                                                                            *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void Taggit::Cmd_QuickHelp ( void )
{
   const short
         dlgROWS = this->termRows - 1, // dialog rows
         dlgCOLS = this->fwinCols + 2, // dialog columns
         ulY = 1,                      // dialog position in Y
         ulX = fieldwinX,              // dialog position in X
         colOne = this->cfgOpt.rtl ? (dlgCOLS - 3) : 2 ; // Data column 1
   attr_t dColor = this->cs.sd,        // dialog interior color
          hColor = this->cs.em ;       // bold dialog text
   AppLang lang = this->cfgOpt.appLanguage ; // language index

   const char* Combo[][4] = 
   {  
      {  // English
         // (0) Key nav
         "UpArrow   - - - - Move file highlight upward.\n"
         "DownArrow - - - - Move file highlight downward.\n"
         "PageUp    - - - - Move file highlight upward by one page.\n"
         "PageDown  - - - - Move file highlight downward by one page.\n"
         "Home key  - - - - Move file highlight to the top of the list.\n"
         "End key   - - - - Move file highlight to the bottom of the list.\n"
         "LeftArrow - - - - Scroll tag fields toward the right.\n"
         "CTRL+LeftArrow  - Scroll first tag field into view.\n"
         "RightArrow  - - - Scroll tag fields toward the left.\n"
         "CTRL+RightArrow - Scroll last tag field into view.\n\n",

         // (1) mouse nav
         "Scroll forward  - - - - Move file highlight upward\n"
         "Scroll backward - - - - Move file highlight downward\n"
         "CTRL+Scroll forward - - Scroll tag fields to the right\n"
         "CTRL+Scroll backward  - Scroll tag fields to the left\n"
         "Left Click  - - - - - - Edit the object under the mouse pointer\n\n",

         // (2) hotkeys
         "ALT+M  = Edit Metadata               ALT+N      = Edit Filename\n"
         "CTRL+S = Save file                   ALT+CTRL+S = Save with rename\n"
         "CTRL+Q = Quit (Exit)                 CTRL+Z     = Undo (discard) edits\n"
         "CTRL+D = Duplicate field             ALT+UP     = Shift file upward\n"
         "CTRL+T = Track sequence              ALT+DOWN   = Shift file downward\n",

         // (3) text-edit keys
         "CTRL+A = select all       CTRL+C = Copy        CTRL+R = Reverse text\n"
         "CTRL+X = Cut              CTRL+V = Paste\n",
      },
      {  // Espanol
         // (0) Key nav
         "Flecha Arriba  - - Mueva el archivo resaltado hacia arriba.\n"
         "Flecha Abajo - - - Mueva el archivo resaltado hacia abajo.\n"
         "Página Arriba  - - Mueva el destaque hacia arriba en una página.\n"
         "Página Abajo - - - Mueva el destaque hacia abajo en una página.\n"
         "Tecla de Inicio  - Mueva el resaltado a la parte superior de la lista.\n"
         "Tecla Final  - - - Mueva el resaltado al final de la lista.\n"
         "Flecha Izquierda  - - - Desplaza los campos de etiquetas hacia la derecha.\n"
         "CTRL+Flecha Izquierda - Mover al primer campo de etiqueta.\n"
         "Flecha Derecha  - - - - Mueva los campos de etiqueta a la izquierda.\n"
         "CTRL+Flecha Derecha - - Mover al último campo de etiqueta.\n\n",

         // (1) mouse nav
         "Rodar hacia adelante  - - - Desplazar el cursor hacia arriba.\n"
         "Rodar hacia atrás - - - - - Desplazar el cursor hacia atrás.\n"
         "CTRL+Rodar hacia adelante - Desplaza los campos de etiqueta a la derecha.\n"
         "CTRL+Rodar hacia atrás  - - Desplácese los campos de etiqueta a la izquierda.\n"
         "click izquierdo - - - - - - Editar objeto bajo el puntero del ratón\n\n",

         // (2) hotkeys
         "ALT+M  = Editar Los Metadatos        ALT+N      = Editar nombre de archivo\n"
         "CTRL+S = Guardar Archivo             ALT+CTRL+S = Guardar Con Renombrar\n"
         "CTRL+Q = Salida (dejar)              CTRL+Z     = Anular Ediciones\n"
         "CTRL+D = Duplicar un Campo           ALT+UP     = Desplazar el archivo hacia arriba\n"
         "CTRL+T = Pista Relleno Automático    ALT+DOWN   = Desplazar el archivo hacia abajo\n",

         // (3) text-edit keys
         "CTRL+A = seleccionar todo CTRL+C = Copiar   CTRL+R = Invierta el texto\n"
         "CTRL+X = Cortar           CTRL+V = Pegar\n",
      },
      {  // Zhongwen
         // (0) Key nav
         "向上箭头  - - - - 向上移动文件指示器。\n"
         "向下箭头  - - - - 向下移动文件指针。\n"
         "向上翻页  - - - - 将文件指针向上移动一页。\n"
         "向下翻页  - - - - 将文件指针向下移动一页。\n"
         "家        - - - - 将文件指针移动到列表顶部。\n"
         "结束      - - - - 将文件指针移动到列表的底部。\n"
         "左箭头    - - - - 向右滚动标签字段。\n"
         "CTRL+左箭头     - 将第一个标签字段滚动到视图中。\n"
         "右箭头      - - - 向左滚动标签字段。\n"
         "CTRL+右箭头     - 将最后一个标签字段滚动到视图中。\n\n",

         // (1) mouse nav
         "向前滚动  - - - - - - - 向上移动文件指针。\n"
         "向后滚动  - - - - - - - 向下移动文件指针。\n"
         "CTRL+向前滚动 - - - - - 向右滚动标签字段。\n"
         "CTRL+向后滚动 - - - - - 向左滚动标签字段。\n"
         "左键单击  - - - - - - - 编辑鼠标指针下的对象。\n\n",

         // (2) hotkeys
         "ALT+M  = 编辑元数据                  ALT+N      = 编辑文件名\n"
         "CTRL+S = 保存文件                    ALT+CTRL+S = 重命名并保存\n"
         "CTRL+Q = 放弃                        CTRL+Z     = 舍弃（撤消）编辑\n"
         "CTRL+D = 复制数据字段                ALT+向上   = 列表中向上移动文件\n"
         "CTRL+T = 初始化 “曲目” 序列          ALT+向下   = 在列表中向下移动文件\n",

         // (3) text-edit keys
         "CTRL+A = 全选             CTRL+C = 复制        CTRL+R = 反向文字\n"
         "CTRL+X = 剪切             CTRL+V = 粘贴\n",
      },
      {  // Tieng Viet
         // (0) Key nav
         "Mũi tên lên   - - - - - Di chuyển điểm nổi bật đến tệp tin trước đó.\n"
         "mũi tên xuống   - - - - Di chuyển điểm nổi bật đến tệp tiếp theo.\n"
         "Trang Lên   - - - - - - Di chuyển lên một trang trong danh sách.\n"
         "Trang Dươi  - - - - - - Di chuyển xuống một trang trong danh sách.\n"
         "Phím Bắt đầu  - - - - - Cuộn xuống đầu danh sách.\n"
         "Phím chung cuộc - - - - Cuộn xuống cuối danh sách.\n"
         "Mũi tên trái  - - - - - Di chuyển đến trường nhãn kế tiếp.\n"
         "CTRL+Mũi tên trái - - - Di chuyển đến trường nhãn đầu tiên.\n"
         "Mũi Tên Bên Phải  - - - Di chuyển đến trường nhãn trước đó.\n"
         "CTRL+Mũi Tên Bên Phải - Di chuyển đến trường nhãn cuối cùng.\n\n",

         // (1) mouse nav
         "Cuộn về phía trước  - - - Di chuyển điểm nổi bật đến tệp tin trước đó.\n"
         "Cuộn về phía sau  - - - - Di chuyển điểm nổi bật đến tệp tiếp theo.\n"
         "CTRL+Cuộn về phía trước - Di chuyển đến trường nhãn trước đó.\n"
         "CTRL+Cuộn về phía sau - - Di chuyển đến trường nhãn kế tiếp.\n"
         "ấn chuột trái - - - - - - Chỉnh sửa đối tượng dưới con trỏ chuột.\n\n",

         // (2) hotkeys
         "ALT+M  = Chỉnh sửa siêu dữ liệu      ALT+N      = Chỉnh sửa tên tệp\n"
         "CTRL+S = Lưu tệp                     ALT+CTRL+S = Lưu tệp với đổi tên\n"
         "CTRL+Q = Thoát                       CTRL+Z     = Hoàn tác (huỷ bỏ) chỉnh sửa\n"
         "CTRL+D = Chép lại trường dữ liệu     ALT+UP     = Di chuyển tệp lên trên\n"
         "CTRL+T = Đặt trình tự bài hát        ALT+DOWN   = Di chuyển tệp xuống\n",

         // (3) text-edit keys
         "CTRL+A = Chọn tất cả  CTRL+C = Sao chép      CTRL+R = Văn bản ngược\n"
         "CTRL+X = Cắt,         CTRL+V = Dán\n",
      },
   } ;

   const char* Labels[][6] = 
   {
      {  // English
         "  Quick Help  ",                               // (0) dialog title
         "Navigation using the navigation keys\n",       // (1) key navigation
         "Navigation using mouse Scroll Wheel\n",        // (2) mouse navigation
         "Hotkeys (shortcuts)\n",                        // (3) hotkeys
         "Text Edit: ",                                  // (4) text edit
         " Press Any Key To Exit. ",                     // (5) exit message
      },
      {  // Espanol
         "  Ayuda Rapida  ",                             // (0) dialog title
         "Navegación con las teclas de navegación\n",    // (1) key navigation
         "Navegación con la rueda de desplazamiento del ratón\n", // (2) mouse navigation
         "Teclas de acceso rápido (atajos)\n",           // (3) hotkeys
         "Edición de texto: ",                           // (4) text edit
         " Presiona cualquier tecla para salir. ",       // (5) exit message
      },
      {  // Zhongwen
         "  快速帮助  ",                                  // (0) dialog title
         "导航 使用导航键\n",                              // (1) key navigation
         "导航 使用鼠标滚轮\n",                            // (2) mouse navigation
         "热键 (捷径)\n",                                 // (3) hotkeys
         "文字编辑:  ",                                  // (4) text edit
         "  按任何一个键退出。 ",                          // (5) exit message
      },
      {  // TiengView
         "  Trợ Giúp Nhanh  ",                           // (0) dialog title
         "Điều Hướng sử dụng các phím điều hướng\n",     // (1) key navigation
         "Điều Hướng sử dụng bánh xe cuộn chuột\n",      // (2) mouse navigation
         "Phím nóng (Phím tắt)\n",                       // (3) hotkeys
         "Chỉnh sửa văn bản: ",                          // (4) text edit
         " Nhấn bất kì phím nào để thoát. ",             // (5) exit message
      },
   } ;

   this->dPtr->SetDialogObscured () ;  // save parent dialog

   //* Initial parameters for dialog window *
   InitNcDialog dInit( dlgROWS,        // number of display lines
                       dlgCOLS,        // number of display columns
                       ulY,            // Y offset from upper-left of terminal 
                       ulX,            // X offset from upper-left of terminal 
                       NULL,           // dialog title
                       ncltDUAL,       // border line-style
                       dColor,         // border color attribute
                       dColor,         // interior color attribute
                       NULL            // pointer to list of control definitions
                     ) ;

   //* Instantiate the dialog window *
   NcDialog* dp = new NcDialog ( dInit ) ;

   //* If user interface language is an RTL language, *
   //* set the internal NcDialog flag for RTL output. *
   if ( this->cfgOpt.rtl )
      dp->DrawLabelsAsRTL () ;

   //* Open the dialog window *
   if ( (dp->OpenWindow()) == OK )
   {
      //* Set the dialog title *
      dp->SetDialogTitle ( Labels[lang][0], hColor ) ;

      winPos wp( 1, colOne ) ;
      wp = dp->WriteParagraph ( wp, Labels[lang][1], hColor, false, this->cfgOpt.rtl ) ;
      wp = dp->WriteParagraph ( wp, Combo[lang][0], dColor, false, this->cfgOpt.rtl ) ;
      wp = dp->WriteParagraph ( wp, Labels[lang][2], hColor, false, this->cfgOpt.rtl ) ;
      wp = dp->WriteParagraph ( wp, Combo[lang][1], dColor, false, this->cfgOpt.rtl ) ;
      wp = dp->WriteParagraph ( wp, Labels[lang][3], hColor, false, this->cfgOpt.rtl ) ;
      wp = dp->WriteParagraph ( wp, Combo[lang][2], dColor, false, this->cfgOpt.rtl ) ;
      wp = dp->WriteParagraph ( wp, Labels[lang][4], hColor, false, this->cfgOpt.rtl ) ;
      wp = dp->WriteParagraph ( wp, Combo[lang][3], dColor, false, this->cfgOpt.rtl ) ;
      gString gs( Labels[lang][5] ) ;
      wp.ypos = 1 ;
      wp.xpos = this->cfgOpt.rtl ? (gs.gscols() + 2)
                                 : (dlgCOLS - (gs.gscols() + 2)) ;
      wp = dp->WriteString ( wp, gs, this->cs.pf, false, this->cfgOpt.rtl ) ;

      dp->RefreshWin () ;                    // make the text visible
      nckPause();
   }        // OpenWindow()
   if ( dp != NULL )                         // close the window
      delete ( dp ) ;

   this->dPtr->RefreshWin () ;               // restore parent dialog

}  //* End Cmd_QuickHelp() *

//*************************
//*     Cmd_HelpAbout     *
//*************************
//******************************************************************************
//* Display the application's Help-About window.                               *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************
//* Programmer's Note: This dialog is very simple for LTR languages, but       *
//* raises multiple issues for RTL languages because a large percentage of the *
//* data are both LTR and English.                                             *
//*                                                                            *
//******************************************************************************

void Taggit::Cmd_HelpAbout ( void )
{
   const short
         dlgROWS = 16,                 // dialog rows
         dlgCOLS = 70,                 // dialog columns
         dciROWS = dlgROWS - 4,        // sub-dialog rows
         dciCOLS = dlgCOLS - 2 ;       // sub-dialog columns
   short ctrY = this->termRows / 2,    // center of terminal in Y
         ctrX = this->termCols / 2,    // center of terminal in X
         ulY = ctrY - dlgROWS / 2,     // dialog position in Y
         ulX = ctrX - dlgCOLS / 2,     // dialog position in X
         dciulY   = ulY + 3,           // sub-dialog position in Y
         dciulX   = ulX + 1 ;          // sub-dialog position in X
   attr_t dColor = this->cs.sd,        // dialog interior color
          vColor = this->cs.tn ;       // color for version strings
   AppLang lang = this->cfgOpt.appLanguage ; // language index

   enum ctrls : short { closePB = ZERO, suppPB, haCTRLS } ;

   static const char* DialogText[] = 
   {
   //** English **
   " Taggit  : Audio file metadata (tag) editor and demo program for\n"
   "           development of a multilingual user interface.\n"
   "  Version:\n"
   "Copyright: (c) %s Mahlon R. Smith, The Software Samurai\n"
   "               Beijing University of Technology - Beijing, PRC\n"
   "               马伦教授 北京工业大学 - 北京，中华人民共和国\n"
   "\n"
   "Developed under Fedora Linux 20, using GNU G++ (Gcc v: 4.8.3)\n"
   "Software released under GNU General Public License, version 3,\n"
   "and documentation under GNU Free Documentation License, version 1.3\n"
   "               Bugs, suggestions, or possible praise?\n"
   "      Please contact the author at:  http://www.SoftwareSam.us/",

   //** Espanol **
   " Taggit  : Editor de metadatos de archivo de audio y ejemplo de \n"
   "           desarrollo para la interfaz de usuario multilingüe.\n"
   "  Versión:\n"
   "Propiedad Artística: (c) %s Mahlon R. Smith,\n"
   "                                   The Software Samurai\n"
   "                     Universidad Tecnológica de Beijing\n"
   "\n"
   "Desarrollado con Fedora Linux 20, utilizando GNU G++ (Gcc v: 4.8.3)\n"
   "Software lanzado bajo GNU General Public License, versión 3,\n"
   "y documentación bajo GNU Free Documentation License, versión 1.3\n"
   "       Informe de error, sugerencias, o posiblemente alabanza?\n"
   "    Por favor, contactar el autor en: http://www.SoftwareSam.us/",

   //** Zhongwen **
   " Taggit  : 用于开发多语言用户界面的音频文件元数据（标签）\n"
   "           编辑器和演示程序。\n"
   "     版本:\n"
   "     版权: (c) %s Mahlon R. Smith, The Software Samurai\n"
   "               马伦教授 北京工业大学 - 北京，中华人民共和国\n"
   "\n"
   "  在 Fedora Linux 20 上开发，使用 GNU G++ (Gcc v: 4.8.3)\n"
   "  根据GNU通用公共许可证发布的软件，版本3，和\n"
   "  根据GNU自由文档许可证，版本1.3发布的文档\n"
   "\n"
   "              对于软件错误，建议或可能的赞扬，\n"
   "              请联系作者：  http://www.SoftwareSam.us/\n",

   //** TiengViet **
   " Taggit  : Audio file metadata (tag) editor and demo program for\n"
   "           development of a multilingual user interface.\n"
   "Phiên Bản:\n"
   "Bản Quyền: (c) %s Mahlon R. Smith, The Software Samurai\n"
   "               Đại học Công nghệ Bắc Kinh - Bắc Kinh, Trung Quốc\n"
   "\n"
   "Được phát triển dưới Fedora Linux 20, sử dụng GNU G++ (Gcc v: 4.8.3)\n"
   "Phần mềm phát hành theo giấy phép GNU General Public, phiên bản 3,\n"
   "và tài liệu phát hành theo Giấy phép Tài liệu Tự do\n"
   "GNU, phiên bản 1.3\n"
   "     Đối với các lỗi phần mềm, đề nghị, hoặc có thể khen ngợi,\n"
   "     Vui lòng liên hệ với tác giả tại:  http://www.SoftwareSam.us/",
   } ;

   static const char* Labels[][3] = 
   {
      {  //** English **
         "    About Taggit    ",       // dialog title
         "    CLOSE    ",              // 'close' pushbutton
         "   SUPPORT INFORMATION   ",  // 'support' pushbutton
      },

      {  //** Espanol **
         "    Sobre Taggit    ",       // dialog title
         "   CIERRE    ",              // 'close' pushbutton
         " Información de Soporte  ",  // 'support' pushbutton
      },

      {  //** Zhongwen **
         "    关于  Taggit    ",        // dialog title
         "    关闭     ",               // 'close' pushbutton
         "      技术支持信息       ",    // 'support' pushbutton
         
      },

      {  //** TiengViet **
         "   Mọi Nơi  Taggit   ",       // dialog title
         " Đóng Cửa Sổ ",              // 'close' pushbutton
         "    Thông Tin Hỗ Trợ     ",  // 'support' pushbutton
      },

   } ;
   this->dPtr->SetDialogObscured () ;  // save parent dialog

   InitCtrl ic[haCTRLS] =        // array of dialog control info
   {
      {  //* 'CLOSE' pushbutton  - - - - - - - - - - - - - - - - - -   closePB *
         dctPUSHBUTTON,                // type:      
         rbtTYPES,                     // rbSubtype: (n/a)
         false,                        // rbSelect:  (n/a)
         short(dlgROWS - 2),           // ulY:       upper left corner in Y
         short(dlgCOLS / 2 - 7),       // ulX:       upper left corner in X
         1,                            // lines:     (n/a)
         13,                           // cols:      control columns
         Labels[lang][1],              // dispText:  
         this->cs.pn,                  // nColor:    non-focus color
         this->cs.pf,                  // fColor:    focus color
         tbPrint,                      // filter:    (n/a)
         "",                           // label:     (n/a)
         ZERO,                         // labY:      (n/a)
         ZERO,                         // labX       (n/a)
         ddBoxTYPES,                   // exType:    (n/a)
         1,                            // scrItems:  (n/a)
         1,                            // scrSel:    (n/a)
         NULL,                         // scrColor:  (n/a)
         NULL,                         // spinData:  (n/a)
         true,                         // active:    allow control to gain focus
         &ic[suppPB],                  // nextCtrl:  link in next structure
      },
      {  //* 'SUPPORT INFO' pushbutton   - - - - - - - - - - - - - - -  suppPB *
         dctPUSHBUTTON,                // type:      
         rbtTYPES,                     // rbSubtype: (n/a)
         false,                        // rbSelect:  (n/a)
         ic[closePB].ulY,              // ulY:       upper left corner in Y
         short(ic[closePB].ulX + ic[closePB].cols + 2), // ulX: upper left corner in X
         1,                            // lines:     (n/a)
         25,                           // cols:      control columns
         Labels[lang][2],              // dispText:  
         this->cs.pn,                  // nColor:    non-focus color
                                       // fColor:    focus color
         attr_t(this->cs.scheme == ncbcGR ? nc.brR : nc.grR), 
         tbPrint,                      // filter:    (n/a)
         "",                           // label:     (n/a)
         ZERO,                         // labY:      (n/a)
         ZERO,                         // labX       (n/a)
         ddBoxTYPES,                   // exType:    (n/a)
         1,                            // scrItems:  (n/a)
         1,                            // scrSel:    (n/a)
         NULL,                         // scrColor:  (n/a)
         NULL,                         // spinData:  (n/a)
         true,                         // active:    allow control to gain focus
         NULL                          // nextCtrl:  link in next structure
      },
   } ;

   //* Initial parameters for dialog window *
   InitNcDialog dInit( dlgROWS,        // number of display lines
                       dlgCOLS,        // number of display columns
                       ulY,            // Y offset from upper-left of terminal 
                       ulX,            // X offset from upper-left of terminal 
                       NULL,           // dialog title
                       ncltDUAL,       // border line-style
                       dColor,         // border color attribute
                       dColor,         // interior color attribute
                       ic              // pointer to list of control definitions
                     ) ;

   //* Instantiate the dialog window *
   NcDialog* dp = new NcDialog ( dInit ) ;

   //* If user interface language is an RTL language, *
   //* set the internal NcDialog flag for RTL output. *
   if ( this->cfgOpt.rtl )
      dp->DrawLabelsAsRTL () ;

   //* Open the dialog window *
   if ( (dp->OpenWindow()) == OK )
   {
      //* Set the dialog title *
      dp->SetDialogTitle ( Labels[lang][0], this->cs.em ) ;

      //* Print dialog window's static text *
      gString gsOut( DialogText[this->cfgOpt.appLanguage], copyrightYears ) ;
      dp->WriteParagraph ( 1, 1, gsOut, dColor ) ;

      //* Format application version string *
      gString verString( " %s ", AppVersion ) ;
      dp->WriteString ( 3, 12, verString, vColor ) ;

      dp->RefreshWin () ;                    // make the text visible

      uiInfo Info ;                 // user interface data returned here
      bool   done = false ;         // loop control
      while ( ! done )
      {
         if ( Info.viaHotkey )
            Info.HotData2Primary () ;
         else
            dp->EditPushbutton ( Info ) ;
         if ( Info.dataMod != false )
         {
            if ( Info.ctrlIndex == closePB )
               done = true ;
            else if ( Info.ctrlIndex == suppPB )
            {  //* Open a dialog to display interesting data needed to submit *
               //* a tech support request.                                    *
               //* Current dialog will be obscured, so save its display data. *
               dp->SetDialogObscured () ;
               this->haSupportInfo ( dciulY, dciulX, dciROWS, dciCOLS ) ;
               dp->NextControl () ;
               dp->RefreshWin () ;
            }
         }
         if ( !done && !Info.viaHotkey )
         {
            if ( Info.keyIn == nckSTAB )
               dp->PrevControl () ; 
            else if ( Info.keyIn != ZERO )
               dp->NextControl () ;
         }
      }     // while()
   }        // OpenWindow()
   if ( dp != NULL )                         // close the window
      delete ( dp ) ;

   this->dPtr->RefreshWin () ;               // restore parent dialog

}  //* End Cmd_HelpAbout() *

//*************************
//*     haSupportInfo     *
//*************************
//******************************************************************************
//* Display version numbers for NcDialog API, ncursesw library.                *
//* Also reports current 'locale' setting and dynamic load (shared) libraries. *
//*                                                                            *
//* Input  : ulY   : upper-left corner of dialog in Y                          *
//*          ulX   : upper-left corner of dialog in X                          *
//*          dRows : dialog rows                                               *
//*          dCols : dialog columns                                            *
//*                                                                            *
//* Returns: 'true' if support request saved, else 'false'                     *
//******************************************************************************
//* Notes: For technical support requests, ask user to save the information    *
//* to a file:                                                                 *
//*                                                                            *
//* 1) Header                                                                  *
//* 1) Taggit version                                                          *
//* 2) NcDialog API version                                                    *
//* 3) ncursesw version                                                        *
//* 4) locale                                                                  *
//* 5) DLLs                                                                    *
//* 6) system info: uname -a                                                   *
//* 7) Environment variables: printenv SHELL TERM COLORTERM LANG               *
//* 8) UDC time                                                                *
//* 9) Prompt for description of issue                                         *
//*                                                                            *
//******************************************************************************

bool Taggit::haSupportInfo ( short ulY, short ulX, short dRows, short dCols )
{
   static const char* Labels[][6] = 
   {
      {  //** English **
         "  Technical Support Information  ",      // dialog title
         "    CLOSE    ",                          // 'close' pushbutton
         "     SAVE TO FILE      ",                // 'save' pushbutton

         "    Please include this information\n"
         "    with all tech support requests.\n",

         "Dynamic (shared) Libs\n",

         "\nPlease describe the reason for this request."
         "\n (Please write in English, if possible.)"
         "\nInclude: a) the operation you were attempting to complete,"
         "\n            or the feature you would like to have added."
         "\n         b) what you believe should happen"
         "\n         c) what actually happened"
         "\n         d) any other information you think would be helpful",
      },

      {  //** Espanol **
         "  Información de Soporte Técnico  ",     // dialog title
         "   CIERRE    ",                          // 'close' pushbutton
         " Guardar En Un Archivo ",                // 'save' pushbutton

         "Por favor, incluya esta información con\n"
         "todas las solicitudes de soporte técnico.\n",
         "Bibliotecas Dinámicas\n",

         "\nDescriba el motivo de esta solicitud."
         "\n (Por favor escriba en inglés, si es posible.)"
         "\nIncluya: a) la operación que estaba intentando completar,"
         "\n            o la característica que usted quisiera haber agregado."
         "\n         b) ¿Qué crees que debería suceder?"
         "\n         c) ¿Lo que realmente pasó?"
         "\n         d) Cualquier otra información que usted piensa que sería útil.",
      },

      {  //** Zhongwen **
         "          技术支持信息          ",         // dialog title
         "    关闭     ",                           // 'close' pushbutton
         "       保存到文件      ",                  // 'save' pushbutton

         "    请将此信息与您的技术支\n"
         "    持请求一起提供。\n",
         "动态（共享）库\n",

         "\n请说明此请求的原因。"
         "\n （如果可能，请用英文写。）"
         "\n包括： a) 您尝试完成的操作，或您要添加的功能。"
         "\n       b) 你认为应该发生什么？"
         "\n       c) 实际发生了什么？"
         "\n       d) 任何其他您认为有用的信息。",
      },

      {  //** TiengViet **
         "   Thông Tin Hỗ Trợ Kỹ Thuật   ",        // dialog title
         " Đóng Cửa Sổ ",                          // 'close' pushbutton
         "  Lưu Vào Một Tập Tin  ",                // 'save' pushbutton

         "Xin bao gồm thông tin này với tất\n"
         "cả sự hỗ trợ kỹ thuật được yêu cầu.\n",
         "Năng Động Thư Viện\n",

         "\nVui lòng mô tả lý do cho yêu cầu này."
         "\n (Vui lòng viết bằng tiếng Anh, nếu có thể.)"
         "\nBao gồm: a) hoạt động bạn đang cố gắng để hoàn thành,"
         "\n            hoặc các tính năng mới, bạn mong muốn."
         "\n         b) Bạn nghĩ gì nên xảy ra?"
         "\n         c) Điều gì thực sự đã xảy ra?"
         "\n         d) Thêm bất kỳ thông tin hữu ích khác.",
      },

   } ;

   AppLang lang = this->cfgOpt.appLanguage ; // language index
   attr_t dColor = this->cs.sd,     // dialog interior color
          vColor = this->cs.tn,     // color for version strings
          hColor = this->cs.em ;    // highlight color
   bool tsRequestSaved = false ;    // return value

   enum ctrls : short { closePB = ZERO, savePB, controlsDEFINED } ;


   InitCtrl ic[controlsDEFINED] =      // array of dialog control info
   {
   {  //* 'CLOSE' pushbutton  - - - - - - - - - - - - - - - - - - - -  closePB *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(dRows - 2),             // ulY:       upper left corner in Y
      short(dCols / 2 - 7),         // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      13,                           // cols:      control columns
      Labels[lang][1],              // dispText:  
      this->cs.pn,                  // nColor:    non-focus color
      this->cs.pf,                  // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "",                           // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[savePB]                   // nextCtrl:  link in next structure
   },
   {  //* 'SAVE' pushbutton   - - - - - - - - - - - - - - - - - - - -   savePB *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      ic[closePB].ulY,              // ulY:       upper left corner in Y
      2,                            // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      23,                           // cols:      control columns
      Labels[lang][2],              // dispText:  
      this->cs.pn,                  // nColor:    non-focus color
      attr_t(this->cs.scheme == ncbcGR ? nc.brR : nc.grR),
      tbPrint,                      // filter:    (n/a)
      "",                           // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      NULL                          // nextCtrl:  link in next structure
   },
   } ;

   //* Initial parameters for dialog window *
   InitNcDialog dInit( dRows,          // number of display lines
                       dCols,          // number of display columns
                       ulY,            // Y offset from upper-left of terminal 
                       ulX,            // X offset from upper-left of terminal 
                       NULL,           // dialog title
                       ncltDUAL,       // border line-style
                       dColor,         // border color attribute
                       dColor,         // interior color attribute
                       ic              // pointer to list of control definitions
                     ) ;

   //* Instantiate the dialog window *
   NcDialog* dp = new NcDialog ( dInit ) ;

   //* If user interface language is an RTL language, *
   //* set the internal NcDialog flag for RTL output. *
   if ( this->cfgOpt.rtl )
      dp->DrawLabelsAsRTL () ;

   //* Open the dialog window *
   if ( (dp->OpenWindow()) == OK )
   {
      //* Set the dialog title *
      dp->SetDialogTitle ( Labels[lang][0], hColor ) ;

      localTime lt ;                   // formatted timestamp
      this->GetLocalTime ( lt ) ;      // get the local time
      gString gsOut( "  Taggit Tech Support Request\n"
                     "===============================\n"
                     "Taggit v:%s\n"
                     "Requested: %04hd-%02hd-%02hdT%02hd:%02hd:%02hd\n", 
                     AppVersion, &lt.year, &lt.month, &lt.date,
                     &lt.hours, &lt.minutes, &lt.seconds ) ;

      //* Create a temporary log *
      gString gsLog ;
      this->CreateTempname ( gsLog ) ;
      ofstream ofs( gsLog.ustr(), ofstream::out | ofstream::trunc ) ;
      ofs << gsOut.ustr() ;

      winPos wp( 1, 2 ) ;
      gsOut = Labels[lang][3] ;
      wp = dp->WriteParagraph ( wp, gsOut, hColor ) ;
      ofs << '\n' << gsOut.ustr() << '\n' ;
      gsOut = "NcDialog API library : " ;
      wp = dp->WriteString ( wp, gsOut, dColor ) ;
      ofs << gsOut.ustr() ;
      gsOut.compose( " %s \n\n", dp->Get_NcDialog_Version () ) ;
      wp = dp->WriteParagraph ( wp, gsOut, vColor ) ;
      ofs << gsOut.ustr() ;

      gsOut = "    ncursesw library : " ;
      wp = dp->WriteString ( wp.ypos, 2, gsOut, dColor ) ;
      ofs << gsOut.ustr() ;
      gsOut.compose( L" %s \n\n", nc.Get_nclibrary_Version () ) ;
      wp = dp->WriteParagraph ( wp, gsOut, vColor ) ;
      ofs << gsOut.ustr() ;

      gsOut = "      Locale setting : " ;
      wp = dp->WriteString ( wp.ypos, 2, gsOut, dColor ) ;
      ofs << gsOut.ustr() ;
      gsOut.compose( L" %s \n", nc.GetLocale () ) ;
      wp = dp->WriteParagraph ( wp, gsOut, vColor ) ;
      ofs << gsOut.ustr() ;
      gsOut.compose( L" (%S)\n", alStrings[lang] ) ;
      wp = dp->WriteParagraph ( wp, gsOut, dColor ) ;
      ofs << "                       " << gsOut.ustr() << '\n' ;

      //* Display a list of shared libraries used in the application *
      gString gsTmp, gx ;
      this->CreateTempname ( gsTmp ) ;
      short libCount = ZERO ;
      wp = { 1, 43 } ;
      gsOut = Labels[lang][4] ;
      wp = dp->WriteParagraph ( wp, gsOut, hColor ) ;
      ofs << gsOut.ustr() ;
      gsOut.compose( "ldd \"%s/taggit\" 1>\"%S\" 2>/dev/null", 
                     this->cfgOpt.appPath, gsTmp.gstr() ) ;
      system ( gsOut.ustr() ) ;
      gsOut.clear() ;
      ifstream ifs( gsTmp.ustr(), ifstream::in ) ;
      if ( ifs.is_open() )
      {
         char  lineData[gsMAXBYTES] ;
         short sIndex ;
         bool  done = false ;

         while ( ! done )
         {
            ifs.getline( lineData, gsMAXBYTES, NEWLINE ) ;
            if ( ifs.good() || (ifs.gcount() > ZERO) )
            {
               ofs << &lineData[1] << '\n' ; // write all entries to log

               if ( libCount < 8 )
               {
                  gx = lineData ;
                  if ( ((sIndex = gx.find( L' ' )) > ZERO) &&
                       ((gx.find( "lib" )) == 1) ) // (line begins with "\tlib")
                  {
                     gx.limitChars( sIndex ) ;
                     gsOut.append( "%S\n", gx.gstr() ) ;
                     ++libCount ;
                  }
               }
            }
            else
               done = true ;
         }
         ifs.close() ;
      }
      if ( libCount == ZERO )
         gsOut = "(Unable to determine.)\n" ;
      wp = dp->WriteParagraph ( wp, gsOut, dColor ) ;
      this->DeleteTempname ( gsTmp ) ;

      //* Supplementary information for the log file *
      ofs << "\nSystem Information:" << endl ;
      ofs.close() ;     // temporarily close the log file

      gsOut.compose( "printenv SHELL TERM COLORTERM LANG 1>>\"%S\" 2>/dev/null",
                     gsLog.gstr() ) ;
      system ( gsOut.ustr() ) ;
      gsOut.compose( "uname -srvmpio 1>>\"%S\" 2>/dev/null", gsLog.gstr() ) ;
      system ( gsOut.ustr() ) ;

      ofs.open( gsLog.ustr(), ofstream::out | ofstream::app ) ;
      if ( ofs.is_open() )
      {
         ofs << "\n------------------------------------------------------------"
             << Labels[lang][5] << "\n\n\n" << endl ;
         ofs.close() ;              // close the log file
      }

      dp->RefreshWin () ;        // make everything visible

      uiInfo   Info ;            // user interface data returned here
      short    icIndex = ZERO ;  // index of control with input focus
      bool     done = false ;    // loop control
      while ( ! done )
      {
         if ( ic[icIndex].type == dctPUSHBUTTON )
         {
            if ( Info.viaHotkey )
               Info.HotData2Primary () ;
            else
               icIndex = dp->EditPushbutton ( Info ) ;

            if ( Info.dataMod != false )
            {
               if ( icIndex == closePB )
                  done = true ;
               else if ( icIndex == savePB )
               {
#if 0    // TEMP TEMP TEMP
dp->CaptureDialog ( "capturedlg.txt" ) ;
dp->CaptureDialog ( "capturedlg.html", true, false, 
                    "infodoc-styles.css", 4, false, nc.blR ) ;
#endif   // TEMP TEMP TEMP
                  //* Do a simple 'move-with-rename' of the temp file *
                  //* to the specified source directory.              *
                  char cmd[gsMAXCHARS * 2] ;
                  if ( (snprintf ( cmd, (gsMAXCHARS * 2), 
                             "mv \"%s\" \"%s/%s\" 1>/dev/null 2>/dev/null", 
                             gsLog.ustr(), this->cfgOpt.srcPath, tsFName )) 
                       >= (gsMAXCHARS * 2) ) ;
                  system ( cmd ) ;
                  tsRequestSaved = true ;
                  gsOut.compose( "Saved to: %s", tsFName ) ;
                  dp->WriteString ( ic[savePB].ulY - 1, ic[savePB].ulX,
                                    gsOut, hColor, true ) ;
               }
            }
         }

         //* Move focus to appropriate control *
         if ( ! done && ! Info.viaHotkey )
         {
            if ( Info.keyIn == nckSTAB )
               icIndex = dp->PrevControl () ; 
            else
               icIndex = dp->NextControl () ;
         }
      }        // while(!done)

      this->DeleteTempname ( gsLog ) ;       // delete temp log (if any)
   }
   if ( dp != NULL )
      delete ( dp ) ;                        // close the window

   return tsRequestSaved ;

}  //* End haSupportInfo() *

//*************************
//*     Cmd_ShellOut      *
//*************************
//******************************************************************************
//* Put the application into hibernation mode and return user to the command   *
//* line with an 'Exit-To-Return' message.                                     *
//*                                                                            *
//* Input  : prompt  : (optional, 'false' by default)                          *
//*                    if 'false' go directly to the shell                     *
//*                    if 'true'  prompt user for an external command to be    *
//*                               executed [CURRENTLY IGNORED]                 *
//*          extCmd  : (optional, NULL pointer by default)                     *
//*                    if specified, instruct the shell to immediately execute *
//*                    the specified command [CURRENTLY IGNORED]               *
//*                                                                            *
//* Returns: 'false' if successful AND if source files are unchanged           *
//*          'true'  if one or more source files are no longer accessible.     *
//******************************************************************************
//* Notes:                                                                     *
//*                                                                            *
//* -- Important Note: We must ASSUME that the calling thread is the only      *
//*    thread that is currently awake. It would be embarrassing if a secondary *
//*    thread were still trying to execute when the application is in          *
//*    hibernation mode.                                                       *
//*                                                                            *
//* -- 'prompt' parameter: [NOT FULLY IMPLEMENTED]                             *
//*     Currently, we ignore this parameter and assume that caller has sent us *
//*     an external command ('extCmd'). In future, we may expand this method   *
//*     prompt the user for an external command to be executed.                *
//*                                                                            *
//* -- 'extCmd' parameter:                                                     *
//*     We trust the caller has created a valid command string, and that the   *
//*     command executed will allow us to return immediately to application    *
//*     control when it finishes.                                              *
//*                                                                            *
//* -- A user will occasionally forget that he/she/it is shelled out, but      *
//*    will remember when they try to close the terminal window and get an     *
//*    'active-process' message.                                               *
//******************************************************************************

void Taggit::Cmd_ShellOut ( bool prompt, const char* extCmd )
{
   //* Set the parameters *
   soOptions option = ((extCmd != NULL) ? soX : soE) ;

   //* Shell out *
   this->dPtr->ShellOut ( option, extCmd ) ;

}  //* End Cmd_ShellOut() *

//*************************
//*    Cmd_TagSummary     *
//*************************
//******************************************************************************
//* Display metadata summary for file(s), and optionally save a copy of the    *
//* summary/summaries to a file.                                               *
//*                                                                            *
//* Input  : viewData : 'true' shell out and display the summary with 'less'   *
//*                     'false' save the summary data to user space but do     *
//*                             not display it immediately                     *
//*          saveData : 'true' write the summary data to a file in user space  *
//*                     'false discard the summary data after viewing          *
//*          allFiles : (optional, 'true' by default)                          *
//*                     'true' summarize all source files                      *
//*                     'false' summarize only the highlighted file            *
//*          prompt   : (optional, 'false' by default)                         *
//*                     'true'  open a sub-dialog so user can set parameters   *
//*                     'false' no user interaction                            *
//*                                                                            *
//* Returns: 'true'  if summary file successfully written to user space        *
//*          'false' if data discarded OR if file-creation error(s)            *
//******************************************************************************
//* Notes:                                                                     *
//* 1) Called from three places:                                               *
//*    a) in response to the '-D' (Dump) command-line option                   *
//*       -- All data are summarized.                                          *
//*       -- The data are saved to a file.                                     *
//*       -- The data are not displayed.                                       *
//*    b) in response to the View Menu 'View Tag Data Summary' option          *
//*       -- All data are sumarized.                                           *
//*       -- The data are not saved to a user-space file.                      *
//*       -- The data are displayed using the 'less' utility.                  *
//*    c) in response to the File Menu 'Save Tag Summary' option               *
//*       -- This is an interactive option which will open a sub-dialog.       *
//*          -- All data or highlighted file only?                             *
//*          -- All active fields included?, or only populated fields?         *
//*          -- The data are saved to a file unless user cancels the operation.*
//*          -- The data are not displayed.                                    *
//*                                                                            *
//* 2) Inactive fields are never included in the summary.                      *
//* 3) Active, but empty fields are included in the summary only by user       *
//*    request.                                                                *
//* 4) RTL UI languages:                                                       *
//*    a) MP3 field labels are written as RTL (not OGG/Vorbis labels).         *
//*    b) Field data are written as RTL by user request.                       *
//*       -- directly through the prompt dialog                                *
//*       -- indirectly through the 'RTL Invert Field Data' menu item          *
//*       -- For the Dump option field data are at this time, always           *
//*          written as LTR.                                                   *
//*                                                                            *
//******************************************************************************

bool Taggit::Cmd_TagSummary ( bool viewData, bool saveData, 
                              bool allFiles,  bool prompt )
{
   gString tmpPath ;                // create a temporary file
   bool  goodTPath = this->CreateTempname ( tmpPath ),
         goodTFile = false,         // 'true' if summary captured
         allData   = false,         // if 'true' include unpopulated fields
         fldRTL    = false,         // toggle for RTL display of tag fields
         status    = false ;        // return value


   //* Interactive experience - called from File/Save Tag Summary *
   if ( prompt )
   {
      if ( (this->ctsPrompt ( allFiles, allData, fldRTL )) == false )
         viewData = saveData = false ;
   }
   else if ( viewData )
      fldRTL = this->cfgOpt.rtlf ;  // write field data as RTL
   if ( ! goodTPath )    // file creation error (this is unlikely)
      viewData = saveData = false ;

   //* If user has not aborted, create the summary file *
   if ( viewData || saveData )
      goodTFile = this->ctsSummarize ( tmpPath, allFiles, allData, fldRTL ) ;

   //* Save only - called from command-line (Dump option) *
   //* OR from File/Save Tag Summary.                     *
   if ( goodTFile && saveData && ! viewData )
   {
      //* Copy the temp file to user's directory.*
      gString trgPath( "%s/%s", this->cfgOpt.srcPath, mdDumpName ) ;
      if ( (this->CopyFile ( tmpPath, trgPath )) != false )
      {  //* If interactive, then use info dialog to alert   *
         //* user that summary file was written successfully.*
         if ( prompt )
            this->ctsVerifySave ( trgPath ) ;

         status = true ;   // alert caller that user has a copy of file
      }
   }

   //* View Only - called from View/View Tag Data Summary *
   else if ( goodTFile && viewData && ! saveData )
   {
      gString gs( "less -c '%S'", tmpPath.gstr() ) ;
      this->Cmd_ShellOut ( false, gs.ustr() ) ;
   }
   //* Delete the temp file *
   this->DeleteTempname ( tmpPath ) ;

   return status ;

}  //* End Cmd_TagSummary() *

//*************************
//*       ctsPrompt       *
//*************************
//******************************************************************************
//* Ask user to specify parameters for the Tag Summary report.                 *
//*                                                                            *
//* Input  : allFiles : (by refrence, initial value ignored)                   *
//*                     'true' summarize all source files                      *
//*                     'false' summarize only the highlighted file            *
//*          allData  : (by refrence, initial value ignored)                   *
//*                     'true' include all active fields in output             *
//*                     'false' include only non-empty fields in output        *
//*          fldRTL   : (by reference, initial value ignored)                  *
//*                     referenced ONLY when RTL UI language                   *
//*                     'false' write field data to file as LTR                *
//*                     'true'  write field data to file as RTL                *
//*                                                                            *
//* Returns: 'true'  if parameters selected                                    *
//*          'false' if user aborts the operation                              *
//******************************************************************************

bool Taggit::ctsPrompt ( bool& allFiles, bool& allData, bool& fldRTL )
{
   const short dlgROWS = 16,              // dialog size
               dlgCOLS = 52,
               dlgY    = fieldwinY + 1,   // dialog position
               dlgX    = fieldwinX + 1 ;
   short       rbX     = 2,               // radiobutton X offset
               rblX    = 7 ;              // radiobutton label X offset
   const wchar_t lArrow[] = L"\x21E6\x21E6\x21E6" ;
   const wchar_t rArrow[] = L"\x21E8\x21E8\x21E8" ;
   attr_t dColor = this->cs.sd,           // dialog interior color
          hColor = this->cs.em ;          // bold dialog text
   AppLang lang = this->cfgOpt.appLanguage ; // language index
   gString gs ;                           // text formatting
   bool status = false ;                  // return value

   enum ctspControls : short 
   { savePB = ZERO, cancelPB, allFileRB, oneFileRB, 
     dtaFieldRB, allFieldRB, toggPB, ctCTRLS } ;

   //* Control labels *
   const char* Labels[][ctCTRLS + 1] = 
   {
      {  // English
         "     SAVE     ",             // 'save' pushbutton
         "    CANCEL    ",             // 'cancel' pushbutton
         "Summarize all source files", // 'all files' radiobutton
         "Summarize only the highlighted file", // 'one file' radiobutton
         "Write only fields which contain data",// 'populated fields' radiobutton
         "Write all active fields,\n"  // 'all active fields' radiobutton
         "including blank fields",
         "  Save Tag Data Summary  ",  // Dialog title
      },
      {  // Espanol
         "   GUARDAR    ",             // 'save' pushbutton
         "   CANCELAR   ",             // 'cancel' pushbutton
         "Resumir todos los archivos de origen", // 'all files' radiobutton
         "Resumir sólo el archivo resaltado", // 'one file' radiobutton
         "Escribir sólo los campos que contienen\ndatos",   // 'populated fields' radiobutton
         "Escribir todos los campos activos,\n"  // 'all active fields' radiobutton
         "incluidos los campos en blanco",
         "  Guardar el Resumen de Datos de Etiqueta  ",  // Dialog title
      },
      {  // Zhongwen
         "     保存     ",              // 'save' pushbutton
         "     取消     ",              // 'cancel' pushbutton
         "汇总所有源文件", // 'all files' radiobutton
         "仅汇总突出显示的文件", // 'one file' radiobutton
         "仅写入包含数据的字段",   // 'populated fields' radiobutton
         "写所有活动字段，这包括空白字段",  // 'all active fields' radiobutton
         "  保存标签数据摘要  ",  // Dialog title
      },
      {  // TiengViet
         "    Để Dành   ",             // 'save' pushbutton
         "    Hủy Bỏ    ",             // 'cancel' pushbutton
         "Tóm tắt tất cả các tập tin nguồn", // 'all files' radiobutton
         "Tóm tắt của một tập tin duy nhất", // 'one file' radiobutton
         "Viết chỉ các trường có chứa dữ liệu",   // 'populated fields' radiobutton
         "Viết tất cả các dữ liệu,\n"  // 'all active fields' radiobutton
         "bao gồm các lĩnh vực có sản phẩm nào",
         "  Lưu Tóm Tắt Dữ Liệu Nhãn  ",  // Dialog title
      },
#if 0    // EXPERIMENTAL
      {  // Arabic
         "      حفظ     ",                // 'save' pushbutton
         "     إلغاء    ",                // 'cancel' pushbutton
         "تلخيص جميع ملفات المصدر.",      // 'all files' radiobutton
         "تلخيص الملف الحالي فقط.",       // 'one file' radiobutton
         "اكتب الحقول التي تحتوي على بيانات فقط.",// 'populated fields' radiobutton
         "اكتب جميع الحقول النشطة، \n"    // 'all active fields' radiobutton
         "بما في ذلك الحقول الفارغة.",
         "  حفظ ملخص بيانات العلامة  ",  // Dialog title
      },
      // HTML5 NOTES: dir=[auto | ltr | rtl]; 
      //               &lrm;  &rlm;
      //              <bdo dir="ltr> w x y z </bdo>
      //              <bdi dir="ltr> w x y z </bdo>
      // https://www.w3.org/International/articles/inline-bidi-markup/
#endif   // EXPERIMENTAL
   } ;
// lang = AppLang(4) ; // TEMP - TEST ARABIC OUTPUT.


   //* If user interface language is an RTL language, *
   //* adjust control and label positions.            *
   if ( this->cfgOpt.rtl )
   {
      rbX = (dlgCOLS - 7) ;
      rblX = -2 ;
   }

   //* Control definitions *
   InitCtrl ic[ctCTRLS] =              // array of dialog control info
   {
   {  //* 'Save' pushbutton - - - - - - - - - - - - - - - - - - - - -   savePB *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(dlgROWS - 2),           // ulY:       upper left corner in Y
      short(dlgCOLS / 2 - 15),      // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      14,                           // cols:      control columns
      Labels[lang][savePB],         // dispText:  
      this->cs.pn,                  // nColor:    non-focus color
      this->cs.pf,                  // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "",                           // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[cancelPB],                // nextCtrl:  link in next structure
   },
   {  //* 'Cancel' pushbutton - - - - - - - - - - - - - - - - - - -   cancelPB *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      ic[savePB].ulY,               // ulY:       upper left corner in Y
      short(ic[savePB].ulX + ic[savePB].cols + 2), // ulX: upper left corner in X
      1,                            // lines:     (n/a)
      14,                           // cols:      control columns
      Labels[lang][cancelPB],       // dispText:  
      this->cs.pn,                  // nColor:    non-focus color
      this->cs.pf,                  // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "",                           // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[allFileRB]                // nextCtrl:  link in next structure
   },
   {  //* 'All Files' radiobutton  - - - - - - - - - - - - - - - -   allFileRB *
      dctRADIOBUTTON,               // type:      
      rbtS5s,                       // sbSubtype: alt font, 5-wide
      true,                         // rbSelect:  initially set
      2,                            // ulY:       upper left corner in Y
      rbX,                          // ulX: upper left corner in X
      1,                            // lines:     (n/a)
      1,                            // cols:      (n/a)
      NULL,                         // dispText:  
      dColor,                       // nColor:    non-focus color
      this->cs.pf,                  // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      Labels[lang][allFileRB],      // label:     
      0,                            // labY:      
      rblX,                         // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[oneFileRB]                // nextCtrl:  link in next structure
   },
   {  //* 'Highlighted File' radiobutton  - - - - - - - - - - - - -  oneFileRB *
      dctRADIOBUTTON,               // type:      
      rbtS5s,                       // sbSubtype: alt font, 5-wide
      false,                        // rbSelect:  initially reset
      short(ic[allFileRB].ulY + 2), // ulY:       upper left corner in Y
      rbX,                          // ulX: upper left corner in X
      1,                            // lines:     (n/a)
      1,                            // cols:      (n/a)
      NULL,                         // dispText:  
      dColor,                       // nColor:    non-focus color
      this->cs.pf,                  // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      Labels[lang][oneFileRB],      // label:     
      0,                            // labY:      
      rblX,                         // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[dtaFieldRB]               // nextCtrl:  link in next structure
   },
   {  //* 'Non-empty Fields' radiobutton  - - - - - - - - - - - - - dtaFieldRB *
      dctRADIOBUTTON,               // type:      
      rbtS5s,                       // sbSubtype: alt font, 5-wide
      true,                         // rbSelect:  initially set
      short(ic[oneFileRB].ulY + 3), // ulY:       upper left corner in Y
      rbX,                          // ulX: upper left corner in X
      1,                            // lines:     (n/a)
      1,                            // cols:      (n/a)
      NULL,                         // dispText:  
      dColor,                       // nColor:    non-focus color
      this->cs.pf,                  // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      Labels[lang][dtaFieldRB],     // label:     
      0,                            // labY:      
      rblX,                         // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[allFieldRB]               // nextCtrl:  link in next structure
   },
   {  //* 'All-active Fields' radiobutton   - - - - - - - - - - - - allFieldRB *
      dctRADIOBUTTON,               // type:      
      rbtS5s,                       // sbSubtype: alt font, 5-wide
      false,                        // rbSelect:  initially reset
      short(ic[dtaFieldRB].ulY + 2),// ulY:       upper left corner in Y
      rbX,                          // ulX: upper left corner in X
      1,                            // lines:     (n/a)
      1,                            // cols:      (n/a)
      NULL,                         // dispText:  
      dColor,                       // nColor:    non-focus color
      this->cs.pf,                  // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      Labels[lang][allFieldRB],     // label:     
      0,                            // labY:      
      rblX,                         // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      NULL                          // nextCtrl:  link in next structure
   },
   {  //* 'Toggle' pushbutton - - - - - - - - - - - - - - - - - - - -   toggPB *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      ic[savePB].ulY,               // ulY:       upper left corner in Y
      2,                            // ulX: upper left corner in X
      1,                            // lines:     (n/a)
      5,                            // cols:      control columns
      " LT^R ",                     // dispText:  
      this->cs.pn,                  // nColor:    non-focus color
      this->cs.pf,                  // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "",                           // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      NULL                          // nextCtrl:  link in next structure
   },
   } ;

   //* Dialog definition *
   InitNcDialog dInit( dlgROWS,         // number of display lines
                       dlgCOLS,         // number of display columns
                       dlgY,            // Y offset from upper-left of terminal 
                       dlgX,            // X offset from upper-left of terminal 
                       NULL,            // dialog title
                       ncltDUAL,        // border line-style
                       dColor,          // border color attribute
                       dColor,          // interior color attribute
                       ic               // pointer to list of control definitions
                     ) ;

   this->dPtr->SetDialogObscured () ;  // save parent dialog

   //* If user interface is an RTL language, enable the optional *
   //* Pushbutton for toggling display of the written records.   *
   if ( this->cfgOpt.rtl )
      ic[allFieldRB].nextCtrl = &ic[toggPB] ;

   //* Open the dialog *
   NcDialog* dp = new NcDialog ( dInit ) ;

   //* If user interface language is an RTL language, *
   //* set the internal NcDialog flag for RTL output. *
   if ( this->cfgOpt.rtl )
      dp->DrawLabelsAsRTL () ;

   if ( (dp->OpenWindow()) == OK )
   {
      //* Set the dialog title *
      dp->SetDialogTitle ( Labels[lang][ctCTRLS - 1], hColor ) ;

      //* Create radiobutton XOR groups *
      short xorGroup[3] { oneFileRB, allFileRB, -1 } ;
      dp->GroupRadiobuttons ( xorGroup ) ;
      xorGroup[0] = dtaFieldRB ; xorGroup[1] = allFieldRB ;
      dp->GroupRadiobuttons ( xorGroup ) ;

      //* Static arrow indicating LTR/RTL field display direction *
      if ( this->cfgOpt.rtl )
         dp->WriteString ( ic[toggPB].ulY - 1, ic[toggPB].ulX + 1,
                           (fldRTL ? lArrow : rArrow), hColor, true ) ;
         
      dp->RefreshWin () ;                 // make the text visible

      //* Interact with the user *
      uiInfo   Info ;                     // user interface data returned here
      short    icIndex = savePB ;         // index of control with focus
      bool     done = false ;             // loop control
      while ( ! done )
      {
         if ( ic[icIndex].type == dctPUSHBUTTON )
         {
            if ( Info.viaHotkey )
               Info.HotData2Primary () ;
            else
               icIndex = dp->EditPushbutton ( Info ) ;
            if ( Info.dataMod != false )
            {
               if ( Info.ctrlIndex == savePB )
               {
                  dp->GetRadiobuttonState ( allFileRB, allFiles ) ;
                  dp->GetRadiobuttonState ( allFieldRB, allData ) ;
                  status = done = true ;
               }
               else if ( Info.ctrlIndex == cancelPB )
               {
                  done = true ;
               }
               else if ( Info.ctrlIndex == toggPB )
               {  //* Note: We will arrive here ONLY if UI language is RTL. *
                  fldRTL = fldRTL ? false : true ;
                  dp->WriteString ( ic[toggPB].ulY - 1, ic[toggPB].ulX + 1,
                                    (fldRTL ? lArrow : rArrow), hColor, true ) ;
               }
            }
         }

         else if ( ic[icIndex].type == dctRADIOBUTTON )
         {
            if ( Info.viaHotkey )
               Info.HotData2Primary () ;
            else
               icIndex = dp->EditRadiobutton ( Info ) ;
            if ( Info.dataMod != false )
            { /* Nothing to do */ }
         }

         //* Move focus to appropriate control *
         if ( ! done && ! Info.viaHotkey )
         {
            if ( Info.keyIn == nckSTAB )
               icIndex = dp->PrevControl () ; 
            else
               icIndex = dp->NextControl () ;
         }
      }  // while()
   }
   if ( dp != NULL )                         // close the window
      delete ( dp ) ;

   this->dPtr->RefreshWin () ;               // restore parent dialog

   return status ;

}  //* end ctsPrompt() *

//*************************
//*     ctsSummarize      *
//*************************
//******************************************************************************
//* Write metadata summary to a temporary file.                                *
//* Called only by Cmd_TagSummary().                                           *
//*                                                                            *
//* Input  : tmpPath  : filespec of allocated (not written) temp file          *
//*          allFiles : 'true' metadata summary for ALL source files           *
//*                     'false' metadata summary for highlighted file only     *
//*          allData  : 'true' write all active fields                         *
//*                     'false' write only non-empty fields                    *
//*          fldRTL   : 'true' write the tag text as RTL data                  *
//*                     'false' write the tag text as LTR data                 *
//*                                                                            *
//* Returns: 'true'  if summary file successfully written to the temp file     *
//*          'false' if file-creation error(s)                                 *
//******************************************************************************
//* Notes:                                                                     *
//* ------                                                                     *
//* RTL Output:                                                                *
//* 1) Because the output stream doesn't know anything about RTL text, we need *
//*    to manually reverse RTL text for writing to the file.                   *
//* 2) The MP3 field labels can be reversed without too much trouble because   *
//*    starting output at the left margin, while not consistent with the       *
//*    language, does not inhibit clarity.                                     *
//* 3) Because OGG/Vorbis tag titles are predefined, ASCII English text,       *
//*    we do not reverse them.                                                 *
//* 4) The real problem is that we don't know if the tag text is LTR or RTL    *
//*    data, so we must ask the user to decide. The user makes this decision   *
//*    at a higher level, (see ctsPrompt()) via the optional RTL Pushbutton    *
//*    (hotkey: CTRL+R).                                                       *
//*                                                                            *
//******************************************************************************

bool Taggit::ctsSummarize ( const gString& tmpPath, bool allFiles, 
                            bool allData, bool fldRTL )
{
   #define SUMMARIZE_IMAGES (1)     // include image summary in output
   #define SUMMARIZE_POPCNT (1)     // include popularimeter/counter summary

   const char* dSPC = "  " ;       // Two consecutive space characters
   const wchar_t* baseDash  = L"--------------------------------" ;
   const char* NoFiles[] = 
   {
      "#  No source files specified.\n",
      "#  No se especificaron archivos de origen.\n",
      "#  没有指定源文件。\n",
      "#  Không có tập tin nguồn đã được quy định.\n",
   } ;
   const char* NoData[] = 
   {
      "  [file contains no metadata]\n",
      "  [el archivo no contiene metadatos]\n",
      "  [文件不包含元数据]\n",
      "  [Các tập tin không chứa siêu dữ liệu]\n",
   } ;
   const char* DataHdr[] = 
   {
      "# Metadata Summary",
      "# Resumen de Metadatos",
      "# 元数据摘要",
      "# Tóm tắt siêu dữ liệu",
      // Programmer's Note: For RTL languages, reverse the direction of 
      // characters for the header text. Example: "# yrammuS atadateM"
      // Example: # ملخص البيانات الوصفية
   } ;
   const char* ImgTemplate[] = 
   {
      "  Image: %S bytes, MIME:%s, Type:%S, Desc:",          // English
      "  Imagen: %S bytes, MIME:%s, Tipo:%S, Descripción:", // Espanol
      "  图片: %S 字节, MIME:%s, 类型:%S, 描述:",              // Zhongwen
      "  Hình ảnh: %S bytes, MIME:%s, Kiểu:%S, Sự vẻ:",       // TiengViet
      // Programmer's Note: Support for RTL languages is not straightforward for
      // output of image data. The words "Image:" "bytes" "Type:" and "Desc:" 
      // in the above template are always displayed as LTR data. This can be 
      // overcome for RTL languages by reversing the characters for these words 
      // within the template. 
      // Example: "egamI: %S setyb, MIME:%s epyT:%S cseD:"
   } ;
   const char* PopTemplate[] = 
   {
      "  Popularity: %hhu of 255, (%u) Message:'%s' Play-count: %u",   // English
      "  Popularidad: %hhu de 255, (%u) el Texto:'%s' el Contador: %u", // Espanol
      "  人气: %hhu 的 255, (%u) 信息:'%s' 回放计数器: %u",                // Zhongwen
      "  Xếp hạng phổ biến: %hhu của 255, (%u) Thông điệp:'%s' Tổng số: %u", // TiengViet
   } ;

   AppLang lang = this->cfgOpt.appLanguage ;
   const char* const* umwHPtr = umwHeaders[lang] ; // tag-field headings
   gString gs, gx ;                          // data formatting
   short indx,
         fcnt ;                              // non-null field count
   bool  status = false ;                    // return value

   //* Open the temp file for output *
   ofstream ofs( tmpPath.ustr(), ofstream::out | ofstream::trunc ) ;
   if ( (ofs.is_open()) != false )
   {  //* Write file header info *
      localTime ft ;                // get local time
      this->GetLocalTime ( ft ) ;
      gs.compose( "%s - %04hd-%02hd-%02hdT%02hd:%02hd:%02hd (Taggit)\n", 
                  DataHdr[lang], &ft.year, &ft.month, &ft.date,
                  &ft.hours, &ft.minutes, &ft.seconds ) ;
      ofs << gs.ustr() 
          << "------------------------------------------------------------\n\n" ;

      //* Format the metadata and write it to the temp file. If 'all', *
      //* start at index ZERO, else set index of highlighted file.     *
      short fIndex = allFiles ? ZERO : this->tData.sfFocus ;
      for ( ; fIndex < this->tData.sfCount ; ++fIndex )
      {
         //* Write the filename *
         gs = this->tData.sf[fIndex].sfName ;
         ofs << "# " << gs.ustr() << endl ;

         //* Write the tag data *
         fcnt = ZERO ;
         for ( short fldIndex = ZERO ; fldIndex < tfCOUNT ; ++fldIndex )
         {  //* If field contains data OR 'allData' specified *
            if ( allData || (*this->tData.sf[fIndex].sfTag.field[fldIndex] != NULLCHAR) )
            {
               if ( this->tData.sf[fIndex].sfType == mftOGG )
               {
                  gs = this->oggUnmapField ( fldIndex ) ;
                  //* The default field mapping for OGG fields is the     *
                  //* 'tfTxxx' (comment) field, but if the display field  *
                  //* contains  no data, skip over it.                    *
                  if ( (*this->tData.sf[fIndex].sfTag.field[fldIndex] == NULLCHAR)
                       && ((gs.compare( OGG_DFLT_COMMENT )) == ZERO) )
                  continue ;
                  gs.append( baseDash ) ;
               }
               else     // MP3 is the default
               {
                  //* Get the field text *
                  gs = umwHPtr[fldIndex] ;
                  //* For narrow fields, pad to full width *
                  if ( (fldIndex == tfTrck) || (fldIndex == tfTyer) )
                     gs.append( baseField ) ;
                  indx = (gs.find( dSPC )) + 1 ;
                  if ( this->cfgOpt.rtl )
                  {
                     gx = gs ;
                     gx.limitChars( indx - 1 ) ;
                     this->gsRevChars ( gx ) ;
                     gs.shiftChars( -(indx) ) ;
                     gs.insert( gx.gstr() ) ;
                  }
                  gs.replace( dSPC, L"--", indx, false, true ) ;
               }

               gs.limitCols( stdFIELD - 2 ) ;
               ofs << dSPC << gs.ustr() << " " ;
               gs = this->tData.sf[fIndex].sfTag.field[fldIndex] ;
               if ( fldRTL )
                  this->gsRevChars ( gs ) ;
               ofs << gs.ustr() << '\n' ;
               ++fcnt ;
            }
         }

         #if SUMMARIZE_IMAGES != 0
         gString gsize, gdesc ;
         EmbeddedImage* ePtr = &this->tData.sf[fIndex].sfTag.ePic ;
         if ( ePtr->inUse )
         {
            while ( ePtr != NULL && ePtr->inUse )
            {
               gsize.formatInt( ePtr->mp3img.picSize, 13, true ) ;
               gdesc = pType[lang][ePtr->mp3img.picType] ;
               if ( fldRTL )
                  this->gsRevChars ( gdesc ) ;
               gs.compose( ImgTemplate[lang], gsize.gstr(), 
                           ePtr->mp3img.mimType, gdesc.gstr() ) ;
               ofs << gs.ustr() ;
               gs = ePtr->mp3img.txtDesc ;
               if ( fldRTL )
                  this->gsRevChars ( gs ) ;
               ofs << gs.ustr() << endl ;
               ePtr = ePtr->next ;
               ++fcnt ;
            }
         }
         #endif   // SUMMARIZE_IMAGES

         #if SUMMARIZE_POPCNT != 0
         if ( (this->tData.sf[fIndex].sfTag.pop.popdata != false) ||
              (this->tData.sf[fIndex].sfTag.pop.cntdata != false) )
         {
            gs.compose( PopTemplate[lang],
                        &this->tData.sf[fIndex].sfTag.pop.popStar,
                        &this->tData.sf[fIndex].sfTag.pop.popCount,
                        this->tData.sf[fIndex].sfTag.pop.popEmail,
                        &this->tData.sf[fIndex].sfTag.pop.playCount ) ;
            ofs << gs.ustr() << endl ;
         }
         #endif   // SUMMARIZE_POPCNT

         //* If file contains no metadata, so indicate *
         if ( fcnt == ZERO )
            ofs << NoData[lang] ;
         ofs << endl ;

         //* If summary only for file with focus, we're done *
         if ( ! allFiles )
            break ;
      }
      if ( this->tData.sfCount == ZERO )  // if no source files
         ofs << NoFiles[lang] << endl ;

      ofs.close() ;     // close the temp file
      status = true ;   // file written successfuly
   }
   return status ;

}  //* End ctsSummarize() *

//*************************
//*     ctsVerifySave     *
//*************************
//******************************************************************************
//* Alert user that Tag Summary data were successfully written to a file.      *
//*                                                                            *
//* Input  : trgPath  : filespec for saved file                                *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void Taggit::ctsVerifySave ( const gString& trgPath )
{
   const char* Labels[] = 
   {
      "Summary data saved to:",        // English
      "Resumen de datos guardados en:",// Espanol
      "摘要数据已保存到 :",               // Zhongwen
      "Tóm tắt dữ liệu được lưu vào:", // TiengViet
   } ;

   short fnIndex = ((trgPath.findlast( L'/' )) + 1) ;
   gString gs( "      %S", &trgPath.gstr()[fnIndex] ) ;

   const char* dlgText[] = 
   {
      "",
      "",
      Labels[this->cfgOpt.appLanguage],
      gs.ustr(),
      "",
      NULL
   } ;
   attr_t dColor = this->cs.sd,
          hColor = this->cs.em ;
   attr_t dlgAttr[] = 
   { dColor, dColor, dColor, hColor, dColor, dColor } ;

   genDialog gd( dlgText, dColor, 8, 48, -1, -1, dlgAttr, this->cfgOpt.rtl, 
                 attrDFLT, attrDFLT, okText[this->cfgOpt.appLanguage] ) ;

   this->dPtr->InfoDialog ( gd ) ;     // alert user

}  //* End ctsVerifySave()

//*************************
//*  Cmd_RefreshMetadata  *
//*************************
//******************************************************************************
//* User has requested a global refresh of all metadata.                       *
//*                                                                            *
//*  1) Ask user to verify the operation:                                      *
//*     a) If edits are pending, ask whether to save them before refresh.      *
//*        If no edits pending, then skip this step.                           *
//*     b) Save all pending edits, including filename updates if they were     *
//*        previously edited. Else discard all pending filename changes AND    *
//*        all metadata.                                                       *
//*     c) Re-read the metadata for all source files and update the display.   *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************
//* Notes:                                                                     *
//* 1) The primary purpose of this operation is to allow user to verify that   *
//*    the edits performed during the session have been correctly written to   *
//*    the target files.                                                       *
//* 2) This method is called after the 'Save All' command has been executed.   *
//* 3) User may also select this option manually from the 'File' menu 'Refresh'*
//*    if he/she/it gets confused about what edits have been made.             *
//******************************************************************************

void Taggit::Cmd_RefreshMetadata ( void )
{
   gString gspc ;                   // message formatting
   AppLang lang = this->cfgOpt.appLanguage ; // user-interface language
   short pendCount = ZERO ;         // number of files with pending edits
   short fails = ZERO ;             // count of failed restores

   #if SIMULATED_DATA != 0    // DEVELOPMENT ONLY
   this->InitSimulatedData () ;
   this->UpdateDataWindows () ;
   return ;
   #endif   // SIMULATED_DATA - DEVELOPMENT ONLY

   const short dDlgLINES = 8 ;
   const char* dDlgText[][dDlgLINES + 3] = 
   {
      {  // English
         "  Refresh All Metadata  ",
         " ",
         gspc.ustr(),
         " ",
         "Do you want to discard these edits and re-read",
         "all the source files?",
         " ",
         NULL,
         "  ^YES  ",
         "  ^NO  ",
         "There are %hd files with edits pending."
      },
      {  // Espanol
         "  Refrescar Todos los Metadatos  ",
         " ",
         gspc.ustr(),
         " ",
         "¿Desea descartar estas ediciones y volver a leer",
         " todos los archivos de origen?",
         " ",
         NULL,
         "  ^SI  ",
         "  ^NO  ",
         "Hay %hd archivos con ediciones pendientes."
      },
      {  // Zhongwen
         "    刷新所有元数据    ",
         " ",
         gspc.ustr(),
         " ",
         " ",
         "     您要舍弃这些编辑并重新读取所有源文件吗？",
         " ",
         NULL,
         "   是   ",
         "  没有  ",
         "有 %hd 个文件，编辑等待。"
      },
      {  // Tieng Viet
         "  Làm Mới Tất Cả Siêu Dữ liệu  ",
         " ",
         gspc.ustr(),
         " ",
         "Bạn có muốn loại bỏ các chỉnh sửa này và đọc lại",
         "tất cả các tệp nguồn không?",
         " ",
         NULL,
         "  ^Vâng  ",
         "  ^Không  ",
         "Có %hd tệp với các chỉnh sửa đang chờ xử lý."
      },
   } ;
   const attr_t dDlgAttr[dDlgLINES] = 
   {
      this->cs.em, this->cs.sd, this->cs.sd, this->cs.sd, this->cs.sd, 
      this->cs.sd, this->cs.sd, this->cs.sd
   } ;

   //* Count the files which have pending edits.*
   pendCount = this->EditsPending () ;
   gspc.compose( dDlgText[lang][dDlgLINES + 2], &pendCount ) ;

   if ( pendCount > ZERO )
   {
      //* Ask user if it's OK to discard pending edits *
      genDialog gdlg( (const char**)&dDlgText[lang], 
                      this->cs.sd, (dDlgLINES + 2), 54, -1, -1, dDlgAttr,
                      this->cfgOpt.rtl, this->cs.pn, this->cs.pf, 
                      dDlgText[lang][dDlgLINES], dDlgText[lang][dDlgLINES + 1] ) ;
      bool userOk = this->dPtr->DecisionDialog ( gdlg ) ;

      //* If we have the user's permission *
      if ( userOk )
      {
         for ( short fIndex = ZERO ; fIndex < this->tData.sfCount ; ++fIndex )
         {
            if ( (this->crmRefreshMetadata ( fIndex, false )) == false )
               ++fails ;            // error during restore
         }
   
         //* Move to top of list and redisplay the data *
         this->tData.sfFirst = this->tData.sfFocus = ZERO ;
         this->InitIndices () ;
         this->UpdateDataWindows () ;
      }
   }

   //* Else no edits are pending, so just warn user.*
   else
   {
      const char* xDlgText[] = 
      {
         dDlgText[lang][0],
         " ",
         gspc.ustr(),
         " ",
         NULL,
      } ;
      genDialog gdlg( (const char**)&xDlgText, 
                      this->cs.sd, (dDlgLINES), 54, -1, -1, dDlgAttr,
                      this->cfgOpt.rtl, this->cs.pn, this->cs.pf, okText[lang] ) ;
      this->dPtr->InfoDialog ( gdlg ) ;
   }

}  //* End Cmd_RefreshMetadata() *

//*************************
//*  crmRefreshMetadata   *
//*************************
//******************************************************************************
//* Re-read the metadata for the specified source file and optionally update   *
//* the display.                                                               *
//*                                                                            *
//* Input  : fIndex  : index of file whose metadata are to be restored         *
//*          refresh : (optional, 'true' by default)                           *
//*                    if 'true' redraw the display data                       *
//*                    if 'false', do not update the display                   *
//*                                                                            *
//* Returns: 'true' if restore successful, else 'false'                        *
//******************************************************************************
//* Notes:                                                                     *
//* 1) This method is called after user saves the data for the specified file. *
//* 2) User may also select this option manually from the 'Edit' menu via the  *
//*    'Undo' command. This allows the user to escape if he/she/it gets        *
//*    confused about what edits have been made.                               *
//* 3) Discard all pending edits:                                              *
//*    a) If the filename has been edited (and not yet written), discard the   *
//*       request for filename change.                                         *
//*    b) Discard any pending edits for the metadata fields (including any     *
//*       pending image insertion or popularimiter/counter updates).           *
//* 4) Re-read the source file's metadata and store it in the file's display   *
//*    object.                                                                 *
//*    a) Note that restoring the original metadata presumes that user has not *
//*       already written changes to the specified source file, because in     *
//*       that case the data restored would be of the most recent save.        *
//* 5) Therefore neither filename nor field content edits are pending, so we   *
//*    reset those flags ('sfMod' and 'tfMod').                                *
//*                                                                            *
//*                                                                            *
//******************************************************************************

bool Taggit::crmRefreshMetadata ( short fIndex, bool refresh )
{
   gString gs ;                  // text analysis
   short fndx ;                  // filename index
   bool status = false ;         // return value

   //* If edit pending on filename, restore original filename *
   if ( this->tData.sf[fIndex].sfMod )
   {
      gs = this->tData.sf[fIndex].sfPath ;
      fndx = (gs.findlast( L'/' )) + 1 ;
      gs.shiftChars( -(fndx) ) ;
      gs.copy( this->tData.sf[fIndex].sfName, MAX_FNAME ) ;
      this->tData.sf[fIndex].sfMod = false ; // reset the edit-pending flag
   }

   //* If edits pending for metadata fields, re-read the source file *
   if ( this->tData.sf[fIndex].sfTag.tfMod != false )
   {
      this->tData.sf[fIndex].sfTag.tfMod = false ;
      status = this->LoadMetadata ( fIndex ) ;
   }

   if ( refresh )
      this->UpdateDataWindows () ;

   return status ;

}  //* End crmRefreshMetadata() *

//*************************
//*   Cmd_ClearMetadata   *
//*************************
//******************************************************************************
//* Clear (erase) display data from all metadata fields for all files, and     *
//* update the display.                                                        *
//*                                                                            *
//* Input  : fIndex  : index of file whose metadata are to be restored         *
//*          refresh : (optional, 'true' by default)                           *
//*                    if 'true' redraw the display data                       *
//*                    if 'false', do not update the display                   *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void Taggit::Cmd_ClearMetadata ( void )
{
   short termCount = this->tData.sfCount - 1 ;

   for ( short fIndex = ZERO ; fIndex < this->tData.sfCount ; ++ fIndex )
      this->ccmClearMetadata ( fIndex, bool(fIndex == termCount ? true : false) ) ;

   //* Move to top of list *
   this->tData.sfFirst = this->tData.sfFocus = ZERO ;
   this->InitIndices () ;
   this->UpdateDataWindows () ;

}  //* End Cmd_ClearMetadata() *

//*************************
//*   ccmClearMetadata    *
//*************************
//******************************************************************************
//* Clear (erase) display data from all metadata fields for the specified      *
//* source file, including embedded picture(s) (if any).                       *
//* Optionally update the display and report success.                          *
//*                                                                            *
//* Input  : fIndex  : index of file whose metadata are to be restored         *
//*          refresh : (optional, 'true' by default)                           *
//*                    if 'true' redraw the display data                       *
//*                    if 'false', do not update the display                   *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void Taggit::ccmClearMetadata ( short fIndex, bool refresh )
{
   for ( short fldIndex = ZERO ; fldIndex < tfCOUNT ; ++fldIndex )
   {
      //* Test for existing metadata in the fields.           *
      //* Edits MAY already be pending, but if we erase data, *
      //* there WILL BE edit(s) pending.                      *
      // Programmer's Note: This logic is not foolproof: If a field was originally
      // empty, then the user wrote data to it, then we erased it, then technically
      // an edit is no longer pending for that field. Testing for this possibility
      // is more trouble than it's worth.
      if ( *this->tData.sf[fIndex].sfTag.field[fldIndex] != NULLCHAR )
      {
         this->tData.sf[fIndex].sfTag.tfMod = true ;
         break ;
      }
   }
   this->tData.sf[fIndex].sfTag.clear_data() ;  // clear all fields

   if ( refresh )
   {
      this->UpdateDataWindows () ;        // update the display

      //* Tell user what has just happened *
      const char* dlgText[][6] = 
      {
         {  // English
            "  Discard Metadata  ",
            "",
            "All metadata fields have been cleared.",
            "To retrieve data, select \"Refresh\" from the FILE menu.",
            "",
            NULL
         },
         {  // Espanol
            "  Descartar los Metadatos  ",
            "",
            "Todos los campos de metadatos se han borrado.",
            "Para recuperar los datos, seleccione \"Refrescar\"",
            " en el menú Archivo.",
            NULL
         },
         {  // Zhongwen
            "  舍弃元数据  ",
            "",
            "所有元数据字段已被清除。",
            "要检索数据，请从文件菜单中选择\"刷新\"。",
            "",
            NULL
         },
         {  // Tieng Viet
            "  Hủy siêu dữ liệu  ",
            "",
            "Tất cả các trường siêu dữ liệu đã bị xóa.",
            "Để lấy dữ liệu, chọn \"Đọc Lại\" từ trình đơn tập tin.",
            "",
            NULL
         },
      } ;
      const short dlgROWS  =  8,             // dialog lines
                  dlgCOLS  = 57,             // dialog columns
                  dlgY     = fieldwinY + 1,  // dialog position
                  dlgX     = fieldwinX + 1 ;
      attr_t dColor = this->cs.sd,           // dialog interior color
             hColor = this->cs.em ;          // bold dialog text
      AppLang lang = this->cfgOpt.appLanguage ; // language index
      attr_t dlgAttr[] = 
      { hColor, dColor, dColor, dColor, dColor } ;

      genDialog gd( dlgText[lang], dColor, dlgROWS, dlgCOLS, 
                    dlgY, dlgX, dlgAttr, this->cfgOpt.rtl, 
                    attrDFLT, attrDFLT, okText[lang]) ;

      this->dPtr->InfoDialog ( gd ) ;     // alert user
   }

}  //* End ccmClearMetadata() *

//*************************
//*  Cmd_DuplicateField   *
//*************************
//******************************************************************************
//* User has selected the Edit/Duplicate Field menu item.                      *
//* 1) Open a dialog and allow user to select the field to be duplicated.      *
//* 2) Optionally, allow edit of the data before writing to ALL files.         *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************
//* Notes:                                                                     *
//*                                                                            *
//* 1) We allow the user to duplicate any active field; however, duplication   *
//*    of some fields makes no sense. For instance, the 'Title' should almost  *
//*    always be unique, as should the 'Track' field and others.               *
//*    This means that we must asume some intelligence on the part of the user,*
//*    although this is always a dangerous assumption.                         *
//* 2) We DO NOT allow user access to inactive fields, although theoretically, *
//*    we could do so. Depending on the save option, the inactive fields which *
//*    contain data would be saved during the write to target. The problem is  *
//*    that duplicating an inactive field would have no visible effect, so     *
//*    user would get no feedback for his/her/its action and that is a major   *
//*    no-no in user interface design.                                         *
//*                                                                            *
//******************************************************************************

void Taggit::Cmd_DuplicateField ( void )
{
   const short dlgROWS = 24,              // dialog size
               dlgCOLS = 90,
               dlgY    = fieldwinY + 1,   // dialog position
               dlgX    = fieldwinX + 1,
               editHEIGHT = 5,            // Textbox dimensions
               editWIDTH  = (stdFIELD / 2 * 3),
               nameHEIGHT = (dlgROWS - 2),// Scrollext dimensions
               nameWIDTH  = (stdFIELD + 2),
               nameBYTES  = (stdFIELD * 4) ; // second dimension of sextText[][] array
   attr_t dColor = this->cs.sd,           // dialog interior color
          hColor = this->cs.em,           // bold dialog text
          xColor = this->cs.tf,           // Scrollext item colors
          iColor = this->cs.pn ;
   AppLang lang = this->cfgOpt.appLanguage ; // language index
   const char* const* umwHPtr = umwHeaders[lang] ; // tag-field headings
   gString gs ;                           // text formatting
   bool redraw = false,                   // 'true' if field(s) updated
        fldRTL = false ;                  // toggle for RTL display in editTB control

   enum ctspControls : short 
   { nameSE = ZERO, editTB, dupePB, cancelPB, toggPB, ctCTRLS } ;

   //* Control labels *
   const char* Labels[][ctCTRLS + 1] = 
   {
      {  // English
         "  FIELD NAME  ",                // 'field name' Scrollext
         "Source Text",                   // 'edit' Textbox
         "    DUPLICATE    ",             // 'duplicate' Pushbutton
         "     CANCEL      ",             // 'cancel' Pushbutton
         "  Duplicate Field Data  ",      // Dialog title
      },
      {  // Espanol
         "  NOMBRE DEL CAMPO  ",          // 'field name' Scrollext
         "Texto de Origen",                   // 'edit' Textbox
         "     DUPLICAR    ",             // 'duplicate' pushbutton
         "     CANCELAR    ",             // 'cancel' pushbutton
         "  Datos de Campo Duplicados  ", // Dialog title
      },
      {  // Zhongwen
         "   领域名称   ",                  // 'field name' Scrollext
         "源文本",                          // 'edit' Textbox
         "      复制       ",              // 'duplicate' pushbutton
         "      取消       ",              // 'cancel' pushbutton
         "   重复文本数据   ",               // Dialog title
      },
      {  // TiengViet
         "  Tên trường dữ liệu  ",        // 'field name' Scrollext
         "Văn bản nguồn",                 // 'edit' Textbox
         "  Chụp hai bản   ",               // 'duplicate' pushbutton
         "      Hủy Bỏ     ",             // 'cancel' pushbutton
         "  Trường lặp lại cho tất cả các tệp  ", // Dialog title
      },
   } ;
   const char* Msg[][3] = 
   {  
      {
         // English ---------------------------------------|
         "Select the field to be duplicated.\n"
         "The contents of the corresponding field in the\n"
         "highlighted file will be displayed in the textbox.\n"
         "The displayed data may be edited.\n"
         "Press the \"Duplicate\" button to write the text\n"
         "into the target field of all source files.\n",
         "\n(Note: Inactive fields may not be duplicated.)",
         "Sorry, the selected field is not active.",
      },
      {
         // Espanol ---------------------------------------|
         "Seleccione el campo que desea duplicar.\n"
         "El contenido del campo correspondiente en el\n"
         "archivo resaltado se mostrará en el cuadro de\n"
         "texto.\n"
         "Los datos visualizados pueden ser editados.\n"
         "Presiona el botón: \"Duplicar\" Para escribir\n"
         "el texto en el campo de destino de todos los\n"
         "archivos de origen.\n",
         "(Nota: Los campos inactivos no se pueden duplicar)",
         "Lo sentimos, el campo seleccionado no está activo.",
      },
      {
         // Zhongwen --------------------------------------|
         "选择要复制的字段。\n"
         "突出显示的文件中相应字段的内容将显示在文本框中。\n"
         "可以编辑显示的数据。\n"
         "按下按钮：\"复制\" 将文本写入所有源文件的目标字段。\n",
         "\n(注意：非活动字段可能不会复制。)",
         "对不起，所选字段未激活。",
      },
      {
         // TiengViet -------------------------------------|
         "Chọn trường cần sao chép.\n"
         "Nội dung của trường tương ứng trong tệp được đánh\n"
         "dấu sẽ được hiển thị trong hộp văn bản.\n"
         "Dữ liệu được hiển thị có thể được chỉnh sửa.\n"
         "Nhấn nút \"Chụp hai bản\" để viết văn bản vào trường\n"
         "mục tiêu của tất cả các tệp nguồn.\n",
         "\n(Lưu ý:Không thể trùng lặp trường không hoạt động)",
         "Xin lỗi, trường đã chọn không hoạt động.",
      },
   } ;


   //* Control definitions *
   InitCtrl ic[ctCTRLS] =              // array of dialog control info
   {
   { //* 'FieldName' Scrollext - - - - - - - - - - - - - - - - - - - -  nameSE *
      dctSCROLLEXT,                 // type:      define a scrolling-data control
      rbtTYPES,                     // rbSubtype: (na)
      false,                        // rbSelect:  (n/a)
      1,                            // ulY:       upper left corner in Y
      3,                            // ulX:       upper left corner in X
      nameHEIGHT,                   // lines:     control lines
      nameWIDTH,                    // cols:      control columns
      NULL,                         // dispText:  (n/a - set below)
      this->cs.em,                  // nColor:    non-focus border color
      this->cs.pf,                  // fColor:    focus border color
      tbPrint,                      // filter:    (n/a)
      Labels[lang][nameSE],         // label:     
      ZERO,                         // labY:      offset from control's ulY
      ZERO,                         // labX       offset from control's ulX
      ddBoxTYPES,                   // exType:    (n/a)
      tfCOUNT,                      // scrItems:  (n/a - set below)
      ZERO,                         // scrSel:    (n/a - set below)
      NULL,                         // scrColor:  (n/a - set below)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[editTB]                   // nextCtrl:  link in next structure
   },
   {  //* 'Edit' Textbox - - - - - - - - - - - - - - - - - - - - - - -  editTB *
      dctTEXTBOX,                   // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      2,                            // ulY:       upper left corner in Y
      short(dlgCOLS - editWIDTH - 3), // ulX:       upper left corner in X
      editHEIGHT,                   // lines:     (n/a)
      editWIDTH,                    // cols:      control columns
      NULL,                         // dispText:  
      this->cs.tn,                  // nColor:    non-focus color
      this->cs.tf,                  // fColor:    focus color
      tbPrint,                      // filter:    all printing characters
      Labels[lang][editTB],         // label:     
      -1,                           // labY:      
      short(this->cfgOpt.rtl ? (editWIDTH - 1) : ZERO), // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[dupePB]                   // nextCtrl:  link in next structure
   },
   {  //* 'Duplicate' pushbutton  - - - - - - - - - - - - - - - - - -   dupePB *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(dlgROWS / 2),           // ulY:       upper left corner in Y
      short(dlgCOLS / 2),           // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      17,                           // cols:      control columns
      Labels[lang][dupePB],         // dispText:  
      this->cs.pn,                  // nColor:    non-focus color
      this->cs.pf,                  // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "",                           // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[cancelPB],                // nextCtrl:  link in next structure
   },
   {  //* 'Cancel' pushbutton - - - - - - - - - - - - - - - - - - -   cancelPB *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      ic[dupePB].ulY,               // ulY:       upper left corner in Y
      short(ic[dupePB].ulX + ic[dupePB].cols + 2), // ulX: upper left corner in X
      1,                            // lines:     (n/a)
      17,                           // cols:      control columns
      Labels[lang][cancelPB],       // dispText:  
      this->cs.pn,                  // nColor:    non-focus color
      this->cs.pf,                  // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "",                           // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      NULL                          // nextCtrl:  link in next structure
   },
   {  //* 'Toggle' pushbutton - - - - - - - - - - - - - - - - - - - -   toggPB *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      ic[dupePB].ulY,               // ulY:       upper left corner in Y
      short(ic[nameSE].ulX + ic[nameSE].cols + 1), // ulX: upper left corner in X
      1,                            // lines:     (n/a)
      5,                            // cols:      control columns
      " LT^R ",                     // dispText:  
      this->cs.pn,                  // nColor:    non-focus color
      this->cs.pf,                  // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "",                           // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      NULL                          // nextCtrl:  link in next structure
   },
   } ;

   //* Construct the display data for the Scrollext control *
   char sextText[tfCOUNT][nameBYTES] ; // raw text for list
   const char* sextData[tfCOUNT] =     // pointers to text for list
   {
      sextText[ 0], sextText[ 1], sextText[ 2], sextText[ 3], sextText[ 4], 
      sextText[ 5], sextText[ 6], sextText[ 7], sextText[ 8], sextText[ 9], 
      sextText[10], sextText[11], sextText[12], sextText[13], sextText[14], 
      sextText[15], sextText[16], sextText[17], sextText[18], sextText[19], 
      sextText[20], sextText[21], sextText[22], sextText[23], sextText[24], 
      sextText[25], sextText[26], sextText[27], sextText[28], sextText[29], 
      sextText[30], sextText[31], sextText[32], sextText[33], sextText[34], 
      sextText[35], sextText[36], sextText[37], sextText[38], 
   } ;
   attr_t sextAttr[tfCOUNT] ;          // color attributes for list

   //* Set the display text and mark inactive fields *
   for ( short i = ZERO ; i < tfCOUNT ; ++i )
   {
      //* Format the display text *
      gs = umwHPtr[i] ;
      gs.append( baseField ) ;
      gs.limitCols( stdFIELD ) ;
      if ( this->tData.sffDisp[i] != false ) // if field is active (visible)
         sextAttr[i] = xColor ;
      else                                   // if field is inactive
         sextAttr[i] = iColor ;
      gs.copy( sextText[i], nameBYTES ) ;
   }

   //* Dialog definition *
   InitNcDialog dInit( dlgROWS,         // number of display lines
                       dlgCOLS,         // number of display columns
                       dlgY,            // Y offset from upper-left of terminal 
                       dlgX,            // X offset from upper-left of terminal 
                       NULL,            // dialog title
                       ncltDUAL,        // border line-style
                       dColor,          // border color attribute
                       dColor,          // interior color attribute
                       ic               // pointer to list of control definitions
                     ) ;

   this->dPtr->SetDialogObscured () ;  // save parent dialog

   //* If user interface is an RTL language, enable the optional *
   //* Pushbutton for toggling display of the descriptions.      *
   if ( this->cfgOpt.rtl )
      ic[cancelPB].nextCtrl = &ic[toggPB] ;

   //* Open the dialog *
   NcDialog* dp = new NcDialog ( dInit ) ;
   cdfDialog = dp ;           // give callback method access to the dialog
   cdfTData  = &this->tData ; // give callback method access to our member data
   cdfLang  = this->cfgOpt.appLanguage ; // give callback method our UI language
   // Programmer's Note: The above pointers give a non-member method access to 
   // our private data. This is incredibly dangerous, but the callback method 
   // promises to behave itself.

   //* Define a list of keycodes reserved for *
   //* access to clipboard functionality and  *
   //* enable invalid-text-input alert.       *
   this->uiDefineReservedKeys ( dp ) ;
   dp->TextboxAlert ( editTB, true ) ;

   //* If user interface language is an RTL language, *
   //* set the internal NcDialog flags for RTL output.*
   if ( this->cfgOpt.rtl )
   {
      dp->DrawLabelsAsRTL () ;
      dp->DrawContentsAsRTL ( nameSE ) ;
   }

   if ( (dp->OpenWindow()) == OK )
   {
      //* Set the dialog title *
      dp->SetDialogTitle ( Labels[lang][ctCTRLS], hColor ) ;

      //* Write the dialog's static data *
      winPos msgPos( (ic[dupePB].ulY + 2), 
                     this->cfgOpt.rtl ? (ic[editTB].ulX + ic[editTB].cols) : ic[editTB].ulX ) ;
      msgPos = dp->WriteParagraph ( msgPos, Msg[lang][0], dColor, false, this->cfgOpt.rtl ) ;
      dp->WriteParagraph ( msgPos, Msg[lang][1], this->cs.dm, false, this->cfgOpt.rtl ) ;
      msgPos = { short(ic[editTB].ulY + 6), ic[editTB].ulX } ;

      //* Initialize the Scrollext control's display data *
      ssetData sData( sextData, sextAttr, tfCOUNT, this->tData.sffOffset, true ) ;
      dp->SetScrollextText ( nameSE, sData ) ;

      dp->RefreshWin () ;                 // make the text visible

      //* Establish a call-back method for updating Textbox data.*
      dp->EstablishCallback ( &cdfCallback ) ;

      //* Interact with the user *
      uiInfo   Info ;                     // user interface data returned here
      short    icIndex = dupePB ;         // index of control with focus
      bool     done = false ;             // loop control
      while ( ! done )
      {
         if ( ic[icIndex].type == dctPUSHBUTTON )
         {
            if ( Info.viaHotkey )
               Info.HotData2Primary () ;
            else
               icIndex = dp->EditPushbutton ( Info ) ;
            if ( Info.dataMod != false )
            {
               if ( Info.ctrlIndex == dupePB )
               {
                  dp->GetTextboxText ( editTB, gs ) ;
                  short fldIndex = dp->GetScrollextSelect ( nameSE ) ;
                  this->cdfDuplicateField ( fldIndex, gs, false ) ;
                  redraw = true ;
                  done = true ;
               }
               else if ( Info.ctrlIndex == cancelPB )
               { done = true ; }
               else if ( Info.ctrlIndex == toggPB )
               {  //* Note: We will arrive here ONLY if UI language is RTL. *
                  fldRTL = fldRTL ? false : true ;
                  dp->DrawContentsAsRTL ( editTB, fldRTL ) ;
                  dp->RefreshWin () ;
               }
            }
         }

         if ( ic[icIndex].type == dctSCROLLEXT )
         {
            Info.viaHotkey = false ;      // ignore hotkey data
            icIndex = dp->EditScrollext ( Info ) ;

            //* Because we cannot disable individual items in a Scrollext *
            //* control, we must disallow selection of inactive items in  *
            //* another way. Our solution is to trap the user inside the  *
            //* Scrollext control until an active item is selected.       *
            //* OR until user hits the panic button (ESC), in which case  *
            //* the highlight is moved back to the top of the list.       *
            dp->ClearLine ( msgPos.ypos, false, msgPos.xpos ) ;
            if ( this->tData.sffDisp[Info.selMember] == false )
            {
               if ( Info.keyIn == nckESC )
               {
                  dp->MoveScrollextHighlight ( short(nameSE), ZERO, ZERO ) ;
               }
               else
               {
                  dp->WriteString ( msgPos, Msg[lang][2], hColor, true ) ;
                  dp->UserAlert () ;
                  Info.keyIn = nckTAB ;
                  dp->PrevControl () ;
               }
            }
         }

         if ( ic[icIndex].type == dctTEXTBOX )
         {
            Info.viaHotkey = false ;      // ignore hotkey data
            icIndex = dp->EditTextbox ( Info ) ;
         }

         //* Move focus to appropriate control *
         if ( ! done && ! Info.viaHotkey )
         {
            if ( Info.keyIn == nckSTAB )
               icIndex = dp->PrevControl () ; 
            else
               icIndex = dp->NextControl () ;
         }
      }  // while()
   }
   if ( dp != NULL )                         // close the window
   {
      delete ( dp ) ;
      cdfDialog = NULL ;
      cdfTData = NULL ;
      cdfLang = enLang ;
   }

   this->dPtr->RefreshWin () ;               // restore parent dialog
   if ( redraw )                             // if parent dialog's data have changed
      this->UpdateDataWindows () ;

}  //* End Cmd_DuplicateField() *

//*************************
//*      cdfCallback      *
//*************************
//******************************************************************************
//* This is a callback method for manually updating the controls in the        *
//* Cmd_DuplicateField() dialog.                                               *
//*                                                                            *
//*  The following are updated by this callback method:                        *
//*   a) If the Textbox does not have focus, AND                               *
//*   b) if the Scrollext index has changed,                                   *
//*   c) extract the field contents from the target field of the highlighted   *
//*      file                                                                  *
//*   d) Set the contents of the Textbox control.                              *
//*                                                                            *
//* Input  : currIndex: index of control that currently has focus              *
//*          wkey     : user's key input data                                  *
//*          firstTime: the EstablishCallback() method calls this method once  *
//*                     with firstTime==true, to perform any required          *
//*                     initialization. Subsequently, the NcDialog class       *
//*                     always calls with firstTime==false.                    *
//* Returns: OK                                                                *
//******************************************************************************
//* Important Note: This method makes some intimate assumptions about what     *
//* is happening in the method that established the callback. If changes are   *
//* made in the establishing method that affect the callback functionality,    *
//* be sure to update this method to address those changes.                    *
//*                                                                            *
//* Programmer's Note: See the static dialog pointer, 'cdfDialog'.             *
//******************************************************************************

static short cdfCallback ( const short currIndex, const wkeyCode wkey, bool firstTime )
{
   const short  nameSE = 0,         // Index of dialog's Scrollext control
                editTB = 1 ;        // Index of dialog's Textbox control
   static short oldSelect = ZERO ;  // Scrollext selection on previous call
   short        newSelect = 1 ;     // Scrollext selection on this call

   //* Be safe... *
   if ( (cdfDialog != NULL) && (cdfTData != NULL) )
   {
      //*************************
      //* First-time processing *
      //*************************
      if ( firstTime != false )
      {
         /* Nothing to do. */
      }

      //***********************
      //* Standard processing *
      //***********************
      if ( currIndex != editTB )
      {
         newSelect = cdfDialog->GetScrollextSelect ( nameSE ) ;
         if ( newSelect != oldSelect )
         {
            gString gs ;
            //* We do not display the data (if any) for inactive fields.    *
            if ( cdfTData->sffDisp[newSelect] != false ) // if field is active (visible)
               gs = cdfTData->sf[cdfTData->sfFocus].sfTag.field[newSelect] ;
            cdfDialog->SetTextboxText ( editTB, gs ) ;
            oldSelect = newSelect ;
         }
      }
   }
   return OK ;
   
}  //* End cdfCallback() *

//*************************
//*   cdfDuplicateField   *
//*************************
//******************************************************************************
//* Duplicate specified field's contents to all source files.                  *
//* 1) Update the target edit field with the specified text for all files.     *
//*    NOTE: If the file is currently under edit, the caller must update the   *
//*          temporary buffer for the field to match the live-data value.      *
//* 2) If the target field contents is changed, set the edits-pending flag for *
//*    the file.                                                               *
//* 3) If specified, redraw the display to show new field contents.            *
//*                                                                            *
//* Input  : fldIndex  : index of field to be duplicated                       *
//*          newValue  : new field contents                                    *
//*          redraw    : (optional, 'true' by default)                         *
//*                      if 'true', redraw the display                         *
//*                      if 'false', display is not updated                    *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void Taggit::cdfDuplicateField ( short fldIndex, const gString& newValue, bool redraw )
{
   for ( short fIndex = ZERO ; fIndex < this->tData.sfCount ; ++fIndex )
   {
      if ( (newValue.compare( this->tData.sf[fIndex].sfTag.field[fldIndex] )) != ZERO )
      {
         newValue.copy( this->tData.sf[fIndex].sfTag.field[fldIndex], gsMAXCHARS ) ;
         this->tData.sf[fIndex].sfTag.tfMod = true ;
      }
   }
   if ( redraw )                      // redraw the display data
      this->UpdateDataWindows () ;

}  //* End cdfDuplicateField() *

//*************************
//*   Cmd_SetSongTitle    *
//*************************
//******************************************************************************
//* For all files, set the 'Title' field based on the filename.                *
//* Query the user before proceeding.                                          *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void Taggit::Cmd_SetSongTitle ( void )
{
   const short dDlgLINES = 9 ;
   const char* dDlgText[][dDlgLINES + 2] = 
   {
      {  // English
         "  Set Song Title  ",
         " ",
         "Do you want to set the \"Title\" field for all files",
         "based on the name of the file?",
         " ",
         "Example Filename: My Heart Will Go On.mp3",
         "   Title becomes: My Heart Will Go On",
         " ",
         NULL,
         "  ^YES  ",
         "  ^NO  "
      },
      {  // Espanol
         "  Establecer el Título de la Canción  ",
         " ",
         "¿Desea establecer el campo \"Título\" para todos",
         "los archivos basado en el nombre del archivo?",
         " ",
         "Ejemplo de nombre de archivo: My Heart Will Go On.mp3",
         "   El título se convierte en: My Heart Will Go On",
         " ",
         NULL,
         "  ^SI  ",
         "  ^NO  "
      },
      {  // Zhongwen
         "   设置歌曲标题   ",
         " ",
         " ",
         "是否要根据文件名设置所有文件的 “标题” 字段？",
         " ",
         "示例文件名： My Heart Will Go On.mp3",
         "  标题变成： My Heart Will Go On",
         " ",
         NULL,
         "   是   ",
         "  没有  "
      },
      {  // Tieng Viet
         "  Đặt Tiêu đề Bài Hát  ",
         " ",
         "Bạn có muốn đặt trường \"Tiêu đề\" cho tất cả các tệp",
         "tin dựa trên tên tệp?",
         " ",
         "Ví dụ Tên tệp    : My Heart Will Go On.mp3",
         "Tiêu đề trở thành: My Heart Will Go On",
         " ",
         NULL,
         "  ^Vâng  ",
         "  ^Không  "
      },
   } ;
   const attr_t dDlgAttr[dDlgLINES] = 
   {
      this->cs.em, this->cs.sd, this->cs.sd, this->cs.sd, this->cs.sd, 
      this->cs.sd, this->cs.sd, this->cs.sd, this->cs.sd
   } ;
   AppLang lang = this->cfgOpt.appLanguage ;

   // Query user *
   genDialog gdlg( (const char**)&dDlgText[lang], 
                   this->cs.sd, 11, 57, -1, -1, dDlgAttr,
                   this->cfgOpt.rtl, this->cs.pn, this->cs.pf, 
                   dDlgText[lang][dDlgLINES], dDlgText[lang][dDlgLINES + 1] ) ;
   bool userOk = this->dPtr->DecisionDialog ( gdlg ) ;

   if ( userOk )
   {
      for ( short fIndex = ZERO ; fIndex < this->tData.sfCount ; ++fIndex )
      {
         this->csstSetSongTitle ( fIndex ) ;
         this->tData.sf[fIndex].sfTag.tfMod = true ; // set the edits-pending flag
      }
   }
   this->UpdateDataWindows () ;

}  //* End Cmd_SetSongTitle() *

//*************************
//*   csstSetSongTitle    *
//*************************
//******************************************************************************
//* Set the 'Title' field based on the filename.                               *
//*                                                                            *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void Taggit::csstSetSongTitle ( short fIndex )
{
  //* Get the filename and strip the extension *
  gString newTitle = this->tData.sf[fIndex].sfName ;
  short i = newTitle.findlast( L'.' ) ;
  newTitle.limitChars( i ) ;
  newTitle.copy( this->tData.sf[fIndex].sfTag.field[tfTit2], gsMAXCHARS ) ;

}  //* End csstSetSongTitle() *

//*************************
//*   Cmd_Popularimeter   *
//*************************
//******************************************************************************
//* Edit or reset Popularimeter and/or Play Counter.                           *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************
//* Notes:                                                                     *
//* ======                                                                     *
//* 1) This method combines editing of the POPM and PCNT tag frames.           *
//* 2) The method and the data format are designed according to MP3 standard.  *
//* 3) Of the audio files we've seen:                                          *
//*    a) the popEmail field has never been used for an email address          *
//*    b) the popCounter is always zero                                        *
//*    c) the playCounter is always zero, BUT we know that some media players  *
//*       can and do update it.                                                *
//* 4) We always set popCount to zero.                                         *
//* 5) playCounter is set to user's selection.                                 *
//* 6) popEmail is set to user's value.                                        *
//*    We are considering defaulting this field to                             *
//*       "Taggit v:x.x.xx The Software Samurai"                               *
//*                                                                            *
//* Notes on "star" rating: (Wikipedia)                                        *
//* ===================================                                        *
//* There is a loose de facto standard for implementation of song ratings.     *
//* Most apps will display 0 to 5 stars for any given song, and how the stars  *
//* are expressed can vary. For instance, when rating a song in iTunes, the    *
//* rating is not embedded in the tag in the music file, but is instead stored *
//* in a separate database that contains all of the iTunes metadata. Other     *
//* media players can embed rating tags in music files, but not necessarily    *
//* the same way, so as a result a song which is rated on one media player     *
//* sometimes won't display the rating the same way, or at all, when played    *
//* on other software or mobile device.                                        *
//*                                                                            *
//* However, there is a "Popularimeter" frame in the ID3v2 specification meant *
//* for this purpose. The frame is called POPM and Windows Explorer, Windows   *
//* Media Player, Winamp, foobar2000, MediaMonkey, and other software all map  *
//* roughly the same ranges of 0–255 to a 0–5 stars value for display.         *
//*                                                                            *
//* The following list details how Windows Explorer reads and writes           *
//* the POPM frame:                                                            *
//* 224-255 = 5 stars when READ, writes 255                                    *
//* 160-223 = 4 stars when READ, writes 196                                    *
//* 096-159 = 3 stars when READ, writes 128                                    *
//* 032-095 = 2 stars when READ, writes 64                                     *
//* 001-031 = 1 star  when READ, writes 1                                      *
//* -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  *
//*                                                                            *
//* Mathematically speaking: 255 units / 10 == 25.5 units per 0.5 star         *
//* We can quibble about where the breaks should go, but we weight it toward   *
//* _whole_ stars by assigning the larger range to whole stars.                *
//* POPM     Rating (stars)                                                    *
//* ====     ==============                                                    *
//* 0        0                                                                 *
//* 1-25     0.5                                                               *
//* 26-51    1.0                                                               *
//* 52-76    1.5                                                               *
//* 77-102   2.0                                                               *
//* 103-127  2.5                                                               *
//* 128-153  3.0                                                               *
//* 154-178  3.5                                                               *
//* 179-204  4.0                                                               *
//* 205-229  4.5                                                               *
//* 230-255  5.0                                                               *
//*                                                                            *
//******************************************************************************

void Taggit::Cmd_Popularimeter ( void )
{
   const short dlgROWS  = 23,             // dialog lines
               dlgCOLS  = 67,             // dialog columns
               dlgY     = fieldwinY + 1,  // dialog position
               dlgX     = fieldwinX + 1 ;
   attr_t dColor = this->cs.sd,           // dialog interior color
          hColor = this->cs.em ;          // bold dialog text
   AppLang lang = this->cfgOpt.appLanguage ; // language index
   winPos wp( 3, 2 ) ;                    // position of static text
   gString gs ;                           // text formatting
   bool   contRTL = false,                // 'true' if Textbox text dispayed as RTL
          displayUpdate = false ;         // if 'true' update parent window

   enum popControls : short
   {
      savePB = ZERO, cancelPB, popSP, msgTB, cntSP, toggPB, popCTRLS, popText
   } ;
   const char* Labels[][8] = 
   {
      {  // English
         "     SAVE     ",                   // 'save' pushbutton
         "    CANCEL    ",                   // 'cancel' pushbutton
         "Set Popularity Meter",             // 'popularity' spinner
         "Popularity Message",               // 'email' textbox
         "Set Play Counter",                 // 'play count' spinner
         " LT^R ",                           // 'RTL' pushbutton
         "  Adjust Popularity Meter  ",      // dialog title
       //* Static text       |         |         |         |         |  |<=63  *
         "The Popularity Meter is implemented as a numeric value\n"
         "between 0 and 255. Media players generally represent this\n"
         "as a number of \"stars\" (between 0 and 5 stars).\n"
         "Note: mapping of this value to the number of \"stars\" varies\n"
         "among media players. Please see documentation for details.\n"
         "\n\n\n"
         "Use this field to enter an email address or other message\n"
         "related to the Popularity Meter.\n"
         "\n\n\n"
         "The Play Counter is an integer value representing the number\n"
         "of times the song has been played by the media player.\n"
         "Many media players have a configuration option to enable\n"
         "or disable this feature.",
      },
      {  // Espanol
         "   GUARDAR    ",                   // 'save' pushbutton
         "   CANCELAR   ",                   // 'cancel' pushbutton
         "Establecer el Medidor de Popularidad",   // 'popularity' spinner
         "Mensaje sobre popularidad",              // 'email' textbox
         "Establecer Contador de Reproducción",    // 'play count' spinner
         " LT^R ",                                 // 'RTL' pushbutton
         "  Ajuste el Medidor de Popularidad  ",   // dialog title
       //* Static text       |         |         |         |         |  |<=63  *
         "El medidor de popularidad se implementa como un valor numérico\n"
         "entre 0 y 255. Los reproductores de medios generalmente\n"
         "representan esto como un número de \"estrellas\" (entre 0 y 5\n"
         "estrellas). Nota: la asignación de este valor al número de\n"
         "\"estrellas\" varía entre los reproductores multimedia.\n"
         "Consulte la documentación para obtener más detalles.\n"
         "\n\n"
         "Utilice este campo para ingresar una dirección de correo\n"
         "electrónico u otro mensaje relacionado con el Medidor de\n"
         "Popularidad.\n"
         "\n\n"
         "El contador de reproducción es un valor entero que representa\n"
         "el número de veces que la canción ha sido reproducida por el\n"
         "reproductor multimedia. Muchos reproductores multimedia tienen\n"
         "una opción de configuración para activar o desactivar esta\n"
         "función.",
      },
      {  // Zhongwen
         "     保存     ",                 // 'save' pushbutton
         "     取消     ",                 // 'cancel' pushbutton
         "设置人气价值",                    // 'popularity' spinner
         "关于人气的简短消息",               // 'email' textbox
         "回放 计数器",                    // 'play count' spinner
         " LT^R ",                        // 'RTL' pushbutton
         "    调整人气指标    ",            // dialog title
         //* Static text *
         "流行度计实现为0到255之间的数值。\n"
         "媒体播放器通常将此值显示为“星”（0-5星）。\n"
         "注意：这个值和“星”的数量之间的对应关系在媒体播放器之间是不同的。\n"
         "请参阅文档了解详情。\n"
         "\n\n\n\n"
         "使用此字段输入电子邮件地址或与受欢迎程度相关的其他消息。\n"
         "\n"
         "\n\n\n"
         "播放计数器是表示媒体播放器播放歌曲的次数的整数值。\n"
         "许多媒体播放器具有启用或禁用此功能的配置选项。",
      },
      {  // TiengViet
         "    Để Dành   ",                // 'save' pushbutton
         "    Hủy Bỏ    ",                // 'cancel' pushbutton
         "Đặt tính phổ biến đồng hồ",      // 'popularity' spinner
         "Thông báo về xếp hạng phổ biến", // 'email' textbox
         "Đặt Số lượt phát",              // 'play count' spinner
         " LT^R ",                        // 'RTL' pushbutton
         "  Điều chỉnh chỉ mục phổ biến  ",// dialog title
       //* Static text       |         |         |         |         |  |<=63  *
         "Chỉ số phổ biến là được thực hiện như là một giá trị số giữa\n"
         "0 và 255. Phần mềm chơi media thường đại diện cho giá trị này\n"
         "như một số \"ngôi sao\" (từ 0 đến 5 sao). Lưu ý: ánh xạ giá\n"
         "trị này với số lượng \"sao\" khác nhau giữa các trình phát\n"
         "đa phương tiện. Vui lòng xem tài liệu để biết chi tiết.\n"
         "\n\n\n"
         "Sử dụng trường này để nhập địa chỉ email hoặc tin nhắn khác\n"
         "liên quan đến chỉ mục phổ biến.\n"
         "\n\n\n"
         "Sự phát lại bộ đếm là một giá trị số nguyên cho biết số lần\n"
         "bài hát đã được phát. Nhiều người chơi media có một tùy chọn\n"
         "cấu hình để kích hoạt hoặc vô hiệu hóa tính năng này.\n",
      },
   } ;

   //* Additional parameters for dctSPINNER object initializations *
   dspinData dsDataPop ( 0, 255, 
                         this->tData.sf[this->tData.sfFocus].sfTag.pop.popStar, 
                         dspinINTEGER, this->cs.saved ),
             dsDataCnt ( 0, 99999, 
                         this->tData.sf[this->tData.sfFocus].sfTag.pop.popCount, 
                         dspinINTEGER, this->cs.saved ) ;

   //* Control definitions *
   InitCtrl ic[popCTRLS] =             // array of dialog control info
   {
   {  //* 'Save' pushbutton - - - - - - - - - - - - - - - - - - - - -   savePB *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(dlgROWS - 2),           // ulY:       upper left corner in Y
      short(dlgCOLS / 2 - 15),      // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      14,                           // cols:      control columns
      Labels[lang][savePB],         // dispText:  
      this->cs.pn,                  // nColor:    non-focus color
      this->cs.pf,                  // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "",                           // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[cancelPB],                // nextCtrl:  link in next structure
   },
   {  //* 'Cancel' pushbutton - - - - - - - - - - - - - - - - - - -   cancelPB *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      ic[savePB].ulY,               // ulY:       upper left corner in Y
      short(ic[savePB].ulX + ic[savePB].cols + 2), // ulX: upper left corner in X
      1,                            // lines:     (n/a)
      14,                           // cols:      control columns
      Labels[lang][cancelPB],       // dispText:  
      this->cs.pn,                  // nColor:    non-focus color
      this->cs.pf,                  // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "",                           // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[popSP]                    // nextCtrl:  link in next structure
   },
   {  //* 'Popularity' spinner - - - - - - - - - - - - - - - - - - - - - popSP *
      dctSPINNER,                   // type:      
      rbtTYPES,                     // sbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      2,                            // ulY:       upper left corner in Y
      2,                            // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      6,                            // cols:      control columns
      NULL,                         // dispText:  (n/a)
      this->cs.tf,                  // nColor:    non-focus color
      this->cs.unsaved,             // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      Labels[lang][popSP],          // label:     
      ZERO,                         // labY:      
      7,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      &dsDataPop,                   // spinData:  spinner init
      true,                         // active:    (initially disabled)
      &ic[msgTB]                    // nextCtrl:  link in next structure
   },
   {  //* 'Message Textbox - - - - - - - - - - - - - - - - - - - - - - - msgTB *
      dctTEXTBOX,                   // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(ic[popSP].ulY + 8),     // ulY:       upper left corner in Y
      ic[popSP].ulX,                // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      31,                           // cols:      control columns
      this->tData.sf[this->tData.sfFocus].sfTag.pop.popEmail, // dispText:  
      this->cs.tn,                  // nColor:    non-focus color
      this->cs.tf,                  // fColor:    focus color
      tbPrint,                      // filter:    all printing characters
      Labels[lang][msgTB],          // label:     
      ZERO,                         // labY:      
      32,                           // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[cntSP]                    // nextCtrl:  link in next structure
   },
   {  //* 'Play Count' spinner - - - - - - - - - - - - - - - - - - - - - cntSP *
      dctSPINNER,                   // type:      
      rbtTYPES,                     // sbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(ic[msgTB].ulY + 5),     // ulY:       upper left corner in Y
      ic[popSP].ulX,                // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      6,                            // cols:      control columns
      NULL,                         // dispText:  (n/a)
      this->cs.tf,                  // nColor:    non-focus color
      this->cs.unsaved,             // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      Labels[lang][cntSP],          // label:     
      ZERO,                         // labY:      
      7,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      &dsDataCnt,                   // spinData:  spinner init
      true,                         // active:    (initially disabled)
      NULL                          // nextCtrl:  link in next structure
   },
   {  //* 'Toggle' pushbutton - - - - - - - - - - - - - - - - - - - -   toggPB *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      ic[savePB].ulY,               // ulY:       upper left corner in Y
      2,                            // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      5,                            // cols:      control columns
      Labels[lang][toggPB],         // dispText:  
      this->cs.pn,                  // nColor:    non-focus color
      this->cs.pf,                  // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "",                           // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      NULL                          // nextCtrl:  link in next structure
   },
   } ;
   //* Dialog definition *
   InitNcDialog dInit( dlgROWS,         // number of display lines
                       dlgCOLS,         // number of display columns
                       dlgY,            // Y offset from upper-left of terminal 
                       dlgX,            // X offset from upper-left of terminal 
                       NULL,            // dialog title
                       ncltDUAL,        // border line-style
                       dColor,          // border color attribute
                       dColor,          // interior color attribute
                       ic               // pointer to list of control definitions
                     ) ;

   //* If user interface language is RTL *
   if ( this->cfgOpt.rtl )
   {  //* Reposition the controls *
      ic[popSP].ulX = dlgCOLS - ic[popSP].cols - 2 ;
      ic[popSP].labX = -2 ;
      ic[msgTB].ulX = dlgCOLS - ic[msgTB].cols - 2 ;
      ic[msgTB].labX = -2 ;
      ic[cntSP].ulX = dlgCOLS - ic[cntSP].cols - 2 ;
      ic[cntSP].labX = -2 ;
      wp.xpos = dlgCOLS - 3 ;    // reposition the static text
      //* Enable the optional Pushbutton for *
      //* toggling Textbox display.          *
      ic[cntSP].nextCtrl = &ic[toggPB] ;

   }

   this->dPtr->SetDialogObscured () ;  // save parent dialog

   //* Open the dialog *
   NcDialog* dp = new NcDialog ( dInit ) ;
   cpDialog = dp ;
   cpColor  = hColor ;
   cpWp     = { short(ic[popSP].ulY - 1), short(ic[popSP].ulX + 1) } ;

   //* Define a list of keycodes reserved for *
   //* access to clipboard functionality and  *
   //* enable invalid-text-input alert.       *
   this->uiDefineReservedKeys ( dp ) ;
   dp->TextboxAlert ( msgTB, true ) ;

   //* If user interface language is RTL *
   if ( this->cfgOpt.rtl )
      dp->DrawLabelsAsRTL () ;

   if ( (dp->OpenWindow()) == OK )
   {
      //* Set the dialog title *
      dp->SetDialogTitle ( Labels[lang][popCTRLS], hColor ) ;

      //* Emphasize control labels *
      dp->WriteString ( ic[popSP].ulY, (ic[popSP].ulX + ic[popSP].labX),
                        Labels[lang][popSP], hColor, false, this->cfgOpt.rtl ) ;
      dp->WriteString ( ic[msgTB].ulY, (ic[msgTB].ulX + ic[msgTB].labX),
                        Labels[lang][msgTB], hColor, false, this->cfgOpt.rtl ) ;
      dp->WriteString ( ic[cntSP].ulY, (ic[cntSP].ulX + ic[cntSP].labX),
                        Labels[lang][cntSP], hColor, false, this->cfgOpt.rtl ) ;
      //* Display dialog's static text *
      wp = dp->WriteParagraph ( wp, Labels[lang][popText], dColor,
                                false, this->cfgOpt.rtl ) ;
      
      dp->RefreshWin () ;

      //* Establish a call-back method for updating Textbox data.*
      dp->EstablishCallback ( &cpCallback ) ;

      //* Interact with the user *
      uiInfo   Info ;                     // user interface data returned here
      short    icIndex = ZERO ;           // index of control with focus
      bool     done = false ;             // loop control
      while ( ! done )
      {
         if ( ic[icIndex].type == dctPUSHBUTTON )
         {
            if ( Info.viaHotkey )
               Info.HotData2Primary () ;
            else
               icIndex = dp->EditPushbutton ( Info ) ;
            if ( Info.dataMod != false )
            {
               if ( Info.ctrlIndex == savePB )
               {  //* Collect data from controls and if different *
                  //* from live data, update the live data.       *
                  dp->GetTextboxText ( msgTB, gs ) ;
                  int s, c, si = this->tData.sfFocus ;
                  dp->GetSpinnerValue ( popSP, s ) ;
                  dp->GetSpinnerValue ( cntSP, c ) ;
                  if ( ((gs.compare( this->tData.sf[si].sfTag.pop.popEmail,
                                    true )) != ZERO)
                       || (this->tData.sf[si].sfTag.pop.popCount != ZERO)
                       || (s != this->tData.sf[si].sfTag.pop.popStar)
                       || ((UINT)c != this->tData.sf[si].sfTag.pop.playCount) )
                  {
                     gs.copy( this->tData.sf[si].sfTag.pop.popEmail, gsMAXBYTES ) ;
                     this->tData.sf[si].sfTag.pop.popCount  = ZERO ;
                     this->tData.sf[si].sfTag.pop.playCount = c ;
                     this->tData.sf[si].sfTag.pop.popStar   = (uint8_t)s ;

                     //* We now have live data for both frames.     *
                     //* Set the data flags and edits-pending flags.*
                     this->tData.sf[si].sfTag.pop.popdata =
                     this->tData.sf[si].sfTag.pop.cntdata =
                     this->tData.sf[si].sfTag.pop.pcMod   =
                     this->tData.sf[si].sfTag.tfMod       = true ;
                     displayUpdate = true ;
                  }
#if 0    // TEMP TEMP TEMP
dp->CaptureDialog ( "capturedlg.txt" ) ;
dp->CaptureDialog ( "capturedlg.html", true, false, 
                    "infodoc-styles.css", 4, false, nc.blR ) ;
#endif   // TEMP TEMP TEMP
                  done = true ;
               }
               else if ( Info.ctrlIndex == cancelPB )
                  done = true ;
               else if ( Info.ctrlIndex == toggPB )
               {  //* Note: We will arrive here ONLY if UI language is RTL. *
                  contRTL = contRTL ? false : true ;
                  dp->DrawContentsAsRTL ( msgTB, contRTL ) ;
                  dp->RefreshWin () ;
               }
            }
         }

         else if ( ic[icIndex].type == dctTEXTBOX )
         {
            Info.viaHotkey = false ;      // ignore hotkey data
            icIndex = dp->EditTextbox ( Info ) ;
         }

         else if ( ic[icIndex].type == dctSPINNER )
         {
            Info.viaHotkey = false ;      // ignore hotkey data
            icIndex = dp->EditSpinner ( Info ) ;
         }

         //* Move focus to appropriate control *
         if ( ! done && ! Info.viaHotkey )
         {
            if ( Info.keyIn == nckSTAB )
               icIndex = dp->PrevControl () ; 
            else
               icIndex = dp->NextControl () ;
         }
      }        // while()
   }
   if ( dp != NULL )                         // close the window
      delete ( dp ) ;

   this->dPtr->RefreshWin () ;               // restore parent dialog
   if ( displayUpdate )    // if data may have changed, update the indicator
      this->UpdateFilenameWindow () ;

}  //* End Cmd_Popularimeter() *

//*************************
//*      cpCallback       *
//*************************
//******************************************************************************
//* This is a callback method for display of the "star rating" for the         *
//* Cmd_Popularimeter() method.                                                *
//*                                                                            *
//* Input  : currIndex: index of control that currently has focus              *
//*          wkey     : user's key input data                                  *
//*          firstTime: the EstablishCallback() method calls this method once  *
//*                     with firstTime==true, to perform any required          *
//*                     initialization. Subsequently, the NcDialog class       *
//*                     always calls with firstTime==false.                    *
//* Returns: OK                                                                *
//******************************************************************************

static short cpCallback ( const short currIndex, const wkeyCode wkey, bool firstTime )
{
   enum starRating : int
   {
      sr00 = ZERO,      // no stars
      sr05 = 25,        // 0.5 star max
      sr10 = 51,        // 1.0 star max
      sr15 = 76,        // 1.5 star max
      sr20 = 102,       // 2.0 star max
      sr25 = 127,       // 2.5 star max
      sr30 = 153,       // 3.0 star max
      sr35 = 178,       // 3.5 star max
      sr40 = 204,       // 4.0 star max
      sr45 = 229,       // 4.5 star max
      sr50 = 255        // 5.0 star max
   } ;
   const char* const sRating[11] = 
   {
      "     ",    // 0.0 stars
      "½    ",    // 0.5 stars
      "☻    ",    // 1.0 stars
      "☻½   ",    // 1.5 stars
      "☻☻   ",    // 2.0 stars
      "☻☻½  ",    // 2.5 stars
      "☻☻☻  ",    // 3.0 stars
      "☻☻☻½ ",    // 3.5 stars
      "☻☻☻☻ ",    // 4.0 stars
      "☻☻☻☻½",    // 4.5 stars
      "☻☻☻☻☻",    // 5.0 stars
   } ;
   const short popSP = 2 ;          // index of popStar spinner control
   static int oldVal = ZERO ;       // previous spinner value
   int        newVal = ZERO ;       // current spinner value

   //*************************
   //* First-time processing *
   //*************************
   if ( firstTime != false )
   {
      /* nothing to do */
   }

   //***********************
   //* Standard processing *
   //***********************
   cpDialog->GetSpinnerValue ( popSP, newVal ) ;
   if ( firstTime || ((currIndex == popSP) && (newVal != oldVal)) )
   {
      oldVal = newVal ;
      gString gs ;
      if ( newVal > sr45 )       gs = sRating[10] ;
      else if ( newVal > sr40 )  gs = sRating[ 9] ;
      else if ( newVal > sr35 )  gs = sRating[ 8] ;
      else if ( newVal > sr30 )  gs = sRating[ 7] ;
      else if ( newVal > sr25 )  gs = sRating[ 6] ;
      else if ( newVal > sr20 )  gs = sRating[ 5] ;
      else if ( newVal > sr15 )  gs = sRating[ 4] ;
      else if ( newVal > sr10 )  gs = sRating[ 3] ;
      else if ( newVal > sr05 )  gs = sRating[ 2] ;
      else if ( newVal > sr00 )  gs = sRating[ 1] ;
      else                       gs = sRating[ 0] ;
      cpDialog->WriteString ( cpWp, gs, cpColor, true ) ;
   }

   return OK ;

}  //* End cpCallback() *

//*************************
//*  Cmd_SequenceTracks   *
//*************************
//******************************************************************************
//* Automatically calculate the Track field for all files based on the current *
//* display order. This overwrites any previous contents of the Track fields.  *
//* The tag-data edits-pending flag will be set.                               *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************
//* Notes:                                                                     *
//* -- If the Track field is inactive it will still be filled; however,        *
//*    the user won't be able to see the field and may be confused about       *
//*    what happened.In future, we may post an info dialog describing why      *
//*    the action was moronic. Currently, we simply leave the edits-pending    *
//*    flag unmodified.                                                        *
//*    -- Within the edit routine, the error is signalled, but in that case,   *
//*       the field will always be active.                                     *
//* -- Note that we make an unnecessary call to uiemFormatTrack() for          *
//*    consistency, and in case we want to globally change the format at a     *
//*    later date.                                                             *
//*                                                                            *
//*                                                                            *
//******************************************************************************

void Taggit::Cmd_SequenceTracks ( void )
{
   //* Track and Total-tracks counters *
   short trk = 1, tot = this->tData.sfCount ;
   gString gs ;
   bool fieldActive = this->tData.sffDisp[tfTrck] ;

   for ( short fIndex = ZERO ; fIndex < this->tData.sfCount ; ++fIndex, ++trk )
   {
      gs.compose( "%2hd/%02hd", &trk, &tot ) ;
      this->uiemFormatTrack ( gs ) ;   // (see note above)
      gs.copy( this->tData.sf[fIndex].sfTag.field[tfTrck], gsMAXCHARS ) ;
      if ( fieldActive )
         this->tData.sf[fIndex].sfTag.tfMod = true ;
   }

   if ( fieldActive != false )
   {
      this->UpdateDataWindows () ;
   }
   else
   {
      // see note above
   }

}  //* End CmdSequenceTracks() *

//*************************
//* Cmd_RTL_ToggleFields  *
//*************************
//******************************************************************************
//* For RTL user interface languages, toggle direction of text in fields.      *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void Taggit::Cmd_RTL_ToggleFields ( void )
{
   if ( this->cfgOpt.rtl )
   {
      this->cfgOpt.rtlf = this->cfgOpt.rtlf ? false : true ;
      this->UpdateDataWindows () ;
   }

}  //* End Cmd_RTL_ToggleFields() *

//*************************
//*    Cmd_ReportImage    *
//*************************
//******************************************************************************
//* Display the setup and text information associated with one or more images  *
//* embedded in the referenced file.                                           *
//*                                                                            *
//* Input  : fIndex  : index of source file                                    *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************
//* Notes:                                                                     *
//* ------                                                                     *
//* a) We call the system default 'Image Viewer' application to display the    *
//*    image. See the LaunchDefaultApplication() method for details.           *
//* b) Currently, this method assumes that the image is located in the         *
//*    id3v2_image member (MP3) of the EmbeddedImage class. When we support    *
//*    image capture for additional audio formats, we will revisit this.       *
//*                                                                            *
//* To set the default viewer:                                                 *
//*  a) open Nautilus (or other file manager)                                  *
//*  b) find a file with an extension of 'jpg'                                 *
//*  c) right-click the file and select "properties"                           *
//*  d) click the "open-with" tab                                              *
//*  e) select the desired application                                         *
//*  f) click the "set-as-default" button                                      *
//*  g) repeat steps b through f for a file with an extension of 'png'         *
//*                                                                            *
//* xdg-open filename    // invokes the default viewer                         *
//* gnome-open filename  // invokes the default viewer (if running Gnome)      *
//* eog filename         // invokes Eye of Gnome (image viewer)                *
//* To invoke an external application, use fork() + exec() and then detatch    *
//* from the terminal window.                                                  *
//* If running in a multi-threaded environment, use pthread_atfork() instead.  *
//*                                                                            *
//* Notes on LTR/RTL:                                                          *
//* a) The filename is displayed as LTR text unless _BOTH_ 'rtl' and 'rtlf'    *
//*    flags are set. This is done because filenames, _especially_ filename    *
//*    extensions are almost always LTR in nature.                             *
//* b) The dialog title and the Pushbutton labels will follow the 'rtl' flag.  *
//* c) The contents of the Billboard control have two parts: 1) the internal   *
//*    data labels, and 2) the reported data.                                  *
//*    -- The labels will follow the 'rtl' flag.                               *
//*       The label portion of the template strings ('bbTemplate[]') is        *
//*       defined as all text UP-TO-AND-INCLUDING the space after the colon    *
//*       character ( ':' ).                                                   *
//*    -- The reported data are ALMOST entirely native LTR data and will be    *
//*       dynamically reversed when 'rtl' is set so that they will be displayed*
//*       correctly when written as RTL data.                                  *
//*       -- 'Picture type' code is a hex value and will be reversed.          *
//*       -- 'MIME type' is an ASCII string and will be reversed.              *
//*       -- 'Image size' is a numeric value (with a language-specific "bytes" *
//*          appended). The numeric portion will be reversed.                  *
//*       -- 'Image file' is a path/filename specification and will be reversed*
//*       -- 'Description may be either LTR or RTL depending on how it was     *
//*          written. Since neither the application nor the NcDialog API knows *
//*          what language a field is written in, the user must decide how the *
//*          description is to be displayed (CTRL+R hotkey).                   *
//*          The description is assumed to be LTR data and is therefore        *
//*          initially reversed for display. However, the non-reversed         *
//*          description is kept in the local 'wtmp' array in case user        *
//*          wants to display it in the other direction.                       *
//*    -- Programmer's Note: This sequence relies on a properly-formatted      *
//*       template string for the active UI language. Colon characters ( ':' ) *
//*       and newline characters ( '\n' ) must be in the expected positions    *
//*       for correct parsing of the data.                                     *
//*    -- Programmer's Note: If filespec is wider than the available space,    *
//*       ugliness will occur in the output because the Billboard line-wraps   *
//*       on word or on full line. This is not a problem for the temp files,   *
//*       which have short paths, but external images may uglify the output.   *
//*       Billboard columns minus width of label is the maximum amount of      *
//*       space available for filespec if line-wrap is to be avoided, so the   *
//*       problem _could be_ solved, but since all the data will be displayed, *
//*       at this time a fix doesn't seem worth it.                            *
//*                                                                            *
//******************************************************************************

void Taggit::Cmd_ReportImage ( short fIndex )
{
   const short dlgROWS  = 17,             // dialog lines
               dlgCOLS  = 60,             // dialog columns
               bbROWS   = (dlgROWS - 7),  // billboard lines
               dlgY     = fieldwinY + 1,  // dialog position
               dlgX     = fieldwinX + 1 ;
   attr_t dColor = this->cs.sd,           // dialog interior color
          hColor = this->cs.em,           // bold dialog text
          bColor = attr_t(this->cs.scheme == ncbcBR ? nc.blR : nc.brR) ; // billboard color
   AppLang lang = this->cfgOpt.appLanguage ; // language index
   gString gsOut, gstmp ;                 // text formatting
   bool rtlFields = (this->cfgOpt.rtl && this->cfgOpt.rtlf) ; // fields and filenames as RTL
   bool descRTL = false ;                 // if 'true' display description text as RTL

   attr_t bbAttr[bbROWS] = { bColor, bColor, bColor, bColor, bColor, 
                             bColor, bColor, bColor, bColor, bColor } ;
   enum riControls : short 
      { donePB = ZERO, viewPB, nextPB, dataBB, toggPB, riCONTROLS } ;

   const char* Labels[][7] = 
   {
      {  // English
         "  View Image Data  ",                    // (0) dialog title
         "    DONE    ",                           // (1) 'done' pushbutton
         "        View Image        ",             // (2) view image(s)
         "    NEXT    ",                           // (3) 'next' pushbutton
         "\nThis file contains no images.",        // (4) no-image message
         "(none)",                                 // (5) no-description message
         "Sorry, unable to launch external application.", // (6) no-launch message
      },
      {  // Espanol
         "  Ver Resumen de Datos  ",               // (0) dialog title
         "   HECHO    ",                           // (1) 'done' pushbutton
         "       Ver Imágenes       ",             // (2) view image(s)
         " SIGUIENTE  ",                           // (3) 'next' pushbutton
         "\nEste archivo no contiene imágenes.",   // (4) no-image message
         "(ninguna)",                              // (5) no-description message
         "Lo sentimos, no se pudo iniciar la aplicación externa.", // (6) no-launch message
      },
      {  // Zhongwen
         "   查看图像数据   ",                       // (0) dialog title
         "    执行    ",                            // (1) 'done' pushbutton
         "         查看图像         ",              // (2) view image(s)
         "   下一个   ",                            // (3) 'next' pushbutton
         "\n此文件不包含图像。",                      // (4) no-image message
         "（没有）",                                // (5) no-description message
         "对不起，我们无法启动外部应用程序。",           // (6) no-launch message
      },
      {  // TiengView
         "  Xem dữ liệu hình ảnh  ",               // (0) dialog title
         "  Làm Xong  ",                           // (1) 'done' pushbutton
         "       Xem hình ảnh       ",             // (2) view image(s)
         "  Kế Tiếp   ",                           // (3) 'next' pushbutton
         "\nTập tin này không chứa hình ảnh.",     // (4) no-image message
         "(không ai)",                             // (5) no-description message
         "Rất tiếc, không thể khởi chạy ứng dụng bên ngoài.", // (6) no-launch message
      },
   } ;
   const char* bbTemplate[] = 
   {  // English
      "\n"
      "Picture type : %02hhXh - %s\n"
      "MIME type    : %s\n"
      "Image size   : %S bytes\n"
      "Image file   : %s\n"
      "Description  : \n"
      "%s\n",
      // Espanol
      "\n"
      "Tipo de imagen      : %02hhXh - %s\n"
      "Tipo de MIME        : %s\n"
      "Tamaño de la imagen : %S bytes\n"
      "Archivo de imagen   : %s\n"
      "Descripción    : \n"
      "%s\n",
      // Zhongwen
      "\n"
      "图像类型  : %02hhXh - %s\n"
      "MIME 类型 : %s\n"
      "图片大小  : %S 字节数\n"
      "图像文件  : %s\n"
      "描述      : \n"
      "%s\n",
      // Tieng Viet
      "\n"
      "Loại hình ảnh    : %02hhXh - %s\n"
      "MIME loại        : %s\n"
      "Kích cỡ hình     : %S byte\n"
      "Tập tin hình ảnh : %s\n"
      "Sự miêu tả       : \n"
      "%s\n",
   } ;

InitCtrl ic[riCONTROLS] =      // array of dialog control info
{
   {  //* 'Done' pushbutton - - - - - - - - - - - - - - - - - - - - -   donePB *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(dlgROWS - 2),           // ulY:       upper left corner in Y
      short(dlgCOLS / 2 + 1),       // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      12,                           // cols:      control columns
      Labels[lang][1],              // dispText:  
      this->cs.pn,                  // nColor:    non-focus color
      this->cs.pf,                  // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "",                           // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[viewPB],                  // nextCtrl:  link in next structure
   },
   {  //* 'View' pushbutton  - - - - - - - - - - - - - - - - - - - -    viewPB *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(ic[donePB].ulY - 2),    // ulY:       upper left corner in Y
      short(ic[donePB].ulX - ic[donePB].cols - 2), // ulX: upper left corner in X
      1,                            // lines:     (n/a)
      26,                           // cols:      control columns
      Labels[lang][2],              // dispText:  
      attr_t(this->cs.scheme == ncbcGR ? nc.brR : nc.grR), // nColor: non-focus color
      this->cs.pf,                  // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "",                           // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[nextPB]                   // nextCtrl:  link in next structure
   },
   {  //* 'Next' pushbutton - - - - - - - - - - - - - - - - - - - -     nextPB *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      ic[donePB].ulY,               // ulY:       upper left corner in Y
      short(ic[donePB].ulX - ic[donePB].cols - 2), // ulX: upper left corner in X
      1,                            // lines:     (n/a)
      12,                           // cols:      control columns
      Labels[lang][3],              // dispText:  
      this->cs.pn,                  // nColor:    non-focus color
      this->cs.pf,                  // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "",                           // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[dataBB]                   // nextCtrl:  link in next structure
   },
   {  //* 'Billboard' (read-only)  - - - - - - - - - - - - - - - - - -  dataBB *
      dctBILLBOARD,                 // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      2,                            // ulY:       upper left corner in Y
      2,                            // ulX:       upper left corner in X
      bbROWS,                       // lines:     control lines
      short(dlgCOLS - 4),           // cols:      control columns
      NULL,                         // dispText:  initial display text
      bColor,                       // nColor: non-focus color
      bColor,                       // fColor: focus color
      tbPrint,                      // filter:    (n/a)
      NULL,                         // label:     control label
      ZERO,                         // labY:      label offset Y
      ZERO,                         // labX       label offset X
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      bbAttr,                       // scrColor:  if specified, attribute array
      NULL,                         // spinData:  (n/a)
      false,                        // active:    view-only
      NULL                          // nextCtrl:  link in next structure
   },
   {  //* 'Toggle' pushbutton - - - - - - - - - - - - - - - - - - - -   toggPB *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      ic[donePB].ulY,               // ulY:       upper left corner in Y
      2,                            // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      5,                            // cols:      control columns
      " LT^R ",                     // dispText:  
      this->cs.pn,                  // nColor:    non-focus color
      this->cs.pf,                  // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "",                           // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      NULL                          // nextCtrl:  link in next structure
   },
} ;

   //* Dialog definition *
   InitNcDialog dInit( dlgROWS,         // number of display lines
                       dlgCOLS,         // number of display columns
                       dlgY,            // Y offset from upper-left of terminal 
                       dlgX,            // X offset from upper-left of terminal 
                       NULL,            // dialog title
                       ncltDUAL,        // border line-style
                       dColor,          // border color attribute
                       dColor,          // interior color attribute
                       ic               // pointer to list of control definitions
                     ) ;

   this->dPtr->SetDialogObscured () ;  // save parent dialog

   //* If user interface is an RTL language, enable the optional *
   //* Pushbutton for toggling display of the descriptions.      *
   if ( this->cfgOpt.rtl )
      ic[dataBB].nextCtrl = &ic[toggPB] ;

   //* Open the dialog *
   NcDialog* dp = new NcDialog ( dInit ) ;

   //* If user interface language is an RTL language, *
   //* set the internal NcDialog flag for RTL output. *
   if ( this->cfgOpt.rtl )
   {
      dp->DrawLabelsAsRTL () ;
      dp->DrawContentsAsRTL ( dataBB ) ;
   }

   if ( (dp->OpenWindow()) == OK )
   {
      //* Set the dialog title *
      dp->SetDialogTitle ( Labels[lang][0], hColor ) ;

      //* This is a simple way to make it APPEAR that *
      //* the Billboard data are inset by one column. *
      dp->ClearArea ( 2, 1, bbROWS, (dlgCOLS - 2), bColor ) ;

      //* Display source filename. (see notes above) *
      winPos wpFName( 1, 2 ) ;
      if ( rtlFields )
         wpFName.xpos = (dlgCOLS - 3) ;
      dp->WriteString ( wpFName, this->tData.sf[this->tData.sfFocus].sfName, 
                        hColor, false, rtlFields ) ;

      EmbeddedImage *basePtr = &this->tData.sf[this->tData.sfFocus].sfTag.ePic,
                    *eiPtr = basePtr ;
      id3v2_image*   epPtr = &eiPtr->mp3img ;
      if ( eiPtr->inUse )
      {
         if ( this->cfgOpt.rtl )
            this->criTransform ( bbTemplate[lang], gsOut, epPtr, Labels[lang][5], descRTL ) ;

         else
         {
            gstmp.formatInt( epPtr->picSize, 11, true ) ;
            gsOut.compose( bbTemplate[lang], 
                           &epPtr->picType, epPtr->picExpl, 
                           epPtr->mimType, gstmp.gstr(), epPtr->picPath,
                           (char*)(*epPtr->txtDesc ? epPtr->txtDesc : Labels[lang][5]) ) ;
         }
      }
      else
      {
         gsOut = Labels[lang][4] ;
         dp->ControlActive ( viewPB, false ) ;
         dp->ControlActive ( nextPB, false ) ;
      }
      dp->SetBillboard ( dataBB, gsOut, bbAttr ) ;

      dp->RefreshWin () ;           // make everything visible

      //* Interact with the user *
      uiInfo   Info ;                     // user interface data returned here
      short    icIndex = ZERO ;           // index of control with focus
      bool     done = false ;             // loop control
      while ( ! done )
      {
         if ( ic[icIndex].type == dctPUSHBUTTON )
         {
            if ( Info.viaHotkey )
               Info.HotData2Primary () ;
            else
               icIndex = dp->EditPushbutton ( Info ) ;
            if ( Info.dataMod != false )
            {
               if ( Info.ctrlIndex == donePB )
                  done = true ;
               else if ( Info.ctrlIndex == viewPB )
               {
                  if ( (this->LaunchDefaultApplication ( epPtr->picPath )) < ZERO )
                  {
                     dp->WriteString ( (ic[viewPB].ulY - 1), 2, 
                                       Labels[lang][6], dColor, true ) ;
                  }
               }
               else if ( Info.ctrlIndex == nextPB )
               {
                  //* If we have not reached the end of the list     *
                  //* OR if there is more than one image in the list *
                  if ( (eiPtr->next != NULL) || (eiPtr != basePtr) )
                  {
                     //* Format and display next record *
                     if ( eiPtr->next != NULL )
                        eiPtr = eiPtr->next ;
                     else     // return to top of list
                        eiPtr = basePtr ;
                     epPtr = &eiPtr->mp3img ;
                     if ( this->cfgOpt.rtl )
                     {
                        this->criTransform ( bbTemplate[lang], gsOut, 
                                             epPtr, Labels[lang][5], descRTL ) ;
                     }
                     else
                     {
                        gstmp.formatInt( epPtr->picSize, 11, true ) ;
                        gsOut.compose( bbTemplate[lang], 
                           &epPtr->picType, epPtr->picExpl, 
                           epPtr->mimType, gstmp.gstr(), epPtr->picPath,
                           (char*)(*epPtr->txtDesc ? epPtr->txtDesc : Labels[lang][5]) ) ;
                     }
                     dp->SetBillboard ( dataBB, gsOut, bbAttr ) ;
                  }
                  else     // no more images
                  {
                     dp->UserAlert () ;
                  }
               }
               else if ( Info.ctrlIndex == toggPB )
               {  //* Note: We will arrive here ONLY if UI language is RTL. *
                  descRTL = descRTL ? false : true ;
                  this->criTransform ( bbTemplate[lang], gsOut, 
                                       epPtr, Labels[lang][5], descRTL ) ;
                  dp->SetBillboard ( dataBB, gsOut, bbAttr ) ;
               }
            }     // dataMod != false
         }        // Pushbuttons

         //* Move focus to appropriate control *
         if ( ! done && ! Info.viaHotkey )
         {
            if ( Info.keyIn == nckSTAB )
               icIndex = dp->PrevControl () ; 
            else
               icIndex = dp->NextControl () ;
         }
      }  // while()
   }
   if ( dp != NULL )                         // close the window
      delete ( dp ) ;

   this->dPtr->RefreshWin () ;         // restore parent dialog

}  //* End Cmd_ReportImage() *

//*************************
//*     criTransform      *
//*************************
//******************************************************************************
//* Called only by Cmd_ReportImage, and ONLY when the user interface language  *
//* is RTL. Convert data for Billboard control to mixed LTR/RTL data.          *
//*                                                                            *
//* Programmer's Note: This method makes intimate assumptions about the        *
//* 'Template' parameter. See notes in Cmd_ReportImage().                      *
//*                                                                            *
//* Input  : Template: sprintf-style formatting string                         *
//*          gsOut   : (by reference) receives the formatted data              *
//*          epPtr   : pointer to image data                                   *
//*          noDesc  : default data if description field is empty              *
//*          rtlDesc : (optional, 'false' by default)                          *
//*                    if 'false', format description text as LTR              *
//*                    if 'true',  format description text as RTL              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************
//* Notes:                                                                     *
//* ------                                                                     *
//* This method is a way of smoothing the interface for RTL languages. Because *
//* we are LTR-centric in our design, there are a few convoluted tricks used   *
//* to integrate RTL user-interface languages. Not all these tricks are        *
//* actually good ideas.                                                       *
//*                                                                            *
//* For the 'rtlDesc' parameter, we can display the description as LTR which   *
//* is the default, or we can at user's discression display the description as *
//* RTL text.                                                                  *
//*  1) If 'rtlDesc' == false:                                                 *
//*     a) If a description is present, display it as LTR.                     *
//*        Note that if the description line wraps, the Billboard formatting   *
//*        algorithm will wrap the BEGINNING of the string to the second       *
//*        description line because we have told it that it was displaying RTL *
//*        data when it was not. (Actual RTL text will of course be displayed  *
//*        properly which is the whole point of the exercise.)                 *
//*        While we _could_ reorder the words of the LTR string to get a       *
//*        proper display, it is not a trivial task, so at this time we ignore *
//*        the issue.                                                          *
//*     b) If there is no description, then display 'noDesc' according to the  *
//*        UI language used, i.e. RTL.                                         *
//*        (This method is only called when cfgOpt.rtl is set.)                *
//*  2) If 'rtlDesc' != false:                                                 *
//*     Display description (or 'noDesc' message) as RTL text (no reversal).   *
//******************************************************************************

void Taggit::criTransform ( const char* Template, gString& gsOut, 
                            const id3v2_image* epPtr, const char* noDesc, bool rtlDesc )
{
   wchar_t wtmp[gsMAXCHARS] ;
   gString gsSize( epPtr->picSize, 11, true ) ;
   this->gsRevChars ( gsSize ) ;
   gString gstmp( epPtr->picPath ) ;
   this->gsRevChars ( gstmp ) ;

   gsOut.compose( Template, 
                  &epPtr->picType, epPtr->picExpl, 
                  epPtr->mimType, gsSize.gstr(), gstmp.ustr(),
                  (char*)(*epPtr->txtDesc != NULLCHAR ? epPtr->txtDesc : noDesc) ) ;
   short bi = (gsOut.after( L':' ) + 1),
         ai = gsOut.find( L'\n', bi ) ;
   gsOut.substr( gstmp, bi, 3 ) ;
   gstmp.copy( wtmp, gsMAXCHARS ) ;
   this->gsRevChars( gstmp ) ;            // reverse the picType code
   gsOut.replace( wtmp, gstmp.gstr() ) ;

   bi = (gsOut.after( L':', ai ) + 1) ;
   ai = gsOut.find( L'\n', bi ) ;
   gsOut.substr( gstmp, bi, (ai - bi) ) ;
   gstmp.copy( wtmp, gsMAXCHARS ) ;
   this->gsRevChars( gstmp ) ;            // reverse the MIME-type string
   gsOut.replace( wtmp, gstmp.gstr() ) ;

   if ( rtlDesc == false && *epPtr->txtDesc != NULLCHAR )
   {
      bi = (gsOut.findlast( L':' )) + 3 ;    // index of description
      ai = (gsOut.find( L'\n', bi )) ;       // limit of description
      gsOut.substr( gstmp, bi, (ai - bi) ) ;
      gstmp.copy( wtmp, gsMAXCHARS ) ;
      this->gsRevChars( gstmp ) ;            // reverse the description string
      gsOut.replace( wtmp, gstmp.gstr(), bi ) ;
   }

}  //* End criTransform() *

//*************************
//*    Cmd_InsertImage    *
//*************************
//******************************************************************************
//* Embed an image into the target file OR edit the description of a           *
//* previously-specified image.                                                *
//*                                                                            *
//* Input  : fIndex  : index of source file                                    *
//*          editImg : (optional, 'false' by default)                          *
//*                    if 'false', prompt user to insert a new image           *
//*                    if 'true',  prompt user to edit the Picture Category    *
//*                                and Description of an image previously      *
//*                                embedded into the source file               *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************
//* Notes:                                                                     *
//* ------                                                                     *
//* 1) See mptDecodeImageFrame() for an excerpt from the id3v2.3 standard      *
//*    the layout of an image frame.                                           *
//*                                                                            *
//* 2) This is a dual-purpose method, and as such is one of the more complex   *
//*    application-level methods. Modify it at your peril.                     *
//*    a) insert a new image                                                   *
//*       -- Filename Textbox is active for editing filespec.                  *
//*       -- Dropdown is active for changing the picture category.             *
//*       -- Description Textbox is active for editing description.            *
//*       -- 'Save to Current' Pushbutton is active.                           *
//*       -- 'Save to All' Pushbutton is active.                               *
//*       -- The 'RTL' Pushbutton is available (CTRL+R) for toggling           *
//*          LTR/RTL Textbox contents.                                         *
//*       -- 'Cancel' Pushbutton is active.                                    *
//*    b) edit stats for an existing image                                     *
//*       -- Filename Textbox is view-only.                                    *
//*       -- Dropdown is active for changing the picture category.             *
//*       -- Description Textbox is active for editing description.            *
//*       -- 'Save' Pushbutton is active.                                      *
//*       -- 'Next Image' Pushbutton is active.                                *
//*       -- 'Delete Image' Pushbutton is active.                              *
//*       -- The 'RTL' Pushbutton is available (CTRL+R) for toggling           *
//*          LTR/RTL Textbox contents.                                         *
//*       -- 'Cancel' Pushbutton is active.                                    *
//*                                                                            *
//* 3) When user "deletes" one or more images.                                 *
//*    a) The corresponding delPix[n] flag is set to indicate the deletion.    *
//*    b) During edit, there are two copies of the image data:                 *
//*       -- the live EmbeddedImage linked list                                *
//*       -- the local array of id3v2_image objects                            *
//*    c) Logically, only the retained images should be copied back to the     *
//*       live-data list; however, technically, it is simpler and more         *
//*       reliable to copy all the modified data back to the live-data list    *
//*       and THEN delete the specified image records. While this may seem     *
//*       counterintuitive, it avoids relying upon pointers that may have been *
//*       invalidated, or compressing an array that might be overrun or other  *
//*       potentially embarrassing processing errors.                          *
//*       Programmer's Note: During this exercise we have reverified the addage*
//*       that: "simple is better than complex."                               *
//*       It took three(3) days of trying various more efficient but           *
//*       error-prone algorithms before we settled on the most straightforward *
//*       (although inefficient) algorithm which is used below.                *
//*    d) The live data are modified only if user selects the Save pushbutton. *
//*       1. Copy all modified (or "deleted") image records from the local     *
//*          array back to their original positions in the live-data list.     *
//*       2. Walk through the live-data list (from tail to head) and delete    *
//*          the nodes whose data have been reset.                             *
//*       3. The actual deletion occurs in tagFields::delPic( EmbeddedImage* );*
//*                                                                            *
//******************************************************************************

void Taggit::Cmd_InsertImage ( short fIndex, bool editImg )
{
   #define DEBUG_CII (0)                  // for debugging only

   const short dlgROWS = 17,              // dialog size
               dlgCOLS = 68,
               dlgY    = fieldwinY + 1,   // dialog position
               dlgX    = fieldwinX + 1,
               pbX     = editImg ? 7 : 12 ;// X offset of 'Save' pushbutton
   attr_t dColor = this->cs.sd,           // dialog interior color
          hColor = this->cs.em ;          // bold dialog text
   AppLang lang = this->cfgOpt.appLanguage ; // language index
   gString gs, gsn ;                      // text formatting
   EmbeddedImage *eiPtr = NULL ;          // pointer to image live data
   attr_t dDataAttr[1] = { dtbmFcolor } ; // color attribute for message in filename field
   dtbmData dData( " ", dDataAttr ) ;     // for display of message in filename field
   short iCount = ZERO,                   // number of embedded images
         iIndex = ZERO ;                  // index into 'tempImg'
   bool rtlFields = bool(this->cfgOpt.rtl && this->cfgOpt.rtlf) ; // write description as RTL

   enum iiControls : short 
   { iiSavePB = ZERO, iiSaveAllPB, iiDeletePB, iiCancelPB, 
     iiFileTB, iiTypeDD, iiDescTB, iiToggPB, iiCONTROLS,
     iiTIT2, iiTYPE, iiSIZE, iiUNSUP, iiNOSUCH, iiNOIMG, iiALTSAVE, iiNXTIMG
   } ;

   //* For edit functionality only:                      *
   //* Count the number of images in the live-data array.*
   //* Allocate a one-dimensional array as temp storage, *
   //* and copy the live data to temp storage.           *
   eiPtr = &this->tData.sf[fIndex].sfTag.ePic ;
   iCount = this->tData.sf[fIndex].sfTag.countPics() ;
   id3v2_image imgtmp[iCount] ;           // working copy of image data
   bool delPix[iCount] ;                  // for tracking "deleted" images
   for ( short i = ZERO ; i < iCount ; ++i )
      delPix[i] = false ;
   if ( editImg && (eiPtr->inUse) )
   {
      for ( short i = ZERO ; eiPtr != NULL; ++i )
      {
         imgtmp[i] = eiPtr->mp3img ;
         eiPtr = eiPtr->next ;
      }
      eiPtr = &this->tData.sf[fIndex].sfTag.ePic ;
   }

   //* Control labels *
   const char* Labels[][iiCONTROLS + 9] = 
   {
      {  // English
         "     SAVE     \n"            // 'save' pushbutton
         "      TO      \n"
         " CURRENT FILE ",
         "     SAVE     \n"            // 'save-all' pushbutton
         "      TO      \n"
         "   ALL FILES  ",
         "    DELETE    \n"            // 'delete' pushbutton
         "     THIS     \n"
         "    IMAGE     ",
         "              \n"            // 'cancel' pushbutton
         "    CANCEL    \n"
         "              ",
         "Image Filename",             // 'filename' textbox
         "Picture \n"                  // 'picType' identifier byte
         "Category",
         "Image Description (64 chars max)", // 'description' string
         " LT^R ",                     // RTL pushbutton
         "    Insert Image    ",       // dialog title               [iiCONTROLS]
         "   Edit Image Setup   ",     // 'edit' title               [iiTIT2]
         "MIME Type",                  // 'MIME' identifier string   [iiTYPE]
         "Image Size: %S bytes",       // Image-size template        [iiSIZE]
         " Unsupported image format!", // MIME-type error            [iiUNSUP]
         " Image file not found!",     // source file not found      [iiNOSUCH]
         "Audio file contains no image data.", // no image data      [iiNOIMG]
         "              \n"            // alternate 'save' button    [iiALTSAVE]
         "     SAVE     \n"
         "              ",
         "     NEXT     \n"            // next-image pushbutton      [iiNXTIMG]
         "              \n"
         "    IMAGE     ",
      },
      {  // Espanol
         "  GUARDAR EN  \n"            // 'save' pushbutton
         "  EL ARCHIVO  \n"
         "    ACTUAL    ",
         "  GUARDAR EN  \n"            // 'save-all' pushbutton
         "   TODOS LOS  \n"
         "   ARCHIVOS   ",
         "   ELIMINAR   \n"            // 'delete' pushbutton
         "     ESTA     \n"
         "    IMAGEN    ",
         "              \n"            // 'cancel' pushbutton
         "   CANCELAR   \n"
         "              ",
         "Nombre de archivo de imagen", // 'filename' textbox
         "Categoría de\n"               // 'picType' identifier byte
         " Imagen",
         "Descripción de la imagen (máximo de 64 caracteres)",   // 'description' string
         " LT^R ",                     // RTL pushbutton
         "    Insertar Imagen    ",    // dialog title
         "  Editar Ajustes de Imagen  ", // 'edit' title             [iiTIT2]
         "MIME Tipo",                  // 'MIME' identifier string   [iiTYPE]
         "Tamaño de la Imagen: %S bytes", // Image-size template     [iiSIZE]
         " ¡El formato de imagen no compatible!", // MIME-type error [iiUNSUP]
         " ¡Archivo de imagen no encontrado!",// source file not found[iiNOSUCH]
         "Audio file contains no image data.", // no image data      [iiNOIMG]
         "              \n"            // alternate 'save' button    [iiALTSAVE]
         "   GUARDAR    \n"
         "              ",
         "  SIGUIENTE   \n"            // next-image pushbutton      [iiNXTIMG]
         "              \n"
         "    IMAGEN    ",
      },
      {  // Zhongwen
         "     保存     \n"             // 'save' pushbutton
         "      到      \n"
         "   当前文件   ",
         "     保存     \n"            // 'save-all' pushbutton
         "      到      \n"
         "   全部文件   \n",
         "              \n"            // 'delete' pushbutton
         "  删除此图片  \n"
         "              ",
         "              \n"            // 'cancel' pushbutton
         "     取消     \n"
         "              ",
         "图像文件名",                  // 'filename' textbox
         "图片类别 \n",                 // 'picType' identifier byte
         "图像描述 （最多64个字符）",     // 'description' string
         " LT^R ",                     // RTL pushbutton
         "    插入图像    ",           // dialog title
         "   编辑图片设置   ",           // 'edit' title               [iiTIT2]
         "MIME 类型",                  // 'MIME' identifier string    [iiTYPE]
         "图片大小: %S 字节",           // Image-size template         [iiSIZE]
         "  不支持的图像格式！",         // MIME-type error             [iiUNSUP]
         "  图像文件未找到！",           // source file not found       [iiNOSUCH]
         "Audio file contains no image data.", // no image data      [iiNOIMG]
         "              \n"            // alternate 'save' button    [iiALTSAVE]
         "     保存     \n"
         "              ",
         "              \n"            // next-image pushbutton      [iiNXTIMG]
         "  下一张图片  \n"
         "              ",
      },
      {  // TiengViet
         "     Lưu      \n"             // 'save' pushbutton
         "   vào tệp    \n"
         "   hiện tại   ",
         "     Lưu      \n"            // 'save-all' pushbutton
         "   vào tất    \n"
         "  cả các tệp  ",
         "   Xóa hình  x\n"            // 'delete' pushbutton
         "     ảnh     x\n"
         "      này    x",
         "              \n"            // 'cancel' pushbutton
         "    Hủy Bỏ    \n"
         "              ",
         "Tên tệp hình ảnh",           // 'filename' textbox
         "Danh mục \n"                 // 'picType' identifier byte
         "hình ảnh",
         "Mô tả hình ảnh (tối đa 64 ký tự)", // 'description' string
         " LT^R ",                     // RTL pushbutton
         "    Chèn hình ảnh    ",      // dialog title
         "  Chỉnh sửa cài đặt hình ảnh  ", // 'edit' title           [iiTIT2]
         "MIME kiểu",                  // 'MIME' identifier string   [iiTYPE]
         "Kích cỡ hình: %S bytes",     // Image-size template        [iiSIZE]
         " Định dạng hình ảnh không được hỗ trợ!", // MIME-type error [iiUNSUP]
         " Không tìm thấy tệp hình ảnh!", // source file not found   [iiNOSUCH]
         "Audio file contains no image data.", // no image data      [iiNOIMG]
         "              \n"            // alternate 'save' button    [iiALTSAVE]
         "     Lưu      \n"
         "              ",
         "     Hình     \n"            // next-image pushbutton     [iiNXTIMG]
         "    ảnh kế    \n"
         "     Tiếp     ",
      }
   } ;

   const short pdCOLS_EN = 35,
               pdCOLS_ES = 50,
               pdCOLS_ZH = 33,
               pdCOLS_VI = 49 ;

   //* Construct the "picture type" array for the dropdown control *
   //* using the specified user interface language of the text     *
   //* descriptions in the pType[][PIC_TYPES] array.               *
   // IMPORTANT NOTE: There is a bug in the compiler (G++ 4.8.3) which tries 
   // to create the ppType[][] array on the stack even though it may be 
   // marginally too large, in which case, the memory gets trashed when passed 
   // to the Dropdown-class constructor. For this reason, we force the array 
   // onto the heap even though it is a bit inconvenient to allocate and 
   // initialize a two-dimensional array as a one-dimensional object.
   short arrayCols,        // display columns for dropdown text
         arrayWidth ;      // data bytes in each array item
   switch ( lang )
   {
      case esLang: arrayCols  = pdCOLS_ES ; arrayWidth = pdCOLS_ES + 3 ;   break ;
      case zhLang: arrayCols  = pdCOLS_ZH ; arrayWidth = pdCOLS_ZH + 11 ;  break ;
      case viLang: arrayCols  = pdCOLS_VI ; arrayWidth = pdCOLS_VI + 17 ;  break ;
      default:     arrayCols  = pdCOLS_EN ; arrayWidth = pdCOLS_EN + 1 ;   break ;
   } ;
   char* ppType = new char[PIC_TYPES * arrayWidth] ;  // dynamic allocation (see note)
   for ( short items = ZERO, ppIndex = ZERO ; items < PIC_TYPES ; ++items )
   {
      gs = pType[lang][items] ;
      if ( gs.gscols() > arrayCols )   // this should not happen
         gs.limitCols( arrayCols ) ;
      gs.padCols( arrayCols ) ;
      gs.copy( &ppType[ppIndex], gs.utfbytes() ) ;
      ppIndex += gs.utfbytes() ;
   }

   //* Calculate X offsets for: *
   //* a) Dropdown label        *
   //* b) Textbox labels        *
   short ddOffset = ZERO, ttOffset = ZERO ;
   if ( this->cfgOpt.rtl )
   {
      ddOffset = -2 ;
      ttOffset = (dlgCOLS - 5) ;
   }
   else
   {
      gs = Labels[lang][iiTypeDD] ;
      ddOffset = gs.find( L'\n' ) ;
      gs.limitChars( ddOffset ) ;
      ddOffset = -(gs.gscols() + 1) ;
   }
   

   //* Control definitions *
   InitCtrl ic[iiCONTROLS] =           // array of dialog control info
   {
   {  //* 'Save' pushbutton - - - - - - - - - - - - - - - - - - - - - iiSavePB *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(dlgROWS - 4),           // ulY:       upper left corner in Y
      pbX,                          // ulX:       upper left corner in X
      3,                            // lines:     (n/a)
      14,                           // cols:      control columns
      Labels[lang][iiSavePB],       // dispText:  
      this->cs.pn,                  // nColor:    non-focus color
      this->cs.pf,                  // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "",                           // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[iiSaveAllPB]              // nextCtrl:  link in next structure
   },
   {  //* 'SaveAll' pushbutton - - - - - - -- - - - - - - - - - -  iiSaveAllPB *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      ic[iiSavePB].ulY,             // ulY:       upper left corner in Y
      short(ic[iiSavePB].ulX + ic[iiSavePB].cols + 1), // ulX: upper left corner in X
      3,                            // lines:     (n/a)
      14,                           // cols:      control columns
      Labels[lang][iiSaveAllPB],    // dispText:  
      this->cs.pn,                  // nColor:    non-focus color
      this->cs.pf,                  // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "",                           // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[iiDeletePB]               // nextCtrl:  link in next structure
   },
   {  //* 'Delete' pushbutton - - - - - - - - - - - - - - - - - - - iiDeletePB *
      // Programmer's Note: We play a dirty trick here. If in 'Insert' mode,
      // this control is defined but inactive. The trick is that it occupies 
      // the same position as the 'Cancel' pushbutton, so it is invisible.
      // This trick only works because 'Cancel' is defined _after_ 'Delete'.
      // When in 'Edit' mode, the 'Delete' and 'Cancel' buttons occupy their own 
      // space so both are visible and active.
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      ic[iiSaveAllPB].ulY,          // ulY:       upper left corner in Y
      short(ic[iiSaveAllPB].ulX + ic[iiSaveAllPB].cols + 1), // ulX: upper left corner in X
      3,                            // lines:     (n/a)
      14,                           // cols:      control columns
      Labels[lang][iiDeletePB],     // dispText:  
      this->cs.pn,                  // nColor:    non-focus color
      this->cs.pf,                  // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "",                           // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      false,                        // active:    allow control to gain focus
      &ic[iiCancelPB]               // nextCtrl:  link in next structure
   },
   {  //* 'Cancel' pushbutton - - - - - - - - - - - - - - - - - - - iiCancelPB *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      ic[iiSaveAllPB].ulY,          // ulY:       upper left corner in Y
      short(ic[iiSaveAllPB].ulX + ic[iiSaveAllPB].cols + 1), // ulX: upper left corner in X
      3,                            // lines:     (n/a)
      14,                           // cols:      control columns
      Labels[lang][iiCancelPB],     // dispText:  
      this->cs.pn,                  // nColor:    non-focus color
      this->cs.pf,                  // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "",                           // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[iiFileTB]                 // nextCtrl:  link in next structure
   },
   {  //* 'File' Textbox - - - - - - - - - - - - - - - - - - - - - -  iiFileTB *
      dctTEXTBOX,                   // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      2,                            // ulY:       upper left corner in Y
      2,                            // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      short(dlgCOLS - 4),           // cols:      control columns
      NULL,                         // dispText:  
      this->cs.tn,                  // nColor:    non-focus color
      this->cs.tf,                  // fColor:    focus color
      tbPathName,                   // filter:    path/filename characters
      Labels[lang][iiFileTB],       // label:     
      -1,                           // labY:      
      ttOffset,                     // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[iiTypeDD]                 // nextCtrl:  link in next structure
   },
   {  //* 'Picture Description' dropdown - - - - - - - - - - - - - -  iiTypeDD *
      dctDROPDOWN,                  // type:      define a scrolling-data control
      rbtTYPES,                     // rbSubtype: (na)
      false,                        // rbSelect:  (n/a)
      short(ic[iiFileTB].ulY + 4),  // ulY:       upper left corner in Y
      15,                           // ulX:       upper left corner in X
      short(dlgROWS - 4 - 1),       // lines:     control lines
      short(arrayCols + 2),         // cols:      control columns
      ppType,                       // dispText
      this->cs.pn,                  // nColor:    non-focus border color
      this->cs.pf,                  // fColor:    focus border color
      tbPrint,                      // filter:    (n/a)
      Labels[lang][iiTypeDD],       // label:     
      ZERO,                         // labY:      offset from control's ulY
      ddOffset,                     // labX       offset from control's ulX
      ddBoxDOWN,                    // exType:    expansion direction
      PIC_TYPES,                    // scrItems:  number of elements in text/color arrays
      ZERO,                         // scrSel:    index of initial highlighted element
      this->cs.menu,                // scrColor:  single-color data display
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[iiDescTB]                 // nextCtrl:  link in next structure
   },
   {  //* 'Desc' Textbox - - - - - - - - - - - - - - - - - - - - - -  iiDescTB *
      dctTEXTBOX,                   // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(ic[iiTypeDD].ulY + 5),  // ulY:       upper left corner in Y
      ic[iiFileTB].ulX,             // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      short(dlgCOLS - 4),           // cols:      control columns
      NULL,                         // dispText:  
      this->cs.tn,                  // nColor:    non-focus color
      this->cs.tf,                  // fColor:    focus color
      tbPrint,                      // filter:    all printing characters
      Labels[lang][iiDescTB],       // label:     
      -1,                           // labY:      
      ttOffset,                     // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      NULL                          // nextCtrl:  link in next structure
   },
   {  //* 'Toggle' pushbutton - - - - - - - - - - - - - - - - - - -   iiToggPB *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(dlgROWS - 2),           // ulY:       upper left corner in Y
      1,                            // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      5,                            // cols:      control columns
      Labels[lang][iiToggPB],       // dispText:  (note that text is entered backward)
      this->cs.pn,                  // nColor:    non-focus color
      this->cs.pf,                  // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "",                           // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      NULL                          // nextCtrl:  link in next structure
   },
   } ;

   //* Dialog definition *
   InitNcDialog dInit( dlgROWS,         // number of display lines
                       dlgCOLS,         // number of display columns
                       dlgY,            // Y offset from upper-left of terminal 
                       dlgX,            // X offset from upper-left of terminal 
                       NULL,            // dialog title
                       ncltDUAL,        // border line-style
                       dColor,          // border color attribute
                       dColor,          // interior color attribute
                       ic               // pointer to list of control definitions
                     ) ;

   this->dPtr->SetDialogObscured () ;  // save parent dialog

   //* If user interface is an RTL language, enable the optional *
   //* Pushbutton for toggling display of the descriptions.      *
   if ( this->cfgOpt.rtl )
      ic[iiDescTB].nextCtrl = &ic[iiToggPB] ;

   //* If editing existing image, reconfigure controls *
   if ( editImg != false )
   {
      ic[iiFileTB].active = false ;                         // file TB inactive
      ic[iiFileTB].nColor = ic[iiFileTB].fColor | ncrATTR ; // file TB non-focus color
      ic[iiFileTB].fColor = this->cs.pn ;                   // file TB focus color
      ic[iiSavePB].dispText = Labels[lang][iiALTSAVE] ;     // label text "Save"
      ic[iiSaveAllPB].dispText = Labels[lang][iiNXTIMG] ;   // label text "next image"
      ic[iiDeletePB].active = true ;
      ic[iiCancelPB].ulX = (ic[iiDeletePB].ulX + ic[iiDeletePB].cols + 1) ;// reposition cancel PB
      //* If number of embedded images < 2, disable next-image Pushbutton.*
      if ( !(eiPtr->inUse) || (eiPtr->next == NULL) )
         ic[iiSaveAllPB].active = false ;
   }

   //* Open the dialog *
   NcDialog* dp = new NcDialog ( dInit ) ;

   //* Define a list of keycodes reserved for *
   //* access to clipboard functionality and  *
   //* enable invalid-text-input alert.       *
   this->uiDefineReservedKeys ( dp ) ;
   dp->TextboxAlert ( iiFileTB, true ) ;
   dp->TextboxAlert ( iiDescTB, true ) ;

   //* If user interface language is an RTL language, *
   //* set the internal NcDialog flag for RTL output. *
   if ( this->cfgOpt.rtl )
   {
      dp->DrawLabelsAsRTL () ;
      dp->DrawContentsAsRTL ( iiTypeDD ) ;   // Contents of Dropdown control
   }

   //* Open the dialog window *
   if ( (dp->OpenWindow()) == OK )
   {
      //* Set the dialog title *
      dp->SetDialogTitle ( Labels[lang][editImg ? iiTIT2 : iiCONTROLS], hColor ) ;

      //* Calculate position for MIME-type message *
      winPos wpMimeType( (ic[iiFileTB].ulY + 1),
                         (this->cfgOpt.rtl ? (dlgCOLS - 3) : 2) ) ;
      bool goodMimeType = true ;             // 'true' if supported image format

      //* Display stats for first saved image *
      if ( editImg )
      {
         if ( iCount > ZERO )
         {
            //* Initialize filename and description textboxes *
            dp->SetTextboxText ( iiFileTB, imgtmp[iIndex].picPath ) ;
            dp->DisplayTextboxTail ( iiFileTB ) ;
            if ( *imgtmp[iIndex].txtDesc != NULLCHAR )
               dp->SetTextboxText ( iiDescTB, imgtmp[iIndex].txtDesc ) ;
   
            //* Validate MIME type string and picture type code. *
            ceiMime ( imgtmp[iIndex], gs, this->cfgOpt.rtl ) ;
            winPos wp = dp->WriteString ( wpMimeType, Labels[lang][iiTYPE], dColor,
                                          false, this->cfgOpt.rtl ) ;
            if ( this->cfgOpt.rtl )
               wp.xpos -= gs.gscols() - 1 ;
            dp->WriteString ( wp, gs, dColor ) ;
            
            wp = { short(wpMimeType.ypos + 1), wpMimeType.xpos } ; // image size
            gsn.formatInt( imgtmp[iIndex].picSize, 11, true ) ;
            if ( this->cfgOpt.rtl )
               this->gsRevChars ( gsn ) ;
            gs.compose( Labels[lang][iiSIZE], gsn.gstr() ) ;
            dp->WriteString ( wp, gs.gstr(), dColor, false, this->cfgOpt.rtl ) ;
            dp->SetDropdownSelect ( iiTypeDD, imgtmp[iIndex].picType ) ;
         }
         else     // no image data
         {  //* Display a message and disable all controls except 'Cancel'.*
            while ( (dp->NextControl ()) != iiCancelPB ) ;
            dp->ControlActive ( iiSavePB, false ) ;
            dp->ControlActive ( iiTypeDD, false ) ;
            dp->ControlActive ( iiDescTB, false ) ;
            if ( this->cfgOpt.rtl )
               dp->ControlActive ( iiToggPB, false ) ;
            attr_t uMsgColor = dtbmNFcolor ;
            dtbmData msg( Labels[lang][iiNOIMG], &uMsgColor, false, 
                          ic[iiFileTB].cols, this->cfgOpt.rtl ) ;
            dp->DisplayTextboxMessage ( iiFileTB, msg ) ;
         }
      }
      //* Display stats for external image to be inserted *
      else if ( !editImg && *this->cfgOpt.extImage.picPath != NULLCHAR )
      {
         gs = this->cfgOpt.extImage.picPath ;
         goodMimeType = this->ciiMime ( gs, this->cfgOpt.extImage ) ;
         dp->SetTextboxText ( iiFileTB, this->cfgOpt.extImage.picPath ) ;
         dp->DisplayTextboxTail ( iiFileTB ) ;

         //* Format and display MIME-type string. Note that _label_ is *
         //* either LTR or RTL, but MIME-type string is always LTR.    *
         winPos wp = dp->WriteString ( wpMimeType, Labels[lang][iiTYPE], dColor,
                                       false, this->cfgOpt.rtl ) ;
         if ( this->cfgOpt.rtl )
         {
            gs.compose( "\"%s\":", this->cfgOpt.extImage.mimType ) ;
            wp.xpos -= gs.gscols() - 1 ;
            dp->WriteString ( wp.ypos, wp.xpos--, gs, dColor ) ;
         }
         else
         {
            gs.compose( ":\"%s\"", this->cfgOpt.extImage.mimType ) ;
            wp = dp->WriteString ( wp, gs, dColor ) ;
         }

         if ( ! goodMimeType )      // if target is not a supported type
         {
            dp->WriteString ( wp, 
                Labels[lang][(this->cfgOpt.extImage.picSize > ZERO) ? iiUNSUP : iiNOSUCH], 
                hColor, false, this->cfgOpt.rtl ) ;
         }
      }

      dp->RefreshWin () ;                 // make everything visible

      //* Interact with the user *
      uiInfo   Info ;                     // user interface data returned here
      short    icIndex = iiSavePB ;       // index of control with focus
      bool     done = false ;             // loop control
      while ( ! done )
      {
         if ( ic[icIndex].type == dctPUSHBUTTON )
         {
            if ( Info.viaHotkey )
               Info.HotData2Primary () ;
            else
               icIndex = dp->EditPushbutton ( Info ) ;
            if ( Info.dataMod != false )
            {
               //* Embedded image options *
               if ( editImg != false &&
                    ((Info.ctrlIndex == iiSavePB) || (Info.ctrlIndex == iiSaveAllPB)
                       || (Info.ctrlIndex == iiDeletePB)) )
               {
                  //* Save embedded-image edits *
                  if ( Info.ctrlIndex == iiSavePB )
                  {
                     //* For current image, get picture type, pic-type   *
                     //* description and user's image description string.*
                     imgtmp[iIndex].picType = dp->GetDropdownSelect ( iiTypeDD ) ;
                     gs = &ppType[arrayWidth * (imgtmp[iIndex].picType)] ;
                     gs.strip();
                     gs.copy( imgtmp[iIndex].picExpl, gsMAXBYTES ) ;
                     dp->GetTextboxText ( iiDescTB, gs ) ;
                     gs.copy( imgtmp[iIndex].txtDesc, gsMAXBYTES ) ;

                     #if DEBUG_CII != 0   // display local array
                     dp->SetDialogObscured () ;
                     gString gs( "DEBUG_CII\n" ) ;
                     winPos wp( 10, 2 ) ;
                     short j = ZERO ;
                     wp = this->dPtr->WriteParagraph ( wp, gs, dColor, true ) ;
                     eiPtr = &this->tData.sf[fIndex].sfTag.ePic ;
                     do
                     {
                        gs.compose( "eiPtr:%p next:%p\n"
                                    "  %s (%hhd)\n",
                                    eiPtr, eiPtr->next, eiPtr->mp3img.picPath, &delPix[j++] ) ;
                        wp = this->dPtr->WriteParagraph ( wp, gs, dColor, true ) ;
                        eiPtr = eiPtr->next ;
                     }
                     while ( eiPtr != NULL ) ;
                     ++wp.ypos ;
                     #endif   // DEBUG_CII

                     //* Copy the array of temp records back to *
                     //* the linked list of live data records.  *
                     if ( iCount > ZERO )
                     {
                        eiPtr = &this->tData.sf[fIndex].sfTag.ePic ;
                        for ( short i = ZERO ; eiPtr != NULL ; ++i )
                        {  //* If the local copy has been marked for deletion,*
                           //* reset its data members. This guarantees that   *
                           //* it will be seen as modified.                   *
                           if ( delPix[i] != false )
                              imgtmp[i].reset() ;

                           //* Test for changes in picType, and txtDesc; and   *
                           //* set edit-pending flag if differences. Note that *
                           //* mimType may also have changed, but we consider  *
                           //* that as a 'correction' rather than as a         *
                           //* modification. Note also that picExpl is not     *
                           //* tested because it is not part of the record     *
                           //* which is written to target.                     *
                           gs = imgtmp[i].txtDesc ;
                           if ( (imgtmp[i].picType != eiPtr->mp3img.picType) ||
                                ((gs.compare( eiPtr->mp3img.txtDesc )) != ZERO) ||
                                (imgtmp[i].picPath[0] == NULLCHAR) )
                              eiPtr->imMod = this->tData.sf[fIndex].sfTag.tfMod = true ;
                           else
                              eiPtr->imMod = false ;
                           if ( eiPtr->imMod != false )
                              eiPtr->mp3img = imgtmp[i] ;

                           eiPtr = eiPtr->next ;
                        }     // for(;;)

                        //* Start at the  tail of the linked list and *
                        //* walk backward, removing the records which *
                        //* have been marked for deletion.            *
                        EmbeddedImage *delPtr ;
                        eiPtr = &this->tData.sf[fIndex].sfTag.ePic ;
                        while ( eiPtr->next != NULL )
                           eiPtr = eiPtr->next ;
                        while ( eiPtr != NULL )
                        {
                           delPtr = eiPtr ;
                           eiPtr = eiPtr->prev ;
                           if ( delPtr->mp3img.picPath[0] == NULLCHAR )
                           {
                              this->tData.sf[fIndex].sfTag.delPic( delPtr ) ;
                           }
                        }
                     }

                     #if DEBUG_CII != 0
                     eiPtr = &this->tData.sf[fIndex].sfTag.ePic ;
                     do
                     {
                        gs.compose( "eiPtr:%p  next:%p\n"
                                    "  %s\n",
                                    eiPtr,  eiPtr->next, eiPtr->mp3img.picPath ) ;
                        wp = this->dPtr->WriteParagraph ( wp, gs, dColor, true ) ;
                        eiPtr = eiPtr->next ;
                     }
                     while ( eiPtr != NULL ) ;
                     nckPause();
                     dp->RefreshWin () ;
                     #endif   // DEBUG_CII

                     done = true ;
                  }

                  //* "Next Image" : Display stats for next embedded image *
                  //* Note that we arrive here only if more than one image.*
                  else if ( Info.ctrlIndex == iiSaveAllPB )
                  {
                     //* For current image, save picture type, pic-type  *
                     //* description and user's image description string.*
                     imgtmp[iIndex].picType = dp->GetDropdownSelect ( iiTypeDD ) ;
                     gs = &ppType[arrayWidth * (imgtmp[iIndex].picType)] ;
                     gs.strip();
                     gs.copy( imgtmp[iIndex].picExpl, gsMAXBYTES ) ;
                     dp->GetTextboxText ( iiDescTB, gs ) ;
                     gs.copy( imgtmp[iIndex].txtDesc, gsMAXBYTES ) ;

                     //* Format and display next record *
                     if ( ++iIndex >= iCount )
                        iIndex = ZERO ;

                     //* Initialize filename and description textboxes *
                     dp->SetTextboxText ( iiFileTB, imgtmp[iIndex].picPath ) ;
                     if ( delPix[iIndex] == false )
                        dp->DisplayTextboxTail ( iiFileTB ) ;
                     else     // image has been provisionally deleted
                     {
                        gs = imgtmp[iIndex].picPath ;
                        gs.padCols( ic[iiFileTB].cols ) ;
                        gs.copy( dData.textData, gsMAXCHARS ) ;
                        dp->DisplayTextboxMessage ( iiFileTB, dData ) ;
                     }
                     dp->SetTextboxText ( iiDescTB, imgtmp[iIndex].txtDesc ) ;

                     //* Validate MIME type string and picture type code. *
                     ceiMime ( imgtmp[iIndex], gs, this->cfgOpt.rtl ) ;
                     winPos wp = dp->WriteString ( wpMimeType, Labels[lang][iiTYPE], 
                                                   dColor, false, this->cfgOpt.rtl ) ;
                     if ( this->cfgOpt.rtl )
                        wp.xpos -= gs.gscols() - 1 ;
                     dp->WriteString ( wp, gs, dColor ) ;
                     //* Display image size *
                     wp = { short(wpMimeType.ypos + 1), wpMimeType.xpos } ;
                     gsn.formatInt( imgtmp[iIndex].picSize, 11, true ) ;
                     if ( this->cfgOpt.rtl )
                        this->gsRevChars ( gsn ) ;
                     gs.compose( Labels[lang][iiSIZE], gsn.gstr() ) ;
                     dp->WriteString ( wp, gs.gstr(), dColor, false, this->cfgOpt.rtl ) ;
                     dp->SetDropdownSelect ( iiTypeDD, imgtmp[iIndex].picType ) ;
                  }

                  //* Delete the currently-indexed image *
                  else if ( Info.ctrlIndex == iiDeletePB )
                  {
                     delPix[iIndex] = true ;      // mark as deleted
                     //* Grey-out the filename field *
                     dp->GetTextboxText ( iiFileTB, gs ) ;
                     gs.padCols( ic[iiFileTB].cols ) ;
                     gs.copy( dData.textData, gsMAXCHARS ) ;
                     dp->DisplayTextboxMessage ( iiFileTB, dData ) ;
                  }
               }

               //* Insert-image save *
               else if ( (Info.ctrlIndex == iiSavePB) || (Info.ctrlIndex == iiSaveAllPB) )
               {
                  //* If we have a validated image file *
                  if ( *this->cfgOpt.extImage.picPath != NULLCHAR )
                  {  //* Get picture type and type description *
                     this->cfgOpt.extImage.picType = dp->GetDropdownSelect ( iiTypeDD ) ;
                     gs = &ppType[arrayWidth * (this->cfgOpt.extImage.picType)] ;
                     gs.strip();
                     gs.copy( this->cfgOpt.extImage.picExpl, gsMAXBYTES ) ;
                     //* Get image description *
                     dp->GetTextboxText ( iiDescTB, gs ) ;
                     gs.copy( this->cfgOpt.extImage.txtDesc, gsMAXBYTES ) ;
                  }

                  //* Copy the image info to the target record(s) *
                  EmbeddedImage* newPtr = NULL ;
                  for ( short indx = (Info.ctrlIndex == iiSavePB ? fIndex : ZERO ) ;
                        indx < this->tData.sfCount ; ++indx )
                  {
                     newPtr = this->tData.sf[indx].sfTag.addPic() ;
                     newPtr->mp3img = this->cfgOpt.extImage ;
                     newPtr->inUse = newPtr->imMod = true ;
                     this->tData.sf[indx].sfTag.tfMod = true ; // edits-pending flag
                     if ( Info.ctrlIndex == iiSavePB )
                        break ;
                  }
                  done = true ;
               }
               else if ( Info.ctrlIndex == iiToggPB )
               {  //* Note: We will arrive here ONLY if UI language is RTL. *
                  //* Toggle the NcDialog's internal flag.                  *
                  rtlFields = rtlFields ? false : true ;
                  dp->DrawContentsAsRTL ( iiDescTB, rtlFields ) ;
                  dp->RefreshWin () ;
               }
               else  // (Info.ctrlIndex == iiCancelPB)
               {
                  done = true ;
               }
            }
         }

         else if ( ic[icIndex].type == dctTEXTBOX )
         {
            Info.viaHotkey = false ;      // ignore hotkey data
            icIndex = dp->EditTextbox ( Info ) ;
            if ( Info.dataMod != false )
            {
               if ( Info.ctrlIndex == iiFileTB )
               {  // Note: We will arrive here only during 'Insert', not 'Edit'
                  dp->ClearLine ( wpMimeType.ypos, false, wpMimeType.xpos, this->cfgOpt.rtl ) ;
                  dp->GetTextboxText ( iiFileTB, gs ) ;
                  goodMimeType = this->ciiMime ( gs, this->cfgOpt.extImage ) ;

                  //* Format and display MIME-type string. Note that _label_ is *
                  //* either LTR or RTL, but MIME-type string is always LTR.    *
                  winPos wp = dp->WriteString ( wpMimeType, Labels[lang][iiTYPE], dColor,
                                                false, this->cfgOpt.rtl ) ;
                  if ( this->cfgOpt.rtl )
                  {
                     gs.compose( "\"%s\":", this->cfgOpt.extImage.mimType ) ;
                     wp.xpos -= gs.gscols() - 1 ;
                     dp->WriteString ( wp.ypos, wp.xpos--, gs, dColor, true ) ;
                  }
                  else
                  {
                     gs.compose( ":\"%s\"", this->cfgOpt.extImage.mimType ) ;
                     wp = dp->WriteString ( wp, gs, dColor, true ) ;
                  }
                  if ( goodMimeType )
                  {  //* Display the full filespec *
                     dp->NextControl () ;
                     dp->NextControl () ;
                     dp->SetTextboxText ( iiFileTB, this->cfgOpt.extImage.picPath ) ;
                     dp->DisplayTextboxTail ( iiFileTB ) ;
                     Info.keyIn = nckSTAB ;
                  }
                  else                       // if target is not a supported type
                     dp->WriteString ( wp,  
                         Labels[lang][(this->cfgOpt.extImage.picSize > ZERO) ? iiUNSUP : iiNOSUCH], 
                         hColor, true, this->cfgOpt.rtl ) ;
               }
               else  // (Info.ctrlIndex == iiDescTB)
               {  //* For MP3 files, silently limit message length *
                  if ( this->tData.sf[fIndex].sfType == mftMP3 )
                  {
                     dp->GetTextboxText ( iiDescTB, gs ) ;
                     if ( (gs.gschars()) > imgMAX_DESC )
                     {
                        while ( (gs.gschars()) > imgMAX_DESC )
                           gs.limitChars( gs.gschars() - 2 ) ;
                        dp->PrevControl () ;
                        dp->SetTextboxText ( iiDescTB, gs ) ;
                        dp->NextControl () ;
                     }
                  }
               }
            }
         }

         else if ( ic[icIndex].type == dctDROPDOWN )
         {
            icIndex = dp->EditDropdown ( Info ) ;
         }

         //* Move focus to appropriate control *
         if ( ! done && ! Info.viaHotkey )
         {
            if ( Info.keyIn == nckSTAB )
               icIndex = dp->PrevControl () ; 
            else
               icIndex = dp->NextControl () ;
         }
      }  // while()
   }
   if ( dp != NULL )                         // close the window
      delete ( dp ) ;
   delete [] ppType ;                        // release dynamic allocations

   this->dPtr->RefreshWin () ;               // restore parent dialog
   this->UpdateFilenameWindow () ;           // update the edits-pending indicators

   #undef DEBUG_CII

}  //* End Cmd_InsertImage() *

//*************************
//*      ciiMime          *
//*************************
//******************************************************************************
//* Called only by Cmd_InsertImage().                                          *
//* Given the filespec of an image file, initialize the caller's image stats.  *
//*                                                                            *
//* The id3v2.3 standard defines three(3) MIME-type strings:                   *
//*       "image/jpeg"      for JPEG images                                    *
//*       "image/png"       for PNG images                                     *
//*       "image/"          for unknown or unsupported MIME type               *
//*                                                                            *
//* Input  : imgPath  : (by reference) contains image filespec                 *
//*          extImage : (by reference) receives image information              *
//*                     1) object is reset on entry                            *
//*                     2) if target exists:                                   *
//*                        a) 'picPath' member gets full filespec of target    *
//*                        b) 'mimType' member receives the MIME-type string   *
//*                           If target's filename extension does not describe *
//*                           a supported image type, then a 'mimType' gets a  *
//*                           a default string and a 'false' status is returned*
//*                        c) 'picSize' member received size of image in bytes *
//*                     3) if target not found:                                *
//*                        a) 'picPath' member gets a copy of 'imgPath'        *
//*                        b) 'mimType' member gets default value              *
//*                        c) all other members uninitialized                  *
//*                        d) a 'false' status is returned                     *
//*                                                                            *
//* Returns: 'true'  if supported JPEG or PNG image filename extension         *
//*          'false' if unknown or unsupported filename extension              *
//******************************************************************************

bool Taggit::ciiMime ( const gString& imgPath, id3v2_image& extImage )
{
   gString gs = imgPath ;
   bool status = false ;      // return value - prepare for the worst

   extImage.reset() ;         // reset the receiving object

// UNDER CONSTRUCTION - IF picPath INDICATES A URL, THEN TEST FOR IT.
// ref: "--> + [relative-or-absolute-path]"
// NOTE: If a relative path, keep it relative on the assumption that it is relative
//       to the audio file.
   //* If target exists, get its full filespec *
   if ( (this->Realpath ( gs )) != false )
   {
      status = true ;                              // hope for the best
      gs.copy( extImage.picPath, MAX_PATH ) ;      // full filespec

      tnFName fStats ;                             // target file stats
      this->GetFileStats ( gs, fStats ) ;
      this->cfgOpt.extImage.picSize = fStats.fBytes ;

      //* For OGG images, we need the size of the _encoded_ data. *
      //* This calculation indicates that each three(3) binary    *
      //* source bytes are encoded as four(4) Base64 pseudotext   *
      //* bytes. Target size must be an even multiple of 4.       *
      this->cfgOpt.extImage.picEncSize = this->cfgOpt.extImage.picSize / 3 * 4 ;
      if ( (this->cfgOpt.extImage.picSize / 3) > ZERO )
         this->cfgOpt.extImage.picEncSize += 4 ;

      //* MIME-type string *
      if ( (imgPath.find( ".jpg" ) >= ZERO) || (imgPath.find( ".jpeg" ) >= ZERO) )
         gs = mimeJPG ;
      else if ( (imgPath.find( ".png" )) >= ZERO )
         gs = mimePNG ;
      else
      {
         gs = mimeUnk ;
         status = false ;
      }
      gs.copy( extImage.mimType, gsMAXBYTES ) ;
   }
   else
   {
      imgPath.copy( extImage.picPath, MAX_PATH ) ;
      gs = mimeUnk ;
      gs.copy( extImage.mimType, gsMAXBYTES ) ;
   }
   return status ;

}  //* End ciiMime() *

//*************************
//*      ceiMime          *
//*************************
//******************************************************************************
//* Non-member method.                                                         *
//* Called only by Cmd_InsertImage(edit).                                      *
//* Validate and format data for an image under edit.                          *
//*                                                                            *
//* Input  : img   : (by reference) image information                          *
//*                    (will be updated as necessary)                          *
//*                  a) 'mimType' member receives the MIME-type string         *
//*                     If target's filename extension does not describe a     *
//*                     supported image type, then a 'mimType' gets a          *
//*                     default string                                         *
//*                  b) picType code is range checked and reset if necessary   *
//*                  c) all other members unchanged                            *
//*          gsmt  : (by reference) receives formatted MIME-type display string*
//*          rtl   : 'false' LTR MIME type                                     *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************
//* Note: Because the image file and associated data have been validated at a  *
//* higher level, this method is much simpler than the ciiMime() method above. *
//*                                                                            *
//******************************************************************************

static void ceiMime ( id3v2_image& img, gString& gsmt, bool rtl )
{
   gString imgPath( img.picPath ), gs ;

   //* Set MIME-type string *
   if ( (imgPath.find( ".jpg" ) >= ZERO) || (imgPath.find( ".jpeg" ) >= ZERO) )
      gs = mimeJPG ;
   else if ( (imgPath.find( ".png" )) >= ZERO )
      gs = mimePNG ;
   else
      gs = mimeUnk ;
   gs.copy( img.mimType, gsMAXBYTES ) ;
   if ( rtl )
      gsmt.compose( "\"%S\":", gs.gstr() ) ;
   else
      gsmt.compose( ":\"%S\"", gs.gstr() ) ;

   //* Range check the picType code *
   if ( img.picType > PIC_TYPES )
      img.picType = ZERO ;   // 'other'

}  //* End ceiMime() *

#if 0    // CURRENTLY UNUSED
//*************************
//*      togCallback      *
//*************************
//******************************************************************************
//* This is a callback method for toggling LTR/RTL input and display for       *
//* controls in the following subdialog methods:                               *
//*  1) Cmd_InsertImage()                                                      *
//*  2) uiEditFilename()                                                       *
//*  3) uiEditMetadata()                                                       *
//*                                                                            *
//* Input  : currIndex: index of control that currently has focus              *
//*          wkey     : user's key input data                                  *
//*          firstTime: the EstablishCallback() method calls this method once  *
//*                     with firstTime==true, to perform any required          *
//*                     initialization. Subsequently, the NcDialog class       *
//*                     always calls with firstTime==false.                    *
//*                                                                            *
//* Returns: OK                                                                *
//******************************************************************************
//* Notes:                                                                     *
//* This method makes intimate assumptions about the target dialog:            *
//*  a) The target control(s) are listed in togCtrl[] array.                   *
//*  b) There are at least two active controls in the dialog.                  *
//*  c) The NcDialog internal RTL flag is initially 'false'.                   *
//*                                                                            *
//* Programmer's Note: If an expanded control is under edit (Menuwin or        *
//* Dropdown), then updating the display may trash the expanded control's      *
//* display. This cannot be helped because at this level, we don't have enough *
//* information to prevent it. The good news is that it's unlikely a user will *
//* try to toggle the LTR/RTL data while in an expanded control.               *
//******************************************************************************

short togCallback ( const short currIndex, const wkeyCode wkey, bool firstTime )
{
   static bool rtl = false ;

   if ( firstTime != false )
      rtl = false ;

   if ( wkey.type == wktFUNKEY && wkey.key == nckC_R )
   {
      //* Toggle our internal tracking flag *
      rtl = rtl ? false : true ;

      short icIndex = currIndex ;
      for ( short indx = ZERO ; togCtrl[indx] != (-1) ; ++indx )
      {
         //* Move the focus if necessary *
         if ( togCtrl[indx] == icIndex )
            icIndex = togDp->NextControl () ;

         //* Toggle the NcDialog flag *
         togDp->DrawContentsAsRTL ( togCtrl[indx], rtl ) ;
      }
      while ( icIndex != currIndex )      // return focus to control under edit
         icIndex = togDp->NextControl () ;
      togDp->RefreshWin () ;              // make changes visible
   }
   return OK ;

}  //* End togCallback()
#endif   // CURRENTLY UNUSED

//*************************
//*     Cmd_Playback      *
//*************************
//******************************************************************************
//* Invoke the default media player for the specified file.                    *
//*                                                                            *
//*                                                                            *
//* Input  : fIndex  : index of source file                                    *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************
//* Notes:                                                                     *
//* ------                                                                     *
//* -- With any luck, the user will have set the same default media player for *
//*    all supported audio filetypes. If not, then for multiple audio filetypes*
//*    multiple media players may be invoked.                                  *
//* -- It is also hoped that the user has configured the default media player  *
//*    to open a single instance regardless of the number of invocations.      *
//*    If not, it may make us look bad although it has nothing to do with us.  *
//* -- If the child process is unable to open the target media player, then    *
//*    we may not be able to detect the error, which will make us look stupid, *
//*    but again, it has nothing to do with us.                                *
//* -- Note that the media player may spew startup information all over our    *
//*    terminal window if we allow it, so the stdout and stderr streams of the *
//*    spawned process are redirected to the bit bucket.                       *
//*                                                                            *
//******************************************************************************

void Taggit::Cmd_Playback ( short fIndex )
{
   if ( (this->LaunchDefaultApplication ( this->tData.sf[fIndex].sfPath )) < ZERO )
   {  //* If the child process was not launched. (This is unlikely.) *
      /* A launch error is not reported or handled at this time. */
   }
}  //* End Cmd_Playback() *

//*************************
//*                       *
//*************************
//******************************************************************************
//*                                                                            *
//*                                                                            *
//*                                                                            *
//* Input  :                                                                   *
//*                                                                            *
//* Returns:                                                                   *
//******************************************************************************

