//******************************************************************************
//* File       : TagDialog.cpp                                                 *
//* Author     : Mahlon R. Smith                                               *
//*              Copyright (c) 2016-2020 Mahlon R. Smith, The Software Samurai *
//*                  GNU GPL copyright notice located in Taggit.hpp            *
//* Date       : 17-Oct-2020                                                   *
//* Version    : (see AppVersion string)                                       *
//*                                                                            *
//* Description: This module contains the application dialog definition and    *
//* user interface loop.                                                       *
//*                                                                            *
//*                                                                            *
//******************************************************************************
//* Version History (see Taggit.cpp).                                          *
//******************************************************************************

//****************
//* Header Files *
//****************
#include "Taggit.hpp"         // Taggit-class definitions and data
                              // plus general definitions and NcDialogAPI definition
#include "TagDialog.hpp"      // Text data displayed in the dialog windows

//**********************
//* Non-member methods *
//**********************
MenuCode MenuItem2MenuCode ( short menuIndex, short menuMember ) ;

//**************
//* Local data *
//**************

#define SIMULATED_DATA (0)             // FOR DEBUGGING ONLY
#define DIALOG_DEBUG   (0)             // FOR DEBUGGING ONLY


//* Indicator character for scrolling *
static const wchar_t sortARROW = 0x25BC ; // '▼'

//* Used to enclose the Menu Bar *
static const wchar_t HalfBlockRight = 0x2590,
                     HalfBlockLeft  = 0x258C,
                     HalfBlockLower = 0x2584,
                     HalfBlockUpper = 0x2580,
                     FullBlock      = 0x2588 ;

//* Used to underline full-width fields for the item with focus.*
//* This is the block character, HalfBlockUpper.                *
static const wchar_t focusLine[] =
   L"\u2580\u2580"
    "\u2580\u2580\u2580\u2580\u2580\u2580\u2580\u2580\u2580\u2580"
    "\u2580\u2580\u2580\u2580\u2580\u2580\u2580\u2580\u2580\u2580"
    "\u2580\u2580\u2580\u2580\u2580\u2580\u2580\u2580\u2580\u2580" ;

//* Placeholder for display of empty fields
static const wchar_t* NoFieldContents = L"_" ;

#if 0    // CURRENTLY UNUSED
//* Generic CTRL+R toggle callback *
extern short togCallback ( const short currIndex, const wkeyCode wkey, 
                           bool firstTime = false ) ;
       extern NcDialog* togDp ;                 // callback access to dialog
       extern short togCtrl[MAX_DIALOG_CONTROLS] ;// target control indices
#endif   // CURRENTLY UNUSED

//* Controls for uiEditMetadata() method *
enum emControls : short { em_editTB, em_donePB, em_cancelPB, em_undoPB, 
                          em_fxPB, em_nextPB, em_prevPB, em_upPB, em_downPB, 
                          em_rtlPB, em_controlsDEFINED } ;


//*************************
//*    InitColorScheme    *
//*************************
//******************************************************************************
//* The application's color scheme is defined in this method.                  *
//*                                                                            *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void Taggit::InitColorScheme ( void )
{
   //* Application's default color scheme. *
   this->cs.bb = nc.cyR ;              // application borders and backgrounds
   this->cs.sd = nc.blR ;              // sub-dialog color
   this->cs.sb = nc.gybl | ncbATTR ;   // sub-dialog Bold color
   this->cs.em = nc.brbl ;             // sub-dialog Emphasis color
   this->cs.dm = nc.gybl ;             // sub-dialog Dim color
   this->cs.mn = nc.blR ;              // menu border (without focus) color
   this->cs.mf = nc.cyR ;              // menu border (with focus) color
   this->cs.pn = nc.gyR ;              // pushbutton (without focus) color
   this->cs.pf = nc.reG ;              // pushbutton (with focus) color
   this->cs.tn = nc.bw ;               // text box (without focus) color
   this->cs.tf = nc.gr ;               // text box (with focus) color
   this->cs.menu[0] = attrDFLT ;       // menu interior color
   this->cs.menu[1] = this->cs.mn ;
   this->cs.title   = this->cs.em ;    // application title color
   this->cs.saved   = nc.grcy ;        // data-saved indicator color
   this->cs.unsaved = nc.recy ;        // data-NOT-saved indicator color

   //* Adjust for non-default color schemes according to user's config option. *
   if ( this->cs.scheme != ncbcCOLORS )
   {
      switch ( this->cs.scheme )
      {
         case ncbcBK:      // black
            this->cs.bb = nc.bk ;            // application borders and backgrounds
            this->cs.sd = nc.bk ;            // sub-dialog color
            this->cs.sb = nc.bk & ~ncbATTR ; // sub-dialog Bold color
            this->cs.em = nc.brbk ;          // sub-dialog Emphasis color
            this->cs.dm = nc.gybk ;          // sub-dialog Dim color
            this->cs.mn = nc.cyR ;           // menu border (without focus) color
            this->cs.mf = nc.bk ;            // menu border (with focus) color
            this->cs.pn = nc.gyR ;           // pushbutton (without focus) color
            this->cs.pf = nc.reG ;           // pushbutton (with focus) color
            this->cs.tn = nc.bw ;            // text box (without focus) color
            this->cs.tf = nc.bl ;            // text box (with focus) color
            this->cs.menu[1] = this->cs.mf ; // menu interior
            this->cs.title   = this->cs.em ; // application title color
            this->cs.saved   = nc.grbk ;     // data-saved indicator color
            this->cs.unsaved = nc.rebk ;     // data-NOT-saved indicator color
            break ;
         case ncbcRE:      // red
            this->cs.bb = nc.reR ;           // application borders and backgrounds
            this->cs.sd = nc.reR ;           // sub-dialog color
            this->cs.sb = nc.gyre | ncbATTR ;// sub-dialog Bold color
            this->cs.em = nc.brre ;          // sub-dialog Emphasis color
            this->cs.dm = nc.gyre ;          // sub-dialog Dim color
            this->cs.mn = nc.re ;            // menu border (without focus) color
            this->cs.mf = nc.reR ;           // menu border (with focus) color
            this->cs.pn = nc.gyR ;           // pushbutton (without focus) color
            this->cs.pf = nc.maG ;           // pushbutton (with focus) color
            this->cs.tn = nc.bw ;            // text box (without focus) color
            this->cs.tf = nc.bl ;            // text box (with focus) color
            this->cs.menu[1] = this->cs.mf ; // menu interior
            this->cs.title   = this->cs.em ; // application title color
            this->cs.saved   = nc.grre ;     // data-saved indicator color
            this->cs.unsaved = nc.mare ;     // data-NOT-saved indicator color
            break ;
         case ncbcGR:      // green
            this->cs.bb = nc.grR ;           // application borders and backgrounds
            this->cs.sd = nc.grR ;           // sub-dialog color
            this->cs.sb = nc.gygr | ncbATTR ;// sub-dialog Bold color
            this->cs.em = nc.brgr ;          // sub-dialog Emphasis color
            this->cs.dm = nc.gygr ;          // sub-dialog Dim color
            this->cs.mn = nc.gr ;            // menu border (without focus) color
            this->cs.mf = nc.grR ;           // menu border (with focus) color
            this->cs.pn = nc.gyR ;           // pushbutton (without focus) color
            this->cs.pf = nc.reG ;           // pushbutton (with focus) color
            this->cs.tn = nc.bw ;            // text box (without focus) color
            this->cs.tf = nc.bl ;            // text box (with focus) color
            this->cs.menu[1] = this->cs.mf ; // menu interior
            this->cs.title   = this->cs.em ; // application title color
            this->cs.saved   = nc.brgr ;     // data-saved indicator color
            this->cs.unsaved = nc.regr ;     // data-NOT-saved indicator color
            break ;
         case ncbcBR:      // brown
            this->cs.bb = nc.brR ;           // application borders and backgrounds
            this->cs.sd = nc.brR ;           // sub-dialog color
            this->cs.sb = nc.gybr | ncbATTR ;// sub-dialog Bold color
            this->cs.em = nc.cybr ;          // sub-dialog Emphasis color
            this->cs.dm = nc.gybr ;          // sub-dialog Dim color
            this->cs.mn = nc.br ;            // menu border (without focus) color
            this->cs.mf = nc.brR ;           // menu border (with focus) color
            this->cs.pn = nc.gyR ;           // pushbutton (without focus) color
            this->cs.pf = nc.maG ;           // pushbutton (with focus) color
            this->cs.tn = nc.bw ;            // text box (without focus) color
            this->cs.tf = nc.bl ;            // text box (with focus) color
            this->cs.menu[1] = this->cs.mf ; // menu interior
            this->cs.title   = this->cs.em ; // application title color
            this->cs.saved   = nc.grbr ;     // data-saved indicator color
            this->cs.unsaved = nc.rebr ;     // data-NOT-saved indicator color
            break ;
         case ncbcBL:      // blue
            this->cs.bb = nc.blR ;           // application borders and backgrounds
            this->cs.sd = nc.blR ;           // sub-dialog color
            this->cs.sb = nc.gybl | ncbATTR ;// sub-dialog Bold color
            this->cs.em = nc.brbl ;          // sub-dialog Emphasis color
            this->cs.dm = nc.gybl ;          // sub-dialog Dim color
            this->cs.mn = nc.bl ;            // menu border (without focus) color
            this->cs.mf = nc.blR ;           // menu border (with focus) color
            this->cs.pn = nc.gyR ;           // pushbutton (without focus) color
            this->cs.pf = nc.reG ;           // pushbutton (with focus) color
            this->cs.tn = nc.bw ;            // text box (without focus) color
            this->cs.tf = nc.gr ;            // text box (with focus) color
            this->cs.menu[1] = this->cs.mf ; // menu interior
            this->cs.title   = this->cs.em ; // application title color
            this->cs.saved   = nc.grbl ;     // data-saved indicator color
            this->cs.unsaved = nc.rebl ;     // data-NOT-saved indicator color
            break ;
         case ncbcMA:      // magenta
            this->cs.bb = nc.maR ;           // application borders and backgrounds
            this->cs.sd = nc.maR ;           // sub-dialog color
            this->cs.sb = nc.gyma | ncbATTR ;// sub-dialog Bold color
            this->cs.em = nc.brma ;          // sub-dialog Emphasis color
            this->cs.dm = nc.gyma ;          // sub-dialog Dim color
            this->cs.mn = nc.ma ;            // menu border (without focus) color
            this->cs.mf = nc.maR ;           // menu border (with focus) color
            this->cs.pn = nc.gyR ;           // pushbutton (without focus) color
            this->cs.pf = nc.reG ;           // pushbutton (with focus) color
            this->cs.tn = nc.bw ;            // text box (without focus) color
            this->cs.tf = nc.bl ;            // text box (with focus) color
            this->cs.menu[1] = this->cs.mf ; // menu interior
            this->cs.title   = this->cs.em ; // application title color
            this->cs.saved   = nc.grma ;     // data-saved indicator color
            this->cs.unsaved = nc.rema ;     // data-NOT-saved indicator color
            break ;
         case ncbcCY:      // cyan
            this->cs.bb = nc.cyR ;           // application borders and backgrounds
            this->cs.sd = nc.cyR ;           // sub-dialog color
            this->cs.sb = nc.gycy | ncbATTR ;// sub-dialog Bold color
            this->cs.em = nc.brcy ;          // sub-dialog Emphasis color
            this->cs.dm = nc.bkcy ;          // sub-dialog Dim color
            this->cs.mn = nc.cy ;            // menu border (without focus) color
            this->cs.mf = nc.cyR ;           // menu border (with focus) color
            this->cs.pn = nc.gyR ;           // pushbutton (without focus) color
            this->cs.pf = nc.reG ;           // pushbutton (with focus) color
            this->cs.tn = nc.bw ;            // text box (without focus) color
            this->cs.tf = nc.bl ;            // text box (with focus) color
            this->cs.menu[1] = this->cs.mf ; // menu interior
            this->cs.title   = this->cs.em ; // application title color
            this->cs.saved   = nc.grcy ;     // data-saved indicator color
            this->cs.unsaved = nc.recy ;     // data-NOT-saved indicator color
            break ;
         case ncbcGY:      // gray
            this->cs.bb = nc.gyR ;           // application borders and backgrounds
            this->cs.sd = nc.gyR ;           // sub-dialog color
            this->cs.sb = nc.bkgy ;          // sub-dialog Bold color
            this->cs.em = nc.blgy ;          // sub-dialog Emphasis color
            this->cs.dm = nc.bkgy ;          // sub-dialog Dim color
            this->cs.mn = nc.cy ;            // menu border (without focus) color
            this->cs.mf = nc.cyR ;           // menu border (with focus) color
            this->cs.pn = nc.bw ;            // pushbutton (without focus) color
            this->cs.pf = nc.reG ;           // pushbutton (with focus) color
            this->cs.tn = nc.bw ;            // text box (without focus) color
            this->cs.tf = nc.bl ;            // text box (with focus) color
            this->cs.menu[1] = this->cs.mf ; // menu interior
            this->cs.title   = this->cs.em ; // application title color
            this->cs.saved   = nc.grgy ;     // data-saved indicator color
            this->cs.unsaved = nc.regy ;     // data-NOT-saved indicator color
            break ;
         default:          // default values handled above
            break ;
      }
   }     // if(this->cs.scheme!=ncbcCOLORS)

}  //* End InitColorScheme() *

//*************************
//*    InitAppControls    *
//*************************
//******************************************************************************
//* The application's global control objects are defined in this method.       *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void Taggit::InitAppControls ( void )
{
   //* If the control definitions have not been instantiated, do it now. *
   if ( this->ic == NULL )
   {
      AppLang lang = this->cfgOpt.appLanguage ;    // language index
      const char* Saveall_Text = Sapb_Text[lang] ;
      const char* Fmw_Text = (const char*)Fmw_Text_EN ;
      const char* Emw_Text = (const char*)Emw_Text_EN ;
      const char* Vmw_Text = (const char*)Vmw_Text_EN ;
      const char* Hmw_Text = (const char*)Hmw_Text_EN ;
      const char* Smw_Text = (const char*)Smw_Text_EN ;
      if ( lang == esLang )
      {
         Fmw_Text = (const char*)Fmw_Text_ES ;
         Emw_Text = (const char*)Emw_Text_ES ;
         Vmw_Text = (const char*)Vmw_Text_ES ;
         Hmw_Text = (const char*)Hmw_Text_ES ;
         Smw_Text = (const char*)Smw_Text_ES ;
      }
      else if ( lang == zhLang )
      {
         Fmw_Text = (const char*)Fmw_Text_ZH ;
         Emw_Text = (const char*)Emw_Text_ZH ;
         Vmw_Text = (const char*)Vmw_Text_ZH ;
         Hmw_Text = (const char*)Hmw_Text_ZH ;
         Smw_Text = (const char*)Smw_Text_ZH ;
      }
      else if ( lang == viLang )
      {
         Fmw_Text = (const char*)Fmw_Text_VI ;
         Emw_Text = (const char*)Emw_Text_VI ;
         Vmw_Text = (const char*)Vmw_Text_VI ;
         Hmw_Text = (const char*)Hmw_Text_VI ;
         Smw_Text = (const char*)Smw_Text_VI ;
      }

      this->ic = new InitCtrl[twCTRLS] ;

      //*************************************
      //* Initialize control-array contents *
      //*************************************

      this->ic[twSavePB] = 
      {  //* 'Save Modified Data' Pushbutton - - - - - - - - - - - -  twSavePB *
         dctPUSHBUTTON,                // type:      
         rbtTYPES,                     // rbSubtype: (n/a)
         false,                        // rbSelect:  (n/a)
         3,                            // ulY:       upper left corner in Y
         2,                            // ulX:       upper left corner in X
         1,                            // lines:     control lines
         SAPB_COLS,                    // cols:      control columns
         Saveall_Text,                 // dispText:  
         this->cs.pn,                  // nColor:    non-focus color
         this->cs.pf,                  // fColor:    focus color
         tbPrint,                      // filter:    (n/a)
         "",                           // label:     (n/a)
         ZERO,                         // labY:      (n/a)
         ZERO,                         // labX       (n/a)
         ddBoxTYPES,                   // exType:    (n/a)
         1,                            // scrItems:  (n/a)
         1,                            // scrSel:    (n/a)
         NULL,                         // scrColor:  (n/a)
         NULL,                         // spinData:  (n/a)
         true,                         // active:    allow control to gain focus
         &this->ic[twFileMW]           // nextCtrl:  link in next structure
      } ;

      this->ic[twFileMW] = 
      {  //* 'File' Menuwin  - - - - - - - - - - - - - - - - - - - -  twFileMW *
         dctMENUWIN,                   // type:      define a scrolling-data control
         rbtTYPES,                     // rbSubtype: (n/a)
         false,                        // rbSelect:  (n/a)
         2,                            // ulY:       upper left corner in Y
         short(fieldwinX + 1),         // ulX:   upper left corner in X
         1,                            // lines:     (n/a)
         FMW_COLS,                     // cols:      control columns
         (const char*)Fmw_Text,        // dispText:  text-data array
         this->cs.mn,                  // nColor:    non-focus border color
         this->cs.mf,                  // fColor:    focus border color
         tbPrint,                      // filter:    (n/a)
         MenuName[lang][0],            // label:     label text
         ZERO,                         // labY:      (n/a)
         ZERO,                         // labX       (n/a)
         ddBoxTYPES,                   // exType:    (n/a)
         FMW_ITEMS,                    // scrItems:  elements in text/color arrays
         ZERO,                         // scrSel:    (n/a)
         this->cs.menu,                // scrColor:  color-attribute list
         NULL,                         // spinData:  (n/a)
         true,                         // active:    allow control to gain focus
         &this->ic[twEditMW]           // nextCtrl:  link in next structure
      } ;

      this->ic[twEditMW] = 
      {  //* 'Edit' Menuwin  - - - - - - - - - - - - - - - - - - - -  twEditMW *
         dctMENUWIN,                   // type:      define a scrolling-data control
         rbtTYPES,                     // rbSubtype: (n/a)
         false,                        // rbSelect:  (n/a)
         this->ic[twFileMW].ulY,       // ulY:       upper left corner in Y
         short(this->ic[twFileMW].ulX + COLLAPSED_WIDTH + 1), // ulX:  upper left corner in X
         1,                            // lines:     (n/a)
         EMW_COLS,                     // cols:      control columns
         (const char*)Emw_Text,        // dispText:  text-data array
         this->cs.mn,                  // nColor:    non-focus border color
         this->cs.mf,                  // fColor:    focus border color
         tbPrint,                      // filter:    (n/a)
         MenuName[lang][1],            // label:     label text
         ZERO,                         // labY:      (n/a)
         ZERO,                         // labX       (n/a)
         ddBoxTYPES,                   // exType:    (n/a)
         EMW_ITEMS,                    // scrItems:  elements in text/color arrays
         ZERO,                         // scrSel:    (n/a)
         this->cs.menu,                // scrColor:  color-attribute list
         NULL,                         // spinData:  (n/a)
         true,                         // active:    allow control to gain focus
         &this->ic[twViewMW]           // nextCtrl:  link in next structure
      } ;

      this->ic[twViewMW] = 
      {  //* 'View' Menuwin  - - - - - - - - - - - - - - - - - - - -  twViewMW *
         dctMENUWIN,                   // type:      define a scrolling-data control
         rbtTYPES,                     // rbSubtype: (n/a)
         false,                        // rbSelect:  (n/a)
         this->ic[twEditMW].ulY,       // ulY:       upper left corner in Y
         short(this->ic[twEditMW].ulX + COLLAPSED_WIDTH + 1), // ulX:  upper left corner in X
         1,                            // lines:     (n/a)
         VMW_COLS,                     // cols:      control columns
         (const char*)Vmw_Text,        // dispText:  text-data array
         this->cs.mn,                  // nColor:    non-focus border color
         this->cs.mf,                  // fColor:    focus border color
         tbPrint,                      // filter:    (n/a)
         MenuName[lang][2],            // label:     label text
         ZERO,                         // labY:      (n/a)
         ZERO,                         // labX       (n/a)
         ddBoxTYPES,                   // exType:    (n/a)
         VMW_ITEMS,                    // scrItems:  elements in text/color arrays
         ZERO,                         // scrSel:    (n/a)
         this->cs.menu,                // scrColor:  color-attribute list
         NULL,                         // spinData:  (n/a)
         true,                         // active:    allow control to gain focus
         &this->ic[twHelpMW]           // nextCtrl:  link in next structure
      } ;

      this->ic[twHelpMW] = 
      {  //* 'Help' Menuwin  - - - - - - - - - - - - - - - - - - - -  twViewMW *
         dctMENUWIN,                   // type:      define a scrolling-data control
         rbtTYPES,                     // rbSubtype: (n/a)
         false,                        // rbSelect:  (n/a)
         this->ic[twViewMW].ulY,       // ulY:       upper left corner in Y
         short(this->ic[twViewMW].ulX + COLLAPSED_WIDTH + 1), // ulX:  upper left corner in X
         1,                            // lines:     (n/a)
         HMW_COLS,                     // cols:      control columns
         (const char*)Hmw_Text,        // dispText:  text-data array
         this->cs.mn,                  // nColor:    non-focus border color
         this->cs.mf,                  // fColor:    focus border color
         tbPrint,                      // filter:    (n/a)
         MenuName[lang][3],            // label:     label text
         ZERO,                         // labY:      (n/a)
         ZERO,                         // labX       (n/a)
         ddBoxTYPES,                   // exType:    (n/a)
         HMW_ITEMS,                    // scrItems:  elements in text/color arrays
         ZERO,                         // scrSel:    (n/a)
         this->cs.menu,                // scrColor:  color-attribute list
         NULL,                         // spinData:  (n/a)
         true,                         // active:    allow control to gain focus
         &this->ic[twSortMW]           // nextCtrl:  link in next structure
      } ;

      this->ic[twSortMW] = 
      {  //* Sort-order sub-menu - - - - - - - - - - - - - - - - - -  twSortMW *
         dctMENUWIN,                   // type:      define a Menu-win control
         rbtTYPES,                     // sbSubtype: (n/a)
         false,                        // rbSelect:  (n/a)
         (short)(ic[twViewMW].ulY + 3),// ulY:       upper left corner in Y
         (short)(ic[twViewMW].ulX + ic[twViewMW].cols - 1), // UlX: upper left corner in X
         1,                            // lines:     (n/a)
         SMW_COLS,                     // cols:      menu width
         (const char*)Smw_Text,        // dispText:  text-data array
         this->cs.mn,                  // nColor:    non-focus border color
         this->cs.mf,                  // fColor:    focus border color
         tbPrint,                      // filter:    (n/a)
         NULL,                         // label:     no label for sub-menus
         ZERO,                         // labY:      (n/a)
         ZERO,                         // labX:      (n/a)
         ddBoxTYPES,                   // exType:    (n/a)
         SMW_ITEMS,                    // scrItems:  number of elements in text/color arrays
         ZERO,                         // scrSel:    (n/a)
         this->cs.menu,                // scrColor:  color-attribute list
         NULL,                         // spinData:  (n/a)
         false,                        // active:    initially inactive and invisible
         NULL                          // nextCtrl:  link in next structure
      } ;
   }
}  //* End InitAppControls()

//*************************
//*   InitStaticDisplay   *
//*************************
//******************************************************************************
//* Called when the dialog opens (or reopens) to write the static data into    *
//* the dialog.                                                                *
//*                                                                            *
//* Input  : nothing                                                           *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void Taggit::InitStaticDisplay ( void )
{
   const wchar_t navLEFT[]  = { L" \n\u25c0\n " },    // 'navigation left' symbol
                 navRIGHT[] = { L" \n\u25B6\n " },    // 'navigation right' symbol
                 navBLOCK[] = { L"\u2585\u2585" },    // vertical half-block
                 navUP   = 0x025B2,                   // 'navigation up' symbol
                 navDOWN = 0x025BC,                   // 'navigation down' symbol
                 wUP = 0x2191, wDN = 0x2193,          // arrows
                 wLT = 0x2190, wRT = 0x2192 ;
   AppLang lang = this->cfgOpt.appLanguage ;          // language index

   //* Construct and display the main application title *
   gString  gs( "%s%s (c)%s %s  ", 
                AppTitle1, AppVersion, copyrightYears, AppTitle2 ) ;
   this->dPtr->SetDialogTitle ( gs, this->cs.title ) ;

   //* Connect the menu controls to create a Menu Bar *
   short mbList[] = { twFileMW, twEditMW, twViewMW, twHelpMW, -1 } ;
   if ( (this->dPtr->GroupMenuwinControls ( mbList )) == OK )
   {
      //* Attach sub-menus to main menu controls *
      const short mbSubListView[] =
      { MAX_DIALOG_CONTROLS, MAX_DIALOG_CONTROLS, twSortMW, MAX_DIALOG_CONTROLS, 
        MAX_DIALOG_CONTROLS, MAX_DIALOG_CONTROLS, -1 } ;
      this->dPtr->AttachMenuwinSubmenus ( twViewMW, mbSubListView ) ;
      
   }

   //* Disable the unassigned/unimplemented menu items.*
   bool Fmw_Active[FMW_ITEMS] = // File Menu
   { true,  // save all
     true,  // save file
     true,  // save as
     true,  // save audio
     true,  // save tag summary
     true,  // playback
     true,  // refresh
     true,  // command shell
     true   // exit
   } ;
   attr_t Fmw_Colors[FMW_ITEMS] = 
   {
     this->cs.menu[1], this->cs.menu[1], this->cs.menu[1], this->cs.menu[1], 
     this->cs.menu[1], this->cs.menu[1], this->cs.menu[1], this->cs.menu[1], 
     this->cs.menu[1]
   } ;
   this->dPtr->SetActiveMenuItems ( twFileMW, Fmw_Active, Fmw_Colors ) ;
   
   bool Emw_Active[EMW_ITEMS] = // Edit Menu
   { true,  // edit filename
     true,  // edit metadata
     true,  // add image
     true,  // edit image
     true,  // undo edits
     true,  // autofill track
     true,  // duplicate field
     true,  // set title
     true,  // popularimeter
     true,  // clear file metadata 
     true,  // clear all metadata
     false, // edit preferences (not implemented)
   } ;
   attr_t Emw_Colors[EMW_ITEMS] = 
   {
     this->cs.menu[1], this->cs.menu[1], this->cs.menu[1], this->cs.menu[1], 
     this->cs.menu[1], this->cs.menu[1], this->cs.menu[1], this->cs.menu[1], 
     this->cs.menu[1], this->cs.menu[1], this->cs.menu[1], this->cs.pn
   } ;
   this->dPtr->SetActiveMenuItems ( twEditMW, Emw_Active, Emw_Colors ) ;
   
   bool Vmw_Active[VMW_ITEMS] = // View Menu
   { true,  // specify fields
     true,  // tag summary
     true,  // sort
     true,  // shift file down
     true,  // shift file up
     true,  // free space
     true,  // image data
     this->cfgOpt.rtl, // RTL field data
     true,  // color scheme
   } ;
   attr_t Vmw_Colors[VMW_ITEMS] = 
   {
     this->cs.menu[1], this->cs.menu[1], this->cs.menu[1], this->cs.menu[1], 
     this->cs.menu[1], this->cs.menu[1], this->cs.menu[1], 
     (this->cfgOpt.rtl ? this->cs.menu[1] : this->cs.pn), 
     this->cs.menu[1]
   } ;
   this->dPtr->SetActiveMenuItems ( twViewMW, Vmw_Active, Vmw_Colors ) ;


   //* Enclose the Menu Bar *
   winPos wp( (fieldwinY - 3), fieldwinX ) ;
   for ( short i = ZERO ; i <= (COLLAPSED_WIDTH * 4 + 4) ; ++i )
   {
      if (   i == ZERO || i == (COLLAPSED_WIDTH + 1) || i == (COLLAPSED_WIDTH * 2 + 2)
          || i == (COLLAPSED_WIDTH * 3 + 3) || i == (COLLAPSED_WIDTH * 4 + 4) )
         this->dPtr->WriteChar ( wp.ypos + 1, wp.xpos, FullBlock, this->cs.saved )  ;
      this->dPtr->WriteChar ( wp.ypos + 2, wp.xpos, HalfBlockUpper, this->cs.saved ) ;
      this->dPtr->WriteChar ( wp.ypos, wp.xpos++, HalfBlockLower, this->cs.saved ) ;
   }

   //* Enable the 'stable mouse' interface *
   if ( this->cfgOpt.enableMouse )
      this->cfgOpt.enableMouse = bool( (this->dPtr->meEnableStableMouse ()) == OK ) ;

   //* Interior dimensions of field-display window *
   this->fwinRows = this->termRows - fieldwinY - 2 ;
   this->fwinCols = this->termCols - fieldwinX - 2 ;

   //* Separate the application controls from the data grid *
   LineDef  lDefH(ncltHORIZ, ncltDUAL, fieldwinY, ZERO, this->termCols, this->cs.bb) ;
   this->dPtr->DrawLine ( lDefH ) ;
   //* Separate static horizontal left columns *
   //* from 4-direction scrolling fields.      *
   LineDef  lDefV(ncltVERT, ncltSINGLE, fieldwinY, fieldwinX, (this->termRows - 4), this->cs.bb) ;
   this->dPtr->DrawLine ( lDefV ) ;

   //* Write the Up/Down/Left/Right navigation *
   //* indicators for mouse-wielding users.    *
   wp = { short(fieldwinY + 12), fieldwinX } ;
   this->dPtr->WriteParagraph ( wp, navLEFT, this->cs.pf ) ;
   wp.xpos = this->termCols - 1 ;
   this->dPtr->WriteParagraph ( wp, navRIGHT, this->cs.pf ) ;
   wp = { fieldwinY, short(fieldwinX + (fwinCols / 2) - 2) } ;
   wp = this->dPtr->WriteString ( wp, navBLOCK, this->cs.unsaved ) ;
   wp = this->dPtr->WriteChar   ( wp, navUP,    this->cs.pf ) ;
   wp = this->dPtr->WriteString ( wp, navBLOCK, this->cs.unsaved ) ;
   wp = { short(this->termRows - 1), short(fieldwinX + (fwinCols / 2) - 2) } ;
   wp = this->dPtr->WriteString ( wp, navBLOCK, this->cs.unsaved ) ;
   wp = this->dPtr->WriteChar   ( wp, navDOWN,  this->cs.pf ) ;
   wp = this->dPtr->WriteString ( wp, navBLOCK, this->cs.unsaved ) ;

   //* Display basic navigation help *
   // (Note that this message is obscured when debugging output is active.)
   wp.ypos = 1 ;
   if ( this->cfgOpt.rtl )
      wp.xpos = this->termCols - 2 ;
   else
      wp.xpos = fieldwinX + COLLAPSED_WIDTH * 4 + 6 ; // (94)
   gs.compose( navHelp[lang], &wUP, &wDN, &wLT, &wRT ) ;
   this->dPtr->WriteParagraph ( wp, gs, this->cs.bb, false, this->cfgOpt.rtl ) ;

   //* If no source files specified, create a dummy record *
   if ( this->tData.sfCount == ZERO )
   {
      gs = " " ;
      gs.copy( this->tData.sf[this->tData.sfCount].sfPath, gsMAXBYTES ) ;
      gs.copy( this->tData.sf[this->tData.sfCount++].sfName, MAX_FNAME ) ;
   }
   //* Calculate index of last filename to be displayed in the window *
   //* and if necessary, adjust focus item to keep focus visible.     *
   this->InitIndices () ;

   this->UpdateDataWindows () ;     // display filename list and metadata matrix

   this->dPtr->RefreshWin () ;      // make everything visible

}  //* End InitStaticDisplay() *

//*************************
//*      InitIndices      *
//*************************
//******************************************************************************
//* Calculate the data-display indices.                                        *
//* a) 'sfLast': Calculate index of last filename to be displayed in the data  *
//*    window.                                                                 *
//* b) 'sfFirst': Value is not modified UNLESS dialog has been expanded        *
//*    vertically and maximum number of items is no longer displayed.          *
//* c) 'sfFocus': Value is not modified UNLESS the calculation for 'sfFirst'   *
//*    and 'sfLast' has moved the focus out of the visible area.               *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void Taggit::InitIndices ( void )
{
   short sfSlots = (this->fwinRows - 1) / 2 ;      // max # of filesnames displayed
   this->tData.sfLast = this->tData.sfCount - 1 ;  // index tail of list

   //* Walk upward from the tail of the list *
   while ( (this->tData.sfLast - this->tData.sfFirst + 1) > sfSlots )
      --this->tData.sfLast ;

   //* If maximum number of items is not displayed, shift items down
   while ( (this->tData.sfFirst > ZERO) &&
           ((this->tData.sfLast - this->tData.sfFirst + 1) < sfSlots) )
      --this->tData.sfFirst ;

   //* If item with focus has moved outside the visible window, *
   //* shift focus to nearest visible item.                     *
   if ( this->tData.sfFocus < this->tData.sfFirst )
      this->tData.sfFocus = this->tData.sfFirst ;
   else if ( this->tData.sfFocus > this->tData.sfLast )
      this->tData.sfFocus = this->tData.sfLast ;

#if DIALOG_DEBUG != 0   // TEMP TEMP TEMP
gString gs( " fwinRows:%02hd sfCount:%02hd \n"
            " sfFirst:%02hd  sfLast:%02hd  \n"
            " sfFocus:%02hd             ", 
            &this->fwinRows, &this->tData.sfCount, 
            &this->tData.sfFirst, &this->tData.sfLast, &this->tData.sfFocus ) ;
this->DebugMsg ( gs, true ) ;
#endif   // TEMP TEMP TEMP

}  //* End InitIndices() *

//*************************
//*   InitSimulatedData   *
//*************************
//******************************************************************************
//* DEBUGGING METHOD:                                                          *
//* Initialize the tData data members with simulated filenames and field       *
//* contents.                                                                  *
//*                                                                            *
//* Input  : fIndex  : (optional, -1 by default)                               *
//*                    If a single valid file index specified, then refresh    *
//*                    ONLY that file.                                         *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

#if SIMULATED_DATA != 0    // FOR DEBUGGING ONLY
void Taggit::InitSimulatedData ( short fIndex )
{
   const char* titles[] =
   {
      "Apple Sauce",
      "Wane Manner",
      "Greenbryer Steakhouse",
      "Chatty Cathy",
      "Tarnished Love",
      "Ham and Eggs On Toast",
      "Tommie's Ho",
      "Friends At Last",
      "Tiny Tim's First Lay",
      "Can't Drive Home Alone Tonight",
      "Roger Rabbit Gets Funky",
      "Not Since Tuesday",
      "You're Kidding, Right?",
      "Charlie's Angel",
      "Bonus Track - Down the Road",
      "Bonus Track - Eat My Shorts",
      "Bonus Track - Fluff My Muff",
   } ;

   gString gsTitle,                             // 'Title' fields
           gsArtist( "Creempuff Girls" ),       // 'Artist' fields
           gsAlbum( "Can't Drive Home Alone" ), // 'Album' fields
           gs ;                                 //output formatting
   short trk, items = 17 ;                      // number of filenames

   //* To refresh the data for a single file*
   if ( fIndex >= ZERO && fIndex < this->tData.sfCount )
   {
      trk = fIndex + 1 ;
      gs.compose( "~/Music/Test/Track %02hd.mp3", &trk ) ;
      gs.copy( this->tData.sf[this->tData.sfCount].sfPath, MAX_PATH ) ;
      gs.shiftChars( -13 ) ;
      gs.copy( this->tData.sf[this->tData.sfCount].sfName, MAX_PATH ) ;
      short f = fIndex ;
      this->tData.sf[f].sfTag.clear_data() ;    // erase any existing data
      gsTitle = titles[f] ;
      gsTitle.copy( this->tData.sf[f].sfTag.field[tfTit2], gsMAXCHARS ) ;
      gsArtist.copy( this->tData.sf[f].sfTag.field[tfTpe1], gsMAXCHARS ) ;
      gsAlbum.copy( this->tData.sf[f].sfTag.field[tfTalb], gsMAXCHARS ) ;
      gs.compose( "%2hd/%02hd", &trk, &this->tData.sfCount ) ;
      this->uiemFormatTrack ( gs ) ;      // this is done for consistency
      gs.copy( this->tData.sf[f].sfTag.field[tfTrck], gsMAXCHARS ) ;
      gs = "2013" ;
      gs.copy( this->tData.sf[f].sfTag.field[tfTyer], gsMAXCHARS ) ;
      return ;
   }

   //* Simulate file specs *
   this->tData.sfCount = ZERO ;  // discard any command-line filenames
   while ( this->tData.sfCount < items )
   {
      trk = this->tData.sfCount + 1 ;
      gs.compose( "~/Music/Test/Track %02hd.mp3", &trk ) ;
      gs.copy( this->tData.sf[this->tData.sfCount].sfPath, MAX_PATH ) ;
      gs.shiftChars( -13 ) ;
      gs.copy( this->tData.sf[this->tData.sfCount++].sfName, MAX_PATH ) ;
   }

   //* Populate selected metadata fields for each file *
   for ( short f = ZERO ; f < this->tData.sfCount ; ++f )
   {
      this->tData.sf[f].sfTag.clear_data() ;    // erase any existing data

      gsTitle = titles[f] ;
      gsTitle.copy( this->tData.sf[f].sfTag.field[tfTit2], gsMAXCHARS ) ;

      gsArtist.copy( this->tData.sf[f].sfTag.field[tfTpe1], gsMAXCHARS ) ;

      gsAlbum.copy( this->tData.sf[f].sfTag.field[tfTalb], gsMAXCHARS ) ;

      trk = f + 1 ;
      gs.compose( "%2hd/%02hd", &trk, &this->tData.sfCount ) ;
      this->uiemFormatTrack ( gs ) ;      // this is done for consistency
      gs.copy( this->tData.sf[f].sfTag.field[tfTrck], gsMAXCHARS ) ;

      gs = "2013" ;
      gs.copy( this->tData.sf[f].sfTag.field[tfTyer], gsMAXCHARS ) ;
   }

}  //* End InitSimulatedData() *
#endif   // SIMULATED_DATA

//*************************
//*      OpenDialog       *
//*************************
//******************************************************************************
//* Instantiate the application dialog and make it visible.                    *
//*                                                                            *
//*                                                                            *
//* Input  : odCode : member of enum odWarn indicating type of warning or error*
//*                   message to be displayed                                  *
//*          abort  : (by reference) if a serious error, set caller's flag to  *
//*                   indicate that user interactions cannot continue and that *
//*                   application should exit.                                 *
//*                                                                            *
//* Returns: 'true'  if dialog was opened successfully                         *
//*          'false' if unable to open dialog window                           *
//******************************************************************************
//* Notes: The leading dots (left column) indicate:                            *
//* ● ●                                                                        *
//* | +-- metadata-fields edits pending                                        *
//* +---- source filename edits pending                                        *
//*                                                                            *
//* Filename: Only files with supported filename extensions will be displayed. *
//*           An invalid metadata format for the specified extension will      *
//*           (it is hoped) be discovered and reported later.                  *
//*                                                                            *
//* Visible fields: Initially, all fields visible OR visibility is dependent   *
//*                 of config file setting                                     *
//*                                                                            *
//* Visible data: Pre-existing field data AND field data specified on command  *
//*               line are initially displayed for all visible fields.         *
//*                                                                            *
//*                                                                            *
//******************************************************************************

bool Taggit::OpenDialog ( odWarn odCode, bool& abort )
{
   bool status = false ;

   abort = false ;                  // hope for the best

   //* Define the application controls. Referenced by the 'ic' member. *
   this->InitAppControls () ;

   //* Initial parameters for dialog window *
   InitNcDialog dInit( this->termRows,  // number of display lines
                       this->termCols,  // number of display columns
                       ZERO,            // Y offset from upper-left of terminal 
                       ZERO,            // X offset from upper-left of terminal 
                       NULL,            // dialog title
                       ncltSINGLE,      // border line-style
                       this->cs.bb,     // border color attribute
                       this->cs.bb,     // interior color attribute
                       this->ic         // pointer to list of control definitions
                     ) ;

   //* Instantiate the dialog window *
   this->dPtr = new NcDialog ( dInit ) ;

   //* If selected UI language is an RTL language, set the *
   //* NcDialog internal RTL flag for titles and labels,   *
   //* as well as for the contents of the MenuBar.         *
   if ( this->cfgOpt.rtl )
   {
      this->dPtr->DrawLabelsAsRTL () ;
      this->dPtr->DrawContentsAsRTL ( twFileMW ) ;
      this->dPtr->DrawContentsAsRTL ( twEditMW ) ;
      this->dPtr->DrawContentsAsRTL ( twViewMW ) ;
      this->dPtr->DrawContentsAsRTL ( twHelpMW ) ;
      this->dPtr->DrawContentsAsRTL ( twSortMW ) ;
   }

   //* Open the dialog window *
   if ( (this->dPtr->OpenWindow()) == OK )
   {
      #if SIMULATED_DATA != 0    // FOR DEBUGGING ONLY
      this->InitSimulatedData () ;
      odCode = odGOOD_DATA ;
      #endif   // SIMULATED_DATA

      this->InitStaticDisplay () ;

      if ( odCode != odGOOD_DATA )
      {  //* A serious or fatal error has been detected during start-up. *
         //* Alert user, and if error is fatal, set caller's abort flag. *
         this->Warning ( odCode ) ;

         if ( odCode == odFILE_ACCESS ||
              odCode == odNO_SOURCE   ||
              odCode == odNO_ACCESS   ||
              odCode == odSRC_FORMAT  ||
              odCode == odNOT_DIR )
            abort = true ;
      }

      //* If the "wl-clipboard" utilities, "wl-copy" and "wl-paste" are *
      //* installed on the local system, then establish communications  *
      //* between the Wayland system clipboard and the NcDialog's       *
      //* Textbox Local Clipboard.                                      *
      //* If unable to establish a connection with the system clipboard,*
      //* copy/cut/paste operations will use the local clipboard only.  *
      //* CTRL+C == Copy, CTRL+X == Cut, CTRL+V == Paste,               *
      //* CTRL+A == Select all,                                         *
      //* SHIFT+LeftArrow and SHIFT+RightArrow == Select by character.  *
      //* Error conditions:                                             *
      //* a) If the external utilities are not installed, then silently *
      //*    continue with local clipboard only.                        *
      //* b) If the external utilities ARE installed, but connection    *
      //*    not established, then report the error.                    *
      if ( !(this->dPtr->wcbEnable ()) )
         /*this->dPtr->wcbUserAlert ( wcbsNOCONNECT )*/ ;

      status = true ;               // return success
   }  // OpenWindow()
   else
   {
      if ( this->suVerbose )
         this->suPos.ypos = (this->termRows - 1) ;
      nc.ClearLine ( this->suPos.ypos, nc.reG ) ;
      nc.WriteString ( this->suPos, 
                       "  Unable to open application dialog window! "
                       "Press any key to exit...  ", nc.reG ) ;
      nckPause();
   }
   return status ;

}  //* End OpenDialog() *

//*************************
//*     ResizeDialog      *
//*************************
//******************************************************************************
//* Caller has received a nckRESIZE keycode indicating that user has resized   *
//* the terminal window.                                                       *
//*                                                                            *
//* If the terminal is at least the minimum size, close, and reopen the dialog *
//* so it will continue to exactly fill the terminal window.                   *
//*                                                                            *
//* Input  : forceUpdate : (optional, 'false' by default)                      *
//*                        If specified, force a resize even if the terminal   *
//*                        window's size has not changed. (Used to dynamically *
//*                        update the dialog color scheme.)                    *
//*                                                                            *
//* Returns: 'false' if successful                                             *
//*          'true'  if unable to reopen dialog                                *
//******************************************************************************
//* Modifications needed to resize the dialog:                                 *
//* ------------------------------------------                                 *
//* -- The source data are unchanged, only the display will change.            *
//* -- 'dPtr' will get a new value                                             *
//* -- 'termRows' and/or 'termCols' WILL change                                *
//* -- 'fwinRows' and/or 'fwinCols' WILL change                                *
//* -- 'sfFirst' and 'sfLast':                                                 *
//*    'sfLast'  i.e. the last item currently displayed may be modified:       *
//*     -- If dialog is smaller in Y, then 'sfFirst' will remain stable and    *
//*        'sfLast will decrease IF there are no longer enough rows to display *
//*        all the previously-displayed files.                                 *
//*     -- If dialog is larger in Y, then:                                     *
//*        a) 'sfLast' will increase IF there were undisplayed files below     *
//*           the previous 'sfLast'.                                           *
//*        b) 'sfFirst' will decrease IF no undisplayed files below 'sfLast'.  *
//*           were undisplayed files above the previous 'sfFirst'              *
//* ALL DISPLAYED   HEAD, NOT TAIL    TAIL, NOT HEAD    NEITHER TAIL NOR HEAD  *
//*                                                                            *
//*                                   f-2               f-2                    *
//*  - - -          - - -             f-1               f-1                    *
//* >0             >0                >f                >f                      *
//*  1              1                 f+1               f+1                    *
//*  2              2                 f+2               f+2                    *
//*  ...            ...               ...               ...                    *
//* >n             >n                >n                >n                      *
//*  - - -          n+1              - - -              n+1                    *
//*                 n+2                                 n+2                    *
//*                                                                            *
//* -- The mouse interface MAY fail to re-initialize, but it is unlikely.      *
//*                                                                            *
//* =  =  =  =  =  =  =  =  =  =  =  =  =  =  =  =  =  =  =  =  =  =  =  =  =  *
//* Important Note:                                                            *
//* 1) An nckRESIZE signal is sent by the ncurses library for each row or      *
//*    column change in the size of the terminal window. This means that       *
//*    there are potentially dozens of resize signals in what the user would   *
//*    consider to be a single resize operation.                               *
//* 2) Because we want to redraw the dialog only once, we wait until the       *
//*    stream of resize signals is complete before sizing the new dialog       *
//*    window.                                                                 *
//*    a) This is not straightforward because the user's action could take     *
//*       anwhere between 100ms to several seconds.                            *
//*    b) We can't really be sure when or how many additional signals will be  *
//*       generated, so we want to wait as long as possible for the user to    *
//*       finish.                                                              *
//*    c) On the other hand, the user would be peeved if there were a          *
//*       noticeable delay between the request and the response.               *
//*    d) Our solution is to wait 100 milliseconds and test for input.         *
//*       - If there is input, discard it and keep waiting.                    *
//*       - If there is no input, decrement a counter and keep waiting.        *
//*       - When the no-input counter reaches zero, we flush the queue again,  *
//*         just to be safe, and only then will the user's resize operation be *
//*         considered complete.                                               *
//*       - The delay between the end of event stream and the resize is        *
//*         determined by the initial value of the 'nonEvent' counter.         *
//*         This value was determined empirically as a noticeable-but-small    *
//*         delay.                                                             *
//*       - The delay algorithm is not foolproof, but if the user DOES fool    *
//*         us, the result is only that the display will be ugly until the     *
//*         next key/mouse event.                                              *
//******************************************************************************

bool Taggit::ResizeDialog ( bool forceUpdate )
{
   const short neCOUNT = 5 ;
   wkeyCode wk ;                    // user input
   short newY, newX,                // new terminal dimensions
         nonEvent = neCOUNT ;       // non-event counter (see note above)
   bool status = false ;            // return value

   //* Because there may be several nckRESIZE signals in the input queue, *
   //* we wait until the user is finished resizing the terminal window    *
   //* before requesting the new window size.                             *
   chrono::duration<short, std::milli>aMoment( 100 ) ;
   do
   {
      this_thread::sleep_for( aMoment ) ;
      if ( (this->dPtr->KeyPeek ( wk )) != wktERR )
         this->dPtr->GetKeyInput ( wk ) ;    // discard the input
      else
         --nonEvent ;
   }
   while ( nonEvent > ZERO ) ;
   this->dPtr->FlushKeyInputStream () ;

   this->dPtr->ScreenDimensions ( newY, newX ) ; // get new terminal dimensions

   //* If terminal dimensions have changed *
   if ( (newY != this->termRows) || (newX != this->termCols) || forceUpdate )
   {
      bool done = false ;
      while ( ! done )
      {
         if ( (newY < MIN_APP_ROWS) || (newX < MIN_APP_COLS) )
         {
            gString gs( "  Please expand to at least %02hd rows by %03hd columns.   ", 
                        &MIN_APP_ROWS, &MIN_APP_COLS ) ;
            nc.ClearScreen () ;
            nc.WriteString ( 0, 0,
                             "        ERROR! - Terminal window is too small.        ",
                             nc.reR ) ;
            nc.WriteString ( 1, 0, gs.gstr(), nc.reR ) ;
            nc.WriteString ( 2, 0,
                             "              OR: Press any key to exit.              ",
                             nc.reR ) ;
            this->dPtr->GetKeyInput ( wk ) ;
            if ( (wk.type == wktFUNKEY) && (wk.key == nckRESIZE) )
            {  //* Wait for the input queue to clear *
               nonEvent = neCOUNT ;
               do
               {
                  this_thread::sleep_for( aMoment ) ;
                  if ( (this->dPtr->KeyPeek ( wk )) != wktERR )
                     this->dPtr->GetKeyInput ( wk ) ;    // discard the input
                  else
                     --nonEvent ;
               }
               while ( nonEvent > ZERO ) ;
               this->dPtr->FlushKeyInputStream () ;
               this->dPtr->ScreenDimensions ( newY, newX ) ;
            }
            else
               status = done = true ;
         }
         else
         {
            //* Save the new terminal size *
            this->termRows = newY ;
            this->termCols = newX ;

            //* Close the dialog *
            delete ( this->dPtr ) ;

            //* Clear any messages previously written to the NCurses screen *
            nc.ClearScreen () ;

            //* Set the new dimensions for instantiation *
            InitNcDialog dInit( this->termRows,  // number of display lines
                                this->termCols,  // number of display columns
                                ZERO,            // Y offset from upper-left of terminal 
                                ZERO,            // X offset from upper-left of terminal 
                                NULL,            // dialog title
                                ncltSINGLE,      // border line-style
                                this->cs.bb,     // border color attribute
                                this->cs.bb,     // interior color attribute
                                this->ic         // pointer to list of control definitions
                              ) ;

            //* Instantiate the dialog window *
            this->dPtr = new NcDialog ( dInit ) ;

            //* If selected UI language is an RTL language, set the *
            //* NcDialog internal RTL flag for titles and labels.   *
            if ( this->cfgOpt.rtl )
            {
               this->dPtr->DrawLabelsAsRTL () ;
               this->dPtr->DrawContentsAsRTL ( twFileMW ) ;
               this->dPtr->DrawContentsAsRTL ( twEditMW ) ;
               this->dPtr->DrawContentsAsRTL ( twViewMW ) ;
               this->dPtr->DrawContentsAsRTL ( twHelpMW ) ;
               this->dPtr->DrawContentsAsRTL ( twSortMW ) ;
            }

            //* Open the dialog window *
            if ( (this->dPtr->OpenWindow()) == OK )
            {
               this->InitStaticDisplay () ;
            }
            else     // (this is fairly unlikely)
            {
               nc.WriteString ( 10, 0, L" RESIZE FAILED ", nc.reR ) ;
               nckPause();
            }
            done = true ;
         }
      }     // while(!done)
   }
   return status ;

}  //* End ResizeDialog() *

//*************************
//*      CloseDialog      *
//*************************
//******************************************************************************
//* Close the dialog window. This is done by deleting the NcDialog object(s)   *
//* established on startup.                                                    *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void Taggit::CloseDialog ( void )
{

   if ( this->cfgOpt.enableMouse )  // shut down the mouse interface
      this->dPtr->meDisableMouse () ;

   delete ( this->dPtr ) ;          // close the dialog
   this->dPtr = NULL ;

   delete [] this->ic ;             // release the control definitions
   this->ic = NULL ;

}  //* End CloseDialog() *

//*************************
//*     EditsPending      *
//*************************
//******************************************************************************
//* Scan the data for pending-edit flags. Called when user wants to exit the   *
//* application. Prevents accidental loss of edited data.                      *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: the number of source files which have unsaved edits               *
//******************************************************************************

short Taggit::EditsPending ( void )
{
   short modFiles = ZERO ;    // return value, number of modified files

   for ( short si = ZERO ; si < this->tData.sfCount ; ++si )
   {
      if ( this->tData.sf[si].sfMod || this->tData.sf[si].sfTag.tfMod )
         ++modFiles ;
   }
   return modFiles ;

}  //* End EditsPending() *

//*************************
//*        Warning        *
//*************************
//******************************************************************************
//* Report a serious problem and wait for user response.                       *
//*                                                                            *
//* Input  : odCode : member of enum odWarn, indicates the message(s) to be    *
//*                   displayed                                                *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************
//* Important Note: The number of columns specified for the dialog is          *
//*                 calculated to allow centering for the minimum dialog width *
//*                 without extending into the filename display area.          *
//*                 Don't change it!                                           *
//******************************************************************************

void Taggit::Warning ( odWarn odCode )
{
   const char* Msg01[][10] = 
   {
      {  // English
         "  UNRECOVERABLE ERROR!  ",
         " ",
         "One or more of the specified source files",
         " (shown in alternate color) ",
         "  a) is missing, or",
         "  b) is not a supported audio file type, or",
         "  c) is not a 'regular' file, or",
         "  d) you don't have read/write access",
         " ",
         NULL
      },
      {  // Espanol
         "  ¡ERROR IRRECUPERABLE!  ",
         " ",
         "Uno o más de los archivos especificados",
         " (mostrado en color alterno) ",
         "  a) no se encuentra, o",
         "  b) es un tipo de audio no admitido, o",
         "  c) no es un archivo 'regular', o",
         "  d) no tiene acceso de lectura y escritura",
         " ",
         NULL
      },
      {  // Zhongwen
         "  不可恢复的错误!  ",
         " ",
         "一个或多个指定的文件：",
         " (名称以替代颜色显示) ",
         "  a) 文件丢失",
         "  b) 不支持音频格式",
         "  c) 该文件不是“常规”类型",
         "  d) 您无权访问该文件",
         " ",
         NULL
      },
      {  // TiengViet
         "   Lỗi Nghiêm Trọng!   ",
         " ",
         "Một hoặc nhiều hơn các tập tin chỉ định",
         " (hiển thị trong một màu sắc khác nhau) ",
         "  a) các tập tin không thể được tìm thấy, hoặc",
         "  b) định dạng âm thanh sai, hoặc",
         "  c) không phải là loại tập tin dự kiến, hoặc",
         "  d) bạn không thể đọc và ghi dữ liệu",
         " ",
         NULL
      },
   } ;

   //** odNO_SOURCE **
   const char* Msg02[][6] = 
   {
      {  // English
         "  UNRECOVERABLE ERROR!  ",
         " ",
         "No individual source files specified, and ",
         "no music files found in target directory.",
         " ",
         NULL
      },
      {  // Espanol
         "  ¡ERROR IRRECUPERABLE!  ",
         " ",
         "No se especificaron archivos de fuentes,",
         "individuales y no se encontraron archivos",
         "de música en el directorio de destino.",
         NULL
      },
      {  // Zhongwen
         "  不可恢复的错误!  ",
         " ",
         "没有指定单个源文件， 和 ",
         "在目标目录中找不到音乐文件。",
         " ",
         NULL
      },
      {  // TiengViet
         "   Lỗi Nghiêm Trọng!   ",
         " ",
         "Không có tập tin nguồn cá nhân quy định,",
         "và không có các file nhạc được tìm thấy",
         "trong thư mục đích.",
         NULL
      },
   } ;

   //** odDUPLICATES **
   const char* Msg03[][8] = 
   {
      {  // English
         "     WARNING!     ",
         " ",
         "One or more duplicate source files specified!",
         " ",
         " Duplicates have been removed from the list.",
         NULL
      },
      {  // Espanol
         "    ¡Advertencia!    ",
         " ",
         "¡Se especificaron uno o más archivos de origen",
         " duplicados!",
         " ",
         "        Los archivos duplicados se han",
         "             eliminado de la lista.",
         NULL
      },
      {  // Zhongwen
         "    警告！    ",
         " ",
         "         一个或多个重复的源文件指定！",
         " ",
         "         列表中已经删除了重复的记录。",
         NULL
      },
      {  // TiengViet
         "    Cảnh Báo!    ",
         " ",
         "Đã chỉ định một hoặc nhiều tệp nguồn trùng lặp!",
         " ",
         "  Hồ sơ trùng lặp đã được xóa khỏi danh sách.",
         NULL
      },
   } ;

   //** odSRC_FORMAT **
   const char* Msg04[][6] = 
   {
      {  // English
         "  UNRECOVERABLE ERROR!  ",
         " ",
         "The internal format of one or more source files",
         "is inconsistent with its filename extension.",
         " ",
         NULL
      },
      {  // Espanol
         "  ¡ERROR IRRECUPERABLE!  ",
         " ",
         "El formato interno de uno o más archivos",
         "fuente no coincide con la extensión de nombre",
         "de archivo.",
         NULL
      },
      {  // Zhongwen
         "  不可恢复的错误!  ",
         " ",
         "一个或多个源文件的内部格式与文件扩展名不匹配。",
         " ",
         " ",
         NULL
      },
      {  // TiengViet
         "   Lỗi Nghiêm Trọng!   ",
         " ",
         "Định dạng nội bộ của một hoặc nhiều tệp nguồn",
         "không phù hợp với đuôi tên tệp tin.",
         " ",
         NULL
      },
   } ;

   attr_t amColors[] =
   {
      this->cs.em,
      this->cs.sd, this->cs.sd,
      this->cs.sd,
      this->cs.sd, this->cs.sd, this->cs.sd, this->cs.sd,
      this->cs.sd, this->cs.sd, this->cs.sd, this->cs.sd,
   } ;

   //* Point to the appropriate message group in the application language *
   AppLang lang = this->cfgOpt.appLanguage ; // language index
   const char** msgPtr = NULL ;
   if ( odCode == odFILE_ACCESS )
      msgPtr = Msg01[lang] ;
   else if ( odCode == odNO_SOURCE )
      msgPtr = Msg02[lang] ;
   else if ( odCode == odDUPLICATES )
      msgPtr = Msg03[lang] ;
   else if ( odCode == odSRC_FORMAT )
      msgPtr = Msg04[lang] ;
   genDialog gd( msgPtr, this->cs.sd, 11, 50, -1, -1, amColors, 
                 this->cfgOpt.rtl, attrDFLT, attrDFLT, okText[lang] ) ;
   this->dPtr->InfoDialog ( gd ) ;

}  //* End Warning() *

//*************************
//* UpdateFilenameWindow  *
//*************************
//******************************************************************************
//* Format the data for the filename list and the edits-pending indicators     *
//* and write it to the display.                                               *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void Taggit::UpdateFilenameWindow ( void )
{
   const char* ufwFilename[] = 
   {
      "File Name ",
      "Nombre del Archivo ",
      "文件名 ",
      "Tên Của Tập Tin ",
   } ;

   gString gsOut, gsFile ;
   winPos wp = { headerROW, filehdrCOL } ;
   bool rtlFields = bool(this->cfgOpt.rtl && this->cfgOpt.rtlf) ;

   this->dPtr->ClearArea ( headerROW, (filehdrCOL - 1), 
                           (this->termRows - wp.ypos - 1), FNAME_WIN_WIDTH ) ;

   //* Write column headings *
   wp = this->dPtr->WriteChar ( wp.ypos, ++wp.xpos, L'F', this->cs.bb ) ;
   wp = this->dPtr->WriteChar ( wp.ypos, ++wp.xpos, L'M', this->cs.bb ) ;
   if ( this->cfgOpt.rtl )
      wp.xpos = FNAME_WIN_WIDTH - 1 ;
   else
      wp.xpos += 2 ;
   gsOut = ufwFilename[this->cfgOpt.appLanguage] ;
   if ( this->cfgOpt.sortBy == sbFNAME )
      gsOut.append( sortARROW ) ;
   this->dPtr->WriteString ( wp, gsOut, this->cs.bb, false, this->cfgOpt.rtl ) ;
   --wp.ypos ;

   //* Write the filename list along with edit-pending indicators.*
   for ( short i = this->tData.sfFirst ; i < this->tData.sfCount ; ++i )
   {
      wp.ypos += 2 ;             // write position
      wp.xpos = filehdrCOL ;

      //* Write edits-pending indicators *
      wp = this->dPtr->WriteChar ( wp.ypos, ++wp.xpos, fillCIRCLE, 
             this->tData.sf[i].sfMod ? this->cs.unsaved : this->cs.saved ) ;
      wp = this->dPtr->WriteChar ( wp.ypos, ++wp.xpos, fillCIRCLE, 
             this->tData.sf[i].sfTag.tfMod ? this->cs.unsaved : this->cs.saved ) ;

      //* Write filename field *
      if ( rtlFields )
         wp.xpos = FNAME_WIN_WIDTH ;
      else
         wp.xpos += 2 ;
      if ( *this->tData.sf[i].sfName != NULLCHAR )
      {
         gsOut.compose( " %s                                ", this->tData.sf[i].sfName ) ;
         gsOut.limitCols( stdFIELD ) ;
         wp = this->dPtr->WriteString ( wp, gsOut, this->cs.tf, false, rtlFields ) ;
      }
      else
      {  //* If sf[i].sfName is empty, it indicates that sf[i].sfPath is an    *
         //* unverified item. Extract name, and display it an alternate color. *
         //* Note: This will happen only the first time this method is called  *
         //* and indicates an unrecoverable error.                             *
         gsFile = this->tData.sf[i].sfPath ;
         short indx = (gsFile.findlast( L'/' )) + 1 ;
         gsOut.compose( " %S                                ", &gsFile.gstr()[indx] ) ;
         gsOut.limitCols( stdFIELD ) ;
         wp = this->dPtr->WriteString ( wp, gsOut, this->cs.pn, false, 
                                        bool(this->cfgOpt.rtl && this->cfgOpt.rtlf) ) ;
      }

      //* For the highlighted file, underline to show item focus *
      if ( i == this->tData.sfFocus )
         this->dPtr->WriteString ( (wp.ypos + 1), (filehdrCOL + 6), focusLine, 
                                   this->cs.unsaved ) ;

      //* If window is filled, exit the loop *
      if ( wp.ypos > (this->termRows - 5) )
         break ;
   }
   this->dPtr->RefreshWin () ;

}  //* End UpdateFilenameWindow() *

//*************************
//* UpdateMetadataWindow  *
//*************************
//******************************************************************************
//* Format the data for the metadata fields and write it to the display.       *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************
//* Notes:                                                                     *
//* ------                                                                     *
//* -- This window is laid out like a spreadsheet. Although it contains no     *
//*    controls or editable fields, these fields are represented in the window *
//*    so the user may select the one to edit.                                 *
//*                                                                            *
//* -- The array of column headings is designed for LTR languages. For this    *
//*    reason, we must play some tricks to set width and position of headings  *
//*    for RTL languages.                                                      *
//*    - Initially we tried to set the insertion point to actually write the   *
//*      headings as RTL text; however, the math was too complex and prone to  *
//*      errors for multicolumn characters.                                    *
//*    - Instead we have opted to simply reverse the order of characters in    *
//*      the output string and then write it as LTR. While this actually works *
//*      very well with the test data, BE AWARE: some Arabic and other         *
//*      characters are actually a combination of two or more wchar_t values.  *
//*      While this is uncommon, it may be a problem for some RTL characters   *
//*      because a straight reversal of wchar_t values may yield an occasional *
//*      garbage character. This is not easily rectified without teaching the  *
//*      application how to read Arabic which is far beyond the scope of this  *
//*      application's design.                                                 *
//*                                                                            *
//* -- Display of field contents for RTL languages:                            *
//*    - Numeric fields tfTRCK, tfTYER, tfTLEN, tfTSIZ, tfTBPM, tfTPOS(?),     *
//*      tfTDLY                                                                *
//*      Hindu-Arabic numeric data are intrinsically LTR.                      *
//*    - Date fields tfTDAT, tfTIME                                            *
//*      Dates are a form of numeric data, so always LTR.                      *
//*    - Genre field tfCON is from a fixed, nontranslated list, so always LTR. *
//*    - Language field tfTLAN is from a fixed list of codes, so always LTR.   *
//*    - tfTCOP (copyright) is a formatted field with a numeric date, so       *
//*      conceivably is mixed LTR/RTL.                                         *
//*    - tfTSRC (ISRC code) is numeric field, so always LTR.                   *
//*    - tfTSEE (settings) field is alphanumeric, so always LTR.               *
//*    - tfTFLT (filetype) is an English alphanumeric field, so always LTR.    *
//*    - tfTMED (media type) field is one of the English abbreviations, so LTR.*
//*    Other fields may be toggled between RTL/LTR display based on            *
//*    cfgOpt.rtl != false && the state of cfgOpt.rtlf.                        *
//*                                                                            *
//*                                                                            *
//******************************************************************************

void Taggit::UpdateMetadataWindow ( void )
{
   const short fldSPC = 2 ;      // spacing between fields
   const char* trackField = "    " ;
   const char* yearField = "      " ;
   gString gsOut, gsFocus ;
   winPos wp = { headerROW, metahdrCOL } ;
   short remCols = this->fwinCols - 1,    // display columns remaining
         fldCols,                         // display columns in current field
         dispFields = ZERO ;              // number of fields successfully displayed
   bool rtlFields = (this->cfgOpt.rtl && this->cfgOpt.rtlf) ; // display fields as RTL

   this->dPtr->ClearArea ( headerROW, (metahdrCOL - 1), this->fwinRows, this->fwinCols ) ;

   //* Display column headings *
   const char* const* umwHPtr = umwHeaders[this->cfgOpt.appLanguage] ;
   for ( short i = this->tData.sffOffset ; i < tfCOUNT ; ++i )
   {
      if ( ! this->tData.sffDisp[i] )     // skip disabled fields
         continue ;
      gsOut = umwHPtr[i] ;
      //* If list is sorted on this field, insert the indicator *
      if ( (this->cfgOpt.sortBy != sbFNAME) && (this->cfgOpt.sortBy != sbNONE)
          && (((this->cfgOpt.sortBy == sbTITLE)  && (i == tfTit2)) ||
              ((this->cfgOpt.sortBy == sbALBUM)  && (i == tfTalb)) ||
              ((this->cfgOpt.sortBy == sbTRACK)  && (i == tfTrck)) ||
              ((this->cfgOpt.sortBy == sbARTIST) && (i == tfTpe1))
             ) )
      {
         const wchar_t* wPtr = gsOut.gstr() ;
         short indx = gsOut.findlast( SPACE ) ;
         while ( (wPtr[indx - 2] == SPACE) && (indx > ZERO) )
            --indx ;
         gsOut.replace( L" ", sortARROW, indx ) ;
      }
      if ( this->cfgOpt.rtl )
      {
         //* Reverse the heading text, discarding the inter-column padding *
         short wCnt = ZERO, t = ZERO ;
         wchar_t wbuff[gsMAXCHARS] ;
         const wchar_t* wPtr = gsOut.gstr( wCnt ) ;
         for ( short s = wCnt - 4 ; s >= ZERO ; --s )
            wbuff[t++] = wPtr[s] ;
         wbuff[t++] = L' ' ;     // re-insert the inter-column padding
         wbuff[t++] = L' ' ;
         wbuff[t] = NULLCHAR ;
         gsOut = wbuff ;
      }
      gsOut.limitCols( remCols ) ;
      wp = this->dPtr->WriteString ( wp, gsOut, this->cs.bb ) ;
      ++dispFields ;
      if ( (remCols -= (gsOut.gscols())) < fldSPC )
         break ;
   }

   //* Draw the tag fields for each source file *
   this->tData.sffLastField = false ;     // reset last-field-displayed flag
   bool lastFlag ;                        // local l-f-d flag
   wp = { short(wp.ypos + 1), metahdrCOL } ;
   for ( short f = this->tData.sfFirst ; f < this->tData.sfCount ; ++f )
   {
      remCols = this->fwinCols - 1 ;            // free columns remaining

      //* 'sffOffset' indicates the leftmost field to be displayed.     *
      //* 'sffLastField' == true if last (rightmost) field displayed.   *
      lastFlag = false ;
      for ( short i = this->tData.sffOffset ; i < tfCOUNT ; ++i )
      {
         if ( ! this->tData.sffDisp[i] )        // skip disabled fields
            continue ;

         switch ( i )
         {
            case tfTrck:         // Track Number, and optionally, Track Count
               {
               //* Under MP3, the TRCK field is formatted as:               *
               //* track/totaltracks. Example: 4/12                         *
               //* Consistent formatting across media formats is necessary  *
               //* for displaying this field, so pre-processing is performed*
               //* on this field in the LoadMetadata() method.              *
               gsOut = trackField ;
               if ( f == this->tData.sfFocus )
               {
                  gsFocus = focusLine ;
                  gsFocus.limitCols( (gsOut.gscols()) * 2 + 1) ;
                  if ( remCols < (gsFocus.gscols()) )
                     gsFocus.limitCols( remCols ) ;
                  this->dPtr->WriteString ( (wp.ypos + 1), wp.xpos, gsFocus,
                                            this->cs.unsaved ) ;
               }

               int trk = ZERO, seq = ZERO, convCount = ZERO ;
               gString gsSeq( this->tData.sf[f].sfTag.field[tfTrck] ) ;
               if ( gsSeq.gschars() > 1 )
                  convCount = swscanf ( gsSeq.gstr(), L"%d / %d", &trk, &seq ) ;
               else
                  gsSeq = trackField ;
               if ( convCount == 2 )
               {
                  if ( trk > 99 )   trk = 99 ;
                  if ( seq > 99 )   seq = 99 ;
                  gsOut.compose( " %2d ", &trk ) ;
                  gsSeq.compose( " %2d ", &seq ) ;
               }
               else if ( convCount == 1 )
               {
                  if ( trk > 99 )   trk = 99 ;
                  gsOut.compose( " %2d ", &trk ) ;
                  gsSeq = trackField ;
               }
               else
               {
                  gsOut = trackField ;
                  gsSeq = trackField ;
               }
               if ( remCols >= (gsOut.gscols()) )
               {
                  wp = this->dPtr->WriteString ( wp, gsOut, this->cs.tn ) ;
                  remCols -= gsOut.gscols() ;
                  if ( remCols-- >= 1 )
                  {
                     wp = this->dPtr->WriteChar ( wp, L'/', this->cs.bb ) ;
                     if ( (gsSeq.limitCols( remCols )) > ZERO )
                     {
                        wp = this->dPtr->WriteString ( wp, gsSeq, this->cs.tn ) ;
                        if ( (remCols -= (gsSeq.gscols() + 2)) < ZERO )
                           remCols = ZERO ;
                     }
                  }
               }
               else
               {
                  if ( (gsOut.limitCols( remCols )) > ZERO )
                  {
                     wp = this->dPtr->WriteString ( wp, gsOut, this->cs.tn ) ;
                     remCols -= gsOut.gscols() ;
                  }
               }
               }
               break ;

            case tfTyer:         // Year published
               gsOut = yearField ;     // empty field
               fldCols = gsOut.gscols() ;
               if ( this->tData.sf[f].sfTag.field[i][0] != NULLCHAR )
               {  // Note: The specification says that this field is exactly 
                  // four(4) characters. (It SHOULD HAVE said 4 digits.)
                  gsOut.compose( " %S     ", this->tData.sf[f].sfTag.field[i] ) ;
                  gsOut.limitCols( 5 ) ;
                  gsOut.append( L' ' ) ;
               }
               fldCols = gsOut.limitCols( (remCols >= yearFIELD) ? 
                                          yearFIELD : remCols ) ;
               if ( f == this->tData.sfFocus )
               {
                  gsFocus = focusLine ;
                  gsFocus.limitCols( fldCols ) ;
                  this->dPtr->WriteString ( (wp.ypos + 1), wp.xpos, gsFocus,
                                            this->cs.unsaved ) ;
               }
               wp = this->dPtr->WriteString ( wp, gsOut, this->cs.tn ) ;
               remCols -= (gsOut.gscols() + 2) ;
               break ;
                                 //                           ALWAYS LTR
            case tfTit2:         // Title                     ==========
            case tfTpe1:         // Artist
            case tfTalb:         // Album
            case tfTcon:         // Genre                         Yes
            case tfTpe2:         // Guest artist
            case tfTpub:         // Publisher
            case tfTcom:         // Composer
            case tfText:         // Lyricist
            case tfTpe4:         // Remixed by
            case tfTcop:         // Copyright                     ?
            case tfTown:         // File owner/licensee
            case tfTxxx:         // Comment (4KB max)
            case tfTdat:         // Record Date                   Yes
            case tfTime:         // Record Time                   Yes
            case tfTrda:         // Record Date Supplement
            case tfTlen:         // Playback Time (mSec)
            case tfTsiz:         // Audio Data Size               Yes
            case tfTbpm:         // Beats Per Minute              Yes
            case tfTit1:         // Content Group
            case tfTit3:         // Subtitle
            case tfTpe3:         // Conductor
            case tfTpos:         // Part Of Set                   ?
            case tfTkey:         // Initial Key
            case tfTlan:         // Language                      Yes
            case tfTope:         // Original Artist
            case tfToal:         // Original Album
            case tfTofn:         // Original Filename
            case tfToly:         // Original lyricist
            case tfTory:         // Original Release Year
            case tfTrsn:         // Internet Radio Station Name
            case tfTrso:         // Internet Radio Station Owner
            case tfTsrc:         // ISRC Recording Code           Yes
            case tfTsee:         // Software/Hardware Settings    Yes
            case tfTflt:         // File Type                     Yes
            case tfTdly:         // Playlist Delay                Yes
            case tfTenc:         // Encoded By
            case tfTmed:         // Media Type                    Yes
            default:             // all other fields are 'standard' width
               gsOut = baseField ;     // empty field
               if ( this->tData.sf[f].sfTag.field[i][0] != NULLCHAR )
                  gsOut.insert( this->tData.sf[f].sfTag.field[i], 1 ) ;
               //* If alphabetical or alphanumeric, non-formatted field, *
               //* reverse the characters. (see notes above)             *
               if ( rtlFields && 
                    (i != tfTcon) &&   // Genre
                    (i != tfTcop) &&   // Copyright
                    (i != tfTdat) &&   // Date
                    (i != tfTime) &&   // Time
                    (i != tfTlen) &&   // Length
                    (i != tfTsiz) &&   // Size
                    (i != tfTbpm) &&   // Beats-Per-Minute
                    (i != tfTpos) &&   // Part-of-Set
                    (i != tfTlan) &&   // Language
                    (i != tfTsrc) &&   // ISRC code
                    (i != tfTsee) &&   // Settings
                    (i != tfTflt) &&   // File type
                    (i != tfTdly) &&   // Delay
                    (i != tfTmed)      // Media type
                  )
               {
                  this->gsRevChars ( gsOut ) ;
                  fldCols = gsOut.shiftCols( -((remCols >= stdFIELD) ? 
                                               (gsOut.gscols() - stdFIELD) : 
                                               (gsOut.gscols() - remCols)) ) ;
               }
               else
                  fldCols = gsOut.limitCols( (remCols >= stdFIELD) ? stdFIELD : remCols ) ;
               if ( f == this->tData.sfFocus )
               {
                  gsFocus = focusLine ;
                  gsFocus.limitCols( fldCols ) ;
                  this->dPtr->WriteString ( (wp.ypos + 1), wp.xpos, gsFocus,
                                            this->cs.unsaved ) ;
               }
               wp = this->dPtr->WriteString ( wp, gsOut, this->cs.tn ) ;
               remCols -= (gsOut.gscols() + 2) ;
               break ;
         } ;
         wp.xpos += fldSPC ;        // step to next field position
         if (   (i == (this->LastActiveField ())) 
             && (gsOut.gscols() == stdFIELD) )
         {
            this->tData.sffLastField = lastFlag = true ;
         }

         //* If we are out of horizontal space OR if last *
         //* field is fully displayed, then we're done.   *
         if ( (remCols < fldSPC) || (lastFlag != false) )
            break ;
      }
      wp.ypos += 2 ; wp.xpos = metahdrCOL ;

      //* If window is filled, exit the loop *
      if ( wp.ypos > (this->termRows - 3) )
         break ;
   }
   this->dPtr->RefreshWin () ;

}  //* End UpdateMetadataWindow() *

//*************************
//*   UpdateDataWindows   *
//*************************
//******************************************************************************
//* Format and redisplay the data for both the filename window and the         *
//* metadata-field window.                                                     *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void Taggit::UpdateDataWindows ( void )
{

   this->UpdateFilenameWindow () ;  // display filename list
   this->UpdateMetadataWindow () ;  // display metadata matrix

}  //* End UpdateDataWindow() *

//*************************
//*    NextActiveField    *
//*************************
//******************************************************************************
//* Scan from 'sffOffset' forward to the next active tag field.                *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: index of next active field                                        *
//******************************************************************************

short Taggit::NextActiveField ( void )
{
   short naIndex = this->tData.sffOffset ;   // return value

   if ( naIndex < (tfCOUNT - 1) )
   {
      while ( (this->tData.sffDisp[++naIndex] == false) && (naIndex < tfCOUNT) ) ;
      if ( naIndex == tfCOUNT )     // if we overran the array, return current index
         naIndex = this->tData.sffOffset ;
   }
   return naIndex ;

}  //* End NextActiveField() *

//*************************
//*    PrevActiveField    *
//*************************
//******************************************************************************
//* Scan from 'sffOffset' backward to the previous active tag field.           *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: index of previous active field                                    *
//******************************************************************************

short Taggit::PrevActiveField ( void )
{
   short paIndex = this->tData.sffOffset ; // return value

   if ( paIndex > ZERO )
   {
      while ( (this->tData.sffDisp[--paIndex] == false) && (paIndex > ZERO) ) ;
   }
   return paIndex ;

}  //* End PrevActiveField() *

//*************************
//*    LastActiveField    *
//*************************
//******************************************************************************
//* Scan the list of tag fields and return the index of the last (rightmost)   *
//* active field. This will be the last field displayed in the field window.   *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: index of last active field                                        *
//******************************************************************************
//* Programmer's Note: We scan backward through the field list beginning with  *
//* the last text field i.e. (tfCOUNT - 1).                                    *
//*                                                                            *
//******************************************************************************

short Taggit::LastActiveField ( void )
{
   short laIndex = (tfCOUNT - 1) ;  // return value

   while (   (this->tData.sffDisp[laIndex] == false)
          && (laIndex > ZERO) )
      --laIndex ;

   return laIndex ;

}  //* End LastActiveField() *

#if DIALOG_DEBUG != 0   // FOR DEBUGGING ONLY
//*************************
//*       DebugMsg        *
//*************************
//******************************************************************************
//* DEBUGGING ONLY:                                                            *
//* Optionally clear some space in the UI window, and then write the specified *
//* message.                                                                   *
//*                                                                            *
//* Input  : gsMsg  : message to be displayed                                  *
//*          clear  : (optional, 'false' by default) if 'true', clear msg area *
//*          dbLine : (optional, ZERO by default) line offset                  *
//*          cAttr  : (optional, nc.blR by default) color attribute for message*
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void Taggit::DebugMsg ( const gString& gsMsg, bool clear, short dbLine, attr_t cAttr )
{
   winPos wp( 1, 102 ) ;      // debug message position

   if ( clear )
      this->dPtr->ClearArea ( wp.ypos, wp.xpos, 3, (this->termCols - 103), nc.blR ) ;

   wp.ypos += dbLine ;
   ++wp.xpos ;
   this->dPtr->WriteParagraph ( wp, gsMsg, cAttr, true ) ;

}  //* End DebugMsg() *

void Taggit::DebugMsg ( const char* csMsg, bool clear, short dbLine, attr_t cAttr )
{
   gString gs( csMsg ) ;
   this->DebugMsg ( gs, clear, dbLine, cAttr ) ;

}  //* End DebugMsg() *
#endif   // DIALOG_DEBUG

//*************************
//*     UserInterface     *
//*************************
//******************************************************************************
//* User interface loop. Gather user input and grant user requests.            *
//*                                                                            *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: OK                                                                *
//******************************************************************************
//* --                                                                         *
//* --                                                                         *
//* --                                                                         *
//*                                                                            *
//* Hotkeys:                                                                   *
//*                                                                            *
//*                                                                            *
//*                                                                            *
//* -- CTRL+R           For RTL user-interface languages,                      *
//*                     toggle the cfgOpt.rtlr flag                            *
//*                                                                            *
//*                                                                            *
//******************************************************************************

short Taggit::UserInterface ( void )
{
   gString gsOut ;
   wkeyCode wk ;
   bool handled = false,
        done = false ;

   //* Metadata (if any) has been loaded since the *
   //* dialog was opened. Update the display.      *
   this->UpdateDataWindows () ;

   //* If external image file specified, *
   //* prompt user for needed info.      *
   if ( *this->cfgOpt.extImage.picPath != NULLCHAR )
      this->Cmd_InsertImage ( this->tData.sfFocus ) ;

   while ( ! done )
   {
      handled = false ;                      // reset the flag
      this->dPtr->GetKeyInput ( wk ) ;       // get user input

      //* If user has resized the terminal window,         *
      //* close and re-open the dialog window for new size *
      if ( (wk.type == wktFUNKEY) && (wk.key == nckRESIZE) )
      {
         done = this->ResizeDialog () ;
         continue ;
      }

      //* If mouse event, convert it to equivalent keycode *
      if ( wk.type == wktMOUSE )
      {
#if 0    // TEMP TEMP TEMP
this->dPtr->CaptureDialog ( "capturedlg.txt" ) ;
this->dPtr->CaptureDialog ( "capturedlg.html", true, false, 
                            "infodoc-styles.css", 4, false, nc.cyR ) ;
#endif   // TEMP TEMP TEMP
         #if DIALOG_DEBUG != 0   // FOR DEBUGGING ONLY
         gsOut.compose( "MOUSE: %02hd/%02hd meType:%hX", 
                        &wk.mevent.ypos, &wk.mevent.xpos, &wk.mevent.meType ) ;
         this->DebugMsg ( gsOut, true ) ;
         #endif   // DIALOG_DEBUG

         //* If mouse event occurred in the controls window *
         if ( wk.mevent.ypos < fieldwinY )
         {
            done = this->uiControls ( wk ) ;
            handled = true ;
         }
         else
         {
            //* Test for scroll-wheel event *
            if ( wk.mevent.meType == metSW_D )
            {  //* In filename and field windows, scroll up, UNLESS *
               //*  mouse is in field window and Ctrl key is down,  *
               //* in which case, scroll left.                      *
               wk.type = wktFUNKEY ;
               wk.key = nckUP ;
               if ( wk.mevent.cKey && (wk.mevent.xpos >= fieldwinX) )
                  wk.key = nckLEFT ;
            }
            else if ( wk.mevent.meType == metSW_U )
            {  //* In filename and field windows, scroll down, UNLESS *
               //*  mouse is in field window and Ctrl key is down,    *
               //* in which case, scroll right.                       *
               wk.type = wktFUNKEY ;
               wk.key = nckDOWN ;
               if ( wk.mevent.cKey && (wk.mevent.xpos >= fieldwinX) )
                  wk.key = nckRIGHT ;
            }
            //* Test for click events *
            else if ( (wk.mevent.meType == metB1_S) ||
                    (wk.mevent.meType == metB1_D) ||
                    (wk.mevent.meType == metB1_T) )
            {  //* Currently Button-1 single, double and triple    *
               //* clicks all mean the same thing.                 *
               //* Modifier keys affect the interpretation:        *
               //* CTRL: go all the way in the specified direction *

               //* Top border of field window *
               if ( wk.mevent.ypos == fieldwinY )
               {
                  wk.type = wktFUNKEY ;
                  if ( wk.mevent.cKey )      // if CTRL modifier key
                     wk.key = nckHOME ;
                  else
                     wk.key = nckUP ;
               }

               //* Bottom border of field window *
               else if ( wk.mevent.ypos == (fieldwinY + this->fwinRows + 1) )
               {
                  wk.type = wktFUNKEY ;
                  if ( wk.mevent.cKey )      // if CTRL modifier key
                     wk.key = nckEND ;
                  else
                     wk.key = nckDOWN ;
               }

               //* Left border of field window OR left border of dialog *
               else if ( (wk.mevent.xpos == ZERO) || (wk.mevent.xpos == fieldwinX) )
               {
                  wk.type = wktFUNKEY ;
                  if ( wk.mevent.cKey )      // if CTRL modifier key
                     wk.key = nckCLEFT ;
                  else
                     wk.key = nckLEFT ;
               }

               //* Right border of field window *
               else if ( wk.mevent.xpos == (fieldwinX + this->fwinCols + 1) )
               {
                  wk.type = wktFUNKEY ;
                  if ( wk.mevent.cKey )      // if CTRL modifier key
                     wk.key = nckCRIGHT ;
                  else
                     wk.key = nckRIGHT ;
               }

               else
               {
                  //* Calculate whether click indicates a file to be edited *
                  short fnIndex = this->uiItemSelected ( wk ) ;
                  if ( fnIndex >= ZERO )
                  {
                     //* Interior of filename window *
                     if ( wk.mevent.xpos < fieldwinX )
                     {
                        this->uiEditFilename () ;
                     }
                     //* Interior of field window *
                     else
                     {
                        short fld = this->uiFieldSelected ( wk ) ;
                        this->uiEditMetadata ( fld ) ;
                     }
                  }
                  handled = true ;  // event handled
               }
            }
            else
            {
               //* This is unlikely because all other mouse events are *
               //* filtered inside the mouse-translation interface.    *
            }
         }
      }

      if ( ! handled && (wk.type == wktFUNKEY) )
      {
         //*************
         //*  UpArrow  *
         //*************
         if ( wk.key == nckUP )
         {
            if ( this->tData.sfFocus > ZERO )
            {
               --this->tData.sfFocus ;
               if ( this->tData.sfFocus < this->tData.sfFirst )
               {
                  --this->tData.sfFirst ;
                  --this->tData.sfLast ;
               }
               this->UpdateDataWindows () ;
            }
            else
               this->dPtr->UserAlert () ;
         }

         //*************
         //* DownArrow *
         //*************
         else if ( wk.key == nckDOWN )
         {
            if ( this->tData.sfFocus < (this->tData.sfCount - 1) )
            {
               ++this->tData.sfFocus ;
               if ( this->tData.sfFocus > this->tData.sfLast )
               {
                  ++this->tData.sfFirst ;
                  ++this->tData.sfLast ;
               }
               this->UpdateDataWindows () ;
            }
            else
               this->dPtr->UserAlert () ;
         }

         //************************
         //* Home OR CTRL+UpArrow *
         //************************
         else if ( wk.key == nckHOME || wk.key == nckCUP )
         {
            if ( (this->tData.sfFirst > ZERO) || (this->tData.sfFocus > ZERO) )
            {
               this->tData.sfFirst = this->tData.sfFocus = ZERO ;
               this->InitIndices () ;
               this->UpdateDataWindows () ;
            }
            else
               this->dPtr->UserAlert () ;
         }

         //*************************
         //* End OR CTRL+DownArrow *
         //*************************
         else if ( wk.key == nckEND || wk.key == nckCDOWN )
         {
            if (    (this->tData.sfLast < (this->tData.sfCount - 1))
                 || (this->tData.sfFocus < (this->tData.sfCount - 1)) )
            {
               this->tData.sfLast = this->tData.sfFocus = (this->tData.sfCount - 1) ;
               this->tData.sfFirst = this->tData.sfLast - 1 ;  // dummy value
               this->InitIndices () ;
               this->UpdateDataWindows () ;
            }
            else
               this->dPtr->UserAlert () ;
         }

         //*************
         //* LeftArrow *
         //*************
         else if ( wk.key == nckLEFT )
         {
            if ( this->tData.sffOffset > ZERO )
            {  //* Step over disabled fields *
               this->tData.sffOffset = this->PrevActiveField () ;
               this->UpdateMetadataWindow () ;
            }
            else     //* Already at head of list *
               this->dPtr->UserAlert () ;
         }

         //**************
         //* RightArrow *
         //**************
         else if ( wk.key == nckRIGHT )
         {
            if ( ! this->tData.sffLastField )
            {  //* Step over disabled fields *
               this->tData.sffOffset = this->NextActiveField () ;
               this->UpdateMetadataWindow () ;
            }
            else     //* Already at tail of list *
               this->dPtr->UserAlert () ;
         }

         //******************
         //* CTRL+LeftArrow *
         //******************
         else if ( wk.key == nckCLEFT )
         {
            if ( this->tData.sffOffset > ZERO )
            {
               this->tData.sffOffset = ZERO ;
               this->UpdateMetadataWindow () ;
            }
            else     //* Already at head of list *
               this->dPtr->UserAlert () ;
         }

         //*******************
         //* CTRL+RightArrow *
         //*******************
         else if ( wk.key == nckCRIGHT )
         {
            if ( ! this->tData.sffLastField )
            {
               // Programmer's Note: This operation is a bit tricky.
               // We want to fully display the last active field; BUT do not 
               // want to push all the preceeding fields out of the window.
               // This is easy when moving one field at a time (see RightArrow),
               // but we do not know which active fields will be displayed 
               // simultaneously with the last active field.
               // We COULD do this calculation as a special case, but we choose
               // to leverage the existing code. To do this, we index an active 
               // field a "safe" distance before the last active field. In this 
               // case, "safe" means that we assume that even a very wide window 
               // can display no more than 5 fields, so we start there and walk 
               // to the end, monitoring the 'sffLastField' flag.
               this->tData.sffOffset = this->LastActiveField () ;
               for ( short i = 5 ; i > ZERO ; --i )
                  this->tData.sffOffset = this->PrevActiveField () ;
               this->UpdateMetadataWindow () ;
               while ( ! this->tData.sffLastField )
               {
                  this->tData.sffOffset = this->NextActiveField () ;
                  this->UpdateMetadataWindow () ;
               }
            }
            else     //* Already at tail of list *
               this->dPtr->UserAlert () ;
         }

         //************
         //*  PageUp  *
         //************
         else if ( wk.key == nckPGUP )
         {
            if (   (this->tData.sfFirst > ZERO)
                || (this->tData.sfFocus > this->tData.sfFirst) )
            {
               if ( this->tData.sfFocus > this->tData.sfFirst )
                  this->tData.sfFocus = this->tData.sfFirst ;
               else
               {
                  short loopy = this->tData.sfLast - this->tData.sfFirst + 1,
                        rem   = this->tData.sfFirst ;
                  if ( rem < loopy )
                     loopy = rem ;
                  do
                  {
                     --this->tData.sfLast ;
                     --this->tData.sfFirst ;
                  }
                  while ( --loopy > ZERO ) ;
                  this->tData.sfFocus = this->tData.sfFirst ;
               }
               this->UpdateDataWindows () ;
            }
            else     //* Already at head of list *
               this->dPtr->UserAlert () ;
         }

         //************
         //* PageDown *
         //************
         else if ( wk.key == nckPGDOWN )
         {
            if (   (this->tData.sfLast < (this->tData.sfCount - 1))
                || (this->tData.sfFocus < this->tData.sfLast) )
            {
               if ( this->tData.sfFocus < this->tData.sfLast )
                  this->tData.sfFocus = this->tData.sfLast ;
               else
               {
                  short loopy = this->tData.sfLast - this->tData.sfFirst + 1,
                        rem   = ((this->tData.sfCount - 1) - this->tData.sfLast) ;
                  if ( rem < loopy )
                     loopy = rem ;
                  do
                  {
                     ++this->tData.sfLast ;
                     ++this->tData.sfFirst ;
                  }
                  while ( --loopy > ZERO ) ;
                  this->tData.sfFocus = this->tData.sfLast ;
               }
               this->UpdateDataWindows () ;
            }
            else     //* Already at tail of list *
               this->dPtr->UserAlert () ;
         }

         //*******************
         //*     CTRL+R      *
         //*******************
         else if ( wk.key == nckC_R )
         {  //* For RTL language user interface only, toggle  *
            //* display of data in fields between LTR and RTL.*
            this->Cmd_RTL_ToggleFields () ;
         }

         //* Auto-translated mouse event, hotkey, shortcut key, etc. *
         else
         {
            done = this->uiControls ( wk ) ;
            handled = true ;
         }
      }

      else if ( ! handled && wk.type == wktEXTEND )
      {
         done = this->uiControls ( wk ) ;
         handled = true ;
      }

      //* Not handled and not wktFUNKEY/wktEXTEND *
      else if ( !handled )
      {
         #if DIALOG_DEBUG != 0   // FOR DEBUGGING ONLY
         gsOut.compose( "!HANDLED: type:%02hd key:%04X\n"
                        "          %02hd/%02hd meType:%hX",
                        &wk.type, &wk.key,
                        &wk.mevent.ypos, &wk.mevent.xpos, &wk.mevent.meType ) ;
         this->DebugMsg ( gsOut, true ) ;
         #endif   // DIALOG_DEBUG

         //* Most printing characters cannot be commands; however,    *
         //* a) the SPACE key indicates a request to edit a tag field *
         if ( wk.type == wktPRINT )
         {
            done = this->uiControls ( wk ) ;
         }
      }

      //* If user has requested to close the application, check *
      //* whether edits are pending and if so, warn the user.   *
      if ( done )
      {
         short modFiles = this->EditsPending () ;
         if ( modFiles > ZERO )
         {  //* Ask user what to do *
            short action = this->uiEditsPending ( modFiles ) ;
            if ( action > ZERO )          // save to target(s) and exit
            {
               this->Cmd_WriteTargetFile () ; // write all files with edits pending
            }
            else if ( action == ZERO )    // continue editing
            {
               done = false ;    // user has changed his/her/its mind
            }
            else                          // discard edits and exit
            { /* nothing left to be done */ }
         }
      }
   }     // while()

   return OK ;

}  //* End UserInterface() *

//*************************
//*      uiControls       *
//*************************
//******************************************************************************
//* Determine whether the mouse click or keystroke references one of the       *
//* user-interface controls. If so, call the corresponding EditXX() method.    *
//*                                                                            *
//* Input  : wk  : user input                                                  *
//*                - if mouse click, caller has verified that click is         *
//*                  _inside_ the controls' window.                            *
//*                                                                            *
//* Returns: 'true' if user selected 'Exit' from the menu, else 'false'        *
//******************************************************************************

bool Taggit::uiControls ( wkeyCode& wk )
{
   uiInfo Info ;                    // user-interface data
   bool   status = false ;          // return value

   //* If the mouse brought us here *
   if ( wk.type == wktMOUSE )
   {
      //* Test for scroll-wheel event (ignored in controls window) *
      if ( !(wk.mevent.meType == metSW_U || wk.mevent.meType == metSW_D) )
      {
         //* If user has indicated a menu title, place input focus *
         //* on indicated menu and call the edit routine.          *
         if ( (wk.mevent.cIndex >= twFileMW) && (wk.mevent.cIndex <= twHelpMW) )
         {
            this->dPtr->FlushKeyInputStream () ;
            status = this->uiEditControl ( wk, wk.mevent.cIndex ) ;
         }

         //* If user has indicated the 'Save All' pushbutton *
         else if ( wk.mevent.cIndex == twSavePB )
         {
            status = this->uiEditControl ( wk, wk.mevent.cIndex ) ;
         }
      }
      else
      { /* Ignore scroll-wheel event */ }
   }

   //* Keystroke OR translated mouse event *
   else
   {
      //* Extended-key-definition group *
      if ( wk.type == wktEXTEND )
      {
         //* Menu-invocation group *
         //* File Menu *
         if ( (wk.key == nckA_F) || (wk.key == nckAS_F) )
            status = this->uiEditControl ( wk, twFileMW ) ;

         //* Edit Menu *
         else if ( (wk.key == nckA_E) || (wk.key == nckAS_E) )
            status = this->uiEditControl ( wk, twEditMW ) ;

         //* View Menu *
         else if ( (wk.key == nckA_V) || (wk.key == nckAS_V) )
            status = this->uiEditControl ( wk, twViewMW ) ;

         //* Help Menu *
         else if ( (wk.key == nckA_H) || (wk.key == nckAS_H) )
            status = this->uiEditControl ( wk, twHelpMW ) ;

         //* Edit the filename field for highlighted file.*
         //* NOTE: This is ALT+ENTER or ALT+CTRL+ENTER or *
         //* ALT+CTRL+SHIFT+ENTER or ALT+N                *
         else if ( (wk.key == nckAC_J) || (wk.key == nckA_N) )
            this->uiEditFilename () ;
         else if ( wk.key == nckA_M )
            this->uiEditMetadata ( this->tData.sffOffset ) ;

         //* ALT+CTRL+S: Save file as *
         else if ( wk.key == nckAC_S )
         {
            if ( (this->uiEditFilename ()) != false )
               this->Cmd_WriteTargetFile ( this->tData.sfFocus ) ;
         }

         //* View source file images *
         else if ( wk.key == nckA_I )
         {
            this->Cmd_ReportImage ( this->tData.sfFocus ) ;
         }

         //* Insert image into target file *
         else if ( wk.key == nckAC_I )
         {
            this->Cmd_InsertImage ( this->tData.sfFocus ) ;
         }

         //* Edit embedded-image stat data *
         else if ( wk.key == nckAC_E )
         {
            this->Cmd_InsertImage ( this->tData.sfFocus, true ) ;
         }

         //* Unrecognized key command *
         else
            this->dPtr->UserAlert () ;
      }

      //* 'FUNKEY' command-key combinations *
      else if ( wk.type == wktFUNKEY )
      {
         //* Rearrange items: move highlighted item up or down.*
         if ( (wk.key == nckAUP) || (wk.key == nckASUP) ||
                   (wk.key == nckADOWN) || (wk.key == nckASDOWN) )
         {
            wkeyCode pushKey( nckUP, wktFUNKEY ) ;
            bool updateDisplay = false ;
            if ( (wk.key == nckAUP) || (wk.key == nckASUP) )
               updateDisplay = this->Cmd_SwapItems ( this->tData.sfFocus, 
                                                   this->tData.sfFocus - 1 ) ;
            else
            {
               updateDisplay = this->Cmd_SwapItems ( this->tData.sfFocus, 
                                                   this->tData.sfFocus + 1 ) ;
               pushKey.key = nckDOWN ;
            }

            //* If swap was successful, update the display by pushing *
            //* the appropriate command into the key-input queue.     *
            if ( updateDisplay )
            {
               this->dPtr->UngetKeyInput ( pushKey ) ;
            }
         }

         //* Edit the tag field at left of field window for highlighted file.*
         //* (This is the clunky way of invoking the edit routine. Invoking) *
         //* (the edit with the mouse will indicate the actual field offset) *
         else if ( wk.key == nckENTER || wk.key == SPACE )
         {
            this->uiEditMetadata ( this->tData.sffOffset ) ;
         }

         //**************************************
         //*  Defined control-key combinations  *
         //**************************************
         //* CTRL+S: Save file       *
         else if ( wk.key == nckC_S )
         {
            this->Cmd_WriteTargetFile ( this->tData.sfFocus ) ;
         }

         //* CTRL+D: Duplicate field *
         else if ( wk.key == nckC_D )
         {
            Cmd_DuplicateField () ;
         }

         //* CTRL+T: Autosequence tracks *
         else if ( wk.key == nckC_T )
         {
            this->Cmd_SequenceTracks () ;
         }

         //* CTRL+Z: Undo Tag edits *
         else if ( wk.key == nckC_Z )
         {
            #if SIMULATED_DATA == 0 // PRODUCTION
            this->crmRefreshMetadata ( this->tData.sfFocus ) ;
            #else                   // SIMULATED_DATA
            this->InitSimulatedData ( this->tData.sfFocus ) ;
            this->UpdateDataWindows () ;
            #endif                  // SIMULATED_DATA
         }

         //* CTRL+P: Playback of highlighted file *
         else if ( wk.key == nckC_P )
         {
            this->Cmd_Playback ( this->tData.sfFocus ) ;
         }

         //* CTRL+K: Undocumented debugging options *
         else if ( wk.key == nckC_K )
         {
            #if 1    // DEBUGGING ONLY - BUT WILL PROBABLY LEAVE IT ACTIVE
            //* The function key must be followed *
            //* by a digit, else it is ignored.   *
            this->dPtr->GetKeyInput ( wk ) ;
            if ( (wk.type == wktPRINT) && ((wk.key >= '0') && (wk.key <= '9')) )
            {
               switch ( wk.key )
               {
                  case '0':
                     break ;
                  case '1':
#if 0    // EXPERIMENTAL - TEST "SPECIAL" CHARACTERS IN FILENAMES.
                     {
                        const wchar_t* tstTxt[] = 
                        {
                           L"(~/Qu\"ack/N'oodle*$HOME/Brew?)",
                           L"~/Documents/My Diary (2012) and (most secret) thoughts().odt",
                           L"${HOME};(horses && cows);<ducks || geese>",
                        } ;
                        gString gs( tstTxt[0] ) ;
                        winPos wp( 12, 43 ) ;
                        this->dPtr->WriteString ( wp.ypos++, wp.xpos, gs, nc.blR, true ) ;
                        this->Realpath ( gs ) ;
                        this->dPtr->WriteString ( wp.ypos++, wp.xpos, gs, nc.blR, true ) ;

                        gs = tstTxt[1] ; ++wp.ypos ;
                        this->dPtr->WriteString ( wp.ypos++, wp.xpos, gs, nc.blR, true ) ;
                        this->Realpath ( gs ) ;
                        this->dPtr->WriteString ( wp.ypos++, wp.xpos, gs, nc.blR, true ) ;

                        gs = tstTxt[2] ; ++wp.ypos ;
                        this->dPtr->WriteString ( wp.ypos++, wp.xpos, gs, nc.blR, true ) ;
                        this->Realpath ( gs ) ;
                        this->dPtr->WriteString ( wp.ypos++, wp.xpos, gs, nc.blR, true ) ;
                     }
#endif   // EXPERIMENTAL
                     break ;
                  case '2':
                     {  //* Test access to the HTML-formatted documentation *
                        gString gs( "%s/Texinfo/taggit.html", this->cfgOpt.appPath ) ;
                        this->LaunchDefaultApplication ( gs.ustr() ) ;
                        break ;
                     }
                  case '3':
                     break ;
                  case '4':
                     break ;
                  case '5':
                     //* Toggle the 'rtl' flag.                               *
                     //* Test cursor positioning and output for RTL languages.*
                     #if FAKE_RTL != 0 
                     this->cfgOpt.rtl = this->cfgOpt.rtl ? false : true ;
                     this->ResizeDialog ( true ) ;
                     #endif   // FAKE_RTL 
                     break ;
                  case '6':
                     break ;
                  case '7':
                     break ;
                  case '8':
                     break ;
                  case '9':
                     break ;
               }
            }
            else
               this->dPtr->UngetKeyInput ( wk ) ;
            #endif   // DEBUGGING ONLY - BUT WILL PROBABLY LEAVE IT ACTIVE
         }
         //* Quit: exit the application *
         else if ( wk.key == nckC_Q )
         {  //* Alert caller that user wants to exit *
            status = true ;
         }

         //* Unrecognized key command *
         else
            this->dPtr->UserAlert () ;
      }

      //* 'PRINT' i.e. printing characters *
      else if ( wk.type == wktPRINT )
      {
         if ( wk.key == SPACE )
            this->uiEditMetadata ( this->tData.sffOffset ) ;

         else     // all other printing characters are currently ignored
         {
            this->dPtr->UserAlert () ;
         }
      }

      //* Unrecognized or garbled key input *
      else
         this->dPtr->UserAlert () ;
   }

   return status ;

}  //* End uiControls() *

//*************************
//*     uiEditControl     *
//*************************
//******************************************************************************
//* Call the EditXXX() method for the specified control.                       *
//* If user makes a selection, honor the user's request.                       *
//*                                                                            *
//* Input  : wk     : key or mouse input that brought us here                  *
//*          ctrlObj: control object to be edited                              *
//*                   NOTE: This is actually a member of enum topWinControls.  *
//*                                                                            *
//* Returns: 'true' if user selected 'Exit' from the menu, else 'false'        *
//******************************************************************************

bool Taggit::uiEditControl ( wkeyCode& wk, short ctrlObj )
{
   uiInfo Info ;                    // user-interface data
   bool status = false ;            // return value

   //* Place focus on target control *
   while ( (this->dPtr->NextControl ()) != ctrlObj ) ;
   Info.viaHotkey = true ;          // selected via hotkey (or mouse equivalent)

   //* NOTE: This Pushbutton is accessible ONLY via mouse click.       *
   //* User cannot TAB/STAB to it nor can it be selected via hotkey.   *
   //* This Pushbutton is equivalent to selecting the 'Save All' item  *
   //* in the File Menu. Therefore, if user clicks this button, we set *
   //* the 'Info' parameters to simulate selection of that option.     *
   if ( ctrlObj == twSavePB )             
   {
      Info.ctrlIndex = twFileMW ;
      Info.selMember = ZERO ;
      Info.dataMod = true ;
   }
   else
   {
      this->dPtr->EditMenuwin ( Info ) ;
   }
   #if 0    // DUMP Info object - DEBUG ONLY
   this->dPtr->SetDialogObscured () ;
   winPos wp( 5, 1 ) ;
   this->dPtr->Dump_uiInfo ( wp, Info ) ;
   this->dPtr->RefreshWin () ;
   #endif   // DUMP Info object - DEBUG ONLY

   //* Move focus away from MenuBar *
   while ( (this->dPtr->NextControl ()) != twSavePB ) ;

   //* If user made a selection *
   if ( Info.dataMod )
   {
      //* Translate menu selection index to MenuCode enum value *
      MenuCode selection = MenuItem2MenuCode ( Info.ctrlIndex, Info.selMember ) ;

      //* Perform the selected operation *
      switch ( selection )
      {
         //***************
         //** File Menu **
         //***************
         case mcFileMB_SAVEALL:     //* Save metadata for all files
            this->Cmd_WriteTargetFile () ;
            break ;
         case mcFileMB_SAVEFILE:    //* Save metadata for highlighted file
            this->Cmd_WriteTargetFile ( this->tData.sfFocus ) ;
            break ;
         case mcFileMB_SAVEAS:      //* Save (with rename) for highlighted file
            //* Get a new filename for the target from *
            //* the user, then call the save method.   *
            if ( (this->uiEditFilename ()) != false )
               this->Cmd_WriteTargetFile ( this->tData.sfFocus ) ;
            break ;
         case mcFileMB_SAVEAUDIO:   //* Save audio data only for highlighted file
            this->Cmd_WriteTargetFile ( this->tData.sfFocus, true ) ;
            break ;
         case mcFileMB_SUMMARY:     //* Write metadata summary to a (text) file
            this->Cmd_TagSummary ( false, true, true, true ) ;
            break ;
         case mcFileMB_PLAYBACK:    //* Open the default media player for playback
            this->Cmd_Playback ( this->tData.sfFocus ) ;
            break ;
         case mcFileMB_REFRESH:     //* Refresh (reread) all displayed data
            this->Cmd_RefreshMetadata () ;
            break ;
         case mcFileMB_SHELL:       //* go to command shell
            this->Cmd_ShellOut () ;
            break ;
         case mcFileMB_EXIT:        //* exit the application
            status = true ;
            break ;

         //***************
         //** Edit Menu **
         //***************
         case mcEditMB_FILENAME:   //* Edit filename of highlighted item
            this->uiEditFilename () ;
            break ;
         case mcEditMB_METADATA:   //* Edit metadata tag fields of highlighted item
            this->uiEditMetadata ( this->tData.sffOffset ) ;
            break ;
         case mcEditMB_IMAGE:      //* Add/modify an image tag for highlighted item
            this->Cmd_InsertImage ( this->tData.sfFocus ) ;
            break ;
         case mcEditMB_IEDIT:      //* Edit stats for an existing image tag
            this->Cmd_InsertImage ( this->tData.sfFocus, true ) ;
            break ;
         case mcEditMB_REFRESH:    //* Undo edits for highlighted file
            #if SIMULATED_DATA == 0 // PRODUCTION
            this->crmRefreshMetadata ( this->tData.sfFocus ) ;
            #else                   // SIMULATED_DATA
            this->InitSimulatedData ( this->tData.sfFocus ) ;
            this->UpdateDataWindows () ;
            #endif                  // SIMULATED_DATA
            break ;
         case mcEditMB_TRACKFILL:   //* Auto-fill track (tfTRCK) field
            this->Cmd_SequenceTracks () ;
            break ;
         case mcEditMB_DUPLICATE:   //* Duplicate contents of a field to all files
            this->Cmd_DuplicateField () ;
            break ;
         case mcEditMB_SETTITLE:    //* For all files, set Title field from filename
            this->Cmd_SetSongTitle () ;
            break ;
         case mcEditMB_POPMETER:    //* Popularimeter and Play counter
            this->Cmd_Popularimeter () ;
            break ;
         case mcEditMB_CLEARFILE:   //* Clear (erase) all metadata fields for current file
            this->ccmClearMetadata ( this->tData.sfFocus ) ;
            break ;
         case mcEditMB_CLEARALL:    //* Clear (erase) all metadata from all files
            this->Cmd_ClearMetadata () ;
            break ;
         case mcEditMB_PREFS:       // Request to open Preferences dialog
            break ;

         //***************
         //** View Menu **
         //***************
         case mcViewMB_FSELECT:     //* Set all text tag fields active
            this->Cmd_ActiveFields ( true, true ) ;
            break ;
         case mcViewMB_SUMMARY:     //* View metadata summary for file
            this->Cmd_TagSummary ( true, false ) ;
            break ;
         case mcViewMB_SORTLIST:    //* Sort the file list
            break ;
         case mcViewMB_SHIFTDOWN:   //* Move highlighted file downward in list
            if ( this->Cmd_SwapItems ( this->tData.sfFocus, this->tData.sfFocus + 1 ) )
            {
               wkeyCode pushKey( nckDOWN, wktFUNKEY ) ;
               this->dPtr->UngetKeyInput ( pushKey ) ;
            }
            break ;
         case mcViewMB_SHIFTUP:     //* Move highlighted file upward in list
            if ( this->Cmd_SwapItems ( this->tData.sfFocus, this->tData.sfFocus - 1 ) )
            {
               wkeyCode pushKey( nckUP, wktFUNKEY ) ;
               this->dPtr->UngetKeyInput ( pushKey ) ;
            }
            break ;
         case mcViewMB_FREESPACE:   //* Report the target free space
            this->Cmd_ReportTargetStats () ;
            break ;
         case mcViewMB_IMAGE:       //* View embedded-image data
            this->Cmd_ReportImage ( this->tData.sfFocus ) ;
            break ;
         case mcViewMB_TOGGLE:      //* Toggle LTR/RTL field contents
            this->Cmd_RTL_ToggleFields () ;
            break ;
         case mcViewMB_CSCHEME:     //* Set the dialog color scheme
            this->Cmd_SetColorScheme () ;
            break ;

         //***************
         //** Help Menu **
         //***************
         case mcHelpMB_F01HELP:     //* request to open info reader help
            this->Cmd_InfoHelp ( false ) ;
            break ;
         case mcHelpMB_HTML:        //* request to open browser help
            this->Cmd_InfoHelp ( true ) ;
            break ;
         case mcHelpMB_QUICKHELP:   //* display help summary
           this->Cmd_QuickHelp () ;
            break ;
         case mcHelpMB_HELPABOUT:   //* display Help-about dialog
            this->Cmd_HelpAbout () ;
            break ;


         //*******************
         //** Sort Sub-menu **
         //*******************
         case mcSortMB_FNAME:       //* Sort by 'Filename'
         case mcSortMB_TITLE:       //* Sort by 'Title'
         case mcSortMB_ALBUM:       //* Sort by 'Album'
         case mcSortMB_TRACK:       //* Sort by 'Track'
         case mcSortMB_ARTIST:      //* Sort by 'Artist'
            {
            bool resort = true ;
            if ( (selection == mcSortMB_FNAME) && (this->cfgOpt.sortBy != sbFNAME) )
               this->cfgOpt.sortBy = sbFNAME ;
            else if ( (selection == mcSortMB_TITLE) && (this->cfgOpt.sortBy != sbTITLE) )
               this->cfgOpt.sortBy = sbTITLE ;
            else if ( (selection == mcSortMB_ALBUM) && (this->cfgOpt.sortBy != sbALBUM) )
               this->cfgOpt.sortBy = sbALBUM ;
            else if ( (selection == mcSortMB_TRACK) && (this->cfgOpt.sortBy != sbTRACK) )
               this->cfgOpt.sortBy = sbTRACK ;
            else if ( (selection == mcSortMB_ARTIST) && (this->cfgOpt.sortBy != sbARTIST) )
               this->cfgOpt.sortBy = sbARTIST ;
            else
               resort = false ;
            if ( resort )
            {
               this->Cmd_SortList () ;
               this->UpdateDataWindows () ;
            }
            }
            break ;

         //* All valid codes are handled, so should never arrive here.*
         default:
            break ;
      }
   }
   return status ;

}  //* End uiEditControl() *

//*************************
//*    uiItemSelected     *
//*************************
//******************************************************************************
//* Caller has determined that the mouse click is in the interior of either    *
//* the filename sub-window or the tag-fields sub-window.                      *
//*                                                                            *
//* a) Map the click row to the corresponding item.                            *
//* b) If focus is not already on selected item, set the focus.                *
//* c) Return index of selected item.                                          *
//*                                                                            *
//* Input  : wk  : user input                                                  *
//*                - caller has verified that the click is _inside_ the        *
//*                  filename window                                           *
//*                                                                            *
//* Returns: index of selected item                                            *
//*          (-1) if mouse event does not indicate an item to be edited        *
//******************************************************************************
//* -- The row containing display data for an item OR the row which follows it *
//*    i.e. the underline row, indicates that user wants to edit that item.    *
//* -- A click in the first (header) row of the sub-window is ignored.         *
//* -- A click in any empty rows below the last displayed item (and its        *
//*    underline row) do not indicate item selection and so are ignored.       *
//*                                                                            *
//******************************************************************************

short Taggit::uiItemSelected ( const wkeyCode& wk )
{
   #define DEBUG_UIIS (0)              // For debugging only

   short fnIndex = -1,                 // return value
         firstIRow = (fieldwinY + 2),  // first and last rows containing records
         lastIRow  = firstIRow + this->tData.sfCount * 2 - 1,
         offset    = wk.mevent.ypos - firstIRow ;
   if ( lastIRow >= (this->termRows - 1) )
      lastIRow = (this->termRows - 2) ;

   #if DEBUG_UIIS != 0     // DEBUG ONLY
   gString gsx( "uiItemSelected: %02hd:%02hd", &wk.mevent.ypos, &wk.mevent.xpos ) ;
   this->DebugMsg ( gsx, true ) ;
   winPos wpx( short(fieldwinY + 1), 1 ) ;
   this->dPtr->WriteString ( firstIRow, wpx.xpos + 2, L"=>", nc.reG ) ;
   this->dPtr->WriteString ( lastIRow, wpx.xpos + 2, L"=>", nc.reG ) ;
   while ( wpx.ypos < (this->termRows - 1) )
   {
      if ( wpx.ypos == wk.mevent.ypos )
         this->dPtr->WriteString ( wpx.ypos, wpx.xpos + 2, L">>", nc.brR ) ;
      gsx.compose( "%02hd\n", &wpx.ypos ) ;
      wpx = this->dPtr->WriteParagraph ( wpx, gsx, nc.brR ) ;
   }
   this->dPtr->RefreshWin () ;
   #endif                  // DEBUG_UIIS

   if ( (wk.mevent.ypos >= firstIRow) && (wk.mevent.ypos <= lastIRow) )
   {
      //* For each two rows beyond the first, add one to the index *
      fnIndex = this->tData.sfFirst + offset / 2 ;

      #if DEBUG_UIIS != 0     // DEBUG ONLY
      gsx.compose( "Item: sfFirst:%02hd offset:%02hd\nfnIndex:%02hd", 
                   &fnIndex, &offset, &fnIndex ) ;
      this->DebugMsg ( gsx, false, 1 ) ;
      if ( this->tData.sfFocus != fnIndex )
         sleep ( 2 ) ;
      #endif                  // DEBUG_UIIS

      if ( this->tData.sfFocus != fnIndex )
      {
         this->tData.sfFocus = fnIndex ;
         this->UpdateDataWindows () ;
      }
   }
   return fnIndex ;

   #undef DEBUG_UIIS
}  //* End uiItemSelected() *

//*************************
//*    uiFieldSelected    *
//*************************
//******************************************************************************
//* Caller has determined that the mouse click is in the interior of the       *
//* tag-fields sub-window. Map the click column to the corresponding field.    *
//*                                                                            *
//* Input  : wk  : user input (mouse click, position)                          *
//*                                                                            *
//* Returns: index of target field                                             *
//*          (-1) if mouse event does not indicate a field to be edited        *
//******************************************************************************
//* -- The open space before a field is included with that field.              *
//*    -- There is one space at the left edge of the window.                   *
//*    -- There are two spaces between fields.                                 *
//* -- The open space at the right edge of the window (if any) is included     *
//*    with the preceeding field.                                              *
//* -- Most fields have 'stdFIELD' columns.                                    *
//*    -- Target width (leftmost field): 1 + stdFIELD                          *
//*    -- Target width (other standard fields): 2 + stdFIELD                   *
//* -- Exceptions to standard field width are as defined in UpdateMetadata().  *
//*    a) tfTrck: (at left) 1 + 9 == 10,  (else) 2 + 9 == 11                   *
//*    b) tfTyer: (at left) 1 + 6 == 7,   (else) 2 + 6 == 8                    *
//* -- The Y offset may point to the file line OR the line below it.           *
//*    File display lines always have an _odd_ Y offset from top of field      *
//*    window and an _even_ offset from top of application dialog.             *
//******************************************************************************

short Taggit::uiFieldSelected ( const wkeyCode& wk )
{
   const short trk1off = 10,     // field-width offsets (see notes above)
               trk2off = 11,
               yer1off = 7,
               yer2off = 8,
               std1off = (1 + stdFIELD),
               std2off = (2 + stdFIELD) ;
   short findx = this->tData.sffOffset,               // field index (return value)
         xoff = wk.mevent.xpos - (fieldwinX + 1),     // relative X offset
         foff = (findx == tfTrck) ? trk1off :         // offset to next field
                (findx == tfTyer) ? yer1off : std1off ;

   while ( xoff >= foff )
   {
      ++findx ;
      foff += (findx == tfTrck) ? trk2off :         // offset to next field
              (findx == tfTyer) ? yer2off : std2off ;
      if ( foff >= (this->fwinCols - 2) )
         break ;
   }
   return findx ;

}  //* End uiFieldSelected() *

//*************************
//*    uiEditFilename     *
//*************************
//******************************************************************************
//* User has selected a filename to edit. 'sfFocus' indicates the target file. *
//*                                                                            *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: 'true'  if filename field has changed                             *
//*                  (original filename will still be in 'sfPath[]')           *
//*          'false' if filename field unchanged                               *
//******************************************************************************
//* 1) open edit dialog with instructions                                      *
//* 2) allow user to edit filename.                                            *
//* 3) if name changed, save it to 'sfName',                                   *
//*   (original name is still in 'sfPath')                                     *
//* 4) filename construction options:                                          *
//*    a) Copy title field as filename (extension unchanged).                  *
//*    b) Construct filename as Title + Artist.                                *
//*    c) Construct filename as Artist + Title.                                *
//*    d) Prepend the track number to existing filename.                       *
//* 5) duplication option:                                                     *
//*    a) Duplicate the construction pattern and/or track number insertion     *
//*       to all filenames.                                                    *
//*    b) If none of the construction options are active, this has no effect.  *
//*                                                                            *
//******************************************************************************

bool Taggit::uiEditFilename ( void )
{
   const char* Labels[][13] = 
   {
      {  //* English *
         "    Edit File Name    ",     // dialog title
         "     DONE     ",             // 'done' pushbutton
         "    CANCEL    ",             // 'cancel' pushbutton
         "    Clear Edits     ",       // 'clear' pushbutton
         // NOTE: RB labels are actually the examples to preserve color contrast
         "Example:  Let It Be.mp3",             // Title pushbutton
         "Example:  Let It Be - Beatles.mp3",   // Title+Artist pushbutton
         "Example:  Beatles - Let It Be.mp3",   // Artist+Title pushbutton
         "Example:  12 Get Back - Beatles.mp3", // Insert-track pushbutton
         "Duplicate pattern to all file names.",// 'Duplicate' radiobutton label
         "Current File Name: ",                 // orig-filename header
         "New File Name:",                      // textbox label
         "Ctrl+C = Copy, Ctrl+X = Cut, Ctrl+V = Paste\n" // Instructions
         "INSERT key toggles insert/overstrike.",
         "Copy from Title field\n\n"            // Example array
         "Combine Title + Artist\n\n"
         "Combine Artist + Title\n\n"
         "Insert track number",
      },
      {  //* Espanol *
         "  Editar el Nombre del Archivo  ",    // dialog title
         "    HECHO     ",                      // 'done' pushbutton
         "   CANCELAR   ",                      // 'cancel' pushbutton
         "  Borrar Ediciones  ",                // 'clear' pushbutton
         "Ejemplo:  Let It Be.mp3",             // Title pushbutton
         "Ejemplo:  Let It Be - Beatles.mp3",   // Title+Artist pushbutton
         "Ejemplo:  Beatles - Let It Be.mp3",   // Artist+Title pushbutton
         "Ejemplo:  12 Get Back - Beatles.mp3", // Insert-track pushbutton
         "Duplicar patrón a todos nombres de archivo.",// 'Duplicate' radiobutton label
         "Nombre actual: ",                     // orig-filename header
         "Nuevo nombre de archivo:",            // textbox label
         "Ctrl+C =Copiar, Ctrl+X =Cortar, Ctrl+V =Pegar\n" // Instructions
         "INSERT cambia entre Insertar y Sobrescribir.",
         "Copiar desde el campo Título\n\n"     // Example array
         "Combinar Título + Artista\n\n"
         "Combinar Artista + Título\n\n"
         "Inserte el número de pista",          // insert track #
      },
      {  //* Zhongwen *
         "    编辑文件名    ",
         "     执行     ",                   // 'done' pushbutton
         "     取消     ",                   // 'cancel' pushbutton
         "      清除编辑      ",              // 'clear' pushbutton
         "例子:  Let It Be.mp3",             // Title pushbutton
         "例子:  Let It Be - Beatles.mp3",   // Title+Artist pushbutton
         "例子:  Beatles - Let It Be.mp3",   // Artist+Title pushbutton
         "例子:  12 Get Back - Beatles.mp3", // Insert-track pushbutton
         "复制模式 为每个文件名.",              // 'Duplicate' radiobutton label
         "当前文件名： ",                     // orig-filename header
         "新文件名：",                       // textbox label
         "Ctrl+C = 复制, Ctrl+X = 剪切, Ctrl+V = 粘贴\n" // Instructions
         "INSERT键 切换之间 插入/覆盖模式。",
         "从标题字段复制\n\n"                   // Example array
         "结合 标题 + 艺术家\n\n"
         "结合 艺术家 + 标题\n\n"
         "插入轨道号",                         // insert track #
      },
      {  //* TiengViet *
         "  Chỉnh Sửa Tên Tệp  ",               // dialog title
         "   Làm Xong   ",                      // 'done' pushbutton
         "    Hủy Bỏ    ",                      // 'cancel' pushbutton
         "   Hủy Chỉnh Sửa    ",                // 'clear' pushbutton
         "Thí dụ:  Let It Be.mp3",             // Title pushbutton
         "Thí dụ:  Let It Be - Beatles.mp3",   // Title+Artist pushbutton
         "Thí dụ:  Beatles - Let It Be.mp3",   // Artist+Title pushbutton
         "Thí dụ:  12 Get Back - Beatles.mp3", // Insert-track pushbutton
         "Chép lại định dạng đến tất cả tên tệp.",// 'Duplicate' radiobutton label
         "Tên tệp hiện tại: ",                 // orig-filename header
         "Tên tệp mới:",                       // textbox label
         "Ctrl+C = Sao chép, Ctrl+X = Cắt, Ctrl+V = Dán\n" // Instructions
         "Phím INSERT chuyển đổi giữa chèn và ghi đè lên.",
         "Sao chép từ trường Tiêu đề\n\n"      // Example array
         "Phối hợp Tiêu đề + Nghệ sĩ\n\n"
         "Phối hợp Nghệ sĩ + Tiêu đề\n\n"
         "Chèn số thứ tự của bài hát",         // insert track #
      },
   } ;

   //* Access the members of 'Labels[][]' *
   enum LabelIndex : short
   {
      liTitle = ZERO, liDone, liCancel, liClear, liTit, liTita, 
      liArtt, liTrk, liDup, liOrig, liNew, liInstr, liExam
   } ;

   const short dlgROWS = 24,              // dialog size
               dlgCOLS = 52,
               dlgY    = fieldwinY + 1,   // dialog position
               dlgX    = fieldwinX + 1 ;
   attr_t dColor = this->cs.sd,           // dialog interior color
          hColor = this->cs.em ;          // bold dialog text
   AppLang lang = this->cfgOpt.appLanguage ; // language index
   gString gs ;                           // text formatting
   short fIndex = this->tData.sfFocus,    // index of item with focus
         rbXPOS = (this->cfgOpt.rtl ? (dlgCOLS - 7) : 2), // Radio button X offset
         rbLABX = (this->cfgOpt.rtl ? -2 : 6) ; // Radio button label X offset
   bool trkFlag = false,                  // 'true' if insert track nuber
        titFlag = false,                  // 'true' if Title-only pattern
        t2aFlag = false,                  // 'true' if Title + Artist pattern
        a2tFlag = false,                  // 'true' if Artist + Title pattern
        dupFlag = false,                  // 'true' if Duplicate-pattern
        rtlFields = bool(this->cfgOpt.rtl && this->cfgOpt.rtlf), // write filename as RTL
        status = false ;                  // return value

   enum Controls : short { fnTB, donePB, cancelPB, clearPB, 
                           titRB, titaRB, atitRB, trkRB, dupRB, toggPB, 
                           controlsDEFINED } ;

InitCtrl ic[controlsDEFINED] =      // array of dialog control info
{
   {  //* 'New Filename' Textbox - - - - - - - - - - - - - - - - - - -    fnTB *
      dctTEXTBOX,                   // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      3,                            // ulY:       upper left corner in Y
      2,                            // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      short(dlgCOLS - 4),           // cols:      control columns
      this->tData.sf[fIndex].sfName, // dispText:  
      this->cs.tn,                  // nColor:    non-focus color
      this->cs.tf,                  // fColor:    focus color
      tbFileName,                   // filter:    valid filename characters
      NULL,                         // label:     
      ZERO,                         // labY:      
      ZERO,                         // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[donePB]                   // nextCtrl:  link in next structure
   },
   {  //* 'Done' pushbutton - - - - - - - - - - - - - - - - - - - - -   donePB *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(ic[fnTB].ulY + 6),      // ulY:       upper left corner in Y
      short(dlgCOLS / 2 - 15),      // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      14,                           // cols:      control columns
      Labels[lang][liDone],         // dispText:  
      this->cs.pn,                  // nColor:    non-focus color
      this->cs.pf,                  // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "",                           // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[cancelPB],                       // nextCtrl:  link in next structure
   },
   {  //* 'Cancel' pushbutton - - - - - - - - - - - - - - - - - - -   cancelPB *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      ic[donePB].ulY,               // ulY:       upper left corner in Y
      short(ic[donePB].ulX + ic[donePB].cols + 2), // ulX: upper left corner in X
      1,                            // lines:     (n/a)
      14,                           // cols:      control columns
      Labels[lang][liCancel],       // dispText:  
      this->cs.pn,                  // nColor:    non-focus color
      this->cs.pf,                  // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "",                           // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[clearPB]                  // nextCtrl:  link in next structure
   },
   {  //* 'Clear' pushbutton - - - - - - - - - - - - - - - - - - - -   clearPB *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(ic[donePB].ulY + 2),    // ulY:       upper left corner in Y
      short(dlgCOLS / 2 - 10),      // ulX: upper left corner in X
      1,                            // lines:     (n/a)
      20,                           // cols:      control columns
      Labels[lang][liClear],        // dispText:  
      this->cs.pn,                  // nColor:    non-focus color
      this->cs.pf,                  // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "",                           // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[titRB]                    // nextCtrl:  link in next structure
   },
   {  //* 'Title' radiobutton  - - - - - - - - - - - - - - - - - - -     titRB *
      dctRADIOBUTTON,               // type:      
      rbtS5s,                       // sbSubtype: alt font, 5-wide
      false,                        // rbSelect:  initially 'insert' mode
      short(ic[clearPB].ulY + 2),   // ulY:       upper left corner in Y
      rbXPOS,                       // ulX: upper left corner in X
      1,                            // lines:     (n/a)
      1,                            // cols:      (n/a)
      NULL,                         // dispText:  
      dColor,                       // nColor:    non-focus color
      this->cs.pf,                  // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      Labels[lang][liTit],          // label:     
      1,                            // labY:      
      rbLABX,                       // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[titaRB]                   // nextCtrl:  link in next structure
   },
   {  //* 'Title+Artist' radiobutton  - - - - - - - - - - - - - - - -   titaPB *
      dctRADIOBUTTON,               // type:      
      rbtS5s,                       // sbSubtype: alt font, 5-wide
      false,                        // rbSelect:  initially 'insert' mode
      short(ic[titRB].ulY + 2),     // ulY:       upper left corner in Y
      rbXPOS,                       // ulX: upper left corner in X
      1,                            // lines:     (n/a)
      1,                            // cols:      (n/a)
      NULL,                         // dispText:  
      dColor,                       // nColor:    non-focus color
      this->cs.pf,                  // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      Labels[lang][liTita],         // label:     
      1,                            // labY:
      rbLABX,                       // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[atitRB]                   // nextCtrl:  link in next structure
   },
   {  //* 'Artist+Title' radiobutton  - - - - - - - - - - - - - - - -   atitRB *
      dctRADIOBUTTON,               // type:      
      rbtS5s,                       // sbSubtype: alt font, 5-wide
      false,                        // rbSelect:  initially 'insert' mode
      short(ic[titaRB].ulY + 2),    // ulY:       upper left corner in Y
      rbXPOS,                       // ulX: upper left corner in X
      1,                            // lines:     (n/a)
      1,                            // cols:      (n/a)
      NULL,                         // dispText:  
      dColor,                       // nColor:    non-focus color
      this->cs.pf,                  // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      Labels[lang][liArtt],         // label:     
      1,                            // labY:      
      rbLABX,                       // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[trkRB]                    // nextCtrl:  link in next structure
   },
   {  //* 'Insert-Track' radiobutton  - - - - - - - - - - - - - - - -    trkRB *
      dctRADIOBUTTON,               // type:      
      rbtS5s,                       // sbSubtype: alt font, 5-wide
      false,                        // rbSelect:  initially 'insert' mode
      short(ic[atitRB].ulY + 2),    // ulY:       upper left corner in Y
      rbXPOS,                       // ulX: upper left corner in X
      1,                            // lines:     (n/a)
      1,                            // cols:      (n/a)
      NULL,                         // dispText:  
      dColor,                       // nColor:    non-focus color
      this->cs.pf,                  // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      Labels[lang][liTrk],          // label:     
      1,                            // labY:      
      rbLABX,                       // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[dupRB]                    // nextCtrl:  link in next structure
   },
   {  //* 'Duplicate' radio button  - - - - - - - - - - - - - - - - -    dupRB *
      dctRADIOBUTTON,               // type:      
      rbtS5s,                       // sbSubtype: alt font, 5-wide
      false,                        // rbSelect:  initially 'insert' mode
      short(ic[trkRB].ulY + 3),     // ulY:       upper left corner in Y
      rbXPOS,                       // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a)
      dColor,                       // nColor:    non-focus color
      this->cs.pf,                  // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      Labels[lang][liDup],          // label:
      ZERO,                         // labY:      
      rbLABX,                       // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      NULL,                         // nextCtrl:  link in next structure
   },
   {  //* 'Toggle' pushbutton - - - - - - - - - - - - - - - - - - - -   toggPB *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      ic[clearPB].ulY,              // ulY:       upper left corner in Y
      2,                            // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      5,                            // cols:      control columns
      " LT^R ",                     // dispText:  
      this->cs.pn,                  // nColor:    non-focus color
      this->cs.pf,                  // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "",                           // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      NULL                          // nextCtrl:  link in next structure
   },
} ;


   //* Initial parameters for dialog window *
   InitNcDialog dInit( dlgROWS,         // number of display lines
                       dlgCOLS,         // number of display columns
                       dlgY,            // Y offset from upper-left of terminal 
                       dlgX,            // X offset from upper-left of terminal 
                       NULL,            // dialog title
                       ncltDUAL,        // border line-style
                       dColor,          // border color attribute
                       dColor,          // interior color attribute
                       ic               // pointer to list of control definitions
                     ) ;

   this->dPtr->SetDialogObscured () ;  // save parent dialog

   //* For RTL user-interface languages, enable the RTL Pushbutton *
   if ( this->cfgOpt.rtl )
   {
      ic[dupRB].nextCtrl = &ic[toggPB] ;
   }

   //* Instantiate the dialog window *
   NcDialog* dp = new NcDialog ( dInit ) ;

   //* Define a list of keycodes reserved for *
   //* access to clipboard functionality and  *
   //* enable invalid-text-input alert.       *
   this->uiDefineReservedKeys ( dp ) ;
   dp->TextboxAlert ( fnTB, true ) ;

   //* If user interface language is an RTL language,      *
   //* set the internal NcDialog flag(s) for RTL output.   *
   //* NOTE: Display and edit of Textbox controls defaults *
   //* to LTR; however, this can be toggled by the user by *
   //* pressing the CTRL+R hotkey.                         *
   //* NOTE: Static display of the filename under edit is  *
   //* controlled by the 'cfgOpt.rtlf' flag (see below).   *
   if ( this->cfgOpt.rtl )
   {
      dp->DrawLabelsAsRTL () ;
      if ( rtlFields )
         dp->DrawContentsAsRTL ( fnTB ) ;
   }

   //* Open the dialog window *
   if ( (dp->OpenWindow()) == OK )
   {
      //* Set the dialog title *
      dp->SetDialogTitle ( Labels[lang][liTitle], hColor ) ;


      //* Print dialog window's static text           *
      //* a) Original name heading (LTR or RTL) 'rtl' *
      //* b) Original name (LTR or RTL)        'rtlf' *
      //* c) New name heading (LTR or RTL)      'rtl' *
      winPos wp( 1, (this->cfgOpt.rtl ? (dlgCOLS - 3) : 2) ) ;
      wp = dp->WriteString ( wp, Labels[lang][liOrig], hColor, false, this->cfgOpt.rtl ) ;
      //* Extract the filename from the filespec *
      gString origName( this->tData.sf[fIndex].sfPath ) ;
      short origIndex = (origName.findlast( L'/' )) + 1 ;
      origName.shiftChars( -origIndex ) ;

      //* Size the filename to fit into the available space *
      short onameSpace = (this->cfgOpt.rtl ? (wp.xpos - 1) : 
                                             (dlgCOLS - (wp.xpos + 1))) ;
      if ( (origName.gscols()) > onameSpace )
         origName.limitCols( onameSpace ) ;
      if ( this->cfgOpt.rtl && ! rtlFields ) // adjust cursor position
         wp.xpos -= (origName.gscols() - 1) ;
      dp->WriteString ( wp, origName, dColor, false, rtlFields ) ;
      ++wp.ypos ;
      wp.xpos = (this->cfgOpt.rtl ? (ic[fnTB].ulX + ic[fnTB].cols - 1) : 2) ;
      dp->WriteString ( wp, Labels[lang][liNew], 
                        hColor, false, this->cfgOpt.rtl ) ;

      origIndex = origName.findlast( L'.' ) ;
      gString nameExt( &origName.gstr()[origIndex] ) ;   // filename extension

      dp->WriteParagraph ( ic[titRB].ulY, ic[titRB].ulX + ic[titRB].labX, 
                           Labels[lang][liExam], hColor, false, this->cfgOpt.rtl ) ;

      wp.ypos += 2 ; 
      wp.xpos = ic[fnTB].ulX + (this->cfgOpt.rtl ? (ic[fnTB].cols - 3) : 2) ;
      dp->WriteParagraph ( wp, Labels[lang][liInstr], 
                           this->cs.dm, false, this->cfgOpt.rtl ) ;

      dp->RefreshWin () ;                 // make the text visible

      //* Establish a call-back method for cut-and-paste operations.*
      //dp->EstablishCallback ( &gnfControlUpdate ) ;

      uiInfo   Info ;                     // user interface data returned here
      short    icIndex = fnTB ;           // index of control with focus
      bool     done = false ;             // loop control
      while ( ! done )
      {
         if ( ic[icIndex].type == dctPUSHBUTTON )
         {
            if ( Info.viaHotkey )
               Info.HotData2Primary () ;
            else
               icIndex = dp->EditPushbutton ( Info ) ;
            if ( Info.dataMod != false )
            {
               if ( Info.ctrlIndex == donePB )
               {
                  //* Copy contents of Textbox to the live data *
                  dp->GetTextboxText ( fnTB, gs ) ;
                  if ( gs != origName )
                  {
                     gs.copy( this->tData.sf[fIndex].sfName, MAX_FNAME ) ;
                     this->tData.sf[fIndex].sfMod = true ; // set edits-pending flag
                  }

                  //* If one of the duplication patterns is in effect, *
                  //* duplicate the filename pattern to all files.     *
                  if ( dupFlag && (titFlag || t2aFlag || a2tFlag || trkFlag) )
                  {
                     this->uiefDuplicate ( titFlag, t2aFlag, a2tFlag, trkFlag ) ;
                  }
                  status = done = true ;
               }
               else if ( Info.ctrlIndex == cancelPB )
               {  //* Any edits or pattern specifications are discarded *
                  done = true ;
               }
               else if ( Info.ctrlIndex == clearPB )
               {
                  //* Set contents of textbox back to the original filename *
                  dp->SetTextboxText ( fnTB, origName ) ;
                  while ( (icIndex = dp->NextControl ()) != dupRB ) ; // return to Textbox
               }
               else if ( Info.ctrlIndex == toggPB )
               {  //* Note: We will arrive here ONLY if UI language is RTL. *
                  //* Toggle the NcDialog's internal flag.                  *
                  rtlFields = rtlFields ? false : true ;
                  dp->DrawContentsAsRTL ( fnTB, rtlFields ) ;
                  
               }
            }     // dataMod
         }        // dctPUSHBUTTON

         else if ( ic[icIndex].type == dctRADIOBUTTON )
         {
            if ( Info.viaHotkey )
               Info.HotData2Primary () ;
            else
               icIndex = dp->EditRadiobutton ( Info ) ;
            if ( Info.dataMod != false && Info.ctrlIndex == dupRB )
            {
               dupFlag = Info.isSel ;
            }
            if ( Info.dataMod != false && Info.ctrlIndex == trkRB )
            {
               dp->GetTextboxText ( fnTB, gs ) ;

               //* Insert track number (if any) into existing filename *
               if ( (trkFlag = Info.isSel) != false )
               {
                  int t ;
                  if ( (*this->tData.sf[fIndex].sfTag.field[tfTrck] != NULLCHAR) &&
                       ((swscanf ( this->tData.sf[fIndex].sfTag.field[tfTrck], L"%d", &t)) == 1) )
                  {
                     gString gsx( "%02d ", &t ) ;
                     gs.insert( gsx.gstr() ) ;
                  }
                  dp->SetTextboxText ( fnTB, gs ) ;
               }
               //* Reconstruct filename without track number *
               else
               {
                  this->uiefConstruct ( gs, fIndex, titFlag, t2aFlag, a2tFlag, trkFlag ) ;
                  dp->SetTextboxText ( fnTB, gs ) ;
               }
            }
            else if ( Info.dataMod != false )
            {
               //* Update the flags *
               if ( Info.ctrlIndex == titRB )
               {
                  titFlag = Info.isSel ;
                  dp->SetRadiobuttonState ( titaRB, false ) ;
                  dp->SetRadiobuttonState ( atitRB, false ) ;
                  t2aFlag = a2tFlag = false ;
                  while ( (icIndex = dp->NextControl ()) != atitRB ) ;
               }
               else if ( Info.ctrlIndex == titaRB )
               {
                  t2aFlag = Info.isSel ;
                  dp->SetRadiobuttonState ( titRB, false ) ;
                  dp->SetRadiobuttonState ( atitRB, false ) ;
                  titFlag = a2tFlag = false ;
                  while ( (icIndex = dp->NextControl ()) != atitRB ) ;
               }
               else if ( Info.ctrlIndex == atitRB )
               {
                  a2tFlag = Info.isSel ;
                  dp->SetRadiobuttonState ( titRB, false ) ;
                  dp->SetRadiobuttonState ( titaRB, false ) ;
                  titFlag = t2aFlag = false ;
               }

               //* Construct filename according to pattern *
               this->uiefConstruct ( gs, fIndex, titFlag, t2aFlag, a2tFlag, trkFlag ) ;
               //* Update textbox contents *
               dp->SetTextboxText ( fnTB, gs ) ;
               //* Advance the focus to end of (pseudo)XOR group *
               if ( (Info.ctrlIndex == titRB) || (Info.ctrlIndex == titaRB) )
                  while ( (icIndex = dp->NextControl ()) != atitRB ) ;
            }
         }

         else if ( ic[icIndex].type == dctTEXTBOX )
         {
            Info.viaHotkey = false ;      // ignore hotkey data
            icIndex = dp->EditTextbox ( Info ) ;
         }

         //* Move focus to appropriate control *
         if ( ! done && ! Info.viaHotkey )
         {
            if ( Info.keyIn == nckSTAB )
               icIndex = dp->PrevControl () ; 
            else
               icIndex = dp->NextControl () ;
         }
      }     // while(!done)
   }
   if ( dp != NULL )                         // close the window
      delete ( dp ) ;

   this->dPtr->RefreshWin () ;               // restore parent dialog

   //* If filename(s) has changed, update the display *
   if ( status != false )
      this->UpdateDataWindows () ;

   return status ;

}  //* End uiEditFilename() *

//*************************
//*     uiefDuplicate     *
//*************************
//******************************************************************************
//* Apply the filename construction pattern to all files.                      *
//*                                                                            *
//* Input  : titFlag : if Title only                                           *
//*          t2aFlag : if Title _and_ Artist                                   *
//*          a2tFlag : if Artist _and_ Title                                   *
//*          trkFlag : if insert track number                                  *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void Taggit::uiefDuplicate ( bool titFlag, bool t2aFlag, bool a2tFlag, bool trkFlag )
{
   gString gsOut ;            // constructed filename

   for ( short fIndex = ZERO ; fIndex < this->tData.sfCount ; ++fIndex )
   {
      this->uiefConstruct ( gsOut, fIndex, titFlag, t2aFlag, a2tFlag, trkFlag ) ;

      //* Save to name field and set edits-pending flag. *
      //*     (path is not modified at this time)        *
      gsOut.copy( this->tData.sf[fIndex].sfName, MAX_FNAME ) ;
      this->tData.sf[fIndex].sfMod = true ;
   }

}  //* End uiefDuplicate() *

//*************************
//*     uiefConstruct     *
//*************************
//******************************************************************************
//* Construct a filename from the specified pattern.                           *
//*                                                                            *
//* Input  : gsOut   : receives the constructed filename                       *
//*          fIndex  : index of record to be modified                          *
//*          titFlag : if Title only                                           *
//*          t2aFlag : if Title _and_ Artist                                   *
//*          a2tFlag : if Artist _and_ Title                                   *
//*          trkFlag : if insert track number                                  *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void Taggit::uiefConstruct ( gString& gsOut, short fIndex, bool titFlag, 
                             bool t2aFlag, bool a2tFlag, bool trkFlag )
{
   gString origName,          // filename as extracted from path
           nameExt,           // filename extension
           title,             // 'Title' field
           artist,            // 'Artist' field
           gst ;              // track-number formatting
   short  oIndex,             // index into array of file-record objects
          track ;             // numeric track number

   gsOut.clear() ;            // initialize caller's buffer

   //* Get the original filename from path *
   origName = this->tData.sf[fIndex].sfPath ;
   oIndex = (origName.findlast( L'/' )) + 1 ;
   origName.shiftChars( -oIndex ) ;

   //* Extract the filename extension *
   oIndex = origName.findlast( L'.' ) ;
   nameExt = &origName.gstr()[oIndex] ;

   //* Get the Title and Artist data *
   title  = (*this->tData.sf[fIndex].sfTag.field[tfTit2] != NULLCHAR) ?
            this->tData.sf[fIndex].sfTag.field[tfTit2] : NoFieldContents ;
   artist = (*this->tData.sf[fIndex].sfTag.field[tfTpe1] != NULLCHAR) ?
            this->tData.sf[fIndex].sfTag.field[tfTpe1] : NoFieldContents ;

   //* If title already contains the    *
   //* filename extension remove it now.*
   if ( (oIndex = title.find( nameExt.gstr() )) >= ZERO )
      title.limitChars( oIndex ) ;

   if ( titFlag )                // Title only
      gsOut = title ;

   else if ( t2aFlag )           // Title and Artist
      gsOut.compose( "%S - %S", title.gstr(), artist.gstr() ) ;

   else if ( a2tFlag )           // Artist and Title
      gsOut.compose( "%S - %S", artist.gstr(), title.gstr() ) ;

   else  // insert Track number only, OR no modifications required
      gsOut = origName ;

   if ( trkFlag )
   {
      //* Get the track number and insert it at the beginning of *
      //* the filename. If no track number, skip this step.      *
      if ( (*this->tData.sf[fIndex].sfTag.field[tfTrck] != NULLCHAR) &&
           ((swscanf ( this->tData.sf[fIndex].sfTag.field[tfTrck], L"%hd", &track)) == 1) )
      {
         gst.compose( "%02hd ", &track ) ;
         gsOut.insert( gst.gstr() ) ;
      }
   }

   //* Append the filename extension (if needed) *
   if ( (gsOut.find( nameExt.gstr() )) < ZERO )
      gsOut.append( nameExt.gstr() ) ;

}  //* End uiefConstruct() *

//*************************
//*    uiEditMetadata     *
//*************************
//******************************************************************************
//* User has selected a tag field to edit. 'sfFocus' indicates the target file.*
//*                                                                            *
//* Input  : fldIndex : initial field to be edited                             *
//*                                                                            *
//* Returns: 'true'  if contents of one or more fields have been modified      *
//*                  Original field contents in the live-data ('undo') record. *
//*          'false' if no data modified                                       *
//******************************************************************************
//* Note on the 'tfMod' flag:                                                  *
//* -------------------------                                                  *
//* 'tfMod' is set if user has modified any of the tag-data fields (or image). *
//* There is, however, a logical problem:                                      *
//* There are 3 logical data sets:                                             *
//*   1) scanned-data    : the data taken directly from the source file.       *
//*   2) live-data buffer: the data which will be written to the target file.  *
//*                        this->tData.sf[]                                    *
//*   3) edit-data buffer: a working copy of the live-data while tag fields    *
//*                        are under edit. (eBuff)                             *
//*                                                                            *
//* a) Initially, live-data is an exact copy of the scanned source data,       *
//*    so tfMod == false.                                                      *
//* b) On edit, the edit buffer 'eBuff' gets a copy of the live data.          *
//*    The user may or may not modify the 'eBuff' data during edit:            *
//*    -- If the 'eBuff' data ARE NOT modified, then the live-data 'tfMod'     *
//*       flag will not be modified.                                           *
//*    -- If the 'eBuff' data ARE modified, then on return from this method    *
//*       'tfMod' will be set to indicate that the data were changed.          *
//*                                                                            *
//* The logical problem is that the data in 'eBuff' _technically_ should be    *
//* compared to the contents of the source file, but since that is impractical,*
//* the 'eBuff' data are compared with the live-data buffer and if they are    *
//* different, then 'tfMod' is set.                                            *
//*         if ( eBuff != live-data ) tfmod = true ;                           *
//* See the uiemTransfer() method for implementation of this algorithm.        *
//******************************************************************************

bool Taggit::uiEditMetadata ( short fldIndex )
{
   const short dlgROWS = 15,              // dialog size
               dlgCOLS = 90,
               ctrX    = dlgCOLS / 2,
               dlgY    = fieldwinY + 1,   // dialog position
               dlgX    = fieldwinX + 1,
               editHEIGHT = 5,            // Textbox dimensions
               editWIDTH  = (stdFIELD / 2 * 3) ;
#if 1    // EXPERIMENTAL 1
#else    // EXPERIMENTAL 2
   const short nextrecOFFSET = (this->cfgOpt.appLanguage == esLang) ? 3 : 
                               (this->cfgOpt.appLanguage == zhLang) ? 1 : 2 ;
#endif   // EXPERIMENTAL 2
   attr_t dColor = this->cs.sd,           // dialog interior color
          hColor = this->cs.em ;          // bold dialog text
   AppLang lang = this->cfgOpt.appLanguage ; // language index
   gString gs ;                           // text formatting
   short fIndex = this->tData.sfFocus,    // index of item with focus
         sffo = this->tData.sffOffset ;   // initial field offset
   bool rtlFields = (this->cfgOpt.rtl && this->cfgOpt.rtlf) ; // direction of field contents
   //* Create a working copy of all text-tag fields.*
   tagFields eBuff ;
   this->uiemTransfer ( fIndex, eBuff ) ;

   //* Access the members of 'Labels[][]' *
   enum LabelIndex : short
   {
      liTitle = ZERO, liDone, liCancel, liUndo, liNext, liPrev, liUp, liDown, 
      liFName, liFmod
   } ;

   const char* Labels[][13] = 
   {
      {  //* English *
         "    Edit Metadata Fields    ",  // dialog title
         "     DONE     ",             // 'done' pushbutton
         "    CANCEL    ",             // 'cancel' pushbutton
         "    Undo  Changes   ",       // 'clear' pushbutton
         " ^next ╺━▶ ",                // 'next field' pushbutton
         " ◀━╸ ^prev ",                // 'previous field' pushbutton
         " ∆  ^up  ∆ \n",              // 'record up'   pushbutton
         " ∇ ^down ∇ \n",              // 'record down' pushbutton
         "File Name: ",                // filename header
         "(modified)",                 // field modified indicator
      },
      {  //* Espanol *
         "   Editar Campos de Metadatos   ",  // dialog title
         "    HECHO     ",             // 'done' pushbutton
         "   CANCELAR   ",             // 'cancel' pushbutton
         "Descartar el Cambio ",       // 'clear' pushbutton
         " ^next ╺━▶ ",                // 'next field' pushbutton
         " ◀━╸ ^prev ",                // 'previous field' pushbutton
         " ∆  ^up  ∆ \n",              // 'record up'   pushbutton
         " ∇ ^down ∇ \n",              // 'record down' pushbutton
         "Nombre del Archivo: ",       // filename header
         "(modificado)",               // field modified indicator
      },
      {  //* Zhongwen *
         "       编辑元数据字段       ",  // dialog title
         "     执行     ",              // 'done' pushbutton
         "     取消     ",             // 'cancel' pushbutton
         "      清除修改      ",        // 'clear' pushbutton
         " ^next ╺━▶ ",                // 'next field' pushbutton
         " ◀━╸ ^prev ",                // 'previous field' pushbutton
         " ∆  ^up  ∆ \n",              // 'record up'   pushbutton
         " ∇ ^down ∇ \n",              // 'record down' pushbutton
         "文件名: ",                    // filename header
         "（改性）",                     // field modified indicator
      },
      {  //* TiengViet *
         "  Chỉnh sửa trường siêu dữ liệu  ",  // dialog title
         "   Làm Xong   ",             // 'done' pushbutton
         "    Hủy Bỏ    ",             // 'cancel' pushbutton
         "   Huỷ Chỉnh Sửa    ",       // 'clear' pushbutton
         " ^next ╺━▶ ",                // 'next field' pushbutton
         " ◀━╸ ^prev ",                // 'previous field' pushbutton
         " ∆  ^up  ∆ \n",              // 'record up'   pushbutton
         " ∇ ^down ∇ \n",              // 'record down' pushbutton
         "Tên Tệp: ",                  // filename header
         "(Dữ liệu sửa đổi)",          // field modified indicator
      },
   } ;

InitCtrl ic[em_controlsDEFINED] =   // array of dialog control info
{
   {  //* 'Edit Field' Textbox - - - - - - - - - - - - - - - - - -   em_editTB *
      dctTEXTBOX,                   // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      3,                            // ulY:       upper left corner in Y
      short(ctrX - editWIDTH / 2),  // ulX: upper left corner in X
      editHEIGHT,                   // lines:     multi-line textbox
      editWIDTH,                    // cols:      control columns
      NULL,                         // dispText:  
      this->cs.tn,                  // nColor:    non-focus color
      this->cs.tf,                  // fColor:    focus color
      tbPrint,                      // filter:    valid filename characters
      NULL,                         // label:     
      ZERO,                         // labY:      
      ZERO,                         // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[em_donePB]                // nextCtrl:  link in next structure
   },
   {  //* 'Done' pushbutton - - - - - - - - - - - - - - - - - - - -  em_donePB *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(ic[em_editTB].ulY + 8), // ulY:       upper left corner in Y
      short(ctrX - 15),             // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      14,                           // cols:      control columns
      Labels[lang][liDone],         // dispText:  
      this->cs.pn,                  // nColor:    non-focus color
      this->cs.pf,                  // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "",                           // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[em_cancelPB],             // nextCtrl:  link in next structure
   },
   {  //* 'Cancel' pushbutton - - - - - - - - - - - - - - - - - -  em_cancelPB *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      ic[em_donePB].ulY,            // ulY:       upper left corner in Y
      short(ic[em_donePB].ulX + ic[em_donePB].cols + 2), // ulX: upper left corner in X
      1,                            // lines:     (n/a)
      14,                           // cols:      control columns
      Labels[lang][liCancel],       // dispText:  
      this->cs.pn,                  // nColor:    non-focus color
      this->cs.pf,                  // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "",                           // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[em_undoPB]                // nextCtrl:  link in next structure
   },
   {  //* 'Clear' pushbutton - - - - - - - - - - - - - - - - - - -   em_undoPB *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(ic[em_donePB].ulY + 2), // ulY:       upper left corner in Y
      short(ctrX - 10),             // ulX: upper left corner in X
      1,                            // lines:     (n/a)
      20,                           // cols:      control columns
      Labels[lang][liUndo],         // dispText:  
      this->cs.pn,                  // nColor:    non-focus color
      this->cs.pf,                  // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "",                           // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[em_fxPB]                  // nextCtrl:  link in next structure
   },
   {  //* 'Speciai Effects' pushbutton - - - - - - - - - - - - - - -   em_fxPB *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      ic[em_undoPB].ulY,            // ulY:       upper left corner in Y
      short(ic[em_undoPB].ulX + ic[em_undoPB].cols + 1), // ulX: upper left corner in X
      1,                            // lines:     (n/a)
      32,                           // cols:      control columns
      "                                ", // dispText:  (initially blank)
      dColor,                       // nColor:    non-focus color
      this->cs.pf,                  // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "",                           // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[em_nextPB]                // nextCtrl:  link in next structure
   },
   {  //* 'Next Field' pushbutton  - - - - - - - - - - - - - - - -   em_nextPB *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      ic[em_editTB].ulY,            // ulY:       upper left corner in Y
      short(dlgCOLS - 12),          // ulX: upper left corner in X
      1,                            // lines:     (n/a)
      10,                           // cols:      control columns
      Labels[lang][liNext],         // dispText:  
      this->cs.tf | ncrATTR,        // nColor:    non-focus color
      this->cs.pf,                  // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "",                           // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[em_prevPB]                // nextCtrl:  link in next structure
   },
   {  //* 'Prev Field' pushbutton  - - - - - - - - - - - - - - - -   em_prevPB *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      ic[em_editTB].ulY,            // ulY:       upper left corner in Y
      2,                            // ulX: upper left corner in X
      1,                            // lines:     (n/a)
      10,                           // cols:      control columns
      Labels[lang][liPrev],         // dispText:  
      this->cs.tf | ncrATTR,        // nColor:    non-focus color
      this->cs.pf,                  // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "",                           // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[em_upPB]                  // nextCtrl:  link in next structure
   },
   {  //* 'Record Up' pushbutton   - - - - - - - - - - - - - - - - -   em_upPB *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(ic[em_prevPB].ulY + 2), // ulY:       upper left corner in Y
      ic[em_prevPB].ulX,            // ulX: upper left corner in X
      1,                            // lines:     (n/a)
      10,                           // cols:      control columns
      Labels[lang][liUp],           // dispText:  
      this->cs.tf | ncrATTR,        // nColor:    non-focus color
      this->cs.pf,                  // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "",                           // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[em_downPB]                // nextCtrl:  link in next structure
   },
   {  //* 'Record Down' pushbutton   - - - - - - - - - - - - - - -   em_downPB *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(ic[em_upPB].ulY + 2),   // ulY:       upper left corner in Y
      ic[em_upPB].ulX,              // ulX: upper left corner in X
      1,                            // lines:     (n/a)
      10,                           // cols:      control columns
      Labels[lang][liDown],         // dispText:  
      this->cs.tf | ncrATTR,        // nColor:    non-focus color
      this->cs.pf,                  // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "",                           // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      NULL                          // nextCtrl:  link in next structure
   },
   {  //* 'Toggle' pushbutton - - - - - - - - - - - - - - - - - - -   em_rtlPB *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(dlgROWS - 2),           // ulY:       upper left corner in Y
      2,                            // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      5,                            // cols:      control columns
      " LT^R ",                     // dispText:  
      this->cs.pn,                  // nColor:    non-focus color
      this->cs.pf,                  // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "",                           // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      NULL                          // nextCtrl:  link in next structure
   },
} ;

   //* Initial parameters for dialog window *
   InitNcDialog dInit( dlgROWS,         // number of display lines
                       dlgCOLS,         // number of display columns
                       dlgY,            // Y offset from upper-left of terminal 
                       dlgX,            // X offset from upper-left of terminal 
                       NULL,            // dialog title
                       ncltDUAL,        // border line-style
                       dColor,          // border color attribute
                       dColor,          // interior color attribute
                       ic               // pointer to list of control definitions
                     ) ;

   this->dPtr->SetDialogObscured () ;  // save parent dialog

   //* For RTL user-interface languages, enable the RTL Pushbutton *
   if ( this->cfgOpt.rtl )
   {
      ic[em_downPB].nextCtrl = &ic[em_rtlPB] ;
   }

   //* Instantiate the dialog window *
   NcDialog* dp = new NcDialog ( dInit ) ;

   //* Define a list of keycodes reserved for *
   //* access to clipboard functionality and  *
   //* enable invalid-text-input alert.       *
   this->uiDefineReservedKeys ( dp ) ;
   dp->TextboxAlert ( em_editTB, true ) ;

   //* If user interface language is an RTL language,      *
   //* set the internal NcDialog flag(s) for RTL output.   *
   //* NOTE: Display and edit of Textbox controls defaults *
   //* to LTR; however, this can be toggled by the user by *
   //* pressing the CTRL+R hotkey.                         *
   //* NOTE: Static display of the filename under edit is  *
   //* controlled by the 'cfgOpt.rtlf' flag (see below).   *
   if ( this->cfgOpt.rtl )
   {
      dp->DrawLabelsAsRTL () ;
      if ( rtlFields )
         dp->DrawContentsAsRTL ( em_editTB ) ;
   }

   //* Open the dialog window *
   if ( (dp->OpenWindow()) == OK )
   {
      //* Set the dialog title *
      dp->SetDialogTitle ( Labels[lang][liTitle], hColor ) ;

      dp->ControlActive ( em_fxPB, false ) ; // special effects initially inactive

      //* Print dialog window's static text   *
      //* Note the calculations for position, *
      //* length and display of the filename. *
      winPos wp( 1, (this->cfgOpt.rtl ? (dlgCOLS - 3) : 2) ) ;
      wp = dp->WriteString ( wp, Labels[lang][liFName], hColor, false, this->cfgOpt.rtl ) ;
      gs = this->tData.sf[fIndex].sfName ;
      if ( this->cfgOpt.rtl )
      {
         if ( this->cfgOpt.rtlf )
            gs.limitCols( wp.xpos ) ;
         else
         {
            wp.xpos -= (gs.gscols() - 1) ;
            if ( wp.xpos < 1 )
            {
               short trunc = gs.gscols() - (-wp.xpos + 1) ;
               wp.xpos = 1 ;
               gs.limitCols( trunc ) ;
            }
         }
      }
      else
         gs.limitCols( dlgCOLS - wp.xpos - 1 ) ;
      dp->WriteString ( wp, gs, dColor, false, rtlFields ) ;

      //* Calculate positioning and initialize the field to be edited. *
      wp = { short(wp.ypos + 1), 
             short(this->cfgOpt.rtl 
                     ? (ic[em_editTB].ulX + ic[em_editTB].cols - 1) 
                     : ic[em_editTB].ulX) } ;
      this->uiemLoadField ( dp, wp, eBuff, fIndex, fldIndex ) ;
      winPos mwp( (ic[em_editTB].ulY + editHEIGHT - 2), 
                  (ic[em_editTB].ulX + editWIDTH + 1) ) ;

      gs = Labels[lang][liFmod] ;   // label for' field-modified' indicator
      short lifOffset = (this->cfgOpt.rtl ? (gs.gscols() + 1) : 2) ;
      dp->WriteString ( mwp.ypos, mwp.xpos + lifOffset, gs, 
                        dColor, false, this->cfgOpt.rtl ) ;

      dp->RefreshWin () ;                 // make the text visible

      //* Establish a call-back method for cut-and-paste operations.*
      //dp->EstablishCallback ( &gnfControlUpdate ) ;

      wkeyCode wkd ;                   // push user keystrokes into input stream
      uiInfo   Info ;                  // user interface data returned here
      short    icIndex = em_editTB ;   // index of control with focus
      bool     done = false ;          // loop control
      while ( ! done )
      {
         if ( ic[icIndex].type == dctPUSHBUTTON )
         {
            if ( Info.viaHotkey )
               Info.HotData2Primary () ;
            else
               icIndex = dp->EditPushbutton ( Info ) ;

            if ( Info.dataMod != false )
            {
               if ( (Info.ctrlIndex == em_donePB) || 
                    (Info.ctrlIndex == em_downPB) || (Info.ctrlIndex == em_upPB) )
               {
                  //* Copy edited data (if any) to the live-data array. *
                  //* (Caller's 'tfMod' is initialized as appropriate.) *
                  this->uiemTransfer ( eBuff, fIndex ) ;

                  //* If hotkey indicates 'next-record', push the keycode *
                  //* which will caused next record to receive focus.     *
                  if ( Info.ctrlIndex != em_donePB )
                  {
                     if ( (Info.ctrlIndex == em_downPB) &&
                          (this->tData.sfFocus < (this->tData.sfCount - 1)) )
                        wkd = { nckDOWN, wktFUNKEY } ;
                     else if ( (Info.ctrlIndex == em_upPB) &&
                          (this->tData.sfFocus > ZERO) )
                        wkd = { nckUP, wktFUNKEY } ;
                     dp->UngetKeyInput ( wkd ) ;
                     wkd.key = nckENTER ;
                     dp->UngetKeyInput ( wkd ) ;
                  }
                  done = true ;
               }
               else if ( Info.ctrlIndex == em_cancelPB )
               {  //* Discard any edits *
                  done = true ;
               }
               else if ( Info.ctrlIndex == em_undoPB )
               {  //* Restore original contents to field under edit *
                  //* from live data and update the work buffer.    *
                  gs = this->tData.sf[fIndex].sfTag.field[fldIndex] ; // restore from live data
                  gs.copy( eBuff.field[fldIndex], gsMAXCHARS ) ;
                  dp->SetTextboxText ( em_editTB, gs ) ; // restore Textbox
                  
                  //* Reset the edits-pending indicator *
                  this->uiemFieldModified ( dp, mwp, eBuff, fldIndex, true ) ;
               }
               else if ( Info.ctrlIndex == em_nextPB )
               {  //* Move to next metadata field *
                  if ( fldIndex > this->tData.sffOffset )
                     this->tData.sffOffset = fldIndex ;
                  fldIndex = this->uiemLoadNext ( dp, wp, eBuff, fIndex, fldIndex ) ;
                  while ( (icIndex = dp->NextControl ()) != em_editTB ) ;
                  icIndex = dp->PrevControl () ;
               }
               else if ( Info.ctrlIndex == em_prevPB )
               {  //* Move to previous metadata field *
                  if ( fldIndex > this->tData.sffOffset )
                     this->tData.sffOffset = fldIndex ;
                  fldIndex = this->uiemLoadPrev ( dp, wp, eBuff, fIndex, fldIndex ) ;
                  while ( (icIndex = dp->NextControl ()) != em_editTB ) ;
                  icIndex = dp->PrevControl () ;
               }
               else if ( Info.ctrlIndex == em_fxPB )
               {  //* Multi-purpose pushbutton for opening lists  *
                  //* and other sub-dialogs for certain fields.   *
                  //* See notes uiemFxSelect() method for details.*
                  dp->SetDialogObscured () ;
                  short dRows = dlgROWS,
                        dCols = dlgCOLS - ic[em_donePB].ulX,
                        dY    = dlgY,
                        dX    = dlgX + ic[em_donePB].ulX ;
                  gString gsg ;
                  bool fxMod = 
                  this->uiemFxSelect ( fldIndex, gsg, dRows, dCols, dY, dX ) ;
                  dp->RefreshWin () ;
                  if ( fxMod != false )
                  {
                     //* 'Title' field OR 'Language' field *
                     if (   ((fldIndex == tfTit2) || (fldIndex == tfTlan))
                         && ((gsg.gschars()) > 1) )
                     {
                        dp->SetTextboxText ( em_editTB, gsg ) ;
                        gsg.copy( eBuff.field[fldIndex], gsMAXCHARS ) ;
                     }

                     //* 'Artist' field OR 'Album' field *
                     else if ( (fldIndex == tfTpe1) || (fldIndex == tfTalb) )
                     {  //* Duplicate field to all files *
                        dp->GetTextboxText ( em_editTB, gs ) ;
                        dp->SetDialogObscured () ;
                        this->cdfDuplicateField ( fldIndex, gs ) ;
                        this->dPtr->SetDialogObscured () ;  // re-save parent dialog
                        dp->RefreshWin () ;
                     }

                     //* 'Genre' field *
                     else if ( (fldIndex == tfTcon) && ((gsg.gschars()) > 1) )
                     {
                        dp->GetTextboxText ( em_editTB, gs ) ;
                        if ( ((gs.gschars()) > 1) && (*(gs.gstr()) == L'(' ) )
                        {
                           gs.append( L',' ) ;
                           gsg.insert( gs.gstr() ) ;
                        }
                        dp->SetTextboxText ( em_editTB, gsg ) ;
                        gsg.copy( eBuff.field[fldIndex], gsMAXCHARS ) ;
                     }
                  }  // fxMod
                  //* Update the edits-pending indicator *
                  this->uiemFieldModified ( dp, mwp, eBuff, fldIndex, true ) ;
               }     // em_fxPB

               else if ( Info.ctrlIndex == em_rtlPB )
               {  //* Note: We will arrive here ONLY if UI language is RTL. *
                  //* Toggle the NcDialog's internal flag.                  *
                  rtlFields = rtlFields ? false : true ;
                  dp->DrawContentsAsRTL ( em_editTB, rtlFields ) ;
                  
               }
            }        // Pushbutton dataMod
         }           // Pushbutton

         else if ( ic[icIndex].type == dctRADIOBUTTON )
         {
            //if ( Info.viaHotkey )
            //   Info.HotData2Primary () ;
            //else
            //   icIndex = dp->EditRadiobutton ( Info ) ;
            //if ( Info.dataMod != false )
            //{
            //}
         }

         else if ( ic[icIndex].type == dctTEXTBOX )
         {
            Info.viaHotkey = false ;      // ignore hotkey data
            icIndex = dp->EditTextbox ( Info ) ;
            if ( Info.dataMod != false )
            {
               dp->GetTextboxText ( em_editTB, gs ) ;

               //* For the formatted fields, verify that the format is correct.*
               bool autoReformat = false ;
               if ( fldIndex == tfTrck )
                  autoReformat = this->uiemFormatTrack ( gs ) ;
               else if ( fldIndex == tfTyer )
                  autoReformat = this->uiemFormatYear ( gs ) ;
               if ( autoReformat )
               {
                  dp->PrevControl () ;
                  dp->SetTextboxText ( em_editTB, gs ) ;
                  attr_t dtmAttr[editWIDTH + 1] ;
                  for ( short i = ZERO ; i < editWIDTH ; ++i )
                     dtmAttr[i] = this->cs.pf ;
                  dtbmData dData( (gs.gstr()), dtmAttr, true, editWIDTH ) ;
                  dp->DisplayTextboxMessage ( em_editTB, dData ) ;
                  AudibleAlert aa( 5, 2 ) ;
                  dp->UserAlert ( &aa ) ;
                  dp->NextControl () ;
               }

               //* Copy the data to edit buffer *
               gs.copy( eBuff.field[fldIndex], gsMAXCHARS ) ;
               this->uiemFieldModified ( dp, mwp, eBuff, fldIndex, true ) ;
            }
         }

         //* Move focus to appropriate control *
         if ( ! done && ! Info.viaHotkey )
         {
            if ( Info.keyIn == nckSTAB )
               icIndex = dp->PrevControl () ; 
            else
               icIndex = dp->NextControl () ;
         }
      }     // while(!done)
   }
   if ( dp != NULL )                         // close the window
      delete ( dp ) ;

   this->dPtr->RefreshWin () ;               // restore parent dialog

   //* If contents of field(s) has changed, OR if user has      *
   //* scrolled the data fields horizontally,update the display.*
   if ( (this->tData.sf[fIndex].sfTag.tfMod) || (this->tData.sffOffset != sffo) )
      this->UpdateDataWindows () ;

   return this->tData.sf[fIndex].sfTag.tfMod ;

}  //* End uiEditMetadata() *

//*************************
//*   uiemForamatTrack    *
//*************************
//******************************************************************************
//* Verify that user input for the 'Track' field (tfTrck) is formatted         *
//* correctly.                                                                 *
//*                                                                            *
//* Programer's Note: Because this is magic manipulation of the data, be sure  *
//* to document it thoroughly. Non-numeric data will be discarded and          *
//* out-of-range data will be set to the maximum value.                        *
//*                                                                            *
//* Input  : trkField  : contains the unformatted Track data                   *
//*                                                                            *
//* Returns: 'true' if trkField modified, else 'false'                         *
//******************************************************************************

bool Taggit::uiemFormatTrack ( gString& trkField )
{
   bool reformat = false ;          // return value
   if ( (trkField.gschars()) > 1 )
   {
      int trk = ZERO, cnt = ZERO, 
      convCount = swscanf ( trkField.gstr(), L"%d / %d", &trk, &cnt ) ;
      if ( convCount == ZERO )   // no valid track data
      {
         trkField.clear() ;
         reformat = true ;
      }
      else if ( convCount >= 1 )
      {
         if ( (trk < ZERO) || (trk > MAX_SRCFILES) )
         {
            trk = MAX_SRCFILES ;
            reformat = true ;
         }
         if ( convCount > 1 )
         {
            if ( (cnt < ZERO) || (cnt > MAX_SRCFILES) )
            {
               cnt = MAX_SRCFILES ;
               reformat = true ;
            }
         }
         if ( reformat )
         {
            if ( convCount == 1 )
               trkField.compose( "%2d", &trk ) ;
            else
               trkField.compose( "%2d/%02d", &trk, &cnt ) ;
         }
      }
   }
   return reformat ;

}  //* End uiemFormatTrack() *

//*************************
//*    uiemFormatYear     *
//*************************
//******************************************************************************
//* Verify that user input for the 'Track' field (tfTrck) is formatted         *
//* correctly.                                                                 *
//*                                                                            *
//* Programer's Note: Because this is magic manipulation of the data, be sure  *
//* to document it thoroughly. Non-numeric data will be discarded and          *
//* out-of-range data will be set to the maximum value.                        *
//*                                                                            *
//* Input  : trkField  : contains the unformatted Track data                   *
//*                                                                            *
//* Returns: 'true' if yearField modified, else 'false'                        *
//******************************************************************************

bool Taggit::uiemFormatYear ( gString& yearField )
{
   bool reformat = false ;       // return value
   if ( (yearField.gschars()) > 1 )
   {
      int yr = ZERO, 
      convCount = swscanf ( yearField.gstr(), L"%d", &yr ) ;
      if ( convCount == ZERO )   // no valid year data
      {
         yearField.clear() ;
         reformat = true ;
      }
      else if ( convCount == 1 )
      {
         if ( (yr < ZERO) || (yr > 9999) )
         {
            yr = 9999 ;
            yearField.compose( "%d", &yr ) ;
            reformat = true ;
         }
      }
   }
   return reformat ;

}  //* End uiemFormatYear() *

//*************************
//*     uiemTransfer      *
//*************************
//******************************************************************************
//* Called by uiEditMetadata() to transfer the live-data tag array to a work   *
//* buffer for editing. (Image data are not transferred.)                      *
//*                                                                            *
//*                                                                            *
//* Input  : fIndex  : index of file under edit                                *
//*          eBuff   : (by reference) temporary buffer for pending edits       *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void Taggit::uiemTransfer ( short fIndex, tagFields& eBuff ) 
{
   gString gs ;

   for ( short i = ZERO ; i < tfCOUNT ; ++i )   // for each metadata field
   {
      gs = this->tData.sf[fIndex].sfTag.field[i] ;
      gs.copy( eBuff.field[i], gsMAXCHARS ) ;
   }
   eBuff.tfMod = false ;         // no NEW edits pending

}  //* End uiemTransfer() *

//*************************
//*     uiemTransfer      *
//*************************
//******************************************************************************
//* Called by uiEditMetadata() to transfer edited data from the work buffer    *
//* to the live-data array.                                                    *
//*                                                                            *
//* Input  : eBuff   : (by reference) temporary buffer for pending edits       *
//*          fIndex  : index of file under edit                                *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************
//* Notes:                                                                     *
//* This functionality is isolated here because:                               *
//* a) we want to control the 'tfMod' flag, (see notes in uiEditMetadata())    *
//* b) in future we may want to massage the data as we save back to the        *
//*    live-data object.                                                       *
//*                                                                            *
//******************************************************************************

void Taggit::uiemTransfer ( tagFields& eBuff, short fIndex )
{
   gString gs ;

   eBuff.tfMod = false ;   // reset the data-modified flag

   for ( short i = ZERO ; i < tfCOUNT ; ++i )   // for each metadata field
   {
      gs = eBuff.field[i] ;
      if ( (gs.compare( this->tData.sf[fIndex].sfTag.field[i] )) != ZERO )
      {
         gs.copy( this->tData.sf[fIndex].sfTag.field[i], gsMAXCHARS ) ;
         eBuff.tfMod = true ;
      }
   }
   if ( eBuff.tfMod )
      this->tData.sf[fIndex].sfTag.tfMod = true ;

}  //* End uiemTransfer() *

//*************************
//*     uiemLoadNext      *
//*************************
//******************************************************************************
//* Called by uiEditMetadata() to prepare the next active field for edit.      *
//*                                                                            *
//* Input  : dp      : pointer to caller's dialog window                       *
//*          wp      : position of header text                                 *
//*          eBuff   : temporary buffer for pending edits                      *
//*          fIndex  : index of file under edit                                *
//*          fldIndex: index of metadata field to be loaded                    *
//*                                                                            *
//* Returns: field loaded for edit                                             *
//*          If already on last field, returns index of current field          *
//******************************************************************************

short Taggit::uiemLoadNext ( NcDialog* dp, const winPos& wp, tagFields& eBuff, 
                             short fIndex, short fldIndex )
{
   //* Test for next active tag field *
   this->tData.sffOffset = this->NextActiveField () ;
   if ( this->tData.sffOffset > fldIndex )
   {
      //* Save the value under edit to the edit buffer *
      gString gs ;
      dp->GetTextboxText ( em_editTB, gs ) ;
      gs.copy( eBuff.field[fldIndex], gsMAXCHARS ) ;

      //* Prepare new target field for edit *
      fldIndex = this->tData.sffOffset ;
      this->uiemLoadField ( dp, wp, eBuff, fIndex, fldIndex ) ;
   }
   else
      dp->UserAlert () ;

   return fldIndex ;

}  //* End uiemLoadNext() *

//*************************
//*     uiemLoadPrev      *
//*************************
//******************************************************************************
//* Called by uiEditMetadata() to prepare the previous active field for edit.  *
//*                                                                            *
//* Input  : dp      : pointer to caller's dialog window                       *
//*          wp      : position of header text                                 *
//*          eBuff   : temporary buffer for pending edits                      *
//*          fIndex  : index of file under edit                                *
//*          fldIndex: index of metadata field to be loaded                    *
//*                                                                            *
//* Returns: field loaded for edit                                             *
//*          If already on first field, returns index of current field         *
//******************************************************************************

short Taggit::uiemLoadPrev ( NcDialog* dp, const winPos& wp, tagFields& eBuff, 
                             short fIndex, short fldIndex )
{
   //* Test for previous active tag field *
   this->tData.sffOffset = this->PrevActiveField () ;
   if ( this->tData.sffOffset < fldIndex )
   {
      //* Save the value under edit to the edit buffer *
      gString gs ;
      dp->GetTextboxText ( em_editTB, gs ) ;
      gs.copy( eBuff.field[fldIndex], gsMAXCHARS ) ;

      //* Prepare new target field for edit *
      fldIndex = this->tData.sffOffset ;
      this->uiemLoadField ( dp, wp, eBuff, fIndex, fldIndex ) ;
   }
   else
      dp->UserAlert () ;

   return fldIndex ;

}  //* End uiemLoadPrev() *

//*************************
//*    uiemLoadField      *
//*************************
//******************************************************************************
//* Called by uiEditMetadata(), uiemLoadNext() and uiemLoadPrev() to prepare   *
//* the current field for edit.                                                *
//* It is assumed that caller will never send us invalid data.                 *
//*                                                                            *
//* Input  : dp      : pointer to caller's dialog window                       *
//*          wp      : position of header text                                 *
//*          eBuff   : temporary buffer for pending edits                      *
//*          fIndex  : index of file under edit                                *
//*          fldIndex: index of metadata field to be loaded                    *
//*                                                                            *
//* Returns: index of displayed field                                          *
//******************************************************************************

short Taggit::uiemLoadField ( NcDialog* dp, const winPos& wp, 
                              tagFields& eBuff, short fIndex, short fldIndex )
{
   //* Field descriptions, formats, comments *
   const char* fldHelp[][tfCOUNT] = 
   {
      {  //* English *
/*TIT2*/ "Title/Songname/Content description is the actual name of the piece.\n"
         "Examples: 'Get Back', 'Moonlight Sonata'",
/*TPE1*/ "Lead artist(s)/Lead performer(s)/Soloist(s)/Performing group.\n"
         "Names are seperated with the '/' character.\n"
         "Example: 'Paul McCartney/Stevie Wonder'",
/*TALB*/ "Album/Movie/Show title: the title of the recording or source\n"
         "of the sound from which the audio in the file is taken.\n"
         "Example: 'Dazed and Confused Soundtrack'",
/*TRCK*/ "Track number/Position in set: the sequence number of the audio file\n"
         "on its original recording. May be extended with a '/' and the total\n"
         "numer of tracks on the original recording. Examples: '2' '4/9'.",
/*TYER*/ "'Year' is a numeric string with a year of the recording.\n"
         "This string is always four characters long. Example: '2013'",
/*TCON*/ "Genre (Content Type) may be taken from one or more items\n"
         "in the list, or create your own description.",
/*TPE2*/ "Band/Orchestra/Accompaniment: provides additional information\n"
         "about the performers in the recording.",
/*TPUB*/ "Publisher: contains the name of the record label or publisher.",
/*TCOM*/ "Composer: name of the composer(s), seperated with a '/' character.\n"
         "Example: 'John Lennon/Paul McCartney'",
/*TEXT*/ "Lyricist/Text writer: author(s) of the text or lyrics in the\n"
         "recording, seperated with a '/' character.",
/*TPE4*/ "Remixed/Interpreted/Modified by: contains more information about\n"
         "the people behind a remix and similar interpretations of another\n"
         "existing piece.",
/*TCOP*/ "Copyright message: copyright holder of the original sound.\n"
         "Must begin with a year and a space character (making five\n"
         "characters minimum). Example: '1970 Apple Studios'",
/*TOWN*/ "File owner/licensee: name of the owner or licensee of the file\n"
         "and its contents.",
/*TXXX*/ "Comment field. A single, free-form text string of up to 1024\n"
         "characters OR a heading and a free-form string separated by\n"
         "a CARET ('^'). Example: \"My Review:^a happy song!\"",
/*TDAT*/ "A numeric string in the DDMM format containing the date for the\n"
         "recording. This field is always four characters. Example: '2804'",
/*TIME*/ "A numeric string in the HHMM format containing the time for the\n"
          "recording. This field is always four characters. Example:'0735'",
/*TRDA*/ "Recording dates: used as complement to the 'Year', 'Date' and\n"
         "'Time' fields. Example: '12th June'",
/*TLEN*/ "Length of the audio file playback in milliseconds.\n"
         "Represented as a numeric string. Example: '210255'",
/*TSIZ*/ "Size of the audiofile in bytes, excluding the tag data.\n"
         "Represented as a numeric string. Example: '3285679'",
/*TBPM*/ "Contains the number of beats per minute in the main part of the\n"
         "audio. The BPM is an integer value represented as a numeric string.",
/*TIT1*/ "Content group description: used if the sound belongs to a larger\n"
         "category of sounds/music. For example, classical music is often\n"
         "categorized by subtype: 'Piano Concerto', 'String Quartet, etc.",
/*TIT3*/ "Subtitle/Description refinement: used for information directly\n"
         "related to the contents of the 'Title' field.\n"
         "Example: 'Live at Covent Gardens'.",
/*TPE3*/ "Name of the conductor. Example: 'Leonard Bernstein'",
/*TPOS*/ "Part of a set: a numeric string that describes which part of a set\n"
         "the audio came from. For instance, if the source described in the\n"
         "'Album' field is part of a multi-disc set. Example: '1/2'",
/*TKEY*/ "The initial musical key in which the sound starts. This is a string\n"
         "with three(3) characters maximum. Examples: 'G' 'Cm' 'DbM' 'E#'",
/*TLAN*/ "Language(s) of the text or lyrics spoken or sung in the audio,\n"
         "represented by a three character code according to ISO-639-2.",
/*TOPE*/ "Original artist(s)/performer(s): the performer(s) of the original\n"
         "recording. For example, if the music in the file is a cover of a\n"
         "previously released song. Example: 'Rod Stewart/Kim Carnes'",
/*TOAL*/ "Original album/movie/show title: the title of the original\n"
         "recording (or source of sound). Example: 'Patch Adams Soundtrack'",
/*TOFN*/ "Original filename: contains the preferred filename for the file,\n"
         "because some media restrict filename length. Name is case sensitive\n"
         "and includes its suffix. Example: 'Faith of the Heart.mp3'",
/*TOLY*/ "Original lyricist/text writer: authors of the lyrics/text of the\n"
         "original recording. Example: 'Diane Warren'",
/*TORY*/ "Original release year: the year of the original recording.\n"
         "The year is represented as four(4) digits. Example: '1998'",
/*TRSN*/ "Name of the internet radio station from which\n"
         "the audio is streamed.",
/*TRSO*/ "Name of the owner of the internet radio station from which\n"
         "the audio is streamed.",
/*TSRC*/ "The International Standard Recording Code (ISRC) (12 characters).\n"
         "Example: 'CCXXXYYNNNNN' CC=Country Code, XXX=Issuer Code,\n"
         "YY=2-digit year, NNNNN=recording identification number.",
/*TSSE*/ "Software/Hardware and settings used for encoding: includes the\n"
         "audio encoder software and hardware and the settings used.\n"
         "Normally used only by recording-studio engineers.",
/*TFLT*/ "File type: indicates which type of audio this tag defines.\n"
         "Examples: 'MPG' for .mp3 audio, 'OGG' for .oga audio, etc.",
/*TDLY*/ "Playlist delay: defines the number of milliseconds of silence\n"
         "between every song in a playlist. The time is represented as a\n"
         "numeric string.",
/*TENC*/ "Encoded by: contains the name of the person or organisation that\n"
         "encoded the audio. Normally used only by recording-studio engineers\n"
         "or by transcoding software.",
/*TMED*/ "Media type: describes the media from which the sound originated.\n"
         "This is either a predefined media type: DIG, ANA, CD, LD, TT, MD,\n"
         "DAT, DCC, DVD, TV, VID, RAD, TEL, MC, REE, or a text string.",
      },
      {  //* Espanol *
/*TIT2*/ "Título/nombre de la canción/contenido. La descripción es el nombre\n"
         "real de la pieza. Ejemplos: 'Get Back', 'Moonlight Sonata'",
/*TPE1*/ "Artista principal/Actor principal/Solista/Grupo de rendimiento.\n"
         "Los nombres se separan con el carácter '/'.\n"
         "Ejemplo: 'Paul McCartney/Stevie Wonder",
/*TALB*/ "Álbum/Película/Mostrar título: el título de la grabación o fuente\n"
         "del sonido del que se toma el audio.\n"
         "Ejemplo: 'Dazed and Confused Soundtrack'",
/*TRCK*/ "Número de pista o posición en el conjunto: el número de secuencia\n"
         "del archivo de audio en su grabación original. Puede extenderse con\n"
         "una '/' y el número total de pistas. Ejemplos: '2' '4/9'",
/*TYER*/ "'Año' es una cadena numérica con un año de grabación.\n"
         "Esta cadena tiene siempre cuatro caracteres. Ejemplo: '2013'",
/*TCON*/ "El género (tipo de contenido) puede tomarse de uno o más\n"
         "elementos de la lista, o crear su propia descripción.",
/*TPE2*/ "Banda, Orquesta, Acompañamiento: proporciona información adicional\n"
         "sobre los intérpretes en la grabación.",
/*TPUB*/ "Editor: contiene el nombre del sello discográfico o editor.",
/*TCOM*/ "Compositor: nombre de los compositores, separados con un '/'.\n"
         "Ejemplo: 'John Lennon/Paul McCartney'",
/*TEXT*/ "Letrista, Escritor de textos: autor(es) del texto o letras\n"
         "de la grabación, separados con un carácter '/'.",
/*TPE4*/ "Remixado, interpretado o modificado por: contiene más información\n"
         "acerca de las personas detrás de un remix e interpretaciones\n"
         "similares de otra pieza existente.",
/*TCOP*/ "Derechos de Autor Mensaje: titular de los derechos de autor del\n"
         "sonido original. Debe comenzar con un año y un carácter de espacio\n"
         "(cinco caracteres como mínimo). Ejemplo: '1970 Apple Studios'",
/*TOWN*/ "Propietario de archivo o licenciatario: nombre del propietario o\n"
         "licenciatario del archivo y su contenido.",
/*TXXX*/ "Columna de comentario: una cadena de texto libre de hasta 1024\n"
         "caracteres, o un encabezado y una cadena de forma libre separados\n"
         "por un ('^'). Ejemplo: \"My Review:^a happy song!\"",
// UNDER CONSTRUCTION - TRANSLATE
/*TDAT*/ "A numeric string in the DDMM format containing the date for the\n"
         "recording. This field is always four characters. Example: '2804'",
/*TIME*/ "A numeric string in the HHMM format containing the time for the\n"
          "recording. This field is always four characters. Example:'0735'",
/*TRDA*/ "Recording dates: used as complement to the 'Year', 'Date' and\n"
         "'Time' fields. Example: '12th June'",
/*TLEN*/ "Length of the audio file playback in milliseconds.\n"
         "Represented as a numeric string. Example: '210255'",
/*TSIZ*/ "Size of the audiofile in bytes, excluding the tag data.\n"
         "Represented as a numeric string. Example: '3285679'",
/*TBPM*/ "Contains the number of beats per minute in the main part of the\n"
         "audio. The BPM is an integer value represented as a numeric string.",
/*TIT1*/ "Content group description: used if the sound belongs to a larger\n"
         "category of sounds/music. For example, classical music is often\n"
         "categorized by subtype: 'Piano Concerto', 'String Quartet, etc.",
/*TIT3*/ "Subtitle/Description refinement: used for information directly\n"
         "related to the contents of the 'Title' field.\n"
         "Example: 'Live at Covent Gardens'.",
/*TPE3*/ "Name of the conductor. Example: 'Leonard Bernstein'",
/*TPOS*/ "Part of a set: a numeric string that describes which part of a set\n"
         "the audio came from. For instance, if the source described in the\n"
         "'Album' field is part of a multi-disc set. Example: '1/2'",
/*TKEY*/ "The initial musical key in which the sound starts. This is a string\n"
         "with three(3) characters maximum. Examples: 'G' 'Cm' 'DbM' 'E#'",
/*TLAN*/ "Language(s) of the text or lyrics spoken or sung in the audio,\n"
         "represented by a three character code according to ISO-639-2.",
/*TOPE*/ "Original artist(s)/performer(s): the performer(s) of the original\n"
         "recording. For example, if the music in the file is a cover of a\n"
         "previously released song. Example: 'Rod Stewart/Kim Carnes'",
/*TOAL*/ "Original album/movie/show title: the title of the original\n"
         "recording (or source of sound). Example: 'Patch Adams Soundtrack'",
/*TOFN*/ "Original filename: contains the preferred filename for the file,\n"
         "because some media restrict filename length. Name is case sensitive\n"
         "and includes its suffix. Example: 'Faith of the Heart.mp3'",
/*TOLY*/ "Original lyricist/text writer: authors of the lyrics/text of the\n"
         "original recording. Example: 'Diane Warren'",
/*TORY*/ "Original release year: the year of the original recording.\n"
         "The year is represented as four(4) digits. Example: '1998'",
/*TRSN*/ "Name of the internet radio station from which\n"
         "the audio is streamed.",
/*TRSO*/ "Name of the owner of the internet radio station from which\n"
         "the audio is streamed.",
/*TSRC*/ "The International Standard Recording Code (ISRC) (12 characters).\n"
         "Example: 'CCXXXYYNNNNN' CC=Country Code, XXX=Issuer Code,\n"
         "YY=2-digit year, NNNNN=recording identification number.",
/*TSSE*/ "Software/Hardware and settings used for encoding: includes the\n"
         "audio encoder software and hardware and the settings used.\n"
         "Normally used only by recording-studio engineers.",
/*TFLT*/ "File type: indicates which type of audio this tag defines.\n"
         "Examples: 'MPG' for .mp3 audio, 'OGG' for .oga audio, etc.",
/*TDLY*/ "Playlist delay: defines the number of milliseconds of silence\n"
         "between every song in a playlist. The time is represented as a\n"
         "numeric string.",
/*TENC*/ "Por lo general, esto solo lo utiliza el ingeniero de estudio\n"
         "o el software de transcodificación.",
/*TMED*/ "Media type: describes the media from which the sound originated.\n"
         "This is either a predefined media type: DIG, ANA, CD, LD, TT, MD,\n"
         "DAT, DCC, DVD, TV, VID, RAD, TEL, MC, REE, or a text string.",
      },
      {  //* Zhongwen *
/*TIT2*/ "标题 / 歌名 / 内容 描述是音乐片段的实际名称。\n"
         "例子: 'Get Back', 'Moonlight Sonata'",
/*TPE1*/ "领先的艺术家 / 领先的表演者 / 独唱 / 表演团体。\n"
         "使用斜杠字 ('/') 符分隔名称。\n"
         "例子：  'Paul McCartney/Stevie Wonder'",
/*TALB*/ "专辑，电影，节目标题： 录音或声音源的标题来自。\n"
         "例子： 'Dazed and Confused 配乐'",
/*TRCK*/ "曲目编号 / 位置： 用于原始录制的音频文件的序列号。\n"
         "可以用斜线字符扩展 ('/') 和总曲目在原录音上。 例子： '2' '4/9'.",
/*TYER*/ "'年' 值是表示录音年份。\n"
         "这个字符串总是四个字符长。 例子： '2013'",
/*TCON*/ "类型 (内容类型) 从列表中获取一个或多个项目，或创建自己的描述。",
/*TPE2*/ "嘉宾表演者或团体: 提供有关录音中表演者的其他信息。",
/*TPUB*/ "唱片公司： 包含记录标签或发布者的名称。",
/*TCOM*/ "作曲家： 作曲家的名字, 用斜杠字符分隔 ('/')。\n"
         "例子： 'John Lennon/Paul McCartney'",
// UNDER CONSTRUCTION - TRANSLATE
/*TEXT*/ "Lyricist/Text writer: author(s) of the text or lyrics in the\n"
         "recording, seperated with a '/' character.",
/*TPE4*/ "Remixed/Interpreted/Modified by: contains more information about\n"
         "the people behind a remix and similar interpretations of another\n"
         "existing piece.",
/*TCOP*/ "Copyright message: copyright holder of the original sound.\n"
         "Must begin with a year and a space character (making five\n"
         "characters minimum). Example: '1970 Apple Studios'",
/*TOWN*/ "文件所有者或被许可人： 文件的所有者或被许可人的名称及其内容。",
/*TXXX*/ "评论栏。 一个最多1024个字符的自由格式的文本字符串，或 一个标题和一个由插入\n"
         "符号分开的自由格式的字符串 ('^'). 例子：\"My Review:^a happy song!\"",
/*TDAT*/ "格式的数字字符串： DDMM 包含录制日期。\n"
         "此字段始终为四个字符。 例子： '2804' (4月28日)",
/*TIME*/ "格式的数字字符串： HHMM 包含录音的小时和分钟。\n"
         "此字段始终为四个字符。 例子： '0735' (上午07:35)",
/*TRDA*/ "记录日期：用作补充 '年', '日期' 和 '时' 数据字段. 例子： '12th June'",
/*TLEN*/ "Length of the audio file playback in milliseconds.\n"
         "Represented as a numeric string. Example: '210255'",
/*TSIZ*/ "Size of the audiofile in bytes, excluding the tag data.\n"
         "Represented as a numeric string. Example: '3285679'",
/*TBPM*/ "Contains the number of beats per minute in the main part of the\n"
         "audio. The BPM is an integer value represented as a numeric string.",
/*TIT1*/ "Content group description: used if the sound belongs to a larger\n"
         "category of sounds/music. For example, classical music is often\n"
         "categorized by subtype: 'Piano Concerto', 'String Quartet, etc.",
/*TIT3*/ "Subtitle/Description refinement: used for information directly\n"
         "related to the contents of the 'Title' field.\n"
         "Example: 'Live at Covent Gardens'.",
/*TPE3*/ "Name of the conductor. Example: 'Leonard Bernstein'",
/*TPOS*/ "Part of a set: a numeric string that describes which part of a set\n"
         "the audio came from. For instance, if the source described in the\n"
         "'Album' field is part of a multi-disc set. Example: '1/2'",
/*TKEY*/ "The initial musical key in which the sound starts. This is a string\n"
         "with three(3) characters maximum. Examples: 'G' 'Cm' 'DbM' 'E#'",
/*TLAN*/ "Language(s) of the text or lyrics spoken or sung in the audio,\n"
         "represented by a three character code according to ISO-639-2.",
/*TOPE*/ "Original artist(s)/performer(s): the performer(s) of the original\n"
         "recording. For example, if the music in the file is a cover of a\n"
         "previously released song. Example: 'Rod Stewart/Kim Carnes'",
/*TOAL*/ "Original album/movie/show title: the title of the original\n"
         "recording (or source of sound). Example: 'Patch Adams Soundtrack'",
/*TOFN*/ "Original filename: contains the preferred filename for the file,\n"
         "because some media restrict filename length. Name is case sensitive\n"
         "and includes its suffix. Example: 'Faith of the Heart.mp3'",
/*TOLY*/ "Original lyricist/text writer: authors of the lyrics/text of the\n"
         "original recording. Example: 'Diane Warren'",
/*TORY*/ "Original release year: the year of the original recording.\n"
         "The year is represented as four(4) digits. Example: '1998'",
/*TRSN*/ "Name of the internet radio station from which\n"
         "the audio is streamed.",
/*TRSO*/ "Name of the owner of the internet radio station from which\n"
         "the audio is streamed.",
/*TSRC*/ "The International Standard Recording Code (ISRC) (12 characters).\n"
         "Example: 'CCXXXYYNNNNN' CC=Country Code, XXX=Issuer Code,\n"
         "YY=2-digit year, NNNNN=recording identification number.",
/*TSSE*/ "Software/Hardware and settings used for encoding: includes the\n"
         "audio encoder software and hardware and the settings used.\n"
         "Normally used only by recording-studio engineers.",
/*TFLT*/ "File type: indicates which type of audio this tag defines.\n"
         "Examples: 'MPG' for .mp3 audio, 'OGG' for .oga audio, etc.",
/*TDLY*/ "Playlist delay: defines the number of milliseconds of silence\n"
         "between every song in a playlist. The time is represented as a\n"
         "numeric string.",
/*TENC*/ "编码：创建音频的人员或组织的名称。 通常仅由录音室工程师或转码软件使用。",
/*TMED*/ "媒体类型：描述媒体类型歌曲来自哪里。这是一种预定义的媒体类型：\n"
         "DIG, ANA, CD, LD, TT, MD, DAT, DCC, DVD, TV, VID, RAD, TEL, MC,\n"
         "REE, 或文本字符串。",
      },
      {  //* TiengViet *
// UNDER CONSTRUCTION - TRANSLATE
/*TIT2*/ "Title/Songname/Content description is the actual name of the piece.\n"
         "Examples: 'Get Back', 'Moonlight Sonata'",
/*TPE1*/ "Lead artist(s)/Lead performer(s)/Soloist(s)/Performing group.\n"
         "Names are seperated with the '/' character.\n"
         "Example: 'Paul McCartney/Stevie Wonder",
/*TALB*/ "Album/Movie/Show title: the title of the recording or source\n"
         "of the sound from which the audio in the file is taken.\n"
         "Example: 'Dazed and Confused Soundtrack'",
/*TRCK*/ "Track number/Position in set: the sequence number of the audio file\n"
         "on its original recording. May be extended with a '/' and the total\n"
         "numer of tracks on the original recording. Examples: '2' '4/9'.",
/*TYER*/ "'Year' is a numeric string with a year of the recording.\n"
         "This string is always four characters long. Example: '2013'",
/*TCON*/ "Genre (Content Type) may be taken from one or more items\n"
         "in the list, or create your own description.",
/*TPE2*/ "Band/Orchestra/Accompaniment: provides additional information\n"
         "about the performers in the recording.",
/*TPUB*/ "Publisher: contains the name of the record label or publisher.",
/*TCOM*/ "Composer: name of the composer(s), seperated with a '/' character.\n"
         "Example: 'John Lennon/Paul McCartney'",
/*TEXT*/ "Lyricist/Text writer: author(s) of the text or lyrics in the\n"
         "recording, seperated with a '/' character.",
/*TPE4*/ "Remixed/Interpreted/Modified by: contains more information about\n"
         "the people behind a remix and similar interpretations of another\n"
         "existing piece.",
/*TCOP*/ "Copyright message: copyright holder of the original sound.\n"
         "Must begin with a year and a space character (making five\n"
         "characters minimum). Example: '1970 Apple Studios'",
/*TOWN*/ "File owner/licensee: name of the owner or licensee of the file\n"
         "and it's contents.",
/*TXXX*/ "Comment field. A single, free-form text string\n"
         "of up to 1024 characters.",
/*TDAT*/ "A numeric string in the DDMM format containing the date for the\n"
         "recording. This field is always four characters. Example: '2804'",
/*TIME*/ "A numeric string in the HHMM format containing the time for the\n"
          "recording. This field is always four characters. Example:'0735'",
/*TRDA*/ "Recording dates: used as complement to the 'Year', 'Date' and\n"
         "'Time' fields. Example: '12th June'",
/*TLEN*/ "Length of the audio file playback in milliseconds.\n"
         "Represented as a numeric string. Example: '210255'",
/*TSIZ*/ "Size of the audiofile in bytes, excluding the tag data.\n"
         "Represented as a numeric string. Example: '3285679'",
/*TBPM*/ "Contains the number of beats per minute in the main part of the\n"
         "audio. The BPM is an integer value represented as a numeric string.",
/*TIT1*/ "Content group description: used if the sound belongs to a larger\n"
         "category of sounds/music. For example, classical music is often\n"
         "categorized by subtype: 'Piano Concerto', 'String Quartet, etc.",
/*TIT3*/ "Subtitle/Description refinement: used for information directly\n"
         "related to the contents of the 'Title' field.\n"
         "Example: 'Live at Covent Gardens'.",
/*TPE3*/ "Name of the conductor. Example: 'Leonard Bernstein'",
/*TPOS*/ "Part of a set: a numeric string that describes which part of a set\n"
         "the audio came from. For instance, if the source described in the\n"
         "'Album' field is part of a multi-disc set. Example: '1/2'",
/*TKEY*/ "The initial musical key in which the sound starts. This is a string\n"
         "with three(3) characters maximum. Examples: 'G' 'Cm' 'DbM' 'E#'",
/*TLAN*/ "Language(s) of the text or lyrics spoken or sung in the audio,\n"
         "represented by a three character code according to ISO-639-2.",
/*TOPE*/ "Original artist(s)/performer(s): the performer(s) of the original\n"
         "recording. For example, if the music in the file is a cover of a\n"
         "previously released song. Example: 'Rod Stewart/Kim Carnes'",
/*TOAL*/ "Original album/movie/show title: the title of the original\n"
         "recording (or source of sound). Example: 'Patch Adams Soundtrack'",
/*TOFN*/ "Original filename: contains the preferred filename for the file,\n"
         "because some media restrict filename length. Name is case sensitive\n"
         "and includes its suffix. Example: 'Faith of the Heart.mp3'",
/*TOLY*/ "Original lyricist/text writer: authors of the lyrics/text of the\n"
         "original recording. Example: 'Diane Warren'",
/*TORY*/ "Original release year: the year of the original recording.\n"
         "The year is represented as four(4) digits. Example: '1998'",
/*TRSN*/ "Name of the internet radio station from which\n"
         "the audio is streamed.",
/*TRSO*/ "Name of the owner of the internet radio station from which\n"
         "the audio is streamed.",
/*TSRC*/ "The International Standard Recording Code (ISRC) (12 characters).\n"
         "Example: 'CCXXXYYNNNNN' CC=Country Code, XXX=Issuer Code,\n"
         "YY=2-digit year, NNNNN=recording identification number.",
/*TSSE*/ "Software/Hardware and settings used for encoding: includes the\n"
         "audio encoder software and hardware and the settings used.\n"
         "Normally used only by recording-studio engineers.",
/*TFLT*/ "File type: indicates which type of audio this tag defines.\n"
         "Examples: 'MPG' for .mp3 audio, 'OGG' for .oga audio, etc.",
/*TDLY*/ "Playlist delay: defines the number of milliseconds of silence\n"
         "between every song in a playlist. The time is represented as a\n"
         "numeric string.",
/*TENC*/ "Encoded by: contains the name of the person or organisation that\n"
         "encoded the audio. Normally used only by recording-studio engineers.",
/*TMED*/ "Media type: describes the media from which the sound originated.\n"
         "This is either a predefined media type: DIG, ANA, CD, LD, TT, MD,\n"
         "DAT, DCC, DVD, TV, VID, RAD, TEL, MC, REE, or a text string.",
      },
   } ;
   const short fhOFFSET  = 6,    // field help offset
               fmOFFSETY = 4,    // field-modified indicator offset
               fmOFFSETX = 70 ;

   //* Labels for 'Special Effects' pushbutton *
   enum fxFields : short { fxTitle = ZERO, fxArtist, fxAlbum, fxGenre, 
                           fxLanguage, fxBlank, fxCOUNT } ;
   const char* fxLabels[][fxCOUNT] = 
   {
      {
      "  Create Title From File Name   ",    // English (Title)
      "    Duplicate For All Files     ",    //         (Artist)
      "    Duplicate For All Files     ",    //         (Album)
      "         List of Genres         ",    //         (Genre)
      "  ISO-639-2 List of Languages   ",    //         (Language)
      "                                ",    // blank text
      },
      {
      " Crear título nombre de archivo ",    // Espanol (Title)
      " Duplicar en todos los archivos ",    //         (Artist)
      " Duplicar en todos los archivos ",    //         (Album)
      "        Lista de géneros        ",    //         (Genre)
      "  Idioma de la lista ISO-639-2  ",    //         (Language)
      "                                ",    // blank text
      },
      {
      "        从文件名创建标题        ",       // Zhongwen (Title)
      "       将值复制到所有文件       ",       //         (Artist)
      "       将值复制到所有文件       ",       //         (Album)
      "         音乐流派的列表         ",       //         (Genre)
      "     语言来自 ISO-639-2 表      ",      //          (Language)
      "                                ",    // blank text
      },
      {
      " Tạo Tiêu Đề Từ Các Tên Tập Tin ",    // TiengViet (Title)
      "Nhân đôi cho tất cả các tập tin ",    //           (Artist)
      "Nhân đôi cho tất cả các tập tin ",    //           (Album)
      " Danh Sách Các Thể Loại âm Nhạc ",    //           (Genre)
      "Danh Sách Ngôn Ngữ Từ ISO-639-2 ",    //           (Language)
      "                                ",    // blank text
      },
   } ;


   AppLang lang = this->cfgOpt.appLanguage ; // language index
   const char* const* umwHPtr = umwHeaders[lang] ; // tag-field headings
   attr_t dColor = this->cs.sd,           // dialog interior color
          hColor = this->cs.em ;          // bold dialog text

   //* Erase the old header and write the new one *
   dp->ClearLine ( wp.ypos, false, wp.xpos, this->cfgOpt.rtl ) ;
   dp->WriteString ( wp, umwHPtr[fldIndex], hColor, false, this->cfgOpt.rtl ) ;

   //* Erase the old help message and write the new one *
   dp->ClearLine ( wp.ypos + fhOFFSET, false, wp.xpos, this->cfgOpt.rtl ) ;
   dp->ClearLine ( wp.ypos + fhOFFSET + 1, false, wp.xpos, this->cfgOpt.rtl ) ;
   dp->ClearLine ( wp.ypos + fhOFFSET + 2, false, wp.xpos, this->cfgOpt.rtl ) ;

   //* Display the edits-pending indicator *
   winPos mwp( wp.ypos + fmOFFSETY, fmOFFSETX ) ;
   this->uiemFieldModified ( dp, mwp, eBuff, fldIndex ) ;

   dp->WriteParagraph ( wp.ypos + fhOFFSET, wp.xpos, fldHelp[lang][fldIndex], 
                        dColor, true, this->cfgOpt.rtl ) ;

   //* Target control must not have the input focus *
   short focus = dp->GetCurrControl () ;
   if ( focus == em_editTB )
      dp->PrevControl () ;

   //* Copy the field text from edit buffer into the target Textbox *
   dp->SetTextboxText ( em_editTB, eBuff.field[fldIndex] ) ;

   if ( focus == em_editTB )  // restore input focus
      dp->NextControl () ;

   // Programmer's Note: It is theoretically possible, though unlikely that
   // the control will already be active (but with the wrong title).
   // It is also possible, but VERY unlikely that the target control has focus.
   // However, to quote Indiana Jones, "You know what a cautions fellow I am."
   if ( focus == em_fxPB )
      dp->PrevControl () ;

   //* For fields that require selection lists or other special controls *
   if ( (fldIndex == tfTit2) || (fldIndex == tfTpe1) || (fldIndex == tfTalb) ||
        (fldIndex == tfTcon) || (fldIndex == tfTlan) )
   {
      //* Enable the special-effects control *
      dp->ControlActive ( em_fxPB, true ) ;
      //* Set the appropriate Pushbutton text *
      if ( fldIndex == tfTit2 )                 // 'Title' field
      {
         dp->SetPushbuttonText ( em_fxPB, fxLabels[lang][fxTitle], this->cs.pf, this->cs.pn ) ;
      }
      else if ( fldIndex == tfTpe1 )            // 'Artist' field
      {
         dp->SetPushbuttonText ( em_fxPB, fxLabels[lang][fxArtist], this->cs.pf, this->cs.pn ) ;
      }
      else if ( fldIndex == tfTalb )            // 'Album' field
      {
         dp->SetPushbuttonText ( em_fxPB, fxLabels[lang][fxAlbum], this->cs.pf, this->cs.pn ) ;
      }
      else if ( fldIndex == tfTcon )            // 'Genre' field
      {
         dp->SetPushbuttonText ( em_fxPB, fxLabels[lang][fxGenre], this->cs.pf, this->cs.pn ) ;
      }
      else if ( fldIndex == tfTlan )            // 'Language' field
      {
         dp->SetPushbuttonText ( em_fxPB, fxLabels[lang][fxLanguage], this->cs.pf, this->cs.pn ) ;
      }
   }
   //* If the special-effects control is not needed, *
   //* be sure it is inactive and invisible.         *
   else
   {
      dp->ControlActive ( em_fxPB, false ) ;
      dp->SetPushbuttonText ( em_fxPB, fxLabels[lang][fxBlank], this->cs.sd, this->cs.sd ) ;
   }

   if ( focus == em_fxPB )       // restore input focus
      dp->NextControl () ;

   return fldIndex ;

}  //* End uiemLoadField() *

//*************************
//*     uiemFxSelect      *
//*************************
//******************************************************************************
//* Called by uiEditMetadata().                                                *
//* User has selected the special-effects control.                             *
//* Display the options available for the specified field and allow user to    *
//* make a selection. See notes below.                                         *
//*                                                                            *
//* Input  : fldIndex : index of displayed metadata field                      *
//*          newValue : receives selected:                                     *
//*                     a) genre identification string                         *
//*                     b) language identification code                        *
//*          dRows    : sub-dialog rows                                        *
//*          dCols    : sub-dialog cols                                        *
//*          dY       : sub-dialog Y position                                  *
//*          dX       : sub-dialog X position                                  *
//*                                                                            *
//* Returns: 'true'  if new data specified                                     *
//*          'false' if selection cancelled                                    *
//******************************************************************************
//* Notes on special-effects pushbutton (em_fxPB) functionality.               *
//* ------------------------------------------------------------               *
//* This method has access to the application dialog's data, but does not      *
//* have access to the Textbox control in the editing window. Therefore if     *
//* the contents of the Textbox need to change, we return the value in caller's*
//* 'newValue' object i.e. (newValue.gschars() > 1)                            *
//*                                                                            *
//* 1) 'Title' field (tfTit2):                                                 *
//*    a) Use the filename to create the song title. Title becomes the         *
//*       filename without the filename extension.                             *
//*       Example: Low Rider.mp3  -->  Low Rider                               *
//*    b) This overwrites any existing data in the field.                      *
//*    c) No dialog is opened for this operation.                              *
//*    d) Note that a menu item exists to set the Title field for all files    *
//*       from the filename.                                                   *
//*                                                                            *
//* 2) 'Artist' field:                                                         *
//*    a) Duplicate the value in the field under edit to the corresponding     *
//*       field in all other files.                                            *
//*    b) This is handled in uiemFxSelect(), BUT the application dialog will   *
//*       need to be redrawn on return.                                        *
//*                                                                            *
//* 3) 'Album' field:                                                          *
//*    a) Duplicate the value in the field under edit to the corresponding     *
//*       field in all other files.                                            *
//*    b) This is handled in uiemFxSelect(), BUT the application dialog will   *
//*       need to be redrawn on return.                                        *
//*                                                                            *
//* 4) 'Genre' Field (tfTcon):                                                 *
//*    a) Display a list of pre-defined genres and allow user to make a        *
//*       selection.                                                           *
//*    b) If field contains no data OR if data are not in the 'standard'       *
//*       format, overwrite any existing data in the field with the user's     *
//*       selection.                                                           *
//*    c) If field already contains a correctly formatted genre entry, then    *
//*       append the new selection to the existing data.                       *
//*    d) If no selection, field is not modified.                              *
//*                                                                            *
//* 5) 'Language' field:                                                       *
//*    a) Display a list of the 50-60 most used languages (according to        *
//*       Wikipedia's list of languages by number of native speakers).         *
//*    b) Overwrite existing data in the field with user's selection.          *
//*    c) If no selection, field is not modified.                              *
//*                                                                            *
//*                                                                            *
//******************************************************************************

bool Taggit::uiemFxSelect ( short fldIndex, gString& newValue,
                            short dRows, short dCols, short dY, short dX )
{
   AppLang lang = this->cfgOpt.appLanguage ; // language index
   NcDialog* dp = NULL ;                     // pointer to dialog window
   winPos wp( 1, (this->cfgOpt.rtl ? (dCols / 2) : 2) ) ; // data offset
   bool status = false ;                     // return value

   //* Define the dialog window *
   InitNcDialog dInit( dRows,           // number of display lines
                       dCols,           // number of display columns
                       dY,              // Y offset from upper-left of terminal 
                       dX,              // X offset from upper-left of terminal 
                       NULL,            // dialog title
                       ncltSINGLE,      // border line-style
                       this->cs.sd,     // border color attribute
                       this->cs.sd,     // interior color attribute
                       NULL             // pointer to list of control definitions
                     ) ;

   const char* Labels[][5] = 
   {
      {  // English
         "Pre-defined Genres",
         "Highlight the desired genre\n"
         "and press ENTER.\n"
         "Highlight \"No Selection\"\n"
         "or press ESCAPE to\n"
         "leave field unchanged.",
         "**     NO SELECTION      **",
         "Language codes",
         "Highlight the desired\n"
         "language and press ENTER.\n"
         "Highlight \"No Selection\"\n"
         "or press ESCAPE to\n"
         "leave field unchanged.",
      },
      {  // Espanol
         "Géneros Predefinidos",
         "Resalte el género deseado\n"
         "y presione ENTER. Resalte\n"
         "\"Sin selección\" o presione\n"
         "ESCAPE para dejar el campo\n"
         "sin cambios.",
         "**     Sin selección     **",
         "Códigos de Idioma",
         "Resalte el idioma deseado\n"
         "y presione ENTER. Resalte\n"
         "\"Sin selección\" o presione\n"
         "ESCAPE para dejar el campo\n"
         "sin cambios.",
      },
      {  // Zhongwen
         "预定义类型",
         "突出显示所需的类型，\n"
         "然后按 ENTER。\n"
         "突出显示 “无选择” 或按\n"
         "ESCAPE 保留字段不变。",
         "**        无选择         **",
         "  语言代码",
         "突出显示所需语言，\n"
         "然后按 ENTER。\n"
         "突出显示 “无选择” 或按\n"
         "ESCAPE 保留字段不变。",
      },
      {  // TiengViet
         "Thể Loại Được Xác Định Trước",
         "Làm nổi bật các thể loại\n"
         "mong muốn và nhấn ENTER.\n"
         "Đánh dấu \"Không có lựa\n"
         "chọn\" hoặc nhấn ESCAPE để\n"
         "rời sân không thay đổi.",
         "**   Không có lựa chọn   **",
         "Mã ngôn ngữ",
         "Làm nổi bật các ngôn ngữ\n"
         "mong muốn và nhấn ENTER.\n"
         "Đánh dấu \"Không có lựa\n"
         "chọn\" hoặc nhấn ESCAPE để\n"
         "rời sân không thay đổi.",
      },
   } ;
   //* For the 'Genre' metadata field, list of pre-defined genres as defined by*
   //* the MPEG standard (including the WINAMP extensions). For other media    *
   //* formats, we MAY in future define expanded options for this field.       *
   // Programmer's Note: We have made a design decision to NOT translate this
   // list. The reason is that many of the items in this list are made-up English 
   // words which cannot be directly translated. We spent a full day trying to
   // do the Chinese translation, and even with help from native Chinese speakers,
   // only about half the entries were adequately translated.
   const short geITEMS = 127, geWIDTH = 29 ;
   const char geText[geITEMS][geWIDTH] =    // English
   {
      "**     NO SELECTION      **",
      "(0) Blues                  ",
      "(1) Classic Rock           ",
      "(2) Country                ",
      "(3) Dance                  ",
      "(4) Disco                  ",
      "(5) Funk                   ",
      "(6) Grunge                 ",
      "(7) Hip-Hop                ",
      "(8) Jazz                   ",
      "(9) Metal                  ",
      "(10) New Age               ",
      "(11) Oldies                ",
      "(12) Other                 ",
      "(13) Pop                   ",
      "(14) R&B                   ",
      "(15) Rap                   ",
      "(16) Reggae                ",
      "(17) Rock                  ",
      "(18) Techno                ",
      "(19) Industrial            ",
      "(20) Alternative           ",
      "(21) Ska                   ",
      "(22) Death Metal           ",
      "(23) Pranks                ",
      "(24) Soundtrack            ",
      "(25) Euro-Techno           ",
      "(26) Ambient               ",
      "(27) Trip-Hop              ",
      "(28) Vocal                 ",
      "(29) Jazz+Funk             ",
      "(30) Fusion                ",
      "(31) Trance                ",
      "(32) Classical             ",
      "(33) Instrumental          ",
      "(34) Acid                  ",
      "(35) House                 ",
      "(36) Game                  ",
      "(37) Sound Clip            ",
      "(38) Gospel                ",
      "(39) Noise                 ",
      "(40) AlternRock            ",
      "(41) Bass                  ",
      "(42) Soul                  ",
      "(43) Punk                  ",
      "(44) Space                 ",
      "(45) Meditative            ",
      "(46) Instrumental Pop      ",
      "(47) Instrumental Rock     ",
      "(48) Ethnic                ",
      "(49) Gothic                ",
      "(50) Darkwave              ",
      "(51) Techno-Industrial     ",
      "(52) Electronic            ",
      "(53) Pop-Folk              ",
      "(54) Eurodance             ",
      "(55) Dream                 ",
      "(56) Southern Rock         ",
      "(57) Comedy                ",
      "(58) Cult                  ",
      "(59) Gangsta               ",
      "(60) Top 40                ",
      "(61) Christian Rap         ",
      "(62) Pop/Funk              ",
      "(63) Jungle                ",
      "(64) Native American       ",
      "(65) Cabaret               ",
      "(66) New Wave              ",
      "(67) Psychadelic           ",
      "(68) Rave                  ",
      "(69) Showtunes             ",
      "(70) Trailer               ",
      "(71) Lo-Fi                 ",
      "(72) Tribal                ",
      "(73) Acid Punk             ",
      "(74) Acid Jazz             ",
      "(75) Polka                 ",
      "(76) Retro                 ",
      "(77) Musical               ",
      "(78) Rock & Roll           ",
      "(79) Hard Rock             ",
      "(80) Folk                  ",
      "(81) Folk-Rock             ",
      "(82) National Folk         ",
      "(83) Swing                 ",
      "(84) Fast Fusion           ",
      "(85) Bebob                 ",
      "(86) Latin                 ",
      "(87) Revival               ",
      "(88) Celtic                ",
      "(89) Bluegrass             ",
      "(90) Avantgarde            ",
      "(91) Gothic Rock           ",
      "(92) Progressive Rock      ",
      "(93) Psychedelic Rock      ",
      "(94) Symphonic Rock        ",
      "(95) Slow Rock             ",
      "(96) Big Band              ",
      "(97) Chorus                ",
      "(98) Easy Listening        ",
      "(99) Acoustic              ",
      "(100) Humour               ",
      "(101) Speech               ",
      "(102) Chanson              ",
      "(103) Opera                ",
      "(104) Chamber Music        ",
      "(105) Sonata               ",
      "(106) Symphony             ",
      "(107) Booty Bass           ",
      "(108) Primus               ",
      "(109) Porn Groove          ",
      "(110) Satire               ",
      "(111) Slow Jam             ",
      "(112) Club                 ",
      "(113) Tango                ",
      "(114) Samba                ",
      "(115) Folklore             ",
      "(116) Ballad               ",
      "(117) Power Ballad         ",
      "(118) Rhythmic Soul        ",
      "(119) Freestyle            ",
      "(120) Duet                 ",
      "(121) Punk Rock            ",
      "(122) Drum Solo            ",
      "(123) A capella            ",
      "(124) Euro-House           ",
      "(125) Dance Hall           ",
   } ;

   //* For the 'Language' metadata field, list of language codes as defined by *
   //* the ISO-639-2 standard. Only the top 50-60 codes (according to number of*
   //* native speakers are listed. For the full ISO-639-2 list, please see:    *
   //*        <http://www.loc.gov/standards/iso639-2/php/code_list.php>        *
   // Programmer's Note: We have made a design decision to NOT translate this
   // list. The likelihood of errors is very high, AND we're just too lazy.
   const short laITEMS = 60, laWIDTH = 29 ;
   const char laText[laITEMS][laWIDTH + 2] =    // English
   {
      "**     NO SELECTION      **",
      "zho  Mandarin group (官話) ",
      "spa  Spanish               ",
      "eng  English               ",
      "hin  Hindi                 ",
      "ara  Arabic                ",
      "por  Portuguese            ",
      "ben  Bengali               ",
      "rus  Russian               ",
      "jpn  Japanese              ",
      "pan  Punjabi               ",
      "deu  German                ",
      "jav  Javanese              ",
      "wuu  Wu (incl. Hu dialect) ",
      "msa  Malay                 ",
      "tel  Telugu                ",
      "vie  Vietnamese            ",
      "kor  Korean                ",
      "fra  French                ",
      "mar  Marathi               ",
      "tam  Tamil                 ",
      "urd  Urdu                  ",
      "tur  Turkish               ",
      "ita  Italian               ",
      "yue  Yue                   ",
      "tha  Thai                  ",
      "guj  Gujarati              ",
      "cjy  Jin                   ",
      "nan  Southern Min(Hokkien) ",
      "fas  Persian               ",
      "pol  Polish                ",
      "pus  Pashto                ",
      "kan  Kannada               ",
      "hsn  Xiang                 ",
      "mal  Malayalam             ",
      "sun  Sundanese             ",
      "hau  Hausa                 ",
      "ori  Odiya                 ",
      "bur  Burmese               ",
      "hak  Hakka                 ",
      "ukr  Ukrainian             ",
      "bho  Bhojpuri              ",
      "tgl  Tagalog               ",
      "yor  Yoruba                ",
      "mai  Maithili              ",
      "uzb  Uzbek                 ",
      "snd  Sindhi                ",
      "amh  Amharic               ",
      "ful  Fula                  ",
      "ron  Romanian              ",
      "orm  Oromo                 ",
      "ibo  Igbo                  ",
      "aze  Azerbaijani           ",
      "awa  Awadhi                ",
      "zhx  Gan                   ",
      "ceb  Cebuano               ",
      "nld  Dutch                 ",
      "kur  Kurdish               ",
      "srp  Serbian               ",
      "hrv  Croatian              ",
   } ;

   newValue.clear() ;      // initialize caller's buffer

   if ( fldIndex == tfTit2 )        // 'Title'
   {  //* Get the filename and strip the extension *
      newValue = this->tData.sf[this->tData.sfFocus].sfName ;
      short i = newValue.findlast( L'.' ) ;
      newValue.limitChars( i ) ;
      status = true ;
   }
   else if ( fldIndex == tfTpe1 )   // 'Artist'
   {
      status = true ;
   }
   else if ( fldIndex == tfTalb )   // 'Album'
   {
      status = true ;
   }
   else if ( fldIndex == tfTcon )   // 'Genre'
   {
      //* Insert a 'no selection' item into the genre list *
      char geTemp[geITEMS * (geWIDTH + 4)] ;
      gString gstmp( Labels[lang][2] ) ;
      short indx = gstmp.copy( geTemp, gsMAXBYTES ) ;
      for ( short i = 1 ; i < geITEMS ; ++i )
      {
         gstmp = geText[i] ;
         indx += gstmp.copy( &geTemp[indx], gsMAXBYTES ) ;
      }
      //* Define a dctDROPDOWN control *
      InitCtrl dd = 
      {
         dctDROPDOWN, rbtTYPES, false, 2, 2, short(dRows - 3), short(geWIDTH), 
         (char*)geTemp,
         nc.gyR,                       // nColor:    non-focus border color
         nc.gr,                        // fColor:    focus border color
         tbPrint,                      // filter:    (n/a)
         NULL,                         // label:     
         ZERO,                         // labY:      offset from control's ulY
         ZERO,                         // labX       offset from control's ulX
         ddBoxDOWN,                    // exType:    (n/a)
         geITEMS,                      // scrItems:  number of elements in text/color arrays
         ZERO,                         // scrSel:    index of initial highlighted element
         this->cs.menu,                // scrColor:  single-color data display
         NULL,                         // spinData:  (n/a)
         true,                         // active:    allow control to gain focus
         NULL,                         // nextCtrl:  link in next structure
      } ;
      dInit.ctrlPtr = &dd ;

      dp = new NcDialog ( dInit ) ;

      //* If user interface language is an RTL language, *
      //* set the internal NcDialog flag for RTL output. *
      // Programmer's Note: Because the Genre list is not translated, we 
      // do not set the contents of the Scrollext as RTL.
      if ( this->cfgOpt.rtl )
         dp->DrawLabelsAsRTL () ;

      if ( (dp->OpenWindow()) == OK )
      {
         dp->WriteString ( wp, Labels[lang][0], this->cs.em, false, this->cfgOpt.rtl ) ;
         ++wp.ypos ;
         wp.xpos = (this->cfgOpt.rtl ? (dCols - 2) : (dCols / 2 + 2)) ;
         dp->WriteParagraph ( wp, Labels[lang][1], this->cs.sd, false, this->cfgOpt.rtl ) ;
         dp->RefreshWin () ;
         uiInfo Info ;
         Info.viaHotkey = true ;
         dp->EditDropdown ( Info ) ;
         if ( Info.selMember > ZERO )
         {  //* Get User's selection *
            newValue = geText[Info.selMember] ;
            newValue.strip( false ) ;
            status = true ;
         }
      }
   }
   else if ( fldIndex == tfTlan )   // 'Language'
   {
      //* Insert a 'no selection' item into the language list *
      char laTemp[laITEMS * (laWIDTH + 4)] ;
      gString gstmp( Labels[lang][2] ) ;
      short indx = gstmp.copy( laTemp, gsMAXBYTES ) ;
      for ( short i = 1 ; i < laITEMS ; ++i )
      {
         gstmp = laText[i] ;
         indx += gstmp.copy( &laTemp[indx], gsMAXBYTES ) ;
      }
      //* Define a dctDROPDOWN control *
      InitCtrl dd = 
      {
         dctDROPDOWN, rbtTYPES, false, 2, 2, short(dRows - 3), short(laWIDTH), 
         (char*)laTemp,
         nc.gyR,                       // nColor:    non-focus border color
         nc.gr,                        // fColor:    focus border color
         tbPrint,                      // filter:    (n/a)
         NULL,                         // label:     
         ZERO,                         // labY:      offset from control's ulY
         ZERO,                         // labX       offset from control's ulX
         ddBoxDOWN,                    // exType:    (n/a)
         laITEMS,                      // scrItems:  number of elements in text/color arrays
         ZERO,                         // scrSel:    index of initial highlighted element
         this->cs.menu,                // scrColor:  single-color data display
         NULL,                         // spinData:  (n/a)
         true,                         // active:    allow control to gain focus
         NULL,                         // nextCtrl:  link in next structure
      } ;
      dInit.ctrlPtr = &dd ;

      dp = new NcDialog ( dInit ) ;

      //* If user interface language is an RTL language, *
      //* set the internal NcDialog flag for RTL output. *
      if ( this->cfgOpt.rtl )
      {
         dp->DrawLabelsAsRTL () ;
         dp->DrawContentsAsRTL ( ZERO ) ;
      }

      if ( (dp->OpenWindow()) == OK )
      {
         dp->WriteString ( wp, Labels[lang][3], this->cs.em, false, this->cfgOpt.rtl ) ;
         ++wp.ypos ;
         wp.xpos = (this->cfgOpt.rtl ? (dCols - 2) : (dCols / 2 + 2)) ;
         dp->WriteParagraph ( wp, Labels[lang][4], this->cs.sd, false, this->cfgOpt.rtl ) ;

         dp->RefreshWin () ;
         uiInfo Info ;
         Info.viaHotkey = true ;
         dp->EditDropdown ( Info ) ;
         if ( Info.selMember > ZERO )
         {  //* Get User's selection. Keep only the   *
            //* 3-character code, discard explanation *
            newValue = laText[Info.selMember] ;
            newValue.limitChars( 3 ) ;
            status = true ;
         }
      }
   }

   if ( dp != NULL )       // close the window
      delete ( dp ) ;

   return status ;

}  //* End uiemFxSelect() *

//*************************
//*   uiemFieldModified   *
//*************************
//******************************************************************************
//* Compare the field data in the edit buffer to the corrosponding field of    *
//* the live data and display the edits-pending indicator in the appropriate   *
//* color.                                                                     *
//*                                                                            *
//* Input  : dp      : pointer to caller's dialog window                       *
//*          wp      : position of indicator                                   *
//*          eBuff   : temporary buffer for pending edits                      *
//*          fldIndex: index of metadata field to be tested                    *
//*          refresh : (optional, 'false' by default)                          *
//*                    'true'  refresh the display                             *
//*                    'false' do not refresh the display                      *
//*                                                                            *
//* Returns: 'true'  if field contains modified data, else false               *
//******************************************************************************
//* Note: The configuration provides the edits pending/not-pending colors to   *
//*       correspond with the application's background color. However, for     *
//*       the default color scheme, the sub-dialog background color differs    *
//*       from the background color of the main application window.            *
//*       Therefore, for the default color scheme only, we must select colors  *
//*       which match the sub-dialog background. This requires an assumption   *
//*       about what's happening in the InitColorScheme() method. Be Aware!    *
//******************************************************************************

bool Taggit::uiemFieldModified ( NcDialog* dp, const winPos& wp, 
                                 const tagFields& eBuff, short fldIndex, bool refresh )
{
   //* Set the pending/not-pending colors for *
   //* the color scheme. (see note above)     *
   attr_t sColor = this->cs.saved,        // color for unmodified data
          uColor = this->cs.unsaved ;     // color for modified data
   if ( this->cs.scheme == ncbcCOLORS )
   {
      sColor = nc.grbl ;
      uColor = nc.rebl ;
   }

   bool fmod = this->uiemCompareField ( eBuff, fldIndex ) ;
   dp->WriteChar ( wp, fillCIRCLE, (fmod ? uColor : sColor), refresh ) ;

   return fmod ;

}  //* End uiemFieldModified() *

//*************************
//*   uiemCompareField    *
//*************************
//******************************************************************************
//* Compare the specified tag field(s) in the edit buffer 'eBuff' with the     *
//* corresponding live data for the file under edit.                           *
//* This is used to determine the status of the edits-pending flag.            *
//*                                                                            *
//* Input  : eBuff    : (by reference) buffer containing the tag data for the  *
//*                     file under edit                                        *
//*          fldIndex : (optional, tfCOUNT by default) member of enum TagFields*
//*                     -- if 'tfCOUNT', compare all active fields of eBuff    *
//*                        with the live data for the file                     *
//*                     -- else, compare only the specified field              *
//*                                                                            *
//* Returns: 'true'  if one or more compared fields are different              *
//*          'false' if compared field(s) are identical                        *
//******************************************************************************

bool Taggit::uiemCompareField ( const tagFields& eBuff, short fldIndex )
{
   gString gs ;
   short indx = fldIndex == tfCOUNT ? ZERO : fldIndex,
         fi = this->tData.sfFocus ;    // file index
   bool status = false ;               // return value

   for ( ; indx < tfCOUNT ; ++indx )
   {
      if ( this->tData.sffDisp[indx] != false )  // if the field is active
      {
         //* Compare the fields (test is case-sensitive) *
         gs = this->tData.sf[fi].sfTag.field[indx] ;
         if ( (gs.compare( eBuff.field[indx], true )) != ZERO )
         {
            status = true ;
            break ;
         }
      }
      if ( fldIndex != tfCOUNT )    // if comparing only one field, we're done
         break ;
   }
   return status ;

}  //* End uiemCompareField() *

//*************************
//*    uiEditsPending     *
//*************************
//******************************************************************************
//* Warn user that edits are pending and ask whether to save before exit.      *
//*                                                                            *
//* Input  : modFiles : number of files with edits pending                     *
//*                                                                            *
//* Returns: 1 == user wants to save edits to target files                     *
//*          0 == user wants to cancel exit and continue editing               *
//*         -1 == user wants to discard edits and exit immediately             *
//******************************************************************************

short Taggit::uiEditsPending ( short modFiles )
{
   const short dlgROWS  = 12,             // dialog lines
               dlgCOLS  = 60,             // dialog columns
               dlgY     = fieldwinY + 1,  // dialog position
               dlgX     = fieldwinX + 1 ;
   attr_t dColor = this->cs.sd,           // dialog interior color
          hColor = this->cs.em,           // bold dialog text
          uColor = this->cs.unsaved ;     // color for modified data
   //* Set the edits-pending color for the color scheme *
   if ( this->cs.scheme == ncbcCOLORS )
      uColor = nc.rebl ;

   AppLang lang = this->cfgOpt.appLanguage ; // language index
   gString gsOut, gstmp ;                 // text formatting
   short action = ZERO ;                  // return value
   enum Controls : short { savePB = ZERO, discardPB, cancelPB, CONTROLS } ;


   const char* Labels[][7] = 
   {
      {  // English
         "  Warning! Edits Pending.  ",            // (0) dialog title
         "    Save  Data    \n"                    // (1) Save pushbutton
         "       then       \n"
         "       Exit       ",

         " Discard  Changes \n"                    // (2) Discard pushbutton
         "       and        \n"
         "       Exit       ",

         "   Do Not Exit.   \n"                    // (3) Return pushbutton
         "    Return to     \n"
         "   Application    ",

         "Caution!  %hd file, indicated by the X has "      // (4) template A
         "been edited,\nbut not saved.%s",

         "Caution!  %hd files, indicated by the X have "    // (5) template B
         "been edited,\nbut not saved.%s",

         "\n\n             Do you want to save your work?", // (6) save message
      },
      {  // Espanol
         "  ¡Advertencia! Cambios pendientes.  ",  // (0) dialog title
         " Guarde los Datos \n"                    // (1) Save pushbutton
         "     y luego      \n"
         "      Salga.      ",

         " Desechar cambios \n"                    // (2) Discard pushbutton
         "        y         \n"
         "      Salir.      ",

         "     No Salga.    \n"                    // (3) Return pushbutton
         "   Volver a la    \n"
         "    Aplicación.   ",

         "¡Precaución!  %hd archivo, indicado por la X ha sido\n"    // (4) template A
         "modificado, pero no se guardado.%s",

         "¡Precaución!  %hd archivos, indicados por la X han sido\n" // (5) template B
         "modificados, pero no guardados.%s",

         "\n\n              ¿Quiere guardar su trabajo?", // (6) save message
      },
      {  // Zhongwen    [别担心，快乐！ "Don't worry, be happy!"]
         "   警报 已经进行了编辑。   ",                // (0) dialog title
         "    保存数据，    \n"                      // (1) Save pushbutton
         "                  \n"
         "    然后退出。    \n",

         "    放弃更改，    \n"                      // (2) Discard pushbutton
         "                  \n"
         "    然后退出。    \n",

         "    不要退出。    \n"                      // (3) Return pushbutton
         "                  \n"
         "   返回应用程序。 ",

         "请谨慎行事！ 由 X 标记的 %hd 个文件已被修改，但尚未保存。%s",  // (4) template A

         "请谨慎行事！ 由 X 标记的 %hd 个文件已被修改，但尚未保存。%s",  // (5) template B

         "\n\n                    您要保存更改吗？",// (6) save message
      },
      {  // TiengView
         "  Cảnh báo! Chỉnh sửa Đang chờ xử lý.  ", // (0) dialog title
         "   Lưu dữ liệu,   \n"                    // (1) Save pushbutton
         "     sau đó       \n"
         "      thoát.      ",

         "   Hủy thay đổi,  \n"                    // (2) Discard pushbutton
         "     sau đó       \n"
         "      thoát.      ",

         "    Đừng thoát.   \n"                    // (3) Return pushbutton
         "     Trở lại      \n"
         "     ứng dụng.    ",

         "Chú ý!  %hd tập tin, được chỉ bởi X đã được sửa đổi,\n" // (4) template A
         "nhưng không được lưu.%s",

         "Chú ý!  %hd tệp, được chỉ bởi X đã được sửa đổi,\n"     // (5) template B
         "nhưng chưa được lưu.%s",

         "\n\n      Bạn có muốn lưu lại công việc của bạn không?", // (6) save message
      },
   } ;

InitCtrl ic[CONTROLS] =       // array of dialog control info
{
   {  //* 'Save' pushbutton - - - - - - - - - - - - - - - - - - - - -   savePB *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(dlgROWS - 4),           // ulY:       upper left corner in Y
      2,                            // ulX:       upper left corner in X
      3,                            // lines:     (n/a)
      18,                           // cols:      control columns
      Labels[lang][1],              // dispText:  
      this->cs.pn,                  // nColor:    non-focus color
      this->cs.pf,                  // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "",                           // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[discardPB],               // nextCtrl:  link in next structure
   },
   {  //* 'Discard' pushbutton - - - - - - - - - - - - - - - - - -   discardPB *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      ic[savePB].ulY,               // ulY:       upper left corner in Y
      short(ic[savePB].ulX + ic[savePB].cols + 1), // ulX: upper left corner in X
      3,                            // lines:     (n/a)
      18,                           // cols:      control columns
      Labels[lang][2],              // dispText:  
      this->cs.pn,                  // nColor: non-focus color
      this->cs.pf,                  // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "",                           // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[cancelPB]                 // nextCtrl:  link in next structure
   },
   {  //* 'Cancel' pushbutton - - - - - - - - - - - - - - - - - - -   cancelPB *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      ic[discardPB].ulY,            // ulY:       upper left corner in Y
      short(ic[discardPB].ulX + ic[discardPB].cols + 1), // ulX: upper left corner in X
      3,                            // lines:     (n/a)
      18,                           // cols:      control columns
      Labels[lang][3],              // dispText:  
      this->cs.pn,                  // nColor:    non-focus color
      this->cs.pf,                  // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "",                           // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      NULL                          // nextCtrl:  link in next structure
   },
} ;

   //* Dialog definition *
   InitNcDialog dInit( dlgROWS,         // number of display lines
                       dlgCOLS,         // number of display columns
                       dlgY,            // Y offset from upper-left of terminal 
                       dlgX,            // X offset from upper-left of terminal 
                       NULL,            // dialog title
                       ncltDUAL,        // border line-style
                       dColor,          // border color attribute
                       dColor,          // interior color attribute
                       ic               // pointer to list of control definitions
                     ) ;

   this->dPtr->SetDialogObscured () ;  // save parent dialog

   //* Open the dialog *
   NcDialog* dp = new NcDialog ( dInit ) ;

   //* If user interface language is an RTL language, *
   //* set the internal NcDialog flag for RTL output. *
   if ( this->cfgOpt.rtl )
      dp->DrawLabelsAsRTL () ;

   if ( (dp->OpenWindow()) == OK )
   {
      //* Set the dialog title *
      dp->SetDialogTitle ( Labels[lang][0], hColor ) ;

      gsOut.compose( Labels[lang][modFiles == 1 ? 4 : 5], &modFiles, Labels[lang][6] ) ;
      winPos wp( 2, (this->cfgOpt.rtl ? (dlgCOLS - 3) : 2) ) ;
      dp->WriteParagraph ( wp, gsOut, hColor, false, this->cfgOpt.rtl ) ;
      short x = gsOut.find( L'X' ) ;
      gsOut.limitChars( x ) ;
      x = gsOut.gscols() ;
      wp.xpos += (this->cfgOpt.rtl ? -(x) : x) ;
      dp->WriteChar ( wp, fillCIRCLE, uColor ) ;

      dp->RefreshWin () ;           // make everything visible


      //* Interact with the user *
      uiInfo   Info ;                     // user interface data returned here
      short    icIndex = ZERO ;           // index of control with focus
      bool     done = false ;             // loop control
      while ( ! done )
      {
         if ( ic[icIndex].type == dctPUSHBUTTON )
         {
            if ( Info.viaHotkey )
               Info.HotData2Primary () ;
            else
               icIndex = dp->EditPushbutton ( Info ) ;
            if ( Info.dataMod != false )
            {
               if ( Info.ctrlIndex == savePB )
                  action = 1 ;

               else if ( Info.ctrlIndex == discardPB )
                  action = -1 ;

               else if ( Info.ctrlIndex == cancelPB )
                  action = ZERO ;

               done = true ;
            }
         }

         //* Move focus to appropriate control *
         if ( ! done && ! Info.viaHotkey )
         {
            if ( Info.keyIn == nckSTAB )
               icIndex = dp->PrevControl () ; 
            else
               icIndex = dp->NextControl () ;
         }
      }  // while()
   }
   if ( dp != NULL )                         // close the window
      delete ( dp ) ;

   this->dPtr->RefreshWin () ;         // restore parent dialog

   return action ;

}  //* End uiEditsPending() *

//*************************
//* uiDefineReservedKeys  *
//*************************
//******************************************************************************
//* Create a list of keycodes that will be used for access to clipboard        *
//* functionality AND instruct the NcDialog class EditTextbox() method to      *
//* handle these keys as a special case. These reserved keycodes will mask any *
//* hotkeys of the same value which may be defined for the dialog.             *
//*                                                                            *
//* These keycodes give all dialog Textbox controls access to the NcDialog API *
//* local clipboard, and if the appropriate extended code is in place, the     *
//* system (X) clipboard as well. Note that the X-clipboard access from the    *
//* NcDialog API is currently broken.                                          *
//*                                                                            *
//* Input  : pointer to caller's dialog window                                 *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void Taggit::uiDefineReservedKeys ( NcDialog* dp )
{
   wkeyCode copy( caCOPY_SELECTED, wktFUNKEY ), // copy selected text
             cut( caCUT_SELECTED, wktFUNKEY ),  // cut selected text
           paste( caPASTE, wktFUNKEY ),         // paste clipboard to target
          selAll( caSELECT_ALL, wktFUNKEY ),    // select 'all' text in source
         selLeft( caSELECT_LEFT, wktFUNKEY ),   // select from cursor leftward
        selRight( caSELECT_RIGHT, wktFUNKEY ) ; // select from cursor rightward
    reservedKeys rk( copy, cut, paste, selAll, selLeft, selRight ) ;

   dp->SetTextboxReservedKeys ( rk ) ;
   
}  //* End uiDefineReservedKeys() *

//*************************
//*   MenuItem2MenuCode   *
//*************************
//******************************************************************************
//* Non-member Method:                                                         *
//* ------------------                                                         *
//* A selection has been made in the menu system. Translate the selection      *
//* into an action code, so command may be executed.                           *
//*                                                                            *
//* Input  : menuIndex:  index of control from which selection was made        *
//*          menuMember: item number selected within the menu                  *
//*                                                                            *
//* Returns: member of enum MenuCode                                           *
//******************************************************************************

MenuCode MenuItem2MenuCode ( short menuIndex, short menuMember )
{
   MenuCode    selection = mcNO_SELECTION ;     // return value

   switch ( menuIndex )
   {
      case twFileMW:                      // twFileMW menu
         selection = mbFileCodes[menuMember] ;
         break ;
      case twEditMW:                      // twEditMW menu
         selection = mbEditCodes[menuMember] ;
         break ;
      case twViewMW:                      // twViewMW menu
         selection = mbViewCodes[menuMember] ;
         break ;
      case twHelpMW:                      // twHelpMW menu
         selection = mbHelpCodes[menuMember] ;
         break ;
      case twSortMW:                      // twSortMW menu
         selection = mbSortCodes[menuMember] ;
         break ;

      default:                            // invalid menu index (unlikely)
         break ;
   }
   return selection ;

}  //* End MenuItem2MenuCode() *

//*************************
//*                       *
//*************************
//******************************************************************************
//*                                                                            *
//*                                                                            *
//*                                                                            *
//* Input  :                                                                   *
//*                                                                            *
//* Returns:                                                                   *
//******************************************************************************

