//******************************************************************************
//* File       : TagData.hpp                                                   *
//* Author     : Mahlon R. Smith                                               *
//*              Copyright (c) 2016-2020 Mahlon R. Smith, The Software Samurai *
//*                 GNU GPL copyright notice below                             *
//* Date       : 08-Oct-2020                                                   *
//* Version    : (see AppVersion string below)                                 *
//*                                                                            *
//* Description: Data structure class definition and miscellaneous constant    *
//*              definitions for the Taggit application.                       *
//*                                                                            *
//*                                                                            *
//******************************************************************************
//* Copyright Notice:                                                          *
//* This program is free software: you can redistribute it and/or modify it    *
//* under the terms of the GNU General Public License as published by the Free *
//* Software Foundation, either version 3 of the License, or (at your option)  *
//* any later version.                                                         *
//*                                                                            *
//* This program is distributed in the hope that it will be useful, but        *
//* WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY *
//* or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License   *
//* for more details.                                                          *
//*                                                                            *
//* You should have received a copy of the GNU General Public License along    *
//* with this program.  If not, see <http://www.gnu.org/licenses/>.            *
//*                                                                            *
//*         Full text of the GPL License may be found in the TexInfo           *
//*         documentation for this program under 'Copyright Notice'.           *
//******************************************************************************


//***************
//* Definitions *
//***************

const short MAX_PATH  = gsMAXBYTES ;      // POSIX definition for path/filename length
const short MAX_FNAME = MAX_PATH / 16 ;   // POSIX definition for filename length

//*************************************************************
//* List of filetypes supported for decoding of file metadata *
//*************************************************************
enum MediafileType : short { mftMP3, mftOGG, mftASF, mftNONE } ;


//*******************************************
//* Indices into the tagFields-class arrays *
//*******************************************
//* This enum determines the order of storage and display of the metadata tag *
//* fields. All ordered operations must conform to this sequence.             *
//* -- The list consists of all text fields defined by the ID3v2.3.0 (MP3)    *
//*    specification, and all other supported media formats are adapted to it.*
//* -- Ordering of this list is according to the author's idea of which tags  *
//*    are most popular and useful, greatest-to-least.                        *
//* -- If this list changes, all dependent code must also be updated.         *
//*    -- 'CfgParm' array in TagConfig.cpp                                    *
//*    -- The 'Enable_xxx' group in the configuration file.                   *
//*    -- Headers for field display (UpdateMetadataWindow())                  *
//*    -- Switch cases for field display (UpdateMetadataWindow())             *
//*    -- Records for field display.                                          *
//*    -- Comparison and summary-display arrays in TagMp3.hpp                 *
//* ---  ---  ---  ---  ---  ---  ---  ---  ---  ---  ---  ---  ---  ---  --- *
enum TagFields : short
{
   tfTit2 = ZERO, // #TIT2 Title/songname/content description
   tfTpe1,        // #TPE1 Artist/Lead performer(s)/Soloist(s)
   tfTalb,        // #TALB Album/Movie/Show title
   tfTrck,        // #TRCK Track number/(and optionally, track count)
   tfTyer,        // #TYER Year (4-character field: "YYYY")
   tfTcon,        // #TCON Content type (genre), one of the defined codes OR freeform
                  //       (see spec for use of parentheses)
   tfTpe2,        // #TPE2 Guest artist/band/orchestra/accompaniment
   tfTpub,        // #TPUB Publisher, record label
   tfTcom,        // #TCOM Composer(s) (separated by the '/' character)
   tfText,        // #TEXT Lyricist/Text writer
   tfTpe4,        // #TPE4 Interpreted, remixed, or otherwise modified by
   tfTcop,        // #TCOP Copyright message (first 5 chars: "YYYY " + any additional info)
                  //       When displayed: "Copyright © YYYY nnnnnn..."
   tfTown,        // #TOWN File owner/licensee
   tfTxxx,        // #TXXX User defined text information frame

   tfTdat,        // #TDAT Date of the recording (4-character field: "DDMM")
   tfTime,        // #TIME Time (4-character field: "HHMM")
   tfTrda,        // #TRDA Recording dates
   tfTlen,        // #TLEN Length (numeric string: play length of audio in milliseconds)
   tfTsiz,        // #TSIZ Size (numeric string: size of audio in bytes, excl. metadata)

   tfTbpm,        // #TBPM BPM (beats per minute)
   tfTit1,        // #TIT1 Content group description
   tfTit3,        // #TIT3 Subtitle/Description refinement
   tfTpe3,        // #TPE3 Conductor/performer refinement
   tfTpos,        // #TPOS Part of a set
   tfTkey,        // #TKEY Initial key
   tfTlan,        // #TLAN Language(s)

   tfTope,        // #TOPE Original artist(s)/performer(s)
   tfToal,        // #TOAL Original album/movie/show title
   tfTofn,        // #TOFN Original filename
   tfToly,        // #TOLY Original lyricist(s)/text writer(s)
   tfTory,        // #TORY Original release year

   tfTrsn,        // #TRSN Internet radio station name
   tfTrso,        // #TRSO Internet radio station owner

   tfTsrc,        // #TSRC ISRC (international standard recording code)
   tfTsee,        // #TSEE Software/Hardware and settings used for encoding
   tfTflt,        // #TFLT File type (MPG /1 /2 /3 /2.5 /AAC VQF PCM)
   tfTdly,        // #TDLY Playlist delay (numeric string: Milliseconds delay between songs)
   tfTenc,        // #TENC Encoded by
   tfTmed,        // #TMED Media type (DIG ANA CD LD TT TV RAD, etc.)

   //*** Any specialized fields for non-MP3 formats are to be inserted here. ***

   tfCOUNT        // NUMBER OF ELEMENTS IN ARRAYS
} ;


//* Information for an image embedded in the metadata.                         *
class EmbeddedImage
{
   public:

   ~EmbeddedImage ( void ) {}       // destructor
   EmbeddedImage ( void )
   { this->reset() ; }

   void reset ( void )
   {
      this->mp3img.reset() ;
      this->next = this->prev = NULL ;
      this->imMod = this->inUse = false ;
   }

   //***********************
   //* Public Data Members *
   //***********************
   //* id3v2 image frame data definition                                       *
   id3v2_image mp3img ;

   //* 'true' if image or associated data have been modified, else false.      *
   bool imMod ;

   //* 'true' if 'mp3img' member contains data.                                *
   bool inUse ;

   EmbeddedImage* next ;            // pointers for a linked list of objects
   EmbeddedImage* prev ;

} ;

//* Tag data for the POPM 'Popularimeter' and PCNT 'Play Counter'
class popMeter
{
   public:
   popMeter ( void )
   {
      this->reset () ;
   }
   void reset ( void )
   {
      this->popEmail[0] = NULLCHAR ;
      this->popCount = this->playCount = ZERO ;
      this->popStar = ZERO ;
      this->popdata = this->cntdata = this->pcMod = false ;
   }

   //* Convert a 4-byte sequence (MSB at offset 0) to an integer value. *
   //* Programmer's Note: This clunky construct avoids the C library's  *
   //* "helpful" automatic sign extension.                              *
   UINT intConv ( const uint8_t* ucp ) const
   {
      UINT i =    (UINT)ucp[3] 
               | ((UINT)ucp[2] << 8)
               | ((UINT)ucp[1] << 16)
               | ((UINT)ucp[0] << 24) ;
      return i ;
   }

   //* Convert a 32-bit integer into a big-endian byte stream.    *
   //* (used for both 'frame-size' and 'decomp')                  *
   short intConv ( UINT uval, char* obuff ) const
   {
      const int bMASK = 0x000000FF ;
      short indx = ZERO ;     // return value

      obuff[indx++] = (char)((uval >> 24) & bMASK) ;
      obuff[indx++] = (char)((uval >> 16) & bMASK) ;
      obuff[indx++] = (char)((uval >>  8) & bMASK) ;
      obuff[indx++] = (char)(uval & bMASK) ;

      return indx ;
   }

   char popEmail[gsMAXBYTES] ; // User's email address, etc.
   uint32_t popCount ;  // Popularimeter (personal) play counter
   uint32_t playCount ; // Play-count counter
   uint8_t  popStar ;   // Popularimeter value (star rating)
   bool     popdata ;   // 'true' if POPM data read from source file OR entered by user
   bool     cntdata ;   // 'true' if PCNT data read from source file OR entered by user
   bool     pcMod ;     // 'true' if user has modified the data
} ;   // popMeter

//* Tag fields for an individual source file *
class tagFields
{
   public:

   ~tagFields ( void )              // destructor
   { this->delPic () ; }
   tagFields ( void )               // constructor
   { this->reset () ; }
   void reset ( void )
   {
      this->clear_data () ;         // set all text fields to empty string
      this->tfMod = false ;         // no pending edits
      for ( short i = ZERO ; i < tfCOUNT ; ++i ) // assume no pre-existing metadata
         this->prex[i] = false ;
   }

   //* Clear (set to empty string) all text fields.   *
   //* Also discard image data and popularimeter data.*
   void clear_data ( void )
   {
      for ( short i = ZERO ; i < tfCOUNT ; ++i )
         this->field[i][0] = NULLCHAR ;
      this->delPic() ;     // detach all image files
      this->pop.reset() ;  // discard popularimeter/counter data
   }

   //* Returns 'true' if any field contains non-null data *
   bool contains_data ( void )
   {
      bool is_data = false ;
      for ( short i = ZERO ; i < tfCOUNT ; ++i )
      {
         if ( *this->field[i] != NULLCHAR )
         {
            is_data = true ;
            break ;
         }
      }
      if ( ! is_data )
      {
         if ( this->ePic.inUse || this->pop.popdata || this->pop.cntdata )
            is_data = true ;
      }
      return is_data ;
   }

   //* Create a node on the 'ePic' linked list of image data objects.          *
   EmbeddedImage* addPic ( void )
   {
      EmbeddedImage* newPtr = &this->ePic ;
      while ( newPtr->inUse )
      {
         if ( newPtr->next == NULL )
         {
            newPtr->next = new EmbeddedImage ;
            newPtr->next->prev = newPtr ;
         }
         newPtr = newPtr->next ;
      }
      return newPtr ;
   }

   //* Delete the indicated node from the 'ePic' linked list of image data     *
   //* objects. Relink any succeeding images to parent of the deleted image.   *
   //* If an image file is attached to the object ('trgPtr->picPath[]'), then  *
   //* caller is responsible for managing the file since it will be orphaned   *
   //* by this operation.                                                      *
   void delPic ( EmbeddedImage* trgPtr )
   {
      //* Locate the indicated image in the list *
      EmbeddedImage *srcPtr = &this->ePic,
                    *src, 
                    *trg ;
      while ( (srcPtr != trgPtr) && (srcPtr != NULL) )
         srcPtr = srcPtr->next ;

      if ( srcPtr != NULL )      // if specified node was found
      {  // LOGIC:
         // 1) target is base node
         //    a) base node is the only node
         //       reset the base node
         //    b) base node has attached children
         //       -- move each child's data upward
         //       -- set last active node's 'next' to NULL
         //       -- delete the tail node
         // 2) target is a child node
         //    a) target has children
         //       -- move each child's data upward
         //       -- set last active node's 'next' to NULL
         //       -- delete the tail node
         //    b) target is the last node in the list
         //       -- set target's parent node 'next' to NULL
         //       -- delete the tail node
         // Note that 1b and 2a, once initialized, can share the processing loop.
         if ( (srcPtr == &this->ePic) && (srcPtr->next == NULL) )
            this->ePic.reset() ;             // reset base node
         else if ( (srcPtr != &this->ePic) && (srcPtr->next == NULL) )
         {
            srcPtr->prev->next = NULL ;      // detach tail node
            delete srcPtr ;                  // delete tail node
         }
         else
         {
            if ( srcPtr == &this->ePic )
               trg = srcPtr ;
            else     // srcPtr is neither base node nor tail node
               trg = srcPtr ;
            src = trg->next ;
            do
            {
               trg->mp3img = src->mp3img ;
               trg->imMod  = src->imMod ;
               trg->inUse  = src->inUse ;
               // Note: 'prev' member is correct as-is
               trg = src ;
               src = trg->next ;
            }
            while ( src != NULL ) ;
            trg->prev->next = NULL ;      // detach tail node
            delete trg ;                  // delete tail node
         }
      }
   }     // delPic()

   //* Delete all nodes from the 'ePic' linked list of image data objects.     *
   //* If image files are attached to the objects ('ePic.picPath[]'), then     *
   //* caller is responsible for managing them.                                *
   void delPic ( void )
   {
      EmbeddedImage* trgPtr = &this->ePic ;
      while ( trgPtr->next != NULL )      // walk to the end of the list
         trgPtr = trgPtr->next ;
      while ( trgPtr->prev != NULL )      // walk to top of list, deleting each node
      {
         trgPtr = trgPtr->prev ;
         delete trgPtr->next ;
         trgPtr->next = NULL ;
      }
      this->ePic.reset() ; // reinitialize the base node
   }

   //* Count the number of entries in the EmbeddedImage linked list.           *
   short countPics ( void )
   {
      EmbeddedImage* eiPtr = &this->ePic ;   // base node of list
      short iVectors = ZERO ;                // return value

      if ( eiPtr->inUse != false )
      {
         ++iVectors ;
         while ( eiPtr->next != NULL )
         { ++iVectors ; eiPtr = eiPtr->next ; }
      }
      return iVectors ;
   }

   //******************
   //** Data Members **
   //******************

   //* Text for tag fields *
   wchar_t field[tfCOUNT][gsMAXCHARS] ;

   //* This flag indicates whether the file had PRE-EXISTING (non-blank)       *
   //* metadata in the corresponding field when the file was first scanned.    *
   //*            'true'    pre-existing data                                  *
   //*            'false'   field not present or field blank in original file  *
   bool prex[tfCOUNT] ;

   //* Image definition: This is the base node of a linked list which may be   *
   //* dynamically allocated if multiple images are found in a source file.    *
   //* -- Note that the MP3 standard allows multiple images in a single file,  *
   //     so long as there is no duplication of "content descriptor".          *
   EmbeddedImage ePic ;

   //* Popularimeter and Play Count frame data.                                *
   popMeter pop ;

   bool tfMod ;      // 'true'  if one or more fields contains unsaved data
                     //         or if image or popMeter data have been updated
                     // 'false' if all fields match the corresponding source file data
} ;   // tagFields

//* Source file definition with metadata tag fields *
class SrcFile
{
   public:
   ~SrcFile ( void ) {}             // destructor
   SrcFile ( void )                 // default constructor
   { this->reset () ; }
   void reset ( void )              // set data members to default values
   {
      this->sfPath[0] = NULLCHAR ;  // no path/filename string
      this->sfName[0] = NULLCHAR ;  // no filename string
      this->sfType    = mftNONE ;   // media format unknown
      this->sfMod     = false ;     // no edits pending
   }

   //** Data Members **
   char sfPath[gsMAXBYTES] ;        // Source path/filename
   char sfName[MAX_FNAME] ;         // Source filename
   MediafileType sfType ;           // Media file format
   tagFields sfTag ;                // Metadata (tag fields)
   bool sfMod ;                     // 'true' if filename edits pending
} ;   // SrcFile

//* Tag data display and editing structure *
const short MAX_SRCFILES = 99 ;
class tagData
{
   public:

   ~tagData ( void ) {}          // destructor
   tagData ( void )              // constructor
   { this->reset () ; }
   void reset ( bool enable_all = true )
   {
      this->sfCount = this->sfFirst = this->sfLast = 
      this->sfFocus = this->sffOffset = ZERO ;
      this->sffLastField = false ;
      if ( enable_all )
         this->enable_all ( true ) ;   // set all display fields active
   }

   //* Enable or disable all fields.   *
   //* Exceptions:                     *
   //*   Title (tfTit2) is always set. *
   //*   Image-active flag is unchanged*
   void enable_all ( bool enable )
   {
      for ( short i = ZERO ; i < tfCOUNT ; ++i )
         this->sffDisp[i] = enable ;
      this->sffDisp[tfTit2] = true ;
   }

   //******************
   //** Data Members **
   //******************

   SrcFile sf[MAX_SRCFILES] ;       // array of source file records
   short sfCount ;                  // number of source records in sFile array
   short sfFirst ;                  // index of first filename displayed at top of window
   short sfLast ;                   // index of last filename displayed at bottom of window
   short sfFocus ;                  // index of filename with the input focus
   bool  sffDisp[tfCOUNT] ;         // indicates whether field is included in display/edit
   short sffOffset ;                // field-display offset (displayed at left of window LTR)
   bool  sffLastField ;             // 'true' if last field displayed (right of window LTR)
} ;

