//******************************************************************************
//* File       : TagAsf.hpp                                                    *
//* Author     : Mahlon R. Smith                                               *
//*              Copyright (c) 2016-2020 Mahlon R. Smith, The Software Samurai *
//*                  GNU GPL copyright notice located in Taggit.hpp            *
//* Date       : 15-Oct-2020                                                   *
//* Version    : (see AppVersion string)                                       *
//*                                                                            *
//* Description: This header defines the metadata fields for ASF (WMA/WMV)     *
//*              audio and video files.                                        *
//*                                                                            *
//******************************************************************************

//****************
//* Header Files *
//****************


//************************
//* Constant Definitions *
//************************

const short asfASF_HDR_BYTES = 30 ; // number of bytes in ASF container header
const short asfGUID_BYTES = 16 ;    // number of bytes in a GUID field
const short asfGUID_ASCII = 37 ;    // ASCII-hex equivalent of asfGUID_BYTES (incl. null)
const short asfSIZE_BYTES = 8 ;     // number of bytes in an object size value (uint64_t)
const short asfHOBJ_BYTES = 4 ;     // number of bytes in header-object count value (uint32_t)
const uint8_t asfRES1     = 0x01 ;  // value of header reserved-byte #1
const uint8_t asfRES2     = 0x02 ;  // value of header reserved-byte #1
const short asfMIN_CDO    = 34 ;    // minimum number of bytes in CDO
const short asfMIN_ECDO   = 26 ;    // minimum number of bytes in ECDO
const short asfDESC_BYTES = 2 ;     // number of bytes in descriptor-count and descriptor sizes
const short asfECD_MAX_COUNT = 64 ; // maximum number of Extended Content Descriptors (array size)
extern const char* HEADER_GUID ;    // ASF_Header_Object
extern const char* CDO_GUID ;       // ASF_Content_Description_Object
extern const char* ECDO_GUID ;      // ASF_Extended_Content_Description_Object
extern const char* CBO_GUID ;       // ASF_Content_Branding_Object
extern const char* MO_GUID ;        // ASF_Metadata_Object
extern const char* MLO_GUID ;       // ASF_Metadata_Library_Object
extern const uint8_t EncodedGUID[][asfGUID_BYTES] ; //* Encoded GUID VALUES


//* Convert GUID from little-endian (mixed with lunacy) *
//* to the character order defined in the specification.*
const short canonicalOrder[asfGUID_ASCII] = 
{  6,  7,  4,  5,  2,  3,  0,  1, 0x002D, 10, 11,  8,  9, 0x002D, 14, 15, 
  12, 13, 0x002D, 16, 17, 18, 19, 0x002D, 20, 21, 22, 23, 24, 25, 26, 27, 
  28, 29, 30, 31, 32 } ;

//* Data types for Extended ContentDescriptionObject descriptor value fields. *
//* Note: non-text data converted to ASCII hex for local storage and display. *
enum asfDataType : short
{
   asfdtUNKNOWN = (-1),
   asfdtUTF16,             // UTF-16LE (no BOM)
   asfdtBYTE,              // byte array    (image, or other binary data)
   asfdtBOOL,              // boolean value (source: 32 bits) [not text]
   asfdtDWORD,             // "DWORD"       (source: 32 bits) [not text]
   asfdtQWORD,             // "QWORD"       (source: 64 bits) [not text]
   asfdtWORD,              // "WORD"        (source: 16 bits) [not text]
} ;

//* Return value from asfConDesc::validateObjectHeader().   *
//* These are the valid metadata objects specified by the   *
//* Advanced Systems Format (ASF) specification.            *
enum ObjHdrType : short
{
   ohtCDO,                 // Content Description Object
   ohtECDO,                // Extended Content Description Object
   ohtCBO,                 // Content Branding Object
   ohtMO,                  // Metadata Object
   ohtMLO,                 // Metadata Library Object
   ohtNONE,                // Invalid or unrecognized object structure
} ;


//***********************************
//* Classes for processing ASF data *
//***********************************

//* Top-level Header Object data.*
class asfHeader
{
   public:
   ~asfHeader ( void ) {}        // destructor
   asfHeader ( void )            // default constructor
   {
      this->reset() ;
   }

   void reset ( void )
   {
      this->guid[0] = '\0' ;
      this->size = 0 ;
      this->hdrobj = 0 ;
      this->res1 = this->res2 = 0 ;
   }


   //* Normalize, validate and store the GUID value.   *
   //* Decode and store the object size.               *
   //* Decode and store the number of embedded objects.*
   //* Verify and store the two reserved bytes.        *
   bool validateHeader ( const char* srcPtr )
   {
      uint8_t uBuff[asfGUID_ASCII+1] ;
      uint8_t rawByte, lowNybble, highNybble ;
      short   srcindx = ZERO, trgindx = ZERO ;
      bool valid = false ;

      //* Decode the GUID byte stream into ASCII hex *
      for ( ; srcindx < asfGUID_BYTES ; ++srcindx )
      {
         rawByte = srcPtr[srcindx] ;
         highNybble = (uint8_t)(rawByte >> 4) ;
         if ( highNybble <= 0x09 )  highNybble += 0x30 ;          // '0'
         else                       highNybble += 0x41 - 0x0A ;   // 'A'
         lowNybble = (uint8_t)(rawByte & 0x0F) ;
         if ( lowNybble <= 0x09 )   lowNybble += 0x30 ;           // '0'
         else                       lowNybble += 0x41 -0x0A ;     // 'A'
         uBuff[trgindx++] = highNybble ;
         uBuff[trgindx++] = lowNybble ;
      }
      uBuff[trgindx] = '\0' ;

      //* Place data in canonical order *
      for ( trgindx = ZERO ; trgindx <= asfGUID_ASCII ; ++trgindx )
      {
         if ( canonicalOrder[trgindx] == 0x002D )
            this->guid[trgindx] = '-' ;
         else
         {
            this->guid[trgindx] = uBuff[canonicalOrder[trgindx]] ;
            if ( this->guid[trgindx] == '\0' )
               break ;
         }
      }
      this->guid[trgindx] = '\0' ;  // be sure sequence is terminated

      //* Extract header size (uint64_t) *
      this->size  =  ((uint64_t)srcPtr[srcindx++] & 0x00FF) ;
      this->size |= (((uint64_t)srcPtr[srcindx++] & 0x00FF) << 8) ;
      this->size |= (((uint64_t)srcPtr[srcindx++] & 0x00FF) << 16) ;
      this->size |= (((uint64_t)srcPtr[srcindx++] & 0x00FF) << 24) ;
      this->size |= (((uint64_t)srcPtr[srcindx++] & 0x00FF) << 32) ;
      this->size |= (((uint64_t)srcPtr[srcindx++] & 0x00FF) << 40) ;
      this->size |= (((uint64_t)srcPtr[srcindx++] & 0x00FF) << 48) ;
      this->size |= (((uint64_t)srcPtr[srcindx++] & 0x00FF) << 56) ;

      //* Extract object count (uint32_t) *
      this->hdrobj  =  ((uint32_t)srcPtr[srcindx++] & 0x00FF) ;
      this->hdrobj |= (((uint32_t)srcPtr[srcindx++] & 0x00FF) << 8) ;
      this->hdrobj |= (((uint32_t)srcPtr[srcindx++] & 0x00FF) << 16) ;
      this->hdrobj |= (((uint32_t)srcPtr[srcindx++] & 0x00FF) << 24) ;

      //* Extract the reserved markers *
      this->res1 = srcPtr[srcindx++] ;
      this->res2 = srcPtr[srcindx] ;

      if ( ((strncmp ( (const char*)this->guid, HEADER_GUID, asfGUID_ASCII )) == 0) &&
           (this->res1 == asfRES1) && (this->res2 == asfRES2) )
      {
         valid = true ;
      }
      return valid ;
   }     // validateHeader()

   //* Encode a 64-bit QWORD into an eight-byte byte stream. *
   short encode64Bit ( uint64_t val64, char* trgBuff ) const
   {
      for ( short i = 0 ; i < asfSIZE_BYTES ; ++i )
      {
         trgBuff[i] = (val64 & 0x00FF) ;
         val64 >>= 8 ;
      }
      return asfSIZE_BYTES ;  // return size of encoded value
   }

   //* Encode a 32-bit DWORD into a four-byte byte stream. *
   short encode32Bit ( uint32_t val32, char* trgBuff ) const
   {
      for ( short i = 0 ; i < 4 ; ++i )
      {
         trgBuff[i] = (val32 & 0x00FF) ;
         val32 >>= 8 ;
      }
      return 4 ;     // return size of encoded value
   }

   //* Return the encoded GUID byte sequence.                  *
   //* Encoded sequences are defined in EncodedGUID[] array.   *
   void encodeGUID ( uint8_t *ubuff ) const
   {
      short indx = 5 ;           // index into EncodedGUID[] array

      for ( short i = 0 ; i < asfGUID_BYTES ; ++i )
         ubuff[i] = EncodedGUID[indx][i] ;

   }  // encodeGUID()

   //* Public data members *
   uint8_t  guid[asfGUID_ASCII+1] ; // Global Unique Identifier
   uint64_t size ;         // Size of object 24 bytes + size of data (>= 30)
   uint32_t hdrobj ;       // Number of object contained in header (excl. this object)
   uint8_t  res1 ;         // Reserved (always 0x01)
   uint8_t  res2 ;         // Reserved (always 0x02)
} ;

//* Extended Content Description Object decoded data *
class asfExDesc
{
   public:
   ~asfExDesc ( void ) {}        // destructor
   asfExDesc ( void )            // default constructor
   {
      this->reset() ;
   }

   void reset ( void )
   {
      this->descName[0] = this->descValue[0] = '\0' ;
      this->nameSize = this->valueSize = 0 ;
      this->dataType = asfdtUNKNOWN ;
   }

   //* Public data members *
   char descName[gsMAXBYTES] ;   // Name of descriptor  (UTF-8 text)
   char descValue[gsMAXBYTES] ;  // Value of descriptor (UTF-8 text)
   uint16_t nameSize ;           // bytes of encoded source for 'descName'
   uint16_t valueSize ;          // bytes of encoded source for 'descValue'
   asfDataType dataType ;        // type of source data for 'descValue' field
} ;

//* Content Description Object and Extended Content Description Object data.  *
//* Text data are stored as UTF-8 regardless of the native encoding used.     *
class asfConDesc
{
   public:
   ~asfConDesc ( void ) {}       // destructor
   asfConDesc ( void )           // default constructor
   {
      this->reset() ;
   }

   void reset ( void )
   {
      //* CDO data *
      this->guid[0] = this->title[0]  = this->author[0] = this->copyrt[0] = 
      this->desc[0] = this->rating[0] = 0 ;
      this->cdoSize =  0 ;
      this->titleSize = this->authorSize = this->copyrtSize = 
      this->descSize  = this->ratingSize = 0 ;
      this->cdoVerified = false ;
      //* ECDO data *
      this->eguid[0]  = '\0' ;   this->ecdoSize = 0 ; this->ecdoVerified = false ;
      this->descCount = 0 ;
      //* CBO data *
      this->cboguid[0] = '\0' ;  this->cboSize = 0 ;  this->cboVerified = false ;
      //* MO data
      this->moguid[0] = '\0' ;   this->moSize = 0 ;   this->moVerified = false ;
      //* MLO data *
      this->mloguid[0] = '\0' ;  this->mloSize = 0 ;  this->mloVerified = false ;
   }

   //* Decode the object GUID and object size.*
   //* If one of the following objects found, *
   //* store the data and return confirmation.*
   //* 1) Content Description Object          *
   //* 2) Extended Content Description Object *
   //* 3) Content Branding Object             *
   //* 4) Metadata Object                     *
   //* 5) Metadata Library Object             *
   ObjHdrType validateObjectHeader ( const char* srcPtr, uint64_t& objSize )
   {
      uint8_t uBuff[asfGUID_ASCII+1] ;          // temp storage (raw guid)
      uint8_t gtmp[asfGUID_ASCII+1] ;           // temp storage (formatted guid)
      uint8_t rawByte, lowNybble, highNybble ;  // integer conversion
      short   srcindx = ZERO, trgindx = ZERO ;
      ObjHdrType metaHdr = ohtNONE ;            // return value

      //* Decode the GUID byte stream into ASCII hex *
      for ( ; srcindx < asfGUID_BYTES ; ++srcindx )
      {
         rawByte = srcPtr[srcindx] ;
         highNybble = (uint8_t)(rawByte >> 4) ;
         if ( highNybble <= 0x09 )  highNybble += 0x30 ;          // '0'
         else                       highNybble += 0x41 - 0x0A ;   // 'A'
         lowNybble = (uint8_t)(rawByte & 0x0F) ;
         if ( lowNybble <= 0x09 )   lowNybble += 0x30 ;           // '0'
         else                       lowNybble += 0x41 -0x0A ;     // 'A'
         uBuff[trgindx++] = highNybble ;
         uBuff[trgindx++] = lowNybble ;
      }
      uBuff[trgindx] = '\0' ;

      //* Place data in canonical order *
      for ( trgindx = ZERO ; trgindx <= asfGUID_ASCII ; ++trgindx )
      {
         if ( canonicalOrder[trgindx] == 0x002D )
            gtmp[trgindx] = '-' ;
         else
         {
            gtmp[trgindx] = uBuff[canonicalOrder[trgindx]] ;
            if ( gtmp[trgindx] == '\0' )
               break ;
         }
      }
      gtmp[trgindx] = '\0' ;     // be sure sequence is terminated

      //* Decode the object size *
      objSize = this->decode64Bit ( &srcPtr[srcindx] ) ;

      //* Test the captured value against the five(5) valid metadata objects *
      //* If a match, save it to our data member.                            *
      // Programmer's Note: We can take some shortcuts with the test because 
      // the specification mandates that there be only one instance of each.
      if ( ((strncmp ( (char*)gtmp, CDO_GUID, asfGUID_ASCII )) == 0) )
      {
         strncpy ( (char*)this->guid, (char*)gtmp, asfGUID_ASCII ) ;
         this->cdoSize = objSize ;
         this->cdoVerified = true ;
         metaHdr = ohtCDO ;
      }
      else if ( ((strncmp ( (char*)gtmp, ECDO_GUID, asfGUID_ASCII )) == 0) )
      {
         strncpy ( (char*)this->eguid, (char*)gtmp, asfGUID_ASCII ) ;
         this->ecdoSize = objSize ;
         this->ecdoVerified = true ;
         metaHdr = ohtECDO ;
      }
      else if ( ((strncmp ( (char*)gtmp, CBO_GUID, asfGUID_ASCII )) == 0) )
      {
         strncpy ( (char*)this->cboguid, (char*)gtmp, asfGUID_ASCII ) ;
         this->cboSize = objSize ;
         this->cboVerified = true ;
         metaHdr = ohtCBO ;
      }
      else if ( ((strncmp ( (char*)gtmp, MO_GUID, asfGUID_ASCII )) == 0) )
      {
         strncpy ( (char*)this->moguid, (char*)gtmp, asfGUID_ASCII ) ;
         this->moSize = objSize ;
         this->moVerified = true ;
         metaHdr = ohtMO ;
      }
      else if ( ((strncmp ( (char*)gtmp, MLO_GUID, asfGUID_ASCII )) == 0) )
      {
         strncpy ( (char*)this->mloguid, (char*)gtmp, asfGUID_ASCII ) ;
         this->mloSize = objSize ;
         this->mloVerified = true ;
         metaHdr = ohtMLO ;
      }
      else  // save found-but-uninteresting GUID for debugging by caller
      {
         strncpy ( (char*)this->xguid, (char*)gtmp, asfGUID_ASCII ) ;
      }

      return metaHdr ;

   }  // validateObjectHeader()

   //* Extract the five(5) 16-bit tag-size values from the byte stream.*
   void decodeTagSizes ( const char* srcPtr )
   {
      short srcindx = ZERO ;
      this->titleSize = this->decode16Bit ( &srcPtr[srcindx] ) ;
      srcindx += asfDESC_BYTES ;
      this->authorSize = this->decode16Bit ( &srcPtr[srcindx] ) ;
      srcindx += asfDESC_BYTES ;
      this->copyrtSize = this->decode16Bit ( &srcPtr[srcindx] ) ;
      srcindx += asfDESC_BYTES ;
      this->descSize = this->decode16Bit ( &srcPtr[srcindx] ) ;
      srcindx += asfDESC_BYTES ;
      this->ratingSize = this->decode16Bit ( &srcPtr[srcindx] ) ;
   }

   //* Decode eight(8) bytes of the byte stream into a 64-bit QWORD *
   uint64_t decode64Bit ( const char* srcPtr )
   {
      uint64_t qword ;
      short srcindx = ZERO ;

      qword  =  ((uint64_t)srcPtr[srcindx++] & 0x00FF) ;
      qword |= (((uint64_t)srcPtr[srcindx++] & 0x00FF) << 8) ;
      qword |= (((uint64_t)srcPtr[srcindx++] & 0x00FF) << 16) ;
      qword |= (((uint64_t)srcPtr[srcindx++] & 0x00FF) << 24) ;
      qword |= (((uint64_t)srcPtr[srcindx++] & 0x00FF) << 32) ;
      qword |= (((uint64_t)srcPtr[srcindx++] & 0x00FF) << 40) ;
      qword |= (((uint64_t)srcPtr[srcindx++] & 0x00FF) << 48) ;
      qword |= (((uint64_t)srcPtr[srcindx++] & 0x00FF) << 56) ;
      return qword ;

   }  // decode64Bit()

   //* Encode a 64-bit QWORD into an eight-byte byte stream. *
   short encode64Bit ( uint64_t val64, char* trgBuff )
   {
      for ( short i = 0 ; i < asfSIZE_BYTES ; ++i )
      {
         trgBuff[i] = (val64 & 0x00FF) ;
         val64 >>= 8 ;
      }
      return asfSIZE_BYTES ;  // return size of encoded value
   }

   //* Decode four(4) bytes of the byte stream into a 32-bit DWORD *
   uint32_t decode32Bit ( const char* srcPtr )
   {
      uint32_t dword ;
      short srcindx = ZERO ;

      dword  =  ((uint32_t)srcPtr[srcindx++] & 0x00FF) ;
      dword |= (((uint32_t)srcPtr[srcindx++] & 0x00FF) << 8) ;
      dword |= (((uint32_t)srcPtr[srcindx++] & 0x00FF) << 16) ;
      dword |= (((uint32_t)srcPtr[srcindx++] & 0x00FF) << 24) ;

      return dword ;

   }  // decode32Bit()

   //* Encode a 32-bit DWORD into a four-byte byte stream. *
   short encode32Bit ( uint32_t val32, char* trgBuff )
   {
      for ( short i = 0 ; i < 4 ; ++i )
      {
         trgBuff[i] = (val32 & 0x00FF) ;
         val32 >>= 8 ;
      }
      return 4 ;     // return size of encoded value
   }

   //* Decode two(2) bytes of the byte stream into a 16-bit WORD *
   uint16_t decode16Bit ( const char* srcPtr )
   {
      uint16_t word ;
      short srcindx = ZERO ;

      word  =  ((uint16_t)srcPtr[srcindx++] & 0x00FF) ;
      word |= (((uint16_t)srcPtr[srcindx++] & 0x00FF) << 8) ;

      return word ;

   }  // decode16Bit()

   //* Encode a 16-bit WORD into a two-byte byte stream. *
   short encode16Bit ( uint16_t val16, char* trgBuff )
   {
      trgBuff[0] = (val16 & 0x00FF) ;
      trgBuff[1] = (val16 >> 8) ;
      return 2 ;     // return size of encoded value
   }

   //* Convert a Unicode-16 string (little-endian, no byte-order mark)  *
   //* to gString format. Source may or may not be null terminated.     *
   short utf16Decode ( const char* rawPtr, short rawBytes, gString& gsTrg )
   {
      int ti = ZERO ;                        // source index
      gsTrg.clear() ;                        // initialize the target buffer
      uint32_t msUnit, lsUnit,               // for converting unit pairs
               cx ;                          // decoded 16-bit character
      while ( rawBytes > ZERO )
      {
         //* Little-endian conversion *
         cx = (uint32_t(rawPtr[ti++]) & 0x000000FF) ;
         cx |= ((uint32_t(rawPtr[ti++]) << 8) & 0x0000FF00) ;
         rawBytes -= 2 ;

         //* If the character is fully represented by 16 bits *
         //* (most characters are)                            *
         if ( ((cx >= ZERO) && (cx < usxMSU16)) || 
              ((cx >= usxHIGH16) && (cx <= usxMAX16)) )
         {
            gsTrg.append( cx ) ;
            if ( cx == '\0' )    // if premature newline, we're done
               break ;
         }

         //* Character is represented by 32 bits    *
         //* (20 bits actually used), 16 MSBs first.*
         else
         {
            msUnit = cx ;
            lsUnit = ZERO ;
            //* Little-endian conversion *
            lsUnit = (UINT(rawPtr[ti++]) & 0x000000FF) ;
            lsUnit |= ((UINT(rawPtr[ti++]) << 8) & 0x0000FF00) ;
            rawBytes -= 2 ;

            //* Validate the range *
            if ( ((msUnit >= usxMSU16) && (msUnit <= usxMSU16e))
                 &&
                 ((lsUnit >= usxLSU16) && (lsUnit <= usxLSU16e)) )
            {
               cx = usxMIN20 + ((msUnit & usxMASK10) << 10) ;
               cx |= (lsUnit & usxMASK10) ;
            }
            else                 // invalid UTF-16 codepoint
               cx = L'?' ;
            gsTrg.append( cx ) ; // add the chararacter to output buffer
         }
      }
      return ( gsTrg.gschars() ) ;
   }  //* End utf16Decode() *

   //* Convert a wchar_t string to UTF-16LE (little-endian, no BOM).*
   //* Returns number of bytes encoded data.                        *
   short utf16Encode ( char* trg, const gString& src )
   {
      short trgCnt = 0 ;      // index into output array (return value)

      //* Establish the source array of wchar_t (wide) characters *
      short wcnt ;   // number of source characters (including the null terminator)
      const wchar_t *wstr = src.gstr( wcnt ) ;
      uint32_t cx ;  // source character

      //* If an empty string, (wstr[0] == '\0') return zero byte count *
      // Programmer's Note: Some playback software (VLC) gets confused if they 
      // see a null string, so an empty string is preferred.
      if ( wcnt == 1 )
         wcnt = 0 ;

      for ( short wIndex = ZERO ; wIndex < wcnt ; ++wIndex )
      {
         cx = wstr[wIndex] ;     // get a character from the input stream

         //* If the character can be encoded with a single, 16-bit value *
         if ( ((cx >= ZERO) && (cx < usxMSU16)) || 
              ((cx >= usxHIGH16) && (cx <= usxMAX16)) )
         {  //* Encode the bytes in little-endian order *
            trg[trgCnt++] = cx & 0x000000FF ;
            trg[trgCnt++] = (cx >> 8) & 0x000000FF ;
         }

         //* Else, character requires a pair of 16-bit values *
         else
         {
            uint32_t msUnit = ZERO, lsUnit = ZERO ;
            if ( (cx >= usxMIN20) && (cx <= usxMAX20) )
            {
               msUnit = ((cx - usxMIN20) >> 10) | usxMSU16 ;
               lsUnit = (cx & usxMASK10) | usxLSU16 ;
               trg[trgCnt++] = msUnit & 0x000000FF ;
               trg[trgCnt++] = (msUnit >> 8) & 0x000000FF ;
               trg[trgCnt++] = lsUnit & 0x000000FF ;
               trg[trgCnt++] = (lsUnit >> 8) & 0x000000FF ;
            }
            //* Character cannot be encoded as UTF-16. *
            //* Encode a question mark character.      *
            else
            {
               cx = L'?' ;
               trg[trgCnt++] = (cx >> 8) & 0x000000FF ;
               trg[trgCnt++] = cx & 0x000000FF ;
            }
         }
      }
      return trgCnt ;
   }  //* End utf16Encode() *

   //* Given an ASCII-hex GUID string, return the equivalent   *
   //* encoded GUID byte sequence.                             *
   //* Encoded sequences are defined in EncodedGUID[] array.   *
   //* Returns 'true' if recognized GUID string, else 'false'. *
   bool encodeGUID ( const gString& guid, uint8_t *ubuff )
   {
      short indx ;               // index into EncodedGUID[] array
      bool  encoded = true ;     // hope for the best
      if ( (guid.compare( CDO_GUID )) == ZERO )
         indx = ZERO ;
      else if ( (guid.compare( ECDO_GUID )) == ZERO )
         indx = 1 ;
      else if ( (guid.compare( CBO_GUID )) == ZERO )
         indx = 2 ;
      else if ( (guid.compare( MO_GUID )) == ZERO )
         indx = 3 ;
      else if ( (guid.compare( MLO_GUID )) == ZERO )
         indx = 4 ;
      else if ( (guid.compare( HEADER_GUID )) == ZERO )
         indx = 5 ;
      else
         encoded = false ;
      if ( encoded )
      {
         for ( short i = 0 ; i < asfGUID_BYTES ; ++i )
            ubuff[i] = EncodedGUID[indx][i] ;
      }
      return encoded ;
   }


   //* Public data members *
   uint8_t  guid[asfGUID_ASCII+1] ;    // CDO Global Unique Identifier (normalized)
   uint8_t  eguid[asfGUID_ASCII+1] ;   // Extended CDO GUID (if present)
   uint8_t  cboguid[asfGUID_ASCII+1] ; // Content Branding GUID (if present)
   uint8_t  moguid[asfGUID_ASCII+1] ;  // Metadata GUID (if present)
   uint8_t  mloguid[asfGUID_ASCII+1] ; // Metadata Library GUID (if present)
   uint8_t  xguid[asfGUID_ASCII+1] ;   // GUID for non-metadata objects
                                       // (used primarily for debugging)
   char     title[gsMAXBYTES] ;     // 'title' tag       UTF-16LE
   char     author[gsMAXBYTES] ;    // 'author' tag         "
   char     copyrt[gsMAXBYTES] ;    // 'copyright' tag      "
   char     desc[gsMAXBYTES] ;      // 'description' tag    "
   char     rating[gsMAXBYTES] ;    // 'rating' tag         "
   uint64_t cdoSize ;               // Size of CDO source object (bytes)
   uint64_t ecdoSize ;              // Size of Extended CDO source object (bytes)
   uint64_t cboSize ;               // Size of CBO source object (bytes)
   uint64_t moSize ;                // Size of MO source object (bytes)
   uint64_t mloSize ;               // Size of MLO source object (bytes)
   uint16_t descCount ;             // Number of ECDO descriptors
   uint16_t titleSize ;             // length of 'title' tag (bytes, UTF-16 string)
   uint16_t authorSize ;            // length of 'author' tag (bytes, UTF-16 string)
   uint16_t copyrtSize ;            // length of 'copyrt' tag (bytes, UTF-16 string)
   uint16_t descSize ;              // length of 'description' tag (bytes, UTF-16 string)
   uint16_t ratingSize ;            // length of 'rating' tag (bytes, UTF-16 string)
   bool     cdoVerified ;           // 'true' if CDO GUID and Size initialized
   bool     ecdoVerified ;          // 'true' if ECDO GUID and Size initialized
   bool     cboVerified ;           // 'true' if CBO GUID and Size initialized
   bool     moVerified ;            // 'true' if MO GUID and Size initialized
   bool     mloVerified ;           // 'true' if MLO GUID and Size initialized
   // Array of Descriptor Name / Descriptor Value records
   asfExDesc exDesc[asfECD_MAX_COUNT] ;

} ;   // asfConDesc

