//******************************************************************************
//* File       : SrcProf_File.cpp                                              *
//* Author     : Mahlon R. Smith                                               *
//*              Copyright (c) 1998-2015 Mahlon R. Smith, The Software Samurai *
//*                  GNU GPL copyright notice located in SrcProf.hpp           *
//* Date       : 22-Oct-2015                                                   *
//* Version    : (see AppVersion string in SrcProf.hpp)                        *
//*                                                                            *
//* Description: This module contains methods for accessing directory files    *
//* and extracting information about each entry.                               *
//*                                                                            *
//* These methods are adapted from a small subset of the FMgr class written    *
//* for the FileMangler file-management project by the same author.            *
//*                                                                            *
//*                                                                            *
//******************************************************************************
//* Copyright Notice:                                                          *
//* This program is free software: you can redistribute it and/or modify it    *
//* under the terms of the GNU General Public License as published by the Free *
//* Software Foundation, either version 3 of the License, or (at your option)  *
//* any later version.                                                         *
//*                                                                            *
//* This program is distributed in the hope that it will be useful, but        *
//* WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY *
//* or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License   *
//* for more details.                                                          *
//*                                                                            *
//* You should have received a copy of the GNU General Public License along    *
//* with this program.  If not, see <http://www.gnu.org/licenses/>.            *
//*                                                                            *
//*         Full text of the GPL License may be found in the Texinfo           *
//*         documentation for this program under 'Copyright Notice'.           *
//******************************************************************************
//* Version History (see SrcProf.cpp)                                          *
//*                                                                            *
//******************************************************************************

//****************
//* Header Files *
//****************
#include "SrcProf.hpp"

//****************
//* Local Data   *
//****************


//*************************
//*   spfReadDirectory    *
//*************************
//******************************************************************************
//* Read the contents of the specified directory and create a list of files    *
//* found. Note that we report only sub-directory names (fmDIR_TYPE) and       *
//* regular (fmREG_TYPE) files which are recognized as source code files.      *
//* - 'regular' files are source code files                                    *
//* - 'sub-directories' may contain source code files                          *
//*   - Note: The '..' parent directory is retained, but the '.' i.e.          *
//*           this-directory is not.                                           *
//*                                                                            *
//* Input  : dPath : full path specification of directory to scan              *
//*          inclDir: (optional, false by default)                             *
//*                  - if 'true' include subdirectory names and parent ('..")  *
//*                    in the list                                             *
//*                  -if 'false' include only eligible source code files       *
//*                                                                            *
//* Returns: Number of files in list                                           *
//*           this->tnfPtr references an array of file-description objects     *
//*           this->tnfCount == number of elements in tnfPtr array             *
//*          no files found or if error reading the directory, returns ZERO    *
//******************************************************************************

UINT SrcProf::spfReadDirectory ( const gString& dPath, bool inclDir )
{
   DIR*      dirPtr ;            // pointer to open directory file
   deStats   destat, *sptr ;     // entry read from directory file
   gString   fnPath ;            // target file's path/filename
   FileStats rawStats ;          // target file's stat info
   fmFType   fileType ;          // target file's file type
   UINT      rawfCount = ZERO,   // total entries in directory
             fCount = ZERO ;     // directory entries returned to caller

   //**************************************************
   //* Count the elegible filenames in the directory. *
   //**************************************************
   if ( ((this->spfTargetExists ( dPath, true )) != false) &&
        ((dirPtr = opendir ( dPath.ustr() )) != NULL) )
   {
      while ( (readdir64_r ( dirPtr, &destat, &sptr )) == ZERO )
      {
         // NOTE: This compensates for a bug in the return value of readdir64_r.
         //       If sptr == NULL, it means end-of-list.
         if ( sptr == NULL )
            break ;

         //* DO NOT include 'current dir' name but DO include 'parent' ('..')  *
         //* and 'hidden' files, i.e. filenames whose first character is       *
         //* a '.' (PERIOD).                                                   *
         bool  showFile = true ; // true if file to be included in list
         if ( destat.d_name[ZERO] == PERIOD )
         {
            if ( destat.d_name[1] == NULLCHAR )
               showFile = false ;
         }
         //* If this is a file we may want to store in the list *
         if ( showFile != false )
         {
            //* 'lstat' the file *
            this->spfCatPathFilename ( fnPath, dPath, destat.d_name ) ;
            if ( (lstat64 ( fnPath.ustr(), &rawStats )) == OK )
            {
               //* Find out what kind of file it is.                       *
               //* We are interested only in directories and regular files.*
               if ( inclDir && (S_ISDIR(rawStats.st_mode)) != false )
                  fileType = fmDIR_TYPE ;
               else if ( (S_ISREG(rawStats.st_mode)) != false )
                  fileType = fmREG_TYPE ;
               else
                  fileType = fmUNKNOWN_TYPE ;

               //* Include the file to our list *
               if ( fileType == fmDIR_TYPE || fileType == fmREG_TYPE )
                  ++rawfCount ;
            }
            else { /* Can't access the file, so ignore it */ }
         }
      }  // while()
      closedir ( dirPtr ) ;               // close the directory
   }     // opendir()

   //**************************************************
   //* Allocate data space and collect elegible files.*
   //**************************************************
   if ( rawfCount > ZERO )
   {
      //* Allocate the data objects (we may have more than we need). *
      //* Dynamic allocation for array is referenced by this->tnfPtr *
      //* and this->tnfCount.                                        *
      this->tnfAllocate ( rawfCount ) ;

      if ( (dirPtr = opendir ( dPath.ustr() )) != NULL )
      {
         while ( (readdir64_r ( dirPtr, &destat, &sptr )) == ZERO )
         {
            // NOTE: This compensates for a bug in the return value of readdir64_r.
            //       If sptr == NULL, it means end-of-list.
            if ( sptr == NULL )
               break ;
   
            //* DO NOT include 'current dir' name but DO include 'parent'('..')*
            //* and 'hidden' files, i.e. filenames whose first character is    *
            //* a '.' (PERIOD).                                                *
            bool  showFile = true ; // 'true' if file to be included in list
            if ( destat.d_name[ZERO] == PERIOD )
            {
               if ( destat.d_name[1] == NULLCHAR )
                  showFile = false ;
            }
            //* If this is a file we may want to store in the list *
            if ( showFile != false )
            {
               //* 'lstat' the file *
               this->spfCatPathFilename ( fnPath, dPath, destat.d_name ) ;
               if ( (lstat64 ( fnPath.ustr(), &rawStats )) == OK )
               {
                  //* Find out what kind of file it is.                       *
                  //* We are interested only in directories and regular files.*
                  if ( inclDir && ((S_ISDIR(rawStats.st_mode)) != false) )
                     fileType = fmDIR_TYPE ;
                  else if ( (S_ISREG(rawStats.st_mode)) != false )
                  {  //* Determine whether 'regular' file is recognized *
                     //* as a source code file.                         *
                     if ( (this->GetFileFamily ( fnPath.ustr() )) != sffUnk )
                        fileType = fmREG_TYPE ;
                     else
                        showFile = false ;
                  }
                  else
                  {
                     fileType = fmUNKNOWN_TYPE ;
                     showFile = false ;
                  }
   
                  //* Add the file to our list *
                  if ( showFile )
                  {
                     this->tnfPtr[fCount].rawStats = rawStats ;
                     strncpy ( this->tnfPtr[fCount].fName, destat.d_name, MAX_FNAME ) ;
                     this->spfFormatEpochTime ( rawStats.st_mtime, 
                                                tnfPtr[fCount].modTime ) ;
                     this->tnfPtr[fCount].fType    = fileType ;
                     this->tnfPtr[fCount].fBytes   = rawStats.st_size ;
                     this->tnfPtr[fCount].readAcc  = true ;
                     //Write Access unknown at this point, set to false in constructor
                     ++fCount ;
                  }
               }
               else { /* Can't access the file, so ignore it */ }
            }
         }  // while()
         closedir ( dirPtr ) ;               // close the directory
      }     // opendir()

     //* Adjust tnfCount to indicate the number of valid files in the array *
     if ( fCount > ZERO && fCount < this->tnfCount )
        this->tnfCount = fCount ;
     else if ( fCount == ZERO )  // if no valid file, release the temp storage
        this->tnfRelease () ;
   }     // if(rawfCount>ZERO)

   return fCount ;

}  //* End spfReadDirectory() *

//***************************
//* spfSortDirEntriesByName *
//***************************
//******************************************************************************
//* Sort this->tnfPtr array:                                                   *
//*  - '..' [parent directory] comes first (if any)                            *
//*  - directory names next, sub-sorted by name                                *
//*  - source code filenames, sub-sorted by name                               *
//* Note that we use a case-insensitive bubble sort (we're in no hurry).       *
//*                                                                            *
//* Input  : tPtr : pointer to array of file-description objects               *
//*          tCnt : number of elements in the array                            *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void SrcProf::spfSortDirEntriesByName ( tnFName* tPtr, UINT tCnt )
{
   if ( tCnt > 1 )                  // if there is something to do
   {
      gString nameA, nameB ;        // compare names
      tnFName tmp ;                 // swap buffer
      UINT    indexA = ZERO,        // upper index
              indexB = 1,           // lower index
              moves ;               // sort operations performed each loop
      bool    parentFound = false ; // parent directory entry found

      //* Bring '..' (parent directory) if any to the top *
      while ( indexB < tCnt )
      {
         nameB = tPtr[indexB].fName ;
         if ( (wcsncasecmp ( nameB.gstr(), L"..", MAX_FNAME )) == ZERO )
         {
            tmp = tPtr[indexA] ;
            tPtr[indexA] = tPtr[indexB] ;
            tPtr[indexB] = tmp ;
            ++indexA ;
            parentFound = true ;
            break ;
         }
         ++indexB ;
      }

      //* Sort by file type *
      do
      {
         moves = ZERO ;
         while ( (indexA < (tCnt - 1)) && 
                  tPtr[indexA].fType == fmDIR_TYPE )
            ++indexA ;
         if ( indexA < (tCnt - 1) )
         {
            for ( indexB = indexA + 1 ; indexB < tCnt ; indexB++ )
            {
               if ( tPtr[indexB].fType == fmDIR_TYPE && 
                    tPtr[indexA].fType != fmDIR_TYPE )
               {
                  tmp = tPtr[indexA] ;
                  tPtr[indexA] = tPtr[indexB] ;
                  tPtr[indexB] = tmp ;
                  ++indexA ;
                  ++moves ;
               }
            }
         }
      }
      while ( (moves > ZERO) && (indexA < (tCnt - 1)) ) ;

      //* If there are two or more sub-directory names, sort them by name.*
      if ( (parentFound && indexA > 2) || (!parentFound && indexA > 1) )
      {
         int iMin = parentFound ? 1 : ZERO,
             iMax = indexA - 1 ;
         while ( iMin <= (iMax - 1) )
         {
            for ( int i = iMin + 1 ; i <= iMax ; i++ )
            {
               nameA = tPtr[iMin].fName ;
               nameB = tPtr[i].fName ;
               if ( (wcsncasecmp ( nameB.gstr(), nameA.gstr(), gsMAXCHARS )) < ZERO )
               {
                  tmp = tPtr[iMin] ;
                  tPtr[iMin] = tPtr[i] ;
                  tPtr[i] = tmp ;
               }            
            }
            ++iMin ;
         }
      }

      //* Sort non-directory files by name *
      while ( indexA < (tCnt - 1) )
      {
         for ( indexB = indexA + 1 ; indexB < tCnt ; indexB++ )
         {
            nameA = tPtr[indexA].fName ;
            nameB = tPtr[indexB].fName ;
            if ( (wcsncasecmp ( nameB.gstr(), nameA.gstr(), gsMAXCHARS )) < ZERO )
            {
               tmp = tPtr[indexA] ;
               tPtr[indexA] = tPtr[indexB] ;
               tPtr[indexB] = tmp ;
            }            
         }
         ++indexA ;
      }
   }

}  //* End spfSortDirEntriesByName() *

//********************************
//* spfSortDirEntriesByExtension *
//********************************
//******************************************************************************
//* Sort the this->tnfPtr array by filename extension.                         *
//*  - sort source code filenames by extension, then sub-sort by filename      *
//* Note that we use a case-insensitive bubble sort (we're in no hurry).       *
//*                                                                            *
//* Input  : fCount ; number of elements in the array                          *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void SrcProf::spfSortDirEntriesByExtension ( UINT fCount )
{
   if ( fCount > 1 )                // if there is something to do
   {
      gString nameA, nameB ;        // compare names
      tnFName tmp ;                 // swap buffer
      UINT indexA = ZERO,           // upper index
           indexB = 1 ;             // lower index

      //* Sort by filename extension *
      while ( indexA < (fCount - 1) )
      {
         for ( indexB = indexA + 1 ; indexB < fCount ; indexB++ )
         {
            this->spfExtractFileExtension ( nameA, this->tnfPtr[indexA].fName ) ;
            this->spfExtractFileExtension ( nameB, this->tnfPtr[indexB].fName ) ;
            if ( (wcsncasecmp ( nameB.gstr(), nameA.gstr(), gsMAXCHARS )) < ZERO )
            {
               tmp = this->tnfPtr[indexA] ;
               this->tnfPtr[indexA] = this->tnfPtr[indexB] ;
               this->tnfPtr[indexB] = tmp ;
            }            
         }
         ++indexA ;
      }

      //* Sub-sort by filename *
      indexA = indexB = ZERO ;
      int   diff, comp ;
      while ( indexA < (fCount - 1) )
      {  //* Delimit a block of filenames with the same extension *
         this->spfExtractFileExtension ( nameA, this->tnfPtr[indexA].fName ) ;
         do
         {
            if ( ++indexB >= fCount )
               break ;
            this->spfExtractFileExtension ( nameB, this->tnfPtr[indexB].fName ) ;
            comp = wcsncasecmp ( nameB.gstr(), nameA.gstr(), gsMAXCHARS ) ;
         }
         while ( comp == ZERO ) ;

         //* Within the block with same extension, sort by filename *
         diff = indexB - indexA ;
         if ( diff > 1 )
         {
            this->spfSortDirEntriesByName ( &this->tnfPtr[indexA], diff ) ;
         }
         indexA = indexB ;    // reference first entry in next block
      }
   }

}  //* End spfSortDirEntriesByExtension() *

//*************************
//*      spfGetCWD        *
//*************************
//******************************************************************************
//* Returns the path of user's current working directory.                      *
//*                                                                            *
//* Input  : dPath:(by reference, initial value ignored)                       *
//*                receives path string                                        *
//*                                                                            *
//* Returns: OK if successful, ERR if path too long to fit the buffer          *
//******************************************************************************

short SrcProf::spfGetCWD ( gString& dPath )
{
char     buff[MAX_PATH] ;
short    status = OK ;

   if ( (getcwd ( buff, MAX_PATH )) != NULL )
      dPath = buff ;
   else
      status = ERR ;    // (this is very unlikely)

   return status ;

}  //* End spfGetCWD() *

//*************************
//*   spfCreateTemppath   *
//*************************
//******************************************************************************
//* Create a unique path/filename prefix for creating the application's        *
//* temporary files. Resulting string is stored in the 'tmpDir' data member.   *
//*                                                                            *
//* This method calls the 'tmpnam_r' function to create a unique name, then    *
//* we extract the path and call 'mkdtemp' to create a unique directory name.  *
//*                                                                            *
//* Finally, we create the path/filename for the file which will contain the   *
//* list of files to be processed and store it in the 'lstFile' data member.   *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: 'true'  if path/filename prefix created successfully              *
//*          'false' if library call failed                                    *
//******************************************************************************

bool SrcProf::spfCreateTemppath ( void )
{
   //* Name of temporary file containing names of files to process *
   const wchar_t* const listFile = L"SRCPROF_list" ;

   bool  status = false ;

   char tn[gsMAXBYTES] ;
   if ( (tmpnam_r ( tn )) != NULL )
   {
      gString tmpPath( tn ), basePath ;
      this->spfExtractPathname ( basePath, tmpPath ) ;
      basePath.append( "/SRCPROF_XXXXXX" ) ;
      basePath.copy( tn, gsMAXBYTES ) ;
      if ( (mkdtemp ( tn )) != NULL )
      {
         this->tmpDir = tn ;
         this->spfCatPathFilename ( this->lstFile, this->tmpDir, listFile ) ;
         status = true ;
      }
   }
   return status ;

}  //* End spfCreateTemppath() *

//*************************
//*   spfCreateTempname   *
//*************************
//******************************************************************************
//* Create a unique path/filename for a temporary file.                        *
//*                                                                            *
//* Input  : tmpPath: (by reference) receives the path/filename                *
//*                                                                            *
//* Returns: 'true'  if path/filename created successfully                     *
//*          'false' if library call failed                                    *
//******************************************************************************

bool SrcProf::spfCreateTempname ( gString& tmpPath )
{
   bool  status = false ;

   char tn[gsMAXBYTES] ;
   if ( (tmpnam_r ( tn )) != NULL )
   {
      gString gs = tn,
              tmpName ;
      this->spfExtractFilename ( tmpName, gs ) ;
      this->spfCatPathFilename ( tmpPath, this->tmpDir, tmpName.gstr() ) ;
      status = true ;
   }
   return status ;

}  //* End spfCreateTempname() *

//*************************
//*     spfCdParent       *
//*************************
//******************************************************************************
//* Change the current working directory to parent of current CWD.             *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: OK if successful, else ERR                                        *
//******************************************************************************

short SrcProf::spfCdParent ( void )
{
   gString parentPath ;             // path of new CWD
   short pathChars ;                // number of characters in path string
   short status = ERR ;             // return value

   //* Get CWD path *
   this->spfGetCWD ( parentPath ) ;
   //* Truncate CWD path to create path to parent directory *
   const wchar_t* pathPtr = parentPath.gstr( pathChars ) ;
   do
   { --pathChars ; }
   while ( pathChars > ZERO && pathPtr[pathChars] != SLASH ) ;
 
   //* If not already at root directory *
   if ( pathChars > ZERO )
   {
      parentPath.limitCharacters ( pathChars ) ;
      status = chdir ( parentPath.ustr() ) ;
   }
   return status ;

}  //* End spfCdParent() *

//*************************
//*      spfCdChild       *
//*************************
//******************************************************************************
//* Change the current working directory to specified subdirectory of CWD.     *
//*                                                                            *
//* Input  : dirName: name of target directory                                 *
//*                                                                            *
//* Returns: OK if successful, else ERR                                        *
//******************************************************************************

short SrcProf::spfCdChild ( const char* dirName )
{
   gString cwdPath,                 // old CWD
           childPath ;              // path of new CWD

   this->spfGetCWD ( cwdPath ) ; 
   this->spfCatPathFilename ( childPath, cwdPath, dirName ) ;
   return ( chdir ( childPath.ustr() ) ) ;

}  //* End spfCdChild() *

//*************************
//*    CatPathFilename    *
//*************************
//******************************************************************************
//* Concatenate a path string with a filename string to create a path/filename *
//* specification.                                                             *
//*                                                                            *
//* Input  : pgs  : gString object (by reference) to hold the path/filename    *
//*                 On return, pgs contains the path/filename string in both   *
//*                 UTF-8 and wchar_t formats.                                 *
//*          wPath: gString object (by reference) path string                  *
//*          uFile: pointer to UTF-8 filename string                           *
//*               OR                                                           *
//*          wFile: pointer to wchar_t filename string                         *
//*                                                                            *
//* Returns: OK if success                                                     *
//*          ERR if string truncated                                           *
//******************************************************************************

short SrcProf::spfCatPathFilename ( gString& pgs, const gString& wPath, const char* uFile )
{
short    success = OK ;

   pgs.compose( L"%S/%s", wPath.gstr(), uFile ) ;
   if ( pgs.gschars() >= (gsMAXCHARS-1) )
      success = ERR ;   // (this is very unlikely)
   return success ;

}  //* End CatPathFilename() *

short SrcProf::spfCatPathFilename ( gString& pgs, const gString& wPath, const wchar_t* wFile )
{
short    success = OK ;

   pgs.compose( L"%S/%S", wPath.gstr(), wFile ) ;
   if ( pgs.gschars() >= (gsMAXCHARS-1) )
      success = ERR ;   // (this is very unlikely)
   return success ;

}  //* End CatPathFilename() *

//*************************
//*  spfExtractPathname   *
//*************************
//******************************************************************************
//* Extract the path from the path/filename provided.                          *
//* (i.e. truncate the string to remove the filename)                          *
//*                                                                            *
//* Input  : eName : (by reference) receives the extracted path                *
//*          fPath : source path/filename string                               *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void SrcProf::spfExtractPathname ( gString& ePath, const gString& fPath )
{
   ePath = fPath ;
   short slashIndex = ZERO, i ;
   while ( (i = ePath.find( L'/', slashIndex )) >= ZERO )
      slashIndex = i + 1 ;

   ePath.limitChars( slashIndex > ZERO ? --slashIndex : ZERO ) ;

}  //* End of spfExtractPathname() *

//*************************
//*  spfExtractFilename   *
//*************************
//******************************************************************************
//* Extract the filename from the path/filename provided.                      *
//*                                                                            *
//* Input  : eName : (by reference) receives the extracted filename            *
//*          fPath : source path/filename string                               *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void SrcProf::spfExtractFilename ( gString& eName, const gString& fPath )
{
   short slashIndex = ZERO, i ;
   while ( (i = fPath.find( L'/', slashIndex )) >= ZERO )
      slashIndex = i + 1 ;

   eName = &fPath.gstr()[slashIndex] ;

}  //* End spfExtractFilename() *

void SrcProf::spfExtractFilename ( gString& gsName, const char* fPath )
{

   gString gs( fPath ) ;
   this->spfExtractFilename ( gsName, gs ) ;

}  //* End ExtractFilename() *

//***************************
//* spfExtractFileExtension *
//***************************
//******************************************************************************
//* Extract the filename extension (including the '.') from the path/filename  *
//* provided.                                                                  *
//*                                                                            *
//* Input  : eExt  : (by reference, initial contents ignored)                  *
//*                  receives the extracted filename extension                 *
//*          fPath : source filename or path/filename string                   *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void SrcProf::spfExtractFileExtension ( gString& eExt, const gString& fPath )
{
   eExt.clear() ;                   // initialize the target buffer

   short pIndex ;
   const wchar_t* wptr = fPath.gstr( pIndex ) ;
   --pIndex ;                       // reference the NULLCHAR
   while ( wptr[pIndex] != PERIOD && wptr[pIndex] != SLASH && pIndex > ZERO )
      --pIndex ;

   //* If we found a period character, we have a filename extension.*
   if ( wptr[pIndex] == PERIOD )
      eExt = &wptr[pIndex] ;
   else
      eExt = L"" ;

}  //* End spfExtractFileExtension() *

//*************************
//* ExtractFileExtension  *
//*************************
//******************************************************************************
//* Extract the filename extension (including the '.') from the path/filename  *
//* provided.                                                                  *
//*                                                                            *
//* Input  : gsName: (by reference, initial contents ignored)                  *
//*                  receives the extracted filename                           *
//*          fName : filename or path/filename string                          *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void SrcProf::spfExtractFileExtension ( gString& gsExt, const char* fName )
{

   gString gs( fName ) ;
   this->spfExtractFileExtension ( gsExt, gs ) ;

}  //* End ExtractFileExtension() *

//*************************
//*     spfRealpath       *
//*************************
//******************************************************************************
//* Decode the relative or aliased path for the specified target.              *
//*                                                                            *
//*                                                                            *
//* Input  : rawPath  : caller's path specification                            *
//*          realPath : (by reference) receives decoded path specification     *
//*                                                                            *
//* Returns: 'true'  if conversion successful                                  *
//*          'false' if invalid path or if system call fails                   *
//******************************************************************************
//* Programmer's Notes:                                                        *
//* a) The 'realpath' C library function cannot handle a symbolic substitution *
//*    such as: ${HOME}/Docs/filename.txt                                      *
//*    It returns: NULL, and if the optional buffer is specified, then it      *
//*    receives something like: /home/Sam/Docs/(CWD)/${HOME}                   *
//*    This is a little bit nuts, but (sort-of) matches 'realpath'             *
//*    documentation. For this reason, we use the 'wordexp' library function   *
//*    to expand any symbolic substitutions.                                   *
//*                                                                            *
//* b) Note that the 'realpath' C library function always follows symbolic     *
//*    links, which we DO NOT WANT in this application. In order to get the    *
//*    full path to a symlink file, we use the coreutils 'realpath' utility    *
//*    which can be told NOT to follow simlinks.                               *
//*                                                                            *
//* c) Note that just because 'realpath' can create a full path/filename spec, *
//*    DOES NOT mean that the file actually exists.                            *
//******************************************************************************

bool SrcProf::spfRealpath ( const char* rawPath, gString& realPath )
{
   bool status = false ;

   //* Expand environment variables *
   gString rp ;
   wordexp_t wexp ;              // target structure
   if ( (wordexp ( rawPath, &wexp, ZERO )) == ZERO )
   {
      if ( wexp.we_wordc > ZERO )   // if we have at least one element
      {
         for ( UINT i = ZERO ; i < wexp.we_wordc ; )
         {
            rp.append( wexp.we_wordv[i++] ) ;
            if ( i < wexp.we_wordc )
               rp.append( L' ' ) ;
         }
      }
      wordfree ( &wexp ) ;

      //* Create a new tempfile name, and direct the coreutils *
      //* 'realpath' output to it. Then read the file.         *
      gString stdoutCapture ;
      if ( (this->spfCreateTempname ( stdoutCapture )) != false )
      {
         char cmdBuff[MAX_PATH*3] ;
         snprintf ( cmdBuff, (MAX_PATH*3), "realpath -sm %s 1>%s 2>/dev/null", 
                    rp.ustr(), stdoutCapture.ustr() );
         if ( (system ( cmdBuff )) == OK )
         {
            //* Read the only line in the file *
            ifstream ifsIn ( stdoutCapture.ustr(), ifstream::in ) ;
            ifsIn.getline ( cmdBuff, (MAX_PATH*3) ) ;
            ifsIn.close () ;     // close the temp file
            //* If we have something other than an empty string *
            rp = cmdBuff ;
            if ( rp.gschars() > 1 )
            {
               realPath = rp ;
               status = true ;
            }
         }

         //* Delete the tempfile if it exists *
         this->spfDeleteFile ( stdoutCapture ) ;
      }
   }
   return status ;

}  //* End spfRealpath() *

//*************************
//*    spfTargetExists    *
//*************************
//******************************************************************************
//* 'stat' the specified path/filename to see if it exists.                    *
//* Target must be a 'regular' file or optionally, a 'directory' file.         *
//* We also test whether user has read access.                                 *
//*                                                                            *
//* Input  : fPath: target path/filename                                       *
//*          isDir: (optional, false by default)                               *
//*                 if 'false', test whether file type is 'regular'            *
//*                 if 'true',  test whether file type is 'directory'          *
//*                                                                            *
//* Returns: 'true' if file exists AND is a Regular file (or optionally, a     *
//*                 'directory' file), AND user has read access                *
//*          'false' otherwise                                                 *
//******************************************************************************

bool SrcProf::spfTargetExists ( const gString& fPath, bool isDir )
{
   bool exists = false ;
   FileStats rawStats ;

   //* If the file exists *
   if ( (lstat64 ( fPath.ustr(), &rawStats )) == OK )
   {  //* Read-access test *
      if ( (access ( fPath.ustr(), R_OK )) == ZERO )
      {
         //* If testing for the existence of a directory *
         if ( isDir != false )
         {
            if ( (S_ISDIR(rawStats.st_mode)) != false )
               exists = true ;
         }
         //* We operate only on 'regular' files *
         else
         {
            if ( (S_ISREG(rawStats.st_mode)) != false )
               exists = true ;
         }
      }
   }
   return exists ;

}  //* End spfTargetExists() *

//*************************
//*    FormatEpochTime    *
//*************************
//******************************************************************************
//* Convert GNU/UNIX epoch time code (seconds since the epoch: Jan. 01, 1970)  *
//* to local time in a structured format (class localTime).                    *
//*                                                                            *
//* See the info page for the 'stat' command for more information.             *
//*                                                                            *
//* Input  : eTime  : epoch time (time_t variable == long int                  *
//*                   Note: GNU defines a time_t type for epoch time as        *
//*                         time_t == long int, but we use ULONG on            *
//*                         the assumption that we will still be using         *
//*                         computers after 18 Jan. 2038.                      *
//*          ftTime : (by reference, initial values ignored)                   *
//*                                                                            *
//* Returns: nothing, but all members of ftTime will be initialized            *
//******************************************************************************
//* Conversion from time_t code to local time is returned in this structure.   *
//*                                                                            *
//*  struct tm                                                                 *
//*  {                                                                         *
//*     int  tm_sec ;          // 0-59  (leap seconds to 61)                   *
//*     int  tm_min ;          // 0-59                                         *
//*     int  tm_hour ;         // 0-23                                         *
//*     int  tm_mday ;         // 1-31                                         *
//*     int  tm_mon ;          // 0-11                                         *
//*     int  tm_year ;         // since 1900                                   *
//*     int  tm_wday ;         // 0-6                                          *
//*     int  tm_yday ;         // 0-365                                        *
//*     int  tm_isdst ;        // >0 == is DST, 0 == not DST, <0 == unknown    *
//*  } ;                                                                       *
//*                                                                            *
//******************************************************************************

void SrcProf::spfFormatEpochTime ( ULONG eTime, localTime& ftTime )
{
Tm    tm ;

   //* Convert epoch-encoded time to local time *
   time_t   teaTime = (time_t)eTime ;  // see note
   localtime_r ( &teaTime, &tm ) ;

   //* Copy it into a convenient format *
   ftTime.day     = tm.tm_wday + 1 ;
   ftTime.date    =  tm.tm_mday ;
   ftTime.month   =  tm.tm_mon + 1 ;
   ftTime.year    = 1900 + tm.tm_year ;
   ftTime.hours   = tm.tm_hour ;
   ftTime.minutes = tm.tm_min ;
   ftTime.seconds = tm.tm_sec ; 

}  //* End FormatEpochTime() *

//*************************
//*    spfDeleteFile      *
//*************************
//******************************************************************************
//* Delete the specified file.                                                 *
//*                                                                            *
//* Input  : trgPath : full path/filename specification of target              *
//*                                                                            *
//* Returns: 'true' if successful, else 'false'                                *
//******************************************************************************

bool SrcProf::spfDeleteFile ( const gString& trgPath )
{

   return ( bool((unlink ( trgPath.ustr() )) == ZERO) ) ;

}  //* End spfDeleteFile() *

//*************************
//*    spfRenameFile      *
//*************************
//******************************************************************************
//* Rename the specified file.                                                 *
//*                                                                            *
//* Input  : srcPath : full path/filename specification of source              *
//*          trgPath : full path/filename specification of target              *
//*                                                                            *
//* Returns: OK if successful, else ERR                                        *
//******************************************************************************
//* For the rename() primitive: If srcPath refers to a symbolic link the       *
//* link is renamed; if trgPath refers to a symbolic link the link will be     *
//* overwritten.                                                               *
//******************************************************************************

short SrcProf::spfRenameFile ( const gString& srcPath, const gString& trgPath )
{

   return ( rename ( srcPath.ustr(), trgPath.ustr() ) ) ;

}  //* End spfRenameFile() *

//*************************
//*      spfReadLine      *
//*************************
//******************************************************************************
//* Read one line of text from the specified input stream.                     *
//* This is a generic ReadLine.                                                *
//*                                                                            *
//* Input  : ifs    : open input stream (by reference)                         *
//*          gs     : (by reference) receives text data                        *
//*                                                                            *
//* Returns: 'true'  if line read successfully, else 'false'                   *
//******************************************************************************

bool SrcProf::spfReadLine ( ifstream& ifs, gString& gs )
{
   char  tmp[gsMAXBYTES] ; // UTF-8 line input from file
   bool  status = false ;  // return value

   ifs.getline ( tmp, gsMAXBYTES, NEWLINE ) ;   // read a source line
   if ( ifs.good() || ifs.gcount() > ZERO )     // if a good read
   {
      gs = tmp ;                                // convert to wide data
      status = true ;                           // return with good news
   }
   return status ;

}  //* End spfReadLine() *

//*************************
//*  spfDeleteTempFiles   *
//*************************
//******************************************************************************
//* Delete the application's temporary files.                                  *
//*                                                                            *
//* Input  : all  : if 'false' do not delete the source code file list         *
//*                 if 'true'  delete all temporary files                      *
//*                            (only the SrcProf destructor calls with 'true') *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void SrcProf::spfDeleteTempFiles ( bool all )
{
   //* Delete the current source-file list *
   if ( this->spfTargetExists ( this->lstFile ) )
      this->spfDeleteFile ( this->lstFile ) ;

   if ( all )
   {
      if ( this->spfTargetExists ( this->tmpDir, true ) )
      {  //* Delete all files in our temporary directory, *
         //* then delete the directory itself.            *
         gString tmpFile ;       // temp file path/filename
         bool  procFile ;        // true if file to be deleted
         DIR*  dirPtr ;          // pointer to open directory file
         deStats   destat, *sptr ; // entry read from directory file
         if ( (dirPtr = opendir ( this->tmpDir.ustr() )) != NULL )
         {
            while ( (readdir64_r ( dirPtr, &destat, &sptr )) == ZERO )
            {
               if ( sptr == NULL ) break ;   // (compensate for readdir_64_r bug)
               procFile = true ;
               if ( destat.d_name[0] == PERIOD && 
                    (destat.d_name[1] == PERIOD || destat.d_name[1] == NULLCHAR) )
                  procFile = false ;
               if ( procFile )
               {
                  this->spfCatPathFilename ( tmpFile, this->tmpDir, destat.d_name ) ;
                  this->spfDeleteFile ( tmpFile ) ;
               }
            }  // while()
            closedir ( dirPtr ) ;               // close the directory
         }     // opendir()
         rmdir ( this->tmpDir.ustr() ) ;
      }
   }

}  //* End spfDeleteTempFiles() *

//*************************
//*                       *
//*************************
//******************************************************************************
//*                                                                            *
//*                                                                            *
//*                                                                            *
//* Input  :                                                                   *
//*                                                                            *
//* Returns:                                                                   *
//******************************************************************************


