//******************************************************************************
//* File       : SrcProf_Analyze.cpp                                           *
//* Author     : Mahlon R. Smith                                               *
//*              Copyright (c) 1998-2015 Mahlon R. Smith, The Software Samurai *
//*                  GNU GPL copyright notice located in SrcProf.hpp           *
//* Date       : 29-Oct-2015                                                   *
//* Version    : (see AppVersion string in SrcProf.hpp)                        *
//*                                                                            *
//* Description: This module contains the source code analysis methods.        *
//*                                                                            *
//*                                                                            *
//******************************************************************************
//* Copyright Notice:                                                          *
//* This program is free software: you can redistribute it and/or modify it    *
//* under the terms of the GNU General Public License as published by the Free *
//* Software Foundation, either version 3 of the License, or (at your option)  *
//* any later version.                                                         *
//*                                                                            *
//* This program is distributed in the hope that it will be useful, but        *
//* WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY *
//* or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License   *
//* for more details.                                                          *
//*                                                                            *
//* You should have received a copy of the GNU General Public License along    *
//* with this program.  If not, see <http://www.gnu.org/licenses/>.            *
//*                                                                            *
//*         Full text of the GPL License may be found in the Texinfo           *
//*         documentation for this program under 'Copyright Notice'.           *
//******************************************************************************
//* SOME NOTES ON HOW WE DO THE SCAN:                                          *
//* ---------------------------------                                          *
//* A minor annoyance with source files created under WinDoze or other systems *
//* whose text file lines end with a CR/LF combination:                                                               *
//* 1. We may report one too many lines in the file because ifstream::getline()*
//*    may read an invisible last line in such files and reports ifs.good().   *
//*    Trying to compensate for this might cause more problems than it solves, *
//*    so we leave it alone.                                                   *
//*                                                                            *
//* 2. In addition: If there is no newline character ('\n') on the last line   *
//*    of a file whose lines end with CR/LF, ifs.getline() will report failure *
//*    (ifs.good() == false) even though whatever was on the line was actually *
//*    read successfully. To compensate for this, we will process source-file  *
//*    lines if:  ifs.good() != false || ifs.gcount > ZERO  || !ifs.eof        *
//*    so the last line of the file will not be discarded. This probably has   *
//*    more to do with the text editor that created the file than with the     *
//*    CR/LF format, but it's handled, so no harm done.                        *
//*                                                                            *
//* 3. Note that ifstream: getline() strips the newline ('\n') character from  *
//*    the end of each line read. We manually strip the carriage return (CR)   *
//*    from the end of source lines that had ended with a CR/LF sequence.      *
//*                                                                            *
//* Another annoyance: Some source code modules have a binary value in the     *
//* first line indicating various things (see Test28.as and Test29.as).        *
//* Because we open the source file as a plain-text file, the conversion from  *
//* 'assumed' UTF-8 to the wchar_t string would fail because the binary value  *
//* is not a valid UTF-8 character. However, to compensate, if the first thing *
//* on the first line of the source code module is a binary number, we         *
//* overwrite it with SPACE characters. This does not affect the accuracy of   *
//* the analysis because it's only whitespace.                                 *
//*                                                                            *
//******************************************************************************

//****************
//* Header Files *
//****************
#include "SrcProf.hpp"

#define DEBUG_01  (0)      // For debugging only
#define DEBUG_02  (0)
#if DEBUG_01 != 0
const UINT lcMIN =   1 ;
const UINT lcMAX = 200 ;
#endif   // DEBUG_01


//****************
//* Local Data   *
//****************

//* Formatting specification for display output *
static const wchar_t* statTemplate = L"%S %s  %s   %s  %s %s   %3.2lf%%" ;

//****************
//* Prototypes   *
//****************


//*************************
//*   AnalyzeFileList     *
//*************************
//******************************************************************************
//* Perform analysis on each source file in the list.                          *
//*                                                                            *
//*                                                                            *
//* Input  : none (source data are in this->pd[])                              *
//*                                                                            *
//* Returns: 'true' if successful, or 'false' if serious error                 *
//*           (currently, we don't deem anything to be serious enough to abort)*
//******************************************************************************

bool SrcProf::AnalyzeFileList ( void )
{
   bool  success = true ;

   for ( UINT i = ZERO ; i < this->pd.fileCount ; i++ )
   {
      //* Analyze the source code file *
      if ( (AnalyzeFile ( this->pd.fileData[i] )) != false )
      {
         //* Add file's data to accumulators *
         this->pd.tLineCount    += this->pd.fileData[i].lineCount ;
         this->pd.tSourceLines  += this->pd.fileData[i].sourceLines ;
         this->pd.tCommentLines += this->pd.fileData[i].commentLines ;
         this->pd.tMixedLines   += this->pd.fileData[i].mixedLines ;
         this->pd.tBlankLines   += this->pd.fileData[i].blankLines ;
         this->pd.tMaintIndex   += this->pd.fileData[i].maintIndex ;
         this->pd.tFileChars    += this->pd.fileData[i].fileChars ;
         this->pd.tFileBytes    += this->pd.fileData[i].fileBytes ;
      }
      //else   // (currently not needed)
      //   success = false ;
   }

   //* Sort the analytical data according to *
   //* specified option (this->pd.fileSort). *
   this->SortFileData () ;

   //* Format the data for display *
   this->CreateDisplayStrings () ;

   return success ;

}  //* End AnalyzeFileList() *

//*************************
//*     AnalyzeFile       *
//*************************
//******************************************************************************
//* Scan the source file and gather statistics on the contents.                *
//*                                                                            *
//* Input  : fd: (by reference) file name and data members for the file        *
//*                                                                            *
//* Returns: 'true' if successful, or 'false' if serious error                 *
//*           (currently, we don't deem anything to be serious enough to abort)*
//******************************************************************************

bool SrcProf::AnalyzeFile ( FileData& fd )
{
   bool  success = true ;           // return value

   //* Determine the type of source file *
   fd.sfType = this->GetFileFamily ( fd.fPath.ustr() ) ;

   switch ( fd.sfType )
   {
      case sffC:                    // C-language family
         success = this->af_sffC ( fd ) ;
         break ;
      case sffA:                    // assembly-language family
         success = this->af_sffA ( fd ) ;
         break ;
      case sffPY:                   // Python
         success = this->af_sffPY ( fd ) ;
         break ;
      case sffPE:                   // Perl
         success = this->af_sffPE ( fd ) ;
         break ;
      case sffPH:                   // PHP
         success = this->af_sffPH ( fd ) ;
         break ;
      case sffRU:                   // Ruby
         success = this->af_sffRU ( fd ) ;
         break ;
      case sffVB:                   // Visual Basic
         success = this->af_sffVB ( fd ) ;
         break ;
      case sffSQ:                   // PL/SQL
         success = this->af_sffSQ ( fd ) ;
         break ;
      case sffTX:                   // Texinfo
         success = this->af_sffTX ( fd ) ;
         break ;
      case sffHT:                   // HTML
         success = this->af_sffHT ( fd ) ;
         break ;
      case sffUnk:                  // unknown or unsupported file type
      default:                      // (unlikely)
         success = false ;
   }

   return success ;

}  //* End AnalyzeFile() *

//*************************
//*        af_sffC        *
//*************************
//******************************************************************************
//* Scan source file of the sffC family:                                       *
//*  C, C++, Java, , Javascript, Flash ActionScript, Objective-C, C#, CSS      *
//*                                                                            *
//* Input  : fd: (by reference) file name and data members for the file        *
//*                                                                            *
//* Returns: 'true' if successful, or 'false' if serious error                 *
//******************************************************************************
//* NOTE: Yes, you naughty boys, it is possible to force a mis-count by        *
//*       fooling this method with some outrageous use of multiple C-style     *
//*       comments on a single line. Most other source-code constructs will    *
//*       be handled in a robust manner.                                       *
//*                                                                            *
//******************************************************************************
//* C# special:  '///' preceding XML tags for generating                       *
//*                    auto-documentation files                                *
//******************************************************************************

bool SrcProf::af_sffC ( FileData& fd )
{
   bool  success = true ;

   //* Open the source file *
   ifstream ifs ( fd.fPath.ustr(), ifstream::in ) ;
   if ( ifs.is_open() )             // if input file open
   {
      UINT  lc = ZERO,        // line count
            sl = ZERO,        // source lines
            cl = ZERO,        // comment lines
            ml = ZERO,        // mixed source/comment lines
            bl = ZERO ;       // blank (whitespace) lines
      USHORT index ;          // offest into line buffer
      char  tmp[gsMAXBYTES] ; // UTF-8 line input from file
      gString gs ;            // conversion from UTF-8 to wchar_t
      const wchar_t* inp ;    // pointer to 'wide character' array
      short lineLen ;         // number of characters in line (incl. NULLCHAR)
      bool  open_comment = false, // 'true' if C-style comment is open
            done = false ;    // loop control

      while ( ! done )
      {
         //* Read a source line *
         ifs.getline ( tmp, gsMAXBYTES, NEWLINE ) ;

         if ( ifs.good() || ifs.gcount() > ZERO || ifs.eof() )
         {
            ++lc ;                        // line counter
           if ( ifs.eof() )               // if last line has been read
              done = true ;

            //* Convert to 'wide' character string.             *
            //* Note that if the first line of the file begins  *
            //* with a non-UTF-8 sequence i.e. a binary number, *
            //* we erase the number. (see note in module header)*
            if ( (lc == 1) && tmp[ZERO] < ZERO )
            {
               for ( short i = ZERO ; tmp[i] < ZERO ; i++ )
                  tmp[i] = SPACE ;
            }
            gs = tmp ;
            fd.fileChars += gs.gschars() - 1 ;  // update character count
            fd.fileBytes += gs.utfbytes() - 1 ; // update byte count
            inp = gs.gstr( lineLen ) ;
            if ( lineLen > 1 && inp[lineLen-2] == CR )
               gs.limitCharacters(lineLen-2) ;
            index = ZERO ;                // reference beginning of line
            #if DEBUG_01 != 0
            if ( lc >= lcMIN && lc <= lcMAX )
            {
               wcout << L"LINE " << lc << L" sl:" << sl << L" ml:" 
                     << ml << L" cl:" << cl << L" bl:" << bl << endl ;
               wcout << L" '" << inp << L"'" << endl ;
            }
            #endif   // DEBUG_01
            this->SkipWhiteSpace ( inp, index ) ;   // step over initial whitespace

            //* If we have an on-going C-style comment *
            if ( open_comment )
            {
               #if DEBUG_02 != 0
               if ( lc >= lcMIN && lc <= lcMAX )
                  wcout << L" TL Open: '" << &inp[index] << L"'" << endl ;
               #endif   // DEBUG_02
               open_comment = this->Scan4CommentClose ( inp, index ) ;
               if ( open_comment )
               {  //* Entire line is part of a C-style comment *
                  #if DEBUG_02 != 0
                  if ( lc >= lcMIN && lc <= lcMAX )
                     wcout << L" +Still open '" << &inp[index] << L"'" << endl ;
                  #endif   // DEBUG_02
                  ++cl ;
               }
               else
               {  //* C-style comment has closed, but  *
                  //* something may follow the comment.*
                  #if DEBUG_02 != 0
                  if ( lc >= lcMIN && lc <= lcMAX )
                     wcout << L" +Close C-style: '" << &inp[index] << L"'" << endl ;
                  #endif   // DEBUG_02
                  this->SkipWhiteSpace ( inp, index ) ;
                  //* If end-of-line OR C++ comment, then *
                  //* only comments on this line.         *
                  if ( (inp[index] == NULLCHAR) ||
                       (inp[index] == SLASH && inp[index+1] == SLASH) )
                  {
                     ++cl ;
                     #if DEBUG_02 != 0
                     if ( lc >= lcMIN && lc <= lcMAX )
                        wcout << L" +EOL || C++ comment" << endl ;
                     #endif   // DEBUG_02
                  }
                  //* Else if a second C-style comment on the same line, *
                  //* with no intervening code, so it's likely the       *
                  //* programmer is trying to fool us and may succeed.   *
                  else if ( inp[index] == SLASH && inp[index+1] == STAR )
                  {
                     #if DEBUG_02 != 0
                     if ( lc >= lcMIN && lc <= lcMAX )
                        wcout << L" +Re-open C-style: '" << &inp[index] << L"'" << endl ;
                     #endif   // DEBUG_02
                     open_comment = this->Scan4CommentClose ( inp, index ) ;
                     #if DEBUG_02 != 0
                     if ( lc >= lcMIN && lc <= lcMAX )
                     {
                        if ( open_comment )
                           wcout << L" +still open: '" << &inp[index] << L"'" << endl ;
                        else
                           wcout << L" +re-closed: '" << &inp[index] << L"'" << endl ;
                     }
                     #endif   // DEBUG_02
                  }
                  //* Code follows comment *
                  else
                  {
                     #if DEBUG_02 != 0
                     if ( lc >= lcMIN && lc <= lcMAX )
                        wcout << L" +Code after C-style: '" << &inp[index] << L"'" << endl ;
                     #endif   // DEBUG_02
                     ++sl ;            // source code on the line AND
                     ++ml ;            // comments on the line
                     bool commentFound = this->Scan2SourceClose ( inp, index ) ;
                     if ( commentFound )
                        open_comment = this->Scan4CommentClose ( inp, index ) ;
                     #if DEBUG_02 != 0
                     if ( lc >= lcMIN && lc <= lcMAX )
                     {
                        wcout << L" +Deep C-style: '" << &inp[index] << L"'" << endl ;
                        if ( open_comment )
                           wcout << L" +still open: '" << &inp[index] << L"'" << endl ;
                        else
                           wcout << L" +re-closed: '" << &inp[index] << L"'" << endl ;
                     }
                     #endif   // DEBUG_02
                  }
               }
            }

            //* Opening a C-style comment? *
            else if ( inp[index] == '/' && inp[index+1] == '*' )
            {
               #if DEBUG_02 != 0
               if ( lc >= lcMIN && lc <= lcMAX )
                  wcout << L" Open C-style: '" << &inp[index] << L"'" << endl ;
               #endif   // DEBUG_02
               if ( (open_comment = this->Scan4CommentClose ( inp, index )) == false )
               {  //* C-style comment is closed. *
                  //* Is there more on the line? *
                  this->SkipWhiteSpace ( inp, index ) ;
                  #if DEBUG_02 != 0
                  if ( lc >= lcMIN && lc <= lcMAX )
                     wcout << L" +Closed: '" << &inp[index] << L"'" << endl ;
                  #endif   // DEBUG_02

                  //* End of line OR C++ comment *
                  if ( inp[index] == NULLCHAR || 
                       (inp[index] == SLASH && inp[index+1] == SLASH) )
                  {
                     #if DEBUG_02 != 0
                     wcout << L" +comment only: '" << &inp[index] << L"'" << endl ;
                     #endif   // DEBUG_02
                     ++cl ;
                  }
                  //* A new C-style comment on the same line? *
                  //* and without intervening source? Dumb!   *
                  else if ( inp[index] == SLASH && inp[index+1] == STAR )
                  {
                     #if DEBUG_02 != 0
                     if ( lc >= lcMIN && lc <= lcMAX )
                        wcout << L" +Re-open C-style: '" << &inp[index] << L"'" << endl ;
                     #endif   // DEBUG_02
                     open_comment = this->Scan4CommentClose ( inp, index ) ;
                     #if DEBUG_02 != 0
                     if ( lc >= lcMIN && lc <= lcMAX )
                     {
                        if ( open_comment )
                           wcout << L" +still open: '" << &inp[index] << L"'" << endl ;
                        else
                           wcout << L" +re-closed: '" << &inp[index] << L"'" << endl ;
                     }
                     #endif   // DEBUG_02
                     ++cl ;
                  }
                  //* Mixed source and comments *
                  else
                  {
                     ++sl ;
                     ++ml ;
                     #if DEBUG_02 != 0
                     if ( lc >= lcMIN && lc <= lcMAX )
                        wcout << L" +Closed and Mixed L:" << lc << "'" << &inp[index] << L"'" << endl ;
                     #endif   // DEBUG_02
                     bool commentFound = this->Scan2SourceClose ( inp, index ) ;
                     if ( commentFound && (inp[index+1] == STAR) )
                        open_comment = this->Scan4CommentClose ( inp, index ) ;
                     #if DEBUG_02 != 0
                     if ( lc >= lcMIN && lc <= lcMAX )
                     {
                        if ( open_comment ) wcout << L" +reopen" << endl ;
                        else wcout << L" +re-closed" << endl ;
                     }
                     #endif   // DEBUG_02
                  }
               }
               else
               {
                  #if DEBUG_02 != 0
                  if ( lc >= lcMIN && lc <= lcMAX )
                     wcout << L" +Still open '" << &inp[index] << L"'" << endl ;
                  #endif   // DEBUG_02
                  ++cl ;
               }
            }

            //* C++ Comment? *
            else if ( inp[index] == '/' && inp[index+1] == '/' )
            {
               #if DEBUG_02 != 0
               if ( lc >= lcMIN && lc <= lcMAX )
                  wcout << L" C++ comment: '" << &inp[index] << L"'" << endl ;
               #endif   // DEBUG_02
               ++cl ;
            }

            //* Else presummed source line (or blank line) *
            else
            {
               if ( inp[index] == NULLCHAR )
               {
                  #if DEBUG_02 != 0
                  if ( lc >= lcMIN && lc <= lcMAX )
                     wcout << L" Blank Line" << endl ;
                  #endif   // DEBUG_02
                  ++bl ;
               }
               else
               {
                  #if DEBUG_02 != 0
                  if ( lc >= lcMIN && lc <= lcMAX )
                     wcout << L" C source: '" << &inp[index] << L'\'' << endl ;
                  #endif   // DEBUG_02
                  ++sl ;                     // assume a source line
                  bool commentFound = this->Scan2SourceClose ( inp, index ) ;
                  if ( commentFound )
                  {
                     #if DEBUG_02 != 0
                     if ( lc >= lcMIN && lc <= lcMAX )
                        wcout << L" +Comment/mixed, L:" << lc << endl ;
                     #endif   // DEBUG_02
                     ++ml ;
                     if ( inp[index] == SLASH && inp[index + 1] == STAR )
                        open_comment = this->Scan4CommentClose ( inp, index ) ;
                     #if DEBUG_02 != 0
                     if ( lc >= lcMIN && lc <= lcMAX )
                        if ( open_comment ) wcout << L" +Still open" << endl ;
                     #endif   // DEBUG_02
                  }
                  #if DEBUG_02 != 0
                  else if ( lc >= lcMIN && lc <= lcMAX )
                     wcout << L" +EOL: '" << &inp[index] << L"'" << endl ;
                  #endif   // DEBUG_02
               }
            }
         }  // good
         else                    // end of file (or read error)
         {
            #if 0    // TESTING ONLY
            wcout << L"Last line char count:" << ifs.gcount() << L" '" 
                  << tmp << L"'" << endl ;
            #endif   // TESTING ONLY
            done = true ;
         }
      }     // while()

      ifs.close() ;                 // close the file

      //* If no errors, transfer data to caller *
      if ( success )
      {
         fd.lineCount    = lc ;
         fd.sourceLines  = sl ;
         fd.commentLines = cl ;
         fd.mixedLines   = ml ;
         fd.blankLines   = bl ;
         if ( fd.lineCount > ZERO ) // (avoid divide-by-zero)
         {
            fd.maintIndex  =
               (((double)fd.commentLines + (double)fd.mixedLines 
                 + fd.blankLines) / (double)fd.lineCount * 100.0) ;
         }
         else
            fd.maintIndex  = (double)0.00 ;
      }
   }
   return success ;

}  //* End af_sffC() *

//*************************
//*        af_sffA        *
//*************************
//******************************************************************************
//* Scan source file of the sffA family:                                       *
//*  ASM, M51, S07, S33 assembly languages                                     *
//*                                                                            *
//* Input  : fd: (by reference) file name and data members for the file        *
//*                                                                            *
//* Returns: 'true' if successful, or 'false' if serious error                 *
//******************************************************************************

bool SrcProf::af_sffA ( FileData& fd )
{
   bool  success = true ;

   //* Open the source file *
   ifstream ifs ( fd.fPath.ustr(), ifstream::in ) ;
   if ( ifs.is_open() )             // if input file open
   {
      UINT  lc = ZERO,        // line count
            sl = ZERO,        // source lines
            cl = ZERO,        // comment lines
            ml = ZERO,        // mixed source/comment lines
            bl = ZERO ;       // blank (whitespace) lines
      USHORT index ;          // offest into line buffer
      char  tmp[gsMAXBYTES] ; // UTF-8 line input from file
      gString gs ;            // conversion from UTF-8 to wchar_t
      const wchar_t* inp ;    // pointer to 'wide character' array
      short lineLen ;         // number of characters in line (incl. NULLCHAR)
      bool  done = false ;    // loop control

      while ( ! done )
      {
         //* Read a source line *
         ifs.getline ( tmp, gsMAXBYTES, NEWLINE ) ;

         if ( ifs.good() || ifs.gcount() > ZERO || ifs.eof() )
         {
            ++lc ;                        // line counter
           if ( ifs.eof() )               // if last line has been read
              done = true ;

            //* Convert to 'wide' character string.             *
            //* Note that if the first line of the file begins  *
            //* with a non-UTF-8 sequence i.e. a binary number, *
            //* we erase the number. (see note in module header)*
            if ( (lc == 1) && tmp[ZERO] < ZERO )
            {
               for ( short i = ZERO ; tmp[i] < ZERO ; i++ )
                  tmp[i] = SPACE ;
            }
            gs = tmp ;
            fd.fileChars += gs.gschars() - 1 ;  // update character count
            fd.fileBytes += gs.utfbytes() - 1 ; // update byte count
            inp = gs.gstr( lineLen ) ;
            if ( lineLen > 1 && inp[lineLen-2] == CR )
               gs.limitCharacters(lineLen-2) ;
            index = ZERO ;                // reference beginning of line
            #if DEBUG_01 != 0
            if ( lc >= lcMIN && lc <= lcMAX )
            {
               wcout << L"LINE " << lc << L" sl:" << sl << L" ml:" 
                     << ml << L" cl:" << cl << L" bl:" << bl << endl ;
               wcout << L" '" << inp << L"'" << endl ;
            }
            #endif   // DEBUG_01
            this->SkipWhiteSpace ( inp, index ) ;   // step over initial whitespace

            if ( inp[index] == NULLCHAR ) // a blank line?
            {
               ++bl ;                     // blank-line counter
            }
            //* If an ASM or IAR HC11 S07 comment *
            else if ( inp[index] == ';' || inp[index] == '*' )
            {
               ++cl ;                     // this line contains only a comment
            }
            else
            {
               ++sl ;                     // assume a source line
               //* Does a comment follow the source code? *
               while ( inp[index] != NULLCHAR )
               {
                  //* String constants and character constants *
                  if ( inp[index] == SGLQUOTE || inp[index] == DBLQUOTE )
                  {
                     if ( (this->Scan2ConstClose ( inp, index )) != false )
                        break ;  // EOL reached
                  }

                  if ( inp[index] == ';' || inp[index] == '*' )
                  {
                     ++ml ;
                     break ;
                  }
                  ++index ;
               }
            }
         }  // good
         else                    // end of file (or read error)
            done = true ;
      }     // while()

      ifs.close() ;                 // close the file

      //* If no errors, transfer data to caller *
      if ( success )
      {
         fd.lineCount    = lc ;
         fd.sourceLines  = sl ;
         fd.commentLines = cl ;
         fd.mixedLines   = ml ;
         fd.blankLines   = bl ;
         if ( fd.lineCount > ZERO ) // (avoid divide-by-zero)
         {
            fd.maintIndex  =
               (((double)fd.commentLines + (double)fd.mixedLines 
                 + fd.blankLines) / (double)fd.lineCount * 100.0) ;
         }
         else
            fd.maintIndex = (double)0.00 ;
      }
   }
   return success ;

}  //* End af_sffA() *

//*************************
//*       af_sffPY        *
//*************************
//******************************************************************************
//* Scan source file of the sffPY family:                                      *
//*  Python                                                                    *
//*                                                                            *
//* Input  : fd: (by reference) file name and data members for the file        *
//*                                                                            *
//* Returns: 'true' if successful, or 'false' if serious error                 *
//******************************************************************************
//*  Python:  .py (Linux/UNIX) .pyw (Winzode)                                  *
//*            '#' to end of line                                              *
//*                # this is a comment                                         *
//*                print "users suck"  # and so is this                        *
//*            .pyc     Python compiled bytecodes                              *
//*            .pyo     Python optimized compiled bytecodes                    *
//*            .pyd     Python compiled DLL                                    *
//******************************************************************************

bool SrcProf::af_sffPY ( FileData& fd )
{
   bool  success = true ;

   //* Open the source file *
   ifstream ifs ( fd.fPath.ustr(), ifstream::in ) ;
   if ( ifs.is_open() )             // if input file open
   {
      UINT  lc = ZERO,        // line count
            sl = ZERO,        // source lines
            cl = ZERO,        // comment lines
            ml = ZERO,        // mixed source/comment lines
            bl = ZERO ;       // blank (whitespace) lines
      USHORT index ;          // offest into line buffer
      char  tmp[gsMAXBYTES] ; // UTF-8 line input from file
      gString gs ;            // conversion from UTF-8 to wchar_t
      const wchar_t* inp ;    // pointer to 'wide character' array
      short lineLen ;         // number of characters in line (incl. NULLCHAR)
      bool  done = false ;    // loop control

      while ( ! done )
      {
         //* Read a source line *
         ifs.getline ( tmp, gsMAXBYTES, NEWLINE ) ;

         if ( ifs.good() || ifs.gcount() > ZERO || ifs.eof() )
         {
            ++lc ;                        // line counter
           if ( ifs.eof() )               // if last line has been read
              done = true ;

            //* Convert to 'wide' character string.             *
            //* Note that if the first line of the file begins  *
            //* with a non-UTF-8 sequence i.e. a binary number, *
            //* we erase the number. (see note in module header)*
            if ( (lc == 1) && tmp[ZERO] < ZERO )
            {
               for ( short i = ZERO ; tmp[i] < ZERO ; i++ )
                  tmp[i] = SPACE ;
            }
            gs = tmp ;
            fd.fileChars += gs.gschars() - 1 ;  // update character count
            fd.fileBytes += gs.utfbytes() - 1 ; // update byte count
            inp = gs.gstr( lineLen ) ;
            if ( lineLen > 1 && inp[lineLen-2] == CR )
               gs.limitCharacters(lineLen-2) ;
            index = ZERO ;                // reference beginning of line
            #if DEBUG_01 != 0
            if ( lc >= lcMIN && lc <= lcMAX )
            {
               wcout << L"LINE " << lc << L" sl:" << sl << L" ml:" 
                     << ml << L" cl:" << cl << L" bl:" << bl << endl ;
               wcout << L" '" << inp << L"'" << endl ;
            }
            #endif   // DEBUG_01
            this->SkipWhiteSpace ( inp, index ) ;   // step over initial whitespace

            if ( inp[index] == NULLCHAR ) // a blank line?
            {
               ++bl ;                     // blank-line counter
            }
            //* If a comment-only line *
            else if ( inp[index] == HASH )
            {
               ++cl ;                     // this line contains only a comment
            }
            else
            {
               ++sl ;                     // assume a source line
               //* Does a comment follow the source code? *
               while ( inp[index] != NULLCHAR )
               {
                  //* String constants and character constants *
                  if ( inp[index] == SGLQUOTE || inp[index] == DBLQUOTE )
                  {
                     if ( (this->Scan2ConstClose ( inp, index )) != false )
                        break ;  // EOL reached
                  }

                  if ( inp[index] == HASH )
                  {
                     ++ml ;
                     break ;
                  }
                  ++index ;
               }
            }
         }  // good
         else                    // end of file (or read error)
            done = true ;
      }     // while()

      ifs.close() ;                 // close the file

      //* If no errors, transfer data to caller *
      if ( success )
      {
         fd.lineCount    = lc ;
         fd.sourceLines  = sl ;
         fd.commentLines = cl ;
         fd.mixedLines   = ml ;
         fd.blankLines   = bl ;
         if ( fd.lineCount > ZERO ) // (avoid divide-by-zero)
         {
            fd.maintIndex  =
               (((double)fd.commentLines + (double)fd.mixedLines 
                 + fd.blankLines) / (double)fd.lineCount * 100.0) ;
         }
         else
            fd.maintIndex = (double)0.00 ;
      }
   }
   return success ;

}  //* End af_sffPY() *

//*************************
//*       af_sffPE        *
//*************************
//******************************************************************************
//* Scan source file of the sffPE family:                                      *
//*  Perl                                                                      *
//*                                                                            *
//* Input  : fd: (by reference) file name and data members for the file        *
//*                                                                            *
//* Returns: 'true' if successful, or 'false' if serious error                 *
//******************************************************************************
//*  Perl:  .pl  .pm   .t                                                      *
//*          '#' to end of line i.e. single-line comment                       *
//*          A perldoc (Pod) sequence is often used. Examples:                 *
//*          =begin comment                                                    *
//*           ...                                                              *
//*          =cut                                                              *
//*             OR                                                             *
//*          =for comment                                                      *
//*           ...                                                              *
//*          =cut                                                              *
//*             OR                                                             *
//*          =begin comment                                                    *
//*           ...                                                              *
//*          =end comment                                                      *
//*             OR                                                             *
//*          Paragraph block with the =for Pod comand is terminated by the     *
//*          next whitespace line or the next Pod command                      *
//*          =for                                                              *
//*           ...                                                              *
//*                                                                            *
//*          And sooo many other clunky options/delimiters, etc.               *
//*          (these sequences must begin in the first column)                  *
//*          The question remains, is a Pod sequence code or comment?          *
//*          Our "official" solution: Yes, Pod sequences are _code_ EXCEPT:    *
//*          =begin comment                                                    *
//*           ...                                                              *
//*          =end comment                                                      *
//*          =begin comment                                                    *
//*           ...                                                              *
//*          =cut                                                              *
//*                                                                            *
//*           Beginning in Perl6 the delimited comment is supported.           *
//*           #{ comment                                                       *
//*              here                                                          *
//*            }                                                               *
//*           #[ comment                                                       *
//*              here                                                          *
//*            ]                                                               *
//*           #( comment here )                                                *
//*           #< comment here >                                                *
//*           #« comment etc. »                                                *
//*           Any open/close set of delimiters is possible, however, we cannot *
//*           possibly test for every potential open/close pair. For this      *
//*           reason, we test for the four(5) most common delimiter sets.      *
//*              {} , [] , () , <> , «»                                        *
//*                                                                            *
//******************************************************************************

bool SrcProf::af_sffPE ( FileData& fd )
{
   const wchar_t LBRACE   = (L'{') ;   // left curley brace
   const wchar_t RBRACE   = (L'}') ;   // right curley brace
   const wchar_t LBRACKET = (L'[') ;   // left curley brace
   const wchar_t RBRACKET = (L']') ;   // right curley brace
   const wchar_t LPAREN   = (L'(') ;   // left curley brace
   const wchar_t RPAREN   = (L')') ;   // right curley brace
   const wchar_t LANGLE   = (L'<') ;   // left angle bracket
   const wchar_t RANGLE   = (L'>') ;   // right angle bracket
   const wchar_t LDANGLE  = (L'«') ;   // left double-angle quotation mark
   const wchar_t RDANGLE  = (L'»') ;   // right double-angle quotation mark
   const wchar_t* OPENBLK = L"=begin comment" ;
   const wchar_t* CLOSEBLK1 = L"=end comment" ;
   const wchar_t* CLOSEBLK2 = L"=cut" ;

   bool  success = true ;

   //* Open the source file *
   ifstream ifs ( fd.fPath.ustr(), ifstream::in ) ;
   if ( ifs.is_open() )             // if input file open
   {
      UINT  lc = ZERO,        // line count
            sl = ZERO,        // source lines
            cl = ZERO,        // comment lines
            ml = ZERO,        // mixed source/comment lines
            bl = ZERO ;       // blank (whitespace) lines
      USHORT index ;          // offest into line buffer
      char  tmp[gsMAXBYTES] ; // UTF-8 line input from file
      gString gs ;            // conversion from UTF-8 to wchar_t
      wchar_t bdelim, edelim ;// beginning and ending delimiter for block comment
      const wchar_t* inp ;    // pointer to 'wide character' array
      short lineLen ;         // number of characters in line (incl. NULLCHAR)
      bool  open_comment = false,    // 'true' if block comment is open
            open_podcomment = false, // 'true' if POD block comment is open
            done = false ;    // loop control

      while ( ! done )
      {
         //* Read a source line *
         ifs.getline ( tmp, gsMAXBYTES, NEWLINE ) ;

         if ( ifs.good() || ifs.gcount() > ZERO || ifs.eof() )
         {
            ++lc ;                        // line counter
           if ( ifs.eof() )               // if last line has been read
              done = true ;

            //* Convert to 'wide' character string.             *
            //* Note that if the first line of the file begins  *
            //* with a non-UTF-8 sequence i.e. a binary number, *
            //* we erase the number. (see note in module header)*
            if ( (lc == 1) && tmp[ZERO] < ZERO )
            {
               for ( short i = ZERO ; tmp[i] < ZERO ; i++ )
                  tmp[i] = SPACE ;
            }
            gs = tmp ;
            fd.fileChars += gs.gschars() - 1 ;  // update character count
            fd.fileBytes += gs.utfbytes() - 1 ; // update byte count
            inp = gs.gstr( lineLen ) ;
            if ( lineLen > 1 && inp[lineLen-2] == CR )
               gs.limitCharacters(lineLen-2) ;
            index = ZERO ;                // reference beginning of line
            #if DEBUG_01 != 0
            if ( lc >= lcMIN && lc <= lcMAX )
            {
               wcout << L"LINE " << lc << L" sl:" << sl << L" ml:" 
                     << ml << L" cl:" << cl << L" bl:" << bl << endl ;
               wcout << L" '" << inp << L"'" << endl ;
            }
            #endif   // DEBUG_01
            this->SkipWhiteSpace ( inp, index ) ;   // step over initial whitespace

            //* If we have an on-going block comment *
            if ( open_comment )
            {
               ++cl ;                     // this line contains only a comment
               if ( (gs.find( edelim, index )) >= ZERO )
               {
                  open_comment = false ;
               }
            }
            else if ( open_podcomment )
            {
               ++cl ;                     // this line contains only a comment
               if ( ((gs.find( CLOSEBLK1, index )) >= ZERO) ||
                    ((gs.find( CLOSEBLK2, index )) >= ZERO) )
               {
                  open_podcomment = false ;
               }
            }

            else if ( inp[index] == NULLCHAR ) // a blank line?
            {
               ++bl ;                     // blank-line counter
            }
            //* If a comment-only line _OR_ beginning of a block comment * *
            else if ( inp[index] == HASH )
            {
               ++cl ;                     // this line contains only a comment
               bdelim = inp[index + 1] ;
               if ( bdelim == LBRACE || bdelim == LBRACKET || bdelim == LPAREN || 
                    bdelim == LANGLE || bdelim == LDANGLE )
               {  //* Scan to end of current line for closing delimiter.*
                  //* If not found, then comment continues on next line.*
                  switch ( bdelim )
                  {
                     case LBRACE:   edelim = RBRACE ;    break ;
                     case LBRACKET: edelim = RBRACKET ;  break ;
                     case LPAREN:   edelim = RPAREN ;    break ;
                     case LANGLE:   edelim = RANGLE ;    break ;
                     case LDANGLE:  edelim = RDANGLE ;   break ;
                  }
                  if ( (gs.find( edelim, (index + 2) )) < ZERO )
                  {
                     open_comment = true ;
                  }
               }
            }
            else if ( (gs.find( OPENBLK, index )) == index )
            {
               ++cl ;                     // this line begins a POD comment
               open_podcomment = true ;   // and it continues to the next line
            }
            else
            {
               ++sl ;
            }
         }  // good
         else                    // end of file (or read error)
            done = true ;
      }     // while()

      ifs.close() ;                 // close the file

      //* If no errors, transfer data to caller *
      if ( success )
      {
         fd.lineCount    = lc ;
         fd.sourceLines  = sl ;
         fd.commentLines = cl ;
         fd.mixedLines   = ml ;
         fd.blankLines   = bl ;
         if ( fd.lineCount > ZERO ) // (avoid divide-by-zero)
         {
            fd.maintIndex  =
               (((double)fd.commentLines + (double)fd.mixedLines 
                 + fd.blankLines) / (double)fd.lineCount * 100.0) ;
         }
         else
            fd.maintIndex = (double)0.00 ;
      }
   }
   return success ;

}  //* End af_sffPE() *

//*************************
//*       af_sffPH        *
//*************************
//******************************************************************************
//* Scan source file of the sffPH family:                                      *
//*  PHP                                                                       *
//*                                                                            *
//* Input  : fd: (by reference) file name and data members for the file        *
//*                                                                            *
//* Returns: 'true' if successful, or 'false' if serious error                 *
//******************************************************************************
//*  PHP  .php .phtml .php3 .php4 .php5 .phps                                  *
//*        '#' or '//'  single-line comment, OR                                *
//*        /*                                                                  *
//*          this is a multi-line comment                                      *
//*         */                                                                 *
//* Example:                                                                   *
//*        <?php the_excerpt(); // Show excerpt and not full post content ?>   *
//*                                                                            *
//******************************************************************************
bool SrcProf::af_sffPH ( FileData& fd )
{
   bool  success = true ;

#if 1    // UNDER CONSTRUCTION
   //* Open the source file *
   ifstream ifs ( fd.fPath.ustr(), ifstream::in ) ;
   if ( ifs.is_open() )             // if input file open
   {
      UINT  lc = ZERO,        // line count
            sl = ZERO,        // source lines
            cl = ZERO,        // comment lines
            ml = ZERO,        // mixed source/comment lines
            bl = ZERO ;       // blank (whitespace) lines
      USHORT index ;          // offest into line buffer
      char  tmp[gsMAXBYTES] ; // UTF-8 line input from file
      gString gs ;            // conversion from UTF-8 to wchar_t
      const wchar_t* inp ;    // pointer to 'wide character' array
      short lineLen ;         // number of characters in line (incl. NULLCHAR)
      bool  done = false ;    // loop control

      while ( ! done )
      {
         //* Read a source line *
         ifs.getline ( tmp, gsMAXBYTES, NEWLINE ) ;

         if ( ifs.good() || ifs.gcount() > ZERO || ifs.eof() )
         {
            ++lc ;                        // line counter

            //* Convert to 'wide' character string.             *
            //* Note that if the first line of the file begins  *
            //* with a non-UTF-8 sequence i.e. a binary number, *
            //* we erase the number. (see note in module header)*
            if ( (lc == 1) && tmp[ZERO] < ZERO )
            {
               for ( short i = ZERO ; tmp[i] < ZERO ; i++ )
                  tmp[i] = SPACE ;
            }
            gs = tmp ;
            fd.fileChars += gs.gschars() - 1 ;  // update character count
            fd.fileBytes += gs.utfbytes() - 1 ; // update byte count
            inp = gs.gstr( lineLen ) ;
            if ( lineLen > 1 && inp[lineLen-2] == CR )
               gs.limitCharacters(lineLen-2) ;
            index = ZERO ;                // reference beginning of line
            #if DEBUG_01 != 0
            if ( lc >= lcMIN && lc <= lcMAX )
            {
               wcout << L"LINE " << lc << L" sl:" << sl << L" ml:" 
                     << ml << L" cl:" << cl << L" bl:" << bl << endl ;
               wcout << L" '" << inp << L"'" << endl ;
            }
            #endif   // DEBUG_01
            this->SkipWhiteSpace ( inp, index ) ;   // step over initial whitespace

// UNDER CONSTRUCTION
         }  // good
         else                    // end of file (or read error)
            done = true ;
      }     // while()

      ifs.close() ;                 // close the file

      //* If no errors, transfer data to caller *
      if ( success )
      {
         fd.lineCount    = lc ;
         fd.sourceLines  = sl ;
         fd.commentLines = cl ;
         fd.mixedLines   = ml ;
         fd.blankLines   = bl ;
         if ( fd.lineCount > ZERO ) // (avoid divide-by-zero)
         {
            fd.maintIndex  =
               (((double)fd.commentLines + (double)fd.mixedLines 
                 + fd.blankLines) / (double)fd.lineCount * 100.0) ;
         }
         else
            fd.maintIndex = (double)0.00 ;
      }
   }
#endif   // UNDER CONSTRUCTION
   return success ;

}  //* End af_sffPH() *

//*************************
//*       af_sffRU        *
//*************************
//******************************************************************************
//* Scan source file of the sffRU family:                                      *
//*  Ruby                                                                      *
//*                                                                            *
//* Input  : fd: (by reference) file name and data members for the file        *
//*                                                                            *
//* Returns: 'true' if successful, or 'false' if serious error                 *
//******************************************************************************
//*  Ruby  .rb  .rbw                                                           *
//*         '#' to end of line i.e. single-line comment, OR multi:             *
//*         '=begin' alone on a line with '=end' alone on a line               *
//******************************************************************************

bool SrcProf::af_sffRU ( FileData& fd )
{
   bool  success = true ;

#if 1    // UNDER CONSTRUCTION
   //* Open the source file *
   ifstream ifs ( fd.fPath.ustr(), ifstream::in ) ;
   if ( ifs.is_open() )             // if input file open
   {
      UINT  lc = ZERO,        // line count
            sl = ZERO,        // source lines
            cl = ZERO,        // comment lines
            ml = ZERO,        // mixed source/comment lines
            bl = ZERO ;       // blank (whitespace) lines
      USHORT index ;          // offest into line buffer
      char  tmp[gsMAXBYTES] ; // UTF-8 line input from file
      gString gs ;            // conversion from UTF-8 to wchar_t
      const wchar_t* inp ;    // pointer to 'wide character' array
      short lineLen ;         // number of characters in line (incl. NULLCHAR)
      bool  done = false ;    // loop control

      while ( ! done )
      {
         //* Read a source line *
         ifs.getline ( tmp, gsMAXBYTES, NEWLINE ) ;

         if ( ifs.good() || ifs.gcount() > ZERO || ifs.eof() )
         {
            ++lc ;                        // line counter

            //* Convert to 'wide' character string.             *
            //* Note that if the first line of the file begins  *
            //* with a non-UTF-8 sequence i.e. a binary number, *
            //* we erase the number. (see note in module header)*
            if ( (lc == 1) && tmp[ZERO] < ZERO )
            {
               for ( short i = ZERO ; tmp[i] < ZERO ; i++ )
                  tmp[i] = SPACE ;
            }
            gs = tmp ;
            fd.fileChars += gs.gschars() - 1 ;  // update character count
            fd.fileBytes += gs.utfbytes() - 1 ; // update byte count
            inp = gs.gstr( lineLen ) ;
            if ( lineLen > 1 && inp[lineLen-2] == CR )
               gs.limitCharacters(lineLen-2) ;
            index = ZERO ;                // reference beginning of line
            #if DEBUG_01 != 0
            if ( lc >= lcMIN && lc <= lcMAX )
            {
               wcout << L"LINE " << lc << L" sl:" << sl << L" ml:" 
                     << ml << L" cl:" << cl << L" bl:" << bl << endl ;
               wcout << L" '" << inp << L"'" << endl ;
            }
            #endif   // DEBUG_01
            this->SkipWhiteSpace ( inp, index ) ;   // step over initial whitespace

// UNDER CONSTRUCTION
         }  // good
         else                    // end of file (or read error)
            done = true ;
      }     // while()

      ifs.close() ;                 // close the file

      //* If no errors, transfer data to caller *
      if ( success )
      {
         fd.lineCount    = lc ;
         fd.sourceLines  = sl ;
         fd.commentLines = cl ;
         fd.mixedLines   = ml ;
         fd.blankLines   = bl ;
         if ( fd.lineCount > ZERO ) // (avoid divide-by-zero)
         {
            fd.maintIndex  =
               (((double)fd.commentLines + (double)fd.mixedLines 
                 + fd.blankLines) / (double)fd.lineCount * 100.0) ;
         }
         else
            fd.maintIndex = (double)0.00 ;
      }
   }
#endif   // UNDER CONSTRUCTION
   return success ;

}  //* End af_sffRU() *

//*************************
//*       af_sffVB        *
//*************************
//******************************************************************************
//* Scan source file of the sffVB family:                                      *
//*  Visual Basic                                                              *
//*                                                                            *
//* Input  : fd: (by reference) file name and data members for the file        *
//*                                                                            *
//* Returns: 'true' if successful, or 'false' if serious error                 *
//******************************************************************************
//*  Visual Basic:   .vb  .vbs                                                 *
//*  and VBScript     ' this is a comment                                      *
//*                     DoSomeStuff ()  ' this function does something         *
//******************************************************************************

bool SrcProf::af_sffVB ( FileData& fd )
{
   bool  success = true ;

   //* Open the source file *
   ifstream ifs ( fd.fPath.ustr(), ifstream::in ) ;
   if ( ifs.is_open() )             // if input file open
   {
      UINT  lc = ZERO,        // line count
            sl = ZERO,        // source lines
            cl = ZERO,        // comment lines
            ml = ZERO,        // mixed source/comment lines
            bl = ZERO ;       // blank (whitespace) lines
      USHORT index ;          // offest into line buffer
      char  tmp[gsMAXBYTES] ; // UTF-8 line input from file
      gString gs ;            // conversion from UTF-8 to wchar_t
      const wchar_t* inp ;    // pointer to 'wide character' array
      short lineLen ;         // number of characters in line (incl. NULLCHAR)
      bool  done = false ;    // loop control

      while ( ! done )
      {
         //* Read a source line *
         ifs.getline ( tmp, gsMAXBYTES, NEWLINE ) ;

         if ( ifs.good() || ifs.gcount() > ZERO || ifs.eof() )
         {
            ++lc ;                        // line counter
           if ( ifs.eof() )               // if last line has been read
              done = true ;

            //* Convert to 'wide' character string.             *
            //* Note that if the first line of the file begins  *
            //* with a non-UTF-8 sequence i.e. a binary number, *
            //* we erase the number. (see note in module header)*
            if ( (lc == 1) && tmp[ZERO] < ZERO )
            {
               for ( short i = ZERO ; tmp[i] < ZERO ; i++ )
                  tmp[i] = SPACE ;
            }
            gs = tmp ;
            fd.fileChars += gs.gschars() - 1 ;  // update character count
            fd.fileBytes += gs.utfbytes() - 1 ; // update byte count
            inp = gs.gstr( lineLen ) ;
            if ( lineLen > 1 && inp[lineLen-2] == CR )
               gs.limitCharacters(lineLen-2) ;
            index = ZERO ;                // reference beginning of line
            #if DEBUG_01 != 0
            if ( lc >= lcMIN && lc <= lcMAX )
            {
               wcout << L"LINE " << lc << L" sl:" << sl << L" ml:" 
                     << ml << L" cl:" << cl << L" bl:" << bl << endl ;
               wcout << L" '" << inp << L"'" << endl ;
            }
            #endif   // DEBUG_01
            this->SkipWhiteSpace ( inp, index ) ;   // step over initial whitespace

            if ( inp[index] == NULLCHAR ) // a blank line?
            {
               ++bl ;                     // blank-line counter
            }
            //* If a comment-only line *
            else if ( inp[index] == SGLQUOTE )
            {
               ++cl ;                     // this line contains only a comment
            }
            else
            {
               ++sl ;                     // assume a source line
               //* Does a comment follow the source code? *
               while ( inp[index] != NULLCHAR )
               {
                  //* String constants may have embedded SGLQUOTE characters *
                  if ( inp[index] == DBLQUOTE )
                  {
                     if ( (this->Scan2ConstClose ( inp, index )) != false )
                        break ;  // EOL reached
                  }

                  if ( inp[index] == SGLQUOTE )
                  {
                     ++ml ;
                     break ;
                  }
                  ++index ;
               }
            }
         }  // good
         else                    // end of file (or read error)
            done = true ;
      }     // while()

      ifs.close() ;                 // close the file

      //* If no errors, transfer data to caller *
      if ( success )
      {
         fd.lineCount    = lc ;
         fd.sourceLines  = sl ;
         fd.commentLines = cl ;
         fd.mixedLines   = ml ;
         fd.blankLines   = bl ;
         if ( fd.lineCount > ZERO ) // (avoid divide-by-zero)
         {
            fd.maintIndex  =
               (((double)fd.commentLines + (double)fd.mixedLines 
                 + fd.blankLines) / (double)fd.lineCount * 100.0) ;
         }
         else
            fd.maintIndex = (double)0.00 ;
      }
   }
   return success ;

}  //* End af_sffVB() *

//*************************
//*       af_sffSQ        *
//*************************
//******************************************************************************
//* Scan source file of the sffSQ family:                                      *
//*  PL/SQL                                                                    *
//*                                                                            *
//* Input  : fd: (by reference) file name and data members for the file        *
//*                                                                            *
//* Returns: 'true' if successful, or 'false' if serious error                 *
//******************************************************************************
//*  PL/SQL .sql '--' ANSI/ISO,  '{...}' INFORMIX,  '/*...*/' SQL-99           *
//*          If the product that you use supports all of these comment         *
//*          symbols, your choice of a comment symbol depends on               *
//*          requirements for ANSI/ISO compliance. If ANSI/ISO                 *
//*          compliance is not an issue, your choice of comment                *
//*          symbols is a matter of personal preference.                       *
//*          - Double hyphen ( -- ) complies with the ANSI/ISO                 *
//*            standard for SQL. [single-line,everything that follows]         *
//*          - Braces ( { } ) are an Informix extension to the                 *
//*            ANSI/ISO standard. [single/multiline]                           *
//*          - C-style slash-and-asterisk ( /* . . . */ ) comply with          *
//*            the SQL-99 standard. [single/multiline]                         *
//******************************************************************************

bool SrcProf::af_sffSQ ( FileData& fd )
{
   bool  success = true ;

#if 1    // UNDER CONSTRUCTION
   //* Open the source file *
   ifstream ifs ( fd.fPath.ustr(), ifstream::in ) ;
   if ( ifs.is_open() )             // if input file open
   {
      UINT  lc = ZERO,        // line count
            sl = ZERO,        // source lines
            cl = ZERO,        // comment lines
            ml = ZERO,        // mixed source/comment lines
            bl = ZERO ;       // blank (whitespace) lines
      USHORT index ;          // offest into line buffer
      char  tmp[gsMAXBYTES] ; // UTF-8 line input from file
      gString gs ;            // conversion from UTF-8 to wchar_t
      const wchar_t* inp ;    // pointer to 'wide character' array
      short lineLen ;         // number of characters in line (incl. NULLCHAR)
      bool  done = false ;    // loop control

      while ( ! done )
      {
         //* Read a source line *
         ifs.getline ( tmp, gsMAXBYTES, NEWLINE ) ;

         if ( ifs.good() || ifs.gcount() > ZERO || ifs.eof() )
         {
            ++lc ;                        // line counter

            //* Convert to 'wide' character string.             *
            //* Note that if the first line of the file begins  *
            //* with a non-UTF-8 sequence i.e. a binary number, *
            //* we erase the number. (see note in module header)*
            if ( (lc == 1) && tmp[ZERO] < ZERO )
            {
               for ( short i = ZERO ; tmp[i] < ZERO ; i++ )
                  tmp[i] = SPACE ;
            }
            gs = tmp ;
            fd.fileChars += gs.gschars() - 1 ;  // update character count
            fd.fileBytes += gs.utfbytes() - 1 ; // update byte count
            inp = gs.gstr( lineLen ) ;
            if ( lineLen > 1 && inp[lineLen-2] == CR )
               gs.limitCharacters(lineLen-2) ;
            index = ZERO ;                // reference beginning of line
            #if DEBUG_01 != 0
            if ( lc >= lcMIN && lc <= lcMAX )
            {
               wcout << L"LINE " << lc << L" sl:" << sl << L" ml:" 
                     << ml << L" cl:" << cl << L" bl:" << bl << endl ;
               wcout << L" '" << inp << L"'" << endl ;
            }
            #endif   // DEBUG_01
            this->SkipWhiteSpace ( inp, index ) ;   // step over initial whitespace

// UNDER CONSTRUCTION
         }  // good
         else                    // end of file (or read error)
            done = true ;
      }     // while()

      ifs.close() ;                 // close the file

      //* If no errors, transfer data to caller *
      if ( success )
      {
         fd.lineCount    = lc ;
         fd.sourceLines  = sl ;
         fd.commentLines = cl ;
         fd.mixedLines   = ml ;
         fd.blankLines   = bl ;
         if ( fd.lineCount > ZERO ) // (avoid divide-by-zero)
         {
            fd.maintIndex  =
               (((double)fd.commentLines + (double)fd.mixedLines 
                 + fd.blankLines) / (double)fd.lineCount * 100.0) ;
         }
         else
            fd.maintIndex = (double)0.00 ;
      }
   }
#endif   // UNDER CONSTRUCTION
   return success ;

}  //* End af_sffSQ() *

//*************************
//*       af_sffTX        *
//*************************
//******************************************************************************
//* Scan source file of the sffTX family:                                      *
//*  Texinfo                                                                   *
//*                                                                            *
//* Input  : fd: (by reference) file name and data members for the file        *
//*                                                                            *
//* Returns: 'true' if successful, or 'false' if serious error                 *
//******************************************************************************
//* Texinfo specifies two kinds of comments:                                   *
//* 1) '@comment'  or  '@c '  anywhere in the line                             *
//* 2) '@ignore' / '@end ignore' blocks                                        *
//*                                                                            *
//*                                                                            *
//* NOTE: Because our '.texi' test data contains a large number of supra-ASCII *
//*       characters, it is important that the application is running in a     *
//*       UTF-8 locale, which it is...                                         *
//*                                                                            *
//******************************************************************************

bool SrcProf::af_sffTX ( FileData& fd )
{
   bool  success = true ;

   //* Open the source file *
   ifstream ifs ( fd.fPath.ustr(), ifstream::in ) ;
   if ( ifs.is_open() )             // if input file open
   {
      UINT  lc = ZERO,        // line count
            sl = ZERO,        // source lines
            cl = ZERO,        // comment lines
            ml = ZERO,        // mixed source/comment lines
            bl = ZERO ;       // blank (whitespace) lines
      USHORT index ;          // offest into line buffer
      short  i2 ;
      char  tmp[gsMAXBYTES] ; // UTF-8 line input from file
      gString gs ;            // conversion from UTF-8 to wchar_t
      const wchar_t* inp ;    // pointer to 'wide character' array
      short lineLen ;         // number of characters in line (incl. NULLCHAR)
      bool  open_comment = false, // 'true' if @ignore-style comment is open
            line_comment = false, // 'true' if @c-style comment detected
            done = false ;    // loop control

      while ( ! done )
      {
         //* Read a source line *
         ifs.getline ( tmp, gsMAXBYTES, NEWLINE ) ;

         if ( ifs.good() || ifs.gcount() > ZERO || ifs.eof() )
         {
            ++lc ;                        // line counter
           if ( ifs.eof() )               // if last line has been read
              done = true ;

            //* Convert to 'wide' character string.             *
            //* Note that if the first line of the file begins  *
            //* with a non-UTF-8 sequence i.e. a binary number, *
            //* we erase the number. (see note in module header)*
            if ( (lc == 1) && tmp[ZERO] < ZERO )
            {
               for ( short i = ZERO ; tmp[i] < ZERO ; i++ )
                  tmp[i] = SPACE ;
            }
            gs = tmp ;
            fd.fileChars += gs.gschars() - 1 ;  // update character count
            fd.fileBytes += gs.utfbytes() - 1 ; // update byte count
            inp = gs.gstr( lineLen ) ;
            if ( lineLen > 1 && inp[lineLen-2] == CR )
               gs.limitCharacters(lineLen-2) ;
            index = ZERO ;                // reference beginning of line
            this->SkipWhiteSpace ( inp, index ) ;   // step over initial whitespace
            line_comment = false ;
            if ( (i2 = gs.find( L"@c", index )) >= ZERO )
            {
               if ( ((gs.find( L"@c ", i2 )) >= ZERO) || 
                    ((gs.find( L"@comment", i2 )) >= ZERO) )
                  line_comment = true ;
            }

            //* If we have an on-going @ignore-style comment *
            if ( open_comment )
            {
               if ( (wcsncmp ( &inp[index], L"@end ignore", 11 )) == ZERO )
                  open_comment = false ;
               ++cl ;
            }
            //* Opening an @ignore-style comment? *
            else if ( (wcsncmp ( &inp[index], L"@ignore", 7)) == ZERO )
            {
               open_comment = true ;
               ++cl ;
            }

            //* If single-line comment previously detected (see above) *
            else if ( line_comment != false )
            {
               if ( i2 > index )    // if we have code FOLLOWED BY a comment
                  ++ml ;
               else
                  ++cl ;                     // else, a comment only
            }

            //* Else presummed source line (or blank line) *
            else
            {
               if ( inp[index] == NULLCHAR )
                  ++bl ;                     // blank line
               else
                  ++sl ;                     // assume a source line
            }
         }  // good
         else                    // end of file (or read error)
         {
            done = true ;
         }
      }     // while()

      ifs.close() ;                 // close the file

      //* If no errors, transfer data to caller *
      if ( success )
      {
         fd.lineCount    = lc ;
         fd.sourceLines  = sl ;
         fd.commentLines = cl ;
         fd.mixedLines   = ml ;
         fd.blankLines   = bl ;
         if ( fd.lineCount > ZERO ) // (avoid divide-by-zero)
         {
            fd.maintIndex  =
               (((double)fd.commentLines + (double)fd.mixedLines 
                 + fd.blankLines) / (double)fd.lineCount * 100.0) ;
         }
         else
            fd.maintIndex  = (double)0.00 ;
      }
   }
   return success ;

}  //* End af_sffTX() *

//*************************
//*       af_sffHT        *
//*************************
//******************************************************************************
//* Scan source file of the sffHT family:                                      *
//*  HTML                                                                      *
//*                                                                            *
//* Input  : fd: (by reference) file name and data members for the file        *
//*                                                                            *
//* Returns: 'true' if successful, or 'false' if serious error                 *
//******************************************************************************
//* An HTML comment begins with "<!--", ends with "-->" AND                    *
//* does not contain "--" or ">" anywhere in the comment.                      *
//*                                                                            *
//* Examples of legal comments:                                                *
//* <!-- Hello -->                                                             *
//* <!---->                                                                    *
//* <!-- Hello                                                                 *
//*      World! -->                                                            *
//*                                                                            *
//* Conditional comments, executed by IE only.                                 *
//* (We ignore this fraking mess!)                                             *
//* <!--[if IE 8]>                                                             *
//*     .... some HTML here ....                                               *
//* <![endif]-->                                                               *
//*                                                                            *
//* CSS comments are equivalent to C-language comments:                        *
//*   /* Hello */                                                              *
//*   /* Hello                                                                 *
//*      World */                                                              *
//* Because we may have CSS embedded within the HTML, we need to look for      *
//* this too. Note: We assume that HTML comments can enclose CSS comments,     *
//* but NOT the other way around.                                              *
//* IMPORTANT NOTE: We scan for CSS comments _ONLY_ within the <head></head>   *
//*    block because this is where most pre-defined style data will live.      *
//*    Beyond the <head></head> block, CSS comments are ignored on the         *
//*    supposition that the HTML code may be discussing 'C' code, which would  *
//*    cause an invalid comment count.                                         *
//*                                                                            *
//* Also, we may find PHP inside the HTML, and PHP comments have their own     *
//* varieties:                                                                 *
//*  a) # Hello         (this form would not be legal in HTML)                 *
//*  b) // Hello                                                               *
//*  c) /* Hello */                                                            *
//*  d) /* Hello                                                               *
//*        World */                                                            *
//* NOTE: We do not scan for PHP '//' comments at this time.                   *
//*       PHP '/* ... */' comments are unlikely to appear in the <head> block, *
//*       which is the only place we scan for 'C'-style comments; therefore,   *
//*       no PHP comments will be reported.                                    *
//*                                                                            *
//*                                                                            *
//*                                                                            *
//******************************************************************************

bool SrcProf::af_sffHT ( FileData& fd )
{
   bool  success = true ;

   //* Open the source file *
   ifstream ifs ( fd.fPath.ustr(), ifstream::in ) ;
   if ( ifs.is_open() )             // if input file open
   {
      UINT  lc = ZERO,        // line count
            sl = ZERO,        // source lines
            cl = ZERO,        // comment lines
            ml = ZERO,        // mixed source/comment lines
            bl = ZERO ;       // blank (whitespace) lines
      USHORT index ;          // offest into line buffer
      short  i2 ;
      char  tmp[gsMAXBYTES] ; // UTF-8 line input from file
      gString gs ;            // conversion from UTF-8 to wchar_t
      const wchar_t* inp ;    // pointer to 'wide character' array
      short lineLen ;         // number of characters in line (incl. NULLCHAR)
      bool  h_comment = false, // 'true' if HTML comment is open
            c_comment = false, // 'true' if C-style comment is open
            after_head = false,// 'true' if scan has moved beyond the </head> tag
            done = false ;    // loop control

      while ( ! done )
      {
         //* Read a source line *
         ifs.getline ( tmp, gsMAXBYTES, NEWLINE ) ;

         if ( ifs.good() || ifs.gcount() > ZERO || ifs.eof() )
         {
            ++lc ;                        // line counter
           if ( ifs.eof() )               // if last line has been read
              done = true ;

            //* Convert to 'wide' character string.             *
            //* Note that if the first line of the file begins  *
            //* with a non-UTF-8 sequence i.e. a binary number, *
            //* we erase the number. (see note in module header)*
            if ( (lc == 1) && tmp[ZERO] < ZERO )
            {
               for ( short i = ZERO ; tmp[i] < ZERO ; i++ )
                  tmp[i] = SPACE ;
            }
            gs = tmp ;
            fd.fileChars += gs.gschars() - 1 ;  // update character count
            fd.fileBytes += gs.utfbytes() - 1 ; // update byte count
            inp = gs.gstr( lineLen ) ;
            if ( lineLen > 1 && inp[lineLen-2] == CR )
               gs.limitCharacters(lineLen-2) ;
            index = ZERO ;                // reference beginning of line
            this->SkipWhiteSpace ( inp, index ) ;   // step over initial whitespace

            //* If end of <head></head> block, then CSS   *
            //* ('C') comments will no longer be counted. *
            if ( (gs.find( L"</head>", index )) >= ZERO )
               after_head = true ;

            //* If we have an on-going HTML-style comment *
            if ( h_comment )
            {
               if ( (i2 = gs.find( L"-->", index )) >= ZERO )
               {  //* Comment is closed. Does code follow the comment? *
                  h_comment = false ;
                  USHORT ix = i2 + 3 ;
                  this->SkipWhiteSpace ( inp, ix ) ; // skip trailing whitespace
                  if ( inp[ix] == nckNULLCHAR )
                     ++cl ;               // entire line is comment
                  else
                     ++ml ;               // mixed code and comment
               }
               else
                  ++cl ;                  // entire line is comment
            }

            //* If we have the beginning of an HTML-style comment *
            else if ( (i2 = gs.find( L"<!--", index )) >= ZERO )
            {
               short i3 = gs.find( L"-->", i2 ) ;
               if ( i2 == index )
               {  //* Comment starts at beginning of line *
                  if ( i3 < ZERO )
                  {  //* Entire line is comment AND comment not closed *
                     ++cl ;
                     h_comment = true ;
                  }
                  else
                  {  //* Comment is closed. Does code follow the comment? *
                     USHORT ix = i3 + 3 ;
                     this->SkipWhiteSpace ( inp, ix ) ; // skip trailing whitespace
                     if ( inp[ix] == nckNULLCHAR )
                        ++cl ;
                     else
                        ++ml ;
                  }
               }
               else
               {  //* Code preceeds beginning of comment *
                  ++ml ;                  // mixed code and comment
                  if ( i3 < ZERO )
                     h_comment = true ;   // comment not closed
               }
            }

            //* If we have an on-going c-style comment *
            else if ( c_comment )
            {
               if ( (i2 = gs.find( L"*/", index )) >= ZERO )
               {  //* Comment is closed. Does code follow the comment? *
                  c_comment = false ;
                  USHORT ix = i2 + 2 ;
                  this->SkipWhiteSpace ( inp, ix ) ; // skip trailing whitespace
                  if ( inp[ix] == nckNULLCHAR )
                     ++cl ;               // entire line is comment
                  else
                     ++ml ;               // mixed code and comment
               }
               else
                  ++cl ;                  // entire line is comment
            }

            //* If we have the beginning of a C-style comment
            else if ( (after_head == false) && (i2 = gs.find( L"/*", index )) >= ZERO )
            {
               short i3 = gs.find( L"*/", i2 ) ;
               if ( i2 == index )
               {  //* Comment starts at beginning of line *
                  if ( i3 < ZERO )
                  {  //* Entire line is comment AND comment not closed *
                     ++cl ;
                     c_comment = true ;
                  }
                  else
                  {  //* Comment is closed. Does code follow the comment? *
                     USHORT ix = i3 + 2 ;
                     this->SkipWhiteSpace ( inp, ix ) ; // skip trailing whitespace
                     if ( inp[ix] == nckNULLCHAR )
                        ++cl ;
                     else
                        ++ml ;
                  }
               }
               else
               {  //* Code preceeds beginning of comment *
                  ++ml ;                  // mixed code and comment
                  if ( i3 < ZERO )
                     c_comment = true ;   // comment not closed
               }
            }

            //* Else presummed source line (or blank line) *
            else
            {
               if ( inp[index] == NULLCHAR )
                  ++bl ;                     // blank line
               else
                  ++sl ;                     // assume a source line
            }
         }  // good
         else                    // end of file (or read error)
         {
            done = true ;
         }
      }     // while()

      ifs.close() ;                 // close the file

      //* If no errors, transfer data to caller *
      if ( success )
      {
         fd.lineCount    = lc ;
         fd.sourceLines  = sl ;
         fd.commentLines = cl ;
         fd.mixedLines   = ml ;
         fd.blankLines   = bl ;
         if ( fd.lineCount > ZERO ) // (avoid divide-by-zero)
         {
            fd.maintIndex  =
               (((double)fd.commentLines + (double)fd.mixedLines 
                 + fd.blankLines) / (double)fd.lineCount * 100.0) ;
         }
         else
            fd.maintIndex  = (double)0.00 ;
      }
   }
   return success ;

}  //* End af_sffHT() *

//*************************
//*    SkipWhiteSpace     *
//*************************
//******************************************************************************
//* Skip over any whitespace placing caller's index at first non-whitespace    *
//* character.                                                                 *
//*                                                                            *
//* Input  : inp  : pointer to a source line of data                           *
//*          index: (by reference) index into current position in data line    *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void SrcProf::SkipWhiteSpace ( const wchar_t* inp, USHORT& index )
{

   while ( inp[index] == SPACE || inp[index] == TAB )
      ++index ;

}  //* End SkipWhiteSpace() *

//*************************
//*   Scan4CommentClose   *
//*************************
//******************************************************************************
//* Beginning of a C-style comment was previously identified. Scan for the     *
//* close-comment sequence: "*/".                                              *
//*                                                                            *
//* Input  : inp  : pointer to a source line of data                           *
//*          index: (by reference) index into current position in data line    *
//*                                                                            *
//* Returns: 'false' if close-comment found, else 'true'                       *
//******************************************************************************

bool SrcProf::Scan4CommentClose ( const wchar_t* inp, USHORT& index )
{
   bool still_open = true ;

   while ( inp[index] != NULLCHAR && still_open != false )
   {
      //* If beginning of string constant or character constant *
      if ( inp[index] == DBLQUOTE || inp[index] == SGLQUOTE )
      {
         //* If we have an apostrophe, ignore it.                    *
         //* If we have a const char DBLQUOTE: '"' OR '"n, step over.*
         if ( inp[index] == SGLQUOTE && inp[index + 1] == DBLQUOTE )
            ++index ;

         else if ( inp[index] == DBLQUOTE )
         {  // NOTE: This is not a definitive scan. Several combinations of 
            //       unfortunate may break it, but it catches most common 
            //       sequences and a few uncommon ones.
            //* Determine whether a pair of double-quote marks on the line.*
            bool pairFound = false, possibleFalseClose = false, actualFalseClose = false ;
            for ( short i = index + 1 ; inp[i] != NULLCHAR ; i++ )
            {
               if ( inp[i] == DBLQUOTE )
               {
                  pairFound = true ;
                  if ( possibleFalseClose )
                     actualFalseClose = true ;
               }
               else if ( (inp[i] == STAR && inp[i + 1] == SLASH) && !pairFound )
               {
                  possibleFalseClose = true ;
               }
            }
            //* Lonely DBLQUOTE, ignore it *
            if ( !pairFound )
            {  /* do nothing */ }

            //* Pair of DBLQUOTEs found, sequencing is everything *
            else  // possible && actual
            {  //* If there is an end-of-comment embedded within the string    *
               //* constant, scan to the end of the constant.                  *
               if ( actualFalseClose )
               {
                  if ( (this->Scan2ConstClose ( inp, index )) != false )
                     break ;     // EOL reached
               }
               //* Else, continue the scan for end-of-comment.        *
               //* We will find it, but there is a DBLQUOTE beyond it *
               //* which may, or may not be code.                     *
               else
               { /* do nothing */ }
            }
         }
      }     // if(inp[index]==DBLQUOTE || inp[index]==SGLQUOTE)

      if ( inp[index] == STAR && inp[index + 1] == SLASH )
      {
         ++index ;
         still_open = false ;
      }
      ++index ;
   }
   return still_open ;

}  //* End Scan4CommentClose() *

//*************************
//*   Scan2SourceClose    *
//*************************
//******************************************************************************
//* Source code has been detected. Scan until comment or end-of-line found.    *
//*                                                                            *
//* Input  : inp  : pointer to a source line of data                           *
//*          index: (by reference) index into current position in data line    *
//*                                                                            *
//* Returns: 'true' if comment follows source code                             *
//*          'false' if end-of-line found                                      *
//******************************************************************************

bool SrcProf::Scan2SourceClose ( const wchar_t* inp, USHORT& index )
{
   bool  more_follows = false ;     // return value

   while ( inp[index] != NULLCHAR )
   {
      //* If beginning of string constant or character constant, step over it. *
      if ( inp[index] == DBLQUOTE || inp[index] == SGLQUOTE )
      {
         if ( (this->Scan2ConstClose ( inp, index )) != false )
            break ;  // EOL reached
      }

      if ( (inp[index] == SLASH) && 
           (inp[index + 1] == SLASH || inp[index + 1] == STAR) )
      {
         more_follows = true ;
         break ;
      }
      ++index ;
   }
   return more_follows ;

}  //* End Scan2SourceClose() *

//*************************
//*    Scan2ConstClose    *
//*************************
//******************************************************************************
//* The beginning of a constant string or constant char has been identified    *
//* and the initial SGLQUOTE or DBLQUOTE character is currently indexed. Scan  *
//* though source the statement on the line until end-of-constant (or EOL).    *
//*                                                                            *
//* Input  : inp  : pointer to a source line of data                           *
//*          index: (by reference) index into current position in data line    *
//*                                                                            *
//* Returns: 'false' if end of constant sequence found (but not EOL)           *
//*          'true' if end-of-line reached (with or without end of constant)   *
//******************************************************************************
//* Programmer's Note: There are potential holes in the analysis which may     *
//* incorrectly identify a comment delimiter '//'  '/*'  '*/'  ';'  '*'        *
//* sequence within a string or character constant.                            *
//* In fact, this error is by far the most common complaint about comment      *
//* strippers; and after all, Source Profiler is the anti-stripper.            *
//*                                                                            *
//* The good news is that we can pretty safely assume two things about string  *
//* and character constants:                                                   *
//*  1) They are bounded by a set of double-quote or single-quote marks.       *
//*  2) They begin and end on the same line.                                   *
//*                                                                            *
//* Thus, in our scan methods, if we encounter a double-quote or single-quote  *
//* mark, we can ignore everything until the matching mark which ends the      *
//* sequence.. The exception would be an escaped mark. Examples:               *
//*    const char* const msg = "Tom often says \"Hello\" to strangers." ;      *
//*    msg  DB "Tom often says \"Hello\" to strangers."                        *
//*    msg  FCC 'Tom often says \"Hello\" to strangers.'                       *
//*    DB "We like your resume; however, we currently have no positions open"  *
//*      (Note the ';' character embedded within the string.)                  *
//* A const char '"' or '\'' should be handled by the same scan.               *
//*                                                                            *
//* This method correctly analyzes Test21.c and Test22.asm, so we have a       *
//* general confidence in it; however, there may be additional scenarios we    *
//* haven't yet identified. For instance, if some language syntax uses a       *
//* single or double-quote mark as an operator (i.e. Visual Basic) OR if some  *
//* unfortunate sequence occurs inside a C comment which either hides the      *
//* end-of-comment sequence or gives a false end-of-comment, we're screwed.    *
//******************************************************************************

bool SrcProf::Scan2ConstClose ( const wchar_t* inp, USHORT& index )
{
   wchar_t  endMark = inp[index] ;  // terminal character
   bool endOfLine = true ;          // return value

   ++index ;                        // step over initial SGLQUOTE or DBLQUOTE
   while ( (inp[index] != NULLCHAR) && (endOfLine != false) )
   {
      //* Skip over escaped end-mark *
      if ( inp[index] == BSLASH && inp[index + 1] == endMark )
         ++index ;

      //* End of constant? *
      else if ( (inp[index] == endMark) && (inp[index + 1] != NULLCHAR) )
         endOfLine = false ;
      ++index ;
   }
   return endOfLine ;

}  //* End Scan2ConstClose() *

//*************************
//*    FormatFilename     *
//*************************
//******************************************************************************
//* Extract the filename from the path/filename provided.                      *
//* Be sure the string fits exactly into the display field.                    *
//*                                                                            *
//* Input  : gsName: (by reference, initial contents ignored)                  *
//*                  receives the extracted filename                           *
//*          fName : filename string                                           *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void SrcProf::FormatFilename ( gString& gsName, const gString& fName )
{
   //* Create the output template *
   const char* padTemplate = "%%-%02hdS" ;
   short padWidth = maxFNAMECOLS + 10 ;
   gString gsTemplate( padTemplate, &padWidth ) ;

   //* Format the filename for display *
   gsName.compose( gsTemplate.gstr(), fName.gstr() ) ;
   gsName.limitCols( maxFNAMECOLS ) ;

}  //* End FormatFilename() *

//*************************
//*    GetFileFamily      *
//*************************
//******************************************************************************
//* Scan the file's path/filename string and isolate the filename extension.   *
//* Determine which source code family the file belongs to.                    *
//*                                                                            *
//* Input  : fPath: pointer to path/filename string                            *
//*                                                                            *
//* Returns: member of sfFamily                                                *
//******************************************************************************

sfFamily SrcProf::GetFileFamily ( const char* fPath )
{
   gString gsExt ;                              // isolate filename extension
   this->spfExtractFileExtension ( gsExt, fPath ) ;
   const wchar_t* extPtr = gsExt.gstr() ;
   sfFamily family = sffUnk ;                   // return value
   if ( gsExt.gschars() > 1 )
   {
      for ( short i = ZERO ; i < sftMAX ; i++ )
      {
         if ( (wcsncasecmp ( extPtr, fnExtensions[i], gsMAXCHARS )) == ZERO )
         {
            switch ( i )
            {
               case sftH:
               case sftC:
               case sftHPP:
               case sftCPP:
               case sftJAVA:
               case sftJS:
               case sftAS:
               case sftM:
               case sftMM:
               case sftCS:
               case sftCSS:
                  family = sffC ;
                  break ;

               case sftASM:
               case sftINC:
               case sftM51:
               case sftS07:
               case sftS33:
                  family = sffA ;
                  break ;

               case sftPY:
               case sftPYW:
               case sftSH:
                  family = sffPY ;
                  break ;

               case sftPL:
               case sftPM:
               case sftT:
                  family = sffPE ;
                  break ;

#if 0    // UNDER CONSTRUCTION
               case sftPHP:
               case sftPHTML:
               case sftPHP3:
               case sftPHP4:
               case sftPHP5:
               case sftPHPS:
                  family = sffPH ;
                  break ;

               case sftRB:
               case sftRBW:
                  family = sffRU ;
                  break ;
#endif   // UNDER CONSTRUCTION

               case sftVB:
               case sftVBS:
                  family = sffVB ;
                  break ;

#if 0    // UNDER CONSTRUCTION
               case sftSQL:
                  family = sffSQ ;
                  break ;
#endif   // UNDER CONSTRUCTION

               case sftTEXI:
               case sftTEXINFO:
                  family = sffTX ;
                  break ;

               case sftHTML:
               case sftHTM:
               case sftXHTML:
               case sftXHT:
               case sftXML:
                  family = sffHT ;
                  break ;

               default:
                  family = sffUnk ;
                  break ;
            }
            break ;
         }
      }
   }

   //* Filename has no extension. Test for shell scripts.*
   else if ( *extPtr == NULLCHAR )
   {
      ifstream ifs ( fPath, ifstream::in ) ;
      if ( ifs.is_open() )             // if input file open
      {
         //* Read first line of file *
         char  tmp[gsMAXBYTES] ; // UTF-8 line input from file
         ifs.getline ( tmp, gsMAXBYTES, NEWLINE ) ;
         if ( ifs.good() || ifs.gcount() > ZERO )
         {
            const wchar_t *Shebang = L"#!",
                          *Bash    = L"/bash",
                          *Bourne  = L"/sh",
                          *Cshell  = L"/csh",
                          *Tshell  = L"/tcsh" ;
            gString gs( tmp ) ;
            if ( ((gs.find( Shebang )) == ZERO) &&
                   (((gs.find( Bash )) >= ZERO) ||
                    ((gs.find( Bourne )) >= ZERO) ||
                    ((gs.find( Cshell )) >= ZERO) ||
                    ((gs.find( Tshell )) >= ZERO)) )
            {
               family = sffPY ;
            }
         }
         ifs.close() ;              // close the file
      }
   }
   return family ;

}  //* GetFileFamily() *

//*************************
//* CreateDisplayStrings  *
//*************************
//******************************************************************************
//* Convert the analytical data to display data.                               *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void SrcProf::CreateDisplayStrings ( void )
{
   gString gsInt, gsName ;
   short ibuffSIZE = FI_MAX_FIELDWIDTH + 1 ;
   char lcBuff[ibuffSIZE], slBuff[ibuffSIZE], clBuff[ibuffSIZE], 
        mlBuff[ibuffSIZE], blBuff[ibuffSIZE] ;

   for ( UINT i = ZERO ; i < this->pd.fileCount ; i++ )
   {
      this->FormatFilename ( gsName, this->pd.fileData[i].fName ) ;
      gsInt.formatInteger( (ULONG)this->pd.fileData[i].lineCount, 5 ) ;
      gsInt.copy( lcBuff, ibuffSIZE ) ;
      gsInt.formatInteger( (ULONG)this->pd.fileData[i].sourceLines, 6 ) ;
      gsInt.copy( slBuff, ibuffSIZE ) ;
      gsInt.formatInteger( (ULONG)this->pd.fileData[i].commentLines, 6 ) ;
      gsInt.copy( clBuff, ibuffSIZE ) ;
      gsInt.formatInteger( (ULONG)this->pd.fileData[i].mixedLines, 6 ) ;
      gsInt.copy( mlBuff, ibuffSIZE ) ;
      gsInt.formatInteger( (ULONG)this->pd.fileData[i].blankLines, 6 ) ;
      gsInt.copy( blBuff, ibuffSIZE ) ;
      this->pd.dispData[i].compose( statTemplate, gsName.gstr(), lcBuff, slBuff, 
               clBuff, mlBuff, blBuff, &this->pd.fileData[i].maintIndex ) ;
   }

   //* Calculate the average maintainability, *
   //* and format the summary data for output.*
   if ( this->pd.tLineCount > ZERO )
      this->pd.tMaintIndex =
         (((double)this->pd.tCommentLines + (double)this->pd.tMixedLines 
           + (double)this->pd.tBlankLines) / (double)this->pd.tLineCount * 100.0) ;
   else
      this->pd.tMaintIndex = 0.0 ; // prevent divide-by-zero

   gsInt.formatInteger( (ULONG)this->pd.tLineCount, 6 ) ;
   gsInt.copy( lcBuff, ibuffSIZE ) ;
   gsInt.formatInteger( (ULONG)this->pd.tSourceLines, 6 ) ;
   gsInt.copy( slBuff, ibuffSIZE ) ;
   gsInt.formatInteger( (ULONG)this->pd.tCommentLines, 6 ) ;
   gsInt.copy( clBuff, ibuffSIZE ) ;
   gsInt.formatInteger( (ULONG)this->pd.tMixedLines, 6 ) ;
   gsInt.copy( mlBuff, ibuffSIZE ) ;
   gsInt.formatInteger( (ULONG)this->pd.tBlankLines, 6 ) ;
   gsInt.copy( blBuff, ibuffSIZE ) ;
   gsName.compose( statSummaryT, &this->pd.fileCount ) ;
   gsName.limitColumns( maxFNAMECOLS - 1 ) ;
   this->pd.summaryData.compose( statTemplate, gsName.gstr(), lcBuff, slBuff, 
                           clBuff, mlBuff, blBuff, &this->pd.tMaintIndex ) ;

}  //* End CreateDisplayStrings() *

//*************************
//*     SortFileData      *
//*************************
//******************************************************************************
//* Sort the array of FileData-class objects according to the specified        *
//* criteria.                                                                  *
//*                                                                            *
//* Input  : none, sort option is controlled by this->pd.fileSort              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void SrcProf::SortFileData ( void )
{
   switch ( this->pd.fileSort )
   {
      case soFilename:     // sort by filename (ascending)
         this->Compare_Filename ( this->pd.fileData, this->pd.fileCount ) ;
         break ;
      case soMaintain:     // sort by maintainability index (ascending)
         this->Compare_Maintainability ( this->pd.fileData, this->pd.fileCount ) ;
         break ;
      case soSourcelines:  // sort by number of source code lines (ascending)
         this->Compare_Sourcelines ( this->pd.fileData, this->pd.fileCount ) ;
         break ;
      case soExtension:    // sort by filename extension (ascending)
      default:             //  sub-sort by filename
         this->Compare_Extension ( this->pd.fileData, this->pd.fileCount ) ;
         break ;
   }
}  //* End SortFileData() *

//*************************
//*   Compare_Extension   *
//*************************
//******************************************************************************
//* Sort data by filename extension (ascending), then sub-sort by filename.    *
//*                                                                            *
//* Input  : fdPtr  : pointer to an array of FileData-class objects            *
//*          fdCount: number of elements in the array                          *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void SrcProf::Compare_Extension ( FileData* fdPtr, UINT fdCount )
{
   if ( fdCount > 1 )               // if there is something to do
   {
      gString nameA, nameB ;        // compare names
      FileData tmp ;                // swap buffer
      UINT indexA = ZERO,           // upper index
           indexB = 1,              // lower index
           lastIndex = fdCount - 1 ;// index last element in the array

      //* Primary sort by filename extension *
      while ( indexA < lastIndex )
      {
         for ( indexB = indexA + 1 ; indexB <= lastIndex ; indexB++ )
         {
            this->spfExtractFileExtension ( nameA, fdPtr[indexA].fName ) ;
            this->spfExtractFileExtension ( nameB, fdPtr[indexB].fName ) ;
            if ( (wcsncasecmp ( nameB.gstr(), nameA.gstr(), gsMAXCHARS )) < ZERO )
            {
               tmp = fdPtr[indexA] ;
               fdPtr[indexA] = fdPtr[indexB] ;
               fdPtr[indexB] = tmp ;
            }            
         }
         ++indexA ;
      }

      //* Sub-sort by filename *
      indexA = indexB = ZERO ;
      int   diff, comp ;
      while ( indexA < lastIndex )
      {  //* Delimit a block of filenames with the same extension *
         this->spfExtractFileExtension ( nameA, fdPtr[indexA].fName ) ;
         do
         {
            if ( ++indexB > lastIndex )
               break ;
            this->spfExtractFileExtension ( nameB, fdPtr[indexB].fName ) ;
            comp = wcsncasecmp ( nameB.gstr(), nameA.gstr(), gsMAXCHARS ) ;
         }
         while ( comp == ZERO ) ;

         //* Within the block with same extension, sort by filename *
         diff = indexB - indexA ;
         if ( diff > 1 )
         {
            this->Compare_Filename ( &fdPtr[indexA], diff ) ;
         }
         indexA = indexB ;    // reference first entry in next block
      }
   }
}  //* End Compare_Extension() *

//*************************
//*   Compare_Filename    *
//*************************
//******************************************************************************
//* Sort data by filename (ascending).                                         *
//*                                                                            *
//* Input  : fdPtr  : pointer to an array of FileData-class objects            *
//*          fdCount: number of elements in the array                          *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void SrcProf::Compare_Filename ( FileData* fdPtr, UINT fdCount )
{
   if ( fdCount > 1 )               // if there is something to do
   {
      FileData tmp ;                // swap buffer
      UINT indexA = ZERO,           // upper index
           indexB = 1,              // lower index
           lastIndex = fdCount - 1 ;// index last element in the array

      while ( indexA < lastIndex )
      {
         for ( indexB = indexA + 1 ; indexB <= lastIndex ; indexB++ )
         {
            if ( (wcsncasecmp ( fdPtr[indexB].fName.gstr(), fdPtr[indexA].fName.gstr(), gsMAXCHARS )) < ZERO )
            {
               tmp = fdPtr[indexA] ;
               fdPtr[indexA] = fdPtr[indexB] ;
               fdPtr[indexB] = tmp ;
            }            
         }
         ++indexA ;
      }
   }
}  //* End Compare_Filename() *

//***************************
//* Compare_Maintainability *
//***************************
//******************************************************************************
//* Sort data by maintainability index (ascending).                            *
//*                                                                            *
//* Input  : fdPtr  : pointer to an array of FileData-class objects            *
//*          fdCount: number of elements in the array                          *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void SrcProf::Compare_Maintainability ( FileData* fdPtr, UINT fdCount )
{
   if ( fdCount > 1 )               // if there is something to do
   {
      FileData tmp ;                // swap buffer
      UINT indexA = ZERO,           // upper index
           indexB = 1,              // lower index
           lastIndex = fdCount - 1 ;// index last element in the array

      while ( indexA < lastIndex )
      {
         for ( indexB = indexA + 1 ; indexB <= lastIndex ; indexB++ )
         {
            if ( fdPtr[indexB].maintIndex < fdPtr[indexA].maintIndex )
            {
               tmp = fdPtr[indexA] ;
               fdPtr[indexA] = fdPtr[indexB] ;
               fdPtr[indexB] = tmp ;
            }            
         }
         ++indexA ;
      }
   }
}  //* End Compare_Maintainability() *

//*************************
//*  Compare_Sourcelines  *
//*************************
//******************************************************************************
//* Sort data by number of source code lines (ascending).                      *
//*                                                                            *
//* Input  : fdPtr  : pointer to an array of FileData-class objects            *
//*          fdCount: number of elements in the array                          *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void SrcProf::Compare_Sourcelines ( FileData* fdPtr, UINT fdCount )
{
   if ( fdCount > 1 )               // if there is something to do
   {
      FileData tmp ;                // swap buffer
      UINT indexA = ZERO,           // upper index
           indexB = 1,              // lower index
           lastIndex = fdCount - 1 ;// index last element in the array

      while ( indexA < lastIndex )
      {
         for ( indexB = indexA + 1 ; indexB <= lastIndex ; indexB++ )
         {
            if ( fdPtr[indexB].sourceLines < fdPtr[indexA].sourceLines )
            {
               tmp = fdPtr[indexA] ;
               fdPtr[indexA] = fdPtr[indexB] ;
               fdPtr[indexB] = tmp ;
            }            
         }
         ++indexA ;
      }
   }
}  //* End Compare_Sourcelines() *

//*************************
//*                       *
//*************************
//******************************************************************************
//*                                                                            *
//*                                                                            *
//*                                                                            *
//* Input  :                                                                   *
//*                                                                            *
//* Returns:                                                                   *
//******************************************************************************

