//******************************************************************************
//* File       : NCurses.hpp                                                   *
//* Author     : Mahlon R. Smith                                               *
//*              Copyright (c) 2005-2015 Mahlon R. Smith, The Software Samurai *
//*                 GNU GPL copyright notice below                             *
//* Date       : 05-Jul-2015                                                   *
//* Version    : (see NCursesVersion string in NCurses.cpp)                    *
//*                                                                            *
//* Description: This class implements a wrapper for the ncurses environment,  *
//* allowing a more easily understood interface and simplifying access to      *
//* many of the more obscure elements of the ncurses engine.                   *
//*                                                                            *
//*                                                                            *
//* Development Tools: See NcDialog.cpp.                                       *
//******************************************************************************
//* Version History:                                                           *
//*   See version history in NCurses.cpp                                       *
//*                                                                            *
//******************************************************************************
//* Copyright Notice:                                                          *
//* This program is free software: you can redistribute it and/or modify it    *
//* under the terms of the GNU General Public License as published by the Free *
//* Software Foundation, either version 3 of the License, or (at your option)  *
//* any later version, PROVIDED THAT the copyright notices for both code and   *
//* documentation are included and unmodified.                                 *
//*                                                                            *
//* This program is distributed in the hope that it will be useful, but        *
//* WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY *
//* or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License   *
//* for more details.                                                          *
//*                                                                            *
//* You should have received a copy of the GNU General Public License along    *
//* with this program.  If not, see <http://www.gnu.org/licenses/>.            *
//*                                                                            *
//*         Full text of the GPL License may be found in the Texinfo           *
//*         documentation for this program under 'Copyright Notice'.           *
//******************************************************************************
//* Programmer's Notes:                                                        *
//* - In NCurses world, OK == 0 and ERR == (-1)                                *
//*                     TRUE == 1 and FALSE == 0                               *
//*                                                                            *
//* - IMPORTANT NOTE: The ncurses C-language library is not thread safe;       *
//*   thus, the NCurses class is not thread safe.                              *
//*   Because applications will seldom, if ever, use NCurses-class I/O routines*
//*   directly, this is not a major problem. Just be aware.                    *
//*                                                                            *
//******************************************************************************

#ifndef NCURSES_INCLUDED
#define NCURSES_INCLUDED

//* This definition is necessary in order for the ncurses.h macros to be       *
//* configured properly for 'wide' character output.                           *
#define _X_OPEN_SOURCE_EXTENDED
#include <ncurses.h>          //* definition file for ncurses library

//** Definitiions**

//***********************
//* Keycode Definitions *
//***********************
#include "NCursesKeyDef.hpp"


//* Return values for the GetKeyInput() method.                                *
//* See GetKeyInput() method prototype, below for an explanation of each code. *
enum wkeyType : short { wktPRINT = OK, wktFUNKEY = KEY_CODE_YES, 
                        wktMOUSE = KEY_MOUSE, wktEXTEND = 'X',
                        wktESCSEQ = nckESC, wktERR = ERR } ;

//* Special mouse-event mask definition for detecting Scroll-Wheel events.     *
//* This cannot be guaranteed for all systems, but on the primary test system, *
//* Scroll-down == BUTTON4_PRESSED and Scroll-up == REPORT_MOUSE_POSITION      *
const long int REPORT_SCROLL_WHEEL = BUTTON4_PRESSED | REPORT_MOUSE_POSITION ;

//* Special values for setting the maximum interval for receipt of a series of *
//* mouse press/release events for which a mouse-click event will be reported. *
//*          See SetMouseClickInterval() method for more information.          *
enum ncmInterval : short 
{
   ncmiREPORT = (-1),   // used internally, and ONLY by GetMouseClickInterval() method
   ncmiNONE   = ZERO,   // disable interval-resolution checking
   ncmiDFLT   = 166,    // default interval: 166/1000 or 1/6th second
                        // (ncurses docs incorrectly identify this as the 'maximum')
   ncmiSTABLE = 220,    // for NcDialog 'stable' mode: default double-click interval
   ncmiMAX    = 30000   // maximum interval: 30 seconds
                        // (arbitrarily chosen as a large, positive, 16-bit value)
} ;
//* This class is similar to the 'MEVENT' structure defined in ncurses.h;      *
//* but conforms to the NCurses-class data-types for cursor positioning, and   *
//* provides for translation from screen-relative to window-relative position. *
class MouseEvent
{
   public:
   MouseEvent ( void )
   { this->reset() ; }
   mmask_t  eventType ; // this is the event-type format used by the ncurses library
   short    deviceID ;  // id of mouse or mouse-like device (for multiple devices)
   //* Screen-relative group (relative to ACTIVE portion of terminal window)
   short    sypos ;
   short    sxpos ;
   short    szpos ;  // for touchpads, not currently supported by ncurses library

   //* Window-relative group                                     *
   //* (used by NcWindow class, not referenced by NCurses class) *
   short    ypos ;
   short    xpos ;
   short    zpos ;   // for touchpads, not currently supported by ncurses library
   short    cIndex ; // NcDialog-class control index
   short    meType ; // NcDialog-class event-type (member of enum meTypes)
   bool     cKey ;   // 'control' key down during event
   bool     sKey ;   // 'shift'   key down during event
   bool     aKey ;   // 'alt'     key down during event
   bool     mPos ;   // 'report-mouse-position' bit
   bool     conv ;   // 'true' if event data have been converted to keycode data

   private:
   void reset ( void )
   {
      this->sypos = this->sxpos = this->szpos = this->deviceID = ZERO ;
      this->eventType = ZERO ;

      this->ypos = this->xpos = this->zpos = ZERO ;
      this->cIndex = -1 ;
      this->meType = ZERO ;
      this->cKey = this->sKey = this->aKey = this->mPos = this->conv = false ;
   }
   friend class NCurses ;
} ;

//* This class is for passing the information returned from the GetKeyInput()  *
//* method. Key input data consists of a keycode TYPE (member of enum wkeyType)*
//* (see above), as well as the actual keycode. If there is any possibility of *
//* ambiguity in interpretation when passing a keycode between application     *
//* methods, pass both pieces of information using this class.                 *
class wkeyCode
{
   public:
   //* Default constructor *
   wkeyCode ( void ) { key = nckNULLCHAR ; type = wktERR ; }
   //* Initialization constructor *
   wkeyCode ( const wchar_t kc, const wkeyType kt ) : key(kc), type(kt) {}
   //* Compare member data *
   bool operator== ( const wkeyCode& wk2 ) const
   {
      return ( (this->key == wk2.key) && (this->type == wk2.type) ) ;
   }
   bool operator!= ( const wkeyCode& wk2 ) const
   {
      return ( (this->key != wk2.key) || (this->type != wk2.type) ) ;
   }

   //* Public data members *
   wchar_t   key ;         // value assigned to an individual keypress
   wkeyType  type ;        // classification of keycode
   MouseEvent mevent ;     // if 'type' == wktMOUSE, then this receives the
                           // mouse event data, else unitialized
} ;

//* Key input pushback queue. This is a FIFO queue.                            *
const short KEYPUSH_MAX = 16 ;      //* Maximum consecutive pushes without a pull
class keyPushback
{
   public:
   keyPushback ( void ) { kCount = ZERO ; }
   wkeyCode    kPush[KEYPUSH_MAX] ; 
   short       kCount ;             //* Number of keycodes in pushback queue
} ;

//* Bitmasks for available color attributes.                                   *
//* These color-attribute modifier bits may be used to MODIFY a color based on *
//* the color constants defined as public members of the NCurses class.        *
//*  Example:  attr_t mycolor = nc.reR | ncuATTR ;                             *
//*           'mycolor' combines the base color constant nc.reR (red text with *
//*           'Reverse' attribute) with the ncuATTR ('Underline') modifier bit.*
//* Note: The ncurses library defines all of the following attribute           *
//*       modifiers; however, your terminal or terminal emulator may not       *
//*       support the use of all these attributes. In general emulators do     *
//*       not support blinking text or the last six attribute modifiers, which *
//*       were used only on some dedicated (dumb) terminals.                   *
const attr_t ncnATTR = A_NORMAL ;         // 0x00000000  NORMAL bit mask
const attr_t ncsATTR = A_STANDOUT ;       // 0x00010000  STANDOUT bit mask
const attr_t ncuATTR = A_UNDERLINE ;      // 0x00020000  UNDERLINE bit mask
const attr_t ncrATTR = A_REVERSE ;        // 0x00040000  REVERSE bit mask
const attr_t nckATTR = A_BLINK ;          // 0x00080000  BLINK bit mask (not used)
const attr_t ncdATTR = A_DIM ;            // 0x00100000  DIM bit mask
const attr_t ncbATTR = A_BOLD ;           // 0x00200000  BOLD bit mask
const attr_t ncaATTR = A_ALTCHARSET ;     // 0x00600000  ALT CHAR SET bit mask
const attr_t ncgATTR = ncrATTR | ncbATTR ;// 0x00C00000  REVERSE+BOLD bit mask
const attr_t nciATTR = A_INVIS ;          // 0x00800000  INVISIBLE bit mask
const attr_t ncpATTR = A_PROTECT ;        // 0x01000000  PROTECT bit mask (UNSUPPORTED)
const attr_t nchATTR = A_HORIZONTAL ;     // 0x02000000  HORIZONTAL bit mask (UNSUPPORTED)
const attr_t nclATTR = A_LEFT ;           // 0x04000000  LEFT (UNSUPPORTED)
const attr_t ncwATTR = A_LOW ;            // 0x08000000  LOW (UNSUPPORTED)
const attr_t ncqATTR = A_RIGHT ;          // 0x10000000  RIGHT (UNSUPPORTED)
const attr_t nctATTR = A_TOP ;            // 0x20000000  TOP (UNSUPPORTED)
//* For completeness,the remaining bits of the attr_t type are *
//* shown below, but these masks have little practical value.  *
const attr_t ncmCHAR_MASK = 0x000000FF ;  // 'narrow' character code (not used)
const attr_t ncmPAIR_MASK = 0x0000FF00 ;  // foreground/background 'color pair' (256 pairs)
const attr_t ncmATTR_MASK = 0x00FF0000 ;  // color attribute bits (only the supported bits)

//* Notes on color mapping:
//*    "BK" indicates BLACK, "RE" indicates RED, "GR" indicates GREEN, "BR" indicates BROWN,
//*    "BL" indicates BLUE, "MA" = MAGENTA, "CY" indicates CYAN, "GY" == GRAY, 
//*    ("BW" indicates BrightWhite)
//* See also color pair variables below.

//* Basic choices for foreground OR background color                           *
//* (used in calls to init_pair())                                             *
//* The definition, ncbcBW denotes the default background at the time the      *
//* NCurses  engine is started (usually Bright White).                         *
//* These definitions equate our NCurses class defines with the ncurses        *
//* library defines.                                                           *
enum NcBaseColors : short
{
   ncbcDEFAULT = ( -1),       // signal to ncurses library to use defaults
   ncbcBW = ncbcDEFAULT,      // standard white background (ofen remapped by terminal)
   ncbcBK = COLOR_BLACK,      // black background
   ncbcRE = COLOR_RED,        // red background
   ncbcGR = COLOR_GREEN,      // green background
   ncbcBR = COLOR_YELLOW,     // brown background
   ncbcBL = COLOR_BLUE,       // blue background
   ncbcMA = COLOR_MAGENTA,    // magenta background
   ncbcCY = COLOR_CYAN,       // cyan background
   ncbcGY = COLOR_WHITE,      // grey background
   ncbcCOLORS                 // ncurses library's limit of support
} ;

//* Extended terminal foreground OR background colors. For terminals that      *
//* support more than the basic eight (8) colors provided by the ncurses       *
//* library. If available, these are generally 'bright' or 'intense' versions  *
//* of the eight base colors.                                                  *
enum NcExtendedColors : short
{
   ncecBK = ncbcCOLORS,       // bright black
   ncecRE,                    // bright red
   ncecGR,                    // bright green
   ncecBR,                    // bright brown (yellow)
   ncecBL,                    // bright blue
   ncecMA,                    // bright magenta
   ncecCY,                    // bright cyan
   ncecGY,                    // bright grey
   ncecCOLORS                 // NCurses class limit of support
} ;

//* Parameter for TerminalColorSupport() method. *
//* Get terminal's color capabilities.           *
class termColorInfo
{
   public:
   short term_colors ;        // number of colors supported
   short rgb_regs ;           // number of RGB register sets supported (16 max)
   short fgbg_pairs ;         // number of foreground/background color pairs supported
   bool  mod_pairs ;          // 'true' if terminal supports color-pair modification
   bool  mod_rgb ;            // 'true' if terminal supports RGB register modification
} ;

//* Color-mapping parameter for calls to StartNCursesEngine(), *
//* StartColorEngine(), GetColorMap() and SetColorMap()        *
const short ncPAIR_MAX = 256 ;  // Maximum foreground/background color pairs
const short ncCOLOR_MAX = 16 ;  // Maximum RGB (virtual) register sets
const short ncSUPPORTED_PAIRS = 128 ; // Maximum color pairs supported by NCurses class
const short minRGBvalue = ZERO ;// Minimum setting for an RGB register
const short maxRGBvalue = 1000 ;// Maximum setting for an RGB register
const short dfltRGBvalue = 680 ;// Default setting for an RGB register if data out of range
class NcColorPair { public: NcBaseColors fgnd ; NcBaseColors bkgnd ; } ; // Color-pair class
class NcRgbSet { public: short r ; short g ; short b ; } ; // RGB set class
class NcColorMap
{
   public:
   //* Initialization constructor *
   NcColorMap ( NcBaseColors bkgnd = ncbcDEFAULT )
   {
      this->bkgndColor = ncbcDEFAULT ;
      if ( bkgnd >= ncbcBW && bkgnd < ncbcCOLORS )
         this->bkgndColor = bkgnd ;
      for ( short i = ZERO ; i < ncPAIR_MAX ; i++ )
         this->pairMap[i].fgnd = this->pairMap[i].bkgnd = ncbcDEFAULT ;
      for ( short i = ZERO ; i < ncCOLOR_MAX ; i++ )
         this->rgbMap[i].r = this->rgbMap[i].g = this->rgbMap[i].b = ZERO ;
      this->pairCount = this->rgbCount = this->supColors = ZERO ;
      this->canModifyPairs = this->canModifyRGB = false ;
   }
   //* Data Members *
   NcColorPair pairMap[ncPAIR_MAX] ;//* Foreground/background color-pairs
   NcRgbSet    rgbMap[ncCOLOR_MAX] ;//* R-G-B register values (virtualized)
   NcBaseColors bkgndColor ;     //* Default backgroud for all foregrounds
   short pairCount ;             //* Number of color pairs supported by terminal
   short rgbCount ;              //* Number of RGB registers supported by terminal
   short supColors ;             //* Number of colors supported by terminal (info only)
   bool  canModifyPairs ;        //* 'true' if terminal supports color-pair modification
   bool  canModifyRGB ;          //* 'true' if terminal supports RGB register modification
} ;

//* Cursor visibility state *
enum ncCurVis : short { nccvINVISIBLE, nccvNORMAL, nccvBOLD } ;

//* Alternate Character Set (alt font) characters.                             *
//* NOTE: USE OF THE ACS IS NOT RECOMMENDED.                                   *
//* Draw these characters by setting the ncaATTR bit in the color attribute    *
//* value. To see these values displayed, please refer to the methods:         *
//* NCurses::DisplayAlternateFont()  or  NcDialog::DisplayAltCharacterSet().   *
//* These methods are incorporated into the Dialog2 test application,          *
//* Tests #9 and #10.                                                          *
//* The UNIX/Linux ACS characters were implemented as a means of extending the *
//* 8-bit character set, but have limited use in modern systems based on UTF-8 *
//* character encoding. It is recommended that the 'wide' equivalents to these *
//* characters be used instead (see wcsXXXX character definitions, below).     *
//* Although the acsXXXX characters can be promoted to wchar_t characters for  *
//* use in 'wide character' strings, it is often more convenient to use the    *
//* wcsXXXX characters rather than the alternate character set because they    *.                                                   *
//* DO NOT require the ncaATTR color attribute.                                *
//* Programmer's Note: In order to define these characters as constants, we    *
//*  have had to compromise. Because the ncurses definitions for these         *
//*  characters are not available until the NCurses engine has been started,   *
//*  we have compromised by assigning the values the ncurses code WOULD HAVE   *
//*  assigned if it had been awake at this point.                              *
const char acsSPACE   = (SPACE) ;         // space (blank)
const char acsRARROW  = ('+') ;           // arrow pointng right
const char acsLARROW  = (',') ;           // arrow pointng left
const char acsUARROW  = ('-') ;           // arrow pointng up
const char acsDARROW  = ('.') ;           // arrow pointng down
const char acsBLOCK   = ('0') ;           // solid block
const char acsDIAMOND = ('`') ;           // diamond
const char acsPATTERN = ('a') ;           // patterned block
const char acsDEGREE  = ('f') ;           // degree symbol
const char acsPLMINUS = ('g') ;           // plus-and-minus symbol
const char acsBOARD   = ('h') ;           // pattern of squares
const char acsLANTERN = ('i') ;           // [unknown symbol]
const char acsLR      = ('j') ;           // drawing character - lower right corner
const char acsUR      = ('k') ;           // drawing character - upper right corner
const char acsUL      = ('l') ;           // drawing character - upper left corner
const char acsLL      = ('m') ;           // drawing character - lower left corner
const char acsINSECT  = ('n') ;           // drawing character - line intersection
const char acsSCAN1   = ('o') ;           // horizontal scanline #1
const char acsSCAN3   = ('p') ;           // horizontal scanline #3
const char acsHORIZ   = ('q') ;           // drawing character - horizontal line
const char acsSCAN7   = ('r') ;           // horizontal scanline #7
const char acsSCAN9   = ('s') ;           // horizontal scanline #9
const char acsLTEE    = ('t') ;           // drawing character - T for left-side connect
const char acsRTEE    = ('u') ;           // drawing character - T for right-edge connect
const char acsBTEE    = ('v') ;           // drawing character - T for bottom-edge connect
const char acsTTEE    = ('w') ;           // drawing character - T for top-edge connect
const char acsVERT    = ('x') ;           // drawing character - vertical line
const char acsLEQUAL  = ('y') ;           // less-than-or-equal symbol
const char acsGEQUAL  = ('z') ;           // greater-than-or equal symbol
const char acsPI      = ('{') ;           // greek letter Pi
const char acsNEQUAL  = ('|') ;           // not-equal-to symbol
const char acsSTRLING = ('}') ;           // U.K Pound Sterling symbol
const char acsBULLET  = ('~') ;           // bullet mark (centered dot)

//* 'WIDE' Alternate Character Set characters.                                 *
//* The following character definitions are the 'wide' (multi-byte) equivalents*
//* of the 'acsXXXX' 8-bit Alt Character Set (ACS) characters defined above.   *
//*                                                                            *
//* The wcsXXXX group of character definitions are the wchar_t character codes *
//* that correspond to the UTF-8 character encoding for line-drawing characters*
//* and other miscellaneous characters found in the Linux/UNIX ACS character   *
//* set. To see these values displayed, please refer to the NcDialog methods:  *
//* DisplayWideCharacterSet() and DisplayLineDrawingSet(). These methods       *
//* are incorporated into the Dialog2 test application, Tests #8 and #10.      *
//*                                                                            *
//* Important Note:                                                            *
//* These characters DO NOT require the use of the ncaATTR color attribute bit.*
//*                                                                            *
//* Programmer's Note: As with the acsXXXX character group above, We really    *
//* want to define these as constants, AND we want to use the ncurses.h        *
//* definitions for the WACS characters; however, defining constants based on  *
//* as-yet-unitialized ncurses pointer-to-cchar_t-structure values yields      *
//* garbage. Example:  const wchar_t wcsLR = WACS_LRCORNER->chars[0] ;         *
//* On the other hand, using #define's is not very C++ cool.                   *
//* Therefore, we have compromised by assigning values the ncurses code        *
//* WOULD HAVE assigned if it had been awake at this point.                    *
const wchar_t wcsSPACE   = (SPACE) ;      // space (blank)
const wchar_t wcsRARROW  = 0x002192 ;     // arrow pointng right
const wchar_t wcsLARROW  = 0x002190 ;     // arrow pointng left
const wchar_t wcsUARROW  = 0x002191 ;     // arrow pointng up
const wchar_t wcsDARROW  = 0x002193 ;     // arrow pointng down
const wchar_t wcsBLOCK   = 0x0025AE ;     // solid block
const wchar_t wcsDIAMOND = 0x0025C6 ;     // diamond
const wchar_t wcsPATTERN = 0x002592 ;     // patterned block
const wchar_t wcsDEGREE  = 0x0000B0 ;     // degree symbol
const wchar_t wcsPLMINUS = 0x0000B1 ;     // plus-and-minus symbol
const wchar_t wcsBOARD   = 0x002592 ;     // pattern of squares
const wchar_t wcsLANTERN = 0x002603 ;     // light bulb
const wchar_t wcsLR      = 0x002518 ;     // drawing character - lower right corner
const wchar_t wcsUR      = 0x002510 ;     // drawing character - upper right corner
const wchar_t wcsUL      = 0x00250C ;     // drawing character - upper left corner
const wchar_t wcsLL      = 0x002514 ;     // drawing character - lower left corner
const wchar_t wcsINSECT  = 0x00253C ;     // drawing character - line intersection
const wchar_t wcsSCAN1   = 0x0023BA ;     // horizontal scanline #1
const wchar_t wcsSCAN3   = 0x0023BB ;     // horizontal scanline #3
const wchar_t wcsHORIZ   = 0x002500 ;     // drawing character - horizontal line
const wchar_t wcsSCAN7   = 0x0023BC ;     // horizontal scanline #7
const wchar_t wcsSCAN9   = 0x0023BD ;     // horizontal scanline #9
const wchar_t wcsLTEE    = 0x00251C ;     // drawing character - T for left-side connect
const wchar_t wcsRTEE    = 0x002524 ;     // drawing character - T for right-edge connect
const wchar_t wcsBTEE    = 0x002534 ;     // drawing character - T for bottom-edge connect
const wchar_t wcsTTEE    = 0x00252C ;     // drawing character - T for top-edge connect
const wchar_t wcsVERT    = 0x002502 ;     // drawing character - vertical line
const wchar_t wcsLEQUAL  = 0x002264 ;     // less-than-or-equal symbol
const wchar_t wcsGEQUAL  = 0x002265 ;     // greater-than-or equal symbol
const wchar_t wcsPI      = 0x0003C0 ;     // greek letter Pi
const wchar_t wcsNEQUAL  = 0x002260 ;     // not-equal-to symbol
const wchar_t wcsSTRLING = 0x0000A3 ;     // U.K Pound Sterling symbol
const wchar_t wcsBULLET  = 0x0000B7 ;     // bullet mark (centered dot)
//* Extra line-drawing characters defined in the Unicode character set.        *
const wchar_t wcsHORIZs  = wcsHORIZ ;  // single-line norm - horizontal line
const wchar_t wcsHORIZb  = 0x002501 ;  // single-line bold - horizontal line
const wchar_t wcsHORIZd  = 0x002550 ;  // dual-line        - horizontal line
const wchar_t wcsVERTs   = wcsVERT  ;  // single-line norm - vertical line
const wchar_t wcsVERTb   = 0x002503 ;  // single-line bold - vertical line
const wchar_t wcsVERTd   = 0x002551 ;  // dual-line        - vertical line
const wchar_t wcsULs     = wcsUL ;     // single-line norm - upper left corner 
const wchar_t wcsULb     = 0x00250F ;  // single-line bold - upper left corner
const wchar_t wcsULd     = 0x002554 ;  // dual-line        - upper left corner
const wchar_t wcsURs     = wcsUR ;     // single-line norm - upper right corner 
const wchar_t wcsURb     = 0x002513 ;  // single-line bold - upper right corner
const wchar_t wcsURd     = 0x002557 ;  // dual-line        - upper right corner
const wchar_t wcsLLs     = wcsLL ;     // single-line norm - lower left corner 
const wchar_t wcsLLb     = 0x002517 ;  // single-line bold - lower left corner
const wchar_t wcsLLd     = 0x00255A ;  // dual-line        - lower left corner
const wchar_t wcsLRs     = wcsLR ;     // single-line norm - lower right corner 
const wchar_t wcsLRb     = 0x00251B ;  // single-line bold - lower right corner
const wchar_t wcsLRd     = 0x00255D ;  // dual-line        - lower right corner
const wchar_t wcsLTEEs   = wcsLTEE  ;  // single-line norm - T for left-edge connect
const wchar_t wcsLTEEb   = 0x002523 ;  // single-line bold - T for left-edge connect
const wchar_t wcsLTEEd   = 0x002560 ;  // dual-line        - T for left-edge connect
const wchar_t wcsRTEEs   = wcsRTEE  ;  // single-line norm - T for right-edge connect
const wchar_t wcsRTEEb   = 0x00252B ;  // single-line bold - T for right-edge connect
const wchar_t wcsRTEEd   = 0x002563 ;  // dual-line        - T for right-edge connect
const wchar_t wcsBTEEs   = wcsBTEE  ;  // single-line norm - T for bottom-edge connect
const wchar_t wcsBTEEb   = 0x00253B ;  // single-line bold - T for bottom-edge connect
const wchar_t wcsBTEEd   = 0x002569 ;  // dual-line        - T for bottom-edge connect
const wchar_t wcsTTEEs   = wcsTTEE  ;  // single-line norm - T for top-edge connect
const wchar_t wcsTTEEb   = 0x002533 ;  // single-line bold - T for top-edge connect
const wchar_t wcsTTEEd   = 0x002566 ;  // dual-line        - T for top-edge connect
const wchar_t wcsINSECTs = wcsINSECT;  // single-line norm - line intersection
const wchar_t wcsINSECTb = 0x00254B ;  // single-line bold - line intersection
const wchar_t wcsINSECTd = 0x00256C ;  // dual-line        - line intersection
//* Dash (dotted) lines *
const wchar_t wcsHDASH2s = 0x00254C ;  // norm - 2-dash horizontal line
const wchar_t wcsHDASH2b = 0x00254D ;  // bold - 2-dash horizontal line
const wchar_t wcsVDASH2s = 0x00254E ;  // norm - 2-dash vertical line
const wchar_t wcsVDASH2b = 0x00254F ;  // bold - 2-dash vertical line
const wchar_t wcsHDASH3s = 0x002504 ;  // norm - 3-dash horizontal line
const wchar_t wcsHDASH3b = 0x002505 ;  // bold - 3-dash horizontal line
const wchar_t wcsVDASH3s = 0x002506 ;  // norm - 3-dash vertical line
const wchar_t wcsVDASH3b = 0x002507 ;  // bold - 3-dash vertical line
const wchar_t wcsHDASH4s = 0x002508 ;  // norm - 4-dash horizontal line
const wchar_t wcsHDASH4b = 0x002509 ;  // bold - 4-dash horizontal line
const wchar_t wcsVDASH4s = 0x00250A ;  // norm - 4-dash vertical line
const wchar_t wcsVDASH4b = 0x00250B ;  // bold - 4-dash vertical line

//* Miscellaneous 'special characters defined in the Unicode character set.    *
//* There are many additional text-mode drawing characters defined by the      *
//* Unicode standard. We define only a subset of them here. For a complete list*
//* visit the Unicode Consortium at  http://www.unicode.org - specifically the *
//* listings for 0x2500-0x257f and 0x2580-0x259F.                              *
const wchar_t wcsLTEEbv  = 0x002520 ;  // T for left-edge connect - bold vert. norm hori.
const wchar_t wcsLTEEbh  = 0x00251D ;  // T for left-edge connect - bold horiz. norm vert.
const wchar_t wcsLTEEdv  = 0x00255F ;  // T for left-edge connect - dual vert. single horiz.
const wchar_t wcsLTEEdh  = 0x00255E ;  // T for left-edge connect - dual horiz. single vert.

const wchar_t wcsRTEEbv  = 0x002528 ;  // T for right-edge connect - bold vert. norm hori.
const wchar_t wcsRTEEbh  = 0x002525 ;  // T for right-edge connect - bold horiz. norm vert.
const wchar_t wcsRTEEdv  = 0x002562 ;  // T for right-edge connect - dual vert. single horiz.
const wchar_t wcsRTEEdh  = 0x002561 ;  // T for right-edge connect - single vert. dual horiz.

const wchar_t wcsBTEEbv  = 0x002538 ;  // T for bottom-edge connect - bold vert. norm hori.
const wchar_t wcsBTEEbh  = 0x002537 ;  // T for bottom-edge connect - bold horiz. norm vert.
const wchar_t wcsBTEEdv  = 0x002568 ;  // T for bottom-edge connect - dual vert. single horiz.
const wchar_t wcsBTEEdh  = 0x002567 ;  // T for bottom-edge connect - single vert. dual horiz.

const wchar_t wcsTTEEbv  = 0x002530 ;  // T for top-edge connect - bold vert. norm hori.
const wchar_t wcsTTEEbh  = 0x00252F ;  // T for top-edge connect - bold horiz. norm vert.
const wchar_t wcsTTEEdv  = 0x002565 ;  // T for top-edge connect - dual vert. single horiz.
const wchar_t wcsTTEEdh  = 0x002564 ;  // T for top-edge connect - single vert. dual horiz.

const wchar_t wcsINSECTbv= 0x002542 ;  // line intersection - bold vert. norm horiz. 
const wchar_t wcsINSECTbh= 0x00253F ;  // line intersection - bold horiz. norm vert.
const wchar_t wcsINSECTdv= 0x00256B ;  // line intersection - dual vert. single horiz.
const wchar_t wcsINSECTdh= 0x00256A ;  // line intersection - single vert. dual horiz.
const wchar_t wcsBLOCKl  = 0x002591 ;  // block character - light pattern
const wchar_t wcsBLOCKm  = 0x002592 ;  // block character - medium pattern
const wchar_t wcsBLOCKd  = 0x002593 ;  // block character - dark pattern
const wchar_t wcsBLOCKs  = 0x002588 ;  // block character - solid black

//* This class implements an Y/X coordinate pair for use in positioning  *
//* objects (or the cursor) on the screen. The functionality supports    *
//* the initialization of arrays of certain class objects under the      *
//* C++11 standard (formerly known as the 'ISO 0x' extensions).          *
//* See initialization example in the test application, Dialog1, within  *
//* the Test8() method.                                                  *
class winPos
{
   public:
   //* Default constructor *
   winPos ( void ) { ypos = ZERO ; xpos = ZERO ; }
   //* Initialization constructor *
   winPos( const short& yval, const short& xval ) : ypos(yval), xpos(xval) {}
   //* Public data members *
   short ypos ;               // Line number (zero-based)
   short xpos ;               // Column number (zero-based)
} ;

//* Key input pre-processing level:                                            *
//* nckpCOOKED:                                                                *
//*  Key input data is held by the ncurses library code until the Enter key is *
//*  pressed. In addition, non-ASCII key data such as function keys and control*
//*  codes are filtered out, making 'Cooked' mode nearly useless for           *
//*  development of any non-trivial application.                               *
//* nckpUNBUFFERED:                                                            *
//*  Allows most of the data through, although it lets the system consume the  *
//*  Break, Kill, Stop keys, so is not an ideal choice unless you enjoy getting*
//*  angry emails from your clients.                                           *
//* nckpRAW (default):                                                         *
//*  'Raw' mode is the logical choice for most terminal applications.          *
//*  In Raw mode, the application can receive data from the maximum number of  *
//*  keys, while as many of the escape sequences as possible are captured and  *
//*  translated by the NCurses key-input methods so the developer need not     *
//*  worry too much about multiple-key sequences.                              *
//* nckpNO_TRANSLATION:                                                        *
//*  If you absolutely WANT the untranslated escape sequences, you can disable *
//*  translation after the NCurses engine has been instantiated by calling     *
//*  SetKeyProcState(nckpNO_TRANSLATION); This will disable the key translation*
//*  in both the ncurses library functions and in the NCurses class methods.   *
//*  Of course the kernel and the terminal emulation program get first and     *
//*  second choice on the keys they will consume or modify before they reach   *
//*  the ncurses library and the NCurses class, so your mileage will vary      *
//*  depending upon system implementation and configuration.                   *
enum ncKeyProc : short { nckpCOOKED, nckpUNBUFFERED, nckpRAW, nckpNO_TRANSLATION } ;

//* Parameters for call to SetKeyDelay() method.                               *
//* 1. Key-input delay range is 1 to nckdMAXDELAY milliseconds.                *
//*    Setting a delay in this range causes all key input methods except       *
//*    GetKeyInput(void) (i.e. 'pause') to wait for the specified time for key *
//*    input. If key data becomes available during the delay period, it is     *
//*    placed in the caller's target variable and a non-error status           *
//*    (see enum wkeyType) is returned. If no key data becomes available       *
//*    during the specified delay period, a status of wktERR is returned and   *
//*    the value of caller's key data variable undefined.                      *
//* 2. Setting a delay of nckdNODELAY causes all keyinput methods except for   *
//*    GetKeyInput(void) (i.e. 'pause') to check for available key data, and   *
//*    if no data is available, to return immediately with a status of wktERR  *
//*    and with the value of caller's key data variable undefined.             *
//* 3. Setting a delay of nckdFOREVER causes all key input methods to block    *
//*    i.e. sleep, until a key is pressed.                                     *
enum ncKeyDelay : short { nckdFOREVER = (-1), nckdNODELAY = 0, nckdMAXDELAY = 1000 } ;

//* Number of significant bits in key input from keyboard controller.          *
//* See SetKeycodeWidth().                                                     *
enum ncKeyBits : short { nckbSEVEN, nckbEIGHT } ;

//* Max bytes in keyboard input escape sequence *
const short MAX_SEQ = (10) ;

//* Macro for Pause-for-key-input *
#define nckPause() (nc.GetKeyInput())


//****************************
//* NCurses class definition *
//****************************
class NCurses
{
public:
   NCurses () ;                              //* Default constructor
   virtual ~NCurses () ;                     //* Destructor

//* Initialize the NCurses engine for text-based application support.          *
//*  Must be called after instantiation of the NCurses class (which happens    *
//*  automagically when the application is executed) but before any other      *
//*  method of this class. See examples in test applications.                  *
//* Default Settings On Startup:                                               *
//*  Locale: set according to terminal window's environment                    *
//*  Cursor shape: nccvNORMAL                                                  *
//*  Key processing: nckpRAW                                                   *
//*  Key echo: disabled                                                        *
//*  Key delay: nckdFOREVER                                                    *
//*  Keycode width: nckbEIGHT (note that UTF-8 encoding requires 8-bit width)  *
//*  Escape sequence translation: enabled                                      *
//*  Delay after ESC key detected: none                                        *
//*  Display colors: If a color-map pointer is not specified, the NCurses      *
//*                  default color mapping is used.                            *
//*                                                                            *
//* Input  : autostartColorEngine: (optional, true by default)                 *
//*             - if 'true' (AND if terminal supports color output), call the  *
//*               StartColorEngine() method before return to caller            *
//*             - if 'false', do not start the Color Engine, only 2-color      *
//*               output will be available unless StartColorEngine() is called *
//*               before any color output is performed.                        *
//*          colorMap: (optional, NULL* pointer by default)                    *
//*             - if autostartColorEngine != false AND this pointer != NULL* , *
//*               then this is a pointer to a fully-initialized NcColorMap     *
//*               object containing the color-pair and RGB values for the      *
//*               available color attributes.                                  *
//*               - If colorMap.rgbCount == actual number of RGB register sets *
//*                 supported by the terminal AND the terminal allows          *
//*                 modification of the RGB registers, then colorMap.rgbMap[]  *
//*                 data is applied to the registers.                          *
//*               - If colorMap.pairCount == actual number of color pairs      *
//*                 supported by the terminal AND the terminal allows          *
//*                 modification of the color pairs, then colorMap.pairMap[]   *
//*                 data is applied to the terminal's color pairs.             *
//*             - if autostartColorEngine != false AND this pointer == NULL* , *
//*               then the NCurses default color mapping will be used to       *
//*               initialize the Color Engine.                                 *
//* Returns: OK if successful                                                  *
//*          ERR if unable to gain access to terminal firmware (or emulator)   *
//*              OR if provided color data do not match terminal definition    *
//*              OR if NCurses engine previously initialized                   *
short StartNCursesEngine ( bool autostartColorEngine = true, 
                           NcColorMap* colorMap = NULL ) ;

//* Shut down the NCurses Engine and release all associated resources.         *
//* (to be called just before driving application exits.)                      *
//* Input  : none                                                              *
//* Returns: nothing                                                           *
void  StopNCursesEngine ( void ) ;

//* Initialize the NCurses color engine for color text output.                 *
//* This method is called automatically as part of the StartNCursesEngine()    *
//*                               UNLESS                                       *
//* the application specifically overrides the default start-up sequence:      *
//* (autostartColorEngine parameter == false) to start the NCurses Engine in   *
//* monochrome mode.
//* If multi-color output is desired, but automatic Color Engine startup was   *
//* disabled in the call to StartNCursesEngine(), then call this method        *
//* immediately after a successful call to StartNCursesEngine() OR at the      *
//* latest, before any color-text output is performed.                         *
//*                                                                            *
//* Input  : colorMap: (optional, NULL* pointer by default)                    *
//*             - if specified, then this is a pointer to a fully-initialized  *
//*               NcColorMap object containing the RGB register set values     *
//*               and/or the color-pair values.                                *
//*               - to set RGB registers: set colorMap.rgbCount == number of   *
//*                 register sets supported by the terminal                    *
//*                 (see the TerminalColorSupport() method)                    *
//*               - to set Color Pairs: set colorMap.pairCount == number of    *
//*                 color pairs supported by the terminal                      *
//*             - if not specified, then the NCurses default color mapping     *
//*               will be used to initialize the Color Engine.                 *
//* Returns: OK if successful                                                  *
//*          ERR if unable to gain access to terminal firmware (or emulator)   *
//*              OR if provided color data do not match terminal definition    *
short StartColorEngine ( NcColorMap* colorMap = NULL ) ;

//* Determine whether the Color Engine has been successfully initialized.      *
//*                                                                            *
//* Input  : none                                                              *
//* Returns: true if Color Engine has been successfully initialized            *
//*          else false                                                        *
bool  ColorText_Initialized ( void ) ;

//* Temporarily disable the NCurses engine so external terminal applications   *
//* can gain full access to the terminal's functionality.                      *
//* Call Wake() or RefreshScreen() method to re-activate the NCurses engine.   *
//*                                                                            *
//* Input  : none                                                              *
//* Returns: nothing                                                           *
void  Hibernate ( void ) ;

//* Re-enable the NCurses engine after a previous call to Hibernate().         *
//* Also flushes any remaining data from the keyboard input stream.            *
//*                                                                            *
//* Input  : none                                                              *
//* Returns: nothing                                                           *
void  Wake ( void ) ;

//* Get a copy of the terminal's color-pair and RGB-register settings.         *
//* Each element of a color-pair is indicated as a member of enum NcBaseColors.*
//* RGB register value range: minRGBvalue to maxRGBvalue (virtual values).     *
//*                                                                            *
//* Input  : cmapPtr: (by reference, initial values ignored)                   *
//*                   NcColorMap-class object to receive the color-pair and    *
//*                   RGB register data                                        *
//* Returns: OK if successful,                                                 *
//*          ERR if Color Engine not started                                   *
short GetColorMap ( NcColorMap& cmapPtr ) ;

//* Modify the terminal's foreground/background color pairs AND/OR modify      *
//* the RGB (Red-Green-Blue) register values (color hue).                      *
//* The safe way to modify the color data is to first get a copy of the        *
//* existing color data and then modify it as needed.                          *
//*   NcColorMap colorMap ;                                                    *
//*   nc.GetColorMap ( colorMap ) ;                                            *
//*     [ perform desired modifications to the data ]                          *
//*   nc.SetColorMap ( colorMap, true, true ) ;                                *
//*                                                                            *
//* Input  : cmapPtr : pointer to an initialized NcColorMap-class object       *
//*                    containing the new color-pair AND/OR RGB register data  *
//*          setPairs: if 'true', use data provided in cmapPtr.pairMap[] array *
//*                    to modify ALL color pairs supported by the terminal     *
//*                     - Each value is a member of enum NcBaseColors.         *
//*                     - Each terminal implementation has a 'default'         *
//*                       foreground and background color, usually             *
//*                       white-on-black OR black-on-white. Use of defaults    *
//*                       may be specified for either or both fgnd/bkgnd by    *
//*                       entering ncbcDEFAULT in the appropriate field(s).    *
//*          setRGB  : if 'true', use data provided in cmapPtr.rgbMap[] array  *
//*                    to modify ALL RGB registers supported by the terminal   *
//*                     - RGB register value range: minRGBvalue to maxRGBvalue.*
//* Returns: OK if successful, else ERR                                        *
short SetColorMap ( NcColorMap* cmapPtr, bool setPairs, bool setRGB ) ;

//* Modify a single foreground/background color-pair.                          *
//*                                                                            *
//* Input  : pairIndex: index of color pair to be modified                     *
//*          pairData : new foreground and background color numbers            *
//* Returns: OK if successful, else ERR                                        *
short SetColorMap ( short pairIndex, const NcColorPair& pairData ) ;

//* Modify a single RGB register set.                                          *
//*                                                                            *
//* Input  : rgbIndex : index of RGB register set to be modified               *
//*          rgbData  : new red/green/blue register-set values                 *
//* Returns: OK if successful, else ERR                                        *
short SetColorMap ( short rgbIndex, const NcRgbSet& rgbData ) ;

//* Establish a default background color to be used with the basic             *
//* text (foreground) colors.                                                  *
//* Note that the results of this operation will likely be disappointing,      *
//* especially if a black background (ncbcBK) is specified.                    *
//* See also: SetColorMap().                                                   *
//*                                                                            *
//* Input  : bkGround : background color for the basic color pairs directly    *
//*                     supported by the ncurses library (usually pairs 0-7)   *
//*          pzfgnd   : (optional, ncbcDEFAULT by default)                     *
//*                     foreground text color for color pair zero (ncbcBK)     *
//* Returns: nothing                                                           *
void SetDefaultBackground ( NcBaseColors bkgnd, NcBaseColors pzfgnd = ncbcDEFAULT ) ;

//* Returns the terminal window dimensions in lines and columns.               *
//* Input  : scrLines (by reference) receives screen line-count                *
//*          scrColumns (by reference) receives screen column-count            *
//* Returns: nothing                                                           *
void ScreenDimensions ( short& scrLines, short& scrColumns ) ;

//* Specify a new locale setting for multi-byte character interpretation.      *
//* Note that when the NCurses class is instantiated, the locale used is the   *
//* one specified by the terminal window's environment variables. If the       *
//* terminal window's locale is not a UTF-8 locale, then this method must be   *
//* called to specify a UTF-8-compliant locale before any I/O takes place.     *
//* If the locale is not UTF-8 compliant, then NCurses methods will not be able*
//* to handle characters beyond simple ASCII (7-bit) characters.               *
//* Input  : string containing name of new locale                              *
//*           NOTE: For a complete list of locales supported by your terminal  *
//*                 window, type:  locale -a    at the command prompt.         *
//* Returns: OK if successful, else ERR                                        *
short SetLocale ( const char* newLocaleName ) ;

//* Returns a pointer to a const string containing the name of the currently-  *
//* active locale setting.                                                     *
//* Input  : none                                                              *
//* Returns: pointer to locale string                                          *
const char* GetLocale ( void ) ;

//* Attempts to verify whether the local name provided specifies a locale that *
//* supports UTF-8 encoding.                                                   *
//* Input  : (optional, NULL by default)                                       *
//*            NULL indicates that current local setting should be tested      *
//*            If non-NULL, test the locale name provided                      *
//* Returns: OK if specified locale known to support UTF-8 encoding, else ERR  *
short VerifyLocale ( const char* lcName=NULL ) ;

//* Refresh the entire contents of the terminal window.                        *
//* Input  : none                                                              *
//* Returns: nothing                                                           *
void RefreshScreen ( void ) ;

//* Erase the entire contents of the terminal window.                          *
//* Input  : none                                                              *
//* Returns: nothing                                                           *
void ClearScreen ( void ) ;

//* Determine whether the terminal or terminal emulator is capable of          *
//* displaying color text. This will almost always be true on a computer, but  *
//* may return false if running on a monochrome dumb terminal.                 *
//*                                                                            *
//* Input  : none                                                              *
//* Returns: 'true' if the terminal supports multi-color text output           *
//*          'false' if the terminal does not supoort multi-color text output  *
//*                  OR if NCurses Engine has not yet been started i.e. info   *
//*                     is not yet available                                   *
bool ColorText_Available ( void ) ;

//* Report terminal (or terminal emulator) color output capabilities.          *
//* See 'termColorInfo' class for mor information.                             *
//*                                                                            *
//* Input  : cinfo : (by reference) receives terminal color information        *
//* Returns: 'true' if the terminal supports multi-color text output           *
//*                 parameters will be initialized                             *
//*          'false' if the terminal does not support multi-color text output  *
//*                  OR if Color Engine has not yet been started i.e. info is  *
//*                     not yet available                                      *
//*                  (term_colors, rgb_regs and fgbg_pairs set to ZERO)        *
//*                  (mod_pairs and mod_rgb set to 'false')                    *
bool  TerminalColorSupport ( termColorInfo& cinfo ) ;

//* Write a string at the specified position in the stdscr window i.e. the     *
//* window in which the NCurses engine was started.  Window is refreshed.      *
//*  NOTE: String output will be truncated at the edge of the window.          *
//*  NOTE: ASCII control characters, 0x01 through 0x1F are ignored.            *
//*                                                                            *
//* Input  : startY : Y cursor start position                                  *
//*          startX : X cursor start position                                  *
//*                OR                                                          *
//*          startYX: Y/X cursor start position                                *
//*                                                                            *
//*          -- FOR UTF-8-ENCODED CHARACTERS (INCLUDING ASCII)                 *
//*            uStr : pointer to null-terminated UTF-8-encoded string          *
//*                    (maximum number of bytes incl. NULLCHAR == gsMAXBYTES ) *
//*          -- FOR wchar_t 'WIDE' CHARACTERS                                  *
//*            wStr : pointer to null-terminated 'wide' wchar_t string         *
//*                    (maximum number of chars incl. NULLCHAR == gsMAXCHARS)  *
//*                                                                            *
//*          cAttr  : color attribute                                          *
//*          rtl    : (optional, 'false' by default) for RTL written languages *
//*                   if 'true', cursor moves from right-to-left               *
//*                                                                            *
//* Returns: returns final position of cursor                                  *
//*          If specified starting Y/X position is valid, then cursor position *
//*           returned will be the the column following the last character of  *
//*           displayed data (or the last column of the target line).          *
//*          If specified starting position is invalid, then cursor position   *
//*           is set to window origin before writing to window.                *
winPos WriteString ( short startY, short startX, const char* uStr, 
                     attr_t cAttr, bool rtl = false ) ;
winPos WriteString ( const winPos& startYX, const char* uStr, 
                     attr_t cAttr, bool rtl = false ) ;
winPos WriteString ( short startY, short startX, const wchar_t* wStr, 
                     attr_t cAttr, bool rtl = false ) ;
winPos WriteString ( const winPos& startYX, const wchar_t* wStr, 
                     attr_t cAttr, bool rtl = false ) ;

//* Write a single character at the specified position.                        *
//* Optionally, refresh the display before return to caller.                   *
//*  NOTE: ASCII control characters, 0x01 through 0x1F are ignored.            *
//*                                                                            *
//* Input  : startY : Y cursor position                                        *
//*          startX : X cursor position                                        *
//*                OR                                                          *
//*          startYX: Y/X cursor start position                                *
//*                                                                            *
//*          -- FOR UTF-8-ENCODED CHARACTERS (INCLUDING ASCII)                 *
//*            uChar: pointer to character to display                          *
//*                    (single-byte or multi-byte character)                   *
//*          -- FOR wchar_t 'WIDE' CHARACTERS                                  *
//*            wChar: character to display in wchar_t format                   *
//*                                                                            *
//*          cAttr  : color attribute                                          *
//*          refresh: (optional, default==false) refresh display window        *
//*          rtl    : (optional, 'false' by default) for RTL written languages *
//*                   if 'true', cursor moves from right-to-left               *
//*                                                                            *
//* Returns: returns final position of cursor                                  *
//*          If specified starting Y/X position is valid, AND the specified    *
//*           character is a printing character, then cursor position returned *
//*           will be the the column following the character (or the last      *
//*           column on the target line).                                      *
//*          If invalid character specified, then position returned will be    *
//*           start position.                                                  *
//*          If specified starting position is invalid, then cursor position   *
//*           is set to window origin before writing to window.                *
winPos WriteChar ( short startY, short startX, const char* uChar, 
                   attr_t cAttr, bool refresh = false, bool rtl = false ) ;
winPos WriteChar ( const winPos& startYX, const char* uChar, 
                   attr_t cAttr, bool refresh = false, bool rtl = false ) ;
winPos WriteChar ( short startY, short startX, wchar_t wChar, 
                   attr_t cAttr, bool refresh = false, bool rtl = false ) ;
winPos WriteChar ( const winPos& startYX, wchar_t wChar, 
                   attr_t cAttr, bool refresh = false, bool rtl = false ) ;

//* Clear the line. Sets all characters of specified line to SPACE character.  *
//* Terminal window is refreshed.                                              *
//*                                                                            *
//* Input  : ypos : line number (zero-based)                                   *
//*          cAttr: (optional, default==terminal background color: nc.bw       *
//* Returns: OK if successful, ERR if invalid line index                       *
short ClearLine ( short ypos, attr_t cAttr = ZERO ) ;

//* Get the NCurses main screen (terminal window) cursor position.             *
//* Input  : none                                                              *
//* Returns: current cursor position in a winPos class object                  *
winPos GetCursor ( void ) ;

//* Set the main screen's cursor position.                                     *
//* Input  : ypos : Y offset from upper-left corner (zero-based)               *
//*          xpos : X offset from upper-left corner (zero-based)               *
//*            OR                                                              *
//*          yxpos: Y/X offset from upper-left corner (zero-based)             *
//* Returns: OK if successful, ERR if parameter(s) out of range                *
short  SetCursor ( short ypos, short xpos ) ;
short  SetCursor ( winPos yxpos ) ;

//* Returns the current visibility state of the cursor.                        *
//* Input  : none                                                              *
//* Returns: current cursor state (member of enum ncCurVis)                    *
ncCurVis GetCursorState ( void ) ;

//* Set the visibility state of the cursor.                                    *
//* Saves previous state for later call to RestoreCursorState().               *
//* Input  : member of enum ncCurVis                                           *
//* Returns: OK                                                                *
short SetCursorState ( ncCurVis state ) ;

//* Restore the previous visibility state of the cursor.                       *
//* Input  : none                                                              *
//* Returns: OK                                                                *
short RestoreCursorState ( void ) ;

//* Enable or disable automatic echo of key input.                             *
//* Input echo is disabled by default, and should remain disabled at all times.*
//* Use with caution.                                                          *
//* Input  : none                                                              *
//* Returns: OK if successful, else ERR                                        *
short KeyEcho ( bool enable ) ;

//* Returns the current level of key-input processing done by the kernel,      *
//* the ncurses library, and the NCurses class before data are passed to the   *
//* application.                                                               *
//* Input  : none                                                              *
//* Returns: member of ncKeyProc                                               *
ncKeyProc GetKeyProcState ( void ) ;

//* Set the level of key-input processing done by the kernel, the ncurses      *
//* library, and the NCurses class before data are passed to the application   *
//* code.                                                                      *
//* The default value on startup is nckpRAW because key input is fully         *
//* encapsulated in the NCurses class and the maximum number of keys are       *
//* available to the driving application. Have a good reason before modifying  *
//* this parameter.                                                            *
//* Applications should always request key data through the NCurses class and  *
//* NOT  directly from the system or through C or C++ functionality.           *
//* Input  : member of ncKeyProc                                               *
//* Returns: OK if successful, else ERR                                        *
short SetKeyProcState ( ncKeyProc procState ) ;

//* Returns the current key-input delay in milliseconds.                       *
//* Input  : none                                                              *
//* Returns: delay in milliseconds OR nckdFOREVER OR nckdNODELAY               *
short GetKeyDelay ( void ) ;

//* Set the number of milliseconds to delay while waiting for key input.       *
//* If no key data arrives before the timeout, the key input methods           *
//* will return ERR and the target variable will be undefined.                 *
//* Input  : delay in milliseconds (1 to 1000)                                 *
//*           nckdFOREVER == wait indefinitely for input (blocking read)       *
//*           nckdNODELAY == returns immediately if no key data available      *
//* Returns: OK                                                                *
short SetKeyDelay ( short delay ) ;

//* Set the number of significant bits of key input from keyboard controller.  *
//* Start-up default is nckbEIGHT.                                             *
//* IMPORTANT NOTE: UTF-8 encoding requires 8-bit width, so don't modify       *
//*                 default without a good reason.                             *
//*                                                                            *
//* Input  : member of ncKeyBits                                               *
//*                                                                            *
//* Returns: OK if successful, else ERR                                        *
short SetKeycodeWidth ( ncKeyBits codeWidth ) ;


//*******************************
//** Keyboard data input group **
//*******************************
//*  - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -   *
//* Get a keycode and keytype from the ncursesw input stream.                  *
//* This includes all supported international character codes as well as       *
//* captured-and-translated escape sequences representing function keys,       *
//* cursor keys and other special-purpose key combinations.                    *
//*                                                                            *
//* If mouse input has been enabled, mouse events will also be returned in     *
//* caller's wkeyCode-class object.                                            *
//* See also the GetMouseEvent() method for additional information.            *
//*                                                                            *
//* Input  : wKey   : wkeyCode-class object (by reference) to hold the key     *
//*                    information (initial values ignored)                    *
//*          rawData: (optional, false by default)                             *
//*                   if false, capture and translate escape sequences         *
//*                   if true, no local processing by the NCurses class, BUT   *
//*                   the ncurses library will still do its translations       *
//*                   UNLESS there has been a prior call to                    *
//*                   SetKeyProcState(nckpNO_TRANSLATION);                     *
//* Returns: member of enum wkeyType                                           *
//*          wktPRINT  if key is a valid, printing wchar_t character code      *
//*          wktFUNKEY if key is a translated function key (cursor key, Fx,    *
//*                     etc) OR if key is (non-printing) control code < 0x0020.*
//*          wktMOUSE  if a mouse event in the input stream has been detected, *
//*                    then the mouse event will be returned in wKey.mevent.   *
//*                    (wKey.key will be set to ZERO and should be ignored.)   *
//*          wktEXTEND This is an extension to the key input captured by the   *
//*                     ncurses library primitives. It provides additional     *
//*                     keycodes to the application: Alt+'a' through Alt+'z',  *
//*                     Alt+Shift+'a' through Alt+Shift+'z' and Alt+Ctrl+'a'   *
//*                     through Alt+Ctrl+'z'. (See notes in NCursesKeyDef.hpp.)*
//*          wktESCSEQ if an untranslated escape sequence was captured.        *
//*                     The keycode (wKey.key or wChar) will be set to ZERO.   *
//*                     This should only happen if application has turned      *
//*                     off key-input translation (or in the rare case that    *
//*                     neither the ncurses library nor the NCurses class can  *
//*                     translate the sequence).                               *
//*                     The captured escape sequence is held in a static       *
//*                     buffer and can be retrieved via GetEscapeSequence()    *
//*                     before it is overwritten by the next untranslated      *
//*                     sequence.                                              *
//*          wktERR    if no key data available.                               *
//*                     (this method performs a non-blocking read if)          *
//*                     (SetKeyDelay() method not set to nckdFOREVER),         *
//*                     OR if system call returned an error.                   *
//*                     The keycode (wKey.key or wChar) will be set to ZERO.   *
//*                                                                            *
//*    IT IS STRONGLY RECOMMENDED THAT THE RETURN VALUE ALWAYS BE CHECKED!     *
wkeyType GetKeyInput ( wkeyCode& wKey, bool rawData = false ) ;

//* Get a key from the input stream.                                           *
//*  Important Note: Internally, the NCurses family of classes uses 32-bit     *
//*  input and output exclusively. However this 16-bit method is available     *
//*  if it is known that 'wide' characters will never be input by the user.    *
//* Input  : sChar  : returned key value (by reference, initial value ignored) *
//*          rawData: (optional, false by default)                             *
//*                   if false, capture and translate escape sequences         *
//*                   if true, no local processing by the NCurses class        *
//* Returns: member of enum wkeyType   See above for details.                  *
wkeyType GetKeyInput ( short& sChar, bool rawData = false ) ;

//* Wait for a keystroke i.e. PAUSE.  The key input data are discarded.        *
//* See the macro, 'nckPause()' above.                                         *
//* Input  : none                                                              *
//* Returns: OK                                                                *
short GetKeyInput ( void ) ;

//* Return the wchar_t keycode to the input queue.                             *
//* Note that the push buffer holds only one code.                             *
//*                                                                            *
//* Input  : keycode and key type to be returned to the input queue            *
//*          (note that key 'type' is currently ignored)                       *
//* Returns: OK if successful, else ERR                                        *
short UngetKeyInput ( wkeyCode& wKey ) ;

//* If there is key data waiting in buffer, return a copy of it.               *
//* Does not remove key data from input stream.                                *
//*                                                                            *
//* Input  : wkeyCode-class object (by reference) to hold the key information  *
//*           (initial values ignored)                                         *
//* Returns: member of enum wkeyType                                           *
//*          if key data waiting:                                              *
//*             wKey.type == wktPRINT, wktFUNKEY or wktESCSEQ                  *
//*             wKey.key  == key code                                          *
//*          if no key data waiting:                                           *
//*             wKey.type == wktERR                                            *
//*             wKey.key  == ZERO and should be ignored                        *
wkeyType KeyPeek ( wkeyCode& wKey ) ;

//* If an escape sequence has not been translated into a keycode within a call *
//* to GetKeyInput() -- either because application has disabled translation,   *
//* or because the sequence is unknown -- GetKeyInput() returns wktESCSEQ and  *
//* the sequence is stored in a static buffer for potential retrieval.         *
//* This method returns a pointer to the most recently captured sequence.      *
//* Input  : charCount (by reference, initial value ignored)                   *
//* Returns: pointer to static buffer containing untranslated escape sequence. *
//*             charCount' gets number of characters in the sequence.          *
//*          OR returns a pointer to an empty string if no untranslated        *
//*            sequence has been captured and 'charCount' gets ZERO.           *
const wchar_t* GetEscapeSequence ( short& charCount ) ;

//* Flush (discard) any key input data currently in the key input stream.      *
//* If mouse input enabled, then flush the mouse-event queue also.             *
//* Input  : none                                                              *
//* Returns: OK                                                                *
short FlushKeyInputStream ( void ) ;

//* Sound an audible alert. The actual sound produced is system dependent.     *
//* Note that if the system does not support audible alerts, then then the     *
//* screen will visibly flash instead.                                         *
//* Input  : none                                                              *
//* Returns: OK if audible alert supported, else ERR                           *
short UserAlert ( void ) ;

//*******************************
//**  Mouse data input group   **
//*******************************
//*  - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -   *
//* Allow mouse events to be reported                                          *
//*  indirectly: through the GetKeyInput methods and                           *
//*   directly : through GetMouseEvent method                                  *
//*                                                                            *
//* Input  : eventTypes : bit mask of event types to be enabled                *
//*                       (at least one event-type bit must be set)            *
//*                       Bit definitions are the BUTTONxxxx group of          *
//*                       defines in ncurses.h                                 *
//* Returns: OK  if mouse enabled AND all bits of the event-type mask          *
//*              have been set successfully                                    *
//*          ERR if mouse not enabled OR actual event-type mask differs from   *
//*              that specified OR eventTypes == ZERO                          *
short EnableMouseEvents ( mmask_t eventTypes ) ;

//* Modify the type(s) of mouse events will be reported.                       *
//*                                                                            *
//* Input  : eventTypes: new mouse-event-type mask                             *
//* Returns: OK  if mouse enabled AND a all bits of the event-type mask        *
//*              have been set successfully                                    *
//*          ERR if mouse not enabled OR actual event-type mask differs from   *
//*              that specified OR eventTypes == ZERO                          *
short SetMouseEventTypes ( mmask_t eventTypes ) ;

//* Returns the current mouse-event-type bitmask.                              *
//*                                                                            *
//* Input  : eventTypes: (by reference, initial value ignored)                 *
//*                      receives the mouse-event-type mask                    *
//*                                                                            *
//* Returns: OK  if successtul                                                 *
//*          ERR if mouse-event input disabled                                 *
short GetMouseEventTypes ( mmask_t& eventTypes ) ;

//* Disable reporting of mouse events.                                         *
//*                                                                            *
//* Input  : oldeventTypes : (optional, NULL pointer by default)               *
//*                          if !NULL, receives previous event-type bit mask   *
//* Returns: OK  if mouse-event reporting disabled                             *
//*              (mouse driver MAY BE disabled also - system dependent)        *
//*          ERR if mouse-event reporting could not be disabled                *
short DisableMouseEvents ( mmask_t* oldeventTypes = NULL ) ;

//* Reports whether mouse input is currently enabled.                          *
//*                                                                            *
//* Input  : none                                                              *
//* Returns: 'true'  if mouse events can be detected and reported, else 'false'*
bool  MouseEventsEnabled ( void ) ;

//* Specify the maximum interval between a button-press event and a following  *
//* button-release event that will result in a mouse 'click' being reported.   *
//* Note: this interval applies to single-, double- and triple-click events.   *
//* If the interval between press and release is greater than this value, then *
//* a 'click' event will not be reported.                                      *
//*                                                                            *
//* Input  : max4Click : interval in thousandths (0.001) of a second           *
//*                      specify an interval OR a member of enum ncmInterval   *
//*          oldMax    : (optional, NULL pointer by default)                   *
//*                      pointer to variable to receive previous interval value*
//* Returns: OK  if successtul                                                 *
//*          ERR if value out-of-range OR mouse-event input disabled           *
short SetMouseClickInterval ( short max4Click, short* oldMax = NULL ) ;

//* Get the current maximum mouse-click interval in thousandths of a second.   *
//*                                                                            *
//* Input  : max4Click : (by reference, initial value ignored)                 *
//*                      receives current interval                             *
//* Returns: OK  if successtul                                                 *
//*          ERR if mouse-event input disabled                                 *
short GetMouseClickInterval ( short& max4Click ) ;

//* Extract the newest mouse event from the event queue.                       *
//* - Events located beyond the writable area of the terminal window belong to *
//*   someone else and are therefore not available to us.                      *
//* - Some mouse events within the terminal window are captured by the GUI     *
//*   desktop, and are therefore not available to us.                          *
//*         Example: Button 3 click is the terminal's context menu.            *
//* - Some filtering of spurious events may be performed to prevent unexpected *
//*   events related to timing artifacts. See FilterSpuriousMouseEvents()      *
//*   method for more information.                                             *
//* - IT IS RECOMMENDED that all mouse-event capture be done through the       *
//*   GetKeyInput() method, and that the GetMouseEvent() and UngetMouseEvent() *
//*   methods should not be called directly.                                   *
//*                                                                            *
//* Input  : mevent : (by reference, initial value ignored)                    *
//*                   receives the mouse-event data                            *
//* Returns: OK  if mouse event extracted successfully                         *
//*          ERR if no mouse event in the queue or mouse-event input disabled  *
short GetMouseEvent ( MouseEvent& mevent ) ;

//* Push a previously-extracted mouse event back into the input queue.         *
//* - IT IS RECOMMENDED that all mouse-event capture be done through the       *
//*   GetKeyInput() method, and that the GetMouseEvent() and UngetMouseEvent() *
//*   methods should not be called directly.                                   *
//*                                                                            *
//* Input  : mevent : (by reference, initial value ignored)                    *
//*                   receives the mouse-event data                            *
//* Returns: OK  if mouse event successfully re-queued                         *
//*          ERR if event queue is full OR if mouse-event input disabled       *
short UngetMouseEvent ( MouseEvent& mevent ) ;

//* Remove any mouse events waiting in the mouse-event queue.                  *
//* The GetMouseEvent method returns events in LIFO (last-in-first-out) order. *
//* If you have received the event you're looking for and don't want any more  *
//* events, OR if your mouse has become tempramental and is sending undesired  *
//* or unexpected events, then this method can be used to flush the            *
//* mouse-event queue.                                                         *
//*                                                                            *
//* Input  : none                                                              *
//* Returns: OK if successful, or ERR if mouse-event input disabled            *
short FlushMouseEventQueue ( void ) ;

// NOT YET IMPLEMENTED! - Note that the nucurses library has no mechanism for 
// reporting the mouse-pointer position UNLESS a button event is detected.
// (sorry about that)
short GetMousePosition ( MouseEvent& mpos ) ;

//* Enable or disable filtering of mouse events that arrive in the queue even  *
//* though events of that type have not been requested via the event mask.     *
//* - Filtering is enabled by default, and it is recommended that it remain    *
//*   enabled unless there is a specific need to see the spurious events.      *
//* While filtering is enabled:                                                *
//* - If an unrequested event is received by the GetMouseEvent method, then    *
//*   it is discarded, and reported as no event.                               *
//* - If an unrequested event is received in the keyboard-input queue via the  *
//*   GetKeyInput method, then the event is discarded AND:                     *
//*    a) if GetKeyInput is set to block, then wait for the next keycode       *
//*    b) if GetKeyInput is set for timeout, then wktERR is returned.          *
//* - In either case, if an unrequested event is received, then the mouse-input*
//*   queue is flushed.                                                        *
//*                                                                            *
//* Input  : filter : 'true' enables filtering                                 *
//*                   'false disables filtering                                *
//* Returns: OK if successful, or ERR if mouse-event input disabled            *
short FilterSpuriousMouseEvents ( bool filter ) ;

const char* Get_NCurses_Version ( void ) ;      // returns NCurses class version
const char* Get_nclibrary_Version ( void ) ;    // returns ncurses library version

            //*********************************************
            //* Application development tools             *
            //* Do not use these methods in finished code.*
            //*********************************************

//* Write data to the main terminal window ('stdscr'). This method is to be    *
//* called ONLY to output data to the terminal window when output-stream       *
//* scrolling has been enabled. (see the OutputStreamScrolling() method)       *
//* If output-stream scrolling has not been enabled, then this method calls    *
//* the standard NCurses-class 'WriteString' method to produce formatted,      *
//* non-scrolling data.                                                        *
//*                                                                            *
//* This method does not filter the output in any way. While the standard      *
//* output methods ignore ASCII control characters to ensure consistent        *
//* formatting, WriteStream() outputs all characters, regardless of type.      *
//* Note however, that the C language output primitives will either ignore     *
//* most ASCII control characters, or will choke and output garbage.           *
//* Only newline '\n' and tab '\t' will be reliably recognized on all systems. *
//*                                                                            *
//* IMPORTANT NOTE:                                                            *
//* This method gives the underlying 'ncurses' C-language library direct       *
//* control of the output, which is generally not a good idea. That is why     *
//* this method is classified as a debugging method. Use with caution.         *
//*                                                                            *
//* IMPORTANT NOTE:                                                            *
//* This method is not to be used when the higher-level NcWindow and NcDialog  *
//* classes have been instantiated in the terminal window. Doing so will cause *
//* all kinds of ugliness. You have been warned.                               *
//*                                                                            *
//* Input  : wStr : pointer to null-terminated wchar_t 'wide'-character string *
//*          cAttr: color attribute                                            *
//* Returns: returns final position of cursor                                  *
winPos WriteStream ( const wchar_t* wStr, attr_t cAttr ) ;

//* Enable or Disable "standard output" style scrolling in the stdscr window.  *
//*                                                                            *
//* By default, the underlying 'ncurses' library disables scrolling.           *
//* The NCurses library also leaves scrolling disabled by default because it   *
//* produces ugly and undisciplined output.                                    *
//*                                                                            *
//* If, however, you have a need to emulate a common console application which *
//* is unaware of screen edges and data formatting then use this method to     *
//* enable/disable scrolling.                                                  *
//*                                                                            *
//* a) If the cursor reaches the right edge of the terminal window, the cursor *
//*    is automatically repositioned to the leftmost column of the following   *
//*    display line. (This is dumb-terminal character wrapping, NOT word wrap.)*
//* b) If the cursor reaches the right edge of the last display line in the    *
//*    terminal window, then all data in the window will be scrolled up by one *
//*    line, and the cursor will be repositioned to the leftmost column of the *
//*    last display line.                                                      *
//*                                                                            *
//* IMPORTANT INFORMATION ABOUT AUTOMATIC SCROLLING:                           *
//*  - The NCurses-class output methods 'WriteString' and 'WriteChar' are fully*
//*    aware of the terminal-window edges and therefore truncate output at the *
//*    edge of the window REGARDLESS of the state of Output-stream scrolling.  *
//*  - Only the 'WriteStream' method ignores the terminal window size and      *
//*    allows for automatic scrolling.                                         *
//*  - Scrolling output applies ONLY to the main terminal window and NOT to    *
//*    any sub-window objects within the terminal window. Therefore, do not    *
//*    define any sub-windows when output-stream scrolling is enabled. In      *
//*    practical terms, this means that you must not call the methods of the   *
//*    higher-level NcWindow and NcDialog classes while scrolling is enabled.  *
//*  - The cursor remains where is was placed after the last character in the  *
//*    string is written, so subsequent output will begin where the previous   *
//*    output left off unless the 'SetCursor' method is explicitly called.     *
//*    Note that any attempt to set the cursor at a position outside the       *
//*    terminal window will be ignored.                                        *
//*  - This functionality IS NOT THREAD SAFE. It is the application's          *
//*    responsibility to ensure that only one thread writes data to the output *
//*    stream.                                                                 *
//*  - Because output-stream scrolling emulates output to a dumb terminal,     *
//*    this method does not support RTL languages.                             *
//*                                                                            *
//* Input  : enable : if 'true', enable automatic line wrap and scrolling      *
//*                   if 'false', disable automatic line wrap and scrolling    *
//* Returns: OK                                                                *
short OutputStreamScrolling ( bool enable ) ;

void  DisplayColorOptions ( short ypos=3, short xpos=1, bool pause=true ) ;
void  DisplayAlternateFont ( short ypos=3, short xpos=1, bool pause=true ) ;
void  DisplayDefinedColors ( short ypos=3, short xpos=1, bool pause=true ) ;
void  DebugBorder ( short wLines, short wCols, short wulY, short wulX, attr_t attr ) ;
//* Display strings for names of keycodes. (See Dialog2 application, Test 5).  *
//* Note that keycodes BETWEEN Table1 and Table2 are ASCII printing characters.*
//* wktFUNKEY: NULLCHAR, Ctrl+A through Ctrl+Z                                 *
void  GetTransTable1 ( const char**& tt1, short& tt1Min, short& tt1Max ) ;
//* wktFUNKEY: 0x7F through 0x0236                                             *
void  GetTransTable2 ( const char**& tt2, short& tt2Min, short& ttsMax ) ;
//* wktEXTEND: Alt+A through ALT+Z AND Alt+Shift+A through Alt+Shift+Z         *
void  GetTransTable3 ( const char**& tt3, short& tt3Entries ) ;


//* Returns the address of the standard screen (stdscr).                       *
//* This is the pointer returned by the ncurses library call to initscr().     *
WINDOW* Get_stdscr_Address ( void ) ;


//***************************
//* Public Access Variables *
//***************************
//******************************************************************************
//* IMPORTANT NOTES:                                                           *
//*  1. These variables should be treated as READ-ONLY constants.              *
//*     The only reason they aren't constants is that they cannot be           *
//*     initialized until after the NCurses Engine has been started.           *
//*     The Linux/UNIX ncurses library protects its variables through the use  *
//*     of nested macros, but we consider that to be one level of indirection  *
//*     too many.   !! Please codify, don't modify !!                          *
//*  2. Color attribute access. Access color attribute variables ONLY after    *
//*     the color engine has been initialized.                                 *
//*  3. Under the default mapping, there will be a duplicate foreground/       *
//*     background pair. Which color-pairs are duplicates depends upon the     *
//*     background color selected.                                             *
//*  4. Except for extraordinary color combinations, the driving applicaton    *
//*     should specify colors using these variables rather than literals or    *
//*     application-defined constants. The reason for this is that not all the *
//*     color attribute definitions are POSSIBLE at any given time, but these  *
//*     colors are guaranteed to be available.                                 *
//*                                                                            *
//******************************************************************************

//* Terminal default foreground/background     *
//* (usually black-on-white or white-on-black) *
attr_t   bw ;

//* Eight basic foreground colors on default,  *
//* (or application-specified) background      *
attr_t   bk ;     // Black (this is 'bw' with fg/bg swapped)
attr_t   re ;     // Red
attr_t   gr ;     // Green
attr_t   br ;     // Brown
attr_t   bl ;     // Blue
attr_t   ma ;     // Magenta
attr_t   cy ;     // Cyan
attr_t   gy ;     // Grey

//* For systems which have 16 basic terminal colors.(if system has only 8      *
//* basic colors, then these duplicate the first 8).                           *
//* Note that depending on the background you choose, one member of this group *
//* will produce the same foreground as background, making the text for that   *
//* variable invisible. For the default background this will likely be 'bgy'   *
//* (light grey on white).                                                     *
attr_t   bbk ;    // Bright Black
attr_t   bre ;    // Bright Red
attr_t   bgr ;    // Bright Green
attr_t   bbr ;    // Bright Brown (yellow)
attr_t   bbl ;    // Bright Blue
attr_t   bma ;    // Bright Magenta
attr_t   bcy ;    // Bright Cyan
attr_t   bgy ;    // Bright Grey (text is likely invisible, see note)

attr_t   bwB ;    // Base Colors with BOLD attribute bit set
attr_t   bkB ;
attr_t   reB ;
attr_t   grB ;
attr_t   brB ;
attr_t   blB ;
attr_t   maB ;
attr_t   cyB ;
attr_t   gyB ;
      
attr_t   bwR ;    // Base Colors with REVERSE attribute bit set
attr_t   bkR ;
attr_t   reR ;
attr_t   grR ;
attr_t   brR ;
attr_t   blR ;
attr_t   maR ;
attr_t   cyR ;
attr_t   gyR ;

attr_t   bwG ;    // Base Colors with both BOLD nad REVERSE attribute bits set
attr_t   bkG ;
attr_t   reG ;
attr_t   grG ;
attr_t   brG ;
attr_t   blG ;
attr_t   maG ;
attr_t   cyG ;
attr_t   gyG ;

attr_t   bwS ;    // Base Colors with STANDOUT attribute bit set
attr_t   bkS ;
attr_t   reS ;
attr_t   grS ;
attr_t   brS ;
attr_t   blS ;
attr_t   maS ;
attr_t   cyS ;
attr_t   gyS ;
      
attr_t   bwD ;    // Base Colors with DIM attribute bit set
attr_t   bkD ;
attr_t   reD ;
attr_t   grD ;
attr_t   brD ;
attr_t   blD ;
attr_t   maD ;
attr_t   cyD ;
attr_t   gyD ;
      
attr_t   bwU ;    // Base Colors with UNDERLINE attribute bit set
attr_t   bkU ;
attr_t   reU ;
attr_t   grU ;
attr_t   brU ;
attr_t   blU ;
attr_t   maU ;
attr_t   cyU ;
attr_t   gyU ;
      
//* Additional foreground/background combinations.             *
//* For optimal contrast, values are initialized as follows:   *
//* - The bkxx group and the gyxx group are initialized using  *
//*   standard foreground colors on standard background colors.*
//* - The other color groups are initialized using 'bold'      *
//*   foreground colors on standard background colors.         *
//* The naming scheme is based on 64 or more available color   *
//* pairs. If the terminal supports fewer than 64 pairs, the   *
//* extra values will be initialized to the terminal default   *
//* color. For example output, please see the test application,*
//* Dialog2, tests 4 and 6.                                    *
attr_t   bkre ;   // black-on-red
attr_t   bkgr ;   // black-on-green
attr_t   bkbr ;   // black-on-brown
attr_t   bkbl ;   // black-on-blue
attr_t   bkma ;   // black-on-magenta
attr_t   bkcy ;   // black-on-cyan
attr_t   bkgy ;   // black-on-grey

attr_t   rebk ;   // red-on-black
attr_t   regr ;   // red-on-green
attr_t   rebr ;   // red-on-brown
attr_t   rebl ;   // red-on-blue
attr_t   rema ;   // red-on-magenta
attr_t   recy ;   // red-on-cyan
attr_t   regy ;   // red-on-grey

attr_t   grbk ;   // green-on-black
attr_t   grre ;   // green-on-red
attr_t   grbr ;   // green-on-brown
attr_t   grbl ;   // green-on-blue
attr_t   grma ;   // green-on-magenta
attr_t   grcy ;   // green-on-cyan
attr_t   grgy ;   // green-on-grey

attr_t   brbk ;   // brown-on-black
attr_t   brre ;   // brown-on-red
attr_t   brgr ;   // brown-on-green
attr_t   brbl ;   // brown-on-blue
attr_t   brma ;   // brown-on-magenta
attr_t   brcy ;   // brown-on-cyan
attr_t   brgy ;   // brown-on-grey

attr_t   blbk  ;  // blue-on-black
attr_t   blre  ;  // blue-on-red
attr_t   blgr  ;  // blue-on-green
attr_t   blbr  ;  // blue-on-brown
attr_t   blma  ;  // blue-on-magenta
attr_t   blcy  ;  // blue-on-cyan
attr_t   blgy  ;  // blue-on-grey

attr_t   mabk  ;  // magenta-on-black
attr_t   mare  ;  // magenta-on-red
attr_t   magr  ;  // magenta-on-green
attr_t   mabr  ;  // magenta-on-brown
attr_t   mabl  ;  // magenta-on-blue
attr_t   macy  ;  // magenta-on-cyan
attr_t   magy  ;  // magenta-on-grey
           
attr_t   cybk  ;  // cyan-on-black
attr_t   cyre  ;  // cyan-on-red
attr_t   cygr  ;  // cyan-on-green
attr_t   cybr  ;  // cyan-on-brown
attr_t   cybl  ;  // cyan-on-blue
attr_t   cyma  ;  // cyan-on-magenta
attr_t   cygy  ;  // cyan-on-grey

attr_t   gybk  ;  // grey-on-black
attr_t   gyre  ;  // grey-on-red
attr_t   gygr  ;  // grey-on-green
attr_t   gybr  ;  // grey-on-brown
attr_t   gybl  ;  // grey-on-blue
attr_t   gyma  ;  // grey-on-magenta
attr_t   gycy  ;  // grey-on-cyan

//* For systems which have 16 basic terminal colors, 'bright' colors. *
//* (if system has only 8 basic colors, then these will be duplicates *
attr_t  bbkre ;   // black-on-red
attr_t  bbkgr ;   // black-on-green
attr_t  bbkbr ;   // black-on-brown
attr_t  bbkbl ;   // black-on-blue
attr_t  bbkma ;   // black-on-magenta
attr_t  bbkcy ;   // black-on-cyan
attr_t  bbkgy ;   // black-on-grey

attr_t  brebk ;   // red-on-black
attr_t  bregr ;   // red-on-green
attr_t  brebr ;   // red-on-brown
attr_t  brebl ;   // red-on-blue
attr_t  brema ;   // red-on-magenta
attr_t  brecy ;   // red-on-cyan
attr_t  bregy ;   // red-on-grey

attr_t  bgrbk ;   // green-on-black
attr_t  bgrre ;   // green-on-red
attr_t  bgrbr ;   // green-on-brown
attr_t  bgrbl ;   // green-on-blue
attr_t  bgrma ;   // green-on-magenta
attr_t  bgrcy ;   // green-on-cyan
attr_t  bgrgy ;   // green-on-grey

attr_t  bbrbk ;   // brown-on-black
attr_t  bbrre ;   // brown-on-red
attr_t  bbrgr ;   // brown-on-green
attr_t  bbrbl ;   // brown-on-blue
attr_t  bbrma ;   // brown-on-magenta
attr_t  bbrcy ;   // brown-on-cyan
attr_t  bbrgy ;   // brown-on-grey

attr_t  bblbk  ;  // blue-on-black
attr_t  bblre  ;  // blue-on-red
attr_t  bblgr  ;  // blue-on-green
attr_t  bblbr  ;  // blue-on-brown
attr_t  bblma  ;  // blue-on-magenta
attr_t  bblcy  ;  // blue-on-cyan
attr_t  bblgy  ;  // blue-on-grey

attr_t  bmabk  ;  // magenta-on-black
attr_t  bmare  ;  // magenta-on-red
attr_t  bmagr  ;  // magenta-on-green
attr_t  bmabr  ;  // magenta-on-brown
attr_t  bmabl  ;  // magenta-on-blue
attr_t  bmacy  ;  // magenta-on-cyan
attr_t  bmagy  ;  // magenta-on-grey
           
attr_t  bcybk  ;  // cyan-on-black
attr_t  bcyre  ;  // cyan-on-red
attr_t  bcygr  ;  // cyan-on-green
attr_t  bcybr  ;  // cyan-on-brown
attr_t  bcybl  ;  // cyan-on-blue
attr_t  bcyma  ;  // cyan-on-magenta
attr_t  bcygy  ;  // cyan-on-grey

attr_t  bgybk  ;  // grey-on-black
attr_t  bgyre  ;  // grey-on-red
attr_t  bgygr  ;  // grey-on-green
attr_t  bgybr  ;  // grey-on-brown
attr_t  bgybl  ;  // grey-on-blue
attr_t  bgyma  ;  // grey-on-magenta
attr_t  bgycy  ;  // grey-on-cyan


private:
   WINDOW*     stdscrAddress ;   //* Address of WINDOW structure returned by initscr()
   short       sRows ;           //* Number of lines on stdscr
   short       sCols ;           //* Number of columns on stdscr
   short       maxColors ;       //* Maximum number of colors supported by terminal
   short       maxRgb ;          //* Maximum number of RGB register sets supported by terminal
   short       maxPairs ;        //* Maximum number of fg/bg color pairs supported by terminal
   ncCurVis    cursorState ;     //* Cursor visibility state
   ncCurVis    cursorRestore ;   //* Previous cursor visibility state
   ncKeyProc   keyProcState ;    //* Key-input preprocessing level 
   short       keyInputDelay ;   //* Key-input delay in milliseconds
   ncKeyBits   keyBits ;         //* number of significant bits in key input
   keyPushback keyPush ;         //* Key input pushback buffer
   mmask_t     mouseMask ;       //* Current mouse-event bitmask
   mmask_t     oldMouseMask ;    //* Previous mouse-event bitmask
   bool        inputEchoEnabled ;//* true, if key-input-echo is enabled
   bool        mouseEnabled ;    //* true, if mouse driver active AND event reporting enabled
   bool        mouseFilter ;     //* true, if filtering of spurious mouse-event reports enabled
   bool        modRgb ;          //* true, if terminal supports RGB register modification
   bool        modPairs ;        //* true, if terminal supports color-pair modification
   bool        outputScroll ;    //* true, if output-stream scrolling enabled

void  InitColorPairs ( NcBaseColors bkGround = ncbcDEFAULT, bool initExt = true ) ;
void  InitColorVariables ( bool enableColor = true ) ;
void  StoreEscapeSequence ( const wchar_t* srcBuff, short srcCount ) ;
winPos WriteString ( short startY, short startX, gString& ogs, attr_t 
                     cAttr, bool rtl = false ) ;
winPos WriteChar ( short startY, short startX, gString& ogs, 
                   attr_t cAttr, bool refresh = false, bool rtl = false ) ;
wkeyType GetKeyInput ( wchar_t& wChar, bool rawData = false ) ;
short RegetKeyInput ( wkeyCode& wKey ) ;

} ;

#endif   // NCURSES_INCLUDED


