//******************************************************************************
//* File       : Salmon.cpp                                                    *
//* Author     : Mahlon R. Smith                                               *
//*              Copyright (c) 2012-2019 Mahlon R. Smith, The Software Samurai *
//*                  GNU GPL copyright notice below.                           *
//* Date       : 02-Oct-2019                                                   *
//* Version    : (see Version History, below)                                  *
//*                                                                            *
//* Description: This is a simple application to test the functionality of the *
//* 'exec' group of C library functions.                                       *
//*                                                                            *
//* Development Tools:                                                         *
//* Fedora 30     GNOME Terminal v:3.32.2    GCC (G++) v:9.2.1 20190827        *
//* Fedora 20     GNOME Terminal v:3.2.1     GCC (G++) v:4.8.3 20140911        *
//* Fedora 12     GNOME Terminal v:2.x       GCC (G++) v:4.4.4 20100429        *
//******************************************************************************
//* Copyright Notice:                                                          *
//* This program is free software: you can redistribute it and/or modify it    *
//* under the terms of the GNU General Public License as published by the Free *
//* Software Foundation, either version 3 of the License, or (at your option)  *
//* any later version.                                                         *
//*                                                                            *
//* This program is distributed in the hope that it will be useful, but        *
//* WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY *
//* or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License   *
//* for more details.                                                          *
//*                                                                            *
//* You should have received a copy of the GNU General Public License along    *
//* with this program.  If not, see <http://www.gnu.org/licenses/>.            *
//*                                                                            *
//******************************************************************************
//* Version History (most recent first):                                       *
//*                                                                            *
//* v: 0.0.04 27-Sep-2019                                                      *
//*    -- The compiler upgrade to GCC (G++) v:9.0.1 introduced some new        *
//*       warning messages. Specifically, the C function snprintf() now causes *
//*       complaints about potential buffer overrun.                           *
//*            -Wformat-truncation, pragma -Wno-format-truncation              *
//*       We now test for output truncation in snprintf() calls which use      *
//*       an "%s" in the formatting template. Because our data are tightly     *
//*       controlled, this warning is more of an inconvenience than an actual  *
//*       problem, so we do not actually perform error recovery, but in the    *
//*       spirit of best practices we have tweaked the code slightly to        *
//*       silence the warnings. (Note that C formatting functions are too      *
//*       dangerous to be used in a "real" application, but here it's OK.)     *
//*    -- Fedora's move from Xorg to the Wayland compositor has caused a       *
//*       mountain of problems for everyone, and it is no different for this   *
//*       application. Information that was once easily obtainable under the   *
//*       X11 protocol is now literally impossible to obtain. All this is being*
//*       done (theoretically) in the interest of security and hardware        *
//*       independence, but this independence comes at a price.                *
//*       Please see the notes in DefineChildEmulator() method.                *
//*       To overcome Wayland's shortcommings, we are forced to ask you, the   *
//*       user to tell us which terminal emulator to launch in response to the *
//*       "-t" command-line option.                                            *
//*    -- Konsole no longer accepts the "geometry" parameter. Formerly, it     *
//*       just ignored the parameter, but now it chokes.                       *
//*    -- Documentation update.                                                *
//*    -- Posted to website 02-Oct-2019.                                       *
//*                                                                            *
//* v: 0.0.03 04-Jun-2017                                                      *
//*    -- Documentation update only.                                           *
//*                                                                            *
//* v: 0.0.02 05-May-2017 With enhancements                                    *
//*    -- Add command-line help.                                               *
//*    -- Add option '-s' to spawn a child process with stdout and stderr      *
//*       redirected to files. This is an experiment in launching an arbitrary *
//*       external application from within another application without         *
//*       disturbing the parent's display data.                                *
//*    -- Add option '-t' to spawn a child process which open a new terminal   *
//*       window based on the terminal emulator and shell program of the       *
//*       parent process.                                                      *
//*    -- Enhance the existing options to more clearly demonstrate the         *
//*       similarities and differences of the various 'exec' functions.        *
//*    -- Create a README file and basic HTML documentation.                   *
//*                                                                            *
//* v: 0.0.01 25-Aug-2012 First pass.                                          *
//*                        Created using GNU G++ (Gcc v: 4.4.2)                *
//*                        under Fedora Linux, 2.6.31-5-127.fs12.i686.PAE      *
//*    -- We apologize for the application name. This a project written        *
//*       strictly for fun. "Salmon" swim upstream to spawn fish eggs, not     *
//*       process IDs, but we hope you get the idea.                           *
//******************************************************************************
//* Programmer's Notes:                                                        *
//*                                                                            *
//* Please see the 'README' file for more extensive notes.                     *
//*                                                                            *
//* Exec functions called by this application:                                 *
//* ------------------------------------------                                 *
//* '-v' option:                                                               *
//*    int execv ( const char* fileName, char const* argV[] ) ;                *
//*        argV[0] s/b bare filename                                           *
//*        argV[1], ...  additional arguments                                  *
//*        argV[n] NULL*                                                       *
//*                                                                            *
//* '-l' and '-f' options:                                                     *
//* int execl ( const char* fileName, const char* argVa, ... ) ;               *
//*     argVa   s/b bare filename                                              *
//*     argVb, ...  addional arguments                                         *
//*     argVn   NULL*                                                          *
//*                                                                            *
//* '-p' and '-t' options:                                                     *
//* int execlp ( const char* fileName, const char* argVa, ... ) ;              *
//*     same as execl, except it searches the PATH for fileName                *
//*                                                                            *
//* '-s' option:                                                               *
//* int execvp ( const char* fileName, char *const argV[] ) ;                  *
//*     same as execv, except it searches the PATH for fileName                *
//*                                                                            *
//******************************************************************************

//** Include files **
#include "GlobalDef.hpp"

//** Local Definitions **
static short DefineChildEmulator ( char* emuName, int enSIZE, char* shellName,
                                   int& xPos, int& yPos, int& cols, int& rows ) ;
static void helpMe ( void ) ;

static const char* const AppVersion = "0.0.04" ;
static const char* const AppYear = "2019" ;

//*************************
//*        main           *
//*************************
//******************************************************************************
//* Program entry point.                                                       *
//*                                                                            *
//* Command-line Usage: see the helpMe() method                                *
//*                                                                            *
//******************************************************************************

int main ( int argc, char const* argv[], char const* argenv[] )
{
   if ( argc > 1 && argv[1][0] == DASH && argv[1][1] != NULLCHAR )
   {
      system ( "clear" ) ;          // clear the decks

      //* Say hello and report the user's command-line arguments *
      wcout << "Salmon Swim Upstream To Spawn!\n"
            << " Invocation: \"" ;
      for ( short i = ZERO ; i < argc ; ++i )
      {
         if ( i > 1 )  wcout << L"'" ;
         wcout << argv[i] ;
         if ( i > 1 ) wcout << L"'" ;
         if ( i < (argc - 1) )
            wcout << " " ;
      }
      pid_t pPID = getpid () ;      // get process ID
      wcout << "\"\n" 
            << " Parent PID: " << pPID << endl ;
   
      char oparg = argv[1][1] ;     // command switch
      
      //*****************
      //**  -v option  **
      //*****************
      if ( oparg == 'v' )
      {
         //* Default arguments for calling the child application *
         const char* trgName   = "childApp" ;
         const char* trgSwitch = "-v" ;
         const char* trgTitle  = "Types of Salmon:" ;
         const char* salmonA   = "Trout" ;
         const char* salmonB   = "Char" ;
         const char* salmonC   = "Grayling" ;
         const char* salmonD   = "Whitefish" ;
         const char* avArray[] = 
         {
            trgName,
            trgSwitch,
            trgTitle,
            salmonA,
            salmonB,
            salmonC,
            salmonD,
            NULL
         } ;
         //* If user has sent us at least one non-option argument,  *
         //* replace static arguments with user's arguments (max 4).*
         if ( argc > 2 )
            avArray[3] = argv[2] ;
         if ( argc > 3 )
            avArray[4] = argv[3] ;
         if ( argc > 4 )
            avArray[5] = argv[4] ;
         if ( argc > 5 )
            avArray[6] = argv[5] ;

         //* Invoke the external application (this application terminates) *
         wcout << " calling execv( childApp, argv[] )\n" << endl ;
         execv ( "./childApp", (char* const*)avArray ) ;
         wcout << " Error! execv() failed." << endl ;
      }

      //*****************
      //**  -l option  **
      //*****************
      else if ( oparg == 'l' )
      {
         //* If user has sent us non-option arguments (max 4), pass them on *
         //* to the child application. Else, send only argv[0] and argv[1]. *
         //* If execl() call is successful, this application terminates.    *
         wcout << " calling execl( childApp ... )\n" << endl ;
         int status = execl ( "./childApp", "childApp", "-l", 
                              (argc > 2 ? argv[2] : NULL),
                              (argc > 3 ? argv[3] : NULL),
                              (argc > 4 ? argv[4] : NULL),
                              (argc > 5 ? argv[5] : NULL),
                              NULL ) ;
         if ( status == -1 )
            wcout << " Error! execl() failed." << endl ;
      }

      //*****************
      //**  -p option  **
      //*****************
      else if ( oparg == 'p' )
      {
         //* Invoke the 'grep' utility which is an example of an         *
         //* application on the path.on the path. Use 'grep' to scan     *
         //* both makefiles for the "COMPILE" definition.                *
         //* If execlp() call is successful, this application terminates.*
         wcout << " calling execlp( grep -n 'COMPILE' Makefile MakeChild )\n" << endl ;
         execlp ( "grep", "grep", "-n", "COMPILE", "Makefile", "MakeChild", NULL ) ;
         wcout << " Error! execlp() failed." << endl ;
      }

      //*****************
      //**  -f option  **
      //*****************
      else if ( oparg == 'f' )
      {
         wcout << " Spawn a child process using fork(), then call execl( childApp ... )\n" 
               << endl ;

         pid_t fpid = vfork () ;       // create the child process

         if ( fpid == ZERO)
         {  //* The child process (if created) executes here.                  *
            //* If user has sent us non-option arguments (max 4), pass them on *
            //* to the child application. Else, send only argv[0] and argv[1]. *
            int status = execl ( "./childApp", "childApp", "-f", 
                                 (argc > 2 ? argv[2] : NULL),
                                 (argc > 3 ? argv[3] : NULL),
                                 (argc > 4 ? argv[4] : NULL),
                                 (argc > 5 ? argv[5] : NULL),
                                 NULL ) ;
            if ( status == -1 )
               wcout << " Error! execl() failed." << endl ;

            //* In case the exec call fails: child process MUST NOT continue.*
            _exit ( 0 ) ;
         }
         else
         {  //* The parent process continues execution here *
            if ( fpid > ZERO )      // child successfully launched
            {
               wchar_t cStatus[16] ;
               int childStatus = ZERO ;   // receives child's exit status
               int statusOptions = ZERO ; // status bitmask (none defined at this time)

               //* Sleep until the child process terminates.  *
               //* waitpid() _should_ return the child's PID, *
               //* but if not, then check errno for:          *
               //* EINTR || ECHILD || EINVAL                  *
               int  rval = waitpid ( fpid, &childStatus, statusOptions ) ;

               wcout << "Parent (" << pPID << ") has regained control." << endl ;
               if ( rval == fpid )
               {
                  swprintf ( cStatus, 16, L"0x%04X", childStatus ) ;
                  wcout << "Child process " << rval 
                        << " has terminated with a status of " << cStatus << endl ;
               }
               else
               {
                  swprintf ( cStatus, 16, L"0x%04X", errno ) ;
                  wcout << "waitpid() has returned an error condition. errno: " 
                        << cStatus << endl ;
               }
            }
            else
            {
               wcout << " fork() failed. PID: " << fpid << endl ;
            }
         }
      }

      //*****************
      //**  -s option  **
      //*****************
      else if ( oparg == 's' )
      {  //* Launch a child process with all its output redirected to *
         //* temporary files, so that it does not interfere with the  *
         //* output of the parent process.                            *
         const char* childApp = "./childApp" ;  // name of app that child invokes
         const char* childArgs[] =              // child's argv[] array
                     {
                        "childApp",             // argv[0]
                        "-s",                   // argv[1]
                        "Hello stdout!\n",      // argv[2] (written to stdout)
                        "Hello stderr!\n",      // argv[3] (written to stderr)
                        NULL                    // terminate the list
                     } ;
         const char* soFName = "./sout.txt" ;   // stdout target file
         const char* seFName = "./serr.txt" ;   // stderr target file
         int soDesc = ZERO,                     // stdout target descriptor
             seDesc  ;                          // stderr target descriptor

         pid_t fpid = vfork () ;       // create the child process

         if ( fpid == ZERO)
         {  //* The child process (if created) executes here.         *
            //* 1) Create temporary files for 'stdout' and 'stderr'.  *
            //* 2) Redirect 'stdout' and 'stderr' to their respective *
            //*    temporary files using the 'dup2()' function.       *
            //* 3) Close the temporary files.                         *
            // Programmer's Note: We open and close these files using 
            // kernel-level file descriptors because getting the underlying 
            // file descriptor from the C++ fstream object is implementation 
            // dependent and therefore unreliable.
            soDesc = open ( soFName, O_WRONLY | O_CREAT | O_TRUNC, 
                              S_IRUSR | S_IWUSR | S_IRGRP | S_IROTH ) ;
            seDesc = open ( seFName, O_WRONLY | O_CREAT | O_TRUNC, 
                              S_IRUSR | S_IWUSR | S_IRGRP | S_IROTH ) ;
            dup2 ( soDesc, STDOUT_FILENO) ;
            dup2 ( seDesc, STDERR_FILENO) ;
            close ( soDesc ) ;
            close ( seDesc ) ;

            //* If execvp() call is successful, the child process will not *
            //* return to the parent application, else the return value    *
            //* will be written to the terminal.                           *
            //* Possible causes of execvp() failure would be insufficient  *
            //* system resources or maximum number of processes reached.   *
            wcerr << execvp ( childApp, (char* const* )childArgs ) ;

            wcerr << " Error! Child process call to execvp() failed." << endl ;

            //* In case the exec call fails: child process MUST NOT continue.*
            _exit ( 0 ) ;
         }
         else
         {  //* The parent process continues execution here *
            //* Wait a moment (in case the child PID *
            //* writes an error message to display)  *
            chrono::duration<short, std::milli>aMoment( 250 ) ;
            this_thread::sleep_for( aMoment ) ;

            if ( fpid > ZERO )      // child successfully launched
            {
               wcout << " Mother salmon spawned child PID: " << fpid 
                     << "\n Type  'cat *.txt'  to view redirected output."
                     << endl ;
            }
            else
            {
               wcout << " Spawn failed: (" << fpid << "), Mother salmon is childless." << endl ;
            }
         }
      }

      //*****************
      //**  -t option  **
      //*****************
      else if ( oparg == 't' )
      {  //* Launch a child process which will open a new terminal window *
         //* and will executes commands within it.                        *
         const int enSIZE = 64 ;
         int xPos, yPos, cols, rows ;
         char emulatorName[enSIZE] ;
         char shellName[enSIZE] ;

         //* Scan the system configuration to define the new terminal window.*
         short cmdOption = 
         DefineChildEmulator ( emulatorName, enSIZE, shellName, xPos, yPos, cols, rows ) ;

         // Programmer's Note: Please note the test for output truncation in 
         // snprintf() calls which use an "%s" in the formatting template.
         // See notes in module header.
         char geo[64] ;
         char cmd[128] ;
         switch ( cmdOption )
         {
            case 0:     // gnome-terminal
               snprintf ( geo, 64, "--geometry=%dx%d+%d+%d", cols, rows, xPos, yPos ) ;
               if ( (snprintf ( cmd, 128, 
                     "%s -c \"grep -n \'COMPILE\' Makefile MakeChild ; exec %s\"",
                     shellName, shellName )) >= 128 ) ;
               break ;
            case 1:     // konsole
               snprintf ( geo, 64, "--geometry=%dx%d+%d+%d", cols, rows, xPos, yPos ) ;
               if ( (snprintf ( cmd, 128, "%s -c \"grep -n \'COMPILE\' Makefile MakeChild ; exec %s\"",
                          shellName, shellName )) >= 128 ) ;
               break ;
            case 2:     // xterm
               snprintf ( geo, 64, "%dx%d+%d+%d", cols, rows, xPos, yPos ) ;
               if ( (snprintf ( cmd, 128, "%s -c \"grep -n \'COMPILE\' Makefile MakeChild ; exec %s\"",
                          shellName, shellName )) >= 128 ) ;
               break ;
         } ;

         pid_t fpid = vfork () ;       // create the child process

         if ( fpid == ZERO)
         {  //* The child process (if created) executes here.              *
            //* Programmer's Note: Because this command is rather complex, *
            //* let's go through it step-by-step. Note that each argument  *
            //* to the execlp() function is a string. To put it another    *
            //* way, each argument is interpreted as a const char*, and    *
            //* arguments are separated by commas creating an array of     *
            //* pointers, which cumulatively are the argv[] array.         *
            //* The arguments described are specific to the GNOME terminal *
            //* and to the 'bash' shell program but other terminals and    *
            //* shells follow the same general pattern.                    *
            //* 1) 'execlp'                                                *
            //*    This function carries the arguments to the system.      *
            //* 2) 'emulatorName'                                          *
            //*    This is the name of the terminal emulation program,     *
            //*    which is assumed to be on the path where execlp() can   *
            //*    find it.                                                *
            //* 3) 'emulatorName'                                          *
            //*    This is the path-less emulator name which constitutes   *
            //*    argv[0].                                                *
            //* 4) 'geo'                                                   *
            //*    This string is the geometry i.e. the size and position  *
            //*    of the new window. --geometry=COLSxROWS+XOFFSET+YOFFSET *
            //*    Example: --geometry=65x8+400+300                        *
            //*       Note: KDE Konsole has a long-standing bug which      *
            //*             ignores this command, and/or interprets the    *
            //*             dimensions as pixels counts rather than        *
            //*             as columns and rows. Both of these scenarios   *
            //*             are inexcusable. Shame on you, KDE Project!    *
            //*             Their solution is to remove the "geometry"     *
            //*             parameter altogether and to handle window      *
            //*             dimensions entirely within the "profile".      *
            //*             Again, shame on you KDE Project!               *
            //*       Note: XTerm options begin with a single dash and do  *
            //*             not allow the '=' to join the option and its   *
            //*             argument.                                      *
            //*                 Example: -geometry 65x8+400+300            *
            //*                 (this is two separate arguments)           *
            //* 5) 'hide-menubar'                                          *
            //*    Does what it says, prevents the menu bar from being     *
            //*    displayed in the new window.                            *
            //* 6) '--window' (gnome-terminal only): By default, gnome-    *
            //*    terminal opens a new tab in the current window, so we   *
            //*    must explicitly specify that the new terminal be opened *
            //*    in a new window.                                        *
            //* 7) '-q' switch (gnome-terminal only): silent launch i.e.   *
            //*    child process will not write to the parent window.      *
            //*    The other emulators do not offer this option.           *
            //* 8) '-e' switch: This is the terminal emulator's "Execute"  *
            //*    command. What follows it is the command to be executed  *
            //*    inside the new terminal window. The '-e' switch is used *
            //*    by all three emulators: gnome-terminal, konsole and     *
            //*    xterm,                                                  *
            //*    Programmer's Note: The Gnome developers have deprecated *
            //*    the '-e' option in favor of '--' however, we have       *
            //*    encouraged them to re-think that decision because it    *
            //*    is needed when starting a terminal with multiple tabs.  *
            //* 9) 'cmd'                                                   *
            //*    This is the main argument. These are the commands to    *
            //*    be executed within the new terminal window.             *
            //*    See the template above which was used to create the     *
            //*    contents of the 'cmd' string.                           *
            //*    a) Note that the entire argument is enclosed in         *
            //*       quotation marks AND that the part of the argument    *
            //*       that is the command to be executed is ALSO enclosed  *
            //*       in (escaped) quotation marks.                        *
            //*    b) 'bash -c'                                            *
            //*       This runs a copy of the 'bash' shell program inside  *
            //*       the new terminal window.                             *
            //*       The '-c' argument to the shell program indicates     *
            //*       that the following argument is to be executed by the *
            //*       shell. Most shell programs use the "-c" command      *
            //*       switch for this, but if you are using a shell        *
            //*       program other than bash, zsh or csh, please verify   *
            //*       the switch used by your shell.                       *
            //*       Note that if you do not require the window to remain *
            //*       open after execution, the shell specification may    *
            //*       be omitted.                                          *
            //*    c) 'grep -n \'COMPILE\' Makefile MakeChild'             *
            //*       This is the command to be executed by the shell      *
            //*       program. In this case it is the 'grep' command and   *
            //*       its arguments, exactly the same as in the '-p'       *
            //*       option of the application (see above). The only      *
            //*       difference between this call to 'grep' and the one   *
            //*       in the '-p' option above is that the call is         *
            //*       preceeded by an escaped quotation mark: \"grep...    *
            //*       The matching escaped quotation mark follows          *
            //*       (see item e) below:  ...exec bash\"                  *
            //*       Placing the backslash before the quotation marks     *
            //*       prevents the shell from removing them before the     *
            //*       command is executed, allowing it to be seen as a     *
            //*       single argument.                                     *
            //*       Similarly, the single quotes surrounding the search  *
            //*       string, 'COMPILE' are also escaped to prevent the    *
            //*       shell from removing them before passing the arguments*
            //*       to grep. This allows for whitespace to be included   *
            //*       in the search string. For example \'OFILES = *\'     *
            //*    d) ';'                                                  *
            //*       The semicolon character separates the the 'grep'     *
            //*       call from the 'exec' call, i.e. these are two        *
            //*       SEPARATE commands.                                   *
            //*    e) 'exec bash'                                          *
            //*       This command _replaces_ the currently running copy   *
            //*       of 'bash' with a new copy.                           *
            //*       This may seem odd, but without the 'exec' two copies *
            //*       of the shell would be left running in the window.    *
            //*       Note that an escaped quotation mark follows this     *
            //*       command which closes the escaped quotation pair      *
            //*       (see item c) above.                                  *
            //*       The _effect_ of this command is to keep the new      *
            //*       window open and to return to the command prompt      *
            //*       after all the specified commands have finished.      *
            //*       If you want the window to close when finished,       *
            //*       then omit the 'exec bash'.                           *
            //* 10) NULL                                                   *
            //*    This is the NULL pointer indicating the end of the      *
            //*    array of pointers.                                      *
            //*       -----  -----  -----  -----  -----  -----  -----      *
            //* Working directory: For gnome-terminal and xterm the        *
            //* working directory is taken from the parent's $PWD          *
            //* environment variable. However, Konsole defaults to the     *
            //* directory specified in the profile, so we must explicitly  *
            //* redirect it to the PWD using: "--workdir=."                *

            switch ( cmdOption )
            {
               case 0:     // gnome-terminal
                  execlp ( emulatorName, emulatorName, geo, "--hide-menubar", 
                           "--window", "-q",
                           "-e", cmd, NULL ) ;  // Note: "-e" option is deprecated.
                  break ;
               case 1:     // konsole
                  execlp ( emulatorName, emulatorName, "--hide-menubar", 
                           "--workdir=.",
                           "-e", cmd, NULL ) ;
                  break ;
               case 2:     // xterm
                  execlp ( emulatorName, emulatorName, "-geometry", geo, 
                           "-fa", "Monospace", "-fs", "12",
                           "-e", cmd, NULL ) ;
                  break ;
            } ;

            wcout << " Error! Child process call to execlp() failed." << endl ;

            //* In case the exec call fails: child process MUST NOT continue.*
            _exit ( 0 ) ;
         }
         else
         {  //* The parent process continues execution here *
            //* Wait a moment (in case the child PID *
            //* writes an error message to display)  *
            chrono::duration<short, std::milli>aMoment( 250 ) ;
            this_thread::sleep_for( aMoment ) ;

            if ( fpid > ZERO )      // child successfully launched
            {
               wcout << " Mother salmon spawned child PID: " << fpid 
                     << " in a new terminal window." << endl ;
            }
            else
            {
               wcout << " Spawn failed: (" << fpid << "), Mother salmon is childless." << endl ;
            }
         }
      }
      else if ( oparg == 'D' )   // TEMP - DEVELOPMENT AND EXPERIMENTATION
      {  //* For development and debugging only.                  *
         //* Execute experimental commands at the command prompt. *
         wcout << L"Experimental Commands:" << endl ;
//         system ( "loginctl show-session $(loginctl | grep $(whoami) | awk '{print $1}') -p Type --value" ) ;
//         system ( "loginctl show-session $(awk '/tty/ {print $1}' <(loginctl)) -p Type | awk -F= '{print $2}'" ) ;
//         system ( "ps -o 'ppid=' -p $$" ) ;
//         system ( "ps -o 'cmd=' -p $(ps -o 'ppid=' -p $$)" ) ;
//         system ( "ps -o 'cmd=' -p $(ps -o 'ppid=' -p $(ps -o 'ppid=' -p $$))" ) ;
//         system ( "ps -o 'cmd=' -p $(ps -o 'ppid=' -p $(ps -o 'ppid=' -p $(ps -o 'ppid=' -p $$)))" ) ;
//         system ( "echo $WINDOWID" ) ;
//         system ( "xwininfo -id $WINDOWID -wm | grep 'Process id:'" ) ;
//         system ( "xwininfo -wm -id $WINDOWID | grep 'Process id:' | grep -o '[[:digit:]]*' | grep '[[:digit:]]' --max-count=1" ) ;
//         system ( "ps -o 'cmd=' -p $(xwininfo -wm -id $WINDOWID | grep 'Process id:' | grep -o '[[:digit:]]*' | grep '[[:digit:]]' --max-count=1)" ) ;
//         system ( "xterm -geometry 65x10+400+300 -fn fixed -fs 16 -e \"bash -c grep --version\" ; exec bash" ) ;
//         execlp ( "xterm", "xterm", "-fn", "fixed", "-fs", "18", "-geometry", "80x16+550+350",
//         execlp ( "xterm", "xterm", "-fn", "fixed", "-fs", "9x18", "-geometry", "80x8+550+350",
//         execlp ( "gnome-terminal", "gnome-terminal", "--geometry=65x9+400+300", "--hide-menubar",
//                  "-e", "bash -c \"grep -n \'COMPILE\' Makefile MakeChild ; exec bash\"", NULL ) ;
//         execlp ( "xterm", "xterm", "-fa", "Monospace", "-fs", "12", "-geometry", "80x8+550+350",
//                  "-e", "bash -c \"gcc --version ; exec bash\"", NULL ) ;
//         execlp ( "konsole", "konsole", "--geometry=1200x800+550+350", "--hide-menubar",
//                  "-e", "bash -c \"gcc --version ; exec bash\"", NULL ) ;
      }
      else
         helpMe () ;
   }
   else
      helpMe () ;

   wcout << endl ;

   return ZERO ;

}  //* End main() *

//*************************
//*  DefineChildEmulator  *
//*************************
//******************************************************************************
//* Determine the size and position for a new terminal-emulator window.        *
//*  Dimensions: large enough to hold our message.                             *
//*    Position: centered in the parent terminal's window.                     *
//*                                                                            *
//* Input  : emuName : receives the name of the terminal emulator program in   *
//*                    which this application is running                       *
//*          enSize  : maximum bytes in 'emuName' and 'shlName' buffers        *
//*          shlName : receives the name of the shell program as specified in  *
//*                    the terminal environment                                *
//*          xPos    : (by reference) Receives upper left corner X (in pixels) *
//*                    for the new window                                      *
//*          yPos    : (by reference) Receives upper left corner Y (in pixels) *
//*                    for the new window                                      *
//*          cols    : (by reference) Receives number of columns for the new   *
//*                    terminal window                                         *
//*          rows    : (by reference) Receives number of rows for the new      *
//*                    terminal window                                         *
//*                                                                            *
//* Returns: 0   gnome-terminal (or unknown terminal)                          *
//*          1   konsole                                                       *
//*          2   xterm                                                         *
//******************************************************************************
//* Programmer's Note:                                                         *
//* This method is used to define a new terminal-emulator window which will be *
//* launched by a child process.                                               *
//*                                                                            *
//*         It is not necessary to understand what this method                 *
//*         is doing in order to understand the exec functions.                *
//*                                                                            *
//* 1) The algorithm is not foolproof; it is a simplistic way of identifying   *
//*    the terminal emulator in which we are running. While there are at least *
//*    twenty terminal emulator programs in general use, for our purposes we   *
//*    identify only three:                                                    *
//*    a) GNOME terminal                                                       *
//*    b) KDE konsole                                                          *
//*    c) xterm                                                                *
//*    If your terminal emulator is not one of these, and if 'gnome-terminal'  *
//*    is not installed on your system, then the '-t' option of this           *
//*    application will probably fail.                                         *
//*                                                                            *
//*    Also, there are many different shell programs in common use, so we      *
//*    rely on the $SHELL environment variable to tell us which one to use.    *
//*    If unable to get this information, we default to 'bash'.                *
//*                                                                            *
//* 2) We could simply open a small terminal window anywhere on the desktop,   *
//*    but instead, we demonstrate how to gather information about the desktop *
//*    screen and the existing terminal-emulator window, and how to position   *
//*    and size a new terminal window. We use the 'xwininfo' utility for this  *
//*    operation.                                                              *
//*    a) If you are not running under the X-server or if there is some aspect *
//*       of your system or environment which affects the 'xwininfo' utility,  *
//*       then the values returned will be an approximation only.              *
//*    b) Note that some intimate assumptions are made about the format of the *
//*       information provided by 'xwininfo'.                                  *
//*    c) Note also that our parsing of the data is grossly inefficient.       *
//*       In a production-worthy application, we would use our gString text    *
//*       tool for parsing. However, since we _refuse_ to use the seriously    *
//*       brain-damaged std::string class, we instead use the C-language       *
//*       sscanf() primitive for scanning the data.                            *
//*                                                                            *
//*                                                                            *
//* Opening a new terminal window.                                             *
//* ------------------------------                                             *
//* 1) Opening a new terminal emulator window requires knowing what terminal   *
//*    emulator is in use. Unfortunately, we have not found a way to directly  *
//*    query the system for this information. The most direct (but klunky) way *
//*    we have so far discovered is the 'ps' (process status) utility.         *
//* 2) Using 'ps':                                                             *
//*    Get the parent's PID: ps -o 'ppid=' -p $$                               *
//*    Use the parent's PID to find the command which was used to invoke it,   *
//*    i.e. the name of the parent process.                                    *
//*            ps -o 'cmd=' -p $(ps -o 'ppid=' -p $$)                          *
//*              yields: /usr/libexec/gnome-terminal-server                    *
//*    See notes in "Terminal Emulator Name.odt" for more information.         *
//*                                                                            *
//* Positioning a new window on the desktop.                                   *
//* ----------------------------------------                                   *
//* 1) To size and position the new window, we need the dimensions of the      *
//*    desktop in pixels, the size of the active terminal window in pixels as  *
//*    well as the size of the active terminal in rows and columns.            *
//* 2) There are a number of tools for obtaining this information.             *
//*    a) 'xwininfo' is an interactive tool. It gives exactly the information  *
//*       needed (indicated by the arrows below).                              *
//*       The utility requires direct user action by default; but if you       *
//*       have the window ID, then:                                            *
//*         xwininfo -id $WINDOWID -stats    yields:                           *
//* --->>      Absolute upper-left X:  41                     (pixels)         *
//* --->>      Absolute upper-left Y:  66                     (pixels)         *
//*            Relative upper-left X:  10                     (pixels)         *
//*            Relative upper-left Y:  47                     (pixels)         *
//* --->>      Width: 1874                                    (pixels)         *
//* --->>      Height: 952                                    (pixels)         *
//*            Depth: 32                                                       *
//*            Visual: 0xaa                                                    *
//*            Visual Class: TrueColor                                         *
//*            Border width: 0                                                 *
//*            Class: InputOutput                                              *
//*            Colormap: 0x1a00006 (not installed)                             *
//*            Bit Gravity State: NorthWestGravity                             *
//*            Window Gravity State: NorthWestGravity                          *
//*            Backing Store State: NotUseful                                  *
//*            Save Under State: no                                            *
//*            Map State: IsViewable                                           *
//*            Override Redirect State: no                                     *
//*            Corners:  +41+66  -5+66  -5-62  +41-62         (pixels)         *
//* --->>      -geometry 156x37--5+19                         (cols and rows)  *
//*            NOTE: For Konsole, the columns and rows are returned as pixel   *
//*                  counts. This is actually a bug in Konsole.                *
//*       This, and other information (including the terminal window's         *
//*       process ID (PID) may be obtained.                                    *
//*         xwininfo -id $WINDOWID -wm     yields:                             *
//*         Window manager hints:                                              *
//*             Client accepts input or input focus: Yes                       *
//*             Initial state is Normal State                                  *
//*             Displayed on desktop 0                                         *
//*             Window type:                                                   *
//*                 Normal                                                     *
//*             Window state:                                                  *
//*                 Focused                                                    *
//*             Process id: 2304 on host sam540                                *
//*             Frame extents: 1, 1, 39, 1                                     *
//*                                                                            *
//*    b) 'xprop' is a very similar utility which also requires the user to    *
//*       click on the window by default. However the 'xprop -id ID' can be    *
//*       used instead of the click if the window ID is known.                 *
//*       Similarly, 'xprop -name NAME' can select a window by name.           *
//*       To get the terminal window's ID or NAME, we need to access the       *
//*       X Server to get reliable info.                                       *
//*       It may not be reliable, but if the environment includes the variable:*
//*           WINDOWID=1234567                                                 *
//*       Therefore: xprop -id $WINDOWID gives us information on the current   *
//*       terminal window.                                                     *
//*            WM_WINDOW_ROLE(STRING) =                                        *
//*                          "gnome-terminal-window-2326-475376441-1494415641" *
//*       A specific entry may be specified:                                   *
//*            xprop -id $WINDOWID WM_WINDOW_ROLE                              *
//*                                                                            *
//*    c) 'xdpyinfo' gives us a lot of information about the display screen.   *
//*          xdpyinfo | grep 'dimensions:'                                     *
//*            yields: dimensions:    1920x1080 pixels (508x285 millimeters)   *
//*          xdpyinfo | grep -A2 'screen #'                                    *
//*            yields: screen #0:                                              *
//*                    dimensions:    1920x1080 pixels (508x285 millimeters)   *
//*                    resolution:    96x96 dots per inch                      *
//*                                                                            *
//*    d) 'xrandr'                                                             *
//*         xrandr | grep '*'                                                  *
//*           yields: 1920x1080     60.01*+                                    *
//*       This is the dimensions of the display in pixels.                     *
//*                                                                            *
//*    e) 'xdotool':                                                           *
//*       This is a useful utility, but is generally not installed by default. *
//*       Opening a new tab in the current terminal window:                    *
//*          WID= xprop -root | grep "_NET_ACTIVE_WINDOW(WINDOW)"|             *
//*            awk '{print $5}'; xdotool windowfocus $WID;                     *
//*            xdotool key ctrl+shift+t $WID                                   *
//*                                                                            *
//*                                                                            *
//* Opening a new terminal window:                                             *
//* ------------------------------                                             *
//*                                                                            *
//* gnome-terminal --geometry=80x8+400+300                                     *
//*        -e "bash -c \"grep -n \'COMPILE\' Makefile MakeChild ; exec bash\"" *
//*                                                                            *
//* a) Calling bash before and after the command will keep the terminal        *
//*    window open.                                                            *
//* b) Current-working-directory is default for GNOME --working-directory.     *
//*    Home directory is default for Konsole --workdir.                        *
//*    Current-working-directory is default for XTerm -working-directory.      *
//*    To explicity set the directory: --working-directory=$PWD                *
//*    To specify another directory  : --working-directory=$HOME/temp          *
//*    Keep in mind that the specification must be an absolute path, not a     *
//*    relative path an that the '~' shortcut character is not recognized.     *
//* c) Pipes and redirection of output also work:                              *
//*    gnome-terminal --geometry=80x8+400+300                                  *
//*      -e "grep -n 'COMPILE' Makefile MakeChild | grep 'childApp' >tmp.txt"  *
//*    In this case, the window closes immediately, but our answer is in       *
//*    the temp file.                                                          *
//* d) Note that the example above describes the invocation of the Gnome       *
//*    terminal, and the bash shell. The equivalent command for Konsole or     *
//*    other terminals and for zsh and other shells may require a slightly     *
//*    modified command structure.                                             *
//*                                                                            *
//* Wayland:                                                                   *
//* ========                                                                   *
//* -- When creating a new terminal window, we need to identify which terminal *
//*    emulator we are running under. Unfortunately, gnome-terminal (which is  *
//*    now a native Wayland app) no longer provides the $WINDOWID environment  *
//*    variable. Also, "wmctrl -ia $gnome_term_winid" no longer works because  *
//*    Wayland does not support 'wmctrl'.                                      *
//*    -- Because the call to 'ps' requires a window ID we don't have access   *
//*       to, we get the error:                                                *
//*            xwininfo: error: -id requires argument                          *
//*            error: list of process IDs must follow -p                       *
//*    -- gnome-terminal '--command' ('-e') option is now deprecated.          *
//*       It is recommended to use '--' instead; however, that is not          *
//*       functional.                                                          *
//*    -- Launching a new gnome-terminal now opens in a new tab by default.    *
//*       We asked for this feature when GNOME 3.0 was first release.          *
//*       Well done, GNOME team.                                               *
//*       However, in this case we _want_ a new window, not a tab, so          *
//*       add "--window" parameter.                                            *
//*    -- Konsole has dropped the "geometry" parameter in favor of specifying  *
//*       a different profile for each situation. Inflexible, but in some way  *
//*       an understandable move. The problem is that we are stuck using the   *
//*       default profile, so we can't create a cute little window.            *
//*                                                                            *
//*                                                                            *
//******************************************************************************

static short DefineChildEmulator ( char* emuName, int enSIZE, char* shellName,
                                  int& xPos, int& yPos, int& cols, int& rows )
{
   #define EXAMINE_KONSOLE_BUG (0)      // enable debug of konsole size/position

   const char* soFName = "./sout.txt" ; // name of temp file
   short status = ZERO ;                // return value (gnome is default)

   //* Read the name of the shell program from the terminal environment *
   system ( "echo ${SHELL} >./sout.txt" ) ;

   //*********************************
   //* Are we running under Wayland? *
   //*********************************
   bool wayland = false ;
   system ( "loginctl show-session $(loginctl | grep $(whoami) "
            "| awk '{print $1}') -p Type --value >>./sout.txt" ) ;
   ifstream ifs ( soFName, ifstream::in ) ;  // open the file
   if ( ifs.is_open() )             // if input file open
   {
      //* Read the first line in the file (shell program) *
      ifs.getline ( shellName, enSIZE, NEWLINE ) ;
      if ( (ifs.good()) || (ifs.gcount() > ZERO) )
      {
         //* Isolate the filename *
         short i = ZERO ;
         while ( (i < enSIZE) && (shellName[i] != '\0') )  // end-of-text
            ++i ;
         while ( (i >= 1) && (shellName[i - 1] != '/') )   // app name
            --i ;
         for ( short j = ZERO ; j < enSIZE ; ++j )
         {
            shellName[j] = shellName[i++] ;
            if ( shellName[j] == '\0' )
               break ;
         }

         //* Read the second line in the file.     *
         //* (probably "Wayland", "X11" or "Xorg") *
         ifs.getline ( emuName, enSIZE, NEWLINE ) ;
         if ( (ifs.good()) || (ifs.gcount() > ZERO) )
         {
            if ( (strncmp ( emuName, "wayland", 7 )) == ZERO )
               wayland = true ;
         }
      }
      else     // file access error, return default shell
         strncpy ( shellName, "bash", enSIZE ) ;

      ifs.close() ;                 // close the source file
   }

   if ( wayland != false )
   {
      //* Ask user what terminal emulator to spawn. *
      wcout << "Wayland is blocking access to the terminal-emulator's name.\n"
               "Please indicate the current terminal:\n"
               "  1) Konsole\n"
               "  2) XTerm\n"
               "  3) Gnome\n"
               "  4) Other\n"
               "  " ;
      char response = (char)getwchar () ;
      switch ( response )
      {
         case '1':
            strncpy ( emuName, "konsole", enSIZE ) ;
            status = 1 ;
         break ;
         
         case '2':
            strncpy ( emuName, "xterm", enSIZE ) ;
            status = 2 ;
            break ;
         
         case '3':
         case '4':
         default:
            strncpy ( emuName, "gnome-terminal", enSIZE ) ;
            status = 0 ;
         break ;
      } ;

      //* Set terminal window size and position ("geometry").  *
      //* Note that the terminal invocation may ignore some or *
      //* all of these values. Note also that Wayland does not *
      //* allow window positioning.                            *
      xPos = 100 ; yPos = 70 ; cols = 65 ; rows = 8 ;
   }

   //* Else, assume that we are running undex X11 *
   else
   {
      //* Extract the process ID (PID) of the       *
      //* terminal emulator in which we are running.*
      system ( "ps -o 'cmd=' -p $(xwininfo -wm -id $WINDOWID | grep 'Process id:' "
               "| grep -o '[[:digit:]]*' | grep '[[:digit:]]' --max-count=1) >./sout.txt" ) ;
   
      //* Open the file *
      ifstream ifs ( soFName, ifstream::in ) ;
      if ( ifs.is_open() )             // if input file open
      {
         //* Read the first line in the file *
         ifs.getline ( emuName, enSIZE, NEWLINE ) ;
         if ( (ifs.good()) || (ifs.gcount() > ZERO) )
         {
            short i = ZERO ;
            while ( (i < enSIZE) && (emuName[i] != '\0') )     // end-of-text
               ++i ;
            while ( (i >= 1) && (emuName[i - 1] != '/') )      // app name
               --i ;
   
            //* If Konsole Terminal *
            if ( (emuName[i] == 'k') && (emuName[i + 1] == 'o') )
            {
               strncpy ( emuName, "konsole", enSIZE ) ;
               status = 1 ;
            }
            //* If XTERM Terminal *
            else if ( (emuName[i] == 'x') && (emuName[i + 1] == 't') )
            {
               strncpy ( emuName, "xterm", enSIZE ) ;
               status = 2 ;
            }
            /* add more terminal tests here... */
   
            //* Default to GNOME terminal *
            else
            {
               strncpy ( emuName, "gnome-terminal", enSIZE ) ;
               status = ZERO ;
            }
         }
         else     /* file access error */ 
         {
            strncpy ( emuName, "gnome-terminal", enSIZE ) ;
            status = ZERO ;
         }
         ifs.close() ;                 // close the source file
      }

      //* Now that we know what terminal we need to open, *
      //* determine its size and position.                *
      system ( "xwininfo -id $WINDOWID >./sout.txt" ) ;

      //* Open the file *
      ifs.open( soFName, ifstream::in ) ;
      if ( ifs.is_open() )             // if input file open
      {  //* Templates for scanning contents of temp file *
         const char* absX = " Absolute upper-left X: %d" ;
         const char* absY = " Absolute upper-left Y: %d" ;
         const char* winW = " Width: %d" ;
         const char* winH = " Height: %d" ;
         const char* geo  = " -geometry %dx%d" ;
         const int buffBYTES = 128 ;      // size of input buffer
         char buff[buffBYTES] ;           // input buffer
         int termXOffset = 0,             // position of active terminal window in pixels
             termYOffset = 0,    
             termXPixels = 0,             // dimensions of active terminal window in pixels
             termYPixels = 0,
             termCols    = 0,             // Size of active terminal window in rows and columns
             termRows    = 0, 
             tempint, ti2 ;

         //* Read the file and extract needed information.  *
         do
         {
            ifs.getline ( buff, buffBYTES, NEWLINE ) ;
            if ( (ifs.good()) || (ifs.gcount() > ZERO) )
            {
               if ( (sscanf ( buff, absX, &tempint )) == 1 )
                  termXOffset = tempint ;
               else if ( (sscanf ( buff, absY, &tempint )) == 1 )
                  termYOffset = tempint ;
               else if ( (sscanf ( buff, winW, &tempint )) == 1 )
                  termXPixels = tempint ;
               else if ( (sscanf ( buff, winH, &tempint )) == 1 )
                  termYPixels = tempint ;
               else if ( (sscanf ( buff, geo, &tempint, &ti2 )) == 2 )
               {
                  termCols = tempint ;
                  termRows = ti2 ;
               }
            }
         }
         while ( (ifs.gcount()) > ZERO ) ;
         ifs.close() ;                 // close the source file

         //* Compensate for bug in Konsole which returns 'termCols' and *
         //* 'termRows' as pixel count rather than column and row count.*
         if ( termCols == termXPixels )
         {
            termCols = termXPixels / 12 ;
            termRows = termYPixels / 18 ;
         }

         #if EXAMINE_KONSOLE_BUG != 0
         if ( status == 1 )
         {
            wcout << "\nKonsole '--working-directory' command:\n" ;
            wcout << absX << "\b\b" << termXOffset << endl ;
            wcout << absY << "\b\b" << termYOffset << endl ;
            wcout << winW << "\b\b" << termXPixels << endl ;
            wcout << winH << "\b\b" << termYPixels << endl ;
            wcout << geo << "\b\b\b\b\b" << termCols << " x " << termRows << endl ;
         }
         #endif   // EXAMINE_KONSOLE_BUG

         //* Calculate the parameters for the new window *
         int pixpercol = termXPixels / termCols,      // pixels per term column
             pixperrow = termYPixels / termRows ;     // pixels per term row
         cols = 65 ;
         rows =  8 ;
         xPos = termXOffset + (termXPixels / 2) - ((cols / 2) * pixpercol) ;
         yPos = termYOffset + (termYPixels / 2) - ((rows / 2) * pixperrow) ;
   
         #if EXAMINE_KONSOLE_BUG != 0
         if ( status == 1 )
         {
            wcout << L" pixpercol:" << pixpercol << endl ;
            wcout << L" pixperrow:" << pixperrow << endl ;
            wcout << L" cols:" << cols << endl ;
            wcout << L" rows:" << rows << endl ;
            wcout << L" xPos:" << xPos << endl ;
            wcout << L" yPos:" << yPos << endl ;
         }
         #endif   // EXAMINE_KONSOLE_BUG
      }
   }
   return status ;

   #undef EXAMINE_KONSOLE_BUG
}  //* End DefineChildEmuator() *

//*************************
//*       helpMe          *
//*************************
//******************************************************************************
//* Display command-line help.                                                 *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

static void helpMe ( void )
{
   wcout << "\nSalmon v:" << AppVersion 
         << " Copyright " << AppYear << " The Software Samurai\n"
              "   Released under GNU General Public License, v:3.0\n"
              "===================================================\n"
              "Salmon invocation: salmon -[v|l|p|f|s|t|D] [ARG1[ ARG2[ ARG3[ ARG4]]]]\n"
              "   v  execv()  pass an argv[] array to childApp\n"
              "               (calling application terminates)\n"
              "   l  execl()  pass individual arguments to childApp\n"
              "               (calling application terminates)\n"
              "   p  execlp() pass individual arguments to grep utility\n"
              "               Note that 'grep' utility is on the system PATH.\n"
              "               (calling application terminates)\n"
              "   f  execl()  'fork' a child process and pass individual arguments to childApp\n"
              "               (wait for child process to terminate)\n"
              "   s  execvp() 'fork' a child process to call childApp with stdout and stderr\n"
              "               redirected to sout.txt and serr.txt, respectively.\n"
              "               (calling application continues immediately)\n"
              "   t  execlp() 'fork' a child process to open a new terminal window,\n"
              "               which executes the 'grep' utility.\n"
              "               (calling application continues immediately)\n"
              "   D           Sandbox area for debugging and experimentation.\n" ;

}  //* End helpMe()) *

//*************************
//*                       *
//*************************
//******************************************************************************
//*                                                                            *
//*                                                                            *
//*                                                                            *
//* Input  :                                                                   *
//*                                                                            *
//* Returns:                                                                   *
//******************************************************************************

