//********************************************************************************
//* File       : Dialogx.cpp                                                     *
//* Author     : Mahlon R. Smith                                                 *
//*              Copyright (c) 2019-2025 Mahlon R. Smith, The Software Samurai   *
//*                 GNU GPL copyright notice located below                       *
//* Date       : 19-Mar-2025                                                     *
//* Version    : (see AppVersion string in header file)                          *
//*                                                                              *
//* Description:                                                                 *
//* This application replaces the previous 'Dialogx' application which was       *
//* designed to interface with the X11 clipboard. That application was           *
//* functional in a rudimentary way, but X is on its way out. The most likely    *
//* replacement for X is Wayland. While Wayland is certainly a work-in-progress  *
//* it has become relatively stable as of early 2019.                            *
//*                                                                              *
//* This application is designed as a simple test of access to the               *
//* GNOME/Wayland compositor's two clipboards: "Primary" and "Regular".          *
//*                                                                              *
//* From inside the terminal window, access to the Wayland clipboard is          *
//* through the wl-clipboard utilities, "wl-copy" and "wl-paste".                *
//* These utilities provide a straightforward method for accessing the system    *
//* clipboard from shell scripts, Perl scripts, etc. For console applications,   *
//* however, the opportunities for communication errors make direct access       *
//* rather dangerous.                                                            *
//* The WaylandCB class provides a uniform (and relatively bug-free) way to      *
//* access the system clipboard without subjecting application programmers to    *
//* the tedious communications protocols necessary for interacting with other    *
//* console utilities.                                                           *
//*                                                                              *
//* The WaylandCB class handles all direct access to the clipboards, while       *
//* thr Dialogx application simply acts as a test suite for that class.          *
//*                                                                              *
//*                        ---  ---  ---  ---                                    *
//* Note: A pure console version of the Dialogx application, "Wayclip" is        *
//* available. Both applications communicate with the system clipboard through   *
//* the WaylandCB class. The console version of the application will be          *
//* distributed with the stand-alone WaylandCB package, while the Dialogx        *
//* version of the application is distributed as part of the NcDialogAPI.        *
//*                                                                              *
//* Development Tools: See NcDialog.cpp.                                         *
//********************************************************************************
//* Copyright Notice:                                                            *
//* This program is free software: you can redistribute it and/or modify it      *
//* under the terms of the GNU General Public License as published by the Free   *
//* Software Foundation, either version 3 of the License, or (at your option)    *
//* any later version.                                                           *
//*                                                                              *
//* This program is distributed in the hope that it will be useful, but          *
//* WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY   *
//* or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License     *
//* for more details.                                                            *
//*                                                                              *
//* You should have received a copy of the GNU General Public License along      *
//* with this program.  If not, see <http://www.gnu.org/licenses/>.              *
//*                                                                              *
//*         Full text of the GPL License may be found in the Texinfo             *
//*         documentation for this program under 'Copyright Notice'.             *
//********************************************************************************
//* Version History (most recent first):                                         *
//*                                                                              *
//* v: 0.0.08 19-Mar-2025                                                        *
//*   - Update to reflect gString changes. Update copyright years.               *
//*                                                                              *
//* v: 0.0.07 23-Jun-2024                                                        *
//*   - Update to reference G++13 and C++11. Update copyright years.             *
//*                                                                              *
//* v: 0.0.06 25-Apr-2020                                                        *
//*   -- Minor changes to reflect changes in return value for some               *
//*      WaylandCB-class methods.                                                *
//*                                                                              *
//* v: 0.0.05 27-Dec-2019                                                        *
//*   -- Complete rewrite:                                                       *
//*      -- Test command-line access to the Wayland clipboard using the          *
//*         wl-clipboard pair of utilities: "wl-copy" and "wl-paste".            *
//*         These are simple (possibly too simple) utilities that communicate    *
//*         the terminal and the Wayland clipboards. See the WaylandCB class     *
//*         for research results.                                                *
//*         Note that these utilities may not be installed by default, even      *
//*         on a GNOME/Wayland desktop. If invoked on a system where they are    *
//*         not installed, yum/dnf offers to install them. (/usr/bin)            *
//*      -- Access to the clipboard is built into the GTK+ development suite;    *
//*         however, we are philosophically opposed to third-party libraries     *
//*         for two reasons: 1) it complicates development, 2) it reduces the    *
//*         chance for fun/education. (Code warriors are all about the fun :-)   *
//*   -- Basic application structure is derived from "Dialogx" which was         *
//*      written to exercise access to the X11 clipboard. If there is a call     *
//*      for continuing support of X, the similaries between thes applications   *
//*      will allow for semi-painless re-integration of the X functionality      *
//*      after the WaylandCB class is stable.                                    *
//*   -- Escaping of "special" characters in clipboard data is borrowed from     *
//*      the FileMangler algorithm used to escape characters in filespecs.       *
//*      This escaping is necessary for passing commands to the shell program.   *
//*   -- This release is the first draft of the Wayland clipboard access code,   *
//*      so it is likely that bugs will hatch. Please send a note about          *
//*      your experiences.                                                       *
//*                                                                              *
//* v: 0.0.04 29-Jan-2018                                                        *
//*   - Allow equals sign ('=') to delimit '-a' option and its argument.         *
//*                                                                              *
//* v: 0.00.03 06-Jul-2015                                                       *
//*   - Begin redesign to support the new:                                       *
//*     ’xcb’ (X-protocol C-language Bindings)                                   *
//*     - For the first effort, we strip out all use of X, and concentrate on    *
//*       Local Clipboard cut-and-paste.                                         *
//*     - Text selection within a Textbox redesigned and simplified.             *
//*     - Selection/Copy/Cut/Paste of LTR, RTL and right-justified data to       *
//*       and from the Local Clipboard are now fully supported.                  *
//*   - Enable mouse support to test ScrollWheel translation for selecting       *
//*     text.                                                                    *
//*   - Can now scan multiple command-line arguemnts.                            *
//*                                                                              *
//* v: 0.00.02 01-Jun-2015                                                       *
//*   - Attempt port to 64-bit system, but XLIB, XMU not supported.              *
//*                                                                              *
//* v: 0.00.01 15-Apr-2014                                                       *
//*   - Created from a copy of the NcDialog group's Dialog3 test application.    *
//*   - The dxClip class does most of the work in this application.              *
//********************************************************************************

//****************
//* Header Files *
//****************
//* All necessary information for:                  *
//* NCurses, NcWindow, NcDialog and gString classes.*
#include "GlobalDef.hpp"
#include "Dialogx.hpp"           //* Application-class definition

//***************
//* Definitions *
//***************
//* Text format identifier. Corresponds to order of items *
//* in Dropdown control 'fmtDD'. Used to determine data   *
//* format for interaction with Wayland clipboard.        *
enum txtfmt : short { txtfmtUTF8 = ZERO, txtfmtUTF32 = 1, txtfmtGSTRING = 2 } ;


//***************
//* Prototypes  *
//***************
//* Callback method for monitoring the Textbox edit sequence.*
static short TextboxMonitor 
       ( const short currIndex, const wkeyCode wkey, bool firstTime = false ) ;

//**********
//*  Data  *
//**********

enum Controls : short { singaTB = ZERO, singbTB, multaTB, multbTB, 
                        donePB, clrPB, reptPB, tstPB, iniPB, regRB, priRB, ncdRB, 
                        wenRB, ovrRB, fmtDD, statBB, ctrlCNT } ;

//* Callback method access to application data members. *
//* See note in callback-method header.                 *
static Dialogx*  wcPtr = NULL ;


//*************************
//*         main          *
//*************************
//******************************************************************************
//* Program entry point.                                                       *
//*                                                                            *
//* Command-line Usage:  See "GetCommandLineArgs() method.                     *
//*                                                                            *
//*                                                                            *
//******************************************************************************

int main ( int argc, char* argv[], char* argenv[] )
{
   //* User may have specified one or more command-line arguments.*
   commArgs clArgs( argc, argv, argenv ) ;

   //* Create the application class object and interact with the user.*
   Dialogx* dxptr = new Dialogx( clArgs ) ;

   //* Before returning to the system, delete the Dialogx object.    *
   //* While the class object and all its resources will be released *
   //* by the system on exit, we want to force execution of the      *
   //* destructor for an orderly termination.                        *
   delete dxptr ;

   exit ( ZERO ) ;

}  //* End main() *

//*************************
//*       ~Dialogx        *
//*************************
//******************************************************************************
//* Destructor. Return all resources to the system.                            *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

Dialogx::~Dialogx ( void )
{

   //* Delete the connection to the Wayland clipboard.*
   delete this->wclip ;

}  //* End ~Dialogx() *

//*************************
//*        Dialogx        *
//*************************
//******************************************************************************
//* Constructor.                                                               *
//*                                                                            *
//*                                                                            *
//* Input  : commArgs class object (by reference)                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

Dialogx::Dialogx ( commArgs& clArgs )
{
   //* Initialize our data members *
   this->dPtr        = NULL ;
   this->sic         = NULL ;
   *this->appPath    = NULLCHAR ;
   this->termRows    = ZERO ;
   this->termCols    = ZERO ;
   this->dlgRows     = MIN_ROWS ;
   this->dlgCols     = MIN_COLS ;
   this->suPos = { 1, 0 } ;
   this->wclip       = NULL ;
   this->wcbConnect  = false ;

   //* Dangerous trick to allow the non-member callback method *
   //* access to our class data (but it promises to behave).   *
   //* See note in callback method header.                     *
   wcPtr = this ;

   //* See what the user has to say *
   this->GetCommandLineArgs ( clArgs ) ;

   gString gsOut ;
   gsOut = clArgs.appPath ;                  // application path
   gsOut.copy( this->appPath, MAX_PATH ) ;

   //***  Initialize the NCurses Engine  ***
   if ( !(clArgs.helpFlag || clArgs.verFlag) && 
         ((nc.StartNCursesEngine ()) == OK) )
   {
      //* Get the size of our playground *
      nc.ScreenDimensions ( this->termRows, this->termCols ) ;

      //* Verify that the current locale supports UTF-8 character encoding.    *
      gString envLocale( nc.GetLocale() ) ;  // name of locale from environment
      short localeSetOk = nc.VerifyLocale ();// env locale supports UTF-8

      //* If user has specified an alternate locale for interpretation of      *
      //* character output and input, it must be passed to the NCurses class   *
      //* before any I/O occurs.                                               *
      short altLocaleOk = OK ;
      if ( *clArgs.altLocale != NULLCHAR )
         altLocaleOk = nc.SetLocale ( clArgs.altLocale ) ;

      nc.SetCursorState ( nccvINVISIBLE ) ;  // hide the cursor
      nc.SetKeyProcState ( nckpRAW ) ;       // allow CTRL keys through unprocessed
      gsOut.compose( titleTemplate, AppTitle, AppVersion, crYears ) ;
      nc.WriteString ( ZERO, ZERO, gsOut.gstr(), nc.bwU ) ; // draw app title line to console
      this->DiagMsg ( "NCurses engine initialized.", nc.bw ) ;


      //* Determine whether the hardware supports display of color text. *
      if ( nc.ColorText_Available () )
      {
         this->DiagMsg ( "Multi-color screen output supported.", nc.gr ) ;

         //* Color Engine successfully initialized? *
         if ( nc.ColorText_Initialized () )
         {
            termColorInfo tci ;
            nc.TerminalColorSupport ( tci ) ;
            gsOut.compose( L"Color Engine started. (%hd RGB Registers and "
                            "%hd fgnd/bkgnd Color Pairs)", 
                           &tci.rgb_regs, &tci.fgbg_pairs ) ;
            this->DiagMsg ( gsOut.ustr(), nc.gr ) ;
         }
         else
         {
            this->DiagMsg ( "Unable to start Color Engine.", nc.bw ) ;
         }
      }
      else
      {
         this->DiagMsg ( "Terminal does not support color output.", nc.bw ) ;
         this->DiagMsg ( " Starting in monochrome mode.", nc.bw ) ;
      }

      //* Report the results of locale settings for character-encoding *
      if ( *clArgs.altLocale != NULLCHAR )
      {
         this->DiagMsg ( "Alternate locale specified: ", nc.gr, false ) ;
         gsOut.compose( L"\"%s\"", clArgs.altLocale ) ;
         this->DiagMsg ( gsOut.ustr(), nc.bw, false ) ;
      }
      else
      {
         this->DiagMsg ( "Locale from environment: ", nc.gr, false ) ;
         gsOut.compose( L"\"%S\"", envLocale.gstr() ) ;
         this->DiagMsg ( gsOut.ustr(), nc.bw, false ) ;
      }
      if ( localeSetOk == OK && altLocaleOk == OK )
         this->DiagMsg ( "  UTF-8 encoding support verified.", nc.gr ) ;
      else
         this->DiagMsg ( "  may not support UTF-8 encoding.", nc.reB ) ;

      //* Report terminal-window dimensions, and *
      //* if window is too small, complain.      *
      gsOut.compose( "Terminal window dimensions: (%hd x %hd)",
                     &this->termRows, &this->termCols ) ;
      this->DiagMsg ( gsOut.ustr(), nc.gr ) ;
      if ( (this->termRows < MIN_ROWS) || (this->termCols < MIN_COLS) )
      {
         gsOut.compose( "Application requires at least %hd rows x %hd columns.",
                        &MIN_ROWS, &MIN_COLS ) ;
         this->DiagMsg ( gsOut.ustr(), nc.re ) ;
         this->DiagMsg ( "Please resize terminal window and try again.", nc.re ) ;
         clArgs.diagPause = 2 ;     // force pause for diagnostic messages
         // Programmer's Note: The call to OpenDialog() will fail when the 
         // terminal window is too small.
      }

      //* If specified on the command-line  *
      //* wait for user to read diagnostics.*
      if ( clArgs.diagPause > ZERO )
      {
         if ( clArgs.diagPause == 1 )
         {
            chrono::duration<short>aWhile( 4 ) ;
            this_thread::sleep_for( aWhile ) ;
         }
         else
         {
            this->DiagMsg ( " * Press Any Key to Continue *", nc.bl ) ;
            nckPause();
         }
      }

      //* Set application color attributes *
      this->dColor   = nc.blR ;        // dialog background color
      this->hColor   = nc.brbl ;       // highlight (emphasis) color
      this->eColor   = nc.rebl ;       // error-message color
      this->pbfColor = nc.reR ;        // pushbutton focus color
      this->pbnColor = nc.gyR ;        // pushbutton non-focus color
      this->tbfColor = nc.gr ;         // pushbutton focus color
      this->tbnColor = nc.bw ;         // pushbutton non-focus color
      this->rbfColor = nc.reR ;        // radiobutton focus color
      this->rbnColor = nc.blG ;        // radiobutton non-focus color
      this->bbfColor = nc.grbk ;       // billboard focus color
      this->bbnColor = nc.gybk ;       // billboard non-focus color

      //* Establish the connection between the application *
      //* and the Wayland clipboard. If connection cannot  *
      //* be established, it will be reported when the     *
      //* dialog opens.                                    *
      this->wclip = new WaylandCB() ;
      this->wcbConnect = this->wclip->wcbIsConnected() ;


      //*******************
      //* Open the dialog *
      //*******************
      if ( (this->OpenDialog ( clArgs )) == OK )
      {
         //***************************
         //* Interact with the user. *
         //***************************
         this->UserInterface () ;
      }

      //* Close the dialog window.*
      if ( this->dPtr != NULL )
         delete this->dPtr ;
      wcPtr = NULL ; // dialog is closed, invalidate callback pointer

      //* Release the dynamically-allocated control definitionns.*
      if ( this->sic != NULL )
         delete [] this->sic ;

      nc.OutputStreamScrolling ( false ) ;   // disable auto-scrolling window
      nc.RestoreCursorState () ;             // make cursor visible
      nc.StopNCursesEngine () ;              // Deactivate the NCurses engine
   }
   else
   {
      if ( clArgs.verFlag )
         this->DisplayVersion () ;
      else if ( clArgs.helpFlag )
         this->DisplayHelp () ;
      else
         wcout << "\n ERROR! Unable to initialize NCurses engine.\n" << endl ;
   }

}  //* End Dialogx() *

//*************************
//*     UserInterface     *
//*************************
//******************************************************************************
//* Interact with the user.                                                    *
//*                                                                            *
//* Input  : none                                                              *
//* Returns: nothing                                                           *
//******************************************************************************

void Dialogx::UserInterface ( void )
{
   gString gsOut ;               // output formatting
   uiInfo Info ;                 // user interface data returned here
   short  icIndex = ZERO ;       // index of control with input focus
   bool   done = false ;         // loop control

   //* Establish a callback method to peek at the user's key input *
   this->dPtr->EstablishCallback ( &TextboxMonitor ) ;

   //* If connection to Wayland clipboard failed, alert user.*
   if ( ! this->wcbConnect )
      this->WheresWayland () ;

   while ( ! done )
   {
      //*******************************************
      //* If focus is currently on a Pushbutton   *
      //*******************************************
      if ( this->sic[icIndex].type == dctPUSHBUTTON )
      {
         //* Get user input for this control.                           *
         //* Returns when the Pushbutton is pressed OR when the user    *
         //* requests that the input focus moves to another control.    *
         if ( !Info.viaHotkey )
            icIndex = this->dPtr->EditPushbutton ( Info ) ;
         //* However, If we arrived here via 'Hotkey', then calling the *
         //* edit method is unnecessary for binary controls like        *
         //* Pushbuttons. Just retrieve the edit data.                  *
         else
            Info.HotData2Primary () ;

         //* If the Pushbutton was pressed *
         if ( Info.dataMod != false )
         {
            if ( Info.ctrlIndex == donePB )        // finish and exit
            {
               done = true ;
            }
            else if ( Info.ctrlIndex == clrPB )    // clear clipboard
            {
               this->ClearClipboard () ;
            }
            else if ( Info.ctrlIndex == reptPB )   // report available data formats
            {
               this->ReportFormats () ;
            }
            else if ( Info.ctrlIndex == tstPB )    // test the clipboard connection
            {
               this->TestConnection () ;
            }
            else if ( Info.ctrlIndex == iniPB )    // reinitialize clipboard connection
            {
               this->ReinitWaylandCB () ;
            }
            // (no other pushbuttons defined in this dialog)
         }
         else
         { /* No button press, so nothing to do */ }
      }

      //*******************************************
      //* If focus is currently on a Text Box     *
      //*******************************************
      else if ( this->sic[icIndex].type == dctTEXTBOX )
      {
         //* Allow user to modify the text data in the text box.    *
         //* Returns when edit is complete.                         *
         icIndex = this->dPtr->EditTextbox ( Info ) ;

         //* If a control's data have changed *
         if ( Info.dataMod != false )
         {
            /* Nothing to do at this time. */
         }
      }

      //*******************************************
      //* If focus is currently on a Radio Button *
      //*******************************************
      else if ( this->sic[icIndex].type == dctRADIOBUTTON )
      {
         //* Get user input for this control                            *
         //* Returns when radio-button control or radio-button group is *
         //* about to lose the input focus.                             *
         //* A selection may or may not have been made.                 *
         icIndex = this->dPtr->EditRadiobutton ( Info ) ;

         //* If 'selected' button is a member of a button group *
         if ( Info.selMember < MAX_DIALOG_CONTROLS )
         {
            //* If a new 'selected' member of the group *
            //* OR previous selection was re-selected   *
            if ( Info.dataMod != false || Info.keyIn == nckENTER )
            {
               /* Nothing to do at this time. */
            }
         }
         //* Else, button is an independent button *
         //* (Info.selMember==MAX_DIALOG_CONTROLS) *
         else
         {
            //* If a radio button's state has changed *
            if ( Info.dataMod != false )
            {
               /* Nothing to do at this time. */
            }
         }
      }

      //***********************************************
      //* If focus is currently on a Dropdown control *
      //***********************************************
      else if ( this->sic[icIndex].type == dctDROPDOWN )
      {
         icIndex = this->dPtr->EditDropdown ( Info ) ;
         //* If a radio button's state has changed *
         if ( Info.dataMod != false )
         {
            /* Nothing to do at this time. */
         }
      }

      //* If user exited the control edit method via a hotkey,*
      //* then the new control already has focus. Otherwise,  *
      //* move input focus to next/previous control.          *
      if ( done == false && Info.viaHotkey == false )
      {
         if ( Info.keyIn == nckSTAB )
            icIndex = this->dPtr->PrevControl () ; 
         else
            icIndex = this->dPtr->NextControl () ;
      }
   }
}  //* End UserInterface() *

//*****************************
//*      TextboxMonitor       *
//*****************************
//******************************************************************************
//* NON-MEMBER METHOD.                                                         *
//* Callback method: called each time through the user input loop of an        *
//* NcDialog control object's edit routine.                                    *
//*                                                                            *
//*                                                                            *
//* Input  : currIndex  : index of control with input focus                    *
//*          wkey       : user's most recent key input                         *
//*          firstTime  : (optional, 'false' by default) 'true' on first call  *
//*                                                                            *
//* Returns: OK                                                                *
//******************************************************************************
//* Notes:                                                                     *
//* ------                                                                     *
//* This callback method is defined for access by the user-input loops of the  *
//* editing routines of the NcDialog-class control objects. This gives us      *
//* access to user requests for cut-and-paste operations.                      *
//* Note that because the compiler will not allow us to specify a class-member *
//* method as a callback method, we must play a trick: we establish this       *
//* non-member callback method and give it access to our Dialogx-class         *
//* member methods and data through the module-scope pointer, 'wcPtr'.         *
//* All methods and data of the Dialogx class are public, so this is not a     *
//* huge violation, but in a real application, a non-member method should      *
//* only have INDIRECT access to the member methods and data, i.e. _requests_  *
//* for information and _requests_ to modify data.                             *
//* However, because this is only a test application, we skip the safety       *
//* layering. Don't take risks like this is "real" applications.               *
//*                                                                            *
//* ---  ----  ----  ----  ----  ----  ----  ----  ----  ----  ----  ----  --- *
//*                                                                            *
//* Actions taken by this method.                                              *
//* 1) Update the ovrRB Radiobutton control to track the state of the          *
//*    Insert/Overstrike input mode used in the textbox controls.              *
//*                                                                            *
//* 2) Process user's 'Copy', 'Cut' and 'Paste' commands (only during Textbox  *
//*     edit). We are interested in user key input related to:                 *
//*      - 'copy'   (nckC_C)                                                   *
//*      - 'cut'    (nckC_X)                                                   *
//*      - 'paste'  (nckC_V)                                                   *
//*     These keys are the signal to invoke access to the X clipboard.         *
//*                                                                            *
//*     For our initial tests we worked with the X 'cut buffers'. These are    *
//*     eight (8) simple text buffers maintained in the X Server's data space  *
//*     which are accessed through X-server command functions. This is a       *
//*     primitive interface, but gives us practice in talking to the           *
//*     X Server.                                                              *
//*                                                                            *
//*     When cut-buffer access had been understood, we moved on to the         *
//*     'Selections' interface functions of the X-server clipboard.            *
//*     This is the standard method for accessing the X clipboard. This is a   *
//*     complex interface and does FAR MORE that we have any use for in a      *
//*     text-based application. However, the lessons learned here will help    *
//*     in future projects (and it's fun :-)                                   *
//*                                                                            *
//* 3) Process user's 'selection' commands:                                    *
//*     These are handled entirely within NcDialog, but can be monitored here. *
//*      - 'select current character and character to right'   (nckSRIGHT)     *
//*      - 'select current character and character to left'    (nckSLEFT)      *
//*      - 'select all text in target textbox control'         (nckC_A)        *
//*    a) Note that if no selection has been received, when 'copy' or 'cut'    *
//*       command is received, then 'All Text' is assumed.                     *
//*    b) Any non-clipboard or non-select key received will clear any previous *
//*       selections.                                                          *
//*                                                                            *
//* 4) For convenience during development, a keycode of ALT+C will clear       *
//*    the Billboard control.                                                  *
//*                                                                            *
//*                                                                            *
//******************************************************************************

static short TextboxMonitor ( const short currIndex, 
                                         const wkeyCode wkey, bool firstTime )
{
   static bool oldOverstrikeFlag = false ;   // track insert/overstrike state
   static short bbwidth = ZERO ;             // width of Billboard control

   //*************************
   //* First-time processing *
   //*************************
   if ( firstTime != false )
   {  //* Establish current state of insert/overstrike flag *
      oldOverstrikeFlag = wcPtr->dPtr->IsOverstrike () ;
      //* Max width of Billboard display data *
      bbwidth = wcPtr->sic[statBB].cols - 4 ;
   }
   
   //***********************
   //* Standard processing *
   //***********************

   //* Reflect any changes in the Insert/Overstrike mode.*
   bool newOverstrikeFlag = wcPtr->dPtr->IsOverstrike () ;
   if ( newOverstrikeFlag != oldOverstrikeFlag )
   {
      wcPtr->dPtr->SetRadiobuttonState ( ovrRB, newOverstrikeFlag ) ;
      oldOverstrikeFlag = newOverstrikeFlag ;
   }

   //* If signal to clear the Billboard control *
   if ( (wkey.type == wktEXTEND) && (wkey.key == nckA_C) )
      wcPtr->dPtr->ClearBillboard ( statBB ) ;

   //* Capture and process 'Copy', 'Cut' and 'Paste' commands. *
   //* For 'Select All' command and for character-by-character *
   //* selection commands, report only.                        *
   if ( (wcPtr->sic[currIndex].type == dctTEXTBOX) &&
        (wkey.type == wktFUNKEY) &&
        (wkey.key == caCOPY_SELECTED || wkey.key == caCUT_SELECTED || 
         wkey.key == caPASTE || wkey.key == caSELECT_ALL || 
         wkey.key == caSELECT_RIGHT || wkey.key == caSELECT_LEFT) )
   {  //* Report the recognized key input *
      const char* keyStrings[] = 
      {
         "'Copy Selected'",
         "'Cut Selected'",
         "'Paste'",
         "'Select All'",
         "'Select Right'",
         "'Select Left'",
      } ;

      short msgIndex ;
      switch ( wkey.key )
      {
         case caCOPY_SELECTED:   msgIndex = 0 ;    break ;
         case caCUT_SELECTED:    msgIndex = 1 ;    break ;
         case caPASTE:           msgIndex = 2 ;    break ;
         case caSELECT_ALL:      msgIndex = 3 ;    break ;
         case caSELECT_RIGHT:    msgIndex = 4 ;    break ;
         case caSELECT_LEFT:     msgIndex = 5 ;    break ;
      } ;
      wcPtr->dPtr->Append2Billboard ( statBB, keyStrings[msgIndex], wcPtr->bbfColor ) ;

      gString gs, gscb ;      // text analysis and formatting
      short textFormat = wcPtr->dPtr->GetDropdownSelect ( fmtDD ) ; // format specification

      //***********************
      //** 'Copy' and 'Cut'  **
      //***********************
      if ( wkey.key == caCOPY_SELECTED || wkey.key == caCUT_SELECTED )
      {
         //* Display the results of the copy/cut operation.*
         wcPtr->dPtr->GetLocalClipboard ( gscb ) ;
         gs = gscb ;
         if ( (gs.gscols()) > bbwidth )
         {
            gs.limitCols( bbwidth ) ;
            gs.append( L"..." ) ;
         }
         gs.insert( L' ' ) ;
         while ( (gs.erase( L"\n" )) >= ZERO ) ; 
         wcPtr->dPtr->Append2Billboard ( statBB, gs, wcPtr->bbnColor ) ;

         //* If connected to the Wayland clipboard, AND     *
         //* if user has not limited us to local clipboard, *
         //* copy local clipboard to system clipboard.      *
         short srcCb = wcPtr->dPtr->GetRbGroupSelection ( regRB ) ;
         if ( wcPtr->wcbConnect && (srcCb != ncdRB) )
         {
            short setStatus ;
            if ( textFormat == txtfmtUTF8 )
            {
               char buff[gsALLOCMED] ;
               gscb.copy( buff, gsALLOCMED ) ;
               setStatus = wcPtr->wclip->wcbSet( buff, bool(srcCb == priRB) ) ;
            }
            else if ( textFormat == txtfmtUTF32 )
            {
               wchar_t buff[gsALLOCDFLT] ;
               gscb.copy( buff, gsALLOCDFLT ) ;
               setStatus = wcPtr->wclip->wcbSet( buff, bool(srcCb == priRB) ) ;
            }
            else  // (textFormat == txtfmtGSTRING)
            {
               setStatus = wcPtr->wclip->wcbSet( gscb, bool(srcCb == priRB) ) ;
            }
            if ( setStatus < ZERO )
            {
               gs.compose( " [Error writing to Wayland clipboard.]" ) ;
               wcPtr->dPtr->Append2Billboard ( statBB, gs, wcPtr->bbnColor ) ;
            }
         }
      }     // copy/cut

      //***********************
      //**      'Paste'      **
      //***********************
      else if ( wkey.key == caPASTE )
      {
         const wchar_t* const srcCbDisp[3] = 
         {
            L" (from Wayland \"Regular\" clipboard)",
            L" (from Wayland \"Primary\" clipboard)",
            L" (from local clipboard)"
         } ;
         short srcCb = wcPtr->dPtr->GetRbGroupSelection ( regRB ),   // source clipboard
               textFormat = wcPtr->dPtr->GetDropdownSelect ( fmtDD ),// format specification
               syscb_chars = -2 ;

         //* If connected to the Wayland clipboard,   *
         //* copy system clipboard to local clipboard.*
         if ( wcPtr->wcbConnect && (srcCb != ncdRB) )
         {
            if ( textFormat == txtfmtUTF8 )
            {
               char buff[gsALLOCMED] ;
               syscb_chars = wcPtr->wclip->wcbGet( buff, bool(srcCb==priRB) ) ;
               gscb = buff ;
            }
            else if ( textFormat == txtfmtUTF32 )
            {
               wchar_t buff[gsALLOCDFLT] ;
               syscb_chars = wcPtr->wclip->wcbGet( buff, bool(srcCb==priRB) ) ;
               gscb = buff ;
            }
            else  // (textFormat == txtfmtGSTRING)
            {
               syscb_chars = wcPtr->wclip->wcbGet( gscb, bool(srcCb==priRB) ) ;
            }
            if ( syscb_chars > ZERO )
               wcPtr->dPtr->SetLocalClipboard ( gscb ) ;
         }
         //* Else, system clipboard is inactive.  *
         //* Get size of local clipboard contents.*
         else
            syscb_chars = wcPtr->dPtr->GetLocalClipboard_Bytes () ;

         //* Display the source of the pasted data.*
         // Programmer's Note: The state of the radiobutton is assumed 
         // to indicate the actual source i.e. no programmer error.
         if ( srcCb == regRB )
         {
            gs.compose( "%S [%S]", srcCbDisp[ZERO],
                        syscb_chars > 1 ? L"OK" : 
                        syscb_chars == 1 ? L"clipboard empty" : L"system error" ) ;
         }
         else if ( srcCb == priRB )
         {
            gs.compose( "%S [%S]", srcCbDisp[1],
                        syscb_chars > 1 ? L"OK" : 
                        syscb_chars == 1 ? L"clipboard empty" : L"system error" ) ;
         }
         else
         {
            gs.compose( "%S [%S]", srcCbDisp[2],
                        syscb_chars > ZERO ? L"OK" : L"clipboard empty" ) ;
         }
         wcPtr->dPtr->Append2Billboard ( statBB, gs, wcPtr->bbnColor ) ;

      }     // paste
   }

   return OK ;

}  //* End TextboxMonitor() *

//*************************
//*    ClearClipboard     *
//*************************
//******************************************************************************
//* Clear the active clipboard. That is, set contents of the clipboard to      *
//* an empty string. Write an informational message to the Billboard control.  *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void Dialogx::ClearClipboard ( void )
{
   gString gsOut, gscb ;

   //* Determine which clipboard is active, *
   //* then clear it.                       *
   short srcCb = this->dPtr->GetRbGroupSelection ( regRB ) ;
   if ( srcCb == ncdRB )   // clear the local clipboard
   {
      gsOut.clear() ;
      this->dPtr->SetLocalClipboard ( gsOut ) ;
   }
   else                    // clear specified Wayland clipboard
   {
      #if 1    // PRODUCTION
      this->wclip->wcbClear( bool(srcCb == priRB) ) ;
      #else    // TESTING ONLY
      //* Tests the wl-copy "--clear" argument which is *
      //* non-functional in the 2018-10-03 release.     *
      this->wclip->wcbClear( bool(srcCb == priRB), true ) ;
      #endif   // TESTING ONLY

      this->wclip->wcbGet ( gscb, (srcCb == priRB) ) ; // SHOULD be an empty string
   }

   //* Write a message to the status window.*
   gsOut.compose( "Clear Active Clipboard (\"%S\")", gscb.gstr() ) ;
   wcPtr->dPtr->Append2Billboard ( statBB, gsOut, wcPtr->bbfColor ) ;

}  //* End ClearClipboard() *

//*************************
//*      GetCbBytes       *
//*************************
//******************************************************************************
//* Report the number of data bytes stored on the specified clipboard.         *
//*                                                                            *
//* Input  : srcCb : specifies which clipboard to query: regRB, priRB, ncdRB.  *
//*                                                                            *
//* Returns: number of data bytes                                              *
//******************************************************************************

short Dialogx::GetCbBytes ( short srcCb )
{
   short cbBytes = ZERO ;     // return value

   if ( srcCb == ncdRB )
      cbBytes = this->dPtr->GetLocalClipboard_Bytes () ;
   else
   {
      //* If error condition, returns wcbsNOCONNECT or wcbsNOINSTALL. *
      if ( (cbBytes = this->wclip->wcbBytes ( bool(srcCb == priRB) )) < ZERO )
         cbBytes = ZERO ;
   }

   return cbBytes ;

}  //* End GetCbBytes() *

//*************************
//*     ReportFormats     *
//*************************
//******************************************************************************
//* Report the available formats (MIME types) for retrieving the data on the   *
//* active clipboard. Also report the number of bytes of data currently on     *
//* the active clipboard. Results are written to the Billboard control.        *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void Dialogx::ReportFormats ( void )
{
   gString gsOut, gsTypes ;

   //* Determine which clipboard is active.*
   short srcCb = this->dPtr->GetRbGroupSelection ( regRB ) ;

   //* Get the number of bytes of data on the active clipboard.*
   short cbBytes = this->GetCbBytes ( srcCb ) ;

   //* If local clipboard, we can return either UTF-8 or UTF-32.*
   if ( srcCb == ncdRB )
   {
      gsTypes = "text/plain;charset=UTF-8\ntext/plain;charset=UTF-32" ;
   }
   else
   {
      if ( this->wcbConnect )
      {
         //* Determine the user's desired communications format.*
         short txtFormat = wcPtr->dPtr->GetDropdownSelect ( fmtDD ) ;
         if ( txtFormat == txtfmtUTF8 )
         {
            char buff[gsALLOCMED] ;
            this->wclip->wcbTypes( buff, (-1) ,bool(srcCb == priRB) ) ;
            gsTypes = buff ;
         }
         else if ( txtFormat == txtfmtUTF32 )
         {
            wchar_t buff[gsALLOCDFLT] ;
            this->wclip->wcbTypes( buff, (-1) ,bool(srcCb == priRB) ) ;
            gsTypes = buff ;
         }
         else  // (txtFormat == txtfmtGSTRING)
         {
            this->wclip->wcbTypes( gsTypes, (-1) ,bool(srcCb == priRB) ) ;
         }
      }
      else  // (this should never happen)
         gsTypes = "Program Error! Wayland clipboard not available." ;
   }
   
   this->dPtr->Append2Billboard ( statBB, L"Available Data Formats", wcPtr->bbfColor ) ;

   const short minMIMELINES = 6 ;   // threshold for reporting two types per line
   short nlIndx, nlcnt = ZERO ;

   //* If the returned data may span more display lines than lines, *
   //* in the control, then display two type records per line.      *
   nlIndx = -1 ;
   while ( (nlIndx = gsTypes.find( L'\n', nlIndx + 1 )) >= ZERO )
      ++nlcnt ;

   while ( (gsTypes.gschars()) > 1 )
   {
      gsOut = gsTypes ;
      if ( (nlIndx = gsOut.find( L'\n' )) >= ZERO )
      {
         if ( nlcnt <= minMIMELINES )
            gsOut.limitChars( nlIndx ) ;
         else
         {
            // Programmer's Note: The formatting in this section is rather 
            // kludgy. It works, but it's ugly. Sorry about that :(
            short ni, gstindx = ZERO ;
            if ( ((ni = gsOut.find( L'\n', (nlIndx + 1) )) > nlIndx) || 
                 ((gstindx = gsTypes.find( L'\n', (nlIndx + 1) )) < ZERO) )
            {
               short i = nlIndx ;
               gsOut.replace( L"\n", L" " ) ;
               for ( ; i < 35 ; ++i )
                  gsOut.insert( L' ', i ) ;
               if ( gstindx >= ZERO )
                  gsOut.limitChars( ni + (i - nlIndx) ) ;
               nlIndx = gsTypes.find( L'\n', (nlIndx + 1) ) ;
            }
         }
      }
      gsOut.insert( L' ' ) ;
      this->dPtr->Append2Billboard ( statBB, gsOut, wcPtr->bbnColor ) ;
      if ( nlIndx >= ZERO )
         gsTypes.shiftChars( -(nlIndx + 1) ) ;
      else
         break ;
   }
   gsOut.compose( " Clipboard Bytes: %hd",  &cbBytes ) ;
   this->dPtr->Append2Billboard ( statBB, gsOut, wcPtr->bbnColor ) ;
   gsOut.compose( " WaylandCB v:%s", wcPtr->wclip->wcbVersion() ) ;
   this->dPtr->Append2Billboard ( statBB, gsOut, wcPtr->bbnColor ) ;

}  //* End ReportFormats() *

//*************************
//*    TestConnection     *
//*************************
//******************************************************************************
//* Send a short test message to the Wayland clipboard.                        *
//* The called method in WaylandCB class will compare the data sent to the     *
//* clipboard with the data retrieved from the clipboard.                      *
//*                                                                            *
//* Note that this method ignores the 'wcbConnect' member and tries the        *
//* connection regardless.                                                     *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void Dialogx::TestConnection ( void )
{
   const char* testData = "Is anyone home?" ;
   gString gsOut ;

   if ( (this->wclip->wcbTest( testData )) == wcbsACTIVE )
      gsOut = "Clipboard connection verified." ;
   else
      gsOut = "Communication test failed." ;

   this->dPtr->Append2Billboard ( statBB, gsOut, wcPtr->bbfColor ) ;

}  //* End TestConnection() *

//*************************
//*    ReinitWaylandCB    *
//*************************
//******************************************************************************
//* Perform a full reset of the WaylandCB connection with the Wayland          *
//* clipboard.                                                                 *
//*                                                                            *
//* Note that this method ignores the 'wcbConnect' member and tries the        *
//* connection regardless.                                                     *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void Dialogx::ReinitWaylandCB ( void )
{
   bool status ;
   gString gsOut ;

   if ( (status = this->wclip->wcbReinit()) )
      gsOut = "Reinitialization successful." ;
   else
      gsOut = "Reinitialization failed." ;

   this->dPtr->Append2Billboard ( statBB, gsOut, wcPtr->bbfColor ) ;

   //* If previously not connected, but now connected, *
   //* enable the inactive user-interface controls and *
   //* set the "Regular" clipboard as active.          *
   if ( status && ! this->wcbConnect )
   {
      if ( (this->wcbConnect = this->wclip->wcbIsConnected ()) )
      {
         this->dPtr->ControlActive ( regRB, true ) ;
         this->dPtr->ControlActive ( priRB, true ) ;
         this->dPtr->ControlActive ( ncdRB, true ) ;
         this->dPtr->SetRbGroupSelection ( regRB ) ;
      }
   }

}  //* End ReinitWaylandCB() *

//*************************
//*      OpenDialog       *
//*************************
//******************************************************************************
//* Instantiate the main dialog window.                                        *
//*                                                                            *
//* Input  : ca   : (by reference) command-line arguments                      *
//*                                                                            *
//* Returns: OK if successful, else ERR                                        *
//******************************************************************************

short Dialogx::OpenDialog ( const commArgs& ca )
{
   short status = ERR ;       // return value

   //* Parameters for the Dropdown box.*
   const short fmtITEMS = 3,
               fmtWIDTH = 16 ;
   const char ddboxData[fmtITEMS][fmtWIDTH] = 
   {
      "UTF-8          ",
      "UTF-32(wchar_t)",
      "gString object ",
   } ;
   attr_t monoColor[2] = { attrDFLT, this->rbnColor } ; // Dropdown box color

   this->sic = new InitCtrl[ctrlCNT] ;
   this->sic[singaTB] = 
   {  //* Single-line Textbox A  - - - - - - - - - - - - - - - - -     singaTB *
      dctTEXTBOX,                   // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      3,                            // ulY:       upper left corner in Y
      2,                            // ulX:       upper left corner in X
      1,                            // lines:     control lines
      tbWIDTH,                      // cols:      control columns
      "Single-line textbox A test data.", // dispText:  
      this->tbnColor,               // nColor:    non-focus color
      this->tbfColor,               // fColor:    focus color
      tbPrint,                      // filter:    any printing character
      "Single-line Textbox A",      // label:     
      -1,                           // labY:      
      ZERO,                         // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &this->sic[singbTB]           // nextCtrl:  link in next structure
   } ;
   this->sic[singbTB] = 
   {  //* Single-line Textbox B  - - - - - - - - - - - - - - - - -     singbTB *
      dctTEXTBOX,                   // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      this->sic[singaTB].ulY,       // ulY:       upper left corner in Y
      short(this->sic[singaTB].ulX + this->sic[singaTB].cols + 4), // ulX: upper left corner in X
      1,                            // lines:     control lines
      tbWIDTH,                      // cols:      control columns
      NULL,                         // dispText:  
      this->tbnColor,               // nColor:    non-focus color
      this->tbfColor,               // fColor:    focus color
      tbPrint,                      // filter:    any printing character
      "Single-line Textbox B",      // label:     
      -1,                           // labY:      
      ZERO,                         // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &this->sic[multaTB]           // nextCtrl:  link in next structure
   } ;
   this->sic[multaTB] = 
   {  //* 'Multi-line' Textbox A  - - - - - - - - - - - - - - - - - -  multaTB *
      dctTEXTBOX,                   // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(this->sic[singaTB].ulY + 3), // ulY: upper left corner in Y
      this->sic[singaTB].ulX,       // ulX:       upper left corner in X
      tbHEIGHT,                     // lines:     control lines
      tbWIDTH,                      // cols:      control columns
      "\"She had been hunted by "   // dispText:  initial display text
      "bloodhounds, shocked by cattle prods, chewed by police dogs, peppered "
      "lightly with shotgun pellets. She had loved every minute of it, showing "
      "me quite proudly (and, I might add, suggestively) a fang mark on her "
      "upper thigh.\"      -- A Confederacy of Dunces",
      this->tbnColor,               // nColor:    non-focus color
      this->tbfColor,               // fColor:    focus color
      tbPrint,                      // filter:    any printing character
      "Multi-line Textbox A",       // label:     
      -1,                           // labY:      
      ZERO,                         // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &this->sic[multbTB]           // nextCtrl:  link in next structure
   } ;
   this->sic[multbTB] = 
   {  //* 'Multi-line' Textbox B  - - - - - - - - - - - - - - - - - -  multbTB *
      dctTEXTBOX,                   // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(this->sic[singbTB].ulY + 3), // ulY: upper left corner in Y
      this->sic[singbTB].ulX,       // ulX:       upper left corner in X
      tbHEIGHT,                     // lines:     control lines
      tbWIDTH,                      // cols:      control columns
      NULL,                         // dispText:  initial display text
      this->tbnColor,               // nColor:    non-focus color
      this->tbfColor,               // fColor:    focus color
      tbPrint,                      // filter:    any printing character
      "Multi-line Textbox B",       // label:     
      -1,                           // labY:      
      ZERO,                         // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &this->sic[donePB]            // nextCtrl:  link in next structure
   } ;
   this->sic[donePB] = 
   {  //* 'DONE' pushbutton - - - - - - - - - - - - - - - - - - - - -   donePB *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(this->dlgRows - 2),     // ulY:       upper left corner in Y
      35,                           // ulX:       upper left corner in X
      1,                            // lines:     control lines
      6,                            // cols:      control columns
      " ^DONE ",                    // dispText:  
      this->pbnColor,               // nColor:    non-focus color
      this->pbfColor,               // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      NULL,                         // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &this->sic[clrPB]             // nextCtrl:  link in next structure
   } ;
   this->sic[clrPB] = 
   {  //* 'CLEAR' pushbutton - - - - - - - - - - - - - - - - - - - - -   clrPB *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      this->sic[donePB].ulY,        // ulY:       upper left corner in Y
      short(this->sic[donePB].ulX + this->sic[donePB].cols + 2), // ulX: upper left corner in X
      1,                            // lines:     control lines
      7,                            // cols:      control columns
      " CL^EAR ",                   // dispText:  
      this->pbnColor,               // nColor:    non-focus color
      this->pbfColor,               // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      NULL,                         // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &this->sic[reptPB]            // nextCtrl:  link in next structure
   } ;
   this->sic[reptPB] = 
   {  //* 'REPORT' pushbutton - - - - - - - - - - - - - - - - - - - -   reptPB *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      this->sic[donePB].ulY,        // ulY:       upper left corner in Y
      short(this->sic[clrPB].ulX + this->sic[clrPB].cols + 2), // ulX: upper left corner in X
      1,                            // lines:     control lines
      8,                            // cols:      control columns
      " ^REPORT ",                  // dispText:  
      this->pbnColor,               // nColor:    non-focus color
      this->pbfColor,               // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      NULL,                         // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &this->sic[tstPB]             // nextCtrl:  link in next structure
   } ;
   this->sic[tstPB] = 
   {  //* 'TEST' pushbutton - - - - - - - - - - - - - - - - - - - - -    tstPB *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      this->sic[donePB].ulY,        // ulY:       upper left corner in Y
      short(this->sic[reptPB].ulX + this->sic[reptPB].cols + 2), // ulX: upper left corner in X
      1,                            // lines:     control lines
      8,                            // cols:      control columns
      "  TEST  ",                   // dispText:  
      this->pbnColor,               // nColor:    non-focus color
      this->pbfColor,               // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      NULL,                         // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &this->sic[iniPB]             // nextCtrl:  link in next structure
   } ;
   this->sic[iniPB] = 
   {  //* 'REINIT' pushbutton - - - - - - - - - - - - - - - - - - - - -  iniPB *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      this->sic[donePB].ulY,        // ulY:       upper left corner in Y
      short(this->sic[tstPB].ulX + this->sic[tstPB].cols + 2), // ulX: upper left corner in X
      1,                            // lines:     control lines
      8,                            // cols:      control columns
      " REINIT ",                   // dispText:  
      this->pbnColor,               // nColor:    non-focus color
      this->pbfColor,               // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      NULL,                         // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &this->sic[regRB]             // nextCtrl:  link in next structure
   } ;
   this->sic[regRB] = 
   {  //* 'Regular Clipboard' radiobutton  - - - - - - - - - - - - - - - regRB *
      dctRADIOBUTTON,               // type:      
      rbtS3p,                       // rbSubtype: standard, 3 chars wide
      true,                         // rbSelect:  default selection
      short(this->sic[multbTB].ulY + this->sic[multbTB].lines + 1),// ulY: upper left corner in Y
      short(this->sic[singbTB].ulX + this->sic[singbTB].cols + 5), // ulX: upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a) for standard button types
      this->rbnColor,               // nColor:    non-focus color
      this->rbfColor,               // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "Wayland \"Re^gular\" Clipboard", // label:
      ZERO,                         // labY:      
      5,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    view only
      &this->sic[priRB]             // nextCtrl:  link in next structure
   } ;
   this->sic[priRB] = 
   {  //* 'Primary Clipboard' radiobutton  - - - - - - - - - - - - - - - priRB *
      dctRADIOBUTTON,               // type:      
      rbtS3p,                       // rbSubtype: standard, 3 chars wide
      false,                        // rbSelect:  default selection
      short(this->sic[regRB].ulY + 1), // ulY: upper left corner in Y
      this->sic[regRB].ulX,         // ulX: upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a) for standard button types
      this->rbnColor,               // nColor:    non-focus color
      this->rbfColor,               // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "Wayland \"^Primary\" Clipboard", // label:
      ZERO,                         // labY:      
      5,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    view only
      &this->sic[ncdRB]             // nextCtrl:  link in next structure
   } ;
   this->sic[ncdRB] = 
   {  //* 'NcDialog Local Clipboard' radiobutton  - - - - - - - - - - -  ncdRB *
      dctRADIOBUTTON,               // type:      
      rbtS3p,                       // rbSubtype: standard, 3 chars wide
      false,                        // rbSelect:  default selection
      short(this->sic[priRB].ulY + 1), // ulY: upper left corner in Y
      this->sic[priRB].ulX,         // ulX: upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a) for standard button types
      this->rbnColor,               // nColor:    non-focus color
      this->rbfColor,               // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "Local ^NcDialog Clipboard Only", // label:
      ZERO,                         // labY:      
      5,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    view only
      &this->sic[wenRB]             // nextCtrl:  link in next structure
   } ;
   this->sic[wenRB] = 
   {  //* 'Wayland Enabled' radiobutton  - - - - - - - - - - - - - - - - wenRB *
      dctRADIOBUTTON,               // type:      
      rbtS3p,                       // rbSubtype: standard, 3 chars wide
      this->wcbConnect,             // rbSelect:  default selection
      short(this->sic[ncdRB].ulY + 3), // ulY: upper left corner in Y
      this->sic[ncdRB].ulX,         // ulX: upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a) for standard button types
      this->pbnColor,               // nColor:    non-focus color
      this->pbnColor,               // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "Wayland Clipboard Active",   // label:
      ZERO,                         // labY:      
      5,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      false,                        // active:    view only
      &this->sic[ovrRB]             // nextCtrl:  link in next structure
   } ;
   this->sic[ovrRB] = 
   {  //* 'Insert/Overstrike' indicator radiobutton (inactive) - - - - - ovrRB *
      dctRADIOBUTTON,               // type:      
      rbtS3p,                       // rbSubtype: standard, 3 chars wide
      false,                        // rbSelect:  default selection
      short(this->sic[wenRB].ulY + 1), // ulY: upper left corner in Y
      this->sic[wenRB].ulX,         // ulX: upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a) for standard button types
      this->pbnColor,               // nColor:    non-focus color
      this->pbnColor,               // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "Overstrike Mode",            // label:
      ZERO,                         // labY:      
      5,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      false,                        // active:    view only
      &this->sic[fmtDD]             // nextCtrl:  link in next structure
   } ;
   this->sic[fmtDD] = 
   { //* 'Data Format' Dropdown  - - - - - - - - - - - - - - - - - - - - fmtDD *
      dctDROPDOWN,                  // type:      define a scrolling-data control
      rbtTYPES,                     // rbSubtype: (na)
      false,                        // rbSelect:  (n/a)
      short(this->sic[wenRB].ulY -1),// ulY:      upper left corner in Y
      short(this->dlgCols - 19),    // ulX:       upper left corner in X
      fmtITEMS + 2,                 // lines:     control lines
      fmtWIDTH + 1,                 // cols:      control columns
      (char*)&ddboxData,            // dispText:  n/a
      this->dColor,                 // nColor:    non-focus border color
      this->pbfColor,               // fColor:    focus border color
      tbPrint,                      // filter:    (n/a)
      "Data Format",                // label:     
      -1,                           // labY:      offset from control's ulY
      6,                            // labX       offset from control's ulX
      ddBoxDOWN,                    // exType:    (n/a)
      fmtITEMS,                     // scrItems:  number of elements in text/color arrays
      ZERO,                         // scrSel:    index of initial highlighted element
      monoColor,                    // scrColor:  single-color data display
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &this->sic[statBB],           // nextCtrl:  link in next structure
   } ;
   this->sic[statBB] =
   {  //* 'Status' Billboard - - - - - - - - - - - - - - - - - - - - -  statBB *
      dctBILLBOARD,                 // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      this->sic[regRB].ulY,         // ulY: upper left corner in Y
      this->sic[multaTB].ulX,       // ulX:       upper left corner in X
      short(this->dlgRows / 2 - 3), // lines:     control lines
      short(this->sic[multaTB].cols * 2 + 7), // cols: control columns
      NULL,                         // dispText:  initial display text
      this->bbnColor,               // nColor:    non-focus color
      this->bbfColor,               // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "Clipboard Status",           // label:     control label
      13,                           // labY:      label offset Y
      ZERO,                         // labX       label offset X
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  if specified, attribute array
      NULL,                         // spinData:  (n/a)
      false,                        // active:    view-only
      NULL                          // nextCtrl:  link in next structure
   } ;

   if ( ! this->wcbConnect )
   {
      //* Disable the regRB, priRB and ncdRB radiobuttons.*
      this->sic[regRB].active = this->sic[regRB].rbSelect = false ;
      this->sic[priRB].active = this->sic[priRB].rbSelect = false ;
      this->sic[ncdRB].active = false ;
      this->sic[ncdRB].rbSelect = true ;
   }

   //* Initial parameters for dialog window *
   InitNcDialog dInit( this->dlgRows,  // number of display lines
                       this->dlgCols,  // number of display columns
                       ZERO,           // Y offset from upper-left of terminal 
                       ZERO,           // X offset from upper-left of terminal 
                       NULL,           // dialog title
                       ncltSINGLE,     // border line-style
                       this->dColor,   // border color attribute
                       this->dColor,   // interior color attribute
                       this->sic       // pointer to list of control definitions
                     ) ;

   //* Instantiate the dialog window *
   this->dPtr = new NcDialog ( dInit ) ;

   //* Open the dialog window *
   if ( (this->dPtr->OpenWindow()) == OK )
   {
      status = OK ;
      
      //* Print dialog window's static text *
      this->dPtr->SetDialogTitle ( "  Dialogx -- Exercise the System Clipboards  ",
                                   this->hColor ) ;
      winPos wp( short(this->sic[regRB].ulY - 1), 
                 short(this->sic[regRB].ulX) ) ;
      this->dPtr->WriteString ( wp, "                Active Clipboard                ", 
                                (this->hColor | ncuATTR) ) ;

      wp.ypos = short(this->sic[wenRB].ulY - 1) ;
      this->dPtr->WriteString ( wp, "        Status Flags         ", 
                                (this->hColor | ncuATTR) ) ;

      wp.ypos = short(this->sic[ovrRB].ulY + 2) ;
      this->dPtr->WriteString ( wp, "             Clipboard Access Keys              ", 
                                this->hColor | ncuATTR ) ;
      ++wp.ypos ;
      this->dPtr->WriteParagraph ( wp, 
                                   "CTRL+C\nCTRL+X\nCTRL+V\nCTRL+A\n"
                                   "SHIFT+RightArrow\nSHIFT+LeftArrow", this->hColor ) ;
      wp.xpos += 8;
      wp = this->dPtr->WriteParagraph ( wp,
                                       "Copy selection to clipboard.\n"
                                       "Cut selection to clipboard.\n"
                                       "Paste clipboard data to Textbox control.\n"
                                       "Select all data in Textbox control.\n", 
                                       this->dColor ) ;
      wp.xpos += 9 ;
      wp = this->dPtr->WriteParagraph ( wp,
                                        "Select chars to the right.\n"
                                        "Select chars to the left.\n", this->dColor ) ;

      this->dPtr->WriteString ( this->sic[fmtDD].ulY + this->sic[fmtDD].labY,
                                this->sic[fmtDD].ulX + this->sic[fmtDD].labX,
                                this->sic[fmtDD].label, this->hColor ) ;

      this->dPtr->WriteString ( this->sic[statBB].ulY + this->sic[statBB].lines,
                                this->sic[statBB].ulX + 18,
                                "(ALT+C clears the status window)", nc.gybl ) ;

      //* Group radiobuttons into an XOR group.*
      short xorGroup[4] { regRB, priRB, ncdRB, -1 } ;
      this->dPtr->GroupRadiobuttons ( xorGroup ) ;

      //* Set the keycodes to be used for user access   *
      //* to the clipboard for select, copy, cut, paste.*
      wkeyCode copy( caCOPY_SELECTED, wktFUNKEY ),
                cut( caCUT_SELECTED, wktFUNKEY ),
              paste( caPASTE, wktFUNKEY ),
             selAll( caSELECT_ALL, wktFUNKEY ),
            selLeft( caSELECT_LEFT, wktFUNKEY ),
           selRight( caSELECT_RIGHT, wktFUNKEY ) ;
      reservedKeys rk( copy, cut, paste, selAll, selLeft, selRight, true ) ;
      this->dPtr->SetTextboxReservedKeys ( rk ) ;

      //* Enable boinks for invalid Textbox input   *
      //* Because we allow all PRINTING characters, *
      //* we will only be boinked for function keys.*
      this->dPtr->TextboxAlert ( singaTB, true ) ;
      this->dPtr->TextboxAlert ( singbTB, true ) ;
      this->dPtr->TextboxAlert ( multaTB, true ) ;
      this->dPtr->TextboxAlert ( multbTB, true ) ;

      //* Enable scrolling of color attributes in the Billboard control.*
      this->dPtr->ScrollBillboardColors ( statBB, true ) ;

      //* If specified, enable mouse support.*
      if ( ca.mouseFlag )
         this->dPtr->meEnableStableMouse () ;

      //* If specified, configure singbTB and multbTB for RTL data.*
      if ( ca.bRtl )
      {
         this->dPtr->DrawContentsAsRTL ( singbTB ) ;
         this->dPtr->DrawContentsAsRTL ( multbTB ) ;
         this->dPtr->WriteParagraph ( this->sic[singbTB].ulY + 1,
                                      this->sic[singbTB].ulX + 31,
                                      "(RTL)\n(RTL)", nc.gybl ) ;
      }

      //* If specified, configure singbTB for right-justified data.*
      else if ( ca.bRj )
      {
         this->dPtr->SetTextboxCursor ( singbTB, tbcpRIGHTJUST ) ;
         this->dPtr->WriteString ( this->sic[singbTB].ulY + 1,
                                   this->sic[singbTB].ulX + 24,
                                   "(right-just)", nc.gybl ) ;
      }

      this->dPtr->RefreshWin () ;
   }
   return status ;

}  //* End OpenDialog() *

//*************************
//*     WheresWayland     *
//*************************
//******************************************************************************
//* If we were unable to establish a connection with the Wayland clipboard,    *
//* disable the clipboard-option radiobuttons or alert the user.               *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void Dialogx::WheresWayland ( void )
{
   const char* msg[] = 
   {
      "  Warning!  Warning!  ",
      " ",
      "  Unable to establish a connection with the ",
      "              Wayland clipboard.",
      "The most likely cause of failure would be ",
      "that the \"wl-copy\" and \"wl-paste\" utilities ",
      "are not installed or are out-of-date.  Try:",
      "       sudo dnf install 'wl-clipboard'",
      " ",
      "Continuing with NcDialog local clipboard only.",
      " ",
      NULL
   } ;
   const attr_t attr[] = 
   { nc.brre, nc.maR, nc.maR, nc.brma, nc.maR, nc.maR, nc.maR, 
     nc.grma, nc.maR, nc.maR, nc.maR, nc.maR } ;
   genDialog gd( msg, nc.maR, 13, 49, 2, 41, attr ) ;

   this->dPtr->InfoDialog ( gd ) ;

}  //* End WhereWayland() *

//*************************
//*  GetCommandLineArgs   *
//*************************
//******************************************************************************
//* Capture user's command-line arguments.                                     *
//* Valid Arguments: see DisplayHelp() method.                                 *
//*                                                                            *
//* Input  : commArgs class object (by reference)                              *
//*                                                                            *
//* Returns: 'false' if normal startup                                         *
//*          'true'  if request for help or invalid command-line argument      *
//******************************************************************************
//* Programmer's Note: Even though it is not documented, we accept both        *
//* lowercase and uppercase option and sub-option characters where possible.   *
//* This allows us to add new, case-sensitive options at a later date.         *
//******************************************************************************

bool Dialogx::GetCommandLineArgs ( commArgs& ca )
{
   #define DEBUG_GCA (0)

   //* Get the application executable's directory path.        *
   //* Save the actual, absolute path specification, replacing *
   //* symbolic links and relative path specifications.        *
   if ( (realpath ( ca.argList[0], ca.appPath )) != NULL )
   {
      gString gs( ca.appPath ) ;
      gs.limitChars( (gs.findlast( L'/' )) ) ;
      gs.copy( ca.appPath, MAX_PATH ) ;
   }

   //* If user provided command-line arguments *
   if ( ca.argCount > 1 )
   {
      gString gs ;               // text formatting
      const char* argPtr ;       // pointer to option argument
      short j = ZERO ;           // for multiple arguments in same token
      bool multiarg = false ;    // 'true' if concatenated options

      for ( short i = 1 ; (i < ca.argCount) || (multiarg != false) ; i++ )
      {
         if ( multiarg != false )   // if additional switches in same argument
            --i ;
         else
            j = ZERO ;

         //* If a command-line switch OR continuing previous switch argument *
         if ( ca.argList[i][j] == DASH || multiarg != false )
         {
            #if DEBUG_GCA != 0  // debugging only
            wcout << L"TOKEN:" << i << L" '" << &ca.argList[i][j] << L"'" << endl ;
            #endif   // DEBUG_GCA

            multiarg = false ;
            ++j ;

            if ( ca.argList[i][j] == DASH ) // (double dash)
            {  //* Long-form command switches *
               gs = ca.argList[i] ;

               //* Request for application version number *
               if ( (gs.compare( L"--version" )) == ZERO )
                  ca.helpFlag = ca.verFlag = true ;

               //* Long-form request for Help *
               else if ( (gs.compare( L"--help" )) == ZERO )
                  ca.helpFlag = true ;

               else  // invalid argument
               { ca.helpFlag = true ; break ; }

               continue ;     // finished with this argument
            }  // (double dash)

            //** Short-form Arguments **
            char argLetter = ca.argList[i][j] ;

            //* Specify an alternate locale *
            if ( argLetter == 'a' || argLetter == 'A' )
            {  //* If user specified an alternate locale string, use it instead *
               //* of the default locale found in the terminal environment.     *
               if ( ca.argList[i][++j] == '=' )
                  argPtr = &ca.argList[i][++j] ;
               else if ( i < (ca.argCount - 1) )
                  argPtr = ca.argList[++i] ;
               else
               { ca.helpFlag = true ; break ; }    // invalid syntax for argument
               gs = argPtr ;
               gs.copy( ca.altLocale, LN_NAME ) ;
               continue ;     // finished with this argument
            }

            //* Configure Single-B and Multi-B textboxes    *
            //* to display text as RTL.                     *
            //* NOTE: 'r' and 'j' options are incompatible, *
            //*       so 'r' takes precedence.              *
            if ( argLetter == 'r' || argLetter == 'R' )
            {
               ca.bRtl = true ;
               ca.bRj  = false ;
            }

            //* Configure Single-B textbox for right-justified input.*
            //* NOTE: 'r' and 'j' options are incompatible, so       *
            //*       'r' takes precedence.                          *
            if ( argLetter == 'j' || argLetter == 'J' )
            {
               if ( ! ca.bRtl )
                  ca.bRj = true ;
            }

            //* Enable/disable Mouse support. *
            else if ( argLetter == 'm' || argLetter == 'M' )
            {
               if ( ca.argList[i][++j] == '=' )
                  argPtr = &ca.argList[i][++j] ;
               else if ( i < (ca.argCount - 1) )
                  argPtr = ca.argList[++i] ;
               else
               { ca.helpFlag = true ; break ; }    // invalid syntax for argument
               if ( (*argPtr == 'e') || (*argPtr == 'E') )
                  ca.mouseFlag = true ;
               else if ( (*argPtr == 'd') || (*argPtr == 'D') )
                  ca.mouseFlag = false ;
               else
               { ca.helpFlag = true ; break ; }    // invalid syntax for argument
               continue ;
            }

            //* Short-form request for help *
            else if ( argLetter == 'h' || argLetter == 'H' )
            { ca.helpFlag = true ; break ; }

            //* Pause after start-up sequence so user can see diagnostics *
            else if ( argLetter == 'p' || argLetter == 'P' )
            {
               char subchar = tolower ( ca.argList[i][j + 1] ) ;
               if ( subchar == 'v' )
               {
                  ca.diagPause = 2 ;
                  ++j ;
               }
               else
                  ca.diagPause = 1 ;
            }

            //* If separate tokens have been concatenated *
            if ( ca.argList[i][j + 1] != nckNULLCHAR )
               multiarg = true ;
         }

         else  // invalid argument, token does not begin with a DASH character
         { ca.helpFlag = true ; break ; }

      }     // for(;;)

      #if DEBUG_GCA != 0  // debugging only
      wcout << L"commArgs Settings" << endl ;
      gs.compose( L"%hd", &ca.argCount ) ;
      wcout << L"argCount   : " << gs.gstr() << endl ;
      for ( short i = ZERO ; i < ca.argCount ; i++ )
      {
         gs.compose( L"argList[%hd] : '%s'", &i, ca.argList[i] ) ;
         wcout << gs.gstr() << endl ;
      }
      gs.compose( L"appPath    : '%s'", ca.appPath ) ;
      wcout << gs.gstr() << endl ;
      gs.compose( L"diagPause  : %hd", &ca.diagPause ) ;
      wcout << gs.gstr() << endl ;
      gs.compose( L"altLocale  : '%s'", ca.altLocale ) ;
      wcout << gs.gstr() << endl ;
      gs.compose( L"bRtl       : %hhd", &ca.bRtl ) ;
      wcout << gs.gstr() << endl ;
      gs.compose( L"bRj        : %hhd", &ca.bRj ) ;
      wcout << gs.gstr() << endl ;
      gs.compose( L"mouseFlag  : %hhd", &ca.mouseFlag ) ;
      wcout << gs.gstr() << endl ;
      gs.compose( L"verFlag    : %hhd", &ca.verFlag ) ;
      wcout << gs.gstr() << endl ;
      gs.compose( L"helpFlag   : %hhd", &ca.helpFlag ) ;
      wcout << gs.gstr() << endl ;
      wcout << L"\n Press Enter" << endl ;
      getchar () ;
      #endif   // DEBUG_GCA
   }
   return ca.helpFlag ;

   #undef DEBUG_GCA
}  //* End GetCommandLineArgs() *

//*************************
//*       DiagMsg         *
//*************************
//******************************************************************************
//* Display start-up diagnostic message in console window.                     *
//* DO NOT call this method after dialog window has opened.                    *
//*                                                                            *
//* Input  : msg    : message to be displayed                                  *
//*          color  : color attribute for message                              *
//*          newLine: (optional, 'true' by default)                            *
//*                   if 'true'  move cursor to next message position          *
//*                   if 'false' leave cursor at end of message                *
//* Returns: nothing                                                           *
//******************************************************************************

void Dialogx::DiagMsg ( const char* msg, attr_t color, bool newLine )
{
   static short xCol = ZERO, xColBaseY = 11 ;

   this->suPos = nc.WriteString ( this->suPos.ypos, this->suPos.xpos, msg, color ) ;
   if ( newLine )
   {
      this->suPos = { short(this->suPos.ypos + 1), xCol  } ;
      if ( xCol == ZERO && (this->suPos.ypos >= this->termRows) )
      {
         xCol = this->termCols / 2 + 1 ;
         this->suPos = { short(xColBaseY + 1), xCol } ;
      }
   }

}  //* End DiagMsg() *

//*************************
//*    DisplayVersion     *
//*************************
//******************************************************************************
//* Print application version number and copyright notice to tty (bypassing    *
//* NCurses output).                                                           *
//* The NCurses engine has been shut down (or didn't start) so we use simple   *
//* console I/O to display the version/copyright text.                         *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void Dialogx::DisplayVersion ( void )
{
   const short fsWidth = 68 ;
   static const char* const freeSoftware = 
    "License GPLv3+: GNU GPL version 3 <https://www.gnu.org/licenses/>.\n"
    "This is free software: you are free to modify and/or redistribute it\n"
    "under the terms set out in the license. There is NO warranty;\n"
    "not even for merchantability or fitness for a particular purpose.\n" ;

   gString gsOut( titleTemplate, AppTitle, AppVersion, crYears ) ;
   gsOut.append( wNEWLINE ) ;
   gsOut.padCols( ((gsOut.gscols()) + fsWidth), L'=' ) ;
   wcout << wNEWLINE << gsOut.gstr() << wNEWLINE << freeSoftware << endl ;

}  //* End DisplayVersion() *

//*************************
//*    DisplayHelp        *
//*************************
//******************************************************************************
//* Print command-line help to tty (bypassing NCurses output).                 *
//* The NCurses engine has been shut down (or didn't start) so we use simple   *
//* console I/O to display the help text.                                      *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void Dialogx::DisplayHelp ( void )
{
   static const char* Help = 
    //12345678901234567890123456789012345678901234567890123456789012345678901234567890
   "\nUSAGE: Dialogx [OPTIONS]   Example: Dialogx --version"
   "\n       Options may be specified in any order. If an option is not specified,"
   "\n       then its default value will be used."
   "\n "
   "\n -a   Specify an alternate 'locale'."
   "\n           Example: -a wo_SN.utf8  or  -ayi_US.utf8"
   "\n -r   Configure Single-B and Multi-B textboxes for RTL data."
   "\n -j   Configure Single-line B textbox for right-justified data."
   "\n -m   Enable/disable mouse support within the application dialog."
   "\n      Mouse support is enabled by default."
   "\n -p[v] Pause to display startup diagnostics before entering main program."
   "\n --version        Display application version number, then exit."
   "\n --help, -h, -?   Help for command line options"
   "\n" ;

   gString gsOut( titleTemplate, AppTitle, AppVersion, crYears ) ;
   gsOut.append( wNEWLINE ) ;
   gsOut.padCols( ((gsOut.gscols()) * 2), L'=' ) ;
   wcout << wNEWLINE << gsOut.gstr() << Help << endl ;

}  //* End DisplayHelp() *

//*************************
//*                       *
//*************************
//********************************************************************************
//*                                                                              *
//*                                                                              *
//*                                                                              *
//* Input  :                                                                     *
//*                                                                              *
//* Returns:                                                                     *
//********************************************************************************

