//******************************************************************************
//* File       : infoDocTest.cpp                                               *
//* Author     : Mahlon R. Smith                                               *
//*              Copyright (c) 2014-2025 Mahlon R. Smith, The Software Samurai *
//*                  GNU GPL copyright notice located in NcDialog.hpp          *
//* Date       : 22-Mar-2025                                                   *
//* Version    : (see below)                                                   *
//*                                                                            *
//* Description: Class definition for the infoDocTest class.                   *
//*              This class includes examples used in the NcDialog API         *
//*              Texinfo documentation.                                        *
//*              It is instantiated by the Dialogw application, Test07.        *
//*                                                                            *
//* Development Tools: See NcDialog.cpp.                                       *
//******************************************************************************
//* Version History (most recent first):                                       *
//*                                                                            *
//* v: 0.00.04 27-Dec-2020                                                     *
//*   - Enable mouse support for tabbed-window example.                        *
//*                                                                            *
//* v: 0.00.03 14-Jan-2016                                                     *
//*   - Add Spinner example for the section describing the 'dspinData' class.  *
//*                                                                            *
//* v: 0.00.02 12-Apr-2015                                                     *
//*   - Add Line-drawing example for 'NcDialog Display Output' chapter.        *
//*   - Add set-mouse-click example for 'Mouse Configuration' chapter.         *
//*   - Add Hotkey selection example for 'Select Control via Hotkey' chapter.  *
//*   - Add genDialog test for LTR/RTL for 'Secondary Classes' chapter.        *
//*   - Add Tabbed Dialogs demo for 'Tabbed Dialogs' chapter.                  *
//*   - Add Billboard control demo for 'Billboard Controls' chapter.           *
//*   - Add Dropdown control demo for 'Dropdown Controls' chapter.             *
//*   - Add Menuwin control demo for 'Menuwin Controls' chapter.               *
//*                                                                            *
//* v: 0.00.01 20-Jun-2014                                                     *
//*   - Attach class instantiation to Dialogw, Test07.                         *
//*   - Add RTL language output examples.                                      *
//*   - Add examples from 'gString' chapter                                    *
//*                                                                            *
//******************************************************************************

#include "infoDocTest.hpp"

//**********************
//** Local Prototypes **
//**********************
static void gsBasicStuff ( NcDialog* dp, attr_t dColor ) ;
static void gsInsertAppend ( NcDialog* dp, attr_t dColor ) ;
static void gsShiftCompose ( NcDialog* dp, attr_t dColor ) ;
static void gsSubstrings ( NcDialog* dp, attr_t dColor ) ;
static void MixedRTL_LTR ( NcDialog* dp, attr_t dColor ) ;
static void Top2BottomRTL ( NcDialog* dp, attr_t dColor ) ;
static void QuickDialog ( NcDialog* dp, attr_t dColor ) ;
static void LineDialog01 ( void ) ;
static void Set_ClickInterval ( void ) ;
static void Hotkey_Selection ( void ) ;
static void Tabby_Cat ( void ) ;
static void tcCatBox ( NcDialog* dp, const char* title, const char* tName, 
                       attr_t fColor, attr_t sColor, attr_t bColor, 
                       short dRows, short dCols, short tabI ) ;
static short tcSpanish ( const InitNcDialog* idptr, const char** tabNames, 
                         const attr_t* tabfColor, const attr_t* tabnColor,
                         const attr_t tabsColor, short ctrlCount ) ;
static short tcVietnamese ( const InitNcDialog* idptr, const char** tabNames, 
                         const attr_t* tabfColor, const attr_t* tabnColor,
                         const attr_t tabsColor, short ctrlCount ) ;
static short tcChinese ( const InitNcDialog* idptr, const char** tabNames, 
                         const attr_t* tabfColor, const attr_t* tabnColor,
                         const attr_t tabsColor, short ctrlCount ) ;

static void Billboard_Demo ( void ) ;
static void Dropdown_Demo ( void ) ;
static void Menuwin_Demo ( void ) ;
static void TimeSpinner ( void ) ;
//* Module-scope access to Set_clickInterval dialog *
static NcDialog* sciPtr = NULL ;
static short ciCallback ( const short currIndex, 
                          const wkeyCode wkey, bool firstTime ) ;


//****************
//** Local Data **
//****************

//* Keep ddData and enum testNumber in synch *
static const short ddITEMS = 15, ddWIDTH = 34 ;
static const char ddData[ddITEMS][ddWIDTH + 2] = 
{
   " gString class 1 of 4             ",
   " gString class 2 of 4             ",
   " gString class 3 of 4             ",
   " gString class 4 of 4             ",
   " Mixed LTR and RTL output         ",
   " RTL + top-to-bottom output       ",
   " Line Drawing Example 1           ",
   " Set Mouse Click Interval         ",
   " Hotkey Demonstration             ",
   " genDialog Demonstration          ",
   " Tabbed Data Presentation         ",
   " Billboard Demonstration          ",
   " Dropdown Demonstration           ",
   " Menuwin Demonstration            ",
   " Time Spinner Demonstration       ",
} ;
enum testNumber : short
{
   tn_gString1 = ZERO, tn_gString2, tn_gString3, tn_gString4, 
   tn_MixLR, tn_RtlTtb, tn_Line1, tn_Clicker, tn_Hottie, tn_genDlg,
   tn_Tabby, tn_Billy, tn_Dropy, tn_Manny, tn_Spiny, 
   
   tn_TESTS_DEFINED
} ;
//* Color scheme for dctDROPDOWN control *
static attr_t monoColor[2] = { attrDFLT, nc.bw } ;

//* Control definitions for prompt dialog *
static InitCtrl promptIc[idpCONTROLS_DEFINED] = 
{
   { //* 'SELECT' Dropdown - - - - - - - - - - - - - - - - - - - - - -   idpDD *
      dctDROPDOWN,                  // type:      define a scrolling-data control
      rbtTYPES,                     // rbSubtype: (na)
      false,                        // rbSelect:  (n/a)
      3,                            // ulY:       upper left corner in Y
      2,                            // ulX: upper left corner in X
      short(tselROWS - 5),          // lines:     control lines
      short(ddWIDTH + 2),           // cols:      control columns
      (char*)&ddData,               // dispText:  items displayed
      nc.bw,                        // nColor:    non-focus border color
      nc.bw,                        // fColor:    focus border color
      tbPrint,                      // filter:    (n/a)
      NULL,                         // label:     
      ZERO,                         // labY:      offset from control's ulY
      ZERO,                         // labX       offset from control's ulX
      ddBoxDOWN,                    // exType:    (n/a)
      ddITEMS,                      // scrItems:  number of elements in text/color arrays
      ZERO,                         // scrSel:    index of initial highlighted element
      monoColor,                    // scrColor:  single-color data display
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &promptIc[idpPB],             // nextCtrl:  link in next structure
   },
   {  //* 'DONE' Pushbutton  - - - - - - - - - - - - - - - - - - - - -   idpPB *
   dctPUSHBUTTON,                // type:      
   rbtTYPES,                     // rbSubtype: (n/a)
   false,                        // rbSelect:  (n/a)
   short(tselROWS - 2),          // ulY:       upper left corner in Y
   short(tselCOLS / 2 - 4),      // ulX:       upper left corner in X
   1,                            // lines:     control lines
   8,                            // cols:      control columns
   "  DONE  ",                   // dispText:  
   nc.gyR,                       // nColor:    non-focus color
   nc.reG,                       // fColor:    focus color
   tbPrint,                      // filter:    (n/a)
   NULL,                         // label:     (n/a)
   ZERO,                         // labY:      (n/a)
   ZERO,                         // labX       (n/a)
   ddBoxTYPES,                   // exType:    (n/a)
   1,                            // scrItems:  (n/a)
   1,                            // scrSel:    (n/a)
   NULL,                         // scrColor:  (n/a)
   NULL,                         // spinData:  (n/a)
   true,                         // active:    allow control to gain focus
   NULL                          // nextCtrl:  link in next structure
   },
} ;

InitCtrl twic[twDEFINED] =       // array of dialog control initialization objects
{
   {  //* 'DONE' pushbutton - - - - - - - - - - - - - - - - - - - -   twDonePB *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(twinROWS - 2),          // ulY:       upper left corner in Y
      short(twinCOLS / 2 - 4),      // ulX:       upper left corner in X
      1,                            // lines:     control lines
      8,                            // cols:      control columns
      "  DO^NE  ",                  // dispText:  
      nc.gyR,                       // nColor:    non-focus color
      nc.reG,                       // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      NULL,                         // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      NULL,                         // nextCtrl:  link in next structure
   },
} ;

   //* Tabby_Cat() : names for basic set of controls *
   enum tcControls : short
   {
      tcDonePB = ZERO, tcTabE, tcTabS, tcTabV, tcTabC, 
      tcCONTROLS_DEFINED
   } ;



//*************************
//*     ~infoDocTest      *
//*************************
//******************************************************************************
//* Destructor.                                                                *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

infoDocTest::~infoDocTest ( void )
{

   if ( this->dp != NULL )                // close the window
      delete ( this->dp ) ;

}  //* End ~infoDocTest() 

//*************************
//*      infoDocTest      *
//*************************
//******************************************************************************
//* Constructor.                                                               *
//*                                                                            *
//* Input  : tLines     : number of display line in terminal window            *
//*          tCols      : number of display columns in terminal window         *
//*          minY       : first available display line                         *
//*                                                                            *
//* Returns: implicitly return class reference                                 *
//******************************************************************************

infoDocTest::infoDocTest ( short tLines, short tCols, short minY )
{
   //* Initialize data members *
   this->termLines = tLines ;
   this->termCols  = tCols ;
   this->minULY    = minY ;
   this->dRows     = 1 ;                        // dialog dimensions
   this->dCols     = 1 ;
   this->icPtr     = NULL ;                     // control-definition pointer
   this->dp        = NULL ;                     // object pointer
   this->dColor    = nc.blR ;                   // dialog base color
   this->tColor    = this->dColor | ncbATTR ;   // dialog title color
   this->bColor    = nc.gybl | ncbATTR ;        // dialog border color
   this->hColor    = nc.brbl | ncbATTR ;        // dialog highlight color
   this->idOpen    = false ;                    // not yet opened

   //* Initialize remainder of control-definition array.           *
   //* (colors become available after NCurses engine instantiated) *
   promptIc[idpDD].nColor = nc.gyR ;
   promptIc[idpDD].fColor = nc.rebl ;

   monoColor[1] = nc.bkG ;

   twic[twDonePB].nColor = promptIc[idpPB].nColor = nc.gyR ;
   twic[twDonePB].fColor = promptIc[idpPB].fColor = nc.reG ;

   //* Open the prompt dialog. If dialog opens caller may continue.*
   if ( (this->idOpen = this->idOpenPrompt ()) != false )
   {

   }  // OpenWindow()

}  //* End infoDocTest() 

//*************************
//*     idOpenPrompt      *
//*************************
//******************************************************************************
//* Open the prompt dialog window.                                             *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: 'true' if dialog window opened successfully, else 'false'         *
//******************************************************************************

bool infoDocTest::idOpenPrompt ( void )
{
   this->icPtr = promptIc ;
   this->dRows = tselROWS ;
   this->dCols = tselCOLS ;

   short ulY     = this->minULY + 1,         // upper left corner in Y
         ulX     = 40 - (this->dCols / 2) ;  // upper left corner in X
   bool  success = false ;

   //* Initial parameters for dialog window *
   InitNcDialog dInit( this->dRows,    // number of display lines
                       this->dCols,    // number of display columns
                       ulY,            // Y offset from upper-left of terminal 
                       ulX,            // X offset from upper-left of terminal 
                       NULL,           // dialog title
                       ncltSINGLE,     // border line-style
                       this->bColor,   // border color attribute
                       this->dColor,   // interior color attribute
                       this->icPtr     // pointer to list of control definitions
                     ) ;

   //* Instantiate the dialog window *
   this->dp = new NcDialog ( dInit ) ;

   //* Open the dialog window *
   if ( (this->dp->OpenWindow()) == OK )
   {
      this->dp->SetDialogTitle ( "  Select a Test Item  ", this->tColor ) ;
      this->dp->WriteParagraph ( 1, 3, 
                       "Use up/down arrow keys to highlight\n"
                       "desired test, then press ENTER key.", this->hColor ) ;

      #if 0    // ONLY FOR HIGHLIGHTING A TEST UNDER DEVELOPMENT
      this->dp->PrevControl () ;
      this->dp->SetDropdownSelect ( idpDD, 10 ) ;
      this->dp->NextControl () ;
      #endif   // DEVELOPMENT ONLY

      this->dp->RefreshWin () ;
      success = true ;
   }
   return success ;

}  //* End idOpenPrompt() *

//*************************
//*    idDialogOpened     *
//*************************
//******************************************************************************
//* Satisfy caller's curiosity.                                                *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: 'true' if dialog opened successfully, else 'false'                *
//******************************************************************************

bool infoDocTest::idDialogOpened ( void )
{

   return this->idOpen ;

}  //* End idDialogOpened() *

//*************************
//*      idInteract       *
//*************************
//******************************************************************************
//* User interactive experience. Calls the appropriate user-interface method.  *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void infoDocTest::idInteract ( void )
{
   if ( this->idOpen )
   {
      uiInfo info ;
      short icIndex = ZERO ;
      bool done = false ;
      while ( ! done )
      {
         if ( this->idOpen != false && this->dp != NULL && this->icPtr == promptIc )
         {
            if ( icIndex == idpPB )
            {
               this->dp->EditPushbutton ( info ) ;
               if ( info.dataMod != false )
                  done = true ;
               else
                  icIndex = this->dp->NextControl () ;
            }

            else if ( icIndex == idpDD )
            {
               info.viaHotkey = true ;
               this->dp->EditDropdown ( info ) ;
               icIndex = this->dp->NextControl () ;

               if ( info.keyIn == nckENTER || info.keyIn == nckSPACE )
               {
                  if ( info.selMember < tn_TESTS_DEFINED )
                  {
                     if ( info.selMember == tn_gString1 ||
                          info.selMember == tn_gString2 ||
                          info.selMember == tn_gString3 ||
                          info.selMember == tn_gString4 ||
                          info.selMember == tn_MixLR    ||
                          info.selMember == tn_RtlTtb   ||
                          info.selMember == tn_genDlg
                        )
                     {
                        this->dp->SetDialogObscured () ; // protect prompt dialog
                        this->idStdDialog ( info.selMember ) ;
                     }
                     else if ( info.selMember == tn_Line1 )
                     {
                        this->dp->HideWin () ;  // hide the prompt dialog
                        LineDialog01 () ;
                     }
                     else if ( info.selMember == tn_Clicker )
                     {
                        this->dp->HideWin () ;  // hide the prompt dialog
                        Set_ClickInterval () ;
                     }
                     else if ( info.selMember == tn_Hottie )
                     {
                        this->dp->HideWin () ;  // hide the prompt dialog
                        Hotkey_Selection () ;
                     }
                     else if ( info.selMember == tn_Tabby )
                     {
                        this->dp->HideWin () ;  // hide the prompt dialog
                        Tabby_Cat () ;
                     }
                     else if ( info.selMember == tn_Billy )
                     {
                        this->dp->HideWin () ;  // hide the prompt dialog
                        Billboard_Demo () ;
                     }
                     else if ( info.selMember == tn_Dropy )
                     {
                        this->dp->HideWin () ;  // hide the prompt dialog
                        Dropdown_Demo () ;
                     }
                     else if ( info.selMember == tn_Manny )
                     {
                        this->dp->HideWin () ;  // hide the prompt dialog
                        Menuwin_Demo () ;
                     }
                     else if ( info.selMember == tn_Spiny )
                     {
                        this->dp->HideWin () ;  // hide the prompt dialog
                        TimeSpinner () ;
                     }

                     this->dp->RefreshWin () ;  // restore prompt dialog
                  }
                  else
                     this->dp->UserAlert () ;
               }
            }
         }
         else
         {  //* Re-open the prompt dialog *
            this->idOpen = this->idOpenPrompt () ;
         }
      }
   }

}  //* End idInteract() *

//*************************
//*    ididStdDialog      *
//*************************
//******************************************************************************
//* Open an empty dialog window and perform specified operation.               *
//*                                                                            *
//* Input  : testNum : indicates which test to perform                         *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void infoDocTest::idStdDialog ( short testNum )
{
   //* Initial parameters for dialog window *
   InitNcDialog dInit( twinROWS,       // number of display lines
                       twinCOLS,       // number of display columns
                       this->minULY,   // Y offset from upper-left of terminal 
                       ZERO,           // X offset from upper-left of terminal 
                       NULL,           // dialog title
                       ncltDUAL,       // border line-style
                       this->hColor,   // border color attribute
                       this->dColor,   // interior color attribute
                       twic            // pointer to list of control definitions
                     ) ;

   //* Instantiate the dialog window *
   NcDialog* dp = new NcDialog ( dInit ) ;

   bool   done = false ;         // loop control

   //* Open the dialog window *
   if ( (dp->OpenWindow()) == OK )
   {
      switch ( testNum )
      {
         case tn_gString1:
            gsBasicStuff ( dp, this->dColor ) ;
            break ;
         case tn_gString2:
            gsInsertAppend ( dp, this->dColor ) ;
            break ;
         case tn_gString3:
            gsShiftCompose ( dp, this->dColor ) ;
            break ;
         case tn_gString4:
            gsSubstrings ( dp, this->dColor ) ;
            break ;
         case tn_MixLR:
            MixedRTL_LTR ( dp, this->dColor ) ;
            break ;
         case tn_RtlTtb:
            Top2BottomRTL ( dp, this->dColor ) ;
            break ;
         case tn_genDlg:
            QuickDialog ( dp, this->dColor ) ;
            done = true ;
            break ;
         default:
            dp->WriteString ( 2, 2, 
                              "Application Error: Request for undefined test.",
                              this->dColor ) ;
      }

      dp->RefreshWin () ;           // make everything visible

      //* Interact with user *
      uiInfo Info ;                 // user interface data returned here
      short  icIndex = ZERO ;       // index of control with input focus
      while ( ! done )
      {
         if ( twic[icIndex].type == dctPUSHBUTTON )
         {
            icIndex = dp->EditPushbutton ( Info ) ;
            if ( Info.dataMod != false )
            {
               if ( Info.ctrlIndex == twDonePB )
                  done = true ;
            }
         }
         else if ( twic[icIndex].type == dctTEXTBOX )
         {
            icIndex = dp->EditTextbox ( Info ) ;
         }
         else if ( twic[icIndex].type == dctBILLBOARD )
         {
            icIndex = dp->EditBillboard ( Info ) ;
         }
         else if ( twic[icIndex].type == dctRADIOBUTTON )
         {
            icIndex = dp->EditRadiobutton ( Info ) ;
         }
         else if ( twic[icIndex].type == dctSCROLLBOX )
         {
            icIndex = dp->EditScrollbox ( Info ) ;
         }
         else if ( twic[icIndex].type == dctDROPDOWN )
         {
            icIndex = dp->EditDropdown ( Info ) ;
         }
         else if ( twic[icIndex].type == dctMENUWIN )
         {
            icIndex = dp->EditMenuwin ( Info ) ;
         }
         else if ( twic[icIndex].type == dctSCROLLEXT )
         {
            icIndex = dp->EditScrollext ( Info ) ;
         }
         else if ( twic[icIndex].type == dctSPINNER )
         {
            icIndex = dp->EditSpinner ( Info ) ;
         }

         //* Move input focus to next/previous control.*
         if ( done == false && Info.viaHotkey == false )
         {
            if ( Info.keyIn == nckSTAB )
               icIndex = dp->PrevControl () ; 
            else
               icIndex = dp->NextControl () ;
         }
      }  // while()
   }
   else
   {  //* Most likely cause of dialog not opening is that the terminal window  *
      //* is too small. Give user a clue...                                    *
      AudibleAlert aa( 3, 3, 2, 5 ) ;
      dp->UserAlert ( &aa ) ;
   }

   if ( dp != NULL )                         // close the window
      delete ( dp ) ;

}  //* End idStdDialog()


//** - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -  **
//** - - - - - - - - NON-MEMBER METHODS FOR THIS MODULE  - - - - - - - - - -  **
//** - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -  **

//*************************
//*    gsInsertAppend     *
//*************************
//******************************************************************************
//* Demonstrate gString-class basic functionality from the 'gStringExamples'   *
//* section of the API documentation.                                          *
//*                                                                            *
//* Input  :  dp    : pointer to dialog window                                 *
//*           dColor: dialog background color                                  *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

static void gsBasicStuff ( NcDialog* dp, attr_t dColor )
{
   winPos wp( 1, 2 ) ;              // cursor position
   dp->SetDialogTitle ( " gString Class Methods " ) ;
   
   //* concatenate strings *
   wp = dp->WriteParagraph ( wp, "concatenate strings\n", nc.brbl ) ;
   const char* Head = "Where" ;
   const wchar_t* Tail = L"is Carmen Sandiego?\n" ;
   gString gsOut( L" in the world " ) ;
   dp->WriteString ( wp, gsOut, dColor ) ;
   ++wp.ypos ;
   gsOut.append( Tail ) ;
   wp = dp->WriteParagraph ( wp, gsOut, dColor ) ;
   gsOut.insert( Head, ZERO ) ;
   wp = dp->WriteParagraph ( wp, gsOut, dColor ) ;
   
   //* 'compose' method *
   wp = dp->WriteParagraph ( wp, "\n'compose' method\n", nc.brbl ) ;
   const char* utf8String = "We present" ;
   const wchar_t* wideString = L"for your enjoyment:" ;
   const char utf8Char = 'W' ;
   const wchar_t wideChar = L'C' ;
   short int ways = 100 ;
   double dish = 17.57 ;

   gsOut.compose( "%s %S %hd %cays to %Cook %Chicken,\n"
                  "and %.2lf side dishes and beverages!\n",
                  utf8String,
                  wideString,
                  &ways,
                  &utf8Char,
                  &wideChar, &wideChar,
                  &dish ) ;
   wp = dp->WriteParagraph ( wp, gsOut, dColor ) ;

   //* line-break example *
   wp = dp->WriteParagraph ( wp, "\nline-break example "
                                 "(breaks on column, not word)\n", nc.brbl ) ;
   // previous message without line breaks
   gsOut = "We present for your enjoyment: 100 Ways to Cook Chicken, "
           "and 17.57 side dishes and beverages!" ;

   const short ddROWS = 7, ddCOLS = 21 ;
   dp->DrawBox ( wp.ypos, wp.xpos, ddROWS, ddCOLS, nc.brbl ) ;
   short tWidth = (ddCOLS - 2),          // inside width of target window
         maxWidth = tWidth * (ddROWS-2), // max columns for message
         tLines = ZERO ;                 // loop counter
   ++wp.ypos ;
   ++wp.xpos ;

   gString gsw( gsOut.gstr() ), gso ;
   if ( gsw.gscols() > maxWidth )
   { // truncate the string if necessary
      gsw.limitCols ( maxWidth - 3 ) ;
      gsw.append( L"..." ) ;
   }
   do
   {  // break the source into convenient widths
      gso = gsw ;
      gso.limitCols( tWidth ) ;
      gso.append( L'\n' ) ;
      gsw.shiftCols( -tWidth ) ;
//      this->dpd->ClearLine ( wpos.ypos ) ; // clear target display line
      wp = dp->WriteParagraph ( wp, gso, dColor ) ;
   }
   while ( ++tLines <= (ddROWS - 2) && gsw.gschars() > 1 ) ;

   dp->RefreshWin () ;     // make everything visible

}  //* End gsBasicStuff() *

//*************************
//*    gsInsertAppend     *
//*************************
//******************************************************************************
//* Demonstrate gString-class insert, append, limitChars, limitCols methods.   *
//*                                                                            *
//* Input  :  dp    : pointer to dialog window                                 *
//*           dColor: dialog background color                                  *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

static void gsInsertAppend ( NcDialog* dp, attr_t dColor )
{
   winPos wp( 1, 2 ) ;              // cursor position
   dp->SetDialogTitle ( " gString Class Methods " ) ;
   
   //* 'append()' *
   wp = dp->WriteParagraph ( wp, "'append' method\n", nc.brbl ) ;
   gString gsOut( L"Be kind to your manager.\n" ) ;
   wp = dp->WriteParagraph ( wp, gsOut.gstr(), dColor ) ;
   gsOut.limitChars( gsOut.gschars() - 3 ) ;
   gsOut.append( L", and other lower forms of life.\n" ) ;
   wp = dp->WriteParagraph ( wp, gsOut.gstr(), dColor ) ;

   //* 'insert()' *
   wp = dp->WriteParagraph ( wp, "\n'insert' method\n", nc.brbl ) ;
   gsOut = L"Remember to hurt people!\n" ;
   wp = dp->WriteParagraph ( wp, gsOut.gstr(), dColor ) ;
   gsOut.insert( L"NOT ", 9 ) ;
   wp = dp->WriteParagraph ( wp, gsOut.gstr(), dColor ) ;

   //* 'limitChars()' *
   wp = dp->WriteParagraph ( wp, "\n'limitChars' method\n", nc.brbl ) ;
   gsOut = "This shirt is available in yellow or red.\n" ;
   wp = dp->WriteParagraph ( wp, gsOut.gstr(), dColor ) ;
   gsOut.limitChars( 34 ) ;
   gsOut.append( "only.\n" ) ;
   wp = dp->WriteParagraph ( wp, gsOut.gstr(), dColor ) ;

   //* 'limitCols()' *
   wp = dp->WriteParagraph ( wp, "\n'limitCols' method\n", nc.brbl ) ;
   gsOut = "The manual is located at:\n"
           "http://cdn.funcom.com/aoc/pdf/aoc_manual.pdf\n" ;
   wp = dp->WriteParagraph ( wp, gsOut.gstr(), dColor ) ;
   gsOut.limitCols( 55 ) ;
   wp = dp->WriteParagraph ( wp, gsOut.gstr(), dColor ) ;

   dp->RefreshWin () ;     // make everything visible

}  //* End gsInsertAppend() *

//*************************
//*    gsShiftCompose     *
//*************************
//******************************************************************************
//* Demonstrate gString-class compose, shiftChars, shiftCols methods.          *
//* Also demonstrates the use of gscols(charCount) method.                     *
//*                                                                            *
//* Input  :  dp    : pointer to dialog window                                 *
//*           dColor: dialog background color                                  *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

static void gsShiftCompose ( NcDialog* dp, attr_t dColor )
{
   winPos wp( 1, 2 ) ;              // cursor position
   dp->SetDialogTitle ( " gString Class Methods " ) ;

   //* 'shiftChars()' *
   wp = dp->WriteParagraph ( wp, "'shiftChars' method\n", nc.brbl ) ;
   gString gsOut( "Your balance is: 2,521,697.56 USD\n" ) ;
   wp = dp->WriteParagraph ( wp, gsOut.gstr(), dColor ) ;
   gsOut.shiftChars( -17 ) ;
   wp = dp->WriteParagraph ( wp, gsOut.gstr(), dColor ) ;
   gsOut.shiftChars( 5, L'#' ) ;
   wp = dp->WriteParagraph ( wp, gsOut.gstr(), dColor ) ;

   //* 'shiftCols()' *
   wp = dp->WriteParagraph ( wp, "\n'shiftCols' method\n", nc.brbl ) ;
   gsOut = "您的帐户余额是500元\n" ; // "Your account balance is 500 yuan"
   wp = dp->WriteParagraph ( wp, gsOut.gstr(), dColor ) ;
   gsOut.shiftCols( -14 ) ;
   wp = dp->WriteParagraph ( wp, gsOut.gstr(), dColor ) ;
   gsOut.shiftCols( 5, L'.' ) ;
   wp = dp->WriteParagraph ( wp, gsOut.gstr(), dColor ) ;

   //* 'compose()' *
   wp = dp->WriteParagraph ( wp, "\n'compose' method\n", nc.brbl ) ;
   char      Greeting[] = { "Hello!" } ;
   int       iValue = 27064 ;
   long long int qValue = 7842561 ;
   long int  lValue = 485772 ;
   short int sValue1 = 28875, sValue2 = -261, sValue3 = 529 ;
   bool      flagValue = 1 ;
   float     fltValue = 278.5610234 ;
   double    dblValue = 9982.5610234 ;
   gsOut.compose( "%s - %d %12hd, %-hi, %#hx %08lXh %llu %hhd\n",
                  Greeting, &iValue, &sValue1, &sValue2, &sValue3,
                  &lValue, &qValue, &flagValue ) ;
   wp = dp->WriteParagraph ( wp, gsOut.gstr(), dColor ) ;
   gsOut.compose( "floating downstream:%10.2f and doubling our pace:%.4lf\n",
                  &fltValue, &dblValue ) ;
   wp = dp->WriteParagraph ( wp, gsOut.gstr(), dColor ) ;

   //* 'gscols(short)' *
   wp = dp->WriteParagraph ( wp, 
                             "\n'gscols' method (with array of column counts)\n",
                             nc.brbl ) ;
   gsOut = "Full Path: /home/Fred/Documents/Klingon Dictionary/addendum\n"
           "Trimmed Path to fit space: " ;
   wp = dp->WriteParagraph ( wp, gsOut.gstr(), dColor ) ;
   gsOut.limitChars( 60 ) ;
   gsOut.shiftChars( -11 ) ;
   short colsAvail = 31 ;
   dp->DrawBox ( wp.ypos, wp.xpos, 3, (colsAvail + 2), nc.brbl ) ;
   ++wp.ypos ;
   ++wp.xpos ;
   if ( (gsOut.gscols()) > colsAvail )
   {
      gString gst = gsOut ;
      int   width = gst.gscols(),
            offset = ZERO,
            charCount ;
      const short* colArray = gst.gscols( charCount ) ;

      while ( width > (colsAvail - 3) )
         width -= colArray[offset++] ;
      gsOut.compose( L"...%S", &gst.gstr()[offset] ) ;
   }
   dp->WriteParagraph ( wp, gsOut.gstr(), dColor ) ;

   dp->RefreshWin () ;     // make everything visible

}  //* End gsShiftCompose() *

static void gsSubstrings ( NcDialog* dp, attr_t dColor )
{
   winPos wp( 1, 2 ) ;              // cursor position
   dp->SetDialogTitle ( " gString Class Methods " ) ;

   //************
   //* 'find()' *
   //************
   wp = dp->WriteParagraph ( wp, "'find' method\n", nc.brbl ) ;

   // find first instance of substring "tim" (not case sensitive)
   gString gs( L"Tiny Tim had a titillating tour of Times Square." ) ;
   gString gsOut( "%S\n", gs.gstr() ) ;
   wp = dp->WriteParagraph ( wp, gsOut, dColor ) ;

   short tIndex = gs.find( "tim" ) ;
   gsOut.compose( "1st \"tim\":: %S\n", &gs.gstr()[tIndex] ) ;
   wp = dp->WriteParagraph ( wp, gsOut, dColor ) ;

   // find the next instance of "tim"
   gString gsSub( "tim" ) ;      // search string in a gString object
   tIndex = gs.find( gsSub, tIndex + 1 ) ;
   gsOut.compose( "2nd \"tim\":: %S\n", &gs.gstr()[tIndex] ) ;
   wp = dp->WriteParagraph ( wp, gsOut, dColor ) ;

   // find first instance of substring "ti" (case sensitive)
   tIndex = gs.find( L"ti", 0, true ) ;
   gsOut.compose( "case sensitive \"ti\":: %S\n", &gs.gstr()[tIndex] ) ;
   wp = dp->WriteParagraph ( wp, gsOut, dColor ) ;

   // match the first three characters of the search string
   tIndex = gs.find( L"squirrel", 0, false, 3 ) ;
   gsOut.compose( "1st 3 chars of \"squirrel\":: %S\n", &gs.gstr()[tIndex] ) ;
   wp = dp->WriteParagraph ( wp, gsOut, dColor ) ;

   // find first instance of L'R' (not case sensitive)
   tIndex = gs.find( L'R' ) ;
   gsOut.compose( "1st 'r':: %S\n", &gs.gstr()[tIndex] ) ;
   wp = dp->WriteParagraph ( wp, gsOut, dColor ) ;
   ++wp.ypos ;

   //*************
   //* 'after()' *
   //*************
   wp = dp->WriteParagraph ( wp, "'after' method\n", nc.brbl ) ;

   gs = "I think that a parrot would be an ideal pet." ;
   gsOut.compose( "%S\n", gs.gstr() ) ;
   wp = dp->WriteParagraph ( wp, gsOut, dColor ) ;

   short pIndex = gs.after( L"would" ) ;
   gs.insert( " NOT", pIndex ) ;
   gsOut.compose( "%S\n", gs.gstr() ) ;
   wp = dp->WriteParagraph ( wp, gsOut, dColor ) ;
   ++wp.ypos ;

   //**************
   //* 'substr()' *
   //**************
   wp = dp->WriteParagraph ( wp, "'substr' method\n", nc.brbl ) ;

   // get a copy of the first word starting with 'c'
   gString AusAnimal( "Aardvark Kangaroo Cockatoo Dingo Wombat " ) ;
   gsOut.compose( "%S\n", AusAnimal.gstr() ) ;
   wp = dp->WriteParagraph ( wp, gsOut, dColor ) ;

   gString gsc ;
   short b = AusAnimal.find( " c" ) ;
   if ( b >= 0 )
   {
      short e = AusAnimal.find( L' ', b + 1 ) ;
      if ( e > b )
      {
         AusAnimal.substr( gsc, (b + 1), (e - b - 1) ) ;
         gsOut.compose( L"1st word beginning with 'c': '%S'\n", gsc.gstr() ) ;
         wp = dp->WriteParagraph ( wp, gsOut, dColor ) ;
      }
   }

   dp->RefreshWin () ;     // make everything visible

}  //* End gsShiftCompose() *

//*************************
//*     MixedRTL_LTR      *
//*************************
//******************************************************************************
//* Demonstrate mixed RTL and LTR output.                                      *
//*                                                                            *
//* Input  :  dp    : pointer to dialog window                                 *
//*           dColor: dialog background color                                  *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

static void MixedRTL_LTR ( NcDialog* dp, attr_t dColor )
{
   // Yiddish is an RTL language: "The price is " + " including tax!"
   const wchar_t* mHead = L"די פּרייַז איז " ;
   const wchar_t* mTail = L" אַרייַנגערעכנט שטייַער!" ;
   wchar_t mCost[] = { L"$123,456.78usd" } ;
   dp->WriteParagraph ( 2, 2,
               L"This is a test of mixed LTR and RTL output:\n"
               "  First, the unmodified data output as RTL  :\n\n"
               "  Next, output separately with cursor mods  :\n\n"
               "  Last, output as RTL with reversed LTR data: ", dColor ) ;
   winPos wp( 4, 51 ) ;                      // cursor position
   gString gsOut ;                           // work buffer

   // (Incorrectly) output the unmodified data.
   gsOut.compose( "%S%S%S\n\n", mHead, mCost, mTail ) ;
   wp = dp->WriteParagraph ( wp, gsOut, nc.brbl, false, true ) ;

   // Output sections separately, adjusting cursor position as necessary.
   winPos wx = dp->WriteString ( wp, mHead, nc.brbl, false, true ) ;
   gsOut = mCost ;
   wx.xpos -= gsOut.gscols() - 1 ;
   dp->WriteString ( wx, gsOut, nc.brbl ) ;
   --wx.xpos ;
   dp->WriteString ( wx, mTail, nc.brbl, false, true ) ;

   // Contatenate the modified data and output as RTL.
   wp.ypos += 2 ;
   gsOut = mCost ;
   gsOut.textReverse() ;
   gsOut.insert( mHead ) ;
   gsOut.append( mTail ) ;
   dp->WriteParagraph ( wp, gsOut, nc.brbl, false, true ) ;
   dp->RefreshWin () ;        // make everything visible

}  //* End MixedRTL_LTR() *

//*************************
//*     Top2BottomRTL     *
//*************************
//******************************************************************************
//* Demonstrate top-to-bottom, right-to-left output.                           *
//*                                                                            *
//* Input  :  dp    : pointer to dialog window                                 *
//*           dColor: dialog background color                                  *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

static void Top2BottomRTL ( NcDialog* dp, attr_t dColor )
{
   const short pLines = 5, pCols = 8 ;
   const wchar_t aPoem[pLines][pCols] = 
   {  // (yes, my teachers and I agree that my Chinese is abysmal.)
     L"我的静的花园",         // In my quiet garden
     L"鱼是很有礼貌，",        // the fish are so polite,
     L"他们很少说话",         // they seldom speak
     L"打岔我的梦想。",        // to interrupt my reverie.
     L"马伦"              // Software Sam
   } ;
   wchar_t outBuff[] = { nckNULLCHAR, nckNEWLINE, nckNULLCHAR } ;
   attr_t  outColor[] = { nc.gr, nc.br, nc.bl, nc.ma, nc.reG, nc.cy } ;

   dp->WriteString ( 1, 2,
     "A (bad) poem output as top-to-bottom, right-to-left.", dColor ) ;
   const short startY = 3 ;
   winPos wp( startY, 20 ) ;
   for ( short i = ZERO ; i < pLines ; i++ )
   {
     winPos wx = wp ;
     for ( short j = ZERO ; aPoem[i][j] != nckNULLCHAR ; j++ )
     {
        outBuff[ZERO] = aPoem[i][j] ;
        wx = dp->WriteParagraph ( wx, outBuff, outColor[i], false, true ) ;
     }
     wp.xpos -= 2 ;
     if ( i == 3 )
     { wp.ypos += 5 ; --wp.xpos ; }
   }
   dp->RefreshWin () ;     // make everything visible

}  //* End Top2BottomRTL() *

//*************************
//*      QuickDialog      *
//*************************
//******************************************************************************
//* Demonstrate the genDialog class for both LTR and RTL data.                 *
//*                                                                            *
//* Input  :  dp    : pointer to dialog window                                 *
//*           dColor: dialog background color                                  *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

static void QuickDialog ( NcDialog* dp, attr_t dColor )
{
   attr_t colors[] = 
   {
      nc.brR, nc.recy, nc.blcy, nc.macy, nc.bkcy, nc.brcy, nc.cyR
   } ;
   const char* poemY[] =       // Yiddish translation
   {
      "  ראָמאַנס  ",            // "romance"
      "רויזן זענען רויט",      // "Roses are red"
      "וויילאַץ זענען בלוי",    // Violets are blue
      "צוקער איז זיס",         // Sugar is sweet
      "און מאכט איר פעט",      // And makes you fat
      " ",
      NULL
   } ;
   const char* poemE[] =      // English translation
   {
      "  Romance  ",
      "Roses are red",
      "Violets are blue",
      "Sugar is sweet",
      "And makes you fat.",
      " ",
      NULL
   };
   const char* poemC[] =      // Chinese translation
   {
      "  浪漫的爱情  ",         // "Romantic Love"
      "玫瑰是昂贵",             // "Roses are expensive"
      "不是紫罗兰的季节",        // "Violets aren't in season"
      "糖是育肥",              // "Sugar is fattening"
      "此外，你永远不会",        // "And, you will never"
      "      找到一个丈夫",    // "find a husband.",
      NULL
   };

   genDialog gdy( poemY, nc.cyR, 8, 22, -1, -1, colors, true ) ;
   dp->InfoDialog ( gdy ) ;

   genDialog gde( poemE, nc.cyR, 8, 22, -1, -1, colors, false ) ;
   dp->InfoDialog ( gde ) ;

   genDialog gdc( poemC, nc.cyR, 8, 22, -1, -1, colors, false ) ;
   dp->InfoDialog ( gdc ) ;

}  //* End QuickDialog() *

//*************************
//*     LineDialog01      *
//*************************
//******************************************************************************
//* Line drawing and auto-connection example.                                  *
//* See Texinfodocs: 'NcDialog Display Output'.                                *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************
//* Note that the CaptureDialog() calls are commented out. We only needed them *
//* to capture data for the Texinfo documentation.                             *
//******************************************************************************

static void LineDialog01 ( void )
{
   const short  dialogROWS = 17,       // display lines
                dialogCOLS  = 37,      // display columns
                ulY = 3,               // upper left corner in Y
                ulX = 2 ;              // upper left corner in X
   const attr_t dColor = nc.blR,       // dialog background color
                bColor = nc.brbl ;     // dialog border color

   //* Initial parameters for dialog window *
   InitNcDialog dInit( dialogROWS,     // number of display lines
                       dialogCOLS,     // number of display columns
                       ulY,            // Y offset from upper-left of terminal 
                       ulX,            // X offset from upper-left of terminal 
                       NULL,           // no title
                       ncltDUAL,       // border line-style
                       bColor,         // border color attribute
                       dColor,         // interior color attribute
                       NULL            // no controls defined
                     ) ;

   //* Instantiate the dialog window *
   NcDialog* dp = new NcDialog ( dInit ) ;

   //* Open the dialog window *
   if ( (dp->OpenWindow()) == OK )
   {  //* Write some data into the dialog window *
      dp->WriteParagraph ( 1, 4, "Section #1         Section #2\n\n\n\n\n\n\n\n"
                                 "Section #3         Section #4", dColor ) ;
      dp->RefreshWin () ;

      //* 'Before' dialog capture *
//      dp->CaptureDialog ( "sbcap1.txt" ) ;
//      dp->CaptureDialog ( "sbcap1.htm", true, false, "infodoc-styles.css", 4, false, nc.blR ) ;
      sleep ( 1 ) ;

      winPos wp( dialogROWS / 2, ZERO ) ;
      LineDef  lDef(ncltHORIZ,      // direction of line
                    ncltDUAL,       // line style
                    dialogROWS / 2, // Y start
                    ZERO,           // X start
                    dialogCOLS,     // line length (columns)
                    bColor ) ;      // line color

      dp->DrawLine ( lDef ) ;
      dp->RefreshWin () ;
      sleep ( 1 ) ;

      lDef.type = ncltVERT ;        // direction of line
      lDef.startY = ZERO ;          // Y start
      lDef.startX = dialogCOLS / 2; // X start
      lDef.length = dialogROWS ;    // line length (rows)
      dp->DrawLine ( lDef ) ;
      dp->RefreshWin () ;

      //* 'After' dialog capture *
//      dp->CaptureDialog ( "sbcap2.txt" ) ;
//      dp->CaptureDialog ( "sbcap2.htm", true, false, "infodoc-styles.css", 4, false, nc.blR ) ;

      sleep ( 1 ) ;
      dp->WriteParagraph ( (dialogROWS - 3), 3, 
                           "Press any key\n  to exit...", nc.grbl, true ) ; 
      nckPause();                   // wait for user
   }
   if ( dp != NULL )                         // close the window
      delete ( dp ) ;

}  //* End LineDialog01() *

//*************************
//*   Set_ClickInterval   *
//*************************
//******************************************************************************
//* Example of using a Spinner control to help user adjust the mouse-click     *
//* interval.                                                                  *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

static void Set_ClickInterval ( void )
{
   const short  dialogROWS = 10,       // display lines
                dialogCOLS  = 41,      // display columns
                ulY = 3,               // upper left corner in Y
                ulX = 21 ;             // upper left corner in X
   const attr_t dColor = nc.blR,       // dialog background color
                bColor = nc.brbl ;     // dialog border color
   //* Data for the ciSP spinner control *
   dspinData dsData( 20, ncmiMAX, ncmiSTABLE, dspinINTEGER, nc.grG ) ;
   enum ciControls : short { ciSP = ZERO, ciPB, ciControlsDEFINED } ;

   InitCtrl ic[ciControlsDEFINED] =   // control initialization structures
   {
      {  //* 'DELAY' spinner - - - - - - - - - - - - - - - - - - -        ciSP *
         dctSPINNER,                   // type:      
         rbtTYPES,                     // rbSubtype: (n/a)
         false,                        // rbSelect:  (n/a)
         short(dialogROWS - 3),        // ulY:       upper left corner in Y
         short(dialogCOLS / 2 - 7),    // ulX:       upper left corner in X
         1,                            // lines:     (n/a)
         11,                           // cols:      control columns
         NULL,                         // dispText:  (n/a)
         nc.gyR,                       // nColor:    non-focus color
         nc.mabk,                      // fColor:    focus color
         tbPrint,                      // filter:    (n/a)
         "Click Delay",                // label:     
         1,                            // labY:      
         ZERO,                         // labX       
         ddBoxTYPES,                   // exType:    (n/a)
         1,                            // scrItems:  (n/a)
         1,                            // scrSel:    (n/a)
         NULL,                         // scrColor:  (n/a)
         &dsData,                      // spinData:  spinner init
         true,                         // active:    allow control to gain focus
         &ic[ciPB],                    // nextCtrl:  link in next structure
      },
      {  //* 'DONE' pushbutton - - - - - - - - - - - - - - - - - - -      ciPB *
         dctPUSHBUTTON,                // type:      
         rbtTYPES,                     // rbSubtype: (n/a)
         false,                        // rbSelect:  (n/a)
         ic[ciSP].ulY,                 // ulY:       upper left corner in Y
         short(dialogCOLS - 10),       // ulX:       upper left corner in X
         1,                            // lines:     control lines
         8,                            // cols:      control columns
         "  DONE  ",                   // dispText:  
         nc.gyR,                       // nColor:    non-focus color
         nc.reG,                       // fColor:    focus color
         tbPrint,                      // filter:    (n/a)
         "",                           // label:     (n/a)
         ZERO,                         // labY:      (n/a)
         ZERO,                         // labX       (n/a)
         ddBoxTYPES,                   // exType:    (n/a)
         1,                            // scrItems:  (n/a)
         1,                            // scrSel:    (n/a)
         NULL,                         // scrColor:  (n/a)
         NULL,                         // spinData:  (n/a)
         true,                         // active:    allow control to gain focus
         NULL                          // nextCtrl:  link in next structure
      },
   } ;
   //* Initial parameters for dialog window *
   InitNcDialog dInit( dialogROWS,     // number of display lines
                       dialogCOLS,     // number of display columns
                       ulY,            // Y offset from upper-left of terminal 
                       ulX,            // X offset from upper-left of terminal 
                       "  Adjust Click Interval  ", // title
                       ncltDUAL,       // border line-style
                       bColor,         // border color attribute
                       dColor,         // interior color attribute
                       ic              // list of controls
                     ) ;

   //* Instantiate the dialog window *
   NcDialog* dp = new NcDialog ( dInit ) ;

   //* Open the dialog window *
   if ( (dp->OpenWindow()) == OK )
   {
      bool     done = false ;             // loop control

      //* Enable stable-mouse input:                                      *
      //* Button 1, single/double/triple clicks, plus ScrollWheel events. *
      if ( (dp->meEnableStableMouse ()) == OK )
      {
//         dp->meAutoConvert ( false ) ; // Do not convert mouse events to keycodes

         dp->WriteString ( 2, 4, "Click the mouse inside the window.", bColor ) ;
   
         //* Callback method to monitor the Spinner Value *
         sciPtr = dp ;
         dp->EstablishCallback ( &ciCallback ) ;
         dp->RefreshWin () ;
      }
      else
      {
         dp->WriteString ( 2, 5, "Unable to initialize the mouse.", nc.rebl, true ) ;
         nckPause();
         done = true ;
      }

      //* Track user's selections *
      uiInfo   Info ;                     // user interface data returned here
      short    icIndex = ZERO ;           // index of control with input focus

      while ( ! done )
      {
         if ( ic[icIndex].type == dctPUSHBUTTON )
         {
            if ( Info.viaHotkey != false )
               Info.HotData2Primary () ;
            else
               icIndex = dp->EditPushbutton ( Info ) ;
            if ( Info.dataMod != false )
               done = true ;
         }
         else if ( ic[icIndex].type == dctSPINNER )
         {
            icIndex = dp->EditSpinner ( Info ) ;
         }
         //* Move to next/previous control *
         if ( done == false && !Info.viaHotkey )
         {
            if ( Info.keyIn == nckSTAB )
               icIndex = dp->PrevControl () ; 
            else if ( Info.keyIn != ZERO )
               icIndex = dp->NextControl () ;
         }
      }  // while()
   }
   if ( (dp->meMouseEnabled ()) )            // disable mouse interface
      dp->meDisableMouse () ;
   if ( dp != NULL )                         // close the window
      delete ( dp ) ;

}  //* End Set_ClickInterval() *

//*************************
//*      ciCallback       *
//*************************
//******************************************************************************
//* Callback method for adjusting the mouse click interval.                    *
//*                                                                            *
//* a) Monitors the dctSPINNER controls and adjusts the mouse click interval.  *
//* b) Reports detected mouse events.                                          *
//*                                                                            *
//* NOTE: This method make intimate assumptions about the parent dialog.       *
//*       Keep it synchronized!                                                *
//*                                                                            *
//* Input  : currIndex: index of control that currently has focus              *
//*          wkey     : user's key input data                                  *
//*          firstTime: the EstablishCallback() method calls this method once  *
//*                     with firstTime==true, to perform any required          *
//*                     initialization. Subsequently, the NcDialog class       *
//*                     always calls with firstTime==false.                    *
//* Returns: OK                                                                *
//******************************************************************************
//* Programmer's Note: Testing for an event that occurs INSIDE the window is   *
//* a bit tricky. 'ypos' || 'xpos' > ZERO, then IS inside window               *
//*               'ypos' == ZERO && 'xpos' == ZERO may be a valid origin OR    *
//*               an event outside the window.                                 *
//* For this reason, we don't capture a valid event at window-relative 0/0     *
//* without the extra test.                                                    *
//******************************************************************************

static short ciCallback ( const short currIndex, 
                          const wkeyCode wkey, bool firstTime )
{
   static int cInterval = ncmiNONE ;
   const short msgROW = 3, msgCOL = 4, posROW = 5, posCOL = 12 ;
   const attr_t dColor = nc.blR ;

   //* First-time processing *
   if ( firstTime != false )
   {
      sciPtr->GetSpinnerValue ( ZERO, cInterval ) ;
   }
   //* If user has updated Spinner value, adjust click interval. *
   else
   {
      int newInterval ;
      sciPtr->GetSpinnerValue ( ZERO, newInterval ) ;
      if ( newInterval != cInterval )
      {
         cInterval = newInterval ;
         sciPtr->meDisableMouse () ;
         sciPtr->meEnableStableMouse ( cInterval ) ;
      }
   }

   //* Report selected mouse events *
   if ( (! firstTime && (wkey.type == wktMOUSE)) &&
        ((wkey.mevent.ypos > ZERO || wkey.mevent.xpos > ZERO)
         ||
         (wkey.mevent.sypos == 3 && wkey.mevent.sxpos == 21)) )
   {
      sciPtr->UserAlert();
      gString gs( "ypos:%2hd xpos:%2hd", &wkey.mevent.ypos, &wkey.mevent.xpos ) ;
      sciPtr->ClearLine ( posROW ) ;
      sciPtr->WriteString ( posROW, posCOL, gs, dColor, true ) ;

      sciPtr->ClearLine ( msgROW ) ;
      if ( wkey.mevent.meType == metB1_S )
         sciPtr->WriteString ( msgROW, msgCOL, "Button #1 - Single Click", dColor, true ) ;
      else if ( wkey.mevent.meType == metB1_D )
         sciPtr->WriteString ( msgROW, msgCOL, "Button #1 - Double Click", dColor, true ) ;
      else if ( wkey.mevent.meType == metB1_T )
         sciPtr->WriteString ( msgROW, msgCOL, "Button #1 - Triple Click", dColor, true ) ;
      else
      {
         mmask_t emt ;
         sciPtr->meGetEventTypes ( emt ) ;
         if ( (emt & REPORT_SCROLL_WHEEL) == REPORT_SCROLL_WHEEL )
         {
            if ( wkey.mevent.meType == metSW_D )
               sciPtr->WriteString ( msgROW, msgCOL, "ScrollWheel = Scroll Down", dColor, true ) ;
            else if ( wkey.mevent.meType == metSW_U )
               sciPtr->WriteString ( msgROW, msgCOL, "ScrollWheel = Scroll Up", dColor, true ) ;
         }
      }
   }

   //* Take a screenshot for the documentation *
   else if ( wkey.type == wktEXTEND )
   {
      if ( wkey.key == nckA_Z)
         sciPtr->CaptureDialog ( "capturedlg.txt" ) ;
      if ( wkey.key == nckAS_Z)
         sciPtr->CaptureDialog ( "capturedlg.html", true, false, 
                                 "infodoc-styles.css", 4, false, dColor ) ;
   }

   return OK ;

}  //* End ciCallback() *

//*************************
//*   Hotkey_Selection    *
//*************************
//******************************************************************************
//* Examples of control selection and line-item selection via Hotkey.          *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

static void Hotkey_Selection ( void )
{
   const short  dialogROWS = 8,        // display lines
                dialogCOLS  = 28,      // display columns
                ulY = 3,               // upper left corner in Y
                ulX = 21 ;             // upper left corner in X
   const attr_t dColor = nc.blR,       // dialog background color
                bColor = nc.brbl ;     // dialog border color
   enum hsControls : short { hsPB = ZERO, hsMW, hsControlsDEFINED } ;

   const short DATA_ITEMS = 12; // items in list
   const short ITEM_WIDTH = 20; // chars per item (incl. NULLCHAR)
   //* Display text for Scrollbox control *
   const char TeamNames[DATA_ITEMS][ITEM_WIDTH + 1] = 
   {
      " ^Arsenal           ",
      " Aston ^Villa       ",
      " ^Blackburn         ",
      " B^ournemouth       ",
      " B^urnley           ",
      " ^Chelsea           ",
      " ^Liverpool         ",
      " ^Manchester United ",
      " Mancheste^r City   ",
      " ^Newcastle United  ",
      " ^Southhampton      ",
      " ^West Ham United   ",
   };
   attr_t monoColor[2] = { attrDFLT, nc.gr }; // color attributes for list

   InitCtrl ic[hsControlsDEFINED] = // array of dialog control initialization objects
   {
   {  //* 'DONE' pushbutton - - - - - - - - - - - - - - - - - - - - - -   hsPB *
      dctPUSHBUTTON,    // type:      
      rbtTYPES,         // rbSubtype: (n/a)
      false,            // rbSelect:  (n/a)
      short(dialogROWS - 2),     // ulY: upper left corner in Y
      short(dialogCOLS / 2 - 4), // ulX: upper left corner in X
      1,                // lines:     control lines
      8,                // cols:      control columns
      "  ^DONE  ",      // dispText:  
      nc.gyR,           // nColor:    non-focus color
      nc.reG,           // fColor:    focus color
      tbPrint,          // filter:    (n/a)
      NULL,             // label:     (n/a)
      ZERO,             // labY:      (n/a)
      ZERO,             // labX       (n/a)
      ddBoxTYPES,       // exType:    (n/a)
      1,                // scrItems:  (n/a)
      1,                // scrSel:    (n/a)
      NULL,             // scrColor:  (n/a)
      NULL,             // spinData:  (n/a)
      true,             // active:    allow control to gain focus
      &ic[hsMW]         // nextCtrl:  link in next structure
   },
   {  //* 'Team' Menuwin  - - - - - - - - - - - - - - - - - - - - - - -   hsMW *
      dctMENUWIN,       // type:      define a scrolling-data control
      rbtTYPES,         // rbSubtype: (n/a)
      false,            // rbSelect:  (n/a)
      2,                // ulY:       upper left corner in Y
      3,                // ulX:       upper left corner in X
      1,                // lines:     (n/a)
      (ITEM_WIDTH + 2), // cols:      control columns
      (const char*)&TeamNames, // dispText:  text-data array
      nc.gr,            // nColor:    non-focus border color
      nc.grR,           // fColor:    focus border color
      tbPrint,          // filter:    (n/a)
      " Football ^team ", // label:     label text
      -1,               // labY:      label offset
      ZERO,             // labX       
      ddBoxTYPES,       // exType:    (n/a)
      DATA_ITEMS,       // scrItems:  elements in text/color arrays
      ZERO,             // scrSel:    (n/a)
      monoColor,        // scrColor:  color-attribute list
      NULL,             // spinData:  (n/a)
      true,             // active:    allow control to gain focus
      NULL              // nextCtrl:  link in next structure
   },
   } ;

   //* Initial parameters for dialog window *
   InitNcDialog dInit( dialogROWS,     // number of display lines
                       dialogCOLS,     // number of display columns
                       ulY,            // Y offset from upper-left of terminal 
                       ulX,            // X offset from upper-left of terminal 
                       "  Premier League  ", // title
                       ncltDUAL,       // border line-style
                       bColor,         // border color attribute
                       dColor,         // interior color attribute
                       ic              // list of controls
                     ) ;

   //* Instantiate the dialog window *
   NcDialog* dp = new NcDialog ( dInit ) ;

   //* Open the dialog window *
   if ( (dp->OpenWindow()) == OK )
   {
      //* Track user's selections *
      uiInfo   Info ;                     // user interface data returned here
      short    icIndex = ZERO ;           // index of control with input focus
      bool     done = false ;             // loop control

      while ( ! done )
      {
         if ( ic[icIndex].type == dctPUSHBUTTON )
         {  //* If we arrived here via hotkey, just unpack the data.*
            if ( Info.viaHotkey )
               Info.HotData2Primary () ;
            //* Else, call the edit routine to discover what the user wants.*
            else
               icIndex = dp->EditPushbutton ( Info ) ;
            if ( Info.dataMod != false )
               done = true ;
         }

         else if ( ic[icIndex].type == dctMENUWIN )
         {
            icIndex = dp->EditMenuwin ( Info ) ;

            //* Because the expanded menu extends below the parent *
            //* dialog window, we must refresh the terminal window *
            //* to remove the artifacts, THEN refresh the dialog.  *
            nc.ClearScreen () ;

            //* (this string manipulation is too cute :-) *
            gString gs( TeamNames[Info.selMember] ) ;
            short i = gs.find( CARET ) ;
            gs.limitChars( i++ ) ;
            gs.append( &TeamNames[Info.selMember][i] ) ;
            dp->WriteString ( (ic[hsMW].ulY + 2), (ic[hsMW].ulX + 2), 
                              gs, dColor ) ;
            dp->RefreshWin () ;
         }

         //* Move to next/previous control *
         if ( done == false && !Info.viaHotkey )
         {
            if ( Info.keyIn == nckSTAB )
               icIndex = dp->PrevControl () ; 
            else if ( Info.keyIn != ZERO )
               icIndex = dp->NextControl () ;
         }
      }  // while()
   }
   if ( dp != NULL )                         // close the window
      delete ( dp ) ;

}  //* End Hotkey_Selection() *

//*************************
//*       Tabby_Cat       *
//*************************
//******************************************************************************
//* Demonstrates a tabbed-dialog group construct.                              *
//* We open a parent dialog and then call the method for the initially-visible *
//* tabbed object. The user may then select among the tabbed objects and       *
//* interact with the controls of that object.                                 *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************
//* Notes:                                                                     *
//* -- The tabs indicating the separate dialog windows are actually implemented*
//*    in each window, independently. This allows the user to select a         *
//*    different tab, regardless of which dialog is currently displayed.       *
//* -- The default dialog (English in this case) defines the InitCtrl class    *
//*    objects for these tab controls, as well as the 'DONE' control.          *
//*    These definitions are passed to each of the other tab dialogs with the  *
//*    understanding that they will be returned in the same state in which     *
//*    they arrived.                                                           *
//* -- To implement the tabs on the top line of the dialog:                    *
//*    a) we erase the border from the top line of the dialog                  *
//*    b) we draw a box around the remaining dialog area                       *
//* -- To visually manage the tabs, we explicitly set the color of the tab     *
//*    (Pushbutton) associated with the currently-displayed dialog.            *
//* -- To add additional controls, the called dialog makes a copy of the basic *
//*    controls and interleaves the new controls. The sequence of control      *
//*    access is adjusted as necessary. Give special attention to indexing of  *
//*    basic versus customized arrays.                                         *
//* -- Note that we have assigned Hotkeys only to the Tab controls, NOT        *
//*    the 'Done' Pushbutton. This is because the text of the 'Done' button    *
//*    changes in each dialog, and trying to use a consistent Hotkey would     *
//*    only confuse the user.                                                  *
//*    -- Note also that because Hotkeys may only be assigned to ASCII         *
//*       characters, we had to add a 'c' to the title of the Chinese tab.     *
//*                                                                            *
//******************************************************************************

static void Tabby_Cat ( void )
{
   const short  dialogROWS = 21,       // display lines
                dialogCOLS  = 69,      // display columns
                ulY = 3,               // upper left corner in Y
                ulX = 7 ;              // upper left corner in X
   const attr_t dColor = nc.blR,       // dialog background color
                bColor = nc.brbl ;     // dialog border color
   winPos wp( dialogROWS - 7, dialogCOLS / 2 - 11 ) ;

   const char* dialogTitle = "  English Translation  " ;
   const char* Chuck_English = 
               "    Despite his manly name, Chuck was seldom asked to join the\n"
               "games of the other boys. First, he was rather small for his age\n"
               "and had the upper body strength of a 5-year-old, even though he\n"
               "was nearly nine. The continual asthma attacks and the\n"
               "prepubescent OCD contributed something as well.\n"
               "    Yes, Chuck was a nerd, but he had a burning desire to express\n"
               "himself and an overwhelming need to create something that would\n"
               "be of lasting value to someone, though his peers would probably\n"
               "still steal his lunch, and he was sure that his family would\n"
               "just continue to stare at him in slack-jawed incomprehension." ;
   const char* tabNames[] =   // Names for basic controls
   {
      "  DONE  ",
      "│ Englis^h  ",
      "│ Españo^l  ",
      "│Tiếng Việ^t  ",
      "│  华语  ^c│",
   } ;
   attr_t tabnColor[] =       // Non-focus colors for basic controls
   { nc.gyR, nc.gyR, nc.gyR, nc.gyR, nc.gyR } ;
   attr_t tabfColor[] =       // Focus colors for basic controls
   { nc.reR, nc.reR, nc.reR, nc.reR, nc.reR } ;
   attr_t tabsColor = nc.bkgr ; // 'Selected' color for basic controls

//* Control definitions for prompt dialog *
InitCtrl ic[tcCONTROLS_DEFINED] = 
{
   {  //* 'DONE' Pushbutton  - - - - - - - - - - - - - - - - - - -    tcDonePB *
   dctPUSHBUTTON,                // type:      
   rbtTYPES,                     // rbSubtype: (n/a)
   false,                        // rbSelect:  (n/a)
   short(dialogROWS - 2),        // ulY:       upper left corner in Y
   8,                            // ulX:       upper left corner in X
   1,                            // lines:     control lines
   8,                            // cols:      control columns
   tabNames[tcDonePB],           // dispText:  
   tabnColor[tcDonePB],          // nColor:    non-focus color
   tabfColor[tcDonePB],          // fColor:    focus color
   tbPrint,                      // filter:    (n/a)
   NULL,                         // label:     (n/a)
   ZERO,                         // labY:      (n/a)
   ZERO,                         // labX       (n/a)
   ddBoxTYPES,                   // exType:    (n/a)
   1,                            // scrItems:  (n/a)
   1,                            // scrSel:    (n/a)
   NULL,                         // scrColor:  (n/a)
   NULL,                         // spinData:  (n/a)
   true,                         // active:    allow control to gain focus
   &ic[tcTabE]                   // nextCtrl:  link in next structure
   },
   {  //* 'ENGLISH TAB' Pushbutton   - - - - - - - - - - - - - - - -    tcTabE *
   dctPUSHBUTTON,                // type:      
   rbtTYPES,                     // rbSubtype: (n/a)
   false,                        // rbSelect:  (n/a)
   ZERO,                         // ulY:       upper left corner in Y
   1,                            // ulX:       upper left corner in X
   1,                            // lines:     control lines
   11,                           // cols:      control columns
   tabNames[tcTabE],             // dispText:  
   tabnColor[tcTabE],            // nColor:    non-focus color
   tabfColor[tcTabE],            // fColor:    focus color
   tbPrint,                      // filter:    (n/a)
   NULL,                         // label:     (n/a)
   ZERO,                         // labY:      (n/a)
   ZERO,                         // labX       (n/a)
   ddBoxTYPES,                   // exType:    (n/a)
   1,                            // scrItems:  (n/a)
   1,                            // scrSel:    (n/a)
   NULL,                         // scrColor:  (n/a)
   NULL,                         // spinData:  (n/a)
   true,                         // active:    allow control to gain focus
   &ic[tcTabS]                   // nextCtrl:  link in next structure
   },
   {  //* 'SPANISH TAB' Pushbutton   - - - - - - - - - - - - - - - -    tcTabS *
   dctPUSHBUTTON,                // type:      
   rbtTYPES,                     // rbSubtype: (n/a)
   false,                        // rbSelect:  (n/a)
   ZERO,                         // ulY:       upper left corner in Y
   short(ic[tcTabE].ulX + ic[tcTabE].cols), // ulX: upper left corner in X
   1,                            // lines:     control lines
   11,                           // cols:      control columns
   tabNames[tcTabS],             // dispText:  
   tabnColor[tcTabS],            // nColor:    non-focus color
   tabfColor[tcTabS],            // fColor:    focus color
   tbPrint,                      // filter:    (n/a)
   NULL,                         // label:     (n/a)
   ZERO,                         // labY:      (n/a)
   ZERO,                         // labX       (n/a)
   ddBoxTYPES,                   // exType:    (n/a)
   1,                            // scrItems:  (n/a)
   1,                            // scrSel:    (n/a)
   NULL,                         // scrColor:  (n/a)
   NULL,                         // spinData:  (n/a)
   true,                         // active:    allow control to gain focus
   &ic[tcTabV]                   // nextCtrl:  link in next structure
   },
   {  //* 'VIETNAMESE TAB' Pushbutton    - - - - - - - - - - - - - -    tcTabV *
   dctPUSHBUTTON,                // type:      
   rbtTYPES,                     // rbSubtype: (n/a)
   false,                        // rbSelect:  (n/a)
   ZERO,                         // ulY:       upper left corner in Y
   short(ic[tcTabS].ulX + ic[tcTabS].cols), // ulX: upper left corner in X
   1,                            // lines:     control lines
   12,                           // cols:      control columns
   tabNames[tcTabV],             // dispText:  
   tabnColor[tcTabV],            // nColor:    non-focus color
   tabfColor[tcTabV],            // fColor:    focus color
   tbPrint,                      // filter:    (n/a)
   NULL,                         // label:     (n/a)
   ZERO,                         // labY:      (n/a)
   ZERO,                         // labX       (n/a)
   ddBoxTYPES,                   // exType:    (n/a)
   1,                            // scrItems:  (n/a)
   1,                            // scrSel:    (n/a)
   NULL,                         // scrColor:  (n/a)
   NULL,                         // spinData:  (n/a)
   true,                         // active:    allow control to gain focus
   &ic[tcTabC]                   // nextCtrl:  link in next structure
   },
   {  //* 'CHINESE TAB' Pushbutton     - - - - - - - - - - - - - - -    tcTabC *
   dctPUSHBUTTON,                // type:      
   rbtTYPES,                     // rbSubtype: (n/a)
   false,                        // rbSelect:  (n/a)
   ZERO,                         // ulY:       upper left corner in Y
   short(ic[tcTabV].ulX + ic[tcTabV].cols), // ulX: upper left corner in X
   1,                            // lines:     control lines
   11,                           // cols:      control columns
   tabNames[tcTabC],             // dispText:  
   tabnColor[tcTabC],            // nColor:    non-focus color
   tabfColor[tcTabC],            // fColor:    focus color
   tbPrint,                      // filter:    (n/a)
   NULL,                         // label:     (n/a)
   ZERO,                         // labY:      (n/a)
   ZERO,                         // labX       (n/a)
   ddBoxTYPES,                   // exType:    (n/a)
   1,                            // scrItems:  (n/a)
   1,                            // scrSel:    (n/a)
   NULL,                         // scrColor:  (n/a)
   NULL,                         // spinData:  (n/a)
   true,                         // active:    allow control to gain focus
   NULL                          // nextCtrl:  link in next structure
   },
} ;

   //* Initial parameters for dialog window *
   InitNcDialog dInit( dialogROWS,     // number of display lines
                       dialogCOLS,     // number of display columns
                       ulY,            // Y offset from upper-left of terminal 
                       ulX,            // X offset from upper-left of terminal 
                       NULL,           // no title for the basic dialog
                       ncltDUAL,       // border line-style
                       bColor,         // border color attribute
                       dColor,         // interior color attribute
                       ic              // list of controls
                     ) ;

   //* Instantiate the dialog window *
   NcDialog* dp = new NcDialog ( dInit ) ;

   //* Open the dialog window *
   if ( (dp->OpenWindow()) == OK )
   {
      //* Draw initial static text *
      tcCatBox ( dp, dialogTitle, tabNames[tcTabE], 
                 tabfColor[tcTabE], tabsColor, bColor, 
                 dialogROWS, dialogCOLS, tcTabE ) ;

      dp->DrawBox ( wp.ypos, wp.xpos, 6, dialogCOLS / 2 + 10, dColor, 
                    " Welcome to the Language Center ", ncltDASH3 ) ;
      dp->WriteParagraph ( wp.ypos + 1, wp.xpos + 1,
                           "This story is written in moderately-fluent\n"
                           "American English. If you know one of the\n"
                           "other languages listed, we invite you to\n"
                           "evaluate our translations.   Enjoy!", dColor );

      //* Enable stable-mouse input:                                      *
      //* Button 1, single/double/triple clicks, plus ScrollWheel events. *
      if ( (dp->meEnableStableMouse ()) != OK )
      {
         dp->WriteParagraph ( (dialogROWS - 6), 2, 
                              "Mouse-initialization\n"
                              "       failed.", nc.rebl ) ;
      }

      //* Tell our story *
      dp->WriteParagraph ( 2, 2, Chuck_English, dColor ) ;

      dp->RefreshWin () ;                 // make everything visible

      //* Track user's selections *
      uiInfo   Info ;                     // user interface data returned here
      short    icIndex = ZERO ;           // index of control with input focus
      bool     done = false ;             // loop control

      while ( ! done )
      {
         if ( ic[icIndex].type == dctPUSHBUTTON )
         {
            if ( Info.viaHotkey )
               Info.HotData2Primary () ;
            else
               icIndex = dp->EditPushbutton ( Info ) ;

            if ( Info.dataMod != false || Info.viaHotkey != false )
            {
               if ( icIndex == tcDonePB )
                  done = true ;
               else if ( icIndex == tcTabE )
               {
                  while ( (icIndex = dp->PrevControl ()) != tcDonePB ) ;
                  icIndex = dp->PrevControl () ;
                  Info.viaHotkey = false ;
               }
               else
               {
                  //* Hide our dialog and call the *
                  //* dialog for the selected Tab. *
                  dp->HideWin () ;
                  short newIndex = tcDonePB ;
                  
                  if ( icIndex == tcTabS )
                  {
                     newIndex = tcSpanish ( &dInit, tabNames, tabfColor, tabnColor, 
                                            tabsColor, tcCONTROLS_DEFINED ) ;
                  }
                  else if ( icIndex == tcTabV )
                  {
                     newIndex = tcVietnamese ( &dInit, tabNames, tabfColor, tabnColor, 
                                               tabsColor, tcCONTROLS_DEFINED ) ;
                  }
                  else if ( icIndex == tcTabC )
                  {
                     newIndex = tcChinese ( &dInit, tabNames, tabfColor, tabnColor, 
                                            tabsColor, tcCONTROLS_DEFINED ) ;
                  }

                  //* Refresh our dialog window, and set focus on the target *
                  //* control. Set 'viaHotkey' and 'dataMod' to simulate     *
                  //* selection via Hotkey (for immediate action).           *
                  dp->RefreshWin () ;
                  while ( (icIndex = dp->NextControl ()) != newIndex ) ;
                  Info.viaHotkey = Info.dataMod = true ;
               }
            }
         }

         //* Move to next/previous control *
         if ( done == false && !Info.viaHotkey )
         {
            if ( Info.keyIn == nckSTAB )
               icIndex = dp->PrevControl () ; 
            else
               icIndex = dp->NextControl () ;
         }
      }  // while()
   }
   if ( (dp->meMouseEnabled ()) )            // disable mouse interface
      dp->meDisableMouse () ;
   if ( dp != NULL )                         // close the window
      delete ( dp ) ;
}  //* End Tabby_Cat() *

//******************************************************************************
//* Initial display modification for caller's dialog.                          *
//*                                                                            *
//******************************************************************************
static void tcCatBox ( NcDialog* dp, const char* title, const char* tName, 
                       attr_t fColor, attr_t sColor, attr_t bColor, 
                       short dRows, short dCols, short tabI )
{
   //* Pretend that the dialog starts one line lower than *
   //* it actually does. The tabs live in the top line.   *
   dp->ClearLine ( ZERO, nc.bw, true ) ;
   dp->DrawBox ( 1, ZERO, (dRows - 1), dCols, bColor,
                 title, ncltDUAL ) ;

   //* Visually indicate which dialog is active *
   dp->SetPushbuttonText ( tabI, tName, fColor, sColor ) ;

}  //* End tcCatBox() *

//******************************************************************************
//* Tabby_Cat ( Spanish )                                                      *
//* Returns: member of enum tcControls, indicating which control was selected  *
//* to cause return to caller.                                                 *
//******************************************************************************
static short tcSpanish ( const InitNcDialog* idptr, 
                         const char** tabNames, 
                         const attr_t* tabfColor, 
                         const attr_t* tabnColor,
                         const attr_t tabsColor,
                         short ctrlCount )
{
   const char* dialogTitle = "  Traducción Idioma Español  " ;
   const char* Chuck_Spanish = 
               "    Aunque su nombre machismo implícito, Carlos fue rara vez le\n"
               "preguntó a jugar con los otros niños. En primer lugar, él era\n"
               "bastante pequeña para su edad y su fortaleza en la parte superior\n"
               "del cuerpo era el de un niño con 5 años, a pesar de que era casi\n"
               "nueve años. Los continuos ataques de asma, y su trastorno\n"
               "obsesivo-compulsivo incipiente contribuyó así.\n"
               "    Sí, Carlos era un nerd, pero tenía un deseo ardiente de\n"
               "expresarse; además, una abrumadora necesidad de crear algo que\n"
               "alguien reconocería como valioso. Era probable que sus compañeros\n"
               "no quisieron dejar de robar su, almuerzo, y estaba seguro de que\n"
               "su familia seguirá con incomprensión obtuso.\n" ;

   const short dialogROWS = idptr->dLines,
               dialogCOLS = idptr->dColumns ;
        attr_t dColor = idptr->interiorColor,
               bColor = idptr->borderColor ;
   winPos wp( dialogROWS - 7, dialogCOLS / 2 - 11 ) ;
   enum txControls : short
   { txDonePB = tcDonePB, tcx1RB, tcx2RB, tcx3RB, tcx4RB, txTabE, txTabS, txTabV, txTabC, txCONTROLS } ;

   //* Initial parameters for dialog window are the same as caller *
   InitNcDialog dInit = *idptr ;

   //* Create a local copy of the control definitions and *
   //* integrate our controls into the basic set.         *
   InitCtrl ic[txCONTROLS] =
   {
      idptr->ctrlPtr[tcDonePB],
      {  // 'Evaluation Radiobutton #1' - - - index == tcx1RB
      dctRADIOBUTTON,               // type:      
      rbtS3a,                       // rbSubtype: standard, 3 chars wide
      true,                         // rbSelect:  default selection
      short(wp.ypos + 1),           // ulY:       upper left corner in Y
      short(wp.xpos + 1),           // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a)
      nc.brR,                       // nColor:    non-focus color
      nc.br,                        // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "¿Nació en Cuernavaca?",      // label: "Were you born in Cuernavaca?"
      ZERO,                         // labY:      
      4,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[tcx2RB]                   // nextCtrl:  link in next structure
      },
      {  // 'Evaluation Radiobutton #2' - - - index == tcx2RB
      dctRADIOBUTTON,               // type:      
      rbtS3a,                       // rbSubtype: standard, 3 chars wide
      false,                        // rbSelect:  default selection
      short(ic[tcx1RB].ulY + 1),    // ulY:       upper left corner in Y
      ic[tcx1RB].ulX,               // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a)
      nc.brR,                       // nColor:    non-focus color
      nc.br,                        // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "¿Has estudiado en la UPN?",  // label: "Have you studied at UPN?"
      ZERO,                         // labY:      
      4,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[tcx3RB]                   // nextCtrl:  link in next structure
      },
      {  // 'Evaluation Radiobutton #3' - - - index == tcx3RB
      dctRADIOBUTTON,               // type:      
      rbtS3a,                       // rbSubtype: standard, 3 chars wide
      false,                        // rbSelect:  default selection
      short(ic[tcx2RB].ulY + 1),    // ulY:       upper left corner in Y
      ic[tcx2RB].ulX,               // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a)
      nc.brR,                       // nColor:    non-focus color
      nc.br,                        // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "Creo que su español es un poco torpe.", // label: "I think you Spanish is a bit awkward."
      ZERO,                         // labY:      
      4,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[tcx4RB]                   // nextCtrl:  link in next structure
      },
      {  // 'Evaluation Radiobutton #4' - - - index == tcx4RB
      dctRADIOBUTTON,               // type:      
      rbtS3a,                       // rbSubtype: standard, 3 chars wide
      false,                        // rbSelect:  default selection
      short(ic[tcx3RB].ulY + 1),    // ulY:       upper left corner in Y
      ic[tcx3RB].ulX,               // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a)
      nc.brR,                       // nColor:    non-focus color
      nc.br,                        // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "Usted es sorprendentemente ignorante.", // label: "You are surprisingly ignorant."
      ZERO,                         // labY:      
      4,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      NULL                          // nextCtrl:  link in next structure
      },
      idptr->ctrlPtr[tcTabE],
      idptr->ctrlPtr[tcTabS],
      idptr->ctrlPtr[tcTabV],
      idptr->ctrlPtr[tcTabC],
   } ;
   dInit.ctrlPtr = ic ;
   ic[txDonePB].dispText = " Cerrar la Ventana " ;
   ic[txDonePB].cols     = 19 ;
   ic[txDonePB].ulX     -= 6 ;
   ic[txDonePB].nextCtrl = &ic[tcx1RB] ;
   ic[tcx4RB].nextCtrl   = &ic[txTabE] ;
   ic[txTabE].nextCtrl   = &ic[txTabS] ;
   ic[txTabS].nextCtrl   = &ic[txTabV] ;
   ic[txTabV].nextCtrl   = &ic[txTabC] ;
   ic[txTabC].nextCtrl   = NULL ;

   //* Instantiate the dialog window *
   NcDialog* dp = new NcDialog ( dInit ) ;

   //* Open the dialog window *
   short icIndex = ZERO ;  // control index (and return value)
   if ( (dp->OpenWindow()) == OK )
   {
      //* Draw initial static text *
      tcCatBox ( dp, dialogTitle, tabNames[tcTabS], 
                 tabfColor[tcTabS], tabsColor, bColor, 
                 dialogROWS, dialogCOLS, txTabS ) ;

      dp->DrawBox ( wp.ypos, wp.xpos, 6, dialogCOLS / 2 + 10, dColor, 
                    " Su evaluación de esta traducción ", ncltDASH3 ) ;

      //* Create an XOR Radiobutton group *
      short XorGroup1[] = { tcx1RB, tcx2RB, tcx3RB, tcx4RB, -1 } ;
      dp->GroupRadiobuttons ( XorGroup1 ) ;

      //* Tell our story *
      dp->WriteParagraph ( 2, 2, Chuck_Spanish, dColor ) ;

      dp->RefreshWin () ;                 // make everything visible

      //* Track user's selections *
      uiInfo   Info ;                     // user interface data returned here
      bool     done = false ;             // loop control

      while ( ! done )
      {
         if ( ic[icIndex].type == dctPUSHBUTTON )
         {
            if ( Info.viaHotkey )
               Info.HotData2Primary () ;
            else
               icIndex = dp->EditPushbutton ( Info ) ;
            if ( Info.dataMod != false )
            {
               if ( icIndex == txDonePB || icIndex == txTabE ||
                    icIndex == txTabV || icIndex == txTabC )
               {
                  //* Adjust the index to reference CALLER'S array *
                  if ( icIndex != txDonePB )
                     icIndex -= 4 ;
                  done = true ;
               }
               else
               {  //* Current language selected again, so do nothing.*
                  while ( (icIndex = dp->PrevControl ()) != txDonePB ) ;
                  icIndex = dp->PrevControl () ;
               }
            }
         }
         else if ( ic[icIndex].type == dctRADIOBUTTON )
         {
            icIndex = dp->EditRadiobutton ( Info ) ;
            if ( Info.dataMod != false )
            {
               while ( (icIndex = dp->PrevControl ()) != txDonePB ) ;
               icIndex = dp->PrevControl () ;
            }
         }

         //* Move to next/previous control *
         if ( done == false && !Info.viaHotkey )
         {
            if ( Info.keyIn == nckSTAB )
               icIndex = dp->PrevControl () ; 
            else
               icIndex = dp->NextControl () ;
         }
      }  // while()
   }
   if ( dp != NULL )                         // close the window
      delete ( dp ) ;

   return icIndex ;

}  //* End tcSpanish() *

//******************************************************************************
//* Tabby_Cat ( Vietnamese )                                                   *
//* Returns: member of enum tcControls, indicating which control was selected  *
//* to cause return to caller.                                                 *
//******************************************************************************
static short tcVietnamese ( const InitNcDialog* idptr, 
                            const char** tabNames, 
                            const attr_t* tabfColor, 
                            const attr_t* tabnColor,
                            const attr_t tabsColor,
                            short ctrlCount )
{
   const char* dialogTitle = "    Dịch Tiếng Việt    " ;
   const char* Chuck_Viet =
               //===============================================================z
               "    Mặc dù cái tên nam tính của mình, Chiến đã không được mời\n"
               "tham gia các trò chơi của các chàng trai khác. Đầu tiên, ông khá\n"
               "nhỏ so với tuổi của mình, và sức mạnh thân của mình cũng tương tự\n"
               "như một đứa trẻ 5 tuổi, mặc dù tuổi của ông đã gần chín năm.\n"
               "    Có, Chiến là mọt sách, nhưng ông đã có một mong muốn cháy\n"
               "để bày tỏ cảm xúc của mình, cũng như một sự ép buộc áp đảo để tạo\n"
               "ra một cái gì đó giá trị lâu dài mà sẽ được đánh giá bởi ai đó.\n"
               "Điều này đúng ngay cả khi ông tin rằng các bạn cùng lớp của ông\n"
               "sẽ tiếp tục ăn cắp ăn trưa của mình, và anh biết chắc chắn rằng\n"
               "các thành viên trong gia đình của ông sẽ tiếp tục nhìn chằm chằm\n"
               "vào anh, khuôn mặt của họ chứa đầy vô minh không hiểu." ;

   const short dialogROWS = idptr->dLines,
               dialogCOLS = idptr->dColumns ;
        attr_t dColor = idptr->interiorColor,
               bColor = idptr->borderColor ;
   attr_t monoColor[2] = { attrDFLT, nc.brR } ; // scroll box data colors
   const short sDATA_ITEMS = 5,                 // scroll box data items
               sDATA_WIDTH = 40 ;               // scrollbox width
   const char  sboxData[sDATA_ITEMS][sDATA_WIDTH * 2] =
   {
      "Đây là ngữ pháp hoàn hảo!             ", // This is perfect grammar!
      "Bạn đã làm tốt, cho người nước ngoài. ", // You have done well, for a foreigner.
      "Bạn nói như một đứa trẻ nhỏ.          ", // You speak like a small child.
      "Tôi hiểu bạn, nhưng với khó khăn.     ", // I understand you, but with difficulty.
      "Tôi xin lỗi, tôi không thể ngừng cười.", // I'm sorry, I can't stop laughing.
   } ;
   winPos wp( dialogROWS - 7, dialogCOLS / 2 - 9 ) ;
   enum txControls : short
   { txDonePB = tcDonePB, tcxDD, txTabE, txTabS, txTabV, txTabC, txCONTROLS } ;

   //* Initial parameters for dialog window are the same as caller *
   InitNcDialog dInit = *idptr ;

   //* Create a local copy of the control definitions and *
   //* integrate our controls into the basic set.         *
   InitCtrl ic[txCONTROLS] =
   {
      idptr->ctrlPtr[tcDonePB],
      {  // 'Evaluation Dropdown' - - - index == tcxDD
      dctDROPDOWN,                  // type:      define a scrolling-data control
      rbtTYPES,                     // rbSubtype: (na)
      false,                        // rbSelect:  (n/a)
      short(wp.ypos + 1),           // ulY:       upper left corner in Y
      short(wp.xpos + 1),           // ulX:       upper left corner in X
      4,                            // lines:     (n/a)
      sDATA_WIDTH,                  // cols:      control columns
      (char*)&sboxData,             // dispText:  n/a
      nc.brR,                       // nColor:    non-focus border color
      nc.grbr,                      // fColor:    focus border color
      tbPrint,                      // filter:    (n/a)
      "chọn một mục",               // label:     
      3,                            // labY:      offset from control's ulY
      ZERO,                         // labX       offset from control's ulX
      ddBoxDOWN,                    // exType:    expand downward
      sDATA_ITEMS,                  // scrItems:  number of elements in text/color arrays
      ZERO,                         // scrSel:    index of initial highlighted element
      monoColor,                    // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      NULL                          // nextCtrl:  link in next structure
      },
      idptr->ctrlPtr[tcTabE],
      idptr->ctrlPtr[tcTabS],
      idptr->ctrlPtr[tcTabV],
      idptr->ctrlPtr[tcTabC],
   } ;
   dInit.ctrlPtr = ic ;
   ic[txDonePB].dispText = " Đóng cửa sổ " ;
   ic[txDonePB].cols     = 13 ;
   ic[txDonePB].ulX     -= 2 ;
   ic[txDonePB].nextCtrl = &ic[tcxDD] ;
   ic[tcxDD].nextCtrl    = &ic[txTabE] ;
   ic[txTabE].nextCtrl   = &ic[txTabS] ;
   ic[txTabS].nextCtrl   = &ic[txTabV] ;
   ic[txTabV].nextCtrl   = &ic[txTabC] ;
   ic[txTabC].nextCtrl   = NULL ;

   //* Instantiate the dialog window *
   NcDialog* dp = new NcDialog ( dInit ) ;

   //* Open the dialog window *
   short icIndex = ZERO ;  // control index (and return value)
   if ( (dp->OpenWindow()) == OK )
   {
      //* Draw initial static text *
      tcCatBox ( dp, dialogTitle, tabNames[tcTabV], 
                 tabfColor[tcTabV], tabsColor, bColor, 
                 dialogROWS, dialogCOLS, txTabV ) ;
      dp->DrawBox ( wp.ypos, wp.xpos, 6, dialogCOLS / 2 + 8, dColor,
                    " Đánh giá của bạn dịch này ", ncltDASH3 ) ;

      //* Tell our story *
      dp->WriteParagraph ( 2, 2, Chuck_Viet, dColor ) ;

      dp->RefreshWin () ;                 // make everything visible

      //* Track user's selections *
      uiInfo   Info ;                     // user interface data returned here
      bool     done = false ;             // loop control

      while ( ! done )
      {
         if ( ic[icIndex].type == dctPUSHBUTTON )
         {
            if ( Info.viaHotkey )
               Info.HotData2Primary () ;
            else
               icIndex = dp->EditPushbutton ( Info ) ;
            if ( Info.dataMod != false )
            {
               if ( icIndex == txDonePB || icIndex == txTabE ||
                    icIndex == txTabS || icIndex == txTabC )
               {
                  //* Adjust the index to reference CALLER'S array *
                  if ( icIndex != txDonePB )
                     --icIndex ;
                  done = true ;
               }
               else
               {  //* Current language selected again, so do nothing.*
                  while ( (icIndex = dp->PrevControl ()) != txDonePB ) ;
                  icIndex = dp->PrevControl () ;
               }
            }
         }
         else if ( ic[icIndex].type == dctDROPDOWN )
         {
            Info.viaHotkey = true ; // force the control to expand immediately
            icIndex = dp->EditDropdown ( Info ) ;
            if ( Info.dataMod != false )
            {
               while ( (icIndex = dp->PrevControl ()) != txDonePB ) ;
               icIndex = dp->PrevControl () ;
            }
         }

         //* Move to next/previous control *
         if ( done == false && !Info.viaHotkey )
         {
            if ( Info.keyIn == nckSTAB )
               icIndex = dp->PrevControl () ; 
            else
               icIndex = dp->NextControl () ;
         }
      }  // while()
   }
   if ( dp != NULL )                         // close the window
      delete ( dp ) ;

   return icIndex ;

}  //* End tcVietnamese() *

//******************************************************************************
//* Tabby_Cat ( Chinese )                                                      *
//* Returns: member of enum tcControls, indicating which control was selected  *
//* to cause return to caller.                                                 *
//******************************************************************************
static short tcChinese ( const InitNcDialog* idptr, 
                         const char** tabNames, 
                         const attr_t* tabfColor, 
                         const attr_t* tabnColor,
                         const attr_t tabsColor,
                         short ctrlCount )
{
// Changrui : 长锐     given name (mountain near Qing tombs)
// See Texinfo docs for details on this translation.
   const char* dialogTitle = "    中文翻译    " ;
   const char* Chuck_China =
               "尽管长锐有一个很男人化的名字，其他男孩们却很少与他玩游戏。\n"
               "首先，是因为他的身躯相当矮小，且上半体的力气只有5岁孩子般那样小，\n"
               "虽然他已近9岁了。其次，频繁的哮喘病的发作和青春期前的强迫症\n"
               "也促成了他这种身材。\n"
               "\n"
               "没错，长锐是个书呆子，但是他有着强烈的愿望想表达自己，\n"
               "以及有一种势不可挡地做出某种创造性工作的需求，\n"
               "该创造性工作对某些人或许有永久的价值。虽然他的同伴可能\n"
               "继续偷窃他的午餐，并且除了这确信他的家人将以喋喋不休、\n"
               "愚昧无知的方式继续盯着着他的言行。" ;


   const short dialogROWS = idptr->dLines,
               dialogCOLS = idptr->dColumns ;
        attr_t dColor = idptr->interiorColor,
               bColor = idptr->borderColor ;
   winPos wp( dialogROWS - 7, dialogCOLS / 2 - 9 ) ;
   enum txControls : short
   { txDonePB = tcDonePB, tcxTB, txTabE, txTabS, txTabV, txTabC, txCONTROLS } ;

   //* Initial parameters for dialog window are the same as caller *
   InitNcDialog dInit = *idptr ;

   //* Create a local copy of the control definitions and *
   //* integrate our controls into the basic set.         *
   InitCtrl ic[txCONTROLS] =
   {
      idptr->ctrlPtr[tcDonePB],
      {  // 'Evaluation Textbox' - - - index == tcxTB
      dctTEXTBOX,                   // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(wp.ypos + 1),           // ulY:       upper left corner in Y
      short(wp.xpos + 1),           // ulX:       upper left corner in X
      4,                            // lines:     (n/a)
      40,                           // cols:      control columns
      NULL,                         // dispText:  (initially blank)
      nc.brR,                       // nColor:    non-focus color
      nc.br,                        // fColor:    focus color
      tbPrint,                      // filter:    any printing character
      NULL,                         // label:     
      ZERO,                         // labY:      
      ZERO,                         // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    cannot gain focus
      NULL                          // nextCtrl:  link in next structure
      },
      idptr->ctrlPtr[tcTabE],
      idptr->ctrlPtr[tcTabS],
      idptr->ctrlPtr[tcTabV],
      idptr->ctrlPtr[tcTabC],
   } ;
   dInit.ctrlPtr = ic ;
   ic[txDonePB].dispText = " 关闭窗口 " ;
   ic[txDonePB].cols     = 10 ;
   ic[txDonePB].nextCtrl = &ic[tcxTB] ;
   ic[tcxTB].nextCtrl    = &ic[txTabE] ;
   ic[txTabE].nextCtrl   = &ic[txTabS] ;
   ic[txTabS].nextCtrl   = &ic[txTabV] ;
   ic[txTabV].nextCtrl   = &ic[txTabC] ;
   ic[txTabC].nextCtrl   = NULL ;

   //* Instantiate the dialog window *
   NcDialog* dp = new NcDialog ( dInit ) ;

   //* Open the dialog window *
   short icIndex = ZERO ;  // control index (and return value)
   if ( (dp->OpenWindow()) == OK )
   {
      //* Draw initial static text *
      tcCatBox ( dp, dialogTitle, tabNames[tcTabC], 
                 tabfColor[tcTabC], tabsColor, bColor, 
                 dialogROWS, dialogCOLS, txTabC ) ;

      dp->DrawBox ( wp.ypos, wp.xpos, 6, dialogCOLS / 2 + 8, dColor,
                    " 您这个翻译评估 ", ncltDASH3 ) ;

      //* Be too cute for our own good :-) *
      dtbmData dt( "你好，北京！\n请在这里写下你的评价。" ) ;
      dp->DisplayTextboxMessage ( tcxTB, dt ) ;

      //* Tell our story *
      dp->WriteParagraph ( 2, 2, Chuck_China, dColor ) ;

      dp->RefreshWin () ;                 // make everything visible

      //* Track user's selections *
      uiInfo   Info ;                     // user interface data returned here
      bool     done = false ;             // loop control

      while ( ! done )
      {
         if ( ic[icIndex].type == dctPUSHBUTTON )
         {
            if ( Info.viaHotkey )
               Info.HotData2Primary () ;
            else
               icIndex = dp->EditPushbutton ( Info ) ;
            if ( Info.dataMod != false )
            {
               if ( icIndex == txDonePB || icIndex == txTabE ||
                    icIndex == txTabS || icIndex == txTabV )
               {
                  //* Adjust the index to reference CALLER'S array *
                  if ( icIndex != txDonePB )
                     --icIndex ;
                  done = true ;

                  #if 0    // Testing only - capture on exit
                  dp->CaptureDialog ( "capturedlg.txt" ) ;
                  dp->CaptureDialog ( "capturedlg.htm", true, false, 
                                      "infodoc-styles.css", 4, false, dColor ) ;
                  #endif   // capture on exit
               }
               else
               {  //* Current language selected again, so do nothing.*
                  while ( (icIndex = dp->PrevControl ()) != txDonePB ) ;
                  icIndex = dp->PrevControl () ;
               }
            }
         }
         else if ( ic[icIndex].type == dctTEXTBOX )
         {
            icIndex = dp->EditTextbox ( Info ) ;
            if ( Info.dataMod != false )
            {
               while ( (icIndex = dp->PrevControl ()) != txDonePB ) ;
               icIndex = dp->PrevControl () ;
            }
         }

         //* Move to next/previous control *
         if ( done == false && !Info.viaHotkey )
         {
            if ( Info.keyIn == nckSTAB )
               icIndex = dp->PrevControl () ; 
            else
               icIndex = dp->NextControl () ;
         }
      }  // while()
   }
   if ( dp != NULL )                         // close the window
      delete ( dp ) ;

   return icIndex ;

}  //* End tcChinese() *

//*************************
//*    Billboard_Demo     *
//*************************
//******************************************************************************
//* Show the basic functionality of a Billboard control.                       *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

static void Billboard_Demo ( void )
{
   const short  dialogROWS = 10,       // display lines
                dialogCOLS  = 29,      // display columns
                ulY = 3,               // upper left corner in Y
                ulX = 21,              // upper left corner in X
                stepCount = 11 ;       // demonstration steps
   const attr_t dColor = nc.blR,       // dialog background color
                bColor = nc.brbl ;     // dialog border color
   attr_t lineColors[] = 
   { nc.recy, nc.blcy, nc.brcy, nc.bkcy, nc.bw } ;
   winPos stepPos( 6, 2 ) ;
   const wchar_t* stepMsg[stepCount] = 
   {
      L"Empty four-line Billboard",
      L"Set Billboard Contents",
      L"Append a Line (1)",
      L"Append a Line (2)",
      L"Append a Line (3)",
      L"Append a Line (4)",
      L"Append a Line (5)",
      L"Insert Line at Top",
      L"Replace Third Line",
      L"Extend Third Line",
      L"Clear Billboard Contents"
   } ;
   const wchar_t* Poem_1 = L"Roses are red,\n"
                            "Violets are blue,\n"
                            "That's why I picked some\n"
                            "just for you!" ;
   const wchar_t* Poem_2[] = 
   {
      L" --  --  --  --  --  --  --",
      L"Roses are pink",
      L"Violets are purple",
      L"Sugar is sweet",
      L"And so's maple syrple."
   } ;

   enum bdControls : short { bdNextPB = ZERO, bdDonePB, bdPoemBB, bdControlsDEFINED } ;


   InitCtrl ic[bdControlsDEFINED] = // array of dialog control initialization objects
   {
   {  //* 'NEXT' pushbutton - - - - - - - - - - - - - - - - - - - - - bdNextPB *
      dctPUSHBUTTON,    // type:      
      rbtTYPES,         // rbSubtype: (n/a)
      false,            // rbSelect:  (n/a)
      short(dialogROWS - 2),     // ulY: upper left corner in Y
      5,                // ulX: upper left corner in X
      1,                // lines:     control lines
      8,                // cols:      control columns
      "  NEXT  ",       // dispText:  
      nc.gyR,           // nColor:    non-focus color
      nc.reG,           // fColor:    focus color
      tbPrint,          // filter:    (n/a)
      NULL,             // label:     (n/a)
      ZERO,             // labY:      (n/a)
      ZERO,             // labX       (n/a)
      ddBoxTYPES,       // exType:    (n/a)
      1,                // scrItems:  (n/a)
      1,                // scrSel:    (n/a)
      NULL,             // scrColor:  (n/a)
      NULL,             // spinData:  (n/a)
      true,             // active:    allow control to gain focus
      &ic[bdDonePB]     // nextCtrl:  link in next structure
   },
   {  //* 'DONE' pushbutton - - - - - - - - - - - - - - - - - - - - - bdDonePB *
      dctPUSHBUTTON,    // type:      
      rbtTYPES,         // rbSubtype: (n/a)
      false,            // rbSelect:  (n/a)
      ic[bdNextPB].ulY, // ulY: upper left corner in Y
      short(ic[bdNextPB].ulX + ic[bdNextPB].cols + 3), // ulX: upper left corner in X
      1,                // lines:     control lines
      8,                // cols:      control columns
      "  DONE  ",       // dispText:  
      nc.gyR,           // nColor:    non-focus color
      nc.reG,           // fColor:    focus color
      tbPrint,          // filter:    (n/a)
      NULL,             // label:     (n/a)
      ZERO,             // labY:      (n/a)
      ZERO,             // labX       (n/a)
      ddBoxTYPES,       // exType:    (n/a)
      1,                // scrItems:  (n/a)
      1,                // scrSel:    (n/a)
      NULL,             // scrColor:  (n/a)
      NULL,             // spinData:  (n/a)
      true,             // active:    allow control to gain focus
      &ic[bdPoemBB]     // nextCtrl:  link in next structure
   },
   {  //* 'Billboard'  - - - - - - - - - - - - - - - - - - - - - -    bdPoemBB *
      dctBILLBOARD,                 // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      1,                            // ulY:       upper left corner in Y
      1,                            // ulX:       upper left corner in X
      4,                            // lines:     control lines
      short(dialogCOLS - 2),        // cols:      control columns
      NULL,                         // dispText:  initial display text
      nc.cyR,                       // nColor:    non-focus color
      nc.reR,                       // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      NULL,                         // label:     control label
      -1,                           // labY:      label offset Y
      ZERO,                         // labX       label offset X
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      lineColors,                   // scrColor:  if specified, attribute array
      NULL,                         // spinData:  (n/a)
      false,                        // active:    view-only
      NULL                          // nextCtrl:  link in next structure
   },
   } ;

   //* Initial parameters for dialog window *
   InitNcDialog dInit( dialogROWS,     // number of display lines
                       dialogCOLS,     // number of display columns
                       ulY,            // Y offset from upper-left of terminal 
                       ulX,            // X offset from upper-left of terminal 
                       "  Poetry Corner  ", // title
                       ncltDUAL,       // border line-style
                       bColor,         // border color attribute
                       dColor,         // interior color attribute
                       ic              // list of controls
                     ) ;

   //* Instantiate the dialog window *
   NcDialog* dp = new NcDialog ( dInit ) ;

   //* Open the dialog window *
   if ( (dp->OpenWindow()) == OK )
   {
      dp->WriteString ( 5, 13, "---", dColor ) ;
      short stepIndex = ZERO ;
      dp->WriteString ( stepPos, stepMsg[stepIndex], dColor ) ;
      dp->RefreshWin () ;                 // make everything visible

      //* Track user's selections *
      uiInfo   Info ;                     // user interface data returned here
      short    icIndex = ZERO ;           // index of control with input focus
      bool     done = false ;             // loop control

      while ( ! done )
      {
         if ( ic[icIndex].type == dctPUSHBUTTON )
         {
            icIndex = dp->EditPushbutton ( Info ) ;
            if ( Info.dataMod != false )
            {
               if ( Info.ctrlIndex == bdDonePB )
                  done = true ;
               else if ( Info.ctrlIndex == bdNextPB )
               {  //* Indicate the current step *
                  dp->ClearLine ( stepPos.ypos ) ;
                  dp->WriteString ( stepPos, stepMsg[++stepIndex], dColor, true ) ;

                  switch ( stepIndex )
                  {
                     case 1:
                        dp->SetBillboard ( bdPoemBB, Poem_1, lineColors ) ;
                        break ;
                     case 2:
                        dp->Append2Billboard ( bdPoemBB, Poem_2[0] ) ;
                        break ;
                     case 3:
                        dp->Append2Billboard ( bdPoemBB, Poem_2[1] ) ;
                        break ;
                     case 4:
                        dp->Append2Billboard ( bdPoemBB, Poem_2[2] ) ;
                        break ;
                     case 5:
                        dp->Append2Billboard ( bdPoemBB, Poem_2[3] ) ;
                        break ;
                     case 6:
                        dp->Append2Billboard ( bdPoemBB, Poem_2[4] ) ;
                        break ;
                     case 7:
                        dp->Insert2Billboard ( bdPoemBB,
                                    L"       !STUPID POEM!       ", nc.reR ) ;
                        break ;
                     case 8:
                        dp->Append2Billboard ( bdPoemBB, 
                                    L"Forget the flowers,", 
                                    nc.grbr, 2 ) ;
                        break ;
                     case 9:
                        dp->Append2Billboard ( bdPoemBB, 
                                    L" give jewelry! ", 
                                    nc.grbr, 2, true ) ;
                        break ;
                     default:
                        stepIndex = ZERO ;   // restart the sequence
                        dp->ClearBillboard ( bdPoemBB ) ;
                        break ;
                  } ;

                  #if 0    // For screenshots only
                  dp->CaptureDialog ( "capturedlg.txt" ) ;
                  dp->CaptureDialog ( "capturedlg.html", true, false, 
                                      "infodoc-styles.css", 4, false, nc.blR ) ;
                  #endif   // For screenshots only
               }
            }
         }

         //* Move to next/previous control *
         if ( done == false && !Info.viaHotkey )
         {
            if ( Info.keyIn == nckSTAB )
               icIndex = dp->PrevControl () ; 
            else if ( Info.keyIn != ZERO )
               icIndex = dp->NextControl () ;
         }
      }  // while()
   }
   if ( dp != NULL )                         // close the window
      delete ( dp ) ;

}  //* End Billboard_Demo() *

//*************************
//*     Dropdown_Demo     *
//*************************
//******************************************************************************
//*                                                                            *
//*                                                                            *
//* Input  :                                                                   *
//*                                                                            *
//* Returns:                                                                   *
//******************************************************************************

static void Dropdown_Demo ( void )
{
   const short  dialogROWS = 16,       // display lines
                dialogCOLS = 84,       // display columns
                ulY = 3,               // upper left corner in Y
                ulX = 1,               // upper left corner in X
                divTEAMS   = 8,        // item count
                ddWIDTH    = 24 ;      // control columns
   const attr_t dColor = nc.blR,       // dialog background color
                bColor = nc.brbl ;     // dialog border color
   attr_t monoBrown[2] = { attrDFLT, nc.br } ; // line item colors
   attr_t monoBlue[2]  = { attrDFLT, nc.bl } ;
   attr_t monoCyan[2]  = { attrDFLT, nc.bkcy } ;

   //* Control indices *
   enum ddControls : short { ddDonePB = ZERO, ddWestDD, ddCentDD, ddEastDD,
                             ddControlsDEFINED } ;

   const char divWest[divTEAMS][ddWIDTH - 1] = // Western Division
   {
      " Arizona Cardinals    ",
      " Denver Broncos       ",
      " Kansas City Chiefs   ",
      " Oakland Raiders      ",
      " St. Louis Rams       ",
      " San Diego Chargers   ",
      " San Francisco 49ers  ",
      " Seattle Seahawks     ",
   } ;
   const char divCent[divTEAMS * 2][ddWIDTH +3] = // North and South Divisions
   {
      " Atlanta Falcons      (S) ",
      " Baltimore Ravens     (N) ",
      " Carolina Panthers    (S) ",
      " Chicago Bears        (N) ",
      " Cincinnati Bengals   (N) ",
      " Cleveland Browns     (N) ",
      " Detroit Lions        (N) ",
      " Green Bay Packers    (N) ",
      " Houston Texans       (S) ",
      " Indianapolis Colts   (S) ",
      " Jacksonville Jaguars (S) ",
      " Minnesota Vikings    (N) ",
      " New Orleans Saints   (S) ",
      " Pittsburgh Steelers  (N) ",
      " Tampa Bay Buccaneers (S) ",
      " Tennessee Titans     (S) ",
   } ;
   const char divEast[divTEAMS][ddWIDTH - 1] = // Eastern Division
   {
      " Buffalo Bills        ",
      " Dallas Cowboys       ",
      " Miami Dolphins       ",
      " New England Patriots ",
      " New York Giants      ",
      " New York Jets        ",
      " Philadelphia Eagles  ",
      " Washington Redskins  ",
   } ;

   InitCtrl ic[ddControlsDEFINED] = // array of dialog control initialization objects
   {
   {  //* 'DONE' pushbutton - - - - - - - - - - - - - - - - - - - - - ddDonePB *
      dctPUSHBUTTON,    // type:      
      rbtTYPES,         // rbSubtype: (n/a)
      false,            // rbSelect:  (n/a)
      dialogROWS - 2,   // ulY: upper left corner in Y
      short(dialogCOLS / 2 - 4), // ulX: upper left corner in X
      1,                // lines:     control lines
      8,                // cols:      control columns
      "  ^DONE  ",      // dispText:  
      nc.gyR,           // nColor:    non-focus color
      nc.maR,           // fColor:    focus color
      tbPrint,          // filter:    (n/a)
      NULL,             // label:     (n/a)
      ZERO,             // labY:      (n/a)
      ZERO,             // labX       (n/a)
      ddBoxTYPES,       // exType:    (n/a)
      1,                // scrItems:  (n/a)
      1,                // scrSel:    (n/a)
      NULL,             // scrColor:  (n/a)
      NULL,             // spinData:  (n/a)
      true,             // active:    allow control to gain focus
      &ic[ddWestDD]     // nextCtrl:  link in next structure
   },
   { //* Western Division Dropdown  - - - - - - - - - - - - - - - -   ddWestDD *
      dctDROPDOWN,                  // type:      define a drop-down control
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      4,                            // ulY:       upper left corner in Y
      2,                            // ulX:       upper left corner in X
      divTEAMS,                     // lines:     control lines (expanded)
      ddWIDTH,                      // cols:      control columns
      (const char*)&divWest,        // dispText:  text-data array
      nc.brR,                       // nColor:    non-focus border color
      nc.maR,                       // fColor:    focus border color
      tbPrint,                      // filter:    (n/a)
      "^Western Division\n(expands downward)",  // label: label text
      -2,                           // labY:      label offset
      2,                            // labX       
      ddBoxDOWN,                    // exType:    expansion type
      divTEAMS,                     // scrItems:  number of elements in text/color arrays
      6,                            // scrSel:    index of initial highlighted element
      monoBrown,                    // scrColor:  color-attribute list
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[ddCentDD]                 // nextCtrl:  link in next structure
   },
   { //* North and South Division Dropdown  - - - - - - - - - - - -   ddCentDD *
      dctDROPDOWN,                  // type:      define a drop-down control
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(ic[ddWestDD].ulY + 2),  // ulY:       upper left corner in Y
      short(ic[ddWestDD].ulX + ic[ddWestDD].cols + 2), // ulX: upper left corner in X
      divTEAMS,                     // lines:     control lines (expanded)
      ddWIDTH + 4,                  // cols:      control columns
      (const char*)&divCent,        // dispText:  text-data array
      nc.cyR,                       // nColor:    non-focus border color
      nc.maR,                       // fColor:    focus border color
      tbPrint,                      // filter:    (n/a)
      "^North and South Divisions\n(expands up and down)",  // label: label text
      -4,                           // labY:      label offset
      2,                            // labX
      ddBoxCENTER,                  // exType:    expansion type
      divTEAMS * 2,                 // scrItems:  number of elements in text/color arrays
      3,                            // scrSel:    index of initial highlighted element
      monoCyan,                     // scrColor:  color-attribute list
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[ddEastDD]                 // nextCtrl:  link in next structure
   },
   { //* Eastern Division Dropdown  - - - - - - - - - - - - - - - -   ddEaseDD *
      dctDROPDOWN,                  // type:      define a drop-down control
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(ic[ddCentDD].ulY + 2),  // ulY:       upper left corner in Y
      short(ic[ddCentDD].ulX + ic[ddCentDD].cols + 2), // ulX: upper left corner in X
      divTEAMS,                     // lines:     control lines (expanded)
      ddWIDTH,                      // cols:      control columns
      (const char*)&divEast,        // dispText:  text-data array
      nc.bl,                        // nColor:    non-focus border color
      nc.maR,                       // fColor:    focus border color
      tbPrint,                      // filter:    (n/a)
      "^Eastern Division\n(expands upward)",  // label: label text
      3,                            // labY:      label offset
      2,                            // labX       
      ddBoxUP,                      // exType:    expansion type
      divTEAMS,                     // scrItems:  number of elements in text/color arrays
      2,                            // scrSel:    index of initial highlighted element
      monoBlue,                     // scrColor:  color-attribute list
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      NULL                          // nextCtrl:  link in next structure
   },
   } ;

   //* Initial parameters for dialog window *
   InitNcDialog dInit( dialogROWS,     // number of display lines
                       dialogCOLS,     // number of display columns
                       ulY,            // Y offset from upper-left of terminal 
                       ulX,            // X offset from upper-left of terminal 
                       "  National Football League  ", // dialog title
                       ncltSINGLE,     // border line-style
                       bColor,         // border color attribute
                       dColor,         // interior color attribute
                       ic              // list of controls
                     ) ;

   //* Instantiate the dialog window *
   NcDialog* dp = new NcDialog ( dInit ) ;

   //* Open the dialog window *
   if ( (dp->OpenWindow()) == OK )
   {
      dp->RefreshWin () ;              // make everything visible

      //* Track user's selections *
      uiInfo   Info ;                  // user interface data returned here
      short    icIndex = ZERO ;        // index of control with input focus
      bool     done = false ;          // loop control

      while ( ! done )
      {
         if ( ic[icIndex].type == dctPUSHBUTTON )
         {
            if ( Info.viaHotkey != false )
               Info.HotData2Primary () ;
            else
               icIndex = dp->EditPushbutton ( Info ) ;

            if ( Info.dataMod != false )
               done = true ;
         }
         else if ( ic[icIndex].type == dctDROPDOWN )
         {
            if ( Info.viaHotkey != false )
            { /* do nothing */ }

            icIndex = dp->EditDropdown ( Info ) ;
         }

         //* Move to next/previous control *
         if ( done == false && !Info.viaHotkey )
         {
            if ( Info.keyIn == nckSTAB )
               icIndex = dp->PrevControl () ; 
            else if ( Info.keyIn != ZERO )
               icIndex = dp->NextControl () ;
         }
      }  // while()
   }
   if ( dp != NULL )                         // close the window
      delete ( dp ) ;

}  //* End Dropdown_Demo() *

//*************************
//*     Menuwin_Demo      *
//*************************
//******************************************************************************
//* Create a simple menu with hotkeys and a submenu.                           *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

static void Menuwin_Demo ( void )
{
   const short  dialogROWS = 20,       // display lines
                dialogCOLS = 51,       // display columns
                ulY = 3,               // upper left corner in Y
                ulX = 21,              // upper left corner in X
                parITEMS = 13,         // Parent menu items
                parWIDTH = 19,         // Parent text column count
                subITEMS = 3,          // Sub-menu items
                subWIDTH = 19 ;        // Sub-menu text column count
   const attr_t dColor = nc.blR,       // dialog background color
                bColor = nc.brbl ;     // dialog border color

   //* Control indices *
   enum mwControls : short { mwDonePB = ZERO, mwParMW, mwSubMW, mwControlsDEFINED } ;

   //* Text for parent menu (use care with column count) *
   const char ParText[parITEMS][parWIDTH + 2] = 
   {
      " Flavors:         ",
      " ^Chocolate         ",
      " ^Vanilla           ",
      " ^Strawberry       >",
      " ^Peach             ",
      " ^Rocky Road        ",
      " ---------------- ",
      " Toppings:        ",
      " Cashe^ws           ",
      " M & M Plai^n       ",
      " Marshmallow ^fluff ",
      " Sprin^kles         ",
      " ^Butterscotch      ",
   } ;

   //* Color attributes for parent menu *
   attr_t ParAttr[parITEMS] = 
   { nc.gyR, nc.bl, nc.bl, nc.bl, nc.bl, nc.bl, nc.gy, nc.gyR, 
     nc.bl, nc.bl, nc.bl, nc.bl, nc.bl  } ;

   //* Active/Inactive flags for parent menu *
   const bool ParActive[parITEMS] = 
   { false, true, true, true, true, true, false, 
     false, true, true, true, true, true } ;

   //* Text for sub-menu *
   const char SubText[subITEMS][subWIDTH + 1] = 
   {
      " with fruit        ",
      " with jam          ",
      " smooth (no fruit) ",
   } ;

   //* Color attributes for sub-menu (monochrome) *
   attr_t SubAttr[2] = { attrDFLT, nc.bl } ;


   InitCtrl ic[mwControlsDEFINED] = // array of dialog control initialization objects
   {
   {  //* 'DONE' pushbutton - - - - - - - - - - - - - - - - - - - - - mwDonePB *
      dctPUSHBUTTON,    // type:      
      rbtTYPES,         // rbSubtype: (n/a)
      false,            // rbSelect:  (n/a)
      dialogROWS - 2,   // ulY: upper left corner in Y
      short(dialogCOLS - 10), // ulX: upper left corner in X
      1,                // lines:     control lines
      8,                // cols:      control columns
      "  ^DONE  ",      // dispText:  
      nc.gyR,           // nColor:    non-focus color
      nc.maR,           // fColor:    focus color
      tbPrint,          // filter:    (n/a)
      NULL,             // label:     (n/a)
      ZERO,             // labY:      (n/a)
      ZERO,             // labX       (n/a)
      ddBoxTYPES,       // exType:    (n/a)
      1,                // scrItems:  (n/a)
      1,                // scrSel:    (n/a)
      NULL,             // scrColor:  (n/a)
      NULL,             // spinData:  (n/a)
      true,             // active:    allow control to gain focus
      &ic[mwParMW]      // nextCtrl:  link in next structure
   },
   {  //* 'Parent' Menuwin  - - - - - - - - - - - - - - - - - - - - -  mwParMW *
      dctMENUWIN,       // type:      define a Menuwin control
      rbtTYPES,         // rbSubtype: (n/a)
      false,            // rbSelect:  (n/a)
      1,                // ulY:       upper left corner in Y
      4,                // ulX:       upper left corner in X
      1,                // lines:     (n/a)
      (parWIDTH + 2),   // cols:      control columns
      (const char*)&ParText, // dispText:  text-data array
      nc.re,            // nColor:    non-focus border color
                        //            and focus title color
      nc.reR,           // fColor:    focus border color
                        //            and non-focus title color
      tbPrint,          // filter:    (n/a)
      "  Ice ^Cream  ", // label:     label text
      ZERO,             // labY:      (n/a)
      ZERO,             // labX       
      ddBoxTYPES,       // exType:    (n/a)
      parITEMS,         // scrItems:  elements in text/color arrays
      ZERO,             // scrSel:    (n/a)
      ParAttr,          // scrColor:  color-attribute list
      NULL,             // spinData:  (n/a)
      true,             // active:    allow control to gain focus
      &ic[mwSubMW]      // nextCtrl:  link in next structure
   },
   {  //* 'Sub-menu' Menuwin  - - - - - - - - - - - - - - - - - - - -  mwSubMW *
      dctMENUWIN,       // type:      define a Menuwin control
      rbtTYPES,         // rbSubtype: (n/a)
      false,            // rbSelect:  (n/a)
      short(ic[mwParMW].ulY + 4),            // ulY: upper left corner in Y
      short(ic[mwParMW].ulX + parWIDTH + 1), // ulX: upper left corner in X
      1,                // lines:     (n/a)
      (subWIDTH + 2),   // cols:      control columns
      (const char*)&SubText, // dispText:  text-data array
      nc.re,            // nColor:    non-focus border color
      nc.reR,           // fColor:    focus border color
      tbPrint,          // filter:    (n/a)
      NULL,             // label:     label text (no label for sub-menus)
      ZERO,             // labY:      (n/a)
      ZERO,             // labX       
      ddBoxTYPES,       // exType:    (n/a)
      subITEMS,         // scrItems:  elements in text/color arrays
      ZERO,             // scrSel:    (n/a)
      SubAttr,          // scrColor:  color-attribute list
      NULL,             // spinData:  (n/a)
      false,            // active:    sub-menu initially inactive
      NULL
   },
   } ;

   //* Initial parameters for dialog window *
   InitNcDialog dInit( dialogROWS,     // number of display lines
                       dialogCOLS,     // number of display columns
                       ulY,            // Y offset from upper-left of terminal 
                       ulX,            // X offset from upper-left of terminal 
                       NULL,           // no title
                       ncltSINGLE,     // border line-style
                       bColor,         // border color attribute
                       dColor,         // interior color attribute
                       ic              // list of controls
                     ) ;

   //* Instantiate the dialog window *
   NcDialog* dp = new NcDialog ( dInit ) ;

   //* Open the dialog window *
   if ( (dp->OpenWindow()) == OK )
   {
      //* Set the active items in parent menu item list.*
      dp->SetActiveMenuItems ( mwParMW, ParActive ) ;

      //* Attach the sub-menu to the parent menu at item index 3.*
      short smList[] = { MAX_DIALOG_CONTROLS, MAX_DIALOG_CONTROLS,
                         MAX_DIALOG_CONTROLS, mwSubMW, -1 } ;
      dp->AttachMenuwinSubmenus ( mwParMW, smList );

      dp->RefreshWin () ;              // make everything visible

      //* Track user's selections *
      uiInfo   Info ;                  // user interface data returned here
      short    icIndex = ZERO ;        // index of control with input focus
      bool     done = false ;          // loop control

      while ( ! done )
      {
         if ( ic[icIndex].type == dctPUSHBUTTON )
         {
            if ( Info.viaHotkey != false )
            {  // Selection of a Pushbutton via hotkey
               // means it was 'pressed'.
               Info.HotData2Primary () ;
            }
            else
               icIndex = dp->EditPushbutton ( Info ) ;

            if ( Info.dataMod != false )
               done = true ;
         }
         if ( ic[icIndex].type == dctMENUWIN )
         {
            if ( Info.viaHotkey != false )
            {
               // Nothing to be done here.
               // Menu will be expanded immediately
               // because 'viaHotkey' member is 'true'
            }

            icIndex = dp->EditMenuwin ( Info ) ;
         }

         //* Move to next/previous control *
         if ( done == false && !Info.viaHotkey )
         {
            if ( Info.keyIn == nckSTAB )
               icIndex = dp->PrevControl () ; 
            else if ( Info.keyIn != ZERO )
               icIndex = dp->NextControl () ;
         }
      }  // while()
   }
   if ( dp != NULL )                         // close the window
      delete ( dp ) ;

}  //* End Menuwin_Demo() *

//*************************
//*   Set_ClickInterval   *
//*************************
//******************************************************************************
//* Example of using a Spinner control to help user adjust the mouse-click     *
//* interval.                                                                  *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

static void TimeSpinner ( void )
{
   const short  dialogROWS = 10,       // display lines
                dialogCOLS  = 41,      // display columns
                ulY = 3,               // upper left corner in Y
                ulX = 21 ;             // upper left corner in X
   const attr_t dColor = nc.blR,       // dialog background color
                bColor = nc.brbl,      // dialog border color
                snColor = nc.bw,       // Spinner non-focus color
                sfColor = nc.bwR ;     // Spinner non-focus color

   dspinData hspData( 0, 23, 0, dspinINTEGER, dColor, true, L':' ) ;
   dspinData mspData( 0, 59, 0, dspinINTEGER, dColor, true, L':' ) ;
   dspinData sspData( 0, 59, 0, dspinINTEGER, dColor, true, nckNULLCHAR ) ;

   enum tsControls : short { tsPB = ZERO, tshSP, tsmSP, tssSP, tsControlsDEFINED } ;

   InitCtrl ic[tsControlsDEFINED] =   // control initialization structures
   {
   {  //* 'DONE' pushbutton  - - - - - - - - - - - - - - - - - - - - -    tsPB *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(dialogROWS - 3),         // ulY:       upper left corner in Y
      short(dialogCOLS / 2 - 4),    // ulX:       upper left corner in X
      1,                            // lines:     control lines
      8,                            // cols:      control columns
      "  DONE  ",                   // dispText:  
      nc.gyR,                       // nColor:    non-focus color
      nc.reG,                       // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "",                           // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[tshSP]                    // nextCtrl:  link in next structure
   },
   { //* 'Hour' Spinner    - - - - - - - - - - - - - - - - - - - - - -   tshSP *
      dctSPINNER,                   // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      4,                            // ulY:       upper left corner in Y
      16,                           // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      3,                            // cols:      control columns
      NULL,                         // dispText:  (n/a)
      snColor,                      // nColor:    non-focus color
      sfColor,                      // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "hh",                         // label:     
      1,                            // labY:      
      0,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      &hspData,                     // spinData:  spinner init
      true,                         // active:    allow control to gain focus
      &ic[tsmSP],                   // nextCtrl:  link in next structure
   },
   { //* 'Minute' Spinner  - - - - - - - - - - - - - - - - - - - - - -   tsmSP *
      dctSPINNER,                   // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      ic[tshSP].ulY,                // ulY:       upper left corner in Y
      short(ic[tshSP].ulX + 3),     // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      3,                            // cols:      control columns
      NULL,                         // dispText:  (n/a)
      snColor,                      // nColor:    non-focus color
      sfColor,                      // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "mm",                         // label:     
      1,                            // labY:      
      0,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      &mspData,                     // spinData:  spinner init
      true,                         // active:    allow control to gain focus
      &ic[tssSP],                   // nextCtrl:  link in next structure
   },
   { //* 'Second' Spinner  - - - - - - - - - - - - - - - - - - - - - -   tssSP *
      dctSPINNER,                   // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      ic[tsmSP].ulY,                // ulY:       upper left corner in Y
      short(ic[tsmSP].ulX + 3),     // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      2,                            // cols:      control columns
      NULL,                         // dispText:  (n/a)
      snColor,                      // nColor:    non-focus color
      sfColor,                      // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "ss",                         // label:     
      1,                            // labY:      
      0,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      &sspData,                     // spinData:  spinner init
      true,                         // active:    allow control to gain focus
      NULL                          // nextCtrl:  link in next structure
   },
   } ;


   //* Initial parameters for dialog window *
   InitNcDialog dInit( dialogROWS,     // number of display lines
                       dialogCOLS,     // number of display columns
                       ulY,            // Y offset from upper-left of terminal 
                       ulX,            // X offset from upper-left of terminal 
                       "  Time  Spinner  ", // title
                       ncltDUAL,       // border line-style
                       bColor,         // border color attribute
                       dColor,         // interior color attribute
                       ic              // list of controls
                     ) ;

   //* Instantiate the dialog window *
   NcDialog* dp = new NcDialog ( dInit ) ;

   //* Open the dialog window *
   if ( (dp->OpenWindow()) == OK )
   {
      //* Enable stable-mouse input *
      if ( (dp->meEnableStableMouse ()) == OK )
         dp->WriteString ( (dialogROWS - 2), 1, 
                           "(Mouse input is active in this dialog.)", dColor ) ;

      dp->WriteString ( 2, 9, "Enter the time (24-hour)", bColor ) ;

      dp->RefreshWin () ;

      //* Track user's selections *
      uiInfo   Info ;                     // user interface data returned here
      short    icIndex = ZERO ;           // index of control with input focus
      bool     done = false ;             // loop control

      while ( ! done )
      {
         if ( ic[icIndex].type == dctPUSHBUTTON )
         {
            if ( Info.viaHotkey != false )
               Info.HotData2Primary () ;
            else
               icIndex = dp->EditPushbutton ( Info ) ;
            if ( Info.dataMod != false )
            {
               /* Retrive spinner values to create the timestamp */
               
               done = true ;

               #if 0    // CAPTURE ONLY
               dp->CaptureDialog ( "capturedlg.txt" ) ;
               dp->CaptureDialog ( "capturedlg.html", true, false, 
                                   "infodoc-styles.css", 4, false, dColor ) ;
               #endif   // CAPTURE ONLY
            }
         }
         else if ( ic[icIndex].type == dctSPINNER )
         {
            icIndex = dp->EditSpinner ( Info ) ;
         }
         //* Move to next/previous control *
         if ( !done && !Info.viaHotkey )
         {
            if ( Info.keyIn == nckSTAB )
               icIndex = dp->PrevControl () ; 
            else if ( Info.keyIn != ZERO )
               icIndex = dp->NextControl () ;
         }
      }  // while()
   }
   if ( (dp->meMouseEnabled ()) )            // disable mouse interface
      dp->meDisableMouse () ;
   if ( dp != NULL )                         // close the window
      delete ( dp ) ;

}  //* End TimeSpinner() *

//*************************
//*                       *
//*************************
//******************************************************************************
//*                                                                            *
//*                                                                            *
//* Input  :                                                                   *
//*                                                                            *
//* Returns:                                                                   *
//******************************************************************************

