//********************************************************************************
//* File       : gStringTest.cpp                                                 *
//* Author     : Mahlon R. Smith                                                 *
//*              Copyright (c) 2013-2025 Mahlon R. Smith, The Software Samurai   *
//*                  GNU GPL copyright notice located in NcDialog.hpp            *
//* Date       : 24-Mar-2025                                                     *
//* Version    : (see below)                                                     *
//*                                                                              *
//* Description: gStringTest is a simple class definition for exercising the     *
//*              functionality of the gString class.                             *
//*                                                                              *
//* Developed using GNU G++ (Gcc v: 4.8.0)                                       *
//*  under Fedora Release 16, Kernel Linux 3.6.11-4.fc16.i686.PAE                *
//********************************************************************************
//* Version History (most recent first):                                         *
//*                                                                              *
//* v: 0.00.08 07-Feb-2025                                                       *           
//*   - The gString class now supports dyanamic allocation/re-allocation of      *
//*     buffer space. These changes have been re-integrated into the mainline    *
//*     code for gString v:0.0.37.                                               *
//*     Every attempt has been made to minimize impact on existing functionality;*
//*     however, a few significant changes are necessary at the application      *
//*     level:                                                                   *
//*     - Certain gString method parameters required promotion from 'short'      *
//*       to 'int32_t'.                                                          *
//*     - The gString 'gsForm&' constructor is obsolete. Replace gString         *
//*       constructors which take a 'gsForm&' argument with standard             *
//*       formatting-template constructors.                                      *
//*   - The dataset needed to validate and stress-test the dynamic buffer        *
//*     allocation/reallocation is huge, so those tests are not included in      *
//*     this module. Instead, a stand-alone test app, 'gsmeter' has been         *
//*     implemented to perform those tests. 'gsmeter' is distributed as part     *
//*     of the stand-alone gString package. Please see the gString documentation *
//*     for further details.                                                     *
//*                                                                              *
//* v: 0.00.08 27-Sep-2023                                                       *
//*   - Create additional tests for the new findr() method. This method          *
//*     mirrors the find() method by scanning toward the top of the data.        *
//*   - Create additional test for the new formatParagraph() method.             *
//*     This method formats the text into a rectangular block (paragraph) by     *
//*     performing automatic word wrap at the right edge of the specified area.  *
//*   - Create additional test for the erase() method to exercise the new        *
//*     optional parameter, 'all', which deletes all instances of the            *
//*     specified substring.                                                     *
//*                                                                              *
//* v: 0.00.07 02-May-2021                                                       *
//*   - Create additional tests for textReverse() method to exercise enhanced    *
//*     functionality.                                                           *
//*   - Create an additional test page to hold the new functionality.            *
//*                                                                              *
//* v: 0.00.06 11-Mar-2021                                                       *
//*   - Create an additional test for the 'padCols()' method of the gString      *
//*     class.                                                                   *
//*                                                                              *
//* v: 0.00.05 15-Dec-2020                                                       *
//*   - Add tests for 'textReverse()' method of gString class.                   *
//*                                                                              *
//* v: 0.00.04 29-May-2020                                                       *
//*   - Add tests for 'gscanf()' method.                                         *
//*   - Shift 'gtBasicPage7' to 'gtBasicPage8' and 'gtBasicPage8' to             *
//*     'gtBasicPage9'. This leaves 'gtBasicPage7' free for addition of new      *
//*     tests.                                                                   *
//*                                                                              *
//* v: 0.00.03 21-Jul-2017                                                       *
//*   - Add tests for 'strip()' method.                                          *
//*   - Add 'gtBasicPage8' in preparation for additional tests.                  *
//*                                                                              *
//* v: 0.00.02 18-Nov-2015                                                       *
//*   - Restructure tests for fixed-width integer formatting to support          *
//*     changes to the formatInt() method group.                                 *
//*                                                                              *
//* v: 0.00.01 01-Dec-2013                                                       *
//*   - Transfer functionality from Dialogw, Test06 and Test07.                  *
//*   - Add tests for 'insert()' method.                                         *
//*                                                                              *
//********************************************************************************

#include "gStringTest.hpp"

//****************
//** Local Data **
//****************

//* Control definitions for prompt dialog *
static InitCtrl promptIc[pCONTROLS_DEFINED] = 
{
   {  //* 'BASIC' pushbutton - - - - - - - - - - - - - - - - - - -    pBasicPB *
   dctPUSHBUTTON,                // type:      
   rbtTYPES,                     // rbSubtype: (n/a)
   false,                        // rbSelect:  (n/a)
   2,                            // ulY:       upper left corner in Y
   2,                            // ulX:       upper left corner in X
   1,                            // lines:     control lines
   15,                           // cols:      control columns
   "  BASIC TESTS  ",            // dispText:  
   nc.gyR,                       // nColor:    non-focus color
   nc.reG,                       // fColor:    focus color
   tbPrint,                      // filter:    (n/a)
   NULL,                         // label:     (n/a)
   ZERO,                         // labY:      (n/a)
   ZERO,                         // labX       (n/a)
   ddBoxTYPES,                   // exType:    (n/a)
   1,                            // scrItems:  (n/a)
   1,                            // scrSel:    (n/a)
   NULL,                         // scrColor:  (n/a)
   NULL,                         // spinData:  (n/a)
   true,                         // active:    allow control to gain focus
   &promptIc[pIntFmtPB],         // nextCtrl:  link in next structure
   },
   {  //* 'INT FMT' pushbutton - - - - - - - - - - - - - - - - - -   pIntFmtPB *
   dctPUSHBUTTON,                // type:      
   rbtTYPES,                     // rbSubtype: (n/a)
   false,                        // rbSelect:  (n/a)
   promptIc[pBasicPB].ulY,       // ulY:       upper left corner in Y
   short(promptIc[pBasicPB].ulX + promptIc[pBasicPB].cols + 2), // ulX:
   1,                            // lines:     control lines
   18,                           // cols:      control columns
   "  INTEGER FORMAT  ",         // dispText:  
   nc.gyR,                       // nColor:    non-focus color
   nc.reG,                       // fColor:    focus color
   tbPrint,                      // filter:    (n/a)
   NULL,                         // label:     (n/a)
   ZERO,                         // labY:      (n/a)
   ZERO,                         // labX       (n/a)
   ddBoxTYPES,                   // exType:    (n/a)
   1,                            // scrItems:  (n/a)
   1,                            // scrSel:    (n/a)
   NULL,                         // scrColor:  (n/a)
   NULL,                         // spinData:  (n/a)
   true,                         // active:    allow control to gain focus
   &promptIc[pDonePB],           // nextCtrl:  link in next structure
   },
   {  //* 'DONE' pushbutton - - - - - - - - - - - - - - - - - - - - -  pDonePB *
   dctPUSHBUTTON,                // type:      
   rbtTYPES,                     // rbSubtype: (n/a)
   false,                        // rbSelect:  (n/a)
   promptIc[pBasicPB].ulY,       // ulY:       upper left corner in Y
   short(promptIc[pIntFmtPB].ulX + promptIc[pIntFmtPB].cols + 2), // ulX:
   1,                            // lines:     control lines
   8,                            // cols:      control columns
   "  DONE  ",                   // dispText:  
   nc.gyR,                       // nColor:    non-focus color
   nc.reG,                       // fColor:    focus color
   tbPrint,                      // filter:    (n/a)
   NULL,                         // label:     (n/a)
   ZERO,                         // labY:      (n/a)
   ZERO,                         // labX       (n/a)
   ddBoxTYPES,                   // exType:    (n/a)
   1,                            // scrItems:  (n/a)
   1,                            // scrSel:    (n/a)
   NULL,                         // scrColor:  (n/a)
   NULL,                         // spinData:  (n/a)
   true,                         // active:    allow control to gain focus
   NULL                          // nextCtrl:  link in next structure
   },
} ;   // 'prompt' controls

//* Control definitions for 'Basic Test' dialog *
static InitCtrl basicIc[bCONTROLS_DEFINED] = 
{
   {  //* 'NEXT' pushbutton - - - - - - - - - - - - - - - - - - - -    bNextPB *
   dctPUSHBUTTON,                // type:      
   rbtTYPES,                     // rbSubtype: (n/a)
   false,                        // rbSelect:  (n/a)
   short(basicROWS - 2),         // ulY:       upper left corner in Y
   short(basicCOLS / 2 - 4),     // ulX:       upper left corner in X
   1,                            // lines:     control lines
   8,                            // cols:      control columns
   "  NEXT  ",                   // dispText:  
   nc.gyR,                       // nColor:    non-focus color
   nc.gyre | ncbATTR,            // fColor:    focus color
   tbPrint,                      // filter:    (n/a)
   "",                           // label:     (n/a)
   ZERO,                         // labY:      (n/a)
   ZERO,                         // labX       (n/a)
   ddBoxTYPES,                   // exType:    (n/a)
   1,                            // scrItems:  (n/a)
   1,                            // scrSel:    (n/a)
   NULL,                         // scrColor:  (n/a)
   NULL,                         // spinData:  (n/a)
   true,                         // active:    allow control to gain focus
   &basicIc[bDonePB],            // nextCtrl:  link in next structure
   },
   {  //* 'DONE' pushbutton - - - - - - - - - - - - - - - - - - - -    bDonePB *
   dctPUSHBUTTON,                // type:      
   rbtTYPES,                     // rbSubtype: (n/a)
   false,                        // rbSelect:  (n/a)
   basicIc[bNextPB].ulY,         // ulY:       upper left corner in Y
   short(basicIc[bNextPB].ulX + 16),// ulX:       upper left corner in X
   1,                            // lines:     control lines
   8,                            // cols:      control columns
   "  DONE  ",                   // dispText:  
   nc.gyR,                       // nColor:    non-focus color
   nc.gyre | ncbATTR,            // fColor:    focus color
   tbPrint,                      // filter:    (n/a)
   "",                           // label:     (n/a)
   ZERO,                         // labY:      (n/a)
   ZERO,                         // labX       (n/a)
   ddBoxTYPES,                   // exType:    (n/a)
   1,                            // scrItems:  (n/a)
   1,                            // scrSel:    (n/a)
   NULL,                         // scrColor:  (n/a)
   NULL,                         // spinData:  (n/a)
   true,                         // active:    allow control to gain focus
   &basicIc[bPrevPB],            // nextCtrl:  link in next structure
   },
   {  //* 'PREV' pushbutton - - - - - - - - - - - - - - - - - - - -    bPrevPB *
   dctPUSHBUTTON,                // type:      
   rbtTYPES,                     // rbSubtype: (n/a)
   false,                        // rbSelect:  (n/a)
   basicIc[bNextPB].ulY,         // ulY:       upper left corner in Y
   short(basicIc[bNextPB].ulX - 16),// ulX:       upper left corner in X
   1,                            // lines:     control lines
   8,                            // cols:      control columns
   "  PREV  ",                   // dispText:  
   nc.gyR,                       // nColor:    non-focus color
   nc.gyre | ncbATTR,            // fColor:    focus color
   tbPrint,                      // filter:    (n/a)
   "",                           // label:     (n/a)
   ZERO,                         // labY:      (n/a)
   ZERO,                         // labX       (n/a)
   ddBoxTYPES,                   // exType:    (n/a)
   1,                            // scrItems:  (n/a)
   1,                            // scrSel:    (n/a)
   NULL,                         // scrColor:  (n/a)
   NULL,                         // spinData:  (n/a)
   true,                         // active:    allow control to gain focus
   NULL,                         // nextCtrl:  link in next structure
   },
} ;   // 'basic test' controls

static const short SUFFIX_ITEMS = 5 ;  // number of items in list
static const short SUFFIX_WIDTH = 9 ;  // characters per item (incl. NULLCHAR)
static char SuffixData[SUFFIX_ITEMS][SUFFIX_WIDTH] = 
{
   " K      ",
   " k      ",
   " Kb     ",
   " kB     ",
   " KiB    "
} ;
attr_t monoColor[2] = { attrDFLT, nc.br } ;

//* Control definitions for 'Integer-format Test' dialog *
static InitCtrl intfmtIc[iCONTROLS_DEFINED] = 
{
   {  //* 'DONE' pushbutton - - - - - - - - - - - - - - - - - - - -    iDonePB *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      1,                            // ulY:       upper left corner in Y
      2,                            // ulX:       upper left corner in X
      1,                            // lines:     control lines
      8,                            // cols:      control columns
      "  DONE  ",                   // dispText:  
      nc.gyR,                       // nColor:    non-focus color
      nc.reG,                       // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      NULL,                         // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &intfmtIc[iRJustRB],          // nextCtrl:  link in next structure
   },
   {  //* 'RJUST' pushbutton  - - - - - - - - - - - - - - - - - - -   iRJustRB *
      dctRADIOBUTTON,               // type:      
      rbtS3a,                       // rbSubtype: standard, 5-wide
      true,                         // rbSelect:  initially set
      short(intfmtIc[iDonePB].ulY + 2), // ulY:       upper left corner in Y
      intfmtIc[iDonePB].ulX,        // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a)
      nc.blR,                       // nColor:    non-focus color
      nc.reG,                       // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "Rjust",                      // label:     
      ZERO,                         // labY:      
      4,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &intfmtIc[iLJustRB],          // nextCtrl:  link in next structure
   },
   {  //* 'LJUST' radio button  - - - - - - - - - - - - - - - - - - - iLJustRB *
      dctRADIOBUTTON,               // type:      
      rbtS3a,                       // rbSubtype: standard, 5-wide
      false,                        // rbSelect:  initially reset
      short(intfmtIc[iRJustRB].ulY + 1), // ulY:       upper left corner in Y
      intfmtIc[iDonePB].ulX,        // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a)
      nc.blR,                       // nColor:    non-focus color
      nc.reG,                       // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "Ljust",                      // label:     
      ZERO,                         // labY:      
      4,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &intfmtIc[iBit32RB],          // nextCtrl:  link in next structure
   },
   {  //* '32-BIT' radio button - - - - - - - - - - - - - - - - - - - iBit32RB *
      dctRADIOBUTTON,               // type:      
      rbtS3a,                       // rbSubtype: standard, 5-wide
      true,                         // rbSelect:  initially set
      short(intfmtIc[iLJustRB].ulY + 2), // ulY:       upper left corner in Y
      intfmtIc[iDonePB].ulX,        // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a)
      nc.blR,                       // nColor:    non-focus color
      nc.reG,                       // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "32-Bit",                     // label:     
      ZERO,                         // labY:      
      4,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &intfmtIc[iBit64RB],          // nextCtrl:  link in next structure
   },
   {  //* '64-BIT' radio button - - - - - - - - - - - - - - - - - - - iBit64RB *
      dctRADIOBUTTON,               // type:      
      rbtS3a,                       // rbSubtype: standard, 5-wide
      false,                        // rbSelect:  initially reset
      short(intfmtIc[iBit32RB].ulY + 1), // ulY:       upper left corner in Y
      intfmtIc[iDonePB].ulX,        // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a)
      nc.blR,                       // nColor:    non-focus color
      nc.reG,                       // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "64-Bit",                     // label:     
      ZERO,                         // labY:      
      4,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &intfmtIc[iDeciRB]            // nextCtrl:  link in next structure
   },
   {  //* 'Decimal Values" radio button   - - - - - - - - - - - - -    iDeciRB *
      dctRADIOBUTTON,               // type:      
      rbtS3a,                       // rbSubtype: standard, 5-wide
      true,                         // rbSelect:  initially reset
      short(intfmtIc[iBit64RB].ulY + 2), // ulY:       upper left corner in Y
      intfmtIc[iDonePB].ulX,        // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a)
      nc.blR,                       // nColor:    non-focus color
      nc.reG,                       // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "Decimal",                    // label:     
      ZERO,                         // labY:      
      4,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &intfmtIc[iKibiRB]            // nextCtrl:  link in next structure
   },
   {  //* 'Binary Values" radio button  - - - - - - - - - - - - - -    iKibiRB *
      dctRADIOBUTTON,               // type:      
      rbtS3a,                       // rbSubtype: standard, 5-wide
      false,                        // rbSelect:  initially reset
      short(intfmtIc[iDeciRB].ulY + 1), // ulY:       upper left corner in Y
      intfmtIc[iDonePB].ulX,        // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a)
      nc.blR,                       // nColor:    non-focus color
      nc.reG,                       // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "Binary",                     // label:     
      ZERO,                         // labY:      
      4,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &intfmtIc[iNegRB]             // nextCtrl:  link in next structure
   },
   {  //* 'Negative Value' radio button  - - - - - - - - - - - - - - -  iNegRB *
      dctRADIOBUTTON,               // type:      
      rbtS3a,                       // rbSubtype: standard, 5-wide
      false,                        // rbSelect:  initially reset
      short(intfmtIc[iKibiRB].ulY + 2), // ulY:       upper left corner in Y
      intfmtIc[iDonePB].ulX,        // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a)
      nc.blR,                       // nColor:    non-focus color
      nc.reG,                       // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "NegVal",                     // label:     
      ZERO,                         // labY:      
      4,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &intfmtIc[iSgnRB]             // nextCtrl:  link in next structure
   },
   {  //* 'Signed (+/-)" radio button - - - - - - - - - - - - - - - -   iSgnRB *
      dctRADIOBUTTON,               // type:      
      rbtS3a,                       // rbSubtype: standard, 5-wide
      false,                        // rbSelect:  initially reset
      short(intfmtIc[iNegRB].ulY + 2), // ulY:       upper left corner in Y
      intfmtIc[iDonePB].ulX,        // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a)
      nc.blR,                       // nColor:    non-focus color
      nc.reG,                       // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "Signed",                     // label:     
      ZERO,                         // labY:      
      4,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &intfmtIc[iUnitSB]            // nextCtrl:  link in next structure
   },
   {  //* 'Suffix Units" Scrollbox - - - - - - - - - - - - - - - - -   iUnitSB *
      dctSCROLLBOX,                 // type:      define a scrolling-data control
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(intfmtIc[iSgnRB].ulY + 2), // ulY:       upper left corner in Y
      intfmtIc[iDonePB].ulX,        // ulX:       upper left corner in X
      SUFFIX_ITEMS + 2,             // lines:     control lines
      SUFFIX_WIDTH + 2,             // cols:      control columns
      (const char*)&SuffixData,     // dispText:  text-data array
      nc.gyR,                       // nColor:    non-focus border color
      nc.gr,                        // fColor:    focus border color
      tbPrint,                      // filter:    (n/a)
      " UNITS ",                    // label:     label text
      ZERO,                         // labY:      label offset
      ZERO,                         // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      SUFFIX_ITEMS,                 // scrItems:  number of elements in text/color arrays
      ZERO,                         // scrSel:    index of initial highlighted element
      monoColor,                    // scrColor:  color-attribute list
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      NULL                          // nextCtrl:  link in next structure
   },
} ;   // 'integer-format test' controls

//*************************
//*     ~gStringTest      *
//*************************
//******************************************************************************
//* Destructor.                                                                *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

gStringTest::~gStringTest ( void )
{

   if ( this->dp != NULL )                // close the window
      delete ( this->dp ) ;

}  //* End ~gStringTest() 

//*************************
//*      gStringTest      *
//*************************
//******************************************************************************
//* Constructor.                                                               *
//*                                                                            *
//* Input  : tLines     : number of display line in terminal window            *
//*          tCols      : number of display columns in terminal window         *
//*          minY       : first available display line                         *
//*                                                                            *
//* Returns: implicitly return class reference                                 *
//******************************************************************************

gStringTest::gStringTest ( short tLines, short tCols, short minY )
{
   //* Initialize data members *
   this->termRows  = tLines ;
   this->termCols  = tCols ;
   this->minULY    = minY ;
   this->dRows     = 1 ;
   this->dCols     = 1 ;
   this->icPtr     = NULL ;
   this->dp        = NULL ;
   this->dColor    = nc.blR ;
   this->tColor    = this->dColor | ncbATTR ;
   this->bColor    = nc.gybl | ncbATTR ;
   this->hColor    = nc.brbl | ncbATTR ;
   this->gtOpen    = false ;

   //* Initialize remainder of control-definition array.           *
   //* (colors become available after NCurses engine instantiated) *
   promptIc[pBasicPB].nColor = promptIc[pIntFmtPB].nColor = 
   promptIc[pDonePB].nColor  = nc.gyR ;
   promptIc[pBasicPB].fColor = promptIc[pIntFmtPB].fColor = 
   promptIc[pDonePB].fColor  = nc.gyre | ncbATTR ;

   basicIc[bNextPB].nColor = basicIc[bDonePB].nColor = 
   basicIc[bPrevPB].nColor = nc.gyR ;
   basicIc[bNextPB].fColor = basicIc[bDonePB].fColor = 
   basicIc[bPrevPB].fColor = nc.gyre | ncbATTR ;

   intfmtIc[iDonePB].nColor = nc.gyR ;
   intfmtIc[iDonePB].fColor = nc.gyre | ncbATTR ;
   intfmtIc[iRJustRB].nColor = intfmtIc[iLJustRB].nColor = 
   intfmtIc[iBit32RB].nColor = intfmtIc[iBit64RB].nColor = 
   intfmtIc[iDeciRB].nColor = intfmtIc[iKibiRB].nColor = 
   intfmtIc[iNegRB].nColor = intfmtIc[iSgnRB].nColor = this->dColor ;
   intfmtIc[iRJustRB].fColor = intfmtIc[iLJustRB].fColor = 
   intfmtIc[iBit32RB].fColor = intfmtIc[iBit64RB].fColor = 
   intfmtIc[iDeciRB].fColor = intfmtIc[iKibiRB].fColor = 
   intfmtIc[iNegRB].fColor = intfmtIc[iSgnRB].fColor = nc.gyre | ncbATTR ;
   intfmtIc[iUnitSB].nColor = this->dColor ;
   intfmtIc[iUnitSB].fColor = nc.gyre | ncbATTR ; ;
   monoColor[1] = nc.br ;

   //* Open the prompt dialog. If dialog opens caller may continue.*
   if ( (this->gtOpen = this->gtOpenPrompt ()) != false )
   {

   }  // OpenWindow()

}  //* End gStringTest() 

//*************************
//*     gtOpenPrompt      *
//*************************
//******************************************************************************
//* Open the prompt dialog window.                                             *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: 'true' if dialog window opened successfully, else 'false'         *
//******************************************************************************

bool gStringTest::gtOpenPrompt ( void )
{
   short ctrX    = this->termCols/2,         // dialog center in X
         ulY     = this->minULY,             // upper left corner in Y
         ulX     = ctrX - promptCOLS / 2 ;   // upper left corner in X
   bool  success = false ;

   //* Initial parameters for dialog window *
   this->icPtr = promptIc ;
   this->dRows = promptROWS ;
   this->dCols = promptCOLS ;
   InitNcDialog dInit( this->dRows,    // number of display lines
                       this->dCols,    // number of display columns
                       ulY,            // Y offset from upper-left of terminal 
                       ulX,            // X offset from upper-left of terminal 
                       NULL,           // dialog title
                       ncltSINGLE,     // border line-style
                       this->bColor,   // border color attribute
                       this->dColor,   // interior color attribute
                       this->icPtr     // pointer to list of control definitions
                     ) ;

   //* Instantiate the dialog window *
   this->dp = new NcDialog ( dInit ) ;

   //* Open the dialog window *
   if ( (this->dp->OpenWindow()) == OK )
   {
      this->dp->SetDialogTitle ( "  Select a Test Sequence  ", this->tColor ) ;

      this->dp->RefreshWin () ;
      success = true ;
   }
   return success ;

}  //* End gtOpenPrompt() *

//*************************
//*    gtDialogOpened     *
//*************************
//******************************************************************************
//* Satisfy caller's curiosity.                                                *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: 'true' if dialog opened successfully, else 'false'                *
//******************************************************************************

bool gStringTest::gtDialogOpened ( void )
{

   return this->gtOpen ;

}  //* End gtDialogOpened() *

//*************************
//*      gtInteract       *
//*************************
//******************************************************************************
//* User interactive experience. Calls the appropriate user-interface method.  *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void gStringTest::gtInteract ( void )
{
   if ( this->gtOpen )
   {
      bool done = false ;
      while ( ! done )
      {
         if ( this->gtOpen != false && this->dp != NULL && this->icPtr == promptIc )
         {  //* Get user's selection *
            short sel = this->gtPromptUI () ;
            this->gtCloseDialog () ;      // close the dialog

            if ( sel == pDonePB )         // if we're done
               done = true ;
            else if ( sel == pBasicPB )   // execute 'basic test' sequence
            {
               if ( (this->gtOpen = this->gtOpenBasic ()) != false )
               {
                  this->gtBasicUI () ;
                  this->gtCloseDialog () ;
               }
            }
            else if ( sel == pIntFmtPB )  // execute 'integer format' sequence
            {
               if ( (this->gtOpen = this->gtOpenIntFmt ()) != false )
               {
                  this->gtIntFmtUI () ;
                  this->gtCloseDialog () ;
               }
            }
         }
         else
         {  //* Re-open the prompt dialog *
            this->gtOpen = this->gtOpenPrompt () ;
         }
      }
   }
}  //* End gtInteract() *

//*************************
//*      gtPromptUI       *
//*************************
//******************************************************************************
//* PRIVATE METHOD: Prompt user for test sequence to execute.                  *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: member of enum gtpControls                                        *
//******************************************************************************

short gStringTest::gtPromptUI ( void )
{
   short selection = pDonePB ;
   uiInfo Info ;                 // user interface data returned here
   bool   done = false ;         // loop control
   while ( ! done )
   {
      this->dp->EditPushbutton ( Info ) ;
      if ( Info.dataMod != false )
      {
         selection = Info.ctrlIndex ; 
         done = true ;
      }
      else
      {  //* Move to next/previous control *
         if ( Info.keyIn == nckSTAB )  this->dp->PrevControl () ; 
         else                          this->dp->NextControl () ;
      }
   }
   return selection ;

}  //* End gtPromptUI() *

//*************************
//*      gtOpenBasic      *
//*************************
//******************************************************************************
//* Open the 'basic test' dialog window.                                       *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: 'true' if dialog window opened successfully, else 'false'         *
//******************************************************************************

bool gStringTest::gtOpenBasic ( void )
{
   short ctrX    = this->termCols/2,         // dialog center in X
         ulY     = this->minULY,             // upper left corner in Y
         ulX     = ctrX - basicCOLS / 2 ;    // upper left corner in X
   bool  success = false ;

   //* Initial parameters for dialog window *
   this->icPtr = basicIc ;
   this->dRows = basicROWS ;
   this->dCols = basicCOLS ;
   InitNcDialog dInit( this->dRows,    // number of display lines
                       this->dCols,    // number of display columns
                       ulY,            // Y offset from upper-left of terminal 
                       ulX,            // X offset from upper-left of terminal 
                       NULL,           // dialog title
                       ncltSINGLE,     // border line-style
                       this->bColor,   // border color attribute
                       this->dColor,   // interior color attribute
                       this->icPtr     // pointer to list of control definitions
                     ) ;

   //* Instantiate the dialog window *
   this->dp = new NcDialog ( dInit ) ;

   //* Open the dialog window *
   if ( (this->dp->OpenWindow()) == OK )
   {
      this->dp->SetDialogTitle ( "  gString-class Basic Functionality  ", this->tColor ) ;

      this->dp->RefreshWin () ;
      success = true ;
   }
   return success ;

}  //* End gtOpenBasic() *

//*************************
//*       gtBasicUI       *
//*************************
//******************************************************************************
//* PRIVATE METHOD: 'Basic Test' user interface.                               *
//* Dialog was opened by caller.                                               *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void gStringTest::gtBasicUI ( void )
{
   this->displayPage = 1 ;    // track which page of data is displayed
   this->gtBasicPage1 () ;    // display the first page of test data

   //* Allow user to play with the controls.      *
   //* (Please see the notes in NcDialog.hpp on   *
   //* values returned (from the editing routines)*
   uiInfo Info ;                 // user interface data returned here
   short  icIndex = ZERO ;       // index of control with input focus
   bool   done = false ;         // loop control
   while ( ! done )
   {
      //*******************************************
      //* If focus is currently on a Pushbutton   *
      //*******************************************
      if ( basicIc[icIndex].type == dctPUSHBUTTON )
      {
         icIndex = this->dp->EditPushbutton ( Info ) ;
         if ( Info.dataMod != false )
         {
            if ( Info.ctrlIndex == bDonePB )
               done = true ;
            else if ( Info.ctrlIndex == bNextPB )
            {
               if ( this->displayPage == 1 )
               {
                  ++this->displayPage ;
                  this->gtBasicPage2 () ;
                  icIndex = this->dp->PrevControl () ;
               }
               else if ( this->displayPage == 2 )
               {
                  ++this->displayPage ;
                  this->gtBasicPage3 () ;
                  icIndex = this->dp->PrevControl () ;
               }
               else if ( this->displayPage == 3 )
               {
                  ++this->displayPage ;
                  this->gtBasicPage4 () ;
                  icIndex = this->dp->PrevControl () ;
               }
               else if ( this->displayPage == 4 )
               {
                  ++this->displayPage ;
                  this->gtBasicPage5 () ;
                  icIndex = this->dp->PrevControl () ;
               }
               else if ( this->displayPage == 5 )
               {
                  ++this->displayPage ;
                  this->gtBasicPage6 () ;
                  icIndex = this->dp->PrevControl () ;
               }
               else if ( this->displayPage == 6 )
               {
                  ++this->displayPage ;
                  this->gtBasicPage7 () ;
                  icIndex = this->dp->PrevControl () ;
               }
               else if ( this->displayPage == 7 )
               {
                  ++this->displayPage ;
                  this->gtBasicPage8 () ;
                  icIndex = this->dp->PrevControl () ;
               }
               else if ( this->displayPage == 8 )
               {
                  ++this->displayPage ;
                  this->gtBasicPage9 () ;
                  icIndex = this->dp->PrevControl () ;
               }
               else if ( this->displayPage == 9 )
               {
                  ++this->displayPage ;
                  this->gtBasicPage10 () ;
                  icIndex = this->dp->PrevControl () ;
               }
               else if ( this->displayPage == 10 )
               {
                  ++this->displayPage ;
                  this->gtBasicPage11 () ;
                  icIndex = this->dp->PrevControl () ;
               }
            }
            else if ( Info.ctrlIndex == bPrevPB )
            {
               if ( this->displayPage == 11 )
               {
                  --this->displayPage ;
                  this->gtBasicPage10 () ;
                  icIndex = this->dp->PrevControl () ;
               }
               else if ( this->displayPage == 10 )
               {
                  --this->displayPage ;
                  this->gtBasicPage9 () ;
                  icIndex = this->dp->PrevControl () ;
               }
               else if ( this->displayPage == 9 )
               {
                  --this->displayPage ;
                  this->gtBasicPage8 () ;
                  icIndex = this->dp->PrevControl () ;
               }
               else if ( this->displayPage == 8 )
               {
                  --this->displayPage ;
                  this->gtBasicPage7 () ;
                  icIndex = this->dp->PrevControl () ;
               }
               else if ( this->displayPage == 7 )
               {
                  --this->displayPage ;
                  this->gtBasicPage6 () ;
                  icIndex = this->dp->PrevControl () ;
               }
               else if ( this->displayPage == 6 )
               {
                  --this->displayPage ;
                  this->gtBasicPage5 () ;
                  icIndex = this->dp->PrevControl () ;
               }
               else if ( this->displayPage == 5 )
               {
                  --this->displayPage ;
                  this->gtBasicPage4 () ;
                  icIndex = this->dp->PrevControl () ;
               }
               else if ( this->displayPage == 4 )
               {
                  --this->displayPage ;
                  this->gtBasicPage3 () ;
                  icIndex = this->dp->PrevControl () ;
               }
               else if ( this->displayPage == 3 )
               {
                  --this->displayPage ;
                  this->gtBasicPage2 () ;
                  icIndex = this->dp->PrevControl () ;
               }
               else if ( this->displayPage == 2 )
               {
                  --this->displayPage ;
                  this->gtBasicPage1 () ;
               }
            }
         }
      }
      else
      { /* no other control types defined for this method */ }

      //* Move input focus to next/previous control.*
      if ( done == false && Info.viaHotkey == false )
      {
         if ( Info.keyIn == nckSTAB )
            icIndex = this->dp->PrevControl () ; 
         else
            icIndex = this->dp->NextControl () ;
      }
   }  // while()

}  //* End gtBasicUI() *

//*************************
//*     gtBasicPage1      *
//*************************
//******************************************************************************
//* Display a page of test output for gtBasic().                               *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void gStringTest::gtBasicPage1 ( void )
{
const char uColHead[] = { "gsColsgsCharsutfBytesisASCII" } ;
const char* uSubHead[] = 
{
   "Instantiation Test:",
   "Assignment Test:",
   "Object-Copy Test:",
   "Character-Limit Test       :|1---5----0----5----0----5----0----5----0|",
   "Full display string        : ",
   "Display first 25 characters: ",
   "Column-Limit Test          :|1---5----0----5----0----5----0----5----0|",
   "Display first 25 columns   : ",
} ;
const char  uAsciiText[] =
{
   "Happy~Birthday~to~You! "
   "- src:UTF-8    out:UTF-8   : "
} ;
const char  uC01[] =  
{
   " 这种评论荒谬绝伦。    "
   "- src:UTF-8    out:UTF-8   : "
} ;
const char  uC02[] =  
{
   " 这种评论荒谬绝伦。    "
   "- src:UTF-8    out:wchar_t : "
} ;
const wchar_t  wR01[] = 
{
   L" Как поживаете?        "
    "- src:wchar_t  out:wchar_t : "
} ;
const wchar_t  wR02[] = 
{
   L" Как поживаете?        "
    "- src:wchar_t  out:UTF-8   : "
} ;
const char  uK01[] = 
{
   " こんにちは始めまして。"
   "- src:UTF-8    out:UTF-8   : "
} ;
const char  uK02[] = 
{
   " こんにちは始めまして。"
   "- src:UTF-8    out:wchar_t : "
} ;
const wchar_t wP01[] = 
{
   L" On patrzy na tę piękną"
    "- src:wchar_t  out:wchar_t : "
} ;
const wchar_t wP02[] = 
{
   L" On patrzy na tę piękną"
    "- src:wchar_t  out:UTF-8   : "
} ;
// limitChars() test 1                                | (25th character and column)
const char  uChCount01[] = { "Hello everyone - I am very happy to meet you today!" } ;
// limitChars() test 2           (25th column) |      | (25th character)
const char  uChCount02[] = { "Hello-你好-안녕하세요!-Mountainous views are beautiful." } ;

   this->dp->ClearWin () ;       // erase old data from window

   winPos wp( 1, 53 ) ;          // cursor positioning
   int    ccnt, bcnt ;           // statistics
   int    srcOffset ;            // offset into source array

   //*********************************************************************
   //* Write column headings while testing the gString limitChars()      *
   //* method as well as assignment operator and byte count using UTF-8  *
   //* ASCII data.                                                       *
   //*********************************************************************
   gString gsHead(uColHead, 6) ;
   wp = this->dp->WriteString ( wp.ypos, (wp.xpos), gsHead.ustr(ccnt, bcnt), 
                                this->dColor | ncuATTR ) ;
   srcOffset = bcnt - 1 ;
   gsHead = &uColHead[srcOffset] ;
   gsHead.limitChars(7) ;
   wp = this->dp->WriteString ( wp.ypos, (wp.xpos += 2), gsHead.ustr(ccnt, bcnt), 
                                this->dColor | ncuATTR ) ;
   srcOffset += bcnt - 1 ;
   gsHead = &uColHead[srcOffset] ;
   gsHead.limitChars(8) ;
   wp = this->dp->WriteString ( wp.ypos, (wp.xpos += 2), gsHead.ustr(ccnt, bcnt), 
                                this->dColor | ncuATTR ) ;
   srcOffset += bcnt - 1 ;
   gsHead = &uColHead[srcOffset] ;
   gsHead.limitChars(8) ;
   wp = this->dp->WriteString ( wp.ypos++, (wp.xpos += 2), gsHead.ustr(ccnt, bcnt), 
                                this->dColor | ncuATTR ) ;

   this->dp->WriteString ( wp.ypos++, (wp.xpos=1), uSubHead[0], this->tColor ) ; // sub-heading

   //****************************************************************
   //* Instantiation test using the available instantiation methods *
   //****************************************************************
   gString  gsA01( uAsciiText ) ;
   wp = this->dp->WriteString ( wp.ypos, wp.xpos, gsA01.ustr(), this->dColor ) ;
   wp = this->gtBasicStats ( gsA01, wp, this->dColor ) ;

   gString  gsC01( uC01 ) ;
   wp = this->dp->WriteString ( wp.ypos, wp.xpos, gsC01.ustr(), this->dColor ) ;
   wp = this->gtBasicStats ( gsC01, wp, dColor ) ;

   gString  gsC02( uC02 ) ;
   wp = this->dp->WriteString ( wp.ypos, (wp.xpos=1), gsC02.gstr(), this->dColor ) ;
   wp = this->gtBasicStats ( gsC02, wp, this->dColor ) ;

   gString  gsR01( wR01 ) ;
   wp = this->dp->WriteString ( wp.ypos, (wp.xpos=1), gsR01.gstr(), this->dColor ) ;
   wp = this->gtBasicStats ( gsR01, wp, this->dColor ) ;

   gString  gsR02( wR02 ) ;
   wp = this->dp->WriteString ( wp.ypos, (wp.xpos=1), gsR02.ustr(), this->dColor ) ;
   wp = this->gtBasicStats ( gsR02, wp, this->dColor ) ;

   //***********************************************************************
   //* Assignment tests for UTF-8 data and wchar_t data, using 'operator=' *
   //***********************************************************************
   this->dp->WriteString ( ++wp.ypos, (wp.xpos=1), uSubHead[1], this->tColor ) ; // sub-heading

   gsC01 = uK01 ;
   wp = this->dp->WriteString ( ++wp.ypos, wp.xpos, gsC01.ustr(), this->dColor ) ;
   wp = this->gtBasicStats ( gsC01, wp, this->dColor ) ;

   gsC02 = uK02 ;
   wp = this->dp->WriteString ( wp.ypos, (wp.xpos=1), gsC02.gstr(), this->dColor ) ;
   wp = this->gtBasicStats ( gsC02, wp, this->dColor ) ;

   gsR01 = wP01 ;
   wp = this->dp->WriteString ( wp.ypos, (wp.xpos=1), gsR01.gstr(), this->dColor ) ;
   wp = this->gtBasicStats ( gsR01, wp, this->dColor ) ;

   gsR02 = wP02 ;
   wp = this->dp->WriteString ( wp.ypos, (wp.xpos=1), gsR02.ustr(), this->dColor ) ;
   wp = this->gtBasicStats ( gsR02, wp, this->dColor ) ;

   //********************************************************
   //* Copy one gString object to another using 'operator=' *
   //********************************************************
   this->dp->WriteString ( ++wp.ypos, (wp.xpos=1), uSubHead[2], this->tColor ) ; // sub-heading

   gsC02 = uC01 ;
   gsC01 = gsC02 ;
   wp = this->dp->WriteString ( ++wp.ypos, wp.xpos, gsC01.ustr(), this->dColor ) ;
   wp = this->gtBasicStats ( gsC01, wp, this->dColor ) ;
   gsC01 = uC02 ;
   gsC02 = gsC01 ;
   wp = this->dp->WriteString ( wp.ypos, (wp.xpos=1), gsC02.gstr(), this->dColor ) ;
   wp = this->gtBasicStats ( gsC02, wp, this->dColor ) ;

   //************************
   //* Character-limit Test *
   //************************
   this->dp->WriteString ( ++wp.ypos, (wp.xpos=1), uSubHead[3], this->tColor ) ; // sub-heading
   wp = this->dp->WriteString ( ++wp.ypos, wp.xpos, uSubHead[4], this->dColor ) ;
   gString ccount(uChCount01) ;
   this->dp->WriteString ( wp.ypos++, wp.xpos, ccount.ustr(), this->dColor ) ;
   wp = this->dp->WriteString ( wp.ypos, (wp.xpos=1), uSubHead[5], this->dColor ) ;
   ccount.limitChars(25) ;
   this->dp->WriteString ( wp.ypos++, wp.xpos, ccount.ustr(), this->dColor ) ;
   wp = this->dp->WriteString ( wp.ypos, (wp.xpos=1), uSubHead[4], this->dColor ) ;
   ccount = uChCount02 ;
   this->dp->WriteString ( wp.ypos++, wp.xpos, ccount.gstr(), this->dColor ) ;
   wp = this->dp->WriteString ( wp.ypos, (wp.xpos=1), uSubHead[5], this->dColor ) ;
   ccount.limitChars(25) ;
   this->dp->WriteString ( wp.ypos++, wp.xpos, ccount.gstr(), this->dColor ) ;

   //*********************
   //* Column-limit Test *
   //*********************
   this->dp->WriteString ( ++wp.ypos, (wp.xpos=1), uSubHead[6], this->tColor ) ; // sub-heading
   wp = this->dp->WriteString ( ++wp.ypos, wp.xpos, uSubHead[4], this->dColor ) ;
   ccount = uChCount01 ;
   this->dp->WriteString ( wp.ypos++, wp.xpos, ccount.gstr(), this->dColor ) ;
   wp = this->dp->WriteString ( wp.ypos, (wp.xpos=1), uSubHead[7], this->dColor ) ;
   ccount.limitCols(25) ;
   this->dp->WriteString ( wp.ypos++, wp.xpos, ccount.gstr(), this->dColor ) ;
   wp = this->dp->WriteString ( wp.ypos, (wp.xpos=1), uSubHead[4], this->dColor ) ;
   ccount = uChCount02 ;
   this->dp->WriteString ( wp.ypos++, wp.xpos, ccount.ustr(), this->dColor ) ;
   wp = this->dp->WriteString ( wp.ypos, (wp.xpos=1), uSubHead[7], this->dColor ) ;
   ccount.limitCols(25) ;
   this->dp->WriteString ( wp.ypos++, wp.xpos, ccount.ustr(), this->dColor ) ;

   this->dp->RefreshWin () ;     // make everything visible

}  //* End gtBasicPage1() *

//*************************
//*     gtBasicPage2      *
//*************************
//******************************************************************************
//* Display a page of test output for gtBasic().                               *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void gStringTest::gtBasicPage2 ( void )
{
const char* uSubHead[] = 
{
   "Formatted Output with 'compose()'  : ",
   "String Comparison with '=='/'!=' operators and 'compare()' methods :",
   "Make a copy using 'copy()' method  :|1---5----0----5----0----5----0----5----0| ",
} ;

   winPos   wp( 1, 1 ) ;

   this->dp->ClearWin () ;    // erase old data from window

   //*************************************
   //* Formatted output with 'compose()' *
   //*************************************
   wp = this->dp->WriteString ( wp.ypos, wp.xpos, uSubHead[0], this->tColor ) ; // sub-heading
   wchar_t  wChinesePointer[] = { L"指针" } ;
   char     uStr[]   = { "Hi" } ;
   wchar_t  wStr[]   = { L"There" } ;
   char     cVal     = 'W' ;
   wchar_t  wVal     = L'C' ;
   const char* const    uaWidth = "thin" ;
   const wchar_t* const waWidth = L"phat" ;
   const char* const    ucWidth = "修长的" ;    // 'slender'
   const wchar_t* const wcWidth = L"肥胖的" ;   // fat
   short    shVal    = 25 ;
   int      intVal   = 50 ;
   float    fltVal   = 71.4328751 ;
   double   dblVal   = 44.7281459 ;
   USHORT   ushVal   = 0x8401 ;
   unsigned uintVal  = 0xC4123B56 ;
   UINT64   qiVal    = (UINT64)(0x014D) ;    // (quiet compiler warning     )
            qiVal += (UINT64)(0xFFFFFFFF) ;  // (about constant being too   )
            qiVal += (UINT64)(0xFFFFFFFF) ;  // (large for 'long int'       )
                                             // ( UINT64 qiVal=0x20000014B; )
   long double LfVal = -821.255 ;
   int      nCount ;    // target value for "%n" parameter
   #if 1    // gString class 'compose' method
   #if 1    // instantiate, THEN compose
   gString  fgs ;
   fgs.compose( L"* '%-3s%ls %c.%C.'-'%04hd'-'%-3d'-'%-4.3f'-'%2.2lf'-'%hu'-'%#08x' *",
                uStr,    // char string
                wStr,    // wchar_t string
                &cVal,   // char
                &wVal,   // wchar_t
                &shVal,  // short int
                &intVal, // int
                &fltVal, // float
                &dblVal, // double
                &ushVal, // unsigned short int
                &uintVal // unsigned int (hexadecimal output)
              ) ;
   #else    // instantiate and compose in one step
   gString  fgs(
                "* '%-3s%ls %c.%C.'-'%04hd'-'%-3d'-'%-4.3f'-'%2.2lf'-'%hu'-'%#08x' *",
                uStr,    // char string
                wStr,    // wchar_t string
                &cVal,   // char
                &wVal,   // wchar_t
                &shVal,  // short int
                &intVal, // int
                &fltVal, // float
                &dblVal, // double
                &ushVal, // unsigned short int
                &uintVal // unsigned int (hexadecimal output)
              ) ;
   #endif   // instantiate and compose in one step

   this->dp->WriteString ( wp.ypos++, wp.xpos, fgs, this->dColor ) ;
   fgs.compose( L"* '%S':'%p' - 'n' type n:'%n' 'm' (errno) type:'%m' *", 
                wChinesePointer, wChinesePointer, &nCount, (void*)NULL ) ;
   this->dp->WriteString ( wp.ypos, wp.xpos, fgs, this->dColor ) ;
   fgs.compose( L"(%%n value s/b 34, value is: %02d)", &nCount ) ;
   this->dp->WriteString ( wp.ypos++, 3, fgs, this->dColor ) ;
   fgs.compose( L"* unsigned long long int: '%#Lx' - long double: '%.4Lf'   *", 
                &qiVal, &LfVal ) ;
   this->dp->WriteString ( wp.ypos++, wp.xpos, fgs, this->dColor ) ;

   #else    // Same output as above, but using the gsForm class
   gsForm   gsf
   {
      L"* '%-3s%ls %c.%C.'-'%04hd'-'%-3d'-'%-4.3f'-'%2.2lf'-'%hu'-'%#08x' *",
      {
         uStr,    // char string
         wStr,    // wchar_t string
         &cVal,   // char
         &wVal,   // wchar_t
         &shVal,  // short int
         &intVal, // int
         &fltVal, // float
         &dblVal, // double
         &ushVal, // unsigned short int
         &uintVal,// unsigned int (hexadecimal output)
      },
   } ;
   gString  fgs( gsf ) ;
   this->dp->WriteString ( wp.ypos++, wp.xpos, fgs, this->dColor ) ;
   gsf.fmtSpec = L"* '%S':'%p' - 'n' type n:'%n' 'm' (errno) type:'%m'     *" ;
   gsf.argList[0] = wChinesePointer ;
   gsf.argList[1] = wChinesePointer ;
   gsf.argList[2] = &nCount ;
   gsf.argList[3] = (void*)NULL ;
   fgs = gsf ;
   this->dp->WriteString ( wp.ypos, wp.xpos, fgs, this->dColor ) ;
   gsf.fmtSpec = L"(%%n value s/b 34, value is: %02d)", 
   gsf.argList[0] = &nCount ;
   fgs = gsf ;
   this->dp->WriteString ( wp.ypos++, 3, fgs, this->dColor ) ;
   gsf.fmtSpec = L"* unsigned long long int: '%#Lx' - long double: '%.4Lf'   *" ;
   gsf.argList[0] = &qiVal ;
   gsf.argList[1] = &LfVal ;
   fgs = gsf ;
   this->dp->WriteString ( wp.ypos++, wp.xpos, fgs, this->dColor ) ;
   #endif   // gsForm class formatting

   //* swprintf function bug-fix for for incorrect padding width *
   wp = this->dp->WriteString ( wp.ypos, (wp.xpos=1), "*swprintf bug-fix ", 
                          this->hColor ) ;
   fgs.compose( L"!%5s!!%-5s!!%5S!!%-5S!!%3S!"
                 " !%s!!%7s!!%-7s! !%S!!%7S!!%-7S!*",
                uaWidth, uaWidth, waWidth, waWidth, waWidth, 
                ucWidth, ucWidth, ucWidth, wcWidth, wcWidth, wcWidth ) ;
   this->dp->WriteString ( wp.ypos++, wp.xpos, fgs, this->dColor ) ;

   //* Binary output formatting extension to the swprintf output specifications*
   ++wp.ypos ; wp.xpos = 1 ;
   wp = this->dp->WriteString ( wp, "Binary Bit Formatting: ", this->hColor ) ;
   fgs.compose( L"byte %#hhx", &cVal ) ;
   wp = this->dp->WriteString ( wp, fgs, this->bColor | ncuATTR ) ;
   fgs.compose( L" -> %hhb", &cVal ) ;
   wp = this->dp->WriteString ( wp, fgs, this->dColor ) ;
   wp = this->dp->WriteString ( wp, "  OR  ", this->bColor ) ;
   fgs.compose( L"%#hhb", &cVal ) ;
   wp = this->dp->WriteString ( wp, fgs, this->dColor ) ;
   wp = this->dp->WriteString ( wp, "  OR  ", this->bColor ) ;
   fgs.compose( L"%-#hhb  ", &cVal ) ;
   this->dp->WriteString ( wp.ypos++, wp.xpos, fgs, this->dColor ) ;

   shVal += 0x2B00 ;
   fgs.compose ( L"short %#hx", &shVal ) ;
   wp = this->dp->WriteString ( wp.ypos, (wp.xpos=1), fgs, this->bColor | ncuATTR ) ;
   fgs.compose( L" -> %#hB", &shVal ) ;
   wp = this->dp->WriteString ( wp, fgs, this->dColor ) ;
   wp = this->dp->WriteString ( wp, "  OR  ", this->bColor ) ;
   fgs.compose( L"%-# .4hb", &shVal ) ;
   wp = this->dp->WriteString ( wp, fgs, this->dColor ) ;
   wp = this->dp->WriteString ( wp, "  OR  ", this->bColor ) ;
   fgs.compose( L"%-#-.8hb", &shVal ) ;
   this->dp->WriteString ( wp, fgs, this->dColor ) ;

   ++wp.ypos ; wp.xpos = 1 ;
   fgs.compose ( L"int 0x%08X", &uintVal ) ;
   wp = this->dp->WriteString ( wp, fgs, this->bColor | ncuATTR ) ;
   fgs.compose( L" -> %-#_B", &uintVal ) ;
   wp = this->dp->WriteString ( wp, fgs, this->dColor ) ;
   wp = this->dp->WriteString ( wp, "  OR  ", this->bColor ) ;
   fgs.compose( L"%-# .8b", &uintVal ) ;
   this->dp->WriteString ( wp, fgs, this->dColor ) ;

   qiVal <<= 24 ; qiVal *= 3 ; qiVal += 0x4A5B6C7D ;
   fgs.compose( L"UINT64 0x%016LX", &qiVal ) ;
   wp = this->dp->WriteString ( ++wp.ypos, (wp.xpos=1), fgs, this->bColor | ncuATTR ) ;
   fgs.compose( L" -> %#`.8llB", &qiVal ) ;
   this->dp->WriteString ( wp.ypos++, wp.xpos, fgs, this->dColor ) ;

   //*******************************************************
   //* String comparison with '==' and 'compare()' methods *
   //*******************************************************
   ++wp.ypos ;
   this->dp->WriteString ( wp.ypos++, (wp.xpos=1), uSubHead[1], this->tColor ) ; // sub-heading
   fgs = L"Alphabet soup" ;
   gString ggs( "Alphabet song" ) ;
   wp = this->dp->WriteString ( wp.ypos, wp.xpos, "operator==               : '", this->dColor ) ;
   wp = this->dp->WriteString ( wp.ypos, wp.xpos, fgs, this->dColor ) ;
   wp = this->dp->WriteChar ( wp.ypos, wp.xpos, L'\'', this->dColor ) ;
   if ( fgs == ggs )
      wp = this->dp->WriteString ( wp.ypos, wp.xpos, " == '", this->dColor ) ;
   else
      wp = this->dp->WriteString ( wp.ypos, wp.xpos, " != '", this->dColor ) ;
   wp = this->dp->WriteString ( wp.ypos, wp.xpos, ggs, this->dColor ) ;
   wp = this->dp->WriteChar ( wp.ypos, wp.xpos, L'\'', this->dColor ) ;
   winPos col2 = wp ;
   col2.xpos += 3 ;

   wp = this->dp->WriteString ( ++wp.ypos, (wp.xpos=1), 
                                "compare( UTF-8 string )  : '", this->dColor ) ;
   wp = this->dp->WriteString ( wp.ypos, wp.xpos, fgs.ustr(), this->dColor ) ;
   short diff = fgs.compare( ggs.ustr() ) ;
   if ( diff > ZERO )
      wp = this->dp->WriteString ( wp.ypos, wp.xpos, L"' >  '", this->dColor ) ;
   else if ( diff < ZERO )
      wp = this->dp->WriteString ( wp.ypos, wp.xpos, L"' <  '", this->dColor ) ;
   else
      wp = this->dp->WriteString ( wp.ypos, wp.xpos, L"' == '", this->dColor ) ;
   wp = this->dp->WriteString ( wp.ypos, wp.xpos, ggs, this->dColor ) ;
   this->dp->WriteChar ( wp.ypos, wp.xpos, L'\'', this->dColor ) ;

   wp = this->dp->WriteString ( ++wp.ypos, (wp.xpos=1), 
                                "compare( wchar_t string ): '", this->dColor ) ;
   wp = this->dp->WriteString ( wp.ypos, wp.xpos, ggs.gstr(), this->dColor ) ;
   diff = ggs.compare( fgs.gstr() ) ;
   if ( diff > ZERO )
      wp = this->dp->WriteString ( wp.ypos, wp.xpos, L"' >  '", this->dColor ) ;
   else if ( diff < ZERO )
      wp = this->dp->WriteString ( wp.ypos, wp.xpos, L"' <  '", this->dColor ) ;
   else
      wp = this->dp->WriteString ( wp.ypos, wp.xpos, L"' == '", this->dColor ) ;
   wp = this->dp->WriteString ( wp.ypos, wp.xpos, fgs, this->dColor ) ;
   this->dp->WriteChar ( wp.ypos, wp.xpos, L'\'', this->dColor ) ;

   wp = this->dp->WriteString ( ++wp.ypos, (wp.xpos=1), 
                               "compare( uStr, 1, 11, 0) : '", this->dColor ) ;
   gString gstmp(fgs.ustr(), 11 ) ;
   wp = this->dp->WriteString ( wp.ypos, wp.xpos, gstmp, this->dColor ) ;
   diff = fgs.compare( ggs.ustr(), true, 11, ZERO ) ;
   if ( diff > ZERO )
      wp = this->dp->WriteString ( wp.ypos, wp.xpos, L"' >  '", this->dColor ) ;
   else if ( diff < ZERO )
      wp = this->dp->WriteString ( wp.ypos, wp.xpos, L"' <  '", this->dColor ) ;
   else
      wp = this->dp->WriteString ( wp.ypos, wp.xpos, L"' == '", this->dColor ) ;
   wp = this->dp->WriteString ( wp.ypos, wp.xpos, gstmp, this->dColor ) ;
   this->dp->WriteChar ( wp.ypos, wp.xpos, L'\'', this->dColor ) ;

   wp = this->dp->WriteString ( ++wp.ypos, (wp.xpos=1), 
                                "compare( wStr, 1, 8, 0 ) : '", this->dColor ) ;
   gstmp.limitChars( 8 ) ;
   wp = this->dp->WriteString ( wp.ypos, wp.xpos, gstmp, this->dColor ) ;
   diff = ggs.compare( fgs.gstr(), true, 8, ZERO ) ;
   if ( diff > ZERO )
      wp = this->dp->WriteString ( wp.ypos, wp.xpos, L"' >  '", this->dColor ) ;
   else if ( diff < ZERO )
      wp = this->dp->WriteString ( wp.ypos, wp.xpos, L"' <  '", this->dColor ) ;
   else
      wp = this->dp->WriteString ( wp.ypos, wp.xpos, L"' == '", this->dColor ) ;
   wp = this->dp->WriteString ( wp.ypos, wp.xpos, gstmp, this->dColor ) ;
   this->dp->WriteChar ( wp.ypos, wp.xpos, L'\'', this->dColor ) ;

   wp = this->dp->WriteString ( ++wp.ypos, (wp.xpos=1), 
                                "compare( wStr, 1, 12, 0 ): '", this->dColor ) ;
   gstmp = ggs ;
   gstmp.limitChars( 12 ) ;
   wp = this->dp->WriteString ( wp.ypos, wp.xpos, gstmp, this->dColor ) ;
   diff = ggs.compare( fgs.gstr(), true, 12, ZERO ) ;
   if ( diff > ZERO )
      wp = this->dp->WriteString ( wp.ypos, wp.xpos, L"' >  '", this->dColor ) ;
   else if ( diff < ZERO )
      wp = this->dp->WriteString ( wp.ypos, wp.xpos, L"' <  '", this->dColor ) ;
   else
      wp = this->dp->WriteString ( wp.ypos, wp.xpos, L"' == '", this->dColor ) ;
   gstmp = fgs ;
   gstmp.limitChars( 12 ) ;
   wp = this->dp->WriteString ( wp.ypos, wp.xpos, gstmp, this->dColor ) ;
   this->dp->WriteChar ( wp.ypos, wp.xpos, L'\'', this->dColor ) ;

   wp = col2 ;
   fgs = "在动物园-Bonobo" ;
   ggs = "在动物园-Chimpanzee" ;
   wp = this->dp->WriteChar ( wp.ypos, wp.xpos, L'\'', this->dColor ) ;
   wp = this->dp->WriteString ( wp.ypos, wp.xpos, fgs, this->dColor ) ;
   if ( fgs == ggs )
      wp = this->dp->WriteString ( wp.ypos, wp.xpos, "' == '", this->dColor ) ;
   else
      wp = this->dp->WriteString ( wp.ypos, wp.xpos, "' != '", this->dColor ) ;
   wp = this->dp->WriteString ( wp.ypos, wp.xpos, ggs, this->dColor ) ;
   this->dp->WriteChar ( wp.ypos, wp.xpos, L'\'', this->dColor ) ;

   gstmp = fgs ;
   gstmp.limitChars( 4 ) ;
   wp = this->dp->WriteChar ( ++wp.ypos, (wp.xpos=col2.xpos), L'\'', this->dColor ) ;
   wp = this->dp->WriteString ( wp.ypos, wp.xpos, gstmp, this->dColor ) ;
   diff = fgs.compare( ggs.ustr(), true, 4 ) ;
   if ( diff > ZERO )
      wp = this->dp->WriteString ( wp.ypos, wp.xpos, L"' >  '", this->dColor ) ;
   else if ( diff < ZERO )
      wp = this->dp->WriteString ( wp.ypos, wp.xpos, L"' <  '", this->dColor ) ;
   else
      wp = this->dp->WriteString ( wp.ypos, wp.xpos, L"' == '", this->dColor ) ;
   wp = this->dp->WriteString ( wp.ypos, wp.xpos, gstmp, this->dColor ) ;
   this->dp->WriteChar ( wp.ypos, wp.xpos, L'\'', this->dColor ) ;

   gstmp = fgs ;
   gstmp.limitChars( 6 ) ;
   wp = this->dp->WriteChar ( ++wp.ypos, (wp.xpos=col2.xpos), L'\'', this->dColor ) ;
   wp = this->dp->WriteString ( wp.ypos, wp.xpos, gstmp, this->dColor ) ;
   diff = fgs.compare( ggs.gstr(), true, 6 ) ;
   if ( diff > ZERO )
      wp = this->dp->WriteString ( wp.ypos, wp.xpos, L"' >  '", this->dColor ) ;
   else if ( diff < ZERO )
      wp = this->dp->WriteString ( wp.ypos, wp.xpos, L"' <  '", this->dColor ) ;
   else
      wp = this->dp->WriteString ( wp.ypos, wp.xpos, L"' == '", this->dColor ) ;
   gstmp = ggs ;
   gstmp.limitChars( 6 ) ;
   wp = this->dp->WriteString ( wp.ypos, wp.xpos, gstmp, this->dColor ) ;
   this->dp->WriteChar ( wp.ypos, wp.xpos, L'\'', this->dColor ) ;

   gstmp = fgs ;
   gstmp.limitChars( 4 ) ;
   wp = this->dp->WriteString ( ++wp.ypos, (wp.xpos=col2.xpos), 
                                L"compare(wch, 1, 2, 2): '", this->dColor ) ;
   wp = this->dp->WriteString ( wp.ypos, wp.xpos, &gstmp.gstr()[2], this->dColor ) ;
   diff = fgs.compare( &ggs.gstr()[2], true, 2, 2 ) ;
   if ( diff > ZERO )
      wp = this->dp->WriteString ( wp.ypos, wp.xpos, L"' >  '", this->dColor ) ;
   else if ( diff < ZERO )
      wp = this->dp->WriteString ( wp.ypos, wp.xpos, L"' <  '", this->dColor ) ;
   else
      wp = this->dp->WriteString ( wp.ypos, wp.xpos, L"' == '", this->dColor ) ;
   gstmp = ggs ;
   gstmp.limitChars( 4 ) ;
   wp = this->dp->WriteString ( wp.ypos, wp.xpos, &gstmp.gstr()[2], this->dColor ) ;
   this->dp->WriteChar ( wp.ypos, wp.xpos, L'\'', this->dColor ) ;

   gstmp = fgs ;
   gstmp.limitChars( 6 ) ;
   wp = this->dp->WriteString ( ++wp.ypos, (wp.xpos=col2.xpos), 
                                L"compare(wch, 1, 4, 2): '", this->dColor ) ;
   wp = this->dp->WriteString ( wp.ypos, wp.xpos, &gstmp.gstr()[2], this->dColor ) ;
   diff = fgs.compare( &ggs.gstr()[2], true, 4, 2 ) ;
   if ( diff > ZERO )
      wp = this->dp->WriteString ( wp.ypos, wp.xpos, L"' >  '", this->dColor ) ;
   else if ( diff < ZERO )
      wp = this->dp->WriteString ( wp.ypos, wp.xpos, L"' <  '", this->dColor ) ;
   else
      wp = this->dp->WriteString ( wp.ypos, wp.xpos, L"' == '", this->dColor ) ;
   gstmp = ggs ;
   gstmp.limitChars( 6 ) ;
   wp = this->dp->WriteString ( wp.ypos, wp.xpos, &gstmp.gstr()[2], this->dColor ) ;
   this->dp->WriteChar ( wp.ypos, wp.xpos, L'\'', this->dColor ) ;

   gstmp = fgs ;
   gstmp.limitChars( 11 ) ;
   wp = this->dp->WriteString ( ++wp.ypos, (wp.xpos=col2.xpos), 
                                L"compare(wch, 1, 6, 5): '", this->dColor ) ;
   wp = this->dp->WriteString ( wp.ypos, wp.xpos, &gstmp.gstr()[5], this->dColor ) ;
   diff = fgs.compare( &ggs.gstr()[5], true, 6, 5 ) ;
   if ( diff > ZERO )
      wp = this->dp->WriteString ( wp, L"' > '", this->dColor ) ;
   else if ( diff < ZERO )
      wp = this->dp->WriteString ( wp, L"' < '", this->dColor ) ;
   else
      wp = this->dp->WriteString ( wp, L"' == '", this->dColor ) ;
   gstmp = ggs ;
   gstmp.limitChars( 11 ) ;
   wp = this->dp->WriteString ( wp, &gstmp.gstr()[5], this->dColor ) ;
   this->dp->WriteChar ( wp, L'\'', this->dColor ) ;
   ++wp.ypos ;

   wp.xpos = 1 ;
   ggs = "Hello, Hippo!" ;
   gstmp = "hello, hippo!" ;
   wp = this->dp->WriteString ( wp, "case-sensitive: ", this->hColor ) ;
   wp = this->dp->WriteChar ( wp, L'\'', this->dColor ) ;
   wp = this->dp->WriteString ( wp, ggs, this->dColor ) ;
   wp = this->dp->WriteChar ( wp, "'  ", this->dColor ) ;
   #if 1    //* Compare using source string *
   diff = ggs.compare( gstmp.ustr() ) ;
   #else    //* Compare using source gString object *
   diff = ggs.compare( gstmp ) ;
   #endif   //* Compare using source gString object *
   if ( diff > ZERO )
      wp = this->dp->WriteString ( wp, L"' > '", this->dColor ) ;
   else if ( diff < ZERO )
      wp = this->dp->WriteString ( wp, L"' < '", this->dColor ) ;
   else
      wp = this->dp->WriteString ( wp, L"' == '", this->dColor ) ;
   wp = this->dp->WriteString ( wp, gstmp, this->dColor ) ;
   wp = this->dp->WriteChar ( wp, L'\'', this->dColor ) ;

   wp = this->dp->WriteString ( wp, "  case-insensitive: ", this->hColor ) ;
   wp = this->dp->WriteChar ( wp, L'\'', this->dColor ) ;
   wp = this->dp->WriteString ( wp, ggs, this->dColor ) ;
   wp = this->dp->WriteChar ( wp, "'  ", this->dColor ) ;
   #if 1    //* Compare using source string *
   diff = ggs.compare( gstmp.ustr(), false ) ;
   #else    //* Compare using source gString object *
   diff = ggs.compare( gstmp, false ) ;
   #endif   //* Compare using source gString object *
   if ( diff > ZERO )
      wp = this->dp->WriteString ( wp, L"' > '", this->dColor ) ;
   else if ( diff < ZERO )
      wp = this->dp->WriteString ( wp, L"' < '", this->dColor ) ;
   else
      wp = this->dp->WriteString ( wp, L"' == '", this->dColor ) ;
   wp = this->dp->WriteString ( wp, gstmp, this->dColor ) ;
   wp = this->dp->WriteChar ( wp, L'\'', this->dColor ) ;
   ++wp.ypos ;

   //*****************************
   //* Test the 'copy()' methods *
   //*****************************
   // copy() bytes/cols test 1-2         (30th column) |      | (30 bytes)  ??     
   // copy() chars/cols test 3-4         (30th column) |      | (30th character)
   const char     uChCount03[] = { "Hello-你好-안녕하세요!-Mountainous views." } ;

   gString  gsc( uChCount03 ), gst1, gst2 ;
   short    tch, tco, tby ;
   char     ucStr[gsDFLTBYTES] ;
   wchar_t  wcStr[gsALLOCDFLT] ;
   wp = this->dp->WriteString ( ++wp.ypos, (wp.xpos=1), uSubHead[2], 
                                this->tColor ) ; // sub-heading
   this->dp->WriteString ( wp.ypos++, wp.xpos, 
                           "chars cols bytes", this->tColor | ncuATTR ) ;
   wp.xpos = 1 ;
   wp = this->dp->WriteString ( wp.ypos, (wp.xpos + 24), 
                                "Base String: ", this->dColor ) ;
   this->dp->WriteString ( wp.ypos, wp.xpos, gsc.gstr(), this->dColor ) ;
   tch  = gsc.gschars() ;
   tco = gsc.gscols() ;
   tby = gsc.utfbytes() ;
   gst2.compose ( L" %02hd    %02hd   %02hd", &tch, &tco, &tby ) ;
   this->dp->WriteString ( wp.ypos++, 80, gst2, this->dColor ) ;

   //* Copy entire string *
   wp = this->dp->WriteString ( wp.ypos, (wp.xpos=2), 
                               "Copy entire string                : ", this->dColor ) ;
   gsc.copy( ucStr, gsALLOCDFLT ) ;
   this->dp->WriteString ( wp.ypos, wp.xpos, ucStr, this->dColor ) ;
   gst1 = ucStr ;       // load resulting string for analysis
   tch  = gst1.gschars() ;
   tco = gst1.gscols() ;
   tby = gst1.utfbytes() ;
   gst2.compose ( L" %02hd    %02hd   %02hd", &tch, &tco, &tby ) ;
   this->dp->WriteString ( wp.ypos++, 80, gst2, this->dColor ) ;

   //* Copy 30 bytes (UTF-8) *
   wp = this->dp->WriteString ( wp.ypos, (wp.xpos=2), 
                               "Copy first 30 bytes        (UTF-8): ", this->dColor ) ;
   gsc.copy( ucStr, 30 ) ;
   this->dp->WriteString ( wp.ypos, wp.xpos, ucStr, this->dColor ) ;
   gst1 = ucStr ;       // load resulting string for analysis
   tch  = gst1.gschars() ;
   tco = gst1.gscols() ;
   tby = gst1.utfbytes() ;
   gst2.compose ( L" %02hd    %02hd   %02hd", &tch, &tco, &tby ) ;
   this->dp->WriteString ( wp.ypos++, 80, gst2, this->dColor ) ;

   //* Copy 28 bytes (UTF-8) (should return 27, rounded down to whole characters) *
   wp = this->dp->WriteString ( wp.ypos, (wp.xpos=2), 
                               "Copy first 28 bytes        (UTF-8): ", this->dColor ) ;
   gsc.copy( ucStr, 28 ) ;
   this->dp->WriteString ( wp.ypos, wp.xpos, ucStr, this->dColor ) ;
   gst1 = ucStr ;       // load resulting string for analysis
   tch  = gst1.gschars() ;
   tco = gst1.gscols() ;
   tby = gst1.utfbytes() ;
   gst2.compose ( L" %02hd    %02hd   %02hd", &tch, &tco, &tby ) ;
   wp = this->dp->WriteString ( wp.ypos, 80, gst2, this->dColor ) ;
   gst1 = "요!" ;
   tby = gst1.utfbytes() - 1 ;   // don't report the NULLCHAR
   gst2.compose ( L"(\"%S\"=%hdbytes)", gst1.gstr(), &tby ) ;
   this->dp->WriteString ( wp.ypos++, wp.xpos, gst2, this->dColor ) ;

   //* Copy 30 columns (UTF-8) *
   wp = this->dp->WriteString ( wp.ypos, (wp.xpos=2), 
                               "Copy first 30 columns      (UTF-8): ", this->dColor ) ;
   gsc.copy( ucStr, gsDFLTBYTES, 30 ) ;
   this->dp->WriteString ( wp.ypos, wp.xpos, ucStr, this->dColor ) ;
   gst1 = ucStr ;       // load resulting string for analysis
   tch  = gst1.gschars() ;
   tco = gst1.gscols() ;
   tby = gst1.utfbytes() ;
   gst2.compose ( L" %02hd    %02hd   %02hd", &tch, &tco, &tby ) ;
   this->dp->WriteString ( wp.ypos++, 80, gst2, this->dColor ) ;
   
   //* Copy 30 characters (wchar_t) *
   wp = this->dp->WriteString ( wp.ypos, (wp.xpos=2), 
                               "Copy first 30 characters (wchar_t): ", this->dColor ) ;
   gsc.copy( wcStr, 30 ) ;
   this->dp->WriteString ( wp.ypos, wp.xpos, wcStr, this->dColor ) ;
   gst1 = wcStr ;       // load resulting string for analysis
   tch  = gst1.gschars() ;
   tco = gst1.gscols() ;
   tby = gst1.utfbytes() ;
   gst2.compose ( L" %02hd    %02hd   %02hd", &tch, &tco, &tby ) ;
   this->dp->WriteString ( wp.ypos++, 80, gst2, this->dColor ) ;

   //* Copy 30 columns (wchar_t) *
   wp = this->dp->WriteString ( wp.ypos, (wp.xpos=2), 
                               "Copy first 30 columns    (wchar_t): ", this->dColor ) ;
   gsc.copy( wcStr, gsDFLTBYTES, 30 ) ;
   this->dp->WriteString ( wp.ypos, wp.xpos, wcStr, this->dColor ) ;
   gst1 = wcStr ;       // load resulting string for analysis
   tch  = gst1.gschars() ;
   tco = gst1.gscols() ;
   tby = gst1.utfbytes() ;
   gst2.compose ( L" %02hd    %02hd   %02hd", &tch, &tco, &tby ) ;
   this->dp->WriteString ( wp.ypos++, 80, gst2, this->dColor ) ;

   this->dp->RefreshWin () ;     // make everything visible

}  //* End gtBasicPage2() *

//*************************
//*     gtBasicPage3      *
//*************************
//******************************************************************************
//* Display a page of test output for gtBasic().                               *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void gStringTest::gtBasicPage3 ( void )
{
const char* uSubHead[] = 
{
   "Append text with 'append()' method : 1---5----0----5----0----5----0----5----0"
                                        "----5----0----5----0----5----0",
   "Shift data by Characters with 'shiftChars()' method: ",
   "Shift data by Columns with 'shiftCols()' method    : ",
} ;
const char* Ruler = 
   "|1---5----0----5----0----5----0----5----0----5----0"
    "     |----5----0----5----0----5----0----5----0----5|" ;
const wchar_t* wData[] = 
{
   // 45 characters, 44 columns
   L"LoveLoveLove, Sing loud, Dance around naked!",
   // 29 characters, 50 columns
   L"一二三四五六七八九十 十一 十二 十三 十四 十五 十六",
} ;
wchar_t  chPeriod = L'。' ;     // Chinese period
//wchar_t  chComma  = L'，' ;     // Chinese comma

   winPos   wp( 1, 1 ) ;

   this->dp->ClearWin () ;    // erase old data from window


   //*******************************
   //* Test the 'append()' methods *
   //*******************************
   const char     uAppendBase[] = { "At Spring Festival" } ;
   const char     uAppend1[] = { " I ate 饺子。" } ;   // dumplings. (jiǎozi.) 
   // "I always add some sesame oil into the vinegar when eating dumplings."
   // "Wǒ chī jiǎozi shí zǒngshì zài cù zhōng jiā yìxiē zhīmáyóu."
   const char     uAppend2[] = { "我吃饺子时总是在醋中加一些芝麻油,Tasty!" } ;

   gString  gsA( uAppendBase ), gsB( uAppend2 ), gsMsg ;
   short    cCount ;
   this->dp->WriteString ( wp.ypos, (wp.xpos=1), uSubHead[0], this->tColor ) ; // sub-heading
   cCount = gsA.gschars() ;
   gsMsg.compose( L"Base String   (%02hd chars incl. null): ", &cCount ) ;
   wp = this->dp->WriteString ( ++wp.ypos, wp.xpos, gsMsg, this->dColor ) ;
   this->dp->WriteString ( wp.ypos, wp.xpos, gsA, this->dColor ) ;

   //* Append UTF-8 text *
   cCount = gsA.append( uAppend1 ) ;
   gsMsg.compose( L"Append UTF-8 text data   (%02hd chars): ", &cCount ) ;
   wp = this->dp->WriteString ( ++wp.ypos, (wp.xpos=1), gsMsg, this->dColor ) ;
   this->dp->WriteString ( wp.ypos, wp.xpos, gsA, this->dColor ) ;

   //* Append wchar_t text *
   cCount = gsA.append( gsB.gstr() ) ;
   gsMsg.compose( L"Append wchar_t text data (%02hd chars): ", &cCount ) ;
   wp = this->dp->WriteString ( ++wp.ypos, (wp.xpos=1), gsMsg, this->dColor ) ;
   this->dp->WriteString ( wp.ypos++, wp.xpos, gsA, this->dColor ) ;

   //* Append formatted (composed) text *
   wp.xpos = 1 ;
   wp = this->dp->WriteString ( wp, "Append formatted text data : ", this->dColor ) ;
   short gaddress = 2840 ;
   wchar_t gdirection = L'E' ;
   const wchar_t* gstreet = L"Colorado Blvd." ;
   double gcost = 29.25 ;
   gsA = "Gorilla Men's Clothing" ;
   this->dp->WriteString ( wp.ypos++, wp.xpos, gsA, this->dColor ) ;
   gsA.append( ", %hd %C %S", &gaddress, &gdirection, gstreet ) ;
   this->dp->WriteString ( wp.ypos++, wp.xpos, gsA, this->dColor ) ;
   gsA.append( "  Dress shirts on sale, $%.2lf.", &gcost ) ;
   this->dp->WriteString ( wp.ypos++, wp.xpos, gsA, this->dColor ) ;

   //*************************************
   //* Shifting with 'shiftChars()'      *
   //*************************************
   ++wp.ypos ; wp.xpos = 1 ;
   gString fgs( wData[0] ) ;
   this->dp->WriteString ( wp.ypos++, wp.xpos, uSubHead[1], this->tColor ) ; // sub-heading
   this->dp->WriteString ( wp.ypos++, wp.xpos, Ruler, this->dColor ) ;
   ++wp.xpos ;
   winPos wpl = this->dp->WriteString ( wp.ypos++, wp.xpos, fgs, this->dColor ) ;
   fgs.shiftChars( -8 ) ;
   this->dp->WriteString ( wp.ypos++, wp.xpos, fgs, this->dColor ) ;
   fgs.shiftChars( 8, '.' ) ;
   this->dp->WriteString ( wp.ypos++, wp.xpos, fgs, this->dColor ) ;
   fgs.shiftChars( -7 ) ;
   this->dp->WriteString ( wp.ypos++, wp.xpos, fgs, this->dColor ) ;
   fgs.shiftChars( 7 ) ;
   this->dp->WriteString ( wp.ypos++, wp.xpos, fgs, this->dColor ) ;
   fgs.shiftChars( -8 ) ;
   this->dp->WriteString ( wp.ypos++, wp.xpos, fgs, this->dColor ) ;
   fgs.shiftChars( 1009, '#' ) ;
   wp = this->dp->WriteString ( wp, L"[...]", this->hColor ) ;
   wp = this->dp->WriteString ( wp.ypos++, wp.xpos, &fgs.gstr()[1009], this->dColor ) ;
   this->dp->WriteString ( wp, L"<<==|", this->hColor ) ;

   wpl.xpos += 2 ;
   this->dp->WriteString ( wpl,                  "( base )", this->dColor ) ;
   this->dp->WriteString ( ++wpl.ypos, wpl.xpos, "(  -8  )", this->dColor ) ;
   this->dp->WriteString ( ++wpl.ypos, wpl.xpos, "(+8fill)", this->dColor ) ;
   this->dp->WriteString ( ++wpl.ypos, wpl.xpos, "(  -7  )", this->dColor ) ;
   this->dp->WriteString ( ++wpl.ypos, wpl.xpos, "(+7 spc)", this->dColor ) ;
   this->dp->WriteString ( ++wpl.ypos, wpl.xpos, "(  -8  )", this->dColor ) ;
   this->dp->WriteString ( ++wpl.ypos, wpl.xpos, "(+1009 )", this->dColor ) ;

   wp = { 11, short(wpl.xpos + 10) } ;
   fgs = wData[1] ;
   this->dp->WriteString ( wp.ypos++, wp.xpos, fgs, this->dColor ) ;
   fgs.shiftChars( -8 ) ;
   this->dp->WriteString ( wp.ypos++, wp.xpos, fgs, this->dColor ) ;
   fgs.shiftChars( 8, chPeriod ) ;
   this->dp->WriteString ( wp.ypos++, wp.xpos, fgs, this->dColor ) ;
   fgs.shiftChars( -7 ) ;
   this->dp->WriteString ( wp.ypos++, wp.xpos, fgs, this->dColor ) ;
   fgs.shiftChars( 7, ',' ) ;
   this->dp->WriteString ( wp.ypos++, wp.xpos, fgs, this->dColor ) ;
   fgs.shiftChars( -8 ) ;
   this->dp->WriteString ( wp.ypos++, wp.xpos, fgs, this->dColor ) ;
   fgs.shiftChars( 1009, '#' ) ;
   wp = this->dp->WriteString ( wp, L"[...]", this->hColor ) ;
   wp = this->dp->WriteString ( wp.ypos++, wp.xpos, &fgs.gstr()[1009], this->dColor ) ;
   this->dp->WriteString ( wp, L"<<==|", this->hColor ) ;

   //*************************************
   //* Shifting with 'shiftCols()'       *
   //*************************************
   wp.ypos += 2 ; wp.xpos = 1 ;
   wpl.ypos += 4 ;

   fgs = wData[0] ;
   this->dp->WriteString ( wp.ypos++, wp.xpos, uSubHead[2], this->tColor ) ; // sub-heading
   this->dp->WriteString ( wp.ypos++, wp.xpos, Ruler, this->dColor ) ;
   ++wp.xpos ;
   this->dp->WriteString ( wp.ypos++, wp.xpos, fgs, this->dColor ) ;

   this->dp->WriteString ( wpl,                  "( base )", this->dColor ) ;
   this->dp->WriteString ( ++wpl.ypos, wpl.xpos, "(  -8  )", this->dColor ) ;
   this->dp->WriteString ( ++wpl.ypos, wpl.xpos, "(+8fill)", this->dColor ) ;
   this->dp->WriteString ( ++wpl.ypos, wpl.xpos, "(  -7  )", this->dColor ) ;
   this->dp->WriteString ( ++wpl.ypos, wpl.xpos, "(+7 spc)", this->dColor ) ;
   this->dp->WriteString ( ++wpl.ypos, wpl.xpos, "(  -8  )", this->dColor ) ;
   this->dp->WriteString ( ++wpl.ypos, wpl.xpos, "(+1009 )", this->dColor ) ;

   fgs.shiftCols( -8 ) ;
   this->dp->WriteString ( wp.ypos++, wp.xpos, fgs, this->dColor ) ;
   fgs.shiftCols( 8, '.' ) ;
   this->dp->WriteString ( wp.ypos++, wp.xpos, fgs, this->dColor ) ;
   fgs.shiftCols( -7 ) ;
   this->dp->WriteString ( wp.ypos++, wp.xpos, fgs, this->dColor ) ;
   fgs.shiftCols( 7 ) ;
   this->dp->WriteString ( wp.ypos++, wp.xpos, fgs, this->dColor ) ;
   fgs.shiftCols( -8 ) ;
   this->dp->WriteString ( wp.ypos++, wp.xpos, fgs, this->dColor ) ;
   fgs.shiftCols( 1009, '#' ) ;
   wp = this->dp->WriteString ( wp, L"[...]", this->hColor ) ;
   wp = this->dp->WriteString ( wp.ypos++, wp.xpos, &fgs.gstr()[1009], this->dColor ) ;
   this->dp->WriteString ( wp, L"<<==|", this->hColor ) ;

   wp = { 21, short(wpl.xpos + 10) } ;
   fgs = wData[1] ;
   this->dp->WriteString ( wp.ypos++, wp.xpos, fgs, this->dColor ) ;
   fgs.shiftCols( -8 ) ;
   this->dp->WriteString ( wp.ypos++, wp.xpos, fgs, this->dColor ) ;
   fgs.shiftCols( 8, ',' ) ;
   this->dp->WriteString ( wp.ypos++, wp.xpos, fgs, this->dColor ) ;
   fgs.shiftCols( -7 ) ;
   this->dp->WriteString ( wp.ypos++, wp.xpos, fgs, this->dColor ) ;
   fgs.shiftCols( 7 ) ;
   this->dp->WriteString ( wp.ypos++, wp.xpos, fgs, this->dColor ) ;
   fgs.shiftCols( -8 ) ;
   this->dp->WriteString ( wp.ypos++, wp.xpos, fgs, this->dColor ) ;
   // NOTE: The following shift correctly inserts 1009 columns at the head of 
   //       the array. The static calculation is that the 14 rightmost columns 
   //       should be truncated; however, 15 columns (9 characters) are actually 
   //       truncated because the non-pad display data starts on an odd column.
   //       Fret not, for all is well  :-)  
   fgs.shiftCols( 1009, '#' ) ;
   wp = this->dp->WriteString ( wp, L"[...]", this->hColor ) ;
   wp = this->dp->WriteString ( wp.ypos++, wp.xpos, &fgs.gstr()[1009], this->dColor ) ;
   this->dp->WriteString ( wp, L"<<==|", this->hColor ) ;
   wp.ypos += 2 ;
   wp.xpos = 2 ;


// Insert additional tests here...


   dp->RefreshWin () ;     // make everything visible

}  //* End gtBasicPage3() *

//*************************
//*     gtBasicPage4      *
//*************************
//******************************************************************************
//* Display a page of test output for gtBasic().                               *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void gStringTest::gtBasicPage4 ( void )
{
   const char* uSubHead[] = 
   {
      "Insert text with 'insert()' method : |--insertion point (default)    "
      "                                      ",
      "             |--insertion point                                      ",
      "                            insertion point beyond the end--|        ",
      "                              (null terminator here)--|             ",
      "Erase text with 'erase()' method :                                   "
      "                                      \n",
      "Strip leading and trailing whitespace with 'strip()'.\n",
   } ;
   const char  uInsertBase[] = 
   { "Lovers, be fruitful and do long division!" } ;
   const char  uInsert1[] = { "Hello, " } ;
   const char  uInsert2[] = { " and friends" } ;
   const char  uInsert3[] = { " (happily)" } ;
   const char* uLandM = 
   "And in the end, the love you take is equal to the "
   "love you make.--Lennon and McCartney" ;
   const short lmChars = 87 ;    // (not really)
   const char* uInsert4 = " if you're very lucky," ;
                        //"  Lennon and McCartney"
   const wchar_t* baseTemplate = L"Base String (%02hd chars incl. null)  : " ;
   const wchar_t* newTemplate = L"New String  (%02hd chars incl. null)  : " ;
   gString  gsA( uInsertBase ), gsMsg ;
   short cCount = gsA.gschars() ;
   const short insPOINT = 13 ;
   const short dispOFFSET = gsALLOCDFLT - lmChars ;
   const short insPOINT2 = dispOFFSET + 15 ;
   winPos   wp( 2, 1 ), wphead ;
   this->dp->ClearWin () ;    // erase old data from window

   //*******************************
   //* Test the 'insert()' methods *
   //*******************************
   this->dp->WriteString ( wp, uSubHead[0], this->tColor ) ; // sub-heading
   gsMsg.compose( baseTemplate, &cCount ) ;
   wp = this->dp->WriteString ( ++wp.ypos, wp.xpos, gsMsg, this->dColor ) ;
   this->dp->WriteString ( wp, gsA, this->dColor ) ;
   wphead = wp ;  wphead.ypos += 2 ;
   gsA.insert( uInsert1 ) ;
   cCount = gsA.gschars() ;
   gsMsg.compose( newTemplate, &cCount ) ;
   wp = this->dp->WriteString ( ++wp.ypos, (wp.xpos=1), gsMsg, this->dColor ) ;
   this->dp->WriteString ( wp.ypos++, wp.xpos, gsA, this->dColor ) ;

   this->dp->WriteString ( wphead, uSubHead[1], this->tColor ) ; // sub-heading
   wphead.ypos += 2 ;
   gsA.insert( uInsert2, insPOINT ) ;
   cCount = gsA.gschars() ;
   gsMsg.compose( newTemplate, &cCount ) ;
   wp = this->dp->WriteString ( ++wp.ypos, (wp.xpos=1), gsMsg, this->dColor ) ;
   this->dp->WriteString ( wp, gsA, this->dColor ) ;

   this->dp->WriteString ( wphead, uSubHead[2], this->tColor ) ; // sub-heading
   wphead.ypos += 2 ;
   gsA.insert( uInsert3, (gsA.gschars() + 5) ) ;   // (beyond end of string)
   cCount = gsA.gschars() ;
   gsMsg.compose( newTemplate, &cCount ) ;
   wp = this->dp->WriteString ( (wp.ypos+=2), (wp.xpos=1), gsMsg, this->dColor ) ;
   this->dp->WriteString ( wp, gsA, this->dColor ) ;

   //* Test string-length limits *
   wp = this->dp->WriteString ( ++wp.ypos, (wp.xpos=1), 
                                "Truncate at Window Limit: ", this->hColor ) ;
   this->dp->WriteString ( wphead, uSubHead[3], this->tColor ) ; // sub-heading
   wp = this->dp->WriteString ( ++wp.ypos, (wp.xpos=1), "[...]", this->hColor ) ;
   // Note that this truncation test is no longer valid because gString will now 
   // reallocate storage to accomodate the overrun. 23-Mar-2025
   gsA.clear() ;                    // fill the object
   do
   { gsA.append( "0123456789" ) ; } while ( (gsA.gschars()) < dispOFFSET ) ;
   gsA.limitChars( dispOFFSET ) ;
   gsA.append( uLandM ) ;
   this->dp->WriteString ( wp, &(gsA.gstr()[dispOFFSET]), this->dColor ) ;
   wp = this->dp->WriteString ( ++wp.ypos, (wp.xpos=1), "[...]", this->hColor ) ;
   gsA.insert( uInsert4, insPOINT2 ) ;
   this->dp->WriteString ( wp, &(gsA.gstr()[dispOFFSET]), this->dColor ) ;
   this->dp->WriteString ( ++wp.ypos, (wp.xpos=1), 
                     "                     -- ------ ---- ------", this->hColor ) ;
   wp.ypos += 2 ; wp.xpos = 1 ;

   //******************************
   //* Test the 'erase()' methods *
   //******************************
   const char* heat = "In the heat of the night, when the music is right, "
                      "the stars all align and the romance is fine.\n" ;
   gString gse( heat ) ;
   wp = this->dp->WriteParagraph ( wp, uSubHead[4], this->tColor ) ; // sub-heading
   wp = this->dp->WriteString ( wp, "Basic Text: ", this->hColor ) ;
   wp = this->dp->WriteParagraph ( wp, gse, this->dColor ) ;
   wp.xpos = 1 ;
   wp = this->dp->WriteString ( wp, "Truncate  : ", this->hColor ) ;
   gse.erase( &gse.gstr()[70] ) ;
   gse.append( L'\n' ) ;
   wp = this->dp->WriteParagraph ( wp, gse, this->dColor ) ;
   wp.xpos = 1 ;
   wp = this->dp->WriteString ( wp, "Cut       : ", this->hColor ) ;
   gse.erase( L" all" ) ;
   wp = this->dp->WriteParagraph ( wp, gse, this->dColor ) ;
   wp.xpos = 1 ;
   wp = this->dp->WriteString ( wp, "Cut       : ", this->hColor ) ;
   gse.erase( " HeAt oF ThE", ZERO, false ) ;
   wp = this->dp->WriteParagraph ( wp, gse, this->dColor ) ;
   wp.xpos = 1 ;
   wp = this->dp->WriteString ( wp, "Cut       : ", this->hColor ) ;
   gse.erase( " when the music is right,", ZERO, true ) ;
   wp = this->dp->WriteParagraph ( wp, gse, this->dColor ) ;
   wp.xpos = 1 ;
   wp = this->dp->WriteString ( wp, "Cut range : ", this->hColor ) ;
   gse.erase( 14, 4 ) ;
   wp = this->dp->WriteString ( wp, gse, this->dColor ) ;
   ++wp.ypos ; wp.xpos = 1 ;
   gse = heat ;
   wp = this->dp->WriteString ( wp, "Basic Text: ", this->hColor ) ;
   wp = this->dp->WriteParagraph ( wp, gse, this->dColor ) ;
   wp.xpos = 1 ;
   wp = this->dp->WriteString ( wp, "Cut all   : ", this->hColor ) ;
   gse.erase( L"the ", ZERO, false, true ) ;
   wp = this->dp->WriteParagraph ( wp, gse, this->dColor ) ;
   ++wp.ypos ; wp.xpos = 1 ;

   //*********************************
   //*   Test the 'strip()' method   *
   //*********************************
   gString gss( "     This is a test.     " ) ;
   wp = dp->WriteParagraph ( wp, uSubHead[5], this->tColor ) ; // sub-heading
   wp = dp->WriteParagraph ( wp, "Strip ASCII whitespace: '", this->hColor ) ;
   wp = dp->WriteParagraph ( wp, gss, this->dColor ) ;
   wp = dp->WriteParagraph ( wp, "'  becomes:  '", this->hColor ) ;
   gss.strip() ;
   wp = dp->WriteParagraph ( wp, gss, this->dColor ) ;
   wp = dp->WriteParagraph ( wp, "'\n", this->hColor ) ;
   wp.xpos = 1 ;
   gss = L"\x3000\x3000\x3000这是一个测试。\x3000\x3000\x3000" ;
   wp = dp->WriteParagraph ( wp, "Strip  CJK  whitespace: '", this->hColor ) ;
   wp = dp->WriteParagraph ( wp, gss, this->dColor ) ;
   wp = dp->WriteParagraph ( wp, "' becomes:  '", this->hColor ) ;
   gss.strip() ;
   wp = dp->WriteParagraph ( wp, gss, this->dColor ) ;
   wp = dp->WriteParagraph ( wp, "'\n", this->hColor ) ;

   dp->RefreshWin () ;     // make everything visible

}  //* End gtBasicPage4() *

//*************************
//*     gtBasicPage5      *
//*************************
//******************************************************************************
//* Display a page of test output for gtBasic().                               *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void gStringTest::gtBasicPage5 ( void )
{
   const char* uSubHead[] = 
   {
      "Replace text with 'replace()' method :                               "
      "                                      \n",
      "Find index of substring with 'find()' method. Retrieve substring "
      "at specified index with 'substr()' method.\n",
      "Use the 'findr()' method to scan backward from the end of the string to "
      "find the \"truth\".\n",
   } ;
   const char* contentsHdr = "Contents of gString object being searched:\n" ;
   winPos   wp( 2, 1 ), wphead ;
   this->dp->ClearWin () ;    // erase old data from window

   //********************************
   //* Test the 'replace()' methods *
   //********************************
   wp = this->dp->WriteParagraph ( wp, uSubHead[0], this->tColor ) ; // sub-heading
   wp = this->dp->WriteString ( wp, "Basic Text                          : ", this->hColor ) ;
   gString gsr( "I live within that special place, That is defined by here embrace.\n" ) ;
   wp = this->dp->WriteParagraph ( wp, gsr, this->dColor ) ;
   wp.xpos = 1 ;
   wp = this->dp->WriteString ( wp, "Correct the typo ('here' to 'her')  : ", this->hColor ) ;
   gsr.replace( "here", "her" ) ;
   wp = this->dp->WriteParagraph ( wp, gsr, this->dColor ) ;
   wp.xpos = 1 ;
   wp = this->dp->WriteString ( wp, "Replace \"That\" (case sensitive)     : ", this->hColor ) ;
   gsr.replace( "That", "Which", ZERO, true ) ;
   wp = this->dp->WriteParagraph ( wp, gsr, this->dColor ) ;
   wp.xpos = 1 ;
   wp = this->dp->WriteString ( wp, "Replace \"That\" (case insensitive)   : ", this->hColor ) ;
   gsr.replace( "That", "a" ) ;
   wp = this->dp->WriteParagraph ( wp, gsr, this->dColor ) ;
   wp.xpos = 1 ;
   wp = this->dp->WriteString ( wp, "Define a more realistic living space: ", this->hColor ) ;
   gsr.replace( L" her embrace.", ": lack of income!", ZERO, false, true ) ;
   wp = this->dp->WriteParagraph ( wp, gsr, this->dColor ) ;
   wp.xpos = 1 ;
   wp = this->dp->WriteString ( wp, "Replace all spaces with underscores : ", this->hColor ) ;
   gsr.replace( L' ', L'_', ZERO, false, true ) ;
   wp = this->dp->WriteParagraph ( wp, gsr, this->dColor ) ;
   ++wp.ypos ; wp.xpos = 1 ;

   //****************************************************
   //* Test the 'find()' methods and 'substr()' methods *
   //****************************************************
   wp = this->dp->WriteParagraph ( wp, uSubHead[1], this->tColor ) ; // sub-heading
   
   wp = dp->WriteParagraph ( wp, contentsHdr, this->hColor ) ;

   const wchar_t* wsubText = L"Br" ;   // search string
   gString gsIn,                       // target buffer
           gsOut ;                     // output formatting
   const char*    usubText = "Br" ;
   gString gs( "Brown Brighton Beach brooms, blithely brush "
               "brightly-lit brownstone balconies for Doctor Bryant." ) ;
   dp->WriteString ( wp.ypos++, wp.xpos, gs, this->dColor ) ;
   dp->WriteString ( wp, "Case Insensitive search for words beginning with \"br\":", this->hColor ) ;
   ++wp.ypos ;

   short bi = ZERO, cnt = ZERO ;
   do //* Case-insensitive scan              *
   {  //* This loop uses a wide-text pattern *
      if ( (bi = gs.find( wsubText, bi )) >= ZERO )
      {
         short ei ;     // search for either SPACE or FULL STOP
         if ( (ei = gs.find( L' ', bi )) < ZERO )
            ei = gs.find( L'.', bi ) ;
         if ( ei >= ZERO )
         {
            ++cnt ;
            if ( (gs.substr( gsIn, bi, (ei - bi))) > ZERO )
            {
               gsOut.compose( "%hd) %S  ", &cnt, gsIn.gstr() ) ;
               wp = dp->WriteString ( wp, gsOut, this->dColor ) ;
            }
            else
               wp = dp->WriteString ( wp, "substr(ERR)  ", this->dColor ) ;
         }
         bi = ei ;
      }
   }
   while ( bi >= ZERO ) ;
   ++wp.ypos ;  wp.xpos = 1 ;

   dp->WriteString ( wp, "Case Sensitive search for words beginning with \"Br\":", this->hColor ) ;
   ++wp.ypos ;

   char csubIn[gsDFLTBYTES] ;    // target buffer (UTF-8)
   wchar_t wsubIn[gsALLOCDFLT] ; // target buffer (wchar_t)
   bi = cnt = ZERO ;
   do //* Case-sensitive scan                 *
   {  //* This loop uses a UTF-8 text pattern *
      if ( (bi = gs.find( usubText, bi, true )) >= ZERO )
      {
         short ei ;     // search for either SPACE or FULL STOP
         if ( (ei = gs.find( L' ', bi )) < ZERO )
            ei = gs.find( L'.', bi ) ;
         if ( ei >= ZERO )
         {
            ++cnt ;
            if ( (gs.substr( csubIn, bi, (ei - bi))) > ZERO )
            {
               gsOut.compose( "%hd) %s  ", &cnt, csubIn ) ;
               wp = dp->WriteString ( wp, gsOut, this->dColor ) ;
            }
            else
               wp = dp->WriteString ( wp, "substr(ERR)  ", this->dColor ) ;
         }
         bi = ei ;
      }
   }
   while ( bi >= ZERO ) ;
   ++wp.ypos ;  wp.xpos = 1 ;

   dp->WriteString ( wp, "Retrieve substring from \"Doctor\" to end of string:  "
                         "index=find(\"Doctor\");  then  substr(index, gsALLOCDFLT);", 
                     this->hColor ) ;
   ++wp.ypos ;
   bi = cnt = ZERO ;
   if ( (bi = gs.find( "Doctor" )) >= ZERO )
   {
      if ( (gs.substr( wsubIn, bi, gsALLOCDFLT)) > ZERO )
      {
         gsOut.compose( "'%S'", wsubIn ) ;
         wp = dp->WriteString ( wp, gsOut, this->dColor ) ;
      }
      else
         wp = dp->WriteString ( wp, "substr(ERR)  ", this->dColor ) ;
   }
   wp.ypos += 2 ;  wp.xpos = 1 ;

   //*******************************
   //*  Test the 'findr()' method  *
   //*******************************
   const char* truthString = 
         "When in doubt, find the truth by starting at the end and working backward.\n" ; 
   const wchar_t* wSearch = L"Truth" ;   // search string
   gs = truthString ;
   wp = dp->WriteParagraph ( wp, uSubHead[2], this->tColor ) ; // sub-heading
   wp = dp->WriteParagraph ( wp, contentsHdr, this->hColor ) ;
   wp = dp->WriteParagraph ( wp, gs, this->dColor ) ;
   bi = gs.findr( wSearch ) ;
   gs.insert( L"-->", bi ) ;
   wp = dp->WriteParagraph ( wp, gs, this->dColor ) ;

   dp->RefreshWin () ;     // make everything visible

}  //* End gtBasicPage5() *

//*************************
//*     gtBasicPage6      *
//*************************
//********************************************************************************
//* Display a page of test output for gtBasic().                                 *
//*                                                                              *
//* Input  : none                                                                *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************

void gStringTest::gtBasicPage6 ( void )
{
   const char* uSubHead[] = 
   {
      "Find index of next character which does not match using 'findx()' method.\n",
      "Find index of last occurance of substring with 'findlast()'.\n",
      "Step over whitespace (20h, 0Ah, 3000h) with 'scan()' method.\n",
   } ;
   const char* contentsHdr = "Contents of gString object being searched:\n" ;
   winPos   wp( 2, 1 ), wphead ;
   gString gs ;
   this->dp->ClearWin () ;    // erase old data from window

   //*******************************
   //*  Test the 'findx()' method  *
   //*******************************
   wp = dp->WriteParagraph ( wp, uSubHead[0], this->tColor ) ; // sub-heading
   gs = "        ----------Call of Duty - Black Ops III (Zombie Splat edition)" ;
   wp = dp->WriteParagraph ( wp, "Basic text               : '", this->hColor ) ;
   wp = dp->WriteString ( wp, gs, this->dColor ) ;
   wp = dp->WriteParagraph ( wp, "'\n", this->hColor ) ;
   wp.xpos = 1 ;
   short offset = gs.findx() ;
   wp = dp->WriteParagraph ( wp, "Scan over leading spaces : '", this->hColor ) ;
   wp = dp->WriteString ( wp, &gs.gstr()[offset], this->dColor ) ;
   wp = dp->WriteParagraph ( wp, "'\n", this->hColor ) ;
   wp.xpos = 1 ;
   offset = gs.findx( L'-', offset ) ;
   wp = dp->WriteParagraph ( wp, "Locate the title         : '", this->hColor ) ;
   wp = dp->WriteString ( wp, &gs.gstr()[offset], this->dColor ) ;
   wp = dp->WriteParagraph ( wp, "'\n\n", this->hColor ) ;
   wp.xpos = 1 ;

   //*********************************
   //* Test the 'findlast()' methods *
   //*********************************
   wp = this->dp->WriteParagraph ( wp, uSubHead[1], this->tColor ) ; // sub-heading
   wp = dp->WriteParagraph ( wp, contentsHdr, this->hColor ) ;
   gs = "ABC10  abc11  ABC12  abc13  ABC14  abc15  ABC16  abc17  ABC18  abc19 xyz20\n" ;
   wp = dp->WriteParagraph ( wp, gs, this->dColor ) ;

   wp = dp->WriteParagraph ( wp, "Find index of last \"ABC\" (case insensitive)\n", this->hColor ) ;
   short bi = gs.findlast( L"ABC" ) ;
   wp = dp->WriteParagraph ( wp, &gs.gstr()[bi], this->dColor ) ;
   
   wp = dp->WriteParagraph ( wp, "Find index of last \"ABC\" (case sensitive)\n", this->hColor ) ;
   bi = gs.findlast( L"ABC", true ) ;
   wp = dp->WriteParagraph ( wp, &gs.gstr()[bi], this->dColor ) ;
   ++wp.ypos ;

   wp = dp->WriteParagraph ( wp, contentsHdr, this->hColor ) ;
   gs = "~/SoftwareDesign/NcDialog/Dialogw/gStringTest.cpp\n" ;
   wp = dp->WriteParagraph ( wp, gs, this->dColor ) ;
   wp = dp->WriteParagraph ( wp, "Isolate filename in path: filename follows "
                            "last occurance of '/' character\n", this->hColor ) ;
   bi = gs.findlast( L'/' ) ;
   ++bi ;
   wp = dp->WriteParagraph ( wp, &gs.gstr()[bi], this->dColor ) ;
   ++wp.ypos ;

   //******************************
   //*  Test the 'scan()' method  *
   //******************************
   wp = dp->WriteParagraph ( wp, uSubHead[2], this->tColor ) ; // sub-heading
   gs = L"                 Hello World! \x3000    \n      Wassup?" ;
   wp = dp->WriteParagraph ( wp, "Basic text                   : '", this->hColor ) ;
   wp = dp->WriteString ( wp, gs, this->dColor ) ;
   wp = dp->WriteParagraph ( wp, "'\n", this->hColor ) ;
   wp.xpos = 1 ;
   offset = gs.scan() ;
   wp = dp->WriteParagraph ( wp, "Scan over leading spaces     : '", this->hColor ) ;
   wp = dp->WriteString ( wp, &gs.gstr()[offset], this->dColor ) ;
   wp = dp->WriteParagraph ( wp, "'\n", this->hColor ) ;
   wp.xpos = 1 ;
   offset = gs.after( L'!' ) ;
   offset = gs.scan( offset ) ;
   wp = dp->WriteParagraph ( wp, "Scan over complex whitespace : '", this->hColor ) ;
   wp = dp->WriteString ( wp, &gs.gstr()[offset], this->dColor ) ;
   wp = dp->WriteParagraph ( wp, "'\n\n", this->hColor ) ;
   wp.xpos = 1 ;

   dp->RefreshWin () ;     // make everything visible

}  //* End gtBasicPage6() *

//*************************
//*     gtBasicPage7      *
//*************************
//********************************************************************************
//* Display a page of test output for gtBasic().                                 *
//*                                                                              *
//* Input  : none                                                                *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************

void gStringTest::gtBasicPage7 ( void )
{
   const char* uSubHead[] = 
   {
      "Add padding characters to the data with 'padCols()'. \n",
      "Load specified number of characters with 'loadChars()'. \n",
      "Scan data and extract components using 'gscanf()' method.\n",
   } ;
   gString  gs ;
   winPos   wp( 2, 1 ), wphead ;
   this->dp->ClearWin () ;    // erase old data from window

   //*********************************
   //*  Test the 'padCols()' method  *
   //*********************************
   wp = dp->WriteParagraph ( wp, uSubHead[0], this->tColor ) ; // sub-heading
   gs = "This is a test." ;
   wp = dp->WriteParagraph ( wp, "Basic ASCII string  : '", this->hColor ) ;
   wp = dp->WriteParagraph ( wp, gs, this->dColor ) ;
   wp = dp->WriteParagraph ( wp, "'\n", this->hColor ) ;
   wp.xpos = 1 ;
   wp = dp->WriteParagraph ( wp, "     add ten columns: '", this->hColor ) ;
   gs.padCols( gs.gscols() + 10, L'#' ) ;
   wp = dp->WriteParagraph ( wp, gs, this->dColor ) ;
   wp = dp->WriteParagraph ( wp, "'\n", this->hColor ) ;
   wp.ypos -= 2 ; wp.xpos += 5 ;
   gs = "这是一个测试。" ;
   wp = dp->WriteParagraph ( wp, "Basic CJK string    : '", this->hColor ) ;
   wp = dp->WriteParagraph ( wp, gs, this->dColor ) ;
   wp = dp->WriteParagraph ( wp, "'\n", this->hColor ) ;
   wp.xpos = 54 ;
   wp = dp->WriteParagraph ( wp, " add fifteen columns: '", this->hColor ) ;
   gs.padCols( gs.gscols() + 15, L'$' ) ;
   wp = dp->WriteParagraph ( wp, gs, this->dColor ) ;
   wp = dp->WriteParagraph ( wp, "'\n", this->hColor ) ;
   wp.xpos = 1 ;

   wp = dp->WriteParagraph ( wp, "123456789-123456789-123456789-123456789-", this->hColor ) ;
   dp->WriteParagraph ( wp, "  Center text in field using padding.", this->hColor ) ;
   ++wp.ypos ;
   wp.xpos = 1 ;
   gs = "Even column split." ;
   gs.padCols( 40, L'#', true ) ;
   wp = dp->WriteParagraph ( wp, gs, this->dColor ) ;
   wp = dp->WriteParagraph ( wp, "  (leading pad columns == trailing pad columns)\n", this->hColor ) ;
   wp.xpos = 1 ;
   gs = "Odd column split." ;
   gs.padCols( 40, L'#', true ) ;
   wp = dp->WriteParagraph ( wp, gs, this->dColor ) ;
   wp = dp->WriteParagraph ( wp, "  (trailing pad contains the extra column)\n\n", this->hColor ) ;
   wp.xpos = 1 ;

   //***********************************
   //*  Test the 'LoadChars()' method  *
   //***********************************
   wp = dp->WriteParagraph ( wp, uSubHead[1], this->tColor ) ; // sub-heading
   gs = "When the going gets tough, " ;
   gString g2( "the tough go fishing.or they cut bait." ) ;
   wp = dp->WriteParagraph ( wp, "Existing Contents    : '", this->hColor ) ;
   wp = dp->WriteParagraph ( wp, gs, this->dColor ) ;
   wp = dp->WriteParagraph ( wp, "'\n", this->hColor ) ;
   wp.xpos = 1 ;
   wp = dp->WriteParagraph ( wp, "Add exactly 21 chars : '", this->hColor ) ;
   gs.loadChars( g2.gstr(), 21, true ) ;
   wp = dp->WriteParagraph ( wp, gs, this->dColor ) ;
   wp = dp->WriteParagraph ( wp, "'\n", this->hColor ) ;
   wp.xpos = 1 ;

   gs = "当道路被石头覆盖时，" ;     // "When the path is covered with stones,"
   g2 = "懦夫做借口，回国。什么烦鬼！" ;  // "the coward makes an excuse and returns home."
   wp = dp->WriteParagraph ( wp, "Existing Contents    : '", this->hColor ) ;
   wp = dp->WriteParagraph ( wp, gs, this->dColor ) ;
   wp = dp->WriteParagraph ( wp, "'\n", this->hColor ) ;
   wp.xpos = 1 ;
   wp = dp->WriteParagraph ( wp, "Add exactly 9 chars  : '", this->hColor ) ;
   gs.loadChars( g2.ustr(), 9, true ) ;
   wp = dp->WriteParagraph ( wp, gs, this->dColor ) ;
   wp = dp->WriteParagraph ( wp, "'\n", this->hColor ) ;
   wp.xpos = 1 ;
   wp = dp->WriteParagraph ( wp, "Replace existing text: '", this->hColor ) ;
   gs.loadChars( "When the path becomes rocky, the coward excuses himself and "
                 "turns toward home.wxyz", 78 ) ;
   wp = dp->WriteParagraph ( wp, gs, this->dColor ) ;
   wp = dp->WriteParagraph ( wp, "'\n\n", this->hColor ) ;
   wp.xpos = 1 ;

   //********************************
   //*  Test the 'gscanf()' method  *
   //********************************
   wp = dp->WriteParagraph ( wp, uSubHead[2], this->tColor ) ; // sub-heading
   gs = "17 % 18 21 22 48A2B 720451 24.325 "
        "A country song is three chords and the truth. - Willie Nelson\n" ;
   short sval1, sval2, sval3, sval4 ;
   int   ival ;
   long long int llval ;
   double dval ;
   char   str1[64], str2[16], str3[16], ch1 ;
   short cnt = gs.gscanf( L"%hd %% %hd %hd %hd %X %lld %lf %63[^-] %c %16s %16s", 
                          &sval1, &sval2, &sval3, &sval4, &ival, &llval, 
                          &dval, str1, &ch1, str2, str3 ) ;
   wp = dp->WriteParagraph ( wp, "Source   : ", this->hColor ) ;
   wp = dp->WriteParagraph ( wp, gs, dColor ) ;
   wp.xpos = 1 ;
   wp = dp->WriteParagraph ( wp, "Format   : \n"
                                 "Items    : \n"
                                 "Numeric  : \n"
                                 "Text Data: ", this->hColor ) ;
   wp.ypos -= 3 ;
   gString gsOut( "\"%%hd %%%% %%hd %%hd %%hd %%X %%lld %%lf %%63[^-] %%c %%16s %%16s\"\n"
                  "%hd\n"
                  "%hd  %hd  %hd  %hd  0x%04X  %lld  %4.6lf\n"
                  "\"%s\" %c %s %s\n",
                  &cnt, &sval1, &sval2, &sval3, &sval4, &ival, &llval, &dval, 
                  str1, &ch1, str2, str3 ) ;
   wp = dp->WriteParagraph ( wp, gsOut, this->dColor ) ;
   ++wp.ypos ;
   wp.xpos = 1 ;

// Insert additional tests here...


   dp->RefreshWin () ;     // make everything visible

}  //* End gtBasicPage7() *

//*************************
//*     gtBasicPage8      *
//*************************
//********************************************************************************
//* Display a page of test output for gtBasic().                                 *
//*                                                                              *
//* Input  : none                                                                *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************

void gStringTest::gtBasicPage8 ( void )
{
   const char* uSubHead[] = 
   {
      "Reverse character order with 'textReverse()'. (LTR and RTL languages, written both directions)             \n",
      "Reverse character order with 'textReverse()', but do not reverse terminal punctuation. \n",
      "Reverse multi-line text with 'textReverse()'. Menu in Arabic and English. (currency in Moroccan dirham) \n",
   } ;
   gString  gs ;
   winPos   wp( 2, 1 ), wphead ;
   this->dp->ClearWin () ;    // erase old data from window

   //************************************
   //* Test the 'textReverse()' method. *
   //************************************
   //* Control group: single-column LTR text. *
   const wchar_t* const oneColRev = L"Are you ready for lunch?" ;
   //* Control group: two-column LTR text. *
   const wchar_t* const twoColRev = L"你准备吃午饭了吗？" ;
   //* Yiddish RTL text. Char to be displayed on the right is at offset zero.  *
   const wchar_t* const YiddishRTL = L"?שטנָאל רַאֿפ טיירג ריא טנעז" ;
   //* Hebrew RTL text. Char to be displayed on the right is at offset zero.   *
   const wchar_t* const HebrewRTL = L"?םיירהצ תחוראל ןכומ התא םאה" ;
   //* Arabic RTL text. Char to be displayed on the right is at offset zero.   *
   const wchar_t* const ArabicRTL = L"هل انت جاهز للغداء؟" ;

   wp = dp->WriteParagraph ( wp, uSubHead[0], this->tColor ) ; // sub-heading
   wp.xpos = 36 ;
   dp->WriteString ( wp.ypos, wp.xpos - 6, " RTL ", this->hColor | ncuATTR ) ;
   dp->WriteString ( wp.ypos, wp.xpos + 2, " LTR ", this->hColor | ncuATTR ) ;
   ++wp.ypos ;
   gs =  L".\n.\n.\n.\n.\n.\n.\n.\n.\n." ;
   dp->WriteParagraph ( wp, gs, this->hColor ) ;
   gs = "Fwd:\nRev:\nFwd:\nRev:\nFwd:\nRev:\nFwd:\nRev:\nFwd:\nRev:" ;
   dp->WriteParagraph ( wp.ypos, 2, gs, this->hColor ) ;
   wp.xpos += 2 ;

   winPos wpBox( wp.ypos - 1, 67 ) ;
   dp->DrawBox ( wpBox.ypos, wpBox.xpos, 11, 41, this->bColor ) ;
   wpBox = { short(wpBox.ypos + 1), short(wpBox.xpos + 2) } ;
   dp->WriteParagraph ( wpBox, "Each sample is written four times:\n"
                               " 1) Canonical order as LTR\n"
                               " 2) Canonical order as RTL\n"
                               " 3) Reversed order as LTR\n"
                               " 4) Reversed order as RTL\n"
                               "LTR data are displayed correctly in\n"
                               " the first instance of the LTR column.\n"
                               "RTL data are displayed correctly in\n"
                               " the first instance of the RTL column.",
                               this->dColor ) ;

   gs = oneColRev ;
   dp->WriteString ( wp, gs, this->dColor, false, false ) ;
   wp.xpos -= 4 ;
   dp->WriteString ( wp, gs, this->dColor, false, true ) ;
   wp = { (short)(wp.ypos + 1), (short)(wp.xpos + 4) }  ;
   gs.textReverse() ;
   dp->WriteString ( wp, gs, this->dColor, false, false ) ;
   wp.xpos -= 4 ;
   dp->WriteString ( wp, gs, this->dColor, false, true ) ;
   wp = { (short)(wp.ypos + 1), (short)(wp.xpos + 4) }  ;

   gs = twoColRev ;
   dp->WriteString ( wp, gs, this->dColor, false, false ) ;
   wp.xpos -= 5 ;
   dp->WriteString ( wp, gs, this->dColor, false, true ) ;
   wp = { (short)(wp.ypos + 1), (short)(wp.xpos + 5) }  ;
   gs.textReverse() ;
   dp->WriteString ( wp, gs, this->dColor, false, false ) ;
   wp.xpos -= 5 ;
   dp->WriteString ( wp, gs, this->dColor, false, true ) ;
   wp = { (short)(wp.ypos + 1), (short)(wp.xpos + 5) }  ;

   gs = YiddishRTL ;
   dp->WriteString ( wp, gs, this->bColor, false, false ) ;
   wp.xpos -= 4 ;
   dp->WriteString ( wp, gs, this->bColor, false, true ) ;
   wp = { (short)(wp.ypos + 1), (short)(wp.xpos + 4) }  ;
   gs.textReverse() ;
   dp->WriteString ( wp, gs, this->bColor, false, false ) ;
   wp.xpos -= 4 ;
   dp->WriteString ( wp, gs, this->bColor, false, true ) ;
   wp = { (short)(wp.ypos + 1), (short)(wp.xpos + 4) }  ;

   gs = HebrewRTL ;
   dp->WriteString ( wp, gs, this->bColor, false, false ) ;
   wp.xpos -= 4 ;
   dp->WriteString ( wp, gs, this->bColor, false, true ) ;
   wp = { (short)(wp.ypos + 1), (short)(wp.xpos + 4) }  ;
   gs.textReverse() ;
   dp->WriteString ( wp, gs, this->bColor, false, false ) ;
   wp.xpos -= 4 ;
   dp->WriteString ( wp, gs, this->bColor, false, true ) ;
   wp = { (short)(wp.ypos + 1), (short)(wp.xpos + 4) }  ;

   gs = ArabicRTL ;
   dp->WriteString ( wp, gs, this->dColor, false, false ) ;
   wp.xpos -= 4 ;
   dp->WriteString ( wp, gs, this->dColor, false, true ) ;
   wp = { (short)(wp.ypos + 1), (short)(wp.xpos + 4) }  ;
   gs.textReverse() ;
   dp->WriteString ( wp, gs, this->dColor, false, false ) ;
   wp.xpos -= 4 ;
   dp->WriteString ( wp, gs, this->dColor, false, true ) ;
   wp.ypos += 2 ; wp.xpos = 1 ;

   //* Terminal punctuation not reversed. *
   wp = dp->WriteParagraph ( wp, uSubHead[1], this->tColor ) ; // sub-heading
   gs = "¿ozreumla le arap atsil sátsE?" ;
   wp = dp->WriteString ( wp.ypos, wp.xpos + 1, "As Written   : ", this->hColor ) ;
   dp->WriteString ( wp, gs, this->dColor ) ;
   wp = { (short)(wp.ypos + 1), 2 } ;
   gs.textReverse( true ) ;
   wp = dp->WriteString ( wp, "Text Reversed: ", this->hColor ) ;
   dp->WriteString ( wp, gs, this->dColor ) ;
   wp.ypos += 2 ; wp.xpos = 1 ;

   //* Display a Morrocan menu in both Arabic and English.*
   //* Exchange rate as of 2021 May: $1.00 == DM8.95      *
   wp = dp->WriteParagraph ( wp, uSubHead[2], this->tColor ) ; // sub-heading
   wp.xpos = this->dCols / 2 + 4 ;
   const wchar_t* const ArabicMenu = 
   L"كسكس بالدجاج\n"
    "كسكس مع لحم الضأن\n"
    "طاجن مانجو\n"
    "الحريرة\n"
    "صفوف\n" ;
   const wchar_t* const EnglishMenu = 
   {
      L"Couscous with chicken\n"
       "Couscous with lamb\n"
       "Mango tajine\n"
       "Harira\n"
       "Sfouff\n"
   } ;
   const float Prices[] = { 58.00, 80.50, 107.00, 62.00, 44.50 } ;
 
   //* Write RTL paragraph, left-justified *
   gs = ArabicMenu ;
   gs.textReverse( false, true ) ;                 // reverse the RTL text
   dp->WriteParagraph ( wp, gs, this->dColor ) ;   // and write it as LTR
   wp.xpos -= 9 ;

   //* Write the standard LTR numeric prices *
   gs.compose( L"%6.2fد\n%6.2fد\n%6.2fد\n%6.2fد\n%6.2fد",
                &Prices[0], &Prices[1], &Prices[2], &Prices[3], &Prices[4] ) ;
   dp->WriteParagraph ( wp, gs, this->dColor ) ;      // and write it as LTR
   wp.xpos -= 2 ;

   //* Write the LTR paragraph, right-justified *
   gs = EnglishMenu ;
   gs.textReverse( false, true, false ) ;          // reverse the LTR text
   wp = dp->WriteParagraph ( wp, gs, this->dColor, false, true ) ; // and write it as RTL
   wp.ypos += 1 ; wp.xpos = 1 ;

   dp->RefreshWin () ;     // make everything visible

}  //* End gtBasicPage8() *

//*************************
//*     gtBasicPage9      *
//*************************
//********************************************************************************
//* Display a page of test output for gtBasic().                                 *
//*                                                                              *
//* Input  : none                                                                *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************

void gStringTest::gtBasicPage9 ( void )
{
   const char* uSubHead[] = 
   {
      "Use formatParagraph() to reformat for display an unformatted text string "
      "to fit within a rectangular area. \n",
   } ;
   gString  gs ;
   winPos   wp( 2, 1 ), wphead ;
   this->dp->ClearWin () ;    // erase old data from window

   //****************************************
   //* Test the 'formatParagraph()' method. *
   //****************************************
   const char* unformData = 
      "What I’ve got no use for is a woman who won’t stand up for herself... \n"
      "You don’t wait to be told when to breathe... "
      "Be like a woman, not a petrified child. \n"
      "There are more that seventy earths spinning about the galaxy, "
      "and the meek have \ninherited not a one. "
      "-- Captain Reynolds, Firefly “Our Mrs. Reynolds” (deleted scene)" ;
   wp = dp->WriteParagraph ( wp, uSubHead[0], this->tColor ) ; // sub-heading
   wp = dp->WriteString ( wp, "Source text: ", this->hColor ) ;
   short boxr, boxc, boxh = 9, boxw = 51 ;
   gs = unformData ;
   wp = dp->WriteParagraph ( wp, gs, this->dColor ) ;
   wp.ypos += 3 ; wp.xpos = 1 ;
   wp = dp->WriteString ( wp, "Reformatted text:  ", this->hColor ) ;
   boxr = wp.ypos -1 ;
   boxc = wp.xpos ;
   dp->DrawBox ( boxr, boxc, boxh, boxw, this->hColor ) ;
   gs.formatParagraph( (boxh - 2), (boxw - 4) ) ;
   wp = { short(boxr + 1), short(boxc + 2) } ;
   wp = dp->WriteParagraph ( wp, gs, this->dColor ) ;

   const char* longTokens = 
      "This text contains a token which is wider than 50% of the specified "
      "column width for the Supercalifragilisticexpialidocious paragraph. "
      "The token is split with the insertion of an ndash." ;
   gs = longTokens ;
   boxc += boxw + 2 ;
   boxw = 33 ;
   dp->DrawBox ( boxr, boxc, boxh, boxw, this->hColor ) ;
   gs.formatParagraph( (boxh - 2), (boxw - 4) ) ;
   wp = { short(boxr + 1), short(boxc + 1) } ;
   wp = dp->WriteParagraph ( wp, gs, this->dColor ) ;
   wp.ypos += 3 ; wp.xpos = 1 ;

// Insert additional tests here...

   dp->RefreshWin () ;     // make everything visible

}  //* End gtBasic9() *

//*************************
//*     gtBasicPage10     *
//*************************
//********************************************************************************
//* Display a page of test output for gtBasic().                                 *
//*                                                                              *
//* Input  : none                                                                *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************

void gStringTest::gtBasicPage10 ( void )
{
   const char* uSubHead[] = 
   {
      "  Constructors for formatted integer fields and direct calls to 'formatInt()'  \n"
   } ;
   winPos   wp( 2, 2 ), wphead ;
   this->dp->ClearWin () ;    // erase old data from window
   wp = this->dp->WriteParagraph ( wp, uSubHead[0], this->tColor ) ; // sub-heading

   short ssize = sizeof(short),     // report integer sizes
         isize = sizeof(int),
         lsize = sizeof(long),
         qsize = sizeof(long long) ;
   gString gsOut ;

   //*******************************************************
   //* Test 'unsigned short' constructor and 'formatInt()' *
   //*******************************************************
   winPos wph = wp ;
   wph = this->dp->WriteString ( wph, "Integer Width", this->hColor | ncuATTR ) ;
   wph = this->dp->WriteString ( wph, "  ", this->dColor ) ;
   wph = this->dp->WriteString ( wph, "Full-width unsigned ", this->hColor | ncuATTR ) ;
   wph = this->dp->WriteString ( wph, "  ", this->dColor ) ;
   wph = this->dp->WriteString ( wph, "Full-width signed   ", this->hColor | ncuATTR ) ;
   wph = this->dp->WriteString ( wph, "  ", this->dColor ) ;
   wph = this->dp->WriteString ( wph, "Full-width force sign", this->hColor | ncuATTR ) ;
   wph = this->dp->WriteString ( wph, "  ", this->dColor ) ;
   wph = this->dp->WriteString ( wph, "Left-justified      ", this->hColor | ncuATTR ) ;
   ++wp.ypos ;
   gsOut.compose( "    short (%hd)\n"
                  "    short (%hd)\n"
                  "      int (%hd)\n"
                  "      int (%hd)\n"
                  "     long (%hd)\n"
                  "     long (%hd)\n"
                  "long long (%hd)\n"
                  "long long (%hd)",
                  &ssize, &ssize, &isize, &isize, &lsize, &lsize, &qsize, &qsize ) ;
   this->dp->WriteParagraph ( wp, gsOut, nc.grbl ) ;

   wp.xpos += 15 ;
   const short FWIDTH = FI_MAX_8BYTE_WIDTH ;

   //***************************
   //** Full-width - unsigned **
   //***************************
   unsigned short sVal = 999 ;
   gString gss( sVal, FWIDTH ) ;
   gsOut.compose( "'%S'\n", gss.gstr() ) ;
   wp = this->dp->WriteParagraph ( wp, gsOut, dColor ) ;

   sVal = 32768 ;
   gss.formatInt( sVal, FWIDTH ) ;
   gsOut.compose( "'%S'\n", gss.gstr() ) ;
   wp = this->dp->WriteParagraph ( wp, gsOut, dColor ) ;

   unsigned int iVal = 7888999 ;
   gString gsi( iVal, FWIDTH ) ;
   gsOut.compose( "'%S'\n", gsi.gstr() ) ;
   wp = dp->WriteParagraph ( wp, gsOut, dColor ) ;

   iVal = 2147483647 ;
   gsi.formatInt( iVal, FWIDTH ) ;
   gsOut.compose( "'%S'\n", gsi.gstr() ) ;
   wp = dp->WriteParagraph ( wp, gsOut, dColor ) ;

   unsigned long int lVal = 2111222333 ;
   gString gsl( lVal, FWIDTH ) ;
   gsOut.compose( "'%S'\n", gsl.gstr() ) ;
   wp = dp->WriteParagraph ( wp, gsOut, dColor ) ;

   if ( lsize == isize )   // sizeof(long) == sizeof(int)
      lVal = 2147483647 ;
   else                    // sizeof(long) > sizeof(int)
   {  // (This prevents compiler warnings on 32-bit systems)
      lVal = 999999999 ;
      lVal = (lVal * 10000) + 9999 ;
   }
   gsl.formatInt( lVal, FWIDTH ) ;
   gsOut.compose( "'%S'\n", gsl.gstr() ) ;
   wp = dp->WriteParagraph ( wp, gsOut, dColor ) ;

   unsigned long long int llVal = 999888777666 ;
   gString gsll( llVal, FWIDTH ) ;
   gsOut.compose( "'%S'\n", gsll.gstr() ) ;
   wp = dp->WriteParagraph ( wp, gsOut, dColor ) ;

   llVal = 9999999999999 ;
   gsll.formatInt( llVal, FWIDTH ) ;
   gsOut.compose( "'%S'\n", gsll.gstr() ) ;
   wp = dp->WriteParagraph ( wp, gsOut, dColor ) ;

   //***************************
   //** Full-width - signed   **
   //***************************
   wp = { 4, 39 } ;
   short ssVal = -999 ;
   gString gsss( ssVal, FWIDTH ) ;
   gsOut.compose( "'%S'\n", gsss.gstr() ) ;
   wp = dp->WriteParagraph ( wp, gsOut, dColor ) ;

   ssVal = -32768 ;
   gsss.formatInt( ssVal, FWIDTH ) ;
   gsOut.compose( "'%S'\n", gsss.gstr() ) ;
   wp = dp->WriteParagraph ( wp, gsOut, dColor ) ;

   int siVal = -7888999 ;
   gString gssi( siVal, FWIDTH ) ;
   gsOut.compose( "'%S'\n", gssi.gstr() ) ;
   wp = dp->WriteParagraph ( wp, gsOut, dColor ) ;

   siVal = -2147483647 ;
   gssi.formatInt( siVal, FWIDTH ) ;
   gsOut.compose( "'%S'\n", gssi.gstr() ) ;
   wp = dp->WriteParagraph ( wp, gsOut, dColor ) ;

   long slVal = -2111222333 ;
   gString gssl( slVal, FWIDTH ) ;
   gsOut.compose( "'%S'\n", gssl.gstr() ) ;
   wp = dp->WriteParagraph ( wp, gsOut, dColor ) ;

   if ( lsize == isize )   // sizeof(long) == sizeof(int)
      slVal = -2147483647 ;
   else                    // sizeof(long) > sizeof(int)
   {  // (This prevents compiler warnings on 32-bit systems)
      slVal = -999999999 ;
      slVal = (slVal * 10000) - 9999 ;
   }
   gssl.formatInt( slVal, FWIDTH ) ;
   gsOut.compose( "'%S'\n", gssl.gstr() ) ;
   wp = dp->WriteParagraph ( wp, gsOut, dColor ) ;

   long long int sllVal = -999888777666 ;
   gString gssll( sllVal, FWIDTH ) ;
   gsOut.compose( "'%S'\n", gssll.gstr() ) ;
   wp = dp->WriteParagraph ( wp, gsOut, dColor ) ;

   sllVal = -9999999999999 ;
   gssll.formatInt( sllVal, FWIDTH ) ;
   gsOut.compose( "'%S'\n", gssll.gstr() ) ;
   wp = dp->WriteParagraph ( wp, gsOut, dColor ) ;

   //*****************************
   //** Full-width - force sign **
   //*****************************
   wp = { 4, 61 } ;
   sVal = 999 ;
   gString gssp( sVal, FWIDTH, false, true ) ;
   gsOut.compose( "'%S'\n", gssp.gstr() ) ;
   wp = dp->WriteParagraph ( wp, gsOut, dColor ) ;

   sVal = 32768 ;
   gssp.formatInt( sVal, FWIDTH, false, true ) ;
   gsOut.compose( "'%S'\n", gssp.gstr() ) ;
   wp = dp->WriteParagraph ( wp, gsOut, dColor ) ;

   iVal = 7888999 ;
   gString gsip( iVal, FWIDTH, false, true ) ;
   gsOut.compose( "'%S'\n", gsip.gstr() ) ;
   wp = dp->WriteParagraph ( wp, gsOut, dColor ) ;

   iVal = 2147483647 ;
   gsip.formatInt( iVal, FWIDTH, false, true ) ;
   gsOut.compose( "'%S'\n", gsip.gstr() ) ;
   wp = dp->WriteParagraph ( wp, gsOut, dColor ) ;

   lVal = 2111222333 ;
   gString gslp( lVal, FWIDTH, false, true ) ;
   gsOut.compose( "'%S'\n", gslp.gstr() ) ;
   wp = dp->WriteParagraph ( wp, gsOut, dColor ) ;

   if ( lsize == isize )   // sizeof(long) == sizeof(int)
      lVal = 2147483647 ;
   else                    // sizeof(long) > sizeof(int)
   {  // (This prevents compiler warnings on 32-bit systems)
      lVal = 999999999 ;
      lVal = (lVal * 10000) + 9999 ;
   }
   gslp.formatInt( lVal, FWIDTH, false, true ) ;
   gsOut.compose( "'%S'\n", gslp.gstr() ) ;
   wp = dp->WriteParagraph ( wp, gsOut, dColor ) ;

   llVal = 999888777666 ;
   gString gsllp( llVal, FWIDTH, false, true ) ;
   gsOut.compose( "'%S'\n", gsllp.gstr() ) ;
   wp = dp->WriteParagraph ( wp, gsOut, dColor ) ;

   llVal = 9999999999999 ;
   gsllp.formatInt( llVal, FWIDTH, false, true ) ;
   gsOut.compose( "'%S'\n", gsllp.gstr() ) ;
   wp = dp->WriteParagraph ( wp, gsOut, dColor ) ;

   //*********************************
   //** Full-width - left-justified **
   //*********************************
   wp = { 4, 84 } ;
   sVal = 999 ;   // test justified constructor
   gString gssj( sVal, FWIDTH, true, true ) ;
   gsOut.compose( "'%S'\n", gssj.gstr() ) ;
   wp = dp->WriteParagraph ( wp, gsOut, dColor ) ;

   sVal = 32768 ;
   gss.formatInt( sVal, FWIDTH, true ) ;
   gsOut.compose( "'%S'\n", gss.gstr() ) ;
   wp = dp->WriteParagraph ( wp, gsOut, dColor ) ;

   iVal = 7888999 ;   // test justified constructor
   gString gsij( iVal, FWIDTH, true, true ) ;
   gsOut.compose( "'%S'\n", gsij.gstr() ) ;
   wp = dp->WriteParagraph ( wp, gsOut, dColor ) ;

   iVal = 2147483647 ;
   gsi.formatInt( iVal, FWIDTH, true ) ;
   gsOut.compose( "'%S'\n", gsi.gstr() ) ;
   wp = dp->WriteParagraph ( wp, gsOut, dColor ) ;

   lVal = 2111222333 ;   // test justified constructor
   gString gslj( lVal, FWIDTH, true, true ) ;
   gsOut.compose( "'%S'\n", gslj.gstr() ) ;
   wp = dp->WriteParagraph ( wp, gsOut, dColor ) ;

   if ( lsize == isize )   // sizeof(long) == sizeof(int)
      lVal = 2147483647 ;
   else                    // sizeof(long) > sizeof(int)
   {  // (This prevents compiler warnings on 32-bit systems)
      lVal = 999999999 ;
      lVal = (lVal * 10000) + 9999 ;
   }
   gsl.formatInt( lVal, FWIDTH, true ) ;
   gsOut.compose( "'%S'\n", gsl.gstr() ) ;
   wp = dp->WriteParagraph ( wp, gsOut, dColor ) ;

   llVal = 999888777666 ;   // test justified constructor
   gString gsllj( llVal, FWIDTH, true, true ) ;
   gsOut.compose( "'%S'\n", gsllj.gstr() ) ;
   wp = dp->WriteParagraph ( wp, gsOut, dColor ) ;

   llVal = 9999999999999 ;
   gsll.formatInt( llVal, FWIDTH, true ) ;
   gsOut.compose( "'%S'\n", gsll.gstr() ) ;
   wp = dp->WriteParagraph ( wp, gsOut, dColor ) ;

   //*************************************
   //** Rounding for three-column field **
   //*************************************
   ++wp.ypos ;
   wp.xpos = 2 ;
   short fWidth = 3 ;
   wp = dp->WriteParagraph ( wp, "Rounding Test for three-column fields:\n", 
                             this->hColor | ncuATTR ) ;
   //**********************
   //** Unsigned Integer **
   //**********************
   iVal = 750 ;
   gsi.formatInt( iVal, fWidth ) ;
   gsOut.compose( L"'%S'  (%'u)\n", gsi.gstr(), &iVal ) ;
   wp = dp->WriteParagraph ( wp, gsOut, dColor ) ;
   iVal = 999 ;
   gsi.formatInt( iVal, fWidth ) ;
   gsOut.compose( L"'%S'  (%'u)\n", gsi.gstr(), &iVal ) ;
   wp = dp->WriteParagraph ( wp, gsOut, dColor ) ;
   iVal = 7500 ;
   gsi.formatInt( iVal, fWidth ) ;
   gsOut.compose( L"'%S'  (%'u)\n", gsi.gstr(), &iVal ) ;
   wp = dp->WriteParagraph ( wp, gsOut, dColor ) ;
   iVal = 7501 ;
   gsi.formatInt( iVal, fWidth ) ;
   gsOut.compose( L"'%S'  (%'u)\n", gsi.gstr(), &iVal ) ;
   wp = dp->WriteParagraph ( wp, gsOut, dColor ) ;
   iVal = 75500 ;
   gsi.formatInt( iVal, fWidth ) ;
   gsOut.compose( L"'%S'  (%'u)\n", gsi.gstr(), &iVal ) ;
   wp = dp->WriteParagraph ( wp, gsOut, dColor ) ;
   iVal = 75501 ;
   gsi.formatInt( iVal, fWidth ) ;
   gsOut.compose( L"'%S'  (%'u)\n", gsi.gstr(), &iVal ) ;
   wp = dp->WriteParagraph ( wp, gsOut, dColor ) ;
   iVal = 99500 ;
   gsi.formatInt( iVal, fWidth ) ;
   gsOut.compose( L"'%S'  (%'u)\n", gsi.gstr(), &iVal ) ;
   wp = dp->WriteParagraph ( wp, gsOut, dColor ) ;
   iVal = 99501 ;
   gsi.formatInt( iVal, fWidth ) ;
   gsOut.compose( L"'%S'  (%'u)\n", gsi.gstr(), &iVal ) ;
   wp = dp->WriteParagraph ( wp, gsOut, dColor ) ;

   //**********************
   //** Signed Integer   **
   //**********************
   siVal = -75 ;
   gsi.formatInt( siVal, fWidth ) ;
   gsOut.compose( L"'%S'  (%'d)\n", gsi.gstr(), &siVal ) ;
   wp = dp->WriteParagraph ( wp, gsOut, dColor ) ;
   siVal = -950 ;
   gsi.formatInt( siVal, fWidth ) ;
   gsOut.compose( L"'%S'  (%'d)\n", gsi.gstr(), &siVal ) ;
   wp = dp->WriteParagraph ( wp, gsOut, dColor ) ;
   siVal = -951 ;
   gsi.formatInt( siVal, fWidth ) ;
   gsOut.compose( L"'%S'  (%'d)\n", gsi.gstr(), &siVal ) ;
   wp = dp->WriteParagraph ( wp, gsOut, dColor ) ;
   siVal = -7500 ;
   gsi.formatInt( siVal, fWidth ) ;
   gsOut.compose( L"'%S'  (%'d)\n", gsi.gstr(), &siVal ) ;
   wp = dp->WriteParagraph ( wp, gsOut, dColor ) ;
   siVal = -7501 ;
   gsi.formatInt( siVal, fWidth ) ;
   gsOut.compose( L"'%S'  (%'d)\n", gsi.gstr(), &siVal ) ;
   wp = dp->WriteParagraph ( wp, gsOut, dColor ) ;
   siVal = -9500 ;
   gsi.formatInt( siVal, fWidth ) ;
   gsOut.compose( L"'%S'  (%'d)\n", gsi.gstr(), &siVal ) ;
   wp = dp->WriteParagraph ( wp, gsOut, dColor ) ;
   siVal = -9501 ;
   gsi.formatInt( siVal, fWidth ) ;
   gsOut.compose( L"'%S'  (%'d)\n", gsi.gstr(), &siVal ) ;
   wp = dp->WriteParagraph ( wp, gsOut, dColor ) ;

   //***************************
   //** Unsigned Long Integer **
   //***************************
   wp = { 14, 20 } ;
   lVal = 950000 ;
   gsi.formatInt( lVal, fWidth ) ;
   gsOut.compose( L"'%S'  (%'lu)\n", gsi.gstr(), &lVal ) ;
   wp = dp->WriteParagraph ( wp, gsOut, dColor ) ;
   lVal = 950001 ;
   gsi.formatInt( lVal, fWidth ) ;
   gsOut.compose( L"'%S'  (%'lu)\n", gsi.gstr(), &lVal ) ;
   wp = dp->WriteParagraph ( wp, gsOut, dColor ) ;
   lVal = 7500000 ;
   gsi.formatInt( lVal, fWidth ) ;
   gsOut.compose( L"'%S'  (%'lu)\n", gsi.gstr(), &lVal ) ;
   wp = dp->WriteParagraph ( wp, gsOut, dColor ) ;
   lVal = 7500001 ;
   gsi.formatInt( lVal, fWidth ) ;
   gsOut.compose( L"'%S'  (%'lu)\n", gsi.gstr(), &lVal ) ;
   wp = dp->WriteParagraph ( wp, gsOut, dColor ) ;
   lVal = 75500000 ;
   gsi.formatInt( lVal, fWidth ) ;
   gsOut.compose( L"'%S'  (%'lu)\n", gsi.gstr(), &lVal ) ;
   wp = dp->WriteParagraph ( wp, gsOut, dColor ) ;
   lVal = 75500001 ;
   gsi.formatInt( lVal, fWidth ) ;
   gsOut.compose( L"'%S'  (%'lu)\n", gsi.gstr(), &lVal ) ;
   wp = dp->WriteParagraph ( wp, gsOut, dColor ) ;
   lVal = 99500000 ;
   gsi.formatInt( lVal, fWidth ) ;
   gsOut.compose( L"'%S'  (%'lu)\n", gsi.gstr(), &lVal ) ;
   wp = dp->WriteParagraph ( wp, gsOut, dColor ) ;
   lVal = 99500001 ;
   gsi.formatInt( lVal, fWidth ) ;
   gsOut.compose( L"'%S'  (%'lu)\n", gsi.gstr(), &lVal ) ;
   wp = dp->WriteParagraph ( wp, gsOut, dColor ) ;

   //***************************
   //** Signed Long Integer   **
   //***************************
   ++wp.ypos ;
   slVal = -950000 ;
   gsi.formatInt( slVal, fWidth ) ;
   gsOut.compose( L"'%S'  (%'ld)\n", gsi.gstr(), &slVal ) ;
   wp = dp->WriteParagraph ( wp, gsOut, dColor ) ;
   slVal = -950001 ;
   gsi.formatInt( slVal, fWidth ) ;
   gsOut.compose( L"'%S'  (%'ld)\n", gsi.gstr(), &slVal ) ;
   wp = dp->WriteParagraph ( wp, gsOut, dColor ) ;
   slVal = -7500000 ;
   gsi.formatInt( slVal, fWidth ) ;
   gsOut.compose( L"'%S'  (%'ld)\n", gsi.gstr(), &slVal ) ;
   wp = dp->WriteParagraph ( wp, gsOut, dColor ) ;
   slVal = -7500001 ;
   gsi.formatInt( slVal, fWidth ) ;
   gsOut.compose( L"'%S'  (%'ld)\n", gsi.gstr(), &slVal ) ;
   wp = dp->WriteParagraph ( wp, gsOut, dColor ) ;
   slVal = -9500000 ;
   gsi.formatInt( slVal, fWidth ) ;
   gsOut.compose( L"'%S'  (%'ld)\n", gsi.gstr(), &slVal ) ;
   wp = dp->WriteParagraph ( wp, gsOut, dColor ) ;
   slVal = -9500001 ;
   gsi.formatInt( slVal, fWidth ) ;
   gsOut.compose( L"'%S'  (%'ld)\n", gsi.gstr(), &slVal ) ;
   wp = dp->WriteParagraph ( wp, gsOut, dColor ) ;

   //********************************
   //** Unsigned Long Long Integer **
   //********************************
   wp = { 14, 42 } ;
   unsigned long long int qVal = 950000000 ;
   gsi.formatInt( qVal, fWidth ) ;
   gsOut.compose( L"'%S'  (%'lu)\n", gsi.gstr(), &qVal ) ;
   wp = dp->WriteParagraph ( wp, gsOut, dColor ) ;
   qVal = 950000001 ;
   gsi.formatInt( qVal, fWidth ) ;
   gsOut.compose( L"'%S'  (%'lu)\n", gsi.gstr(), &qVal ) ;
   wp = dp->WriteParagraph ( wp, gsOut, dColor ) ;
   qVal = 7500000000 ;
   gsi.formatInt( qVal, fWidth ) ;
   gsOut.compose( L"'%S'  (%'llu)\n", gsi.gstr(), &qVal ) ;
   wp = dp->WriteParagraph ( wp, gsOut, dColor ) ;
   qVal = 7500000001 ;
   gsi.formatInt( qVal, fWidth ) ;
   gsOut.compose( L"'%S'  (%'llu)\n", gsi.gstr(), &qVal ) ;
   wp = dp->WriteParagraph ( wp, gsOut, dColor ) ;
   qVal = 75500000000 ;
   gsi.formatInt( qVal, fWidth ) ;
   gsOut.compose( L"'%S'  (%'llu)\n", gsi.gstr(), &qVal ) ;
   wp = dp->WriteParagraph ( wp, gsOut, dColor ) ;
   qVal = 75500000001 ;
   gsi.formatInt( qVal, fWidth ) ;
   gsOut.compose( L"'%S'  (%'llu)\n", gsi.gstr(), &qVal ) ;
   wp = dp->WriteParagraph ( wp, gsOut, dColor ) ;
   qVal = 99500000000 ;
   gsi.formatInt( qVal, fWidth ) ;
   gsOut.compose( L"'%S'  (%'llu)\n", gsi.gstr(), &qVal ) ;
   wp = dp->WriteParagraph ( wp, gsOut, dColor ) ;
   qVal = 99500000001 ;
   gsi.formatInt( qVal, fWidth ) ;
   gsOut.compose( L"'%S'  (%'llu)\n", gsi.gstr(), &qVal ) ;
   wp = dp->WriteParagraph ( wp, gsOut, dColor ) ;

   //********************************
   //** Signed Long Long Integer   **
   //********************************
   ++wp.ypos ;
   long long int sqVal = -950000000 ;
   gsi.formatInt( sqVal, fWidth ) ;
   gsOut.compose( L"'%S'  (%'lld)\n", gsi.gstr(), &sqVal ) ;
   wp = dp->WriteParagraph ( wp, gsOut, dColor ) ;
   sqVal = -950000001 ;
   gsi.formatInt( sqVal, fWidth ) ;
   gsOut.compose( L"'%S'  (%'lld)\n", gsi.gstr(), &sqVal ) ;
   wp = dp->WriteParagraph ( wp, gsOut, dColor ) ;
   sqVal = -7500000000 ;
   gsi.formatInt( sqVal, fWidth ) ;
   gsOut.compose( L"'%S'  (%'lld)\n", gsi.gstr(), &sqVal ) ;
   wp = dp->WriteParagraph ( wp, gsOut, dColor ) ;
   sqVal = -7500000001 ;
   gsi.formatInt( sqVal, fWidth ) ;
   gsOut.compose( L"'%S'  (%'lld)\n", gsi.gstr(), &sqVal ) ;
   wp = dp->WriteParagraph ( wp, gsOut, dColor ) ;
   sqVal = -9500000000 ;
   gsi.formatInt( sqVal, fWidth ) ;
   gsOut.compose( L"'%S'  (%'lld)\n", gsi.gstr(), &sqVal ) ;
   wp = dp->WriteParagraph ( wp, gsOut, dColor ) ;
   sqVal = -9500000001 ;
   gsi.formatInt( sqVal, fWidth ) ;
   gsOut.compose( L"'%S'  (%'lld)\n", gsi.gstr(), &sqVal ) ;
   wp = dp->WriteParagraph ( wp, gsOut, dColor ) ;

   //*********************
   //** Terabyte Values **
   //*********************
   wp = { 14, 68 } ;
   qVal = 950000000000 ;      // 950.0 Gbytes
   gsi.formatInt( qVal, fWidth ) ;
   gsOut.compose( L"'%S'  (%'llu)\n", gsi.gstr(), &qVal ) ;
   wp = dp->WriteParagraph ( wp, gsOut, dColor ) ;
   qVal = 950000000001 ;      // > 950.0 Gbytes
   gsi.formatInt( qVal, fWidth ) ;
   gsOut.compose( L"'%S'  (%'llu)\n", gsi.gstr(), &qVal ) ;
   wp = dp->WriteParagraph ( wp, gsOut, dColor ) ;
   qVal = 7500000000000 ;
   gsi.formatInt( qVal, fWidth ) ;
   gsOut.compose( L"'%S'  (%'llu)\n", gsi.gstr(), &qVal ) ;
   wp = dp->WriteParagraph ( wp, gsOut, dColor ) ;
   qVal = 7500000000001 ;
   gsi.formatInt( qVal, fWidth ) ;
   gsOut.compose( L"'%S'  (%'llu)\n", gsi.gstr(), &qVal ) ;
   wp = dp->WriteParagraph ( wp, gsOut, dColor ) ;
   qVal = 9500000000000 ;
   gsi.formatInt( qVal, fWidth ) ;
   gsOut.compose( L"'%S'  (%'llu)\n", gsi.gstr(), &qVal ) ;
   wp = dp->WriteParagraph ( wp, gsOut, dColor ) ;
   qVal = 9500000000001 ;
   gsi.formatInt( qVal, fWidth ) ;
   gsOut.compose( L"'%S'  (%'llu)\n", gsi.gstr(), &qVal ) ;
   wp = dp->WriteParagraph ( wp, gsOut, dColor ) ;

   wp.ypos += 3 ;
   sqVal = -950000000000 ;      // -999.5 Gbytes
   gsi.formatInt( sqVal, fWidth ) ;
   gsOut.compose( L"'%S'  (%'lld)\n", gsi.gstr(), &sqVal ) ;
   wp = dp->WriteParagraph ( wp, gsOut, dColor ) ;
   sqVal = -950000000001 ;      // < -999.5 Gbytes
   gsi.formatInt( sqVal, fWidth ) ;
   gsOut.compose( L"'%S'  (%'lld)\n", gsi.gstr(), &sqVal ) ;
   wp = dp->WriteParagraph ( wp, gsOut, dColor ) ;
   sqVal = -7500000000000 ;
   gsi.formatInt( sqVal, fWidth ) ;
   gsOut.compose( L"'%S'  (%'lld)\n", gsi.gstr(), &sqVal ) ;
   wp = dp->WriteParagraph ( wp, gsOut, dColor ) ;
   sqVal = -7500000000001 ;
   gsi.formatInt( sqVal, fWidth ) ;
   gsOut.compose( L"'%S'  (%'lld)\n", gsi.gstr(), &sqVal ) ;
   wp = dp->WriteParagraph ( wp, gsOut, dColor ) ;
   sqVal = -9500000000000 ;
   gsi.formatInt( sqVal, fWidth ) ;
   gsOut.compose( L"'%S'  (%'lld)\n", gsi.gstr(), &sqVal ) ;
   wp = dp->WriteParagraph ( wp, gsOut, dColor ) ;
   sqVal = -9500000000001 ;
   gsi.formatInt( sqVal, fWidth ) ;
   gsOut.compose( L"'%S'  (%'lld)\n", gsi.gstr(), &sqVal ) ;
   wp = dp->WriteParagraph ( wp, gsOut, dColor ) ;

   this->dp->RefreshWin () ;     // make everything visible

}  //* End gtBasicPage10() *

//*************************
//*    gtBasicPage11      *
//*************************
//********************************************************************************
//* Display a page of test output for gtBasic().                                 *
//*                                                                              *
//* Input  : none                                                                *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************

void gStringTest::gtBasicPage11 ( void )
{
   const char* uSubHead[] = 
   {
      "  Continued testing for formatted-integer fields: 'formatInt()'  \n"
   } ;
   winPos   wp( 2, 2 ), wphead ;
   this->dp->ClearWin () ;    // erase old data from window
   wp = this->dp->WriteParagraph ( wp, uSubHead[0], this->tColor ) ; // sub-heading

   winPos wph = wp ;
   wph = this->dp->WriteString ( wph, "COLS kBytes   ", this->hColor | ncuATTR ) ;
   wph = this->dp->WriteString ( wph, "  ", this->dColor ) ;
   wph = this->dp->WriteString ( wph, "COLS mBytes       ", this->hColor | ncuATTR ) ;
   wph = this->dp->WriteString ( wph, "  ", this->dColor ) ;
   wph = this->dp->WriteString ( wph, "COLS mBytes       ", this->hColor | ncuATTR ) ;
   wph = this->dp->WriteString ( wph, "  ", this->dColor ) ;
   wph = this->dp->WriteString ( wph, "COLS +gBytes          ", this->hColor | ncuATTR ) ;
   wph = this->dp->WriteString ( wph, "  ", this->dColor ) ;
   wph = this->dp->WriteString ( wph, "COLS -gBytes          ", this->hColor | ncuATTR ) ;
   ++wp.ypos ;

   gString gsOut, gsi ;
   //********************************
   //** Unsigned Int - Kbyte range **
   //********************************
   unsigned int iVal = 654521 ;
   short fWidth = 9 ;
   for ( short i = ZERO ; i < 9 ; ++i )
   {
      gsi.formatInt( iVal, fWidth ) ;
      gsOut.compose( "%hd) '%S'\n", &fWidth, gsi.gstr() ) ;
      wp = this->dp->WriteParagraph ( wp, gsOut, this->dColor ) ;
      --fWidth ;
   }

   //********************************
   //** Signed Int - Kbyte range   **
   //********************************
   ++wp.ypos ;
   fWidth = 9 ;
   int niVal = -999876 ;
   for ( short i = ZERO ; i < 9 ; ++i )
   {
      gsi.formatInt( niVal, fWidth ) ;
      gsOut.compose( "%hd) '%S'\n", &fWidth, gsi.gstr() ) ;
      wp = this->dp->WriteParagraph ( wp, gsOut, this->dColor ) ;
      --fWidth ;
   }

   //*****************************************
   //** Unsigned Long Int - mid-MByte range **
   //*****************************************
   wp = { 4, short(wp.xpos + 15) } ;
   unsigned long lVal = 123654321 ;
   fWidth = 12 ;
   for ( short i = ZERO ; i < 12 ; ++i )
   {
      gsi.formatInt( lVal, fWidth ) ;
      gsOut.compose( "% 3hd) '%S'\n", &fWidth, gsi.gstr() ) ;
      wp = dp->WriteParagraph ( wp, gsOut, dColor ) ;
      --fWidth ;
   }

   //*****************************************
   //** Signed Long Int - mid-MByte range   **
   //*****************************************
   ++wp.ypos ;
   fWidth = 12 ;
   int nlVal = -123654321 ;
   for ( short i = ZERO ; i < 12 ; ++i )
   {
      gsi.formatInt( nlVal, fWidth ) ;
      gsOut.compose( "% 3hd) '%S'\n", &fWidth, gsi.gstr() ) ;
      wp = dp->WriteParagraph ( wp, gsOut, dColor ) ;
      --fWidth ;
   }

   //*****************************************
   //** Unsigned Long Int - max-MByte range **
   //*****************************************
   wp = { 4, short(wp.xpos + 20) } ;
   lVal = 999987876 ;
   fWidth = 12 ;
   for ( short i = ZERO ; i < 12 ; ++i )
   {
      gsi.formatInt( lVal, fWidth ) ;
      gsOut.compose( "% 3hd) '%S'\n", &fWidth, gsi.gstr() ) ;
      wp = dp->WriteParagraph ( wp, gsOut, dColor ) ;
      --fWidth ;
   }

   //*****************************************
   //** Signed Long Int - max-MByte range   **
   //*****************************************
   ++wp.ypos ;
   fWidth = 12 ;
   nlVal = -999876543 ;
   for ( short i = ZERO ; i < 12 ; ++i )
   {
      gsi.formatInt( nlVal, fWidth ) ;
      gsOut.compose( "% 3hd) '%S'\n", &fWidth, gsi.gstr() ) ;
      wp = dp->WriteParagraph ( wp, gsOut, dColor ) ;
      --fWidth ;
   }

   //*******************************************
   //** Unsigned Long Long Int - +GByte range **
   //*******************************************
   wp = { 4, short(wp.xpos + 20) } ;
   unsigned long long qVal = 999987654321 ;
   fWidth = 16 ;
   for ( short i = ZERO ; i < 15 ; ++i )
   {
      gsi.formatInt( qVal, fWidth ) ;
      gsOut.compose( "% 3hd) '%S'\n", &fWidth, gsi.gstr() ) ;
      wp = dp->WriteParagraph ( wp, gsOut, dColor ) ;
      --fWidth ;
   }

   //*******************************************
   //** Signed Long Long Int - -GByte range   **
   //*******************************************
   wp = { 4, short(wp.xpos + 24) } ;
   long long sqVal = -999987654321 ;
   fWidth = 16 ;
   for ( short i = ZERO ; i < 15 ; ++i )
   {
      gsi.formatInt( sqVal, fWidth ) ;
      gsOut.compose( "% 3hd) '%S'\n", &fWidth, gsi.gstr() ) ;
      wp = dp->WriteParagraph ( wp, gsOut, dColor ) ;
      --fWidth ;
   }

   //*********************
   //** Terabyte values **
   //*********************
   wp = { 20, 54 } ;
   dp->WriteParagraph ( (wp.ypos - 1), (wp.xpos + 1),
                        "COLS +tBytes      COLS -tBytes      COLS more tBytes", 
                        this->hColor | ncuATTR ) ;
   qVal = 9855432100000 ;
   fWidth = 10 ;
   for ( short i = ZERO ; i < 9 ; ++i )
   {
      gsi.formatInt( qVal, fWidth ) ;
      gsOut.compose( "% 3hd) '%S'\n", &fWidth, gsi.gstr() ) ;
      wp = dp->WriteParagraph ( wp, gsOut, dColor ) ;
      --fWidth ;
   }

   wp = { 20, short(wp.xpos + 18) } ;
   sqVal = -9855432100000 ;
   fWidth = 10 ;
   for ( short i = ZERO ; i < 9 ; ++i )
   {
      gsi.formatInt( sqVal, fWidth ) ;
      gsOut.compose( "% 3hd) '%S'\n", &fWidth, gsi.gstr() ) ;
      wp = dp->WriteParagraph ( wp, gsOut, dColor ) ;
      --fWidth ;
   }

   wp = { 20, short(wp.xpos + 17) } ;
   qVal = 8765432100000 ;
   fWidth = 6 ;
   for ( short i = ZERO ; i < 5 ; ++i )
   {
      gsi.formatInt( qVal, fWidth ) ;
      gsOut.compose( "% 3hd) '%S'\n", &fWidth, gsi.gstr() ) ;
      wp = dp->WriteParagraph ( wp, gsOut, dColor ) ;
      --fWidth ;
   }
   sqVal = -8765432100000 ;
   fWidth = 6 ;
   for ( short i = ZERO ; i < 5 ; ++i )
   {
      gsi.formatInt( sqVal, fWidth ) ;
      gsOut.compose( "% 3hd) '%S'\n", &fWidth, gsi.gstr() ) ;
      wp = dp->WriteParagraph ( wp, gsOut, dColor ) ;
      --fWidth ;
   }

   this->dp->RefreshWin () ;     // make everything visible

}  //* End gtBasicPage11() *

//*************************
//*     gtBasicStats      *
//*************************
//******************************************************************************
//* Gather and display statistics on the specified gString object.             *
//*                                                                            *
//* Input  : gs : (by reference) gString object on which to report             *
//*          wp : cursor position at which to write statistics                 *
//*          dColor: text color attribute                                      *
//*                                                                            *
//* Returns: current cursor position                                           *
//******************************************************************************

winPos gStringTest::gtBasicStats ( gString& gs, winPos& wp, attr_t dColor )
{
short    gcols = gs.gscols(),       // columns needed for display
         gchars = gs.gschars(),     // characters to be displayed
         ubytes = gs.utfbytes() ;   // bytes in UTF-8 array
bool     isAscii = gs.isASCII() ;   // 'true' if ASCII string

   gString fgs( "%6hd  %7hd  %8hd    %s", 
                &gcols, &gchars, &ubytes, (isAscii ? "yes" : " no") ) ;
//   gsForm gsf { L"%6hd  %7hd  %8hd    %s",
//                { &gcols, &gchars, &ubytes, (isAscii ? "yes" : " no") },
//              } ;
//   gString fgs( gsf ) ;
   this->dp->WriteString ( wp.ypos++, wp.xpos, fgs, this->dColor ) ;
   wp.xpos = 1 ;
   return wp ;
   
}  //* End gtBasicStats() *

//*************************
//*     gtOpenIntFmt      *
//*************************
//******************************************************************************
//* Open the 'integer-formatting' dialog window.                               *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: 'true' if dialog window opened successfully, else 'false'         *
//******************************************************************************

bool gStringTest::gtOpenIntFmt ( void )
{
   short ctrX    = this->termCols/2,         // dialog center in X
         ulY     = 1,                        // upper left corner in Y
         ulX     = ctrX - intfmtCOLS / 2 ;   // upper left corner in X
   bool  success = false ;

   //* Initial parameters for dialog window *
   this->icPtr = intfmtIc ;
   this->dRows = intfmtROWS ;
   this->dCols = intfmtCOLS ;
   InitNcDialog dInit( this->dRows,    // number of display lines
                       this->dCols,    // number of display columns
                       ulY,            // Y offset from upper-left of terminal 
                       ulX,            // X offset from upper-left of terminal 
                       NULL,           // dialog title
                       ncltSINGLE,     // border line-style
                       this->bColor,   // border color attribute
                       this->dColor,   // interior color attribute
                       this->icPtr     // pointer to list of control definitions
                     ) ;

   //* Instantiate the dialog window *
   this->dp = new NcDialog ( dInit ) ;

   //* Open the dialog window *
   if ( (this->dp->OpenWindow()) == OK )
   {
      this->dp->SetDialogTitle ( 
            "  gString-class Fixed-width Integer Formatting  ", this->tColor ) ;

      //* Group radio buttons *
      short XorGroup[] = { iRJustRB, iLJustRB, -1 } ;
      dp->GroupRadiobuttons ( XorGroup ) ;
      XorGroup[0] = iBit32RB ; XorGroup[1] = iBit64RB ;
      dp->GroupRadiobuttons ( XorGroup ) ;
      XorGroup[0] = iDeciRB ; XorGroup[1] = iKibiRB ;
      dp->GroupRadiobuttons ( XorGroup ) ;

      this->dp->RefreshWin () ;
      success = true ;
   }
   return success ;

}  //* End gtOpenIntFmt() *

//*************************
//*      gtIntFmtUI       *
//*************************
//******************************************************************************
//* PRIVATE METHOD: 'Integer Format Test' user interface.                      *
//* Dialog was opened by caller.                                               *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void gStringTest::gtIntFmtUI ( void )
{
const short COLCNT = 13 ;
const short MAX_NUMS = 10 ;         // entries per column
const UINT64 oneGB  = (1000000000),
             nineGB = (oneGB * 9),
             tenGB  = (oneGB * 10),
             hunGB  = (tenGB * 10),
             oneTB  = (hunGB * 10) ;

const wchar_t* colHead[COLCNT] = 
{
   L"INT%S %hd wide\n%S",    // 13
   L"   %hd wide\n%S",       // 12
   L"  %hd wide\n%S",        // 11
   L"  %hd wide\n%S",        // 10
   L"  %hd wide\n%S",        //  9
   L" %hd wide\n%S",         //  8
   L" %hd wide\n%S",         //  7
   L"%hd wide\n%S",          //  6
   L" %hd w\n%S",            //  5
   L"%hd w\n%S",             //  4
   L"%hd w\n%S",             //  3
   L"%hdw\n%S",              //  2
   L"%hd\n%S",               //  1
} ;

const wchar_t* wIndicator[COLCNT] = 
{
 L"-------------\n",
 L"------------\n",
 L"-----------\n",
 L"----------\n",
 L"---------\n",
 L"--------\n",
 L"-------\n",
 L"------\n",
 L"-----\n",
 L"----\n",
 L"---\n",
 L"--\n",
 L"-\n"
} ;

winPos colPos[13] = 
{
   { 1, 15 },
   { 1, 30 },
   { 1, 44 },
   { 1, 57 },
   { 1, 69 },
   { 1, 80 },
   { 1, 90 },
   { 1, 99 },
   { 1, 107 },
   { 1, 114 },
   { 1, 120 },
   { 1, 125 },
   { 1, 129 }
} ;

const unsigned long long int pIntA[] =  // positive values that do not need rounding
{ 1, 12, 123, 1234, 12345, 123456, 1234567, 12345678, 123456789, 1234567891 } ;
const signed long long int   nIntA[] =  // negative values that do not need rounding
{ -1, -12, -123, -1234, -12345, -123456, -1234567, -12345678, -123456789, -1234567891 } ;
const unsigned long long int pIntB[] =  // positive values at the rounding point
{ 9, 95, 995, 9995, 99995, 999995, 9999995, 99999995, 999999995, 1399999996 } ;
const signed long long int   nIntB[] =  // negative values at the rounding point
{ -9, -95, -995, -9995, -99995, -999995, -9999995, -99999995, -999999995, -1399999996 } ;
const unsigned long long int pIntC[] =   // positive max values
{ 9, 99, 999, 9999, 99999, 999999, 9999999, 99999999, 999999999, 9999999999 } ;
const signed long long int   nIntC[] =   // positive max values
{ -9, -99, -999, -9999, -99999, -999999, -9999999, -99999999, -999999999, -9999999999 } ;

   uiInfo Info ;                 // user interface data returned here
   short  icIndex = ZERO ;       // index of control with input focus
   fiUnits suffix = fiK ;        // Suffix units
   bool   done = false,          // loop control
          firstTime = true,      // formatting instructions
          lJust = false,
          binCompress = false,
          sgnOutput = false,
          negValues = false,
          bigValues = false ;

   while ( ! done )
   {
      if ( ! firstTime )
      {
         do
         {
            if ( intfmtIc[icIndex].type == dctPUSHBUTTON )
            {
               icIndex = this->dp->EditPushbutton ( Info ) ;
               if ( Info.dataMod != false )
               {
                  if ( icIndex == iDonePB )
                     done = true ;
               }
            }
            else if ( intfmtIc[icIndex].type == dctRADIOBUTTON )
            {
               icIndex = this->dp->EditRadiobutton ( Info ) ;
               if ( Info.dataMod != false )
               {
                  if ( Info.selMember == iRJustRB )
                     lJust = false ;
                  else if ( Info.selMember == iLJustRB )
                     lJust = true ;
                  else if ( Info.selMember == iBit32RB )
                     bigValues = false ;
                  else if ( Info.selMember == iBit64RB )
                     bigValues = true ;
                  else if ( Info.selMember == iDeciRB )
                     binCompress = false ;
                  else if ( Info.selMember == iKibiRB )
                     binCompress = true ;
                  else if ( Info.ctrlIndex == iNegRB )
                     negValues = Info.isSel ;
                  else if ( Info.ctrlIndex == iSgnRB )
                     sgnOutput = Info.isSel ;
               }
            }
            else if ( intfmtIc[icIndex].type == dctSCROLLBOX )
            {
               icIndex = this->dp->EditScrollbox ( Info ) ;
               if ( Info.dataMod != false )
               {
                  switch ( Info.selMember )
                  {
                     case 0:  suffix = fiK ;    break ;
                     case 1:  suffix = fik ;    break ;
                     case 2:  suffix = fiKb ;   break ;
                     case 3:  suffix = fikB ;   break ;
                     case 4:  suffix = fiKiB ;  break ;
                  }
               }
            }
            //* Move focus to next/previous control *
            if ( Info.dataMod == false )
            {
               if ( Info.keyIn == nckSTAB )
                  icIndex = this->dp->PrevControl () ;
               else
                  icIndex = this->dp->NextControl () ;
            }
            //* Return to 'Done' pushbutton *
            else
            {
               while ( icIndex != iDonePB )
                  icIndex = this->dp->NextControl () ;
            }
         }
         while ( ! Info.dataMod ) ;
         if ( ! done )
         {
            //* Clear the old data from the window *
            winPos cPos = colPos[ZERO] ;
            do
            {
               this->dp->ClearLine ( cPos.ypos++, false, cPos.xpos ) ;
            }
            while ( cPos.ypos < (intfmtROWS - 1) ) ;
         }
      }
      firstTime = false ;

      //******************************
      //* Display the formatted data *
      //******************************
      winPos   wPos ;
      short    i, width ;
      gString  oBuff ;

      if ( ! done )
      {
         long long int qVal ;             // working value (signed)
         unsigned long long int uVal ;    // working value (unsigned)
         width = 13 ;
         for ( short column = ZERO ; column < 13 ; ++column )
         {
            //* Set the base cursor position and write column heading *
            wPos = colPos[column] ;
            if ( column == ZERO )
               oBuff.compose( colHead[column], (bigValues ? L"64" : L"32"), 
                              &width, wIndicator[column] ) ;
            else
               oBuff.compose( colHead[column], &width, wIndicator[column] ) ;
            wPos = this->dp->WriteParagraph ( wPos, oBuff, this->dColor ) ;

            for ( i = ZERO ; i < MAX_NUMS ; i++ )  //** Row A **
            {
               if ( negValues )     // negative integers
                  oBuff.formatInt( nIntA[i], width, lJust, sgnOutput, binCompress, suffix ) ;
               else                 // positive integers
                  oBuff.formatInt( pIntA[i], width, lJust, sgnOutput, binCompress, suffix ) ;
               this->dp->WriteString ( wPos.ypos++, wPos.xpos, oBuff, this->dColor ) ;
            }
            wPos = this->dp->WriteParagraph ( wPos, wIndicator[column], this->dColor ) ;

            for ( i = ZERO ; i < MAX_NUMS ; i++ )  //** Row B **
            {
               if ( negValues )     // negative integers
               {
                  qVal = nIntB[i] ;
                  if ( bigValues ) qVal -= (tenGB * i) + nineGB ;
                  oBuff.formatInt( qVal, width, lJust, sgnOutput, binCompress, suffix ) ;
               }
               else                 // positive integers
               {
                  uVal = pIntB[i] ;
                  if ( bigValues ) uVal += (tenGB * i) + nineGB ;
                  oBuff.formatInt( uVal, width, lJust, sgnOutput, binCompress, suffix ) ;
               }
               this->dp->WriteString ( wPos.ypos++, wPos.xpos, oBuff, this->dColor ) ;
            }
            wPos = this->dp->WriteParagraph ( wPos, wIndicator[column], this->dColor ) ;

            for ( i = ZERO ; i < MAX_NUMS ; i++ )  //** Row C **
            {
               if ( negValues )     // negative integers
               {
                  qVal = nIntC[i] ;
                  if ( bigValues ) qVal -= ((oneTB * (i + 1)) + 
                                            (hunGB * (i + 1)) + 
                                            (tenGB * (i + 1)));
                  oBuff.formatInt( qVal, width, lJust, sgnOutput, binCompress, suffix ) ;
               }
               else                 // positive integers
               {
                  uVal = pIntC[i] ;
                  if ( bigValues ) uVal += ((oneTB * (i + 1)) + 
                                            (hunGB * (i + 1)) + 
                                            (tenGB * (i + 1)));
                  oBuff.formatInt( uVal, width, lJust, sgnOutput, binCompress, suffix ) ;
               }
               this->dp->WriteString ( wPos.ypos++, wPos.xpos, oBuff, this->dColor ) ;
            }
            --width ;      // decrease column width by 1
         }  // for(;;)
         dp->RefreshWin () ;
      }     // if(!done)
   }        // while()

}  //* End gtIntFmtUI() *

//*************************
//*     gtCloseDialog     *
//*************************
//******************************************************************************
//* Close the open dialog (if any).                                            *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void gStringTest::gtCloseDialog ( void )
{
   if ( this->dp != NULL )
      delete this->dp ;
   this->dp = NULL ;
   this->icPtr = NULL ;
   this->dRows     = 1 ;
   this->dCols     = 1 ;
   this->gtOpen    = false ;

}  //* End gtCloseDialog() *

//*************************
//*                       *
//*************************
//******************************************************************************
//*                                                                            *
//*                                                                            *
//*                                                                            *
//* Input  :                                                                   *
//*                                                                            *
//* Returns:                                                                   *
//******************************************************************************


