//********************************************************************************
//* File       : Dialogw.cpp                                                     *
//* Author     : Mahlon R. Smith                                                 *
//*              Copyright (c) 2011-2025 Mahlon R. Smith, The Software Samurai   *
//* Date       : 22-Mar-2025                                                     *
//* Version    : (see AppVersion string below)                                   *
//*                                                                              *
//* Description: Dialogw is a test and demonstration application for             *
//*              development of the NcDialog class and associated classes.       *
//*              The focus of this application is to exercise the 'wide' I/O     *
//*              functions of the ncursesw C-language library, and to            *
//*              incorporate the necessary subset of these functions into the    *
//*              NCurses class, NcWindow class, and NcDialog class.              *
//*                                                                              *
//*              See also the Dialog1, Dialog2, Dialog3 and Dialog4 test         *
//*              applications for additional examples of using the NCurses       *
//*              class, NcWindow class, NcDialog class and gString class         *
//*              methods for building your own NcDialog-based applications.      *
//*                                                                              *
//*                                                                              *
//* IMPORTANT NOTE:                                                              *
//*  This application optionally uses the Gnome TookKit Glib::ustring class      *
//*  for some tests. See also the ustringTest utility for more information.      *
//*  Official documentation for the Glib::ustring class may be found at:         *
//*  http://library.gnome.org/devel/glibmm/unstable/classGlib_1_1ustring.html    *
//*                                                                              *
//*  The use of third-party libraries is always a difficult decision because     *
//*  it introduces additional complexity to what should be a simple test         *
//*  application. The reason we chose to include Glib::ustring is because we     *
//*  are so very disappointed in the std::string class, and we wanted to know    *
//*  what was happening in the world of native support for UTF-8 encoded text.   *
//*                                                                              *
//*  The use of Glib::ustring class requires installation of both glib and       *
//*  glibmm libraries. The libsigc++ library is also bundled with these          *
//*  libraries, but is not used within the Glib::ustring class.                  *
//*  If you have these libraries installed and configured on your system,        *
//*  then please see the conditional compile flag GLIB_USTRING_ENABLE below      *
//*  and set it to 1 (one).                                                      *
//*  Also, edit 'Makefile' to select the linker invocation that includes         *
//*  these libraries.                                                            *
//*                                                                              *
//*                   Glib::ustring IS NOT USED BY THE                           *
//*                      NCurses/NcWindow/NcDialog                               *
//*                         FAMILY OF CLASSES                                    *
//*  Internally, these classes use the much simpler gString class (source code   *
//*  provided) to perform all conversions between UTF-8 encoding and wchar_t     *
//*  encoding.                                                                   *
//*                                                                              *
//*                                                                              *
//* Development Tools: See NcDialog.cpp.                                         *
//********************************************************************************
//* Version History (most recent first):                                         *
//*                                                                              *
//* v: 0.00.24 21-Mar-2025                                                       *
//*   - The gString 'gsForm&' constructor is obsolete. Replace gString           *
//*     constructors which take a 'gsForm&' argument with standard               *
//*     formatting-template constructors.                                        *
//*   - Certain gString method parameters required promotion from 'short' to     *
//*     'int32_t'.                                                               *
//*                                                                              *
//* v: 0.0.23 02-Jun-2024                                                        *
//*   - Enable microspacing adjustment on screen capture in T09_ControlUpdate()  *
//*     This is done because the dialog contains two-column (Chinese)            *
//*     characters which cause the HTML output to be mis-aligned.                *
//*   - Modify the capture-dialog stress-test code in Test09 to verify           *
//*     color-attribute alignment                                                *
//*   - Update Help dialog to reference G++13 and C++11.                         *
//*                                                                              *
//* v: 0.0.22 19-Dec-2020                                                        *
//*   - Bug Fix: The new version of the G++ compiler had some intermittant       *
//*     trouble allocating a moderate-sized array of 'cchar_t' structures on     *
//*     the stack, (see Test01()). This is disturbing. We have moved the         *
//*     allocation to the heap which seems to have cleared things up.            *
//*                                                                              *
//* v: 0.0.21 22-Apr-2020                                                        *
//*   - Because the WaylandCB class has now been integrated into the             *
//*     NcDialogAPI, activate the WaylandCB class for Test10() to exercise       *
//*     the interface between the system clipboard and the local Textbox         *
//*     clipboard.                                                               *
//*                                                                              *
//* v: 0.0.20 29-Jan-2018                                                        *
//*   - Allow equals sign ('=') to delimit '-a' option and its argument.         *
//*                                                                              *
//* v: 0.0.19 21-Jul-2017                                                        *
//*   - Add a new page of tests to gStringTest.cpp to support gString::strip()   *
//*     and gString::padCols(). Space for additional tests is now available.     *
//*                                                                              *
//* v: 0.0.18 20-Jun-2014                                                        *
//*   - Activate Test07 to access various examples used in the TexInfo           *
//*     documentation for the NcDialog link library API.                         *
//*     See infoDocTest.hpp and infoDocTest.cpp.                                 *
//*                                                                              *
//* v: 0.0.17 28-Nov-2013                                                        *
//*   - Move the integer-formatting test from Dialog2 test application.          *
//*     This was done because the integer-formatting functionality has been      *
//*     moved into the gString class, so its test logically belong here.         *
//*   - Move direct testing of the gString class to gStringTest.cpp to reduce    *
//*     the size of the source files. See Test06().                              *
//*                                                                              *
//* v: 0.0.16 27-Jun-2013                                                        *
//*   - Add test case to Test06, page 2 to exercise the gString-class binary     *
//*     formatting specifier.                                                    *
//*   - Add a test to the Test_wcout() method for using the return value of      *
//*     gString::compose() as input to std::wcout.                               *
//*            (This is actually much hotter than it sounds.)                    *
//*                                                                              *
//* v: 0.0.15 23-Apr-2012                                                        *
//*   - Update the startup routine to match the changes made in the              *
//*     StartNCursesEngine() method. These changes enable remapping of color     *
//*     pairs and RGB registers on startup.                                      *
//*                                                                              *
//* v: 0.0.14 09-Mar-2012                                                        *
//*   - Update certain tests of NCurses, NcWindow and NcDialog output methods    *
//*     to accomodate changes/bugs introduced to the ncurses library between     *
//*     v:5.7 and v:5.9.                                                         *
//*   - Update Test06 to reflect changes in the gString class related to the     *
//*     ncurses library changes.                                                 *
//*                                                                              *
//* v: 0.0.13 01-Sep-2011                                                        *
//*   - Move certain shared definitions, prototypes and data common to all the   *
//*     test and demonstration applications to a header, DialogAppShared.hpp.    *
//*                                                                              *
//* v: 0.0.12 30-Jul-2011                                                        *
//*   - Moved code for this application from the Cw development hack into the    *
//*     Dialogw test application. This was done as a first step in including     *
//*     Dialogw utility as part of the NCurses/NcWindows/NcDialog distribution   *
//*     package. This is warranted because the application has gone beyond the   *
//*     experimental stage and into serious testing of support for UTF-8         *
//*     encoded I/O for these classes.                                           *
//*   - Encapsulated all Glib::ustring functionality within the                  *
//*     GLIB_USTRING_ENABLE conditional compile flag. (see note above)           *
//*                                                                              *
//* v: 0.0.11 19-Mar-2011                                                        *
//*   - Replaced manual UTF-8 / wchar_t conversions with gString class           *
//*     implicit conversions for everything except Test01().                     *
//*                                                                              *
//* v: 0.0.10 05-Feb-2011                                                        *
//*   - Base code is a simplified version of the Dialog1 test application,       *
//*     but exercises all three levels: NCurses methods, NcWindow methods,       *
//*     and NcDialog methods, as well as the ncursesw primitives using           *
//*     multiple-language I/O supported by UTF-8 character encoding.             *
//*   - The std::wcout class is also tested for comparison purposes.             *
//*     Note that std::wcout is tested without starting the NCurses Engine.      *
//********************************************************************************
//* Programmer's Notes:                                                          *
//* - Because internationalization of I/O is always a possibility in any         *
//*   application, we use as few ASCII-oriented C/C++ library functions as       *
//*   possible. e.g. sprintf(), strlen(), toupper(), etc. Where such             *
//*   functionality is needed for character manipulation, we strongly advise     *
//*   using the equivalent 'wide chararacter' functions OR the gString class.    *
//*                                                                              *
//********************************************************************************


//* Enable or disable use of the glib and glibmm libraries in this application *
//*             (Disabled in the distribution package     )                    *
//*             (for better odds of a clean first compile.)                    *
#define GLIB_USTRING_ENABLE 0

//****************
//* Header Files *
//****************
//* All necessary information for:                  *
//* NCurses, NcWindow, NcDialog and gString classes.*
#include "GlobalDef.hpp"

//* Definitions, prototypes and data shared among the test applications *
#include "DialogAppShared.hpp"

//* gStringTest-class definition for Test06() *
#include "gStringTest.hpp"

//* infoDocTest-class definition for Test07() *
#include "infoDocTest.hpp"

#if GLIB_USTRING_ENABLE != 0     // Include Glib::ustring class testing
#include <glib.h>          //* GTK+ C library base
#include <glibmm.h>        //* GTK+ C++ wrapper for glib
#endif   // GLIB_USTRING_ENABLE

//***************
//* Definitions *
//***************

//* Used only for Test01() ncursesw primitives *
const short wsMAXCHARS = 128 + 1 ;

//**************
//* Prototypes *
//**************

void     Test_wcout ( bool adjustLocaleSettings = true, const char* newLocale = NULL ) ;
short    ncChar2Cchart ( cchar_t* dst, const char* src, attr_t attr = nc.bw ) ;
void     Help_wcout ( void ) ;
void     T10_FilterTest ( void ) ;

//**********
//*  Data  *
//**********
//* Application version string. Keep it current! *
const char AppVersion[] = "0.0.24" ;
const char* crYears = "2011-2025" ;


//* Programmer's Note: For any dialogs that use a callback method, a set of 
//* constants should be defined at file scope for the dialog's control indices, 
//* so both the instantiating method and its callback method have access to the 
//* same definitions. These could be done either as enums beginning with ZERO, 
//* or as 'const short' definitions.

enum T08Controls : short         // Test08() dialog control constants
                 {
                   T8donePush = ZERO, T8groupA_1, T8groupA_2, T8groupA_3, 
                   T8groupA_4, T8groupB_1, T8groupB_2,  T8groupB_3,
                   T8standAlone8, T8standAlone9, T8tbText, T8utBOX1,
                   T8utBOX2, T8utBOX3, T8utBOX4
                 } ;

enum T09Controls : short         // Test09() dialog control constants
                 {
                   T9donePush = 0, T9trackrBox, T9positionControl, 
                   T9planetBox, T9icecreamBox, T9tuscaloosaButton, 
                   T9sandiegoButton, T9cucamongaButton, T9valparisoButton, 
                   T9dumpPush, T9scottyButton, T9spinnerControl, T9dropdownBox, 
                   T9scrollextControl, T9menuwin01, T9menuwin02, T9menuwin03, 
                   T9menuwin04, T9controlsDEFINED
                 } ;

enum T10Controls : short         // Test10() dialog control constants
                 {
                   T10donePush = ZERO, T10tboxLJ1, T10tboxLJ2, T10tboxRJ1, 
                   T10tboxRJ2, T10tboxAJ, T10tboxLC, T10tboxRC, T10tboxINS, 
                   T10SysCbRadio, T10etRadio, T10filterPush, T10controlsDEFINED
                 } ;


//*************************
//*         main          *
//*************************
//******************************************************************************
//* Program entry point.                                                       *
//*                                                                            *
//* Command-line Usage:  SEE BELOW FOR COMMAND-LINE ARGUMENT PROCESSING        *
//*                                                                            *
//*                                                                            *
//******************************************************************************

int main ( int argc, char* argv[], char* argenv[] )
{
   //* Test the iostream WITHOUT the NCurses engine. *
   //* See notes in Test_wcout() below.              *
   if ( argc > 1 && argv[1][0] == '-' && (argv[1][1] == 'w' || argv[1][1] == 'W') )
   {
      //* Adjust locale settings before performing any i/o operations *
      if ( argv[1][2] == 'l' || argv[1][2] == 'L' )
      {
         bool  adjustLocaleSettings = true ;
         char* newLocale = NULL ;
         //* If no locale option is provided by user, use the locale specified *
         //* in the terminal's environment settings.                           *
         // NOTE: If user specified an invalid alternate locale name, then the 
         //       application will exit with an exception.
         //       This is user stupidity, not an application error.
         if ( argc > 2 )
            newLocale = argv[2] ;   // name of alternate locale
         Test_wcout ( adjustLocaleSettings, newLocale ) ;
      }
      //* Run test without adjusting the locale settings *
      else if ( argv[1][2] == 'n' || argv[1][2] == 'N' )
         Test_wcout ( false ) ;
      else
         Help_wcout () ;
      return ZERO ;
   }

   //******************************************************
   //* Begin the main application.                        *
   //******************************************************
short    testNum = - 1 ;         // number of test to execute (default TestMenu)
bool     winTooSmall = false ;   // set 'true' if terminal window is too small

   //* User may have specified one or more command-line arguments *
   commArgs clArgs( argc, argv, argenv ) ;
   testNum = GetCommandLineArgs ( clArgs ) ;

   #if 0    // PRETEST THE LOCALE SPECIFIED BY THE TERMINAL ENVIRONMENT
   // setenv ( "LANG", "yi_US.cp1255", 1 ) ; // (set non-UTF-8 locale)
   //* Note that on start-up, the NCurses Engine sets the 'locale' (character  *
   //* encoding for internationalization) according to the terminal window's   *
   //* environment variables; however, if the default locale does not support  *
   //* UTF-8 encoding, you should adjust the terminal environment variables    *
   //* LANG=  ( || LC_ALL=  ||  LC_CTYPE= ).                                   *
   //* If your application is not ASCII only, please set the terminal          *
   //* environment variable to 'en_US.UTF-8' or another locale that supports   *
   //* UTF-8 character encoding, before running the application, or at least   *
   //* before starting the NCurses Engine.                                     *
   //* -At the command prompt    :  LANG=en_US.UTF-8                           *
   //* -or within the application:  setenv("LANG", "en_US.UTF-8", 1);          *
   //*                                                                         *
   //* Note that UTF-8 is the native language of Linux.                        *
   //*                                                                         *
   //* If user has not specified a locale on the command line, test the default*
   if ( clArgs.altLocale == false )
   {
      locale loco("") ;
      short dfltLocaleOk = nc.VerifyLocale ( loco.name().c_str() ) ;
      if ( dfltLocaleOk != OK )
      {
         //* Default locale does not support UTF-8 character encoding *
         setenv ( "LANG", "en_US.UTF-8", 1 ) ;
      }
   }
   #endif   // PRETEST

   //*******************************
   //** Initialize NCurses Engine **
   //*******************************
   //* NCurses class has been previously instantiated at the time the          *
   //* application was loaded, and is available through the name: 'nc'.        *
   //* (see note in GlobalDef.hpp)                                             *
   if( (nc.StartNCursesEngine ()) == OK )
   {
      //* Verify that the current locale supports UTF-8 character encoding.    *
      short localeSetOk = nc.VerifyLocale () ;

      //* If user has specified an alternate locale for interpretation of      *
      //* character output and input, it must be passed to the NCurses class   *
      //* before any I/O occurs.                                               *
      //* (NOTE: If clArgs.localeName does not specify a locale supported by ) *
      //* (      the terminal program, OR the locale specified is not a      ) *
      //* (      UTF-8-encoded locale, OR the default encoding specified in  ) *
      //* (      the environment was not UTF-8, then the results will be     ) *
      //* (      ambiguous at best.                                          ) *
      short altLocaleOk = OK ;
      if ( clArgs.altLocale != false )
         altLocaleOk = nc.SetLocale ( clArgs.localeName ) ;

      //* Even though no windows have been opened yet, you can think of the    *
      //* entire screen as a big window which can be written to and which      *
      //* is the parent/grandparent of all windows and NcDialog objects that   *
      //* will be opened. (cursor state: nccvINVISIBLE and key state: nckpRAW) *
      ClearTerminalWindow () ;               // display application title
      winPos   wpos ;

      //* Initialize and open the application's status window *
      sWin = new NcWindow ( swLINES, swCOLS, swOFFy, swOFFx ) ;
      if ( (sWin->OpenWindow ()) != ERR )
      {
         bool  done = false ;    // loop control

         //* Report any error in initializing color output*
         if ( !(nc.ColorText_Initialized ()) )
         {
            if ( ! nc.ColorText_Available () )
               StatWin ( "This terminal does not support multi-color output..." ) ;
            else
               StatWin ( "Initialization of multi-color output failed..." ) ;
            sleep ( 5 ) ;
         }
         //* Report any error in setting default or alternate locale.  *
         //* (Even if error, ASCII-only I/O SHOULD cause no problems.) *
         if ( localeSetOk != OK || altLocaleOk != OK )
         {
            if ( localeSetOk != OK )
               StatWin ( "Current locale setting does not support UTF-8 character encoding..." ) ;
            else
               StatWin ( "Locale setting specified on command line is not valid..." ) ;
            sleep ( 5 ) ;
         }

         //* Refresh main window *
         nc.RefreshScreen () ;
         nc.ScreenDimensions ( termRows, termCols ) ;   // get terminal-window size
         if ( termRows < minTERMROWS || termCols < minTERMCOLS )
         {
            //* We need room to play *
            winTooSmall = done = true ;
         }

         //* Find out what the user wants *
         while ( ! done )
         {
            //* If command-line agrument for test number provided, use it.     *
            //* Else, open a dialog so user can choose which test(s) to run.   *
            if ( testNum < ZERO )
               testNum = TestMenu ( termRows / 2, termCols / 2 ) ;

            //* Check whether user has re-sized the terminal window *
            nc.ScreenDimensions ( termRows, termCols ) ;
            if ( termRows < minTERMROWS || termCols < minTERMCOLS )
            {
               winTooSmall = done = true ;
               continue ;
            }

            //* Execute selected test *            
            switch ( testNum )
            {
               case indexT01:          // Using ncursesw output primitives
                  Test01 () ;
                  break ;
               case indexT02:          // NCurses class wide output routines
                  Test02 () ; 
                  break ;
               case indexT03:          // NcWindow multibyte character output
                  Test03 () ;
                  break ;
               case indexT04:          // NcDialog multibyte character output
                  Test04 () ;
                  break ;
               case indexT05:          // NcDialog output - Wrap-around test
                  Test05 () ;
                  break ;
               case indexT06:          // gString class functionality test
                  Test06 () ;
                  break ;
               case indexT07:          // unassigned
                  Test07 () ;
                  break ;
               case indexT08:          // NcDialog controls, multi-byte chars
                  Test08 () ;
                  break ;
               case indexT09:          // Move-dialog window test (hide and show dialog)
                  Test09 () ;
                  break ;
               case indexT10:          // Textbox Controls - UTF-8 Data I/O
                  Test10 () ;
                  break ;
               case indexEXIT:         // exit the application
                  StatWin ( " Exiting Application . . . ", nc.blG ) ;
                  done = true ;
                  break ;
               case indexHELP:         // help-about dialog
               default:
                  HelpAbout ( termRows / 2 + 1, termCols / 2 ) ;
                  break ;
            }
            testNum = -1 ;    // force use of the menu next time through loop
         }        // while()
      }
      else
      {
         winTooSmall = true ;
      }

      //****************************
      //* Shut down ncurses engine *
      //****************************
      delete sWin ;                          // release status line resources       
      nc.RefreshScreen () ;                  // refresh the main window
      nc.StopNCursesEngine () ;              // Deactivate the NCurses engine

      //* Most likely cause of failure in opening the background *
      //* window is that the terminal window is too small.       *
      //* Memory allocation error is also a possibility.         *
      if ( winTooSmall != false )      
         cout << "\n ******************************************"
                 "\n * ERROR! Unwilling to open main window.  *"
                 "\n * Terminal window size must be >= 80x25  *"
                 "\n ******************************************\n\n" ;
   }
   else
   {
      cout << "\n **********************************************"
              "\n * ERROR! Unable to initialize NCurses engine.*"
              "\n **********************************************\n\n" ;
   }

   return ZERO ;

}  //* End main() *

//*************************
//*       TestMenu        *
//*************************
//******************************************************************************
//* Open a dialog and display a list of test choices.                          *
//*                                                                            *
//*                                                                            *
//* Input  : ctrY: Y center position for dialog window                         *
//*          ctrX: X center position for dialog window                         *
//*                                                                            *
//* Returns: member of cIndices                                                *
//******************************************************************************

short    TestMenu ( short ctrY, short ctrX )
{
static const short dialogROWS = 19 ;         // display lines
static const short dialogCOLS = 57 ;         // display columns
static const char* testNames[] = 
{// 1234567890123456789012345678901234567890123456

   "Test #1 : Using ncursesw output ^primitives   ",
   "Test #2 : ^NCurses multibyte character output ",
   "Test #3 : Nc^Window multibyte character output",
   "Test #4 : Nc^Dialog multibyte character output",
   "Test #5 : NcDialog output - W^rap-around test ",
   "Test #6 : ^gString class functionality test   ",
   "Test #7 : ^Examples included in TexInfo docs  ",
   "Test #8 : ^Textbox and Radio button controls  ",
   "Test #9 : M^ove Dialog Window (可移动的 窗口) ",
   "Test #10: Text ^Box With Multi-byte Char I/O  ",
} ;
static NcDialog*  dp ;                       // dialog window pointer
static bool dialogOpen = false ;             // set to true when dialog opened
short    ulY     = ctrY - dialogROWS / 2,    // upper left corner in Y
         ulX     = ctrX - dialogCOLS / 2 ;   // upper left corner in X
attr_t   dColor  = nc.blR,                   // dialog's base color
         rnColor = nc.bw,                    // radio button non-focus color
         rfColor = nc.cyG ;                  // radio button focus color
short    testNum ;                           // selected test, return value

   StatWin ( "Opening Test-Selection Menu" ) ; // keep user up-to-date

   //* Instantiate the dialog window and install its control objects *
   if ( dialogOpen == false )
   {
      //****************************************
      //* Initial parameters for dialog window *
      //****************************************
      InitCtrl ic[indexCOUNT] =                    // control initialization structures
      {
         {  //* TEST01 radiobutton  - - - - - - - - - - - - - - - - -    indexT01 *
            dctRADIOBUTTON,               // type:      
            rbtS3s,                       // rbSubtype: standard, 3 chars wide
            false,                        // rbSelect:  default selection
            2,                            // ulY:       upper left corner in Y
            4,                            // ulX:       upper left corner in X
            1,                            // lines:     (n/a)
            0,                            // cols:      (n/a)
            NULL,                         // dispText:  (n/a)
            rnColor,                      // nColor:    non-focus color
            rfColor,                      // fColor:    focus color
            tbPrint,                      // filter:    (n/a)
            testNames[0],                 // label:
            ZERO,                         // labY:      
            5,                            // labX       
            ddBoxTYPES,                   // exType:    (n/a)
            1,                            // scrItems:  (n/a)
            1,                            // scrSel:    (n/a)
            NULL,                         // scrColor:  (n/a)
            NULL,                         // spinData:  (n/a)
            true,                         // active:    allow control to gain focus
            &ic[1]                        // nextCtrl:  link in next structure
         },
         {  //* TEST02 radiobutton  - - - - - - - - - - - - - - - - -    indexT02 *
            dctRADIOBUTTON,               // type:      
            rbtS3s,                       // rbSubtype: standard, 3 chars wide
            false,                        // rbSelect:  default selection
            3,                            // ulY:       upper left corner in Y
            4,                            // ulX:       upper left corner in X
            1,                            // lines:     (n/a)
            0,                            // cols:      (n/a)
            NULL,                         // dispText:  (n/a)
            rnColor,                      // nColor:    non-focus color
            rfColor,                      // fColor:    focus color
            tbPrint,                      // filter:    (n/a)
            testNames[1],                 // label:
            ZERO,                         // labY:      
            5,                            // labX       
            ddBoxTYPES,                   // exType:    (n/a)
            1,                            // scrItems:  (n/a)
            1,                            // scrSel:    (n/a)
            NULL,                         // scrColor:  (n/a)
            NULL,                         // spinData:  (n/a)
            true,                         // active:    allow control to gain focus
            &ic[2]                        // nextCtrl:  link in next structure
         },
         {  //* TEST03 radiobutton  - - - - - - - - - - - - - - - - -    indexT03 *
            dctRADIOBUTTON,               // type:      
            rbtS3s,                       // rbSubtype: standard, 3 chars wide
            false,                        // rbSelect:  default selection
            4,                            // ulY:       upper left corner in Y
            4,                            // ulX:       upper left corner in X
            1,                            // lines:     (n/a)
            0,                            // cols:      (n/a)
            NULL,                         // dispText:  (n/a)
            rnColor,                      // nColor:    non-focus color
            rfColor,                      // fColor:    focus color
            tbPrint,                      // filter:    (n/a)
            testNames[2],                 // label:
            ZERO,                         // labY:      
            5,                            // labX       
            ddBoxTYPES,                   // exType:    (n/a)
            1,                            // scrItems:  (n/a)
            1,                            // scrSel:    (n/a)
            NULL,                         // scrColor:  (n/a)
            NULL,                         // spinData:  (n/a)
            true,                         // active:    allow control to gain focus
            &ic[3]                        // nextCtrl:  link in next structure
         },
         {  //* TEST04 radiobutton  - - - - - - - - - - - - - - - - -    indexT04 *
            dctRADIOBUTTON,               // type:      
            rbtS3s,                       // rbSubtype: standard, 3 chars wide
            false,                        // rbSelect:  default selection
            5,                            // ulY:       upper left corner in Y
            4,                            // ulX:       upper left corner in X
            1,                            // lines:     (n/a)
            0,                            // cols:      (n/a)
            NULL,                         // dispText:  (n/a)
            rnColor,                      // nColor:    non-focus color
            rfColor,                      // fColor:    focus color
            tbPrint,                      // filter:    (n/a)
            testNames[3],                 // label:
            ZERO,                         // labY:      
            5,                            // labX       
            ddBoxTYPES,                   // exType:    (n/a)
            1,                            // scrItems:  (n/a)
            1,                            // scrSel:    (n/a)
            NULL,                         // scrColor:  (n/a)
            NULL,                         // spinData:  (n/a)
            true,                         // active:    allow control to gain focus
            &ic[4]                        // nextCtrl:  link in next structure
         },
         {  //* TEST05 radiobutton  - - - - - - - - - - - - - - - - -    indexT05 *
            dctRADIOBUTTON,               // type:      
            rbtS3s,                       // rbSubtype: standard, 3 chars wide
            false,                        // rbSelect:  default selection
            6,                            // ulY:       upper left corner in Y
            4,                            // ulX:       upper left corner in X
            1,                            // lines:     (n/a)
            0,                            // cols:      (n/a)
            NULL,                         // dispText:  (n/a)
            rnColor,                      // nColor:    non-focus color
            rfColor,                      // fColor:    focus color
            tbPrint,                      // filter:    (n/a)
            testNames[4],                 // label:
            ZERO,                         // labY:      
            5,                            // labX       
            ddBoxTYPES,                   // exType:    (n/a)
            1,                            // scrItems:  (n/a)
            1,                            // scrSel:    (n/a)
            NULL,                         // scrColor:  (n/a)
            NULL,                         // spinData:  (n/a)
            true,                         // active:    allow control to gain focus
            &ic[5]                        // nextCtrl:  link in next structure
         },
         {  //* TEST06 radiobutton  - - - - - - - - - - - - - - - - -    indexT06 *
            dctRADIOBUTTON,               // type:      
            rbtS3s,                       // rbSubtype: standard, 3 chars wide
            false,                        // rbSelect:  default selection
            7,                            // ulY:       upper left corner in Y
            4,                            // ulX:       upper left corner in X
            1,                            // lines:     (n/a)
            0,                            // cols:      (n/a)
            NULL,                         // dispText:  (n/a)
            rnColor,                      // nColor:    non-focus color
            rfColor,                      // fColor:    focus color
            tbPrint,                      // filter:    (n/a)
            testNames[5],                 // label:
            ZERO,                         // labY:      
            5,                            // labX       
            ddBoxTYPES,                   // exType:    (n/a)
            1,                            // scrItems:  (n/a)
            1,                            // scrSel:    (n/a)
            NULL,                         // scrColor:  (n/a)
            NULL,                         // spinData:  (n/a)
            true,                         // active:    allow control to gain focus
            &ic[6]                        // nextCtrl:  link in next structure
         },
         {  //* TEST07 radiobutton  - - - - - - - - - - - - - - - - -    indexT07 *
            dctRADIOBUTTON,               // type:      
            rbtS3s,                       // rbSubtype: standard, 3 chars wide
            false,                        // rbSelect:  default selection
            8,                            // ulY:       upper left corner in Y
            4,                            // ulX:       upper left corner in X
            1,                            // lines:     (n/a)
            0,                            // cols:      (n/a)
            NULL,                         // dispText:  (n/a)
            rnColor,                      // nColor:    non-focus color
            rfColor,                      // fColor:    focus color
            tbPrint,                      // filter:    (n/a)
            testNames[6],                 // label:
            ZERO,                         // labY:      
            5,                            // labX       
            ddBoxTYPES,                   // exType:    (n/a)
            1,                            // scrItems:  (n/a)
            1,                            // scrSel:    (n/a)
            NULL,                         // scrColor:  (n/a)
            NULL,                         // spinData:  (n/a)
            true,                         // active:    allow control to gain focus
            &ic[7]                        // nextCtrl:  link in next structure
         },
         {  //* TEST08 radiobutton  - - - - - - - - - - - - - - - - -    indexT08 *
            dctRADIOBUTTON,               // type:      
            rbtS3s,                       // rbSubtype: standard, 3 chars wide
            false,                        // rbSelect:  default selection
            9,                            // ulY:       upper left corner in Y
            4,                            // ulX:       upper left corner in X
            1,                            // lines:     (n/a)
            0,                            // cols:      (n/a)
            NULL,                         // dispText:  (n/a)
            rnColor,                      // nColor:    non-focus color
            rfColor,                      // fColor:    focus color
            tbPrint,                      // filter:    (n/a)
            testNames[7],                 // label:
            ZERO,                         // labY:      
            5,                            // labX       
            ddBoxTYPES,                   // exType:    (n/a)
            1,                            // scrItems:  (n/a)
            1,                            // scrSel:    (n/a)
            NULL,                         // scrColor:  (n/a)
            NULL,                         // spinData:  (n/a)
            true,                         // active:    allow control to gain focus
            &ic[8]                        // nextCtrl:  link in next structure
         },
         {  //* TEST09 radiobutton  - - - - - - - - - - - - - - - - -    indexT09 *
            dctRADIOBUTTON,               // type:      
            rbtS3s,                       // rbSubtype: standard, 3 chars wide
            false,                        // rbSelect:  default selection
            10,                           // ulY:       upper left corner in Y
            4,                            // ulX:       upper left corner in X
            1,                            // lines:     (n/a)
            0,                            // cols:      (n/a)
            NULL,                         // dispText:  (n/a)
            rnColor,                      // nColor:    non-focus color
            rfColor,                      // fColor:    focus color
            tbPrint,                      // filter:    (n/a)
            testNames[8],                 // label:
            ZERO,                         // labY:      
            5,                            // labX       
            ddBoxTYPES,                   // exType:    (n/a)
            1,                            // scrItems:  (n/a)
            1,                            // scrSel:    (n/a)
            NULL,                         // scrColor:  (n/a)
            NULL,                         // spinData:  (n/a)
            true,                         // active:    allow control to gain focus
            &ic[9]                        // nextCtrl:  link in next structure
         },
         {  //* TEST10 radiobutton  - - - - - - - - - - - - - - - - -    indexT10 *
            dctRADIOBUTTON,               // type:      
            rbtS3s,                       // rbSubtype: standard, 3 chars wide
            false,                        // rbSelect:  default selection
            11,                           // ulY:       upper left corner in Y
            4,                            // ulX:       upper left corner in X
            1,                            // lines:     (n/a)
            0,                            // cols:      (n/a)
            NULL,                         // dispText:  (n/a)
            rnColor,                      // nColor:    non-focus color
            rfColor,                      // fColor:    focus color
            tbPrint,                      // filter:    (n/a)
            testNames[9],                 // label:
            ZERO,                         // labY:      
            5,                            // labX       
            ddBoxTYPES,                   // exType:    (n/a)
            1,                            // scrItems:  (n/a)
            1,                            // scrSel:    (n/a)
            NULL,                         // scrColor:  (n/a)
            NULL,                         // spinData:  (n/a)
            true,                         // active:    allow control to gain focus
            &ic[10]                       // nextCtrl:  link in next structure
         },
// INSERT ANY ADDITIONAL TEST SELECTORS HERE
         {  //* HELP radiobutton    - - - - - - - - - - - - - - - - -   indexHELP *
            dctRADIOBUTTON,               // type:      
            rbtS3s,                       // rbSubtype: standard, 3 chars wide
            true,                         // rbSelect:  default selection
            12,                           // ulY:       upper left corner in Y
            4,                            // ulX:       upper left corner in X
            1,                            // lines:     (n/a)
            0,                            // cols:      (n/a)
            NULL,                         // dispText:  (n/a)
            rnColor,                      // nColor:    non-focus color
            rfColor,                      // fColor:    focus color
            tbPrint,                      // filter:    (n/a)
            "^Help About dialog",         // label:
            ZERO,                         // labY:      
            5,                            // labX       
            ddBoxTYPES,                   // exType:    (n/a)
            1,                            // scrItems:  (n/a)
            1,                            // scrSel:    (n/a)
            NULL,                         // scrColor:  (n/a)
            NULL,                         // spinData:  (n/a)
            true,                         // active:    allow control to gain focus
            &ic[11]                       // nextCtrl:  link in next structure
         },
         {  //* EXIT radiobutton    - - - - - - - - - - - - - - - - -   indexEXIT *
            dctRADIOBUTTON,               // type:      
            rbtS3s,                       // rbSubtype: standard, 3 chars wide
            false,                        // rbSelect:  default selection
            13,                           // ulY:       upper left corner in Y
            4,                            // ulX:       upper left corner in X
            1,                            // lines:     (n/a)
            0,                            // cols:      (n/a)
            NULL,                         // dispText:  (n/a)
            rnColor,                      // nColor:    non-focus color
            rfColor,                      // fColor:    focus color
            tbPrint,                      // filter:    (n/a)
            "E^xit",                      // label:
            ZERO,                         // labY:      
            5,                            // labX       
            ddBoxTYPES,                   // exType:    (n/a)
            1,                            // scrItems:  (n/a)
            1,                            // scrSel:    (n/a)
            NULL,                         // scrColor:  (n/a)
            NULL,                         // spinData:  (n/a)
            true,                         // active:    allow control to gain focus
            NULL                          // nextCtrl:  link in next structure
         },
      } ;
   
      //* Initial parameters for dialog window *
      InitNcDialog dInit( dialogROWS,     // number of display lines
                          dialogCOLS,     // number of display columns
                          ulY,            // Y offset from upper-left of terminal 
                          ulX,            // X offset from upper-left of terminal 
                          "  Select Test To Be Executed  ", // dialog title
                          ncltSINGLE,     // border line-style
                          dColor,         // border color attribute
                          dColor,         // interior color attribute
                          ic              // pointer to list of control definitions
                        ) ;

      //* Instantiate the dialog window *
      dp = new NcDialog ( dInit ) ;
   
      //* Open the dialog window *
      if ( (dp->OpenWindow()) == OK )
      {
         dialogOpen = true ;              // remember that dialog is open
         
         dp->DrawBox ( 1,  2, dialogROWS-5, 53, dColor ) ; // enclose radio-button group
         //* Display instructions *
         gString gsOut( "Use Tab and Shift+Tab or arrow keys to scroll through\n"
                        "options, then Space or Enter key to select;\n"
                        "or select a test via 'hotkey' (underlined character)" ) ;
         dp->WriteParagraph ( dialogROWS-4, 2, gsOut, dColor ) ;
         dp->RefreshWin () ;
   
         //* Group all radio buttons in the dialog *
         short XorGroup[] = { indexT01, indexT02, indexT03, indexT04, indexT05, 
                              indexT06, indexT07, indexT08, indexT09, indexT10, 
                              indexHELP, indexEXIT, -1 } ;
         dp->GroupRadiobuttons ( XorGroup ) ;
      }
   }           // If(dialogOpen==false)
   
   //* Un-hide the existing dialog window *
   else
   {
      dp->RefreshWin () ;
   }

   if ( dialogOpen != false )
   {
      //* Give the focus to the currently 'selected' button *
      short sIndex = dp->GetRbGroupSelection ( indexEXIT ) ;
      short cIndex = dp->GetCurrControl () ;
      while ( cIndex != sIndex )
         cIndex = dp->PrevControl () ;

      uiInfo Info ;                       // user interface info
      bool   testSelected = false ;       // loop control
      while ( !testSelected )
      {
         //* Move among the members of this Radio Button group *
         //* allowing user to select a test to be executed.    *
         // NOTE: This interface loop is fairly simple because *
         //       all controls in this dialog are Radio Buttons*
         //       and they all belong to the same button group.*
         dp->EditRadiobutton ( Info ) ;
         
         //* If a new selection has been made, OR  *
         //* if previous selection was re-selected.*
         if ( Info.dataMod || Info.keyIn == nckENTER )
            testSelected = true ;
      }
      testNum = Info.selMember ; // return value
   }

   //* If we are about to exit application, close the dialog. *
   if ( testNum == indexEXIT )
   {
      if ( dp != NULL )                      // close the window
         delete ( dp ) ;
      dp = NULL ;
   }
   //* Else, hide the window for future use *
   else
   {
      dp->HideWin () ;
   }

   return testNum ;

}  //* End TestMenu() *

//*************************
//*       Test01          *
//*************************
//******************************************************************************
//* This is a test of the ncursesw library output primitives.                  *
//* The 'w' stands for 'wide', and supposedly, the ncursesw library supports   *
//* UTF-8-encoded text. Let's find out, shall we?                              *
//*                                                                            *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************
//* This is the definiiton of the cchar_t structure defined in curses.h.       *
//* This is passed to the ncursesw primitive method, wadd_wch() as parameter   *
//* two.                                                                       *
//* - A string of from one to CCHARW_MAX characters may be printed to the      *
//*   display with a single call to wadd_wch().                                *
//* - A zero ('\0') value signals the end of the string.                       *
//* - A single color attribute value will be used for all specified characters.*
//* - In general, it is better to send only one character per call; however,   *
//*   certain characters (e.g. Modern Standard Arabic characters) are          *
//*   dependent on their context i.e. the character before or after, and so    *
//*   must be sent as a group.                                                 *
//*                                                                            *
//* #define CCHARW_MAX	5                                                      *
//* typedef struct                                                             *
//* {                                                                          *
//*    attr_t	attr;                                                           *
//*    wchar_t	chars[CCHARW_MAX];                                              *
//* } cchar_t ;                                                                *
//*                                                                            *
//* Note that cursor positioning is a difficult concept when not all characters*
//* are the same width. Therefore, it is important not to make assumptions     *
//* about where the cursor is after writing to the display, except that it is  *
//* positioned just after whatever you just wrote. Also be cautious about      *
//* your text length to prevent running past your display area.                *
//*  THERE IS A DIFFERENCE BETWEEN NUMBER OF BYTES AND NUMBER OF CHARACTERS!!  *
//*                                                                            *
//* RTL Languages:                                                             *
//* Unicode provides embedded codes to explicity designate text as flowing     *
//* left-to-right or right-to-left.                                            *
//*                                                                            *
//* Implicit encoding of left-to-right or right-to-left characters is handled  *
//* by some terminals through "bidirectional hints." This works sometimes;     *
//* however, when you rely on magic, especially someone else's magic, you may  *
//* spend your life as a frog, searching for the kiss of a princess.           *
//* Personally, I have found that princesses are rather high-maintenance, so   *
//* RTL text is EXPLICITLY written right-to-left.                              *
//*                                                                            *
//* Unicode provides embedded codes for specifying the direction of text flow. *
//* These are most useful when mixed LTR and RTL text are written.             *
//*                                                                            *
//*            Please see http://www.unicode.org/reports/tr9/                  *
//*            - - - - - - - - - - - - - - - - - - - - - - -                   *
//* Explicit Directional Embedding and Override Formatting Characters          *
//*   LRE  U+202A   Embedded left-to-right text follows                        *
//*   RLE  U+202B   Embedded right-to-left text follows                        *
//*   PDF  U+202C   Terminate the previous embedding code                      *
//*   LRO  U+202D   Force following text to be treated as strong LTR (avoid)   *
//*   RLO  U+202E   Force following text to be treated as strong RTL (avoid)   *
//*                                                                            *
//* Explicit Directional Isolate Formatting Characters                         *
//*   LRI  U+2066   Isolated left-to-right text follows                        *
//*   RLI  U+2067   Isolated right-to-left text follows                        *
//*   FSI  U+2068   Isolated text follows, and in the direction if the first   *
//*                 non-isolated 'strong directional character' (avoid)        *
//*   PDI  U+2069   Terminate the previous isolation code                      *
//*                                                                            *
//* Implicit Directional Formatting Characters                                 *
//*   LRM  U+200E   Left-to-right zero-width character                         *
//*   RLM  U+200F   Right-to-left zero-width non-Arabic character              *
//*   ALM  U+061C   Right-to-left zero-width Arabic character                  *
//*                                                                            *
//*                                                                            *
//*                                                                            *
//******************************************************************************

void Test01 ( void )
{
   StatWin ( "Test #01 - Using ncursesw output primitives" ) ;  // keep user up-to-date

   const char test01[] = { "This is a test of ncursesw primitives (narrow: waddch())" } ;
   const char endTest[] = { " PRESS ANY KEY TO CONTINUE " } ;
   const cchar_t test02[] = 
   { 
      {nc.bw, {'T'}}, {nc.bw, {'h'}}, {nc.bw, {'i'}}, {nc.bw, {'s'}}, {nc.bw, {' '}}, 
      {nc.bw, {'i'}}, {nc.bw, {'s'}}, {nc.bw, {' '}}, {nc.bw, {'a'}}, {nc.bw, {' '}}, 
      {nc.bw, {'t'}}, {nc.bw, {'e'}}, {nc.bw, {'s'}}, {nc.bw, {'t'}}, {nc.bw, {' '}},
      {nc.bw, {'o'}}, {nc.bw, {'f'}}, {nc.bw, {' '}}, 
      {nc.bw, {'n'}}, {nc.bw, {'c'}}, {nc.bw, {'u'}}, {nc.bw, {'r'}}, {nc.bw, {'s'}}, 
         {nc.bw, {'e'}}, {nc.bw, {'s'}}, {nc.bw, {'w'}}, {nc.bw, {' '}}, 
      {nc.bw, {'p'}}, {nc.bw, {'r'}}, {nc.bw, {'i'}}, {nc.bw, {'m'}}, {nc.bw, {'i'}}, 
         {nc.bw, {'t'}}, {nc.bw, {'i'}}, {nc.bw, {'v'}}, {nc.bw, {'e'}}, {nc.bw, {'s'}}, 
         {nc.bw, {' '}}, 
      {nc.bw, {'('}}, {nc.bw, {'w'}}, {nc.bw, {'i'}}, {nc.bw, {'d'}}, {nc.bw, {'e'}}, 
         {nc.bw, {':'}}, {nc.bw, {' '}}, 
      {nc.bw, {'w'}}, {nc.bw, {'a'}}, {nc.bw, {'d'}}, {nc.bw, {'d'}}, {nc.bw, {'_'}}, 
         {nc.bw, {'w'}}, {nc.bw, {'c'}}, {nc.bw, {'h'}}, {nc.bw, {'('}}, {nc.bw, {')'}}, 
         {nc.bw, {')'}},  
      {nc.bw, {'\0'}}, 
   } ;
   const short t3STRUCTS = 15 ;
   const cchar_t test03[t3STRUCTS] = 
   { 
      {nc.bw, {'C', 'o', 'm', 'p', 'a' }}, {nc.bw, {'c', 't', ' ', 's', 'i' }}, 
      {nc.bw, {'n', 'g', 'l', 'e', '-' }}, {nc.bw, {'c', 'o', 'l', 'o', 'r' }}, 
      {nc.bw, {' ', 's', 't', 'r', 'i' }}, {nc.bw, {'n', 'g', ' ', 'e', 'm' }}, 
      {nc.bw, {'b', 'e', 'd', 'd', 'e' }}, {nc.bw, {'d', ' ', 'i', 'n', ' ' }}, 
      {nc.bw, {'a', ' ', 'c', 'c', 'h' }}, {nc.bw, {'a', 'r', '_', 't', ' ' }}, 
      {nc.bw, {'a', 'r', 'r', 'a', 'y' }}, {nc.bw, {' ', '(', 'w', 'i', 'd' }}, 
      {nc.bw, {'e', ':', ' ', 'w', 'a' }}, {nc.bw, {'d', 'd', '_', 'w', 'c' }}, 
      {nc.bw, {'h', '(', ')', ')','\0' }},  
   } ;
   const cchar_t test04[t3STRUCTS] = 
   { 
      {nc.bw, {'C', 'o', 'm', 'p', 'a' }}, {nc.re, {'c', 't', ' ', 'm', 'u' }}, 
      {nc.gr, {'l', 't', 'i', '-', 'c' }}, {nc.br, {'o', 'l', 'o', 'r', ' ' }}, 
      {nc.bl, {'s', 't', 'r', 'i', 'n' }}, {nc.ma, {'g', ' ', 'e', 'm', 'b' }}, 
      {nc.cy, {'e', 'd', 'd', 'e', 'd' }}, {nc.gy, {' ', 'i', 'n', ' ', 'a' }}, 
      {nc.bw, {' ', 'c', 'c', 'h', 'a' }}, {nc.re, {'r', '_', 't', ' ','\0' }}, 
      {nc.gr, {'a', 'r', 'r', 'a', 'y' }}, {nc.br, {' ', '(', 'w', 'i', 'd' }}, 
      {nc.bl, {'e', ':', ' ', 'w', 'a' }}, {nc.ma, {'d', 'd', '_', 'w', 'c' }}, 
      {nc.cy, {'h', '(', ')', ')','\0' }},  
   } ;
   const char cSubHead01[] = { "NEXT: Characters requiring more than 8 bits: 'supra-ASCII'" } ;
   const short t5STRUCTS = 8 ;
   const cchar_t test05[t5STRUCTS] = 
   {
      {nc.bw, {L"** á"}}, {nc.bw, {L" Á à"}}, {nc.bw, {L" À ã"}}, {nc.bw, {L" Ã â"}}, 
      {nc.bw, {L" Â ạ"}}, {nc.bw, {L" ā ả"}}, {nc.bw, {L" ȧ ă"}}, {nc.bw, {L" **"}}, 
   } ;
   const char cChinese01[]  = { "Chinese  - 这种评论荒谬绝伦。" } ;
   const char cChinese02[]  = { "Chinese  - 你为什么不能在家过年啊？" } ;
   const char cKorean01[]   = { "Korean   - 안녕하세요. 만나서 반갑습니다." } ;
   const char cKorean02[]   = { "Korean   - 저는 골프와 테니스 등 스포츠를 좋아해요." } ;
   const char cKanji01[]    = { "Kanji    - こんにちは。始めまして。" } ;
   const char cKanji02[]    = { "Kanji    - 始めまして。お元気ですか。" } ;
   const char cHiragana01[] = { "Hiragana - こんにちは。はじめまして。" } ;
   const char cHiragana02[] = { "Hiragana - はじめまして。おげんきですか。" } ;
   const char cRussian[]    = { "Russian  - Доброе утро. Как поживаете?" } ;
   const char cPolish[]     = { "Polish   - On patrzy na tę piękną dziewczynę." } ;
   const char cArabic[]     = { "هَل تَتَكَلَّم الْانْجِلِيزِيَّة؟"} ;
   const char cArabicHead[] = { "Arabic   - " } ;
   const char cHebrew[]     = { "הלו חבר! מה נשמע? התגעגעתי אליך." } ;
   const char cHebrewHead[] = { "Hebrew   - " } ;

   winPos wp( 3, 0 ) ;                             // cursor positioning
   WINDOW*  scrAddr = nc.Get_stdscr_Address () ;   // address of main ncurses window

   //* ASCII C string *
   wmove ( scrAddr, wp.ypos, wp.xpos ) ;           // set cursor
   for ( short i = ZERO ; test01[i] != NULLCHAR ; ++i )
      waddch ( scrAddr, (int)test01[i] ) ;

   //* ASCII cchar_t string *
   wmove ( scrAddr, ++wp.ypos, wp.xpos ) ;         // set cursor
   for ( short i = ZERO ; test02[i].chars[0] != NULLCHAR ; ++i )
      wadd_wch ( scrAddr, &test02[i] ) ;

   //* What's up with the definition of cchar_t ?? *
   wmove ( scrAddr, ++wp.ypos, wp.xpos ) ;         // set cursor
   for ( short i = ZERO ; i < t3STRUCTS ; ++i )
      wadd_wch ( scrAddr, &test03[i] ) ;

   wmove ( scrAddr, ++wp.ypos, wp.xpos ) ;         // set cursor
   for ( short i = ZERO ; i < t3STRUCTS ; ++i )
      wadd_wch ( scrAddr, &test04[i] ) ;

   wmove ( scrAddr, ++wp.ypos, wp.xpos ) ;         // set cursor
   for ( short i = ZERO ; cSubHead01[i] != NULLCHAR ; ++i )
      waddch ( scrAddr, (int)cSubHead01[i] | nc.blB | ncuATTR  ) ;

   wmove ( scrAddr, ++wp.ypos, ++wp.xpos ) ;       // set cursor
   for ( short i = ZERO ; i < t5STRUCTS ; ++i )
      wadd_wch ( scrAddr, &test05[i] ) ;

   //* Allocate an array of cchar_t structures *
   cchar_t* outBuff = new cchar_t[wsMAXCHARS] ;

   wmove ( scrAddr, ++wp.ypos, wp.xpos ) ;         // set cursor
   short loopCount = ncChar2Cchart ( outBuff, cChinese01 ) ;
   for ( short i = ZERO ; i < loopCount ; i++ )
      wadd_wch ( scrAddr, &outBuff[i] ) ;

   wmove ( scrAddr, ++wp.ypos, wp.xpos ) ;         // set cursor
   loopCount = ncChar2Cchart ( outBuff, cChinese02, nc.re ) ;
   for ( short i = ZERO ; i < loopCount ; i++ )
      wadd_wch ( scrAddr, &outBuff[i] ) ;

   wmove ( scrAddr, ++wp.ypos, wp.xpos ) ;         // set cursor
   loopCount = ncChar2Cchart ( outBuff, cKorean01 ) ;
   for ( short i = ZERO ; i < loopCount ; i++ )
      wadd_wch ( scrAddr, &outBuff[i] ) ;

   wmove ( scrAddr, ++wp.ypos, wp.xpos ) ;         // set cursor
   loopCount = ncChar2Cchart ( outBuff, cKorean02, nc.gr ) ;
   for ( short i = ZERO ; i < loopCount ; i++ )
      wadd_wch ( scrAddr, &outBuff[i] ) ;

   wmove ( scrAddr, ++wp.ypos, wp.xpos ) ;         // set cursor
   loopCount = ncChar2Cchart ( outBuff, cKanji01 ) ;
   for ( short i = ZERO ; i < loopCount ; i++ )
      wadd_wch ( scrAddr, &outBuff[i] ) ;

   wmove ( scrAddr, ++wp.ypos, wp.xpos ) ;         // set cursor
   loopCount = ncChar2Cchart ( outBuff, cKanji02, nc.br ) ;
   for ( short i = ZERO ; i < loopCount ; i++ )
      wadd_wch ( scrAddr, &outBuff[i] ) ;

   wmove ( scrAddr, ++wp.ypos, wp.xpos ) ;         // set cursor
   loopCount = ncChar2Cchart ( outBuff, cHiragana01 ) ;
   for ( short i = ZERO ; i < loopCount ; i++ )
      wadd_wch ( scrAddr, &outBuff[i] ) ;

   wmove ( scrAddr, ++wp.ypos, wp.xpos ) ;         // set cursor
   loopCount = ncChar2Cchart ( outBuff, cHiragana02, nc.bl ) ;
   for ( short i = ZERO ; i < loopCount ; i++ )
      wadd_wch ( scrAddr, &outBuff[i] ) ;

   wmove ( scrAddr, ++wp.ypos, wp.xpos ) ;         // set cursor
   loopCount = ncChar2Cchart ( outBuff, cRussian, nc.ma ) ;
   for ( short i = ZERO ; i < loopCount ; i++ )
      wadd_wch ( scrAddr, &outBuff[i] ) ;

   wmove ( scrAddr, ++wp.ypos, wp.xpos ) ;         // set cursor
   loopCount = ncChar2Cchart ( outBuff, cPolish, nc.cy ) ;
   for ( short i = ZERO ; i < loopCount ; i++ )
      wadd_wch ( scrAddr, &outBuff[i] ) ;

   //* For the ncurses primitives, RTL (right-to-left) languages   *
   //* must be manually written at their correct positions.        *
   //* The width of RTL characters in our test-data set is 'known' *
   //* to be one (1) column, but such an assumption would be       *
   //* inexcusable in real life.                                   *
   wmove ( scrAddr, ++wp.ypos, wp.xpos ) ;         // set cursor
   loopCount = ncChar2Cchart ( outBuff, cArabicHead, nc.br ) ;
   for ( short i = ZERO ; i < loopCount ; i++ )
      wadd_wch ( scrAddr, &outBuff[i] ) ;
   short xoffset = wp.xpos + 30 ;
   loopCount = ncChar2Cchart ( outBuff, cArabic, nc.br ) ;
   for ( short i = ZERO ; i < loopCount ; i++ )
   {
      mvwadd_wch ( scrAddr, wp.ypos, xoffset, &outBuff[i] ) ;
      --xoffset ;    // horrible-but-convenient kludge (see note)
   }
   wmove ( scrAddr, ++wp.ypos, wp.xpos ) ;         // set cursor
   loopCount = ncChar2Cchart ( outBuff, cHebrewHead, nc.gr ) ;
   for ( short i = ZERO ; i < loopCount ; i++ )
      wadd_wch ( scrAddr, &outBuff[i] ) ;
   xoffset = wp.xpos + 41 ;
   loopCount = ncChar2Cchart ( outBuff, cHebrew, nc.gr ) ;
   for ( short i = ZERO ; i < loopCount ; i++ )
   {
      mvwadd_wch ( scrAddr, wp.ypos, xoffset, &outBuff[i] ) ;
      --xoffset ;    // horrible-but-convenient kludge (see note)
   }

   wp.ypos += 2 ;
   wp.xpos  = 8 ;
   wmove ( scrAddr, wp.ypos, wp.xpos ) ;           // set cursor
   for ( short i = ZERO ; endTest[i] != NULLCHAR ; ++i )
      waddch ( scrAddr, (int)endTest[i] | nc.reG ) ;

   wrefresh ( scrAddr ) ;     // redraw the window
   nckPause() ;               // wait for keypress

   delete [] outBuff ;        // release the dynamic allocation

   //* Clear screen of the test data *
   ClearTerminalWindow () ;

}  //* End Test01() *

//*************************
//*     ncChar2Cchart     *
//*************************
//******************************************************************************
//* Convert a string of UTF-8-encoded characters into a sequence of wchar_t    *
//* characters embedded in an array of ncursesw cchar_t structures.            *
//*                                                                            *
//* This is how the ncursesw library supports characters requiring more than   *
//* eight bits. It's a horrible kludge, of course, but we work with what we    *
//* have. Note: Placing nullchar characters into unused cchar_t 'chars' array  *
//* is necessary to avoid artifacts.                                           *
//*                                                                            *
//* IMPORTANT NOTE:                                                            *
//* This method is used by Test01() for demonstration purposes only.           *
//* For any more serious code, please see the gString class which automagically*
//* converts to/from UTF-8 and wchar_t. Also, it is both unnecessary AND       *
//* inadvisable for any application based on the NCurses, NcWindow and NcDialog*
//* classes to know anything about wide character output using the nasty       *
//* cchar_t structure because all such output is handled transparently by      *
//* member methods of these classes.                                           *
//*                                                                            *
//* Input  : dst : array of cchar_t structures to hold the converted data      *
//*          src : UTF-8-encoded 'C' string                                    *
//*          attr: (optional, nc.bw by default)                                *
//*                color attribute to be used for display of characters        *
//*                                                                            *
//* Returns: number of target structures initialized which is the number of    *
//*          times the wadd_wch() method should be called to display the string*
//*          NOTE: A ZERO return value indicates an error in conversion from   *
//*                UTF-8 to wchar_t.                                           *
//******************************************************************************
//* Programer's Note:                                                          *
//* For the intimate details of conversion between UTF-8 and wchar_t           *
//* characters, AND calculation of the number of columns required for display  *
//* of a character, please refer to the standard library functions:            *
//*   mbsrtowcs()      convert from UTF-8 to wchar_t                           *
//*   wcsrtombs()      convert from wchar_t to UTF-8                           *
//*   wcwidth()        look up number of display columns required for character*
//*                                                                            *
//* These library functions are tricky, and overall, it is STRONGLY recommended*
//* that you let the gString class handle ALL character conversions.           *
//* If you must investigate the use of these functions, please refer to the    *
//* source code for the gString class.                                         *
//*                                                                            *
//* Note that the formatting algorithm used in this method ASSUMES that a      *
//* printing (non-zero-width) character may be modified by either leading      *
//* zero-width characters OR by trailing zero-width characters, BUT NOT BOTH!  *
//*                                                                            *
//******************************************************************************

short ncChar2Cchart ( cchar_t* dst, const char* src, attr_t attr )
{
   int wI = ZERO, dstI = ZERO, cci, strLen ;
   gString  gs( src ) ;
   const wchar_t* wPtr = gs.gstr() ;
   const short*   cPtr = gs.gscols( strLen ) ;
   --strLen ;     // don't include the null terminator

   if ( strLen > ZERO )
   {
      for ( dstI = ZERO ; (wI < strLen) && (dstI < wsMAXCHARS) ; dstI++ )
      {  //* Initialize the structure *
         dst[dstI].attr = attr ;
         dst[dstI].chars[0] = 
         dst[dstI].chars[1] = dst[dstI].chars[2] = 
         dst[dstI].chars[3] = dst[dstI].chars[4] = nckNULLCHAR ;

         //* Leading, zero-width characters *
         bool leadzero = false ;
         for ( cci = ZERO ; cci < CCHARW_MAX && wPtr[wI] != nckNULLCHAR && 
                            cPtr[wI] == ZERO ; )
         {
            dst[dstI].chars[cci++] = wPtr[wI++] ;
            leadzero = true ;
         }
         //* Printing character (non-zero width) *
         if ( (cci < CCHARW_MAX) && (wPtr[wI] != nckNULLCHAR) )
         { 
            dst[dstI].chars[cci++] = wPtr[wI++] ;
         }
         //* Trailing, zero-width characters *
         if ( !leadzero )
         {
            while ( cci < CCHARW_MAX && wPtr[wI] != nckNULLCHAR && cPtr[wI] == ZERO )
               dst[dstI].chars[cci++] = wPtr[wI++] ;
         }
      }
   }
   return dstI ;

}  //* End ncChar2Cchart() *

//*************************
//*       Test02          *
//*************************
//******************************************************************************
//* This dialog tests the functionality of the NCurses methods that support    *
//* 'wide character' output; specifically, the WriteString() method.           *
//*                                                                            *
//* Note that we have provided only minimal support for 'wide' (supra-ASCII)   *
//* characters in the NCurses class because no one would use this class as a   *
//* base for serious code without also including the NcWindow or NcDialog      *
//* class as well.                                                             *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************
//*                                                                            *
//*                                                                            *
//******************************************************************************

void Test02 ( void )
{
   StatWin ( "Test #2 : NCurses multi-byte character output methods" ) ; // keep user up-to-date

   const wchar_t test01[] = 
   { L"This is a test of NCurses WriteString() and WriteChar() for 'wide' characters." } ;
   const wchar_t subHead01[] = { L"NEXT: Characters requiring more than 8 bits: 'supra-ASCII'" } ;
   const wchar_t endTest[] = { L" PRESS ANY KEY TO CONTINUE " } ;
   const wchar_t test02[] =  { L"** á Á à À ã Ã â Â ạ ā ả ȧ ă **" } ;
   const char cChinese01[]  = { "Chinese  - 这种评论荒谬绝伦。" } ;
   const char cChinese02[]  = { "Chinese  - 你为什么不能在家过年啊？" } ;
   const char cKorean01[]   = { "Korean   - 안녕하세요. 만나서 반갑습니다." } ;
   const char cKorean02[]   = { "Korean   - 저는 골프와 테니스 등 스포츠를 좋아해요." } ;
   const char cKanji01[]    = { "Kanji    - こんにちは。始めまして。" } ;
   const char cKanji02[]    = { "Kanji    - 始めまして。お元気ですか。" } ;
   const char cHiragana01[] = { "Hiragana - こんにちは。はじめまして。" } ;
   const char cHiragana02[] = { "Hiragana - はじめまして。おげんきですか。" } ;
   const char cRussian[]    = { "Russian  - Доброе утро. Как поживаете?" } ;
   const char cPolish[]     = { "Polish   - On patrzy na tę piękną dziewczynę." } ;
   const char cArabic[]     = { "هَل تَتَكَلَّم الْانْجِلِيزِيَّة؟"} ;
   const char cArabicHead[] = { "Arabic   - " } ;
   const char cHebrew[]     = { "הלו חבר! מה נשמע? התגעגעתי אליך." } ;
   const char cHebrewHead[] = { "Hebrew   - " } ;

   winPos wp( 3, 0 ) ;           // cursor positioning
   for ( short i = ZERO ; test01[i] != NULLCHAR ; i++ )  // testing WriteChar() method
      nc.WriteChar ( wp.ypos, wp.xpos++, test01[i], nc.bw ) ;
   wp = { short(wp.ypos+1), 0 } ;

   const wchar_t st01[] = 
   { //---------1---------2---------3---------4---------5---------6---------7
     L"This is a stress test. Attempting to induce wrap-around. Display "
      "should be truncated at the edge of the screen and NEVER wrap-around "
      "to the beginning of the next line. "
   } ;
   const wchar_t st02[] = 
   { //---------1---------2---------3---------4---------5---------6---------7
     L"Cursor positioning test. (arrows should be touching) >"
   } ;
   const wchar_t st03[] = 
   { //---------1---------2---------3---------4---------5---------6---------7
     L"< AND "
   } ;
   const wchar_t st04[] = 
   { //---------1---------2---------3---------4---------5---------6---------7
     L"á Á à À ã Ã >"
   } ;
   const wchar_t st05[] = 
   { //---------1---------2---------3---------4---------5---------6---------7
     L"< â Â ạ ā ả ȧ ă"
   } ;
   nc.WriteString ( wp.ypos++, wp.xpos, st01, nc.bw ) ;

   wp = nc.WriteString ( wp.ypos++, wp.xpos, st02, nc.bw ) ;
   wp = nc.WriteString ( wp.ypos, wp.xpos, st03, nc.bw ) ;
   wp = nc.WriteString ( wp.ypos, wp.xpos, st04, nc.bw ) ;
   wp = nc.WriteString ( wp.ypos, wp.xpos, st05, nc.bw ) ;
   wp = { short(wp.ypos + 1), ZERO } ;

   nc.WriteString ( wp.ypos++, wp.xpos++, subHead01, nc.blB | ncuATTR ) ;
   nc.WriteString ( wp.ypos++, wp.xpos, test02, nc.bw ) ;

   nc.WriteString ( wp.ypos++, wp.xpos, cChinese01, nc.bw ) ;
   nc.WriteString ( wp.ypos++, wp.xpos, cChinese02, nc.re ) ;
   nc.WriteString ( wp.ypos++, wp.xpos, cKorean01, nc.bw ) ;
   nc.WriteString ( wp.ypos++, wp.xpos, cKorean02, nc.gr ) ;
   nc.WriteString ( wp.ypos++, wp.xpos, cKanji01, nc.bw ) ;
   nc.WriteString ( wp.ypos++, wp.xpos, cKanji02, nc.br ) ;
   nc.WriteString ( wp.ypos++, wp.xpos, cHiragana01, nc.bw ) ;
   nc.WriteString ( wp.ypos++, wp.xpos, cHiragana02, nc.bl ) ;
   nc.WriteString ( wp.ypos++, wp.xpos, cRussian, nc.ma ) ;
   nc.WriteString ( wp.ypos++, wp.xpos, cPolish, nc.cy ) ;
   nc.WriteString ( wp.ypos, wp.xpos, cArabicHead, nc.br ) ;
   nc.WriteString ( wp.ypos++, (wp.xpos+30), cArabic, nc.br, true ) ;  // RTL output
   nc.WriteString ( wp.ypos, wp.xpos, cHebrewHead, nc.br ) ;
   nc.WriteString ( wp.ypos, (wp.xpos+42), cHebrew, nc.br, true ) ;  // RTL output

   wp = { short(wp.ypos + 2), short(wp.xpos + 7) } ;
   nc.WriteString ( wp.ypos++, wp.xpos, endTest, nc.reG ) ;
   nckPause() ;                     // wait for user input

   //* Clear screen of the test data *
   ClearTerminalWindow () ;

}  //* End Test02() *

//*************************
//*       Test03          *
//*************************
//******************************************************************************
//* This test exercises the NcWindow class methods related to 'wide character' *
//* output and UTF-8-encoded character output.                                 *
//*                                                                            *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************
//*                                                                            *
//*                                                                            *
//******************************************************************************

void Test03 ( void )
{
   StatWin ( "Test #3 : NcWindow multi-byte character output" ) ; // keep user up-to-date

   const wchar_t wTitle[] = 
   {
     L"                                         "
      "This is an NcWindow Object"
      "                                        "
   } ;
   const wchar_t endTest[] = { L" PRESS ANY KEY TO CONTINUE " } ;
   const wchar_t wWideString[2][90] = 
   { 
      L" 'Wide' characters (Polish alphabet): aA ąĄ bB cC ćĆ dD eE ęĘ fF gG hH iI jJ kK lL łŁ",
      L"                                      mM nN ńŃ oO óÓ pP rR sS śŚ tT uU wW yY zZ źŹ żŻ"
   } ;
   const char cSubHead01[2][101] = 
   { 
      "The data in this window are displayed using the NcWindow class WriteString() and WriteChar() methods",
      " for wchar_t 'wide' character data and for UTF-8-encoded char-width character data."
   } ;
   const char cSubHead02[] = { "UTF-8-encoded char strings:" } ;
   const char  cChinese01[] =  
   {
      " 这种评论荒谬绝伦。                       "
      "- Chinese: \"This comment is absolutely preposterous.\""
   } ;
   const char  cChinese02[] =  
   {
      " 你为什么不能在家过年啊？                 "
      "- Chinese: \"Why can't you celebrate the Spring Festival at home?\""
   } ;
   const char  cKorean01[] = 
   {
      " 안녕하세요. 만나서 반갑습니다.           "
      "- Korean: \"Hello. It is a pleasure to meet you.\""
   } ;
   const char  cKorean02[] = 
   {
      " 저는 골프와 테니스 등 스포츠를 좋아해요. "
      "- Korean: \"I love sports, especially golf and tennis.\""
   } ;
   const char  cKanji01[] = 
   {
      " こんにちは。始めまして。                 "
      "- Kanji: \"Hello. Nice to meet you.\""
   } ;
   const char  cKanji02[] = 
   {
      " 始めまして。お元気ですか。               "
      "- Kanji: \"Nice to meet you. How are you?\""
   } ;
   const char  cHiragana01[] = 
   {
      " こんにちは。はじめまして。               "
      "- Hiragana: \"Hello. Nice to meet you.\""
   } ;
   const char  cHiragana02[] = 
   {
      " はじめまして。おげんきですか。           "
      "- Hiragana: \"Nice to meet you. How are you?\""
   } ;
   const char  cRussian[] = 
   {
      " Доброе утро. Как поживаете?              "
      "- Russian: \"Good morning. How are you?\""
   } ;
   const char  cPolish[] = 
   {
      " On patrzy na tę piękną dziewczynę.       "
      "- Polish: \"He is looking at the beautiful girl.\""
   } ;
   const char cArabic[]     = { "هَل تَتَكَلَّم الْانْجِلِيزِيَّة؟"} ;
   const char cArabicTrans[] = { "- Arabic: \"Do you speak English?\"" } ;
   const char cHebrew[]     = { "הלו חבר! מה נשמע? התגעגעתי אליך." } ;
   const char cHebrewTrans[] = 
   { "- Hebrew: \"Hey, friend! What's new? I missed you.\"" } ;

   const short wLINES = 25, wCOLS = 109, ctrX = termCols / 2,
               wULY = 3, wULX = ctrX - (wCOLS / 2) ;
   const attr_t winColor = nc.bl ;

   NcWindow* ncWin = new NcWindow ( wLINES, wCOLS, wULY, wULX ) ;
   if ( (ncWin->OpenWindow ()) != ERR )
   {
      ncWin->DrawBorder ( winColor ) ;  // draw a border around our window
      LineDef  lDef(ncltHORIZ, ncltSINGLE, 2, ZERO, wCOLS, winColor) ;
      ncWin->DrawLine ( lDef ) ;

      //* Writing title line tests the gString version of WriteString()        *
      gString gs( wTitle ) ;
      ncWin->WriteString ( 1, 1, gs, (winColor | ncgATTR) ) ;

      winPos wp( 3, 1 ) ;           // cursor positioning

      //* Display native 'wide' data sample *
      ncWin->WriteString ( wp.ypos++, (wp.xpos+3), cSubHead01[0], nc.bwB ) ;
      ncWin->WriteString ( wp.ypos++, (wp.xpos+11), cSubHead01[1], nc.bwB ) ;
      ++wp.ypos ;
      ncWin->WriteString ( wp.ypos++, wp.xpos, wWideString[0], nc.bw ) ;
      for ( short i = ZERO ; wWideString[1][i] != NULLCHAR ; i++ )
         ncWin->WriteChar ( wp.ypos, wp.xpos++, wWideString[1][i], nc.bw ) ;
      
      //* Display the UTF-8 data *
      wp = { short(wp.ypos+2), 1 } ;
      for ( short i = ZERO ; cSubHead02[i] != NULLCHAR ; i++ )
         ncWin->WriteChar ( wp.ypos, wp.xpos++, cSubHead02[i], (nc.bwB | ncuATTR) ) ;
      wp = { short(wp.ypos+1), 1 } ;
      ncWin->WriteString ( wp.ypos++, wp.xpos, cChinese01, nc.bw ) ;
      ncWin->WriteString ( wp.ypos++, wp.xpos, cChinese02, nc.bw ) ;
      ncWin->WriteString ( wp.ypos++, wp.xpos, cKorean01, nc.bw ) ;
      ncWin->WriteString ( wp.ypos++, wp.xpos, cKorean02, nc.bw ) ;
      ncWin->WriteString ( wp.ypos++, wp.xpos, cKanji01, nc.bw ) ;
      ncWin->WriteString ( wp.ypos++, wp.xpos, cKanji02, nc.bw ) ;
      ncWin->WriteString ( wp.ypos++, wp.xpos, cHiragana01, nc.bw ) ;
      ncWin->WriteString ( wp.ypos++, wp.xpos, cHiragana02, nc.bw ) ;
      ncWin->WriteString ( wp.ypos++, wp.xpos, cRussian, nc.bw ) ;
      ncWin->WriteString ( wp.ypos++, wp.xpos, cPolish, nc.bw ) ;

      wp.xpos += 20 ;   // RTL output
      ncWin->WriteString ( wp.ypos, wp.xpos, cArabic, nc.bw, false, true ) ;
      ncWin->WriteString ( wp.ypos++, (wp.xpos+22), cArabicTrans, nc.bw ) ;
      wp.xpos += 12 ;   // RTL output
      ncWin->WriteString ( wp.ypos, wp.xpos, cHebrew, nc.bw, false, true ) ;
      ncWin->WriteString ( wp.ypos, (wp.xpos+10), cHebrewTrans, nc.bw ) ;

      ncWin->WriteString ( (wLINES - 2), (wCOLS / 2 - 14), endTest, nc.reG ) ;
      ncWin->RefreshWin () ;
      nckPause() ;                  // wait for keypress
   }
   else
   {  //* Most likely cause of dialog not opening is that the terminal window  *
      //* is too small. Give user a clue...                                    *
      short neededLines = (wULY + wLINES) >= minTERMROWS ? 
                          (wULY + wLINES) : minTERMROWS, 
            neededCols  = wCOLS >= minTERMCOLS ? wCOLS : minTERMCOLS ;
      gString gs ;
      gs.compose ( L" Sorry, size of terminal window must be at least %hd x %hd for this test. ", 
                   &neededCols, &neededLines ) ;
      StatWin ( gs.gstr(), nc.cyG ) ;
      sleep ( 5 ) ;
   }

   if ( ncWin != NULL )             // close the window
      delete ncWin ;

}  //* End Test03() *

//*************************
//*       Test04          *
//*************************
//******************************************************************************
//* This test exercises the NcDialog class methods related to 'wide character' *
//* output and UTF-8-encoded character output.                                 *
//*                                                                            *
//*                                                                            *
//*                                                                            *
//*                                                                            *
//* Input  :                                                                   *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************
//*                                                                            *
//*                                                                            *
//******************************************************************************

void Test04 ( void )
{
   StatWin ( "Test #4 : NcDialog class multi-byte character output" ) ; // keep user up-to-date

   const char cPolishAlphaHdr[] = 
   {
      " 'Wide' characters (Polish alphabet): "
   } ;
   const wchar_t wPolishAlphabet[3][50] = 
   { 
      L"aA ąĄ bB cC ćĆ dD eE ęĘ fF gG hH iI jJ kK lL łŁ",
      L"                                      ",
      L"mM nN ńŃ oO óÓ pP rR sS śŚ tT uU wW yY zZ źŹ żŻ",
   } ;
   const char cSubHead01[2][128] = 
   { 
      "The data in this window are displayed using the NcDialog class "
      "WriteString() and WriteChar() methods ",
      "         for wchar_t 'wide' character data and for UTF-8-encoded "
      "char-width character data.          "
   } ;
   const char cSubHead02[] = { "UTF-8-encoded char strings:" } ;
   const char  cChinese01[] =  
   {
      " 这种评论荒谬绝伦。                       "
      "- Chinese: \"This comment is absolutely preposterous.\""
   } ;
   const char  cChinese02[] =  
   {
      " 你为什么不能在家过年啊？                 "
      "- Chinese: \"Why can't you celebrate the Spring Festival at home?\""
   } ;
   const char  cKorean01[] = 
   {
      " 안녕하세요. 만나서 반갑습니다.           "
      "- Korean: \"Hello. It is a pleasure to meet you.\""
   } ;
   const char  cKorean02[] = 
   {
      " 저는 골프와 테니스 등 스포츠를 좋아해요. "
      "- Korean: \"I love sports, especially golf and tennis.\""
   } ;
   const char  cKanji01[] = 
   {
      " こんにちは。始めまして。                 "
      "- Kanji: \"Hello. Nice to meet you.\""
   } ;
   const char  cKanji02[] = 
   {
      " 始めまして。お元気ですか。               "
      "- Kanji: \"Nice to meet you. How are you?\""
   } ;
   const char  cHiragana01[] = 
   {
      " こんにちは。はじめまして。               "
      "- Hiragana: \"Hello. Nice to meet you.\""
   } ;
   const char  cHiragana02[] = 
   {
      " はじめまして。おげんきですか。           "
      "- Hiragana: \"Nice to meet you. How are you?\""
   } ;
   const char  cRussian[] = 
   {
      " Доброе утро. Как поживаете?              "
      "- Russian: \"Good morning. How are you?\""
   } ;
   const char  cPolish[] = 
   {
      " On patrzy na tę piękną dziewczynę.       "
      "- Polish: \"He is looking at the beautiful girl.\""
   } ;
   const char cArabic[]      = { "هَل تَتَكَلَّم الْانْجِلِيزِيَّة؟"} ;
   const char cArabicTrans[] = { "- Arabic: \"Do you speak English?\"" } ;
   const char cHebrew[]     = { "הלו חבר! מה נשמע? התגעגעתי אליך." } ;
   const char cHebrewTrans[] = 
   { "- Hebrew: \"Hey, friend! What's new? I missed you.\"" } ;

   const short dialogROWS = 22 ;   // display lines
   const short dialogCOLS = 109 ;  // display columns
   short ctrY    = termRows/2 - 2, // dialog center in Y
         ctrX    = termCols/2,     // dialog center in X
         //* Upper left corner in Y (cannot obscure status window) *
         ulY     = (ctrY - dialogROWS/2) >= minTestULY ? 
                   (ctrY - dialogROWS/2) : minTestULY,
         ulX     = ctrX - dialogCOLS/2 ; // upper left corner in X
attr_t   dColor  = nc.blR ;        // display color
InitCtrl ic =                      // array of dialog control info
{
   //* 'OK' pushbutton - - - - - - - - - - - - - - - - donePush *
   dctPUSHBUTTON,                // type:      
   rbtTYPES,                     // rbSubtype: (n/a)
   false,                        // rbSelect:  (n/a)
   short(dialogROWS - 2),        // ulY:       upper left corner in Y
   short(dialogCOLS / 2 - 14),   // ulX:       upper left corner in X
   1,                            // lines:     control lines
   27,                           // cols:      control columns
   " PRESS ANY KEY TO CONTINUE ",// dispText:  
   nc.gyR,                       // nColor:    non-focus color
   nc.gyre | ncbATTR,            // fColor:    focus color
   tbPrint,                      // filter:    (n/a)
   "",                           // label:     (n/a)
   ZERO,                         // labY:      (n/a)
   ZERO,                         // labX       (n/a)
   ddBoxTYPES,                   // exType:    (n/a)
   1,                            // scrItems:  (n/a)
   1,                            // scrSel:    (n/a)
   NULL,                         // scrColor:  (n/a)
   NULL,                         // spinData:  (n/a)
   true,                         // active:    allow control to gain focus
   NULL                          // nextCtrl:  link in next structure
} ;

   //* Initial parameters for dialog window *
   InitNcDialog dInit( dialogROWS,     // number of display lines
                       dialogCOLS,     // number of display columns
                       ulY,            // Y offset from upper-left of terminal 
                       ulX,            // X offset from upper-left of terminal 
                       "  This is an NcDialog Object  ", // dialog title
                       ncltSINGLE,     // border line-style
                       dColor,         // border color attribute
                       dColor,         // interior color attribute
                       &ic             // pointer to list of control definitions
                     ) ;

   //* Instantiate the dialog window *
   NcDialog* dp = new NcDialog ( dInit ) ;

   if ( (dp->OpenWindow()) == OK )
   {
      winPos wp( 1, 1 ) ;        // cursor positioning

      //* Writing explanation lines tests the gString version of WriteString() *
      gString gs( cSubHead01[0] ) ;
      dp->WriteString ( wp.ypos++, (wp.xpos+3), gs, nc.bw ) ;
      gs = cSubHead01[1] ; 
      dp->WriteString ( wp.ypos++, (wp.xpos+3), gs, nc.bw ) ;

      //* Display native 'wide' data sample *
      //* Exercise char-width version of WriteChar(), then wchar_t version of  *
      //* WriteString(), then wchar_t version of WriteChar().                  *
      //* Note on indexing a UTF-8-encoded character string: we 'know' that    *
      //* cPolishAlphaHdr[] is pure ASCII, so we use an index increment. If the*
      //* string had multi-byte chacters, we would need a more sophisticated   *
      //* indexing scheme. See other test cases in this application for more   *
      //* information.                                                         *
      wp = { short(wp.ypos+1), short(wp.xpos+6) } ;
      for ( short i = ZERO ; cPolishAlphaHdr[i] != NULLCHAR ; i++ )
         wp = dp->WriteChar ( wp.ypos, wp.xpos, cPolishAlphaHdr[i], dColor ) ;
      wp = dp->WriteString ( wp.ypos, wp.xpos, wPolishAlphabet[0], dColor ) ;
      wp = { short(wp.ypos + 1), 7 } ;
      wp = dp->WriteString ( wp.ypos, wp.xpos, wPolishAlphabet[1], dColor ) ;
      for ( short i = ZERO ; wPolishAlphabet[2][i] != NULLCHAR ; i++ )
         wp = dp->WriteChar ( wp.ypos, wp.xpos, wPolishAlphabet[2][i], dColor ) ;
      wp.xpos = 1 ;

      //* Display the UTF-8 data *
      ++wp.ypos ;
      dp->WriteString ( wp.ypos++, wp.xpos, cSubHead02, dColor | ncbATTR | ncuATTR ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, cChinese01, dColor ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, cChinese02, dColor ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, cKorean01, dColor ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, cKorean02, dColor ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, cKanji01, dColor ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, cKanji02, dColor ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, cHiragana01, dColor ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, cHiragana02, dColor ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, cRussian, dColor ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, cPolish, dColor ) ;

      wp.xpos += 20 ;   // RTL output
      dp->WriteString ( wp.ypos, wp.xpos, cArabic, dColor, false, true ) ;
      dp->WriteString ( wp.ypos++, (wp.xpos+22), cArabicTrans, dColor ) ;
      wp.xpos += 12 ;   // RTL output
      dp->WriteString ( wp.ypos, wp.xpos, cHebrew, dColor, false, true ) ;
      dp->WriteString ( wp.ypos, (wp.xpos+10), cHebrewTrans, dColor ) ;

      dp->RefreshWin () ;

      //* The Pushbutton has input focus - wait for user to press a key *
      uiInfo Info ;
      dp->EditPushbutton ( Info ) ;
   }
   else
   {  //* Most likely cause of dialog not opening is that the terminal window  *
      //* is too small. Give user a clue...                                    *
      short neededLines = (dialogROWS + minTestULY) >= minTERMROWS ?
                          (dialogROWS + minTestULY) : minTERMROWS, 
            neededCols  = dialogCOLS >= minTERMCOLS ? dialogCOLS : minTERMCOLS ;
      gString gs ;
      gs.compose ( L" Sorry, size of terminal window must be at least %hd x %hd for this test. ", 
                   &neededCols, &neededLines ) ;
      StatWin ( gs.gstr(), nc.cyG ) ;
      sleep ( 5 ) ;
   }

   if ( dp != NULL )                         // close the window
      delete ( dp ) ;

}  //* End Test04() *

//*************************
//*       Test05          *
//*************************
//******************************************************************************
//* This dialog tests the functionality of wrap-around control for the NcDialog*
//* output methods. The ncurses/ncursesw library allows text to wrap around    *
//* at the end of the current line and continues output at the beginning of    *
//* the next line of the window. This makes for an unacceptably-ugly display.  *
//*                                                                            *
//* For ASCII text, it is a simple matter to truncate the output at the right  *
//* edge of the dialog window because one character requires one display       *
//* column. However, in the international world, we need to be aware of how    *
//* many display columns are required for a character or for a string to be    *
//* displayed, so that we can avoid wrap-around.                               *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************
//*                                                                            *
//*                                                                            *
//******************************************************************************

void Test05 ( void )
{
   StatWin ( "Test #5 : NcDialog output - Wrap-around test  " ) ; // keep user up-to-date
   
static const short dialogROWS = 33 ;         // display lines
static const short dialogCOLS = 80 ;         // display columns
short    ctrY    = termRows/2,               // dialog center in Y
         ctrX    = termCols/2,               // dialog center in X
         //* Upper left corner in Y (cannot obscure status window) *
         ulY     = (ctrY - dialogROWS/2) >= minTestULY ? 
                   (ctrY - dialogROWS/2) : minTestULY,
         ulX     = ctrX - dialogCOLS/2,      // upper left corner in X
         leftCtr = dialogCOLS / 2 - 19 ;
attr_t   dColor  = nc.blR,                   // display color
         legendColor = nc.blG ;
InitCtrl ic =                                // array of dialog control info
{
   //* 'OK' pushbutton - - - - - - - - - - - - - - - - donePush *
   dctPUSHBUTTON,                // type:      
   rbtTYPES,                     // rbSubtype: (n/a)
   false,                        // rbSelect:  (n/a)
   short(dialogROWS - 2),        // ulY:       upper left corner in Y
   short(dialogCOLS / 2 - 14),   // ulX:       upper left corner in X
   1,                            // lines:     control lines
   27,                           // cols:      control columns
   " PRESS ANY KEY TO CONTINUE ",// dispText:  
   nc.gyR,                       // nColor:    non-focus color
   nc.gyre | ncbATTR,            // fColor:    focus color
   tbPrint,                      // filter:    (n/a)
   "",                           // label:     (n/a)
   ZERO,                         // labY:      (n/a)
   ZERO,                         // labX       (n/a)
   ddBoxTYPES,                   // exType:    (n/a)
   1,                            // scrItems:  (n/a)
   1,                            // scrSel:    (n/a)
   NULL,                         // scrColor:  (n/a)
   NULL,                         // spinData:  (n/a)
   true,                         // active:    allow control to gain focus
   NULL                          // nextCtrl:  link in next structure
} ;

   //* Initial parameters for dialog window *
   InitNcDialog dInit( dialogROWS,     // number of display lines
                       dialogCOLS,     // number of display columns
                       ulY,            // Y offset from upper-left of terminal 
                       ulX,            // X offset from upper-left of terminal 
                       "  NcDialog Class Wrap-around Test  ", // dialog title
                       ncltSINGLE,     // border line-style
                       dColor,         // border color attribute
                       dColor,         // interior color attribute
                       &ic             // pointer to list of control definitions
                     ) ;

   //* Instantiate the dialog window *
   NcDialog* dp = new NcDialog ( dInit ) ;

   if ( (dp->OpenWindow()) == OK )
   {
      winPos wp( (short)(dialogROWS - 9), 10 ) ;
      const char* exNote =    // explanatory Note:
      { //123456789012345678901234567890123456789012345678901234567890
         "Note that the ncurses library primitives allow wrap-around\n"
         "at the right edge of the window, with text continuing at\n"
         "the beginning of the next line. The NcDialog class captures\n"
         "as much of this untidy behaviour as possible.\n"
         "Captured string and char output wrap-arounds are truncated.",
      } ;
      dp->WriteParagraph ( wp, exNote, dColor ) ;

      //* Display a grid for ease in interpreting the results *
      for ( short i = ZERO ; i < dialogROWS ; i++ )
      {
         gString  nBuff( "%02hd", &i ) ;
         dp->WriteString ( i, 6, nBuff, dColor ) ;
         dp->WriteString ( i, (short)(dialogCOLS-10), nBuff, dColor ) ;
      }

      //* Start-outside-window test: Any attempt to BEGIN writing outside the  *
      //* boundaries of the window will be captured, and the output will begin *
      //* at offset ZERO/ZERO (upper-left corner) of the dialog window.        *
      //* Instead of simply not writing the data, thus creating a mystery, this*
      //* feature allows the application programmer to have immediate, visual  *
      //* feedback about specifying a cursor position that is out of range.    *
      #if 0    // TEST CURSOR RANGE - ws UTF-8
      dp->WriteString ( dialogROWS, 1, "Begin write below window.", nc.bw ) ;
      dp->RefreshWin () ;sleep ( 1 ) ;
      dp->WriteString ( -2, 1, "Begin write above window.", nc.bw ) ;
      dp->RefreshWin () ;sleep ( 1 ) ;
      dp->WriteString ( 3, dialogCOLS, "Begin write to right of window.", nc.bw ) ;
      dp->RefreshWin () ; sleep ( 1 ) ;
      dp->WriteString ( 3, -4, "Begin write to left of window.", nc.bw ) ;
      dp->RefreshWin () ; sleep ( 1 ) ;
      #elif 0  // ws wchar_t
      dp->WriteString ( dialogROWS, 1, L"Begin write below window.", nc.bw ) ;
      dp->RefreshWin () ;sleep ( 1 ) ;
      dp->WriteString ( -2, 1, L"Begin write above window.", nc.bw ) ;
      dp->RefreshWin () ;sleep ( 1 ) ;
      dp->WriteString ( 3, dialogCOLS, L"Begin write to right of window.", nc.bw ) ;
      dp->RefreshWin () ; sleep ( 1 ) ;
      dp->WriteString ( 3, -4, L"Begin write to left of window.", nc.bw ) ;
      dp->RefreshWin () ; sleep ( 1 ) ;
      #elif 0  // wc UTF-8 and wchar_t
      char  cChar = 'X' ;
      wchar_t  wChar = 'Y' ;
      dp->WriteChar ( dialogROWS, 1, acsINSECT, nc.bw | ncaATTR ) ;
      dp->RefreshWin () ;sleep ( 3 ) ;
      dp->WriteChar ( dialogROWS, 1, wChar, nc.bw ) ;
      dp->RefreshWin () ;sleep ( 1 ) ;
      dp->WriteChar ( 2, dialogCOLS, &cChar, nc.bw ) ;
      dp->RefreshWin () ;sleep ( 1 ) ;
      dp->WriteChar ( 2, dialogCOLS, wChar, nc.bw ) ;
      dp->RefreshWin () ;sleep ( 1 ) ;

      dp->WriteChar ( -2, 1, &cChar, nc.bw ) ;
      dp->RefreshWin () ;sleep ( 1 ) ;
      dp->WriteChar ( -2, 1, wChar, nc.bw ) ;
      dp->RefreshWin () ;sleep ( 1 ) ;
      dp->WriteChar ( 2, -2, &cChar, nc.bw ) ;
      dp->RefreshWin () ;sleep ( 1 ) ;
      dp->WriteChar ( 2, -2, wChar, nc.bw ) ;
      dp->RefreshWin () ;sleep ( 1 ) ;
      #endif   // TEST CURSOR RANGE

      wp.ypos = 2 ;
      //* Wrap to next line test - WriteString() UTF-8 *
      dp->WriteString ( wp.ypos, (short)(leftCtr+12), 
                         "Begin ASCII write near right border.", legendColor ) ;
      dp->WriteString ( wp.ypos, (short)(dialogCOLS - 7), 
                                         "ABCDEFGHIJKLMNOPQRSTUVWXYZ", nc.bw ) ;

      wp.ypos += 2 ;
      dp->WriteString ( wp.ypos, (short)(leftCtr-2), 
           "Begin multi-byte-char write near right border (a).", legendColor ) ;
      dp->WriteString ( wp.ypos, (short)(dialogCOLS - 7), "-你为什么不能在家过年啊？", nc.bl ) ;
      
      wp.ypos += 2 ;
      dp->WriteString ( wp.ypos, (short)(leftCtr-2), 
           "Begin multi-byte-char write near right border (b).", legendColor ) ;
      dp->WriteString ( wp.ypos, (short)(dialogCOLS - 7), "你为什么不能在家过年啊？", nc.bl ) ;
      
      //* Wrap to next line test - WriteString() wchar_t *
      wp.ypos += 2 ;
      dp->WriteString ( wp.ypos, (short)(leftCtr+7), 
                    "Begin wide ASCII write near right border.", legendColor ) ;
      dp->WriteString ( wp.ypos, (short)(dialogCOLS - 7), 
                                        L"ABCDEFGHIJKLMNOPQRSTUVWXYZ", nc.br ) ;

      wp.ypos += 2 ;
      dp->WriteString ( wp.ypos, (short)(leftCtr-7), 
      "Begin wide multi-byte-char write near right border (a).", legendColor ) ;
      char doubleColCharsA[] = { "-你为什么不能在家过年啊？" } ;
      wchar_t  wBuff[128] ;
      mbstowcs ( wBuff, doubleColCharsA, 50 ) ;
      dp->WriteString ( wp.ypos, (short)(dialogCOLS - 7), wBuff, nc.re ) ;

      wp.ypos += 2 ;
      dp->WriteString ( wp.ypos, (short)(leftCtr-7), 
      "Begin wide multi-byte-char write near right border (b).", legendColor ) ;
      char doubleColCharsB[] = { "你为什么不能在家过年啊？" } ;
      mbstowcs ( wBuff, doubleColCharsB, 50 ) ;
      dp->WriteString ( wp.ypos, (short)(dialogCOLS - 7), wBuff, nc.bw ) ;

      //* Wrap to next line test - WriteChar() UTF-8 *
      wp.ypos += 2 ;
      dp->WriteString ( wp.ypos, (short)(leftCtr+16), 
                             "ASCII character at right border.", legendColor ) ;
      char aChar = 'A' ;
      dp->WriteChar ( wp.ypos, (short)(dialogCOLS-1), aChar, nc.bw ) ;

      wp.ypos += 2 ;
      dp->WriteString ( wp.ypos, (short)(leftCtr + 1),
                "UTF-8 multi-column character at right border-1.", legendColor ) ;
      char  cChar[] = "你" ;
      dp->WriteChar ( wp.ypos, (short)(dialogCOLS-2), cChar, nc.cy ) ;

      wp.ypos += 2 ;
      dp->WriteString ( wp.ypos, (short)(leftCtr+3), 
                  "UTF-8 multi-column character at right border.", legendColor ) ;
      dp->WriteChar ( wp.ypos, (short)(dialogCOLS-1), cChar, nc.cy ) ;

      //* Wrap to next line test - WriteChar() wchar_t *
      wp.ypos += 2 ;
      dp->WriteString ( wp.ypos, (short)(leftCtr+2), 
                 "Wide multi-column character at right border-1.", legendColor ) ;
      wchar_t  wChar ;
      mbtowc ( &wChar, cChar, 4 ) ;
      dp->WriteChar ( wp.ypos, (short)(dialogCOLS-2), wChar, nc.re ) ;

      wp.ypos += 2 ;
      dp->WriteString ( wp.ypos, (short)(leftCtr+4), 
                   "Wide multi-column character at right border.", legendColor ) ;
      dp->WriteChar ( wp.ypos, (short)(dialogCOLS-1), wChar, nc.ma ) ;

      #if 0    // SCROLLING IS TURNED OFF, SO THIS LINE DOES NOT WRAP
      dp->WriteString ( (short)(dialogROWS-1), (short)(dialogCOLS-5), 
                                               "ASCIIASCIIASCIIASCII", nc.bw ) ;
      #endif   // SCROLLING TEST

      dp->RefreshWin () ;

      //* The Pushbutton has input focus - wait for user to press a key *
      uiInfo Info ;
      dp->EditPushbutton ( Info ) ;
   }
   else
   {  //* Most likely cause of dialog not opening is that the terminal window  *
      //* is too small. Give user a clue...                                    *
      short neededLines = (dialogROWS + minTestULY) >= minTERMROWS ?
                          (dialogROWS + minTestULY) : minTERMROWS, 
            neededCols  = dialogCOLS >= minTERMCOLS ? dialogCOLS : minTERMCOLS ;
      gString gs ;
      gs.compose ( L" Sorry, size of terminal window must be at least %hd x %hd for this test. ", 
                   &neededCols, &neededLines ) ;
      StatWin ( gs.gstr(), nc.cyG ) ;
      sleep ( 5 ) ;
   }

   if ( dp != NULL )                         // close the window
      delete ( dp ) ;

}  //* End Test05() *

//*************************
//*       Test06          *
//*************************
//******************************************************************************
//* This dialog tests the functionality of the gString class.                  *
//* The gString class implements implicit conversion between UTF-8-encoded     *
//* char data and wchar_t 'wide' data, and provides simple output formatting   *
//* tools.                                                                     *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************
//*                                                                            *
//*                                                                            *
//******************************************************************************

void Test06 ( void )
{
   StatWin ( "Test #6 : gString class functionality test " ) ; // keep user up-to-date

   short neededLines = (intfmtROWS + 1), 
         neededCols  = intfmtCOLS ;
   if ( (termRows >= neededLines) && (termCols >= neededCols) )
   {
      gStringTest* gsTest = new gStringTest ( termRows, termCols, minTestULY ) ;
      if ( (gsTest->gtDialogOpened ()) != false )
      {
         gsTest->gtInteract () ;
      }
      else { /* Memory allocation error - unlikely */ }

      if ( gsTest != NULL )      // close the dialog
         delete gsTest ;
   }
   else
   {  //* Most likely cause of dialog not opening is that the terminal window  *
      //* is too small. Give user a clue...                                    *
      gString gs ;
      gs.compose ( L" Sorry, size of terminal window must be "
                    "at least %hd x %hd for this test. ", 
                   &neededCols, &neededLines ) ;
      StatWin ( gs.gstr(), nc.cyG ) ;
      sleep ( 4 ) ;
   }

}  //* End Test06() *

//*************************
//*       Test07          *
//*************************
//******************************************************************************
//* Mixed LTR and RTL data.                                                    *
//*                                                                            *
//*                                                                            *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void Test07 ( void )
{
   StatWin ( "Test #7 : Examples included in TexInfo docs " ) ; // keep user up-to-date

   short neededLines = (twinROWS + 1), 
         neededCols  = twinCOLS ;
   if ( (termRows >= neededLines) && (termCols >= neededCols) )
   {
      infoDocTest* idTest = new infoDocTest ( termRows, termCols, minTestULY ) ;
      if ( (idTest->idDialogOpened ()) != false )
      {
         idTest->idInteract () ;
      }
      else { /* Memory allocation error - unlikely */ }

      if ( idTest != NULL )      // close the dialog
         delete idTest ;
   }
   else
   {  //* Most likely cause of dialog not opening is that the terminal window  *
      //* is too small. Give user a clue...                                    *
      gString gs ;
      gs.compose ( L" Sorry, size of terminal window must be "
                    "at least %hd x %hd for this test. ", 
                   &neededCols, &neededLines ) ;
      StatWin ( gs.gstr(), nc.cyG ) ;
      sleep ( 4 ) ;
   }

}  //* End Test07() *

//*************************
//*       Test08          *
//*************************
//******************************************************************************
//* This dialog tests the functionality of multi-byte characters within        *
//* NcDialog Control objects.                                                  *
//*                                                                            *
//*                                                                            *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************
//*                                                                            *
//*                                                                            *
//******************************************************************************

void Test08 ( void )
{
   StatWin ( "Test #8 : Textbox and Radio button controls "
             "- labels with hotkeys (UTF-8)" ) ; // keep user up-to-date

static const short dialogROWS = 12 ;         // display lines
static const short dialogCOLS = 70 ;         // display columns
static const short controlsDEFINED = 15 ;    // # of controls in dialog
short    ctrY    = termRows/2,               // dialog center in Y
         ctrX    = termCols/2,               // dialog center in X
         //* Upper left corner in Y (cannot obscure status window) *
         ulY     = (ctrY - dialogROWS/2) >= minTestULY ? 
                   (ctrY - dialogROWS/2) : minTestULY,
         ulX     = ctrX - dialogCOLS / 2 ;   // upper left corner in X
attr_t   dColor = nc.blR ;

   //****************************************
   //* Initial parameters for dialog window *
   //****************************************
InitCtrl ic[controlsDEFINED] =      // array of dialog control info
{
   {  //* 'DONE' pushbutton - - - - - - - - - - - - - - - - - - - - T8donePush *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(dialogROWS - 2),        // ulY:       upper left corner in Y
      short(dialogCOLS / 2 - 4),    // ulX:       upper left corner in X
      1,                            // lines:     control lines
      8,                            // cols:      control columns
      "  DON^E  ",                  // dispText:  
      nc.gyR,                       // nColor:    non-focus color
      nc.gyre | ncbATTR,            // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "",                           // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[1],                       // nextCtrl:  link in next structure
   },
   {  //* Radio Button Group A - - - - - - - - - - - - - - - - - -  T8groupA_1 *
      dctRADIOBUTTON,               // type:      
      rbtS3s,                       // rbSubtype: standard, 3 chars wide
      true,                         // rbSelect:  default selection
      2,                            // ulY:       upper left corner in Y
      4,                            // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a)
      nc.bw,                        // nColor:    non-focus color
      nc.cyG,                       // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "^rbtS3s Radio Button 一",     // label:
      ZERO,                         // labY:      
      5,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[2],                       // nextCtrl:  link in next structure
   },
   {  //* Radio Button Group A - - - - - - - - - - - - - - - - - -  T8groupA_2 *
      dctRADIOBUTTON,               // type:      
      rbtS3s,                       // rbSubtype: standard, 3 chars wide
      false,                        // rbSelect:  default selection
      3,                            // ulY:       upper left corner in Y
      4,                            // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a)
      nc.bw,                        // nColor:    non-focus color
      nc.cyG,                       // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "r^btS3s Radio Button 二",     // label:
      ZERO,                         // labY:      
      5,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[3],                       // nextCtrl:  link in next structure
   },
   {  //* Radio Button Group A - - - - - - - - - - - - - - - - - -  T8groupA_3 *
      dctRADIOBUTTON,               // type:      
      rbtS3s,                       // rbSubtype: standard, 3 chars wide
      false,                        // rbSelect:  default selection
      4,                            // ulY:       upper left corner in Y
      4,                            // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a)
      nc.bw,                        // nColor:    non-focus color
      nc.cyG,                       // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "rbt^S3s Radio Button 三",     // label:
      ZERO,                         // labY:      
      5,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[4],                       // nextCtrl:  link in next structure
   },
   {  //* Radio Button Group A - - - - - - - - - - - - - - - - - -  T8groupA_4 *
      dctRADIOBUTTON,               // type:      
      rbtS3s,                       // rbSubtype: standard, 3 chars wide
      false,                        // rbSelect:  default selection
      5,                            // ulY:       upper left corner in Y
      4,                            // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a)
      nc.bw,                        // nColor:    non-focus color
      nc.cyG,                       // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "rbtS3s Radio B^utton 四",     // label:
      ZERO,                         // labY:      
      5,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[5],                       // nextCtrl:  link in next structure
   },
   {  //* Radio Button Group B - - - - - - - - - - - - - - - - - -  T8groupB_1 *
      dctRADIOBUTTON,               // type:      
      rbtS3s,                       // rbSubtype: standard, 3 chars wide
      true,                         // rbSelect:  default selection
      2,                            // ulY:       upper left corner in Y
      39,                           // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a)
      nc.bw,                        // nColor:    non-focus color
      nc.cyG,                       // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "rbtS3s Radio But^ton 五",     // label:
      ZERO,                         // labY:      
      5,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[6],                       // nextCtrl:  link in next structure
   },
   {  //* Radio Button Group B - - - - - - - - - - - - - - - - - -  T8groupB_2 *
      dctRADIOBUTTON,               // type:      
      rbtS3s,                       // rbSubtype: standard, 3 chars wide
      false,                        // rbSelect:  default selection
      3,                            // ulY:       upper left corner in Y
      39,                           // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a)
      nc.bw,                        // nColor:    non-focus color
      nc.cyG,                       // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "rbtS3s Radio Butt^on 六",     // label:
      ZERO,                         // labY:      
      5,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[7],                       // nextCtrl:  link in next structure
   },
   {  //* Radio Button Group B - - - - - - - - - - - - - - - - - -  T8groupB_3 *
      dctRADIOBUTTON,               // type:      
      rbtS3s,                       // rbSubtype: standard, 3 chars wide
      false,                        // rbSelect:  default selection
      4,                            // ulY:       upper left corner in Y
      39,                           // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a)
      nc.bw,                        // nColor:    non-focus color
      nc.cyG,                       // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "rbtS3s Radio Butto^n 七",     // label:
      ZERO,                         // labY:      
      5,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[8],                       // nextCtrl:  link in next structure
   },
   {  //* Stand-alone Radio Button 8 - - - - - - - - - - - - - - T8standAlone8 *
      dctRADIOBUTTON,               // type:      
      rbtS3s,                       // rbSubtype: standard, 3 chars wide
      false,                        // rbSelect:  default selection
      6,                            // ulY:       upper left corner in Y
      39,                           // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a)
      nc.bw,                        // nColor:    non-focus color
      nc.cyG,                       // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "St^and-alone 按钮 八",          // label:
      ZERO,                         // labY:      
      5,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[9],                       // nextCtrl:  link in next structure
   },
   {  //* Stand-alone Radio Button 8 - - - - - - - - - - - - - - T8standAlone9 *
      dctRADIOBUTTON,               // type:      
      rbtS3s,                       // rbSubtype: standard, 3 chars wide
      true,                         // rbSelect:  default selection
      8,                            // ulY:       upper left corner in Y
      39,                           // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a)
      nc.bw,                        // nColor:    non-focus color
      nc.cyG,                       // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "Stan^d-alone 按钮 九",          // label:
      ZERO,                         // labY:      
      5,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[10],                      // nextCtrl:  link in next structure
   },
   {  //* Text Box - - - - - - - - - - - - - - - - - - - - - - - - -  T8tbText *
      dctTEXTBOX,                   // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      9,                            // ulY:       upper left corner in Y
      11,                           // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      8,                            // cols:      control columns
      NULL,                         // dispText:  (initially blank)
      nc.bw,                        // nColor:    non-focus color
      nc.reG,                       // fColor:    focus color
      tbPrint,                      // filter:    any printing character
      "Text Bo^x",                  // label:     
      ZERO,                         // labY:      
      -9,                           // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    cannot gain focus
      &ic[11],                      // nextCtrl:  link in next structure
   },
   {  //* Non-active Text Box holds Group 'A' selection - - - - - - - T8utBOX1 *
      dctTEXTBOX,                   // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      7,                            // ulY:       upper left corner in Y
      21,                           // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      4,                            // cols:      control columns
      "",                           // dispText:  (initially blank)
      nc.grG,                       // nColor:    non-focus color
      nc.grG,                       // fColor:    focus color
      tbPrint,                      // filter:    any printing character
      "Group A Selection",          // label:     
      ZERO,                         // labY:      
      -19,                          // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      false,                        // active:    cannot gain focus
      &ic[12],                      // nextCtrl:  link in next structure
   },
   {  //* Non-active Text Box holds Group 'B' selection - - - - index T8utBOX2 *
      dctTEXTBOX,                   // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      8,                            // ulY:       upper left corner in Y
      21,                           // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      4,                            // cols:      control columns
      NULL,                         // dispText:  (initially blank)
      nc.grB,                       // nColor:    non-focus color
      nc.grB,                       // fColor:    focus color
      tbPrint,                      // filter:    any printing character
      "Group B Selection",          // label:     
      ZERO,                         // labY:      
      -19,                          // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      false,                        // active:    cannot gain focus
      &ic[13],                      // nextCtrl:  link in next structure
   },
   {  //* Non-active Text Box holds Button '8' state  - - - - - - - - T8utBOX3 *
      dctTEXTBOX,                   // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      7,                            // ulY:       upper left corner in Y
      57,                           // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      12,                           // cols:      control columns
      NULL,                         // dispText:  (initially blank)
      nc.grG,                       // nColor:    non-focus color
      nc.grG,                       // fColor:    focus color
      tbPrint,                      // filter:    any printing character
      "按钮 八 状态",                    // label:     ("Button #8 State")
      ZERO,                         // labY:      
      -13,                          // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      false,                        // active:    cannot gain focus
      &ic[14],                      // nextCtrl:  link in next structure
   },
   {  //* Non-active Text Box holds Button '9' state  - - - - - - - - T8utBOX4 *
      dctTEXTBOX,                   // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      9,                            // ulY:       upper left corner in Y
      57,                           // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      12,                           // cols:      control columns
      NULL,                         // dispText:  (initially blank)
      nc.grG,                       // nColor:    non-focus color
      nc.grG,                       // fColor:    focus color
      tbPrint,                      // filter:    any printing character
      "按钮 九 状态",                    // label:     ("Button #9 State")
      ZERO,                         // labY:      
      -13,                          // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      false,                        // active:    cannot gain focus
      NULL,                         // nextCtrl:  link in next structure
   },
} ;

   //* Initial parameters for dialog window *
   InitNcDialog dInit( dialogROWS,     // number of display lines
                       dialogCOLS,     // number of display columns
                       ulY,            // Y offset from upper-left of terminal 
                       ulX,            // X offset from upper-left of terminal 
                       "  UTF-8 Encoding 文本框(Text-Boxes) and 单选按钮(Radio-Buttons) ",
                       ncltSINGLE,     // border line-style
                       dColor,         // border color attribute
                       dColor,         // interior color attribute
                       ic              // pointer to list of control definitions
                     ) ;

   //* Instantiate the dialog window *
   NcDialog* dp = new NcDialog ( dInit ) ;
   Test08Dialog = dp ;              // give callback method access to the dialog

   //* Open the dialog window *
   if ( (dp->OpenWindow()) == OK )
   {
      dp->DrawBox ( 1,  2, 6, 31, dColor, " 组(Group) A " ) ; // group 1 box
      dp->DrawBox ( 1, 37, 5, 31, dColor, " 组(Group) B " ) ; // group 2 box
      dp->RefreshWin () ;

      //* Set radio buttons 1, 2, 3, and 4 as an exclusive-OR group.*
      //* Set radio buttons 5, 6, and 7 as an exclusive-OR group.   *
      //* (only one from each group can be selected at any moment). *
      short XorGroup1[5] = { 1, 2, 3, 4, -1 } ;
      dp->GroupRadiobuttons ( XorGroup1 ) ;
      short XorGroup2[4] = { 5, 6, 7, -1 } ;
      dp->GroupRadiobuttons ( XorGroup2 ) ;

      //* Establish a call-back method that can be called from within the *
      //* NcDialog code. Called each time through a control's user-input  *
      //* loop so we can manually modify the controls when necessary.     *
      dp->EstablishCallback ( &T08_ControlUpdate ) ;

      //* Allow user to play with the controls.      *
      //* (Please see the notes in NcDialog.hpp on   *
      //* values returned (from the editing routines)*
      uiInfo Info ;                 // user interface data returned here
      short  icIndex = ZERO ;       // index of control with input focus
      bool   done = false ;         // loop control
      while ( ! done )
      {
         //*******************************************
         //* If focus is currently on a Pushbutton   *
         //*******************************************
         if ( ic[icIndex].type == dctPUSHBUTTON )
         {
            //* Get user input for this control                            *
            //* Returns when the Pushbutton is pressed OR when the user    *
            //* requests that the input focus moves to another control.    *
            if ( Info.viaHotkey == false )
            {
               icIndex = dp->EditPushbutton ( Info ) ;
            }
            //* If user selected the current control object via hotkey, we *
            //* don't need to call the input routine; but simply process   *
            //* the existing data.                                         *
            else
            {
               //* Information arrived in the Info.h_xxx fields.*
               //* Transfer it to the primary positions.        *
               //* Previous data in these fields has already    *
               //* been handled in prior loop iteration.        *
               Info.HotData2Primary () ;
            }

            //* If a Pushbutton was pressed *
            if ( Info.dataMod != false )
            {
               if ( Info.ctrlIndex == T8donePush ) // request to exit the dialog
                  done = true ;
               else
               { /* no other Pushbuttons defined for this dialog */ }
            }
            else
            {
               // No button press, so nothing to do
            }
         }

         //*******************************************
         //* If focus is currently on a Text Box     *
         //*******************************************
         else if ( ic[icIndex].type == dctTEXTBOX )
         {
            //* It doesn't matter how this control gained the input    *
            //* focus, by Tab/ShiftTab or via hotkey. If by hotkey, no *
            //* processing was done anyway, so discard any hotkey data *
            Info.viaHotkey = false ;

            //* Allow user to modify the text data in the text box.    *
            //* Returns when edit is complete.                         *
            icIndex = dp->EditTextbox ( Info ) ;

            //* If the data in the text box being edited has changed *
            if ( Info.dataMod != false )
            {
               //* Do any processing related to a change in the text box data *
               if ( Info.ctrlIndex == T8tbText )
               {
                  // DO STUFF
               }
               else
                  { /* no other text boxes defined in this dialog */ }
            }
         }

         //*******************************************
         //* If focus is currently on a Radio Button *
         //*******************************************
         else if ( ic[icIndex].type == dctRADIOBUTTON )
         {
            //* Get user input for this control                            *
            //* Returns when radio-button control or radio-button group is *
            //* about to lose the input focus.                             *
            //* A selection may or may not have been made.                 *
            if ( Info.viaHotkey == false )
            {
               icIndex = dp->EditRadiobutton ( Info ) ;
            }
            //* If user selected the current control object via hotkey, we *
            //* don't need to call the input routine; but simply process   *
            //* the existing data.                                         *
            //* For radio buttons, display update will already have been   *
            //* done, so we can just move the focus to the next control.   *
            else
            {
               //* Information arrived in the Info.h_xxx fields.*
               //* Transfer it to the primary positions.        *
               //* Previous data in these fields has already    *
               //* been handled in prior loop iteration.        *
               Info.HotData2Primary () ;
            }

            //* If 'selected' button is a member of a button group *
            if ( Info.selMember < MAX_DIALOG_CONTROLS )
            {
               //* If a new 'selected' member of the group *
               //* OR previous selection was re-selected   *
               if ( Info.dataMod != false || Info.keyIn == nckENTER )
               {
                  //* Process the user's selection.             *
                  //* There are two Radio Button groups defined *
                  switch ( Info.selMember )
                  {
                     case T8groupA_1:
                     case T8groupA_2:
                     case T8groupA_3:
                     case T8groupA_4:
                        // DO STUFF
                        break ;
                     case T8groupB_1:
                     case T8groupB_2:
                     case T8groupB_3:
                        // DO STUFF
                        break ;
                     default:
                        // no other radio buttons associated with a 
                        // button group have been defined for this dialog
                        break ;
                  }
               }
            }
            //* Else, button is an independent button *
            //* (Info.selMember==MAX_DIALOG_CONTROLS) *
            else
            {
               //* If the selected/deselected state of an * 
               //* independent radio button has changed,  *
               if ( Info.dataMod != false )
               {
                  // NOTE: all the interesting action for this test dialog 
                  //       is happening in the callback method.
                  switch ( Info.ctrlIndex )
                  {
                     case T8standAlone8:
                        // DO STUFF
                        break ;
                     case T8standAlone9:
                        // DO STUFF
                        break ;
                     default:
                        // no other independent radio buttons defined for this dialog
                        break ;
                  }
               }
            }
         }

         //* If user exited the control edit method via a hotkey,*
         //* then the new control already has focus. Otherwise,  *
         //* move input focus to next/previous control.          *
         if ( done == false && Info.viaHotkey == false )
         {
            if ( Info.keyIn == nckSTAB )
               icIndex = dp->PrevControl () ; 
            else
               icIndex = dp->NextControl () ;
         }
      }  // while()
   }        // if((dp->OpenWindow())==OK)
   else
   {  //* Most likely cause of dialog not opening is that the terminal window  *
      //* is too small. Give user a clue...                                    *
      short neededLines = (dialogROWS + minTestULY) >= minTERMROWS ?
                          (dialogROWS + minTestULY) : minTERMROWS, 
            neededCols  = dialogCOLS >= minTERMCOLS ? dialogCOLS : minTERMCOLS ;
      gString gs ;
      gs.compose ( L" Sorry, size of terminal window must be at least %hd x %hd for this test. ", 
                   &neededCols, &neededLines ) ;
      StatWin ( gs.gstr(), nc.cyG ) ;
      sleep ( 5 ) ;
   }

   if ( dp != NULL )                         // close the window
      delete ( dp ) ;

}  //* End Test08() *

//*************************
//*  T08_ControlUpdate    *
//*************************
//******************************************************************************
//* This is a callback method for manually updating the controls in the        *
//* Test08() dialog. This method is called at the beginning of the user-input  *
//* loop for each of the NcDialog control-editing methods.                     *
//* Thus, if the user changes any data in a control, the application can       *
//* immediately respond to that change.                                        *
//* This method handles situations that the dialog code cannot conveniently    *
//* handle in a timely manner.                                                 *
//*                                                                            *
//*  For this test, the following are updated by the callback method:          *
//*    1. indicate the 'selected' member of radio button group A               *
//*    2. indicate the 'selected' member of radio button group B               *
//*    3. indicate the state of radio buttons 8 and 9                          *
//*                                                                            *
//* Input  : currIndex: index of control that currently has focus              *
//*          wkey     : user's key input data                                  *
//*          firstTime: the EstablishCallback() method calls this method once  *
//*                     with firstTime==true, to perform any required          *
//*                     initialization. Subsequently, the NcDialog class       *
//*                     always calls with firstTime==false.                    *
//* Returns: OK                                                                *
//******************************************************************************
//* Important Note: This method makes some intimate assumptions about what     *
//* is happening in the method that established the callback. If changes are   *
//* made in the establishing method that affect the callback functionality,    *
//* be sure to update this method to address those changes.                    *
//******************************************************************************

short T08_ControlUpdate ( const short currIndex, const wkeyCode wkey, bool firstTime )
{
const char tbText[][16] = { { " 归零 Reset " }, { " 设定 Set   " } } ;
const char tbNum[][12] = 
{
   { " 零 (00) " },
   { " 一 (01) " },
   { " 二 (02) " },
   { " 三 (03) " },
   { " 四 (04) " },
   { " 五 (05) " },
   { " 六 (06) " },
   { " 七 (07) " },
} ;
               // initial values will force a display update on first call
static short   oldRBG_A = -1, oldRBG_B = -1 ;
static bool    oldRB8, oldRB9 ;
short          gIndex ;
bool           bState ;

   //*************************
   //* First-time processing *
   //*************************
   if ( firstTime != false )
   {
      //** Initialize the tracking data **
      //* Get state of Radio Button 8 and indicate the state in Text Box 3
      Test08Dialog->GetRadiobuttonState ( T8standAlone8, oldRB8 ) ;
      Test08Dialog->SetTextboxText ( T8utBOX3, (oldRB8 ? (char*)tbText[1] : (char*)tbText[0]) ) ;

      //* Get state of Radio Button 9 and indicate the state in Text Box 4
      Test08Dialog->GetRadiobuttonState ( T8standAlone9, oldRB9 ) ;
      Test08Dialog->SetTextboxText ( T8utBOX4, (oldRB9 ? (char*)tbText[1] : (char*)tbText[0]) ) ;
   }

   //***********************
   //* Standard processing *
   //***********************
   //* Get number (index) of 'selected' Radio Button in Group A and indicate it in Text Box 1 *
   gIndex = Test08Dialog->GetRbGroupSelection ( T8groupA_1 ) ;
   if ( gIndex != oldRBG_A )
   {
      oldRBG_A = gIndex ;
      Test08Dialog->SetTextboxText ( T8utBOX1, tbNum[gIndex] ) ;
   }

   //* Get number (index) of 'selected' Radio Button in Group B and indicate it in Text Box 2 *
   gIndex = Test08Dialog->GetRbGroupSelection ( T8groupB_1 ) ;
   if ( gIndex != oldRBG_B )
   {
      oldRBG_B = gIndex ;
      Test08Dialog->SetTextboxText ( T8utBOX2, tbNum[gIndex] ) ;
   }

   //* Get state of Radio Button 8 and indicate the state in Text Box 3
   Test08Dialog->GetRadiobuttonState ( T8standAlone8, bState ) ;
   if ( bState != oldRB8 )
   {
      Test08Dialog->SetTextboxText ( T8utBOX3, (bState ? (char*)tbText[1] : (char*)tbText[0]) ) ;
      oldRB8 = bState ;
   }

   //* Get state of Radio Button 9 and indicate the state in Text Box 4
   Test08Dialog->GetRadiobuttonState ( T8standAlone9, bState ) ;
   if ( bState != oldRB9 )
   {
      Test08Dialog->SetTextboxText ( T8utBOX4, (bState ? (char*)tbText[1] : (char*)tbText[0]) ) ;
      oldRB9 = bState ;
   }
   return OK ;

}  //* End T08_ControlUpdate() *

//*************************
//*       Test09          *
//*************************
//******************************************************************************
//* Test the NcDialog methods related to capture and restore of dialog         *
//* window display data.                                                       *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************
//* Notes on Menuwin Controls:                                                 *
//* - Internal calculations are performed by the control's constructor for     *
//*   size and position of both the 'collapsed' and 'expanded' control. The    *
//*   'collapsed' control is positioned at the specified ulY/ulX position.     *
//*   The expanded control is positioned one display line below the specified  *
//*   ulY/ulX position.                                                        *
//* - The label text specified is written in the 'collapsed' control. The      *
//*   length (display columns) of the label string (not including ^ (caret)    *
//*   if any) determines the width of the 'collapsed' control. Note that it is *
//*   generally prudent to have one space at each end of the label string for  *
//*   visual effect.  The 'cols' field of the initialization structure         *
//*   determines the width of the 'expanded' control i.e.the menu data display.*
//*   This includes the left and right borders.                                *
//*   If no label is specified (label field == NULL pointer or pointer to      *
//*   empty string), then there is no visible 'collapsed' control (i.e. it is  *
//*   a context menu), and the position of the 'expanded' control is specified *
//*   by the ulY/ulX fields.                                                   *
//*                                                                            *
//* Note on exercising the CaptureDialog() method:                             *
//*   This test includes a pair of special keycode definitions that will be    *
//*   interpreted by the callback method for capture of the dialog window's    *
//*   display data, either as plain text or as HTML. This dialog includes      *
//*   examples of most control types, so is a good candidate for this test.    *
//*   See also Dialog4 test application, Test #5 for capture of a dialog       *
//*   containing dctBILLBOARD controls.                                        *
//******************************************************************************

void Test09 ( void )
{
   StatWin ( "Test #9 : Move Dialog Window (可移动的 窗口) "
             "- testing screen data capture" ) ; // keep user up-to-date

static const short dialogROWS = 28 ;      // display lines
static const short dialogCOLS = 60 ;      // display columns
static const short sDATA_ITEMS = 5 ;      // elements in SB string and color arrays
static const short sCOL_WIDTH = 21 ;      // width of control in display columns
static attr_t monoColor[2] = { attrDFLT, nc.gr } ; // scroll box data colors
attr_t dColor = nc.blR ;                  // text color for writing to dialog
short ctrY    = termRows/2,               // dialog center in Y
      ctrX    = termCols/2,               // dialog center in X
      //* Upper left corner in Y (cannot obscure status window) *
      ulY     = (ctrY - dialogROWS/2) >= minTestULY ? 
                (ctrY - dialogROWS/2) : minTestULY,
      ulX     = ctrX - dialogCOLS / 2 ;   // upper left corner in X

//* Data for the dctSCROLLBOX control.                                         *
//* Note that some characters in this array are multi-byte characters.         *
#if 1    // CORRECTLY FORMATTED
static const char sboxData[sDATA_ITEMS][sCOL_WIDTH+6] =   
{
   "Upper Left 上左侧   ",   // "上 左侧"   // upper part - left-hand side
   "LowerLeft 下层的左侧",     // "下层的 左侧" // bottom - left-hand side
   "Centered 居中图像   ",    // "居中图像"   //centered image
   "Upper Right 上右边的",    // "上 右边的"  // upper part - right-hand side
   "LowRight下层的右边的",      // "下层的 右边的"// bottom - right-hand side 
} ;
#elif 0  // TEMP - SINGLE-STRING (STRESS-TESTING)
static const char sboxData[sDATA_ITEMS * (sCOL_WIDTH+6)] =   
{
   "Upper Left 上左侧   "    // "上 左侧"   // upper part - left-hand side
   "LowerLeft 下层的左侧"      // "下层的 左侧" // bottom - left-hand side
   "Centered 居中图像   "     // "居中图像"   //centered image
   "Upper Right 上右边的"     // "上 右边的"  // upper part - right-hand side
   "LowRight下层的右边的"       // "下层的 右边的"// bottom - right-hand side 
} ;
#elif 0  // TEMP - STRINGS TOO LONG (STRESS-TESTING)
static const char sboxData[sDATA_ITEMS][sCOL_WIDTH+13] =   
{
   "Upper Left abcdefghijk",
   "Lower Left abcdefg",
   "Centered 居中图像abcdefghij",
   "Upper Rightxxxxxxxxx",
   "Lower Rightxxxxx", 
} ;
#elif 0  // TEMP - STRINGS TOO SHORT (STRESS-TESTING)
static const char sboxData[sDATA_ITEMS][sCOL_WIDTH+4] =   
{
   "Upper Left 上左侧",
   "Lower Left",
   "Centered 居中图像x",
   "Upper Right",
   "Lower Right", 
} ;
#elif 0  // TEMP - MIXED-LENGTH DATA ITEMS (STRESS-TESTING)
static const char sboxData[sDATA_ITEMS][sCOL_WIDTH+13] =   
{
   "Upper Left abcdefghijk",
   "Lower Left",
   "Centered 居中图像",
   "Upper Right abcdefgh",
   "Lower Right qrstuvwxyz", 
} ;
#endif   // TEMP -  (STRESS-TESTING)

//* Specifies a list of (y,x) positions for *
//* the dialog within the terminal window   *
winPos winPosList[sDATA_ITEMS] 
{
   winPos( 2,  2 ),                          // Upper Left
   winPos( termRows-dialogROWS-1,  2 ),      // Lower Left
   winPos( termRows/2-dialogROWS/2, termCols/2-dialogCOLS/2 ), // Centered
   winPos( 2, termCols-dialogCOLS-2 ),       // Upper Right
   winPos( termRows-dialogROWS-1, termCols-dialogCOLS-2 ), // Lower Right
} ;
           
const short scDATA_ITEMS = 32, scITEM_LEN = 15 ; // elements in Dropdown and Scrollext
//* Data for the dctDROPDOWN control.                                          *
//* Note that some characters in this array are multi-byte characters.         *
#if 1    // CORRECTLY FORMATTED UTF8 DATA
static const char ddboxData[scDATA_ITEMS][scITEM_LEN+5] =   
{  // "Why are you buying so many vitamins?"
   "00你干吗买这00",  "01么多维他命01",  "02啊？      02",
   // "For my health. I don't think I eat enough fruits and vegetables."
   "03为了身体嘛03",  "04。我觉得我04",  "05吃的蔬菜水05",  "06果不够。  06", 
   // "Do you think they are really beneficial?"
   "07你觉得真的07",  "08有好处吗？08",  
   // "Actually, I'm not quite sure, but I guess they are, or there wouldn't be 
   //  so many people buying vitamins."
   "09说不好。但09",  "10应该有好处10",   "11吧，否则不11", "12会有那么多12",  "13人买维他命13",  
   "14。        14", 
   // "I read in an article that vitamins are beneficial but the human body can 
   //  only absorb so much. The rest just pass through."
   "15我从一个文15",  "16章看到，维16",  "17他命确实对17",  "18人体有益，18",   "19但人体只能19",
   "20吸收一些，20",  "21其余部分都21",  "22流失了。  22", 
   // "Are you saying they are wasted?"
   "23你是说维他23", "24命都被浪费24",  "25了？      25",
   // "I'm saying I think you'd better eat more fruits and vegetables instead 
   //  of taking vitamin pills."
   "26我觉得你还26", "27是多吃蔬菜27", "28水果，别吃28", "29维他命了。29", 
   // "I will give it some thought. Thanks."
   "30我会考虑下30", "31的，谢谢。31", 
} ;
#elif 0  // TEMP - CORRECTLY FORMATTED ASCII DATA
static const char ddboxData[scDATA_ITEMS][scITEM_LEN] =   
{
   "00AAAAAAAAAA00",  "01BBBBBBBBBB01",  "02CCCCCCCCCC02",  "03DDDDDDDDDD03",
   "04EEEEEEEEEE04",  "05FFFFFFFFFF05",  "06GGGGGGGGGG06",  "07HHHHHHHHHH07",
   "08IIIIIIIIII08",  "09JJJJJJJJJJ09",  "10KKKKKKKKKK10",  "11LLLLLLLLLL11",
   "12MMMMMMMMMM12",  "13NNNNNNNNNN13",  "14OOOOOOOOOO14",  "15PPPPPPPPPP15",
   "16QQQQQQQQQQ16",  "17RRRRRRRRRR17",  "18SSSSSSSSSS18",  "19TTTTTTTTTT19",
   "20UUUUUUUUUU20",  "21VVVVVVVVVV21",  "22WWWWWWWWWW22",  "23XXXXXXXXXX23",
   "24YYYYYYYYYY24",  "25ZZZZZZZZZZ25",  "26   THIS   26",  "27    IS    27", 
   "28   THE    28",  "29UPPER-CASE29",  "30  ASCII   30",  "31 ALPHABET 31", 
} ;
#elif 0  // TEMP - SINGLE-STRING (STRESS-TESTING)
#elif 0  // TEMP - MIXED-LENGTH DATA ITEMS (STRESS-TESTING)
#endif   // TEMP -  (STRESS-TESTING)

//* Data for the dctSCROLLEXT control.                                         *
//* Note that some characters in this array are multi-byte characters.         *
// "第一确定真相"       "First, establish the truth."
// "未逢龙虎会，一任马牛呼"  "not every dragon will come when you whistle"
//                (not meet-by-chance dragon vigorous meet allow horse cow whistle)
const short seDATA_ITEMS = 25 ;
#if 1    // CORRECTLY FORMATTED
static const char* seboxData[seDATA_ITEMS] =   
{
   "00道德是什么00",   // "what is the moral"
   "01英文翻译谢01",   // "thanks for the English translation"
   "02春季与秋季02",   // "spring and autumn"
   "03战争总动员03",   // "war story"
   "04寒暄乏味的04",   // "small talk is boring"
   "05你在干什么05",   // "what are you doing?"
   "06西方是白虎06",   // "West is the white tiger"
   "07发风骂坐  07",  // "to sit cursing the wind"
   "08间接民主  08",  // "representative democracy"
   "09不奉承我 !09",  // "Don't flatter me!"
   "10不废话我 !10",  // "Don't bullshit me!"
   "11各花入各眼11",   // "Beauty is in the eye of the beholder."
   "12言多必失。12",   // "Say too much and you have made an error"
   "13不作虧心事13",   // "A clear conscience is a good pillow."
   "14天高皇帝遠14",   // "The mountains are high, the emperor is far away."
   "15操你妈    15", // "fuck your mother"
   "16一见钟情  16",  // "fall in love at first sight"
   "17你真有眼光17",   // "you have good taste"
   "18走开！    18", // "Get lost!"
   "19坐便器座圈19",   // "toilet seat"
   "20试穿一下吗20",   // "Would you like to try it on?"
   "21马是强健的21",   // "the horse is strong"
   "22恶毒的女人22",   // "she is such a cat!" (bitch)
   "23他是粗俗人23",   // "he is a pig" (vulgar - unkind remark to a person)
   "24兔子是柔软24",   // "the rabbit is soft"
} ;
#elif 0  // TEMP - MIXED-LENGTH DATA ITEMS (STRESS-TESTING)
//* Note that because these data are used as-is, and are not tested and        *
//* massaged as they are with the dctSCROLLBOX control's data; therefore,      *
//* an ugly array == an ugly display. Display items which are too long, are    *
//* truncated, and display items which are too short are not padded to fill    *
//* the display space.                                                         *
static const char* seboxData[seDATA_ITEMS] =   
{
   "00道德德是什么00",
   "01英文翻译谢德德德01",
   "02春德德德季与秋季02",
   "03战争总动03",
   "04寒暄的04",
   "05什么05",
   "06虎06",
   "07发风骂坐07",
   "08间接民主08",
   "09不奉承我!09",
   "10不废话我!10",
   "11各花入各眼11",
   "12言多必失。12",
   "13不作虧心事13",
   "14天高皇帝遠14",
   "15操你妈15",
   "16一见钟情一见钟情16",
   "17你真有眼光17",
   "18走开！abcd18",
   "19坐便器座圈abc19",
   "20试穿一下吗def20",
   "21马是强健的ghij21",
   "22恶毒的女人klmno22",
   "23他是粗俗人23",
   "2424",
} ;
#endif   // TEMP -  (STRESS-TESTING)

static const attr_t seboxColors[seDATA_ITEMS+1] = 
{
   nc.gr, nc.gr, nc.gr, nc.gr, nc.gr, nc.gr, nc.gr, nc.gr, nc.gr, nc.gr, 
   nc.gr, nc.gr, nc.gr, nc.gr, nc.gr, nc.gr, nc.gr, nc.gr, nc.gr, nc.gr, 
   nc.gr, nc.gr, nc.gr, nc.gr, nc.gr
} ;
static const char dumpitStrings[][19] = { " Enable UI D^ump  ", " Disable UI D^ump " } ;
const dspinData dsData( -999, 999, 0, dspinINTEGER, nc.brR ) ;
const short mw01DATA_ITEMS = 8,  // menu items in array
            mw01ITEM_LEN = 17 ;  // display columns required for menu item
            // note compensation for multi-byte characters and hotkey indicator
#if 1    // CORRECTLY FORMATTED
static const char mw01Data[mw01DATA_ITEMS][mw01ITEM_LEN+6] =   
{  // data with hotkeys
   "菜单命令 ^First   ",
   "菜单命令 ^Second >",
   "菜单命令 ^Third   ",
   "菜单命令 ^Fourth  ",
   "菜单命令 F^ifth   ",
   "菜单命令 Si^xth   ",
   "菜单命令 Se^venth ",
   "菜单命令 ^Eighth  ",
} ;
#elif 0  // TEMP - STRINGS TOO LONG (STRESS-TESTING)
static const char mw01Data[mw01DATA_ITEMS][mw01ITEM_LEN+8] =   
{  // data with hotkeys
   "菜单命令 ^First   xx",
   "菜单命令 ^Second >xx",
   "菜单命令 ^Third   xx",
   "菜单命令 ^Fourth  xx",
   "菜单命令 F^ifth   xx",
   "菜单命令 Si^xth   xx",
   "菜单命令 Se^venth xx",
   "菜单命令 ^Eighth  xx",
} ;
#elif 0  // TEMP - STRINGS TOO SHORT (STRESS-TESTING)
static const char mw01Data[mw01DATA_ITEMS][mw01ITEM_LEN+5] =   
{  // data with hotkeys
   "菜单命令 ^First",
   "菜单命令 ^Second",
   "菜单命令 ^Third",
   "菜单命令 ^Fourth",
   "菜单命令 F^ifth",
   "菜单命令 Si^xth",
   "菜单命令 Se^venth",
   "菜单命令 ^Eighth",
} ;
#elif 0  // TEMP - MIXED-LENGTH DATA ITEMS (STRESS-TESTING)
static const char mw01Data[mw01DATA_ITEMS][mw01ITEM_LEN+18] =   
{  // data with hotkeys
   "菜单命令 ^First",
   "菜单命令 ^Second gggggggggg",
   "菜单命令 ^Third",
   "菜单命令 ^Fourth",
   "菜单命令 F^ifth",
   "菜单命令 Si^xth",
   "菜单命令 Se^venthrrrrrrrr",
   "菜单命令 ^Eighth nnnnnnnnnnnnn",
} ;
#elif 0  // TEMP - SINGLE STRING, SINGLE COL TEXT (STRESS-TESTING)
static const char* mw01Data =   
{  // data with hotkeys
   "^First aaaaaaaaaaa"
   "^Second bbbbbbbb>"
   "^Third cccccccccc"
   "^Fourth ddddddddd"
   "F^ifth eeeeeeeeee"
   "Si^xth ffffffffff"
   "Se^venth gggggggg"
   "^Eighth hhhhhhhhh"
} ;
#else    // TEMP - SINGLE STRING, MULTI-COL TEXT (STRESS-TESTING)
static const char* mw01Data =   
{  // data with hotkeys
   "菜单命令 ^First   "
   "菜单命令 ^Second >"
   "菜单命令 ^Third   "
   "菜单命令 ^Fourth  "
   "菜单命令 F^ifth   "
   "菜单命令 Si^xth   "
   "菜单命令 Se^venth "
   "菜单命令 ^Eighth  "
} ;
#endif   // TEMP -  (STRESS-TESTING)

const short mw02DATA_ITEMS = 8,  // menu items in array
            mw02ITEM_LEN = 17 ;  // display columns required for menu item
            // note compensation for multi-byte characters (no hotkey indicator)
static const char mw02Data[mw02DATA_ITEMS][mw02ITEM_LEN+5] =   
{  // data without hotkeys
   "菜单命令 9th     ",
   "菜单命令 10th    ",
   "菜单命令 11th    ",
   "菜单命令 12th   >",
   "菜单命令 13th    ",
   "菜单命令 14th    ",
   "菜单命令 15th    ",
   "菜单命令 16th    ",
} ;

const short mw03DATA_ITEMS = 4,  // menu items in array
            mw03ITEM_LEN = 12 ;  // display columns required for menu item
            // note compensation for multi-byte characters and hotkey indicator
static const char mw03Data[mw03DATA_ITEMS][mw03ITEM_LEN+2] =   
{  // data with hotkeys
   " ^ILMT       ",
   " I^LHE       ",
   " IL^HM       ",
   " IHS^WMMOD   ",
} ;

const short mw04DATA_ITEMS = 4,  // menu items in array
            mw04ITEM_LEN = 12 ;  // display columns required for menu item
            // note compensation for multi-byte characters (no hotkey indicator)
static const char mw04Data[mw04DATA_ITEMS][mw04ITEM_LEN+1] =   
{  // data with hotkeys
   " ILMT       ",
   " ILHE       ",
   " ILHM       ",
   " IHSWMMOD   ",
} ;


InitCtrl ic[T9controlsDEFINED] =    // array of dialog control info
{
   { //* 'DONE' pushbutton - - - - - - - - - - - - - - - - - - - - T09donePush *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(dialogROWS - 2),        // ulY:       upper left corner in Y
      short(dialogCOLS / 2 - 16),   // ulX:       upper left corner in X
      1,                            // lines:     control lines
      15,                           // cols:      control columns
      "  ^DONE (完蛋)  ",             // dispText:  
      nc.gyR,                       // nColor:    non-focus color
      nc.gyre | ncbATTR,            // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      NULL,                         // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[T9trackrBox],             // nextCtrl:  link in next structure
   },
   { //* Non-active text box (report dialog position)  - - - - - - T9trackrBox *
      dctTEXTBOX,                   // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(ic[T9donePush].ulY - 2),// ulY:       upper left corner in Y
      short(ic[T9donePush].ulX),    // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      15,                           // cols:      control columns
      NULL,                         // dispText:  (initially blank)
      nc.brR,                       // nColor:    non-focus color
      nc.grR,                       // fColor:    focus color
      tbPrint,                      // filter:    any printing character
      "Current Dialog Position",    // label:     
      1,                            // labY:      
      -4,                           // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      false,                        // active:    cannot gain focus
      &ic[T9positionControl],       // nextCtrl:  link in next structure
   },
   { //* Scrollbox (set dialog position)   - - - - - - - - - T9positionControl *
      dctSCROLLBOX,                 // type:      define a scrolling-data control
      rbtTYPES,                     // rbSubtype: (na)
      false,                        // rbSelect:  (n/a)
      2,                            // ulY:       upper left corner in Y
      2,                            // ulX:       upper left corner in X
      short(sDATA_ITEMS + 2),       // lines:     control lines
      short(sCOL_WIDTH + 1),        // cols:      control columns
      (char*)&sboxData,             // dispText:  n/a
      nc.gyR,                       // nColor:    non-focus border color
      nc.gr,                        // fColor:    focus border color
      tbPrint,                      // filter:    (n/a)
      "^Where To?",                 // label:     
      -1,                           // labY:      offset from control's ulY
      4,                            // labX       offset from control's ulX
      ddBoxTYPES,                   // exType:    (n/a)
      sDATA_ITEMS,                  // scrItems:  number of elements in text/color arrays
      2,                            // scrSel:    index of initial highlighted element
      monoColor,                    // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[T9planetBox],             // nextCtrl:  link in next structure
   },
   { //* 'Planet' textbox  - - - - - - - - - - - - - - - - - - - - T9planetBox *
      dctTEXTBOX,                   // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      2,                            // ulY:       upper left corner in Y
      41,                           // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      18,                           // cols:      control columns
      NULL,                         // dispText:  (initially blank)
      nc.bw,                        // nColor:    non-focus color
      nc.cyG,                       // fColor:    focus color
      tbPrint,                      // filter:    any printing character
      "Favorite ^Planet",           // label:     label text
      ZERO,                         // labY:      label offset
      -16,                          // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[T9icecreamBox],           // nextCtrl:  link in next structure
   },
   { //* 'Ice Cream' textbox - - - - - - - - - - - - - - - - - - T9icecreamBox *
      dctTEXTBOX,                   // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      4,                            // ulY:       upper left corner in Y
      41,                           // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      18,                           // cols:      control columns
      NULL,                         // dispText:  (initially blank)
      nc.bw,                        // nColor:    non-focus color
      nc.cyG,                       // fColor:    focus color
      tbPrint,                      // filter:    any printing character
      "^Fave Ice Cream",            // label:     label text
      ZERO,                         // labY:      label offset
      -16,                          // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[T9tuscaloosaButton],      // nextCtrl:  link in next structure
   },
   { //* Radio Button Group  - - - - - - - - - - - - - - -  T9tuscaloosaButton *
      dctRADIOBUTTON,               // type:      
      rbtS3s,                       // rbSubtype: standard, 3 chars wide
      true,                         // rbSelect:  default selection
      6,                            // ulY:       upper left corner in Y
      41,                           // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a)
      nc.bw,                        // nColor:    non-focus color
      nc.cyG,                       // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "Tuscal^oosa",                // label:
      ZERO,                         // labY:      
      5,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[T9sandiegoButton],        // nextCtrl:  link in next structure
   },
   { //* Radio Button Group  - - - - - - - - - - - - - - - -  T9sandiegoButton *
      dctRADIOBUTTON,               // type:      
      rbtS3s,                       // rbSubtype: standard, 3 chars wide
      false,                        // rbSelect:  default selection
      7,                            // ulY:       upper left corner in Y
      41,                           // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a)
      nc.bw,                        // nColor:    non-focus color
      nc.cyG,                       // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "San Die^go",                 // label:
      ZERO,                         // labY:      
      5,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[T9cucamongaButton],       // nextCtrl:  link in next structure
   },
   { //* Radio Button Group  - - - - - - - - - - - - - - - - T9cucamongaButton *
      dctRADIOBUTTON,               // type:      
      rbtS3s,                       // rbSubtype: standard, 3 chars wide
      false,                        // rbSelect:  default selection
      8,                            // ulY:       upper left corner in Y
      41,                           // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a)
      nc.bw,                        // nColor:    non-focus color
      nc.cyG,                       // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "Cu^camonga",                 // label:
      ZERO,                         // labY:      
      5,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[T9valparisoButton],       // nextCtrl:  link in next structure
   },
   { //* Radio Button Group  - - - - - - - - - - - - - - - - T9valparisoButton *
      dctRADIOBUTTON,               // type:      
      rbtS3s,                       // rbSubtype: standard, 3 chars wide
      false,                        // rbSelect:  default selection
      9,                            // ulY:       upper left corner in Y
      41,                           // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a)
      nc.bw,                        // nColor:    non-focus color
      nc.cyG,                       // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "Valpa^riso",                 // label:
      ZERO,                         // labY:      
      5,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[T9dumpPush],              // nextCtrl:  link in next structure
   },
   { //* 'DUMP UI' pushbutton  - - - - - - - - - - - - - - - - - -  T9dumpPush *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(ic[T9valparisoButton].ulY + 2),// ulY:       upper left corner in Y
      short(ic[T9valparisoButton].ulX + 1),// ulX:       upper left corner in X
      1,                            // lines:     control lines
      17,                           // cols:      control columns
      dumpitStrings[0],             // dispText:  
      nc.cyR,                       // nColor:    non-focus color
      nc.grG,                       // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "",                           // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[T9scottyButton],          // nextCtrl:  link in next structure
   },
   { //* Stand-alone Radio Button -- - - - - - - - - - - - - -  T9scottyButton *
      dctRADIOBUTTON,               // type:      
      rbtS3s,                       // rbSubtype: standard, 3 chars wide
      false,                        // rbSelect:  default selection
      12,                           // ulY:       upper left corner in Y
      26,                           // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a)
      nc.bw,                        // nColor:    non-focus color
      nc.cyG,                       // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "\"Be^am me up, Scotty!\"",   // label:
      ZERO,                         // labY:      
      5,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[T9spinnerControl],        // nextCtrl:  link in next structure
   },
   { //* Spinner Control   - - - - - - - - - - - - - - - - -  T9spinnerControl *
      dctSPINNER,                   // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      10,                           // ulY:       upper left corner in Y
      2,                            // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      5,                            // cols:      control columns
      NULL,                         // dispText:  (n/a)
      nc.bw,                        // nColor:    non-focus color
      nc.grbk,                      // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "^Spin Control",              // label:     
      ZERO,                         // labY:      
      7,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      &dsData,                      // spinData:  spinner init
      true,                         // active:    allow control to gain focus
      &ic[T9dropdownBox],           // nextCtrl:  link in next structure
   },
   { //* DropDown Box      - - - - - - - - - - - - - - - - - - - T9dropdownBox *
      dctDROPDOWN,                  // type:      define a scrolling-data control
      rbtTYPES,                     // rbSubtype: (na)
      false,                        // rbSelect:  (n/a)
      13,                           // ulY:       upper left corner in Y
      2,                            // ulX:       upper left corner in X
      scDATA_ITEMS / 2,             // lines:     control lines
      scITEM_LEN + 1,               // cols:      control columns
      (char*)&ddboxData,            // dispText:  n/a
      nc.gyR,                       // nColor:    non-focus border color
      nc.gr,                        // fColor:    focus border color
      tbPrint,                      // filter:    (n/a)
      "DropDown a We^ll!",          // label:     
      -1,                           // labY:      offset from control's ulY
      ZERO,                         // labX       offset from control's ulX
      ddBoxDOWN,                    // exType:    (n/a)
      scDATA_ITEMS,                 // scrItems:  number of elements in text/color arrays
      ZERO,                         // scrSel:    index of initial highlighted element
      monoColor,                    // scrColor:  single-color data display
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[T9scrollextControl],      // nextCtrl:  link in next structure
   },
   { //* Scrollext Control - - - - - - - - - - - - - - - -  T9scrollextControl *
      dctSCROLLEXT,                 // type:      define a scrolling-data control
      rbtTYPES,                     // rbSubtype: (na)
      false,                        // rbSelect:  (n/a)
      17,                           // ulY:       upper left corner in Y
      2,                            // ulX:       upper left corner in X
      5,                            // lines:     control lines
      scITEM_LEN + 1,               // cols:      control columns
      NULL,                         // dispText:  n/a
      nc.gyR,                       // nColor:    non-focus border color
      nc.gr,                        // fColor:    focus border color
      tbPrint,                      // filter:    (n/a)
      "Scroll O^n Down!",           // label:     
      -1,                           // labY:      offset from control's ulY
      ZERO,                         // labX       offset from control's ulX
      ddBoxTYPES,                   // exType:    (n/a)
      seDATA_ITEMS,                 // scrItems:  number of elements in text/color arrays
      ZERO,                         // scrSel:    index of initial highlighted element
      monoColor,                    // scrColor:  single-color data display
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[T9menuwin01],             // nextCtrl:  link in next structure
   },
   { //* Menuwin Control #1  - - - - - - - - - - - - - - - - - -   T9menuwin01 *
      dctMENUWIN,                   // type:      define a menu-window control
      rbtTYPES,                     // rbSubtype: (na)
      false,                        // rbSelect:  (n/a)
      16,                           // ulY:       upper left corner in Y
      20,                           // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      mw01ITEM_LEN + 2,             // cols:      control columns (incl. borders)
      (const char*)mw01Data,        // dispText:  n/a
      nc.gr,                        // nColor:    non-focus border color
      nc.grR,                       // fColor:    focus border color
      tbPrint,                      // filter:    (n/a)
      " Menu^Y (菜单 第一) ",        // label:     'primary menu'
      -1,                           // labY:      offset from control's ulY
      ZERO,                         // labX       offset from control's ulX
      ddBoxTYPES,                   // exType:    (n/a)
      mw01DATA_ITEMS,               // scrItems:  number of elements in text/color arrays
      ZERO,                         // scrSel:    (n/a)
      monoColor,                    // scrColor:  single-color data display
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[T9menuwin02],             // nextCtrl:  link in next structure
   },
   { //* Menuwin Control #2  - - - - - - - - - - - - - - - - - -   T9menuwin02 *
      dctMENUWIN,                   // type:      define a menu-window control
      rbtTYPES,                     // rbSubtype: (na)
      false,                        // rbSelect:  (n/a)
      16,                           // ulY:       upper left corner in Y
      40,                           // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      mw02ITEM_LEN + 2,             // cols:      control columns (incl. borders)
      (const char*)mw02Data,        // dispText:  n/a
      nc.gr,                        // nColor:    non-focus border color
      nc.grR,                       // fColor:    focus border color
      tbPrint,                      // filter:    (n/a)
      " Menu^Z (菜单 第二) ",           // label:     'secondary menu'
      -1,                           // labY:      offset from control's ulY
      ZERO,                         // labX       offset from control's ulX
      ddBoxTYPES,                   // exType:    (n/a)
      mw01DATA_ITEMS,               // scrItems:  number of elements in text/color arrays
      ZERO,                         // scrSel:    (n/a)
      monoColor,                    // scrColor:  single-color data display
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[T9menuwin03],             // nextCtrl:  link in next structure
   },
   { //* Menuwin Sub-menu (for menu 01)  - - - - - - - - - - - -   T9menuwin03 *
      dctMENUWIN,                   // type:      define a menu-window control
      rbtTYPES,                     // rbSubtype: (na)
      false,                        // rbSelect:  (n/a)
      19,                           // ulY:       upper left corner in Y
      38,                           // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      mw03ITEM_LEN + 2,             // cols:      control columns (incl. borders)
      (const char*)mw03Data,        // dispText:  n/a
      nc.gr,                        // nColor:    non-focus border color
      nc.grR,                       // fColor:    focus border color
      tbPrint,                      // filter:    (n/a)
      NULL,                         // label:     'sub-menu'
      ZERO,                         // labY:      offset from control's ulY
      ZERO,                         // labX       offset from control's ulX
      ddBoxTYPES,                   // exType:    (n/a)
      mw03DATA_ITEMS,               // scrItems:  number of elements in text/color arrays
      ZERO,                         // scrSel:    (n/a)
      monoColor,                    // scrColor:  single-color data display
      NULL,                         // spinData:  (n/a)
      false,                        // active:    initially unavailable to user
      &ic[T9menuwin04],             // nextCtrl:  link in next structure
   },
   { //* Menuwin Sub-menu (for menu 02)  - - - - - - - - - - - -   T9menuwin04 *
      dctMENUWIN,                   // type:      define a menu-window control
      rbtTYPES,                     // rbSubtype: (na)
      false,                        // rbSelect:  (n/a)
      21,                           // ulY:       upper left corner in Y
      27,                           // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      mw04ITEM_LEN + 2,             // cols:      control columns (incl. borders)
      (const char*)mw04Data,        // dispText:  n/a
      nc.gr,                        // nColor:    non-focus border color
      nc.grR,                       // fColor:    focus border color
      tbPrint,                      // filter:    (n/a)
      NULL,                         // label:     'sub-menu'
      ZERO,                         // labY:      offset from control's ulY
      ZERO,                         // labX       offset from control's ulX
      ddBoxTYPES,                   // exType:    (n/a)
      mw04DATA_ITEMS,               // scrItems:  number of elements in text/color arrays
      ZERO,                         // scrSel:    (n/a)
      monoColor,                    // scrColor:  single-color data display
      NULL,                         // spinData:  (n/a)
      false,                        // active:    initially unavailable to user
      NULL,                         // nextCtrl:  link in next structure
   },
} ;

   //* Initial parameters for dialog window *
   InitNcDialog dInit( dialogROWS,     // number of display lines
                       dialogCOLS,     // number of display columns
                       ulY,            // Y offset from upper-left of terminal 
                       ulX,            // X offset from upper-left of terminal 
                       "  Moving the Dialog Window (可移动的 窗口)  ", // dialog title
                       ncltSINGLE,     // border line-style
                       dColor,         // border color attribute
                       dColor,         // interior color attribute
                       ic              // pointer to list of control definitions
                     ) ;

   //* Instantiate the dialog window *
   NcDialog* dp = new NcDialog ( dInit ) ;
   Test09Dialog = dp ;              // give callback method access to the dialog

   //* Open the dialog window *
   if ( (dp->OpenWindow()) == OK )
   {
      //* Button-group label *
      dp->WriteString ( ic[T9tuscaloosaButton].ulY, 
                        ic[T9tuscaloosaButton].ulX - 16, "4-syllable City", dColor ) ;

      //* Initialize display data for dctSCROLLEXT control *
      ssetData sData( seboxData, seboxColors, seDATA_ITEMS, 21, true ) ;
      dp->SetScrollextText ( T9scrollextControl, sData ) ;

      //* Use Right-justify format for editing text in icecreamBox *
      dp->SetTextboxCursor ( T9icecreamBox, tbcpRIGHTJUST ) ;

      //* Give user a clue *
      dp->WriteParagraph ( dialogROWS-6, 2, 
               "Use Tab or hotkeys to navigate among controls, and then\n"
               "select a new dialog position from the ScrollBox control.", dColor ) ;
      dp->WriteParagraph ( dialogROWS - 4, dialogCOLS - 24,
                           "Capture Dialog To File\n"
                           "  ALT+z == plain text\n"
                           "  ALT+SHIFT+z == html", nc.brbl | ncbATTR ) ;

      //* Some color text to test synchronization of character/color *
      //* on window capture-hide-restore.                            *
      winPos wp( 8, 25 ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, "=Magenta Stout=", nc.maG ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, "==Green Stout==", nc.grG ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, "===Bold Blue===", nc.blB ) ;
      wp.ypos += 3 ; wp.xpos -= 2 ;
      gString gsOut ( "Menu Bar (菜单栏) Initially Hidden\n"
                      "  Press CTRL+V To View/Hide Menu  " ) ;
      dp->WriteParagraph ( wp, gsOut, dColor | ncbATTR ) ;

      #if 0    // TESTING ONLY - Replace dialog text
      // Used only for stress-testing text alignment of the CaptureDialog() method.
      // Text-alignment data are written in the left margin. Color attribute 
      // shift data are positioned to be overwritten by Dropdown.control.
      // See T09_ControlUpdate().
      dp->WriteString ( wp, L"Menu Bar", nc.blre ) ;
      wchar_t someChars[52] = L"a\n菜\nb\n菜\nc\n菜\nd\n菜\ne\n菜\nf\n菜\n"
                              "g\n菜\nh\n菜\ni\n菜\nj\n菜\nk\n菜\nl\n菜\nm" ; 
      dp->WriteParagraph ( 2, 0, someChars, dColor ) ;
      winPos wps (22, 14 ) ;
      attr_t ats[12] = { nc.bw, nc.re, nc.gr, nc.br, nc.bl, nc.ma, nc.cy, nc.gy, 
                         nc.bw, nc.reR 
                       } ;
      dp->ClearLine ( wps.ypos ) ; dp->ClearLine ( wps.ypos + 1 ) ;
      for ( short i = ZERO, j = ZERO ; i < 10 ; ++i, j+=2 )
      {
         wps = dp->WriteChar ( wps, someChars[j], ats[i] ) ;
         ++wps.xpos ;
      }
      #endif   // TESTING ONLY - Replace dialog text

      //* Create a Menuwin group (MenuBar) and make it initially invisible     *
      short mwList[] = { T9menuwin01, T9menuwin02, -1 } ;
      if ( (dp->GroupMenuwinControls ( mwList, nckC_V )) == OK )
      {
         //* Attach sub-menus to main menu controls *
         //* One sub-menu is attached to T9menuwin01 at item 2 (index 1).         *
         //* One sub-menu is attached to T9menuwin02 at item 4 (index 3).         *
         short submenuList01[] = { MAX_DIALOG_CONTROLS, T9menuwin03, -1 } ;
         if ( (dp->AttachMenuwinSubmenus ( T9menuwin01, submenuList01 )) != OK )
            dp->DebugMsg ( "Attach Sub-menu Error!" ) ;
         short submenuList02[] = { MAX_DIALOG_CONTROLS, MAX_DIALOG_CONTROLS, 
                                   MAX_DIALOG_CONTROLS, T9menuwin04, -1 } ;
         if ( (dp->AttachMenuwinSubmenus ( T9menuwin02, submenuList02 )) != OK )
            dp->DebugMsg ( "Attach Sub-menu Error!" ) ;

         //* Make the Menu Bar invisible *
         if ( (dp->HideMenuBar ( T9menuwin01 )) != OK )
            dp->DebugMsg ( "Hide-Menu-Bar Error!" ) ;
      }
      else
         dp->DebugMsg ( "Group Menuwin Error!" ) ;

      //* Set radio buttons as an exclusive-OR group.              *
      //* (only one from each group can be selected at any moment).*
      short XorGroup1[5] = { 
                             T9tuscaloosaButton, T9sandiegoButton, 
                             T9cucamongaButton, T9valparisoButton, -1 
                           } ;
      dp->GroupRadiobuttons ( XorGroup1 ) ;

      //* Make our data visible *
      dp->RefreshWin () ;

      //* Establish a call-back method that can be called from within the *
      //* NcDialog code. Called each time through a control's user-input  *
      //* loop so we can manually modify the controls when necessary.     *
      dp->EstablishCallback ( &T09_ControlUpdate ) ;

      //* Allow user to play with the controls.      *
      //* (Please see the notes in NcDialog.hpp on   *
      //* values returned (from the editing routines)*
      uiInfo Info ;                 // user interface data returned here
      short  icIndex = ZERO ;       // index of control with input focus
      bool   uidumpEnabled = false, // enable user-interface info dump
             done = false ;         // loop control
      while ( ! done )
      {
         //*******************************************
         //* If focus is currently on a Pushbutton   *
         //*******************************************
         if ( ic[icIndex].type == dctPUSHBUTTON )
         {
            //* Get user input for this control                            *
            //* Returns when the Pushbutton is pressed OR when the user    *
            //* requests that the input focus moves to another control.    *
            if ( Info.viaHotkey == false )
            {
               icIndex = dp->EditPushbutton ( Info ) ;
               if ( uidumpEnabled )
               {  // To view contents of Info
                  //* Note that the coordinates for display are absolute 
                  //* screen coordinates, not dialog offsets
                  winPos wPos(2,1) ;
                  dp->Dump_uiInfo ( wPos, Info ) ;
               }
            }
            //* If user selected the current control object via hotkey, we *
            //* don't need to call the input routine; but simply process   *
            //* the existing data.                                         *
            else
            {
               if ( uidumpEnabled )
               {  // To view contents of Info
                  winPos wPos(2,1) ;
                  dp->Dump_uiInfo ( wPos, Info ) ;
               }
               //* Information arrived in the Info.h_xxx fields.*
               //* Transfer it to the primary positions.        *
               //* Previous data in these fields has already    *
               //* been handled in prior loop iteration.        *
               Info.HotData2Primary () ;
               if ( uidumpEnabled )
               {  // To view contents of Info
                  winPos wPos(2,1) ;
                  dp->Dump_uiInfo ( wPos, Info ) ;
               }
            }

            //* If a Pushbutton was pressed *
            if ( Info.dataMod != false )
            {
               if ( Info.ctrlIndex == T9donePush ) // request to exit the dialog
                  done = true ;
               else if ( Info.ctrlIndex == T9dumpPush )
               {
                  uidumpEnabled = (uidumpEnabled ? false : true) ;
                  dp->SetPushbuttonText ( T9dumpPush, dumpitStrings[uidumpEnabled] ) ;
               }
               // (no other pushbuttons defined in this dialog)
            }
            else
            {
               // No button press, so nothing to do
            }
         }

         //*******************************************
         //* If focus is currently on a Text Box     *
         //*******************************************
         else if ( ic[icIndex].type == dctTEXTBOX )
         {
            //* It doesn't matter how this control gained the input    *
            //* focus, by Tab/ShiftTab or via hotkey. If by hotkey, no *
            //* processing was done anyway, so discard any hotkey data *
            Info.viaHotkey = false ;

            //* Allow user to modify the text data in the text box.    *
            //* Returns when edit is complete.                         *
            icIndex = dp->EditTextbox ( Info ) ;
            if ( uidumpEnabled )
            {  // To view contents of Info
               winPos wPos(2,1) ;
               dp->Dump_uiInfo ( wPos, Info ) ;
            }

            //* If the data in the text box being edited has changed *
            if ( Info.dataMod != false )
            {
               //* Do any processing related to a change in the text box data *
               if ( Info.ctrlIndex == T9planetBox )
               {
                  // DO STUFF
                  dp->DebugMsg ( "planetBox changed", 2 ) ;
               }
               else if ( Info.ctrlIndex == T9icecreamBox )
               {
                  // DO STUFF
                  dp->DebugMsg ( "icecreamBox changed", 2 ) ;
               }
               // no other text boxes defined in this dialog
               dp->DebugMsg ( "%" ) ;     // clear the message
            }
         }

         //*******************************************
         //* If focus is currently on a Scroll Box   *
         //*******************************************
         else if ( ic[icIndex].type == dctSCROLLBOX )
         {
            //* It doesn't matter how this control gained the input    *
            //* focus, by Tab/ShiftTab or via hotkey. If by hotkey, no *
            //* processing was done anyway, so discard any hotkey data *
            Info.viaHotkey = false ;

            //* Allow user to modify the selected member of the scroll box.*
            //* Returns when edit is complete.                             *
            icIndex = dp->EditScrollbox ( Info ) ;
            if ( uidumpEnabled )
            {  // To view contents of Info
               winPos wPos(2,1) ;
               dp->Dump_uiInfo ( wPos, Info ) ;
            }

            //* If 'selected' member of Scroll Box has changed *
            if ( Info.dataMod != false || Info.keyIn == nckENTER )
            {
               // NOTE: We have only one dctSCROLLBOX defined in this dialog.
               //       If we had more, we would test for that here.
               if ( Info.ctrlIndex == T9positionControl )
               {
                  //* Use the specified standard position *
                  winPos wPos (winPosList[Info.selMember].ypos, 
                                       winPosList[Info.selMember].xpos) ;
                  //* Move the window to the specified position *
                  if ( (dp->MoveWin ( wPos )) == OK )
                  {  //* Update the tracking control *
                     wkeyCode wktmp ;
                     T09_ControlUpdate ( icIndex, wktmp ) ;
                  }
                  else
                  {  // (this won't happen under our controlled circumstances)
                     dp->DebugMsg ( "   Invalid dialog position specified!!   ", 4 ) ;
                     dp->DebugMsg ( "%" ) ;
                  }
               }
            }
         }

         //*******************************************
         //* If focus is currently on a Radio Button *
         //*******************************************
         else if ( ic[icIndex].type == dctRADIOBUTTON )
         {
            //* Get user input for this control                            *
            //* Returns when radio-button control or radio-button group is *
            //* about to lose the input focus.                             *
            //* A selection may or may not have been made.                 *
            if ( Info.viaHotkey == false )
            {
               icIndex = dp->EditRadiobutton ( Info ) ;
               if ( uidumpEnabled )
               {  // To view contents of Info
                  winPos wPos(2,1) ;
                  dp->Dump_uiInfo ( wPos, Info ) ;
               }
            }
            //* If user selected the current control object via hotkey, we *
            //* don't need to call the input routine; but simply process   *
            //* the existing data.                                         *
            //* For radio buttons, display update will already have been   *
            //* done, so we can just move the focus to the next control.   *
            else
            {
               if ( uidumpEnabled )
               {  // To view contents of Info
                  winPos wPos(2,1) ;
                  dp->Dump_uiInfo ( wPos, Info ) ;
               }
               //* Information arrived in the Info.h_xxx fields.*
               //* Transfer it to the primary positions.        *
               //* Previous data in these fields has already    *
               //* been handled in prior loop iteration.        *
               Info.HotData2Primary () ;
               if ( uidumpEnabled )
               {  // To view contents of Info
                  winPos wPos(2,1) ;
                  dp->Dump_uiInfo ( wPos, Info ) ;
               }
            }
            
            //* If 'selected' button is a member of a button group *
            if ( Info.selMember < MAX_DIALOG_CONTROLS )
            {
               //* If a new 'selected' member of the group *
               //* OR previous selection was re-selected   *
               if ( Info.dataMod != false || Info.keyIn == nckENTER )
               {
                  //* Process the user's selection *
                  switch ( Info.selMember )
                  {
                     case T9tuscaloosaButton:
                        // DO STUFF
                        dp->DebugMsg ( "  Tuscaloosa!  ", 2 ) ;
                        break ;
                     case T9sandiegoButton:
                        // DO STUFF
                        dp->DebugMsg ( "  San Diego!  ", 2 ) ;
                        break ;
                     case T9cucamongaButton:
                        // DO STUFF
                        dp->DebugMsg ( "  Cucamonga!  ", 2 ) ;
                        break ;
                     case T9valparisoButton:
                        // DO STUFF
                        dp->DebugMsg ( "  Valpariso!  ", 2 ) ;
                        break ;
                  }
                  dp->DebugMsg ( "%" ) ;
               }
            }
            //* Else, button is an independent button *
            //* (Info.selMember==MAX_DIALOG_CONTROLS) *
            else
            {
               //* If it is the scottyButton AND its state has changed *
               if ( Info.ctrlIndex == T9scottyButton && Info.dataMod != false )
               {
                  if ( Info.isSel != false )
                  {
                     // Beam Kirk Up to the ship - saving his ass once again
                     dp->DebugMsg ( "  ~~~ Beam Up ~~~  ", 2 ) ;
                     dp->DebugMsg ( "%" ) ;
                  }
                  else
                  {
                     // Don't Beam Kirk Up - let him spend the night 
                     //                      with the sexy alien wench
                     dp->DebugMsg ( "  ~~~ No Beam Up ~~~  ", 2 ) ;
                     dp->DebugMsg ( "%" ) ;
                  }
               }
               // (no other independent buttons in this dialog)

            }
         }  // radio button

         //**********************************************
         //* If focus is currently on a Spinner Control *
         //**********************************************
         else if ( ic[icIndex].type == dctSPINNER )
         {
            //* It doesn't matter how this control gained the input    *
            //* focus, by Tab/ShiftTab or via hotkey. If by hotkey, no *
            //* processing was done anyway, so discard any hotkey data *
            Info.viaHotkey = false ;

            //* Allow user to modify the spinner control's value.      *
            //* Returns when edit is complete.                         *
            icIndex = dp->EditSpinner ( Info ) ;
            if ( uidumpEnabled )
            {  // To view contents of Info
               winPos wPos(2,1) ;
               dp->Dump_uiInfo ( wPos, Info ) ;
            }

            //* If displayed value has been modified *
            if ( Info.dataMod != false )
            {
               //* Take any necessary actions here.  *
               //* For now, we just report the value.*
               #if GLIB_USTRING_ENABLE != 0     // Include Glib::ustring class testing
               int spval ;
               dp->GetSpinnerValue ( T9spinnerControl, spval ) ;
               Glib::ustring str = Glib::ustring::compose("Spinner Value %i", spval ) ;
               dp->DebugMsg ( str.c_str(), 2 ) ;
               #else    // NON GLIB METHOD
               int spval ;
               dp->GetSpinnerValue ( T9spinnerControl, spval ) ;
               std::ostringstream ostr(ostringstream::out) ;
               ostr << "Spinner Value " << spval ;
               std::string str = ostr.str() ;
               dp->DebugMsg ( str.c_str(), 2 ) ;
               #endif   // GLIB_USTRING_ENABLE
               dp->DebugMsg ( "%" ) ;
            }
            else
            { /* Spinner data not modified, so nothing to do */ }
         }

         //**********************************************
         //* If focus is currently on a DropDown Box    *
         //**********************************************
         else if ( ic[icIndex].type == dctDROPDOWN )
         {
            //* It doesn't matter how this control gained the input    *
            //* focus, by Tab/ShiftTab or via hotkey. If by hotkey, no *
            //* processing was done anyway, so discard any hotkey data *
            //*                                                        *
            //* However, if you want the control to expand immediately *
            //* when EditDropDownBox() is called, then set             *
            //* Info.viaHotkey==true. Otherwise, the control will      *
            //* expand when the user presses Enter or one of the       *
            //* vertical scrolling keys. This is useful if the user    *
            //* arrived at the current control via hotkey because by   *
            //* selecting the control via hotkey, he/she has already   *
            //* demonstrated the intention to edit the control.        *
            //* (all other data in the Info class is ignored on entry) *
            //Info.viaHotkey = false ;

            //* Allow user to modify which member of the drop-down box *
            //* is selected.  Returns when edit is complete.           *
            icIndex = dp->EditDropdown ( Info ) ;
            if ( uidumpEnabled )
            {  // To view contents of Info
               //* Note that the coordinates for display are absolute 
               //* screen coordinates, not dialog offsets
               winPos wPos(2,1) ;
               dp->Dump_uiInfo ( wPos, Info ) ;
            }

            //* If 'selected' member of Drop-Down Box has changed *
            if ( Info.dataMod != false || Info.keyIn == nckENTER )
            {
               // selection has been modified - do any processing here
            }
         }

         //************************************************
         //* If focus is currently on a Scrollext Control *
         //************************************************
         else if ( ic[icIndex].type == dctSCROLLEXT )
         {
            //* It doesn't matter how this control gained the input    *
            //* focus, by Tab/ShiftTab or via hotkey. If by hotkey, no *
            //* processing was done anyway, so discard any hotkey data *
            Info.viaHotkey = false ;

            //* Allow user to modify the selected member of the scroll ext.*
            //* Returns when edit is complete.                             *
            icIndex = dp->EditScrollext ( Info ) ;
            if ( uidumpEnabled )
            {  // To view contents of Info
               winPos wPos(2,1) ;
               dp->Dump_uiInfo ( wPos, Info ) ;
            }

            //* If 'selected' member of Scroll Box has changed *
            if ( Info.dataMod != false || Info.keyIn == nckENTER )
            {
               // NOTE: We have only one dctSCROLLEXT defined in this dialog.
               //       If we had more, we would test for that here.
               if ( Info.ctrlIndex == T9scrollextControl )
               {
               }
            }
         }

         //**********************************************
         //* If focus is currently on a Menuwin Control *
         //**********************************************
         else if ( ic[icIndex].type == dctMENUWIN )
         {
            //* Note: If we arrived via hotkey, menu is expanded immediately *
            icIndex = dp->EditMenuwin ( Info ) ;
            if ( uidumpEnabled )
            {  // To view contents of Info
               winPos wPos(2,1) ;
               dp->Dump_uiInfo ( wPos, Info ) ;
            }

            //* If a selection has been made *
            if ( Info.dataMod != false )
            {
               #if 0    // TEST ONLY
               //* Most menu items in these menu controls have no functionality*
               //* attached; however, we informally use the following menu     *
               //* items to test the following NcDialog-class methods.         *
               //* SetScrollboxSelect and GetScrollboxSelect methods           *
               //*   T8menuwin01, index 2 and 3: get/set dctSCROLLBOX item     *
               //* SetScrollextSelect and GetScrollextSelect methods           *
               //*   T8menuwin01, index 4 and 5: get/set dctSCROLLEXT item     *
               //* SetDropdownSelect and GetDropdownSelect methods             *
               //*   T8menuwin01, index 6 and 7: get/set dctDROPDOWN item      *
               if ( Info.ctrlIndex == T9menuwin01 )
               {
                  short oldItem, newItem ;
                  //* Set scrollbox index *
                  if ( Info.selMember == 2 || Info.selMember == 3 )
                  {
                     oldItem = dp->GetScrollboxSelect ( T9positionControl ) ;
                     newItem = Info.selMember == 2 ? 2 : 3 ;
                     if ( newItem != oldItem )
                     {
                        if ( (dp->SetScrollboxSelect ( T9positionControl, 
                                                       newItem )) == newItem )
                        {
                           dp->DebugMsg ( "dctScrollBox selection modified", 2 ) ;
                           //* Move the window to the specified position *
                           winPos wPos (winPosList[newItem].ypos, 
                                                winPosList[newItem].xpos) ;
                           dp->MoveWin ( wPos ) ;
                        }
                        else     // error condition
                        {
                           dp->DebugMsg ( "dctScrollBox selection error", 2 ) ;
                        }
                        dp->DebugMsg ( "%" ) ;
                     }
                  }
                  else if ( Info.selMember == 4 || Info.selMember == 5 )
                  {
                     oldItem = dp->GetScrollextSelect ( T9scrollextControl ) ;
                     newItem = Info.selMember == 4 ? 10 : 23 ;
                     if ( newItem != oldItem )
                     {
                        if ( (dp->SetScrollextSelect ( T9scrollextControl, 
                                                       newItem )) == newItem )
                           dp->DebugMsg ( "dctScrollext selection modified", 2 ) ;
                        else     // error condition
                           dp->DebugMsg ( "dctScrollext selection error", 2 ) ;
                        dp->DebugMsg ( "%" ) ;
                     }
                  }
                  else if ( Info.selMember == 6 || Info.selMember == 7 )
                  {
                     oldItem = dp->GetDropdownSelect ( T9dropdownBox ) ;
                     newItem = Info.selMember == 6 ? 15 : 31 ;
                     if ( newItem != oldItem )
                     {
                        if ( (dp->SetDropdownSelect ( T9dropdownBox, 
                                                      newItem )) == newItem )
                           dp->DebugMsg ( "dctDropDown selection modified", 2 ) ;
                        else     // error condition
                           dp->DebugMsg ( "dctDropDown selection error", 2 ) ;
                        dp->DebugMsg ( "%" ) ;
                     }
                  }
               }
               #endif   // TEST ONLY
            }
         }

         else
            {  /* no other active controls defined for this dialog */ }

         //* If user exited the control edit method via a hotkey,*
         //* then the new control already has focus. Otherwise,  *
         //* move input focus to next/previous control.          *
         if ( done == false && Info.viaHotkey == false )
         {
            if ( Info.keyIn == nckSTAB )
               icIndex = dp->PrevControl () ; 
            else
               icIndex = dp->NextControl () ;
         }
      }  // while()
   }  // OpenWindow()
   else
   {  //* Most likely cause of dialog not opening is that the terminal window  *
      //* is too small. Give user a clue...                                    *
      short neededLines = (dialogROWS + minTestULY) >= minTERMROWS ?
                          (dialogROWS + minTestULY) : minTERMROWS, 
            neededCols  = dialogCOLS >= minTERMCOLS ? dialogCOLS : minTERMCOLS ;
      gString gs ;
      gs.compose ( L" Sorry, size of terminal window must be at least %hd x %hd for this test. ", 
                   &neededCols, &neededLines ) ;
      StatWin ( gs.gstr(), nc.cyG ) ;
      sleep ( 5 ) ;
   }

   if ( dp != NULL )                         // close the window
      delete ( dp ) ;

}  //* End Test09() *

//*************************
//*  T09_ControlUpdate    *
//*************************
//******************************************************************************
//* This is a callback method for manually updating the controls in the        *
//* Test09() dialog.                                                           *
//*                                                                            *
//*  For this test, the following are updated by the callback method:          *
//*   1. manual update of non-active (display-only) text box that reports      *
//*      the current position of the dialog window                             *
//*   2. translation of the Chinese item highlighted in the T9scrollextControl *
//*   3. Capture of dialog's text/attribute data to a file.                    *
//*      a) nckA_Z  : capture as plain text                                    *
//*      b) nckAS_Z : capture as full-color HTML                               *
//*      c) nckAS_S : capture as basic-color HTML (for Texinfo docs)           *
//*                                                                            *
//* Input  : currIndex: index of control that currently has focus              *
//*          wkey     : user's key input data                                  *
//*          firstTime: the EstablishCallback() method calls this method once  *
//*                     with firstTime==true, to perform any required          *
//*                     initialization. Subsequently, the NcDialog class       *
//*                     always calls with firstTime==false.                    *
//* Returns: OK                                                                *
//******************************************************************************
//* Important Note: This method makes some intimate assumptions about what     *
//* is happening in the method that established the callback. If changes are   *
//* made in the establishing method that affect the callback functionality,    *
//* be sure to update this method to address those changes.                    *
//*                                                                            *
//* Programmer's Note: An unlikely-but-possible error is induced if the dialog *
//* is moved while a menu-open sequence is being performed. Any control        *
//* obscured by the opening of the menu will cause an ugly display with the    *
//* obscured control superimposed on the menu. In such a case the dialog       *
//* window needs to be refreshed. See below.                                   *
//*                                                                            *
//******************************************************************************

short T09_ControlUpdate ( const short currIndex, const wkeyCode wkey, bool firstTime )
{
static winPos  oldPos(0,0) ;     // previous position of the dialog window
winPos         newPos(0,0) ;     // current position of the dialog window

   //*************************
   //* First-time processing *
   //*************************
   if ( firstTime != false )
   {
      //* Initialize the tracking data *
      // nothing to do at this time...
   }

   //***********************
   //* Standard processing *
   //***********************
   //* If dialog window has moved, update its coordinates *
   //* displayed in the T9trackrBox Text Box.             *
   newPos = Test09Dialog->GetDialogPosition () ;
   if ( newPos.ypos != oldPos.ypos || newPos.xpos != oldPos.xpos || firstTime != false )
   {
      gString  tboxBuff( " Y:%03hd   X:%03hd ", &newPos.ypos, &newPos.xpos ) ;
      Test09Dialog->SetTextboxText ( T9trackrBox, tboxBuff.gstr() ) ;
      oldPos = newPos ;
      //* Test for stale display data (see note above) *
      if ( currIndex == T9menuwin01 )
         Test09Dialog->RefreshWin () ;
   }
   if ( currIndex == T9scrollextControl || firstTime != false )
   {
      static const char* seTranslation[] =   
      {
         "What is the moral?                     ",  // 00
         "Thanks for the English translation.    ",  // 01
         "Spring and Autumn                      ",  // 02
         "War story                              ",  // 03
         "Small talk is boring.                  ",  // 04
         "What are you doing?                    ",  // 05
         "West is the white tiger.               ",  // 06
         "To sit cursing the wind                ",  // 07
         "Representative democracy               ",  // 08
         "Don't flatter me!                      ",  // 09
         "Don't bulls**t me!                     ",  // 10
         "Beauty is in the eye of the beholder.  ",  // 11
         "Say too much and you have made an error",  // 12
         "A clear conscience is a good pillow.   ",  // 13
         "Mountains are high,the emperor far away",  // 14
         "F*** you!                              ",  // 15
         "Fall in love at first sight            ",  // 16
         "You have good taste.                   ",  // 17
         "Get lost!                              ",  // 18
         "Toilet seat                            ",  // 19
         "Would you like to try it on?           ",  // 20
         "The horse is strong.                   ",  // 21
         "She is such a cat!                     ",  // 22
         "He is a pig!                           ",  // 23
         "The rabbit is soft.                    ",  // 24
      } ;
      short sextItem = Test09Dialog->GetScrollextSelect ( T9scrollextControl ) ;
      Test09Dialog->WriteString ( 19, 20, seTranslation[sextItem], nc.blR, true ) ;
   }
   //* If command to capture dialog's text/attribute data to a file.*
   //* This tests the CaptureDialog() method.                       *
   if ( (wkey.type == wktEXTEND) && 
        ((wkey.key == nckA_Z) || (wkey.key == nckAS_Z) || (wkey.key == nckAS_S)) )
   {
      const char* const capText = "./capturedlg.txt" ;
      const char* const capHtml = "./capturedlg.html" ;
      bool fmt = bool(wkey.key == nckAS_Z || wkey.key == nckAS_S) ;
      short capStatus ;

      //* Special, simplified HTML capture for insertion into Texinfo         *
      //* documentation. References CSS definition file 'infodoc-styles.css'. *
      if ( wkey.key == nckAS_S )
         capStatus = Test09Dialog->CaptureDialog ( capHtml, fmt, 
                  false, "../Texinfo/infodoc-styles.css", 1, false, nc.blR, true ) ;

      //* Standard, plain-text or table-based HTML capture *
      else
         capStatus = Test09Dialog->CaptureDialog ( (fmt ? capHtml : capText), fmt ) ;

      //* Report the results *
      gString gs ;
      gs.compose( L"Screen Capture to: '%s'", (fmt ? capHtml : capText) ) ;
      if ( capStatus != OK )
         gs.append( L" FAILED!" ) ;
      Test09Dialog->DebugMsg ( gs.ustr(), 2, true ) ;
   }

   return OK ;

}  //* End T09_ControlUpdate() *

//*************************
//*       Test10          *
//*************************
//******************************************************************************
//* This dialog tests the functionality of TextBox controls, specifically the  *
//* editing of text that includes a mixture of single-column and 2-column      *
//* characters in both LTR and RTL languages. Also exercised is the ability to *
//* edit text strings that are longer than the control is wide.                *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************
//*                                                                            *
//*                                                                            *
//******************************************************************************

void Test10 ( void )
{
   StatWin ( "Test #10: Textbox Controls - UTF-8 Data I/O" ) ; // keep user up-to-date

   //* "The best jihad is saying the truth in the face of a dictator." *
   const char* rtlText = "خير جهاد كلمة عدل لسلطان ظالم" ;
//OLD   const char* rtlText = "أفْضَلُ الجهاد كلمة عدل عند سلطان جائر" ;

   static const short dialogROWS = 29 ;      // display lines
   static const short dialogCOLS = 72 ;      // display columns
   short ctrY    = termRows/2,               // dialog center in Y
         ctrX    = termCols/2,               // dialog center in X
         //* Upper left corner in Y (cannot obscure status window) *
         ulY     = (ctrY - dialogROWS/2) >= minTestULY ? 
                   (ctrY - dialogROWS/2) : minTestULY,
         ulX     = ctrX - dialogCOLS / 2 ;   // upper left corner in X
   attr_t dColor = nc.blR ;                  // dialog window base color

   //****************************************
   //* Initial parameters for dialog window *
   //****************************************
InitCtrl ic[T10controlsDEFINED] =   // array of dialog control info
{
   { //* 'DONE' pushbutton - - - - - - - - - - - - - - - - - - - - T09donePush *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      
      short(dialogROWS - 4),        // ulY:       upper left corner in Y
      short(dialogCOLS / 2 - 8),    // ulX:       upper left corner in X
      1,                            // lines:     control lines
      15,                           // cols:      control columns
      "  DONE (完蛋)  ",             // dispText:  
      nc.gyR,                       // nColor:    non-focus color
      nc.gyre | ncbATTR,            // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "",                           // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[1],                       // nextCtrl:  link in next structure
   },
   {  //* Text Box, left-justified, init. text longer than field  - T10tboxLJ1 *
      dctTEXTBOX,                   // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      3,                            // ulY:       upper left corner in Y
      4,                            // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      40,                           // cols:      control columns
      // dispText    initial display text, several flavours for testing wide I/O
      #if 1    // mixed-width text - exercise many scrolling-column combinations
      "abcdefghij？kl你mno你pqrxxxxxxxxxxxxxxxRs你t命啊uv命wyz啊你ab\0",
//  0 1 2 3 4 5 6 7 8 9 10 11 12 13 14 15 16 17 18 19 20   
// "a b c d e f g h i j ？  k  l  你  m  n  o  你  p  q  r  . 
//  - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
//                    36 37 38 39 40 41 42 43 44 45 46 47 48 49 50 51 52
//                    R  s  你  t  命  啊  u  v  命  w  y  z  啊  你  a  b  \0",
      #elif 0  // plain ASCII one-column text (67 characters plus NULL)
      "This is the 'easy' test of scrolling through single-column text. ~~",
      #elif 0  // plain ASCII one-column text (120 characters plus NULL)
      "abcdefgABCDEFGhijklmnHIJKLMNopqrstuOPQRSTUvwxyzVWXYZ.!?-0060"
      "abcdefgABCDEFGhijklmnHIJKLMNopqrstuOPQRSTUvwxyzVWXYZ.!?-0120",
      #elif 0  // all two-column text (48 characters plus NULL)
      "你命啊？你命啊？你命啊？你命啊？你命啊？你命啊？你命啊？你命啊？你命啊？你命啊？你命啊？你命啊？",
      #elif 0  // long string for testing the gsALLOCDFLT limit (1024 plus NULL)
      "abcdefgABCDEFGhijklmnHIJKLMNopqrstuOPQRSTUvwxyzVWXYZ.!?-0060"
      "abcdefgABCDEFGhijklmnHIJKLMNopqrstuOPQRSTUvwxyzVWXYZ.!?-0120"
      "abcdefgABCDEFGhijklmnHIJKLMNopqrstuOPQRSTUvwxyzVWXYZ.!?-0180"
      "abcdefgABCDEFGhijklmnHIJKLMNopqrstuOPQRSTUvwxyzVWXYZ.!?-0240"
      "abcdefgABCDEFGhijklmnHIJKLMNopqrstuOPQRSTUvwxyzVWXYZ.!?-0300"
      "abcdefgABCDEFGhijklmnHIJKLMNopqrstuOPQRSTUvwxyzVWXYZ.!?-0360"
      "abcdefgABCDEFGhijklmnHIJKLMNopqrstuOPQRSTUvwxyzVWXYZ.!?-0420"
      "abcdefgABCDEFGhijklmnHIJKLMNopqrstuOPQRSTUvwxyzVWXYZ.!?-0480"
      "abcdefgABCDEFGhijklmnHIJKLMNopqrstuOPQRSTUvwxyzVWXYZ.!?-0540"
      "abcdefgABCDEFGhijklmnHIJKLMNopqrstuOPQRSTUvwxyzVWXYZ.!?-0600"
      "abcdefgABCDEFGhijklmnHIJKLMNopqrstuOPQRSTUvwxyzVWXYZ.!?-0660"
      "abcdefgABCDEFGhijklmnHIJKLMNopqrstuOPQRSTUvwxyzVWXYZ.!?-0720"
      "abcdefgABCDEFGhijklmnHIJKLMNopqrstuOPQRSTUvwxyzVWXYZ.!?-0780"
      "abcdefgABCDEFGhijklmnHIJKLMNopqrstuOPQRSTUvwxyzVWXYZ.!?-0840"
      "abcdefgABCDEFGhijklmnHIJKLMNopqrstuOPQRSTUvwxyzVWXYZ.!?-0900"
      "abcdefgABCDEFGhijklmnHIJKLMNopqrstuOPQRSTUvwxyzVWXYZ.!?-0960"
      "abcdefgABCDEFGhijklmnHIJKLMNopqrstuOPQRSTUvwxyzVWXYZ.!?-1020"
      "*234",
      #elif 0  // not-quite-so-long string
      "abcdefgABCDEFGhijklmnHIJKLMNopqrstuOPQRSTUvwxyzVWXYZ.!?-0060"
      "abcdefgABCDEFGhijklmnHIJKLMNopqrstuOPQRSTUvwxyzVWXYZ.!?-0120"
      "abcdefgABCDEFGhijklmnHIJKLMNopqrstuOPQRSTUvwxyzVWXYZ.!?-0180"
      "abcdefgABCDEFGhijklmnHIJKLMNopqrstuOPQRSTUvwxyzVWXYZ.!?-0240"
      "abcdefgABCDEFGhijklmnHIJKLMNopqrstuOPQRSTUvwxyzVWXYZ.!?-0300"
      "abcdefgABCDEFGhijklmnHIJKLMNopqrstuOPQRSTUvwxyzVWXYZ.!?-0360"
      "abcdefgABCDEFGhijklmnHIJKLMNopqrstuOPQRSTUvwxyzVWXYZ.!?-0420"
      "abcdefgABCDEFGhijklmnHIJKLMNopqrstuOPQRSTUvwxyzVWXYZ.!?-0480"
      "abcdefgABCDEFGhijklmnHIJKLMNopqrstuOPQRSTUvwxyzVWXYZ.!?-0540"
      "abcdefgABCDEFGhijklmnHIJKLMNopqrstuOPQRSTUvwxyzVWXYZ.!?-0600"
      "abcdefgABCDEFGhijklmnHIJKLMNopqrstuOPQRSTUvwxyzVWXYZ.!?-0660"
      "abcdefgABCDEFGhijklmnHIJKLMNopqrstuOPQRSTUvwxyzVWXYZ.!?-0720"
      "abcdefgABCDEFGhijklmnHIJKLMNopqrstuOPQRSTUvwxyzVWXYZ.!?-0780"
      "abcdefgABCDEFGhijklmnHIJKLMNopqrstuOPQRSTUvwxyzVWXYZ.!?-0840"
      "abcdefgABCDEFGhijklmnHIJKLMNopqrstuOPQRSTUvwxyzVWXYZ.!?-0900"
      "abcdefgABCDEFGhijklmnHIJKLMNopqrstuOPQRSTUvwxyzVWXYZ.!?-0960"
      "abcdefgABCDEFGhijklmnHIJKLMNopqrstuOPQRS-1005",
      #else    // blank line
      "",   // blank/empty/nothing/zip/zilch/nada
      #endif
      nc.bw,                        // nColor:    non-focus color
      nc.cyG,                       // fColor:    focus color
      tbPrint,                      // filter:    any printing character
      "Left-justified text - data width > field", // label:     
      -1,                           // labY:      
      0,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[T10tboxLJ2]               // nextCtrl:  link in next structure
   },
   {  //* Text Box, left-justified, with initial data  - - - - - -  T10tboxLJ2 *
      dctTEXTBOX,                   // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(ic[T10tboxLJ1].ulY + 3),// ulY:       upper left corner in Y
      short(ic[T10tboxLJ1].ulX),    // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      66,                           // cols:      control columns
      "Why are you buying so many vitamins? - 你干吗买这么多维他命啊？", // dispText:  
      nc.bl,                        // nColor:    non-focus color
      nc.cyG,                       // fColor:    focus color
      tbPrint,                      // filter:    any printing character
      "Left-justified text - data width < field", // label:     
      -1,                           // labY:      
      0,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[T10tboxRJ1]               // nextCtrl:  link in next structure
   },
   {  //* Text Box, right-justified, initially blank    - - - - - - T10tboxRJ1 *
      dctTEXTBOX,                   // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(ic[T10tboxLJ2].ulY + 3),// ulY:       upper left corner in Y
      short(ic[T10tboxLJ2].ulX),    // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      40,                           // cols:      control columns
      NULL,                         // dispText:  
      nc.bw,                        // nColor:    non-focus color
      nc.cyG,                       // fColor:    focus color
      tbPrint,                      // filter:    any printing character
      "Right-justified text - initially blank", // label:     
      -1,                           // labY:      
      0,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[T10tboxRJ2]               // nextCtrl:  link in next structure
   },
   {  //* Text Box, right-justified, with initial data  - - - - - - T10tboxRJ2 *
      dctTEXTBOX,                   // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(ic[T10tboxRJ1].ulY + 3),// ulY:       upper left corner in Y
      short(ic[T10tboxRJ1].ulX),    // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      66,                           // cols:      control columns
      "你在干什么 - What are you doing?", // dispText:  
      nc.bl,                        // nColor:    non-focus color
      nc.cyG,                       // fColor:    focus color
      tbPrint,                      // filter:    any printing character
      "Right-justified text - with initial data", // label:     
      -1,                           // labY:      
      0,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[T10tboxAJ]                // nextCtrl:  link in next structure
   },
   {  //* Text Box, RTL text    - - - - - - - - - - - - - - - - - -  T10tboxAJ *
      dctTEXTBOX,                   // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(ic[T10tboxRJ2].ulY + 3),// ulY:       upper left corner in Y
      short(ic[T10tboxRJ2].ulX),    // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      66,                           // cols:      control columns
      rtlText,                      // dispText:  
      nc.bl,                        // nColor:    non-focus color
      nc.cyG,                       // fColor:    focus color
      tbPrint,                      // filter:    any printing character
      "Right-To-Left languages - Arabic example", // label:     
      -1,                           // labY:      
      0,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[T10tboxLC]                // nextCtrl:  link in next structure
   },
   {  //* Text Box (inactive), display contents of T10tboxLJ1 - - -  T10tboxLC *
      dctTEXTBOX,                   // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(ic[T10tboxAJ].ulY + 3), // ulY:       upper left corner in Y
      short(ic[T10tboxAJ].ulX - 2), // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      68,                           // cols:      control columns
      NULL,                         // dispText:  (initially blank)
      nc.brR,                       // nColor:    non-focus color
      nc.brR,                       // fColor:    focus color
      tbPrint,                      // filter:    any printing character
      "Contents of Text Box #1",    // label:     
      -1,                           // labY:      
      0,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      false,                        // active:    cannot gain input focus
      &ic[T10tboxRC]                // nextCtrl:  link in next structure
   },
   {  //* Text Box (inactive), display contents of T10tboxRJ1 - - -  T10tboxRC *
      dctTEXTBOX,                   // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(ic[T10tboxLC].ulY + 2), // ulY:       upper left corner in Y
      short(ic[T10tboxLC].ulX),     // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      68,                           // cols:      control columns
      NULL,                         // dispText:  (initially blank)
      nc.brR,                       // nColor:    non-focus color
      nc.brR,                       // fColor:    focus color
      tbPrint,                      // filter:    any printing character
      "Contents of Text Box #3",    // label:     
      -1,                           // labY:      
      0,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      false,                        // active:    cannot gain input focus
      &ic[T10tboxINS]               // nextCtrl:  link in next structure
   },
   {  //* Text Box (inactive), display Insert/Overstrike Mode  - -  T10tboxINS *
      dctTEXTBOX,                   // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(ic[T10tboxRC].ulY + 2), // ulY:       upper left corner in Y
      short(ic[T10tboxRC].ulX),     // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      23,                           // cols:      control columns
      NULL,                         // dispText:  (initially blank)
      nc.bw,                        // nColor:    non-focus color
      nc.bw,                        // fColor:    focus color
      tbPrint,                      // filter:    any printing character
      "Use 'Insert' key to toggle Insert/Overstrike",    // label:     
      ZERO,                         // labY:      
      24,                           // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      false,                        // active:    cannot gain input focus
      &ic[T10SysCbRadio]            // nextCtrl:  link in next structure
   },
   {  //* Wayland Clipboard Indicator Radio button   - - - - - - T10SysCbRadio *
      dctRADIOBUTTON,               // type:      
      rbtS3a,                       // rbSubtype: standard, 3 chars wide
      false,                        // rbSelect:  default selection
      short(ic[T10tboxINS].ulY + 2),// ulY:       upper left corner in Y
      ic[T10tboxINS].ulX,           // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a)
      nc.grR,                       // nColor:    non-focus color
      nc.grR,                       // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "System Clipboard Connected", // label:
      ZERO,                         // labY:      
      4,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      false,                        // active:    allow control to gain focus
      &ic[T10etRadio]               // nextCtrl:  link in next structure
   },
   {  //* Extended Text Data Radio button  - - - - - - - - - - - -  T10etRadio *
      dctRADIOBUTTON,               // type:      
      rbtS5s,                       // rbSubtype: standard, 5 chars wide
      true,                         // rbSelect:  default selection
      short(dialogROWS - 6),        // ulY:       upper left corner in Y
      short(dialogCOLS - 7),        // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a)
      nc.bw,                        // nColor:    non-focus color
      nc.cyG,                       // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "Enable TextBox #1 Extended Text Input", // label:
      ZERO,                         // labY:      
      -39,                          // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[T10filterPush]            // nextCtrl:  link in next structure
   },
   { //* 'Filter Test' pushbutton  - - - - - - - - - - - - - - - T10filterPush *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(dialogROWS - 5),        // ulY:       upper left corner in Y
      short(dialogCOLS - 15),       // ulX:       upper left corner in X
      1,                            // lines:     control lines
      13,                           // cols:      control columns
      " Filter Test ",              // dispText:  
      nc.gyR,                       // nColor:    non-focus color
      nc.grG,                       // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "",                           // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      NULL,                         // nextCtrl:  link in next structure
   },
} ;

   //* Initial parameters for dialog window *
   InitNcDialog dInit( dialogROWS,     // number of display lines
                       dialogCOLS,     // number of display columns
                       ulY,            // Y offset from upper-left of terminal 
                       ulX,            // X offset from upper-left of terminal 
                       "  Text Box With Multi-byte Character I/O  ", // dialog title
                       ncltSINGLE,     // border line-style
                       dColor,         // border color attribute
                       dColor,         // interior color attribute
                       ic              // pointer to list of control definitions
                     ) ;

   //* Instantiate the dialog window *
   NcDialog* dp = new NcDialog ( dInit ) ;
   Test10Dialog = dp ;              // give callback method access to the dialog

   //* Open the dialog window *
   if ( (dp->OpenWindow()) == OK )
   {
      dp->WriteParagraph ( 3, 2, "1.\n\n\n2.\n\n\n3.\n\n\n4.\n\n\n5.", dColor ) ;

      //* Set T10tboxRJ1 and T10tboxRJ2 as a right-justified fields *
      dp->SetTextboxCursor ( T10tboxRJ1, tbcpRIGHTJUST ) ;
      dp->SetTextboxCursor ( T10tboxRJ2, tbcpRIGHTJUST ) ;

      //* Configure T10tboxAJ for RTL (right-to-left) editing. *
      dp->DrawContentsAsRTL ( T10tboxAJ ) ;

      //* Set cursor for T10tboxLJ2 to the 'append' position *
      dp->SetTextboxCursor ( T10tboxLJ2, tbcpAPPEND ) ;

      //* Mirror contents of primary left-justified and right-justified *
      //* controls. Exercises retrieve and set of text data.            *
      char uText[gsDFLTBYTES] ;
      dp->GetTextboxText ( T10tboxLJ1, uText ) ;
      dp->SetTextboxText ( T10tboxLC, uText  ) ;
      dp->GetTextboxText ( T10tboxRJ1, uText ) ;
      dp->SetTextboxText ( T10tboxRC, uText  ) ;

      dp->WriteParagraph ( (dialogROWS-3), 2, 
         "NOTE: To input characters not directly supported by your keyboard,\n"
         "      you need an IME (Input Method Editor) such as 'iBus' or 'scim'.", dColor ) ;

      //* Make our data visible *
      dp->RefreshWin () ;

      //* If the "wl-clipboard" utilities, "wl-copy" and "wl-paste" are        *
      //* installed on the local system, then establish communications         *
      //* between the Wayland system clipboard and the NcDialog's              *
      //* Textbox Local Clipboard.                                             *
      //* If unable to establish a connection with the system clipboard,       *
      //* copy/cut/paste operations will use the local clipboard only.         *
      //* CTRL+C == Copy, CTRL+X == Cut, CTRL+V == Paste, CTRL+A == Select all,*
      //* SHIFT+LeftArrow and SHIFT+RightArrow == Select by character.         *
      if ( (dp->wcbEnable ()) != false )
      {
         dp->SetRadiobuttonState ( T10SysCbRadio, true ) ;
      }

      //* Establish a call-back method that can be called from within the *
      //* NcDialog code. Called each time through a control's user-input  *
      //* loop so we can manually modify the controls when necessary.     *
      dp->EstablishCallback ( &T10_ControlUpdate ) ;

      //* Allow user to play with the controls.      *
      //* (Please see the notes in NcDialog.hpp on   *
      //* values returned (from the editing routines)*
      uiInfo Info ;                 // user interface data returned here
      short  icIndex = ZERO ;       // index of control with input focus
      bool   done = false ;         // loop control
      while ( ! done )
      {
         //*******************************************
         //* If focus is currently on a Pushbutton   *
         //*******************************************
         if ( ic[icIndex].type == dctPUSHBUTTON )
         {
            //* Get user input for this control *
            icIndex = dp->EditPushbutton ( Info ) ;

            //* If a Pushbutton was pressed *
            if ( Info.dataMod != false )
            {
               if ( Info.ctrlIndex == T10donePush )
                  done = true ;
               else if ( Info.ctrlIndex == T10filterPush )
               {
                  T10_FilterTest () ;
               }
            }
            else
            {
               // No button press, so nothing to do
            }
         }

         //*******************************************
         //* If focus is currently on a Text Box     *
         //*******************************************
         else if ( ic[icIndex].type == dctTEXTBOX )
         {
            //* Allow user to modify the text data in the text box. *
            //* Returns when edit is complete.                      *
            icIndex = dp->EditTextbox ( Info ) ;

            //* If the data in the text box being edited has changed *
           if ( Info.dataMod != false )
            {
               if ( Info.ctrlIndex == T10tboxLJ1 )
               {  //* 'Set' data in target text box *
                  dp->GetTextboxText ( T10tboxLJ1, uText ) ;
                  dp->SetTextboxText ( T10tboxLC, uText  ) ;
               }
               else if ( Info.ctrlIndex == T10tboxRJ1 )
               {  //* 'Display Message' data in target text box *
                  attr_t nfColor = dtbmNFcolor ;
                  dp->GetTextboxText ( T10tboxRJ1, uText ) ;
                  dtbmData uData( uText, &nfColor, true ) ;
                  dp->DisplayTextboxMessage ( T10tboxRC, uData ) ;
               }
            }
         }

         //*******************************************
         //* If focus is currently on a Radio Button *
         //*******************************************
         else if ( ic[icIndex].type == dctRADIOBUTTON )
         {
            icIndex = dp->EditRadiobutton ( Info ) ;
            if ( Info.ctrlIndex == T10etRadio && Info.dataMod != false )
            {  //* Toggle extended-text input for Text Box #1 *
               dp->FixedWidthTextbox ( T10tboxLJ1, Info.isSel ) ;

               //* If text box data may have been trucated, (extended text     *
               //* data disabled) update the secondary display of data in the  *
               //* display-only control.                                       *
               if ( Info.isSel == false )
               {
                  dp->GetTextboxText ( T10tboxLJ1, uText ) ;
                  dp->SetTextboxText ( T10tboxLC, uText  ) ;
               }
            }
         }

         //* Move input focus to next/previous control *
         if ( done == false )
         {
            if ( Info.keyIn == nckSTAB )
               icIndex = dp->PrevControl () ; 
            else
               icIndex = dp->NextControl () ;
         }
      }  // while()

      //* If currently connected to the system clipboard, *
      //* terminate the connection, delete temp files and *
      //* return resources to the system.                 *
      dp->wcbDisable () ;

   }  // OpenWindow()
   else
   {  //* Most likely cause of dialog not opening is that the terminal window  *
      //* is too small. Give user a clue...                                    *
      short neededLines = (dialogROWS + minTestULY) >= minTERMROWS ?
                          (dialogROWS + minTestULY) : minTERMROWS, 
            neededCols  = dialogCOLS >= minTERMCOLS ? dialogCOLS : minTERMCOLS ;
      gString gs ;
      gs.compose ( L" Sorry, size of terminal window must be at least %hd x %hd for this test. ", 
                   &neededCols, &neededLines ) ;
      StatWin ( gs.gstr(), nc.cyG ) ;
      sleep ( 5 ) ;
   }

   if ( dp != NULL )                         // close the window
      delete ( dp ) ;

}  //* End Test10() *

//*************************
//*  T10_ControlUpdate    *
//*************************
//******************************************************************************
//* This is a callback method for manually updating the controls in the        *
//* Test10() dialog.                                                           *
//*                                                                            *
//*  For this test, the following are updated by the callback method:          *
//*   1. T10tboxINS control indicates current state of Insert/Overstrike flag. *
//*   2.                                                                       *
//*   3.                                                                       *
//*                                                                            *
//* Input  : currIndex: index of control that currently has focus              *
//*          wkey     : user's key input data                                  *
//*          firstTime: the EstablishCallback() method calls this method once  *
//*                     with firstTime==true, to perform any required          *
//*                     initialization. Subsequently, the NcDialog class       *
//*                     always calls with firstTime==false.                    *
//* Returns: OK                                                                *
//******************************************************************************
//* Important Note: This method makes some intimate assumptions about what     *
//* is happening in the method that established the callback. If changes are   *
//* made in the establishing method that affect the callback functionality,    *
//* be sure to update this method to address those changes.                    *
//******************************************************************************

short T10_ControlUpdate ( const short currIndex, const wkeyCode wkey, bool firstTime )
{
static const char* ioText[2] = 
{
   " Overstrike(叠印) Mode ",
   "   Insert(插入) Mode   ",
} ;
static attr_t ioAttr[2][22] = 
{
   {  // overstrike mode
      nc.gr, nc.gr, nc.gr, nc.gr, nc.gr, nc.gr, nc.gr, nc.gr, nc.gr, nc.gr, nc.gr, nc.gr, 
      nc.re, nc.re,  // chinses characters in contrasting color
      nc.gr, nc.gr, 
      nc.bl, nc.bl, nc.bl, nc.bl, nc.bl, nc.bl  // "Mode "
   },
   {  // insert mode
      nc.re, nc.re, nc.re, nc.re, nc.re, nc.re, nc.re, nc.re, nc.re, nc.re, 
      nc.gr, nc.gr,  // chinese characters in contrasting color
      nc.re, nc.re, 
      nc.bl, nc.bl, nc.bl, nc.bl, nc.bl, nc.bl, nc.bl, nc.bl   // "Mode   "
   },
} ;
static bool oldInsState = false ;

   //*************************
   //* First-time processing *
   //*************************
   if ( firstTime != false )
   {
      oldInsState = Test10Dialog->IsOverstrike () ;
   }

   //***********************
   //* Standard processing *
   //***********************
   bool newInsState = Test10Dialog->IsOverstrike () ;
   if ( (newInsState && !oldInsState) || (!newInsState && oldInsState) || 
        (firstTime != false) )
   {
      const char*    textPtr ;
      const attr_t*  attrPtr ;
      if ( newInsState != false )
      {
         textPtr = ioText[0] ;
         attrPtr = ioAttr[0] ;
      }
      else
      {
         textPtr = ioText[1] ;
         attrPtr = ioAttr[1] ;
      }
      dtbmData ioData( textPtr, attrPtr ) ;
      Test10Dialog->DisplayTextboxMessage ( T10tboxINS, ioData) ;
      oldInsState = newInsState ;
   }
   return OK ;

}  //* End T10_ControlUpdate() *

//*************************
//*   T10_FilterTest      *
//*************************
//******************************************************************************
//* Pass various UTF-8 strings to the VerifyTbString() method to exercise the  *
//* text filters.                                                              *
//*                                                                            *
//* Called by Test10().                                                        *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void T10_FilterTest ( void )
{
const char  uFilter[]  = { "Invalid character detected by filter...." } ;
const char  uPrint[]   = {  "Aa Bb Cc Dd Ee Ff Gg Hh Ii Jj Kk Ll Mm - á Á à À ã Ã â Â ạ ā ả ȧ ă" } ;
const char  uUppLow[]  = {  "Aa Bb Cc Dd Ee Ff Gg Hh Ii Jj Kk Ll Mm   á Á à À ã Ã â Â ạ ā ả ȧ ă" } ;
const char uPrinting[][128] =
{  // variety of printing characters
 // ASCII    Russian  Polish  Chinese  Korean  Kanji  Hiragana  
   "AbCdEfGh поживае  piękną  这种评论荒谬绝  서반갑습니다  こんにちは  こんにちは。はじ ",
   // fail on non-printing characters
   "AbCdEfGh  \x07  IjKlMnOpQrStUvWxYz",
   // ASCII-only printing characters
   " !\"#$%&'()*+,-./0123456789:;<=>?@AbCdEfGhIjKlMnOpQrStUvWxYz[\\]^_`{|}~",
} ;
const char uAlpha[][80] = 
{
   "AbCdEfGhпоживаеpięknąáÁàÀãÃâÂạāảȧă",
   "AbCdEfGh поживае piękną á Á à À ã Ã â Â ạ ā ả ȧ ă",
   "AbCdEfGh!@#$поживаеpięknąáÁàÀãÃâÂạāảȧă",
} ;
const char uNumerics[][64] = 
{
   "0123+4567-89.0",                   // tbNumeric
   "0123+4567-89.0=4601",              // tbNumeric (invalid character)
   "0123456789",                       // tbNumber
   "0123456789零一二三四五六七八九壹贰叁",          // tbNumber (fail on Chinese number)
                                       // (0123456789 + 123 financial)
   "12A78B90C34D56E78F1a2b3c4d5e6f7",  // HexNum
   "12g78B90C34D56E78F",               // HexNum (invalid character)
} ;
const char uAlphaNums[][128] = 
{
   "AbCживięą这种评서반갑こんにこんに.0123零一二三四五六七八九壹贰叁",       // AlNum with PERIOD
   "AbCd жива pięą 这种评论 서반갑습 こんにち こんにち.0123456789 " // AlNum with PERIOD and SPACE
      "零一二三四五六七八九 壹贰叁",
   "AbC@живięą这种评서반갑こんにこんに.0123零一二三四五六七八九壹贰叁",      // AlNum (with invalid char)
} ;
const char uFNames[][128] = 
{
   "http://www.ThisIsMy+-_.~+0123456789+!*'();:@&=+$,/?#[Website].com",   // URL specification
   "AbCd жива pięą 这种评论 서반갑습 こんにち こんにち.0123456789.odt",
   "/home/жива-pięą这[种评论]/서반갑습@こんに#2/PhotoMauiTrip 0456.jpg",
} ;
attr_t   dColor = nc.blR ;          // message color
winPos   wp( 17, 26 ) ;             // message area
gString  gs,                        // format conversion
         gsFiltered ;               // source text for filtering
//char     uBuff[gsDFLTBYTES] ;       // UTF-8 work buffer

   //* Instantiate and initialize the dtbmData class for displaying messages   *
   //* in a text box.                                                          *
   attr_t nfColor = dtbmNFcolor ;
   dtbmData uData( "Beginning TextBox Text-filtering test", &nfColor ) ;

   Test10Dialog->WriteString ( wp, 
               "[Press Any Key To Begin Test               ]", dColor, true ) ;
   Test10Dialog->DisplayTextboxMessage ( T10tboxRC, uData ) ;
   nckPause() ;

   //*****************************
   //* All Printing Characters   *
   //*****************************
   Test10Dialog->WriteString ( wp, 
               "[tbPrint: All printing characters          ]", dColor, true ) ;
   gsFiltered = uPrinting[0] ;
   if ( (Test10Dialog->VerifyTbText ( T10tboxRC, gsFiltered, tbPrint )) == OK )
      uData = gsFiltered.ustr() ;
   else
      uData = uFilter ;    // invalid character found
   Test10Dialog->DisplayTextboxMessage ( T10tboxRC, uData ) ;
   nckPause() ;

   Test10Dialog->WriteString ( wp, 
               "[tbPrint: All printing chars to upper case ]", dColor, true ) ;
   gsFiltered = uPrinting[0] ;
   if ( (Test10Dialog->VerifyTbText ( T10tboxRC, gsFiltered, tbPrintUpper )) == OK )
      uData = gsFiltered.ustr() ;
   else
      uData = uFilter ;    // invalid character found
   Test10Dialog->DisplayTextboxMessage ( T10tboxRC, uData ) ;
   nckPause() ;

   Test10Dialog->WriteString ( wp, 
               "[tbPrint: All printing chars to lower case ]", dColor, true ) ;
   gsFiltered = uPrinting[0] ;
   if ( (Test10Dialog->VerifyTbText ( T10tboxRC, gsFiltered, tbPrintLower )) == OK )
      uData = gsFiltered.ustr() ;
   else
      uData = uFilter ;    // invalid character found
   Test10Dialog->DisplayTextboxMessage ( T10tboxRC, uData ) ;
   nckPause() ;

   Test10Dialog->WriteString ( wp, 
               "[tbPrint: All printing (fail on non-print) ]", dColor, true ) ;
   gsFiltered = uPrinting[1] ;
   if ( (Test10Dialog->VerifyTbText ( T10tboxRC, gsFiltered, tbPrint )) == OK )
      uData = gsFiltered.ustr() ;
   else
      uData = uFilter ;    // invalid character found
   Test10Dialog->DisplayTextboxMessage ( T10tboxRC, uData ) ;
   nckPause() ;

   Test10Dialog->WriteString ( wp.ypos, wp.xpos, 
               "[tbAsciiPrint: all ASCII printing chars    ]", dColor, true ) ;
   gsFiltered = uPrinting[2] ;
   if ( (Test10Dialog->VerifyTbText ( T10tboxRC, gsFiltered, tbAsciiPrint )) == OK )
      uData = gsFiltered.ustr() ;
   else
      uData = uFilter ;    // invalid character found
   Test10Dialog->DisplayTextboxMessage ( T10tboxRC, uData ) ;
   nckPause() ;

   Test10Dialog->WriteString ( wp.ypos, wp.xpos, 
               "[tbAsciiPrint: ASCII print (fail non-ascii)]", dColor, true ) ;
   gsFiltered = uPrinting[0] ;
   if ( (Test10Dialog->VerifyTbText ( T10tboxRC, gsFiltered, tbAsciiPrint )) == OK )
      uData = gsFiltered.ustr() ;
   else
      uData = uFilter ;    // invalid character found
   Test10Dialog->DisplayTextboxMessage ( T10tboxRC, uData ) ;
   nckPause() ;

   //*****************************
   //* Upper and Lower Case      *
   //*****************************
   Test10Dialog->WriteString ( wp.ypos, wp.xpos, 
               "[Mixed ASCII/Non-ASCII, and mixed case     ]", dColor, true ) ;
   gsFiltered = uUppLow ;
   if ( (Test10Dialog->VerifyTbText ( T10tboxRC, gsFiltered, tbPrint )) == OK )
      uData = gsFiltered.ustr() ;
   else
      uData = uFilter ;    // invalid character found
   Test10Dialog->DisplayTextboxMessage ( T10tboxRC, uData ) ;
   nckPause() ;

   Test10Dialog->WriteString ( wp.ypos, wp.xpos, 
               "[tbLowerSp: all to lower case              ]", dColor, true ) ;
   gsFiltered = uUppLow ;
   if ( (Test10Dialog->VerifyTbText ( T10tboxRC, gsFiltered, tbLowerSp )) == OK )
      uData = gsFiltered.ustr() ;
   else
      uData = uFilter ;    // invalid character found
   Test10Dialog->DisplayTextboxMessage ( T10tboxRC, uData ) ;
   nckPause() ;

   Test10Dialog->WriteString ( wp.ypos, wp.xpos, 
               "[tbUpperSp: all to upper case              ]", dColor, true ) ;
   gsFiltered = uUppLow ;
   if ( (Test10Dialog->VerifyTbText ( T10tboxRC, gsFiltered, tbUpperSp )) == OK )
      uData = gsFiltered.ustr() ;
   else
      uData = uFilter ;    // invalid character found
   Test10Dialog->DisplayTextboxMessage ( T10tboxRC, uData ) ;
   nckPause() ;

   Test10Dialog->WriteString ( wp.ypos, wp.xpos, 
               "[tbLower: all to lower (fail on space)     ]", dColor, true ) ;
   gsFiltered = uUppLow ;
   if ( (Test10Dialog->VerifyTbText ( T10tboxRC, gsFiltered, tbLower )) == OK )
      uData = gsFiltered.ustr() ;
   else
      uData = uFilter ;    // invalid character found
   Test10Dialog->DisplayTextboxMessage ( T10tboxRC, uData ) ;
   nckPause() ;

   Test10Dialog->WriteString ( wp.ypos, wp.xpos, 
               "[tbLower: all to lower (fail on non-u/l)   ]", dColor, true ) ;
   gsFiltered = uPrint ;
   if ( (Test10Dialog->VerifyTbText ( T10tboxRC, gsFiltered, tbLower )) == OK )
      uData = gsFiltered.ustr() ;
   else
      uData = uFilter ;    // invalid character found
   Test10Dialog->DisplayTextboxMessage ( T10tboxRC, uData ) ;
   nckPause() ;

   //*****************************
   //* Alphabetic Characters     *
   //*****************************
   Test10Dialog->WriteString ( wp.ypos, wp.xpos, 
               "[tbAlpha: Alphabetic - all alphabets       ]", dColor, true ) ;
   gsFiltered = uAlpha[0] ;
   if ( (Test10Dialog->VerifyTbText ( T10tboxRC, gsFiltered, tbAlpha )) == OK )
      uData = gsFiltered.ustr() ;
   else
      uData = uFilter ;    // invalid character found
   Test10Dialog->DisplayTextboxMessage ( T10tboxRC, uData ) ;
   nckPause() ;

   Test10Dialog->WriteString ( wp.ypos, wp.xpos, 
               "[tbAlphaSp: Alphabetic - all alpha + SPACE ]", dColor, true ) ;
   gsFiltered = uAlpha[1] ;
   if ( (Test10Dialog->VerifyTbText ( T10tboxRC, gsFiltered, tbAlphaSp )) == OK )
      uData = gsFiltered.ustr() ;
   else
      uData = uFilter ;    // invalid character found
   Test10Dialog->DisplayTextboxMessage ( T10tboxRC, uData ) ;
   nckPause() ;

   Test10Dialog->WriteString ( wp.ypos, wp.xpos, 
               "[tbAlpha: Alphabetic (fail on SPACE)       ]", dColor, true ) ;
   gsFiltered = uAlpha[1] ;
   if ( (Test10Dialog->VerifyTbText ( T10tboxRC, gsFiltered, tbAlpha )) == OK )
      uData = gsFiltered.ustr() ;
   else
      uData = uFilter ;    // invalid character found
   Test10Dialog->DisplayTextboxMessage ( T10tboxRC, uData ) ;
   nckPause() ;

   Test10Dialog->WriteString ( wp.ypos, wp.xpos, 
               "[tbAlpha: Alphabetic (fail on non-Alpha)   ]", dColor, true ) ;
   gsFiltered = uAlpha[2] ;
   if ( (Test10Dialog->VerifyTbText ( T10tboxRC, gsFiltered, tbAlpha )) == OK )
      uData = gsFiltered.ustr() ;
   else
      uData = uFilter ;    // invalid character found
   Test10Dialog->DisplayTextboxMessage ( T10tboxRC, uData ) ;
   nckPause() ;

   Test10Dialog->WriteString ( wp.ypos, wp.xpos, 
               "[tbAlphaSp: Alphabetic+SP (fail non-Alpha) ]", dColor, true ) ;
   gsFiltered = uPrinting[0] ;
   if ( (Test10Dialog->VerifyTbText ( T10tboxRC, gsFiltered, tbAlphaSp )) == OK )
      uData = gsFiltered.ustr() ;
   else
      uData = uFilter ;    // invalid character found
   Test10Dialog->DisplayTextboxMessage ( T10tboxRC, uData ) ;
   nckPause() ;

   //*****************************
   //* Alphanumeric Characters   *
   //*****************************
   Test10Dialog->WriteString ( wp.ypos, wp.xpos, 
               "[tbAlNum: all alphabets and number systems ]", dColor, true ) ;
   gsFiltered = uAlphaNums[0] ;
   if ( (Test10Dialog->VerifyTbText ( T10tboxRC, gsFiltered, tbAlNum )) == OK )
      uData = gsFiltered.ustr() ;
   else
      uData = uFilter ;    // invalid character found
   Test10Dialog->DisplayTextboxMessage ( T10tboxRC, uData ) ;
   nckPause() ;

   Test10Dialog->WriteString ( wp.ypos, wp.xpos, 
               "[tbAlNumSp: all alpha and num plus SPACE   ]", dColor, true ) ;
   gsFiltered = uAlphaNums[1] ;
   if ( (Test10Dialog->VerifyTbText ( T10tboxRC, gsFiltered, tbAlNumSp )) == OK )
      uData = gsFiltered.ustr() ;
   else
      uData = uFilter ;    // invalid character found
   Test10Dialog->DisplayTextboxMessage ( T10tboxRC, uData ) ;
   nckPause() ;

   Test10Dialog->WriteString ( wp.ypos, wp.xpos, 
               "[tbAlNum: all alpha and num (fail on SPACE)]", dColor, true ) ;
   gsFiltered = uAlphaNums[1] ;
   if ( (Test10Dialog->VerifyTbText ( T10tboxRC, gsFiltered, tbAlNum )) == OK )
      uData = gsFiltered.ustr() ;
   else
      uData = uFilter ;    // invalid character found
   Test10Dialog->DisplayTextboxMessage ( T10tboxRC, uData ) ;
   nckPause() ;

   Test10Dialog->WriteString ( wp.ypos, wp.xpos, 
               "[tbAlNum: all alpha/num (fail on non-alnum)]", dColor, true ) ;
   gsFiltered = uAlphaNums[2] ;
   if ( (Test10Dialog->VerifyTbText ( T10tboxRC, gsFiltered, tbAlNum )) == OK )
      uData = gsFiltered.ustr() ;
   else
      uData = uFilter ;    // invalid character found
   Test10Dialog->DisplayTextboxMessage ( T10tboxRC, uData ) ;
   nckPause() ;

   //*****************************
   //* Numeric Chars and Numbers *
   //*****************************
   Test10Dialog->WriteString ( wp.ypos, wp.xpos, 
               "[tbNumeric: 0-9 + PERIOD/MINUS/PLUS (ASCII)]", dColor, true ) ;
   gsFiltered = uNumerics[0] ;
   if ( (Test10Dialog->VerifyTbText ( T10tboxRC, gsFiltered, tbNumeric )) == OK )
      uData = gsFiltered.ustr() ;
   else
      uData = uFilter ;    // invalid character found
   Test10Dialog->DisplayTextboxMessage ( T10tboxRC, uData ) ;
   nckPause() ;

   Test10Dialog->WriteString ( wp.ypos, wp.xpos, 
               "[tbNumeric: Numeric (fail on non-numeric)  ]", dColor, true ) ;
   gsFiltered = uNumerics[1] ;
   if ( (Test10Dialog->VerifyTbText ( T10tboxRC, gsFiltered, tbNumeric )) == OK )
      uData = gsFiltered.ustr() ;
   else
      uData = uFilter ;    // invalid character found
   Test10Dialog->DisplayTextboxMessage ( T10tboxRC, uData ) ;
   nckPause() ;

   Test10Dialog->WriteString ( wp.ypos, wp.xpos, 
               "[tbNumber: zero through nine (0-9) (ASCII) ]", dColor, true ) ;
   gsFiltered = uNumerics[2] ;
   if ( (Test10Dialog->VerifyTbText ( T10tboxRC, gsFiltered, tbNumber )) == OK )
      uData = gsFiltered.ustr() ;
   else
      uData = uFilter ;    // invalid character found
   Test10Dialog->DisplayTextboxMessage ( T10tboxRC, uData ) ;
   nckPause() ;

   Test10Dialog->WriteString ( wp.ypos, wp.xpos, 
               "[tbNumber: zero through nine (fail non-num)]", dColor, true ) ;
   gsFiltered = uNumerics[3] ;
   if ( (Test10Dialog->VerifyTbText ( T10tboxRC, gsFiltered, tbNumber )) == OK )
      uData = gsFiltered.ustr() ;
   else
      uData = uFilter ;    // invalid character found
   Test10Dialog->DisplayTextboxMessage ( T10tboxRC, uData ) ;
   nckPause() ;

   Test10Dialog->WriteString ( wp.ypos, wp.xpos, 
               "[tbHexNum: '0'-'9','A'-'F','a'-'f' (ASCII) ]", dColor, true ) ;
   gsFiltered = uNumerics[4] ;
   if ( (Test10Dialog->VerifyTbText ( T10tboxRC, gsFiltered, tbHexNum )) == OK )
      uData = gsFiltered.ustr() ;
   else
      uData = uFilter ;    // invalid character found
   Test10Dialog->DisplayTextboxMessage ( T10tboxRC, uData ) ;
   nckPause() ;

   Test10Dialog->WriteString ( wp.ypos, wp.xpos, 
               "[tbHexNum: Hexidecimal, force uppercase    ]", dColor, true ) ;
   gsFiltered = uNumerics[4] ;
   if ( (Test10Dialog->VerifyTbText ( T10tboxRC, gsFiltered, tbHexNumUp )) == OK )
      uData = gsFiltered.ustr() ;
   else
      uData = uFilter ;    // invalid character found
   Test10Dialog->DisplayTextboxMessage ( T10tboxRC, uData ) ;
   nckPause() ;

   Test10Dialog->WriteString ( wp.ypos, wp.xpos, 
               "[tbHexNum: Hexidecimal (fail on non-hex)   ]", dColor, true ) ;
   gsFiltered = uNumerics[5] ;
   if ( (Test10Dialog->VerifyTbText ( T10tboxRC, gsFiltered, tbHexNum )) == OK )
      uData = gsFiltered.ustr() ;
   else
      uData = uFilter ;    // invalid character found
   Test10Dialog->DisplayTextboxMessage ( T10tboxRC, uData ) ;
   nckPause() ;

   //*****************************
   //* Filename/Pathname/URL     *
   //*****************************
   // Programmer's Note: The characters which should be valid for URL specifications 
   // is still under discussion. When a decision is made, we will impliment it.
   // Currently, this filter only allows characters specified in RFC3986.
   Test10Dialog->WriteString ( wp.ypos, wp.xpos, 
               "[tbURL: characters valid for URL spec.     ]", dColor, true ) ;
   gsFiltered = uFNames[0] ;
   if ( (Test10Dialog->VerifyTbText ( T10tboxRC, gsFiltered, tbURL )) == OK )
      uData = gsFiltered.ustr() ;
   else
      uData = uFilter ;    // invalid character found
   Test10Dialog->DisplayTextboxMessage ( T10tboxRC, uData ) ;
   nckPause() ;

   Test10Dialog->WriteString ( wp.ypos, wp.xpos, 
               "[tbFileName: valid filename characters     ]", dColor, true ) ;
   gsFiltered = uFNames[1] ;
   if ( (Test10Dialog->VerifyTbText ( T10tboxRC, gsFiltered, tbFileName )) == OK )
      uData = gsFiltered.ustr() ;
   else
      uData = uFilter ;    // invalid character found
   Test10Dialog->DisplayTextboxMessage ( T10tboxRC, uData ) ;
   nckPause() ;

   Test10Dialog->WriteString ( wp.ypos, wp.xpos, 
               "[tbPathName: filename characters plus SLASH]", dColor, true ) ;
   gsFiltered = uFNames[2] ;
   if ( (Test10Dialog->VerifyTbText ( T10tboxRC, gsFiltered, tbPathName )) == OK )
      uData = gsFiltered.ustr() ;
   else
      uData = uFilter ;    // invalid character found
   Test10Dialog->DisplayTextboxMessage ( T10tboxRC, uData ) ;
   nckPause() ;

   //* Signal that we're done... *
   Test10Dialog->WriteString ( wp, 
               "[Press Any Key To End Test                 ]", dColor, true ) ;
   uData = " ** Press Any Key To End Test **" ;
   Test10Dialog->DisplayTextboxMessage ( T10tboxRC, uData ) ;
   nckPause() ;

   //* Clear the text box and the message area *
   uData = "" ;
   Test10Dialog->DisplayTextboxMessage ( T10tboxRC, uData ) ;
   Test10Dialog->WriteString ( wp.ypos, wp.xpos, 
               "                                            ", dColor, true ) ;

}  //* End T10_FilterTest() *

//*************************
//*    Test_wcout         *
//*************************
//******************************************************************************
//* Test the basic functionality of "wide" i/o as implemented by the C++       *
//* standard library iostream classes, wcout and wcin.                         *
//*                                                                            *
//* Note that the NCurses engine is not running at this time. This is a        *
//* standard, C++ console application, so far as this test is concerned.       *
//*                                                                            *
//*                                                                            *
//*                                                                            *
//* Input  : adjustLocaleSettings (optional, true by default)                  *
//*             if 'true', adjust the global and iostream locale settings      *
//*             before performing any i/o operations                           *
//*          newLocale (optional, NULL by default)                             *
//*             if adjustLocaleSettings == true AND newLocale != NULL, then    *
//*             points to a locale string to be used instead of the default    *
//*             specified by the terminal's environment variables.             *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************
//* Programmer's Notes:                                                        *
//* Note that on Fedora 12, Gnome Terminal, /bin/bash, en_US.UTF-8 is the      *
//* default locale for U.S. machines. Any locale of the form: xx_XX.UTF-8      *
//* should work fine with this test. The only requirements are that the        *
//* character encoding must be UTF-8, and that the terminal font support all . *
//* or most of the UTF-8 character set.                                        *
//* Note that the setlocale() C-language function queries the LC_ALL=,         *
//* LC_CTYPE=, or LANG= environment variables in that order and uses the       *.                                                               *
//* first one that has a value. Because this is a test of C++, however, we do  *
//* things the hard way with the locale class (see below).                     *
//*                                                                            *
//* 'C' library version of setting locale:                                     *
//*   setlocale ( LC_ALL, "" ) ;  // set locale from the environment variable  *
//*   setlocale ( LC_ALL, "en_US.UTF-8" ) ;  // set locale explicitly          *
//*                                                                            *
//* As of 07 Feb, 2011, the date of the original test, with                    *
//* (Fedora 12, Kernel Linux 2.6.32.26-175.fc12.i686.PAE                       *
//*  GCC, 4,4,4 20100630  -  lib                                               *
//* whatever internal members of wcout.getloc(), wcout.imbue() and             *
//* cout.getloc(), cout.imbue() are used for their locale information, are     *
//* ignored or overridden by the global settings made by the call to           *
//* locale::global(). The timing of the call to locale::global() may also be   *
//* critical to proper handling of the output-stream data.                     *
//*                                                                            *
//* Note that if we don't set the global locale at all, any supra-ASCII        *
//* characters will not be displayed properly, but will instead be displayed   *
//* as Windings, question marks, boxes, character-without-its-accent-mark, etc.*
//* This is true regardless of the locale values explicitly set in the wcout   *
//* and cout streams.                                                          *
//*                                                                            *
//* Note that attempts to use cout to output anything after the first call to  *
//* wcout fail utterly. Nothing gets out using that stream once the locale is  *
//* set and wcout has been used.                                               *
//*                                                                            *
//* If, on the other hand, cout is called after locale is set but BEFORE any   *
//* call to wcout, the cout (ASCII) text works fine, BUT any supra-ASCII data  *
//* subsequently sent through wcout will result in question marks, boxes, etc. *
//*                                                                            *
//* The mbstowcs() method prototype indicates that a size_t value is returned, *
//* indicating the number of characters converted (not including the NULLCHAR).*
//* size_t is apparently an UNSIGNED int because in the case of error, the     *
//* value returned is 4294967295. This would be a -1 in a signed integer.      *
//* This is a bug either in the documentation or in the library. We compensate *
//* by capturing the return value in a signed integer, so we can test for a    *
//* minus one (-1) return value. It is important to test the return value      *
//* because otherwise the receiving array is filled with garbage from the      *
//* previous call - beginning at the first UTF-8 character encountered.        *
//*                                                                            *
//* It appears that mbstowcs() always returns an error IF the locale is not    *
//* set to support UTF-8 AND it is given a string containing UTF-8 characters  *
//* to convert. This is reasonable, but it should be clearly documented.       *
//*                                                                            *
//* If we do not allow the initial adjustment of the locale, but we do allow   *
//* the "C" and back again setting, then the second time we convert the UTF-8  *
//* data to wchar_t, we do not get the return value of minus one (-1).         *
//* Although it is not clear why, it is localized around the second call to    *
//* loco.global(logo) near the bottom of the first page of display data.       *
//* Either the contents of loco may be different the second time or the second *
//* wcout.imbue(loco) does does something different the second time.           *
//* When we put the "C" and back again locale setting inside the               *
//* if(adjustLocaleSettings) conditional execution, we get consistent          *
//* mbstowcs errors throughout the test. When we enable the "back again" calls *
//* the mbstowcs no longer fails. It cannot convert the UTF-8 characters, and  *
//* reports them as question marks (presumably because i/o has already         *
//* occurred, blocking a complete locale setting change, but it converts the   *
//* ASCII portion of the strings. There is a clue here about the library's     *
//* internal inconsisencies, but I can't determine exactly what it is.         *
//*                                                                            *
//* Kanji and Hiragana found at:                                               *
//* http://www.freejapaneselessons.com/lesson10.cfm                            *
//*                                                                            *
//* Arabic found at:                                                           *
//* http://mylanguages.org/arabic_questions.php                                *
//*                                                                            *
//* Arabic found at:                                                           *
//* http://www.linguanaut.com/english_hebrew.htm                               *
//*                                                                            *
//*                     - - - - - - - - - - - - - - - -                        *
//* Certain characters in the wchar_t strings (such as my Chinese name), and   *
//* and UTF-8 encoded characters in the cXXX strings defined below, may not    *
//* display properly in your editor; however, they should display properly     *
//* when written to the console (presuming that your editor is UTF-8 enabled). *
//*                                                                            *
//******************************************************************************

void Test_wcout ( bool adjustLocaleSettings, const char* newLocale )
{
const wchar_t  wTitle[] =
{ 
   L"Dialogw - (c) 2011-2024  Mahlon R. Smith 马伦"
    "\n   (size terminal to at least 132 x 30 for best results)"
    "\n--------------------------------------------------------"
} ;
const wchar_t  wSubHeadTa[] = 
{
  L"\nThis is a test of std::wcout, mbstowcs and wcstombs. "
   "The NCurses engine is not running at this time."
} ;
const wchar_t  wSubHeadTb[] = 
{
  L"\nWe will now test output from the std::wstring class "
   "using the same UTF-8 data set."
} ;
const wchar_t  wSubHeadTc[] = 
{
  L"\n\nWe will now test output from the Glib::ustring class "
   "using the same UTF-8 data set."
} ;
const wchar_t  wSubHead01a[] = 
{
   L"These are UTF-8 strings, converted to wchar_t for display "
    "(note that some glyphs require two display columns):"
} ;
const wchar_t  wSubHead01b[] = 
{
   L"These are UTF-8 strings, converted to wstring for display "
    "(note that some glyphs require two display columns):"
} ;
const wchar_t  wSubHead01c[] = 
{
   L"These are UTF-8 strings, copied directly into the ustring class for display."
} ;
const wchar_t  wSubHead02[] = 
{
   L"We will now test the robustness of the iostream class. "
    "Switch locale of global and wcout to \"C\" and back again.:"
} ;
const wchar_t  mbstowcsError[] = { L"  mbstowcs error" } ;
const char cAsciiCstring01[] = 
{
   "This string was converted from a 7-bit ASCII char string to a "
   "wchar_t string for display."
} ;
const char cAsciiCstring02[] = 
{
   "This is a 7-bit ASCII character string."
} ;
const wchar_t  wideCstring1[] =
{
   L"This is a native wchar_t string containing mixed-width "
   "characters á Á à À ã Ã â Â ạ ā ả ȧ ă."
} ;
const char  cChinese01[] =  
{
   "  这种评论荒谬绝伦。                       "
   "- Chinese: \"This comment is absolutely preposterous.\""
} ;
const char  cChinese02[] =  
{
   "  你为什么不能在家过年啊？                 "
   "- Chinese: \"Why can't you celebrate the Spring Festival at home?\""
} ;
const char  cKorean01[] = 
{
   "  안녕하세요. 만나서 반갑습니다.           "
   "- Korean: \"Hello. It is a pleasure to meet you.\""
} ;
const char  cKorean02[] = 
{
   "  저는 골프와 테니스 등 스포츠를 좋아해요. "
   "- Korean: \"I love sports, especially golf and tennis.\""
} ;
const char  cKanji01[] = 
{
   "  こんにちは。始めまして。                 "
   "- Kanji: \"Hello. Nice to meet you.\""
} ;
const char  cKanji02[] = 
{
   "  始めまして。お元気ですか。               "
   "- Kanji: \"Nice to meet you. How are you?\""
} ;
const char  cHiragana01[] = 
{
   "  こんにちは。はじめまして。               "
   "- Hiragana: \"Hello. Nice to meet you.\""
} ;
const char  cHiragana02[] = 
{
   "  はじめまして。おげんきですか。           "
   "- Hiragana: \"Nice to meet you. How are you?\""
} ;
const char  cRussian[] = 
{
   "  Доброе утро. Как поживаете?              "
   "- Russian: \"Good morning. How are you?\""
} ;
const char  cPolish[] = 
{
   "  On patrzy na tę piękną dziewczynę.       "
   "- Polish: \"He is looking at the beautiful girl.\""
} ;
const char  cArabic[] = 
{
   "  هَل تَتَكَلَّم الْانْجِلِيزِيَّة؟"
} ;
const char  cArabicTrans[] = 
{ "                     - Arabic: \"Do you speak English?\"" } ;
const char cHebrew[]     = { "  הלו חבר! מה נשמע?" } ;
const char  cHebrewTrans[] = 
{ "                        - Hebrew: \"Hey, friend! What's new?\"" } ;

//const wchar_t wRussian[] = { L"Доброе утро. Как поживаете?" } ;
//const char    uRussian[] = {  "Доброе утро. Как поживаете?" } ;
//const wchar_t wPolish[]  = { L"On patrzy na tę piękną dziewczynę." } ;
//const char    uPolish[]  = {  "On patrzy na tę piękną dziewczynę." } ;
//const wchar_t wPrint[]   = { L"Aa Bb C Dd Ee Ff Gg Hh Ii Jj Kk Ll Mm - á Á à À ã Ã â Â ạ ā ả ȧ ă" } ;



wchar_t  w[1024] ;                  // wide-character strings (int[])
char     b[1024] ;                  // byte-character strings
std::string    sString ;            // for capturing various info
std::wstring   wString ;
int      strLen ;                   // characters converted (see note above)
locale   current_loco ;             // holds current global locale setting
locale   current_wloco ;            // holds current wcout locale setting
locale   current_cloco ;            // holds current cout locale setting
locale   current_eloco ;            // holds current cerr locale setting

   //* Get the original locale settings, both global locale and ostream locales*
   strncpy (b, setlocale (LC_ALL, NULL), 1024) ; // get original global locale
   current_wloco = wcout.getloc () ;      // get original wcout locale
   current_cloco = cout.getloc () ;       // get original cout locale
   current_eloco = cerr.getloc () ;       // get original cerr locale

   //* Set the locale according to the console environmentbefore doing any I/O.*
   //* This will probably be some flavour of UTF-8 on all modern OSs.          *
   //* To set locale explicitly use: locale loco ( "en_US.UTF-8" ) ;           *
   //*                         then: loco.global ( loco ) ;                    *
   //* Note that the sequence below SHOULD set the iostream locale members,    *
   //* but it does not. However, it does have the EFFECT of having set the     *
   //* iostream members. See ios_base::getloc() and ios_base::imbue()          *
   
   locale loco( "" ) ;              // get locale from the environment variable
   //* Set the 'locale' before performing any i/o operations. See notes above. *
   if ( adjustLocaleSettings != false )
   {
      if ( newLocale != NULL )
      {                             // set global locale with caller's string
         locale userLoco(newLocale) ;
         loco = userLoco ;
      }
      loco.global ( loco ) ;        // set global locale
      sString = loco.name () ;      // get current locale setting
   }
   else
      sString = b ;

   system ( "clear" ) ;             // clear the screen
   #if 0    // Stress Test - Do you want to see a mess? Let this print!
   cout << "!!! call cout before wcout to see what happens !!!" << endl ;
   #endif   // Stress Test
   //* Display application title and test description *
   gString gs ;   // Here, we test the return value of the 'compose' method
   wcout << gs.compose( L"%S%S", wTitle, wSubHeadTa ) << endl ;
   strLen = mbstowcs ( w, sString.c_str(), 1024 ) ;
   wcout << L"Your default 'locale' uses the following character encoding: \"" 
         << w << "\"" << endl ;
   wcout << L"Original APPLICATION locale settings:" 
         << "global:\"" << b << "\"  " ;
         sString = current_wloco.name() ;
         mbstowcs ( w, sString.c_str(), 1024 ) ;
         wcout << "wcout:\"" << w << "\"  " ;
         sString = current_cloco.name() ;
         mbstowcs ( w, sString.c_str(), 1024 ) ;
         wcout << "cout:\"" << w << "\"  " ;
         sString = current_eloco.name() ;
         mbstowcs ( w, sString.c_str(), 1024 ) ;
         wcout << "cerr:\"" << w << "\"" << endl ;

   //* "imbue" the iostream with the correct encoding.                         *
   //* Note that with or without this step, the wcout output is the same.      *
   if ( adjustLocaleSettings != false )
   {
      wcout.imbue ( loco ) ;
      cout.imbue ( loco ) ;
   }
   current_wloco = wcout.getloc () ;      // get adjusted wcout locale
   current_cloco = cout.getloc () ;       // get adjusted cout locale
   strncpy (b, setlocale (LC_ALL, NULL), 1024) ; // get adjusted global locale
   wcout << L"Adjusted APPLICATION locale settings:" 
         << "global:\"" << b << "\"  " ;
         sString = current_wloco.name() ;
         mbstowcs ( w, sString.c_str(), 1024 ) ;
         wcout << "wcout:\"" << w << "\"  " ;
         sString = current_cloco.name() ;
         mbstowcs ( w, sString.c_str(), 1024 ) ;
         wcout << "cout:\"" << w << "\"  " ;
         sString = current_eloco.name() ;
         mbstowcs ( w, sString.c_str(), 1024 ) ;
         wcout << "cerr:\"" << w << "\"" << endl ;

   wcout << endl ;                  // display test data
   wcout << wideCstring1 << endl ;
   strLen = mbstowcs ( w, cAsciiCstring01, 1024 ) ;
   wcout << w << endl ;
   wcout << cAsciiCstring02 << L"    (displayed with wcout)" << endl ;

   //** Pause **
   wcout << "\n  ** Please Press 'Enter' Key To Continue Test..." ;
   cin.get() ;
   system ( "clear" ) ;             // clear the screen

   wcout << endl << wSubHead01a << endl ;
   strLen = mbstowcs ( w, cChinese01, 1024 ) ;
   wcout << (strLen > ZERO ? w : mbstowcsError) << endl ;
   strLen = mbstowcs ( w, cChinese02, 1024 ) ;
   wcout << (strLen > ZERO ? w : mbstowcsError) << endl ;
   
   strLen = mbstowcs ( w, cKorean01, 1024 ) ;
   wcout << (strLen > ZERO ? w : mbstowcsError) << endl ;
   strLen = mbstowcs ( w, cKorean02, 1024 ) ;
   wcout << (strLen > ZERO ? w : mbstowcsError) << endl ;
   
   strLen = mbstowcs ( w, cKanji01, 1024 ) ;
   wcout << (strLen > ZERO ? w : mbstowcsError) << endl ;
   strLen = mbstowcs ( w, cKanji02, 1024 ) ;
   wcout << (strLen > ZERO ? w : mbstowcsError) << endl ;
   
   strLen = mbstowcs ( w, cHiragana01, 1024 ) ;
   wcout << (strLen > ZERO ? w : mbstowcsError) << endl ;
   strLen = mbstowcs ( w, cHiragana02, 1024 ) ;
   wcout << (strLen > ZERO ? w : mbstowcsError) << endl ;
   
   strLen = mbstowcs ( w, cRussian, 1024 ) ;
   wcout << (strLen > ZERO ? w : mbstowcsError) << endl ;
   strLen = mbstowcs ( w, cPolish, 1024 ) ;
   wcout << (strLen > ZERO ? w : mbstowcsError) << endl ;

   wchar_t rtlStart = 0x202B , rtlEnd = 0x202C ;
   strLen = mbstowcs ( w, cArabic, 1024 ) ;
   if ( strLen > ZERO )
   {
      wcout << rtlStart << w << rtlEnd ;
      strLen = mbstowcs ( w, cArabicTrans, 1024 ) ;
      wcout << (strLen > ZERO ? w : mbstowcsError) 
            << L" (RTL codes not recognized)" ;
   }
   else
      wcout << mbstowcsError ;
   wcout << endl ;

   strLen = mbstowcs ( w, cHebrew, 1024 ) ;
   if ( strLen > ZERO )
   {
      wcout << rtlStart << w << rtlEnd ;
      strLen = mbstowcs ( w, cHebrewTrans, 1024 ) ;
      wcout << (strLen > ZERO ? w : mbstowcsError) 
            << L" (RTL codes not recognized)" ;
   }
   else
      wcout << mbstowcsError ;
   wcout << endl ;

   //* Switch global and wcout locales to "C" *
   wcout << endl << wSubHead02 << endl ;
   locale old_loco = loco ;
   locale c_loco ( "C" ) ;
   loco = c_loco ;
   if ( adjustLocaleSettings != false )
   {
      loco.global ( loco ) ;
      wcout.imbue ( loco ) ;
   }
   current_loco = wcout.getloc () ;
   sString = current_loco.name () ;
   wcout << "global locale and wcout locale set to: \"" << sString.c_str() << "\"" << endl ;
   wcout << cAsciiCstring02 << endl ;
   wcout << wideCstring1 << endl ;
   strLen = mbstowcs ( w, cKorean02, 25 ) ;
   if ( strLen > ZERO )    w[strLen] = '\0' ;   // no null terminator was converted
   else                    wcscpy ( w, mbstowcsError ) ;
   wcout << "substring ("<< strLen 
      << " chars) of Korean UTF-8 string converted to wchar_t: " << w << endl ;

   //* Switch global and wcout locales back to earlier setting *
   loco = old_loco ;
   if ( adjustLocaleSettings != false )
   {
      loco.global ( loco ) ;
      wcout.imbue ( loco ) ;
   }
   current_loco = wcout.getloc () ;
   sString = current_loco.name () ;
   wcout << "\nglobal locale and wcout locale set to: \"" << sString.c_str() << "\"" << endl ;
   wcout << cAsciiCstring02 << endl ;
   wcout << wideCstring1 << endl ;
   strLen = mbstowcs ( w, cKorean02, 25 ) ;
   if ( strLen > ZERO )    w[strLen] = '\0' ;   // no null terminator was converted
   else                    wcscpy ( w, mbstowcsError ) ;   
   wcout << "substring ("<< strLen 
      << " chars) of Korean UTF-8 string converted to wchar_t: " << w << endl ;

   wcout << "\n  ** Please Press 'Enter' Key To Continue Test..." ;
   cin.get() ;

   #if 0    // Experimental Only
   // Turn ON UTF-8 locale settings - too late, but ON.
   // This causes the mbctowcs return error go away (see note above)
   loco.global ( loco ) ;
   #endif   // Experimental Only

   //* Display the UTF-8 data again, but this time through the 'wstring' class *
   system ( "clear" ) ;             // clear the screen
   wcout << wTitle << wSubHeadTb << endl ;// display application title and test description
   wcout << endl << wSubHead01b << endl ;

   strLen = mbstowcs ( w, cChinese01, 1024 ) ;
   wString = w ;
   wcout << (strLen > ZERO ? wString : mbstowcsError) << endl ;
   strLen = mbstowcs ( w, cChinese02, 1024 ) ;
   wString = w ;
   wcout << (strLen > ZERO ? wString : mbstowcsError) << endl ;

   strLen = mbstowcs ( w, cKorean01, 1024 ) ;
   wString = w ;
   wcout << (strLen > ZERO ? wString : mbstowcsError) << endl ;
   strLen = mbstowcs ( w, cKorean02, 1024 ) ;
   wString = w ;
   wcout << (strLen > ZERO ? wString : mbstowcsError) << endl ;

   strLen = mbstowcs ( w, cKanji01, 1024 ) ;
   wString = w ;
   wcout << (strLen > ZERO ? wString : mbstowcsError) << endl ;
   strLen = mbstowcs ( w, cKanji02, 1024 ) ;
   wString = w ;
   wcout << (strLen > ZERO ? wString : mbstowcsError) << endl ;

   strLen = mbstowcs ( w, cHiragana01, 1024 ) ;
   wString = w ;
   wcout << (strLen > ZERO ? wString : mbstowcsError) << endl ;
   strLen = mbstowcs ( w, cHiragana02, 1024 ) ;
   wString = w ;
   wcout << (strLen > ZERO ? wString : mbstowcsError) << endl ;

   strLen = mbstowcs ( w, cRussian, 1024 ) ;
   wString = w ;
   wcout << (strLen > ZERO ? wString : mbstowcsError) << endl ;
   strLen = mbstowcs ( w, cPolish, 1024 ) ;
   wString = w ;
   wcout << (strLen > ZERO ? wString : mbstowcsError) << endl ;

   strLen = mbstowcs ( w, cArabic, 1024 ) ;
   if ( strLen > ZERO )
   {
      wString = w ;
      wcout << rtlStart << wString << rtlEnd ;
      strLen = mbstowcs ( w, cArabicTrans, 1024 ) ;
      wString = w ;
      wcout << (strLen > ZERO ? wString : mbstowcsError)
            << L" (RTL codes not recognized)" ;
   }
   else
      wcout << mbstowcsError ;
   wcout << endl ;

   strLen = mbstowcs ( w, cHebrew, 1024 ) ;
   if ( strLen > ZERO )
   {
      wString = w ;
      wcout << rtlStart << wString << rtlEnd ;
      strLen = mbstowcs ( w, cHebrewTrans, 1024 ) ;
      wString = w ;
      wcout << (strLen > ZERO ? wString : mbstowcsError)
            << L" (RTL codes not recognized)" ;
   }
   else
      wcout << mbstowcsError ;
   wcout << endl ;

   //* Display the UTF-8 data again, but this time through the 'ustring' class *
   wcout << wSubHeadTc << endl ;    // display test description
   wcout << wSubHead01c << endl ;
#if GLIB_USTRING_ENABLE != 0     // Include Glib::ustring class testing
   Glib::ustring uString ;
   uString = cChinese01 ;     wcout << uString << endl ;
   uString = cChinese02 ;     wcout << uString << endl ;
   uString = cKorean01 ;      wcout << uString << endl ;
   uString = cKorean02 ;      wcout << uString << endl ;
   uString = cKanji01 ;       wcout << uString << endl ;
   uString = cKanji02 ;       wcout << uString << endl ;
   uString = cHiragana01 ;    wcout << uString << endl ;
   uString = cHiragana02 ;    wcout << uString << endl ;
   uString = cRussian ;       wcout << uString << endl ;
   uString = cPolish ;        wcout << uString << endl 
   uString = cArabic ;        wcout << uString ;
   uString = cArabicTrans ;   wcout << uString << endl ;
   uString = cHebrew ;        wcout << uString ;
   uString = cHebrewTrans ;   wcout << uString << endl << endl ;
#else    // NON GLIB METHOD
   wcout << L"  [Dialogw was compiled without the libraries "
             "needed to support Glib::ustring]\n" << endl ;
#endif   // GLIB_USTRING_ENABLE

   wcout << endl ;

}  //* End Test_wcout() *

//*************************
//*     Help_wcout        *
//*************************
//******************************************************************************
//* Display command-line options for the iostream test.                        *
//*                                                                            *
//*                                                                            *
//* Input  :                                                                   *
//*                                                                            *
//* Returns:                                                                   *
//******************************************************************************

void Help_wcout ( void )
{
string   helpText = 
   "\n Invoking the iostream test without starting the NCurses engine:"
   "\n  Dialogw -w | -W([ n | l [locale_name] | h])"
   "\n            Invoke the 'wide' iostream (std::wcout) output test."
   "\n"
   "\n        n   use the default, application startup locale, usually \"C\", which will"
   "\n            probably cause the supra-ASCII characters to display incorrectly."
   "\n        l   (default) set locale before performing any i/o operations"
   "\n        'locale_name' (optional)"
   "\n            if specified, it indicates an alternate locale to be used"
   "\n            instead of the locale specified by the terminal's environment."
   "\n            For a complete list of locales available on your system,"
   "\n            run the console command, 'locale -a'"
   "\n        h (or invalid sub-option), this message"
   "\n IMPORTANT NOTE: To function properly, this test requires a locale"
   "\n                 that supports UTF-8 character encoding." ;

   system ( "clear" ) ;             // clear the screen
   cout << helpText << endl << endl;// help the confused user
   
}  //* End Help_wcout() *

//*************************
//*  GetCommandLineArgs   *
//*************************
//******************************************************************************
//* Capture user's command-line arguments.                                     *
//*                                                                            *
//* Valid Arguments:                                                           *
//*   Test number: -[1 | 2 | 3 | 4 | 5 | 6 | 7 | 8 | 9 | 10]                   *
//*   Alt Locale : -A, -a     Specify an alternate 'locale' for interpretation *
//*                           of character output and input conversions.       *
//*   Help       : -H, -h     Start application with display of Help Dialog.   *
//*                                                                            *
//*                                                                            *
//* Input  : commArgs class object (by reference)                              *
//*                                                                            *
//* Returns: test number to be executed OR -1 if no test specified             *
//******************************************************************************
//* NOTE: Command-line argument [-w | -w] is captured at the top of main(),    *
//*       and we don't see it here.                                            *
//******************************************************************************

short GetCommandLineArgs ( commArgs& ca )
{
   if ( ca.argCount > 1 )
   {
      short v = 1,         // argv index
            i ;            // current arg index

      while ( v < ca.argCount )
      {
         i = ZERO ;
         if ( ca.argList[v][i] == '-' )  ++i ;     // step over dashes
         if ( ca.argList[v][i] == '-' )  ++i ;

         //* Test number specified? *
         if ( ca.argList[v][i] >= '1' && ca.argList[v][i] <= '9' )
         {
            sscanf ( &ca.argList[v][i], "%hd", &ca.testNum ) ;
            --ca.testNum ; // make it a zero-based index
         }

         //* Alternate locale specified? *
         else if ( ca.argList[v][i] == 'A' || ca.argList[v][i] == 'a' )
         {  //* If user specified an alternate locale string, use it instead *
            //* of the default locale found in the terminal environment.     *
            ++i ;
            if ( ca.argList[v][i] != NULLCHAR )
            {
               if ( ca.argList[v][i] == '=' )
                  ++i ;
               strncpy ( ca.localeName, &ca.argList[v][i], 30 ) ;
               ca.altLocale = true ;
            }
            else if ( ca.argCount > 2 )
            {
               strncpy ( ca.localeName, ca.argList[v+1], 30 ) ;
               ca.altLocale = true ;
            }
         }

         //* Else, request for help OR invalid command-line argument *
         else
         {
            ca.testNum = indexHELP ;
            break ;
         }

         ++v ;          // next command-line argument
      }
   }
   return ca.testNum ;
   
}  //* End GetCommandLineArgs() *

//*************************
//*     HelpAbout         *
//*************************
//******************************************************************************
//* Display the application's Help-About window.                               *
//* This is a simple informational dialog with a 'CLOSE' button.               *
//*                                                                            *
//*                                                                            *
//* Input  : ctrY : position at which to center dialog in Y                    *
//*          ctrX : position at which to center dialog in X                    *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void HelpAbout ( short ctrY, short ctrX )
{
const gString DialogText(
   "Dialogw\n"
   "Copyright (c) %s Mahlon R. Smith, The Software Samurai\n"
   "                        Beijing University of Technology\n"
   " \n"
   "A simple utility for testing 'wide' and UTF-8-encoded input/output.\n"
   " \n"
   "Command Line arguments (optional): ./Dialogw (-[n | w | a | h])\n"
   "    -n           where 'n' is a test number\n"
   "    -a | -A      specify an alternate 'locale' setting to override\n"
   "                 locale specified in terminal environment\n"
   "                 (for complete list, run console cmd:  locale -a)\n"
   "                  Ex: ./Dialog -Avi_VN.utf8 ./Dialogw -A yi_US.utf8\n"
   "    -h, -H       Help (this dialog)\n"
   "    -W[n | l | h] test of std:wcout (NCurses engine not initialized)\n"
   "       n == run test without initializing 'locale' Ex: ./Dialogw -Wn\n"
   "       l == run test with alternate locale setting\n"
   "            Ex: ./Dialogw -Wl locale_name          h == context help\n"
   "Tools: GNU G++ v:13 (C++11) under Fedora and Ubuntu Linux.\n"
   "       Requires ncursesw library v:6.4+ ", crYears
);//xxxxxxxxx0xxxxxxxxx0xxxxxxxxx0xxxxxxxxx0xxxxxxxxx0xxxxxxxxx0xxxxxxxx|
static const short dialogROWS = 23 ;      // # of dialog lines
static const short dialogCOLS = 70 ;      // # of dialog columns
static const short controlsDEFINED = 3 ;  // # of controls in dialog
enum ctrls { closePUSH = ZERO, classPUSH, appverTBOX } ;
short ulY = ctrY - dialogROWS / 2,
      ulX = ctrX - dialogCOLS / 2 ;
attr_t dColor = nc.blR ;

      
   //* Format application version string for display *
   gString verString( " %s ", AppVersion ) ;


InitCtrl ic[controlsDEFINED] =      // array of dialog control info
{
   {  //* 'CLOSE' pushbutton  - - - - - - - - - - - - -  closePUSH *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(dialogROWS - 2),        // ulY:       upper left corner in Y
      short(dialogCOLS / 2 - 4),    // ulX:       upper left corner in X
      1,                            // lines:     control lines
      7,                            // cols:      control columns
      " CLOSE ",                    // dispText:  
      nc.gyR,                       // nColor:    non-focus color
      nc.gyre | ncbATTR,            // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "",                           // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[1],                       // nextCtrl:  link in next structure
   },
   {  //* 'CLASS VERSION' pushbutton  - - - - - - - - -  classPUSH *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(dialogROWS - 2),        // ulY:       upper left corner in Y
      short(dialogCOLS - 25),       // ulX:       upper left corner in X
      1,                            // lines:     control lines
      24,                           // cols:      control columns
      " DISPLAY CLASS VERSIONS ",   // dispText:  
      nc.gyR,                       // nColor:    non-focus color
      nc.gyre | ncbATTR,            // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "",                           // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[2],                       // nextCtrl:  link in next structure
   },
   {  //* Non-active textbox for app. version  - - - -  appverTBOX *
      dctTEXTBOX,                   // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      1,                            // ulY:       upper left corner in Y
      19,                           // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      short(verString.gscols()),    // cols:      control columns
      verString.ustr(),             // dispText:  (value of t3indexO3)
      nc.bl,                        // nColor:    non-focus color
      nc.bl,                        // fColor:    focus color
      tbPrint,                      // filter:    any printing character
      "Version:",                   // label:     
      ZERO,                         // labY:      
      -9,                           // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      false,                        // active:    cannot gain focus
      NULL,                         // nextCtrl:  link in next structure
   },
} ;

   //* Initial parameters for dialog window *
   InitNcDialog dInit( dialogROWS,     // number of display lines
                       dialogCOLS,     // number of display columns
                       ulY,            // Y offset from upper-left of terminal 
                       ulX,            // X offset from upper-left of terminal 
                       NULL,           // no dialog title
                       ncltSINGLE,     // border line-style
                       dColor,         // border color attribute
                       dColor,         // interior color attribute
                       ic              // pointer to list of control definitions
                     ) ;

   //* Instantiate the dialog window *
   NcDialog* dp = new NcDialog ( dInit ) ;

   //* Open the dialog window *
   if ( (dp->OpenWindow()) == OK )
   {
      dp->WriteParagraph ( 1, 1, DialogText, dColor ) ;
      dp->RefreshWin () ;

      uiInfo Info ;                 // user interface data returned here
      bool   done = false ;         // loop control
      while ( ! done )
      {
         dp->EditPushbutton ( Info ) ;
         if ( Info.dataMod != false )
         {
            if ( Info.ctrlIndex == closePUSH )
               done = true ;
            else if ( Info.ctrlIndex == classPUSH )
            {  //* Open a dialog to display class version numbers and other    *
               //* interesting data. Current dialog will be obscured, so save  *
               //* its display data.                                           *
               dp->SetDialogObscured () ;
               DisplayClassInfo ( (ulY + 4), (ctrX - 20), 15, 40 ) ;
               dp->NextControl () ;
               dp->RefreshWin () ;
            }
         }
         if ( done == false )
         {
            if ( Info.keyIn == nckSTAB )
               dp->PrevControl () ; 
            else if ( Info.keyIn != ZERO )
               dp->NextControl () ;
         }
      }
   }
   if ( dp != NULL )
      delete ( dp ) ;                        // close the window

}  //* End HelpAbout() *

//***********************
//* ClearTerminalWindow *
//***********************
//****************************************************************************
//* Clear the terminal window and display the application title.             *
//*                                                                          *
//*                                                                          *
//* Input Values : none                                                      *
//*                                                                          *
//* Return Value : none                                                      *
//****************************************************************************

void ClearTerminalWindow ( void )
{
   winPos   wpos( 0, 0 ) ;
   nc.ClearScreen () ;
   gString gsTitle( "  Dialogw - v:%s (c) %s Mahlon R. Smith    : Locale: %s      ",
                    AppVersion, crYears, nc.GetLocale() ) ;
   nc.WriteString ( wpos, gsTitle.gstr(), nc.bw | ncuATTR ) ;
   nc.RefreshScreen () ;

}  //* End ClearTerminalWindow() *

//*************************
//*                       *
//*************************
//******************************************************************************
//*                                                                            *
//*                                                                            *
//*                                                                            *
//* Input  :                                                                   *
//*                                                                            *
//* Returns:                                                                   *
//******************************************************************************


