//******************************************************************************
//* File       : SliderTest.cpp                                                *
//* Author     : Mahlon R. Smith                                               *
//*              Copyright (c) 2021-2025 Mahlon R. Smith, The Software Samurai *
//*                  GNU GPL copyright notice located in NcDialog.hpp          *
//* Date       : 21-Mar-2025                                                   *
//* Version    : (see below)                                                   *
//*                                                                            *
//* Description: Class definition for the SlideTest class.                     *
//*              This class provides a dialog window in which all functionality*
//*              of the dctSLIDER class can be tested and demonstrated.        *
//*              The dctSLIDER class defines an NcDialog user-interface        *
//*              control object.                                               *
//*              This test is called as a menu item within the Dialog4         *
//*              application, Test10.                                          *
//*                                                                            *
//* Development Tools: See NcDialog.cpp.                                       *
//******************************************************************************
//* Version History (most recent first):                                       *
//* v: 0.00.01 19-Aug-2021                                                     *
//*   - First draft of test code for the new (August 2021) dctSLIDER control.  *
//*     Based on simplified test code developed in Dialog1.cpp, Test #8.       *
//*                                                                            *
//*                                                                            *
//******************************************************************************

//****************
//* Header Files *
//****************
#include "SliderTest.hpp"

//***************************
//* File-access definitions *
//***************************
const short MAX_FNAME = 255 ;          //* Size of buffer to hold filename (POSIX)

//* Local definitions for file type encoded in the "st_mode" *
//* element of the stat64{} structure                        *
enum fmFType : short
{ fmDIR_TYPE,   fmREG_TYPE,  fmLINK_TYPE, fmCHDEV_TYPE, 
  fmBKDEV_TYPE, fmFIFO_TYPE, fmSOCK_TYPE, fmUNKNOWN_TYPE, fmTYPES } ;

//* The following c-type structures defined in the standard header files 
//* need to be C++ data types: stat{} dirent{}, etc.
typedef struct stat64 FileStats ;
//typedef struct dirent64 deStats ;
typedef struct tm Tm ;

//* Class for reporting the system local date/time.                            *
//* ---------------------------------------------------------------------------*
#define DECODE_TZ (0)   // if non-zero, decode time-zone fields of localTime
const int64_t maxETIME32 = 0x07FFFFFFF,      // largest 32-bit time_t
              minETIME32 = -maxETIME32 - 1 ; // smallest 32-bit time_t
const short ltFMT_LEN = 16 ;                 // length of time string buffers
const short ltISO_LEN = 24 ;
//* Important note about the year 2038 bug:                                    *
//* Because 'time_t' is defined as a signed, 32-bit value on 32-bit systems,   *
//* time reporting will break down for date/time values                        *
//*                     >= 19 Jan 2038 at 11:14:08                             *
//* For this reason and for others, we have defined our own local-time class.  *
//* This class uses a signed, 64-bit value for storing the epoch time.         *
//* It is hoped that all Linux systems will have been updated long before this *
//* becomes an issue; however, even now (2013), some applications are using    *
//* date/time manipulations that exceed the 32-bit limit. BEWARE!!             *
//* NOTE: time arithmetic should always be performed on signed 64-bit values.  *
//*                                                                            *
class localTime
{
   public:
   localTime ( void ) { this->reset() ; }
   void reset ( void )
   {  //* Set to epoch date: Thursday, 01 Jan. 1970 @ 08:00:00
      this->sysepoch = ZERO ;
      this->epoch = ZERO ;
      this->day = 4 ;
      this->year = 1970 ;
      this->month = this->date = 1 ;
      this->hours = 8 ;
      this->minutes = this->seconds = ZERO ;
      this->julian = ZERO ;
      *this->iso = NULLCHAR ;
      #if DECODE_TZ != 0   // time-zone data
      *this->timezone = NULLCHAR ;
      *this->utc_zone  = NULLCHAR ;
      this->gmtoffset = ZERO ;
      this->dst       = false ;
      #endif   // DECODE_TZ
   }
   int64_t  epoch ;        //* Seconds since the epoch (01-Jan-1970)
   time_t   sysepoch ;     //* system epoch time (see note)
   uint16_t day ;          //* Day of the week (0 == Sun, ..., 6 == Sat)
   uint16_t date ;         //* Day of the month (1-[28|29|30|31])
   uint16_t month ;        //* Month of the year (1 == Jan, ... 12 == Dec)
   uint16_t year ;         //* Year (four digit year)
   uint16_t hours ;        //* Time of day, hour (0-23)
   uint16_t minutes ;      //* Time of day, minutes (0-59)
   uint16_t seconds ;      //* Time of day, seconds (0-59)
   uint16_t julian ;       //* Completed days since beginning of year
   char iso[ltISO_LEN] ;   //* String representation: ISO 8601-1:2019 "yyyy-mm-ddThh:mm:ss"

   #if DECODE_TZ != 0   // time-zone data
   // See DecodeEpochTime method for more information.
   char     timezone[ltFMT_LEN] ; //* string representing the local time zone (useless)
   char     utc_zone[ltFMT_LEN] ; //* string representing the UTC offset for local time zone
   short    gmtoffset ;    //* Expressed as (signed) number of seconds offset from UTC+0:00
   bool     dst ;          //* 'true' if daylight savings time in effect,
                           //* 'false if not DST or info not available
   #endif   // DECODE_TZ
} ;

//* Class definition for human-readable file stats.                            *
class tnFName
{
   public:
   ~tnFName ( void ) {}       //* Destructor
   tnFName ( void )           //* Constructor
   { this->ReInit () ; }
   void ReInit(void)
   {
      fName[0] = NULLCHAR ;
      fType    = fmTYPES ;
      fBytes   = ZERO ;
      modTime.day = modTime.date = modTime.month = modTime.year 
                  = modTime.hours = modTime.minutes = modTime.seconds = ZERO ;
      readAcc  = false ;
      writeAcc = false ;
      // NOTE: data member, rawStats is left unitialized
   }
   char        fName[MAX_FNAME] ; //* Filename string
   fmFType     fType ;     //* File type (enum fmFType)
   UINT64      fBytes ;    //* File size (in bytes)
   localTime   modTime ;   //* Date/Time file last modified (human readable)
   FileStats   rawStats ;  //* Copy of system's "stat" structure (all file info)
   bool        readAcc ;   //* 'true' if user has read access to file
   bool        writeAcc ;  //* 'true' if user has write access to file 
} ;

//**********************
//** Local Prototypes **
//**********************
static bool TargetExists ( const gString& trgPath, fmFType& fType ) ;
static bool GetFileStats ( const gString& trgPath, tnFName& fStats ) ;

//****************
//** Local Data **
//****************
static const short 
   dROWS      = 34,        // dialog rows
   dCOLS      = 122 ;      // dialog columns

//* Used by ControlUpdate() method *
//* method to access member data.  *
static SliderTest *stPtr ;

//*************************************
//** Test-dialog control information **
//*************************************
//* Slider initialization structures:                                         *
//* a) Odd numbered structures grow in the "standard" direction (up/right).   *
//* b) Even numbered structures grow in the "reverse" direction (down/left).  *
//* c) sl1Data, sl2Data, sl9Data and sl10Data have invalid parameters. This   *
//*    tests the internal error correction and default configuration values.  *                  *
//* d) sl3Data and sl4Data specify a range of positive values.                *
//* e) sl5Data and sl6Data specify a range of negative values.                *
//* f) sl7Data and sl8Data specify a range from negative to positive values.  *
//* g) sl11Data and sl12Data specify a range of positive values.              *
//* h) sl13Data and sl14Data specify a range from negative to positive values.*
static sliderData sl1Data( 0.0, 0.0, 0.0, true, false ) ;
static sliderData sl2Data( 0.0, 0.0, 0.0, true, true ) ;
static sliderData sl3Data( 0.0, 100.0, 0.0, true, false ) ;
static sliderData sl4Data( 0.0, 100.0, 0.0, true, true ) ;
static sliderData sl5Data( -150.0, 0.0, -150.0, true, false ) ;
static sliderData sl6Data( -150.0, 0.0, -150.0, true, true ) ;
static sliderData sl7Data( -500.0, 500.0, -500.0, true, false ) ;
static sliderData sl8Data( -500.0, 500.0, -500.0, true, true ) ;

static sliderData sl9Data( 0.0, 0.0, 0.0, false, false ) ;
static sliderData sl10Data( 0.0, 0.0, 0.0, false, true ) ;
static sliderData sl11Data( 0.0, 100.0, 0.0, false, false ) ;
static sliderData sl12Data( 0.0, 100.0, 0.0, false, true ) ;

enum ctControls : short    // Uniquely name each dialog control
{
   stClosePB = ZERO,       // 'Close Dialog' Pushbutton
   stVstd1SL,              // Slider #1,  vertical,   one column,    grows upward
   stVrev1SL,              // Slider #2,  vertical,   one column,    grows downward
   stVstd2SL,              // Slider #3,  vertical,   two columns,   grows upward
   stVrev2SL,              // Slider #4,  vertical,   two columns,   grows downward
   stVstd3SL,              // Slider #5,  vertical,   three columns, grows upward
   stVrev3SL,              // Slider #6,  vertical,   three columns, grows downward
   stVstd7SL,              // Slider #7,  vertical,   seven columns, grows upward
   stVrev7SL,              // Slider #8,  vertical,   seven columns, grows downward
   stHstd1SL,              // Slider #9,  horizontal. one row,       grows rightward
   stHrev1SL,              // Slider #10, horizontal. one row,       grows leftward
   stHstd2SL,              // Slider #11, horizontal. two rows,      grows rightward
   stHrev2SL,              // Slider #12, horizontal. two rows,      grows leftward
   stLinkRB,               // Couple or de-couple sliders #1 and #2
   stHelpPB,               // 'Info Help' Pushbutton
   stCONTROLS              // number of controls defined
} ;

//**********************************************************
//* Initialization of dialog control objects.              *
//* Color attributes initialized by SliderTest constructor.*
//**********************************************************
InitCtrl SliderTestIC[stCONTROLS] =
{
   {  //* 'CLOSE' pushbutton  - - - - - - - - - - - - - - - - - - -  stClosePB *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      2,                            // ulY:       upper left corner in Y
      2,                            // ulX:       upper left corner in X
      1,                            // lines:     control lines
      9,                            // cols:      control columns
      "  ^CLOSE  ",                 // dispText:  
      nc.bw,                        // nColor:    non-focus color
      nc.bw,                        // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "",                           // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &SliderTestIC[stVstd1SL]      // nextCtrl:  link in next structure
   },
   { //* Slider #1   - - - - - - - - - - - - - - - - - - - - - - -   stVstd1SL *
      dctSLIDER,                    // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(SliderTestIC[stClosePB].ulY + 4), // ulY: upper left corner in Y
      SliderTestIC[stClosePB].ulX,  // ulX:       upper left corner in X
      12,                           // lines:     control rows
      1,                            // cols:      control columns
      NULL,                         // dispText:  (none)
      nc.bw,                        // nColor:    non-focus color
      nc.bw,                        // fColor:    focus color
      tbPrint,                      // filter:    any printing character
      "^Slider 1",                  // label:     label text
      -2,                           // labY:      label offset
      ZERO,                         // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &SliderTestIC[stVrev1SL]      // nextCtrl:  link in next structure
   },
   { //* Slider #2   - - - - - - - - - - - - - - - - - - - - - - -   stVrev1SL *
      dctSLIDER,                    // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      SliderTestIC[stVstd1SL].ulY,  // ulY: upper left corner in Y
      short(SliderTestIC[stVstd1SL].ulX +   // ulX: upper left corner in X
            SliderTestIC[stVstd1SL].cols + 11),
      12,                           // lines:     control rows
      1,                            // cols:      control columns
      NULL,                         // dispText:  (none)
      nc.bw,                        // nColor:    non-focus color
      nc.bw,                        // fColor:    focus color
      tbPrint,                      // filter:    any printing character
      "Slider 2",                   // label:     label text
      -2,                           // labY:      label offset
      ZERO,                         // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &SliderTestIC[stVstd2SL]      // nextCtrl:  link in next structure
   },
   { //* Slider #3   - - - - - - - - - - - - - - - - - - - - - - -   stVstd2SL *
      dctSLIDER,                    // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      SliderTestIC[stVrev1SL].ulY,  // ulY: upper left corner in Y
      short(SliderTestIC[stVrev1SL].ulX +   // ulX: upper left corner in X
            SliderTestIC[stVrev1SL].cols + 11),
      12,                           // lines:     control rows
      2,                            // cols:      control columns
      NULL,                         // dispText:  (none)
      nc.bw,                        // nColor:    non-focus color
      nc.bw,                        // fColor:    focus color
      tbPrint,                      // filter:    any printing character
      "Slider 3",                   // label:     label text
      -2,                           // labY:      label offset
      ZERO,                         // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &SliderTestIC[stVrev2SL]      // nextCtrl:  link in next structure
   },
   { //* Slider #4   - - - - - - - - - - - - - - - - - - - - - - -   stVrev2SL *
      dctSLIDER,                    // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      SliderTestIC[stVstd2SL].ulY,  // ulY: upper left corner in Y
      short(SliderTestIC[stVstd2SL].ulX +   // ulX: upper left corner in X
            SliderTestIC[stVstd2SL].cols + 10),
      12,                           // lines:     control rows
      2,                            // cols:      control columns
      NULL,                         // dispText:  (none)
      nc.bw,                        // nColor:    non-focus color
      nc.bw,                        // fColor:    focus color
      tbPrint,                      // filter:    any printing character
      "Slider 4",                   // label:     label text
      -2,                           // labY:      label offset
      ZERO,                         // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &SliderTestIC[stVstd3SL]      // nextCtrl:  link in next structure
   },
   { //* Slider #5   - - - - - - - - - - - - - - - - - - - - - - -   stVstd3SL *
      dctSLIDER,                    // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      SliderTestIC[stVrev2SL].ulY,  // ulY: upper left corner in Y
      short(SliderTestIC[stVrev2SL].ulX +   // ulX: upper left corner in X
            SliderTestIC[stVrev2SL].cols + 10),
      12,                           // lines:     control rows
      3,                            // cols:      control columns
      NULL,                         // dispText:  (none)
      nc.bw,                        // nColor:    non-focus color
      nc.bw,                        // fColor:    focus color
      tbPrint,                      // filter:    any printing character
      "Slider 5",                   // label:     label text
      -2,                           // labY:      label offset
      ZERO,                         // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &SliderTestIC[stVrev3SL]      // nextCtrl:  link in next structure
   },
   { //* Slider #6   - - - - - - - - - - - - - - - - - - - - - - -   stVrev3SL *
      dctSLIDER,                    // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      SliderTestIC[stVstd3SL].ulY,  // ulY: upper left corner in Y
      short(SliderTestIC[stVstd3SL].ulX +   // ulX: upper left corner in X
            SliderTestIC[stVstd3SL].cols + 9),
      12,                           // lines:     control rows
      3,                            // cols:      control columns
      NULL,                         // dispText:  (none)
      nc.bw,                        // nColor:    non-focus color
      nc.bw,                        // fColor:    focus color
      tbPrint,                      // filter:    any printing character
      "Slider 6",                   // label:     label text
      -2,                           // labY:      label offset
      ZERO,                         // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &SliderTestIC[stVstd7SL]      // nextCtrl:  link in next structure
   },
   { //* Slider #7   - - - - - - - - - - - - - - - - - - - - - - -   stVstd7SL *
      dctSLIDER,                    // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      SliderTestIC[stVrev3SL].ulY,  // ulY: upper left corner in Y
      short(SliderTestIC[stVrev3SL].ulX +   // ulX: upper left corner in X
            SliderTestIC[stVrev3SL].cols + 9),
      12,                           // lines:     control rows
      7,                            // cols:      control columns
      NULL,                         // dispText:  (none)
      nc.bw,                        // nColor:    non-focus color
      nc.bw,                        // fColor:    focus color
      tbPrint,                      // filter:    any printing character
      "Slider 7",                   // label:     label text
      -2,                           // labY:      label offset
      ZERO,                         // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      (const dspinData*)&sl7Data,   // spinData:  (pointer cast from sliderData)
      true,                         // active:    allow control to gain focus
      &SliderTestIC[stVrev7SL]      // nextCtrl:  link in next structure
   },
   { //* Slider #8   - - - - - - - - - - - - - - - - - - - - - - -   stVrev7SL *
      dctSLIDER,                    // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      SliderTestIC[stVstd7SL].ulY,  // ulY: upper left corner in Y
      short(SliderTestIC[stVstd7SL].ulX +   // ulX: upper left corner in X
            SliderTestIC[stVstd7SL].cols + 5),
      12,                           // lines:     control rows
      7,                            // cols:      control columns
      NULL,                         // dispText:  (none)
      nc.bw,                        // nColor:    non-focus color
      nc.bw,                        // fColor:    focus color
      tbPrint,                      // filter:    any printing character
      "Slider 8",                   // label:     label text
      -2,                           // labY:      label offset
      ZERO,                         // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &SliderTestIC[stHstd1SL]      // nextCtrl:  link in next structure
   },
   { //* Slider #9   - - - - - - - - - - - - - - - - - - - - - - -   stHstd1SL *
      dctSLIDER,                    // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(SliderTestIC[stVrev7SL].ulY - 1),// ulY: upper left corner in Y
      short(SliderTestIC[stVrev7SL].ulX +    // ulX: upper left corner in X
            SliderTestIC[stVrev7SL].cols + 6),
      1,                            // lines:     control rows
      21,                           // cols:      control columns
      NULL,                         // dispText:  (none)
      nc.bw,                        // nColor:    non-focus color
      nc.bw,                        // fColor:    focus color
      tbPrint,                      // filter:    any printing character
      "Slide^r 9",                  // label:     label text
      -1,                           // labY:      label offset
      ZERO,                         // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &SliderTestIC[stHrev1SL]      // nextCtrl:  link in next structure
   },
   { //* Slider #10  - - - - - - - - - - - - - - - - - - - - - - -   stHrev1SL *
      dctSLIDER,                    // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(SliderTestIC[stHstd1SL].ulY + // ulY: upper left corner in Y
            SliderTestIC[stHstd1SL].lines + 6),
      SliderTestIC[stHstd1SL].ulX,    // ulX: upper left corner in X
      1,                            // lines:     control rows
      21,                           // cols:      control columns
      NULL,                         // dispText:  (none)
      nc.bw,                        // nColor:    non-focus color
      nc.bw,                        // fColor:    focus color
      tbPrint,                      // filter:    any printing character
      "Slider 10",                  // label:     label text
      -1,                           // labY:      label offset
      ZERO,                         // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &SliderTestIC[stHstd2SL]      // nextCtrl:  link in next structure
   },
   { //* Slider #11  - - - - - - - - - - - - - - - - - - - - - - -   stHstd2SL *
      dctSLIDER,                    // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(SliderTestIC[stHrev1SL].ulY + // ulY: upper left corner in Y
            SliderTestIC[stHrev1SL].lines + 6),
      SliderTestIC[stHrev1SL].ulX,  // ulX: upper left corner in X
      2,                            // lines:     control rows
      21,                           // cols:      control columns
      NULL,                         // dispText:  (none)
      nc.bw,                        // nColor:    non-focus color
      nc.bw,                        // fColor:    focus color
      tbPrint,                      // filter:    any printing character
      "Slider 11",                  // label:     label text
      -1,                           // labY:      label offset
      ZERO,                         // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &SliderTestIC[stHrev2SL]      // nextCtrl:  link in next structure
   },
   { //* Slider #12  - - - - - - - - - - - - - - - - - - - - - - -   stHrev2SL *
      dctSLIDER,                    // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(SliderTestIC[stHstd2SL].ulY + // ulY: upper left corner in Y
            SliderTestIC[stHstd2SL].lines + 6),
      SliderTestIC[stHstd2SL].ulX,    // ulX: upper left corner in X
      2,                            // lines:     control rows
      21,                           // cols:      control columns
      NULL,                         // dispText:  (none)
      nc.bw,                        // nColor:    non-focus color
      nc.bw,                        // fColor:    focus color
      tbPrint,                      // filter:    any printing character
      "Slider 12",                  // label:     label text
      -1,                           // labY:      label offset
      ZERO,                         // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      (const dspinData*)&sl12Data,  // spinData:  (pointer cast from sliderData)
      true,                         // active:    allow control to gain focus
      &SliderTestIC[stLinkRB]       // nextCtrl:  link in next structure
   },
   {  //* "Link" Radio Button  - - - - - - - - - - - - - - - - - - -  stLinkRB *
      dctRADIOBUTTON,               // type:      
      rbtS5s,                       // rbSubtype: standard, 3 chars wide
      false,                        // rbSelect:  default selection
      SliderTestIC[stClosePB].ulY,  // ulY:       upper left corner in Y
      short(SliderTestIC[stClosePB].ulX +  // ulX:  upper left corner in X
            SliderTestIC[stClosePB].cols + 4),
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a)
      nc.bw,                        // nColor:    non-focus color
      nc.bw,                        // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "^Link S1/S2",                // label:
      -1,                           // labY:      
      -2,                           // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      -1,                           // scrItems:  (n/a)
      ZERO,                         // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &SliderTestIC[stHelpPB]       // nextCtrl:  link in next structure
   },
   {  //* 'HELP' pushbutton  - - - - - - - - - - - - - - - - - - - -  stHelpPB *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      SliderTestIC[stLinkRB].ulY,   // ulY:       upper left corner in Y
      short(SliderTestIC[stLinkRB].ulX + 9), // ulX: upper left corner in X
      1,                            // lines:     control lines
      8,                            // cols:      control columns
      "  ^HELP  ",                  // dispText:  
      nc.bw,                        // nColor:    non-focus color
      nc.bw,                        // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "",                           // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      NULL                          // nextCtrl:  link in next structure
   },
} ;


//*************************
//*      SliderTest       *
//*************************
//******************************************************************************
//* Constructor: Exercise the options of the Slider class.                     *
//*                                                                            *
//* Input  : wpOrig : (by reference) origin (upper left corner) for test dialog*
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

SliderTest::SliderTest ( const winPos& wpOrig )
{
   //* Initialize our data members and control definitions *
   this->Setup ( wpOrig ) ;

   //* Define the dialog object *
   InitNcDialog dInit( dROWS,          // number of display lines
                       dCOLS,          // number of display columns
                       this->wpOrig.ypos, // Y offset from upper-left of terminal 
                       this->wpOrig.xpos, // X offset from upper-left of terminal 
                       NULL,           // dialog title
                       ncltDUAL,       // border line-style
                       this->bColor,   // border color attribute
                       this->dColor,   // interior color attribute
                       SliderTestIC    // pointer to list of control definitions
                     ) ;

   //* Instantiate the dialog window *
   this->dp = new NcDialog ( dInit ) ;

   //* Open the dialog window *
   if ( (this->dlgOpen = bool(this->dp->OpenWindow()) == OK) )
   {
      //* Set dialog title *
      this->dp->SetDialogTitle ( " Test Slider Controls  ", this->bColor ) ;

      //* Enable Mouse Support *
      if ( (this->dp->meEnableStableMouse ()) != OK )
         ;  // warn user that mouse support was not enabled

      //* Initialize the dctSLIDER objects *
      //* Note that Slider7 and Slider12 are initialized during instantiation *
      //* by a pointer cast in the InitCtrl structure for those controls.     *
      this->dp->SetSliderConfig ( stVstd1SL, sl1Data ) ;
      this->dp->SliderAlert ( stVstd1SL, true ) ;
      this->dp->SetSliderConfig ( stVrev1SL, sl2Data ) ;
      this->dp->SliderAlert ( stVrev1SL, true ) ;
      this->dp->SetSliderConfig ( stVstd2SL, sl3Data ) ;
      this->dp->SliderAlert ( stVstd2SL, true ) ;
      this->dp->SetSliderConfig ( stVrev2SL, sl4Data ) ;
      this->dp->SliderAlert ( stVrev2SL, true ) ;
      this->dp->SetSliderConfig ( stVstd3SL, sl5Data ) ;
      this->dp->SliderAlert ( stVstd3SL, true ) ;
      this->dp->SetSliderConfig ( stVrev3SL, sl6Data ) ;
      this->dp->SliderAlert ( stVrev3SL, true ) ;
      //this->dp->SetSliderConfig ( stVstd7SL, sl7Data ) ; // (initialized during instantiation)
      this->dp->SliderAlert ( stVstd7SL, true ) ;
      this->dp->SetSliderConfig ( stVrev7SL, sl8Data ) ;
      this->dp->SliderAlert ( stVrev7SL, true ) ;

      this->dp->SetSliderConfig ( stHstd1SL, sl9Data ) ;
      this->dp->SliderAlert ( stHstd1SL, true ) ;
      this->dp->SetSliderConfig ( stHrev1SL, sl10Data ) ;
      this->dp->SliderAlert ( stHrev1SL, true ) ;
      this->dp->SetSliderConfig ( stHstd2SL, sl11Data ) ;
      this->dp->SliderAlert ( stHstd2SL, true ) ;
      //this->dp->SetSliderConfig ( stHrev2SL, sl12Data ) ; // (initialized during instantiation)
      this->dp->SliderAlert ( stHrev2SL, true ) ;

      //* Test the SliderAutoReport() method for Sliders 7 and 8 *
      this->dp->SliderAutoReport ( stVstd7SL, true, 
                                   (SliderTestIC[stVstd7SL].ulY - 1),
                                   (SliderTestIC[stVstd7SL].ulX), 
                                   "%-6.1lf", nc.bkbl, false ) ;
      this->dp->SliderAutoReport ( stVrev7SL, true, 
                                   (SliderTestIC[stVrev7SL].ulY - 1),
                                   (SliderTestIC[stVrev7SL].ulX), 
                                   "%-6.1lf", nc.bkbl, true ) ;

      //* Write static text *
      this->dp->WriteParagraph ( 1, 40, 
         "Adjust controls using cursor keys: Up, Down, Right, Left, PageUp, PageDown,\n"
         " Home, End; or use the mouse scroll wheel while hovering over the control.",
         this->bColor ) ;

      this->dp->WriteParagraph ( (dROWS - 6), 2, 
         "LEGEND:\n"
         "vCur:current value, vPct:current value percentage, vMin/vMax:minimum and max values\n"
         "dCur:current divisions, dTot:total divisions, vert:vertical growth, rev:reversed growth.\n"
         "Link S1/S2 Control: Link Slider 1 and Slider 2 so that when the value in one changes,\n"
         "                    the value in the other changes also.",
         this->gColor ) ;

      this->dp->RefreshWin () ;     // make everything visible

      //* Establish a callback method which has access to member data *
      stPtr = this ;
      this->dp->EstablishCallback ( &this->ControlUpdate ) ;


      //****************************************
      //* Interact with user (filthy beasts :) *
      //****************************************
      uiInfo Info ;                 // user interface data returned here
      short  icIndex = ZERO ;       // index of control with input focus
      bool   done = false,          // loop control
             linkedSliders = false ;// track the state of the radiobutton
      while ( ! done )
      {
         if ( this->icptr[icIndex].type == dctPUSHBUTTON )
         {
            if ( Info.viaHotkey != false )
               Info.HotData2Primary () ;
            else
               icIndex = this->dp->EditPushbutton ( Info ) ;
            if ( Info.dataMod != false )
            {
               if ( Info.ctrlIndex == stClosePB) 
               {
                  done = true ;
               }
               else if ( Info.ctrlIndex == stHelpPB )
               {
                  this->Cry4Help () ;
               }
            }
         }

         if ( this->icptr[icIndex].type == dctRADIOBUTTON )
         {
            if ( Info.viaHotkey != false )
            {
               Info.HotData2Primary () ;
               // Programmer's Note: This sequence allows user to both select 
               // and deselect the Radiobutton via hotkey by locally tracking 
               // the Radiobutton's state.
               if ( linkedSliders )
               {
                  this->dp->PrevControl () ;    // remove focus from target control
                  this->dp->SetRadiobuttonState ( stLinkRB, false ) ;
                  this->dp->NextControl () ;    // restore focus
                  Info.dataMod = true ;
                  linkedSliders = Info.isSel = false ;
               }
               else
                  linkedSliders = true ;
            }
            else
            {
               icIndex = this->dp->EditRadiobutton ( Info ) ;
               if ( Info.dataMod )
                  linkedSliders = Info.isSel ;
            }
            //* If link has been disconnected, redraw *
            //* the stats for the affected controls.  *
            //* Call the callback method directly.    *
            if ( (Info.dataMod != false) && (Info.isSel == false) )
            {
               wkeyCode k ;   //* (note that keycode is not referenced)
               this->ControlUpdate ( stVstd1SL, k, false ) ;
               this->ControlUpdate ( stVrev1SL, k, false ) ;
            }
         }

         else if ( this->icptr[icIndex].type == dctSLIDER )
         {
            Info.viaHotkey = false ;   // discard any hotkey info
            icIndex = this->dp->EditSlider ( Info ) ;
         }

         //* Move input focus to next/previous control.*
         if ( done == false && Info.viaHotkey == false )
         {
            if ( Info.keyIn == nckSTAB )
               icIndex = this->dp->PrevControl () ; 
            else
               icIndex = this->dp->NextControl () ;
         }
      }  // while()
   }

   if ( this->dp != NULL )             // close the window
      delete ( this->dp ) ;

   stPtr = NULL ;                      // reset callback pointer

}  //* End SliderTest() *

//*************************
//*         Setup         *
//*************************
//******************************************************************************
//* Called by SliderTest() method to initialize the test data.                 *
//*                                                                            *
//* Input  : wpo : (by reference) position of dialog window                    *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void SliderTest::Setup ( const winPos& wpo )
{
   //* Color Attributes *
   this->dColor    = nc.blR ;                // dialog base color
   this->bColor    = nc.brbl ;               // dialog border color
   this->gColor    = nc.gybl ;               // dialog dim color
   this->pbnColor  = nc.gyR ;                // Pushbutton colors
   this->pbfColor  = nc.reG ;
   this->rbnColor  = nc.gyR ;                // Radiobutton colors
   this->rbfColor  = nc.reG ;
   this->slnColorA = nc.bk ;                 // Slider group A colors
   this->slfColorA = nc.re ;
   this->slnColorB = nc.bkcy ;               // Slider group B colors
   this->slfColorB = nc.brgr ;
   this->slnColorC = nc.bkcy ;               // Slider group C colors
   this->slfColorC = nc.brre ;
   this->slnColorD = nc.bkcy ;               // Slider group D colors
   this->slfColorD = nc.brma ;

   this->icptr    = SliderTestIC ;           // access to InitCtrl array
   this->wpOrig   = wpo ;                    // main dialog origin
   this->dlgOpen  = false ;                  // dialog is not yet open

   //* Initialize colors for control definitions *
   SliderTestIC[stClosePB].nColor  = this->pbnColor ;
   SliderTestIC[stClosePB].fColor  = this->pbfColor ;

   SliderTestIC[stVstd1SL].nColor  = this->slnColorA ;
   SliderTestIC[stVstd1SL].fColor  = this->slfColorA ;
   SliderTestIC[stVrev1SL].nColor  = this->slnColorA ;
   SliderTestIC[stVrev1SL].fColor  = this->slfColorA ;

   SliderTestIC[stVstd2SL].nColor  = this->slnColorB ;
   SliderTestIC[stVstd2SL].fColor  = this->slfColorB ;
   SliderTestIC[stVrev2SL].nColor  = this->slnColorB ;
   SliderTestIC[stVrev2SL].fColor  = this->slfColorB ;

   SliderTestIC[stVstd3SL].nColor  = this->slnColorC ;
   SliderTestIC[stVstd3SL].fColor  = this->slfColorC ;
   SliderTestIC[stVrev3SL].nColor  = this->slnColorC ;
   SliderTestIC[stVrev3SL].fColor  = this->slfColorC ;

   SliderTestIC[stVstd7SL].nColor  = this->slnColorD ;
   SliderTestIC[stVstd7SL].fColor  = this->slfColorD ;
   SliderTestIC[stVrev7SL].nColor  = this->slnColorD ;
   SliderTestIC[stVrev7SL].fColor  = this->slfColorD ;

   SliderTestIC[stHstd1SL].nColor  = this->slnColorB ;
   SliderTestIC[stHstd1SL].fColor  = this->slfColorB ;
   SliderTestIC[stHrev1SL].nColor  = this->slnColorB ;
   SliderTestIC[stHrev1SL].fColor  = this->slfColorB ;

   SliderTestIC[stHstd2SL].nColor  = this->slnColorD ;
   SliderTestIC[stHstd2SL].fColor  = this->slfColorD ;
   SliderTestIC[stHrev2SL].nColor  = this->slnColorD ;
   SliderTestIC[stHrev2SL].fColor  = this->slfColorD ;

   SliderTestIC[stLinkRB].nColor   = this->rbnColor ;
   SliderTestIC[stLinkRB].fColor   = this->rbfColor ;
   SliderTestIC[stHelpPB].nColor   = this->pbnColor ;
   SliderTestIC[stHelpPB].fColor   = this->pbfColor ;

   //* Get the filespec of the current working directory (CWD), *
   //* Initialize the 'helpPath' and 'helpUrl' members. *
   this->InitHelpPath () ;

}  //* End Setup() *

//*************************
//*     GetDimensions     *
//*************************
//******************************************************************************
//* Returns the rows/columns needed to open the main dialog.                   *
//* If dialog does not open, caller can retrive and report the terminal-window *
//* dimensions required for this test.                                         *
//*                                                                            *
//* Input  : rowsNeeded : (by reference) receives number of rows required      *
//*          colsNeeded : (by reference) receives number of columns required   *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void SliderTest::GetDimensions ( short& rowsNeeded, short& colsNeeded )
{

   rowsNeeded = dROWS ;
   colsNeeded = dCOLS ;

}  //* End GetDimensions() *

//*************************
//*     InitHelpPath      *
//*************************
//******************************************************************************
//* Initialize the 'helpPath' and 'helpUrl' members.                           *
//* These point to the NcDialog API documentation, in info-reader format and   *
//* HTML format, respectively. If file not found, then the member is           *
//* initialized to the NULL string.                                            *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void SliderTest::InitHelpPath ( void )
{
   // Programmer's Note: POSIX defines 'PATH_MAX' as 4096 bytes, but unfortunately,
   // the compiler defines the default buffer size for getcwd() as 4100 bytes.
   // We love the GNU compiler, but sometimes the developers fail to think clearly.
   // We "know" that the system will not return a string longer than PATH_MAX.
   const short MAX_PATH = PATH_MAX + 16 ; //* Size of buffer to hold path/filename

   const char* infoSpec  = "Texinfo/ncdialogapi.info" ;
   const char* htmlSpec  = "Texinfo/ncdialogapi.html" ;
   const char* urlPrefix = "file:///" ;
   const char* urlOffset = "#Slider-Controls" ;

   gString gsBase,               // text formatting
           gs ;
   char cwd[MAX_PATH] ;          // current working directory

   //* Initialize the member variables *
   *this->helpPath = NULLCHAR ;
   *this->helpUrl  = NULLCHAR ;

   //* Get the current working directory *
   if ( (getcwd ( cwd, MAX_PATH )) == cwd )
   {
      gsBase = cwd ;
      short indx = gsBase.findlast( L'/' ) ;
      if ( indx > ZERO )
      {
         fmFType ft ;
         gsBase.limitChars( indx + 1 ) ;

         //* Construct path to '.info' document *
         gs.compose( "%S%s", gsBase.gstr(), infoSpec ) ;
         if ( (TargetExists ( gs, ft )) && (ft == fmREG_TYPE) )
            gs.copy( this->helpPath, gsDFLTBYTES ) ;

         //* Construct path to '.html' document *
         gs.compose( "%S%s", gsBase.gstr(), htmlSpec ) ;
         if ( (TargetExists ( gs, ft )) && (ft == fmREG_TYPE) )
         {
            gs.insert( urlPrefix ) ;
            gs.append( urlOffset ) ;
            gs.copy( this->helpUrl, gsDFLTBYTES ) ;
         }
      }
   }

}  //* End InitHelpPath() *

//*************************
//*       Cry4Help        *
//*************************
//******************************************************************************
//* Private Method                                                             *
//* --------------                                                             *
//* Ask the user whether the '.info' or '.html' format of the document should  *
//* be displayed. Based on user's selection, open the application Help file.   *
//*  a) Shell out and invoke the 'info' documentation reader.                  *
//*  b) Invoke the default browser with help in HTML format.                   *
//*                                                                            *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************
//* Notes:                                                                     *
//* a) This method and the methods is calls are a simplified version of the    *
//*    Exercalc application's algorithm for invoking external help.            *
//*    For this application, the target help file(s) almost certainly exist,   *
//*    but this sequence demonstrates how to validate the target file before   *
//*    attempting to access it.                                                *
//* b) If HTML format is requested, but HTML file is not found, the '.info'    *
//*    document is invoked instead.                                            *
//* c) Invoking the HTML document with 'xdg-open' opens the document at the    *
//*    top by default. To automagically scan to the target chapter, it is      *
//*    necessary to construct the full, absolute filespec with the chapter     *
//*    name appended. Example:                                                 *
//*    /home/sam/Software/NcDialog/Texinfo/ncdialogapi.html#Slider-Controls    *
//*    The prefix "file:///" is added to indicate that the target file is a    *
//*    LOCAL file, NOT a remote URL.                                           *
//*                                                                            *
//******************************************************************************

void SliderTest::Cry4Help ( void )
{
   const short cryROWS = 6,
               cryCOLS = 38,
               cryYPOS = (this->wpOrig.ypos + this->icptr[stClosePB].ulY),
               cryXPOS = (this->wpOrig.xpos + this->icptr[stClosePB].ulX) ;
   enum cry : short { cryInfoPB = ZERO, cryHtmlPB, cryAbortPB, cryITEMS } ;

   attr_t bAttr = nc.brgr,
          dAttr = nc.bkgr ;
   bool htmlFormat = false ;


   InitCtrl ic[cryITEMS] =
   {
   {  //* 'INFO' pushbutton  - - - - - - - - - - - - - - - - - - - - cryInfoPB *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(cryROWS - 2),           // ulY:       upper left corner in Y
      4,                            // ulX:       upper left corner in X
      1,                            // lines:     control lines
      8,                            // cols:      control columns
      "  INFO  ",                   // dispText:  
      this->pbnColor,               // nColor:    non-focus color
      this->pbfColor,               // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "",                           // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[cryHtmlPB]                // nextCtrl:  link in next structure
   },
   {  //* 'HTML' pushbutton  - - - - - - - - - - - - - - - - - - - - cryHtmlPB *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      ic[cryInfoPB].ulY,            // ulY:       upper left corner in Y
      short(cryCOLS / 2 - 4),       // ulX:       upper left corner in X
      1,                            // lines:     control lines
      8,                            // cols:      control columns
      "  HTML  ",                   // dispText:  
      this->pbnColor,               // nColor:    non-focus color
      this->pbfColor,               // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "",                           // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[cryAbortPB]               // nextCtrl:  link in next structure
   },
   {  //* 'CANCEL' pushbutton  - - - - - - - - - - - - - - - - - -  cryAbortPB *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      ic[cryHtmlPB].ulY,            // ulY:       upper left corner in Y
      short(cryCOLS - 12),          // ulX:       upper left corner in X
      1,                            // lines:     control lines
      8,                            // cols:      control columns
      " CANCEL ",                   // dispText:  
      this->pbnColor,               // nColor:    non-focus color
      this->pbfColor,               // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "",                           // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      NULL                          // nextCtrl:  link in next structure
   },
   } ;

   //* Define the dialog object *
   InitNcDialog dInit( cryROWS,        // number of display lines
                       cryCOLS,        // number of display columns
                       cryYPOS,        // Y offset from upper-left of terminal 
                       cryXPOS,        // X offset from upper-left of terminal 
                       "  Launch Application Help  ", // dialog title
                       ncltDUAL,       // border line-style
                       bAttr,          // border color attribute
                       dAttr,          // interior color attribute
                       ic              // pointer to list of control definitions
                     ) ;

   //* Save the parent dialog *
   this->dp->SetDialogObscured () ;

   //* Instantiate the dialog window *
   NcDialog *dp = new NcDialog ( dInit ) ;

   //* Open the dialog window *
   if ( (dp->OpenWindow ()) == OK )
   {
      winPos wp( 2, 2 ) ;
      dp->WriteParagraph ( wp, "Please select Help document format.", dAttr, true ) ;

      //* Ask user which format to use *
      uiInfo Info ;                 // user interface data returned here
      short  icIndex = ZERO ;       // index of control with input focus
      bool   done = false,          // loop control
             abort = false ;        // 'true' if user aborts

      while ( ! done )
      {
         if ( ic[icIndex].type == dctPUSHBUTTON )
         {
            if ( Info.viaHotkey != false )
               Info.HotData2Primary () ;
            else
               icIndex = dp->EditPushbutton ( Info ) ;
            if ( Info.dataMod != false )
            {
               if ( Info.ctrlIndex == cryInfoPB )
                  htmlFormat = false ;
               else if ( Info.ctrlIndex == cryHtmlPB )
                  htmlFormat = true ;
               else if ( Info.ctrlIndex == cryAbortPB) 
                  abort = true ;
               done = true ;
            }
         }

         //* Move input focus to next/previous control.*
         if ( done == false && Info.viaHotkey == false )
         {
            if ( Info.keyIn == nckSTAB )
               icIndex = dp->PrevControl () ; 
            else
               icIndex = dp->NextControl () ;
         }
      }  // while()

      //* If user has made a selection *
      if ( ! abort )
      {
         if ( htmlFormat && (*this->helpUrl != NULLCHAR) )  // help (HTML format)
         {
            pid_t fpid = vfork () ;                // create the child process

            if ( fpid == ZERO)
            {
               execlp ( "xdg-open", "xdg-open", this->helpUrl, NULL ) ;
               //* In case the exec call fails: child process MUST NOT return.*
               _exit ( 0 ) ;
            }
            else
            {  //* The parent process continues execution here *
               if ( fpid > ZERO )      // child successfully launched
               { /* Currently, nothing to be done. */ }
            }
         }
         else if ( *this->helpPath != NULLCHAR )            // help (info format)
         {
            gString gs( "info -f \"%s\" -n \"Slider Controls\"", this->helpPath ) ;
            this->dp->ShellOut ( soX, gs.ustr() ) ;
         }
         else                    // Error: help file not found
         {
            gString gs( "  Sorry, Help file was not found.  \n"
                        "  \"../Texinfo/ncdialogapi.info\"" ) ;
            dp->WriteParagraph ( wp, gs, dAttr, true ) ;
            this->dp->UserAlert ( 2 ) ;
            chrono::duration<short>aWhile( 3 ) ;
            this_thread::sleep_for( aWhile ) ;
         }
      }  // (!abort)
   }     // dialog opened
   if ( dp != NULL )             // close the window
      delete ( dp ) ;

   //* Restore parent dialog *
   this->dp->RefreshWin () ;

}  //* End Cry4Help() *

//*************************
//*     ControlUpdate     *
//*************************
//******************************************************************************
//* This is a callback method for manually updating the dialog controls.       *
//* NOTE: This method is a member of the SliderTest class, and is declared     *
//*       as "static". The fact that this is a "static" member means that the  *
//*       the method belongs to the _class_, and not to any specific instance  *
//*       of the class. This is necessary to enable a member method to         *
//*       specified as the callback method. Because "static" members do not    *
//*       have access to private data members, we must rely on the external    *
//*       pointer to the specific _instance_ of the SliderTest class.          *
//*             https://en.cppreference.com/w/cpp/language/lambda              *
//*                                                                            *
//* For future versions of the callback mechanism:                             *
//* ----------------------------------------------                             *
//* C++11 provides "lambda expressions" which will capture the call arguments. *
//* Use care, however, because lambda-expression syntax has been update in     *
//* C++14, C++17, C++20 and C++23 (at least), so to retain compatibility       *
//* with C++11 don't use the updated syntax.                                   *
//* <https://towardsdatascience.com/c-basics-understanding-lambda-7df00705fa48>*
//*                                                                            *
//*  For this test, the following are updated by the callback method:          *
//*   a) Display configuration data for the Slider control.                    *
//*   b) For Sliders stVstd1SL and stVrev1SL, transfer the value in the        *
//*      control with focus to the control without focus.                      *
//*                                                                            *
//* Input  : currIndex: index of control that currently has focus              *
//*          wkey     : user's key input data                                  *
//*          firstTime: the EstablishCallback() method calls this method once  *
//*                     with firstTime==true, to perform any required          *
//*                     initialization. Subsequently, the NcDialog class       *
//*                     always calls with firstTime==false.                    *
//* Returns: OK                                                                *
//******************************************************************************

short SliderTest::ControlUpdate ( const short currIndex, 
                                  const wkeyCode wkey, bool firstTime )
{
   //* Display configuration info for vertical sliders *
   const char* cfgVTemplate = "vCur:%-6.1lf\n"
                              "vPct:%-6.1lf\n"
                              "vMin:%-6.1lf\n"
                              "vMax:%-6.1lf\n"
                              "dCur:%hd\n"
                              "dTot:%hd\n"
                              "vert:%hhd\n"
                              "rev :%hhd" ;
   //* Display configuration info for horizontal sliders *
   const char* cfgHTemplate = "vCur:%-6.1lf dCur:%hd\n"
                              "vPct:%-6.1lf dTot:%hd\n"
                              "vMin:%-6.1lf vert:%hhd\n"
                              "vMax:%-6.1lf rev :%hhd" ;
   const char* cfgPctTemplate = "%-6.1lf" ;
   const short vHeight = 8, vWidth = 11 ;

   sliderData sd ;      // slider configuration data
   winPos wp ;          // message position
   gString gs ;         // message formatting

   if ( firstTime  )
   {
      //* Display initial slider configurations *
      for ( short i = ZERO ; i < stCONTROLS ; ++i )
      {
         if ( stPtr->icptr[i].type == dctSLIDER )
         {
            if ( (stPtr->dp->GetSliderConfig ( i, sd )) == OK )
            {
               //* Test for direction of growth. For this test, *
               //* we assume that growth direction is larger.   *
               if ( stPtr->icptr[i].lines > stPtr->icptr[i].cols )
               {
                  wp.ypos = stPtr->icptr[i].ulY + stPtr->icptr[i].lines + 1 ;
                  wp.xpos = stPtr->icptr[i].ulX ;
                  stPtr->dp->ClearArea ( wp.ypos, wp.xpos, vHeight, vWidth ) ;
                  gs.compose( cfgVTemplate,
                              &sd.curValue, &sd.curPercent, &sd.minValue, &sd.maxValue,
                              &sd.divCurrent, &sd.divSteps, &sd.vertical, &sd.reverse ) ;
               }
               else     // horizontal sliders
               {
                  wp.ypos = stPtr->icptr[i].ulY + stPtr->icptr[i].lines ;
                  wp.xpos = stPtr->icptr[i].ulX ;
                  gs.compose( cfgHTemplate,
                              &sd.curValue,   &sd.divCurrent, 
                              &sd.curPercent, &sd.divSteps, 
                              &sd.minValue,   &sd.vertical, 
                              &sd.maxValue,   &sd.reverse ) ;
               }
               stPtr->dp->WriteParagraph ( wp, gs, stPtr->dColor ) ;
            }
         }
      }
      stPtr->dp->RefreshWin () ;
   }

   //* For the slider with focus, display the current configuration *
   if ( (stPtr->icptr[currIndex].type == dctSLIDER) &&
        ((stPtr->dp->GetSliderConfig ( currIndex, sd )) == OK) )
   {
      //* If Slider #1 and Slider #2 are linked *
      if ( (currIndex == stVstd1SL) || (currIndex == stVrev1SL) )
      {
         double focusPct ;
         short  nfIndex = currIndex == stVstd1SL ? stVrev1SL : stVstd1SL ;
         bool rbState ;
         stPtr->dp->GetRadiobuttonState ( stLinkRB, rbState ) ;
         if ( rbState )
         {  //* Set non-focus-slider value to focus-slider value. *
            // Programmer's Note: We COULD use sd.curPercent as the source value,
            // but we need to exercise the alternate 'GetSliderValue()' method.
            stPtr->dp->GetSliderValue ( currIndex, focusPct, true ) ;
            stPtr->dp->SetSliderValue ( nfIndex, focusPct, true ) ;
            //* Update display of non-focus stats *
            wp.ypos = stPtr->icptr[nfIndex].ulY + stPtr->icptr[nfIndex].lines + 2 ;
            wp.xpos = stPtr->icptr[nfIndex].ulX + 5 ;
            gs.compose( cfgPctTemplate, &focusPct ) ;
            stPtr->dp->WriteParagraph ( wp, gs, stPtr->bColor, true ) ;
         }
      }

      //* Test for direction of growth. For this test, *
      //* we assume that growth direction is larger.   *
      if ( stPtr->icptr[currIndex].lines > stPtr->icptr[currIndex].cols )
      {
         wp.ypos = stPtr->icptr[currIndex].ulY + stPtr->icptr[currIndex].lines + 1 ;
         wp.xpos = stPtr->icptr[currIndex].ulX ;
         stPtr->dp->ClearArea ( wp.ypos, wp.xpos, vHeight, vWidth ) ;
         gs.compose( cfgVTemplate,
                     &sd.curValue, &sd.curPercent, &sd.minValue, &sd.maxValue,
                     &sd.divCurrent, &sd.divSteps, &sd.vertical, &sd.reverse ) ;
      }
      else     // horizontal sliders
      {
         wp.ypos = stPtr->icptr[currIndex].ulY + stPtr->icptr[currIndex].lines ;
         wp.xpos = stPtr->icptr[currIndex].ulX ;
         gs.compose( cfgHTemplate,
                     &sd.curValue,   &sd.divCurrent, 
                     &sd.curPercent, &sd.divSteps, 
                     &sd.minValue,   &sd.vertical, 
                     &sd.maxValue,   &sd.reverse ) ;
                     
      }
      stPtr->dp->WriteParagraph ( wp, gs, stPtr->dColor, true ) ;
   }

   return OK ;

}  //* ControlUpdate() *

//** - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -  **
//** - - - - - - - - NON-MEMBER METHODS FOR THIS MODULE  - - - - - - - - - -  **
//** - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -  **

//*************************
//*     TargetExists      *
//*************************
//******************************************************************************
//* NON-MEMBER METHOD                                                          *
//* -----------------                                                          *
//* Determine whether specified file exists and return its filetype.           *
//*                                                                            *
//* Input  : trgPath : filespec of target file                                 *
//*          fType   : (by reference)                                          *
//*                  : if target exists, initialized to target fileType        *
//*                    else fmUNKNOWN_TYPE                                     *
//*                                                                            *
//* Returns: 'true' if target exists, else 'false'                             *
//******************************************************************************
//* Programmer's Note: This method is grossly inefficient, but this            *
//* application requires simplicity of design over speed.                      *
//******************************************************************************

static bool TargetExists ( const gString& trgPath, fmFType& fType )
{
   tnFName fs ;               // file stats
   bool exists = false ;      // return value

   if ( (GetFileStats ( trgPath, fs )) )
   {
      fType = fs.fType ;
      exists = true ;
   }
   return exists ;

}  //* End TargetExists() *

//*************************
//*     GetFileStats      *
//*************************
//******************************************************************************
//* NON-MEMBER METHOD                                                          *
//* -----------------                                                          *
//* Perform a 'stat' ('lstat') on the file specified by the full path string.  *
//*                                                                            *
//* Input  : trgPath: full path/filename specification                         *
//*          fStats : (by reference) receives the stat data for the file       *
//*                                                                            *
//* Returns: 'true'  if file exists, else 'false'                              *
//******************************************************************************

static bool GetFileStats ( const gString& trgPath, tnFName& fStats )
{
   bool exists = false ;            // return value

   //* Provides an orderly crash in case of failure *
   fStats.modTime.reset() ;
   fStats.fBytes = ZERO ;
   fStats.fType = fmUNKNOWN_TYPE ;
   fStats.readAcc = fStats.writeAcc = false ;

   if ( (lstat64 ( trgPath.ustr(), &fStats.rawStats )) == OK )
   {
      //* Extract filename from caller's string *
      short indx = trgPath.findlast( L'/' ) ;
      if ( indx < ZERO )   indx = ZERO ;
      else                 ++indx ;
      gString fName( &trgPath.gstr()[indx] ) ;
      fName.copy( fStats.fName, MAX_FNAME ) ;

      fStats.fBytes = fStats.rawStats.st_size ;    // number of bytes in file

      //* Decode the filetype *
      UINT mode = fStats.rawStats.st_mode ;
      if ( (S_ISREG(mode)) != false )        fStats.fType = fmREG_TYPE ;
      else if ( (S_ISDIR(mode)) != false )   fStats.fType = fmDIR_TYPE ;
      else if ( (S_ISLNK(mode)) != false )   fStats.fType = fmLINK_TYPE ;
      else if ( (S_ISCHR(mode)) != false )   fStats.fType = fmCHDEV_TYPE ;
      else if ( (S_ISBLK(mode)) != false )   fStats.fType = fmBKDEV_TYPE ;
      else if ( (S_ISFIFO(mode)) != false )  fStats.fType = fmFIFO_TYPE ;
      else if ( (S_ISSOCK(mode)) != false )  fStats.fType = fmSOCK_TYPE ;

      //* Decode the mod time *
      fStats.modTime.epoch = (int64_t)fStats.rawStats.st_mtime ;
      fStats.modTime.sysepoch = (time_t)fStats.rawStats.st_mtime ; // (possible narrowing)
      Tm bdt ;       // receives broken-down time
      if ( (localtime_r ( &fStats.modTime.sysepoch, &bdt )) != NULL )
      {
         //* Translate to localTime format (timezone data not decoded) *
         fStats.modTime.date    = bdt.tm_mday ;         // today's date
         fStats.modTime.month   = bdt.tm_mon + 1 ;      // month
         fStats.modTime.year    = bdt.tm_year + 1900 ;  // year
         fStats.modTime.hours   = bdt.tm_hour ;         // hour
         fStats.modTime.minutes = bdt.tm_min ;          // minutes
         fStats.modTime.seconds = bdt.tm_sec ;          // seconds
         fStats.modTime.day     = bdt.tm_wday ;         // day-of-week (0 == Sunday)
         fStats.modTime.julian  = bdt.tm_yday ;         // Julian date (0 == Jan.01)
      }

      //* Initialize the read-access and write-access flags *
      fStats.readAcc = bool((access ( trgPath.ustr(), R_OK )) == ZERO) ;
      if ( fStats.readAcc != false && fStats.fType == fmDIR_TYPE )
         fStats.readAcc = bool((access ( trgPath.ustr(), X_OK )) == ZERO) ;
      fStats.writeAcc = bool((access ( trgPath.ustr(), W_OK )) == ZERO) ;

      exists = true ;
   }
   return exists ;

}  //* End GetFileStats() *

//*************************
//*                       *
//*************************
//******************************************************************************
//*                                                                            *
//*                                                                            *
//* Input  :                                                                   *
//*                                                                            *
//* Returns:                                                                   *
//******************************************************************************

