//******************************************************************************
//* File       : Progbar.hpp                                                   *
//* Author     : Mahlon R. Smith                                               *
//*              Copyright (c) 2015-2025 Mahlon R. Smith, The Software Samurai *
//*                  GNU GPL copyright notice located in NcDialog.hpp          *
//* Date       : 12-May-2021                                                   *
//*                                                                            *
//* Description: Class definition for the Progbar class.                       *
//******************************************************************************

//* All necessary information for:                  *
//* NCurses, NcWindow, NcDialog and gString classes.*
#include "GlobalDef.hpp"

//* Conditional compilation flag to enable/disable debugging data. *
//* Note that the parent dialog must pass us a pointer, so we can  *
//* display the diagnostic data in the parent window.              *
#define DEBUG_PBAR (0)

//* Maximum width of begining and ending label text.*
const short maxLABELCOLS = 4 ;

class Progbar ;      //* dummy declaration

//* Macro for callback method used by secondary monitor thread.*
//* See the pbar::Init.threadCb member.                        *
typedef short (*PBMON)( short stepsRemaining ) ;

//* Call parameters for creating a Progbar object.*
class pbarInit
{
   public:
   //* Initialization Constructor *
   pbarInit ( attr_t dc, attr_t bc, short c, bool h )
   {
      this->dlgColor = dc ;
      this->barColor = bc ;
      this->cells    = c ;
      this->horiz    = h ;

      //* Defaults are calculated for remaining fields. *
      //* Custom values may be assigned to these fields *
      //* after instantiation.                          *
      this->steps        = this->cells ;  // one cell per step
      this->beginText    =                // no text displayed
      this->endText      = 
      this->titleText    = 
      this->cancelText   = 
      this->closeText    = NULL ;
      this->enableCancel =                // pushbutton controls disabled
      this->enableClose  = false ;
      this->border       = true ;         // dialog will have a border
      this->borderColor = this->dlgColor ;// border is same color as dialog
      this->yOffset      =                // center dialog in terminal window
      this->xOffset      = (-1) ;
      this->threadCb     = NULL ;         // callback method not specified
      #if DEBUG_PBAR != 0  // FOR DEBUGGING ONLY
      this->pdp = NULL ;
      #endif   // DEBUG_PBAR
   }

   //** Optional Fields - default values are calculated **
   //** and may be updated after instantiation.         **
   const char* beginText ;    // text to display at   0% end
   const char* endText ;      // text to display at 100% end
   const char* titleText ;    // title text (horizontal layout only)
   const char* cancelText ;   // text for 'Cancel' Pushbutton (default: " Cancel ")
   const char* closeText ;    // text for 'Close' Pushbutton (default: " Close ")
   attr_t   borderColor ;     // border color
   short    steps ;           // number of discrete steps across the bar (>= 'cells')
   bool     enableCancel ;    // enable the 'Cancel' pushbutton
   bool     enableClose ;     // enable the 'Close' pushbutton
   bool     border ;          // enable border 
   short    yOffset ;         // offset from terminal origin
   short    xOffset ;
   PBMON    threadCb ;        // address of callback method for secondary thread
   #if DEBUG_PBAR != 0        // FOR DEBUGGING ONLY
   NcDialog* pdp ;            // access to parent dialog
   #endif   // DEBUG_PBAR

   //** Required Fields are set on instantiation.**
   private:
   pbarInit ( void ) { /* do not allow instantiation without required parms */ }
   attr_t   dlgColor ;     // dialog color
   attr_t   barColor ;     // color of progress bar
   short    cells ;        // number of character cells in progress bar
   bool     horiz ;        // 'true'  for horizontal layout
                           // 'false' for vertical layout
   friend class Progbar ;
} ;

//* Block character definitions for both horizontal and vertical *
//* progress-bar action.                                         *
const short cellDIV = 8 ;     // character cell divided into eighths
const wchar_t hblock[cellDIV + 1] =
{
   L' ', 
   0x0258F,    // ▏
   0x0258E,    // ▎
   0x0258D,    // ▍
   0x0258C,    // ▌
   0x0258B,    // ▋
   0x0258A,    // ▊
   0x02589,    // ▉
   0x02588,    // █
} ;
const wchar_t vblock[cellDIV + 1] =
{
   L' ', 
   0x02581,    // ▁
   0x02582,    // ▂
   0x02583,    // ▃
   0x02584,    // ▄
   0x02585,    // ▅
   0x02586,    // ▆
   0x02587,    // ▇
   0x02588,    // █
} ;

const short maxCONTROLS = 1 ;
const short labelCHARS = 24 ;
const short labelBYTES = (labelCHARS * 4) ;
class Progbar
{
   public:
   virtual ~Progbar ( void ) ;      // destructor (this is an abstract class)

   Progbar ( const pbarInit& init ) ;// constructor

   //* Open the dialog.                                            *
   //* Input  : none                                               *
   //* Returns: 'true'  if successful                              *
   //*          'false' if window did not open                     *
   //*                  (position error is likely cause)           *
   bool open ( void ) ;

   //* If Pushbutton control(s) have been enabled, AND if a        *
   //* callback method has been specified for the secondary        *
   //* thread, then let the user interact with the Pushbuttons.    *
   //* Input  : abort : (optional, NULL ptr by default)            *
   //*            pointer to a boolean flag monitored during the   *
   //*            operation. If set ('true') it indicates that the *
   //*            thread should immediately return to caller.      *
   //* Returns: percentage completion:                             *
   //*          100  : operation complete                          *
   //*          0-99 : user aborted operation before completion    *
   //*          ERR  : method called inappropriately               *
   //*                 a) dialog not open                          *
   //*                 b) no auto-update thread established        *
   //*                 c) no Pushbutton control defined, OR        *
   //*                    window too small to include controls     *
   short UserInteract ( bool* abort = NULL ) ;
   
   //* Update the progress bar: move one step forward.             *
   //* (display is refreshed)                                      *
   //* Input  : none                                               *
   //* Returns: percentage of steps completed (integer value 0-100)*
   short update ( void ) ;

   //* Update the progress bar: move specified number if steps.    *
   //* positive value: move forward, negative value: move backward *
   //* zero value: clear the bar,                                  *
   //* value larger than remaining steps: fill the bar             *
   //* (display is refreshed)                                      *
   //* Input  : inc :                                              *
   //*            a) positive values move forward                  *
   //*            b) negative values move backward                 *
   //*            c) zero value clears the bar to ZERO steps       *
   //*            d) any value greater than the remaining number   *
   //*               of steps fills the bar to maximum steps       *
   //* Returns: percentage of steps completed (integer value 0-100)*
   short update ( short inc ) ;

   //* Refresh (redraw) the Progbar object *
   //* Input  : none                                               *
   //* Returns: nothing                                            *
   void  refresh ( void ) ;

   private:
   //* Default constructor - included for safety. If a parameterless instance  *
   //* is declared, then the compiler will complain. (you're welcome :-)       *
   Progbar ( void ) {}
   void ProgressMonitor ( void ) ;  // periodically access application's callback
   void clear ( bool filled = false ) ; // clear (or fill) the bar
   void center ( void ) ;           // center dialog in terminal window
   void initControls ( const char* canPtr, const char*cloPtr ) ; // setup Pushbuttons
   void increment ( void ) ;        // increment bar by one step
   void decrement ( void ) ;        // decrement bar by one step
   short pct ( void ) ;             // calculate percentage completion

   NcDialog* dPtr ;                 // pointer to embedded NcDialog object
   InitCtrl* icPtr ;                // pointer to control definitions
   PBMON     cbPtr ;                // address of callback method for independent thread
   thread    monitor ;              // handle of secondary thread
   wchar_t begText[labelCHARS] ;    // text displayed at beginning of bar
   wchar_t endText[labelCHARS] ;    // text displayed at end of bar
   char    titText[labelBYTES] ;    // title for sub-dialog
   char    canText[labelBYTES] ;    // text for 'Cancel' pushbutton
   char    cloText[labelBYTES] ;    // text for 'Close' pushbutton
   attr_t  dColor ;                 // dialog background color
   attr_t  eColor ;                 // dialog border color
   attr_t  bColor ;                 // progress bar color
   short   dRows ;                  // number of dialog rows (1 through 6)
   short   dCols ;                  // number of dialog columns
   short   ulY ;                    // offset from top left of terminal window
   short   ulX ;
   short   barCells ;               // number of character cells in bar
   winPos  barPos ;                 // base position of bar
   winPos  begPos ;                 // position of beginning label text
   winPos  endPos ;                 // position of ending label text
   short   barSteps ;               // number of divisions for progress bar
   short   curStep ;                // number of completed steps
   short   perStep ;                // number of scanlines per step (8 per cell)
   short   curCell ;                // index of current bar cell
   short   curScan ;                // number of scanlines written in current cell
   short   canIndex ;               // index of 'Cancel' button
   short   cloIndex ;               // index of 'Close' button
   bool    border ;                 // 'true' if dialog has a border
   bool    canBut ;                 // 'true' if 'Cancel" button displayed
   bool    cloBut ;                 // 'true' if 'Close" button displayed
   bool    horiz ;                  // 'true' for horizontal left-to-right
                                    // 'false' for vertical bottom-to-top
   bool    monActive ;              // 'true' if secondary thread is active
   bool    monAbort ;               // 'true' to terminate secondary thread
   bool    parmError ;              // 'true' if user has passed garbage parameter(s)
   bool    isOpen ;                 // 'true' if dialog is open
   #if DEBUG_PBAR != 0        // FOR DEBUGGING ONLY
   NcDialog* pdp ; // parent dialog access
   void debugPB ( bool after ) ;
   #endif   // DEBUG_PBAR

} ;

