//********************************************************************************
//* File       : Progbar.cpp                                                     *
//* Author     : Mahlon R. Smith                                                 *
//*              Copyright (c) 2015-2025 Mahlon R. Smith, The Software Samurai   *
//*                  GNU GPL copyright notice located in NcDialog.hpp            *
//* Date       : 12-May-2025                                                     *
//*                                                                              *
//* Description: Class definition for the Progbar class.                         *
//*              This class implements a widget based on the NcDialog API.       *
//*              It is a dialog window which displays a 'progress bar' so user   *
//*              may visually monitor the status of a task in progress.          *
//*                                                                              *
//* Development Tools: See NcDialog.cpp.                                         *
//********************************************************************************
//* Version History (most recent first):                                         *
//*                                                                              *
//* v: 0.00.03 13-Oct-2016                                                       *
//*   - Call to clear() method now refreshes the display.                        *
//*                                                                              *
//* v: 0.00.02 27-Jul-2015                                                       *
//*   - Meets all specifications; however, this is still a baby application,     *
//*     so bugs are a definite possibility.                                      *
//*                                                                              *
//* v: 0.00.01 18-Jul-2015                                                       *
//*   - First effort: define the class and options.                              *
//*   - Create test dialog: Dialog4, Test09.                                     *
//********************************************************************************
//* Programmer's Notes:                                                          *
//*                                                                              *
//*                                                                              *
//*                                                                              *
//*                                                                              *
//*                                                                              *
//********************************************************************************

//* All necessary information for:                  *
//* NCurses, NcWindow, NcDialog and gString classes.*
#include "GlobalDef.hpp"

//* Definition of the Progbar dialog widget.*
#include "Progbar.hpp"


//*************************
//*      ~Progbar         *
//*************************
//******************************************************************************
//* Destructor: release dynamically-allocated memory.                          *
//*                                                                            *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

Progbar::~Progbar ( void )
{
   //* If the auto-update monitor was launched, *
   //* be sure the thread has returned to us.   *
   if ( this->monActive )
   {
      this->monAbort = true ;
      this->monitor.join() ;
   }

   //* If controls defined, delete the definitions. *
   if ( this->icPtr != NULL )
      delete [] this->icPtr ;

   delete this->dPtr ;           // close the dialog

}  //* End ~Progbar() *

//*************************
//*       Progbar         *
//*************************
//********************************************************************************
//* Initialization constructor.                                                  *
//*                                                                              *
//* Notes:                                                                       *
//* -- The actual dimensions of the Progbar object are calculated based on       *
//*    the options provided.                                                     *
//* -- The pbarInit constructor initializes all fields, both specified           *
//*    required fields and optionally-specified fields:                          *
//* -- If 'steps' specified, then:                                               *
//*    minimum   : cells i.e. cellDIV divisions-per-step (default)               *
//*    maximum   : cells * cellDIV                                               *
//*    caveat    : divisions-per-step: 1, 2, 3, 4, 5, 6, 7, 8  however,          *
//*                (divs-per-step * steps) must be >= (cells * cellDIV),         *
//*                otherwise the sequence would complete before the bar is       *
//*                filled, so we pad with extra steps as necessary.              *
//*                                                                              *
//*                                                                              *
//* Input  : init : (ProgInit-class object, by reference) setup parameters       *
//*                                                                              *
//* Returns: implicitly returns a pointer to the object                          *
//*          - If any bad parameter detected, set defaults for critical values   *
//             and draw "PARM ERROR!" message in the bar area.                   *
//********************************************************************************
//* NOTES:                                                                       *
//*                                                                              *
//*                                                                              *
//********************************************************************************

Progbar::Progbar ( const pbarInit& init )
{
   //* Initialize reserved members *
   this->dPtr      = NULL ;         // dialog not yet defined
   this->icPtr     = NULL ;         // controls not yet defined
   this->cbPtr     = NULL ;         // no thread callback target
   this->canBut    =                // no buttons yet defined
   this->cloBut    = false ;
   *this->canText  =                // no button text yet defined
   *this->cloText  = nckNULLCHAR ;
   this->canIndex  = ZERO ;         // index of first (only) control (if any)
   this->monActive = false ;        // no secondary thread running
   this->monAbort  = false ;        //   and not in need of abort
   this->parmError = false ;        // hope for the best
   this->isOpen    = false ;        // dialog not yet open

   // Programmer's Note: Because the 'cells' parameter drives everything else,
   // we do range checking, and if necessary, go all alpha-nerd on their ass.
   this->barCells = init.cells ;
   if ( this->barCells < 1 )
   {
      this->barCells  = 11 ;        // make room for the error message
      this->parmError = true ;
   }
   this->barSteps = init.steps ;
   if (   (this->barSteps < this->barCells) 
       || (this->barSteps > (this->barCells * cellDIV)) )
   {
      if ( this->barCells < 11 )
         this->barCells = 11 ;
      this->barSteps = this->barCells ;
      this->parmError = true ;
   }
   //* Ensure that (steps * divs-per-step) >= total divisions.*
   //* (see note above)                                       *
   short divs = this->barCells * cellDIV ;
   this->perStep = divs / this->barSteps ;
   while ( (this->barSteps * this->perStep) < divs )
      ++this->barSteps ;

   //* Set the label text *
   gString gs( init.beginText ) ;
   gs.limitCols( maxLABELCOLS ) ;
   gs.copy( this->begText, labelCHARS ) ;
   gs = init.endText ;
   gs.limitCols( maxLABELCOLS ) ;
   gs.copy( this->endText, labelCHARS ) ;
   if ( init.horiz && init.border )
   {  // title placement works only with bordered, horizontal layout
      gs = init.titleText ;
      gs.copy( this->titText, (labelBYTES) ) ;
   }
   else
      *this->titText = nckNULLCHAR ;

   //* Set the dialog's layout and color scheme *
   this->horiz  = init.horiz ;
   this->border = init.border ;
   if ( this->horiz && this->border )
   {  // horizontal layout with border
      this->dCols = this->barCells + 2 ;
      this->dRows = 3 ;
      if ( *this->begText != nckNULLCHAR || *this->endText != nckNULLCHAR )
      {
         ++this->dRows ;
         this->begPos.ypos = this->endPos.ypos = 2 ;
         this->begPos.xpos = 1 ;
         this->endPos.xpos = this->dCols - (maxLABELCOLS + 1) ;
      }
   }
   else if ( this->horiz )
   {  // horizontal layout (no border)
      this->dCols = this->barCells ;
      this->dRows = 1 ;
      if ( *this->begText != nckNULLCHAR || *this->endText != nckNULLCHAR )
      {
         ++this->dRows ;
         this->begPos.ypos = this->endPos.ypos = 1 ;
         this->begPos.xpos = ZERO ;
         this->endPos.xpos = this->dCols - maxLABELCOLS ;
      }
   }
   else if ( ! this->horiz && this->border )
   {  // vertical layout with border
      this->dRows = this->barCells + 2 ;
      this->dCols = 3 ;
      if ( *this->begText != nckNULLCHAR || *this->endText != nckNULLCHAR )
      {
         this->dRows += 2 ;
         this->dCols += 3 ;
         this->begPos.xpos = this->endPos.xpos = 1 ;
         this->begPos.ypos = this->dRows - 2 ;
         this->endPos.ypos = 1 ;
      }
   }
   else
   {  // vertical layout (no border)
      this->dRows = this->barCells ;
      this->dCols = 1 ;
      if ( *this->begText != nckNULLCHAR || *this->endText != nckNULLCHAR )
      {
         this->dRows += 2 ;
         this->dCols += 3 ;
         this->begPos.xpos = this->endPos.xpos = ZERO ;
         this->begPos.ypos = this->dRows - 1 ;
         this->endPos.ypos = ZERO ;
      }
   }
   this->ulY      = init.yOffset ;  // provisionally accept caller's values for these
   this->ulX      = init.xOffset ;
   this->dColor   = init.dlgColor ;
   this->eColor   = init.borderColor ;
   this->bColor   = init.barColor ;


   //* Define the visual progress bar.                       *
   //* ('barCells', 'barSteps' and 'perStep were set above.) *
   this->curStep  = ZERO ;
   this->curCell  = ZERO ;
   this->curScan  = ZERO ;
   if ( this->border )
   {
      if ( this->horiz )   this->barPos = { 1, 1 } ;
      else
      {
         this->barPos = { short(this->dRows - 2), 1 } ;
         if ( *this->begText != nckNULLCHAR || *this->endText != nckNULLCHAR )
         {
            --this->barPos.ypos ;
            this->barPos.xpos += 2 ;
         }
      }
   }
   else
   {
      if ( this->horiz )   this->barPos = { ZERO, ZERO } ;
      else
      {
         this->barPos = { short(this->dRows - 1), ZERO } ;
         if ( *this->begText != nckNULLCHAR || *this->endText != nckNULLCHAR )
         {
            --this->barPos.ypos ;
            this->barPos.xpos += 2 ;
         }
      }
   }

   //* Auto-update monitor (if specified) *
   this->cbPtr = init.threadCb ;

   //* Define the Pushbutton control if specified.            *
   //* User interface controls make sense ONLY if caller has  *
   //* specified a callback method for the auto-update thread.*
   if ( init.enableCancel || init.enableClose )
   {
      if ( init.threadCb != NULL )
      {
         //* These are both the same physical button and 'Cancel' controls  *
         //* the action. The 'Close' button is an optional change of text   *
         //* for the button after the operation being monitored is complete.*
         this->canBut = bool(init.enableCancel || init.enableClose) ;
         this->cloBut = init.enableClose ;
         this->initControls ( init.cancelText, init.closeText ) ;
      }
      else
      {  //* Without an auto-update thread, the bar would never be updated. *
         // Programmer's Note: There are situations where this might not be 
         // true, but we enforce it at this time.
         if ( this->barCells < 11 )
            this->barCells = 11 ;
         this->barSteps = this->barCells ;
         this->parmError = true ;
      }
   }

   //* If position not specified, center dialog in terminal window *
   if ( this->ulY < ZERO || this->ulX < ZERO )
      this->center() ;

   //* As a final test, verify that our dialog stays entirely within *
   //* the terminal window. This won't cure all parameter problems,  *
   //* but at least the dialog will open.                            *
   short ydim, xdim ;
   nc.ScreenDimensions ( ydim, xdim ) ;
   while ( (this->ulY + this->dRows) > ydim )
   {
      --this->dRows ;
      --this->barCells ;
      --this->barPos.ypos ;
      --this->begPos.ypos ;
      this->parmError = true ;
   }
   while ( (this->ulX + this->dCols) > xdim )
   {
      --this->dCols ;
      --this->barCells ;
      --this->endPos.xpos ;
      this->parmError = true ;
   }

   #if DEBUG_PBAR != 0        // FOR DEBUGGING ONLY
   this->pdp = init.pdp ;
   #endif   // DEBUG_PBAR

}  //* End Progbar() *

//*************************
//*         open          *
//*************************
//******************************************************************************
//* Open the dialog: instantiate the dialog using specified parameters, and    *
//* then make the dialog visible.                                              *
//*                                                                            *
//* Callback Method:                                                           *
//* -- If specified, then a secondary thread is launched to run in the         *
//*    callback method, and the Progbar will be dynamically updated through    *
//*    the callback method.                                                    *
//* -- If not specified, then no secondary thread is created, and the Progbar  *
//*    object is passive, waiting for updates from the application.            *
//* Primary thread will return to caller.                                      *
//*                                                                            *
//*                                                                            *
//* NOTE: If the Progbar object obscures any part of the parent dialog, then   *
//*       be sure to call your 'SetDialogObscured' method before opening the   *
//*       Progbar widget. ALSO, if the parent dialog is refreshed, then        *
//*       remember to also refresh the Progbar object.                         *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: 'true'  if successful                                             *
//*          'false' if window did not open (position error is likely cause)   *
//******************************************************************************

bool Progbar::open ( void )
{
   //* Initial parameters for dialog window *
   InitNcDialog dInit( this->dRows,    // number of display lines
                       this->dCols,    // number of display columns
                       this->ulY,      // Y offset from upper-left of terminal 
                       this->ulX,      // X offset from upper-left of terminal 
                       this->titText,  // dialog title
                       ncltSINGLE,     // border line-style
                       this->eColor,   // border color attribute
                       this->dColor,   // interior color attribute
                       this->icPtr     // pointer to list of control definitions
                     ) ;
   this->dPtr = new NcDialog( dInit ) ;

   if ( (this->isOpen = ((this->dPtr->OpenWindow ()) == OK)) )
   {
      //* Because windows open with borders by default, if *
      //* the dialog is defined as borderless, we need to  *
      //* verify that there are no visual artifacts.       *
      // Programmer's Note: Because only Pushbutton controls are defined
      // within this dialog, we aren't erasing any label data here.
      if ( ! this->border )
         this->dPtr->ClearWin ( true ) ;

      //* Draw the empty progress bar *
      this->clear() ;

      //* If text legend specified *
      if ( *this->begText != nckNULLCHAR )
         this->dPtr->WriteString ( this->begPos, this->begText, this->dColor ) ;
      if ( *this->endText != nckNULLCHAR )
         this->dPtr->WriteString ( this->endPos, this->endText, this->dColor ) ;

      //* If the constructor detected a bad parameter, try to report it.*
      if ( this->parmError )
      {
         gString gs ;
         winPos wp = this->barPos ;
         if ( this->horiz )
            gs = "PARM ERROR!" ;
         else
         {
            gs = "P\nA\nR\nM\n \nE\nR\nR\nO\nR\n!" ;
            wp.ypos -= this->barCells - 1 ;
         }
         this->dPtr->WriteParagraph ( wp, gs, this->bColor ) ;
      }

      this->dPtr->RefreshWin () ;     // make everything visible

      //* If auto-update callback specified, BUT no user-interface *
      //* controls defined, split the threads and let caller's     *
      //* thread return. Local thread will monitor the update.     *
      if ( this->cbPtr != NULL && !(this->canBut || this->cloBut) )
      {
         this->monitor = thread( &Progbar::ProgressMonitor, this ) ;
      }
   }
   return this->isOpen ;

}  //* End open() *

//*************************
//*    ProgressMonitor    *
//*************************
//******************************************************************************
//* PRIVATE METHOD:                                                            *
//* Home to the optional secondary thread which monitors progress of the       *
//* activity being reported by periodically calling the callback method to     *
//* get the step increment/decrement.                                          *
//*                                                                            *
//* Continues until:                                                           *
//*  a) operation reaches completion (this->currStep == this->barSteps)        *
//*  b) operation is aborted (this->monAbort != false )                        *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void Progbar::ProgressMonitor ( void )
{
   this->monActive = true ;      // signal that we're starting

   short inc, newSteps ;
   bool  done = false ;
   do
   {
      chrono::duration<short, std::milli>aMoment( 200 ) ;
      this_thread::sleep_for( aMoment ) ;

      if ( (inc = (this->barSteps - this->curStep)) > ZERO )
      {
         newSteps = this->cbPtr ( inc ) ;
         if ( newSteps == 1 )
         {
            this->update() ;
         }
         else if ( newSteps != ZERO )
         {
            this->update ( newSteps ) ;
         }
         else     // no progress to report
            ;
      }
      else
         done = true ;
   }
   while ( ! done && ! this->monAbort ) ;

   // NOTE: The secondary thread stops executing here, BUT it is not dead.
   // Only when the destructor call 'join()' is the thread actually terminated.

}  //* End ProgressMonitor() *

//*************************
//*     UserInteract      *
//*************************
//******************************************************************************
//* If Pushbutton control defined, interact directly the user until the        *
//* operation is complete or user aborts the operation.                        *
//*                                                                            *
//*                                                                            *
//* Input  : abort : (optional, NULL pointer by default)                       *
//*                  pointer to a boolean flag monitored during the operation. *
//*                  If set ('true') it indicates that the thread should       *
//*                  immediately return to caller.                             *
//*                                                                            *
//* Returns: percentage completion:                                            *
//*          100  : operation complete                                         *
//*          0-99 : user aborted operation before completion                   *
//*          ERR  : method called inappropriately                              *
//*                 a) dialog not open                                         *
//*                 b) no auto-update thread established                       *
//*                 c) no Pushbutton control defined, OR                       *
//*                    window too small to include controls                    *
//******************************************************************************
//* NOTE: We cannot call the EditPushbutton() method because the operation     *
//*       might complete while inside the edit. Therefore, we peek at the key  *
//*       input stream and accept the same key input the Edit method would     *
//*       accept: nckENTER, nckSPACE, nckESC or mouse click.                   *
//*                                                                            *
//* Programmer's Note: If the application has enabled the mouse, we can see    *
//* the mouse events here, HOWEVER, they are not translated because we are     *
//* not inside an edit method. Therefore, if a click event occurs over our     *
//* Pushbutton, consider it a keypress.                                        *
//*                                                                            *
//******************************************************************************

short Progbar::UserInteract ( bool* abort )
{
   short opComplete = ERR ;      // return status

   if ( this->isOpen && this->cbPtr != NULL && (this->canBut || this->cloBut) )
   {
      //* Start the monitor thread *
      this->monitor = thread( &Progbar::ProgressMonitor, this ) ;

      //* Interact with the user (treacherous little beasts). *
      wkeyCode wk ;           // user input
      bool cset = false ;     // 'true' if 'Close' text already set
      do
      {
         if ( (opComplete = this->pct()) == 100 )
         {
            //* If specified, change to 'Close' text *
            //* and continue waiting for user input. *
            if ( this->cloBut && ! cset )
            {
               this->dPtr->SetPushbuttonText ( this->canIndex, this->cloText ) ;
               cset = true ;
            }
         }

         //* If user wants to abort *
         if ( (this->dPtr->KeyPeek ( wk )) != wktERR )
         {
            this->dPtr->GetKeyInput ( wk ) ;

            //* If we have a non-ScrollWheel mouse event, be sure *
            //* it belongs to us, and if it does, translate it.   *
            if ( wk.type == wktMOUSE )
            {
               if ( (wk.mevent.ypos > ZERO || wk.mevent.xpos > ZERO) &&
                    (wk.mevent.meType != metSW_D && wk.mevent.meType != metSW_U) )
               {  //* The coordinates have been translated to our dialog.*
                  // Programmer's Note: A 0/0 _MAY BE_ our dialog origin,
                  // but we have no control there, so ignore it.

                  if ( this->horiz )
                  {
                     //* If we're on the same line *
                     if ( wk.mevent.ypos == this->icPtr[this->canIndex].ulY )
                     {
                        short bx = this->icPtr[this->canIndex].ulX,
                              lx = bx + this->icPtr[this->canIndex].cols - 1 ;
                        if ( wk.mevent.xpos >= bx && wk.mevent.xpos <= lx )
                        {
                           wk.type = wktFUNKEY ;
                           wk.key = nckENTER ;
                        }
                     }
                  }
                  else
                  {
                     //* If we're on the same column *
                     if ( wk.mevent.xpos == this->icPtr[this->canIndex].ulX )
                     {
                        short by = this->icPtr[this->canIndex].ulY,
                              ly = by + this->icPtr[this->canIndex].lines - 1 ;
                        if ( wk.mevent.ypos >= by && wk.mevent.ypos <= ly )
                        {
                           wk.type = wktFUNKEY ;
                           wk.key = nckENTER ;
                        }
                     }
                  }
               }
            }
            if (   (wk.type == wktFUNKEY && (wk.key == nckENTER || wk.key == nckESC))
                || (wk.type == wktPRINT && (wk.key == nckSPACE)) )
            {
               opComplete = this->pct() ;
               break ;
            }
         }

         //* If calling thread wants to abort *
         else if ( abort != NULL && *abort != false )
         {
            opComplete = this->pct() ;
            break ;
         }
      }
      while ( (this->curStep < this->barSteps) || (this->cloBut) ) ;

      //* Disable the monitor thread *
      this->monAbort = true ;
   }

   return opComplete ;

}  //* End UserInteract() *

//*************************
//*        update         *
//*************************
//******************************************************************************
//* Move the progress bar forward by one step.                                 *
//*                                                                            *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: percentage of steps completed (integer value: 0 - 100)            *
//******************************************************************************

short Progbar::update ( void )
{
   short pctComplete = ZERO ;

   if ( this->isOpen )
   {
      this->increment() ;
      pctComplete = this->pct() ;
   }
   return pctComplete ;

}  //* End update() *

//*************************
//*        update         *
//*************************
//******************************************************************************
//* Move the progress bar forward or backward by the specified number of steps.*
//*                                                                            *
//* Input  : inc : a) positive value moves forward                             *
//*                b) negative value moves backward                            *
//*                c) zero value clears the bar to ZERO steps                  *
//*                d) any value greater than the remaining number of steps     *
//*                   fills the bar to maximum steps                           *
//*                                                                            *
//* Returns: percentage of steps completed (integer value: 0 - 100)            *
//******************************************************************************

short Progbar::update ( short inc )
{
   short pctComplete = ZERO ;
   if ( this->isOpen )
   {
      if ( inc == ZERO )
         this->clear() ;
      else if ( inc >= (this->barSteps - this->curStep) )
         this->clear( true ) ;
      else if ( inc > ZERO )
      {
         for ( short i = ZERO ; i < inc && this->curStep < this->barSteps ; i++ )
            this->increment() ;
      }
      else
      {
         for ( short i = ZERO ; i > inc && this->curStep > ZERO ; i-- )
            this->decrement() ;
      }
       pctComplete = this->pct() ;
   }
   return pctComplete ;

}  //* End update() *

//*************************
//*        refresh        *
//*************************
//******************************************************************************
//* Refresh the display. Call after updates to the parent window.              *
//*                                                                            *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void Progbar::refresh ( void )
{

   if ( this->isOpen )
      this->dPtr->RefreshWin () ;

}  //* End refresh() *

//*************************
//*        center         *
//*************************
//******************************************************************************
//* PRIVATE METHOD:                                                            *
//* Center the dialog in the terminal window based on the dialog size          *
//* specified by 'dRows' and 'dCols'                                           *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void Progbar::center ( void )
{
   short ydim, xdim, ctry, ctrx ;
   nc.ScreenDimensions ( ydim, xdim ) ;
   ctry = ydim / 2 ;
   ctrx = xdim / 2 ;
   this->ulY = ctry - (this->dRows / 2) ;
   if ( this->ulY < ZERO )
      this->ulY = ZERO ;
   this->ulX = ctrx - (this->dCols / 2) ;
   if ( this->ulX < ZERO )
      this->ulX = ZERO ;

}  //* End center() *

//*************************
//*         clear         *
//*************************
//******************************************************************************
//* PRIVATE METHOD:                                                            *
//* Clear the progress bar.                                                    *
//*                                                                            *
//* Input  : filled : (optional, false by default)                             *
//*                   if false, fill with spaces                               *
//*                   if true,  fill with blocks                               *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void Progbar::clear ( bool filled )
{
   short blki = filled ? (cellDIV) : ZERO ;
   winPos wp = this->barPos ;
   if ( this->horiz )               // horizontal scan
   {
      for ( short i = ZERO ; i < this->barCells ; i++ )
         wp = this->dPtr->WriteChar ( wp, hblock[blki], this->bColor ) ;
   }
   else                             // vertical scan
   {
      for ( short i = ZERO ; i < this->barCells ; i++ )
         this->dPtr->WriteChar ( wp.ypos--, wp.xpos, hblock[blki], this->bColor ) ;
   }
   this->refresh() ;

   //* Reset the step counter, and display info *
   if ( blki == ZERO )
   {
      this->curStep = this->curCell = this->curScan = ZERO ;
   }
   else
   {
      this->curStep = this->barSteps ;
      this->curCell = (this->barCells - 1) ;
      this->curScan = cellDIV ;
   }

}  //* End clear() *

//*************************
//*     initControls      *
//*************************
//******************************************************************************
//* PRIVATE METHOD:                                                            *
//* Setup for optional Pushbutton controls.                                    *
//*                                                                            *
//* NOTE: These calculations are not foolproof. If the application sends us    *
//*       something ugly, we will probably display something ugly.             *
//*                                                                            *
//* Input  : canPtr : pointer to optional text for 'Cancel' button             *
//*          cloPtr : pointer to optional text for 'Close' button              *
//*                                                                            *
//* Returns: nothing : if error, this->parmError is set                        *
//******************************************************************************
//* Expand the dialog dimensions if necessary (and possible).                  *
//* Horiz. Layout: " Cancel " and " Close  " require 8 columns                 *
//* Vert; Layout : " Cancel " and " Close  " require 8 rows                    *
//* If custom text is specified, then the calculation is based on that text.   *
//*                                                                            *
//* Free space required is the width of the text plus one space on either      *
//* end for beauty.                                                            *
//*                                                                            *
//* If label text is specified, both specifications must be the same length.   *
//* (we can pad if necessary, OR refuse to play)                               *
//*                                                                            *
//******************************************************************************

void Progbar::initControls ( const char* canPtr, const char*cloPtr )
{
   //* Default label text *
   const char* dfltCancel = " Cancel " ;
   const char* dfltClose  = " Close  " ;

   //* Allocate memory for the control *
   this->icPtr = new InitCtrl[maxCONTROLS] ;

   //* Define the controls *
   this->canIndex = ZERO ;
   this->icPtr[this->canIndex].type     = dctPUSHBUTTON ;
   this->icPtr[this->canIndex].lines    = 1 ;   // default for horizontal layout
   this->icPtr[this->canIndex].ulY      = ZERO ;
   this->icPtr[this->canIndex].ulX      = ZERO ;
   this->icPtr[this->canIndex].dispText = NULL ;
   this->icPtr[this->canIndex].nColor   = nc.gyR ;
   this->icPtr[this->canIndex].fColor   = nc.gyre | ncbATTR ;
   this->icPtr[this->canIndex].active   = true ;
   this->icPtr[this->canIndex].nextCtrl = NULL ;
   //* NOTE: All other fields are unused for Pushbutton controls.*

   //* Width of text determies width of control.*
   // Programmer's Note: Because non-default text is very likely NOT English,
   // we have to be careful of our char/byte/column counts.
   gString gs ;
   char canTmp[labelBYTES], cloTmp[labelBYTES] ;
   short canWidth, cloWidth ;
   if ( canPtr != NULL )
   {
      gs = canPtr ;
      gs.copy( canTmp, labelBYTES ) ;
      gs = canTmp ;
   }
   else
      gs = dfltCancel ;
   canWidth = gs.gscols() ;
   gs.copy( canTmp, labelBYTES ) ;

   if ( cloPtr != NULL )
   {
      gs = cloPtr ;
      gs.copy( cloTmp, labelBYTES ) ;
      gs = cloTmp ;
   }
   else
      gs = dfltClose ;
   cloWidth = gs.gscols() ;
   gs.copy( cloTmp, labelBYTES ) ;

   //* If strings are not the same length, canWidth controls.*
   if ( cloWidth > canWidth )
   {
      gs = cloTmp ;
      gs.limitCols( canWidth ) ;
      gs.copy(cloTmp, labelBYTES ) ;
   }

   //* Calculate the available space (see notes above) *
   short pWidth = ZERO,
         pNeed = canWidth + 2 ;

   if ( this->horiz )
   {
      pWidth = this->dCols ;
      if ( this->border )
         pWidth -= 2 ;
      if ( *this->begText != nckNULLCHAR || *this->endText != nckNULLCHAR )
         pWidth -= maxLABELCOLS * 2 + 2 ;
      else     // if no labels, we need to expand height to make room
         ++this->dRows ;
   }
   else
   {
      pWidth = this->dRows ;
      if ( this->border )
         pWidth -= 2 ;
      if ( *this->begText != nckNULLCHAR || *this->endText != nckNULLCHAR )
         pWidth -= 2 + 2 ;
   }

   if ( pWidth >= pNeed )
   {

      //* Configure horizontal layout *
      if ( this->horiz )
      {
         gs = canTmp ;
         gs.copy( this->canText, labelBYTES ) ;
         this->icPtr[this->canIndex].dispText = this->canText ;
         this->icPtr[this->canIndex].ulY = this->border ? 2 : 1 ;
         this->icPtr[this->canIndex].cols = canWidth ;
         if ( this->cloBut )
         {
            gs = cloTmp ;
            gs.copy( this->cloText, labelBYTES ) ;
         }

         short ctrX = this->dCols / 2 ;
         this->icPtr[this->canIndex].ulX = ctrX - canWidth / 2 ;
      }

      //* Configure vertical layout *
      else
      {
         //* Format the label control text to fit in a single, vertical column.*
         gs = canTmp ;
         wchar_t wtmp[gsALLOCDFLT] ;
         gs.copy( wtmp, gs.gschars() ) ;
         gs.clear() ;
         for ( short i = ZERO ; wtmp[i] != nckNULLCHAR ; )
         {
            gs.append( wtmp[i++] ) ;
            if ( wtmp[i] != nckNULLCHAR )
               gs.append( L'\n' ) ;
            gs.copy(canTmp, labelBYTES ) ;
            gs = canTmp ;
         }
         gs.copy( this->canText, labelBYTES ) ;
         this->icPtr[this->canIndex].dispText = this->canText ;
         this->icPtr[this->canIndex].lines = canWidth ;
         this->icPtr[this->canIndex].cols = 1 ;
         this->icPtr[this->canIndex].ulX = this->border ? 1 : ZERO ;
         this->icPtr[this->canIndex].ulY = (this->dRows / 2) - (canWidth / 2) ;
         if ( this->cloBut )
         {
            gs = cloTmp ;
            gs.copy( wtmp, gs.gschars() ) ;
            gs.clear() ;
            for ( short i = ZERO ; wtmp[i] != nckNULLCHAR ; )
            {
               gs.append( wtmp[i++] ) ;
               if ( wtmp[i] != nckNULLCHAR )
                  gs.append( L'\n' ) ;
               gs.copy(cloTmp, labelBYTES ) ;
               gs = cloTmp ;
               gs.copy( this->cloText, labelBYTES ) ;
            }
         }

         //* Expand the dialog width if and move the bar to the right *
         //* if necessary. If we have labels, this is already done.   *
         //* Move the bar to the right if necessary.*
         if ( !(*this->begText != nckNULLCHAR || *this->endText != nckNULLCHAR) )
         {
            ++this->dCols ;
            ++this->barPos.xpos ;
         }
      }
   }
   else
   {
      if ( this->icPtr != NULL )
         delete [] this->icPtr ; // release the dynamic allocation
      this->icPtr = NULL ;
      this->canBut = this->cloBut = false ;
      this->cbPtr = NULL ;
      if ( this->barCells < 11 )
         this->barCells = 11 ;
      this->barSteps = this->barCells ;
      this->parmError = true ;
   }

}  //* End initControls() *

//*************************
//*       increment       *
//*************************
//******************************************************************************
//* PRIVATE METHOD:                                                            *
//* Increment the progress bar by one step.                                    *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void Progbar::increment ( void )
{
   if ( this->curStep < this->barSteps )
   {
      #if DEBUG_PBAR != 0        // FOR DEBUGGING ONLY
      this->debugPB( false ) ;
      #endif   // DEBUG_PBAR

      const wchar_t* cArray;

      //* Locate the character cell. wp == y/x index *
      winPos wp = this->barPos ;
      if ( this->horiz )
      {
         wp.xpos += this->curCell ;
         cArray = hblock ;
      }
      else
      {
         wp.ypos -= this->curCell ;
         cArray = vblock ;
      }

      //* Determine index of character to write *
      short cIndex = this->curScan + this->perStep ;

      //* If cell will become filled OR overfilled.*
      if ( cIndex >= cellDIV )
      {
         //* If current cell not full, fill it *
         if ( this->curScan < cellDIV )
         {
            this->dPtr->WriteChar ( wp, cArray[cellDIV], this->bColor, true ) ;
            cIndex -= cellDIV ;
            this->curScan = cellDIV ;
         }
         //* Else, current cell IS full *
         else
            cIndex -= cellDIV ;

         //* If we have remaining scanlines *
         if ( cIndex > ZERO )
         {
            //* If there are unoccupied cells, move forward *
            if ( this->curCell < (this->barCells - 1) )
            {
               ++this->curCell ;
               this->curScan = ZERO ;
               if ( this->horiz )   ++wp.xpos ;
               else                 --wp.ypos ;
            }
            else
            {
               //* If we arrive here, we have a non-integer * 
               //* step, and these scanlines are orphans.   *
               cIndex = ZERO ;
            }
         }
      }

      //* Current cell will be partially filled, just write the character. *
      if ( cIndex > ZERO )
      {
         this->dPtr->WriteChar ( wp, cArray[cIndex], this->bColor, true ) ;
         this->curScan = cIndex ;
      }
      ++this->curStep ;

      #if DEBUG_PBAR != 0        // FOR DEBUGGING ONLY
      this->debugPB( true ) ;
      #endif   // DEBUG_PBAR
   }

}  //* End increment() *

//*************************
//*       decrement       *
//*************************
//******************************************************************************
//* PRIVATE METHOD:                                                            *
//* Decrement the progress bar by one step.                                    *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void Progbar::decrement ( void )
{
   if ( this->curStep > ZERO )
   {
      #if DEBUG_PBAR != 0        // FOR DEBUGGING ONLY
      this->debugPB( false ) ;
      #endif   // DEBUG_PBAR

      const wchar_t* cArray;

      //* Locate the character cell. wp == y/x index *
      winPos wp = this->barPos ;
      if ( this->horiz )
      {
         wp.xpos += this->curCell ;
         cArray = hblock ;
      }
      else
      {
         wp.ypos -= this->curCell ;
         cArray = vblock ;
      }

      //* Determine index of character to write *
      short cIndex = this->curScan - this->perStep ;

      //* If cell will become empty or less than empty.*
      if ( cIndex < ZERO )
      {
         //* If current cell not yet empty, empty it.*
         if ( this->curScan > ZERO )
         {
            this->dPtr->WriteChar ( wp, cArray[ZERO], this->bColor, true ) ;
            cIndex += this->curScan ;
            this->curScan = ZERO ;
         }
         else
         { /* Else, current cell is already empty.*/ }

         //* If we have remaining scanlines *
         if ( cIndex < ZERO )
         {
            //* If there are previous cells, move backward *
            if ( this->curCell > ZERO )
            {
               --this->curCell ;
               this->curScan = cellDIV ;
               cIndex = cellDIV + cIndex ;   // calculate positive index (note sign)
               if ( this->horiz )   --wp.xpos ;
               else                 ++wp.ypos ;
            }
            else
            {
               //* If we arrive here, we have a non-integer * 
               //* step, and these scanlines are orphans.   *
               cIndex = ZERO ;
            }
         }
      }

      //* Current cell is full or partially filled, subtract scanlines.*
      if ( cIndex >= ZERO )
      {
         this->dPtr->WriteChar ( wp, cArray[cIndex], this->bColor, true ) ;
         this->curScan = cIndex ;
      }
      --this->curStep ;

      #if DEBUG_PBAR != 0        // FOR DEBUGGING ONLY
      this->debugPB( true ) ;
      #endif   // DEBUG_PBAR
   }

}  //* End decrement() *

//*************************
//*          pct          *
//*************************
//******************************************************************************
//* PRIVATE METHOD:                                                            *
//* Calculate the percentage completion.                                       *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: percentage of steps completed (integer value: 0 - 100)            *
//******************************************************************************

short Progbar::pct ( void )
{
   short pct = ZERO ;

   if ( this->curStep > ZERO )
   {
      if ( this->curStep == this->barSteps )
         pct = 100 ;
      else
      {
         double bs = this->barSteps,
                cs = this->curStep,
                dpct = cs / bs * 100.0 ;
         // Note: We may want to do rounding here...
         pct = short(dpct) ;
      }
   }
   return pct ;

}  //* End pct() *

//*************************
//*       debugPB         *
//*************************
//******************************************************************************
//* PRIVATE METHOD:                                                            *
//* Write diagnostic data into our parent's window.                            *
//* If parent changes its display layout, we will need to update this method.  *
//*                                                                            *
//* Input  : before/after bar update                                           *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

#if DEBUG_PBAR != 0        // FOR DEBUGGING ONLY
void Progbar::debugPB ( bool after )
{
   if ( this->pdp != NULL )
   {
      gString gs ;
      short y = 11, x = after ? 14 : 2 ;
      if ( ! after )
      {
         gs.compose( "barCells:%3hd\n"
                     "barSteps:%3hd\n" 
                     "perStep :%3hd\n" 
                     "curStep :%3hd\n" 
                     "curCell :%3hd\n" 
                     "curScan :%3hd\n"
                     "                         \n" 
                     "                         \n" 
                     "                         \n", 
                     &this->barCells,
                     &this->barSteps,
                     &this->perStep,
                     &this->curStep,
                     &this->curCell,
                     &this->curScan ) ;
      }
      else
      {
         gs.compose( " :%3hd        \n"
                     " :%3hd        \n" 
                     " :%3hd        \n" 
                     " :%3hd        \n" 
                     " :%3hd        \n" 
                     " :%3hd        \n", 
                     &this->barCells,
                     &this->barSteps,
                     &this->perStep,
                     &this->curStep,
                     &this->curCell,
                     &this->curScan ) ;
      }
      this->pdp->WriteParagraph ( y, x, gs, nc.blR, true ) ;
   }

}  //* End debugPB() *
#endif   // DEBUG_PBAR

//*************************
//*                       *
//*************************
//******************************************************************************
//*                                                                            *
//*                                                                            *
//*                                                                            *
//* Input  :                                                                   *
//*                                                                            *
//* Returns:                                                                   *
//******************************************************************************

