//******************************************************************************
//* File       : PinwheelTest.cpp                                              *
//* Author     : Mahlon R. Smith                                               *
//*              Copyright (c) 2021-2025 Mahlon R. Smith, The Software Samurai *
//*                  GNU GPL copyright notice located in NcDialog.hpp          *
//* Date       : 27-May-2021                                                   *
//* Version    : (see below)                                                   *
//*                                                                            *
//* Description: Class definition for testing of the Pinwheel class.           *
//*              This class provides a dialog window in which all functionality*
//*              of the Pinwheel class can be tested and demonstrated.         *
//*              This test is called as a menu item within the Dialog4         *
//*              application, Test10.                                          *
//*                                                                            *
//* Development Tools: See NcDialog.cpp.                                       *
//******************************************************************************
//* Version History (most recent first):                                       *
//*                                                                            *
//* v: 0.00.01 15-May-2021                                                     *
//*   - Move the Pinwheel test code from ExpandTest.cpp. The development had   *
//*     progressed far enough that it now deserves its own source file.        *
//*                                                                            *
//*                                                                            *
//*                                                                            *
//******************************************************************************

//* For debugging only. Display contents of      *
//* PinwheelInit object whenever the data change.*
#define DEBUG_PINWHEEL_INIT (0)

//****************
//* Header Files *
//****************
#include "PinwheelTest.hpp"

//**********************
//** Local Prototypes **
//**********************


//****************
//** Local Data **
//****************
const short 
   dROWS = 31,             // dialog rows
   dCOLS = 122,            // dialog columns
   statWIDTH = 58,         // width of status-message area
   baseTICK = 250,         // initial tick value
   spinWIDTH = 6,          // width of main spinner control
   aspinWIDTH = 8,         // width of advanced spinner controls
   boxULY  = 9,            // position and size of "advanced" box
   boxULX  = 68,
   boxROWS = dROWS - boxULY - 1,
   boxCOLS = dCOLS - boxULX - 3,
   advY    = boxULY + 2,   // initial position of "advanced" Pinwheel
   advX    = boxULX + 3,
   advMinW = 1,            // minimum width of "advanced" Pinwheel
   advMaxW = 10,           // maximum width of "advanced" Pinwheel
   advMinH = 1,            // minimum height of "advanced" Pinwheel
   advMaxH = 4 ;           // maximum height of "advanced" Pinwheel

//* Uniquely name each dialog control *
enum ptControls : short
{
   ptClosePB = ZERO,       // 'Close' Pushbutton
   ptSpeedSP,              // Speed adjust Spinner
   ptActive1RB,            // Activate Radiobutton (pswWheelC group)
   ptActive2RB,            // Activate Radiobutton (pswWheelA group)
   ptActive3RB,            // Activate Radiobutton (pswVerticalU group)
   ptActive4RB,            // Activate Radiobutton (pswVerticalD group)
   ptActive5RB,            // Activate Radiobutton (pswHorizontalR group)
   ptActive6RB,            // Activate Radiobutton (pswHorizontalL group)
   ptAllRB,                // Activate/deactivate all Pinwheel groups
   ptA_StyleDD,            // Advanced Style Dropdown control
   ptA_ColorDD,            // Advanced Color Dropdown control
   ptA_TickSP,             // Advanced Tick Spinner
   ptA_HeightSP,           // Advanced Height Spinner
   ptA_WidthSP,            // Advanced Width Spinner
   ptA_ActiveRB,           // Advanced Active Radiobutton
   ptA_MultiRB,            // Advanced Multi-color Radiobutton
   ptA_AltposRB,           // Advanced Alt-position Radiobutton
   ptA_ExflagRB,           // Advanced External loop control Radiobutton
   ptA_BidirRB,            // Advanced Bidirectional Radiobutton
   ptA_AudibleRB,          // Advanced Audible Radiobutton
   ptA_ResetRB,            // Advanced: Reset-all-parameters Radiobutton
   ptA_ExplicitRB,         // Advanced Require-explicit-deactivate Radiobutton
   ptDEFINED               // number of controls defined
} ;

//* Data for the style-selection Dropdown *
const short sddITEMS = 6 ;
const short sddHEIGHT = 10 ;
const short sddWIDTH = 7 ;
const char styleDDdata[sddITEMS][sddWIDTH + 1] = 
{  //* Synchronized with enum PinwheelStyle for convenience (keep it that way).*
   "WheelC ",
   "WheelA ",
   "VertiU ",
   "VertiD ",
   "HorizR ",
   "HorizL ",
} ;

//* Data for the color-selection Dropdown *
const short cddITEMS = 8 ;
const short cddHEIGHT = 10 ;
const short cddWIDTH = 7 ;
const char colorDDdata[cddITEMS][cddWIDTH + 1] = 
{
   "Green ",
   " Red  ",
   "Black ",
   "Brown ",
   " Blue ",
  "Magenta",
  " Cyan  ",
  " Grey  ",
} ;

//* Miscellaneous status messages. *
const char* threadActivated = "Pinwheel Activated." ;
const char* threadDeactivated = "Pinwheel Deactivated." ;
//* Message indicating that Pinwheel management-thread *
//* allocation/activation failed.                      *
const char* threadError = "System Error! - Thread allocation failed." ;


//*************************
//*     PinwheelTest      *
//*************************
//******************************************************************************
//* Exercise the options of the Pinwheel class.                                *
//*                                                                            *
//* Input  : wpOrig : (by reference) origin (upper left corner) for test dialog*
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************
//* Programmer's Note: There may be some logical holes in the advanced testing *
//* group of methods. Our tests go far beyond anything the Pinwheel class      *
//* would be asked to do in production code. The test is much more complex     *
//* the code we're testing, but hey, that's the nature of testing.             *
//* The motto of the QA enginer is: "Break the motherf&!%#*r if you can!"      *
//*                                                                            *
//* Note that the selection of color attribute(s) has a profound effect on     *
//* visual appeal of the Pinwheel object. This test defines three arrays of    *
//* color attributes:                                                          *
//* 1) advAttr1u[] : An array of eight complementary attributes to be used     *
//*                  with the pwsWheelC/pwsWheelA/pwsVertiU/pwsHorizR group    *
//*                  of styles. These attributes may be used individually,     *
//*                  or in sequence.                                           *
//* 2) advAttr1d[] : An array of eight complementary attributes to be used     *
//*                  with the pwsVertiD/pwsHorizL group of styles. These       *
//*                  attributes may be used individually, or in sequence.      *
//* 3) advAttr8[]  : An array of eight attributes to be used in sequence       *
//*                  (multi-color display). This array is designed so that     *
//*                  each attribute blends smoothly with the next for a better *
//*                  visual flow.                                              *
//*                                                                            *
//* Please experiment freely with color attributes and color sequences in your *
//* own applications to obtain an optimal visual effect.                       *
//*                                                                            *
//******************************************************************************

PinwheelTest::PinwheelTest ( const winPos& wpOrig )
{
   //* Initialize our data members *
   this->dColor   = nc.blR ;                 // dialog base color
   this->tColor   = this->dColor | ncbATTR ; // dialog title color
   this->bColor   = nc.gybl | ncbATTR ;      // dialog border color
   this->hColor   = nc.brbl | ncbATTR ;      // dialog highlight color
   this->aColor   = nc.grbl ;                // advanced-testing box color
   this->msgAttr  = nc.bgrbk ;               // advanced-testing message color
   this->advAttr  = nc.grbk ;                // advanced-testing-area theme color
   this->AdvancedAttrInit () ;               // advanced-testing color arrays
   this->objWidth  = 1 ;                     // width of object (columns)
   this->objHeight = 1 ;                     // height of object (rows)
   this->advFlag   = false ;                 // reset "advanced" loop-control flag
   this->dlgOpen   = false ;                 // dialog is not yet open
   //* The 'icptr' member will be initialized below.        *
   //* The 'pw' and 'wp' members will be initialized below. *
   //* The 'msgPos' member will be initialized below.       *
   //* The 'advInit' member will be initialized below.      *
   //* The 'amsgPos' member will be initialized below.      *

   //* Color scheme for 'Style' and 'Color' dctDROPDOWN controls *
   attr_t monoColor[2] = { attrDFLT, this->advAttr } ;

   attr_t spinFattr = nc.bkgr ;        // spinner focus color
   attr_t dimAttr = nc.gybl ;          // dim-text color
   gString gs ;                        // text formatting
   int  iValue ;                       // receives raw value from Spinners
   short ddIndex,                      // receives index of selection in Dropdowns
         prevColorDD ;                 // previous index of ptA_ColorDD
   bool rbFlag,                        // receives state of Radiobutton flag
        forceUpdate ;                  // used as 'forceOff' parm to AdvancedSetup()

   //* Setup information for tick-interval spinner controls *
   dspinData tsData(minTick, maxTick, baseTICK, dspinINTEGER, nc.brgr ) ;

   //* Setup information for "advanced" Width spinner control *
   dspinData wsData(advMinW, advMaxW, advMinW, dspinINTEGER, nc.brgr ) ;

   //* Setup information for "advanced" Height spinner control *
   dspinData hsData(advMinH, advMaxH, advMinH, dspinINTEGER, nc.brgr ) ;

   InitCtrl ic[ptDEFINED] =            // control initialization structures
   {
      {  //* 'CLOSE' pushbutton  - - - - - - - - - - - - - - - - - - ptClosePB *
         dctPUSHBUTTON,                // type:      
         rbtTYPES,                     // rbSubtype: (n/a)
         false,                        // rbSelect:  (n/a)
         2,                            // ulY:       upper left corner in Y
         4,                            // ulX:       upper left corner in X
         1,                            // lines:     control lines
         9,                            // cols:      control columns
         "  ^CLOSE  ",                 // dispText:  
         nc.gyR,                       // nColor:    non-focus color
         nc.reG,                       // fColor:    focus color
         tbPrint,                      // filter:    (n/a)
         "",                           // label:     (n/a)
         ZERO,                         // labY:      (n/a)
         ZERO,                         // labX       (n/a)
         ddBoxTYPES,                   // exType:    (n/a)
         1,                            // scrItems:  (n/a)
         1,                            // scrSel:    (n/a)
         NULL,                         // scrColor:  (n/a)
         NULL,                         // spinData:  (n/a)
         true,                         // active:    allow control to gain focus
         &ic[ptSpeedSP]                // nextCtrl:  link in next structure
      },
      {  //* 'SPEED' spinner - - - - - - - - - - - - - - - - - - - - ptSpeedSP *
         dctSPINNER,                   // type:      
         rbtTYPES,                     // rbSubtype: (n/a)
         false,                        // rbSelect:  (n/a)
         ic[ptClosePB].ulY,            // ulY:       upper left corner in Y
         short(ic[ptClosePB].ulX + 13),// ulX:    upper left corner in X
         1,                            // lines:     (n/a)
         spinWIDTH,                    // cols:      control columns
         NULL,                         // dispText:  (n/a)
         nc.gyR,                       // nColor:    non-focus color
         spinFattr,                    // fColor:    focus color
         tbPrint,                      // filter:    (n/a)
         "^TICK SPEED\n        ms",    // label:     
         -1,                           // labY:      
         -2,                           // labX       
         ddBoxTYPES,                   // exType:    (n/a)
         1,                            // scrItems:  (n/a)
         1,                            // scrSel:    (n/a)
         NULL,                         // scrColor:  (n/a)
         &tsData,                      // spinData:  spinner init
         true,                         // active:    allow control to gain focus
         &ic[ptActive1RB]              // nextCtrl:  link in next structure
      },
      {  //* ACTIVE 1 radiobutton  - - - - - - - - - - - - - - -   ptActive1RB *
         dctRADIOBUTTON,               // type:      
         rbtS3a,                       // rbSubtype: angle, 3 chars wide
         true,                         // rbSelect:  default selection
         4,                            // ulY:       upper left corner in Y
         61,                           // ulX:       upper left corner in X
         1,                            // lines:     (n/a)
         0,                            // cols:      (n/a)
         NULL,                         // dispText:  (n/a)
         nc.gyR,                       // nColor:    non-focus color
         nc.grG,                       // fColor:    focus color
         tbPrint,                      // filter:    (n/a)
         "ACTIVE",                     // label:
         -1,                           // labY:      
         -1,                           // labX       
         ddBoxTYPES,                   // exType:    (n/a)
         ZERO,                         // scrItems:  (n/a)
         ZERO,                         // scrSel:    (n/a)
         NULL,                         // scrColor:  (n/a)
         NULL,                         // spinData:  (n/a)
         true,                         // active:    allow control to gain focus
         &ic[ptActive2RB]              // nextCtrl:  link in next structure
      },
      {  //* ACTIVE 2 radiobutton  - - - - - - - - - - - - - - -   ptActive2RB *
         dctRADIOBUTTON,               // type:      
         rbtS3a,                       // rbSubtype: angle, 3 chars wide
         true,                         // rbSelect:  default selection
         short(ic[ptActive1RB].ulY + 4),// ulY:       upper left corner in Y
         ic[ptActive1RB].ulX,          // ulX:       upper left corner in X
         1,                            // lines:     (n/a)
         0,                            // cols:      (n/a)
         NULL,                         // dispText:  (n/a)
         nc.gyR,                       // nColor:    non-focus color
         nc.grG,                       // fColor:    focus color
         tbPrint,                      // filter:    (n/a)
         NULL,                         // label:
         ZERO,                         // labY:      
         ZERO,                         // labX       
         ddBoxTYPES,                   // exType:    (n/a)
         ZERO,                         // scrItems:  (n/a)
         ZERO,                         // scrSel:    (n/a)
         NULL,                         // scrColor:  (n/a)
         NULL,                         // spinData:  (n/a)
         true,                         // active:    allow control to gain focus
         &ic[ptActive3RB]              // nextCtrl:  link in next structure
      },
      {  //* ACTIVE 3 radiobutton  - - - - - - - - - - - - - - -   ptActive3RB *
         dctRADIOBUTTON,               // type:      
         rbtS3a,                       // rbSubtype: angle, 3 chars wide
         true,                         // rbSelect:  default selection
         short(ic[ptActive2RB].ulY + 4),// ulY:       upper left corner in Y
         ic[ptActive2RB].ulX,          // ulX:       upper left corner in X
         1,                            // lines:     (n/a)
         0,                            // cols:      (n/a)
         NULL,                         // dispText:  (n/a)
         nc.gyR,                       // nColor:    non-focus color
         nc.grG,                       // fColor:    focus color
         tbPrint,                      // filter:    (n/a)
         NULL,                         // label:
         ZERO,                         // labY:      
         ZERO,                         // labX       
         ddBoxTYPES,                   // exType:    (n/a)
         ZERO,                         // scrItems:  (n/a)
         ZERO,                         // scrSel:    (n/a)
         NULL,                         // scrColor:  (n/a)
         NULL,                         // spinData:  (n/a)
         true,                         // active:    allow control to gain focus
         &ic[ptActive4RB]              // nextCtrl:  link in next structure
      },
      {  //* ACTIVE 4 radiobutton  - - - - - - - - - - - - - - -   ptActive4RB *
         dctRADIOBUTTON,               // type:      
         rbtS3a,                       // rbSubtype: angle, 3 chars wide
         true,                         // rbSelect:  default selection
         short(ic[ptActive3RB].ulY + 4),// ulY:       upper left corner in Y
         ic[ptActive3RB].ulX,          // ulX:       upper left corner in X
         1,                            // lines:     (n/a)
         0,                            // cols:      (n/a)
         NULL,                         // dispText:  (n/a)
         nc.gyR,                       // nColor:    non-focus color
         nc.grG,                       // fColor:    focus color
         tbPrint,                      // filter:    (n/a)
         NULL,                         // label:
         ZERO,                         // labY:      
         ZERO,                         // labX       
         ddBoxTYPES,                   // exType:    (n/a)
         ZERO,                         // scrItems:  (n/a)
         ZERO,                         // scrSel:    (n/a)
         NULL,                         // scrColor:  (n/a)
         NULL,                         // spinData:  (n/a)
         true,                         // active:    allow control to gain focus
         &ic[ptActive5RB]              // nextCtrl:  link in next structure
      },
      {  //* ACTIVE 5 radiobutton  - - - - - - - - - - - - - - -   ptActive5RB *
         dctRADIOBUTTON,               // type:      
         rbtS3a,                       // rbSubtype: angle, 3 chars wide
         true,                         // rbSelect:  default selection
         short(ic[ptActive4RB].ulY + 4),// ulY:       upper left corner in Y
         ic[ptActive4RB].ulX,          // ulX:       upper left corner in X
         1,                            // lines:     (n/a)
         0,                            // cols:      (n/a)
         NULL,                         // dispText:  (n/a)
         nc.gyR,                       // nColor:    non-focus color
         nc.grG,                       // fColor:    focus color
         tbPrint,                      // filter:    (n/a)
         NULL,                         // label:
         ZERO,                         // labY:      
         ZERO,                         // labX       
         ddBoxTYPES,                   // exType:    (n/a)
         ZERO,                         // scrItems:  (n/a)
         ZERO,                         // scrSel:    (n/a)
         NULL,                         // scrColor:  (n/a)
         NULL,                         // spinData:  (n/a)
         true,                         // active:    allow control to gain focus
         &ic[ptActive6RB]              // nextCtrl:  link in next structure
      },
      {  //* ACTIVE 6 radiobutton  - - - - - - - - - - - - - - -   ptActive6RB *
         dctRADIOBUTTON,               // type:      
         rbtS3a,                       // rbSubtype: angle, 3 chars wide
         true,                         // rbSelect:  default selection
         short(ic[ptActive5RB].ulY + 4),// ulY:       upper left corner in Y
         ic[ptActive5RB].ulX,          // ulX:       upper left corner in X
         1,                            // lines:     (n/a)
         0,                            // cols:      (n/a)
         NULL,                         // dispText:  (n/a)
         nc.gyR,                       // nColor:    non-focus color
         nc.grG,                       // fColor:    focus color
         tbPrint,                      // filter:    (n/a)
         NULL,                         // label:
         ZERO,                         // labY:      
         ZERO,                         // labX       
         ddBoxTYPES,                   // exType:    (n/a)
         ZERO,                         // scrItems:  (n/a)
         ZERO,                         // scrSel:    (n/a)
         NULL,                         // scrColor:  (n/a)
         NULL,                         // spinData:  (n/a)
         true,                         // active:    allow control to gain focus
         &ic[ptAllRB]                  // nextCtrl:  link in next structure
      },
      {  //* ALL radiobutton  - - - - - - - - - - - - - - - - - - -    ptAllRB *
         dctRADIOBUTTON,               // type:      
         rbtS3a,                       // rbSubtype: angle, 3 chars wide
         true,                         // rbSelect:  default selection
         short(ic[ptActive6RB].ulY + 3),// ulY:       upper left corner in Y
         ic[ptActive6RB].ulX,          // ulX:       upper left corner in X
         1,                            // lines:     (n/a)
         0,                            // cols:      (n/a)
         NULL,                         // dispText:  (n/a)
         nc.gyR,                       // nColor:    non-focus color
         nc.grG,                       // fColor:    focus color
         tbPrint,                      // filter:    (n/a)
         "ALL ^ACTIVE",                // label:
         ZERO,                         // labY:      
         -11,                          // labX       
         ddBoxTYPES,                   // exType:    (n/a)
         ZERO,                         // scrItems:  (n/a)
         ZERO,                         // scrSel:    (n/a)
         NULL,                         // scrColor:  (n/a)
         NULL,                         // spinData:  (n/a)
         true,                         // active:    allow control to gain focus
         &ic[ptA_StyleDD]              // nextCtrl:  link in next structure
      },
      { //* 'STYLE SELECT' Dropdown - - - - - - - - - - - - - - - -  ptStyleDD *
         dctDROPDOWN,                  // type:      define a scrolling-data control
         rbtTYPES,                     // rbSubtype: (na)
         false,                        // rbSelect:  (n/a)
         short(boxULY + 1),            // ulY:       upper left corner in Y
         short(boxULX + 22),           // ulX:       upper left corner in X
         short(sddITEMS + 2),          // lines:     control lines
         short(sddWIDTH + 2),          // cols:      control columns
         (const char*)styleDDdata,     // dispText:  items displayed
         nc.gybl,                      // nColor:    non-focus border color
         nc.rebl,                      // fColor:    focus border color
         tbPrint,                      // filter:    (n/a)
         "Set ^Style",                 // label:     
         1,                            // labY:      offset from control's ulY
         10,                           // labX       offset from control's ulX
         ddBoxDOWN,                    // exType:    expands downward
         sddITEMS,                     // scrItems:  number of elements in text/color arrays
         ZERO,                         // scrSel:    index of initial highlighted element
         monoColor,                    // scrColor:  single-color data display
         NULL,                         // spinData:  (n/a)
         true,                         // active:    allow control to gain focus
         &ic[ptA_ColorDD]               // nextCtrl:  link in next structure
      },
      { //* 'COLOR SELECT' Dropdown - - - - - - - - - - - - - - - -  ptColorDD *
         dctDROPDOWN,                  // type:      define a scrolling-data control
         rbtTYPES,                     // rbSubtype: (na)
         false,                        // rbSelect:  (n/a)
         short(ic[ptA_StyleDD].ulY + 3),// ulY:       upper left corner in Y
         ic[ptA_StyleDD].ulX,          // ulX: upper left corner in X
         short(cddITEMS + 2),          // lines:     control lines
         short(cddWIDTH + 2),          // cols:      control columns
         (const char*)colorDDdata,     // dispText:  items displayed
         nc.gybl,                      // nColor:    non-focus border color
         nc.rebl,                      // fColor:    focus border color
         tbPrint,                      // filter:    (n/a)
         "Set (single) Color",         // label:     
         1,                            // labY:      offset from control's ulY
         10,                           // labX       offset from control's ulX
         ddBoxDOWN,                    // exType:    (n/a)
         cddITEMS,                     // scrItems:  number of elements in text/color arrays
         ZERO,                         // scrSel:    index of initial highlighted element
         monoColor,                    // scrColor:  single-color data display
         NULL,                         // spinData:  (n/a)
         true,                         // active:    allow control to gain focus
         &ic[ptA_TickSP]               // nextCtrl:  link in next structure
      },
      {  //* 'ATICK' spinner - - - - - - - - - - - - - - - - - - -  ptA_TickSP *
         dctSPINNER,                   // type:      
         rbtTYPES,                     // rbSubtype: (n/a)
         false,                        // rbSelect:  (n/a)
         short(ic[ptA_ColorDD].ulY + 4), // ulY:       upper left corner in Y
         short(ic[ptA_ColorDD].ulX + 1), // ulX:    upper left corner in X
         1,                            // lines:     (n/a)
         aspinWIDTH,                   // cols:      control columns
         NULL,                         // dispText:  (n/a)
         this->advAttr,                // nColor:    non-focus color
         spinFattr,                    // fColor:    focus color
         tbPrint,                      // filter:    (n/a)
         "Tick Speed",                 // label:     
         ZERO,                         // labY:      
         9,                            // labX       
         ddBoxTYPES,                   // exType:    (n/a)
         1,                            // scrItems:  (n/a)
         1,                            // scrSel:    (n/a)
         NULL,                         // scrColor:  (n/a)
         &tsData,                      // spinData:  spinner init
         true,                         // active:    allow control to gain focus
         &ic[ptA_HeightSP]             // nextCtrl:  link in next structure
      },
      {  //* 'AHEIGHT' spinner   - - - - - - - - - - - - - - - -  ptA_HeightSP *
         dctSPINNER,                   // type:      
         rbtTYPES,                     // rbSubtype: (n/a)
         false,                        // rbSelect:  (n/a)
         short(ic[ptA_TickSP].ulY + 2),// ulY:       upper left corner in Y
         ic[ptA_TickSP].ulX,           // ulX:       upper left corner in X
         1,                            // lines:     (n/a)
         aspinWIDTH,                   // cols:      control columns
         NULL,                         // dispText:  (n/a)
         this->advAttr,                // nColor:    non-focus color
         nc.bbkgr,                     // fColor:    focus color
         tbPrint,                      // filter:    (n/a)
         "Set Height (rows)",          // label:     
         ZERO,                         // labY:      
         9,                            // labX       
         ddBoxTYPES,                   // exType:    (n/a)
         1,                            // scrItems:  (n/a)
         1,                            // scrSel:    (n/a)
         NULL,                         // scrColor:  (n/a)
         &hsData,                      // spinData:  spinner init
         true,                         // active:    allow control to gain focus
         &ic[ptA_WidthSP]              // nextCtrl:  link in next structure
      },
      {  //* 'AWIDTH' spinner  - - - - - - - - - - - - - - - - -   ptA_WidthSP *
         dctSPINNER,                   // type:      
         rbtTYPES,                     // rbSubtype: (n/a)
         false,                        // rbSelect:  (n/a)
         short(ic[ptA_HeightSP].ulY + 2),// ulY:       upper left corner in Y
         ic[ptA_HeightSP].ulX,         // ulX:    upper left corner in X
         1,                            // lines:     (n/a)
         aspinWIDTH,                   // cols:      control columns
         NULL,                         // dispText:  (n/a)
         this->advAttr,                // nColor:    non-focus color
         spinFattr,                    // fColor:    focus color
         tbPrint,                      // filter:    (n/a)
         "Set Width (cols)",           // label:     
         ZERO,                         // labY:      
         9,                            // labX       
         ddBoxTYPES,                   // exType:    (n/a)
         1,                            // scrItems:  (n/a)
         1,                            // scrSel:    (n/a)
         NULL,                         // scrColor:  (n/a)
         &wsData,                      // spinData:  spinner init
         true,                         // active:    allow control to gain focus
         &ic[ptA_ActiveRB]             // nextCtrl:  link in next structure
      },
      {  //* A_ACTIVE radiobutton  - - - - - - - - - - - - - - -  ptA_ActiveRB *
         dctRADIOBUTTON,               // type:      
         rbtS3s,                       // rbSubtype: square, 3 chars wide
         true,                         // rbSelect:  default selection
         short(boxULY + 8),            // ulY:       upper left corner in Y
         short(boxULX + 2),            // ulX:       upper left corner in X
         1,                            // lines:     (n/a)
         0,                            // cols:      (n/a)
         NULL,                         // dispText:  (n/a)
         this->advAttr,                // nColor:    non-focus color
         nc.bbkgr,                     // fColor:    focus color
         tbPrint,                      // filter:    (n/a)
         "Acti^ve",                    // label:
         ZERO,                         // labY:      
         4,                            // labX       
         ddBoxTYPES,                   // exType:    (n/a)
         ZERO,                         // scrItems:  (n/a)
         ZERO,                         // scrSel:    (n/a)
         NULL,                         // scrColor:  (n/a)
         NULL,                         // spinData:  (n/a)
         true,                         // active:    allow control to gain focus
         &ic[ptA_MultiRB]              // nextCtrl:  link in next structure
      },
      {  //* A_MULTI radiobutton  - - - - - - - - - - - - - - - -  ptA_MultiRB *
         dctRADIOBUTTON,               // type:      
         rbtS3s,                       // rbSubtype: square, 3 chars wide
         false,                        // rbSelect:  default selection
         short(ic[ptA_ActiveRB].ulY + 1), // ulY:       upper left corner in Y
         ic[ptA_ActiveRB].ulX,         // ulX:       upper left corner in X
         1,                            // lines:     (n/a)
         0,                            // cols:      (n/a)
         NULL,                         // dispText:  (n/a)
         this->advAttr,                // nColor:    non-focus color
         nc.bbkgr,                     // fColor:    focus color
         tbPrint,                      // filter:    (n/a)
         "Multi-color",                // label:
         ZERO,                         // labY:      
         4,                            // labX       
         ddBoxTYPES,                   // exType:    (n/a)
         ZERO,                         // scrItems:  (n/a)
         ZERO,                         // scrSel:    (n/a)
         NULL,                         // scrColor:  (n/a)
         NULL,                         // spinData:  (n/a)
         true,                         // active:    allow control to gain focus
         &ic[ptA_AltposRB]             // nextCtrl:  link in next structure
      },
      {  //* A_ALTPOS radiobutton  - - - - - - - - - - - - - - -  ptA_AltposRB *
         dctRADIOBUTTON,               // type:      
         rbtS3s,                       // rbSubtype: square, 3 chars wide
         false,                        // rbSelect:  default selection
         short(ic[ptA_MultiRB].ulY + 1), // ulY:       upper left corner in Y
         ic[ptA_MultiRB].ulX,          // ulX:       upper left corner in X
         1,                            // lines:     (n/a)
         0,                            // cols:      (n/a)
         NULL,                         // dispText:  (n/a)
         this->advAttr,                // nColor:    non-focus color
         nc.bbkgr,                     // fColor:    focus color
         tbPrint,                      // filter:    (n/a)
         "Alt Position",               // label:
         ZERO,                         // labY:      
         4,                            // labX       
         ddBoxTYPES,                   // exType:    (n/a)
         ZERO,                         // scrItems:  (n/a)
         ZERO,                         // scrSel:    (n/a)
         NULL,                         // scrColor:  (n/a)
         NULL,                         // spinData:  (n/a)
         true,                         // active:    allow control to gain focus
         &ic[ptA_ExflagRB]             // nextCtrl:  link in next structure
      },
      {  //* A_EXFLAG radiobutton  - - - - - - - - - - - - - - -  ptA_ExflagRB *
         dctRADIOBUTTON,               // type:      
         rbtS3s,                       // rbSubtype: square, 3 chars wide
         true,                         // rbSelect:  default selection
         short(ic[ptA_AltposRB].ulY + 1), // ulY:    upper left corner in Y
         ic[ptA_AltposRB].ulX,         // ulX:       upper left corner in X
         1,                            // lines:     (n/a)
         0,                            // cols:      (n/a)
         NULL,                         // dispText:  (n/a)
         this->advAttr,                // nColor:    non-focus color
         nc.bbkgr,                     // fColor:    focus color
         tbPrint,                      // filter:    (n/a)
         "External Flag",              // label:
         ZERO,                         // labY:      
         4,                            // labX       
         ddBoxTYPES,                   // exType:    (n/a)
         ZERO,                         // scrItems:  (n/a)
         ZERO,                         // scrSel:    (n/a)
         NULL,                         // scrColor:  (n/a)
         NULL,                         // spinData:  (n/a)
         true,                         // active:    allow control to gain focus
         &ic[ptA_BidirRB]              // nextCtrl:  link in next structure
      },
      {  //* A_BIDIR radiobutton  - - - - - - - - - - - - - - - -  ptA_BidirRB *
         dctRADIOBUTTON,               // type:      
         rbtS3s,                       // rbSubtype: square, 3 chars wide
         false,                        // rbSelect:  default selection
         short(ic[ptA_ExflagRB].ulY + 1), // ulY:       upper left corner in Y
         ic[ptA_ExflagRB].ulX,         // ulX:       upper left corner in X
         1,                            // lines:     (n/a)
         0,                            // cols:      (n/a)
         NULL,                         // dispText:  (n/a)
         this->advAttr,                // nColor:    non-focus color
         nc.bbkgr,                     // fColor:    focus color
         tbPrint,                      // filter:    (n/a)
         "Bidirectional",              // label:
         ZERO,                         // labY:      
         4,                            // labX       
         ddBoxTYPES,                   // exType:    (n/a)
         ZERO,                         // scrItems:  (n/a)
         ZERO,                         // scrSel:    (n/a)
         NULL,                         // scrColor:  (n/a)
         NULL,                         // spinData:  (n/a)
         true,                         // active:    allow control to gain focus
         &ic[ptA_AudibleRB]            // nextCtrl:  link in next structure
      },
      {  //* A_AUDIBLE radiobutton  - - - - - - - - - - - - - -  ptA_AudibleRB *
         dctRADIOBUTTON,               // type:      
         rbtS3s,                       // rbSubtype: square, 3 chars wide
         false,                        // rbSelect:  default selection
         short(ic[ptA_BidirRB].ulY + 1), // ulY:       upper left corner in Y
         ic[ptA_BidirRB].ulX,          // ulX:       upper left corner in X
         1,                            // lines:     (n/a)
         0,                            // cols:      (n/a)
         NULL,                         // dispText:  (n/a)
         this->advAttr,                // nColor:    non-focus color
         nc.bbkgr,                     // fColor:    focus color
         tbPrint,                      // filter:    (n/a)
         "Audible",                    // label:
         ZERO,                         // labY:      
         4,                            // labX       
         ddBoxTYPES,                   // exType:    (n/a)
         ZERO,                         // scrItems:  (n/a)
         ZERO,                         // scrSel:    (n/a)
         NULL,                         // scrColor:  (n/a)
         NULL,                         // spinData:  (n/a)
         true,                         // active:    allow control to gain focus
         &ic[ptA_ResetRB]              // nextCtrl:  link in next structure
      },
      {  //* A_RESET radiobutton  - - - - - - - - - - - - - - - -  ptA_ResetRB *
         dctRADIOBUTTON,               // type:      
         rbtS3s,                       // rbSubtype: square, 3 chars wide
         false,                        // rbSelect:  default selection
         short(ic[ptA_AudibleRB].ulY + 1), // ulY:       upper left corner in Y
         ic[ptA_AudibleRB].ulX,          // ulX:       upper left corner in X
         1,                            // lines:     (n/a)
         0,                            // cols:      (n/a)
         NULL,                         // dispText:  (n/a)
         this->advAttr,                // nColor:    non-focus color
         nc.bbkgr,                     // fColor:    focus color
         tbPrint,                      // filter:    (n/a)
         "Reset All",                  // label:
         ZERO,                         // labY:      
         4,                            // labX       
         ddBoxTYPES,                   // exType:    (n/a)
         ZERO,                         // scrItems:  (n/a)
         ZERO,                         // scrSel:    (n/a)
         NULL,                         // scrColor:  (n/a)
         NULL,                         // spinData:  (n/a)
         true,                         // active:    allow control to gain focus
         &ic[ptA_ExplicitRB]           // nextCtrl:  link in next structure
      },
      {  //* A_EXPLICIT radiobutton  - - - - - - - - - - - - -  ptA_ExplicitRB *
         dctRADIOBUTTON,               // type:      
         rbtS3s,                       // rbSubtype: square, 3 chars wide
         false,                        // rbSelect:  default selection
         short(ic[ptA_ResetRB].ulY + 1), // ulY:       upper left corner in Y
         ic[ptA_ResetRB].ulX,          // ulX:       upper left corner in X
         1,                            // lines:     (n/a)
         0,                            // cols:      (n/a)
         NULL,                         // dispText:  (n/a)
         this->advAttr,                // nColor:    non-focus color
         nc.bbkgr,                     // fColor:    focus color
         tbPrint,                      // filter:    (n/a)
         "^Require Explicit Deactivate Before Update",  // label:
         ZERO,                         // labY:      
         4,                            // labX       
         ddBoxTYPES,                   // exType:    (n/a)
         ZERO,                         // scrItems:  (n/a)
         ZERO,                         // scrSel:    (n/a)
         NULL,                         // scrColor:  (n/a)
         NULL,                         // spinData:  (n/a)
         true,                         // active:    allow control to gain focus
         NULL                          // nextCtrl:  link in next structure
      },
   } ;

   InitNcDialog dInit( dROWS,          // number of display lines
                       dCOLS,          // number of display columns
                       wpOrig.ypos,    // Y offset from upper-left of terminal 
                       wpOrig.xpos,    // X offset from upper-left of terminal 
                       NULL,           // dialog title
                       ncltDUAL,       // border line-style
                       this->hColor,   // border color attribute
                       this->dColor,   // interior color attribute
                       ic              // pointer to list of control definitions
                     ) ;

   //* Instantiate the dialog window *
   this->dp = new NcDialog ( dInit ) ;

   //* Open the dialog window *
   if ( (this->dlgOpen = bool(this->dp->OpenWindow()) == OK) )
   {
      //* Set dialog title *
      this->dp->SetDialogTitle ( "  Pinwheel-class Styles and Functionality  ",
                                 this->tColor ) ;

      //* Initialize the 'icptr' member. Access to InitCtrl array. *
      this->icptr = ic ;
      //* Initialize stat-message position *
      this->msgPos = { short(ic[ptAllRB].ulY + 1), 6 } ;
      //* Set the message position for the "advanced" area *
      this->amsgPos = { short(ic[ptA_ExplicitRB].ulY + 1), ic[ptA_ExplicitRB].ulX } ;
      //* Initialize the 'forceUpdate' flag. This acts as the *
      //* 'forceOff' argument for calls to AdvancedSetup().   *
      this->dp->GetRadiobuttonState ( ptA_ExplicitRB, rbFlag ) ;
      forceUpdate = rbFlag ? false : true ;
      //* Backup copy of ptA_ColorDD Dropdown control index *
      prevColorDD = this->dp->GetDropdownSelect ( ptA_ColorDD ) ;

      //* Write the dialog's static text *
      winPos wpt( 4, 6 ),     // cursor positioning for text
             wps( 4, 33 ),
             wptmp( short(wps.ypos - 2), short(wps.xpos - 3) ) ;
      //* Column headings *
      wptmp = this->dp->WriteParagraph ( wptmp,
                             " Single  Multi   Auto\n", this->hColor ) ;
              this->dp->WriteParagraph ( wptmp,
                             "  Color  Color   Reverse \n", (this->hColor | ncuATTR) ) ;
      //* Write instructions *
      wptmp = { ic[ptClosePB].ulY, short(ic[ptActive1RB].ulX + 8) } ;
      wptmp = this->dp->WriteParagraph ( wptmp,
         "Use the \"Tick Speed\" spinner to adjust execution\n"
         "speed for the objects. \"Tick Speed\" is the display\n"
         "update rate for each object (in milliseconds).\n"
         "Use the \"Active\" radiobuttons to enable or disable\n"
         "the objects of that row. This will reduce the\n"
         "inter-thread latency among active Pinwheel objects.\n\n",
         this->dColor ) ;

      //* Define the advanced-options enclosure *
      this->dp->DrawBox ( boxULY, boxULX, boxROWS, boxCOLS, this->aColor,
                          "  Advanced Testing  " ) ;
      this->dp->DrawBox ( boxULY + 1, boxULX + 2, 6, 12, this->aColor ) ;

      this->dp->WriteParagraph ( this->amsgPos.ypos + 1, this->amsgPos.xpos + 1,
         "1) Clear the \"Active\" radiobutton.\n"
         "2) Make desired changes to setup parameters.\n"
         "3) Set the \"Active\" radiobutton.",
         this->aColor ) ;

      //* Calculate the position of each Pinwheel object *
      //* in the main group, then instantiate and launch *
      //* all Pinwheel objects in the main group.        *
      this->Setup ( wpt, baseTICK ) ;

      //* All members of the 'advInit' object are initialized to default *
      //* values. Instantiate and launch the "advanced" Pinwheel object. *
      //* Note: Be sure these values match control settings.             *
      this->AdvancedInit () ;
      Pinwheel advPw( this->advInit ) ;
      if ( !(advPw.launch()) )
      {
         //* Reset the Radiobutton state *
         this->dp->SetRadiobuttonState ( ptA_ActiveRB, false ) ;
         //* If system error, alert user *
         this->AdvancedMsg ( threadError ) ;
      }

      this->dp->RefreshWin () ;     // make everything visible


      //****************************************
      //* Interact with user (filthy beasts :) *
      //****************************************
      uiInfo Info ;                 // user interface data returned here
      short  icIndex = ZERO ;       // index of control with input focus
      bool   done = false ;         // loop control
      while ( ! done )
      {
         if ( ic[icIndex].type == dctPUSHBUTTON )
         {
            if ( Info.viaHotkey != false )
               Info.HotData2Primary () ;
            else
               icIndex = this->dp->EditPushbutton ( Info ) ;
            if ( Info.dataMod != false )
            {
               this->statMsg ( "Shutting down the execution threads, please wait... ", false ) ;
               done = true ;
            }
         }

         else if ( ic[icIndex].type == dctDROPDOWN )
         {
            icIndex = this->dp->EditDropdown ( Info ) ;

            //* If user has made a selection, modify the *
            //* advanced Pinwheel object's parameters.   *
            if ( Info.dataMod )
            {
               //* Because this could be a slow operation, *
               //* first make the new setting visible.     *
               icIndex = this->dp->NextControl () ;
               icIndex = this->dp->PrevControl () ;

               //* Get a copy of the current Pinwheel configuration *
               advPw.getParms ( this->advInit ) ;

               //*****************
               //** ptA_StyleDD **
               //*****************
               if ( icIndex == ptA_StyleDD )
               {
                  //* Set the new style *
                  this->advInit.dispStyle = PinwheelStyle(Info.selMember) ;

                  //* Calculate the color attribute(s) for this style *
                  ddIndex = this->dp->GetDropdownSelect ( ptA_ColorDD ) ;
                  //* If single color display *
                  if ( this->advInit.colorCount == 1 )
                  {
                     if ( (this->advInit.dispStyle == pwsVertiD) || 
                          (this->advInit.dispStyle == pwsHorizL) )
                     { this->advInit.colorPtr = &this->advAttr1d[ddIndex] ; }
                     else
                        this->advInit.colorPtr = &this->advAttr1u[ddIndex] ;
                  }
                  //* Else, multi-color display *
                  else
                  {
                     if ( (this->advInit.dispStyle == pwsWheelC) || 
                          (this->advInit.dispStyle == pwsWheelA) )
                     { this->advInit.colorPtr = this->advAttr1u ; }
                     else
                        this->advInit.colorPtr = this->advAttr8 ;
                  }
                  //* Apply modifications (3rd parm indicates a color change) *
                  this->AdvancedSetup ( advPw, forceUpdate, true ) ;
               }

               //*****************
               //** ptA_ColorDD **
               //*****************
               else if ( icIndex == ptA_ColorDD )
               {
                  this->advInit.colorCount = 1 ;   // single-color display
                  //* Choose the color attribute according to Pinwheel style *
                  ddIndex = this->dp->GetDropdownSelect ( ptA_StyleDD ) ;

                  //* Style: pwsVertiD/pwsHorizL *
                  if ( (ddIndex == pwsVertiD) || (ddIndex == pwsHorizL) )
                     this->advInit.colorPtr = &this->advAttr1d[Info.selMember] ;
                  //* Style: pwsWheelC/pwsWheelA/pwsVertiU/pwsHorizL *
                  else
                     this->advInit.colorPtr = &this->advAttr1u[Info.selMember] ;
                  //* Be sure multi-color option is disabled *
                  this->dp->SetRadiobuttonState ( ptA_MultiRB, false ) ;
                  //* Apply modifications (3rd parm indicates a color change) *
                  //* If operation successful, remember the new index.        *
                  //* If operation fails, restore previous index.             *
                  // Programmer's Note: We track the previous color here because
                  // the Pinwheel class is not able to tell us about the actual
                  // color attribute(s) in its local storage.
                  if ( (this->AdvancedSetup ( advPw, forceUpdate, true )) )
                     prevColorDD = this->dp->GetDropdownSelect ( ptA_ColorDD ) ;
                  else
                  {
                     this->dp->NextControl () ;
                     this->dp->SetDropdownSelect ( ptA_ColorDD, prevColorDD ) ;
                     this->dp->PrevControl () ;
                  }
               }
            }
         }

         else if ( ic[icIndex].type == dctSPINNER )
         {
            icIndex = this->dp->EditSpinner ( Info ) ;
            if ( Info.dataMod != false )
            {
               //***************
               //** ptSpeedSP **
               //***************
               if ( icIndex == ptSpeedSP )
               {
                  int ms ;
                  if ( (this->dp->GetSpinnerValue ( ptSpeedSP, ms )) == OK )
                     this->Speed ( ZERO, pwCNT, short(ms) ) ;
               }

               //****************
               //** ptA_TickSP **
               //****************
               else if ( icIndex == ptA_TickSP )
               {
                  this->dp->GetSpinnerValue ( ptA_TickSP, iValue ) ;
                  this->advInit.msTick = short(iValue) ;
                  //* Apply modifications *
                  this->AdvancedSetup ( advPw, forceUpdate ) ;
               }

               //******************
               //** ptA_HeightSP **
               //******************
               else if ( icIndex == ptA_HeightSP )
               {
                  this->dp->GetSpinnerValue ( ptA_HeightSP, iValue ) ;
                  this->advInit.rows = short(iValue) ;

                  //* If target is currently in the the alternate  *
                  //* position, return it to the standard position.*
                  this->dp->GetRadiobuttonState ( ptA_AltposRB, rbFlag ) ;
                  if ( rbFlag )
                  {
                     this->dp->SetRadiobuttonState ( ptA_AltposRB, false ) ;
                     this->AdvancedPosition ( false ) ;
                  }
                  //* For multi-cell object, set origChar to L' '. *
                  if ( (this->advInit.rows > 1) || (this->advInit.cols > 1) )
                     this->advInit.origChar = L' ' ;
                  else
                     this->advInit.origChar = L'.' ;

                  //* Apply modifications *
                  if ( (this->AdvancedSetup ( advPw, forceUpdate )) )
                  {
                     //* If height > 1 OR if width > 1,     *
                     //* disable the 'ptA_AltposRB' control.*
                     attr_t labelAttr ;
                     if ( (this->advInit.rows > 1) || (this->advInit.cols > 1) )
                     { rbFlag = false ; labelAttr = dimAttr ; }
                     else
                     { rbFlag = true ; labelAttr = this->dColor ; }
                     this->dp->ControlActive ( ptA_AltposRB, rbFlag ) ;
                     this->dp->WriteString ( (this->icptr[ptA_AltposRB].ulY +
                                              this->icptr[ptA_AltposRB].labY),
                                             (this->icptr[ptA_AltposRB].ulX +
                                              this->icptr[ptA_AltposRB].labX),
                                             this->icptr[ptA_AltposRB].label,
                                             labelAttr, true ) ;
                  }
               }

               //*****************
               //** ptA_WidthSP **
               //*****************
               else if ( icIndex == ptA_WidthSP )
               {
                  this->dp->GetSpinnerValue ( ptA_WidthSP, iValue ) ;
                  this->advInit.cols = short(iValue) ;

                  //* If target is currently in the the alternate  *
                  //* position, return it to the standard position.*
                  this->dp->GetRadiobuttonState ( ptA_AltposRB, rbFlag ) ;
                  if ( rbFlag )
                  {
                     this->dp->SetRadiobuttonState ( ptA_AltposRB, false ) ;
                     this->AdvancedPosition ( false ) ;
                  }
                  //* For multi-cell object, set origChar to L' '. *
                  if ( (this->advInit.rows > 1) || (this->advInit.cols > 1) )
                     this->advInit.origChar = L' ' ;
                  else
                     this->advInit.origChar = L'.' ;

                  //* Apply modifications *
                  if ( (this->AdvancedSetup ( advPw, forceUpdate )) )
                  {
                     //* If height > 1 OR if width > 1,     *
                     //* disable the 'ptA_AltposRB' control.*
                     attr_t labelAttr ;
                     if ( (this->advInit.rows > 1) || (this->advInit.cols > 1) )
                     { rbFlag = false ; labelAttr = dimAttr ; }
                     else
                     { rbFlag = true ; labelAttr = this->dColor ; }
                     this->dp->ControlActive ( ptA_AltposRB, rbFlag ) ;
                     this->dp->WriteString ( (this->icptr[ptA_AltposRB].ulY +
                                              this->icptr[ptA_AltposRB].labY),
                                             (this->icptr[ptA_AltposRB].ulX +
                                              this->icptr[ptA_AltposRB].labX),
                                             this->icptr[ptA_AltposRB].label,
                                             labelAttr, true ) ;
                  }
               }
            }
         }

         else if ( ic[icIndex].type == dctRADIOBUTTON )
         {
            if ( Info.viaHotkey != false )
            {
               Info.HotData2Primary () ;
               // Programmer's Note: The NcDialog API allows radiobuttons 
               // to be SET via hotkey, but not reset. This sequence implements 
               // a _trick_ allowing the ptAllRB, ptA_ActiveRB and ptA_ExplicitRB 
               // controls to also be RESET via hotkey. This trick requires 
               // a way to externally verify that the Radiobutton was SET 
               // before it was selected for RESET via hotkey.
               if ( icIndex == ptAllRB )
               {
                  //* If ANY of the Pinwheel groups is currently active, *
                  //* then selection of the 'All' Radiobutton by hotkey  *
                  //* is intended to reset (deactivate) all groups.      *
                  for ( short i = ZERO ; i < pwCNT ; ++i )
                  {
                     if ( this->pw[i]->isActive() )
                     {
                        this->dp->NextControl () ;
                        this->dp->SetRadiobuttonState ( ptAllRB, false ) ;
                        this->dp->PrevControl () ;
                        Info.dataMod = true ;
                        Info.isSel = false ;
                        break ;
                     }
                  }
               }
               else if ( icIndex == ptA_ActiveRB )
               {
                  //* If Pinwheel object is currently active, *
                  //* then selection by hotkey is intended to *
                  //* reset the control.                      *
                  if ( (advPw.isActive()) )
                  {
                     this->dp->NextControl () ;
                     this->dp->SetRadiobuttonState ( ptA_ActiveRB, false ) ;
                     this->dp->PrevControl () ;
                     Info.dataMod = true ;
                     Info.isSel = false ;
                  }
               }
               else if ( icIndex == ptA_ExplicitRB )
               {
                  //* Current Radiobutton state s/b the complement *
                  //* of the 'forceUpdate' flag. Therefore, if     *
                  //* forceUpdate != false, then s/b reset.        *
                  if ( ! forceUpdate )
                  {
                     this->dp->NextControl () ;
                     this->dp->SetRadiobuttonState ( ptA_ExplicitRB, false ) ;
                     this->dp->PrevControl () ;
                     Info.dataMod = true ;
                     Info.isSel = false ;
                  }
               }
            }
            else
               icIndex = this->dp->EditRadiobutton ( Info ) ;
            if ( Info.dataMod != false )
            {
               //* Because this could be a slow operation, *
               //* first make the toggle visible.          *
               icIndex = this->dp->NextControl () ;
               icIndex = this->dp->PrevControl () ;

               //*********************************************
               //** ptAllRB: Activate/deactivate all groups **
               //*********************************************
               if ( icIndex == ptAllRB )
               {
                  this->ToggleAll ( pwCNT, Info.isSel ) ;
                  this->dp->SetRadiobuttonState ( ptActive1RB, Info.isSel ) ;
                  this->dp->SetRadiobuttonState ( ptActive2RB, Info.isSel ) ;
                  this->dp->SetRadiobuttonState ( ptActive3RB, Info.isSel ) ;
                  this->dp->SetRadiobuttonState ( ptActive4RB, Info.isSel ) ;
                  this->dp->SetRadiobuttonState ( ptActive5RB, Info.isSel ) ;
                  this->dp->SetRadiobuttonState ( ptActive6RB, Info.isSel ) ;
               }

               //*************************************************************
               //** ptActive[123456]RB: Activate/deactivate a group of four **
               //*************************************************************
               else if ( icIndex == ptActive1RB || icIndex == ptActive2RB ||
                         icIndex == ptActive3RB || icIndex == ptActive4RB ||
                         icIndex == ptActive5RB || icIndex == ptActive6RB )
               {
                  this->ToggleGroup ( (icIndex == ptActive1RB ?  0 :
                                       icIndex == ptActive2RB ?  4 :
                                       icIndex == ptActive3RB ?  8 :
                                       icIndex == ptActive4RB ? 12 :
                                       icIndex == ptActive5RB ? 16 : 20), 4 ) ;
               }

               //***************************************
               //** ptA_ActiveRB: Activate/deactivate **
               //** the "advanced" Pinwheel object.   **
               //***************************************
               else if ( icIndex == ptA_ActiveRB )
               {
                  if ( Info.isSel )
                  {
                     if ( (advPw.launch()) )
                        gs = threadActivated ;
                     else
                        gs = threadError ;
                  }
                  else
                  {
                     advPw.deactivate() ;
                     gs = threadDeactivated ;
                  }
                  this->AdvancedMsg ( gs ) ;
               }

               //********************************************
               //** ptA_MultiRB: Enable/disable multi-color**
               //** display for "advanced" Pinwheel object.**
               //********************************************
               else if ( icIndex == ptA_MultiRB )
               {
                  //* Retrieve the current style setting *
                  ddIndex = this->dp->GetDropdownSelect ( ptA_StyleDD ) ;

                  if ( Info.isSel )
                  {
                     //* Point to the multi-color attribute array     *
                     //* associated with the current style (8 colors).*
                     this->advInit.colorCount = pwTicks ;   // 8 colors
                     if ( (ddIndex == pwsWheelC) || (ddIndex == pwsWheelA) )
                        this->advInit.colorPtr = this->advAttr1u ;
                     else
                        this->advInit.colorPtr = this->advAttr8 ;
                  }
                  else
                  {
                     //* Point to the specified attribute within the array *
                     //* associated with the current style (single color). *
                     this->advInit.colorCount = 1 ;         // 1 color
                     short caIndex = this->dp->GetDropdownSelect ( ptA_ColorDD ) ;
                     if ( (ddIndex == pwsVertiD) || (ddIndex == pwsHorizL) )
                        this->advInit.colorPtr = &this->advAttr1d[caIndex] ;
                     else
                        this->advInit.colorPtr = &this->advAttr1u[caIndex] ;
                  }
                  //* Apply modifications (3rd parm indicates a color change) *
                  this->AdvancedSetup ( advPw, forceUpdate, true ) ;
               }

               //**********************************************
               //** ptA_AltposRB: Specify alternate/standard **
               //** position for "advanced" Pinwheel object. **
               //**********************************************
               else if ( icIndex == ptA_AltposRB )
               {
                  //* This Radiobutton is active ONLY when the  *
                  //* size of the target object is 1x1. This    *
                  //* greatly simplifies configuration.         *
                  this->AdvancedPosition ( Info.isSel ) ;

                  //* Apply modifications *
                  this->AdvancedSetup ( advPw, forceUpdate ) ;
               }

               //********************************************************
               //** ptA_ExflagRB: Set the loop-control pointer.        **
               //** Reset: NULL: point to Pinwheel-class internal flag.**
               //** Set  : point to 'advFlag' inPinwheelTest class.    **
               //********************************************************
               else if ( icIndex == ptA_ExflagRB )
               {
                  this->advInit.exitFlag = Info.isSel ? &this->advFlag : NULL ;
                  //* Apply modifications *
                  this->AdvancedSetup ( advPw, forceUpdate ) ;
               }

               //********************************************************
               //** ptA_BidirRB: Enable/disable bidirectional mode for **
               //** "advanced" Pinwheel object.                        **
               //** Note that the bidirectional flag is not part of    **
               //** the PinwheelInit class, so we use the 4th parameter**
               //** to alert the called method to test target's flag.  **
               //********************************************************
               else if ( icIndex == ptA_BidirRB )
               {
                  this->AdvancedSetup ( advPw, forceUpdate, false, true ) ;
               }

               //********************************************************
               //** ptA_Audible: Enable/disable audible tick cycle for **
               //** "advanced" Pinwheel object. Note that the audible  **
               //** flag is not part of the PinwheelInit class. Note   **
               //** also that toggling the audible flag may be done    **
               //** asynchronously and does not require that the       **
               //** target object be inactive.                         **
               //********************************************************
               else if ( icIndex == ptA_AudibleRB )
               {
                  advPw.setAudible( Info.isSel ) ;
               }

               //***************************************************
               //** ptA_ResetRB: Reset all members of 'advInit'   **
               //** to initial values. Reinitialize and re-launch **
               //** the Pinwheel object.                          **
               //***************************************************
               else if ( icIndex == ptA_ResetRB )
               {
                  //* Reset local copy of parameters and reinitialize target  *
                  //* data members. Then re-launch the target object.         *
                  this->AdvancedInit () ;
                  if ( (advPw.resetAll( this->advInit, true )) )
                  {
                     this->AdvancedMsg ( "All parameters set to defaults." ) ;
                     if ( !(advPw.launch()) )
                     {
                        //* If system error, alert user *
                        this->AdvancedMsg ( threadError ) ;
                     }
                  }
                  //* Reset the Radiobutton state *
                  this->dp->SetRadiobuttonState ( ptA_ActiveRB, false ) ;
                  //* Set controls to match actual configuration.*
                  this->AdvancedResync ( advPw, true ) ;
               }

               //***************************************************************
               //** ptA_ExplicitRB: Mirror the state of the 'explicit-disable'**
               //** flag, by setting/resetting local 'forceUpdate' flag.      **
               //***************************************************************
               else if ( icIndex == ptA_ExplicitRB )
               {
                  forceUpdate = Info.isSel ? false : true ;
                  gs.compose( "Auto-disable: %s", (forceUpdate ? "set" : "reset") ) ;
                  this->AdvancedMsg ( gs ) ;
               }
            }
         }

         //* Move input focus to next/previous control.*
         if ( done == false && Info.viaHotkey == false )
         {
            if ( Info.keyIn == nckSTAB )
               icIndex = this->dp->PrevControl () ; 
            else
               icIndex = this->dp->NextControl () ;
         }
      }  // while()

      //* In the intereset of speed, the debugging    *
      //* method 'debugKill()' is used for the mass   *
      //* deactivation rather than sequentially       *
      //* calling 'deactivate()' for active objects   *
      //* which means waiting for each thread to wake *
      //* up and respond.                             *
      //* The 'debugKill()' method should not be used *
      //* in production-worthy code. It is use here   *
      //* only because we have 25 execution threads   *
      //* simultaneously spinning 25 Pinwheel objects.*
      for ( short indx = ZERO ; indx < pwCNT ; ++indx )
         this->pw[indx]->debugKill() ;

      //* Deactivate and delete the target objects.   *
      for ( short indx = ZERO ; indx < pwCNT ; ++indx )
      {
         this->pw[indx]->deactivate() ;   // restore display area
         delete this->pw[indx] ;          // delete the object
      }
      //* Deactivate the advanced-testing Pinwheel object. *
      //* This is an 'automatic' variable and will be      *
      //* deleted when it goes out of scope.               *
      advPw.deactivate() ;
   }

   if ( this->dp != NULL )                         // close the window
      delete ( this->dp ) ;

}  //* End PinwheelTest() *

//*************************
//*         Setup         *
//*************************
//******************************************************************************
//* Called by PinwheelTest() method to initialize the test data.               *
//* 1) Write the static text into the window.                                  *
//* 2) Calculate the Y/X position for each Pinwheel object.                    *
//* 3) Define an array of Pinwheel objects and launch (activate) each object.  *
//*                                                                            *
//* Input  : wpt    : (by reference) start position for display of text data   *
//*          iTick  : initial tick value for each object                       *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void PinwheelTest::Setup ( winPos& wpt, short iTick )
{
   const attr_t color1  = nc.re ;      // single-color objects dark-on-light
   const attr_t color1r = nc.reG ;      // single-color objects light-on-dark
   const attr_t color8u[pwTicks] =     // 8-color, upward or rightward
   { nc.bw, nc.re, nc.gr, nc.br, nc.bl, nc.ma, nc.cy, nc.gy } ;
   const attr_t color8d[pwTicks] =     // 8-color, downward or leftward
   { nc.bwR, nc.reR, nc.grR, nc.brR, nc.blR, nc.maR, nc.cyR, nc.gyR } ;
   const attr_t color16[pwbTicks] =    // 16-color, bidirectional
   { nc.bw, nc.re, nc.gr, nc.br, nc.bl, nc.ma, nc.cy, nc.gy,
     nc.bwB, nc.reB, nc.grB, nc.brB, nc.blB, nc.maB, nc.cyB, nc.gyB } ;
   const short intSPACE = 7 ;       // spacing between Pinwheel objects

   //** pwsWheelC row **
   this->wp[0] = this->dp->WriteParagraph ( wpt, "pwsWheelC ---------------  ",
                                            this->hColor ) ;
   this->wp[1] = { this->wp[0].ypos, short(this->wp[0].xpos + intSPACE) } ;
   this->wp[2] = { this->wp[1].ypos, short(this->wp[1].xpos + intSPACE) } ;
   this->wp[3] = { this->wp[2].ypos, short(this->wp[2].xpos + intSPACE) } ;
   wpt = this->dp->WriteParagraph ( wpt, "\nPinwheel\n"
                                    "(spins clockwise)\n\n", this->dColor ) ;

   //** pwsWheelA row **
   this->wp[4] = this->dp->WriteParagraph ( wpt, "pwsWheelA ---------------  ",
                                            this->hColor ) ;
   this->wp[5] = { this->wp[4].ypos, short(this->wp[4].xpos + intSPACE) } ;
   this->wp[6] = { this->wp[5].ypos, short(this->wp[5].xpos + intSPACE) } ;
   this->wp[7] = { this->wp[6].ypos, short(this->wp[6].xpos + intSPACE) } ;
   wpt = this->dp->WriteParagraph ( wpt, "\nPinwheel\n"
                                    "(spins anti-clockwise)\n\n", this->dColor ) ;

   //* pwsVertiU row *
   this->wp[8] = this->dp->WriteParagraph ( wpt, "pwsVertiU ---------------  ", 
                                            this->hColor ) ;
   this->wp[9] = { this->wp[8].ypos, short(this->wp[8].xpos + intSPACE) } ;
   this->wp[10] = { this->wp[9].ypos, short(this->wp[9].xpos + intSPACE) } ;
   this->wp[11] = { this->wp[10].ypos, short(this->wp[10].xpos + intSPACE) } ;
   wpt = this->dp->WriteParagraph ( wpt, "\nVertical Bar\n"
                                    "(grows upward)\n\n", this->dColor ) ;

   //* pwsVertiD row *
   this->wp[12] = this->dp->WriteParagraph ( wpt, "pwsVertiD ---------------  ",
                                             this->hColor ) ;
   this->wp[13] = { this->wp[12].ypos, short(this->wp[12].xpos + intSPACE) } ;
   this->wp[14] = { this->wp[13].ypos, short(this->wp[13].xpos + intSPACE) } ;
   this->wp[15] = { this->wp[14].ypos, short(this->wp[14].xpos + intSPACE) } ;
   wpt = this->dp->WriteParagraph ( wpt, "\nVertical Bar\n"
                                    "(grows downward)\n\n", this->dColor ) ;

   //* pswHorizR row *
   this->wp[16] = this->dp->WriteParagraph ( wpt, "pwsHorizR ---------------  ",
                                             this->hColor ) ;
   this->wp[17] = { wp[16].ypos, short(this->wp[16].xpos + intSPACE) } ;
   this->wp[18] = { wp[17].ypos, short(this->wp[17].xpos + intSPACE) } ;
   this->wp[19] = { wp[18].ypos, short(this->wp[18].xpos + intSPACE) } ;
   wpt = this->dp->WriteParagraph ( wpt, "\nHorizontal Bar\n"
                                    "(grows left-to-right)\n\n", this->dColor ) ;

   //* pswHorizL row *
   this->wp[20] = this->dp->WriteParagraph ( wpt, "pwsHorizL ---------------  ",
                                             this->hColor ) ;
   this->wp[21] = { this->wp[20].ypos, short(this->wp[20].xpos + intSPACE) } ;
   this->wp[22] = { this->wp[21].ypos, short(this->wp[21].xpos + intSPACE) } ;
   this->wp[23] = { this->wp[22].ypos, short(this->wp[22].xpos + intSPACE) } ;
   wpt = this->dp->WriteParagraph ( wpt, "\nHorizontal Bar\n"
                                    "(grows right-to-left)\n\n", this->dColor ) ;

   //* Instantiate pwsWheelC group and launch the control threads. *
   wchar_t origChar = L'.' ;
   attr_t  origAttr = this->dColor ;
   short pwindx = ZERO ;

   //*******************
   //* pwsWheelC group *
   //*******************
   //* Pinwheel object #00 *
   this->pw[pwindx] = new Pinwheel( this->dp, pwsWheelC, &color1r, 1, this->wp[pwindx], 
                                    iTick, 1, 1, NULL, origChar, origAttr ) ;
   this->pw[pwindx++]->launch() ;

   //* Pinwheel object #01 *
   this->pw[pwindx] = new Pinwheel( this->dp, pwsWheelC, color8d, pwTicks, 
                                    this->wp[pwindx], iTick, 1, 1, NULL, 
                                    origChar, origAttr ) ;
   this->pw[pwindx++]->launch() ;

   //* Pinwheel object #02 *
   this->pw[pwindx] = new Pinwheel( this->dp, pwsWheelC, &color1r, 1, this->wp[pwindx], 
                                    iTick, 1, 1, NULL, origChar, origAttr ) ;
   this->pw[pwindx]->setBidir( true ) ;
   this->pw[pwindx++]->launch() ;

   //* Pinwheel object #03 *
   this->pw[pwindx] = new Pinwheel( this->dp, pwsWheelC, color8d, pwTicks, 
                                    this->wp[pwindx], iTick, 1, 1, NULL, 
                                    origChar, origAttr ) ;
   this->pw[pwindx]->setBidir( true ) ;
   this->pw[pwindx++]->launch() ;

   //*******************
   //* pwsWheelA group *
   //*******************
   //* Pinwheel object #04 *
   this->pw[pwindx] = new Pinwheel( this->dp, pwsWheelA, &color1r, 1, this->wp[pwindx], 
                                    iTick, 1, 1, NULL, origChar, origAttr ) ;
   this->pw[pwindx++]->launch() ;

   //* Pinwheel object #05 *
   this->pw[pwindx] = new Pinwheel( this->dp, pwsWheelA, color8d, pwTicks, 
                                    this->wp[pwindx], iTick, 1, 1, NULL, 
                                    origChar, origAttr ) ;
   this->pw[pwindx++]->launch() ;

   //* Pinwheel object #06 *
   this->pw[pwindx] = new Pinwheel( this->dp, pwsWheelA, &color1r, 1, this->wp[pwindx], 
                                    iTick, 1, 1, NULL, origChar, origAttr ) ;
   this->pw[pwindx]->setBidir( true ) ;
   this->pw[pwindx++]->launch() ;

   //* Pinwheel object #07 *
   this->pw[pwindx] = new Pinwheel( this->dp, pwsWheelA, color8d, pwTicks, 
                                    this->wp[pwindx], iTick, 1, 1, NULL, 
                                    origChar, origAttr ) ;
   this->pw[pwindx]->setBidir( true ) ;
   this->pw[pwindx++]->launch() ;

   //*******************
   //* pwsVertiU group *
   //*******************
   //* Pinwheel object #08 *
   this->pw[pwindx] = new Pinwheel( this->dp, pwsVertiU, &color1, 1, this->wp[pwindx], 
                                    iTick, 1, 1, NULL, origChar, origAttr ) ;
   this->pw[pwindx++]->launch() ;

   //* Pinwheel object #09 *
   this->pw[pwindx] = new Pinwheel( this->dp, pwsVertiU, color8u, pwTicks, 
                                    this->wp[pwindx], iTick, 1, 1, NULL, 
                                    origChar, origAttr ) ;
   this->pw[pwindx++]->launch() ;

   //* Pinwheel object #10 *
   this->pw[pwindx] = new Pinwheel( this->dp, pwsVertiU, &color1, 1, this->wp[pwindx], 
                                    iTick, 1, 1, NULL, origChar, origAttr ) ;
   this->pw[pwindx]->setBidir( true ) ;
   this->pw[pwindx++]->launch() ;

   //* Pinwheel object #11 *
   this->pw[pwindx] = new Pinwheel( this->dp, pwsVertiU, color16, pwbTicks, 
                                    this->wp[pwindx], iTick, 1, 1, NULL, 
                                    origChar, origAttr ) ;
   this->pw[pwindx]->setBidir( true ) ;
   this->pw[pwindx++]->launch() ;

   //*******************
   //* pwsVertiD group *
   //*******************
   //* Pinwheel object #12 *
   this->pw[pwindx] = new Pinwheel( this->dp, pwsVertiD, &color1r, 1, this->wp[pwindx], 
                                    iTick, 1, 1, NULL, origChar, origAttr ) ;
   this->pw[pwindx++]->launch() ;

   //* Pinwheel object #13 *
   this->pw[pwindx] = new Pinwheel( this->dp, pwsVertiD, color8d, pwTicks, 
                                    this->wp[pwindx], iTick, 1, 1, NULL, 
                                    origChar, origAttr ) ;
   this->pw[pwindx++]->launch() ;

   //* Pinwheel object #14 *
   this->pw[pwindx] = new Pinwheel( this->dp, pwsVertiD, &color1r, 1, this->wp[pwindx], 
                                    iTick, 1, 1, NULL, origChar, origAttr ) ;
   this->pw[pwindx]->setBidir( true ) ;
   this->pw[pwindx++]->launch() ;

   //* Pinwheel object #15 *
   this->pw[pwindx] = new Pinwheel( this->dp, pwsVertiD, color16, pwbTicks, 
                                    this->wp[pwindx], iTick, 1, 1, NULL, 
                                    origChar, origAttr ) ;
   this->pw[pwindx]->setBidir( true ) ;
   this->pw[pwindx++]->launch() ;

   //*******************
   //* pwsHorizR group *
   //*******************
   //* Pinwheel object #16 *
   this->pw[pwindx] = new Pinwheel( this->dp, pwsHorizR, &color1, 1, this->wp[pwindx], 
                                    iTick, 1, 1, NULL, origChar, origAttr ) ;
   this->pw[pwindx++]->launch() ;

   //* Pinwheel object #17 *
   this->pw[pwindx] = new Pinwheel( this->dp, pwsHorizR, color8u, pwTicks, 
                                    this->wp[pwindx], iTick, 1, 1, NULL, 
                                    origChar, origAttr ) ;
   this->pw[pwindx++]->launch() ;

   //* Pinwheel object #18 *
   this->pw[pwindx] = new Pinwheel( this->dp, pwsHorizR, &color1, 1, this->wp[pwindx], 
                                    iTick, 1, 1, NULL, origChar, origAttr ) ;
   this->pw[pwindx]->setBidir( true ) ;
   this->pw[pwindx++]->launch() ;

   //* Pinwheel object #19 *
   this->pw[pwindx] = new Pinwheel( this->dp, pwsHorizR, color16, pwbTicks, 
                                    this->wp[pwindx], iTick, 1, 1, NULL, 
                                    origChar, origAttr ) ;
   this->pw[pwindx]->setBidir( true ) ;
   this->pw[pwindx++]->launch() ;

   //*******************
   //* pwsHorizL group *
   //*******************
   //* Pinwheel object #20 *
   this->pw[pwindx] = new Pinwheel( this->dp, pwsHorizL, &color1r, 1, this->wp[pwindx], 
                                    iTick, 1, 1, NULL, origChar, origAttr ) ;
   this->pw[pwindx++]->launch() ;

   //* Pinwheel object #21 *
   this->pw[pwindx] = new Pinwheel( this->dp, pwsHorizL, color8d, pwTicks, 
                                    this->wp[pwindx], iTick, 1, 1, NULL, 
                                    origChar, origAttr ) ;
   this->pw[pwindx++]->launch() ;

   //* Pinwheel object #22 *
   this->pw[pwindx] = new Pinwheel( this->dp, pwsHorizL, &color1r, 1, this->wp[pwindx], 
                                    iTick, 1, 1, NULL, origChar, origAttr ) ;
   this->pw[pwindx]->setBidir( true ) ;
   this->pw[pwindx++]->launch() ;

   //* Pinwheel object #23 *
   this->pw[pwindx] = new Pinwheel( this->dp, pwsHorizL, color16, pwbTicks, 
                                    this->wp[pwindx], iTick, 1, 1, NULL, 
                                    origChar, origAttr ) ;
   this->pw[pwindx]->setBidir( true ) ;
   this->pw[pwindx++]->launch() ;

}  //* End Setup() *

//*************************
//*     AdvancedSetup     *
//*************************
//******************************************************************************
//* For the advanced-testing Pinwheel object, adjust the operational           *
//* parameters and re-launch the object.                                       *
//* The new parameters are located in the 'advInit' data member.               *
//*                                                                            *
//* The bidirectional-output flag is not a member of the PinwheelInit class.   *
//* It is handled as a separate process.                                       *
//*                                                                            *
//* Note that the audible-tick flag is not handled by this method because      *
//* toggling this flag may be done asynchronously without deactivating the     *
//* object.                                                                    *
//*                                                                            *
//* Modification of all other parameters requires that the object be           *
//* deactivated before modification occurs.                                    *
//*                                                                            *
//* NOTE: If one or more modifications fail, then on return,the 'advInit'      *
//*       member will have been intialized with the actual setting of the      *
//*       target object. Note also that in case of error all dialog controls   *
//*       associated with the advanced parameters will be adjusted to match    *
//*       the actual values. (see AdvancedResync() method)                     *
//*                                                                            *
//* Input  : advPw   : (by reference) advanced-testing Pinwheel object         *
//*          forceOff: if 'false', Do not deactivate the object before         *
//*                                attempting to modify the parameters.        *
//*                                This tests the ability of the Pinwheel class*
//*                                to reject stupid requests.                  *
//*                    if 'true',  explicitly deactivate the object before     *
//*                                attempting to modify the parameters.        *
//*          cChange : (optional, 'false' by default)                          *
//*                    if 'true', the display colors for the target have       *
//*                               changed ('colorPtr' and/or 'colorCount')     *
//*                    There is no convenient way to directly test for a color *
//*                    change because the call to Pinwheel::getParms() returns *
//*                    a NULL pointer in the 'colorPtr' member in order to     *
//*                    protect its private data.                               *
//*          biDir   : (optional, 'false' by default)                          *
//*                    if 'true',  toggle the target's bidirectional flag      *
//*                                                                            *
//* Returns: 'true'  if modifications successful                               *
//*          'false' if one or more modifications failed                       *
//*                  The most common reason for failure is that the target     *
//*                  was not deactivated before attempting modifications.      *
//******************************************************************************

bool PinwheelTest::AdvancedSetup ( Pinwheel& advPw, bool forceOff, 
                                   bool cChange, bool biDir )
{
   gString gs ;                        // text formatting
   PinwheelInit tmpInit ;              // working target parameters
   short modCount = ZERO,              // number of parameter modifications made
         errCount = ZERO ;             // number of errors
   bool pwActive = advPw.isActive(),   // initial state of target Pinwheel
        status = false ;               // return value

   //* If specified, deactivate object before adjusting parameters *
   if ( forceOff && pwActive )
   {
      advPw.deactivate() ;
   }

   //* Get current target parameters and       *
   //* compare them with new parameters.       *
   //* (Note: dialog pointer is not compared.) *
   advPw.getParms( tmpInit ) ;

   //* STYLE: Change display style of target. *
   if ( this->advInit.dispStyle != tmpInit.dispStyle )
   {
      if ( (advPw.setStyle( this->advInit.dispStyle )) )
      { gs = "Style updated." ; ++modCount ; }
      else
         ++errCount ;
   }

   //* COLORS: Change the display color attribute(s). *
   //* (see note above)                               *
   if ( cChange || (this->advInit.colorCount != tmpInit.colorCount) )
   {
      if ( (advPw.setColors( this->advInit.colorPtr, this->advInit.colorCount )) )
      {
         gs = "Display color(s) updated." ;
         ++modCount ;
      }
      else
         ++errCount ;
   }

   //* TICK-RATE: Change the tick rate (range is controlled) *
   if ( this->advInit.msTick != tmpInit.msTick )
   {
      if ( (advPw.setTickRate( this->advInit.msTick )) )
      {
         gs = "Tick-rate updated." ;
         ++modCount ;
      }
      else
         ++errCount ;
   }

   //* HEIGHT/WIDTH: Change Height and/or Width of object *
   if ( (this->advInit.rows != tmpInit.rows) || (this->advInit.cols != tmpInit.cols) )
   {
      if ( (advPw.setDimensions( this->advInit.rows, this->advInit.cols )) )
      {
         gs = "Object dimensions updated." ;
         ++modCount ;
      }
      else
         ++errCount ;
   }

   //* POSITION: Change position of target. Caller has verified *
   //* that after repositioning, object display will remain     *
   //* entirely within its little corral.                       *
   if ( (this->advInit.yPos != tmpInit.yPos) ||
        (this->advInit.xPos != tmpInit.xPos) )
   {
      winPos wp( this->advInit.yPos, this->advInit.xPos ) ;
      if ( (advPw.setPosition( wp, this->advInit.origChar, 
            this->advInit.origAttr )) )
      {
         gs = "Position updated." ;
         ++modCount ;
      }
      else
         ++errCount ;
   }

   //* LOOP-CONTROL: Change pointer to loop-control flag. *
   //* 'exitFlag' points either to 'advFlag' member, or   *
   //* is a NULL pointer, indicating that Pinwheel-class  *
   //* internal flag should be used.                      *
   if ( this->advInit.exitFlag != tmpInit.exitFlag ) 
   {
      if ( (advPw.setLoopflagPtr( this->advInit.exitFlag )) )
      {
         gs = "Pointer to loop control flag updated." ;
         ++modCount ;
      }
      else
         ++errCount ;
   }

   //* ORIG_CHAR/ORIG_ATTR: Change the obscured character *
   //* and attribute: 'origChar' and 'origAttr'.          *
   if ( (this->advInit.origChar != tmpInit.origChar) ||
        (this->advInit.origAttr != tmpInit.origAttr) )
   {
      if ( (advPw.setOrigChar( this->advInit.origChar, this->advInit.origAttr )) )
      {
         gs = "Obscured char/attribute updated." ;
         ++modCount ;
      }
      else
         ++errCount ;
   }

   //* BIDIRECTIONAL: If specified, toggle target's bidirectional flag *
   if ( biDir )
   {
      bool oldState = advPw.getBidir(),         // current state of target's flag
           newState = oldState ? false : true ; // new state of target's flag
      if ( (advPw.setBidir( newState )) == newState )
      {
         gs.compose( "Bidirectional flag %s.", (newState ? "set" : "reset") ) ;
         ++modCount ;
      }
      else
         ++errCount ;
   }

   //* If no errors encountered *
   if ( errCount == ZERO )
   {
      //* If multiple parameters modified, create generic message *
      if ( modCount > 1 )
         gs.compose( "%hd parameters updated.", &modCount ) ;

      //* If object was active on entry *
      if ( pwActive )
      {
         //* If one or more parameters have changed, BUT *
         //* no errors encountered, re-launch the object.*
         if ( modCount > ZERO )
         {
            if ( (advPw.launch()) )
               status = true ;
            else
               gs = threadError ;
         }
         else     // (this is unlikely, but be thorough)
            gs = "No modifications requested." ;
      }
      //* Object was inactive on entry.         *
      //* Declare success, but do not re-launch.*
      else
         status = true ;

   }
   //* One or more errors. Set the 'advInit' members to    *
   //* actual configuration of the 'advPw' object, then    *
   //* synchronize the dialog control objects to match the *
   //* actual settings.                                    *
   else
   {
      advPw.getParms( this->advInit ) ;
      this->AdvancedResync ( advPw ) ;
      gs.compose( "Operation failed. %hd errors.", &errCount ) ;
   }

   #if DEBUG_PINWHEEL_INIT != 0
   this->AdvancedDump ( &tmpInit ) ;
   #endif   // DEBUG_PINWHEEL_INIT

   //* Report the status *
   this->AdvancedMsg ( gs ) ;

   return status ;

}  //* End AdvancedSetup() *

//*************************
//*    AdvancedResync     *
//*************************
//******************************************************************************
//* Re-synchronize the dialog controls with the actual object settings.        *
//* This method is called when one or more errors has occurred while adjusting *
//* the target objects parameters, resulting in potentially misleading data    *
//* being presented to the user.                                               *
//*                                                                            *
//*                                                                            *
//* Input  : advPw   : (by reference) advanced-testing Pinwheel object         *
//*          all     : (optional, 'false' by default)                          *
//*                    if 'false', sync only the Pinwheel configuration        *
//*                    if 'true',  sync Pinwheel configuration AND reinitialize*
//*                                application-level Radiobuttons:             *
//*                                'ptA_ResetRB' and 'ptA_Explicit'.           *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void PinwheelTest::AdvancedResync ( Pinwheel& advPw, bool all )
{
   int   spValue ;            // value set in Spinner control
   short icIndex ;            // index of control which currently has focus
   bool  rbState ;            // state of Radiobutton control

   //* Retrieve the current configuration *
   advPw.getParms( this->advInit ) ;

   //* Move input focus away from advanced-control group *
   icIndex = this->dp->GetCurrControl () ;
   while ( (this->dp->NextControl ()) != ptClosePB ) ;

   //* Set style selection in Dropdown.      *
   //* (assumes synchronization between enum)*
   //* (PinwheelStyle and Dropdown contents )*
   this->dp->SetDropdownSelect ( ptA_StyleDD, this->advInit.dispStyle ) ;

   //* Setting of ptA_ColorDD index is handled in the mail loop,*
   //* so in this method, we simply update the ptA_MultiRB flag.*
   if ( this->advInit.colorCount == 1 )
   {  //* Reset the multi-color Radiobutton *
      this->dp->SetRadiobuttonState ( ptA_MultiRB, false ) ;
   }
   //* Else, set the multi-color Radiobutton *
   else
      this->dp->SetRadiobuttonState ( ptA_MultiRB, true ) ;

   //* Set tick-rate Spinner *
   spValue = this->advInit.msTick ;
   this->dp->SetSpinnerValue ( ptA_TickSP, spValue ) ;

   //* Set Height and Width Spinners *'
   spValue = this->advInit.rows ;
   this->dp->SetSpinnerValue ( ptA_HeightSP, spValue ) ;
   spValue = this->advInit.cols ;
   this->dp->SetSpinnerValue ( ptA_WidthSP, spValue ) ;

   //* Initialize the 'Active' Radiobutton *
   rbState = advPw.isActive() ;
   this->dp->SetRadiobuttonState ( ptA_ActiveRB, rbState ) ;

   //* Initialize the 'Alternate Position' Radiobutton *
   rbState = bool( (this->advInit.yPos != advY) || (this->advInit.xPos != advX)) ;
   this->dp->SetRadiobuttonState ( ptA_AltposRB, rbState ) ;

   //* Initialize the 'External-loop-control' Radiobutton *
   rbState = bool(this->advInit.exitFlag != NULL) ;
   this->dp->SetRadiobuttonState ( ptA_ExflagRB, rbState ) ;

   //* Initialize the 'Bidirectional' Radiobutton *
   rbState = advPw.getBidir() ;
   this->dp->SetRadiobuttonState ( ptA_BidirRB, rbState ) ;

   //* Initialize the 'Audible' Radiobutton *
   rbState = advPw.getAudible() ;
   this->dp->SetRadiobuttonState ( ptA_AudibleRB, rbState ) ;

   //* If specified, re-initialize the application-level Radiobuttons *
   if ( all )
   {
      //* Reset the _independent_ 'ResetAll' Radiobutton *
      this->dp->SetRadiobuttonState ( ptA_ResetRB, false ) ;

      //* Initialize the _independent_ 'Explicit Disable' Radiobutton *
      rbState = this->icptr[ptA_ExplicitRB].rbSelect ;
      this->dp->SetRadiobuttonState ( ptA_ExplicitRB, rbState ) ;
   }

   //* Return focus to previous control *
   while ( (this->dp->PrevControl ()) != icIndex ) ;

}  //* End AdvancedResync() *

//*************************
//*     AdvancedInit      *
//*************************
//******************************************************************************
//* Set all members of the the PinwheelInit class associated with the advanced *
//* Pinwheel object to their default values.                                   *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void PinwheelTest::AdvancedInit ( void )
{
   //* Initialize our data members *
   this->advInit = 
   {
      this->dp,            // dialog pointer
      advY,                // origin Y
      advX,                // origin X
      pwsWheelC,           // style
      &this->advAttr1u[0], // single color
      1,                   // color count
      baseTICK,            // tick rate
      1,                   // height (rows)
      1,                   // width (cols)
      &this->advFlag,      // loop control
      L'.',                // saved character
      this->dColor         // saved attribute
   } ;

}  //* End AdvancedInit() *

//*************************
//*   AdvancedAttrInit    *
//*************************
//******************************************************************************
//* Initialize the color arrays used by the advanced-testing group.            *
//*                                                                            *
//* Additional arrays may be defined for testing purposes.                     *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void PinwheelTest::AdvancedAttrInit ( void )
{
   const attr_t color1u[pwTicks] =     // 1-color, upward or rightward (or clock)
   { nc.grbk, nc.rebk, nc.bkgr, (nc.brbk ^ ncbATTR), nc.blgy, (nc.mabk ^ ncbATTR), (nc.cybk ^ ncbATTR), nc.gyma } ;
   const attr_t color1d[pwTicks] =     // 1-color, downward or leftward
   { nc.bkgr, nc.bkre, nc.grbk, nc.bkbr, nc.bgybl, nc.bkma, nc.bkcy, nc.magy } ;
   const attr_t color8[pwTicks] =      // 8-colors, walking-sequence
   { nc.grbk, nc.regr, nc.brre, nc.blbr, nc.bmabl, nc.cyma, nc.gycy, nc.bkgy } ;

   for ( short i = ZERO ; i < pwTicks ; ++i )
   {
      this->advAttr1u[i] = color1u[i] ;
      this->advAttr1d[i] = color1d[i] ;
      this->advAttr8[i]  = color8[i] ;
   }

}  //* End AdvancedAttrInit() *

//*************************
//*   AdvancedPosition    *
//*************************
//******************************************************************************
//* Set the Y/X coordinates for the advanced Pinwheel object as well as the    *
//* character which will be obscured by the object.                            *
//*                                                                            *
//* Call this method ONLY when the Pinwheel dimensions are 1x1.                *
//*                                                                            *
//* Input  : altPos : if 'true',  set the alternate position                   *
//*                   if 'false', set the standard position                    *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void PinwheelTest::AdvancedPosition ( bool altPos )
{
   if ( altPos )        // alternate position
   {
      this->advInit.origChar = L' ' ;  // underlying character
      //* (underlying attribute assumed correct) *
      this->advInit.yPos = advY + advMaxH - 1 ;
      this->advInit.xPos = advX + advMaxW - 1 ;
   }
   else                 // standard position
   {
      this->advInit.origChar = L'.' ;
      //* (underlying attribute assumed correct) *
      this->advInit.yPos = advY ;
      this->advInit.xPos = advX ;
   }

}  //* End AdvancedPosition() *

//*************************
//*      AdvancedMsg      *
//*************************
//******************************************************************************
//* Write a message to the message line of the advanced-testing area.          *
//* The message is centered on the display line.                               *
//*                                                                            *
//* Input  : msg  : text to be written                                         *
//*          pause: (optional, 'true' by default)                              *
//*                 if 'true',  pause after writing message so user has time   *
//*                             to read it, then erase the message             *
//*                 if 'false', write the message and return immediately       *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void PinwheelTest::AdvancedMsg ( gString& msg, bool pause )
{
   chrono::duration<short>aWhile( 2 ) ;
   const short msgWIDTH = (boxCOLS - 4) ;

   msg.padCols( msgWIDTH, L' ', true ) ;
   this->dp->WriteString ( this->amsgPos, msg, this->msgAttr, true ) ;

   if ( pause )
   {
      this_thread::sleep_for( aWhile ) ;
      msg.clear() ;
      msg.padCols( msgWIDTH, L' ', true ) ;
      this->dp->WriteString ( this->amsgPos, msg, this->dColor, true ) ;
   }

}  //* End AdvancedMsg() *

void PinwheelTest::AdvancedMsg ( const char* msg, bool pause )
{
   gString gs( msg ) ;
   this->AdvancedMsg ( gs, pause ) ;

}  //* End AdvancedMsg() *

//*************************
//*      AdvancedDump      *
//*************************
//******************************************************************************
//* For Debugging Only:                                                        *
//* -------------------                                                        *
//* Display the contents of the 'advInit' member, and if specified, a second   *
//* PinwheelInit-class object.                                                 *
//*                                                                            *
//* Input  : pwInit: (optional, NULL pointer by default)                       *
//*                  if specified, a pointer to a second Pinwheel-class object *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void PinwheelTest::AdvancedDump ( const PinwheelInit *pwiPtr )
{
   #if DEBUG_PINWHEEL_INIT != 0
   const short dulY  = this->icptr[ptClosePB].ulY + 2,
               dulX  = 2,
               dRows = dROWS - 4,
               dCols = 65 ;

   //* Main Pinwheel group must have been disabled *
   //* to avoid conflict with child threads.       *
   bool rbState ;
   this->dp->GetRadiobuttonState ( ptAllRB, rbState ) ;
   if ( rbState )
   {
      this->dp->UserAlert ( 2 ) ;   // warn the user
      return ;                      // Note the early return
   }

   //* Save the parent dialog.*
   this->dp->SetDialogObscured () ;

   //* Define the dialog *
   InitNcDialog dInit( dRows,          // number of display lines
                       dCols,          // number of display columns
                       dulY,           // Y offset from upper-left of terminal 
                       dulX,           // X offset from upper-left of terminal 
                       " Display PinwheelInit ",// dialog title
                       ncltSINGLE,     // border line-style
                       this->hColor,   // border color attribute
                       this->dColor,   // interior color attribute
                       NULL            // pointer to list of control definitions
                     ) ;

   //* Instantiate the dialog window *
   NcDialog *dpd = new NcDialog ( dInit ) ;

   //* Open the dialog window *
   if ( (dpd->OpenWindow ()) == OK )
   {
      //* Display the data *
      winPos wp( 2, 2 ) ;
      wp = dpd->WriteParagraph ( wp, "'advInit' Member                      \n", 
                                 (this->hColor | ncuATTR) ) ;
      gString gs( "NcDialog *ncdPtr ------ : %p\n"
                  "short yPos ------------ : %hd\n"
                  "short xPos ------------ : %hd\n"
                  "PinwheelStyle dispStyle : %s\n"
                  "const attr_t *colorPtr  : %p\n"
                  "short colorCount ------ : %hd\n"
                  "short msTick ---------- : %hd\n"
                  "short rows ------------ : %hd\n"
                  "short cols ------------ : %hd\n"
                  "bool *exitFlag -------- : %p\n"
                  "wchar_t --------------- : '%C' (0x%04X)\n"
                  "attr_t origAttr ------- : 0x%08X\n\n\n",
                  this->advInit.ncdPtr, &this->advInit.yPos, &this->advInit.xPos,
                  (this->advInit.dispStyle == pwsWheelC ? "pwsWheelC" :
                   this->advInit.dispStyle == pwsWheelA ? "pwsWheelA" :
                   this->advInit.dispStyle == pwsVertiU ? "pwsVertiU" :
                   this->advInit.dispStyle == pwsVertiD ? "pwsVertiD" :
                   this->advInit.dispStyle == pwsHorizL ? "pwsHorizL" : "pwsHorizR"),
                   this->advInit.colorPtr, &this->advInit.colorCount, 
                   &this->advInit.msTick, &this->advInit.rows, &this->advInit.cols, 
                   this->advInit.exitFlag, &this->advInit.origChar, 
                   &this->advInit.origChar, &this->advInit.origAttr
                ) ;
      wp = dpd->WriteParagraph ( wp, gs, this->dColor ) ;

      //* Display the color-attribute arrays *
      winPos a = wp,
             b( short(a.ypos + 2), a.xpos ),
             c( short(b.ypos + 2), b.xpos ) ;
      for ( short i = ZERO ; i < pwTicks ; ++i )
      {
         a = dpd->WriteChar ( a, L'▄', this->advAttr1u[i] ) ;
         b = dpd->WriteChar ( b, L'▄', this->advAttr1d[i] ) ;
         c = dpd->WriteChar ( c, L'▄', this->advAttr8[i] ) ;
      }
      gs.compose( " advAttr1u : %p", this->advAttr1u ) ;
      dpd->WriteString ( a, gs, this->dColor ) ;
      gs.compose( " advAttr1d : %p", this->advAttr1d ) ;
      dpd->WriteString ( b, gs, this->dColor ) ;
      gs.compose( " advAttr8  : %p", this->advAttr8 ) ;
      dpd->WriteString ( c, gs, this->dColor ) ;

      //* If a second instance was provided, display it in right column.*
      if ( pwiPtr != NULL )
      {
         wp = { 2, 42 } ;
         wp = dpd->WriteParagraph ( wp, "Alt. PinwheelInit   \n", 
                                 (this->hColor | ncuATTR) ) ;
         gs.compose( "--- %p\n"
                     "--- %hd\n"
                     "--- %hd\n"
                     "--- %s\n"
                     "--- %p\n"
                     "--- %hd\n"
                     "--- %hd\n"
                     "--- %hd\n"
                     "--- %hd\n"
                     "--- %p\n"
                     "--- '%C' (0x%04X)\n"
                     "--- 0x%08X\n\n",
                     pwiPtr->ncdPtr, &pwiPtr->yPos, &pwiPtr->xPos,
                     (pwiPtr->dispStyle == pwsWheelC ? "pwsWheelC" :
                      pwiPtr->dispStyle == pwsWheelA ? "pwsWheelA" :
                      pwiPtr->dispStyle == pwsVertiU ? "pwsVertiU" :
                      pwiPtr->dispStyle == pwsVertiD ? "pwsVertiD" :
                      pwiPtr->dispStyle == pwsHorizL ? "pwsHorizL" : "pwsHorizR"),
                      pwiPtr->colorPtr, &pwiPtr->colorCount, &pwiPtr->msTick,
                      &pwiPtr->rows, &pwiPtr->cols, pwiPtr->exitFlag,
                      &pwiPtr->origChar, &pwiPtr->origChar, &pwiPtr->origAttr
                   ) ;
         wp = dpd->WriteParagraph ( wp, gs, this->dColor ) ;
      }

      //* Draw a Pushbutton image * 
      dpd->WriteString ( (dRows - 3), (dCols / 2 - 4),
                         "  CLOSE  ", this->icptr[ptClosePB].fColor ) ;


      dpd->RefreshWin () ;       // make everything visible

      nckPause();                // wait for a keypress
   }

   //* Close the dialog *
   if ( dpd != NULL )            // close the window
      delete ( dpd ) ;

   //* Restore the parent dialog *
   this->dp->RefreshWin () ;

   #endif   // DEBUG_PINWHEEL_INIT

}  //* End AdvancedDump() *

//*************************
//*      ToggleGroup      *
//*************************
//******************************************************************************
//* Called by PinwheelTest() method to enable or disable the specified group   *
//* of Pinwheel objects. If the target object is currently active, deactivate  *
//* it. If the target object is not active, activate it.                       *
//*                                                                            *
//* Input  : objIndex: index of first (of four) objects to be processed        *
//*          objCount: number of objects to be processed                       *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void PinwheelTest::ToggleGroup ( short objIndex, short objCount )
{
   for ( short i = ZERO ; i < objCount ; ++i, ++objIndex )
   {
      if ( (this->pw[objIndex]->isActive()) )
         this->pw[objIndex]->deactivate() ;
      else
         this->pw[objIndex]->launch() ;
   }

}  //* End ToggleGroup() *

//*************************
//*       ToggleAll       *
//*************************
//******************************************************************************
//* Called by PinwheelTest() method to enable or disable all Pinwheel objects  *
//* in the 'pw' array.                                                         *
//*                                                                            *
//* Input  : objCount: number of objects to be processed                       *
//*          launch  : if 'true',  enable all objects in the array             *
//*                    if 'false', disable all object in the array             *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void PinwheelTest::ToggleAll ( short objCount, bool launch )
{
   //* If launching the objects *
   if ( launch )
   {
      for ( short i = ZERO ; i < objCount ; ++i )
         this->pw[i]->launch() ;
   }

   //* If deactivating the objects *
   else
   {
      //* Use the quick-kill method, followed by cleanup. *
      for ( short i = ZERO ; i < objCount ; ++i )
         this->pw[i]->debugKill() ;
      for ( short i = ZERO ; i < objCount ; ++i )
         this->pw[i]->deactivate() ;
   }
}  //* End ToggleAll() *

//*************************
//*         Speed         *
//*************************
//******************************************************************************
//* Modify the tick speed for the specified Pinwheel objects.                  *
//* 1) Deactivate all objects.                                                 *
//* 2) Set the new tick interval for each object, and                          *
//* 3) Re-launch each object.                                                  *
//*                                                                            *
//* Input  : objIndex: index of first (of four) objects to be processed        *
//*          objCount: number of objects to be processed                       *
//*          newTick : new tick speed in milliseconds                          *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void PinwheelTest::Speed ( short objIndex, short objCount, short newTick )
{
   #if 1    // Fast method
   //* Force all objects to an inactive state.*
   //* Note: This uses the debugging method   *
   //* in the interest of speed.              *
   for ( short i = ZERO, oIndx = objIndex ; i < objCount ; ++i, ++oIndx )
      this->pw[oIndx]->debugKill() ;
   #endif   // Fast method

   for ( short i = ZERO, oIndx = objIndex ; i < objCount ; ++i, ++oIndx )
   {
      this->pw[oIndx]->deactivate() ;   // return data members to a known state
      this->pw[oIndx]->setTickRate( newTick ) ;
      this->pw[oIndx]->launch() ;
   }

}  //* End Speed() *

//*************************
//*        statMsg        *
//*************************
//******************************************************************************
//* Write a message to the message line of the advanced-testing area.          *
//* The message is centered on the display line.                               *
//*                                                                            *
//* Input  : msg  : text to be written                                         *
//*          pause: (optional, 'true' by default)                              *
//*                 if 'true',  pause after writing message so user has time   *
//*                             to read it, then erase the message             *
//*                 if 'false', write the message and return immediately       *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void PinwheelTest::statMsg ( gString& msg, bool pause )
{
   chrono::duration<short>aWhile( 2 ) ;

   msg.padCols( statWIDTH, L' ', true ) ;
   this->dp->WriteString ( this->msgPos, msg, this->msgAttr, true ) ;
   if ( pause )
   {
      this_thread::sleep_for( aWhile ) ;
      msg.clear() ;
      msg.padCols( statWIDTH, L' ', true ) ;
      this->dp->WriteString ( this->msgPos, msg, this->dColor, true ) ;
   }
}  //* End statMsg() *

void PinwheelTest::statMsg ( const char* msg, bool pause )
{
   gString gs( msg ) ;
   this->statMsg ( gs, pause ) ;

}  //* End statMsg() *


//** - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -  **
//** - - - - - - - - NON-MEMBER METHODS FOR THIS MODULE  - - - - - - - - - -  **
//** - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -  **


//*************************
//*                       *
//*************************
//******************************************************************************
//*                                                                            *
//*                                                                            *
//* Input  :                                                                   *
//*                                                                            *
//* Returns:                                                                   *
//******************************************************************************

