//******************************************************************************
//* File       : Pinwheel.hpp                                                  *
//* Author     : Mahlon R. Smith                                               *
//*              Copyright (c) 2021-2025 Mahlon R. Smith, The Software Samurai *
//*                  GNU GPL copyright notice located below.                   *
//* Date       : 28-May-2021                                                   *
//*                                                                            *
//* Description: Class definition for the Pinwheel class.                      *
//*              This class implements a widget based on the NcDialog API.     *
//*                                                                            *
//*        *** The Pinwheel widget requires that the 'pthread' ***             *
//*        *** library be linked into the application.         ***             *
//*                                                                            *
//* - The Pinwheel class implements a lightweight widget to visually indicate  *
//*   to the user that the application is "Busy" performing some               *
//*   time-consumming task. Only a basic introduction is provided here.        *
//*   Please see the NcDialog API documentation for a more complete            *
//*   description of this functionality.                                       *
//* - The Pinwheel widget by default occupies a single character cell in the   *
//*   parent dialog to display a repeating pattern of characters indicating    *
//*   that the application is still executing and has not "locked-up",         *
//*   "crashed", or "gone off into the weeds". (Users are simple creatures,    *
//*   and need this kind of visual comfort, while software designers just      *
//*   love to play with widgets. We're all just kids at heart. :-)             *
//*   Multi-cell instances of the Pinwheel object may be defined, within       *
//*   certain practical limits.                                                *
//*   - For a more full-featured "In Progress" widget including direct user    *
//*     interaction, please see the Progbar class, also distributed as part of *
//*     the NcDialog API.                                                      *
//* - The Pinwheel widget provides several choices of visual "style".          *
//*   See enum PinwheelStyle, below, for the available visual style options.   *
//* - Setup is through one of the initialization constructors, and execution   *
//*   is initiated through a call to 'launch()'. Execution is terminated by    *
//*   setting the specified boolean flag in the main application space, OR     *
//*   by calling 'deactivate()' method (preferred).                            *
//* - The Pinwheel widget launches a new execution thread to manage the visual *
//*   representation of the widget, allowing the calling thread to return      *
//*   to the application to continue the task that is underway.                *
//*   The widget's thread will continue to execute until the specified         *
//*   termination flag (Boolean) changes state.                                *
//* - Note on display updates: The underlying ncurses library implements an    *
//*   efficient algorithm for updating the dialog window's display area.       *
//*   A single character cell may be updated independently of the dialog as a  *
//*   whole; however, if multiple character cells are updated simultaneously,  *
//*   then all cells of the dialog must be updated. For this reason, the       *
//*   Pinwheel thread updates each character cell of the object individually.  *
//*   This avoids potential conflicts with any display updates simultaneously  *
//*   being performed by the application thread(s).                            *
//* - Note on restoring the original contents of the display area:             *
//*   Whe the deactivate() method is called to terminate the display loop,     *
//*   the specified character and attribute are used to restore each character *
//*   cell that was obscured by the Pinwheel object. For multi-cell objects,   *
//*   it is not possible to specify restoration character/attribute for each   *
//*   individual obscured character cell. All cells are restored using the     *
//*   same character/attribute. We hope this will not be a problem.            *
//*   The object will usually be positioned in an empty area of the dialog.    *
//*                                                                            *
//*   ----  ----  ----  ----  ----  ----  ----  ----  ----  ----  ----  ----   *
//* - Future development:                                                      *
//*   - Offer a horizontal-string format:                                      *
//*        ▁▁▁▁▁▁▁▁▁▁▁▁▁▁▁▁▁▁▁▁  OR  ────────────────────                      *
//*     This could also be implemented within a dctTextbox control.            *
//*                                                                            *
//*   - We have had a request for a version of this class which can be used    *
//*     with pure terminal applications (no nucurses support).                 *
//*     This sounds interesting, and we will experiment when time permits.     *
//*                                                                            *
//*                                                                            *
//*                                                                            *
//******************************************************************************
//* Copyright Notice:                                                          *
//* This program is free software: you can redistribute it and/or modify it    *
//* under the terms of the GNU General Public License as published by the Free *
//* Software Foundation, either version 3 of the License, or (at your option)  *
//* any later version.                                                         *
//*                                                                            *
//* This program is distributed in the hope that it will be useful, but        *
//* WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY *
//* or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License   *
//* for more details.                                                          *
//*                                                                            *
//* You should have received a copy of the GNU General Public License along    *
//* with this program.  If not, see <http://www.gnu.org/licenses/>.            *
//*                                                                            *
//*         Full text of the GPL License may be found in the Texinfo           *
//*         documentation for this program under 'Copyright Notice'.           *
//******************************************************************************
//* Version History (most recent first):                                       *
//*                                                                            *
//* v: 0.00.01 09-May-2021                                                     *
//*   - First release. This widget was originally developed as a module of     *
//*     the FileMangler application by the same author, and is used in that    *
//*     application to indicate a pause while the system kernel's output       *
//*     buffers are being flushed.                                             *
//*                                                                            *
//* v: 0.00.02 25-May-2021                                                     *
//*   - Single-cell Pinwheels (all styles) seem stable.                        *
//*   - Multi-cell Pinwheels (all styles) have been implemented.               *
//*     The code appears stable; however, it is more complex than the          *
//*     single-cell implementation, so keep an eye on it.                      *
//*   - The application's choice of color attribute(s) has a significant effect*
//*     on the visual appeal of the objects (especially multi-cell objects).   *
//*     For this reason, use care in the selection of foreground/background    *
//*     display colors to achieve the desired visual effect.                   *
//*   - Extensive test code which exercises the Pinwheel class may be found in *
//*     the Dialog4 test application.                                          *
//*     (See PinwheelTest.cpp and PinwheelTest.hpp.)                           *
//*   - Full documentation of the Pinwheel class may be found in the           *
//*     NcDialog API documentation, under "Creating Widgets", in the chapter   *
//*     "Pinwheel Widget".                                                     *
//*                                                                            *
//******************************************************************************

//* All necessary information for:                  *
//* NCurses, NcWindow, NcDialog and gString classes.*
#include "GlobalDef.hpp"

//* List of visual styles. (see also the 'setBidir()' method)                *
//* For pwsVertiU and pwsHorizR styles, the color attribute selected is      *
//* assumed to be dark text on a light background, while for pwsVertiD and   *
//* pwsHorizL the color attribute selected is assumed by be light text on a  *
//* dark background. If foreground and background are swapped, the growth    *
//* may appear as shrinkage. Experiment with color attributes to obtain the  *
//* desired display sequence.                                                *
enum PinwheelStyle : short 
{
   pwsWheelC = ZERO,          // display travels in a clockwise circle
   pwsWheelA,                 // display travels in an anti-clockwise circle
   pwsVertiU,                 // display grows Upward, from bottom-to-top
   pwsVertiD,                 // display grows Downward, from top-to-bottom
   pwsHorizR,                 // display grows Rightward, from left-to-right
   pwsHorizL,                 // display grows Leftward, from right-to-left
} ;

const short pwTicks  = 8 ;    // number of ticks in a unidirectional display cycle
const short pwbTicks = 16 ;   // number of ticks in a bidirectional display cycle
const short minTick  = 50 ;   // minimum tick: 50 milliseconds   (0.05 seconds)
const short maxTick  = 5000 ; // maximum tick: 5000 milliseconds (5.00 seconds)
const short dfltTick = 500 ;  // default tick: 500 milliseconds  (0.50 seconds)

const wchar_t pwWheelChars[pwTicks] = // clockwise pinwheel
{
   0x02502,    // L'│'
   0x0002F,    // L'/'
   0x02500,    // L'─'
   0x0005C,    // L'\\'
   0x02551,    // L'║'
   0x0002F,    // L'/'
   0x02550,    // L'═'
   0x0005C,    // L'\\'
} ;
const wchar_t pwWheelAnti[pwTicks] = // anti-clockwise pinwheel
{
   0x0005C,    // L'\\'
   0x02550,    // L'═'
   0x0002F,    // L'/'
   0x02551,    // L'║'
   0x0005C,    // L'\\'
   0x02500,    // L'─'
   0x0002F,    // L'/'
   0x02502,    // L'│'
} ;
const wchar_t pwVertiU[pwTicks] = // fills from bottom to top
{
   0x02581,    // ▁
   0x02582,    // ▂
   0x02583,    // ▃
   0x02584,    // ▄
   0x02585,    // ▅
   0x02586,    // ▆
   0x02587,    // ▇
   0x02588,    // █
} ;
const wchar_t pwVertiD[pwTicks] = // fills from top to bottom
{
   0x02588,    // █
   0x02587,    // ▇
   0x02586,    // ▆
   0x02585,    // ▅
   0x02584,    // ▄
   0x02583,    // ▃
   0x02582,    // ▂
   0x02581,    // ▁
} ;
const wchar_t pwHorizR[pwTicks] = // fills from Right
{
   0x0258F,    // ▏
   0x0258E,    // ▎
   0x0258D,    // ▍
   0x0258C,    // ▌
   0x0258B,    // ▋
   0x0258A,    // ▊
   0x02589,    // ▉
   0x02588,    // █
} ;
const wchar_t pwHorizL[pwTicks] = // fills from Left
{
   0x02588,    // █
   0x02589,    // ▉
   0x0258A,    // ▊
   0x0258B,    // ▋
   0x0258C,    // ▌
   0x0258D,    // ▍
   0x0258E,    // ▎
   0x0258F,    // ▏
} ;
const wchar_t pwBidiWheel[pwbTicks] = // bidirectional pinwheel
{
   0x02502,    // L'│'
   0x0002F,    // L'/'
   0x02500,    // L'─'
   0x0005C,    // L'\\'
   0x02551,    // L'║'
   0x0002F,    // L'/'
   0x02550,    // L'═'
   0x0005C,    // L'\\'
   0x02502,    // L'│'
   0x0005C,    // L'\\'
   0x02550,    // L'═'
   0x0002F,    // L'/'
   0x02551,    // L'║'
   0x0005C,    // L'\\'
   0x02500,    // L'─'
   0x0002F,    // L'/'
} ;
const wchar_t pwBidiVerti[pwbTicks] = // bidirectional vertical fill
{
   0x02581,    // ▁
   0x02582,    // ▂
   0x02583,    // ▃
   0x02584,    // ▄
   0x02585,    // ▅
   0x02586,    // ▆
   0x02587,    // ▇
   0x02588,    // █
   0x02587,    // ▇
   0x02586,    // ▆
   0x02585,    // ▅
   0x02584,    // ▄
   0x02583,    // ▃
   0x02582,    // ▂
   0x02581,    // ▁
   0x00020,    //' '
} ;
const wchar_t pwBidiHoriz[pwbTicks] = // bidirectional horizontal fill
{
   0x0258F,    // ▏
   0x0258E,    // ▎
   0x0258D,    // ▍
   0x0258C,    // ▌
   0x0258B,    // ▋
   0x0258A,    // ▊
   0x02589,    // ▉
   0x02588,    // █
   0x02589,    // ▉
   0x0258A,    // ▊
   0x0258B,    // ▋
   0x0258C,    // ▌
   0x0258D,    // ▍
   0x0258E,    // ▎
   0x0258F,    // ▏
   0x00020,    //' '
} ;

//*********************************************************
//*          Pinwheel initialization parameters.          *
//*             All data members are public.              *
//*  --- --- --- --- --- --- --- --- --- --- --- --- ---  *
//* >> Used to retrieve a copy of the current parameters, *
//*    (see getParms() method).                           *
//* >> Used as a parameter for initialization constructor.*
//*    NOTE: When used as a parametetr for the            *
//*    constructor, the 'ncdPtr' member is required.      *
//*    Initialization of all other data members is        *
//*    optional, but STRONGLY recommended.                *
//*********************************************************
class PinwheelInit
{
   public:       //* Data members *
   NcDialog      *ncdPtr ;
   short         yPos ;
   short         xPos ;
   PinwheelStyle dispStyle ;
   const attr_t  *colorPtr ;
   short         colorCount ;
   short         msTick ;
   short         rows ;
   short         cols ;
   bool          *exitFlag ;
   wchar_t       origChar ;
   attr_t        origAttr ;
} ;

//************************************
//* Definition of the Pinwheel class *
//************************************
class Pinwheel
{
   //* Public methods *
   public:
   //* Destructor *
   ~Pinwheel ( void )
   {
      //* If loop is still active when destructor is    *
      //* called, terminate the loop, then release all  *
      //* resources before deleting the Pinwheel object.*
      this->deactivate () ;

      //* Release the dynamic allocation for color attributes
      if ( this->cPtr != NULL )
      {
         delete [] this->cPtr ;
         this->cPtr = NULL ;
      }
   }

   //* Initialization constructor (with winPos):                         *
   //*                                                                   *
   //* Input: ncdPtr:    pointer to OPEN dialog window                   *
   //*        dispStyle: member of enum PinwheelStyle                    *
   //*                   default==pwsWheelC                              *
   //*        colorPtr:  pointer to array of color attributes            *
   //*        colorCount:number of elements in 'colors' array            *
   //*                   If either 'colors' or 'colorCnt' is             *
   //*                   out-of-range, terminal default color            *
   //*                   is used.                                        *
   //*        yxPos:     Y/X offset in dialog window for display         *
   //*                   Values must be >= ZERO. Upper limit is verified *
   //*                   by the dialog window, and if out of range data  *
   //*                   will be displayed at window origin (0,0).       *
   //*        msTick:    number of milliseconds per tick                 *
   //*                   Value range: minTick to maxTick.                *
   //*        rows:      number of character rows (height) default:1     *
   //*        cols:      number of character columns (width) default:1   *
   //*        exitFlag:  pointer to loop-termination flag                *
   //*                   (if NULL pointer, use internal flag)            *
   //*        origChar:  character which will be obscured                *
   //*                   (default: L' ')                                 *
   //*        origAttr:  color attribute for 'origChar'                  *
   //*                   (default: terminal color)                       *
   //*                                                                   *
   //* Returns: implicitly returns a pointer to object                   *
   Pinwheel ( NcDialog *ncdPtr, PinwheelStyle dispStyle, 
              const attr_t *colorPtr, short colorCount, const winPos& yxPos,
              short msTick, short rows, short cols,
              bool *exitFlag, wchar_t origChar, attr_t origAttr )
   {
      PinwheelInit pwInit
      {
         ncdPtr,
         yxPos.ypos,
         yxPos.xpos,
         dispStyle,
         colorPtr,
         colorCount,
         msTick,
         rows,
         cols,
         exitFlag,
         origChar,
         origAttr,
      } ;
      this->setup ( pwInit ) ;

   }  //* End Pinwheel constructor *

   //* Initialization constructor (with yPos and xPos)                   *
   //*                                                                   *
   //* Input: ncdPtr:    pointer to OPEN dialog window                   *
   //*        dispStyle: member of enum PinwheelStyle                    *
   //*                   default==pwsWheelC                              *
   //*        colorPtr   pointer to array of color attributes            *
   //*        colorCount:number of elements in 'colors' array            *
   //*                   If either 'colors' or 'colorCnt' is             *
   //*                   out-of-range, terminal default color            *
   //*                   is used.                                        *
   //*        yPos:      Y offset into dialog window for display         *
   //*        xPos:      X offset into dialog window for display         *
   //*                   Values must be >= ZERO. Upper limit is verified *
   //*                   by the dialog window, and if out of range data  *
   //*                   will be displayed at window origin (0,0).       *
   //*        msTick:    number of milliseconds per tick                 *
   //*                   Value range: minTick to maxTick.                *
   //*        rows:      number of character rows (height) default:1     *
   //*        cols:      number of character columns (width) default:1   *
   //*        exitFlag:  pointer to loop-termination flag                *
   //*                   (if NULL pointer, use internal flag)            *
   //*        origChar:  character which will be obscured                *
   //*                   (default: L' ')                                 *
   //*        origAttr:  color attribute for 'origChar'                  *
   //*                   (default: terminal color)                       *
   //*                                                                   *
   //* Returns: implicitly returns a pointer to object                   *
   Pinwheel ( NcDialog *ncdPtr, PinwheelStyle dispStyle, 
              const attr_t *colorPtr, short colorCount, short yPos, short xPos, 
              short msTick, short rows, short cols, bool *exitFlag,
              wchar_t origChar, attr_t origAttr )
   {
      PinwheelInit pwInit
      {
         ncdPtr,
         yPos,
         xPos,
         dispStyle,
         colorPtr,
         colorCount,
         msTick,
         rows,
         cols,
         exitFlag,
         origChar,
         origAttr,
      } ;
      this->setup ( pwInit ) ;

   }  //* End Pinwheel constructor *

   //* Initialization constructor (using PinwheelInit class)             *
   //*                                                                   *
   //* 'ncdPtr' member initialization is required.                       *
   //* Initialization of all other data members is optional, but         *
   //* STRONGLY recommended.                                             *
   //*                                                                   *
   //* Input  : (by reference) a fully-initialized instance of the       *
   //*          PinwheelInit class                                       *
   //*                                                                   *
   //* Returns: implicitly returns a pointer to object                   *
   Pinwheel ( const PinwheelInit& pwInit )
   {
      this->setup ( pwInit ) ;

   }  //* End Pinwheel constructor *

   //* Partial-initialization constructor.                               *
   //*                                                                   *
   //* This is really just a proof-of-concept, or placeholder method     *
   //* and should not be used in production code.                        *
   //* NOTE: Because the termination flag is internal to the object,     *
   //*       caller MUST call the 'deactivate()' method to exit the loop.*
   //*                                                                   *
   //* Input: ncdPtr:    pointer to OPEN dialog window                   *
   //*        yxPos:     Y/X offset in dialog window for display         *
   //*                   Values must be >= ZERO. Upper limit is verified *
   //*                   by the dialog window, and if out of range data  *
   //*                   will be displayed at window origin (0,0).       *
   //*                                                                   *
   //* The object will execute with defaults for all other members.      *
   //*        style    : pwsWheelC                                       *
   //*        color    : terminal default color                          *
   //*        colorCnt : 1                                               *
   //*        position : dialog origin (0,0)                             *
   //*        millisec : dfltTick                                        *
   //*        exit flag: internal flag                                   *
   //*        origChar : space (L' ')                                    *
   //*        origAttr:  terminal default color                          *
   //*                                                                   *
   //* Returns: implicitly returns a pointer to object                   *
   Pinwheel ( NcDialog *ncdPtr, const winPos& yxPos )
   {
      this->reset () ;              // initialize all data members
      this->dp = ncdPtr ;           // save pointer to dialog window
      this->SetPosition ( yxPos ) ; // set object's display position
      this->SetDfltAttr () ;        // set default color attribute array
   }  //* End Pinwheel constructor *

   //* Create an execution thread and launch the thread it into the      *
   //* internal display-loop method.                                     *
   //* If the object is already in the active state, this call will have *
   //* no effect, and the existing thread will continue.                 *
   //*                                                                   *
   //* Input  : none                                                     *
   //*                                                                   *
   //* Returns: 'true'  if launch is successful                          *
   //*                  (or object is already active)                    *
   //*          'false' if system calls for allocation or launch fails   *
   bool launch ( void )
   {
      //* If loop is not already active, launch. *
      //* If already active, let it continue.    *
      if ( ! this->active )
      {
         //* If thread was previously allocated, perform   *
         //* reinitialization before continuing the launch.*
         if ( this->xThread != NULL )
            this->deactivate () ;

         //* Ensure that the loop-control flag is reset. *
         //* Otherwise the loop would execute only once. *
         *this->done = false ;

         //* Allocate a new thread object. *
         // Programmer's Note that both allocation and release are handled as 
         // a thread array, even though only one thread is actually allocated.
         // This is the safer protocol for thread allocation/release because 
         // it avoids potential memory leaks.
         chrono::duration<short, std::milli>aMoment( 10 ) ;
         if ( (this->xThread = new (std::nothrow) std::thread[1]) != NULL )
         {
            try
            {
               if ( (this->height == 1) && (this->width == 1) )
                  *this->xThread = thread( &Pinwheel::onecellTarget, this ) ;
               else
                  *this->xThread = thread( &Pinwheel::multicellTarget, this ) ;

               //* Give the called method time to set the 'active' flag.*
               this_thread::sleep_for( aMoment ) ;
            }
            catch ( ... )
            {
               // Launch of thread failed, return the bad news.
            }
         }
         else ;   // Else, allocation failed, return the bad news.
      }
      return this->active ;
   }  //* End launch() *

   //* Force the object to terminate execution.                          *
   //* This is the approved method for terminating the loop. Although    *
   //* caller could simply set the application-space flag to terminate,  *
   //* that would not perform the necessary cleanup, consisting of a     *
   //* call to 'join()' followed by deleting the thread object.          *
   //* If the object is already inactive, simply return the internal     *
   //* parameters to a known state.                                      *
   //*                                                                   *
   //* Input  : none                                                     *
   //*                                                                   *
   //* Returns: 'true'  if successful (or if object is already inactive) *
   //*          'false' if deactivation fails (this is unlikely)         *
   bool deactivate ( void )
   {
      chrono::duration<short, std::milli>aMoment( this->tick ) ;
      short failsafe = 10 ;
      bool  status   = this->active ? false : true ;
      if ( this->active )
      {
         *this->done = true ;             // set the termination flag
         do                               // wait for acknowledgement
         {
            if ( ! *this->done )
            {
               this->active = false ;
               this->xThread->join() ;    // deactivate the thread object
               delete [] this->xThread ;  // delete the thread object
               this->xThread = NULL ;
               status = true ;            // good news
               break ;
            }
            this_thread::sleep_for( aMoment ) ; // sleep for the 'tick' period
         }
         while ( --failsafe > ZERO ) ;
      }
      //* If thread object inactive, but not yet "joined". *
      //* (This assumes that all code sets 'xThread' to a) *
      //* (NULL pointer after deleting thread.           ) *
      //* (A redundant call to 'join()' should do no harm) *
      else if ( this->xThread != NULL )
      {
         this->xThread->join() ;    // deactivate the thread object
         delete [] this->xThread ;  // delete the thread object
         this->xThread = NULL ;
      }
      //* Restore caller's dialog (erase the pinwheel) *
      winPos mp = this->wp ;
      short y, x ;
      for ( y = ZERO ; y < this->height ; ++y )
      {
         for ( x = ZERO ; x < this->width ; ++x )
            mp = this->dp->WriteChar ( mp, this->oChar, this->oAttr, true ) ;
         ++mp.ypos ;
         mp.xpos = this->wp.xpos ;
      }
      return status ;
   }  //* End deactivate() *

   //* Returns a copy of the current configuration.     *
   //* -- The 'colorPtr' member is set to NULL pointer  *
   //*    to prevent access to our internal color array.*
   //* -- The 'exitFlag' member is set to NULL pointer  *
   //*    IF we are pointing at our internal flag.      *
   void getParms ( PinwheelInit& pwInit )
   {
      pwInit.ncdPtr     = this->dp ;
      pwInit.yPos       = this->wp.ypos ;
      pwInit.xPos       = this->wp.xpos ;
      pwInit.dispStyle  = this->style ;
      pwInit.colorPtr   = NULL ;
      pwInit.colorCount = this->cCount ;
      pwInit.msTick     = this->tick ;
      pwInit.rows       = this->height ;
      pwInit.cols       = this->width ;
      pwInit.exitFlag   = this->done == &this->safetyFlag ? NULL : this->done ;
      pwInit.origChar   = this->oChar ;
      pwInit.origAttr   = this->oAttr ;
   }

   //* Returns 'true' is audible alert is enabled, else 'false'.         *
   bool getAudible ( void ) { return this->audible ; }

   //* Returns 'true' is bidirectional output is enabled, else 'false'.  *
   bool getBidir ( void ) { return this->bidir ; }

   //* Enable/disable a beep to be generated at the end of each cycle.   *
   //* This uses the default sound set up in the terminal configuration. *
   //* See the NcDialog-class UserAlert() method for more information.   *
   //*                                                                   *
   //* Input  : enable : 'true',  enable the beep                        *
   //*                   'false', disable the beep                       *
   //*                                                                   *
   //* Returns: current state of the 'audible' flag                      *
   bool setAudible ( bool enable )
   {
      this->audible = enable ;
      return this->audible ;
   }  //* End setAudible() *

   //* Perform a bidirectional cycle based on the specified style:       *
   //*                                                                   *
   //* For Styles:              Bidirectionality looks like:             *
   //* ----------------------   ------------------------------           *
   //* pwsWheelC OR pwsWheelA : clockwise, then anti-clockwise           *
   //* pwsVertiU OR pwsVertiD : upward, then downward                    *
   //* pwsHorizR OR pwsHorizL : rightward, then leftward                 *
   //*                                                                   *
   //* Input  : enable : 'true',  enable bidirectional operation         *
   //*                   'false', disable bidirectional operation        *
   //*                                                                   *
   //* Returns: current state of the 'bidirectional' flag                *
   //*          This command is recognized ONLY when thread is inactive. *
   bool setBidir ( bool enable )
   {
      if ( ! this->active )
      {
         this->bidir = enable ;
         this->SetStyle ( this->style ) ;
      }
      return this->bidir ;
   }  //* End setBidir() *

   //* Set the Pinwheel display style.                                   *
   //*                                                                   *
   //* Input  : dispStyle: member of enum PinwheelStyle                  *
   //*                     (if an invalid style, then pwsWheelC is used) *
   //*                                                                   *
   //* Returns: 'true'  if the style is initialized                      *
   //*          'false' if object is currently active                    *
   //*          This command is recognized ONLY when thread is inactive. *
   bool setStyle ( PinwheelStyle dispStyle )
   {
      bool success = false ;

      if ( ! this->active )
      {
         this->SetStyle ( dispStyle ) ;
         success = true ;
      }
      return success ;
   }

   //* Modify the array of color attributes used to  display    *
   //* the object.                                              *
   //*                                                          *
   //* colorPtr:  pointer to array of color attributes          *
   //* colorCount:number of elements in 'colors' array          *
   //*            If either 'colors' or 'colorCnt' is           *
   //*            out-of-range, terminal default color is used. *
   //*                                                          *
   //* Returns: 'true' if color(s) and count are initialized    *
   //*          'false' if child thread is currently active     *
   //* This command is recognized ONLY when thread is inactive. *
   bool setColors ( const attr_t *colorPtr, short colorCount )
   {
      bool success = false ;

      if ( ! this->active )
      {
         this->SetColors ( colorPtr, colorCount ) ;
         success = true ;
      }
      return success ;
   }

   //* Set the coordinates for display of the pinwheel.         *
   //*                                                          *
   //* yxPos:     Y/X offset in dialog window for display       *
   //*            Values must be >= ZERO. Max value is          *
   //*            range-checked during the call to              *
   //*            WriteChar(). out-of-range set to ZERO.        *
   //* origChar:  character which will be obscured              *
   //*            (default: L' ')                               *
   //* origAttr:  color attribute for 'origChar'                *
   //*            (default: terminal color)                     *
   //*                                                          *
   //* Returns: 'true'  if the coordinates are initialized      *
   //*          'false' if child thread is currently active     *
   //* This command is recognized ONLY when thread is inactive. *
   bool setPosition ( const winPos& yxPos, wchar_t origChar, attr_t origAttr )
   {
      bool success = false ;

      if ( ! this->active )
      {
         this->SetPosition ( yxPos ) ;
         this->oChar = origChar ;
         this->oAttr = origAttr ;
         success = true ;
      }


      return success ;
   }

   //* Modify the height and width of the displayed object.     *
   //*                                                          *
   //* Input  : rows : number of rows (>= 1)                    *
   //*          cols : number of columns (>= 1)                 *
   //*                                                          *
   //* Returns: 'true' if value is initialized                  *
   //*          'false' if child thread is currently active     *
   //* This command is recognized ONLY when thread is inactive. *
   bool setDimensions ( short rows, short cols )
   {
      bool success = false ;

      if ( ! this->active )
      {
         if ( (rows >= 1) && (cols >= 1) )
         {
            this->height = rows ;
            this->width  = cols ;
            success = true ;
         }
      }
      return success ;
   }

   //* Modify the tick rate.                                    *
   //*                                                          *
   //* Input  : msTick: number of milliseconds per tick         *
   //*                  Value range: minTick to maxTick.        *
   //*                                                          *
   //* Returns: 'true' if value is initialized                  *
   //*          'false' if child thread is currently active     *
   //* This command is recognized ONLY when thread is inactive. *
   bool setTickRate ( short msTick )
   {
      bool success = false ;

      if ( ! this->active )
      {
         if ( (msTick >= minTick) && (msTick <= maxTick) )
         {
            this->tick = msTick ;
            success = true ;
         }
      }
      return success ;
   }

   //************************************************************
   //* Point to the loop-termination flag.                      *
   //* If successful, the target flag will be reset.            *
   //*                                                          *
   //* exitFlag:  pointer to loop-termination flag              *
   //*            (if NULL pointer, use internal flag)          *
   //*                                                          *
   //* Returns: 'true'  if flag pointer is initialized          *
   //*          'false' if child thread is currently active     *
   //* This command is recognized ONLY when thread is inactive. *
   //************************************************************
   bool setLoopflagPtr ( bool *exitFlag )
   {
      bool success = false ;

      if ( ! this->active )
      {
         if ( exitFlag != NULL )
            this->done = exitFlag ;
         else
            this->done = &this->safetyFlag ;
         success = true ;
      }
      return success ;
   }

   //************************************************************
   //* Set the obscured character and its attribute.            *
   //* (This is not very useful in the real      )              *
   //* (world, but is helpful during development.)              *
   //*                                                          *
   //* Input  : origChar : character obscured by the object     *
   //*          origAttr : color attribute for obscured char    *
   //*                                                          *
   //* Returns: 'true'  if char/attribute has been updated      *
   //*          'false' if child thread is currently active     *
   //* This command is recognized ONLY when thread is inactive. *
   //************************************************************
   bool setOrigChar ( wchar_t origChar, attr_t origAttr )
   {
      bool success = false ;

      if ( ! this->active )
      {
         this->oChar = origChar ;
         this->oAttr = origAttr ;
         success = true ;
      }
      return success ;
   }

   //************************************************************
   //* Reset all members to specified values.                   *
   //* Also reset the 'bidir' and 'audible' flags.              *
   //*                                                          *
   //* Input  : pwInit    : (by reference) fully initialized    *
   //*                      instance of PinwheelInit class      *
   //*          deactivate: if 'true' : if object is currently  *
   //*                      active, deactivate it               *
   //*                                                          *
   //* Returns: 'true'  if data members initialized             *
   //*          'false' if child thread is currently active     *
   //*                  AND 'deactivate' == false               *
   //************************************************************
   bool resetAll ( const PinwheelInit& pwInit, bool deactivate )
   {
      bool success = false ;

      //* If specified and if active, deactivate *
      if ( deactivate && this->active )
         this->deactivate() ;

      if ( ! this->active )
      {
         this->reset () ;

         this->setup ( pwInit ) ;
         this->bidir = false ;
         this->audible = false ;
         success = true ;
      }
      return success ;
   }

   //************************************************************
   //* Returns the state of the 'active' flag indicating        *
   //* whether the child thread is currently executing.         *
   //************************************************************
   bool isActive ( void ) const
   {
      return this->active ;
   }

   //************************************************************
   //* FOR DEBUGGING ONLY: reset the loop-control flag. This    *
   //* is faster, but less secure than a call to deactive().    *
   //************************************************************
   void debugKill ( void )
   {
      *this->done = true ;
   }

   //************************************
   //* Private methods and data members *
   //************************************
   private:
   //* Private: Default constructor *
   Pinwheel ( void )
   {
      this->reset () ;           // initialize all data members
   }

   //* Private: This method is called by the full-initialization *
   //* constructors to initialize all data members.              *
   void setup ( const PinwheelInit& pwInit )
   {
      this->reset () ;           // initialize all data members

      //* Save pointer to dialog window. If caller sends us an  *
      //* invalid pointer, he/she/it will get what they deserve.*
      this->dp = pwInit.ncdPtr ;

      //* Obtain the dimensions of the target dialog window. *
      //* This information is used to prevent the Pinwheel   *
      //* object from extending beyond the window boundaries.*
      short dwRows, dwCols ;
      this->dp->GetDialogDimensions ( dwRows, dwCols ) ;

      //* Set the position of the object.        *
      //* If necessary adjust the position to    *
      //* ensure object is within the window.    *
      if ( pwInit.yPos < ZERO )           this->wp.ypos = ZERO ;
      else if ( pwInit.yPos >= dwRows )   this->wp.ypos = dwRows - 1 ;
      else                                this->wp.ypos = pwInit.yPos ;
      if ( pwInit.xPos < ZERO )           this->wp.xpos = ZERO ;
      else if ( pwInit.xPos >= dwCols )   this->wp.xpos = dwCols - 1 ;
      else                                this->wp.xpos = pwInit.xPos ;

      //* Set the dimensions of the object.      *
      //* If necessary, adjust the dimensions to *
      //* ensure object is entirely within the   *
      //* window.                                *
      if ( (this->wp.ypos + pwInit.rows - 1) < dwRows )
         this->height = pwInit.rows ;
      else
         this->height = dwRows - this->wp.ypos ;
      if ( (this->wp.xpos + pwInit.cols - 1) < dwCols )
         this->width = pwInit.cols ;
      else
         this->width = dwCols - this->wp.xpos ;

      //* Set the object style (default: pwsWheelC) *
      this->SetStyle ( pwInit.dispStyle ) ;

      //* Copy the color-attribute array. *
      //* If caller error, set default.   *
      this->SetColors ( pwInit.colorPtr, pwInit.colorCount ) ;

      //* Set tick value (milliseconds) *
      if ( (pwInit.msTick >= minTick) && (pwInit.msTick <= maxTick) )
         this->tick = pwInit.msTick ;
      else
         this->tick = dfltTick ;

      //* Save location of termination flag.  *
      //* If caller error, we are pointing at *
      //* the internal flag 'safetyFlag'.     *
      if ( pwInit.exitFlag != NULL )
         this->done = pwInit.exitFlag ;

      //* Save the existing char/attribute at display position *
      this->oChar = pwInit.origChar ;
      this->oAttr = pwInit.origAttr ;

   }  //* End setup() *

   //* Private: Reset all data members *
   void reset ( void )
   {
      this->dp      = NULL ;
      this->wp      = { 0, 0 } ;
      this->wPtr    = pwWheelChars ;
      this->style   = pwsWheelC ;
      this->cPtr    = NULL ;
      this->xThread = NULL ;
      this->oChar   = L' ' ;
      this->oAttr   = ZERO ;
      this->cCount  = ZERO ;
      this->tick    = dfltTick ;
      this->height  = 1 ;
      this->width   = 1 ;
      this->done    = &this->safetyFlag ;
      this->active  = false ;
      this->audible = false ;
      this->bidir   = false ;
      this->safetyFlag = false ;
   }  //* End reset() *

   //* Private: Set Pinwheel style *
   void SetStyle ( PinwheelStyle s )
   {
      //* If bidirectional cycle *
      if ( this->bidir )
      {
         switch ( s )
         {
            case pwsVertiU:
            case pwsVertiD:
               this->style = pwsVertiU ;
               this->wPtr  = pwBidiVerti ;
               break ;
            case pwsHorizR:
            case pwsHorizL:
               this->style = pwsHorizR ;
               this->wPtr  = pwBidiHoriz ;
               break ;
            case pwsWheelC:
            case pwsWheelA:
            default:
               this->style = pwsWheelC ;
               this->wPtr  = pwBidiWheel ;
               break ;
         } ;
      }
      //* If unidirectional style *
      else
      {
         switch ( s )
         {
         case pwsVertiU:
               this->style = pwsVertiU ;
               this->wPtr  = pwVertiU ;
               break ;
            case pwsVertiD:
               this->style = pwsVertiD ;
               this->wPtr  = pwVertiD ;
               break ;
            case pwsHorizR:
               this->style = pwsHorizR ;
               this->wPtr  = pwHorizR ;
               break ;
            case pwsHorizL:
               this->style = pwsHorizL ;
               this->wPtr  = pwHorizL ;
               break ;
            case pwsWheelA:
               this->style = pwsWheelA ;
               this->wPtr  = pwWheelAnti ;
               break ;
            case pwsWheelC:
            default:
               this->style = pwsWheelC ;
               this->wPtr  = pwWheelChars ;
               break ;
         } ;
      }
   }  //* End SetStyle() *

   //* Private: Copy the color-attribute array and set the color count.  *
   //* If invalid parameter, set default color (one-color, term default).*
   void SetColors ( const attr_t *colorPtr, short colorCount )
   {
      if ( colorCount >= 1 || colorPtr != NULL )
      {
         //* Release any previously-defined array *
         if ( this->cPtr != NULL )
            delete [] this->cPtr ;

         //* Allocate a new array and copy data to it *
         this->cCount = colorCount ;
         this->cPtr = new attr_t [this->cCount] ;
         for ( short i = ZERO ; i < this->cCount ; ++i )
            this->cPtr[i] = colorPtr[i] ;
      }
      else
         this->SetDfltAttr () ;
   }  //* End SetColors() *

   //* Private: Set the display position. *
   //* Note: upper limit is not verified. *
   void SetPosition ( const winPos& yxPos )
   {
      this->wp.ypos = (yxPos.ypos >= ZERO ? yxPos.ypos : ZERO) ;
      this->wp.xpos = (yxPos.xpos >= ZERO ? yxPos.xpos : ZERO) ;
   }  //* End SetPosition() *

   //******************************************************
   //* Spawned execution thread for single-cell display   *
   //* starts here. (Caller has verified all parameters.) *
   //******************************************************
   void onecellTarget ( void )
   {
      //* Set up the delay timer *
      chrono::duration<short, std::milli>aMoment( this->tick ) ;
      short wCount = (this->bidir ? pwbTicks : pwTicks), // elements in wchar_array
            wIndx = ZERO,     // index into specified wchar_t array
            cIndx = ZERO ;    // index into color-attribute array
      this->active = true ;   // acknowledge that thread is active

      //* Execute the loop until the termination flag is set *
      while ( ! *this->done )
      {
         //* Write the character (with refresh) *
         this->dp->WriteChar ( this->wp, this->wPtr[wIndx++], 
                               this->cPtr[cIndx], true ) ;
         //* For multi-color display, advance to next color when:    *
         //* If pwsWheelC or pwsWheelA, advance color with each tick.*
         //* For other styles, advance color once per cycle.         *
         if ( (this->cCount > 1) &&
              ((this->style == pwsWheelC) || (this->style == pwsWheelA) ||
               (wIndx >= wCount)) )
         { ++cIndx ; }

         //* Test the indices for end-of-buffer *
         if ( wIndx >= wCount )
         {
            wIndx = ZERO ;
            if ( this->audible )
               this->dp->UserAlert () ;
         }
         if ( cIndx >= this->cCount )
            cIndx = ZERO ;

         //* Sleep for the 'tick' period *
         this_thread::sleep_for( aMoment ) ;
      }
      this->active = false ;     // thread is going inactive
      *this->done = false ;      // acknowledge termination signal

      // NOTE: This thread does not actually return.
      //       The thread goes dormant, waiting for the 'join()' signal.
      //       It is the caller's responsibility to 'join()' with the 
      //       parent thread and release the child thread's resources.
   }

   //******************************************************
   //* Spawned execution thread for multi-cell display    *
   //* starts here. (Caller has verified all parameters.) *
   //******************************************************
   void multicellTarget ( void )
   {
      //* Set up the delay timer *
      chrono::duration<short, std::milli>aMoment( this->tick ) ;
      winPos mp = this->wp,         // grid grows FROM origin
             ap ;                   // grid grows TOWARD origin
      PinwheelStyle pstyle = this->style ; // local copy of current style
      const wchar_t *wptr = this->wPtr ;   // local copy of pointer to char array
      short wCount = (this->bidir ? pwbTicks : pwTicks), // elements in wchar_array
            wIndx = ZERO,           // index into specified wchar_t array
            cIndx = ZERO,           // index into color-attribute array
            hcnt  = this->height,   // object height (vert. char cells)
            wcnt  = this->width ;   // object width (horiz. char cells)
      bool  clearGrid = true,       // set when grid sequence is complete
            boink = false ;         // flag to generate audible indicator

      this->active = true ;         // acknowledge that thread is active

      //* Initialize 'ap' to edge of grid opposite to origin *
      //* (pwsWheelC/pwsWheelA always reference the origin.) *
      if ( (pstyle == pwsVertiU) || (pstyle == pwsVertiD) )
         ap = { short(this->wp.ypos + this->height - 1), this->wp.xpos } ;
      else if ( (pstyle == pwsHorizR) || (pstyle == pwsHorizL) )
         ap = { this->wp.ypos, short(this->wp.xpos + this->width - 1) } ;

      //* If bidirectional display, set up switching between *
      //* up/down or right/left. (Wheel styles not affected).*
      if ( this->bidir && 
           ((pstyle == pwsVertiU) || (pstyle == pwsVertiD) ||
             (pstyle == pwsHorizR) || (pstyle == pwsHorizL)) )
      {
         wCount = pwTicks ;      // length of char array
         if ( pstyle == pwsVertiU )       wptr = pwVertiU ;
         else if ( pstyle == pwsVertiD )  wptr = pwVertiD ;
         else if ( pstyle == pwsHorizR )  wptr = pwHorizR ;
         else if ( pstyle == pwsHorizL )  wptr = pwHorizL ;
      }

      //* Execute the loop until the termination flag is set *
      while ( ! *this->done )
      {
         //* pwsWheelC and pwsWheelA: all cells of the row/column *
         //* grid receive the same character simultaneously.      *
         if ( (pstyle == pwsWheelC) || (pstyle == pwsWheelA) )
         {
            this->mctDrawGrid ( this->wp, this->wPtr[wIndx], this->cPtr[cIndx],
                                hcnt, wcnt, false ) ;
            ++wIndx ;               // advance to next display character
            if ( this->audible ) boink = true ;
         }

         //**********************************************
         //* pwsVertiU: all columns grow simultaneously *
         //* from the bottom, upward.                   *
         //**********************************************
         else if ( pstyle == pwsVertiU )
         {
            //* Clear the area and begin the sequence again *
            //* Note: Clear only if single-color display.   *
            //* Colors for multi-color object should flow.  *
            //* Clear the area using a space L' ' and the   *
            //* first (only) color attribute in the array.  *
            if ( clearGrid )
            {
               if ( this->cCount == 1 )
               {
                  this->mctDrawGrid ( this->wp, L' ', this->cPtr[cIndx],
                                      hcnt, wcnt, false ) ;
               }
               mp = ap ;
               clearGrid = false ;
            }

            //* Write the character to all cells in the row.*
            this->mctDrawRow ( mp, wptr[wIndx], this->cPtr[cIndx], wcnt ) ;
            ++wIndx ;               // advance to next display character

            //* If all characters in the sequence have been written *
            //* to the current row, move to the row above.          *
            if ( wIndx >= wCount )
            {
               //* If the grid is complete *
               if ( (--mp.ypos) < this->wp.ypos )
               {
                  //* If bidirectional output is active, *
                  //* set next cycle to grow downward.   *
                  if ( this->bidir )
                  {
                     mp     = this->wp ;
                     pstyle = pwsVertiD ;
                     wptr   = pwVertiD ;
                  }

                  //* Else, set flag to clear the *
                  //* grid on the next tick.      *
                  else
                     clearGrid = true ;

                  //* If audible enabled, boink now *
                  if ( this->audible ) boink = true ;
               }
            }
         }  // (pstyle == pwsVertiU)

         //**********************************************
         //* pwsVertiD: all columns grow simultaneously *
         //* from the top, downward.                    *
         //**********************************************
         else if ( pstyle == pwsVertiD )
         {
            if ( clearGrid )
            {
               if ( this->cCount == 1 )
               {
                  this->mctDrawGrid ( this->wp, L' ', (this->cPtr[cIndx] ^ ncrATTR),
                                      hcnt, wcnt, false ) ;
               }
               mp = this->wp ;
               clearGrid = false ;
            }
            //* Write the character to all cells in the row.*
            this->mctDrawRow ( mp, wptr[wIndx], this->cPtr[cIndx], wcnt ) ;
            ++wIndx ;               // advance to next display character

            //* If all characters in the sequence have been written *
            //* to the current row, move to the row below.          *
            if ( wIndx >= wCount )
            {
               //* To fill the cell(s) on this row, we add an *
               //* extra tick then write a space to each cell.*
               this_thread::sleep_for( aMoment ) ;
               this->mctDrawRow ( mp, L' ', this->cPtr[cIndx], wcnt ) ;

               //* If the grid is complete *
               if ( ++mp.ypos > ap.ypos )
               {
                  //* If bidirectional output is active, *
                  //* set next cycle to grow downward.   *
                  if ( this->bidir )
                  {
                     mp     = ap ;
                     pstyle = pwsVertiU ;
                     wptr   = pwVertiU ;
                  }

                  //* Else, set flag to clear the *
                  //* grid on the next tick.      *
                  else
                     clearGrid = true ;

                  //* If audible enabled, boink now *
                  if ( this->audible ) boink = true ;
               }
            }
         }  // (pstyle == pwsVertiD)

         //***********************************************
         //* For pwsHorizR, all rows grow simultaneously *
         //* from the left, rightward.                   *
         //***********************************************
         else if ( pstyle == pwsHorizR )
         {
            if ( clearGrid )
            {
               if ( this->cCount == 1 )
               {
                  this->mctDrawGrid ( this->wp, L' ', this->cPtr[cIndx],
                                hcnt, wcnt, false ) ;
               }
               mp = this->wp ;
               clearGrid = false ;
            }

            //* Write the character to all cells in the column.*
            this->mctDrawCol ( mp, this->wPtr[wIndx], this->cPtr[cIndx], hcnt ) ;
            ++wIndx ;               // advance to next display character

            //* If all characters in the sequence have been written    *
            //* to the current column, move to the column to the right.*
            if ( wIndx >= wCount )
            {
               //* If the grid is complete *
               if ( ++mp.xpos > ap.xpos )
               {
                  //* If bidirectional output is active, *
                  //* set next cycle to grow leftward.   *
                  if ( this->bidir )
                  {
                     mp     = ap ;
                     pstyle = pwsHorizL ;
                     wptr   = pwHorizL ;
                  }

                  //* Else, set flag to clear the *
                  //* grid on the next tick.      *
                  else
                     clearGrid = true ;

                  //* If audible enabled, boink now *
                  if ( this->audible ) boink = true ;
               }
            }
         }  // (pstyle == pwsHorizR)

         //***********************************************
         //* For pwsHorizL, all rows grow simultaneously *
         //* from the right, leftward.                   *
         //***********************************************
         else if ( pstyle == pwsHorizL )
         {
            if ( clearGrid )
            {
               if ( this->cCount == 1 )
               {
                  this->mctDrawGrid ( this->wp, L' ', (this->cPtr[cIndx] ^ ncrATTR),
                                hcnt, wcnt, false ) ;
               }
               mp = ap ;
               clearGrid = false ;
            }
            //* Write the character to all cells in the column.*
            this->mctDrawCol ( mp, wptr[wIndx], this->cPtr[cIndx], hcnt ) ;
            ++wIndx ;               // advance to next display character

            //* If all characters in the sequence have been written    *
            //* to the current column, move to the column to the left. *
            if ( wIndx >= wCount )
            {
               //* To fill the cell(s) in this column, we add an *
               //* extra tick then write a space to each cell.   *
               this_thread::sleep_for( aMoment ) ;
               this->mctDrawCol ( mp, L' ', this->cPtr[cIndx], hcnt ) ;

               //* If the grid is complete *
               if ( --mp.xpos < this->wp.xpos )
               {
                  //* If bidirectional output is active, *
                  //* set next cycle to grow rightward.  *
                  if ( this->bidir )
                  {
                     mp     = this->wp ;
                     pstyle = pwsHorizR ;
                     wptr   = pwHorizR ;
                  }

                  //* Else, set flag to clear the *
                  //* grid on the next tick.      *
                  else
                     clearGrid = true ;

                  //* If audible enabled, boink now *
                  if ( this->audible ) boink = true ;
               }
            }
         }  // (pstyle == pwsHorizL)

         //* For multi-color display, advance to next color when:    *
         //* If pwsWheelC or pwsWheelA, advance color with each tick.*
         //* For other styles, advance color once per cycle.         *
         if ( (this->cCount > 1) &&
              ((pstyle == pwsWheelC) || (pstyle == pwsWheelA) ||
               (wIndx >= wCount)) )
         { ++cIndx ; }

         //* Test the indices for end-of-char-array *
         if ( wIndx >= wCount )
         {
            wIndx = ZERO ;
            if ( this->audible && boink )
            {
               this->dp->UserAlert () ;
               boink = false ;
            }
         }
         if ( cIndx >= this->cCount )
            cIndx = ZERO ;

         //* Sleep for the 'tick' period *
         this_thread::sleep_for( aMoment ) ;
      }
      this->active = false ;     // thread is going inactive
      *this->done = false ;      // acknowledge termination signal

      // NOTE: This thread does not actually return.
      //       The thread goes dormant, waiting for the 'join()' signal.
      //       It is the caller's responsibility to 'join()' with the 
      //       parent thread and release the child thread's resources.
   }

   //* Called by multicellTarget() to draw a grid of characters. *
   //* 1) To clear the grid.                                     *
   //* 2) To draw pwsWheelC or pwsWheelA in each cell of grid.   *
   //*                                                           *
   //* Input  : basePos: base cursor position                    *
   //*          wchar  : character to write in each cell         *
   //*          attr   : color attribute for character           *
   //*          hcnt   : height (number of rows)                 *
   //*          wcnt   : width  (number of columns)              *
   //*          up     : if 'true',  step upward for next row    *
   //*                   if 'false', step downward for next row  *
   //*                                                           *
   //* Returns: nothing                                          *
   void mctDrawGrid ( const winPos& basePos, wchar_t wchar, attr_t attr, 
                      short hcnt, short wcnt, bool up )
   {
      winPos mp = basePos ;
      for ( short y = ZERO ; y < hcnt ; ++y )
      {
         for ( short x = ZERO ; x < wcnt ; ++x )
            mp = this->dp->WriteChar ( mp, wchar, attr, true ) ;

         if ( up )
            --mp.ypos ;
         else
            ++mp.ypos ;
         mp.xpos = basePos.xpos ;
      }
   }

   //* Called by multicellTarget() to draw one row of characters.*
   //*                                                           *
   //* Input  : basePos: base cursor position                    *
   //*          wchar  : character to write in each cell         *
   //*          attr   : color attribute for character           *
   //*          wcnt   : width  (number of columns)              *
   //*                                                           *
   //* Returns: nothing                                          *
   void mctDrawRow ( const winPos& basePos, wchar_t wchar, attr_t attr, short wcnt )
   {
      winPos mp = basePos ;
      for ( short x = ZERO ; x < wcnt ; ++x )
         mp = this->dp->WriteChar ( mp, wchar, attr, true ) ;
   }

   //* Called by multicellTarget() to draw one column of         *
   //* characters.                                               *
   //*                                                           *
   //* Input  : basePos: base cursor position                    *
   //*          wchar  : character to write in each cell         *
   //*          attr   : color attribute for character           *
   //*          hcnt   : height (number of rows)                 *
   //*                                                           *
   //* Returns: nothing                                          *
   void mctDrawCol ( const winPos& basePos, wchar_t wchar, attr_t attr, short hcnt )
   {
      winPos mp = basePos ;
      for ( short y = ZERO ; y < hcnt ; ++y )
         this->dp->WriteChar ( mp.ypos++, mp.xpos, wchar, attr, true ) ;
   }

   //* If partial-initialization constructor or user error, *
   //* set the default color-attribute array.               *
   void SetDfltAttr ( void )
   {
      if ( this->cPtr != NULL )  // discard any existing array
         delete [] this->cPtr ;
      this->cCount = 1 ;
      this->cPtr = new attr_t [this->cCount] ;
      this->cPtr[0] = ZERO ;
      //* In ncurses world 0x00000000 is the default color attribute.*
   }

   //******************
   //** Data Members **
   //******************
   NcDialog *dp ;       // pointer to target dialog window
   winPos   wp ;        // position of character cell used for object display
   const wchar_t *wPtr ;// pointer to array of display characters
   attr_t *cPtr ;       // pointer to array of color attributes
   thread *xThread ;    // pointer to created execution thread
   wchar_t oChar ;      // character obscured by the Pinwhell object
   attr_t  oAttr ;      // color attribute of obscured character (origChar)
   bool *done ;         // pointer to loop-termination flag
   PinwheelStyle style ;// member of enum PinwheelStyle
   short cCount ;       // number of elements in the 'cPtr' array
   short tick ;         // number of milliseconds per tick
   short height ;       // number of rows occupied by object [CURRENTLY IGNORED]
   short width ;        // number of columns occupied by object [CURRENTLY IGNORED]
   bool  active ;       // true if object is currently executing
   bool  bidir ;        // if true, perform bi-directional cycle
   bool  audible ;      // if true, beep after each cycle
   bool  safetyFlag ;   // if constructor called with 'exitFlag' set to NULL, 
                        // then this is the termination flag
} ;   // Pinwheel class

