//********************************************************************************
//* File       : Dialog4.cpp                                                     *
//* Author     : Mahlon R. Smith                                                 *
//*              Copyright (c) 2013-2025 Mahlon R. Smith, The Software Samurai   *
//* Date       : 15-Jul-2025                                                     *
//* Version    : (see AppVersion string below)                                   *
//*                                                                              *
//* Description: Dialog4 is a test and demonstration application for             *
//*              development of the NcDialog class and associated classes.       *
//*                                                                              *
//*              See also the Dialog1, Dialog2, Dialog3 and Dialogw test         *
//*              applications for additional examples of using the NCurses       *
//*              class, NcWindow class, NcDialog class and gString class         *
//*              methods for building your own NcDialog-based applications.      *
//*                                                                              *
//* Development Tools: See NcDialog.cpp.                                         *
//********************************************************************************
//* Version History (most recent first):                                         *
//*                                                                              *
//* v: 0.00.08 21-Mar-2025                                                       *
//*   - The gString 'gsForm&' constructor is obsolete. Replace gString           *
//*     constructors which take a 'gsForm&' argument with standard               *
//*     formatting-template constructors.                                        *
//*   - Certain gString method parameters required promotion from 'short' to     *
//*     'int32_t'.                                                               *
//*                                                                              *
//* v: 0.00.07 23-Jun-2024                                                       *
//*   - Update Help dialog to reference G++13 and C++11.                         *
//*                                                                              *
//* v: 0.00.06 10-May-2021                                                       *
//*   - Modify Test #10 (previously unused). Create a menu of miscellaneous      *
//*     tests. This allows for expanding the number of tests inplemented         *
//*     within this application. This was originally done to accomodate testing  *
//*     of the Pinwheel-class functionality.                                     *
//*                                                                              *
//* v: 0.00.05 29-Jan-2018                                                       *
//*   - Allow equals sign ('=') to delimit '-a' option and its argument.         *
//*                                                                              *
//* v: 0.00.04 28-Jun-2016                                                       *
//*   - Update Test #02 to reflect changes in the UserAlert() method.            *
//*                                                                              *
//* v: 0.00.03 18-Jul-2015                                                       *
//*   - Add Test #09, Progbar class (progress bar widget)                        *
//*                                                                              *
//* v: 0.00.02 30-Apr-2014                                                       *
//*   - Add Test #06, Test RTL language support                                  *
//*   - Add Test #07, Test NcWindow-class thread safety.                         *
//*   - Add Test #08, Exercise access to command shell                           *
//*                                                                              *
//* v: 0.00.01 20-May-2013 Copy of Dialog 3.                                     *
//*   - Add Test #01, Sandbox                                                    *
//*   - Add Test #02, Exercise the UserAlert() method.                           *
//*   - Add Test #03, Test NCurses-class mouse methods                           *
//*   - Add Test #04, Test NcDialog-class mouse methods                          *
//*   - Add Test #05, Test DialogBillboard controls                              *
//*                                                                              *
//********************************************************************************
//* Programmer's Notes:                                                          *
//* - Because internationalization of I/O is always a possibility in any         *
//*   application, we use as few ASCII-oriented C/C++ library functions as       *
//*   possible. e.g. sprintf(), strlen(), toupper(), etc. Where such             *
//*   functionality is needed for character manipulation, we strongly advise     *
//*   using the equivalent 'wide chararacter' functions OR the gString class.    *
//*                                                                              *
//* - This application makes a minor use of multithreading in Tests #5 and #6.   *
//*    AND                                                                       *
//* - Thread-safety of the NcDialog class is exercised in Test #7.               *
//*                                                                              *
//* - If your system does not support multi-threading, see the conditional       *
//*   compilation flags in BillboardTest.hpp, RTL_ContentTest and                *
//*   ThreadTest.hpp.                                                            *
//*   For a more thorough introduction to C++ multithreading, please see the     *
//*   author's solution to the 'Sleeping Barber' problem in the 'Shave'          *
//*   demonstration program.                                                     *
//********************************************************************************

//****************
//* Header Files *
//****************
//* All necessary information for:                  *
//* NCurses, NcWindow, NcDialog and gString classes.*
#include "GlobalDef.hpp"

//* Definitions, prototypes and data shared among the test applications *
#include "DialogAppShared.hpp"

//* CMouseTest-class definition for Test03() *
#include "CMouseTest.hpp"

//* DMouseTest-class definition for Test04() *
#include "DMouseTest.hpp"

//* BillboardTest-class definition for Test05() *
#include "BillboardTest.hpp"

//* RTL_ContentTest-class definition for Test06() *
#include "RTL_ContentTest.hpp"

//* ThreadTest-class definition for Test07() *
#include "ThreadTest.hpp"

//* ShellTest-class definition for Test08() *
#include "ShellTest.hpp"

//* ProgbarTest-class definition for Test09() *
#include "ProgbarTest.hpp"

//* Miscellaneous Tests: Test10() *
#include "ExpandTest.hpp"


//***************
//* Definitions *
//***************

//***************
//* Prototypes  *
//***************

//**********
//*  Data  *
//**********
//* Application version string. Keep it current! *
static const char AppVersion[] = "0.0.08" ;
const char* crYears = "2013-2025" ;


//* Programmer's Note: For any dialogs that use a callback method, a set of 
//* constants should be defined at file scope for the dialog's control indices, 
//* so both the instantiating method and its callback method have access to the 
//* same definitions. These is done as an enum with initial value == ZERO.

//* Test01() dialog control constants. *
enum T01Controls {
                   T1donePB = 0, 
                   T1controlsDEFINED      // number of controls in dialog
                 } ;

//* Test05() dialog control constants. *
enum T05Controls {
                   T5DonePB = 0, T5BilaBB, T5BilbBB, 
                   T5usRB, T5ulRB, T5wsRB, T5wlRB, T5clrRB, T5MultiRB, T5NextPB, 
                   T5custTB, T5custRB, T5appPB, T5repPB, T5lineSP, T5extPB, 
                   T5controlsDEFINED      // number of controls in dialog
                 } ;


//*************************
//*         main          *
//*************************
//******************************************************************************
//* Program entry point.                                                       *
//*                                                                            *
//* Command-line Usage:  SEE BELOW FOR COMMAND-LINE ARGUMENT PROCESSING        *
//*                                                                            *
//*                                                                            *
//******************************************************************************

int main ( int argc, char* argv[], char* argenv[] )
{
short    testNum = - 1 ;         // number of test to execute (default TestMenu)
bool     winTooSmall = false ;   // set 'true' if terminal window is too small

   //* User may have specified one or more command-line arguments *
   commArgs clArgs( argc, argv, argenv ) ;
   testNum = GetCommandLineArgs ( clArgs ) ;

   #if 0    // PRETEST THE LOCALE SPECIFIED BY THE TERMINAL ENVIRONMENT
   //* Note that on start-up, the NCurses Engine sets the 'locale' (character  *
   //* encoding for internationalization) according to the terminal window's   *
   //* environment variables; however, if the default locale does not support  *
   //* UTF-8 encoding, you should adjust the terminal environment variables    *
   //* LANG=  ( || LC_ALL=  ||  LC_CTYPE= ).                                   *
   //* If your application is not ASCII only, please set the terminal          *
   //* environment variable to 'en_US.UTF-8' or another locale that supports   *
   //* UTF-8 character encoding, before running the application, or at least   *
   //* before starting the NCurses Engine.                                     *
   //* -At the command prompt    :  LANG=en_US.UTF-8                           *
   //* -or within the application:  setenv("LANG", "en_US.UTF-8", 1);          *
   //*                                                                         *
   //* Note that UTF-8 is the native language of Linux.                        *
   //*                                                                         *
   //* If user has not specified a locale on the command line, test the default*
   if ( clArgs.altLocale == false )
   {
      locale loco("") ;
      short dfltLocaleOk = nc.VerifyLocale ( loco.name().c_str() ) ;
      if ( dfltLocaleOk != OK )
      {
         //* Default locale does not support UTF-8 character encoding *
         setenv ( "LANG", "en_US.UTF-8", 1 ) ;
      }
   }
   #endif   // PRETEST

   //*******************************
   //** Initialize NCurses Engine **
   //*******************************
   //* NCurses class has been previously instantiated at the time the          *
   //* application was loaded, and is available through the name: 'nc'.        *
   //* (see note in GlobalDef.hpp)                                             *
   if( (nc.StartNCursesEngine ()) == OK )
   {
      //* Verify that the current locale supports UTF-8 character encoding.    *
      short localeSetOk = nc.VerifyLocale () ;  // verify default locale setting

      //* If user has specified an alternate locale for interpretation of      *
      //* character output and input, it must be passed to the NCurses class   *
      //* before any I/O occurs.                                               *
      //* (NOTE: If clArgs.localeName does not specify a locale supported by ) *
      //* (      the terminal program, OR the locale specified is not a      ) *
      //* (      UTF-8-encoded locale, OR the default encoding specified in  ) *
      //* (      the environment was not UTF-8, then the results will be     ) *
      //* (      ambiguous at best.                                          ) *
      short altLocaleOk = OK ;
      if ( clArgs.altLocale != false )
         altLocaleOk = nc.SetLocale ( clArgs.localeName ) ;

      //* Even though no windows have been opened yet, you can think of the    *
      //* entire screen as a big window which can be written to and which      *
      //* is the parent/grandparent of all windows and NcDialog objects that   *
      //* will be opened. (cursor state: nccvINVISIBLE and key state: nckpRAW) *
      ClearTerminalWindow () ;               // display application title

      //* Initialize and open the application's status window *
      sWin = new NcWindow ( swLINES, swCOLS, swOFFy, swOFFx ) ;
      if ( (sWin->OpenWindow ()) != ERR )
      {
         bool  done = false ;    // loop control

         //* Report any error in initializing color output*
         if ( !(nc.ColorText_Initialized ()) )
         {
            if ( ! nc.ColorText_Available () )
               StatWin ( "This terminal does not support multi-color output..." ) ;
            else
               StatWin ( "Initialization of multi-color output failed..." ) ;
            sleep ( 5 ) ;
         }
         //* Report any error in setting default or alternate locale.  *
         //* (Even if error, ASCII-only I/O SHOULD cause no problems.) *
         if ( localeSetOk != OK || altLocaleOk != OK )
         {
            if ( localeSetOk != OK )
               StatWin ( "Current locale setting does not support UTF-8 character encoding..." ) ;
            else
               StatWin ( "Locale setting specified on command line is not valid..." ) ;
            sleep ( 5 ) ;
         }

         //* Refresh main window *
         nc.RefreshScreen () ;
         nc.ScreenDimensions ( termRows, termCols ) ;   // get terminal-window size
         if ( termRows < minTERMROWS || termCols < minTERMCOLS )
         {
            //* We need room to play *
            winTooSmall = done = true ;
         }

         //* Find out what the user wants *
         while ( ! done )
         {
            //* If command-line agrument for test number provided, use it.     *
            //* Else, open a dialog so user can choose which test(s) to run.   *
            if ( testNum < ZERO )
               testNum = TestMenu ( termRows / 2, termCols / 2 ) ;

            //* Check whether user has re-sized the terminal window *
            nc.ScreenDimensions ( termRows, termCols ) ;
            if ( termRows < minTERMROWS || termCols < minTERMCOLS )
            {
               winTooSmall = done = true ;
               continue ;
            }

            //* Execute selected test *
            switch ( testNum )
            {
               case indexT01:          // Sandbox - Temporary Testing
                  Test01 () ;
                  break ;
               case indexT02:          // Audible Alerts
                  Test02 () ; 
                  break ;
               case indexT03:          // NCurses-class Mouse Methods
                  Test03 () ;
                  break ;
               case indexT04:          // NcDialog-class Mouse Methods
                  Test04 () ;
                  break ;
               case indexT05:          // Billboard Controls
                  Test05 () ;
                  break ;
               case indexT06:          // RTL Title, Labels and Contents
                  Test06 () ;
                  break ;
               case indexT07:          // Thread Safety - I/O Stress Test
                  Test07 () ;
                  break ;
               case indexT08:          // ShellOut command and options
                  Test08 () ;    
                  break ;
               case indexT09:          // Progress-bar Widget
                  Test09 () ;
                  break ;
               case indexT10:          // Menu of miscellaneous tests
                  Test10 () ;
                  break ;
               case indexEXIT:         // exit the application
                  StatWin ( " Exiting Application . . . ", nc.blG ) ;
                  done = true ;
                  break ;
               case indexHELP:         // help-about dialog
               default:
                  HelpAbout ( termRows / 2 + 1, termCols / 2 ) ;
                  break ;
            }
            testNum = -1 ;    // force use of the menu
         }        // while()
      }
      else
      {
         winTooSmall = true ;
      }

      //****************************
      //* Shut down ncurses engine *
      //****************************
      delete sWin ;                          // release status line resources       
      nc.RefreshScreen () ;                  // refresh the main window
      nc.StopNCursesEngine () ;              // Deactivate the NCurses engine

      //* Most likely cause of failure in opening the background *
      //* window is that the terminal window is too small.       *
      //* Memory allocation error is also a possibility.         *
      if ( winTooSmall != false )      
         cout << "\n ******************************************"
                 "\n * ERROR! Unwilling to open main window.  *"
                 "\n * Terminal window must be >= 80x25 chars.*"
                 "\n ******************************************\n\n" ;
   }
   else
   {
      cout << "\n **********************************************"
              "\n * ERROR! Unable to initialize NCurses engine.*"
              "\n **********************************************\n\n" ;
   }

   return ZERO ;

}  //* End main() *

//*************************
//*       TestMenu        *
//*************************
//******************************************************************************
//* Open a dialog and display a list of test choices.                          *
//*                                                                            *
//*                                                                            *
//* Input  : ctrY: Y center position for dialog window                         *
//*          ctrX: X center position for dialog window                         *
//*                                                                            *
//* Returns: member of cIndices                                                *
//******************************************************************************

short    TestMenu ( short ctrY, short ctrX )
{
static const short dialogROWS = 19 ;   // display lines
static const short dialogCOLS = 57 ;   // display columns
static const char* testNames[] = 
{// 1234567890123456789012345678901234567890123456

   "Test #1: ^Sandbox - Temporary Testing         ",
   "Test #2: ^Audible Alerts                      ",
   "Test #3: ^NCurses-class Mouse Methods         ",
   "Test #4: Nc^Dialog-class Mouse Methods        ",
   "Test #5: ^Billboard Controls                  ",
   "Test #6: ^RTL Title, Labels and Contents      ",
   "Test #7: ^Thread Safety - I/O Stress Test     ",
   "Test #8: Shell^Out command and options        ",
   "Test #9: ^Progress-bar Widget                 ",
   "Test#10: ^Miscellaneous Tests                 ",
} ;
static NcDialog*  dp ;                       // dialog window pointer
static bool    dialogOpen = false ;          // set to true when dialog opened
short    ulY     = ctrY - dialogROWS / 2,    // upper left corner in Y
         ulX     = ctrX - dialogCOLS / 2 ;   // upper left corner in X
attr_t   dColor  = nc.blR,                   // dialog's base color
         rnColor = nc.bw,                    // radio button non-focus color
         rfColor = nc.cyG ;                  // radio button focus color
short    testNum ;                           // return value

   StatWin ( "Opening Test-Selection Menu" ) ; // keep user up-to-date

   //* Instantiate the dialog window and install its control objects *
   if ( dialogOpen == false )
   {
      //****************************************
      //* Initial parameters for dialog window *
      //****************************************
      InitCtrl ic[indexCOUNT] =                    // control initialization structures
      {
         {  //* TEST01 radiobutton  - - - - - - - - - - - - - - - - -    indexT01 *
            dctRADIOBUTTON,               // type:      
            rbtS3s,                       // rbSubtype: standard, 3 chars wide
            false,                        // rbSelect:  default selection
            2,                            // ulY:       upper left corner in Y
            4,                            // ulX:       upper left corner in X
            1,                            // lines:     (n/a)
            0,                            // cols:      (n/a)
            NULL,                         // dispText:  (n/a)
            rnColor,                      // nColor:    non-focus color
            rfColor,                      // fColor:    focus color
            tbPrint,                      // filter:    (n/a)
            testNames[0],                 // label:
            ZERO,                         // labY:      
            5,                            // labX       
            ddBoxTYPES,                   // exType:    (n/a)
            1,                            // scrItems:  (n/a)
            1,                            // scrSel:    (n/a)
            NULL,                         // scrColor:  (n/a)
            NULL,                         // spinData:  (n/a)
            true,                         // active:    allow control to gain focus
            &ic[1]                        // nextCtrl:  link in next structure
         },
         {  //* TEST02 radiobutton  - - - - - - - - - - - - - - - - -    indexT02 *
            dctRADIOBUTTON,               // type:      
            rbtS3s,                       // rbSubtype: standard, 3 chars wide
            false,                        // rbSelect:  default selection
            3,                            // ulY:       upper left corner in Y
            4,                            // ulX:       upper left corner in X
            1,                            // lines:     (n/a)
            0,                            // cols:      (n/a)
            NULL,                         // dispText:  (n/a)
            rnColor,                      // nColor:    non-focus color
            rfColor,                      // fColor:    focus color
            tbPrint,                      // filter:    (n/a)
            testNames[1],                 // label:
            ZERO,                         // labY:      
            5,                            // labX       
            ddBoxTYPES,                   // exType:    (n/a)
            1,                            // scrItems:  (n/a)
            1,                            // scrSel:    (n/a)
            NULL,                         // scrColor:  (n/a)
            NULL,                         // spinData:  (n/a)
            true,                         // active:    allow control to gain focus
            &ic[2]                        // nextCtrl:  link in next structure
         },
         {  //* TEST03 radiobutton  - - - - - - - - - - - - - - - - -    indexT03 *
            dctRADIOBUTTON,               // type:      
            rbtS3s,                       // rbSubtype: standard, 3 chars wide
            false,                        // rbSelect:  default selection
            4,                            // ulY:       upper left corner in Y
            4,                            // ulX:       upper left corner in X
            1,                            // lines:     (n/a)
            0,                            // cols:      (n/a)
            NULL,                         // dispText:  (n/a)
            rnColor,                      // nColor:    non-focus color
            rfColor,                      // fColor:    focus color
            tbPrint,                      // filter:    (n/a)
            testNames[2],                 // label:
            ZERO,                         // labY:      
            5,                            // labX       
            ddBoxTYPES,                   // exType:    (n/a)
            1,                            // scrItems:  (n/a)
            1,                            // scrSel:    (n/a)
            NULL,                         // scrColor:  (n/a)
            NULL,                         // spinData:  (n/a)
            true,                         // active:    allow control to gain focus
            &ic[3]                        // nextCtrl:  link in next structure
         },
         {  //* TEST04 radiobutton  - - - - - - - - - - - - - - - - -    indexT04 *
            dctRADIOBUTTON,               // type:      
            rbtS3s,                       // rbSubtype: standard, 3 chars wide
            false,                        // rbSelect:  default selection
            5,                            // ulY:       upper left corner in Y
            4,                            // ulX:       upper left corner in X
            1,                            // lines:     (n/a)
            0,                            // cols:      (n/a)
            NULL,                         // dispText:  (n/a)
            rnColor,                      // nColor:    non-focus color
            rfColor,                      // fColor:    focus color
            tbPrint,                      // filter:    (n/a)
            testNames[3],                 // label:
            ZERO,                         // labY:      
            5,                            // labX       
            ddBoxTYPES,                   // exType:    (n/a)
            1,                            // scrItems:  (n/a)
            1,                            // scrSel:    (n/a)
            NULL,                         // scrColor:  (n/a)
            NULL,                         // spinData:  (n/a)
            true,                         // active:    allow control to gain focus
            &ic[4]                        // nextCtrl:  link in next structure
         },
         {  //* TEST05 radiobutton  - - - - - - - - - - - - - - - - -    indexT05 *
            dctRADIOBUTTON,               // type:      
            rbtS3s,                       // rbSubtype: standard, 3 chars wide
            false,                        // rbSelect:  default selection
            6,                            // ulY:       upper left corner in Y
            4,                            // ulX:       upper left corner in X
            1,                            // lines:     (n/a)
            0,                            // cols:      (n/a)
            NULL,                         // dispText:  (n/a)
            rnColor,                      // nColor:    non-focus color
            rfColor,                      // fColor:    focus color
            tbPrint,                      // filter:    (n/a)
            testNames[4],                 // label:
            ZERO,                         // labY:      
            5,                            // labX       
            ddBoxTYPES,                   // exType:    (n/a)
            1,                            // scrItems:  (n/a)
            1,                            // scrSel:    (n/a)
            NULL,                         // scrColor:  (n/a)
            NULL,                         // spinData:  (n/a)
            true,                         // active:    allow control to gain focus
            &ic[5]                        // nextCtrl:  link in next structure
         },
         {  //* TEST06 radiobutton  - - - - - - - - - - - - - - - - -    indexT06 *
            dctRADIOBUTTON,               // type:      
            rbtS3s,                       // rbSubtype: standard, 3 chars wide
            false,                        // rbSelect:  default selection
            7,                            // ulY:       upper left corner in Y
            4,                            // ulX:       upper left corner in X
            1,                            // lines:     (n/a)
            0,                            // cols:      (n/a)
            NULL,                         // dispText:  (n/a)
            rnColor,                      // nColor:    non-focus color
            rfColor,                      // fColor:    focus color
            tbPrint,                      // filter:    (n/a)
            testNames[5],                 // label:
            ZERO,                         // labY:      
            5,                            // labX       
            ddBoxTYPES,                   // exType:    (n/a)
            1,                            // scrItems:  (n/a)
            1,                            // scrSel:    (n/a)
            NULL,                         // scrColor:  (n/a)
            NULL,                         // spinData:  (n/a)
            true,                         // active:    allow control to gain focus
            &ic[6]                        // nextCtrl:  link in next structure
         },
         {  //* TEST07 radiobutton  - - - - - - - - - - - - - - - - -    indexT07 *
            dctRADIOBUTTON,               // type:      
            rbtS3s,                       // rbSubtype: standard, 3 chars wide
            false,                        // rbSelect:  default selection
            8,                            // ulY:       upper left corner in Y
            4,                            // ulX:       upper left corner in X
            1,                            // lines:     (n/a)
            0,                            // cols:      (n/a)
            NULL,                         // dispText:  (n/a)
            rnColor,                      // nColor:    non-focus color
            rfColor,                      // fColor:    focus color
            tbPrint,                      // filter:    (n/a)
            testNames[6],                 // label:
            ZERO,                         // labY:      
            5,                            // labX       
            ddBoxTYPES,                   // exType:    (n/a)
            1,                            // scrItems:  (n/a)
            1,                            // scrSel:    (n/a)
            NULL,                         // scrColor:  (n/a)
            NULL,                         // spinData:  (n/a)
            true,                         // active:    allow control to gain focus
            &ic[7]                        // nextCtrl:  link in next structure
         },
         {  //* TEST08 radiobutton  - - - - - - - - - - - - - - - - -    indexT08 *
            dctRADIOBUTTON,               // type:      
            rbtS3s,                       // rbSubtype: standard, 3 chars wide
            false,                        // rbSelect:  default selection
            9,                            // ulY:       upper left corner in Y
            4,                            // ulX:       upper left corner in X
            1,                            // lines:     (n/a)
            0,                            // cols:      (n/a)
            NULL,                         // dispText:  (n/a)
            rnColor,                      // nColor:    non-focus color
            rfColor,                      // fColor:    focus color
            tbPrint,                      // filter:    (n/a)
            testNames[7],                 // label:
            ZERO,                         // labY:      
            5,                            // labX       
            ddBoxTYPES,                   // exType:    (n/a)
            1,                            // scrItems:  (n/a)
            1,                            // scrSel:    (n/a)
            NULL,                         // scrColor:  (n/a)
            NULL,                         // spinData:  (n/a)
            true,                         // active:    allow control to gain focus
            &ic[8]                        // nextCtrl:  link in next structure
         },
         {  //* TEST09 radiobutton  - - - - - - - - - - - - - - - - -    indexT09 *
            dctRADIOBUTTON,               // type:      
            rbtS3s,                       // rbSubtype: standard, 3 chars wide
            false,                        // rbSelect:  default selection
            10,                           // ulY:       upper left corner in Y
            4,                            // ulX:       upper left corner in X
            1,                            // lines:     (n/a)
            0,                            // cols:      (n/a)
            NULL,                         // dispText:  (n/a)
            rnColor,                      // nColor:    non-focus color
            rfColor,                      // fColor:    focus color
            tbPrint,                      // filter:    (n/a)
            testNames[8],                 // label:
            ZERO,                         // labY:      
            5,                            // labX       
            ddBoxTYPES,                   // exType:    (n/a)
            1,                            // scrItems:  (n/a)
            1,                            // scrSel:    (n/a)
            NULL,                         // scrColor:  (n/a)
            NULL,                         // spinData:  (n/a)
            true,                         // active:    allow control to gain focus
            &ic[9]                        // nextCtrl:  link in next structure
         },
         {  //* TEST10 radiobutton  - - - - - - - - - - - - - - - - -    indexT10 *
            dctRADIOBUTTON,               // type:      
            rbtS3s,                       // rbSubtype: standard, 3 chars wide
            false,                        // rbSelect:  default selection
            11,                           // ulY:       upper left corner in Y
            4,                            // ulX:       upper left corner in X
            1,                            // lines:     (n/a)
            0,                            // cols:      (n/a)
            NULL,                         // dispText:  (n/a)
            rnColor,                      // nColor:    non-focus color
            rfColor,                      // fColor:    focus color
            tbPrint,                      // filter:    (n/a)
            testNames[9],                 // label:
            ZERO,                         // labY:      
            5,                            // labX       
            ddBoxTYPES,                   // exType:    (n/a)
            1,                            // scrItems:  (n/a)
            1,                            // scrSel:    (n/a)
            NULL,                         // scrColor:  (n/a)
            NULL,                         // spinData:  (n/a)
            true,                         // active:    allow control to gain focus
            &ic[10]                       // nextCtrl:  link in next structure
         },
         {  //* HELP radiobutton    - - - - - - - - - - - - - - - - -   indexHELP *
            dctRADIOBUTTON,               // type:      
            rbtS3s,                       // rbSubtype: standard, 3 chars wide
            true,                         // rbSelect:  default selection
            12,                           // ulY:       upper left corner in Y
            4,                            // ulX:       upper left corner in X
            1,                            // lines:     (n/a)
            0,                            // cols:      (n/a)
            NULL,                         // dispText:  (n/a)
            rnColor,                      // nColor:    non-focus color
            rfColor,                      // fColor:    focus color
            tbPrint,                      // filter:    (n/a)
            "^Help About dialog",         // label:
            ZERO,                         // labY:      
            5,                            // labX       
            ddBoxTYPES,                   // exType:    (n/a)
            1,                            // scrItems:  (n/a)
            1,                            // scrSel:    (n/a)
            NULL,                         // scrColor:  (n/a)
            NULL,                         // spinData:  (n/a)
            true,                         // active:    allow control to gain focus
            &ic[11]                       // nextCtrl:  link in next structure
         },
         {  //* EXIT radiobutton    - - - - - - - - - - - - - - - - -   indexEXIT *
            dctRADIOBUTTON,               // type:      
            rbtS3s,                       // rbSubtype: standard, 3 chars wide
            false,                        // rbSelect:  default selection
            13,                           // ulY:       upper left corner in Y
            4,                            // ulX:       upper left corner in X
            1,                            // lines:     (n/a)
            0,                            // cols:      (n/a)
            NULL,                         // dispText:  (n/a)
            rnColor,                      // nColor:    non-focus color
            rfColor,                      // fColor:    focus color
            tbPrint,                      // filter:    (n/a)
            "E^xit",                      // label:
            ZERO,                         // labY:      
            5,                            // labX       
            ddBoxTYPES,                   // exType:    (n/a)
            1,                            // scrItems:  (n/a)
            1,                            // scrSel:    (n/a)
            NULL,                         // scrColor:  (n/a)
            NULL,                         // spinData:  (n/a)
            true,                         // active:    allow control to gain focus
            NULL                          // nextCtrl:  link in next structure
         },
      } ;
   
   //* Initial parameters for dialog window *
   InitNcDialog dInit( dialogROWS,     // number of display lines
                       dialogCOLS,     // number of display columns
                       ulY,            // Y offset from upper-left of terminal 
                       ulX,            // X offset from upper-left of terminal
                       "  Select Test To Be Executed  ", // dialog title
                       ncltSINGLE,     // border line-style
                       dColor,         // border color attribute
                       dColor,         // interior color attribute
                       ic              // pointer to list of control definitions
                     ) ;

   //* Instantiate the dialog window *
   dp = new NcDialog ( dInit ) ;
   
      //* Open the dialog window *
      if ( (dp->OpenWindow()) == OK )
      {
         dialogOpen = true ;              // remember that dialog is open
         
         dp->DrawBox ( 1,  2, dialogROWS-5, 53, dColor ) ;  // enclose radio-button group
         //* Display instructions *
         dp->WriteString ( dialogROWS-4, 2, 
                        "Use Tab and Shift+Tab or arrow keys to scroll through", dColor ) ;
         dp->WriteString ( dialogROWS-3, 2, 
                        "options, then Space or Enter key to select;", dColor ) ;
         dp->WriteString ( dialogROWS-2, 2, 
                        "or select a test via 'hotkey' (underlined character)", dColor ) ;
         dp->RefreshWin () ;
   
         //* Group all radio buttons in the dialog *
         short XorGroup[] = { indexT01, indexT02, indexT03, indexT04, indexT05, 
                              indexT06, indexT07, indexT08, indexT09, indexT10, 
                              indexHELP, indexEXIT, -1 } ;
         dp->GroupRadiobuttons ( XorGroup ) ;
      }
   }           // If(dialogOpen==false)
   
   //* Un-hide the existing dialog window *
   else
   {
      dp->RefreshWin () ;
   }

   if ( dialogOpen != false )
   {
      //* Give the focus to the currently 'selected' button *
      short sIndex = dp->GetRbGroupSelection ( indexEXIT ) ;
      short cIndex = dp->GetCurrControl () ;
      while ( cIndex != sIndex )
         cIndex = dp->PrevControl () ;

      uiInfo Info ;                       // user interface info
      bool  testSelected = false ;        // loop control
      while ( !testSelected )
      {
         //* Move among the members of this Radio Button group *
         //* allowing user to select a test to be executed.    *
         // NOTE: This interface loop is fairly simple because *
         //       all controls in this dialog are Radio Buttons*
         //       and they all belong to the same button group.*
         dp->EditRadiobutton ( Info ) ;
         
         //* If a new selection has been made, OR  *
         //* if previous selection was re-selected.*
         if ( Info.dataMod || Info.keyIn == nckENTER )
            testSelected = true ;
      }
      testNum = Info.selMember ; // return value
   }

   //* If we are about to exit application, close the dialog. *
   if ( testNum == indexEXIT )
   {
      if ( dp != NULL )                      // close the window
         delete ( dp ) ;
      dp = NULL ;
   }
   //* Else, hide the window for future use *
   else
   {
      dp->HideWin () ;
   }

   return testNum ;

}  //* End TestMenu() *

//*************************
//*       Test01          *
//*************************
//******************************************************************************
//* This dialog is available for use as a Sandbox. It's a safe place to build  *
//* sand castles and let your imagination fly free!                            *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************
//* Notes:                                                                     *
//*                                                                            *
//******************************************************************************

void Test01 ( void )
{
   StatWin ( "Test #01 : - Sandbox" ) ;

   const short dialogROWS = 24 ;    // display lines
   const short dialogCOLS  = 80 ;   // display columns

   attr_t dColor = nc.blR,          // dialog background color
          bColor = nc.brbl ;        // dialog border color
   short ulY = minTestULY,          // upper left corner in Y
         ulX = 1 ;                  // upper left corner in X

InitCtrl ic[T1controlsDEFINED] =    // array of dialog control initialization objects
{
   {  //* 'DONE' pushbutton - - - - - - - - - - - - - - - - - - - -   T1donePB *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(dialogROWS - 2),        // ulY:       upper left corner in Y
      short(dialogCOLS / 2 - 4),    // ulX:       upper left corner in X
      1,                            // lines:     control lines
      8,                            // cols:      control columns
      "  DO^NE  ",                  // dispText:  
      nc.gyR,                       // nColor:    non-focus color
      nc.reG,                       // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      NULL,                         // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      NULL,                         // nextCtrl:  link in next structure
   },
} ;

   //* Initial parameters for dialog window *
   InitNcDialog dInit( dialogROWS,     // number of display lines
                       dialogCOLS,     // number of display columns
                       ulY,            // Y offset from upper-left of terminal 
                       ulX,            // X offset from upper-left of terminal 
                       "  Sandbox - Temporary Test Code  ", // dialog title
                       ncltDUAL,       // border line-style
                       bColor,         // border color attribute
                       dColor,         // interior color attribute
                       ic              // pointer to list of control definitions
                     ) ;

   //* Instantiate the dialog window *
   NcDialog* dp = new NcDialog ( dInit ) ;

   //* Open the dialog window *
   if ( (dp->OpenWindow()) == OK )
   {
      #if 1    // TESTING HERE - TESTING HERE - TESTING HERE
      //-------------
      
      
      
      #endif   // TESTING HERE - TESTING HERE - TESTING HERE

      dp->RefreshWin () ;           // make everything visible

      //* Interact with user *
      uiInfo Info ;                 // user interface data returned here
      short  icIndex = ZERO ;       // index of control with input focus
      bool   done = false ;         // loop control
      while ( ! done )
      {
         if ( ic[icIndex].type == dctPUSHBUTTON )
         {
            if ( Info.viaHotkey )         // arrived via hotkey
               Info.HotData2Primary () ;  // unpack the data
            else
               icIndex = dp->EditPushbutton ( Info ) ;
            if ( Info.dataMod != false )
            {
               if ( Info.ctrlIndex == T1donePB )
                  done = true ;
            }
         }
         else if ( ic[icIndex].type == dctTEXTBOX )
         {
            if ( Info.viaHotkey )         // arrived via hotkey
               Info.viaHotkey = false ;   // ignore hotkey data
            icIndex = dp->EditTextbox ( Info ) ;
         }
         else if ( ic[icIndex].type == dctBILLBOARD )
         {
            if ( Info.viaHotkey )         // arrived via hotkey
               Info.viaHotkey = false ;   // ignore hotkey data
            icIndex = dp->EditBillboard ( Info ) ;
         }
         else if ( ic[icIndex].type == dctRADIOBUTTON )
         {
            if ( Info.viaHotkey )         // arrived via hotkey
               Info.HotData2Primary () ;  // unpack the data
            else
               icIndex = dp->EditRadiobutton ( Info ) ;
         }
         else if ( ic[icIndex].type == dctSCROLLBOX )
         {
            if ( Info.viaHotkey )         // arrived via hotkey
               Info.viaHotkey = false ;   // ignore hotkey data
            icIndex = dp->EditScrollbox ( Info ) ;
         }
         else if ( ic[icIndex].type == dctDROPDOWN )
         {
            if ( Info.viaHotkey )         // arrived via hotkey
               Info.viaHotkey = false ;   // ignore hotkey data
            icIndex = dp->EditDropdown ( Info ) ;
         }
         else if ( ic[icIndex].type == dctMENUWIN )
         {
            if ( Info.viaHotkey )         // arrived via hotkey
               Info.viaHotkey = false ;   // ignore hotkey data
            icIndex = dp->EditMenuwin ( Info ) ;
         }
         else if ( ic[icIndex].type == dctSCROLLEXT )
         {
            if ( Info.viaHotkey )         // arrived via hotkey
               Info.viaHotkey = false ;   // ignore hotkey data
            icIndex = dp->EditScrollext ( Info ) ;
         }
         else if ( ic[icIndex].type == dctSPINNER )
         {
            if ( Info.viaHotkey )         // arrived via hotkey
               Info.viaHotkey = false ;   // ignore hotkey data
            icIndex = dp->EditSpinner ( Info ) ;
         }

         //* Move input focus to next/previous control.*
         if ( done == false && Info.viaHotkey == false )
         {
            if ( Info.keyIn == nckSTAB )
               icIndex = dp->PrevControl () ; 
            else
               icIndex = dp->NextControl () ;
         }
      }  // while()
   }
   else
   {  //* Most likely cause of dialog not opening is that the terminal window  *
      //* is too small. Give user a clue...                                    *
      short neededLines = (dialogROWS + minTestULY) >= minTERMROWS ?
                          (dialogROWS + minTestULY) : minTERMROWS, 
            neededCols  = dialogCOLS >= minTERMCOLS ? dialogCOLS : minTERMCOLS ;
      gString gs ;
      gs.compose ( L" Sorry, size of terminal window must "
                    "be at least %hd x %hd for this test. ", 
                   &neededCols, &neededLines ) ;
      StatWin ( gs.gstr(), nc.cybr | ncbATTR ) ;
      sleep ( 5 ) ;
   }

   if ( dp != NULL )                         // close the window
      delete ( dp ) ;

}  //* End Test01() *

//*************************
//*  T01_ControlUpdate    *
//*************************
//******************************************************************************
//* This is a callback method for manually updating the dialog controls.       *
//*                                                                            *
//*  For this test, the following are updated by the callback method:          *
//*   1.                                                                       *
//*   2.                                                                       *
//*   3.                                                                       *
//*                                                                            *
//* Input  : currIndex: index of control that currently has focus              *
//*          wkey     : user's key input data                                  *
//*          firstTime: the EstablishCallback() method calls this method once  *
//*                     with firstTime==true, to perform any required          *
//*                     initialization. Subsequently, the NcDialog class       *
//*                     always calls with firstTime==false.                    *
//* Returns: OK                                                                *
//******************************************************************************
//* Notes:                                                                     *
//******************************************************************************

short T01_ControlUpdate ( const short currIndex, const wkeyCode wkey, bool firstTime )
{

   return OK ;

}  //* End T01_ControlUpdate() *

//*************************
//*       Test02          *
//*************************
//******************************************************************************
//* This dialog tests the functionality of the NcDialog-class UserAlert()      *
//* method.  The noise is annoying, but making the noise is fun!               *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************
//* Notes:                                                                     *
//******************************************************************************

void Test02 ( void )
{
   StatWin ( "Test #02 : 'A noise annoys an oyster!'" ) ;

   const short dialogROWS = 14 ;    // display lines
   const short dialogCOLS = 70 ;    // display columns
   
   enum Controls : short
   {
       testPB = ZERO, donePB, countSP, intervalSP, repeatSP, delaySP, fancyPB, 
       controlsDEFINED
   } ;
   attr_t dColor  = nc.blR,         // dialog background color
          pnColor = nc.gyR,         // pushbutton non-focus color
          pfColor = nc.gyre|ncbATTR,// pushbutton non-focus color
          snColor = nc.bw,          // spinner non-focus color
          sfColor = nc.gycy|ncbATTR,// spinner focus color
          siColor = nc.gygr,        // spinner indicator color
          emColor = nc.brbl ;       // emphasis color
   short ctrY   = termRows/2,       // dialog center in Y
         ctrX   = termCols/2,       // dialog center in X
   //* Upper left corner in Y (cannot obscure status window) *
   ulY = (ctrY - dialogROWS/2) >= minTestULY ? 
         (ctrY - dialogROWS/2) : minTestULY,
   ulX = ctrX - dialogCOLS / 2 ; // upper left corner in X
   const dspinData cspData(  1, 32768, 5, dspinINTEGER, siColor ), 
                   ispData(  minUAINTERVAL, 32768, 4, dspinINTEGER, siColor ), 
                   rspData( -1, 32768, 0, dspinINTEGER, siColor ), 
                   dspData(  0, 32768, 0, dspinINTEGER, siColor ) ;
   const short fancyPattern[] = 
   { 9, minUAINTERVAL, minUAINTERVAL, minUAINTERVAL, 9, 18, 10, 9 } ;


InitCtrl ic[controlsDEFINED] =   // array of dialog control initialization objects
{
   {  //* 'TEST' pushbutton - - - - - - - - - - - - - - - - - - -       testPB *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(dialogROWS - 4),        // ulY:       upper left corner in Y
      short(dialogCOLS / 2 - 4),    // ulX:       upper left corner in X
      1,                            // lines:     control lines
      8,                            // cols:      control columns
      "  TEST  ",                   // dispText:  
      pnColor,                      // nColor:    non-focus color
      pfColor,                      // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      NULL,                         // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[donePB],                  // nextCtrl:  link in next structure
   },
   {  //* 'DONE' pushbutton - - - - - - - - - - - - - - - - - - -       donePB *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      ic[testPB].ulY,               // ulY:       upper left corner in Y
      short(ic[testPB].ulX + ic[testPB].cols + 4), // ulX:
      1,                            // lines:     control lines
      8,                            // cols:      control columns
      "  DONE  ",                   // dispText:  
      pnColor,                      // nColor:    non-focus color
      pfColor,                      // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      NULL,                         // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[countSP],                 // nextCtrl:  link in next structure
   },
   { //* 'Count' Spinner   - - - - - - - - - - - - - - - - -           countSP *
      dctSPINNER,                   // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      7,                            // ulY:       upper left corner in Y
      8,                            // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      6,                            // cols:      control columns
      NULL,                         // dispText:  (n/a)
      snColor,                      // nColor:    non-focus color
      sfColor,                      // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "Ping Count",                 // label:     
      -1,                           // labY:      
      -2,                           // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      &cspData,                     // spinData:  spinner init
      true,                         // active:    allow control to gain focus
      &ic[intervalSP],              // nextCtrl:  link in next structure
   },
   { //* 'Interval' Spinner    - - - - - - - - - - - - - - -        intervalSP *
      dctSPINNER,                   // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      ic[countSP].ulY,              // ulY:       upper left corner in Y
      short(ic[countSP].ulX + 16),  // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      6,                            // cols:      control columns
      NULL,                         // dispText:  (n/a)
      snColor,                      // nColor:    non-focus color
      sfColor,                      // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "Ping Interval\n\n   (0.05s)",// label:     
      -1,                           // labY:      
      -3,                           // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      &ispData,                     // spinData:  spinner init
      true,                         // active:    allow control to gain focus
      &ic[repeatSP],                // nextCtrl:  link in next structure
   },
   { //* 'Repeat' Spinner    - - - - - - - - - - - - - - - -          repeatSP *
      dctSPINNER,                   // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      ic[intervalSP].ulY,           // ulY:       upper left corner in Y
      short(ic[intervalSP].ulX +16),// ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      6,                            // cols:      control columns
      NULL,                         // dispText:  (n/a)
      snColor,                      // nColor:    non-focus color
      sfColor,                      // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "Seq. Repeat",                // label:     
      -1,                           // labY:      
      -2,                           // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      &rspData,                     // spinData:  spinner init
      true,                         // active:    allow control to gain focus
      &ic[delaySP],                 // nextCtrl:  link in next structure
   },
   { //* 'Delay' Spinner    - - - - - - - - - - - - - - - - -          delaySP *
      dctSPINNER,                   // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      ic[repeatSP].ulY,             // ulY:       upper left corner in Y
      short(ic[repeatSP].ulX + 16), // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      6,                            // cols:      control columns
      NULL,                         // dispText:  (n/a)
      snColor,                      // nColor:    non-focus color
      sfColor,                      // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "Seq. Delay\n\n  (0.10s)",   // label:     
      -1,                           // labY:      
      -2,                           // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      &dspData,                     // spinData:  spinner init
      true,                         // active:    allow control to gain focus
      &ic[fancyPB],                 // nextCtrl:  link in next structure
   },
   {  //* 'FANCY TEST' pushbutton - - - - - - - - - - - - - - - -      fancyPB *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      ic[testPB].ulY,               // ulY:       upper left corner in Y
      short(ic[testPB].ulX - 16),   // ulX:
      1,                            // lines:     control lines
      12,                           // cols:      control columns
      " FANCY TEST ",               // dispText:  
      pnColor,                      // nColor:    non-focus color
      pfColor,                      // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      NULL,                         // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      NULL,                         // nextCtrl:  link in next structure
   },
} ;
   //* Initial parameters for dialog window *
   InitNcDialog dInit( dialogROWS,     // number of display lines
                       dialogCOLS,     // number of display columns
                       ulY,            // Y offset from upper-left of terminal 
                       ulX,            // X offset from upper-left of terminal 
                       NULL,           // dialog title
                       ncltDUAL,       // border line-style
                       nc.brbl,        // border color attribute
                       dColor,         // interior color attribute
                       ic              // pointer to list of control definitions
                     ) ;

   //* Instantiate the dialog window *
   NcDialog* dp = new NcDialog ( dInit ) ;

   //* Open the dialog window *
   if ( (dp->OpenWindow()) == OK )
   {
      dp->SetDialogTitle ( "  Audible Alerts  ", nc.gybr | ncbATTR ) ;
      dp->WriteParagraph ( 2, 1, 
         "       Music, it's not, but it will get the user's attention.\n"
         "     The actual tone produced for the ping is system dependent.\n"
         "     Set call parameters as desired, then press 'Test' button.",
         dColor ) ;

      dp->WriteString ( dialogROWS - 2, 13,
                        "(press any key to interrupt test in progress)", emColor ) ;

      dp->RefreshWin () ;

      //* Interact with user *
      uiInfo Info ;                 // user interface data returned here
      short  icIndex = ZERO ;       // index of control with input focus
      bool   done = false ;         // loop control
      while ( ! done )
      {
         //*******************************************
         //* If focus is currently on a Pushbutton   *
         //*******************************************
         if ( ic[icIndex].type == dctPUSHBUTTON )
         {
            icIndex = dp->EditPushbutton ( Info ) ;
            if ( Info.dataMod != false )
            {
               if ( Info.ctrlIndex == donePB )
                  done = true ;
               else if ( Info.ctrlIndex == testPB )
               {
                  //* Use contents of spinner controls to     *
                  //* initialize AudibleAlert object for call.*
                  int c, i, r, d ;
                  dp->GetSpinnerValue ( countSP, c ) ;
                  dp->GetSpinnerValue ( intervalSP, i ) ;
                  dp->GetSpinnerValue ( repeatSP, r ) ;
                  dp->GetSpinnerValue ( delaySP, d ) ;
                  AudibleAlert aa( c, i, r, d ) ;
                  dp->UserAlert ( &aa ) ;
               }
               else if ( Info.ctrlIndex == fancyPB )
               {
                  AudibleAlert ab ( 8, 2, 1, 2, fancyPattern ) ;
                  dp->UserAlert ( &ab ) ;
               }
            }
         }
         //*******************************************
         //* If focus is currently on a Spinner      *
         //*******************************************
         if ( ic[icIndex].type == dctSPINNER )
         {
            icIndex = dp->EditSpinner ( Info ) ;
         }
         else
         { /* no other control types defined for this method */ }

         //* Move input focus to next/previous control.*
         if ( done == false && Info.viaHotkey == false )
         {
            if ( Info.keyIn == nckSTAB )
               icIndex = dp->PrevControl () ; 
            else
               icIndex = dp->NextControl () ;
         }
      }  // while()

   }
   if ( dp != NULL )                         // close the window
      delete ( dp ) ;

}  //* End Test02() *

//*************************
//*       Test03          *
//*************************
//******************************************************************************
//* This dialog tests the functionality of the NCurses-class mouse methods.    *
//*                                                                            *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************
//* Notes:                                                                     *
//******************************************************************************

void Test03 ( void )
{
   StatWin ( "Test #03 : - NCurses-class Mouse Methods" ) ;

   short neededLines = (ncmROWS + minTestULY) >= minTERMROWS ?
                       (ncmROWS + minTestULY) : minTERMROWS, 
         neededCols  = ncmCOLS >= minTERMCOLS ? ncmCOLS : minTERMCOLS ;
   if ( (termRows >= neededLines) && (termCols >= neededCols) )
   {
      CMouseTest* cmTest = new CMouseTest ( termRows, termCols, minTestULY ) ;
      if ( (cmTest->cmDialogOpened ()) != false )
      {
         cmTest->cmInteract () ;
      }
      else { /* Memory allocation error - unlikely */ }

      if ( cmTest != NULL )   // close the dialog
         delete cmTest ;
   }
   else
   {  //* Most likely cause of dialog not opening is that the terminal window  *
      //* is too small. Give user a clue...                                    *
      gString gs ;
      gs.compose ( L" Sorry, size of terminal window must "
                    "be at least %hd x %hd for this test. ", 
                   &neededCols, &neededLines ) ;
      StatWin ( gs.gstr(), nc.cybr | ncbATTR ) ;
      sleep ( 5 ) ;
   }

}  //* End Test03() *

//*************************
//*       Test04          *
//*************************
//******************************************************************************
//* This dialog tests the functionality of the NcDialog-class mouse methods.   *
//*                                                                            *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************
//* Notes: Size of terminal needed is larger than dialog because one control   *
//*        extends below the dialog window.                                    *
//******************************************************************************

void Test04 ( void )
{
   StatWin ( "Test #04 : - NcDialog-class Mouse Methods" ) ;

   short neededLines = (ndmROWS + minTestULY + 5) >= minTERMROWS ?
                       (ndmROWS + minTestULY + 5) : minTERMROWS, 
         neededCols  = ndmCOLS >= minTERMCOLS ? ndmCOLS : minTERMCOLS ;
   if ( (termRows >= neededLines) && (termCols >= neededCols) )
   {
      DMouseTest* dmTest = new DMouseTest ( termRows, termCols, minTestULY ) ;
      if ( (dmTest->dmDialogOpened ()) != false )
      {
         dmTest->dmInteract () ;
      }
      else { /* Memory allocation error - unlikely */ }

      if ( dmTest != NULL )   // close the dialog
         delete dmTest ;
   }
   else
   {  //* Most likely cause of dialog not opening is that the terminal window  *
      //* is too small. Give user a clue...                                    *
      gString gs ;
      gs.compose ( L" Sorry, size of terminal window must "
                    "be at least %hd x %hd for this test. ", 
                   &neededCols, &neededLines ) ;
      StatWin ( gs.gstr(), nc.cybr | ncbATTR ) ;
      sleep ( 5 ) ;
   }

}  //* End Test04() *

//*************************
//*       Test05          *
//*************************
//******************************************************************************
//* This dialog tests the functionality of the dctBILLBOARD control.           *
//* Please see BillboardTest.hpp and BillboardTest.cpp.                        *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************
//* Notes:                                                                     *
//******************************************************************************

void Test05 ( void )
{
   StatWin ( "Test #05 : - Billboard Controls" ) ;

   short neededLines = (btROWS + minTestULY) >= minTERMROWS ?
                       (btROWS + minTestULY) : minTERMROWS, 
         neededCols  = btCOLS >= minTERMCOLS ? btCOLS : minTERMCOLS ;
   if ( (termRows >= neededLines) && (termCols >= neededCols) )
   {
      BillboardTest* btTest = new BillboardTest ( termRows, termCols, minTestULY ) ;
      if ( (btTest->btDialogOpened ()) != false )
      {
         btTest->btInteract () ;
      }
      else { /* Memory allocation error - unlikely */ }

      if ( btTest != NULL )   // close the dialog
         delete btTest ;
   }
   else
   {  //* Most likely cause of dialog not opening is that the terminal window  *
      //* is too small. Give user a clue...                                    *
      gString gs ;
      gs.compose ( L" Sorry, size of terminal window must "
                    "be at least %hd x %hd for this test. ", 
                   &neededCols, &neededLines ) ;
      StatWin ( gs.gstr(), nc.cybr | ncbATTR ) ;
      sleep ( 5 ) ;
   }

}  //* End Test05() *

//*************************
//*       Test06          *
//*************************
//******************************************************************************
//* This dialog tests the functionality of configuring the dialog to draw      *
//* the control labels and dialog title as RTL (right-to-left) text.           *
//* Contents of all controls are also configured as RTL data.                  *
//* Please see RTL_ContentTest.hpp and RTL_ContentTest.cpp.                    *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************
//* Notes:                                                                     *
//* - One control of each control type is defined, and its label is configured *
//*   for display as RTL text. Control labels also include 'hotkey'            *
//*   specifications for a robust test.                                        *
//* - The dialog title is also displayed as RTL text.                          *
//* - Each control that has text content is initialized with data that is      *
//*   displayed as RTL text data.                                              *
//* - For simplicity and ease of debugging, most data are in English, BUT are  *
//*   constructed backward, so that they will be displayed in an understandable*
//*   way when written as RTL. This does not reduce the validity of the test.  *
//*   We may substitute actual RTL data later, but your author is not fluent   *
//*   in any of the RTL languages, which makes a realistic test more difficult.*
//*                                                                            *
//******************************************************************************

void Test06 ( void )
{
   StatWin ( "Opening Test 06 - RTL Title, Labels and Contents" ) ;

   short neededLines = (rtROWS + minTestULY) >= minTERMROWS ?
                       (rtROWS + minTestULY) : minTERMROWS, 
         neededCols  = rtCOLS >= minTERMCOLS ? rtCOLS : minTERMCOLS ;
   if ( (termRows >= neededLines) && (termCols >= neededCols) )
   {
      RTL_ContentTest* rtTest = 
                     new RTL_ContentTest ( termRows, termCols, minTestULY ) ;
      if ( (rtTest->rtDialogOpened ()) != false )
      {
         rtTest->rtInteract () ;
      }
      else { /* Memory allocation error - unlikely */ }

      if ( rtTest != NULL )   // close the dialog
         delete rtTest ;
   }
   else
   {  //* Most likely cause of dialog not opening is that the terminal window  *
      //* is too small. Give user a clue...                                    *
      gString gs ;
      gs.compose ( L" Sorry, size of terminal window must "
                    "be at least %hd x %hd for this test. ", 
                   &neededCols, &neededLines ) ;
      StatWin ( gs.gstr(), nc.cybr | ncbATTR ) ;
      sleep ( 5 ) ;
   }

}  //* End Test06() *

//*************************
//*  T06_ControlUpdate    *
//*************************
//******************************************************************************
//* This is a callback method for manually updating the dialog controls.       *
//*                                                                            *
//*  For this test, the following are updated by the callback method:          *
//*   1.                                                                       *
//*   2.                                                                       *
//*   3.                                                                       *
//*                                                                            *
//* Input  : currIndex: index of control that currently has focus              *
//*          wkey     : user's key input data                                  *
//*          firstTime: the EstablishCallback() method calls this method once  *
//*                     with firstTime==true, to perform any required          *
//*                     initialization. Subsequently, the NcDialog class       *
//*                     always calls with firstTime==false.                    *
//* Returns: OK                                                                *
//******************************************************************************
//* Notes:                                                                     *
//******************************************************************************

short T06_ControlUpdate ( const short currIndex, const wkeyCode wkey, bool firstTime )
{

   return OK ;

}  //* End T06_ControlUpdate() *

//*************************
//*       Test07          *
//*************************
//******************************************************************************
//* This dialog tests the functionality of                                     *
//*                                                                            *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************
//* Notes:                                                                     *
//******************************************************************************

void Test07 ( void )
{
   StatWin ( "Opening Test 07 - Thread Safety - I/O Stress Test" ) ;

   short neededLines = (ttROWS + minTestULY) >= minTERMROWS ?
                       (ttROWS + minTestULY) : minTERMROWS, 
         neededCols  = ttCOLS >= minTERMCOLS ? ttCOLS : minTERMCOLS ;
   if ( (termRows >= neededLines) && (termCols >= neededCols) )
   {
      ThreadTest* ttTest = 
                     new ThreadTest ( termRows, termCols, minTestULY ) ;
      if ( (ttTest->ttDialogOpened ()) != false )
      {
         ttTest->ttInteract () ;
      }
      else { /* Memory allocation error - unlikely */ }

      if ( ttTest != NULL )   // close the dialog
         delete ttTest ;
   }
   else
   {  //* Most likely cause of dialog not opening is that the terminal window  *
      //* is too small. Give user a clue...                                    *
      gString gs ;
      gs.compose ( L" Sorry, size of terminal window must "
                    "be at least %hd x %hd for this test. ", 
                   &neededCols, &neededLines ) ;
      StatWin ( gs.gstr(), nc.cybr | ncbATTR ) ;
      sleep ( 5 ) ;
   }

}  //* End Test07() *

//*************************
//*  T07_ControlUpdate    *
//*************************
//******************************************************************************
//* This is a callback method for manually updating the dialog controls.       *
//*                                                                            *
//*  For this test, the following are updated by the callback method:          *
//*   1.                                                                       *
//*   2.                                                                       *
//*   3.                                                                       *
//*                                                                            *
//* Input  : currIndex: index of control that currently has focus              *
//*          wkey     : user's key input data                                  *
//*          firstTime: the EstablishCallback() method calls this method once  *
//*                     with firstTime==true, to perform any required          *
//*                     initialization. Subsequently, the NcDialog class       *
//*                     always calls with firstTime==false.                    *
//* Returns: OK                                                                *
//******************************************************************************
//* Notes:                                                                     *
//******************************************************************************

short T07_ControlUpdate ( const short currIndex, const wkeyCode wkey, bool firstTime )
{

   return OK ;

}  //* End T07_ControlUpdate() *

//*************************
//*       Test08          *
//*************************
//******************************************************************************
//* This dialog tests the functionality of the NcDialog ShellOut() and         *
//* PseudoShell() methods. See ShellTest.cpp and ShellTest.hpp.                *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************
//* Notes:                                                                     *
//******************************************************************************

void Test08 ( void )
{
   StatWin ( "Opening Test 08 - ShellOut command and options" ) ;

   short neededLines = (soROWS + minTestULY) >= minTERMROWS ?
                       (soROWS + minTestULY) : minTERMROWS, 
         neededCols  = soCOLS >= minTERMCOLS ? soCOLS : minTERMCOLS ;
   if ( (termRows >= neededLines) && (termCols >= neededCols) )
   {
      ShellTest* soTest = 
                     new ShellTest ( termRows, termCols, minTestULY ) ;
      if ( (soTest->soDialogOpened ()) != false )
      {
         soTest->soInteract () ;
      }
      else { /* Memory allocation error - unlikely */ }

      if ( soTest != NULL )   // close the dialog
         delete soTest ;
   }
   else
   {  //* Most likely cause of dialog not opening is that the terminal window  *
      //* is too small. Give user a clue...                                    *
      gString gs ;
      gs.compose ( L" Sorry, size of terminal window must "
                    "be at least %hd x %hd for this test. ", 
                   &neededCols, &neededLines ) ;
      StatWin ( gs.gstr(), nc.cybr | ncbATTR ) ;
      sleep ( 4 ) ;
   }

}  //* End Test08() *

//*************************
//*       Test09          *
//*************************
//******************************************************************************
//* This dialog tests the functionality of the Progress-bar Widget.            *
//*                                                                            *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************
//* Notes:                                                                     *
//******************************************************************************

void Test09 ( void )
{
   StatWin ( "Opening Test 09 - Progress-bar Widget" ) ;

   short neededRows = (pbROWS + minTestULY + 2) >= minTERMROWS ?
                       (pbROWS + minTestULY + 2) : minTERMROWS, 
         neededCols  = minTERMCOLS ;
   if ( (termRows >= neededRows) && (termCols >= neededCols) )
   {
      ProgbarTest* pbTest = 
                     new ProgbarTest ( termRows, termCols, minTestULY ) ;
      if ( (pbTest->pbDialogOpened ()) != false )
      {
         pbTest->pbInteract () ;
      }
      else { /* Memory allocation error - unlikely */ }

      if ( pbTest != NULL )   // close the dialog
         delete pbTest ;
   }
   else
   {  //* Most likely cause of dialog not opening is that the terminal window  *
      //* is too small. Give user a clue...                                    *
      gString gs ;
      gs.compose ( L" Sorry, size of terminal window must "
                    "be at least %hd x %hd for this test. ", 
                   &neededCols, &neededRows ) ;
      StatWin ( gs.gstr(), nc.cybr | ncbATTR ) ;
      sleep ( 4 ) ;
   }

}  //* End Test09() *

//*************************
//*  T09_ControlUpdate    *
//*************************
//******************************************************************************
//* This is a callback method for manually updating the dialog controls.       *
//*                                                                            *
//*  For this test, the following are updated by the callback method:          *
//*   1.                                                                       *
//*   2.                                                                       *
//*   3.                                                                       *
//*                                                                            *
//* Input  : currIndex: index of control that currently has focus              *
//*          wkey     : user's key input data                                  *
//*          firstTime: the EstablishCallback() method calls this method once  *
//*                     with firstTime==true, to perform any required          *
//*                     initialization. Subsequently, the NcDialog class       *
//*                     always calls with firstTime==false.                    *
//* Returns: OK                                                                *
//******************************************************************************
//* Notes:                                                                     *
//******************************************************************************

short T09_ControlUpdate ( const short currIndex, const wkeyCode wkey, bool firstTime )
{

   return OK ;

}  //* End T09_ControlUpdate() *

//*************************
//*       Test10          *
//*************************
//******************************************************************************
//* This dialog tests the functionality of                                     *
//*                                                                            *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************
//* Notes:                                                                     *
//******************************************************************************

void Test10 ( void )
{
   StatWin ( "Test #10 : Miscellaneous Tests" ) ;

   //* Establish minimum terminal-window dimensions for these tests.*
   short neededLines = (twinROWS + 1), 
         neededCols  = twinCOLS ;
   if ( (termRows >= neededLines) && (termCols >= neededCols) )
   {
      expandTest *etPtr = new expandTest ( termRows, termCols, minTestULY ) ;
      if ( (etPtr->etDialogOpened ()) != false )
      {
         etPtr->etInteract () ;
      }
      else { /* Memory allocation error - unlikely */ }

      if ( etPtr != NULL )        // close the dialog
         delete etPtr ;
   }
   else
   {  //* Most likely cause of dialog not opening is that the terminal window  *
      //* is too small. Give user a clue...                                    *
      gString gs ;
      gs.compose ( L" Sorry, size of terminal window must be "
                    "at least %hd x %hd for this test. ", 
                   &neededCols, &neededLines ) ;
      StatWin ( gs.gstr(), nc.cyG ) ;
      chrono::duration<short>aWhile( 4 ) ;
      this_thread::sleep_for( aWhile ) ;
   }

}  //* End Test10() *

//*************************
//*  T10_ControlUpdate    *
//*************************
//******************************************************************************
//* This is a callback method for manually updating the dialog controls.       *
//* (currently unused)                                                         *
//*                                                                            *
//*  For this test, the following are updated by the callback method:          *
//*   1.                                                                       *
//*   2.                                                                       *
//*   3.                                                                       *
//*                                                                            *
//* Input  : currIndex: index of control that currently has focus              *
//*          wkey     : user's key input data                                  *
//*          firstTime: the EstablishCallback() method calls this method once  *
//*                     with firstTime==true, to perform any required          *
//*                     initialization. Subsequently, the NcDialog class       *
//*                     always calls with firstTime==false.                    *
//* Returns: OK                                                                *
//******************************************************************************
//* Notes:                                                                     *
//******************************************************************************

short T10_ControlUpdate ( const short currIndex, const wkeyCode wkey, bool firstTime )
{

   return OK ;

}  //* End T10_ControlUpdate() *

//*************************
//*  GetCommandLineArgs   *
//*************************
//******************************************************************************
//* Capture user's command-line arguments.                                     *
//*                                                                            *
//* Valid Arguments:                                                           *
//*   Test number: -[1 | 2 | 3 | 4 | 5 | 6 | 7 | 8 | 9 | 10]                   *
//*   Alt Locale : -A, -a     Specify an alternate 'locale' for interpretation *
//*                           of character output and input conversions.       *
//*   Help       : -H, -h     Start application with display of Help Dialog.   *
//*                                                                            *
//* Input  : commArgs class object (by reference)                              *
//*                                                                            *
//* Returns: test number to be executed OR -1 if no test specified             *
//******************************************************************************

short GetCommandLineArgs ( commArgs& ca )
{
   if ( ca.argCount > 1 )
   {
      short v = 1,         // argv index
            i ;            // current arg index

      while ( v < ca.argCount )
      {
         i = ZERO ;
         if ( ca.argList[v][i] == '-' )  ++i ;     // step over dashes
         if ( ca.argList[v][i] == '-' )  ++i ;

         //* Test number specified? *
         if ( ca.argList[v][i] >= '1' && ca.argList[v][i] <= '9' )
         {
            sscanf ( &ca.argList[v][i], "%hd", &ca.testNum ) ;
            --ca.testNum ; // make it a zero-based index
         }

         //* Alternate locale specified? *
         else if ( ca.argList[v][i] == 'A' || ca.argList[v][i] == 'a' )
         {  //* If user specified an alternate locale string, use it instead *
            //* of the default locale found in the terminal environment.     *
            ++i ;
            if ( ca.argList[v][i] != NULLCHAR )
            {
               if ( ca.argList[v][i] == '=' )
                  ++i ;
               strncpy ( ca.localeName, &ca.argList[v][i], 30 ) ;
               ca.altLocale = true ;
            }
            else if ( ca.argCount > 2 )
            {
               strncpy ( ca.localeName, ca.argList[v+1], 30 ) ;
               ca.altLocale = true ;
            }
         }

         //* Else, request for help OR invalid command-line argument *
         else
         {
            ca.testNum = indexHELP ;
            break ;
         }

         ++v ;          // next command-line argument
      }
   }
   return ca.testNum ;
   
}  //* End GetCommandLineArgs() *

//*************************
//*      HelpAbout        *
//*************************
//******************************************************************************
//* Display the application's Help-About window.                               *
//* This is a simple informational dialog with a 'CLOSE' button.               *
//*                                                                            *
//*                                                                            *
//* Input  : ctrY : position at which to center dialog in Y                    *
//*          ctrX : position at which to center dialog in X                    *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void HelpAbout ( short ctrY, short ctrX )
{
const gString DialogText( 
   "Dialog4, \n"
   "Copyright (c) %s Mahlon R. Smith, The Software Samurai\n"
   "                        Beijing University of Technology\n"
   " \n"
   "A simple utility for development and testing of the NcDialog family\n"
   "of classes based on the Linux/UNIX ncurses library.\n"
   " \n"
   "Command Line arguments (optional):\n"
   "    -n      where 'n' is a test number\n"
   "    -A, -a  specify alternate 'locale' Ex: -Awo_SN.utf8 \n"
   "    -H, -h  Help (this dialog)\n"
   " \n"
   "The API behind this utility provides a means of easily designing\n"
   "applications for curses, while in a modern, C++ environment.\n"
   "For bugs, suggestions, possible praise: http://www.SoftwareSam.us\n"
   " \n"
   "Tools: GNU G++13 (C++11) under Fedora and Ubuntu Linux.\n"
   "       Requires ncursesw library v:6.4+ and 'pthread' library.", crYears
);//xxxxxxxxx0xxxxxxxxx0xxxxxxxxx0xxxxxxxxx0xxxxxxxxx0xxxxxxxxx0xxxxxxxx|
const short dialogROWS = 23 ;       // # of display lines
const short dialogCOLS = 70 ;       // # of display columns
const short controlsDEFINED = 3 ;   // # of controls in dialog
enum ctrls { closePUSH = ZERO, classPUSH, appverTBOX } ;
short ulY = ctrY - dialogROWS / 2,
      ulX = ctrX - dialogCOLS / 2 ;
attr_t dColor = nc.blR ;

   //* Format application version string for display *
   gString verString( " %s ", AppVersion ) ;


InitCtrl ic[controlsDEFINED] =      // array of dialog control info
{
   {  //* 'CLOSE' pushbutton  - - - - - - - - - - - - -  closePUSH *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(dialogROWS - 2),        // ulY:       upper left corner in Y
      short(dialogCOLS / 2 - 4),    // ulX:       upper left corner in X
      1,                            // lines:     control lines
      7,                            // cols:      control columns
      " CLOSE ",                    // dispText:  
      nc.gyR,                       // nColor:    non-focus color
      nc.reR,                       // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "",                           // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[1],                       // nextCtrl:  link in next structure
   },
   {  //* 'CLASS VERSION' pushbutton  - - - - - - - - -  classPUSH *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(dialogROWS - 2),        // ulY:       upper left corner in Y
      short(dialogCOLS - 25),       // ulX:       upper left corner in X
      1,                            // lines:     control lines
      24,                           // cols:      control columns
      " DISPLAY CLASS VERSIONS ",   // dispText:  
      nc.gyR,                       // nColor:    non-focus color
      nc.reR,                       // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "",                           // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[2],                       // nextCtrl:  link in next structure
   },
   {  //* Non-active textbox for app. version  - - - -  appverTBOX *
      dctTEXTBOX,                   // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      1,                            // ulY:       upper left corner in Y
      19,                           // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      short(verString.gscols()),    // cols:      control columns
      verString.ustr(),             // dispText:  (value of t3indexO3)
      nc.bl,                        // nColor:    non-focus color
      nc.bl,                        // fColor:    focus color
      tbPrint,                      // filter:    any printing character
      "Version:",                   // label:     
      ZERO,                         // labY:      
      -9,                           // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      false,                        // active:    cannot gain focus
      NULL,                         // nextCtrl:  link in next structure
   },
} ;

   //* Initial parameters for dialog window *
   InitNcDialog dInit( dialogROWS,     // number of display lines
                       dialogCOLS,     // number of display columns
                       ulY,            // Y offset from upper-left of terminal 
                       ulX,            // X offset from upper-left of terminal 
                       NULL,           // dialog title
                       ncltSINGLE,     // border line-style
                       dColor,         // border color attribute
                       dColor,         // interior color attribute
                       ic              // pointer to list of control definitions
                     ) ;

   //* Instantiate the dialog window *
   NcDialog* dp = new NcDialog ( dInit ) ;

   //* Open the dialog window *
   if ( (dp->OpenWindow()) == OK )
   {
      dp->WriteParagraph ( 1, 1, DialogText, dColor ) ;
      dp->RefreshWin () ;

      uiInfo Info ;                 // user interface data returned here
      bool   done = false ;         // loop control
      while ( ! done )
      {
         dp->EditPushbutton ( Info ) ;
         if ( Info.dataMod != false )
         {
            if ( Info.ctrlIndex == closePUSH )
               done = true ;
            else if ( Info.ctrlIndex == classPUSH )
            {  //* Open a dialog to display class version numbers and other    *
               //* interesting data. Current dialog will be obscured, so save  *
               //* its display data.                                           *
               dp->SetDialogObscured () ;
               DisplayClassInfo ( (ulY + 4), (ctrX - 20), 15, 40 ) ;
               dp->NextControl () ;
               dp->RefreshWin () ;
            }
         }
         if ( done == false )
         {
            if ( Info.keyIn == nckSTAB )
               dp->PrevControl () ; 
            else if ( Info.keyIn != ZERO )
               dp->NextControl () ;
         }
      }
   }
   if ( dp != NULL )
      delete ( dp ) ;                        // close the window

}  //* End HelpAbout() *

//***********************
//* ClearTerminalWindow *
//***********************
//****************************************************************************
//* Clear the terminal window and display the application title.             *
//*                                                                          *
//*                                                                          *
//* Input Values : none                                                      *
//*                                                                          *
//* Return Value : none                                                      *
//****************************************************************************

void ClearTerminalWindow ( void )
{
   winPos   wpos( 0, 0 ) ;
   nc.ClearScreen () ;
   gString gsTitle( "  Dialog4 - v:%s (c) %s Software Samurai   : Locale: %s      ",
//   gString gsTitle( "  Dialog4 - v:%s (c) %s Mahlon R. Smith    : Locale: %s      ",
                    AppVersion, crYears, nc.GetLocale() ) ;
   nc.WriteString ( wpos, gsTitle.gstr(), nc.bw | ncuATTR ) ;
   nc.RefreshScreen () ;

}  //* End ClearTerminalWindow() *

//*************************
//*                       *
//*************************
//******************************************************************************
//*                                                                            *
//*                                                                            *
//*                                                                            *
//* Input  :                                                                   *
//*                                                                            *
//* Returns:                                                                   *
//******************************************************************************


