//******************************************************************************
//* File       : Chart.cpp                                                     *
//* Author     : Mahlon R. Smith                                               *
//*              Copyright (c) 2020-2025 Mahlon R. Smith, The Software Samurai *
//*                 GNU GPL copyright notice below                             *
//* Date       : 21-Mar-2025                                                   *
//* Version    : (see ChartVersion string below)                               *
//*                                                                            *
//* Description: Implementation of the "Chart" class.                          *
//*              This is a widget built on the NcDialog API by the same author.*
//*              Draw a bar chart representing the specified data.             *
//*                                                                            *
//******************************************************************************
//* Copyright Notice:                                                          *
//* This program is free software: you can redistribute it and/or modify it    *
//* under the terms of the GNU General Public License as published by the Free *
//* Software Foundation, either version 3 of the License, or (at your option)  *
//* any later version.                                                         *
//*                                                                            *
//* This program is distributed in the hope that it will be useful, but        *
//* WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY *
//* or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License   *
//* for more details.                                                          *
//*                                                                            *
//* You should have received a copy of the GNU General Public License along    *
//* with this program.  If not, see <http://www.gnu.org/licenses/>.            *
//*                                                                            *
//*         Full text of the GPL License may be found in the Texinfo           *
//*         documentation for this program under 'Copyright Notice'.           *
//******************************************************************************
//* Version History (most recent first):                                       *
//*                                                                            *
//* v: 0.0.04 30-Jul-2021                                                      *
//*   -- Implement ReloadDataset() method. This is used to update the data     *
//*      display when the data in application space has changed.               *
//*   -- Add a screen capture pass-through method. Enhanced screenshots for    *
//*      use in documentation.                                                 *
//*                                                                            *
//* v: 0.0.03 16-Jul-2021                                                      *
//*   -- All major features implemented and functional.                        *
//*   -- Add user-friendly enhancements.                                       *
//*      -- Variable height for header area                                    *
//*      -- Variable height for footer area                                    *
//*      -- Variable dimensions for margin area (left margin)                  *
//*      -- Implement method to prevent writes to header, footer and margin    *
//*         areas from overrunning area boundaries.                            *
//*   -- GetStats() returns statistics on the dataset.                         *
//*   -- Save and restore display for independent dialog window. This allows   *
//*      application dialogs to overlay the independent dialog without         *
//*      corrupting the display.                                               *
//*   -- Eliminate the 'barTips' member of the ChartDef class, and use a       *
//*      zero(0) value in the 'barWidth' member to indicate that only bar tips *
//*      should be displayed.                                                  *
//*   -- Create first draft of formal documentation. See "ncdialogapi.info" or *
//*      "ncdialogapi.html", chapter "Chart Widget".                           *
//*      -- Implement a conditional-compilation flag which allows an           *
//*         undocumented keycode in the ShiftData() method to perform screen   *
//*         captures for use in the documentation.                             *
//*                                                                            *
//* v: 0.0.02 01-Jul-2021                                                      *
//*   -- Integrate the Chart class into the NcDialog API, Dialog4 application. *
//*      Few changes have been initially required.                             *
//*   -- Add 'cartChar' member to the ChartDef class. This is the character    *
//*      used to plot Cartesian data points.                                   *
//*   -- Reformat the debugging output to make more room in the footer for     *
//*      user data.                                                            *
//*   -- Clean up the mess of temporary and experimental code.                 *
//*   -- Refine the mapping algorithm and institute layers of error checking.  *
//*                                                                            *
//* v: 0.0.01 01-Mar-2021                                                      *
//*   -- First effort is an extension and generalization of chart generation   *
//*      code in an early release of Exercalc, an exercise tracking application*
//*      by the same author.                                                   *
//*   -- Development of the Chart class is assisted by the Exercalc method     *
//*      "DebugChartWidget()". This method will be integrated into a future    *
//*      version of one of the NcDialog test applications.                     *
//*                                                                            *
//******************************************************************************
//* Notes:                                                                     *
//* ------                                                                     *
//* Arithmetic Mean:                                                           *
//* The arithmetic mean value is the sum of all elements of a data set,        *
//* divided by the number of elements in the set.                              *
//*   (Wikipedia)                                                              *
//*                                                                            *
//* Median Value:                                                              *
//* The median is the middle number of the group when they are ranked in order.*
//* (If there are an even number of numbers, the mean of the middle two is     *
//* taken.)                                                                    *
//* Thus to find the median, order the list according to its elements'         *
//* magnitude and then repeatedly remove the pair consisting of the highest    *
//* and lowest values until either one or two values are left. If exactly one  *
//* value is left, it is the median; if two values, the median is the          *
//* arithmetic mean of these two.                                              *
//*   (Wikipedia)                                                              *
//*                                                                            *
//* The median is sometimes used as opposed to the mean when there are         *
//* outliers in the sequence that might skew the average of the values.        *
//* The median of a sequence can be less affected by outliers than the mean.   *
//*                                                                            *
//*   ----  ----  ----  ----  ----  ----  ----  ----  ----  ----  ----  ----   *
//* Dialog data integrity:                                                     *
//* ----------------------                                                     *
//* The Chart-class object can be launched either:                             *
//*  a) in a new dialog which is controlled entirely by the Chart class, OR    *
//*  b) the object can be launched within an existing dialog.                  *
//* If the object is given a pointer to the parent dialog, all data within the *
//* specified chart area will be overwritten. It is the caller's responsibility*
//* to save and restore any data that will be overwritten. No display data     *
//* outside the specified area will be modified. (The exception is if the      *
//* dialog title is set.)                                                      *
//* In addition, it is important that no dialog controls be within the         *
//* specified chart area. Dialog controls are actually independent windows     *
//* which are positioned within the parent dialog. As such, they may be        *
//* refreshed asynchronously by various activities, which would interfere with *
//* the chart display.                                                         *
//*                                                                            *
//* In order to avoid conflicts, the Chart class calls very few NcDialog       *
//* methods. These methods are:                                                *
//*  this->dp->GetDialogDimensions()                                           *
//*  this->dp->SetDialogTitle()                                                *
//*  this->dp->WriteParagraph()                                                *
//*  this->dp->WriteString()                                                   *
//*  this->dp->WriteChar()                                                     *
//*  this->dp->DrawLine()                                                      *
//*  this->dp->DrawBox()                                                       *
//*  this->dp->GetKeyInput()                                                   *
//*  this->dp->UserAlert()                                                     *
//*  this->dp->RefreshWin ()                                                   *
//* None of these methods access dialog controls or modify the internal data   *
//* of the dialog.                                                             *
//*   ----  ----  ----  ----  ----  ----  ----  ----  ----  ----  ----  ----   *
//*                                                                            *
//* Future Development:                                                        *
//* -------------------                                                        *
//* -- Allow 'barWidth' to be 0-8, and also multiples of 8 (16, 32, 48, 64).   *
//*    This would implement ver wide bars                                      *
//* -- If 'barWidth' == 1-7, implement an option to use the bar-tip character  *
//*    to draw the terminating character.                                      *
//* -- Add a method to dump the current configuration to memory in addition    *
//*    to the dump to screen.                                                  *
//* -- Evaluate the use of colors, which colors are used for what type of text.*
//*                                                                            *
//* -- Improve display of terminating character when narrow bars are specified.*
//*    see the "quadrant" group of Unicode codepoints:                         *
//*                                                                            *
//*    U+2596  ▖  QUADRANT LOWER LEFT                                          *
//*    U+2597  ▗  QUADRANT LOWER RIGHT                                         *
//*    U+2598  ▘  QUADRANT UPPER LEFT                                          *
//*    U+2599  ▙  QUADRANT UPPER LEFT AND LOWER LEFT AND LOWER RIGHT           *
//*    U+259A  ▚  QUADRANT UPPER LEFT AND LOWER RIGHT                          *
//*    U+259B  ▛  QUADRANT UPPER LEFT AND UPPER RIGHT AND LOWER LEFT           *
//*    U+259C  ▜  QUADRANT UPPER LEFT AND UPPER RIGHT AND LOWER RIGHT          *
//*    U+259D  ▝  QUADRANT UPPER RIGHT                                         *
//*    U+259E  ▞  QUADRANT UPPER RIGHT AND LOWER LEFT                          *
//*    U+259F  ▟  QUADRANT UPPER RIGHT AND LOWER LEFT AND LOWER RIGHT          *
//*                                                                            *
//* -- Enhance ShiftData() to include an optional? loop-termination key and    *
//*    if specified, to ignore all non-defined keycodes.                       *
//*                                                                            *
//* --                                                                         *
//*                                                                            *
//*                                                                            *
//*                                                                            *
//*                                                                            *
//******************************************************************************

//****************
//* Header Files *
//****************
#include "Chart.hpp"          // class definition and support data

//***************
//* Definitions *
//***************

//* For debugging only: enable display of config data *
#define DEBUG_CONFIG (0)
#if DEBUG_CONFIG != 0
#define DEBUG_SHIFT (0)    // Display shift offset
static winPos wpShift( 0, 0 ) ;
#endif   // DEBUG_CONFIG

//* For debugging only: Display Y offsets for chart area *
#define DEBUG_Y_OFFSETS (0)
#if DEBUG_Y_OFFSETS != 0
static winPos wpOutside( 0, 0 )  ;
#endif   // DEBUG_Y_OFFSETS

//* Define a special, reserved keycode for capture of *
//* screenshots. See ShiftData() method for details.  *
#define DEBUG_SCREENSHOT (0)


//* Class version number *
static const char* const ChartVersion = "0.0.04" ;

//* Minimum useful height and width for coordinate grid *
static const short MIN_GRID_ROWS = 4 ;
static const short MIN_GRID_COLS = 12 ;

//* Percentage of chart area for display of minimum and maximum values *
static const double minBAR = 2.0 ;  // number of divisions displayed for minimum value

//* For Cartesian charts only: scaling multiplier for X coordinate used *
//* to make the aspect ratio of the plot appear (approximately) square. *
static const double xCONST = 1.5 ;

//***************
//* Prototypes  *
//***************
static short InvertFractionalCell ( short blockIndex ) ;
static attr_t InvertColorAttribute ( attr_t normColor ) ;

//***************
//* Local Data  *
//***************

//* Character definitions for drawing graphs *
static const short wcINDX = 7 ;         // index of whole-cell character
static const wchar_t hBlock[cellDIVS] = // for horizontal bar graphs
{
   0x0258F,    // ▏
   0x0258E,    // ▎
   0x0258D,    // ▍
   0x0258C,    // ▌
   0x0258B,    // ▋
   0x0258A,    // ▊
   0x02589,    // ▉
   0x02588,    // █
} ;
static const wchar_t vBlock[cellDIVS] = // for vertical bar graphs
{
   0x02581,    // ▁
   0x02582,    // ▂
   0x02583,    // ▃
   0x02584,    // ▄
   0x02585,    // ▅
   0x02586,    // ▆
   0x02587,    // ▇
   0x02588,    // █
} ;

//* Character used when full bar is not displayed, only the terminal point.*
static const wchar_t hTipChar = 0x025AE ;  // character '▮' for horizontal bars
static const wchar_t vTipChar = 0x025AC ;  // character '▬' for vertical bars


//*************************
//*        ~Chart         *
//*************************
//******************************************************************************
//* Destructor. Return all resources to the system.                            *
//*                                                                            *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

Chart::~Chart ( void )
{
   //* If dynamic allocation of a data array to hold   *
   //* converted data, return the memory to the system.*
   if ( this->dynoData && (this->dataPtr != NULL) )
   { delete [] this->dataPtr ; this->dataPtr = NULL ; this->dynoData = false ; }

   //* If locally-instantiated dialog, delete it now. *
   if ( this->localDlg && this->dp != NULL )
   { delete this->dp ; this->dp = NULL ; }

   #if DEBUG_Y_OFFSETS != 0
   //* Erase debugging info in dialog window but OUTSIDE the chart area.*
   if ( ! this->localDlg )
      this->WriteYOffsets ( wpOutside, true ) ;
   #endif   // DEBUG_Y_OFFSETS

}  //* End ~Chart() *

//*************************
//*         Chart         *
//*************************
//******************************************************************************
//* Full-initialization constructor. All data members set according to         *
//* provided parameters.                                                       *
//*                                                                            *
//* Input  : cdef   : (by reference) a fully-initialized instance of the       *
//*                   'ChartDef' class.                                        *
//*          refresh: (optional, 'false' by default)                           *
//*                   if 'false', setup is performed, but display not updated  *
//*                   if 'true',  after setup, update display immediately      *
//*                                                                            *
//* Returns: implicitly returns a pointer to instantiated object               *
//******************************************************************************

Chart::Chart ( const ChartDef& cdef, bool refresh )
{
   //* Configure all data members *
   this->Configure ( cdef ) ;

   bool isOpen = false ;            // 'true' if target window is open

   if ( this->localDlg && this->dp == NULL ) // if we are to open a dialog locally
   {
      InitNcDialog id
         {
            this->ldRows,              // dialog rows (expanded to include borders)
            this->ldCols,              // dialog columns (expanded to include borders)
            this->wpTerm.ypos,         // dialog position Y
            this->wpTerm.xpos,         // dialog position X
            this->titText,             // title (if any)
            this->bStyle,              // border style
            this->bdrColor,            // border color
            this->dColor,              // interior color
            NULL                       // no controls
         } ;
      this->dp = new NcDialog ( id ) ;
      if ( (this->dp->OpenWindow ()) == OK )
         isOpen = true ;
   }
   else                             // if caller has provided a dialog
   {
      //* Defensive programming: test whether dialog is accessible *
      short py = -1, px = -1 ;
      this->dp->GetDialogDimensions ( py, px) ;
      if ( py >= ZERO && px >= ZERO ) 
         isOpen = true ;
   }

   //* If we have a valid target window *
   if ( isOpen )
   {
      //* If specified, set the dialog title *
      if ( this->titText != NULL )
         this->dp->SetDialogTitle ( this->titText, this->titColor ) ;

      //* Clear the display area,and if specified, draw a border.*
      this->ClearArea () ;

      #if DEBUG_CONFIG != 0
      //* Display configuration in target dialog.*
      this->DumpCfg () ;
      #endif   // DEBUG_CONFIG

      //* Draw the static text (if any) *
      if ( this->hdrText != NULL )
         this->DrawHeaderText ( this->hdrText, this->tColor ) ;
      if ( this->ftrText != NULL )
         this->DrawFooterText ( this->ftrText, this->tColor ) ;
      if ( this->marText != NULL )
         this->DrawMarginText ( this->marText, this->tColor ) ;

      //* Draw the vertical and horizontal axes *
      //* and axis labels (if provided).        *
      this->DrawGrid ( true ) ;

      //* Display the data *
      if ( (this->dataPtr != NULL) && (this->dataCount > ZERO) )
         this->MapData2Grid () ;

      //* If specified, make everything visible *
      if ( refresh )
         this->dp->RefreshWin () ;
   }

}  //* End Chart() *

//*************************
//*       ShiftData       *
//*************************
//******************************************************************************
//* Interact with user to shift the data visible in the chart window forward   *
//* and backward through the data array.                                       *
//* An audible alert (beep) is generated if unable to perform user-specified)  *
//* specified shift. The audible alert can be disabled via AudibleShift().     *
//*                                                                            *
//* If all data are currently displayed, returns immediately.                  *
//* Otherwise, returns when an un-handled keycode is received.                 *
//*                                                                            *
//* This method translates the defined keycodes (defaults listed below) into   *
//* parameters used by the private ShiftData() method, below.                  *
//*                                                                            *
//* If the 'sdPtr' argument is not provided, then the following default        *
//* shift-operation/keycode combinations are used.                             *
//*    Shift Operation           Keycode              Key type                 *
//*    -----------------------   -------------------------------               *
//*    top of data             : Home key (sbFirstPage)                        *
//*    end of data             : End key  (sbLastPage)                         *
//*    next page of data       : PageDown (sbNextPage)                         *
//*    previous page of data   : PageUp   (sbPrevPage)                         *
//*     1 step toward end      : RightArrow           wktFUNKEY                *
//*     1 step toward top      : LeftArrow            wktFUNKEY                *
//*     5 steps toward end     : Shift + RightArrow   wktFUNKEY                *
//*     5 steps toward top     : Shift + LeftArrow    wktFUNKEY                *
//*    10 steps toward end     : Ctrl + RightArrow    wktFUNKEY                *
//*    10 steps toward top     : Ctrl + LeftArrow     wktFUNKEY                *
//*                                                                            *
//* Input  : wkey : (by reference) receives the unhandled                      *
//*                 keycode that triggered return                              *
//*          sdCnt: (optional, ZERO by default)                                *
//*                 If specified, this is the number of objects in the         *
//*                 'sdPtr' array.                                             *
//*          sdPtr: (optional, NULL pointer by default)                        *
//*                 If specifed, this is a pointer to an array of ShiftDef     *
//*                 objects which define the valid shift options and           *
//*                 associated keycodes. See documentation for examples.       *
//*                                                                            *
//* Returns: index of first currently-displayed data item                      *
//*          'wkey' contains the unhandled key input. Note:                    *
//*          if all data currently displayed, wkey.key==nullchar               *
//******************************************************************************
//* Additional Notes:                                                          *
//* -- Secondary keycode CTRL+R is used to load an alternate dataset into the  *
//*    active Chart object.                                                    *
//* -- Hidden keycode CTRL+ALT+P is used to capture a screenshot of the dialog.*
//*    This is intended primarily for creating documentation.                  *
//*                                                                            *
//*                                                                            *
//******************************************************************************

int32_t Chart::ShiftData ( wkeyCode& wkey, short sdCnt, ShiftDef* sdPtr )
{
   int32_t    shiftCnt = ZERO ;        // parameters for call to private shiftData()
   ShiftBlock shiftBlk = sbNoShift ;
   short indx ;                        // index into sdPtr array
   bool dfltKeys = false,              // 'true' if default key definitions used
        done = false ;                 // loop control

   //* If all data are currently displayed, return immediately.*
   //* For these chart types, all data are currently displayed *
   //* because data have been scaled to fit the grid.          *
   if ( (this->hBars && this->dataCount <= this->vCells)  ||
        (!this->hBars && this->dataCount <= this->hCells) ||
        (this->chType == ctCartesian) ) 
   {
      wkey = { nckNULLCHAR, wktERR } ;
      done = true ;
   }

   //* Create the keycode-to-shift-operation map *
   if ( ! done && (sdPtr == NULL) )
   {
      sdCnt = 10 ;
      sdPtr = new ShiftDef[sdCnt]
      { //  wk.key     wk.type     sb           sc
         { {nckHOME,   wktFUNKEY}, sbFirstPage, ZERO },
         { {nckEND,    wktFUNKEY}, sbLastPage,  ZERO },
         { {nckPGDOWN, wktFUNKEY}, sbNextPage,  ZERO },
         { {nckPGUP,   wktFUNKEY}, sbPrevPage,  ZERO },
         { {nckRIGHT,  wktFUNKEY}, sbNoShift,     1  },
         { {nckLEFT,   wktFUNKEY}, sbNoShift,    -1  },
         { {nckSRIGHT, wktFUNKEY}, sbNoShift,     5  },
         { {nckSLEFT,  wktFUNKEY}, sbNoShift,    -5  },
         { {nckCRIGHT, wktFUNKEY}, sbNoShift,    10  },
         { {nckCLEFT,  wktFUNKEY}, sbNoShift,   -10  },
      } ;
      dfltKeys = true ;
   }

   while ( ! done )
   {
      shiftCnt = ZERO ;          // reset the call parameters
      shiftBlk = sbNoShift ;

      //* Get user input *
      this->dp->GetKeyInput ( wkey ) ;

      #if DEBUG_SCREENSHOT != 0  // save screenshot as text and HTML
      if ( (wkey.type == wktEXTEND) && (wkey.key == nckAS_P) )
      {
         this->dp->CaptureDialog ( "./capturedlg.txt" ) ;
         this->dp->CaptureDialog ( "./capturedlg.html", true, false, 
                                   "infodoc-styles.css", 4, false, nc.blR ) ;
         this->dp->UserAlert ( 3 ) ;
         continue ;
      }
      #endif   // DEBUG_SCREENSHOT

      //* If valid keycodes are defined locally, we silently remap     *
      //* certain keycodes that the user might confuse with valid keys.*
      //* If caller defined the key list, remapping does not apply.    *
      if ( dfltKeys )
      {
         if      ( wkey.key == nckDOWN )     wkey.key = nckRIGHT ;   // toward bottom
         else if ( wkey.key == nckSDOWN )    wkey.key = nckSRIGHT ;
         else if ( wkey.key == nckADOWN )    wkey.key = nckSRIGHT ;
         else if ( wkey.key == nckCDOWN )    wkey.key = nckCRIGHT ;
         else if ( wkey.key == nckARIGHT )   wkey.key = nckCRIGHT ;
         else if ( wkey.key == nckASDOWN )   wkey.key = nckPGDOWN ;
         else if ( wkey.key == nckACDOWN )   wkey.key = nckPGDOWN ;
         else if ( wkey.key == nckCSDOWN )   wkey.key = nckPGDOWN ;
         else if ( wkey.key == nckCSRIGHT )  wkey.key = nckPGDOWN ;
         else if ( wkey.key == nckASRIGHT )  wkey.key = nckPGDOWN ;
         else if ( wkey.key == nckACRIGHT )  wkey.key = nckPGDOWN ;
         else if ( wkey.key == nckUP )       wkey.key = nckLEFT ;    // toward top
         else if ( wkey.key == nckSUP )      wkey.key = nckSLEFT ;
         else if ( wkey.key == nckAUP )      wkey.key = nckSLEFT ;
         else if ( wkey.key == nckCUP )      wkey.key = nckCLEFT ;
         else if ( wkey.key == nckALEFT )    wkey.key = nckCLEFT ;
         else if ( wkey.key == nckASUP )     wkey.key = nckPGUP ;
         else if ( wkey.key == nckACUP )     wkey.key = nckPGUP ;
         else if ( wkey.key == nckCSUP )     wkey.key = nckPGUP ;
         else if ( wkey.key == nckCSLEFT )   wkey.key = nckPGUP ;
         else if ( wkey.key == nckASLEFT )   wkey.key = nckPGUP ;
         else if ( wkey.key == nckACLEFT )   wkey.key = nckPGUP ;
      }

      //* Scan for matching keycode *
      for ( indx = ZERO ; indx < sdCnt ; ++indx )
      {
         if ( (wkey.type == sdPtr[indx].wk.type) &&
              (wkey.key  == sdPtr[indx].wk.key) )
         {
            if ( (sdPtr[indx].sb == sbFirstPage) ||
                 (sdPtr[indx].sb == sbLastPage)  ||
                 (sdPtr[indx].sb == sbNextPage)  ||
                 (sdPtr[indx].sb == sbPrevPage) )
            {
               shiftBlk = sdPtr[indx].sb ;
               break ;
            }
            //* Set specific shift count *
            else if ( (sdPtr[indx].sb == sbNoShift) && 
                      (sdPtr[indx].sc != ZERO) )
            {
               shiftCnt = sdPtr[indx].sc ;
               break ;
            }
            //* Else, invalid ShiftDef definition, return to caller.*
            else
               done = true ;
         }
      }

      //* If no keycode match, return to caller.*
      if ( indx == sdCnt )
         done = true ;

      //* Execute the shift, beep if error returned.*
      if ( ! done )
      {
         if ( (this->shiftData ( shiftCnt, shiftBlk )) && this->audible )
            this->dp->UserAlert () ; // call user a dumbguy
      }
   }  // while()

   if ( dfltKeys && sdPtr != NULL )    // release local dynamic allocation
      delete [] sdPtr ;

   return ( this->dataOffset ) ;       // return current data offset (index)

}  //* End ShiftData() *

//*************************
//*     ReloadDataset     *
//*************************
//******************************************************************************
//* Replace the currently-displayed dataset with a new dataset.                *
//* Use this method to refresh the data displayed in the Chart object when the *
//* data are dynamically changing in application space, or to alternately      *
//* display different datasets.                                                *
//*                                                                            *
//* Note that only the dataset will be updated. Header, Footer and Margin      *
//* areas are not affected. The coordinate-grid style and dimensions are not   *
//* affected. Data-bar orientation is not affected.                            *
//*                                                                            *
//* It is an application's judgement call when changing datasets whether to    *
//* update the data in the existing Chart object, or to close the existing     *
//* Chart object and launch a new Chart object to display the new data.        *
//* It is also possible to have multiple, independent Chart objects and simply *
//* bring them into the foregroud on a rotating basis. From a performance      *
//* point-of-view, it is somewhat faster to update the data in an existing     *
//* Chart object because it cuts the number of calculations and system calls   *
//* by about 40%, but use your best judgement.                                 *
//*                                                                            *
//* Only four(4) parameters are updated by the ReloadDataset() call:           *
//*   1) the array of source data items                                        *
//*   2) the number of source data items                                       *
//*   3) the type of source data                                               *
//*   4) for Cartesian charts ONLY, indicate whether data                      *
//*      are X/Y pairs or Y/X pairs.                                           *
//*                                                                            *
//* Input  : dPtr     : pointer to new dataset                                 *
//*          dCount   : number of items in 'dataPtr' array                     *
//*          dType    : data type (member of enum idataType)                   *
//*          YX_pairs : (optional, 'false' by default)                         *
//*                     (for Cartesian chart type only)                        *
//*                     'false' == X/Y pairs                                   *
//*                     'true'  == Y/X pairs                                   *
//*                                                                            *
//* Returns: 'true' if data successfully updated                               *
//*          'false' if invalid parameter                                      *
//******************************************************************************

bool Chart::ReloadDataset ( const void *dPtr, int32_t dCount,
                            idataType dType, bool YX_pairs )
{
   bool status = false ;            // return value

   //* Validate caller's data *
   if ( dPtr == NULL )
      dCount = ZERO ;
   if ( dCount < ZERO )
      dCount = ZERO ;
   //* Protect against an odd number of values *
   //* when interpreting as coordinate pairs.  *
   if ( (this->chType == ctCartesian) && ((dCount % 2) != ZERO) )
      --dCount ;
   //* If offset is no longer in range, set offset to zero.*
   if ( this->dataOffset >= dCount )
      this->dataOffset = ZERO ;

   //* If existing dynamic allocation, release it *
   if ( this->dynoData )
   { delete [] this->dataPtr ; this->dataPtr = NULL ; this->dynoData = false ; }

   //* Initialize the source data as an array of *
   //* double-precision floating point values.   *
   if ( (this->ConvertData ( dPtr, dCount, dType, this->dataPtr )) )
      this->dataCount = dCount ;

   //* For Cartesian charts only, initialize the pair-order flag.*
   if ( this->chType == ctCartesian )
      this->hBars = YX_pairs ;

   //* Do a preliminary scan of the data to set minimum  *
   //* and maximum values, data range and average value. *
   if ( this->dataCount > ZERO )
      this->SetRange () ;

   //* Re-scale the data to available grid divisions.*
   this->perDiv = this->rngVal / (this->maxDiv - this->minDiv) ;
   if ( this->perDiv < 1.0 )
      this->perDiv = 1.0 ;

   //* Clear the grid area and re-draw the grid.*
   this->ClearGrid () ;

   #if DEBUG_CONFIG != 0
   //* Display configuration in target dialog.*
   if ( this->ftrRows > ZERO )
   {
      this->ClearArea ( this->ftrPos, this->ftrRows, this->ftrCols, this->tColor ) ;
      this->DumpCfg () ;
   }
   #endif   // DEBUG_CONFIG

   //* Display the data *
   if ( (this->dataPtr != NULL) && (this->dataCount > ZERO) )
      this->MapData2Grid () ;
   this->dp->RefreshWin () ;     // make changes visible

   return status ;

}  //* End ReloadDataset() *

//*************************
//*  GetHeaderDimensions  *
//*************************
//******************************************************************************
//* Get dimensions of header area.                                             *
//* The vertical dimension EXCLUDES the row reserved for the axis label.       *
//*                                                                            *
//* Input  : rows   : (by reference) receives number of display rows           *
//*          cols   : (by reference) receives number of display columns        *
//*                                                                            *
//* Returns: 'true' if successful                                              *
//*          'false' if target area not defined (rows == ZERO)                 *
//******************************************************************************

bool Chart::GetHeaderDimensions ( short& rows, short& cols ) const
{

   rows  = this->hdrRows ;
   cols  = this->hdrCols ;
   return ( bool(rows > ZERO) ) ;

}  //* End GetHeaderDimensions() *

//*************************
//*    DrawHeaderText     *
//*************************
//******************************************************************************
//* Clear the header-text area and write the specified text to the header area.*
//*                                                                            *
//* Note: To clear the area without writing new text,                          *
//*       set 'txt' to an empty string ("").                                   *
//*                                                                            *
//* Input  : txt    : text to be written                                       *
//*          attr   : color attribute for text                                 *
//*          border : (optional, 'false' by default)                           *
//*                   if 'true', draw a horizontal line across the bottom row  *
//*                   of the header area. This creates a visual division       *
//*                   between the header and the chart grid.                   *
//*          refresh: (optional, 'false' by default)                           *
//*                   if 'false', do not refresh the display after writing     *
//*                   if 'true',  refresh the display (make changes visible)   *
//*                                                                            *
//* Returns: 'true' if successful                                              *
//*          'false' if target area not defined                                *
//******************************************************************************

bool Chart::DrawHeaderText ( const char* txt, attr_t txtAttr, 
                             bool border, bool refresh )
{
   gString gs ;                     // text formatting
   bool  status = false ;           // return value

   //* Verify that target area is defined and clear the target area.*
   if ( (this->ClearArea ( this->hdrPos, this->hdrRows, this->hdrCols, txtAttr )) )
   {
      //* Range-check the text, then write it to target area *
      if ( (status = this->TrimText ( txt, gs, this->hdrRows, this->hdrCols )) )
         this->dp->WriteParagraph ( this->hdrPos, gs, txtAttr ) ;

      //* If specified, draw the horizontal dividing line *
      //* on last row of header area. Note that this may  *
      //* overwrite some or all the text we wrote above,  *
      //* but that is the caller's responsibility.        *
      if ( border )
         this->DrawDivider ( true, this->bStyle, (this->hdrRows - 1), this->bdrColor ) ;

      if ( refresh )
         this->dp->RefreshWin () ;
      status = true ;
   }
   return status ;

}  //* End DrawHeaderText() *

//*************************
//*    Add2HeaderText     *
//*************************
//******************************************************************************
//* Add text to the header area at the specified offset. For example, display  *
//* of instructions or explanations at different positions or in different     *
//* colors.                                                                    *
//* -- See GetHeaderDimensions() for obtaining the dimensions of the area.     *
//* -- Unlike the DrawHeaderText() method above, this method DOES NOT clear    *
//*    the header area before writing. Instead, the specified text is added    *
//*    to any existing text in the header area.                                *
//* -- The start position (pos) is validated to be within the target area.     *
//*    If start position is out-of-range, the text will not be written, and a  *
//*    cursor offset of 0:0 will be returned to indicate the error.            *
//* -- If any portion of the source text (txt) would overrun the boundaries of *
//*    the target area, that portion of the text will be silently discarded.   *
//* -- If all source text is displayed successfully, the offset to the space   *
//*    following the text will be returned.                                    *
//*    If, however, some of the text was discarded due to attempted boundary   *
//*    violations, then the cursor offset returned may not be as expected.     *
//*    For example, if the text was truncated to avoid overrunning the right   *
//*    edge of the target area, the X offset will reference the right edge of  *
//*    the target area. Similarly, if the text was truncated to avoid          *
//*    overrunning the bottom edge of the target area, the Y offset will       *
//*    reference the last row of the target area.                              *
//*                                                                            *
//*                                                                            *
//* Input  : pos    : (by reference) Y/X offset from header base position      *
//*                   Y offset must be >= ZERO and within the header area      *
//*                   X offset Must be >= ZERO and within the header area      *
//*          txt    : text to be written. Line breaks are indicated by         *
//*                   newline '\n' characters.                                 *
//*          attr   : color attribute for text                                 *
//*          refresh: (optional, 'false' by default)                           *
//*                   if 'false', do not refresh the display                   *
//*                               after writing                                *
//*                   if 'true',  refresh the display                          *
//*                               (make changes visible)                       *
//* Returns: Y/X offset at end of text written.                                *
//*          Returns Y/X==0/0 if 'pos' is out-of-range OR                      *
//*            if end position is out-of-range (unlikely)                      *
//******************************************************************************

winPos Chart::Add2HeaderText ( const winPos& pos, const char* txt,
                               attr_t txtAttr, bool refresh )
{
   winPos wp( ZERO, ZERO ) ;        // return value

   //* If a header area has been defined AND the offset is positive *
   if ( (pos.ypos >= ZERO) && (pos.xpos >= ZERO) && 
        (this->hdrRows > ZERO) && (this->hdrCols > ZERO) )
   {
      //* Calculate the start position and free space available *
      winPos trgPos( (this->hdrPos.ypos + pos.ypos),
                     (this->hdrPos.xpos + pos.xpos) ) ;
      gString gs ;
      short trgRows = this->hdrRows - pos.ypos,
            trgCols = this->hdrCols - pos.xpos ;

      //* Range-check the text, then write it to target area *
      if ( (this->TrimText ( txt, gs, trgRows, trgCols )) )
      {
         wp = this->dp->WriteParagraph ( trgPos, gs, txtAttr, true ) ;

         //* Convert 'wp' from dialog window offset to *
         //* an offset from base of header area.       *
         wp.ypos = wp.ypos - this->hdrPos.ypos ;
         wp.xpos = wp.xpos - this->hdrPos.xpos ;
      }
   }
   return ( wp ) ;

}  //* End Add2HeaderText() *

//*************************
//*  GetFooterDimensions  *
//*************************
//******************************************************************************
//* Get dimensions of footer area.                                             *
//* This area EXCLUDES the row reserved for the axis label.                    *
//*                                                                            *
//* Input  : rows   : (by reference) receives number of display rows           *
//*          cols   : (by reference) receives number of display columns        *
//*                                                                            *
//* Returns: 'true' if successful                                              *
//*          'false' if target area not defined (rows == ZERO)                 *
//******************************************************************************

bool Chart::GetFooterDimensions ( short& rows, short& cols ) const
{

   rows  = this->ftrRows ;
   cols  = this->ftrCols ;
   return ( bool(rows > ZERO) ) ;

}  //* End GetFooterDimensions() *

//*************************
//*    DrawFooterText     *
//*************************
//******************************************************************************
//* Clear the footer-text area and write the specified text starting at the    *
//* base position of the footer area.                                          *
//*                                                                            *
//* Note: To clear the area without writing new text,                          *
//*       set 'txt' to an empty string ("").                                   *
//*                                                                            *
//* Input  : txt    : text to be written                                       *
//*          attr   : color attribute for text                                 *
//*          border : (optional, 'false' by default)                           *
//*                   if 'true', draw a horizontal line across the top row     *
//*                   of the footer area. This creates a visual division       *
//*                   between the footer and the chart grid.                   *
//*          refresh: (optional, 'false' by default)                           *
//*                   if 'false', do not refresh the display after writing     *
//*                   if 'true',  refresh the display (make changes visible)   *
//*                                                                            *
//* Returns: 'true' if successful                                              *
//*          'false' if target area not defined                                *
//******************************************************************************

bool Chart::DrawFooterText ( const char* txt, attr_t txtAttr, 
                             bool border, bool refresh )
{
   bool  status = false ;           // return value

   //* Verify that target area is defined and clear the target area.*
   if ( (this->ClearArea ( this->ftrPos, this->ftrRows, this->ftrCols, txtAttr )) )
   {
      winPos txtPos = this->ftrPos ;   // working copy of area position
      short  txtRows = this->ftrRows ; // working copy of area height

      //* If specified, draw the horizontal dividing line *
      //* on last row of header area. Note that this may  *
      //* overwrite some or all the text we wrote above,  *
      //* but that is the caller's responsibility.        *
      if ( border )
      {
         this->DrawDivider ( false, this->bStyle, ZERO, this->bdrColor ) ;
         ++txtPos.ypos ;
         --txtRows ;
      }

      //* Range-check the text, then write it to target area *
      gString gs ;
      if ( (status = this->TrimText ( txt, gs, txtRows, this->ftrCols )) )
         this->dp->WriteParagraph ( txtPos, gs, txtAttr ) ;
      if ( refresh )
         this->dp->RefreshWin () ;
   }
   return status ;

}  //* End DrawFooterText() *

//*************************
//*    Add2FooterText     *
//*************************
//******************************************************************************
//* Add text to the footer area at the specified offset. For example, display  *
//* of instructions or explanations at different positions or in different     *
//* colors.                                                                    *
//* -- See GetFooterDimensions() for obtaining the dimensions of the area.     *
//* -- Unlike the DrawFooterText() method above, this method DOES NOT clear    *
//*    the footer area before writing. Instead, the specified text is added    *
//*    to any existing text in the footer area.                                *
//* -- The start position (pos) is validated to be within the target area.     *
//*    If start position is out-of-range, the text will not be written, and a  *
//*    cursor offset of 0:0 will be returned to indicate the error.            *
//* -- If any portion of the source text (txt) would overrun the boundaries of *
//*    the target area, that portion of the text will be silently discarded.   *
//* -- If all source text is displayed successfully, the offset to the space   *
//*    following the text will be returned.                                    *
//*    If, however, some of the text was discarded due to attempted boundary   *
//*    violations, then the cursor offset returned may not be as expected.     *
//*    For example, if the text was truncated to avoid overrunning the right   *
//*    edge of the target area, the X offset will reference the right edge of  *
//*    the target area. Similarly, if the text was truncated to avoid          *
//*    overrunning the bottom edge of the target area, the Y offset will       *
//*    reference the last row of the target area.                              *
//*                                                                            *
//*                                                                            *
//* Input  : pos    : (by reference) Y/X offset from footer base position      *
//*                   Y offset must be >= ZERO and within the footer area      *
//*                   X offset Must be >= ZERO and within the footer area      *
//*          txt    : text to be written. Line breaks are indicated by         *
//*                   newline '\n' characters.                                 *
//*          attr   : color attribute for text                                 *
//*          refresh: (optional, 'false' by default)                           *
//*                   if 'false', do not refresh the display                   *
//*                               after writing                                *
//*                   if 'true',  refresh the display                          *
//*                               (make changes visible)                       *
//* Returns: Y/X offset at end of text written.                                *
//*          Returns Y/X==0/0 if 'pos' is out-of-range OR                      *
//*            if end position is out-of-range (unlikely)                      *
//******************************************************************************

winPos Chart::Add2FooterText ( const winPos& pos, const char* txt,
                               attr_t txtAttr, bool refresh )
{
   winPos wp( ZERO, ZERO ) ;        // return value

   //* If a footer area has been defined AND the offset is positive *
   if ( (pos.ypos >= ZERO) && (pos.xpos >= ZERO) && 
        (this->ftrRows > ZERO) && (this->ftrCols > ZERO) )
   {
      //* Calculate the start position and free space available *
      winPos trgPos( (this->ftrPos.ypos + pos.ypos),
                     (this->ftrPos.xpos + pos.xpos) ) ;
      gString gs ;
      short trgRows = this->ftrRows - pos.ypos,
            trgCols = this->ftrCols - pos.xpos ;

      //* Range-check the text, then write it to target area *
      if ( (this->TrimText ( txt, gs, trgRows, trgCols )) )
      {
         wp = this->dp->WriteParagraph ( trgPos, gs, txtAttr, true ) ;

         //* Convert 'wp' from dialog window offset to *
         //* an offset from base of footer area.       *
         wp.ypos = wp.ypos - this->ftrPos.ypos ;
         wp.xpos = wp.xpos - this->ftrPos.xpos ;
      }
   }
   return ( wp ) ;

}  //* End Add2FooterText() *

//*************************
//*  GetMarginDimensions  *
//*************************
//******************************************************************************
//* Get dimensions of left margin area.                                        *
//*                                                                            *
//* Input  : rows   : (by reference) receives number of display rows           *
//*          cols   : (by reference) receives number of display columns        *
//*                                                                            *
//* Returns: 'true' if successful                                              *
//*          'false' if target area not defined (cols == ZERO)                 *
//******************************************************************************

bool Chart::GetMarginDimensions ( short& rows, short& cols ) const
{

   rows  = this->marRows ;
   cols  = this->marCols ;
   return ( bool(cols > ZERO) ) ;

}  //* End GetMarginDimensions() *

//*************************
//*    DrawMarginText     *
//*************************
//******************************************************************************
//* Clear the margin-text area and write the specified text to the margin area.*
//*                                                                            *
//* Note: To clear the area without writing new text,                          *
//*       set 'txt' to an empty string ("").                                   *
//*                                                                            *
//* Input  : txt    : text to be written                                       *
//*          attr   : color attribute for text                                 *
//*          border : (optional, 'false' by default) draw an interior border   *
//*                   around the Margin area.                                  *
//*          refresh: (optional, 'false' by default)                           *
//*                   if 'false', do not refresh the display after writing     *
//*                   if 'true',  refresh the display (make changes visible)   *
//*                                                                            *
//* Returns: 'true' if successful                                              *
//*          'false' if target area not defined                                *
//******************************************************************************

bool Chart::DrawMarginText ( const char* txt, attr_t txtAttr, 
                             bool border, bool refresh )
{
   gString gs ;                  // text formatting
   bool status = false ;         // return value

   //* If a margin area is defined *
   if ( (this->marRows > ZERO) && (this->marCols > ZERO) )
   {
      //* Clear the target area. If interior border, remove *
      //* it and return the area to its original size.      *
      if ( this->marginBdr )
      {
         --this->marPos.ypos ;      // origin moves up by one row
         this->marRows += 2 ;       // include top and bottom border lines
         ++this->marCols ;          // include right border column
         if ( ! this->bdrFlag )     // include left border column
         {
            --this->marPos.xpos ;   // origin moves left by one column
            ++this->marCols ;       // include the left border column
         }
         this->marginBdr = false ;
      }
      this->ClearArea ( this->marPos, this->marRows, this->marCols, txtAttr ) ;

      winPos trgPos  = this->marPos ;
      short  trgRows = this->marRows,
             trgCols = this->marCols ;

      //* If specified, draw interior border *
      if ( border )
      {
         //* If chart area has no border, draw interior border as a box.*
         if ( ! this->bdrFlag )
         {
            this->dp->DrawBox ( trgPos.ypos, trgPos.xpos,
                                trgRows, trgCols, this->bdrColor,
                                NULL, this->bStyle ) ;
         }
         //* Else, draw three connected lines to complete the interior border.*
         else
         {
            wchar_t urc, lrc ;         // characters for corners
            switch ( this->bStyle )
            {
               case ncltDUAL:
                  urc = wcsURd ; lrc = wcsLRd ; break ;
               case ncltSINGLEBOLD:
               case ncltDASH2BOLD:
               case ncltDASH3BOLD:
               case ncltDASH4BOLD:
                  urc = wcsURb ; lrc = wcsLRb ; break ;
               default:
                  urc = wcsURs ; lrc = wcsLRs ; break ;
            } ;

            //* Define the line referencing top line of margin area, *
            //* and draw the top line.                               *
            LineDef ld { ncltHORIZ, this->bStyle, trgPos.ypos, this->wpBase.xpos, 
                         trgCols, this->bdrColor } ;
            this->dp->DrawLine ( ld ) ;

            //* Reference the right end of the top line and draw the corner.*
            ld.startX += trgCols ;
            this->dp->WriteChar ( ld.startY, ld.startX, urc, this->bdrColor ) ;

            //* Draw the vertical line *
            ++ld.startY ;
            ld.type = ncltVERT ;
            ld.length = trgRows - 2 ;
            this->dp->DrawLine ( ld ) ;

            //* Draw the bottom-right corner *
            ld.startY += trgRows - 2 ;
            this->dp->WriteChar ( ld.startY, ld.startX, lrc, this->bdrColor ) ;

            //* Reference bottom line of margin area and draw bottom line.*
            ld.startX -= trgCols ;
            ld.type = ncltHORIZ ;
            ld.length = trgCols ;
            this->dp->DrawLine ( ld ) ;
         }

         //* Set the margin-border flag and adjust member variables *
         this->marginBdr = true ;
         ++this->marPos.ypos ;
         this->marRows -= 2 ;
         --this->marCols ;
         if ( ! this->bdrFlag )
         { ++this->marPos.xpos ; --this->marCols ; }
      }

      //* Range-check the text, then write it to target area *
      if ( (status = this->TrimText ( txt, gs, this->marRows, this->marCols )) )
         this->dp->WriteParagraph ( this->marPos, gs, txtAttr, refresh ) ;
   }

   return status ;

}  //* DrawMarginText() *

//*************************
//*    Add2MarginText     *
//*************************
//******************************************************************************
//* Add text to the margin area at the specified offset. For example, display  *
//* of instructions or explanations at different positions or in different     *
//* colors.                                                                    *
//* -- See GetMarginDimensions() for obtaining the dimensions of the area.     *
//* -- Unlike the DrawMarginText() method above, this method DOES NOT clear    *
//*    the margin area before writing. Instead, the specified text is added    *
//*    to any existing text in the margin area.                                *
//* -- The start position (pos) is validated to be within the target area.     *
//*    If start position is out-of-range, the text will not be written, and a  *
//*    cursor offset of 0:0 will be returned to indicate the error.            *
//* -- If any portion of the source text (txt) would overrun the boundaries of *
//*    the target area, that portion of the text will be silently discarded.   *
//* -- If all source text is displayed successfully, the offset to the space   *
//*    following the text will be returned.                                    *
//*    If, however, some of the text was discarded due to attempted boundary   *
//*    violations, then the cursor offset returned may not be as expected.     *
//*    For example, if the text was truncated to avoid overrunning the right   *
//*    edge of the target area, the X offset will reference the right edge of  *
//*    the target area. Similarly, if the text was truncated to avoid          *
//*    overrunning the bottom edge of the target area, the Y offset will       *
//*    reference the last row of the target area.                              *
//*                                                                            *
//*                                                                            *
//* Input  : pos    : (by reference) Y/X offset from margin base position      *
//*                   Y offset must be >= ZERO and within the margin area      *
//*                   X offset Must be >= ZERO and within the margin area      *
//*          txt    : text to be written. Line breaks are indicated by         *
//*                   newline '\n' characters.                                 *
//*          attr   : color attribute for text                                 *
//*          refresh: (optional, 'false' by default)                           *
//*                   if 'false', do not refresh the display                   *
//*                               after writing                                *
//*                   if 'true',  refresh the display                          *
//*                               (make changes visible)                       *
//* Returns: Y/X offset at end of text written.                                *
//*          Returns Y/X==0/0 if 'pos' is out-of-range OR                      *
//*            if end position is out-of-range (unlikely)                      *
//******************************************************************************

winPos Chart::Add2MarginText ( const winPos& pos, const char* txt,
                               attr_t txtAttr, bool refresh )
{
   winPos wp( ZERO, ZERO ) ;        // return value

   //* If a footer area has been defined AND the offset is positive *
   if ( (pos.ypos >= ZERO) && (pos.xpos >= ZERO) && 
        (this->marRows > ZERO) && (this->marCols > ZERO) )
   {
      //* Calculate the start position and free space available *
      winPos trgPos( (this->marPos.ypos + pos.ypos),
                     (this->marPos.xpos + pos.xpos) ) ;
      gString gs ;
      short trgRows = this->marRows - pos.ypos,
            trgCols = this->marCols - pos.xpos ;

      //* Range-check the text, then write it to target area *
      if ( (this->TrimText ( txt, gs, trgRows, trgCols )) )
      {
         wp = this->dp->WriteParagraph ( trgPos, gs, txtAttr, true ) ;

         //* Convert 'wp' from dialog window offset to *
         //* an offset from base of footer area.       *
         wp.ypos = wp.ypos - this->marPos.ypos ;
         wp.xpos = wp.xpos - this->marPos.xpos ;
      }
   }
   return ( wp ) ;

}  //* End Add2MarginText() *

//*************************
//*      DrawDivider      *
//*************************
//******************************************************************************
//* Draw a horizontal line across the Footer Area. Line position is specified  *
//* as an OFFSET, and must fall within the footer area.                        *
//* Minumum Offset: 0 == top line of footer area.                              *
//* Maximum Offset: footer_rows - 1 == bottom line of footer area.             *
//*    See GetFooterDimensions() method.                                       *
//* The line is drawn the full width of the target area.                       *
//* If the line intersects another line (e.g. the border), a visual connection *
//* will be made.                                                              *
//*                                                                            *
//* Input  : lineStyle: line style (member of enum ncLineType,      )          *
//*                                (excluding ncltHORIZ and ncltVERT)          *
//*          offset   : offset from top of target area                         *
//*                        Range: >= 0 and <= last row of target area          *
//*                        Note that if a border has been specified,           *
//*                        the bottom border is outside the footer area.       *
//*          lineAttr : color attribute for line                               *
//*          refresh  : (optional, 'false' by default)                         *
//*                     if 'false', do not refresh the display                 *
//*                                 after writing                              *
//*                     if 'true',  refresh the display                        *
//*                                 (make changes visible)                     *
//*                                                                            *
//* Returns: 'true'  if successful                                             *
//*          'false' if 'offset' is out-of-range or if target area is not      *
//*                  defined (target rows == ZERO)                             *
//******************************************************************************

bool Chart::DrawDivider ( ncLineType lineStyle,
                          short offset, attr_t lineAttr, bool refresh )
{

   return ( (this->DrawDivider ( false, lineStyle, offset, lineAttr, refresh )) ) ;

}  //* End DrawDivider() *

//*************************
//*       GetStats        *
//*************************
//******************************************************************************
//* Get basic statistics on the dataset:                                       *
//*                                                                            *
//* Input  : cStats : (by reference) an instance of the ChartStats class to    *
//*                   receive the statistical information                      *
//*                   Note: 'dataItems' member == number of data values        *
//*                         EXCEPT for the ctCartesian chart type where        *
//*                         'dataItems' == number of coordinate pairs.         *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************
//* Selecting the median value from the sorted array of values:                *
//*                                                                            *
//*  ----|----   odd number of items, median is central item                   *
//*  012345678                                                                 *
//*  ----||----  even number of items, halfway between the two central items   *
//*  0123456789                                                                *
//*                                                                            *
//******************************************************************************

void Chart::GetStats ( ChartStats& cStats ) const
{
   //* Reset the ChartStats object *
   cStats.dataItems = ZERO ;
   cStats.minValue  = cStats.maxValue    = 0.0 ;
   cStats.meanValue = cStats.medianValue = 0.0 ;
   cStats.cartRange.minValX    = cStats.cartRange.maxValX    = 
   cStats.cartRange.minValY    = cStats.cartRange.maxValY    = 
   cStats.cartRange.meanValX   = cStats.cartRange.meanValY   = 
   cStats.cartRange.medianValX = cStats.cartRange.medianValY = 0.0 ;

   //************************
   //* Cartesian chart type *
   //************************
   if ( this->chType == ctCartesian )
   {
      int32_t dataPairs = this->dataCount / 2 ;
      //* Copy the known data *
      cStats.dataItems = dataPairs ;
      cStats.cartRange.minValX  = this->cartRng.minValX ;
      cStats.cartRange.maxValX  = this->cartRng.maxValX ;
      cStats.cartRange.minValY  = this->cartRng.minValY ;
      cStats.cartRange.maxValY  = this->cartRng.maxValY ;
      cStats.cartRange.meanValX = this->cartRng.meanValX ;
      cStats.cartRange.meanValY = this->cartRng.meanValY ;

      //* Create a sorted (low-to-high) copy of the X and Y  *
      //* subsets of the dataset.                            *
      //* Perform a simple coctail sort of the each subset,  *
      //* and scan for median value.                         *
      double *dList = new double[dataPairs] ;
      double saveLow, saveHigh ;
      int32_t lIndex = ZERO, gIndex = (dataPairs - 1), src, trg = ZERO, i ;
      //* Working copy of X coordinates *
      for ( src = this->hBars ? 1 : 0 ; trg < dataPairs ; src += 2, ++trg )
         dList[trg] = this->dataPtr[src] ;
      //* Sort the X data and scan for the median value *
      for ( ; (gIndex-lIndex) > ZERO ; lIndex++, gIndex-- )
      {
         for ( i = lIndex+1 ; i <= gIndex ; i++ )
         {
            if ( dList[i] < dList[lIndex] )
            {
               saveLow = dList[lIndex] ;
               dList[lIndex] = dList[i] ;
               dList[i] = saveLow ;
            }

            if ( i < gIndex )
            {
               if ( dList[gIndex] < dList[i] )
               {
                  saveHigh = dList[gIndex] ;
                  dList[gIndex] = dList[i] ;
                  dList[i] = saveHigh ;
                  
                  if ( dList[i] < dList[lIndex] )
                  {
                     saveLow = dList[lIndex] ;
                     dList[lIndex] = dList[i] ;
                     dList[i] = saveLow ;
                  }
               }
            }
         }
      }
      //* Select the median value in X (see note above) *
      if ( (dataPairs % 2) > ZERO )
         cStats.cartRange.medianValX = dList[dataPairs / 2] ;
      else  // (this->dataCount % 2) == ZERO)
      {
         i = dataPairs / 2 ;
         cStats.cartRange.medianValX = ((dList[i] + dList[i - 1]) / 2.0) ;
      }

      //* Working copy of Y coordinates *
      for ( src = this->hBars ? 0 : 1 ; trg < dataPairs ; src += 2, ++trg )
         dList[trg] = this->dataPtr[src] ;
      //* Sort the Y data and scan for the median value *
      for ( ; (gIndex-lIndex) > ZERO ; lIndex++, gIndex-- )
      {
         for ( i = lIndex+1 ; i <= gIndex ; i++ )
         {
            if ( dList[i] < dList[lIndex] )
            {
               saveLow = dList[lIndex] ;
               dList[lIndex] = dList[i] ;
               dList[i] = saveLow ;
            }

            if ( i < gIndex )
            {
               if ( dList[gIndex] < dList[i] )
               {
                  saveHigh = dList[gIndex] ;
                  dList[gIndex] = dList[i] ;
                  dList[i] = saveHigh ;
                  
                  if ( dList[i] < dList[lIndex] )
                  {
                     saveLow = dList[lIndex] ;
                     dList[lIndex] = dList[i] ;
                     dList[i] = saveLow ;
                  }
               }
            }
         }
      }
      //* Select the median value in X (see note above) *
      if ( (dataPairs % 2) > ZERO )
         cStats.cartRange.medianValY = dList[dataPairs / 2] ;
      else  // (this->dataCount % 2) == ZERO)
      {
         i = dataPairs / 2 ;
         cStats.cartRange.medianValY = ((dList[i] + dList[i - 1]) / 2.0) ;
      }

      delete [] dList ;
   }  // Cartesian charts

   //***********************
   //* Non-Cartesian types *
   //***********************
   else
   {
      //* Copy the known data. These were calculated in SetRange().*
      cStats.dataItems = this->dataCount ;
      cStats.minValue  = this->minVal ;
      cStats.maxValue  = this->maxVal ;
      cStats.meanValue = this->meaVal ;

      //* Create a sorted (low-to-high) copy of the dataset. *
      //* Perform a simple coctail sort of the dataset.      *
      //* Scan the list for median value.                    *
      double *dList = new double[this->dataCount] ;
      double saveLow, saveHigh ;
      int32_t lIndex = ZERO, gIndex = (this->dataCount - 1), i ;
      for ( int32_t i = ZERO ; i < this->dataCount ; ++i )  // working copy of data
         dList[i] = this->dataPtr[i] ;

      for ( ; (gIndex-lIndex) > ZERO ; lIndex++, gIndex-- ) // sort the data
      {
         for ( i = lIndex+1 ; i <= gIndex ; i++ )
         {
            if ( dList[i] < dList[lIndex] )
            {
               saveLow = dList[lIndex] ;
               dList[lIndex] = dList[i] ;
               dList[i] = saveLow ;
            }
               
            if ( i < gIndex )
            {
               if ( dList[gIndex] < dList[i] )
               {
                  saveHigh = dList[gIndex] ;
                  dList[gIndex] = dList[i] ;
                  dList[i] = saveHigh ;
                  
                  if ( dList[i] < dList[lIndex] )
                  {
                     saveLow = dList[lIndex] ;
                     dList[lIndex] = dList[i] ;
                     dList[i] = saveLow ;
                  }
               }
            }
         }
      }
      //* Select the median value (see note above) *
      if ( (this->dataCount % 2) > ZERO )
         cStats.medianValue = dList[this->dataCount / 2] ;
      else  // (this->dataCount % 2) == ZERO)
      {
         i = this->dataCount / 2 ;
         cStats.medianValue = ((dList[i] + dList[i - 1]) / 2.0) ;
      }

      delete [] dList ;

   }  // non-Cartesian

}  //* End GetStats() *

//*************************
//*     AudibleShift      *
//*************************
//******************************************************************************
//* Enable or disable audible alert in ShiftData() method.                     *
//* If the data cannot be shifted further in the specified direction, for      *
//* example, if already at the end of dataset, then a beep will sound to       *
//* indicate that it is not possible to shift further in that direction.       *
//*                                                                            *
//* Input  : enable: 'true'  to enable alert                                   *
//*                  'false' to disable alert                                  *
//*                                                                            *
//* Returns: 'true' if alert enabled, 'false' if disabled                      *
//******************************************************************************

bool Chart::AudibleShift ( bool enable )
{
   this->audible = enable ;

   return ( this->audible ) ;

}  //* End AudibleShift() *

//*************************
//*   SetCartesianChar    *
//*************************
//******************************************************************************
//* Set the display character for the datapoints of a Cartesian chart.         *
//* Any single-column, printable, non-whitespace character may be specified,   *
//* but be aware that for best results, the character should relatively small  *
//* compared to the size of the character cell, and should be centered in the  *
//* cell. By default, the '◈' character, codepoint U+25C8 is used.             *
//*                                                                            *
//* The standard bullet characters and similar would be good choices:          *
//* U+25CF (●) black-circle                U+25CB (○) white-circle             *
//* U+26AB (⚫) medium-black-circle         U+26AA (⚪) medium-white-circle      *
//* U+23FA (⏺) black-circle-for-record     U+26AC (⚬) medium-small-white-circle*
//* U+2219 (∙) bullet operator (math)      U+25E6 (◦) white bullet             *
//* U+2022 (•) medium-small-black-circle   U+2660 (♠) spade                    *
//* U+25A0 (■) black-square                U+2665 (♥) heart                    *
//* U+25FC (◼) black-medium-square         U+2666 (♦) diamond                  *
//* U+25FE (◾) black-medium-small-square   U+2663 (♣) club                     *
//* U+25AA (▪) black-small-square          U+263A (☺) smiley face              *
//* U+2699 (⚙) gear symbol                 U+263B (☻) smiley face              *
//*                                                                            *
//* Input  : cartchar : character to be displayed at each datapoint            *
//*                                                                            *
//* Returns: 'true'  if success                                                *
//*          'false' if not a single-column printing character                 *
//******************************************************************************

bool Chart::SetCartesianChar ( wchar_t cartchar )
{
   bool status = false ;      // return value

   //* Verify that the specified character is both a single-column *
   //* character AND that it is a printing character.              *
   gString gs( &cartchar ) ;
   if ( (gs.gscols() == 1) && isgraph(cartchar) )
   {
      if ( cartchar != this->cartChar )
      {
         this->cartChar = cartchar ;         // set the new chararacter
         if ( this->chType == ctCartesian )  // redraw the chart
            this->MapCartesianPairs () ;
      }
      status = true ;
   }
   return status ;

}  //* End SetCartesianChar() *

//***************************
//* OverlayCartesianDataset *
//***************************
//******************************************************************************
//* Superimpose an additional dataset onto an existing Cartesian chart.        *
//*                                                                            *
//* 1) 'cartData' pairs MUST be in the same order as the original dataset,     *
//*    either X/Y pairs or Y/X pairs.                                          *
//* 2) The absolute range of the data in X and Y must be less than or equal    *
//*    to the range of the original data. Datapoints which fall outside the    *
//*    established range i.e. off-the-chart, will be silently discarded.       *
//* 3) 'cartCount' is the number of VALUES in the array, NOT the number of     *
//*    data pairs.                                                             *
//* 4) 'cartAttr' will typically be a color which contrasts with the color     *
//*    used for the original plot so that the new data may be easily           *
//*    identified.                                                             *
//* 5) 'dType' is optional, and indicates the type of data referenced by the   *
//*    'cartData' parameter. 'dType' indicates double-precision floating-point *
//*    data by default. If 'cartData' points to data of another type, indicate *
//*    the type as a member of enum ChartType.                                 *
//* 6) 'cartChar' is optional. By default, the character used for the original *
//*    data will also be used for the new data; however, if desired a different*
//*    character may be used. (See SetCartesianChar() method for details.)     *
//*                                                                            *
//* Input  : cartData : pointer to array of data values to be plotted,         *
//*                     arranged as X/Y pairs (or as Y/X pairs)                *
//*          cartCount: number of elements in 'cartData' array                 *
//*          cartAttr : color attribute for datapoint display                  *
//*          dType    : (optional, idtDouble by default) type of data          *
//*                     referenced by 'cartData'                               *
//*          cartChar : (optional, "dblDiamond" (0x25C8) by default) character *
//*                     to be displayed at each datapoint                      *
//*          refresh: (optional, 'false' by default)                           *
//*                   if 'false', setup is performed, but display not updated  *
//*                   if 'true',  after setup, update display immediately      *
//*                                                                            *
//* Returns: 'true'  if success,                                               *
//*          'false' if invalid parameter(s)                                   *
//******************************************************************************

bool Chart::OverlayCartesianDataset ( const void* cartData, int32_t cartCount,
                                      attr_t cartAttr, idataType dType, 
                                      wchar_t cartChar, bool refresh )
{
   const double *dblPtr = NULL ; // pointer to converted dataset
   bool status = false ;         // return value

   //* Validate the input *
   if ( (cartData != NULL) && (cartCount > ZERO) )
   {
      status = true ;         // declare success

      //* Initialize the source data as an array of *
      //* double-precision floating point values.   *
      this->ConvertData ( cartData, cartCount, dType, dblPtr ) ;

      //**********************
      //* Plot the X/Y pairs *
      //**********************
      this->PlotCartPairs ( dblPtr, cartCount, cartChar, cartAttr, NULL, true ) ;

      if ( refresh )
         this->dp->RefreshWin () ;
   }

   return status ;

}  //* End OverlayCartesianDataset() *

//*************************
//*     ProtectDialog     *
//*************************
//******************************************************************************
//* Save the display data for the independent dialog window.                   *
//* This implements a call to the local NcDialog object's                      *
//* 'SetDialogObscured()' method.                                              *
//* If dialog was not created locally, this call is ignored.                   *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void Chart::ProtectDialog ( void ) const
{
   if ( this->localDlg )
      this->dp->SetDialogObscured () ;

}  //* End ProtectDialog() *

//*************************
//*     RestoreDialog     *
//*************************
//******************************************************************************
//* Restore display data for the independent dialog window                     *
//* which was saved by a previous call to ProtectDialog().                     *
//* This implements a call to the local NcDialog object's                      *
//* 'RefreshWin()' method.                                                     *
//* If dialog was not created locally, this call is ignored.                   *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void Chart::RestoreDialog ( void ) const
{

   if ( this->localDlg )
      this->dp->RefreshWin () ;

}  //* End RestoreDialog() *

//*************************
//*      GetVersion       *
//*************************
//******************************************************************************
//* Returns the version number of the Chart class.                             *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: pointer to const string                                           *
//******************************************************************************

const char* Chart::GetVersion ( void ) const
{

   return ( ChartVersion ) ;

}  //* End GetVersion


//*** ---  ---  ---  ---  ---  ---  ---  ---  ---  ---  ---  ---  ---  ---  ***
//*** ---  ---  ---  ---    Private Methods   ---  ---  ---  ---  ---  ---  ***
//*** ---  ---  ---  ---  ---  ---  ---  ---  ---  ---  ---  ---  ---  ---  ***

//*************************
//*         Chart         *
//*************************
//******************************************************************************
//* Private Method:                                                            *
//* ---------------                                                            *
//* Default constructor. All data members set to default values.               *
//* This method is defined as private to prevent the calling application from  *
//* embarrassment resulting from an incomplete setup sequence.                 *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: implicitly returns a pointer to instantiated object               *
//******************************************************************************

Chart::Chart ( void )
{

   this->reset () ;           // initialize the data members

}  //* End Chart() *

//*************************
//*         reset         *
//*************************
//******************************************************************************
//* Private Method:                                                            *
//* ---------------                                                            *
//* Reset all data members to default values.                                  *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void Chart::reset ( void )
{
   this->dp = NULL ;
   this->wpTerm.ypos = this->wpTerm.xpos = ZERO ;
   this->wpBase.ypos = this->wpBase.xpos = ZERO ;
   this->wpOrig.ypos = this->wpOrig.xpos = ZERO ;
   this->hdrPos.ypos = this->hdrPos.xpos = ZERO ;
   this->ftrPos.ypos = this->ftrPos.xpos = ZERO ;
   this->marPos.ypos = this->marPos.xpos = ZERO ;
   this->wpYlab.ypos = this->wpYlab.xpos = ZERO ;
   this->wpXlab.ypos = this->wpXlab.xpos = ZERO ;
   this->jYlab = this->jXlab = tjLeft ;
   this->dispRows = this->dispCols = ZERO ;
   this->gridRows = this->gridCols = ZERO ;
   this->hdrRows  = this->hdrCols  = ZERO ;
   this->ftrRows  = this->ftrCols  = ZERO ;
   this->marRows  = this->marCols  = ZERO ;
   this->yOffset  = 1 ;             // default offset from top of display area to top of grid (space for axis label)
   this->xOffset  = 1 ;             // default offset from left of display area to left edge of grid
   this->ftrRows  = ZERO ;          // default freespace below grid (does not incl. axis label)
   this->barWidth = cellDIVS ;      // one full cell
   this->barSpace = ZERO ;          // no space between bars
   this->chType   = ctLowerLeft ;   // default chart type
   this->vCells   = this->hCells = ZERO ;
   this->bdrColor = this->titColor = this->barColor = this->negColor = 
   this->dColor   = this->tColor   = this->gColor   = ZERO ;
   this->dataCount = ZERO ;
   this->dataOffset = ZERO ;
   this->dataPtr  = NULL ;
   this->attrPtr  = NULL ;
   this->titText = this->hdrText = this->ftrText = this->marText = NULL ;
   this->vLabel  = this->hLabel  = NULL ;
   this->bStyle  = ncltSINGLE ;     // single-line border style
   this->gStyle  = ncltSINGLE ;     // single-line grid style

   this->minVal = this->maxVal = this->rngVal = this->meaVal = this->medVal = 0.0 ;
   this->verDiv = this->horDiv = this->minDiv = this->maxDiv = this->rngDiv = 0.0 ;
   this->perDiv = 0.0 ;
   this->cartRng.minValX = this->cartRng.maxValX = 
   this->cartRng.minValY = this->cartRng.maxValY = 0.0 ;
   this->vaxisChar = wcsLTEEs ;     // default: L'├'
   this->haxisChar = wcsBTEEs ;     // default: L'┴'
   this->axisCross = wcsLLs ;       // default: L'└'
   this->axisCap  = wcsVERTs ;      // default: L'│' (for vertical axis)
   this->cartChar  = dblDiamond ;   // default: Cartesian datapoint marker
   this->dynoData  = false ;        // no dynamic allocation
   this->localDlg  = false ;        // assume caller's dialog
   this->bdrFlag   = false ;        // assume no border
   this->marginBdr = false ;        // assume no Margin-area border
   this->hBars     = false ;        // assume vertical bars
   this->audible   = true ;         // audible shift alert enabled by default

}  //* End reset() *

//*************************
//*       Configure       *
//*************************
//******************************************************************************
//* Private Method:                                                            *
//* ---------------                                                            *
//* 1) Calculate dimensions, layout and color scheme of display area.          *
//* 2) Calculate the number of character Y/X character cells for the grid.     *
//* 3) Scan the data for max, min, median, average.                            *
//* 4) Size the data layout to fit the grid matrix.                            *
//*                                                                            *
//* Input  : cdef  : (by reference) a fully-initialized ChartDef-class object  *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************
//* Notes:                                                                     *
//* ======                                                                     *
//* 1) The sanity check on provided parameters attempts to reserve _at least_  *
//*    four(4) rows and twelve(12) columns for the coordinate grid; however,   *
//*    a really stupid set of parameters from the application could potentially*
//*    circumvent this. In that case, the resulting meshugas would be obvious. *
//*                                                                            *
//*                                                                            *
//******************************************************************************

void Chart::Configure ( const ChartDef& cdef )
{
   this->reset () ;                 // initialize all data members

   this->dp = cdef.dPtr ;                 // dialog pointer (or NULL pointer)
   if ( this->dp == NULL )                // if local dialog to be created
   {
      this->wpTerm.ypos = cdef.ulY ;      // local dialog absolute terminal Y offset
      this->wpTerm.xpos = cdef.ulX ;      // local dialog absolute terminal X offset
      this->ldRows      = cdef.rows ;     // local dialog height
      this->ldCols      = cdef.cols ;     // local dialog width
      this->dispRows    = cdef.rows - 2 ; // rows in display area (inside the borders)
      this->dispCols    = cdef.cols - 2 ; // columns in display area (inside the borders)
      this->wpBase.ypos = 1 ;             // Y origin inside upper left corner
      this->wpBase.xpos = 1 ;             // X origin inside upper left corner
      this->localDlg    = true ;          // set the local-dialog-definition flag
   }
   else
   {
      this->wpBase.ypos = cdef.ulY ;         // Y origin
      this->wpBase.xpos = cdef.ulX ;         // X origin
      this->dispRows    = cdef.rows ;        // rows in display area
      this->dispCols    = cdef.cols ;        // columns in display area
   }
   this->yOffset     = cdef.yOffset ;     // space above grid
   this->xOffset     = cdef.xOffset ;     // space at left of grid
   this->ftrRows     = cdef.footRows ;    // footer area rows
   this->barWidth    = cdef.barWidth ;    // width of each bar (divisions)
   this->barSpace    = cdef.barSpace ;    // spacing between bars (0 or 1 columns)
   this->chType      = cdef.chType ;      // chart type (member of enum chartType)
   this->bdrColor    = cdef.borderColor ; // border color attribute
   this->titColor    = cdef.titleColor ;  // title-text color attribute
   this->barColor    = cdef.barColor ;    // default bar color attribute
   this->negColor    = cdef.negColor ;    // bar color for negative values
   this->dColor      = cdef.baseColor ;   // dialog background, axis label color
   this->tColor      = cdef.textColor ;   // text color attribute
   this->gColor      = cdef.gridColor ;   // vertical/horizontal axes color attribute
   this->dataCount   = cdef.dataCount ;   // number of data elements in 'dataPtr'
   this->dataOffset  = cdef.dataOffset ;  // index of first data item to be displayed
   this->attrPtr     = cdef.attrPtr ;     // array of color attributes for bars (optional)
   this->cartChar    = cdef.cartChar ;    // Cartesian datapoint character
   this->bStyle      = cdef.borderStyle ; // line style for border
   this->gStyle      = cdef.gridStyle ;   // line style for vertical/horizontal axes
   this->vLabel      = cdef.vaxisLabel ;  // Y-axis label text
   this->hLabel      = cdef.haxisLabel ;  // X-axis label text
   this->titText     = cdef.titleText ;   // title text
   this->hdrText     = cdef.headText ;    // header text
   this->ftrText     = cdef.footText ;    // footer text
   this->marText     = cdef.marginText ;  // margin text
   this->hBars       = cdef.horizBars ;   // horizontal bars == 'true', vertical == 'false'
   this->bdrFlag     = cdef.drawBorder ;  // border flag
   this->localDlg    = this->dp == NULL ? true : false ; // locally-instantiated-dialog flag
   //* NOTE: 'hdrRows', 'hdrCols', 'hdrPos',                                  *
   //*       'marRows', 'marCols', 'marPos',                                  *
   //*       'ftrCols', 'ftrPos'                                              *
   //* members are initialized by the call to SetDimensions().                *
   //* NOTE: The 'dataPtr' member is initialized by the call to ConvertData().*
   // All other members will be calculated below.                             *

   //* Perform sanity check on data provided by caller.          *
   //* NOTE: This test IS NOT a guarantee against user stupidity.*
   if ( this->wpBase.ypos < ZERO || this->wpBase.xpos < ZERO )
      this->wpBase.ypos = this->wpBase.xpos = ZERO ;
   if ( this->dispRows < 1 )
      this->dispRows = 1 ;
   if ( this->dispCols < 1 )
      this->dispCols = 1 ;
   if ( this->yOffset < 1 || this->yOffset >= this->dispRows )
      this->yOffset = 1 ;
   if ( this->xOffset < ZERO || 
        ((this->dispCols - this->xOffset) < MIN_GRID_COLS) )
      this->xOffset = 1 ;
   if ( this->ftrRows < ZERO || 
      ((this->dispRows - (this->ftrRows + this->yOffset)) < MIN_GRID_ROWS) )
      this->ftrRows = ZERO ;
   if ( this->barWidth < ZERO || this->barWidth > cellDIVS )
      this->barWidth = cellDIVS ;
   if ( this->barSpace < ZERO || this->barSpace > 1 )
      this->barSpace = ZERO ;
   if ( this->chType < ctLowerLeft || this->chType > ctCartesian )
      this->chType = ctLowerLeft ;
   if ( this->dataCount < ZERO )
      this->dataCount = ZERO ;
   if ( (this->dataOffset < ZERO) || (this->dataOffset >= this->dataCount) )
      this->dataOffset = ZERO ;
   //* Protect against an odd number of values *
   //* when interpreting as coordinate pairs.  *
   if ( (this->chType == ctCartesian) && ((this->dataCount % 2) != ZERO) )
      --this->dataCount ;
   //* Validate the datapoint character, but do not redraw the data map.*
   // Programmer's Note: This is a trick: Because the character under test is
   // ALREADY the character in the data member, the data member will not be 
   // updated, and therefore the call will not redraw the data set.
   if ( !(this->SetCartesianChar ( this->cartChar )) )
      this->cartChar = dblDiamond ;
   if ( !((this->bStyle != ncltSINGLE)     || (this->bStyle != ncltDUAL)  ||
          (this->bStyle != ncltSINGLEBOLD) || 
          (this->bStyle != ncltDASH2)      || (this->bStyle != ncltDASH2BOLD) ||
          (this->bStyle != ncltDASH3)      || (this->bStyle != ncltDASH3BOLD) ||
          (this->bStyle != ncltDASH4)      ||  (this->bStyle != ncltDASH4BOLD)) )
   {
      this->bStyle = ncltSINGLE ;
   }
   if ( this->gStyle != ncltSINGLE && this->gStyle != ncltDUAL )
      this->gStyle = ncltSINGLE ;
   //* For locally-defined dialog window, the border *
   //* is built in, so disable the border flag.      *
   if ( this->dp == NULL )
      this->bdrFlag = false ;
   //* These chart types support only vertical bars.*
   if ( (this->chType == ctCenterLeft) || (this->chType == ctCenterRight) )
      this->hBars = false ;
   //* These chart types support only horizontal bars.*
   if ( (this->chType == ctLowerCenter) || (this->chType == ctUpperCenter) )
      this->hBars = true ;
   // (all non-null char* members are assumed to point at actual text)
   // (color attributes and 'dataAttr' if specified, are assumed to be valid)
   // (dimensions are assumed to fit within the terminal window and/or dialog window)

   //* Initialize the source data as an array of *
   //* double-precision floating point values.   *
   this->ConvertData ( cdef.dataPtr, cdef.dataCount, cdef.dataType, this->dataPtr ) ;

   //* Set grid characters for specified grid style *
   this->SetStyle () ;

   //* Calculate the number of character cells available for the chart. *
   //* (See notes in method header about this calculation.)             *
   this->SetDimensions () ;

   //* Set base position of chart grid. This identifies *
   //* the cell where vertical and horizontal axes meet.*
   this->SetOrigin () ;

   //* Do a preliminary scan of the data to set minimum  *
   //* and maximum values, data range and average value. *
   if ( this->dataCount > ZERO )
      this->SetRange () ;

   //* Calculate available grid divisions.                                 *
   //* A "division" is the smallest visible bar element (1/8 of one cell). *
   this->verDiv = this->vCells * cellDIVS ;  // total vertical divisions
   this->horDiv = this->hCells * cellDIVS ;  // total horizontal divisions

   //* Set the maximum number of divisions available for a bar.            *
   //* -- For chart types which allow the data to extend the full width of *
   //*    the grid, all physical divisions are available.                  *
   //* -- For chart types with a centered axis, positive values may extend *
   //*    from the axis across half the chart, while negative values extend*
   //*    in the opposite direction across the other half of the chart.    *
   //*    -- Note that for chart types with a centered axis, the 'minDiv'  *
   //*       limit is ignored, so: minDiv==ZERO and rngDiv==maxDiv         *
   int32_t tmpDiv ;                    // integer result of floating-point math
   this->minDiv = minBAR ;             // divisions displayed for minimum value
   switch ( this->chType )             // maximum display divisions per bar
   {
      case ctLowerLeft:
      case ctLowerRight:
      case ctUpperLeft:
      case ctUpperRight:
         this->maxDiv = (this->hBars ? this->horDiv : this->verDiv) ;
         break ;
      case ctLowerCenter:
      case ctUpperCenter:
         //* Note that these chart types are designed for *
         //* horizontal bars ONLY, but safety first.      *
         //* Whole cells only. Discard fractional cell.   *
         tmpDiv = (this->hBars ? (this->horDiv / 2) : (this->verDiv / 2)) ;
         this->maxDiv = (tmpDiv - (tmpDiv % cellDIVS)) ;
         this->minDiv = ZERO ;
         break ;
      case ctCenterLeft:
      case ctCenterRight:
         //* Note that these chart types are designed for *
         //* vertical bars ONLY, but safety first.        *
         //* Whole cells only. Discard fractional cell.   *
         tmpDiv = (this->hBars ? (this->horDiv / 2) : (this->verDiv / 2)) ;
         this->maxDiv = (tmpDiv - (tmpDiv % cellDIVS)) ;
         this->minDiv = ZERO ;
         break ;
      case ctCartesian:
         /* Note that Cartesian charts do not use cell-division data. */
         break ;
   } ;
   //* Range of divisions for graph orientation *
   this->rngDiv = this->maxDiv - this->minDiv ;
   //* Data-units per division *
   this->perDiv = this->rngVal / (this->maxDiv - this->minDiv) ;
   //* If data range is less than division range, promote 'perDiv' to 1.0 *
   if ( this->perDiv < 1.0 )
      this->perDiv = 1.0 ;

}  //* End Configure() *

//*************************
//*      ConvertData      *
//*************************
//******************************************************************************
//* If the user has provided data in a format other than as doubles, convert   *
//* the data to doubles for processing.                                        *
//* The 'dataPtr' member is initialized to point to the working copy of the    *
//* input data (double-precision floating-point values).                       *
//*                                                                            *
//* Be Aware!! : If application provides an invalid pointer or a data type     *
//*              which doesn't match the provided data, then the most likely   *
//*              result will be crash-and-burn, or at best, garbled data.      *
//*                                                                            *
//* Input  : vdPtr  : void pointer to input data ('dType' == source format)    *
//*          dCount : number of elements referenced by 'vdPtr'                 *
//*          dType  : type of data referenced by 'vdPtr'                       *
//*          trgPtr : (by reference) receives a pointer to the converted data  *
//*                                                                            *
//* Returns: 'true' if conversion successful, else 'false'                     *
//******************************************************************************

bool Chart::ConvertData ( const void* vdPtr, int32_t dCount, idataType dType, 
                          const double*& trgPtr )
{
   bool status = false ;      // return value

   //* Verify that source data were provided, AND *
   //* that a supported data type was specified.  *
   if ( (vdPtr != NULL) && (dCount > ZERO) &&
        ((dType == idtDouble)     || (dType == idtFloat)   ||
         (dType == idtByte_s)     || (dType == idtByte_u)  ||
         (dType == idtShort_s)    || (dType == idtShort_u) ||
         (dType == idtInt_s)      || (dType == idtInt_u)   ||
         (dType == idtLong_s)     || (dType == idtLong_u)  ||
         (dType == idtLongLong_s) || (dType == idtLongLong_u)) )
   {
      //* If the data are already in the correct format, *
      //* simply copy the pointer.                       *
      if ( dType == idtDouble )
         trgPtr = (const double*)vdPtr ;

      //* Perform data-type conversion *
      else
      {
         //* Allocate the new data array (non-const pointer) *
         double *dptr = new double[dCount] ;
         this->dynoData = true ;    // set the dynamic-allocation flag

         if ( dType == idtFloat )
         {
            //* Use the specified pointer type.*
            const float *aptr = (const float*)vdPtr ;
            for ( int32_t i = ZERO ; i < dCount ; ++i )
               dptr[i] = double(aptr[i]) ;
         }
         else if ( dType == idtShort_s )
         {
            //* Use the specified pointer type.*
            const short *aptr = (const short*)vdPtr ;
            for ( int32_t i = ZERO ; i < dCount ; ++i )
               dptr[i] = double(aptr[i]) ;
         }
         else if ( dType == idtShort_u )
         {
            //* Use the specified pointer type.*
            const unsigned short *aptr = (const unsigned short*)vdPtr ;
            for ( int32_t i = ZERO ; i < dCount ; ++i )
               dptr[i] = double(aptr[i]) ;
         }
         else if ( dType == idtInt_s )
         {
            const int *aptr = (const int*)vdPtr ;
            for ( int32_t i = ZERO ; i < dCount ; ++i )
               dptr[i] = double(aptr[i]) ;
         }
         else if ( dType == idtInt_u )
         {
            const unsigned int *aptr = (const unsigned int*)vdPtr ;
            for ( int32_t i = ZERO ; i < dCount ; ++i )
               dptr[i] = double(aptr[i]) ;
         }
         else if ( dType == idtLong_s )
         {
            const long *aptr = (const long*)vdPtr ;
            for ( int32_t i = ZERO ; i < dCount ; ++i )
               dptr[i] = double(aptr[i]) ;
         }
         else if ( dType == idtLong_u )
         {
            const unsigned long *aptr = (const unsigned long*)vdPtr ;
            for ( int32_t i = ZERO ; i < dCount ; ++i )
               dptr[i] = double(aptr[i]) ;
         }
         else if ( dType == idtLongLong_s )
         {
            const long long *aptr = (const long long*)vdPtr ;
            for ( int32_t i = ZERO ; i < dCount ; ++i )
               dptr[i] = double(aptr[i]) ;
         }
         else if ( dType == idtLongLong_u )
         {
            const unsigned long long *aptr = (const unsigned long long*)vdPtr ;
            for ( int32_t i = ZERO ; i < dCount ; ++i )
               dptr[i] = double(aptr[i]) ;
         }
         else if ( dType == idtByte_s )
         {
            const char *aptr = (const char*)vdPtr ;
            for ( int32_t i = ZERO ; i < dCount ; ++i )
               dptr[i] = double(aptr[i]) ;
         }
         else if ( dType == idtByte_u )
         {
            const unsigned char *aptr = (const unsigned char*)vdPtr ;
            for ( int32_t i = ZERO ; i < dCount ; ++i )
               dptr[i] = double(aptr[i]) ;
         }

         trgPtr = dptr ;      // initialize caller's pointer
      }
      status = true ;
   }
   return status ;

}  //* End ConvertData() *

//*************************
//*       SetStyle        *
//*************************
//******************************************************************************
//* Set the X and Y axis-line style for the chart.                             *
//*                                                                            *
//* Input  : none (references the 'chType' and 'gStyle' members)               *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void Chart::SetStyle ( void )
{
   switch ( this->chType )
   {
      case ctLowerLeft:    // origin at lower-left (default type)
         if ( this->gStyle == ncltDUAL )
         {
            this->vaxisChar = wcsLTEEdv ;    // L'╟'
            this->haxisChar = wcsBTEEdh;     // L'╧'
            this->axisCross = wcsLLd ;       // L'╚'
            this->axisCap   = (this->hBars ? wcsHORIZd : wcsVERTd) ; // L'═' or L'║'
         }
         else  // this->gStyle == ncltSINGLE || invalid value
         {
            this->vaxisChar = wcsLTEEs ;     // L'├'
            this->haxisChar = wcsBTEEs ;     // L'┴'
            this->axisCross = wcsLLs ;       // L'└'
            this->axisCap   = (this->hBars ? wcsHORIZs : wcsVERTs) ; // L'─' or L'│'
         }
         break ;
      case ctLowerRight:   // origin at lower-right
         if ( this->gStyle == ncltDUAL )
         {
            this->vaxisChar = wcsRTEEdv ;    // L'╢'
            this->haxisChar = wcsBTEEdh;     // L'╧'
            this->axisCross = wcsLRd ;       // L'╝'
            this->axisCap   = (this->hBars ? wcsHORIZd : wcsVERTd) ; // L'═' or L'║'
         }
         else  // this->gStyle == ncltSINGLE
         {
            this->vaxisChar = wcsRTEEs ;     // L'┤'
            this->haxisChar = wcsBTEEs ;     // L'┴'
            this->axisCross = wcsLRs ;       // L'┘'
            this->axisCap   = (this->hBars ? wcsHORIZs : wcsVERTs) ; // L'─' or L'│'
         }
         break ;
      case ctUpperLeft:    // origin at upper-left
         if ( this->gStyle == ncltDUAL )
         {
            this->vaxisChar = wcsLTEEdv ;    // L'╟'
            this->haxisChar = wcsTTEEdh;     // L'╤'
            this->axisCross = wcsULd ;       // L'╔'
            this->axisCap   = (this->hBars ? wcsHORIZd : wcsVERTd) ; // L'═' or L'║'
         }
         else  // this->gStyle == ncltSINGLE || invalid value
         {
            this->vaxisChar = wcsLTEEs ;     // L'├'
            this->haxisChar = wcsTTEEs ;     // L'┬'
            this->axisCross = wcsULs ;       // L'┌'
            this->axisCap   = (this->hBars ? wcsHORIZs : wcsVERTs) ; // L'─' or L'│'
         }
         break ;
      case ctUpperRight:   // origin at upper-right
         if ( this->gStyle == ncltDUAL )
         {
            this->vaxisChar = wcsRTEEdv ;    // L'╢'
            this->haxisChar = wcsTTEEdh;     // L'╤'
            this->axisCross = wcsURd ;       // L'╗'
            this->axisCap   = (this->hBars ? wcsHORIZd : wcsVERTd) ; // L'═' or L'║'
         }
         else  // this->gStyle == ncltSINGLE
         {
            this->vaxisChar = wcsRTEEs ;     // L'┤'
            this->haxisChar = wcsTTEEs ;     // L'┬'
            this->axisCross = wcsURs ;       // L'┐'
            this->axisCap   = (this->hBars ? wcsHORIZs : wcsVERTs) ; // L'─' or L'│'
         }
         break ;
      case ctLowerCenter:  // origin at lower-center
         if ( this->gStyle == ncltDUAL )
         {
            this->vaxisChar = wcsINSECTdv ;  // L'╫'
            this->haxisChar = wcsBTEEdh ;    // L'╧'
            this->axisCross = wcsBTEEd ;     // L'╩'
            this->axisCap   = (this->hBars ? wcsHORIZd : wcsVERTd) ; // L'═' or L'║'
         }
         else  // this->gStyle == ncltSINGLE
         {
            this->vaxisChar = wcsINSECTs ;   // L'┼'
            this->haxisChar = wcsBTEEs ;     // L'┴'
            this->axisCross = wcsBTEEs ;     // L'┴'
            this->axisCap   = (this->hBars ? wcsHORIZs : wcsVERTs) ; // L'─' or L'│'
         }
         break ;
      case ctUpperCenter:  // origin at upper-center
         if ( this->gStyle == ncltDUAL )
         {
            this->vaxisChar = wcsINSECTdv ;  // L'╫'
            this->haxisChar = wcsTTEEdh ;    // L'╤'
            this->axisCross = wcsTTEEd ;     // L'╦'
            this->axisCap   = (this->hBars ? wcsHORIZd : wcsVERTd) ; // L'═' or L'║'
         }
         else  // this->gStyle == ncltSINGLE
         {
            this->vaxisChar = wcsINSECTs ;   // L'┼'
            this->haxisChar = wcsTTEEs ;     // L'┬'
            this->axisCross = wcsTTEEs ;     // L'┬'
            this->axisCap   = (this->hBars ? wcsHORIZs : wcsVERTs) ; // L'─' or L'│'
         }
         break ;
      case ctCenterLeft:   // origin at center-left
         if ( this->gStyle == ncltDUAL )
         {
            this->vaxisChar = wcsLTEEdv ;    // L'╟'
            this->haxisChar = wcsINSECTdh ;  // L'╪'
            this->axisCross = wcsLTEEd ;     // L'╠'
            this->axisCap   = (this->hBars ? wcsHORIZd : wcsVERTd) ; // L'═' or L'║'
         }
         else  // this->gStyle == ncltSINGLE
         {
            this->vaxisChar = wcsLTEEs ;     // L'├'
            this->haxisChar = wcsINSECTs ;   // L'┼'
            this->axisCross = wcsLTEEs ;     // L'├'
            this->axisCap   = (this->hBars ? wcsHORIZs : wcsVERTs) ; // L'─' or L'│'
         }
         break ;
      case ctCenterRight:  // origin at center-right
         if ( this->gStyle == ncltDUAL )
         {
            this->vaxisChar = wcsRTEEdv ;    // L'╢'
            this->haxisChar = wcsINSECTdh ;  // L'╪'
            this->axisCross = wcsRTEEd ;     // L'╣'
            this->axisCap   = (this->hBars ? wcsHORIZd : wcsVERTd) ; // L'═' or L'║'
         }
         else  // this->gStyle == ncltSINGLE
         {
            this->vaxisChar = wcsRTEEs ;     // L'┤'
            this->haxisChar = wcsINSECTs ;   // L'┼'
            this->axisCross = wcsRTEEs ;     // L'┤'
            this->axisCap   = (this->hBars ? wcsHORIZs : wcsVERTs) ; // L'─' or L'│'
         }
         break ;
      case ctCartesian:    // Cartesian coordinates
         if ( this->gStyle == ncltDUAL )
         {
            this->vaxisChar = wcsINSECTdv ;  // L'╫'
            this->haxisChar = wcsINSECTdh ;  // L'╪'
            this->axisCross = wcsINSECTd ;   // L'╬'
            this->axisCap   = wcsINSECTdv ;  // L'╫'
         }
         else  // this->gStyle == ncltSINGLE
         {
            this->vaxisChar = wcsINSECTs ;   // L'┼'
            this->haxisChar = wcsINSECTs ;   // L'┼'
            this->axisCross = wcsINSECTs ;   // L'┼'
            this->axisCap   = wcsINSECTs ;   // L'┼'
         }
         break ;
      default:             // origin at lower-left
         if ( this->gStyle == ncltDUAL )
         {
            this->vaxisChar = wcsLTEEdv ;    // L'╟'
            this->haxisChar = wcsBTEEdh;     // L'╧'
            this->axisCross = wcsLLd ;       // L'╚'
            this->axisCap   = (this->hBars ? wcsHORIZd : wcsVERTd) ; // L'═' or L'║'
         }
         else  // this->gStyle == ncltSINGLE || invalid value
         {
            this->vaxisChar = wcsLTEEs ;     // L'├'
            this->haxisChar = wcsBTEEs ;     // L'┴'
            this->axisCross = wcsLLs ;       // L'└'
            this->axisCap   = (this->hBars ? wcsHORIZs : wcsVERTs) ; // L'─' or L'│'
         }
         break ;
   }
}  //* End SetStyle() *

//*************************
//*     SetDimensions     *
//*************************
//******************************************************************************
//* Set the X and Y axis dimensions for the chart.                             *
//*                                                                            *
//* Input  : none (references the 'chType' and 'gStyle' members)               *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************
//* Notes on calculating the number of active cells within the chart grid.     *
//*                                                                            *
//* Vertical dimension (vCells):                                               *
//* ----------------------------                                               *
//* The preliminary height of the chart (_active_ character cells) is the      *
//* height of the display area MINUS the specified 'Y-offset' and 'Y-footer'   *
//* reserved areas, MINUS the row occupied by the horizontal-axis.             *
//*  -- The minumum for 'Y-offset' and 'Y-footer' is one(1). This reserves     *
//*     one row above and below the chart area used for axis labels.           *
//* This preliminary height is adjusted for three(3) conditions:               *
//*  1) The X-axis column is included as an active cell for charts with a      *
//*     centered X-axis.                                                       *
//*  2) For some chart types, when the data bars are displayed parallel to the *
//*     Y-axis, the Y-axis cell furthest from the origin is reserved as a      *
//*     "cap" cell which may be used to display annotations.                   *
//*  3) If a border around the chart area is defined, then the height of the   *
//*     chart is reduced to compensate for the top and bottom borders.         *
//*                                                                            *
//* Horizontal dimension (hCells):                                             *
//* ------------------------------                                             *
//* The preliminary width of the chart (_active_ character cells) is the       *
//* width of the display area MINUS the specified X-offset, MINUS the column   *
//* occupied by the vertical-axis, MINUS one open column between the end of    *
//* the X-axis and the edge of the display area.                               *
//* This preliminary width is adjusted for three(3) conditions:                *
//*  1) The Y-axis column is included as an active cell for charts with a      *
//*     centered Y-axis.                                                       *
//*  2) For some chart types, when the data bars are displayed parallel to the *
//*     X-axis, the last cell of the X-axis (furthest from origin) is reserved.*
//*     This is the "cap" cell which may be used to display annotations.       *
//*  3) If a border around the chart area is defined, then the width of the    *
//*     chart is reduced to compensate for the left and right borders.         *
//*                                                                            *
//* Special Case:                                                              *
//* For Cartesian charts, all cells of both horizontal and vertical axes are   *
//* active and are included in hCells and vCells, respectively. No "cap" cell  *
//* is defined for either axis of Cartesian charts.                            *
//*                                                                            *
//* -- -- --                                                                   *
//* Header Area:                                                               *
//* 'this->yOffset' determines whether a header area will be defined.          *
//* Height of header area == yOffset - 1.                                      *
//* If yOffset==1, then no header area. The single row is reserved for the     *
//* axis label.                                                                *
//* Header area height _does not_ include the top border (if any).             *
//* Header area width is the full width of the defined chart area minus        *
//* left and right borders (if any).                                           *
//*                                                                            *
//* Footer Area:                                                               *
//* 'this->ftrRows' determines whether a footer area will be defined.          *
//* If ftrRows==ZERO, then no footer area.                                     *
//* Otherwise, validate 'ftrRows' and establish the dimensions of the area.    *
//* Footer area height _does not_ include the bottom border (if any).          *
//* Footer area width is the full width of the defined chart area minus        *
//* left and right borders (if any).                                           *
//*                                                                            *
//* Margin Area:                                                               *
//* 'this->xOffset' determines whether a margin area will be defined.          *
//* If xOffset==ZERO, then no margin area.                                     *
//* Otherwise, validate 'xOffset' and establish dimension of the area.         *
//* Margin area width _does not_ include the left border (if any).             *
//* Margin height == grid height.                                              *
//******************************************************************************

void Chart::SetDimensions ( void )
{
   //*****************************************
   //* Set the base height of the chart grid *
   //*****************************************
   this->gridRows = this->dispRows - this->ftrRows - this->yOffset - 1 ;
   this->vCells = this->gridRows - 1 ;

   //* For bars that run parallel to the Y-axis, reserve *
   //* the "cap" cell for specific chart types.          *
   //* The remaining chart types have no "cap" cell.     *
   if ( ! this->hBars &&
        ((this->chType == ctLowerLeft)   ||
         (this->chType == ctLowerRight)  ||
         (this->chType == ctUpperLeft)   ||
         (this->chType == ctUpperRight)) )
   { --this->vCells ; }

   //* Include the H-axis column as an active cell for these chart types.*
   if ( this->chType == ctCartesian )
   { ++this->vCells ; }

   //****************************************
   //* Set the base width of the chart grid *
   //****************************************
   this->gridCols = this->dispCols - this->xOffset - 1 ;
   this->hCells = this->gridCols - 1 ;

   //* Include the V-axis column as an active cell for these chart types.*
   if ( (this->chType == ctLowerCenter) ||
        (this->chType == ctUpperCenter) ||
        (this->chType == ctCartesian) )
   { ++this->hCells ; }

   //* For bars that run parallel to the X-axis, reserve the "cap" *
   //* cell for specific chart types. The remaining chart types    *
   //* have no "cap" cell.                                         *
   if ( this->hBars &&
        ((this->chType == ctLowerLeft)  ||
         (this->chType == ctLowerRight) ||
         (this->chType == ctUpperLeft)  ||
         (this->chType == ctUpperRight)) )
   { --this->hCells ; }

   //* If caller has specified a border around the display area, *
   //* reduce the dimensions of the chart accordingly.           *
   if ( this->bdrFlag )
   {
      //* Reduce the horizontal dimension of the chart *
      //* to compensate for left and right borders.    *
      this->gridCols -= 2 ;
      this->hCells -= 2 ;

      //* Reduce the vertical dimension of the chart *
      //* to compensate for top and bottom borders,  *
      //* then advance Y offset to maintain spacing. *
      this->gridRows -= 2 ;
      this->vCells -= 2 ;
      ++this->yOffset ;
   }

   //* Establish the Header Area, Footer Area and Margin Area.    *
   //* 1) 'yOffset' determines height of header.                  *
   //* 2) 'xOffset' determines width of margin.                   *
   //* 3) 'ftrRows' determines height of footer.                  *
   this->hdrPos = { short(this->wpBase.ypos + (this->bdrFlag ? 1 : 0)),
                    short(this->wpBase.xpos + (this->bdrFlag ? 1 : 0)) } ;
   this->hdrRows = this->yOffset - (this->bdrFlag ? 2 : 1) ;
   this->hdrCols = this->dispCols - (this->bdrFlag ? 2 : 0) ;
 
   this->marPos = { short(this->wpBase.ypos + this->yOffset),
                    short(this->wpBase.xpos + (this->bdrFlag ? 1 : 0)) } ;
   this->marRows = this->gridRows ;
   this->marCols = this->xOffset - (this->bdrFlag ? 1 : 0);

   //* Initialize parameters to default values (avoids problems)  *
   this->ftrPos = { short(this->wpBase.ypos + this->dispRows - 1),
                    short(this->wpBase.xpos + (this->bdrFlag ? 1 : 0)) } ;
   this->ftrCols = ZERO ;

   //* If a footer area has been defined *
   if ( this->ftrRows > ZERO )
   {
      this->ftrPos.ypos = this->wpBase.ypos + this->dispRows - this->ftrRows
                          - (this->bdrFlag ? 1 : 0) ;

      //* Full area width except if border defined, area width minus 2 *
      this->ftrCols = this->dispCols - (this->bdrFlag ? 2 : 0) ;
   }

}  //* End SetDimensions() *

//*************************
//*       SetOrigin       *
//*************************
//******************************************************************************
//* For the specified chart type, set the origin (the point at which the       *
//* X and Y axes meet.                                                         *
//*                                                                            *
//* Input  : none (references the 'chType' member)                             *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************
//* Special Case: If an axis is centered, then we must be aware of the number  *
//* of cells on either size of the centered axis. If there is an odd number    *
//* cells in the crossing axis, then there will be an equal number of cells on *
//* either side. However, if there are an even number of cells for the crossing*
//* axis we must be consistent about which side the extra cell inhabits.       *
//* Logically (at least in the author's head), the POSITIVE side of the axis   *
//* should receive the extra cell. Arithematically, this is natural because    *
//* the divide-by-two operation used to calculate the halfway point will       *
//* truncate the fractional cell.                                              *
//*                                                                            *
//* Example: If the vertical axis is centered:                                 *
//*  a) If the horizontal axis has 61 cells, there will be 30 cells on either  *
//*     side of the vertical axis and the origin cell: 30 + 1 + 30 == 61       *
//*  b) If the horizontal axis has 80 cells, there will be 39 cells on the     *
//*     negative (lower) side, 40 cells on the positive (upper) side and the   *
//*     origin cell: 39 + 1 + 40 == 80                                         *
//*                                                                            *
//* Example: If the horizontal axis is centered:                               *
//*  a) If the vertical axis has 25 cells, there will be 12 cells on either    *
//*     side of the horizontal axis and the origin cell: 12 + 1 + 12 == 25     *
//*  b) If the vertical axis has 22 cells, there will be 10 cells on the       *
//*     negative (left) side, 11 cells on the positive (right) side and the    *
//*     origin cell: 10 + 1 + 11 == 22                                         *
//*                                                                            *
//******************************************************************************

void Chart::SetOrigin ( void )
{
   switch ( this->chType )
   {
      case ctLowerLeft:    // origin at lower-left (default type)
         this->wpOrig.ypos = this->wpBase.ypos + this->yOffset + this->vCells 
                             + 1 - (this->hBars ? 1 : 0) ;
         this->wpOrig.xpos = this->wpBase.xpos + this->xOffset 
                             + (this->bdrFlag ? 1 : 0) ;
         break ;
      case ctLowerRight:   // origin at lower-right
         this->wpOrig.ypos = this->wpBase.ypos + this->yOffset + this->vCells 
                             + 1 - (this->hBars ? 1 : 0) ;
         this->wpOrig.xpos = this->wpBase.xpos + this->xOffset + this->hCells 
                             + (this->hBars ? 1 : 0) + (this->bdrFlag ? 1 : 0) ;
         break ;
      case ctUpperLeft:    // origin at upper-left
         this->wpOrig.ypos = this->wpBase.ypos + this->yOffset ;
         this->wpOrig.xpos = this->wpBase.xpos + this->xOffset 
                             + (this->bdrFlag ? 1 : 0) ;
         break ;
      case ctUpperRight:   // origin at upper-right
         this->wpOrig.ypos = this->wpBase.ypos + this->yOffset ;
         this->wpOrig.xpos = this->wpBase.xpos + this->xOffset + this->hCells 
                             + (this->hBars ? 1 : 0) + (this->bdrFlag ? 1 : 0) ;
         break ;
      case ctLowerCenter:  // origin at lower-center
         this->wpOrig.ypos = this->wpBase.ypos + this->yOffset + this->vCells 
                             + (this->hBars ? 0 : 1) ;
         this->wpOrig.xpos = this->wpBase.xpos + this->xOffset + (this->hCells / 2)
                             + (this->bdrFlag ? 1 : 0) ;
         break ;
      case ctUpperCenter:  // origin at upper-center
         this->wpOrig.ypos = this->wpBase.ypos + this->yOffset ;
         this->wpOrig.xpos = this->wpBase.xpos + this->xOffset + (this->hCells / 2)
                             + (this->bdrFlag ? 1 : 0) ;
         break ;
      case ctCenterLeft:   // origin at center-left
         this->wpOrig.ypos = this->wpBase.ypos + this->yOffset + this->vCells
                             - (this->vCells / 2) ;
         this->wpOrig.xpos = this->wpBase.xpos + this->xOffset
                             + (this->bdrFlag ? 1 : 0) ;
         break ;
      case ctCenterRight:  // origin at center-right
         this->wpOrig.ypos = this->wpBase.ypos + this->yOffset + this->vCells
                             - (this->vCells / 2) ;
         this->wpOrig.xpos = this->wpBase.xpos + this->xOffset + this->hCells 
                             + (this->hBars ? 1 : 0) + (this->bdrFlag ? 1 : 0) ;
         break ;
      case ctCartesian:    // Cartesian coordinates
         this->wpOrig.ypos = this->wpBase.ypos + this->yOffset + (this->vCells / 2) ;
         //* - For an even number of vertical cells, the number of cells below *
         //*   the axis is one greater than the number of cells above the axis.*
         //*   Because the horizontal axis is considered positive, there will  *
         //*   be an equal number of positive and negative cells.              *
         //* - For an odd number of vertical cells, the horizontal axis is     *
         //*   centered with an equal number cells above and below the axis.   *
         //*   Because the horizontal axis is considered positive, there is    *
         //*   effectively one more positive cell than negative cells.         *
         if ( (this->vCells % 2) == ZERO )   // if even number of vertical cells
            --this->wpOrig.ypos ;
         this->wpOrig.xpos = this->wpBase.xpos + this->xOffset + (this->hCells / 2)
                             + (this->bdrFlag ? 1 : 0) ;
         break ;
      default:             // (unlikely to happen, bug origin at lower-left)
         this->wpOrig.ypos = this->wpBase.ypos + this->yOffset + this->vCells + 1 ;
         this->wpOrig.xpos = this->wpBase.xpos + this->xOffset ;
         break ;
   }

}  //* End SetOrigin() *

//*************************
//*       SetRange        *
//*************************
//******************************************************************************
//* Private Method:                                                            *
//* ---------------                                                            *
//* Determine the minimum and maximum data values, the establish the data      *
//* range and average value.                                                   *
//* This method calculates the values for bar charts only. Calculations for    *
//* the ctCartesian chart type may be found in MapCartesianPairs().            *
//*                                                                            *
//* Note that the median value (medVal member) is calculated on demand because *
//* it requires sorting the data, which could be slow.                         *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void Chart::SetRange ( void )
{
   if ( this->chType != ctCartesian )
   {
      //* Establish value range *
      this->meaVal = 0.0 ;
      this->minVal = this->maxVal = this->dataPtr[ZERO] ;
      for ( int32_t indx = ZERO ; indx < this->dataCount ; ++indx )
      {
         if ( this->dataPtr[indx] < this->minVal )
            this->minVal = this->dataPtr[indx] ;   // minimum value
         if ( this->dataPtr[indx] > this->maxVal )
            this->maxVal = this->dataPtr[indx] ;   // maximum value
         this->meaVal += this->dataPtr[indx] ;     // accumulate the values
      }
      this->meaVal /= double(this->dataCount) ;    // arithmetic mean value
      this->rngVal = this->maxVal - this->minVal ; // data range
   }

}  //* End SetRange() *

//*************************
//*       DrawGrid        *
//*************************
//******************************************************************************
//* Private Method:                                                            *
//* ---------------                                                            *
//* Draw the vertical and horizontal axes of the grid.                         *
//*                                                                            *
//* Input  : labels : (optional, 'false' by default)                           *
//*                   if 'true' draw axis labels if provided                   *
//*          refresh: (optional, 'false' by default)                           *
//*                   if 'true' refresh the display (make axes visible)        *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void Chart::DrawGrid ( bool labels, bool refresh )
{
   winPos  wp ;            // cursor position
   gString gs ;            // text formatting
   short   trgpos ;        // target position

   if ( this->chType == ctLowerLeft )
   {
      wp = { short(this->wpOrig.ypos - this->vCells - (this->hBars ? 0 : 1)), 
             this->wpOrig.xpos } ;

      #if DEBUG_Y_OFFSETS != 0
      this->WriteYOffsets ( wp ) ;
      #endif   // DEBUG_Y_OFFSETS

      this->jYlab = tjLeft ;        // position the Y label 
      this->wpYlab = { short(wp.ypos - 1), wp.xpos } ;
      if ( ! this->hBars )
      {
         gs.compose( "%C\n", &this->axisCap ) ;
         wp = this->dp->WriteParagraph ( wp, gs, this->gColor ) ;
      }
      gs.compose( "%C\n", &this->vaxisChar ) ;
      while ( wp.ypos < this->wpOrig.ypos )
         wp = this->dp->WriteParagraph ( wp, gs, this->gColor ) ;
      wp = this->dp->WriteChar ( wp, this->axisCross, this->gColor ) ;

      //* Draw the horiontal axis *
      this->jXlab = tjLeft ;        // position the X label
      this->wpXlab = { short(wp.ypos + 1), wp.xpos } ;
      trgpos = this->hCells ;
      for ( short i = ZERO ; i < trgpos ; ++i )
         wp = this->dp->WriteChar ( wp, this->haxisChar, this->gColor ) ;
      if ( this->hBars )
         this->dp->WriteChar ( wp, this->axisCap, this->gColor ) ;
   }
   else if ( this->chType == ctLowerRight )
   {
      wp = { short(this->wpOrig.ypos - this->vCells - (this->hBars ? 0 : 1)),
             this->wpOrig.xpos } ;
      this->jYlab = tjRight ;       // position the Y label
      this->wpYlab = { short(wp.ypos - 1), wp.xpos } ;
      if ( ! this->hBars )
      {
         gs.compose( "%C\n", &this->axisCap ) ;
         wp = this->dp->WriteParagraph ( wp, gs, this->gColor ) ;
      }
      gs.compose( "%C\n", &this->vaxisChar ) ;
      while ( wp.ypos < this->wpOrig.ypos )
         wp = this->dp->WriteParagraph ( wp, gs, this->gColor ) ;
      wp = this->dp->WriteChar ( wp, this->axisCross, this->gColor ) ;

      //* Draw the horiontal axis *
      this->jXlab = tjRight ;       // position the X label
      this->wpXlab = { short(this->wpOrig.ypos + 1), short(this->wpOrig.xpos - 1) } ;
      wp.xpos -= this->hCells + (this->hBars ? 2 : 1) ;
      if ( this->hBars )
         wp = this->dp->WriteChar ( wp, this->axisCap, this->gColor ) ;
      while ( wp.xpos <  this->wpOrig.xpos )
         wp = this->dp->WriteChar ( wp, this->haxisChar, this->gColor ) ;
   }
   else if ( this->chType == ctUpperLeft )
   {
      //* Draw the vertical axis *
      wp = this->wpOrig ;
      trgpos = wp.ypos + this->vCells + 1 ;
      gs.compose( "%C\n", &this->axisCross ) ;
      wp = this->dp->WriteParagraph ( wp, gs, this->gColor ) ;
      gs.compose( "%C\n", &this->vaxisChar ) ;
      while ( wp.ypos < trgpos )
         wp = this->dp->WriteParagraph ( wp, gs, this->gColor ) ;
      if ( ! this->hBars )
         this->dp->WriteChar ( wp.ypos++, wp.xpos, this->axisCap, this->gColor ) ;
      this->jYlab = tjLeft ;        // position the Y label
      this->wpYlab = wp ;

      //* Draw the horiontal axis *
      this->jXlab = tjLeft ;        // position the X label
      this->wpXlab = { short(this->wpOrig.ypos - 1), short(this->wpOrig.xpos + 1) } ;
      wp = { this->wpOrig.ypos, short(this->wpOrig.xpos + 1) } ;
      trgpos = wp.xpos + this->hCells ;
      while ( wp.xpos < trgpos )
         wp = this->dp->WriteChar ( wp, this->haxisChar, this->gColor ) ;
      if ( this->hBars )
         this->dp->WriteChar ( wp, this->axisCap, this->gColor ) ;
   }
   else if ( this->chType == ctUpperRight )
   {
      //* Draw the vertical axis *
      wp = this->wpOrig ;
      trgpos = wp.ypos + this->vCells + 1 ;
      gs.compose( "%C\n", &this->axisCross ) ;
      wp = this->dp->WriteParagraph ( wp, gs, this->gColor ) ;
      gs.compose( "%C\n", &this->vaxisChar ) ;
      while ( wp.ypos < trgpos )
         wp = this->dp->WriteParagraph ( wp, gs, this->gColor ) ;
      if ( ! this->hBars )
         this->dp->WriteChar ( wp.ypos++, wp.xpos, this->axisCap, this->gColor ) ;
      this->jYlab = tjRight ;       // position the Y label
      this->wpYlab = wp ;

      //* Draw the horiontal axis *
      this->jXlab = tjRight ;       // position the X label
      this->wpXlab = { short(this->wpOrig.ypos - 1), short(this->wpOrig.xpos - 1) } ;
      wp = { this->wpOrig.ypos, 
             short(this->wpOrig.xpos - this->hCells - (this->hBars ? 1 : 0)) } ;
      if ( this->hBars )
         wp = this->dp->WriteChar ( wp, this->axisCap, this->gColor ) ;
      while ( wp.xpos < this->wpOrig.xpos )
         wp = this->dp->WriteChar ( wp, this->haxisChar, this->gColor ) ;
   }
   else if ( this->chType == ctLowerCenter ) //** ctLowerCenter **
   {
      //* Draw the vertical axis *
      // Programmer's Note: This chart type has no axis caps and supports 
      // only horizontal bars.
      wp = { short(this->wpOrig.ypos - this->vCells), this->wpOrig.xpos } ;
      this->jYlab = tjCenter ;      // position the Y label
      this->wpYlab = { short(wp.ypos - 1), wp.xpos } ;
      gs.compose( "%C\n", &this->vaxisChar ) ;
      while ( wp.ypos < this->wpOrig.ypos  )
         wp = this->dp->WriteParagraph ( wp, gs, this->gColor ) ;

      //* Draw the horiontal axis *
      this->jXlab = tjCenter ;      // position the X label
      this->wpXlab = { short(this->wpOrig.ypos + 1), this->wpOrig.xpos } ;
      wp.xpos = this->wpOrig.xpos - (this->hCells / 2) ;
      trgpos = wp.xpos + this->hCells ;
      while ( wp.xpos < trgpos )
         wp = this->dp->WriteChar ( wp, this->haxisChar, this->gColor ) ;
      this->dp->WriteChar ( this->wpOrig, this->axisCross, this->gColor ) ;
   }
   else if ( this->chType == ctUpperCenter ) //** ctUpperCenter **
   {
      //* Draw the vertical axis *
      // Programmer's Note: This chart type has no axis caps and supports 
      // only horizontal bars.
      wp = { short(this->wpOrig.ypos + 1), this->wpOrig.xpos } ;
      trgpos = wp.ypos + this->vCells ;
      this->jYlab = tjCenter ;      // position the Y label
      this->wpYlab = { short(trgpos + (this->hBars ? 0 : 1)), wp.xpos } ;
      gs.compose( "%C\n", &this->vaxisChar ) ;
      while ( wp.ypos < trgpos  )
         wp = this->dp->WriteParagraph ( wp, gs, this->gColor ) ;

      //* Draw the horiontal axis *
      wp = { this->wpOrig.ypos, short(this->wpOrig.xpos - (this->hCells / 2)) } ;
      trgpos = wp.xpos + this->hCells ;
      this->jXlab = tjCenter ;      // position the X label
      this->wpXlab = { short(this->wpOrig.ypos - 1), this->wpOrig.xpos } ;
      while ( wp.xpos < trgpos )
         wp = this->dp->WriteChar ( wp, this->haxisChar, this->gColor ) ;
      this->dp->WriteChar ( this->wpOrig, this->axisCross, this->gColor ) ;
   }
   else if ( this->chType == ctCenterLeft )
   {
      //* Draw the vertical axis *
      wp = { short(this->wpBase.ypos + this->yOffset), this->wpOrig.xpos } ;
      trgpos = wp.ypos + this->vCells + (this->hBars ? 0 : 1) ;
      this->jYlab = tjLeft ;        // position the Y label
      this->wpYlab = { short(wp.ypos - 1), wp.xpos } ;
      if ( ! this->hBars )
      {
         gs.compose( "%C\n", &this->axisCap ) ;
         wp = this->dp->WriteParagraph ( wp, gs, this->gColor ) ;
      }
      gs.compose( "%C\n", &this->vaxisChar ) ;
      while ( wp.ypos < trgpos  )
         wp = this->dp->WriteParagraph ( wp, gs, this->gColor ) ;

      //* Draw the horiontal axis *
      this->jXlab = tjRight ;       // position the Y label
      this->wpXlab = { wp.ypos, short(wp.xpos + this->hCells) } ;
      wp = this->dp->WriteChar ( this->wpOrig, this->axisCross, this->gColor ) ;
      trgpos = wp.xpos + this->hCells ;
      while ( wp.xpos < trgpos )
         wp = this->dp->WriteChar ( wp, this->haxisChar, this->gColor ) ;
   }
   else if ( this->chType == ctCenterRight )
   {
      //* Draw the vertical axis *
      wp = { short(this->wpBase.ypos + this->yOffset), this->wpOrig.xpos } ;
      trgpos = wp.ypos + this->vCells + (this->hBars ? 0 : 1) ;
      this->jYlab = tjRight ;       // position the Y label
      this->wpYlab = { short(wp.ypos - 1), wp.xpos } ;
      if ( ! this->hBars )
      {
         gs.compose( "%C\n", &this->axisCap ) ;
         wp = this->dp->WriteParagraph ( wp, gs, this->gColor ) ;
      }
      gs.compose( "%C\n", &this->vaxisChar ) ;
      while ( wp.ypos < trgpos )
         wp = this->dp->WriteParagraph ( wp, gs, this->gColor ) ;

      //* Draw the horiontal axis *
      this->jXlab = tjLeft ;        // position the Y label
      this->wpXlab = { wp.ypos, short(this->wpOrig.xpos - this->hCells) } ;
      wp = { this->wpOrig.ypos, short(this->wpOrig.xpos - this->hCells) } ;
      while ( wp.xpos < this->wpOrig.xpos )
         wp = this->dp->WriteChar ( wp, this->haxisChar, this->gColor ) ;
      wp = this->dp->WriteChar ( this->wpOrig, this->axisCross, this->gColor ) ;
   }
   else if ( this->chType == ctCartesian )
   {
      //* Draw the vertical axis *
      wp = { short(this->wpBase.ypos + this->yOffset), this->wpOrig.xpos } ;

      #if DEBUG_Y_OFFSETS != 0
      this->WriteYOffsets ( wp ) ;
      #endif   // DEBUG_Y_OFFSETS

      trgpos = wp.ypos + this->vCells - 1 ;
      this->jYlab = tjCenter ;      // position the Y label
      this->wpYlab = { short(wp.ypos - 1), wp.xpos } ;
      gs.compose( "%C\n", &this->vaxisChar ) ;
      while ( wp.ypos < trgpos  )
         wp = this->dp->WriteParagraph ( wp, gs, this->gColor ) ;
      this->dp->WriteChar ( wp, this->axisCap, this->gColor ) ;
      this->wpXlab.ypos = wp.ypos + 1 ;

      //* Draw the horiontal axis *
      wp = { this->wpOrig.ypos, short(this->wpOrig.xpos - (this->hCells / 2)) } ;
      trgpos = wp.xpos + this->hCells ;
      this->jXlab = tjLeft ;        // position the X label
      this->wpXlab.xpos = wp.xpos ;
      while ( wp.xpos < trgpos )
         wp = this->dp->WriteChar ( wp, this->haxisChar, this->gColor ) ;
      wp = this->dp->WriteChar ( this->wpOrig, this->axisCross, this->gColor ) ;
   }

   //* Draw label(s) if specified *
   if ( labels && this->vLabel != NULL )
      this->DrawAxisLabel ( this->vLabel, this->wpYlab, this->jYlab ) ;
   if ( labels && this->hLabel != NULL )
      this->DrawAxisLabel ( this->hLabel, this->wpXlab, this->jXlab ) ;

}  //* End DrawGrid() *

//*************************
//*     MapData2Grid      *
//*************************
//******************************************************************************
//* Map the data onto the grid.                                                *
//* This method selects the appropriate lower-level method based on the chart  *
//* type and the direction of the bars.                                        *
//*                                                                            *
//* There are two basic chart types:                                           *
//*   1) those with both axes on the edges of the field, and                   *
//*   2) those with one or both axes centered in the field.                    *
//* There are two basic bar types:                                             *
//*   1) Vertical, and                                                         *
//*   2) Horizontal                                                            *
//* Cartesian charts support both bar types _plus_ X/Y pairs (scatter chart).  *
//*                                                                            *
//* For data bars which start at one edge of the field and moves toward the    *
//* opposite edge WITHOUT crossing an axis, data mapping is a simple operation.*
//*                                                                            *
//*    Chart Type      Bar Direction                                           *
//*   --------------- ------------------                                       *
//*    ctLowerLeft     B-to-T (vert) or L-to-R (horiz)                         *
//*    ctLowerRight    B-to-T (vert) or R-to-L (horiz)                         *
//*    ctUpperLeft     T-to-B (vert) or L-to-R (horiz)                         *
//*    ctUpperRight    T-to-B (vert) or R-to-L (horiz)                         *
//*    ctLowerCenter   B-to-T (vert)     (ONLY HORIZ CENTERED BARS MAKE SENSE) *
//*    ctUpperCenter   T-to-B (vert)     (ONLY HORIZ CENTERED BARS MAKE SENSE) *
//*    ctCenterLeft    L-to-R (horiz)    (ONLY VERT CENTERED BARS MAKE SENSE)  *
//*    ctCenterRight   R-to-L (horiz)    (ONLY VERT CENTERED BARS MAKE SENSE)  *
//*                                                                            *
//* For data bars which start on a centered axis and grow in both directions   *
//* at right angles to that axis, the calculation is a special case.           *
//*                                                                            *
//*    Chart Type      Bar Direction                                           *
//*   --------------- ------------------                                       *
//*    ctLowerCenter   L-and-R (horiz)                                         *
//*    ctUpperCenter   L-and-R (horiz)                                         *
//*    ctCenterLeft    U-and-D (vert)                                          *
//*    ctCenterRight   U-and-D (vert)                                          *
//*    ctCartesian     U-and-D (vert) or   (not fully implemented)             *
//*                    L-and-R (horiz)     (not fully implemented)             *
//*                                                                            *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void Chart::MapData2Grid ( void )
{
   if ( this->hBars )      // draw horizontal bars
   {
      if ( (this->chType == ctLowerLeft)   ||
           (this->chType == ctLowerRight)  ||
           (this->chType == ctUpperLeft)   ||
           (this->chType == ctUpperRight) )
      {
         this->MapHorizontalBars () ;
      }
      else if ( (this->chType == ctLowerCenter) ||
                (this->chType == ctUpperCenter) )
      {
         this->MapHorizontalCentered () ;
      }
      // Programmer's Note: Cartesian charts produce X/Y data points, 
      // not bars, but the 'hBars' member is used to indicate pair order.
      else if ( this->chType == ctCartesian )
         this->MapCartesianPairs () ;
      // Programmer's Note: Chart types ctCenterLeft and ctCenterRight
      // do not support horizontal bars. See Configuration().
   }
   else                    // draw vertical bars
   {
      if ( (this->chType == ctLowerLeft)   ||
           (this->chType == ctLowerRight)  ||
           (this->chType == ctUpperLeft)   ||
           (this->chType == ctUpperRight) )
      {
         this->MapVerticalBars () ;
      }
      else if ( (this->chType == ctCenterLeft)  ||
                (this->chType == ctCenterRight) )
      {
         this->MapVerticalCentered () ;
      }
      // Programmer's Note: Cartesian charts produce X/Y data points, 
      // not bars, but the 'hBars' member is used to indicate pair order.
      else if ( this->chType == ctCartesian )
         this->MapCartesianPairs () ;
      // Programmer's Note: Chart types ctLowerCenter and ctUpperCenter
      // do not support vertical bars. See Configuration().
   }

}  //* End MapData2Grid() *

//*************************
//*    MapVerticalBars    *
//*************************
//******************************************************************************
//* Private Method:                                                            *
//* ---------------                                                            *
//* Draw the dataset as vertical bars in sequential order.                     *
//* For chart styles: ctLowerLeft, ctLowerRight, ctUpperLeft, ctUpperRight.    *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void Chart::MapVerticalBars ( void )
{
   double curVal,                      // data value to be mapped
          curDiv,                      // display divisions representing 'curVal'
          fWhole,                      // number of divisions (whole cells)
          fFract ;                     // number of divisions (fractional cell)
   short  iWhole,                      // number of filled cells (integer)
          iFract,                      // divisions in the fractional cell (integer)
          vStep,                       // step direction in which bar grows
          hStep,                       // step direction/distance between bars
          lastCol ;                    // last writable cell (loop control)
   winPos wp, wpo ;                    // cursor positioning
   wchar_t fcChar = hBlock[this->barWidth - 1], // full-cell bar character
           fracChar ;                  // fractional cell bar character (bar tip)
   attr_t  fcAttr = this->barColor,    // full-cell-character color attribute
           fracAttr = fcAttr ;         // color attribute for fractional bar cell

   //* Establish the origin-point cell, direction in which bar grows, step *
   //* direction/distance between bars, color attribute and endpoint.      *
   switch ( this->chType )
   {
      case ctLowerLeft:
         wpo = { short(this->wpOrig.ypos - 1), short(this->wpOrig.xpos + 1) } ;
         vStep = -1 ;
         hStep = this->barSpace + 1 ;
         lastCol = this->wpOrig.xpos + this->hCells ;
         break ;
      case ctLowerRight:
         wpo =  { short(this->wpOrig.ypos - 1), short(this->wpOrig.xpos - 1) } ;
         vStep = -1 ;
         hStep = -(this->barSpace + 1) ;
         lastCol = this->wpOrig.xpos - this->hCells ;
         if ( this->barWidth != cellDIVS )
            fcAttr = InvertColorAttribute ( fcAttr ) ;
         break ;
      case ctUpperLeft:
         wpo = { short(this->wpOrig.ypos + 1), short(this->wpOrig.xpos + 1) } ;
         vStep = 1 ;
         hStep = this->barSpace + 1 ;
         lastCol = this->wpOrig.xpos + this->hCells ;
         fracAttr = InvertColorAttribute ( fcAttr ) ;
         break ;
      case ctUpperRight:
         wpo = { short(this->wpOrig.ypos + 1), short(this->wpOrig.xpos - 1) } ;
         vStep = 1 ;
         hStep = -(this->barSpace + 1) ;
         lastCol = this->wpOrig.xpos - this->hCells ;
         if ( this->barWidth != cellDIVS )
            fcAttr = InvertColorAttribute ( fcAttr ) ;
         fracAttr = InvertColorAttribute ( fracAttr ) ;
         break ;
      case ctLowerCenter:     // silence the compiler warning
      case ctUpperCenter:     // silence the compiler warning
      case ctCenterLeft:      // silence the compiler warning
      case ctCenterRight:     // silence the compiler warning
      case ctCartesian:       // silence the compiler warning
         break ;
   } ;

   //* Display as much of the data as will fit on the horizonal axis *
   for ( int32_t indx = this->dataOffset ; indx < this->dataCount ; ++indx )
   {
      wp = wpo ;                       // position of bar-graph character
      if ( this->attrPtr != NULL )     // sequenced color attribute
      {
         fcAttr = fracAttr = this->attrPtr[indx] ;
         // If bar grows downward, invert fractional-cell color attribute
         if ( vStep > ZERO )     
            fracAttr = InvertColorAttribute ( fracAttr ) ;
         //* If narrow bar AND moving leftward, invert full-cell attribute *
         if ( (this->barWidth != cellDIVS) && hStep < ZERO )
            fcAttr = InvertColorAttribute ( fcAttr ) ;
      }
      curVal = this->dataPtr[indx] ;   // get a value

      //* Number of display divisions needed to represent the value *
      curDiv = this->rngDiv * ((curVal - this->minVal) / this->rngVal) ;
      if ( curDiv < this->minDiv ) curDiv = this->minDiv ;  // range limiting
      if ( curDiv > this->maxDiv ) curDiv = this->maxDiv ;

      fFract = modf ( curDiv, &fWhole ) ; // separate whole and fractional divisions
      iWhole = short(fWhole) ;            // whole divisions
      iFract = iWhole % cellDIVS ;        // divisions in fractional cell
      iWhole /= cellDIVS ;                // whole cells
      if ( fFract > 0.5 )                 // round up to next division
      {  //* If this yields a whole cell *
         if ( ++iFract >= cellDIVS )
         { ++iWhole ; iFract = ZERO ; }
      }
      iFract -= 1 ;                       // convert fractional cell to index
      //* If bar grows downward, mirror the fractional cell. *
      if ( (vStep > ZERO) && (iFract >= ZERO && iFract < cellDIVS) )
         iFract = InvertFractionalCell ( iFract ) ;

      //*************************
      //* Draw the vertical bar *
      //*************************
      for ( short i = ZERO ; i < iWhole ; ++i )
      {
         if ( this->barWidth > ZERO )
            this->dp->WriteChar ( wp, fcChar, fcAttr ) ;
         wp.ypos += vStep ;
      }
      if ( (iFract >= ZERO) || (this->barWidth == ZERO) )
      {
         fracChar = (this->barWidth == ZERO) ? vTipChar : vBlock[iFract] ;
         this->dp->WriteChar ( wp, fracChar, fracAttr ) ;
      }

      wpo.xpos += hStep ;                 // advance to next bar position
      //* If last column of chart has been filled, we're done *
      if ( ((hStep > ZERO) && (wpo.xpos > lastCol)) ||
           ((hStep < ZERO) && (wpo.xpos < lastCol)) )
         break ;
   }

}  //* MapVerticalBars() *

//*************************
//*   MapHorizontalBars   *
//*************************
//******************************************************************************
//* Private Method:                                                            *
//* ---------------                                                            *
//* Draw the dataset as horizontal bars in sequential order.                   *
//* For chart styles: ctLowerLeft, ctLowerRight, ctUpperLeft, ctUpperRight.    *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void Chart::MapHorizontalBars ( void )
{
   double curVal,                      // data value to be mapped
          curDiv,                      // display divisions representing 'curVal'
          fWhole,                      // number of divisions (whole cells)
          fFract ;                     // number of divisions (fractional cell)
   short  iWhole,                      // number of filled cells (integer)
          iFract,                      // divisions in the fractional cell (integer)
          vStep,                       // step direction in which bar grows
          hStep,                       // step direction/distance between bars
          lastRow ;                    // last writable cell
   winPos wp, wpo ;                    // cursor positioning
   wchar_t fcChar = vBlock[this->barWidth - 1], // full-cell bar character
           fracChar ;                  // fractional cell bar character (bar tip)
   attr_t  fcAttr = this->barColor,    // full-cell-character color attribute
           fracAttr = fcAttr ;         // color attribute for fractional bar cell

   //* Establish the origin-point cell, direction in which bar grows, step *
   //* direction/distance between bars, color attribute and endpoint.      *
   switch ( this->chType )
   {
      case ctLowerLeft:
         wpo = { short(this->wpOrig.ypos - 1), short(this->wpOrig.xpos + 1) } ;
         vStep = -(this->barSpace + 1) ;
         hStep = 1 ;
         lastRow = this->wpOrig.ypos - this->vCells ;
         break ;
      case ctLowerRight:
         wpo=  { short(this->wpOrig.ypos - 1), short(this->wpOrig.xpos - 1) } ;
         vStep = -(this->barSpace + 1) ;
         hStep = -1 ;
         lastRow = this->wpOrig.ypos - this->vCells ;
         fracAttr = InvertColorAttribute ( fracAttr ) ;
         break ;
      case ctUpperLeft:
         wpo = { short(this->wpOrig.ypos + 1), short(this->wpOrig.xpos + 1) } ;
         vStep = this->barSpace + 1 ;
         hStep = 1 ;
         lastRow = this->wpOrig.ypos + this->vCells ;
         if ( this->barWidth != cellDIVS )
            fcAttr = InvertColorAttribute ( fcAttr ) ;
         break ;
      case ctUpperRight:
         wpo = { short(this->wpOrig.ypos + 1), short(this->wpOrig.xpos - 1) } ;
         vStep = this->barSpace + 1 ;
         hStep = -1 ;
         lastRow = this->wpOrig.ypos + this->vCells ;
         if ( this->barWidth != cellDIVS )
            fcAttr = InvertColorAttribute ( fcAttr ) ;
         fracAttr = InvertColorAttribute ( fracAttr ) ;
         break ;
      case ctLowerCenter:     // silence the compiler warning
      case ctUpperCenter:     // silence the compiler warning
      case ctCenterLeft:      // silence the compiler warning
      case ctCenterRight:     // silence the compiler warning
      case ctCartesian:       // silence the compiler warning
         break ;
   } ;

   //* Display as much of the data as will fit on the vertical axis *
   for ( int32_t indx = this->dataOffset ; indx < this->dataCount ; ++indx )
   {
      wp = wpo ;                       // position of bar-graph character
      if ( this->attrPtr != NULL )     // sequenced color attribute
      {
         fcAttr = fracAttr = this->attrPtr[indx] ;
         //* If bar grows leftward, invert fractional-cell color attribute.*
         if ( hStep < ZERO )     
            fracAttr = InvertColorAttribute ( fracAttr ) ;
         //* If narrow bar AND moving downward, invert full-cell attribute *
         if ( (this->barWidth != cellDIVS) && vStep > ZERO )
            fcAttr = InvertColorAttribute ( fcAttr ) ;
      }
      curVal = this->dataPtr[indx] ;   // get a value

      //* Number of display divisions needed to represent the value *
      curDiv = this->rngDiv * ((curVal - this->minVal) / this->rngVal) ;
      if ( curDiv < this->minDiv ) curDiv = this->minDiv ;  // range limiting
      if ( curDiv > this->maxDiv ) curDiv = this->maxDiv ;

      fFract = modf ( curDiv, &fWhole ) ; // separate whole and fractional divisions
      iWhole = short(fWhole) ;            // whole divisions
      iFract = iWhole % cellDIVS ;        // divisions in fractional cell
      iWhole /= cellDIVS ;                // whole cells
      if ( fFract > 0.5 )                 // round up to next division
      {  //* If this yields a whole cell *
         if ( ++iFract >= cellDIVS )
         { ++iWhole ; iFract = ZERO ; }
      }
      iFract -= 1 ;                       // convert fractional cell to index
      //* If bar grows rightward, mirror the fractional cell. *
      if ( (hStep < ZERO) && (iFract >= ZERO && iFract < cellDIVS) )
         iFract = InvertFractionalCell ( iFract ) ;

      //* Draw the horizontal bar *
      for ( short i = ZERO ; i < iWhole ; ++i )
      {
         if ( this->barWidth > ZERO )
            this->dp->WriteChar ( wp, fcChar, fcAttr ) ;
         wp.xpos += hStep ;
      }
      if ( (iFract >= ZERO) || (this->barWidth == ZERO) )
      {
         fracChar = (this->barWidth == ZERO) ? hTipChar : hBlock[iFract] ;
         this->dp->WriteChar ( wp, fracChar, fracAttr ) ;
      }

      wpo.ypos += vStep ;                 // advance to next bar position
      //* If last row of chart has been filled, we're done *
      if ( ((vStep > ZERO) && (wpo.ypos > lastRow)) ||
           ((vStep < ZERO) && (wpo.ypos < lastRow)) )
         break ;
   }

}  //* MapHorizontalBars() *

//*************************
//*  MapVerticalCentered  *
//*************************
//******************************************************************************
//* Private Method:                                                            *
//* ---------------                                                            *
//* Draw the dataset as vertical bars extending above and below the horizontal *
//* axis.                                                                      *
//* For chart styles: ctCenterLeft, ctCenterRight.                             *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void Chart::MapVerticalCentered ( void )
{
   double curVal,                      // data value to be mapped
          curDiv,                      // display divisions representing 'curVal'
          fWhole,                      // number of divisions (whole cells)
          fFract ;                     // number of divisions (fractional cell)
   short  iWhole,                      // number of filled cells (integer)
          iFract,                      // divisions in the fractional cell (integer)
          vStep,                       // step direction in which bar grows
          hStep,                       // step direction/distance between bars
          lastCol ;                    // last writable cell (opposite vertical axis)
   winPos wp, wpo ;                    // cursor positioning
   wchar_t fcChar = hBlock[this->barWidth - 1], // full-cell bar character
           bChar ;                     // working bar character
   attr_t  fcAttr = this->barColor,    // full-cell-character color attribute
           fracAttr = fcAttr,          // color attribute for fractional bar cell
           bAttr ;                     // working bar attribute

   //* Establish the origin-point cell, direction in which bar grows, step *
   //* direction/distance between bars, color attribute and endpoint.      *
   switch ( this->chType )
   {
      case ctCenterLeft:
         wpo = { this->wpOrig.ypos, short(this->wpOrig.xpos + 1) } ;
         hStep = this->barSpace + 1 ;
         lastCol = this->wpOrig.xpos + this->hCells ;
         break ;
      case ctCenterRight:
         wpo = { this->wpOrig.ypos, short(this->wpOrig.xpos - 1) } ;
         hStep = -(this->barSpace + 1) ;
         lastCol = this->wpOrig.xpos - this->hCells ;
         fcAttr = InvertColorAttribute ( fcAttr ) ;
         break ;
      case ctLowerLeft:       // silence the compiler warning
      case ctLowerRight:      // silence the compiler warning
      case ctUpperLeft:       // silence the compiler warning
      case ctUpperRight:      // silence the compiler warning
      case ctLowerCenter:     // silence the compiler warning
      case ctUpperCenter:     // silence the compiler warning
      case ctCartesian:       // silence the compiler warning
         break ;
   }

   //* Display as much of the data as will fit on the horizontal axis *
   for ( int32_t indx = this->dataOffset ; indx < this->dataCount ; ++indx )
   {
      wp = wpo ;                       // position of bar-graph character
      curVal = this->dataPtr[indx] ;   // get a value

      //* Number of display divisions needed to represent the value *
      if ( curVal > 0.0 )
         curDiv = this->rngDiv * (curVal / this->maxVal) ;
      else if ( curVal < 0.0 )
         curDiv = this->rngDiv * (std::abs(curVal) / std::abs(this->minVal)) ;
      else
         curDiv = 0.0 ;
      //* Range limiting: must be <= 'maxDiv'.      *
      //* No minimum is specified for centered axes.*
      if ( curDiv > this->maxDiv ) curDiv = this->maxDiv ;

      //* Replace axis character with first block of bar.*
      // Programmer's Note: There is no character to accurately represent 
      // less than half a cell for for this position. Therefore, for all 
      // values in this group are represented by a half-cell character.
      if ( curVal > 0.0 )
      {
         vStep = -1 ;                     // direction of bar growth
         fracAttr = this->attrPtr != NULL ? this->attrPtr[indx] : this->barColor ;
         fracAttr = InvertColorAttribute ( fracAttr ) ; // color attribute for cell
         bChar = vBlock[3] ;              // half-block character
         if ( curDiv < (cellDIVS / 2) )   // subtract divisions for half-cell
            curDiv = 0.0 ;
         else
            curDiv -= 4.0 ;
      }
      else if ( curVal < 0.0 )
      {
         vStep = 1 ;                      // direction of bar growth
         fracAttr = this->attrPtr != NULL ? this->attrPtr[indx] : this->negColor ;
         bChar = vBlock[3] ;              // half-block character
         if ( curDiv < (cellDIVS / 2) )   // subtract divisions for half-cell
            curDiv = 0.0 ;
         else
            curDiv -= 4.0 ;
      }
      else  // curVal==0.0
      {
         vStep = ZERO ;                   // direction of bar growth
         fracAttr = this->barColor ;      // color attribute for cell
         bChar = this->gStyle == ncltDUAL ? wcsHORIZd : wcsHORIZb ;
      }
      //* Special test: If bar tips only, replace axis character *
      //* ONLY if no whole cells in bar.                         *
      if ( (curVal == 0.0) || (this->barWidth > ZERO) || (curDiv < cellDIVS) )
         this->dp->WriteChar ( wp, bChar, fracAttr ) ; // replace the axis character
      wp.ypos += vStep ;                  // advance to next vertical cell

      //* Set color attributes for the cell *
      if ( this->attrPtr != NULL )     // sequenced color attribute
         bAttr = fracAttr = this->attrPtr[indx] ;
      else
      {
         if ( curVal > 0.0 )
            fracAttr = bAttr = this->barColor ;
         else
            fracAttr = bAttr = this->negColor ;
      }
      //* If narrow bar AND moving leftward, invert 'bAttr'.*
      if ( (this->barWidth != cellDIVS) && (hStep < ZERO) )
         bAttr = InvertColorAttribute ( bAttr ) ;

      fFract = modf ( curDiv, &fWhole ) ; // separate whole and fractional divisions
      iWhole = short(fWhole) ;            // whole divisions
      iFract = iWhole % cellDIVS ;        // divisions in fractional cell
      iWhole /= cellDIVS ;                // whole cells
      if ( fFract > 0.5 )                 // round up to next division
      {  //* If this yields a whole cell *
         if ( ++iFract >= cellDIVS )
         { ++iWhole ; iFract = ZERO ; }
      }
      iFract -= 1 ;                       // convert fractional cell to index
      //* If entire bar was displayed on axis line, *
      //* then display of tip would be redundant.   *
      if ( (this->barWidth == ZERO) && (iWhole == ZERO) )
         iFract = -1 ;

      //* If bar grows downward, mirror the fractional cell. *
      if ( (vStep > ZERO) && (iFract >= ZERO && iFract < cellDIVS) )
      {
         iFract = InvertFractionalCell ( iFract ) ;
         if ( this->barWidth > ZERO )
            fracAttr = InvertColorAttribute ( fracAttr ) ;
      }

      //*************************
      //* Draw the vertical bar *
      //*************************
      if ( curVal != 0.0 )
      {
         for ( short i = ZERO ; i < iWhole ; ++i )
         {
            if ( this->barWidth > ZERO )
               this->dp->WriteChar ( wp, fcChar, bAttr ) ;
            wp.ypos += vStep ;
         }
         if ( (iFract >= ZERO) || ((this->barWidth == ZERO) && iWhole > ZERO) )
         {
            bChar = (this->barWidth == ZERO) ? vTipChar : vBlock[iFract] ;
            this->dp->WriteChar ( wp, bChar, fracAttr ) ;
         }
      }

      wpo.xpos += hStep ;                 // advance to next bar position
      //* If last column of chart has been filled, we're done *
      if ( ((hStep > ZERO) && (wpo.xpos > lastCol)) ||
           ((hStep < ZERO) && (wpo.xpos < lastCol)) )
         break ;
   }

}  //* End MapVerticalCentered() *

//*************************
//* MapHorizontalCentered *
//*************************
//******************************************************************************
//* Private Method:                                                            *
//* ---------------                                                            *
//* Draw the dataset as horizontal bars extending left and right from the      *
//* the vertical axis.                                                         *
//* For chart styles: ctLowerCenter, ctUpperCenter.                            *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void Chart::MapHorizontalCentered ( void )
{
   double curVal,                      // data value to be mapped
          curDiv,                      // display divisions representing 'curVal'
          fWhole,                      // number of divisions (whole cells)
          fFract ;                     // number of divisions (fractional cell)
   short  iWhole,                      // number of filled cells (integer)
          iFract,                      // divisions in the fractional cell (integer)
          vStep,                       // step direction/distance between bars
          hStep,                       // step direction in which bar grows
          lastRow ;                    // last writable cell (opposite horizontal axis)
   winPos wp, wpo ;                    // cursor positioning
   wchar_t fcChar = vBlock[this->barWidth - 1], // full-cell bar character
           bChar ;                     // working bar character
   attr_t  fcAttr = this->barColor,    // full-cell-character color attribute
           fracAttr = fcAttr,          // color attribute for fractional bar cell
           bAttr ;                     // working bar attribute

   switch ( this->chType )
   {
      case ctLowerCenter:
         wpo = { short(this->wpOrig.ypos - 1), this->wpOrig.xpos } ;
         vStep = -(this->barSpace + 1) ;
         lastRow = this->wpOrig.ypos - this->vCells ;
         break ;
      case ctUpperCenter:
         wpo = { short(this->wpOrig.ypos + 1), this->wpOrig.xpos } ;
         vStep = this->barSpace + 1 ;
         fcAttr = InvertColorAttribute ( fcAttr ) ;
         lastRow = this->wpOrig.ypos + this->vCells ;
         break ;
      case ctLowerLeft:       // silence the compiler warning
      case ctLowerRight:      // silence the compiler warning
      case ctUpperLeft:       // silence the compiler warning
      case ctUpperRight:      // silence the compiler warning
      case ctCenterLeft:      // silence the compiler warning
      case ctCenterRight:     // silence the compiler warning
      case ctCartesian:       // silence the compiler warning
         break ;
   }

   //* Display as much of the data as will fit on the horizontal axis *
   for ( int32_t indx = this->dataOffset ; indx < this->dataCount ; ++indx )
   {
      wp = wpo ;                       // position of bar-graph character
      curVal = this->dataPtr[indx] ;   // get a value

      //* Number of display divisions needed to represent the value *
      if ( curVal > 0.0 )
         curDiv = this->rngDiv * (curVal / this->maxVal) ;
      else if ( curVal < 0.0 )
         curDiv = this->rngDiv * (std::abs(curVal) / std::abs(this->minVal)) ;
      else
         curDiv = 0.0 ;
      //* Range limiting: must be <= 'maxDiv'.      *
      //* No minimum is specified for centered axes.*
      if ( curDiv > this->maxDiv ) curDiv = this->maxDiv ;

      //* Replace axis character with first block of bar.*
      // Programmer's Note: There is no character to accurately represent 
      // less than half a cell for for this position. Therefore, for all 
      // values in this group are represented by a half-cell character.
      if ( curVal > 0.0 )
      {
         hStep = 1 ;                      // direction of bar growth
         fracAttr = this->attrPtr != NULL ? this->attrPtr[indx] : this->barColor ;
         fracAttr = InvertColorAttribute ( fracAttr ) ; // color attribute for cell
         bChar = hBlock[3] ;              // half-block character
         if ( curDiv < (cellDIVS / 2) )   // subtract divisions for half-cell
            curDiv = 0.0 ;
         else
            curDiv -= 4.0 ;
      }
      else if ( curVal < 0.0 )
      {
         hStep = -1 ;                     // direction of bar growth
         fracAttr = this->attrPtr != NULL ? this->attrPtr[indx] : this->negColor ;
         bChar = hBlock[3] ;              // half-block character
         if ( curDiv < (cellDIVS / 2) )   // subtract divisions for half-cell
            curDiv = 0.0 ;
         else
            curDiv -= 4.0 ;
      }
      else  // curVal==0.0
      {
         hStep = ZERO ;                   // direction of bar growth
         fracAttr = this->barColor ;      // color attribute for cell
         bChar = this->gStyle == ncltDUAL ? wcsVERTd : wcsVERTb ;
      }
      //* Special test: If bar tips only, replace axis character *
      //* ONLY if no whole cells in bar.                         *
      if ( (curVal == 0.0) || (this->barWidth > ZERO) || (curDiv < cellDIVS) )
         this->dp->WriteChar ( wp, bChar, fracAttr ) ; // replace the axis character
      wp.xpos += hStep ;                  // advance to next horizontal cell

      //* Set color attributes for the cell *
      if ( this->attrPtr != NULL )     // sequenced color attribute
         bAttr = fracAttr = this->attrPtr[indx] ;
      else
      {
         if ( curVal > 0.0 )
            fracAttr = bAttr = this->barColor ;
         else
            fracAttr = bAttr = this->negColor ;
      }
      //* If narrow bar AND moving downward, invert 'bAttr'.*
      if ( (this->barWidth != cellDIVS) && (vStep < ZERO) )
         bAttr = InvertColorAttribute ( bAttr ) ;

      fFract = modf ( curDiv, &fWhole ) ; // separate whole and fractional divisions
      iWhole = short(fWhole) ;            // whole divisions
      iFract = iWhole % cellDIVS ;        // divisions in fractional cell
      iWhole /= cellDIVS ;                // whole cells
      if ( fFract > 0.5 )                 // round up to next division
      {  //* If this yields a whole cell *
         if ( ++iFract >= cellDIVS )
         { ++iWhole ; iFract = ZERO ; }
      }
      iFract -= 1 ;                       // convert fractional cell to index
      //* If entire bar was displayed on axis line, *
      //* then display of tip would be redundant.   *
      if ( (this->barWidth == ZERO) && (iWhole == ZERO) )
         iFract = -1 ;

      //* If bar grows leftward, mirror the fractional cell. *
      if ( (hStep < ZERO) && (iFract >= ZERO && iFract < cellDIVS) )
      {
         iFract = InvertFractionalCell ( iFract ) ;
         if ( this->barWidth > ZERO )
            fracAttr = InvertColorAttribute ( fracAttr ) ;
      }

      //***************************
      //* Draw the horizontal bar *
      //***************************
      if ( curVal != 0.0 )
      {
         for ( short i = ZERO ; i < iWhole ; ++i )
         {
            if ( this->barWidth > ZERO )
               this->dp->WriteChar ( wp, fcChar, bAttr ) ;
            wp.xpos += hStep ;
         }
         if ( (iFract >= ZERO) || ((this->barWidth == ZERO) && iWhole > ZERO) )
         {
            bChar = (this->barWidth == ZERO) ? hTipChar : hBlock[iFract] ;
            this->dp->WriteChar ( wp, bChar, fracAttr ) ;
         }
      }

      wpo.ypos += vStep ;                 // advance to next bar position
      //* If last column of chart has been filled, we're done *
      if ( ((vStep > ZERO) && (wpo.ypos > lastRow)) ||
           ((vStep < ZERO) && (wpo.ypos < lastRow)) )
         break ;
   }

}  //* End MapHorizontalCentered() *

//*************************
//*   MapCartesianPairs   *
//*************************
//******************************************************************************
//* Private Method:                                                            *
//* ---------------                                                            *
//* Draw the dataset as Y/X pairs extending from the axis origin (center).     *
//* For chart style: ctCartesian.                                              *
//*                                                                            *
//* The 'horizBars' flag is re-purposed for Cartesian charts.                  *
//*    'false' : source data are X/Y pairs                                     *
//*    'true;  : source data are Y/X pairs                                     *
//*                                                                            *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************
//* Notes:                                                                     *
//* ------                                                                     *
//* 1) Ideally, the units in X should be equal to the units in Y,              *
//*    i.e. an aspect ratio of 1:1. Unfortunately, the character cells in a    *
//*    terminal window are not square. Nor are the pixels in either dimension  *
//*    an even multiple of the other dimension. This leads to a dilema:        *
//*    How do we make the display appear "square-ish"?                         *
//* 2) To make the above problem even more difficult, there is no _portable_   *
//*    way to query the terminal for the aspect ratio. All we can do is guess  *
//*    at an aspect ratio for the grid that approximates a square and map the  *
//*    data according to that ratio. Not a very satisfying solution.           *
//*    Typical font aspect ratio is between 1.5 and 1.7 vertical-to-horizontal.*
//*    This translates to approximately 2 vertical cells to 3 horizontal cells.*
//* 3) Note that any attempt to subdivide character cells to contain multiple  *
//*    datapoints would yield an unsightly mess (as well as being              *
//*    mathematically suspect).                                                *
//*                                                                            *
//* Mapping the Data                                                           *
//* ----------------                                                           *
//* To create a "square-ish" layout, we use the X/Y character-cell aspect ratio*
//* described above:                                                           *
//*  -- For each coordinate pair, the X coordinate is multiplied by 'xCONST'   *
//*     to achieve this squarish aspect ratio.                                 *
//*  -- The resulting coordinate pair is scaled to fit on the physical grid.   *
//* This is a lot of (slow) floating-point math, but we must assume that the   *
//* dataset is fairly small. The larger the dataset, the more likely that the  *
//* chart will just be a big, useless blob.                                    *
//*                                                                            *
//*                       ---  ---  ---  ---  ---                              *
//* Programmer's Note: The minimum- and maximum-value variables are initialized*
//* to "safe" values, i.e. the first pair in the array.                        *
//* The X and Y mean (average) values are calculated here.                     *
//* Median values are not calculated here; instead, medians are calculated on  *
//* demand only because it is a very slow operation. See GetStats() method.    *
//******************************************************************************

void Chart::MapCartesianPairs ( void )
{
   winPos  wp ;                     // cell positioning
   bool    yxPair  = this->hBars ;  // source value-pair order

   //* Initialize the limit values *
   this->cartRng.minValX = this->cartRng.maxValX = 
                           (yxPair ? this->dataPtr[1] : this->dataPtr[0]) ;
   this->cartRng.minValY = this->cartRng.maxValY = 
                           (yxPair ? this->dataPtr[0] : this->dataPtr[1]) ;
   //* Initialize the mean-value accumulators *
   this->cartRng.meanValX = this->cartRng.meanValY = 0.0 ;

   //* Establish data range *
   for ( int32_t i = ZERO ; i < this->dataCount ; i += 2 )
   {
      if ( yxPair )     // Y/X pairs
      {
         if ( this->dataPtr[i] < this->cartRng.minValY )
            this->cartRng.minValY = this->dataPtr[i] ;
         if ( this->dataPtr[i] > this->cartRng.maxValY )
            this->cartRng.maxValY = this->dataPtr[i] ;
         if ( this->dataPtr[i+1] < this->cartRng.minValX )
            this->cartRng.minValX = this->dataPtr[i+1] ;
         if ( this->dataPtr[i+1] > this->cartRng.maxValX )
            this->cartRng.maxValX = this->dataPtr[i+1] ;
         this->cartRng.meanValX += this->dataPtr[i+1] ;
         this->cartRng.meanValY += this->dataPtr[i] ;
      }
      else              // X/Y pairs
      {
         if ( this->dataPtr[i] < this->cartRng.minValX )
            this->cartRng.minValX = this->dataPtr[i] ;
         if ( this->dataPtr[i] > this->cartRng.maxValX )
            this->cartRng.maxValX = this->dataPtr[i] ;
         if ( this->dataPtr[i+1] < this->cartRng.minValY )
            this->cartRng.minValY = this->dataPtr[i+1] ;
         if ( this->dataPtr[i+1] > this->cartRng.maxValY )
            this->cartRng.maxValY = this->dataPtr[i+1] ;
         this->cartRng.meanValX += this->dataPtr[i] ;
         this->cartRng.meanValY += this->dataPtr[i+1] ;
      }
   }

   //* Apply the aspect-ratio correction to 'minValX' and 'maxValX'.*
   this->cartRng.minValX *= xCONST ;
   this->cartRng.maxValX *= xCONST ;

   //* Calculate the mean values *
   this->cartRng.meanValX /= this->dataCount / 2 ;
   this->cartRng.meanValY /= this->dataCount / 2 ;

   //**********************
   //* Plot the X/Y pairs *
   //**********************
   this->PlotCartPairs ( this->dataPtr, this->dataCount, this->cartChar, 
                         this->barColor, this->attrPtr ) ;

}  //* End MapCartesianPairs() *

//*************************
//*     PlotCartPairs     *
//*************************
//******************************************************************************
//* Private Method:                                                            *
//* ---------------                                                            *
//* Plot the Cartesian datapoints according to caller's parameters.            *
//*                                                                            *
//* Called by MapCartesianPair() to plot the original dataset.                 *
//* Called by OverlayCartesianDataset() to plot secondary dataset(s).          *
//*                                                                            *
//* Input  : dPtr    : pointer to an array of double-precision values,         *
//*                    arranged as X/Y pairs (or as Y/X pairs)                 *
//*          dCnt    : number of values in 'dPtr' array                        *
//*          pntChar : character to display at each datapoint                  *
//*          pntAttr : color attribute for 'pntChar'                           *
//*          atrArray: (optional, NULL pointer by default) if specified, points*
//*                    to an array of color attribytes, one per X/Y coordinate *
//*                    pair. 'pntAttr' will be ignored.                        *
//*          truncate: (optional, 'false' by default)                          *
//*                    'false' plot all data points                            *
//*                    'true'  silently discard data points which fall outside *
//*                            the specified data range                        *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void Chart::PlotCartPairs ( const double* dPtr, int32_t dCnt, 
                            wchar_t pntChar, attr_t pntAttr, 
                            const attr_t *atrArray, bool truncate )
{
   #define DEBUG_CART_PLOT (0)         // For debugging only

   double  xVal, yVal,                 // current coordinate pair
           fWhole, fFract,             // whole and fractional cells
           xpCells = double(this->hCells / 2),  // available cells in X
           xnCells = double(this->hCells / 2),
           ypCells = double(this->vCells / 2),  // available cells in Y
           ynCells = double(this->vCells / 2) ;
           if ( (this->vCells % 2) == ZERO )    // adjust cell count in positive Y
              ypCells -= 1.0 ;
           if ( (this->hCells % 2) == ZERO )    // adjust cell count in positive X
              xpCells -= 1.0 ;
   int32_t attrIndx = ZERO ;           // if color array specified, array index
   winPos  wp ;                        // cell positioning
   short   yOrig = this->wpOrig.ypos,  // local copy of chart origin Y
           xOrig = this->wpOrig.xpos,  // local copy of chart origin X
           tCell = yOrig - (this->vCells / 2),  // topmost available display cell
           bCell = yOrig + (this->vCells / 2),  // bottom available display cell
           lCell = xOrig - (this->hCells / 2),  // leftmost available display cell
           rCell = xOrig + (this->hCells / 2) ; // rightmost available display cell
           if ( (this->vCells % 2) == ZERO )
              ++tCell ;
           if ( (this->hCells % 2) == ZERO )
              --rCell ;
   bool    yxPair  = this->hBars ;     // source value-pair order

   #if DEBUG_CART_PLOT != 0
   gString gx ;
   ofstream ofs( "./A_mcp.txt", ofstream::out | ofstream::trunc ) ;
   if ( ofs.is_open() )
   {
      gx.compose( "MapCartesianPairs\n-----------------"
                  "\nwpOrig: %hd/%hd"
                  "\nthis->vCells: %hd this->hCells: %hd"
                  "\ntCell: %2hd"
                  "\nbCell: %2hd"
                  "\nlCell: %2hd"
                  "\nrCell: %2hd"
                  "\nminValX:%5.1lf" 
                  "\nmaxValX:%5.1lf" 
                  "\nminValY:%5.1lf" 
                  "\nmaxValY:%5.1lf"
                  "\nxpCells:% 2.1lf xnCells:% 2.1lf"
                  "\nypCells:% 5.1lf ynCells:% 5.1lf",
                  &this->wpOrig.ypos, &this->wpOrig.xpos, &this->vCells, &this->hCells,
                  &tCell, &bCell, &lCell, &rCell,
                  &this->cartRng.minValX, &this->cartRng.maxValX,
                  &this->cartRng.minValY, &this->cartRng.maxValY,
                  &xpCells, &xnCells, &ypCells, &ynCells ) ;
      ofs << gx.ustr() << endl ;
   }
   #endif   // DEBUG_CART_PLOT

   for ( int32_t i = ZERO ; i < dCnt ; i += 2 )
   {
      if ( yxPair )     // Y/X pairs
      {
         yVal = dPtr[i] ;
         xVal = dPtr[i+1] * xCONST ;
      }
      else              // X/Y pairs
      {
         xVal = dPtr[i] * xCONST ;
         yVal = dPtr[i+1] ;
      }

      if ( xVal > 0.0 )
      {
         fFract = modf ( (xpCells * xVal / this->cartRng.maxValX), &fWhole ) ;
         wp.xpos = short(xOrig + fWhole - 1) ;
         if ( (fFract > 0.5) && (wp.xpos < rCell) )
            ++wp.xpos ;
      }
      else if ( xVal < 0.0 )
      {
         fFract = modf ( (xnCells * xVal / this->cartRng.minValX), &fWhole ) ;
         wp.xpos = short(xOrig - (fWhole - (fFract < -0.5 ? 1.0 : 0.0))) ;
      }
      else     // (xVal == 0.0)
         wp.xpos = xOrig ;

      #if DEBUG_CART_PLOT != 0
      if ( ofs.is_open() )
      {
         gx.compose( "\ni:% 3d y/x:%.1lf/%.1lf  yVal/xVal:%.1lf/%.1lf"
                     "\n   fWhole:%.1lf fFract:%.1lf = xpos:%hd",
                     &i, &dPtr[i+1], &dPtr[i], &yVal, &xVal,
                     &fWhole, &fFract, &wp.xpos ) ;
         if ( (wp.xpos < lCell) || (wp.xpos > rCell) )
            gx.append( " (xpos out-of-range)" ) ;
         ofs << gx.ustr() ;
      }
      #endif   // DEBUG_CART_PLOT

      if ( yVal > 0.0 )
      {
         fFract = modf ( (ypCells * yVal / this->cartRng.maxValY), &fWhole ) ;
         wp.ypos = short(yOrig - (fWhole + (fFract > 0.5 ? 1.0 : 0.0))) ;
      }
      else if ( yVal < 0.0 )
      {
         fFract = modf ( (ynCells * yVal / this->cartRng.minValY), &fWhole ) ;
         wp.ypos = short(yOrig + (fWhole - (fFract < -0.5 ? 1.0 : 0.0))) ;
      }
      else     // (yVal == 0.0)
         wp.ypos = yOrig ;

      #if DEBUG_CART_PLOT != 0
      if ( ofs.is_open() )
      {
         gx.compose( "\n   fWhole:%.1lf fFract:%.1lf = ypos:%hd",
                     &fWhole, &fFract, &wp.ypos ) ;
         if ( (wp.ypos < tCell) || (wp.ypos > bCell) )
            gx.append( " (ypos out-of-range)" ) ;
         ofs << gx.ustr() ;
      }
      #endif   // DEBUG_CART_PLOT

      #if DEBUG_CART_PLOT == 0   // PRODUCTION
      if ( atrArray != NULL )    // if color array specified, set attribute for datapoint
         pntAttr = atrArray[attrIndx++] ;

      if ( ! truncate ||
           (truncate && 
            ((wp.ypos <= bCell) && (wp.ypos >= tCell) &&
             (wp.xpos <= rCell) && (wp.xpos >= lCell))) )
      this->dp->WriteChar ( wp, pntChar, pntAttr ) ;
      #else    // DEBUG_CARTPLOT - Plot out-of-range datapoints in a constrasting color.
      if ( ((wp.ypos > bCell) || (wp.ypos < tCell) ||
           (wp.xpos > rCell) || (wp.xpos < lCell)) )
         this->dp->WriteChar ( wp, pntChar, nc.gr ) ;
      else
         this->dp->WriteChar ( wp, pntChar, pntAttr ) ;
      #endif   // U/C
   }

   #if DEBUG_CART_PLOT != 0
   if ( ofs.is_open() )
   {
      ofs << endl ;
      ofs.close() ;
   }
   #endif   // DEBUG_CART_PLOT

   #undef DEBUG_CART_PLOT
}  //* End PlotCartPairs() *

//*************************
//*       shiftData       *
//*************************
//******************************************************************************
//* Private Method:                                                            *
//* ---------------                                                            *
//* Shift the data display forward or backward by the specified amount.        *
//*                                                                            *
//* IMPORTANT NOTE: This method assumes that some of the data are not currently*
//* displayed. DO NOT call this method if all data are currently displayed.    *
//*                                                                            *
//* Members of enum ShiftBlock:                                                *
//* ---------------------------                                                *
//* sbFirstPage: Start display with the first element of the data array and    *
//*              continue until chart is filled or end of data. If already on  *
//*              first page or if all data are currently displayed, this       *
//*              command will have no effect.                                  *
//* sbLastPage : Fill the chart starting at the offset which will allow the    *
//*              last element of the data array to be displayed. If already on *
//*              last page, or if all data are currently displayed, this       *
//*              command will have no effect.                                  *
//* sbNextPage : Shift the currently-displayed data out of the frame and       *
//*              display the next full page of data. If end of data is already *
//*              displayed, this command has no effect.                        *
//* sbPrevPage : Shift the currently-displayed data out of the frame and       *
//*              display the previous full page of data. If start of data is   *
//*              already displayed, this command has no effect.                *
//* sbNoShift  : Ignore this argument and use the shift value specified by     *
//*              'shiftCnt'. (This is the default value for 'shiftBlk'.)       *
//*                                                                            *
//*                                                                            *
//* Input  : shiftCnt : if a positive value, shift toward tail of data         *
//*                     if a negative value, shift toward head of data         *
//*          shiftBlk : (optional, sbNoShift by default)                       *
//*                     If specified, 'shiftCnt' is ignored and data shift is  *
//*                     calculated based on specified member of enum ShiftBlock*
//*                                                                            *
//* Returns: 'false' if shift successful                                       *
//*          'true'  if shift failed                                           *
//******************************************************************************

bool Chart::shiftData ( int32_t shiftCnt, ShiftBlock shiftBlk )
{
   int32_t oldOffset = this->dataOffset ; // remember current offset
   int32_t avail = ZERO,      // number of elements available in specified direction
           bCells = ZERO ;    // number of cells in bar direction
   if ( this->hBars )         // horizontal bars
      bCells = this->vCells ;
   else                       // vertical bars
      bCells = this->hCells ;

   bool shiftError = false ;  // return value

   //* If block-shift not specified, shift index by 'shiftCnt' elements *
   if ( (shiftBlk == sbNoShift) && (shiftCnt != ZERO) )
   {
      //* Un-displayed elements in selected direction *
      if ( shiftCnt > ZERO )
         avail = this->dataCount - this->dataOffset - bCells ;
      else
         avail = this->dataOffset ;

      //* If un-displayed elements >= requested shift,   *
      //* shift by specified number of elements.         *
      //* Otherwise, set shift to display remaining data.*
      if ( avail > ZERO )
      {
         if ( avail >= abs ( shiftCnt ) )
            this->dataOffset = this->dataOffset + shiftCnt ;
         else
         {
            if ( shiftCnt > ZERO )
               this->dataOffset = this->dataCount - bCells ;
            else
               this->dataOffset = ZERO ;
         }
      }
   }

   //* If block shift was specified. Calculate new start index *
   else if ( shiftBlk != sbNoShift )
   {
      if ( shiftBlk == sbFirstPage )
         this->dataOffset = ZERO ;
      else if ( shiftBlk == sbLastPage )
         this->dataOffset = this->dataCount - bCells ;
      else if ( shiftBlk == sbNextPage )
      {
         avail = this->dataCount - this->dataOffset - bCells ;
         if ( avail > ZERO )
         {
            if ( avail >= bCells )
               this->dataOffset += bCells ;
            else
               this->dataOffset = this->dataCount - bCells ;
         }
      }
      else if ( shiftBlk == sbPrevPage )
      {
         avail = this->dataOffset ;
         if ( avail > ZERO )
         {
            if ( avail >= bCells )
               this->dataOffset -= bCells ;
            else
               this->dataOffset = ZERO ;
         }
      }
   }

   //* If data are to be shifted *
   if ( this->dataOffset != oldOffset )
   {
      this->ClearGrid () ;             // clear and re-draw the chart grid
      this->MapData2Grid () ;          // map the data into the grid

      #if DEBUG_CONFIG != 0 && DEBUG_SHIFT != 0 // Report shift positioning
      //* If display position has been initialized (see DumpCfg()).*
      if ( (wpShift.ypos != ZERO) && (wpShift.xpos != ZERO) )
      {
         bCells = this->dataOffset + bCells - 1 ;
         gString gs( "Offsets %03d -> %03d of %03d items.",
                     &this->dataOffset, &bCells, &this->dataCount ) ;
         this->dp->WriteString ( wpShift, gs, nc.grB ) ;
      }
      #endif   // DEBUG_CONFIG && DEBUG_SHIFT

      this->dp->RefreshWin () ;        // make changes visible
   }
   else
      shiftError = true ;

   return shiftError ;

}  //* End shiftData() *

//*************************
//*     DrawAxisLabel     *
//*************************
//******************************************************************************
//* Private Method:                                                            *
//* ---------------                                                            *
//* Draw the specified axis label (or other text) at the specified position.   *
//*                                                                            *
//* Input  : label  : text to be written                                       *
//*          wpl    : cursor position for text                                 *
//*          justify: (optional, tjLeft by default) member of enum txtJust:    *
//*                   tjLeft   : left-justified, write starting at 'wpl'       *
//*                   tjCenter : center-justified, center text on 'wpl'        *
//*                   tjRight  : right-justified, end text at 'wpl'            *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void Chart::DrawAxisLabel ( const char* label, const winPos& wpl, txtJust justify )
{
   gString gs( label ) ;      // text formatting
   winPos  wp = wpl ;         // cursor position

   if ( justify == tjCenter )
      wp.xpos -= (gs.gscols() / 2) ;
   else if ( justify == tjRight )
      wp.xpos -= (gs.gscols() - 1) ;
   this->dp->WriteParagraph ( wp, gs, this->dColor ) ;

}  //* DrawAxisLabel() *

//*************************
//*       TrimText        *
//*************************
//******************************************************************************
//* Truncate the text in X and/or Y to fit within a rectangle of the           *
//* specified dimensions. This method is called by various text-display        *
//* methods to ensure that the text does not overrun the boundaries of the     *
//* target area. These methods are:                                            *
//*   DrawHeaderText()                                                         *
//*   DrawMarginText()                                                         *
//*   Add2MarginText()                                                         *
//*   DrawFooterText()                                                         *
//*   Add2FooterText()                                                         *
//*                                                                            *
//* Input  : src    : source text                                              *
//*          trg    : (by reference) receives processed text                   *
//*          maxRow : height of target area (rows)                             *
//*          maxCol : width of target area (columns)                           *
//*                                                                            *
//* Returns: 'true'  if 'trg' contains text data to be written                 *
//*          'false' if target area is zero height or zero width, OR           *
//*                  if none of the source text will fit in the target area    *
//******************************************************************************
//* Programmer's Note: This algorithm uses alot of CPU cycles for each         *
//* character processed, but it is simple and unlikely to break.               *
//* We may enhance it for performance after the class functionality is stable. *
//******************************************************************************

bool Chart::TrimText ( const char* src, gString& trg, short maxRow, short maxCol )
{
   trg.clear() ;        // initialize the target buffer

   //* Verify that the target area is non-zero height and width *
   if ( (maxRow > ZERO) && (maxCol > ZERO) )
   {
      gString gsRow,                   // build a row of text
              gs( src ) ;              // used for analysis
      short srcRows = (*src == NULLCHAR ? ZERO : 1), // number of source rows required
            chIndx = ZERO,             // character index
            tmpi ;                     // temporary index

      //* Count the text rows and limit rows to 'maxRow' *
      while ( (chIndx = gs.find( L'\n', chIndx )) >= ZERO )
      {
         tmpi = chIndx ;
         if ( gs.gstr()[++chIndx] != NULLCHAR )
         {
            if ( ++srcRows > maxRow )
            {
               --srcRows ;
               gs.limitChars( tmpi ) ; // truncate extra rows
               break ;
            }
         }
      }

      int   chCount,             // number of (wchar_t) characters in source array
            colCount ;           // accumulator for number of columns
      const wchar_t *wPtr = gs.gstr() ; // pointer to wchar_t array
      const short *colPtr = gs.gscols( chCount ) ;
      bool eol ;                 // end-of-line flag
      chIndx = ZERO ;            // index into character and column arrays

      //* Parse each line of the text, limiting the number of   *
      //* columns for each line to the width of the target area.*
      for ( short r = ZERO ; r < srcRows ; ++r )
      {
         gsRow.clear() ;         // initialize the row buffer
         colCount = ZERO ;       // initialize accumulator
         eol = false ;

         while ( chIndx < chCount )
         {
            if ( wPtr[chIndx] == nckNULLCHAR )   // if end of source text
            { ++chIndx ; eol = true ; break ; }

            else if ( wPtr[chIndx] == nckNEWLINE ) // if end of source line
            { gsRow.append( wPtr[chIndx++] ) ; eol = true ; break ; }

            else if ( (colCount + colPtr[chIndx]) < maxCol ) // add character
            {
               gsRow.append( wPtr[chIndx] ) ;   // add character to row
               colCount += colPtr[chIndx] ;     // total columns in row
               ++chIndx ;
            }

            else  // row will be filled unless overflow would occur
            {
               //* Row exactly filled *
               if ( (colCount + colPtr[chIndx]) == maxCol )
               {
                  gsRow.append( wPtr[chIndx] ) ;// add character to row
                  //colCount += colPtr[chIndx] ;  // total columns in row
                  ++chIndx ;
               }
               //* Discard remainder of text for this row *
               while ( (wPtr[chIndx] != nckNEWLINE) && (wPtr[chIndx] != nckNULLCHAR) )
                  ++chIndx ;
            }
         }  // while()

         //* If text line is complete, add it to target buffer *
         if ( eol )
            trg.append( gsRow.gstr() ) ;

      }     // for(;;)
   }
   return ( bool(trg.gschars() > 1) ) ; // if target contains more than null terminator

}  //* End TrimText() *

//*************************
//*      DrawDivider      *
//*************************
//******************************************************************************
//* Private Method                                                             *
//* --------------                                                             *
//* Called indirectly by the public method of the same name, or called         *
//* directly by member methods (seeDrawHeaderText()).                          *
//*                                                                            *
//* Draw a horizontal line across either the Header Area or Footer Area.       *
//* 1) Draw a horizontal line in the header area. The line must be at or below *
//*    the header position and will extend the full width of the chart area.   *
//*    See GetHeaderDimensions() method.                                       *
//* 2) Draw a horizontal line in the footer area. The line must be at or below *
//*    the footer position and will extend the full width of the chart area.   *
//*    See GetFooterDimensions() method.                                       *
//*                                                                            *
//*                                                                            *
//* If the line intersects another line, a visual connection will be made.     *
//*                                                                            *
//* Input  : targArea : target area for drawing divider:                       *
//*                      'false' == Footer Area                                *
//*                      'true'  == Header Area                                *
//*          lineStyle: line style (member of enum ncLineType,      )          *
//*                                (excluding ncltHORIZ and ncltVERT)          *
//*          offset   : offset from top of target area                         *
//*                        Range: >= 0 and <= bottom edge of footer area       *
//*                        Note that if a border has been specified, the top   *
//*                        border is outside the header area, and the bottom   *
//*                        border is outside the footer area.                  *
//*          lineAttr : color attribute for line                               *
//*          refresh  : (optional, 'false' by default)                         *
//*                     if 'false', do not refresh the display                 *
//*                                 after writing                              *
//*                     if 'true',  refresh the display                        *
//*                                 (make changes visible)                     *
//*                                                                            *
//* Returns: 'true'  if successful                                             *
//*          'false' if 'offset' is out-of-range or if target area is not      *
//*                  defined (target rows == ZERO)                             *
//******************************************************************************

bool Chart::DrawDivider ( bool targArea, ncLineType lineStyle,
                          short offset, attr_t lineAttr, bool refresh )
{
   bool  status = false ;           // return value

   //* Validate the line style and silently correct errors.*
   if ( (lineStyle == ncltHORIZ) || (lineStyle == ncltVERT) )
      lineStyle = ncltSINGLE ;

   //***********************************
   //** Horizontal Divider for Header **
   //***********************************
   if ( targArea )
   {
      //* Validate the Y offset.                          *
      //* 1) If a header area is defined, and             *
      //* 2) If the specified offset is not negative, and *
      //* 3) If offset is within the header area.         *
      if ( (this->hdrRows > ZERO) &&
           (offset >= ZERO) && (offset < this->hdrRows) )
      {
         //* Line is the full width of the defined chart area.*
         LineDef ld { ncltHORIZ, lineStyle, 
                      short(this->hdrPos.ypos + offset), 
                      short(this->localDlg ? ZERO : this->wpBase.xpos), 
                      short(this->localDlg ? this->ldCols : this->dispCols), 
                      lineAttr } ;
         this->dp->DrawLine ( ld ) ;
         if ( refresh )
            this->dp->RefreshWin () ;
         status = true ;
      }
   }

   //***********************************
   //** Horizontal Divider for Footer **
   //***********************************
   else
   {
      //* Validate the Y offset.                          *
      //* 1) If a footer area is defined, and             *
      //* 2) If the specified offset is not negative, and *
      //* 3) If one of the following:                     *
      //*    a) Dialog is locally defined and offset from *
      //*       base position is within the dialog window.*
      //*    b) Dialog defined by application and offset  *
      //*       from base position is within the footer   *
      //*       area.                                     *
      //* Will not overwrite the bottom border (if any).  *
      if ( (this->ftrRows > ZERO) &&   // if there is a footer area 
           (offset >= ZERO) &&         // and offset not negative
           ((this->localDlg && ((this->ftrPos.ypos + offset) < (this->ldRows - 1))) ||
            ((!this->localDlg) && (offset < this->ftrRows))) )
      {
         //* Line is the full width of the defined chart area.*
         LineDef ld { ncltHORIZ, lineStyle, 
                      short(this->ftrPos.ypos + offset), 
                      short(this->localDlg ? ZERO : this->wpBase.xpos), 
                      short(this->localDlg ? this->ldCols : this->dispCols), 
                      lineAttr } ;
         this->dp->DrawLine ( ld ) ;
         if ( refresh )
            this->dp->RefreshWin () ;
         status = true ;
      }
   }
   return status ;

}  //* End DrawHorizontalLine() *

//*************************
//*       ClearArea       *
//*************************
//******************************************************************************
//* Private Method:                                                            *
//* ---------------                                                            *
//* Clear the display area, and if specified draw a border around the area.    *
//*                                                                            *
//* Note that for a locally-defined dialog window, the actual clearing         *
//* operation is unnecessary because it is a new window.                       *
//*                                                                            *
//* Programmer's Note: The NcDialog 'ClearArea()' and 'ClearLine()' methods    *
//* both refresh the display, but this method does not refresh.                *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void Chart::ClearArea ( void )
{
   if ( ! this->localDlg )
   {
      gString gs( " " ) ;
      gs.padCols( this->dispCols ) ;
      gs.append( L'\n' ) ;
      winPos wp = this->wpBase ;
      for ( short r = ZERO ; r < this->dispRows ; ++r )
         wp = this->dp->WriteParagraph ( wp, gs, this->dColor ) ;
   }

   //* If specified, draw a border around the chart area *
   if ( this->bdrFlag )
   {
      this->dp->DrawBox ( this->wpBase.ypos, this->wpBase.xpos,
                          this->dispRows, this->dispCols, this->bdrColor,
                          this->titText, this->bStyle ) ;
   }

}  //* End ClearArea() *

//*************************
//*       ClearArea       *
//*************************
//******************************************************************************
//* Private Method:                                                            *
//* ---------------                                                            *
//* Clear the target area. Called to clear the header, margin or footer area.  *
//*                                                                            *
//*                                                                            *
//* Input  : wpPos  : upper-left corner of target area                         *
//*          trgRows: number of rows in target area                            *
//*          trgCols: number of columns in target area                         *
//*          trgAttr: background color                                         *
//*                                                                            *
//* Returns: 'true'  if area cleared                                           *
//*          'false' if target area is not defined                             *
//******************************************************************************

bool Chart::ClearArea ( const winPos& wpPos, short trgRows, short trgCols, 
                        attr_t trgAttr )
{
   bool  status = false ;

   if ( (trgRows > ZERO) && (trgCols > ZERO) )
   {
      winPos wp = wpPos ;
      gString gs( " " ) ;
      gs.padCols( trgCols ) ;
      gs.append( L'\n' ) ;

      for ( short i = ZERO ; i < trgRows ; ++i )
         wp = this->dp->WriteParagraph ( wp, gs, trgAttr ) ;

      status = true ;
   }
   return status ;

}  //* ClearArea() *

//*************************
//*       ClearGrid       *
//*************************
//******************************************************************************
//* Private Method:                                                            *
//* ---------------                                                            *
//* Clear the area occupied by the chart grid and re-draw the grid.            *
//*                                                                            *
//* Programmer's Note: The NcDialog 'ClearArea()' and 'ClearLine()' methods    *
//* both refresh the display, but this method does not refresh.                *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void Chart::ClearGrid ( void )
{
   //* Test size and position of cleared area. For debugging only.*
   #define DEBUG_CLEAR (0)

   //* Erase data from chart area including axes *
   winPos wp( short(this->wpBase.ypos + this->yOffset), 
              short(this->wpBase.xpos + this->xOffset + (this->bdrFlag ? 1 : 0)) ) ;
   short eog = wp.ypos + this->gridRows ;

   gString gs( " " ) ;
   gs.padCols( this->gridCols ) ;
   gs.append( L'\n' ) ;

   #if DEBUG_CLEAR != 0    // Position test only
   gs.replace( L' ', L'/', ZERO, false, true ) ;
   #endif   // DEBUG_CLEAR

   while ( wp.ypos < eog )
      wp = this->dp->WriteParagraph ( wp, gs, this->gColor ) ;

   #if DEBUG_CLEAR != 0
   this->dp->RefreshWin () ;
   chrono::duration<short>aWhile( 2 ) ;
   this_thread::sleep_for( aWhile ) ;
   #endif   // DEBUG_CLEAR

   this->DrawGrid () ;     // re-draw the axes
   #undef DEBUG_CLEAR

}  //* End ClearGrid() *

//*************************
//*        DumpCfg        *
//*************************
//******************************************************************************
//* Debugging Only: display data configuration                                 *
//* ---------------                                                            *
//* -- Programmer's Note: The position is calculated with the assumption that  *
//*    the caller is the "ChartTest" class which is aware of the internal      *
//*    functionality of this method.                                           *
//* -- This method is called before other data are written to the chart area   *
//*    so that "real" data may overwrite it if necessary.                      *
//* -- Note that if called, DrawFooterText() erases the entire footer area,    *
//*    so this config data may be partially or fully erased.                   *
//*                                                                            *
//* Input  : wpos   : start position for data display                          *
//*          refresh: (optional, 'false' by default)                           *
//*                   if 'true', refresh the display (make data visible)       *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void Chart::DumpCfg ( bool refresh )
{
   #if DEBUG_CONFIG != 0
   const short textHEIGHT = 9 ;

   short dlgRows, dlgCols ;
   this->dp->GetDialogDimensions ( dlgRows, dlgCols ) ;
   winPos wpos( (dlgRows - textHEIGHT - ((!this->localDlg && this->bdrFlag) ? 1 : 0)), 
                this->wpBase.xpos + 1 ) ;

   //* Do we have enough space below the footer for the entire data dump? *
   if ( wpos.ypos >= this->ftrPos.ypos )
   {
      winPos wp = wpos ;
      gString gsOut( 
         "wpBase(%02hd,%02hd)\n"
         "wpOrig(%02hd,%02hd)\n"
         "dispRows:%02hd\n"
         "dispCols:%02hd\n"
         "gridRows:%02hd\n"
         "gridCols:%02hd\n"
         "yOffset :%02hd\n"
         "xOffset :%02hd\n",
         &this->wpBase.ypos, &this->wpBase.xpos,
         &this->wpOrig.ypos, &this->wpOrig.xpos,
         &this->dispRows, &this->dispCols,
         &this->gridRows, &this->gridCols,
         &this->yOffset, &this->xOffset ) ;
      wp = this->dp->WriteParagraph ( wp, gsOut, nc.gy ) ;

      wp.ypos = wpos.ypos ;
      wp.xpos += 15 ;
      gsOut.compose(
         "barWidth:%02hd\n"
         "barSpace:%02hd\n"
         "hdrRows :%02hd\n"
         "ftrRows :%02hd\n"
         "vCells:%02hd\n"
         "hCells:%02hd\n"
         "minVal:%.3lf\n"
         "maxVal:%.3lf\n",
         &this->barWidth, &this->barSpace,
         &this->hdrRows, &this->ftrRows, 
         &this->vCells, &this->hCells,
         &this->minVal, &this->maxVal ) ;
      wp = this->dp->WriteParagraph ( wp, gsOut, nc.gy ) ;

      wp.ypos = wpos.ypos + 1 ;
      wp.xpos += 15 ;
      gsOut.compose( 
         "chType  :%s\n"
         "rngVal:%.3lf\n"
         "meaVal:%.3lf\n"
         "medVal:%.3lf\n"
         "verDiv:%.3lf\n"
         "horDiv:%.3lf\n",
         (this->chType == ctLowerLeft   ? "LL" :
          this->chType == ctUpperLeft   ? "UL" :
          this->chType == ctLowerRight  ? "LR" :
          this->chType == ctUpperRight  ? "UR" :
          this->chType == ctLowerCenter ? "LC" :
          this->chType == ctUpperCenter ? "UC" :
          this->chType == ctCenterLeft  ? "CL" :
          this->chType == ctCenterRight ? "CR" : "CT"),
         &this->rngVal, &this->meaVal, &this->medVal, &this->verDiv, &this->horDiv ) ;
      wpShift = this->dp->WriteParagraph ( wp, gsOut, nc.gy ) ;

      wp.ypos = wpos.ypos + 1 ;
      wp.xpos += 16 ;
      gsOut.compose( 
         "perDiv:%.3lf\n"
         "minDiv:%.3lf\n"
         "maxDiv:%.3lf\n"
         "rngDiv:%.3lf\n",
         &this->perDiv, &this->minDiv, &this->maxDiv, &this->rngDiv ) ;
      this->dp->WriteParagraph ( wp, gsOut, nc.gy ) ;

      if ( refresh )
         this->dp->RefreshWin () ;
   }
   #endif   // DEBUG_CONFIG
}  //* End DumpCfg() *

//*************************
//*     WriteYOffsets     *
//*************************
//******************************************************************************
//* Debugging Only: display area line numbers.                                 *
//* ---------------                                                            *
//* -- Programmer's Note: The display position is calculated with the          *
//*    assumption that the caller is the "ChartTest" class which is aware of   *
//*    the internal functionality of this method.                              *
//* -- The line numbers are written OUTSIDE the left edge of the chart area.   *
//*    For this reason, the write occurs ONLY when chart is drawn to caller's  *
//*    dialog window.                                                          *
//*                                                                            *
//* Input  : gridTop : upper left corner of the grid                           *
//*          clear   : (optional, 'false' by default)                          *
//*                    if 'true' write spaces instead of line numbers          *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void Chart::WriteYOffsets ( const winPos gridTop, bool clear )
{
   #if DEBUG_Y_OFFSETS != 0
   if ( ! this->localDlg && !(gridTop.ypos == ZERO && gridTop.xpos == ZERO) )
   {
      //* Reference top of chart area *
      winPos wp( this->wpBase.ypos, this->wpBase.xpos - 2 ),
             wpFoot = this->ftrPos ;
      gString gs( L"  \n" ) ;             // initialize for "clear"
      attr_t gAttr = nc.rebl | ncbATTR,   // grid-line attribute
             fAttr = nc.grbl ;            // header/footer-line attribute

      //* Save the initial position for a future call to clear the area.*
      if ( ! clear ) wpOutside = gridTop ;

      //* Header lines *
      for ( short r = ZERO ; wp.ypos < (gridTop.ypos - 1) ; ++r )
      {
         if ( ! clear )
            gs.compose( "%02hd\n", &r ) ;
         wp = this->dp->WriteParagraph ( wp, gs, fAttr ) ;
      }
      if ( ! clear )
         gs = L"--\n" ;
      wp = this->dp->WriteParagraph ( wp, gs, fAttr ) ;
      wp.ypos = gridTop.ypos ;         // reference top of grid area

      //* Note: If no footer area defined, ftrPos.ypos     *
      //* references the last row of the chart area, i.e.  *
      //* the axis-label line. Increment ftrPos.ypos so    *
      //* chart-lines loop will reach that point.          *
      //* (The footer-lines loop will not execute.)        *
      if ( this->ftrRows == ZERO )
         ++wpFoot.ypos ;

      //* Chart lines *
      for ( short r = ZERO ; wp.ypos < (wpFoot.ypos - 1) ; ++r )
      {
         if ( ! clear )
            gs.compose( "%02hd\n", &r ) ;
         wp = this->dp->WriteParagraph ( wp, gs, gAttr ) ;
      }
      //* Axis-label line *
      if ( ! clear )
         gs = L"--\n" ;
      wp = this->dp->WriteParagraph ( wp, gs, gAttr ) ;

      //* Footer lines *
      wp.ypos = wpFoot.ypos ;
      for ( short r = ZERO ; r < this->ftrRows ; ++r )
      {
         if ( ! clear )
            gs.compose( "%02hd\n", &r ) ;
         wp = this->dp->WriteParagraph ( wp, gs, fAttr ) ;
      }
      this->dp->RefreshWin () ;
   }
   #endif   // DEBUG_Y_OFFSETS
}  //* End WriteYOffsets()

//*************************
//*     CaptureDialog     *
//*************************
//******************************************************************************
//* For Debugging Only!                                                        *
//* ===================                                                        *
//* Perform a screen capture of the dialog window.                             *
//* This is a pass-through to the NcDialog method of the same name.            *
//*                                                                            *
//* Input  : (see NcDialog.hpp)                                                *
//*                                                                            *
//* Returns: OK if successful                                                  *
//*          ERR  : a) if parameter out-of-range                               *
//*                 b) if unable to write to specified file                    *
//*                 c) if development methods are disabled                     *
//******************************************************************************

short Chart::CaptureDialog ( const char* fPath, bool fhtml, bool timeStamp, 
                             const char* stylePath, short cellsPerLine,
                             bool lineComments, attr_t simpleColor, bool microSpace ) const
{
   #if DEBUG_SCREENSHOT != 0
   return ( (this->dp->CaptureDialog ( fPath, fhtml, timeStamp, stylePath,
                                       cellsPerLine, lineComments,
                                       simpleColor, microSpace )) ) ;
   #else    // METHOD IS DISABLED
   return ( ERR ) ;
   #endif   // DEBUG_SCREENSHOT
}  //* End CaptureDialog() *


//*** ---  ---  ---  ---  ---  ---  ---  ---  ---  ---  ---  ---  ---  ---  ***
//*** ---  ---  ---  ---  Non-member Methods  ---  ---  ---  ---  ---  ---  ***
//*** ---  ---  ---  ---  ---  ---  ---  ---  ---  ---  ---  ---  ---  ---  ***

//*************************
//* InvertFractionalCell  *
//*************************
//******************************************************************************
//* Non-member method:                                                         *
//* ------------------                                                         *
//* Invert the index into the block-character arrays, hBlock[] and vBlock[].   *
//* This allows a visual inversion of the filled vs. empty pixes withing the   *
//* character cell.                                                            *
//*                                                                            *
//* When used in conjunction with the mirrored color attribute, it allows for  *
//* fractionally-filled cells used to terminate a bar of the graph.            *
//*                                                                            *
//* Input  : blkIndx : index into character array referencing character to     *
//*                    be inverted. NOTE: blkIndx should be strictly controlled*
//*                    in the range of 0-6 (inclusive). The arrays are exactly *
//*                    eight(8) elements AND offset 7 is the full-cell block   *
//*                    character.                                              *
//*                                                                            *
//* Returns: index of block character which mirrors the character refrenced    *
//*          by blkIndx                                                        *
//******************************************************************************

static short InvertFractionalCell ( short blkIndx )
{
   short invIndx ;

   switch ( blkIndx )
   {
      case 0: invIndx = 6 ; break ;
      case 1: invIndx = 5 ; break ;
      case 2: invIndx = 4 ; break ;
      case 3: invIndx = 3 ; break ;
      case 4: invIndx = 2 ; break ;
      case 5: invIndx = 1 ; break ;
      case 6: invIndx = 0 ; break ;
      default: invIndx = blkIndx ; break ;   // should never happen
   } ;

   return invIndx ;
}  //* End InvertFractionalCell() *

//*************************
//* InvertColorAttribute  *
//*************************
//******************************************************************************
//* Non-member method:                                                         *
//* ------------------                                                         *
//* For the specified color attribute return a color attribute with the        *
//* foreground and background colors swapped.                                  *
//*                                                                            *
//* When used in conjunction with the mirrored block character, it allows for  *
//* fractionally-filled cells used to terminate a bar of the graph.            *
//*                                                                            *
//* IMPORTANT NOTE: This methods uses intimate knowledge of the way color      *
//* attributes are encoded within the ncurses world, specifically as used by   *
//* the NcDialog API. All inversions have been manually tested and visually    *
//* verified. In the unlikely event that the ncurses library changes the way   *
//* it handles color attributes, this method may break.                        *
//*                                                                            *
//* Bit Definitions for a 32-bit character/attribute value (7-bit ASCII).      *
//* This is known as 'attr_t'. (see curses.h)                                  *
//* Note that 'wide' characters are combined with their color attribute        *
//* internally by the ncurses library; however, the color attribute bits       *
//* are the same for both narrow and wide characters.                          *
//*  BIT  31    = unused    -----------+                                       *
//*  BIT  30    = vertical             |                                       *
//*  BIT  29    = top                  |                                       *
//*  BIT  28    = right                +-- we make no use of these             *
//*  BIT  27    = low                  |   definitions in our implementation   *
//*  BIT  26    = left                 |                                       *
//*  BIT  25    = horizontal           |                                       *
//*  BIT  24    = "protected" mode ----+                                       *
//*  BIT  23    = invisible (background color is also used for foreground)     *
//*  BIT  22    = alternate character set (with line draw characters, etc.)    *
//*  BIT  21    = bold foreground against default background                   *
//*  BIT  20    = dim                                                          *
//*  BIT  19    = blinking (not functional under most terminal implementations)*
//*  BIT  18    = reverse (foreground/background swap)                         *
//*  BIT  17    = underline                                                    *
//*  BIT  16    = standout (foreground-to-background, with fg & bg bolded)     *
//*  BITS 15-8  = color pair number  (8, 16, 64, 256.  see init_pair() )       *
//*  BITS 7-0   = character code                                               *
//*                                                                            *
//* Refer to ISO 6429 for a description of color support for terminals.        *
//*                                                                            *
//* In this method, we simply toggle the 'Reverse' bit.                        *
//* In some cases we may swap the 8-bit color-pair sequence, and/or toggle     *
//* the 'Bold' bit.                                                            *
//* IMPORTANT NOTE: Inverting BOLD colors will likely fail due to inconsistent *
//*                 combinations of BOLD and REVERSE.                          *
//*                                                                            *
//* Input  : normColor : non-inverted color attribute                          *
//*                                                                            *
//* Returns: color attribute with foreground and background swapped.           *
//******************************************************************************

static attr_t InvertColorAttribute ( attr_t normColor )
{
   const attr_t mask = ~ncrATTR ;
   attr_t invColor ;

   if ( normColor & ncrATTR )
   {
      invColor = normColor & mask ;
   }
   else
   {
      invColor = normColor | ncrATTR ;
   }
   return invColor ;

}  //* End InvertColorAttribute() *

//*************************
//*                       *
//*************************
//******************************************************************************
//*                                                                            *
//*                                                                            *
//*                                                                            *
//* Input  :                                                                   *
//*                                                                            *
//* Returns:                                                                   *
//******************************************************************************

