//******************************************************************************
//* File       : CMouseTest.cpp                                                *
//* Author     : Mahlon R. Smith                                               *
//*              Copyright (c) 2013-2025 Mahlon R. Smith, The Software Samurai *
//*                  GNU GPL copyright notice located in NcDialog.hpp          *
//* Date       : 26-Jul-2021                                                   *
//* Version    : (see below)                                                   *
//*                                                                            *
//* Description: Class definition for the CMouseTest class.                    *
//*              This class exercises the NCurses-class mouse methods.         *
//*              It is instantiated by the Dialog4 application, Test03.        *
//*                                                                            *
//* Development Tools: See NcDialog.cpp.                                       *
//******************************************************************************
//* Version History (most recent first):                                       *
//*                                                                            *
//* v: 0.00.02 01-Jan-2020                                                     *
//*    -- Output formatting of the mouse record in the Billboard control       *
//*       modified due to a change in the definition of "mmask_t" in the       *
//*       underlying ncursesw library from 'long in' to 'int'. This may be an  *
//*       artifact of the port from 32-bit to 64-bit platforms.                *
//*       In any case, "mmask_t" is a 4-byte int.                              *
//*    -- Due to a new bug (feature?) in the underlying ncurses library        *
//*       (on or prior to v:6.1.20180923), the "REPORT_MOUSE_POSITION" flag    *
//*       of the 'bstate' field is no longer reported in response to a         *
//*       scroll-wheel-up event. (The position flag is sporatically reported   *
//*       during a scroll-wheel button-up event.)                              *
//*       -- Instead, the scroll-wheel-scroll-up event is now reported through *
//*         "BUTTON5_PRESSED". This is actually a better solution, but spent   *
//*         a whole afternoon chasing down the change in functionality, and    *
//*         all of the next morning updating our code and testing the changes. *
//*         We hates meeces to pieces!                                         *
//*       -- In addition, scroll-wheel events which include the CTRL/ALT/SHIFT *
//*          modifier keys no longer report the modifier flags. This may be    *
//*          due to the Wayland mouse driver and not the ncursesw mouse code.  *
//*       NcDialog library has been updated to reflect these changes.          *
//*                                                                            *
//* v: 0.00.01 03-Jun-2013 Layout based on ColorTest-class test in Dialog2.    *
//*                                                                            *
//******************************************************************************
//* Programmer's Note:                                                         *
//* - Click is reported only if press and release are in same character cell.  *
//*                                                                            *
//* - ncurses library mouse events are very sensitive to timing-related issues.*
//*                                                                            *
//* - BUTTON2_PRESSED is 'paste' in Gnome world. This can be disabled in       *
//*   Xterm if desired.                                                        *
//*                                                                            *
//*                                                                            *
//*                                                                            *
//*                                                                            *
//******************************************************************************

#include "CMouseTest.hpp"

//****************
//** Local Data **
//****************
static const short dialogROWS = ncmROWS ; // display lines
static const short dialogCOLS = ncmCOLS ; // display columns

static dspinData dsData ;                 // data for the cmDelSP spinner control

//* Display data and color attributes for event-selector control.*
static const short SelectBITS = 32, SelectWIDTH = 18 ;
static char     selectText[SelectBITS][SelectWIDTH] ;
static char*    selectPntr[SelectBITS] ;
static attr_t   selectAttr[SelectBITS] ;
static attr_t   selectColor, deselectColor ;
static short    lastSelectable = NCURSES_MOUSE_VERSION <= 1 ? 23 : 24 ;
static ssetData setSelect( (const char**)selectPntr, selectAttr, 
                            SelectBITS, ZERO, false ) ;


static InitCtrl ic[cmControlsDEFINED] =   // control initialization structures
{
   {  //* 'DONE' pushbutton - - - - - - - - - - - - - - - - - - -     cmDonePB *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(dialogROWS - 3),        // ulY:       upper left corner in Y
      short(dialogCOLS / 2 - 4),    // ulX:       upper left corner in X
      1,                            // lines:     control lines
      8,                            // cols:      control columns
      "  DONE  ",                   // dispText:  
      nc.gyR,                       // nColor:    non-focus color
      nc.reG,                       // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "",                           // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[cmOutBB],                 // nextCtrl:  link in next structure
   },
   {  //* 'OUTPUT' Billboard - - - - - - - - - - - - - - - - - - -     cmOutBB *
      dctBILLBOARD,                 // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      1,                            // ulY:       upper left corner in Y
      1,                            // ulX:       upper left corner in X
      14,                           // lines:     control lines
      short(dialogCOLS - 2),        // cols:      control columns
      NULL,                         // dispText:  initial display text
      nc.gybr | ncbATTR,            // nColor:    non-focus color
      nc.gybr | ncbATTR,            // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "Mouse Events",               // label:     
      short(ic[cmOutBB].lines),     // labY:      
      short(ic[cmOutBB].cols / 2 - 6), // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  if specified, attribute array
      NULL,                         // spinData:  (n/a)
      false,                        // active:    allow control to gain focus
      &ic[cmMaskSE],                // nextCtrl:  link in next structure
   },
   {  //* 'MASK scroll-ext control - - - - - - - - - - - - - - - - - cmMaskSE *
      dctSCROLLEXT,                 // type:      define a scrolling-data control
      rbtTYPES,                     // rbSubtype: (na)
      false,                        // rbSelect:  (n/a)
      short(dialogROWS / 2),        // ulY:       upper left corner in Y
      short(dialogCOLS - 20),       // ulX:       upper left corner in X
      8,                            // lines:     control lines
      short(SelectWIDTH + 1),       // cols:      control columns
      NULL,                         // dispText:  n/a
      nc.gybl,                      // nColor:    non-focus border color
      nc.rebl,                      // fColor:    focus border color
      tbPrint,                      // filter:    (n/a)
      "Event Mask Select",          // label:     
      8,                            // labY:      offset from control's ulY
      1,                            // labX       offset from control's ulX
      ddBoxTYPES,                   // exType:    (n/a)
      ZERO,                         // scrItems:  (n/a)
      ZERO,                         // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    view-only
      &ic[cmMaskTB],                // nextCtrl:  link in next structure
   },
   {  //* Mask-display text box (non-active) - - - - - - - - - -      cmMaskTB *
      dctTEXTBOX,                   // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(ic[cmMaskSE].ulY + 9),  // ulY:       upper left corner in Y
      short(ic[cmMaskSE].ulX + 4),  // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      11,                           // cols:      control columns
      " 00000000h ",                // dispText:  
      nc.gr,                        // nColor:    non-focus color
      nc.bw,                        // fColor:    focus color
      tbPrint,                      // filter:    any printing character
      NULL,                         // label:     
      ZERO,                         // labY:      
      ZERO,                         // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      false,                        // active:    cannot gain focus
      &ic[cmFilterRB],              // nextCtrl:  link in next structure
   },
   {  //* 'FILTER' radio button - - - - - - - - - - - - - - - - -   cmFilterRB *
      dctRADIOBUTTON,               // type:      
      rbtS3a,                       // rbSubtype: standard, 3 chars wide
      true,                         // rbSelect:  default selection
      short(ic[cmMaskTB].ulY + 1),  // ulY:       upper left corner in Y
      ic[cmMaskSE].ulX,             // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a)
      nc.gr,                        // nColor:    non-focus color
      nc.re,                        // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "Event Filter",               // label:
      ZERO,                         // labY:      
      4,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[cmWheelRB],               // nextCtrl:  link in next structure
   },
   {  //* 'SCROLL_WHEEL' radio button - - - - - - - - - - - - - - -  cmWheelRB *
      dctRADIOBUTTON,               // type:      
      rbtS3a,                       // rbSubtype: standard, 3 chars wide
      false,                        // rbSelect:  default selection
      short(ic[cmFilterRB].ulY + 1),// ulY:       upper left corner in Y
      ic[cmFilterRB].ulX,           // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a)
      nc.gr,                        // nColor:    non-focus color
      nc.re,                        // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "Scroll Wheel",               // label:
      ZERO,                         // labY:      
      4,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[cmDelaySP],               // nextCtrl:  link in next structure
   },
   {  //* 'DELAY' spinner - - - - - - - - - - - - - - - - - - - - -  cmDelaySP *
      dctSPINNER,                   // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      ic[cmDonePB].ulY,             // ulY:       upper left corner in Y
      short(dialogCOLS - 14),       // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      8,                            // cols:      control columns
      NULL,                         // dispText:  (n/a)
      nc.gyR,                       // nColor:    non-focus color
      nc.bw,                        // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "Click Delay",                // label:     
      1,                            // labY:      
      -1,                           // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      &dsData,                      // spinData:  spinner init
      true,                         // active:    allow control to gain focus
      &ic[cmClearPB],               // nextCtrl:  link in next structure
   },
   {  //* 'CLEAR' pushbutton  - - - - - - - - - - - - - - - - - -    cmClearPB *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      ic[cmDonePB].ulY,             // ulY:       upper left corner in Y
      short(ic[cmDonePB].ulX - 18), // ulX:       upper left corner in X
      1,                            // lines:     control lines
      15,                           // cols:      control columns
      "  CLEAR INPUT  ",            // dispText:  
      nc.gyR,                       // nColor:    non-focus color
      nc.maG,                       // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "",                           // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      NULL,                         // nextCtrl:  link in next structure
   },
} ;

//* Container for key-type label lookup *
class tLab
{
   public:
   short       type ;
   const char* label ;
} ;

//* Container for mouse-event-type label lookup *
class meLab
{
   public:
   ULONG       type ;
   const char* label ;
} ;


//*************************
//*     ~CMouseTest       *
//*************************
//******************************************************************************
//* Destructor.                                                                *
//*                                                                            *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

CMouseTest::~CMouseTest ( void )
{

   if ( this->dp != NULL )                // close the window
      delete ( this->dp ) ;

}  //* End ~CMouseTest() 

//*************************
//*      CMouseTest       *
//*************************
//******************************************************************************
//* Constructor.                                                               *
//*                                                                            *
//* Input  : tLines     : number of display line in terminal window            *
//*          tCols      : number of display columns in terminal window         *
//*          minY       : first available display line                         *
//*                                                                            *
//* Returns: implicitly return class reference                                 *
//******************************************************************************

CMouseTest::CMouseTest ( short tLines, short tCols, short minY )
{
   //* Initialize data members *
   this->termRows = tLines ;
   this->termCols = tCols ;
   this->minULY   = minY ;
   this->dp       = NULL ;
   this->dColor   = nc.gybl | ncbATTR ;
   this->bColor   = nc.brbl ;
   this->cmOpen   = false ;

   //* Initialize remainder of control-definition array.           *
   //* (colors become available after NCurses engine instantiated) *
   ic[cmDonePB].nColor   = nc.gyR ;
   ic[cmDonePB].fColor   = nc.reR ;
   ic[cmOutBB].nColor    = nc.gybr | ncbATTR ;
   ic[cmOutBB].fColor    = nc.gybr | ncbATTR ;
   ic[cmMaskSE].nColor   = nc.gybl ;
   ic[cmMaskSE].fColor   = nc.rebl ;
   ic[cmMaskTB].nColor   = nc.gr ;
   ic[cmMaskTB].fColor   = nc.bw ;
   ic[cmFilterRB].nColor = nc.gr ;
   ic[cmFilterRB].fColor = nc.re ;
   ic[cmWheelRB].nColor  = nc.gr ;
   ic[cmWheelRB].fColor  = nc.re ;
   ic[cmDelaySP].nColor  = nc.gyR ;
   ic[cmDelaySP].fColor  = nc.bw ;
   ic[cmClearPB].nColor  = nc.gyR ;
   ic[cmClearPB].fColor  = nc.maG ;
   selectColor   = nc.grB ;
   deselectColor = nc.gy ;

   //* Initialize the dctSCROLLEXT control data *
   this->cmInitSE () ;

   //* Initialize the dctSPINNER control data *
   dsData.minValue = ncmiNONE ;
   dsData.maxValue = ncmiMAX ;
   dsData.iniValue = ncmiDFLT ;
   dsData.dsFormat = dspinINTEGER ;
   dsData.indiAttr = nc.grG ;

   if ( (this->cmOpen = this->cmOpenDialog ()) != false )
   {  //* Initialize data for event selector *
      this->dp->SetScrollextText ( cmMaskSE, setSelect ) ;
      this->cmSynch2EMask () ;      // indicate which mask bits are active

   }  // OpenWindow()

}  //* End CMouseTest() 

//*************************
//*     cmOpenDialog      *
//*************************
//******************************************************************************
//* Open the dialog window.                                                    *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: 'true' if dialog window opened successfully, else 'false'         *
//******************************************************************************

bool CMouseTest::cmOpenDialog ( void )
{
   short ctrY    = this->termRows/2 + 1,     // dialog center in Y
         ctrX    = this->termCols/2,         // dialog center in X
         //* Upper left corner in Y (cannot obscure status window) *
         ulY     = (ctrY - dialogROWS/2) >= this->minULY ? 
                   (ctrY - dialogROWS/2) : this->minULY,
         ulX     = ctrX - dialogCOLS / 2 ;   // upper left corner in X
   bool  success = false ;

   //* Initial parameters for dialog window *
   InitNcDialog dInit( dialogROWS,     // number of display lines
                       dialogCOLS,     // number of display columns
                       ulY,            // Y offset from upper-left of terminal 
                       ulX,            // X offset from upper-left of terminal 
                       "  NCurses-class Mouse Methods  ", // dialog title
                       ncltDUAL,       // border line-style
                       bColor,         // border color attribute
                       dColor,         // interior color attribute
                       ic              // pointer to list of control definitions
                     ) ;

   //* Instantiate the dialog window *
   this->dp = new NcDialog ( dInit ) ;

   //* Open the dialog window *
   if ( (this->dp->OpenWindow()) == OK )
   {
      #if 0    // TESTING ONLY - SET ALL BITS
      mmask_t  newMask = ALL_MOUSE_EVENTS | REPORT_MOUSE_POSITION,
               eventTypes ;
      nc.EnableMouseEvents ( newMask ) ;
      nc.GetMouseEventTypes ( eventTypes ) ;
      gString gs ;
      gs.compose( L"newMask 0x%08X eventTypes:0x%08X", &newMask, &eventTypes ) ;
      this->dp->WriteString ( dialogROWS-2, 2, gs, this->dColor ) ;
      #else    // PROD
      //* Enable mouse input - (Button 1,2,3, single/double/triple clicks) *
      mmask_t newMask = 
         BUTTON1_CLICKED | BUTTON1_DOUBLE_CLICKED | BUTTON1_TRIPLE_CLICKED |
         BUTTON2_CLICKED | BUTTON2_DOUBLE_CLICKED | BUTTON2_TRIPLE_CLICKED |
         BUTTON3_CLICKED | BUTTON3_DOUBLE_CLICKED | BUTTON3_TRIPLE_CLICKED ;
      if ( (nc.EnableMouseEvents ( newMask )) == OK )
      {
         short initialInterval ;
         if ( (nc.GetMouseClickInterval ( initialInterval )) == OK )
         {
            dsData.iniValue = initialInterval ;
            this->dp->SetSpinnerValue ( cmDelaySP, dsData.iniValue ) ;
         }
      }
      else
      {
         this->dp->WriteString ( (dialogROWS - 2), (dialogCOLS / 2 - 20),
               " ERROR! UNABLE TO ACTIVATE MOUSE INPUT! ", nc.bkma ) ;
      }
      #endif   // PROD

      winPos wp( 17, 2 ) ;
      wp = this->dp->WriteString ( wp, "ncurses mouse version: ", bColor ) ;
      short mver = NCURSES_MOUSE_VERSION ;
      gString gs( "%hd", &mver ) ;
      this->dp->WriteString ( wp, gs, dColor ) ;

      this->dp->RefreshWin () ;     // make everything visible
      success = true ;
   }
   return success ;

}  //* End cmOpenDialog() *

//*************************
//*    cmDialogOpened     *
//*************************
//******************************************************************************
//* Satisfy caller's curiosity.                                                *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: 'true' if dialog opened successfully, else 'false'                *
//******************************************************************************

bool CMouseTest::cmDialogOpened ( void )
{
   return this->cmOpen ;

}  //* End cmDialogOpened() *

//*************************
//*      cmInteract       *
//*************************
//******************************************************************************
//* User interactive experience.                                               *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void CMouseTest::cmInteract ( void )
{
   //* Types of key input *
   const short tlCount = 6 ;
   tLab typeLabels[tlCount] = 
   {
      { wktPRINT,  "wktPRINT " },
      { wktFUNKEY, "wktFUNKEY" },
      { wktMOUSE,  "wktMOUSE " },
      { wktEXTEND, "wktEXTEND" },
      { wktESCSEQ, "wktESCSEQ" },
      { wktERR,    "wktERR   " },
   } ;
   //* Mouse-event types *
   meLab meLabels[32] = 
   {
      { ZERO,                    "none:" },
      #if NCURSES_MOUSE_VERSION <= 1
      { BUTTON1_RELEASED,        "B1rel" },
      { BUTTON1_PRESSED,         "B1pre" },
      { BUTTON1_CLICKED,         "B1cl1" },
      { BUTTON1_DOUBLE_CLICKED,  "B1cl2" },
      { BUTTON1_TRIPLE_CLICKED,  "B1cl3" },
      { BUTTON1_RESERVED_EVENT,  "B1res" },

      { BUTTON2_RELEASED,        "B2rel" },
      { BUTTON2_PRESSED,         "B2pre" },
      { BUTTON2_CLICKED,         "B2cl1" },
      { BUTTON2_DOUBLE_CLICKED,  "B2cl2" },
      { BUTTON2_TRIPLE_CLICKED,  "B2cl3" },
      { BUTTON2_RESERVED_EVENT,  "B2res" },
                                 
      { BUTTON3_RELEASED,        "B3rel" },
      { BUTTON3_PRESSED,         "B3pre" },
      { BUTTON3_CLICKED,         "B3cl1" },
      { BUTTON3_DOUBLE_CLICKED,  "B3cl2" },
      { BUTTON3_TRIPLE_CLICKED,  "B3cl3" },
      { BUTTON3_RESERVED_EVENT,  "B3res" },
                                 
      { BUTTON4_RELEASED,        "B4rel" },
      { BUTTON4_PRESSED,         "B4pre" },
      { BUTTON4_CLICKED,         "B4cl1" },
      { BUTTON4_DOUBLE_CLICKED,  "B4cl2" },
      { BUTTON4_TRIPLE_CLICKED,  "B4cl3" },
      { BUTTON4_RESERVED_EVENT,  "B4res" },
      #else    // NCURSES_MOUSE_VERSION > 1
      { BUTTON1_RELEASED,        "B1rel" },
      { BUTTON1_PRESSED,         "B1pre" },
      { BUTTON1_CLICKED,         "B1cl1" },
      { BUTTON1_DOUBLE_CLICKED,  "B1cl2" },
      { BUTTON1_TRIPLE_CLICKED,  "B1cl3" },

      { BUTTON2_RELEASED,        "B2rel" },
      { BUTTON2_PRESSED,         "B2pre" },
      { BUTTON2_CLICKED,         "B2cl1" },
      { BUTTON2_DOUBLE_CLICKED,  "B2cl2" },
      { BUTTON2_TRIPLE_CLICKED,  "B2cl3" },
                                 
      { BUTTON3_RELEASED,        "B3rel" },
      { BUTTON3_PRESSED,         "B3pre" },
      { BUTTON3_CLICKED,         "B3cl1" },
      { BUTTON3_DOUBLE_CLICKED,  "B3cl2" },
      { BUTTON3_TRIPLE_CLICKED,  "B3cl3" },

      { BUTTON4_RELEASED,        "B4rel" },
      { BUTTON4_PRESSED,         "B4pre" },
      { BUTTON4_CLICKED,         "B4cl1" },
      { BUTTON4_DOUBLE_CLICKED,  "B4cl2" },
      { BUTTON4_TRIPLE_CLICKED,  "B4cl3" },

      { BUTTON5_RELEASED,        "B5rel" },
      { BUTTON5_PRESSED,         "B5pre" },
      { BUTTON5_CLICKED,         "B5cl1" },
      { BUTTON5_DOUBLE_CLICKED,  "B5cl2" },
      { BUTTON5_TRIPLE_CLICKED,  "B5cl3" },
      #endif   // NCURSES_MOUSE_VERSION > 1
   } ;

   if ( this->cmOpen )
   {
      gString gsOut ;                  // for formatting output
      winPos   wp( 2, 2 ) ;            // output position
      wkeyCode wk ;                    // key input
      const char* tlPtr ;              // pointer to key-type label
      const char* mePtr ;              // pointer to mouse-event-type label
      char cK, sK, aK, pF ;            // Ctrl, Shift, Alt keys, position flag

      //* Track user's selections *
      uiInfo   Info ;                  // user interface data returned here
      short    icIndex = ZERO ;        // index of control with input focus
      bool     done = false ;          // loop control
      while ( ! done )
      {
         //*************************************
         //* Scan user input for mouse events. *
         //*************************************
         nc.KeyPeek ( wk ) ;
         if ( wk.type == wktMOUSE || wk.type == wktERR )
         {
            //* If we have a mouse event, report it *
            if ( wk.type == wktMOUSE )
            {
               nc.GetKeyInput ( wk ) ;
               //* Report key type *
               for (short i = ZERO ; i < tlCount ; i++ )
               {
                  if (wk.type == typeLabels[i].type)
                  { tlPtr = typeLabels[i].label ; break ; }
               }
               //* Report mouse-event type *
               mePtr = meLabels[0].label ;      // if no event
               for (short i = ZERO ; i < lastSelectable ; i++ )
               {
                  if ( wk.mevent.eventType & meLabels[i].type )
                  { mePtr = meLabels[i].label ; break ; }
               }
               //* Report conditioning keys *
               cK = sK = aK = pF = '-' ;
               if ( (wk.mevent.eventType & BUTTON_CTRL)  != ZERO )    cK = 'C' ;
               if ( (wk.mevent.eventType & BUTTON_SHIFT) != ZERO )    sK = 'S' ;
               if ( (wk.mevent.eventType & BUTTON_ALT)   != ZERO )    aK = 'A' ;
               if ( (wk.mevent.eventType & REPORT_MOUSE_POSITION) != ZERO ) pF = 'P' ;
               gsOut.compose( 
                  L" type:%s key:%04Xh  bits:%08Xh (%s %c%c%c%c)"
                   " y:%-2hd x:%-3hd id:%hd", 
                  tlPtr, &wk.key, &wk.mevent.eventType, mePtr, &cK, &sK, &aK, &pF, 
                  &wk.mevent.ypos, &wk.mevent.xpos, &wk.mevent.deviceID ) ;
               this->dp->Append2Billboard ( cmOutBB, gsOut ) ;
            }
            continue ;  // return to top of loop
         }


         //*******************************************
         //* If focus is currently on a Pushbutton   *
         //*******************************************
         if ( ic[icIndex].type == dctPUSHBUTTON )
         {  //* Get user input *
            icIndex = this->dp->EditPushbutton ( Info ) ;

            //* If a Pushbutton was pressed *
            if ( Info.dataMod != false )
            {
               if ( Info.ctrlIndex == cmDonePB )
               {  //* We're done *
                  done = true ;
               }
               else if ( Info.ctrlIndex == cmClearPB )
               {  //* Clear the input area *
                  this->dp->ClearBillboard ( cmOutBB ) ;
                  nc.FlushMouseEventQueue () ;  // discard pending data
                  nc.FlushKeyInputStream () ;
                  Info.keyIn = nckTAB ;   // allow focus to move forward
               }
            }
         }     // dctPUSHBUTTON

         //*********************************************
         //* If focus is currently on a Scroll Ext Box *
         //*********************************************
         else if ( ic[icIndex].type == dctSCROLLEXT )
         {
            this->dp->RefreshScrollextText ( cmMaskSE, true ) ; // enable highlight
            do
            {
               icIndex = this->dp->EditScrollext ( Info ) ;
               //* We act only if user has selected/deselected an item, *
               //* not based on the highlight position.                 *
               if ( Info.keyIn == nckENTER || Info.keyIn == nckSPACE )
                  Info.dataMod = true ;
               else
                  Info.dataMod = false ;
               if ( (Info.dataMod != false) && (Info.selMember <= lastSelectable) )
               {  //* If user has selected/deselected an item, update the display *
                  //* text and refresh the control. Then update the event mask.   *
                  selectAttr[Info.selMember] = 
                     (selectAttr[Info.selMember] == selectColor) ?
                     deselectColor : selectColor ;
                  this->dp->MoveScrollextHighlight ( cmMaskSE, nckDOWN ) ;
                  this->cmSetEMask () ;
               }
            }
            while ( Info.dataMod != false ) ;
            this->dp->RefreshScrollextText ( cmMaskSE, false ) ; // remove highlight
         }

         //*******************************************
         //* If focus is currently on a Radio Button *
         //*******************************************
         else if ( ic[icIndex].type == dctRADIOBUTTON )
         {
            icIndex = this->dp->EditRadiobutton ( Info ) ;
            if ( Info.dataMod != false )
            {
               if ( Info.ctrlIndex == cmFilterRB )
               {  //* Enable/disable filtering *
                  nc.FilterSpuriousMouseEvents ( Info.isSel ) ;
               }
               else if ( Info.ctrlIndex == cmWheelRB )
               {  //* Enable/disable the ScrollWheel bits *
                  this->cmSetEMask () ;
                  this->cmSynch2EMask () ;
               }
            }
         }

         //**************************************
         //* If focus is currently on a Spinner *
         //**************************************
         else if ( ic[icIndex].type == dctSPINNER )
         {
            icIndex = this->dp->EditSpinner ( Info ) ;
            if ( Info.dataMod != false )
            {
               int newDelay ;
               this->dp->GetSpinnerValue ( cmDelaySP, newDelay ) ;
               nc.SetMouseClickInterval ( short(newDelay) ) ;
            }
         }

         //****************************************
         //* If focus is currently on a Billboard *
         //****************************************
         else if ( ic[icIndex].type == dctBILLBOARD )
         {
            icIndex = this->dp->EditBillboard ( Info ) ;
            if ( Info.dataMod != false )
            {
               //* Because Billboard data are not editable *
               //* we should never arrive here.            *
            }
         }

         //* Move to next/previous control *
         if ( done == false )
         {
            if ( Info.keyIn == nckSTAB )
               icIndex = this->dp->PrevControl () ; 
            else if ( Info.keyIn != ZERO )
               icIndex = this->dp->NextControl () ;

            //* Spcial case: If focus in on the dctSCROLLEXT control, then   *
            //* we need to refresh the control NOW to make highlight visible *
            //* because scan for mouse events delays the control update.     *
            if ( ic[icIndex].type == dctSCROLLEXT )
               this->dp->RefreshScrollextText ( cmMaskSE, true ) ;

         }
      }  // while()
      nc.DisableMouseEvents () ;    // disable mouse input
   }
   else
      { /* Caller is an idiot */ }

}  //* End cmInteract() *

//*************************
//*       cmInitSE        *
//*************************
//******************************************************************************
//* Construct display strings for the event-selector control.                  *
//* NOTE: This is a bit clunky because we assume that the bitmask definitions  *
//*       in ncurses.h will not change. If they do, the display will be wrong. *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void CMouseTest::cmInitSE ( void )
{
   #if NCURSES_MOUSE_VERSION <= 1
   const char* iStrings[SelectBITS] =        // Mouse version <=1 definitions
   { //12|4567890123456|
      " BUTTON1_RELEASE ", " BUTTON1_PRESS   ", " BUTTON1_CLICK1  ",
      " BUTTON1_CLICK2  ", " BUTTON1_CLICK3  ", " BUTTON1_RESERVE ",
      " BUTTON2_RELEASE ", " BUTTON2_PRESS   ", " BUTTON2_CLICK1  ",
      " BUTTON2_CLICK2  ", " BUTTON2_CLICK3  ", " BUTTON2_RESERVE ",
      " BUTTON3_RELEASE ", " BUTTON3_PRESS   ", " BUTTON3_CLICK1  ",
      " BUTTON3_CLICK2  ", " BUTTON3_CLICK3  ", " BUTTON3_RESERVE ",
      " BUTTON4_RELEASE ", " BUTTON4_PRESS   ", " BUTTON4_CLICK1  ",
      " BUTTON4_CLICK2  ", " BUTTON4_CLICK3  ", " BUTTON4_RESERVE ",
      " CTRL Key (read) ", " SHFT Key (read) ",
      " ALT  Key (read) ", " Position (read) ",
      " (unassigned)    ", " (unassigned)    ",
      " (unassigned)    ", " (unassigned)    ",
   } ;                                       
   #else    // NCURSES_MOUSE_VERSION >1
   const char* iStrings[SelectBITS] =        // Mouse version >1 definitions
   { //12|4567890123456|
      " BUTTON1_RELEASE ", " BUTTON1_PRESS   ", " BUTTON1_CLICK1  ",
      " BUTTON1_CLICK2  ", " BUTTON1_CLICK3  ",
      " BUTTON2_RELEASE ", " BUTTON2_PRESS   ", " BUTTON2_CLICK1  ",
      " BUTTON2_CLICK2  ", " BUTTON2_CLICK3  ",
      " BUTTON3_RELEASE ", " BUTTON3_PRESS   ", " BUTTON3_CLICK1  ",
      " BUTTON3_CLICK2  ", " BUTTON3_CLICK3  ",
      " BUTTON4_RELEASE ", " BUTTON4_PRESS   ", " BUTTON4_CLICK1  ",
      " BUTTON4_CLICK2  ", " BUTTON4_CLICK3  ",
      " BUTTON5_RELEASE ", " BUTTON5_PRESS   ", " BUTTON5_CLICK1  ",
      " BUTTON5_CLICK2  ", " BUTTON5_CLICK3  ",
      " CTRL Key (read) ", " SHFT Key (read) ",
      " ALT  Key (read) ", " Position (read) ",
      " (unassigned)    ", " (unassigned)    ", " (unassigned)    ",
   } ;
   #endif   // NCURSES_MOUSE_VERSION >1
   gString gs ;
   for ( short items = ZERO ; items < SelectBITS ; items++ )
   {
      gs = iStrings[items] ;
      gs.limitCols( SelectWIDTH - 1 ) ;
      gs.copy( selectText[items], SelectWIDTH ) ;
      selectPntr[items] = selectText[items] ;
      selectAttr[items] = items <= lastSelectable ? deselectColor : nc.bw ;
   }

}  //* End cmInitSE() *

//*************************
//*      cmSetEMask       *
//*************************
//******************************************************************************
//* Set a new value for the NCurses-class event mask based on the selections   *
//* in the cmMaskSE control.                                                   *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: 'true' if value set successfully, else 'false'                    *
//******************************************************************************
//* Programmer's Note: The two bits that are used for the ScrollWheel are      *
//* forced on or off according to 'wheelFlag' regardless of the individual bits*
//* selected by the user.                                                      *
//*                                                                            *
//******************************************************************************

bool CMouseTest::cmSetEMask ( void )
{
   mmask_t  cmask = ZERO, cmark = 0x00000001 ;
   for ( short i = ZERO ; i <= lastSelectable ; i++ )
   {
      if ( setSelect.dispColor[i] == selectColor )
         cmask |= cmark ;
      cmark <<= 1 ;
   }

   //* Test whether scroll-wheel reporting is enabled *
   bool wheelFlag ;
   this->dp->GetRadiobuttonState ( cmWheelRB, wheelFlag ) ;
   if ( wheelFlag )
      cmask |= REPORT_SCROLL_WHEEL ;
   else
      cmask &= ~REPORT_SCROLL_WHEEL ;

   bool status = ((nc.SetMouseEventTypes ( cmask )) == OK) ;
   this->cmUpdateTBox ( cmask ) ;
   return status ;

}  //* End cmSetEMask() *

//*************************
//*    cmSynch2EMask      *
//*************************
//******************************************************************************
//* Synchronize the display of data in the event-mask selector with the active *
//* event mask AND update the hex value displayed.                             *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void CMouseTest::cmSynch2EMask ( void )
{
   mmask_t  cmask ;
   if ( (nc.GetMouseEventTypes ( cmask )) == OK )
   {
      short si = ZERO ;
      selectAttr[si++] = ((cmask & BUTTON1_RELEASED) != ZERO) ? 
                           selectColor : deselectColor ;
      selectAttr[si++] = ((cmask & BUTTON1_PRESSED ) != ZERO) ? 
                           selectColor : deselectColor ;
      selectAttr[si++] = ((cmask & BUTTON1_CLICKED ) != ZERO) ? 
                           selectColor : deselectColor ;
      selectAttr[si++] = ((cmask & BUTTON1_DOUBLE_CLICKED ) != ZERO) ? 
                           selectColor : deselectColor ;
      selectAttr[si++] = ((cmask & BUTTON1_TRIPLE_CLICKED ) != ZERO) ? 
                           selectColor : deselectColor ;
      #if NCURSES_MOUSE_VERSION <= 1
      selectAttr[si++] = ((cmask & BUTTON1_RESERVED_EVENT ) != ZERO) ? 
                           selectColor : deselectColor ;
      selectAttr[si++] = ((cmask & BUTTON2_RELEASED       ) != ZERO) ? 
                           selectColor : deselectColor ;
      selectAttr[si++] = ((cmask & BUTTON2_PRESSED        ) != ZERO) ? 
                           selectColor : deselectColor ;
      selectAttr[si++] = ((cmask & BUTTON2_CLICKED        ) != ZERO) ? 
                           selectColor : deselectColor ;
      selectAttr[si++] = ((cmask & BUTTON2_DOUBLE_CLICKED ) != ZERO) ? 
                           selectColor : deselectColor ;
      selectAttr[si++] = ((cmask & BUTTON2_TRIPLE_CLICKED ) != ZERO) ? 
                           selectColor : deselectColor ;
      selectAttr[si++] = ((cmask & BUTTON2_RESERVED_EVENT ) != ZERO) ? 
                           selectColor : deselectColor ;
      selectAttr[si++] = ((cmask & BUTTON3_RELEASED       ) != ZERO) ? 
                           selectColor : deselectColor ;
      selectAttr[si++] = ((cmask & BUTTON3_PRESSED        ) != ZERO) ? 
                           selectColor : deselectColor ;
      selectAttr[si++] = ((cmask & BUTTON3_CLICKED        ) != ZERO) ? 
                           selectColor : deselectColor ;
      selectAttr[si++] = ((cmask & BUTTON3_DOUBLE_CLICKED ) != ZERO) ? 
                           selectColor : deselectColor ;
      selectAttr[si++] = ((cmask & BUTTON3_TRIPLE_CLICKED ) != ZERO) ? 
                           selectColor : deselectColor ;
      selectAttr[si++] = ((cmask & BUTTON3_RESERVED_EVENT ) != ZERO) ? 
                           selectColor : deselectColor ;
      selectAttr[si++] = ((cmask & BUTTON4_RELEASED       ) != ZERO) ? 
                           selectColor : deselectColor ;
      selectAttr[si++] = ((cmask & BUTTON4_PRESSED        ) != ZERO) ? 
                           selectColor : deselectColor ;
      selectAttr[si++] = ((cmask & BUTTON4_CLICKED        ) != ZERO) ? 
                           selectColor : deselectColor ;
      selectAttr[si++] = ((cmask & BUTTON4_DOUBLE_CLICKED ) != ZERO) ? 
                           selectColor : deselectColor ;
      selectAttr[si++] = ((cmask & BUTTON4_TRIPLE_CLICKED ) != ZERO) ? 
                           selectColor : deselectColor ;
      selectAttr[si++] = ((cmask & BUTTON4_RESERVED_EVENT ) != ZERO) ? 
                           selectColor : deselectColor ;
      #else    // NCURSES_MOUSE_VERSION >1
      selectAttr[si++] = ((cmask & BUTTON2_RELEASED       ) != ZERO) ? 
                           selectColor : deselectColor ;
      selectAttr[si++] = ((cmask & BUTTON2_PRESSED        ) != ZERO) ? 
                           selectColor : deselectColor ;
      selectAttr[si++] = ((cmask & BUTTON2_CLICKED        ) != ZERO) ? 
                           selectColor : deselectColor ;
      selectAttr[si++] = ((cmask & BUTTON2_DOUBLE_CLICKED ) != ZERO) ? 
                           selectColor : deselectColor ;
      selectAttr[si++] = ((cmask & BUTTON2_TRIPLE_CLICKED ) != ZERO) ? 
                           selectColor : deselectColor ;
      selectAttr[si++] = ((cmask & BUTTON3_RELEASED       ) != ZERO) ? 
                           selectColor : deselectColor ;
      selectAttr[si++] = ((cmask & BUTTON3_PRESSED        ) != ZERO) ? 
                           selectColor : deselectColor ;
      selectAttr[si++] = ((cmask & BUTTON3_CLICKED        ) != ZERO) ? 
                           selectColor : deselectColor ;
      selectAttr[si++] = ((cmask & BUTTON3_DOUBLE_CLICKED ) != ZERO) ? 
                           selectColor : deselectColor ;
      selectAttr[si++] = ((cmask & BUTTON3_TRIPLE_CLICKED ) != ZERO) ? 
                           selectColor : deselectColor ;
      selectAttr[si++] = ((cmask & BUTTON4_RELEASED       ) != ZERO) ? 
                           selectColor : deselectColor ;
      selectAttr[si++] = ((cmask & BUTTON4_PRESSED        ) != ZERO) ? 
                           selectColor : deselectColor ;
      selectAttr[si++] = ((cmask & BUTTON4_CLICKED        ) != ZERO) ? 
                           selectColor : deselectColor ;
      selectAttr[si++] = ((cmask & BUTTON4_DOUBLE_CLICKED ) != ZERO) ? 
                           selectColor : deselectColor ;
      selectAttr[si++] = ((cmask & BUTTON4_TRIPLE_CLICKED ) != ZERO) ? 
                           selectColor : deselectColor ;
      selectAttr[si++] = ((cmask & BUTTON5_RELEASED       ) != ZERO) ? 
                           selectColor : deselectColor ;
      selectAttr[si++] = ((cmask & BUTTON5_PRESSED        ) != ZERO) ? 
                           selectColor : deselectColor ;
      selectAttr[si++] = ((cmask & BUTTON5_CLICKED        ) != ZERO) ? 
                           selectColor : deselectColor ;
      selectAttr[si++] = ((cmask & BUTTON5_DOUBLE_CLICKED ) != ZERO) ? 
                           selectColor : deselectColor ;
      selectAttr[si++] = ((cmask & BUTTON5_TRIPLE_CLICKED ) != ZERO) ? 
                           selectColor : deselectColor ;
      #endif   // NCURSES_MOUSE_VERSION >1

      this->dp->RefreshScrollextText ( cmMaskSE, false ) ;
      this->cmUpdateTBox ( cmask ) ;
   }

}  //* End cmSynch2EMask() *

//*************************
//*     cmUpdateTBox      *
//*************************
//******************************************************************************
//* Update the dispay data with current mouse-event mask.                      *
//*                                                                            *
//* Input  : cmask  : event mask                                               *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void CMouseTest::cmUpdateTBox ( mmask_t cmask )
{
   gString gs ;
   gs.compose( L" %08Xh ", &cmask ) ;
   this->dp->SetTextboxText ( cmMaskTB, gs ) ;

}  //* End cmUpdateTBox() *

//* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -  *
//* - - - - - - - - - - - - Non-member Methods- - - - - - - - - - - - - - - -  *
//* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -  *

//*************************
//*                       *
//*************************
//******************************************************************************
//*                                                                            *
//*                                                                            *
//*                                                                            *
//* Input  :                                                                   *
//*                                                                            *
//* Returns:                                                                   *
//******************************************************************************


