//******************************************************************************
//* File       : BillboardTest.cpp                                             *
//* Author     : Mahlon R. Smith                                               *
//*              Copyright (c) 2013-2025 Mahlon R. Smith, The Software Samurai *
//* Date       : 03-Jan-2020                                                   *
//* Version    : (see below)                                                   *
//*                                                                            *
//* Description: Class definition for the BillboartTest class.                 *
//*              This class exercises the DialogBillboard-class control        *
//*              defined in NcDialog.hpp.                                      *
//*              It is instantiated by the Dialog4 application, Test05.        *
//*                                                                            *
//*              Note that this module uses multi-threading. If your system    *
//*              does not support multithreading, you can disable the secondary*
//*              thread using the ENABLE_BT_MULTITHREAD definition in          *
//*              BillboardTest.hpp.                                            *
//*                                                                            *
//* Developed using GNU G++ (Gcc v: 4.8.0)                                     *
//*  under Fedora Release 16, Kernel Linux 3.6.11-4.fc16.i686.PAE              *
//******************************************************************************
//* Version History (most recent first):                                       *
//*                                                                            *
//* v: 0.00.02 06-Feb-2014                                                     *
//*   - Add test for NcDialog-class CaptureDialog() method. This test is       *
//*     invoked by the nckA_Z or nckAS_Z key while in the interactive loop.    *
//*     See btControlUpdate().                                                 *
//*                                                                            *
//* v: 0.00.01 24-Jun-2013                                                     *
//*   - Layout based on ColorTest-class test in Dialog2.                       *
//*   - Note on ENABLE_BT_MULTITHREAD != 0                                     *
//*     Lack of thread safety in NcDialog output routines can cause occasional *
//*     screen artifacts, but these do not affect functionality.               *
//******************************************************************************
//* Programmer's Note:                                                         *
//*                                                                            *
//*                                                                            *
//*                                                                            *
//*                                                                            *
//*                                                                            *
//*                                                                            *
//******************************************************************************

#include "BillboardTest.hpp"

//****************
//** Local Data **
//****************
static const short dialogROWS = btROWS ;  // display lines
static const short dialogCOLS = btCOLS ;  // display columns

//** Data for btBilaBB control **
//-------------------------------
//* Multi-color data array - enabled/disabled by btMultiRB *
static attr_t bba_cData[Bill1ROWS] ;

//* Data invoked by btUsBB - UTF-8 character data that fits the control *
static const char* bba_uDatas = 
{
   "Welcome\n  to\n   your\n    nightmare!\n"
   "     Ho!\n      ho!\n       ho!\n        ho!\n"
   "         A\n         Bruja\n          is\n           a\n"
   "            witch --\n"
   "             and a gargle is a gargoyle...  :-)\n"
   "    If you see this line, the programmer screwed up!    "
} ;

//* Data invoked by btWsBB - UTF-8 character data that fits the control *
static const wchar_t* bba_wDatas = 
{
  L"Apples and oranges are things to share\n"
   "Bears and beasties are things that scare\n"
   "Children and burdens are things to bear, but\n"
   "Ducks and chickens are things to cook.\n"
   "Elba and Crete are in the sea\n"
   "Finches and magpies are in a tree\n"
   "Guanyin and honey are in our tea, but\n"
   "Homer and Hamlet are in a book.\n"
   "Ice cream and impulse are everywhere\n"
   "Jenga and Jarjar, we just don't care\n"
   "Kids and koalas are likely to stare, but\n"
   "Laughter is lovely, so have a look.\n"
   "  Maybe\t you're wondering about our sanity, but\n"
   "     Never \tdoubt our capacity for drinking "
        "fresh coffee until we exude \t\tpoetry!"// (note extraneous Tab characters)
} ;

//* Data invoked by btUlBB - UTF-8 character data for line-by-line scrolling *
// (Note extraneous Tab characters inserted for testing.  )
// (these s/b stripped AND return value properly adjusted )
static const char* bba_uDatal = 
{
 "   The town had endured a mischievous winter. The eaves dripped ice talons, "
 "giving the houses a monstrous, carnivorous quality. The streets translated "
 "their winter burden into a city-wide grid of slalom runs, over which the "
 "children joyously ran and fell\t, but over which those unfortunate enough to "
 "need distant travel, crawled with obvious dis-ease. Harold, the constable\t, "
 "rolled slowly along the thoroughfare with his studded snow tires ripping "
 "regular chinks from the street's January armor.  Harold's mother, eighty-"
 "seven, but as she would tell you “nobody's fool,” had packed a quart of "
 "boiling coffee with his customary ham-and-cheese, so Harold scarcely felt the "
 "chill wind shouldering it\t's way through the broken rear window of the cruiser.\n"
 "   Harold had stopped at Sam's garage the evening before, after depositing "
 "Tommy Sands, the despoiler of Harold's cruiser in the drunk tank, but as Sam "
 "could not get a replacement window until Tuesday\t, Harold had resigned himself "
 "to a few days of closer communion with God's natural wonders.  \n"
 // End of first page of data

 "   Harold held a healthy respect for God and his wonders, having peeled old\n"
 "Josiah \tRenquist from behind the wheel of his pickup two-days since, after\n"
 "Josiah \thad missed a turn and had frozen to death, dashboard-deep in the creek\n"
 "over the course of a long, New Hampshire night. Frieda Willhoby, shovel in\n"
 "hand, stood knee-deep in the drift that had enveloped her front porch, and\n"
 "Harold stopped, leaning out the driver's window to chat for a few minutes.\n"

 "   Frieda never missed a chance to ask, “Cold enough for ya'?,” and Harold\n"
 "could not disappoint her by passing on by. Although the small-town telegraph\n"
 "had already transmitted every fact concerning Josiah Renquist's death to\n"
 "everyone old enough to use a telephone, Frieda let fly as professional an\n"
 "interrogation on the matter as Harold had ever encountered.  When Frieda\n"
 "seemed satisfied as to the particulars, Harold waved, slipping the cruiser\n"
 "back into gear, and crunched on down the street.\n"

 "   As he rounded the corner at Baker's Drugs, Harold noticed a knot of men,\n"
 "women, and children in their shirtsleeves converged in the Hudson's driveway,\n"
 "and pulled over to see what had lured them into a 14-degree morning without\n"
 "their coats. Ernie Hudson saw him coming and clambered over the snow plow's\n"
 "drift to meet him.\n"
 "   “Harold, we got us a situation here and can't seem to decide what to do\n"
 "about it.”\n"

 "   It seems that a family of skunks, near-frozen and unable to dig back into\n"
 "their den, had set up camp under the Hudson's kitchen stove. Unwilling to\n"
 "arouse them and thereby make their house uninhabitable for several days, the\n"
 "Hudsons had relinquished the field of battle with the kettle still singing\n"
 "away and biscuits in the oven. Harold eased out of the cruiser, withholding\n"
 "his smile only with difficulty, and began to rummage in the trunk, coming up\n"
 "with a burlap bag tied with string.\n"
 "   “Ernie, will you just slip around and ease open the kitchen door?” With\n"
 "this, Harold and his bag disappeared through the front door while the growing\n"
 "crowd huddled together in the driveway, speculating under their breaths about\n"
 "the odds of Harold coming out of that house ripe as head cheese left a week\n"
 "in the sun.\n"
 "   Ernie had just returned from the back yard with a hint of musk about him\n"
 "when five or six skunks scurried across in front of the garage and disappeared\n"
 "into the woods, followed about a minute later by Harold, his mysterious bag\n"
 "slung over his shoulder, and whistling a little tune. Harold tossed the bag\n"
 "into the trunk and punched up a number on his phone.\n"
 "   “Hi, Mom. Listen, breakfast at the Hudson's didn't go too smoothly this\n"
 "morning. Could they visit with you for a couple of hours until their kitchen\n"
 "airs out?”\n"
 "   . . . .\n"
 "   “Skunks.”\n"
 "   . . . .\n"
 "   “Some fox pelts I confiscated from a poacher a couple of weeks ago. Hadn't\n"
 "had a chance to turn them over to Fish and Game.”\n"
 "   . . .\n"
 "   “Skunks don't have the brains to know the difference, Mom. See you soon.”\n"
 "   The Hudson's all piled into the cruiser, glancing at the missing rear\n"
 "window, and then at Harold. The cruiser's tires crackled merrily as Harold\n"
 "made a U-turn and headed toward B Street.\n"
 "                   [ “Breakfast, Interrupted” - (c) 2009 ]\n"
} ;
static const short bba_uDatalBYTES = strlen ( bba_uDatal ) ;

//* Data invoked by btWlBB - wide character data for page-by-page scrolling *
// (Note extraneous Tab characters inserted for testing.  )
// (these s/b stripped AND return value properly adjusted )
static const wchar_t* bba_wDatal = 
{
  L"This revisionist take on the ever-popular Batman broke box office records i"
   "n the US, and won critical acclaim as one of the greatest superhero movies "
   "ever made. A sequel to 2006's BATMAN BEGINS, also directed by Christopler N"
   "olan, which restarted the Caped Crusader as a young, martial-arts-trained c"
   "rimefighter in a corrupt city\t, the plot deals with two of the iconic char"
   "acters of the Batman comic books; Two-Face and the Joker.\n"
   " \n"
   "CHINESE TRANSLATION\n"
   " \n"
   "The Joker, a deranged criminal with his face permanently set into a rictus "
   "grin, has always been Batman's most popular villain. Famously played by Jac"
   "k Nicholson as a suave crimelord in Tim Burton's version of Batman, Heath L"
   "edger\ntakes on the role here in a masterpiece performance. Ledger plays th"
   "e Joker as\na true psychopath\t\t\t\t, a man with no moral compunctions and"
   " a sick sense of humor which has the audience laughing and wincing at the s"
   "ame time. In sickly make-up, looking like a street vagrant, this screen-ste"
   "aling performance is sadly also his last\t, as he died of an accidental sle"
   "eping-pill overdose earlier this\nyear.\n"
   " \n"
   "CHINESE TRANSLATION\n"
   " \n"
   "Starring alongside Ledger are Christian Bale as Batman, a solid role played"
   " with his normal physical intensity, and Aaron Eckhart as Harvey Dent, an i"
   "dealistic lawyer transformed into the divided Two-Face by the Joker's machi"
   "nations. A strong supporting cast, including Maggie Gyllenhaal, Gary Oldman"
   ", and the ever-wise Morgan Freeman, fills out the movie, but Ledger's Joker"
   " dominates both the action and the acting. Batman's trademark gadgets, incl"
   "uding a souped-up Bat-Car, get plenty of use, and the psychological tension"
   " is matched by outstanding action scenes. Thoroughly recommended for both t"
   "hose new to Batman and to long-standing fans, but also extremely dark\n"
   "in tone - these heroes are definitely not for children.\n"
   " \n"
   "CHINESE TRANSLATION\n"
   "                   [Copyright \"Billingual Time\" Nov. 2008]\n"
} ;
static const short bba_wDatalCHARS = wcslen ( bba_wDatal ) ;

//* 'Next' pushbutton text *
static const char* NextText[] = 
{
   "  Next Line of Story  ",     // UTF-8-character story
   "  Next Page of Story  ",     // wide-character story
} ;
static const char* defaultTbText = "(some additional text)" ;

static dspinData lspData( 0, 13, 3, dspinINTEGER ) ;



static InitCtrl ic[btControlsDEFINED] =   // control initialization structures
{
   {  //* 'DONE' pushbutton - - - - - - - - - - - - - - - - - - -     btDonePB *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(dialogROWS - 5),        // ulY:       upper left corner in Y
      short(dialogCOLS / 2 - 12),   // ulX:       upper left corner in X
      1,                            // lines:     control lines
      8,                            // cols:      control columns
      "  DONE  ",                   // dispText:  
      nc.bw,                        // nColor:    non-focus color
      nc.bw,                        // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      NULL,                         // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[btBilaBB],                // nextCtrl:  link in next structure
   },
   {  //* 'Billboard A'    - - - - - - - - - - - - - - - - - - - -    btBilaBB *
      dctBILLBOARD,                 // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      2,                            // ulY:       upper left corner in Y
      1,                            // ulX:       upper left corner in X
      14,                           // lines:     control lines
      short(dialogCOLS - 2),        // cols:      control columns
      bba_uDatas,                   // dispText:  initial display text
      nc.bw,                        // nColor:    non-focus color
      nc.bw,                        // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "Billboard A",                // label:     control label
      -1,                           // labY:      label offset Y
      ZERO,                         // labX       label offset X
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      bba_cData,                    // scrColor:  if specified, attribute array
      NULL,                         // spinData:  (n/a)
      false,                        // active:    view-only
      &ic[btBilbBB],                // nextCtrl:  link in next structure
   },
   {  //* 'Billboard B'    - - - - - - - - - - - - - - - - - - - -    btBilbBB *
      dctBILLBOARD,                 // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      18,                           // ulY:       upper left corner in Y
      2,                            // ulX:       upper left corner in X
      10,                           // lines:     control lines
      Bill2COLS,                    // cols:      control columns
      "          Breaking News           "
      #if ENABLE_BT_MULTITHREAD == 0
      "    (multithreading disabled)     "
      #endif   // ENABLE_BT_MULTITHREAD
                                  , // dispText:  initial display text
      nc.bw,                        // nColor:    non-focus color
      nc.bw,                        // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "Billboard B",                // label:     control label
      -1,                           // labY:      label offset Y
      ZERO,                         // labX       label offset X
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  if specified, attribute array
      NULL,                         // spinData:  (n/a)
      false,                        // active:    view-only
      &ic[btUsRB],                  // nextCtrl:  link in next structure
   },
   {  //* UTF-8 data (short) Radio Button   - - - - - - - - - - - -     btUsRB *
      dctRADIOBUTTON,               // type:      
      rbtS3s,                       // rbSubtype: standard, 3 chars wide
      true,                         // rbSelect:  default selection
      short(ic[btBilbBB].ulY - 1),  // ulY:       upper left corner in Y
      short(dialogCOLS / 2),        // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a)
      nc.bw,                        // nColor:    non-focus color
      nc.bw,                        // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "UTF-8 character data (short)", // label:
      ZERO,                         // labY:      
      5,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[btUlRB],                  // nextCtrl:  link in next structure
   },
   {  //* UTF-8 data (long) Radio Button    - - - - - - - - - - - -     btUlRB *
      dctRADIOBUTTON,               // type:      
      rbtS3s,                       // rbSubtype: standard, 3 chars wide
      false,                        // rbSelect:  default selection
      short(ic[btUsRB].ulY + 1),    // ulY:       upper left corner in Y
      ic[btUsRB].ulX,               // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a)
      nc.bw,                        // nColor:    non-focus color
      nc.bw,                        // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "UTF-8 character data (long)",// label:
      ZERO,                         // labY:      
      5,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[btWsRB],                  // nextCtrl:  link in next structure
   },
   {  //* Wide data (short) Radio Button      - - - - - - - - - - -     btWsRB *
      dctRADIOBUTTON,               // type:      
      rbtS3s,                       // rbSubtype: standard, 3 chars wide
      false,                        // rbSelect:  default selection
      short(ic[btUlRB].ulY + 1),    // ulY:       upper left corner in Y
      ic[btUlRB].ulX,               // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a)
      nc.bw,                        // nColor:    non-focus color
      nc.bw,                        // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "Wide-character data (short)",// label:
      ZERO,                         // labY:      
      5,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[btWlRB],                  // nextCtrl:  link in next structure
   },
   {  //* Wide data (long) Radio Button       - - - - - - - - - - -     btWlRB *
      dctRADIOBUTTON,               // type:      
      rbtS3s,                       // rbSubtype: standard, 3 chars wide
      false,                        // rbSelect:  default selection
      short(ic[btWsRB].ulY + 1),    // ulY:       upper left corner in Y
      ic[btWsRB].ulX,               // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a)
      nc.bw,                        // nColor:    non-focus color
      nc.bw,                        // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "Wide-character data (long)", // label:
      ZERO,                         // labY:      
      5,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[btClrRB],                 // nextCtrl:  link in next structure
   },
   {  //* 'Clear' Radio Button      - - - - - - - - - - - - - - - -    btClrRB *
      dctRADIOBUTTON,               // type:      
      rbtS3s,                       // rbSubtype: standard, 3 chars wide
      false,                        // rbSelect:  default selection
      short(ic[btWlRB].ulY + 1),    // ulY:       upper left corner in Y
      ic[btWlRB].ulX,               // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a)
      nc.bw,                        // nColor:    non-focus color
      nc.bw,                        // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "Clear Billboard A",          // label:
      ZERO,                         // labY:      
      5,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[btMultiRB],               // nextCtrl:  link in next structure
   },
   {  //* Multi-color Radio Button    - - - - - - - - - - - - - -    btMultiRB *
      dctRADIOBUTTON,               // type:      
      rbtS3a,                       // rbSubtype: standard, 3 chars wide
      true,                         // rbSelect:  default selection
      short(ic[btClrRB].ulY + 2),   // ulY:       upper left corner in Y
      ic[btClrRB].ulX,              // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a)
      nc.bw,                        // nColor:    non-focus color
      nc.bw,                        // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "Multi-color output",         // label:
      ZERO,                         // labY:      
      5,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[btNextPB],                // nextCtrl:  link in next structure
   },
   {  //* 'NEXT' pushbutton - - - - - - - - - - - - - - - - - - -     btNextPB *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(ic[btMultiRB].ulY + 1), // ulY:       upper left corner in Y
      ic[btMultiRB].ulX,            // ulX:       upper left corner in X
      1,                            // lines:     control lines
      22,                           // cols:      control columns
      NextText[0],                  // dispText:  
      nc.bw,                        // nColor:    non-focus color
      nc.bw,                        // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      NULL,                         // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      false,                        // active:    allow control to gain focus
      &ic[btCustTB],                // nextCtrl:  link in next structure
   },
   {  //* 'Custom Text' textbox - - - - - - - - - - - - - - - - - -   btCustTB *
      dctTEXTBOX,                   // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(ic[btNextPB].ulY + 2),  // ulY:       upper left corner in Y
      short(ic[btNextPB].ulX + 9),  // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      30,                           // cols:      control columns
      defaultTbText,                // dispText:  (value of t3indexO3)
      nc.bw,                        // nColor:    non-focus color
      nc.bw,                        // fColor:    focus color
      tbPrint,                      // filter:    any printing character
      "Custom Text",                // label:     
      ZERO,                         // labY:      
      -12,                          // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    cannot gain focus
      &ic[btCustRB],                // nextCtrl:  link in next structure
   },
   {  //* 'Custom Text' radio button  - - - - - - - - - - - - - - -   btCustRB *
      dctRADIOBUTTON,               // type:      
      rbtS3a,                       // rbSubtype: standard, 3 chars wide
      true,                         // rbSelect:  default selection
      short(ic[btCustTB].ulY + 1),  // ulY:       upper left corner in Y
      ic[btCustTB].ulX,             // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a)
      nc.bw,                        // nColor:    non-focus color
      nc.bw,                        // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "Insert as 'wide' text",      // label:
      ZERO,                         // labY:      
      5,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[btAppPB],                 // nextCtrl:  link in next structure
   },
   {  //* 'APPEND' pushbutton - - - - - - - - - - - - - - - - - -      btAppPB *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(ic[btCustRB].ulY + 1),  // ulY:       upper left corner in Y
      short(ic[btCustRB].ulX - 11), // ulX:       upper left corner in X
      1,                            // lines:     control lines
      22,                           // cols:      control columns
     "  Append Custom Text  ",      // dispText:  
      nc.bw,                        // nColor:    non-focus color
      nc.bw,                        // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      NULL,                         // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[btPosRB],                 // nextCtrl:  link in next structure
   },
   {  //* 'Append Position' radio button  - - - - - - - - - - - - -    btPosRB *
      dctRADIOBUTTON,               // type:      
      rbtS3a,                       // rbSubtype: standard, 3 chars wide
      true,                         // rbSelect:  default selection
      ic[btAppPB].ulY,              // ulY:       upper left corner in Y
      short(ic[btAppPB].ulX + ic[btAppPB].cols + 2),// ulX:
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a)
      nc.bw,                        // nColor:    non-focus color
      nc.bw,                        // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "At End",                     // label:
      ZERO,                         // labY:      
      5,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[btRepPB],                 // nextCtrl:  link in next structure
   },
   {  //* 'REPLACE' pushbutton  - - - - - - - - - - - - - - - - -      btRepPB *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(ic[btAppPB].ulY + 1),   // ulY:       upper left corner in Y
      ic[btAppPB].ulX,              // ulX:       upper left corner in X
      1,                            // lines:     control lines
      22,                           // cols:      control columns
      " Replace Indexed Line ",     // dispText:  
      nc.bw,                        // nColor:    non-focus color
      nc.bw,                        // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      NULL,                         // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[btLineSP],                // nextCtrl:  link in next structure
   },
   {  //* 'LINE INDEX' spinner  - - - - - - - - - - - - - - - - -     btLineSP *
      dctSPINNER,                   // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      ic[btRepPB].ulY,              // ulY:       upper left corner in Y
      short(ic[btRepPB].ulX + ic[btRepPB].cols + 2), // ulX:
      1,                            // lines:     (n/a)
      4,                            // cols:      control columns
      NULL,                         // dispText:  (n/a)
      nc.bw,                        // nColor:    non-focus color
      nc.bw,                        // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "Line Index",                 // label:     
      ZERO,                         // labY:      
      5,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      &lspData,                     // spinData:  spinner init
      true,                         // active:    allow control to gain focus
      &ic[btExtPB],                 // nextCtrl:  link in next structure
   },
   {  //* 'EXTEND' pushbutton - - - - - - - - - - - - - - - - - -      btExtPB *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(ic[btRepPB].ulY + 1),   // ulY:       upper left corner in Y
      ic[btRepPB].ulX,              // ulX:       upper left corner in X
      1,                            // lines:     control lines
      22,                           // cols:      control columns
      " Extend Indexed Line  ",     // dispText:  
      nc.bw,                        // nColor:    non-focus color
      nc.bw,                        // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      NULL,                         // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[btSwpPB],                 // nextCtrl:  link in next structure
   },
   {  //* 'SWAP' pushbutton   - - - - - - - - - - - - - - - - - -      btSwpPB *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      ic[btExtPB].ulY,              // ulY:       upper left corner in Y
      short(ic[btExtPB].ulX + ic[btExtPB].cols + 2), // ulX:
      1,                            // lines:     control lines
      14,                           // cols:      control columns
      " Swap 3 and 5 ",             // dispText:  
      nc.bw,                        // nColor:    non-focus color
      nc.bw,                        // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      NULL,                         // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      NULL,                         // nextCtrl:  link in next structure
   },
} ;

//* Module-scope access to the dialog for the non-member callback method *
static NcDialog* btcuPtr = NULL ;
static short btControlUpdate ( const short currIndex, 
                               const wkeyCode wkey, bool firstTime ) ;


//*************************
//*    ~BillboardTest     *
//*************************
//******************************************************************************
//* Destructor.                                                                *
//*                                                                            *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

BillboardTest::~BillboardTest ( void )
{

   if ( this->dp != NULL )          // close the window
      delete ( this->dp ) ;
   btcuPtr = NULL ;                 // detach callback method's access

}  //* End ~BillboardTest() 

//*************************
//*     BillboardTest     *
//*************************
//******************************************************************************
//* Constructor.                                                               *
//*                                                                            *
//* Input  : tLines     : number of display line in terminal window            *
//*          tCols      : number of display columns in terminal window         *
//*          minY       : first available display line                         *
//*                                                                            *
//* Returns: implicitly return class reference                                 *
//******************************************************************************

BillboardTest::BillboardTest ( short tLines, short tCols, short minY )
{
   //* Initialize data members *
   this->termRows = tLines ;
   this->termCols = tCols ;
   this->minULY   = minY ;
   this->dp       = NULL ;
   this->dColor   = nc.gybl | ncbATTR ;
   this->bColor   = nc.brbl ;
   this->btOpen   = false ;

   //* Initialize remainder of control-definition array.           *
   //* (colors become available after NCurses engine instantiated) *
   attr_t pnColor = nc.gyR,            // pushbutton non-focus color
          pfColor = nc.gyre | ncbATTR, // pushbutton focus color
          rnColor = dColor,            // radio button non-focus color
          rfColor = pfColor,           // radio button focus color
          bnColor = nc.gybr | ncbATTR, // Billboard A non-focus color
          bfColor = nc.gyma | ncbATTR, // Billboard A focus color
          snColor = nc.bw,             // spinner non-focus color
          sfColor = pfColor,           // spinner focus color
          siColor = nc.gygr ;          // spinner indicator color
   ic[btDonePB].nColor   = pnColor ;
   ic[btDonePB].fColor   = pfColor ;
   ic[btBilaBB].nColor   = bnColor ;
   ic[btBilaBB].fColor   = bfColor ;
   ic[btBilbBB].nColor = ic[btBilbBB].fColor = nc.rebk ;
   ic[btUsRB].nColor = ic[btUlRB].nColor  = ic[btWsRB].nColor = 
   ic[btWlRB].nColor = ic[btClrRB].nColor = ic[btMultiRB].nColor = 
   ic[btCustRB].nColor = ic[btPosRB].nColor = rnColor ;
   ic[btUsRB].fColor = ic[btUlRB].fColor  = ic[btWsRB].fColor = 
   ic[btWlRB].fColor = ic[btClrRB].fColor = ic[btMultiRB].fColor = 
   ic[btCustRB].fColor = ic[btPosRB].fColor = rfColor ;
   ic[btNextPB].nColor   = pnColor ;
   ic[btNextPB].fColor   = nc.gygr | ncbATTR ;
   ic[btCustTB].nColor   = nc.bw ;
   ic[btCustTB].fColor   = nc.grB ;
   ic[btAppPB].nColor    = nc.brgy | ncbATTR ;
   ic[btAppPB].fColor    = pfColor ;
   ic[btRepPB].nColor    = nc.cygy ;
   ic[btRepPB].fColor    = pfColor ;
   ic[btLineSP].nColor   = snColor ;
   ic[btLineSP].fColor   = sfColor ;
   ic[btExtPB].nColor    = nc.brgy | ncbATTR ;
   ic[btExtPB].fColor    = pfColor ;
   ic[btSwpPB].nColor    = pnColor ;
   ic[btSwpPB].fColor    = nc.gyma | ncbATTR ;
   bba_cData[ 0] = nc.gybr | ncbATTR ;
   bba_cData[ 1] = nc.bkbr ;
   bba_cData[ 2] = nc.rebr ;
   bba_cData[ 3] = nc.grbr ;
   bba_cData[ 4] = nc.blbr ;
   bba_cData[ 5] = nc.mabr ;
   bba_cData[ 6] = nc.cybr ;
   bba_cData[ 7] = nc.gybr | ncbATTR ; 
   bba_cData[ 8] = nc.bkbr | ncbATTR ;
   bba_cData[ 9] = nc.rebr | ncbATTR ;
   bba_cData[10] = nc.grbr | ncbATTR ;
   bba_cData[11] = nc.blbr | ncbATTR ;
   bba_cData[12] = nc.mabr | ncbATTR ;
   bba_cData[13] = nc.cybr | ncbATTR ;

   //* Initialize the dctSPINNER control color *
   lspData.indiAttr = siColor ;

   if ( (this->btOpen = this->btOpenDialog ()) != false )
   {

   }  // OpenWindow()

}  //* End BillboardTest() 

//*************************
//*     btOpenDialog      *
//*************************
//******************************************************************************
//* Open the dialog window.                                                    *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: 'true' if dialog window opened successfully, else 'false'         *
//******************************************************************************
//* Note that the compiler will not allow us to use a member method as the     *
//* callback target (unless we use the -fpermissive flag), so we must give the *
//* non-member method access to our dialog. Although this is a dangerous       *
//* thing, our target method promises to behave itself :-)                     *
//******************************************************************************

bool BillboardTest::btOpenDialog ( void )
{
   short ctrY    = this->termRows/2,         // dialog center in Y
         ctrX    = this->termCols/2,         // dialog center in X
         //* Upper left corner in Y (cannot obscure status window) *
         ulY     = (ctrY - dialogROWS/2) >= this->minULY ? 
                   (ctrY - dialogROWS/2) : this->minULY,
         ulX     = ctrX - dialogCOLS / 2 ;   // upper left corner in X
   bool  success = false ;

   //* Initial parameters for dialog window *
   InitNcDialog dInit( dialogROWS,     // number of display lines
                       dialogCOLS,     // number of display columns
                       ulY,            // Y offset from upper-left of terminal 
                       ulX,            // X offset from upper-left of terminal 
                       "  DialogBillboard-class Controls  ", // dialog title
                       ncltDUAL,       // border line-style
                       this->bColor,   // border color attribute
                       this->dColor,   // interior color attribute
                       ic              // pointer to list of control definitions
                     ) ;

   //* Instantiate the dialog window *
   this->dp = new NcDialog ( dInit ) ;

   //* Open the dialog window *
   if ( (this->dp->OpenWindow()) == OK )
   {
      LineDef  lDef(ncltHORIZ, ncltSINGLE, (dialogROWS - 4), 
                    ZERO, dialogCOLS, this->bColor) ;
      this->dp->DrawLine ( lDef ) ; // divide window horizontally

      //* Set radio button group for text selecton. *
      //* (only one can be selected at any moment). *
      short XorGroup[] = { btUsRB, btUlRB, btWsRB, btWlRB, btClrRB, -1 } ;
      this->dp->GroupRadiobuttons ( XorGroup ) ;

      this->dp->WriteString ( (ic[btDonePB].ulY - 2), 5, 
                              "(ALT+P to pause Billboard-B)", 
                              this->dColor & ~ncbATTR ) ;
      this->dp->RefreshWin () ;     // make everything visible

      //* Establish a callback method for display of context help.        *
      btcuPtr = this->dp ;
      this->dp->EstablishCallback ( &btControlUpdate ) ;

      success = true ;
   }
   return success ;

}  //* End btOpenDialog() *

//*************************
//*    btDialogOpened     *
//*************************
//******************************************************************************
//* Satisfy caller's curiosity.                                                *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: 'true' if dialog opened successfully, else 'false'                *
//******************************************************************************

bool BillboardTest::btDialogOpened ( void )
{
   return this->btOpen ;

}  //* End btDialogOpened() *

//*************************
//*      btInteract       *
//*************************
//******************************************************************************
//* User interactive experience.                                               *
//* (Not quite "California Adventure"(tm), but still cool.)                    *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void BillboardTest::btInteract ( void )
{

   if ( this->btOpen )
   {
      //* Track user's selections *
      uiInfo Info ;                 // user interface data returned here
      short  icIndex = ZERO,        // index of control with input focus
             storyNumbr = ZERO,     // for advancing the story display
             storyIndex = ZERO ;
      bool   done = false ;         // loop control

      #if ENABLE_BT_MULTITHREAD != 0
      //* Note the odd syntax required to declare a member method *
      //* as the target of a new thread.                          *
      //* NOTE: Because the ncurses library is not thread safe,   *
      //*       screen artifacts may sometimes appear. We are     *
      //*       working on this problem from a higher level, and  *
      //*       hope to have it fixed soon....                    *
      thread BNews( &BillboardTest::btBreakingNews, this, &done ) ;
      #endif   // ENABLE_BT_MULTITHREAD

      while ( ! done )
      {
         //*******************************************
         //* If focus is currently on a Pushbutton   *
         //*******************************************
         if ( ic[icIndex].type == dctPUSHBUTTON )
         {
            icIndex = this->dp->EditPushbutton ( Info ) ;
            if ( Info.dataMod != false )
            {
               if ( Info.ctrlIndex == btDonePB )
               {
                  done = true ;
                  #if ENABLE_BT_MULTITHREAD != 0
                  BNews.join() ;    // wait for secondary thread to return
                  #endif   // ENABLE_BT_MULTITHREAD
               }
               else if ( Info.ctrlIndex == btNextPB )
               {
                  if ( storyNumbr == ZERO )  // UTF-8-character story
                  {
                     if ( storyIndex < bba_uDatalBYTES )
                     {
                        storyIndex += 
                           this->dp->Append2Billboard ( btBilaBB, 
                                                  &bba_uDatal[storyIndex] ) ;
                        icIndex = this->dp->PrevControl () ;
                     }
                     else
                     {
                        icIndex = this->dp->PrevControl () ;
                        this->dp->ControlActive ( btNextPB, false ) ;
                     }
                  }
                  else                       // wide-character story
                  {
                     if ( storyIndex < bba_wDatalCHARS )
                     {
                        storyIndex += 
                           this->dp->SetBillboard ( btBilaBB, 
                                                    &bba_wDatal[storyIndex] ) ;
                        icIndex = this->dp->PrevControl () ;
                     }
                     else
                     {
                        icIndex = this->dp->PrevControl () ;
                        this->dp->ControlActive ( btNextPB, false ) ;
                     }
                  }
               }
               else if ( Info.ctrlIndex == btAppPB )
               {  //* Append a line of text *
                  gString gs ;
                  this->dp->GetTextboxText ( btCustTB, gs ) ;
                  bool multiColor, wideData, atEnd ;
                  this->dp->GetRadiobuttonState ( btMultiRB, multiColor ) ;
                  this->dp->GetRadiobuttonState ( btCustRB, wideData ) ;
                  this->dp->GetRadiobuttonState ( btPosRB, atEnd ) ;
                  attr_t lineColor = multiColor ? dtbmFcolor : dtbmNFcolor ;
                  if ( atEnd )
                  {
                     if ( wideData != false )
                        this->dp->Append2Billboard ( btBilaBB, gs.gstr(), lineColor ) ;
                     else
                        this->dp->Append2Billboard ( btBilaBB, gs.ustr(), lineColor ) ;
                  }
                  else
                  {
                     if ( wideData != false )
                        this->dp->Insert2Billboard ( btBilaBB, gs.gstr(), lineColor ) ;
                     else
                        this->dp->Insert2Billboard ( btBilaBB, gs.ustr(), lineColor ) ;
                  }
               }
               else if ( Info.ctrlIndex == btRepPB )
               {  //* Replace specified line of text
                  gString gs ;
                  this->dp->GetTextboxText ( btCustTB, gs ) ;
                  int l ;
                  this->dp->GetSpinnerValue ( btLineSP, l ) ;
                  short lineNum = short(l) ;
                  bool multiColor, wideData ;
                  this->dp->GetRadiobuttonState ( btMultiRB, multiColor ) ;
                  this->dp->GetRadiobuttonState ( btCustRB, wideData ) ;
                  attr_t lineColor = multiColor ? dtbmFcolor : attrDFLT ;
                  if ( wideData != false )
                     this->dp->Append2Billboard ( btBilaBB, gs.gstr(), 
                                                  lineColor, lineNum ) ;
                  else
                     this->dp->Append2Billboard ( btBilaBB, gs.ustr(), 
                                                  lineColor, lineNum ) ;
               }
               else if ( Info.ctrlIndex == btExtPB )
               {  //* Extend the specified line of text *
                  gString gs ;
                  this->dp->GetTextboxText ( btCustTB, gs ) ;
                  int l ;
                  this->dp->GetSpinnerValue ( btLineSP, l ) ;
                  short lineNum = short(l) ;
                  bool multiColor, wideData ;
                  this->dp->GetRadiobuttonState ( btMultiRB, multiColor ) ;
                  this->dp->GetRadiobuttonState ( btCustRB, wideData ) ;
                  attr_t lineColor = multiColor ? dtbmFcolor : attrDFLT ;
                  if ( wideData != false )
                     this->dp->Append2Billboard ( btBilaBB, gs.gstr(), 
                                                  lineColor, lineNum, true ) ;
                  else
                     this->dp->Append2Billboard ( btBilaBB, gs.ustr(), 
                                                  lineColor, lineNum, true ) ;
               }
               else if ( Info.ctrlIndex == btSwpPB )
               {
                  short lA = 3, lB = 5, lineCount ;
                  multiText srcText[ic[btBilaBB].lines] ;
                  this->dp->GetBillboard ( btBilaBB, srcText ) ;
                  const attr_t* srcAttr = 
                     this->dp->GetBillboardColors ( btBilaBB, lineCount ) ;
                  attr_t lbAttr = srcAttr[lB] ;
                  this->dp->Append2Billboard ( btBilaBB, srcText[lA].ln,
                                               srcAttr[lA], lB ) ;
                  this->dp->Append2Billboard ( btBilaBB, srcText[lB].ln,
                                               lbAttr, lA ) ;
               }
            }
         }
         //*******************************************
         //* If focus is currently on a Radio button *
         //*******************************************
         else if ( ic[icIndex].type == dctRADIOBUTTON )
         {
            icIndex = this->dp->EditRadiobutton ( Info ) ;

            //* If 'selected' button is a member of a button group *
            if ( Info.selMember < MAX_DIALOG_CONTROLS )
            {
               //* If a new 'selected' member of the group *
               //* OR previous selection was RE-selected   *
               if ( Info.dataMod != false || Info.keyIn == nckENTER )
               {
                  bool multiColor ;
                  this->dp->GetRadiobuttonState ( btMultiRB, multiColor ) ;
                  this->dp->ClearBillboard ( btBilaBB, true, 
                        (const attr_t*)(multiColor ? bba_cData : NULL) ) ;
                  this->dp->ControlActive ( btNextPB, false ) ;
                  switch ( Info.selMember )
                  {
                     case btUsRB:      // UTF-8 (short)
                        this->dp->SetBillboard ( btBilaBB, bba_uDatas,
                              (const attr_t*)(multiColor ? bba_cData : NULL) ) ;
                        break ;
                     case btUlRB:      // UTF-8 (long)
                        storyIndex = this->dp->SetBillboard ( btBilaBB, bba_uDatal,
                              (const attr_t*)(multiColor ? bba_cData : NULL) ) ;
                        storyNumbr = 0 ;
                        this->dp->SetPushbuttonText ( btNextPB, NextText[storyNumbr] ) ;
                        this->dp->ControlActive ( btNextPB, true ) ;
                        icIndex = this->dp->NextControl () ;
                        break ;
                     case btWsRB:      // Wide (short)
                        this->dp->SetBillboard ( btBilaBB, bba_wDatas,
                              (const attr_t*)(multiColor ? bba_cData : NULL) ) ;
                        break ;
                     case btWlRB:      // Wide (long)
                        storyIndex = this->dp->SetBillboard ( btBilaBB, bba_wDatal,
                                 (const attr_t*)(multiColor ? bba_cData : NULL) ) ;
                        storyNumbr = 1 ;
                        this->dp->SetPushbuttonText ( btNextPB, NextText[storyNumbr] ) ;
                        this->dp->ControlActive ( btNextPB, true ) ;
                        icIndex = this->dp->NextControl () ;
                        break ;
                     case btClrRB:     // Clear Billboard A
                        // We just did that above
                        break ;
                     default:
                        break ;
                  }
               }
            }
            //* Else, button is an independent button *
            //* (Info.selMember==MAX_DIALOG_CONTROLS) *
            else
            {
               if ( Info.dataMod != false )
               {
                  if ( Info.ctrlIndex == btMultiRB )
                  {
                     //* Move the focus to the place user *
                     //* is most likely to want it.       *
                     while ( (icIndex = this->dp->NextControl ()) != btDonePB ) ;
                  }
                  else if ( Info.ctrlIndex == btCustRB )
                     ;  // nothing to do
                  else if ( Info.ctrlIndex == btPosRB )
                     ;  // nothing to do
               }
            }
         }
         //**********************************************
         //* If focus is currently on a TextBox control *
         //**********************************************
         else if ( ic[icIndex].type == dctTEXTBOX )
         {
            icIndex = this->dp->EditTextbox ( Info ) ;
         }
         //**********************************************
         //* If focus is currently on a Spinner control *
         //**********************************************
         else if ( ic[icIndex].type == dctSPINNER )
         {
            icIndex = this->dp->EditSpinner ( Info ) ;
         }
         //************************************************
         //* If focus is currently on a Billboard control *
         //************************************************
         else if ( ic[icIndex].type == dctBILLBOARD )
         {
            //* Nothing to do because the Billboard controls   *
            //* are not directly editable AND are set inactive.*
            icIndex = this->dp->EditBillboard ( Info ) ;
         }
         else
         { /* no other control types defined for this method */ }

         //* Move input focus to next/previous control.*
         if ( done == false && Info.viaHotkey == false )
         {
            if ( Info.keyIn == nckSTAB )
               icIndex = this->dp->PrevControl () ; 
            else
               icIndex = this->dp->NextControl () ;
            // Programmer's Note: The context help message is displayed by the 
            // callback method; however, the message gets out-of-synch because 
            // of the timing of the call from inside the edit routines. For this 
            // reason, we manually call the callback method when the focus changes.                                *
            btControlUpdate ( icIndex, Info.wk, false ) ;
         }
      }  // while()
   }
   else
      { /* Caller is an idiot */ }

}  //* End btInteract() *

//*************************
//*    btBreakingNews     *
//*************************
//******************************************************************************
//* Launched by btInteract().                                                  *
//* A secondary thread starts here and scrolls through the Breaking News       *
//* stories until main thread signals that we should come home                 *
//*                                                                            *
//* Note that in the real world, the semaphore 'quittingTime', should have an  *
//* access lock, but this is a test program, and timing is not critical.       *
//*                                                                            *
//* Input  : quittingTime : (pointer to boolean)                               *
//*                         when flag changes to 'true', then rejoin the       *
//*                         main thread                                        *
//* Returns: nothing                                                           *
//******************************************************************************

#if ENABLE_BT_MULTITHREAD != 0
static bool btBN_Pause = false ;
void BillboardTest::btBreakingNews ( bool* quittingTime )
{
const short MAX_nIndex = 39 ;
const char* bbb_uData[MAX_nIndex] = 
{ //----+----|----+----|----+----|---|
   "          Breaking News           ",
   "Muggles are vastly better educated",
   "than witches and wizards, fortu-  ",
   "nately, magic compensates nicely. ",
   "               ---                ",
   "The average college graduate knows",
   "everything except how ignorant he ",
   "or she really is. (but we learn)  ",
   "               ---                ",
   "\"Internationalization of user in- ",
   "terfaces is not for the timid, nor",
   "for the impatient.\" -Software Sam ",
   "               ---                ",
   "Sheldon Cooper is a smelly pooper.",
   "               ---                ",
   "\"Stay focused: never let yourself ",
   "become famous enough that people  ",
   "start wanting a piece of you.\"    ",
   "               ---                ",
   "如果你不生气, 那么你需要关注你的大",
   "脑！  \"If you aren't angry, then ",
   "you just aren't paying attention!\"",
   "               ---                ",
   "只有傻瓜才有梦想，只有傻瓜才能实现",
   "梦想。 其他所有的人都会像阳光下的",
   "剩菜一样腐烂掉。 \"Only fools have ",
   "dreams, and only fools achieve    ",
   "them. Everyone else just rots like",
   "old vegetables left too long in   ",
   "the sun\"                         ",
   " ***  ***  ***  ***  ***  *** *** ",
   "                ..                ",
   "                ..                ",
   "                ..                ",
   "                ..                ",
   "                ..                ",
   "                ..                ",
   "                ..                ",
   "                ..                ",
} ;

   short nIndex = 1 ;
   do
   {
      if ( ! btBN_Pause )
      {
         this->dp->Append2Billboard ( btBilbBB, bbb_uData[nIndex++] ) ;
         if ( nIndex >= MAX_nIndex )
            nIndex = ZERO ;
      }
      chrono::duration<short, std::milli>aMoment( 950 ) ;
      this_thread::sleep_for( aMoment ) ;
   }
   while ( *quittingTime == false ) ;

}  //* End btBreakingNews() *
#endif   // ENABLE_BT_MULTITHREAD

//* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -  *
//* - - - - - - - - - - - - Non-member Methods- - - - - - - - - - - - - - - -  *
//* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -  *

//*************************
//*    btControlUpdate    *
//*************************
//******************************************************************************
//* This is a callback method for manually updating the dialog controls.       *
//*                                                                            *
//*  For this test, the following are updated by the callback method:          *
//*   1. Context help messages for each control                                *
//*      (but see note in input loop)                                          *
//*   2. Pause/Un-pause the auto-scroll thread.                                *
//*   3. Capture of dialog's text/attribute data to a file.                    *
//*                                                                            *
//* Input  : currIndex: index of control that currently has focus              *
//*          wkey     : user's key input data                                  *
//*          firstTime: the EstablishCallback() method calls this method once  *
//*                     with firstTime==true, to perform any required          *
//*                     initialization. Subsequently, the NcDialog class       *
//*                     always calls with firstTime==false.                    *
//* Returns: OK                                                                *
//******************************************************************************
//* Notes:                                                                     *
//******************************************************************************

static short btControlUpdate ( const short currIndex, 
                               const wkeyCode wkey, bool firstTime )
{
static const char* ContextHelp[btControlsDEFINED] = 
{
 " Exit Billboard-test dialog.",
 " Billboard A",   // inactive control
 " Billboard B",   // inactive control
 " If 'selected', display a UTF-8 encoded data set that fits\n"
 " in the Billboard-A display area.",
 " If 'selected', display a UTF-8 encoded data set that is larger\n"
 " than the Billboard-A display area.",
 " If 'selected', display a 'wide' (wchar_t) data set that fits\n"
 " in the Billboard-A display area.",
 " If 'selected', display a 'wide' (wchar_t) data set that is larger\n"
 " than the Billboard-A display area.",
 " If 'selected', erase all data in Billboard-A display area.\n"
 " (color attributes are unchanged)",
 " If 'selected', use a different color attribute for each display line\n"
 " of Billboard-A. Otherwise, used the default foreground/background color.",
 " For data sets larger than Billboard-A, display the next line or page of data.",
 " Enter desired text here to be used for append/replace/extend operations in\n"
 " Billboard-A.",
 " If 'selected', then append/replace/extend operations on Billboard-A\n"
 " will use 'wide' data, else use UTF-8 data.",
 " Append a line of text containing your custom data to the currently-displayed\n"
 " data in Billboard-A. (color determined by state of 'multi-color' button)",
 " For 'append' operations, specifies insertion point: if 'selected', then\n"
 " append to end of data. else insert as first line of data.",
 " Replace an existing line of text in Billboard-A with your custom text.\n"
 " (color determined by state of 'multi-color' button)",
 " Billboard-A target-display-line INDEX for replace/extend operations.",
 " Extend an existing line of text in Billboard-A using your custom text.\n"
 " (color determined by state of 'multi-color' button)",
 " Exchange the text and color data currently at line indices 3 and 5.\n"
 " Exercises the 'GetBillboard' methods.",
} ;

   static short dRows, dCols ;
   if ( firstTime )
   {
      btcuPtr->GetDialogDimensions ( dRows, dCols ) ;
   }
   #if ENABLE_BT_MULTITHREAD != 0
   else if ( wkey.type == wktEXTEND && wkey.key == 'p' )
   { // Convenient debugging signal: stop/start the auto-scroll thread
      btBN_Pause = btBN_Pause ? false : true ;
   }
   #endif   // ENABLE_BT_MULTITHREAD

   //* Display context help *
   attr_t hColor = nc.gybl | ncbATTR ;    // context-help color
   winPos helpPos( (dRows - 3), 1 ) ;     // context-help position
   btcuPtr->ClearLine ( helpPos.ypos, hColor ) ;
   btcuPtr->ClearLine ( helpPos.ypos + 1, hColor ) ;
   btcuPtr->WriteParagraph ( helpPos, ContextHelp[currIndex], hColor, true ) ;

   if ( (wkey.type == wktEXTEND) && 
        ((wkey.key == nckA_Z) || (wkey.key == nckAS_Z)) )
   {
      const char* const capText = "./capturedlg.txt" ;
      const char* const capHtml = "./capturedlg.html" ;

      //* Temporarily disable the auto-scroll thread, *
      //* so we don't capture during an update, and   *
      //* wait a moment for current update to finish. *
      bool btBN_Pause_Old = btBN_Pause ;
      btBN_Pause = false ;
      chrono::duration<short, std::milli>aMoment( 1500 ) ;
      this_thread::sleep_for( aMoment ) ;

      bool fmt = bool(wkey.key == nckAS_Z) ;
      short capStatus =
         btcuPtr->CaptureDialog ( (fmt ? capHtml : capText), fmt ) ;
      gString gs ;
      gs.compose( L"Screen Capture to: '%s'", 
                  (fmt ? capHtml : capText) ) ;
      if ( capStatus != OK )
         gs.append( L" FAILED!" ) ;
      btcuPtr->DebugMsg ( gs.ustr(), 3, true ) ;

      //* Re-enable the auto-scroll thread.*
      btBN_Pause = btBN_Pause_Old ;
   }

   return OK ;

}  //* End btControlUpdate() *

//*************************
//*                       *
//*************************
//******************************************************************************
//*                                                                            *
//*                                                                            *
//*                                                                            *
//* Input  :                                                                   *
//*                                                                            *
//* Returns:                                                                   *
//******************************************************************************


