//******************************************************************************
//* File       : StreamTest.cpp                                                *
//* Author     : Mahlon R. Smith                                               *
//*              Copyright (c) 2010-2025 Mahlon R. Smith, The Software Samurai *
//*                  GNU GPL copyright notice located in NCurses.hpp           *
//* Date       : 21-Mar-2025                                                   *
//* Version    : (see AppVersion string below)                                 *
//*                                                                            *
//* Description: This module contains code and data to exercise the            *
//*              NCurses-class output stream functionality. This is Test11().  *
//*              Stream output emulates stdout/wcout functionality by wrapping *
//*              the text output at the right edge of the terminal window.     *
//*                                                                            *
//******************************************************************************
//* Version History (most recent first):                                       *
//*                                                                            *
//* v: 0.00.01 28-Apr-2015 Test new NCurses-class functionality.               *
//******************************************************************************

// All definitions needed for your application
#include "GlobalDef.hpp"

// C++ access to files
#include <fstream>

//** Local Prototypes **
static short ReadLine ( ifstream& ifs, gString& gs ) ;



//*************************
//*        Test11         *
//*************************
//******************************************************************************
//* Test the NCurses-class stream output functionality.                        *
//*                                                                            *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************
//* Programmer's Note: The scan of our 'grep' output is rather brain-dead,     *
//* with many logical holes; however, for our controlled data, it performs     *
//* adequately.                                                                *
//*                                                                            *
//******************************************************************************

void Test11 ( void )
{
   const char*    cursorTemplate = 
                  "\n Final cursor position (see red 'X'): y:%02hd x:%02hd " ;
   const wchar_t* paK = L"  PRESS ANY KEY...  \n\n" ;
   const wchar_t* Popeye = L"Iyam what Iyam, and that's all what "
                            "Iyam! - Popeye the Sailor **  " ;
   short sCnt = 10 ;

   if( (nc.StartNCursesEngine ()) == OK )
   {
      const attr_t pColor[20] = 
      {
         nc.bw,  nc.re,  nc.gr,   nc.br,  nc.bl,  nc.ma,  nc.cy,  nc.gy,
         nc.bk,  nc.reB, nc.grR,  nc.brR, nc.blB, nc.maR, nc.cyB, nc.gyR,
         nc.reU, nc.grU, nc.macy, nc.brgr
      } ;
      nc.ClearScreen () ;                    // clear terminal window
      nc.OutputStreamScrolling ( true ) ;    // enable stream output
      winPos wp( ZERO, ZERO ) ;              // tracks cursor position
      nc.SetCursor ( wp ) ;                  // set cursor to window origin

      wp = nc.WriteStream ( L"  ** Test of NCurses::WriteStream() Method **  \n\n", nc.blR ) ;

      gString gsOut ;
      gsOut.compose( L" >>Outputting a string %02hd times (monochrome): \n", &sCnt ) ;
      wp = nc.WriteStream ( gsOut.gstr(), nc.brbl ) ;
      for ( short i = ZERO ; i < sCnt ; i++ )
         wp = nc.WriteStream ( Popeye, nc.br ) ;
      nc.WriteStream ( L"X", nc.reR ) ;
      gString gs( cursorTemplate, &wp.ypos, &wp.xpos ) ;
      wp = nc.WriteStream ( gs.gstr(), nc.brbl ) ;
      wp = nc.WriteStream ( paK, nc.rebk ) ;
      nckPause();

      sCnt *= 2 ;
      gsOut.compose( L" >>Outputting a string %02hd times (multi-color): \n", &sCnt ) ;
      wp = nc.WriteStream ( gsOut.gstr(), nc.brbl ) ;
      for ( short i = ZERO ; i < sCnt ; i++ )
         wp = nc.WriteStream ( Popeye, pColor[i] ) ;
      nc.WriteStream ( L"X", nc.reR ) ;
      gs.compose( cursorTemplate, &wp.ypos, &wp.xpos ) ;
      wp = nc.WriteStream ( gs.gstr(), nc.brbl ) ;
      wp = nc.WriteStream ( paK, nc.rebk ) ;
      nckPause();

      sCnt /= 2 ;
      gsOut.compose( L" >>Outputting a string %02hd times (with newlines): \n", &sCnt ) ;
      wp = nc.WriteStream ( gsOut.gstr(), nc.brbl ) ;
      gsOut.compose( "%S\n", Popeye ) ;
      for ( short i = ZERO ; i < sCnt ; i++ )
         wp = nc.WriteStream ( gsOut.gstr(), pColor[sCnt + i] ) ;
      nc.WriteStream ( L"X", nc.reR ) ;
      gs.compose( cursorTemplate, &wp.ypos, &wp.xpos ) ;
      wp = nc.WriteStream ( gs.gstr(), nc.brbl ) ;
      wp = nc.WriteStream ( paK, nc.rebk ) ;
      nckPause();

      wp = nc.WriteStream ( 
           L"Execute the following command: grep -n 'NCurses-class' *.[hc]pp \n"
            " Then capture the output, and colorize the results for display: \n", nc.brbl ) ;
      system ( "grep -n 'NCurses-class' *.[hc]pp >t.txt" ) ;

      //* Open the file with captured data *
      ifstream ifs ( "t.txt", ifstream::in ) ;
      if ( ifs.is_open() )
      {
         const wchar_t* findIt = L"NCurses-class" ;
         short i ;
         while ( (ReadLine ( ifs, gs )) == OK )
         {
            gsOut = gs ;
            i = gsOut.find( L':' ) + 1 ;
            gsOut.limitChars( i ) ;
            nc.WriteStream ( gsOut.gstr(), nc.re ) ;
            gs.shiftChars( (0 - i) ) ;
            gsOut = gs ;
            i = gsOut.find( L':' ) + 1 ;
            gsOut.limitChars( i ) ;
            winPos wp = nc.WriteStream ( gsOut.gstr(), nc.gr ) ;
            while ( wp.xpos < 20 )
               wp = nc.WriteChar ( wp, nckSPACE, nc.bw ) ;
            gs.shiftChars( (0 - i) ) ;

            while ( gs.gstr()[ZERO] == nckSPACE )
               gs.shiftChars ( -1 ) ;

            i = gs.find( findIt ) ;
            if ( i >= ZERO )
            {
               gsOut = gs ;
               gsOut.limitChars( i ) ;
               nc.WriteStream ( gsOut.gstr(), nc.bw ) ;
               gs.shiftChars( (0 - i) ) ;
               gsOut = gs ;
               gsOut.limitChars ( 13 ) ;
               nc.WriteStream ( gsOut.gstr(), (nc.blgr & ~ncbATTR) ) ;
               gs.shiftChars( (0 - 13) ) ;
               if ( (i = gs.find( findIt )) >= ZERO )
               { /* currently no duplicates */ }
               gsOut = gs ;
               gsOut.append( L'\n' ) ;
               nc.WriteStream ( gsOut.gstr(), nc.bw ) ;
            }
            else
            {  //* This in unlikey to happen *
               gs.append( L'\n' ) ;
               nc.WriteStream ( gs.gstr(), nc.bw ) ;
            }
         }

         ifs.close() ;
      }
      else
      {
         nc.WriteStream ( L"  Command capture failed! File not found.  ", nc.reR ) ;
      }
      nc.WriteStream ( L"\n\n", nc.bw ) ;
      nc.WriteStream ( paK, nc.rebk ) ;
      nckPause();

      nc.OutputStreamScrolling ( false ) ;   // disable stream output
      nc.ClearScreen () ;                    // clear terminal window
      nc.StopNCursesEngine () ;              // Deactivate the NCurses engine
   }
   else
   {
      cout << "\n **********************************************"
              "\n * ERROR! Unable to initialize NCurses engine.*"
              "\n **********************************************\n\n" ;
   }
}  //* End Test11() *

//*************************
//*       ReadLine        *
//*************************
//******************************************************************************
//* Read one line of text from the specified input stream.                     *
//*                                                                            *
//* Input  : ifs    : open input stream (by reference)                         *
//*          gs     : (by reference) receives text data                        *
//*                                                                            *
//* Returns: OK if successful, ERR processing error                            *
//******************************************************************************
//* Programmer's Note: Either the 'grep' command or the redirection-to-file    *
//* inserts carriage-returns at random spots in the output. We can't bear this *
//* foolishness, so we scan the stream and remove the carriage-returns.        *
//*                                                                            *
//******************************************************************************

static short ReadLine ( ifstream& ifs, gString& gs )
{
   char  tmp[gsDFLTBYTES] ;   // UTF-8 line input from file
   short status = ERR ;       // return value

   ifs.getline ( tmp, gsDFLTBYTES, NEWLINE ) ;  // read a source line
   if ( ifs.good() || ifs.gcount() > ZERO )     // if a good read
   {
      gs = tmp ;                                // convert to wide data

      //* Strip trailing carriage-returns *
      const wchar_t CR = 0x000D ;
      if ( (gs.find( CR )) >= ZERO )
         gs.limitChars( gs.gschars() - 2 ) ;

      status = OK ;                             // return with good news
   }
   return status ;

}  //* End ReadLine() *

//*************************
//*                       *
//*************************
//******************************************************************************
//*                                                                            *
//*                                                                            *
//*                                                                            *
//* Input  :                                                                   *
//*                                                                            *
//* Returns:                                                                   *
//******************************************************************************

