//******************************************************************************
//* File       : MenuTest.cpp                                                  *
//* Author     : Mahlon R. Smith                                               *
//*              Copyright (c) 2010-2024 Mahlon R. Smith, The Software Samurai *
//*                  GNU GPL copyright notice located in NcDialog.hpp          *
//* Date       : 07-Jun-2024                                                   *
//* Version    : (see below)                                                   *
//*                                                                            *
//* Description: MenuTest is a simple class definition which exercises the     *
//*              NcDialog DialogMenuwin class as well as the DialogScrollext   *
//*              class. It is instantiated by the Dialog2 application, Test01. *
//*                                                                            *
//******************************************************************************
//* Version History (most recent first):                                       *
//*                                                                            *
//* v: 0.00.02 22-Mar-2015                                                     *
//*   -- Consolidate and simplify this test. Dialog layout and useful code     *
//*      from FileMangler application retained, and unneeded code and data     *
//*      discarded.                                                            *
//*                                                                            *
//* v: 0.00.01 12-Aug-2014                                                     *
//*   -- This code taken from the FileMangler application, v:0.00.27.          *
//******************************************************************************

#include "MenuTest.hpp"

//****************
//** Local Data **
//****************
//* Application Title (AppVersion is inserted for display) *
static const char AppTitle1[] = "  FileMangler - v:" ;
static const char AppTitle2[] = " (c) 2005-13  [Press F2 for Menu]  " ;
static const char AppVersion[] = { "0.00.27" } ; // FileMangler version
static const char quickHelpMsg[] = "    Ctrl+Q=Quit, Shift+F1=Help, F2=Menu " ;
static attr_t monoColor[2] = { attrDFLT, nc.blR } ;// menu data colors (monochrome data)
static const short COLLAPSED_WIDTH = 11 ;          // width of a collapsed Menuwin control


//***********************************************
//* Text strings for populating the menu system *
//* and the associated enum MenuCode values.    *
//***********************************************

//* 'File' menu data *
static const short mbFileDATA_ITEMS = 10, mbFileITEM_LEN = 18 ;
static const char mbFileData[mbFileDATA_ITEMS][mbFileITEM_LEN] =   
{
   "^File Commands  >",
   "^SortOptions c+S>",
   "F^avorites... c+O",
   "^View File      >",
   "Fin^d Files...c+F",
   "^Mount...     c+Y",
   "^Unmount...      ",
   "  (unassigned)   ",
   "  (unassigned)   ",
   "E^xit FileMgr c+Q"
} ;
static const MenuCode mbFileCodes[mbFileDATA_ITEMS] = 
{
   mcHANDLED,           //* sub-menu access (no code)
   mcFileMB_SORTOPTION, //* request to open Sort Option dialog
   mcFileMB_FAVORITES,  //* request to open Favorites dialog
   mcFileMB_VIEWFILE,   //* SUBMENU - request to view file as text/hex/binary
   mcFileMB_FINDFILES,  //* request to open find-files dialog
   mcFileMB_MOUNT,      //* request to open mount-filesystem dialog
   mcFileMB_UNMOUNT,    //* request to open unmount-filesystem dialog
   mcHANDLED,           //* TEMP - PLACEHOLDER
   mcHANDLED,           //* TEMP - PLACEHOLDER
   mcFileMB_EXIT,       //* exit the application
} ;
//* 'Edit' menu data *
static const short mbEditDATA_ITEMS = 7, mbEditITEM_LEN = 18 ;
static const char mbEditData[mbEditDATA_ITEMS][mbEditITEM_LEN] =   
{
   "C^opy         c+C",
   "^Cut          c+X",
   "^Paste        c+V",
   "Pas^teSpecial c+P",
   "Select ^All   c+A",
   "^UnTrashcan   c+U",
   "Preferences...   "
} ;
static const MenuCode mbEditCodes[mbEditDATA_ITEMS] = 
{
   mcEditMB_COPY,       // copy files
   mcEditMB_CUT,        // cut files
   mcEditMB_PASTE,      // paste files
   mcEditMB_PASTESP,    // paste-special files
   mcEditMB_SELECTALL,  // select all files in window
   mcEditMB_UNDODEL,    // undo delete-to-trashcan
   mcEditMB_PREFS,      // request to open Preferences dialog
} ;
//* 'View' menu data *
static const short mbViewDATA_ITEMS = 10, mbViewITEM_LEN = 18 ;
static const char mbViewData[mbViewDATA_ITEMS][mbViewITEM_LEN] =   
{
   "^Dual Window  a+W",
   "^Single Win   a+W",
   "^Tree view    a+T",
   "Menu Bar ^Visible",
   "^User Info       ",
   "^Clipboard Info  ",
   "File S^ystem Info",
   "^Hidden Files    ",
   "^FMG Disk Usage  ",
   "  (unassigned)   "
} ;
static const MenuCode mbViewCodes[mbViewDATA_ITEMS] = 
{
   mcViewMB_DUALWIN,    // dial-window display
   mcViewMB_SINGLEWIN,  // single-window display
   mcViewMB_DIRTREE,    // display file-system tree structure
   mcViewMB_LOCKMBAR,   // lock menu bar in visible state
   mcViewMB_USERINFO,   // display user info
   mcViewMB_CLIPBRD,    // browse clipboard
   mcViewMB_FILESYS,    // file system summary
   mcViewMB_HIDDEN,     // display/hide hidden files
   mcViewMB_DISKUSAGE,  // application's disk usage
   mcHANDLED,           // TEMP - PLACEHOLDER
} ;

//* 'Util' menu data *
static const short mbUtilDATA_ITEMS = 6, mbUtilITEM_LEN = 18 ;
static const char mbUtilData[mbUtilDATA_ITEMS][mbUtilITEM_LEN] = 
{
   "^Backup       a+B",
   "^Synchronize  a+Z",
   "Compare^Directory",
   "Compare ^Files   ",
   "Create ^Log File ",
   "^Configure...    ",
} ;
static const MenuCode mbUtilCodes[mbUtilDATA_ITEMS] = 
{
   mcUtilMB_BACKUP,                 //* back-up directory contents
   mcUtilMB_SYNCH,                  //* synchronnize directory contents
   mcUtilMB_CMPDIR,                 //* sompare directory contents
   mcUtilMB_CMPFILE,                //* compare two files (diff)
   mcUtilMB_LOGFILE,                //* enable/disable transacton log
   mcUtilMB_CONFIG,                 //* request to open configuration dialog
} ;
//* 'Help' menu data *
static const short mbHelpDATA_ITEMS = 3, mbHelpITEM_LEN = 18 ;
static const char mbHelpData[mbHelpDATA_ITEMS][mbHelpITEM_LEN] =   
{
   "FileMangler ^Help",
   "Key B^indings    ",
   "^About FileMgr   ",
} ;
static const MenuCode mbHelpCodes[mbHelpDATA_ITEMS] = 
{
   mcHelpMB_F01HELP,    // request to open main help dialog
   mcHelpMB_KEYBIND,    // display current key bindings
   mcHelpMB_HELPABOUT,  // display Help-about dialog
} ;
//* File-command sub-menu data *
static const short mbFcmdDATA_ITEMS = 13, mbFcmdITEM_LEN = 19 ;
static const char mbFcmdData[mbFcmdDATA_ITEMS][mbFcmdITEM_LEN] =   
{
   "^Select File   spc",
   "Select ^All    c+A",
   "Tras^hcanFiles c+D",
   "C^opy Files    c+C",
   "C^ut Files     c+X",
   "^Paste Files   c+V",
   "Paste Specia^l c+P",
   "^DeleteFiles s+DEL",
   "^Rename Files  c+R",
   "^Touch Files   c+T",
   "^Write Protect c+W",
   "Write ^Enable  c+E",
   "^New Directory c+N"
} ;
static const MenuCode mbFcmdCodes[mbFcmdDATA_ITEMS] = 
{
   mcFcmdMB_SELECT,     //* select/deselect file under highlight
   mcFcmdMB_SELECTALL,  //* select/deselect all files in window
   mcFcmdMB_TRASH,      //* move files to trashcan
   mcFcmdMB_COPY,       //* copy files
   mcFcmdMB_CUT,        //* cut files
   mcFcmdMB_PASTE,      //* paste files
   mcFcmdMB_PASTESP,    //* paste-special files
   mcFcmdMB_DELETE,     //* delete files
   mcFcmdMB_RENAME,     //* rename files
   mcFcmdMB_TOUCH,      //* touch files (set mod date)
   mcFcmdMB_WENABLE,    //* write-enable files
   mcFcmdMB_WPROT,      //* write-protect files
   mcFcmdMB_NEWDIR,     //* create new directory
} ;
//* Sort-order sub-menu data *
static const short mbSortDATA_ITEMS = 11, mbSortITEM_LEN = 19 ;
static const char mbSortData[mbSortDATA_ITEMS][mbSortITEM_LEN] =   
{
   "^Name             ",
   "Na^me Reverse     ",
   "^Date             ",
   "D^ate Reverse     ",
   "^Size             ",
   "Si^ze Reverse     ",
   "^Extension        ",
   "E^xtension Reverse",
   "File ^Type        ",
   "FileT^ype Reverse ",
   "N^o Sort          "
} ;
static const MenuCode mbSortCodes[mbSortDATA_ITEMS] = 
{
   mcSortMB_NAME,       // sort by filename (low-to-high)
   mcSortMB_NAMEREV,    // sort by filename (high-to-low)
   mcSortMB_DATE,       // sort by mod date (low-to-high)
   mcSortMB_DATEREV,    // sort by mod date (high-to-low)
   mcSortMB_SIZE,       // sort by file size (low-to-high)
   mcSortMB_SIZEREV,    // sort by file size (high-to-low)
   mcSortMB_EXT,        // sort by file extension (low-to-high)
   mcSortMB_EXTREV,     // sort by file extension (high-to-low)
   mcSortMB_TYPE,       // sort by file type (low-to-high)
   mcSortMB_TYPEREV,    // sort by file type (high-to-low)
   mcSortMB_NOSORT,     // no sort
} ;
//* File-view context menu *
static const short cmVfileDATA_ITEMS = 5, cmVfileITEM_LEN = 19 ;
static const char cmVfileData[cmVfileDATA_ITEMS][cmVfileITEM_LEN] =   
{
   " View file ^Stats ",
   " View ^Contents   ",
   " ^Execute (Run)   ",
   " Find Link Target ",
   " Cance^l          "
} ;
static const MenuCode cmVfileCodes[cmVfileDATA_ITEMS] = 
{
   mcVfileCM_VIEWSTATS,             //* view file statistics
   mcVfileCM_CONTENTS,              //* view file contents
   mcVfileCM_EXECUTE,               //* if file is an executable, run it
   mcVfileCM_FINDTARGET,            //* if file is a symbolic link, find link target
   mcVfileCM_CANCELVIEW,            //* cancel view-file operation
} ;

static char MenuCodeStrings[][24] = 
{
   //* GENERAL    *
   "mcNO_SELECTION",                  //* Menu closed without a selection,
   "mcHANDLED",                       //* Selection handled before return
   "mcSYSERROR",                      //* If system call returned an error

   //* FILE MENU  *
   "mcFileMB_SORTOPTION",             //* request to open Sort Option dialog
   "mcFileMB_FAVORITES",              //* request to open Favorites dialog
   "mcFileMB_VIEWFILE",               //* request to view file as text/hex/binary
   "mcFileMB_FINDFILES",              //* request to open find-files dialog
   "mcFileMB_MOUNT",                  //* request to open mount-filesystem dialog
   "mcFileMB_UNMOUNT",                //* request to open unmount-filesystem dialog
   "mcFileMB_EXIT",                   //* exit the application

   //* EDIT MENU  *
   "mcEditMB_COPY",                   //* copy files
   "mcEditMB_CUT",                    //* cut files
   "mcEditMB_PASTE",                  //* paste files
   "mcEditMB_PASTESP",                //* paste-special files
   "mcEditMB_SELECTALL",              //* select all files in window
   "mcEditMB_UNDODEL",                //* undo delete-to-trashcan
   "mcEditMB_PREFS",                  //* request to open Preferences dialog

   //* VIEW MENU  *
   "mcViewMB_DUALWIN",                //* dial-window display
   "mcViewMB_SINGLEWIN",              //* single-window display
   "mcViewMB_DIRTREE",                //* display file-system tree structure
   "mcViewMB_LOCKMBAR",               //* lock menu bar in visible state
   "mcViewMB_USERINFO",               //* display user info
   "mcViewMB_CLIPBRD",                //* browse clipboard
   "mcViewMB_FILESYS",                //* file system summary
   "mcViewMB_HIDDEN",                 //* display/hide hidden files
   "mcViewMB_DISKUSAGE",              //* application's disk usage

   //* UTIL MENU  *
   "mcUtilMB_BACKUP",                 //* back-up directory contents
   "mcUtilMB_SYNCH",                  //* synchronnize directory contents
   "mcUtilMB_CMPDIR",                 //* sompare directory contents
   "mcUtilMB_CMPFILE",                //* compare two files (diff)
   "mcUtilMB_LOGFILE",                //* enable/disable transacton log
   "mcUtilMB_CONFIG",                 //* request to open configuration dialog

   //* HELP MENU  *
   "mcHelpMB_F01HELP",                //* request to open main help dialog
   "mcHelpMB_KEYBIND",                //* display current key bindings
   "mcHelpMB_HELPABOUT",              //* display Help-about dialog

   //* FCMD MENU  *
   "mcFcmdMB_SELECT",                 //* select/deselect file under highlight
   "mcFcmdMB_SELECTALL",              //* select/deselect all files in window
   "mcFcmdMB_TRASH",                  //* move files to trashcan
   "mcFcmdMB_COPY",                   //* copy files
   "mcFcmdMB_CUT",                    //* cut files
   "mcFcmdMB_PASTE",                  //* paste files
   "mcFcmdMB_PASTESP",                //* paste-special files
   "mcFcmdMB_DELETE",                 //* delete files
   "mcFcmdMB_RENAME",                 //* rename files
   "mcFcmdMB_TOUCH",                  //* touch files (set mod date)
   "mcFcmdMB_WENABLE",                //* write-enable files
   "mcFcmdMB_WPROT",                  //* write-protect files
   "mcFcmdMB_NEWDIR",                 //* create new directory

   //* SORT MENU  *
   "mcSortMB_NAME",                   //* sort by filename (low-to-high)
   "mcSortMB_NAMEREV",                //* sort by filename (high-to-low)
   "mcSortMB_DATE",                   //* sort by mod date (low-to-high)
   "mcSortMB_DATEREV",                //* sort by mod date (high-to-low)
   "mcSortMB_SIZE",                   //* sort by file size (low-to-high)
   "mcSortMB_SIZEREV",                //* sort by file size (high-to-low)
   "mcSortMB_EXT",                    //* sort by file extension (low-to-high)
   "mcSortMB_EXTREV",                 //* sort by file extension (high-to-low)
   "mcSortMB_TYPE",                   //* sort by file type (low-to-high)
   "mcSortMB_TYPEREV",                //* sort by file type (high-to-low)
   "mcSortMB_NOSORT",                 //* no sort

   //* VIEW-FILE MENU *
   "mcVfileCM_VIEWSTATS",             //* view file statistics
   "mcVfileCM_CONTENTS",              //* view file contents
   "mcVfileCM_EXECUTE",               //* if file is an executable, run it
   "mcVfileCM_FINDTARGET",            //* if file is a symbolic link, find link target
   "mcVfileCM_CANCELVIEW",            //* cancel view-file operation

} ;

//* Test data for the DialogScrollext class object.                    *
//* Replaces the live file data when FileDlg class is not instantiated.*
static const char testStrings[64][80] = 
{
   "Test String #0                                                             x",
   "Test String #1                                                             x",
   "Test String #2                                                             x",
   "Test String #3                                                             x",
   "Test String #4                                                             x",
   "Test String #5                                                             x",
   "Test String #6                                                             x",
   "Test String #7                                                             x",
   "Test String #8                                                             x",
   "Test String #9                                                             x",
   "Test String #10                                                            x",
   "Test String #11                                                            x",
   "Test String #12                                                            x",
   "Test String #13                                                            x",
   "Test String #14                                                            x",
   "Test String #15                                                            x",
   "Test String #16                                                            x",
   "Test String #17                                                            x",
   "Test String #18                                                            x",
   "Test String #19                                                            x",
   "Test String #20                                                            x",
   "Test String #21                                                            x",
   "Test String #22                                                            x",
   "Test String #23                                                            x",
   "Test String #24                                                            x",
   "Test String #25                                                            x",
   "Test String #26                                                            x",
   "Test String #27                                                            x",
   "Test String #28                                                            x",
   "Test String #29                                                            x",
   "Test String #30                                                            x",
   "Test String #31                                                            x",
   "Test String #32                                                            x",
   "Test String #33                                                            x",
   "Test String #34                                                            x",
   "Test String #35                                                            x",
   "Test String #36                                                            x",
   "Test String #37                                                            x",
   "Test String #38                                                            x",
   "Test String #39                                                            x",
   "Test String #40                                                            x",
   "Test String #41                                                            x",
   "Test String #42                                                            x",
   "Test String #43                                                            x",
   "Test String #44                                                            x",
   "Test String #45                                                            x",
   "Test String #46                                                            x",
   "Test String #47                                                            x",
   "Test String #48                                                            x",
   "Test String #49                                                            x",
   "Test String #50                                                            x",
   "Test String #51                                                            x",
   "Test String #52                                                            x",
   "Test String #53                                                            x",
   "Test String #54                                                            x",
   "Test String #55                                                            x",
   "Test String #56                                                            x",
   "Test String #57                                                            x",
   "Test String #58                                                            x",
   "Test String #59                                                            x",
   "Test String #60                                                            x",
   "Test String #61                                                            x",
   "Test String #62                                                            x",
   "Test String #63 - Last                                                     x",
//   "Test String #64                                                            x",
//   "Test String #65                                                            x",
//   "Test String #66                                                            x",
//   "Test String #67                                                            x",
//   "Test String #68                                                            x",
//   "Test String #69                                                            x",
} ;
static attr_t testAttributes [] =
{
   0x00000000, 0x00000100, 0x00000200, 0x00000300, 0x00000400, 0x00000500, 0x00000600, 0x00000700, 
   0x00000000, 0x00000100, 0x00000200, 0x00000300, 0x00000400, 0x00000500, 0x00000600, 0x00000700, 
   0x00000000, 0x00000100, 0x00000200, 0x00000300, 0x00000400, 0x00000500, 0x00000600, 0x00000700, 
   0x00000000, 0x00000100, 0x00000200, 0x00000300, 0x00000400, 0x00000500, 0x00000600, 0x00000700, 
   0x00000000, 0x00000100, 0x00000200, 0x00000300, 0x00000400, 0x00000500, 0x00000600, 0x00000700, 
   0x00000000, 0x00000100, 0x00000200, 0x00000300, 0x00000400, 0x00000500, 0x00000600, 0x00000700, 
   0x00000000, 0x00000100, 0x00000200, 0x00000300, 0x00000400, 0x00000500, 0x00000600, 0x00000700, 
   0x00000000, 0x00000100, 0x00000200, 0x00000300, 0x00000400, 0x00000500, 0x00000600, 0x00000700, 
   0x00000000, 0x00000100, 0x00000200, 0x00000300, 0x00000400, 0x00000500, 0x00000600, 0x00000700, 
   0x00000000, 0x00000100, 0x00000200, 0x00000300, 0x00000400, 0x00000500, 0x00000600, 0x00000700, 
} ;
static const char* testData[] = 
{
   testStrings[0],  testStrings[1],  testStrings[2],  testStrings[3],  testStrings[4], 
   testStrings[5],  testStrings[6],  testStrings[7],  testStrings[8],  testStrings[9], 
   testStrings[10], testStrings[11], testStrings[12], testStrings[13], testStrings[14], 
   testStrings[15], testStrings[16], testStrings[17], testStrings[18], testStrings[19], 
   testStrings[20], testStrings[21], testStrings[22], testStrings[23], testStrings[24], 
   testStrings[25], testStrings[26], testStrings[27], testStrings[28], testStrings[29], 
   testStrings[30], testStrings[31], testStrings[32], testStrings[33], testStrings[34], 
   testStrings[35], testStrings[36], testStrings[37], testStrings[38], testStrings[39], 
   testStrings[40], testStrings[41], testStrings[42], testStrings[43], testStrings[44], 
   testStrings[45], testStrings[46], testStrings[47], testStrings[48], testStrings[49], 
   testStrings[50], testStrings[51], testStrings[52], testStrings[53], testStrings[54], 
   testStrings[55], testStrings[56], testStrings[57], testStrings[58], testStrings[59], 
   testStrings[60], testStrings[61], testStrings[62], testStrings[63], //testStrings[64], 
//   testStrings[65], testStrings[66], testStrings[67], testStrings[68], testStrings[69], 
} ;





//* Index numbers for controls in the Dialog Window *
enum dwControls { dwFileSE = ZERO, dwMsgsTB, dwPathTB, dwStatTB, dwFileMB, 
                  dwEditMB, dwViewMB, dwUtilMB, dwHelpMB, dwFcmdMB, dwSortMB, 
                  dwVfileCM, dwCTRLS } ;

//* Controls that are members of the Menu Bar and sub-menu associations *
static const short mbList[] = { dwFileMB, dwEditMB, dwViewMB, dwUtilMB, dwHelpMB, -1 } ;
static const short mbSubListFile[] = { dwFcmdMB, dwSortMB, MAX_DIALOG_CONTROLS, dwVfileCM, -1 } ;

//* Control Initialization Structures.                  *
//* Note that some of the following data need to be     *
//* initialized AFTER instantiation of the parent class *
static InitCtrl ic[dwCTRLS] = 
{
   {  //* File Data - - - - - - - - dwFileSE *
      dctSCROLLEXT,                 // type:      define a scrolling-data control
      rbtTYPES,                     // sbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      4,                            // ulY:       upper left corner in Y
      ZERO,                         // ulX:       upper left corner in X
      0,                            // lines:     control lines
      0,                            // cols:      control columns
      NULL,                         // dispText:  n/a
      nc.cyR,                       // nColor:    non-focus border color
      nc.cyR,                       // fColor:    focus border color
      tbPrint,                      // filter:    (n/a)
      "",                           // label:     no label
      ZERO,                         // labY:      offset from control's ulY
      ZERO,                         // labX       offset from control's ulX
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[dwMsgsTB]                 // nextCtrl:  link in next structure
   },
   {  //* Status Messages - - - - - dwMsgsTB *
      dctTEXTBOX,                   // type:      define a plain-text textbox
      rbtTYPES,                     // sbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      1,                            // ulY:       upper left corner in Y
      8,                            // ulX:       upper left corner in X
      1,                            // lines:     one display line
      0,                            // cols:      display columns
      quickHelpMsg,                 // dispText: initial display text
      nc.reB,                       // nColor:    normal message text color
      nc.bw,                        // fColor:    alternate message text color
      tbPrint,                      // filter:    accepts all printing ASCII characters
      "Status:",                    // label:     label text
      ZERO,                         // labY:      offset from control's ulY
      -7,                           // labX:      offset from control's ulX
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      false,                        // active:    view only
      &ic[dwPathTB]                 // nextCtrl:  link in next structure
   },
   {  //* Directory Path  - - - - - dwPathTB *
      dctTEXTBOX,                   // type:      define a plain-text textbox
      rbtTYPES,                     // sbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      3,                            // ulY:       offset in Y from corner of dialog
      1,                            // ulX:       offset in X from corner of dialog
      1,                            // lines:     one display line
      0,                            // cols:      display columns
      "",                           // dispText:  initial display text
      nc.bl,                        // nColor:    normal message text color
      nc.re,                        // fColor:    alternate message text color
      tbPrint,                      // filter:    accepts all printing ASCII characters
      "",                           // label:      no label text (empty string)
      ZERO,                         // labY:      offset from control's ulY
      ZERO,                         // labX:      offset from control's ulX
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      false,                        // active:    view only
      &ic[dwStatTB]                 // nextCtrl:  link in next structure
   },
   {  //* File Statistics - - - - - dwStatTB *
      dctTEXTBOX,                   // type:      define a plain-text textbox
      rbtTYPES,                     // sbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      0,                            // ulY:       offset in Y from corner of dialog
      1,                            // ulX:       offset in X from corner of dialog
      1,                            // lines:     one display line
      0,                            // cols:      display columns
      "",                           // dispText:  initial display text (empty string)
      nc.bl,                        // nColor:    normal message text color
      nc.reB,                       // fColor:    alternate message text color
      tbPrint,                      // filter:    accepts all printing ASCII characters
      "",                           // label:      no label text
      ZERO,                         // labY:      offset from control's ulY
      ZERO,                         // labX:      offset from control's ulX
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      false,                        // active:    view only
      &ic[dwFileMB]                 // nextCtrl:  link in next structure
   },
   {  //* "File" Menu - - - - - - - dwFileMB *
      dctMENUWIN,                   // type:      define a Menu-win control
      rbtTYPES,                     // sbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      0,                            // ulY:       offset in Y from corner of dialog
      0,                            // ulX:       offset in X from corner of dialog
      1,                            // lines:     (n/a)
      mbFileITEM_LEN,               // cols:      menu width
      (char*)&mbFileData,           // dispText:  pointer to display-text array
      nc.blR,                       // nColor:    non-focus color for collapsed menu
      nc.cyR,                       // fColor:    focus color for collapsed menu
      tbPrint,                      // filter:    (n/a)
      "  File     ",                // label:     text for 'collapsed' menu
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX:      (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      mbFileDATA_ITEMS,             // scrItems:  number of elements in text/color arrays
      ZERO,                         // scrSel:    (n/a)
      monoColor,                    // scrColor:  single-color data strings
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[dwEditMB]                 // nextCtrl:  link in next structure
   },
   {  //* "Edit" Menu - - - - - - - dwEditMB *
      dctMENUWIN,                   // type:      define a Menu-win control
      rbtTYPES,                     // sbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      0,                            // ulY:       upper left corner in Y
      0,                            // UlX: upper left corner in X
      1,                            // lines:     (n/a)
      mbEditITEM_LEN,               // cols:      menu width
      (char*)&mbEditData,           // dispText:  pointer to display-text array
      nc.blR,                       // nColor:    non-focus color for collapsed menu
      nc.cyR,                       // fColor:    focus color for collapsed menu
      tbPrint,                      // filter:    (n/a)
      "  Edit     ",                // label:     text for 'collapsed' menu
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX:      (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      mbEditDATA_ITEMS,             // scrItems:  number of elements in text/color arrays
      ZERO,                         // scrSel:    (n/a)
      monoColor,                    // scrColor:  single-color data strings
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[dwViewMB]                 // nextCtrl:  link in next structure
   },
   {  //* "View" Menu - - - - - - - dwViewMB *
      dctMENUWIN,                   // type:      define a Menu-win control
      rbtTYPES,                     // sbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      0,                            // ulY:       upper left corner in Y
      0,                            // UlX: upper left corner in X
      1,                            // lines:     (n/a)
      mbViewITEM_LEN,               // cols:      menu width
      (char*)&mbViewData,           // dispText:  pointer to display-text array
      nc.blR,                       // nColor:    non-focus color for collapsed menu
      nc.cyR,                       // fColor:    focus color for collapsed menu
      tbPrint,                      // filter:    (n/a)
      "  View     ",                // label:     text for 'collapsed' menu
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX:      (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      mbViewDATA_ITEMS,             // scrItems:  number of elements in text/color arrays
      ZERO,                         // scrSel:    (n/a)
      monoColor,                    // scrColor:  single-color data strings
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[dwUtilMB]                 // nextCtrl:  link in next structure
   },
   {  //* "Util" Menu - - - - - - - dwUtilMB *
      dctMENUWIN,                   // type:      define a Menu-win control
      rbtTYPES,                     // sbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      0,                            // ulY:       upper left corner in Y
      0,                            // UlX: upper left corner in X
      1,                            // lines:     (n/a)
      mbUtilITEM_LEN,               // cols:      menu width
      (char*)&mbUtilData,           // dispText:  pointer to display-text array
      nc.blR,                       // nColor:    non-focus color for collapsed menu
      nc.cyR,                       // fColor:    focus color for collapsed menu
      tbPrint,                      // filter:    (n/a)
      "  Util     ",                // label:     text for 'collapsed' menu
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX:      (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      mbUtilDATA_ITEMS,             // scrItems:  number of elements in text/color arrays
      ZERO,                         // scrSel:    (n/a)
      monoColor,                    // scrColor:  single-color data strings
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[dwHelpMB]                 // nextCtrl:  link in next structure
   },
   {  //* "Help" Menu - - - - - - - dwHelpMB *
      dctMENUWIN,                   // type:      define a Menu-win control
      rbtTYPES,                     // sbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      0,                            // ulY:       upper left corner in Y
      0,                            // ulX: upper left corner in X
      1,                            // lines:     (n/a)
      mbHelpITEM_LEN,               // cols:      menu width
      (char*)&mbHelpData,           // dispText:  pointer to display-text array
      nc.blR,                       // nColor:    non-focus color for collapsed menu
      nc.cyR,                       // fColor:    focus color for collapsed menu
      tbPrint,                      // filter:    (n/a)
      "  Help     ",                // label:     text for 'collapsed' menu
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX:      (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      mbHelpDATA_ITEMS,             // scrItems:  number of elements in text/color arrays
      ZERO,                         // scrSel:    (n/a)
      monoColor,                    // scrColor:  single-color data strings
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[dwFcmdMB]                 // nextCtrl:  link in next structure
   },
   {  //* File commands sub-menu- - dwFcmdMB *
      dctMENUWIN,                   // type:      define a Menu-win control
      rbtTYPES,                     // sbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      0,                            // ulY:       upper left corner in Y
      0,                            // ulX: upper left corner in X
      1,                            // lines:     (n/a)
      mbFcmdITEM_LEN,               // cols:      menu width
      (char*)&mbFcmdData,           // dispText:  pointer to display-text array
      nc.blR,                       // nColor:    non-focus color for collapsed menu
      nc.cyR,                       // fColor:    focus color for collapsed menu
      tbPrint,                      // filter:    (n/a)
      "",                           // label:     no label for sub-menus
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX:      (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      mbFcmdDATA_ITEMS,             // scrItems:  number of elements in text/color arrays
      ZERO,                         // scrSel:    (n/a)
      monoColor,                    // scrColor:  single-color data strings
      NULL,                         // spinData:  (n/a)
      false,                        // active:    initially inactive and invisible
      &ic[dwSortMB]                 // nextCtrl:  link in next structure
   },
   {  //* Sort-order sub-menu - - - dwSortMB *
      dctMENUWIN,                   // type:      define a Menu-win control
      rbtTYPES,                     // sbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      0,                            // ulY:       upper left corner in Y
      0,                            // UlX: upper left corner in X
      1,                            // lines:     (n/a)
      mbSortITEM_LEN,               // cols:      menu width
      (char*)&mbSortData,           // dispText:  pointer to display-text array
      nc.blR,                       // nColor:    non-focus color for collapsed menu
      nc.cyR,                       // fColor:    focus color for collapsed menu
      tbPrint,                      // filter:    (n/a)
      "",                           // label:     no label for sub-menus
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX:      (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      mbSortDATA_ITEMS,             // scrItems:  number of elements in text/color arrays
      ZERO,                         // scrSel:    (n/a)
      monoColor,                    // scrColor:  single-color data strings
      NULL,                         // spinData:  (n/a)
      false,                        // active:    initially inactive and invisible
      &ic[dwVfileCM]                // nextCtrl:  link in next structure
   },
   {  //* View-file context menu - - - dwVfileCM *
      dctMENUWIN,                   // type:      define a Menu-win control
      rbtTYPES,                     // sbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      (short)(ic[dwFileSE].ulY + 2),// ulY:       upper left corner in Y
      (short)(ic[dwFileMB].ulX + ic[dwFileMB].cols - 1), // UlX: upper left corner in X
      1,                            // lines:     (n/a)
      cmVfileITEM_LEN,              // cols:      menu width
      (char*)&cmVfileData,          // dispText:  pointer to display-text array
      nc.blR,                       // nColor:    non-focus color for collapsed menu
      nc.cyR,                       // fColor:    focus color for collapsed menu
      tbPrint,                      // filter:    (n/a)
      "",                           // label:     no label for sub-menus
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX:      (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      cmVfileDATA_ITEMS,            // scrItems:  number of elements in text/color arrays
      ZERO,                         // scrSel:    (n/a)
      monoColor,                    // scrColor:  single-color data strings
      NULL,                         // spinData:  (n/a)
      false,                        // active:    initially inactive and invisible
      NULL                          // nextCtrl:  link in next structure
   }
} ;   // InitCtrl array

//*************************
//*      ~MenuTest        *
//*************************
//******************************************************************************
//* Destructor.                                                                *
//*                                                                            *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

MenuTest::~MenuTest ( void )
{

   if ( this->dp != NULL )                // close the window
      delete ( this->dp ) ;

}  //* End ~ColorTest() 

//*************************
//*       MenuTest        *
//*************************
//******************************************************************************
//* Constructor.                                                               *
//*                                                                            *
//* Input  : tLines     : number of display line in terminal window            *
//*          tCols      : number of display columns in terminal window         *
//*          minY       : first available display line                         *
//*                                                                            *
//* Returns: implicitly return class reference                                 *
//******************************************************************************

MenuTest::MenuTest ( short tLines, short tCols, short minY )
{
   //* Initialize data members *
   this->termRows   = tLines ;      // terminal window dimensions
   this->termCols   = tCols ;
   this->minULY     = minY ;        // first available display line
   this->dialogRows = this->termRows - this->minULY, // height of dialog
   this->dialogCols = mtMIN_COLS ;  // width of dialog
   this->menuBase   = { 0, 11 } ;   // position of Menu Bar
   this->titlePos   = { 0, 0 } ;    // initial position of dialog title
   this->dp         = NULL ;        // dialog instance pointer
   this->bColor     = nc.brcy;      // dialog border color
   this->dColor     = nc.cyR ;      // dialog background color
   this->mnColor    = nc.cyR,       // menu color (with input focus)
   this->mfColor    = nc.blR,       // menu color (without input focus)
   this->lockMenuBar = false ;      // menu bar is not locked in visible state
   this->mtOpen     = false ;

   //* Initialize remainder of control-definition array.           *
   //* (colors become available after NCurses engine instantiated) *
   monoColor[1] = nc.blR ;    // Menu interior color

   ic[dwFileSE].lines  = this->dialogRows - 6 ;
   ic[dwFileSE].cols   = this->dialogCols ;
   ic[dwFileSE].nColor = this->bColor ;
   ic[dwFileSE].fColor = this->bColor ;

   ic[dwMsgsTB].cols   = this->dialogCols - 11 ;
   ic[dwMsgsTB].nColor = nc.reB ;
   ic[dwMsgsTB].fColor = nc.bw ;

   ic[dwPathTB].cols   = this->dialogCols - 2 ;
   ic[dwPathTB].nColor = nc.bl ;
   ic[dwPathTB].fColor = nc.maS ;

   ic[dwStatTB].ulY    = this->dialogRows - 2 ;
   ic[dwStatTB].cols   = this->dialogCols - 2 ;
   ic[dwStatTB].nColor = nc.bl ;
   ic[dwStatTB].fColor = nc.reB ;

   ic[dwFileMB].ulY    = this->menuBase.ypos ;
   ic[dwFileMB].ulX    = this->menuBase.xpos ;
   ic[dwFileMB].nColor = this->mnColor ;
   ic[dwFileMB].fColor = this->mfColor ;

   ic[dwEditMB].ulY    = ic[dwFileMB].ulY ;
   ic[dwEditMB].ulX    = ic[dwFileMB].ulX + COLLAPSED_WIDTH ;
   ic[dwEditMB].nColor = this->mnColor ;
   ic[dwEditMB].fColor = this->mfColor ;

   ic[dwViewMB].ulY    = ic[dwEditMB].ulY ;
   ic[dwViewMB].ulX    = ic[dwEditMB].ulX + COLLAPSED_WIDTH ;
   ic[dwViewMB].nColor = this->mnColor ;
   ic[dwViewMB].fColor = this->mfColor ;

   ic[dwUtilMB].ulY    = ic[dwViewMB].ulY ;
   ic[dwUtilMB].ulX    = ic[dwViewMB].ulX + COLLAPSED_WIDTH ;
   ic[dwUtilMB].nColor = this->mnColor ;
   ic[dwUtilMB].fColor = this->mfColor ;

   ic[dwHelpMB].ulY    = ic[dwUtilMB].ulY ;
   ic[dwHelpMB].ulX    = ic[dwUtilMB].ulX + COLLAPSED_WIDTH ;
   ic[dwHelpMB].nColor = this->mnColor ;
   ic[dwHelpMB].fColor = this->mfColor ;

   ic[dwFcmdMB].ulY    = ic[dwFileMB].ulY + 2 ;
   ic[dwFcmdMB].ulX    = ic[dwFileMB].ulX + ic[dwFileMB].cols - 1 ;
   ic[dwFcmdMB].nColor = this->mnColor ;
   ic[dwFcmdMB].fColor = this->mfColor ;

   ic[dwSortMB].ulY    = ic[dwFileMB].ulY + 3 ;
   ic[dwSortMB].ulX    = ic[dwFileMB].ulX + ic[dwFileMB].cols - 1 ;
   ic[dwSortMB].nColor = this->mnColor ;
   ic[dwSortMB].fColor = this->mfColor ;

   ic[dwVfileCM].ulY    = ic[dwFileSE].ulY + 2 ;
   ic[dwVfileCM].ulX    = ic[dwFileMB].ulX + ic[dwFileMB].cols - 1 ;
   ic[dwVfileCM].nColor = this->mnColor ;
   ic[dwVfileCM].fColor = this->mfColor ;

   if ( (this->mtOpen = this->mtOpenDialog ()) != false )
   {
      /* nothing to do at this time */
   }  // OpenWindow()

}  //* End MenuTest() *

//*************************
//*     mtOpenDialog      *
//*************************
//******************************************************************************
//* Open the dialog window.                                                    *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: 'true' if dialog window opened successfully, else 'false'         *
//******************************************************************************

bool MenuTest::mtOpenDialog ( void )
{
   short ulY = this->minULY,
         ulX = ZERO ;
   bool  success = false ;

   //* Initial parameters for dialog window *
   InitNcDialog dInit( this->dialogRows, // number of display lines
                       this->dialogCols, // number of display columns
                       ulY,            // Y offset from upper-left of terminal 
                       ulX,            // X offset from upper-left of terminal 
                       NULL,           // dialog title
                       ncltSINGLE,     // border line-style
                       bColor,         // border color attribute
                       this->dColor,   // interior color attribute
                       ic              // pointer to list of control definitions
                     ) ;

   //* Instantiate the dialog window *
   this->dp = new NcDialog ( dInit ) ;

   //* Open the dialog window *
   if ( (this->dp->OpenWindow()) == OK )
   {
      //* Establish a Menu Bar (group of DialogMenuwin controls).           *
      //* Menu Bar is initially invisible (obscured by the dialog title).   *
      this->mtDialogTitle ( this->bColor, false, true ) ;

      //* Draw a line under the status display *
      LineDef  lDef(ncltHORIZ, ncltSINGLE, 2, ZERO, this->dialogCols, this->bColor) ;
      this->dp->DrawLine ( lDef ) ;

      //* Make a visual connection for the dctSCROLLEXT control that *
      //* overlaps the dialog window's border.                       *
      cdConnect cdConn ;
      cdConn.ul2Left = true ;
      cdConn.ll2Left = true ;
      cdConn.ur2Right = true ;
      cdConn.lr2Right = true ;
      cdConn.connection = true ;
      this->dp->ConnectControl2Border ( dwFileSE, cdConn ) ;

//      dtbmData  msgData("  <-- path of current working directory is displayed here -->") ;
//      this->dp->DisplayTextboxMessage ( dwPathTB, msgData ) ;
      this->dp->SetTextboxText ( dwPathTB, 
               "  <-- path of current working directory is displayed here -->" ) ;

      dtbmData msgData ;
      for ( short i = ZERO ; i < 76 ; i++ )  // initialize message color-attribute data
         msgData.colorData[i] = testAttributes[i] ;

      msgData = "0123456789012345  Display Test Data In File Display Control  5432109876543210" ;
      this->dp->DisplayTextboxMessage ( dwStatTB, msgData ) ;

      ssetData sData( (const char**)testData, testAttributes, 64, ZERO, true ) ;
      this->dp->SetScrollextText ( dwFileSE, sData ) ;

      msgData.textData[0] = NULLCHAR ; // redisplay control's saved message
      this->dp->DisplayTextboxMessage ( dwMsgsTB, msgData ) ;

      this->dp->RefreshWin () ;        // make everything visible
      success = true ;
   }
   return success ;

}  //* End mtOpenDialog() *

//*************************
//*    mtDialogOpened     *
//*************************
//******************************************************************************
//* Satisfy caller's curiosity.                                                *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: 'true' if dialog opened successfully, else 'false'                *
//******************************************************************************

bool MenuTest::mtDialogOpened ( void )
{
   return this->mtOpen ;

}  //* End mtDialogOpened() *

//*************************
//*      mtInteract       *
//*************************
//******************************************************************************
//* User interactive experience.                                               *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void MenuTest::mtInteract ( void )
{
   if ( this->mtOpen )
   {
      short    icIndex = this->dp->GetCurrControl () ;  // index of control with focus
      bool     done = false ;
      uiInfo   Info ;
      MenuCode menuCode = mcFileMB_EXIT ;

      while ( done == false )
      {
         if ( icIndex == dwFileSE )
         {
            //* Let user scroll through the data until a non-scroll key detected *
            icIndex = this->dp->EditScrollext ( Info ) ;
            
            //* If we are still focused on the dwFileSE control.     *
            //* (if not, we have moved to another control via hotkey *
            if ( icIndex == dwFileSE )
            {
               switch ( Info.keyIn )
               {
                  case nckTAB:               // move to next ACTIVE control
                     break ;
                  case nckSTAB:              // move to previous ACTIVE control
                     break ;
                  case nckF02:               // access the Menu Bar
                     if ( (menuCode = this->mtMenuBarAccess ()) != mcFileMB_EXIT )
                     {
                        attr_t c = dtbmFcolor ;
                        gString msgBuff ;
                        msgBuff.compose( L"MenuCode: 0x%04X - %s", 
                                         &menuCode, &MenuCodeStrings[menuCode] ) ;
                        dtbmData  msgData( msgBuff.gstr(), &c, true, msgBuff.gschars() ) ;
                        this->dp->DisplayTextboxMessage ( dwPathTB, msgData ) ;
                        sleep ( 2 ) ;
                        msgData = "" ;
                        this->dp->DisplayTextboxMessage ( dwPathTB, msgData ) ;
                     }
                     else
                        done = true ;
                     break ;
                  case 's':   // test only re-draw data with highlight
                     this->dp->RefreshScrollextText ( dwFileSE, true ) ;
                     break ;
                  case 'h':   // test only: re-draw data without highlight
                     this->dp->RefreshScrollextText ( dwFileSE, false ) ; 
                     break ;
                  case 'x':   // test only
                  case nckC_Q:               // Ctrl+q == exit to original directory
                  case nckESC:               // Escape
                     menuCode = mcFileMB_EXIT ;
                     done = true ;
                     break ;
                  default:                   // invalid input in this context
                     break ;
               }
            }
         }
         else
         {
            // no other active controls defined in this dialog
         }
         if ( done == false )
         {
            if ( Info.keyIn == nckSTAB )
               icIndex = this->dp->PrevControl () ; 
            else
               icIndex = this->dp->NextControl () ;
         }
      }     // while()
   }
   else
      { /* Caller is an idiot */ }

}  //* End mtInteract() *

//*************************
//*    mtDialogTitle      *
//*************************
//******************************************************************************
//* Draw the application title at the top of the dialog window.                *
//*                                                                            *
//* On first call after Single/Dual window mode is started, 'establish'        *
//* must be 'true', and 'false' at all other times.                            *
//*                                                                            *
//*                                                                            *
//* Input: colorAttr : color attribute for writing the title                   *
//*        showMenu  : if true, make the Menu Bar visible and place focus on   *
//*                     on first menu of the group.                            *
//*                    if false, make the Menu Bar invisible and display the   *
//*                     dialog title in that position.                         *
//*        establish : (optional, false by default)                            *
//*                    if 'true': establish the Menu Bar and the dialog title  *
//*                     (on return, dialog title will be displayed)            *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************
//* THIS IS A SUBSET OF THE METHOD FOUND IN FileMangler.                       *
//******************************************************************************

void MenuTest::mtDialogTitle ( attr_t colorAttr, bool showMenu, bool establish )
{
   //* Create the dialog title string *
   gString  gs ;
   gs.compose( L"%s%s%s", AppTitle1, AppVersion, AppTitle2 ) ;

   //* If dialog has just been instantiated, group the DialogMenuwin controls  *
   //* to create a Menu Bar, and establish the dialog title.                   *
   if ( establish != false )
   {
      //* Create a Menuwin group (MenuBar) and make it initially invisible  *
      if ( (dp->GroupMenuwinControls ( mbList )) == OK )
      {
         //* Attach sub-menus to the primary menu controls *
         if ( (this->dp->AttachMenuwinSubmenus ( dwFileMB, mbSubListFile )) != OK )
         { /* ERROR - unlikely */ }

         //* Make the MenuBar inactive and invisible *
         this->dp->HideMenuBar ( dwFileMB ) ;
      }
      else
      { /* ERROR - unlikely */ }

      //* Create and display the dialog title *
      this->titlePos = this->dp->SetDialogTitle ( gs.ustr(), colorAttr ) ;
   }

   //* Menu Bar and dialog title previously established *
   else
   {  //* If Menu Bar is to be displayed *
      if ( showMenu )
      {
         char  msgBuff[MAX_DIALOG_WIDTH] ;
         short spCount = gs.gscols(), i = ZERO ;
         while ( i < spCount )
            msgBuff[i++] = nckSPACE ;
         msgBuff[i] = NULLCHAR ;
         this->dp->WriteString ( this->titlePos, msgBuff, colorAttr, true ) ;
         this->dp->ShowMenuBar ( dwFileMB, true ) ;
      }
      //* Else, dialog title is to be displayed *
      else
      {
         if ( this->dp->MenuBarVisible ( dwFileMB ) )
            this->dp->HideMenuBar ( dwFileMB ) ;
         this->dp->WriteString ( this->titlePos, gs, colorAttr, true ) ;
      }
   }

}  //* End mtDialogTitle() *

//*************************
//*    mtMenuBarAccess    *
//*************************
//******************************************************************************
//* This method implements user access to all the menus and sub-menus of the   *
//* Menu Bar defined and instantiated in InitMenuSystem().                     *
//*                                                                            *
//* Considerable functionality, as well as context menus, may be accessed      *
//* directly through hotkeys, but the menu system provides a sure, global      *
//* access to all commands.                                                    *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: code indicating the menu item chosen                              *
//******************************************************************************
//* THIS IS A SUBSET OF THE METHOD FOUND IN FileMangler.                       *
//******************************************************************************

MenuCode MenuTest::mtMenuBarAccess ( void )
{
   MenuCode selection = mcNO_SELECTION ;        // return value
   short    cIndex = dp->GetCurrControl () ;  // index of control that has focus

   //* If the Menu Bar is currently hidden, make it visible, *
   //* and place the focus on the first menu                 *
   if ( ! (this->dp->MenuBarVisible ( dwFileMB )) )
      this->mtDialogTitle ( this->bColor, true ) ;

   //* User instructions *
   attr_t uMsgColor = dtbmNFcolor ;
   dtbmData uMsg( "  ESC exits menu--ESC exits menu--ESC exits menu--ESC exits menu",
                  &uMsgColor ) ;
   this->dp->DisplayTextboxMessage ( dwMsgsTB, uMsg ) ;

   //* Give user a chance to make a decision *
   uiInfo Info ;
   bool   done = false ;
   while ( ! done )
   {
      Info.viaHotkey = true ;          // expand the menu immediately on entry
      cIndex = this->dp->EditMenuwin ( Info ) ;
      if ( Info.dataMod == false && Info.keyIn != nckESC )
      {
         if ( cIndex == dwFileMB )
         {
            while ( cIndex < dwHelpMB )
               cIndex = this->dp->NextControl () ;
         }
         else
         {
            while ( cIndex > dwFileMB )
               cIndex = this->dp->PrevControl () ;
         }
      }
      else
         done = true ;
   }
   
   //* Erase user instructions *
   uMsg = "" ;
   this->dp->DisplayTextboxMessage ( dwMsgsTB, uMsg ) ;

   //* Translante menu selection index to MenuCode enum value *
   if ( Info.dataMod != false )
   {
      selection = this->mtMenuItem2MenuCode ( Info.ctrlIndex, Info.selMember ) ;
   }

   //* If Menu Bar is not locked in the visible state, hide it *
   if ( this->lockMenuBar == false )
   {
      while ( cIndex > dwFileSE )
         cIndex = dp->PrevControl () ;
      this->mtDialogTitle ( bColor | ncbATTR, false ) ;
   }
   dp->RefreshWin () ;

   return selection ;

}  //* End mtMenuBarAccess() *

//*************************
//*  mtMenuItem2MenuCode  *
//*************************
//******************************************************************************
//* A selection has been made in the menu system. Translate the selection      *
//* into an action code, so command may be executed.                           *
//*                                                                            *
//* Input  : menuIndex:  index of control from which selection was made        *
//*          menuMember: item number selected within the menu                  *
//*                                                                            *
//* Returns: member of enum MenuCode                                           *
//******************************************************************************
//* THIS IS A SUBSET OF THE METHOD FOUND IN FileMangler.                       *
//******************************************************************************

MenuCode MenuTest::mtMenuItem2MenuCode ( short menuIndex, short menuMember )
{
MenuCode    selection = mcNO_SELECTION ;     // return value

   switch ( menuIndex )
   {
      case dwFileMB:                      // dwFileMB menu
         selection = mbFileCodes[menuMember] ;
         break ;
      case dwEditMB:                      // dwEditMB menu
         selection = mbEditCodes[menuMember] ;
         break ;
      case dwViewMB:                      // dwViewMB menu
         selection = mbViewCodes[menuMember] ;
         break ;
      case dwUtilMB:                      // dwUtilMB menu
         selection = mbUtilCodes[menuMember] ;
         break ;
      case dwHelpMB:                      // dwHelpMB menu
         selection = mbHelpCodes[menuMember] ;
         break ;
      case dwFcmdMB:                      // dwFcmdMB menu
         selection = mbFcmdCodes[menuMember] ;
         break ;
      case dwSortMB:                      // dwSortMB menu
         selection = mbSortCodes[menuMember] ;
         break ;
      case dwVfileCM:                     // dwVfileCM menu
         selection = cmVfileCodes[menuMember] ;
         break ;
      default:                            // invalid menu index (unlikely)
         break ;
   }
   return selection ;

}  //* End mtMenuItem2MenuCode() *

