//********************************************************************************
//* File       : Dialog2.cpp                                                     *
//* Author     : Mahlon R. Smith                                                 *
//*              Copyright (c) 2010-2025 Mahlon R. Smith, The Software Samurai   *
//*                  GNU GPL copyright notice located in NCurses.hpp             *
//* Date       : 21-Mar-2025                                                     *
//* Version    : (see AppVersion string below)                                   *
//*                                                                              *
//* Description: Dialog2 is a test and demonstration application for             *
//*              development of the NcDialog class and associated classes.       *
//*                                                                              *
//*              See also the Dialog1, Dialog3, Dialog4 and Dialogw test         *
//*              applications for additional examples of using the NCurses       *
//*              class, NcWindow class, NcDialog class and gString class         *
//*              methods for building your own NcDialog-based applications.      *
//*                                                                              *
//* Development Tools: See NcDialog.cpp.                                         *
//********************************************************************************
//* Version History (most recent first):                                         *
//*                                                                              *
//* v: 0.00.23 21-Mar-2025                                                       *
//*   - The gString 'gsForm&' constructor is obsolete. Replace gString           *
//*     constructors which take a 'gsForm&' argument with standard               *
//*     formatting-template constructors.                                        *
//*   - Certain gString method parameters required promotion from 'short' to     *
//*     'int32_t'.                                                               *
//*                                                                              *
//* v: 0.0.22 23-Jun-2024                                                        *
//*   - Update Help dialog to reference G++13 and C++11.                         *
//*                                                                              *
//* v: 0.0.21 31-Aug-2021                                                        *
//*   - Update Test07 to include an instance of the new dctSLIDER control.       *
//*     See CapTest.cpp.                                                         *
//*                                                                              *
//* v: 0.00.20 02-Nov-2017                                                       *
//*   - Update Test03 to exercise the new hexadecimal spinner format.            *
//*   - Allow equals sign ('=') to delimit '-a' option and its argument.         *
//*                                                                              *
//* v: 0.00.19 20-Mar-2015                                                       *
//*   - Move the dialog-capture code from the experimental application to        *
//*     Test07. The source module CapTest.cpp contains the new functionality.    *
//*   - Reduce the complexity of Test01(), the FileMangler Menu Bar example.     *
//*     Because the functionality is now stable, the test no longer needs to     *
//*     be so complex. Move everything into MenuTest.cpp and MenuTest.hpp and    *
//*     eliminate all code borrowed from FileMangler.                            *
//*   - Add Test11() to exercise the new NCurses-class WriteStream() method.     *
//*     See source module StreamTest.cpp.                                        *
//*   - Update Test02 for dctSCROLLEXT early initialization.                     *
//*                                                                              *
//* v: 0.00.18 24-Nov-2013                                                       *
//*   - Move integer-formatting test to Dialogw to reflect that the integer      *
//*     formatting functionality has been moved into the gString class.          *
//*     This eliminates the header file FormatInteger.hpp from our build.        *
//*                                                                              *
//* v: 0.00.17 22-Jun-2013                                                       *
//*   - Due to significant enhancements in the FileMangler application, it is    *
//*     no longer practical to continue support for the FileMangler application  *
//*     subset in Test01. Instead, we have simplified Test01 to use only dummy   *
//*     display data rather than live file and directory data. Thus, linking     *
//*     of the FileDlg.a library is no longer necessary. Certain header files    *
//*     from the FileMangler application are still required.                     *
//*   - Test01 now simply demonstrates a complex dialog with dctSCROLLEXT        *
//*     control and a MenuBar group, which are maintainable. The remainder of    *
//*     the actual test code has been moved to FileMangler_Subset.cpp.           *
//*   - This change also means that we must return to using FormatInteger.hpp    *
//*     as the base code for Test07.                                             *
//*                                                                              *
//* v: 0.00.16 23-May-2013                                                       *
//*   - Update Test03() instructions to reflect enhancement in spinner-control   *
//      user interface.                                                          *
//*   - Because Test05 has grown so rapidly, create a new class, KeyTest to      *
//*     contain the key-input testing.                                           *
//*     - Add simple mouse-event reporting. Comprehensive mouse-event testing    *
//*       is handled elsewhere.                                                  *
//*   - Similarly, the color-mapping test has become rather too large, so its    *
//*     functionality has been moved to a new class, ColorTest which contains    *
//*     all the interesting functionality for Test04.                            *
//*                                                                              *
//* v: 0.00.15 23-Apr-2012                                                       *
//*   - Update the startup routine to match the changes changes made in the      *
//*     StartNCursesEngine() method. These changes enable remapping of color     *
//*     pairs and RGB registers on startup.                                      *
//*   - Add Test04() to exercise color initialization methods.                   *
//*     See also ColorMap2File() and File2ColorMap().                            *
//*                                                                              *
//* v: 0.00.14 29-Oct-2011                                                       *
//*   - Re-integrate the FileMangler version of FileDlg class link library.      *
//*     FileDlg is no longer built in the Dialog2 application directory.         *
//*   - FileMangler code for Test01() moved to a new module:                     *
//*     FileMangler_Subset.cpp.                                                  *
//*                                                                              *
//* v: 0.00.13 01-Sep-2011                                                       *
//*   - Move certain shared definitions, prototypes and data common to all the   *
//*     test and demonstration applications to a header, DialogAppShared.hpp.    *
//*                                                                              *
//* v: 0.00.12 01-Aug-2011                                                       *
//*   - Restructure control flow in main() to be more like the more polished     *
//*     Dialogw test application.                                                *
//*   - Integrate changes to-date in NCurses library in support of UTF-8         *
//*     encoding.                                                                *
//*   - Beautification and comment clean-up.                                     *
//*     (Some sloppy programmer was in a big fat hurry in)                       *
//*     (the previous release. Oh yeah, that was me    :-)                       *
//*                                                                              *
//* v: 0.00.11 21-Jan-2011 Add text case for DialogSpinner controls.             *
//*                                                                              *
//* v: 0.00.10 13-Apr-2010                                                       *
//*   - Base code is from the Dialog1 application.                               *
//*   - Additional test routines for various NcWindow and NcDialog class         *
//*     functionality, as well as basic testing for the FMgr and FileDialog      *
//*     classes.                                                                 *
//********************************************************************************
//* Programmer's Notes:                                                          *
//* - Because internationalization of I/O is always a possibility in any         *
//*   application, we use as few ASCII-oriented C/C++ library functions as       *
//*   possible. e.g. sprintf(), strlen(), toupper(), etc. Where such             *
//*   functionality is needed for character manipulation, we strongly advise     *
//*   using the equivalent 'wide chararacter' functions OR the gString class.    *
//*                                                                              *
//********************************************************************************

//****************
//* Header Files *
//****************
//* All necessary information for:                  *
//* NCurses, NcWindow, NcDialog and gString classes.*
#include "GlobalDef.hpp"

//* Definitions, prototypes and data shared among the test applications *
#include "DialogAppShared.hpp"

//* MenuTest-class definition for Test01() *
#include "MenuTest.hpp"

//* ColorTest-class definition for Test04() *
#include "ColorTest.hpp"

//* KeyTest-class definition for Test05() *
#include "KeyTest.hpp"


//***************
//* Definitions *
//***************


//**************
//* Prototypes *
//**************
//* Prototypes for Test07() only - see CapTest.cpp *
void CapSimple ( const winPos wp ) ;
void CapComplex ( const winPos wp ) ;

//* Prototype for Test11(). NCurses Engine testing (see main()). *
void Test11 ( void ) ;


//**********
//*  Data  *
//**********
//* Application version string. Keep it current! *
const char* AppVersion = "0.0.23" ;
const char* crYears = "2010-2025" ;

//* Programmer's Note: For any dialogs that use a callback method, a set of 
//* constants should be defined at file scope for the dialog's control indices, 
//* so both the instantiating method and its callback method have access to the 
//* same definitions. These is done as an enum with initial value == ZERO.

//* Test03() dialog control constants. *
enum T03Controls {
                   T3donePush=ZERO, 
                   T3spinnerA, T3spinnerB, T3spinnerC, T3spinnerD, T3spinnerE, 
                   T3spinnerF, T3spinnerG, T3spinnerH, T3spinnerI, 
                   T3spinOvr3, T3textOvr3, T3spinOvr5, T3textOvr5, 
                   T3copyPush, T3dumpPush
                 } ;


//*************************
//*        main           *
//*************************
//******************************************************************************
//* Program entry point.                                                       *
//*                                                                            *
//* Command-line Usage:  SEE BELOW FOR COMMAND-LINE ARGUMENT PROCESSING        *
//*                                                                            *
//*                                                                            *
//******************************************************************************

int main ( int argc, char* argv[], char* argenv[] )
{
short    testNum = - 1 ;         // number of test to execute (default TestMenu)
bool     winTooSmall = false ;   // set 'true' if terminal window is too small

   //* User may have specified one or more command-line arguments *
   commArgs clArgs( argc, argv, argenv ) ;
   testNum = GetCommandLineArgs ( clArgs ) ;

   //* This section is used only for testing the color map (See Test04()) *
   bool startInColor = true ;
   NcColorMap* cmapPtr = NULL ;
   if ( clArgs.altColors != false )
   {  //* Either start in monochrome mode or use alternate color map *
      if ( ! clArgs.altColorMap )
         startInColor = false ;  // monochrome mode
      else
      {
         NcColorMap* cmptr= new NcColorMap ; // NOTE: this will persist until exit
         if ( (File2ColorMap ( *cmptr, cmapFilename )) == OK )
            cmapPtr = cmptr ;
         else
         {
            delete cmptr ;
            clArgs.altColorMap = false ;
         }
      }
   }

   //* If test eleven(11) specified, then do the test BEFORE *
   //* starting the main application. Exercises the          *
   //* NCurses-class methods OutputStreamScrolling() and     *
   //* WriteStream(). See StreamTest.cpp.                    *
   if ( testNum == indexCOUNT )
   {
      Test11 () ;
      testNum = (-1) ;     // start main application with menu
   }

   //*******************************
   //** Initialize NCurses Engine **
   //*******************************
   //* NCurses class has been previously instantiated at the time the          *
   //* application was loaded, and is available through the name: 'nc'.        *
   //* (see note in GlobalDef.hpp)                                             *
   if( (nc.StartNCursesEngine ( startInColor, cmapPtr )) == OK )
   {
      //* Verify that the current locale supports UTF-8 character encoding.    *
      short localeSetOk = nc.VerifyLocale () ;

      //* If user has specified an alternate locale for interpretation of      *
      //* character output and input, it must be passed to the NCurses class   *
      //* before any I/O occurs.                                               *
      //* (NOTE: If clArgs.localeName does not specify a locale supported by ) *
      //* (      the terminal program, OR the locale specified is not a      ) *
      //* (      UTF-8-encoded locale, OR the default encoding specified in  ) *
      //* (      the environment was not UTF-8, then the results will be     ) *
      //* (      ambiguous at best.                                          ) *
      short altLocaleOk = OK ;
      if ( clArgs.altLocale != false )
         altLocaleOk = nc.SetLocale ( clArgs.localeName ) ;

      //* Even though no windows have been opened yet, you can think of the    *
      //* entire screen as a big window which can be written to and which      *
      //* is the parent/grandparent of all windows and NcDialog objects that   *
      //* will be opened. (cursor state: nccvINVISIBLE and key state: nckpRAW) *
      ClearTerminalWindow () ;               // display application title

      //* Initialize and open the application's status window *
      sWin = new NcWindow ( swLINES, swCOLS, swOFFy, swOFFx ) ;
      if ( (sWin->OpenWindow ()) != ERR )
      {
         bool  done = false ;    // loop control

         //* Report any error in initializing color output*
         if ( !(nc.ColorText_Initialized ()) )
         {
            if ( ! nc.ColorText_Available () )
               StatWin ( "This terminal does not support multi-color output..." ) ;
            else
               StatWin ( "Initialization of multi-color output failed..." ) ;
            sleep ( 4 ) ;
         }
         //* Report any error in setting default or alternate locale.  *
         //* (Even if error, ASCII-only I/O SHOULD cause no problems.) *
         if ( localeSetOk != OK || altLocaleOk != OK )
         {
            if ( localeSetOk != OK )
               StatWin ( "Current locale setting does not support UTF-8 character encoding..." ) ;
            else
               StatWin ( "Locale setting specified on command line is not valid..." ) ;
            sleep ( 5 ) ;
         }

         //* Refresh main window *
         nc.RefreshScreen () ;
         nc.ScreenDimensions ( termRows, termCols ) ;   // get terminal-window size
         if ( termRows < minTERMROWS || termCols < minTERMCOLS )
         {
            //* We need room to play *
            winTooSmall = done = true ;
         }

         //* Find out what the user wants *
         while ( ! done )
         {
            //* If command-line agrument for test number provided, use it.     *
            //* Else, open a dialog so user can choose which test(s) to run.   *
            if ( testNum < ZERO )
               testNum = TestMenu ( termRows / 2, termCols / 2 ) ;

            //* Check whether user has re-sized the terminal window *
            nc.ScreenDimensions ( termRows, termCols ) ;
            if ( termRows < minTERMROWS || termCols < minTERMCOLS )
            {
               winTooSmall = done = true ;
               continue ;
            }

            //* Execute selected test *            
            switch ( testNum )
            {
               case indexT01:          // basic file display via dctSCROLLEXT control
                  Test01 () ;
                  break ;
               case indexT02:          // test the ConnectControl2Border() method
                  Test02 () ;
                  break ;
               case indexT03:          // test DialogSpinner controls
                  Test03 () ;
                  break ;
               case indexT04:          // Test color mapping
                  Test04 () ;
                  break ;
               case indexT05:          // Test key-input stream
                  Test05 () ;
                  break ;
               case indexT06:          // Display Text Color Mapping
                  Test06 () ;
                  break ;
               case indexT07:          // Capture Display Data to Text or HTML
                  Test07 () ;
                  break ;
               case indexT08:          // test line-drawing methods
                  Test08 () ;
                  break ;
               case indexT09:          // display of alternate character set (NCurses)
                  Test09 () ;
                  break ;
               case indexT10:          // display of alternate character set (NcDialog)
                  Test10 () ;
                  break ;
               case indexEXIT:         // exit the application
                  StatWin ( " Exiting Application . . . ", nc.blG ) ;
                  done = true ;
                  break ;
               case indexHELP:         // help-about dialog
               default:
                  HelpAbout ( termRows / 2 + 1, termCols / 2 ) ;
                  break ;
            }
            testNum = -1 ;    // force use of the menu
         }        // while()
      }
      else
      {
         winTooSmall = true ;
      }

      //****************************
      //* Shut down ncurses engine *
      //****************************
      delete sWin ;                          // release status line resources       
      nc.RefreshScreen () ;                  // refresh the main window
      nc.StopNCursesEngine () ;              // Deactivate the NCurses engine

      //* Most likely cause of failure in opening the background *
      //* window is that the terminal window is too small.       *
      //* Memory allocation error is also a possibility.         *
      if ( winTooSmall != false )      
         cout << "\n ******************************************"
                 "\n * ERROR! Unwilling to open main window.  *"
                 "\n * Terminal window must be >= 80x25 chars.*"
                 "\n ******************************************\n\n" ;
   }
   else
   {
      cout << "\n **********************************************"
              "\n * ERROR! Unable to initialize NCurses engine.*"
              "\n **********************************************\n\n" ;
   }

   return ZERO ;

}  //* End main() *

//*************************
//*       TestMenu        *
//*************************
//******************************************************************************
//* Open a dialog and display a list of test choices.                          *
//*                                                                            *
//*                                                                            *
//* Input  : ctrY: Y center position for dialog window                         *
//*          ctrX: X center position for dialog window                         *
//*                                                                            *
//* Returns: member of cIndices                                                *
//******************************************************************************

short    TestMenu ( short ctrY, short ctrX )
{
static const short dialogROWS = 19 ;         // display lines
static const short dialogCOLS = 57 ;         // display columns
static const char* testNames[] = 
{// 1234567890123456789012345678901234567890123456

   "Test #1: ^FileMangler Scrolling and Menu Bar  ",
   "Test #2: Control-to-Dialog ^Border Test       ",
   "Test #3: ^Spinner Controls                    ",
   "Test #4: Adjusting the ^Text-color Map        ",
   "Test #5: Keyboard Input and ^Key Translation  ",
   "Test #6: Display-Text ^Color Mapping          ",
   "Test #7: Ca^pture Display Data to Text or HTML",
   "Test #8: Line-Dra^wing Methods                ",
   "Test #9: A^lternate Character Set, NCurses    ",
   "Test #10: ^Alternate Character Set, NcDialog  ",
} ;
static NcDialog*  dp ;                       // dialog window pointer
static bool    dialogOpen = false ;          // set to true when dialog opened
short    ulY     = ctrY - dialogROWS / 2,    // upper left corner in Y
         ulX     = ctrX - dialogCOLS / 2 ;   // upper left corner in X
attr_t   dColor   = nc.blR,                  // dialog's base color
         rnColor = nc.bw,                    // radio button non-focus color
         rfColor = nc.cyG ;                  // radio button focus color
short    testNum ;

   StatWin ( "Opening Test-Selection Menu" ) ; // keep user up-to-date

   //* Instantiate the dialog window and install its control objects *
   if ( dialogOpen == false )
   {
      //****************************************
      //* Initial parameters for dialog window *
      //****************************************
      InitCtrl ic[indexCOUNT] =           // control initialization structures
      {
         {  //* TEST01 radiobutton  - - - - - - - - - - - - - - - -   indexT01 *
            dctRADIOBUTTON,               // type:      
            rbtS3s,                       // rbSubtype: standard, 3 chars wide
            false,                        // rbSelect:  default selection
            2,                            // ulY:       upper left corner in Y
            4,                            // ulX:       upper left corner in X
            1,                            // lines:     (n/a)
            0,                            // cols:      (n/a)
            NULL,                         // dispText:  (n/a)
            rnColor,                      // nColor:    non-focus color
            rfColor,                      // fColor:    focus color
            tbPrint,                      // filter:    (n/a)
            testNames[0],                 // label:
            ZERO,                         // labY:      
            5,                            // labX       
            ddBoxTYPES,                   // exType:    (n/a)
            1,                            // scrItems:  (n/a)
            1,                            // scrSel:    (n/a)
            NULL,                         // scrColor:  (n/a)
            NULL,                         // spinData:  (n/a)
            true,                         // active:    allow control to gain focus
            &ic[1]                        // nextCtrl:  link in next structure
         },
         {  //* TEST02 radiobutton  - - - - - - - - - - - - - - - -   indexT02 *
            dctRADIOBUTTON,               // type:      
            rbtS3s,                       // rbSubtype: standard, 3 chars wide
            false,                        // rbSelect:  default selection
            3,                            // ulY:       upper left corner in Y
            4,                            // ulX:       upper left corner in X
            1,                            // lines:     (n/a)
            0,                            // cols:      (n/a)
            NULL,                         // dispText:  (n/a)
            rnColor,                      // nColor:    non-focus color
            rfColor,                      // fColor:    focus color
            tbPrint,                      // filter:    (n/a)
            testNames[1],                 // label:
            ZERO,                         // labY:      
            5,                            // labX       
            ddBoxTYPES,                   // exType:    (n/a)
            1,                            // scrItems:  (n/a)
            1,                            // scrSel:    (n/a)
            NULL,                         // scrColor:  (n/a)
            NULL,                         // spinData:  (n/a)
            true,                         // active:    allow control to gain focus
            &ic[2]                        // nextCtrl:  link in next structure
         },
         {  //* TEST03 radiobutton  - - - - - - - - - - - - - - - -   indexT03 *
            dctRADIOBUTTON,               // type:      
            rbtS3s,                       // rbSubtype: standard, 3 chars wide
            false,                        // rbSelect:  default selection
            4,                            // ulY:       upper left corner in Y
            4,                            // ulX:       upper left corner in X
            1,                            // lines:     (n/a)
            0,                            // cols:      (n/a)
            NULL,                         // dispText:  (n/a)
            rnColor,                      // nColor:    non-focus color
            rfColor,                      // fColor:    focus color
            tbPrint,                      // filter:    (n/a)
            testNames[2],                 // label:
            ZERO,                         // labY:      
            5,                            // labX       
            ddBoxTYPES,                   // exType:    (n/a)
            1,                            // scrItems:  (n/a)
            1,                            // scrSel:    (n/a)
            NULL,                         // scrColor:  (n/a)
            NULL,                         // spinData:  (n/a)
            true,                         // active:    allow control to gain focus
            &ic[3]                        // nextCtrl:  link in next structure
         },
         {  //* TEST04 radiobutton  - - - - - - - - - - - - - - - -   indexT04 *
            dctRADIOBUTTON,               // type:      
            rbtS3s,                       // rbSubtype: standard, 3 chars wide
            false,                        // rbSelect:  default selection
            5,                            // ulY:       upper left corner in Y
            4,                            // ulX:       upper left corner in X
            1,                            // lines:     (n/a)
            0,                            // cols:      (n/a)
            NULL,                         // dispText:  (n/a)
            rnColor,                      // nColor:    non-focus color
            rfColor,                      // fColor:    focus color
            tbPrint,                      // filter:    (n/a)
            testNames[3],                 // label:
            ZERO,                         // labY:      
            5,                            // labX       
            ddBoxTYPES,                   // exType:    (n/a)
            1,                            // scrItems:  (n/a)
            1,                            // scrSel:    (n/a)
            NULL,                         // scrColor:  (n/a)
            NULL,                         // spinData:  (n/a)
            true,                         // active:    allow control to gain focus
            &ic[4]                        // nextCtrl:  link in next structure
         },
         {  //* TEST05 radiobutton  - - - - - - - - - - - - - - - -   indexT05 *
            dctRADIOBUTTON,               // type:      
            rbtS3s,                       // rbSubtype: standard, 3 chars wide
            false,                        // rbSelect:  default selection
            6,                            // ulY:       upper left corner in Y
            4,                            // ulX:       upper left corner in X
            1,                            // lines:     (n/a)
            0,                            // cols:      (n/a)
            NULL,                         // dispText:  (n/a)
            rnColor,                      // nColor:    non-focus color
            rfColor,                      // fColor:    focus color
            tbPrint,                      // filter:    (n/a)
            testNames[4],                 // label:
            ZERO,                         // labY:      
            5,                            // labX       
            ddBoxTYPES,                   // exType:    (n/a)
            1,                            // scrItems:  (n/a)
            1,                            // scrSel:    (n/a)
            NULL,                         // scrColor:  (n/a)
            NULL,                         // spinData:  (n/a)
            true,                         // active:    allow control to gain focus
            &ic[5]                        // nextCtrl:  link in next structure
         },
         {  //* TEST06 radiobutton  - - - - - - - - - - - - - - - -   indexT06 *
            dctRADIOBUTTON,               // type:      
            rbtS3s,                       // rbSubtype: standard, 3 chars wide
            false,                        // rbSelect:  default selection
            7,                            // ulY:       upper left corner in Y
            4,                            // ulX:       upper left corner in X
            1,                            // lines:     (n/a)
            0,                            // cols:      (n/a)
            NULL,                         // dispText:  (n/a)
            rnColor,                      // nColor:    non-focus color
            rfColor,                      // fColor:    focus color
            tbPrint,                      // filter:    (n/a)
            testNames[5],                 // label:
            ZERO,                         // labY:      
            5,                            // labX       
            ddBoxTYPES,                   // exType:    (n/a)
            1,                            // scrItems:  (n/a)
            1,                            // scrSel:    (n/a)
            NULL,                         // scrColor:  (n/a)
            NULL,                         // spinData:  (n/a)
            true,                         // active:    allow control to gain focus
            &ic[6]                        // nextCtrl:  link in next structure
         },
         {  //* TEST07 radiobutton  - - - - - - - - - - - - - - - -   indexT07 *
            dctRADIOBUTTON,               // type:      
            rbtS3s,                       // rbSubtype: standard, 3 chars wide
            false,                        // rbSelect:  default selection
            8,                            // ulY:       upper left corner in Y
            4,                            // ulX:       upper left corner in X
            1,                            // lines:     (n/a)
            0,                            // cols:      (n/a)
            NULL,                         // dispText:  (n/a)
            rnColor,                      // nColor:    non-focus color
            rfColor,                      // fColor:    focus color
            tbPrint,                      // filter:    (n/a)
            testNames[6],                 // label:
            ZERO,                         // labY:      
            5,                            // labX       
            ddBoxTYPES,                   // exType:    (n/a)
            1,                            // scrItems:  (n/a)
            1,                            // scrSel:    (n/a)
            NULL,                         // scrColor:  (n/a)
            NULL,                         // spinData:  (n/a)
            true,                         // active:    allow control to gain focus
            &ic[7]                        // nextCtrl:  link in next structure
         },
         {  //* TEST08 radiobutton  - - - - - - - - - - - - - - - -   indexT08 *
            dctRADIOBUTTON,               // type:      
            rbtS3s,                       // rbSubtype: standard, 3 chars wide
            false,                        // rbSelect:  default selection
            9,                            // ulY:       upper left corner in Y
            4,                            // ulX:       upper left corner in X
            1,                            // lines:     (n/a)
            0,                            // cols:      (n/a)
            NULL,                         // dispText:  (n/a)
            rnColor,                      // nColor:    non-focus color
            rfColor,                      // fColor:    focus color
            tbPrint,                      // filter:    (n/a)
            testNames[7],                 // label:
            ZERO,                         // labY:      
            5,                            // labX       
            ddBoxTYPES,                   // exType:    (n/a)
            1,                            // scrItems:  (n/a)
            1,                            // scrSel:    (n/a)
            NULL,                         // scrColor:  (n/a)
            NULL,                         // spinData:  (n/a)
            true,                         // active:    allow control to gain focus
            &ic[8]                        // nextCtrl:  link in next structure
         },
         {  //* TEST09 radiobutton  - - - - - - - - - - - - - - - -   indexT09 *
            dctRADIOBUTTON,               // type:      
            rbtS3s,                       // rbSubtype: standard, 3 chars wide
            false,                        // rbSelect:  default selection
            10,                           // ulY:       upper left corner in Y
            4,                            // ulX:       upper left corner in X
            1,                            // lines:     (n/a)
            0,                            // cols:      (n/a)
            NULL,                         // dispText:  (n/a)
            rnColor,                      // nColor:    non-focus color
            rfColor,                      // fColor:    focus color
            tbPrint,                      // filter:    (n/a)
            testNames[8],                 // label:
            ZERO,                         // labY:      
            5,                            // labX       
            ddBoxTYPES,                   // exType:    (n/a)
            1,                            // scrItems:  (n/a)
            1,                            // scrSel:    (n/a)
            NULL,                         // scrColor:  (n/a)
            NULL,                         // spinData:  (n/a)
            true,                         // active:    allow control to gain focus
            &ic[9]                        // nextCtrl:  link in next structure
         },
         {  //* TEST10 radiobutton  - - - - - - - - - - - - - - - -   indexT10 *
            dctRADIOBUTTON,               // type:      
            rbtS3s,                       // rbSubtype: standard, 3 chars wide
            false,                        // rbSelect:  default selection
            11,                           // ulY:       upper left corner in Y
            4,                            // ulX:       upper left corner in X
            1,                            // lines:     (n/a)
            0,                            // cols:      (n/a)
            NULL,                         // dispText:  (n/a)
            rnColor,                      // nColor:    non-focus color
            rfColor,                      // fColor:    focus color
            tbPrint,                      // filter:    (n/a)
            testNames[9],                 // label:
            ZERO,                         // labY:      
            5,                            // labX       
            ddBoxTYPES,                   // exType:    (n/a)
            1,                            // scrItems:  (n/a)
            1,                            // scrSel:    (n/a)
            NULL,                         // scrColor:  (n/a)
            NULL,                         // spinData:  (n/a)
            true,                         // active:    allow control to gain focus
            &ic[10]                       // nextCtrl:  link in next structure
         },
// INSERT ANY ADDITIONAL TEST SELECTORS HERE
         {  //* HELP radiobutton    - - - - - - - - - - - - - - - -  indexHELP *
            dctRADIOBUTTON,               // type:      
            rbtS3s,                       // rbSubtype: standard, 3 chars wide
            true,                         // rbSelect:  default selection
            12,                           // ulY:       upper left corner in Y
            4,                            // ulX:       upper left corner in X
            1,                            // lines:     (n/a)
            0,                            // cols:      (n/a)
            NULL,                         // dispText:  (n/a)
            rnColor,                      // nColor:    non-focus color
            rfColor,                      // fColor:    focus color
            tbPrint,                      // filter:    (n/a)
            "^Help About dialog",         // label:
            ZERO,                         // labY:      
            5,                            // labX       
            ddBoxTYPES,                   // exType:    (n/a)
            1,                            // scrItems:  (n/a)
            1,                            // scrSel:    (n/a)
            NULL,                         // scrColor:  (n/a)
            NULL,                         // spinData:  (n/a)
            true,                         // active:    allow control to gain focus
            &ic[11]                       // nextCtrl:  link in next structure
         },
         {  //* EXIT radiobutton    - - - - - - - - - - - - - - - -  indexEXIT *
            dctRADIOBUTTON,               // type:      
            rbtS3s,                       // rbSubtype: standard, 3 chars wide
            false,                        // rbSelect:  default selection
            13,                           // ulY:       upper left corner in Y
            4,                            // ulX:       upper left corner in X
            1,                            // lines:     (n/a)
            0,                            // cols:      (n/a)
            NULL,                         // dispText:  (n/a)
            rnColor,                      // nColor:    non-focus color
            rfColor,                      // fColor:    focus color
            tbPrint,                      // filter:    (n/a)
            "E^xit",                      // label:
            ZERO,                         // labY:      
            5,                            // labX       
            ddBoxTYPES,                   // exType:    (n/a)
            1,                            // scrItems:  (n/a)
            1,                            // scrSel:    (n/a)
            NULL,                         // scrColor:  (n/a)
            NULL,                         // spinData:  (n/a)
            true,                         // active:    allow control to gain focus
            NULL                          // nextCtrl:  link in next structure
         },
      } ;

      //* Initial parameters for dialog window *
      InitNcDialog dInit( dialogROWS,     // number of display lines
                          dialogCOLS,     // number of display columns
                          ulY,            // Y offset from upper-left of terminal 
                          ulX,            // X offset from upper-left of terminal
                          "  Select Test To Be Executed  ", // dialog title
                          ncltSINGLE,     // border line-style
                          dColor,         // border color attribute
                          dColor,         // interior color attribute
                          ic              // pointer to list of control definitions
                        ) ;

      //* Instantiate the dialog window *
      dp = new NcDialog ( dInit ) ;
   
      //* Open the dialog window *
      if ( (dp->OpenWindow()) == OK )
      {
         dialogOpen = true ;              // remember that dialog is open
         
         dp->DrawBox ( 1,  2, dialogROWS-5, 53, dColor ) ;  // enclose radio-button group
         //* Display instructions *
         gString gsOut( "Use Tab and Shift+Tab or arrow keys to scroll through\n"
                        "options, then Space or Enter key to select;\n"
                        "or select a test via 'hotkey' (underlined character)" ) ;
         dp->WriteParagraph ( dialogROWS-4, 2, gsOut, dColor ) ;
         dp->RefreshWin () ;
   
         //* Group all radio buttons in the dialog *
         short XorGroup[] = { indexT01, indexT02, indexT03, indexT04, indexT05, 
                              indexT06, indexT07, indexT08, indexT09, indexT10, 
                              indexHELP, indexEXIT, -1 } ;
         dp->GroupRadiobuttons ( XorGroup ) ;
      }
   }           // if(dialogOpen==false)
   
   //* Un-hide the existing dialog window *
   else
   {
      dp->RefreshWin () ;
   }

   if ( dialogOpen != false )
   {
      //* Give the focus to the currently 'selected' button *
      short sIndex = dp->GetRbGroupSelection ( indexEXIT ) ;
      short cIndex = dp->GetCurrControl () ;
      while ( cIndex != sIndex )
         cIndex = dp->PrevControl () ;

      uiInfo Info ;                       // user interface info
//      short  icIndex = ZERO ;             // index into ic[] array
      bool   testSelected = false ;       // loop control
      while ( !testSelected )
      {
         //* Move among the members of this Radio Button group *
         //* allowing user to select a test to be executed.    *
         // NOTE: This interface loop is fairly simple because *
         //       all controls in this dialog are Radio Buttons*
         //       and they all belong to the same button group.*
         dp->EditRadiobutton ( Info ) ;
         
         //* If a new selection has been made, OR  *
         //* if previous selection was re-selected.*
         if ( Info.dataMod || Info.keyIn == nckENTER )
            testSelected = true ;
      }
      testNum = Info.selMember ; // return value
   }

   //* If we are about to exit application, close the dialog. *
   if ( testNum == indexEXIT )
   {
      if ( dp != NULL )                      // close the window
         delete ( dp ) ;
      dp = NULL ;
   }
   //* Else, hide the window for future use *
   else
   {
      dp->HideWin () ;
   }

   return testNum ;

}  //* End TestMenu() *

//*************************
//*       Test01          *
//*************************
//******************************************************************************
//* Demonstrate a complex dialog application featuring a dctSCROLLEXT-type     *
//* control and several dctMENUWIN-type controls grouped to create a MenuBar.  *
//*                                                                            *
//* This test is located in the FileMangler_Subset.cpp module.                 *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void Test01 ( void )
{
   StatWin ( "Opening Test 1 - FileMangler Scrolling and Menu Bar : Press F2 for Menu" ) ;

   short neededLines = minTERMROWS + minTestULY, 
         neededCols  = minTERMCOLS ;
   if ( (termRows >= neededLines) && (termCols >= neededCols) )
   {
      short tRows, tCols ;
      nc.ScreenDimensions ( tRows, tCols ) ;
      MenuTest* menuTest = new MenuTest ( tRows, tCols, minTestULY ) ;
      if ( (menuTest->mtDialogOpened ()) != false )
      {
         menuTest->mtInteract () ;
      }
      else { /* Memory allocation error - unlikely */ }

      if ( menuTest != NULL )   // close the dialog
         delete menuTest ;
   }
   else
   {  //* Most likely cause of dialog not opening is that the terminal window  *
      //* is too small. Give user a clue...                                    *
      gString gs ;
      gs.compose ( L" Sorry, size of terminal window must "
                    "be at least %hd x %hd for this test. ", 
                   &neededCols, &neededLines ) ;
      StatWin ( gs.gstr(), nc.cyG ) ;
      sleep ( 5 ) ;
   }

}  //* End Test01() *

//*************************
//*  T01_ControlUpdate    *
//*************************
//******************************************************************************
//* This is a callback method for manually updating the controls in the        *
//* Test01() dialog.                                                           *
//*                                                                            *
//*  For this test, the following are updated by the callback method:          *
//*   1.                                                                       *
//*   2.                                                                       *
//*   3.                                                                       *
//*                                                                            *
//* Input  : currIndex: index of control that currently has focus              *
//*          wkey     : user's key input data                                  *
//*          firstTime: the EstablishCallback() method calls this method once  *
//*                     with firstTime==true, to perform any required          *
//*                     initialization. Subsequently, the NcDialog class       *
//*                     always calls with firstTime==false.                    *
//* Returns: OK                                                                *
//******************************************************************************
//* Important Note: This method makes some intimate assumptions about what     *
//* is happening in the method that established the callback. If changes are   *
//* made in the establishing method that affect the callback functionality,    *
//* be sure to update this method to address those changes.                    *
//*                                                                            *
//* Programmer's Note: See dialog control index constants at file scope above. *
//******************************************************************************

short T01_ControlUpdate ( const short currIndex, const wkeyCode wkey, bool firstTime )
{

   return OK ;

}  //* End T01_ControlUpdate() *

//*************************
//*       Test02          *
//*************************
//******************************************************************************
//* Test the NcDialog method, ConnectControl2Border().                         *
//*                                                                            *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void Test02 ( void )
{
const short dialogROWS = 35 ;       // display lines
const short dialogCOLS = 99 ;       // display columns
const short pcontrolsDEFINED = 3 ;  // # of controls in prequel dialog
const short controlsDEFINED = 10 ;  // # of controls in main dialog
const short DD_WIDTH = 15 ;         // display-data width
const short DD_ITEMS = 8 ;          // display-data items
char dispData[DD_ITEMS][DD_WIDTH] = 
{
   "Test Data #000",
   "Test Data #001",
   "Test Data #002",
   "Test Data #003",
   "Test Data #004",
   "Test Data #005",
   "Test Data #006",
   "Test Data #007",
} ;
const char* dispDataPtr[DD_ITEMS] = 
{
   dispData[0], dispData[1], dispData[2], dispData[3], 
   dispData[4], dispData[5], dispData[6], dispData[7],  
} ;
attr_t dispAttr[DD_ITEMS + 1] = 
{
   0x00000000, 0x00000100, 0x00000200, 0x00000300, 
   0x00000400, 0x00000500, 0x00000600, 0x00000700, 0x00000000
} ;
short    icIndex = ZERO ;                    // index into ic[] array
attr_t   dColor = nc.cyR ;
short    ctrY    = termRows/2,               // dialog center in Y
         ctrX    = termCols/2,               // dialog center in X
         ulY     = ctrY - dialogROWS/2,      // upper left corner in Y
         ulX     = ctrX - dialogCOLS/2 ;     // upper left corner in X
bool     scrollBoxControls = false,
         cancelTest = false ;

   StatWin ( "Opening Test 02 - Test ConnectControl2Border() method" ) ;

   //*************************************************
   //* Open a dialog to ask whether we are testing   *
   //* dctSCROLLBOX controls or dctSCROLLEXT controls*
   //*************************************************
   static const short PREQUEL_LINES = 7 ;    // display lines
   static const short PREQUEL_COLS = 36 ;    // display columns
   InitCtrl icp[pcontrolsDEFINED] =          // control initialization structures
   {
      {  //* dctSCROLLEXT pushbutton - - - - - - index 0 *
         dctPUSHBUTTON,                // type:      
         rbtTYPES,                     // rbSubtype: (n/a)
         false,                        // rbSelect:  (n/a)
         2,                            // ulY:       upper left corner in Y
         3,                            // ulX:       upper left corner in X
         1,                            // lines:     control lines
         14,                           // cols:      control columns
         " dctSCROLLEXT ",             // dispText:  
         nc.gyR,                       // nColor:    non-focus color
         nc.reG,                       // fColor:    focus color
         tbPrint,                      // filter:    (n/a)
         "",                           // label:     (n/a)
         ZERO,                         // labY:      (n/a)
         ZERO,                         // labX       (n/a)
         ddBoxTYPES,                   // exType:    (n/a)
         1,                            // scrItems:  (n/a)
         1,                            // scrSel:    (n/a)
         NULL,                         // scrColor:  (n/a)
         NULL,                         // spinData:  (n/a)
         true,                         // active:    allow control to gain focus
         &icp[1]                       // nextCtrl:  link in next structure
      },
      {  //* dctSCROLLBOX pushbutton - - - - - - index 1 *
         dctPUSHBUTTON,                // type:      
         rbtTYPES,                     // rbSubtype: (n/a)
         false,                        // rbSelect:  (n/a)
         2,                            // ulY:       upper left corner in Y
         19,                           // ulX:       upper left corner in X
         1,                            // lines:     control lines
         14,                           // cols:      control columns
         " dctSCROLLBOX ",             // dispText:  
         nc.gyR,                       // nColor:    non-focus color
         nc.reG,                       // fColor:    focus color
         tbPrint,                      // filter:    (n/a)
         "",                           // label:     (n/a)
         ZERO,                         // labY:      (n/a)
         ZERO,                         // labX       (n/a)
         ddBoxTYPES,                   // exType:    (n/a)
         1,                            // scrItems:  (n/a)
         1,                            // scrSel:    (n/a)
         NULL,                         // scrColor:  (n/a)
         NULL,                         // spinData:  (n/a)
         true,                         // active:    allow control to gain focus
         &icp[2]                       // nextCtrl:  link in next structure
      },
      {  //* 'CANCEL' pushbutton - - - - - - - - index 2 *
         dctPUSHBUTTON,                // type:      
         rbtTYPES,                     // rbSubtype: (n/a)
         false,                        // rbSelect:  (n/a)
         PREQUEL_LINES-2,              // ulY:       upper left corner in Y
         PREQUEL_COLS/2 - 4,           // ulX:       upper left corner in X
         1,                            // lines:     control lines
         8,                            // cols:      control columns
         " CANCEL ",                   // dispText:  
         nc.gyR,                       // nColor:    non-focus color
         nc.reG,                       // fColor:    focus color
         tbPrint,                      // filter:    (n/a)
         "",                           // label:     (n/a)
         ZERO,                         // labY:      (n/a)
         ZERO,                         // labX       (n/a)
         ddBoxTYPES,                   // exType:    (n/a)
         1,                            // scrItems:  (n/a)
         1,                            // scrSel:    (n/a)
         NULL,                         // scrColor:  (n/a)
         NULL,                         // spinData:  (n/a)
         true,                         // active:    allow control to gain focus
         NULL,                         // nextCtrl:  link in next structure
      },
   } ;

   //* Initial parameters for dialog window *
   InitNcDialog pdInit( PREQUEL_LINES,  // number of display lines
                        PREQUEL_COLS,   // number of display columns
                        short(ctrY-PREQUEL_LINES/2), // Y offset from upper-left 
                        short(ctrX-PREQUEL_COLS/2),  // X offset from upper-left
                        "  Select Control Type  ", // dialog title
                        ncltSINGLE,     // border line-style
                        dColor,         // border color attribute
                        dColor,         // interior color attribute
                        icp             // pointer to list of control definitions
                      ) ;

   //* Instantiate the dialog window *
   NcDialog* dp = new NcDialog ( pdInit ) ;

   //* Open the dialog window *
   if ( (dp->OpenWindow()) == OK )
   {
      dp->RefreshWin () ;

      uiInfo Info ;                 // user interface data returned here
      bool   done = false ;         // loop control
      while ( ! done )
      {
         //*******************************************
         //* If focus is currently on a Pushbutton   *
         //*******************************************
         if ( icp[icIndex].type == dctPUSHBUTTON )
         {
            icIndex = dp->EditPushbutton ( Info ) ;

            //* If a Pushbutton was pressed *
            if ( Info.ctrlType == dctPUSHBUTTON && Info.dataMod != false )
            {
               switch ( Info.ctrlIndex )
               {
                  case ZERO:        // dctSCROLLEXT controls
                     scrollBoxControls = false ;
                     break ;
                  case 1:           // dctSCROLLBOX controls
                     scrollBoxControls = true ;
                     break ;
                  case 2:           // cancel test
                     cancelTest = true ;
                     break ;
               }
               done = true ;
            }
            else
            {
               // No button press, so nothing to do
            }
         }
         if ( done == false )
         {
            if ( Info.keyIn == nckSTAB )
               icIndex = dp->PrevControl () ; 
            else
               icIndex = dp->NextControl () ;
         }
      }  // while()
   }  // OpenWindow()
   else
      StatWin ( "Unable to open the dialog !!" ) ;
   if ( dp != NULL )                         // close the window
   {
      delete ( dp ) ;
      dp = NULL ;
   }
   // NOTE: It is terrible programming practice to have multiple
   //       exit points from any function, but it's only a test
   if ( cancelTest != false )
      return ;

   //**********************
   //** MAIN TEST DIALOG **
   //**********************
icIndex = ZERO ;
InitCtrl    ic[controlsDEFINED] = 
{
   {  //* 'OK' pushbutton - - - - - - - - - - - - - - - - -index 0 *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      dialogROWS/2,                 // ulY:       upper left corner in Y
      dialogCOLS/2 - 3,             // ulX:       upper left corner in X
      1,                            // lines:     control lines
      6,                            // cols:      control columns
      "  OK  ",                     // dispText:  
      nc.gyR,                       // nColor:    non-focus color
      nc.reG,                       // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "",                           // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[1],                       // nextCtrl:  link in next structure
   },
   {  //* Control in upper left corner of dialog - - - - - index 1 *
      dctSCROLLEXT,                 // type:      define a scrolling-data control
      rbtTYPES,                     // rbSubtype: (na)
      false,                        // rbSelect:  (n/a)
      ZERO,                         // ulY:       upper left corner in Y
      ZERO,                         // ulX:       upper left corner in X
      DD_ITEMS + 2,                 // lines:     control lines
      DD_WIDTH + 1,                 // cols:      control columns
      (char*)&dispDataPtr,          // dispText:  pointer to text data
      dColor,                       // nColor:    non-focus border color
      dColor,                       // fColor:    focus border color
      tbPrint,                      // filter:    (n/a)
      "Control #1",                 // label:     
      10,                           // labY:      offset from control's ulY
      3,                            // labX       offset from control's ulX
      ddBoxTYPES,                   // exType:    (n/a)
      DD_ITEMS,                     // scrItems:  display item count
      ZERO,                         // scrSel:    highlight index
      dispAttr,                     // scrColor:  pointer to color attrib. array
      NULL,                         // spinData:  (n/a)
      false,                        // active:    view-only
      &ic[2],                       // nextCtrl:  link in next structure
   },
   {  //* Control in center left corner of dialog - - - - -index 2 *
      dctSCROLLEXT,                 // type:      define a scrolling-data control
      rbtTYPES,                     // rbSubtype: (na)
      false,                        // rbSelect:  (n/a)
      12,                           // ulY:       upper left corner in Y
      ZERO,                         // ulX:       upper left corner in X
      DD_ITEMS + 2,                 // lines:     control lines
      DD_WIDTH + 1,                 // cols:      control columns
      (char*)&dispDataPtr,          // dispText:  pointer to text data
      dColor,                       // nColor:    non-focus border color
      dColor,                       // fColor:    focus border color
      tbPrint,                      // filter:    (n/a)
      "Control #2",                 // label:     
      10,                           // labY:      offset from control's ulY
      3,                            // labX       offset from control's ulX
      ddBoxTYPES,                   // exType:    (n/a)
      DD_ITEMS,                     // scrItems:  display item count
      ZERO,                         // scrSel:    highlight index
      dispAttr,                     // scrColor:  pointer to color attrib. array
      NULL,                         // spinData:  (n/a)
      false,                        // active:    view-only
      &ic[3],                       // nextCtrl:  link in next structure
   },
   {  //* Control in lower left corner of dialog - - - - - index 3 *
      dctSCROLLEXT,                 // type:      define a scrolling-data control
      rbtTYPES,                     // rbSubtype: (na)
      false,                        // rbSelect:  (n/a)
      25,                           // ulY:       upper left corner in Y
      ZERO,                         // ulX:       upper left corner in X
      DD_ITEMS + 2,                 // lines:     control lines
      DD_WIDTH + 1,                 // cols:      control columns
      (char*)&dispDataPtr,          // dispText:  pointer to text data
      dColor,                       // nColor:    non-focus border color
      dColor,                       // fColor:    focus border color
      tbPrint,                      // filter:    (n/a)
      "Control #3",                 // label:     
      -1,                           // labY:      offset from control's ulY
      3,                            // labX       offset from control's ulX
      ddBoxTYPES,                   // exType:    (n/a)
      DD_ITEMS,                     // scrItems:  display item count
      ZERO,                         // scrSel:    highlight index
      dispAttr,                     // scrColor:  pointer to color attrib. array
      NULL,                         // spinData:  (n/a)
      false,                        // active:    view-only
      &ic[4],                       // nextCtrl:  link in next structure
   },
   {  //* Control in left-center (full length) - - - - - - index 4 *
      dctSCROLLEXT,                 // type:      define a scrolling-data control
      rbtTYPES,                     // rbSubtype: (na)
      false,                        // rbSelect:  (n/a)
      ZERO,                         // ulY:       upper left corner in Y
      19,                           // ulX:       upper left corner in X
      dialogROWS,                   // lines:     control lines
      DD_WIDTH + 1,                 // cols:      control columns
      (char*)&dispDataPtr,          // dispText:  pointer to text data
      dColor,                       // nColor:    non-focus border color
      dColor,                       // fColor:    focus border color
      tbPrint,                      // filter:    (n/a)
      "Control #4",                 // label:     
      2,                            // labY:      offset from control's ulY
      DD_WIDTH + 3,                 // labX       offset from control's ulX
      ddBoxTYPES,                   // exType:    (n/a)
      DD_ITEMS,                     // scrItems:  display item count
      ZERO,                         // scrSel:    highlight index
      dispAttr,                     // scrColor:  pointer to color attrib. array
      NULL,                         // spinData:  (n/a)
      false,                        // active:    view-only
      &ic[5],                       // nextCtrl:  link in next structure
   },
   {  //* Control in right-center, top - - - - - - - - - - index 5 *
      dctSCROLLEXT,                 // type:      define a scrolling-data control
      rbtTYPES,                     // rbSubtype: (na)
      false,                        // rbSelect:  (n/a)
      ZERO,                         // ulY:       upper left corner in Y
      64,                           // ulX:       upper left corner in X
      DD_ITEMS + 2,                 // lines:     control lines
      DD_WIDTH + 1,                 // cols:      control columns
      (char*)&dispDataPtr,          // dispText:  pointer to text data
      dColor,                       // nColor:    non-focus border color
      dColor,                       // fColor:    focus border color
      tbPrint,                      // filter:    (n/a)
      "Control #5",                 // label:     
      10,                           // labY:      offset from control's ulY
      3,                            // labX       offset from control's ulX
      ddBoxTYPES,                   // exType:    (n/a)
      DD_ITEMS,                     // scrItems:  display item count
      ZERO,                         // scrSel:    highlight index
      dispAttr,                     // scrColor:  pointer to color attrib. array
      NULL,                         // spinData:  (n/a)
      false,                        // active:    view-only
      &ic[6],                       // nextCtrl:  link in next structure
   },
   {  //* Control in right-center, bottom - - - - - - - - -index 6 *
      dctSCROLLEXT,                 // type:      define a scrolling-data control
      rbtTYPES,                     // rbSubtype: (na)
      false,                        // rbSelect:  (n/a)
      25,                           // ulY:       upper left corner in Y
      64,                           // ulX:       upper left corner in X
      DD_ITEMS + 2,                 // lines:     control lines
      DD_WIDTH + 1,                 // cols:      control columns
      (char*)&dispDataPtr,          // dispText:  pointer to text data
      dColor,                       // nColor:    non-focus border color
      dColor,                       // fColor:    focus border color
      tbPrint,                      // filter:    (n/a)
      "Control #6",                 // label:     
      -1,                           // labY:      offset from control's ulY
      3,                            // labX       offset from control's ulX
      ddBoxTYPES,                   // exType:    (n/a)
      DD_ITEMS,                     // scrItems:  display item count
      ZERO,                         // scrSel:    highlight index
      dispAttr,                     // scrColor:  pointer to color attrib. array
      NULL,                         // spinData:  (n/a)
      false,                        // active:    view-only
      &ic[7],                       // nextCtrl:  link in next structure
   },
   {  //* Control in upper right corner of dialog - - - - -index 7 *
      dctSCROLLEXT,                 // type:      define a scrolling-data control
      rbtTYPES,                     // rbSubtype: (na)
      false,                        // rbSelect:  (n/a)
      ZERO,                         // ulY:       upper left corner in Y
      dialogCOLS - DD_WIDTH - 1,    // ulX:       upper left corner in X
      DD_ITEMS + 2,                 // lines:     control lines
      DD_WIDTH + 1,                 // cols:      control columns
      (char*)&dispDataPtr,          // dispText:  pointer to text data
      dColor,                       // nColor:    non-focus border color
      dColor,                       // fColor:    focus border color
      tbPrint,                      // filter:    (n/a)
      "Control #7",                 // label:     
      10,                           // labY:      offset from control's ulY
      3,                            // labX       offset from control's ulX
      ddBoxTYPES,                   // exType:    (n/a)
      DD_ITEMS,                     // scrItems:  display item count
      ZERO,                         // scrSel:    highlight index
      dispAttr,                     // scrColor:  pointer to color attrib. array
      NULL,                         // spinData:  (n/a)
      false,                        // active:    view-only
      &ic[8],                       // nextCtrl:  link in next structure
   },
   {  //* Control in center-right of dialog - - - - - - - -index 8 *
      dctSCROLLEXT,                 // type:      define a scrolling-data control
      rbtTYPES,                     // rbSubtype: (na)
      false,                        // rbSelect:  (n/a)
      12,                           // ulY:       upper left corner in Y
      dialogCOLS - DD_WIDTH - 1,    // ulX:       upper left corner in X
      DD_ITEMS + 2,                 // lines:     control lines
      DD_WIDTH + 1,                 // cols:      control columns
      (char*)&dispDataPtr,          // dispText:  pointer to text data
      dColor,                       // nColor:    non-focus border color
      dColor,                       // fColor:    focus border color
      tbPrint,                      // filter:    (n/a)
      "Control #8",                 // label:     
      10,                           // labY:      offset from control's ulY
      3,                            // labX       offset from control's ulX
      ddBoxTYPES,                   // exType:    (n/a)
      DD_ITEMS,                     // scrItems:  display item count
      ZERO,                         // scrSel:    highlight index
      dispAttr,                     // scrColor:  pointer to color attrib. array
      NULL,                         // spinData:  (n/a)
      false,                        // active:    view-only
      &ic[9],                       // nextCtrl:  link in next structure
   },
   {  //* Control in center-right of dialog - - - - - - - -index 8 *
      dctSCROLLEXT,                 // type:      define a scrolling-data control
      rbtTYPES,                     // rbSubtype: (na)
      false,                        // rbSelect:  (n/a)
      25,                           // ulY:       upper left corner in Y
      dialogCOLS - DD_WIDTH - 1,    // ulX:       upper left corner in X
      DD_ITEMS + 2,                 // lines:     control lines
      DD_WIDTH + 1,                 // cols:      control columns
      (char*)&dispDataPtr,          // dispText:  pointer to text data
      dColor,                       // nColor:    non-focus border color
      dColor,                       // fColor:    focus border color
      tbPrint,                      // filter:    (n/a)
      "Control #9",                 // label:     
      -1,                           // labY:      offset from control's ulY
      3,                            // labX       offset from control's ulX
      ddBoxTYPES,                   // exType:    (n/a)
      DD_ITEMS,                     // scrItems:  display item count
      ZERO,                         // scrSel:    highlight index
      dispAttr,                     // scrColor:  pointer to color attrib. array
      NULL,                         // spinData:  (n/a)
      false,                        // active:    view-only
      NULL,                         // nextCtrl:  link in next structure
   },
} ;

   //* If the controls are Scroll-Box controls, make adjustments *
   if ( scrollBoxControls != false )
   {
      ic[1].type = ic[2].type = ic[3].type = ic[4].type = ic[5].type = 
      ic[6].type = ic[7].type = ic[8].type = ic[9].type = dctSCROLLBOX ;
      ic[1].dispText = ic[2].dispText = ic[3].dispText = ic[4].dispText = 
      ic[5].dispText = ic[6].dispText = ic[7].dispText = ic[8].dispText = 
      ic[9].dispText = ic[10].dispText = (char*)&dispData ;
   }

   //* Initial parameters for dialog window *
   InitNcDialog dInit( dialogROWS,     // number of display lines
                       dialogCOLS,     // number of display columns
                       ulY,            // Y offset from upper-left 
                       ulX,            // X offset from upper-left
                       "  Connections  ", // dialog title
                       ncltSINGLE,     // border line-style
                       dColor,         // border color attribute
                       dColor,         // interior color attribute
                       ic              // pointer to list of control definitions
                     ) ;

   //* Instantiate the dialog window *
   dp = new NcDialog ( dInit ) ;

   //* Open the dialog window *
   if ( (dp->OpenWindow()) == OK )
   {
      //* Make appropriate visual connections for the controls *
      cdConnect cdConn ;
      cdConn.connection = true ;
      cdConn.ll2Left  = cdConn.ur2Top   = true ;
      dp->ConnectControl2Border ( 1, cdConn ) ;
      cdConn.ll2Left  = cdConn.ur2Top   = false ;
      cdConn.ul2Left  = cdConn.ll2Left  = true ;
      dp->ConnectControl2Border ( 2, cdConn ) ;
      cdConn.ul2Left  = cdConn.ll2Left  = false ;
      cdConn.ul2Left  = cdConn.lr2Bot   = true ;
      dp->ConnectControl2Border ( 3, cdConn ) ;
      cdConn.ul2Left  = cdConn.lr2Bot   = false ;
      cdConn.ul2Top   = cdConn.ll2Bot   = cdConn.ur2Top   = cdConn.lr2Bot = true ;
      dp->ConnectControl2Border ( 4, cdConn ) ;
      cdConn.ul2Top   = cdConn.ll2Bot   = cdConn.ur2Top   = cdConn.lr2Bot = false ;
      cdConn.ul2Top   = cdConn.ur2Top   = true ;
      dp->ConnectControl2Border ( 5, cdConn ) ;
      cdConn.ul2Top   = cdConn.ur2Top   = false ;
      cdConn.ll2Bot   = cdConn.lr2Bot   = true ;
      dp->ConnectControl2Border ( 6, cdConn ) ;
      cdConn.ll2Bot   = cdConn.lr2Bot   = false ;
      cdConn.ul2Top   = cdConn.lr2Right = true ;
      dp->ConnectControl2Border ( 7, cdConn ) ;
      cdConn.ul2Top   = cdConn.lr2Right = false ;
      cdConn.ur2Right = cdConn.lr2Right = true ;
      dp->ConnectControl2Border ( 8, cdConn ) ;
      cdConn.ur2Right = cdConn.lr2Right = false ;
      cdConn.ll2Bot   = cdConn.ur2Right = true ;
      dp->ConnectControl2Border ( 9, cdConn ) ;

      //* If the controls are Scroll-Ext controls, initialize the display data *
      if ( scrollBoxControls == false )
      {
         dp->WriteString ( dialogROWS/2 + 2, dialogCOLS/2 - 12, 
                           " dctSCROLLEXT Controls ", nc.bw ) ;
      }
      else
         dp->WriteString ( dialogROWS/2 + 2, dialogCOLS/2 - 12, 
                           " dctSCROLLBOX Controls ", nc.bw ) ;

      dp->RefreshWin () ;
      nckPause() ;
      
   }
   else
   {  //* Most likely cause of dialog not opening is that the terminal window  *
      //* is too small. Give user a clue...                                    *
      short neededLines = (dialogROWS + minTestULY) >= minTERMROWS ?
                          (dialogROWS + minTestULY) : minTERMROWS, 
            neededCols  = dialogCOLS >= minTERMCOLS ? dialogCOLS : minTERMCOLS ;
      gString gs ;
      gs.compose ( L" Sorry, size of terminal window must be at least %hd x %hd for this test. ", 
                   &neededCols, &neededLines ) ;
      StatWin ( gs.gstr(), nc.cyG ) ;
      sleep ( 5 ) ;
   }

   if ( dp != NULL )                         // close the window
      delete ( dp ) ;

}  //* End Test02() *

//*************************
//*  T02_ControlUpdate    *
//*************************
//******************************************************************************
//* This is a callback method for manually updating the controls in the        *
//* Test02() dialog.                                                           *
//*                                                                            *
//*  For this test, the following are updated by the callback method:          *
//*   1.                                                                       *
//*   2.                                                                       *
//*   3.                                                                       *
//*                                                                            *
//* Input  : currIndex: index of control that currently has focus              *
//*          wkey     : user's key input data                                  *
//*          firstTime: the EstablishCallback() method calls this method once  *
//*                     with firstTime==true, to perform any required          *
//*                     initialization. Subsequently, the NcDialog class       *
//*                     always calls with firstTime==false.                    *
//* Returns: OK                                                                *
//******************************************************************************
//* Important Note: This method makes some intimate assumptions about what     *
//* is happening in the method that established the callback. If changes are   *
//* made in the establishing method that affect the callback functionality,    *
//* be sure to update this method to address those changes.                    *
//*                                                                            *
//* Programmer's Note: See dialog control index constants at file scope above. *
//******************************************************************************

short T02_ControlUpdate ( const short currIndex, const wkeyCode wkey, bool firstTime )
{

   return OK ;
   
}  //* End T02_ControlUpdate() *

//*************************
//*       Test03          *
//*************************
//******************************************************************************
//* Testing for the DialogSpinner class of controls.                           *
//*                                                                            *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void Test03 ( void )
{
const short dialogROWS = 35 ;                   // display lines
const short dialogCOLS = 68 ;                   // display columns
const short controlsDEFINED = 16 ;              // # of controls in dialog
//* Additional parameters for dctSPINNER object initializations *
dspinData 
dsData1( 0, 999, 0, dspinINTEGER, nc.blG ),     // int 4-wide
dsData2( -999, 999, 0, dspinINTEGER, nc.blG ),  // int 5-wide (neg)
dsData3( 0, 999, 25, dspinDEC1, nc.grG ),       // dec 5-wide (1 dec place)
dsData4( -999, 999, 0, dspinDEC1, nc.grG ),     // dec 6-wide (neg, 1 dec place)
dsData5( 0, 9999, 0, dspinDEC2, nc.maG ),       // dec 6-wide (2 dec places)
dsData6( -9999, 9999, 0, dspinDEC2, nc.maG ),   // dec 7-wide (neg, 2 dec places)
dsData7( 0, 10000, 0, dspinDEC3, nc.brG ),      // dec 7-wide (3 dec places)
dsData8( -10000, 10000, 0, dspinDEC3, nc.brG ), // dec 8-wide (neg, 3 dec places)
dsData9( 0, 1024, 0, dspinHEX1, nc.brR ),       // hex 9-wide
dsData10( 0, 1000, 0, dspinINTEGER, nc.blG ),   // int 3-wide (overflow at 100)
dsData11( -1000, 1000, 0, dspinDEC2, nc.blG ) ; // dec 5-wide (overflow at -0.01 and 10.00)

attr_t   dColor = nc.cyR ;
short    ctrY    = termRows/2,               // dialog center in Y
         ctrX    = termCols/2,               // dialog center in X
         //* Upper left corner in Y (cannot obscure status window) *
         ulY     = (ctrY - dialogROWS/2) >= minTestULY ? 
                   (ctrY - dialogROWS/2) : minTestULY,
         ulX     = ctrX - dialogCOLS / 2 ;   // upper left corner in X

   StatWin ( "Opening Test 3 - Testing for DialogSpinner controls" ) ;

   InitCtrl ic[controlsDEFINED] =      // control initialization structures
   {
      {  //* 'DONE' pushbutton - - - - -- - - - - - - - - - - - -   T3donePush *
         dctPUSHBUTTON,                // type:      
         rbtTYPES,                     // rbSubtype: (n/a)
         false,                        // rbSelect:  (n/a)
         (short)(dialogROWS - 2),      // ulY:       upper left corner in Y
         (short)(dialogCOLS / 2 - 3),  // ulX:       upper left corner in X
         1,                            // lines:     control lines
         6,                            // cols:      control columns
         " DO^NE ",                    // dispText:  
         nc.gyR,                       // nColor:    non-focus color
         nc.reG,                       // fColor:    focus color
         tbPrint,                      // filter:    (n/a)
         "",                           // label:     (n/a)
         ZERO,                         // labY:      (n/a)
         ZERO,                         // labX       (n/a)
         ddBoxTYPES,                   // exType:    (n/a)
         1,                            // scrItems:  (n/a)
         1,                            // scrSel:    (n/a)
         NULL,                         // scrColor:  (n/a)
         NULL,                         // spinData:  (n/a)
         true,                         // active:    allow control to gain focus
         &ic[T3spinnerA]               // nextCtrl:  link in next structure
      },
      {  //* 'INTEGER' spinner (4-wide) - - - - - - - - - - - - -   T3spinnerA *
         dctSPINNER,                   // type:      
         rbtTYPES,                     // rbSubtype: (n/a)
         false,                        // rbSelect:  (n/a)
         1,                            // ulY:       upper left corner in Y
         48,                           // ulX:       upper left corner in X
         1,                            // lines:     (n/a)
         4,                            // cols:      control columns
         NULL,                         // dispText:  (n/a)
         nc.brR,                       // nColor:    non-focus color
         nc.bw,                        // fColor:    focus color
         tbPrint,                      // filter:    (n/a)
         "Integer, 4-wide (range: 0 to 999):          ^A",   // label:     
         ZERO,                         // labY:      
         -46,                          // labX       
         ddBoxTYPES,                   // exType:    (n/a)
         1,                            // scrItems:  (n/a)
         1,                            // scrSel:    (n/a)
         NULL,                         // scrColor:  (n/a)
         &dsData1,                     // spinData:  spinner init
         true,                         // active:    allow control to gain focus
         &ic[T3spinnerB]               // nextCtrl:  link in next structure
      },
      {  //* Integer spinner (5-wide) - - - - - - - - - - - - - -   T3spinnerB *
         dctSPINNER,                   // type:      
         rbtTYPES,                     // rbSubtype: (n/a)
         false,                        // rbSelect:  (n/a)
         short(ic[T3spinnerA].ulY + 2),// ulY:       upper left corner in Y
         48,                           // ulX:       upper left corner in X
         1,                            // lines:     (n/a)
         5,                            // cols:      control columns
         NULL,                         // dispText:  (n/a)
         nc.brR,                       // nColor:    non-focus color
         nc.bw,                        // fColor:    focus color
         tbPrint,                      // filter:    (n/a)
         "Integer, 5-wide (range: -999 to 999):       ^B",   // label:     
         ZERO,                         // labY:      
         -46,                          // labX       
         ddBoxTYPES,                   // exType:    (n/a)
         1,                            // scrItems:  (n/a)
         1,                            // scrSel:    (n/a)
         NULL,                         // scrColor:  (n/a)
         &dsData2,                     // spinData:  spinner init
         true,                         // active:    allow control to gain focus
         &ic[T3spinnerC]               // nextCtrl:  link in next structure
      },
      {  //* Decimal spinner (5-wide) - - - - - - - - - - - - - -   T3spinnerC *
         dctSPINNER,                   // type:      
         rbtTYPES,                     // rbSubtype: (n/a)
         false,                        // rbSelect:  (n/a)
         short(ic[T3spinnerB].ulY + 2),// ulY:       upper left corner in Y
         48,                           // ulX:       upper left corner in X
         1,                            // lines:     (n/a)
         5,                            // cols:      control columns
         NULL,                         // dispText:  (n/a)
         nc.brR,                       // nColor:    non-focus color
         nc.bw,                        // fColor:    focus color
         tbPrint,                      // filter:    (n/a)
         "Decimal, 5-wide (range: 0.0 to 99.9):       ^C",   // label:     
         ZERO,                         // labY:      
         -46,                          // labX       
         ddBoxTYPES,                   // exType:    (n/a)
         1,                            // scrItems:  (n/a)
         1,                            // scrSel:    (n/a)
         NULL,                         // scrColor:  (n/a)
         &dsData3,                     // spinData:  spinner init
         true,                         // active:    allow control to gain focus
         &ic[T3spinnerD]               // nextCtrl:  link in next structure
      },
      {  //* Decimal spinner (6-wide) - - - - - - - - - - - - - -   T3spinnerD *
         dctSPINNER,                   // type:      
         rbtTYPES,                     // rbSubtype: (n/a)
         false,                        // rbSelect:  (n/a)
         short(ic[T3spinnerC].ulY + 2),// ulY:       upper left corner in Y
         48,                           // ulX:       upper left corner in X
         1,                            // lines:     (n/a)
         6,                            // cols:      control columns
         NULL,                         // dispText:  (n/a)
         nc.brR,                       // nColor:    non-focus color
         nc.bw,                        // fColor:    focus color
         tbPrint,                      // filter:    (n/a)
         "Decimal, 6-wide (range: -99.9 to 99.9):     ^D",   // label:     
         ZERO,                         // labY:      
         -46,                          // labX       
         ddBoxTYPES,                   // exType:    (n/a)
         1,                            // scrItems:  (n/a)
         1,                            // scrSel:    (n/a)
         NULL,                         // scrColor:  (n/a)
         &dsData4,                     // spinData:  spinner init
         true,                         // active:    allow control to gain focus
         &ic[T3spinnerE]               // nextCtrl:  link in next structure
      },
      {  //* Decimal spinner (6-wide) - - - - - - - - - - - - - -   T3spinnerE *
         dctSPINNER,                   // type:      
         rbtTYPES,                     // rbSubtype: (n/a)
         false,                        // rbSelect:  (n/a)
         short(ic[T3spinnerD].ulY + 2),// ulY:       upper left corner in Y
         48,                           // ulX:       upper left corner in X
         1,                            // lines:     (n/a)
         6,                            // cols:      control columns
         NULL,                         // dispText:  (n/a)
         nc.blR,                       // nColor:    non-focus color
         nc.bw,                        // fColor:    focus color
         tbPrint,                      // filter:    (n/a)
         "Decimal, 6-wide (range: 0.00 to 99.99):     ^E",   // label:     
         ZERO,                         // labY:      
         -46,                          // labX       
         ddBoxTYPES,                   // exType:    (n/a)
         1,                            // scrItems:  (n/a)
         1,                            // scrSel:    (n/a)
         NULL,                         // scrColor:  (n/a)
         &dsData5,                     // spinData:  spinner init
         true,                         // active:    allow control to gain focus
         &ic[T3spinnerF]               // nextCtrl:  link in next structure
      },
      {  //* Decimal spinner (7-wide)  - - - - - - - - - - - - - -  T3spinnerF *
         dctSPINNER,                   // type:      
         rbtTYPES,                     // rbSubtype: (n/a)
         false,                        // rbSelect:  (n/a)
         short(ic[T3spinnerE].ulY + 2),// ulY:       upper left corner in Y
         48,                           // ulX:       upper left corner in X
         1,                            // lines:     (n/a)
         7,                            // cols:      control columns
         NULL,                         // dispText:  (n/a)
         nc.blR,                       // nColor:    non-focus color
         nc.bw,                        // fColor:    focus color
         tbPrint,                      // filter:    (n/a)
         "Decimal, 7-wide (range: -99.99 to 99.99):   ^F",   // label:     
         ZERO,                         // labY:      
         -46,                          // labX       
         ddBoxTYPES,                   // exType:    (n/a)
         1,                            // scrItems:  (n/a)
         1,                            // scrSel:    (n/a)
         NULL,                         // scrColor:  (n/a)
         &dsData6,                     // spinData:  spinner init
         true,                         // active:    allow control to gain focus
         &ic[T3spinnerG]               // nextCtrl:  link in next structure
      },
      {  //* Decimal spinner (7-wide) - - - - - - - - - - - - - - - T3spinnerG *
         dctSPINNER,                   // type:      
         rbtTYPES,                     // rbSubtype: (n/a)
         false,                        // rbSelect:  (n/a)
         short(ic[T3spinnerF].ulY + 2),// ulY:       upper left corner in Y
         48,                           // ulX:       upper left corner in X
         1,                            // lines:     (n/a)
         7,                            // cols:      control columns
         NULL,                         // dispText:  (n/a)
         nc.blR,                       // nColor:    non-focus color
         nc.bw,                        // fColor:    focus color
         tbPrint,                      // filter:    (n/a)
         "Decimal, 7-wide (range: 0 to 10.000):       ^G",   // label:     
         ZERO,                         // labY:      
         -46,                          // labX       
         ddBoxTYPES,                   // exType:    (n/a)
         1,                            // scrItems:  (n/a)
         1,                            // scrSel:    (n/a)
         NULL,                         // scrColor:  (n/a)
         &dsData7,                     // spinData:  spinner init
         true,                         // active:    allow control to gain focus
         &ic[T3spinnerH]               // nextCtrl:  link in next structure
      },
      {  //* Decimal spinner (8-wide) - - - - - - - - - - - - - - - T3spinnerH *
         dctSPINNER,                   // type:      
         rbtTYPES,                     // rbSubtype: (n/a)
         false,                        // rbSelect:  (n/a)
         short(ic[T3spinnerG].ulY + 2),// ulY:       upper left corner in Y
         48,                           // ulX:       upper left corner in X
         1,                            // lines:     (n/a)
         8,                            // cols:      control columns
         NULL,                         // dispText:  (n/a)
         nc.blR,                       // nColor:    non-focus color
         nc.bw,                        // fColor:    focus color
         tbPrint,                      // filter:    (n/a)
         "Decimal, 8-wide (range: -10.000 to 10.000): ^H",   // label:     
         ZERO,                         // labY:      
         -46,                          // labX       
         ddBoxTYPES,                   // exType:    (n/a)
         1,                            // scrItems:  (n/a)
         1,                            // scrSel:    (n/a)
         NULL,                         // scrColor:  (n/a)
         &dsData8,                     // spinData:  spinner init
         true,                         // active:    allow control to gain focus
         &ic[T3spinnerI]               // nextCtrl:  link in next structure
      },
      {  //* Hexaecimal spinner (8-wide)  - - - - - - - - - - - - - T3spinnerI *
         dctSPINNER,                   // type:      
         rbtTYPES,                     // rbSubtype: (n/a)
         false,                        // rbSelect:  (n/a)
         short(ic[T3spinnerH].ulY + 2),// ulY:       upper left corner in Y
         48,                           // ulX:       upper left corner in X
         1,                            // lines:     (n/a)
         8,                            // cols:      control columns
         NULL,                         // dispText:  (n/a)
         nc.grR,                       // nColor:    non-focus color
         nc.bw,                        // fColor:    focus color
         tbPrint,                      // filter:    (n/a)
         "Hexadecimal, 8-wide (range: 0 to 1024):     ^K",   // label:     
         ZERO,                         // labY:      
         -46,                          // labX       
         ddBoxTYPES,                   // exType:    (n/a)
         1,                            // scrItems:  (n/a)
         1,                            // scrSel:    (n/a)
         NULL,                         // scrColor:  (n/a)
         &dsData9,                     // spinData:  spinner init
         true,                         // active:    allow control to gain focus
         &ic[T3spinOvr3]               // nextCtrl:  link in next structure
      },
      {  //* Integer spinner (3-wide) - - - - - - - - - - - - -     T3spinOvr3 *
         dctSPINNER,                   // type:      
         rbtTYPES,                     // rbSubtype: (n/a)
         false,                        // rbSelect:  (n/a)
         short(ic[T3spinnerI].ulY + 2),// ulY:       upper left corner in Y
         2,                            // ulX:       upper left corner in X
         1,                            // lines:     (n/a)
         3,                            // cols:      control columns
         NULL,                         // dispText:  (n/a)
         nc.brR,                       // nColor:    non-focus color
         nc.re,                        // fColor:    focus color
         tbPrint,                      // filter:    (n/a)
         "^Overflow Test 1 (range: 0 to 1000)",   // label:     
         ZERO,                         // labY:      
         8,                            // labX       
         ddBoxTYPES,                   // exType:    (n/a)
         1,                            // scrItems:  (n/a)
         1,                            // scrSel:    (n/a)
         NULL,                         // scrColor:  (n/a)
         &dsData10,                    // spinData:  spinner init
         true,                         // active:    allow control to gain focus
         &ic[T3textOvr3]               // nextCtrl:  link in next structure
      },
      {  //* Non-active text box      - - - - - - - - - - - - -     T3textOvr3 *
         dctTEXTBOX,                   // type:      
         rbtTYPES,                     // rbSubtype: (n/a)
         false,                        // rbSelect:  (n/a)
         short(ic[T3spinOvr3].ulY + 1),// ulY:       upper left corner in Y
         2,                            // ulX:       upper left corner in X
         1,                            // lines:     (n/a)
         6,                            // cols:      control columns
         "     0",                     // dispText:  (value of T3spinOvr3)
         nc.gyR,                       // nColor:    non-focus color
         nc.bw,                        // fColor:    focus color
         tbPrint,                      // filter:    any printing character
         "actual value",               // label:     
         ZERO,                         // labY:      
         8,                            // labX       
         ddBoxTYPES,                   // exType:    (n/a)
         1,                            // scrItems:  (n/a)
         1,                            // scrSel:    (n/a)
         NULL,                         // scrColor:  (n/a)
         NULL,                         // spinData:  (n/a)
         false,                        // active:    cannot gain focus
         &ic[T3spinOvr5]               // nextCtrl:  link in next structure
      },
      {  //* Decimal spinner (5-wide) - - - - - - - - - - - - - -   T3spinOvr5 *
         dctSPINNER,                   // type:      
         rbtTYPES,                     // rbSubtype: (n/a)
         false,                        // rbSelect:  (n/a)
         short(ic[T3textOvr3].ulY + 1),// ulY:       upper left corner in Y
         2,                            // ulX:       upper left corner in X
         1,                            // lines:     (n/a)
         5,                            // cols:      control columns
         NULL,                         // dispText:  (n/a)
         nc.brR,                       // nColor:    non-focus color
         nc.re,                        // fColor:    focus color
         tbPrint,                      // filter:    (n/a)
         "Overflow ^Test 2 (range: -10.00 to 10.00)",   // label:     
         ZERO,                         // labY:      
         8,                            // labX       
         ddBoxTYPES,                   // exType:    (n/a)
         1,                            // scrItems:  (n/a)
         1,                            // scrSel:    (n/a)
         NULL,                         // scrColor:  (n/a)
         &dsData11,                    // spinData:  spinner init
         true,                         // active:    allow control to gain focus
         &ic[T3textOvr5]               // nextCtrl:  link in next structure
      },
      {  //* Non-active text box      - - - - - - - - - - - - -     T3textOvr5 *
         dctTEXTBOX,                   // type:      
         rbtTYPES,                     // rbSubtype: (n/a)
         false,                        // rbSelect:  (n/a)
         short(ic[T3spinOvr5].ulY + 1),// ulY:       upper left corner in Y
         2,                            // ulX:       upper left corner in X
         1,                            // lines:     (n/a)
         6,                            // cols:      control columns
         "  0.00",                     // dispText:  (value of T3spinOvr5)
         nc.gyR,                       // nColor:    non-focus color
         nc.bw,                        // fColor:    focus color
         tbPrint,                      // filter:    any printing character
         "actual value",               // label:     
         ZERO,                         // labY:      
         8,                            // labX       
         ddBoxTYPES,                   // exType:    (n/a)
         1,                            // scrItems:  (n/a)
         1,                            // scrSel:    (n/a)
         NULL,                         // scrColor:  (n/a)
         NULL,                         // spinData:  (n/a)
         false,                        // active:    cannot gain focus
         &ic[T3copyPush]               // nextCtrl:  link in next structure
      },
      {  //* 'COPY' pushbutton - - - - -- - - - - - - - - - - - -   T3copyPush *
         dctPUSHBUTTON,                // type:      
         rbtTYPES,                     // rbSubtype: (n/a)
         false,                        // rbSelect:  (n/a)
         ic[T3textOvr5].ulY,           // ulY:       upper left corner in Y
         short(dialogCOLS -26),        // ulX:       upper left corner in X
         1,                            // lines:     control lines
         24,                           // cols:      control columns
         " Co^py 'C' To Overflow 2 ",  // dispText:  
         nc.gyR,                       // nColor:    non-focus color
         nc.reG,                       // fColor:    focus color
         tbPrint,                      // filter:    (n/a)
         "",                           // label:     (n/a)
         ZERO,                         // labY:      (n/a)
         ZERO,                         // labX       (n/a)
         ddBoxTYPES,                   // exType:    (n/a)
         1,                            // scrItems:  (n/a)
         1,                            // scrSel:    (n/a)
         NULL,                         // scrColor:  (n/a)
         NULL,                         // spinData:  (n/a)
         true,                         // active:    allow control to gain focus
         &ic[T3dumpPush]               // nextCtrl:  link in next structure
      },
      {  //* 'DUMP' pushbutton - - - - -- - - - - - - - - - - - -   T3dumpPush *
         dctPUSHBUTTON,                // type:      
         rbtTYPES,                     // rbSubtype: (n/a)
         false,                        // rbSelect:  (n/a)
         (short)(dialogROWS - 2),      // ulY:       upper left corner in Y
         (short)(dialogCOLS -21),      // ulX:       upper left corner in X
         1,                            // lines:     control lines
         18,                           // cols:      control columns
         " D^ump uiInfo Data ",        // dispText:  
         nc.gyR,                       // nColor:    non-focus color
         nc.reG,                       // fColor:    focus color
         tbPrint,                      // filter:    (n/a)
         "",                           // label:     (n/a)
         ZERO,                         // labY:      (n/a)
         ZERO,                         // labX       (n/a)
         ddBoxTYPES,                   // exType:    (n/a)
         1,                            // scrItems:  (n/a)
         1,                            // scrSel:    (n/a)
         NULL,                         // scrColor:  (n/a)
         NULL,                         // spinData:  (n/a)
         true,                         // active:    allow control to gain focus
         NULL                          // nextCtrl:  link in next structure
      },
   } ;

   //* Initial parameters for dialog window *
   InitNcDialog dInit( dialogROWS,     // number of display lines
                       dialogCOLS,     // number of display columns
                       ulY,            // Y offset from upper-left of terminal 
                       ulX,            // X offset from upper-left of terminal 
                       "  DialogSpinner Class Controls  ", // dialog title
                       ncltSINGLE,     // border line-style
                       dColor,         // border color attribute
                       dColor,         // interior color attribute
                       ic              // pointer to list of control definitions
                     ) ;

   //* Instantiate the dialog window *
   NcDialog* dp = new NcDialog ( dInit ) ;
   Test03Dialog = dp ;              // give callback method access to the dialog


   //* Open the dialog window *
   if ( (dp->OpenWindow()) == OK )
   {
      //* Draw a line to beautify the display *
      winPos wp ( dialogROWS - 12, 2 ) ;
      LineDef  lDef(ncltHORIZ, ncltSINGLE, wp.ypos, ZERO, dialogCOLS, dColor) ;
      dp->DrawLine ( lDef ) ;
      ++wp.ypos ;
      
      //* Display dialog's static text *
      const char* const T3Instructions = 
         "Use Tab and Shift+Tab keys to place focus on desired control,\n"
         "or select control via 'Hotkey' indicated by underlined character.\n"
         "Adjusting spinner-control values:\n"
         " increment/decrement by 1   : up/down keys (or plus/minus keys)\n"
         " increment/decrement by 10  : Shift + up/down keys\n"
         " increment/decrement by 100 : Control + up/down keys\n"
         " increment/decrement by 1000: Alt + up/down keys\n"
         " increment/decrement by 10% : PageUp/PageDown keys\n"
         " maximum/minimum value      : Home/End keys" ;
      gString gs( T3Instructions ) ;
      dp->WriteParagraph ( wp, gs, dColor ) ;

      dp->RefreshWin () ;

      //* Establish a call-back method that can be called from within the *
      //* NcDialog code. Called each time through a control's user-input  *
      //* loop so we can manually modify the controls when necessary.     *
      dp->EstablishCallback ( &T03_ControlUpdate ) ;

      //* Position for user-interface dump. Note that the coordinates for *
      //* display are absolute screen coordinates, not dialog offsets     *
      winPos wPos(4,(ulX + dialogCOLS)) ;

      uiInfo Info ;                 // user interface data returned here
      short  icIndex = ZERO ;       // index of control with input focus
      bool   uidumpEnabled = false, // enable user-interface info dump
             done = false ;         // loop control
      while ( ! done )
      {
         //*******************************************
         //* If focus is currently on a Pushbutton   *
         //*******************************************
         if ( ic[icIndex].type == dctPUSHBUTTON )
         {
            if ( Info.viaHotkey == false )
            {
               icIndex = dp->EditPushbutton ( Info ) ;

               //* If no hotkey activity AND button was pressed via nckENTER or*
               //* SPACE keys. then Info.keyIn == ZERO indicating no change of *
               //* focus; however, since the keypress will be processed during *
               //* this loop iteration, allow focus to advance.                *
               if ( ! Info.viaHotkey && Info.keyIn == ZERO ) 
                  Info.keyIn = nckTAB ;

               if ( uidumpEnabled ) // dump user input info
                  dp->Dump_uiInfo ( wPos, Info ) ;
            }
            else
            {
               //* Information arrived in the Info.h_xxx fields.*
               //* Transfer it to the primary positions.        *
               //* Previous data in these fields has already    *
               //* been handled in prior loop iteration.        *
               if ( uidumpEnabled ) // dump user input info
                  dp->Dump_uiInfo ( wPos, Info ) ;
               Info.HotData2Primary () ;

               //* Button press arrived via hotkey. Since processing *
               //* will be completed during this loop iteration,     *
               //* allow focus to advance to the next control.       *
               Info.keyIn = nckTAB ;

               if ( uidumpEnabled )  // dump user input info
                  dp->Dump_uiInfo ( wPos, Info ) ;
            }

            //* If a Pushbutton was pressed *
            if ( Info.ctrlType == dctPUSHBUTTON && Info.dataMod != false )
            {  //* Return to test selection *
               if ( icIndex == T3donePush )
                  done = true ;

               //* Copy contents of spinner 'C' to spinner 'Overflow Test 2'   *
               //* This exercises the GetSpinnerValue() and SetSpinnerValue()  *
               //* methods.                                                    *
               else if ( icIndex == T3copyPush )
               {  //* Get current contents of spinner 'C' *
                  int spC_value ;
                  dp->GetSpinnerValue ( T3spinnerC, spC_value ) ;
                  // Because spinner 'C' is formatted for one decimal place,   *
                  // but spinner 'Overflow Test 2' is formatted for two decimal*
                  //* places, we must adjust our value accordingly.            *
                  //* (For more information, see notes for definition of )     *
                  //* (dspinData class, located in NcDialog.hpp          )     *
                  spC_value *= 10 ;
                  //* Set contents of spinner 'Overflow Test 2' *
                  // Note: We do not range-test the captured value, allowing us 
                  //       to exercise error reporting returned by the dialog 
                  //       method call. In your application PLEASE do a range 
                  //       test before manually setting a spinner's value.
                  //       You'll be glad you did!
                  if ( (dp->SetSpinnerValue (T3spinOvr5, spC_value)) != OK )
                  {
                     //* Error condition - of course we never make misteaks,   *
                     //* but if the value in spinner 'C' is outside the valid  *
                     //* range for the target control, we will end up here.    *
                     dp->DebugMsg ( "Error returned from SetSpinnerValue()", 
                                    3, true ) ;
                  }
               }
               //* Toggle flag for display of user input info structure *
               else if ( icIndex == T3dumpPush )
                  uidumpEnabled = uidumpEnabled ? false : true ;
            }
            else
            {  /* No button press, so nothing to do */ }
         }

         //*******************************************
         //* If focus is currently on a Spinner      *
         //*******************************************
         if ( ic[icIndex].type == dctSPINNER )
         {
            //* It doesn't matter how this control gained the input    *
            //* focus, by Tab/ShiftTab or via hotkey. If by hotkey, no *
            //* processing was done anyway, so discard any hotkey data *
            Info.viaHotkey = false ;

            icIndex = dp->EditSpinner ( Info ) ;
            if ( uidumpEnabled ) // dump user input info
               dp->Dump_uiInfo ( wPos, Info ) ;

            //* If displayed value has been modified *
            if ( Info.ctrlType == dctSPINNER && Info.dataMod != false )
            {
               //* Take any necessary actions here *
               switch ( icIndex )
               {
                  case T3spinnerA:  break ;  // Spinner A
                  case T3spinnerB:  break ;  // Spinner B
                  case T3spinnerC:  break ;  // Spinner C
                  case T3spinnerD:  break ;  // Spinner D
                  case T3spinnerE:  break ;  // Spinner E
                  case T3spinnerF:  break ;  // Spinner F
                  case T3spinnerG:  break ;  // Spinner G
                  case T3spinnerH:  break ;  // Spinner H
                  case T3spinOvr3:  break ;  // Overflow Test 1
                  case T3spinOvr5:  break ;  // Overflow Test 2
                  default:          break ;  // should never happen
                     
               }
            }
            else
            { /* Spinner data not modified, so nothing to do */ }
         }
         else
            {  /* no other active controls defined for this dialog */ }

         if ( done == false )
         {
            if ( Info.keyIn == nckSTAB )
               icIndex = dp->PrevControl () ; 
            else if ( Info.keyIn != ZERO )
               icIndex = dp->NextControl () ;
         }
      }  // while()
   }  // OpenWindow()
   else
   {  //* Most likely cause of dialog not opening is that the terminal window  *
      //* is too small. Give user a clue...                                    *
      short neededLines = (dialogROWS + minTestULY) >= minTERMROWS ?
                          (dialogROWS + minTestULY) : minTERMROWS, 
            neededCols  = dialogCOLS >= minTERMCOLS ? dialogCOLS : minTERMCOLS ;
      gString gs ;
      gs.compose ( L" Sorry, size of terminal window must be at least %hd x %hd for this test. ", 
                   &neededCols, &neededLines ) ;
      StatWin ( gs.gstr(), nc.cyG ) ;
      sleep ( 5 ) ;
   }

   if ( dp != NULL )                         // close the window
      delete ( dp ) ;

}  //* End Test03() *

//*************************
//*  T03_ControlUpdate    *
//*************************
//******************************************************************************
//* This is a callback method for manually updating the controls in the        *
//* Test03() dialog.                                                           *
//*                                                                            *
//*  For this test, the following are updated by the callback method:          *
//*   1. T3textOvr3 tracks actual value of T3spinOvr3 for spinner overflow test*
//*   2. T3textOvr5 tracks actual value of T3spinOvr5 for spinner overflow test*
//*   3.                                                                       *
//*                                                                            *
//* Input  : currIndex: index of control that currently has focus              *
//*          wkey     : user's key input data                                  *
//*          firstTime: the EstablishCallback() method calls this method once  *
//*                     with firstTime==true, to perform any required          *
//*                     initialization. Subsequently, the NcDialog class       *
//*                     always calls with firstTime==false.                    *
//* Returns: OK                                                                *
//******************************************************************************
//* Important Note: This method makes some intimate assumptions about what     *
//* is happening in the method that established the callback. If changes are   *
//* made in the establishing method that affect the callback functionality,    *
//* be sure to update this method to address those changes.                    *
//*                                                                            *
//* Programmer's Note: See dialog control index constants at file scope above. *
//******************************************************************************

short T03_ControlUpdate ( const short currIndex, const wkeyCode wkey, bool firstTime )
{
   static int  old3Value, old5Value ;
   int   ovr3Value, ovr5Value ; 
         Test03Dialog->GetSpinnerValue ( T3spinOvr3, ovr3Value ) ;
         Test03Dialog->GetSpinnerValue ( T3spinOvr5, ovr5Value ) ;
   gString textBuff ;

   if ( firstTime != false )
   {  //* Remember the most recent value *
      old3Value = ovr3Value ;
      old5Value = ovr5Value ;
   }

   //* Update the dctTEXTBOX objects that report contents  *
   //* of spinners used for testing display-field overflow.*
   if ( ovr3Value != old3Value )
   {
      textBuff.compose( L"% 6d", &ovr3Value ) ;
      Test03Dialog->SetTextboxText ( T3textOvr3, textBuff ) ;
      old3Value = ovr3Value ;             // update our static tracking data
   }
   if ( ovr5Value != old5Value )
   {
      double val = (double)ovr5Value / 100.0 ;
      textBuff.compose( L"%6.2lf", &val ) ;
      Test03Dialog->SetTextboxText ( T3textOvr5, textBuff ) ;
      old5Value = ovr5Value ;             // update our static tracking data
   }

   return OK ;

}  //* End T03_ControlUpdate() *

//*************************
//*       Test04          *
//*************************
//******************************************************************************
//* Exercise the NCurses-class / NcDialog-class color-mapping functionality.   *
//*                                                                            *
//* Remapping of available terminal colors takes two forms:                    *
//*  1. Reassign colors for the foreground/background color pairs.             *
//*  2. Adjust the RGB (Red-Green-Blue) register values.                       *
//* Both options are available here.                                           *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void Test04 ( void )
{
   StatWin ( "Opening Test 4 - Adjusting the Text-color Map" ) ;

   short neededLines = (ctdROWS + minTestULY) >= minTERMROWS ?
                       (ctdROWS + minTestULY) : minTERMROWS, 
         neededCols  = ctdCOLS >= minTERMCOLS ? ctdCOLS : minTERMCOLS ;
   if ( (termRows >= neededLines) && (termCols >= neededCols) )
   {
      ColorTest* colorTest = new ColorTest ( termRows, termCols, minTestULY ) ;
      if ( (colorTest->ctDialogOpened ()) != false )
      {
         colorTest->ctInteract () ;
      }
      else { /* Memory allocation error - unlikely */ }

      if ( colorTest != NULL )   // close the dialog
         delete colorTest ;
   }
   else
   {  //* Most likely cause of dialog not opening is that the terminal window  *
      //* is too small. Give user a clue...                                    *
      gString gs ;
      gs.compose ( L" Sorry, size of terminal window must "
                    "be at least %hd x %hd for this test. ", 
                   &neededCols, &neededLines ) ;
      StatWin ( gs.gstr(), nc.cyG ) ;
      sleep ( 5 ) ;
   }

}  //* End Test04() *

//*************************
//*       Test05          *
//*************************
//******************************************************************************
//* Key input testing.                                                         *
//* Note that this test uses a simple class definition to contain the methods  *
//* and data needed for this test. See KeyTest.hpp and KeyTest.cpp.            *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void Test05 ( void )
{
   StatWin ( "Opening Test 5 - Keyboard Input and Key Translation" ) ;

   const short dialogROWS = ktdROWS ;
   const short dialogCOLS = ktdCOLS ;
   short neededLines = (dialogROWS + minTestULY) >= minTERMROWS ?
                       (dialogROWS + minTestULY) : minTERMROWS, 
         neededCols  = dialogCOLS >= minTERMCOLS ? dialogCOLS : minTERMCOLS ;
   if ( (termRows >= neededLines) && (termCols >= neededCols) )
   {
      KeyTest* keyTest = new KeyTest ( termRows, termCols, minTestULY ) ;
      if ( (keyTest->ktDialogOpened ()) != false )
      {
         keyTest->ktInteract () ;
      }
      else { /* Memory allocation error - unlikely */ }

      if ( keyTest != NULL )     // close the dialog
         delete keyTest ;
   }
   else
   {  //* Most likely cause of dialog not opening is that the terminal window  *
      //* is too small. Give user a clue...                                    *
      gString gs ;
      gs.compose ( L" Sorry, size of terminal window must be at least %hd x %hd for this test. ", 
                   &neededCols, &neededLines ) ;
      StatWin ( gs.gstr(), nc.cyG ) ;
      sleep ( 5 ) ;
   }

}  //* End Test05() *

//*************************
//*       Test06          *
//*************************
//******************************************************************************
//* Display the NCurses diagnostics for foreground/background color mapping.   *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void Test06 ( void )
{
   if ( termRows >= 27 && termCols >= minTERMCOLS )
   {
      const short dialogROWS = 4,         // display lines
                  dialogCOLS = 45,        // display columns
                  controlsDEFINED = 3,    // # of controls in dialog
                  ulY = 5,                // dialog position
                  ulX = 18,
                  basePush = ZERO,        // control indices
                  variPush = 1,
                  donePush = 2 ;
      attr_t dColor = nc.blG ;         // dialog background color

      StatWin ( "Opening Test 6 - Display Text Color Mapping" ) ;

      InitCtrl ic[controlsDEFINED] =   // array of dialog control initialization objects
      {
         {  //* 'BASE COLORS' pushbutton  - - - - - - - - - - - - - - donePush *
            dctPUSHBUTTON,                // type:      
            rbtTYPES,                     // rbSubtype: (n/a)
            false,                        // rbSelect:  (n/a)
            short(dialogROWS - 2),        // ulY:       upper left corner in Y
            2,                            // ulX:       upper left corner in X
            1,                            // lines:     control lines
            13,                           // cols:      control columns
            " BASE COLORS ",              // dispText:  
            nc.gyR,                       // nColor:    non-focus color
            nc.reG,                       // fColor:    focus color
            tbPrint,                      // filter:    (n/a)
            NULL,                         // label:     (n/a)
            ZERO,                         // labY:      (n/a)
            ZERO,                         // labX       (n/a)
            ddBoxTYPES,                   // exType:    (n/a)
            1,                            // scrItems:  (n/a)
            1,                            // scrSel:    (n/a)
            NULL,                         // scrColor:  (n/a)
            NULL,                         // spinData:  (n/a)
            true,                         // active:    allow control to gain focus
            &ic[variPush],                // nextCtrl:  link in next structure
         },
         {  //* 'DEFINED COLORS' pushbutton   - - - - - - - - - -     basePush *
            dctPUSHBUTTON,                // type:      
            rbtTYPES,                     // rbSubtype: (n/a)
            false,                        // rbSelect:  (n/a)
            short(ic[basePush].ulY),      // ulY:       upper left corner in Y
            short(ic[basePush].ulX + 15), // ulX:       upper left corner in X
            1,                            // lines:     control lines
            16,                           // cols:      control columns
            " DEFINED COLORS ",           // dispText:  
            nc.gyR,                       // nColor:    non-focus color
            nc.reG,                       // fColor:    focus color
            tbPrint,                      // filter:    (n/a)
            NULL,                         // label:     (n/a)
            ZERO,                         // labY:      (n/a)
            ZERO,                         // labX       (n/a)
            ddBoxTYPES,                   // exType:    (n/a)
            1,                            // scrItems:  (n/a)
            1,                            // scrSel:    (n/a)
            NULL,                         // scrColor:  (n/a)
            NULL,                         // spinData:  (n/a)
            true,                         // active:    allow control to gain focus
            &ic[donePush],                // nextCtrl:  link in next structure
         },
         {  //* 'DONE' pushbutton - - - - - - - - - - - - - - - - -   donePush *
            dctPUSHBUTTON,                // type:      
            rbtTYPES,                     // rbSubtype: (n/a)
            false,                        // rbSelect:  (n/a)
            short(ic[variPush].ulY),      // ulY:       upper left corner in Y
            short(ic[variPush].ulX + 18), // ulX:       upper left corner in X
            1,                            // lines:     control lines
            8,                            // cols:      control columns
            "  DONE  ",                   // dispText:  
            nc.gyR,                       // nColor:    non-focus color
            nc.reG,                       // fColor:    focus color
            tbPrint,                      // filter:    (n/a)
            NULL,                         // label:     (n/a)
            ZERO,                         // labY:      (n/a)
            ZERO,                         // labX       (n/a)
            ddBoxTYPES,                   // exType:    (n/a)
            1,                            // scrItems:  (n/a)
            1,                            // scrSel:    (n/a)
            NULL,                         // scrColor:  (n/a)
            NULL,                         // spinData:  (n/a)
            true,                         // active:    allow control to gain focus
            NULL,                         // nextCtrl:  link in next structure
         },
      } ;
      //* Initial parameters for dialog window *
      InitNcDialog dInit( dialogROWS,     // number of display lines
                          dialogCOLS,     // number of display columns
                          ulY,            // Y offset from upper-left of terminal 
                          ulX,            // X offset from upper-left of terminal 
                          "  Select Color Test  ", // dialog title
                          ncltSINGLE,     // border line-style
                          dColor,         // border color attribute
                          dColor,         // interior color attribute
                          ic              // pointer to list of control definitions
                        ) ;

      //* Instantiate the dialog window *
      NcDialog* dp = new NcDialog ( dInit ) ;
   
      //* Open the dialog window *
      if ( (dp->OpenWindow()) == OK )
      {
         uiInfo Info ;                 // user interface data returned here
         short  icIndex = ZERO ;       // index of control with input focus
         bool   done = false ;         // loop control
         while ( ! done )
         {
            //*******************************************
            //* If focus is currently on a Pushbutton   *
            //*******************************************
            if ( ic[icIndex].type == dctPUSHBUTTON )
            {
               icIndex = dp->EditPushbutton ( Info ) ;
               if ( Info.dataMod != false )
               {
                  if ( Info.ctrlIndex == basePush )
                  {
                     dp->SetDialogObscured () ;
                     nc.DisplayColorOptions ( 3, 1, false ) ;
                     sWin->RefreshWin () ;
                     nckPause() ;
                     ClearTerminalWindow () ;
                     sWin->RefreshWin () ;
                     dp->RefreshWin () ;
                  }
                  else if ( Info.ctrlIndex == variPush )
                  {
                     dp->SetDialogObscured () ;
                     nc.DisplayDefinedColors ( 3, 1, false ) ;
                     sWin->RefreshWin () ;
                     nckPause() ;
                     ClearTerminalWindow () ;
                     sWin->RefreshWin () ;
                     dp->RefreshWin () ;
                  }
                  else if ( Info.ctrlIndex == donePush )
                     done = true ;
               }
            }
   
            //* Advance the input focus *
            if ( done == false )
            {
               if ( Info.keyIn == nckSTAB )
                  icIndex = dp->PrevControl () ; 
               else
                  icIndex = dp->NextControl () ;
            }
         }  // while()
      }
      if ( dp != NULL )                      // close the window
         delete ( dp ) ;

   }
   else
   {
      StatWin ( "Sorry, this test requires that terminal be at "
                "least 80 columns by 40 lines.", nc.cyG ) ;
      sleep ( 3 ) ;
   }

}  //* End Test06() *

//*************************
//*       Test07          *
//*************************
//******************************************************************************
//* This dialog tests the functionality of the NcWindow/NcDialog display       *
//* capture methods and writes the captured data to a file, either plain       *
//* text or HTML format.                                                       *
//*                                                                            *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************
//*                                                                            *
//*                                                                            *
//******************************************************************************

void Test07 ( void )
{
   //* Target window dimensions and position *
   const short trgROWS = 36, trgCOLS = 120, trgULY = 3, trgULX = 2 ;

   if ( termRows >= trgROWS && termCols >= trgCOLS )
   {
      const short dialogROWS = 4,
                  dialogCOLS = 48,
                  ulY = 5,                // dialog position
                  ulX = 18 ;
      enum ctrl : short { ccPB = ZERO, cdPB, donePB, controlsDEFINED } ;
      attr_t dColor = nc.blR ;         // dialog background color

      StatWin ( "Opening Test 7 - Capture Display Data to Text or HTML" ) ;

      InitCtrl ic[controlsDEFINED] =   // array of dialog control initialization objects
      {
         {  //* 'CAPTURE COLORS' pushbutton - - - - - - - - - - - - - -   ccPB *
            dctPUSHBUTTON,                // type:      
            rbtTYPES,                     // rbSubtype: (n/a)
            false,                        // rbSelect:  (n/a)
            short(dialogROWS - 2),        // ulY:       upper left corner in Y
            2,                            // ulX:       upper left corner in X
            1,                            // lines:     control lines
            16,                           // cols:      control columns
            " CAPTURE COLORS ",           // dispText:  
            nc.gyR,                       // nColor:    non-focus color
            nc.reG,                       // fColor:    focus color
            tbPrint,                      // filter:    (n/a)
            NULL,                         // label:     (n/a)
            ZERO,                         // labY:      (n/a)
            ZERO,                         // labX       (n/a)
            ddBoxTYPES,                   // exType:    (n/a)
            1,                            // scrItems:  (n/a)
            1,                            // scrSel:    (n/a)
            NULL,                         // scrColor:  (n/a)
            NULL,                         // spinData:  (n/a)
            true,                         // active:    allow control to gain focus
            &ic[cdPB]                     // nextCtrl:  link in next structure
         },
         {  //* 'CAPTURE DIALOG' pushbutton   - - - - - - - - - - - - - - cdPB *
            dctPUSHBUTTON,                // type:      
            rbtTYPES,                     // rbSubtype: (n/a)
            false,                        // rbSelect:  (n/a)
            short(ic[ccPB].ulY),          // ulY:       upper left corner in Y
            short(ic[ccPB].ulX + ic[ccPB].cols + 2), // ulX:       upper left corner in X
            1,                            // lines:     control lines
            16,                           // cols:      control columns
            " CAPTURE DIALOG ",           // dispText:  
            nc.gyR,                       // nColor:    non-focus color
            nc.reG,                       // fColor:    focus color
            tbPrint,                      // filter:    (n/a)
            NULL,                         // label:     (n/a)
            ZERO,                         // labY:      (n/a)
            ZERO,                         // labX       (n/a)
            ddBoxTYPES,                   // exType:    (n/a)
            1,                            // scrItems:  (n/a)
            1,                            // scrSel:    (n/a)
            NULL,                         // scrColor:  (n/a)
            NULL,                         // spinData:  (n/a)
            true,                         // active:    allow control to gain focus
            &ic[donePB]                   // nextCtrl:  link in next structure
         },
         {  //* 'DONE' pushbutton - - - - - - - - - - - - - - - - - -   donePB *
            dctPUSHBUTTON,                // type:      
            rbtTYPES,                     // rbSubtype: (n/a)
            false,                        // rbSelect:  (n/a)
            short(ic[cdPB].ulY),          // ulY:       upper left corner in Y
            short(ic[cdPB].ulX + ic[cdPB].cols + 2), // ulX:       upper left corner in X
            1,                            // lines:     control lines
            8,                            // cols:      control columns
            "  DONE  ",                   // dispText:  
            nc.gyR,                       // nColor:    non-focus color
            nc.reG,                       // fColor:    focus color
            tbPrint,                      // filter:    (n/a)
            NULL,                         // label:     (n/a)
            ZERO,                         // labY:      (n/a)
            ZERO,                         // labX       (n/a)
            ddBoxTYPES,                   // exType:    (n/a)
            1,                            // scrItems:  (n/a)
            1,                            // scrSel:    (n/a)
            NULL,                         // scrColor:  (n/a)
            NULL,                         // spinData:  (n/a)
            true,                         // active:    allow control to gain focus
            NULL,                         // nextCtrl:  link in next structure
         },
      } ;
      //* Initial parameters for dialog window *
      InitNcDialog dInit( dialogROWS,     // number of display lines
                          dialogCOLS,     // number of display columns
                          ulY,            // Y offset from upper-left of terminal 
                          ulX,            // X offset from upper-left of terminal 
                          "  Select Capture Test  ", // dialog title
                          ncltSINGLE,     // border line-style
                          dColor,         // border color attribute
                          dColor,         // interior color attribute
                          ic              // pointer to list of control definitions
                        ) ;

      //* Instantiate the dialog window *
      NcDialog* dp = new NcDialog ( dInit ) ;
   
      //* Open the dialog window *
      if ( (dp->OpenWindow()) == OK )
      {
         uiInfo Info ;                 // user interface data returned here
         short  icIndex = ZERO ;       // index of control with input focus
         bool   done = false ;         // loop control
         while ( ! done )
         {
            //*******************************************
            //* If focus is currently on a Pushbutton   *
            //*******************************************
            if ( ic[icIndex].type == dctPUSHBUTTON )
            {
               icIndex = dp->EditPushbutton ( Info ) ;
               if ( Info.dataMod != false )
               {
                  if ( Info.ctrlIndex == ccPB )
                  {
                     dp->SetDialogObscured () ;
                     winPos wp( trgULY, trgULX ) ;
                     CapSimple ( wp ) ;
                     sWin->RefreshWin () ;
                     ClearTerminalWindow () ;
                     sWin->RefreshWin () ;
                     dp->RefreshWin () ;
                  }
                  else if ( Info.ctrlIndex == cdPB )
                  {
                     dp->SetDialogObscured () ;
                     winPos wp( trgULY, trgULX ) ;
                     CapComplex ( wp ) ;
                     sWin->RefreshWin () ;
                     ClearTerminalWindow () ;
                     sWin->RefreshWin () ;
                     dp->RefreshWin () ;
                  }
                  else if ( Info.ctrlIndex == donePB )
                     done = true ;
               }
            }
   
            //* Advance the input focus *
            if ( done == false )
            {
               if ( Info.keyIn == nckSTAB )
                  icIndex = dp->PrevControl () ; 
               else
                  icIndex = dp->NextControl () ;
            }
         }  // while()
      }
      if ( dp != NULL )                      // close the window
         delete ( dp ) ;
   }
   else
   {
      gString gs( "Sorry, this test requires that terminal be at "
                  "least %hd rows by %hd columns.", &trgROWS, &trgCOLS ) ;
      StatWin ( gs.ustr(), nc.cyG ) ;
      sleep ( 3 ) ;
   }

}  //* End Test07() *

//*************************
//*  T07_ControlUpdate    *
//*************************
//******************************************************************************
//* This is a callback method for manually updating the controls in the        *
//* Test07() dialog.                                                           *
//*                                                                            *
//*  For this test, the following are updated by the callback method:          *
//*   1.                                                                       *
//*   2.                                                                       *
//*   3.                                                                       *
//*                                                                            *
//* Input  : currIndex: index of control that currently has focus              *
//*          wkey     : user's key input data                                  *
//*          firstTime: the EstablishCallback() method calls this method once  *
//*                     with firstTime==true, to perform any required          *
//*                     initialization. Subsequently, the NcDialog class       *
//*                     always calls with firstTime==false.                    *
//* Returns: OK                                                                *
//******************************************************************************
//* Important Note: This method makes some intimate assumptions about what     *
//* is happening in the method that established the callback. If changes are   *
//* made in the establishing method that affect the callback functionality,    *
//* be sure to update this method to address those changes.                    *
//*                                                                            *
//* Programmer's Note: See dialog control index constants at file scope above. *
//******************************************************************************

short T07_ControlUpdate ( const short currIndex, const wkeyCode wkey, bool firstTime )
{

   return OK ;

}  //* End T07_ControlUpdate() *

//*************************
//*       Test08          *
//*************************
//******************************************************************************
//* Test NcWindow class line drawing methods. Note that these are accessed     *
//* through the NcDialog object because NcDialog is a derived class of         *
//* NcWindow.                                                                  *
//*                                                                            *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void Test08 ( void )
{
const short dialogROWS = 20 ;                // display lines
const short dialogCOLS = 60 ;                // display columns
const short controlsDEFINED = 13 ;           // # of controls in dialog
const short minNeededLines = 34,             // min hight for ACS window display
            minNeededCols  = 90 ;            // min width for ACS window display
enum T8controls : short                      // control indices 
{ 
   T8dispePush = ZERO, T8dispcPush, T8donePush, T8bwRadio, T8reRadio, 
   T8grRadio, T8brRadio, T8blRadio, T8maRadio, T8cyRadio, T8gyRadio, 
   T8revRadio, T8boldRadio
} ;
attr_t   dColor = nc.cyR ;
short    ctrY    = termRows/2,               // dialog center in Y
         ctrX    = termCols/2,               // dialog center in X
         ulY     = ctrY - dialogROWS/2-7,    // upper left corner in Y
         ulX     = ctrX - dialogCOLS/2-24 ;  // upper left corner in X
   //* Adjust dialog position to accomodate the rather large window *
   if ( ulY < minTestULY )
      ulY = minTestULY ;
   if ( ulX < ZERO )
      ulX = ZERO ;

   StatWin ( "Opening Test 8 - Test Line Drawing Methods" ) ;

InitCtrl ic[controlsDEFINED] =   // array of dialog control info
{
   {  //* 'DISPLAY EXAMPLES' pushbutton  - - - - - - - - - - - -   T8dispePush *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(dialogROWS - 6),        // ulY:       upper left corner in Y
      short(dialogCOLS / 2 - 15),   // ulX:       upper left corner in X
      1,                            // lines:     control lines
      30,                           // cols:      control columns
      "  DISPLAY LINE DRAW EXAMPLES  ",// dispText:  
      nc.gyR,                       // nColor:    non-focus color
      nc.reG,                       // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "",                           // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[1],                       // nextCtrl:  link in next structure
   },
   {  //* 'DISPLAY EXAMPLES' pushbutton  - - - - - - - - - - - -   T8dispePush *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(dialogROWS - 4),        // ulY:       upper left corner in Y
      short(dialogCOLS / 2 - 15),   // ulX:       upper left corner in X
      1,                            // lines:     control lines
      30,                           // cols:      control columns
      "  DISPLAY LINE DRAW CHAR SET  ",// dispText:  
      nc.gyR,                       // nColor:    non-focus color
      nc.reG,                       // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "",                           // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[2],                       // nextCtrl:  link in next structure
   },
   {  //* 'DONE' pushbutton - - - - - - - - - - - - - - - - - - - - - donePush *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(dialogROWS - 2),        // ulY:       upper left corner in Y
      short(dialogCOLS / 2 - 4),    // ulX:       upper left corner in X
      1,                            // lines:     control lines
      8,                            // cols:      control columns
      "  DONE  ",                   // dispText:  
      nc.gyR,                       // nColor:    non-focus color
      nc.reG,                       // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "",                           // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[3],                       // nextCtrl:  link in next structure
   },
   {  //* 'black-on-wihte'   Radio Button   - - - - - - - - - - -   T8bwRadio  *
      dctRADIOBUTTON,               // type:      
      rbtS5a,                       // rbSubtype: standard, 5-wide
      false,                        // rbSelect:  initially set
      2,                            // ulY:       upper left corner in Y
      short(dialogCOLS / 2 - 13),   // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a)
      dColor,                       // nColor:    non-focus color
      nc.bw,                        // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "Black-on-White",             // label:     
      ZERO,                         // labY:      
      7,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[4],                       // nextCtrl:  link in next structure
   },
   {  //* 'red-on white'     Radio Button   - - - - - - - - - - -   T8reRadio  *
      dctRADIOBUTTON,               // type:      
      rbtS5a,                       // rbSubtype: standard, 5-wide
      false,                        // rbSelect:  initially set
      short(ic[T8bwRadio].ulY+1),   // ulY:       upper left corner in Y
      short(ic[T8bwRadio].ulX),     // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a)
      dColor,                       // nColor:    non-focus color
      nc.bw,                        // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "Red-on-White",               // label:     
      ZERO,                         // labY:      
      7,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[5],                       // nextCtrl:  link in next structure
   },
   {  //* 'green-on white'   Radio Button   - - - - - - - - - - -   T8grRadio  *
      dctRADIOBUTTON,               // type:      
      rbtS5a,                       // rbSubtype: standard, 5-wide
      false,                        // rbSelect:  initially set
      short(ic[T8reRadio].ulY+1),   // ulY:       upper left corner in Y
      short(ic[T8reRadio].ulX),     // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a)
      dColor,                       // nColor:    non-focus color
      nc.bw,                        // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "Green-on-White",             // label:     
      ZERO,                         // labY:      
      7,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[6],                       // nextCtrl:  link in next structure
   },
   {  //* 'brown-on white'   Radio Button   - - - - - - - - - - -   T8brRadio  *
      dctRADIOBUTTON,               // type:      
      rbtS5a,                       // rbSubtype: standard, 5-wide
      false,                        // rbSelect:  initially set
      short(ic[T8grRadio].ulY+1),   // ulY:       upper left corner in Y
      short(ic[T8grRadio].ulX),     // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a)
      dColor,                       // nColor:    non-focus color
      nc.bw,                        // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "Brown-on-White",             // label:     
      ZERO,                         // labY:      
      7,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[7],                       // nextCtrl:  link in next structure
   },
   {  //* 'blue-on-white'    Radio Button   - - - - - - - - - - -   T8blRadio  *
      dctRADIOBUTTON,               // type:      
      rbtS5a,                       // rbSubtype: standard, 5-wide
      true,                         // rbSelect:  initially set
      short(ic[T8brRadio].ulY+1),   // ulY:       upper left corner in Y
      short(ic[T8brRadio].ulX),     // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a)
      dColor,                       // nColor:    non-focus color
      nc.bw,                        // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "Blue-on-White",              // label:     
      ZERO,                         // labY:      
      7,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[8],                       // nextCtrl:  link in next structure
   },
   {  //* 'magenta-on-white' Radio Button   - - - - - - - - - - -   T8maRadio  *
      dctRADIOBUTTON,               // type:      
      rbtS5a,                       // rbSubtype: standard, 5-wide
      false,                        // rbSelect:  initially set
      short(ic[T8blRadio].ulY+1),   // ulY:       upper left corner in Y
      short(ic[T8blRadio].ulX),     // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a)
      dColor,                       // nColor:    non-focus color
      nc.bw,                        // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "Magenta-on-White",           // label:     
      ZERO,                         // labY:      
      7,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[9],                       // nextCtrl:  link in next structure
   },
   {  //* 'cyan-on-white'    Radio Button   - - - - - - - - - - -   T8cyRadio  *
      dctRADIOBUTTON,               // type:      
      rbtS5a,                       // rbSubtype: standard, 5-wide
      false,                        // rbSelect:  initially set
      short(ic[T8maRadio].ulY+1),   // ulY:       upper left corner in Y
      short(ic[T8maRadio].ulX),     // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a)
      dColor,                       // nColor:    non-focus color
      nc.bw,                        // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "Cyan-on-White",              // label:     
      ZERO,                         // labY:      
      7,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[10],                      // nextCtrl:  link in next structure
   },
   {  //* 'grey-on-white'    Radio Button   - - - - - - - - - - -   T8gyRadio  *
      dctRADIOBUTTON,               // type:      
      rbtS5a,                       // rbSubtype: standard, 5-wide
      false,                        // rbSelect:  initially set
      short(ic[T8cyRadio].ulY+1),   // ulY:       upper left corner in Y
      short(ic[T8cyRadio].ulX),     // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a)
      dColor,                       // nColor:    non-focus color
      nc.bw,                        // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "Grey-on-White",              // label:     
      ZERO,                         // labY:      
      7,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[11],                      // nextCtrl:  link in next structure
   },
   {  //* 'reverse'          Radio Button   - - - - - - - - - - -   T8revRadio *
      dctRADIOBUTTON,               // type:      
      rbtS5a,                       // rbSubtype: standard, 5-wide
      true,                         // rbSelect:  initially set
      short(ic[T8gyRadio].ulY+2),   // ulY:       upper left corner in Y
      short(ic[T8gyRadio].ulX),     // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a)
      dColor,                       // nColor:    non-focus color
      nc.bw,                        // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "Reverse Fgnd/Bkgnd",         // label:     
      ZERO,                         // labY:      
      7,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[12],                      // nextCtrl:  link in next structure
   },
   {  //* 'bold'             Radio Button   - - - - - - - - - - -  T8boldRadio *
      dctRADIOBUTTON,               // type:      
      rbtS5a,                       // rbSubtype: standard, 5-wide
      false,                        // rbSelect:  initially set
      short(ic[T8revRadio].ulY+1),  // ulY:       upper left corner in Y
      short(ic[T8revRadio].ulX),    // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a)
      dColor,                       // nColor:    non-focus color
      nc.bw,                        // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "Bold Text",                  // label:     
      ZERO,                         // labY:      
      7,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      NULL,                         // nextCtrl:  link in next structure
   },
} ;
   //* Initial parameters for dialog window *
   InitNcDialog dInit( dialogROWS,     // number of display lines
                       dialogCOLS,     // number of display columns
                       ulY,            // Y offset from upper-left of terminal 
                       ulX,            // X offset from upper-left of terminal 
                       "  Some Dialog Window Under Development  ", // dialog title
                       ncltSINGLE,     // border line-style
                       dColor,         // border color attribute
                       dColor,         // interior color attribute
                       ic              // pointer to list of control definitions
                     ) ;

   //* Instantiate the dialog window *
   NcDialog* dp = new NcDialog ( dInit ) ;

   //* Open the dialog window *
   if (   (termRows >= minNeededLines && termCols >= minNeededCols) 
       && ((dp->OpenWindow()) == OK) )
   {
      dp->RefreshWin () ;

      //* Create an exclusive OR group for color-attribute radio buttons *
      short XorGroup[] = { T8bwRadio, T8reRadio, T8grRadio, T8brRadio, 
                           T8blRadio, T8maRadio, T8cyRadio, T8gyRadio, -1 } ;
      dp->GroupRadiobuttons ( XorGroup ) ;

      attr_t callColor = nc.blR ;   // color attribute for call to display method
      uiInfo Info ;                 // user interface data returned here
      short  icIndex = ZERO ;       // index of control with input focus
      bool   done = false ;         // loop control
      while ( ! done )
      {
         //*******************************************
         //* If focus is currently on a Pushbutton   *
         //*******************************************
         if ( ic[icIndex].type == dctPUSHBUTTON )
         {
            icIndex = dp->EditPushbutton ( Info ) ;
            if ( Info.dataMod != false )
            {
               if ( Info.ctrlIndex == T8dispePush )
               {
                  //* Display examples of the Line-drawing characters    *
                  //* Note that the coordinates for display are absolute *
                  //* screen coordinates, not dialog offsets.            *
                  winPos   wPos(ulY+2,ulX+5) ;
                  dp->DisplayLineDrawingExamples ( wPos, callColor ) ;
               }
               else if ( Info.ctrlIndex == T8dispcPush )
               {
                  //* Display examples of the Line-drawing characters    *
                  //* Note that the coordinates for display are absolute *
                  //* screen coordinates, not dialog offsets.            *
                  winPos   wPos(ulY+2,ulX+5) ;
                  dp->DisplayLineDrawingSet ( wPos, callColor ) ;
               }
               else if ( Info.ctrlIndex == T8donePush )
                  done = true ;
            }
         }
         //*******************************************
         //* If focus is currently on a RadioButton  *
         //*******************************************
         else if ( ic[icIndex].type == dctRADIOBUTTON )
         {
            icIndex = dp->EditRadiobutton ( Info ) ;
            if ( Info.dataMod != false )
            {
               //* If 'selected' button is a member of a button group *
               if ( Info.selMember < MAX_DIALOG_CONTROLS )
               {
                  switch ( Info.selMember )
                  {
                     case T8bwRadio:   callColor = nc.bw ;  break ;
                     case T8reRadio:   callColor = nc.re ;  break ;
                     case T8grRadio:   callColor = nc.gr ;  break ;
                     case T8brRadio:   callColor = nc.br ;  break ;
                     case T8maRadio:   callColor = nc.ma ;  break ;
                     case T8cyRadio:   callColor = nc.cy ;  break ;
                     case T8gyRadio:   callColor = nc.gy ;  break ;
                     case T8blRadio:
                     default:
                        callColor = nc.bl ;                 break ;
                  }
                  //* Test the 'reverse fg/bg' flag *
                  bool  rev ;
                  dp->GetRadiobuttonState ( T8revRadio, rev ) ;
                  if ( rev != false )
                     callColor |= ncrATTR ;
               }
               //* 'Reverse' stand-alone button *
               else if ( Info.ctrlIndex == T8revRadio )
               {
                  if ( Info.isSel )
                     callColor |= ncrATTR ;
                  else
                     callColor &= ~ncrATTR ;
               }
               //* 'Bold' stand-alone button *
               else if ( Info.ctrlIndex == T8boldRadio )
               {
                  if ( Info.isSel )
                     callColor |= ncbATTR ;
                  else
                     callColor &= ~ncbATTR ;
               }
            }
         }

         //* Advance the input focus *
         if ( done == false )
         {
            if ( Info.keyIn == nckSTAB )
               icIndex = dp->PrevControl () ; 
            else
               icIndex = dp->NextControl () ;
         }
      }  // while()
   }  // OpenWindow()
   else
   {  //* Most likely cause of dialog not opening is that the terminal window  *
      //* is too small. Give user a clue...                                    *
      short neededLines = minNeededLines, 
            neededCols  = minNeededCols ;
      gString gs ;
      gs.compose ( L" Sorry, size of terminal window must be at least %hd x %hd for this test. ", 
                   &neededCols, &neededLines ) ;
      StatWin ( gs.gstr(), nc.cyG ) ;
      sleep ( 5 ) ;
   }

   if ( dp != NULL )                         // close the window
      delete ( dp ) ;

}  //* End Test08() *

//*************************
//*  T08_ControlUpdate    *
//*************************
//******************************************************************************
//* This is a callback method for manually updating the controls in the        *
//* Test08() dialog.                                                           *
//*                                                                            *
//*  For this test, the following are updated by the callback method:          *
//*   1.                                                                       *
//*   2.                                                                       *
//*   3.                                                                       *
//*                                                                            *
//* Input  : currIndex: index of control that currently has focus              *
//*          wkey     : user's key input data                                  *
//*          firstTime: the EstablishCallback() method calls this method once  *
//*                     with firstTime==true, to perform any required          *
//*                     initialization. Subsequently, the NcDialog class       *
//*                     always calls with firstTime==false.                    *
//* Returns: OK                                                                *
//******************************************************************************
//* Important Note: This method makes some intimate assumptions about what     *
//* is happening in the method that established the callback. If changes are   *
//* made in the establishing method that affect the callback functionality,    *
//* be sure to update this method to address those changes.                    *
//*                                                                            *
//* Programmer's Note: See dialog control index constants at file scope above. *
//******************************************************************************

short T08_ControlUpdate ( const short currIndex, const wkeyCode wkey, bool firstTime )
{

   return OK ;

}  //* End T08_ControlUpdate() *

//*************************
//*       Test09          *
//*************************
//******************************************************************************
//* Display the Alternate Character Set using the NCurses debugging method.    *
//* Displays all the printing characters in the ALC in each of the basic       *
//* reverse-video colors.                                                      *
//*                                                                            *
//* See also Test10().                                                         *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void Test09 ( void )
{

   StatWin ( "Opening Test 9 - Display Alternate Character Set: NCurses" ) ;
   nc.DisplayAlternateFont ( 3, 4 ) ;
   ClearTerminalWindow () ;

}  //* End Test09() *

//*************************
//*       Test10          *
//*************************
//******************************************************************************
//* Display the Alternate Character Set using the NcDialog debugging method.   *
//* Displays all the printing characters in the ACS with additional information*
//* and in a pleasant format.                                                  *
//*                                                                            *
//* Also displays the wchar_t 'wide' character equivalents to the Alternate    *
//* Character Set.                                                             *
//*                                                                            *
//* See also Test09().                                                         *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void Test10 ( void )
{
const short dialogROWS = 20 ;                // display lines
const short dialogCOLS = 60 ;                // display columns
const short controlsDEFINED = 3 ;            // # of controls in dialog
const short minNeededLines = 34,             // min hight for ACS window display
            minNeededCols  = 90 ;            // min width for ACS window display
const short acsPush = ZERO,                  // control indices
            wcsPush = 1,
            donePush = 2 ;
attr_t   dColor = nc.cyR ;
short    ctrY    = termRows/2,               // dialog center in Y
         ctrX    = termCols/2,               // dialog center in X
         ulY     = ctrY - dialogROWS/2-7,    // upper left corner in Y
         ulX     = ctrX - dialogCOLS/2-24 ;  // upper left corner in X
   //* Adjust dialog position to accomodate the rather large ACS window *
   if ( ulY < minTestULY )
      ulY = minTestULY ;
   if ( ulX < ZERO )
      ulX = ZERO ;

InitCtrl ic[controlsDEFINED] =      // control initialization structures
{
   {  //* ACS pushbutton - - - - - - - - - - - - - - - - - - - - - - - acsPush *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(dialogROWS - 2),        // ulY:       upper left corner in Y
      short(dialogCOLS - 54),       // ulX:       upper left corner in X
      1,                            // lines:     control lines
      20,                           // cols:      control columns
      " ALTERNATE CHAR SET ",       // dispText:  
      nc.gyR,                       // nColor:    non-focus color
      nc.reG,                       // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "",                           // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[1],                       // nextCtrl:  link in next structure
   },
   {  //* WCS pushbutton - - - - - - - - - - - - - - - - - - - - - - - wcsPush *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(dialogROWS - 2),        // ulY:       upper left corner in Y
      short(dialogCOLS - 32),       // ulX:       upper left corner in X
      1,                            // lines:     control lines
      20,                           // cols:      control columns
      " 'WIDE' EQUIVALENTS ",       // dispText:  
      nc.gyR,                       // nColor:    non-focus color
      nc.reG,                       // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "",                           // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[2],                       // nextCtrl:  link in next structure
   },
   {  //* DONE pushbutton - - - - - - - - - - - - - - - - - - - - - - donePush *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(dialogROWS - 2),        // ulY:       upper left corner in Y
      short(dialogCOLS - 10),       // ulX:       upper left corner in X
      1,                            // lines:     control lines
      8,                            // cols:      control columns
      "  DONE  ",                   // dispText:  
      nc.gyR,                       // nColor:    non-focus color
      nc.reG,                       // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "",                           // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      NULL,                         // nextCtrl:  link in next structure
   },
} ;

   StatWin ( "Opening Test 10 - Display Alternate Character Set: NcDialog" ) ;

   //* Initial parameters for dialog window *
   InitNcDialog dInit( dialogROWS,     // number of display lines
                       dialogCOLS,     // number of display columns
                       ulY,            // Y offset from upper-left of terminal 
                       ulX,            // X offset from upper-left of terminal 
                       "  Some Dialog Window Under Development  ", // dialog title
                       ncltSINGLE,     // border line-style
                       dColor,         // border color attribute
                       dColor,         // interior color attribute
                       ic              // pointer to list of control definitions
                     ) ;

   //* Instantiate the dialog window *
   NcDialog* dp = new NcDialog ( dInit ) ;

   //* Open the dialog window *
   if (   (termRows >= minNeededLines && termCols >= minNeededCols) 
       && ((dp->OpenWindow()) == OK) )
   {
      dp->RefreshWin () ;
      
      //* Display examples of the Alternate Character Set (ACS) characters*
      //* OR 'wide' equivalent characters.                                *
      //* Note that the coordinates for display are absolute              * 
      //* screen coordinates, not dialog offsets.                         *
      winPos   wPos(ulY+3,ulX+5) ;

      uiInfo Info ;                 // user interface data returned here
      bool   done = false ;         // loop control
      while ( ! done )
      {
         dp->EditPushbutton ( Info ) ;
         if ( Info.dataMod != false )
         {
            if ( Info.ctrlIndex == acsPush )
               dp->DisplayAltCharacterSet ( wPos, dColor ) ;
            else if ( Info.ctrlIndex == wcsPush )
               dp->DisplayWideCharacterSet ( wPos, dColor ) ;
            else if ( Info.ctrlIndex == donePush )
               done = true ;
         }
         //* Move focus to next/previous control *
         if ( done == false )
         {
            if ( Info.keyIn == nckSTAB )
               dp->PrevControl () ;
            else
               dp->NextControl () ;
         }
      }
   }
   else
   {  //* Most likely cause of dialog not opening is that the terminal window  *
      //* is too small. Give user a clue...                                    *
      short neededLines = minNeededLines, 
            neededCols  = minNeededCols ;
      gString gs ;
      gs.compose ( L" Sorry, size of terminal window must be at least %hd x %hd for this test. ", 
                   &neededCols, &neededLines ) ;
      StatWin ( gs.gstr(), nc.cyG ) ;
      sleep ( 5 ) ;
   }

   if ( dp != NULL )                         // close the window
      delete ( dp ) ;

}  //* End Test10() *

//*************************
//*  GetCommandLineArgs   *
//*************************
//******************************************************************************
//* Capture user's command-line arguments.                                     *
//*                                                                            *
//* Valid Arguments:                                                           *
//*   Test number: -[1 | 2 | 3 | 4 | 5 | 6 | 7 | 8 | 9 | 10 | 11]              *
//*   Alt Locale : -A, -a     Specify an alternate 'locale' for interpretation *
//*                           of character output and input conversions.       *
//*   Alt Colors : -M, -m     Start application in monochrome mode             *
//*                           (undocumented sub-option, -Mf :             )    *
//*                           (start app using color map data in cmap.txt )    *
//*   Help       : -H, -h     Start application with display of Help Dialog.   *
//*                                                                            *
//* Input  : commArgs class object (by reference)                              *
//*                                                                            *
//* Returns: test number to be executed OR -1 if no test specified             *
//******************************************************************************

short GetCommandLineArgs ( commArgs& ca )
{
   if ( ca.argCount > 1 )
   {
      short v = 1,         // argv index
            i ;            // current arg index

      while ( v < ca.argCount )
      {
         i = ZERO ;
         if ( ca.argList[v][i] == '-' )  ++i ;     // step over dashes
         if ( ca.argList[v][i] == '-' )  ++i ;

         //* Test number specified? *
         if ( ca.argList[v][i] >= '1' && ca.argList[v][i] <= '9' )
         {
            sscanf ( &ca.argList[v][i], "%hd", &ca.testNum ) ;
            --ca.testNum ; // make it a zero-based index

            if ( ca.testNum > indexT10 )
            {
               //* Special test for Test11() *
               if ( ca.testNum == indexHELP )
                  ca.testNum = indexCOUNT ;
               //* Else, value out of range *
               else
                  ca.testNum = indexHELP ;
            }
         }

         //* Alternate locale specified? *
         else if ( ca.argList[v][i] == 'A' || ca.argList[v][i] == 'a' )
         {  //* If user specified an alternate locale string, use it instead *
            //* of the default locale found in the terminal environment.     *
            ++i ;
            if ( ca.argList[v][i] != NULLCHAR )
            {
               if ( ca.argList[v][i] == '=' )
                  ++i ;
               strncpy ( ca.localeName, &ca.argList[v][i], 30 ) ;
               ca.altLocale = true ;
            }
            else if ( ca.argCount > 2 )
            {
               strncpy ( ca.localeName, ca.argList[v+1], 30 ) ;
               ca.altLocale = true ;
            }
         }

         //* Option to start the NCurses Engine in monochrome   *
         //* mode OR to specify a custom Color Map.             *
         //* (useful only for testing color/attribute mapping). *
         else if ( ca.argList[v][i] == 'M' || ca.argList[v][i] == 'm' )
         {
            ca.altColors = true ;
            if ( ca.argList[v][i+1] == 'f' )
               ca.altColorMap = true ;
         }

         //* Else, request for help OR invalid command-line argument *
         else
         {
            ca.testNum = indexHELP ;
            break ;
         }

         ++v ;       // next command-line argument
      }
   }
   return ca.testNum ;
   
}  //* End GetCommandLineArgs() *

//***********************
//*      HelpAbout      *
//***********************
//****************************************************************************
//* Display the application's Help-About window.                             *
//* This is a simple informational dialog with a 'CLOSE' button.             *
//*                                                                          *
//*                                                                          *
//* Input Values : ctrY : position at which to center dialog in Y            *
//*                ctrX : position at which to center dialog in X            *
//*                                                                          *
//* Return Value : none                                                      *
//****************************************************************************

void HelpAbout ( short ctrY, short ctrX )
{
static const short dialogROWS = 24 ;
static const short dialogCOLS = 70 ;
static const short controlsDEFINED = 3 ;
const gString DialogText( 
   "Dialog2, \n"
   "A simple utility for development and testing of the NcDialog family\n"
   "of classes based on the Linux/UNIX ncurses library.\n"
   " \n"
   "Copyright (c) %s Mahlon R. Smith, The Software Samurai\n"
   "                        Beijing University of Technology\n"
   " \n"
   "Command Line arguments (optional):\n"
   "    -n      where 'n' is a test number\n"
   "            (Note: Test 11 executes before main dialog is opened.)\n"
   "    -A, -a  specify an alternate 'locale' setting to override\n"
   "            locale specified in terminal environment\n"
   "            (for complete list, run console cmd:  locale -a)\n"
   "             Ex: ./Dialog2 -Avi_VN.utf8 ./Dialog2 -A yi_US.utf8\n"
   "    -M, -m  start in monochrome mode (testing only - ugly!)\n"
   "    -H, -h  Help (this dialog)\n"
   " \n"
   "For bugs, suggestions, possible praise: http://www.SoftwareSam.us\n"
   " \n"
   "Tools: GNU G++ v:13 (C++11) under Fedora and Ubuntu Linux.\n"
   "       Requires ncursesw library v:6.4+ ", crYears
);//xxxxxxxxx0xxxxxxxxx0xxxxxxxxx0xxxxxxxxx0xxxxxxxxx0xxxxxxxxx0xxxxxxxx|

enum ctrls { closePUSH = ZERO, classPUSH, appverTBOX } ;
short       ulY = ctrY - dialogROWS / 2,
            ulX = ctrX - dialogCOLS / 2 ;
attr_t      dColor = nc.blR ;

   //* Format application version string for display *
   gString verString( " %s ", AppVersion ) ;


InitCtrl ic[controlsDEFINED] =      // array of dialog control info
{
   {  //* 'CLOSE' pushbutton  - - - - - - - - - - - - -  closePUSH *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(dialogROWS - 2),        // ulY:       upper left corner in Y
      short(dialogCOLS / 2 - 4),    // ulX:       upper left corner in X
      1,                            // lines:     control lines
      7,                            // cols:      control columns
      " CLOSE ",                    // dispText:  
      nc.gyR,                       // nColor:    non-focus color
      nc.reR,                       // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "",                           // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[1],                       // nextCtrl:  link in next structure
   },
   {  //* 'CLASS VERSION' pushbutton  - - - - - - - - -  classPUSH *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(dialogROWS - 2),        // ulY:       upper left corner in Y
      short(dialogCOLS - 25),       // ulX:       upper left corner in X
      1,                            // lines:     control lines
      24,                           // cols:      control columns
      " DISPLAY CLASS VERSIONS ",   // dispText:  
      nc.gyR,                       // nColor:    non-focus color
      nc.reR,                       // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "",                           // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[2],                       // nextCtrl:  link in next structure
   },
   {  //* Non-active textbox for app. version  - - - -  appverTBOX *
      dctTEXTBOX,                   // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      1,                            // ulY:       upper left corner in Y
      19,                           // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      short(verString.gscols()),    // cols:      control columns
      verString.ustr(),             // dispText:  (value of t3indexO3)
      nc.bl,                        // nColor:    non-focus color
      nc.bl,                        // fColor:    focus color
      tbPrint,                      // filter:    any printing character
      "Version:",                   // label:     
      ZERO,                         // labY:      
      -9,                           // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      false,                        // active:    cannot gain focus
      NULL,                         // nextCtrl:  link in next structure
   },
} ;

   //* Initial parameters for dialog window *
   InitNcDialog dInit( dialogROWS,     // number of display lines
                       dialogCOLS,     // number of display columns
                       ulY,            // Y offset from upper-left of terminal 
                       ulX,            // X offset from upper-left of terminal 
                       NULL,           // dialog title
                       ncltSINGLE,     // border line-style
                       dColor,         // border color attribute
                       dColor,         // interior color attribute
                       ic              // pointer to list of control definitions
                     ) ;

   //* Instantiate the dialog window *
   NcDialog* dp = new NcDialog ( dInit ) ;

   //* Open the dialog window *
   if ( (dp->OpenWindow()) == OK )
   {
      dp->WriteParagraph ( 1, 1, DialogText, dColor ) ;
      dp->RefreshWin () ;

      uiInfo Info ;                 // user interface data returned here
      bool   done = false ;         // loop control
      while ( ! done )
      {
         dp->EditPushbutton ( Info ) ;
         if ( Info.dataMod != false )
         {
            if ( Info.ctrlIndex == closePUSH )
               done = true ;
            else if ( Info.ctrlIndex == classPUSH )
            {  //* Open a dialog to display class version numbers and other    *
               //* interesting data. Current dialog will be obscured, so save  *
               //* its display data.                                           *
               dp->SetDialogObscured () ;
               DisplayClassInfo ( (ulY + 4), (ctrX - 20), 15, 40 ) ;
               dp->NextControl () ;
               dp->RefreshWin () ;
            }
         }
         if ( done == false )
         {
            if ( Info.keyIn == nckSTAB )
               dp->PrevControl () ; 
            else if ( Info.keyIn != ZERO )
               dp->NextControl () ;
         }
      }
   }
   if ( dp != NULL )
      delete ( dp ) ;                        // close the window

}  //* End HelpAbout() *

//***********************
//* ClearTerminalWindow *
//***********************
//****************************************************************************
//* Clear the terminal window and display the application title.             *
//*                                                                          *
//*                                                                          *
//* Input Values : none                                                      *
//*                                                                          *
//* Return Value : none                                                      *
//****************************************************************************

void ClearTerminalWindow ( void )
{
   winPos   wpos( 0, 0 ) ;
   nc.ClearScreen () ;
   gString gsTitle( "  Dialog2 - v:%s (c) %s Mahlon R. Smith    : Locale: %s      ",
                    AppVersion, crYears, nc.GetLocale() ) ;
   nc.WriteString ( wpos, gsTitle.gstr(), nc.bw | ncuATTR ) ;
   nc.RefreshScreen () ;

}  //* End ClearTerminalWindow() *

//*************************
//*                       *
//*************************
//******************************************************************************
//*                                                                            *
//*                                                                            *
//*                                                                            *
//* Input  :                                                                   *
//*                                                                            *
//* Returns:                                                                   *
//******************************************************************************

