//********************************************************************************
//* File       : NcwScroll.cpp                                                   *
//* Author     : Mahlon R. Smith                                                 *
//*              Copyright (c) 2006-2025 Mahlon R. Smith, The Software Samurai   *
//*                 GNU GPL copyright notice located in NcWindow.hpp             *
//* Date       : 21-Mar-2021                                                     *
//* Version    : (see NcWindowVersion string in NcWindow.cpp)                    *
//*                                                                              *
//* Description: This module includes the methods of the NcWindow class that     *
//* address the manipulation of scrolling-data in a window.                      *
//*                                                                              *
//*                                                                              *
//* Development Tools: See NcWindow.cpp.                                         *
//********************************************************************************
//* Version History (most recent first):                                         *
//*   See version history in NcWindow.cpp.                                       *
//********************************************************************************
//* Programmer's Notes:                                                          *
//*                                                                              *
//*                                                                              *
//********************************************************************************

//*****************
//* Include Files *
//*****************

#include "GlobalDef.hpp"               //* General definitions

#ifndef NCURSES_INCLUDED
#include "NCurses.hpp"
#endif

#ifndef NCWINDOW_INCLUDED
#include "NcWindow.hpp"
#endif

//*********************
//* Local Definitions *
//*********************


//*************************
//*      PaintData        *
//*************************
//******************************************************************************
//* Write into the window the display data to be scrolled and initialize       *
//* the tracking variables. See also RepaintData().                            *
//* Note that the data item specified by the hIndex parameter is positioned    *
//* as near to top of window as possible, even when withHilte == false. This   *
//* guarentees that the specified item will be displayed when the window is    *
//* first opened.                                                              *
//*                                                                            *
//* This version of PaintData() is for single-color data only.                 *
//*                                                                            *
//* Note: It is assumed that the caller has properly sized the display         *
//* strings to fit within the window's width. If not, we may experience        *
//* unsightly or uneven right edges.                                           *
//*                                                                            *
//* Input  : dPtr   : pointer to an array of character pointers                *
//*          count  : number of data items                                     *
//*          hIndex : index of highlighted data item                           *
//*          color  : text color attribute                                     *
//*          hilite : (optional, true by default)                              *
//*                   hilight currently indexed item                           *
//*          rtlFmt : (optional, false by default)                             *
//*                   draw data as RTL (right-to-left) text                    *
//*                                                                            *
//* Returns: index of currently-highlighted item                               *
//******************************************************************************

short NcWindow::PaintData ( const char* dPtr[], short count, short hIndex, 
                            attr_t color, bool hilite, bool rtlFmt )
{
   short lineIndex, y = ZERO, x = rtlFmt ? (this->wCols - 1) : ZERO ;

   //* Save references to caller's data *
   this->scrollData.dptr = dPtr ;            // text data
   this->scrollData.cptr = NULL ;            // no color data
   this->scrollData.monocolor = color ;      // monochrome color
   this->scrollData.count = count ;          // item count
   this->scrollData.rtl = rtlFmt ;           // indicates LTR or RTL data
   if ( hIndex < ZERO || hIndex >= count )   // error correction test
      hIndex = ZERO ;
   this->scrollData.hIndex = hIndex ;        // index of highlighted item
   this->scrollData.tIndex = ZERO ;          // initial index of top line of data
   if ( count >= wLines - 1 )                // initial index of bottom line of data
      this->scrollData.bIndex = wLines - 1 ;
   else
      this->scrollData.bIndex = count - 1 ;
   if ( hIndex <= this->scrollData.bIndex )  // if highlight is in first page of list
      this->scrollData.hRow = hIndex ;       // row of highlighted item
   else
   {
      //* Advance top- and bottom-line indices until highlighted line is visible *
      while ( (this->scrollData.bIndex < hIndex) && 
              (this->scrollData.bIndex < (count-1)) )
      {
         ++this->scrollData.tIndex ;
         ++this->scrollData.bIndex ;
      }
      scrollData.hRow = wLines - 1 ;
      //* Bring highlight as close to top of window as possible *
      while ( (this->scrollData.tIndex < hIndex) && 
              (this->scrollData.bIndex < (count-1)) )
      {
         ++this->scrollData.tIndex ;
         ++this->scrollData.bIndex ;
         --this->scrollData.hRow ;
      }
   }

   this->AcquireOutputLock () ;     // acquire exclusive access to output stream

   this->ClearWin ( true, false ) ; // clear the old data (no display update)
   for ( lineIndex = this->scrollData.tIndex ; 
         lineIndex <= this->scrollData.bIndex ; lineIndex++ )
      this->WriteString ( y++, x, dPtr[lineIndex], this->scrollData.monocolor,
                          false, this->scrollData.rtl ) ;
   if ( hilite != false )
      this->HilightItem ( true ) ;
   this->RefreshWin () ;                     // refresh the display

   this->ReleaseOutputLock () ;     // release lock on output stream
   return hIndex ;

}  //* End PaintData() *

//*************************
//*      PaintData        *
//*************************
//******************************************************************************
//* Write into the window the display data to be scrolled and initialize       *
//* the tracking variables. See also RepaintData().                            *
//* Note that the data item specified by the hIndex parameter is positioned    *
//* as near to top of window as possible, even when withHilte == false. This   *
//* guarentees that the specified item will be displayed when the window is    *
//* first opened.                                                              *
//*                                                                            *
//* This version of PaintData() is for multi-color data (one color per         *
//* line).                                                                     *
//*                                                                            *
//* Note: It is assumed that the caller has properly sized the display         *
//* strings to fit within the window's width. If not, we may experience        *
//* unsightly or uneven right edges.                                           *
//*                                                                            *
//* Input  : dPtr   : pointer to an array of character pointers                *
//*          cPtr   : pointer to an array of (attr_t) color attributes         *
//*          count  : number of data items                                     *
//*          hIndex : index of highlighted data item                           *
//*          hilite: (optional, true by default)                               *
//*                  display hilight on indexed item, hIndex                   *
//*          rtlFmt : (optional, false by default)                             *
//*                   draw data as RTL (right-to-left) text                    *
//*                                                                            *
//* Returns: index of currently-highlighted item                               *
//******************************************************************************

short NcWindow::PaintData ( const char* dPtr[], const attr_t cPtr[], short count, 
                            short hIndex, bool hilite, bool rtlFmt )
{
   short lineIndex, y = ZERO, x = rtlFmt ? (this->wCols - 1) : ZERO ;

   //* Save references to caller's data *
   this->scrollData.dptr = dPtr ;            // text data
   this->scrollData.cptr = cPtr ;            // color data
   this->scrollData.monocolor = ZERO ;       // not used for color display
   this->scrollData.count = count ;          // item count
   this->scrollData.rtl = rtlFmt ;           // indicates LTR or RTL data

   if ( hIndex < ZERO || hIndex >= count )   // error correction test
      hIndex = ZERO ;
   this->scrollData.hIndex = hIndex ;        // index of highlighted item
   this->scrollData.tIndex = ZERO ;          // initial index of top line of data
   if ( count >= wLines )                    // initial index of bottom line of data
      this->scrollData.bIndex = wLines - 1 ;
   else
      this->scrollData.bIndex = count - 1 ;
   if ( hIndex <= this->scrollData.bIndex )  // if highlight is in first page of list
      this->scrollData.hRow = hIndex ;       // row of highlighted item
   else
   {
      //* Advance top- and bottom-line indices until highlighted line is visible *
      while ( (this->scrollData.bIndex < hIndex) && 
              (this->scrollData.bIndex < (count-1)) )
      {
         ++this->scrollData.tIndex ;
         ++this->scrollData.bIndex ;
      }
      this->scrollData.hRow = wLines - 1 ;
      //* Bring highlight as close to top of window as possible *
      while ( (this->scrollData.tIndex < hIndex) && 
              (this->scrollData.bIndex < (count-1)) )
      {
         ++this->scrollData.tIndex ;
         ++this->scrollData.bIndex ;
         --this->scrollData.hRow ;
      }
   }

   this->AcquireOutputLock () ;     // acquire exclusive access to output stream

   this->ClearWin ( true, false ) ; // clear the old data (no display update)
   for ( lineIndex = this->scrollData.tIndex ; 
         lineIndex <= this->scrollData.bIndex ; lineIndex++ )
      this->WriteString ( y++, x, dPtr[lineIndex], cPtr[lineIndex],
                          false, this->scrollData.rtl ) ;
   if ( hilite != false )
      this->HilightItem ( true ) ;
   this->RefreshWin () ;                     // refresh the display

   this->ReleaseOutputLock () ;     // release lock on output stream
   return hIndex ;

}  //* End PaintData() *

//*************************
//*     RepaintData       *
//*************************
//******************************************************************************
//* Redraw data controlled by the scrollData structure.                        *
//*                                                                            *
//* It is assumed that all data members of the control structure are valid.    *
//*                                                                            *
//* Input  : (optional, true by default): hilight currently                    *
//*              indexed item                                                  *
//*                                                                            *
//* Returns: none                                                              *
//******************************************************************************

void NcWindow::RepaintData ( bool hilite )
{
   this->AcquireOutputLock () ;     // acquire exclusive access to output stream

   short lineIndex, dataIndex = this->scrollData.tIndex,  
         y = ZERO, x = this->scrollData.rtl ? (this->wCols - 1) : ZERO ;
   attr_t oColor ;

   this->ClearWin ( true, false ) ; // clear the old data (no display update)

   for ( lineIndex = ZERO ; lineIndex < this->wLines && 
         lineIndex < this->scrollData.count ; lineIndex++ )
   {
      if ( this->scrollData.cptr != NULL )
         oColor = this->scrollData.cptr[dataIndex] ;
      else
         oColor = this->scrollData.monocolor ;
      this->WriteString ( y++, x, this->scrollData.dptr[dataIndex], 
                          oColor, false, this->scrollData.rtl ) ;
      ++dataIndex ;
   }
   if ( hilite != false )           // highlight currently-indexed line
      this->HilightItem ( true ) ;
   this->RefreshWin () ;            // refresh the display

   this->ReleaseOutputLock () ;     // release lock on output stream

}  //* End RepaintData() *

//*************************
//*    PaintMenuData      *
//*************************
//******************************************************************************
//* Write menu data including strings with hotkey indicators into a window     *
//* to be scrolled, and initialize the scroll-tracking variables.              *
//*                                                                            *
//* This version of PaintMenuData() is for single-color data only.             *
//*                                                                            *
//* Note: We do not make a copy of the caller's text strings because menu      *
//*       text can change dynamically based on action in the application       *
//*       code. Therefore, we use the actual application data each time this   *
//*       method is called.                                                    *
//*                                                                            *
//* Note: This method scans the display strings for the hotkey character       *
//*       indicator '^' (caret) and displays the character that follows it     *
//*       with the Underline attribute.                                        *
//*                                                                            *
//* Note: It is assumed that the caller has properly sized the display         *
//* strings to fit within the window's width. If not, we may experience        *
//* unsightly or uneven edges.                                                 *
//*                                                                            *
//* Input  : dPtr   : pointer to an array of character pointers                *
//*          count  : number of data items                                     *
//*          hIndex : index of highlighted data item                           *
//*          color  : text color attribute                                     *
//*          hilite : (optional, true by default)                              *
//*                   hilight currently indexed item                           *
//*          rtlFmt : (optional, false by default)                             *
//*                   draw data as RTL (right-to-left) text                    *
//*                                                                            *
//* Returns: index of currently-highlighted item                               *
//******************************************************************************

short NcWindow::PaintMenuData ( const char* dPtr[], short count, short hIndex, 
                                attr_t color, bool hilite, bool rtlFmt )
{
   short lineIndex, y = ZERO, x = rtlFmt ? (this->wCols - 1) : ZERO ;

   //* Save references to caller's data *
   this->scrollData.dptr = dPtr ;               // text data
   this->scrollData.cptr = (const attr_t*)NULL ;// no color data
   this->scrollData.monocolor = color ;         // monochrome color
   this->scrollData.count = count ;             // item count
   this->scrollData.rtl = rtlFmt ;              // indicates LTR or RTL data

   if ( hIndex < ZERO || hIndex >= count )   // error correction test
      hIndex = ZERO ;
   this->scrollData.hIndex = hIndex ;        // index of highlighted item
   this->scrollData.tIndex = ZERO ;          // initial index of top line of data
   if ( count >= wLines - 1 )                // initial index of bottom line of data
      this->scrollData.bIndex = wLines - 1 ;
   else
      this->scrollData.bIndex = count - 1 ;
   if ( hIndex <= scrollData.bIndex )        // if highlight is in first page of list
      this->scrollData.hRow = hIndex ;       // row of highlighted item
   else
   {
      //* Advance top- and bottom-line indices until highlighted line is visible *
      while ( (this->scrollData.bIndex < hIndex) && 
              (this->scrollData.bIndex < (count-1)) )
      {
         ++this->scrollData.tIndex ;
         ++this->scrollData.bIndex ;
      }
      this->scrollData.hRow = wLines - 1 ;
      //* Bring highlight as close to top of window as possible *
      while ( (this->scrollData.tIndex < hIndex) && 
              (this->scrollData.bIndex < (count-1)) )
      {
         ++this->scrollData.tIndex ;
         ++this->scrollData.bIndex ;
         --this->scrollData.hRow ;
      }
   }

   this->AcquireOutputLock () ;     // acquire exclusive access to output stream

   //* Draw the visible lines of data *
   for ( lineIndex = this->scrollData.tIndex ; 
         lineIndex <= this->scrollData.bIndex ; lineIndex++ )
      this->WriteHotString ( y++, x, dPtr[lineIndex], this->scrollData.monocolor ) ;

   if ( hilite != false )           // highlight currently-indexed line
      this->HilightItem ( true ) ;
   this->RefreshWin () ;            // refresh the display

   this->ReleaseOutputLock () ;     // release lock on output stream

   return this->scrollData.hIndex ;

}  //* End PaintMenuData() *

//*************************
//*     PaintMenuData     *
//*************************
//******************************************************************************
//* Write menu data including strings with hotkey indicators into a window     *
//* to be scrolled, and initialize the scroll-tracking variables.              *
//*                                                                            *
//* This version of PaintMenuData() is for multi-color data.                   *
//*                                                                            *
//* Note: We do not make a copy of the caller's text strings because menu      *
//*       text can change dynamically based on action in the application       *
//*       code. Therefore, we use the actual application data each time this   *
//*       method is called.                                                    *
//*                                                                            *
//* Note: This method scans the display strings for the hotkey character       *
//*       indicator '^' (caret) and displays the character that follows it     *
//*       with the Underline attribute.                                        *
//*                                                                            *
//* Note: It is assumed that the caller has properly sized the display         *
//* strings to fit within the window's width. If not, we may experience        *
//* unsightly or uneven edges.                                                 *
//*                                                                            *
//* Input  : dPtr   : pointer to an array of character pointers                *
//*          cPtr   : pointer to an array of color attributes                  *
//*          count  : number of data items                                     *
//*          hIndex : index of highlighted data item                           *
//*          hilite : (optional, true by default)                              *
//*                      hilight currently indexed item                        *
//*          rtlFmt : (optional, false by default)                             *
//*                   draw data as RTL (right-to-left) text                    *
//*                                                                            *
//* Returns: index of currently-highlighted item                               *
//******************************************************************************

short NcWindow::PaintMenuData ( const char* dPtr[], attr_t cPtr[], short count, 
                                short hIndex, bool hilite, bool rtlFmt )
{
   short lineIndex, y = ZERO, x = rtlFmt ? (this->wCols - 1) : ZERO ;

   //* Save references to caller's data *
   this->scrollData.dptr = dPtr ;            // text data
   this->scrollData.cptr = cPtr ;            // each string has it own color
   this->scrollData.monocolor = ZERO ;       // monochrome color not used
   this->scrollData.count = count ;          // item count
   this->scrollData.rtl = rtlFmt ;           // indicates LTR or RTL data

   if ( hIndex < ZERO || hIndex >= count )   // error correction test
      hIndex = ZERO ;
   this->scrollData.hIndex = hIndex ;        // index of highlighted item
   this->scrollData.tIndex = ZERO ;          // initial index of top line of data
   if ( count >= this->wLines - 1 )          // initial index of bottom line of data
      this->scrollData.bIndex = this->wLines - 1 ;
   else
      this->scrollData.bIndex = count - 1 ;
   if ( hIndex <= this->scrollData.bIndex )  // if highlight is in first page of list
      this->scrollData.hRow = hIndex ;       // row of highlighted item
   else
   {
      //* Advance top- and bottom-line indices until highlighted line is visible *
      while ( (this->scrollData.bIndex < hIndex) && 
              (this->scrollData.bIndex < (count-1)) )
      {
         ++this->scrollData.tIndex ;
         ++this->scrollData.bIndex ;
      }
      this->scrollData.hRow = wLines - 1 ;
      //* Bring highlight as close to top of window as possible *
      while ( (this->scrollData.tIndex < hIndex) && 
              (this->scrollData.bIndex < (count-1)) )
      {
         ++this->scrollData.tIndex ;
         ++this->scrollData.bIndex ;
         --this->scrollData.hRow ;
      }
   }

   this->AcquireOutputLock () ;     // acquire exclusive access to output stream

   //* Draw the visible lines of data *
   for ( lineIndex = this->scrollData.tIndex ; 
         lineIndex <= this->scrollData.bIndex ; lineIndex++ )
      this->WriteHotString ( y++, x, dPtr[lineIndex], 
                             this->scrollData.cptr[lineIndex] ) ;

   if ( hilite != false )           // highlight currently-indexed line
      this->HilightItem ( true ) ;
   this->RefreshWin () ;            // refresh the display

   this->ReleaseOutputLock () ;     // release lock on output stream

   return this->scrollData.hIndex ;

}  // End PaintMenuData()

//*************************
//*      EraseData        *
//*************************
//******************************************************************************
//* Erase currently displayed data and reset the tracking variables.           *
//*                                                                            *
//* Note that the display and color data belong to the application's memory    *
//* space; there is no memory allocated for it within the NcWindow class.      *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: OK                                                                *
//******************************************************************************

short NcWindow::EraseData ( void )
{
   this->scrollData.dptr = NULL ;            // text data
   this->scrollData.cptr = NULL ;            // color data
   this->scrollData.monocolor = ZERO ;       // monochrome data
   this->scrollData.count = ZERO ;           // item count
   this->scrollData.hRow = ZERO ;            // row of highlighted item
   this->scrollData.hIndex = ZERO ;          // index of highlighted item
   this->scrollData.tIndex = ZERO ;          // index of top line of data
   this->scrollData.bIndex = ZERO ;          // index of bottom line of data

   this->AcquireOutputLock () ;     // acquire exclusive access to output stream
   wclear ( this->wp ) ;                     // clear and refresh the window
   wrefresh ( this->wp ) ;
   this->ReleaseOutputLock () ;     // release lock on output stream
   return OK ;

}  //* End EraseData() *

//*************************
//*      ScrollData       *
//*************************
//******************************************************************************
//* Scroll the disp display data by one scrolling unit.                        *
//*                 = scroll up by one line                                    *
//*                 = scroll down by one line                                  *
//*                 = scroll up by one page                                    *
//*                 = scroll down by one page                                  *
//*                 = scroll to top of data                                    *
//*                 = scroll to bottom of data                                 *
//*                                                                            *
//* Input  : key input (non-scroll-keys are ignored)                           *
//*                                                                            *
//* Returns: index of currently highlighted data item                          *
//******************************************************************************

short NcWindow::ScrollData ( wchar_t scrollKey )
{
   this->AcquireOutputLock () ;     // acquire exclusive access to output stream

   #define DEBUG_SD (0)
   attr_t   lineColor ;

   #if DEBUG_SD != 0
   gString tbuff ;
   tbuff.compose( L"wL:%02hd ct:%02hd hR:%02hd hI:%02hd tI:%02hd bI:%02hd",
                  &wLines, &scrollData.count, &scrollData.hRow, 
                  &scrollData.hIndex, &scrollData.tIndex, &scrollData.bIndex ) ;
   nc.WriteString ( 22, 2, tbuff.gstr(), ZERO ) ;
   #endif   // DEBUG_SD

   //*****************
   //* Down Arrow    *
   //*****************
   if ( scrollKey == nckDOWN )
   {
      //* If not already highlighting last data item *
      if ( this->scrollData.hIndex < (this->scrollData.count - 1) )
      {
         //* If highlight is not on last item currently  *
         //* displayed, move the highlight to next item. *
         if ( this->scrollData.hIndex < this->scrollData.bIndex )
         {
            if ( this->scrollData.cptr != NULL )   // color data
               lineColor = this->scrollData.cptr[this->scrollData.hIndex] ;
            else                             // monochrome data
               lineColor = this->scrollData.monocolor ;

            //* Re-display highlighted item to remove highlight *
            this->HilightItem ( false ) ;

            //* Highlight the next item *
            ++this->scrollData.hIndex ;
            ++this->scrollData.hRow ;
            if ( this->scrollData.cptr != NULL )   // color data
               lineColor = this->scrollData.cptr[this->scrollData.hIndex] ;
            if ( (lineColor & ncrATTR) == ZERO ) lineColor |= ncrATTR ;
            else                                 lineColor &= ~ncrATTR ;
            this->HilightItem ( true, true ) ;
         }
         
         //* Highlight is on last displayed item. *
         //* Scroll all items up by one position. *
         else
         {
            ++this->scrollData.tIndex ;
            ++this->scrollData.bIndex ;
            ++this->scrollData.hIndex ;
            this->RepaintData () ;
         }
      }
   }  // (scrollKey==nckDOWN)

   //*****************
   //* Up Arrow      *
   //*****************
   else if ( scrollKey == nckUP )
   {
      //* If not already highlighting first data item *
      if ( this->scrollData.hIndex > ZERO )
      {
         //* If highlight is not on first item currently     *
         //* displayed, move the highlight to previous item. *
         if ( this->scrollData.hIndex > this->scrollData.tIndex )
         {
            if ( this->scrollData.cptr != NULL )   // color data
               lineColor = this->scrollData.cptr[this->scrollData.hIndex] ;
            else                             // monochrome data
               lineColor = this->scrollData.monocolor ;

            //* Re-display highlighted item to remove highlight *
            this->HilightItem ( false ) ;

            //* Highlight the previous item *
            --this->scrollData.hIndex ;
            --this->scrollData.hRow ;
            if ( this->scrollData.cptr != NULL )   // color data
               lineColor = this->scrollData.cptr[this->scrollData.hIndex] ;
            if ( (lineColor & ncrATTR) == ZERO ) lineColor |= ncrATTR ;
            else                                 lineColor &= ~ncrATTR ;
            this->HilightItem ( true, true ) ;
         }

         //* Highlight is on first displayed item.  *
         //* Scroll all items down by one position. *
         else
         {
            --this->scrollData.tIndex ;
            --this->scrollData.bIndex ;
            --this->scrollData.hIndex ;
            this->RepaintData () ;
         }
      }
   }  // (scrollKey==nckUP)

   //*****************
   //* Top of List   *
   //*****************
   else if ( scrollKey == nckHOME )
   {
      if ( this->scrollData.hIndex > ZERO )  // if not already at 'home' position
      {
         this->scrollData.tIndex = ZERO ;
         this->scrollData.hIndex = ZERO ;
         this->scrollData.hRow   = ZERO ;
         if ( this->scrollData.count < this->wLines )
            this->scrollData.bIndex = this->scrollData.count - 1 ;
         else
            this->scrollData.bIndex = wLines - 1 ;
         this->RepaintData () ;
      }
   }  // (scrollKey==nckHOME)

   //*****************
   //* End of List   *
   //*****************
   else if ( scrollKey == nckEND )
   {  //* If not already at 'end' position *
      if ( this->scrollData.hIndex < (this->scrollData.count - 1) )
      {
         this->scrollData.bIndex = this->scrollData.count - 1 ;// index of last data item
         this->scrollData.hIndex = this->scrollData.count - 1 ;// highlight last data item
         if ( this->scrollData.count < this->wLines ) // if fewer data items than display lines
         {
            this->scrollData.hRow = this->scrollData.count - 1 ;
            this->scrollData.tIndex = ZERO ;
         }
         else                                // data items >= display lines
         {
            this->scrollData.hRow = this->wLines - 1 ;
            this->scrollData.tIndex = this->scrollData.count - this->wLines ;
         }
         this->RepaintData () ;
      }
   }  // (scrollKey==nckEND)

   //*****************
   //* Previous Page *
   //*****************
   else if ( scrollKey == nckPGUP )
   {
      if ( this->scrollData.hIndex > ZERO )  // if not already at 'home' position
      {
         //* If highlight not at first displayed line, *
         //* move highlight to first displayed line.   *
         if ( this->scrollData.hIndex > this->scrollData.tIndex )
         {
            if ( this->scrollData.cptr != NULL )   // color data
               lineColor = this->scrollData.cptr[this->scrollData.hIndex] ;
            else                             // monochrome data
               lineColor = this->scrollData.monocolor ;

            this->HilightItem ( false ) ;
            this->scrollData.hIndex = this->scrollData.tIndex ;
            this->scrollData.hRow = ZERO ;
            if ( this->scrollData.cptr != NULL )   // color data
               lineColor = this->scrollData.cptr[this->scrollData.hIndex] ;
            if ( (lineColor & ncrATTR) == ZERO ) lineColor |= ncrATTR ;
            else                                 lineColor &= ~ncrATTR ;
            this->HilightItem ( true, true ) ;
         }
         
         //* Else display new screenful of data *
         else if ( this->scrollData.tIndex > ZERO )
         {
            //* Highlight is on first displayed item AND     *
            //* first displayed item IS NOT first data item. *
            //* Note that hRow is already ZERO.              *
            short b = this->scrollData.tIndex - 1 ;
            while ( this->scrollData.bIndex > b && this->scrollData.tIndex > ZERO )
            {
               --this->scrollData.tIndex ;
               --this->scrollData.bIndex ;
               --this->scrollData.hIndex ;
            }
            this->RepaintData () ;
         }
      }
   }  // (scrollKey==nckPGUP)

   //*****************
   //* Next Page     *
   //*****************
   else if ( scrollKey == nckPGDOWN )
   {  //* If not already at 'end' position
      if ( this->scrollData.hIndex < (this->scrollData.count - 1) )
      {
         //* If highlight not at last displayed line, *
         //* move highlight to last displayed line.   *
         if ( this->scrollData.hIndex < this->scrollData.bIndex )
         {
            if ( this->scrollData.cptr != NULL )   // color data
               lineColor = this->scrollData.cptr[this->scrollData.hIndex] ;
            else                             // monochrome data
               lineColor = this->scrollData.monocolor ;

            this->HilightItem ( false ) ;
            while ( (this->scrollData.hIndex < this->scrollData.bIndex) && 
                    (this->scrollData.hRow < this->wLines-1) )
            {
               ++this->scrollData.hRow ;
               ++this->scrollData.hIndex ;
            }
            if ( this->scrollData.cptr != NULL )   // color data
               lineColor = this->scrollData.cptr[this->scrollData.hIndex] ;
            if ( (lineColor & ncrATTR) == ZERO ) lineColor |= ncrATTR ;
            else                                 lineColor &= ~ncrATTR ;
            this->HilightItem ( true, true ) ;
         }
         
         //* Else display new screenful of data *
         else if ( this->scrollData.bIndex < (this->scrollData.count-1) )
         {
            //* Highlight is on last displayed item AND    *
            //* last displayed item IS NOT last data item. *
            //* Note that hRow is already wLines-1.        *
            short t = this->scrollData.bIndex + 1 ;
            while ( (this->scrollData.tIndex < t) && 
                    (this->scrollData.bIndex < (this->scrollData.count-1)) )
            {
               ++this->scrollData.tIndex ;
               ++this->scrollData.bIndex ;
               ++this->scrollData.hIndex ;
            }
            this->RepaintData () ;
         }
      }
   }  // (scrollKey==nckPGDOWN)

   //*****************
   //* Shift Right   *
   //*****************
   else if ( scrollKey == nckLEFT )
   {
      /* Not valid within this context. */
   }  // (scrollKey==nckLEFT)

   //*****************
   //* Shift Left    *
   //*****************
   else if ( scrollKey == nckRIGHT )
   {
      /* Not valid within this context. */
   }  // (scrollKey==nckRIGHT)
   
   #if DEBUG_SD != 0
   tbuff.compose( L"            hR:%02hd hI:%02hd tI:%02hd bI:%02hd",
                  &scrollData.hRow, &scrollData.hIndex,
                  &scrollData.tIndex, &scrollData.bIndex ) ;
   DebugMsg ( tbuff.ustr() ) ;
   #endif   // DEBUG_SD

   this->ReleaseOutputLock () ;     // release lock on output stream

   return this->scrollData.hIndex ; // index of currently-highlighted item
   
}  //* End ScrollData() *

//*************************
//*    ScrollMenuData     *
//*************************
//******************************************************************************
//* Scroll through the items in a menu. Menu data may or may not have          *
//* hot-key associations, indicated by the caret ('^') before the character.   *
//* If a caret is present in the data, the data are displayed with the         *
//* character following the caret underlined to indicate the hot-key.          *
//* See also PaintMenuData().                                                  *
//*                                                                            *
//* Note that unlike regular scrolling display lists, menu data are all        *
//* visible in the menu window. For this reason, only the highlight is moved.  *
//* Scrolling the data elements up and down is not required.                   *
//*                                                                            *
//* Note also that unlike regular scrolling display lists, if nckUP is pressed *
//* while highlight is on first item, highlight will wrap to the last item;    *
//* and similarly, if nckDOWN is pressed while highlight is on last item,      *
//* highlight will wrap to the first item.                                     *
//*                                                                            *
//* Input  : key input (non-scroll-keys are ignored)                           *
//*                                                                            *
//* Returns: index of currently highlighted data item                          *
//*          (if highlight moved, window is refreshed before return)           *
//******************************************************************************
//* Programmer's Note: This method may incorrectly draw the highlighted item   *
//* if the color data is not already in reverse attribute.                     *
//*                                                                            *
//******************************************************************************

short NcWindow::ScrollMenuData ( wchar_t scrollKey )
{
   this->AcquireOutputLock () ;     // acquire exclusive access to output stream

   //* Handle top-to-bottom and bottom-to-top wrap *
   // Programmer's Note: To disable wrapping, comment out this section.
   if ( scrollKey == nckUP && this->scrollData.hIndex == ZERO )
      scrollKey = nckEND ;
   else if ( scrollKey == nckDOWN && 
             this->scrollData.hIndex == (this->scrollData.count - 1) )
      scrollKey = nckHOME ;

   //*****************
   //* Down Arrow    *
   //*****************
   if ( scrollKey == nckDOWN )
   {
      //* If not already highlighting last data item *
      if ( this->scrollData.hIndex < (this->scrollData.count - 1) )
      {
         //* If highlight is not on last item currently  *
         //* displayed, move the highlight to next item. *
         if ( this->scrollData.hIndex < this->scrollData.bIndex )
         {
            //* Re-display highlighted item to remove highlight *
            this->HilightItem ( false ) ;

            //* Highlight the next item *
            ++this->scrollData.hIndex ;
            ++this->scrollData.hRow ;
            this->HilightItem ( true ) ;
            this->RefreshWin () ;         // Refresh the display
         }
      }
   }  // (scrollKey==nckDOWN)

   //*****************
   //* Up Arrow      *
   //*****************
   else if ( scrollKey == nckUP )
   {
      //* If not already highlighting first data item *
      if ( this->scrollData.hIndex > ZERO )
      {
         //* If highlight is not on first item currently     *
         //* displayed, move the highlight to previous item. *
         if ( this->scrollData.hIndex > this->scrollData.tIndex )
         {
            //* Re-display highlighted item to remove highlight *
            this->HilightItem ( false ) ;

            //* Highlight the previous item *
            --this->scrollData.hIndex ;
            --this->scrollData.hRow ;
            this->HilightItem ( true ) ;
            this->RefreshWin () ;         // Refresh the display
         }
      }
   }  // (scrollKey==nckUP)

   //*****************
   //* Top of List   *
   //*****************
   else if ( scrollKey == nckHOME || scrollKey == nckPGUP )
   {
      if ( this->scrollData.hIndex > ZERO ) // if not already at 'home' position
      {
         //* Re-display highlighted item to remove highlight *
         this->HilightItem ( false ) ;

         this->scrollData.tIndex = ZERO ; // index of first data item
         this->scrollData.hIndex = ZERO ;
         this->scrollData.hRow = ZERO ;
         this->HilightItem ( true ) ;     // highlight the first item
         this->RefreshWin () ;            // Refresh the display
      }
   }  // (scrollKey==nckHOME)

   //*****************
   //* End of List   *
   //*****************
   else if ( scrollKey == nckEND || scrollKey == nckPGDOWN )
   {  //* If not already at 'end' position *
      if ( this->scrollData.hIndex < (this->scrollData.count - 1) )
      {
         //* Re-display highlighted item to remove highlight *
         this->HilightItem ( false ) ;

         this->scrollData.bIndex = this->scrollData.count - 1 ; // index of last data item
         this->scrollData.hIndex = this->scrollData.bIndex ;
         this->scrollData.hRow   = this->scrollData.bIndex ;
         this->HilightItem ( true ) ;// highlight the last item
         this->RefreshWin () ;      // Refresh the display
      }
   }  // (scrollKey==nckEND)

   this->ReleaseOutputLock () ;     // release lock on output stream

   return this->scrollData.hIndex ; // index of currently-highlighted item

}  //* End ScrollMenuData() *

//*************************
//*     ScrollView        *
//*************************
//******************************************************************************
//* Scroll the disp display data by one scrolling unit.                        *
//*   nckUP         = scroll up by one line                                    *
//*   nckDOWN       = scroll down by one line                                  *
//*   nckPGUP       = scroll up by one page                                    *
//*   nckPGDOWN     = scroll down by one page                                  *
//*   nckHOME       = scroll to top of data                                    *
//*   nckEND        = scroll to bottom of data                                 *
//*                                                                            *
//* Input  : key input (non-scroll-keys are ignored)                           *
//*                                                                            *
//* Returns: index of data item at top of window                               *
//******************************************************************************

short NcWindow::ScrollView ( wchar_t scrollKey )
{
   this->AcquireOutputLock () ;     // acquire exclusive access to output stream

   //*****************
   //* Down Arrow    *
   //*****************
   if ( scrollKey == nckDOWN )
   {
      //* If last data item not already displayed *
      if ( this->scrollData.bIndex < (this->scrollData.count - 1) )
      {
         ++this->scrollData.tIndex ;
         ++this->scrollData.bIndex ;
         ++this->scrollData.hIndex ;
         this->scrollData.hIndex = this->scrollData.tIndex ; // index the first displayed item
         this->RepaintData ( false ) ;
      }
   }  // (scrollKey==nckDOWN)

   //*****************
   //* Up Arrow      *
   //*****************
   else if ( scrollKey == nckUP )
   {
      //* If first item not already displayed *
      if ( this->scrollData.tIndex > ZERO )
      {
         --this->scrollData.tIndex ;
         --this->scrollData.bIndex ;
         --this->scrollData.hIndex ;
         this->scrollData.hIndex = this->scrollData.tIndex ; // index the first displayed item
         this->RepaintData ( false ) ;
      }
   }  // (scrollKey==nckUP)

   //*****************
   //* Top of List   *
   //*****************
   else if ( scrollKey == nckHOME )
   {
      if ( this->scrollData.hIndex > ZERO ) // if not already at 'home' position
      {
         this->scrollData.tIndex = ZERO ;
         this->scrollData.hIndex = ZERO ;
         this->scrollData.hRow = ZERO ;
         if ( this->scrollData.count < this->wLines )
            this->scrollData.bIndex = this->scrollData.count - 1 ;
         else
            this->scrollData.bIndex = this->wLines - 1 ;
         this->RepaintData ( false ) ;
      }
   }  // (scrollKey==nckHOME)

   //*****************
   //* End of List   *
   //*****************
   else if ( scrollKey == nckEND )
   {
      //* If last data item not already displayed *
      if ( this->scrollData.bIndex < (this->scrollData.count - 1) )
      {
         this->scrollData.bIndex = this->scrollData.count - 1 ; // index of last data item
         if ( this->scrollData.count < this->wLines ) // if fewer data items than display lines
         {
            this->scrollData.hRow = this->scrollData.count - 1 ;
            this->scrollData.tIndex = ZERO ;
         }
         else                                // data items >= display lines
         {
            this->scrollData.hRow = wLines - 1 ;
            this->scrollData.tIndex = this->scrollData.count - this->wLines ;
         }
         this->scrollData.hIndex = this->scrollData.tIndex ; // index the first displayed item
         this->RepaintData ( false ) ;
      }
   }  // (scrollKey==nckEND)

   //*****************
   //* Previous Page *
   //*****************
   else if ( scrollKey == nckPGUP )
   {
      //* If first item not already displayed *
      if ( this->scrollData.tIndex > ZERO )
      {
         short b = this->scrollData.tIndex - 1 ;
         while ( this->scrollData.bIndex > b && this->scrollData.tIndex > ZERO )
         {
            --this->scrollData.tIndex ;
            --this->scrollData.bIndex ;
            --this->scrollData.hIndex ;
         }
         this->scrollData.hIndex = this->scrollData.tIndex ; // index the first displayed item
         this->RepaintData ( false ) ;
      }
   }  // (scrollKey==nckPGUP)

   //*****************
   //* Next Page     *
   //*****************
   else if ( scrollKey == nckPGDOWN )
   {
      //* If last data item not already displayed *
      if ( this->scrollData.bIndex < (this->scrollData.count - 1) )
      {
         short t = this->scrollData.bIndex + 1 ;
         while ( (this->scrollData.tIndex < t) && 
                 (this->scrollData.bIndex < (this->scrollData.count-1)) )
         {
            ++this->scrollData.tIndex ;
            ++this->scrollData.bIndex ;
            ++this->scrollData.hIndex ;
         }
         this->scrollData.hIndex = this->scrollData.tIndex ; // index the first displayed item
         this->RepaintData ( false ) ;
      }
   }  // (scrollKey==nckPGDOWN)

   //*****************
   //* Shift Right   *
   //*****************
   else if ( scrollKey == nckLEFT )
   {
   }  // (scrollKey==nckLEFT)

   //*****************
   //* Shift Left    *
   //*****************
   else if ( scrollKey == nckRIGHT )
   {
   }  // (scrollKey==nckRIGHT)
   
   this->ReleaseOutputLock () ;     // release lock on output stream

   return this->scrollData.tIndex ; // index of top item being displayed
   
}  //* End ScrollView() *

//*************************
//*     Hotkey2Index      *
//*************************
//******************************************************************************
//* Convert a menu-item selection made through a hotkey into an index into     *
//* the menu and move highlight to the indexed item.                           *
//* If no hotkey is associated with this keycode, highlight is unchanged.      *
//*                                                                            *
//*                                                                            *
//* Input  : hKey: key input from user                                         *
//*                - If hKey contains a valid hotkey, nckENTER is returned     *
//*                  indicating that the now-highlighted menu item has been    *
//*                  'selected'.                                               *
//*                - If NOT a valid hotkey for the menu, value is unchanged    *
//*                                                                            *
//* Returns: index of (new or unchanged) highlighted menu item                 *
//******************************************************************************
//* Programmer's Note: Hotkeys are only A-Z, a-z, Ctrl+A - Ctrl+Z.             *
//* Therefore we 'know' that the character that follows the caret '^' is an    *
//* ASCII character. HOWEVER, because the data are in UTF-8 format, we must    *
//* be careful about scanning the data as if it were pure ASCII.               *
//*                                                                            *
//* As a special case for menu-item selection the digits '0' through '9' are   *
//* recognized as hotkeys for the first nine(9) menu items. This is similar    *
//* to the selection of menu items in the info reader. Note that ONLY the      *
//* first nine items in the menu may be selected in this way.                  *
//* We have found this useful when creating menus for multiple languages, or   *
//* when creating a menu using a language which does not use ASCII characters  *
//* (Chinese).                                                                 *
//* A '0' key is first set to '1'. This avoids any out-of-range problems.      *
//* The numeric equivalent of the key is then converted to a (zero-based)      *
//* index. If there is a menu item at that index, the highlight is set on      *
//* that item and the ENTER key is returned as if the equivalent hotkey had    *
//* been pressed.                                                              *
//******************************************************************************

short NcWindow::Hotkey2Index ( wchar_t& hKey )
{
   wchar_t  testKey = nckNULLCHAR ;
   if ( hKey >= 'A' && hKey <= 'Z' )         // is uppercase alpha
      testKey = hKey ;
   else if ( hKey >= 'a' && hKey <= 'z' )    // lowercase alpha to uppercase
      testKey = hKey & 0x005F ;
   else if ( (hKey >= nckC_A && hKey <= nckC_H) || // control key to upperase alpha
             (hKey == nckC_K || hKey == nckC_L) ||
             (hKey >= nckC_N && hKey <= nckC_Z) )
   {
      testKey = hKey + 0x0040 ;
   }
   if ( testKey >= 'A' && testKey <= 'Z' )   // if a possible hotkey selection
   {
      //* Scan the menu data strings for a match with our candidate keycode.   *
      short newIndex = -1 ;                  // new item index (initially none)
      const wchar_t* gsPtr ;
      gString  gs ;
      int      gsCount ;
      wchar_t  tChar ;
      for ( short i = ZERO ; i < this->scrollData.count && newIndex < ZERO ; i++ )
      {
         gs = this->scrollData.dptr[i] ;
         gsPtr = gs.gstr( gsCount ) ;
         for ( short j = ZERO ; j < gsCount ; j++ )
         {
            if ( gsPtr[j] == HOTCHAR )
            {
               tChar = towupper ( gsPtr[++j] ) ;
               if ( tChar == testKey )
               {
                  newIndex = i ; // index of menu item containing matching hotkey

                  //* Move highlight to this menu item *
                  wchar_t arrowKey = newIndex > this->scrollData.hIndex 
                                     ? nckDOWN : nckUP ;

                  this->AcquireOutputLock () ;  // acquire exclusive access to output stream

                  while ( this->scrollData.hIndex != newIndex )
                     this->ScrollMenuData ( arrowKey ) ;

                  this->ReleaseOutputLock () ;  // release lock on output stream

                  hKey = nckENTER ;    // signal our success
                  break ;
               }
            }
         }
      }
   }
   else if ( hKey >= '0' && hKey <= '9' )
   {
      short newIndex = hKey & 0x0000000F ;   // convert character to numeric
      if ( newIndex == 0 )                   // avoid embarrassment
         newIndex = 1 ;
      --newIndex ;                           // convert to index

      if ( newIndex < this->scrollData.count ) // range check
      {
         //* Move highlight to this menu item *
         wchar_t arrowKey = newIndex > this->scrollData.hIndex 
                            ? nckDOWN : nckUP ;

         this->AcquireOutputLock () ;  // acquire exclusive access to output stream

         while ( this->scrollData.hIndex != newIndex )
            this->ScrollMenuData ( arrowKey ) ;

         this->ReleaseOutputLock () ;  // release lock on output stream

         hKey = nckENTER ;    // signal our success
      }
   }
   return scrollData.hIndex ;                // return current item index
   
}  //* End Hotkey2Index() *

//*************************
//*   GetHilightIndex     *
//*************************
//******************************************************************************
//* Returns the index of the currently-highlighted item in the scrolling-      *
//* data array.                                                                *
//*                                                                            *
//* Input : none                                                               *
//*                                                                            *
//* Returns: index of currently-highlighted item                               *
//******************************************************************************

short NcWindow::GetHilightIndex ( void )
{

   return this->scrollData.hIndex ;

}  //* End GetHilightIndex() *

//*************************
//*     IsScrollKey       *
//*************************
//******************************************************************************
//* Tests caller's key datum to determine whether it is one of the scrolling   *
//* keys, nckUP, nckDOWN, nckLEFT, nckRIGHT, nckHOME, nckEND, nckPGUP,         *
//* nckPGDOWN.                                                                 *
//*                                                                            *
//*                                                                            *
//* Input  : wkeyCode-class object containing keycode and key classification   *
//*                                                                            *
//* Returns: returns true if a scroll key, else false                          *
//******************************************************************************

bool  NcWindow::IsScrollKey ( wkeyCode wk )
{
   bool result = false ;

   if ( (wk.type == wktFUNKEY) &&
        (wk.key == nckUP     || wk.key == nckDOWN || wk.key == nckPGUP || 
         wk.key == nckPGDOWN || wk.key == nckHOME || wk.key == nckEND || 
         wk.key == nckLEFT   || wk.key == nckRIGHT) )
      result = true ;
      
   return result ;
      
}  //* End IsScrollKey() *

//*************************
//*     HilightItem       *
//*************************
//******************************************************************************
//* Hilight or remove hilight from the display line referenced by hIndex.      *
//*                                                                            *
//* Removing the highlight (un-hilighting) the data means drawing the data     *
//* using the base color attribute.                                            *
//*                                                                            *
//* Highlighting the data means TOGGLING the Reverse video (ncrATTR) bit of    *
//* the base color attribute and then drawing the data. If the base color      *
//* data for the string does not include the ncrATTR bit, then the data are    *
//* drawn with ncrATTR SET. If the base color data for the string does         *
//* include the ncrATTR bit, then the data are drawn with ncrATTR RESET.       *
//*                                                                            *
//* Note also that any HOTCHAR character (optionally be embedded in line item) *
//* is also handled.                                                           *
//*                                                                            *
//* Input  : show   : 'true' to add hilight, 'false' to remove hilight         *
//*          refresh: (optional, 'false' by default)                           *
//*                   if 'true', then refresh the display before return        *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void NcWindow::HilightItem ( bool show, bool refresh )
{
   if ( this->scrollData.hIndex >= ZERO )
   {  //* Start position is X *
      short  x = this->scrollData.rtl ? (this->wCols - 1) : ZERO ;
      attr_t color ;                // color attribute

      //* Basic data color *
      if ( this->scrollData.cptr != NULL )// color data
         color = this->scrollData.cptr[this->scrollData.hIndex] ;
      else                                // monochrome data
         color = this->scrollData.monocolor ;
      //* Determine whether the base data has the ncrATTR (Reverse) bit set *
      bool baseRev = (bool)(color & ncrATTR) ;
      //* Adjust the color attribute appropriately *
      if ( show != false )                // highlight the current data item
      {
         if ( baseRev != false )          // base color includes ncrATTR bit
            color &= ~ncrATTR ;           // mask the ncrATTR bit from base color
         else                             // base color does not include ncrATTR bit
            color |= ncrATTR ;            // add ncrATTR bit to base color
      }
      this->AcquireOutputLock () ;     // acquire exclusive access to output stream
      this->WriteHotString ( this->scrollData.hRow, x, 
                             this->scrollData.dptr[this->scrollData.hIndex], color );
      if ( refresh != false )
         this->RefreshWin () ;
      this->ReleaseOutputLock () ;     // release lock on output stream
   }
   
}  //* End HilightItem() *

//*************************
//*      Track2Top        *
//*************************
//******************************************************************************
//* Move the logical highlight to the first item in window, and optionally     *
//* update the display.                                                        *
//* If display refresh not specified, then call RepaintData() to refresh.      *
//*                                                                            *
//* Input  : refresh: (optional, false by default)                             *
//*                   if 'true', repaint the data and refresh the display      *
//*                              (highlight will be visible)                   *
//*                                                                            *
//* Returns: index of the logically-highlighted item i.e. ZERO                 *
//******************************************************************************

short NcWindow::Track2Top ( bool refresh )
{
   this->scrollData.tIndex = ZERO ;
   this->scrollData.hIndex = ZERO ;
   this->scrollData.hRow   = ZERO ;
   if ( this->scrollData.count >= this->wLines - 1 )
      this->scrollData.bIndex = this->wLines - 1 ;
   else  // fewer data items than window lines
      this->scrollData.bIndex = this->scrollData.count - 1 ;

   if ( refresh != false )
      this->RepaintData () ;

   return this->scrollData.hIndex ;

}  //* End Track2Top() *

//*************************
//*     Track2Bottom      *
//*************************
//******************************************************************************
//* Move the logical highlight to the last item in window, and optionally      *
//* update the display.                                                        *
//* If display refresh not specified, then call RepaintData() to refresh.      *
//*                                                                            *
//* Input  : refresh: (optional, false by default)                             *
//*                   if 'true', repaint the data and refresh the display      *
//*                              (highlight will be visible)                   *
//*                                                                            *
//* Returns: index of the logically-highlighted item                           *
//******************************************************************************

short NcWindow::Track2Bottom ( bool refresh )
{
   this->scrollData.hIndex = 
   this->scrollData.bIndex = this->scrollData.count - 1 ;
   if ( this->scrollData.count >= this->wLines )
   {
      this->scrollData.hRow = this->wLines - 1 ;
      this->scrollData.tIndex = this->scrollData.count - this->wLines ;
   }
   else  // fewer data items than window lines
   {
      this->scrollData.hRow = this->scrollData.hIndex ;
      this->scrollData.tIndex = ZERO ;
   }

   if ( refresh != false )
      this->RepaintData () ;

   return this->scrollData.hIndex ;

}  //* End Track2Bottom() *

//*************************
//*      Track2Next       *
//*************************
//******************************************************************************
//* Move the logical highlight to the next (lower) item in window, and         *
//* optionally update the display.                                             *
//* If display refresh not specified, then call RepaintData() to refresh.      *
//*                                                                            *
//* Input  : refresh: (optional, false by default)                             *
//*                   if 'true', repaint the data and refresh the display      *
//*                              (highlight will be visible)                   *
//*                                                                            *
//* Returns: index of the logically-highlighted item                           *
//******************************************************************************

short NcWindow::Track2Next ( bool refresh )
{
   //* If not already at the bottom of the list *
   if ( this->scrollData.hIndex < this->scrollData.count-1 )
   {
      //* If highlight is not on last item currently  *
      //* displayed, move the highlight to next item. *
      if ( this->scrollData.hIndex < this->scrollData.bIndex )
      {
         ++this->scrollData.hIndex ;
         ++this->scrollData.hRow ;
      }
      //* Highlight is on last displayed item. *
      //* Scroll all items up by one position. *
      else
      {
         ++this->scrollData.tIndex ;
         ++this->scrollData.bIndex ;
         ++this->scrollData.hIndex ;
         // this->scrollData.hRow is unchanged on last display row
      }
   }

   if ( refresh != false )
      this->RepaintData () ;

   return this->scrollData.hIndex ;

}  //* End Track2Next() *

//*************************
//*    Track2Previous     *
//*************************
//******************************************************************************
//* Move the logical highlight to the previous (higher) item in window, and    *
//* optionally update the display.                                             *
//* If display refresh not specified, then call RepaintData() to refresh.      *
//*                                                                            *
//* Input  : refresh: (optional, false by default)                             *
//*                   if 'true', repaint the data and refresh the display      *
//*                              (highlight will be visible)                   *
//*                                                                            *
//* Returns: index of the logically-highlighted item                           *
//******************************************************************************

short NcWindow::Track2Previous ( bool refresh )
{
   //* If not already at top of list *
   if ( this->scrollData.hIndex > ZERO )
   {
      //* If highlight is not on first item currently     *
      //* displayed, move the highlight to previous item. *
      if ( this->scrollData.hIndex > this->scrollData.tIndex )
      {
         --this->scrollData.hIndex ;
         --this->scrollData.hRow ;
      }

      //* Highlight is on first displayed item.  *
      //* Scroll all items down by one position. *
      else
      {
         --this->scrollData.tIndex ;
         --this->scrollData.bIndex ;
         --this->scrollData.hIndex ;
         // this->scrollData.hRow is unchanged on first display row
      }
   }

   if ( refresh != false )
      this->RepaintData () ;

   return this->scrollData.hIndex ;

}  //* End Track2Previous() *

//*************************
//*      Track2Item       *
//*************************
//******************************************************************************
//* Move the logical highlight to the specified item in window, and            *
//* optionally update the display.                                             *
//* If display refresh not specified, then call RepaintData() to refresh.      *
//*                                                                            *
//* Input  : iIndex : index of target item                                     *
//*          refresh: (optional, false by default)                             *
//*                   if 'true', repaint the data and refresh the display      *
//*                              (highlight will be visible)                   *
//*                                                                            *
//* Returns: index of the logically-highlighted item                           *
//*            If iIndex out-of-range, then index of currently-highlighted     *
//*            item is returned.                                               *
//******************************************************************************

short NcWindow::Track2Item ( short iIndex, bool refresh )
{
   this->AcquireOutputLock () ;     // acquire exclusive access to output stream

   //* Range-check the input *
   if ( (iIndex >= ZERO) && (iIndex < this->scrollData.count) )
   {
      while ( this->scrollData.hIndex < iIndex )
         this->Track2Next () ;
      while ( this->scrollData.hIndex > iIndex )
         this->Track2Previous () ;
   }

   if ( refresh != false )
      this->RepaintData () ;

   this->ReleaseOutputLock () ;     // release lock on output stream
   return this->scrollData.hIndex ;

}  //* End Track2Item() *

//*************************
//*                       *
//*************************
//******************************************************************************
//*                                                                            *
//*                                                                            *
//*                                                                            *
//* Input  :                                                                   *
//*                                                                            *
//* Returns:                                                                   *
//******************************************************************************


