//********************************************************************************
//* File       : NcwKey.cpp                                                      *
//* Author     : Mahlon R. Smith                                                 *
//*              Copyright (c) 2005-2025 Mahlon R. Smith, The Software Samurai   *
//*                  GNU GPL copyright notice located in NcWindow.hpp            *
//* Date       : 27-Dec-2020                                                     *
//* Version    : (see NcWindlwVersion string in NcWindow.cpp)                    *
//*                                                                              *
//* Description: This module contains:                                           *
//*   a) Pass-through methods for calling the NCurses-class methods indirectly   *
//*      through the NcWindow object.                                            *
//*      These NcWindow pass-through methods are provided for convenience in     *
//*      accessing often-used NCurses-class methods of the same name using an    *
//*      NcWindow pointer. Please refer to NCurses.hpp for a more complete       *
//*      description of each of these methods.                                   *
//*                                                                              *
//*      NOTE: All NCurses public methods and public data members are available  *
//*            through the global handle 'nc' defined in GlobalDef.hpp.          *
//*                    ** HOWEVER, They are not thread-safe **                   *
//*                                                                              *
//*   b) Access to the NCurses-class keyboard input methods and key-input        *
//*      data-stream conditioning methods are channeled through this module      *
//*      to create a thread-safe input stream. These methods are protected by    *
//*      a mutex, 'InputStream_Lock'to ensure thread-safe behavior.              *
//*                                                                              *
//*      This module contains the methods which implement the exclusive locks    *
//*      for the input stream and output stream:                                 *
//*              AcquireInputLock()   and  ReleaseInputLock()                    *
//*              AcquireOutputLock()  and  ReleaseOutputLock()                   *
//*      See additional notes on thread safety, below.                           *
//*                                                                              *
//*   c) The application layer mouse interface.                                  *
//*      While the NCurses-class mouse methods are screen-oriented, the          *
//*      NcWindow-class methods transform those simpler methods into             *
//*      dialog-window-oriented methods. It is expected that applications will   *
//*      seldom, if ever call the NCurses-class mouse-event methods directly     *
//*      - nor should they. Access is accomplished through an indirect access    *
//*      to the NcWindow-class GetKeyInput() method.                             *
//*                                                                              *
//*                                                                              *
//* Development Tools: See NcWindow.cpp.                                         *
//********************************************************************************
//* NOTE: There are certain NCurses-class methods, such as StartNCursesEngine()  *
//*       and StopNCursesEngine() which it would make no sense to call from      *
//*       inside a window interface, so pass-through stubs are not provided      *
//*       for those methods. Nor are stubs provided for seldom-used NCurses      *
//*       methods. In addition, certain NCurses methods might be easily          *
//*       confused with similar NcWindow methods, so stubs are not provided      *
//*       for those methods.                                                     *
//*                                                                              *
//********************************************************************************

//*****************
//* Include Files *
//*****************

#include "GlobalDef.hpp"               //* General definitions

#ifndef NCURSES_INCLUDED
#include "NCurses.hpp"
#endif

#ifndef NCWINDOW_INCLUDED
#include "NcWindow.hpp"
#endif

#if NCD_THREAD_SAFETY != 0
//* These mutexes are located in NcWindow.cpp.*
extern mutex InputStream_Lock ;
extern mutex OutputStream_Lock ;
#endif   // NCD_THREAD_SAFETY

//* For 'stable' mouse interface (see meEnableStableMouse method),  *
//* only the following event types are passed to meClickSynthesis().*
static const mmask_t csEVENTS = BUTTON1_RELEASED ;


//*************************
//*     GetKeyInput       *
//*************************
//******************************************************************************
//* Get a keycode and keytype from the ncursesw input stream.                  *
//* This includes all supported international character codes as well as       *
//* captured-and-translated escape sequences representing function keys,       *
//* cursor keys and other special-purpose key combinations.                    *
//*                                                                            *
//* If mouse input has been enabled, mouse events will also be returned in     *
//* caller's wkeyCode-class object's 'mevent' data member.                     *
//*                                                                            *
//* Input  : wKey (by reference, initial values ignored)                       *
//*          receives the key input data                                       *
//*                                                                            *
//* Returns: member of enum wkeyType                                           *
//******************************************************************************
//* Programmer's Note:                                                         *
//* If 'Stable' mode mouse interface has been enabled, we also call the        *
//* meClickSynthesis() method to convert from 'release' events to 'click'      *
//* events.                                                                    *
//******************************************************************************

wkeyType NcWindow::GetKeyInput ( wkeyCode& wKey )
{
   //** Acquire an exclusive lock on the input stream.                        **
   //** Those desiring access to the input stream, will sleep on this mutex.  **
   this->AcquireInputLock () ;

   nc.GetKeyInput ( wKey ) ;

   if ( wKey.type == wktMOUSE )
   {
      //* If in 'stable' mode, we synthesize button-1 click events.*
      //* ScrollWheel events are passed along unmodified.          *
      if ( (this->meStable != false) && 
           ((wKey.mevent.eventType & csEVENTS) != ZERO) )
      {
         //* If synthesis is successful, the 'mevent' member will *
         //* contain valid click-event data.                      *
         //* Else, invalid data: wk.type==wktERR and wk.key==ZERO.*
         this->meClickSynthesis ( wKey ) ;
      }

      //* Due to timing issues, we may have garbage in the mouse queue.   *
      //* This is supposed to be a LIFO queue, but it gets routed through *
      //* the keyboard FIFO queue, so we may receive stale data.          *
      //* For this reason, we flush the queue after each mouse event.     *
      nc.FlushMouseEventQueue () ;
   }

   //** Release the exclusive lock **
   this->ReleaseInputLock () ;

   return wKey.type ;

}  //* End GetKeyInput() *

//*************************
//*       KeyPeek         *
//*************************
wkeyType NcWindow::KeyPeek ( wkeyCode& wKey )
{
   this->AcquireInputLock () ;
   nc.KeyPeek ( wKey ) ;
   this->ReleaseInputLock () ;
   return wKey.type ;
   
}  //* End KeyPeek() *

//*************************
//*    UngetKeyInput      *
//*************************
short NcWindow::UngetKeyInput ( wkeyCode& wKey )
{
   this->AcquireInputLock () ;
   short status = nc.UngetKeyInput ( wKey ) ;
   this->ReleaseInputLock () ;
   return status ;

}  //* End UngetKeyInput() *

//*************************
//* FlushKeyInputStream   *
//*************************
short NcWindow::FlushKeyInputStream ( void )
{
   this->AcquireInputLock () ;
   short status = nc.FlushKeyInputStream () ;
   this->ReleaseInputLock () ;
   return status ;

}  //* End FlushKeyInputStream() *

//*************************
//*    GetCursorState     *
//*************************
ncCurVis NcWindow::GetCursorState ( void )
{
   this->AcquireInputLock () ;
   ncCurVis status = nc.GetCursorState () ;
   this->ReleaseInputLock () ;
   return status ;

}  //* End GetCursorState() *

//*************************
//*    SetCursorState     *
//*************************
short NcWindow::SetCursorState ( ncCurVis state )
{
   this->AcquireInputLock () ;
   short status = nc.SetCursorState ( state ) ;
   this->ReleaseInputLock () ;
   return status ;

}  //* End SetCursorState() *

//*************************
//*  RestoreCursorState   *
//*************************
short NcWindow::RestoreCursorState ( void )
{
   this->AcquireInputLock () ;
   short status = nc.RestoreCursorState () ;
   this->ReleaseInputLock () ;
   return status ;

}  //* End RestoreCursorState() *

//*************************
//*   GetKeyProcState     *
//*************************
ncKeyProc NcWindow::GetKeyProcState ( void )
{
   this->AcquireInputLock () ;
   ncKeyProc status = nc.GetKeyProcState () ;
   this->ReleaseInputLock () ;
   return status ;

}  //* End GetKeyProcState() *

//*************************
//*   SetKeyProcState     *
//*************************
short NcWindow::SetKeyProcState ( ncKeyProc procState )
{
   this->AcquireInputLock () ;
   short status = nc.SetKeyProcState ( procState ) ;
   this->ReleaseInputLock () ;
   return status ;

}  //* End SetKeyProcState() *

//*************************
//*     GetKeyDelay       *
//*************************
short NcWindow::GetKeyDelay ( void )
{
   this->AcquireInputLock () ;
   short status = nc.GetKeyDelay () ;
   this->ReleaseInputLock () ;
   return status ;

}  //* End GetKeyDelay() *

//*************************
//*     SetKeyDelay       *
//*************************
short NcWindow::SetKeyDelay ( short delay )
{
   this->AcquireInputLock () ;
   short status = nc.SetKeyDelay ( delay ) ;
   this->ReleaseInputLock () ;
   return status ;

}  //* End SetKeyDelay() *

//*************************
//*   ScreenDimensions    *
//*************************
void NcWindow::ScreenDimensions ( short& scrLines, short& scrColumns )
{

   this->AcquireInputLock () ;
   nc.ScreenDimensions ( scrLines, scrColumns ) ;
   this->ReleaseInputLock () ;

}  //* End ScreenDimensions() *



//*************************
//*      Hibernate        *
//*************************
void NcWindow::Hibernate ( void )
{

   nc.Hibernate () ;

}  //* End Hibernate() *

//*************************
//*        Wake           *
//*************************
void NcWindow::Wake ( void )
{

   nc.Wake () ;

}  //* End Wake() *

//*************************
//* Get_NCurses_Version   *
//*************************
const char* NcWindow::Get_NCurses_Version ( void )
{

   return ( nc.Get_NCurses_Version () ) ;

}  //* End get_NCurses_Version() *

//*************************
//* Get_nclibrary_Version *
//*************************
const char* NcWindow::Get_nclibrary_Version ( void )
{

   return ( nc.Get_nclibrary_Version () ) ;

}  //* End get_nclibrary_Version() *


      //*************************************************
      //**   This Section Implements the Methods for   **
      //**      NcWindow-class Mouse Support.          **
      //** - - - - - - - - - - - - - - - - - - - - - - **
      //**                                             **
      //**                                             **
      //*************************************************

//**************************
//*     meEnableMouse      *
//**************************
//******************************************************************************
//* Enable reporting of mouse events.                                          *
//* Mouse events are reported within the 'GetKeyInput' keyboard input stream.  *
//*                                                                            *
//* Input  : eventTypes : bit mask of event types to be enabled                *
//*                       Bit definitions are the BUTTONxxxx group of          *
//*                       macros defined in ncurses.h (yes, it's clunky,       *
//*                       but you only have to define the mask once).          *
//*                       NOTE:                                                *
//*                       If no mouse-event types are specified,               *
//*                       (eventTypes==ZERO), then the following mask is set   *
//*                       as the default:                                      *
//*                          eventTypes = BUTTON1_CLICKED |                    *
//*                                       BUTTON1_DOUBLE_CLICKED |             *
//*                                       REPORT_SCROLL_WHEEL;                 *
//*          interval  : (optional, ncmiDFLT by default)                       *
//*                      the maximum interval to wait between the start of an  *
//*                      event and the end of the event in thousandths (0.001) *
//*                      of a second. Specify an interval OR a member of       *
//*                      enum ncmInterval.                                     *
//*                                                                            *
//* Returns: OK  if mouse-event reporting enabled AND the system supports all  *
//*              requested event types                                         *
//*                                                                            *
//*          ERR if mouse could not be enabled                                 *
//*           OR if optionally-specified 'interval' value is out of range      *
//*           OR one or more of the requested event types is not supported by  *
//*              the system (remaing event types will still be reported)       *
//*              In this case, the meGetEventTypes() method should be called   *
//*              to determine which events will be reported.                   *
//******************************************************************************

short NcWindow::meEnableMouse ( mmask_t eventTypes, short interval )
{
   this->AcquireInputLock () ;
   short status = ERR ;
   if ( (eventTypes & ALL_MOUSE_EVENTS) == ZERO )
      eventTypes = BUTTON1_CLICKED | BUTTON1_DOUBLE_CLICKED | REPORT_SCROLL_WHEEL ;
   status = nc.EnableMouseEvents ( eventTypes ) ;
   if ( (this->meMouseEnabled ()) != false )
   {
      status = this->meSetClickInterval ( interval ) ;
      this->meStable = false ;      // reset restricted-access flag
      this->meAuto   = true ;       // enable auto-conversion
   }
   this->ReleaseInputLock () ;
   return status ;

}  //* End meEnableMouse() *

//*************************
//*  meEnableStableMouse  *
//*************************
//******************************************************************************
//* Enable enhanced reporting of mouse events.                                 *
//* Mouse events are reported within the 'GetKeyInput' keyboard input stream.  *
//*                                                                            *
//* This form of enabling the mouse interface is different from the more       *
//* general meEnableMouse() method above. Whereas meEnableMouse() will continue*
//* even if the system does not fully support the requested configuration,     *
//* this method requires that all necessary parameters be fully configured.    *
//* If the system cannot be configured as required, then the mouse interface   *
//* is disabled and ERR is returned.                                           *
//*                                                                            *
//* The purpose of this configuration method is to reduce the unfortunate      *
//* timing artifacts inherited from the xterm mouse driver and from the        *
//* ncurses(w) C-language library. It is hoped that this will reduce the loss  *
//* of valid mouse-event data and improve overall mouse performance.           *
//*                                                                            *
//* Enable synthesized timing for mouse events, AND restrict the mouse         *
//* interface to event types:                                                  *
//*      metB1_S : Button 1 - single click                                     *
//*      metB1_D : Button 1 - double click                                     *
//*      metB1_T : Button 1 - triple click                                     *
//*      metSW_U : Scroll-wheel - scroll up   (see 'swheel' parameter)         *
//*      metSW_D : Scroll-wheel - scroll down                                  *
//*      with and without modifier keys (CTRL, SHIFT, ALT).                    *
//*                                                                            *
//*                                                                            *
//* Input  : dclick : (optional, ncmiSTABLE by default)                        *
//*                   This parameter indicates the time to wait after the      *
//*                   first click event is received to determine whether       *
//*                   the event is a double-click.                             *
//*                   Specified in thousandths (0.001) of a second OR a        *
//*                   member of enum ncmInterval.                              *
//*          swheel : (optional, 'true' by default)                            *
//*                   'true'  : ScrollWheel events are reported                *
//*                   'false' : ScrollWheel events ARE NOT reported            *
//*                                                                            *
//* Returns: OK  if mouse-event reporting enabled and the system supports all  *
//*              required mouse event types and specified timing parameters.   *
//*          ERR if:                                                           *
//*            a) mouse could not be enabled                                   *
//*            b) if optionally-specified 'dclick' value is out of range       *
//*            c) one or more required event types not supported by the system *
//*            d) the required timing interval could not be established        *
//*            If ERR, then mouse interface will be disabled.                  *
//******************************************************************************
//* NOTES:                                                                     *
//* Mouse events as received from the low-level ncursesw library suffer from   *
//* severe timing artifacts causing loss and/or corruption of mouse-event data.*
//*                                                                            *
//* It is hoped that by setting a very short delay (or no delay) AND by        *
//* receiving only 'release' events, NOT CLICKS, that we can reduce or         *
//* eliminate the timing problems and make the mouse interface more responsive *
//* to user activity.                                                          *
//*                                                                            *
//* User access to configuration of event types and timing are restricted by   *
//* the 'meStable' member flag, so they won't go mucking about.                *
//*                                                                            *
//* The critical portion of this algorithm is the capture and analysis of      *
//* BUTTON1_RELEASED events to synthesize single-click and double-click events.*
//* To streamline the process flow, this is done when the GeyKeyInput() method *
//* above detects a mouse event, which in turn calls meClickSynthesis().       *
//*                                                                            *
//* -   -   -   -   -   -   -   -   -   -   -   -   -   -   -   -   -   -   -  *
//* Programmer's Notes:                                                        *
//* A metB1_P is almost always followed by a metPOS (if ScrollWheel enabled).  *
//* It seems that the only smart thing to do is to scan only for 'release'     *
//* events (forget about 'press'). One release in the interval will be a       *
//* single-click, and two releases in the interval will be a double-click.     *
//*                                                                            *
//* There is a big delay from the first Scroll-Down, to its being reported.    *
//* Not sure there is anything to be done about that; it's in the ncurses      *
//* timing (or xterm).                                                         *
//*                                                                            *
//* We need to reference the NcWindow member flag 'meStable' to disallow user  *
//* access to modifying the event mask or timing after enabling 'stable'       *
//* interface. Leave meAutoconvert() toggle available for debugging.           *
//*                                                                            *
//* The critical portion is to interpret the metB1_R events and construct      *
//* metB1_S and metB1_D events from it. ScrollWheel events won't need much     *
//* modification.                                                              *
//*                                                                            *
//*                      -------------------------                             *
//* Note that somewhere between ncurses v:5.9 and v6.1 the scroll-upward flag  *
//* has changed from "REPORT_MOUSE_POSITION" to "BUTTON5_PRESSED".             *
//*                                                                            *
//* In addition, the modifier keys are no longer reported in conjunction with  *
//* scrolling events, so our Shift+Scroll selection within Textbox controls    *
//* no longer works. (We may be able to create a work-around for this.)        *
//* 27-Dec-2020: As of this date (Wayland v:1.18), modifier keys are once again*
//* reported along with mouse scroll-wheel events. The Shift key is still not  *
//* being reported in conjunction with click/release events.                   *
//*                                                                            *
//* These changes may be related to (or as a response to) the GNOME/Wayland    *
//* mouse driver, so we won't beat up Mr. Dickey too much about it.            *
//*                                                                            *
//*                                                                            *
//******************************************************************************

short NcWindow::meEnableStableMouse ( short dclick, bool swheel )
{
   this->AcquireInputLock () ;
   short status = ERR ;

   //* Validate the double-click interval *
   if ( dclick >= ncmiNONE && dclick <= ncmiMAX )
   {
      //* Raw mouse events are Button #1 release and the ScrollWheel events.*
      mmask_t eventTypes = swheel ? (BUTTON1_RELEASED | REPORT_SCROLL_WHEEL)
                                   : BUTTON1_RELEASED ;
      this->meStable = false ;      // give ourselves access to configuration

      //* If the mouse is not already enabled, enable it now.*
      if ( (this->meMouseEnabled ()) == false )
         nc.EnableMouseEvents ( eventTypes ) ;

      else  //* Mouse previously enabled. Adjust parameters.*
         this->meSetEventTypes ( eventTypes ) ;

      this->meSetClickInterval ( ncmiNONE ) ; // set the click interval
      this->meAuto   = true ;                 // enable auto-conversion
      this->meStable = true ;                 // set restricted-access flag
      this->meDClick = dclick ;               // save our double-click delay

      //* Verify the results *
      mmask_t et = ZERO ;
      short s1 = this->meGetEventTypes ( et ) ;
      short cint ;
      short s2 = this->meGetClickInterval ( cint ) ;
      if ( (s1 == OK) && (s2 == OK) && (et == eventTypes) && (cint == ncmiNONE) )
         status = OK ;
      else     // if all is not happy in Mooseville, disable the mouse interface.
      {
         this->meDisableMouse () ;
      }
   }
   this->ReleaseInputLock () ;
   return status ;

}  //* End meEnableStableMouse() *

//**************************
//*     meDisableMouse     *
//**************************
//******************************************************************************
//* Disable reporting of mouse events.                                         *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: OK  if mouse events successfully disabled, else ERR               *
//******************************************************************************

short NcWindow::meDisableMouse ( void )
{

   this->AcquireInputLock () ;
   this->meAuto   = true ;       // return auto-conversion to default state
   this->meStable = false ;      // reset restricted-access flag
   short status = nc.DisableMouseEvents () ;
   this->ReleaseInputLock () ;
   return status ;

}  //* End meDisableMouse() *

//**************************
//*     meMouseEnabled     *
//**************************
//******************************************************************************
//* Reports whether mouse input is currently enabled.                          *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: 'true'  if mouse-event reporting enabled, else, 'false'           *
//******************************************************************************

bool NcWindow::meMouseEnabled ( void )
{

   this->AcquireInputLock () ;
   short status = nc.MouseEventsEnabled () ;
   this->ReleaseInputLock () ;
   return status ;

}  //* End meMouseEnabled() *

//**************************
//*    meSetEventTypes     *
//**************************
//******************************************************************************
//* Modify the type(s) of mouse events which will be reported.                 *
//*                                                                            *
//* Input  : eventTypes : bit mask of event types to be enabled                *
//*                                                                            *
//* Returns: OK  if ALL requested event-types are supported by the system      *
//*                                                                            *
//*          ERR if:                                                           *
//*            a) mouse-event reporting not enabled                            *
//*            b) no event types specified                                     *
//*            c) one or more of the requested event types is not supported by *
//*              the system (remaining event types will still be reported)     *
//*            d) mouse interface enabled with meEnableStableMouse,            *
//*               so event types may not be modified                           *
//******************************************************************************

short NcWindow::meSetEventTypes ( mmask_t eventTypes )
{
   this->AcquireInputLock () ;
   short status = ERR ;
   if ( (this->meStable == false) &&
        ((this->meMouseEnabled ()) != false) &&
        ((eventTypes & (ALL_MOUSE_EVENTS | REPORT_MOUSE_POSITION)) != ZERO) )
   {
      status = nc.SetMouseEventTypes ( eventTypes ) ;
   }
   this->ReleaseInputLock () ;
   return status ;

}  //* End mcSetEventTypes() *

//**************************
//*    meGetEventTypes     *
//**************************
//******************************************************************************
//* Returns the mouse-event-type mask for currently-enabled event types.       *
//* Generally useful only if meEnableMouse() method returns an error.          *
//*                                                                            *
//* Input  : eventTypes : (by reference) receives the event mask               *
//*                                                                            *
//* Returns: OK  if successful                                                 *
//*          ERR if mouse-event reporting not enabled                          *
//******************************************************************************

short NcWindow::meGetEventTypes ( mmask_t& eventTypes )
{
   this->AcquireInputLock () ;
   short status = nc.GetMouseEventTypes ( eventTypes ) ;
   this->ReleaseInputLock () ;
   return status ;

}  //* End meGetEventTypes() *

//**************************
//*   meSetClickInterval   *
//**************************
//******************************************************************************
//* Specify the maximum interval between a button-press event and a following  *
//* button-release event that will result in a mouse 'click' being reported.   *
//* Note: this interval applies to single-, double- and triple-click events.   *
//* If the interval between press and release is greater than this value, then *
//* a 'click' event will not be reported.                                      *
//*                                                                            *
//* Note that this inverval is optionally set when mouse-event reporting is    *
//* is enabled through the meEnableMouse() method.                             *
//*                                                                            *
//* Input  : waitMax   : interval in thousandths (0.001) of a second           *
//*                      specify an interval OR a member of enum ncmInterval   *
//*          oldWait   : (optional, NULL pointer by default)                   *
//*                      pointer to variable to receive previous interval value*
//* Returns: OK  if successtul                                                 *
//*          ERR if:                                                           *
//*            a) 'waitMax' value out-of-range                                 *
//*            b) mouse interface enabled with meEnableStableMouse,            *
//*               so intra-click delay may not be modified                     *
//*            c) mouse-event reporting not enabled                            *
//******************************************************************************

short NcWindow::meSetClickInterval ( short waitMax, short* oldMax )
{
   this->AcquireInputLock () ;
   short status = ERR ;
   if ( this->meStable == false )
      status = nc.SetMouseClickInterval ( waitMax, oldMax ) ;
   this->ReleaseInputLock () ;
   return status ;

}  //* End meSetClickInterval() *

//**************************
//*   meGetClickInterval   *
//**************************
//******************************************************************************
//* Get the current maximum mouse-click interval in thousandths of a second.   *
//*                                                                            *
//* Input  : waitMax   : (by reference, initial value ignored)                 *
//*                      receives current interval                             *
//* Returns: OK  if successtul                                                 *
//*          ERR if mouse-event reporting not enabled                          *
//******************************************************************************

short NcWindow::meGetClickInterval ( short& waitMax )
{

   this->AcquireInputLock () ;
   short status = nc.GetMouseClickInterval ( waitMax ) ;
   this->ReleaseInputLock () ;
   return status ;

}  //* End meGetClickInterval() *

//**************************
//*   meFlushEventQueue    *
//**************************
//******************************************************************************
//* Remove any mouse events waiting in the mouse-event queue.                  *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: OK  if successful                                                 *
//*          ERR if mouse-event reporting not enabled                          *
//******************************************************************************

short NcWindow::meFlushEventQueue ( void )
{

   this->AcquireInputLock () ;
   short status = nc.FlushMouseEventQueue () ;
   this->ReleaseInputLock () ;
   return status ;

}  //* End meFlushEventQueue() *

//**************************
//*      meAutoFilter      *
//**************************
//******************************************************************************
//* Temporarily disable filtering of spurious (unrequested) mouse events.      *
//* Filtering is enabled by default. Should REMAIN enabled except for testing. *
//*                                                                            *
//* Input  : filter : 'true' enables filtering                                 *
//*                   'false disables filtering                                *
//* Returns: OK if successful, or ERR if mouse-event reporting not enabled     *
//******************************************************************************

short NcWindow::meAutoFilter ( bool filter )
{

   this->AcquireInputLock () ;
   short status = nc.FilterSpuriousMouseEvents ( filter ) ;
   this->ReleaseInputLock () ;
   return status ;

}  //* End meAutoFilter() *

//**************************
//*     meAutoConvert      *
//**************************
//******************************************************************************
//* Within an NcDialog window, mouse event are, by default, automatically and  *
//* intelligently converted to keycodes (when possible). This eliminates the   *
//* need for the application to handle mouse events manually. In general, the  *
//* less the application needs to know about mouse events, the better.         *
//*                                                                            *
//* There may be circumstances, however, when you need to temporarily disable  *
//* this automatic conversion, but use caution: It means that the edit routines*
//* for the dialog controls will return from the edit with the unprocessed     *
//* mouse event, and your user-input loop needs to know what to do with it.    *
//*                                                                            *
//* Input  : enable : 'true'  to enable auto-conversion (default setting)      *
//*                   'false' to disable auto-conversion                       *
//*                                                                            *
//* Returns: OK  if successful                                                 *
//*          ERR if mouse-event reporting not enabled                          *
//******************************************************************************

short NcWindow::meAutoConvert ( bool enable )
{
   this->AcquireInputLock () ;
   short status = ERR ;
   if ( (this->meMouseEnabled ()) != false )
   {
      this->meAuto = enable ;
      status = OK ;
   }
   this->ReleaseInputLock () ;
   return status ;

}  //* End meAutoConvert() *

//**************************
//*    meClickSynthesis    *
//**************************
//******************************************************************************
//* PRIVATE METHOD. Called only from GetKeyInput(), and ONLY if the event is   *
//* one of the events defined in csEVENTS.                                     *
//* Currently, the only event passed to us is the BUTTON1_RELEASED event, so   *
//* so only the following events can be returned:                              *
//*       BUTTON1_CLICKED                                                      *
//*       BUTTON1_DOUBLE_CLICKED                                               *
//*       BUTTON1_TRIPLE_CLICKED                                               *
//*                                                                            *
//* We wait a decent interval to see if a second or third event matching the   *
//* original event arrives in the queue indicating a double-click or           *
//* triple-click, respectively. Otherwise, we return a single-click event.     *
//*                                                                            *
//* Caller has acquired the input-stream lock, so we can muck about as we      *
//* please (for a short time).                                                 *
//*                                                                            *
//* Input  : wk    : (by reference)                                            *
//*                  wkeyCode object containing the raw mouse event            *
//*                                                                            *
//* Returns: OK  if successful                                                 *
//*          ERR if unable to synthesize valid data                            *
//*              (wk members will be set to indicate an invalid keystroke)     *
//******************************************************************************
//* Perform a series of non-blocking reads of the input stream until we get    *
//* a valid read or until timeout.                                             *
//* a) If it's a mouse event, compare it with the original event.              *
//*    If it compares, convert it, otherwise discard it.                       *
//* b) If it's a keyboard event, it's not ours, so push it back into the       *
//*    stream.                                                                 *
//*                                                                            *
//******************************************************************************

short NcWindow::meClickSynthesis ( wkeyCode& wk )
{
   short status = ERR ;

   //* Interval between polls of the input queue *
   chrono::duration<short, std::milli>pollDelay( ((this->meDClick / 2) + 1) ) ;

   short oldKeyDelay = nc.GetKeyDelay () ;   // save current delay
         nc.SetKeyDelay ( nckdNODELAY ) ;    // set non-blocking reads

   wkeyCode moosePoop,           // gather data from the stream
            dPoop,               // potential double-click event
            tPoop ;              // potential triple-click event
   short mClicks = 1 ;

   short loopCount = 2 ;
   while ( (loopCount--) > ZERO )
   {
      this_thread::sleep_for( pollDelay ) ;
      nc.GetKeyInput ( moosePoop ) ;

      if ( moosePoop.type == wktMOUSE )
      {
         if ( ++mClicks == 2 )
         {
            dPoop = moosePoop ;  // second mouse event
            loopCount += 2 ;     // wait a bit longer
         }
         else
         {
            tPoop = moosePoop ;  // third mouse event
            break ;
         }
      }
      else if ( moosePoop.type != wktERR )
      {  //* Received a non-mouse event (doesn't belong to us) *
         nc.UngetKeyInput ( moosePoop ) ;
         break ;
      }
   }

   //* Restore original key-input delay *
   nc.SetKeyDelay ( oldKeyDelay ) ;

   //* If we have received a second or third mouse event within the   *
   //* delay interval, then compare the event type and Y/X/Z position *
   //* to determine whether it is a double- or triple-click.          *
   if ( mClicks > 1 )
   {
      //* Compare the second event with the original event *
      if ( (dPoop.mevent.eventType == wk.mevent.eventType) && 
           (dPoop.mevent.sypos == wk.mevent.sypos) && 
           (dPoop.mevent.sxpos == wk.mevent.sxpos) && 
           (dPoop.mevent.szpos == wk.mevent.szpos) )
      {
         wk.mevent.eventType = (wk.mevent.eventType & ~(BUTTON1_RELEASED))
                                | BUTTON1_DOUBLE_CLICKED ;
         status = OK ;

         //* If there is a third event, compare it to the second event.*
         if ( (mClicks > 2) && 
              (dPoop.mevent.eventType == tPoop.mevent.eventType) && 
              (dPoop.mevent.sypos == tPoop.mevent.sypos) && 
              (dPoop.mevent.sxpos == tPoop.mevent.sxpos) && 
              (dPoop.mevent.szpos == tPoop.mevent.szpos) )
            
         {
            wk.mevent.eventType = (wk.mevent.eventType & ~(BUTTON1_DOUBLE_CLICKED))
                                   | BUTTON1_TRIPLE_CLICKED ;
         }
      }
      else        // Second event does not match, single-click only
         mClicks = 1 ;
   }

   //* Else, this is a single-click event, and any secondary  *
   //* events have been silently discarded.                   *
   if ( mClicks == 1 )
   {
      if ( (wk.mevent.eventType & BUTTON1_RELEASED) == BUTTON1_RELEASED )
      {
         wk.mevent.eventType = 
               (wk.mevent.eventType & ~(BUTTON1_RELEASED)) | BUTTON1_CLICKED ;
         status = OK ;
      }
      #if 0    // NOT CURRENTLY USED
      else if ( (wk.mevent.eventType & BUTTON2_RELEASED) == BUTTON2_RELEASED )
      {
      }
      else if ( (wk.mevent.eventType & BUTTON3_RELEASED) == BUTTON3_RELEASED )
      {
      }
      else if ( (wk.mevent.eventType & BUTTON4_RELEASED) == BUTTON4_RELEASED )
      {
      }
      #endif   // NOT CURRENTLY USED
      else
      {  //* This should never happen because our input is pre-tested.*
         wk.type = wktERR ;
         wk.key = ZERO ;
      }
   }
   return status ;

}  //* End meClickSynthesis() *

//**************************
//*  meIdentifyClickItem   *
//**************************
//******************************************************************************
//* PROTECTED METHOD. Called ONLY from NcDialog meTransformEvent() method.     *
//*                                                                            *
//* For scrolling controls: Scrollbox, Scrollext, expanded Dropdown and        *
//* expanded Menuwin, this method identifies the index of the line item under  *
//* the mouse pointer.                                                         *
//*                                                                            *
//* NOTE: This is a dangerous method because it digs deep into the NcWindow    *
//*       object's protected data. We must trust our caller to pass us a valid *
//*       window pointer; however, at this level, we don't event know the      *
//*       concept of a user-interface control, so we can't MODIFY the window-  *
//*       level data without risking the integrity of the caller's data.       *
//*                                                                            *
//* Input  : wPtr    : pointer to NcWindow object containing the scrolling data*
//*          yOffset : mouse pointer offset from top border of target control  *
//*                    range: ZERO to target-rows-minus-1                      *
//*                                                                            *
//* Returns: if successful, returns index of target line item                  *
//*          returns ERR if:                                                   *
//*            a) mouse-event reporting not enabled                            *
//*            b) target window does not contain scrolling data                *
//*            c) 'yOffset' is out-of-range                                    *
//******************************************************************************

short NcWindow::meIdentifyClickItem ( NcWindow* wPtr, short yOffset )
{
   short status = ERR ;
   this->AcquireInputLock () ;
   if ( (this->meMouseEnabled ()) && (wPtr->scrollData.count > ZERO) &&
        (yOffset >= ZERO) && (yOffset < wPtr->wLines) )
   {
      status = wPtr->scrollData.tIndex + yOffset ;
   }
   this->ReleaseInputLock () ;
   return status ;

}  //* End meIdentifyClickItem() *

//******************************************************************************
//*                       End of Mouse Interface Methods                       *
//******************************************************************************


//* - * - * - * - * - * - * - * - * - * - * - * - * - * - * - * - * - * - * - **
//** - - - This section implements the mutex locks for thread safety.  - - - -**
//* - * - * - * - * - * - * - * - * - * - * - * - * - * - * - * - * - * - * - **
//* Thread-safe operation:                                                     *
//* 1) The 'ncurses' and 'ncursesw' libraries included as part of Linux/UNIX   *
//*    systems ARE NOT thread safe.                                            *
//* 2) The NCurses class methods provide direct access to the 'ncursesw' input *
//*    and output streams, and are therefore also not thread safe.             *
//* 3) The NcWindow class (and its derived classes) implement thread safety    *
//*    through two exclusive-access-lock <recursive_mutex> objects:            *
//*    a) 'InputStream_Lock' for keyboard/mouse input, and                     *
//*    b) 'OutputStream_Lock' for data output to the display.                  *
//*    These mutexes are located in NcWindow.cpp, and all input/output methods *
//*    must obtain exclusive access to the stream before each operation.       *
//* 4) Cursor positioning: The cursor (output insertion point) is a single,    *
//*    shared resource, so at no time, should you rely upon the cursor being   *
//*    where you left it because Forces Of Evil (another thread) may have      *
//*    moved it. This is why all methods which write to the screen are         *
//*    designed to EXPLICITlY set the cursor position before starting the      *
//*    output.                                                                 *
//*    a) This is also why the 'GetCursor' method is unreliable in             *
//*       multithreaded applications.                                          *
//*    b) Note that if a thread makes the cursor visible, then the user may be *
//*       surprised to see the cursor jumping around as other threads write    *
//*       data to the display. There is no elegant way to prevent this without *
//*       inter-thread communications regarding what each thread is doing with *
//*       the cursor, so, it is recommended that the cursor remain invisible   *
//*       as much as possible in order to reduce user confusion.               *
//*    c) Under extreme duress, a multi-threaded output will occasionally show *
//*       screen artifacts - this is unavoidable due to inherent weaknesses in *
//*       the underlying 'ncursesw' library, and hardware/kernel issues as     *
//*       well. However, we feel an overall satisfaction with the results of   *
//*       our thread-safety campaign.                                          *
//* 5) Hibernate() and Wake() methods:                                         *
//*    Before a thread calls Hibernate(), putting the NCurses Engine to sleep, *
//*    all other program threads must have been either 'join'ed (terminated),  *
//*    OR blocked. It would be quite embarrassing for a thread to run amok,    *
//*    calling NCurses/NcWindow/NcDialog methods that are not available.       *
//* 6) Please note that multithreading IS NOT USED within the NcWindow class   *
//*    nor any of its descendants, and thus, the 'lpthread' library need not   *
//*    be linked to the target application unless the application itself uses  *
//*    multithreading.                                                         *
//* 7) For convenience, thread safety is controlled by the NCD_THREAD_SAFETY   *
//*    flag in NcWindow.hpp, so the mutexes can be temporarily disabled during *
//*    testing; however, this flag should be enabled for all production builds.*
//* - * - * - * - * - * - * - * - * - * - * - * - * - * - * - * - * - * - * - **

//*************************
//*   AcquireInputLock    *
//*************************
//******************************************************************************
//* PRIVATE METHOD:                                                            *
//* Acquire an exclusive lock on the data input stream.                        *
//* 1) Calling thread will sleep until lock acquired.                          *
//* 2) Upon task completion, the thread must have called ReleaseInputLock()    *
//*    exactly once for each time AcquireInputLock() was called.               *
//*    This ensures that the number of locks held by the thread == ZERO.       *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void NcWindow::AcquireInputLock ( void )
{

   #if NCD_THREAD_SAFETY != 0    // Acquire an exclusive lock
   InputStream_Lock.lock() ;
   #endif   // NCD_THREAD_SAFETY

}  //* End AcquireInputLock() *

void NcWindow::ReleaseInputLock ( void )
{

   #if NCD_THREAD_SAFETY != 0    // Release the exclusive lock
   InputStream_Lock.unlock() ;
   #endif   // NCD_THREAD_SAFETY

}  //* End ReleaseInputLock() *

//*************************
//*   AcquireOutputLock   *
//*************************
//******************************************************************************
//* PRIVATE METHOD:                                                            *
//* Acquire an exclusive lock on the data output stream.                       *
//* 1) Calling thread will sleep until lock acquired.                          *
//* 2) Upon task completion, the thread must have called ReleaseOutputLock()   *
//*    exactly once for each time AcquireOutputLock() was called.              *
//*    This ensures that the number of locks held by the thread == ZERO.       *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void NcWindow::AcquireOutputLock ( void )
{

   #if NCD_THREAD_SAFETY != 0    // Acquire an exclusive lock
   OutputStream_Lock.lock() ;
   #endif   // NCD_THREAD_SAFETY

}  //* End AcquireOutputLock() *

void NcWindow::ReleaseOutputLock ( void )
{

   #if NCD_THREAD_SAFETY != 0    // Release the exclusive lock
   OutputStream_Lock.unlock() ;
   #endif   // NCD_THREAD_SAFETY

}  //* End ReleaseOutputLock() *

//*************************
//*                       *
//*************************
//******************************************************************************
//*                                                                            *
//*                                                                            *
//*                                                                            *
//* Input  :                                                                   *
//*                                                                            *
//* Returns:                                                                   *
//******************************************************************************


