//********************************************************************************
//* File       : NcdControlSL.cpp                                                *
//* Author     : Mahlon R. Smith                                                 *
//*              Copyright (c) 2006-2025 Mahlon R. Smith, The Software Samurai   *
//*                  GNU GPL copyright notice located in NcDialog.hpp            *
//* Date       : 31-Aug-2021                                                     *
//* Version    : (see NcDialogVersion string in NcDialog.cpp)                    *
//*                                                                              *
//* Description: Contains the methods of the DialogSlider class.                 *
//*              and the NcDialog class EditSlider() method.                     *
//*              See NcDialog.hpp for the definition of this class.              *
//*                                                                              *
//*                                                                              *
//* Development Tools: See NcDialog.cpp.                                         *
//********************************************************************************
//* Version History (most recent first):                                         *
//*   See version history in NcDialog.cpp.                                       *
//********************************************************************************
//* Notes:                                                                       *
//* ------                                                                       *
//* 1) The application may perform configuration AFTER instantiation through     *
//*    SetSliderConfig(), _OR_ the 'sliderData' object may be attached to the    *
//*    'spinData' member of the InitCtrl class. This is carefully documented.    *
//* 2) Directional sense of user-interface keys is not completely intuitive.     *
//*    Arrow keys grow/shrink the bar in the expected direction, but Home/End,   *
//*    PgUp/PgDown are ambiguous. (Home/End for "standard" directions use the    *
//*    same sense as the Spinner control, and we want to maintain that           *
//*    similarity, but they are swapped for "reverse" direction. Is that right?  *
//*    PgUp/PgDown keys are reasonable for "standard" direction, but are         *
//*    swapped for "reverse" direction. Is that right?                           *
//* 3) The 'miniChar' (minimum-value character) is logical to us because the     *
//*    arrow points to where the bar _would be_ if value was above minimum;      *
//*    however what is logical to us may not be logical to the average user.     *
//*    a) Also, would the smaller arrows look better?                            *
//* 4) An option to format and display the current value (or percentage)         *
//*    during EditSlider() is functional (see SliderAutoReport()), but the       *
//*    algorithm is not pristine. Is there a way to improve our defense          *
//*    against caller's potential dumbassery?                                    *
//*                                                                              *
//*                                                                              *
//* Interesting character codepoints:                                            *
//* ---------------------------------                                            *
//* const wchar_t TRIGRAM_HEAVEN   = 0x2630 ;    // U+2630 '☰'                   *
//* const wchar_t TRIGRAM_LAKE     = 0x2631 ;    // U+2631 '☱'                   *
//* const wchar_t TRIGRAM_FIRE     = 0x2632 ;    // U+2632 '☲'                   *
//* const wchar_t TRIGRAM_THUNDER  = 0x2633 ;    // U+2633 '☳'                   *
//* const wchar_t TRIGRAM_WIND     = 0x2634 ;    // U+2634 '☴'                   *
//* const wchar_t TRIGRAM_WATER    = 0x2635 ;    // U+2635 '☵'                   *
//* const wchar_t TRIGRAM_MOUNTAIN = 0x2636 ;    // U+2636 '☶'                   *
//* const wchar_t TRIGRAM_EARTH    = 0x2637 ;    // U+2637 '☷'                   *
//*                                                                              *
//* const wchar_t TRIANGLE_UPb     = 0x25B2 ;    // U+25B2 '▲'                   *
//* const wchar_t TRIANGLE_UPbs    = 0x25B4 ;    // U+25B4 '▴'                   *
//* const wchar_t TRIANGLE_UPw     = 0x25B3 ;    // U+25B3 '△'                   *
//* const wchar_t TRIANGLE_UPws    = 0x25B5 ;    // U+25B5 '▵'                   *
//* const wchar_t TRIANGLE_DOWNb   = 0x25BC ;    // U+25BC '▼'                   *
//* const wchar_t TRIANGLE_DOWNbs  = 0x25BE ;    // U+25BE '▾'                   *
//* const wchar_t TRIANGLE_DOWNw   = 0x25BD ;    // U+25BD '▽'                   *
//* const wchar_t TRIANGLE_DOWNws  = 0x25BF ;    // U+25BF '▿'                   *
//* const wchar_t TRIANGLE_LEFTb   = 0x25C0 ;    // U+25C0 '◀'                   *
//* const wchar_t TRIANGLE_LEFTbs  = 0x25C2 ;    // U+25C2 '◂'                   *
//* const wchar_t TRIANGLE_LEFTw   = 0x25C1 ;    // U+25C1 '◁'                   *
//* const wchar_t TRIANGLE_LEFTws  = 0x25C3 ;    // U+25C3 '◃'                   *
//* const wchar_t TRIANGLE_RIGHTb  = 0x25B6 ;    // U+25B6 '▶'                   *
//* const wchar_t TRIANGLE_RIGHTbs = 0x25B8 ;    // U+25B8 '▸'                   *
//* const wchar_t TRIANGLE_RIGHTw  = 0x25B7 ;    // U+25B7 '▷'                   *
//* const wchar_t TRIANGLE_RIGHT   = 0x25B9 ;    // U+25B9 '▹'                   *
//*                                                                              *
//* const wchar_t UPPER_DIV        = 0x2594 ;    // U+2594 '▔'                   *
//* const wchar_t RIGHT_DIV        = 0x2595 ;    // U+2595 '▕'                   *
//*                                                                              *
//*                                                                              *
//********************************************************************************


//*****************
//* Include Files *
//*****************

#include "GlobalDef.hpp"               //* General definitions


//***************
//* Prototypes  *
//***************
static short InvertFractionalCell ( short blockIndex ) ;
static attr_t InvertColorAttribute ( attr_t normColor ) ;


//*****************
//*   Local Data  *
//*****************
static const wchar_t TRIANGLE_UPb    = 0x025B2 ;   // U+25B2 '▲'
static const wchar_t TRIANGLE_DOWNb  = 0x025BC ;   // U+25BC '▼'
static const wchar_t TRIANGLE_LEFTb  = 0x025C0 ;   // U+25C0 '◀'
static const wchar_t TRIANGLE_RIGHTb = 0x025B6 ;   // U+25B6 '▶'
static const wchar_t UPPER_DIV       = 0x02594 ;   // U+2594 '▔'
static const wchar_t RIGHT_DIV       = 0x02595 ;   // U+2595 '▕'
static const wchar_t LOWER_DIV       = 0x02581 ;   // U+2581 '▁'
static const wchar_t LEFT_DIV        = 0x0258F ;   // U+258F '▏'
 
//* Divisions per character cell (vertical OR horizontal) *
static const short cellDIVS = 8 ;
//* Character definitions for drawing graphs *
static const short wcINDX = 7 ;         // index of whole-cell character
static const wchar_t hBlock[cellDIVS] = // for horizontal bar graphs
{
   0x0258F,    // ▏
   0x0258E,    // ▎
   0x0258D,    // ▍
   0x0258C,    // ▌
   0x0258B,    // ▋
   0x0258A,    // ▊
   0x02589,    // ▉
   0x02588,    // █
} ;
static const wchar_t vBlock[cellDIVS] = // for vertical bar graphs
{
   0x02581,    // ▁
   0x02582,    // ▂
   0x02583,    // ▃
   0x02584,    // ▄
   0x02585,    // ▅
   0x02586,    // ▆
   0x02587,    // ▇
   0x02588,    // █
} ;

      //*************************************************
      //** THIS SECTION IMPLEMENTS THE METHODS OF THE  **
      //**             DialogSlider CLASS.             **
      //** - - - - - - - - - - - - - - - - - - - - - - **
      //** See below for public NcDialog-class methods **
      //**    related to DialogSlider-class access.    **
      //*************************************************

//*************************
//*     DialogSlider      *
//*************************
//******************************************************************************
//* Constructor for DialogSlider class object.                                 *
//*                                                                            *
//*                                                                            *
//* Input  : pointer to initialization structure                               *
//*                                                                            *
//* Returns: implicitly returns a pointer to the control object                *
//******************************************************************************

DialogSlider::DialogSlider ( InitCtrl* iPtr )
{
   this->type   = dctSLIDER ;       // set control type
   this->ulY    = iPtr->ulY ;       // control's absolute offset within terminal window
   this->ulX    = iPtr->ulX ;
   this->rows   = iPtr->lines ;     // control's vertical and horizontal dimensions
   this->lines = this->rows ;       /* 'lines' IS DEPRECATED */
   this->cols   = iPtr->cols ;

   this->nColor = iPtr->nColor ;    // non-focus color
   this->fColor = iPtr->fColor ;    // focus color
   this->labY   = iPtr->labY ;      // label offsets
   this->labX   = iPtr->labX ;
   this->active = iPtr->active ;    // can control receive input focus?

   //* Copy the display text to data member, wLabel and *
   //* initialize the bHotkey. (allow multi-line label) *
   this->InitHotkey ( iPtr->label, false ) ;

   //* Default values for all other data members *
   this->groupCode = ZERO ;         // sliders are not grouped (not used)
   this->rtlContent = false ;       // control contains no text (not used)
   this->audibleAlert = false ;     // alert disabled by default

   //* If caller has attached a sliderData configuration object *
   //* to the 'spinData' member of the initialization structure,*
   //* perform configuration.                                   *
   if ( iPtr->spinData != NULL )
   {
      sliderData *sdptr = (sliderData*)iPtr->spinData ;  // pointer to object
      this->Configure ( *sdptr ) ;
   }
   //* Configuration data not provided.           *
   //* Set default values for operational members.*
   else
      this->opReset () ;

   //* Instantiate the underlying window object *
   this->wPtr = new NcWindow ( this->rows, this->cols, this->ulY, this->ulX ) ;

}  //* End DialogSlider()

//*************************
//*     ~DialogSlider     *
//*************************
//******************************************************************************
//* Destructor for DialogSlider class object.                                  *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

DialogSlider::~DialogSlider ( void )
{

   //* Close the underlying window *
   delete this->wPtr ;

}  //* End ~DialogSlider() *

//*************************
//*        opReset        *
//*************************
//******************************************************************************
//* Private Method:                                                            *
//* ---------------                                                            *
//* Reset operational parameters. These are the parameters used during user    *
//* modification of the control. The members which define the control object's *
//* position, dimensions and colors are not modified.                          *
//*                                                                            *
//* Note on parameters:                                                        *
//*  a) When called by the constructor, 'force' == false, so parameters are    *
//*     set to the default state because at that time we don't know the        *
//*     intended settings. Note that the control will be fully functional with *
//*     the calculated defaults, but may not be what the application intends.  *
//*     It is expected that the application will subsequently call the         *
//*     configuration method to finalize the settings.                         *
//*  b) When called by SetSliderConfig(), the parameter states are passed      *
//*     through from the calling application.                                  *
//*                                                                            *
//* Input  : force  : (optional, 'false' by default)                           *
//*                   if 'false', assume that the larger dimension is the      *
//*                               growth orientation, and that growth direction*
//*                               is "standard" (up/right).                    *
//*                   if 'true',  use caller's parameters for orientation and  *
//*                               growth direction                             *
//*          vOrient: (optional, 'true' by default, ignored if force==false)   *
//*                   if 'true',  vertical bar orientation                     *
//*                   if 'false', horizontal bar orientation                   *
//*          growRev: (optional, 'false' by default, ignored if force==false)  *
//*                   if 'false', set standard (upward/rightward) growth       *
//*                   if 'true',  set reverse (downward/leftward) growth       *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void DialogSlider::opReset ( bool force, bool vOrient, bool growRev )
{
   //* Set default values:                                  *
   //* Calculate available divisions for growth orientation *
   //* and direction, and set compatible special characters.*
   if ( !force )
   {
      if ( this->rows >= this->cols )              // orientation == vertical
      {
         this->divTotal = this->rows * cellDIVS ;  // Divisions available
         this->vertical = true ;                   // Bar grows vertically
         this->ruleChar = UPPER_DIV ;              // Ruler character
         this->miniChar = TRIANGLE_DOWNb ;         // Minimum-value character
      }
      else                                         // orientation == horizontal
      {
         this->divTotal = this->cols * cellDIVS ;  // Divisions available
         this->vertical = false ;                  // Bar grows horizontally
         this->ruleChar = RIGHT_DIV ;              // Ruler character
         this->miniChar = TRIANGLE_LEFTb ;         // Minimum-value character
      }
      this->reverse = false ;                      // growth direction == standard
   }

   //* Set values according to caller's flags:      *
   //* vOrient and growRev determine initialization.*
   else
   {
      this->vertical = vOrient ;                   // Caller's orientation
      this->reverse  = growRev ;                   // Caller's growth direction
      if ( this->vertical )                        // orientation == vertical
      {
         this->divTotal = this->rows * cellDIVS ;  // Divisions available
         //* Set special characters according to specified growth direction.*
         if ( this->reverse )
         {
            this->ruleChar = vBlock[ZERO] ;        // Ruler character
            this->miniChar = TRIANGLE_UPb ;        // Minimum-value character
         }
         else
         {
            this->ruleChar = UPPER_DIV ;           // Ruler character
            this->miniChar = TRIANGLE_DOWNb ;      // Minimum-value character
         }
      }
      else                                         // orientation == horizontal
      {
         this->divTotal = this->cols * cellDIVS ;  // Divisions available
         //* Set special characters according to specified growth direction.*
         if ( this->reverse )
         {
            this->ruleChar = hBlock[ZERO] ;        // Ruler character
            this->miniChar = TRIANGLE_RIGHTb ;     // Minimum-value character
         }
         else
         {
            this->ruleChar = RIGHT_DIV ;           // Ruler character
            this->miniChar = TRIANGLE_LEFTb ;      // Minimum-value character
         }
      }
   }

   //* Set default max, min and current values *
   this->maxVal = double(this->divTotal) ;      // maximum value
   this->minVal = 0.0 ;                         // minimum value
   this->rngVal = this->maxVal ;                // value range
   this->divCurr = ZERO ;                       // current value divisions

}  //* End opReset() *

//*************************
//*       Configure       *
//*************************
//******************************************************************************
//* Specify the minimum, maximum and current values as well as the control     *
//* orientation (vertical vs. horizontal) and the growth direction             *
//* (right/up vs. left/down).                                                  *
//*                                                                            *
//* Input  : slData: (by reference) a partially-initialized instance of the    *
//*                  sliderData class with the following members initialized:  *
//*                  minValue  : minimum value represented                     *
//*                  maxValue  : maximum value represented                     *
//*                  curValue  : initial value setting                         *
//*                  vertical  : orientation                                   *
//*                              'true' : grows/shrinks vertically             *
//*                              'false': grows/shrinks horizontally           *
//*                  reverse   : direction                                     *
//*                              'false': if 'vertical', grows upward          *
//*                                       if 'horizontal, grows rightward      *
//*                              'true' : if 'vertical', grown downward        *
//*                                       if 'horizontal', grows leftward      *
//*                  ---------                                                 *
//*                  Other member values is are ignored.                       *
//*                                                                            *
//* Returns: 'true'  if successful                                             *
//*          'false' if invalid input (default values used),                   *
//******************************************************************************
//* Notes:                                                                     *
//* ======                                                                     *
//* Parameter validation:                                                      *
//*   a) The minimum and maximum values may be any floating-point values       *
//*      provided that the maximum is above the minimum.                       *
//*   b) Initial value must be within the limits.                              *
//*   c) Growth orientation ('vertical') and growth direction ('reverse') are  *
//*      always valid.                                                         *
//******************************************************************************

bool DialogSlider::Configure ( sliderData& slData )
{
   bool status = false ;

   //* Set operational parameters to defaults, BUT  *
   //* use caller's settings for the boolean values.*
   this->opReset ( true, slData.vertical, slData.reverse ) ;

   //* Validate the numeric input *
   if ( (slData.maxValue > slData.minValue) &&
        ((slData.curValue >= slData.minValue) && 
         (slData.curValue <= slData.maxValue)) )
   {
      this->minVal   = slData.minValue ;
      this->maxVal   = slData.maxValue ;
      this->rngVal   = fabs ( this->maxVal - this->minVal ) ;
      this->divCurr  = short(this->divTotal * 
                     fabs ( slData.curValue - this->minVal ) / this->rngVal) ;
      status = true ;

      //* Initialize the remaining members of caller's structure. *
      // Programmer's Note: There are a few redundant assignments performed by 
      // this call, but this avoids duplication of critical calculations 
      // in multiple places.
      this->Report ( slData ) ;
   }

   return status ;

}  //* End Configure() *

//*************************
//*     SetCurrValue      *
//*************************
//******************************************************************************
//* Set the current value of the control, either as a value or as a percentage *
//* of the range.                                                              *
//*                                                                            *
//* Note: The actual value set may be slightly different from the specified    *
//*       value due to rounding errors during conversion of the floating-point *
//*       value to an integer step size.                                       *
//*                                                                            *
//* Input  : slValue: new current value                                        *
//*          percent: if 'true',  interpret 'slValue' as a percentage          *
//*                   if 'false', interpret 'slValue' as an actual value       *
//*                                                                            *
//* Returns: 'true'  if successful                                             *
//*          'false' if value out-of-range                                     *
//******************************************************************************

bool DialogSlider::SetCurrValue ( double slValue, bool percent )
{
   bool status = true ;

   //* If caller's value is a percentage of the value range, *
   //* and the percentage is in range.                       *
   if ( percent && ((slValue >= 0.0) && (slValue <= 100.0)) )
      this->divCurr  = short(this->divTotal * (slValue / 100.0)) ;

   //* If caller's value is a value within the value range *
   else if ( !percent && ((slValue >= this->minVal) && (slValue <= this->maxVal)) )
   {
      double pctRng = fabs (slValue - this->minVal) / this->rngVal ;
      this->divCurr = short(this->divTotal * pctRng) ;
   }

   //* Specified value is out-of-range *
   else
      status = false ;

   return status ;

}  //* End SetCurrValue() *

//*************************
//*        Report         *
//*************************
//******************************************************************************
//* Returns the current configuration.                                         *
//*                                                                            *
//* Input  : slData: (by reference, initial values ignored)                    *
//*                  receives the current setup information for the control    *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void DialogSlider::Report ( sliderData& slData )
{
   //* Calculate the current value as both a decimal value *
   //* AND as a percentage of the value range.             *
   this->ReportValue ( slData.curValue, slData.curPercent ) ;

   slData.minValue   = this->minVal ;
   slData.maxValue   = this->maxVal ;
   slData.divSteps   = this->divTotal ;
   slData.divCurrent = this->divCurr ;
   slData.vertical   = this->vertical ;
   slData.reverse    = this->reverse ;

}  //* End Report() *

//*************************
//*      ReportValue      *
//*************************
//******************************************************************************
//* Returns the currently-displayed value as both a decimal value AND as a     *
//* percentage of the value range.                                             *
//*                                                                            *
//* Input  : curVal: (by reference, initial value ignored)                     *
//*                  receives currently displayed value                        *
//*          curPct: (by reference, initial value ignored)                     *
//*                  receives value as a percentage of value range             *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void DialogSlider::ReportValue ( double& curVal, double& curPct )
{

   curVal = this->minVal + this->rngVal * ((double)this->divCurr /
                                           (double)this->divTotal) ;
   curPct = (fabs(curVal - this->minVal) / this->rngVal) * 100.0 ;

}  //* End ReportValue() *

//*************************
//*      OpenControl      *
//*************************
//******************************************************************************
//* Draw a dialog slider control i.e. open the window previously               *
//* instantiated by a call to DialogSlider().                                  *
//*                                                                            *
//* Input  : hasFocus : optional boolean, false by default, determines         *
//*                     whether to draw the object in nColor or fColor         *
//*                                                                            *
//* Returns: OK if successful, else ERR                                        *
//******************************************************************************

short DialogSlider::OpenControl ( bool hasFocus )
{
   short result ;

   //* Set the control's background color *
   this->wPtr->SetInteriorColor ( hasFocus ? this->fColor : this->nColor ) ;

   //* Initialize the control's underlying window *
   if ( (result = this->wPtr->OpenWindow ()) == OK )
   {
      //* Draw control on the screen *
      this->RedrawControl ( hasFocus ) ;
   }

   return result ;

}  //* End OpenControl() *

//*************************
//*    RedrawControl      *
//*************************
//******************************************************************************
//* Redraw the data in a Slider window.                                        *
//* (refreshes the control's window using existing data)                       *
//*                                                                            *
//* Input  : hasFocus: if true, control has focus                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void DialogSlider::RedrawControl ( bool hasFocus )
{
   wchar_t fillChar ;                  // character to write
   short   fillCnt = this->divCurr,
           fillIndx = fillCnt - 1 ;    // index into fill-character array
   winPos wp ;
   attr_t barColor, fbarColor, rbarColor, gridColor = this->nColor ;
   if ( hasFocus )
   {
      barColor = fbarColor = this->fColor ;
      rbarColor = InvertColorAttribute ( barColor ) ;
   }
   else
   {
      barColor = fbarColor = this->nColor ;
      rbarColor = InvertColorAttribute ( barColor ) ;
   }

   //* Draw the fill data *
   if ( this->vertical )         // vertical fill
   {
      short yinc, x, tc ;
      wp.xpos = ZERO ;           // left column
      if ( this->reverse )       // fills from top
      { wp.ypos = ZERO ; yinc = 1 ; }
      else
      { wp.ypos = this->rows - 1 ; yinc = -1 ; }
      if ( fillIndx >= wcINDX )  // full-height character
         fillChar = vBlock[wcINDX] ;
      else                       // indicates zero divisions
         fillChar = this->miniChar ;

      for ( short y = ZERO ; y < this->rows ; ++y )
      {
         //* If partial cell, select appropriate character *
         if ( (fillChar != nckSPACE) && (fillIndx < wcINDX) && (fillIndx >= ZERO) )
         {
            if ( this->reverse )
            {
               fillIndx = InvertFractionalCell ( fillIndx ) ;
               barColor = rbarColor ;
            }
            fillChar = vBlock[fillIndx] ;
         }

         //* If a column is reserved for the ruler, draw it in left column *
         x = (this->cols > 1 ? 1 : ZERO) ;
         if ( x == 1 )
            this->wPtr->WriteChar ( wp, this->ruleChar, gridColor ) ;

         if ( fillChar == this->miniChar )
         {
            //* Position the zero-indicator character *
            switch ( this->cols )
            {
               case 1: case 2: tc = x ;         break ;
               case 3:         tc = 2 ;         break ;
               default: tc = this->cols / 2 ;   break ;
            }
            for ( ; x < this->cols ; ++x )
            {
               this->wPtr->WriteChar ( wp.ypos, wp.xpos + x, 
                                       (x == tc ? this->miniChar : nckSPACE),
                                       barColor ) ;
            }
         }
         else
         {
            for ( ; x < this->cols ; ++x )
               this->wPtr->WriteChar ( wp.ypos, wp.xpos + x, fillChar, barColor ) ;
         }
         wp.ypos += yinc ;

         //* Decrement the division index *
         if ( (fillChar == this->miniChar) || ((fillIndx -= cellDIVS) < ZERO) )
         {
            //* End of bar, fill remaining cells with spaces *
            fillIndx = -1 ;
            fillChar = nckSPACE ;
            if ( this->reverse )
               barColor = fbarColor ;
         }
      }
   }
   else                          // horizontal fill
   {
      short xinc, y, tc ;
      wp.ypos = this->rows - 1 ; // bottom row
      if ( this->reverse )       // fills from right
      { wp.xpos = this->cols - 1 ; xinc = -1 ; }
      else
      { wp.xpos = ZERO ; xinc = 1 ; }
      if ( fillIndx >= wcINDX )  // full-height character
         fillChar = hBlock[wcINDX] ;
      else                       // indicates zero divisions
         fillChar = this->miniChar ;

      for ( short x = ZERO ; x < this->cols ; ++x )
      {
         //* If partial cell, select appropriate character *
         if ( (fillChar != nckSPACE) && (fillIndx < wcINDX) && (fillIndx >= ZERO) )
         {
            if ( this->reverse )
            {
               fillIndx = InvertFractionalCell ( fillIndx ) ;
               barColor = rbarColor ;
            }
            fillChar = hBlock[fillIndx] ;
         }

         //* If a row is reserved for the ruler, draw it in lower row *
         y = (this->rows > 1 ? 1 : ZERO) ;
         if ( y == 1 )
            this->wPtr->WriteChar ( wp, this->ruleChar, gridColor ) ;

         if ( fillChar == this->miniChar )
         {
            //* Position the zero-indicator character *
            switch ( this->rows )
            {
               case 1: case 2: tc = y ;         break ;
               case 3:         tc = 2 ;         break ;
               default: tc = this->rows / 2 ;   break ;
            }
            for ( ; y < this->rows ; ++y )
            {
               this->wPtr->WriteChar ( wp.ypos - y, wp.xpos, 
                                       (y == tc ? this->miniChar : nckSPACE),
                                       barColor ) ;
            }
         }
         else
         {
            for ( ; y < this->rows ; ++y )
               this->wPtr->WriteChar ( wp.ypos - y, wp.xpos, fillChar, barColor ) ;
         }
         wp.xpos += xinc ;

         //* Decrement the division index *
         if ( (fillChar == this->miniChar) || ((fillIndx -= cellDIVS) < ZERO) )
         {
            //* End of bar, fill remaining cells with spaces *
            fillIndx = -1 ;
            fillChar = nckSPACE ;
            if ( this->reverse )
               barColor = fbarColor ;
         }
      }
   }

   this->wPtr->RefreshWin () ;              // update the display

}  //* End RedrawControl()

//*************************
//*    SetOutputFormat    *
//*************************
//******************************************************************************
//* Set the text data output format: RTL (right-to-left), or                   *
//* LTR (left-to-right). Does not refresh display.                             *
//* Note: Because the Slider control contains no text, this method is just a   *
//*       placeholder which implements the virtual method in DialogControl     *
//*       class.                                                               *
//*                                                                            *
//* Input  : rtlFormat : if 'true',  then set as RTL format                    *
//*                      if 'false', then set as LTR format                    *
//*                                                                            *
//* Returns: OK  if successful                                                 *
//*          ERR if data format cannot be changed (currently not used)         *
//******************************************************************************

short DialogSlider::SetOutputFormat ( bool rtlFormat )
{

   this->rtlContent = rtlFormat ;
   return OK ;

}  //* End SetOutputFormat() *


      //*************************************************
      //**     THIS SECTION IMPLEMENTS THE PUBLIC,     **
      //**    NcDialog-class METHODS FOR ACCESSING     **
      //**    DialogSlider-class FUNCTIONALITY.        **
      //*************************************************

//*************************
//*      EditSlider       *
//*************************
//******************************************************************************
//* If control with input focus == dctSLIDER, call this method to get user's   *
//* key input. Allows user to adjust the magnitude of the slider setting.      *
//*                                                                            *
//* Returns when the control IS READY to lose the input focus (magnitude may or*
//* may not have changed) -OR- control has lost focus due to a hotkey press.   *
//*                                                                            *
//* User input keys:                                                           *
//* -------------------------------------------------------------------------- *
//*                                                                            *
//*                                                                            *
//*                                                                            *
//*                                                                            *
//*                                                                            *
//* Input  : uiInfo class (by reference) - initial values ignored              *
//*                                                                            *
//* Returns: index of control that currently has the input focus               *
//*           (See note in NcDialog.hpp about interpretation )                 *
//*           (of values returned in the uiInfo-class object.)                 *
//******************************************************************************

short NcDialog::EditSlider ( uiInfo& info )
{
   DialogSlider* cp = NULL ;        // pointer to control object
   short  divOrig,                  // slider value (display divisions) on entry
          divTemp ;                 // temporary div value
   bool   done = false ;            // loop control

   //* Be sure the control with focus is actually a DialogSlider *
   if ( this->dCtrl[this->currCtrl]->type == dctSLIDER )
   {
      cp = (DialogSlider*)this->dCtrl[this->currCtrl] ;
   }
   else            // control is not a slider control i.e. application error
   {
      //* Do what we can to minimize the damage *
      info.ctrlType = this->dCtrl[this->currCtrl]->type ;
      info.ctrlIndex = this->currCtrl ;
      info.keyIn = nckTAB ;               // move on to next control
      info.hasFocus = true ;              // control has focus
      info.dataMod = false ;              // no data modified
      info.selMember = MAX_DIALOG_CONTROLS ; // don't care
      info.isSel = false ;                // don't care
      info.viaHotkey = false ;            // invalidate any hotkey data
      done = true ;                       // don't enter the loop
   }

   divOrig = cp->divCurr ;                // remember the initial value

   //**************************
   //* Interact with the user *
   //**************************
   while ( ! done )
   {
      //* 1) Get user input.                                  *
      //* 2) Check for toggle of Insert/Overstrike mode.      *
      //* 3) Check for terminal-resize event.                 *
      //* 4) Access callback method, if specified.            *
      //* 5) If input has been handled, return to top of loop.*
      this->GetKeyInput ( info.wk ) ;

      if ( (info.wk.type == wktFUNKEY)
            && (info.wk.key == nckINSERT || info.wk.key == nckRESIZE) )
      {
         if ( info.wk.key == nckINSERT )
            this->ToggleIns () ;
         else
            this->TermResized () ;

         if ( ExternalControlUpdate != NULL )
            ExternalControlUpdate ( this->currCtrl, info.wk, false ) ;
         continue ;
      }

      //* ENTER key is transformed to TAB key. *
      if ( info.wk.type == wktFUNKEY && 
           (info.wk.key == nckENTER || info.wk.key == nckpENTER) )
         info.wk.key = nckTAB ;

      //* If a mouse scroll-wheel event was converted to a keycode, associated *
      //* with controls oriented horizontally, then convert the following      *
      //* keycodes: nckUP becomes nckRIGHT, and nckDOWN becomes nckLEFT.       *
      //* This implements horizontal scrolling using the scroll wheel.         *
      //* Mouse events which are converted to nckPGUP, nckPGDOWN, nckHOME and  *
      //* nckEND are processed as they were generated. See meTransformEvent(). *
      if ( info.wk.mevent.conv && !cp->vertical && (info.wk.type == wktFUNKEY) )
      {
         if ( info.wk.key == nckUP )
            info.wk.key = nckRIGHT ;
         else if ( info.wk.key == nckDOWN )
            info.wk.key = nckLEFT ;
      }

      // Programmer's Note: As a possible future enhancement, allow a 
      // single-click in a control with focus to signal a move of the slider 
      // bar to the character cell which received the click. Although the 
      // resolution is the character cell and NOT the step within the cell, 
      // it may be helpful.

      //* Moving to next control to the right/down *
      if ( (info.wk.type == wktFUNKEY) && 
           ((info.wk.key == nckTAB) || (info.wk.key == nckESC) ||
           (cp->vertical && (info.wk.key == nckRIGHT)) ||
           (!cp->vertical && (info.wk.key == nckDOWN))) )
      {
         info.ctrlType = dctSLIDER ;         // this is a text box control
         info.ctrlIndex = this->currCtrl ;   // control with input focus
         info.hasFocus = true ;              // focus was not lost
         info.selMember = MAX_DIALOG_CONTROLS ; // don't care
         info.isSel = false ;                // don't care
         info.viaHotkey = false ;            // control did not get focus via hotkey
         info.keyIn = nckTAB ;               // focus to move forward to next control
         //* If ESC, then discard any edits by restoring original value.*
         if ( info.wk.key == nckESC )
         {
            cp->divCurr = divOrig ;
            info.dataMod = false ;
         }
         //* Else, determine whether data have been modified.*
         else
         {
            info.dataMod = bool(cp->divCurr != divOrig) ;
         }
         done = true ;
      }

      //* Moving to previous control to the left/up *
      else if ( (info.wk.type == wktFUNKEY) && 
                ((info.wk.key == nckSTAB) || 
                (cp->vertical && (info.wk.key == nckLEFT)) ||
                (!cp->vertical && (info.wk.key == nckUP))) )
      {
         info.ctrlType = dctSLIDER ;         // this is a text box control
         info.ctrlIndex = this->currCtrl ;   // control with input focus
         info.hasFocus = true ;              // focus was not lost
         info.selMember = MAX_DIALOG_CONTROLS ; // don't care
         info.isSel = false ;                // don't care
         info.viaHotkey = false ;            // control did not get focus via hotkey
         info.keyIn = nckSTAB ;              // focus to move backward to previous control
         //* Determine whether data have been modified.*
         info.dataMod = bool(cp->divCurr != divOrig) ;
         done = true ;
      }

      //* Increase magnitude by one step *
      else if ( (info.wk.type == wktFUNKEY) && 
                ((!cp->reverse && (info.wk.key == nckUP))    ||   // grow upward
                 (!cp->reverse && (info.wk.key == nckRIGHT)) ||   // grow rightward
                 (cp->reverse && (info.wk.key == nckDOWN))   ||   // grow downward
                 (cp->reverse && (info.wk.key == nckLEFT))) )     // grow leftward
      {
         if ( cp->divCurr < cp->divTotal )
         {
            ++cp->divCurr ;
            cp->RedrawControl ( true ) ;
            //* If automatic report of value is enabled *
            if ( cp->autoval )   this->SliderAutoValueDisplay ( cp ) ;
         }
         else if ( cp->audibleAlert )
            this->UserAlert () ;
      }

      //* Increase magnitude by ten percent *
      else if ( (info.wk.type == wktFUNKEY) && 
                ((!cp->reverse && (info.wk.key == nckPGUP)) ||
                 (cp->reverse  && (info.wk.key == nckPGDOWN))) )
      {
         if ( cp->divCurr < cp->divTotal )
         {
            divTemp = (cp->divTotal / 10) ;
            if ( divTemp == ZERO )
               divTemp = 1 ;
            divTemp += cp->divCurr ;
            if ( divTemp <= cp->divTotal )
               cp->divCurr = divTemp ;
            else
               cp->divCurr = cp->divTotal ;
            cp->RedrawControl ( true ) ;
            //* If automatic report of value is enabled *
            if ( cp->autoval )   this->SliderAutoValueDisplay ( cp ) ;
         }
         else if ( cp->audibleAlert )
            this->UserAlert () ;
      }

      //* Increase magnitude to maximum *
      else if ( (info.wk.type == wktFUNKEY) &&
                ((!cp->reverse && (info.wk.key == nckHOME)) ||
                 (cp->reverse  && (info.wk.key == nckEND))) )
      {
         if ( cp->divCurr < cp->divTotal )
         {
            cp->divCurr = cp->divTotal ;
            cp->RedrawControl ( true ) ;
            //* If automatic report of value is enabled *
            if ( cp->autoval )   this->SliderAutoValueDisplay ( cp ) ;
         }
         else if ( cp->audibleAlert )
            this->UserAlert () ;
      }

      //* Decrease magnitude by one step *
      else if ( (info.wk.type == wktFUNKEY) && 
                ((!cp->reverse && (info.wk.key == nckDOWN))  ||   // shrink downward
                 (!cp->reverse && (info.wk.key == nckLEFT))  ||   // shrink leftward
                 (cp->reverse && (info.wk.key == nckUP))     ||   // shrink upward
                 (cp->reverse && (info.wk.key == nckRIGHT))) )    // shrink rightward
      {
         if ( cp->divCurr > ZERO )
         {
            --cp->divCurr ;
            cp->RedrawControl ( true ) ;
            //* If automatic report of value is enabled *
            if ( cp->autoval )   this->SliderAutoValueDisplay ( cp ) ;
         }
         else if ( cp->audibleAlert )
            this->UserAlert () ;
      }

      //* Decrease magnitude by ten percent *
      else if ( (info.wk.type == wktFUNKEY) && 
                ((!cp->reverse && (info.wk.key == nckPGDOWN)) ||
                 (cp->reverse  && (info.wk.key == nckPGUP))) )
      {
         if ( cp->divCurr > ZERO )
         {
            divTemp = (cp->divTotal / 10) ;
            if ( divTemp == ZERO )
               divTemp = 1 ;
            divTemp = cp->divCurr - divTemp ;
            if ( divTemp >= ZERO )
               cp->divCurr = divTemp ;
            else
               cp->divCurr = ZERO ;
            cp->RedrawControl ( true ) ;
            //* If automatic report of value is enabled *
            if ( cp->autoval )   this->SliderAutoValueDisplay ( cp ) ;
         }
         else if ( cp->audibleAlert )
            this->UserAlert () ;
      }

      //* Decrease magnitude to minimum *
      else if ( (info.wk.type == wktFUNKEY) &&
                ((!cp->reverse && (info.wk.key == nckEND)) ||
                 (cp->reverse  && (info.wk.key == nckHOME))) )
      {
         if ( cp->divCurr > ZERO )
         {
            cp->divCurr = ZERO ;
            cp->RedrawControl ( true ) ;
            //* If automatic report of value is enabled *
            if ( cp->autoval )   this->SliderAutoValueDisplay ( cp ) ;
         }
         else if ( cp->audibleAlert )
            this->UserAlert () ;
      }

      //* Check for hotkey (CTRL+['A' ... 'Z']) OR *
      //* 'default' hotkey if mouse enabled        *
      else if ( (info.wk.type == wktFUNKEY && 
                 (info.wk.key >= nckC_A && info.wk.key <= nckC_Z)) ||
                ((this->meMouseEnabled ()) && 
                 (info.wk.mevent.conv != false)) )
      {
         //* Scan the controls for one whose hotkey matches the input.*
         //* If match found, returns control's index, else -1.        *
         //* Ignore hotkey selection of this text box.                *
         short hotIndex = this->IsHotkey ( info.wk ) ;
         if ( hotIndex >= ZERO && hotIndex != this->currCtrl )
         {  //* Make the indicated control the current/active control *
            //* and indicate the results of our edit.                 *
            info.ctrlType = dctSLIDER ;      // this is a slider control
            info.ctrlIndex = this->currCtrl ;// control with input focus
            info.hasFocus = true ;           // this may be reset by ChangedFocusViaHotkey()
            info.keyIn = ZERO ;              // indicate no shift in focus necessary
            info.selMember = MAX_DIALOG_CONTROLS ; // don't care
            info.isSel = false ;             // don't care
            info.viaHotkey = false ;         // this may be set by ChangedFocusViaHotkey()
            //* Determine whether data have been modified.*
            info.dataMod = bool(cp->divCurr != divOrig) ;
            done = true ;                    // exit the input loop
            this->ChangedFocusViaHotkey ( info.wk.key, hotIndex, info ) ;
            // on return, new control has the focus, and the info.h_XX
            // variables have been adjusted for that control
         }
         else if ( hotIndex < ZERO )
         {
            //* Key input value is not valid in this context, ignore it. *
            //* If audible alert enabled for invalid key data.           *
            if ( cp->audibleAlert && (info.wk.type != wktMOUSE) )
               this->UserAlert () ;
         }
      }

      //* Key input value is not valid in this context, ignore it. *
      else
      {
      }

      //* If caller has specified a callback method, do it now.*
      if ( ExternalControlUpdate != NULL )
         ExternalControlUpdate ( this->currCtrl, info.wk, false ) ;
   }     // while()

   return this->currCtrl;

}  //* End EditSlider() *

//*************************
//*    SetSliderConfig    *
//*************************
//******************************************************************************
//* Specify the minimum, maximum and current values as well as the control     *
//* orientation (vertical vs. horizontal) and the growth direction             *
//* (right/up vs. left/down). This method redraws the control.                 *
//*                                                                            *
//* See also 'GetSliderConfig()' method which returns the current setup.       *
//*                                                                            *
//* Input  : cIndex: index of target dctSLIDER control                         *
//*          slData: (by reference) a partially-initialized instance of the    *
//*                  sliderData class with the following members initialized:  *
//*                  minValue  : minimum value represented                     *
//*                  maxValue  : maximum value represented                     *
//*                  curValue  : initial value setting                         *
//*                  vertical  : orientation                                   *
//*                              'true' : grows/shrinks vertically             *
//*                              'false': grows/shrinks horizontally           *
//*                  reverse   : direction                                     *
//*                              'false': if 'vertical', grows upward          *
//*                                       if 'horizontal, grows rightward      *
//*                              'true' : if 'vertical', grown downward        *
//*                                       if 'horizontal', grows leftward      *
//*                  ---------                                                 *
//*                  Initial values of other members is are ignored.           *
//*                  On return, the remaining fields will be initialized.      *
//*                                                                            *
//* Returns: OK if successful                                                  *
//*          ERR if invalid input (default values used),                       *
//*              OR specified control is not of type dctSLIDER                 *
//******************************************************************************

short NcDialog::SetSliderConfig ( short cIndex, sliderData& slData )
{
   short result = ERR ;

   if ( cIndex >= ZERO && cIndex <= this->lastCtrl
        && this->dCtrl[cIndex]->type == dctSLIDER )
   {
      //* Pointer to the control's methods and data *
      DialogSlider* cp = (DialogSlider*)this->dCtrl[cIndex] ;
      if ( (cp->Configure ( slData )) )
         result = OK ;
      //* Redraw the control with the new value *
      cp->RedrawControl ( (bool)(cIndex==this->currCtrl ? true : false) ) ;
   }
   return result ;

}  //* End SetSliderConfig() *

//*************************
//*    GetSliderConfig    *
//*************************
//******************************************************************************
//* Returns the current configuration and values of the Slider control.        *
//* See also 'SetSliderConfig()' method which sets the initital configuration. *
//*                                                                            *
//* Input  : cIndex: index of target dctSLIDER control                         *
//*          slData: (by reference, initial values ignored)                    *
//*                  receives the current setup information for the control    *
//*                                                                            *
//* Returns: OK if successful                                                  *
//*          ERR if specified control is not of type dctSLIDER                 *
//******************************************************************************

short NcDialog::GetSliderConfig ( short cIndex, sliderData& slData )
{
   short result = ERR ;

   if ( cIndex >= ZERO && cIndex <= this->lastCtrl
        && this->dCtrl[cIndex]->type == dctSLIDER )
   {
      //* Pointer to the control's methods and data *
      DialogSlider* cp = (DialogSlider*)this->dCtrl[cIndex] ;
      cp->Report ( slData ) ;
      result = OK ;
   }
   return result ;

}  //* End GetSliderConfig() *

//************************
//*    SetSliderValue    *
//************************
//******************************************************************************
//* Set a new display value for the specified dctSLIDER control.               *
//* The value may be specified either as a value within the data range,        *
//*             minimum value <= new value <= maximum value                    *
//* OR as a percentage of the data range:  0.0 <= percentage <= 100.0          *
//*                                                                            *
//* Note: The actual value set may be slightly different from the specified    *
//*       value due to rounding errors during conversion of the floating-point *
//*       value to an integer step size.                                       *
//*                                                                            *
//* Input  : cIndex : index into list of controls defined for this dialog      *
//*          slValue: new display value either as a value or as a percentage   *
//*                   depending on the state of the 'percent' flag.            *
//*          percent: if 'false', interpret 'slValue' as a decimal value       *
//*                   if 'true',  interpret 'slValue' as a percentage of the   *
//*                               value range                                  *
//*                                                                            *
//* Returns: OK if successful                                                  *
//*          OR ERR if slValue is out-of-range                                 *
//*                 if specified control currently has the input focus         *
//*                 if specified control is not of type dctSLIDER              *
//******************************************************************************

short NcDialog::SetSliderValue ( short cIndex, double slValue, bool percent )
{
   short status = ERR ;
   if ( cIndex >= ZERO && cIndex <= this->lastCtrl && 
        (this->dCtrl[cIndex])->type == dctSLIDER && cIndex != this->currCtrl )
   {
      DialogSlider* cp = (DialogSlider*)(dCtrl[cIndex]) ;
      if ( (cp->SetCurrValue ( slValue, percent )) )
      {
         cp->RedrawControl ( false ) ;

         //* If auto-value-display is active, update display *
         if ( cp->autoval )
            this->SliderAutoValueDisplay ( cp ) ;

         status = OK ;
      }
   }
   return status ;

}  //* End SetSliderValue() *

//************************
//*    GetSliderValue    *
//************************
//******************************************************************************
//* Return the currently-displayed value for the specified dctSLIDER control.  *
//* The value returned is a decimal (double) value which specifies either the  *
//* the actual current value of the control OR the current value expressed as  *
//* a _percentage_ of the established value range:  0.0 <= percentage <= 100.0 *
//*                                                                            *
//* Input  : cIndex : index into list of controls defined for this dialog      *
//*          slValue: (by reference, initial value ignored)                    *
//*                   receives currently-displayed value as a value or as a    *
//*                   percentage depending on the state of the 'percent' flag. *
//*          percent: if 'false', 'slValue' receives the actual decimal value  *
//*                   if 'true',  'slValue' receives the decimal value         *
//*                               expressed as a percentage of the value range *
//*                                                                            *
//* Returns: OK if successful                                                  *
//*          OR ERR if specified control is not of type dctSLIDER              *
//******************************************************************************

short NcDialog::GetSliderValue ( short cIndex, double& slValue, bool percent )
{
   double val, pct ;
   short status = ERR ;
   if ( cIndex >= ZERO && cIndex <= lastCtrl && dCtrl[cIndex]->type == dctSLIDER )
   {
      DialogSlider* cp = (DialogSlider*)(dCtrl[cIndex]) ;
      cp->ReportValue ( val, pct ) ;
      slValue = (percent ? pct : val) ;
      status = OK ;
   }
   return status ;

}  //* End GetSliderValue() *

//*************************
//*      SliderAlert      *
//*************************
//******************************************************************************
//* Enable or disable audible alert (beep) when user attempts to increase or   *
//* decrease the Slider data beyond the established limit during edit of the   *
//* Slider control.                                                            *
//*                                                                            *
//* Audible alert is initially disabled, and may be enabled after instantiation*
//* of the control. The alert mechanism is a call to the UserAlert() method    *
//* (with no parameters) as described elsewhere.                               *
//*                                                                            *
//* Input  : cIndex : index of target dctSLIDER control                        *
//*          enable : 'true'  to enable audible alerts                         *
//*                   'false' to disable audible alerts                        *
//*                                                                            *
//* Returns: OK if successful                                                  *
//*          ERR if specified control is not of type dctSLIDER                 *
//******************************************************************************

short NcDialog::SliderAlert ( short cIndex, bool enable )
{
   short status = ERR ;

   //* If caller's index number references a Slider control.*
   if ( cIndex >= ZERO && cIndex <= this->lastCtrl && 
        (this->dCtrl[cIndex])->type == dctSLIDER )
   {
      //* Get a pointer to the control's private methods and data *
      DialogSlider* cp = (DialogSlider*)(dCtrl[cIndex]) ;
      cp->audibleAlert = enable ;
      status = OK ;
   }
   return status ;

}  //* End SliderAlert() *

//*************************
//*   SliderAutoReport    *
//*************************
//******************************************************************************
//* Enable or disable automatic reporting of the current Slider value.         *
//* Each time the user adjusts the value of the Slider control, the value      *
//* display is updated. This method formats and writes the current value to    *
//* the dialog window, but does not refresh the display.                       *
//*                                                                            *
//* If the position specified by 'yOffset' and 'xOffset' does not allow the    *
//* value to be fully displayed, the offsets will be silently adjusted as      *
//* necessary.                                                                 *
//*                                                                            *
//* Notes on format specification ('fmtSpec'):                                 *
//*  a) This is a standard format string as used by printf, sprintf, etc.      *
//*     The value is a double-precision floating-point value, so the format    *
//*     specification must reflect the source data.                            *
//*     Format string length: avfmtBYTES bytes max.                            *
//*  b) The format specification should produce a fixed-width output string to *
//*     avoid display artifacts.                                               *
//*  c) The specified position should be free of all other data to avoid       *
//*     display conflicts.                                                     *
//*  d) The format specification may contain optional text indicating units    *
//*     or other information.                                                  *
//*      Examples:  "%6.1lf °C"                                                *
//*                 "%4.0lf"                                                   *
//*                 "$%5.2lf per dozen"                                        *
//*  e) Note that this method does not support RTL output, so if the format    *
//*     specification includes Right-To-Left text in addition to the numeric   *
//*     data, this must be handled at the application level.                   *
//*       Example:  "nopuoc htiw ffo tnecrep %3.0lf teG"                       *
//*  f) When calculating the width of the area to be cleared, we count the     *
//*     columns ONLY on the first line of the output. AND we clear only the    *
//*     first line of the target area.                                         *
//*                                                                            *
//*                                                                            *
//* Input  : cIndex : index of target dctSLIDER control                        *
//*          enable : 'true'  to enable automatic value display                *
//*                   'false' to disable automatic value display               *
//*          yOffset: Y offset from dialog origin for value display            *
//*          xOffset: X offset from dialog origin for value display            *
//*          fmtSpec: format specification for display of value, either value  *
//*                   or percentage with optional units (see note above)       *
//*          txtAttr: color attribute for display text                         *
//*          reportPct: if 'true',  report the current percentage of range     *
//*                     if 'false', report the current value                   *
//*                                                                            *
//* Returns: OK if successful                                                  *
//*          ERR if specified control is not of type dctSLIDER                 *
//******************************************************************************

short NcDialog::SliderAutoReport ( short cIndex, bool enable, 
                                   short yOffset, short xOffset, const char* fmtSpec, 
                                   attr_t txtAttr, bool reportPct )
{
   short status = ERR ;

   //* If caller's index number references a Slider control.*
   if ( cIndex >= ZERO && cIndex <= this->lastCtrl && 
        (this->dCtrl[cIndex])->type == dctSLIDER )
   {
      //* Get a pointer to the control's private methods and data *
      DialogSlider* cp = (DialogSlider*)(dCtrl[cIndex]) ;

      if ( (cp->autoval = enable) != false )    // save the enable/disable flag
      {
         gString gs( fmtSpec ) ;                // save the format specification
         gs.copy( cp->autovalFmt, avfmtBYTES ) ;
         cp->autovalAttr = txtAttr ;            // save the display color attribute
         cp->autovalPct = reportPct ;           // save the value/percentage flag

         //* Calculate the maximum output width *
         short minvalWidth, maxvalWidth, nlIndx ;
         if ( reportPct )        // if formatted percentage
         {
            double maxpct = 100.0 ;
            gs.compose( cp->autovalFmt, &maxpct ) ;
            if ( (nlIndx = gs.find( L'\n' )) >= ZERO )
               gs.limitCols( nlIndx ) ;
            maxvalWidth = minvalWidth = gs.gscols() ;
         }
         else                    // if formatted value
         {
            gs.compose( cp->autovalFmt, &cp->minVal ) ;
            if ( (nlIndx = gs.find( L'\n' )) >= ZERO )
               gs.limitCols( nlIndx ) ;
            minvalWidth = gs.gscols() ;
            gs.compose( cp->autovalFmt, &cp->maxVal ) ;
            if ( (nlIndx = gs.find( L'\n' )) >= ZERO )
               gs.limitCols( nlIndx ) ;
            maxvalWidth = gs.gscols() ;
         }
         cp->autovalMaxw = (maxvalWidth >= minvalWidth ? maxvalWidth : minvalWidth) ;

         //* Validate the position and silently adjust as necessary *
         if ( yOffset >= this->dLines )      yOffset = this->dLines - 1 ;
         else if ( yOffset < ZERO )          yOffset = ZERO ;
         if ( xOffset >= (this->dCols - cp->autovalMaxw) )
            xOffset = this->dCols - cp->autovalMaxw - 1 ;
         if ( xOffset < ZERO )               xOffset = ZERO ;
         cp->autovalPos = { yOffset, xOffset } ;// save the display position

         //* Write the formatted value to the dialog window *
         this->SliderAutoValueDisplay ( cp, false ) ;

         status = true ;
      }
   }
   return status ;

}  //* End SliderAutoReport() *

//**************************
//* SliderAutoValueDisplay *
//**************************
//******************************************************************************
//* Private Method:                                                            *
//* ---------------                                                            *
//* Display the current Slider value using the configuration data specified    *
//* by a previous call to SliderAutoReport().                                  *
//* NOTE: Application is cautioned to use a formatting template which of a     *
//*       fixed width; however, we clear the target area before update just    *
//*       to be safe.                                                          *
//* NOTE: This method writes the data as LTR text.                             *
//*       See note in SliderAutoReport() about support for RTL text.           *
//*                                                                            *
//* Input  : cp      : pointer to a DialogSlider object                        *
//*          refresh : (optional, 'true' by default)                           *
//*                    if 'true',  refresh the display after write             *
//*                    if 'false', return without refreshing display           *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void NcDialog::SliderAutoValueDisplay ( DialogSlider *cp, bool refresh )
{
   gString gs ;
   double cur, pct ;

   //* Retrieve the current value (value AND percentage) *
   cp->ReportValue ( cur, pct ) ;

   //* Clear the display area *
   gs.padCols( avfmtBYTES ) ;
   gs.limitCols ( cp->autovalMaxw ) ;
   this->WriteString ( cp->autovalPos, gs, cp->autovalAttr ) ;

   //* Format and display the data *
   if ( cp->autovalPct )
      gs.compose( cp->autovalFmt, &pct ) ;
   else
      gs.compose( cp->autovalFmt, &cur ) ;
   this->WriteParagraph ( cp->autovalPos, gs, cp->autovalAttr ) ;
   if ( refresh )                // inf specified, refresh the display
      this->RefreshWin () ;

}  //* End AutoValueDisplay() *


//*** ---  ---  ---  ---  ---  ---  ---  ---  ---  ---  ---  ---  ---  ---  ***
//*** ---  ---  ---  ---  Non-member Methods  ---  ---  ---  ---  ---  ---  ***
//*** ---  ---  ---  ---  ---  ---  ---  ---  ---  ---  ---  ---  ---  ---  ***

//*************************
//* InvertFractionalCell  *
//*************************
//******************************************************************************
//* Non-member method:                                                         *
//* ------------------                                                         *
//* Invert the index into the block-character arrays, hBlock[] and vBlock[].   *
//* This allows a visual inversion of the filled vs. empty pixes withing the   *
//* character cell.                                                            *
//*                                                                            *
//* When used in conjunction with the mirrored color attribute, it allows for  *
//* fractionally-filled cells used to terminate a bar of the graph.            *
//*                                                                            *
//* Input  : blkIndx : index into character array referencing character to     *
//*                    be inverted. NOTE: blkIndx should be strictly controlled*
//*                    in the range of 0-6 (inclusive). The arrays are exactly *
//*                    eight(8) elements AND offset 7 is the full-cell block   *
//*                    character.                                              *
//*                                                                            *
//* Returns: index of block character which mirrors the character refrenced    *
//*          by blkIndx                                                        *
//******************************************************************************

static short InvertFractionalCell ( short blkIndx )
{
   short invIndx ;

   switch ( blkIndx )
   {
      case 0: invIndx = 6 ; break ;
      case 1: invIndx = 5 ; break ;
      case 2: invIndx = 4 ; break ;
      case 3: invIndx = 3 ; break ;
      case 4: invIndx = 2 ; break ;
      case 5: invIndx = 1 ; break ;
      case 6: invIndx = 0 ; break ;
      default: invIndx = blkIndx ; break ;   // should never happen
   } ;

   return invIndx ;
}  //* End InvertFractionalCell() *

//*************************
//* InvertColorAttribute  *
//*************************
//******************************************************************************
//* Non-member method:                                                         *
//* ------------------                                                         *
//* For the specified color attribute return a color attribute with the        *
//* foreground and background colors swapped.                                  *
//*                                                                            *
//* When used in conjunction with the mirrored block character, it allows for  *
//* fractionally-filled cells used to terminate a bar of the graph.            *
//*                                                                            *
//* IMPORTANT NOTE: This methods uses intimate knowledge of the way color      *
//* attributes are encoded within the ncurses world, specifically as used by   *
//* the NcDialog API. All inversions have been manually tested and visually    *
//* verified. In the unlikely event that the ncurses library changes the way   *
//* it handles color attributes, this method may break.                        *
//*                                                                            *
//* Bit Definitions for a 32-bit character/attribute value (7-bit ASCII).      *
//* This is known as 'attr_t'. (see curses.h)                                  *
//* Note that 'wide' characters are combined with their color attribute        *
//* internally by the ncurses library; however, the color attribute bits       *
//* are the same for both narrow and wide characters.                          *
//*  BIT  31    = unused    -----------+                                       *
//*  BIT  30    = vertical             |                                       *
//*  BIT  29    = top                  |                                       *
//*  BIT  28    = right                +-- we make no use of these             *
//*  BIT  27    = low                  |   definitions in our implementation   *
//*  BIT  26    = left                 |                                       *
//*  BIT  25    = horizontal           |                                       *
//*  BIT  24    = "protected" mode ----+                                       *
//*  BIT  23    = invisible (background color is also used for foreground)     *
//*  BIT  22    = alternate character set (with line draw characters, etc.)    *
//*  BIT  21    = bold foreground against default background                   *
//*  BIT  20    = dim                                                          *
//*  BIT  19    = blinking (not functional under most terminal implementations)*
//*  BIT  18    = reverse (foreground/background swap)                         *
//*  BIT  17    = underline                                                    *
//*  BIT  16    = standout (foreground-to-background, with fg & bg bolded)     *
//*  BITS 15-8  = color pair number  (8, 16, 64, 256.  see init_pair() )       *
//*  BITS 7-0   = character code                                               *
//*                                                                            *
//* Refer to ISO 6429 for a description of color support for terminals.        *
//*                                                                            *
//* In this method, we simply toggle the 'Reverse' bit.                        *
//* In some cases we may swap the 8-bit color-pair sequence, and/or toggle     *
//* the 'Bold' bit.                                                            *
//* IMPORTANT NOTE: Inverting BOLD colors will likely fail due to inconsistent *
//*                 combinations of BOLD and REVERSE.                          *
//*                                                                            *
//* Input  : normColor : non-inverted color attribute                          *
//*                                                                            *
//* Returns: color attribute with foreground and background swapped.           *
//******************************************************************************

static attr_t InvertColorAttribute ( attr_t normColor )
{
   const attr_t mask = ~ncrATTR ;
   attr_t invColor ;

   if ( normColor & ncrATTR )
   {
      invColor = normColor & mask ;
   }
   else
   {
      invColor = normColor | ncrATTR ;
   }
   return invColor ;

}  //* End InvertColorAttribute() *

//*************************
//*                       *
//*************************
//********************************************************************************
//*                                                                              *
//*                                                                              *
//*                                                                              *
//* Input  :                                                                     *
//*                                                                              *
//* Returns:                                                                     *
//********************************************************************************

