//********************************************************************************
//* File       : NcdControlSE.cpp                                                *
//* Author     : Mahlon R. Smith                                                 *
//*              Copyright (c) 2006-2025 Mahlon R. Smith, The Software Samurai   *
//*                  GNU GPL copyright notice located in NcDialog.hpp            *
//* Date       : 12-Feb-2021                                                     *
//* Version    : (see NcDialogVersion string in NcDialog.cpp)                    *
//*                                                                              *
//* Description: Contains the methods of the DialogScrollext class.              *
//*              and the NcDialog class EditScrollext() method.                  *
//*              See NcDialog.hpp for the definition of this class.              *
//*                                                                              *
//*                                                                              *
//* Development Tools: See NcDialog.cpp.                                         *
//********************************************************************************
//* Version History (most recent first):                                         *
//*   See version history in NcDialog.cpp.                                       *
//********************************************************************************

//*****************
//* Include Files *
//*****************

#include "GlobalDef.hpp"               //* General definitions

#ifndef NCURSES_INCLUDED
#include "NCurses.hpp"
#endif

#ifndef NCDIALOG_INCLUDED
#include "NcDialog.hpp"
#endif


//*****************
//*   Local Data  *
//*****************

//* During instantiation of the control object,   *
//* the pointers need some valid data to point to.*
static const short   defMsgLength = 24 ;
static const char    defaultMsg[defMsgLength+1] = "display data unitialized" ;
static char          defaultString[MAX_DIALOG_WIDTH+1] ;
static const char*   defaultData = defaultString ;
static attr_t        defaultAttribute = nc.bw ;
const static attr_t* defaultColor = &defaultAttribute ;


      //*************************************************
      //** THIS SECTION IMPLEMENTS THE METHODS OF THE  **
      //**           DialogScrollext CLASS.            **
      //** - - - - - - - - - - - - - - - - - - - - - - **
      //** See below for public NcDialog-class methods **
      //**  related to DialogScrollext-class access.   **
      //*************************************************

//*************************
//*    DialogScrollext    *
//*************************
//******************************************************************************
//* Constructor for DialogScrollext class object.                              *
//*                                                                            *
//*                                                                            *
//* Input  : pointer to initialization structure                               *
//*                                                                            *
//* Returns: implicitly returns a pointer to the control object                *
//******************************************************************************

DialogScrollext::DialogScrollext ( InitCtrl* iPtr )
{
   this->type   = dctSCROLLEXT ;    // set control type
   this->ulY    = iPtr->ulY ;       // transfer the initialization data
   this->ulX    = iPtr->ulX ;
   this->lines  = iPtr->lines ;
   this->cols   = iPtr->cols ;
   this->nColor = iPtr->nColor ;    // color of border when control does not have focus
   this->fColor = iPtr->fColor ;    // color of border when control has focus
   this->labY   = iPtr->labY ;      // label offsets
   this->labX   = iPtr->labX ;
   this->active = iPtr->active ;    // indicates whether control can be selected

   //* Copy the label text to data member, wLabel and initialize bHotkey.      *
   //* Note that if a multi-line label was specified for a label embedded      *
   //* within the control's border, we strip the newlines from the string as   *
   //* an indication to the user that he/she/it screwed up.                    *
   this->InitHotkey ( iPtr->label, bool(this->labY == ZERO && this->labX == ZERO) ) ;

   //* Default values for all other data members *
   this->bIndex = ZERO ;      // index for initial position of highlight
   this->bItems = 1 ;         // number of display strings, initially 1
   this->hlShow = true ;      // highlight the current display item
   this->groupCode = ZERO ;   // scroll-ext controls are not grouped (not used)
   this->rtlContent = false ; // left-to-right language content by default
         //* NOTE: cdConnect class bConnect values are all initialized to 'false'

   //* Create a default display item.                            *
   //* (in case application doesn't initialize the display data) *
   short i ;
   for ( i = ZERO ; i < (this->cols-2) && i < defMsgLength && i < (MAX_DIALOG_WIDTH) ; i++ )
      defaultString[i] = defaultMsg[i] ;
   for ( ; i < (this->cols-2) && i < (MAX_DIALOG_WIDTH) ; i++ )
      defaultString[i] = SPACE ;
   defaultString[i] = NULLCHAR ;      // terminate the string

   //* Initialize the text data and color data pointers *
   this->bText  = &defaultData ;
   this->bColor = defaultColor ;

   //* Instantiate the underlying window object *
   this->bPtr = new NcWindow ( lines, cols, ulY, ulX ) ;    // outer window contains border
   this->wPtr = new NcWindow ( lines-2, cols-2, ulY+1, ulX+1 ) ; // inner window contains data

}  //* End DialogScrollext() *

//*************************
//*   ~DialogScrollext    *
//*************************
//******************************************************************************
//* Destructor for DialogScrollext class object.                               *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

DialogScrollext::~DialogScrollext ( void )
{

   //* Close the underlying windows *
   delete this->bPtr ;
   delete this->wPtr ;

}  //* End ~DialogScrollext() *

//*************************
//*      OpenControl      *
//*************************
//******************************************************************************
//* Draw a dialog scroll-ext control i.e. open the window previously           *
//* instantiated by a call to DialogScrollext().                               *
//*                                                                            *
//* Input  : hasFocus : optional boolean, false by default, determines         *
//*                     whether to draw the object in nColor or fColor         *
//*                                                                            *
//* Returns: OK if successful, else ERR                                        *
//******************************************************************************

short DialogScrollext::OpenControl ( bool hasFocus )
{
attr_t   color = hasFocus ? fColor : nColor ;
short    result = ERR ;

   this->wPtr->SetInteriorColor ( color ) ;
   if (   (this->bPtr->OpenWindow () == OK) 
       && (result = this->wPtr->OpenWindow ()) == OK )
   {
      this->RedrawControl ( hasFocus, true, true ) ;
   }
   return result ;

}  //* End OpenControl() *

//*************************
//*    RedrawControl      *
//*************************
//******************************************************************************
//* Redraw the data in a Scrollext window.                                     *
//* (refreshes the control's window using existing data)                       *
//*                                                                            *
//* Input  : hasFocus: if true, control has focus                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void DialogScrollext::RedrawControl ( bool hasFocus )
{

   this->RedrawControl ( hasFocus, false ) ;

}  //* End RedrawControl()

//*************************
//*     RedrawControl     *
//*************************
//******************************************************************************
//* Redraw the data in a Scrollext window, either new or existing data.        *
//*                                                                            *
//* NOTE: Called directly only by members of the DialogScrollext class.        *
//*                                                                            *
//* Input  : hasFocus: if true, control has focus                              *
//*          newData : if true, paint the new data to the display              *
//*                    if false, repaint existing data                         *
//*          refresh : (optional, 'true' by default) refresh display           *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void DialogScrollext::RedrawControl ( bool hasFocus, bool newData, bool refresh )
{
   attr_t borderColor = hasFocus ? fColor : nColor ;

   //* Redraw and refresh the border (either plain border or border with title)*
   if ( this->labY != ZERO || this->labX != ZERO || *this->wLabel == NULLCHAR )
      this->bPtr->DrawBorder ( borderColor ) ;
   else
   {
      gString gs( this->wLabel ) ;  // convert from wchar_t to UTF-8
      short xOffset = this->bPtr->DrawBorder ( borderColor, gs.ustr(), 
                                               ncltSINGLE, this->rtlContent ) ;
      if ( this->bHotkey.hotkey != false )
      {  //* Write the hotkey character with underline *
         short xpos = this->labX + (this->rtlContent ?
                      (xOffset - this->bHotkey.xoffset) :   // RTL
                      (this->bHotkey.xoffset + xOffset)) ;  // LTR
         
         this->bPtr->WriteChar ( this->labY, xpos, 
                                 this->bHotkey.hotchar, borderColor | ncuATTR ) ;
      }
   }

   if ( this->bConnect.connection != false )
   {
      wchar_t  oChar ;                    // single-character output
      oChar = wcsLTEE ;
      if ( this->bConnect.ul2Left != false )
         this->bPtr->WriteChar ( ZERO, ZERO, oChar, borderColor ) ;
      if ( this->bConnect.ll2Left != false )
         this->bPtr->WriteChar ( this->lines-1, ZERO, oChar, borderColor ) ;
      oChar = wcsRTEE ;
      if ( this->bConnect.ur2Right != false )
         this->bPtr->WriteChar ( ZERO, this->cols-1, oChar, borderColor ) ;
      if ( this->bConnect.lr2Right != false )
         this->bPtr->WriteChar ( this->lines-1, this->cols-1, oChar, borderColor ) ;
      oChar = wcsTTEE ;
      if ( this->bConnect.ul2Top != false )
         this->bPtr->WriteChar ( ZERO, ZERO, oChar, borderColor ) ;
      if ( this->bConnect.ur2Top != false )
         this->bPtr->WriteChar ( ZERO, this->cols-1, oChar, borderColor ) ;
      oChar = wcsBTEE ;
      if ( this->bConnect.ll2Bot != false )
         this->bPtr->WriteChar ( this->lines-1, ZERO, oChar, borderColor ) ;
      if ( this->bConnect.lr2Bot != false )
         this->bPtr->WriteChar ( this->lines-1, this->cols-1, oChar, borderColor ) ;
   }
   this->bPtr->RefreshWin () ;
   if ( newData != false )
      this->wPtr->PaintData ( (const char**)this->bText, this->bColor, this->bItems, 
                              this->bIndex, this->hlShow, this->rtlContent ) ;
   else
      this->wPtr->RepaintData ( this->hlShow ) ;
   if ( refresh != false )
      this->wPtr->RefreshWin () ;
   
}  //* End RedrawControl() *

//*************************
//*    RefreshControl     *
//*************************
//******************************************************************************
//* Refresh the display of the specified dialog control object.                *
//*                                                                            *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void DialogScrollext::RefreshControl ( void )
{

   this->bPtr->RefreshWin () ;            // Refresh the outer (border) window
   this->wPtr->RefreshWin () ;            // refresh the inner (data) window

}  //* End RefreshControl() *

//*************************
//*    SetOutputFormat    *
//*************************
//******************************************************************************
//* Set the text data output format: RTL (right-to-left), or                   *
//* LTR (left-to-right). Does not refresh display.                             *
//* Note that the default (unmodified) output format is LTR.                   *
//*                                                                            *
//* Input  : rtlFormat : if 'true',  then set as RTL format                    *
//*                      if 'false', then set as LTR format                    *
//*                                                                            *
//* Returns: OK  if successful                                                 *
//*          ERR if data format cannot be changed (not currently used)         *
//******************************************************************************
//* Programmer's Note: When redrawing the control's contents, we assume that   *
//* control does not have focus. This could be incorrect, but since we don't   *
//* update the display, the user will never know it.                           *
//******************************************************************************

short DialogScrollext::SetOutputFormat ( bool rtlFormat )
{

   this->rtlContent = rtlFormat ;
   this->RedrawControl ( false, true, false ) ; // force data update
   return OK ;

}  //* End SetOutputFormat() *



      //*************************************************
      //**     THIS SECTION IMPLEMENTS THE PUBLIC,     **
      //**    NcDialog-class METHODS FOR ACCESSING     **
      //**    DialogScrollext-class FUNCTIONALITY.     **
      //*************************************************

//*************************
//*     EditScrollext     *
//*************************
//******************************************************************************
//* If control with input focus == dctSCROLLEXT, call this method to get user's*
//* key input. Allows user to edit which item in the control is 'selected'.    *
//*                                                                            *
//* Returns to caller when user has finished selection (selection may or may   *
//* not have changed), -OR- when control has lost focus due to a hotkey press. *
//*                                                                            *
//* Input  : uiInfo class (by reference) - initial values ignored              *
//*                                                                            *
//* Returns: index of control that currently has the input focus               *
//*           (See note in NcDialog.hpp about interpretation )                 *
//*           (of values returned in the uiInfo-class object.)                 *
//*           (highlight IS visible on return)                                 *
//******************************************************************************

short NcDialog::EditScrollext ( uiInfo& info )
{

   return ( (this->EditScrollext ( info, true )) ) ;

}  //* End EditScrollext() *

//************************
//*    ViewScrollext     *
//************************
//******************************************************************************
//* This method is similar to the EditScrollext() method above, except that the*
//* highlight is not displayed for the current data item and the user cannot   *
//* 'select' a data item. This is useful for scrolling through informational   *
//* message text rather than through selectable items.                         *
//*                                                                            *
//* Input  : uiInfo class (by reference) - initial values ignored              *
//*                                                                            *
//* Returns: index of control that currently has the input focus               *
//*           (See note in NcDialog.hpp about interpretation )                 *
//*           (of values returned in the uiInfo-class object.)                 *
//*           (highlight IS NOT visible on return)                             *
//******************************************************************************

short NcDialog::ViewScrollext ( uiInfo& info )
{

   return ( (this->EditScrollext ( info, false )) ) ;

}  //* End ViewScrollext() *

//*************************
//*     EditScrollext     *
//*************************
//******************************************************************************
//* PRIVATE METHOD:                                                            *
//* User interface for editing the position of the highlight within a          *
//* dctSCROLLEXT control.                                                      *
//*                                                                            *
//* Input  : uiInfo class (by reference) - initial values ignored              *
//*           (see note in NcDialog.hpp about values returned)                 *
//*          withHighlight: if 'true', highlight is visible on the current     *
//*                          display item, and data are scrolled one line at   *
//*                          a time.                                           *
//*                         if 'false' highlight is not visible and data are   *
//*                          scrolled as a block.                              *
//*                                                                            *
//* Returns: index of control that currently has the input focus               *
//******************************************************************************
//* Programmer's Notes:                                                        *
//* Even though the dctSCROLLEXT control looks like and behaves like the       *
//* dctSCROLLBOX control in most respects, the edit method is somewhat         *
//* different in that the caller has a more active role in the edit, rather    *
//* than simply waiting for a selection to be made (or edits abandoned).       *
//* This allows for greater flexibility in the way data are displayed and      *
//* selections made, as well as allowing for dynamic changes in the display    *
//* data. Please refer to the Dialog2 test application, Test01() for an        *
//* example of this dynamic edit.                                              *
//*                                                                            *
//* This edit method handles the following user input:                         *
//*  1. Scrolling keys: nckUP, nckDOWN, nckHOME, nckEND, nckPGUP, nckPGDOWN    *
//*                     (nckLEFT and nckRIGHT are not handled here, but are    *
//*                      converted to nckSTAB and nckTAB, respectively to      *
//*                      conform to editing methods for other control types.)  *
//*  2. Hotkeys associated with controls in the dialog window will cause       *
//*     focus to move to the specified ACTIVE control before return.           *
//*  3. The Insert/Overstrike toggle is handled.                               *
//*  4. The nckRESIZE signal causes the screen data to be refreshed.           *
//*  5. The nckENTER and nckSPACE keys _ARE NOT_ handled, and cause a return   *
//*     to caller. These and other non-handled keys are also available to the  *
//*     callback method (if any) for processing if desired.                    *
//*  6. If in 'View Mode' (no highlight), the nckESC key is remapped to the    *
//*     nckTAB key.                                                            *
//*                                                                            *
//* All other key input is returned to caller for processing.                  *
//******************************************************************************

short NcDialog::EditScrollext ( uiInfo& info, bool withHighlight )
{
   DialogScrollext* cp = NULL ;     // pointer to control object
   bool  done = false ;             // loop control

   //* Be sure the control with focus is actually a DialogScrollext *
   if ( this->dCtrl[this->currCtrl]->type == dctSCROLLEXT )
   {
      cp = (DialogScrollext*)this->dCtrl[this->currCtrl] ;
      if ( withHighlight )
      {
         //* If the highlight is invisible, make it visible.*
         if ( cp->hlShow == false )
         {
            cp->hlShow = true ;
            cp->wPtr->HilightItem ( true ) ;
            cp->RefreshControl () ;
         }
      }
      else
         cp->wPtr->HilightItem ( false ) ; // un-highlight the current item
   }
   else            // control is not a scroll-ext control i.e. application error
   {
      //* Do what we can to minimize the damage *
      info.ctrlType = this->dCtrl[this->currCtrl]->type ;
      info.ctrlIndex = this->currCtrl ;
      info.keyIn = nckTAB ;               // move on to next control
      info.hasFocus = true ;              // control has focus
      info.dataMod = false ;              // no data modified
      info.selMember = MAX_DIALOG_CONTROLS ; // don't care
      info.isSel = false ;                // don't care
      info.viaHotkey = false ;            // invalidate any hotkey data
      done = true ;                       // don't enter the loop
   }

   short origIndex = cp->bIndex ;         // remember the initial index

   //**************************
   //* Interact with the user *
   //**************************
   while ( ! done )
   {
      //* 1) Get user input.                                  *
      //* 2) Check for toggle of Insert/Overstrike mode.      *
      //* 3) Check for terminal-resize event.                 *
      //* 4) Access callback method, if specified.            *
      //* 5) If input has been handled, return to top of loop.*
      this->GetKeyInput ( info.wk ) ;

      if ( (info.wk.type == wktFUNKEY)
            && (info.wk.key == nckINSERT || info.wk.key == nckRESIZE) )
      {
         if ( info.wk.key == nckINSERT )
            this->ToggleIns () ;
         else
            this->TermResized () ;

         if ( ExternalControlUpdate != NULL )
            ExternalControlUpdate ( this->currCtrl, info.wk, false ) ;
         continue ;
      }

      //* If we are in 'View' mode and we get an ESCAPE key, *
      //* convert it to a TAB key to let user escape.        *
      if ( (withHighlight == false) && 
           (info.wk.type == wktFUNKEY) && (info.wk.key == nckESC) )
         info.wk.key = nckTAB ;

      //* Done scrolling. Move to next control *
      if ( (info.wk.type == wktFUNKEY) && 
           (info.wk.key == nckTAB || info.wk.key == nckRIGHT) )
      {
         info.ctrlType = dctSCROLLEXT ;      // control type
         info.ctrlIndex = this->currCtrl ;   // control with input focus
         info.hasFocus = true ;              // focus was not lost
         info.dataMod = (cp->bIndex != origIndex) ? true : false ;
         info.keyIn = nckTAB ;               // focus to move forward to next control
         info.selMember = cp->bIndex ;       // current member is the 'selected' member
         info.isSel = false ;                // don't care
         info.viaHotkey = false ;            // control did not get focus via hotkey
         done = true ;                       // returning to caller
      }
      //* Done scrolling. Move to previous control *
      else if ( (info.wk.type == wktFUNKEY) && 
                (info.wk.key == nckSTAB || info.wk.key == nckLEFT) )
      {
         info.ctrlType = dctSCROLLEXT ;      // control type
         info.ctrlIndex = this->currCtrl ;   // control with input focus
         info.hasFocus = true ;              // focus was not lost
         info.dataMod = (cp->bIndex != origIndex) ? true : false ;
         info.keyIn = nckSTAB ;              // focus to move backward to previous control
         info.selMember = cp->bIndex ;       // current member is the 'selected' member
         info.isSel = false ;                // don't care
         info.viaHotkey = false ;            // control did not get focus via hotkey
         done = true ;                       // returning to caller
      }

      //* Scroll up or down through list *
      // Note: nckLEFT and nckRIGHT are scroll keys also, but we eliminated 
      //       them in above tests because they don't have a function in 
      //       a vertical scrolling control.
      else if ( (cp->wPtr->IsScrollKey ( info.wk )) != false )
      {
         if ( withHighlight )    // with visible highlight
            cp->bIndex = cp->wPtr->ScrollData ( info.wk.key ) ;
         else                    // view only, no highlight
         {
            cp->wPtr->ScrollView ( info.wk.key ) ;
            cp->bIndex = cp->wPtr->GetHilightIndex () ;
         }
      }

      else     // test for a possible hotkey
      {
         //* Scan the controls for one whose hotkey matches the input.*
         //* If match found, returns control's index, else -1.        *
         short hotIndex = this->IsHotkey ( info.wk ) ;
         if ( hotIndex >= ZERO )
         {
            //* Ignore hotkey selection of THIS scroll-ext control          *
            //* else, make the indicated control the current/active control *
            if ( hotIndex != this->currCtrl )
            {
               //* Indicate the results of our edit *
               info.ctrlType = dctSCROLLEXT ;   // control type
               info.ctrlIndex = this->currCtrl ;// control with input focus
               info.hasFocus = true ;           // this may be reset by ChangedFocusViaHotkey()
               info.dataMod = (cp->bIndex != origIndex ||
                               info.keyIn == nckENTER || info.keyIn == nckpENTER) ?
                              true : false ;
               info.keyIn = ZERO ;              // indicate no shift in focus necessary
               info.selMember = cp->bIndex ;    // current member is the 'selected' member
               info.isSel = false ;             // don't care
               info.viaHotkey = false ;         // this may be set by ChangedFocusViaHotkey()
               this->ChangedFocusViaHotkey ( info.wk.key, hotIndex, info ) ;
               done = true ;
            }
         }
         else
         {
            //* Not a scroll key and not a hotkey. Return it to caller.*
            info.ctrlType = dctSCROLLEXT ;      // control type
            info.ctrlIndex = this->currCtrl ;   // control with input focus
            info.hasFocus = true ;              // focus was not lost
            info.keyIn = info.wk.key ;          // unhandled key input value
            info.dataMod = ((cp->bIndex != origIndex) || 
                            (info.wk.type == wktPRINT && info.wk.key == nckSPACE) ||
                            ((info.wk.type == wktFUNKEY) &&
                             (info.keyIn == nckENTER || info.keyIn == nckpENTER)) ) ?
                           true : false ;
            info.selMember = cp->bIndex ;       // current member is the 'selected' member
            info.isSel = false ;                // don't care
            info.viaHotkey = false ;            // control did not get focus via hotkey
            done = true ;                       // returning to caller
         }
      }
      //* If caller has specified a callback method, do it now.*
      if ( ExternalControlUpdate != NULL )
         ExternalControlUpdate ( this->currCtrl, info.wk, false ) ;
   }     // while()

   return this->currCtrl;

}  //* End EditScrollext() *

//*************************
//*   SetScrollextText    *
//*************************
//******************************************************************************
//* Specify the display text and associated color attributes for a             *
//* dctSCROLLEXT control. Unlike the display data for other control types, the *
//* display data for dctSCROLLEXT control is external to the NcDialog's memory *
//* space. This is done to allow the application to dynamically update the     *
//* display text and color attribute data.                                     *
//*                ("You'll thank me later." - Adrian Monk)                    *
//*                                                                            *
//* NOTE: It is the application's responsibility to ensure that each item's    *
//*       text string width (number of display columns) exactly fits the width *
//*       of the control window i.e. the 'cols' specified during               *
//*       instantiation, minus 2 (for left and right border columns).          *
//* NOTE: If there are fewer data items than display lines, the unused lines   *
//*       will be filled with the background color of the first display item.  *
//*                                                                            *
//* Input  : cIndex: index of target dctSCROLLEXT control                      *
//*          displayData: an initialized ssetData class object                 *
//*           dispText : pointer to an array of pointers to text strings       *
//*           dispColor: pointer to an array of color attribute codes (int)    *
//*           dispItems: number of items in the display arrays                 *
//*           hlIndex  : specifies the item that is initially highlighted      *
//*           hlShow   : if true, show the highlight, else, hide it            *
//*                      Highlight remains in the specified state until        *
//*                      changed by another call to SetScrollextText() or until*
//*                      RefreshScrollextText() or EditScrollext() is called.  *
//*                                                                            *
//* Returns: OK if successful, else ERR                                        *
//******************************************************************************
//* Programmer's Note: Because the caller's data may not completely fill the   *
//* window, we set the window's background color. Any unused lines will have   *
//* no data in them, so it is only the background color, not the text color    *
//* that matters. However, because the data are intrinsically multi-colored    *
//* data, we must make a decision about what color to use.                     *
//*                                                                            *
//* Here is how I see it: The background color of all display lines is probably*
//* the same. If not, the display will look awful anyway, so our decision won't*
//* make it any worse. Therefore we use the color attribute of the first data  *
//* line (minus some modifier bits) as the background fill color.              *
//*                                                                            *
//* If more control is needed, we will have to add a 'background' member to    *
//* the ssetData class.                                                        *
//******************************************************************************

short NcDialog::SetScrollextText ( short cIndex, ssetData& dData )
{
short    result = ERR ;

   if ( cIndex >= ZERO && cIndex <= this->lastCtrl
        && this->dCtrl[cIndex]->type == dctSCROLLEXT 
        && dData.dispText != NULL 
        && dData.dispColor != NULL && dData.dispItems > ZERO 
        && dData.hlIndex >= ZERO && dData.hlIndex < dData.dispItems )
   {
      //* Pointer to the control's methods and data *
      DialogScrollext* dp = (DialogScrollext*)this->dCtrl[cIndex] ;

      //* Set the window's background color in case data *
      //* do not fill the window. (see note above)       *
      // Programmer's Note: This may not work as expected: border color is
      // actually set as background, but that works, so we haven't fixed it.
      attr_t fillattr = dData.dispColor[ZERO] & ~(ncuATTR) ;
      dp->wPtr->SetInteriorColor ( fillattr ) ;

      //* Re-initialize the data pointer to reference caller's display data *
      dp->bText  = dData.dispText ;
      dp->bColor = dData.dispColor ;
      dp->bItems = dData.dispItems ;
      dp->bIndex = dData.hlIndex ;
      dp->hlShow = dData.hlShow ;
      dp->RedrawControl ( (bool)(cIndex==this->currCtrl ? true : false), true ) ;
      result = OK ;
   }
   return result ;

}  //* End SetScrollextText() *

//************************
//* RefreshScrollextText *
//************************
//******************************************************************************
//* Refresh the dctSCROLLEXT display data previously specified via a call      *
//* to SetScrollextText(). Call this method if the text data and/or color      *
//* attribute data have been modified BUT the number of items in the arrays    *
//* have not. This method provides a much faster display update that to call   *
//* SetScrollextText() again.                                                  *
//*                                                                            *
//* Input : cIndex : index of target dctSCROLLEXT control                      *
//*         hlShow : (optional, default==true): if true, highlight             *
//*                  the current data item, else current data item is          *
//*                  drawn without highlight. Highlight remains in the         *
//*                  specified state until changed by another call to          *
//*                  RefreshScrollextText() or until SetScrollextText() or     *
//*                  EditScrollext() is called.                                *
//*                                                                            *
//* Returns: OK if successful, else ERR                                        *
//******************************************************************************

short NcDialog::RefreshScrollextText ( short cIndex, bool hlShow )
{
short    result = OK ;

   if ( cIndex <= this->lastCtrl && this->dCtrl[cIndex]->type == dctSCROLLEXT )
   {
      //* Pointer to the control's methods and data *
      DialogScrollext* dp = (DialogScrollext*)this->dCtrl[cIndex] ;
      dp->hlShow = hlShow ;
      dp->RedrawControl ( (bool)(cIndex==this->currCtrl ? true : false), false ) ;
   }
   return result ;

}  //* End RefreshScrollextText() *

//**************************
//* MoveScrollextHighlight *
//**************************
//******************************************************************************
//* Move the highlight within a dctSCROLLEXT control.                          *
//* Operates as if the user had pressed the specified scrolling key.           *
//*                                                                            *
//* Input  : cIndex : index of target dctSCROLLEXT control                     *
//*          sKey   : key code for one of the valid scrolling keys:            *
//*                   nckUP, nckDOWN, nckPGUP, nckPGDOWN, nckHOME, nckEND      *
//*                   (all other key values ignored)                           *
//*          iIndex : (optional: default == -1): if specified,                 *
//*                   indicates the index of the data item to be               *
//*                   highlighted (sKey will be ignored)                       *
//*                                                                            *
//* Returns: index of data item which is currently highlighted                 *
//*          or ERR if specified control is not of type dctSCROLLEXT           *
//*            or if invalid iIndex specified                                  *
//******************************************************************************

short NcDialog::MoveScrollextHighlight ( short cIndex, wchar_t sKey, short iIndex )
{
short    memberIndex = ERR ;

   if ( cIndex >= ZERO && cIndex <= this->lastCtrl 
        && this->dCtrl[cIndex]->type == dctSCROLLEXT )
   {
      //* Access the control *
      DialogScrollext* cp = (DialogScrollext*)this->dCtrl[cIndex] ;

      //* Process the key input value (invalid keycodes ignored) *
      if ( iIndex == -1 )
         memberIndex = cp->bIndex = cp->wPtr->ScrollData ( sKey ) ;

      //* Move highlight to specified display item *
      else if ( iIndex >= ZERO && iIndex < cp->bItems )
         memberIndex = cp->bIndex = cp->wPtr->Track2Item ( iIndex, true ) ;

      //* If highlight should be invisible, 'Make it so.' *
      if ( ! cp->hlShow )
         cp->wPtr->HilightItem ( false, true ) ;
   }
   return memberIndex ;

}  //* End MoveScrollextHighlight() *

//*************************
//*  GetScrollextSelect   *
//*************************
//******************************************************************************
//* Returns the index of the highlighted item in the specified dctSCROLLEXT    *
//* control.                                                                   *
//*                                                                            *
//* Input  : cIndex : index number of source control                           *
//*                                                                            *
//* Returns: index of 'selected' (highlighted) item                            *
//*          or ERR if:                                                        *
//*            a) specified control is not of type dctSCROLLEXT                *
//******************************************************************************

short NcDialog::GetScrollextSelect ( short cIndex )
{
   short itemIndex = ERR ;

   //* If caller has passed a valid index number *
   if ( (cIndex >= ZERO) && (cIndex <= this->lastCtrl)
        && (dCtrl[cIndex]->type == dctSCROLLEXT) )
   {
      DialogScrollext* cp = (DialogScrollext*)(dCtrl[cIndex]) ;
      itemIndex = cp->bIndex ;
   }
   return itemIndex ;

}  //* End GetScrollextSelect() *

//*************************
//*  SetScrollextSelect   *
//*************************
//******************************************************************************
//* Set 'selected' (highlighted) item in specified dctSCROLLEXT control.       *
//*                                                                            *
//* Input  : cIndex : index number of target control                           *
//*          selMember : index of item to be set as 'selected'                 *
//*                                                                            *
//* Returns: index of 'selected' member                                        *
//*          or ERR if:                                                        *
//*            a) invalid item index specified                                 *
//*            b) specified control currently has the input focus              *
//*            c) specified control is not of type dctSCROLLEXT                *
//******************************************************************************

short NcDialog::SetScrollextSelect ( short cIndex, short selMember )
{
   short itemIndex = ERR ;
   //* If caller has passed a valid index number *
   if ( (cIndex >= ZERO) && (cIndex <= this->lastCtrl)
        && (dCtrl[cIndex]->type == dctSCROLLEXT)
        && (cIndex != this->currCtrl) )
   {
      DialogScrollext* cp = (DialogScrollext*)(dCtrl[cIndex]) ;

      //* If a non-highlighted item has been specified for selection *
      //* move highlight to that position.                           *
      if ( selMember >= ZERO && selMember < cp->bItems )
      {
         while ( cp->bIndex != selMember )
         {
            if ( selMember > cp->bIndex )    // scroll down
               cp->bIndex = cp->wPtr->ScrollData ( nckDOWN ) ;
            else                             // scroll up
               cp->bIndex = cp->wPtr->ScrollData ( nckUP ) ;
         }
         //* Set currently-highlighted item as selection *
         itemIndex = cp->bIndex ;
      }
   }
   return itemIndex ;

}  //* End SetScrollextSelect() *

//*************************
//*                       *
//*************************
//********************************************************************************
//*                                                                              *
//*                                                                              *
//*                                                                              *
//* Input  :                                                                     *
//*                                                                              *
//* Returns:                                                                     *
//********************************************************************************

