//********************************************************************************
//* File       : NcdControlDD.cpp                                                *
//* Author     : Mahlon R. Smith                                                 *
//*              Copyright (c) 2006-2025 Mahlon R. Smith, The Software Samurai   *
//*                  GNU GPL copyright notice located in NcDialog.hpp            *
//* Date       : 25-Mar-2021                                                     *
//* Version    : (see NcDialogVersion string in NcDialog.cpp)                    *
//*                                                                              *
//* Description: Contains the methods of the DialogDropdown class                *
//*              and the public NcDialog-class methods for accessing             *
//*              the DialogDropdown object's functionality.                      *
//*              See NcDialog.hpp for the definition of this class.              *
//*                                                                              *
//*                                                                              *
//* Development Tools: See NcDialog.cpp.                                         *
//********************************************************************************
//* Version History (most recent first):                                         *
//*   See version history in NcDialog.cpp.                                       *
//********************************************************************************

//*****************
//* Include Files *
//*****************

#include "GlobalDef.hpp"               //* General definitions

#ifndef NCURSES_INCLUDED
#include "NCurses.hpp"
#endif

#ifndef NCDIALOG_INCLUDED
#include "NcDialog.hpp"
#endif


      //*************************************************
      //** THIS SECTION IMPLEMENTS THE METHODS OF THE  **
      //**           DialogDropdown CLASS.             **
      //** - - - - - - - - - - - - - - - - - - - - - - **
      //** See below for public NcDialog-class methods **
      //**   related to DialogDropdown-class access.   **
      //*************************************************

//*************************
//*    DialogDropdown     *
//*************************
//******************************************************************************
//* Constructor for DialogDropdown class object.                               *
//*                                                                            *
//*                                                                            *
//* Input  : iPtr: pointer to initialization structure                         *
//*          dUY : screen offset in Y for first line of parent dialog          *
//*          dLI : number of lines in parent dialog                            *
//*                                                                            *
//* Returns: constructors implicitly return a pointer to the object            *
//******************************************************************************
//* Programmer's Note: We make every effort to satisfy the caller's request    *
//* for the size and position of the 'expanded' control.                       *
//* The minimum number of display lines required for this (or any other)       *
//* scrolling data is three (DDBOX_MIN_LINES) lines: top border, one data      *
//* line, and bottom border.The following is a scenario for positioning and    *
//* sizing the data display while the control is in its 'expanded' form.       *
//* This example uses five (5) VISIBLE data items (there could be many more    *
//* items that are currently outside the display range). Five visible items    *
//* requires seven (7) display lines.                                          *
//* If expanding the control in the specified direction would overwrite the    *
//* top or bottom border of the dialog, we have two choices:                   *
//*  1. Disallow the request (which would cause chaos in the dialog window's   *
//*     constructor i.e. we couldn't let the dialog open), OR                  *
//*  2. Reduce the number of VISIBLE items in the expanded display until the   *
//*     expansion will stay within the borders of the dialog.                  *
//* If the application requested more than five visible items in any of these  *
//* controls, it would cause the control to expand over or beyond the dialog   *
//* window's border. In this case, we would resize the expanded control        *
//* window to maintain the integrity of the dialog's borders. The application  *
//* will receive an OK code, even though it didn't get exactly what it         *
//* requested.                                                                 *
//*                                                                            *
//* __TOP OF DIALOG_________________________________________________________   *
//*                                     +-------------+                        *
//*                                     |1st item     |   Centered Expansion   *
//*                   Expand Downward   |2nd item     |   +-------------+      *
//* Collapsed Form:      (default)      |3rd item     |   |1st item     +      *
//* +-------------+   +-------------+   |4th item     |   |2nd item     +      *
//* |selected item*---|1st item     |---|5th item     |---|3rd item     +      *
//* +-------------+   |2nd item     |   +-------------+   |4th item     +      *
//*                   |3rd item     |    Expand Upward    |5th item     +      *
//*                   |4th item     |                     +-------------+      *
//*                   |5th item     |                                          *
//*                   +-------------+                                          *
//* ____________________________  ____________________________________________ *
//*   BOTTOM OF DIALOG                                                         *
//******************************************************************************

DialogDropdown::DialogDropdown ( InitCtrl* iPtr, short dUY, short dLI )
{
   //* Transfer caller's initialization data *
   this->type   = dctDROPDOWN ;     // set control type
   this->ulY    = iPtr->ulY ;       // control's position in dialog
   this->ulX    = iPtr->ulX ;
   this->lines  = iPtr->lines ;     // preliminary height in 'expanded' mode
   this->cols   = iPtr->cols ;      // width in 'expanded' mode
   this->nColor = iPtr->nColor ;    // color of border when control does not have focus
   this->fColor = iPtr->fColor ;    // color of border when control has focus
   this->labY   = iPtr->labY ;      // label offsets
   this->labX   = iPtr->labX ;
   this->bIndex = iPtr->scrSel ;    // index for initial position of highlight
   this->bItems = iPtr->scrItems ;  // number of display strings
   this->ddxp   = iPtr->exType ;    // direction in which control expands
   this->active = iPtr->active ;    // indicates whether control can be selected

   //* Default values for all other data members of the class *
   this->expanded = false ;         // control is initially collapsed
   this->sdPaint  = false ;         // selection data not yet displayed
   this->tLines = DDBOX_MIN_LINES ; // display lines in 'collapsed' mode
   this->tCols  = this->cols ;      // display columns in 'collapsed' mode
   this->tulY   = this->ulY ;       // position of 'collapsed' control
   this->tulX   = this->ulX ;
   this->tcChar = wcsDIAMOND ;      // diamond shape
   this->tcAttr = iPtr->nColor ;    // (currently unused)
   this->tcYpos = 1 ;               // Y offset for special char (center row)
   this->tcXpos = this->tCols - 1 ; // X offset for special char (right border)
   this->typos  = 1 ;               // Y offest for string display (center row)
   this->txpos  = 1 ;               // X offset for string display (inside left border)
   this->groupCode = ZERO ;         // drop-down controls are not grouped (not used) 
   this->rtlContent = false ;       // left-to-right language content by default

   //* Copy the label text to data member, wLabel and initialize bHotkey.      *
   //* Note that if a multi-line label was specified for a label embedded      *
   //* within the control's border, we strip the newlines from the string as   *
   //* an indication to the user that he/she/it screwed up.                    *
   this->InitHotkey ( iPtr->label, bool(this->labY == ZERO && this->labX == ZERO) ) ;

   //* Allocate space for strings, pointers, and color attributes.             *
   //* This is a single memory allocation, divided into sections.              *
   //* Note that our initial allocation is likely larger than we actually need,*
   //* so we give back the excess, below.                                      *
   void* blkptr ;
   char* dString ;
   int bytesNeeded = bItems * (this->cols) * 4 ;      // space for display strings
   bytesNeeded    += bItems * sizeof(char*) ;         // space for string pointers
   bytesNeeded    += bItems * sizeof(attr_t) ;        // space for color attributes
   if ( (blkptr = calloc ( bytesNeeded, 1 )) != NULL )// allocate a memory block
   {
      //* Pointer to array of menu-item pointers *
      this->bText   = (char**)blkptr ;
      //* Point to array of color attributes *
      this->bColor  = (attr_t*)((char*)this->bText + (this->bItems*sizeof(char*))) ;
      //* Point to array of display-item data strings *
      dString       = (char*)((char*)this->bColor + (this->bItems*sizeof(attr_t))) ;

      //* Copy display data and color attributes to local storage. Normalizing *
      //* the number of display columns required by each menu item.            *
      const char* cPtr = iPtr->dispText ;    // pointer to source text
      short reqCols  = this->cols - 2,       // display columns available
            ubytes ;                         // for item-width calculations
      gString gs ;                           // text formatting

      for ( short iCount = ZERO ; iCount < this->bItems ; iCount++ ) // for each menu item
      {
         this->bText[iCount] = dString ; // initialize pointer to formatted display string

         //* Copy the display item to a formatting buffer.                     *
         //* Note that string length is limited to number of display columns.  *
         gs = cPtr ;

         //* If caller has correctly formatted the display item, just copy it. *
         if ( gs.gscols() == reqCols )
         {
            ubytes = gs.utfbytes () ;
            gs.copy( this->bText[iCount], ubytes ) ;
            cPtr += ubytes ;                 // point to next source string
            if ( iCount < (this->bItems -1) )// step over NULLCHAR filler
            { while ( *cPtr == NULLCHAR ) ++cPtr ; }
            dString += ubytes ;              // point to next target buffer
         }
         //* If a display item is too short, copy what we have and pad on right*
         else if ( gs.gscols() < reqCols )
         {
            ubytes = gs.utfbytes () ;
            while ( gs.gscols() < reqCols )
               gs.append( L' ' ) ;
            gs.copy( this->bText[iCount], gs.utfbytes() ) ;
            cPtr += ubytes ;                 // point to next source string
            if ( iCount < (this->bItems -1) )// step over NULLCHAR filler
            { while ( *cPtr == NULLCHAR ) ++cPtr ; }
            dString += gs.utfbytes() ;       // point to next target buffer
         }
         //* Source string is too long - truncate it to fit control window.    *
         else  // gs.gscols() > reqCols
         {
            gs.limitCols(reqCols) ;          // limit string to available width
            ubytes = gs.utfbytes () ;
            gs.copy( this->bText[iCount], ubytes ) ;
            cPtr += ubytes - 1 ;             // point to next source string
            if ( iCount < (this->bItems -1) )// step over NULLCHAR filler
            { while ( *cPtr == NULLCHAR ) ++cPtr ; }
            dString += ubytes ;              // point to next target buffer
         }

         //* Establish color of display string *
         if ( iPtr->scrColor[0] != attrDFLT )
         {
            this->bColor[iCount] = iPtr->scrColor[iCount] ;
         }
         else
         {
            this->bColor[iCount] = iPtr->scrColor[1] ;
         }
      }  // end for(;;)

      //* Return unneeded portion of memory allocation to the heap.            *
      // Programmer's Note: realloc() MAY copy the block to a new location,    *
      // even when shrinking block, necessitating adjustement of our pointers. *
      void* newblkPtr = realloc ( blkptr, (dString - (char*)blkptr) ) ;
      if ( newblkPtr != blkptr )
      {
         this->bText   = (char**)newblkPtr ;
         this->bColor  = (attr_t*)((char*)this->bText + (this->bItems*sizeof(char*))) ;
         dString       = (char*)((char*)this->bColor + (this->bItems*sizeof(attr_t))) ;
         for ( short iCount = ZERO ; iCount < this->bItems ; iCount++ ) // for each display item
         {
            this->bText[iCount] = dString ;
            while ( *dString != NULLCHAR )
               ++dString ;
            ++dString ;
         }
      }
   }  // calloc()

   //* Memory allocaton error will be reported in control window *
   else
   {
      //* Set all variables to 'safe' values *
      // (of course, nothing is 'safe' after a memory allocation error)
      bPtr     = NULL ;
      wPtr     = NULL ;
      tPtr     = NULL ;
      bText    = NULL ;
      bColor   = NULL ;
      bIndex   = -1 ;
      bItems   = -1 ;
      *wLabel  = NULLCHAR ;
      labY     = ZERO ;
      labX     = ZERO ;
   }

   //* Because this control 'expands' when user in interacting with it, *
   //* AND because there are three directions in which the control may  *
   //* expand, we need to calculate the vertical position and size of   *
   //* the expansion windows.                                           *
   //* NOTE: The this->ulY and dUY values are screen offsets NOT dialog *
   //* offsets.                                                         *
   short    dfirstY = dUY, dlastY = dfirstY + dLI - 1 ;
   if ( this->ddxp == ddBoxUP )  // expands upward from the base position
   {
      //* Bottom of expanded control == bottom of collapsed control */
      this->ulY -= this->lines - 3 ;
      while ( (this->ulY < dfirstY) && (this->lines > DDBOX_MIN_LINES) )
      {
         --this->lines ;
         ++this->ulY ;
      }
   }
   else if ( this->ddxp == ddBoxCENTER )  // expands centered on the base position
   {
      //* If an even number of visible items, then      *
      //* the extra item should be below the centerline *
      this->ulY -= (this->lines-1)/2 - 1 ;

      //* Size the expanded control to fit the vertical space *
      if ( this->ulY < dfirstY )
         this->ulY = dfirstY ;
      while ( ((this->ulY + this->lines - 1) > dlastY) && 
              (this->lines > DDBOX_MIN_LINES) ) 
      {
         //* Reduce the number of display lines required *
         --this->lines ;
         if ( this->lines > DDBOX_MIN_LINES )
         {
            --this->lines ;
            if ( this->ulY < this->tulY )
               ++this->ulY ;
         }
      }
   }
   else  // expands downward (ddBoxDOWN) or incorrect parameter value
   {
      this->ddxp = ddBoxDOWN ;      // force a valid expansion direction

      //* Size the expanded control to fit the vertical space *
      while ( ((this->ulY + this->lines - 1) > dlastY) && 
              (this->lines > DDBOX_MIN_LINES) )
         --this->lines ;
   }

   //*********************************************
   //* Instantiate the underlying window objects.*
   //* No display data are drawn to screen here; *
   //* see DialogDropdown::OpenControl()         *
   //*********************************************
   // 'Collapsed' control's window contains only a copy of the 'selected' item
   this->tPtr = new NcWindow ( this->tLines, this->tCols, this->tulY, this->tulX ) ;
   // 'Expanded' control's outer window contains border
   this->bPtr = new NcWindow ( this->lines, this->cols, this->ulY, this->ulX ) ;
   // 'Expanded' control's inner window contains scrolling data
   this->wPtr = new NcWindow ( this->lines-2, this->cols-2, this->ulY+1, this->ulX+1 ) ;

}  //* End DialogDropdown() *

//*************************
//*    ~DialogDropdown    *
//*************************
//******************************************************************************
//* Destructor for DialogDropdown class object.                                *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

DialogDropdown::~DialogDropdown ( void )
{
   //* Release dynamic allocation for display data *
   free ( (void*)this->bText ) ;

   //* Close the underlying windows *
   delete this->bPtr ;
   delete this->wPtr ;
   delete this->tPtr ;

}  //* End ~DialogDropdown() *

//*************************
//*     OpenControl       *
//*************************
//******************************************************************************
//* Draw a DialogDropdown control i.e. open the window previously              *
//* instantiated by a call to DialogDropdown().                                *
//*                                                                            *
//* Input  : hasFocus : optional boolean, false by default, determines         *
//*                     whether to draw the object in nColor (does             *
//*                     not have focus) or fColor (has focus)                  *
//*                                                                            *
//* Returns: OK if successful, else ERR                                        *
//******************************************************************************

short DialogDropdown::OpenControl ( bool hasFocus )
{
   short    result = ERR ;

   //* Open the control in 'collapsed' form, that  *
   //* is,with the scrolling window invisible.     *
   if (   (this->tPtr->OpenWindow () == OK)
       && (this->bPtr->OpenWindow () == OK) 
       && (result = this->wPtr->OpenWindow ()) == OK )
   {
      //* Hurray! all component windows of the control are open!
      
      //* Set interior color of the scrolling-data window.   *
      //* This is done in case the scrolling data do not fill*
      //* all the rows of the scroll window. The color used  *
      //* is the color of the first display item.            *
      this->wPtr->SetInteriorColor ( this->bColor[ZERO] ) ;

      //* If content defined as RTL, then adjust X position *
      //* for 'collapsed' data.                             *
      if ( this->rtlContent != false )
         this->txpos = this->cols - 2 ;   // 'collapsed' configuration

      //* The call to RedrawControl makes the control visible.*
      this->RedrawControl ( hasFocus ) ;
   }
   return result ;

}  //* End OpenControl() *

//*************************
//*     RedrawControl     *
//*************************
//******************************************************************************
//* Redraw the data in a Dropdown window.                                      *
//* (refreshes the control's window)                                           *
//*                                                                            *
//* There are two modes: 'collapsed' mode and 'expanded mode. The mode in      *
//* which the control is drawn depends on the state of the 'expanded'          *
//* flag.                                                                      *
//*                                                                            *
//* Input  : hasFocus: if true, control has focus                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void DialogDropdown::RedrawControl ( bool hasFocus )
{
   attr_t   borderColor = hasFocus ? fColor : nColor ;

   if ( this->expanded == false )         // control is 'collapsed'
   {
      this->tPtr->DrawBorder ( borderColor, NULL, ncltSINGLE, this->rtlContent ) ;
      this->tPtr->WriteChar ( this->tcYpos, this->tcXpos, this->tcChar, borderColor ) ;
      short xpos = this->txpos ;
      if ( this->rtlContent )    // for RTL data position cursor inside right border
         xpos = this->tCols - 2 ;
      this->tPtr->WriteString ( this->typos, xpos, this->bText[this->bIndex], 
                                this->bColor[this->bIndex], true, this->rtlContent ) ;
   }
   else                                   // control is 'expanded'
   {
      //* NOTE: We arrive here ONLY via a call from inside the EditDropdown
      //* Caller has marked the parent dialog window as 'obscured' i.e. saved 
      //* the display data for the entire dialog; thus, we can overwrite 
      //* whatever we need to when displaying the expanded control window.
      this->bPtr->DrawBorder ( borderColor ) ;// draw the outer border in appropriate color
      this->bPtr->RefreshWin () ;         // display the outer window
      if ( this->bText != NULL )          // are we pointing at valid scroll data?
      {
         if ( this->sdPaint == false )    // if scrolling data not yet painted to window
         {
            this->wPtr->PaintData ( (const char**)this->bText, this->bColor, 
                                    this->bItems, this->bIndex,
                                    true, this->rtlContent ) ; 
            this->sdPaint = true ;
         }
         else
            this->wPtr->RepaintData () ;     // redraw and refresh the scrolling data 
      }
      else     // Else, constructor could not allocate working memory
         this->wPtr->WriteString ( 1, 1, "MEMORY ALLOCATION ERROR!", fColor ) ;
      this->wPtr->RefreshWin () ;         // in the inner window
   }

}  //* End RedrawControl()

//*************************
//*    RefreshControl     *
//*************************
//******************************************************************************
//* Refresh the display of the specified dialog control object.                *
//*                                                                            *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void DialogDropdown::RefreshControl ( void )
{

   if ( this->expanded == false )
   {
      this->tPtr->RefreshWin () ;         // refresh collapsed window
   }
   else
   {
      this->bPtr->RefreshWin () ;         // refresh expanded window
      this->wPtr->RefreshWin () ;         // refresh the inner (data) window
   }

}  //* End RefreshControl() *

//*************************
//*    SetOutputFormat    *
//*************************
//******************************************************************************
//* Set the text data output format: RTL (right-to-left), or                   *
//* LTR (left-to-right). Does not refresh display.                             *
//* Note that the default (unmodified) output format is LTR.                   *
//*                                                                            *
//* Input  : rtlFormat : if 'true',  then set as RTL format                    *
//*                      if 'false', then set as LTR format                    *
//*                                                                            *
//* Returns: OK  if successful                                                 *
//*          ERR if data have already been initialized                         *
//******************************************************************************
//* Formatting is done in the OpenControl() method.                            *
//******************************************************************************

short DialogDropdown::SetOutputFormat ( bool rtlFormat )
{
   short status = ERR ;

   //* If data have not yet been initialized, set the output format.*
   if ( this->sdPaint == false )
   {
      this->rtlContent = rtlFormat ;
      status = OK ;
   }
   return status ;

}  //* End SetOutputFormat() *



      //*************************************************
      //**     THIS SECTION IMPLEMENTS THE PUBLIC,     **
      //**    NcDialog-class METHODS FOR ACCESSING     **
      //**    DialogDropdown-class FUNCTIONALITY.      **
      //*************************************************

//*************************
//*     EditDropdown      *
//*************************
//******************************************************************************
//* User interface for editing the position of the highlight within a          *
//* dctDROPDOWN control.                                                       *
//*                                                                            *
//* On entry, the control is in its 'collapsed' form i.e. only the currently-  *
//* 'selected' data item is visible. We make the remaining data items visible  *
//* at the correct moment, and then make them invisible again before returning *
//* to caller. To put it another way, the data from which the user makes a     *
//* selection are only visible while inside this method.                       *
//*                                                                            *
//* Input  : uiInfo class (by reference) - initial values ignored              *
//*           EXCEPT: If info.viaHotkey != false, expand the control           *
//*                   immediately on entry.                                    *
//*           (see note in NcDialog.hpp about values returned)                 *
//*                                                                            *
//* Returns: index of control that currently has the input focus               *
//******************************************************************************
//* Programmer's Note:                                                         *
//* If the control is about to be drawn in its 'expanded' state, we need to    *
//* mark the entire dialog as obscured, since we can't know what will be       *
//* overwritten when the control is transformed into its 'expanded' state.     *
//* Because the control is only 'expanded' when it has the input focus, it     *
//* will only become 'expanded' during a call to EditDropdown()                *
//*                                                                            *
//* Also, we cannot call the NcDialog::RefreshWin() method during the editing  *
//* process (nor allow it to be called elsewhere) because that would restore   *
//* the previous dialog display, thus destroying the display of the window     *
//* we are playing in.                                                         *
//******************************************************************************

short NcDialog::EditDropdown ( uiInfo& info )
{
   #define CAPTURE_DROPDOWN (0)     // For debugging only, capture screenshot

   DialogDropdown* cp ;
   short       origSelection ;
   bool        done = false ;       // loop control

   //* Be sure the control with focus is actually a DialogDropdown *
   if ( this->dCtrl[this->currCtrl]->type == dctDROPDOWN )
   {
      cp = (DialogDropdown*)this->dCtrl[this->currCtrl] ;
      origSelection = cp->bIndex ;
   }
   else               // control is not a Dropdown i.e. application error
   {
      //* Do what we can to minimize the damage *
      info.ctrlType = dCtrl[currCtrl]->type ;
      info.ctrlIndex = this->currCtrl ;
      info.keyIn = nckTAB ;               // move on to next control
      info.hasFocus = true ;              // control has focus
      info.dataMod = false ;              // no data modified
      info.selMember = MAX_DIALOG_CONTROLS ; // don't care
      info.isSel = false ;                // don't care
      info.viaHotkey = false ;            // invalidate any hotkey data
      done = true ;                       // don't enter the loop
   }

   //* If this control was selected via hotkey, expand the control immediately *
   if ( info.viaHotkey != false )
   {
      //* Save dialog window's display data because we are about to overwrite*
      //* some of it. Then expand the control so user can make a selection.  *
      this->CaptureDialogDisplayData () ;
      cp->expanded = true ;
      cp->RedrawControl ( true ) ;
   }

   //**************************
   //* Interact with the user *
   //**************************
   while ( ! done )
   {
      //* 1) Get user input.                                  *
      //* 2) Check for toggle of Insert/Overstrike mode.      *
      //* 3) Check for terminal-resize event.                 *
      //* 4) Access callback method, if specified.            *
      //* 5) If input has been handled, return to top of loop.*
      this->GetKeyInput ( info.wk ) ;

      if ( (info.wk.type == wktFUNKEY)
            && (info.wk.key == nckINSERT || info.wk.key == nckRESIZE) )
      {
         if ( info.wk.key == nckINSERT )
            this->ToggleIns () ;
         else
            this->TermResized () ;

         if ( ExternalControlUpdate != NULL )
            ExternalControlUpdate ( this->currCtrl, info.wk, false ) ;
         continue ;
      }

      //* If control is in the 'collapsed' state, then *
      //* nckDOWN and nckUP and nckPGUP are converted. *
      if ( (cp->expanded == false) && ((info.wk.type == wktFUNKEY) && 
           (info.wk.key == nckUP || info.wk.key == nckDOWN || info.wk.key == nckPGUP)) )
      {
         info.wk.key = (info.wk.key == nckDOWN) ? nckTAB : nckSTAB ;
      }

      //* Done scrolling. Move to next control *
      if ( (info.wk.type == wktFUNKEY) && 
           (info.wk.key == nckTAB || info.wk.key == nckRIGHT) )
      {
         //* If control is expanded, collapse it *
         if ( cp->expanded != false )
         {
            cp->expanded = false ;
            this->RefreshWin () ;
         }

         info.ctrlType = dctDROPDOWN ;       // 
         info.ctrlIndex = this->currCtrl ;   // control with input focus
         info.hasFocus = true ;              // focus was not lost
         info.keyIn = nckTAB ;               // focus to move forward to next control
         info.dataMod = true ;               // indicate selection change (but see below)
         info.selMember = cp->bIndex ;       // current member is the 'selected' member
         info.isSel = false ;                // don't care
         info.viaHotkey = false ;            // control did not get focus via hotkey
         done = true ;                       // returning to caller
      }
      //* Done scrolling. Move to previous control *
      else if ( (info.wk.type == wktFUNKEY) && 
                (info.wk.key == nckSTAB || info.wk.key == nckLEFT) )
      {
         //* If control is expanded, collapse it *
         if ( cp->expanded != false )
         {
            cp->expanded = false ;
            this->RefreshWin () ;
         }

         info.ctrlType = dctDROPDOWN ;       // 
         info.ctrlIndex = this->currCtrl ;   // control with input focus
         info.hasFocus = true ;              // focus was not lost
         info.keyIn = nckSTAB ;              // focus to move backward to previous control
         info.dataMod = true ;               // indicate selection change (but see below)
         info.selMember = cp->bIndex ;       // current member is the 'selected' member
         info.isSel = false ;                // don't care
         info.viaHotkey = false ;            // control did not get focus via hotkey
         done = true ;                       // returning to caller
      }

      //* If active 'select' of currently-highlighted item *
      else if (   (info.wk.type == wktFUNKEY && 
                   (info.wk.key == nckENTER || info.wk.key == nckpENTER))
               || (info.wk.type == wktPRINT && info.wk.key == SPACE) )
      {
         //* If control is not expanded, expand it now *
         if ( cp->expanded == false )
         {
            this->CaptureDialogDisplayData () ;
            cp->expanded = true ;
            cp->RedrawControl ( true ) ;
         }
         //* Else, Enter key selects currently-highlighted item *
         else
         {
            cp->expanded = false ;        // Collapse the control
            this->RefreshWin () ;

            info.ctrlType = dctDROPDOWN ;    // 
            info.ctrlIndex = this->currCtrl ;// control with input focus
            info.hasFocus = true ;           // focus was not lost
            info.keyIn = nckENTER ;          // active selection, not just leaving the control
            info.dataMod = true ;            // indicate selection change (but see below)
            info.selMember = cp->bIndex ;    // current member is the 'selected' member
            info.isSel = false ;             // don't care
            info.viaHotkey = false ;         // control did not get focus via hotkey
            done = true ;                    // returning to caller
         }
      }

      //* Scroll up or down through list *
      // Note: nckLEFT and nckRIGHT are scroll keys also, but we eliminated 
      //       them in above tests because they don't have a function in 
      //       a vertical scrolling control.
      else if ( (cp->wPtr->IsScrollKey ( info.wk )) != false )
      {
         //* If control is not expanded, expand it now *
         if ( cp->expanded == false )
         {
            this->CaptureDialogDisplayData () ;
            cp->expanded = true ;
            cp->RedrawControl ( true ) ;
         }
         else
            cp->bIndex = cp->wPtr->ScrollData ( info.wk.key ) ;
      }

      else if ( info.wk.type == wktFUNKEY && info.wk.key == nckESC )   // abort selection
      {
         //* If control is expanded i.e. an edit is in progress *
         if ( cp->expanded != false )
         {
            #if CAPTURE_DROPDOWN != 0        // Debugging Only: capture screenshot
            //* Used only to capture screenshots of expanded *
            //* Dropdown objects for documentation purposes. *
            if ( info.wk.key == nckESC )
            {  //* Capture the entire dialog *
               this->CaptureDialog ( "./captureDD.txt" ) ;
               this->CaptureDialog ( "./captureDD.html", true, false, 
                                     "infodoc-styles.css", 4, false, nc.blR ) ;
            }
            #endif // CAPTURE_DROPDOWN != 0

            //* return highlight to initial state *
            wchar_t sKey = cp->bIndex > origSelection ? nckUP : nckDOWN ;
            while ( cp->bIndex != origSelection )
               cp->bIndex = cp->wPtr->ScrollData ( sKey ) ;
            cp->expanded = false ;        // Collapse the control
            this->RefreshWin () ;
            if ( ExternalControlUpdate != NULL )   // in case caller is tracking the scroll
               ExternalControlUpdate ( this->currCtrl, info.wk, false ) ;
         }
         info.ctrlType = dctDROPDOWN ;       // 
         info.ctrlIndex = this->currCtrl ;   // control with input focus
         info.hasFocus = true ;              // focus was not lost
         info.keyIn = nckTAB ;               // focus to move forward to next control
         info.dataMod = false ;              // data unchanged
         info.selMember = cp->bIndex ;       // current member is the 'selected' member
         info.isSel = false ;                // don't care
         info.viaHotkey = false ;            // control did not get focus via hotkey
         done = true ;                       // returning to caller
      }

      else     // test for a possible hotkey
      {
         //* Scan the controls for one whose hotkey matches the input.*
         //* If match found, returns control's index, else -1.        *
         short hotIndex = this->IsHotkey ( info.wk ) ;
         if ( hotIndex >= ZERO )
         {
            //* Ignore hotkey selection of this control, *
            //* unless currently in collapsed state.     *
            if ( hotIndex == this->currCtrl && cp->expanded == false )
            {
               this->CaptureDialogDisplayData () ;
               cp->expanded = true ;
               cp->RedrawControl ( true ) ;
            }
            //Else, make the indicated control the current/active control.*
            else if ( hotIndex != this->currCtrl )
            {
               //* If control is expanded, collapse it *
               if ( cp->expanded != false )
               {
                  cp->expanded = false ;        // Collapse the control
                  this->RefreshWin () ;
               }
               //* Indicate the results of our edit *
               info.ctrlType = dctDROPDOWN ;    // 
               info.ctrlIndex = this->currCtrl ;// control with input focus
               info.hasFocus = true ;           // this may be reset by ChangedFocusViaHotkey()
               info.keyIn = ZERO ;              // indicate no shift in focus necessary
               info.dataMod = true ;            // indicate selection change (but see below)
               info.selMember = cp->bIndex ;    // current member is the 'selected' member
               info.isSel = false ;             // don't care
               info.viaHotkey = false ;         // this may be set by ChangedFocusViaHotkey()
               this->ChangedFocusViaHotkey ( info.wk.key, hotIndex, info ) ;
               done = true ;
            }
         }
         else
         {
            // key input value is not valid in this context, ignore it
         }
      }
      //* If caller has specified a callback method, do it now.*
      if ( ExternalControlUpdate != NULL )
         ExternalControlUpdate ( this->currCtrl, info.wk, false ) ;
   }     // while()

   //* If 'selected' item has not changed, reset dataMod flag *
   if ( cp->bIndex == origSelection )
      info.dataMod = false ;
   return this->currCtrl;

}  //* End EditDropdown() *

//*************************
//*   GetDropdownSelect   *
//*************************
//******************************************************************************
//* Returns the index of the highlighted item in the specified dctDROPDOWN     *
//* control.                                                                   *
//*                                                                            *
//* Input  : cIndex : index number of source control                           *
//*                                                                            *
//* Returns: index of 'selected' (highlighted) item                            *
//*          or ERR if:                                                        *
//*            a) specified control is not of type dctDROPDOWN                 *
//******************************************************************************

short NcDialog::GetDropdownSelect ( short cIndex )
{
   short itemIndex = ERR ;

   //* If caller has passed a valid index number *
   if ( (cIndex >= ZERO) && (cIndex <= this->lastCtrl)
        && (dCtrl[cIndex]->type == dctDROPDOWN) )
   {
      DialogDropdown* cp = (DialogDropdown*)(dCtrl[cIndex]) ;
      itemIndex = cp->bIndex ;
   }
   return itemIndex ;

}  //* End GetDropdownSelect() *

//*************************
//*   SetDropdownSelect   *
//*************************
//******************************************************************************
//* Set 'selected' (highlighted) item in specified dctDROPDOWN control.        *
//*                                                                            *
//* Input  : cIndex : index number of target control                           *
//*          selMember : index of item to be set as 'selected'                 *
//*                                                                            *
//* Returns: index of 'selected' member                                        *
//*          or ERR if:                                                        *
//*            a) invalid item index specified                                 *
//*            b) specified control currently has the input focus              *
//*            c) specified control is not of type dctDROPDOWN                 *
//******************************************************************************

short NcDialog::SetDropdownSelect ( short cIndex, short selMember )
{
   short itemIndex = ERR ;

   //* If caller has passed a valid index number *
   if ( (cIndex >= ZERO) && (cIndex <= this->lastCtrl)
        && (dCtrl[cIndex]->type == dctDROPDOWN)
        && (cIndex != this->currCtrl) )
   {
      DialogDropdown* cp = (DialogDropdown*)(dCtrl[cIndex]) ;

      //* If a non-highlighted item has been specified for selection *
      //* move highlight to that position.                           *
      if ( (selMember >= ZERO) && (selMember < cp->bItems) 
           && (cp->expanded == false) )
      {
         cp->bIndex = selMember ;   // set the highlight
         cp->sdPaint = false ;      // force redraw next time control is expanded
         cp->RedrawControl ( this->currCtrl == cIndex ? true : false ) ;

         //* Set currently-highlighted item as selection *
         itemIndex = cp->bIndex ;
      }
   }
   return itemIndex ;

}  //* End SetDropdownSelect() *

//*************************
//*                       *
//*************************
//******************************************************************************
//*                                                                            *
//*                                                                            *
//*                                                                            *
//* Input  :                                                                   *
//*                                                                            *
//* Returns:                                                                   *
//******************************************************************************


