//********************************************************************************
//* File       : NcdControlBB.cpp                                                *
//* Author     : Mahlon R. Smith                                                 *
//*              Copyright (c) 2006-2025 Mahlon R. Smith, The Software Samurai   *
//*                  GNU GPL copyright notice located in NcDialog.hpp            *
//* Date       : 20-Mar-2025                                                     *
//* Version    : (see NcDialogVersion string in NcDialog.cpp)                    *
//*                                                                              *
//* Description: Contains the methods of the DialogBillboard class               *
//*              and the public NcDialog-class methods for accessing             *
//*              the DialogBillboard object's functionality.                     *
//*              See NcDialog.hpp for the definition of this class.              *
//*                                                                              *
//*                                                                              *
//* Development Tools: See NcDialog.cpp.                                         *
//********************************************************************************
//* Version History (most recent first):                                         *
//*   See version history in NcDialog.cpp.                                       *
//********************************************************************************
//* Programmer's Notes:                                                          *
//*                                                                              *
//*                                                                              *
//*                                                                              *
//*                                                                              *
//*                                                                              *
//********************************************************************************

//*****************
//* Include Files *
//*****************

#include "GlobalDef.hpp"               //* General definitions

#ifndef NCURSES_INCLUDED
#include "NCurses.hpp"
#endif

#ifndef NCDIALOG_INCLUDED
#include "NcDialog.hpp"
#endif

//******************************
//* Local Definitions and Data *
//******************************


      //*************************************************
      //** THIS SECTION IMPLEMENTS THE METHODS OF THE  **
      //**           DialogBillBoard CLASS.            **
      //** - - - - - - - - - - - - - - - - - - - - - - **
      //** See below for public NcDialog-class methods **
      //**   related to DialogBillboard-class access.  **
      //*************************************************

//*************************
//*    DialogBillboard    *
//*************************
//******************************************************************************
//* Constructor for DialogBillboard class object.                              *
//*                                                                            *
//*                                                                            *
//* Input  : pointer to initialization structure                               *
//*                                                                            *
//* Returns: constructors implicitly return a pointer to the object            *
//******************************************************************************

DialogBillboard::DialogBillboard ( InitCtrl* iPtr )
{
   //* Transfer the initialization data *
   this->type   = dctBILLBOARD ;    // set control type
   this->ulY    = iPtr->ulY ;       // control's offset within the dialog
   this->ulX    = iPtr->ulX ;
   this->lines  = iPtr->lines ;     // Display lines
   if ( iPtr->cols > MAX_DIALOG_WIDTH ) // (this will prevent embarrassing)
      iPtr->cols = MAX_DIALOG_WIDTH ;   // (buffer overrun                )
   this->cols   = iPtr->cols ;      // Diaplay columns
   this->nColor = iPtr->nColor ;    // color of text when control does not have focus
   this->fColor = iPtr->fColor ;    // color of text when control has focus
   this->altAttr = false ;          // assume no alternate colors specified
   this->sclAttr = false ;          // by default, color attributes are NOT scrolled
   this->labY   = iPtr->labY ;      // label offsets
   this->labX   = iPtr->labX ;
   this->active = iPtr->active ;    // can control receive input focus?

   //* Copy the label text to data member, wLabel.       *
   //* Note that Billboard controls may not have hotkeys.*
   gString gs( iPtr->label ) ;
   short hi ;
   if ( (hi = gs.find( L'^' )) >= ZERO )
   {  //* Strip the hotkey indicator *
      if ( hi == ZERO )
         gs.shiftChars( -1 ) ;
      else
      {
         gString gstmp( gs.gstr(), hi ) ;
         gs.shiftChars( -(hi + 1) ) ;
         gs.insert( gstmp.gstr() ) ;
      }
   }
   this->InitHotkey ( gs.ustr(), false ) ;

   //* Allocate storage for maximum number of 'wide' display characters *
   this->mlText = new multiText[this->lines+1] ;
   this->mlAttr = new attr_t[this->lines+1] ;

   //* Character capacity of the display area (incl. NULLCHARS) *
   this->maxWChars = this->lines * this->cols + this->lines ;

   //* Initialize display data (if any). Filter, format and store the display  *
   //* text (and optionally, color attributes for each line).                  *
   //* NOTE: Assumes that control DOES NOT have the input focus on             *
   //*       instantiation. This is not a problem because caller will draw the *
   //*       control for the first time with the appropriate color.            *
   if ( iPtr->dispText != NULL )
      this->bbSetText ( iPtr->dispText, false, iPtr->scrColor ) ;
   else
      this->bbClear () ;

   //* Default values for all other data members *
   this->groupCode = ZERO ;   // Billboard controls are not grouped (not used)
   this->rtlContent = false ; // left-to-right language content by default

   //* Instantiate the underlying window object *
   this->wPtr = new NcWindow ( this->lines, this->cols, this->ulY, this->ulX ) ;
   
}  //* End DialogBillboard() *

//*************************
//*   ~DialogBillboard    *
//*************************
//******************************************************************************
//* Destructor for DialogBillboard class object.                               *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

DialogBillboard::~DialogBillboard ( void )
{
   //* Delete dynamically-allocated data *
   if ( this->mlText != NULL )
      delete [] this->mlText ;
   if ( this->mlAttr != NULL )
      delete [] this->mlAttr ;

   //* Close the underlying window *
   delete this->wPtr ;

}  //* End ~DialogBillboard() *

//*************************
//*     OpenControl       *
//*************************
//******************************************************************************
//* Open the window previously instantiated by the constructor, then draw the  *
//* display data to the screen.                                                *
//*                                                                            *
//* Input  : hasFocus : (optional, false by default), determines               *
//*                     whether to draw the object in nColor or fColor         *
//*                                                                            *
//* Returns: OK if successful, else ERR                                        *
//******************************************************************************

short DialogBillboard::OpenControl ( bool hasFocus )
{
short    result ;

   //* Initialize the control's underlying window *
   if ( (result = this->wPtr->OpenWindow ()) == OK )
   {  //* Draw control on the screen with text data (if any) *
      this->RedrawControl ( hasFocus ) ;
   }
   return result ;

}  //* End OpenControl() *

//*************************
//*     RedrawControl     *
//*************************
//******************************************************************************
//* Redraw the data for the control.                                           *
//* - Note that if data are wider than control width, the data output will be  *
//*   truncated at right edge of control.                                      *
//*                                                                            *
//*                (refreshes control window's display)                        *
//*                                                                            *
//* Input  : hasFocus: determines color attribute of background                *
//*                    if true,  control has input focus so use fColor         *
//*                    if false, control does not have focus so use nColor     *
//*                    NOTE: If alternate colors defined for indivitual lines, *
//*                          this overrides the 'hasFocus' parameter.          *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************
//* Programmer's Note: There is a design decision about the color attribute:   *
//* - On entry, the background color of the display area is cleared to the     *
//*   focus/non-focus color.                                                   *
//* - If non-default colors have been specified for the display lines i.e.     *
//*   this->altAttr != false, then we must decide:                             *
//*   1) Draw ONLY the display text in the specified color, leaving the unused *
//*      portion of the line in the background color.                          *
//*      OR                                                                    *
//*   2) Set the entire line to the specified color attribute.                 *
//* - Note that this is not an issue if the background attribute of the text   *
//*   is the same as the background attribute of the control.                  *
//* - It only becomes an issue if the backgrounds are different.               *
//*                                                                            *
//* Our view is that having different background attributes for different      *
//* sections of the line looks unprofessional; therefore we take the extra     *
//* step to draw the entire line in the text color.                            *
//* The implementation is not straightforward because the X offset returned by *
//* the 'WriteString' and 'WriteChar' methods is limited to (cols-1).          *
//*                                                                            *
//******************************************************************************

void DialogBillboard::RedrawControl ( bool hasFocus )
{

   this->RedrawControl ( hasFocus, true ) ;

}  //* End RedrawControl()

//*************************
//*     RedrawControl     *
//*************************
//******************************************************************************
//* Redraw the data for the control and optionally refresh the display.        *
//* Private method, called only by DialogBillboard-class methods.              *
//*                                                                            *
//* Input  : hasFocus: determines color attribute of background                *
//*                    if true,  control has input focus so use fColor         *
//*                    if false, control does not have focus so use nColor     *
//*                    NOTE: If alternate colors defined for indivitual lines, *
//*                          this overrides the 'hasFocus' parameter.          *
//*          refresh : if 'true', then refresh display                         *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void DialogBillboard::RedrawControl ( bool hasFocus, bool refresh )
{
   attr_t dColor = hasFocus ? this->fColor : this->nColor ;
   //* If alternate colors not specified, use focus/non-focus color *
   if ( this->altAttr == false )
   {
      for ( short lIndex = ZERO ; lIndex < this->lines ; lIndex++ )
         this->mlAttr[lIndex] = dColor ;
   }

   //* Set control's interior color and set the background to that color.*
   this->wPtr->SetInteriorColor ( dColor ) ;
   this->wPtr->ClearWin () ;

   //* Draw the text data *
   winPos wp ;
   short x = this->rtlContent ? this->cols - 1 : ZERO ;
   for ( short lIndex = ZERO ; lIndex < this->lines ; lIndex++ )
   {
      wp = this->wPtr->WriteString ( lIndex, x, this->mlText[lIndex].ln, 
                                     this->mlAttr[lIndex], false, this->rtlContent ) ;
      if ( this->mlText[lIndex].lnCols > ZERO && 
           this->mlText[lIndex].lnCols < this->cols )
      {
         for ( short i = this->mlText[lIndex].lnCols ; i < this->cols ; i++ )
         {
            wp.ypos = lIndex ;
            wp = this->wPtr->WriteChar ( wp, L' ', this->mlAttr[lIndex],
                                         false, this->rtlContent ) ;
         }
      }
   }
   if ( refresh != false )
      this->wPtr->RefreshWin () ;              // update the display

}  //* End RedrawControl()

//*************************
//*    SetOutputFormat    *
//*************************
//******************************************************************************
//* Set the text data output format: RTL (right-to-left), or                   *
//* LTR (left-to-right). Does not refresh display.                             *
//* Note that the default (unmodified) output format is LTR.                   *
//*                                                                            *
//* Input  : rtlFormat : if 'true',  then set as RTL format                    *
//*                      if 'false', then set as LTR format                    *
//*                                                                            *
//* Returns: OK  if successful                                                 *
//*          ERR if data format cannot be changed (currently not used)         *
//******************************************************************************
//* Programmer's Note: When redrawing the control's contents, we assume that   *
//* control does not have focus. This is reasonable because Billboard controls *
//* are read-only.                                                             *
//******************************************************************************

short DialogBillboard::SetOutputFormat ( bool rtlFormat )
{
   this->rtlContent = rtlFormat ;
   this->RedrawControl ( false, false ) ;
   return OK ;

}  //* End SetOutputFormat() *

//*************************
//*       bbSetText       *
//*************************
//******************************************************************************
//* Replace current display data, if any, with specified data.                 *
//*                                                                            *
//* Input  : srcText   : new display data                                      *
//*                      gString object, wchar_t array, or UTF-8 array         *
//*          hasFocus  : 'true' if control currently has the input focus       *
//*                      else, 'false'                                         *
//*          srcAttr   : (optional, NULL pointer by default)  if specified,    *
//*                      use the data to initialize this->mlAttr[]             *
//*                                                                            *
//* Returns: for wchar_t data: index of first unconsummed source character     *
//*          for UTF-8 data  : index of first unconsummed source byte          *
//*                   DOES NOT REFRESH THE DISPLAY!                            *
//******************************************************************************

short DialogBillboard::bbSetText ( const wchar_t* srcText, 
                                   bool hasFocus, const attr_t* srcAttr )
{
   //* Erase all existing text and reinitialize the color array *
   this->bbClear ( true, srcAttr ) ;

   short wChars = ZERO ;         // total source characters
   while ( srcText[wChars++] != NULLCHAR ) ;

   gString gs ;                  // formatted data returned here
   short lIndex = ZERO,          // display-line index
         sIndex = ZERO ;         // index into srcText (and return value)
   do
   {
      sIndex += this->bbFormatLine ( &srcText[sIndex], gs ) ;
      this->mlText[lIndex].lnChars = gs.gschars () ;
      this->mlText[lIndex].lnCols  = gs.gscols () ;
      gs.copy( this->mlText[lIndex++].ln, gs.gschars() ) ;
   }
   while ( sIndex < wChars && lIndex < this->lines ) ;

   return sIndex ;

}  //* End of bbSetText() *

short DialogBillboard::bbSetText ( const gString& srcText, 
                                   bool hasFocus, const attr_t* srcAttr )
{

   return ( this->bbSetText ( srcText.gstr(), hasFocus, srcAttr ) ) ;

}  //* End of bbSetText() *

short DialogBillboard::bbSetText ( const char* srcText, 
                                   bool hasFocus, const attr_t* srcAttr )
{
   short sIndex = ZERO,          // index into srcText (and return value)
         uBytes = ZERO ;         // total source bytes
   while ( srcText[uBytes++] != NULLCHAR ) ;

   gString gs ;                  // formatted data returned here

   //* If the entire source can be processed in a single pass.      *
   //* (Note the comparison of bytes to characters. This is       ) *
   //* (intentional as it prevents overflow of the gString object.) *
   if ( uBytes < gsALLOCDFLT )
   {
      gs = srcText ;
      this->bbSetText ( gs.gstr(), hasFocus, srcAttr ) ;
      sIndex = uBytes ; // all data processed
   }
   //* Else, parse the source data one line at a time *
   else
   {
      //* Erase all existing text and reinitialize the color array *
      this->bbClear ( true, srcAttr ) ;

      wchar_t wBuff[gsALLOCDFLT] ;  // unformatted 'wide' source data
      short lIndex = ZERO,          // display-line index
            bStrip ;                // number of non-printing characters discarded
      do
      {
         gs = &srcText[sIndex] ;
         gs.copy( wBuff, gs.gschars() ) ;
         this->bbFormatLine ( wBuff, gs, &bStrip ) ;
         this->mlText[lIndex].lnChars = gs.gschars () ;
         this->mlText[lIndex].lnCols  = gs.gscols () ;
         gs.copy( this->mlText[lIndex++].ln, gs.gschars() ) ;
         //* Advance the source index by the number of source bytes  *
         //* written to target PLUS non-printing characters stripped.*
         sIndex += gs.utfbytes() - 1 + bStrip ;
         if ( sIndex < uBytes && 
             (srcText[sIndex] == nckNEWLINE || srcText[sIndex] == NULLCHAR ||
// NOTE: Potential bug here. We might accidentally step over a SPACE character
//       that should be printed at the beginning of the next line.
//       Not sure how to test for it...
              srcText[sIndex] == nckSPACE) )
            ++sIndex ;
      }
      while ( sIndex < uBytes && lIndex < this->lines ) ;
   }
   return sIndex ;

}  //* End of bbSetText() *

//*************************
//*     bbFormatLine      *
//*************************
//******************************************************************************
//* Extract a line of text from the source data.                               *
//* 1) remove non-printing characters                                          *
//* 2) size the data to fit within the control's display width                 *
//* 3) perform logical line break (at SPACE or DASH) if necessary.             *
//*                                                                            *
//* Input  : srcText   : unformatted source data                               *
//*          gs        : (by reference, initial contents ignored)              *
//*                      receives formatted data for one display line          *
//*          bStrip    : (optional, NULL pointer by default)                   *
//*                      if specified, receives the count of non-printing      *
//*                      characters stripped, NOT including delimiter chars    *
//*                      (useful only if original text was UTF-8 data)         *
//*                                                                            *
//* Returns: index of first unconsummed source character                       *
//******************************************************************************

short DialogBillboard::bbFormatLine ( const wchar_t* srcText, gString& gs,
                                      short* bStrip )
{
   wchar_t wBuff[gsALLOCDFLT] ;  // work buffer
   short sIndex = ZERO,          // index into source data array
         halfCol = this->cols / 2,// inflection point for possible overflow
         cCount = ZERO,          // columns of data processed
         sChars = ZERO,          // total source characters
         wIndex = ZERO,          // index into work buffer
         recentSp = ZERO,        // index of potential breakpoint (wBuff)
         recentSi = ZERO ;       // index of potential breakpoint (srcText)
                                 // (does not include delimiter characters)
   bool delimChar = false ;      // 'true' if delimiter character found
   if ( bStrip != NULL )   *bStrip = ZERO ;  // initialize counter

   while ( srcText[sChars++] != NULLCHAR ) ; // count the source characters

   //* Isolate a raw data line *
   while ( srcText[sIndex] != nckNEWLINE && 
           srcText[sIndex] != NULLCHAR && (cCount < this->cols) )
   {
      //* Remember potential line-breakpoint *
      if ( srcText[sIndex] == nckSPACE || srcText[sIndex] == DASH )
      {
         if ( srcText[sIndex] == nckSPACE )
            recentSp = wIndex ;        // target breakpoint (space)
         else
            recentSp = wIndex + 1 ;    // target breakpoint (dash)
         recentSi = sIndex + 1 ;       // source breakpoint
      }
      if ( srcText[sIndex] >= nckSPACE )
         wBuff[wIndex++] = srcText[sIndex++] ;
      else        // step over non-printing control character
      {
         if ( bStrip != NULL )
            ++(*bStrip) ;
         ++sIndex ;
      }

      //* For multi-column source characters, this is the point at which *
      //* the column count COULD exceed control's width.                 *
      if ( wIndex >= halfCol )
      {
         wBuff[wIndex] = NULLCHAR ;
         gs = wBuff ;
         if ( (cCount = gs.gscols()) >= this->cols ) // if column limit reached
            break ;
      }
   }
   wBuff[wIndex] = NULLCHAR ;

   //* If loop terminated because a line delimiter was found *
   if ( srcText[sIndex] == nckNEWLINE || srcText[sIndex] == NULLCHAR )
   {
      delimChar = true ;
      ++sIndex ;     // step over the delimiter
      gs = wBuff ;
      cCount = gs.gscols() ;
   }

   //* If the data fills or overflows the width of the control *
   if ( cCount >= this->cols )
   {
      //* We may have exact, end-of-word alignment to control width.*
      wchar_t lastChar = wBuff[wIndex-1], 
              nextChar = sIndex < sChars ? srcText[sIndex] : nckSPACE ;
      if ( (cCount == this->cols) && 
           (delimChar != false || lastChar == nckSPACE || 
            lastChar == DASH || nextChar == nckSPACE) )
         ;  // nothing to do

      //* We have either alignment, but not on a natural *
      //* breakpoint, OR too many columns for the line.  *
      else
      {  //* If a natural breakpoint (SPACE OR DASH) *
         //* previously identified, return to it.    *
         if ( recentSp > ZERO )
         {
            gs.limitChars ( recentSp ) ;
            sIndex = recentSi ;
         }
         //* Else, no breakpoint identified, so just *
         //* be sure we don't lose a character.      *
         else if ( cCount > this->cols )
         {
            gs.limitCols( this->cols ) ;
            --sIndex ;
         }
      }
   }
   return sIndex ;

}  //* End bbFormatLine() *

//*************************
//*    bbNextFreeLine     *
//*************************
//******************************************************************************
//* Locate the first unoccupied display line in the control.                   *
//* If all lines are currently occupied, returns this->lines                   *
//*                                                                            *
//* Input  : none      : for private version                                   *
//*          ctrlLines : for call from parent dialog                           *
//*                      (by reference, initial value ignored)                 *
//*                      receives number of display lines in control           *
//*                                                                            *
//* Returns: index of first empty display line                                 *
//*          CAUTION! : If all lines are occupied, returns this->lines         *
//*                                                                            *
//******************************************************************************

short DialogBillboard::bbNextFreeLine ( void )  // private version
{
   short lIndex = ZERO ;
   for ( ; lIndex < this->lines ; lIndex++ )
   {
      if ( *this->mlText[lIndex].ln == NULLCHAR )
         break ;
   }
   return lIndex ;

}  //* End bbNextFreeLine() *

short DialogBillboard::bbNextFreeLine ( short& ctrlLines ) // semi-public version
{
   ctrlLines = this->lines ;
   return ( this->bbNextFreeLine () ) ;

}  //* End bbNextFreeLine() *

//*************************
//*      bbAddText        *
//*************************
//******************************************************************************
//* Append one line of text to the end of existing data, or replace/modify the *
//* data of specified display line.                                            *
//*                                                                            *
//* Input  : srcText  : data to be added                                       *
//*                     - if 'rowIndex' not specified:                         *
//*                       insert the new data at the first unoccupied display  *
//*                       line. If all lines already contain data, then scroll *
//*                       all data up by one line (discarding the first line)  *
//*                       and then add the new data as the last line.          *
//*                       If data are too wide to be displayed, then the       *
//*                       excess will be TRUNCATED to fit on the line.         *
//*                     - if 'rowIndex' specified, see below                   *
//*          byteOffset: (by reference, initial value ignored)                 *
//*                      receives BYTE offset of the first unconsummed         *
//*                      character (useful only if original text was UTF-8)    *
//*          hasFocus : 'true' if control currently has the input focus        *
//*                     else, 'false'                                          *
//*          srcAttr: (optional, attrDFLT by default)                          *
//*                   if specified, text will be written using the specified   *
//*                   color attribute. Options are: a member of enum dtbmColors*
//*                   or one of the color attributes defined in NCurses.hpp.   *
//*          rowIndex : (optional, (-1) by default)                            *
//*                     - if a valid display line index, then replace or append*
//*                       to the existing data (if any)                        *
//*                     - if not a valid line index, then ignored              *
//*          append   : (optional, false by default)                           *
//*                     if 'true',  then append new text to existing text      *
//*                                 (if any), shifting the data to the left    *
//*                                 as necessary to fully display the new text *
//*                                 on the specified display line              *
//*                     if 'false', then replace existing text (if any),       *
//*                                 TRUNCATING the result if necessary to fit  *
//*                                 on the specified display line              *
//*                                                                            *
//* Returns: index of first unconsummed source character                       *
//******************************************************************************
//* Programmer's Note: If 'append' specified AND if new data is too wide for   *
//* the field, i.e. an application error, then new data will be truncated to   *
//* fit the field BUT subsequent calls on the same data stream may be          *
//* incorrectly indexed due to a potentially-corrupted return value.           *
//******************************************************************************

short DialogBillboard::bbAddText ( const wchar_t* srcText, short& byteOffset, 
                                   bool hasFocus, attr_t srcAttr, 
                                   short rowIndex, bool append )
{
   short wIndex = ZERO,       // return value
         lIndex = ZERO ;      // index of target line

   //* Locate the target display line *
   if ( rowIndex >= ZERO && rowIndex < this->lines )
      lIndex = rowIndex ;
   else
   {
      //* Find the first empty line, and if all lines       *
      //* contain text, then scroll the data up by one line.*
      lIndex = this->bbNextFreeLine () ;
      if ( lIndex == this->lines )
      {
         for ( lIndex = 1 ; lIndex < this->lines ; lIndex++ )
         {
            this->mlText[lIndex - 1] = this->mlText[lIndex] ;
            if ( this->sclAttr )
               this->mlAttr[lIndex - 1] = this->mlAttr[lIndex] ;
         }
         --lIndex ;           // index the last line
      }
   }

   //* Set color attribute for target line *
   this->bbSetColor ( lIndex, srcAttr, hasFocus ) ;

   //* Isolate the display data for the line *
   gString newgs ;
   short bStrip ;
   wIndex = this->bbFormatLine ( srcText, newgs, &bStrip ) ;
   byteOffset = newgs.utfbytes() + bStrip ;

   //* If replacing existing text data for target line *
   if ( append == false || newgs.gscols() == this->cols )
   {
      newgs.copy( this->mlText[lIndex].ln, mtLENGTH ) ;
      this->mlText[lIndex].lnChars = newgs.gschars() ;
      this->mlText[lIndex].lnCols  = newgs.gscols() ;
   }
   //* If appending new text to existing text for this line *
   else
   {
      gString oldgs( this->mlText[lIndex].ln ) ;
      short oldCols = oldgs.gscols(),        // columns of existing text
            newCols = newgs.gscols(),        // columns of new text
            totCols = oldCols + newCols,     // total columns
            freCols = this->cols - totCols ; // free columns
      if ( freCols < ZERO )   // discard head of existing text data
         oldgs.shiftCols ( freCols ) ;
      oldgs.append( newgs.gstr() ) ;
      if ( newgs.gscols() > this->cols )
      {  //* Adjust byteOffset to reflect the characters actually displayed.*
         // (see note above about truncating 'new' text)
         short oldBytes = oldgs.utfbytes() ;
         //* Truncate the output to fit control width *
         oldgs.limitCols( this->cols ) ;
         byteOffset -= oldBytes - oldgs.utfbytes() ;
      }
      oldgs.copy( this->mlText[lIndex].ln, mtLENGTH ) ;
      this->mlText[lIndex].lnChars = oldgs.gschars() ;
      this->mlText[lIndex].lnCols  = oldgs.gscols() ;
   }
   return wIndex ;

}  //* End bbAddText() *

//*************************
//*       bbGetText       *
//*************************
//******************************************************************************
//* Returns a copy of the text currently displayed on the specified line.      *
//*                                                                            *
//* Input  : rowText  : (by reference) receives text object for specified line *
//*          rowIndex : index of source display line                           *
//*                                                                            *
//* Returns: OK if successful, or ERR if invalid index                         *
//******************************************************************************

short DialogBillboard::bbGetText ( multiText* rowText, short rowIndex )
{
   short status = ERR ;
   if ( rowIndex >= ZERO && rowIndex < this->lines )
   {
      *rowText = this->mlText[rowIndex] ;
      status = OK ;
   }
   else
      rowText->reset() ;

   return status ;

}  //* End bbGetText() *

//*************************
//*        bbClear        *
//*************************
//******************************************************************************
//* Erase display text data. Optionally, erase/modify 'alternate' color        *
//* attributes.                                                                *
//*                                                                            *
//* Input  : clrAll : (optional, true by default)                              *
//*                   if 'true', then any previously-set custom color          *
//*                   attributes are discarded                                 *
//*          srcAttr : (optional, NULL pointer by default)                     *
//*                    if specified, points to an array of color attributes,   *
//*                    one for each display line defined for the control       *
//*                                (overrides 'clrAll')                        *
//* Returns: OK                                                                *
//******************************************************************************

short DialogBillboard::bbClear ( bool clearAll, const attr_t* srcAttr )
{
   for ( short lIndex = ZERO ; lIndex < this->lines ; lIndex++ )
   {
      this->mlText[lIndex].reset() ;
      if ( srcAttr != NULL )
         this->mlAttr[lIndex] = srcAttr[lIndex] ;
      else if ( clearAll != false )   // unnecessary, but tidy
         this->mlAttr[lIndex] = this->nColor ;
      else
         ;  // existing attributes retained
   }
   if ( srcAttr != NULL )
      this->altAttr = true ;
   else if ( clearAll != false )
      this->altAttr = false ;
   return OK ;

}  //* End bbClear() *

//*************************
//*     bbScrollAttr      *
//*************************
//******************************************************************************
//* Enable/disable scrolling of color attributes along with scrolling text.    *
//* Color-attribute scrolling is disabled by default.                          *
//*                                                                            *
//* Input  : scroll : if 'true', when text data are scrolled upward or         *
//*                      downward in the window, the color attribute associated*
//*                      with that text is scrolled along with the text.       *
//*                   if 'false', when text data are scrolled upward or        *
//*                      downward in the window, the color attribute for each  *
//*                      line remains unchanged.                               *
//*                      Exception: If the text-insertion command specifies a  *
//*                      non-default color attribute, then the insertion line  *
//*                      is set to the specified value.                        *
//*                                                                            *
//* Returns: OK                                                                *
//******************************************************************************

short DialogBillboard::bbScrollAttr ( bool scroll )
{
   
   this->sclAttr = scroll ;
   return OK ;

}  //* End bbScrollAttr() *

//*************************
//*     bbScrollAttr      *
//*************************
//******************************************************************************
//* Returns the current state of the 'sclAttr' member.                         *
//* Called ONLY by NcDialog methods, specifically Insert2Billboard().          *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: current state of the 'sclAttr' flag                               *
//******************************************************************************

bool DialogBillboard::bbScrollAttr ( void )
{

   return ( this->sclAttr ) ;

}  //* End bbScrollAttr() *

//*************************
//*      bbGetColor       *
//*************************
//******************************************************************************
//* Returns a pointer to the list of color attributes, one for each display    *
//* line.                                                                      *
//*                                                                            *
//* Input  : lineCount: (by reference, initial value ignored)                  *
//*                     receives the number of items in the referenced array   *
//*                                                                            *
//* Returns: const pointer to the color-attribute array                        *
//******************************************************************************

const attr_t* DialogBillboard::bbGetColor ( short& lineCount )
{
   
   lineCount = this->lines ;
   return (const attr_t*)this->mlAttr ;

}  //* End bbGetColor() *

//*************************
//*      bbSetColor       *
//*************************
//******************************************************************************
//* Set alternate color attributes for all display lines.                      *
//*                                                                            *
//* Input  : attrList : (optional, NULL pointer by default)                    *
//*                     array of color attribute values, one for each line     *
//*          hasFocus : 'true' if control has input focus, else 'false'        *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void DialogBillboard::bbSetColor ( const attr_t* attrList, bool hasFocus ) 
{
   attr_t dColor = hasFocus ? this->fColor : this->nColor ;
   for ( short lIndex = ZERO ; lIndex < this->lines ; lIndex++ )
   {
      if ( attrList != NULL )
      {
         if ( attrList[lIndex] == attrDFLT )
            this->mlAttr[lIndex] = dColor ;
         else if ( attrList[lIndex] == dtbmFcolor )
            this->mlAttr[lIndex] = this->fColor ;
         else if ( attrList[lIndex] == dtbmNFcolor )
            this->mlAttr[lIndex] = this->nColor ;
         else
            this->mlAttr[lIndex] = attrList[lIndex] ; ;
      }
      else
         this->mlAttr[lIndex] = dColor ;
   }
   this->altAttr = (attrList == NULL) ? false : true ;

}  //* End bbSetColor() *

//*************************
//*      bbSetColor       *
//*************************
//******************************************************************************
//* Set alternate color attribute for specified display line.                  *
//* - If an alternate color list has already been established, then we simply  *
//*   replace the existing value for target line with the new value            *
//* - If default attributes are currently in use, then establish an alternate  *
//*   color-attribute list using existing list and modify the target line color*
//*                                                                            *
//* Input  : trgLine  : index of display line whose color is to be modified    *
//*                      if trgLine is not a valid line index, then            *
//*                      do nothing                                            *
//*          trgAttr  : new color attribute                                    *
//*                      if trgAttr == attrDFLT, then do nothing               *
//*          hasFocus : [currently unused]                                     *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void DialogBillboard::bbSetColor ( short trgLine, attr_t trgAttr, bool hasFocus )
{
   if ( trgAttr != attrDFLT && (trgLine >= ZERO && trgLine < this->lines) )
   {
      attr_t attrib = attrDFLT ;
      if ( trgAttr == dtbmFcolor )
         attrib = this->fColor ;
      else if ( trgAttr == dtbmNFcolor )
         attrib = this->nColor ;
      else
         attrib = trgAttr ;
      //* If new value is different from existing value, then       *
      //* update the value and declare an alternate attribute array.*
      if ( attrib != this->mlAttr[trgLine] )
      {
         this->mlAttr[trgLine] = attrib ;
         this->altAttr = true ;
      }
   }

}  //* End bbSetColor() *



      //*************************************************
      //**     THIS SECTION IMPLEMENTS THE PUBLIC,     **
      //**    NcDialog-class METHODS FOR ACCESSING     **
      //**    DialogBillboard-class FUNCTIONALITY.     **
      //*************************************************

//*************************
//*     EditBillboard     *
//*************************
//******************************************************************************
//* If control with input focus == dctBILLBOARD, call this method to get       *
//* user's key input. Data displayed in this control cannot be directly edited.*
//*                                                                            *
//* Currently this method is simply a stub that waits for key input indicating *
//* that the input focus is moving to another control; thus, it is recommended *
//* that dctBILLBOARD controls be instantiated as 'inactive'.                  *
//*                                                                            *
//* Input  : uiInfo class (by reference) - initial values ignored              *
//*                                                                            *
//* Returns: index of control that currently has the input focus               *
//*           (See note in NcDialog.hpp about interpretation )                 *
//*           (of values returned in the uiInfo-class object.)                 *
//******************************************************************************

short NcDialog::EditBillboard ( uiInfo& info )
{
   //* If the control with focus is actually a dctBILLBOARD control *
   if ( (this->dCtrl[this->currCtrl]->type == dctBILLBOARD) )
   {
      //* Create a pointer to the control *
      DialogBillboard* cp = (DialogBillboard*)this->dCtrl[this->currCtrl] ;

      //**************************
      //* Interact with the user *
      //**************************
      bool  done = false ;                // loop control
      do
      {
         //* 1) Get user input.                                  *
         //* 2) Check for toggle of Insert/Overstrike mode.      *
         //* 3) Check for terminal-resize event.                 *
         //* 4) Access callback method, if specified.            *
         //* 5) If input has been handled, return to top of loop.*
         this->GetKeyInput ( info.wk ) ;

         if ( (info.wk.type == wktFUNKEY)
               && (info.wk.key == nckINSERT || info.wk.key == nckRESIZE) )
         {
            if ( info.wk.key == nckINSERT )
               this->ToggleIns () ;
            else
               this->TermResized () ;

            if ( ExternalControlUpdate != NULL )
            {
               ExternalControlUpdate ( this->currCtrl, info.wk, false ) ;
               cp->wPtr->RefreshWin () ;     // update the display
            }
            continue ;
         }

         //* Transform Enter key (not used here) to Tab key. *
         if ( info.wk.type == wktFUNKEY && 
              (info.wk.key == nckENTER || info.wk.key == nckpENTER) )
            info.wk.key = nckTAB ;

         if ( (info.wk.type == wktFUNKEY) && 
              (info.wk.key == nckTAB   || info.wk.key == nckDOWN || 
               info.wk.key == nckRIGHT || info.wk.key == nckESC  ||
               info.wk.key == nckSTAB  || info.wk.key == nckUP   ||
               info.wk.key == nckLEFT) )
         {                                      // user is leaving the control
            //* Focus moving back to previous control *
            if ( info.wk.key == nckSTAB || info.wk.key == nckUP || info.wk.key == nckLEFT )
               info.keyIn = nckSTAB ;
            //* Focus moving forward to next control *
            else 
               info.keyIn = nckTAB ;
            info.ctrlType = dctBILLBOARD ;      // indicate the control type
            info.ctrlIndex = this->currCtrl ;   // control with input focus
            info.hasFocus = true ;              // focus was not lost
            info.dataMod = false ;              // data unchanged
            info.selMember = MAX_DIALOG_CONTROLS ; // don't care
            info.isSel = false ;                // don't care
            info.viaHotkey = false ;            // control did not get focus via hotkey
            done = true ;                       // exit the input loop
         }
         else
         {  //* Scan the controls for one whose hotkey matches the input.*
            //* If match found, returns control's index, else -1.        *
            short hotIndex = this->IsHotkey ( info.wk ) ;
            if ( hotIndex >= ZERO )
            {
               //* If user has selected THIS control *
               //* via hotkey, treat it an a Tab key *
               if ( hotIndex == this->currCtrl )
               {
                  info.ctrlType = dctBILLBOARD ;      // indicate the control type
                  info.ctrlIndex = this->currCtrl ;   // control with input focus
                  info.hasFocus = true ;              // focus was not lost
                  info.dataMod = false ;              // data unchanged
                  info.selMember = MAX_DIALOG_CONTROLS ; // don't care
                  info.isSel = false ;                // don't care
                  info.keyIn = nckTAB ;               // move forward
                  info.viaHotkey = false ;            // control did not get focus via hotkey
                  done = true ;                       // exit the input loop
               }
               else
               {  //* Control activated via hotkey was not this control *
                  info.ctrlType = dctBILLBOARD ;      // indicate the control type
                  info.ctrlIndex = this->currCtrl ;   // control with input focus
                  info.hasFocus = true ;// this may be reset by ChangedFocusViaHotkey()
                  info.dataMod = false ;              // data unchanged
                  info.isSel = false ;                // don't care
                  info.keyIn = ZERO ;                 // direction focus moves n/a
                  info.selMember = MAX_DIALOG_CONTROLS ; // don't care
                  info.viaHotkey = false ; // this may be set by ChangedFocusViaHotkey()
                  done = true ;                       // exit the input loop

                  //* Make the indicated control the current/active control *
                  this->ChangedFocusViaHotkey ( info.wk.key, hotIndex, info ) ;
                  // on return, new control has the focus, and the info.h_XX
                  // variables adjusted
               }
            }
            else
            {
               // key input value is not valid in this context, ignore it
            }
         }
         //* If caller has specified a callback method, do it now.*
         if ( ExternalControlUpdate != NULL )
         {
            ExternalControlUpdate ( this->currCtrl, info.wk, false ) ;
            cp->wPtr->RefreshWin () ;     // update the display
         }
      }
      while ( ! done ) ;
   }
   else
   {  //* Control is not a Billboard i.e. application error.*
      //* Do what we can to minimize the damage.            *
      info.ctrlType = this->dCtrl[this->currCtrl]->type ;
      info.ctrlIndex = this->currCtrl ;
      info.keyIn = nckTAB ;               // move on to next control
      info.hasFocus = true ;              // control has focus
      info.dataMod = false ;              // no data modified
      info.selMember = MAX_DIALOG_CONTROLS ; // don't care
      info.isSel = false ;                // don't care
      info.viaHotkey = false ;            // invalidate any hotkey data
   }

   return this->currCtrl; 

}  //* End EditBillboard() *

//*************************
//*     SetBillboard      *
//*************************
//******************************************************************************
//* Replace current display data, if any, for a dctBILLBOARD control with      *
//* specified data. The data is a single, null-terminated character array with *
//* each line's data optionally separated by an NEWLINE ('\n') character.      *
//*  NOTE: Automatic inter-word line break calculation is performed, but is    *
//*        rudimentary, so if a line needs to break in a certain place, be     *
//*        safe and indicate the breakpoint with a NEWLINE character.          *
//*                                                                            *
//* Internally, all data are handled as wchar_t type ('wide') characters, and  *
//* the limits described below refer to wchar_t storage.                       *
//* The limits are based on the amount of display area available in the target *
//* control.                                                                   *
//* 1) The text stored for, and displayed on a single line is limited by the   *
//*    number of display columns i.e. the width of the control.                *
//* 2) If source data for any line of the control is wider than the control,   *
//*    the excess will be wrapped to the next line.                            *
//* 3) If the number of delimited lines of data is greater than the number of  *
//*    display lines defined for the control, OR if undelimited text is too    *
//*    long for the control, then the excess data will not be processed.       *
//* 4) All non-printing characters (Ctrl+A - Ctrl+Z, etc.) including newline   *
//*    characters will be silently stripped from the display data.             *
//*                                                                            *
//* Input  : cIndex : index of target dctBILLBOARD control                     *
//*          uPtr   : pointer to new display data in UTF-8 format              *
//*               OR                                                           *
//*          wPtr   : pointer to new display data in wchar_t format            *
//*               OR                                                           *
//*          gsData : gString object (by reference) containing the new data    *
//*          cPtr   : (optional, NULL pointer by default)                      *
//*                   if specified, points to an array of color attributes,    *
//*                   ONE VALUE FOR EACH DISPLAY LINE defined for the control  *
//*                                                                            *
//* Returns: Note on return value (sourceIndex)                                *
//*          The value returned is the number of characters or bytes processed.*
//*          If display area is not large enough to display all the source     *
//*          data, then the value returned may be interpreted as the index of  *
//*          the first UNPROCESSED character or byte.                          *
//*            For UTF-8 source data                                           *
//*               index of first unprocessed source byte (if any)              *
//*               Example: uPtr[sourceIndex] would be the first byte of the    *
//*                        first unprocessed character.                        *
//*            For wchar_t source data                                         *
//*               index of first unprocessed source character (if any)         *
//*               Example: wPtr[sourceIndex] would be the first unprocessed    *
//*                        character.                                          *
//*            For gString object source data                                  *
//*               index of first unprocessed source character (if any)         *
//*               Example: gString::gstr()[sourceIndex] would be the first     *
//*                        unprocessed character.                              *
//*          OR ERR if specified control is not of type dctBILLBOARD           *
//******************************************************************************

short NcDialog::SetBillboard ( short cIndex, 
                               const gString& gsData, const attr_t* cPtr )
{
   short status = ERR ;
   if ( cIndex <= this->lastCtrl && dCtrl[cIndex]->type == dctBILLBOARD )
   {
      DialogBillboard* cp = (DialogBillboard*)dCtrl[cIndex] ; // pointer to control
      bool hasFocus = bool(cIndex == this->currCtrl) ;
      status = cp->bbSetText ( gsData, hasFocus, cPtr ) ;
      cp->RedrawControl ( hasFocus ) ;
   }
   return status ;

}  //* End SetBillboard() *

short NcDialog::SetBillboard ( short cIndex, 
                               const char* uPtr, const attr_t* cPtr )
{
   short status = ERR ;
   if ( cIndex <= this->lastCtrl && dCtrl[cIndex]->type == dctBILLBOARD && 
        uPtr != NULL )
   {
      DialogBillboard* cp = (DialogBillboard*)dCtrl[cIndex] ; // pointer to control
      bool hasFocus = bool(cIndex == this->currCtrl) ;
      status = cp->bbSetText ( uPtr, hasFocus, cPtr ) ;
      cp->RedrawControl ( hasFocus ) ;
   }
   return status ;

}  //* End SetBillboard() *

short NcDialog::SetBillboard ( short cIndex, 
                               const wchar_t* wPtr, const attr_t* cPtr )
{
   short status = ERR ;
   if ( cIndex <= this->lastCtrl && dCtrl[cIndex]->type == dctBILLBOARD &&
        wPtr != NULL )
   {
      DialogBillboard* cp = (DialogBillboard*)dCtrl[cIndex] ; // pointer to control
      bool hasFocus = bool(cIndex == this->currCtrl) ;
      status = cp->bbSetText ( wPtr, hasFocus, cPtr ) ;
      cp->RedrawControl ( hasFocus ) ;
   }
   return status ;

}  //* End SetBillboard() *

//*************************
//*     GetBillboard      *
//*************************
//******************************************************************************
//* Returns a copy of the specified dctBILLBOARD display data.                 *
//* The data are copied to an array of multiText-class objects, one for each   *
//* line of the control. The data returned are 'wide' (wchar_t) characters.    *
//* Example of conversion from wchar_t to UTF-8:                               *
//*   short lineCount = 5 ;                                                    *
//*   multiText billboardData[lineCount];    // wchar_t data (source)          *
//*   char utfData[lineCount][mtLENGTH * 4]; // UTF-8 data (target)            *
//*   gString gs ;                           // conversion tool                *
//*   for ( short i = ZERO ; i < lineCount ; i++ )                             *
//*   {                                                                        *
//*      gs = billboardData[i].ln ;                                            *
//*      gs.copy( utfData[i], (mtLENGTH*4) ) ;                                 *
//*   }                                                                        *
//*                                                                            *
//* Input  : cIndex : index of target dctBILLBOARD control                     *
//*          mtPtr  : pointer to an array of multiText-class objects to        *
//*                   receive the text data                                    *
//*                                                                            *
//* Returns: OK if successful,                                                 *
//*          OR ERR if specified control is not of type dctBILLBOARD           *
//******************************************************************************

short NcDialog::GetBillboard ( short cIndex, multiText* mtPtr )
{
   short status = ERR ;
   if ( cIndex <= this->lastCtrl && dCtrl[cIndex]->type == dctBILLBOARD )
   {
      DialogBillboard* cp = (DialogBillboard*)dCtrl[cIndex] ; // pointer to control

      short ctrlLines = cp->bbDisplayLines () ;
      for ( short rIndex = ZERO ; rIndex < ctrlLines ; rIndex++ )
      {
         cp->bbGetText ( &mtPtr[rIndex], rIndex ) ;
      }
      status = OK ;
   }
   return status ;

}  //* End GetBillboard() *

//*************************
//*     GetBillboard      *
//*************************
//******************************************************************************
//* Returns a copy of the display data for the specified line of a             *
//* dctBILLBOARD control.                                                      *
//*                                                                            *
//* Input  : cIndex : index of target dctBILLBOARD control                     *
//*          gsData : (by reference) receives the text data                    *
//*          lIndex : index of control line                                    *
//*                                                                            *
//* Returns: OK if successful,                                                 *
//*          OR ERR if specified control is not of type dctBILLBOARD           *
//******************************************************************************

short NcDialog::GetBillboard ( short cIndex, gString& gsData, short lIndex )
{
   short status = ERR ;
   if ( cIndex <= this->lastCtrl && dCtrl[cIndex]->type == dctBILLBOARD )
   {
      DialogBillboard* cp = (DialogBillboard*)dCtrl[cIndex] ; // pointer to control

      short ctrlLines = cp->bbDisplayLines () ;
      if ( lIndex >= ZERO && lIndex < ctrlLines )
      {
         multiText mttmp ;
         cp->bbGetText ( &mttmp, lIndex ) ;
         gsData.copy( mttmp.ln, mtLENGTH ) ;
         status = OK ;
      }
      else
         gsData.clear() ;
   }
   return status ;

}  //* End GetBillboard() *

//*************************
//*   GetBillboardIndex   *
//*************************
//******************************************************************************
//* Get the index of the next free display line. This is the line which will   *
//* by default, receive the text from the next Append2Billboard() call.        *
//* (Note that text from Insert2Billboard() is always inserted at index ZERO.  *
//*                                                                            *
//* Input  : cIndex : index of target dctBILLBOARD control                     *
//*          lIndex : (by reference) receives index of next free line          *
//*                   Range: ZERO through total-Billboard-lines-minus-one      *
//*                          -1 if all display lines occupied i.e. all lines   *
//*                          will be scrolled up and last line will become     *
//*                          the free line (insertion line).                   *
//* Returns: OK if successful,                                                 *
//*          OR ERR if specified control is not of type dctBILLBOARD           *
//******************************************************************************

short NcDialog::GetBillboardIndex ( short cIndex, short& lIndex )
{
   short status = ERR ;    // return value
   lIndex = -1 ;           // initialize caller's variable

   if ( cIndex <= this->lastCtrl && dCtrl[cIndex]->type == dctBILLBOARD )
   {
      DialogBillboard* cp = (DialogBillboard*)dCtrl[cIndex] ; // pointer to control

      short ctrlLines ;
      lIndex = cp->bbNextFreeLine ( ctrlLines ) ;
      if ( lIndex == ctrlLines )
         lIndex = -1 ;
      status = OK ;
   }
   return status ;

}

//*************************
//*   Append2Billboard    *
//*************************
//******************************************************************************
//* Append a single line of text data to the currently-displayed data (if any).*
//* If all display lines are already occupied, then scroll all lines up by one,*
//* discarding data on first line, and append the new data at the last line.   *
//*                                                                            *
//* Input  : cIndex : index of target dctBILLBOARD control                     *
//*          uPtr   : pointer to new display data in UTF-8 format              *
//*               OR                                                           *
//*          wPtr   : pointer to new display data in wchar_t format            *
//*               OR                                                           *
//*          gsData : gString object (by reference) containing the new data    *
//*          cAttr  : (optional, attrDFLT by default)                          *
//*                   if specified, text will be written using the specified   *
//*                   color attribute. Options are: a member of enum dtbmColors*
//*                   or one of the color attributes defined in NCurses.hpp.   *
//*          lIndex : (optional, (-1) by default)                              *
//*                   if specified, value is interpreted as an index of        *
//*                   the control line whose data is to be replaced by         *
//*                   the new data. If not a valid line index, then it         *
//*                   will be ignored.                                         *
//*          attach : (optional, false by default)                             *
//*                   if 'true', then                                          *
//*                      for the target line, append new text to existing text *
//*                      (if any), shifting the data to the left as necessary  *
//*                      to fully display the new text on the target display   *
//*                      line                                                  *
//*                      NOTE: If new data is too wide for the field, i.e.     *
//*                      an application error, then new data will be           *
//*                      truncated to fit the field AND subsequent calls on    *
//*                      the same data stream may be incorrectly indexed.      *
//*                   if 'false', then                                         *
//*                      replace existing text (if any) on target line,        *
//*                      TRUNCATING the result if necessary to fit on the      *
//*                      target display line                                   *
//*                                                                            *
//* Returns: number of characters or bytes processed                           *
//*           For note on return value, see 'SetBillboard' above.              *
//*          OR ERR if specified control is not of type dctBILLBOARD           *
//******************************************************************************

short NcDialog::Append2Billboard ( short cIndex, const wchar_t* wPtr, 
                                   attr_t cAttr, short lIndex, bool attach )
{
   short status = ERR ;
   if ( cIndex <= this->lastCtrl && dCtrl[cIndex]->type == dctBILLBOARD )
   {
      DialogBillboard* cp = (DialogBillboard*)dCtrl[cIndex] ; // pointer to control

      //* Append text data to end of list *
      bool hasFocus = bool(cIndex == this->currCtrl) ;
      short byteOffset ;      // (ignored for wide text data)
      status = cp->bbAddText ( wPtr, byteOffset, hasFocus, cAttr, lIndex, attach ) ;

      //* Update the display to reflect the changes *
      cp->RedrawControl ( hasFocus ) ;
   }
   return status ;

}  //* End Append2Billboard() *

short NcDialog::Append2Billboard ( short cIndex, const gString& gsData, 
                                   attr_t cAttr, short lIndex, bool attach )
{
   short status = 
      this->Append2Billboard ( cIndex, gsData.gstr(), cAttr, lIndex, attach ) ;
   return status ;

}  //* End Append2Billboard() *

short NcDialog::Append2Billboard ( short cIndex, const char* uPtr, 
                                   attr_t cAttr, short lIndex, bool attach )
{
   //* NOTE: Because the return value from the call to append the  *
   //*       text will be the CHARACTER index, not the byte index, *
   //*       we return the byte offset to caller.                  *
   short status = ERR ;    // return value
   if ( cIndex <= this->lastCtrl && dCtrl[cIndex]->type == dctBILLBOARD )
   {
      DialogBillboard* cp = (DialogBillboard*)dCtrl[cIndex] ; // pointer to control

      //* Append text data to end of list *
      bool hasFocus = bool(cIndex == this->currCtrl) ;
      gString gs( uPtr ) ;
      cp->bbAddText ( gs.gstr(), status, hasFocus, cAttr, lIndex, attach ) ;

      //* Update the display to reflect the changes *
      cp->RedrawControl ( hasFocus ) ;
   }
   return status ;

}  //* End Append2Billboard() *

//*************************
//*   Insert2Billboard    *
//*************************
//******************************************************************************
//* Insert a single line of text data at the top of the currently-displayed    *
//* data (if any). Any existing data are scrolled DOWN by one line and the new *
//* text is inserted on the first display line of the control.                 *
//*                                                                            *
//* Input  : cIndex : index of target dctBILLBOARD control                     *
//*          uPtr   : pointer to new display data in UTF-8 format              *
//*               OR                                                           *
//*          wPtr   : pointer to new display data in wchar_t format            *
//*               OR                                                           *
//*          gsData : gString object (by reference) containing the new data    *
//*          cAttr  : (optional, attrDFLT by default)                          *
//*                   if specified, text will be written using the specified   *
//*                   color attribute. Options are: a member of enum dtbmColors*
//*                   or one of the color attributes defined in NCurses.hpp.   *
//*                                                                            *
//* Returns: OK if successful                                                  *
//*          OR ERR if specified control is not of type dctBILLBOARD           *
//******************************************************************************
//* Programmer's Note:                                                         *
//* Insert2Billboard() is not fully symmetrical with Append2Billboard().       *
//* The options are more limited, it's slower, and done at a higher level.     *
//* If in future this lack of symmetry becomes a problem, we can enhance the   *
//* control for a full up/down scroll capability. 30 Jun 2013                  *
//******************************************************************************

short NcDialog::Insert2Billboard ( short cIndex, 
                                   const gString& gsData, attr_t cAttr )
{
   short status = ERR ;
   if ( cIndex <= this->lastCtrl && dCtrl[cIndex]->type == dctBILLBOARD )
   {
      DialogBillboard* cp = (DialogBillboard*)dCtrl[cIndex] ; // pointer to control

      //* Get a copy of the currently-displayed data text data.*
      short ctrlLines = cp->bbDisplayLines () ;
      multiText src[ctrlLines] ;
      this->GetBillboard ( cIndex, src ) ;
      //* Get a _local_ copy of the currently-displayed color attributes.*
      short attrCnt ;
      const attr_t* ap = this->GetBillboardColors ( cIndex, attrCnt ) ;
      attr_t srcAttr[attrCnt] ;
      for ( short i = ZERO ; i < attrCnt ; ++i )   // local copy of attributes
         srcAttr[i] = ap[i] ;
      bool attrScroll = cp->bbScrollAttr() ;

      //* Update the control's data *
      bool hasFocus = bool(cIndex == this->currCtrl) ;
      short byteOffset,       // (ignored for wide text data)
            lIndex = ZERO ;
      gString gs ;
      cp->bbAddText ( gsData.gstr(), byteOffset, hasFocus, cAttr, lIndex ) ;
      for ( short tIndex = 1 ; tIndex < ctrlLines ; tIndex++ )
      {
         cp->bbAddText ( src[lIndex].ln, byteOffset, hasFocus, 
                         attr_t(attrScroll ? srcAttr[lIndex] : attrDFLT), 
                         tIndex ) ;
         ++lIndex ;
      }
      cp->RedrawControl ( hasFocus ) ;
      status = OK ;
   }
   return status ;

}  //* End Insert2Billboard() *

short NcDialog::Insert2Billboard ( short cIndex, 
                                   const wchar_t* wPtr, attr_t cAttr )
{
   gString gs( wPtr ) ;
   return ( this->Insert2Billboard ( cIndex, gs, cAttr ) ) ;

}  //* End Insert2Billboard() *

short NcDialog::Insert2Billboard ( short cIndex, 
                                   const char* uPtr, attr_t cAttr )
{
   gString gs( uPtr ) ;
   return ( this->Insert2Billboard ( cIndex, gs, cAttr ) ) ;

}  //* End Insert2Billboard() *

//*************************
//*    ClearBillboard     *
//*************************
//******************************************************************************
//* Erase the text and optionally any 'alternate-color' data from the control. *
//*                                                                            *
//* Input  : cIndex : index of target dctBILLBOARD control                     *
//*          clearAll: (optional, 'true' by default)                           *
//*                    - if 'true',  then the color attribute for each line    *
//*                      will be set to the default color                      *
//*                    - if 'false', then any previously-specified             *
//*                      'alternate' color attributes will be retained         *
//*          cPtr    : (optional, NULL pointer by default)                     *
//*                    if specified, points to an array of color attributes,   *
//*                    ONE VALUE FOR EACH DISPLAY LINE defined for the control *
//*                    to reinitialize attribute array  (overrides 'clearAll') *
//*                                                                            *
//* Returns: OK if successful                                                  *
//*          OR ERR if specified control is not of type dctBILLBOARD           *
//******************************************************************************

short NcDialog::ClearBillboard ( short cIndex, bool clearAll, const attr_t* cPtr )
{
   short status = ERR ;
   if ( cIndex <= this->lastCtrl && dCtrl[cIndex]->type == dctBILLBOARD )
   {
      DialogBillboard* cp = (DialogBillboard*)dCtrl[cIndex] ; // pointer to control
      cp->bbClear ( clearAll, cPtr ) ;
      cp->RedrawControl ( bool(cIndex == this->currCtrl) ) ;
      status = OK ;
   }
   return status ;

}  //* End ClearBillboard() *

//*************************
//* ScrollBillboardColors *
//*************************
//******************************************************************************
//* Enable/disable scrolling of color attributes along with scrolling text.    *
//* Color-attribute scrolling is disabled by default.                          *
//*                                                                            *
//* Input  : cIndex : index of target dctBILLBOARD control                     *
//*          scroll : if 'true', when text data are scrolled upward or         *
//*                      downward in the window, the color attribute associated*
//*                      with that text is scrolled along with the text.       *
//*                   if 'false', when text data are scrolled upward or        *
//*                      downward in the window, the color attribute for each  *
//*                      line remains unchanged.                               *
//*                      Exception: If the text-insertion command specifies a  *
//*                      non-default color attribute, then the insertion line  *
//*                      (only) is set to the specified value.                 *
//*                                                                            *
//* Returns: OK if successful                                                  *
//*          OR ERR if specified control is not of type dctBILLBOARD           *
//******************************************************************************

short NcDialog::ScrollBillboardColors ( short cIndex, bool scroll )
{
   short status = ERR ;
   if ( cIndex <= this->lastCtrl && dCtrl[cIndex]->type == dctBILLBOARD )
   {
      DialogBillboard* cp = (DialogBillboard*)dCtrl[cIndex] ; // pointer to control
      cp->bbScrollAttr ( scroll ) ;
      status = OK ;
   }
   return status ;

}  //* End ScrollBillboardColors() *

//*************************
//*  GetBillboardColors   *
//*************************
//******************************************************************************
//* Returns a const pointer to the color-attribute array for the specified     *
//* dctBILLBOARD control AND the number of items in the array.                 *
//*                Values may not be modified directly.                        *
//*            Please see SetBillboardColors method, below.                    *
//*                                                                            *
//* Input  : cIndex   : index of target dctBILLBOARD control                   *
//*          lineCount: (by reference, initial value ignored)                  *
//*                     receives the number of items in the referenced array   *
//*                                                                            *
//* Returns: pointer to color-attribute array                                  *
//*          OR NULL pointer if specified control is not of type dctBILLBOARD  *
//******************************************************************************

const attr_t* NcDialog::GetBillboardColors ( short cIndex, short& lineCount )
{
   const attr_t* attrPtr = NULL ;
   if ( cIndex <= this->lastCtrl && dCtrl[cIndex]->type == dctBILLBOARD )
   {
      DialogBillboard* cp = (DialogBillboard*)dCtrl[cIndex] ; // pointer to control
      attrPtr = cp->bbGetColor ( lineCount ) ;
   }
   return attrPtr ;
   
}  //* End GetBillboardColors() *

//*************************
//*  SetBillboardColors   *
//*************************
//******************************************************************************
//* Establish an alternate color attribute map for the specified dctBILLBOARD  *
//* display data. Text data are not modified.                                  *
//*                                                                            *
//* Input  : cIndex : index of target dctBILLBOARD control                     *
//*          cPtr   : (optional, NULL pointer by default)                      *
//*                   - if new map not specified, then the map is set to the   *
//*                     default color determined by whether the control        *
//*                     currently has the input focus                          *
//*                   - if specified, points to an array of color attributes,  *
//*                     ONE VALUE FOR EACH DISPLAY LINE defined for the control*
//*                                                                            *
//* Returns: OK if successful                                                  *
//*          OR ERR if specified control is not of type dctBILLBOARD           *
//******************************************************************************

short NcDialog::SetBillboardColors ( short cIndex, const attr_t* cPtr )
{
   short status = ERR ;
   if ( cIndex <= this->lastCtrl && dCtrl[cIndex]->type == dctBILLBOARD )
   {
      DialogBillboard* cp = (DialogBillboard*)dCtrl[cIndex] ; // pointer to control
      cp->bbSetColor ( cPtr, bool(cIndex == this->currCtrl) ) ;
      cp->RedrawControl ( bool(cIndex == this->currCtrl) ) ;
      status = OK ;
   }
   return status ;

}  //* End SetBillboardColors() *

//*************************
//*  SetBillboardColors   *
//*************************
//******************************************************************************
//* Modify the color attribute for the specified display line of a             *
//* dctBILLBOARD control. Text data are not modified.                          *
//*                                                                            *
//* Input  : cIndex : index of target dctBILLBOARD control                     *
//*          lIndex : index of target line                                     *
//*          cAttr  : new color attribute for line                             *
//*                                                                            *
//* Returns: OK if successful                                                  *
//*          OR ERR if lIndex is out-of-range                                  *
//*                 if specified control is not of type dctBILLBOARD           *
//******************************************************************************

short NcDialog::SetBillboardColors ( short cIndex, short lIndex, attr_t cAttr )
{
   short status = ERR ;
   if ( cIndex <= this->lastCtrl && dCtrl[cIndex]->type == dctBILLBOARD )
   {
      DialogBillboard* cp = (DialogBillboard*)dCtrl[cIndex] ; // pointer to control
      short lineCount ;
      const attr_t* cPtr = cp->bbGetColor ( lineCount ) ;
      if ( lIndex >= ZERO && lIndex < lineCount )
      {
         attr_t   cArray[lineCount] ;
         for ( short i = ZERO ; i < lineCount ; i++ )
            cArray[i] = cPtr[i] ;
         cArray[lIndex] = cAttr ;
         cp->bbSetColor ( cArray, bool(cIndex == this->currCtrl) ) ;
         cp->RedrawControl ( bool(cIndex == this->currCtrl) ) ;
         status = OK ;
      }
   }
   return status ;

}  //* End SetBillboardColors() *

//*************************
//*                       *
//*************************
//******************************************************************************
//*                                                                            *
//*                                                                            *
//*                                                                            *
//* Input  :                                                                   *
//*                                                                            *
//* Returns:                                                                   *
//******************************************************************************


