//********************************************************************************
//* File       : idpp.cpp                                                        *
//* Author     : Mahlon R. Smith                                                 *
//*              Copyright (c) 2014-2025 Mahlon R. Smith, The Software Samurai   *
//*                 GNU GPL copyright notice below                               *
//* Date       : 02-Aug-2025                                                     *
//* Version    : (see AppVersion string)                                         *
//*                                                                              *
//* Description: Definitions and data for Infodoc Post-processor (idpp),         *
//* an HTML post-processing utility for use with HTML documents generated from   *
//* Texinfo source.                                                              *
//*                                                                              *
//* For full documentation for idpp is located in 'infodoc.info' :               *
//*             info -f infodoc.info -n 'Infodoc Post-processor'                 *
//*                                                                              *
//*                                                                              *
//* Copyright Notice:                                                            *
//* -----------------                                                            *
//* This program is free software: you can redistribute it and/or modify it      *
//* under the terms of the GNU General Public License as published by the Free   *
//* Software Foundation, either version 3 of the License, or (at your option)    *
//* any later version, PROVIDED THAT the copyright notices for both code and     *
//* documentation are included and unmodified.                                   *
//*                                                                              *
//* This program is distributed in the hope that it will be useful, but          *
//* WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY   *
//* or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License     *
//* for more details.                                                            *
//*                                                                              *
//* You should have received a copy of the GNU General Public License along      *
//* with this program.  If not, see <http://www.gnu.org/licenses/>.              *
//*                                                                              *
//*         Full text of the GPL License may be found in the TexInfo             *
//*         documentation for this program under 'Copyright Notice'.             *
//********************************************************************************
//* Version History (most recent first):                                         *
//*                                                                              *
//* v: 0.0.16 27-May-2024                                                        *
//*   -- Bug Fix: In parsing command-line arguments, if single-characters options*
//*      were concatenated, scan of the token may have stopped too early, causing*
//*      option(s) to be missed.                                                 *
//*   -- Bug Fix: The abort signal from the user was not being recognized under  *
//*      some very specific circumstances, i.e. abort before the first chapter   *
//*      heading.                                                                *
//*   -- Bug Fix: If a "@verbatim" block is declared, but the block contains no  *
//*      data, the scan was not seeing the closing tag for the block because     *
//*      it is located on the same line as the opening tag.                      *
//*           Example: "<pre class="verbatim"></pre><p>following_data"           *
//*   -- The interactive prompt for multitable border now silently accepts a     *
//*      response of 'a' (automatic) as a substitute for the default             *
//*      'b' (bordered) response. This was done for compatibility with the       *
//*      'automatic' response which is valid in prompts for processing other     *
//*      types of objects.                                                       *
//*   -- The makeinfo (texi2any) v:7.1 is inserting the HTML "&para;" entity at  *
//*      the end of headings and certain other links which use a "copiable-link" *
//*      reference.     <h1>, <h2>, <h3>, <h4>, <h5>. <h6>                       *
//*               and:  <a class="copiable-link" ...                             *
//*      There is no reason for this, and it is hoped that it will be repaired   *
//*      soon. Until then, we have added a temporary fix which scans for these   *
//*      entities and automatically removes them. The --no_paraent option        *
//*      has been added to disable this functionality (although we have no idea  *
//*      why you might want to use it).                                          *
//*   -- Documentation update.                                                   *
//*   -- Posted to website 02-Aug-2025                                           *
//*                                                                              *
//* v: 0.0.15 02-Apr-2024                                                        *
//*   -- Updates to accomodate changes to the texi2any processor.                *
//*      Specifically, there have been some new classes defined that replace     *
//*      the older <span>, <div> and <blockquote> sequences. One of these        *
//*      redefines the boundaries for the Table Of Contents which impacted all   *
//*      subsequent parsing.                                                     *
//*      -- The intent of our updates to 'idpp' is support BOTH makeinfo v:6.8   *
//*         AND makeinfo v:7.0+. This broad mandate may or may not be possible.  *
//*      -- Section menus now call out a "mini-toc" class.                       *
//*      -- <ul> lists: These lists are now constructed to declare either two(2) *
//*         class names, OR one(1) class name followed by a style element        *
//*         indicating the bullet character to be used. All declared classes are *
//*         now defined, along with a group of classes for additional characters.*
//*      -- <ol> lists: HTML source syntax has undergone minor adjustments, and  *
//*         idpp has been updated in those areas.                                *
//*         -- Important Change: The interactive prompt for enumerated lists has *
//*            been modified: the order of secondary options has changed.        *
//*            Therefore, existing response files which specify multiple options *
//*            for configuring enumerated lists will need to be verified.        *
//*            The new order of options is: enumeration_type, initial_value,     *
//*            font_size, and sequence_direction.                                *
//*      -- No significant differences were discovered in the HTML syntax for    *
//*         tables and cartouche constructs, but additional formatting options   *
//*         have been implemented for these constructs.                          *
//*      -- Redesign parsing of source data for defined text blocks (@verbatim,  *
//*         @display, @quotation, etc.). This allow for greater flexibility in   *
//*         parsing any new constructs implemented by texi2any in the future.    *
//*      -- No significant differences were discovered in the HTML syntax for    *
//*         "indentedblock" objects.                                             *
//*      -- For "quotation" block objects, the HTML syntax for the @author       *
//*         sub-command has changed, and idpp has been updated in that area.     *
//*   -- Significant enhancements have been made in the area of selecting the    *
//*      relative font size for all the major objects (what the makeinfo         *
//*      documentation calls "environments"). This includes block environments,  *
//*      itemized lists, enumerated lists, multitables, the cartouche, _and_     *
//*      text outside of any environment object. These enhancements include      *
//*      interactive-processing options, as well as an expanded support for      *
//*      invisible styling tokens embedded within the Texi source.               *
//*   -- Significant improvements have been made in the configuration of style   *
//*      options for enumerated lists. Again, these enhancements include         *
//*      interactive-processing options, as well as an expanded support for      *
//*      invisible styling tokens embedded within the Texi source.               *
//*   -- Add the "--cartouche" command-line option to give direct control over   *
//*      text flow within a cartouche object. The "--no_cartouche" option was a  *
//*      binary switch and is now deprecated.                                    *
//*   -- Bug Fix: If user specified to process a file for the second time, the   *
//*      'multiPass' flag was not being set, so a second copy of the container   *
//*      class was being written to target. (Bad programmer! Go to your room!)   *
//*   -- Bug Fix: If -v (verbose) option specified, number of line items in      *
//*      itemized lists was not being counted correctly.                         *
//*   -- Replace the 'c' (custom) enumerated type with 'G' upper-case Greek type.*
//*      Note that at this time (2024) not all browsers can handle upper-case    *
//*      Greek and will output the default decimal enumeration instead.          *
//*   -- Due to enhancements to the texi-2-HTML output, the "--up_target" and    *
//*      "--no_uplink" command-line options are no longer necessary and therefore*
//*      deprecated. Currently, these options are recognized, but ignored, and   *
//*      may be removed in a future release. By default, texi2any now removes the*
//*      up-target link of the first page and replaces it with a reference to    *
//*      the top of the document. This behavior can be controlled by a Texinfo   *
//*      build option. The application now assumes that if the build options     *
//*      TOP_NODE_UP_URL or TOP_FILE or TOP_NODE_FILE_TARGET are used, then the  *
//*      developer must know what he/she/they is/are doing. (We realized that    *
//*      this is a dangerous assumption, but we have faith in you! :-)           *
//*   -- Due to enhancements to the texi-2-HTML output, the "--fixed_list" option*
//*      now has limited usefulness because all unordered lists now declare a    *
//*      CSS class. Currently, this option is recognized, but ignored, and may   *
//*      be removed in a future release.                                         *
//*   -- Due to enhancements in the texi-2-HTML output, which now declares a     *
//*      standard HTML5 MIME type, the "--no_html5" option is now obsolete.      *
//*      If specified, this option will be ignored, and will be removed from     *
//*      the application for the next release.                                   *
//*   -- Update the output generated by the '--version' option.                  *
//*   -- Update command-line help to allow either '--help' or '--helpless'       *
//*      (-h or -hl). This functionality was imported from the author's AnsiCmd  *
//*      library, and optionally pipes the help data through the 'less' utility. *
//*      Thus, helpless :-)                                                      *
//*   -- Cleaned up receipt of the ABORT token from either the response file or  *
//*      within interactive mode.                                                *
//*   -- If an invalid response is read from a response file i.e. the file       *
//*      responses are out-of-synch, the application will now immediately        *
//*      terminate post-processing. This makes it easier for the developer to    *
//*      find where in the sequence the error occurred.                          *
//*   -- Import the nsleep() method from our AnsiCmd library. This allows for    *
//*      inserting a small delay in processing tokens read from the response     *
//*      file.                                                                   *
//*   -- Implement the '--step' command-line option to assist in debugging the   *
//*      scan of source data.                                                    *
//*   -- Implement the '--skip' command-line option to assist in debugging the   *
//*      user prompts.                                                           *
//*   -- Parsing the source now about 5-8% faster. (It wasn't slow before :-)    *
//*   -- Major documentation update. Note that the documentation may still be    *
//*      a bit rough in places, which will be addressed soon.                    *
//*   -- Posted to website: 24-May-2024                                          *
//*                                                                              *
//* v: 0.0.14 14-Jun-2023                                                        *
//*   -- Update: Recent changes to the HTML output formatting generated by       *
//*      makeinfo (texi2any) have required updates to the parsing algorithm      *
//*      of idpp. The major differences relate to changing various constructs    *
//*      to be "classes" instead of simple "<span> etc.                          *
//*      -- The HTMcL format which begins the main section of the data just below*
//*         the <body> was changed from: <span id="TOP">                         *
//*                                  to: <div class="top" id="Top">              *
//*      -- The HTML format which begins the table of contents was changed       *
//*         from: "<span id=\"SEC_Contents\"></span>"                            *
//*           to: "<div class=\"Contents_element\" id=\"SEC_Contents\">"         *
//*      -- The 'info-style' top menu is no longer included in the HTML document.*
//*      -- The Title page is now located _above_ the Table of Contents.         *
//*         This is actually a more logical placement, but it required updating  *
//*         the post-processor to scan for blocks of formatted text _above_      *
//*         the Table of Contents.                                               *
//*      -- The HTML format for the seldom-used "lisp" block has been modified:  *
//*         from: "<div class=\"lisp\">                                          *
//*           to: "<div class=\"example lisp\">                                  *
//*               "<pre class=\"lisp\">"                                         *
//*         We can see why this was done: the Lisp and Example blocks actually   *
//*         use the same format; however, it was a poor design decision.         *
//*   -- Update the gString-class code from v:0.0.30 to v:0.0.32. Note that the  *
//*      changes to gString are unrelated to the way the utility is used in this *
//*      application.                                                            *
//*   -- Consolidate some redundant code e.g. ppfFormattedBlock().               *
//*   -- Reorganize the messages generated by the '--book' option.               *
//*   -- Update copyright dates.                                                 *
//*   -- Correct a statement which generated a compiler warning about comparing  *
//*      signed and unsigned integers. (no actual functionality change)          *
//*   -- Posted to website 24-Jun-2023                                           *
//*                                                                              *
//* v: 0.0.13 01-Aug-2020                                                        *
//*   -- Modifications to post-processing for a second or subsequent pass on     *
//*      the same file. While a second pass on a previously-processed HTML       *
//*      file is allowed, our goal is to do as little _unintentional_            *
//*      modification as practical during the second pass.                       *
//*      For this reason, we have made the following adjustments:                *
//*      -- If a second pass is performed with automatic processing:             *
//*         -- Bug Fix: In ppfProcGNU(), two redundant "<br>" tokens were        *
//*            being inserted after the version-number declaration.              *
//*         -- Bug Fix: When pushing an empty source line into the FIFO object,  *
//*            the corresponding pull of that line from the FIFO object was      *
//*            seeing no data. Refer to ppfUnReadSrcLine() and ppfReadSrcLine()  *
//*            for details.                                                      *
//*         -- Bug Fix: In ppfProcEnumeratedList(), if the source file contained *
//*            a descending <ol> list, (example: 10, 9, 8, ...) the "reversed"   *
//*            token was being discarded.                                        *
//*         -- Bug Fix: The table-border prompts were being skipped on the       *
//*            second pass because there was no test for previously-processed    *
//*            tables.                                                           *
//*      -- If a second pass is performed with interactive processing via a      *
//*         response file:                                                       *
//*         -- The "no-mods" flag is set to prevent all modifications during     *
//*            the second pass. This is necessary to avoid unintended responses  *
//*            due to the response file tokens very likely becomming             *
//*            out-of-synch with the actual promots.                             *
//*            Note: The user can override the "no_mods" flag (for the first     *
//*            source file in the list only) by invoking the application with    *
//*            the '-V' (verify) flag.                                           *
//*   -- Bug Fix: When scanning for <ul> and <ol> tags, there was a hidden       *
//*      bug which accidentally parsed the tag correctly in all known cases.     *
//*      It now does the same parsing intentionally, eliminating the potential   *
//*      error.                                                                  *
//*   -- Documentation update.                                                   *
//*   -- Posted to website 14-Aug-2020                                           *
//*                                                                              *
//* v: 0.0.12 16-Dec-2019                                                        *
//*   -- Bug Fix: If formatted text blocks are present between the Table Of      *
//*      Contents and the top-level menu, they were not always recognized        *
//*      and processed. Now when processing the TOC, we return as soon as the    *
//*      first page header is processed, so everything below the top page        *
//*      header will be processed in the main loop.                              *
//*   -- Enhance user prompt for UL and OL lists by displaying cleaner           *
//*      "first line item" text.                                                 *
//*   -- Cartouche objects are now recognized within UL and OL lists so long     *
//*      as it is preceded by a blank line.                                      *
//*   -- Documentation update.                                                   *
//*   -- Posted to website 03-Jul-2020                                           *
//*                                                                              *
//* v: 0.0.11 12-Dec-2019                                                        *
//*   -- Bug Fix: Application exit code was sometimes not being set correctly.   *
//*      Application should exit with either OK ( 0 ) or ERR ( -1 ).             *
//*   -- Enhance processing of the document chapters which contain the           *
//*      GNU General Public License and the Free Documentation License.          *
//*      As stated elsewhere (ppfProcGNU() method), these are legal documents    *
//*      and therefore should resemble as closely as possible the lawyers'       *
//*      intended format.                                                        *
//*   -- Documentation update.                                                   *
//*   -- Posted to website 15-Dec-2019                                           *
//*                                                                              *
//* v: 0.0.10 26-Sep-2019                                                        *
//*   -- Skip v:0.0.06, v:0.0.07, 0.0.08 and v:0.0.09 to synchronize version     *
//*      number with the Infodoc CSS definitions which have been updated         *
//*      several times since the last Idpp release.                              *
//*   -- Transition from gcc 4.8.3 to gcc 9.2.1 required a change from           *
//*      readdir64_r() which is deprecated. Switch to readdir64() which is       *
//*      now thread-safe under the GNU compiler.                                 *
//*   -- Updates to address changes in makeinfo between v:5.5  and  v:6.6.       *
//*      -- A large number of HTML constructs generated by makeinfo conversion   *
//*         have been modified, causing idpp to incorrectly parse these          *
//*         constructs. For this reason it is necessary to freeze support for    *
//*         makeinfo v:5.x at idpp v:0.0.05. IDPP v:0.0.10 and forward will      *
//*         support makeinfo v:6.x and higher.                                   *
//*      -- Minor changes is header processing.                                  *
//*      -- Minor changes in Table Of Contents processing.                       *
//*      -- Significant enhancements in texi2any in creation of ordered lists    *
//*         in HTML output, allow for much smoother and more detailed            *
//*         post-processing of <ol> lists.                                       *
//*      -- The HTML for itemized lists (@itemize <ul>) is mostly unchanged,     *
//*         however we have enhanced the post-processing of these lists to       *
//*         support more types of bullets and to beautify the output.            *
//*         This includes enhanced automatic post-processing, and also an        *
//*         interactive mode for processing <ul> lists to optionally assign a    *
//*         bullet class to any list which does not include class information.   *
//*      -- texi2any v:6.6 Bug: According to the ChangeLog 2019-02-10 and the    *
//*         v:6.6 documentation, Gavin has removed support for smaller font in   *
//*         all @small... commands in HTML and now processes them as the         *
//*         standard-size versions. "@small..." blocks are now rendered as       *
//*         identical to their standard versions.                                *
//*         -- To compensate for this foolishness, we now offer interactive      *
//*            selection of font size for all the various block constructs:      *
//*            inherited/smaller/larger. More work for the user, but also more   *
//*            flexibility.                                                      *
//*         -- We also support standard/smaller/larger versions of @verbatim     *
//*            blocks which have never been supported in makeinfo.               *
//*      -- The HTML markup generated for @indentedblock blocks has changed      *
//*            From: '<div class="indentedblock">'                               *
//*              To: '<blockquote class="indentedblock">'                        *
//*         This is actually a good change, but it requires a corresponding      *
//*         change in the way we scan for indentedblock sequences because there  *
//*         is now some ambiguity between indentedblock and quotation markup.    *
//*         As noted above, the "@small..." versions of blocks are no longer     *
//*         generated by texi2any, but we do post-processing in a                *
//*         symetrical way: '<blockquote class="smallindentedblock">'            *
//*                         '<blockquote class="largeindentedblock">'            *
//*      -- HTML markup for quotation blocks (@quotation/@smallquotation)        *
//*         remains relatively unchanged, but we must now distinguish between:   *
//*             <blockquote>                        @quotation                   *
//*             <blockquote class="indentedblock">  @indentedblock               *
//*         Although the @smallquotation command is no longer recognized by      *
//*         we do provide standard/smaller/larger options.                       *
//*   -- Post-processing of @multitable blocks and other tables including        *
//*      the @cartouche has been redefined. The 'tabPrompt' member has been      *
//*      removed, and the 'tabBorder' member has been redefined to allow         *
//*      for more flexible processing options.                                   *
//*   -- Enhanced CSS formatting for @cartouche blocks (class=cartouche).        *
//*      The "--no_cartouche" option which allowed the genenerated border        *
//*      to be retained has been redefined to mean that text within a            *
//*      cartouche block is allowed to "flow". Cartouche text is                 *
//*      pre-formatted by default in the HTML document.                          *
//*   -- Enhance beautification of lists inside preformatte blocks, ALTHOUGH     *
//*      lists should never be inside @format, @display, @example, @lisp         *
//*      blocks. To disable this beautification invoke with the                  *
//*      "--no_special" command-line option.                                     *
//*   -- A file containing responses to prompts can be redirected to stdin,      *
//*      but if the number of responses is less than the number of prompts,      *
//*      the program would not finish because user could not regain control      *
//*      of stdin. We have therefore added support for opening and reading       *
//*      the response file directly.                                             *
//*   -- Fully document the "--scan" and "--book" options. Formerly, the         *
//*      documentation mentioned them but referred the user to the source code.  *
//*   -- Documentation update.                                                   *
//*   -- Posted to website 05-Dec-2019                                           *
//*                                                                              *
//* v: 0.0.05 24-Feb-2015                                                        *
//*   - Update calls to 'formatInt', 'compare' and 'limitChars' methods of the   *
//*     gString class to match changes to gString prototypes (no functionality   *
//*     changed).                                                                *
//*                                                                              *
//* v: 0.0.04 20-Feb-2015                                                        *
//*   - Create a more robust test for end-of-block markers. This reduces the     *
//*     chance that really strange constructs will hide the end-of-block tag,    *
//*     causing a processing error. See ppfEndBlock().                           *
//*   - Create a more robust test for the beginning of <ul> and <ol> lists.      *
//*     Occasional strange constructs were sometimes causing the opening tag     *
//*     to be missed, and therefore the list to be unprocessed.                  *
//*     See ppfTestItemizedList() and ppfTestEnumeratedList().                   *
//*   - Bug fix: In ppfProcFormattedBlock(), if source contained one             *
//*     preformatted block nested inside another preformatted block, then        *
//*     processing of the block terminated too soon.                             *
//*   - Lists inside pre-formatted blocks. In the previous release we did not    *
//*     process anything inside a preformatted block. In fact, lists             *
//*     SHOULD NOT be placed inside preformatted blocks, but if they are, we     *
//*     now identify them. Note, however, that we DO NOT process the lists       *
//*     themselves (that would require a second pass through the source HTML),   *
//*     but we do compensate for the unfortunate spacing in the generated HTML.  *
//*     See ppfProcFormattedBlock() and ppfPFB_List().                           *
//*                                                                              *
//* v: 0.0.03 02-Feb-2015                                                        *
//*   - Bug fix: If <ul> or <ol> list ends with a block construct, then the      *
//*     </ul> or </ol> tag was sometimes being missed.                           *
//*   - Bug fix: If first chapter header is not for 'index-main-menu', then      *
//*     end of TOC was not being found.                                          *
//*   - Bug fix: In ppfProcItemizedList(), if user specified that <ul> lists     *
//*     should not be processed, the disable-processing flag was not being set.  *
//*   - Remove the methods ppfCopyUL() and ppfCopyOL(). These made sense when    *
//*     the algorithm for list processing was unstable, but are no longer        *
//*     necessary.                                                               *
//*   - Allow for a '0' (zero) value in response to the <ol> list processing     *
//*     start value prompt IF the specified enumeration type == 'd' (decimal).   *
//*     This is to accomodate the fact that the text of both the GPL and FDL     *
//*     licenses begin with item '0'. Note that the prompt still calls for a     *
//*     value >= 1.                                                              *
//*   - Add a special test for GPL and FDL license text. Process the enumerated  *
//*     lists associated with these licenses (unless the '--no_mods' option      *
//*     has been specified). See 'ppfProcGNU' method for details.                *
//*   - Identify and process tables that are nested inside 'indentedblock'       *
//*     blocks.                                                                  *
//*                                                                              *
//* v: 0.0.02 01-Feb-2015                                                        *
//*   - First public release.                                                    *
//*   - All documented options are now fully functional except '--css_mods'.     *
//*   - All issues reported by beta-testers through 28 Jan 2015 have been        *
//*     addressed.                                                               *
//*   - All basic functionality seems to be working; but there are a number of   *
//*     issues related to Texinfo configuration options which have not yet       *
//*     been investigated.                                                       *
//*   - Channel all user input and display output through a single point for     *
//*     ease in future modifications.                                            *
//*   - Add the '--scan' and '--book' options for debugging only. These provide  *
//*     super-verbose output with line numbers to more easily find mis-handled   *
//*     HTML source constructs. (tip-of-the-hat to Xiaoxiao on this one :)       *
//*   - Texinfo documentation is relatively complete.                            *
//*   - Building with static libraries has not yet been tested.                  *
//*                                                                              *
//* v: 0.0.01 06-Dec-2014                                                        *
//*   - Experimental only. Application structure based on SourceProfiler.        *
//*   - All NcDialog/ncurses code has been temporarily disabled. This will       *
//*     be reinstated when (if) we implement Interactive Mode.                   *
//*                                                                              *
//********************************************************************************
//* Programmer's Notes:                                                          *
//* - The raw HTML mark-up generated by the 'makeinfo' utility is actually       *
//*   pretty good; however, a few things are clunky and backward-looking,        *
//*   while a few things in the displayed output really do look second-rate.     *
//*                                                                              *
//* - There are a number of Texinfo build options that affect the HTML output.   *
//*   The use of some of them may cause the early releases of idpp to report     *
//*   parsing errors or to incorrectly parse and process the data.               *
//*   These build options will be investigated as time permits.                  *
//*                                                                              *
//*                                                                              *
//*                                                                              *
//* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -    *
//* =================                                                            *
//* == To-do List: ==                                                            *
//* =================                                                            *
//*                                                                              *
//* -- Add capture of the panic button so shutdown is more orderly.              *
//*                                                                              *
//* -- Test second-pass scenarios for robustness. Change as little as possible.  *
//*                                                                              *
//* --command-line: --css_mods (modify the CSS definition file)                  *
//*                   Overrides all other options except --help and --version    *
//*                   - Add a method to modify the whole-document definitions    *
//*                     in the CSS file. Uses the '-f' option for the target     *
//*                     filename if specified, else the default definition       *
//*                     file 'infodoc-styles.css'.                               *
//*                     - background color                                       *
//*                     - foreground color                                       *
//*                     - base font size                                         *
//*                     - resize width of infodoc_container class                *
//*                     - enable/disable container border                        *
//*                 -- Instead of modifying the CSS file itself, the container   *
//*                    <div> could specify additional classes or direct styling. *
//*                                                                              *
//* -- Test support for @sub (subscript) and @sup (superscript)                  *
//*                                                                              *
//* -- If the '@anchor{}' command is used in the texi source, then the           *
//*    texi-to-html converter inappropriately concatenates the '@anchor' with    *
//*    the line that follows it. This may cause 'idpp' to miss the line          *
//*    following the '@anchor' command. For now, just be sure to follow all      *
//*    @anchor{} commands with a blank line.                                     *
//*    Looks like: "<a class="pxref" href="..." so probably will not be a problem*
//*                                                                              *
//* -- Re-test for CSS-3 compliance:                                             *
//*    Remember to insert the following at the top of infodoc_css.html           *
//*    <img style="margin:0px 0px 0px 1082px; border:0;width:88px;height:31px;"  *
//*    src="vcss-new.gif" title="W3C Validated CSS3!" alt="Valid CSS3!"/>        *
//*                                                                              *
//* -- Research other Texinfo customization variables.                           *
//*    -- At least SOME of the HTML-only build options in makeinfo cause invalid *
//*       parsing of the document. (BEFORE_TOC, etc.)                            *
//* -- Research how images (or other objects) are embedded in the HTML output.   *
//*    There is a good chance that it's not handled very well.                   *
//* -- Research how footnotes are generated in the HTML output.                  *
//* -- Add a chapter for instructions on installing info document into the       *
//*    info system? This can be borrowed from another application (filemangler). *
//* -- infodoc-styles.css                                                        *
//*    Verify that for each class called out in the HTML source that we at least *
//*    have a stub for it in the CSS file.                                       *
//*                                                                              *
//*                                                                              *
//*                                                                              *
//*                                                                              *
//*                                                                              *
//********************************************************************************

//****************
//* Header Files *
//****************
#include "idpp.hpp"

//****************
//* Local Data   *
//****************
extern const wchar_t* dToken ;
extern const char*    cssLINK0 ;
const wchar_t* dfltTargPath = L"#" ;      // Deprecated
const short    dfltTargPath_len = 2 ;     // Deprecated
const wchar_t* dfltTargText = L"(top)" ;  // Deprecated
const short    dfltTargText_len = 6 ;     // Deprecated
const short    RSTEP_MIN  = 2 ;           // used by the --step debugging option
const short    RSTEP_MAX  = 50 ;
const short    RSTEP_DFLT = 10 ;

//**********************
//* Non-member methods *
//**********************
bool  gclaGetArg ( const gString& gs, Cfg& trg ) ;


//*************************
//*         main          *
//*************************
//********************************************************************************
//* Program entry point.                                                         *
//*                                                                              *
//* Command-line Usage:  See GetCommandLineArgs() below for command-line         *
//*                      argument processing.                                    *
//*                                                                              *
//* Returns: OK  (0)  if all processing completed successfully                   *
//*          ERR (-1) if                                                         *
//*                   a) one or more command-line arguments invalid              *
//*                   b) one or more specified source files not found,           *
//*                      inaccessible, or not valid HTML                         *
//*                   c) CSS definition file not found or corrupted              *
//*                   d) processing error                                        *
//*                   e) user aborted the operation                              *
//********************************************************************************

int main ( int argc, char* argv[], char* argenv[] )
{
   //* Gather our entry-sequence data *
   commArgs clArgs( argc, argv, argenv ) ;

   //* Create the application class object *
   Idpp idpp( clArgs ) ;

   return ( int(idpp.ProcStatus()) ) ;

}  //* End main() *

//*************************
//*         ~Idpp         *
//*************************
//********************************************************************************
//* Destructor. Return all resources to the system.                              *
//*                                                                              *
//* Input  : none                                                                *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************

Idpp::~Idpp ( void )
{

   delete this->pushBack ; // release the dynamic memory allocation

}  //* End ~Idpp() *

//*************************
//*         Idpp          *
//*************************
//********************************************************************************
//* Default constructor.                                                         *
//*                                                                              *
//* Input  : commArgs class object (by reference)                                *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************

Idpp::Idpp ( commArgs& ca )
{
   //* Set the locale to the environment's choice i.e. UTF-8 encoding.*
   //* This affects all input and output streams.                     *
   std::locale::global(std::locale(""));

   //* Initialize our data members *
   this->sfCount = this->emCount = ZERO ;
   this->slCount = this->tlCount = ZERO ; // (unsigned values)
   this->iMode = false ;                  // (automatic processing is default)
   this->cssFile = L"infodoc-styles.css" ;
   this->pushBack = new fifo( 8 ) ;       // fifo buffer has 8 elements
   this->procStatus = OK ;                // exit code (0 or -1)

   //* <ol> and <ul> list processing _including_     *
   //* special-case processing is enabled by default.*
   this->ulLists = this->olLists = cfgAuto ;
   this->liSpec = true ;

   //* Pre-formatted blocks inherit font size from container by default *
   this->blkFont = cfgAuto ;

   //* Borders are applied to all main-sequence tables by default *
   this->tabBorder = cfgAuto ;

   //* Cartouche blocks contain pre-formatted text by default *
   this->cartFormat = cfgAuto ;

   this->allFiles = this->multiPass = this->abort    = 
   this->tocMod   = this->tocDel    = this->verbose  = 
   this->css_mod  = this->no_mods   = 
   
   this->no_body  = this->no_bloc   = this->no_auth  = 
   this->no_meta  = this->no_link   = this->no_cont  = false ;

   this->ulAssign = this->no_utrg =       // DEPRECATED MEMBERS
   this->no_html5 = this->upTarg = false ;

   #if PARA_FIX != 0       // Initialization
   this->no_para = false ; // entities will be automagicaly removed
   this->paraDel = ZERO ;  // count of entities removed
   #endif   // PARA_FIX

   //* Initialize default "Up" target link path and display text - Deprecated *
   wcsncpy ( this->utPath, dfltTargPath, dfltTargPath_len ) ;
   wcsncpy ( this->utText, dfltTargText, dfltTargText_len ) ;

   this->ppfGetCWD ( this->cwDir ) ;   // get path of current-working directory

   //* Initialize debugging option parameters *
   this->scan_beg = ZERO ; this->scan_end = 0xFFFF ; this->scan = false ;
   this->book = false ;
   this->rsteps = ZERO ;               // inter-token pause (disabled)
   this->skip   = ZERO ;               // auto-processing counter reset

   //* User may have specified one or more command-line arguments *
   bool validArgs = this->GetCommandLineArgs ( ca ) ;

   //* If full interactive mode, set the secondary-level processing switches.*
   if ( this->iMode )
   {
      this->ulLists    = cfgSpec ;
      this->olLists    = cfgSpec ;
      this->tabBorder  = cfgSpec ;
      this->blkFont    = cfgSpec ;
      this->cartFormat = cfgSpec ;
   }

   //* If '--up_target' option specified, parse the input string *
   //* into target-path and display-text segments. Deprecated.   *
   if ( this->upTarg != false )  // DEPRECATED OPTION
   {
      gString gstmp( this->utPath ) ;
      short idx = gstmp.find( L',' ) ;
      if ( idx >= ZERO )      // if display text ALSO specified
      {
         wcsncpy ( this->utText, &gstmp.gstr()[idx + 1], TARG_TEXT_LEN ) ;
         gstmp.limitChars( idx ) ;
         gstmp.copy( this->utPath, gsALLOCDFLT ) ;
      }
   }

   //* If valid user input, and not a cry for help *
   if ( validArgs && (ca.helpOption == ZERO) && (ca.verFlag == false) ) 
   {
      //* Determine whether to perform a pre-scan of the   * 
      //* first source file, then process the source files.*
      if ( (this->Prescan ( ZERO, ca.preFlag )) )
      {
         this->procStatus = this->ProcessSource () ;
      }
   }           // valid user input

   else        // explicit or implied cry for help
   {
      if ( ca.verFlag != false )
         this->DisplayAppVersion () ;
      else if ( (ca.helpOption > ZERO) || (this->emCount == ZERO) )
         this->DisplayCommandLineHelp ( bool(ca.helpOption == 2) ) ;
      else     // display accumulated command-line errors
      {
         gString gsOut ;
         this->ComposeTitle ( gsOut ) ;
         this->textOut ( gsOut ) ;
         for ( short i = ZERO ; i < this->emCount ; i++ )
            this->textOut ( this->ErrorMsg[i] ) ;
         this->textOut ( "\nFor more information: 'idpp --help'\n" ) ;
      }
   }

}  //* End Idpp() *

//*************************
//*        Prescan        *
//*************************
//********************************************************************************
//* Pre-scan the specified source file, and if it has been previously            *
//* processed, optionally ask user whether to continue with re-processing.       *
//*                                                                              *
//* Note that the optional user confirmation is requested ONLY for the first     *
//* file in the list of source files to be scanned, and the user's response      *
//* applies only to the first file in the list. Second and subsequent source     *
//* files will be handled according to the rules described below.                *
//*                                                                              *
//* Input  : srcIndex : sourcefile index (index into this->srcFiles[])           *
//*          prompt   : (optional, 'false' by default)                           *
//*                     if 'false', perform the scan but do not prompt for       *
//*                                 user confirmation                            *
//*                     if 'true',  perform the scan, and if file was            *
//*                                 previously processed, prompt user whether    *
//*                                 to continue processing                       *
//*                     On first call, 'prompt' reflects the '-V' (verify)       *
//*                     command-line argument. Flag is reset for second and      *
//*                     subsequent calls.                                        *
//*                                                                              *
//* Returns: 'true'  if source not previously scanned OR                         *
//*                  if user specifies that file is to be scanned again OR       *
//*                  if application invocation specifies automatic processing    *
//*          'false' if user abort                                               *
//********************************************************************************
//* Previous scan is indicated by the presence of the post-processing message    *
//* at the top of the <head> section.                                            *
//*                                                                              *
//* Processing a file multiple times can lead to unintended results:             *
//*    1) If "automatic processing" specified i.e. interactive flags are all     *
//*       reset, then it is likely (though not guaranteed) that the target       *
//*       will be nearly identical to the source file.                           *
//*       For this reason, second and subsequent passes are allowed if           *
//*       invocation parameters specify full automatic processing.               *
//*                                                                              *
//*    2) If user interaction is specified for second and subsequent passes,     *
//*       that interaction could easily override decisions made during the       *
//*       first pass.                                                            *
//*       a) If user is manually entering responses to processing prompts, we    *
//*          must assume that the user knows what he/she is doing.               *
//*          If user is informed that the file was previously processed, but     *
//*          decides to continue anyway, then again, we must assume that the     *
//*          user knows what she is are doing (although she probably doesn't).   *
//*       b) However, if responses to the interactive prompts are provided by    *
//*          a response file, it must be assumed that the response file was      *
//*          designed to apply to a fresh (newly-generated) HTML file.           *
//*          If such a response file is used against a previously-processed      *
//*          source file, it is very likely that the responses will get          *
//*          out-of-synch causing unintended consequence including multiple      *
//*          invalid responses to individual prompts.                            *
//*          For this scenario, we will display a message alerting the user to   *
//*          the danger of processing errors, and then disable modifications     *
//*          for during processing for this source file AND for all subsequent   *
//*          files in the source list.                                           *
//*                                                                              *
//********************************************************************************

bool Idpp::Prescan ( short srcIndex, bool prompt )
{
   const wchar_t* headEND = L"</head>" ;
   char    tmp[gsDFLTBYTES] ;    // UTF-8 line input from file
   gString gs,
           gsTrg( "%S/%S", this->cwDir.gstr(), this->srcFiles[srcIndex] ),
           gscomp( cssLINK0 ) ;
   gscomp.limitChars( 37 ) ;
   bool    proc = false,      // 'true' if source file already processed
           force =            // 'true' if invocation parameters require prescan
              (this->iMode || 
               (this->ulLists == cfgSpec)   || (this->olLists == cfgSpec) ||
               (this->tabBorder == cfgSpec) || (this->blkFont == cfgSpec))
               && ((this->respFile.gschars()) > 1),
           status = true ;    // return value, assume that file will be processed
   this->multiPass = false ;  // initialize the global flag

   if ( prompt || force )
   {
      //* Open the source file and scan for the marker *
      ifstream ifs( gsTrg.ustr(), ifstream::in ) ;
      if ( (ifs.is_open()) )
      {
         bool done = false ;
         while ( ! done )
         {
            ifs.getline( tmp, gsDFLTBYTES, NEWLINE ) ;
            if ( ifs.good() || ifs.gcount() > ZERO )
            {
               gs = tmp ;
               if ( (gs.find( gscomp.gstr()) >= ZERO) )
               { proc = true ; done = true ; }
               else if ( (gs.find( headEND )) >= ZERO )
                  done = true ;
            }
            else
               done = true ;
         }
         ifs.close() ;
      }

      //* If source file was previously processed *
      if ( proc )
      {
         //* If prompt enabled, ask user if we should continue *
         if ( prompt )
         {
            gs.compose( "The source file \"%S\" has already been processed.\n"
                        "Do you want to process it again? (y/n): ", 
                        this->srcFiles[srcIndex] ) ;
            this->textOut( gs, false ) ;
            this->userResponse ( gscomp ) ;
            if ( ((*gscomp.gstr()) == L'y') || ((*gscomp.gstr()) == L'Y') )
               this->multiPass = true ;   // set the 'previously-processed' flag
            else
               status = false ;           // abort
         }

         //* If source file has been previously processed, but user *
         //* prompt not specified, perform internal initializations.*
         else
         {
            this->no_mods = this->multiPass = true ;
         }
      }
   }
   return status ;

}  //* End Prescan() *

//*************************
//*     ProcessSource     *
//*************************
//********************************************************************************
//* Process the specified HTML documents.                                        *
//*                                                                              *
//* Input  : none                                                                *
//*                                                                              *
//* Returns: OK  if all files processed successfully                             *
//*          ERR if processing error(s) or user aborted operation                *
//********************************************************************************

short Idpp::ProcessSource ( void )
{
   gString gsSrc,          // source path/filename
           gsTrg,          // target path/filename
           gsName,         // misc. filename
           gsOut ;         // message output
   short   status = OK ;   // return value

   //* Display the application title, CWD, and CSS filename/version *
   this->ComposeTitle ( gsOut ) ;
   this->textOut ( gsOut ) ;
   gsOut.compose( L"CWD: %S", this->cwDir.gstr() ) ;
   this->textOut ( gsOut ) ;
   gsOut.compose( L"CSS: %S", this->cssFile.gstr() ) ;
   this->textOut ( gsOut ) ;
   gsOut.compose( L"CSS: %S", this->cssVersion ) ;
   this->textOut ( gsOut ) ;

   //* If a response file was specified, open it now. *
   if ( this->respFile.gschars() > 1 )
   {
      this->rifs.open ( this->respFile.ustr(), ifstream::in ) ;

      //* If the response file was opened, be sure the flags *
      //* for interactive processing are enabled.            *
      if ( (this->rifs.is_open()) )
      {
         this->iMode = true ;
         this->ulLists = this->olLists    =this->tabBorder = 
         this->blkFont = this->cartFormat = cfgSpec ;
      }
   }


   //* Process each file in list *
   for ( short i = ZERO ; (i < this->sfCount) && (status == OK) ; ++i )
   {
      //* If target modification is enabled, _provisionally_ test *
      //* whether the source file has been previously processed,  *
      //* and if it has, append a user alert. Note that prescan   *
      //* of first source file was performed by caller.           *
      if ( !this->no_mods && (i > ZERO) )
         this->Prescan ( i ) ;

      //* Declare the file being processed.*
      gsOut.compose( L">>> %S", this->srcFiles[i] ) ;
      if ( this->multiPass )
         gsOut.append( L" (previously processed)" ) ;
      this->textOut ( gsOut ) ;

      //* If processing is enabled *
      if ( this->no_mods == false )
      {
         //* Create full path/filename for source(backup) file *
         this->ppfCatPathFilename ( gsTrg, this->cwDir, this->srcFiles[i] ) ;
         gsSrc.compose( L"%S~", gsTrg.gstr() ) ;
         this->ppfExtractFilename ( gsName, gsSrc ) ;
         gsOut.compose( L"    src: %S", gsName.gstr() ) ;

         //* If existing backup file, delete it *
         if ( this->ppfTargetExists ( gsSrc ) )
         {
            status = this->ppfDeleteFile ( gsSrc ) ;
            if ( this->verbose || status != OK )
               gsOut.append( L"  (delete existing backup)" ) ;
            if ( status != OK )
               gsOut.append( L" FAILED!" ) ;
            this->textOut ( gsOut ) ;
         }

         //* Rename original file as backup file *
         if ( status == OK )
         {
            this->ppfExtractFilename ( gsName, gsTrg ) ;
            gsOut.compose( L"    trg: %S", gsName.gstr() ) ;
            status = this->ppfRenameFile ( gsTrg, gsSrc ) ;
            if ( this->verbose || status != OK )
               gsOut.append( L"   (backup source file)" ) ;
            if ( status != OK )
               gsOut.append( L" FAILED!" ) ;
            this->textOut ( gsOut ) ;
         }
         gsOut.clear() ; this->textOut( gsOut ) ;
      }
      //* Else, we are performing a scan-only pass *
      else
      {
         //* Create full path/filename for source file *
         this->ppfCatPathFilename ( gsSrc, this->cwDir, this->srcFiles[i] ) ;
         this->ppfExtractFilename ( gsName, gsSrc ) ;
         gsOut.compose( L"    src: %S\n"
                         "    trg: (none)", gsName.gstr() ) ;
         this->textOut ( gsOut ) ;
         gsTrg.clear() ;      // target set to null string
      }

      //* Process source file, creating new target file *
      if ( status == OK )
      {
         if ( (status = ppfProcessSrcHTML ( gsSrc, gsTrg )) == OK )
         {
            #if PARA_FIX != 0    // Report results
            //* Report number of "&para;" entities removed from file. *
            if ( ! this->no_mods )
            {
               gsOut.compose( "    \"&para;\" entities removed: %hd", &this->paraDel ) ;
               this->textOut ( gsOut ) ;
            }
            #endif   // PARA_FIX

            this->textOut ( L"    Conversion successful!" ) ;
            if ( this->verbose )
            {
               gsOut.compose( L"    (%4hu source lines processed and )", 
                              &this->slCount ) ;
               if ( this->no_mods != false )
               {
                  gsOut.limitChars( gsOut.gschars() - 6 ) ;
                  gsOut.append( L')' ) ;
               }
               this->textOut ( gsOut ) ;
               if ( this->no_mods == false )
               {
                  gsOut.compose( L"    (%4hu lines written to target    )", 
                                 &this->tlCount ) ;
                  this->textOut ( gsOut ) ;
               }
            }
         }
         else
         {
            gsOut.compose( L"    Processing error on source line: %hu\n"
                            "    Conversion failed: STOP!", &this->slCount ) ;
            this->textOut ( gsOut ) ;
         }
         gsOut.clear() ; this->textOut( gsOut ) ;
      }
      else  // preparing to exit
         { gsOut.clear() ; this->textOut( gsOut ) ; }
   }

   //* If a response file was opened, close it.*
   if ( this->rifs.is_open() )
      this->rifs.close() ;

   return status ;

}  //* End ProcessSource() *

//*************************
//*       textOut         *
//*************************
//********************************************************************************
//* All, or nearly all text written to the display goes through this method,     *
//* so we can control whether it goes through stdout (wide stream) or some       *
//* other channel.                                                               *
//*                                                                              *
//* Input  : tOut   : text data to be displayed                                  *
//*          newln  : (optional, true by default)                                *
//*                   if 'true', then terminate the line with an 'endl'          *
//*                   if 'false', do not termnate the line                       *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************
//* Programmer's Note: We channel as much of the stdout stream as possible       *
//* through this method to avoid future effort if we later add a GUI interface.  *
//********************************************************************************

void Idpp::textOut ( const gString& tOut, bool newln )
{
   wcout << tOut.gstr() ;
   if ( newln != false )
      wcout << endl ;

}  //* End textOut() *
void Idpp::textOut ( const char* tOut, bool newln )
{
   gString gs( tOut ) ;
   this->textOut ( gs, newln ) ;

}  //* End textOut() *
void Idpp::textOut ( const wchar_t* tOut, bool newln )
{
   gString gs( tOut ) ;
   this->textOut ( gs, newln ) ;

}  //* End textOut() *

//*************************
//*     userResponse      *
//*************************
//********************************************************************************
//* Get user response to text-mode prompts.                                      *
//*                                                                              *
//* Input  : gsIn   : (by reference) receives user input                         *
//*                                                                              *
//* Returns: nothing                                                             *
//*          (If abort token is received, the 'abort' data member is set.)       *
//********************************************************************************
//* Programmer's Notes:                                                          *
//* - User is prompted to enter a response for selecting a class type and        *
//*   optionally, a start index for <ol> (enumerated list) tags.                 *
//* - User is prompted to enter a response for selecting whether to place a      *
//*   border around a <table> object.                                            *
//* - Direct user input is straightforward: the interface expects character or   *
//*   string input, terminated by an ENTER key.                                  *
//* - Input may be redirected from a response file, for which we define a        *
//*   placeholder string: 'default_token'.                                       *
//* - User can abort processing by sending the "Quit" string (case insensitive)  *
//*   either from the response file OR typed directly at the prompt.             *
//*   If received, the 'abort' data member will be set indicating that caller    *
//*   should proceed to application shutdown.                                    *
//* - We wait indefinitely for user input.                                       *
//* - If the first character of the response is the :'#' (hash) character,       *
//*   then assume that the token is a comment and ignore it.                     *
//* - First character of input must not be an ASCII control character.           *
//*   (later, we may want to enforce a _printing_ character as the first)        *
//*                                                                              *
//* - If the token comes from a response file, then the shell does not echo it   *
//*   to the display. This throws off the formatting of the prompts.             *
//*   Unfortunately, we can't tell whether the token came from the keyboard or   *
//*   from a response file--they look the same after buffering. So, we output    *
//*   an extra linefeed after a valid token.                                     *
//********************************************************************************

void Idpp::userResponse ( gString& gsIn )
{
   const wchar_t* aToken = L"Quit" ;   // Abort token

   wchar_t userInput[gsALLOCDFLT] = L"" ;
   short wi ;                       // index into response string
   bool  done = false ;             // loop control

   gsIn.clear() ;             // clear caller's old data

   //* If we are running in scan-only mode, don't prompt for responses. *
   //* Instead, just return the 'default_token' string.                 *
   if ( this->no_mods != false )
   {
      gsIn = dToken ;
      this->textOut ( gsIn ) ;
      done = true ;
   }

   //* If we have an open response file, read a line. *
   if ( (this->rifs.is_open()) )
   {
      while ( ! done )
      {
         if ( (this->ppfReadLine ( this->rifs, gsIn )) == OK )
         {
            if ( (wi = gsIn.find( L'#' )) >= ZERO ) // if line contains a comment, delete it
               gsIn.limitChars( wi ) ;
            gsIn.strip() ;                // remove leading/trailing whitespace
            if ( (gsIn.gschars()) > 1 )   // if we have a token, echo it to display
            {
               //* Special Case: If response file returns the abort token:     *
               //* 1) close the response file. Caller will not access it again.*
               if ( (gsIn.find( aToken )) == ZERO )
               {
                  this->rifs.close() ;    // close the response file
                  this->abort = true ;    // set the global abort flag
               }
               this->textOut ( gsIn ) ;   // echo the token to the display

               if ( this->rsteps > ZERO ) // if delay enabled
               {
                  if ( ! this->scan || (this->scan && 
                        ((this->slCount >= this->scan_beg) && 
                         (this->slCount <= this->scan_end))) )
                  { this->nsleep ( this->rsteps ) ; }
               }
               done = true ;              // return token to caller
            }
         }
         //* End-of-file: This response, and any following prompts *
         //* will be taken from 'wcin', below.                     *
         else  // end-of-file
         {
            this->rifs.close() ;          // close the response file
            break ;
         }
      }
   }

   //* Get user response from 'wcin' *
   while ( ! done )
   {
      wcin >> userInput ;     // get user response

      //* Verify that input is not a control character and not a comment *
      if ( *userInput >= SPACE && *userInput != L'#' )
      {
         gsIn = userInput ;   // copy response to caller's buffer
         this->textOut ( L"" ) ; // transmit a newline to display
         if ( (gsIn.find( aToken, false )) == ZERO )
            this->abort = true ;
         break ;
      }
   }

}  //* End userResponse() *

//*************************
//*    invalidResponse    *
//*************************
//********************************************************************************
//* If user provides an invalid response to the prompt, re-prompt.               *
//*                                                                              *
//* Input  : none                                                                *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************

void Idpp::invalidResponse ( void )
{
   const wchar_t* badUser = L"Invalid response, your choice?: " ;
   const wchar_t* badSync = L"Invalid response from response file!" ;

   //* If the invalid response came from the response file, *
   //* it will be out-of-synch, so terminate processing.    *
   if ( (this->rifs.is_open()) )
   { this->textOut ( badSync ) ; this->abort = true ; }
   else
      this->textOut ( badUser, false ) ;

}  //* End invalidResponse() *

//*************************
//*        nsleep         *
//*************************
//********************************************************************************
//* Sleep (pause) execution for the specified time period.                       *
//* This method encapsulates access to the C-language nanosleep() function.      *
//* It is intended to simplify the the interface and replaces direct calls to    *
//* the obsolete C-language sleep() and usleep().                                *
//*                                                                              *
//* The primary delay value is specified in tenths of a second (0.10sec), with   *
//* a secondary delay specification in milliseconds (0.001sec).                  *
//* The sum of the two values specifies the duration of the delay.               *
//* Thus the effective minimum delay is 1 millisecond, while the theoretical     *
//* maximum delay is: 65535/10 or 6553 minutes + 65535 milliseconds or           *
//* approximately 6618 minutes.This is an impractically large number, so we      *
//* arbitrarily limit the maximum delay to 61 minutes, which is approximately    *
//* 60 minutes more than anyone will ever need.                                  *
//*               1 millisecond <= delay <= 61.0 minutes                         *
//*                                                                              *
//* Input  : tenths  : specifies the sleep interval in tenths of a second        *
//*                    (1/10 sec == 0.10 sec)                                    *
//*          millisec: (optional, zero by default) for finer control over the    *
//*                    length of the delay, specify a delay in milliseconds.     *
//*          nanosec : (optional, zero by default) for even finer control over   *
//*                    the length of the delay, specify a delay in nanoseconds.  *
//*                    [CURRENTLY IGNORED]                                       *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************

void Idpp::nsleep ( uint16_t tenths, uint16_t millisec, time_t nanosec ) const
{
   typedef struct timespec TimeSpec ;     // C++ syntax
   const time_t NS_SEC  = 1000000000 ;    // nanoseconds per second (1 billion ns)
   const time_t NS_10TH = NS_SEC / 10 ;   // nanoseconds per 1/10 second (100 million ns)
   const time_t NS_MSEC = NS_SEC / 1000 ; // nanoseconds per millisecond (1 million ns)
   const uint16_t MAX_10THS = 3600 ;      // arbitrary upper limit==60 minutes
   const uint16_t MAX_MSEC  = 60000 ;     // arbitrary upper limit==60 seconds

   if ( tenths > MAX_10THS )  tenths   = MAX_10THS ; // range checking
   if ( millisec > MAX_MSEC ) millisec = MAX_MSEC ;

   time_t   fullSec  = tenths / 10,          // full seconds
            tenthSec = tenths % 10,          // tenths of a second
            nsec     = (NS_10TH * tenthSec) ;// nanoseconds
            nsec += (millisec * NS_MSEC) ;
            if ( nsec >= NS_SEC )             // test for overflow
            { ++fullSec ; nsec -= NS_SEC ; }

   TimeSpec ts,                              // requested sleep interval
            //* If interval is interrupted, this receives remaining time.*
            rem = { ZERO, ZERO } ;
   int      status ;                         // return value from nanosleep()
   short    max_loop = 10 ;                  // limit loop iterations


   ts.tv_sec  = fullSec ;
   ts.tv_nsec = nsec ;

   #if 0    // FOR DEBUGGING ONLY
   gString gsfmt( ts.tv_nsec, FI_MAX_FIELDWIDTH, true ) ;
   gString gx( "tv_sec:%lu tv_nsec:%S\n", &ts.tv_sec, gsfmt.gstr() ) ;
   this->ttyWrite ( gx ) ;
   #endif   // FOR DEBUGGING ONLY

   if ( (ts.tv_nsec > ZERO) || (ts.tv_sec > ZERO) )
   {
      do
      {
         //* Execute the specified delay.*
         status = nanosleep ( &ts, &rem ) ;

         //* If the sleep was interrupted, call *
         //* again with the remaining time.     *
         if ( (status != ZERO) && (errno == EINTR) )
            ts = rem ;
         else           // success (or non-interrupt error)
            break ;
      }
      while ( (status != ZERO) && (--max_loop > ZERO) ) ;
   }

}  //* End nsleep() *

//*************************
//*      skipCounter      *
//*************************
//********************************************************************************
//* Debugging support: Decrement the 'skip' counter.                             *
//* If the decrement causes 'skip' to reach zero, enable full interactive mode.  *
//*                                                                              *
//* It is assumed that 'skip' is currently greater than zero, but if it's not,   *
//* it does no harm.                                                             *
//*                                                                              *
//* Input  : none                                                                *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************

void Idpp::skipCounter ( void )
{
   if ( --this->skip <= ZERO )
   {
      this->iMode = true ;
      this->ulLists    = cfgSpec ;
      this->olLists    = cfgSpec ;
      this->tabBorder  = cfgSpec ;
      this->blkFont    = cfgSpec ;
      this->cartFormat = cfgSpec ;
   }
}  //* End skipCounter() *

//*************************
//*  GetCommandLineArgs   *
//*************************
//********************************************************************************
//* Capture user's command-line arguments.                                       *
//*                                                                              *
//* Valid Arguments: (see DisplayCommandLineHelp())                              *
//*                                                                              *
//* Input  : commArgs class object (by reference)                                *
//*                                                                              *
//* Returns: 'true' if all arguments are valid                                   *
//*          'false' if invalid argument(s)                                      *
//********************************************************************************
//*  All arguments are optional EXCEPT:                                          *
//*    a) at least one source filename must be specified (16 names max)          *
//*       OR                                                                     *
//*    b) the "-a" (all files) switch must be specified                          *
//*                                                                              *
//*  c) parameters with string arguments may be specified as:                    *
//*     - string immediately follows:  idpp -dpublic_html                        *
//*     - string is next argument   :  idpp -d public_html                       *
//*     - string follows '=' sign   :  idpp -d=public_html                       *
//*  d) switches with sub-switch arguments                                       *
//*     - sub-switches follow immediately and in any order                       *
//*     - Note: no sub-switches defined at this time.                            *
//*  e) Request for help (or arg error) overrides all switches except --version  *
//*  f) --version overrides all other switches                                   *
//*                                                                              *
//*                                                                              *
//* -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -    *
//* Debugging options:                                                           *
//* ==================                                                           *
//* --scan=start,end                                                             *
//*   As lines are read from source print them to the display.                   *
//*   Optional arguments: starting source line number and ending source line     *
//*   number, separated by a comma ','. (See ppfReadSrcLine method)              *
//*   a) If no arguments, then display all lines read.                           *
//*      --scan                                                                  *
//*   b) If only first argument, then it is start. Continue scan to the end.     *
//*      --scan=400                                                              *
//*   c) If both arguments, then display all lines in the range, inclusive.      *
//*      --scan=400,525                                                          *
//*                                                                              *
//* --book                                                                       *
//*   For the main processing loop (ppfProcessSrcHTML method), display a         *
//*   message when a sub-process method is called and another message when it    *
//*   returns. This helps to locate the place in the source document where the   *
//*   processing logic has gone off into the weeds, i.e. if the end of the       *
//*   processing block is not properly identified by the sub-process method.     *
//*                                                                              *
//* --skip=skip_count                                                            *
//*   When in full automatic processing mode, count the number of user prompts   *
//*   which WOULD HAVE BEEN displayed in interactive mode, but which were        *
//*   skipped in automatic mode. When the count reaches 'skip_count", set the    *
//*   'iMode' flag to enable interactive mode for remainder of source data.      *
//*                                                                              *
//* --step                                                                       *
//*   When reading tokens from the response file, this option may be used to     *
//*   insert a delay between processing of each token. This slows the processing *
//*   enough that output to the display can be read by the user.                 *
//*   a) If no argument, then 7 (0.7 seconds) is the default.                    *
//*   b) If an argument is specified, it is range-checked:                       *
//*      2 <= TENTHS <= 50  which is interpreted as 2/10s throuth 5 seconds.     *
//********************************************************************************

bool Idpp::GetCommandLineArgs ( commArgs& ca )
{
   #define DEBUG_GCLA (0)        // for debugging only

   gString gs,                   // data formatting
           gsOut ;               // user messages
   const short argTest_len = 9 ; // number of chararacters to compare
   bool status = true ;          // return value

   if ( ca.argCount > 1 )
   {
      short j = ZERO ;
      bool multiarg = false ;
      for ( short i = 1 ; (i < ca.argCount) || (multiarg != false) ; ++i )
      {
         if ( multiarg != false )   // if additional switches in same argument
            --i ;
         else
            j = ZERO ;

         //* If a command-line switch OR continuing previous switch argument *
         if ( ca.argList[i][j] == DASH || multiarg != false )
         {
            multiarg = false ;
            ++j ;
            if ( ca.argList[i][j] == DASH ) // (double dash)
            {  //* Long-form command switches *
               bool goodarg = false ;
               ++j ;

               //* Options that begin with 'a' *
               if ( ca.argList[i][j] == 'a' )
               {
                  // none at this time
               }

               //* Options that begin with 'b' *
               else if ( ca.argList[i][j] == 'b' )
               {
                  gs = ca.argList[i] ;

                  //* Test for bullet-list option *
                  if ( (gs.compare( L"--bullet_list", true, argTest_len )) == ZERO )
                  {
                     goodarg = gclaGetArg ( gs, this->ulLists ) ;
                  }

                  //* Test for font-size option for preformatted blocks *
                  else if ( (gs.compare( L"--block_font", true, argTest_len )) == ZERO )
                  {
                     if ( (goodarg = gclaGetArg ( gs, this->blkFont )) != false )
                     {
                        if ( (this->blkFont != cfgAuto) && (this->blkFont != cfgSpec) )
                           goodarg = false ;
                     }
                  }

                  //* Test for '--book' option (book-end the blocks) *
                  else if ( (gs.compare( L"--book", true, 6 )) == ZERO )
                  {
                     goodarg = this->book = true ;
                  }
               }

               //* Options that begin with 'c' *
               else if ( ca.argList[i][j] == 'c' )
               {
                  gs = ca.argList[i] ;

                  if ( (gs.compare( L"--cartouche", true, 6 )) == ZERO )
                  {
                     // "auto"    == cfgAuto == pre-formatted text
                     // "flow"    == cfgNone == free-flow text
                     // "specify  == cfgSpec == prompt user for formatting option
                     //gs.replace( L"flow", L"none" ) ; // normalize the argument
                     //goodarg = gclaGetArg ( gs, this->cartFormat ) ;
                     // See gclaGetArg() algorithm
                        // after '='
                     short ai = ZERO ;
                     goodarg = false ;    // assume that user is a dumbguy
                     if ( (ai = gs.after( L'=' )) > ZERO )
                     {
                        goodarg = true ;  // user is provisionally intelligent
                        if ( (gs.find( "flow", ai, false )) >= ai )
                           this->cartFormat = cfgNone ;
                        else if ( (gs.find( "auto", ai, false )) >= ai )
                           this->cartFormat = cfgAuto ;
                        else if ( (gs.find( "spec", ai, false )) >= ai )
                           this->cartFormat = cfgSpec ;
                        else              // user was a dumbguy after all
                           goodarg = false ;
                     }

                     #if DEBUG_GCLA != 0
                     gString gstmp( "--cartouche: %s", 
                                    (this->cartFormat == cfgAuto ? "cfgAuto" :
                                     this->cartFormat == cfgNone ? "cfgNone" :
                                                                   "cfgSpec") ) ;
                     this->textOut ( gstmp ) ;
                     #endif   // DEBUG_GCLA
                  }

                  #if 0    // NOT YET IMPLEMENTED
                  //* Request to modify the whole-document CSS options *
                  if ( (gs.compare( L"--css_mods", true, argTest_len ) == ZERO) )
                  {
                     goodarg = this->css_mod = true ;

                     if ( this->emCount < emMAX )
                     {
                        gsOut = "Sorry, interactive modification of CSS definitions "
                                "not yet implemented." ;
                        gsOut.copy( this->ErrorMsg[this->emCount++], gsALLOCDFLT ) ;
                     }
                     status = false ;
                     break ;
                  }
                  #endif   // NOT YET IMPLEMENTED
               }

               //* Options that begin with 'e' *
               else if ( ca.argList[i][j] == 'e' )
               {
                  gs = ca.argList[i] ;

                  //* Test for enumeration-list option *
                  if ( (gs.compare( L"--enum_list", true, argTest_len ) == ZERO) )
                  {
                     goodarg = gclaGetArg ( gs, this->olLists ) ;
                  }
               }

               //* Options that begin with 'f' *
               else if ( ca.argList[i][j] == 'f' )
               {
                  gs = ca.argList[i] ;

                  if ( (gs.compare( L"--fixed_list", true, argTest_len ) == ZERO) )
                  {
                     goodarg = this->ulAssign = true ;
                  }
               }

               //* Options that begin with 'm' *
               else if ( ca.argList[i][j] == 'm' )
               {
                  gs = ca.argList[i] ;

                  //* Request to insert custom data into <head> section *
                  if ( (gs.compare( L"--my_metadata", true, argTest_len ) == ZERO) )
                  {
                     goodarg = true ; // good command, test its argument
                     if ( (status = this->gclaGetFilename ( gs, this->userMeta )) == false )
                     {
                        if ( this->emCount < emMAX )
                        {
                           gsOut.compose( "Error! Metadata file '%S' not found!", 
                                          this->userMeta.gstr() ) ;
                           gsOut.copy( this->ErrorMsg[this->emCount++], gsALLOCDFLT ) ;
                        }
                        this->userMeta.clear() ;
                     }
                  }
               }

               //* Options that begin with 'r' *
               else if ( ca.argList[i][j] == 'r' )
               {
                  gs = ca.argList[i] ;

                  //* Test for 'response' option *
                  if ( (gs.compare( L"--response", true, 6 )) == ZERO )
                  {
                     goodarg = true ; // good command, test its argument
                     if ( (status = this->gclaGetFilename ( gs, this->respFile )) == false )
                     {
                        if ( this->emCount < emMAX )
                        {
                           gsOut.compose( "Error! Specified response file '%S' not found!", 
                                          this->respFile.gstr() ) ;
                           gsOut.copy( this->ErrorMsg[this->emCount++], gsALLOCDFLT ) ;
                        }
                        this->respFile.clear() ;
                     }
                  }
               }

               //* Options that begin with 's' *
               else if ( ca.argList[i][j] == 's' )
               {
                  gs = ca.argList[i] ;

                  //* Test for '--scan' debugging option *
                  if ( (gs.compare( L"--scan", true, 6 )) == ZERO )
                  {
                     goodarg = this->scan = true ;
                     uint16_t b, e ;
                     short args = gs.gscanf( 6, L"=%hu,%hu", &b, &e ) ;
                     switch ( args )
                     {
                        case 1:     // scan from
                           this->scan_beg = b ;
                           break ;
                        case 2:     // scan range
                           this->scan_beg = b ;
                           this->scan_end = e ;
                           break ;
                        case 0:     // scan all
                        case int(WEOF):
                           this->scan_beg = ZERO ;
                           this->scan_end = 0xFFFF ;
                           break ;
                        default:
                           goodarg = this->scan = false ;
                           break ;
                     }
                     #if DEBUG_GCLA != 0
                     gString gstmp( "--scan: specified args:%hd scan_beg:%hu scan_end:%hu", 
                                    &args, &this->scan_beg, &this->scan_end ) ;
                     this->textOut ( gstmp ) ;
                     #endif   // DEBUG_GCLA
                  }

                  //* Test for 'step' debugging option *
                  if ( (gs.compare( L"--step", true, 6 )) == ZERO )
                  {
                     goodarg = true ; // good command, test its argument
                     short tenths,
                           argindx = gs.after( L'=' ) ;
                     if ( (argindx > ZERO) && 
                          ((gs.gscanf( argindx, L"%hd", &tenths )) == 1) )
                     {
                        if ( tenths < RSTEP_MIN )      { tenths = RSTEP_MIN ; }
                        else if ( tenths > RSTEP_MAX ) { tenths = RSTEP_MAX ; }
                     }
                     else { tenths = RSTEP_DFLT ; }
                     this->rsteps = tenths ;
                  }

                  //* Test for '--skip' debugging option. *
                  //* (argument s/b a positive integer.)  *
                  if ( (gs.compare( L"--skip", true, 6 )) == ZERO )
                  {
                     short cnt,
                           argindx = gs.after( L'=' ) ;
                     if ( (argindx > ZERO) && 
                          ((gs.gscanf( argindx, L"%hd", &cnt )) == 1) && (cnt > ZERO) )
                     {
                        this->skip = cnt ;
                        goodarg = true ;

                        #if DEBUG_GCLA != 0
                        gString gstmp( "--skip: specified arg: %hd", &this->skip ) ;
                        this->textOut ( gstmp ) ;
                        #endif   // DEBUG_GCLA
                     }
                  }
               }

               //* Options that begin with 't' *
               else if ( ca.argList[i][j] == 't' )
               {
                  gs = ca.argList[i] ;

                  //* Request to place borders around table objects *
                  if ( (gs.compare( L"--table_border", true, argTest_len ) == ZERO) )
                  {
                     goodarg = gclaGetArg ( gs, this->tabBorder ) ;
                  }
               }

               //* Options that begin with 'u' *
               else if ( ca.argList[i][j] == 'u' )
               {
                  gs = ca.argList[i] ;

                  //* Test for "up target" specification - DEPRECATED OPTION *
                  if ( (gs.compare( L"--up_target", true, argTest_len ) == ZERO) )
                  {
                     goodarg = true ; // good command, test its argument
                     // Programmer's Note: The data are not a real filename, 
                     // so we ignore the returned status.
                     this->gclaGetFilename ( gs, gsOut ) ;
                     if ( gsOut.gschars() > 1 )
                     {
                        gsOut.copy( this->utPath, gsALLOCDFLT ) ;
                        this->upTarg = true ;   // DEPRECATED OPTION
                     }
                  }
               }

               //* Request for application version number *
               else if ( ca.argList[i][j] == 'v' )
               {  //* Most Linux console apps respond to a       *
                  //* '--version' request in a cannonical manner.*
                  //* Overrides everything else on comand line.  *
                  gs = ca.argList[i] ;
                  if ( (gs.compare( L"--version", true, argTest_len ) == ZERO) )
                  {
                     ca.reset() ;
                     ca.helpOption = 1 ;
                     ca.verFlag = true ;
                     #if DEBUG_GCLA != 0
                     this->textOut ( L"--version: specified" ) ;
                     #endif   // DEBUG_GCLA
                     break ;
                  }
               }

               //* Long-form request for Help *
               else if ( ca.argList[i][j] == 'h' || ca.argList[i][j] == 'H' )
               {
                  gs = &ca.argList[i][j] ;
                  ca.helpOption = ((gs.find( L"helpless" )) == ZERO) ? 2 : 1 ;

                  #if DEBUG_GCLA != 0
                  if ( ca.helpOption == 2 ) { this->textOut ( L"--helpless: specified" ) ; }
                  else { this->textOut ( L"--help: specified" ) ; }
                  #endif   // DEBUG_GCLA
               }

               //* Exceptions to default processing *
               else if ( ca.argList[i][j] == 'n' )
               {
                  gs = ca.argList[i] ;
                  if ( (gs.compare( L"--no_mods", true, argTest_len ) == ZERO) )
                     goodarg = this->no_mods = this->verbose = true ;
                  else if ( (gs.compare( L"--no_spec", true, argTest_len ) == ZERO) )
                  { goodarg = true ; this->liSpec = false ; }
                  else if ( (gs.compare( L"--no_html", true, argTest_len ) == ZERO) )
                     goodarg = this->no_html5 = true ; // OBSOLETE OPTION
                  else if ( (gs.compare( L"--no_uplink", true, argTest_len ) == ZERO) )
                     goodarg = this->no_utrg = true ; // DEPRECATED OPTION
                  else if ( (gs.compare( L"--no_body", true, argTest_len ) == ZERO) )
                     goodarg = this->no_body = true ;
                  else if ( (gs.compare( L"--no_meta", true, argTest_len ) == ZERO) )
                     goodarg = this->no_meta = true ;
                  else if ( (gs.compare( L"--no_link", true, argTest_len ) == ZERO) )
                     goodarg = this->no_link = true ;
                  else if ( (gs.compare( L"--no_bloc", true, argTest_len ) == ZERO) )
                     goodarg = this->no_bloc = true ;
                  else if ( (gs.compare( L"--no_auth", true, argTest_len ) == ZERO) )
                     goodarg = this->no_auth = true ;
                  else if ( (gs.compare( L"--no_cont", true, argTest_len ) == ZERO) )
                     goodarg = this->no_cont = true ;
                  #if 1    // Deprecated for v:0.0.15: '--no_cartouche' option.
                  else if ( (gs.compare( L"--no_cart", true, argTest_len ) == ZERO) )
                  { this->cartFormat = cfgNone ; goodarg = true ; }
                  #endif   // Will be removed in the next release
                  #if PARA_FIX != 0    // Commline arg
                  else if ( (gs.compare( L"--no_para", true, argTest_len ) == ZERO) )
                  { this->no_para = true ; goodarg = true ; } // set the flag to disable removal
                  #endif   // PARA_FIX
               }

               if ( goodarg != false )
               {
                  #if DEBUG_GCLA != 0
                  this->textOut ( gs, false ) ;
                  this->textOut ( L": specified" ) ;
                  #endif   // DEBUG_GCLA
               }
               else           // invalid switch
               {  //* Generate an error mesage *
                  if ( this->emCount < emMAX )
                  {
                     gs.compose( "Error! Unrecognized command: '%s' ", ca.argList[i] ) ;
                     gs.copy( this->ErrorMsg[this->emCount++], gsALLOCDFLT ) ;
                  }
                  status = false ;
               }
               continue ;     // finished with this argument
            }  // (double dash)
            char argLetter  = ca.argList[i][j],
                 argLetter2 = tolower( ca.argList[i][j+1] ) ;

            //* Interactive Mode processing *
            if ( argLetter == 'i' || argLetter == 'I' )
            {
               this->iMode = true ;
               if ( ca.argList[i][j+1] != NULLCHAR ) multiarg = true ;
               #if DEBUG_GCLA != 0
               this->textOut ( L"-i: specified" ) ;
               #endif   // DEBUG_GCLA
            }

            //* Verbose output (applies only to text mode) *
            else if ( (argLetter == 'v') && 
                      (strncmp ( ca.argList[i], "--v", 3 )) != ZERO)
            {
               this->verbose = true ;
               if ( ca.argList[i][j+1] != NULLCHAR ) multiarg = true ;
               #if DEBUG_GCLA != 0
               this->textOut ( L"-v: specified" ) ;
               #endif   // DEBUG_GCLA
            }

            //* Perform pre-scan to determine whether  *
            //* source file has already been processed.*
            else if ( argLetter == 'V' )
            {
               ca.preFlag = true ;
               if ( ca.argList[i][j+1] != NULLCHAR ) multiarg = true ;
               #if DEBUG_GCLA != 0
               this->textOut ( L"-V: specified" ) ;
               #endif   // DEBUG_GCLA
            }

            //* Specify alternate CSS definition file *
            else if ( argLetter == 'f' || argLetter == 'F' )
            {
               ++j ;    // index next character after switch character
               if ( ca.argList[i][j] == NULLCHAR ) // if path is in next arg
               { ++i ; j = ZERO ; }
               else if ( ca.argList[i][j] == '=' )
                  ++j ;
               if ( ca.argList[i][j] != DASH && ca.argList[i][j] != NULLCHAR )
               {
                  this->cssFile = &(ca.argList[i][j]) ;
                  #if DEBUG_GCLA != 0
                  this->textOut ( L"-f: ", false ) ;
                  this->textOut ( this->cssFile ) ;
                  #endif   // DEBUG_GCLA
               }
               else
               {
                  if ( this->emCount < emMAX )
                  {
                     gs = "Error! '-f' switch specified without filename" ;
                     gs.copy( this->ErrorMsg[this->emCount++], gsALLOCDFLT ) ;
                  }
                  status = false ;
               }
            }

            //* Process ALL files in directory *
            else if ( argLetter == 'a' || argLetter == 'A' )
            {
               this->allFiles = true ;
               if ( ca.argList[i][j+1] != NULLCHAR ) multiarg = true ;
               #if DEBUG_GCLA != 0
               this->textOut ( L"-a: specified" ) ;
               #endif   // DEBUG_GCLA
            }

            //* Alternate target Directory *
            else if ( argLetter == 'd' )
            {  //* Change the current working directory *
               ++j ;    // index next character after switch character
               if ( ca.argList[i][j] == NULLCHAR ) // if path is in next arg
               { ++i ; j = ZERO ; }
               else if ( ca.argList[i][j] == '=' )
                  ++j ;
               gs = &(ca.argList[i][j]) ;
               #if DEBUG_GCLA != 0
               this->textOut ( L"-d: ", false ) ;
               this->textOut ( gs ) ;
               #endif   // DEBUG_GCLA
               if ( (this->ppfCdTarget ( gs.ustr() )) != OK )
               {
                  if ( this->emCount < emMAX )
                  {
                     gs.compose( "Error! Alternate target directory '%S' not found.", gs.gstr() ) ;
                     gs.copy( this->ErrorMsg[this->emCount++], gsALLOCDFLT ) ;
                  }
                  status = false ;
               }
            }

            //* Process table of Contents *
            else if ( argLetter == 'c' )
            {
               this->tocMod = true ;
               if ( ca.argList[i][j+1] != NULLCHAR ) multiarg = true ;
               #if DEBUG_GCLA != 0
               this->textOut ( L"-c: specified" ) ;
               #endif   // DEBUG_GCLA
            }

            //* Remove table of contents *
            else if ( argLetter == 'r' )
            {
               this->tocDel = true ;
               if ( ca.argList[i][j+1] != NULLCHAR ) multiarg = true ;
               #if DEBUG_GCLA != 0
               this->textOut ( L"-r: specified" ) ;
               #endif   // DEBUG_GCLA
            }

            //* Request for Help *
            else if ( argLetter == 'h' || argLetter == 'H' || argLetter == '?' )
            {  //* A cry for help overrides everything else   *
               //* on the command line except Version.        *
               //* (see below)
               ca.helpOption = (argLetter2 == 'l' ? 2 : 1) ;

               #if DEBUG_GCLA != 0
               if ( ca.helpOption == 2 ) { this->textOut ( L"-hl: specified" ) ; }
               else { this->textOut ( L"-h: specified" ) ; }
               #endif   // DEBUG_GCLA
            }

            else           // invalid switch
            {  //* Generate an error mesage *
               if ( this->emCount < emMAX )
               {
                  gs.compose( "Error! Unrecognized command: '%s' ", ca.argList[i] ) ;
                  gs.copy( this->ErrorMsg[this->emCount++], gsALLOCDFLT ) ;
               }
               status = false ;
            }
         }
         else
         {  //* Interpret argument as a filename. (validation occurs below) *
            if ( this->sfCount < sfMAX )
            {
               gs = ca.argList[i] ;
               gs.copy( this->srcFiles[this->sfCount++], gsALLOCDFLT ) ;
            }
            else
            {  //* Generate an error message *
               if ( this->emCount < emMAX )
               {
                  gs.compose( "Error! Too many source files specified. '%s' ignored.", ca.argList[i] ) ;
                  gs.copy( this->ErrorMsg[this->emCount++], gsALLOCDFLT ) ;
               }
            }
         }
      }     // for(;;)

      if ( status != false )
      {         
         //* Verify that CSS definition file exists and is our file.*
         if ( ((this->ppfRealpath ( gs, this->cssFile )) == OK)
              && ((this->ppfTargetExists ( gs )) != false) )
         {  //* Verify copyright notice and version number *
            gString gsVer ;
            if ( this->ppfTargetIsCSS ( gs, gsVer ) )
            {
               gsVer.copy( this->cssVersion, CSS_VER_LEN ) ;
            }
            else
            {
               if ( this->emCount < emMAX )
               {
                  gs.compose( "Error! CSS definition file '%S'"
                              " is of unknown configuration!", 
                              this->cssFile.gstr() ) ;
                  gs.copy( this->ErrorMsg[this->emCount++], gsALLOCDFLT ) ;
               }
               status = false ;
            }
         }
         else
         {
            if ( this->emCount < emMAX )
            {
               gs.compose( "Error! CSS definition file '%S' was not found!", 
                           this->cssFile.gstr() ) ;
               gs.copy( this->ErrorMsg[this->emCount++], gsALLOCDFLT ) ;
            }
            status = false ;
         }

         //* "skip" debugging option is valid ONLY in auto-processing mode, *
         //* and ONLY if no response file has been specified.               *
         if ( (status != false) && (this->iMode || (this->respFile.gschars() > 1)) )
            this->skip = ZERO ;
      }

      if ( ca.helpOption > ZERO && !ca.verFlag )
      {  //* Help overrides everything except Version *
         short tmp = ca.helpOption ;
         ca.reset() ;
         ca.helpOption = tmp ;
      }
   }        // if(ca.argCount>1)

   //* If the '-a' switch specified, scan all files in the target directory  *
   //* for HTML documents, and create a list of files to be processed.       *
   if ( this->allFiles != false && status != false )
      status = this->ppfScan4Src () ;


   //* Validate the list of source files to be processed.*
   if ( status != false )
   {
      for ( short sfIndex = ZERO ; sfIndex < this->sfCount ; sfIndex++ )
      {
         this->ppfCatPathFilename ( gs, this->cwDir, this->srcFiles[sfIndex] ) ;
         if ( this->ppfTargetExists ( gs ) )
         {
            if ( (this->ppfTargetIsHTML ( gs )) == false )
            {  //* Generate an error message *
               if ( this->emCount < emMAX )
               {
                  gs.compose( "Error! Source file '%S' is not valid HTML!", this->srcFiles[sfIndex] ) ;
                  gs.copy( this->ErrorMsg[this->emCount++], gsALLOCDFLT ) ;
               }
               status = false ;
            }
         }
         else
         {  //* Generate an error mesage *
            if ( this->emCount < emMAX )
            {
               gs.compose( "Error! Source file '%S' was not found!", this->srcFiles[sfIndex] ) ;
               gs.copy( this->ErrorMsg[this->emCount++], gsALLOCDFLT ) ;
            }
            status = false ;
         }
      }
   }

   //* No command line arguments OR no filenames specified.         *
   //* If --css_mods specified then no HTML filenames are necessary.*
   if ( (ca.argCount == 1) || (this->sfCount == ZERO && this->css_mod == false) )
   {
      gs = "--no source filenames specified--" ;
      gs.copy( this->ErrorMsg[this->emCount++], gsALLOCDFLT ) ;
      status = false ;
   }

   #if DEBUG_GCLA != 0
   this->nsleep ( 55 ) ;      // pause for a moment
   #endif   // DEBUG_GCLA

   return status ;

#undef DEBUG_GCLA
}  //* End GetCommandLineArgs() *

//*************************
//*    gclaGetFilename    *
//*************************
//********************************************************************************
//* Called by GetCommandLineArgs() to capture and verify a filename argument     *
//* for a command-line parameter.                                                *
//*                                                                              *
//* Input  : cmd  : (by reference) command-line argument                         *
//*          fname: (by reference) receives the captured and verified filename   *
//*                                                                              *
//* Returns: 'true'  if a valid filename                                         *
//*          'false' if file not found                                           *
//********************************************************************************

bool Idpp::gclaGetFilename ( const gString& cmd, gString& fname )
{
   gString gstmp ;                     // work buffers
   bool  goodarg = false ;             // return value
   short idx = cmd.after( L'=' ) ;     // index of argument

   fname.clear() ;           // initialize caller's buffer

   if ( (idx >= ZERO) && (cmd.gstr()[idx] != NULLCHAR) )
   {
      gstmp = &cmd.gstr()[idx] ;
      if ( ((this->ppfRealpath ( fname, gstmp )) == OK) &&
           ((this->ppfTargetExists ( fname )) != false) )
         goodarg = true ;
      else
         fname = gstmp ;
   }
   return goodarg ;

}  //* End gclaGetFilename() *

//*************************
//*      gclaGetArg       *
//*************************
//********************************************************************************
//* Called by GetCommandLineArgs() to isolate the arguments for long-form        *
//* command-line options.                                                        *
//*                                                                              *
//* Input  : gs   : (by reference) command-line argument                         *
//*          trg  : (by reference) receives the specified argument               *
//*                                                                              *
//* Returns: 'true'  if a valid argument (trg receives a member of enum Cfg)     *
//*          'false' if unknown argument (trg unmodified)                        *
//********************************************************************************

bool gclaGetArg ( const gString& gs, Cfg& trg )
{
   bool  goodarg = false ;    // return value
   short idx = gs.after( L'=' ) ;

   if ( (idx > ZERO) && (gs.gstr()[idx] != NULLCHAR) )
   {
      goodarg = true ;
      if ( (gs.compare( L"specify", true, 4, idx )) == ZERO )
         trg = cfgSpec ;
      else if ( (gs.compare( L"auto", true, 4, idx )) == ZERO )
         trg = cfgAuto ;
      else if ( (gs.compare( L"none", true, 4, idx )) == ZERO )
         trg = cfgNone ;
      else
         goodarg = false ;
   }
   return goodarg ;

}  //* gclaGetArg() *

//*************************
//*     ComposeTitle      *
//*************************
//********************************************************************************
//* Create a text string including the application title, application version    *
//* number and copywrite years.                                                  *
//*                                                                              *
//* Input  : gs     : (by reference) receives the formatting data                *
//*          tailNL : (optional, 'false' by default)                             *
//*                   if 'true',  a newline character is appended to the text    *
//*                   if 'false', no trailing newline                            *
//*          headNL : (optional,  'false' by default)                            *
//*                   if 'true',  place a newline character at start of text     *
//*                   if 'false', no leading newline                             *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************

void Idpp::ComposeTitle ( gString& gs, bool tailNL, bool headNL )
{
   gs.compose( titleTemplate, 
               AppTitle1, AppVersion, AppYears, AppTitle2 ) ;
   gs.padCols( (gs.gscols() *2), L'-' ) ;
   if ( headNL ) { gs.insert( NEWLINE ) ; }
   if ( tailNL ) { gs.append( NEWLINE ) ; }

}  //* End ComposeTitle() *

//*************************
//*  DisplayAppVersion    *
//*************************
//********************************************************************************
//* Display the application's title, version and copyright info.                 *
//*                                                                              *
//* Input  : none                                                                *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************
//* Programmer's Note: We play tricks with our title strings to make this        *
//* message look canonical. If the strings are changed, this message may get     *
//* ugly. That's why Trix are for kids....                                       *
//********************************************************************************

void Idpp::DisplayAppVersion ( void )
{
   const wchar_t* freeSoftware = 
   L"License GPLv3+: GNU GPL version 3 or later <http://gnu.org/licenses/gpl.html>\n"
    "This is free software: you are free to modify and/or redistribute it\n"
    "under the terms set out in the license.\n"
    "There is NO WARRANTY, to the extent permitted by law.\n" ;

   gString gsOut ;
   this->ComposeTitle ( gsOut, false, true ) ;
   this->textOut ( gsOut ) ;
   this->textOut ( freeSoftware ) ;

}  //* End DisplayAppVersion() *

//**************************
//* DisplayCommandLineHelp *
//**************************
//********************************************************************************
//* Display the brief version of command-line help.                              *
//*                                                                              *
//* Input  : helpless : output formatting option                                 *
//*                     'true'  pipe the output through 'less'                   *
//*                     'false' write directly to stdout                         *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************

void Idpp::DisplayCommandLineHelp ( bool helpless )
{
   #define DCLH_Debug (1)     // Enable debugging options

   const char* Help1 = 
   //123456789-123456789-123456789-123456789-123456789-123456789-123456789-123456789-
    "  Apply external CSS style definitions to HTML documents generated by\n"
    "  'makeinfo' utility. Source documents renamed as backup files, e.g.\n"
    "  'home.htm' becomes 'home.htm~'. Modifications written to NEW 'home.htm'.\n\n"
    "Usage  : idpp [OPTIONS][HTML_FILENAMES]\n"
    "             EXAMPLE: idpp -cv --table_border=specify home.html docs.html\n" ;
   const char* Help2 = 
   //123456789-123456789-123456789-123456789-123456789-123456789-123456789-123456789-
    "Options:\n"
    " -i     Interactive mode processing. Prompt for a decision on each item\n"
    "        to be formatted. (automatic formatting is the default)\n"
    " -a     Process all files in the target directory that look like HTML.\n"
    "        recognized filename extensions: html, htm, shtml, shtm, xhtml\n"
    " -d     Specify an alternate source directory.\n"
    "                  (current working directory is the default)\n"
    "        Examples: idpp -d=public_html\n"
    " -f     Specify an alternate filename for the CSS definition file.\n"
    "        ('infodoc-styles.css' in the target directory is the default)\n"
    "        Examples: idpp -f=my-styles.css home.html\n"
    " -c     Contents: process Table of Contents as a multi-level bullet list\n"
    " -r     Remove Table of Contents from the document\n"
    "        (Table of Contents is left unmodified by default)\n"
    " -v     Verbose output, report details of each operation\n"
    " -V     If a file has already been processed, Verify that the file is to\n"
    "        be processed again. (This test is not performed by default.)\n" ;
   const char* Help3 = 
   //123456789-123456789-123456789-123456789-123456789-123456789-123456789-123456789-
    " --bullet_list [=[auto | none | specify]]\n"
    "        'auto'  : automatic formatting of all bullet lists (default)\n"
    "        'none'  : do not apply formatting to bullet lists\n"
    "        'specify: application prompts for formatting option\n"
    " --enum_list [=[auto | none | specify]]\n"
    "        'auto'  : automatic formatting of all enumeration lists (default)\n"
    "        'none'  : do not apply formatting to enumeration lists\n"
    "        'specify: application prompts for enumeration type\n"
    " --table_border [=[auto | none | specify]]\n"
    "         Add a border/grid around the elements of a table.\n"
    "        'auto'   : automatically draw tables with border/gridlines (default)\n"
    "        'none'   : draw tables without border/gridlines\n"
    "        'specify': application prompts for a decision on each table\n" ;
   const char* Help4 = 
   //123456789-123456789-123456789-123456789-123456789-123456789-123456789-123456789-
    " --block_font [=[auto | specify]]\n"
    "        Font size for formatted blocks.\n"
    "        By default, blocks inherit their font size from the parent container,\n"
    "        but font size may be specified during post-processing.\n"
    "        'auto'  : blocks inherit font size from container\n"
    "        'specify': application prompts for font size: (inherit/larger/smaller)\n"
    " --cartouche [=[formatted | flowing | specify]]\n"
    "        A cartouche is a paragraph enclosed within a box.\n"
    "        The text of a cartouche can be pre-formatted (default), or it can be\n"
    "        allowed to flow within the container. The options are:\n"
    "        'formatted: use pre-formatted text in all cartouche objects (default)\n"
    "        'flowing' : use free-flowing text for all cartouche objects\n"
    "        'specify' : application prompts for formatting: (formatted/flowing)\n"
    " --fixed_list - Deprecated (makeinfo v:7 makes this unnecessary)\n"
    "        Do not allow browser to automatically reformat multi-level bullet\n"
    "        lists. (Browsers reformat nested bullet lists by default.)\n" ;
   const char* Help5 = 
   //123456789-123456789-123456789-123456789-123456789-123456789-123456789-123456789-
    " --up_target - Deprecated (makeinfo v:7 deletes this by default.)\n"
    "        If your document is a node on a document tree, specify path to\n"
    "        parent node and optionally the associated display text.\n"
    "        Example: idpp --up_target='../parent_node.htm'         (link only)\n"
    "        Example: idpp --up_target='../parent_node.htm,(home)'  (link and text)\n"
    "        (Up Target is set to top of current page by default.)\n"
    " --my_metadata\n"
    "        Insert custom metadata elements.\n"
    "        Example: idpp --my_metadata=metadata.htm\n"
    "        Contents of the specified file will be copied into the\n"
    "        document's <head> block, just above the </head> tag.\n"
    " --response\n"
    "        Specify a plain text file containing responses to the prompts\n"
    "        displayed by the interactive processing options. Write one token on\n"
    "        each line. Comments (text following a '#') will be ignored.\n"
    "        Example: idpp --response=interactive_responses.txt\n"
//    " --css_mods\n"
//    "        Specify the basic CSS definitions for background color, container\n"
//    "        width, font family, size and color. [NOT YET IMPLEMENTED]\n"
    " ---- ---- ---- ---- ---- ---- ---- ---- ---- ---- ---- ---- ---- ---- ---- \n" ;
   const char* Help6 = 
   //123456789-123456789-123456789-123456789-123456789-123456789-123456789-123456789-
    " --no_mods\n"
    "        Do not perform modifications. Instead, list the operations that\n"
    "        WOULD BE processed for the options specified. (assumes '-v' option)\n"
    " --no_special\n"
    "        Do not perform \"special case\" formatting for bullet lists and\n"
    "        enumeration lists. Default is to apply special processing rules.\n"
    "        (See documentation for details.)\n"
    " --no_html5 - (Obsolete)\n"
    "        Do not update the <!DOCTYPE> tag and various other obsolete HTML\n"
    "        constructs. (updated to HTML5 spec by default)\n"
    " --no_meta\n"
    "        Do not delete the _valid_ <meta> elements in the <head> section.\n"
    "        section. These are: \"application-name\", \"author\", \"description\",\n"
    "        \"generator\", \"keywords\". (all metadata is discarded by default)\n" ;
   const char* Help7 = 
   //123456789-123456789-123456789-123456789-123456789-123456789-123456789-123456789-
    " --no_links\n"
    "        Do not delete the auto-generated <link> elements in the <head>\n"
    "        section (all links are discarded by default)\n"
    " --no_body\n"
    "        Do not update the <body> tag. (extra definitions discarded by default)\n"
    " --no_uplink - Deprecated (makeinfo v:7 deletes this by default.)\n"
    "        Do not modify the target specified in the \"Up\" hyperlink at the top\n"
    "        of the document. (Default is to point the link to top of the document.)\n"
    " --no_block\n"
    "        Do not remove the unnecessary blank line before the block classes\n"
    "        'format', 'display', 'example' and 'lisp'.  (deleted by default)\n"
    " --no_author\n"
    "        For a <blockquote> block followed by an author's name, do not adjust\n"
    "        the horizontal offset of the name. (adjusted by default)\n" ;
   const char* Help8 = 
   //123456789-123456789-123456789-123456789-123456789-123456789-123456789-123456789-
    " --no_cartouche - Deprecated (see --cartouche, above)\n"
    "        Allow text within a cartouche block to \"flow\" (dynamic linebreaks).\n"
    "        In the HTML document, text is preformatted by default (linebreaks \n"
    "        follow the source text).\n"
    " --no_contain\n"
    "        Do not insert the 'infodoc_container' class into the document.\n"
    "        This will cause text to wrap at the edge of the browser window.\n"
    "        (Default is to wrap text at the edge of the container.)\n"
    #if PARA_FIX != 0   // Quick Help
    " --no_paraent\n"
    "        Do not strip the \"&para;\" entities from HTML heading tags.\n"
    "        All such entities are automatically removed by default.\n"
    "        This compensates for a bug in texi2any v:7.1 which inappropriately\n"
    "        adds this entity to HTML heading tags, \"<h[1|2|3|4|5|6]>\"\n"
    "        If this option is specified, then &para; entities ARE NOT removed.\n"
    #endif  // PARA_FIX
    ;
   #if DCLH_Debug != 0     // enable debugging options
   const char* Help9a = 
   //123456789-123456789-123456789-123456789-123456789-123456789-123456789-123456789-
    " ---- ---- ---- ---- ---- ---- ---- ---- ---- ---- ---- ---- ---- ---- ---- \n"
    "                             Debugging Options                              \n"
    " --scan[=START[,END]]\n"
    "     As lines of source data are read, print them to the display.\n"
    "     Starting and ending line number are optional. If not specified,\n"
    "     then all source lines are reported.\n"
    "     If only the starting line is specified, that line and all\n"
    "     subsequent lines will be reported.\n"
    "     If both start and end line numbers are specified (comma separated),\n"
    "     then all source lines in that range (inclusive) will be reported.\n"
    " --book\n"
    "     This option prints a pair of messages which \"bookend\" each text\n"
    "     block, list or other data structure. This helps to locate the point\n"
    "     in the source document where HTML syntax is being misinterpreted.\n" ;
   const char* Help9b = 
   //123456789-123456789-123456789-123456789-123456789-123456789-123456789-123456789-
    " --skip=COUNT\n"
    "     When auto-processing is active, automatically process the first COUNT user\n"
    "     prompts, then switch to interactive mode (see '-i' option).\n"
    "     (This option ignored when a response file is used.)\n"
    " --step[=TENTHS]\n"
    "     Response Step: When reading tokens from the response file, pause for\n"
    "     the specified interval (in 1/10 seconds) after receipt of each token.\n"
    "     range: 2 <= TENTHS <= 50    (default: 10 i.e. 1.0 second)\n"
    "     May be used in combination with the 'scan' option, to delimit the range\n"
    "     in which the delay is active.\n"
    " ---- ---- ---- ---- ---- ---- ---- ---- ---- ---- ---- ---- ---- ---- ---- \n" ;
   #endif   // DCLH_Debug
   const char* Help10 = 
   //123456789-123456789-123456789-123456789-123456789-123456789-123456789-123456789-
    " --version   Display version number and copyright notice.\n"
    " --help, (-h)  Command-line Help\n" 
    " --helpless, (-hl)  Display application help using the 'less' utility.\n" 
    "        For detailed information, see Texinfo documentation:\n"
    "            info -f infodoc.info -n 'Infodoc Post-processor'\n" ;
   //123456789-123456789-123456789-123456789-123456789-123456789-123456789-123456789-

   gString gsTitle ;
   bool write2stdout = true ;
   this->ComposeTitle ( gsTitle ) ;

   //* If specified, pipe the Help text to the 'less' utility.*
   if ( helpless )
   {
      FILE *output = popen ( "less -c", "w" ) ;
      if ( output != NULL )
      {
         gsTitle.append( NEWLINE ) ;
         fprintf ( output, gsTitle.ustr() ) ;
         fprintf ( output, Help1 ) ;
         fprintf ( output, Help2 ) ;
         fprintf ( output, Help3 ) ;
         fprintf ( output, Help4 ) ;
         fprintf ( output, Help5 ) ;
         fprintf ( output, Help6 ) ;
         fprintf ( output, Help7 ) ;
         fprintf ( output, Help8 ) ;
         #if DCLH_Debug != 0     // enable debugging options
         fprintf ( output, Help9a ) ;
         fprintf ( output, Help9b ) ;
         #endif   // DCLH_Debug
         fprintf ( output, Help10 ) ;
         fprintf ( output, "\n" ) ;
         pclose ( output ) ;
         write2stdout = false ;
      }
   }
   //* Write Help text to stdout *
   if ( write2stdout != false )
   {
      this->textOut ( gsTitle ) ;
      this->textOut ( Help1,  false ) ;
      this->textOut ( Help2,  false ) ;
      this->textOut ( Help3,  false ) ;
      this->textOut ( Help4,  false ) ;
      this->textOut ( Help5,  false ) ;
      this->textOut ( Help6,  false ) ;
      this->textOut ( Help7,  false ) ;
      this->textOut ( Help8,  false ) ;
      #if DCLH_Debug != 0     // enable debugging options
      this->textOut ( Help9a,  false ) ;
      this->textOut ( Help9b,  false ) ;
      #endif   // DCLH_Debug
      this->textOut ( Help10 ) ;
   }

   #undef DCLH_Debug
}  //* End DisplayCommandLineHelp() *

//*************************
//*                       *
//*************************
//********************************************************************************
//*                                                                              *
//*                                                                              *
//*                                                                              *
//* Input  :                                                                     *
//*                                                                              *
//* Returns:                                                                     *
//********************************************************************************

