//******************************************************************************
//* File       : idpp_file.cpp                                                 *
//* Author     : Mahlon R. Smith                                               *
//*              Copyright (c) 2014-2020 Mahlon R. Smith, The Software Samurai *
//*                 GNU GPL copyright notice below                             *
//* Date       : 10-Aug-2020                                                   *
//* Version    : (see AppVersion string)                                       *
//*                                                                            *
//* Description: This module contains methods for accessing files to be        *
//* processed, extracting information about each item and creating the         *
//* modified output.                                                           *
//*                                                                            *
//* These methods are adapted from a small subset of the FMgr class written    *
//* for the FileMangler file-management project by the same author.            *
//*                                                                            *
//*                                                                            *
//* Copyright Notice:                                                          *
//* -----------------                                                          *
//* This program is free software: you can redistribute it and/or modify it    *
//* under the terms of the GNU General Public License as published by the Free *
//* Software Foundation, either version 3 of the License, or (at your option)  *
//* any later version, PROVIDED THAT the copyright notices for both code and   *
//* documentation are included and unmodified.                                 *
//*                                                                            *
//* This program is distributed in the hope that it will be useful, but        *
//* WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY *
//* or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License   *
//* for more details.                                                          *
//*                                                                            *
//* You should have received a copy of the GNU General Public License along    *
//* with this program.  If not, see <http://www.gnu.org/licenses/>.            *
//*                                                                            *
//*         Full text of the GPL License may be found in the TexInfo           *
//*         documentation for this program under 'Copyright Notice'.           *
//******************************************************************************
//* Version History (see idpp.cpp)                                             *
//*                                                                            *
//******************************************************************************

//****************
//* Header Files *
//****************
#include "idpp.hpp"

//**********************
//* Non-member methods *
//**********************
static void StripTags ( gString& disp ) ;

//****************
//* Local Data   *
//****************
//* If set, remove the useless comments inserted by the converter.*
//* It is assumed that this comment was some kind of debug marker.*
// Programmer's Note: Because few source lines contain instances of the target 
// substring (or multiple instances) enabling this flag causes a small 
// performance hit. If performance becomes an issue, set to zero (0).
#define REMOVE_STUPID_COMMENT (1)

//* Const strings for scanning files *
const wchar_t* topDOCT = L"<!DOCTYPE HTML" ;
const short    topDOCT_len = 14 ;
const wchar_t* topHTML = L"<HTML" ;
const short    topHTML_len = 5 ;
const wchar_t* cssCOPY = L"Copyright" ;
const short    cssCOPY_len = 9 ;
const wchar_t* cssSAMU = L"The Software Samurai" ;
const short    cssSAMU_len = 20 ;
const wchar_t* cssVERS = L"Version: " ;
const short    cssVERS_len = 9 ;
const wchar_t* headTag = L"<head" ;
const short    headTag_len = 5 ;
const wchar_t* headendTag = L"</head" ;
const short    headendTag_len = 6 ;
const wchar_t* titleTag = L"<title" ;
const short    titleTag_len = 6 ;
const wchar_t* bodyTag = L"<body" ;
const short    bodyTag_len = 5 ;
const wchar_t* bodyendTag = L"</body" ;
const short    bodyendTag_len = 6 ;
const wchar_t* indexTarget = L"<span id=\"Index-1\"></span>" ;
const short    indexTarget_len = 22 ;

const wchar_t* tocHead = L"<h2 class=\"contents-heading\">Table of Contents</h2>" ;
const short    tocHead_len = 51 ;
const wchar_t* tocTarget1 = L"<span id=\"SEC_Contents\"></span>" ;
const short    tocTarget1_len = 31 ;
const wchar_t* tocTarget2 = L"<span id=\"Top\">" ;
const short    tocTarget2_len = 15 ;
const wchar_t* tocTarget3 = L"<span id=\"index-" ;
const short    tocTarget3_len = 16 ;
const wchar_t* tocTitle = L"<h1 class=\"settitle\"" ;
const wchar_t* tocMenuTitle = L"<h1 class=\"top\">" ;
const short    tocMenuTitle_len = 16 ;

const wchar_t* headerClass = L"<div class=\"header\">" ;
const short    headerClass_len = 20 ;
const wchar_t* headerLinks = L"Next: <a href=\"" ;
const short    headerLinks_len = 15 ;

const wchar_t* ulBegin = L"<ul" ;
const short    ulBegin_len = 3 ;
const wchar_t* ulEnd = L"</ul>" ;
const short    ulEnd_len = 5 ;

const wchar_t* olBegin = L"<ol" ;
const short    olBegin_len = 3 ;
const wchar_t* ulNobull = L"<ul class=\"no-bullet\">" ;
const short    ulNobull_len = 22 ;
const wchar_t* olEnd = L"</ol>" ;
const short    olEnd_len = 5 ;
const wchar_t* liOpen  = L"<li" ;
const short    liOpen_len = 3 ;
const wchar_t* liBegin = L"<li>" ;
const short    liBegin_len = 4 ;
const wchar_t* liLongBegin = L"</li><li>" ;
const short    liLongBegin_len = 9 ;
const wchar_t* paraEnd = L"</p>" ;
const short    paraEnd_len = 4 ;
const wchar_t* tabBegin = L"<table" ;
const short    tabBegin_len = 6 ;
const wchar_t* tabSimple = L"<table>" ;
const short    tabSimple_len = 7 ;
const char*    commBegin = "<!--" ;    // HTML comment delimiters
const char*    commEnd   = "-->" ;
const wchar_t* stupidComment = L"<!-- /@w -->" ; // meaningless random comment

const wchar_t* indeInherit = L"<blockquote class=\"indentedblock\">" ;
const wchar_t* indeSmall   = L"<blockquote class=\"smallindentedblock\">" ;
const wchar_t* indeLarge   = L"<blockquote class=\"largeindentedblock\">" ;
const wchar_t* quotSimple  = L"<blockquote>" ;
const wchar_t* quotInherit = L"<blockquote class=\"quotation\">" ;
const wchar_t* quotSmall   = L"<blockquote class=\"smallquotation\">" ;
const wchar_t* quotLarge   = L"<blockquote class=\"largequotation\">" ;
const wchar_t* quotEND     = L"</blockquote>" ;
const wchar_t* tagClose    = L">" ;
const wchar_t degreeSymbol = L'\x00B0' ;
const wchar_t discBullet   = L'\x23FA' ;     // Texinfo/HTML disc bullet
const wchar_t cirtexBullet = L'\x26AC' ;     // Texinfo circle bullet
const wchar_t circleBullet = L'\x26AA' ;     // HTML circle bullet (looks bad in info)
const wchar_t squareBullet = L'\x25AA' ;     // Texinfo/HTML square bullet

//* Const output strings *
const char* stdDOCT = "<!DOCTYPE HTML>" ;
const char* stdHTML = "<html>" ;
const char* cssLINK0 = 
   "<!-- Post-processing of this document performed using %S v:%S -->" ;
const char* cssLINK1 = 
   "<meta charset=\"utf-8\" />  <!-- Import the global stylesheet -->\n"
   "<link rel=\"stylesheet\" href=\"" ;
const char* cssLINK2 = 
   "\" lang=\"en\" type=\"text/css\"/>\n" ;
const char* stdBODY = "<body>" ;
const wchar_t* begCONTAINER = L"<div class=\"infodoc_container\">" ;
const short begCONTAINER_len = 31 ;
const wchar_t* endCONTAINER = L"</div>    <!-- infodoc_container -->" ;
const short endCONTAINER_len = 36 ;
const char* tocDISC   = "<ul class=\"toc-level1\">" ;
const char* tocCIRCLE = "<ul class=\"toc-level2\">" ;
const char* tocSQUARE = "<ul class=\"toc-level3\">" ;
const char* ulDISC   = "<ul class=\"disc-bullet\">" ;
const char* ulCIRCLE = "<ul class=\"circle-bullet\">" ;
const char* ulSQUARE = "<ul class=\"square-bullet\">" ;

//* Placeholder user response. (see 'userResponse' method for details) *
const wchar_t* dToken = L"default_token" ;
const short    dToken_len = 13 ;



//*************************
//*   ppfProcessSrcHTML   *
//*************************
//******************************************************************************
//* Convert raw HTML to CSS-styled HTML.                                       *
//*                                                                            *
//* Input  : src : (by reference) path/filename of source data                 *
//*          trg : (by reference) path/filename of target                      *
//*                                                                            *
//* Returns: OK if successful, ERR processing error or user aborted process    *
//******************************************************************************
//* Programmer's Note: The TexInfo HTML converter always outputs a DOCTYPE     *
//* statement; however, in HTML3, it was allowed to omit it. For this reason,  *
//* we test for both with and without (just to be safe).                       *
//* If the DOCTYPE statement is missing, then the <html> tag probably contains *
//* some definitions similar to those in the converter's DOCTYPE line. Thus,   *
//* if 'no_html5' != false we keep them, else we output a standard <html> tag. *
//*                                                                            *
//* The monospace font tag <tt></tt> (teletype) is obsolete. Unfortunately,    *
//* texi2any still uses it to specify monospace text.                          *
//* Thus, if 'no_html5' == false we replace all <tt>...</tt> sequences with    *
//* a CSS "font-family" specification. If 'no_html5' != false, we leave them   *
//* unprocessed.                                                               *
//******************************************************************************

short Idpp::ppfProcessSrcHTML ( const gString& src, const gString& trg )
{
   //* VALID metadata that may be seen in the <head> section *
   const wchar_t* metaApp = L"<meta name=\"application-name" ;
   const short    metaApp_len = 28 ;
   const wchar_t* metaAuth = L"<meta name=\"author" ;
   const short    metaAuth_len = 18 ;
   const wchar_t* metaDesc = L"<meta name=\"description" ;
   const short    metaDesc_len = 23 ;
   const wchar_t* metaGen = L"<meta name=\"generator" ;
   const short    metaGen_len = 21 ;
   const wchar_t* metaKeys = L"<meta name=\"keywords" ;
   const short    metaKeys_len = 20 ;
   const wchar_t* metaDate = L"<meta name=\"date\"" ;
   const short    metaDate_len = 17 ;
   const wchar_t* linkBegin = L"<link" ;
   const short    linkBegin_len = 5 ;
   const wchar_t* gplBegin = L"<h3 class=\"unnumberedsec\">GNU General Public License</h3>" ;
   const wchar_t* fdlBegin = L"<h3 class=\"unnumberedsec\">GNU Free Documentation License</h3>" ;

   this->slCount = ZERO ;     // reset source-line counter (zero-based counter)
   this->tlCount = ZERO ;     // reset target-line counter (zero-based counter)
   short status = OK ;        // return value

   //* Open the source file, and                      *
   //* if processing is enabled, open the target file.*
   this->ifs.open ( src.ustr(), ifstream::in ) ;
   if ( this->no_mods == false )
      this->ofs.open ( trg.ustr(), ofstream::trunc ) ;

   //* Access to source and target? *
   if ( this->ifs.is_open() && ((this->ofs.is_open()) || this->no_mods != false) )
   {
      gString gs,             // conversion to wide text
              gsOut,          // output data for target file
              gsBook ;        // for debugging output only
      short wi ;              // index into source line

      //* Read the first source line (validates document as HTML) *
      if ( (status = this->ppfReadSrcLine ( gs, wi )) == OK )
      {  //* 'true' if document does not have a doctype statement.*
         bool dtMissing = ((gs.compare( topHTML, false, topHTML_len, wi )) == ZERO ) ;
         //* 'true' if document has a valid doctype statement.*
         bool dtValid = ((gs.compare( topDOCT, false, topDOCT_len, wi )) == ZERO) ;
         if ( dtMissing || (dtValid && this->no_html5 == false) )
         {
            gsOut = stdDOCT ;
            this->ppfWriteLine ( gsOut ) ;

            //* If first line was a valid <html> tag (see note above) *
            if ( dtMissing )
            {
               if ( this->no_html5 == false ) // output a standard tag
                  gsOut = stdHTML ;
               else                          // output the original line
                  gsOut = gs ;
               this->ppfWriteLine ( gsOut ) ;
            }
         }
         //* Else if user wants to keep original DOCTYPE line *
         else if ( dtValid && this->no_html5 )
            this->ppfWriteLine ( gs ) ;
         else
            status = ERR ;
      }

      //* Copy source lines to target until '<head>' tag copied *
      while ( status == OK )
      {
         if ( (status = this->ppfReadSrcLine ( gs, wi )) == OK )
         {
            this->ppfWriteLine ( gs.ustr() ) ;
            if ( (gs.compare( headTag, false, headTag_len, wi )) == ZERO )
            {
               if ( this->book != false )
               {
                  gsBook.compose( L"ML(%hu) <HEAD>", &this->slCount ) ;
                  this->textOut ( gsBook ) ;
               }
               break ;     // <head> found and copied
            }
         }
      }

      //* Discard everything inside <head></head> EXCEPT: retain <title> tag, *
      //* and IF specified, retain the special metadata and/or links.         *
      while ( status == OK )
      {
         if ( (status = this->ppfReadSrcLine ( gs, wi )) == OK )
         {
            if ( (gs.compare( titleTag, false, titleTag_len, wi )) == ZERO )
            {  //* <title> found, insert link to CSS definition file. *
               gsOut.compose( cssLINK0, AppTitle1, AppVersion ) ;
               this->ppfWriteLine ( gsOut ) ;
               gString gscf ;
               gsOut.compose( L"%s%S%s", cssLINK1, this->cssFile.gstr(), cssLINK2 ) ;
               this->ppfWriteLine ( gsOut ) ;   // write file link to target
               this->tlCount += 2 ; // this is a 3-line sequence
               this->ppfWriteLine ( gs ) ;      // copy the title to target
               // NOTE: We assume that the entire title is on one line.
            }

            //* If end of <head></head> block found *
            else if ( (gs.compare( headendTag, false, headTag_len, wi )) == ZERO )
            {
               //* If specified, insert user's custom metadata into the    *
               //* <head>...</head> block. We DO NOT validate these data.*
               if ( (this->userMeta.gschars()) > 1 )
               {
                  if ( this->verbose != false )
                  {
                     gString gsVerb( "(%4hu) Inserting custom data at line %hu.", 
                                     &this->slCount, &this->tlCount ) ;
                     this->textOut ( gsVerb ) ;
                  }

                  //* Copy data from user file to target *
                  status = this->ppfInsertCustomData ( this->userMeta ) ;
               }

               //* Write the end-of-head tag *
               this->ppfWriteLine ( gs.ustr() ) ;
               if ( this->book != false )
               {
                  gsBook.compose( L"ML(%hu) </HEAD>", &this->slCount ) ;
                  this->textOut ( gsBook ) ;
               }
               break ;        // our <head> is now (nearly :) empty
            }

            //* If user wants to retain the VALID metadata *
            else if ( (this->no_meta != false) &&
                      (((gs.compare( metaDesc, false, metaDesc_len, wi )) == ZERO) ||
                      ((gs.compare( metaKeys, false, metaKeys_len, wi )) == ZERO)  ||
                      ((gs.compare( metaAuth, false, metaAuth_len, wi )) == ZERO)  ||
                      ((gs.compare( metaApp, false, metaApp_len, wi )) == ZERO)    ||
                      ((gs.compare( metaGen, false, metaGen_len, wi )) == ZERO)    ||
                      ((gs.compare( metaDate, false, metaDate_len, wi )) == ZERO)) )
            {
               status = this->ppfProcHEAD ( gs ) ;
            }

            //* If user wants to retain the <link> entries *
            else if ( (this->no_link != false) &&
                      ((gs.compare( linkBegin, false, linkBegin_len, wi )) == ZERO) )
            {
               status = this->ppfProcHEAD ( gs ) ;
            }

            else
            { /* Discard the line */ }
         }
      }

      //* Copy lines until we find the <body> tag *
      while ( status == OK )
      {
         if ( (status = this->ppfReadSrcLine ( gs, wi )) == OK )
         {
            if ( (gs.compare( bodyTag, false, bodyTag_len, wi )) == ZERO )
            {  //* <body> found, replace it with a lean <body>. *
               if ( this->no_body == false )
                  gsOut = stdBODY ;
               else
                  gsOut = gs ;
               this->ppfWriteLine ( gsOut ) ;

               if ( this->book != false )
               {
                  gsBook.compose( L"ML(%hu) <BODY>", &this->slCount ) ;
                  this->textOut ( gsBook ) ;
               }

               //* Establish the container just inside the <body> tag. *
               if ( this->no_cont == false )
               {
                  gsOut = begCONTAINER ;
                  this->ppfWriteLine ( gsOut ) ;
               }
               break ;
            }
            this->ppfWriteLine ( gs ) ;
         }
      }

      //* Copy lines until we reach the Table of Contents (if any) *
      //*    a) tocTarget1 is present if TOC is present.           *
      //*    b) tocTarget2 is always present.                      *
      short blankCount = ZERO ;
      while ( status == OK )
      {
         if ( (status = this->ppfReadSrcLine ( gs, wi )) == OK )
         {
            if ( (gs.compare( tocTarget1, false, tocTarget1_len, wi )) == ZERO )
            {  //* Table of Contents intra-page target found. *
               //* Copy it, then process Table of Contents.   *
               this->ppfWriteLine ( gs ) ;
               status = this->ppfProcTOC ( true ) ;
               break ;
            }
            else if ( (gs.compare( tocTarget2, false, tocTarget2_len, wi )) == ZERO )
            {  //* Table of Contents not present. Copy intra-page *
               //* target, then process the page header ONLY.     *
               this->ppfWriteLine ( gs ) ;
               status = this->ppfProcTOC ( false ) ;
               break ;
            }
            else if ( (this->tocDel != false) &&
                      ((gs.find( tocTitle )) >= ZERO) )
            {  //* If discarding Table of Contents, *
               //* then also discard the TOC title. *
               continue ;
            }
            else if ( gs.gschars() == 1 )
            {  //* For some reason the texi-to-HTML converter inserts 41 *
               //* blank lines into the file between the start of the    *
               //* <body> and the Table of Contents. Delete some of them.*
               if ( ++blankCount > 5 )
                  continue ;
            }

            //* If container class was established on a previous pass, *
            //* delete the old copy to avoid duplication.              *
            if ( (gs.compare( begCONTAINER, false, begCONTAINER_len, wi )) == ZERO )
               continue ;

            this->ppfWriteLine ( gs ) ;   // copy the line from source to target
         }
      }

      //*****************************************************************
      //* Copy lines until we reach the end of the <body></body> block. *
      //*****************************************************************
      blkType bType = btNone ;   // type of block found
      while ( status == OK )
      {
         if ( (status = this->ppfReadSrcLine ( gs, wi )) == OK )
         {  //* If </body> (end-of-body) tag found *
            if ( (gs.compare( bodyendTag, false, bodyendTag_len, wi )) == ZERO )
            {
               if ( this->book != false )
               {
                  gsBook.compose( L"ML(%hu) </BODY>", &this->slCount ) ;
                  this->textOut ( gsBook ) ;
               }

               //* Insert close of container just above it. *
               if ( this->no_cont == false )
               {
                  gsOut = endCONTAINER ;
                  this->ppfWriteLine ( gsOut ) ;
               }
               this->ppfWriteLine ( gs ) ;
               break ;
            }

            //* Process embedded HTML comment in the source file.*
            else if ( (this->ppfTestComment ( gs )) != false )
            {
               this->ppfProcComment ( gs ) ;
               // Programmer's Note: Return value is currently ignored.
               // See notes in called method.
               status = OK ;
            }

            //* If beginning an indentedblock, smallindentedblock *
            //* or largeindentedblock.                            *
            else if ( (this->ppfTestIndentedBlock ( bType, gs )) != false )
            {
               if ( this->book != false )
               {
                  gsBook.compose( L"ML(%hu) IBLOCK START", &this->slCount ) ;
                  this->textOut ( gsBook, false ) ;
               }

               status = this->ppfProcIDB ( bType, gs ) ;

               if ( this->book != false )
               {
                  gsBook.compose( L"--(%hu) IBLOCK END", &this->slCount ) ;
                  this->textOut ( gsBook ) ;
               }
            }

            //* If beginning a quotation, smallquotation   *
            //* or largequotation block.                   *
            else if ( (this->ppfTestQuotationBlock ( bType, gs )) != false )
            {
               if ( this->book != false )
               {
                  gsBook.compose( L"ML(%hu) QBLOCK START", &this->slCount ) ;
                  this->textOut ( gsBook, false ) ;
               }

               status = this->ppfProcQUOTE ( bType, gs ) ;

               if ( this->book != false )
               {
                  gsBook.compose( L"--(%hu) QBLOCK END", &this->slCount ) ;
                  this->textOut ( gsBook ) ;
               }
            }

            //* If beginning a verbatim, smallverbatim or largeverbatim block *
            else if ( (this->ppfTestVerbatimBlock ( bType, gs )) != false )
            {
               status = this->ppfProcVerbatim ( bType, gs ) ;
            }

            //* If beginning of a preformatted block is detected: *
            //* 'display, 'format', 'example', 'lisp', and their  *
            //* 'small' or 'large' counterparts.                  *
            else if ( (this->ppfTestFormattedBlock ( bType, gs )) != false )
            {
               if ( this->book != false )
               {
                  gsBook.compose( L"ML(%hu) PBLOCK START", &this->slCount ) ;
                  this->textOut ( gsBook, false ) ;
               }

               //* Copy the block, optionally eliminating           *
               //* the unnecessary leading blank line.              *
               //* NOTE: Data inside these blocks ARE NOT modified. *
               status = this->ppfProcFormattedBlock ( bType, gs ) ;

               if ( this->book != false )
               {
                  gsBook.compose( L"--(%hu) PBLOCK END", &this->slCount ) ;
                  this->textOut ( gsBook ) ;
               }
            }

            //* For <ul> lists *
            else if ( (this->ppfTestUlistBegin ( gs, wi )) != false )
            {
               if ( this->book != false )
               {
                  gsBook.compose( L"ML(%hu) UBLOCK START", &this->slCount ) ;
                  this->textOut ( gsBook, false ) ;
               }

               //* Process the (possibly nested) <ul> / <ol> lists.*
               status = this->ppfProcUL ( gs ) ;

               if ( this->book != false )
               {
                  gsBook.compose( L"--(%hu) UBLOCK END", &this->slCount ) ;
                  this->textOut ( gsBook ) ;
               }
            }

            //* For <ol> lists *
            else if ( (this->ppfTestOlistBegin ( gs, wi )) != false )
            {
               if ( this->book != false )
               {
                  gsBook.compose( L"ML(%hu) OBLOCK START", &this->slCount ) ;
                  this->textOut ( gsBook, false ) ;
               }

               //* Process the (possibly nested) <ol> / <ul> lists.*
               status = this->ppfProcOL ( gs ) ;

               if ( this->book != false )
               {
                  gsBook.compose( L"--(%hu) OBLOCK END", &this->slCount ) ;
                  this->textOut ( gsBook ) ;
               }
            }

            //* If a <table> object found *
            else if ( (gs.find( tabBegin, wi, false )) >= ZERO )
            {
               status = this->ppfProcTAB ( gs ) ;
            }

            //* If the GNU General Public License or GNU Free  *
            //* Documentation License is detected, process it. *
            else if (   ((gs.find( gplBegin, wi, false )) >= ZERO)
                     || ((gs.find( fdlBegin, wi, false )) >= ZERO) )
            {
               this->ppfWriteLine ( gs ) ;      // write the unmodified line
               bool gpl = true ;
               if ( (gs.find( fdlBegin, wi, false )) >= ZERO )
                  gpl = false ;
               status = this->ppfProcGNU ( gpl ) ;
            }

            //* If we have reached the beginning of the Index node. *
            //*          (this is below the node header)            *
            else if ( (gs.find( indexTarget, wi, false )) >= ZERO )
            {
               if ( this->book != false )
               {
                  gsBook.compose( L"ML(%hu) XBLOCK START", &this->slCount ) ;
                  this->textOut ( gsBook, false ) ;
               }
               
               this->ppfWriteLine ( gs ) ;      // write the target line
               status = this->ppfProcINDEX () ;

               if ( this->book != false )
               {
                  gsBook.compose( L"--(%hu) XBLOCK END", &this->slCount ) ;
                  this->textOut ( gsBook ) ;
               }
            }

            //* If container class was established on a previous pass, *
            //* delete the old copy to avoid duplication.              *
            else if ( (gs.find( endCONTAINER, wi, false )) >= ZERO )
            {
               /* do nothing */
            }

            else        // this line gets no processing
               this->ppfWriteLine ( gs ) ;
         }
      }

      //* Finish copying the file *
      while ( status == OK )
      {
         if ( (this->ppfReadSrcLine ( gs, wi )) == OK )
            this->ppfWriteLine ( gs ) ;
         else        // last source line copied to target
            break ;
      }
   }
   if ( this->ifs.is_open() )
      this->ifs.close() ;           // close the source file
   if ( this->ofs.is_open() )
      this->ofs.close() ;           // close the target file
   return status ;

}  //* End ppfProcessSrcHTML()

//*************************
//*      ppfProcTOC       *
//*************************
//******************************************************************************
//* Called by ppfProcessSrcHTML() to process the Table Of Contents list.       *
//* 1) If 'tocDel' == false :                                                  *
//*    a) If 'tocMod' == false :                                               *
//*       Copy Table of Contents unmodified.                                   *
//*    b) If 'tocMod' != false :                                               *
//*       Convert the "no-bullet" classes within the Table of Contents to      *
//*       the appropriate level of bullet class.                               *
//*       - 1st level "no-bullet" class becomes toc-level1" class              *
//*       - 2nd level "no-bullet" class becomes toc-level2" class              *
//*       - 3rd and lower levels "no-bullet" class becomes toc-level3" class   *
//* 2) If 'tocDel' != false :                                                  *
//*    Delete the entire Table of Contents.                                    *
//* 3) If user has specified an alternate "up" target, insert it into the      *
//*    "up" link. Else, just insert a '#' character, so "up" just goes to the  *
//*    Table Of Contents on the same page. Note that if user has specified     *
//*    that the TOC be removed, the link will still work.                      *
//* 4) If TOC is not present at the top of the document, don't process it,     *
//*    BUT, do other housekeeping chores.                                      *
//*                                                                            *
//* Input  : tocFound : 'true' if caller found the TOC, else 'false'           *
//*                                                                            *
//* Returns: OK if successful, ERR processing error                            *
//******************************************************************************

short Idpp::ppfProcTOC ( bool tocFound )
{
   const wchar_t* upLink = L"Up: <a href=\"" ;
   const short    upLink_len = 13 ;

   gString gs,             // conversion to wide text
           gsOut ;         // output data for target file
   short wi, wx,
         tocLevel = ZERO,  // nesting level for TOC line items
         trgs = tocFound ? 2 : 1 ; // when this reaches zero, TOC is processed
   short status = OK ;     // return value

   while ( status == OK )
   {
      if ( (status = this->ppfReadSrcLine ( gs, wi )) == OK )
      {
         if ( ((gs.compare( tocTarget2, false, tocTarget2_len, wi )) == ZERO) ||
              ((gs.compare( tocTarget3, false, tocTarget3_len, wi )) == ZERO) )
         {
            this->ppfWriteLine ( gs ) ;
            if ( --trgs == ZERO )   // count intra-page targets found
               break ;
         }

         //* The source document may, or may not have node headers. If it   *
         //* does, then the "UP" link of the first header points into space.*
         //* Unless user has specified that we should leave it alone, adjust*
         //* link to point to the top of the current page, or to the        *
         //* user-specified target.                                         *
         else if ( (gs.compare( headerLinks, false, headerLinks_len, wi )) == ZERO )
         {
            //* If not prohibited OR if user specified an alternate link target*
            if ( this->no_utrg == false || this->upTarg != false )
            {
               //* Scan for "Up" link specification *
               if ( ((wi = gs.find( upLink )) >= ZERO) && 
                    ((gs.compare( upLink, false, upLink_len, wi )) == ZERO) )
               {
                  wi += upLink_len ;
                  if ( (wx = gs.find( L'"', wi )) > wi ) // isolate the original target path
                     gs.erase( wi, (wx - wi) ) ;         // erase original target path
                  gs.insert ( this->utPath, wi ) ;       // insert user's target path
                  if ( (wi = gs.after( L'>', wi )) > ZERO ) // find the end of current tag
                  {
                     if ( (wx = (gs.find( L'<', wi ))) > wi ) // isolate the original display text
                        gs.erase( wi, (wx - wi) ) ;      // erase original display text
                     gs.insert( this->utText, wi ) ;
                  }
               }
            }
            this->ppfWriteLine ( gs ) ;
            //* Copy the remainder of the page header *
            while ( (gs.find( L"</div>" )) && (status == OK) )
            {
               if ( (status = this->ppfReadSrcLine ( gs, wi )) == OK )
                  this->ppfWriteLine ( gs ) ;
            }
            break ;     // page header processed, end of TOC page
         }
         else if ( this->tocDel == false )
         {
            //* If user specified that TOC should be treated as an    *
            //* unordered lists, process it. Else copy TOC unmodified.*
            if ( this->tocMod != false )
            {
               if ( (gs.compare( ulNobull, false, ulNobull_len, wi )) == ZERO )
               {
                  gs.limitChars( wi ) ;
                  if ( ++tocLevel == 1 )
                     gs.append( tocDISC ) ;
                  else if ( tocLevel == 2 )
                     gs.append( tocCIRCLE ) ;
                  else
                     gs.append( tocSQUARE ) ;
                  this->ppfWriteLine ( gs ) ;
               }
               else
               {
                  this->ppfWriteLine ( gs ) ;
                  if ( (tocLevel > ZERO) &&
                       ((gs.compare( ulEnd, false, ulEnd_len, wi )) == ZERO) )
                  {
                     --tocLevel ;
                  }
               }
            }
            else
               this->ppfWriteLine ( gs ) ;
         }
         else if ( this->tocDel != false &&
                   ((gs.find( tocMenuTitle, wi, false, tocMenuTitle_len )) >= ZERO) )
         {  //* Retain the title of the Main Menu node. *
            this->ppfWriteLine ( gs ) ;
         }
      }
   }
   return status ;

}  //* End ppfProcTOC() *

//*************************
//*     ppfProcINDEX      *
//*************************
//******************************************************************************
//* Called by ppfProcessSrcHTML() to process the Index table.                  *
//* The index header and the index target node have been written by caller.    *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: OK if successful, ERR processing error                            *
//******************************************************************************

short Idpp::ppfProcINDEX ( void )
{
   const char* indexJUMPTO = " class=\"jumpto\"" ;
   //* This is what 'Jump to:' will look like if it's already been processed.*
   const wchar_t* fixedTable = L"<table class=\"jumpto\">" ;
   const short fixedTable_len = 22 ;

   gString gs ;            // conversion to wide text
   short wi,
         trgs = 2 ;        // when this reaches zero, Index is processed
   short status = OK ;     // return value

   while ( status == OK )
   {
      if ( (status = this->ppfReadSrcLine ( gs, wi )) == OK )
      {
         //* If unstyled "<table>" found, assign it to the "jumpto" class *
         if ( (gs.compare( tabSimple, false, tabSimple_len, wi )) == ZERO )
         {
            if ( (wi = gs.find( tagClose, wi )) >= ZERO )
               gs.insert( indexJUMPTO, wi ) ;
            this->ppfWriteLine ( gs ) ;
            if ( --trgs == ZERO )
               break ;
         }
         else
         {
            // Programmer's Note: This includes the remainder of the top "jumpto" 
            // table, the inner table: <table class="index-cp" border="0">
            // and its contents.
            this->ppfWriteLine ( gs ) ;   // just copy the line

            //* If index table already processed, move on.*
            //* This prevents double processing.          *
            if ( (gs.compare( fixedTable, false, fixedTable_len, wi )) == ZERO )
            {
               if ( --trgs == ZERO )
                  break ;
            }
         }
      }
   }
   return status ;

}  //* End ppfProcINDEX() *

//*************************
//*      ppfProcIDB       *
//*************************
//******************************************************************************
//* Called by ppfProcessSrcHTML() to process 'indentedblock' and               *
//* 'smallindentedblock' sequences.                                            *
//*                                                                            *
//* Input  : bType  : indicates block type: member of enum blkType             *
//*          gsib  : (by reference) contains the first line of the block.      *
//*                                                                            *
//* Returns: OK if successful, ERR processing error                            *
//******************************************************************************
//* Programmer's Note: It is possible that this method will be called          *
//* recursively, so be aware.                                                  *
//*                                                                            *
//* Programmer's Note: Processing this block separately is not actually        *
//* necessary because everthing is processed smoothly from the main loop;      *
//* however, it is potentially more robust to process the contents here.       *
//*                                                                            *
//******************************************************************************

short Idpp::ppfProcIDB ( blkType bType, const gString& gsib )
{
   const wchar_t* blockName[] = 
   {
      L"Indentedblock",   L"smallIndentedblock",   L"largeIndentedblock",
   } ;

   gString gs = gsib,         // text formatting
           gsNext ;           // context line
   wchar_t usrResp = L'a';    // user response to prompt
   short wi = ZERO,           // string index
         bi,                  // index of <blockquote...  tag
         bnIndex = ZERO,      // index into blockName[] array
         status = OK ;        // return value
   bool  otherClass = false ; // 'true' if existing class callout in source

   //* Identify the specified block type, then erase the existing tag *
   if ( (bi = gs.find( indeInherit )) >= ZERO )
      gs.erase( indeInherit, bi ) ;
   else if ( (bi = gs.find( indeSmall )) >= ZERO )
   { otherClass = true ; gs.erase( indeSmall, bi ) ; }
   else if ( (bi = gs.find( indeLarge )) >= ZERO )
   { otherClass = true ; gs.erase( indeLarge, bi ) ; }

   //* If user interaction is enabled, *
   //* prompt for user's selection.    *
   if ( this->blkFont == cfgSpec )
   {
      //* Index the block-name text for display *
      if      ( bType == stdI ) bnIndex = ZERO ;
      else if ( bType == smaI ) bnIndex = 1 ;
      else if ( bType == lrgI ) bnIndex = 2 ;

      //* If there is display text on this line (unlikely), use it *
      if ( gs.gstr()[bi] != NULLCHAR )
         gsNext = &gs.gstr()[bi] ;
      //* Else read the next line to give user some context.       *
      else
      {
         status = this->ppfReadSrcLine ( gsNext, wi ) ;

         //* Push the context line back into the stream      *
         //* format the text portion of the line for display.*
         this->ppfUnReadSrcLine ( gsNext ) ;
         gsNext.erase( L"<p>", ZERO, false ) ;
         gsNext.erase( L"<br>", ZERO, false ) ;
      }

      //* Ask the user to select a font-size option *
      usrResp = this->ppfBlockPrompt ( blockName[bnIndex], gsNext.gstr() ) ;
   }

   //* Update the tag according to user's selection *
   if ( bi >= ZERO )
   {
      switch ( usrResp )
      {
         //** inherited font **
         case L'i':  gs.insert( indeInherit, bi ) ; break ;
         //** smaller font **
         case L's':  gs.insert( indeSmall, bi ) ;   break ;
         //** larger font **
         case L'l':  gs.insert( indeLarge, bi ) ;   break ;
         //** Automatic assignment of class       **
         //** If font size previously specified,  **
         //** use it, else use standard font size.**
         case L'a':
         default:
            if ( bType == smaI )
               gs.insert( indeSmall, bi ) ;
            else if ( bType == lrgI )
               gs.insert( indeLarge, bi ) ;
            else
               gs.insert( indeInherit, bi ) ;
            break ;
      } ;

      //* During a second pass, the existing class *
      //* designation overrides user response.     *
      if ( otherClass && (this->respFile.gschars() > 1) )
         gs = gsib ;    // copy line as-written
   }
   else
   {
      gs = gsib ;       // copy line as-written
      status = ERR ;
   }

   //* Write the header line *
   this->ppfWriteLine ( gs ) ;

   //* Process the body of the block *
   bool done = false ;
   while ( ! done && status == OK )
   {  //* If at the end of the block, write the line and jump out
      if ( (status = this->ppfReadSrcLine ( gs, wi )) == OK )
      {
         if ( (gs.find( quotEND, ZERO, false )) >= ZERO )
         {
            this->ppfWriteLine ( gs ) ;
            done = true ;
         }

         //* If beginning an indentedblock, smallindentedblock *
         //* or largeindented block.                           *
         else if ( (this->ppfTestIndentedBlock ( bType, gs )) != false )
         {
            status = this->ppfProcIDB ( bType, gs ) ;
         }

         //* If beginning a quotation, smallquotation   *
         //* or largequotation block.                   *
         else if ( (this->ppfTestQuotationBlock ( bType, gs )) != false )
         {
            status = this->ppfProcQUOTE ( bType, gs ) ;
         }

         //* If beginning a verbatim, smallverbatim or largeverbatim block *
         else if ( (this->ppfTestVerbatimBlock ( bType, gs )) != false )
         {
            status = this->ppfProcVerbatim ( bType, gs ) ;
         }

         //* If beginning of a preformatted block is detected: *
         //* 'display, 'format', 'example', 'lisp', and their  *
         //* 'small' or 'large' counterparts.                  *
         else if ( (this->ppfTestFormattedBlock ( bType, gs )) != false )
         {
            //* Copy the block, optionally eliminating           *
            //* the unnecessary leading blank line.              *
            //* NOTE: Data inside these blocks ARE NOT modified. *
            status = this->ppfProcFormattedBlock ( bType, gs ) ;
         }

         //* If a <table> object found *
         else if ( (gs.find( tabBegin, wi, false )) >= ZERO )
         {
            status = this->ppfProcTAB ( gs ) ;
         }

         //* For <ul> lists *
         else if ( (this->ppfTestUlistBegin ( gs, wi )) != false )
         {
            //* Process the (possibly nested) <ul> / <ol> lists.*
            status = this->ppfProcUL ( gs ) ;
         }

         //* For <ol> lists *
         else if ( (this->ppfTestOlistBegin ( gs, wi )) != false )
         {
            //* Process the (possibly nested) <ol> / <ul> lists.*
            status = this->ppfProcOL ( gs ) ;
         }

         else     // copy the line without processing
            this->ppfWriteLine ( gs ) ;
      }
   }
   return status ;

}  //* End ppfProcIDB() *

//*************************
//*     ppfProcQUOTE      *
//*************************
//******************************************************************************
//* Called by ppfProcessSrcHTML() to process <blockquote> sequences.           *
//* This includes:                                                             *
//*    <blockquote>  (unprocessed @quotation) PLUS:                            *
//*    <blockquote class="quotation">                                          *
//*    <blockquote class="smallquotation">                                     *
//*    <blockquote class="largequotation">                                     *
//* If we find an author's name sequence following follows the block, the      *
//* sequence will have the form:                                               *
//*    <div align=\"center\">&mdash; <em>Author Name>/em>                      *
//*    </div>                                                                  *
//*                                                                            *
//* Input  : bType  : indicates block type: member of enum blkType             *
//*          gsbq  : (by reference) contains the first line of the block.      *
//*                                                                            *
//* Returns: OK if successful, ERR processing error                            *
//******************************************************************************
//* Programmer's Note: The author name sequence is moved INSIDE the block,     *
//* where we believe it should have been all along. In addition, we replace    *
//* the centered style with a simple offset style which looks much better.     *
//*                                                                            *
//* Programmer's Note: The procedure for this modification is not particularly *
//* robust, but it handles all _reasonable_ tests well.                        *
//*                                                                            *
//* Embedded blocks: Because both "quotation" and "indentedblock" use the      *
//* <blockquote> tag, if one is embedded inside the other we need to track     *
//* the number of <blockquote>...</blockquote> pairs we see here so we know    *
//* when the main block is complete. Therefore 'bqCount' is used to track      *
//* the embedded "indentedblock" sections.                                     *
//*                                                                            *
//* However: Other blocks embedded within "quotation" blocks are merely        *
//* _identified_, NOT processed.                                               *
//******************************************************************************

short Idpp::ppfProcQUOTE ( blkType bType, const gString& gsbq )
{
   const wchar_t* quotBEGIN = L"<blockquote" ;
   const wchar_t* endDIV = L"</div>" ;
   const short    endDIV_len = 6 ;
   const wchar_t* authorCMD = L"<div align=\"center\">&mdash; <em>" ;
   const short    authorCMD_len = 32 ;
   const wchar_t* authorINDENT = L"<br><span style=\"margin-left:3.2em;\">&mdash; <em>" ;
   const wchar_t* blockName[] = 
   {
      L"Quotation",   L"smallQuotation",   L"largeQuotation",
   } ;

   gString gs = gsbq,         // text formatting
           gsNext ;           // context line
   wchar_t usrResp = L'd' ;   // user response to prompt
   short wi = ZERO,           // string index
         qi,                  // index of <blockquote...  tag
         bnIndex = ZERO,      // index into blockName[] array
         bqCount = 1,         // count <blockquote...> tags within this block
         status = OK ;        // return value
   bool  otherClass = false ; // 'true' if existing class callout in source

   //* Identify the specified block type.               *
   //* If no class specified, erase entire existing tag.*
   //* If class IS specified, erase class, but remember.*
   if ( (qi = gs.find( quotSimple )) >= ZERO )
      gs.erase( quotSimple, qi ) ;
   else if ( (qi = gs.find( quotInherit )) >= ZERO )
   { otherClass = true ; gs.erase( quotInherit, qi ) ; }
   else if ( (qi = gs.find( quotSmall )) >= ZERO )
   { otherClass = true ; gs.erase( quotSmall, qi ) ; }
   else if ( (qi = gs.find( quotLarge )) >= ZERO )
   { otherClass = true ; gs.erase( quotLarge, qi ) ; }

   //* If user interaction is enabled, *
   //* prompt for user's selection.    *
   if ( this->blkFont == cfgSpec )
   {
      //* Index the block-name text for display *
      if ( bType == stdQ )      bnIndex = ZERO ;
      else if ( bType == smaQ ) bnIndex = 1 ;
      else if ( bType == lrgQ ) bnIndex = 2 ;

      //* If there is display text on this line (unlikely), use it *
      if ( gs.gstr()[qi] != NULLCHAR )
         gsNext = &gs.gstr()[qi] ;
      //* Else read the next line to give user some context.       *
      else
      {
         status = this->ppfReadSrcLine ( gsNext, wi ) ;

         //* Push the context line back into the stream      *
         //* format the text portion of the line for display.*
         this->ppfUnReadSrcLine ( gsNext ) ;
         gsNext.erase( L"<p>", ZERO, false ) ;
         gsNext.erase( L"<br>", ZERO, false ) ;
      }

      //* Ask the user to select a font-size option *
      usrResp = this->ppfBlockPrompt ( blockName[bnIndex], gsNext.gstr() ) ;
   }

   //* Update the tag according to user's selection *
   if ( qi >= ZERO )
   {
      switch ( usrResp )
      {
         //** inherited font **
         case L'i':  gs.insert( quotInherit, qi ) ; break ;
         //** smaller font **
         case L's':  gs.insert( quotSmall, qi ) ;   break ;
         //** larger font **
         case L'l':  gs.insert( quotLarge, qi ) ;   break ;
         //** Automatic assignment of class       **
         //** If font size previously specified,  **
         //** use it, else use standard font size.**
         case L'a':
         default:
            if ( bType == smaQ )
               gs.insert( quotSmall, qi ) ;
            else if ( bType == lrgQ )
               gs.insert( quotLarge, qi ) ;
            else
               gs.insert( quotInherit, qi ) ;
            break ;
      } ;

      //* During a second pass, the existing class *
      //* designation overrides user response.     *
      if ( otherClass && (this->respFile.gschars() > 1) )
         gs = gsbq ;    // copy line as-written
   }
   else
   {
      gs = gsbq ;       // copy line as-written
      status = ERR ;
   }

   //* Write the header tag *
   this->ppfWriteLine ( gs ) ;

   while ( status == OK )
   {
      if ( (status = this->ppfReadSrcLine ( gs, wi )) == OK )
      {
         if ( (gs.find( quotBEGIN )) >= ZERO )
         {
            ++bqCount ;    // count embedded blockquote instances
            this->ppfWriteLine ( gs ) ;
         }
         else if ( (gs.find( quotEND, ZERO, false )) >= ZERO )
         {
            --bqCount ;
            gString gsauth, gsnew, gsdiv  ;
            short pi ;

            if ( (status = this->ppfReadSrcLine ( gsauth, pi )) == OK )
            {
               if ( (this->no_auth == false) &&
                    ((gsauth.compare( authorCMD, false, authorCMD_len, pi )) == ZERO) )
               {
                  //* Reformat and write the 'author' line *
                  gsauth.replace( authorCMD, authorINDENT, pi ) ;
                  this->ppfWriteLine ( gsauth ) ;

                  //* Read the end-of-author (</div> tag) line. If ONLY *
                  //* the tag, discard it. Else, retain trailing data.  *
                  status = this->ppfReadSrcLine ( gsdiv, pi ) ;
                  if ( (gsdiv.compare( endDIV, false, endDIV_len, pi )) == ZERO )
                  {
                     gsdiv.shiftChars( ZERO - (pi + endDIV_len) ) ;
                     if ( gsdiv.gschars() > 1 )
                        this->ppfUnReadSrcLine ( gsdiv ) ;
                  }
                  else  // This is unlikely and would be a source error.
                  {
                     this->ppfWriteLine ( gsdiv ) ;
                     status = ERR ;
                  }
                  this->ppfWriteLine ( gs ) ;   // output the end-of-block line

                  if ( this->verbose != false )
                  {
                     uint16_t authLine = this->tlCount - 1 ;
                     gString gsVerb( "(%4hu) Quotation Author repositioned.", &authLine ) ;
                     this->textOut ( gsVerb ) ;
                  }
               }
               else
               {
                  this->ppfWriteLine ( gs ) ;      // output the end-of-block line
                  this->ppfUnReadSrcLine ( gsauth ) ; // push unused data back into the stream
               }
               if ( bqCount == ZERO )
                  break ;     // block processing complete
            }
         }
         else        // this is not the droid you're looking for
            this->ppfWriteLine ( gs ) ;
      }
   }
   return status ;

}  //* End ppfProcQUOTE() *

//*************************
//*      ppfProcTAB       *
//*************************
//******************************************************************************
//* The beginning of a <table...> ... </table> sequence has been found.        *
//* Determine the TYPE of table object, and optionally modify it if it meets   *
//* our criteria.                                                              *
//*                                                                            *
//* Note that we process ONLY the table header here. Caller must copy the      *
//* remainder of the table.                                                    *
//*                                                                            *
//* Input  : gstab : (by reference) contains the first line of the <table>     *
//*                  sequence                                                  *
//*                                                                            *
//* Returns: OK if successful, ERR processing error                            *
//******************************************************************************
//* Notes:                                                                     *
//* 1) Only tables within the main sequence (after Table of Contents)          *
//*    are seen here.                                                          *
//*                                                                            *
//* 2) The tables representing the auto-generated menus are not modified.      *
//*          <table class="menu" border="0" cellspacing="0">                   *
//*    We may offer an option to modify menus in a future release.             *
//*                                                                            *
//* 3) The tables at the top and bottom of the index node ("Jump to" blocks)   *
//*    <table><tr><th valign="top">Jump to: &nbsp; </th> ....                  *
//*    are always processed as borderless tables. If the index node is         *
//*    properly identified, we should never see these tables here, but we      *
//*    test for them just-in-case.                                             *
//*                                                                            *
//* 4) Simple "<table>" tags are generated by the texi @multitable command.    *
//*    The '--table_border' command-line option initializes the 'tabBorder'    *
//*    member:                                                                 *
//*       cfgAuto : Apply <table class="bordered"> border style to all tables. *
//*                 (default)                                                  *
//*       cfgNone : Apply <table class="borderless"> to all tables.            *
//*       cfgSpec : For each table, prompt the user for a decision whether to  *
//*                 include a border.                                          *
//*                                                                            *
//* 5) <table class="cartouche" ...> tags are generated by the texi @cartouche *
//*    command. This command includes a 'border="1"' style element which is    *
//*    indended to create a one-pixel border. However, this looks like crap in *
//*    combination with the class style, so that style element is removed      *
//*    during post-processing so that the cartouche class can fully control    *
//*    style for the block.                                                    *
//*    -- The cartouche class specifies that text with the block is            *
//*       pre-formatted; however, the "--no_cartouche" command-line option     *
//*       specifies that we should allow text within the cartouche block to    *
//*       "flow" (i.e. dynamic line breaks). In this case, we override the     *
//*       "white-space:pre;" style element specified in the class and adjust   *
//*       other style elements as necessary to position the text.              *
//*                                                                            *
//* 6) All other '<table...> sequence headers are written unmodified to the    *
//*    target file.                                                            *
//*                                                                            *
//* 7) The user prompt for table border inclusion is documented as yes/no;     *
//*    however, we have found it convenient to have a third response,          *
//*    'A' (All) which sets this->tabBorder to 'cfgAuto', allowing the         *
//*    remaining tables to receive borders without further prompting.          *
//******************************************************************************

short Idpp::ppfProcTAB ( const gString& gstab )
{
   const wchar_t* tabCartouche = L"<table class=\"cartouche\" border=\"1\">" ;
   const short tabCartouche_len = 36 ;
   const short tabCartMin_len = 24 ;
   const short tabCartTail_len = tabCartMin_len + 11 ;
   const wchar_t* indexTab = L"<table><tr><th valign=\"top\">Jump to: &nbsp; </th>" ;
   const wchar_t* bdrCLASS = L"<table class=\"bordered\">" ;
   const wchar_t* nobdrCLASS = L"<table class=\"borderless\">" ;
   //* Style for cartouche with "flowing" text *
   const wchar_t* flowStyle = L" style=\"white-space:inherit; margin-right:8.0%; padding-top:1.5em;\"" ;
   //* Used to format the user prompt *
   const wchar_t* thStart = L"<th>" ;
   const wchar_t* thEnd = L"</th>" ;
   const wchar_t* elips1 = L"." ;
   const wchar_t* elips2 = L".." ;
   const wchar_t* elips3 = L"..." ;

   gString gst ;              // text formatting
   short wi = gstab.scan(),   // step over leading whitespace
         ti,                  // index of "<table>" tag
         status = OK ;        // return value

   // Programmer's Note: If the index header was found as it should be, then the 
   // index tables will be processed in ppfProcINDEX() and will not seen here.
   if ( (gstab.find( indexTab, wi )) >= wi )
   {
      gst = gstab ;
      if ( (wi = gst.find( tagClose, wi )) > ZERO )
         gst.insert( L" class=\"jumpto\"", wi ) ;
      this->ppfWriteLine ( gst ) ;  // write the modified header line
   }

   //* Else if a simple table header OR a header that explicity declares *
   //* a bordered or borderless table (previously processed).            *
   else if ( ((ti = gstab.find( tabSimple, wi, false )) >= ZERO) || 
             ((ti = gstab.find( bdrCLASS, wi, false )) >= ZERO) || 
             ((ti = gstab.find( nobdrCLASS, wi, false )) >= ZERO) )
   {
      gString gsNext ;
      wchar_t resp = L'y' ;      
      if ( (this->tabBorder == cfgNone) || 
           ((gstab.find( nobdrCLASS, ti, false )) == ZERO) )
         resp = L'n' ;

      //* If we must prompt user for a decision *
      if ( this->tabBorder == cfgSpec )
      {
         //* Read the next line of the table to give user some context *
         short ni ;
         if ( (status = this->ppfReadSrcLine ( gsNext, ni )) == OK )
         {
            //* Push the context line back into the stream      *
            //* format the text portion of the line for display.*
            this->ppfUnReadSrcLine ( gsNext ) ;
            short ni ;
            if ( (ni = gsNext.after( thStart )) > ZERO )
            {
               gsNext.shiftChars( -(ni) ) ;
               if ( (ni = gsNext.findlast( thEnd )) > ZERO )
                  gsNext.limitChars( ni ) ;
               gsNext.replace( thStart, elips2, ZERO, false, true ) ;
               gsNext.replace( thEnd, elips1, ZERO, false, true ) ;
               gsNext.insert( elips3 ) ;
               gsNext.append( elips3 ) ;
            }
            gString gsIn,
                    gsOut( "____________________________\n"
                            "Table found on Line:%4hu\n"
                            "First Row: %S\n"
                            "Add border to this table?",
                            &this->slCount, gsNext.gstr()
                         ) ;
            this->textOut ( gsOut ) ;
            this->textOut ( L"your choice (y/n/A): ", false ) ;
            while ( true )
            {
               if ( (this->userResponse ( gsIn )) == OK )
               {
                  //* If "default_token" received *
                  if ( (gsIn.compare( dToken, true, dToken_len )) == ZERO )
                     resp = L'y' ;     // default is 'Yes'
                  else
                     resp = *gsIn.gstr() ;

                  if ( resp == L'y' || resp == L'n' || resp == L'A' )
                  { break ; }
                  else  // call user a dumbguy and ask for correct response
                  { this->invalidResponse () ; }
               }
            }
         }
      }

      if ( status == OK )
      {
         gst = gstab ;
         if ( (resp == L'y') || (resp == L'A') )
            gst.replace( tabSimple, bdrCLASS, ti ) ;
         else
            gst.replace( tabSimple, nobdrCLASS, ti ) ;
         this->ppfWriteLine ( gst ) ;        // write the modified header line

         //* If user specified 'All', then set the global indicator *
         //* to process all remaining tables automagically.         *
         if ( this->tabBorder == cfgSpec )
         {
            if ( resp == L'A' )     // switch to auto mode (see note above)
               this->tabBorder = cfgAuto ;
         }
      }
   }

   //* Else if cartouche table *
   else if ( (gstab.compare( tabCartouche, false, tabCartouche_len, wi )) == ZERO )
   {
      gst = gstab ;
      gst.limitChars( wi + tabCartMin_len ) ;
      gst.append( &gstab.gstr()[tabCartTail_len] ) ;
      //* If flowing text specified, override white-space style element *
      if ( this->no_cart )
      {
         wi = gst.find( tagClose ) ;
         gst.insert( flowStyle, wi ) ;
      }
      this->ppfWriteLine ( gst ) ;
   }

   else     // write the unmodified line
   {
      this->ppfWriteLine ( gstab ) ;
   }
   return status ;

}  //* End ppfProcTAB() *

//*************************
//*      ppfProcHEAD      *
//*************************
//******************************************************************************
//* If the '--no_meta' option OR the '--no_links' option OR the '--no_head'    *
//* option has been invoked, then this method is called to retain the entry.   *
//* See ppfProcessSrcHTML method for element selection.                        *
//*                                                                            *
//* Although the default entries are single-line entries, IF the source        *
//* document was generated with custom data, then the entry can span multiple  *
//* lines.                                                                     *
//*                                                                            *
//* Input  : (by reference) contains the first line of the entry to be retained*
//*                                                                            *
//* Returns: OK if successful, ERR processing error                            *
//******************************************************************************

short Idpp::ppfProcHEAD ( const gString& gsmeta )
{
   gString gs ;            // additional source data
   short wi ;              // index into source line
   bool done = false ;     // loop control
   short status = OK ;     // return value

   //* Write the first (and possibly only) line of the entry.*
   this->ppfWriteLine ( gsmeta ) ;

   //* Index the end-of-tag character *
   if ( (wi = gsmeta.find( tagClose )) >= ZERO )
      done = true ;

   //* If metadata entry not yet terminated, scan and copy additional lines.*
   while ( ! done )
   {
      status = this->ppfReadSrcLine ( gs, wi ) ;   // read from source
      this->ppfWriteLine ( gs ) ;                  // write to target
      if ( status == OK )
      {  //* Scan the line for end-of-entry character.*
         if ( (wi = gs.find( tagClose )) >= ZERO )
            done = true ;
      }
   }     // while()
   return status ;

}  //* End ppfProcHEAD() *

//*************************
//*       ppfProcUL       *
//*************************
//******************************************************************************
//* Called by ppfProcessSrcHTML() and by ppfProcOL() to process <ul> lists.    *
//* Also called recursively.                                                   *
//*                                                                            *
//* Note that for automatic processing, only <ul> blocks declared with the     *
//* "no-bullet" class are actively processed. (<ul class="no-bullet">)         *
//* For interactive processing, any list may optionally be assigned one of the *
//* class descriptions recognized by the browser's rendering engine.           *
//*                                                                            *
//*                                                                            *
//* Input  : gsul   : (by reference) contains the first line of the list:      *
//*                   Examples:   '<ul>'   '<ul class="no-bullet">'            *
//*                                                                            *
//* Returns: OK if successful, ERR processing error                            *
//******************************************************************************
//* Programmer's Notes:                                                        *
//* 1) Because lists may be embedded within other lists, this method may be    *
//*    called recursively, so be sure all data are safely on the stack.        *
//*                                                                            *
//* 2) The texi source fully supports only one bullet character, "@bullet",    *
//*    whether explicity called out or implicitly specified. From a practical  *
//*    point-of-view, this is reasonable because the HTML rendering in a       *
//*    browser will automatically convert the standard disc bullet to one of   *
//*    the alternate characters ( '⏺' becomes '⚬' or '▪') when lists are       *
//*    nested. However, this severely limits the range of list types.          *
//*                                                                            *
//* 3) If the texi source specifies a bullet character other than the standard *
//*    default "@bullet" character, the specified character is embedded into   *
//*    each line item. This looks fine in 'info' documents, but appears        *
//*    amateurish when converted to HTML. Our solution is to adjust the        *
//*    indentation for the first line of each line item to shift the embedded  *
//*    character into the position it would have if it were a true bullet      *
//*    character. This requires the writer of the texi source to be aware of   *
//*    the length of the first line and to place a hard return at the          *
//*    appropriate place if the line item extends to second and subsequent     *
//*    lines.                                                                  *
//*                                                                            *
//* 4) The HTML markup may contain the actual character specified, or may      *
//*    substitute a "defined" character for the actual character specified in  *
//*    the texi source. For instance, the HTML converter substitutes '&gt;'    *
//*    for the '>' character and '&deg;' for the '◦' (degree symbol) when they *
//*    are used as bullets in the texi source. There are a limited number of   *
//*    these defined characters in HTML, so we can test for them here. We have *
//*    identified a fairly generous list of characters we support as bullet    *
//*    characters. See ppfFormatLineItem() for the complete list.              *
//*                                                                            *
//* 5) Also, the HTML converter (stupidly) inserts an HTML comment if:         *
//*    texi source specifies a non-standard bullet character or if the         *
//*    no-bullet sequence (@itemize @w{}) is specified:                        *
//*    HTML output:  <li><!-- /@w --> ...                                      *
//*    These useless comment sequences are stripped globally under a           *
//*    conditional-compile flag, but if that flag is not active, we strip      *
//*    out the unnecessary comment HERE to bring the embedded character        *
//*    adjacent to the <li> declaration or to create a true 'no-bullet' list.  *
//*    While it is not technically necessary to do this, this kind of garbage  *
//*    in our output offends us, and the resulting markup is much easier       *
//*    to read.                                                                *
//*    -- We also define the bullet character in monospace font to fill the    *
//*       character cell. This provides better alignment.                      *
//*                                                                            *
//* 6) Previously-processed "no-bullet" lists are passed through unmodified;   *
//*    however, if the user is running a second pass on the document, AND if   *
//*    a automatic response file is used, we must prompt for a response on     *
//*    this previously-processed list to avoid getting the automatic responses *
//*    out-of-synch. In this case, the response is ignored and the list is     *
//*    is passed through as written.                                           *
//*                                                                            *
//******************************************************************************

short Idpp::ppfProcUL ( gString& gsul )
{
   gString gs, gst,           // text analysis and formatting
           origUL,            // copy of unmodified UL tag
           fmtUL,             // formatted UL tag
           origLI,            // copy of unmodified line item
           fmtLI,             // formatted line item
           dispLI,            // stripped line item for display to user
           gsIn,              // user response to prompt
           gsVerb ;           // for 'verbose' output
   wchar_t bullChar,          // embedded bullet character (no-bullet class lists)
           usrResp ;          // user response to prompt
   blkType bType = btNone ;   // block-type
   short   wi = gsul.scan(),  // index the first non-whitespace character
           ul,                // index of UL tag
           li,
           lineItems = ZERO,  // count the <li> tags
           status = OK ;      // return value
   bool  nobullClass = false, // 'true' if "no-bullet" class specified
         otherClass = false,  // 'true' if a class other than "no-bullet" specified
         simpleLi = false,    // 'true' if _unstyled_ line item identified
         eraseBull = false,   // 'true' if user specifies override of line-item formatting
         nomodBull = false,   // 'true' if user specifies no modification to line-items
         apOverride = false ; // 'true' if force user response on Already Processed list

   //* Index the beginning of UL tag, and test for      *
   //* specification of class name or style information.*
   if ( (ul = gsul.find( ulBegin, false, wi )) >= ZERO )
   {
      //* Was this list processed as a "no-bullet" class? *
      if ( (gsul.find( ulNobull, ul )) >= ul )
         nobullClass = true ;

      //* If any other class specified, or if style info specified, *
      //* do not modify the tag.                                    *
      //* (assumes that document author knows what he/she is doing) *
      else if ( ((gsul.find( L"class=\"", ul )) >= ul) ||
                ((gsul.find( L"style=\"", ul )) >= ul) )
         otherClass = true ;

      //* Save a copy of the original UL tag *
      // Programmer's Note: We assume two things here:
      //  1. The complete tag is present in the source line.
      //  2. The tag is the last thing in the source line.
      // This is the case in all our test data; however, if either 
      // of these things becomes false, some data may be lost.
      short endtag, taglen ;
      if ( (endtag = gsul.find( tagClose, ul )) > ul )
         taglen = endtag - ul + 1 ;
      else
         taglen = gsul.gschars() - ul - 1 ;
      origUL = &gsul.gstr()[ul] ;    // make a copy of the original UL tag
      fmtUL = origUL ;
      fmtUL.limitChars( taglen ) ;
      //* Delete the original tag from output data *
      if ( (this->ulLists == cfgAuto) || (this->ulLists == cfgSpec) )
         gsul.limitChars( ul ) ;
   }

   if ( this->verbose )       // verbose diagnostics
   {
      short bLine = this->tlCount + 1 ;   // output line number
      gsVerb.compose( L"(%4hu) '%S' ==>> ", &bLine, fmtUL.gstr() ) ;
   }


   bool done = false ;
   do
   {
      //*************************************************
      //*     Read lines from the source document.      *
      //*************************************************
      if ( (status = this->ppfReadSrcLine ( gs, wi )) == OK )
      {
         //* Test for line item *
         simpleLi = false ;
         if ( (li = gs.find( liOpen, wi )) >= wi )
         {
            ++lineItems ;              // count the line items
            //* Test for _unstyled_ line item *
            if ( (gs.compare( liBegin, false, liBegin_len, li )) == ZERO )
               simpleLi = true ;       // unstyled line item
         }

         //* If at the end of the list, return to caller *
         if ( (gs.find( ulEnd, wi, false )) >= wi )
         {
            this->ppfWriteLine ( gs ) ;   // pass line through unmodified
            if ( this->verbose )          // verbose diagnostics
            {
               gsVerb.append( " (%hd items)", &lineItems ) ;
               this->textOut ( gsVerb ) ;
            }
            done = true ;     // return to caller
            break ;
         }     // (end-of-list)

         //* If source line contains a line item
         else if ( li >= wi )
         {
            //* If the list is formatted as a no-bullet class,   *
            //* AND if the first character of the line item is a *
            //* a recognized bullet character, adjust indentation*
            //* of first line of text within the line item.      *
            //* Optionally remove any embedded bullet character. *
            origLI = gs ;
            fmtLI = origLI ;
            if ( nobullClass && simpleLi &&
                 ((this->ulLists == cfgAuto) || (this->ulLists == cfgSpec)) )
               bullChar = this->ppfFormatLineItem ( fmtLI, li, eraseBull ) ;
            else     // bullet character not identified
               bullChar = L' ' ;

            //**********************************************
            //* If first line item, potentially modify the *
            //* class called out by the <ul> tag.          *
            //**********************************************
            if ( lineItems == 1 )
            {
               //* If automatic processing, skip user prompt.      *
               //* If interactive processing, ask user what to do. *
               if ( (this->ulLists == cfgAuto) || (this->ulLists == cfgSpec) )
               {
                  usrResp = L'a' ;     // initialize user response to "automatic"

                  //* Force a user response for previously-processed *
                  //* "no-bullet" list. Response will be ignored.    *
                  //* See note in method header.                     *
                  if ( nobullClass && !simpleLi && (this->respFile.gschars() > 1) )
                     apOverride = true ;

                  //* If list has not already been formatted, OR if   *
                  //* forcing a user response (which will be ignored) *
                  //* on an already-processed list.                   *
                  if ( (this->ulLists == cfgSpec) && !otherClass &&
                       (simpleLi || apOverride) )
                  {
                     dispLI = fmtLI ;
                     StripTags ( dispLI ) ;
                     gst.compose( L"___________________________________\n"
                                   "Bullet List: (source line:%hu)\n"
                                   "First Line Item: %S\n"
                                   "Auto-definition: %S",
                                  &this->slCount, dispLI.gstr(), fmtUL.gstr() ) ;
                     this->textOut ( gst ) ;
                     this->textOut ( 
                        L"Choose Bullet-list type:\n"
                         "d:disc   (\x23FA)     a:automatic (default)\n"
                         "c:circle (\x26AC)     A:All automatic\n"
                         "s:square (\x25AA)     x:no modification\n"
                         "n:no bullet" ) ;
                     this->textOut ( L"your choice: ", false ) ;

                     while ( true )    // input loop
                     {
                        if ( (this->userResponse ( gsIn )) == OK )
                        {
                           //* If user specified a default response *
                           if ( (gsIn.compare( dToken, true, dToken_len )) == ZERO )
                              usrResp = L'a' ;
                           else        // single-character response expected
                              usrResp = *gsIn.gstr() ;
                           //* If this list, and all remaining UL lists *
                           //* are to be processed automagically        *
                           if ( usrResp == L'A' )
                           { usrResp = L'a' ; this->ulLists = cfgAuto ; }

                           if ( (usrResp == L'a') || (usrResp == L'd') ||
                                (usrResp == L'c') || (usrResp == L's') ||
                                (usrResp == L'n') || (usrResp == L'x') )
                              break ;  // good input
                           //* Call user a dumbguy and ask for correct response.*
                           this->invalidResponse () ;
                        }
                     }  // while()
                  }     // (cfgSpec && !otherClass && simpleLi, etc.)

                  //* Existing class designation overrides *
                  //* user's second-pass response.         *
                  if ( apOverride )
                     usrResp = L'x' ;

                  switch ( usrResp )
                  {
                     case L'a':     //** automatic processing **
                        //* If specified, perform automatic class assignment.*
                        //* This will assign the default class to _unstyled_ *
                        //* <ul> tags.                                       *
                        if ( (this->ulAssign && ! nobullClass && ! otherClass) ||
                             (this->liSpec && (bullChar == discBullet)) )
                           fmtUL = ulDISC ;
                        //* For a list with embedded text-degree symbol, OR     *
                        //* embedded circle bullet, the symbol has been deleted.*
                        //* Declare the "circle-bullet" class.                  *
                        else if ( this->liSpec && (bullChar == circleBullet) )
                           fmtUL = ulCIRCLE ;
                        //* For a list with embedded square-bullet symbol, *
                        //* the symbol has been deleted.                   *
                        //* Declare the "square-bullet" class.             *
                        else if ( this->liSpec && (bullChar == squareBullet) )
                           fmtUL = ulSQUARE ;
                        //* If a true "no-bullet" class, then     *
                        //* there is no embedded bullet character.*
                        //* Declare the "no-bullet" class.        *
                        else if ( nobullClass )
                           fmtUL = ulNobull ;

                        gs = fmtLI ; // copy modified line item to output buffer
                        break ;
                     case L'd':
                        //* Discard any embedded bullet character *
                        fmtLI = gs ;
                        eraseBull = true ;   // signal for all line items
                        if ( nobullClass && simpleLi )
                           this->ppfFormatLineItem ( fmtLI, li, eraseBull ) ;
                        fmtUL = ulDISC ;
                        gs = fmtLI ;
                        break ;
                     case L'c':
                        fmtLI = gs ;
                        eraseBull = true ;   // signal for all line items
                        if ( nobullClass && simpleLi )
                           this->ppfFormatLineItem ( fmtLI, li, eraseBull ) ;
                        fmtUL = ulCIRCLE ;
                        gs = fmtLI ;
                        break ;
                     case L's':
                        fmtLI = gs ;
                        eraseBull = true ;   // signal for all line items
                        if ( nobullClass && simpleLi )
                           this->ppfFormatLineItem ( fmtLI, li, eraseBull ) ;
                        fmtUL = ulSQUARE ;
                        gs = fmtLI ;
                        break ;
                     case L'n':
                        fmtLI = gs ;
                        eraseBull = true ;   // signal for all line items
                        if ( nobullClass && simpleLi )
                           this->ppfFormatLineItem ( fmtLI, li, eraseBull ) ;
                        fmtUL = ulNobull ;
                        gs = fmtLI ;
                        break ;
                     case L'x':
                        nomodBull = true ;   // signal for all line items
                        fmtUL = origUL ;  // use original UL tag, no changes
                        fmtLI = origLI ;  // use original line item, no changes
                        break ;
                  } ;
               }     // (cfgAuto || cfgSpec)


               //* Append the formatted tag to the *
               //* source line and write to target *
               gsul.append( fmtUL.gstr() ) ;
               this->ppfWriteLine ( gsul ) ;

               //* Append the (potentially) modified UL tag to verbose output *
               if ( this->verbose )
                  gsVerb.append( fmtUL.gstr() ) ;
            }
            //************************************
            //* Second and subsequent line items *
            //************************************
            else
            {
               //* Mirror the formatting of the first line item *
               gs = nomodBull ? origLI : fmtLI ;
            }

            //* Write the line item *
            this->ppfWriteLine ( gs ) ;

         }     // (line item)

         //* Test for a <ul> list nested within current list.*
         else if ( (this->ppfTestUlistBegin ( gs, wi )) != false )
         {
            status = this->ppfProcUL ( gs ) ;
         }

         //* Test for an <ol> list nested within the current list *
         else if ( (this->ppfTestOlistBegin ( gs, wi )) != false )
         {
            status = this->ppfProcOL ( gs ) ;
         }

         //* If there is a preformatted block inside the list *
         else if ( (this->ppfTestFormattedBlock ( bType, gs )) != false )
         {
            //* Copy the block, optionally eliminating the unnecessary *
            //* leading blank line. NOTE: Data inside the block        *
            //* IS NOT modified.                                       *
            status = this->ppfProcFormattedBlock ( bType, gs ) ;
         }
         
         //* If beginning a quotation, smallquotation   *
         //* or largequotation block.                   *
         else if ( (this->ppfTestQuotationBlock ( bType, gs )) != false )
         {
            status = this->ppfProcQUOTE ( bType, gs ) ;
         }

         //* If beginning a verbatim, smallverbatim or largeverbatim block *
         else if ( (this->ppfTestVerbatimBlock ( bType, gs )) != false )
         {
            status = this->ppfProcVerbatim ( bType, gs ) ;
         }

         //* If beginning an indentedblock, smallindentedblock *
         //* or largeindentedblock.                            *
         else if ( (this->ppfTestIndentedBlock ( bType, gs )) != false )
         {
            status = this->ppfProcIDB ( bType, gs ) ;
         }

         //* If a <table> object found *
         else if ( (gs.find( tabBegin, wi, false )) >= ZERO )
         {
            status = this->ppfProcTAB ( gs ) ;
         }

         //* Else the line does not contain a control statement *
         else
         {
            this->ppfWriteLine ( gs ) ;   // pass line through unmodified
         }
      }        // (ReadLine())
   }
   while ( ! done && status == OK ) ;

   return status ;

}  //* End ppfProcUL() *

//*************************
//*       ppfProcOL       *
//*************************
//******************************************************************************
//* Called by ppfProcessSrcHTML() to process <ol> lists.                       *
//* Also called recursively, and by ppfProcUL().                               *
//*                                                                            *
//* Caller has determined that the OL tag is the last item on the source line. *
//*                                                                            *
//*                                                                            *
//* Input  : gsol  : (by reference) contains the first line of the list:       *
//*                  '<ol>' || '</pre><ol>' || '</p><ol>' ||                   *
//*                  '<ol type="a" start="1">' or similar                      *
//*          type  : enumeration type ('1' 'a' 'A' etc)                        *
//*          start : value for first item of list ('1-999' 'a-z' 'A-Z' etc)    *
//*                                                                            *
//* Returns: OK if successful, ERR processing error                            *
//******************************************************************************
//* Programmer's Note:                                                         *
//* 1) Because lists may be embedded within other lists, this method may be    *
//*    called recursively, so be sure all data are safely on the stack.        *
//*                                                                            *
//* 2) Ordered list elements, in addition to the global attributes, may        *
//*    include three(3) style attributes:                                      *
//*     a) "reversed"  boolean to indicate descending sequence                 *
//*     b) "type"      enumeration type: '1', 'a', 'A', 'i', 'I'.              *
//*                    ideally, the type should be handled by the CSS          *
//*                    "list-style-type" which has many more options.          *
//*                    OL sub-classes are defined for the most common CSS      *
//*                    enumeration types.                                      *
//*     c) "start"     ordinal offset from beginning of list (one-based except *
//*                    for decimal "type" which also accepts '0')              *
//*                                                                            *
//* 3) texi2any takes an entirely-reasonable shortcut in determining the       *
//*    "type" and "start" parameters based on the .texi source:                *
//*    => Any single-character lower-case alpha will yield the same type:      *
//*       "@enumerate a", "@enumerate c", "@enumerate e" all yield: type="a"   *
//*    => However, these will yield: start="1", start="3" and start="5"        *
//*       respectively.                                                        *
//*    Similarly for upper-case alpha and for unsigned integer values.         *
//*       (Note that multi-character alpha e.g. "@enumerate cc" )              *
//*       (will be handled as an error and defaulted to "<ol>". )              *
//*       (Multi-digit ordinals will be encoded as expected:    )              *
//*       ( "@enumerate 25" yields <ol start="25">              )              *
//*                                                                            *
//*    While this is reasonable, it means that Roman-numeral lists cannot be   *
//*    specified directly because "@enumerate i" will yield type="a" start="9".*
//*    In our opinion, the need for non-alpha enumerated list is significantly *
//*    greater than the need to start lists at offsets > 1.                    *
//*    Therefore, we make a special case of certain type/start combinations.   *
//*    See below for details.                                                  *
//*    This automatic formatting may be avoided by not processing OL lists,    *
//*    or by interactively specifying the enumeration type and start value.    *
//*                                                                            *
//* 4) Two switches control processing of <ol> lists:                          *
//*    - this->olLists  : 'cfgAuto' process of OL lists automatically          *
//*                       'cfgSpec' ask user to specify list type              *
//*                       'cfgNone' do not process OL lists                    *
//*    - this->liSpec   : 'true'  perform special-case interpretation for      *
//*                               certain combinations of 'type and 'start'    *
//*                               parameters (see notes above)                 *
//*                       'false' interpret list style information literally   *
//*                                                                            *
//* 5) Note that HTML will accept alphabetical offsets of any reasonable size  *
//*    and display the sequence as: ... y, z, aa, ab, ac, ...                  *
//*                                                                            *
//* 6) GPL and FDL license text:                                               *
//*    Please see ppfProgGNU() method for a discussion of processing these     *
//*    special enumerated lists.                                               *
//*                                                                            *
//*                                                                            *
//*                                                                            *
//*                                                                            *
//*                                                                            *
//*                                                                            *
//*                                                                            *
//******************************************************************************

short Idpp::ppfProcOL ( gString& gsol )
{
   //* List of supported OL type sub-classes *
   const short OLCL_COUNT = 12 ;
   const wchar_t* OL_ClassList[OLCL_COUNT] = 
   {
      L"enum-decimal",           //  0 decimal ordinals
      L"enum-decimal-zero",      //  1 decimal with leading zero
      L"enum-lower-alpha",       //  2 lower-case English alpha
      L"enum-upper-alpha",       //  3 upper-case English alpha
      L"enum-lower-roman",       //  4 lower case Roman numerals
      L"enum-upper-roman",       //  5 upper-case Roman numerals
      L"enum-lower-greek",       //  6 lower-case classical Greek alpha
      L"enum-cjk-decimal",       //  7 CJK (Han) decimal numeric (informal)
      L"enum-katakana",          //  8 Katakana alpha (Gojūon)
      L"enum-hebrew",            //  9 Hebrew alpha
      L"enum-arabic-indic",      // 10 Arabic-Indic numerals
      L"enum-custom",            // 11 Custom support for sequence
                                 // (CSS class must be modified to specify the sequence)
   } ;
   const wchar_t OL_ClassID[OLCL_COUNT] = 
   {
      L'1',    // decimal ordinals
      L'0',    // decimal with leading zero
      L'a',    // lower-case English alpha
      L'A',    // upper-case English alpha
      L'i',    // lower-case Roman numerals
      L'I',    // upper-case Roman numerals
      L'α',    // lower-case classical Greek alpha
      L'一',   // CJK (Han) decimal numeric (informal)
      L'ア',   // Katakana alpha (Gojūon)
      L'א',    // Hebrew alpha
      L'١',    // Arabic-Indic numerals 
      L'?',    // unsupported sequence ("enum-custom")
   } ;
   const wchar_t* olTemplate   = L"<ol class=\"%S\" start=\"%hd\">" ;
   const wchar_t* olstyleType  = L"type=\"" ;
   const wchar_t* olstyleStart = L"start=\"" ;
   const wchar_t* classSpec    = L"class=\"" ;
   const wchar_t* olstypeLST   = L"list-style-type:" ;
   const wchar_t* revDIR       = L" reversed" ;

   gString gs, gst,           // text formatting
           gsOrig,            // caller's original OL tag
           fmtOL,             // formatted OL tag
           dispLI,            // stripped line item for display to user
           gsVerb,            // for 'verbose' output
           gsIn ;             // receives user input
   wchar_t eType = L'1',      // enumeration type (default == decimal)
           etemp ;            // temporary enum type
   wchar_t eDir = 'a' ;       // enumeration list direction (default == ascending)
   blkType bType = btNone ;   // block-type
   short   eStart = 1,        // enumeration start value (default == 1)
           ol,                // index of <ol> tag
           wi,                // search index
           clIndex,           // index into OL_ClassList[] array
           lineItems = ZERO,  // count the <li> tags
           status = OK ;      // return value
   bool otherClass = false ;  // 'true' if previously-defined class detected


   if ( this->verbose )       // verbose diagnostics
   {
      //* Index the beginning of the <ol ...> tag *
      if ( (ol = gsol.find( olBegin )) < ZERO )
         ol = ZERO ;
      short bLine = this->tlCount + 1 ;
      gsVerb.compose( L"(%4hu) '%S' ==>> ", &bLine, &gsol.gstr()[ol] ) ;
   }

   //* If list is not to be processed, output the original <ol...> tag.*
   if ( ! ((this->olLists == cfgAuto) || (this->olLists == cfgSpec)) )
   {
      this->ppfWriteLine ( gsol ) ;

      if ( this->verbose )
         gsVerb.append( &gsol.gstr()[ol] ) ;
   }

   //* Else, index the beginning of the OL tag,     *
   //* and test for class name or style information.*
   else if ( (ol = gsol.find( olBegin )) >= ZERO )
   {
      //* If tag includes class or style info *
      //* i.e. skip plain <ol>  tag.          *
      if ( gsol.gstr()[ol + olBegin_len] != L'>' )
      {
         if ( (wi = gsol.after( olstyleType )) >= ZERO )    // "type" attribute
         {
            eType = gsol.gstr()[wi] ;
         }

         if ( (wi = gsol.after( olstyleStart )) >= ZERO )   // "start" attribute
         {
            if ( (gsol.gscanf( wi, L"%hd", &eStart )) == 1 )
            {
               if ( (eStart > ZERO) || ((eStart == ZERO) && (eType == 1)) )
                  ;
               else  // out-of range, use default 
                  eStart = 1 ;
            }
         }

         //* If "reversed" (decrement) specified, remember it. *
         if ( (gsol.find( revDIR )) > ZERO )
         {
            eDir = L'd' ;
         }

         if ( (wi = gsol.after( olstypeLST )) >= ZERO )     // "list-style-type" attribute
         {  //* Isolate the "list-style-type" argument *
            wchar_t wbuff[32] ;
            short   w = ZERO ;
            while ( (isalnum(gsol.gstr()[wi]) || (gsol.gstr()[wi] == L'-')) && (w < 31) )
               wbuff[w++] = gsol.gstr()[wi++] ;
            wbuff[w] = NULLCHAR ;   // terminate the string
            //* If isolated type name is a substring *
            //* in class name, then we have a match. *
            // Programmer's Note: If the text-to-HTML was fully implemented, 
            // this would be enough; however, see special cases below.
            for ( short i = ZERO ; i < OLCL_COUNT ; ++i )
            {
               gst = OL_ClassList[i] ;
               if ( (gst.find( wbuff )) >= ZERO )
               {
                  eType = OL_ClassID[i] ;
                  break ;
               }
            }
         }

         //* If source already specifies a defined class, *
         //* assume that user knows what he/she is doing  *
         //* and set 'eType' to match the specified class.*
         //* (An undefined class name will be discarded.) *
         if ( (wi = gsol.after( classSpec )) >= ZERO )
         {
            wchar_t wbuff[32] ;
            short   w = ZERO ;
            while ( (isalnum(gsol.gstr()[wi]) || (gsol.gstr()[wi] == L'-')) && (w < 31) )
               wbuff[w++] = gsol.gstr()[wi++] ;
            wbuff[w] = NULLCHAR ;   // terminate the string
            for ( short i = ZERO ; i < OLCL_COUNT ; ++i )
            {
               gst = OL_ClassList[i] ;
               if ( (gst.compare( wbuff )) == ZERO )
               {
                  eType = OL_ClassID[i] ;
                  otherClass = true ;
                  break ;
               }
            }
         }

         //* Special Case Processing: Override certain texi2any assignments *
         if ( this->liSpec )
         {
            if ( (eType == L'A') && (eStart == 4) )       // 'D' decimal with leading '0's
            { eType = OL_ClassID[1] ; eStart = 1 ; otherClass = false ; }
            else if ( (eType == L'a') && (eStart == 9) )  // 'i' lower-case Roman numerals
            { eType = OL_ClassID[4] ; eStart = 1 ; otherClass = false ; }
            else if ( (eType == L'A') && (eStart == 9) )  // 'I' upper-case Roman numerals
            { eType = OL_ClassID[5] ; eStart = 1 ; otherClass = false ; }
            else if ( (eType == L'a') && (eStart == 7) )  // 'g' lower-case Greek alpha
            { eType = OL_ClassID[6] ; eStart = 1 ; otherClass = false ; }
            else if ( (eType == L'a') && (eStart == 10) ) // 'j' CJK numeric (informal)
            { eType = OL_ClassID[7] ; eStart = 1 ; otherClass = false ; }
            else if ( (eType == L'a') && (eStart == 11) ) // 'k' Katakana alpha
            { eType = OL_ClassID[8] ; eStart = 1 ; otherClass = false ; }
            else if ( (eType == L'a') && (eStart == 8) )  // 'h' Hebrew alpha
            { eType = OL_ClassID[9] ; eStart = 1 ; otherClass = false ; }
            else if ( (eType == L'a') && (eStart == 5) )  // 'e' Eastern (Arabic-Indic)
            { eType = OL_ClassID[10] ; eStart = 1 ; otherClass = false ; }
         }
      }

      //* Index the class name *
      for ( clIndex = ZERO ; clIndex < (OLCL_COUNT - 1) ; ++clIndex )
      {
         if ( eType == OL_ClassID[clIndex] )
            break ;
      }
      gsOrig = &gsol.gstr()[ol] ;   // save a copy of the original tag
      gsol.limitChars( ol ) ;       // truncate existing OL tag

      //* Format the auto-generated OL tag.*
      fmtOL.compose( olTemplate, OL_ClassList[clIndex], &eStart ) ;
   }

   bool done = false ;
   do
   {  //* Read lines from source file.*
      if ( (status = this->ppfReadSrcLine ( gs, wi )) == OK )
      {
         //* If at the end of the list, return to caller *
         if ( (gs.find( olEnd, ZERO, false )) >= ZERO )
         {
            this->ppfWriteLine ( gs ) ;   // pass line through unmodified
            if ( this->verbose )          // verbose diagnostics
            {
               gsVerb.append( " (%hd items)", &lineItems ) ;
               this->textOut ( gsVerb ) ;
            }
            done = true ;
            break ;
         }

         //* Scan for line items *
         if ( (gs.find( liBegin, wi )) >= wi )
         {
            ++lineItems ;              // count the line items

            //* If first line-item in the list *
            if ( (lineItems == 1) &&
                 ((this->olLists == cfgAuto) || (this->olLists == cfgSpec))  )
            {
               //* If interactive, ask user what to do. *
               if ( this->olLists == cfgSpec )
               {
                  dispLI = gs ;
                  StripTags ( dispLI ) ;
                  gst.compose( L"___________________________________\n"
                                "Enumerated List: (source line:%hu)\n"
                                "First Line Item: %S\n"
                                "Auto-definition: %S",
                               &this->slCount, dispLI.gstr(), fmtOL.gstr() ) ;
                  this->textOut ( gst ) ;
                  this->textOut ( 
                     L"Choose Enumeration type:\n"
                      "d:decimal            D:leading-zero decimal  j:CJK(informal)\n"
                      "l:lower case alpha   u:upper case alpha      k:Katakana\n"
                      "i:lower case Roman   I:upper case Roman      h:Hebrew\n"
                      "g:lower case Greek   c:custom enumeration    e:Arabic-Indic\n"
                      "a:automatic(default) A:All automatic         x:no modification\n"
                      "response format: TYPE[,START_VAL[,DIR]]") ;
                  this->textOut ( L"your choice: ", false ) ;
                  while ( true )
                  {
                     if ( (this->userResponse ( gsIn )) == OK )
                     {  //* "default-token" ?
                        if ( (gsIn.compare( dToken, true, dToken_len )) == ZERO )
                           etemp = L'd' ;     // default is 'd'ecimal
                        else     // single-character response
                        {
                           etemp = *gsIn.gstr() ;
                           //* If start value specified *
                           if ( (gsIn.gschars() > 2) && (gsIn.gstr()[1] == L',') )
                           {
                              if ( (swscanf ( &gsIn.gstr()[2], L"%hd,%C", 
                                   &eStart, &eDir )) >= 1 )
                                 ;
                              else     // assume secondary parameters not specified
                              { eStart = 1 ; eDir = L'a' ; }
                           }
                        }

                        if ( etemp == L'd' || etemp == L'D' || 
                             etemp == L'l' || etemp == L'u' || 
                             etemp == L'i' || etemp == L'I' || 
                             etemp == L'g' || etemp == L'j' || 
                             etemp == L'k' || etemp == L'h' || 
                             etemp == L'e' || etemp == L'c' || 
                             etemp == L'a' || etemp == L'A' ||
                             etemp == L'x' )
                        {
                           //* Convert response to class ID *
                           switch ( etemp )
                           {
                              case L'd':  eType = OL_ClassID[clIndex = 0] ;  break ;
                              case L'D':  eType = OL_ClassID[clIndex = 1] ;  break ;
                              case L'l':  eType = OL_ClassID[clIndex = 2] ;  break ;
                              case L'u':  eType = OL_ClassID[clIndex = 3] ;  break ;
                              case L'i':  eType = OL_ClassID[clIndex = 4] ;  break ;
                              case L'I':  eType = OL_ClassID[clIndex = 5] ;  break ;
                              case L'g':  eType = OL_ClassID[clIndex = 6] ;  break ;
                              case L'j':  eType = OL_ClassID[clIndex = 7] ;  break ;
                              case L'k':  eType = OL_ClassID[clIndex = 8] ;  break ;
                              case L'h':  eType = OL_ClassID[clIndex = 9] ;  break ;
                              case L'e':  eType = OL_ClassID[clIndex = 10] ; break ;
                              case L'c':  eType = OL_ClassID[clIndex = 11] ; break ;
                              case L'A':  // 'A' - retain auto-formatting for all
                                 this->olLists = cfgAuto ;
                              case L'x':  // 'x' - do not modify the tag
                              default:    break ;  // 'a' - retain auto-formatting
                           } ;

                           //* Validate the start value and sequence direction *
                           if ( (eStart < ZERO) || 
                                ((eStart == ZERO) && 
                                   !((eType == L'1') || (eType == L'0'))) ||
                                !((eDir == L'a' || eDir == L'd')) )
                           {
                              this->invalidResponse () ;
                              continue ;
                           }

                           //* Format the specified OL tag *
                           fmtOL.compose( olTemplate, OL_ClassList[clIndex], &eStart ) ;

                           if ( eDir == L'd' )
                           {
                              wi = fmtOL.findlast( tagClose ) ;
                              fmtOL.insert( revDIR, wi ) ;
                           }
                           break ;
                        }
                        else
                           this->invalidResponse () ;
                     }
                  }

                  //* During a second pass, the existing class *
                  //* designation overrides user response.     *
                  if ( otherClass && (this->respFile.gschars() > 1) )
                     etemp = L'x' ;
               }

               //* If auto processing AND if existing "reversed" token *
               //* reinsert the token.                                 *
               else if ( eDir == L'd' )
               {
                  wi = fmtOL.findlast( tagClose ) ;
                  fmtOL.insert( revDIR, wi ) ;
               }

               //* Append the modified OL tag and write to target *
               gsol.append( (wchar_t*)(etemp == L'x' ? (gsOrig.gstr()) : (fmtOL.gstr())) ) ;
               this->ppfWriteLine ( gsol ) ;

               if ( this->verbose )
                  gsVerb.append( fmtOL.gstr() ) ;
            }
            this->ppfWriteLine ( gs ) ;   // pass line item data through unmodified
         }

         //* Test for a <ul> list nested within current list.*
         else if ( (this->ppfTestUlistBegin ( gs, wi )) != false )
         {
            status = this->ppfProcUL ( gs ) ;
         }

         //* Test for an <ol> list nested within current list.*
         else if ( (this->ppfTestOlistBegin ( gs, wi )) != false )
         {
            status = this->ppfProcOL ( gs ) ;
         }

         //* If there is a preformatted block inside the list *
         else if ( (this->ppfTestFormattedBlock ( bType, gs )) != false )
         {
            //* Copy the block, optionally eliminating the unnecessary leading *
            //* blank line. NOTE: Data inside the block IS NOT modified.       *
            status = this->ppfProcFormattedBlock ( bType, gs ) ;
         }

         //* If beginning a quotation, smallquotation   *
         //* or largequotation block.                   *
         else if ( (this->ppfTestQuotationBlock ( bType, gs )) != false )
         {
            status = this->ppfProcQUOTE ( bType, gs ) ;
         }

         //* If beginning a verbatim, smallverbatim or largeverbatim block *
         else if ( (this->ppfTestVerbatimBlock ( bType, gs )) != false )
         {
            status = this->ppfProcVerbatim ( bType, gs ) ;
         }

         //* If beginning an indentedblock, smallindentedblock *
         //* or largeindentedblock.                            *
         else if ( (this->ppfTestIndentedBlock ( bType, gs )) != false )
         {
            status = this->ppfProcIDB ( bType, gs ) ;
         }

         //* If a <table> object found *
         else if ( (gs.find( tabBegin, wi, false )) >= ZERO )
         {
            status = this->ppfProcTAB ( gs ) ;
         }

         //* Process line as text data *
         else
         {
            this->ppfWriteLine ( gs ) ;   // pass line through unmodified
         }
      }
      else  done = true ;     // read error, abort the loop
   }
   while ( ! done ) ;

   return status ;

}  //* End ppfProcOL() *

//*************************
//*   ppfTestUlistBegin   *
//*************************
//******************************************************************************
//* Test the provided text for the beginning of a <ul> block.                  *
//*    See notes below.                                                        *
//*                                                                            *
//* Input  : gssrc : (by reference) contains the source data line being        *
//*                  processed                                                 *
//*          wi    : offset into 'gssrc' at which to begin search               *
//*                                                                            *
//* Returns: 'true'  if line contains a <ul class="no-bullet'> tag             *
//*          'false' otherwise                                                 *
//******************************************************************************
//* Programmer's Notes:                                                        *
//* 1) Normally, a <ul> tag is alone on a line, but in some cases it could be  *
//*    preceeded by a </p> (close paragraph) tag or a </pre> (close            *
//*    preformatted) or other data. This happens if the <ul> is immediately    *
//*    preceeded by one of the block commands.                                 *
//*                                                                            *
//* 2) We must distinguish between <ul> lists generated by the texi-to-HTML    *
//*    converter and HTML code embedded within the texi source document.       *
//*    There is no way to be 100% certain about this, but we "know" that the   *
//*    converter always ends the first line of the list immediately after      *
//*    the opening tag. Thus if there is data following the opening tag on the *
//*    same line, we assume that the list WAS NOT generated by the converter   *
//*    (or that it has already been post-processed).                           *
//*                                                                            *
//* 3) It is also remotely possible (though stupid) that the source document   *
//*    will have a list INSIDE a preformatted block. In this case, we could    *
//*    have some other end tag(s) leading the <ul>.  For example: </pre></div> *
//*    Here is the unbearable HTML for an @itemize inside a @display block:    *
//*                                                                            *
//*    <div class="display">                                                   *
//*    <pre class="display">Here we are inside an @display block.              *
//*    </pre><ul>                                                              *
//*    <li> <pre class="display">Bathe.                                        *
//*    </pre></li><li> <pre class="display">Brush your teeth.                  *
//*    </pre></li><li> <pre class="display">Shave unwanted hair.               *
//*    </pre></li></ul>                                                        *
//*    <pre class="display">Here we are inside an @display block.              *
//*    </pre></div>                                                            *
//*                                                                            *
//*    Note that we WILL identify the <ul> tag in the above mess; however, the *
//*    caller may choose not to process it.                                    *
//******************************************************************************

bool Idpp::ppfTestUlistBegin ( const gString& gssrc, short wi )
{
   bool ulListFound = false ;

   //* If source line contains '<ul>' tag AND it is last item on source line *
   if ( (wi = gssrc.find( ulBegin, wi, false, ulBegin_len )) >= ZERO )
   {
      wi = gssrc.after( tagClose, wi ) ;
      if ( (wi > ZERO) && (gssrc.gstr()[wi] == NULLCHAR) )
         ulListFound = true ;
   }

   return ulListFound ;

}  //* End ppfTestUlistBegin() *

//*************************
//*   ppfTestOlistBegin   *
//*************************
//******************************************************************************
//* Test the provided text for the beginning of an UNPROCESSED <ol> block.     *
//*                                                                            *
//* Input  : gssrc : (by reference) contains the source data line being        *
//*                  processed                                                 *
//*          wi    : offset into 'gssrc' at which to begin search               *
//*                                                                            *
//* Returns: 'true'  if line contains a plain (unprocessed) <ol> tag           *
//*          'false' otherwise                                                 *
//******************************************************************************
//* Programmer's Notes:                                                        *
//*  1) Normally, an <ol> tag is alone on a line, but occasionally it is       *
//*     preceeded by a </p> (close paragraph) tag or a </pre> (close           *
//*     preformatted). This happens if the <ol> is immediately preceeded by    *
//*     one of the block commands.                                             *
//* 2a) For makeinfo 5.x, all <ol> tags are declared without a class or style  *
//*     parameters, so if it has a class, then it has already been processed.  *
//* 2b) For makeinfo 6.x, some <ol> tags are plain and others specify a type   *
//*     and start value. "<ol>"  vs.  "<ol type="A" start="1"> or              *
//*     "<ol start="0">                                                        *
//*  3) Also, unprocessed <ol> tags are the last thing on the source line, so  *
//*     if there is additional data AFTER the <ol> tag, then the list has      *
//*     already been processed.                                                *
//*  4) We COULD BE fooled by HTML embedded directly into the texi source, but *
//*     if we are fooled, processing the list will not change its appearance   *
//*     unless user selects a non-default formatting option.                   *
//******************************************************************************

bool Idpp::ppfTestOlistBegin ( const gString& gssrc, short wi )
{
   bool olListFound = false ;

   //* If source line contains '<ol>' tag AND it is last item on source line *
   if ( (wi = gssrc.find( olBegin, wi, false, olBegin_len )) >= ZERO )
   {
      wi = gssrc.after( tagClose, wi ) ;
      if ( (wi > ZERO) && (gssrc.gstr()[wi] == NULLCHAR) )
         olListFound = true ;
   }
   return olListFound ;

}  //* End ppfTestOlistBegin() *

//**************************
//* ppfTestFormattedBlock  *
//**************************
//******************************************************************************
//* Test whether the provided data indicates the beginning of a class block    *
//* for 'format', 'display' 'example' or 'lisp', 'verbatim' or their           *
//* 'small' / 'large' counterparts.                                            *
//*                                                                            *
//*                                                                            *
//*                                                                            *
//* Input  : bType  : (by reference) recieves indicator of block type:         *
//*                   member of enum blkType                                   *
//*          gsln   : line of source data to be scanned                        *
//*                                                                            *
//* Returns: 'true' if this is a preformatted block header, else 'false'       *
//******************************************************************************
//* Programmer's Note:                                                         *
//* 1) All formatted blocks live inside a <div>...</div> sequence, EXCEPT for  *
//*    the "verbatim" block which lives inside a <pre>...</pre> sequence.      *
//* 2) All formatted blocks are defined as being within a class.               *
//* 3) Note that makeinfo does not define a "@smallverbatim" command, but we   *
//*    provide it as a post-processing option.                                 *
//* 4) Note that makeinfo does not define the "@large...." commands, but we    *
//*    provide them as post-processing options.                                *
//*                                                                            *
//******************************************************************************

bool Idpp::ppfTestFormattedBlock ( blkType& bType, const gString& gsln )
{
   const wchar_t* divclassBegin = L"<div class=\"" ;
   const short    divclassBegin_len = 12 ;
   const wchar_t* fClass = L"format\">" ;
   const short    fClass_len = 8 ;
   const wchar_t* sfClass = L"smallformat\">" ;
   const short    sfClass_len = 13 ;
   const wchar_t* lfClass = L"largeformat\">" ;
   const short    lfClass_len = 13 ;
   const wchar_t* dClass = L"display\">" ;
   const short    dClass_len = 9 ;
   const wchar_t* sdClass = L"smalldisplay\">" ;
   const short    sdClass_len = 14 ;
   const wchar_t* ldClass = L"largedisplay\">" ;
   const short    ldClass_len = 14 ;
   const wchar_t* eClass = L"example\">" ;
   const short    eClass_len = 9 ;
   const wchar_t* seClass = L"smallexample\">" ;
   const short    seClass_len = 14 ;
   const wchar_t* leClass = L"largeexample\">" ;
   const short    leClass_len = 14 ;
   const wchar_t* lClass = L"lisp\">" ;
   const short    lClass_len = 6 ;
   const wchar_t* slClass = L"smalllisp\">" ;
   const short    slClass_len = 11 ;
   const wchar_t* llClass = L"largelisp\">" ;
   const short    llClass_len = 11 ;

   short wi ;                 // offset index
   bool fmtBlock = false ;    // return value
   bType = btNone ;           // initialize caller's variable

   if ( (wi = gsln.find( divclassBegin )) >= ZERO )
   {
      fmtBlock = true ;          // tentatively identified
      wi += divclassBegin_len ;  // step over identified partial tag

      //* Now check for type of block *
      if ( (gsln.compare( fClass, false, fClass_len, wi )) == ZERO )
         bType = stdF ;    // 'format' class block
      else if ( (gsln.compare( sfClass, false, sfClass_len, wi )) == ZERO )
         bType = smaF ;    // 'smallformat' class block
      else if ( (gsln.compare( lfClass, false, lfClass_len, wi )) == ZERO )
         bType = lrgF ;    // 'largeformat' class block
      else if ( (gsln.compare( dClass, false, dClass_len, wi )) == ZERO )
         bType = stdD ;    // 'display' class block
      else if ( (gsln.compare( sdClass, false, sdClass_len, wi )) == ZERO )
         bType = smaD ;    // 'smalldisplay' class block
      else if ( (gsln.compare( ldClass, false, ldClass_len, wi )) == ZERO )
         bType = lrgD ;    // 'largedisplay' class block
      else if ( (gsln.compare( eClass, false, eClass_len, wi )) == ZERO )
         bType = stdE ;    // 'example' class block
      else if ( (gsln.compare( seClass, false, seClass_len, wi )) == ZERO )
         bType = smaE ;    // 'smallexample' class block
      else if ( (gsln.compare( leClass, false, leClass_len, wi )) == ZERO )
         bType = lrgE ;    // 'largeexample' class block
      else if ( (gsln.compare( lClass, false, lClass_len, wi )) == ZERO )
         bType = stdL ;    // 'lisp' class block
      else if ( (gsln.compare( slClass, false, slClass_len, wi )) == ZERO )
         bType = smaL ;    // 'smalllisp' class block
      else if ( (gsln.compare( llClass, false, llClass_len, wi )) == ZERO )
         bType = lrgL ;    // 'largelisp' class block
      else           // line does not delimit a formatted data block
         fmtBlock = false ;
   }
   return fmtBlock ;

}  //* End ppfTestFormattedBlock() *

//*************************
//* ppfTestIndentedBlock  *
//*************************
//******************************************************************************
//* Scan the source data for one of the "indentedblock" headers:               *
//*          indentedblock                                                     *
//*          smallindentedblock                                                *
//*          largeindentedblock                                                *
//* Note that an unstyled <blockquote> tag indicates a "quotation" block, NOT  *
//* an indentedblock (see below).                                              *
//*                                                                            *
//* Input  : bType  : (by reference) recieves indicator of block type:         *
//*                   member of enum blkType                                   *
//*          gsln   : line of source data to be scanned                        *
//*                                                                            *
//* Returns: 'true'  if an "indentedblock" header ('bType' initialized)        *
//*          'false' if not indentedblock header ('bType' unchanged)           *
//******************************************************************************

bool Idpp::ppfTestIndentedBlock ( blkType& bType, const gString&gsln )
{
   bool status = true ;

   if ( (gsln.find( indeInherit, ZERO, false )) >= ZERO )
      bType = stdI ;
   else if ( (gsln.find( indeSmall, ZERO, false )) >= ZERO )
      bType = smaI ;
   else if ( (gsln.find( indeLarge, ZERO, false )) >= ZERO )
      bType = lrgI ;
   else
      status = false ;

   return status ;

}  //* End ppfTestIndentedBlock() *

//*************************
//* ppfTestQuotationBlock *
//*************************
//******************************************************************************
//* Scan the source data for one of the "quotation" block headers:             *
//*          quotation (or an unstyled <blockquote>)                           *
//*          smallquotation                                                    *
//*          largequotation                                                    *
//*                                                                            *
//* Input  : bType  : (by reference) recieves indicator of block type:         *
//*                   member of enum blkType                                   *
//*          gsln   : line of source data to be scanned                        *
//*                                                                            *
//* Returns: 'true'  if a "quotation" block header ('bType' initialized)       *
//*          'false' if not quotation block header ('bType' unchanged)         *
//******************************************************************************

bool Idpp::ppfTestQuotationBlock ( blkType& bType, const gString&gsln )
{
   bool status = true ;

   if ( ((gsln.find( quotSimple, ZERO, false )) >= ZERO) ||
        ((gsln.find( quotInherit, ZERO, false )) >= ZERO) )
      bType = stdQ ;
   else if ( (gsln.find( quotSmall, ZERO, false )) >= ZERO )
      bType = smaQ ;
   else if ( (gsln.find( quotLarge, ZERO, false )) >= ZERO )
      bType = lrgQ ;
   else
      status = false ;

   return status ;

}  //* End ppfTestQuotationBlock() *

//*************************
//* ppfTestVerbatimBlock  *
//*************************
//******************************************************************************
//* Scan the source data for one of the "verbatim" block headers:              *
//*          verbatim                                                          *
//*          smallverbatim                                                     *
//*          largeverbatim                                                     *
//*                                                                            *
//* Input  : bType  : (by reference) recieves indicator of block type:         *
//*                   member of enum blkType                                   *
//*          gsln   : line of source data to be scanned                        *
//*                                                                            *
//* Returns: 'true'  if a "verbatim" block header ('bType' initialized)        *
//*          'false' if not verbatim block header ('bType' unchanged)          *
//******************************************************************************

bool Idpp::ppfTestVerbatimBlock ( blkType& bType, const gString&gsln )
{
   const wchar_t* preclassBegin = L"<pre class=\"" ;
   const short    preclassBegin_len = 12 ;
   const wchar_t* vClass = L"verbatim\">" ;
   const short    vClass_len = 10 ;
   const wchar_t* svClass = L"smallverbatim\">" ;
   const short    svClass_len = 15 ;
   const wchar_t* lvClass = L"largeverbatim\">" ;
   const short    lvClass_len = 15 ;

   short wi ;              // text index
   bool  status = false ;  // return value

   if ( (wi = gsln.find( preclassBegin )) >= ZERO )
   {
      status = true ;
      wi += preclassBegin_len ;  // step over identified partial tag
      if ( (gsln.compare( vClass, false, vClass_len, wi )) == ZERO )
         bType = stdV ;    // 'verbatim' class block
      else if ( (gsln.compare( svClass, false, svClass_len, wi )) == ZERO )
         bType = smaV ;    // 'smallverbatim' class block
      else if ( (gsln.compare( lvClass, false, lvClass_len, wi )) == ZERO )
         bType = lrgV ;    // 'largeverbatim' class block
      else           // line does not delimit a verbatim block
         status = false ;
   }
   return status ;

}  //* End ppfTestVerbatimBlock() *

//**************************
//* ppfProcFormattedBlock  *
//**************************
//******************************************************************************
//* Copy the contents of a pre-formatted block from source to target.          *
//* Optionally, eliminate the extra blank line before the block by combining   *
//* the <div> and <pre> tags onto a single line.                               *
//*                                                                            *
//*      Contents of preformatted blocks are copied as plain text,             *
//*                       EXCEPT:                                              *
//*      1) Formatted blocks within formatted blocks are processed.            *
//*      2) Beginning and end of <ol> and <ul> lists (remove inner <pre>).     *
//*         NOTE: There will be no user prompt for lists within formatted      *
//*               blocks.                                                      *
//*                                                                            *
//* Input  : bType  : indicates block type: member of enum blkType             *
//*          gsb    : (by reference) contains the first line of the block.     *
//*                                                                            *
//* Returns: OK if successful, ERR processing error                            *
//******************************************************************************

short Idpp::ppfProcFormattedBlock ( blkType bType, const gString& gsb )
{
   const wchar_t* blockEND = L"</div>" ;
   const wchar_t* preEND   = L"</pre>" ;

   gString gs ;               // source line data
   blkType eType = btNone ;   // if nested block, its type
   short wi,                  // string index
         nestCount = 1,       // depth of nesting
         listCount = ZERO,    // embedded <ul> and <ol> lists
         status = OK ;        // return value

   //* If specified, eliminate the leading blank line *
   // Programmer's Note: "no_block" if set, overrides user interaction.
   if ( this->no_bloc == false )
      status = this->ppfProcInnerBlock ( bType, gsb ) ;

   //* Else, copy the header line as-is *
   else
      this->ppfWriteLine ( gsb ) ;

   bool done = false ;
   while ( ! done && status == OK )
   {
      //*************************************************
      //*     Read lines from the source document.      *
      //*************************************************
      if ( (status = this->ppfReadSrcLine ( gs, wi )) == OK )
      {
         //* If end of block, then return to caller *
         if ( (gs.find( blockEND, ZERO, false )) >= ZERO )
         {
            //* Delete the orphaned block-end tag *
            if ( this->no_bloc == false )
            {
               short xi ;
               if ( (xi = gs.find( preEND )) >= ZERO )
                  gs.erase( preEND, xi ) ;
            }
            this->ppfWriteLine ( gs ) ;      // write line to target
            if ( --nestCount <= ZERO )
            {
               done = true ;
               break ;
            }
            else  // end of nested block found, keep processing
               continue ;
         }

         //* Test for <ol>, <ul> lists and other preformatted blocks INSIDE *
         //* this preformatted block. No intelligent user would do this,    *
         //* but if it happens, we need to identify it.                     *
         if ( ((this->ppfTestOlistBegin ( gs, ZERO )) != false)
              ||
              ((this->ppfTestUlistBegin ( gs, ZERO )) != false) )
         {
            this->ppfPFB_List ( gs ) ;       // strip the garbage
            this->ppfWriteLine ( gs ) ;      // write line to target
            ++listCount ;                    // count embedded lists
         }
         else if ( ((gs.find( ulEnd, ZERO, false )) >= ZERO)
                   ||
                   ((gs.find( olEnd, ZERO, false )) >= ZERO) )
         {
            this->ppfPFB_List ( gs ) ;       // strip the garbage
            this->ppfWriteLine ( gs ) ;      // write line to target
            //* The source line following the list unnecessarily *
            //* re-declares the containing block.                *
            this->ppfReadSrcLine ( gs, wi ) ;
            // Programmer's Note: When the source file has been previously 
            // processed, the source and target would be different without 
            // this conditional statement. If practical, we want a second pass 
            // on a previously-processed file to produce an exact copy of the 
            // source. This is not critical, but it is a matter of pride.
            if ( (gs.gschars()) > 1 )
               this->ppfPFB_List ( gs ) ;    // strip the garbage
            this->ppfWriteLine ( gs ) ;      // write line to target
            --listCount ;                    // embedded list complete
         }
         else if ( (this->ppfTestFormattedBlock ( eType, gs )) != false )
         {
            this->ppfWriteLine ( gs ) ;      // write line to target
            ++nestCount ;                    // nesting level
         }
         else
         {
            //* If inside an embedded list *
            if ( listCount > ZERO )
               this->ppfPFB_List ( gs ) ;    // strip the garbage
            this->ppfWriteLine ( gs ) ;      // write the line
         }
      }
   }
   return status ;

}  //* End ppfProcFormattedBlock() *

//*************************
//*    ppfProcVerbatim    *
//*************************
//******************************************************************************
//* Called by ppfProcFormattedBlock() method to process "verbatim" blocks.     *
//* Note that we simply copy the block, unmodified, from source to target      *
//* UNLESS the user specifies a font size for the block.                       *
//*                                                                            *
//* Input  : bType  : indicates block type: member of enum blkType             *
//*          gsb    : (by reference) contains the first line of the block.     *
//*                                                                            *
//* Returns: OK if successful, ERR processing error                            *
//******************************************************************************
//* Programmer's Note:                                                         *
//* Because the 'verbatim' block is a <pre>...</pre> construct, _AND_ because  *
//* texi2any combines the "</pre>" with whatever HTML follows it.              *
//* -- The HTML output doesn"t care, because the data is actually written to   *
//*    the target.                                                             *
//* -- However, if the post-processor needs to see that information, it will   *
//*    get confused when it doesn't.                                           *
//* -- If a blank line follows the 'verbatim' block, then the </pre> will be   *
//*    on the line by itself, and the caller will not miss anything.           *
//* -- The problem is that the user cannot be relied upon to remember the      *
//*    trailing blank line, so we must parse the line containing the "</pre>"  *
//*    and return any unused data to the caller. The elegant way to do this is *
//*    to push the unused data back into the input stream. Unfortunately, this *
//*    is not supported for input-only streams. Therefore, we have implemented *
//*    an input-stream push-back buffer 'pushBack[]' to preserve any data      *
//*    following the "</pre>" tag.                                             *
//*                                                                            *
//******************************************************************************

short Idpp::ppfProcVerbatim ( blkType bType, const gString& gsb )
{
   const wchar_t* verbInherit = L"<pre class=\"verbatim\">" ;
   const wchar_t* verbSmall   = L"<pre class=\"smallverbatim\">" ;
   const wchar_t* verbLarge   = L"<pre class=\"largeverbatim\">" ;
   const wchar_t* verbEND     = L"</pre>" ;
   const wchar_t* blockName[] = 
   {
      L"Verbatim",   L"smallVerbatim",   L"largeVerbatim",
   } ;

   gString gs = gsb,          // source line data
           gst, gsIn ;        // user interaction
   wchar_t usrResp = L'a' ;   // user response to prompt
   short wi,                  // string index
         pi,                  // index of <pre...  tag
         bnIndex = ZERO,      // index into blockName[] array
         status = OK ;        // return value
   bool  done = false ;       // loop control

   //* Erase the existing tag *
   if ( (pi = gs.find( verbInherit )) >= ZERO )
      gs.erase( verbInherit, pi ) ;
   else if ( (pi = gs.find( verbSmall )) >= ZERO )
      gs.erase( verbSmall, pi ) ;
   else if ( (pi = gs.find( verbLarge )) >= ZERO )
      gs.erase( verbLarge, pi ) ;

   //* If user interaction is enabled, *
   //* prompt for user's selection.    *
   if ( this->blkFont == cfgSpec )
   {
      //* Index the block-name text for display *
      if ( bType == stdV )       bnIndex = ZERO ;
      else if ( bType == smaV )  bnIndex = 1 ;
      else if ( bType == lrgV )  bnIndex = 2 ;

      //* Point to the text of this line *
      if ( (wi = gsb.find( verbInherit, ZERO, false, 4 )) >= ZERO )
         wi = gsb.after( tagClose, wi ) ;
      if ( wi < ZERO )
         wi = ZERO ;

      //* Ask the user to select a font-size option *
      usrResp = this->ppfBlockPrompt ( blockName[bnIndex], &gsb.gstr()[wi] ) ;
   }

   //* Update the tag according to user's selection *
   if ( pi >= ZERO )
   {
      switch ( usrResp )
      {
         //** inherited font **
         case L'i':  gs.insert( verbInherit, pi ) ; break ;
         //** smaller font **
         case L's':  gs.insert( verbSmall, pi ) ;   break ;
         //** larger font **
         case L'l':  gs.insert( verbLarge, pi ) ;   break ;
         //** Automatic assignment of class       **
         //** If font size previously specified,  **
         //** use it, else use standard font size.**
         case L'a':
         default:
            if ( bType == smaV )
               gs.insert( verbSmall, pi ) ;
            else if ( bType == lrgV )
               gs.insert( verbLarge, pi ) ;
            else
               gs.insert( verbInherit, pi ) ;
            break ;
      } ;
   }
   else
   {
      gs = gsb ;     // copy line as-written
      status = ERR ;
   }

   //* Write the header line *
   this->ppfWriteLine ( gs ) ;

   done = false ;
   while ( ! done && status == OK )
   {
      //*************************************************
      //*     Read lines from the source document.      *
      //*************************************************
      if ( (status = this->ppfReadSrcLine ( gs, wi )) == OK )
      {
         if ( (wi = gs.find( verbEND, wi, false )) >= ZERO )
         {
            wi = gs.after( tagClose, wi ) ;
            if ( gs.gstr()[wi] != NULLCHAR )
            {
               gString gstmp = &gs.gstr()[wi] ;
               gs.limitChars( wi ) ;
               this->ppfUnReadSrcLine ( gstmp ) ;
            }
            done = true ;
         }
         this->ppfWriteLine ( gs ) ;
      }
   }
   return status ;

}  //* End ppfProcVerbatim() *

//*************************
//*      ppfPFB_List      *
//*************************
//******************************************************************************
//* Called only by ppfProcFormattedBlock(), and ONLY if embedded lists have    *
//* been identified. Removes extra HTML garbage from the lists to make them    *
//* readable.                                                                  *
//*                                                                            *
//* Input  : gsLine : (by reference) contains the line data to be scanned      *
//*                   On return, contains the reformatted data.                *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************
//* Notes:                                                                     *
//* 1) If the list is IMMEDIATELY follows the block declaration, the block     *
//* and/or the list it contains may not be recognized. Therefore, ALWAYS place *
//* a blank line above a list embedded within a preformatted block.            *
//* 2) The source line following the end of the list is an unnecessary         *
//*    re-declaration of the block containing the list:                        *
//*        Example:  <pre class="display">                                     *
//*    This line has no value and should also be processed here.               *
//******************************************************************************

void Idpp::ppfPFB_List ( gString& gsLine )
{
   const wchar_t* preBEGIN = L"<pre" ;
   const wchar_t* preEND   = L"</pre>" ;

   short wi,         // data indices
         ei ;

   //* If special list processing is enabled *
   if ( this->liSpec )
   {
      if ( (wi = gsLine.compare( preEND, false, 6 )) == ZERO )
      {
         gsLine.erase( preEND, ZERO, false ) ;
      }
      if ( ((wi = gsLine.find( preBEGIN, ZERO, false )) >= ZERO) &&
           ((ei = gsLine.find( tagClose, wi )) > wi ) )
      {
         gsLine.erase( wi, (ei - wi + 1) ) ;
      }
   }
   return ;

}  //* End ppfPFB_List() *

//**************************
//*   ppfProcInnerBlock    *
//**************************
//******************************************************************************
//* Eliminate the extra blank line before the block by deleting the inner      *
//* <pre class="...."> tag. It is the caller's responsibility to delete the    *
//* closing </pre> tag for the block.                                          *
//*                                                                            *
//* If specified, prompt the user for the desired font size for the block.     *
//*                                                                            *
//*                                                                            *
//* Input  : bType  : indicates block type: member of enum blkType             *
//*          gsb    : (by reference) contains the first line of the block.     *
//*                                                                            *
//* Returns: OK if successful, ERR processing error                            *
//******************************************************************************

short Idpp::ppfProcInnerBlock ( blkType bType, const gString& gsb )
{
   const wchar_t* formInherit = L"<div class=\"format\">" ;
   const wchar_t* formSmall   = L"<div class=\"smallformat\">" ;
   const wchar_t* formLarge   = L"<div class=\"largeformat\">" ;
   const wchar_t* form_iPRE   = L"<pre class=\"format\">" ;
   const wchar_t* form_sPRE   = L"<pre class=\"smallformat\">" ;
   const wchar_t* form_lPRE   = L"<pre class=\"largeformat\">" ;

   const wchar_t* dispInherit = L"<div class=\"display\">" ;
   const wchar_t* dispSmall   = L"<div class=\"smalldisplay\">" ;
   const wchar_t* dispLarge   = L"<div class=\"largedisplay\">" ;
   const wchar_t* disp_iPRE   = L"<pre class=\"display\">" ;
   const wchar_t* disp_sPRE   = L"<pre class=\"smalldisplay\">" ;
   const wchar_t* disp_lPRE   = L"<pre class=\"largedisplay\">" ;

   const wchar_t* examInherit = L"<div class=\"example\">" ;
   const wchar_t* examSmall   = L"<div class=\"smallexample\">" ;
   const wchar_t* examLarge   = L"<div class=\"largeexample\">" ;
   const wchar_t* exam_iPRE   = L"<pre class=\"example\">" ;
   const wchar_t* exam_sPRE   = L"<pre class=\"smallexample\">" ;
   const wchar_t* exam_lPRE   = L"<pre class=\"largeexample\">" ;

   const wchar_t* lispInherit = L"<div class=\"lisp\">" ;
   const wchar_t* lispSmall   = L"<div class=\"smalllisp\">" ;
   const wchar_t* lispLarge   = L"<div class=\"largelisp\">" ;
   const wchar_t* lisp_iPRE   = L"<pre class=\"lisp\">" ;
   const wchar_t* lisp_sPRE   = L"<pre class=\"smalllisp\">" ;
   const wchar_t* lisp_lPRE   = L"<pre class=\"largelisp\">" ;

   const wchar_t* unkDisp = L"? ? ?" ;

   const wchar_t* blockName[] = 
   {
      L"Format",   L"smallFormat",   L"largeFormat",
      L"Display",  L"smallDisplay",  L"largeDisplay",
      L"Example",  L"smallExample",  L"largeExample",
      L"Lisp",     L"smallLisp",     L"largeLisp",
   } ;

   gString gs,                // source line data
           gst, gsIn,         // user interaction
           gsB = gsb,         // working copy of caller's data
           gsdisp,            // display text for user interaction
           gsOut ;            // output data for target file
   wchar_t usrResp = L'a' ;   // user response to prompt ("automatic" is the default)
   short wi,                  // string index
         hi,                  // insertion point in header
         pi,                  // index of data following "<pre..." tag
         bnIndex = ZERO ;     // index into blockName[] array
   short status = OK ;        // return value

   if ( (status = this->ppfReadSrcLine ( gs, wi )) == OK )
   {
      if ( (pi = gs.find( L"<pre", wi, false, 4 )) < ZERO )
      {
         this->ppfUnReadSrcLine ( gs ) ;
         gs.clear() ;
      }

      //* If user interaction is enabled, *
      //* prompt for user's selection.    *
      if ( this->blkFont == cfgSpec )
      {
         if ( pi >= ZERO )
         {
            if ( (pi = gs.after( tagClose )) >= ZERO )
               gsdisp = &gs.gstr()[pi] ;
            else  // (unlikely)
               gsdisp = unkDisp ;
         }
         else  // (pi < ZERO)
         {
            if ( (pi = gsB.find( formInherit, ZERO, false, 4 )) >= ZERO )
            {
               if ( (pi = gsB.after( tagClose, pi )) >= ZERO )
                  gsdisp = &gsB.gstr()[pi] ;
               else
                  gsdisp = unkDisp ;
            }
            else
               gsdisp = unkDisp ;
         }

         switch ( bType )
         {
            case stdF:     bnIndex = ZERO ;  break ;
            case smaF:     bnIndex = 1 ;     break ;
            case lrgF:     bnIndex = 2 ;     break ;
            case stdD:     bnIndex = 3 ;     break ;
            case smaD:     bnIndex = 4 ;     break ;
            case lrgD:     bnIndex = 5 ;     break ;
            case stdE:     bnIndex = 6 ;     break ;
            case smaE:     bnIndex = 7 ;     break ;
            case lrgE:     bnIndex = 8 ;     break ;
            case stdL:     bnIndex = 9 ;     break ;
            case smaL:     bnIndex = 10 ;    break ;
            case lrgL:     bnIndex = 11 ;    break ;
            case stdI: case smaI: case lrgI: // silence compiler warning
            case stdQ: case smaQ: case lrgQ: 
            case stdV: case smaV: case lrgV: case btNone: break ;
         } ;

         //* Ask the user to select a font-size option *
         //* usrResp: a == automatic
         //*          i == inherit
         //*          s == smaller
         //*          l == larger
         usrResp = this->ppfBlockPrompt ( blockName[bnIndex], gsdisp.gstr() ) ;
      }        // user prompt

      //* Update the tag according to user's selection *
      bool repo = false ;     // 'true' if block repositioned

      //*********************
      //** "Format" group  **
      //*********************
      if ( (bType == stdF) || (bType == smaF) ||(bType == lrgF) )
      {
         //* Erase the existing tag *
         if ( (hi = gsB.find( formInherit )) >= ZERO )
            gsB.erase( formInherit, hi ) ;
         else if ( (hi = gsB.find( formSmall )) >= ZERO )
            gsB.erase( formSmall, hi ) ;
         else if ( (hi = gsB.find( formLarge )) >= ZERO )
            gsB.erase( formLarge, hi ) ;

         //* Set font size specified by user *
         if ( hi >= ZERO )
         {
            switch ( usrResp )
            {
               //** inherited font **
               case L'i':  gsB.insert( formInherit, hi ) ; break ;
               //** smaller font **
               case L's':  gsB.insert( formSmall, hi ) ;   break ;
               //** larger font **
               case L'l':  gsB.insert( formLarge, hi ) ;   break ;
               //** Automatic assignment of class       **
               //** If font size previously specified,  **
               //** use it, else use standard font size.**
               case L'a':
               default:
                  if ( bType == smaF )
                     gsB.insert( formSmall, hi ) ;
                  else if ( bType == lrgF )
                     gsB.insert( formLarge, hi ) ;
                  else
                     gsB.insert( formInherit, hi ) ;
                  break ;
            } ;
         }

         //* If inner <pre...> found, erase it, then  *
         //* Concatenate <div...> and first data line.*
         if ( (pi = gs.find( form_iPRE, wi, false )) >= ZERO )
            gs.erase( form_iPRE, pi, false ) ;
         else if ( (pi = gs.find( form_sPRE, wi, false )) >= ZERO )
            gs.erase( form_sPRE, pi, false ) ;
         else if ( (pi = gs.find( form_lPRE, wi, false )) >= ZERO )
            gs.erase( form_lPRE, pi, false ) ;
      }

      //*********************
      //** "Display" group **
      //*********************
      else if ( (bType == stdD) || (bType == smaD) || (bType == lrgD) )
      {
         //* Erase the existing tag *
         if ( (hi = gsB.find( dispInherit )) >= ZERO )
            gsB.erase( dispInherit, hi ) ;
         else if ( (hi = gsB.find( dispSmall )) >= ZERO )
            gsB.erase( dispSmall, hi ) ;
         else if ( (hi = gsB.find( dispLarge )) >= ZERO )
            gsB.erase( dispLarge, hi ) ;

         //* Set font size specified by user *
         if ( hi >= ZERO )
         {
            switch ( usrResp )
            {
               //** inherited font **
               case L'i':  gsB.insert( dispInherit, hi ) ; break ;
               //** smaller font **
               case L's':  gsB.insert( dispSmall, hi ) ;   break ;
               //** larger font **
               case L'l':  gsB.insert( dispLarge, hi ) ;   break ;
               //** Automatic assignment of class       **
               //** If font size previously specified,  **
               //** use it, else use standard font size.**
               case L'a':
               default:
                  if ( bType == smaD )
                     gsB.insert( dispSmall, hi ) ;
                  else if ( bType == lrgD )
                     gsB.insert( dispLarge, hi ) ;
                  else
                     gsB.insert( dispInherit, hi ) ;
                  break ;
            } ;
         }

         //* If inner <pre...> found, erase it, then  *
         //* Concatenate <div...> and first data line.*
         if ( (pi = gs.find( disp_iPRE, wi, false )) >= ZERO )
            gs.erase( disp_iPRE, pi, false ) ;
         else if ( (pi = gs.find( disp_sPRE, wi, false )) >= ZERO )
            gs.erase( disp_sPRE, pi, false ) ;
         else if ( (pi = gs.find( disp_lPRE, wi, false )) >= ZERO )
            gs.erase( disp_lPRE, pi, false ) ;
      }

      //*********************
      //** "Example" group **
      //*********************
      else if ( (bType == stdE) || (bType == smaE) || (bType == lrgE) )
      {
         //* Erase the existing tag *
         if ( (hi = gsB.find( examInherit )) >= ZERO )
            gsB.erase( examInherit, hi ) ;
         else if ( (hi = gsB.find( examSmall )) >= ZERO )
            gsB.erase( examSmall, hi ) ;
         else if ( (hi = gsB.find( examLarge )) >= ZERO )
            gsB.erase( examLarge, hi ) ;

         //* Set font size specified by user *
         if ( hi >= ZERO )
         {
            switch ( usrResp )
            {
               //** inherited font **
               case L'i':  gsB.insert( examInherit, hi ) ; break ;
               //** smaller font **
               case L's':  gsB.insert( examSmall, hi ) ;   break ;
               //** larger font **
               case L'l':  gsB.insert( examLarge, hi ) ;   break ;
               //** Automatic assignment of class       **
               //** If font size previously specified,  **
               //** use it, else use standard font size.**
               case L'a':
               default:
                  if ( bType == smaE )
                     gsB.insert( examSmall, hi ) ;
                  else if ( bType == lrgE )
                     gsB.insert( examLarge, hi ) ;
                  else
                     gsB.insert( examInherit, hi ) ;
                  break ;
            } ;
         }

         //* If inner <pre...> found, erase it, then  *
         //* Concatenate <div...> and first data line.*
         if ( (pi = gs.find( exam_iPRE, wi, false )) >= ZERO )
            gs.erase( exam_iPRE, pi, false ) ;
         else if ( (pi = gs.find( exam_sPRE, wi, false )) >= ZERO )
            gs.erase( exam_sPRE, pi, false ) ;
         else if ( (pi = gs.find( exam_lPRE, wi, false )) >= ZERO )
            gs.erase( exam_lPRE, pi, false ) ;
      }

      //*********************
      //** "Lisp" group    **
      //*********************
      else if ( (bType == stdL) || (bType == smaL) || (bType == lrgL) )
      {
         //* Erase the existing tag *
         if ( (hi = gsB.find( lispInherit )) >= ZERO )
            gsB.erase( lispInherit, hi ) ;
         else if ( (hi = gsB.find( lispSmall )) >= ZERO )
            gsB.erase( lispSmall, hi ) ;
         else if ( (hi = gsB.find( lispLarge )) >= ZERO )
            gsB.erase( lispLarge, hi ) ;

         //* Set font size specified by user *
         if ( hi >= ZERO )
         {
            switch ( usrResp )
            {
               //** inherited font **
               case L'i':  gsB.insert( lispInherit, hi ) ; break ;
               //** smaller font **
               case L's':  gsB.insert( lispSmall, hi ) ;   break ;
               //** larger font **
               case L'l':  gsB.insert( lispLarge, hi ) ;   break ;
               //** Automatic assignment of class       **
               //** If font size previously specified,  **
               //** use it, else use standard font size.**
               case L'a':
               default:
                  if ( bType == smaL )
                     gsB.insert( lispSmall, hi ) ;
                  else if ( bType == lrgL )
                     gsB.insert( lispLarge, hi ) ;
                  else
                     gsB.insert( lispInherit, hi ) ;
                  break ;
            } ;
         }

         //* If inner <pre...> found, erase it, then  *
         //* Concatenate <div...> and first data line.*
         if ( (pi = gs.find( lisp_iPRE, wi, false )) >= ZERO )
            gs.erase( lisp_iPRE, pi, false ) ;
         else if ( (pi = gs.find( lisp_sPRE, wi, false )) >= ZERO )
            gs.erase( lisp_sPRE, pi, false ) ;
         else if ( (pi = gs.find( lisp_lPRE, wi, false )) >= ZERO )
            gs.erase( lisp_lPRE, pi, false ) ;
      }
      
      else     // this is unlikely and would indicate programmer error
         status = ERR ;

      if ( status == OK )
      {
         //* Format the data for output
         if ( pi >= ZERO )
         {
            gsOut.compose( L"%S%S", gsB.gstr(), gs.gstr() ) ;
            repo = true ;
         }
         else
            gsOut = gsB ;

         //* Write the re-formatted data to target *
         this->ppfWriteLine ( gsOut ) ;
         if ( repo == false )
            ++this->tlCount ; // if we have just output 2 lines in one write

         if ( this->verbose != false && repo != false )
         {
            gString gsVerb( "(%4hd) Formatted Block repositioned.", &this->tlCount ) ;
            this->textOut ( gsVerb ) ;
         }
      }
   }        // read source line
   else     // read error - save current line
      this->ppfWriteLine ( gsb ) ;
   return status ;

}  //* End ppfProcInnerBlock() *

//*************************
//*    ppfBlockPrompt     *
//*************************
//******************************************************************************
//* When processing one of the block segments AND when user interaction is     *
//* enabled, prompt the user for the font size to be applied to the block.     *
//*                                                                            *
//* If user's response is: L'A', set the 'blkFont' member to 'cfgAuto'         *
//*                              AND return L'a'.                              *
//*                                                                            *
//* Input  : blkName  : display text identifying the type of block being       *
//*                     processed                                              *
//*          firstLine:                                                        *
//*                                                                            *
//* Returns: user selection:                                                   *
//*             L'a'   automatic (default)                                     *
//*             L'i'   inherit font size from container                        *
//*             L's'   10% smaller than inherited font size                    *
//*             L'l'   10% larger than inherited font size                     *
//******************************************************************************

wchar_t Idpp::ppfBlockPrompt ( const wchar_t* blkName, const wchar_t* firstLine )
{
   gString gst,               // output formatting
           gsIn ;             // user response
   wchar_t usrResp = L'a' ;   // return value
   bool    done = false ;     // loop control

   gst.compose( L"___________________________________\n"
                 "Block: \"%S\" (source line:%hu)\n"
                 "First Line    : %S\n"
                 "Choose font size:\n"
                 "i:inherit (std.)\n"
                 "s:smaller (-10%%)   a:automatic (default)\n"
                 "l:larger  (+10%%)   A:All automatic",
                blkName, &this->slCount, firstLine ) ;
   this->textOut ( gst ) ;
   this->textOut ( L"your choice: ", false ) ;

   while ( ! done )
   {
      if ( (this->userResponse ( gsIn )) == OK )
      {
         //* If user specified a default response *
         if ( (gsIn.compare( dToken, true, dToken_len )) == ZERO )
            usrResp = L'a' ;
         else        // single-character response expected
            usrResp = *gsIn.gstr() ;
         //* If this block, and all remaining blocks *
         //* are to be processed automagically       *
         if ( usrResp == L'A' )
         { usrResp = L'a' ; this->blkFont = cfgAuto ; }

         if ( (usrResp == L'a') || (usrResp == L'i') || 
              (usrResp == L's') || (usrResp == L'l') )
            break ;     // good input
         //* Call user a dumbguy and ask for correct response.*
         this->invalidResponse () ;
      }
   }     // while()

   return usrResp ;

}  //* End ppfBlockPrompt() *

//*************************
//*      ppfProcGNU       *
//*************************
//******************************************************************************
//* Process the GNU General Public License sub-document.                       *
//*  OR                                                                        *
//* Process the GNU Free Documentation License sub-document.                   *
//*                                                                            *
//* 1) Start the top-level enumerated list at item '0' (zero).                 *
//* 2) Convert the three (3) nested lists in the GPL to 'a.' 'b.' 'c.' lists.  *
//* 3) Convert the one (1) nested list in the FDL to an 'A.' 'B.' 'C.' list.   *
//*                                                                            *
//* Input  : gpl    : 'true' if we are processing the GPL text                 *
//*                   'false' if we are processing the FDL text                *
//*                                                                            *
//* Returns: OK if successful, ERR processing error                            *
//******************************************************************************
//* NOTES:                                                                     *
//* -- GPL and FDL license text:                                               *
//*    We silently correct the enumerated lists in these documents if detected.*
//*    Because it is boilerplate AND because by law, we as licensees are not   *
//*    allowed to modify it, we can be reasonably confident that it can be     *
//*    correctly identified. For this reason, our tests for the tags are quite *
//;    simple. However, past or future versions of the licenses MAY cause      *
//*    problems. Also, if the texi-to-HTML converter changes the way it formats*
//*    <ol> lists, problems _could_ arise.                                     *
//*    -- Note that the list-formatting methods are not called from this       *
//*       method. Instead, we directly modify the list headers. This means     *
//*       that the user is not prompted for these lists.                       *
//* -- The .texi source for the GPL and FDL licenses call out "smallexample"   *
//*    blocks; however, as of texi2any v:6.6 the "@small...." commands are no  *
//*    longer recognized. This leaves us with a decision:                      *
//*     a) Accept the (lazy-programmer) texi2any mistake and render the blocks *
//*        in the default font size.                                           *
//*     b) Assume that the .texi source intended "@smallexample" and repair    *
//*        the HTML.                                                           *
//*    This decision is encapsulated by the "SMALL_WORLD" definition below.    *
//*    The flag which indicates the user prompt for block constructs is        *
//*    temporarily disabled, so user will not be prompted to specify these     *
//*    blocks.                                                                 *
//* -- The HTML contains a (centered) version number line, immediately followed*
//*    by a "display" formatted block. Because we automatically correct the    *
//*    formatting for this block, we must insert a newline between the version *
//*    number and the "display" block.                                         *
//* -- This code is based on GPL v:3 and FDL v:1.3. These are the current      *
//*    versions of the licenses (2019) and have been stable since 2008.        *
//* -- We aren't sure whether to offer the user a flag to disable these        *
//*    automatic formatting operations; so at this time, we correct the        *
//*    formatting whether the user wants it or not.                            *
//*                                                                            *
//******************************************************************************

short Idpp::ppfProcGNU ( bool gpl )
{
   const wchar_t* gnu0start = L"<ol start=\"0\">" ;
   const wchar_t* gnu0class = L"<ol class=\"enum-decimal\" start=\"0\">" ;
   const wchar_t* gnuastart = L"<ol type=\"a\" start=\"1\">" ;
   const wchar_t* gnuaclass = L"<ol class=\"enum-lower-alpha\" start=\"1\">" ;
   const wchar_t* gnuAstart = L"<ol type=\"A\" start=\"1\">" ;
   const wchar_t* gnuAclass = L"<ol class=\"enum-upper-alpha\" start=\"1\">" ;
   const wchar_t* gnuEnd    = L"The GNU General Public License does not permit "
                               "incorporating your" ;
   const wchar_t* fdlEnd    = L"If your document contains nontrivial examples "
                               "of program code," ;
   const wchar_t* gnuVersion = L"<div align=\"center\">Version" ;

   //* Compensate for texi2any error (see note above). *
   #define SMALL_WORLD (1)
   #if SMALL_WORLD != 0
   Cfg tmpblkFont = this->blkFont ; // remember the global setting
   this->blkFont = cfgAuto ;        // temporarily disable user prompt
   #endif   // SMALL_WORLD

   gString gs ;                     // source data
   short wi ;
   short status = OK ;              // return value
   blkType bType = btNone ;         // type of block found
   bool  done = false ;             // loop control

   do
   {
      if ( (status = this->ppfReadSrcLine ( gs, wi )) == OK )
      {
         //* Each license has a "display" block near the top, and *
         //* at least two(2) "example" ("smallexample") blocks.   *
         if ( (this->ppfTestFormattedBlock ( bType, gs )) != false )
         {
            #if SMALL_WORLD != 0
            //* Pretest the formatted block, and if it it is a "example" *
            //* block, convert it to "smallexample" before processing.   *
            if ( (wi = gs.find( L"<div class=\"example\">" )) >= ZERO )
            {
               wi = gs.after( L'"' ) ;
               gs.insert( L"small", wi ) ;
               bType = smaE ;
            }
            #endif   // SMALL_WORLD

            status = this->ppfProcFormattedBlock ( bType, gs ) ;
            continue ;
         }

         //* If end-of-license found, read through the end *
         //* of the current paragraph.                     *
         if ( ((gs.find( gnuEnd )) >= ZERO) ||
              ((gs.find( fdlEnd )) >= ZERO) )
         {
            do
            {
               this->ppfWriteLine ( gs ) ;   // write the line to target
               if ( (status = this->ppfReadSrcLine ( gs, wi )) != OK )
                  break ;
            }
            while ( (gs.find( L"</p>" )) < ZERO ) ;
            done = true ;
         }

         // Programmer's Note: This is a bit dangerous that the last paragraph 
         // of the license begins with the above text because FSF may change 
         // the text of the license, so we also scan for an "<hr>" which 
         // texi2any uses to end a chapter.
         else if ( (gs.find( L"<hr>" )) >= ZERO )
            done = true ;

         //* If zero-based decimal enumerated list *
         else if ( (wi = (gs.find( gnu0start, ZERO, false ))) >= ZERO )
            gs.replace( gnu0start, gnu0class, wi, false ) ;

         //* If lower-alpha enumerated list *
         if ( (wi = (gs.find( gnuastart, ZERO, true ))) >= ZERO )
            gs.replace( gnuastart, gnuaclass, wi ) ;

         //* If upper-alpha enumerated list *
         else if ( (wi = (gs.find( gnuAstart, ZERO, true ))) >= ZERO )
            gs.replace( gnuAstart, gnuAclass, wi ) ;

         //* If "Version" notice, append a newline (see note above).*
         else if ( ((gs.find( gnuVersion )) >= ZERO) &&
                   ((gs.find( L"<br><br>" )) < ZERO) )
            gs.append( L"<br><br>" ) ;

         this->ppfWriteLine ( gs ) ;   // write the line to target
      }
   }
   while ( !done && (status == OK) ) ;

   #if SMALL_WORLD != 0
   this->blkFont = tmpblkFont ;     // restore the global setting
   #endif   // SMALL_WORLD

   return status ;

}  //* End ppfProcGNU() *

//**************************
//*     ppfTestComment     *
//**************************
//******************************************************************************
//* Test whether the data begin with an open-comment token.                    *
//*                                                                            *
//* Input  : gsln   : (by reference) line of source data to be scanned         *
//*                                                                            *
//* Returns: 'true' if comment identified, else 'false'                        *
//******************************************************************************

bool Idpp::ppfTestComment ( const gString& gsln )
{
   short indx = gsln.scan() ;       // step over leading whitespace
   return ( bool((gsln.find( commBegin, indx )) == indx) ) ;

}  //* End ppfTestComment() *

//**************************
//*     ppfProcComment     *
//**************************
//******************************************************************************
//* Process the comment at the top of 'gsc' and return any portion of the      *
//* data which are not part of the comment.                                    *
//*                                                                            *
//* Input  : gsc    : (by reference)                                           *
//*                                                                            *
//* Returns: 'true'  if full comment processed                                 *
//*                  ('gsc' will contain any non-comment data)                 *
//*          'false' if end-of-comment not found                               *
//*                  (comment continues on next line)                          *
//******************************************************************************
//* Programmer's Note: The texi2any converter will generate no comments as     *
//* part of the texi-2-html conversion; however, the texi source may contain   *
//* HTML comments similar to:                                                  *
//* @html                                                                      *
//* <!-- The code generated by this section may require                        *
//*      manual post-processing. -->                                           *
//* @end html                                                                  *
//*                                                                            *
//* This author often uses this method during development of the 'idpp'        *
//* post-processor to identify texi-source constructs that may not be          *
//* parsed correctly. For our purposes, all comments are contained on a single *
//* source line, AND these comments stand alone on the line, and are not       *
//* combined with other HTML tokens.                                           *
//* Multi-line comments or mixed comment/non-comment data may not be properly  *
//* handled by 'idpp'.                                                         *
//******************************************************************************

bool Idpp::ppfProcComment ( gString& gsc )
{
   gString gsOut ;
   short indx ;
   bool  commComplete = false ;

   if ( (indx = gsc.after( commEnd )) >= ZERO )
   {
      gsc.substr( gsOut, ZERO, indx ) ;   // extract the comment substring
      this->ppfWriteLine ( gsOut ) ;      // write the comment substring
      gsc.shiftChars( -(indx) ) ;         // remove substring from source
      commComplete = true ;               // comment fully processed
   }
   else     // write the entire line
   {
      this->ppfWriteLine ( gsc ) ;
      gsc.clear() ;
   }

   return commComplete ;

}  //* End ppfProcComment() *

//*************************
//*   ppfFormatLineItem   *
//*************************
//******************************************************************************
//* For line items of a UL list that has been declared with:                   *
//*             <ul class="no-bullet">                                         *
//*                                                                            *
//* Scan for the embedded bullet character, and if a recognized bullet         *
//* character is the first non-space character following the <li> tag,         *
//* process it in one of two ways:                                             *
//*  1) Adjust the indentation of the line using a CSS style element.          *
//*     This will cause the line data to be shifted left by one character cell *
//*     which will give the appearance of a bullet list using the embedded     *
//*     bullet character.                                                      *
//*     -- The embedded bullet character (if identified) is enclosed within a  *
//*        monospace style element to ensure that it fills the entire character*
//*        space.                                                              *
//*  2) Special processing is performed for four(4) embedded bullet            *
//*     characters:                                                            *
//*       -- the discBullet character   (\x23FA)                               *
//*       -- the circleBullet character (\x26AC _or_ \x26AA)                   *
//*       -- the squareBullet character (\x25AA)                               *
//*       -- the "&deg;" (degree symbol) character (\x00B0)                    *
//*     If special automatic beautification is enabled, these characters are   *
//*     deleted from the line item so that the caller may define the list      *
//*     using the appropriate bullet-class definition.                         *
//*                                                                            *
//* -- Note that only a plain "<li>" tag should arrive here.                   *
//*    An <li ...> tag with style or other information has already been        *
//*    processed, and should not be seen here.                                 *
//* -- "no-bullet" line items often contain a completely useless HTML comment. *
//*    If such a comment is present, it is removed.                            *
//*                                                                            *
//* Input  : gsitem   : (by reference) source line item                        *
//*          liIndex  : data offset where line item <li> begins                *
//*          eraseBull: (optional, 'false' by default)                         *
//*                     'false' erase embedded bullet characters only for the  *
//*                             special bullets                                *
//*                     'true'  erase all embedded bullet characters           *
//*                                                                            *
//* Returns: if recognized bullet-item character, return it                    *
//*          if recognized bullet-character definition, return '&'             *
//*          if recognized bullet char not found, return space (L' ').         *
//******************************************************************************

wchar_t Idpp::ppfFormatLineItem ( gString& gsitem, short liIndex, bool eraseBull )
{
   const wchar_t* monoStyleBegin = L"<span style=\"font-family:monospace;\">" ;
   const wchar_t* monoStyleEnd = L"</span>" ;
   const wchar_t* liOutdent = L"<li style=\"text-indent:-1.0em;\">" ;
   const short    vbcCOUNT = 23 ;
   const wchar_t ValidBulletChars[] = 
   {
      L'\x002D',     // ASCII minus    (-)
      L'\x00B0',     // &deg;          (°)
      L'\x003E',     // &gt;           (>)
      L'\x2022',     // &bullet        (•)
      L'\x2013',     // &ndash;        (–)
      L'\x2014',     // &mdash;        (—)   (macro @EMDASH)
      L'\x2212',     // Unicode minus  (−)   (macro @UMINUS)
      L'\x2660',     // spade          (♠)
      L'\x2663',     // club           (♣)
      L'\x2665',     // heart          (♥)
      L'\x2666',     // diamond        (♦)   (macro @BDIAMOND)
      L'\x25CF',     // black-circle   (●)
      L'\x26AB',     // med-blk-circle (⚫)
      L'\x23FA',     // black-circle   (⏺)   (macro @BDISC)
      L'\x2219',     // math bullet    (∙)
      L'\x25CB',     // white-circle   (○)
      L'\x26AA',     // med-wh-circle  (⚪)
      L'\x26AC',     // med-sm wh cir  (⚬)   (macro @BCIRCLE)
      L'\x25E6',     // white bullet   (◦)
      L'\x25A0',     // black-square   (■)
      L'\x25FC',     // blk-med-square (◼)
      L'\x25FE',     // blk-med-sm-sq  (◾)
      L'\x25AA',     // blk-sm-square  (▪)   (macro @BSQUARE)
   } ;
   const short    vbdCOUNT = 5 ;
   const wchar_t* ValidBulletDefs[vbdCOUNT] =
   {
      L"&deg;",
      L"&gt;",
      L"&bullet;",
      L"&ndash;",
      L"&mdash;",
   } ;
   const short vbdLen[vbdCOUNT] = 
   { 5, 4, 8, 7, 7 } ;

   wchar_t bullChar ;   // captures embedded bullet character
   short wi = gsitem.after( tagClose, liIndex ) ; // index first char after opening tag

   //* If the stupid comment (see above) is present, delete it.*
   // Programmer's Note: If conditional compile set, then comment is removed globally.
   #if REMOVE_STUPID_COMMENT == 0
   if ( this->no_mods == false )
      gsitem.erase( stupidComment, wi ) ;
   #endif   // REMOVE_STUPID_COMMENT

   //* Step over any whitespace that separates the tag from *
   //* the text and copy the embedded bullet character.     *
   wi = gsitem.scan( wi ) ;
   bullChar = gsitem.gstr()[wi] ;

   //* Validate the bullet character
   short   cIndex = ZERO ;
   while ( cIndex < vbcCOUNT )
   {
      //* If captured bullet character is on of the     *
      //* supported characters, enclose it in monospace *
      //* style and adjust indentation of line item.    *
      if ( bullChar == ValidBulletChars[cIndex] )
      {
         if ( this->liSpec && (bullChar == discBullet) )
         {
            gsitem.erase( ValidBulletChars[cIndex], wi ) ;
            bullChar = discBullet ;
         }
         else if ( this->liSpec && (bullChar == squareBullet) )
         {
            gsitem.erase( ValidBulletChars[cIndex], wi ) ;
            bullChar = squareBullet ;
         }
         else if ( this->liSpec && ((bullChar == degreeSymbol) || 
                   (bullChar == cirtexBullet) || (bullChar == circleBullet)) )
         {
            gsitem.erase( ValidBulletChars[cIndex], wi ) ;
            bullChar = circleBullet ;  // alert caller to set "circle-bullet" class
         }
         else if ( eraseBull )      // forced erase of embedded character
         {
            gsitem.erase( ValidBulletChars[cIndex], wi ) ;
            bullChar = L' ' ;
         }
         else
         {
            if ( cIndex == ZERO )      // replace ASCII minus with Unicode minus
            {
               bullChar = L'\x2212' ;
               gsitem.replace( ValidBulletChars[cIndex], bullChar, wi ) ;
            }
            gsitem.insert( monoStyleEnd, (wi + 1) ) ;
            gsitem.insert(monoStyleBegin, wi ) ;
            gsitem.replace( liBegin, liOutdent, liIndex ) ;
         }
         break ;
      }
      ++cIndex ;
   }  // while()
   if ( cIndex >= vbcCOUNT )  // bullet character not found
   {
      //* Test for one of the defined HTML character constants *
      if ( bullChar == L'&' )
      {
         cIndex = ZERO ;
         while ( cIndex < vbdCOUNT )
         {
            if ( (gsitem.find( ValidBulletDefs[cIndex], wi )) == wi )
            {
               //* If bullet character is "&deg", erase it.         *
               //* Caller will substitute the "circle-bullet" class *
               if ( this->liSpec && cIndex == ZERO )
               {
                  gsitem.erase( ValidBulletDefs[cIndex], wi ) ;
                  bullChar = circleBullet ;
               }
               else if ( eraseBull )      // forced erase of embedded character
               {
                  gsitem.erase( ValidBulletDefs[cIndex], wi ) ;
                  bullChar = L' ' ;
               }
               else
               {
                  gsitem.insert( monoStyleEnd, (wi + vbdLen[cIndex]) ) ;
                  gsitem.insert(monoStyleBegin, wi ) ;
                  gsitem.replace( liBegin, liOutdent, liIndex ) ;
                  // (embedded bullChar identified above)
               }
               break ;
            }
            ++cIndex ;
         }
      }
      if ( cIndex >= vbdCOUNT )  // bullet character not found, set to space char
      {
         bullChar = L' ' ;
      }
   }

   return bullChar ;

}  //* End ppfFormatLineItem() *

//*************************
//*  ppfInsertCustomData  *
//*************************
//******************************************************************************
//* Insert user-supplied data at the current position of the target document.  *
//* (existence of the data file has been previously verified)                  *
//*                                                                            *
//* Input  : fileSpec: path/filename specification of source data              *
//*                                                                            *
//* Returns: OK if successful, ERR processing error                            *
//******************************************************************************

bool Idpp::ppfInsertCustomData ( const gString& fileSpec )
{
   short status = ERR ;
   ifstream dataIn( fileSpec.ustr(), ifstream::in ) ;
   if ( dataIn.is_open() )
   {
      status = OK ;
      gString gsin ;
      while ( (this->ppfReadLine ( dataIn, gsin )) == OK )
         this->ppfWriteLine ( gsin ) ;

      dataIn.close() ;           // close the source file
   }
   return status ;

}  //* End ppfInsertCustomData() *

//*************************
//*    ppfReadSrcLine     *
//*************************
//******************************************************************************
//* Read one line of text from the specified source file,                      *
//* Idpp::slCount is incremented.                                              *
//*                                                                            *
//* Input  : gs     : (by reference) receives text data                        *
//*          windex : (by reference) returned with index of first (wide data)  *
//*                   non-whitespace character on line                         *
//*                                                                            *
//* Returns: OK if successful, ERR processing error                            *
//******************************************************************************
//* Programmer's Note: When pulling a data line from the 'fifo' object, if     *
//* that line contains only a newline character, it indicates that the         *
//* original source line was an empty line, so remove the newline character.   *
//*                                                                            *
//******************************************************************************

short Idpp::ppfReadSrcLine ( gString& gs, short& windex )
{
   char  tmp[gsMAXBYTES] ; // UTF-8 line input from file
   short status = ERR ;    // return value
   windex = ZERO ;         // initialize caller's index

   //* If there are data in the pushback buffer, return it. *
   if ( (this->pushBack->pull( gs )) )
   {
      //* If data is only a newline character (see note above).*
      if ( (gs.gschars() == 2) && (gs.gstr()[ZERO] == L'\n') )
         gs.clear() ;
      windex = gs.scan() ;
      status = OK ;
   }

   else
   {
      this->ifs.getline ( tmp, gsMAXBYTES, NEWLINE ) ; // read a source line
      if ( this->ifs.good() || this->ifs.gcount() > ZERO ) // if a good read
      {
         ++this->slCount ;                         // increment line counter
         gs = tmp ;                                // convert to wide data

         #if REMOVE_STUPID_COMMENT != 0
         if ( this->no_mods == false )
            while ( (gs.erase( stupidComment )) != ERR ) ;
         #endif   // REMOVE_STUPID_COMMENT

         windex = gs.scan() ;                      // ignore leading whitespace
         status = OK ;                             // return with good news
   
         //* For debugging output (super verbose) *
         if ( this->scan != false 
              && (this->slCount >= this->scan_beg && 
                  this->slCount <= this->scan_end) )
         {
            gString gss( "SCAN (s:%4hu t:%4hu) '%S'", 
                         &this->slCount, &this->tlCount, gs.gstr() ) ;
            this->textOut ( gss ) ;
         }
      }
   }
   return status ;

}  //* End ppfReadSrcLine() *

//*************************
//*   ppfUnReadSrcLine    *
//*************************
//******************************************************************************
//* Caller has read data from the source input stream that it cannot process.  *
//* Push it back into the source input stream so that it will be available     *
//* for processing at a higher level.                                          *
//*                                                                            *
//* Input  : gs   : data to be written back into the source stream             *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************
//* Programmer's Note: If caller is pushing an empty line back into the stream,*
//* it means that the original source line contained only a newline.           *
//* Unfortunately, when pulling the line from the queue, the 'fifo' object     *
//* will interpret the empty line as no data.                                  *
//*                                                                            *
//******************************************************************************

void Idpp::ppfUnReadSrcLine ( const gString& gs )
{
   if ( gs.gschars() == 1 )      // if an empty line (see note above)
   {
      gString gstmp( L"\n" ) ;
      this->pushBack->push( gstmp ) ;
   }
   else
      this->pushBack->push( gs ) ;

}  //* End UnReadSrcLine() *

//*************************
//*      ppfReadLine      *
//*************************
//******************************************************************************
//* Read one line of text from the specified input stream.                     *
//* This is a generic ReadLine. See ppfReadSrcLine() for specialized HTML      *
//* source file read.                                                          *
//*                                                                            *
//* Input  : ifs    : open input stream (by reference)                         *
//*          gs     : (by reference) receives text data                        *
//*                                                                            *
//* Returns: OK if successful, ERR processing error                            *
//******************************************************************************

short Idpp::ppfReadLine ( ifstream& ifs, gString& gs )
{
   char  tmp[gsMAXBYTES] ; // UTF-8 line input from file
   short status = ERR ;    // return value

   ifs.getline ( tmp, gsMAXBYTES, NEWLINE ) ;   // read a source line
   if ( ifs.good() || ifs.gcount() > ZERO )     // if a good read
   {
      gs = tmp ;                                // convert to wide data
      status = OK ;                             // return with good news
   }
   return status ;

}  //* End ppfReadLine() *

//*************************
//*     ppfWriteLine      *
//*************************
//******************************************************************************
//* Write a line of data to the target file. the line will be terminated with  *
//* a newline character ('\n'), and the output buffer is flushed.              *
//* Idpp::tlCount is incremented.                                              *
//*                                                                            *
//* Input  : gsOut : (by reference) text to be written.                        *
//*                                                                            *
//* Returns: OK if successful, or ERR if write error.                          *
//*            Note: currently, we don't check for output stream errors.       *
//******************************************************************************

short Idpp::ppfWriteLine ( const gString& gsOut )
{
   short status = OK ;

   if ( this->no_mods == false )
   {
      this->ofs << gsOut.ustr() << endl ;
   }
   ++this->tlCount ;    // increment the output-line counter
   return status ;

}  //* End ppfWriteLine() *

//*************************
//*      ppfScan4Src      *
//*************************
//******************************************************************************
//* Scan the target directory for HTML source documents.                       *
//* Files are selected by filename extension.                                  *
//* Valid extensions:  .html  .htm                                             *
//*                                                                            *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: 'true' if at least one valid file found                           *
//**         'false' if: a) no HTML files found                                *
//*                      b) directory read error                               *
//******************************************************************************

bool Idpp::ppfScan4Src ( void )
{
   const wchar_t* extA = L".html" ;    const short extA_len = 6 ;
   const wchar_t* extB = L".htm" ;     const short extB_len = 5 ;
   const wchar_t* extC = L".shtml" ;   const short extC_len = 7 ;
   const wchar_t* extD = L".shtm" ;    const short extD_len = 6 ;
   const wchar_t* extE = L".xhtml" ;   const short extE_len = 7 ;

   DIR*      dirPtr ;            // pointer to open directory file
   deStats   *destat ;           // pointer to entry read from directory file
   gString   fnPath,             // source file's path/filename
             fnName,             // source file's filename
             fnExt ;             // source file's filename extension
   FileStats rawStats ;          // source file's stat info
   bool      status = false ;    // return value

   //* Discard any filenames specified directly on *
   //* command line. This avoids duplicates.       *
   this->sfCount = ZERO ;

   if ( ((this->ppfTargetExists ( this->cwDir, true )) != false) &&
        ((dirPtr = opendir ( this->cwDir.ustr() )) != NULL) )
   {
      while ( (destat = readdir64 ( dirPtr )) != NULL )
      {
         //* '.' and '..' are not included, but hidden files are *
         if ( (destat->d_name[ZERO] == PERIOD)
              &&
              ((destat->d_name[1] == NULLCHAR) || 
               (destat->d_name[1] == PERIOD && destat->d_name[2] == NULLCHAR)) )
         {
            continue ;
         }

         //* 'lstat' the file to determine its type 'regular' files only *
         fnName = destat->d_name ;
         this->ppfCatPathFilename ( fnPath, this->cwDir, fnName.ustr() ) ;
         if ( (lstat64 ( fnPath.ustr(), &rawStats )) == OK )
         {
            if ( (S_ISREG(rawStats.st_mode)) != false )
            {
               this->ppfExtractFileExtension ( fnExt, fnPath.ustr() ) ;
               if ( ((fnExt.gstr()[fnExt.gschars() - 1]) != L'~') && 
                    (   (fnExt.compare( extA, true, extA_len ) == ZERO)
                     || (fnExt.compare( extB, true, extB_len ) == ZERO)
                     || (fnExt.compare( extC, true, extC_len ) == ZERO)
                     || (fnExt.compare( extD, true, extD_len ) == ZERO)
                     || (fnExt.compare( extE, true, extE_len ) == ZERO))
                  )
               {
                  fnName.copy( this->srcFiles[this->sfCount++], gsMAXCHARS ) ;
                  status = true ;   // at least one valid file found
               }
            }
            else { /* Can't access the file, so ignore it */ }
         }
      }  // while()
      closedir ( dirPtr ) ;               // close the directory
   }     // opendir()
   return status ;

}  //* End ppfScan4Src()

//*************************
//*       ppfGetCWD       *
//*************************
//******************************************************************************
//* Returns the path of user's current working directory.                      *
//*                                                                            *
//* Input  : dPath:(by reference, initial value ignored)                       *
//*                receives path string                                        *
//*                                                                            *
//* Returns: OK if successful, ERR if path too long to fit the buffer          *
//******************************************************************************

short Idpp::ppfGetCWD ( gString& dPath )
{
char     buff[MAX_PATH] ;
short    status = OK ;

   if ( (getcwd ( buff, MAX_PATH )) != NULL )
      dPath = buff ;
   else
      status = ERR ;    // (this is very unlikely)

   return status ;

}  //* End ppfGetCWD() *

//*************************
//*    ppfTargetExists    *
//*************************
//******************************************************************************
//* 'stat' the specified path/filename to see if it exists.                    *
//* Target must be a 'regular' file or optionally, a 'directory' file.         *
//*                                                                            *
//* We also do a simple test for read access: if owner, group, or other has    *
//* read access, we assume that user has read access although this may not be  *
//* true in all cases.                                                         *
//*                                                                            *
//* Input  : fPath: target path/filename                                       *
//*          isDir: (optional, false by default)                               *
//*                 if 'true', test whether file type is 'regular'             *
//*                 if 'false', test whether file type is 'directory'          *
//*                                                                            *
//* Returns: 'true' if file exists AND is a Regular file, or optionally, a     *
//*                 'directory' file                                           *
//*          'false' otherwise                                                 *
//******************************************************************************

bool Idpp::ppfTargetExists ( const gString& fPath, bool isDir )
{
   bool exists = false ;
   FileStats rawStats ;

   //* If the file exists *
   if ( (lstat64 ( fPath.ustr(), &rawStats )) == OK )
   {  //* Simple, read-access test (see note above) *
      if ( (rawStats.st_mode & S_IRUSR) || (rawStats.st_mode & S_IRGRP) || 
           (rawStats.st_mode & S_IROTH) )
      {
         //* If testing for the existence of a directory *
         if ( isDir != false )
         {
            if ( (S_ISDIR(rawStats.st_mode)) != false )
               exists = true ;
         }
         //* We operate only on 'regular' files *
         else
         {
            if ( (S_ISREG(rawStats.st_mode)) != false )
               exists = true ;
         }
      }
   }
   return exists ;

}  //* End ppfTargetExists() *

//*************************
//*    ppfTargetIsHTML    *
//*************************
//******************************************************************************
//* Test the first line of specified file.                                     *
//* A valid HTML (v4 or v5) document must begin with "<!DOCTYPE HTML".         *
//* A valid HTML (v3x) MAY begin with "<HTML".                                 *
//* If neither, then we assume that the document is not valid HTML.            *
//*                                                                            *
//* Input  : fPath: target path/filename                                       *
//*                                                                            *
//* Returns: 'true' if file begins with <!DOCTYPE HTML" OR "<HTML"             *
//*                 (comparison IS NOT case sensitive)                         *
//*          'false' otherwise                                                 *
//******************************************************************************

bool Idpp::ppfTargetIsHTML ( const gString& fPath )
{
   bool status = false ;
   ifstream ifs ( fPath.ustr(), ifstream::in ) ;
   if ( ifs.is_open() )             // if input file open
   {
      //* Read the first source line *
      gString gs ;
      if ( (status = this->ppfReadLine ( ifs, gs )) == OK )
      {
         if ( ((gs.compare( topDOCT, false, topDOCT_len )) == ZERO)
              ||
              ((gs.compare( topHTML, false, topHTML_len )) == ZERO) )
            status = true ;
      }
      ifs.close() ;                 // close the file
   }
   return status ;

}  //* End ppfTargetIsHTML() *

//*************************
//*    ppfTargetIsCSS     *
//*************************
//******************************************************************************
//* Test whether the specified CSS definition file is valid.                   *
//* a) Verify that the copyright notice is intact.                             *
//* b) Extract the version number string.                                      *
//* If version string not found, then we assume that the document is           *
//* incorrectly formatted      *
//*                                                                            *
//* Input  : fPath: target path/filename                                       *
//*          gsVer: receives version number string                             *
//*                                                                            *
//* Returns: 'true' if file format verified                                    *
//*          'false' otherwise                                                 *
//******************************************************************************

bool Idpp::ppfTargetIsCSS ( const gString& fPath, gString& gsVer )
{
   bool status = false ;
   ifstream ifs ( fPath.ustr(), ifstream::in ) ;
   if ( ifs.is_open() )             // if input file open
   {
      gString gs ;
      short i ;
      bool  cright_found = false, done = false ;

      while ( !done )
      {
         //* Read a source line *
         if ( (status = this->ppfReadLine ( ifs, gs )) == OK )
         {
            if ( !cright_found )
            {
               for ( i = ZERO ; gs.gstr()[i] != *cssCOPY && gs.gstr()[i] != NULLCHAR ; i++ ) ;
               if ( (gs.gstr()[i] == *cssCOPY) &&
                       ((gs.compare( cssCOPY, true, cssCOPY_len, i )) == ZERO) )
               {
                  gs.shiftChars( (ZERO - i) ) ;
                  for ( i = ZERO ; gs.gstr()[i] != *cssSAMU && gs.gstr()[i] != NULLCHAR ; i++ ) ;
                  if ( (gs.gstr()[i] == *cssSAMU) && 
                       ((gs.compare( cssSAMU, true, cssSAMU_len, i )) == ZERO) )
                  {
                     cright_found = true ;
                  }
               }
            }
            else
            {
               for ( i = ZERO ; gs.gstr()[i] != *cssVERS && gs.gstr()[i] != NULLCHAR ; i++ ) ;
               if ( (gs.gstr()[i] == *cssVERS) &&
                       ((gs.compare( cssVERS, true, cssVERS_len, i )) == ZERO) )
               {
                  gs.shiftChars( (ZERO - i) ) ;
                  gs.limitChars( CSS_VER_LEN ) ;
                  gsVer = gs ;
                  done = status = true ;
               }
            }
         }
         else
            done = true ;
      }
      ifs.close() ;                 // close the file
   }
   return status ;

}  //* End ppfTargetIsCSS() *

//*************************
//*  ppfCatPathFilename   *
//*************************
//******************************************************************************
//* Concatenate a path string with a filename string to create a path/filename *
//* specification.                                                             *
//*                                                                            *
//* Input  : pgs  : gString object (by reference) to hold the path/filename    *
//*                 On return, pgs contains the path/filename string in both   *
//*                 UTF-8 and wchar_t formats.                                 *
//*          wPath: gString object (by reference) path string                  *
//*          uFile: pointer to UTF-8 filename string                           *
//*               OR                                                           *
//*          wFile: pointer to wchar_t filename string                         *
//*                                                                            *
//* Returns: OK if success                                                     *
//*          ERR if string truncated                                           *
//******************************************************************************

short Idpp::ppfCatPathFilename ( gString& pgs, const gString& wPath, const char* uFile )
{
short    success = OK ;

   pgs.compose( L"%S/%s", wPath.gstr(), uFile ) ;
   if ( pgs.gschars() >= (gsMAXCHARS-1) )
      success = ERR ;   // (this is very unlikely)
   return success ;

}  //* End ppfCatPathFilename() *

short Idpp::ppfCatPathFilename ( gString& pgs, const gString& wPath, const wchar_t* wFile )
{
short    success = OK ;

   pgs.compose( L"%S/%S", wPath.gstr(), wFile ) ;
   if ( pgs.gschars() >= (gsMAXCHARS-1) )
      success = ERR ;   // (this is very unlikely)
   return success ;

}  //* End ppfCatPathFilename() *

//*************************
//*     ppfCdTarget      *
//*************************
//******************************************************************************
//* Change the current working directory to specified target path.             *
//*                                                                            *
//* Input  : dirPath: path of new CWD (relative or absolute)                   *
//*                                                                            *
//* Returns: OK if successful, else ERR                                        *
//******************************************************************************

short Idpp::ppfCdTarget ( const char* dirPath )
{
   short status ;       // return value

   //* If a relative path, convert to absolute path *
   char tmpPath[gsMAXBYTES] ;
   realpath ( dirPath, tmpPath ) ;
   gString targPath( tmpPath ) ;

   //* Change our current working directory and update CWD data member *
   if ( (status = chdir ( targPath.ustr() )) == OK )
      this->ppfGetCWD ( this->cwDir ) ;
   return status ;

}  //* End ppfCdTarget() *

//*************************
//*    ppfDeleteFile      *
//*************************
//******************************************************************************
//* Delete the specified file.                                                 *
//*                                                                            *
//* Input  : trgPath : full path/filename specification of target              *
//*                                                                            *
//* Returns: OK if successful, else ERR                                        *
//******************************************************************************

short Idpp::ppfDeleteFile ( const gString& trgPath )
{

   return ( unlink ( trgPath.ustr() ) ) ;

}  //* End ppfDeleteFile() *

//*************************
//*    ppfRenameFile      *
//*************************
//******************************************************************************
//* Rename the specified file.                                                 *
//*                                                                            *
//* Input  : srcPath : full path/filename specification of source              *
//*          trgPath : full path/filename specification of target              *
//*                                                                            *
//* Returns: OK if successful, else ERR                                        *
//******************************************************************************
//* For the rename() primitive: If srcPath refers to a symbolic link the       *
//* link is renamed; if trgPath refers to a symbolic link the link will be     *
//* overwritten.                                                               *
//******************************************************************************

short Idpp::ppfRenameFile ( const gString& srcPath, const gString& trgPath )
{

   return ( rename ( srcPath.ustr(), trgPath.ustr() ) ) ;

}  //* End ppfRenameFile() *

//*************************
//*     ppfRealpath       *
//*************************
//******************************************************************************
//* Decode the relative or aliased path for the specified target.              *
//*                                                                            *
//*                                                                            *
//* Input  : realPath : (by reference) receives decoded path specification     *
//*          rawPath  : (by reference) caller's path specification             *
//*                                                                            *
//* Returns: OK  if successful                                                 *
//*          ERR if invalid path or if system call fails ('realPath' unchanged)*
//******************************************************************************
//* Programmer's Note: The 'realpath' library function cannot handle a         *
//* symboloc substitution such as: ${HOME}/Docs/filename.txt                   *
//* It returns: /home/Sam/Docs/(CWD)/${HOME}                                   *
//* This is a little bit nuts, but we don't have time to investigate it.       *
//*                                                                            *
//******************************************************************************

short Idpp::ppfRealpath ( gString& realPath, const gString& rawPath )
{
   short status = ERR ;

   const char* rpath ;
   if ( (rpath = realpath ( rawPath.ustr(), NULL )) != NULL )
   {
      realPath = rpath ;
      free ( (void*)rpath ) ;    // release the temp storage
      status = OK ;
   }
   return status ;

}  //* End ppfRealpath() *

//*************************
//*  ppfExtractFilename   *
//*************************
//******************************************************************************
//* Extract the filename from the path/filename provided.                      *
//*                                                                            *
//* Input  : gsName: (by reference, initial contents ignored)                  *
//*                  receives the extracted filename                           *
//*          fPath : path/filename string                                      *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void Idpp::ppfExtractFilename ( gString& gsName, const gString& fPath )
{
   const wchar_t* fptr = fPath.gstr() ;
   short findex = fPath.gschars() - 1 ;
   while ( findex > ZERO && fptr[findex] != SLASH )
      --findex ;
   if ( fptr[findex] == SLASH )
      ++findex ;
   gsName = &fptr[findex] ;

}  //* End ppfExtractFilename() *

//***************************
//* ppfExtractFileExtension *
//***************************
//******************************************************************************
//* Extract the filename extension (including the '.') from the path/filename  *
//* provided.                                                                  *
//*                                                                            *
//* Input  : gsName: (by reference, initial contents ignored)                  *
//*                  receives the extracted filename                           *
//*          fName : filename or path/filename string                          *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void Idpp::ppfExtractFileExtension ( gString& gsExt, const char* fName )
{
   wchar_t tmpName[gsMAXCHARS] ;    // convert to wide character string
   gsExt = fName ;
   gsExt.copy( tmpName, gsMAXCHARS ) ;

   short   tpIndex = gsExt.gschars() - 1 ; // index the NULLCHAR
   while ( tpIndex > ZERO && tmpName[tpIndex] != PERIOD ) // isolate the extension
      --tpIndex ;
   if ( tmpName[tpIndex] == PERIOD )
      gsExt = &tmpName[tpIndex] ;
   else
      gsExt = L"" ;

}  //* End ppfExtractFileExtension() *

//*************************
//*                       *
//*************************
//******************************************************************************
//* Non-member method                                                          *
//* -----------------                                                          *
//*                                                                            *
//* Remove all HTML tags from the data.                                        *
//* Called by ppfProcUL() and ppfProcOL() to create first-line display text    *
//* for the user prompt.                                                       *
//*                                                                            *
//* Input  : disp : (by reference) data to be scanned                          *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

static void StripTags ( gString& disp )
{
   short bi, ei ;
   while ( (bi = disp.find( L'<' )) >= ZERO )
   {
      if ( (ei = disp.after( L'>', false, bi )) > bi )
         disp.erase( bi, (ei - bi) ) ;
   }
}  //* End StripTags() *

//*************************
//*                       *
//*************************
//******************************************************************************
//*                                                                            *
//*                                                                            *
//*                                                                            *
//* Input  :                                                                   *
//*                                                                            *
//* Returns:                                                                   *
//******************************************************************************

