//******************************************************************************
//* File       : idpp.cpp                                                      *
//* Author     : Mahlon R. Smith                                               *
//*              Copyright (c) 2014-2020 Mahlon R. Smith, The Software Samurai *
//*                 GNU GPL copyright notice below                             *
//* Date       : 14-Aug-2020                                                   *
//* Version    : (see AppVersion string)                                       *
//*                                                                            *
//* Description: Definitions and data for Infodoc Post-processor (idpp),       *
//* an HTML post-processing utility for use with HTML documents generated from *
//* Texinfo source.                                                            *
//*                                                                            *
//* For full documentation for idpp is located in 'infodoc.info' :             *
//*             info -f infodoc.info -n 'Infodoc Post-processor'               *
//*                                                                            *
//*                                                                            *
//* Copyright Notice:                                                          *
//* -----------------                                                          *
//* This program is free software: you can redistribute it and/or modify it    *
//* under the terms of the GNU General Public License as published by the Free *
//* Software Foundation, either version 3 of the License, or (at your option)  *
//* any later version, PROVIDED THAT the copyright notices for both code and   *
//* documentation are included and unmodified.                                 *
//*                                                                            *
//* This program is distributed in the hope that it will be useful, but        *
//* WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY *
//* or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License   *
//* for more details.                                                          *
//*                                                                            *
//* You should have received a copy of the GNU General Public License along    *
//* with this program.  If not, see <http://www.gnu.org/licenses/>.            *
//*                                                                            *
//*         Full text of the GPL License may be found in the TexInfo           *
//*         documentation for this program under 'Copyright Notice'.           *
//******************************************************************************
//* Version History (most recent first):                                       *
//*                                                                            *
//* v: 0.0.13 01-Aug-2020                                                      *
//*   -- Modifications to post-processing for a second or subsequent pass on   *
//*      the same file. While a second pass on a previously-processed HTML     *
//*      file is allowed, our goal is to do as little _unintentional_          *
//*      modification as practical during the second pass.                     *
//*      For this reason, we have made the following adjustments:              *
//*      -- If a second pass is performed with automatic processing:           *
//*         -- Bug Fix: In ppfProcGNU(), two redundant "<br>" tokens were      *
//*            being inserted after the version-number declaration.            *
//*         -- Bug Fix: When pushing an empty source line into the FIFO object,*
//*            the corresponding pull of that line from the FIFO object was    *
//*            seeing no data. Refer to ppfUnReadSrcLine() and ppfReadSrcLine()*
//*            for details.                                                    *
//*         -- Bug Fix: In ppfProcOL(), if the source file contained a         *
//*            descending <ol> list, (example: 10, 9, 8, ...) the "reversed"   *
//*            token was being discarded.                                      *
//*         -- Bug Fix: The table-border prompts were being skipped on the     *
//*            second pass because there was no test for previously-processed  *
//*            tables.                                                         *
//*      -- If a second pass is performed with interactive processing via a    *
//*         response file:                                                     *
//*         -- The "no-mods" flag is set to prevent all modifications during   *
//*            the second pass. This is necessary to avoid unintended responses*
//*            due to the response file tokens very likely becomming           *
//*            out-of-synch with the actual promots.                           *
//*            Note: The user can override the "no_mods" flag (for the first   *
//*            source file in the list only) by invoking the application with  *
//*            the '-V' (verify) flag.                                         *
//*   -- Bug Fix: When scanning for <ul> and <ol> tags, there was a hidden     *
//*      bug which accidentally parsed the tag correctly in all known cases.   *
//*      It now does the same parsing intentionally, eliminating the potential *
//*      error.                                                                *
//*   -- Documentation update.                                                 *
//*   -- Posted to website 14-Aug-2020                                         *
//*                                                                            *
//* v: 0.0.12 16-Dec-2019                                                      *
//*   -- Bug Fix: If formatted text blocks are present between the Table Of    *
//*      Contents and the top-level menu, they were not always recognized      *
//*      and processed. Now when processing the TOC, we return as soon as the  *
//*      first page header is processed, so everything below the top page      *
//*      header will be processed in the main loop.                            *
//*   -- Enhance user prompt for UL and OL lists by displaying cleaner         *
//*      "first line item" text.                                               *
//*   -- Cartouche objects are now recognized within UL and OL lists so long   *
//*      as it is preceded by a blank line.                                    *
//*   -- Documentation update.                                                 *
//*   -- Posted to website 03-Jul-2020                                         *
//*                                                                            *
//* v: 0.0.11 12-Dec-2019                                                      *
//*   -- Bug Fix: Application exit code was sometimes not being set correctly. *
//*      Application should exit with either OK ( 0 ) or ERR ( -1 ).           *
//*   -- Enhance processing of the document chapters which contain the         *
//*      GNU General Public License and the Free Documentation License.        *
//*      As stated elsewhere (ppfProcGNU() method), these are legal documents  *
//*      and therefore should resemble as closely as possible the lawyers'     *
//*      intended format.                                                      *
//*   -- Documentation update.                                                 *
//*   -- Posted to website 15-Dec-2019                                         *
//*                                                                            *
//* v: 0.0.10 26-Sep-2019                                                      *
//*   -- Skip v:0.0.06, v:0.0.07, 0.0.08 and v:0.0.09 to synchronize version   *
//*      number with the Infodoc CSS definitions which have been updated       *
//*      several times since the last Idpp release.                            *
//*   -- Transition from gcc 4.8.3 to gcc 9.2.1 required a change from         *
//*      readdir64_r() which is deprecated. Switch to readdir64() which is     *
//*      now thread-safe under the GNU compiler.                               *
//*   -- Updates to address changes in makeinfo between v:5.5  and  v:6.6.     *
//*      -- A large number of HTML constructs generated by makeinfo conversion *
//*         have been modified, causing idpp to incorrectly parse these        *
//*         constructs. For this reason it is necessary to freeze support for  *
//*         makeinfo v:5.x at idpp v:0.0.05. IDPP v:0.0.10 and forward will    *
//*         support makeinfo v:6.x and higher.                                 *
//*      -- Minor changes is header processing.                                *
//*      -- Minor changes in Table Of Contents processing.                     *
//*      -- Significant enhancements in texi2any in creation of ordered lists  *
//*         in HTML output, allow for much smoother and more detailed          *
//*         post-processing of <ol> lists.                                     *
//*      -- The HTML for itemized lists (@itemize <ul>) is mostly unchanged,   *
//*         however we have enhanced the post-processing of these lists to     *
//*         support more types of bullets and to beautify the output.          *
//*         This includes enhanced automatic post-processing, and also an      *
//*         interactive mode for processing <ul> lists to optionally assign a  *
//*         bullet class to any list which does not include class information. *
//*      -- texi2any v:6.6 Bug: According to the ChangeLog 2019-02-10 and the  *
//*         v:6.6 documentation, Gavin has removed support for smaller font in *
//*         all @small... commands in HTML and now processes them as the       *
//*         standard-size versions. "@small..." blocks are now rendered as     *
//*         identical to their standard versions.                              *
//*         -- To compensate for this foolishness, we now offer interactive    *
//*            selection of font size for all the various block constructs:    *
//*            inherited/smaller/larger. More work for the user, but also more *
//*            flexibility.                                                    *
//*         -- We also support standard/smaller/larger versions of @verbatim   *
//*            blocks which have never been supported in makeinfo.             *
//*      -- The HTML markup generated for @indentedblock blocks has changed    *
//*            From: '<div class="indentedblock">'                             *
//*              To: '<blockquote class="indentedblock">'                      *
//*         This is actually a good change, but it requires a corresponding    *
//*         change in the way we scan for indentedblock sequences because there*
//*         is now some ambiguity between indentedblock and quotation markup.  *
//*         As noted above, the "@small..." versions of blocks are no longer   *
//*         generated by texi2any, but we do post-processing in a              *
//*         symetrical way: '<blockquote class="smallindentedblock">'          *
//*                         '<blockquote class="largeindentedblock">'          *
//*      -- HTML markup for quotation blocks (@quotation/@smallquotation)      *
//*         remains relatively unchanged, but we must now distinguish between: *
//*             <blockquote>                        @quotation                 *
//*             <blockquote class="indentedblock">  @indentedblock             *
//*         Although the @smallquotation command is no longer recognized by    *
//*         we do provide standard/smaller/larger options.                     *
//*   -- Post-processing of @multitable blocks and other tables including      *
//*      the @cartouche has been redefined. The 'tabPrompt' member has been    *
//*      removed, and the 'tabBorder' member has been redefined to allow       *
//*      for more flexible processing options.                                 *
//*   -- Enhanced CSS formatting for @cartouche blocks (class=cartouche).      *
//*      The "--no_cartouche" option which allowed the genenerated border      *
//*      to be retained has been redefined to mean that text within a          *
//*      cartouche block is allowed to "flow". Cartouche text is               *
//*      pre-formatted by default in the HTML document.                        *
//*   -- Enhance beautification of lists inside preformatte blocks, ALTHOUGH   *
//*      lists should never be inside @format, @display, @example, @lisp       *
//*      blocks. To disable this beautification invoke with the                *
//*      "--no_special" command-line option.                                   *
//*   -- A file containing responses to prompts can be redirected to stdin,    *
//*      but if the number of responses is less than the number of prompts,    *
//*      the program would not finish because user could not regain control    *
//*      of stdin. We have therefore added support for opening and reading     *
//*      the response file directly.                                           *
//*   -- Fully document the "--scan" and "--book" options. Formerly, the       *
//*      documentation mentioned them but referred the user to the source code.*
//*   -- Documentation update.                                                 *
//*   -- Posted to website 05-Dec-2019                                         *
//*                                                                            *
//* v: 0.0.05 24-Feb-2015                                                      *
//*   - Update calls to 'formatInt', 'compare' and 'limitChars' methods of the *
//*     gString class to match changes to gString prototypes (no functionality *
//*     changed).                                                              *
//*                                                                            *
//* v: 0.0.04 20-Feb-2015                                                      *
//*   - Create a more robust test for end-of-block markers. This reduces the   *
//*     chance that really strange constructs will hide the end-of-block tag,  *
//*     causing a processing error. See ppfEndBlock().                         *
//*   - Create a more robust test for the beginning of <ul> and <ol> lists.    *
//*     Occasional strange constructs were sometimes causing the opening tag   *
//*     to be missed, and therefore the list to be unprocessed.                *
//*     See ppfTestUlistBegin() and ppfTestOlistBegin().                       *
//*   - Bug fix: In ppfProcFormattedBlock(), if source contained one           *
//*     preformatted block nested inside another preformatted block, then      *
//*     processing of the block terminated too soon.                           *
//*   - Lists inside pre-formatted blocks. In the previous release we did not  *
//*     process anything inside a preformatted block. In fact, lists           *
//*     SHOULD NOT be placed inside preformatted blocks, but if they are, we   *
//*     now identify them. Note, however, that we DO NOT process the lists     *
//*     themselves (that would require a second pass through the source HTML), *
//*     but we do compensate for the unfortunate spacing in the generated HTML.*
//*     See ppfProcFormattedBlock() and ppfPFB_List().                         *
//*                                                                            *
//* v: 0.0.03 02-Feb-2015                                                      *
//*   - Bug fix: If <ul> or <ol> list ends with a block construct, then the    *
//*     </ul> or </ol> tag was sometimes being missed.                         *
//*   - Bug fix: If first chapter header is not for 'index-main-menu', then    *
//*     end of TOC was not being found.                                        *
//*   - Bug fix: In ppfProcUL(), if user specified that <ul> lists should not  *
//*     be processed, the disable-processing flag was not being set.           *
//*   - Remove the methods ppfCopyUL() and ppfCopyOL(). These made sense when  *
//*     the algorithm for list processing was unstable, but are no longer      *
//*     necessary.                                                             *
//*   - Allow for a '0' (zero) value in response to the <ol> list processing   *
//*     start value prompt IF the specified enumeration type == 'd' (decimal). *
//*     This is to accomodate the fact that the text of both the GPL and FDL   *
//*     licenses begin with item '0'. Note that the prompt still calls for a   *
//*     value >= 1.                                                            *
//*   - Add a special test for GPL and FDL license text. Process the enumerated*
//*     lists associated with these licenses (unless the '--no_mods' option    *
//*     has been specified). See 'ppfProcGNU' method for details.              *
//*   - Identify and process tables that are nested inside 'indentedblock'     *
//*     blocks.                                                                *
//*                                                                            *
//* v: 0.0.02 01-Feb-2015                                                      *
//*   - First public release.                                                  *
//*   - All documented options are now fully functional except '--css_mods'.   *
//*   - All issues reported by beta-testers through 28 Jan 2015 have been      *
//*     addressed.                                                             *
//*   - All basic functionality seems to be working; but there are a number of *
//*     issues related to Texinfo configuration options which have not yet     *
//*     been investigated.                                                     *
//*   - Channel all user input and display output through a single point for   *
//*     ease in future modifications.                                          *
//*   - Add the '--scan' and '--book' options for debugging only. These provide*
//*     super-verbose output with line numbers to more easily find mis-handled *
//*     HTML source constructs. (tip-of-the-hat to Xiaoxiao on this one :)     *
//*   - Texinfo documentation is relatively complete.                          *
//*   - Building with static libraries has not yet been tested.                *
//*                                                                            *
//* v: 0.0.01 06-Dec-2014                                                      *
//*   - Experimental only. Application structure based on SourceProfiler.      *
//*   - All NcDialog/ncurses code has been temporarily disabled. This will     *
//*     be reinstated when (if) we implement Interactive Mode.                 *
//*                                                                            *
//******************************************************************************
//* Programmer's Notes:                                                        *
//* - The raw HTML mark-up generated by the 'makeinfo' utility is actually     *
//*   pretty good; however, a few things are clunky and backward-looking,      *
//*   while a few things in the displayed output really do look second-rate.   *
//*                                                                            *
//* - There are a number of Texinfo build options that affect the HTML output. *
//*   The use of some of them may cause the early releases of idpp to report   *
//*   parsing errors or to incorrectly parse and process the data.             *
//*   These build options will be investigated as time permits.                *
//*                                                                            *
//*                                                                            *
//*                                                                            *
//* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -  *
//* =================                                                          *
//* == To-do List: ==                                                          *
//* =================                                                          *
//*                                                                            *
//* Notes from texi2any release v:6.7::                                        *
//* texi2any                                                                   *
//*  . for HTML output, mark index nodes in menus and tables of contents       *
//*    with the 'rel' attribute of the 'a' tag.                                *
//*  . TOP_NODE_UP is now only used in HTML if TOP_NODE_UP_URL is set.         *
//*    Also TOP_NODE_UP should now be formatted in the output format.          *
//*    In HTML TOP_NODE_UP should be suitable for inclusion in HTML            *
//*    element attributes, so for instance should not contain elements.        *
//*  . support of noderename.cnf files has been removed                        *
//*  . INPUT_PERL_ENCODING, INPUT_ENCODING_NAME, NODE_FILE_EXTENSION,          *
//*    NODE_FILENAMES, SHORTEXTN and TOP_NODE_FILE removed as customization    *
//*    variables.                                                              *
//*  . TOP_NODE_FILE_TARGET now contains the extension.                        *
//*  . error messages translated when the XS parser module is in use           *
//* Need to verify that if the TOP_NODE_UP construct is not found, that it is  *
//* handled gracefully. By default, we redirect this to the top of the page.   *
//* User can also specify a target address, so if the entry is not found when  *
//* user wants to change it, we need to post a warning.                        *
//*                                                                            *
//*                                                                            *
//* -- Test second-pass scenarios for robustness.                              *
//*    Recent update has broken the second-pass format checks.                 *
//*    -- If all the auto flags are set, then the second-pass results are      *
//*       identical (or nearly so).                                            *
//*    -- If interactive AND all responses are 'a' auto, then the second-pass  *
//*       results are identical (or nearly so).                                *
//*    -- In the boilerplate copyright notices, GPL and FDL, an extra:         *
//*       "<br><br>" is added to the version line.                             *
//*       -- We now count the number of existing <br> tokens.                  *
//*    -- If <ol> list in source is reversed, the reverse token is lost.       *
//*       <ol class="enum-decimal-zero" start="10" reversed>  becomes          *
//*       <ol class="enum-decimal-zero" start="10">                            *
//*                                                                            *
//* Remaining non-identical lines:                                             *
//* ------------------------------                                             *
//* FileA: infodoc_css.html~   FileB: infodoc_css.html                         *
//* <blockquote class="indentedblock">                                         *
//*                 [L:1799]   <blockquote class="largeindentedblock">         *
//* <div class="example">&lt;ol&gt;
//*                 [L:3179]   <div class="largeexample">&lt;ol&gt;
//* <ul class="no-bullet">
//* [L:4120,4232,4917,4928,4944] <ul class="square-bullet">
//* </p><div class="format">&lt;dl compact=&quot;compact&quot;&gt;
//*                 [L:6266]   </p><div class="smallformat">&lt;dl compact=&quot;compact&quot;&gt;
//* </p><div class="example">disc bullets    become  circle bullets
//*                 [L:6837]   </p><div class="largeexample">disc bullets    become  circle bullets
//*   </p><ul>
//*                 [L:6854]   </p><ul class="disc-bullet">
//*                                                                            *
//* -- Document the --scan and --book options in the quick-help list           *
//*    -- Update the documentation                                             *
//*                                                                            *
//* -- Multi-line comments will seldom be seen, but if they are, then second   *
//*    and subsequent lines may be parsed in an ugly way.                      *
//*                                                                            *
//*                                                                            *
//*                                                                            *
//*                                                                            *
//*                                                                            *
//*                                                                            *
//* Busted V:6.1                                                               *
//* -- Update command-line help                                                *
//*                                                                            *
//* --command-line: --css_mods (modify the CSS definition file)                *
//*                   Overrides all other options except --help and --version  *
//*                   - Add a method to modify the whole-document definitions  *
//*                     in the CSS file. Uses the '-f' option for the target   *
//*                     filename if specified, else the default definition     *
//*                     file 'infodoc-styles.css'.                             *
//*                     - background color                                     *
//*                     - foreground color                                     *
//*                     - base font size                                       *
//*                     - resize width of infodoc_container class              *
//*                     - enable/disable container border                      *
//* - command-line: -L Insert a Link (Back-to-Doc-Page) at top and bottom of   *
//*    container. This might or might not be generalized for inserting any     *
//*    object (image?) at top and/or bottom of visible document.               *
//*                                                                            *
//* - Pre-release Texinfo 5.9.90 modifies the way TOP_NODE_UP_URL works.       *
//*   Instead of searching for '(dir)' we may need to search also for something*
//*   like '/manual/'.                                                         *
//*                                                                            *
//* - Pre-release Texinfo 5.9.90 also introduces two new commands:             *
//*                      @sub and @sup                                         *
//*   These are for subscripts and superscripts respectively. Need to see how  *
//*   the texi-to-HTML converter renders these.                                *
//*                                                                            *
//* - Research Gavin's suggestion on using 'pre' only on the inner class of    *
//*   preformatted blocks.                                                     *
//*                                                                            *
//* - If response file ends too soon, then stdin can't see manually-entered    *
//*   keystrokes.                                                              *
//*                                                                            *
//* - If the '@anchor{}' command is used in the texi source, then the          *
//*   texi-to-html converter inappropriately concatenates the '@anchor' with   *
//*   the line that follows it. This may cause 'idpp' to miss the line         *
//*   following the '@anchor' command. For now, just be sure to follow all     *
//*   @anchor{} commands with a blank line.                                    *
//*                                                                            *
//* - Research other Texinfo customization variables.                          *
//*   - At least SOME of the HTML-only build options in makeinfo cause invalid *
//*     parsing of the document. (BEFORE_TOC, etc.)                            *
//* - Research how images (or other objects) are embedded in the HTML output.  *
//*   There is a good chance that it's not handled very well.                  *
//* - Research how footnotes are generated in the HTML output.                 *
//* - Verify that user is not prompted for a response if the file has already  *
//*   been processed for response-worthy objects. I.E. If a table already has  *
//*   a border, don't ask again. If an <ol> list already has a class, don't    *
//*   ask again.                                                               *
//* - Add a chapter for instructions on installing info document into the      *
//*   info system?                                                             *
//* - If TOC is removed, then we should also remove the '[Contents]' links.    *
//*                                                                            *
//* - The @smallformat followed by a @verbatim block suffers from the same     *
//*   extra whitespace as the preformatted block sequences. Not sure if it is  *
//*   worth the code and performance hit to check for it. Think about this.    *
//*                                                                            *
//* - Remember to insert the following at the top of infodoc_css.html          *
//*   <img style="margin:0px 0px 0px 1082px; border:0;width:88px;height:31px;" 
//*   src="vcss-new.gif" title="W3C Validated CSS3!" alt="Valid CSS3!"/>
//*                                                                            *
//* - infodoc-styles.css                                                       *
//*   - Note that the @shortcontents command generates:                        *
//*      <h2 class="shortcontents-heading">Short Table of Contents</h2>        *
//*     and a:                                                                 *
//*      <div class="shortcontents">                                           *
//*     Neither of these classes are defined in our CSS file. This is not      *
//*     critical because it is just an unnumbered list, but we may want to add *
//*     these definitions.                                                     *
//*   - Note that the SIMPLE_MENU customization variable generates:            *
//*     <div class="menu"><pre class="menu-preformatted">                      *
//*                                                                            *
//*                                                                            *
//******************************************************************************

//****************
//* Header Files *
//****************
#include "idpp.hpp"

//****************
//* Local Data   *
//****************
extern const wchar_t* dToken ;
extern const char*    cssLINK0 ;
const wchar_t* dfltTargPath = L"#" ;
const short    dfltTargPath_len = 2 ;
const wchar_t* dfltTargText = L"(top)" ;
const short    dfltTargText_len = 6 ;

//**********************
//* Non-member methods *
//**********************
bool  gclaGetArg ( const gString& gs, Cfg& trg ) ;


//*************************
//*         main          *
//*************************
//******************************************************************************
//* Program entry point.                                                       *
//*                                                                            *
//* Command-line Usage:  See GetCommandLineArgs() below for command-line       *
//*                      argument processing.                                  *
//*                                                                            *
//* Returns: OK  (0)  if all processing completed successfully                 *
//*          ERR (-1) if                                                       *
//*                   a) one or more command-line arguments invalid            *
//*                   b) one or more specified source files not found,         *
//*                      inaccessible, or not valid HTML                       *
//*                   c) CSS definition file not found or corrupted            *
//*                   d) processing error                                      *
//*                   e) user aborted the operation                            *
//******************************************************************************

int main ( int argc, char* argv[], char* argenv[] )
{
   //* Gather our entry-sequence data *
   commArgs clArgs( argc, argv, argenv ) ;

   //* Create the application class object *
   Idpp idpp( clArgs ) ;

   return ( int(idpp.ProcStatus()) ) ;

}  //* End main() *

//*************************
//*         ~Idpp         *
//*************************
//******************************************************************************
//* Destructor. Return all resources to the system.                            *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

Idpp::~Idpp ( void )
{

   delete this->pushBack ; // release the dynamic memory allocation

}  //* End ~Idpp() *

//*************************
//*         Idpp          *
//*************************
//******************************************************************************
//* Default constructor.                                                       *
//*                                                                            *
//* Input  : commArgs class object (by reference)                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

Idpp::Idpp ( commArgs& ca )
{
   //* Set the locale to the environment's choice i.e. UTF-8 encoding.*
   //* This affects all input and output streams.                     *
   std::locale::global(std::locale(""));

   //* Initialize our data members *
   this->sfCount = this->emCount = ZERO ;
   this->slCount = this->tlCount = ZERO ; // (unsigned values)
   this->iMode = false ;                  // (automatic processing is default)
   this->cssFile = L"infodoc-styles.css" ;
   this->pushBack = new fifo( 8 ) ;       // fifo buffer has 8 elements
   this->procStatus = OK ;                // exit code (0 or -1)

   //* <ol> and <ul> list processing _including_     *
   //* special-case processing is enabled by default.*
   this->ulLists = this->olLists = cfgAuto ;
   this->liSpec = true ;

   //* Pre-formatted blocks inherit font size from container by default *
   this->blkFont = cfgAuto ;

   //* Borders are applied to all main-sequence tables by default *
   this->tabBorder = cfgAuto ;

   this->allFiles = this->multiPass = this->ulAssign = this->tocMod   = 
   this->tocDel   = this->upTarg    = this->verbose  = this->css_mod  = 
   this->no_mods  = this->no_html5  = this->no_utrg  = this->no_body  = 
   this->no_bloc  = this->no_auth   = this->no_meta  = this->no_link  = 
   this->no_cont  = this->no_cart   = false ;

   //* Initialize default "Up" target link path and display text *
   wcsncpy ( this->utPath, dfltTargPath, dfltTargPath_len ) ;
   wcsncpy ( this->utText, dfltTargText, dfltTargText_len ) ;

   this->ppfGetCWD ( this->cwDir ) ;   // get path of current-working directory

   //* Initialize debugging option parameters *
   this->scan_beg = ZERO ; this->scan_end = 0xFFFF ; this->scan = false ;
   this->book = false ;
   #if DEBUG_USER_INPUT != 0  // for debugging only
   this->invalidResp = ZERO ;
   #endif   // for debugging only

   //* User may have specified one or more command-line arguments *
   bool validArgs = this->GetCommandLineArgs ( ca ) ;

   //* If full interactive mode, set the secondary-level processing switches.*
   if ( this->iMode )
   {
      this->ulLists   = cfgSpec ;
      this->olLists   = cfgSpec ;
      this->tabBorder = cfgSpec ;
      this->blkFont   = cfgSpec ;
   }

   //* If '--up_target' option specified, parse the input string *
   //* into target-path and display-text segments.               *
   if ( this->upTarg != false )
   {
      gString gstmp( this->utPath ) ;
      short idx = gstmp.find( L',' ) ;
      if ( idx >= ZERO )      // if display text ALSO specified
      {
         wcsncpy ( this->utText, &gstmp.gstr()[idx + 1], TARG_TEXT_LEN ) ;
         gstmp.limitChars( idx ) ;
         gstmp.copy( this->utPath, gsMAXCHARS ) ;
      }
   }

   //* If valid user input, and not a cry for help *
   if ( validArgs && (ca.helpFlag == false) && (ca.verFlag == false) ) 
   {
      //* Determine whether to perform a pre-scan of the   * 
      //* first source file, then process the source files.*
      if ( (this->Prescan ( ZERO, ca.preFlag )) )
      {
         this->procStatus = this->ProcessSource () ;

         #if DEBUG_USER_INPUT != 0  // for debugging only
         if ( this->invalidResp > ZERO )
         {
            gString gstmp( "\n*** Invalid User Responses: %hd ***", 
                           &this->invalidResp ) ;
            wcout << gstmp.gstr() << endl ;
         }
         #endif   // for debugging only
      }
   }           // valid user input

   else        // explicit or implied cry for help
   {
      if ( ca.verFlag != false )
         this->DisplayAppVersion () ;
      else if ( (ca.helpFlag != false) || (this->emCount == ZERO) )
         this->DisplayCommandLineHelp () ;
      else     // display accumulated command-line errors
      {
         gString gsOut ;
         gsOut.compose( AppTitleTemplate, 
                        AppTitle1, AppVersion, AppYears, AppTitle2, AppTitle3 ) ;
         this->textOut ( gsOut ) ;
         for ( short i = ZERO ; i < this->emCount ; i++ )
            this->textOut ( this->ErrorMsg[i] ) ;
         this->textOut ( "\nFor more information: 'idpp --help'\n" ) ;
      }
   }

}  //* End Idpp() *

//*************************
//*        Prescan        *
//*************************
//******************************************************************************
//* Pre-scan the specified source file, and if it has been previously          *
//* processed, optionally ask user whether to continue with re-processing.     *
//*                                                                            *
//* Note that the optional user confirmation is requested ONLY for the first   *
//* file in the list of source files to be scanned, and the user's response    *
//* applies only to the first file in the list. Second and subsequent source   *
//* files will be handled according to the rules described below.              *
//*                                                                            *
//* Input  : srcIndex : sourcefile index (index into this->srcFiles[])         *
//*          prompt   : (optional, 'false' by default)                         *
//*                     if 'false', perform the scan but do not prompt for     *
//*                                 user confirmation                          *
//*                     if 'true',  perform the scan, and if file was          *
//*                                 previously processed, prompt user whether  *
//*                                 to continue processing                     *
//*                     On first call, 'prompt' reflects the '-V' (verify)     *
//*                     command-line argument. Flag is reset for second and    *
//*                     subsequent calls.                                      *
//*                                                                            *
//* Returns: 'true'  if source not previously scanned OR                       *
//*                  if user specifies that file is to be scanned again OR     *
//*                  if application invocation specifies automatic processing  *
//*          'false' if user abort                                             *
//******************************************************************************
//* Previous scan is indicated by the presence of the post-processing message  *
//* at the top of the <head> section.                                          *
//*                                                                            *
//* Processing a file multiple times can lead to unintended results:           *
//*    1) If "automatic processing" specified i.e. interactive flags are all   *
//*       reset, then it is likely (though not guaranteed) that the target     *
//*       will be nearly identical to the source file.                         *
//*       For this reason, second and subsequent passes are allowed            *
//*       invocation parameters specify full automatic processing.             *
//*                                                                            *
//*    2) If user interaction is specified for second and subsequent passes,   *
//*       that interaction could easily override decisions made during the     *
//*       first pass.                                                          *
//*       a) If user is manually entering responses to processing prompts, we  *
//*          must assume that the user knows what he/she is doing.             *
//*          If user is informed that the file was previously processed, but   *
//*          decides to continue anyway, then again, we must assume that the   *
//*          user knows what she is are doing (although she probably doesn't). *
//*       b) However, if responses to the interactive prompts are provided by  *
//*          a response file, it must be assumed that the response file was    *
//*          designed to apply to a fresh (newly-generated) HTML file.         *
//*          If such a response file is used against a previously-processed    *
//*          source file, it is very likely that the responses will get        *
//*          out-of-synch causing unintended consequence including multiple    *
//*          invalid responses to individual prompts.                          *
//*          For this scenario, we will display a message alerting the user to *
//*          the danger of processing errors, and then disable modifications   *
//*          for during processing for this source file AND for all subsequent *
//*          files in the source list.                                         *
//*                                                                            *
//******************************************************************************

bool Idpp::Prescan ( short srcIndex, bool prompt )
{
   const wchar_t* headEND = L"</head>" ;
   char    tmp[gsMAXBYTES] ; // UTF-8 line input from file
   gString gs,
           gsTrg( "%S/%S", this->cwDir.gstr(), this->srcFiles[srcIndex] ),
           gscomp( cssLINK0 ) ;
   gscomp.limitChars( 37 ) ;
   bool    proc = false,      // 'true' if source file already processed
           force =            // 'true' if invocation parameters require prescan
              (this->iMode || 
               (this->ulLists == cfgSpec)   || (this->olLists == cfgSpec) ||
               (this->tabBorder == cfgSpec) || (this->blkFont == cfgSpec))
               && ((this->respFile.gschars()) > 1),
           status = true ;    // return value, assume that file will be processed
   this->multiPass = false ;  // initialize the global flag

   if ( prompt || force )
   {
      //* Open the source file and scan for the marker *
      ifstream ifs( gsTrg.ustr(), ifstream::in ) ;
      if ( (ifs.is_open()) )
      {
         bool done = false ;
         while ( ! done )
         {
            ifs.getline( tmp, gsMAXBYTES, NEWLINE ) ;
            if ( ifs.good() || ifs.gcount() > ZERO )
            {
               gs = tmp ;
               if ( (gs.find( gscomp.gstr()) >= ZERO) )
               { proc = true ; done = true ; }
               else if ( (gs.find( headEND )) >= ZERO )
                  done = true ;
            }
            else
               done = true ;
         }
         ifs.close() ;
      }

      //* If source file was previously processed *
      if ( proc )
      {
         //* If prompt enabled, ask user if we should continue *
         if ( prompt )
         {
            gs.compose( "The source file \"%S\" has already been processed.\n"
                        "Do you want to process it again? (y/n): ", 
                        this->srcFiles[srcIndex] ) ;
            this->textOut( gs, false ) ;
            this->userResponse ( gscomp ) ;
            if ( ((*gscomp.gstr()) != L'y') && ((*gscomp.gstr()) != L'Y') )
               status = false ;  // abort processing
         }

         //* If source file has been previously processed, but user *
         //* prompt not specified, perform internal initializations.*
         else
         {
            this->no_mods = this->multiPass = true ;
         }
      }
   }
   return status ;

}  //* End Prescan() *

//*************************
//*     ProcessSource     *
//*************************
//******************************************************************************
//* Process the specified HTML documents.                                      *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: OK  if all files processed successfully                           *
//*          ERR if processing error(s) or user aborted operation              *
//******************************************************************************

short Idpp::ProcessSource ( void )
{
   gString gsSrc,          // source path/filename
           gsTrg,          // target path/filename
           gsName,         // misc. filename
           gsOut ;         // message output
   short   status = OK ;   // return value

   //* Display the application title, CWD, and CSS filename/version *
   gsOut.compose( AppTitleTemplate, 
                  AppTitle1, AppVersion, AppYears, AppTitle2, AppTitle3 ) ;
   this->textOut ( gsOut ) ;
   gsOut.compose( L"CWD: %S", this->cwDir.gstr() ) ;
   this->textOut ( gsOut ) ;
   gsOut.compose( L"CSS: %S", this->cssFile.gstr() ) ;
   this->textOut ( gsOut ) ;
   gsOut.compose( L"CSS: %S", this->cssVersion ) ;
   this->textOut ( gsOut ) ;

   //* If a response file was specified, open it now. *
   if ( this->respFile.gschars() > 1 )
      this->rifs.open ( this->respFile.ustr(), ifstream::in ) ;


   //* Process each file in list *
   for ( short i = ZERO ; (i < this->sfCount) && (status == OK) ; ++i )
   {
      //* If target modification is enabled, _provisionally_ test *
      //* whether the source file has been previously processed,  *
      //* and if it has, append a user alert. Note that prescan   *
      //* of first source file was performed by caller.           *
      if ( !this->no_mods && (i > ZERO) )
         this->Prescan ( i ) ;

      //* Declare the file being processed.*
      gsOut.compose( L">>> %S", this->srcFiles[i] ) ;
      if ( this->multiPass )
         gsOut.append( L" (previously processed)" ) ;
      this->textOut ( gsOut ) ;

      //* If processing is enabled *
      if ( this->no_mods == false )
      {
         //* Create full path/filename for source(backup) file *
         this->ppfCatPathFilename ( gsTrg, this->cwDir, this->srcFiles[i] ) ;
         gsSrc.compose( L"%S~", gsTrg.gstr() ) ;
         this->ppfExtractFilename ( gsName, gsSrc ) ;
         gsOut.compose( L"    src: %S", gsName.gstr() ) ;
   
         //* If existing backup file, delete it *
         if ( this->ppfTargetExists ( gsSrc ) )
         {
            status = this->ppfDeleteFile ( gsSrc ) ;
            if ( this->verbose || status != OK )
               gsOut.append( L"  (delete existing backup)" ) ;
            if ( status != OK )
               gsOut.append( L" FAILED!" ) ;
            this->textOut ( gsOut ) ;
         }

         //* Rename original file as backup file *
         if ( status == OK )
         {
            this->ppfExtractFilename ( gsName, gsTrg ) ;
            gsOut.compose( L"    trg: %S", gsName.gstr() ) ;
            status = this->ppfRenameFile ( gsTrg, gsSrc ) ;
            if ( this->verbose || status != OK )
               gsOut.append( L"   (backup source file)" ) ;
            if ( status != OK )
               gsOut.append( L" FAILED!" ) ;
            this->textOut ( gsOut ) ;
         }
         gsOut.clear() ; this->textOut( gsOut ) ;
      }
      //* Else, we are performing a scan-only pass *
      else
      {
         //* Create full path/filename for source file *
         this->ppfCatPathFilename ( gsSrc, this->cwDir, this->srcFiles[i] ) ;
         this->ppfExtractFilename ( gsName, gsSrc ) ;
         gsOut.compose( L"    src: %S\n"
                         "    trg: (none)", gsName.gstr() ) ;
         this->textOut ( gsOut ) ;
         gsTrg.clear() ;      // target set to null string
      }

      //* Process source file, creating new target file *
      if ( status == OK )
      {
         if ( (status = ppfProcessSrcHTML ( gsSrc, gsTrg )) == OK )
         {
            this->textOut ( L"    Conversion successful!" ) ;
            if ( this->verbose )
            {
               gsOut.compose( L"    (%4hu source lines processed and )", 
                              &this->slCount ) ;
               if ( this->no_mods != false )
               {
                  gsOut.limitChars( gsOut.gschars() - 6 ) ;
                  gsOut.append( L')' ) ;
               }
               this->textOut ( gsOut ) ;
               if ( this->no_mods == false )
               {
                  gsOut.compose( L"    (%4hu lines written to target    )", 
                                 &this->tlCount ) ;
                  this->textOut ( gsOut ) ;
               }
            }
         }
         else
         {
            gsOut.compose( L"    Processing error on source line: %hu\n"
                            "    Conversion failed: STOP!", &this->slCount ) ;
            this->textOut ( gsOut ) ;
         }
         gsOut.clear() ; this->textOut( gsOut ) ;
      }
      else  // preparing to exit
         { gsOut.clear() ; this->textOut( gsOut ) ; }
   }

   //* If a response file was opened, close it.*
   if ( this->rifs.is_open() )
      this->rifs.close() ;

   return status ;

}  //* End ProcessSource() *

//*************************
//*       textOut         *
//*************************
//******************************************************************************
//* All, or nearly all text written to the display goes through this method,   *
//* so we can control whether it goes through stdout (wide stream) or through  *
//* the NcDialog output stream.                                                *
//*                                                                            *
//* Input  : tOut   : text data to be displayed                                *
//*          newln  : (optional, true by default)                              *
//*                   if 'true', then terminate the line with an 'endl'        *
//*                   if 'false', do not termnate the line                     *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************
//* Programmer's Note: We channel as much of the stdout stream as possible     *
//* through this method to avoid future effort if we later add the NcDialog    *
//* (ncurses) interface which is incompatible with stdin/stdout.               *
//******************************************************************************

void Idpp::textOut ( const gString& tOut, bool newln )
{
   wcout << tOut.gstr() ;
   if ( newln != false )
      wcout << endl ;

}  //* End textOut() *
void Idpp::textOut ( const char* tOut, bool newln )
{
   gString gs( tOut ) ;
   this->textOut ( gs, newln ) ;

}  //* End textOut() *
void Idpp::textOut ( const wchar_t* tOut, bool newln )
{
   gString gs( tOut ) ;
   this->textOut ( gs, newln ) ;

}  //* End textOut() *

//*************************
//*     userResponse      *
//*************************
//******************************************************************************
//* Get user response to text-mode prompts.                                    *
//*                                                                            *
//* Input  : gsIn   : (by reference) receives user input                       *
//*                                                                            *
//* Returns: OK  if data successfully received                                 *
//*          ERR if end of input script encountered, indicating that caller    *
//*              should re-prompt the user. (gsIn will contain an empty string)*
//******************************************************************************
//* Programmer's Notes:                                                        *
//* - User is prompted to enter a response for selecting a class type and      *
//*   optionally, a start index for <ol> (enumerated list) tags.               *
//* - User is prompted to enter a response for selecting whether to place a    *
//*   border around a <table> object.                                          *
//* - Direct user input is straightforward: the interface expects character or *
//*   string input, terminated by an ENTER key.                                *
//* - Input may be redirected from a response file, for which we define a      *
//*   placeholder string: 'default_token'.                                     *
//* - We wait indefinitely for user input.                                     *
//* - If the first character of the response is the :'#' (hash) character,     *
//*   then assume that the token is a comment and ignore it.                   *
//* - First character of input must not be an ASCII control character.         *
//*   (later, we may want to enforce a _printing_ character as the first)      *
//*                                                                            *
//* - If the token comes from a response file, then the shell does not echo it *
//*   to the display. This throws off the formatting of the prompts.           *
//*   Unfortunately, we can't tell whether the token came from the keyboard or *
//*   from a response file--they look the same after buffering. So, we output  *
//*   an extra linefeed after a valid token.                                   *
//******************************************************************************

short Idpp::userResponse ( gString& gsIn )
{
   wchar_t userInput[gsMAXCHARS] = L"" ;
   bool  done = false ;             // loop control
   short status = OK ;              // return value

   gsIn.clear() ;             // clear caller's old data

   //* If we are running in scan-only mode, don't prompt for responses. *
   //* Instead, just return the 'default_token' string.                 *
   if ( this->no_mods != false )
   {
      gsIn = dToken ;
      this->textOut ( gsIn ) ;
      done = true ;
   }

   //* If we have an open response file, read a line. *
   if ( (this->respFile.gschars() > 1) && (this->rifs.is_open()) )
   {
      while ( ! done )
      {
         if ( (this->ppfReadLine ( this->rifs, gsIn )) == OK )
         {
            gsIn.strip() ;                // remove leading/trailing whitespace
            short wi = gsIn.find( L'#' ) ;// remove comments
            if ( wi >= ZERO )
               gsIn.limitChars( wi ) ;
            wi = gsIn.find( L' ' ) ;      // whitespace is the delimiter
            if ( wi >= ZERO )
               gsIn.limitChars( wi ) ;    // remove everything after the first token
            if ( (gsIn.gschars()) > 1 )   // if we have a token, echo it to display
            {
               this->textOut ( gsIn ) ;
               done = true ;              // return token to caller
            }
         }
         else  // end-of-file
         {
            this->rifs.close() ;
            break ;
         }
      }
   }

   //* Get user response from 'wcin' *
   while ( ! done )
   {
      wcin >> userInput ;     // get user response

      //* Verify that input is not a control character and not a comment *
      if ( *userInput >= SPACE && *userInput != L'#' )
      {
         gsIn = userInput ;   // copy response to caller's buffer
         this->textOut ( L"" ) ;
         break ;
      }
   }
   return status ;

}  //* End userResponse() *

//*************************
//*    invalidResponse    *
//*************************
//******************************************************************************
//* If user provides an invalid response to the prompt, re-prompt.             *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void Idpp::invalidResponse ( void )
{
   const wchar_t* badUser = L"Invalid response, your choice?: " ;

   this->textOut ( badUser, false ) ;

   #if DEBUG_USER_INPUT != 0  // for debugging only
   ++this->invalidResp ;
   this->textOut ( L"? ", false ) ;
   wchar_t userInput[gsMAXCHARS] = L"" ;
   wcin >> userInput ;
   #endif   // for debugging only

}  //* End invalidResponse() *

//*************************
//*  GetCommandLineArgs   *
//*************************
//******************************************************************************
//* Capture user's command-line arguments.                                     *
//*                                                                            *
//* Valid Arguments: (see DisplayCommandLineHelp())                            *
//*                                                                            *
//* Input  : commArgs class object (by reference)                              *
//*                                                                            *
//* Returns: 'true' if all arguments are valid                                 *
//*          'false' if invalid argument(s)                                    *
//******************************************************************************
//*  All arguments are optional EXCEPT:                                        *
//*    a) at least one source filename must be specified (16 names max)        *
//*       OR                                                                   *
//*    b) the "-a" (all files) switch must be specified                        *
//*                                                                            *
//*  c) parameters with string arguments may be specified as:                  *
//*     - string immediately follows:  idpp -dpublic_html                      *
//*     - string is next argument   :  idpp -d public_html                     *
//*     - string follows '=' sign   :  idpp -d=public_html                     *
//*  d) switches with sub-switch arguments                                     *
//*     - sub-switches follow immediately and in any order                     *
//*     - Note: no sub-switches defined at this time.                          *
//*  e) -h, -H, -? or arg error overrides all switches except --version        *
//*  f) --version overrides all other switches                                 *
//*                                                                            *
//*                                                                            *
//* -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  *
//* Debugging options:                                                         *
//* ==================                                                         *
//* --scan=start,end                                                           *
//*   As lines are read from source print them to the display.                 *
//*   Optional arguments: starting source line number and ending source line   *
//*   number, separated by a comma ','. (See ppfReadSrcLine method)            *
//*   a) If no arguments, then display all lines read.                         *
//*      --scan                                                                *
//*   b) If only first argument, then it is start. Continue scan to the end.   *
//*      --scan=400                                                            *
//*   c) If both arguments, then display all lines in the range, inclusive.    *
//*      --scan=400,525                                                        *
//*                                                                            *
//* --book                                                                     *
//*   For the main processing loop (ppfProcessSrcHTML method), display a       *
//*   message when a sub-process method is called and another message when it  *
//*   returns. This helps to locate the place in the source document where the *
//*   processing logic has gone off into the weeds, i.e. if the end of the     *
//*   processing block is not properly identified by the sub-process method.   *
//*                                                                            *
//******************************************************************************

bool Idpp::GetCommandLineArgs ( commArgs& ca )
{
   #define DEBUG_GCLA (0)        // for debugging only

   gString gs,                   // data formatting
           gsOut ;               // user messages
   const short argTest_len = 9 ; // number of chararacters to compare
   bool status = true ;          // return value

   if ( ca.argCount > 1 )
   {
      short j = ZERO ;
      bool multiarg = false ;
      for ( short i = 1 ; (i < ca.argCount) || (multiarg != false) ; ++i )
      {
         if ( multiarg != false )   // if additional switches in same argument
            --i ;
         else
            j = ZERO ;

         //* If a command-line switch OR continuing previous switch argument *
         if ( ca.argList[i][j] == DASH || multiarg != false )
         {
            multiarg = false ;
            ++j ;
            if ( ca.argList[i][j] == DASH ) // (double dash)
            {  //* Long-form command switches *
               bool goodarg = false ;
               ++j ;

               //* Options that begin with 'b' *
               if ( ca.argList[i][j] == 'b' )
               {
                  gs = ca.argList[i] ;

                  //* Test for bullet-list option *
                  if ( (gs.compare( L"--bullet_list", true, argTest_len ) == ZERO) )
                  {
                     goodarg = gclaGetArg ( gs, this->ulLists ) ;
                  }

                  //* Test for font-size option for preformatted blocks *
                  else if ( (gs.compare( L"--block_font", true, argTest_len ) == ZERO) )
                  {
                     if ( (goodarg = gclaGetArg ( gs, this->blkFont )) != false )
                     {
                        if ( (this->blkFont != cfgAuto) && (this->blkFont != cfgSpec) )
                           goodarg = false ;
                     }
                  }

                  //* Test for '--book' option (book-end the blocks) *
                  else if ( (gs.compare( L"--book", true, 6 )) == ZERO )
                  {
                     goodarg = this->book = true ;
                  }
               }

               //* Options that begin with 'c' *
               else if ( ca.argList[i][j] == 'c' )
               {  
                  #if 0    // NOT YET IMPLEMENTED
                  gs = ca.argList[i] ;

                  //* Request to modify the whole-document CSS options *
                  if ( (gs.compare( L"--css_mods", true, argTest_len ) == ZERO) )
                  {
                     goodarg = this->css_mod = true ;

                     if ( this->emCount < emMAX )
                     {
                        gsOut = "Sorry, interactive modification of CSS definitions "
                                "not yet implemented." ;
                        gsOut.copy( this->ErrorMsg[this->emCount++], gsMAXCHARS ) ;
                     }
                     status = false ;
                     break ;
                  }
                  #endif   // NOT YET IMPLEMENTED
               }

               //* Options that begin with 'e' *
               else if ( ca.argList[i][j] == 'e' )
               {
                  gs = ca.argList[i] ;

                  //* Test for enumeration-list option *
                  if ( (gs.compare( L"--enum_list", true, argTest_len ) == ZERO) )
                  {
                     goodarg = gclaGetArg ( gs, this->olLists ) ;
                  }
               }

               //* Options that begin with 'f' *
               else if ( ca.argList[i][j] == 'f' )
               {
                  gs = ca.argList[i] ;

                  if ( (gs.compare( L"--fixed_list", true, argTest_len ) == ZERO) )
                  {
                     goodarg = this->ulAssign = true ;
                  }
               }

               //* Options that begin with 'm' *
               else if ( ca.argList[i][j] == 'm' )
               {
                  gs = ca.argList[i] ;

                  //* Request to insert custom data into <head> section *
                  if ( (gs.compare( L"--my_metadata", true, argTest_len ) == ZERO) )
                  {
                     goodarg = true ; // good command, test its argument
                     if ( (status = this->gclaGetFilename ( gs, this->userMeta )) == false )
                     {
                        if ( this->emCount < emMAX )
                        {
                           gsOut.compose( "Error! Metadata file '%S' not found!", 
                                          this->userMeta.gstr() ) ;
                           gsOut.copy( this->ErrorMsg[this->emCount++], gsMAXCHARS ) ;
                        }
                        this->userMeta.clear() ;
                     }
                  }
               }

               //* Options that begin with 'r' *
               else if ( ca.argList[i][j] == 'r' )
               {
                  gs = ca.argList[i] ;

                  //* Test for 'response' option *
                  if ( (gs.compare( L"--response", true, 6 )) == ZERO )
                  {
                     goodarg = true ; // good command, test its argument
                     if ( (status = this->gclaGetFilename ( gs, this->respFile )) == false )
                     {
                        if ( this->emCount < emMAX )
                        {
                           gsOut.compose( "Error! Specified response file '%S' not found!", 
                                          this->respFile.gstr() ) ;
                           gsOut.copy( this->ErrorMsg[this->emCount++], gsMAXCHARS ) ;
                        }
                        this->respFile.clear() ;
                     }
                  }
               }

               //* Options that begin with 's' *
               else if ( ca.argList[i][j] == 's' )
               {
                  gs = ca.argList[i] ;

                  //* Test for '--scan' option *
                  if ( (gs.compare( L"--scan", true, 6 )) == ZERO )
                  {
                     goodarg = this->scan = true ;
                     uint16_t b, e ;
                     int args = swscanf ( &gs.gstr()[6], L"=%hu,%hu", &b, &e ) ;
                     switch ( args )
                     {
                        case 1:     // scan from
                           this->scan_beg = b ;
                           break ;
                        case 2:     // scan range
                           this->scan_beg = b ;
                           this->scan_end = e ;
                           break ;
                        case 0:     // scan all
                        case WEOF:
                           this->scan_beg = ZERO ;
                           this->scan_end = 0xFFFF ;
                           break ;
                        default:
                           goodarg = this->scan = false ;
                           break ;
                     }
                  }
               }

               //* Options that begin with 't' *
               else if ( ca.argList[i][j] == 't' )
               {
                  gs = ca.argList[i] ;

                  //* Request to place borders around table objects *
                  if ( (gs.compare( L"--table_border", true, argTest_len ) == ZERO) )
                  {
                     goodarg = gclaGetArg ( gs, this->tabBorder ) ;
                  }
               }

               //* Options that begin with 'u' *
               else if ( ca.argList[i][j] == 'u' )
               {
                  gs = ca.argList[i] ;

                  //* Test for "up target" specification *
                  if ( (gs.compare( L"--up_target", true, argTest_len ) == ZERO) )
                  {
                     goodarg = true ; // good command, test its argument
                     // Programmer's Note: The data are not a real filename, 
                     // so we ignore the returned status.
                     this->gclaGetFilename ( gs, gsOut ) ;
                     if ( gsOut.gschars() > 1 )
                     {
                        gsOut.copy( this->utPath, gsMAXCHARS ) ;
                        this->upTarg = true ;
                     }
                  }
               }

               //* Request for application version number *
               else if ( ca.argList[i][j] == 'v' )
               {  //* Most Linux console apps respond to a       *
                  //* '--version' request in a cannonical manner.*
                  //* Overrides everything else on comand line.  *
                  gs = ca.argList[i] ;
                  if ( (gs.compare( L"--version", true, argTest_len ) == ZERO) )
                  {
                     ca.reset() ;
                     ca.helpFlag = ca.verFlag = true ;
                     #if DEBUG_GCLA != 0
                     this->textOut ( L"--version: specified" ) ;
                     #endif   // DEBUG_GCLA
                     break ;
                  }
               }

               //* Long-form request for Help *
               else if ( ca.argList[i][j] == 'h' || ca.argList[i][j] == 'H' )
               {
                  ca.helpFlag = true ;
                  #if DEBUG_GCLA != 0
                  this->textOut ( L"--help: specified" ) ;
                  #endif   // DEBUG_GCLA
               }

               //* Exceptions to default processing *
               else if ( ca.argList[i][j] == 'n' )
               {
                  gs = ca.argList[i] ;
                  if ( (gs.compare( L"--no_mods", true, argTest_len ) == ZERO) )
                     goodarg = this->no_mods = this->verbose = true ;
                  else if ( (gs.compare( L"--no_spec", true, argTest_len ) == ZERO) )
                  { goodarg = true ; this->liSpec = false ; }
                  else if ( (gs.compare( L"--no_html", true, argTest_len ) == ZERO) )
                     goodarg = this->no_html5 = true ;
                  else if ( (gs.compare( L"--no_uplink", true, argTest_len ) == ZERO) )
                     goodarg = this->no_utrg = true ;
                  else if ( (gs.compare( L"--no_body", true, argTest_len ) == ZERO) )
                     goodarg = this->no_body = true ;
                  else if ( (gs.compare( L"--no_meta", true, argTest_len ) == ZERO) )
                     goodarg = this->no_meta = true ;
                  else if ( (gs.compare( L"--no_link", true, argTest_len ) == ZERO) )
                     goodarg = this->no_link = true ;
                  else if ( (gs.compare( L"--no_bloc", true, argTest_len ) == ZERO) )
                     goodarg = this->no_bloc = true ;
                  else if ( (gs.compare( L"--no_auth", true, argTest_len ) == ZERO) )
                     goodarg = this->no_auth = true ;
                  else if ( (gs.compare( L"--no_cont", true, argTest_len ) == ZERO) )
                     goodarg = this->no_cont = true ;
                  else if ( (gs.compare( L"--no_cart", true, argTest_len ) == ZERO) )
                     goodarg = this->no_cart = true ;
               }

               if ( goodarg != false )
               {
                  #if DEBUG_GCLA != 0
                  this->textOut ( gs, false ) ;
                  this->textOut ( L": specified" ) ;
                  #endif   // DEBUG_GCLA
               }
               else           // invalid switch
               {  //* Generate an error mesage *
                  if ( this->emCount < emMAX )
                  {
                     gs.compose( "Error! Unrecognized command: '%s' ", ca.argList[i] ) ;
                     gs.copy( this->ErrorMsg[this->emCount++], gsMAXCHARS ) ;
                  }
                  status = false ;
               }
               continue ;     // finished with this argument
            }  // (double dash)
            char argLetter = ca.argList[i][j] ;

            //* Interactive Mode processing *
            if ( argLetter == 'i' || argLetter == 'I' )
            {
               this->iMode = true ;
               if ( ca.argList[i][j+1] != NULLCHAR ) multiarg = true ;
               #if DEBUG_GCLA != 0
               this->textOut ( L"-i: specified" ) ;
               #endif   // DEBUG_GCLA
            }

            //* Verbose output (applies only to text mode) *
            else if ( (argLetter == 'v') && 
                      (strncmp ( ca.argList[i], "--v", 3 )) != ZERO)
            {
               this->verbose = true ;
               if ( ca.argList[i][j+1] != NULLCHAR ) multiarg = true ;
               #if DEBUG_GCLA != 0
               this->textOut ( L"-v: specified" ) ;
               #endif   // DEBUG_GCLA
            }

            //* Perform pre-scan to determine whether  *
            //* source file has already been processed.*
            else if ( argLetter == 'V' )
            {
               ca.preFlag = true ;
               #if DEBUG_GCLA != 0
               this->textOut ( L"-V: specified" ) ;
               #endif   // DEBUG_GCLA
            }

            //* Specify alternate CSS definition file *
            else if ( argLetter == 'f' )
            {
               ++j ;    // index next character after switch character
               if ( ca.argList[i][j] == NULLCHAR ) // if path is in next arg
               { ++i ; j = ZERO ; }
               else if ( ca.argList[i][j] == '=' )
                  ++j ;
               if ( ca.argList[i][j] != DASH && ca.argList[i][j] != NULLCHAR )
               {
                  this->cssFile = &(ca.argList[i][j]) ;
                  #if DEBUG_GCLA != 0
                  this->textOut ( L"-f: ", false ) ;
                  this->textOut ( this->cssFile ) ;
                  #endif   // DEBUG_GCLA
               }
               else
               {
                  if ( this->emCount < emMAX )
                  {
                     gs = "Error! '-f' switch specified without filename" ;
                     gs.copy( this->ErrorMsg[this->emCount++], gsMAXCHARS ) ;
                  }
                  status = false ;
               }
            }

            //* Process ALL files in directory *
            else if ( argLetter == 'a' )
            {
               this->allFiles = true ;
               if ( ca.argList[i][j+1] != NULLCHAR ) multiarg = true ;
               #if DEBUG_GCLA != 0
               this->textOut ( L"-a: specified" ) ;
               #endif   // DEBUG_GCLA
            }

            //* Alternate target Directory *
            else if ( argLetter == 'd' )
            {  //* Change the current working directory *
               ++j ;    // index next character after switch character
               if ( ca.argList[i][j] == NULLCHAR ) // if path is in next arg
               { ++i ; j = ZERO ; }
               else if ( ca.argList[i][j] == '=' )
                  ++j ;
               gs = &(ca.argList[i][j]) ;
               #if DEBUG_GCLA != 0
               this->textOut ( L"-d: ", false ) ;
               this->textOut ( gs ) ;
               #endif   // DEBUG_GCLA
               if ( (this->ppfCdTarget ( gs.ustr() )) != OK )
               {
                  if ( this->emCount < emMAX )
                  {
                     gs.compose( "Error! Alternate target directory '%S' not found.", gs.gstr() ) ;
                     gs.copy( this->ErrorMsg[this->emCount++], gsMAXCHARS ) ;
                  }
                  status = false ;
               }
            }

            //* Process table of Contents *
            else if ( argLetter == 'c' )
            {
               this->tocMod = true ;
               if ( ca.argList[i][j+1] != NULLCHAR ) multiarg = true ;
               #if DEBUG_GCLA != 0
               this->textOut ( L"-c: specified" ) ;
               #endif   // DEBUG_GCLA
            }

            //* Remove table of contents *
            else if ( argLetter == 'r' )
            {
               this->tocDel = true ;
               if ( ca.argList[i][j+1] != NULLCHAR ) multiarg = true ;
               #if DEBUG_GCLA != 0
               this->textOut ( L"-r: specified" ) ;
               #endif   // DEBUG_GCLA
            }

            //* Request for Help *
            else if ( argLetter == 'h' || argLetter == 'H' || argLetter == '?' )
            {  //* A cry for help overrides everything else   *
               //* on the command line except Version.        *
               //* (see below)
               ca.helpFlag = true ;
               #if DEBUG_GCLA != 0
               this->textOut ( L"-h: specified" ) ;
               #endif   // DEBUG_GCLA
            }

            else           // invalid switch
            {  //* Generate an error mesage *
               if ( this->emCount < emMAX )
               {
                  gs.compose( "Error! Unrecognized command: '%s' ", ca.argList[i] ) ;
                  gs.copy( this->ErrorMsg[this->emCount++], gsMAXCHARS ) ;
               }
               status = false ;
            }
         }
         else
         {  //* Interpret argument as a filename. (validation occurs below) *
            if ( this->sfCount < sfMAX )
            {
               gs = ca.argList[i] ;
               gs.copy( this->srcFiles[this->sfCount++], gsMAXCHARS ) ;
            }
            else
            {  //* Generate an error message *
               if ( this->emCount < emMAX )
               {
                  gs.compose( "Error! Too many source files specified. '%s' ignored.", ca.argList[i] ) ;
                  gs.copy( this->ErrorMsg[this->emCount++], gsMAXCHARS ) ;
               }
            }
         }
      }     // for(;;)

      //* Verify that CSS definition file exists and is our file.*
      if ( status != false )
      {
         if ( ((this->ppfRealpath ( gs, this->cssFile )) == OK)
              && ((this->ppfTargetExists ( gs )) != false) )
         {  //* Verify copyright notice and version number *
            gString gsVer ;
            if ( this->ppfTargetIsCSS ( gs, gsVer ) )
            {
               gsVer.copy( this->cssVersion, CSS_VER_LEN ) ;
            }
            else
            {
               if ( this->emCount < emMAX )
               {
                  gs.compose( "Error! CSS definition file '%S'"
                              " is of unknown configuration!", 
                              this->cssFile.gstr() ) ;
                  gs.copy( this->ErrorMsg[this->emCount++], gsMAXCHARS ) ;
               }
               status = false ;
            }
         }
         else
         {
            if ( this->emCount < emMAX )
            {
               gs.compose( "Error! CSS definition file '%S' was not found!", 
                           this->cssFile.gstr() ) ;
               gs.copy( this->ErrorMsg[this->emCount++], gsMAXCHARS ) ;
            }
            status = false ;
         }
      }

      if ( ca.helpFlag && !ca.verFlag )
      {  //* Help overrides everything except Version *
         ca.reset() ;
         ca.helpFlag = true ;
      }
   }        // if(ca.argCount>1)

   //* If the '-a' switch specified, scan all files in the target directory  *
   //* for HTML documents, and create a list of files to be processed.       *
   if ( this->allFiles != false && status != false )
      status = this->ppfScan4Src () ;


   //* Validate the list of source files to be processed.*
   if ( status != false )
   {
      for ( short sfIndex = ZERO ; sfIndex < this->sfCount ; sfIndex++ )
      {
         this->ppfCatPathFilename ( gs, this->cwDir, this->srcFiles[sfIndex] ) ;
         if ( this->ppfTargetExists ( gs ) )
         {
            if ( (this->ppfTargetIsHTML ( gs )) == false )
            {  //* Generate an error message *
               if ( this->emCount < emMAX )
               {
                  gs.compose( "Error! Source file '%S' is not valid HTML!", this->srcFiles[sfIndex] ) ;
                  gs.copy( this->ErrorMsg[this->emCount++], gsMAXCHARS ) ;
               }
               status = false ;
            }
         }
         else
         {  //* Generate an error mesage *
            if ( this->emCount < emMAX )
            {
               gs.compose( "Error! Source file '%S' was not found!", this->srcFiles[sfIndex] ) ;
               gs.copy( this->ErrorMsg[this->emCount++], gsMAXCHARS ) ;
            }
            status = false ;
         }
      }
   }

   //* No command line arguments OR no filenames specified.         *
   //* If --css_mods specified then no HTML filenames are necessary.*
   if ( (ca.argCount == 1) || (this->sfCount == ZERO && this->css_mod == false) )
   {
      gs = "--no source filenames specified--" ;
      gs.copy( this->ErrorMsg[this->emCount++], gsMAXCHARS ) ;
      status = false ;
   }
   return status ;

#undef DEBUG_GCLA
}  //* End GetCommandLineArgs() *

//*************************
//*    gclaGetFilename    *
//*************************
//******************************************************************************
//* Called by GetCommandLineArgs() to capture and verify a filename argument   *
//* for a command-line parameter.                                              *
//*                                                                            *
//* Input  : cmd  : (by reference) command-line argument                       *
//*          fname: (by reference) receives the captured and verified filename *
//*                                                                            *
//* Returns: 'true'  if a valid filename                                       *
//*          'false' if file not found                                         *
//******************************************************************************

bool Idpp::gclaGetFilename ( const gString& cmd, gString& fname )
{
   gString gstmp ;                     // work buffers
   bool  goodarg = false ;             // return value
   short idx = cmd.after( L'=' ) ;     // index of argument

   fname.clear() ;           // initialize caller's buffer

   if ( (idx >= ZERO) && (cmd.gstr()[idx] != NULLCHAR) )
   {
      gstmp = &cmd.gstr()[idx] ;
      if ( ((this->ppfRealpath ( fname, gstmp )) == OK) &&
           ((this->ppfTargetExists ( fname )) != false) )
         goodarg = true ;
      else
         fname = gstmp ;
   }
   return goodarg ;

}  //* End gclaGetFilename() *

//*************************
//*      gclaGetArg       *
//*************************
//******************************************************************************
//* Called by GetCommandLineArgs() to isolate the arguments for long-form      *
//* command-line options.                                                      *
//*                                                                            *
//* Input  : gs   : (by reference) command-line argument                       *
//*          trg  : (by reference) receives the specified argument             *
//*                                                                            *
//* Returns: 'true'  if a valid argument (trg receives a member of enum Cfg)   *
//*          'false' if unknown argument (trg unmodified)                      *
//******************************************************************************

bool gclaGetArg ( const gString& gs, Cfg& trg )
{
   bool  goodarg = false ;    // return value
   short idx = gs.after( L'=' ) ;

   if ( (idx > ZERO) && (gs.gstr()[idx] != NULLCHAR) )
   {
      goodarg = true ;
      if ( (gs.compare( L"specify", true, 4, idx )) == ZERO )
         trg = cfgSpec ;
      else if ( (gs.compare( L"auto", true, 4, idx )) == ZERO )
         trg = cfgAuto ;
      else if ( (gs.compare( L"none", true, 4, idx )) == ZERO )
         trg = cfgNone ;
      else
         goodarg = false ;
   }
   return goodarg ;

}  //* gclaGetArg() *

//*************************
//*  DisplayAppVersion    *
//*************************
//******************************************************************************
//* Display the application's title, version and copyright info.               *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************
//* Programmer's Note: We play tricks with our title strings to make this      *
//* message look canonical. If the strings are changed, this message may get   *
//* ugly. That's why Trix are for kids....                                     *
//******************************************************************************

void Idpp::DisplayAppVersion ( void )
{
   const wchar_t* freeSoftware = 
   L"License GPLv3+: GNU GPL version 3 or later <http://gnu.org/licenses/gpl.html>\n"
    "This is free software: you are free to modify and/or redistribute it\n"
    "under the terms set out in the license.\n"
    "There is NO WARRANTY, to the extent permitted by law.\n" ;

   gString gsOut( "%S(idpp) version: %S\nCopyright (C) %S %S\n", 
                  AppTitle1, AppVersion, AppYears, AppTitle2 ) ;
   this->textOut ( gsOut ) ;
   this->textOut ( freeSoftware ) ;
   
}  //* End DisplayAppVersion() *

//**************************
//* DisplayCommandLineHelp *
//**************************
//******************************************************************************
//* Display the brief version of command-line help.                            *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void Idpp::DisplayCommandLineHelp ( void )
{
   gString gsOut ;
   gsOut.compose( AppTitleTemplate, 
                  AppTitle1, AppVersion, AppYears, AppTitle2, AppTitle3 ) ;
   this->textOut ( gsOut ) ;

   wcout << 
   //123456789-123456789-123456789-123456789-123456789-123456789-123456789-123456789-
   L"  Apply external CSS style definitions to HTML documents generated by\n"
    "  'makeinfo' utility. Source documents renamed as backup files, e.g.\n"
    "  'home.htm' becomes 'home.htm~'. Modifications written to NEW 'home.htm'.\n\n"
    "Usage  : idpp [OPTIONS][HTML_FILENAMES]\n"
    "             EXAMPLE: idpp -cv --table_border=specify home.html docs.html\n"
    "Options:\n"
    " -i     Interactive mode processing. Prompt for a decision on each item\n"
    "        to be formatted. (automatic formatting is the default)\n"
    " -a     Process all files in the target directory that look like HTML.\n"
    "        recognized filename extensions: html, htm, shtml, shtm, xhtml\n"
    " -d     Specify an alternate source directory.\n"
    "                  (current working directory is the default)\n"
    "        Examples: idpp -d=public_html\n"
    " -f     Specify an alternate filename for the CSS definition file.\n"
    "        ('infodoc-styles.css' in the target directory is the default)\n"
    "        Examples: idpp -f=my-styles.css home.html\n"
    " -c     Contents: process Table of Contents as a multi-level bullet list\n"
    " -r     Remove Table of Contents from the document\n"
    "        (Table of Contents is left unmodified by default)\n"
    " -v     Verbose output, report details of each operation\n"
    " -V     If a file has already been processed, Verify that the file is to\n"
    "        be processed again. (This test is not performed by default.)\n"
    " --bullet_list [=[auto | none | specify]]\n"
    "        'auto'  : automatic formatting of all bullet lists (default)\n"
    "        'none'  : do not apply formatting to bullet lists\n"
    "        'specify: application prompts for formatting option\n"
    " --enum_list [=[auto | none | specify]]\n"
    "        'auto'  : automatic formatting of all enumeration lists (default)\n"
    "        'none'  : do not apply formatting to enumeration lists\n"
    "        'specify: application prompts for enumeration type\n"
    " --table_border [=[auto | none | specify]]\n"
    "         Add a border/grid around the elements of a table.\n"
    "        'auto'   : automatically draw tables with border/gridlines (default)\n"
    "        'none'   : draw tables without border/gridlines\n"
    "        'specify': application prompts for a decision on each table\n"
    " --block_font [=[auto | specify]]\n"
    "        Font size for pre-formatted blocks. 'makeinfo' no longer supports\n"
    "        the \"@small...\" (small font) versions of preformatted blocks;\n"
    "        however, the font size for these blocks may optionally be specified\n"
    "        during post-processing \n"
    "        'auto'  : pre-formatted blocks inherit font size from container\n"
    "        'specify: application prompts for font size: (default/larger/smaller)\n"
    " --fixed_list\n"
    "        Do not allow browser to automatically reformat multi-level bullet\n"
    "        lists. (Browsers reformat nested bullet lists by default.)\n"
    " --up_target\n"
    "        If your document is a node on a document tree, specify path to\n"
    "        parent node and optionally the associated display text.\n"
    "        Example: idpp --up_target='../parent_node.htm'         (link only)\n"
    "        Example: idpp --up_target='../parent_node.htm,(home)'  (link and text)\n"
    "        (Up Target is set to top of current page by default.)\n"
    " --my_metadata\n"
    "        Insert custom metadata elements.\n"
    "        Example: idpp --my_metadata=metadata.htm\n"
    "        Contents of the specified file will be copied into the\n"
    "        document's <head> block, just above the </head> tag.\n"
    " --response\n"
    "        Specify a plain text file containing responses to the prompts\n"
    "        displayed by the interactive processing options. Write one token on\n"
    "        each line. Comments (text following a '#') will be ignored.\n"
    "        Example: idpp --response=interactive_responses.txt\n"
//    " --css_mods\n"
//    "        Specify the basic CSS definitions for background color, container\n"
//    "        width, font family, size and color. [NOT YET IMPLEMENTED]\n"
    " ---- ---- ---- ---- ---- ---- ---- ---- ---- ---- ---- ---- ---- ---- ---- \n"
    " --no_mods\n"
    "        Do not perform modifications. Instead, list the operations that\n"
    "        WOULD BE processed for the options specified. (assumes '-v' option)\n"
    " --no_special\n"
    "        Do not perform \"special case\" formatting for bullet lists and\n"
    "        enumeration lists. Default is to apply special processing rules.\n"
    "        (See documentation for details.)\n"
    " --no_html5\n"
    "        Do not update the <!DOCTYPE> tag and various other obsolete HTML\n"
    "        constructs. (updated to HTML5 spec by default)\n"
    " --no_meta\n"
    "        Do not delete the _valid_ <meta> elements in the <head> section.\n"
    "        section. These are: \"application-name\", \"author\", \"description\",\n"
    "        \"generator\", \"keywords\". (all metadata is discarded by default)\n"
    " --no_links\n"
    "        Do not delete the auto-generated <link> elements in the <head>\n"
    "        section (all links are discarded by default)\n"
    " --no_body\n"
    "        Do not update the <body> tag. (extra definitions discarded by default)\n"
    " --no_uplink\n"
    "        Do not modify the target specified in the \"Up\" hyperlink at the top\n"
    "        of the document. (Default is to point the link to top of the document.)\n"
    " --no_block\n"
    "        Do not remove the unnecessary blank line before the block classes\n"
    "        'format', 'display', 'example' and 'lisp'.  (deleted by default)\n"
    " --no_author\n"
    "        For a <blockquote> block followed by an author's name, do not adjust\n"
    "        the horizontal offset of the name. (adjusted by default)\n"
    " --no_cartouche\n"
    "        Allow text within a cartouche block to \"flow\" (dynamic linebreaks).\n"
    "        In the HTML document, text is preformatted by default (linebreaks \n"
    "        follow the source text).\n"
    " --no_contain\n"
    "        Do not insert the 'infodoc_container' class into the document.\n"
    "        This will cause text to wrap at the edge of the browser window.\n"
    "        (Default is to wrap text at the edge of the container.)\n"
    #if 0   // Include descriptions of debugging options.
    " ---- ---- ---- ---- ---- ---- ---- ---- ---- ---- ---- ---- ---- ---- ---- \n"
    "                             Debugging Options                              \n"
    " --scan[=START[,END]]\n"
    "     As lines of source data are read, print them to the display.\n"
    "     Starting and ending line number are optional. If not specified,\n"
    "     then all source lines are reported.\n"
    "     If only the starting line is specified, that line and all\n"
    "     subsequent lines will be reported.\n"
    "     If both start and end line numbers are specified (comma separated),\n"
    "     then all source lines in that range (inclusive) will be reported.\n"
    " --book\n"
    "     This option prints a pair of messages which \"bookend\" each text\n"
    "     block, list or other data structure. This helps to locate the point\n"
    "     in the source document where HTML syntax is being misinterpreted.\n"
    " ---- ---- ---- ---- ---- ---- ---- ---- ---- ---- ---- ---- ---- ---- ---- \n"
    #endif  // debugging options.
    " --version   Display version number and copyright notice.\n"
    " --help, -h  Command-line Help\n" 
    "        For detailed information, see Texinfo documentation:\n"
    "              info -f infodoc.info -n 'Infodoc Post-processor'\n"
   //123456789-123456789-123456789-123456789-123456789-123456789-123456789-123456789-
    << endl ;

}  //* End DisplayCommandLineHelp() *

//*************************
//*                       *
//*************************
//******************************************************************************
//*                                                                            *
//*                                                                            *
//*                                                                            *
//* Input  :                                                                   *
//*                                                                            *
//* Returns:                                                                   *
//******************************************************************************

