//******************************************************************************
//* File       : idpp.hpp                                                      *
//* Author     : Mahlon R. Smith                                               *
//*              Copyright (c) 2014-2015 Mahlon R. Smith, The Software Samurai *
//*                 GNU GPL copyright notice below                             *
//* Date       : 22-Feb-2015                                                   *
//* Version    : (see AppVersion string)                                       *
//*                                                                            *
//* Description: Definitions and data for Infodoc Post-processor (idpp),       *
//* an HTML post-processing utility for use with HTML documents generated from *
//* Texinfo source.                                                            *
//*                                                                            *
//*                                                                            *
//******************************************************************************
//* Copyright Notice:                                                          *
//* This program is free software: you can redistribute it and/or modify it    *
//* under the terms of the GNU General Public License as published by the Free *
//* Software Foundation, either version 3 of the License, or (at your option)  *
//* any later version.                                                         *
//*                                                                            *
//* This program is distributed in the hope that it will be useful, but        *
//* WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY *
//* or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License   *
//* for more details.                                                          *
//*                                                                            *
//* You should have received a copy of the GNU General Public License along    *
//* with this program.  If not, see <http://www.gnu.org/licenses/>.            *
//*                                                                            *
//*         Full text of the GPL License may be found in the TexInfo           *
//*         documentation for this program under 'Copyright Notice'.           *
//******************************************************************************

//****************
//* Header Files *
//****************
//* All necessary information for:                  *
//* NCurses, NcWindow, NcDialog and gString classes.*
#include "GlobalDef.hpp"
//* Access to file i/o and other system functions.  *
#include "idpp_file.hpp"


//***************
//* Definitions *
//***************

//* Holds captured command-line arguments *
//const USHORT maxCL_FILENAMES = 16 ;    // max number of source files per session
class commArgs
{
   public:
   commArgs ( int argc, char** argv, char** argenv ) :
               argCount(argc), argList(argv), envList(argenv)
   {
      this->reset() ;
   }
   void reset ( void )
   {
      this->verFlag = this->helpFlag = false ;
      this->textFlag = true ;
   }
   short    argCount ;     // command-line arguments
   char**   argList ;      // list of argument strings
   char**   envList ;      // pointer to terminal environment
   bool     textFlag ;     // 'true' if text-only, 'false' if interactive dialog
   bool     verFlag ;      // /true' if application version request (overrides all others)
   bool     helpFlag ;     // 'true' if command-line help (overrides all except verFlag)
} ;


//***********************************************
//* Constant Data:                              *
//***********************************************

//* Application version string. Keep it current! *
const wchar_t* const AppVersion = L"0.0.04" ;

const wchar_t CR = 0x000D ;   // carriage-return character
const wchar_t STAR = ('*') ;  // asterisk
const wchar_t SLASH = ('/') ; // forward slash
const wchar_t BSLASH = ('\\') ; // back slash
const wchar_t SGLQUOTE = ('\'') ;// single quote
const wchar_t DBLQUOTE = ('"') ;// double quotation mark
const wchar_t HASH = ('#') ;  // hash mark, pound/number sign

//* Application title *
const wchar_t* const AppTitle1 = L"Infodoc Post-processor v:" ;
const wchar_t* const AppTitle2 = L" (c)2014-2015 The Software Samurai" ;
const wchar_t* const AppTitle3 = // (this string == minTERMCOLS)
L"------------------------------------------------------------------------------" ;

const short sfMAX = 24 ;         // max number of source files to process
const short emMAX = 24 ;         // max number of error messages
const short CSS_VER_LEN = 32 ;   // length of CSS definition file version string
const short TARG_TEXT_LEN = 64 ; // size of upTargText[] member


//********************************
//* Application class definition *
//********************************
class Idpp
{
   public:
   virtual ~Idpp () ;               // destructor
   Idpp ( commArgs& ca ) ;          // constructor
   short ProcStatus ( void )        // returns 'OK' if all files processed successfully
   { return this->procStatus ; }


   private:

   //*** Methods for start-up, memory management ***
   //***-----------------------------------------***
   //* Interpret user's command options and gather specified source-filenames. *
   bool  GetCommandLineArgs ( commArgs& ca ) ;
   //* Display the application's title, version and copyright info.            *
   void  DisplayAppVersion ( void ) ;
   //* Display command-line options *
   void  DisplayCommandLineHelp ( void ) ;
   //* Process specified HTML documents in text-only mode.                     *
   short TextMode ( void ) ;
   //* Process specified HTML documents in interactive (dialog-based) mode.    *
   short InteractiveMode ( void ) ;
   //* Write text to the display                                               *
   void  textOut ( const gString& tOut, bool newln = true ) ;
   void  textOut ( const char*    tOut, bool newln = true ) ;
   void  textOut ( const wchar_t* tOut, bool newln = true ) ;
   //* Get user response to text-mode prompts.                                 *
   short userResponse ( gString& gsIn ) ;

   //*** Methods to read and modify files        ***
   //***-----------------------------------------***
   //* Convert raw HTML to CSS-styled HTML.                                    *
   short ppfProcessSrcHTML ( const gString& src, const gString& trg ) ;
   short ppfProcTOC ( bool tocFound ) ;
   short ppfProcINDEX ( void ) ;
   short ppfProcIDB ( const gString& gsbq ) ;
   short ppfProcQUOTE ( const gString& gsbq ) ;
   short ppfProcTAB ( const gString& gstab ) ;
   short ppfProcHEAD ( const gString& gsmeta ) ;
   short ppfProcUL ( gString& gst ) ;
   short ppfProcOL ( gString& gst ) ;
   bool  ppfTestUlistBegin ( const wchar_t* wptr ) ;
   bool  ppfTestOlistBegin ( const wchar_t* wptr ) ;
   bool  ppfTestFormattedBlock ( wchar_t& bType, const wchar_t* wptr ) ; 
   short ppfProcFormattedBlock ( wchar_t bType, gString& gsb ) ;
   void  ppfPFB_List ( gString& gsLine ) ;
   short ppfProcBlockWhitespace ( wchar_t bType, gString& gsb ) ;
   short ppfProcGNU ( bool gpl ) ;
   bool  ppfEndBlock ( const gString& gsLine, const wchar_t* etag, short elen ) ;
   bool  ppfInsertCustomData ( const gString& fileSpec ) ;
   short ppfReadSrcLine ( gString& gs, const wchar_t*& wptr, short& windex ) ;
   short ppfReadLine ( ifstream& ifs, gString& gs ) ;
   short ppfWriteLine ( const gString& gsOut ) ;

   //* Test whether the specified file (or directory) exists.                  *
   bool  ppfTargetExists ( const gString& fPath, bool isDir = false ) ;
   //* Test whether the specified file is a valid HTML document.               *
   bool  ppfTargetIsHTML ( const gString& fPath ) ;
   //* Test whether the specified CSS definition file is valid.                *
   bool  ppfTargetIsCSS ( const gString& fPath, gString& gsVer ) ;
   //* Scan the target directory for HTML source documents.                    *
   bool  ppfScan4Src ( void ) ;

   //* Get the user's current working directory.                               *
   short ppfGetCWD ( gString& dPath ) ;
   //* Concatenate path and filename strings to create path/filename spec.     *
   short ppfCatPathFilename ( gString& pgs, const gString& wPath, const char* uFile ) ;
   short ppfCatPathFilename ( gString& pgs, const gString& wPath, const wchar_t* wFile ) ;
   //* Change the current working directory to specified target path.          *
   short ppfCdTarget ( const char* dirPath ) ;
   //* Delete a file.                                                          *
   short ppfDeleteFile ( const gString& trgPath ) ;
   //* Rename a file.                                                          *
   short ppfRenameFile ( const gString& srcPath, const gString& trgPath ) ;
   //* Decode the full path/filename specification for target.                 *
   short ppfRealpath ( gString& realPath, const gString& rawPath ) ;
   //* Extract the source filename from the path/filename string.              *
   void  ppfExtractFilename ( gString& gsName, const gString& fPath ) ;
   //* Extract the filename extension (if any) from the filename.              *
   void  ppfExtractFileExtension ( gString& gsExt, const char* fName ) ;

   gString  cwDir ;        //* Current-working-directory on start-up
   gString  cssFile ;      //* Path/filename for CSS definitions
   gString  userMeta ;     //* Path/filename for User Text to be inserted into document head section
   wchar_t  cssVersion[CSS_VER_LEN] ;  //* CSS definition file version
   wchar_t  srcFiles[sfMAX][gsMAXCHARS] ; //* List of source filenames
   short    sfCount ;      //* Number of source files to process
   wchar_t  ErrorMsg[emMAX][gsMAXCHARS] ; //* Formatted error message strings
   short    emCount ;      //* Number of error messages in queue
   USHORT   slCount ;      //* Line number of source file being processed (in case of error)
   USHORT   tlCount ;      //* Line number of target file being written (verbose diagnostics)
   wchar_t  upTargPath[gsMAXCHARS] ; //* path to insert in "up target" link
   wchar_t  upTargText[TARG_TEXT_LEN] ; //* display text to insert for "up target" link
   short    procStatus ;   //* OK if all files processes successfully, else ERR
   ifstream ifs ;          //* Access to HTML source file
   ofstream ofs ;          //* Access to HTML target file
   bool     textMode ;     //* 'true' if TextMode, 'false' if InteractiveMode
   bool     allFiles ;     // 'true' if processing all source files in target directory
   bool     oLists ;       // 'true' if modifying Ordered lists
   bool     oOffst ;       // 'true' if modifying Ordered lists AND prompt user for sequence offset
   bool     tocMod ;       // 'true' if converting Table of Contents to bullet list
   bool     tocDel ;       // 'true' if removing Table of Contents from document
   bool     upTarg ;       // 'true' if user has specified an alternate "up" target (upTargPath)
   bool     tabBorder ;    // 'true' if user has specified to enclose tables within borders
   bool     tabPrompt ;    // 'true' if we are to prompt user to make a decision on table borders
   bool     verbose ;      // 'true' if 'verbose output' (applies only to text mode)
   bool     css_mod ;      // 'true' if user wants to adjust the CSS definition file (CURRENTLY UNREFERENCED)
   bool     my_meta ;      // 'true' if user specifed a file containing data to be inserted into head section
   bool     no_mods ;      // 'true' if scan only, no modifications to be performed
   bool     no_doct ;      // 'true' if <!DOCTYPE> tag not to be processed
   bool     no_utrg ;      // 'true' if retaining original header "Up" target
   bool     no_body ;      // 'true' if <body> tag are not to be processed
   bool     no_bull ;      // 'true' if <ul> tags are not to be processed
   bool     no_bloc ;      // 'true' if unnecessary leading blank lines not to be removed
   bool     no_auth ;      // 'true' if @author sub-command's output not to be adjusted
   bool     no_meta ;      // 'true' if retaining meta-data in <head>
   bool     no_link ;      // 'true' if retaining links in <head>
   bool     no_cont ;      // 'true' if container class is not to be inserted
   bool     no_cart ;      // 'true' if cartouche callout is not to be modified

   //* For debugging only - display the source lines as they are read *
   //* and/or bookend the blocks (report begin and end).              *
   USHORT   scan_beg, scan_end ;
   bool     scan ;
   bool     book ;
} ;

