//******************************************************************************
//* File       : gString.hpp                                                   *
//* Author     : Mahlon R. Smith                                               *
//*              Copyright (c) 2011-2014 Mahlon R. Smith, The Software Samurai *
//*                 GNU GPL copyright notice below                             *
//* Date       : 12-Oct-2014                                                   *
//* Version    : (see gStringVersion string in gString.cpp)                    *
//*                                                                            *
//* Description: This class is used to convert between UTF-8 character         *
//* encoding and wchar_t character encoding. Statistical information for       *
//* analysis and display of the data is generated.                             *
//*                                                                            *
//* This is not a comprehensive string class, but a highly-specialized,        *
//* single-purpose string class for text encoding conversions. The gString     *
//* class was developed to support the NcDialog class, NCurses class, NcWindow *
//* classes and their descendants, and is a compromise between the full        *
//* functionality of Glib::ustring and the need to keep the NCurses, NcWindow, *
//* and NcDialog classes independent from third-party libraries.               *
//*                                                                            *
//* The gString class is a stand-alone class, that is, it requires no third-   *
//* party source or libraries beyond the standard C and C++ libraries found in *
//* all GNU compiler releases. Note however that the flag '_XOPEN_SOURCE'      *
//* must be defined (see below).                                               *
//*                                                                            *
//* Dynamic memory allocation is not performed within the gString class, and   *
//* The capacity of gString (see gsMAXCHARS and gsMAXBYTES) was chosen to      *
//* accomodate the maximum length of a path/filename string,                   *
//* i.e. 'PATH_MAX' (4096 bytes) as defined in the POSIX standard and found    *
//* in 'limits.h'.                                                             *
//*                                                                            *
//* This class is relatively fast and is useful for text data analysis and     *
//* manipulation, but is rather inefficient for data storage (although perhaps *
//* no more inefficient than std::string()).                                   *
//*                                                                            *
//* Developed using GNU G++ (Gcc v: 4.4.2 through 4.8.0)                       *
//*  under Fedora Release 12, Kernel Linux 2.6.32.26-175.fc12.i686.PAE         *
//*         through                                                            *
//*        Fedora Release 16, Kernel Linux 3.6.11-4.fc16.i686.PAE              *
//*  GNOME terminal 2.28.2 through 3.2.1                                       *
//*                                                                            *
//*                                                                            *
//* Copyright Notice:                                                          *
//* -----------------                                                          *
//* This program is free software: you can redistribute it and/or modify it    *
//* under the terms of the GNU General Public License as published by the Free *
//* Software Foundation, either version 3 of the License, or (at your option)  *
//* any later version, PROVIDED THAT the copyright notices for both code and   *
//* documentation are included and unmodified.                                 *
//*                                                                            *
//* This program is distributed in the hope that it will be useful, but        *
//* WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY *
//* or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License   *
//* for more details.                                                          *
//*                                                                            *
//* You should have received a copy of the GNU General Public License along    *
//* with this program.  If not, see <http://www.gnu.org/licenses/>.            *
//*                                                                            *
//*         Full text of the GPL License may be found in the TexInfo           *
//*         documentation for this program under 'Copyright Notice'.           *
//******************************************************************************
//* Version History (most recent first):                                       *
//*   See version history in gString.cpp.                                      *
//*                                                                            *
//******************************************************************************
//* Programmer's Notes:                                                        *
//* 1) To assign data from a std::string class object to a gString object,     *
//*    use one of the following:                                               *
//*      std::string st("Hello World!") ;                                      *
//*      gString gs(st.c_str()) ;                                              *
//*      OR                                                                    *
//*      gString gs ;                                                          *
//*      gs = st.c_str() ;                                                     *
//*                                                                            *
//* 2) To assign data from a Glib::ustring class object to a gString object,   *
//*    use one of the following:                                               *
//*      Glib::ustring ust("Good Morning!") ;                                  *
//*      gString gs(ust.c_str()) ;                                             *
//*      OR                                                                    *
//*      gString gs ;                                                          *
//*      gs = ust.c_str() ;                                                    *
//*                                                                            *
//******************************************************************************

#ifndef GSTRING_INCLUDED
#define GSTRING_INCLUDED

//* _XOPEN_SOURCE must be defined for including cwchar header file *
#ifndef _XOPEN_SOURCE
#define _XOPEN_SOURCE
#endif   // _XOPEN_SOURCE

//*****************
//* Include Files *
//*****************
#include <cstdlib>      //* Misc. functionality including character conversion
#include <cstdarg>      //* Variadic-function macro definitions
#include <cwchar>       //* For wcwidth function

//** For Debugging Only - Disable In Production **
#define ENABLE_GS_DEBUG (0)

const short gsMAXCHARS  = (1025) ;  // maximum number of converted characters
const short gsMAXBYTES  = (gsMAXCHARS * 4) ; // max bytes of source data scanned

//* Max field-width parameter for the formatInteger() methods *
const short FI_MAX_FIELDWIDTH = (13) ;


//* Use the gsForm class with the gString constructor or operator= to create   *
//* a formatted string.                                                        *
//* Example:                                                                   *
//* char utfString[] = { "Hello! " } ;   short sVal = 21 ;   int iVal = 523 ;  *
//* gsForm gsf { L"%s: %hd %d", { utfString, &sVal, &iVal } } ;                *
//*    gString gs ;                                                            *
//*    gs = gsf ;                                                              *
//*       OR                                                                   *
//*    gString gs( gsf ) ;                                                     *
//* See also the description of the compose() method below.                    *
const short gsFormMAXARGS = 24 ;    // Maximum number of conversion parameters
class gsForm
{
   public:
   //* Formatting string that specifies data type and format for each of       *
   //* the parameters in argList.                                              *
   const wchar_t* fmtSpec ;

   //* Pointers to each value to be converted                                  *
   const void*    argList[gsFormMAXARGS] ;

} ;


//******************************
//** Define the gString class **
//******************************
class gString
{
public:
virtual ~gString() ;             // destructor

//* Constructor: Initialize members to default values (NULL string).           *
//*                                                                            *
//* Input  : none                                                              *
//* Returns: nothing                                                           *
gString ( void ) ;

//* Constructor: Convert specified UTF-8-encoded source to gString.            *
//*                                                                            *
//* Input  : usrc  : pointer to a UTF-8-encoded, null-terminated string        *
//*          charLimit: (optional, gsMAXCHARS by default)                      *
//*                  maximum number of characters from source array to convert *
//* Returns: nothing                                                           *
gString ( const char* usrc, short charLimit=gsMAXCHARS ) ;

//* Constructor: Convert specified wchar_t ('wide') source to gString.         *
//*                                                                            *
//* Input  : wsrc  : pointer to a wchar_t-encoded, null-terminated string      *
//*          charLimit: (optional, gsMAXCHARS by default)                      *
//*                  maximum number of characters from source array to convert *
//* Returns: nothing                                                           *
gString ( const wchar_t* wsrc, short charLimit=gsMAXCHARS ) ;

//* Constructor: Convert formatting specification and its arguments to gString.*
//*              Please refer to compose() method for more information.        *
//*                                                                            *
//* Input  : fmt  : a format specification string in the style of sprintf(),   *
//*                 swprintf() and related formatting C/C++ functions.         *
//*          arg1 : pointer to first value to be converted by 'fmt'            *
//*          ...  : optional arguments (between ZERO and gsfMAXARGS - 1)       *
//*                 Each optional argument is a POINTER (address of) the value *
//*                 to be formatted.                                           *
//*                                                                            *
//* Returns: nothing                                                           *
gString ( const char* fmt, const void* arg1, ... ) 
          __attribute__ ((format (gnu_printf, 2, 0))) ;

//* Constructor: Convert formatting instructions in gsForm class object        *
//*              to gString. See compose() for more information.               *
//*                                                                            *
//* Input  : gsf   : initialized gsForm class object containing parameters for *
//*                  creating a formatted text string.                         *
//*          charLimit: (optional, gsMAXCHARS by default)                      *
//*                  maximum number of characters from source array to convert *
//* Returns: nothing                                                           *
gString ( const gsForm& gsf, short charLimit=gsMAXCHARS ) ;

//* Assignment operator: converts UTF-8-encoded source to gString.             *
void operator = ( const char* usrc ) ;

//* Assignment operator: converts wchar_t ('wide') source to gString.          *
void operator = ( const wchar_t* wsrc ) ;

//* Assignment operator: Converts gsForm-class instructions to gString.        *
void operator = ( const gsForm& gsf ) ;

//* Assignment operator. Copies one gString object to another.       .         *
void operator = ( const gString& gssrc ) ;

//* Create formatted text data from a format specification string including    *
//* between ZERO and gsfMAXARGS format specifications and their corresponding  *
//* argument pointers..                                                        *
//*                                                                            *
//* Supported data types:                                                      *
//*  %d, %i  integer (decimal)                                                 *
//*  %o      integer (octal)                                                   *
//*  %u      integer (unsigned)                                                *
//*  %x, %X  integer (hex lower or upper case)                                 *
//*  %f      floating point (fixed point)                                      *
//*  %e, %E  floating point (scientific notation, lower/uppercase)             *
//*  %g, %G  floating point (normal/exponential, lower/uppercase)              *
//*  %a, %A  floating point (hex fraction)                                     *
//*  %c      character                                                         *
//*  %C      character (alias for %lc)                                         *
//*  %s      string                                                            *
//*  %S      string (alias for %ls)                                            *
//*  %p      pointer                                                           *
//*  %b, %B  (extension to swprintf - see description below)                   *
//*  %m      capture 'errno' description string (see /usr/include/errno.h)     *
//*  %n      number of characters printed so far                               *
//*          (value written to corresponding argument's location)              *
//*  %%      literal '%'                                                       *
//*                                                                            *
//* See man pages for the C/C++ function 'swprintf' or                         *
//* 'Table of Output Conversions' for additional details.                      *
//* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -  *
//*                                                                            *
//* Input  : fmt  : a format specification string in the style of sprintf(),   *
//*                 swprintf() and related formatting C/C++ functions.         *
//*          ...  : optional arguments (between ZERO and gsfMAXARGS)           *
//*                 Each optional argument is a POINTER (address of) the value *
//*                 to be formatted.                                           *
//*                 - Important Note: There must be AT LEAST as many optional  *
//*                   arguments as the number of format specifiers defined in  *
//*                   the formatting string. Excess arguments will be ignored; *
//*                   HOWEVER, too few arguments will result in an application *
//*                   crash and ridicule from your peers.                      *
//*                                                                            *
//* Returns: const wchar_t* to formatted data                                  *
//*                                                                            *
//* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -  *
//* Implementation Notes:                                                      *
//* =====================                                                      *
//*                                                                            *
//* 1) Unsupported data types                                                  *
//*    ======================                                                  *
//*    Conversion modifiers that are not fully supported at this time:         *
//*                       'j', 'z', 't', '%['                                  *
//*    Note: The '*' field-width specification or precision specification      *
//*          which uses the following argument as the width/precision value    *
//*          IS NOT supported.                                                 *
//*                                                                            *
//* 2) Binary-format specifier                                                 *
//*    =======================                                                 *
//*    We implement an extension to the swprintf output-conversion-specifiers  *
//*    for binary formatted output. We have found this formatting option useful*
//*    when working with bit masks and for verifying bit-shifting operations.  *
//*    - Base formatting specifier:                                            *
//*       %b , %B                                                              *
//*    - Format modifiers for DATA SIZE are the same as for swprintf:          *
//*       hh , h , l , ll , L , q                                              *
//*    - Format modifier for prepending of a type indicator.                   *
//*       '#' (hash character)                                                 *
//*       This is the same principle as for prepending a '0x' indicator to hex *
//*       output, and will place either a 'b' or 'B' character at the beginning*
//*       of the output. Examples:  %#hhb -> b0101.1010   %#hhB -> B0101.1010  *
//*    - Format modifier for appending of a type indicator.                    *
//*       '-#' (minus sign and hash character)                                 *
//*       Rather than prepending the indicator, the indicator will be append   *
//*       to the end of the output.                                            *
//*       Examples: %-#hhb -> 0101.1010b   %-#hhB -> 0101.1010B                *
//*    - Format modifier for specifying the group-seperator character.         *
//*       By default, the bit groups are seperated by a '.' (fullstop)         *
//*       character. To specify an alternate seperator character:              *
//*       % hB -> 0111 0101 1010 0001    (' ' (space) character as seperator)  *
//*       %_hB -> 0111_0101_1010_0001    ('_' (underscore) char as seperator)  *
//*       %#/hB -> B0111/0101/1010/0001  ('/' (slash) character as seperator)  *
//*       %-#-hB -> 0111-0101-1010-0001B ('-' (dash) character as seperator)   *
//*       Valid seperator characters are any printable ASCII character         *
//*       that IS NOT alphabetical, IS NOT a number, and IS NOT a '.'(fullstop)*
//*    - Format modifier for specifying bit grouping.                          *
//*       By default, bits are formatted in groups of four (4 nybble); however,*
//*       if desired, bits can be formatted in groups of eight (8 byte):       *
//*       %.8hB -> 01110101.10100001                                           *
//*       %-.8hB -> 01110101-10100001                                          *
//*       %# .8hB -> B01110101 10100001                                        *
//*       %-#`.8hb -> 01110101`10100001b                                       *
//*                                                                            *
//* 3) Format-specification Deficiencies                                       *
//*    =================================                                       *
//*    The compiler will scan your UTF-8 formatting string for valid           *
//*    formatting specifications and warn you of potential problems.           *
//*      HOWEVER:                                                              *
//*    The ANSI-C/C++ specification does not yet support parsing of wchar_t    *
//*    (swprintf-style) formatting strings. If you pass a wchar_t formatting   *
//*    string, the compiler will not be able to warn of potential problems,    *
//*    so construct your formatting string carefully. See gcc documentation    *
//*    for more information:  info -f gcc.info -n 'Function Attributes'        *
//*      ADDITIONALLY:                                                         *
//*    Because the optional parameters are passed as POINTERS, similar to the  *
//*    C/C++ library function 'sscanf' and friends, the compiler cannot perform*
//*    automatic promotions from short int* to int* or from float* to double*, *
//*    and so-on. For this reason, please use care that the data type you      *
//*    specify in the formatting string matches the data type of the variable  *
//*    referenced by its parameter pointer. For example, the conversion        *
//*    specifications:  %d, %i, %u, %x, and %X  assume a parameter of 'int'    *
//*    size If your actual variable declaration is for a 8-bit int, short int, *
//*    long int or long long int, then explicitly indicate this in the         *
//*    corresponding conversion specification.                                 *
//*    Examples:                                                               *
//*      char      Greeting[] = { "Hello!" } ;                                 *
//*      int       iValue ;                                                    *
//*      long long int qValue ;                                                *
//*      long int  lValue ;                                                    *
//*      short int sValue1, sValue2, sValue3 ;                                 *
//*      bool      flagValue ;                                                 *
//*      float     fltValue ;                                                  *
//*      double    dblValue ;                                                  *
//*      gString gs ;                                                          *
//*      gs.compose( "%s - %d %12hd, %-hi, %#hx %08lX %llu %hhd",              *
//*                  Greeting, &iValue, &sValue1, &sValue2, &sValue3,          *
//*                  &lValue, &qValue, &flagValue );                           *
//*      gs.compose( "floating downstream:%10.2f and doubling our pace:%.4lf", *
//*                  &fltValue, &dblValue ) ;                                  *
//*                                                                            *
//*      Note: Use care when formatting an 8-bit integer: 'char' ,             *
//*            'unsigned char' or a boolean value declared with the 'bool'     *
//*            keyword. For these types, use the 'hh' conversion modifier.     *
//*            Examples: %hhd , %hhu , %hhX                                    *
//*                                                                            *
//* 4) 'swprintf' bug-fix                                                      *
//*    ==================                                                      *
//*    The standard library 'swprintf' function has a design flaw for format   *
//*    specifications that include a field-width specifier.                    *
//*    'swprintf' pads the string to the specified number of CHARACTERS, not   *
//*    the number of COLUMNS as it should do. For (Arabic) numeric source      *
//*    values this is not a problem because one character == one display       *
//*    column. For string source data, however, if the source string contains  *
//*    characters that require more than one display column each, then the     *
//*    output may be too wide.                                                 *
//*    Therefore, for string-source-formatting specifications ONLY             *
//*          (examples: "%12s"  "%-6s"  "%16ls"  "%5S"  "%-24S")               *
//*    we compensate for this anomalous behavior by interpreting the           *
//*    field-width specifier as number-of-columns, NOT number-of-characters.   *
//*    This will result in output that appears different (and better) than     *
//*    output created directly by the 'swprintf' function.                     *
//*                                                                            *
//* 5) Parameter type checking                                                 *
//*    =======================                                                 *
//*    Unfortunately, type-checking of wchar_t formatting strings is not yet   *
//*    supported by the gnu (v:4.8.0) compiler, (but see wchar.h which is      *
//*    preparing for future expansion). Thus, use care when constructing your  *
//*    'wchar_t fmt' formatting string. The 'char fmt' string is type-checked. *
//* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -  *
const wchar_t* compose ( const wchar_t* fmt, ... ) 
                         /*__attribute__ ((format (gnu_wprintf, 2, 0)))*/ ;
const wchar_t* compose ( const char* fmt, ... ) 
                         __attribute__ ((format (gnu_printf, 2, 0))) ;

//* Convert an integer value into a formatted display string of the specified  *
//* width. Value is right-justified in the field, with leading spaces added    *
//* if necessary (but see 'lJust' parameter).                                  *
//* Maximum field width is FI_MAX_FIELDWIDTH. This is wide enough to display   *
//* 10 decimal digits in comma-formatted form: '1,234,567,890'                 *
//*                                                                            *
//* Actual formatting of the value depends on the combination of value size    *
//* and the specified field-width as well as the value of the LC_NUMERIC locale*
//* environment variable. Examples are for the 'C' and U.S. English locale.    *
//* 1) Simple comma formatted output if specified field-width is sufficient.   *
//*    Examples:  345    654,345    782,654,345    4,294,967,295               *
//* 2) Compression of the value to fit a specified field width.                *
//*    Examples:  12.3K    999K    12.345M    1.234G    4.3G                   *
//*                                                                            *
//* Note on 'K', 'M', 'G' and 'T' suffixes: Because we are working with        *
//* decimal (base-10) values, compressed values are expressed as:              *
//*          kilobytes (x/1000)           NOT kibibytes (x/1024),              *
//*          megabytes (x/1000000)        NOT mibibytes (x/1024000)            *
//*          gigabytes (x/1000000000),    NOT gibibytes (x/1024000000).        *
//*          terabytes (x/1000000000000), NOT tibibytes (x/1024000000000).     *
//* Please see NcDialog test application, 'Dialogw' for more examples.)        *
//*                                                                            *
//* Input  : iVal  : value to be formatted                                     *
//*                  Max supported value: 9.9999999999 terabytes==9.9999999999T*
//*          fWidth: field width (number of display columns)                   *
//*                  range: 1 to FI_MAX_FIELDWIDTH                             *
//*          lJust : (optional, false by default)                              *
//*                  if true, strip leading spaces to left-justify the value   *
//*                  in the field. (resulting string may be less than fWidth)  *
//*          sgnVal: (optional, false by default) [CURRENTLY IGNORED]          *
//*                  if true, then interpret 'iVal' as a signed number         *
//*          pnSign: (optional, false by default) [CURRENTLY IGNORED]          *
//*                  if 'sgnVal' == true AND pnSign == true, then prepend a    *
//*                  '+' or '-' sign to the value. The sign reduces precision  *
//*                  by one digit                                              *
//* Returns: 'true' if successful                                              *
//*          'false' if field overflow (field will be filled with '#' chars)   *
//*                     note that oveflow can occur ONLY:                      *
//*                     if fWidth <= 3 OR if iVal >= 10. terabytes             *
bool  formatInteger ( unsigned short int iVal, short fWidth, bool lJust = false, 
                      bool sgnVal = false, bool pnSign = false ) ;
bool  formatInteger ( unsigned int iVal, short fWidth, bool lJust = false, 
                      bool sgnVal = false, bool pnSign = false ) ;
bool  formatInteger ( unsigned long iVal, short fWidth, bool lJust = false, 
                      bool sgnVal = false, bool pnSign = false ) ;
bool  formatInteger ( unsigned long long iVal, short fWidth, bool lJust = false, 
                      bool sgnVal = false, bool pnSign = false ) ;

//* Returns a const pointer to the wchar_t array AND optionally, the number    *
//* of characters in the array (incl. null terminator).                        *
const wchar_t* gstr ( void ) const { return this->gStr ; }
const wchar_t* gstr ( short& charCount ) const ;

//* Returns a const pointer to the UTF-8-encoded char array AND optionally,    *
//* the number of characters in the array (incl. null terminator) and the      *
//* number of bytes in the array (incl. null terminator).                      *
const char* ustr ( void ) const { return this->uStr ; } ;
const char* ustr ( short& charCount, short& byteCount ) const ;

//* Comparison operator: Compares the text content of two gString objects.     *
//* Actual comparison is done on the objects' wchar_t character arrays.        *
//* Returns: 'true' if the strings are identical, else 'false'                 *
bool operator == ( const gString& gs2 ) const ;

//* Comparison operator: Compares the text content of two gString objects.     *
//* Actual comparison is done on the objects' wchar_t character arrays.        *
//* Returns: 'true' if the strings are different, else 'false'                 *
bool operator != ( const gString& gs2 ) const ;

//* Compares the text content of the gString object with the specified text.   *
//* For both, actual comparison is done on  wchar_t character arrays.          *
//* Input  : utf2 (UTF-8 string) OR wch2 (wchar_t string) to be compared       *
//*          charLimit: (optional, gsMAXCHARS by default)                      *
//*                     maximum number of characters to compare                *
//*          offset   : (optional, ZERO by default)                            *
//*                     If specified, equals the character offset into the     *
//*                     gString character array at which to begin comparison.  *
//* Returns: using the rules of the 'wcsncmp' library function (see string.h): *
//*     ZERO, text data are identical                                          *
//*   > ZERO, first differing char of gString object is numerically larger.    *
//*   < ZERO, first differing char of gString object is numerically smaller.   *
short compare ( const char* utf2, short limit = gsMAXCHARS, short offset = 0 ) const ;
short compare ( const wchar_t* wch2, short limit = gsMAXCHARS, short offset = 0 ) const ;

//* Copy gString text to specified target buffer. (source data unchanged)      *
//*                                                                            *
//* Input  : uTarget : pointer to target array to receive UTF-8-encoded text   *
//*          maxBytes: maximum number of bytes to copy (incl. NULL terminator) *
//*          maxCols : (optional, default == gsMAXCHARS*2)                     *
//*                    maximum number of display-columns to copy               *
//* Returns: number of bytes copied (incl. NULL terminator)                    *
short copy ( char* uTarget, short maxBytes, short maxCols=(gsMAXCHARS*2) ) const ;

//* Copy gString text to specified target buffer. (source data unchanged)      *
//*                                                                            *
//* Input  : wTarget : pointer to target array to receive wchar_t 'wide' text  *
//*          maxChars: maximum number of characters to copy (incl. NULL)       *
//*          maxCols : (optional, default == gsMAXCHARS*2)                     *
//*                    maximum number of display-columns to copy               *
//* Returns: number of characters copied (incl. NULL terminator)               *
short copy ( wchar_t* wTarget, short maxChars, short maxCols=(gsMAXCHARS*2) ) const ;

//* Append text to existing gString text data up to a combined length of       *
//* gsMAXCHARS. Characters in excess of the maximum will not be appended.      *
//*                                                                            *
//* Input  : wPtr: pointer to array of wchar_t 'wide' text to be appended      *
//*           OR                                                               *
//*          uPtr: pointer to array of char UTF-8 text to be appended          *
//*           OR                                                               *
//*          wChar: a single, 'wide' character                                 *
//* Returns: number of characters in resulting string (incl. NULL terminator)  *
//*          Note: if return == gsMAXCHARS, some data MAY HAVE BEEN discarded. *
short append ( const wchar_t* wPtr ) ;
short append ( const char* uPtr ) ;
short append ( const wchar_t wChar ) ;

//* Insert text into existing gString text data up to a combined length of     *
//* gsMAXCHARS. Characters in excess of the maximum will be truncated.         *
//*                                                                            *
//* Input  : wPtr  : pointer to array of wchar_t 'wide' text to be inserted    *
//*           OR                                                               *
//*          uPtr  : pointer to array of char UTF-8 text to be inserted        *
//*           OR                                                               *
//*          wChar : a single wchar_t 'wide' character                         *
//*          offset: (optional, ZERO by default)                               *
//*                  character offset at which to insert specified text into   *
//*                  existing text.                                            *
//*                  Note: if specified 'offset' > number of characters in     *
//*                        existing text, then acts like 'append' method.      *
//* Returns: number of characters in resulting string (incl. NULL terminator)  *
//*          Note: if return == gsMAXCHARS, some data MAY HAVE BEEN discarded. *
short insert ( const wchar_t* wPtr, short offset = 0 ) ;
short insert ( const char* uPtr, short offset = 0 ) ;
short insert ( wchar_t wChar, short offset = 0 ) ;

//* Shift text data by the specified number of characters.                     *
//*                                                                            *
//* Input  : shiftCount: < ZERO: shift data to the left, discarding the        *
//*                              specified number of characters from the       *
//*                              beginning of the array                        *
//*                      > ZERO: shift data to the right, padding the vacated  *
//*                              positions on the left with 'padChar'          *
//*                      ==ZERO: do nothing                                    *
//*          padChar   : (optional, SPACE character, 0x20 by default)          *
//*                      when shifting data to the right, use this character   *
//*                      to fill the vacated character positions               *
//* Returns: number of characters in adjusted array                            *
short shiftChars ( short shiftCount, wchar_t padChar = L' ' ) ;

//* Shift text data by the specified number of display columns.                *
//*                                                                            *
//* Input  : shiftCount: < ZERO: shift data to the left, discarding the        *
//*                              number of characters equivalent to the        *
//*                              specified number of display columns           *
//*                              NOTE: May discard one extra column if count   *
//*                              falls within a multi-column character.        *
//*                      > ZERO: shift data to the right, padding the vacated  *
//*                              positions on the left with 'padChar'          *
//*                      ==ZERO: do nothing                                    *
//*          padChar   : (optional, SPACE character, 0x20 by default)          *
//*                      when shifting data to the right, use this character   *
//*                      to fill the vacated column positions                  *
//*                      NOTE: Specify a one-column character ONLY as the      *
//*                      padding character. (multi-column characters ignored)  *
//* Returns: number of columns in adjusted array                               *
short shiftCols ( short shiftCount, wchar_t padChar = L' ' ) ;

//* Returns the number of characters in the string including null terminator   *
short gschars ( void ) const ;

//* Returns the number of columns required to display the string.              *
short gscols ( void ) const ;

//* Returns a pointer to an array of column counts, one for each character     *
//* of the text data. charCount gets number of characters in display string.  .*
const short* gscols ( short& charCount ) const ;

//* Returns the number of bytes in the UTF-8-encoded string (incl. null)       *
short utfbytes ( void ) const ;

//* Returns 'true' if data are pure 7-bit ASCII, 'false' otherwise.            *
bool  isASCII ( void ) ;

//* Truncate the data at no more than charCount characters.                    *
//* Insert a null terminator after the specified number of characters          *
//* (not including the null). Range: 1 to gsMAXCHARS-1,                        *
//* Returns: number of characters in the adjusted data (including null)        *
short limitCharacters ( short charCount ) ;

//* Truncate the data at no more than colCount display columns.                *
//* Insert a null terminator after the number of characters required to fill   *
//* the specified number of display columns. Range: 1 to (gsMAXCHARS * 2).     *
//* Returns: number of columns needed to display the adjusted data             *
short limitColumns ( short colCount ) ;

//* Reset contents to an empty string i.e. "". The data will consist of a      *
//* single, NULLCHAR character. The character and byte counts are set to       *
//* 1 (one), and the column count is zero.                                     *
void  clear ( void ) ;

//* Returns a pointer to version number string for the gString class.          *
const char* Get_gString_Version ( void ) const ;

#if ENABLE_GS_DEBUG != 0
void dbMsg ( gString& gsmsg ) ;  // for debugging only
#endif   // ENABLE_GS_DEBUG


private:
void  ConvertUTF8toWide ( const char* usrc, short charLimit=gsMAXCHARS ) ;
void  Bytewise_u2w ( const char* usrc, short charLimit ) ;
void  ConvertWidetoUTF8 ( const wchar_t* wsrc=NULL, short charLimit=gsMAXCHARS ) ;
void  Bytewise_w2u ( short charLimit ) ;
void  Convert_gsForm2gs ( const gsForm& gsf, wchar_t wsrc[gsMAXCHARS] ) ;
void  limitBytes ( short maxBytes ) ;
void  ColumnCount ( short maxCols=(gsMAXCHARS * 2) ) ;
void  Reinit ( bool initAll=true ) ;
void  binaryFormat ( wchar_t* convBuff, wchar_t convType, wchar_t sepType, 
                     long long int intData, short dataType, short grpType, 
                     bool append ) ;
bool  intFormat32 ( wchar_t* usrc, unsigned long iVal, short fWidth, 
                    bool lJust, bool sgnVal, bool pnSign ) ;
bool  intFormat64 ( wchar_t* usrc, unsigned long long iVal, short fWidth, 
                    bool lJust, bool sgnVal, bool pnSign ) ;

short composeFieldwidthBugFix ( char convType, wchar_t* wsrcPtr, short wsrcSpace, 
                                const wchar_t* fmt, gString& gsSrc ) ;

wchar_t  gStr[gsMAXCHARS] ;   // text data in wchar_t 'wide' format
char     uStr[gsMAXBYTES] ;   // text data in UTF-8 encoded format
short    cWidth[gsMAXCHARS] ; // column width for each character in gStr
short    gsCols ;             // number of columns needed to display gStr
short    gsChars ;            // number of characters represented (incl. null)
short    utfBytes ;           // number of bytes in UTF-8 data (incl. null)
bool     isAscii ;            // 'true' if data are pure, 7-bit ASCII, else 'false'
// Note that isAscii is unitialized by default and is valid only after 
// a call to the isASCII() method
} ;

#endif   // GSTRING_INCLUDED


