//********************************************************************************
//* File       : FmMenu.cpp                                                      *
//* Author     : Mahlon R. Smith                                                 *
//*              Copyright (c) 2005-2024 Mahlon R. Smith, The Software Samurai   *
//*                  GNU GPL copyright notice located in FileMangler.hpp         *
//* Date       : 06-Dec-2020                                                     *
//* Version    : (see AppVersion string)                                         *
//*                                                                              *
//* Description: This is a support module for the FileMangler application.       *
//* This module contains the methods related to menu system access and           *
//* executing commands which user has selected from the menu system.             *
//*                                                                              *
//* Development Tools: see notes in FileMangler.cpp.                             *
//********************************************************************************
//* Programmer's Notes:                                                          *
//*                                                                              *
//*                                                                              *
//********************************************************************************


//** Header Files **
#include "FileMangler.hpp"    // FileMangler definitions and data

//** Local Data **
//* Color attributes for the menu items of cmVfileData, the 'View File' menu.  *
//* See cmVfileActiveFT and cmVfileInactiveFT in FmMenu.hpp.                   *
attr_t cmVfileAcolorFT[cmVfileDATA_ITEMS] = 
{ ZERO, ZERO, ZERO, ZERO, ZERO } ;
attr_t cmVfileIcolorFT[cmVfileDATA_ITEMS] = 
{ ZERO, ZERO, ZERO, ZERO, ZERO } ;


//*************************
//*     MenuBarAccess     *
//*************************
//******************************************************************************
//* This method implements user access to all the menus and sub-menus of the   *
//* Menu Bar defined and instantiated in the application.                      *
//*                                                                            *
//* Considerable functionality, as well as context menus, may be accessed      *
//* directly through hotkeys, but the menu system provides a sure, global      *
//* access to all commands.                                                    *
//*                                                                            *
//* Input  : mIndex: (optional, -1 by default)                                 *
//*                  If not specified, then the menu with lowest index is      *
//*                   initially expanded.                                      *
//*                  If valid menu index specified, then the indexed menu is   *
//*                   initially expanded.                                      *
//*                  If mIndex == MAX_DIALOG_CONTROLS, then menu with lowest   *
//*                   index gets input focus, but is not expanded.             *
//*                                                                            *
//* Returns: code indicating the menu item chosen                              *
//******************************************************************************

MenuCode FileMangler::MenuBarAccess ( short mIndex )
{
MenuCode selection = mcNO_SELECTION ;        // return value

   //* If the Menu Bar is currently hidden, make it visible.                   *
   //* Activate the member controls and place the focus on the indicated menu. *
   //* (We trust the caller to send a valid index.)                            *
   this->DialogTitle ( true, mIndex ) ;

   //* User instructions *
   attr_t uMsgColor = dtbmNFcolor ;
   dtbmData uMsg( "  ESC exits menu--ESC exits menu--ESC exits menu--ESC exits menu",
                  &uMsgColor, true ) ;
   this->dWin->DisplayTextboxMessage ( dwMsgsTB, uMsg ) ;

   //* Give user a chance to make a decision. Stay in the   *
   //* Menu Bar until we have a selection or an Escape key. *
   uiInfo Info ;
   Info.viaHotkey = !(mIndex == MAX_DIALOG_CONTROLS) ;
   bool   done = false ;
   while ( ! done )
   {
      this->dWin->EditMenuwin ( Info ) ;

      //* If mouse interface enabled, test whether user has clicked *
      //* another member of the MenuBar group. If so, just call the *
      //* edit routine again. This is a special test for MenuBar    *
      //* groups, and indicates that the edit routine has pushed a  *
      //* hotkey keycode into the input stream. This hotkey will    *
      //* cause the target menu to open immediately.                *
      if ( Info.wk.mevent.conv && (this->dWin->meMouseEnabled ()) &&
           Info.wk.type == wktFUNKEY && Info.wk.key == nckESC &&
           (Info.wk.mevent.cIndex >= dwFileMB && Info.wk.mevent.cIndex <= dwHelpMB) )
      { /* Continue the Loop */ }

      //* If user has made a selection, OR has left the       *
      //* MenuBar without making a selection, then we're done.*
      else
         done = true ;
   }

   //* Erase user instructions *
   uMsg = "" ;
   this->dWin->DisplayTextboxMessage ( dwMsgsTB, uMsg ) ;

   //* Translate menu selection index to MenuCode enum value *
   if ( Info.dataMod != false )
   {
      selection = this->MenuItem2MenuCode ( Info.ctrlIndex, Info.selMember ) ;
   }

   //* If Menu Bar is not locked in the visible state, hide it.*
   //* Else, leave it visible, but deactivate it.              *
   this->DialogTitle ( false ) ;
   this->dWin->RefreshWin () ;

   return selection ;

}  //* End MenuBarAccess() *

//*************************
//*  ExecuteMenuCommand   *
//*************************
//******************************************************************************
//* Execute the command returned from user's visit to the menu system.         *
//* (see also MenuBarAccess() and the various context menu interface methods)  *
//*                                                                            *
//* Input  : menuCode: member of enum MenuCode, command code                   *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************
//* Programmer's Note: Keep the switch entries synchronized with the           *
//* enum MenuCode.                                                             *
//******************************************************************************

void FileMangler::ExecuteMenuCommand ( MenuCode menuCode )
{
   switch ( menuCode )
   {
      //* GENERAL    *
      case mcNO_SELECTION:        //* Menu closed without a selection,
      case mcHANDLED:             //* Selection handled before return
      case mcSYSERROR:            //* If system call returned an error
         break ;                  
                                  
      //* FILE MENU  *            
      case mcFileMB_FAVORITES:    //* request to open Favorites dialog
         this->CmdFavorites () ;
         break ;
      case mcFileMB_SORTOPTION:
         this->CmdDialogSort ( menuCode ) ;
         break ;
      case mcFileMB_FINDFILES:    //* request to open find-files dialog
         this->CmdFind () ;
         break ;
      case mcFileMB_MOUNT:        //* request to open mount-filesystem dialog
         this->CmdMount ( menuCode ) ;
         break ;
      case mcFileMB_EJECT:        //* eject optical media
         this->CmdOpticalMedia () ;
         break ;
      case mcFileMB_REFRESH:      //* refresh the directory display
         this->CmdRefresh () ;
         break ;
      case mcFileMB_SETALT:       //* set CWD of alt window equal to active window
         this->CmdAltCWD () ;
         break ;
      case mcFileMB_SHELL:        //* go to command shell
         this->CmdShellOut () ;
         break ;
      case mcFileMB_EXIT:         //* exit the application
         break ;                  
                                  
      //* EDIT MENU  *            
      case mcEditMB_COPY:         //* copy files
         this->CmdCopy () ;
         break ;
      case mcEditMB_CUT:          //* cut files
         this->CmdCut () ;
         break ;
      case mcEditMB_PASTE:        //* paste files
         this->CmdPaste () ;
         break ;
      case mcEditMB_PASTESP:      //* paste-special files
         this->CmdPasteSp () ;
         break ;
      case mcEditMB_SELECTALL:    //* select all files in window
         this->CmdSelectAll () ;
         break ;
      case mcEditMB_PREFS:        //* request to open Preferences dialog
         break ;                  
                                  
      //* VIEW MENU  *            
      case mcViewMB_DUALWIN:      //* dial-window display
      case mcViewMB_SINGLEWIN:    //* single-window display
      case mcViewMB_RESIZE:       //* resize the dialog to fit terminal window
         //* These commands are handled at a higher level *
         break ;
      case mcViewMB_DIRTREE:      //* display file-system tree structure
         this->CmdDirTree () ;
         break ;
      case mcViewMB_LOCKMBAR:     //* lock menu bar in visible state
         this->CmdLockMenuBar ( !this->lockMenuBar ) ;
         break ;
      case mcViewMB_SYNCLOCK:     //* toggle the dual-window synch lock
         this->CmdSynchLock () ;
         break ;
      case mcViewMB_USERINFO:     //* display user info
         this->CmdUserInfo () ;
         break ;
      case mcViewMB_CLIPBRD:      //* browse clipboard
         this->CmdBrowseCB () ;
         break ;
      case mcViewMB_FILESYS:      //* file system summary
         this->CmdFilesysInfo () ;
         break ;
      case mcViewMB_HIDDEN:       //* display/hide hidden files
         this->CmdHiddenFiles () ;
         break ;
      case mcViewMB_ROOTSCAN:     //* toggle full-scan option for root directory
         this->CmdRootScan () ;
         break ;

      //* UTIL MENU  *
      case mcUtilMB_BACKUP:       //* back-up directory contents
         this->CmdBackup ( false ) ;
         break ;
      case mcUtilMB_SYNCH:        //* synchronnize directory contents
         this->CmdBackup ( true ) ;
         break ;
      case mcUtilMB_ARCHIVE:      //* create a tar archive
         this->CmdArchive () ;
         break ;
      case mcUtilMB_CMPFILE:      //* compare two files (diff)
         this->CmdCompareFiles () ;
         break ;
      case mcUtilMB_GREP:         //* scan contents of files (grep)
         this->CmdGrep () ;
         break ;
      case mcUtilMB_HLINK:        //* locate files with shared inode
         this->CmdInodes () ;
         break ;
      case mcUtilMB_UTRASH:       //* un-do move-to-trash operation
      case mcUtilMB_MTRASH:       //* manage the Trashcan
         this->CmdTrash ( menuCode ) ;
         break ;
      case mcUtilMB_MOOSE:        //* enable/disable mouse support
         this->CmdMouse () ;
         break ;
      case mcUtilMB_CONFIG:       //* request to open configuration dialog
         break ;

      //* HELP MENU  *
      case mcHelpMB_F01HELP:      //* request to open info-reader help
         this->CmdCallHelp () ;
         break ;
      case mcHelpMB_HTML:         //* request to open HTML docs in browser
         this->CmdCallHelp ( true ) ;
         break ;
      case mcHelpMB_KEYBIND:      //* display current key bindings
         this->CmdKeyBindings () ;
         break ;
      case mcHelpMB_HELPABOUT:    //* display Help-about dialog
         this->CmdHelpAbout () ;
         break ;

      //* FileCoMmanD SUB-MENU  *            
      case mcFcmdMB_SELECT:       //* select/deselect file under highlight
         {
         wkeyCode wk( nckSPACE, wktPRINT ) ;
         this->CmdSelect ( wk ) ;
         }
         break ;
      case mcFcmdMB_SELECTALL:    //* select/deselect all files in window
         this->CmdSelectAll () ;
         break ;
      case mcFcmdMB_TRASH:        //* move files to trashcan
         this->CmdTrash ( menuCode ) ;
         break ;
      case mcFcmdMB_COPY:         //* copy files
         this->CmdCopy () ;
         break ;
      case mcFcmdMB_CUT:          //* cut files
         this->CmdCut () ;
         break ;
      case mcFcmdMB_PASTE:        //* paste files
         this->CmdPaste () ;
         break ;
      case mcFcmdMB_PASTESP:      //* paste-special files
         this->CmdPasteSp () ;
         break ;
      case mcFcmdMB_DELETE:       //* delete files
         this->CmdDelete () ;
         break ;
      case mcFcmdMB_RENAME:       //* rename files
         this->CmdRename () ;
         break ;
      case mcFcmdMB_TOUCH:        //* touch files (set mod date)
         this->CmdTouch () ;
         break ;
      case mcFcmdMB_WENABLE:      //* write-enable files
         this->CmdWProtect ( false ) ;
         break ;
      case mcFcmdMB_WPROT:        //* write-protect files
         this->CmdWProtect ( true ) ;
         break ;
      case mcFcmdMB_NEWDIR:       //* create new directory
         this->CmdCreateDir () ;
         break ;                  
                                  
      //* VIEW-FILE SUB/CONTEXT MENU *
      case mcVfileCM_CONTENTS:    //* view file contents
      case mcVfileCM_VIEWSTATS:   //* view file statistics
      case mcVfileCM_EXECUTE:     //* if file is an executable, run it
      case mcVfileCM_FINDTARGET:  //* if file is a symbolic link, find link target
      case mcVfileCM_CANCELVIEW:  //* cancel view-file operation
         this->CmdView ( menuCode ) ;
         break ;

      //* Invalid or unimplemented menu code -- this should never happen *
      default:
         gString msgBuff( "MenuCode: 0x%04X - %s", &menuCode, MenuCodeStrings[menuCode] ) ;
         this->FmDebugMsg ( msgBuff.ustr() ) ;
         break ;
   }
   
}  //* End ExecuteMenuCommand() *

//*************************
//*   MenuItem2MenuCode   *
//*************************
//******************************************************************************
//* A selection has been made in the menu system. Translate the selection      *
//* into an action code, so command may be executed.                           *
//*                                                                            *
//* Input  : menuIndex:  index of control from which selection was made        *
//*          menuMember: item number selected within the menu                  *
//*                                                                            *
//* Returns: member of enum MenuCode                                           *
//******************************************************************************

MenuCode FileMangler::MenuItem2MenuCode ( short menuIndex, short menuMember )
{
MenuCode    selection = mcNO_SELECTION ;     // return value

   switch ( menuIndex )
   {
      case dwFileMB:                      // dwFileMB menu
         selection = mbFileCodes[menuMember] ;
         break ;
      case dwEditMB:                      // dwEditMB menu
         selection = mbEditCodes[menuMember] ;
         break ;
      case dwViewMB:                      // dwViewMB menu
         selection = mbViewCodes[menuMember] ;
         break ;
      case dwUtilMB:                      // dwUtilMB menu
         selection = mbUtilCodes[menuMember] ;
         break ;
      case dwHelpMB:                      // dwHelpMB menu
         selection = mbHelpCodes[menuMember] ;
         break ;
      case dwFcmdMB:                      // dwFcmdMB menu
         selection = mbFcmdCodes[menuMember] ;
         break ;
      case dwSortMB:                      // dwSortMB menu
         selection = mbSortCodes[menuMember] ;
         break ;
      case dwVfileCM:                     // dwVfileCM menu
         selection = cmVfileCodes[menuMember] ;
         break ;
      default:                            // invalid menu index (unlikely)
         break ;
   }
   return selection ;

}  //* End MenuItem2MenuCode() *

//*************************
//*    InitFmMenuData     *
//*************************
//******************************************************************************
//* Data which defines the NcDialog object and its component controls must     *
//* be initialized from two sources:                                           *
//*  1) configuration data for the application                                 *
//*     - User's preferences for color scheme are set according to             *
//*       cfgOpt.cScheme.scheme.                                               *
//*  2) the parameters specific to the 'window mode' that will instantiate     *
//*     the dialog object.                                                     *
//*                                                                            *
//*                                                                            *
//* Input  : wMode : member of enum WinMode indicating the type of             *
//*                  initialization to perform                                 *
//*                                                                            *
//* Returns: true if successful, false if error                                *
//******************************************************************************
//* Programmer's Note:                                                         *
//* The application's color scheme is defined in this method; however if this  *
//* defininition changes, be sure to also update:                              *
//* - ecoEditSC_ControlUpdate() method in FmConfigi.cpp. That method is used   *
//*   for color-map selection during interactive configuration.                *
//* - TestColorSchemes() debugging method (below)                              *
//*               (consistency is such a bitch, isn't it?)                     *
//*                                                                            *
//*                                                                            *
//* NOTE: Dynamic key-mapping is to be done here, but is not yet implemented.  *
//*                                                                            *
//******************************************************************************

bool FileMangler::InitFmMenuData ( WinMode wMode )
{
   bool  success = false ;

   this->fWin = NULL ;  // FileDlg* for left/only window not yet opened
   this->rWin = NULL ;  // FileDlg* for right window (dual-win mode) not yet opened
   this->fdPtr  = NULL ;// FileDlg* for window with input focus
   this->altPtr = NULL ;// FileDlg* for window without input focus (dual-win mode only)

   //*****************************************************
   //* Color Scheme: application's border color, default *
   //* background color, menu color, etc.                *
   //*****************************************************
   //* Colors for basic application controls are handled as a special case.* 
   attr_t statusnColor = nc.reB,       // application status window (non-focus)
          statusfColor = nc.bw,        // application status window (focus)
          pathnColor = nc.bl,          // application path window (non-focus)
          pathfColor = nc.re,          // application path window (focus)
          statnColor = nc.bl,          // application statistics window (non-focus)
          statfColor = nc.reB,         // application statistics window (focus)
          inactColor = nc.gyR ;        // application menu, inactive menu item

   //* Application's default color scheme. *
   this->cfgOpt.cScheme.bb = nc.cyR ;        // application borders and backgrounds
   this->cfgOpt.cScheme.sd = nc.blR ;        // sub-dialog color
   this->cfgOpt.cScheme.sb = nc.gybl | ncbATTR ; // sub-dialog Bold color
   this->cfgOpt.cScheme.em = nc.brbl ;       // sub-dialog Emphasis color
   this->cfgOpt.cScheme.wr = nc.rebl ;       // sub-dialog Warning color
   this->cfgOpt.cScheme.mn = nc.blR ;        // menu border (without focus) color
   this->cfgOpt.cScheme.mf = nc.cyR ;        // menu border (with focus) color
   this->cfgOpt.cScheme.pn = nc.gyR ;        // pushbutton (without focus) color
   this->cfgOpt.cScheme.pf = nc.reG ;        // pushbutton (with focus) color
   this->cfgOpt.cScheme.tn = nc.bw ;         // text box (without focus) color
   this->cfgOpt.cScheme.tf = nc.gr ;         // text box (with focus) color
   this->monoColor[0] = (-1) ;               // menu interior
   this->monoColor[1] = this->cfgOpt.cScheme.mn ;
   this->titleColor = this->cfgOpt.cScheme.bb ; // application title color

   //* Adjust for non-default color schemes according to user's config option. *
   if ( this->cfgOpt.cScheme.scheme != ncbcCOLORS )
   {
      switch ( this->cfgOpt.cScheme.scheme )
      {
         case ncbcBK:      // black
            this->cfgOpt.cScheme.bb = nc.bk ;   // application borders and backgrounds
            this->cfgOpt.cScheme.sd = nc.bk ;   // sub-dialog color
            this->cfgOpt.cScheme.sb = nc.bk & ~ncbATTR ; // sub-dialog Bold color
            this->cfgOpt.cScheme.em = nc.brbk ; // sub-dialog Emphasis color
            this->cfgOpt.cScheme.wr = nc.rebk ; // sub-dialog Warning color
            this->cfgOpt.cScheme.mn = nc.cyR ;  // menu border (without focus) color
            this->cfgOpt.cScheme.mf = nc.bk ;   // menu border (with focus) color
            this->cfgOpt.cScheme.pn = nc.gyR ;  // pushbutton (without focus) color
            this->cfgOpt.cScheme.pf = nc.reG ;  // pushbutton (with focus) color
            this->cfgOpt.cScheme.tn = nc.bw ;   // text box (without focus) color
            this->cfgOpt.cScheme.tf = nc.bl ;   // text box (with focus) color
            this->monoColor[1] = this->cfgOpt.cScheme.mf ; // menu interior
            this->titleColor = this->cfgOpt.cScheme.sd ;// application title color
            break ;
         case ncbcRE:      // red
            this->cfgOpt.cScheme.bb = nc.reR ;  // application borders and backgrounds
            this->cfgOpt.cScheme.sd = nc.reR ;  // sub-dialog color
            this->cfgOpt.cScheme.sb = nc.gyre | ncbATTR ; // sub-dialog Bold color
            this->cfgOpt.cScheme.em = nc.brre ; // sub-dialog Emphasis color
            this->cfgOpt.cScheme.wr = nc.mare ; // sub-dialog Warning color
            this->cfgOpt.cScheme.mn = nc.re ;   // menu border (without focus) color
            this->cfgOpt.cScheme.mf = nc.reR ;  // menu border (with focus) color
            this->cfgOpt.cScheme.pn = nc.gyR ;  // pushbutton (without focus) color
            this->cfgOpt.cScheme.pf = nc.maG ;  // pushbutton (with focus) color
            this->cfgOpt.cScheme.tn = nc.bw ;   // text box (without focus) color
            this->cfgOpt.cScheme.tf = nc.bl ;   // text box (with focus) color
            this->monoColor[1] = this->cfgOpt.cScheme.mf ; // menu interior
            this->titleColor = this->cfgOpt.cScheme.sb ;// application title color
            break ;
         case ncbcGR:      // green
            this->cfgOpt.cScheme.bb = nc.grR ;  // application borders and backgrounds
            this->cfgOpt.cScheme.sd = nc.grR ;  // sub-dialog color
            this->cfgOpt.cScheme.sb = nc.gygr | ncbATTR ; // sub-dialog Bold color
            this->cfgOpt.cScheme.em = nc.brgr ; // sub-dialog Emphasis color
            this->cfgOpt.cScheme.wr = nc.regr ; // sub-dialog Warning color
            this->cfgOpt.cScheme.mn = nc.gr ;   // menu border (without focus) color
            this->cfgOpt.cScheme.mf = nc.grR ;  // menu border (with focus) color
            this->cfgOpt.cScheme.pn = nc.gyR ;  // pushbutton (without focus) color
            this->cfgOpt.cScheme.pf = nc.reG ;  // pushbutton (with focus) color
            this->cfgOpt.cScheme.tn = nc.bw ;   // text box (without focus) color
            this->cfgOpt.cScheme.tf = nc.bl ;   // text box (with focus) color
            this->monoColor[1] = this->cfgOpt.cScheme.mf ; // menu interior
            this->titleColor = this->cfgOpt.cScheme.sb ;// application title color
            break ;
         case ncbcBR:      // brown
            this->cfgOpt.cScheme.bb = nc.brR ;  // application borders and backgrounds
            this->cfgOpt.cScheme.sd = nc.brR ;  // sub-dialog color
            this->cfgOpt.cScheme.sb = nc.gybr | ncbATTR ; // sub-dialog Bold color
            this->cfgOpt.cScheme.em = nc.cybr ; // sub-dialog Emphasis color
            this->cfgOpt.cScheme.wr = nc.mabr ; // sub-dialog Warning color
            this->cfgOpt.cScheme.mn = nc.br ;   // menu border (without focus) color
            this->cfgOpt.cScheme.mf = nc.brR ;  // menu border (with focus) color
            this->cfgOpt.cScheme.pn = nc.gyR ;  // pushbutton (without focus) color
            this->cfgOpt.cScheme.pf = nc.maG ;  // pushbutton (with focus) color
            this->cfgOpt.cScheme.tn = nc.bw ;   // text box (without focus) color
            this->cfgOpt.cScheme.tf = nc.bl ;   // text box (with focus) color
            this->monoColor[1] = this->cfgOpt.cScheme.mf ; // menu interior
            this->titleColor = this->cfgOpt.cScheme.sb ;// application title color
            break ;
         case ncbcBL:      // blue
            this->cfgOpt.cScheme.bb = nc.blR ;  // application borders and backgrounds
            this->cfgOpt.cScheme.sd = nc.blR ;  // sub-dialog color
            this->cfgOpt.cScheme.sb = nc.gybl | ncbATTR ; // sub-dialog Bold color
            this->cfgOpt.cScheme.em = nc.brbl ; // sub-dialog Emphasis color
            this->cfgOpt.cScheme.wr = nc.rebl ; // sub-dialog Warning color
            this->cfgOpt.cScheme.mn = nc.bl ;   // menu border (without focus) color
            this->cfgOpt.cScheme.mf = nc.blR ;  // menu border (with focus) color
            this->cfgOpt.cScheme.pn = nc.gyR ;  // pushbutton (without focus) color
            this->cfgOpt.cScheme.pf = nc.reG ;  // pushbutton (with focus) color
            this->cfgOpt.cScheme.tn = nc.bw ;   // text box (without focus) color
            this->cfgOpt.cScheme.tf = nc.gr ;   // text box (with focus) color
            this->monoColor[1] = this->cfgOpt.cScheme.mf ; // menu interior
            this->titleColor = this->cfgOpt.cScheme.sb ;// application title color
            break ;
         case ncbcMA:      // magenta
            this->cfgOpt.cScheme.bb = nc.maR ;  // application borders and backgrounds
            this->cfgOpt.cScheme.sd = nc.maR ;  // sub-dialog color
            this->cfgOpt.cScheme.sb = nc.gyma | ncbATTR ; // sub-dialog Bold color
            this->cfgOpt.cScheme.em = nc.brma ; // sub-dialog Emphasis color
            this->cfgOpt.cScheme.wr = nc.rema ; // sub-dialog Warning color
            this->cfgOpt.cScheme.mn = nc.ma ;   // menu border (without focus) color
            this->cfgOpt.cScheme.mf = nc.maR ;  // menu border (with focus) color
            this->cfgOpt.cScheme.pn = nc.gyR ;  // pushbutton (without focus) color
            this->cfgOpt.cScheme.pf = nc.reG ;  // pushbutton (with focus) color
            this->cfgOpt.cScheme.tn = nc.bw ;   // text box (without focus) color
            this->cfgOpt.cScheme.tf = nc.bl ;   // text box (with focus) color
            this->monoColor[1] = this->cfgOpt.cScheme.mf ; // menu interior
            this->titleColor = this->cfgOpt.cScheme.sb ;// application title color
            break ;
         case ncbcCY:      // cyan
            this->cfgOpt.cScheme.bb = nc.cyR ;  // application borders and backgrounds
            this->cfgOpt.cScheme.sd = nc.cyR ;  // sub-dialog color
            this->cfgOpt.cScheme.sb = nc.gycy | ncbATTR ; // sub-dialog Bold color
            this->cfgOpt.cScheme.em = nc.brcy ; // sub-dialog Emphasis color
            this->cfgOpt.cScheme.wr = nc.recy ; // sub-dialog Warning color
            this->cfgOpt.cScheme.mn = nc.cy ;   // menu border (without focus) color
            this->cfgOpt.cScheme.mf = nc.cyR ;  // menu border (with focus) color
            this->cfgOpt.cScheme.pn = nc.gyR ;  // pushbutton (without focus) color
            this->cfgOpt.cScheme.pf = nc.reG ;  // pushbutton (with focus) color
            this->cfgOpt.cScheme.tn = nc.bw ;   // text box (without focus) color
            this->cfgOpt.cScheme.tf = nc.bl ;   // text box (with focus) color
            this->monoColor[1] = this->cfgOpt.cScheme.mf ; // menu interior
            this->titleColor = this->cfgOpt.cScheme.sb ;// application title color
            break ;
         case ncbcGY:      // grey
            this->cfgOpt.cScheme.bb = nc.gyR ;  // application borders and backgrounds
            this->cfgOpt.cScheme.sd = nc.gyR ;  // sub-dialog color
            this->cfgOpt.cScheme.sb = nc.bkgy ; // sub-dialog Bold color
            this->cfgOpt.cScheme.em = nc.blgy ; // sub-dialog Emphasis color
            this->cfgOpt.cScheme.wr = nc.regy ; // sub-dialog Warning color
            this->cfgOpt.cScheme.mn = nc.cy ;   // menu border (without focus) color
            this->cfgOpt.cScheme.mf = nc.cyR ;  // menu border (with focus) color
            this->cfgOpt.cScheme.pn = nc.bw ;   // pushbutton (without focus) color
            this->cfgOpt.cScheme.pf = nc.reG ;  // pushbutton (with focus) color
            this->cfgOpt.cScheme.tn = nc.bw ;   // text box (without focus) color
            this->cfgOpt.cScheme.tf = nc.bl ;   // text box (with focus) color
            this->monoColor[1] = this->cfgOpt.cScheme.mf ; // menu interior
            this->titleColor = this->cfgOpt.cScheme.sb ;// application title color
            break ;
         case ncbcDEFAULT: // terminal default
            this->cfgOpt.cScheme.bb = nc.bw ;   // application borders and backgrounds
            this->cfgOpt.cScheme.sd = nc.cyR ;  // sub-dialog color
            this->cfgOpt.cScheme.sb = nc.gycy | ncbATTR ; // sub-dialog Bold color
            this->cfgOpt.cScheme.em = nc.brcy ; // sub-dialog Emphasis color
            this->cfgOpt.cScheme.wr = nc.recy ; // sub-dialog Warning color
            this->cfgOpt.cScheme.mn = nc.gyR ;  // menu border (without focus) color
            this->cfgOpt.cScheme.mf = nc.bw ;   // menu border (with focus) color
            this->cfgOpt.cScheme.pn = nc.gyR ;  // pushbutton (without focus) color
            this->cfgOpt.cScheme.pf = nc.reG ;  // pushbutton (with focus) color
            this->cfgOpt.cScheme.tn = nc.bw ;   // text box (without focus) color
            this->cfgOpt.cScheme.tf = nc.blR ;  // text box (with focus) color
            this->monoColor[1] = this->cfgOpt.cScheme.mf ; // menu interior
            this->titleColor = this->cfgOpt.cScheme.bb ;// application title color
            statusnColor = nc.regy | ncbATTR;   // application status window (non-focus)
            statusfColor = nc.bkgy ;            // application status window (focus)
            pathnColor   = nc.blgy ;            // application path window (non-focus)
            pathfColor   = nc.regy | ncbATTR ;  // application path window (focus)
            statnColor   = nc.blgy ;            // application statistics window (non-focus)
            statfColor   = nc.regy | ncbATTR ;  // application statistics window (focus)
            break ;
         default:          // default values handled above
            break ;
      }
   }

   //* Initialize color attributes for the active/inactive menu items for *
   //* for menus which contain items that are sometimes inactive.         *
   // Note: These are local data defined at the top of this module.       *
   attr_t actColor = this->cfgOpt.cScheme.mn ;
   if ( this->cfgOpt.cScheme.scheme != ncbcCOLORS )
      actColor = this->cfgOpt.cScheme.mf ;
   for ( short i = ZERO ; i < cmVfileDATA_ITEMS ; i++ )
   {
      cmVfileAcolorFT[i] = cmVfileActiveFT[i]   == false ? inactColor : actColor ;
      cmVfileIcolorFT[i] = cmVfileInactiveFT[i] == false ? inactColor : actColor ;
   }

   //* Check for terminal window dimensions large enough to hold application's *
   //* display window. Re-size the application window as necessary and report  *
   //* any problems with terminal window dimensions.                           *
   nc.ScreenDimensions ( this->termRows, this->termCols ) ;
   bool  linesOK = (this->termRows >= MIN_ROWS) ;
   bool  swColsOK = (this->termCols >= MIN_COLS) ;
   bool  colsOK  = (   (wMode == wmSINGLE_WIN && swColsOK)
                    || (wMode == wmDUAL_WIN && 
                        (this->termCols >= MIN_DUALWINDOW_WIDTH)) ) ;
   if ( linesOK && colsOK )
   {
      //* Dynamically adjust dialog dimensions to fill terminal window.        *
      //* Note: Width of dialog is not adjusted for Single-Win mode because we *
      //* know that the terminal window is at least wide enough for our needs. *
      //* Also note that Dual-Win mode requires an EVEN number of columns.     *
      this->actualRows = this->termRows ;
      if ( wMode == wmDUAL_WIN )
      {
         this->actualCols = 
            (this->termCols > (MIN_COLS * 2)) ? (MIN_COLS * 2) : this->termCols ;
         if ( this->actualCols % 2 )
            --this->actualCols ;
      }
      else     // wMode == wmSINGLE_WIN)
      {
         this->actualCols = MIN_COLS ;
      }
      success = true ;

      //* If the control definitions have not been instantiated, do it now. *
      //* NOTE: It is not possible to make internal references to the array *
      //*       during instantiation; therefore we must do it in two steps: *
      //        first instantiate, then initialize.                         *
      if ( this->ic == NULL )
      {
         this->ic = new InitCtrl[dwCTRLS] ;
         this->ic[dwFileSE] = 
            {  //* File Data (left side or only) - - - - - - - - - -  dwFileSE *
               dctSCROLLEXT,                 // type:      define a scrolling-data control
               rbtTYPES,                     // sbSubtype: (n/a)
               false,                        // rbSelect:  (n/a)
               4,                            // ulY:       upper left corner in Y
               ZERO,                         // ulX:       upper left corner in X
               short(this->actualRows - 6),  // lines:     control lines
               this->actualCols,             // cols:      control columns
               NULL,                         // dispText:  n/a
               this->cfgOpt.cScheme.bb,      // nColor:    non-focus border color
               this->cfgOpt.cScheme.bb,      // fColor:    focus border color
               tbPrint,                      // filter:    (n/a)
               NULL,                         // label:     no label
               ZERO,                         // labY:      offset from control's ulY
               ZERO,                         // labX       offset from control's ulX
               ddBoxTYPES,                   // exType:    (n/a)
               1,                            // scrItems:  (n/a)
               1,                            // scrSel:    (n/a)
               NULL,                         // scrColor:  (n/a)
               NULL,                         // spinData:  (n/a)
               true,                         // active:    allow control to gain focus
               &ic[dwMsgsTB],                // nextCtrl:  link in next structure
            } ;
         this->ic[dwMsgsTB] = 
            {  //* Status Messages (left side or only) - - - - - - -  dwMsgsTB *
               dctTEXTBOX,                   // type:      define a plain-text textbox
               rbtTYPES,                     // sbSubtype: (n/a)
               false,                        // rbSelect:  (n/a)
               1,                            // ulY:       upper left corner in Y
               8,                            // ulX:       upper left corner in X
               1,                            // lines:     one display line
               // Programmer's Note: This parameter makes an assumption
               // The eleven(11) columns are: borders + "Status:" + "  "
               // which fills out the width of the Single-Win dialog.
               short(MIN_COLS - 11),         // cols:      display columns
               quickHelpMsg,                 // dispText: initial display text
               statusnColor,                 // nColor:    normal message text color
               statusfColor,                 // fColor:    alternate message text color
               tbPrint,                      // filter:    accepts all printing ASCII characters
               "Status:",                    // label:     label text
               ZERO,                         // labY:      offset from control's ulY
               -7,                           // labX:      offset from control's ulX
               ddBoxTYPES,                   // exType:    (n/a)
               1,                            // scrItems:  (n/a)
               1,                            // scrSel:    (n/a)
               NULL,                         // scrColor:  (n/a)
               NULL,                         // spinData:  (n/a)
               false,                        // active:    view only
               &ic[dwPathTB],                // nextCtrl:  link in next structure
            } ;
         this->ic[dwPathTB] = 
            {  //* Directory Path (left side or only)  - - - - - - -  dwPathTB *
               dctTEXTBOX,                   // type:      define a plain-text textbox
               rbtTYPES,                     // sbSubtype: (n/a)
               false,                        // rbSelect:  (n/a)
               3,                            // ulY:       offset in Y from corner of dialog
               1,                            // ulX:       offset in X from corner of dialog
               1,                            // lines:     one display line
               short(this->actualCols - 2),  // cols:      display columns
               NULL,                         // dispText:  initial display text
               pathnColor,                   // nColor:    normal message text color
               pathfColor,                   // fColor:    alternate message text color
               tbPrint,                      // filter:    accepts all printing ASCII characters
               NULL,                         // label:      no label text (empty string)
               ZERO,                         // labY:      offset from control's ulY
               ZERO,                         // labX:      offset from control's ulX
               ddBoxTYPES,                   // exType:    (n/a)
               1,                            // scrItems:  (n/a)
               1,                            // scrSel:    (n/a)
               NULL,                         // scrColor:  (n/a)
               NULL,                         // spinData:  (n/a)
               false,                        // active:    view only
               &ic[dwStatTB],                // nextCtrl:  link in next structure
            } ;
         this->ic[dwStatTB] = 
            {  //* File Statistics (left side or only) - - - - - - -  dwStatTB *
               dctTEXTBOX,                   // type:      define a plain-text textbox
               rbtTYPES,                     // sbSubtype: (n/a)
               false,                        // rbSelect:  (n/a)
               short(this->actualRows - 2),  // ulY:       offset in Y from corner of dialog
               1,                            // ulX:       offset in X from corner of dialog
               1,                            // lines:     one display line
               short(this->actualCols - 2),  // cols:      display columns
               NULL,                         // dispText:  initial display text (empty string)
               statnColor,                   // nColor:    normal message text color
               statfColor,                   // fColor:    alternate message text color
               tbPrint,                      // filter:    accepts all printing ASCII characters
               NULL,                         // label:      no label text
               ZERO,                         // labY:      offset from control's ulY
               ZERO,                         // labX:      offset from control's ulX
               ddBoxTYPES,                   // exType:    (n/a)
               1,                            // scrItems:  (n/a)
               1,                            // scrSel:    (n/a)
               NULL,                         // scrColor:  (n/a)
               NULL,                         // spinData:  (n/a)
               false,                        // active:    view only
               &ic[dwFileMB],                // nextCtrl:  link in next structure
            } ;
         this->ic[dwFileMB] = 
            {  //* "File" Menu - - - - - - - - - - - - - - - - - - -  dwFileMB *
               dctMENUWIN,                   // type:      define a Menu-win control
               rbtTYPES,                     // sbSubtype: (n/a)
               false,                        // rbSelect:  (n/a)
               this->menuBase.ypos,          // ulY:       offset in Y from corner of dialog
               this->menuBase.xpos,          // ulX:       offset in X from corner of dialog
               1,                            // lines:     (n/a)
               mbFileITEM_LEN,               // cols:      menu width
               (char*)&mbFileData,           // dispText:  pointer to display-text array
               this->cfgOpt.cScheme.mn,      // nColor:    non-focus color for collapsed menu
               this->cfgOpt.cScheme.mf,      // fColor:    focus color for collapsed menu
               tbPrint,                      // filter:    (n/a)
               "  File     ",                // label:     text for 'collapsed' menu
               ZERO,                         // labY:      (n/a)
               ZERO,                         // labX:      (n/a)
               ddBoxTYPES,                   // exType:    (n/a)
               mbFileDATA_ITEMS,             // scrItems:  number of elements in text/color arrays
               ZERO,                         // scrSel:    (n/a)
               monoColor,                    // scrColor:  single-color data strings
               NULL,                         // spinData:  (n/a)
               true,                         // active:    allow control to gain focus
               &ic[dwEditMB],                // nextCtrl:  link in next structure
            } ;
         this->ic[dwEditMB] = 
            {  //* "Edit" Menu - - - - - - - - - - - - - - - - - - -  dwEditMB *
               dctMENUWIN,                   // type:      define a Menu-win control
               rbtTYPES,                     // sbSubtype: (n/a)
               false,                        // rbSelect:  (n/a)
               ic[dwFileMB].ulY,             // ulY:       upper left corner in Y
               short(ic[dwFileMB].ulX + COLLAPSED_WIDTH), // UlX: upper left corner in X
               1,                            // lines:     (n/a)
               mbEditITEM_LEN,               // cols:      menu width
               (char*)&mbEditData,           // dispText:  pointer to display-text array
               this->cfgOpt.cScheme.mn,      // nColor:    non-focus color for collapsed menu
               this->cfgOpt.cScheme.mf,      // fColor:    focus color for collapsed menu
               tbPrint,                      // filter:    (n/a)
               "  Edit     ",                // label:     text for 'collapsed' menu
               ZERO,                         // labY:      (n/a)
               ZERO,                         // labX:      (n/a)
               ddBoxTYPES,                   // exType:    (n/a)
               mbEditDATA_ITEMS,             // scrItems:  number of elements in text/color arrays
               ZERO,                         // scrSel:    (n/a)
               monoColor,                    // scrColor:  single-color data strings
               NULL,                         // spinData:  (n/a)
               true,                         // active:    allow control to gain focus
               &ic[dwViewMB],                // nextCtrl:  link in next structure
            } ;
         this->ic[dwViewMB] = 
            {  //* "View" Menu - - - - - - - - - - - - - - - - - - -  dwViewMB *
               dctMENUWIN,                   // type:      define a Menu-win control
               rbtTYPES,                     // sbSubtype: (n/a)
               false,                        // rbSelect:  (n/a)
               ic[dwEditMB].ulY,             // ulY:       upper left corner in Y
               (short)(ic[dwEditMB].ulX + COLLAPSED_WIDTH), // UlX: upper left corner in X
               1,                            // lines:     (n/a)
               mbViewITEM_LEN,               // cols:      menu width
               (char*)&mbViewData,           // dispText:  pointer to display-text array
               this->cfgOpt.cScheme.mn,      // nColor:    non-focus color for collapsed menu
               this->cfgOpt.cScheme.mf,      // fColor:    focus color for collapsed menu
               tbPrint,                      // filter:    (n/a)
               "  View     ",                // label:     text for 'collapsed' menu
               ZERO,                         // labY:      (n/a)
               ZERO,                         // labX:      (n/a)
               ddBoxTYPES,                   // exType:    (n/a)
               mbViewDATA_ITEMS,             // scrItems:  number of elements in text/color arrays
               ZERO,                         // scrSel:    (n/a)
               monoColor,                    // scrColor:  single-color data strings
               NULL,                         // spinData:  (n/a)
               true,                         // active:    allow control to gain focus
               &ic[dwUtilMB],                // nextCtrl:  link in next structure
            } ;
         this->ic[dwUtilMB] = 
            {  //* "Util" Menu - - - - - - - - - - - - - - - - - - -  dwUtilMB *
               dctMENUWIN,                   // type:      define a Menu-win control
               rbtTYPES,                     // sbSubtype: (n/a)
               false,                        // rbSelect:  (n/a)
               ic[dwViewMB].ulY,             // ulY:       upper left corner in Y
               (short)(ic[dwViewMB].ulX + COLLAPSED_WIDTH), // UlX: upper left corner in X
               1,                            // lines:     (n/a)
               mbUtilITEM_LEN,               // cols:      menu width
               (char*)&mbUtilData,           // dispText:  pointer to display-text array
               this->cfgOpt.cScheme.mn,      // nColor:    non-focus color for collapsed menu
               this->cfgOpt.cScheme.mf,      // fColor:    focus color for collapsed menu
               tbPrint,                      // filter:    (n/a)
               "  Util     ",                // label:     text for 'collapsed' menu
               ZERO,                         // labY:      (n/a)
               ZERO,                         // labX:      (n/a)
               ddBoxTYPES,                   // exType:    (n/a)
               mbUtilDATA_ITEMS,             // scrItems:  number of elements in text/color arrays
               ZERO,                         // scrSel:    (n/a)
               monoColor,                    // scrColor:  single-color data strings
               NULL,                         // spinData:  (n/a)
               true,                         // active:    allow control to gain focus
               &ic[dwHelpMB],                // nextCtrl:  link in next structure
            } ;
         this->ic[dwHelpMB] = 
            {  //* "Help" Menu - - - - - - - - - - - - - - - - - - -  dwHelpMB *
               dctMENUWIN,                   // type:      define a Menu-win control
               rbtTYPES,                     // sbSubtype: (n/a)
               false,                        // rbSelect:  (n/a)
               ic[dwUtilMB].ulY,             // ulY:       upper left corner in Y
               (short)(ic[dwUtilMB].ulX + COLLAPSED_WIDTH), // UlX: upper left corner in X
               1,                            // lines:     (n/a)
               mbHelpITEM_LEN,               // cols:      menu width
               (char*)&mbHelpData,           // dispText:  pointer to display-text array
               this->cfgOpt.cScheme.mn,      // nColor:    non-focus color for collapsed menu
               this->cfgOpt.cScheme.mf,      // fColor:    focus color for collapsed menu
               tbPrint,                      // filter:    (n/a)
               "  Help     ",                // label:     text for 'collapsed' menu
               ZERO,                         // labY:      (n/a)
               ZERO,                         // labX:      (n/a)
               ddBoxTYPES,                   // exType:    (n/a)
               mbHelpDATA_ITEMS,             // scrItems:  number of elements in text/color arrays
               ZERO,                         // scrSel:    (n/a)
               monoColor,                    // scrColor:  single-color data strings
               NULL,                         // spinData:  (n/a)
               true,                         // active:    allow control to gain focus
               &ic[dwFcmdMB],                // nextCtrl:  link in next structure
            } ;
         this->ic[dwFcmdMB] = 
            {  //* File commands sub-menu- - - - - - - - - - - - - -  dwFcmdMB *
               dctMENUWIN,                   // type:      define a Menu-win control
               rbtTYPES,                     // sbSubtype: (n/a)
               false,                        // rbSelect:  (n/a)
               (short)(ic[dwFileMB].ulY + 2),// ulY:       upper left corner in Y
               (short)(ic[dwFileMB].ulX + ic[dwFileMB].cols - 1), // UlX: upper left corner in X
               1,                            // lines:     (n/a)
               mbFcmdITEM_LEN,               // cols:      menu width
               (char*)&mbFcmdData,           // dispText:  pointer to display-text array
               this->cfgOpt.cScheme.mn,      // nColor:    non-focus color for collapsed menu
               this->cfgOpt.cScheme.mf,      // fColor:    focus color for collapsed menu
               tbPrint,                      // filter:    (n/a)
               NULL,                         // label:     no label for sub-menus
               ZERO,                         // labY:      (n/a)
               ZERO,                         // labX:      (n/a)
               ddBoxTYPES,                   // exType:    (n/a)
               mbFcmdDATA_ITEMS,             // scrItems:  number of elements in text/color arrays
               ZERO,                         // scrSel:    (n/a)
               monoColor,                    // scrColor:  single-color data strings
               NULL,                         // spinData:  (n/a)
               false,                        // active:    initially inactive and invisible
               &ic[dwSortMB],                // nextCtrl:  link in next structure
            } ;
         this->ic[dwSortMB] = 
            {  //* Sort-order sub-menu - - - - - - - - - - - - - - -  dwSortMB *
               dctMENUWIN,                   // type:      define a Menu-win control
               rbtTYPES,                     // sbSubtype: (n/a)
               false,                        // rbSelect:  (n/a)
               (short)(ic[dwFileMB].ulY + 3),// ulY:       upper left corner in Y
               (short)(ic[dwFileMB].ulX + ic[dwFileMB].cols - 1), // UlX: upper left corner in X
               1,                            // lines:     (n/a)
               mbSortITEM_LEN,               // cols:      menu width
               (char*)&mbSortData,           // dispText:  pointer to display-text array
               this->cfgOpt.cScheme.mn,      // nColor:    non-focus color for collapsed menu
               this->cfgOpt.cScheme.mf,      // fColor:    focus color for collapsed menu
               tbPrint,                      // filter:    (n/a)
               NULL,                         // label:     no label for sub-menus
               ZERO,                         // labY:      (n/a)
               ZERO,                         // labX:      (n/a)
               ddBoxTYPES,                   // exType:    (n/a)
               mbSortDATA_ITEMS,             // scrItems:  number of elements in text/color arrays
               ZERO,                         // scrSel:    (n/a)
               monoColor,                    // scrColor:  single-color data strings
               NULL,                         // spinData:  (n/a)
               false,                        // active:    initially inactive and invisible
               &ic[dwVfileCM],               // nextCtrl:  link in next structure
            } ;
         this->ic[dwVfileCM] = 
            {  //* View-file context menu - - - - - - - - - - - - -  dwVfileCM *
               dctMENUWIN,                   // type:      define a Menu-win control
               rbtTYPES,                     // sbSubtype: (n/a)
               false,                        // rbSelect:  (n/a)
               short(ic[dwFileSE].ulY + 2),// ulY:       upper left corner in Y
               short(ic[dwFileMB].ulX + ic[dwFileMB].cols - 1), // UlX: upper left corner in X
               1,                            // lines:     (n/a)
               cmVfileITEM_LEN,              // cols:      menu width
               (char*)&cmVfileData,          // dispText:  pointer to display-text array
               this->cfgOpt.cScheme.mn,      // nColor:    non-focus color for collapsed menu
               this->cfgOpt.cScheme.mf,      // fColor:    focus color for collapsed menu
               tbPrint,                      // filter:    (n/a)
               NULL,                         // label:     no label for sub-menus
               ZERO,                         // labY:      (n/a)
               ZERO,                         // labX:      (n/a)
               ddBoxTYPES,                   // exType:    (n/a)
               cmVfileDATA_ITEMS,            // scrItems:  number of elements in text/color arrays
               ZERO,                         // scrSel:    (n/a)
               monoColor,                    // scrColor:  single-color data strings
               NULL,                         // spinData:  (n/a)
               false,                        // active:    initially inactive and invisible
               &ic[dwFileSEr],               // nextCtrl:  link in next structure
            } ;
            
            //********************************************
            //* Right-side controls for Dual-Window mode *
            //********************************************
         this->ic[dwFileSEr] = 
            {  //* File Data (right side)  - - - - - - - - - - - -   dwFileSEr *
               dctSCROLLEXT,                 // type:      define a scrolling-data control
               rbtTYPES,                     // sbSubtype: (n/a)
               false,                        // rbSelect:  (n/a)
               4,                            // ulY:       upper left corner in Y
               ZERO,                         // ulX:       upper left corner in X
               short(this->actualRows - 6),  // lines:     control lines
               this->actualCols,             // cols:      control columns
               NULL,                         // dispText:  n/a
               this->cfgOpt.cScheme.bb,      // nColor:    non-focus border color
               this->cfgOpt.cScheme.bb,      // fColor:    focus border color
               tbPrint,                      // filter:    (n/a)
               NULL,                         // label:     no label
               ZERO,                         // labY:      offset from control's ulY
               ZERO,                         // labX       offset from control's ulX
               ddBoxTYPES,                   // exType:    (n/a)
               1,                            // scrItems:  (n/a)
               1,                            // scrSel:    (n/a)
               NULL,                         // scrColor:  (n/a)
               NULL,                         // spinData:  (n/a)
               true,                         // active:    allow control to gain focus
               &ic[dwPathTBr],               // nextCtrl:  link in next structure
            } ;
         this->ic[dwPathTBr] = 
            {  //* Directory Path (right side) - - - - - - - - - -   dwPathTBr *
               dctTEXTBOX,                   // type:      define a plain-text textbox
               rbtTYPES,                     // sbSubtype: (n/a)
               false,                        // rbSelect:  (n/a)
               3,                            // ulY:       offset in Y from corner of dialog
               1,                            // ulX:       offset in X from corner of dialog
               1,                            // lines:     one display line
               (short)(this->actualCols - 2),// cols:      display columns
               NULL,                         // dispText:  initial display text
               pathnColor,                   // nColor:    normal message text color
               pathfColor,                   // fColor:    alternate message text color
               tbPrint,                      // filter:    accepts all printing ASCII characters
               NULL,                         // label:      no label text (empty string)
               ZERO,                         // labY:      offset from control's ulY
               ZERO,                         // labX:      offset from control's ulX
               ddBoxTYPES,                   // exType:    (n/a)
               1,                            // scrItems:  (n/a)
               1,                            // scrSel:    (n/a)
               NULL,                         // scrColor:  (n/a)
               NULL,                         // spinData:  (n/a)
               false,                        // active:    view only
               &ic[dwStatTBr],               // nextCtrl:  link in next structure
            } ;
         this->ic[dwStatTBr] = 
            {  //* File Statistics (right side)  - - - - - - - - -   dwStatTBr *
               dctTEXTBOX,                   // type:      define a plain-text textbox
               rbtTYPES,                     // sbSubtype: (n/a)
               false,                        // rbSelect:  (n/a)
               short(this->actualRows - 2),  // ulY:       offset in Y from corner of dialog
               1,                            // ulX:       offset in X from corner of dialog
               1,                            // lines:     one display line
               short(this->actualCols - 2),  // cols:      display columns
               NULL,                         // dispText:  initial display text (empty string)
               statnColor,                   // nColor:    normal message text color
               statfColor,                   // fColor:    alternate message text color
               tbPrint,                      // filter:    accepts all printing ASCII characters
               NULL,                         // label:      no label text
               ZERO,                         // labY:      offset from control's ulY
               ZERO,                         // labX:      offset from control's ulX
               ddBoxTYPES,                   // exType:    (n/a)
               1,                            // scrItems:  (n/a)
               1,                            // scrSel:    (n/a)
               NULL,                         // scrColor:  (n/a)
               NULL,                         // spinData:  (n/a)
               false,                        // active:    view only
               NULL                          // nextCtrl:  link in next structure
            } ;
      }  // if (this->ic== NULL)

      //***********************************************
      //*      Complete control definition setup      *
      //***********************************************
      //* File Data (left side)              dwFileSE *
      ic[dwFileSE].lines   = this->actualRows - 6 ;   // control lines
      if ( wMode == wmDUAL_WIN )                      // control columns
         ic[dwFileSE].cols = this->actualCols / 2 ;
      else
         ic[dwFileSE].cols = this->actualCols ;

      //* Status Messages (inactive)         dwMsgsTB *
      // (nothing to do)

      //* Directory Path (inactive) (left side)  dwPathTB *
      ic[dwPathTB].cols   = ic[dwFileSE].cols - 2 ;   // control columns

      //* File Statistics (inactive) (left side)  dwStatTB *
      ic[dwStatTB].ulY    = this->actualRows - 2 ;    // offset in Y from corner of dialog
      ic[dwStatTB].cols   = ic[dwFileSE].cols - 2 ;   // control columns

      //* "File" Menu                      dwFileMB *
      // (nothing to do)

      //* "Edit" Menu                      dwEditMB *
      // (nothing to do)

      //* "View" Menu                      dwViewMB *
      // (nothing to do)

      //* "Util" Menu                      dwUtilMB *
      // (nothing to do)

      //* "Help" Menu                      dwHelpMB *
      // (nothing to do)

      //* File Commands sub-menu           dwFcmdMB *
      // (nothing to do)

      //* Sort Options sub-menu            dwSortMB *
      // (nothing to do)

      //* View-file context menu          dwVfileCM *
      ic[dwVfileCM].nextCtrl = NULL ;  // last control in Single-window mode

      //* If we are starting in Dual-Win mode, initialize the remaing controls *
      if ( wMode == wmDUAL_WIN )
      {  //* Allow display of right-side controls *
         ic[dwVfileCM].nextCtrl = &ic[dwFileSEr] ;

         //* File Data (right side)            dwFileSEr *
         ic[dwFileSEr].ulY   = ic[dwFileSE].ulY ;  // Y position
         ic[dwFileSEr].ulX   = ic[dwFileSE].ulX + ic[dwFileSE].cols ; // X position
         ic[dwFileSEr].lines = ic[dwFileSE].lines ;// control lines
         ic[dwFileSEr].cols  = ic[dwFileSE].cols ; // control columns

         //* Directory Path (inactive) (right side)  dwPathTBr *
         ic[dwPathTBr].ulY   = ic[dwPathTB].ulY ;        // Y position
         ic[dwPathTBr].ulX   = ic[dwFileSEr].ulX + 1 ;   // X position
         ic[dwPathTBr].cols  = ic[dwFileSEr].cols - 2 ;  // control columns
   
         //* File Statistics (inactive) (right side)  dwStatTBr *
         ic[dwStatTBr].ulY   = ic[dwStatTB].ulY ;        // Y position
         ic[dwStatTBr].ulX   = ic[dwFileSEr].ulX + 1 ;   // X position
         ic[dwStatTBr].cols  = ic[dwFileSEr].cols - 2 ;  // control columns
      }
   }
   else
   {
      gString oBuff ;
      if ( linesOK == false || swColsOK == false )
         this->DiagMsg ( "Unable to open display window!", nc.reB, false ) ;
      else  // ( ! colsOK )
         this->DiagMsg ( "Unable to start in Dual-Window mode!", nc.reB, false ) ;
      this->DiagMsg ( " Please re-size terminal window.", nc.reB ) ;
      oBuff.compose( L"Current Size :    %2hd lines and    %2hd columns.", &termRows, &termCols ) ;
      this->DiagMsg ( oBuff.ustr(), nc.reB ) ;

      short reqLines = MIN_ROWS,
            reqCols  = (wMode==wmSINGLE_WIN || !linesOK || !swColsOK) ? 
                        MIN_COLS : MIN_DUALWINDOW_WIDTH ;
      oBuff.compose( L"Required Size: >= %2hd lines and >= %2hd columns.", &reqLines, &reqCols ) ;
      this->DiagMsg ( oBuff.ustr(), nc.reB ) ;
      if ( linesOK == false || swColsOK == false )
      {
         this->DiagMsg ( " ", nc.bw ) ;   // skip a line
         this->DiagMsg ( "Press Any Key to Exit . . .", nc.grB ) ;
         nckPause();
      }
      else  // ( ! swColsOK )
      {
         this->DiagMsg ( "Starting in Single-Window mode . . .", nc.grB ) ;
         chrono::duration<short>aWhile( 3 ) ;
         this_thread::sleep_for( aWhile ) ;
      }
   }
   return success ;

}  //* End InitFmMenuData() *

//*************************
//*  SetActiveMenuItems   *
//*************************
//******************************************************************************
//* Some individual menu items within the menuing system are not active under  *
//* all circumstances. This method set/reset the 'item-active' flags and       *
//* menu-item color attributes based on these circumstances.                   *
//*                                                                            *
//*                                                                            *
//* Input  : mIndex: member of dwControls indicating which menu to adjust      *
//*                  a value of dwCTRLS indicates that all updates should be   *
//*                  performed                                                 *
//*                                                                            *
//* Returns: OK  if successful                                                 *
//*          ERR if invalid index or index does not reference a control of     *
//*              dctMENUWIN type (or conflicting adjustment criteria)          *
//******************************************************************************

short FileMangler::SetActiveMenuItems ( dwControls mIndex )
{
short status = ERR ;

   if ( mIndex <= dwCTRLS )
   {  //* View-File context menu *
      if ( mIndex == dwVfileCM || mIndex == dwCTRLS )
      {
         tnFName  fStats ;
         this->fdPtr->GetStats ( fStats ) ;
         if ( fStats.fType == fmLINK_TYPE )
            this->dWin->SetActiveMenuItems ( dwVfileCM, 
                                             cmVfileActiveFT, cmVfileAcolorFT ) ;
         else
            this->dWin->SetActiveMenuItems ( dwVfileCM, 
                                             cmVfileInactiveFT, cmVfileIcolorFT ) ;
      }

#if 0    // NOT CURRENTLY USED
      if ( mIndex == dwViewMB || mIndex == dwCTRLS )
      {  //* 'View' menu: Only one of the two window-mode *
         //* options may be active at any one time.       *
//* Active item flags for 'View' menu.                               *
//* See also dwViewSWcolor[] and dwViewDWcolor defined in FmMenu.cpp.*
const bool dwViewActiveSW[mbViewDATA_ITEMS] = 
{ false, true, true, true, true, true, true, true, true, true, true } ;
const bool dwViewActiveDW[mbViewDATA_ITEMS] = 
{ true, false, true, true, true, true, true, true, true, true, true } ;

      }
#endif   // NOT CURRENTLY USED
   }
   return status ;

}  //* End SetActiveMenuItems() *

//*************************
//*     DialogTitle       *
//*************************
//******************************************************************************
//* Draw the application title at the top of the dialog window.                *
//*                                                                            *
//* The action taken depends not only on the input parameters, but also upon   *
//* data member this->lockMenuBar. The following scenarios may occur:          *
//*                                                                            *
//* 1) If 'establish' != false                                                 *
//*    On first call after Single/Dual window mode is started, 'establish'     *
//*    must be 'true', and the other parameters are ignored.                   *
//*    'establish is 'false' at all other times.                               *
//*    a) If this->lockMenuBar == false, then on return, dialog title will     *
//*       be displayed.                                                        *
//*    b) If this->lockMenuBar != false, then inactive Menu Bar will be        *
//*       displayed.                                                           *
//*                                                                            *
//*                                                                            *
//*                                                                            *
//* Input: showMenu  : if true, make the Menu Bar visible and place focus on   *
//*                     on first menu of the group.                            *
//*                    if false, make the Menu Bar invisible and display the   *
//*                     dialog title in that position.                         *
//*        trgMenu   : (optional, -1 by default)                               *
//*                    If 'showMenu' != false AND 'establish' == false:        *
//*                    if 'trgMenu' specified, then specified menu will be     *
//*                    opened. Else if 'trgMenu' == MAX_DIALOG_CONTROLS, then  *
//*                    Menu Bar will be made visible but no menu will be       *
//*                    expanded (not implemented). Otherwise the menu with     *
//*                    lowest index will be opened.                            *
//*        establish : (optional, false by default)                            *
//*                    if 'true': establish the Menu Bar and the dialog title  *
//*                     (on return, dialog title will be displayed)            *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************
//* Programmer's Note: We are ashamed to do something so kludgy:               *
//* Using the NcDialog SetDialogTitle() method centers the title in the top    *
//* row of the dialog, and while this is ordinarily the correct thing to do,   *
//* we want the Dual-Win title to be in the same position as the Single-Win    *
//* title; therefore, we synthesize the title relative to the MenuBar position,*
//* So long as the title length and MenuBar position do not change this should *
//* work, but if they change, adjust the constant definitions below.           *
//******************************************************************************

void FileMangler::DialogTitle ( bool showMenu, short trgMenu, bool establish )
{
   //* Create the dialog title string *
   gString  gs( "%s%s (c)%s  [Press F2 for Menu]  ", 
                AppTitle1, AppVersion, copyrightYears ) ;

   const short TITLE_YPOS  = ZERO,
               TITLE_XPOS  = this->menuBase.xpos - 2,
               TITLE_WIDTH = gs.gscols() ;
   const char* const Pad = "                                        "
                           "                                        " ;

   //* If dialog has just been instantiated, group the DialogMenuwin controls  *
   //* to create a Menu Bar, and establish the dialog title.                   *
   if ( establish != false )
   {
      //* Create a Menuwin group (MenuBar) and make it initially invisible  *
      if ( (this->dWin->GroupMenuwinControls ( mbList )) == OK )
      {
         //* Attach sub-menus to the primary menu controls *
         const char amwsMsg[] = "AttachMenuwinSubmenus() Error!" ;
         if ( (this->dWin->AttachMenuwinSubmenus ( dwFileMB, mbSubListFile )) != OK )
            this->FmDebugMsg ( amwsMsg, 99 ) ;
         #if 0    // Currently, no sub-menus attached to other primaries
         if ( (this->dWin->AttachMenuwinSubmenus ( dwEditMB, mbSubListEdit )) != OK )
            this->FmDebugMsg ( amwsMsg, 99 ) ;
         if ( (this->dWin->AttachMenuwinSubmenus ( dwViewMB, mbSubListView )) != OK )
            this->FmDebugMsg ( amwsMsg, 99 ) ;
         if ( (this->dWin->AttachMenuwinSubmenus ( dwHelpMB, mbSubListHelp )) != OK )
            this->FmDebugMsg ( amwsMsg, 99 ) ;
         #endif   // Currently, no sub-menus attached to other primaries

         //* Establish the dialog title position and *
         //* enclose its space (see note above).     *
         this->titlePos.ypos = TITLE_YPOS ;
         this->titlePos.xpos = TITLE_XPOS ;
         wchar_t rb[2] = { wcsLTEE, NULLCHAR } ;
         gString gsb ;
         gsb.compose( L"%C%s", &wcsRTEE, Pad ) ;
         gsb.limitCols( TITLE_WIDTH + 1 ) ;
         gsb.append( rb ) ;
         this->dWin->WriteString ( titlePos.ypos, titlePos.xpos - 1, gsb, 
                                   this->cfgOpt.cScheme.bb ) ;

         if ( ! this->lockMenuBar )
         {  //* Make the MenuBar inactive and invisible.*
            this->dWin->HideMenuBar ( dwFileMB ) ;

            //* Display the dialog title *
            this->dWin->WriteString ( this->titlePos, gs, this->titleColor, true ) ;
         }
         else
         {  //* Deactivate MenuBar's controls and leave it visible.*
            //* (Caller is responsible for screen refresh.)        *
            for ( short menuIndex = dwFileMB ; menuIndex <= dwHelpMB ; menuIndex++ )
               this->dWin->ControlActive ( menuIndex, false ) ;
         }
      }
      else
         this->FmDebugMsg ( "GroupMenuwinControls() failed!", 99 ) ;
   }

   //* Menu Bar and dialog title previously established *
   else
   {
      //* In order to modify the settings of the MenuBar *
      //* controls, they must not have the input focus.  *
      short targetControl = this->fdPtr == this->fWin ? dwFileSE : dwFileSEr, 
            cIndex ;
      while ( (cIndex = this->dWin->GetCurrControl ()) != targetControl )
         this->dWin->PrevControl () ;

      //* If Menu Bar is to be displayed *
      if ( showMenu )
      {
         short targetMenu ;
         switch ( trgMenu )
         {
            case MAX_DIALOG_CONTROLS: case -1: targetMenu = dwFileMB ; break ;
            default: targetMenu = trgMenu ;  break ;
         }

         //* Clear the title area and make MenuBar visible.*
         if ( ! (this->dWin->MenuBarVisible ( targetMenu )) )
         {
            gString gsb( Pad ) ;
            gsb.limitCols( TITLE_WIDTH ) ;
            attr_t spaceColor = this->cfgOpt.cScheme.scheme == ncbcGY ?
                                this->cfgOpt.cScheme.mf : this->titleColor ;
            this->dWin->WriteString ( titlePos, gsb, spaceColor, true ) ;
            this->dWin->ShowMenuBar ( targetMenu, true ) ;
         }
         else
         {  //* MenuBar already visible - activate its controls *
            //* and place focus on specified control.           *
            for ( short menuIndex = dwFileMB ; menuIndex <= dwHelpMB ; menuIndex++ )
               this->dWin->ControlActive ( menuIndex, true ) ;
            while ( (this->dWin->NextControl ()) != targetMenu ) ;
            //* (Caller is responsible for screen refresh if needed.) *
         }
      }
      //* Else, dialog title is to be displayed *
      else
      {
         if ( this->lockMenuBar == false )
         {
            if ( this->dWin->MenuBarVisible ( dwFileMB ) )
               this->dWin->HideMenuBar ( dwFileMB ) ;
            this->dWin->WriteString ( titlePos.ypos, titlePos.xpos, gs, 
                                   this->titleColor, true ) ;
         }
         else
         {
            for ( short menuIndex = dwFileMB ; menuIndex <= dwHelpMB ; menuIndex++ )
               this->dWin->ControlActive ( menuIndex, false ) ;
            //* (Caller is responsible for screen refresh if needed.) *
         }
      }
   }
}  //* End DialogTitle() *

//***********************
//* RefreshAltFileList  *
//***********************
//******************************************************************************
//* If currently operating in Dual-Window mode, AND                            *
//* 1. the active file-display control references the same directory as the    *
//*    inactive (alternate) file-display control.                              *
//*    OR                                                                      *
//* 2. the inactive file-display control was the source directory for a        *
//*    cut-and-paste, delete or trashcan operation, (we can know this here     *
//*    because clipboard will have been cleared, so forceRefresh will be true) *
//*    OR                                                                      *
//* 3) If caller wants to set alternate window's CWD to active window's CWD.   *
//*                                                                            *
//* update the file list of the alternate control to reflect any changes made. *
//*                                                                            *
//* Input Values : forceRefresh: (optional, false by default)                  *
//*                       if 'true' force an update of the alternate control   *
//*                synchAlt    : (optional, false by default)                  *
//*                       if 'true', then set CWD of the alternate window to   *
//*                       CWD of active window and update alternate control    *
//*                                                                            *
//* Return Value : true if alternate control updated, else false               *
//******************************************************************************

bool FileMangler::RefreshAltFileList ( bool forceRefresh, bool synchAlt )
{
bool     refreshed = false ;

   //* If operating in Dual-Win mode *
   if ( (this->DualwinMode ()) )
   {
      //* If both active and alternate controls reference the *
      //* same directory OR if caller is forcing the refresh  *
      gString fdPath, altPath ;
      this->fdPtr->GetPath ( fdPath ) ;
      this->altPtr->GetPath ( altPath ) ;
      if ( (fdPath == altPath) || (forceRefresh != false) || (synchAlt != false) )
      {
         //* Update file list in alternate control                             *
         //* Because this is not the active window, draw data without highlight*
         //* (clears clipboard if data source == displayed directory)          *
         if ( synchAlt && !(fdPath == altPath) )
         {
            this->altPtr->SetDirectory ( fdPath ) ;
            this->altPtr->RedrawCurrDir ( false ) ; // remove highlight
         }
         else
            this->altPtr->RefreshCurrDir ( false ) ;
         refreshed = true ;
      }
   }
   return refreshed ;

}  //* End RefreshAltFileList() *

//******************************
//* ClearAltFileListSelections *
//******************************
//******************************************************************************
//* If currently operating in Dual-Window mode, AND the inactive (alternate)   *
//* file-display window has one or more files marked as 'selected', clear those*
//* selections.                                                                *
//*                                                                            *
//* Any data currently on the clipboard is not affected.                       *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: true if alternate control updated, else false                     *
//******************************************************************************

bool FileMangler::ClearAltFileListSelections ( void )
{
const char Sel01[] = " Selections in alternate window deselected." ;
bool     updated = false ;

   //* If operating in Dual-Win mode *
   if ( (this->DualwinMode ()) )
   {
      if ( (this->altPtr->IsSelected ( true )) )
      {
         //* One or more files in alternate window have been marked 'selected'.*
         //* De-select all 'selected' files in the window. Clipboard data are  *
         //* not affected.                                                     *
         this->altPtr->RedrawCurrDir ( false, true ) ;
         dtbmData  msgData( Sel01 ) ;
         this->dWin->DisplayTextboxMessage ( dwMsgsTB, msgData ) ;
         updated = true ;
      }
   }
   return updated ;

}  //* End ClearAltFileListSelections() *

//*************************
//*      DualwinMode      *
//*************************
//******************************************************************************
//* Determne whether we are operating in Dual-window Mode.                     *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: 'true'  if operating in Dual-window Mode                          *
//*          'false' if operating in Single-window Mode                        *
//******************************************************************************

bool FileMangler::DualwinMode ( void )
{
   bool dualWin = false ;

   if ( (this->cfgOpt.winMode == wmDUAL_WIN) && (this->altPtr != NULL) )
   {
      dualWin = true ;
   }
   return dualWin ;

}  //* End DualwinMode() *

#if TEST_CS != 0
//******************************************************************************
//* DEBUG ONLY - TEST COLORS OF COLOR SCHEME                                   *
//******************************************************************************
void FileMangler::TestColorSchemes ( void )
{
   const short neededROWS = 38, neededCOLS = 78 ;
   short dialogROWS = this->actualRows - 1, dialogCOLS = this->actualCols,
         ulY = 1, ulX = ZERO, bRows = 7, bCols = 36 ;
   ColorScheme cs ;
   //* Initial parameters for dialog window *
   InitNcDialog dInit( dialogROWS,
                       dialogCOLS,
                       ulY,            // Y offset from upper-left of terminal 
                       ulX,            // X offset from upper-left of terminal 
                       "  Test Color Scheme  ",// dialog title
                       ncltSINGLE,     // border line-style
                       nc.bk,          // border color attribute
                       nc.bw,          // interior color attribute
                       NULL            // pointer to list of control definitions
                     ) ;

   //* Instantiate the dialog window *
   NcDialog* dp = new NcDialog ( dInit ) ;
   //* Open the dialog window *
   if ( (dp->OpenWindow()) == OK )
   {
      if ( dialogROWS >= neededROWS && dialogCOLS >= neededCOLS )
      {
         //* ncbcBK *
         cs.bb = nc.bk ;
         cs.sd = nc.bk ;
         cs.sb = nc.bk & ~ncbATTR ;
         cs.mn = nc.cyR ;
         cs.mf = nc.bk ;
         cs.pn = nc.gyR ;
         cs.pf = nc.reG ;
         cs.tn = nc.bw ;
         cs.tf = nc.bl ;
         winPos wp( ulY + 1, ulX + 2 ) ;
         this->tcsDialog ( dp, wp, bRows, bCols, "ncbcBK", cs ) ;

      cs.bb = nc.reR ;
      cs.sd = nc.reR ;
      cs.sb = nc.gyre | ncbATTR ;
      cs.mn = nc.re ;
      cs.mf = nc.reR ;
      cs.pn = nc.gyR ;
      cs.pf = nc.cyG ;
      cs.tn = nc.bw ;
      cs.tf = nc.bl ;
      wp.xpos += bCols + 1 ;
      this->tcsDialog ( dp, wp, bRows, bCols, "ncbcRE", cs ) ;

      cs.bb = nc.grR ;
      cs.sd = nc.grR ;
      cs.sb = nc.gygr | ncbATTR ;
      cs.mn = nc.gr ;
      cs.mf = nc.grR ;
      cs.pn = nc.gyR ;
      cs.pf = nc.reG ;
      cs.tn = nc.bw ;
      cs.tf = nc.bl ;
      wp.xpos -= bCols + 1 ;
      wp.ypos += bRows ;
      this->tcsDialog ( dp, wp, bRows, bCols, "ncbcGR", cs ) ;

      cs.bb = nc.brR ;
      cs.sd = nc.brR ;
      cs.sb = nc.gybr | ncbATTR ;
      cs.mn = nc.br ;
      cs.mf = nc.brR ;
      cs.pn = nc.gyR ;
      cs.pf = nc.cyG ;
      cs.tn = nc.bw ;
      cs.tf = nc.bl ;
      wp.xpos += bCols + 1 ;
      this->tcsDialog ( dp, wp, bRows, bCols, "ncbcBR", cs ) ;

      cs.bb = nc.blR ;
      cs.sd = nc.blR ;
      cs.sb = nc.gybl | ncbATTR ;
      cs.mn = nc.bl ;
      cs.mf = nc.blR ;
      cs.pn = nc.gyR ;
      cs.pf = nc.reG ;
      cs.tn = nc.bw ;
      cs.tf = nc.gr ;
      wp.xpos -= bCols + 1 ;
      wp.ypos += bRows ;
      this->tcsDialog ( dp, wp, bRows, bCols, "ncbcBL", cs ) ;

      cs.bb = nc.maR ;
      cs.sd = nc.maR ;
      cs.sb = nc.gyma | ncbATTR ;
      cs.mn = nc.ma ;
      cs.mf = nc.maR ;
      cs.pn = nc.gyR ;
      cs.pf = nc.reG ;
      cs.tn = nc.bw ;
      cs.tf = nc.bl ;
      wp.xpos += bCols + 1 ;
      this->tcsDialog ( dp, wp, bRows, bCols, "ncbcMA", cs ) ;

      cs.bb = nc.cyR ;
      cs.sd = nc.cyR ;
      cs.sb = nc.gycy | ncbATTR ;
      cs.mn = nc.cy ;
      cs.mf = nc.cyR ;
      cs.pn = nc.gyR ;
      cs.pf = nc.reG ;
      cs.tn = nc.bw ;
      cs.tf = nc.bl ;
      wp.xpos -= bCols + 1 ;
      wp.ypos += bRows ;
      this->tcsDialog ( dp, wp, bRows, bCols, "ncbcCY", cs ) ;

      cs.bb = nc.gyR ;
      cs.sd = nc.gyR ;
      cs.sb = nc.bkgy ;
      cs.mn = nc.cy ;
      cs.mf = nc.cyR ;
      cs.pn = nc.bw ;
      cs.pf = nc.reG ;
      cs.tn = nc.bw ;
      cs.tf = nc.bl ;
      wp.xpos += bCols + 1 ;
      this->tcsDialog ( dp, wp, bRows, bCols, "ncbcGY", cs ) ;

      cs.bb = nc.cyR ;
      cs.sd = nc.blR ;
      cs.sb = nc.gybl | ncbATTR ;
      cs.mn = nc.blR ;
      cs.mf = nc.cyR ;
      cs.pn = nc.gyR ;
      cs.pf = nc.reG ;
      cs.tn = nc.bw ;
      cs.tf = nc.gr ;
      wp.xpos -= bCols + 1 ;
      wp.ypos += bRows ;
      this->tcsDialog ( dp, wp, bRows, bCols, "Application Default", cs ) ;

      cs.bb = nc.bw ;
      cs.sd = nc.cyR ;
      cs.sb = nc.gycy | ncbATTR ;
      cs.mn = nc.gyR ;
      cs.mf = nc.bw ;
      cs.pn = nc.gyR ;
      cs.pf = nc.reG ;
      cs.tn = nc.bw ;
      cs.tf = nc.blR ;
      wp.xpos += bCols + 1 ;
      this->tcsDialog ( dp, wp, bRows, bCols, "Terminal Default", cs ) ;
      }
      else
      {
         gString gs ;
         short r = neededROWS + 1 ;
         gs.compose( L" This test requires a big window,\n"
                      "  at least %02hd rows x %02hd columns  ", 
                      &r, &neededCOLS ) ;
         dp->WriteParagraph ( 2, 2, gs, nc.blG, true ) ;
      }
      nckPause() ;
   }
   if ( dp != NULL )        // close the dialog window
      delete ( dp ) ;

}  //* End TestColorSchemes() *

//******************************************************************************
//* DEBUG ONLY - TEST COLORS OF COLOR SCHEME                                   *
//******************************************************************************
void FileMangler::tcsDialog ( NcDialog*dp, winPos wp, short bRows, short bCols, 
                 const char* title, const ColorScheme& cAttr )
{
   const wchar_t rbf[] = { '<', ' ', wcsDIAMOND, ' ', '>', NULLCHAR }, 
                 rbn[] = { '<', ' ', ' ', ' ', '>', NULLCHAR } ;
   dp->DrawBox ( wp.ypos, wp.xpos, bRows, bCols, cAttr.bb, title ) ;
   char clr[MAX_DIALOG_WIDTH] ;
   short i ;
   for ( i = ZERO ; i < (bCols - 2) ; i++ )
      clr[i] = SPACE ;
   clr[i] = NULLCHAR ;
   for ( i = 1 ; i < (bRows - 1) ; i++ )
      dp->WriteString ( wp.ypos + i, wp.xpos + 1, clr, cAttr.sd ) ;

   winPos wpx( wp.ypos + 1, wp.xpos + 2 ) ;
   dp->WriteString ( wpx.ypos++, wpx.xpos, "tnColor  ", cAttr.tn ) ;
   dp->WriteString ( wpx.ypos++, wpx.xpos, "tfColor  ", cAttr.tf ) ;
   dp->WriteString ( wpx.ypos++, wpx.xpos, " pnColor ", cAttr.pn ) ;
   dp->WriteString ( wpx.ypos++, wpx.xpos, " pfColor ", cAttr.pf ) ;
   wpx = dp->WriteString ( wpx.ypos, (wpx.xpos-1), rbn, cAttr.sd ) ;
   wpx = dp->WriteString ( wpx.ypos, (wpx.xpos+1), rbf, cAttr.tf ) ;
   wpx = dp->WriteString ( wpx.ypos, (wpx.xpos+1), rbn, cAttr.tn ) ;
   wpx = dp->WriteString ( wpx.ypos, (wpx.xpos+1), rbf, cAttr.pf ) ;

   wpx = { short(wp.ypos + 1), short(wp.xpos + 14) } ;
#if 1    // NEW
   winPos wp2 = dp->WriteString ( wpx.ypos, wpx.xpos, " mnColor ", cAttr.mn ) ;
   wp2.xpos += 3 ;
   dp->WriteString ( wp2, " sd norm ", cAttr.sd ) ;
   ++wp2.ypos ;
   dp->WriteString ( wp2, " sb bold ", cAttr.sb ) ;
   ++wpx.ypos ;
#else    // OLD
   dp->WriteString ( wpx.ypos++, wpx.xpos, " mnColor ", cAttr.mn ) ;
#endif   // OLD

   //* Application default scheme is slightly modified from the pattern *
   attr_t menuInterior = cAttr.mf ;
   if ( cAttr.bb == nc.cyR && cAttr.sd != cAttr.bb )
      menuInterior = cAttr.mn ;
   dp->DrawBox ( wpx.ypos++, wpx.xpos, 3, 11, cAttr.mn ) ;
   dp->WriteString ( wpx.ypos, (wpx.xpos+1), " mfColor ", menuInterior ) ;
   wpx.xpos += 10 ;
   dp->DrawBox ( wpx.ypos++, wpx.xpos, 3, 11, cAttr.mf ) ;
   dp->WriteString ( wpx.ypos, (wpx.xpos+1), " mfColor ", menuInterior ) ;

   dp->RefreshWin () ;

}  // End tcsDialog() *
#endif   // TEST_CS

//*************************
//*                       *
//*************************
//******************************************************************************
//*                                                                            *
//*                                                                            *
//*                                                                            *
//* Input  :                                                                   *
//*                                                                            *
//* Returns:                                                                   *
//******************************************************************************

