//********************************************************************************
//* File       : FmKeymap.hpp                                                    *
//* Author     : Mahlon R. Smith                                                 *
//*              Copyright (c) 2005-2025 Mahlon R. Smith, The Software Samurai   *
//*                  GNU GPL copyright notice located in FileMangler.hpp         *
//* Date       : 09-Apr-2025                                                     *
//* Version    : (see AppVersion string)                                         *
//*                                                                              *
//* Description: Definitions and text explanations for the FileMangler           *
//*              command-key map. These definitions are used during              *
//*              configuration and during application startup to initialize      *
//*              user-defined command-key assignments.                           *
//********************************************************************************
//* Notes on command-key mapping:                                                *
//* -----------------------------                                                *
//*                                                                              *
//* The list of valid command-key sequences is handled in several places, in     *
//* several ways, and for different purposes.                                    *
//*                                                                              *
//* 1) The FmKeymap.hpp header file. The FmConfig utility controls this master   *
//*    list which defines the valid command-key sequences.                       *
//*    - "const wkeyCode KeymapCfgDflt[]"  This is the controlling list.         *
//*      All other lists of command keys must conform.                           *
//*    - The file FmKeymap.cfg (discussed below) may be generated from the       *
//*      definitions in this file, except for the static file header             *
//*      information which is located at the top of FmConfig.cpp.                *
//*                                                                              *
//* 2) The FmKeymap.cfg - This is the default configuration file used by the     *
//*    configuration utility (or manually by the user) to specify custom         *
//*    key mapping.                                                              *
//*    - This file (or user-specified alternate definition file) must reside     *
//*      in the same directory as the main configuration file.                   *
//*    - During interactive configuration, the user's selections will modify     *
//*      FmKeymap.cfg.                                                           *
//*    - If command-key remapping is enabled in the main configuration file      *
//*      (FileMangler.cfg), then FmKeymap.cfg (or other specified filename) is   *
//*      read and validated on startup. If no remap config file is specified,    *
//*      then the startup sequence will not read this file and will not          *
//*      populate the memory-resident keymap. This makes the startup sequence    *
//*      more efficient.                                                         *
//*    - This file does not list every key combination that COULD THEORETICALLY  *
//*      be used as a command key. Instead, it lists the valid key combinations  *
//*      that are _likely_ to be available to console applications, (with the    *
//*      questionable keycodes clearly marked).                                  *
//*    - The user may only assign operations to the command-key combinations     *
//*      listed in this file. This is ensured by FmConfig during startup.        *
//*                                                                              *
//* 3) The FileMangler::keyMap data member of the FileMangler class is an        *
//*    instance of the RemapKey class.                                           *
//*    - If key mapping is active, then the 'kmActive' member of the 'keyMap'    *
//*      object will be set, and the default commands will be loaded into the    *
//*      table along with any remapped keys.                                     *
//*    - If key mapping is not active, then 'kmActive' will be reset, the table  *
//*      will not be initialized, and no mapping routines will be called.        *
//*                                                                              *
//* 4) The UserInterface() method which actually executes the commands           *
//*    associated with the active command-key sequences.                         *
//*    - This method knows only the key combinations that are actually used.     *
//*      It is the job of the RemapKey::decodeKeyInput() method to translate     *
//*      any user-specified command-key sequences to the default (hard-coded)    *
//*      key sequences used to execute the operations.                           *
//*                                                                              *
//* 5) The Texinfo documentation lists essentially the same list of command-key  *
//*    combinations as FmKeymap.hpp (discussed above), but with additional       *
//*    details and recommendations.                                              *
//*                                                                              *
//* It is important that all these lists remain in synch to avoid user           *
//* confusion and programmer embarrassment.                                      *
//********************************************************************************

#ifndef FMKEYMAP_DEFINED      // be sure file is included only once
#define FMKEYMAP_DEFINED

//****************************
//* Command-key definitions. *
//****************************

const short KEYMAP_KEYS = 238 ;              // number of valid command keys defined
const short KEYMAP_DLGX = 80 ;               // columns in keymap edit dialog
const short KEYMAP_COLS = KEYMAP_DLGX - 2 ;  // display columns for keymap edit entry
const short KEYMAP_TEXT = 96 ;               // bytes of storage for keymap edit entry
const wkeyCode nonCode( nckNULLCHAR, wktPRINT ) ;  // placeholder keycode definition
const char* const kmcfgName = "FmKeymap.cfg" ; // name of keymap configuration file

//* Default command keycode definitions.                     *
//* This list includes all keys which will be recognized by  *
//* the application as valid command-key sequences.          *
//* Keycodes NOT listed may not be assigned as command keys. *
const wkeyCode KeymapCfgDflt[KEYMAP_KEYS] = 
{
   //********************
   //*** CTRL+  group ***
   //********************                                                 OFFSET
   { nckC_A,     wktFUNKEY },    // CTRL+A                                 000
   { nckC_B,     wktFUNKEY },    // CTRL+B
   { nckC_C,     wktFUNKEY },    // CTRL+C
   { nckC_D,     wktFUNKEY },    // CTRL+D
   { nckC_E,     wktFUNKEY },    // CTRL+E
   { nckC_F,     wktFUNKEY },    // CTRL+F
   { nckC_G,     wktFUNKEY },    // CTRL+G
   { nckC_H,     wktFUNKEY },    // CTRL+H
// { nckTAB,     wktFUNKEY },    // CTRL+I duplicate, unavailable
// { nckENTER,   wktFUNKEY },    // CTRL+J duplicate, unavailable
   { nckC_K,     wktFUNKEY },    // CTRL+K
   { nckC_L,     wktFUNKEY },    // CTRL+L
// { 0x0000,     wktFUNKEY },    // CTRL+M carriage-return, unavailable
   { nckC_N,     wktFUNKEY },    // CTRL+N
   { nckC_O,     wktFUNKEY },    // CTRL+O
   { nckC_P,     wktFUNKEY },    // CTRL+P
   { nckC_Q,     wktFUNKEY },    // CTRL+Q
   { nckC_R,     wktFUNKEY },    // CTRL+R
   { nckC_S,     wktFUNKEY },    // CTRL+S
   { nckC_T,     wktFUNKEY },    // CTRL+T
   { nckC_U,     wktFUNKEY },    // CTRL+U
   { nckC_V,     wktFUNKEY },    // CTRL+V
   { nckC_W,     wktFUNKEY },    // CTRL+W
   { nckC_X,     wktFUNKEY },    // CTRL+X
   { nckC_Y,     wktFUNKEY },    // CTRL+Y
   { nckC_Z,     wktFUNKEY },    // CTRL+Z
                                                                       // OFFSET
// { nckENTER,   wktFUNKEY },    // CTRL+ENTER  duplicate, unavailable
// { 0x0000,     wktFUNKEY },    // CTRL+INSERT unseen, unavailable
   { nckCDEL,    wktFUNKEY },    // CTRL+DELETE                            023
   { nckCUP,     wktFUNKEY },    // CTRL+UP
   { nckCDOWN,   wktFUNKEY },    // CTRL+DOWN
   { nckCLEFT,   wktFUNKEY },    // CTRL+LEFT
   { nckCRIGHT,  wktFUNKEY },    // CTRL+RIGHT
// { 0x0000,     wktFUNKEY },    // CTRL+PGUP   captured by default
// { 0x0000,     wktFUNKEY },    // CTRL+PGDN   captured by default
// { nckHOME,    wktFUNKEY },    // CTRL+HOME   duplicate, unavailable
// { nckEND,     wktFUNKEY },    // CTRL+END    duplicate, unavailable
                                                                       // OFFSET
// { nckCF01,    wktFUNKEY },    // CTRL+F01    not seen, unavailable
   { nckCF02,    wktFUNKEY },    // CTRL+F02                               028
   { nckCF03,    wktFUNKEY },    // CTRL+F03
   { nckCF04,    wktFUNKEY },    // CTRL+F04
   { nckCF05,    wktFUNKEY },    // CTRL+F05
   { nckCF06,    wktFUNKEY },    // CTRL+F06
   { nckCF07,    wktFUNKEY },    // CTRL+F07
   { nckCF08,    wktFUNKEY },    // CTRL+F08
   { nckCF09,    wktFUNKEY },    // CTRL+F09
   { nckCF10,    wktFUNKEY },    // CTRL+F10
   { nckCF11,    wktFUNKEY },    // CTRL+F11
   { nckCF12,    wktFUNKEY },    // CTRL+F12

   //*******************
   //*** ALT+  group ***
   //*******************                                                  OFFSET
   { nckA_A,     wktEXTEND },    // ALT+A                                  039
   { nckA_B,     wktEXTEND },    // ALT+B
   { nckA_C,     wktEXTEND },    // ALT+C
   { nckA_D,     wktEXTEND },    // ALT+D
   { nckA_E,     wktEXTEND },    // ALT+E
   { nckA_F,     wktEXTEND },    // ALT+F
   { nckA_G,     wktEXTEND },    // ALT+G
   { nckA_H,     wktEXTEND },    // ALT+H
   { nckA_I,     wktEXTEND },    // ALT+I
   { nckA_J,     wktEXTEND },    // ALT+J
   { nckA_K,     wktEXTEND },    // ALT+K
   { nckA_L,     wktEXTEND },    // ALT+L
   { nckA_M,     wktEXTEND },    // ALT+M
   { nckA_N,     wktEXTEND },    // ALT+N
   { nckA_O,     wktEXTEND },    // ALT+O
   { nckA_P,     wktEXTEND },    // ALT+P
   { nckA_Q,     wktEXTEND },    // ALT+Q
   { nckA_R,     wktEXTEND },    // ALT+R
   { nckA_S,     wktEXTEND },    // ALT+S
   { nckA_T,     wktEXTEND },    // ALT+T
   { nckA_U,     wktEXTEND },    // ALT+U
   { nckA_V,     wktEXTEND },    // ALT+V
   { nckA_W,     wktEXTEND },    // ALT+W
   { nckA_X,     wktEXTEND },    // ALT+X
   { nckA_Y,     wktEXTEND },    // ALT+Y
   { nckA_Z,     wktEXTEND },    // ALT+Z
                                                                       // OFFSET
   { nckA_0,     wktEXTEND },    // ALT+0                                  065
   { nckA_1,     wktEXTEND },    // ALT+1
   { nckA_2,     wktEXTEND },    // ALT+2
   { nckA_3,     wktEXTEND },    // ALT+3
   { nckA_4,     wktEXTEND },    // ALT+4
   { nckA_5,     wktEXTEND },    // ALT+5
   { nckA_6,     wktEXTEND },    // ALT+6
   { nckA_7,     wktEXTEND },    // ALT+7
   { nckA_8,     wktEXTEND },    // ALT+8
   { nckA_9,     wktEXTEND },    // ALT+9
                                                                       // OFFSET
   { nckA_MINU,  wktEXTEND },    // ALT+-                                  075
   { nckA_EQUL,  wktEXTEND },    // ALT+=
   { nckA_LBRK,  wktEXTEND },    // ALT+[  (caution: may be confused with ESC key)",
   { nckA_RBRK,  wktEXTEND },    // ALT+]
   { nckA_BSTR,  wktEXTEND },    // ALT+\.
   { nckA_SEMI,  wktEXTEND },    // ALT+;
   { nckA_SQUO,  wktEXTEND },    // ALT+'
   { nckA_COMM,  wktEXTEND },    // ALT+,
   { nckA_STOP,  wktEXTEND },    // ALT+.
   { nckA_FSTR,  wktEXTEND },    // ALT+/
   { nckA_GRAV,  wktEXTEND },    // ALT+`
                                                                       // OFFSET
   { nckAC_J,    wktEXTEND },    // ALT+ENTER                              086
   { nckAINSERT, wktFUNKEY },    // ALT+INSERT
   { nckADEL,    wktFUNKEY },    // ALT+DELETE
   { nckAUP,     wktFUNKEY },    // ALT+UP
   { nckADOWN,   wktFUNKEY },    // ALT+DOWN
   { nckALEFT,   wktFUNKEY },    // ALT+LEFT
   { nckARIGHT,  wktFUNKEY },    // ALT+RIGHT
   { nckAPGUP,   wktFUNKEY },    // ALT+PAGEUP
   { nckAPGDN,   wktFUNKEY },    // ALT+PAGEDN
// { nckHOME,    wktFUNKEY },    // ALT+HOME    duplicate, unavailable
// { nckEND,     wktFUNKEY },    // ALT+END     duplicate, unavailable

   //*********************
   //*** SHIFT+  group ***
   //*********************                                                OFFSET
   { nckSDEL,    wktFUNKEY },    // SHIFT+DELETE                           095
   { nckSTAB,    wktFUNKEY },    // SHIFT+TAB
   { nckSUP,     wktFUNKEY },    // SHIFT+UP
   { nckSDOWN,   wktFUNKEY },    // SHIFT+DOWN
   { nckSLEFT,   wktFUNKEY },    // SHIFT+LEFT
   { nckSRIGHT,  wktFUNKEY },    // SHIFT+RIGHT
                                                                       // OFFSET
   { nckSF01,    wktFUNKEY },    // CTRL+F01                               101
   { nckSF02,    wktFUNKEY },    // CTRL+F02
   { nckSF03,    wktFUNKEY },    // CTRL+F03
   { nckSF04,    wktFUNKEY },    // CTRL+F04
   { nckSF05,    wktFUNKEY },    // CTRL+F05
   { nckSF06,    wktFUNKEY },    // CTRL+F06
   { nckSF07,    wktFUNKEY },    // CTRL+F07
   { nckSF08,    wktFUNKEY },    // CTRL+F08
   { nckSF09,    wktFUNKEY },    // CTRL+F09
   { nckSF10,    wktFUNKEY },    // CTRL+F10
   { nckSF11,    wktFUNKEY },    // CTRL+F11
   { nckSF12,    wktFUNKEY },    // CTRL+F12

   //************************
   //*** ALT+SHIFT  group ***
   //************************                                             OFFSET
   { nckAS_A,    wktEXTEND },    // ALT+SHIFT+A                            113
   { nckAS_B,    wktEXTEND },    // ALT+SHIFT+B
   { nckAS_C,    wktEXTEND },    // ALT+SHIFT+C
   { nckAS_D,    wktEXTEND },    // ALT+SHIFT+D
   { nckAS_E,    wktEXTEND },    // ALT+SHIFT+E
   { nckAS_F,    wktEXTEND },    // ALT+SHIFT+F
   { nckAS_G,    wktEXTEND },    // ALT+SHIFT+G
   { nckAS_H,    wktEXTEND },    // ALT+SHIFT+H
   { nckAS_I,    wktEXTEND },    // ALT+SHIFT+I
   { nckAS_J,    wktEXTEND },    // ALT+SHIFT+J
   { nckAS_K,    wktEXTEND },    // ALT+SHIFT+K
   { nckAS_L,    wktEXTEND },    // ALT+SHIFT+L
   { nckAS_M,    wktEXTEND },    // ALT+SHIFT+M
   { nckAS_N,    wktEXTEND },    // ALT+SHIFT+N
   { nckAS_O,    wktEXTEND },    // ALT+SHIFT+O
   { nckAS_P,    wktEXTEND },    // ALT+SHIFT+P
   { nckAS_Q,    wktEXTEND },    // ALT+SHIFT+Q
   { nckAS_R,    wktEXTEND },    // ALT+SHIFT+R
   { nckAS_S,    wktEXTEND },    // ALT+SHIFT+S
   { nckAS_T,    wktEXTEND },    // ALT+SHIFT+T
   { nckAS_U,    wktEXTEND },    // ALT+SHIFT+U
   { nckAS_V,    wktEXTEND },    // ALT+SHIFT+V
   { nckAS_W,    wktEXTEND },    // ALT+SHIFT+W
   { nckAS_X,    wktEXTEND },    // ALT+SHIFT+X
   { nckAS_Y,    wktEXTEND },    // ALT+SHIFT+Y
   { nckAS_Z,    wktEXTEND },    // ALT+SHIFT+Z
                                                                       // OFFSET
   { nckAS_0,    wktEXTEND },    // ALT+SHIFT+0                            139
   { nckAS_1,    wktEXTEND },    // ALT+SHIFT+1
   { nckAS_2,    wktEXTEND },    // ALT+SHIFT+2
   { nckAS_3,    wktEXTEND },    // ALT+SHIFT+3
   { nckAS_4,    wktEXTEND },    // ALT+SHIFT+4
   { nckAS_5,    wktEXTEND },    // ALT+SHIFT+5
   { nckAS_6,    wktEXTEND },    // ALT+SHIFT+6
   { nckAS_7,    wktEXTEND },    // ALT+SHIFT+7
   { nckAS_8,    wktEXTEND },    // ALT+SHIFT+8
   { nckAS_9,    wktEXTEND },    // ALT+SHIFT+9
                                                                       // OFFSET
   { nckA_UNDR,  wktEXTEND },    // ALT+SHIFT+-                            149
   { nckA_PLUS,  wktEXTEND },    // ALT+SHIFT+=
   { nckA_LBRC,  wktEXTEND },    // ALT+SHIFT+[
   { nckA_RBRC,  wktEXTEND },    // ALT+SHIFT+]
   { nckA_VBAR,  wktEXTEND },    // ALT+SHIFT+\.
   { nckA_COLO,  wktEXTEND },    // ALT+SHIFT+;
   { nckA_DQUO,  wktEXTEND },    // ALT+SHIFT+'
   { nckA_LCHV,  wktEXTEND },    // ALT+SHIFT+,
   { nckA_RCHV,  wktEXTEND },    // ALT+SHIFT+.
   { nckA_QUES,  wktEXTEND },    // ALT+SHIFT+/
                                                                       // OFFSET
// { nckAC_J,    wktEXTEND },    // ALT+SHIFT+ENTER   duplicate, unavailable
// { 0x0000,     wktFUNKEY },    // ALT+SHIFT+INSERT  unavailable
   { nckASDEL,   wktFUNKEY },    // ALT+SHIFT+DELETE                       159
   { nckASUP,    wktFUNKEY },    // ALT+SHIFT+UP
   { nckASDOWN,  wktFUNKEY },    // ALT+SHIFT+DOWN
   { nckASLEFT,  wktFUNKEY },    // ALT+SHIFT+LEFT
   { nckASRIGHT, wktFUNKEY },    // ALT+SHIFT+RIGHT
// { 0x0000,     wktFUNKEY },    // ALT+SHIFT+PAGEUP
// { 0x0000,     wktFUNKEY },    // ALT+SHIFT+PAGEDN
// { 0x00000,    wktFUNKEY },    // ALT+SHIFT+HOME    duplicate, unavailable
// { 0x0000,     wktFUNKEY },    // ALT+SHIFT+END     duplicate, unavailable

   //***********************
   //*** ALT+CTRL  group ***
   //***********************                                              OFFSET
   { nckAC_A,    wktEXTEND },    // ALT+CTRL+A                             164
   { nckAC_B,    wktEXTEND },    // ALT+CTRL+B
   { nckAC_C,    wktEXTEND },    // ALT+CTRL+C
   { nckAC_D,    wktEXTEND },    // ALT+CTRL+D
   { nckAC_E,    wktEXTEND },    // ALT+CTRL+E
   { nckAC_F,    wktEXTEND },    // ALT+CTRL+F
   { nckAC_G,    wktEXTEND },    // ALT+CTRL+G
   { nckAC_H,    wktEXTEND },    // ALT+CTRL+H
   { nckAC_I,    wktEXTEND },    // ALT+CTRL+I
// { nckAC_J,    wktEXTEND },    // ALT+CTRL+J  (ALT+ENTER not available as a command key)",
   { nckAC_K,    wktEXTEND },    // ALT+CTRL+K
   { nckAC_L,    wktEXTEND },    // ALT+CTRL+L
// { nckAC_M,    wktEXTEND },    // ALT+CTRL+M  (ALT+ENTER not available as a command key)",
   { nckAC_N,    wktEXTEND },    // ALT+CTRL+N
   { nckAC_O,    wktEXTEND },    // ALT+CTRL+O
   { nckAC_P,    wktEXTEND },    // ALT+CTRL+P
   { nckAC_Q,    wktEXTEND },    // ALT+CTRL+Q
   { nckAC_R,    wktEXTEND },    // ALT+CTRL+R
   { nckAC_S,    wktEXTEND },    // ALT+CTRL+S
   { nckAC_T,    wktEXTEND },    // ALT+CTRL+T
   { nckAC_U,    wktEXTEND },    // ALT+CTRL+U
   { nckAC_V,    wktEXTEND },    // ALT+CTRL+V
   { nckAC_W,    wktEXTEND },    // ALT+CTRL+W
   { nckAC_X,    wktEXTEND },    // ALT+CTRL+X
   { nckAC_Y,    wktEXTEND },    // ALT+CTRL+Y
   { nckAC_Z,    wktEXTEND },    // ALT+CTRL+Z
                                                                       // OFFSET
// { 0x0000,     wktFUNKEY },    // ALT+CTRL+INSERT  unavailable
// { 0x0000,     wktFUNKEY },    // ALT+CTRL+DELETE  unavailable
// { 0x0000,     wktFUNKEY },    // ALT+CTRL+UP      unavailable
// { 0x0000,     wktFUNKEY },    // ALT+CTRL+DOWN    unavailable
// { 0x0000,     wktFUNKEY },    // ALT+CTRL+LEFT    unavailable
// { 0x0000,     wktFUNKEY },    // ALT+CTRL+RIGHT   unavailable
   { nckACPGUP,  wktFUNKEY },    // ALT+CTRL+PAGEUP                        188
   { nckACPGDN,  wktFUNKEY },    // ALT+CTRL+PAGEDN
// { 0x0000,     wktFUNKEY },    // ALT+CTRL+HOME    unavailable
// { 0x0000,     wktFUNKEY },    // ALT+CTRL+END     unavailable

   //*********************
   //*** KEYPAD  group ***
   //*********************                                                OFFSET
   { nckpCENTER, wktFUNKEY },    // KEYPAD+CENTER                          190
   { nckpPLUS,   wktFUNKEY },    // KEYPAD++
   { nckpMINUS,  wktFUNKEY },    // KEYPAD+-
   { nckpMULT,   wktFUNKEY },    // KEYPAD+*
   { nckpDIV,    wktFUNKEY },    // KEYPAD+/
   { nckpCPGUP,  wktFUNKEY },    // KEYPAD+CTRL+PAGEUP
   { nckpCPGDN,  wktFUNKEY },    // KEYPAD+CTRL+PAGEDN
   { nckpACUP,   wktFUNKEY },    // KEYPAD+ALT+CTRL+UP
   { nckpACDOWN, wktFUNKEY },    // KEYPAD+ALT+CTRL+DOWN
   { nckpACLEFT, wktFUNKEY },    // KEYPAD+ALT+CTRL+LEFT
   { nckpACRIGHT,wktFUNKEY },    // KEYPAD+ALT+CTRL+RIGHT
   { nckpACDEL,  wktFUNKEY },    // KEYPAD+ALT+CTRL+DELETE                 201
   // (all other keypad combinations are duplicates of main keyboard keys)

  //***************************
  //*** Special-codes group ***
  //***************************
// { nckSPACE,   wktPRINT  },    // SPACE  cannot be remapped
// { nckTAB,     wktFUNKEY },    // TAB    cannot be remapped
// { nckENTER,   wktFUNKEY },    // ENTER  cannot be remapped

   //********************
   //*** F-KEY  group ***
   //********************                                                 OFFSET
   { nckF01,     wktFUNKEY },    // F01 Term Help - may be unavailable     202
   { nckF02,     wktFUNKEY },    // F02
   { nckF03,     wktFUNKEY },    // F03
   { nckF04,     wktFUNKEY },    // F04
   { nckF05,     wktFUNKEY },    // F05
   { nckF06,     wktFUNKEY },    // F06
   { nckF07,     wktFUNKEY },    // F07
   { nckF08,     wktFUNKEY },    // F08
   { nckF09,     wktFUNKEY },    // F09
   { nckF10,     wktFUNKEY },    // F10
   { nckF11,     wktFUNKEY },    // F11 Window Resize - may be unavailable
   { nckF12,     wktFUNKEY },    // F12
                                                                       // OFFSET
   { nckSF01,    wktFUNKEY },    // SHIFT+F01                              214
   { nckSF02,    wktFUNKEY },    // SHIFT+F02
   { nckSF03,    wktFUNKEY },    // SHIFT+F03
   { nckSF04,    wktFUNKEY },    // SHIFT+F04
   { nckSF05,    wktFUNKEY },    // SHIFT+F05
   { nckSF06,    wktFUNKEY },    // SHIFT+F06
   { nckSF07,    wktFUNKEY },    // SHIFT+F07
   { nckSF08,    wktFUNKEY },    // SHIFT+F08
   { nckSF09,    wktFUNKEY },    // SHIFT+F09
   { nckSF10,    wktFUNKEY },    // SHIFT+F10
   { nckSF11,    wktFUNKEY },    // SHIFT+F11
   { nckSF12,    wktFUNKEY },    // SHIFT+F12                              225
                                                                       // OFFSET
   { nckCSF01,   wktFUNKEY },    // CTRL+SHIFT+F01                         226
   { nckCSF02,   wktFUNKEY },    // CTRL+SHIFT+F02
   { nckCSF03,   wktFUNKEY },    // CTRL+SHIFT+F03
   { nckCSF04,   wktFUNKEY },    // CTRL+SHIFT+F04
   { nckCSF05,   wktFUNKEY },    // CTRL+SHIFT+F05
   { nckCSF06,   wktFUNKEY },    // CTRL+SHIFT+F06
   { nckCSF07,   wktFUNKEY },    // CTRL+SHIFT+F07
   { nckCSF08,   wktFUNKEY },    // CTRL+SHIFT+F08
   { nckCSF09,   wktFUNKEY },    // CTRL+SHIFT+F09
   { nckCSF10,   wktFUNKEY },    // CTRL+SHIFT+F10
   { nckCSF11,   wktFUNKEY },    // CTRL+SHIFT+F11
   { nckCSF12,   wktFUNKEY },    // CTRL+SHIFT+F12                         237
} ;

//* Descriptions used to create a Keymap Configuration file. *
//* See the top of FmConfig.cpp for the static header text.  *
//* Also used for descriptions for 'ecoEditKM()' method.     *
const char* const KeymapCfgDesc[KEYMAP_KEYS] = 
{
   //********************
   //*** CTRL+  group ***
   //********************
   "CTRL+A        Select/deselect all files in directory",
   "CTRL+B        unassigned by default",
   "CTRL+C        Copy selected files to clipboard",
   "CTRL+D        Date (timestamp) modification",
   "CTRL+E        write-Enable selected file(s)",
   "CTRL+F        Find files by name in directory tree",
   "CTRL+G        unassigned by default",
   "CTRL+H        unassigned by default",
// "CTRL+I        (not available as a command key)",
// "CTRL+J        (not available as a command key)",
   "CTRL+K        display Keycode map",
   "CTRL+L        unassigned by default",
// "CTRL+M        (not available as a command key)",
   "CTRL+N        create New subdirectory",
   "CTRL+O        Open the 'Favorites' dialog",
   "CTRL+P        unassigned by default",
   "CTRL+Q        Quit the application (see also ALT+Q)",
   "CTRL+R        Rename selected file(s)",
   "CTRL+S        Sort-files-by option",
   "CTRL+T        Tree-view mode",
   "CTRL+U        Update (refresh) the list of displayed files",
   "CTRL+V        Paste from clipboard",
   "CTRL+W        Write-protect selected file(s)",
   "CTRL+X        Cut selected file(s) to clipboard",
   "CTRL+Y        open the 'Mount Filesystem' dialog",
   "CTRL+Z        Undo most recent copy or move operation",

// "CTRL+ENTER    (same as ENTER key)",
// "CTRL+INSERT   (unavailable as a command key)",
   "CTRL+DELETE   Move selected file(s) to Trashcan",
   "CTRL+UP       unassigned by default",
   "CTRL+DOWN     unassigned by default",
   "CTRL+LEFT     unassigned by default",
   "CTRL+RIGHT    unassigned by default",
// "CTRL+PGUP     (unavailable as a command key)",
// "CTRL+PGDOWN   (unavailable as a command key)",
// "CTRL+HOME     (unavailable as a command key)",
// "CTRL+END      (unavailable as a command key)",

// "CTRL+F01      (unavailable as a command key)",
   "CTRL+F02      unassigned by default",
   "CTRL+F03      unassigned by default",
   "CTRL+F04      unassigned by default",
   "CTRL+F05      unassigned by default",
   "CTRL+F06      unassigned by default",
   "CTRL+F07      unassigned by default",
   "CTRL+F08      unassigned by default",
   "CTRL+F09      unassigned by default",
   "CTRL+F10      unassigned by default",
   "CTRL+F11      unassigned by default",
   "CTRL+F12      unassigned by default",

   //*******************
   //*** ALT+  group ***
   //*******************
   "ALT+A         unassigned by default",
   "ALT+B         unassigned by default",
   "ALT+C         unassigned by default",
   "ALT+D         unassigned by default",
   "ALT+E         open 'Edit' menu",
   "ALT+F         open 'File' menu",
   "ALT+G         unassigned by default",
   "ALT+H         open 'Help' menu",
   "ALT+I         Toggle visibility of hidden files",
   "ALT+J         unassigned by default",
   "ALT+K         unassigned by default",
   "ALT+L         unassigned by default",
   "ALT+M         unassigned by default",
   "ALT+N         unassigned by default",
   "ALT+O         unassigned by default",
   "ALT+P         unassigned by default",
   "ALT+Q         Quit application to CWD (see also CTRL+Q)",
   "ALT+R         Toggle full scan of root directory.",
   "ALT+S         unassigned by default",
   "ALT+T         unassigned by default",
   "ALT+U         open 'Util' menu",
   "ALT+V         open 'View' menu",
   "ALT+W         Window mode, toggle Single- and Dual-window",
   "ALT+X         unassigned by default",
   "ALT+Y         unassigned by default",
   "ALT+Z         unassigned by default",

   "ALT+0         unassigned by default",
   "ALT+1         unassigned by default",
   "ALT+2         unassigned by default",
   "ALT+3         unassigned by default",
   "ALT+4         unassigned by default",
   "ALT+5         unassigned by default",
   "ALT+6         unassigned by default",
   "ALT+7         unassigned by default",
   "ALT+8         unassigned by default",
   "ALT+9         unassigned by default",

   "ALT+-         unassigned by default",
   "ALT+=         unassigned by default",
   "ALT+[         (caution: may be confused with ESC key)",
   "ALT+]         unassigned by default",
   "ALT+\\         unassigned by default",
   "ALT+;         unassigned by default",
   "ALT+'         Synch-lock toggle for Dual-window Mode",
   "ALT+,         unassigned by default",
   "ALT+.         unassigned by default",
   "ALT+/         unassigned by default",
   "ALT+`         unassigned by default",

   "ALT+ENTER     View-file context menu for directory names",
   "ALT+INSERT    Paste Special",
   "ALT+DELETE    unassigned by default",
   "ALT+UP        Navigate to parent directory",
   "ALT+DOWN      unassigned by default",
   "ALT+LEFT      unassigned by default",
   "ALT+RIGHT     unassigned by default",
   "ALT+PAGEUP    unassigned by default",
   "ALT+PAGEDN    unassigned by default",
// "ALT+HOME      (unavailable as a command key)",
// "ALT+END       (unavailable as a command key)",

   //*********************
   //*** SHIFT+  group ***
   //*********************
   "SHIFT+DELETE  Delete files unconditionally",
   "SHIFT+TAB     Move input focus to previous control",
   "SHIFT+UP      Select current file and previous file",
   "SHIFT+DOWN    Select current file and next file",
   "SHIFT+LEFT    unassigned by default",
   "SHIFT+RIGHT   unassigned by default",

   "SHIFT+F01     Invoke online Help (Info reader)",
   "SHIFT+F02     unassigned by default",
   "SHIFT+F03     unassigned by default",
   "SHIFT+F04     unassigned by default",
   "SHIFT+F05     unassigned by default",
   "SHIFT+F06     unassigned by default",
   "SHIFT+F07     unassigned by default",
   "SHIFT+F08     unassigned by default",
   "SHIFT+F09     unassigned by default",
   "SHIFT+F10     unassigned by default",
   "SHIFT+F11     unassigned by default",
   "SHIFT+F12     unassigned by default",

   //************************
   //*** ALT+SHIFT  group ***
   //************************
   "ALT+SHIFT+A   create, expand or modify an Archive file",
   "ALT+SHIFT+B   Backup files from source to target",
   "ALT+SHIFT+C   Compare files",
   "ALT+SHIFT+D   unassigned by default",
   "ALT+SHIFT+E   open 'Edit' menu",
   "ALT+SHIFT+F   open 'File' menu",
   "ALT+SHIFT+G   scan file(s) for matching substring (Grep)",
   "ALT+SHIFT+H   open 'Help' menu",
   "ALT+SHIFT+I   scan filesystem for files with shared inode",
   "ALT+SHIFT+J   unassigned by default",
   "ALT+SHIFT+K   (reserved for development and debugging)",
   "ALT+SHIFT+L   unassigned by default",
   "ALT+SHIFT+M   activate MenuBar",
   "ALT+SHIFT+N   unassigned by default",
   "ALT+SHIFT+O   unassigned by default",
   "ALT+SHIFT+P   unassigned by default",
   "ALT+SHIFT+Q   unassigned by default",
   "ALT+SHIFT+R   Restore file from trashcan",
   "ALT+SHIFT+S   Synchronize files between source and target",
   "ALT+SHIFT+T   manage system Trashcan",
   "ALT+SHIFT+U   open 'Util' menu",
   "ALT+SHIFT+V   open 'View' menu",
   "ALT+SHIFT+W   Set alt directory window to CWD",
   "ALT+SHIFT+X   unassigned by default",
   "ALT+SHIFT+Y   unassigned by default",
   "ALT+SHIFT+Z   invoke command-line shell (shell out)",

   "ALT+SHIFT+0   unassigned by default",
   "ALT+SHIFT+1   unassigned by default",
   "ALT+SHIFT+2   unassigned by default",
   "ALT+SHIFT+3   unassigned by default",
   "ALT+SHIFT+4   unassigned by default",
   "ALT+SHIFT+5   unassigned by default",
   "ALT+SHIFT+6   unassigned by default",
   "ALT+SHIFT+7   unassigned by default",
   "ALT+SHIFT+8   unassigned by default",
   "ALT+SHIFT+9   unassigned by default",

   "ALT+SHIFT+-   unassigned by default",
   "ALT+SHIFT+=   unassigned by default",
   "ALT+SHIFT+[   display user information",
   "ALT+SHIFT+]   display filesystem information",
   "ALT+SHIFT+\\   unassigned by default",
   "ALT+SHIFT+;   Browse FileMangler clipboard contents",
   "ALT+SHIFT+'   Eject optical media",
   "ALT+SHIFT+,   unassigned by default",
   "ALT+SHIFT+.   unassigned by default",
   "ALT+SHIFT+/   unassigned by default",

// "ALT+SHIFT+ENTER  (ALT+ENTER unavailable as a command key)",
// "ALT+SHIFT+INSERT (unavailable as a command key)",
   "ALT+SHIFT+DELETE unassigned by default",
   "ALT+SHIFT+UP     unassigned by default",
   "ALT+SHIFT+DOWN   unassigned by default",
   "ALT+SHIFT+LEFT   unassigned by default",
   "ALT+SHIFT+RIGHT  unassigned by default",
// "ALT+SHIFT+PAGEUP (unavailable as a command key)",
// "ALT+SHIFT+PAGEDN (unavailable as a command key)",
// "ALT+SHIFT+HOME   (unavailable as a command key)",
// "ALT+SHIFT+END    (unavailable as a command key)",

   //***********************
   //*** ALT+CTRL  group ***
   //***********************
   "ALT+CTRL+A    unassigned by default",
   "ALT+CTRL+B    unassigned by default",
   "ALT+CTRL+C    unassigned by default",
   "ALT+CTRL+D    unassigned by default",
   "ALT+CTRL+E    unassigned by default",
   "ALT+CTRL+F    unassigned by default",
   "ALT+CTRL+G    unassigned by default",
   "ALT+CTRL+H    unassigned by default",
   "ALT+CTRL+I    unassigned by default",
// "ALT+CTRL+J    (ALT+ENTER not available as a command key)",
   "ALT+CTRL+K    unassigned by default",
   "ALT+CTRL+L    unassigned by default",
// "ALT+CTRL+M    (ALT+ENTER not available as a command key)",
   "ALT+CTRL+N    unassigned by default",
   "ALT+CTRL+O    unassigned by default",
   "ALT+CTRL+P    unassigned by default",
   "ALT+CTRL+Q    unassigned by default",
   "ALT+CTRL+R    Resize application Dialog",
   "ALT+CTRL+S    unassigned by default",
   "ALT+CTRL+T    unassigned by default",
   "ALT+CTRL+U    unassigned by default",
   "ALT+CTRL+V    unassigned by default",
   "ALT+CTRL+W    unassigned by default",
   "ALT+CTRL+X    unassigned by default",
   "ALT+CTRL+Y    unassigned by default",
   "ALT+CTRL+Z    unassigned by default",

// "ALT+CTRL+INSERT (unavailable as a command key)",
// "ALT+CTRL+DELETE (unavailable as a command key)",
// "ALT+CTRL+UP     (unavailable as a command key)",
// "ALT+CTRL+DOWN   (unavailable as a command key)",
// "ALT+CTRL+LEFT   (unavailable as a command key)",
// "ALT+CTRL+RIGHT  (unavailable as a command key)",
   "ALT+CTRL+PAGEUP unassigned by default",
   "ALT+CTRL+PAGEDN unassigned by default",
// "ALT+CTRL+HOME   (unavailable as a command key)",
// "ALT+CTRL+END    (unavailable as a command key)",

   //*********************
   //*** KEYPAD  group ***
   //*********************
   "KPAD+CENTER         unassigned by default",
   "KPAD++              unassigned by default",
   "KPAD+-              unassigned by default",
   "KPAD+*              unassigned by default",
   "KPAD+/              unassigned by default",
   "KPAD+CTRL+PAGEUP    unassigned by default",
   "KPAD+CTRL+PAGEDN    unassigned by default",
   "KPAD+ALT+CTRL+UP    unassigned by default",
   "KPAD+ALT+CTRL+DOWN  unassigned by default",
   "KPAD+ALT+CTRL+LEFT  unassigned by default",
   "KPAD+ALT+CTRL+RIGHT unassigned by default",
   "KPAD+ALT+CTRL+DEL   unassigned by default",
   // (all other keypad combinations are duplicates of main keyboard keys)

   //***************************
   //*** Special-codes group ***
   //***************************
// "SPACE         Select or deselect highlighted file",
// "TAB           Move input focus to next control",
// "ENTER         Navigate to subdirectory, or open ViewFile context menu",

   //********************
   //*** F-KEY  group ***
   //********************
   "F01            (Term Help - may be unavailable)",
   "F02            Access Menu Bar",
   "F03            unassigned by default",
   "F04            unassigned by default",
   "F05            unassigned by default",
   "F06            unassigned by default",
   "F07            unassigned by default",
   "F08            unassigned by default",
   "F09            unassigned by default",
   "F10            unassigned by default",
   "F11            (Term Resize - may be unavailable)",
   "F12            unassigned by default",
             
   "SHIFT+F01      Invoke online Help (info reader)",
   "SHIFT+F02      unassigned by default",
   "SHIFT+F03      unassigned by default",
   "SHIFT+F04      unassigned by default",
   "SHIFT+F05      unassigned by default",
   "SHIFT+F06      unassigned by default",
   "SHIFT+F07      unassigned by default",
   "SHIFT+F08      unassigned by default",
   "SHIFT+F09      unassigned by default",
   "SHIFT+F10      unassigned by default",
   "SHIFT+F11      unassigned by default",
   "SHIFT+F12      unassigned by default",
             
   "CTRL+SHIFT+F01 unassigned by default",
   "CTRL+SHIFT+F02 unassigned by default",
   "CTRL+SHIFT+F03 unassigned by default",
   "CTRL+SHIFT+F04 unassigned by default",
   "CTRL+SHIFT+F05 unassigned by default",
   "CTRL+SHIFT+F06 unassigned by default",
   "CTRL+SHIFT+F07 unassigned by default",
   "CTRL+SHIFT+F08 unassigned by default",
   "CTRL+SHIFT+F09 unassigned by default",
   "CTRL+SHIFT+F10 unassigned by default",
   "CTRL+SHIFT+F11 unassigned by default",
   "CTRL+SHIFT+F12 unassigned by default",
} ;      // KeymapCfgDesc[]

//* Record for a valid command-key sequence *
class rKey
{
   public:

   void operator = ( const rKey& rkSrc )
   {
      short i = ZERO ;
      for ( ; i < KEYMAP_TEXT ; ++i )
      { if ( (this->ktext[i] = rkSrc.ktext[i]) == NULLCHAR ) { break ; } }
      if ( i == KEYMAP_TEXT )       // (should never happen)
         this->ktext[KEYMAP_TEXT - 1] = NULLCHAR ;
      this->kdflt = rkSrc.kdflt ;
      this->kuser = rkSrc.kuser ;
      this->cmd = rkSrc.cmd ;
      this->map = rkSrc.map ;
   }

   //** Public Data Members **
   char ktext[KEYMAP_TEXT] ;  // Formatted description of the key command
   wkeyCode kdflt ;           // command-key sequence
   wkeyCode kuser ;           // remapped value as an alias for 'kdflt'
                              // if not remapped, kuser.type == wktPRINT
   bool cmd ;                 // true if 'kdflt' is assigned to a command operation
   bool map ;                 // true if this command-key has been reassigned
} ;

//* Contains a map of default command shortcut keys, and *
//* equivalent user-selected key code (if any).          *
class RemapKey
{
   public:
   ~RemapKey ( void ) {}      // destructor
   RemapKey ( void )          // constructor
   {
      *this->keyFile = NULLCHAR ;
      this->kmActive = this->kmInit = false ;
   }

   //* Initialize only the top-level fields.  *
   //* Initialization of the 'keydef' array   *
   //* is unnecessary and takes too much time.*
   void reset ( void )
   {
      #if 0
      for ( short i = ZERO ; i < KEYMAP_KEYS ; ++i )
      {
         *this->keydef[i].ktext = NULLCHAR ;
         this->keydef[i].kdflt.type = this->keydef[i].kuser.type = wktPRINT ;
         this->keydef[i].kdflt.key = this->keydef[i].kuser.key = ZERO ;
         this->keydef[i].cmd = this->keydef[i].map = false ;
      }
      #endif
      *this->keyFile = NULLCHAR ;
      this->kmActive = this->kmInit = false ;
   }

   //* Input  : tofile  : (optional, false by default)                         *
   //*                    if 'false', initialize text for display in a dialog  *
   //*                    if 'true', initialize text for writing to config file*
   //*          userDefs: (optional, false by default)                         *
   //*                    if 'false' initialize the 'kuser' members with the   *
   //*                       placeholder value, and format the display text for*
   //*                       the default keycodes only                         *
   //*                    if 'true'  caller has initialized one or more of the *
   //*                       'kuser' members to a non-placeholder value (it is *
   //*                       assumed that the caller has entered only valid    *
   //*                       codes, and that unused records contain the        *
   //*                       placeholder value), so format the display text to *
   //*                       include the user-defined data                     *
   //* Returns: 'true' if keymapping is active (one or more commands remapped) *
   //*          'false' if keymap initialized but no commands remapped.        *
   bool initialize ( bool tofile = false, bool userDefs = false )
   {
      const wchar_t bar = L'|' ;       // formatting character
      gString gs, gsuser ;             // text formatting
      short offset,                    // text index
            kindex ;                   // index of user-defined keycode
      this->kmActive = this->kmInit = false ;   // reset in-use flags
      if ( tofile != false )
      {
         const short dfltOffset = 25,  // index of default-keycode column
                     descOffset = 45 ; // index of description column
         const char* const kmfDfltTemplate = 
            "          --          %C %s" ;
         const char* const kmfUserTemplate = 
            "   %S%C %s" ;

         for ( short i = ZERO ; i < KEYMAP_KEYS ; ++i )
         {
            //* Initialize the default keycode for the entry.*
            this->keydef[i].kdflt = KeymapCfgDflt[i] ;

            //* If user has specified a custom keycode for this entry *
            if ( (userDefs != false) && (this->keydef[i].kuser.type != wktPRINT) &&
                 ((kindex = this->validCode ( this->keydef[i].kuser )) < KEYMAP_KEYS) )
            {
               gsuser = KeymapCfgDesc[kindex] ;
               offset = gsuser.find( L' ' ) ;
               gsuser.limitChars( offset ) ;
               gsuser.padCols( (dfltOffset - 6) ) ;
               this->keydef[i].map = true ;

               //* Format the display text for the entry *
               gs.compose( kmfUserTemplate, gsuser.gstr(), &bar, KeymapCfgDesc[i] ) ;
            }
            else     // no user-defined code
            {
               //* Set user-defined keycode to dummy value *
               this->keydef[i].kuser = nonCode ;
               this->keydef[i].map = false ;

               //* Format the display text for the entry *
               gs.compose( kmfDfltTemplate, &bar, KeymapCfgDesc[i] ) ;
            }

            //* Format and store the display text for the entry.*
            offset = gs.find( L' ', dfltOffset ) ;
            offset = gs.scan( offset ) ;
            while ( offset < descOffset )
               gs.insert( L' ', offset++ ) ;
            this->keydef[i].cmd = bool((gs.find( "unassigned" )) < ZERO) ;
            gs.copy( this->keydef[i].ktext, KEYMAP_TEXT ) ;
         }
      }     // (tofile != false)
      else  // (tofile == false)
      {
         const short dfltOffset = 19 ; // index of default-keycode column
                     //descOffset = 33 ; // index of description column
         const char* const kmDfltTemplate = 
                  "        --       %C %s................................." ;
         const char* const kmUserTemplate = 
                  "%S%C %s................................." ;

         for ( short i = ZERO ; i < KEYMAP_KEYS ; ++i )
         {
            //* Initialize the default keycode for the entry.*
            this->keydef[i].kdflt = KeymapCfgDflt[i] ;

            //* Format and store the display text for the entry.*
            if ( (userDefs != false) && (this->keydef[i].kuser.type != wktPRINT) &&
                 ((kindex = this->validCode ( this->keydef[i].kuser )) < KEYMAP_KEYS) )
            {
               gsuser = KeymapCfgDesc[kindex] ;    // get full text description
               offset = gsuser.find( L' ' ) ;      // isolate the keycode string
               gsuser.limitChars( offset ) ;
               gsuser.insert( L' ' ) ;
               gsuser.padCols( (dfltOffset - 2) ) ;// pad to field width
               gsuser.limitCols( (dfltOffset - 2) ) ; // limit to field width
               gs.compose( kmUserTemplate, gsuser.gstr(), &bar, KeymapCfgDesc[i] ) ;
               this->keydef[i].map = true ;
               this->kmActive = true ;
            }
            else
            {
               gs.compose( kmDfltTemplate, &bar, KeymapCfgDesc[i] ) ;
               this->keydef[i].map = false ;
            }
            gs.limitCols( KEYMAP_COLS ) ;
            this->keydef[i].cmd = bool((gs.find( "unassigned" )) < ZERO) ;
            gs.copy( this->keydef[i].ktext, KEYMAP_TEXT ) ;
         }
      }     // (tofile == false)
      this->kmInit = true ;      // data members have been intialized

      return ( this->kmActive ) ;
   }     //* End initialize() *

   //* Compare the specified keycode to the list of         *
   //* user-defined keycodes, and if found, return the      *
   //* corresponding default keycode.                       *
   //* Input  : wk  : keycode to be tested                  *
   //* Returns: nothing                                     *
   void decodeKeyInput ( wkeyCode& wk )
   {
      if ( this->kmActive )
      {
         //* If this keycode appears in the user-defined list, *
         //* get the corresponding default value.              *
         for ( short i = ZERO ; i < KEYMAP_KEYS ; ++i )
         {
            if ( (this->keydef[i].map != false) &&
                 (wk == this->keydef[i].kuser) )
            {
               wk = this->keydef[i].kdflt ;
               break ;
            }
         }
      }

   }  //* End decodeKeyInput() *

   //* Match the specified keycode to a member of the list. *
   //* Input  : testCode : keycode to be tested             *
   //* Returns: index of matching keycode, or               *
   //*          KEYMAP_KEYS if no match found               *
   short validCode ( const wkeyCode& testCode )
   {
      short kindex ;
      for ( kindex = ZERO ; kindex < KEYMAP_KEYS ; ++kindex )
      {
         if ( testCode == KeymapCfgDflt[kindex] )
            break ;
      }
      return kindex ;

   }  //* End validCode() *

   void operator = ( const RemapKey& rkSrc )
   {
      for ( short i = ZERO ; i < KEYMAP_KEYS ; i++ )
         this->keydef[i] = rkSrc.keydef[i] ;
      for ( short i = ZERO ; i < MAX_FNAME ; ++i )
      { if ( (this->keyFile[i] = rkSrc.keyFile[i]) == NULLCHAR ) { break ; } }
      this->kmActive = rkSrc.kmActive ;
      this->kmInit   = rkSrc.kmInit ;
   }  //* End operator=() *

   //** Public Data Members **
   rKey keydef[KEYMAP_KEYS] ; // array of keycode definition records
   char keyFile[MAX_FNAME] ;  // name of key-map configuration file
   bool kmActive ;            // 'true' when one or more command-keys remapped
   bool kmInit ;              // 'true' if array has been initialized

   private:
} ;   // RemapKey

#endif   // FMKEYMAP_DEFINED
