//********************************************************************************
//* File       : FmInterface.cpp                                                 *
//* Author     : Mahlon R. Smith                                                 *
//*              Copyright (c) 2005-2025 Mahlon R. Smith, The Software Samurai   *
//*                  GNU GPL copyright notice located in FileMangler.hpp         *
//* Date       : 26-Jul-2025                                                     *
//* Version    : (see AppVersion string)                                         *
//*                                                                              *
//* Description: This is a support module for the FileMangler application.       *
//* This module, along with FmCommand.cpp contains the non-menu (command-key)    *
//* portion of the User Interface which gathers key input and executes the       *
//* user's commands.                                                             *
//* Also in this module are the methods specific to FileMangler's                *
//* Single-window and Dual-window operation modes.                               *
//*                                                                              *
//* Development Tools: see notes in FileMangler.cpp.                             *
//********************************************************************************
//* Programmer's Notes:                                                          *
//*                                                                              *
//********************************************************************************

//* Header Files *
#include "FileMangler.hpp"    // FileMangler definitions and data

//* Local Methods *
static short uiCallback ( const short currentIndex, const wkeyCode wkey, bool firstTime ) ;

//** Local Data **
static FileMangler* fmangPtr = NULL ;  // see uiCallback() method


//*************************
//*    SingleWin_Mode     *
//*************************
//********************************************************************************
//* Initialize application for single-window file browsing.                      *
//*                                                                              *
//*                                                                              *
//* Input  : startupPath : (optional, NULL* by default)                          *
//*                        if specified, path of start-up target directory       *
//*                        if not specified, start-up directory is               *
//*                        current-working dir (CWD)                             *
//*          treeView    : (optional, 'false' by default)                        *
//*                        if 'true', then after opening the main dialog         *
//*                        invoke the 'Tree View' command                        *
//*          rootScan    : (optional, 'false' by default)                        *
//*                        if 'true', then enable full scan from CWD of "/"      *
//*                                                                              *
//* Returns: member of enum modeStat                                             *
//********************************************************************************

modeStat FileMangler::SingleWin_Mode ( const char* startupPath, 
                                       bool treeView, bool rootScan )
{
   modeStat mStat = mstatEXIT ;        // return value

   //* Initialize static data defined in FmMenu.hpp *
   bool startupOK = this->InitFmMenuData ( wmSINGLE_WIN ) ;
   if ( startupOK != false )
   {
      short ulY = ZERO,    // dialog's position, upper-left corner
            ulX = ZERO ;

      //* Initial parameters for dialog window *
      InitNcDialog dInit( this->actualRows, // number of display lines
                          this->actualCols, // number of display columns
                          ulY,            // Y offset from upper-left of terminal 
                          ulX,            // X offset from upper-left of terminal 
                          NULL,           // dialog title
                          ncltSINGLE,     // border line-style
                          this->cfgOpt.cScheme.bb, // border color attribute
                          this->cfgOpt.cScheme.bb, // interior color attribute
                          this->ic        // pointer to list of control definitions
                        ) ;

      //* Instantiate the dialog window *
      this->dWin = new NcDialog ( dInit ) ;

      //* Open the dialog window *
      if ( (this->dWin->OpenWindow()) == OK )
      {
         //* Establish a Menu Bar (group of DialogMenuwin controls).           *
         //* Menu Bar is initially invisible (obscured by the dialog title).   *
         this->DialogTitle ( false, -1, true ) ;
         
         //* Draw a line under the status display *
         LineDef  lDef(ncltHORIZ, ncltSINGLE, 2, ZERO, this->actualCols, 
                       this->cfgOpt.cScheme.bb ) ;
         this->dWin->DrawLine ( lDef ) ;

         //* Make a visual connection for the dctSCROLLEXT control that *
         //* overlaps the dialog window's border.                       *
         cdConnect cdConn ;
         cdConn.ul2Left = true ;
         cdConn.ll2Left = true ;
         cdConn.ur2Right = true ;
         cdConn.lr2Right = true ;
         cdConn.connection = true ;
         this->dWin->ConnectControl2Border ( dwFileSE, cdConn ) ;

         //* If mouse support requested *
         if ( this->cfgOpt.enableMouse != false )
         {
            if ( (this->dWin->meEnableStableMouse ()) != OK )
               this->cfgOpt.enableMouse = false ;
         }

         //* Instantiate the FileDlg class for access to the filesystem *
         InitFileDlg    fdInit ;
         fdInit.dPtr          = this->dWin ;
         fdInit.fdispInit     = this->ic[dwFileSE] ;
         fdInit.fdispIndex    = dwFileSE ;
         fdInit.statusInit    = this->ic[dwMsgsTB] ;
         fdInit.statusIndex   = dwMsgsTB ;
         fdInit.pathInit      = this->ic[dwPathTB] ;
         fdInit.pathIndex     = dwPathTB ;
         fdInit.statsInit     = this->ic[dwStatTB] ;
         fdInit.statsIndex    = dwStatTB ;
         fdInit.rootScan      = rootScan ;
         fdInit.refresh       = !treeView ;
         fdInit.startPath     = startupPath ;
         fdInit.cfgInit       = this->cfgOpt ;
         this->fdPtr = this->fWin = new FileDlg ( fdInit ) ;

         //* Determine whether the Trashcan was found and is properly *
         //* configured. If not, Trashcan operations are disabled.    *
         //* Ask user whether to run configuration.                   *
         if ( !(this->fdPtr->TrashcanConfigured ()) )
         {
            if ( (this->Prompt4Config ()) != false )
               mStat = mstatCONFIG ;
         }

         //* Initial status message (obscures the Quick-Help message) *
         dtbmData  msgData(initialStatusMsg, initialStatusMsgColor) ;
         this->dWin->DisplayTextboxMessage ( dwMsgsTB, msgData ) ;

         //* If user is 'superuser', then post an indicator *
         this->Display_SU_Flag () ;

         //* Display our results */
         this->dWin->RefreshWin () ;

         //* If the "wl-clipboard" utilities, "wl-copy" and "wl-paste" are *
         //* installed on the local system, then establish communications  *
         //* between the Wayland system clipboard and the NcDialog's       *
         //* Textbox Local Clipboard.                                      *
         //* If unable to establish a connection with the system clipboard,*
         //* copy/cut/paste operations will use the local clipboard only.  *
         //* CTRL+C == Copy, CTRL+X == Cut, CTRL+V == Paste,               *
         //* CTRL+A == Select all,                                         *
         //* SHIFT+LeftArrow and SHIFT+RightArrow == Select by character.  *
         //* Error conditions:                                             *
         //* a) If the external utilities are not installed, then silently *
         //*    continue with local clipboard only.                        *
         //* b) If the external utilities ARE installed, but connection    *
         //*    not established, then report the error.                    *
         if ( !(this->dWin->wcbEnable ()) )
            this->dWin->wcbUserAlert ( wcbsNOCONNECT ) ;
         #if 0    // For Debugging Only - Verify WaylandCB connection.
         else
         {
            gString gsOut, gsver, gsval ;
            int cbbytes ;
            bool conn = this->dWin->wcbIsConnected () ; // get WCB status
            this->dWin->wcbVersion ( gsver ) ;  // get concatenated version strings
            if ( conn )
            {
               gsver.insert( "WaylandCB version:    " ) ;
               gsver.insert( "wl-clipboard version: ", (gsver.after( NEWLINE )) ) ;
               cbbytes = this->dWin->wcbBytes () ;
               gsval.formatInt( cbbytes, 5, true );
               gsver.append( "\nclipboard contents: %S bytes", gsval.gstr() ) ;
            }
            gsOut.compose( "  Clipboard Status  \n"
                           "System Clipboard Connected: %s\n"
                           "%S\n",
                           (conn ? "yes" : "no"), gsver.gstr() ) ;
            this->dWin->wcbUserAlert ( wcbsACTIVE, &gsOut ) ;
         }
         #endif   // Verify WaylandCB connection.

         //***************************
         //* Interact with the user. *
         //* (Watch out; they bite!) *
         //***************************
         if ( mStat != mstatCONFIG )
         {
            MenuCode initCmd = treeView ? mcViewMB_DIRTREE : mcNO_SELECTION, 
                     mCode = this->UserInterface ( initCmd ) ;
            switch ( mCode )
            {
               case mcViewMB_DUALWIN:     mStat = mstatDUAL ;     break ;
               case mcViewMB_SINGLEWIN:
               case mcViewMB_RESIZE:
                  mStat = mstatSINGLE ;
                  break ;
               case mcUtilMB_CONFIG:
               case mcEditMB_PREFS:       mStat = mstatCONFIGR ;  break ;
               case mcFileMB_EXIT:
               default:
                  mStat = this->CmdExit () ;
                  break ;
            }
         }
      }  // OpenWindow()
      else
      {
         this->DiagMsg ( "Unable to open application dialog window.", nc.reB ) ;
         chrono::duration<short>aWhile( 8 ) ;
         this_thread::sleep_for( aWhile ) ;
      }

      if ( this->fWin != NULL )  // close the FileDlg class instance
      {
         delete ( this->fWin ) ;
         this->fWin = NULL ;
      }
      if ( this->dWin != NULL )  // close the dialog window
      {
         //* Terminate the connection with the system clipboard.*
         this->dWin->wcbDisable () ;

         delete ( this->dWin ) ;
         this->dWin = NULL ;
      }
   }
   return mStat ;
   
}  //* End SingleWin_Mode() *

//*************************
//*    DualWin_Mode       *
//*************************
//********************************************************************************
//* Initialize application for dual-window file browsing.                        *
//*                                                                              *
//*                                                                              *
//* Input  : startupPath : (optional, NULL* by default)                          *
//*                        if specified, path of start-up target directory       *
//*                        if not specified, start-up directory is               *
//*                        current-working dir (CWD)                             *
//*          treeView    : (optional, 'false' by default)                        *
//*                        if 'true', then after opening the main dialog         *
//*                        invoke the 'Tree View' command                        *
//*          rootScan    : (optional, 'false' by default)                        *
//*                        if 'true', then enable full scan from CWD of "/"      *
//*                                                                              *
//* Returns: member of enum modeStat                                             *
//********************************************************************************

modeStat FileMangler::DualWin_Mode ( const char* startupPath, 
                                     bool treeView, bool rootScan )
{
   //* If exiting with 'mcViewMB_RESIZE', then  *
   //* remember which is the active window and  *
   //* the paths of each window.                *
   static char lPath[MAX_PATH] = "\0", // CWD of left window on exit for restart
               rPath[MAX_PATH] = "\0" ;// CWD of right window on exit for restart
   static bool rwinResize = false ;    // 'true' if right window active on exit for restart
   modeStat mStat = mstatEXIT ;        // return value

   //* If clean startup, both windows will display 'startupPath'.*
   //* Else if we are performing a restart after a 'resize'      *
   //* command, use the previously-saved target paths.           *
   if ( lPath[0] == NULLCHAR )
   {
      gString gs( startupPath ) ;
      gs.copy( lPath, MAX_PATH ) ;
      gs.copy( rPath, MAX_PATH ) ;
   }

   //* Initialize static data defined in FmMenu.h *
   bool startupOK = this->InitFmMenuData ( wmDUAL_WIN ) ;
   if ( startupOK != false )
   {
      //* Initial parameters for dialog window *
      InitNcDialog dInit( this->actualRows, // number of display lines
                          this->actualCols, // number of display columns
                          ZERO,           // Y offset from upper-left of terminal 
                          ZERO,           // X offset from upper-left of terminal 
                          NULL,           // dialog title
                          ncltSINGLE,     // border line-style
                          this->cfgOpt.cScheme.bb, // border color attribute
                          this->cfgOpt.cScheme.bb, // interior color attribute
                          this->ic        // pointer to list of control definitions
                        ) ;
   
      //* Instantiate the dialog window *
      this->dWin = new NcDialog ( dInit ) ;

      //* Open the dialog window *
      if ( (this->dWin->OpenWindow()) == OK )
      {
         //* Establish a Menu Bar (group of DialogMenuwin controls).           *
         //* Menu Bar is initially invisible (obscured by the dialog title).   *
         this->DialogTitle ( false, -1, true ) ;

         //* Draw a line under the status display *
         LineDef  lDef(ncltHORIZ, ncltSINGLE, 2, ZERO, this->actualCols, 
                       this->cfgOpt.cScheme.bb) ;
         this->dWin->DrawLine ( lDef ) ;

         //* Make a visual connection for the dctSCROLLEXT controls that *
         //* overlaps the dialog window's border.                        *
         cdConnect cdConn ;
         cdConn.ul2Left = true ;
         cdConn.ll2Left = true ;
         cdConn.connection = true ;
         this->dWin->ConnectControl2Border ( dwFileSE, cdConn ) ;
         cdConn.ul2Left = false ;
         cdConn.ll2Left = false ;
         cdConn.ur2Right = true ;
         cdConn.lr2Right = true ;
         this->dWin->ConnectControl2Border ( dwFileSEr, cdConn ) ;

         //* If mouse support requested *
         if ( this->cfgOpt.enableMouse != false )
         {
            if ( (this->dWin->meEnableStableMouse ()) != OK )
               this->cfgOpt.enableMouse = false ;
         }

         //* Instantiate the FileDlg class for access to the filesystem *
         InitFileDlg    fdInit ;
         fdInit.dPtr          = this->dWin ;
         fdInit.statusInit    = this->ic[dwMsgsTB] ;
         fdInit.statusIndex   = dwMsgsTB ;
         fdInit.fdispInit     = this->ic[dwFileSE] ;
         fdInit.fdispIndex    = dwFileSE ;
         fdInit.pathInit      = this->ic[dwPathTB] ;
         fdInit.pathIndex     = dwPathTB ;
         fdInit.statsInit     = this->ic[dwStatTB] ;
         fdInit.statsIndex    = dwStatTB ;
         fdInit.rootScan      = rootScan ;
         fdInit.refresh       = !treeView ;
         fdInit.startPath     = lPath ;
         fdInit.cfgInit       = this->cfgOpt ;
         this->fdPtr = this->fWin = new FileDlg ( fdInit ) ;  // left-side view

         //* Instantiate the right-side FileDlg window *
         fdInit.fdispInit     = this->ic[dwFileSEr] ;
         fdInit.fdispIndex    = dwFileSEr ;
         fdInit.pathInit      = this->ic[dwPathTBr] ;
         fdInit.pathIndex     = dwPathTBr ;
         fdInit.statsInit     = this->ic[dwStatTBr] ;
         fdInit.statsIndex    = dwStatTBr ;
         fdInit.startPath     = rPath ;
         this->altPtr = this->rWin = new FileDlg ( fdInit ) ;  // right-side view
         //* Remove highlight from right-side (non-focus) window *
         this->dWin->RefreshScrollextText ( dwFileSEr, false ) ;

         //* If we are performing a restart after a 'resize' command, *
         //* verify which is the active window.                       *
         if ( rwinResize )
         {
            while ( (this->dWin->PrevControl ()) != dwFileSEr ) ;
            this->fdPtr->RedrawCurrDir ( false ) ;
            FileDlg* tmp = this->fdPtr ;
            this->fdPtr = this->altPtr ;
            this->altPtr = tmp ;
            this->fdPtr->RedrawCurrDir () ;
            rwinResize = false ;
         }
         lPath[0] = rPath[0] = NULLCHAR ;    // reinitialize saved target dirs

         //* Determine whether the Trashcan was found and is properly *
         //* configured. If not, Trashcan operations are disabled.    *
         //* Ask user whether to run configuration.                   *
         if ( !(this->fdPtr->TrashcanConfigured ()) )
         {
            if ( (this->Prompt4Config ()) != false )
               mStat = mstatCONFIG ;
         }

         //* Establish a callback method to monitor user input. *
         fmangPtr = this ; // allow callback method access to our data (see notes)
         this->dWin->EstablishCallback ( uiCallback ) ;

         //* Initial status message (obscures the Quick-Help message) *
         dtbmData  msgData(initialStatusMsg, initialStatusMsgColor) ;
         this->dWin->DisplayTextboxMessage ( dwMsgsTB, msgData ) ;

         //* If user is 'superuser', then post an indicator *
         this->Display_SU_Flag () ;

         //* Display our results */
         this->dWin->RefreshWin () ;

         //* If the "wl-clipboard" utilities, "wl-copy" and "wl-paste" are *
         //* installed on the local system, then establish communications  *
         //* between the Wayland system clipboard and the NcDialog's       *
         //* Textbox Local Clipboard.                                      *
         //* If unable to establish a connection with the system clipboard,*
         //* copy/cut/paste operations will use the local clipboard only.  *
         //* CTRL+C == Copy, CTRL+X == Cut, CTRL+V == Paste,               *
         //* CTRL+A == Select all,                                         *
         //* SHIFT+LeftArrow and SHIFT+RightArrow == Select by character.  *
         //* Error conditions:                                             *
         //* a) If the external utilities are not installed, then silently *
         //*    continue with local clipboard only.                        *
         //* b) If the external utilities ARE installed, but connection    *
         //*    not established, then report the error.                    *
         if ( !(this->dWin->wcbEnable ()) )
            this->dWin->wcbUserAlert ( wcbsNOCONNECT ) ;

         //***************************
         //* Interact with the user. *
         //* (Watch out; they bite!) *
         //***************************
         if ( mStat != mstatCONFIG )
         {
            MenuCode initCmd = treeView ? mcViewMB_DIRTREE : mcNO_SELECTION, 
                     mCode = this->UserInterface ( initCmd ) ;
            switch ( mCode )
            {
               case mcViewMB_RESIZE:
                  {
                  //* If a 'resize' event, remember the active window, *
                  //* and save the CWD of each window.                 *
                  rwinResize = (this->fdPtr == this->rWin) ? true : false ;
                  gString gs ;
                  this->fdPtr->GetPath ( gs ) ;
                  gs.copy( (rwinResize ? rPath : lPath), MAX_PATH ) ;
                  this->altPtr->GetPath ( gs ) ;
                  gs.copy( (rwinResize ? lPath : rPath), MAX_PATH ) ;
                  }
                  // fall through here
               case mcViewMB_DUALWIN:
                  mStat = mstatDUAL ;
                  this->fdPtr->RefreshCurrDir () ;    // set the CWD
                  break ;
               case mcViewMB_SINGLEWIN:
                  mStat = mstatSINGLE ;
                  this->fdPtr->RefreshCurrDir () ;    // set the CWD
                  break ;
               case mcUtilMB_CONFIG:
               case mcEditMB_PREFS:       mStat = mstatCONFIGR ;  break ;
               case mcFileMB_EXIT:
               default:
                  mStat = this->CmdExit () ;
                  break ;
            }
         }
      }  // OpenWindow()
      else
      {
         this->DiagMsg ( "Unable to open application dialog window.", nc.reB ) ;
         chrono::duration<short>aWhile( 8 ) ;
         this_thread::sleep_for( aWhile ) ;
      }

      if ( this->rWin != NULL )  // delete the right-side FileDlg class instance
      {
         delete ( this->rWin ) ;
         this->rWin = NULL ;
      }
      if ( this->fWin != NULL )  // delete the left-side FileDlg class instance
      {
         delete ( this->fWin ) ;
         this->fWin = NULL ;
      }
      if ( this->dWin != NULL )  // close the dialog window
      {
         //* Terminate the connection with the system clipboard.*
         this->dWin->wcbDisable () ;

         delete ( this->dWin ) ;
         this->dWin = NULL ;
      }
   }
   else
   {
      //* Terminal window is too small for Dual-Window mode, but it may be     *
      //* large enough for Single-Window mode, so don't give up yet.           *
      if ( this->termRows >= MIN_ROWS && this->termCols >= MIN_COLS )
         mStat = mstatSINGLE ;
   }

   return mStat ;

}  //* End DualWin_Mode() *

//*************************
//*    UserInterface      *
//*************************
//********************************************************************************
//* Interact with the user, gathering input and executing commands.              *
//*                                                                              *
//*                                                                              *
//* Input  : initialCmd : (optional, mcNO_SELECTION by default)                  *
//*                                                                              *
//* Returns: member of enum MenuCode:                                            *
//*           mcFileMB_EXIT      == exit application                             *
//*           mcViewMB_DUALWIN   == switch to dual-window display                *
//*           mcViewMB_SINGLEWIN == switch to single-window display              *
//*           mcEditMB_PREFS     == invoke FmConfig utility                      *
//*           mcUtilMB_CONFIG    == invoke FmConfig utility                      *
//*******************************************************************************
//* Under Consideration:                                                         *
//* -- We have a callback method that we COULD use to selected/deselect items    *
//*    dynamically without having to leave the Edit method so often.             *
//* -- We could also do selection/deselection through the ScrollWheel+CTRL key.  *
//* -- This may turn out to be impractical, but it's worth a try.                *
//*                                                                              *
//*                                                                              *
//********************************************************************************

MenuCode FileMangler::UserInterface ( MenuCode initialCmd )
{
   //* Execute the initial command (if any) *
   switch ( initialCmd )
   {
      case mcViewMB_DIRTREE:           // go to tree-view mode
         this->CmdDirTree () ;
         break ;
      default:                         // no initial command
         break ;
   }

   MenuCode  menuCode = mcFileMB_EXIT ;
   uiInfo    Info ;           // user input data returned from edit methods
   short     icIndex = this->dWin->GetCurrControl (), // index of control with focus
             oldIndex ;
   bool      done = false ;   // loop control


   while ( ! done )
   {
      //* Let user scroll through the data until a non-scroll key detected *
      oldIndex = icIndex ;
      icIndex = this->fdPtr->ScrollThruFileList ( Info ) ;
      #if 0    // Debugging: DUMP uiInfo
      winPos wp(7,80);
      this->dWin->Dump_uiInfo ( wp, Info ) ;
      #endif   // DUMP uiInfo

      //* If focus has changed to the other ScrollExt window, then update our *
      //* local tracking data. This only happens through the mouse interface. *
      if ( icIndex != oldIndex )
      {
         //* Remove highlight from previously-active control *
         this->fdPtr->RedrawCurrDir ( false ) ;
         //* Point to FileDlg object which now has the input focus *
         FileDlg* tmp = this->fdPtr ;
         this->fdPtr = this->altPtr ;
         this->altPtr = tmp ;
         //* Redraw data (with highlight) in new control *
         this->fdPtr->RedrawCurrDir () ;
         //* Discard the input to avoid undesired selection.*
         continue ;
      }

      //* If key input has been remapped (see configuration file), *
      //* then map any user-defined keycode back to the default.   *
      this->keyMap.decodeKeyInput ( Info.wk ) ;

      //* Prevent duplication of code by converting nckF02 or the equivalent *
      //* mouse click for Menu Bar access to the extended-key equivalent.    *
      // Programmer's Note: The mouse position test below uses the screen-absolute 
      // mouse coordinates because the event may or may not have been converted to 
      // dialog-relative coordinates.
      if ( (Info.wk.type == wktFUNKEY && Info.wk.key == nckF02) ||
           (
            ((Info.wk.type == wktMOUSE) && (this->dWin->meMouseEnabled ()))
            &&
            (Info.wk.mevent.meType == metB1_S || Info.wk.mevent.meType == metB1_D)
            &&
            ((Info.wk.mevent.sypos == this->menuBase.ypos) &&
             ((this->cfgOpt.winMode == wmDUAL_WIN) || (Info.wk.mevent.sxpos < MIN_COLS)))
           )
         )
      { Info.wk.type = wktEXTEND ; Info.wk.key = nckAS_M ; }

      //* If input is an NcDialog extended-key-definition *
      if ( Info.wk.type == wktEXTEND )
      {
         wchar_t extKey = Info.wk.key ;

         // Please Note:
         // Alt+f, Alt+Shift+f,  (File menu)
         // Alt+e, Alt+Shift+e,  (Edit menu)
         // Alt+v, Alt+Shift+v,  (View menu)
         // Alt+u, Alt+Shift+u,  (Util menu)
         // Alt+h, Alt+Shift+h   (Help menu)
         // key combinations are reserved for accessing 
         // the individual menus of the Menu Bar.
         // In addition, the terminal application, for example the Gnome 
         // terminal, may eat several Alt+n key combinations for access to 
         // it own menu system, so be careful in assigning an Alt+n key 
         // combination to any application functionality.                  *
         //*                                                               *
         //* Alt+Shift+M == activate menu system                           *
         //* Alternate command-key access to the application menu          *
         //* in case the GUI mavens have snarfed the F2 key.               *
         if ( extKey == nckAS_M ||
              extKey == nckA_F || extKey == nckAS_F ||
              extKey == nckA_E || extKey == nckAS_E ||
              extKey == nckA_V || extKey == nckAS_V ||
              extKey == nckA_U || extKey == nckAS_U ||
              extKey == nckA_H || extKey == nckAS_H )
         {
            this->SetActiveMenuItems ( dwCTRLS ) ;
            short trgMenu = -1 ;
            switch ( extKey )
            {
               // Programmer's Note: We're not really happy with how Alt+Shift+M 
               // opens the menu bar, but it's fairly cool.
               case nckAS_M:  trgMenu = MAX_DIALOG_CONTROLS ;    break ;
               case nckA_E:   case nckAS_E: trgMenu = dwEditMB ; break ;
               case nckA_V:   case nckAS_V: trgMenu = dwViewMB ; break ;
               case nckA_U:   case nckAS_U: trgMenu = dwUtilMB ; break ;
               case nckA_H:   case nckAS_H: trgMenu = dwHelpMB ; break ;
               case nckA_F:   case nckAS_F:  default: trgMenu = dwFileMB ; break ;
            }
            menuCode = this->MenuBarAccess ( trgMenu ) ;
            if ( menuCode != mcFileMB_EXIT    && menuCode != mcViewMB_RESIZE &&
                 menuCode != mcViewMB_DUALWIN && menuCode != mcViewMB_SINGLEWIN && 
                 menuCode != mcEditMB_PREFS   && menuCode != mcUtilMB_CONFIG )
            {
               this->ExecuteMenuCommand ( menuCode ) ;
               //* Return input focus to original control *
               while ( (this->dWin->GetCurrControl ()) != icIndex )
                  this->dWin->NextControl () ;
            }
            else
            {
               //* If a) switching to single-window mode *
               //*    b) switching to dual-window mode   *
               //*    c) exiting the application         *
               //*    d) resizing the application dialog *
               //*    e) invoking configuration utility  *
               //* then 'menuCode' tells the tale...     *
               done = true ;
            }
         }

         //* Alt+q == exit application to current directory *
         else if ( extKey == nckA_Q )
         {
            this->exit2cwd = true ;
            menuCode = mcFileMB_EXIT ;
            done = true ;
         }

         //* Alt+Shift+c == compare files *
         else if ( extKey == nckAS_C )
         {
            this->CmdCompareFiles () ;
         }

         //* Alt+Shift+g == grep file contents *
         else if ( extKey == nckAS_G )
         {
            this->CmdGrep () ;
         }

         //* Alt+Shift+i == locate files with shared inode *
         else if ( extKey == nckAS_I )
         {
            this->CmdInodes () ;
         }

         //* Alt+Shift+z == go to the command shell *
         else if ( extKey == nckAS_Z )
         {
            this->CmdShellOut () ;
         }

         //* Alt+w == switch between single-window and *
         //*          dual-window mode                 *
         else if ( extKey == nckA_W )
         {
            menuCode = this->cfgOpt.winMode == wmSINGLE_WIN ? 
                                    mcViewMB_DUALWIN : mcViewMB_SINGLEWIN ;
            this->synchLock = false ;  // ensure that synch-lock is disabled
            done = true ;
         }

         //* Alt+Shift+w == set inactive window to same     *
         //* directory as active window. Dual-win Mode only.*
         else if ( extKey == nckAS_W )
         {
            this->CmdAltCWD () ;
         }

         //* If highlight was on a directory name, user is requesting  *
         //* the View-file context menu. Else, same as nckENTER, below.*
         else if ( extKey == nckAC_J ) // ALT+nckENTER
         {
            this->CmdENTER ( true ) ;
         }

         //* Alt+Shift+b == perform directory-tree backup          *
         //*                         (dual-window mode only)       *
         else if ( extKey == nckAS_B )
         {
            this->CmdBackup ( false ) ;
         }

         //* Alt+Shift+s == perform directory-tree synchronization          *
         //*                         (dual-window mode only)                *
         else if ( extKey == nckAS_S )
         {
            this->CmdBackup ( true ) ;
         }

         //* Create an archive file       *
         else if ( extKey == nckAS_A )
         {
            this->CmdArchive () ;
         }

         //* Alt+Shift+r == restore most recent move to trash *
         else if ( extKey == nckAS_R )
         {
            this->CmdTrash ( mcUtilMB_UTRASH ) ;
         }

         //* Alt+i == toggle visibility of hidden files *
         else if ( extKey == nckA_I )
         {
            this->CmdHiddenFiles () ;
         }

         //* Alt+r == toggle full-scan option for display of root directory *
         else if ( extKey == nckA_R )
         {
            this->CmdRootScan () ;
         }

         //* Alt+Shift+t == manage the system trashcan *
         else if ( extKey == nckAS_T )
         {
            this->CmdTrash ( mcUtilMB_MTRASH ) ;
         }

         //* Alt+Ctrl+r == resize application dialog to fit terminal window.  *
         else if ( extKey == nckAC_R )
         {
            menuCode = mcViewMB_RESIZE ;
            done = true ;
         }

         //* Alt+SingleQuote == lock dual window navigation in synch. *
         //* Valid only while in Dual-window Mode, else ignored.      *
         else if ( extKey == nckA_SQUO )
         {
            this->CmdSynchLock () ;
         }

         //* Open (eject) or close an optical-drive tray.*
         else if ( extKey == nckA_DQUO )
         {
            this->CmdOpticalMedia () ;
         }

         //* Alt+Shift+Semicolon (AKA:Alt+Colon)                *
         //* == browse clipboard contents.                      *
         else if ( extKey == nckA_COLO )
         {
            this->CmdBrowseCB () ;
         }

         //* Alt+Shift+LeftBracket (AKA:Alt+Shift+LeftBrace)    *
         //* == Display user account information.               *
         else if ( extKey == nckA_LBRC )
         {
            this->CmdUserInfo () ;
         }

         //* Alt+Shift+RightBracket (AKA:Alt+Shift+RightBrace)            *
         //* == Display information on the currently-displayed filesystem.*
         else if ( extKey == nckA_RBRC )
         {
            this->CmdFilesysInfo () ;
         }

         //* Alt+Shift+k == testing or debugging functionality.    *
         //* Follow with a digit, 0-9.                             *
         //* For Debug and Test - Hidden Command with Sub-Commands *
         else if ( extKey == nckAS_K )
         {
            wkeyCode wk ;
            this->dWin->GetKeyInput ( wk ) ;
            switch ( wk.key )
            {
               case '0':
                  break ;
               case '1':
                  #if 0    // Debug Exif headers for jpeg files
                  {
                  tnFName  fn ;
                  this->fdPtr->GetStats ( fn ) ;
                  gString cmdBuff( "jhead \"%s\" | less -c", fn.fName ) ;
                  this->dWin->ShellOut ( soX, cmdBuff.ustr() ) ;
                  }
                  #endif   // Debugging
                  break ;
               case '2':
                  break ;
               case '3':
                  break ;
               case '4':
                  break ;
               case '5':
                  break ;
               case '6':
                  break ;
               case '7':
                  break ;
               case '8':
                  break ;
               case '9':
                  #if TEST_CS != 0
                  this->dWin->SetDialogObscured () ;  // save the parent window
                  this->TestColorSchemes () ;
                  this->dWin->RefreshWin () ;   
                  #endif   // TEST_CS
                  break ;
               default:
                  break ;
            }
         }

         //* Else is invalid input in this context *
         else
         {
            this->dWin->UserAlert ( 1 ) ;
         }
      }
      else
      {
         if ( Info.wk.type == wktFUNKEY &&
               (Info.wk.key == nckTAB   ||   // move to next active control
                Info.wk.key == nckRIGHT ||
                Info.wk.key == nckSTAB  ||   // move to previous active control
                Info.wk.key == nckLEFT) )
         {
            //* If we are in Dual-Win mode, move to the alternate *
            //* file-display control, else, do nothing            *
            if ( (this->DualwinMode ()) )
            {
               //* Remove highlight from currently-active control *
               this->fdPtr->RedrawCurrDir ( false ) ;
               //* Move input focus to alternate view *
               icIndex = this->dWin->NextControl () ;
               //* Point to FileDlg object which now has the input focus *
               FileDlg* tmp = this->fdPtr ;
               this->fdPtr = this->altPtr ;
               this->altPtr = tmp ;
               //* Redraw data (with highlight) in new control *
               this->fdPtr->RedrawCurrDir () ;
            }
         }

         //* SPACE key or SHIFT+DownArrow or SHIFT+UpArrow or Mouse *
         //* Click (without modifier keys) indicates file selection.*
         else if ( (Info.wk.type == wktPRINT  && Info.wk.key == nckSPACE) ||
                   (Info.wk.type == wktFUNKEY && 
                    (Info.wk.key == nckSDOWN || Info.wk.key == nckSUP)) ||
                   (Info.wk.mevent.conv && Info.wk.mevent.meType == metB1_S && 
                    !(Info.wk.mevent.cKey || Info.wk.mevent.aKey)) )
         {
            this->CmdSelect ( Info.wk ) ;
         }

         //* ENTER or Mouse Click (with modifier key)
         else if ( Info.wk.type == wktFUNKEY &&
               (Info.wk.key == nckENTER ||   // directory or filename selected
                Info.wk.key == nckpENTER) )  // OR open 'viewFile' context menu
         {
            //* ALT key + Mouse Click yields ViewFile Menu *
            if ( Info.wk.mevent.conv && Info.wk.mevent.aKey )
               this->CmdENTER ( true ) ;

            //* If highlight was on a directory name, go to that *
            //* directory. Else user is requesting ViewFile Menu.*
            else
               this->CmdENTER () ;
         }

         else if ( Info.wk.type == wktFUNKEY &&    // move to parent directory
                   (Info.wk.key == nckBKSP || Info.wk.key == nckAUP) )
         {
            this->CmdParentDir () ;
         }

         else if ( Info.wk.type == wktFUNKEY &&    // select/deselect ALL files
                   (Info.wk.key == nckC_A) )
         {
            this->CmdSelectAll () ;
         }

         else if ( Info.wk.type == wktFUNKEY &&    // copy files to the clipboard
                   (Info.wk.key == nckC_C) )
         {
            this->CmdCopy () ;
         }

         else if ( Info.wk.type == wktFUNKEY &&    // cut files to the clipboard
                   (Info.wk.key == nckC_X) )
         {
            this->CmdCut () ;
         }

         else if ( Info.wk.type == wktFUNKEY &&    // paste files from clipboard
                   (Info.wk.key == nckC_V) )
         {
            this->CmdPaste () ;
         }

         else if ( Info.wk.type == wktFUNKEY &&    // paste-special from clipboard
                   (Info.wk.key == nckAINSERT) )
         {
            this->CmdPasteSp () ;
         }

         else if ( Info.wk.type == wktFUNKEY &&    // undo most recent paste (move or copy)
                   (Info.wk.key == nckC_Z) )
         {
            /* KEYCODE RESERVED FOR FUTURE DEVELOPMENT */
         }

         else if ( Info.wk.type == wktFUNKEY &&    // move files to trashcan
                   (Info.wk.key == nckCDEL) )
         {
            this->CmdTrash ( mcFcmdMB_TRASH ) ;
         }

         else if ( Info.wk.type == wktFUNKEY &&    // delete (unlink) files
                   (Info.wk.key == nckSDEL) )
         {
            this->CmdDelete () ;
         }

         else if ( Info.wk.type == wktFUNKEY &&    // update (refresh) data in 
                   (Info.wk.key == nckC_U) )       // active window
         {
            this->CmdRefresh () ;
         }

         else if ( Info.wk.type == wktFUNKEY &&    // rename selected files
                   (Info.wk.key == nckC_R) )
         {
            this->CmdRename () ;
         }

         else if ( Info.wk.type == wktFUNKEY &&    // touch file date/timestamp
                   (Info.wk.key == nckC_D) )
         {
            this->CmdTouch () ;
         }

         else if ( Info.wk.type == wktFUNKEY &&    // write-protect selected files
                   (Info.wk.key == nckC_W) )
         {
            this->CmdWProtect ( true ) ;
         }

         else if ( Info.wk.type == wktFUNKEY &&    // write-enable selected files
                   (Info.wk.key == nckC_E) )
         {
            this->CmdWProtect ( false ) ;
         }

         else if ( Info.wk.type == wktFUNKEY &&    // open sort-file context menu
                   (Info.wk.key == nckC_S) )
         {
            this->CmdContextSort () ;
         }

         else if ( Info.wk.type == wktFUNKEY &&    // create new subdirectory
                   (Info.wk.key == nckC_N) )
         {
            this->CmdCreateDir () ;
         }

         else if ( Info.wk.type == wktFUNKEY &&    // find files
                   (Info.wk.key == nckC_F) )
         {
            this->CmdFind () ;
         }

         else if ( Info.wk.type == wktFUNKEY &&    // open tree-view window
                   (Info.wk.key == nckC_T) )
         {
            this->CmdDirTree () ;
         }

         else if ( Info.wk.type == wktFUNKEY &&    // open new target directory
                   (Info.wk.key == nckC_O) )       // from favorites list
         {
            this->CmdFavorites () ;
         }

         else if ( Info.wk.type == wktFUNKEY &&    // mount a filesystem
                   (Info.wk.key == nckC_Y) )       // from mount list
         {
            this->CmdMount ( mcFileMB_MOUNT ) ;
         }

         else if ( Info.wk.type == wktFUNKEY &&    // display Texinfo Help
                   (Info.wk.key == nckF01 || Info.wk.key == nckSF01) )
         {
            // NOTE: F01 key probably won't get through to the terminal 
            //       application, so Shift+F01 is the alternate
            this->CmdCallHelp () ;
         }

         else if ( Info.wk.type == wktFUNKEY &&    // display shortcut key menu
                   (Info.wk.key == nckC_K) )
         {
            this->CmdKeyBindings () ;
         }

         //* Exit application to invocation directory *
         else if ( Info.wk.type == wktFUNKEY &&
                   (Info.wk.key == nckC_Q || Info.wk.key == nckESC) )
         {
            this->exit2cwd = false ;
            menuCode = mcFileMB_EXIT ;
            done = true ;
         }

         //* Else, if a printing character, it is a possible attempt by user  *
         //* to find a file in the current view (i.e. first char of filename).*
         else if ( Info.wk.type == wktPRINT )
         {
            this->CmdLocateFile ( Info.wk ) ;
         }

         //* Test un-converted mouse events for association with our special   *
         //* hot zones. These zones are not associated with any control object.*
         // Programmer's Note: We should not get a mouse event unless the
         // mouse is enabled, but be safe, never trust the ncurses interface.
         else if ( (Info.wk.type == wktMOUSE) && (this->dWin->meMouseEnabled ()) )
         {
            //* Only click events are interesting, ScrollWheel events ignored. *
            if ( Info.wk.mevent.meType == metB1_S || Info.wk.mevent.meType == metB1_D )
            {
               //* A click event within the path Textbox above a Scrollext     *
               //* control indicates that user wants to navigate to parent     *
               //* directory for that control.                                 *
               // Programmer's Note: These Texbox controls are inactive, so
               // mouse-events above them will not be converted to keycodes.
               if ( (Info.wk.mevent.ypos == this->ic[dwPathTB].ulY) )
               {
                  if ( (Info.wk.mevent.xpos >= this->ic[dwPathTB].ulX) &&
                       (Info.wk.mevent.xpos < 
                          (this->ic[dwPathTB].ulX + this->ic[dwPathTB].cols)) )
                  {  //* Left (or only) path Textbox is the target *
                     this->fWin->ParentDirectory () ;
                     //* If not the active window, remove highlight *
                     if ( this->fdPtr != this->fWin )
                        this->fWin->RedrawCurrDir ( false ) ;
                     this->RestoreQuickHelpMsg () ;
                  }
                  else if ( (this->rWin != NULL) &&
                            ((Info.wk.mevent.xpos >= this->ic[dwPathTBr].ulX) &&
                             (Info.wk.mevent.xpos < 
                             (this->ic[dwPathTBr].ulX + this->ic[dwPathTBr].cols))) )
                  {  //* Right Textbox is the target *
                     this->rWin->ParentDirectory () ;
                     //* If not the active window, remove highlight *
                     if ( this->fdPtr != this->rWin )
                        this->rWin->RedrawCurrDir ( false ) ;
                     this->RestoreQuickHelpMsg () ;
                  }
               }
            }
         }

         //* Else is invalid input in this context *
         else
         {
            this->dWin->UserAlert ( 1 ) ;
         }
      }
   }           // while()

   return menuCode ;

}  //* End UserInterface() *

//*************************
//*      uiCallback       *
//*************************
//********************************************************************************
//* Callback method to monitor user input during the UserInterface() loop.       *
//*                                                                              *
//* Programmer's Note: We use a totally inappropriate indirection here,          *
//* but it is needed in order to use a member method as the callback target.     *
//*                                                                              *
//* Input  : currIndex : index of control with input focus                       *
//*          wkey      : user's key input                                        *
//*          firstTime : 'true' if first time method is called, else 'false'     *
//*                                                                              *
//* Returns: OK                                                                  *
//********************************************************************************

static short uiCallback ( const short currIndex, const wkeyCode wkey, bool firstTime )
{

   return ( (fmangPtr->uiMonitor ( currIndex, wkey, firstTime )) ) ;

}  //* End uiCallback() *

//*************************
//*       uiMonitor       *
//*************************
//********************************************************************************
//* Callback method to monitor user input during the UserInterface() loop.       *
//* Action is taken ONLY if one of the File-display controls has input focus.    *
//* -- Also called directly by CmdLocateFile() method to handle synchronized     *
//*    scan-for-filename operations.                                             *
//*                                                                              *
//* 1) Handle Synch Lock processing. For Dual-window Mode only, navigate         *
//*    through the inactive window in synch with the active window.              *
//*                                                                              *
//* Input  : currIndex : index of control with input focus                       *
//*          wkey      : user's key input                                        *
//*          firstTime : 'true' if first time method is called, else 'false'     *
//*                                                                              *
//* Returns: OK                                                                  *
//********************************************************************************
//* Notes:                                                                       *
//* ------                                                                       *
//* SynchLock:                                                                   *
//*  a) Note that if the two windows do not contain the same set of              *
//*     filenames, then the scrolling cannot be perfectly synchronized.          *
//*  b) Entering a subdirectory will work ONLY if the highlight in both          *
//*     directories references the same directory name. Entering a child         *
//*     directory in the active window when there is no corresponding child      *
//*     directory in the inactive window will disable the SynchLock. This is     *
//*     necessary because a subsequent move to parent directory will cause       *
//*     serious user confusion (they are simple creatures after all).            *
//*  c) Moving to the parent directory should always work unless one or both     *
//*     windows is already at root directory.                                    *
//*  d) Moving the focus between windows is important to the overall             *
//*     functionality, but may result in an unexpected (out-of-synch)            *
//*     situation which we do not catch. (corrected on next scroll-keypress)     *
//*  e) The current file-sort option may affect smoothness of the synch'd        *
//*     movement. Alphabetical sort is recommended, but not enforced.            *
//*  g) There are logical holes in this algorithm:                               *
//*     1) If there is only one filename displayed, then highlight in            *
//*        inactive window may not always work (but it won't really matter).     *
//*     2) We must do some defensive programming in case the first displayed     *
//*        item in each window does not match                                    *
//*     3) If filename lists are wildly different, the SynchLock will be more    *
//*        confusing than useful. In that case, the user SHOULD realize it       *
//*        fairly quickly and disable it.                                        *
//*     4) The documentation says that cut/copy/paste/etc. will automagically    *
//*        disable the synchlock. They do not! It's OK if the synchlock stays    *
//*        active after the data in one of the windows changes, but if so the    *
//*        highlight will need to be resynched but it requires knowing when      *
//*        the mainline code has refreshed the active window, OR requires this   *
//*        method to explicity monitor all the keycodes which could modify the   *
//*        data and determine when to resynch. This is Big-Brother-ish, and      *
//*        we resist doing it. See conditional compile switch below which        *
//*        isolates this functionality.                                          *
//*                                                                              *
//********************************************************************************

short FileMangler::uiMonitor ( const short currIndex, const wkeyCode wkey, bool firstTime )
{
   #define DEBUG_uiMONITOR (0)      // for debugging only

   static gString substr ;          // match filename

   if ( firstTime != false )
   {
      /* Nothing to do at this time. */
   }

   //* Examine only the keystrokes collected by the file-display controls.*
   else if ( currIndex == dwFileSE || currIndex == dwFileSEr )
   {
      //* Index of inactive Scrollext control (Dual-window Mode only) *
      short inactCtrl = currIndex == dwFileSE ? dwFileSEr : dwFileSE ;

      //* Note that user may have remapped the command key(s),*
      //* so we must decode the input.                        *
      //* Convert user input to default operational key code. *
      wkeyCode wk = wkey ;
      this->keyMap.decodeKeyInput( wk ) ;

      //* If the Synch Lock is active, analyze the user's input.           *
      //* 1) If ENTER key on subdirectory name, go there.                  *
      //* 2) If BACKSPACE or ALT+UPARROW, go to parent directory.          *
      //* 3) If a navigation key, echo that key in the inactive window.    *
      //* 4) If scanning by filename, perform same scan in inactive window.*
      //* 5) If switching focus to the inactive window, do housekeeping.   *
      if ( this->synchLock )
      {
         gString gsCmp ;                              // for comparing filenames, etc.
         attr_t msgAttr[this->ic[dwMsgsTB].cols] ;    // status messages
         for ( short i = ZERO ; i < this->ic[dwMsgsTB].cols ; ++i )
            msgAttr[i] = this->cfgOpt.cScheme.pf ;
         gsCmp.shiftCols( this->ic[dwMsgsTB].cols ) ;
         dtbmData msgData( gsCmp.ustr(), msgAttr, this->ic[dwMsgsTB].cols ) ;

         //* Get information of the highlighted filenames in each window *
         tnFName tnfActive, tnfInactive ;
         short aoff = this->fdPtr->GetStats ( tnfActive ) ;
         short ioff = this->altPtr->GetStats ( tnfInactive ) ;

         //* If one of the navigation keys echo it in the inactive window.     *
         if ( (wk.type == wktFUNKEY) && 
               (wk.key == nckUP   || wk.key == nckDOWN || 
                wk.key == nckPGUP || wk.key == nckPGDOWN ||
                wk.key == nckHOME || wk.key == nckEND) )
         {
            this->dWin->MoveScrollextHighlight ( inactCtrl, wk.key ) ;
            ioff = this->altPtr->GetStats ( tnfInactive ) ;

            //* Remove "highlight" from inactive window.*
            this->altPtr->DeselectFile ( true ) ;
            this->altPtr->RedrawCurrDir ( false ) ;

            //* If highlighted file in inactive window is not *
            //* a match, search for matching file.            *
            gsCmp = tnfActive.fName ;
            if ( (gsCmp.compare( tnfInactive.fName )) != ZERO )
            {
               // Highlight matching inactive file by name if it exists.
               if ( (this->altPtr->Scroll2MatchingFile ( gsCmp, ioff, true )) )
               {
                  ioff = this->altPtr->GetStats ( tnfInactive ) ;
                  this->altPtr->SelectFile ( false, ZERO ) ;
                  this->altPtr->RedrawCurrDir ( false ) ;
                  msgData.textData[ZERO] = NULLCHAR ;
               }
               else
               {  //* Item not found, now out-of-synch. *
                  gsCmp.compose( "Matching file for \"%s\" not found.", tnfActive.fName ) ;
                  gsCmp.copy( msgData.textData, gsALLOCDFLT ) ;
                  msgData.centered = true ;
               }
            }
            else
            {
               //* "Highlight" the matching file *
               this->altPtr->SelectFile ( false, ZERO ) ;
               this->altPtr->RedrawCurrDir ( false ) ;
               msgData.textData[ZERO] = NULLCHAR ;
            }
            this->dWin->DisplayTextboxMessage ( dwMsgsTB, msgData ) ;
         }

         //* If ENTER key AND highlight is on a subdirectory name, navigate to *
         //* the indicated subdirectory in the inactive window.                *
         //* The main loop will handle navigation in the active window.        *
         else if ( (tnfActive.fType == fmDIR_TYPE) && (wk.type == wktFUNKEY &&
                   (wk.key == nckENTER || wk.key == nckpENTER)) )
         {
            gsCmp = tnfActive.fName ;
            if ( (tnfInactive.fType == fmDIR_TYPE) && 
                 ((gsCmp.compare ( tnfInactive.fName )) == ZERO) )
            {
               this->altPtr->ChildDirectory () ;
               this->altPtr->SelectFile ( false, ZERO ) ;   // "highlight" the first item
               this->altPtr->RedrawCurrDir ( false ) ;
            }
            else
            {
               gsCmp = "Matching target directory "
                       "not found. SynchLock disengaged." ;
               gsCmp.copy( msgData.textData, gsALLOCDFLT ) ;
               msgData.centered = true ;
               this->dWin->DisplayTextboxMessage ( dwMsgsTB, msgData ) ;
               this->CmdSynchLock () ;
               chrono::duration<short>aWhile( 2 ) ;
               this_thread::sleep_for( aWhile ) ;
            }
         }

         //* If BACKSPACE or ALT+UPARROW navigate to parent directory (if any).*
         else if ( (wk.type == wktFUNKEY) && 
                   (wk.key == nckBKSP || wk.key == nckAUP) )
         {
            gString childName ;
            this->altPtr->GetPath ( childName ) ;
            if ( childName.gschars() > 2 )   // if not already at root directory
            {
               short fnIndex ;
               childName.shiftChars( -((childName.findlast( L'/' )) + 1) ) ;
               this->altPtr->ParentDirectory () ;
               this->altPtr->Scroll2MatchingFile ( childName, fnIndex, true ) ;
               this->altPtr->SelectFile ( false, ZERO ) ;   // "highlight" the matching item
               this->altPtr->RedrawCurrDir ( false ) ;
            }
         }

         //* If user changed focus to other window.      *
         //*      (It hasn't happened yet.)              *
         //* 1) De-select all files in inactive window.  *
         //* 2) Bump highlight to enable "highlight" of  *
         //*    matching file in newly inactive window.  *
         else if ( (wk.type == wktFUNKEY) && 
            ((wk.key == nckLEFT) || (wk.key == nckRIGHT) || 
             (wk.key == TAB) || (wk.key == nckSTAB)) )
         {
            this->altPtr->DeselectFile ( true ) ;
            wkeyCode wkt( nckDOWN, wktFUNKEY ) ;
            this->dWin->UngetKeyInput ( wkt ) ;
            wkt.key = nckUP ;
            this->dWin->UngetKeyInput ( wkt ) ;
         }

         //* If user is trying to locate a file by typing its name, *
         //* set the highlight of the inactive window to match.     *
         //* SPECIAL CASE: The delete key acts as a backspace key   *
         //* erase the last character entered into 'substr'.        *
         else if ( ((wk.type == wktPRINT) && (isprint ( wk.key ))) ||
                   ((wk.type == wktFUNKEY) && (wk.key == nckDELETE)) )
         {
            short newIndex ;
            if ( wk.key == nckDELETE )
               substr.limitChars( substr.gschars() - 2 ) ;
            else
               substr.append( L"%C", &wk.key ) ;
            if ( (this->altPtr->Scroll2MatchingFile ( substr, newIndex )) )
            {
               this->altPtr->DeselectFile ( true ) ;
               this->altPtr->SelectFile ( false, ZERO ) ;
               this->altPtr->RedrawCurrDir ( false ) ;
               ioff = this->altPtr->GetStats ( tnfInactive ) ;
            }
         }

         #if 0    // auto-disable of SynchLock currently disabled
         //* If an operation key, disable the Synch Lock. *
         //* See notes in method header.                  *
         else if ( (wk.type == wktFUNKEY) && 
              (   (wk.key == nckC_C)   // Copy
               || (wk.key == nckC_X)   // Cut
               || (wk.key == nckC_V)   // Paste
               || (wk.key == nckC_R)   // Rename
               || (wk.key == nckCDEL)  // Sent-to-trashcan
               || (wk.key == nckSDEL)  // Unconditional delete
              )
            )
         {
            this->CmdSynchLock () ;
         }
         #endif   // U/C

         //* If not a printing character, (or DEL) clear the search substring.*
         if ( (wk.type != wktPRINT) && !(wk.type == wktFUNKEY && wk.key == nckDELETE) )
            substr.clear() ;

         //*******************************
         //** Debug uiMonitor synchLock **
         //*******************************
         #if DEBUG_uiMONITOR != 0
         const short LINES = 10, 
                     COLS = this->ic[dwFileSE].cols - 14,
                     ulY = ZERO, 
                     ulX = (currIndex == dwFileSE) ? this->ic[dwFileSE].ulX + 14
                           : this->ic[dwFileSEr].ulX + 14 ;
         const wchar_t* typeNames[] = { L"PRINT ", L"FUNKEY", L"EXTEND", L"MOUSE ", L"ESCSEQ", L"ERROR " } ;
         attr_t dColor = nc.blR,
                hColor = nc.grbl ;
         //* Access to debugging data for key translation *
         const char **TransTable1, **TransTable2, **TransTable3 ;
         short tt1Min, tt1Max, tt2Min, tt2Max, tt3Entries ;
         nc.GetTransTable1 ( TransTable1, tt1Min, tt1Max ) ;
         nc.GetTransTable2 ( TransTable2, tt2Min, tt2Max ) ;
         nc.GetTransTable3 ( TransTable3, tt3Entries ) ;
         short typeIndex ;
         if ( wk.type == wktPRINT )        typeIndex = ZERO ;
         else if ( wk.type == wktFUNKEY )  typeIndex = 1 ;
         else if ( wk.type == wktEXTEND )  typeIndex = 2 ;
         else if ( wk.type == wktMOUSE  )  typeIndex = 3 ;
         else if ( wk.type == wktESCSEQ )  typeIndex = 4 ;
         else                              typeIndex = 5 ;
         gString gsOut ;
         winPos  wp( 1, 1 ) ;
         if ( wk.type == wktPRINT )
         {
            typeIndex = ZERO ;
            gsOut.compose( "i:%hu type:%S key:%C ft:%hhu\n\n", 
                           &currIndex, typeNames[typeIndex], &wk.key, &firstTime ) ;
         }
         else if ( wk.type == wktFUNKEY )
         {
            typeIndex = 1 ;
            const char* nPtr ;
            if ( wk.key >= tt1Min && wk.key <= tt1Max )
               nPtr = TransTable1[wk.key] ;
            else if ( wk.key >= tt2Min && wk.key <= tt2Max )
               nPtr = TransTable2[wk.key - (tt1Max + 1)] ;
            else
               nPtr = "OVERFLOW" ;
            gsOut.compose( "i:%hu type:%S key:%s ft:%hhu\n\n", 
                           &currIndex, typeNames[typeIndex], nPtr, &firstTime ) ;
         }
         else if ( wk.type == wktEXTEND )
         {
            typeIndex = 2 ;
            short nIndex ;
            nIndex = wk.key ;
            if ( nIndex < ZERO || nIndex >= tt3Entries )
               nIndex = tt3Entries - 1 ;
            const char* nPtr = TransTable3[nIndex] ;
            gsOut.compose( "i:%hu type:%S key:%s ft:%hhu\n\n", 
                           &currIndex, typeNames[typeIndex], nPtr, &firstTime ) ;
         }
         else if ( wk.type == wktMOUSE  )
         {
            typeIndex = 3 ;
            gsOut.compose( "i:%hu type:%S key:%s ft:%hhu\n\n", 
                           &currIndex, typeNames[typeIndex], "Unhandled Mouse Event", &firstTime ) ;
         }
         else if ( wk.type == wktESCSEQ )
         {
            typeIndex = 4 ;
            gsOut.compose( "i:%hu type:%S key:%s ft:%hhu\n\n", 
                           &currIndex, typeNames[typeIndex], "Unhandled Escape Sequence", &firstTime ) ;
         }
         else
         {
            typeIndex = 5 ;
            gsOut.compose( "i:%hu type:%S key:%s ft:%hhu\n\n", 
                           &currIndex, typeNames[typeIndex], "Unhandled Key Code", &firstTime ) ;
         }

         InitNcDialog dInit( LINES,    // number of display lines
                             COLS,     // number of display columns
                             ulY,      // Y offset from upper-left of terminal 
                             ulX,      // X offset from upper-left of terminal 
                             "  Debug uiMonitor  ",// dialog title
                             ncltSINGLE,// border line-style
                             dColor,   // border color attribute
                             dColor,   // interior color attribute
                             NULL      // pointer to list of control definitions
                        ) ;
         NcDialog* dp = new NcDialog ( dInit ) ;
         bool dpOpen = false ;
         if ( (dpOpen = (dp->OpenWindow()) == OK) )
         {  //* Report the input *
            wp = dp->WriteParagraph ( wp, gsOut, dColor ) ;

            winPos wp2 = wp ; wp2.xpos += 30 ;
            gsOut.compose( "hIndex: %hd\n%s\nType: %S\n", 
                           &aoff, tnfActive.fName, 
                           (tnfActive.fType == fmDIR_TYPE ? L"DIR" : 
                            (tnfActive.fType == fmREG_TYPE ? L"REG" : L"OTHER")) ) ;
            wp = dp->WriteParagraph ( wp, gsOut, dColor ) ;

            gsOut.compose( "hIndex: %hd\n%s\nType: %S\n", 
                           &ioff, tnfInactive.fName, 
                           (tnfInactive.fType == fmDIR_TYPE ? L"DIR" : 
                            (tnfInactive.fType == fmREG_TYPE ? L"REG" : L"OTHER")) ) ;
            wp2 = dp->WriteParagraph ( wp2, gsOut, dColor ) ;
            if ( (substr.gschars()) > 1 )
            {
               wp2.ypos -= 4 ;
               wp2 = dp->WriteString ( wp2, "substr: ", dColor ) ;
               dp->WriteString ( wp2, substr, hColor ) ;
            }

            dp->WriteString ( dInit.dLines - 2, 2, "Press A key", 
                              this->cfgOpt.cScheme.wr, true ) ;
            nckPause();
            delete ( dp ) ;
            this->dWin->RefreshWin () ;
         }
         #else    // silence the compiler warning
         if ( aoff == ioff ) ;
         #endif   // DEBUG_uiMONITOR
      }
   }

   return OK ;

   #undef DEBUG_uiMONITOR
}  //* End uiMonitor() *

//*************************
//*       CmdENTER        *
//*************************
//********************************************************************************
//* User has pressed the ENTER key (see note below).                             *
//*                                                                              *
//* nckENTER   ('altEnter' == false):                                            *
//* a) If highlight is on a directory name, move to the child directory.         *
//* b) If highlight is on a non-directory filename, open the ViewFile context    *
//*    menu and ask user what he/she wants to see.                               *
//* c) If highlight is on empty space, ignore the request.                       *
//*                                                                              *
//* nckAC_J ('altEnter' != false):                                               *
//* a) If highlight is on a directory name or other filename, open the           *
//*    ViewFile context menu and ask user what he/she wants to see.              *
//* b) If highlight is on empty space, ignore the request.                       *
//*                                                                              *
//*                                                                              *
//* Input  : altEnter : (optional, 'false' by default)                           *
//*                     if 'false', we arrived here via nckENTER                 *
//*                     if 'true',  we arrived here via ALT+nckENTER (nckAC_J)   *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************
//* Note on keycodes:                                                            *
//*  -- ENTER, SHIFT+ENTER and CTRL+ENTER are all the same keycode: nckENTER.    *
//*  -- ALT+ENTER, ALT+CTRL+ENTER, SHIFT+ALT+ENTER and SHIFT+ALT+CTRL+ENTER      *
//*     are all the same keycode: nckAC_J.                                       *
//*                                                                              *
//********************************************************************************

void FileMangler::CmdENTER ( bool altEnter )
{
   //* If current directory is empty, GetStats() will fail *
   tnFName  fn ;
   if ( (this->fdPtr->GetStats ( fn )) != ERR )
   {
      //* Navigate to child directory *
      if ( (fn.fType == fmDIR_TYPE) && (altEnter == false) )
      {
         if ( (this->fdPtr->ChildDirectory ()) != OK )
         {
            // NOTE: If change-directory failed, current directory was re-read
            // so no error will be reported here. (error handled at lower level)
         }
         this->RestoreQuickHelpMsg () ;
      }
      //* Ask user what view of the file to use *
      else
      {
         //* Position the context menu. Within the active dctSCROLLEXT      *
         //* control, open near the top of the control and in a position    *
         //* that does not obscure the filename.                            *
         short cIndex = this->dWin->GetCurrControl () ;
         short offsetY = this->ic[cIndex].ulY + 2,
               offsetX = this->ic[cIndex].ulX + 15 ;
         this->dWin->PositionContextMenu ( dwVfileCM, offsetY, offsetX ) ;

         //* Depending on the file type, of the highlighted file, *
         //* activate/deactivate the 'Find Link Target' menu item.*
         this->SetActiveMenuItems ( dwVfileCM ) ;

         //* Open the context menu for user to select type of file view *
         uiInfo   Info ;
         this->dWin->EditMenuwin ( dwVfileCM, Info ) ;
         if ( Info.dataMod != false )
         {
            MenuCode mCode = this->MenuItem2MenuCode ( Info.ctrlIndex, Info.selMember ) ;
            if ( (fn.fType == fmDIR_TYPE) && (altEnter != false) && 
                 (mCode == mcVfileCM_CONTENTS) )
            {  //* Navigate to child directory *
               this->fdPtr->ChildDirectory () ;
            }
            else
               this->CmdView ( mCode ) ;
         }
         else
         { /* no selection */ }

         //* Restore menu to its sub-menu position *
         offsetY = this->ic[dwVfileCM].ulY ;
         offsetX = this->ic[dwVfileCM].ulX ;
         this->dWin->PositionContextMenu ( dwVfileCM, offsetY, offsetX ) ;
      }
   }
//THIS INTERFERES WITH LOWER-LEVEL MESSAGES   this->RestoreQuickHelpMsg () ;

}  //* End CmdENTER() *

//*************************
//*     CmdParentDir      *
//*************************
//********************************************************************************
//* User has pressed the Backspace key or ALT+UpArrow.                           *
//* Climb up to the next-higher directory level (if any)                         *
//*                                                                              *
//* Highlight will be positioned on the name of the child directory from         *
//* which we just came.                                                          *
//*                                                                              *
//* Input  : none                                                                *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************

void FileMangler::CmdParentDir ( void )
{
   //* If not already at root directory *
   if ( ! (this->isRoot ()) )
   {
      //* Get the name of the current directory *
      gString  childName ;
      this->fdPtr->GetPath ( childName ) ;
      short fnIndex = childName.findlast( L'/' ) + 1 ;
      childName.shiftChars( -(fnIndex) ) ;

      this->fdPtr->ParentDirectory () ;

      //* Highlight the directory from which we have returned *
      this->fdPtr->Scroll2MatchingFile ( childName, fnIndex, true ) ;
   }
   else
      this->dWin->UserAlert () ;

   this->RestoreQuickHelpMsg () ;

}  //* End CmdParentDir() *

//*************************
//*       CmdCopy         *
//*************************
//********************************************************************************
//* Copy files to clipboard.                                                     *
//*                                                                              *
//* Input  : none                                                                *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************

void FileMangler::CmdCopy ( void )
{
   //* If the CWD is the root directory, disallow file operations.*
   if ( (this->isRoot ()) )
   { this->RootWarning () ;  return ; }   // NOTE THE EARLY RETURN

   dtbmData    msgData ;                     // message display

   //* If in Dual-Win mode, clear any file selections in the alternate control *
   this->ClearAltFileListSelections () ;

   //* Copy 'selected' files (or file under cursor) to the clipboard           *
   //* If clipboard already contains data, it will be released, and if previous*
   //* 'selections' for clipboard data are visible, they will be cleared.      *
   if ( (this->fdPtr->Copy2Clipboard ()) != false )
   {
      //* Give user a clue *
      UINT   cbFiles = this->fdPtr->ClipboardFiles () ;
      UINT64 cbSize  = this->fdPtr->ClipboardSize () ;
      short fieldWidth = 7 ;
      char fBuff[fieldWidth+2], sBuff[fieldWidth+2] ;
      gString gs ;
      gs.formatInt( cbFiles, fieldWidth, true ) ;
      gs.copy( fBuff, fieldWidth+1 ) ;
      fieldWidth = 5 ;
      gs.formatInt( cbSize, fieldWidth, true ) ;
      gs.copy( sBuff, fieldWidth+1 ) ;
      gs.compose( L"Clipboard files: %s (%s bytes) 'Paste'= copy files to target.", 
                 fBuff, sBuff ) ;
      gs.copy( msgData.textData, gsDFLTBYTES ) ;
   }
   else
   { /* Copy-to-clipboard failed because no files in this directory */ }
   this->dWin->DisplayTextboxMessage ( dwMsgsTB, msgData ) ;

}  //* End CmdCopy() *

//*************************
//*        CmdCut         *
//*************************
//********************************************************************************
//* Cut files to clipboard.                                                      *
//*                                                                              *
//* Input  : none                                                                *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************

void FileMangler::CmdCut ( void )
{
   //* If the CWD is the root directory, disallow file operations.*
   if ( (this->isRoot ()) )
   { this->RootWarning () ;  return ; }   // NOTE THE EARLY RETURN

   dtbmData    msgData ;                     // message display

   //* If in Dual-Win mode, clear any file selections in the alternate control *
   if ( (this->ClearAltFileListSelections ()) == false )
      this->RestoreQuickHelpMsg () ;

   //* Cut 'selected' files (or file under cursor) to the clipboard *
   if ( (this->fdPtr->Cut2Clipboard ()) != false )
   {
      //* Give user a clue *
      ULONG cbFiles = this->fdPtr->ClipboardFiles (),
            cbSize  = this->fdPtr->ClipboardSize () ;
      short fieldWidth = 7 ;
      char fBuff[fieldWidth+2], sBuff[fieldWidth+2] ;
      gString gs ;
      gs.formatInt( cbFiles, fieldWidth, true ) ;
      gs.copy( fBuff, fieldWidth+1 ) ;
      fieldWidth = 5 ;
      gs.formatInt( cbSize, fieldWidth, true ) ;
      gs.copy( sBuff, fieldWidth+1 ) ;
      gs.compose( L"Clipboard files: %s (%s bytes) 'Paste'= move files to target.", 
                 fBuff, sBuff ) ;
      gs.copy( msgData.textData, gsDFLTBYTES ) ;
   }
   else
   { /* Cut-to-clipboard failed because no files in this directory */ }
   this->dWin->DisplayTextboxMessage ( dwMsgsTB, msgData ) ;

}  //* End CmdCut() *

//*************************
//*       CmdPaste        *
//*************************
//********************************************************************************
//* Paste files from the clipboard and, if previous operation was a 'Cut',       *
//* delete the original files.                                                   *
//*                                                                              *
//* Input  : none                                                                *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************

void FileMangler::CmdPaste ( void )
{
   UINT        attCount,            // number of paste operations attempted
               pstCount,            // number of files successfully pasted
               delCount ;           // number of source files successfully deleted
   gString     gs ;                 // data formatting
   dtbmData    msgData ;            // message display

   //* If the clipboard list contains at least one name *
   if ( (this->fdPtr->ClipboardFiles ()) > ZERO )
   {
      bool cutPend = this->fdPtr->CutPending () ;  // if cut-and-paste operation
      bool forceAltRefresh = false ;         // 'true' if source dir is displayed
      if ( (this->DualwinMode ()) )
      {
         if ( cutPend && (this->altPtr->TargDirEqualsClipDir ( false )) )
            forceAltRefresh = true ;
      }

      //* Copy the files listed on the clipboard to the current working *
      //* directory and re-read contents of target directory.           *
      pstCount = this->fdPtr->PasteClipboardList ( attCount, delCount ) ;

      //* If in Dual-Win mode AND source and target are the same directory,    *
      //* OR if a 'cut-and-paste' operation from source in the alternate       *
      //* window, update the alternate window also.                            *
      this->RefreshAltFileList ( forceAltRefresh ) ;

      if ( ! cutPend )              // 'copy' operation
      {
         if ( pstCount == attCount )
            gs.compose( "  %u files copied", &pstCount ) ;
         else
         {
            UINT badCount = (attCount - pstCount) ;
            gs.compose( "  %u files copied successfully, but ERROR copying %u file(s)!", 
                        &pstCount, &badCount ) ;
         }
         gs.copy( msgData.textData, gsALLOCDFLT ) ;
      }
      else                          // 'move' operation
      {
         if ( pstCount == attCount && delCount == attCount )
            gs.compose( "  %u files moved", &pstCount ) ;
         else
         {
            UINT mCount  = (pstCount - delCount),
                 teCount = (attCount - pstCount),
                 seCount = (attCount - delCount) ;
            gs.compose( "  %u files moved, but %u target errors and %u source errors!", 
                       &mCount, &teCount, &seCount ) ;
         }
         gs.copy( msgData.textData, gsALLOCDFLT ) ;
      }
   }
   else
      msgData = " Clipboard is empty; nothing to paste!" ;
   this->dWin->DisplayTextboxMessage ( dwMsgsTB, msgData ) ;

}  //* End CmdPaste() *

//*************************
//*      CmdPasteSp       *
//*************************
//********************************************************************************
//* Paste files from the clipboard in one of the 'special' ways.                 *
//*  See enum psSubType for available paste-special options.                     *
//*                                                                              *
//* Input  : none                                                                *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************

void FileMangler::CmdPasteSp ( void )
{
UINT        attCount,               // number of paste operations attempted
            pstCount,               // number of files successfully pasted
            delCount ;              // number of source files successfully deleted
gString     gs ;                    // data formatting
dtbmData    msgData ;               // message display

   //* If the clipboard list contains at least one name *
   if ( (this->fdPtr->ClipboardFiles ()) > ZERO )
   {
      bool cutPend = this->fdPtr->CutPending () ; // if cut-and-paste operation
      bool forceAltRefresh = false ;         // 'true' if source dir is displayed
      if ( (this->DualwinMode ()) )
      {
         if ( cutPend && (this->altPtr->TargDirEqualsClipDir ( false )) )
            forceAltRefresh = true ;
      }

      //* Copy/move the files listed on the clipboard to the current *
      //* working directory and re-read contents of target directory.*
      pstCount = this->fdPtr->PasteSpecialList ( attCount, delCount ) ;

      //* If in Dual-Win mode AND source and target are the same directory,    *
      //* OR if a 'cut-and-paste' operation from source in the alternate       *
      //* window, update the alternate window also.                            *
      this->RefreshAltFileList ( forceAltRefresh ) ;

      if ( ! cutPend )              // 'copy' operation
      {
         if ( pstCount == attCount )
            gs.compose( "  %u files copied", &pstCount ) ;
         else
         {
            UINT badCount = (attCount - pstCount) ;
            gs.compose( "  %u files copied successfully, but ERROR copying %u file(s)!", 
                       &pstCount, &badCount ) ;
         }
         gs.copy( msgData.textData, gsALLOCDFLT ) ;
      }
      else                          // 'move' operation
      {
         if ( pstCount == attCount && delCount == attCount )
            gs.compose( "  %u files moved", &pstCount ) ;
         else
         {
            UINT mCount  = (pstCount - delCount),
                 teCount = (attCount - pstCount),
                 seCount = (attCount - delCount) ;
            gs.compose( "  %u files moved, but %u target errors and %u source errors!", 
                        &mCount, &teCount, &seCount ) ;
         }
         gs.copy( msgData.textData, gsALLOCDFLT ) ;
      }
   }
   else
      msgData = " Clipboard is empty; nothing to paste!" ;
   this->dWin->DisplayTextboxMessage ( dwMsgsTB, msgData ) ;

}  //* End CmdPasteSp() *

//*************************
//*      CmdDelete        *
//*************************
//********************************************************************************
//* Delete files.                                                                *
//*                                                                              *
//* Input  : none                                                                *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************
//* NOTE: If in Dual-Window mode AND if the inactive window is anywhere on the   *
//*       path of files being deleted, then after the operation, the path of     *
//*       inactive window will no longer exist and must be updated to a          *
//*       directory that actually does exist.                                    *
//********************************************************************************

void FileMangler::CmdDelete ( void )
{
   //* If the CWD is the root directory, disallow file operations.*
   if ( (this->isRoot ()) )
   { this->RootWarning () ;  return ; }   // NOTE THE EARLY RETURN


   UINT     selFiles,            // number of 'selected' files
            result ;             // number file files successfully deleted
   gString  gs ;                 // data formatting
   dtbmData msgData ;            // message display

   //* If in Dual-Win mode, clear any file selections in the alternate control *
   this->ClearAltFileListSelections () ;

   //* Delete the files that have been marked as 'selected' *
   result = this->fdPtr->DeleteSelectedFiles ( selFiles ) ;

   //* If in Dual-Win mode AND alternate window is the same as source    *
   //* dir (or an affected child dir), update the alternate window also. *
   if ( (this->DualwinMode ()) )
   {
      fmFType apType ;
      gString srcPath, trgPath ;
      this->fdPtr->GetPath ( srcPath ) ;
      this->altPtr->GetPath ( trgPath ) ;
      //* If alt window's target dir has disappeared (see note above) *
      if ( !(this->altPtr->TargetExists ( trgPath, apType )) )
      {
         trgPath = srcPath ;
         if ( (this->altPtr->SetDirectory ( trgPath )) == OK )
            this->altPtr->RedrawCurrDir ( false ) ;
         else
         { /* (this is unlikely) */ }
      }
      else if ( trgPath == srcPath )
         this->RefreshAltFileList () ;
   }

   if ( result == selFiles )
      gs.compose( "  %u files deleted", &result ) ;
   else
   {
      UINT badCount = (selFiles-result) ;
      gs.compose( "  %u files deleted successfully, "
                  "but ERROR deleting %u file(s)!", &result, &badCount ) ;
   }
   gs.copy( msgData.textData, gsALLOCDFLT ) ; 
   this->dWin->DisplayTextboxMessage ( dwMsgsTB, msgData ) ;

}  //* End CmdDelete() *

//***********************
//*     CmdRename       *
//***********************
//********************************************************************************
//* Rename a file or files. The source and target directories are the same       *
//* for renaming files. If source directory is different from target             *
//* directory, then we have a combination of 'Cut' and 'Paste Special'.          *
//*                                                                              *
//* Input  : none                                                                *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************

void FileMangler::CmdRename ( void )
{
   UINT        result,                 // result of calls to FileDlg classs
               selFiles ;              // number of 'selected' files
   gString     gs ;                    // data formatting
   dtbmData    msgData ;               // message display

   if ( (this->DualwinMode ()) )
   {
      //* If there are selections in the alternate window, remove them *
      if ( this->altPtr->IsSelected ( true ) )
      {
         this->altPtr->DeselectFile ( true ) ;
         this->altPtr->RedrawCurrDir ( false ) ;
      }
   }

   //* Rename the files that have been marked as 'selected' *
   result = this->fdPtr->RenameSelectedFiles ( selFiles ) ;

   //* If in Dual-Win mode AND alternate window is the   *
   //* same directory, update the alternate window also. *
   this->RefreshAltFileList () ;

   //* Report the results *   
   if ( result == selFiles )
      gs.compose( "  %u files renamed", &result ) ;
   else
   {
      UINT badCount = (selFiles-result) ;
      gs.compose( "  %u files renamed successfully, but ERROR renaming %u file(s)!", 
                  &result, &badCount ) ; 
   }
   gs.copy( msgData.textData, gsALLOCDFLT ) ; 
   this->dWin->DisplayTextboxMessage ( dwMsgsTB, msgData ) ;

}  //* End CmdRename() *

//***********************
//*     CmdTouch        *
//***********************
//********************************************************************************
//* Update date/timestamp of selected file(s).                                   *
//* Updates both 'modification' date and 'access' date to specified date/time    *
//* stamp.  Updates inode change time, 'ctime', to current system date/time.     *
//*                                                                              *
//* Input  : none                                                                *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************

void FileMangler::CmdTouch ( void )
{
UINT        result,                 // result of calls to FileDlg class
            selFiles ;              // number of 'selected' files
gString     gs ;                    // data formatting
dtbmData    msgData ;               // message display

   if ( (this->DualwinMode ()) )
   {
      //* If there are selections in the alternate window, remove them *
      if ( this->altPtr->IsSelected ( true ) )
      {
         this->altPtr->DeselectFile ( true ) ;
         this->altPtr->RedrawCurrDir ( false ) ;
      }
   }

   //* Interactively update the files' date/time stamp *
   result = this->fdPtr->TouchSelectedFiles ( selFiles ) ;

   //* If in Dual-Win mode AND alternate window is the   *
   //* same directory, update the alternate window also. *
   this->RefreshAltFileList () ;

   //* Report the results *   
   if ( result == selFiles )
      gs.compose( "  %u file dates modified", &result ) ;
   else
   {
      UINT badCount = (selFiles-result) ;
      gs.compose( "  %u file dates modified, but ERROR updating %u file(s)!", 
                  &result, &badCount ) ;
   }
   gs.copy( msgData.textData, gsALLOCDFLT ) ; 
   this->dWin->DisplayTextboxMessage ( dwMsgsTB, msgData ) ;

}  //* End CmdTouch() *

//***********************
//*     CmdRefresh      *
//***********************
//********************************************************************************
//* Refresh (re-read) and re-display the contents of the displayed directory.    *
//* Any file 'selections' will be discarded.                                     *
//*                                                                              *
//* Note that the data (if any) currently on the clipboard came from this        *
//* directory, then the clipboard will be cleared.                               *
//*                                                                              *
//* Input  : altWin    : (optional, 'false' by default)                          *
//*                      if 'false' update the active window                     *
//*                      if 'true' AND if in Dual-window Mode, update the        *
//*                          inactive window                                     *
//*          clipClear : (optional, 'false by default)                           *
//*                      if 'false', the clipboard is cleared only if there      *
//*                          are selected items in the target window             *
//*                      if 'true', then force the update to also clear the      *
//*                          clipboard                                           *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************
//* Programmer's Note: Use the 'clipClear' parameter sparingly. Clearing the     *
//* clipboard has side-effects, so do it only if you're 'sure' that it won't     *
//* harm any operation in progress.                                              *
//*                                                                              *
//********************************************************************************

void FileMangler::CmdRefresh ( bool altWin, bool clipClear )
{
   if ( altWin )
   {
      if ( (this->DualwinMode ()) )
         this->altPtr->RefreshCurrDir ( false, clipClear ) ;
   }
   else
      this->fdPtr->RefreshCurrDir ( true, clipClear ) ;

   dtbmData  msgData(" Directory Display Refreshed") ;
   this->dWin->DisplayTextboxMessage ( dwMsgsTB, msgData ) ;

}  //* End CmdRefresh() *

//***********************
//*    CmdWProtect      *
//***********************
//********************************************************************************
//* Write-protect OR write-enable 'selected' files.                              *
//*                                                                              *
//* Input  : protect: if 'true', write-protect selected files                    *
//*                   if 'false', write-enable selected files                    *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************

void FileMangler::CmdWProtect ( bool protect )
{
   UINT  modFiles,            // number of modified files
         selFiles ;           // number of 'selected' files

   if ( (this->DualwinMode ()) )
   {
      //* If there are selections in the alternate window, remove them *
      if ( this->altPtr->IsSelected ( true ) )
      {
         this->altPtr->DeselectFile ( true ) ;
         this->altPtr->RedrawCurrDir ( false ) ;
      }
   }

   //* Get user input (0==protect, 1==enable, 2==custom bitmap, ERR==aborted) *
   short op = this->fdPtr->ProtectSelectedFiles ( selFiles, modFiles, protect ) ;
   const char* const opName[] = 
   {
      "files write protected",
      "files write enabled",
      "permissions updated"
   } ;
   gString gs ;
   if ( op != ERR )
   {
      gs.compose( L"  %u %s", &modFiles, opName[op] ) ;
      if ( selFiles != modFiles )
      {
         gString gsErr ;
         UINT errFiles = selFiles - modFiles ;
         gsErr.compose( L", but ERROR updating %u file(s)!", &errFiles ) ;
         gs.append( gsErr.gstr() ) ;
      }
   }
   else
      gs = "  Operation cancelled, no files modified." ;

   dtbmData msgData ;               // message display
   gs.copy( msgData.textData, gsDFLTBYTES ) ;
   this->dWin->DisplayTextboxMessage ( dwMsgsTB, msgData ) ;

}  //* End CmdWProtect() *

//*************************
//*    CmdContextSort     *
//*************************
//********************************************************************************
//* Open the context menu for selecting the file sorting option, and let the     *
//* user have a go.                                                              *
//*                                                                              *
//* Input  : none                                                                *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************

void FileMangler::CmdContextSort ( void )
{
   //* Save the current position of the context menu. *
   short origY = this->ic[dwSortMB].ulY,
         origX = this->ic[dwSortMB].ulX ;
   //* Position the context menu. Within the active dctSCROLLEXT   *
   //* control, open near the top of the control and in a position *
   //* that does not obscure the filenames.                        *
   short cIndex = this->dWin->GetCurrControl () ;
   short offsetY = this->ic[cIndex].ulY + 2,
         offsetX = this->ic[cIndex].ulX + 15 ;
   this->dWin->PositionContextMenu ( dwSortMB, offsetY, offsetX ) ;

   //* Give user a clue *
   dtbmData  msgData(" Select a Sort Option (or ESC to close menu)") ;
   this->dWin->DisplayTextboxMessage ( dwMsgsTB, msgData ) ;

   //* Open the context menu for user to select file-sort option *
   uiInfo   Info ;
   this->dWin->EditMenuwin ( dwSortMB, Info ) ;
   if ( Info.dataMod != false )
   {
      //* Get the menu code and convert it to corresponding *
      //* sort-option code and re-sort the data.            *
      MenuCode mCode = this->MenuItem2MenuCode ( Info.ctrlIndex, Info.selMember ) ;
      this->CmdSort ( mCode ) ;
   }
   else
   { /* no selection */ }
   
   //* Restore original position of the menu *
   this->dWin->PositionContextMenu ( dwSortMB, origY, origX ) ;

}  //* End CmdContextSort() *

//*************************
//*       CmdSort         *
//*************************
//********************************************************************************
//* User has selected a file-sort option from the context menu.                  *
//* Translate the menu code into a sort-option code and set the new sorting      *
//* option.                                                                      *
//*                                                                              *
//* Input  : mCode: menu code                                                    *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************

void FileMangler::CmdSort ( MenuCode mCode )
{
bool  soModified = false ;

   fmSort   newSort ;
   switch ( mCode )
   {
      case mcSortMB_NAMEREV:  //* filename (high-to-low)
         newSort = fmNAMEr_SORT ;   break ;
      case mcSortMB_DATE:     //* mod date (low-to-high)
         newSort = fmDATE_SORT ;    break ;
      case mcSortMB_DATEREV:  //* mod date (high-to-low)
         newSort = fmDATEr_SORT ;   break ;
      case mcSortMB_SIZE:     //* file size (low-to-high)
         newSort = fmSIZE_SORT ;    break ;
      case mcSortMB_SIZEREV:  //* file size (high-to-low)
         newSort = fmSIZEr_SORT ;   break ;
      case mcSortMB_EXT:      //* file extension (low-to-high)
         newSort = fmEXT_SORT ;     break ;
      case mcSortMB_EXTREV:   //* file extension (high-to-low)
         newSort = fmEXTr_SORT ;    break ;
      case mcSortMB_TYPE:     //* file type (low-to-high)
         newSort = fmTYPE_SORT ;    break ;
      case mcSortMB_TYPEREV:  //* file type (high-to-low)
         newSort = fmTYPEr_SORT ;   break ;
      case mcSortMB_NOSORT:   //* no sort
         newSort = fmNO_SORT ;      break ;
      case mcSortMB_NAME:     //* sort by filename (low-to-high)
      default:
         newSort = fmNAME_SORT ;
         break ;
   }
   //* If user has changed the sort option *
   if ( newSort != this->cfgOpt.sortOption )
   {
      //* Update our local copy of config options and pass it *
      //* to the FileDlg that currently has the input focus.  *
      this->cfgOpt.sortOption = newSort ;
      if ( (this->fdPtr->UpdateSortOptions ( this->cfgOpt )) != false )
      {
         //* Re-display the data in the new sort order *
         this->fdPtr->RedrawCurrDir () ;

         //* If we are in Dual-Window mode, update *
         //* sort for alternate window also        *
         if ( (this->DualwinMode ()) )
         {
            this->altPtr->UpdateSortOptions ( this->cfgOpt ) ;
            this->altPtr->RedrawCurrDir ( false ) ;
         }
         soModified = true ;
      }
   }
   if ( soModified != false )
   {
      dtbmData  msgData(" Sort Order Modified!") ;
      this->dWin->DisplayTextboxMessage ( dwMsgsTB, msgData ) ;
   }
   else
      this->RestoreQuickHelpMsg () ;

}  //* End CmdSort() *

//*************************
//*    CmdDialogSort      *
//*************************
//********************************************************************************
//* Interactively set sort options for display of file lists.                    *
//*                                                                              *
//* Input  : mCode: menu code (identifies caller)                                *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************

void FileMangler::CmdDialogSort ( MenuCode mCode )
{
   dtbmData msgData( "Sort options not modified" ) ;
   //* Allow user to modify sort options. Data members of cfgOpt *
   //* may change: sortOption, caseSensitive, showHidden, and if *
   //* so, file display for window with focus will be updated.   *
   if ( (this->fdPtr->ModifySortOptions ( this->cfgOpt )) != false )
   {
      //* If in Dual-Win mode, update the alternate window also *
      if ( (this->DualwinMode ()) )
      {
         this->altPtr->UpdateSortOptions ( this->cfgOpt ) ;
         this->altPtr->RefreshCurrDir ( false, true ) ;
      }
      msgData = " Sort Options Modified!" ;
   }

   //* Report our results *
   this->dWin->DisplayTextboxMessage ( dwMsgsTB, msgData ) ;

}  //* End of CmdDialogSort() *

//*************************
//*    CmdHiddenFiles     *
//*************************
//********************************************************************************
//* Toggle between showing and hiding hidden files.                              *
//*                                                                              *
//* Input  : none                                                                *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************

void FileMangler::CmdHiddenFiles ( void )
{
   //* Toggle our local flag, then update the file-display window *
   this->cfgOpt.showHidden = this->cfgOpt.showHidden ? false : true ;
   this->fdPtr->UpdateSortOptions ( this->cfgOpt ) ;
   this->fdPtr->RefreshCurrDir ( true, true ) ;

   //* If in Dual-Win mode, update the alternate window also *
   if ( (this->DualwinMode ()) )
   {
      this->altPtr->UpdateSortOptions ( this->cfgOpt ) ;
      this->altPtr->RefreshCurrDir ( false ) ;
   }

   //* Report our results *
   gString gs( "  Hidden files are%sdisplayed.", 
               this->cfgOpt.showHidden ? " " : " not " ) ;
   dtbmData msgData( gs.ustr() ) ;
   this->dWin->DisplayTextboxMessage ( dwMsgsTB, msgData ) ;

}  //* End CmdHiddenFiles() *

//*************************
//*      CmdRootScan      *
//*************************
//********************************************************************************
//* Toggle the full-scan option for display of root directory.                   *
//* This option is disabled by default because a full scan of the directory      *
//* tree from the root directory is unbearably slow and prone to system          *
//* exceptions. Use with caution!                                                *
//*                                                                              *
//* Input  : none                                                                *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************

void FileMangler::CmdRootScan ( void )
{
   gString gs ;

   //* Get current state of root-scan flag *
   bool rsEnabled = this->fdPtr->ScanFromRoot ( true, true ) ;

   //* Toggle the setting *
   rsEnabled = bool(rsEnabled ? false : true) ;
   this->fdPtr->ScanFromRoot ( rsEnabled ) ;

   //* If full scan has been enabled _and_ CWD is  *
   //* root directory, re-read directory contents. *
   if ( rsEnabled )
   {
      this->fdPtr->GetPath ( gs ) ;
      if ( (gs.compare( ROOT_PATH )) == ZERO )
         this->fdPtr->RefreshCurrDir ( true, true ) ;

      //* If in Dual-Win mode, update the alternate window also *
      if ( (this->DualwinMode ()) )
      {
         this->altPtr->ScanFromRoot ( rsEnabled ) ;
         this->altPtr->GetPath ( gs ) ;
         if ( (gs.compare( ROOT_PATH )) == ZERO )
            this->altPtr->RefreshCurrDir ( false, true ) ;
      }
   }

   //* Report the results *
   gs.compose( "  Full root directory scan is %s.", 
               (rsEnabled ? "enabled" : "disabled") ) ;
   dtbmData msgData( gs.ustr() ) ;
   this->dWin->DisplayTextboxMessage ( dwMsgsTB, msgData ) ;

}  //* End CmdRootScan() *

//*************************
//*     CmdCreateDir      *
//*************************
//********************************************************************************
//* Call the FileDlg method for interactively creating a new directory.          *
//*                                                                              *
//*                                                                              *
//* Input  : none                                                                *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************

void FileMangler::CmdCreateDir ( void )
{
   //* Get directory name from user and create subdirectory in current dir *
   this->fdPtr->CreateDirectory () ;

   //* If in Dual-Win mode AND alternate window is the   *
   //* same directory, update the alternate window also. *
   this->RefreshAltFileList () ;

}  // End CmdCreateDir() *

//*************************
//*      CmdSelect        *
//*************************
//********************************************************************************
//* Select the highlighted filename.                                             *
//* If group filename selection in progress, then also select the previous or    *
//* following filename.                                                          *
//*                                                                              *
//* Input  : wk  : key type and key code with which selection was made           *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************
//* The group-selection sequence emulates (roughly) what GUI apps do in item     *
//* selection when Shift key is held down during up/down arrow keys. This is a   *
//* simple, non-mousy emulation and is not an exact parallel, but if we may      *
//* say so, it is actually more logical than the equivalent GUI sequence.        *
//* Logic:                                                                       *
//* 1) If moving down AND already at the bottom, then                            *
//*    - if prevSel, then select currItem                                        *
//*    - if !prevSel, then deselect currItem                                     *
//* 2) If moving up AND already at the top, then                                 *
//*    - if nextSel, then select currItem                                        *
//*    - if !nextSel, then deselect currItem                                     *
//* -- ELSE --                                                                   *
//* 3) If current item is not selected, then                                     *
//*    a) select current item                                                    *
//*    b) move to next/previous and select it also                               *
//* 4) If current is selected, then                                              *
//*    a) if next/prev NOT selected, go there and select it                      *
//*    b) if next/prev IS selected, then                                         *
//*      - deselect current item                                                 *
//*      - move to next/previous item                                            *
//*                                                                              *
//*                                                                              *
//********************************************************************************

void FileMangler::CmdSelect ( wkeyCode& wk )
{
   //* If the CWD is the root directory, disallow file selection.*
   if ( (this->isRoot ()) )
   { this->RootWarning () ;  return ; }   // NOTE THE EARLY RETURN

   //* If in Dual-Win mode, clear any file selections in the alternate control *
   if ( (this->ClearAltFileListSelections ()) == false )
      this->RestoreQuickHelpMsg () ;

   bool  groupSelect = false ;   // assume single-file selection and
   short dir = 1 ;               // moving highlight downward
   if (wk.type == wktFUNKEY && (wk.key == nckSDOWN || wk.key == nckSUP) )
   {
      groupSelect = true ;
      if ( wk.key == nckSUP )
         dir = -1 ;
   }
   //* If group selection in progress *
   if ( groupSelect )
   {
      short currItem, totalItems ;
      bool  prevSel, nextSel,
            currSel = this->fdPtr->IsSelected ( currItem, totalItems, 
                                                prevSel, nextSel ) ;
      if ( (dir == 1 && currItem < (totalItems - 1)) || 
           (dir == (-1) && currItem > ZERO) )
      {
         if ( ! currSel )
         {
            if ( (dir == 1 && !nextSel) || (dir == (-1) && !prevSel) )
            {
               this->fdPtr->SelectFile ( false, dir ) ;
               this->fdPtr->SelectFile ( false, ZERO ) ;
            }
            else if ( (dir == 1 && nextSel) || (dir == (-1) && prevSel) )
            {
               this->fdPtr->SelectFile ( false, ZERO ) ;
               this->fdPtr->DeselectFile ( false, dir ) ;
               this->fdPtr->DeselectFile ( false, ZERO ) ;
            }
         }
         else
         {
            if ( (dir == 1 && !nextSel) || (dir == -1 && !prevSel) )
            {
               this->fdPtr->DeselectFile ( false, ZERO ) ;
               this->fdPtr->SelectFile ( false, dir ) ;
               this->fdPtr->SelectFile ( false, ZERO ) ;
            }
            else if ( (dir == 1 && nextSel) || (dir == (-1) && prevSel) )
            {
               this->fdPtr->DeselectFile ( false, dir ) ;
            }
         }
      }
      else     // first or last item is highlighted
      {
         if ( !currSel && ((dir == 1 && prevSel) || (dir == (-1) && nextSel)) )
            this->fdPtr->SelectFile ( false, ZERO ) ;
         else if ( currSel && ((dir == 1 && !prevSel) || (dir == (-1) && !nextSel)) )
            this->fdPtr->DeselectFile ( false, ZERO ) ;
      }
   }
   //* Else, select or deselect only highlighted filename *
   else
   {
      if ( this->fdPtr->IsSelected () )
         this->fdPtr->DeselectFile ( false, dir ) ;
      else
         this->fdPtr->SelectFile ( false, dir ) ;
   }

}  //* End CmdSelect() *

//*************************
//*     CmdSelectAll      *
//*************************
//********************************************************************************
//* Select all files in the current window.                                      *
//*                                                                              *
//* Input  : none                                                                *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************

void FileMangler::CmdSelectAll ( void )
{
   //* If the CWD is the root directory, disallow file selection.*
   if ( (this->isRoot ()) )
   { this->RootWarning () ;  return ; }   // NOTE THE EARLY RETURN

   //* If in Dual-Win mode, clear any file selections in the alternate control *
   if ( (this->ClearAltFileListSelections ()) == false )
      this->RestoreQuickHelpMsg () ;

   if ( this->fdPtr->IsSelected ( true ) )// if file(s) already selected, deselect them
      this->fdPtr->DeselectFile ( true ) ;
   else                                // mark all files as 'selected'
      this->fdPtr->SelectFile ( true ) ;

}  //* End CmdSelectAll() *

//*************************
//*      CmdFind          *
//*************************
//********************************************************************************
//* Invoke a dialog from which the user can search for files at or below the     *
//* currently-displayed directory, and optionally change the CWD to the          *
//* directory of that file.                                                      *
//*                                                                              *
//* Input  : none                                                                *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************

void FileMangler::CmdFind ( void )
{
   gString newCwd ;
   winPos ulCorner( 1, 0 ) ;
   short dlgRows, dlgCols ;
   this->dWin->GetDialogDimensions ( dlgRows, dlgCols ) ;
   dlgRows -= 1 ;    // make it fit

   if ( (this->fdPtr->FindFiles ( newCwd, ulCorner, dlgRows, dlgCols )) != false )
      this->fdPtr->SetDirectory ( newCwd ) ;

   this->RestoreQuickHelpMsg () ;

}  //* End CmdFind() *

//*************************
//*       CmdInodes       *
//*************************
//********************************************************************************
//* Invoke a dialog from which the user can search for all files on the current  *
//* filesystem which share the INODE number specified by the user.               *
//*                                                                              *
//* Input  : none                                                                *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************

void FileMangler::CmdInodes ( void )
{
   winPos ulCorner( 1, 0 ) ;
   short dlgRows, dlgCols ;
   this->dWin->GetDialogDimensions ( dlgRows, dlgCols ) ;
   dlgRows -= 1 ;    // make it fit
   gString newCwd ;
   this->fdPtr->FindInodes ( newCwd, ulCorner, dlgRows, dlgCols ) ;

}  //* End CmdInodes() *

//*************************
//*        CmdGrep        *
//*************************
//********************************************************************************
//* Search selected files for substrings that match user's input.                *
//*                                                                              *
//* Input  : none                                                                *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************

void FileMangler::CmdGrep ( void )
{
   const short rows = 18,
               cols = (MIN_COLS - 2) ;
   short dlgRows, dlgCols ;
   this->dWin->GetDialogDimensions ( dlgRows, dlgCols ) ;
   winPos ulCorner( short(((this->ic[dwFileSE].lines - 2) >= rows) ? 
                           (this->ic[dwFileSE].ulY + 1) : 2),
                    short((dlgCols / 2) - (cols / 2)) ) ;
   if ( (this->fdPtr->GrepFiles ( ulCorner, rows, cols )) == false )
      this->RestoreQuickHelpMsg () ;

}  //* End CmdGrep() *

//*************************
//*       CmdView         *
//*************************
//********************************************************************************
//* User has selected an action using the CmdEnter() method or through the       *
//* menu system.                                                                 *
//*  1. View    display file as text/hex/decimal/octal,metadata, etc.            *
//*  2. Stat    display file stats, and if symbolic link, also display           *
//*             stats for target file                                            *
//*  3. Run     execute the file if it is executable, or load appropriate        *
//*             external program.                                                *
//*  4. Find    if highlighted file is a symbolic link, AND link target exists,  *
//*             go to the directory containing the link target                   *
//*                                                                              *
//* Input  : mCode : menu code for an item in the 'ViewFile' menu                *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************

void FileMangler::CmdView ( MenuCode mCode )
{
   switch ( mCode )
   {
      case mcVfileCM_CONTENTS:
         this->fdPtr->ViewFileContents () ;
         break ;
      case mcVfileCM_VIEWSTATS:
         this->fdPtr->DisplayFileStats () ;
         break ;
      case mcVfileCM_EXECUTE:
         this->fdPtr->OpenExternalProgram () ;
         break ;
      case mcVfileCM_FINDTARGET:
         this->fdPtr->FindLinkTarget () ;
         break ;
      case mcVfileCM_CANCELVIEW:
      default:
         //* nothing to do *
         break ;
   }

}  //* End CmdView() *

//*************************
//*      CmdBackup        *
//*************************
//********************************************************************************
//* Backup of files in the file-display window that currently has the input      *
//* focus TO the directory displayed in the alternate file-display window.       *
//*      -- OR --                                                                *
//* Synchronize the directory contents of the directories in the left and        *
//* right file-display windows.                                                  *
//*                                                                              *
//*          !! Requires that application be in Dual-Window mode. !!             *
//*                                                                              *
//* Input  : synch:  'false' if 'Backup' operation                               *
//*                  'true'  if 'Synch' operation                                *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************

void FileMangler::CmdBackup ( bool synch )
{

   if ( (this->DualwinMode ()) )
      this->Backup ( synch ) ;
   else
   {
      gString gs( " %s operation requires that FileMangler be in Dual-window mode.",
                  synch ? "Synch" : "Backup" ) ;
      dtbmData  msgData( gs.gstr() ) ;
      this->dWin->DisplayTextboxMessage ( dwMsgsTB, msgData ) ;
      this->dWin->UserAlert ( 2 ) ;
   }

}  //* End CmdBackup() *

//*************************
//*      CmdArchive       *
//*************************
//********************************************************************************
//* Create an archive from 'selected' items.                                     *
//*      -- OR --                                                                *
//* Expand an archive to the current directory.                                  *
//*      -- OR --                                                                *
//* Append data to existing archive.                                             *
//*                                                                              *
//* Input  : none                                                                *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************

void FileMangler::CmdArchive ( void )
{
   //* If 'selected' files in inactive window, BUT *
   //* no data on clipboard, clear the selections. *
   if ( ((this->fdPtr->ClipboardFiles ()) == ZERO) && 
         (this->DualwinMode ()) && (this->altPtr->IsSelected ( true )) )
      this->ClearAltFileListSelections () ;

   this->fdPtr->Archive () ;

   //* If in Dual-win Mode and active window is the same        *
   //* directory as inactive window, update the inactive window.*
   this->RefreshAltFileList () ;

   this->RestoreQuickHelpMsg () ;

}  //* End CmdArchive() *

//*************************
//*    CmdLockMenuBar     *
//*************************
//********************************************************************************
//* Controls whether the Menu Bar remains visible when not being accessed by     *
//* user.                                                                        *
//*                                                                              *
//* Input  : lockVisible: if 'true', then make Menu Bar visible (but inactive)   *
//*                       else, make Menu Bar invisible AND inactive             *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************

void FileMangler::CmdLockMenuBar ( bool lockVisible )
{

   this->lockMenuBar = lockVisible ;
   this->DialogTitle ( lockVisible, MAX_DIALOG_CONTROLS ) ;

}  //* End CmdLockMenuBar() *

//**************************
//*      CmdSynchLock      *
//**************************
//********************************************************************************
//* Toggle the 'synchLock' flag. (Dual-window Mode only)                         *
//* When Synch Lock is active display a visual indicator. A navigation key       *
//* detected in the active directory window will be propagated into the          *
//* inactive window.                                                             *
//*                                                                              *
//* Input  : none                                                                *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************
//* Programmer's Note: The directories should have (nearly) identical file       *
//* lists. If not, then synched scrolling has little value.                      *
//* Note also that the current sort criterion may also affect the ability to     *
//* synch. Alpha sorts, and size sort should work fairly well, but sort by       *
//* date could be a problem.                                                     *
//********************************************************************************

void FileMangler::CmdSynchLock ( void )
{
   //* Recognized only in Dual-window Mode *
   if ( (this->DualwinMode ()) )
   {
      winPos slPos( this->ic[dwPathTB].ulY, 
                    short(this->ic[dwPathTB].ulX + this->ic[dwPathTB].cols) ) ;

      if ( ! this->synchLock )
      {
         //* Re-scan both windows and set index on first item *
         this->altPtr->RefreshCurrDir ( false, true ) ;
         this->fdPtr->RefreshCurrDir () ;

         this->synchLock = true ;
         this->dWin->WriteString ( slPos, "SL", this->cfgOpt.cScheme.pf, true ) ;

         //* Bump the highlight to enable "highlight" in inactive window.*
         wkeyCode wk( nckDOWN, wktFUNKEY ) ;
         this->dWin->UngetKeyInput ( wk ) ;
         wk.key = nckUP ;
         this->dWin->UngetKeyInput ( wk ) ;
         this->altPtr->SelectFile ( false, ZERO ) ;
      }

      else
      {
         this->synchLock = false ;
         this->altPtr->DeselectFile ( true ) ;
         this->altPtr->RedrawCurrDir ( false ) ;
         this->dWin->WriteString ( slPos, "  ", this->cfgOpt.cScheme.bb, true ) ;
      }
   }
   else     // call user a dumb-guy
      this->dWin->UserAlert () ;

}  //* End CmdSynchLock() *

//*************************
//*      CmdDirTree       *
//*************************
//********************************************************************************
//* Display a tree view of the directory structure and optionally, allow user    *
//* to select a new working directory.                                           *
//*                                                                              *
//* Input  : none                                                                *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************

void FileMangler::CmdDirTree ( void )
{
   winPos ulCorner( 0, 0 ) ;
   short dlgRows, dlgCols ;
   this->dWin->GetDialogDimensions ( dlgRows, dlgCols ) ;
   if ( (this->fdPtr->DisplayFileTree ( ulCorner, dlgRows, dlgCols )) == false )
      this->RestoreQuickHelpMsg () ;

   //* If alt window (Dual-win Mode) has been *
   //* affected by the scan, refresh it now.  *
   this->RefreshAltFileList () ;

}  //* End CmdDirTree() *

//*************************
//*       CmdAltCWD       *
//*************************
//********************************************************************************
//* For DualWin Mode only, set the alternate (inactive) file-display window's    *
//* working directory (CWD) to be the same as the active window's directory.     *
//*                                                                              *
//* Input  : none                                                                *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************

void FileMangler::CmdAltCWD ( void )
{

   if ( (this->DualwinMode ()) )
      this->RefreshAltFileList ( true, true ) ;
   else
      this->dWin->UserAlert () ;

}  //* End CmdAltCWD() *

//*************************
//*    CmdCompareFiles    *
//*************************
//********************************************************************************
//* Compare two 'selected' files, either one in each window (left and right),    *
//* or two in the active window.                                                 *
//*                                                                              *
//*                                                                              *
//* Input  : prompt  : (optional, 'false' by default)                            *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************
//* FileA and FileB specification:                                               *
//* ------------------------------                                               *
//* Single-window Mode:                                                          *
//*  a) Exactly one file must be 'selected' AND it must not be the               *
//*     highlighted file. The selected file will be compared with the            *
//*     highlighted file.                                                        *
//*  b) All other cases are treated as specification errors.                     *
//*                                                                              *
//* Dual-window Mode:                                                            *
//*  a) If two or more files are 'selected' in the active window, then the       *
//*     selected files will be compared with the corresponding filenames         *
//*     (if any) in the inactive window.                                         *
//*  b) If one file is 'selected' in the active window AND if that file is the   *
//*     highlighted file, then that file will be compared to the corresponding   *
//*     filename (if any) in the inactive window.                                *
//*  c) If one file is 'selected' in active window and if that file IS NOT the   *
//*     highlighted file, then the Single-window rules apply and the inactive    *
//*     window will be ignored.                                                  *
//*  d) If no 'selected' files in active window, then the highlighted file       *
//*     will be compared with the highlighted file in the inactive window.       *
//*  e) All other cases are treated as specification errors.                     *
//*                                                                              *
//********************************************************************************

void FileMangler::CmdCompareFiles ( void )
{
   gString altPath ;       // pathspec for inactive window (group compare)
                           // filespec for inactive window (two-file compare)
   tnFName altStat ;       // stats for file in inactive window (two-file compare)

   if ( (this->DualwinMode ()) )
   {
      this->altPtr->GetPath ( altPath ) ;

      //* Deselect all files in inactive window.      *
      this->ClearAltFileListSelections () ;

      //* 1) If there are multiple 'selected' files in the *
      //*    active window, then launch a group comparison.*
      //* 2) If one 'selected' file in the active window,  *
      //*    then its stats are returned in altStat.
      //* 3) If no selections, then highlighted filename   *
      //*    is the file to be compared.                   *
      if ( !(this->fdPtr->MultiCompare ( altPath )) )
      {
         //* Enable the highlight in inactive window. *
         this->altPtr->RedrawCurrDir () ;

         //* Get filespec and stats for highlighted file in inactive window *
         if ( ((this->altPtr->GetStats ( altStat )) >= ZERO) &&
               (altStat.fType == fmREG_TYPE) )
            altPath.append( "/%s", altStat.fName ) ;
         else        // no files in inactive window
            altPath.clear() ;

         //* Compare the highlighted file in the active  *
         //* window with file capture in inactive window.*
         this->fdPtr->CompareFiles ( altPath, altStat ) ;

         //* Remove highlight from inactive window *
         this->altPtr->RedrawCurrDir ( false ) ;
      }
   }

   //* Else, SinglewinMode, compare indicated files.  *
   else
      this->fdPtr->CompareFiles ( altPath, altStat ) ;

}  //* End CmdCompareFiles() *

//*************************
//*      CmdShellOut      *
//*************************
//********************************************************************************
//* Put the application into hibernation mode and return user to the command     *
//* line with an 'Exit-To-Return' message.                                       *
//*                                                                              *
//* Input  : prompt  : (optional, 'false' by default)                            *
//*                    if 'false' go directly to the shell                       *
//*                    if 'true'  prompt user for an external command to be      *
//*                               executed [CURRENTLY IGNORED]                   *
//*          extCmd  : (optional, NULL pointer by default)                       *
//*                    if specified, instruct the shell to immediately execute   *
//*                    the specified command [CURRENTLY IGNORED]                 *
//*                                                                              *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************
//* Notes:                                                                       *
//*                                                                              *
//* -- Important Note: We must ASSUME that the calling thread is the only        *
//*    thread that is currently awake. It would be embarrassing if a secondary   *
//*    thread were still trying to execute when the application is in            *
//*    hibernation mode.                                                         *
//*                                                                              *
//* -- It is assumed that the user will dink with one or more files at or        *
//*    below the displayed directory tree, and if in Dual-window Mode either     *
//*    or both directory trees. Thus, we force an update on return, which        *
//*    includes clearing the clipboard.                                          *
//*                                                                              *
//* -- 'prompt' parameter:                                                       *
//*     Not yet implemented.                                                     *
//*                                                                              *
//* -- 'extCmd' parameter:                                                       *
//*     Not yet implemented.                                                     *
//*                                                                              *
//* -- A user will occasionally forget that he/she/it is shelled out, but        *
//*    will remember when they try to close the terminal window and get an       *
//*    'active-process' message.                                                 *
//********************************************************************************

void FileMangler::CmdShellOut ( bool prompt, const char* extCmd )
{
   //* Shell out *
   this->dWin->ShellOut () ;

   //* If in Dual-window mode, update the inactive window *
   this->RefreshAltFileList ( true ) ;

   //* Update the active window AND clear the clipboard *
   this->CmdRefresh ( false, true ) ;

}  //* End CmdShellOut() *

//*************************
//*     CmdBrowseCB       *
//*************************
//********************************************************************************
//* Let user scan the current contents of the clipboard and optionally, clear    *
//* data from the clipboard.                                                     *
//*                                                                              *
//*                                                                              *
//* Input  : none                                                                *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************

void FileMangler::CmdBrowseCB ( void )
{
   if ( (this->fdPtr->BrowseClipboard ()) != false )
   {
      //* Clipboard has been cleared, be sure all file *
      //* selections are also cleared                  *
      this->fdPtr->DeselectFile ( true ) ;
      this->ClearAltFileListSelections () ;
      this->RestoreQuickHelpMsg () ;
      this->fdPtr->BrowseClipboard () ;
   }

}  //* End CmdBrowseCB() *

//*************************
//*     CmdUserInfo       *
//*************************
//********************************************************************************
//* Display information about user's account.                                    *
//*                                                                              *
//* Input  : none                                                                *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************

void FileMangler::CmdUserInfo ( void )
{

   this->fdPtr->DisplayUserInfo () ;

}  //* End CmdUserInfo() *

//*************************
//*    CmdFilesysInfo     *
//*************************
//********************************************************************************
//* Display information about the currently-displayed filesystem.                *
//*                                                                              *
//* Input  : none                                                                *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************

void FileMangler::CmdFilesysInfo ( void )
{

   this->fdPtr->DisplayFilesystemInfo () ;

}  //* End CmdFilesysInfo() *

//*************************
//*      CmdExit          *
//*************************
//********************************************************************************
//* Called by user interface method when application exit code is received.      *
//* Determines whether application should exit to the original invocation        *
//* directory OR exit to the current working directory.                          *
//*                                                                              *
//* Input  : none                                                                *
//*                                                                              *
//* Returns: member of enum modeStat{} :                                         *
//*           mstatEXIT  == exit to invocation directory                         *
//*           mstatEXITC == exit to CWD                                          *
//********************************************************************************
//* Programmer's Notes:                                                          *
//* Under Linux/UNIX, it is rather tedious for an application (or a script for   *
//* that matter) to get the parent process to change its CWD (current-working-   *
//* directory. Security-wise, this is an excellent design decision; however,     *
//* FileMangler needs to be able to dump the user in any specified directory     *
//* upon exit.                                                                   *
//*                                                                              *
//* The 'cd' (chdir) command is a shell builtin command to be executed           *
//* directly by the shell process to modify its environment.                     *
//* If the 'cd' command or its equivalent is executed inside an application or   *
//* shell script, it will have no effect on the parent process which lives       *
//* within its own environment. When the application or script terminates, its   *
//* environment goes away and the parent (shell) process regains control.        *
//*                                                                              *
//* Thus, if your application or script wants to exit, dumping the user in a     *
//* directory other than the one from which it was invoked, then the shell       *
//* process must be signalled in some way to indicate that it should modify      *
//* its own CWD.                                                                 *
//*                                                                              *
//* The method we have chosen is to create a simple text file which contains     *
//* the target path string AND which will be used by the invocation script as    *
//* input to the 'cd' command which is (optionally) executed after the           *
//* application exits. Even this task is not as straightforward as it sounds.    *
//*                                                                              *
//* Typically, when the shell executes a script, it creates a new child          *
//* process to run the script and goes to sleep until the child process          *
//* terminates. Thus, any 'cd' command that is executed within the script        *
//* affects only the environment of the child process and NOT the environment    *
//* of the shell process itself.                                                 *
//*                                                                              *
//* This default behavior can be circumvented by invoking the script using the   *
//* 'source' command or the equivalent '. ' prefix. Examples:                    *
//*    foo.sh                 // starts a child process to execute the script    *
//*    source foo.sh          // script is directly executed by shell process    *
//*    . foo.sh               // script is directly executed by shell process    *
//*                                                                              *
//* So far, so good -- however, we cannot expect the user to invoke the          *
//* 'fmg.sh' script with the 'source' command. The user wants to invoke by       *
//* calling the script directly, BUT in that case, a child process is executing  *
//* the script and cannot influence the parent process.                          *
//*                                                                              *
//* Our solution is to have a memory-resident function that the user calls       *
//* to invoke the invocation script. This function can be placed in memory       *
//* on an ad-hoc basis, OR may be defined in the user's '.bashrc' or equivalent  *
//* startup file. The function looks something like this:                        *
//*    fmg() { source ~/bin/fmg.sh ; }                                           *
//*                                                                              *
//* and the invocation script 'fmg.sh' lives in the user's 'bin' directory,      *
//* and looks something like this:                                               *
//*                                                                              *
//*   #!/bin/bash                                                                *
//*   # ****************************************************                     *
//*   # Invoke FileMangler Application                                           *
//*   #                                                                          *
//*   # 1) Create and export environment variables for                           *
//*   #    communication with the target application.                            *
//*   # 2) Invoke the application                                                *
//*   # 3) Check the application's exit code, and if                             *
//*   #    specified change the current working directory.                       *
//*   # ****************************************************                     *
//*                                                                              *
//*   IFS=$'\n'                                                                  *
//*   export FMG_HOME='~/Apps/FileMangler'                                       *
//*   export FMG_WD_FILE="$(mktemp --tmpdir FMG.XXX)"                            *
//*   FMG_ORIG_WD=0                                                              *
//*   FMG_CURR_WD=2                                                              *
//*   $FMG_HOME/FileMangler $1 $2 $3 $4 $5 $6 $7 $8 $9                           *
//*                                                                              *
//*   # Check the application's exit code for exit to alternate directory.       *
//*   if [ $? -eq $FMG_CURR_WD ] ; then      # test return value                 *
//*      if [ -f $FMG_WD_FILE ] ; then       # test existence of temp file       *
//*         cd "$(cat $FMG_WD_FILE)"         # change directory                  *
//*         rm $FMG_WD_FILE                  # delete the temp file              *
//*      fi                                                                      *
//*   fi                                                                         *
//*                                                                              *
//* Of course, not all users will be comfortable editing the '.bashrc' file,     *
//* or FileMangler configuration may be incomplete, or the user may not care     *
//* about being able to exit into the CWD. For this reason, a invocation         *
//* script 'fmg', identical to 'fmg.sh' is also placed in the user's 'bin'       *
//* directory to directly invoke the application. However, in this case, the     *
//* cd-on-exit command in the direct invocation would be ineffective.            *
//*                                                                              *
//********************************************************************************

modeStat FileMangler::CmdExit ( void )
{
   modeStat exitCode = mstatEXIT ;     // return value

   if ( this->exit2cwd != false )
   {
      //* Name of input file (if defined) *
      const char* exitFilepath = getenv ( fmgWDfile ) ;
      if ( exitFilepath != NULL )
      {
         gString gsCurrPath ;
         this->fdPtr->GetPath ( gsCurrPath ) ;
         ofstream ofs( exitFilepath, ofstream::out | ofstream::trunc ) ;
         if ( ofs.is_open() )
         {
            ofs << gsCurrPath.ustr() << endl ;
            ofs.close() ;
         }
         exitCode = mstatEXITC ;
      }
   }
   return exitCode ;

}  //* End CmdExit() *

//*************************
//*  RestoreQuickHelpMsg  *
//*************************
//********************************************************************************
//* Erase any stale message in the status control and restore the Quick-Help     *
//* message.                                                                     *
//* This is accomplished by sending a zero-length message to                     *
//* the DisplayTextboxMessage() method.                                          *
//*                                                                              *
//* Input  : none                                                                *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************

void FileMangler::RestoreQuickHelpMsg ( void )
{
   dtbmData    msgData ;
   this->dWin->DisplayTextboxMessage ( dwMsgsTB, msgData ) ;
   
}  //* End RestoreQuickHelpMsg() *

//*************************
//*        isRoot         *
//*************************
//********************************************************************************
//* Determine whether the CWD is the ROOT directory.                             *
//* Used to disable file commands when user is in the root directory.            *
//*                                                                              *
//* Input  : none                                                                *
//*                                                                              *
//* Returns: 'true'  if CWD is '/', else 'false'                                 *
//********************************************************************************
//* Selection of files and file operations in the root directory ( '/' ):        *
//* ---------------------------------------------------------------------        *
//* Because the directory tree is not fully populated when the root directory    *
//* is displayed, it is not possible to "select" a subdirectory located in       *
//* the root directory. For this reason, if the user tries to select a           *
//* file or subdirectory name (or perform any file operation which makes         *
//* implicit selections), while in the root directory, we will complain.         *
//* (see also RootWarning() method)                                              *
//* Note that while it is technically POSSIBLE to select or operate on a         *
//* non-directory file in the root directory, we can't imagine why you would     *
//* want to, and therefore don't bother to test for it.                          *
//********************************************************************************

bool FileMangler::isRoot ( void )
{
   gString cwd ;                 // string comparison
   bool isroot = false ;         // return value

   this->fdPtr->GetPath ( cwd ) ;
   if ( (cwd.compare( ROOT_PATH )) == ZERO )
      isroot = true ;

   return isroot ;

}  //* End isRoot() *

//*************************
//*      RootWarning      *
//*************************
//********************************************************************************
//* Warn user that file operations are not permitted in the root directory.      *
//*                                                                              *
//* Input  : none                                                                *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************

void FileMangler::RootWarning ( void )
{

   dtbmData  msgData( "  WARNING: File operations in the root "
                      "directory are not supported." ) ;
   this->dWin->DisplayTextboxMessage ( dwMsgsTB, msgData ) ;
   this->dWin->UserAlert ( 2, minUAINTERVAL ) ;

}  //* End RootWarning() *

//*************************
//*       CmdMouse        *
//*************************
//********************************************************************************
//* Enable/disable application mouse support.                                    *
//*                                                                              *
//*                                                                              *
//* Input  : none                                                                *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************

void FileMangler::CmdMouse ( void )
{
   if ( (this->dWin->meMouseEnabled ()) ) // mouse currently enabled
      this->dWin->meDisableMouse () ;
   else                                   // mouse currently disabled
      this->dWin->meEnableStableMouse () ;

   //* Update our local copy of config options and pass it *
   //* to the FileDlg that currently has the input focus.  *
   this->cfgOpt.enableMouse = this->dWin->meMouseEnabled () ;
   this->fdPtr->UpdateMouse ( this->cfgOpt.enableMouse ) ;

   //* Display status message *
   dtbmData    msgData ;               // message display
   gString gs( " Mouse Support %s. ",
               (this->cfgOpt.enableMouse ? "Enabled" : "Disabled") ) ;
   gs.copy( msgData.textData, gsALLOCDFLT ) ;
   this->dWin->DisplayTextboxMessage ( dwMsgsTB, msgData ) ;

}  //* CmdMouse() *

//*************************
//*     CmdHelpAbout      *
//*************************
//********************************************************************************
//* Display the HelpAbout dialog at the appropriate position within the          *
//* application window.                                                          *
//*                                                                              *
//* Input  : none                                                                *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************

void FileMangler::CmdHelpAbout ( void )
{
   winPos   wPos( this->actualRows / 2, this->actualCols / 2 ) ;

   this->HelpAbout ( wPos.ypos, wPos.xpos ) ;

}  //* End CmdHelpAbout() *

//*************************
//*     CmdCallHelp       *
//*************************
//********************************************************************************
//* Display the FileMangler 'info' Help document.                                *
//*                                                                              *
//* Input  : html  : (optional, 'false' by default)                              *
//*                  if 'false', invoke the info reader documentation            *
//*                  if 'true',  open the HTML-format documentation in the       *
//*                              default browser                                 *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************

void FileMangler::CmdCallHelp ( bool html )
{

   if ( html != false )
   {
      gString htmlDocs( "%s/%s", this->cfgOpt.appPath, HTML_Filename ) ;
      fmFType fType ;
      bool goodLaunch = false ;
      if ( (this->fdPtr->TargetExists ( htmlDocs, fType )) )
      {
         if ( (this->fdPtr->LaunchDefaultApplication ( htmlDocs.ustr() )) >= ZERO )
            goodLaunch = true ;
      }
      if ( ! goodLaunch )
      {
         dtbmData  msgData( "  HTML docs not found!  " ) ;
         this->dWin->DisplayTextboxMessage ( dwMsgsTB, msgData ) ;
      }
   }
   else
      this->fdPtr->CallContextHelp ( cxtMAIN ) ;

}  //* End CmdCallHelp() *

//*************************
//*    Display_SU_Flag    *
//*************************
//********************************************************************************
//* If user is 'superuser', then indicate it with a prominent message.           *
//*                                                                              *
//* Input  : none                                                                *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************

void FileMangler::Display_SU_Flag ( void )
{
   UserInfo ui ;
   this->fdPtr->GetUserInfo ( ui ) ;
   if ( ui.userID == superUID )
      this->dWin->WriteString ( this->ic[dwMsgsTB].ulY + 1, 1,
                                "[SUPERUSER]", this->cfgOpt.cScheme.pf ) ;

}  //* End Display_SU_Flag() *

//*************************
//*     Prompt4Config     *
//*************************
//********************************************************************************
//* Called if Trashcan is missing or improperly configured.                      *
//* Ask user whether to run the configuration utility.                           *
//*                                                                              *
//* Input  : none                                                                *
//*                                                                              *
//* Returns: 'true'  if user wants to run the configuration utility              *
//*          'false' otherwise                                                   *
//********************************************************************************

bool FileMangler::Prompt4Config ( void )
{
   const char* dlgMsg[] = 
   { //123456789x123456789x123456789x123456789x123456789x12
      "  ALERT!!  ALERT!!  ",
      "       Application is improperly configured.       ",
      "   Default Desktop Trashcan was not found AND      ",
      "   no alternate Trashcan location was specified.   ",
      "   Move-To-Trash operations have been disabled.    ",
      " ",
      "   Do you want to run the configuration utility    ",
      "   to complete repair?",
      NULL
   } ;
   attr_t msgColor[] = 
   {
    this->cfgOpt.cScheme.sb,           // title color attribute
    this->cfgOpt.cScheme.pf,           // Warning message
    this->cfgOpt.cScheme.em,           // message
    this->cfgOpt.cScheme.em,           // message
    this->cfgOpt.cScheme.em,           // message
    this->cfgOpt.cScheme.sd,           // blank line
    this->cfgOpt.cScheme.em,           // message
    this->cfgOpt.cScheme.em,           // message
    this->cfgOpt.cScheme.sd            // null string (to be safe)
   } ;

   bool configure = this->fdPtr->DecisionDialog ( dlgMsg, msgColor ) ;

   if ( ! configure )
   {  //* Warn that Trashcan operations are disabled *
      dlgMsg[0] = "   CAUTION   " ;
      dlgMsg[1] = " " ;
      msgColor[1] = this->cfgOpt.cScheme.sd ;
      dlgMsg[2] = "FileMangler will run without a Trashcan, but for   " ;
      dlgMsg[3] = "full functionality, please complete configuration. " ;
      dlgMsg[4] = NULL ;
      this->fdPtr->InfoDialog ( dlgMsg, msgColor ) ;
   }
   return configure ;

}  //* End Prompt4Config() *

//*************************
//*   FmNotImplemented    *
//*************************
//********************************************************************************
//* For the many (but decreasing) number of things that are not yet              *
//* implemented, display a message.                                              *
//*                                                                              *
//* Input  : func  : (optional, NULL pointer by default)                         *
//*                  If specified, substitute caller's functionality message     *
//*                  for the default message.                                    *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************

void FileMangler::FmNotImplemented ( const char* func )
{
   const wchar_t* dfltFunc = L"requested operation" ;
   gString gs( "  Sorry, %S is not yet implemented.  ", dfltFunc ) ;
   if ( func != NULL )
   {
      gString gt( "'%s'", func ) ;
      gs.replace( dfltFunc, gt.gstr() ) ;
   }
   dtbmData msg = gs.ustr() ;
   this->dWin->DisplayTextboxMessage ( dwMsgsTB, msg ) ;
   this->dWin->UserAlert () ;
   chrono::duration<short>aWhile( 3 ) ;
   this_thread::sleep_for( aWhile ) ;
   this->RestoreQuickHelpMsg () ;

}  //* End FmNotImplemented() *

//*************************
//*      FmDebugMsg       *
//*************************
//********************************************************************************
//* Display a debugging message in the bottom line of the terminal window.       *
//* Note: This is a one-line dialog. Display for main dialog is preserved.       *
//*                                                                              *
//* Input  : msg  : pointer to buffer containing message                         *
//*          pause (optional, two (2) by default)                                *
//*            if 1 <= pause < 10 then sleep for that number of seconds          *
//*            if pause >= 10 then wait for a keypress                           *
//* Returns: nothing                                                             *
//********************************************************************************

void FileMangler::FmDebugMsg ( const char msg[], short pause )
{
   //* Save the display for the parent dialog window *
   this->dWin->SetDialogObscured () ;

   //* Initial parameters for dialog window *
   InitNcDialog dInit( 1,              // number of display lines
                       (this->actualCols-4), // number of display columns
                       (this->termRows-1),   // Y offset from upper-left of terminal 
                       2,              // X offset from upper-left of terminal 
                       NULL,           // dialog title
                       ncltSINGLE,     // border line-style
                       this->cfgOpt.cScheme.tn, // border color attribute
                       this->cfgOpt.cScheme.tn, // interior color attribute
                       NULL            // pointer to list of control definitions
                     ) ;

   //* Instantiate the dialog window *
   NcDialog* dp = new NcDialog ( dInit ) ;

   //* Open the dialog window *
   if ( (dp->OpenWindow()) == OK )
   {
      dp->WriteString ( ZERO, 1, msg, nc.re, true ) ;
      if ( pause < 1 )
         pause = 2 ;
      if ( pause < 10  )
      {
         chrono::duration<short>aWhile( pause ) ;
         this_thread::sleep_for( aWhile ) ;
      }
      else
         nckPause() ;
   }
   if ( dp != NULL )          // close the dialog window
      delete ( dp ) ;

   //* Restore display of parent dialog window *
   this->dWin->RefreshWin () ;
   
}  //* End FmDebugMsg() *

//*************************
//*                       *
//*************************
//********************************************************************************
//*                                                                              *
//*                                                                              *
//*                                                                              *
//* Input  :                                                                     *
//*                                                                              *
//* Returns:                                                                     *
//********************************************************************************

