//********************************************************************************
//* File       : FmConfig.hpp                                                    *
//* Author     : Mahlon R. Smith                                                 *
//*              Copyright (c) 2005-2025 Mahlon R. Smith, The Software Samurai   *
//*                  GNU GPL copyright notice located in FileMangler.hpp         *
//* Date       : 26-Nov-2020                                                     *
//* Version    : (see AppVersion string)                                         *
//*                                                                              *
//* Description: Class definition and miscellaneous constant definitions         *
//*              for the FmConfig class and FmConfig execution thread            *
//*              supporting the FileMangler application.                         *
//*                                                                              *
//********************************************************************************
//* Version History (most recent first):                                         *
//*                 See version history in FileMangler.cpp.                      *
//*                                                                              *
//********************************************************************************

//* Ensure that the critical code is reentrant *
#define _REENTRANT

//****************
//* Header Files *
//****************
#include "GlobalDef.hpp"      //* NcDialog family of classes
#include "FileMangler.hpp"    //* For various constant definitions
#include "FileDlg.hpp"        //* For ConfigOptions class definition and const strings only
#include <sys/shm.h>          //* Support for shared-memory functionality
#include <wordexp.h>          //* For environment-variable substitution
#include <sys/wait.h>         //* For waitpid() prototype
#include <fstab.h>            //* For accessing 'fstab' and 'mtab' files
#include <mntent.h>           //* For accessing 'fstab' and 'mtab' files


//*****************
//*  Definitions  *
//*****************
//* Command options for calling the FileMangler configuration thread *
enum ConfigCommand :short
{
   ccREAD = 1,       //* read the specified configuration file
   ccREADSILENT,     //* same as ccREAD, but no diagnostic message written
   ccINTERACT,       //* interact with user to set/modify configuration options
   ccKEYMAP,         //* create a new keymap configuration file with default values
   ccCREATE,         //* create a new configuration file with default values
   ccCREATEp,        //* create a new configuration file with specified values
   ccWRITE,          //* write specified data to to specified configuration file
                     //* (if target does not exist, create it)
   ccOPTIONS         //* end-of-command-options
} ;

//* Status returned from the FileMangler configuration thread *
enum ConfigStatus : short 
{
   csOK = OK,        //* operation successful
   csARGS = 1,       //* FmConfig invoked with invalid arguments
   csACCESS,         //* unable to create/access shared-memory segment
   csBADFORK,        //* unable to create child process (unlikely)
   csNOTFOUND,       //* specified configuration file not found
   csALTRASH,        //* specified alternate path for trashcan is invalid
   csTMPATH,         //* specified path for temp storage is invalid
   csSYNTAX,         //* syntax error encounter while reading the config file
   csREADONLY,       //* target file or target directory is write-protected
   csNOCURSES,       //* unable to start the NCurses engine for interactive dialogs
   csERR,            //* general error
} ;

//* This is a layout of the shared-memory area used to communicate between *
//* the parent process (FileMangler) and the child process (FmConfig).     *
const short smmMSG_COUNT = 64 ;
const short smmMSG_LEN = 128 ;
class sharedMemMap
{
   public:
   char  appPath[MAX_PATH] ;     // path of parent-application executable
   char  cfgPath[MAX_PATH] ;     // path/filename of configuration file
   char  trashPath[MAX_PATH] ;   // alternate path for desktop's trashcan
   char  favPath[MAX_FAVDIRS][MAX_PATH] ; // 'favorite dir' paths
   short favCount ;
   char  mntCmd[MAX_MNTCMDS][MAX_PATH] ;  // 'mount' commands or paths
   short mntCount ;
   bool  lockMb ;                // if 'true', lock Menu Bar in visible state
   RemapKey keyMap ;             // command-key map
   char  msg[smmMSG_COUNT][smmMSG_LEN] ;// diagnostic messages from child process
   short msgCount ;              // number of diagnostic messages
   ConfigCommand  ccCmd ;        // command option (operation to be performed)
   ConfigStatus   fcStatus ;     // status of FmConfig operation performed
   bool  verbose ;               // if 'true', return extended diagnostics
   ConfigOptions  cfgOpt ;       // configuration options read/written
} ;

//* Comparison Strings *
// Usage Note: Not all members are used in all circumstances.
//       'str' used for each instance
//       'len' used for each instance (needed for comparisons)
//    'argcnt' used only in CfgParm[] : indicates number of args in attached array
//      'dflt' used only in CfgParm[] : indicates default member in attached array
//      'args' used only in CfgParm[] : attach point for argument array
class CfgComp
{
   public:
   const wchar_t* str ;          // pointer to command string
   short len ;                   // command-string length (not incl. NULLCHAR)
   short argcnt ;                // number of valid arguments for command
   short dflt ;                  // 'default' member of the argument array
   const CfgComp* args ;         // pointer to linked-list of arg descriptions
} ;

//* Names for accessing members of CfgParm[] array *
enum cciArguments : short 
{ 
   cciWM, cciSO, cciCS, cciHF, cciCD, cciCO, cciLM,
   cciSC, cciEM, cciEP, /*cciTL, */cciEC, cciArgs 
} ;

//* Names for accessing members of CfgList[] array *
enum cliArguments : short 
{ cliFD = ZERO, cliKM, cliMC, cliAT, cliEF, cliArgs } ;

const wchar_t SLASH = L'/' ;
const wchar_t EQUAL = L'=' ;
const wchar_t DOLLAR = L'$' ;


//*****************
//* Constant Data *
//*****************

//* Base name for interactive configuration dialog *
const char* const CfgTitle = "  FileMangler Configuration - " ;

//* This lives at the top of every configuration *
//* file and identifies it a 'genuine'.          *
const char* const ConfigFileID = "FMCFG" ;
const char* const ConfigFileVersion = "v:0.01.00" ;
const char* const KeymapFileID = "FMKEY" ;
const char* const KeymapFileVersion = "v:0.01.00" ;
const char* const ConfigDfltFavorite = "${HOME}" ;
//* Default location for FileMangler installation *
//*   (and base directory for temporarary files). *
const char* const ConfigDfltInstallPath = "~/Apps/FileMangler" ;
//* Default location of desktop environment's trashcan. *
const char* const ConfigDfltTrashcanPath = "~/.local/share/Trash" ;
const char* const cdtpFiles = "files" ;
const char* const cdtpInfo  = "info" ;

const CfgComp wmArgs[] =               // WindowMode arguments
{
   { L"SingleWin",   9, 0, 0, NULL }, 
   { L"DualWin",     7, 0, 0, NULL }, 
   { L"ByTermSize", 10, 0, 0, NULL }   // (default)
} ;
const CfgComp soArgs[] =               // SortOrder arguments
{
   { L"None",     4, 0, 0, NULL },
   { L"NameR",    5, 0, 0, NULL }, 
   { L"Name",     4, 0, 0, NULL },     // (default)
   { L"DateR",    5, 0, 0, NULL },
   { L"Date",     4, 0, 0, NULL }, 
   { L"SizeR",    5, 0, 0, NULL },
   { L"Size",     4, 0, 0, NULL },
   { L"TypeR",    5, 0, 0, NULL },
   { L"Type",     4, 0, 0, NULL },
   { L"ExtR",     4, 0, 0, NULL },
   { L"Ext",      3, 0, 0, NULL },
} ;
const CfgComp roArgs[] =               // ReverseSort arguments
{
   { L"true",   4, 0, 0, NULL },
   { L"false",  5, 0, 0, NULL }        // (default)
} ;
const CfgComp csArgs[] =               // CaseSensitive arguments
{
   { L"true",   4, 0, 0, NULL },       // (default)
   { L"false",  5, 0, 0, NULL } 
} ;
const CfgComp hfArgs[] =               // HiddenFiles arguments
{
   { L"true",   4, 0, 0, NULL }, 
   { L"false",  5, 0, 0, NULL }        // (default)
} ;
const CfgComp cdArgs[] =               // ConfirmDelete arguments
{
   { L"true",   4, 0, 0, NULL }, 
   { L"false",  5, 0, 0, NULL }        // (default)
} ;
const CfgComp coArgs[] =               // ConfirmOverwrite arguments
{
   { L"Never",         5, 0, 0, NULL }, 
   { L"ConfirmNewer", 12, 0, 0, NULL },// (default)
   { L"Always",        6, 0, 0, NULL }, 
} ;                          
const CfgComp tnArgs[] =               // TrashcanNotify arguments
{
   { L"true",   4, 0, 0, NULL }, 
   { L"false",  5, 0, 0, NULL }        // (default)
} ;                   
const CfgComp lmArgs[] =               // LockMenuBar arguments
{
   { L"true",   4, 0, 0, NULL }, 
   { L"false",  5, 0, 0, NULL }        // (default)
} ;
const CfgComp scArgs[] =               // colorSCheme arguments
{
   { L"Default",   7, 0, 0, NULL },    // (default)
   { L"Black",     5, 0, 0, NULL },
   { L"Red",       3, 0, 0, NULL },
   { L"Green",     5, 0, 0, NULL },
   { L"Brown",     5, 0, 0, NULL },
   { L"Blue",      4, 0, 0, NULL },
   { L"Magenta",   7, 0, 0, NULL },
   { L"Cyan",      4, 0, 0, NULL },
   { L"Grey",      4, 0, 0, NULL },
   { L"Terminal",  8, 0, 0, NULL },
} ;
const CfgComp emArgs[] =               // EnableMouse arguments
{
   { L"enable",  6, 0, 0, NULL }, 
   { L"disable", 7, 0, 0, NULL }       // (default)
} ;
const CfgComp epArgs[] =               // ExternalPrograms arguments
{
   { L"AutoLaunch",   10, 0, 0, NULL }, // (default)
   { L"AutoAudio",     9, 0, 0, NULL },
   { L"SafeLaunch",   10, 0, 0, NULL },
   { L"ManualLaunch", 12, 0, 0, NULL },
} ;
const CfgComp ecArgs[] =               // EndCfgCommands arguments
{
   { L"End",      3, 0, 0, NULL }
} ;
const CfgComp CfgParm[cciArgs] =       // Configuration Parameters
{
   { L"WindowMode",       10,  3, 2, wmArgs },
   { L"SortOrder",         9, 11, 2, soArgs },
   { L"CaseSensitive",    13,  2, 0, csArgs },
   { L"HiddenFiles",      11,  2, 1, hfArgs },
   { L"ConfirmDelete",    13,  2, 1, cdArgs },
   { L"ConfirmOverwrite", 16,  3, 1, coArgs },
   { L"LockMenuBar",      11,  2, 1, lmArgs },
   { L"ColorScheme",      11, 10, 0, scArgs },
   { L"EnableMouse",      11,  2, 1, emArgs },
   { L"ExternalPrograms", 16,  4, 0, epArgs },
   { L"EndCfgCommands",   14,  1, 0, ecArgs },
} ;                               

//* Dynamic configuration data *
const CfgComp CfgList[cliArgs] =       // Configuration list names
{
   { L"Favorite Directories:", 21, 0, 0, NULL },
   { L"KeyMap",                 6, 0, 0, NULL },
   { L"Mount Commands:",       15, 0, 0, NULL },
   { L"Alternate Trashcan:",   19, 0, 0, NULL },
   { L"EndConfigFile",         13, 0, 0, NULL },
} ;                                   


//* Key-bindings template *
const char* const kbColumnHead = 
   "____KEY____|_ DEFAULT__|_____DESCRIPTION________________________________" ;
const wchar_t* const kbTemplate = L"   %-9s%s" ;
const short kbDataItems = 152 ;
const char* const kbDescription[kbDataItems] = 
{
   " | nckC_A    | select all / deselect all files in window",
   " | nckC_B    | back up current directory tree (Dual-window mode only)",
   " | nckC_C    | copy selected files (place on clipboard)",
   " | nckC_D    | unassigned",
   " | nckCDEL   | move selected files to trash",
   " | nckSDEL   | delete selected files (permanent - not to trash)",
   " | nckCSDEL  | empty the trashcan",
   " | nckC_E    | enable write access for selected files",
   " | nckC_F    | find file(s)",
   " | nckC_G    | unassigned",
   " | nckC_H    | unassigned",
   " | nckTAB    | move to alternate window (Dual-window mode only)",
   " | nckSTAB   | move to alternate window (Dual-window mode only)",
   " | nckC_ENTER| 'viewfile' context menu OR navigate to subdirectory",
   " | nckC_K    | unassigned",
   " | nckC_L    | unassigned",
   " | nckC_N    | new subdirectory (create directory)",
   " | nckC_O    | open 'Favorites' dialog",
   " | nckC_P    | paste: open 'paste special' option dialog",
   " | nckC_Q    | exit the application (to original working directory)",
   " | nckC_R    | rename selected files",
   " | nckC_S    | sort option context menu",
   " | nckC_T    | touch selected files (adjust date/time)",
   " | nckC_U    | update the list of displayed files",
   " | nckC_V    | paste files from clipboard to current directory",
   " | nckC_W    | write protect selected files",
   " | nckC_X    | cut selected files (place on clipboard)",
   " | nckC_Y    | mount a file system",
   " | nckC_Z    | undo most recent 'move-to-trashcan' operation",
   " | nckESC    | (meaning is context dependent)",
   " | nckSPACE  | select/deselect the highlighted file",
   " | nckBKSP   | navigate to parent directory",
   " | nckAUP    | navigate to parent directory",
   " | nckF01    | (may not be available to application)",
   " | nckF02    | open the main menu",
   " | nckF03    | ",
   " | nckF04    | ",
   " | nckF05    | ",
   " | nckF06    | ",
   " | nckF07    | ",
   " | nckF08    | ",
   " | nckF09    | ",
   " | nckF10    | (may not be available to application)",
   " | nckF11    | (may not be available to application)",
   " | nckF12    | ",
   " | nckSF01   | FileMangler help (Texinfo help file)",
   " | nckSF02   | unassigned",
   " | nckSF03   | unassigned",
   " | nckSF04   | unassigned",
   " | nckSF05   | unassigned",
   " | nckSF06   | unassigned",
   " | nckSF07   | unassigned",
   " | nckSF08   | unassigned",
   " | nckSF09   | unassigned",
   " | nckSF10   | (may not be available to application)",
   " | nckSF11   | unassigned",
   " | nckSF12   | unassigned",
   " | nckCF01   | unassigned",
   " | nckCF02   | unassigned",
   " | nckCF03   | unassigned",
   " | nckCF04   | unassigned",
   " | nckCF05   | unassigned",
   " | nckCF06   | unassigned",
   " | nckCF07   | unassigned",
   " | nckCF08   | unassigned",
   " | nckCF09   | unassigned",
   " | nckCF10   | unassigned",
   " | nckCF11   | unassigned",
   " | nckCF12   | unassigned",
   " | nckCSF01  | unassigned",
   " | nckCSF02  | unassigned",
   " | nckCSF03  | unassigned",
   " | nckCSF04  | unassigned",
   " | nckCSF05  | unassigned",
   " | nckCSF06  | unassigned",
   " | nckCSF07  | unassigned",
   " | nckCSF08  | unassigned",
   " | nckCSF09  | unassigned",
   " | nckCSF10  | unassigned",
   " | nckCSF11  | unassigned",
   " | nckCSF12  | unassigned",
   // Numeric Keypad Group
   " | nckpCENTER| unassigned",
   " | nckpPLUS  | unassigned",
   " | nckpMINUS | unassigned",
   " | nckpMULT  | unassigned",
   " | nckpDIV   | unassigned",
   " | nckpHOME  | unassigned",
   " | nckpEND   | unassigned",
   " | nckpENTER | unassigned",
   " | nckpAPGDN | unassigned",
   " | nckpAPGUP | unassigned",
   " | nckpCPGDN | unassigned",
   " | nckpCPGUP | unassigned",
   " | nckpACUP  | unassigned",
   " | nckpACDOWN| unassigned",
   " | nckpACRIGH| unassigned",
   " | nckpACLEFT| unassigned",
   " | nckpACPGDN| unassigned",
   " | nckpACPGUP| unassigned",
   " | nckpACDEL | unassigned",
   //* NcDialog Extensions *
   " | nckA_A    | Alt+a unassigned",
   " | nckA_B    | Alt+b unassigned",
   " | nckA_C    | Alt+c unassigned",
   " | nckA_D    | Alt+d unassigned",
   " | nckA_E    | Alt+e unassigned",
   " | nckA_F    | Alt+f unassigned",
   " | nckA_G    | Alt+g unassigned",
   " | nckA_H    | Alt+h unassigned",
   " | nckA_I    | Alt+i unassigned",
   " | nckA_J    | Alt+j unassigned",
   " | nckA_K    | Alt+k unassigned",
   " | nckA_L    | Alt+l unassigned",
   " | nckA_M    | Alt+m unassigned",
   " | nckA_N    | Alt+n unassigned",
   " | nckA_O    | Alt+o unassigned",
   " | nckA_P    | Alt+p unassigned",
   " | nckA_Q    | exit the application (to current working directory)",
   " | nckA_R    | Alt+r unassigned",
   " | nckA_S    | Alt+s unassigned",
   " | nckA_T    | Alt+t unassigned",
   " | nckA_U    | Alt+u unassigned",
   " | nckA_V    | Alt+v unassigned",
   " | nckA_W    | Alt+w toggle Single/Dual-window mode",
   " | nckA_X    | Alt+x unassigned",
   " | nckA_Y    | Alt+y unassigned",
   " | nckA_Z    | Alt+z unassigned",
   " | nckAS_A   | Alt+Shift+a unassigned",
   " | nckAS_B   | Alt+Shift+b backup files",
   " | nckAS_C   | Alt+Shift+c unassigned",
   " | nckAS_D   | Alt+Shift+d unassigned",
   " | nckAS_E   | Alt+Shift+e unassigned",
   " | nckAS_F   | Alt+Shift+f unassigned",
   " | nckAS_G   | Alt+Shift+g unassigned",
   " | nckAS_H   | Alt+Shift+h unassigned",
   " | nckAS_I   | Alt+Shift+i unassigned",
   " | nckAS_J   | Alt+Shift+j unassigned",
   " | nckAS_K   | Alt+Shift+k reserved for development",
   " | nckAS_L   | Alt+Shift+l unassigned",
   " | nckAS_M   | Alt+Shift+m unassigned",
   " | nckAS_N   | Alt+Shift+n unassigned",
   " | nckAS_O   | Alt+Shift+o unassigned",
   " | nckAS_P   | Alt+Shift+p unassigned",
   " | nckAS_Q   | Alt+Shift+q unassigned",
   " | nckAS_R   | Alt+Shift+r unassigned",
   " | nckAS_S   | Alt+Shift+s synchronize files",
   " | nckAS_T   | Alt+Shift+t view directory tree",
   " | nckAS_U   | Alt+Shift+u unassigned",
   " | nckAS_V   | Alt+Shift+v unassigned",
   " | nckAS_W   | Alt+Shift+w Set alt directory window to CWD",
   " | nckAS_X   | Alt+Shift+x unassigned",
   " | nckAS_Y   | Alt+Shift+y unassigned",
   " | nckAS_Z   | Alt+Shift+z shell out: go to command prompt ('exit' to return)",
} ;                

//* For temporary-storage directories, these are the permission *
//* bits that must be set to ensure user read/write access.     *
const mode_t dirMode = (S_IRWXU | S_IRWXG | S_IROTH | S_IXOTH) ;


//********************
//* Class Definition *
//********************
class FmConfig
{
   public:

   virtual ~FmConfig () ;  // destructor

   //* Constructor *
   FmConfig ( sharedMemMap* shmPtr ) ;

   //* Returns status of operation *
   ConfigStatus GetStatus ( void ) { return this->csStatus ; }

   private:
   //****************
   //*   Methods    *
   //****************
   //* Open specified configuration file and capture its content               *
   ConfigStatus ReadConfigFile ( ConfigCommand readType ) ;
   //* Read the section of configuration file containing 'Favorite' directories*
   ConfigStatus rcfLoadFavorites ( ifstream& ifs, short& lineNum ) ;
   //* Read the command-key mapping file.                                      *
   ConfigStatus rcfLoadKeymap ( const gString& kmCmd ) ;
   //* Scan the keycode map for conflicts or duplications.                     *
   ConfigStatus rcfValidateKeymap ( short& dup1, short&dup2 ) ;
   //* Read the section of configuration file containing 'Mount Targets'       *
   ConfigStatus rcfLoadMounts ( ifstream& ifs, short& lineNum ) ;
   //* Determine the location and configuration of the trashcan sub-directory  *
   ConfigStatus rcfLoadAltTrash ( ifstream& ifs, short& lineNum ) ;
   bool  rcfValidateTrashPath ( gString& tcPath ) ;
   bool  vtpTargetDirExists ( const gString& tPath, bool& rdAcc, bool& wrAcc ) ;
   bool  vtpTargetExists ( const gString& tPath, bool& rdAcc, bool& wrAcc ) ;
   bool  vtpCreateDir ( const gString& tPath, bool& rdAcc, bool& wrAcc ) ;
   bool  vtpEnablePermissions ( const gString& tPath, bool& rdAcc, bool& wrAcc ) ;
   //* Read a line of data from source file                                    *
   short rcfReadSourceLine ( ifstream& ifs, short& lineNum, gString& gs ) ;
   //* Perform environment-variable expansion on a path (or other) string      *
   bool  rcfEnvExpansion ( gString& gsPath, bool force = false ) ;
   //* Create a unique path/filename for a temporary file.                     *
   bool  CreateTempname ( gString& tmpPath ) ;

   //* Write configuration data to the target file                             *
   ConfigStatus WriteConfigFile ( void ) ;
   ConfigStatus wcfWriteFavorites ( ofstream& ofs, ifstream& ifs, short& lineNum ) ;
   ConfigStatus wcfWriteFavorites ( ofstream& ofs ) ;
   ConfigStatus wcfWriteKeyMap ( ofstream& ofs, gString& gs ) ;
   ConfigStatus wcfWriteMounts ( ofstream& ofs, ifstream& ifs, short& lineNum ) ;
   ConfigStatus wcfWriteMounts ( ofstream& ofs ) ;
   ConfigStatus wcfWriteAltTrash ( ofstream& ofs, ifstream& ifs, short& lineNum ) ;
   ConfigStatus wcfWriteAltTrash ( ofstream& ofs ) ;
   void  wcfWriteTrailingData ( ofstream& ofs, ifstream& ifs, short& lineNum ) ;

   //* Create a configuration with all options set to defaults                 *
   ConfigStatus CreateConfigFile ( bool populate = false ) ;
   ConfigStatus CreateKeymapFile ( bool populate = false, const char* trgFile = NULL ) ;

   //* Interactively view/edit all configuration options                       *
   ConfigStatus EditConfigOptions ( ConfigCommand cfgCmd ) ;
   bool  ecoEditParms ( void ) ;
   bool  ecoEditWM ( InitNcDialog& dInit ) ;
   bool  ecoEditHF ( InitNcDialog& dInit ) ;
   bool  ecoEditCD ( InitNcDialog& dInit ) ;
   bool  ecoEditCO ( InitNcDialog& dInit ) ;
   bool  ecoEditSO ( InitNcDialog& dInit ) ;
   bool  ecoEditLM ( InitNcDialog& dInit ) ;
   bool  ecoEditTS ( InitNcDialog& dInit ) ;
   bool  ecoEditCS ( InitNcDialog& dInit ) ;
   bool  ecoEditEM ( InitNcDialog& dInit ) ;
   bool  ecoEditEP ( InitNcDialog& dInit ) ;
   bool  ecoEditFD ( InitNcDialog& dInit ) ;
   bool  ecoEditKM ( InitNcDialog& dInit ) ;
   bool  ecoEditMC ( InitNcDialog& dInit ) ;
   bool  ecoSelectFilesys ( InitNcDialog& dInit, gString& gsMnt ) ;
   short ecoScan4Filesystems ( fsInfo*& fsiPtr ) ;
   short ecoUSB_DeviceStats ( usbDevice*& usbdPtr ) ;
   short ecoDVD_DeviceStats ( usbDevice*& usbdPtr ) ;
   bool  ecoEditScrollextData ( NcDialog* dp, uiInfo& Info, ssetData& sData, 
                                gString* fgs, short dspSE, short entTB, 
                                short maxEnt, winPos&pptPos, winPos& msgPos ) ;
   bool  ecoBinaryDecision ( InitNcDialog& dInit, const char* tmsg, 
                             const char* fmsg, bool currSel, bool dfltSel ) ;
   void  ecoDisplayCurrent ( const winPos& wp ) ;
   void  ecoPathFit ( gString& gsPath, short colsAvail ) ;
   void  ConfigHelp ( const winPos& wpStat ) ;

   //* Start or shut down the NCurses Engine used for creating dialog windows. *
   short StartNCursesEngine ( void ) ;
   void  StopNCursesEngine () ;


   //****************
   //* Data Members *
   //****************
   sharedMemMap*  shm ;       //* Pointer to shared memory space
   //* Color attributes for dialog window *
   attr_t dColor,                // dialog background color
         expColor,               // explanation text
         pbnColor,               // pushbutton (non-focus)
         pbfColor,               // pushbutton (focus)
         rbnColor,               // radio button (non-focus)
         rbfColor,               // radio button (focus)
         tbnColor,               // text box (non-focus)
         tbfColor,               // text box (focus)
         spnColor,               // spinner (non-focus)
         spfColor ;              // spinner (focus)
   short termRows,            //* Size of terminal window
         termCols,
         dialogRows,          //* Size of dialog window
         dialogCols,
         dialogPosY,          //* Position (offset) of dialog window
         dialogPosX, 
         icCount ;            //* Number of dialog controls defined
   InitCtrl* ic ;             //* Pointer to list of controls in dialog window
   NcDialog* dPtr ;           //* Pointer to open dialog window object
   ConfigStatus   csStatus ;  //* Status of operation returned to parent process
} ;


