//********************************************************************************
//* File       : FmConfig.cpp                                                    *
//* Author     : Mahlon R. Smith                                                 *
//*              Copyright (c) 2005-2025 Mahlon R. Smith, The Software Samurai   *
//*                  GNU GPL copyright notice located in FileMangler.hpp         *
//* Date       : 19-Apr-2025                                                     *
//* Version    : (see FileMangler AppVersion string)                             *
//*                                                                              *
//* Description: This is the configuration sub-program for the FileMangler       *
//*              application. It is invoked only from within FileMangler, and    *
//*              at this time, is not intended to be run as a stand-alone        *
//*              application.                                                    *
//*                NOTE: In a future release, we hope to add a stand-alone       *
//*                      start-up sequence.                                      *
//*                                                                              *
//* The application relies upon a single argument which is a shared-memory key   *
//* for accessing a shared-memory space created by the parent process            *
//*                                                                              *
//* This application performs the following activities:                          *
//* 1) Invoked by FileMangler during startup to read the 'FileMangler.cfg'       *
//*    configruation file (or other specified configuration file) to establish   *
//*    user's start-up preferences.                                              *
//* 2) Invoked by the FileMangler user to interactively set/modify               *
//*    configuration parameters, which are then stored in 'FileMangler.cfg' or   *
//*    another specified configuration file.                                     *
//* 3) Invoked by the FileMangler application to display/modify key bindings.    *
//* 4) Invoked by the FileMangler application or by the FileMangler user to      *
//*    interactively create a new configuration file.                            *
//*                                                                              *
//* Developed using GNU G++ (Gcc v: 4.4.2)                                       *
//*  under Fedora Release 12, Kernel Linux 2.6.31.5-127.fc12.i686.PAE            *
//********************************************************************************
//* Programmer's Note:                                                           *
//* This module uses 'wide' (wchar_t) comparisons in reading the configuration   *
//* file. Although this may be inefficient, it allows for much easier            *
//* internationalization of the configuration file for future releases.          *
//*                                                                              *
//* Adding New Configuration Options:                                            *
//* When adding a new configuration option, there are a number of sections to    *
//* be updated, depending upon whether the new option is a parameterized         *
//* option or an open-form option.                                               *
//* A. Parameterized Options                                                     *
//*    1. Add a new CfgComp xxArgs array to indicate the valid arguments.        *
//*       (If it is a numeric argument, this step is unnecessary.)               *
//*    2. Add a member to CfgComp CfgParm[] at the appropriate position.         *
//*       This must be somewhere ABOVE the 'EndCfgCommands' entry.               *
//*    3. Add a member to enum cciArguments at the appropriate position.         *
//*    4. Add a section to the default configuration file describing the option  *
//*       including a line with the command parameter and default argument.      *
//*       This section must be somewhere ABOVE the 'EndCfgCommands' section.     *
//*    5. Copy the description to the appropriate place in                       *
//*              'DefaultConfigFile[]' array.                                    *
//*    6. Add a case to ReadConfigFile() method to handle the input.             *
//*    7. Add a case to WriteConfigFile() method to handle the output.           *
//*    8. Add a case to CreateConfigFile() method for writing the new section.   *
//*    9. Add a case to the EditConfigOptions() method group to handle           *
//*       interactive configuration.                                             *
//*       - This includes a case in ecoDisplayCurrent().                         *
//*       - This includes a case in ecoEditParms().                              *
//*       - This includes a new member for enum ecoControls.                     *
//*       - This includes a new item in ContextHelp[] array.                     *
//*                                                                              *
//* B. Open-form Options                                                         *
//*    1. Add a member to CfgComp CfgList[] at the appropriate position.         *
//*       This must be somewhere ABOVE the 'Temporary Storage' entry.            *
//*    2. Add a member to enum cliArguments at the appropriate position.         *
//*    3. Add a section to the default configuration file describing the option  *
//*       including a line with the command parameter.                           *
//*       Include any desired default-value lines (optional).                    *
//*       This section must be somewhere ABOVE the 'Temporary Storage' section.  *
//*    4. Copy the description to the appropriate place in                       *
//*              'DefaultConfigFile[]' array.                                    *
//*    5. Add a case to ReadConfigFile() method to handle the input.             *
//*    6. Add a case to WriteConfigFile() method to handle the output.           *
//*    7. Add a case to CreateConfigFile() method for writing the new section.   *
//*    8. Add a case to the EditConfigOptions() method group to handle           *
//*       interactive configuration.                                             *
//*       - This includes a case in ecoDisplayCurrent().                         *
//*       - This includes a case in ecoEditParms().                              *
//*       - This includes a new member for enum ecoControls.                     *
//*       - This includes a new item in ContextHelp[] array.                     *
//*                                                                              *
//*             There may be other minor tweaks necessary.                       *
//*               That's why we do REGRESSION TESTING !!                         *
//*                                                                              *
//********************************************************************************


//****************
//* Header Files *
//****************
#include "FmConfig.hpp"       //* FmConfig class definition and constant data


//******************************
//* Local Definitions and Data *
//******************************
static const char* const badInvocation = 
   "\n"
   "  FmConfig is not a stand-alone application.\n"
   "  Please invoke through FileMangler's '-C' option.\n"
   "  Example:  ./FileMangler -C\n"
   "  Or from the FileMangler 'Util' menu, select 'Configure'.\n" ;


//* Copy of default configuration file *
static const char* const DefaultCfgFile[] =
{
//* Header Section *
"//******************************************************************************\n"
"//* FileMangler.cfg                                                            *\n"
"//*                                                                            *\n"
"//* This file contains the startup information for the FileMangler application.*\n"
"//* The parameters contained here are used as the initial settings for         *\n"
"//* FileMangler, unless these settings are overridden by command-line options. *\n"
"//*                                                                            *\n"
"//* Many settings can be changed on-the-fly within the application. For these, *\n"
"//* this configuration file provides _initial_ settings only.                  *\n"
"//*                                                                            *\n"
"//* The master copy of this file must reside in the same directory as the      *\n"
"//* FileMangler executable. It need not reside in the directory containing the *\n"
"//* shell script 'fmg'. The shell script contains the hard-coded path string   *\n"
"//* for the directory in which the executable and configuration files reside.  *\n"
"//*                                                                            *\n"
"//* Note that an alternate configuration file may be specified on the command  *\n"
"//* line or by editing the shell script file which by default is located in    *\n"
"//* the user's 'bin' directory:  ${HOME}/bin/fmg                               *\n"
"//* This is useful in cases where FileMangler is installed for all users, and  *\n"
"//* this option allows each to customize the startup options for his/her needs.*\n"
"//*                                                                            *\n"
"//* Settings may be modified manually using a text editor OR by using the      *\n"
"//* interactive configuration utility. The interactive configuration utility   *\n"
"//* may be invoked in one of two ways.                                         *\n"
"//*  1) Command-line option:     fmg -C                                        *\n"
"//*  2) FileMangler menu   :     open 'Util' menu and select 'Configure'       *\n"
"//*                                                                            *\n"
"//* C++ style comments may be added anywhere in the file                       *\n"
"//*  (note that C-style comments ARE NOT supported).                           *\n"
"//*                                                                            *\n"
"//* Each configuration option is preceeded by a short explanation of the       *\n"
"//* option and by a list of the supported values for that option.              *\n"
"//*                                                                            *\n"
"//* The default setting (if any) is indicated for each option. This is the     *\n"
"//* setting used if no configuration file is present or if the configuration   *\n"
"//* file contains syntax errors.                                               *\n"
"//*                                                                            *\n"
"//* Please refer to FileMangler Help (Texinfo: info filemangler) for           *\n"
"//* additional information on configuring the application.                     *\n"
"//*                                                                            *\n"
"//******************************************************************************\n"
"\n",

//* 'WindowMode' Section *
"//* The Window Mode indicates whether the application will be initially started\n" 
"//* in Single-Window Mode or Dual-Window Mode.\n"
"//*   Note: If the terminal window is too small for Single-Window mode, an \n" 
"//*         error message will be displayed and the application will exit. \n"
"//* Arguments:\n"
"//*   SingleWin   //* Start in Single-Window Mode\n"
"//*   DualWin     //* Start in Dual-Window Mode\n"
"//*                   If terminal window is too small for Dual-Window mode, the \n"
"//*                   application will display a warning and then start in \n"
"//*                   Single-Window mode.\n"
"//*   ByTermSize  //* The startup mode is determined by the size of the terminal \n"
"//*                   window in which the application is invoked. If the \n"
"//*                   terminal window is large enough, the application will \n"
"//*                   start in Dual-Window mode, else the application will \n"
"//*                   start in Single-Window mode. (default)",

//* 'SortOrder' Section *
"//* File-display sort option. This option specifies the order in which files are \n"
"//* displayed in the windows. Note that if two file-display windows are open, the \n"
"//* sort option is the same for both.\n"
"//* Arguments:\n"
"//*   Name        //* Sort by filename (default)\n"
"//*   NameR       //* Reverse sort by filename\n"
"//*   Date        //* Sort by date/timestamp (modification date)\n"
"//*   DateR       //* Reverse sort by date/time\n"
"//*   Size        //* Sort by file size\n"
"//*   SizeR       //* Reverse sort by file size\n"
"//*   Ext         //* Sort by filename extension\n"
"//*   ExtR        //* Reverse sort by filename extension\n"
"//*   Type        //* Sort by file type (Regular, FIFO, Socket etc)\n"
"//*   TypeR       //* Reverse sort by file type\n"
"//*   None        //* No Sort: Display files in the order in which they are\n" 
"//*                   stored in the directory file",

//* 'CaseSensitive' Section *
"//* The case-sensitivity flag indicates whether alphabetical sorting of files\n" 
"//* uses a case-sensitive or case-insensitive test. Sort options that use \n"
"//* alphabetical sort are 'Name', 'NameR', 'Ext', and 'ExtR' (see above). \n"
"//* For example, in a case sensitive sort, 'A' (0x041) is less than 'a' (0x061).\n"
"//* In a case-INsensitive sort 'A' and 'a' are treated as having equal value.\n"
"//* Arguments:\n"
"//*   true        //* Case-sensitive alphabetical sort (default)\n"
"//*   false       //* Case-insensitive alphabetical sort",

//* 'HiddenFiles' Section *
"//* The Show-Hidden-Files option is, as the name implies a toggle indicating\n" 
"//* whether hidden files will be displayed in the directory-display window(s).\n"
"//* Arguments:\n"
"//*   true        //* Show hidden files\n"
"//*   false       //* Do not show hidden files (default)",

//* 'ConfirmDelete' Section *
"//* Confirmation of file deletion. This option specifies the level of confirmation\n" 
"//* to provide when moving files to the Trashcan.\n"
"//* (Moving non-empty directory trees to the Trashcan AND            )\n"
"//* (deletions that cannot be un-deleted always require confirmation.)\n"
"//* Arguments:\n"
"//*   true        //* Always prompt for confirmation of file deletion\n"
"//*   false       //* Never confirm file deletion (default)",

//* 'ConfirmOverwrite' Section *
"//* Confirmation of file overwrite. This option specifies the level of confirmation \n"
"//* to provide when pasting a file into a directory that already contains a file of \n"
"//* that name.\n"
"//* Arguments:\n"
"//*   Always         //* Always prompt for confirmation of file overwrite\n"
"//*   ConfirmNewer   //* Confirm overwrite if the existing target file's \n"
"//*                      modification date is newer than the date of the file with \n"
"//*                      which it is to be overwritten. (default)\n"
"//*   Never          //* Never confirm file overwrite\n"
"//*                      (note that an existing directory name is never overwritten,\n"
"//*                       only the directory contents, and overwrite of a \n"
"//*                       write-protected file must always be confirmed.)",

//* 'LockMenuBar' Section
"//* Lock Application Menu Bar in visible state.\n"
"//* The Menu Bar shares screen space with the dialog title, and is normally\n"
"//* only visible when the user is making a menu selection. This configuration\n" 
"//* option may be used to lock the Menu Bar so it is always visible.\n"
"//* Arguments:\n"
"//*   true        //* Menu Bar is always visible\n"
"//*   false       //* Menu Bar visible only during menu-item selection (default)",

//* 'ColorScheme' Section *
"//* Specify the application 'Color Scheme'. \n"
"//* The Color Scheme allows selection of a set of display colors to be used \n"
"//* when drawing borders, dialog windows, menus, and other dialog objects.\n"
"//* Arguments:\n"
"//*   Default     //* (default): Cyan borders and blue dialog windows, menus;\n"
"//*                   optimized for contrast and readability\n"
"//*   Black       //* bright white foreground, black background\n"
"//*   Red         //* white foreground, red background\n"
"//*   Green       //* white foreground, green background\n"
"//*   Brown       //* white foreground, brown background\n"
"//*   Blue        //* white foreground, blue background\n"
"//*   Magenta     //* white foreground, magenta background\n"
"//*   Cyan        //* white foreground, cyan background\n"
"//*   Grey        //* white foreground, grey background\n"
"//*   Terminal    //* Use the terminal default foreground/background for most \n"
"//*                   objects with contrasting colors where necessary. (ugly!)",

//* 'EnableMouse' Section *
"//* Enable mouse support.\n"
"//* Capture all mouse activity which occurs within the application window.\n"
"//* - Left button single-click is generally interpreted as pressing the\n"
"//*   'Space' key, and left-button double-click is generally interpreted as\n"
"//*   pressing the 'Enter' key.\n"
"//* - ScrollWheel mouse events are interpreted as pressing the\n"
"//*   'UpArrow' or 'DownArrow' keys.\n"
"//* - Interpretation of these mouse events is modified when used in combination\n"
"//*   with the 'Ctrl' and 'Alt' modifier keys.\n"
"//* - All other mouse events will be ignored.\n"
"//* Please refer to the FileMangler documentation for more information on\n"
"//* application mouse support.\n"
"//* Arguments:\n"
"//*   enable      //* Enable mouse support\n"
"//*   disable     //* Disable mouse support (default)",

//* 'ExternalPrograms' Section *
"//* Specify how external programs are to be opened.\n"
"//* FileMangler will launch external applications such as binary executable\n" 
"//* programs, shell scripts, interpreted program files (Perl, Python, Ruby),\n" 
"//* or will launch the default application associated with various kinds of\n"
"//* data files such as text documents, LibreOffice documents, images,\n"
"//* audio and video media files and more. These external applications are\n"
"//* launched through the 'ViewFile' context menu.\n"
"//* Arguments:\n"
"//*   AutoLaunch    //* Application determines launch parameters if possible,\n"
"//*                     else open a dialog to get user's preference (default).\n"
"//*   AutoAudio     //* Same as 'AutoLaunch' but also assumes that audio files\n"
"//*                     will be opened by a GUI media player.\n"
"//*   SafeLaunch    //* If program can be SAFELY launched from the current\n"
"//*                     window without resource conflict, do so, otherwise open\n"
"//*                     a new terminal window from which to launch the external\n"
"//*                     program. Setup dialog will not be opened.\n"
"//*   ManualLaunch  //* Always open an interactive dialog to configure\n"
"//*                     launch parameters.",

//* 'EndCfgCommands' Section *
"//* This ends the 'command = parameter' sequence of configuration options. *\n"
"//* The remaining configuration data are of varying formats.               *\n"
"//* Please do not modify this command string.                              *\n"
"EndCfgCommands = End\n\n"
"//* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -*\n\n",

//* 'Favorite Directories' Section *
"//* List of user's favorite directories.\n"
"//* Absolute path specification for often-used directories. Environment \n"
"//* variable substitution is supported. (MUST NOT be a relative path.)\n"
"//* Examples: \n"
"//*  /home/Philip/Games\n"
"//*  ${HOME}/Documents/Taxes\n"
"//*  ~/bin\n"
"//* \n"
"//* Up to twenty-four (24) paths may be stored in this list.\n"
"//* A blank line signals the end of the list.\n"
"Favorite Directories:",

//* 'KeyMap' Section *
"//* Mapping of command-shortcut keys.\n"
"//* Many command-key shortcuts are available for quick access to actions \n"
"//* specified in the FileMangler menus. The default shortcut key assignment\n"
"//* (if any) for each item is listed on the right side of the menu item.\n"
"//* These command-shortcut keys fall into three groups\n"
"//*  Control key + another key  Ex: Control+s invokes the Sort-Option menu.\n"
"//*  Alt key + another key      Ex: Alt+q exits application to current directory\n"
"//*  Function keys              Ex: Shift+F01 opens FileMangler Help\n"
"//* Most of these shortcut keys may be reassigned/redefined to meet your needs.\n"
"//* Plese see the file 'FmKeymap.cfg' in the installation directory for\n"
"//* more information.\n"
"//* \n"
"//* To specify an alternate key mapping, specify the filename of the key map\n"
"//* here. The file specified MUST reside in the same directory as the\n"
"//* configuration file which is read when FileMangler is invoked.\n"
"//* If no key-map filename is specified, the default key map is used.\n",

//* 'Mount Commands' Section *
"//* List of dynamically mountable and un-mountable file systems.\n"
"//* List of absolute mount-point paths (NOT device path) which the\n"
"//* operating system has defined as being associated with particular devices.\n"
"//*   Example:  /run/media/sam/cdrom\n"
"//* The usual suspects for mountable file systems are CD-ROM drives, flash\n"
"//* drives, non-Linux/UNIX partitions, or network drives. Many other types of\n"
"//* file systems may also be mounted if they are fully defined by the \n"
"//* operating system (typically in the file /etc/fstab). The procedure for\n"
"//* defining and tracking mountable file systems is implementation-dependent,\n"
"//* so please refer to the 'mount' and 'umount' commands in your system\n"
"//* documentation for instructions on defining a file-system mount point and\n"
"//* its associated device.\n"
"//* \n"
"//* Important Note: If you wish to create an entry for your smartphone, please\n"
"//* use the Uniform Resource Identifier (URI), not the mountpath.\n"
"//* Example:  \"mtp://SAMSUNG_SAMSUNG_Android_ce012345c012345d01/\"\n"
"//* \n"
"//* Important Note: Certain file systems can only be mounted by the super-user.\n"
"//* Therefore, when mounting filesystems, the \"mount\" command will be sent\n"
"//* through the \"sudo\" command, so the system will ask for your password.\n"
"//* \n"
"//* Up to twenty-four (24) mountable filesystems may be stored in this list.\n"
"//* A blank line signals the end of the list.\n"
"Mount Commands:",

//* 'Alternate Trashcan' Section *
"//* Alternate location for desktop environment's Trashcan.\n"
"//* The trashcan directory for the desktop environment in GNOME, KDE or any\n"
"//* desktop using the freedesktop.org trashcan specification is located at:\n"
"//*                   ~/.local/share/Trash\n"
"//* If this directory exists, AND if you as the user have read/write access\n"
"//* to it, then it will be the location used by FileMangler when sending\n"
"//* files to the trashcan and this parameter (if specified) will be ignored.\n"
"//* \n"
"//* If you invoke FileMangler as the 'root' user, then the above directory\n"
"//* specification will likely be incorrect, and you will need to specify\n"
"//* root's trashcan location here.\n"
"//* \n"
"//* Also, for older versions of the desktop environment or for non-standard\n"
"//* installations, you may need to specify where the trashcan directory\n"
"//* is located. For instance, some older installations use ${HOME}/.Trash\n"
"//* \n"
"//* Be aware that FileMangler WILL RUN WITHOUT TRASHCAN ACCESS, but in that\n"
"//* case, the trashcan-files command will be disabled, and any file deletions\n"
"//* must be done through the delete-files command.\n"
"//* \n"
"//* Please specify an ABSOLUTE path, NOT a relative path.\n"
"//* A blank line following the command, indicates no-alternate-path-specified.\n"
"Alternate Trashcan:",

//* 'EndConfigFile' Section *
"//* This command marks the end of the configuration file.\n"
"//* Any data that follow it will be ignored.\n"
"EndConfigFile\n",
} ;

//* Copy of default key map configuration file (header section) *
//* See FmKeymap.hpp for the command-key descriptions.          *
static const char* const KeymapCfgHdr =
{
"//  Command Key map file for FileMangler configuration\n"
"// ====================================================\n"
"// By default, the command-key interface is defined according to the author's\n"
"// view of the most convenient and intuitive mapping of command-keys to\n"
"// application functionality. For example, the command key CTRL+R invokes the\n"
"// application's \"Rename file\" function, and the CTRL+S command key invokes\n"
"// the application's \"Sort file list\" function.\n"
"// These default values are listed in the column labelled: DEFAULT COMMAND KEY\n"
"// \n"
"// To enable remapping of command keys according to personal preference, this\n"
"// keymap configuration file may be specified by the \"KeyMap = filename\"\n"
"// configuration option in the FileMangler configuration file \n"
"// (named 'FileMangler.cfg' by default).\n"
"// On startup, the application will read this file and set the custom keymap\n"
"// according to the values specified in the column: CUSTOM COMMAND KEY\n"
"// \n"
"// The preferred method of specifying a custom keymap is through the\n"
"// application's configuration utility which interactively sets the desired\n"
"// values while automatically checking syntax and scanning for duplications.\n"
"// If desired, this file may be manually edited using a text editor so long as\n"
"// strict syntax and formatting are observed. Manual specification of custom\n"
"// command keys is subject to the following restrictions.\n"
"// 1) The file identifier must be the first line in the file. This ensures that\n"
"//    the file actually is a keymap configuration file.\n"
"// 2) Lines which begin with a C++ \"//\" comment command are interpreted as\n"
"//    comments and will be ignored. Blank lines are also ignored.\n"
"// 3) All non-comment lines present in the unmodified file must be present in\n"
"//    the modified file, _AND_ these lines must be in the same order.\n"
"// 4) Only the text under the \"CUSTOM COMMAND KEY\" column may be modified.\n"
"//    Data in the \"DEFAULT COMMAND KEY\" column may not be modified in any way.\n"
"//    Data in the \"DESCRIPTION OF COMMAND\" column is for information purposes\n"
"//    only, but should remain unmodified under most circumstances.\n"
"// 5) Data written to the \"CUSTOM COMMAND KEY\" column must be an _EXACT_ match\n"
"//    for one of the entries in the \"DEFAULT COMMAND KEY\" column. Any command\n"
"//    key not listed in the default column is unavailable as a command-key\n"
"//    sequence.\n"
"// 6) A command-key sequence appearing in the \"CUSTOM COMMAND KEY\" column\n"
"//    must appear only once. Duplication within the column is not allowed.\n"
"// 7) Command keys listed in the \"DEFAULT COMMAND KEY\" column may, or may not\n"
"//    be assigned to a function by default.\n"
"//    a) If a command key is assigned to a function by default, a description\n"
"//       of that function will be given.\n"
"//    b) If a command key is not assigned, its description will read:\n"
"//       \"unassigned by default\".\n"
"//    c) Note that assigning a custom command key to an entry listed as\n"
"//       \"unassigned by default\" will have no effect.\n"
"// 8) A command-key sequence may not be simultaneously active in both the\n"
"//    custom and default columns because it would indicate that the same\n"
"//    command-key is simultaneously assigned to two different functions.\n"
"//    a) A command key whose description is \"unassigned by default\" may be\n"
"//       used to directly remap any command function.\n"
"//    b) A command key which is assigned to an application function by default,\n"
"//       remains assigned to that function, unless it is explicitly assigned\n"
"//       to a different operation.\n"
"//    c) A command key which is assigned to an application function by default\n"
"//       may be used to remap any command, PROVIDED THAT its original function\n"
"//       is masked by assigning an unused command-key sequence to its original\n"
"//       function.\n"
"//       For example: the CTRL+A command key is assigned to select/deselect all\n"
"//       files. It's entry is listed as:\n"
"//       \"     --     | CTRL+A        Select/deselect all files in directory\"\n"
"//       To use CTRL+A for a different function, an unused command-key sequence\n"
"//       must be inserted into this entry. For example, the ALT+P command key\n"
"//       is unassigned by default, so to disable the default function of the\n"
"//       CTRL+A key its entry may be modified as follows:\n"
"//       \"ALT+P       | CTRL+A        Select/deselect all files in directory\"\n"
"//       The CTRL+A command key may then be remapped to a different function.\n"
"//       Example:\n"
"//       \"CTRL+A      | ALT+SHIFT+A   create, expand or modify an Archive file\"\n"
"//       Once these changes have been made, the ALT+P command invokes the\n"
"//       \"select all\" function, and the CTRL+A command invokes the\n"
"//        \"create archive\" function.\n"
"// \n"
"// CUSTOM COMMAND KEY   DEFAULT COMMAND KEY  DESCRIPTION OF COMMAND\n"
"// ===================  ===================  =================================================="
} ;


//*************************
//*        main           *
//*************************
//******************************************************************************
//* Program entry point.                                                       *
//*                                                                            *
//* Command-line Usage: (not a stand-alone executable, see above)              *
//*                                                                            *
//* Command Line Arguments:                                                    *
//*                                                                            *
//*   1) string denoting a shared-memory key (int)                             *
//*                                                                            *
//* Input  : argc     : number of command-line arguments                       *
//*          argv[]   : argument strings                                       *
//*          argenv[] : copy of terminal's environment variables               *
//*                                                                            *
//* Returns: member of ConfigStatus. Just in case, we ALSO return this value   *
//*          in parent's fcStatus field of the shared-memory segment because   *
//*          the return value from 'system()' is quite flakey, i.e.            *
//*                 for a return  of  0, parent receives 0    HOWEVER,         *
//*                 for a return  of  1, parent receives 256                   *
//*                 for a return  of  2, parent receives 512                   *
//*                 for a return  of -1, parent receives -256                  *
//*                 Therefore, we don't recommend using the returned exit code.*
//*                                                                            *
//******************************************************************************

int main ( int argc, char* argv[], char* argenv[] )
{
   ConfigStatus status = csARGS ;   // return value assume bad invocation

   //* If interactive configuration, invoked from within      *
   //* FileMangler, get virtual address of shared memory area.*
   int segID ;
   if ( argc > 1 && ((sscanf( argv[1], "%d", &segID )) == 1) )
   {
      //* Get a physical pointer to the shared memory area *
      //* and attach it to the application's memory space. *
      void* segAddress = shmat ( segID, (void *)NULL, ZERO ) ;
      bool isAttached = ( segAddress != (void*)(-1) ) ;
      if ( isAttached )
      {  //* Create a pointer to shared-memory space *
         sharedMemMap* shmPtr = (sharedMemMap*)segAddress ;

         //* If 'verbose diagnostics' mode, spill our guts.*
         if ( shmPtr->verbose )
         {
            //* Get details on the shared space *
            gString gs( "Shared Memory Established" ) ;
            gs.copy ( shmPtr->msg[shmPtr->msgCount++], smmMSG_LEN ) ;
            shmid_ds smStat ;
            int smsStatus = shmctl ( segID, IPC_STAT, &smStat ) ;
            if ( smsStatus == ZERO )
            {
               gs.compose( L" Created by : PID %d", &smStat.shm_cpid ) ;
               gs.copy ( shmPtr->msg[shmPtr->msgCount++], smmMSG_LEN ) ;
               gs.compose( L" Block size : %d bytes", &smStat.shm_segsz ) ;
               gs.copy ( shmPtr->msg[shmPtr->msgCount++], smmMSG_LEN ) ;
               gs.compose( L" Attachments: %d", &smStat.shm_nattch ) ;
               gs.copy ( shmPtr->msg[shmPtr->msgCount++], smmMSG_LEN ) ;
               gs.compose( L" Permissions: 0x%04X", &smStat.shm_perm.mode ) ;
               gs.copy ( shmPtr->msg[shmPtr->msgCount++], smmMSG_LEN ) ;
            }
            else
            {
               int err = errno ;
               gs.compose( L"shmctl failed! errno: %d", &err ) ;
               gs.copy ( shmPtr->msg[shmPtr->msgCount++], smmMSG_LEN ) ;
            }
         }

         //* Create the application class object *
         FmConfig FmConfig( shmPtr ) ;

         //* Get the return value *
         status = shmPtr->fcStatus = FmConfig.GetStatus () ;

         // NOTE: FmConfig class goes out of scope here and is destroyed
      }
      else
         status = csACCESS ;
   }
   #if 0    // FOR DEBUGGING ONLY
   else if ( argc > 1 && 
             ((strncmp (argv[1], "DEBUG_FileManglerConfig", 23 )) == ZERO) )
   {
      system ( "clear" ) ;
      wcout << "Debugging FileManglerConfig\n---------------------------" << endl ;
      if ( argc > 2 )
      {
         //* Create a pointer to FAKE shared-memory space and populate it. *
         sharedMemMap* shmPtr = new sharedMemMap ;
         realpath ( "./1_TestData", shmPtr->appPath ) ;
         gString gt( shmPtr->appPath ) ;
         gt.append( "/fm.cfg" ) ;
         gt.copy( shmPtr->cfgPath, MAX_PATH ) ;
         strncpy ( shmPtr->trashPath, "~/.local/share/Trash", MAX_PATH ) ;
         strncpy ( shmPtr->favPath[0], ConfigDfltFavorite, MAX_PATH ) ;
         shmPtr->favCount = 1 ;
         shmPtr->mntCount = ZERO ;
         shmPtr->msgCount = ZERO ;
         if ( !(strncmp ( argv[2], "ccREAD", 6 )) )             shmPtr->ccCmd = ccREAD ;
         else if ( !(strncmp ( argv[2], "ccREADSILENT", 12 )) ) shmPtr->ccCmd = ccREADSILENT ;
         else if ( !(strncmp ( argv[2], "ccINTERACT", 10 )) )   shmPtr->ccCmd = ccINTERACT ;
         else if ( !(strncmp ( argv[2], "ccKEYMAP", 8 )) )      shmPtr->ccCmd = ccKEYMAP ;
         else if ( !(strncmp ( argv[2], "ccCREATEp", 9 )) )
         {
            shmPtr->ccCmd = ccCREATEp ;
            wcout << "Create initialized configuration file:\n"
                  << shmPtr->cfgPath << endl ;
         }
         else if ( !(strncmp ( argv[2], "ccCREATE", 8 )) )
         {
            shmPtr->ccCmd = ccCREATE ;
            wcout << "Create default configuration file:\n" 
                  << shmPtr->cfgPath << endl ;
         }
         else if ( !(strncmp ( argv[2], "ccWRITE", 7 )) )       shmPtr->ccCmd = ccWRITE ;
         shmPtr->verbose = ( argc > 3 ) ? true : false ; // verbose output

         FmConfig FmConfig( shmPtr ) ;

         for ( short i = ZERO ; i < shmPtr->msgCount ; i++ )
            wcout << shmPtr->msg[i] << endl ;

         delete shmPtr ;
         status = csOK ;
      }
      else
      {
         wcout << 
            "ccREAD       - read the specified configuration file\n"
            "ccREADSILENT - same as ccREAD, but no diagnostic message written\n"
            "ccINTERACT   - interact with user to set/modify configuration options\n"
            "ccKEYMAP     - create a new keymap configuration file with default values\n"
            "ccCREATE     - create a new configuration file with default values\n"
            "ccCREATEp    - create a new configuration file with specified values\n"
            "ccWRITE      - write specified data to to specified configuration file"
            << endl ;
      }
   }
   #endif   // FOR DEBUGGING ONLY
   if ( status == csARGS )
   {
      wcout << badInvocation << endl ;
   }
   exit ( int(status) ) ;

}  //* End main() *

//*************************
//*      ~FmConfig        *
//*************************
//******************************************************************************
//* Destructor.                                                                *
//* Return all resource to system, and unload the NCurses engine.              *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

FmConfig::~FmConfig ( void )
{

   this->termRows = this->termCols = ZERO ;
   this->dialogPosY = this->dialogPosX = ZERO ;
   this->icCount = ZERO ;
   if ( this->dPtr != NULL )
   {
      delete this->dPtr ;
      this->dPtr = NULL ;
   }
   if ( this->ic != NULL )
   {
      delete [] this->ic ;
      this->ic = NULL ;
   }
   this->StopNCursesEngine () ;

}  //* End ~FmConfig() *

//*************************
//*      FmConfig         *
//*************************
//******************************************************************************
//* Constructor.                                                               *
//*                                                                            *
//* Input  : shmPtr : pointer to shared-memory segment laid out as a           *
//*                   sharedMemMap-class object                                *
//*                                                                            *
//* Returns: nothing (implicit pointer to instance)                            *
//******************************************************************************
//* Programmer's Note:                                                         *
//* - We communicate with the parent process through the shared memory space   *
//*   which is defined as a sharedMemMap-class object.                         *
//* - For simply reading the specified configuration file, we output NOTHING   *
//*   to the display. Any errors are indicated in the 'msg' array in the       *
//*   shared space.                                                            *
//*                                                                            *
//******************************************************************************

FmConfig::FmConfig ( sharedMemMap* shmPtr )
{
   //* Initialize our data members.                               *
   //* Some values, we won't know until the NCurses Engine starts.*
   this->shm         = shmPtr ;     // access to the shared-memory space
   this->dColor      =              // NCurses not yet initialized
   this->pbnColor = this->expColor = this->pbfColor = this->rbnColor = 
   this->rbfColor = this->tbnColor = this->tbfColor = this->spnColor = 
   this->spfColor = ZERO ;
   this->termRows    = ZERO ;       // size of terminal window
   this->termCols    = ZERO ;
   this->dialogRows  = MIN_ROWS+4 ; // height of dialog window is fixed
   this->dialogCols  = MIN_COLS ;   // width of dialog window is fixed
   this->dialogPosY  = ZERO ;       // initially upper left of terminal window
   this->dialogPosX  = ZERO ;
   this->icCount     = ZERO ;       // no controls defined yet
   this->ic          = NULL ;
   this->dPtr        = NULL ;       // no dialog window open yet
   this->csStatus    = csOK ;       // hope for the best !

   if ( this->shm->verbose )
   {
      gString gs( "FmConfig constructor" ) ;
      gs.copy ( shmPtr->msg[shmPtr->msgCount++], smmMSG_LEN ) ;
      gs.compose( L" operation: %hd", &this->shm->ccCmd ) ;
      gs.copy ( shmPtr->msg[shmPtr->msgCount++], smmMSG_LEN ) ;
   }

   switch ( this->shm->ccCmd )
   {
      case ccREAD:         // read config file and return diagnostics
      case ccREADSILENT:   // read config file with no diagnostic messages
         this->csStatus = this->ReadConfigFile ( this->shm->ccCmd ) ;
         break ;
      case ccINTERACT:     // interactively set all configuration parameters
         if ( (this->StartNCursesEngine ()) == OK )
         {  //* If caller's config file exists, read it.*
            this->ReadConfigFile ( ccREADSILENT ) ;
            this->csStatus = this->EditConfigOptions ( this->shm->ccCmd ) ;
         }
         else
            this->csStatus = csNOCURSES ;
         break ;
      case ccCREATE: // create a new target file using default data
         this->csStatus = this->CreateConfigFile () ;
         break ;
      case ccCREATEp: // create a new target file using data sent by parent process
         this->csStatus = this->CreateConfigFile ( true ) ;
         break ;
      case ccWRITE:  // save data sent by parent process to existing (or new) target file
         this->WriteConfigFile () ;
         break ;
      case ccKEYMAP:       // create a new keymap configuration file using default data
         //* Note that parent process will never send new or modified *
         //* keymap data, so the default 'false' parameter is used.   *
         this->csStatus = this->CreateKeymapFile () ;
         break ;
      default:       // invalid option (unlikely)
         this->csStatus = csERR ;
         break ;
   }

}  //* End FmConfig() *

//*************************
//*  StartNCursesEngine   *
//*************************
//******************************************************************************
//* Start the NCurses Engine and initialize the FmConfig data members which    *
//* are dependent upon it.                                                     *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: OK if successful, else ERR                                        *
//******************************************************************************

short FmConfig::StartNCursesEngine ( void )
{
   gString gsv ;
   short status ;
   if ( (status = nc.StartNCursesEngine ()) == OK )
   {
      nc.SetCursorState ( nccvINVISIBLE ) ;  // hide the cursor
      nc.SetKeyProcState ( nckpRAW ) ;       // allow CTRL keys through unprocessed
      nc.ClearScreen () ;                    // clear the terminal window 

      //* Verify that default locale supports UTF-8 encoding *
      short localeSetOk = nc.VerifyLocale () ;
      if ( localeSetOk != OK && (this->shm->msgCount < smmMSG_COUNT) )
      {
         gsv = "  Locale may not support UTF-8 encoding." ;
         gsv.copy ( this->shm->msg[this->shm->msgCount++], smmMSG_LEN ) ;
      }

      //* Check for color support *
      short colorSupport = ERR ;
      if ( (nc.ColorText_Available ()) && (nc.ColorText_Initialized ()) )
         colorSupport = OK ;
      else
      {
         if ( this->shm->msgCount < smmMSG_COUNT )
         {
            gsv = "  Multi-color screen output NOT supported." ;
            gsv.copy ( this->shm->msg[this->shm->msgCount++], smmMSG_LEN ) ;
         }
      }
      if ( this->shm->verbose && (this->shm->msgCount < smmMSG_COUNT) )
      {
         gsv = "NCurses engine initialized." ;
         gsv.copy ( this->shm->msg[this->shm->msgCount++], smmMSG_LEN ) ;
         if ( localeSetOk == OK && (this->shm->msgCount < smmMSG_COUNT) )
         {
            gsv = "  UTF-8 encoding support verified." ;
            gsv.copy ( this->shm->msg[this->shm->msgCount++], smmMSG_LEN ) ;
         }
         if ( colorSupport == OK && (this->shm->msgCount < smmMSG_COUNT) )
         {
            gsv = "  Multi-color screen output is supported." ;
            gsv.copy ( this->shm->msg[this->shm->msgCount++], smmMSG_LEN ) ;
         }
      }
      if ( colorSupport == OK )
      {
         this->dColor   = nc.blR ;     // set base dialog color
         this->expColor = nc.brbl ;    // set color for explanation text
         this->pbnColor = nc.gyR ;     // set pushbutton non-focus color
         this->pbfColor = nc.reR ;     // set pushbutton focus color
         this->rbnColor = this->dColor ; // set radio button non-focus color
         this->rbfColor = this->dColor & ~(ncrATTR | ncbATTR) ; // set radio button focus color
         this->tbnColor = nc.bw ;      // set text-box non-focus color
         this->tbfColor = nc.br ;      // set text-box focus color
         this->spnColor = nc.gyR ;     // set spinner non-focus color
         this->spfColor = nc.bw ;      // set spinner focus color
      }
      else     // if running on a monochrome terminal, do the best we can
      {
         this->dColor   = nc.bw ;
         this->expColor = nc.bw ;
         this->pbnColor = nc.bw | ncbATTR ;
         this->pbfColor = nc.bwR ;
         this->rbnColor = nc.bw | ncbATTR ;
         this->rbfColor = nc.bwR ;
         this->tbnColor = nc.bw ;
         this->tbfColor = nc.bwR ;
         this->spnColor = nc.bw ;
         this->spfColor = nc.bwR ;
      }

      //* Get the size of our playground and be sure that    *
      //* the terminal window is large enough for the dialog.*
      wkeyCode wk ;
      bool winSizeOK ;
      do
      {
         nc.ScreenDimensions ( this->termRows, this->termCols ) ;
         this->dialogPosY = ZERO ;
         this->dialogPosX = this->termCols / 2 - (this->dialogCols / 2) ;
         winSizeOK = bool(this->termRows >= this->dialogRows && 
                          this->termCols >= this->dialogCols) ;
         if ( ! winSizeOK )
         {
            nc.ClearScreen () ;
            short y = 2, x = 2 ;
            gsv.compose( L" Terminal window must be at least %02hd rows by %02hd columns ", 
                         &this->dialogRows, &this->dialogCols ) ;
            nc.WriteString ( y++, x, gsv.gstr(), nc.reG ) ;
            nc.WriteString ( y++, x, 
                  L"             for Interactive Configuration.              ", nc.reG ) ;
            nc.WriteString ( y++, x, 
                  L"           Please re-size the terminal window.           ", nc.reG ) ;
            nc.GetKeyInput ( wk ) ;
            //* If user is signalling de-feet *
            if ( wk.type == wktFUNKEY && (wk.key == nckESC || wk.key == nckC_C) )
            {
               if ( this->shm->msgCount < smmMSG_COUNT )
                  gsv.copy ( this->shm->msg[this->shm->msgCount++], smmMSG_LEN ) ;
               StopNCursesEngine () ;
               status = ERR ;
               break ;
            }
         }
      }
      while ( ! winSizeOK ) ;
   }
   else if ( this->shm->msgCount < smmMSG_COUNT )
   {
      gsv = "ERROR! Unable to initialize NCurses engine." ;
      gsv.copy ( this->shm->msg[this->shm->msgCount++], smmMSG_LEN ) ;
   }
   return status ;

}  //* End StartNCursesEngine() *

//*************************
//*   StopNCursesEngine   *
//*************************
//******************************************************************************
//* Shut down the NCurses Engine and reset associated data members.            *
//*                                                                            *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void FmConfig::StopNCursesEngine ( void )
{

   nc.RestoreCursorState () ;             // make cursor visible
   nc.StopNCursesEngine () ;              // Deactivate the NCurses engine

}  //* End StopNCursesEngine() *

//*************************
//*    ReadConfigFile     *
//*************************
//******************************************************************************
//* Read the configuration file and initialize the ConfigOptions structure.    *
//*                                                                            *
//* Input  : readType: either ccREAD or ccREADSILENT                           *
//*                                                                            *
//* Returns: member of enum ConfigStatus                                       *
//******************************************************************************

ConfigStatus FmConfig::ReadConfigFile ( ConfigCommand readType )
{
   gString gs, gsv ;                // work buffers
   bool silent = (readType == ccREADSILENT ? true : false) ;
   ConfigStatus status = csOK ;     // return value

   //* Determine whether default desktop trashcan       *
   //* exists, and if it does, set the path to it.      *
   //* Else, set NULL path indicating (nominal) failure.*
   gs = ConfigDfltTrashcanPath ;
   if ( (this->rcfValidateTrashPath ( gs )) != false )
      gs.copy( this->shm->cfgOpt.trPath, MAX_PATH ) ;
   else
      *this->shm->cfgOpt.trPath = NULLCHAR ;
   if ( !silent && this->shm->verbose && (this->shm->msgCount < smmMSG_COUNT) )
   {
      gsv.compose( L"Trashcan: '%s' ", this->shm->cfgOpt.trPath ) ;
      gsv.append( (*this->shm->cfgOpt.trPath == NULLCHAR) ? "ERR" : "OK" ) ;
      gsv.copy ( this->shm->msg[this->shm->msgCount++], smmMSG_LEN ) ;
   }

   if ( !silent && this->shm->verbose && (this->shm->msgCount < smmMSG_COUNT) )
   {
      short i = ZERO ;
      while ( this->shm->cfgPath[i] != NULLCHAR )
         ++i ;
      while ( (i > ZERO) && (this->shm->cfgPath[i] != '/') )
         --i ;
      if ( this->shm->cfgPath[i] == '/' )
         ++i ;
      gsv.compose( L"Reading: '%s'", &this->shm->cfgPath[i] ) ;
      gsv.copy ( this->shm->msg[this->shm->msgCount++], smmMSG_LEN ) ;
   }

   //* Open the source file *
   ifstream ifs ( this->shm->cfgPath, ifstream::in ) ;
   if ( ifs.is_open() )             // if input file open
   {
      char  lineData[gsDFLTBYTES] ; // raw UTF-8 input
      const wchar_t* wptr ;         // pointer to work buffer
      short readStatus,             // status of line read
            syntaxErrors = ZERO,    // count the syntax error found
            syntaxMax = (silent ? 99 : 2), // max syntax errors before abort
            lineNum = ZERO ;        // source-line counter
      bool  done = false ;          // loop control

      //* Verify that this is actually a configuration file *
      ifs.getline ( lineData, gsDFLTBYTES, NEWLINE ) ;
      if ( ifs.good() && (!(strncmp ( lineData, ConfigFileID, 5 ))) )
         ++lineNum ;
      else
      {
         if ( !silent && this->shm->msgCount < smmMSG_COUNT )
         {
            gs = "Not a valid FileMangler configuration file!" ;
            gs.copy ( this->shm->msg[this->shm->msgCount++], smmMSG_LEN ) ;
         }
         done = true ;
      }

      short ccIndex = ZERO ;              // index into command array
      short aaIndex ;                     // index into argument array
      short aIndex ;                      // index into command string
      bool  endCmds = false ;             // true if 'EndCfgCommands' encountered
      while ( ! done )
      {
         if ( (readStatus = this->rcfReadSourceLine ( ifs, lineNum, gs )) > ZERO )
         {
            wptr = gs.gstr() ;      // point to work buffer

            //* If 'verbose' diagnostics, display source line *
            if ( !silent && this->shm->verbose && (this->shm->msgCount < smmMSG_COUNT) )
            {
               gsv.compose( L"LINE%4hd: %S", &lineNum, gs.gstr() ) ;
               gsv.copy ( this->shm->msg[this->shm->msgCount++], smmMSG_LEN ) ;
            }

            //* Search for the config command in list of valid commands *
            for ( ccIndex = ZERO ; ccIndex < cciArgs ; ccIndex++ )
            {
               if ( (wcsncmp ( wptr, CfgParm[ccIndex].str, CfgParm[ccIndex].len )) == ZERO )
                  break ;
            }
            if ( ccIndex < cciArgs )
            {  //* Command parameter string found.*
               //* Index the command argument.    *
               for ( aIndex = CfgParm[ccIndex].len + 1 ;
                     wptr[aIndex] == nckSPACE
                     || wptr[aIndex] == EQUAL ; aIndex++ ) ;
               aaIndex = ZERO ;
               const wchar_t* argptr = &wptr[aIndex] ;

               //* These parameters have numeric arguments. *
               //* (currently no numeric arguments defined) *
               short numberArg = ZERO ;
               if ( false )
               {
                  //* Get the numeric argument *
                  if ( (swscanf ( argptr, L"%hd", &numberArg)) == 1 )
                  {  // * Range the argument according to parameter *
                     if ( true )
                        aaIndex = (-1) ;  // fool the 'match found' test below
                  }
               }
               //* Compare argument to argument strings *
               else
               {
                  do
                  {
                     if ( (wcsncmp ( argptr, CfgParm[ccIndex].args[aaIndex].str,
                                     CfgParm[ccIndex].args[aaIndex].len )) == ZERO )
                        break ;     // found a match
                     ++aaIndex ;
                  }
                  while ( aaIndex < CfgParm[ccIndex].argcnt ) ;
               }
               
               //* If a match found, process the command *
               if ( aaIndex < CfgParm[ccIndex].argcnt )
               {
                  switch ( ccIndex )
                  {
                     case cciWM:          // "WindowMode"
                        switch ( aaIndex )
                        {
                           case ZERO: this->shm->cfgOpt.winMode = wmSINGLE_WIN ;    break ;
                           case 1:    this->shm->cfgOpt.winMode = wmDUAL_WIN ;      break ;
                           case 2:    this->shm->cfgOpt.winMode = wmTERM_WIN ;      break ;
                           default:   break ;   // should never happen
                        }
                        break ;
                     case cciHF:          // "HiddenFiles"
                        switch ( aaIndex )
                        {
                           case ZERO: this->shm->cfgOpt.showHidden = true ;         break ;
                           case 1:    this->shm->cfgOpt.showHidden = false ;        break ;
                           default:   break ;
                        }
                        break ;
                     case cciCD:          // "ConfirmDelete"
                        switch ( aaIndex )
                        {
                           case ZERO: this->shm->cfgOpt.confirmDelete = true ;      break ;
                           case 1:    this->shm->cfgOpt.confirmDelete = false ;     break ;
                           default:   break ;
                        }
                        break ;
                     case cciCO:          // "ConfirmOverwrite"
                        switch ( aaIndex )
                        {
                           case ZERO: this->shm->cfgOpt.overWrite = owNO_CONFIRM ;  break ;
                           case 1:    this->shm->cfgOpt.overWrite = owCONFIRM_NEWER;break ;
                           case 2:    this->shm->cfgOpt.overWrite = owCONFIRM_ALL ; break ;
                           default:   break ;
                        }
                        break ;
                     case cciSO:          // "SortOption"
                        switch ( aaIndex )
                        {
                           case ZERO: this->shm->cfgOpt.sortOption = fmNO_SORT ;    break ;
                           case 1:    this->shm->cfgOpt.sortOption = fmNAMEr_SORT ; break ;
                           case 2:    this->shm->cfgOpt.sortOption = fmNAME_SORT ;  break ;
                           case 3:    this->shm->cfgOpt.sortOption = fmDATEr_SORT ; break ;
                           case 4:    this->shm->cfgOpt.sortOption = fmDATE_SORT ;  break ;
                           case 5:    this->shm->cfgOpt.sortOption = fmSIZEr_SORT ; break ;
                           case 6:    this->shm->cfgOpt.sortOption = fmSIZE_SORT ;  break ;
                           case 7:    this->shm->cfgOpt.sortOption = fmTYPEr_SORT ; break ;
                           case 8:    this->shm->cfgOpt.sortOption = fmTYPE_SORT ;  break ;
                           case 9:    this->shm->cfgOpt.sortOption = fmEXTr_SORT ;  break ;
                           case 10:   this->shm->cfgOpt.sortOption = fmEXT_SORT ;   break ;
                           default:   break ;
                        }
                        break ;
                     case cciCS:          // "CaseSensitive"
                        switch ( aaIndex )
                        {
                           case ZERO: this->shm->cfgOpt.caseSensitive = true ;      break ;
                           case 1:    this->shm->cfgOpt.caseSensitive = false ;     break ;
                           default:   break ;
                        }
                        break ;
                     case cciLM:          // "LockMenuBar"
                        switch ( aaIndex )
                        {
                           case ZERO: this->shm->lockMb = true ;                    break ;
                           case 1:    this->shm->lockMb = false ;                   break ;
                           default:   break ;
                        }
                        break ;
                     case cciSC:          // "ColorScheme"
                        switch ( aaIndex )
                        {
                           case ZERO: this->shm->cfgOpt.cScheme.scheme = ncbcCOLORS;break ;
                           case 1:    this->shm->cfgOpt.cScheme.scheme = ncbcBK ;   break ;
                           case 2:    this->shm->cfgOpt.cScheme.scheme = ncbcRE ;   break ;
                           case 3:    this->shm->cfgOpt.cScheme.scheme = ncbcGR ;   break ;
                           case 4:    this->shm->cfgOpt.cScheme.scheme = ncbcBR ;   break ;
                           case 5:    this->shm->cfgOpt.cScheme.scheme = ncbcBL ;   break ;
                           case 6:    this->shm->cfgOpt.cScheme.scheme = ncbcMA ;   break ;
                           case 7:    this->shm->cfgOpt.cScheme.scheme = ncbcCY ;   break ;
                           case 8:    this->shm->cfgOpt.cScheme.scheme = ncbcGY ;   break ;
                           case 9:    this->shm->cfgOpt.cScheme.scheme = ncbcDEFAULT;break ;
                           default:   break ;
                        }
                        break ;
                     case cciEM:          // "EnableMouse"
                        switch ( aaIndex )
                        {
                           case ZERO: this->shm->cfgOpt.enableMouse = true ;   break ;
                           case 1:    this->shm->cfgOpt.enableMouse = false ;  break ;
                           default:   break ;
                        }
                        break ;
                     case cciEP:          // "ExternalPrograms"
                        switch ( aaIndex )
                        {
                           case 0:     this->shm->cfgOpt.lCode = lcAutoLaunch ;     break ;
                           case 1:     this->shm->cfgOpt.lCode = lcAutoAudio ;      break ;
                           case 2:     this->shm->cfgOpt.lCode = lcSafeLaunch ;     break ;
                           case 3:     this->shm->cfgOpt.lCode = lcManualLaunch ;   break ;
                           default:    break ;
                        } ;
                     case cciEC:          // "EndCfgCommands"
                        endCmds = true ;
                        break ;
                     default:             // should never happen
                        break ;
                  }  // switch(ccIndex)
               }  // if()
               else
               {
                  if ( !silent && this->shm->msgCount < smmMSG_COUNT )
                  {
                     gsv.compose( L" Syntax error on line %hd ", &lineNum ) ;
                     gsv.copy ( this->shm->msg[this->shm->msgCount++], smmMSG_LEN ) ;
                     if ( this->shm->msgCount < smmMSG_COUNT )
                     {
                        gsv.compose( L" ->%S", gs.gstr() ) ;
                        gsv.copy ( this->shm->msg[this->shm->msgCount++], smmMSG_LEN ) ;
                     }
                  }
                  status = csSYNTAX ;
                  //* Abort after two errors UNLESS we are reading silently *
                  if ( ++syntaxErrors >= syntaxMax )
                     done = true ;
               }
            }

            //* Command is not in the 'CfgParm' array.                    *
            //* If all fixed-arg commands completed, scan 'CfgList' array.*
            else if ( endCmds )
            {
               ConfigStatus rStat = csOK ;
               bool cmdfound = false ;
               for ( ccIndex = ZERO ; ccIndex < cliArgs ; ccIndex++ )
               {
                  if ( (wcsncmp ( wptr, CfgList[ccIndex].str, CfgList[ccIndex].len )) == ZERO )
                  {
                     // NOTE: If we are running in 'silent' mode, discard any 
                     //       messages posted by these sub-methods.
                     short messageCount = this->shm->msgCount ;
                     switch ( ccIndex )
                     {
                        case cliFD: // 'Favorite Directories' header
                           cmdfound = true ;
                           rStat = this->rcfLoadFavorites ( ifs, lineNum ) ;
                           break ;
                        case cliKM: // 'KeyMap' header
                           cmdfound = true ;
                           rStat = this->rcfLoadKeymap ( gs ) ;
                           break ;
                        case cliMC: // 'Mount Commands' header
                           cmdfound = true ;
                           rStat = this->rcfLoadMounts ( ifs, lineNum ) ;
                           break ;
                        case cliAT: // 'Alternate Trashcan' header
                           cmdfound = true ;
                           rStat = this->rcfLoadAltTrash ( ifs, lineNum ) ;
                           break ;
                        case cliEF: // 'End Config File' marker
                           cmdfound = true ;
                           //* Any data following this marker will be ignored.*
                           done = true ;
                           break ;
                        default:    // (should never happen)
                           break ;
                     }
                     if ( rStat != csOK )
                        status = rStat ;
                     if ( silent )
                        this->shm->msgCount = messageCount ;
                     break ;
                  }
               }
               if ( ! cmdfound )
               {
                  if ( !silent && this->shm->msgCount < smmMSG_COUNT )
                  {
                     gsv.compose( L" Syntax error on line %hd ", &lineNum ) ;
                     gsv.copy ( this->shm->msg[this->shm->msgCount++], smmMSG_LEN ) ;
                     if ( this->shm->msgCount < smmMSG_COUNT )
                     {
                        gsv.compose( L" ->%S", gs.gstr() ) ;
                        gsv.copy ( this->shm->msg[this->shm->msgCount++], smmMSG_LEN ) ;
                     }
                  }
                  status = csSYNTAX ;
                  if ( ++syntaxErrors > 1 )  // abort after two errors
                     done = true ;
               }
            }
            else
            {
               if ( !silent && this->shm->msgCount < smmMSG_COUNT )
               {
                  gsv.compose( L" Syntax error on line %hd ", &lineNum ) ;
                  gsv.copy ( this->shm->msg[this->shm->msgCount++], smmMSG_LEN ) ;
                  if ( this->shm->msgCount < smmMSG_COUNT )
                  {
                     gsv.compose( L" ->%S", gs.gstr() ) ;
                     gsv.copy ( this->shm->msg[this->shm->msgCount++], smmMSG_LEN ) ;
                  }
               }
               status = csSYNTAX ;
               if ( ++syntaxErrors > 1 )  // abort after two errors
                  done = true ;
            }
         }
         else if ( readStatus == ZERO )
            { /* ignores comment or blank line */ }
         else     // (readStatus < ZERO) end-of-file
            done = true ;
      }
      ifs.close() ;                 // close the file
   }
   else
   {
      if ( !silent && this->shm->msgCount < smmMSG_COUNT )
      {
         gString gs( "Unable to open configuration file." ) ;
         gs.copy ( this->shm->msg[this->shm->msgCount++], smmMSG_LEN ) ;
      }
      status = csNOTFOUND ;
   }

   return status ;

}  //* End ReadConfigFile() *

//*************************
//*   rcfLoadFavorites    *
//*************************
//******************************************************************************
//* Read the section of configuration file containing 'Favorite' directories.  *
//* Caller has located and verified the header line for this data section.     *
//*                                                                            *
//* Input  : ifs     : handle to open input stream                             *
//*          lineNum : most-recently-read line of source file                  *
//*                                                                            *
//* Returns: member of enum ConfigStatus                                       *
//******************************************************************************
//* Programmer's Note:                                                         *
//* We would like to do a 'realpath' or 'stat' on each of these directory      *
//* specifications; however, even if the path is a valid one, there is no      *
//* guarantee that the device is currently mounted.                            *
//******************************************************************************

ConfigStatus FmConfig::rcfLoadFavorites ( ifstream& ifs, short& lineNum )
{
   gString gs, gsv ;             // work buffers
   short readStatus ;            // status of line read
   ConfigStatus status = csOK ;  // return value
   bool  done = false ;          // loop control

   //* Discard any entries that may have been sent from within *
   //* the application. These would most likely be duplicates. *
   this->shm->favCount = ZERO ;
   while ( ! done )
   {
      //* Read a source line *
      if ( (readStatus = this->rcfReadSourceLine ( ifs, lineNum, gs )) > ZERO )
      {
         if ( this->shm->favCount < MAX_FAVDIRS )
         {
            //* Perform environment-variable substitution, if necessary.*
            if ( this->shm->ccCmd != ccINTERACT )
               this->rcfEnvExpansion ( gs ) ;
            //* Store the path string *
            gs.copy( this->shm->favPath[this->shm->favCount++], MAX_PATH ) ;
         }
         else
            ; // silently ignore entries we don't have room for
      }
      else
      {  //* Ignore comments - list is terminated by a blank line (or EOF) *
         if ( (gs.gschars()) == 1 )
         {
            if ( this->shm->verbose && (this->shm->msgCount < smmMSG_COUNT) )
            {
               gsv.compose( L"          (%hd specified)", &this->shm->favCount ) ;
               gsv.copy ( this->shm->msg[this->shm->msgCount++], smmMSG_LEN ) ;
            }
            //* Fill remainder of array with null strings *
            short i = this->shm->favCount ;
            while ( i < MAX_FAVDIRS )
               *this->shm->favPath[i++] = NULLCHAR ;
            done = true ;
         }
      }
   }
   return status ;

}  //* End rcfLoadFavorites() *

//*************************
//*     rcfLoadKeymap     *
//*************************
//******************************************************************************
//* Read the keymap configuration file.                                        *
//* 1) If no source file is specified, do nothing.                             *
//* 2) If a source file IS specified, then:                                    *
//*    a) Decode each record and if it contains a user-defined keycode, insert *
//*       it into the keymap object.                                           *
//*    b) Initialize the keymap object.                                        *
//*                                                                            *
//* Input  : kmCmd   : contains "KeyMap" command and optionally, its argument  *
//*                    i.e. the keymap-config filename                         *
//*                    Examples: KeyMap =                   (without argument) *
//*                              KeyMap = FmKeymap.cfg      (with argument)    *
//*                                                                            *
//* Returns: member of enum ConfigStatus                                       *
//*          csOK  : either no keymap file specified, OR file read successfully*
//*          csERR :                                                           *
//******************************************************************************
//* Notes:                                                                     *
//* - Verify that all records are present and are in the correct order.        *
//* - Verify that all user-defined values are valid command keys.              *
//* - Verify that there are no doubly-mapped command keys.                     *
//* - Diagnostics:                                                             *
//*   - success messages are written ONLY in verbose mode                      *
//*   - if syntax errors are encountered, messages will be written; however,   *
//*     caller may discard them if in 'silent' mode.                           *
//*                                                                            *
//* Format of a source record:                                                 *
//* --------------------------                                                 *
//*      --        | CTRL+A           Select/deselect all files in directory   *
//*      --        | CTRL+B           unassigned by default                    *
//*      --        | CTRL+C           Copy selected files to clipboard
//* CTRL+T         | CTRL+D           Date (timestamp) modification
//* _______________  ________________ ________________________________________ *
//*      |             |              |                                        *
//*      |             |              |                                        *
//*      |             |              +-- Description of key function (ignored)*
//*      |             +--------- Default key code                              *
//*      |                                                                     *
//*      +---------------------- Remapped key code/keycode (if any)            *
//*                                                                            *
//******************************************************************************

ConfigStatus FmConfig::rcfLoadKeymap ( const gString& kmCmd )
{
   gString gs,                      // input buffer
           gsv ;                    // diagnostics formatting
   short lineNum = ZERO,            // source-line counter
         userValues = ZERO ;        // number of user-defined values scanned
   ConfigStatus status = csOK ;     // return value

   //* Construct the filespec for source file and verify its existence.*
   gString kmSpec ;
   short indx = kmCmd.find( L'=' ) ;
   bool  trgExists = false,         // true if target file exists
         mappingError = false ;     // true if duplicate keycodes or 
                                    //   other non-syntax user error
   if ( indx >= ZERO )
   {
      while ( (kmCmd.gstr()[++indx]) == SPACE ) ;
      if ( kmCmd.gstr()[indx] != NULLCHAR )
      {
         gs = this->shm->cfgPath ;
         gs.limitChars( gs.findlast( L'/' ) ) ;
         kmSpec.compose( "%S/%S", gs.gstr(), &kmCmd.gstr()[indx] ) ;

         FileStats   fstats ;
         if ( (lstat64 (kmSpec.ustr(), &fstats )) == ZERO )
         {
            if ( (S_ISREG(fstats.st_mode)) && 
                 ((access ( kmSpec.ustr(), R_OK )) == ZERO) )
            {
               gs = &kmCmd.gstr()[indx] ;
               gs.copy( this->shm->keyMap.keyFile, MAX_FNAME ) ;
               trgExists = true ;
            }
            else              // specified file not a 'regular' file or no read access
               status = csERR ;
         }
         else                 // specified file does not exist
            status = csERR ;
      }
      else
         ; // no source file specified
   }
   else
      ; // incorrect command format, filename (if any) ignored

   //* If source file exists and user has read access, read the file.*
   if ( trgExists )
   {
      ifstream ifs ( kmSpec.ustr(), ifstream::in ) ;
      if ( ifs.is_open() )             // if input file open
      {
         wchar_t userText[128], dfltText[128], descText[128] ;
         wkeyCode dfltKey,             // default key assignment
                  userKey ;            // remapped key assignment
         short readStatus,             // status of line read
               recIndx = ZERO,         // index of record being processed
               kcIndx = ZERO,          // index of default record matching custom keycode
               cnt = ZERO ;            // number of items captured in record scan
         bool  done = false ;          // loop control

         //* Verify that this is actually a configuration file *
         if ( (readStatus = this->rcfReadSourceLine ( ifs, lineNum, gs )) >= ZERO )
         {
            if ( (gs.find( KeymapFileID, ZERO, false, 5 )) != ZERO )
            {
               status = csERR ;
               done = true ;
            }
         }

         while ( ! done )
         {
            if ( (readStatus = this->rcfReadSourceLine ( ifs, lineNum, gs )) > ZERO )
            {
               //* Parse the record and store the results in the keymap object.*
               cnt = swscanf ( gs.gstr(), L"%S | %S %S", userText, dfltText, descText ) ;

               //* If we have a potentially-valid record *
               if ( cnt == 3 )
               {
                  //* Is the default value for this record    *
                  //* assigned to a command?                  *
                  this->shm->keyMap.keydef[recIndx].cmd = ((gs.find( "unassigned" )) < ZERO) ;

                  //* If user-defined field is populated, AND *
                  //* if this record references a command     *
                  if ( (*userText != L'-') && this->shm->keyMap.keydef[recIndx].cmd )
                  {
                     //* Search the list of default items for a *
                     //* match with the user-defined item.      *
                     gs.compose( "%S ", userText ) ;
                     for ( kcIndx = ZERO ; kcIndx < KEYMAP_KEYS ; ++kcIndx )
                     {
                        if ( (gs.compare( KeymapCfgDesc[kcIndx], true, 
                                          (gs.gschars() - 1) )) == ZERO )
                        {  //* Copy user's keycode definition to storage *
                           this->shm->keyMap.keydef[recIndx].kuser = KeymapCfgDflt[kcIndx] ;
                           ++userValues ;
                           this->shm->keyMap.keydef[recIndx].map = true ;
                           break ;
                        }
                     }
                     //* If no matching record found, then syntax error.*
                     if ( kcIndx >= KEYMAP_KEYS )
                     {
                        status = csERR ;
                        break ;
                     }
                  }
                  else     // otherwise, this is an empty (or meaningless) user value
                  {
                     this->shm->keyMap.keydef[recIndx].kuser = nonCode ;
                     this->shm->keyMap.keydef[recIndx].map = false ;
                  }
               }
               else     // syntax error (malformed record)
               {
                  status = csERR ;
                  done = true ;
               }

               //* Record has been processed *
               if ( ++recIndx >= KEYMAP_KEYS )
                  done = true ;
            }
            else if ( readStatus == ZERO )
               { /* ignores comment or blank line */ }
            else     // (readStatus < ZERO) end-of-file
               done = true ;
         }

         ifs.close() ;                 // close the file
      }
   }

   //* If user has remapped one or more command *
   //* keys, fully initialize the keymap.       *
   //* Otherwise, keymap remains uninitialized. *
   if ( (status == csOK) && (userValues > ZERO) )
   {
      this->shm->keyMap.initialize( false, true ) ;

      //* Test for duplicated or misapplied command keys *
      short dup1, dup2 ;      // (unused in this context)
      if ( (status = rcfValidateKeymap ( dup1, dup2 )) != csOK )
         mappingError = true ;

   }
   if ( status != csOK )   // disable the keymap, default commands will be used
      this->shm->keyMap.reset() ;

   //* Write diagnostic messages *
   if ( this->shm->msgCount < smmMSG_COUNT )
   {
      gsv.clear() ;        // reset the buffer
      if ( status == csOK )
      {
         if ( this->shm->verbose )
         {
            if ( trgExists )
               gsv.compose( "          %hd user commands defined", &userValues ) ;
            else
               gsv = "          (using default key map)" ;
         }
      }
      else  // (status != csOK)
      {
         if ( ! trgExists )
            gsv = "          (keymap file not found, or access error)" ;
         else
         {
            if ( mappingError != false )
               gsv.compose( "          Syntax error (duplicate keycodes)!", &lineNum ) ;
            else
               gsv.compose( "          Syntax error (line:%hd)!", &lineNum ) ;
         }
      }
      if ( gsv.gschars() > 1 )
         gsv.copy ( this->shm->msg[this->shm->msgCount++], smmMSG_LEN ) ;
   }

   return status ;

}  //* End rcfLoadKeymap() *

//*************************
//*   rcfValidateKeymap   *
//*************************
//******************************************************************************
//* Scan the keycode map for conflicts or duplications.                        *
//* The individual keycodes have been validated at a higher level.             *
//*                                                                            *
//* Input  : dup1   : (by reference) receives index of duplicate (or -1)       *
//*          dup2   : (by reference) receives index of duplicate (or -1)       *
//*                                                                            *
//* Returns: member of enum ConfigStatus (csOK or csERR)                       *
//*          If csOK, then dup1 == dup2 == (-1).                               *
//*          If csERR, then dup1 and dup2 are indices of duplicated keycode.   *
//******************************************************************************
//* Notes:                                                                     *
//* A keycode which is assigned to multiple functions will cause the user      *
//* interface to to be unreliable. The problem is that a comprehensive test    *
//* for duplicates is quite labor intensive.                                   *
//* 1) User keycode definitions must be unique. That is, a keycode may be used *
//*    only once as a user-defined keycode.                                    *
//* 2) For a keycode to be used to remap a function, the value must either be  *
//*    unassigned by default, OR it original function must be masked.          *
//* 3) Note that neither the rcfLoadKeymap() method nor the ecoEditKM() method *
//*    will map a user-defined keycode to a record which is not associated     *
//*    with a command. That is, we will not see a user-defined keycode in an   *
//*    unassigned record because it was filtered out at a higher level.        *
//*                                                                            *
//* Programmer's Note:                                                         *
//* ==================                                                         *
//* This algorithm is GROSSLY inefficient. It makes potentially hundreds of    *
//* scans through the keycode map (> 200 records).                             *
//* -- The more user-defined keys, the more scans are required.                *
//* -- The most likely keycodes are at the top of the list (CTRL+n and ALT+n), *
//*    so a scan will seldom need to read the full list.                       *
//* -- It is unlikely that the user will re-define more that a few keycodes;   *
//*    we estimate ten(10) or fewer.                                           *
//*                                                                            *
//******************************************************************************

ConfigStatus FmConfig::rcfValidateKeymap ( short& dup1, short&dup2 )
{
   ConfigStatus status = csOK ;     // return value
   short ukeys[KEYMAP_KEYS] ;       // indices of user-defined keycodes
   short userKeys = ZERO ;          // number of user-defined keycodes
   wkeyCode wktest ;                // value under test

   dup1 = dup2 = -1 ;               // initialize caller's indices

   //* Count the number of user-defined keycodes, *
   //* and remember their indices.                *
   for ( short i = ZERO ; i < KEYMAP_KEYS ; ++i )
   {
      if ( this->shm->keyMap.keydef[i].map )
         ukeys[userKeys++] = i ;
   }

   //* Test for duplicate keycodes within the user definitions *
   if ( userKeys > 1 )
   {
      for ( short i = ZERO ; (i < (userKeys - 1)) && (status == csOK) ; ++i )
      {
         wktest = this->shm->keyMap.keydef[ukeys[i]].kuser ;
         for ( short j = i + 1 ; (j < userKeys) ; ++j )
         {
            if ( wktest == this->shm->keyMap.keydef[ukeys[j]].kuser )
            {
               status = csERR ;     // multiple instances of a user-defined keycode
               dup1 = ukeys[i] ;    // index of first instance
               dup2 = ukeys[j] ;    // index of second instance
               break ;
            }
         }
      }
   }

   //* For each user definition, the keycode must either be unassigned *
   //* by default, OR the default function must be masked.             *
   for ( short i = ZERO ; i < userKeys && (status == csOK) ; ++i )
   {
      wktest = this->shm->keyMap.keydef[ukeys[i]].kuser ;
      for ( short j = ZERO ; j < KEYMAP_KEYS ; ++j )
      {  //* Locate the default position for user's keycode *
         if ( wktest == this->shm->keyMap.keydef[j].kdflt )
         {  //* If the default position references a command, AND *
            //* if this command function is not masked by another *
            //* keycode, then it is a logical error.              *
            if ( (this->shm->keyMap.keydef[j].cmd) &&
                 (! this->shm->keyMap.keydef[j].map) )
               status = csERR ;
            else
               ;  // keycode is unassigned by default and free for reassignment
            break ;
         }
      }
   }

   return status ;

}  //* End rcfValidateKeymap() *

//*************************
//*     rcfLoadMounts     *
//*************************
//******************************************************************************
//* Read the section of configuration file containing 'Mount Targets'.         *
//* Caller has located and verified the header line for this data section.     *
//*                                                                            *
//* Input  : ifs     : handle to open input stream                             *
//*          lineNum : most-recently-read line of source file                  *
//*                                                                            *
//* Returns: member of enum ConfigStatus                                       *
//******************************************************************************

ConfigStatus FmConfig::rcfLoadMounts ( ifstream& ifs, short& lineNum )
{
   gString gs, gsv ;             // work buffers
   short readStatus ;            // status of line read
   ConfigStatus status = csOK ;  // return value
   bool  done = false ;          // loop control

   //* Discard any mount points that may have been sent from within *
   //* the application. These would most likely be duplicates.      *
   this->shm->mntCount = ZERO ;
   while ( ! done )
   {
      //* Read a source line *
      if ( (readStatus = this->rcfReadSourceLine ( ifs, lineNum, gs )) > ZERO )
      {
         //* Perform environment-variable substitution, if necessary.*
         this->rcfEnvExpansion ( gs ) ;

         if ( this->shm->mntCount < MAX_MNTCMDS )
            gs.copy( this->shm->mntCmd[this->shm->mntCount++], MAX_PATH ) ;
         else
            ; // silently ignore entries we don't have room for
      }
      else
      {  //* Ignore comments - list is terminated by a blank line (or EOF) *
         if ( (gs.gschars()) == 1 )
         {
            if ( this->shm->verbose && (this->shm->msgCount < smmMSG_COUNT) )
            {
               gsv.compose( L"          (%hd specified)", &this->shm->mntCount ) ;
               gsv.copy ( this->shm->msg[this->shm->msgCount++], smmMSG_LEN ) ;
            }
            //* Fill remainder of array with null strings *
            short i = this->shm->mntCount ;
            while ( i < MAX_MNTCMDS )
               *this->shm->mntCmd[i++] = NULLCHAR ;
            done = true ;
         }
      }
   }
   return status ;

}  //* End rcfLoadMounts() *

//*************************
//*    rcfLoadAltTrash    *
//*************************
//******************************************************************************
//* If specified, determine whether the non-default or non-standard location   *
//* for the desktop environment's Trashcan subdirectory and the two directories*
//* it should contain, actually exist and that the user has read/write access  *
//* to them, as follows:                                                       *
//*             specified_path/files  and  specified_path/info                 *
//*                                                                            *
//* NOTE: This specification is used ONLY if the default Trashcan (see below)  *
//*       was not found or is inaccessible. If, for instance, the user is      *
//*       'root', then the root user seldom has a true ${HOME} directory, and  *
//*       thus has no Home Trashcan. In such a case, an Alternate Trashcan     *
//*       path should be specified in the configuration file.                  *
//*       Else, FileMangler's move-to-trash command will not function.         *
//*                                                                            *
//* If neither the default Trashcan nor the Alternate Trashcan is accessible,  *
//* then we return a warning. Be aware that the FileMangler application will   *
//* RUN without a trashcan, but, since quantum physics and probability theory  *
//* are beyond the scope of this application, it will not allow the user to    *
//* send things to a non-existent trashcan.                                    *
//*                                                                            *
//*  - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -   *
//* The Default Trashcan                                                       *
//* --------------------                                                       *
//* In GNOME, KDE or any modern freedesktop.org environment the following      *
//* directories will contain the desktop's Trashcan.                           *
//*      ~/.local/share/Trash/files      (files sent to trash)                 *
//*      ~/.local/share/Trash/info       (info file for each file in trash)    *
//*                                                                            *
//* The existence and accessibility of these directories is tested in the      *
//* ReadConfigFile() method above. If these directories exist (or are          *
//* successfully created) and are accessible to the user, that path will have  *
//* been written to this->shm->cfgOpt.trPath[], and it will not be overwritten *
//* here.                                                                      *
//*                                                                            *
//* Input  : ifs     : handle to open input stream                             *
//*          lineNum : most-recently-read line of source file                  *
//*                                                                            *
//* Returns: member of enum ConfigStatus                                       *
//*          Note that failure to specify an Alternate Trashcan or specifying  *
//*          an invalid Alternate Trashcan path is not an error UNLESS the     *
//*          default Trashcan was not found.                                   *
//******************************************************************************

ConfigStatus FmConfig::rcfLoadAltTrash ( ifstream& ifs, short& lineNum )
{
   gString gs, gsv, gsRaw ;            // work buffers
   short readStatus ;                  // status of line read
   ConfigStatus status = csOK ;        // return value
   bool  validAltPath = false,         // 'true' if alt path validated
         done = false ;                // loop control
   *this->shm->trashPath = NULLCHAR ;  // assume no alt path specified

   while ( ! done )
   {
      //* Read a source line *
      if ( (readStatus = this->rcfReadSourceLine ( ifs, lineNum, gs )) > ZERO )
      {
         //* Save the un-expanded specification, then validate the real path.*
         gs.copy( this->shm->trashPath, MAX_PATH ) ;
         if ( (validAltPath = this->rcfValidateTrashPath ( gs )) != false )
         {
            //* If the default Trashcan path is invalid,     *
            //* make the Alternate Trashcan path the default.*
            if ( *this->shm->cfgOpt.trPath == NULLCHAR )
               gs.copy( this->shm->cfgOpt.trPath, MAX_PATH ) ; // (real path)
         }
         if ( this->shm->verbose && (this->shm->msgCount < smmMSG_COUNT) )
         {
            if ( validAltPath )
               gsv = "          (alternate trash path verified)" ;
            else
               gsv = "          (invalid alternate trash path)" ;
            gsv.copy ( this->shm->msg[this->shm->msgCount++], smmMSG_LEN ) ;
         }
         done = true ;
      }
      //* Ignore comments - a blank line (or EOF) indicates no path specified. *
      else
      {
         if ( ((readStatus == ZERO) && ((gs.gschars()) == 1)) || (readStatus < ZERO) )
         {
            if ( this->shm->verbose && (this->shm->msgCount < smmMSG_COUNT) )
            {
               gsv = "          (no alternate trash path)" ;
               gsv.copy ( this->shm->msg[this->shm->msgCount++], smmMSG_LEN ) ;
            }
            done = true ;
         }
      }
   }
   //* Warn user if neither default nor alternate trashcan path found *
   if ( *this->shm->cfgOpt.trPath == NULLCHAR )
      status = csALTRASH ;
   if ( (status != csOK) && (this->shm->ccCmd != ccREADSILENT) && 
        (this->shm->ccCmd != ccINTERACT) && (this->shm->msgCount < smmMSG_COUNT) )
   {
      gsv = "Warning! Trashcan directory not found." ;
      gsv.copy ( this->shm->msg[this->shm->msgCount++], smmMSG_LEN ) ;
   }
   return status ;

}  //* rcfLoadAltTrash() *

//*************************
//* rcfValidateTrashPath  *
//*************************
//******************************************************************************
//* Verify:                                                                    *
//* 1) that the specified base directory for the desktop trashcan exists       *
//*    and that user has read/write access to it.                              *
//* 2) that the two necessary subdirectories 'files' and 'info' exist within   *
//*    the base directory and that user has read/write access to both.         *
//*    If the necessary subdirectories do not exist, try to create them.       *
//*      (we do not attempt to create a missing 'Trash' base directory)        *
//*                                                                            *
//* Input  : tcPath  : base path for trashcan                                  *
//*                                                                            *
//* Returns: 'true' if target directory AND subdirectories exist               *
//*                 AND if user has read/write access to them                  *
//*          'false' otherwise                                                 *
//******************************************************************************

bool FmConfig::rcfValidateTrashPath ( gString& tcPath )
{
   bool success = false ;           // return value

   //* Expand any environment variables and get the real path spec.*
   this->rcfEnvExpansion ( tcPath, true ) ;
   char realPath[MAX_PATH] ;        // decoded full path
   if ( (realpath ( tcPath.ustr(), realPath )) != NULL )
   {
      gString rp( realPath ) ;
      bool  rdAccess, wrAccess,
            rpExist = this->vtpTargetDirExists ( rp, rdAccess, wrAccess ) ;
      if ( rpExist && rdAccess &&  wrAccess )
      {  //* Base directory is accessible *
         tcPath = rp ;

         //* Now test subdirectories under base directory *
         //* and if they are missing, create them.        *
         gString gsSubf, gsSubi ;
         bool gsSubfExists, gsSubiExists, 
              gsSubfRead, gsSubfWrite, gsSubiRead, gsSubiWrite ;
         gsSubf.compose( L"%S/%s", tcPath.gstr(), cdtpFiles ) ;
         if ( !(gsSubfExists = 
                this->vtpTargetDirExists ( gsSubf, gsSubfRead, gsSubfWrite )) )
         {
            gsSubfExists = this->vtpCreateDir ( gsSubf, gsSubfRead, gsSubfWrite ) ;
         }
         gsSubi.compose( L"%S/%s", tcPath.gstr(), cdtpInfo ) ;
         if ( !(gsSubiExists = 
                this->vtpTargetDirExists ( gsSubi, gsSubiRead, gsSubiWrite )) )
         {
            gsSubiExists = this->vtpCreateDir ( gsSubi, gsSubiRead, gsSubiWrite ) ;
         }
         if ( gsSubfExists && gsSubfRead && gsSubfWrite &&
              gsSubiExists && gsSubiRead && gsSubiWrite )
         {
            success = true ;
         }
      }
   }
   return success ;

}  //* End rcfValidateTrashPath() *

//*************************
//*  vtpTargetDirExists   *
//*************************
//******************************************************************************
//* Determine whether specified path specifies an existing file.               *
//* If it exists, determine also user's read/write access to target AND        *
//* whether the specified file is a directory-type file.                       *
//*                                                                            *
//* Input  : tPath  : full path specification for target to be tested          *
//*          rdAcc  : (by reference, initial state ignored)                    *
//*                   on return: if target exists AND user has read access,    *
//*                   set as 'true', else 'false'                              *
//*          wrAcc  : (by reference, initial state ignored)                    *
//*                   on return: if target exists AND user has write access,   *
//*                   set as 'true', else 'false'                              *
//*                                                                            *
//* Returns: 'true'  if target exists AND is a directory-type file             *
//*           else 'false'                                                     *
//******************************************************************************

bool FmConfig::vtpTargetDirExists ( const gString& tPath, bool& rdAcc, bool& wrAcc )
{
   bool  tExist = this->vtpTargetExists ( tPath, rdAcc, wrAcc ) ;
   if ( tExist )
   {
      FileStats   fstats ;
      int lsstat = lstat64 ( tPath.ustr(), &fstats ) ;
      if ( lsstat == ZERO )         // target (still) exists
      {
         if ( (S_ISDIR(fstats.st_mode)) == false )
            tExist = false ;     // not a directory name
      }
   }
   return tExist ;

}  //* End vtpTargetDirExists() *

//*************************
//*   vtpTargetExists     *
//*************************
//******************************************************************************
//* Determine whether specified path specifies an existing file.               *
//* If it exists, determine also user's read/write access to target.           *
//*                                                                            *
//* Input  : tPath  : full path specification for target to be tested          *
//*          rdAcc  : (by reference, initial state ignored)                    *
//*                   on return: if target exists AND user has read access,    *
//*                   set as 'true', else 'false'                              *
//*          wrAcc  : (by reference, initial state ignored)                    *
//*                   on return: if target exists AND user has write access,   *
//*                   set as 'true', else 'false'                              *
//*                                                                            *
//* Returns: 'true'  if target exists, else 'false'                            *
//******************************************************************************

bool FmConfig::vtpTargetExists ( const gString& tPath, bool& rdAcc, bool& wrAcc )
{
   bool  tExist = false ;           // return value
   rdAcc = wrAcc = false ;          // initialize caller's flags

   if ( (access ( tPath.ustr(), F_OK )) == ZERO )
   {
      tExist = true ;
      rdAcc = bool((access ( tPath.ustr(), R_OK )) == ZERO) ;
      wrAcc = bool((access ( tPath.ustr(), W_OK )) == ZERO) ;
   }

   return tExist ;

}  //* End vtpTargetExists() *

//*************************
//*     vtpCreateDir      *
//*************************
//******************************************************************************
//* Create a directory. Set it's permission bits to read AND write.            *
//*                                                                            *
//* Input  : tPath  : full path specification for target to be tested          *
//*          rdAcc  : (by reference, initial state ignored)                    *
//*                   on return: if target exists AND user has read access,    *
//*                   set as 'true', else 'false'                              *
//*          wrAcc  : (by reference, initial state ignored)                    *
//*                   on return: if target exists AND user has write access,   *
//*                   set as 'true', else 'false'                              *
//*                                                                            *
//* Returns: 'true'  if target now exists, else 'false'                        *
//******************************************************************************
//* Assumes that target does not already exist. If a pre-existing target, this *
//* method will return false information. Be Aware!                            *
//******************************************************************************

bool FmConfig::vtpCreateDir ( const gString& tPath, bool& rdAcc, bool& wrAcc )
{
   bool  tExist = false ;           // return value
   rdAcc = wrAcc = false ;          // initialize caller's flags

   if ( (mkdir ( tPath.ustr(), dirMode )) == ZERO )
   {
      tExist = true ;
      this->vtpEnablePermissions ( tPath, rdAcc, wrAcc ) ;
   }
   return tExist ;

}  //* End vtpCreateDir() *

//*************************
//* vtpEnablePermissions  *
//*************************
//******************************************************************************
//* Set permissions for the specified file (directory).                        *
//*                                                                            *
//* We use the same philosophy as the main FileMangler application in choosing *
//* permission bits for the temporary storage directories. That is, we set the *
//* same permission pattern as does the 'mkdir' command-line utility:          *
//* rwxrwxr-x meaning full permissions for owner and group,  with read/execute *
//* for 'others'.                                                              *
//*                                                                            *
//* Input  : tPath  : full path specification for target to be tested          *
//*          rdAcc  : (by reference, initial state ignored)                    *
//*                   on return: if target exists AND user has read access,    *
//*                   set as 'true', else 'false'                              *
//*          wrAcc  : (by reference, initial state ignored)                    *
//*                   on return: if target exists AND user has write access,   *
//*                   set as 'true', else 'false'                              *
//*                                                                            *
//* Returns: 'true'  if operation successful, else 'false'                     *
//*                  Caller's read and write access flags reflect reality.     *
//******************************************************************************

bool FmConfig::vtpEnablePermissions ( const gString& tPath, bool& rdAcc, bool& wrAcc )
{
   FileStats   fstats ;
   bool  success = false ;

   //* Get current permissions *
   if ( (lstat64 ( tPath.ustr(), &fstats )) == ZERO )
   {  //* Set new permissions *
      fstats.st_mode |= dirMode ;
      chmod ( tPath.ustr(), fstats.st_mode ) ;
   }
   //* Adjust caller's permission flags *
   if ( (this->vtpTargetExists ( tPath, rdAcc, wrAcc )) && rdAcc && wrAcc )
      success = true ;

   return success ;

}  //* End vtpEnablePermissions() *

//*************************
//*   rcfReadSourceLine   *
//*************************
//******************************************************************************
//* Read and verify a line of data from the source file.                       *
//*                                                                            *
//* Input  : ifs     : handle to open input stream                             *
//*          lineNum : most-recently-read line of source file                  *
//*          gs      : (by reference, initial contents ignored)                *
//*                    receives input data                                     *
//*                                                                            *
//* Returns: > 0  if successful read of a line containing non-comment data     *
//*          ==0  if successful read of comment OR an empty line               *
//*          < 0  if end-of-file (or read error)                               *
//******************************************************************************

short FmConfig::rcfReadSourceLine ( ifstream& ifs, short& lineNum, gString& gs )
{
   char  lineData[gsDFLTBYTES] ;    // raw UTF-8 input
   short readStatus = (-1) ;        // return value
   gs.clear() ;                     // initialize caller's buffer

   ifs.getline ( lineData, gsDFLTBYTES, NEWLINE ) ;

   if ( ifs.good() || ifs.gcount() > ZERO )
   {
      ++lineNum ;             // line counter
      readStatus = 1 ;        // assume non-empty line

      //* Skip leading whitespace and convert to 'wide' characters *
      short i = ZERO ;
      while ( lineData[i] == nckSPACE || lineData[i] == nckTAB )
         ++i ;
      gs = &lineData[i] ;     // result to caller's buffer

      //* Comment or empty line? *
      if ( ((gs.gschars()) == 1) || ((gs.find( L"//" )) == ZERO) )
         readStatus = ZERO ;
   }
   return readStatus ;

}  //* End rcfReadSourceLine() *

//*************************
//*   rcfEnvExpansion     *
//*************************
//******************************************************************************
//* Perform environment-variable expansion or tilde ('~') expansion on the     *
//* specified string.                                                          *
//*                                                                            *
//*   NOTE: When running in interactive-configuration mode, we do not expand   *
//*         paths because we want the user to see what is actually read from   *
//*         the source configuration file.                                     *
//*         Paths are expanded when verifying user input and of course during  *
//*         runtime.                                                           *
//*                                                                            *
//* Input  : gsPath  : (by reference) contains the string to be scanned        *
//*          force   : (optional, 'false' by default)                          *
//*                    if 'true', force expansion of environment variables     *
//*                                                                            *
//* Returns: 'true'  if expansion successful (or no expansion needed)          *
//*          'false' if unable to expand                                       *
//******************************************************************************
//* Notes on 'wordexp':                                                        *
//* The 'wordexp' function is a pretty cool, but watch out:                    *
//*  a) wordexp returns ZERO on success or WRDE_BADCHAR (2) if an invalid      *
//*     character is detected in the stream.                                   *
//*     - Note that an empty string will pass the scan, but then               *
//*       'wexp.we_wordc' will be ZERO.                                        *
//*  b) Dynamic memory allocation happens, so remember to free it.             *
//*     - If a bad character in the stream, then freeing the dynamic           *
//*       allocation will cause a segmentation fault. This is a Standard       *
//*       Library bug, so the work-around is to call 'wordfree' ONLY if        *
//*      'wordexp' returns success.                                            *
//*  c) SPACE characters delimit the parsing, so if the path contains spaces,  *
//*     then we must concatenate the resulting substrings, reinserting the     *
//*     space characters. Leading and trailing spaces are ignored.             *
//*     (We assume that a path will never contain a TAB character.)            *
//*  d) wordexp will choke on the following characters in the stream:          *
//*             & | ; < >  \n     (unless they are quoted)                     *
//*     - Parentheses and braces should appear ONLY as part of a token to be   *
//*       expanded (or if they are quoted).                                    *
//*  e) The tokens we are most likely to see are '${HOME}' and '~'.            *
//*     These are both expanded as '/home/sam' or the equivalent.              *
//******************************************************************************

bool FmConfig::rcfEnvExpansion ( gString& gsPath, bool force )
{
   bool status = true ;          // return value

   if ( (this->shm->ccCmd != ccINTERACT) || (force != false) )
   {
      wordexp_t wexp ;              // target structure
      if ( (wordexp ( gsPath.ustr(), &wexp, ZERO )) == ZERO )
      {
         if ( wexp.we_wordc > ZERO )   // if we have at least one element
         {
            gsPath.clear() ;
            for ( UINT i = ZERO ; i < wexp.we_wordc ; )
            {
               gsPath.append( wexp.we_wordv[i++] ) ;
               if ( i < wexp.we_wordc )      // re-insert stripped spaces (see note)
                  gsPath.append( L' ' ) ;
            }
         }
         wordfree ( &wexp ) ;
      }
      else
         status = false ;
   }
   return status ;

}  //* End rcfEnvExpansion() *

//*************************
//*    CreateTempname     *
//*************************
//******************************************************************************
//* Create a unique path/filename for a temporary file within the previously   *
//* established application temporary directory. (See CreateTemppath())        *
//*                                                                            *
//* Input  : tmpPath: (by reference) receives the path/filename                *
//*                                                                            *
//* Returns: 'true'  if path/filename created successfully                     *
//*          'false' if library call failed                                    *
//******************************************************************************

bool FmConfig::CreateTempname ( gString& tmpPath )
{
   bool  status = false ;

   char tn[gsDFLTBYTES] ;
   gString gstmp( "%s/FMG_XXXXXX", this->shm->cfgOpt.tfPath ) ;
   gstmp.copy( tn, gsDFLTBYTES ) ;
   int descriptor ;
   if ( (descriptor = mkstemp ( tn )) != (-1) )
   {
      close ( descriptor ) ;     // close the file
      tmpPath = tn ;             // copy target path to caller's buffer
      status = true ;            // declare success
   }
   else
      tmpPath.clear() ;
   return status ;

}  //* End CreateTempname() *

//*************************
//*    WriteConfigFile    *
//*************************
//******************************************************************************
//* Write the data in the shared-memory object to the specified configuration  *
//* file.                                                                      *
//* - If target exists:                                                        *
//*   a. rename the existing file as a backup file                             *
//*   b. open the original file for reading                                    *
//*   c. open the target file for writing                                      *
//*   d. copy the comment data from original file, inserting the command data  *
//*      from the shared-memory object.                                        *
//* - If target DOES NOT exist, call CreateConfigFile ( populate==true ).      *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: member of enum ConfigStatus                                       *
//******************************************************************************
//* Programmer's Note: If a source config file exists, we will transfer its    *
//* comment data to the new target. Note, however, that if the source file is  *
//* missing commands, the target file will not have those commands either.     *
//* This is the nature of trusting the user to do things correctly.            *
//* We do, however, return a syntax error if not all commands found in source. *
//*                                                                            *
//* Major Note: The 'rename' function as used here, silently overwrites any    *
//* existing 'backup' file ( configfilename~ ) when making a backup copy of    *
//* the configuration. This is actually what we WANT to happen, but it is a    *
//* bit cavalier to trash any file without warning.                            *
//*                                                                            *
//* Minor Note: If a source config file exists, we strip any leading whitespace*
//* from each line before writing it to the target file. This does not affect  *
//* functionality, but it may surprise the user - but it is not a bug...       *
//******************************************************************************

ConfigStatus FmConfig::WriteConfigFile ( void )
{
   gString gsOut ;
   ConfigStatus trgStatus = csOK,   // status of target-file write
                status = csERR ;    // return value

   //* Check whether target exists, and *
   //* if it does, rename it as backup. *
   bool  backupExists = false, 
         targetExists = false ;
   gString trgPath( this->shm->cfgPath ),
           backupPath ;
   backupPath.compose( L"%S~", trgPath.gstr() ) ;
   FileStats   fstats ;
   int lsstat = lstat64 ( trgPath.ustr(), &fstats ) ;
   if ( lsstat == ZERO )
      targetExists = true ;
   else
   {
      if ( errno != ENOENT )
         targetExists = true ;
   }
   if ( targetExists )
   {
      if ( (rename ( trgPath.ustr(), backupPath.ustr() )) == ZERO )
         { targetExists = false ; backupExists = true ; }
   }

   //* If backup successful, write the data to new target file *
   if ( backupExists && !targetExists )
   {
      //* Open the source file and target file *
      ifstream ifs ( backupPath.ustr(), ifstream::in ) ;
      ofstream ofs( trgPath.ustr(), ofstream::out | ofstream::trunc ) ;
      if ( ifs.is_open() && ofs.is_open() )
      {
         char  lineData[gsDFLTBYTES] ; // raw UTF-8 input
         const wchar_t *wptr ;         // pointer to work buffer
         gString gs, gsv ;             // work buffers
         short readStatus,             // status of line read
               syntaxErrors = ZERO,    // count the syntax error found
               commandsRead = ZERO,    // count of commands read from source file
               ccIndex,                // index into command array
               aaIndex,                // index into argument array
               lineNum = ZERO ;        // source-line counter
         bool  endCmds = false,        // true if 'EndCfgCommands' encountered
               done = false ;          // loop control
   
         //* Verify that this is actually a configuration file *
         ifs.getline ( lineData, gsDFLTBYTES, NEWLINE ) ;
         if ( ifs.good() && (!(strncmp ( lineData, ConfigFileID, 5 ))) )
         {
            ofs << lineData << endl ;  // copy ID line to target
            ++lineNum ;
         }
         else
         {  //* Source file is not a valid configuration file, so  *
            //* ignore it and create a new file using our data set.*
            status = this->CreateConfigFile ( true ) ;
            done = true ;
         }
         while ( ! done )
         {  //* If a comment line, a blank line or a command *
            if ( (readStatus = this->rcfReadSourceLine ( ifs, lineNum, gs )) >= ZERO )
            {
               if ( readStatus > ZERO )      // source line contains a command
               {
                  //* Validate the command *
                  wptr = gs.gstr() ;         // point to work buffer
                  for ( ccIndex = ZERO ; ccIndex < cciArgs ; ccIndex++ )
                  {
                     if ( (wcsncmp ( wptr, CfgParm[ccIndex].str, CfgParm[ccIndex].len )) == ZERO )
                     {
                        ++commandsRead ;
                        break ;
                     }
                  }

                  //* If command parameter string found.*
                  if ( ccIndex < cciArgs )
                  {
                     switch ( ccIndex )
                     {
                        case cciWM:    // 'WindowMode'
                           gsv.compose( L" = %S", 
                              CfgParm[ccIndex].args[this->shm->cfgOpt.winMode].str ) ;
                           break ;
                        case cciSO:    // 'SortOrder'
                           gsv.compose( L" = %S", 
                              CfgParm[ccIndex].args[this->shm->cfgOpt.sortOption].str ) ;
                           break ;
                        case cciCS:    // 'CaseSensitive'
                           gsv.compose( L" = %S", 
                              CfgParm[ccIndex].args[this->shm->cfgOpt.caseSensitive ? 0 : 1].str ) ;
                           break ;
                        case cciHF:    // 'HiddenFiles'
                           gsv.compose( L" = %S", 
                              CfgParm[ccIndex].args[this->shm->cfgOpt.showHidden ? 0 : 1].str ) ;
                           break ;
                        case cciCD:    // 'ConfirmDelete'
                           gsv.compose( L" = %S", 
                              CfgParm[ccIndex].args[this->shm->cfgOpt.confirmDelete ? 0 : 1].str ) ;
                           break ;
                        case cciCO:    // 'ConfirmOverwrite'
                           gsv.compose( L" = %S", 
                              CfgParm[ccIndex].args[this->shm->cfgOpt.overWrite].str ) ;
                           break ;
                        case cciLM:    // 'LockMenuBar'
                           gsv.compose( L" = %S", 
                              CfgParm[ccIndex].args[this->shm->lockMb ? 0 : 1].str ) ;
                           break ;
                        case cciSC:
                           switch ( this->shm->cfgOpt.cScheme.scheme )
                           {
                              case ncbcBK:      aaIndex = 1 ;  break ;
                              case ncbcRE:      aaIndex = 2 ;  break ;
                              case ncbcGR:      aaIndex = 3 ;  break ;
                              case ncbcBR:      aaIndex = 4 ;  break ;
                              case ncbcBL:      aaIndex = 5 ;  break ;
                              case ncbcMA:      aaIndex = 6 ;  break ;
                              case ncbcCY:      aaIndex = 7 ;  break ;
                              case ncbcGY:      aaIndex = 8 ;  break ;
                              case ncbcDEFAULT: aaIndex = 9 ;  break ;  // terminal dflt
                              case ncbcCOLORS:                          // application default
                              default:          aaIndex = 0 ;  break ;
                           }
                           gsv.compose( L" = %S", scArgs[aaIndex].str ) ;
                           break ;
                        case cciEM:
                           gsv.compose( L" = %S", 
                              CfgParm[ccIndex].args[this->shm->cfgOpt.enableMouse ? 0 : 1].str ) ;
                           break ;
                        case cciEP:          // "ExternalPrograms"
                           gsv.compose( L" = %S", 
                              CfgParm[ccIndex].args[this->shm->cfgOpt.lCode].str ) ;
                           break ;
                        case cciEC:    // 'EndCfgCommands'
                           endCmds = true ;
                           gsv.compose( L" = %S", 
                              CfgParm[ccIndex].args[0].str ) ;
                           break ;
                        default:       // (unlikely)
                           gsv = L"" ;
                           break ;
                     }
                     gs.compose( L"%S%S", CfgParm[ccIndex].str, gsv.gstr() ) ;
                     ofs << gs.ustr() << endl ;
                  }
                  //* Command is not in the 'CfgParm' array. If all      *
                  //* fixed-arg commands completed, scan 'CfgList' array.*
                  else if ( endCmds )
                  {
                     bool cmdfound = false ;
                     ConfigStatus wStat = csOK ;
                     for ( ccIndex = ZERO ; ccIndex < cliArgs ; ccIndex++ )
                     {
                        if ( (wcsncmp ( wptr, CfgList[ccIndex].str, CfgList[ccIndex].len )) == ZERO )
                        {  //* Store the free-form arguments *
                           ++commandsRead ;
                           cmdfound = true ;
                           switch ( ccIndex )
                           {
                              case cliFD: // 'Favorite Directories' header
                                 ofs << gs.ustr() << endl ;
                                 wStat = this->wcfWriteFavorites ( ofs, ifs, lineNum ) ;
                                 break ;
                              case cliKM: // 'KeyMap' command
                                 wStat = this->wcfWriteKeyMap ( ofs, gs ) ;
                                 break ;
                              case cliMC: // 'Mount Commands' header
                                 ofs << gs.ustr() << endl ;
                                 wStat = this->wcfWriteMounts ( ofs, ifs, lineNum ) ;
                                 break ;
                              case cliAT: // 'Alternate Trashcan' header
                                 ofs << gs.ustr() << endl ;
                                 wStat = this->wcfWriteAltTrash ( ofs, ifs, lineNum ) ;
                                 break ;
                              case cliEF: // 'End Config File' marker
                                 //* Any data following this marker will be   *
                                 //* copied to target but its content ignored.*
                                 ofs << gs.ustr() << "\n" << endl ; // copy the marker
                                 this->wcfWriteTrailingData ( ofs, ifs, lineNum ) ;

                                 //* Determine status to be returned to caller.*
                                 if ( trgStatus == csOK && syntaxErrors == ZERO && 
                                      (commandsRead == (cciArgs + cliArgs)) )
                                    status = csOK ;
                                 else if ( trgStatus != csOK )
                                    status = trgStatus ;
                                 else
                                    status = csSYNTAX ;
                                 done = true ;
                                 break ;
                              default:    // (unlikely)
                                 wStat = csSYNTAX ;
                                 break ;
                           }
                           if ( wStat != csOK )
                              trgStatus = wStat ;
                           break ;
                        }
                     }
                     if ( ! cmdfound ) // syntax error in source file
                        ++syntaxErrors ;
                  }
                  else                 // syntax error in source file
                     ++syntaxErrors ;
               }
               else  // source line is a comment or blank, 
               {     // copy it to target, unchanged
                  ofs << gs.ustr() << endl ;
               }
            }
            else     // (readStatus < ZERO) end-of-file
            {
               //* If we reach this point, the 'EndConfigFile' *
               //* marker was not found in the source file.    *
               status = csSYNTAX ;
               done = true ;
            }
         }
      }
      //* Close the files *
      if ( ifs.is_open() )
         ifs.close() ;
      if ( ofs.is_open() )
         ofs.close() ;
   }
   //* Target does not exist, create a new  *
   //* config file containing our data set. *
   else if ( ! targetExists )
   {
      status = this->CreateConfigFile ( true ) ;
   }
   //* Else, target file exists but is not accessible *
   else
      status = csREADONLY ;

   return status ;

}  //* End WriteConfigFile() *

//*************************
//*   wcfWriteFavorites   *
//*************************
//******************************************************************************
//* Copy the list of 'favorite directories' to the target configuration file.  *
//*                                                                            *
//* Input  : ofs    : open output stream (by reference)                        *
//*        : ifs    : open input stream (by reference)                         *
//*          lineNum: source-file line count (by reference)                    *
//*                                                                            *
//* Returns: member of enum ConfigStatus                                       *
//******************************************************************************

ConfigStatus FmConfig::wcfWriteFavorites ( ofstream& ofs, ifstream& ifs, short& lineNum )
{
   ConfigStatus status = csOK ;     // return value

   //* Write favorites list to target file *
   status = wcfWriteFavorites ( ofs ) ;

   //* Step over favorites list (if any) in source file.*
   //* If comment found, copy it to target file.        *
   //* An empty line (gs.gschars() == 1) signals the    *
   //* end of the list.                                 *
   gString gs ;
   short stepStat ;
   do
   {
      stepStat = this->rcfReadSourceLine ( ifs, lineNum, gs ) ;
      if ( (stepStat == ZERO) && (gs.gschars() > 1) )
      {
         ofs << gs.ustr() << endl ;
         continue ;
      }
   }
   while ( stepStat > ZERO ) ;
   if ( stepStat < ZERO )     // if premature end-of-file
      status = csSYNTAX ;

   return status ;

}  //* End wcfWriteFavorites() *

//*************************
//*   wcfWriteFavorites   *
//*************************
//******************************************************************************
//* Copy the list of 'favorite directories' to the target configuration file.  *
//*                                                                            *
//* Input  : ofs    : open output stream (by reference)                        *
//*                                                                            *
//* Returns: member of enum ConfigStatus                                       *
//******************************************************************************

ConfigStatus FmConfig::wcfWriteFavorites ( ofstream& ofs )
{
   //* Write favorites list to target file *
   for ( short i = ZERO ; i < this->shm->favCount ; i++ )
      ofs << this->shm->favPath[i] << endl ;
   ofs << endl ;     // empty line signals end of list

   return csOK ;

}  //* End wcfWriteFavorites() *

//*************************
//*    wcfWriteKeyMap     *
//*************************
//******************************************************************************
//* If a key-map file has been specified, OR if one or more command keys have  *
//* have been set to non-default values, write data to a key-map file and      *
//* append the name of the key-map file to the command string.                 *
//*                                                                            *
//* Input  : cfgofs  : (by reference) open output stream for main config file  *
//*          gs      : (by reference) command with possible key-map filename   *
//*                     - if key-map file specified, write data to it          *
//*                     - if no key-map file specified, then data will be      *
//*                       written to the default keymap file.                  *
//*                                                                            *
//* Returns: member of enum ConfigStatus                                       *
//******************************************************************************
//* When and with what data do we update (or create) the keymap config file?   *
//*                                                                            *
//* 1) During read of the config file:                                         *
//*    a) If a keymap file was specified, the shared-memory keyMap will have   *
//*       been initialized, either with the specified data or if errors in the *
//*       source file, with default data.                                      *
//*    b) If no keymap file was specified, then the keymap is was not          *
//*       initialized during config read.                                      *
//* 2) If the keymap was edited by the user, even if no changes were made, the *
//*    keyMap will have been initialized during the edit if it hadn't been     *
//*    initialized earlier.                                                    *
//* 3) If a keymap file was not specified, AND if user has not edited the      *
//*    keymap, then we will arrive here with the keyMap member still           *
//*    uninitialized, so initialize it here.                                   *
//* 4) Write the keymap, either to the specified file, or if no filename       *
//*    specified, to the default keymap file.                                  *
//* 5) Note that on entry, the data are either initialized for display in a    *
//*    dialog or unitialized. To write the data to a file, we need to reformat *
//*    the data for output to a file. There are two possibilities for this:    *
//*    a) Reformat the shared-memory 'keyMap' for file output and the          *
//*      RE-reformat for display in the main application's dialog.             *
//*    b) Make a COPY of the shared-memory keymap and format the data for      *
//*       output to a file. Note that the RemapKey::operator= method copies    *
//*       ALL the data; however, for our purpose, we need only the user-defined*
//*       keycodes from the shared-memory object.                              *
//*    Clearly, the second option is more efficient overall.                   *
//* 6) Logistics:                                                              *
//*    If user has added comments to the file, we want to retain them; however,*
//*    logistically, this is difficult. Therefore we write a pristine file:    *
//*    a) the boilerplate header                                               *
//*    b) the default keycode data, and                                        *
//*    c) user-defined keycode data from our shared-memory member.             *
//******************************************************************************

ConfigStatus FmConfig::wcfWriteKeyMap ( ofstream& cfgofs, gString& kmCmd )
{
   ConfigStatus status = csOK ;        // return value
   gString kmSpec,                     // keymap filespec
           kmFname( kmcfgName ),       // keymap filename
           gs ;                        // text formatting
   short indx = kmCmd.find( L'=' ) ;   // command parsing index
   bool  trgSpecified = false ;        // if target filename specified

   //* Construct the filespec for source file and verify its existence.*
   if ( indx >= ZERO )
   {
      indx = kmCmd.scan( indx + 1 ) ;
      if ( kmCmd.gstr()[indx] != NULLCHAR )  // filename specified
      {
         kmFname = &kmCmd.gstr()[indx] ;
         trgSpecified = true ;
      }
   }
   kmFname.copy( this->shm->keyMap.keyFile, MAX_FNAME ) ;
   gs = this->shm->cfgPath ;
   gs.limitChars( gs.findlast( L'/' ) ) ;
   kmSpec.compose( "%S/%S", gs.gstr(), kmFname.gstr() ) ;

   //* Write the 'KeyMap' command to the main config file.      *
   //* If 'kmCmd' specifies a filename, write the command as-is.*
   //* If 'kmCmd' does not specify a filename, BUT user-defined *
   //*  keycodes have been selected, then write the command     *
   //*  including the default filename.                         *
   //* Otherwise write a clean command string WITHOUT filename. *
   if ( ! trgSpecified )
   {
      kmCmd.compose( "%S = ", CfgList[cliKM].str ) ;
      if ( this->shm->keyMap.kmActive )
         kmCmd.append( kmFname.gstr() ) ;
   }
   cfgofs << kmCmd.ustr() << endl ;

   //* If the keymap has not been initialized, do it now.*
   //*    (formatted for display, using default data)    *
   if ( ! this->shm->keyMap.kmInit )
      this->shm->keyMap.initialize() ;

   RemapKey* km = new RemapKey ;       // temporary data object

   //* Format the data for output to a file.*
   if ( ! this->shm->keyMap.kmActive ) // if no user-defined keycodes
      km->initialize( true ) ;         // format for file output (default data only)
   else                                // one or more user-defined keycodes
   {
      for ( short i = ZERO ; i < KEYMAP_KEYS ; ++i )
         km->keydef[i].kuser = this->shm->keyMap.keydef[i].kuser ;
      km->initialize( true, true ) ;
   }

   //* Create the output file. Existing file, *
   //* if any, will be renamed as backup.     *
   status = this->CreateKeymapFile ( this->shm->keyMap.kmActive, kmFname.ustr() ) ;

   delete km ;                         // delete the temporary object

   return status ;

}  //* End wcfWriteKeyMap() *

//*************************
//*    wcfWriteMounts     *
//*************************
//******************************************************************************
//* Copy the list of 'mount commands' to the target configuration file.        *
//*                                                                            *
//* Input  : ofs    : open output stream (by reference)                        *
//*        : ifs    : open input stream (by reference)                         *
//*          lineNum: source-file line count (by reference)                    *
//*                                                                            *
//* Returns: member of enum ConfigStatus                                       *
//******************************************************************************

ConfigStatus FmConfig::wcfWriteMounts ( ofstream& ofs, ifstream& ifs, short& lineNum )
{
   ConfigStatus status = csOK ;     // return value

   //* Write mount-point list to target file *
   status = wcfWriteMounts ( ofs ) ;

   //* Step over mounts list (if any) in source file.   *
   //* If comment found, copy it to target file.        *
   //* An empty line (gs.gschars() == 1) signals the    *
   //* end of the list.                                 *
   gString gs ;
   short stepStat ;
   do
   {
      stepStat = this->rcfReadSourceLine ( ifs, lineNum, gs ) ;
      if ( (stepStat == ZERO) && (gs.gschars() > 1) )
      {
         ofs << gs.ustr() << endl ;
         continue ;
      }
   }
   while ( stepStat > ZERO ) ;
   if ( stepStat < ZERO )     // if premature end-of-file
      status = csSYNTAX ;

   return status ;

}  //* End wcfWriteMounts() *

//*************************
//*    wcfWriteMounts     *
//*************************
//******************************************************************************
//* Copy the list of 'mount commands' to the target configuration file.        *
//*                                                                            *
//* Input  : ofs    : open output stream (by reference)                        *
//*                                                                            *
//* Returns: member of enum ConfigStatus                                       *
//******************************************************************************

ConfigStatus FmConfig::wcfWriteMounts ( ofstream& ofs )
{
   //* Write mount-point list to target file *
   for ( short i = ZERO ; i < this->shm->mntCount ; i++ )
      ofs << this->shm->mntCmd[i] << endl ;
   ofs << endl ;     // empty line signals end of list

   return csOK ;

}  //* End wcfWriteMounts() *

//*************************
//*   wcfWriteAltTrash    *
//*************************
//******************************************************************************
//* Copy the base path for 'alternate trashcan' directory to the target        *
//* configuration file.                                                        *
//*                                                                            *
//* Input  : ofs    : open output stream (by reference)                        *
//*        : ifs    : open input stream (by reference)                         *
//*          lineNum: source-file line count (by reference)                    *
//*                                                                            *
//* Returns: member of enum ConfigStatus                                       *
//******************************************************************************

ConfigStatus FmConfig::wcfWriteAltTrash ( ofstream& ofs, ifstream& ifs, short& lineNum )
{
   ConfigStatus status = csOK ;     // return value

   //* Write alt-trash path to target file *
   status = wcfWriteAltTrash ( ofs ) ;

   //* Step over existing entry in source file.     *
   //* If comments found before entry, copy them to *
   //* target file. Discard extra empty lines       *
   //* (gs.gschars() == 1) for beauty.              *
   gString gs ;
   short stepStat ;
   do
   {
      stepStat = this->rcfReadSourceLine ( ifs, lineNum, gs ) ;
      if ( stepStat == ZERO && gs.gschars() > 1 )
         ofs << gs.ustr() << endl ;
   }
   while ( stepStat == ZERO ) ;
   if ( stepStat < ZERO )     // if premature end-of-file
      status = csSYNTAX ;

   return status ;

}  //* End wcfWriteAltTrash() *

//*************************
//*   wcfWriteAltTrash    *
//*************************
//******************************************************************************
//* Copy the base path for 'alternate trashcan' directory to the target        *
//* configuration file.                                                        *
//*                                                                            *
//* Input  : ofs    : open output stream (by reference)                        *
//*                                                                            *
//* Returns: member of enum ConfigStatus                                       *
//******************************************************************************

ConfigStatus FmConfig::wcfWriteAltTrash ( ofstream& ofs )
{

   ofs << this->shm->trashPath << endl ;
   return csOK ;

}  //* End wcfWriteAltTrash() *

#if 0    // OBSOLETE - OBSOLETE - OBSOLETE
//*************************
//*  wcfWriteTempStorage  *
//*************************
//******************************************************************************
//* Copy the base path for 'temporary files' directory to the target           *
//* configuration file.                                                        *
//*                                                                            *
//* Input  : ofs    : open output stream (by reference)                        *
//*        : ifs    : open input stream (by reference)                         *
//*          lineNum: source-file line count (by reference)                    *
//*                                                                            *
//* Returns: member of enum ConfigStatus                                       *
//******************************************************************************

ConfigStatus FmConfig::wcfWriteTempStorage ( ofstream& ofs, ifstream& ifs, 
                                             short& lineNum )
{
   //* Write the temp-storage path *
   ConfigStatus status = this->wcfWriteTempStorage ( ofs ) ;

   //* Step over existing entry in source file.     *
   //* If comments found before entry, copy them to *
   //* target file. Discard extra empty lines       *
   //* (gs.gschars() == 1) for beauty.              *
   gString gs ;
   short stepStat ;
   do
   {
      stepStat = this->rcfReadSourceLine ( ifs, lineNum, gs ) ;
      if ( stepStat == ZERO && gs.gschars() > 1 )
         ofs << gs.ustr() << endl ;
   }
   while ( stepStat == ZERO ) ;
   if ( stepStat < ZERO )     // if premature end-of-file
      status = csSYNTAX ;

   return status ;

}  //* End wcfWriteTempStorage() *

//*************************
//*  wcfWriteTempStorage  *
//*************************
//******************************************************************************
//* Copy the base path for 'temporary files' directory to the target           *
//* configuration file.                                                        *
//*                                                                            *
//* Input  : ofs    : open output stream (by reference)                        *
//*                                                                            *
//* Returns: member of enum ConfigStatus                                       *
//******************************************************************************

ConfigStatus FmConfig::wcfWriteTempStorage ( ofstream& ofs )
{

   ofs << this->shm->cfgOpt.xxPath << endl ;
   return csOK ;

}  //* End wcfWriteTempStorage() *
#endif   // OBSOLETE - OBSOLETE - OBSOLETE

//*************************
//* wcfWriteTrailingData  *
//*************************
//******************************************************************************
//* When updating an existing configuration file, there may be unused data     *
//* following the 'End Configuration File' marker.                             *
//* We copy this unused data from the source file to the target file.          *
//*                                                                            *
//* We do however, strip any blank lines that follow immediately after the     *
//* marker. If we didn't, another blank line would be added to the file each   *
//* time it was updated.                                                       *
//*                                                                            *
//* Input  : ofs    : open output stream (by reference)                        *
//*        : ifs    : open input stream (by reference)                         *
//*          lineNum: source-file line count (by reference)                    *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void FmConfig::wcfWriteTrailingData ( ofstream& ofs, ifstream& ifs, short& lineNum )
{
   gString gs ;
   short stepStat ;
   bool strip = true ;
   do
   {
      stepStat = this->rcfReadSourceLine ( ifs, lineNum, gs ) ;
      if ( stepStat >= ZERO )
      {
         if ( strip && (gs.gschars() == 1) )
            continue ;
         ofs << gs.ustr() << endl ;
         strip = false ;
      }
   }
   while ( stepStat >= ZERO ) ;

}  //* End wcfWriteTrailingData() *

//*************************
//*   CreateConfigFile    *
//*************************
//******************************************************************************
//* Create a new configuration file with default settings for all parameters,  *
//* OR use the parameter values in the shared-memory object.                   *
//*                                                                            *
//* Input  : populate : (optional, 'false' by default)                         *
//*                     if 'true', copy the template using data values from    *
//*                      the shared-memory segment: this->shm                  *
//*                     if 'false' copy the template with default settings     *
//*                                                                            *
//* Returns: member of enum ConfigStatus                                       *
//******************************************************************************
//* Programmer's Note: When writing a configuration file:                      *
//* 1. If target exists, rename existing file as a backup:                     *
//*    Config.cfg --> Config.cfg~                                              *
//*    a. If a file with backup name already exists, it will be overwrittten.  *
//*       EXCEPTION: If rename-as-backup operation fails, then existing target *
//*                  will be truncated and overwritten.                        *
//* 2. Copy the static data, section-by-section from our template.             *
//*    The sections are in a specific order. See 'DefaultConfig' above.        *
//* 3. Add the command string from the lookup table, and then select the       *
//*    value string:                                                           *
//*    a. if 'populate' != false, use the string corresponding to the value in *
//*       our data member.                                                     *
//*    b. if 'populate' == false, use the string corresponding to the default  *
//*       value. See 'DefaultString'.                                          *
//*                                                                            *
//* The FmKeymap.cfg file:                                                     *
//* See CreateKeymapFile(), below.                                             *
//******************************************************************************

ConfigStatus FmConfig::CreateConfigFile ( bool populate )
{
   gString gsOut ;
   ConfigStatus status = csERR ;    // return value

   //* Check whether target exists, and if it does, rename it as backup.*
   gString trgPath( this->shm->cfgPath ),
           backupPath ;
   backupPath.compose( L"%S~", trgPath.gstr() ) ;
   FileStats   fstats ;
   int verrno = ZERO,
       lsstat = lstat64 ( trgPath.ustr(), &fstats ) ;
   if ( lsstat != ZERO )   verrno = errno ;
   if ( verrno != ENOENT )
   {
      rename ( trgPath.ustr(), backupPath.ustr() ) ;
   }

   //* Open the target file for writing *
   ofstream ofs( trgPath.ustr(), ofstream::out | ofstream::trunc ) ;
   if ( ofs.is_open() )             // if input file open
   {
      //* First line is config-file ID tag and version number *
      gsOut.compose( L"%s %s", ConfigFileID, ConfigFileVersion ) ;
      ofs << gsOut.ustr() << endl ;

      //* Write the header section *
      short section = ZERO ;
      ofs << DefaultCfgFile[section++] << endl ;

      //* Write the 'WindowMode' section *
      ofs << DefaultCfgFile[section++] << endl ;
      if ( populate )
      {
         gsOut.compose( L"%S = %S\n", CfgParm[cciWM].str, 
                        CfgParm[cciWM]. args[this->shm->cfgOpt.winMode].str ) ;
      }
      else
         gsOut.compose( L"%S = %S\n", CfgParm[cciWM].str,
                        CfgParm[cciWM].args[CfgParm[cciWM].dflt].str ) ;
      ofs << gsOut.ustr() << endl ;

      //* Write the 'SortOrder' section *
      ofs << DefaultCfgFile[section++] << endl ;
      if ( populate )
      {
         gsOut.compose( L"%S = %S\n", CfgParm[cciSO].str, 
                        CfgParm[cciSO].args[this->shm->cfgOpt.sortOption].str ) ;
      }
      else
         gsOut.compose( L"%S = %S\n", CfgParm[cciSO].str,
                        CfgParm[cciSO].args[CfgParm[cciSO].dflt].str ) ;
      ofs << gsOut.ustr() << endl ;

      //* Write the 'CaseSensitive' section *
      ofs << DefaultCfgFile[section++] << endl ;
      if ( populate )
      {
         gsOut.compose( L"%S = %S\n", CfgParm[cciCS].str, 
                        CfgParm[cciCS].args[this->shm->cfgOpt.caseSensitive ? 0 : 1].str ) ;
      }
      else
         gsOut.compose( L"%S = %S\n", CfgParm[cciCS].str,
                        CfgParm[cciCS].args[CfgParm[cciCS].dflt].str ) ;
      ofs << gsOut.ustr() << endl ;

      //* Write the 'HiddenFiles' section *
      ofs << DefaultCfgFile[section++] << endl ;
      if ( populate )
      {
         gsOut.compose( L"%S = %S\n", CfgParm[cciHF].str, 
                        CfgParm[cciHF].args[this->shm->cfgOpt.showHidden ? 0 : 1].str ) ;
      }
      else
         gsOut.compose( L"%S = %S\n", CfgParm[cciHF].str,
                        CfgParm[cciHF].args[CfgParm[cciHF].dflt].str ) ;
      ofs << gsOut.ustr() << endl ;

      //* Write the 'ConfirmDelete' section *
      ofs << DefaultCfgFile[section++] << endl ;
      if ( populate )
      {
         gsOut.compose( L"%S = %S\n", CfgParm[cciCD].str, 
                        CfgParm[cciCD].args[this->shm->cfgOpt.confirmDelete ? 0 : 1].str ) ;
      }
      else
         gsOut.compose( L"%S = %S\n", CfgParm[cciCD].str,
                        CfgParm[cciCD].args[CfgParm[cciCD].dflt].str ) ;
      ofs << gsOut.ustr() << endl ;

      //* Write the 'ConfirmOverwrite' section *
      ofs << DefaultCfgFile[section++] << endl ;
      if ( populate )
      {
         gsOut.compose( L"%S = %S\n", CfgParm[cciCO].str, 
                        CfgParm[cciCO].args[this->shm->cfgOpt.overWrite].str ) ;
      }
      else
         gsOut.compose( L"%S = %S\n", CfgParm[cciCO].str,
                        CfgParm[cciCO].args[CfgParm[cciCO].dflt].str ) ;
      ofs << gsOut.ustr() << endl ;

      //* Write the 'LockMenuBar' section *
      ofs << DefaultCfgFile[section++] << endl ;
      if ( populate )
      {
         gsOut.compose( L"%S = %S\n", CfgParm[cciLM].str, 
                        CfgParm[cciLM].args[this->shm->lockMb ? 0 : 1].str ) ;
      }
      else
         gsOut.compose( L"%S = %S\n", CfgParm[cciLM].str,
                        CfgParm[cciLM].args[CfgParm[cciLM].dflt].str ) ;
      ofs << gsOut.ustr() << endl ;

      //* Write the 'ColorScheme' section *
      ofs << DefaultCfgFile[section++] << endl ;
      if ( populate )
      {
         short scIndex ;
         switch ( this->shm->cfgOpt.cScheme.scheme )
         {
            case ncbcBK:      scIndex = 1 ;  break ;
            case ncbcRE:      scIndex = 2 ;  break ;
            case ncbcGR:      scIndex = 3 ;  break ;
            case ncbcBR:      scIndex = 4 ;  break ;
            case ncbcBL:      scIndex = 5 ;  break ;
            case ncbcMA:      scIndex = 6 ;  break ;
            case ncbcCY:      scIndex = 7 ;  break ;
            case ncbcGY:      scIndex = 8 ;  break ;
            case ncbcDEFAULT: scIndex = 9 ;  break ;  // terminal dflt
            case ncbcCOLORS:                          // application default
            default:          scIndex = 0 ;  break ;
         }
         gsOut.compose( L"%S = %S\n", CfgParm[cciSC].str, scArgs[scIndex].str ) ;
      }
      else
         gsOut.compose( L"%S = %S\n", CfgParm[cciSC].str,
                        CfgParm[cciSC].args[CfgParm[cciSC].dflt].str ) ;
      ofs << gsOut.ustr() << endl ;

      //* Write the 'EnableMouse' section *
      ofs << DefaultCfgFile[section++] << endl ;
      if ( populate )
      {
         gsOut.compose( L"%S = %S\n", CfgParm[cciEM].str, 
                        CfgParm[cciEM].args[this->shm->cfgOpt.enableMouse ? 0 : 1].str ) ;
      }
      else
         gsOut.compose( L"%S = %S\n", CfgParm[cciEM].str,
                        CfgParm[cciEM].args[CfgParm[cciEM].dflt].str ) ;
      ofs << gsOut.ustr() << endl ;

      //* Write the 'ExternalPrograms' section *
      ofs << DefaultCfgFile[section++] << endl ;
      if ( populate )
      {
         gsOut.compose( L"%S = %S\n", CfgParm[cciEP].str, 
                                      epArgs[this->shm->cfgOpt.lCode].str ) ;
      }
      else
         gsOut.compose( L"%S = %S\n", CfgParm[cciEP].str,
                        CfgParm[cciEP].args[CfgParm[cciEP].dflt].str ) ;
      ofs << gsOut.ustr() << endl ;

      //* Write the 'EndCfgCommands' section *
      ofs << DefaultCfgFile[section++] << endl ;

      ConfigStatus wStat = csOK ;   // value returned from sub-methods

      //* Write the 'Favorite Directories' section *
      ofs << DefaultCfgFile[section++] << endl ;
      if ( populate )
      {
         wStat = this->wcfWriteFavorites ( ofs ) ;
         if ( wStat != csOK )
            status = wStat ;
      }
      else
         ofs << ConfigDfltFavorite << "\n" << endl ;

      //* Write the 'KeyMap' section *
      ofs << DefaultCfgFile[section++] ;
      //* Create a keymap config file with, or without user-defined data.   *
      //* a) If the keymap file will contain user-defined data, then append *
      //*    the filename to the command option.                            *
      //* b) If the keymap file contains only default data, the filename    *
      //*    is omitted from the command option.
      gsOut.compose( "%S = ", CfgList[cliKM].str ) ;
      if ( populate )
         gsOut.append( kmcfgName ) ;
      ofs << gsOut.ustr() << '\n' << endl ;
      wStat = this->CreateKeymapFile ( this->shm->keyMap.kmActive ) ;

      //* Write the 'Mount Commands' section *
      ofs << DefaultCfgFile[section++] << endl ;
      if ( populate )
      {
         wStat = this->wcfWriteMounts ( ofs ) ;
         if ( wStat != csOK )
            status = wStat ;
      }
      else
         ofs << endl ;

      //* Write the 'Alternate Trashcan' section *
      ofs << DefaultCfgFile[section++] << endl ;
      if ( populate )
      {
         wStat = this->wcfWriteAltTrash ( ofs ) ;
         ofs << endl ;        // add a blank line
         if ( wStat != csOK )
            status = wStat ;
      }
      else
      {  //* Default entry is expanded version of default trashcan path. *
         //* If not a valid path, then we leave the entry blank.         *
         gsOut = ConfigDfltTrashcanPath ;
         if ( (this->rcfValidateTrashPath ( gsOut )) != false )
            ofs << gsOut.ustr() << endl ;
         ofs << endl ;        // add a blank line
      }

      //* Write the 'EndConfigFile' marker *
      ofs << DefaultCfgFile[section++] << endl ;

      ofs.close() ;                 // close the file
      status = csOK ;               // return the good news
   }

   return status ;

}  //* End CreateConfigFile() *

//*************************
//*   CreateKeymapFile    *
//*************************
//******************************************************************************
//* Create a new FmKeymap.cfg file in the same directory as the primary        *
//* configuration file.                                                        *
//*   a) with only the list of default settings, OR                            *
//*   b) with both default, and user-defined settings.                         *
//*                                                                            *
//* Input  : populate : (optional, 'false' by default)                         *
//*                     if 'true', copy the template using data values from    *
//*                      the shared-memory segment: this->shm                  *
//*                     if 'false' copy the template with default settings     *
//*          trgFile  : (optional, NULL pointer by default)                    *
//*                     if specified, provides the target filename             *
//*                     if not specified, the default filename will be used    *
//*                                                                            *
//* Returns: member of enum ConfigStatus                                       *
//******************************************************************************
//* 1. If target exists, rename existing file as a backup:                     *
//*    FmKeymap.cfg --> FmKeymap.cfg~                                          *
//*    a. If a file with backup name already exists, it will be overwrittten.  *
//*       EXCEPTION: If rename-as-backup operation fails, then existing target *
//*                  will be truncated and overwritten.                        *
//* 2. Copy the static data from our template data:                            *
//*                  'KeymapCfgDflt' and 'KeymapCfgDesc'                       *
//*    This is done through the RemapKey::initialize() method.                 *
//*      a) keydef.[x].kdflt <== KeymapCfgDflt[x]                              *
//*      b) keydef.[x].kuser:                                                  *
//*          i) 'populate' != false: caller has initialized keydef[x].kuser.   *
//*         ii. 'populate' == false: initialize keydef[x].kuser <== 'nonCode'  *
//*      c) keydef.ktext[x]: Text description is based on the combination of   *
//*         KeymapCfgDesc[x], keydef[x].kdflt and keydef[x].kuser according to *
//*         the appropriate template.                                          *
//* 3. The target file is written using the boilerplate, 'KeymapCfgHdr' and    *
//*    the array of text descriptions.                                         *
//******************************************************************************

ConfigStatus FmConfig::CreateKeymapFile ( bool populate, const char* trgFile )
{
   #define DEBUGckf (0)             // set as non-ZERO for debug only

   gString gsOut ;
   ConfigStatus status = csERR ;    // return value

   //* Create the target filespec.*
   gString trgPath( this->shm->cfgPath ),
           bakPath ;

   short offset = (trgPath.findlast( L'/' )) + 1 ;
   trgPath.limitChars( offset ) ;
   #if DEBUGckf == 0 // Production
   trgPath.append( trgFile != NULL ? trgFile : kmcfgName ) ;
   #else    // clfDEBUG
   trgPath.append( "KMtest.cfg" ) ;
   #endif   // clfDEBUG
   bakPath.compose( "%S~", trgPath.gstr() ) ;

   #if DEBUGckf != 0
   if ( this->shm->msgCount < smmMSG_COUNT )
   {
      gsOut = "** ccKEYMAP **" ;
      gsOut.copy ( this->shm->msg[this->shm->msgCount++], smmMSG_LEN ) ;
      gsOut.compose( "trgPath: %S", trgPath.gstr() ) ;
      gsOut.copy ( this->shm->msg[this->shm->msgCount++], smmMSG_LEN ) ;
      gsOut.compose( "bakPath: %S", bakPath.gstr() ) ;
      gsOut.copy ( this->shm->msg[this->shm->msgCount++], smmMSG_LEN ) ;
   }
   #endif   // DEBUGckf

   //* Check whether target exists, and if it does, rename it as backup.*
   FileStats   fstats ;
   int verrno = ZERO,
       lsstat = lstat64 ( trgPath.ustr(), &fstats ) ;
   if ( lsstat != ZERO )   verrno = errno ;
   if ( verrno != ENOENT )
   {
      #if DEBUGckf == 0 // Production
      rename ( trgPath.ustr(), bakPath.ustr() ) ;
      #endif   // DEBUGckf == 0
   }

   //* Open the target file for writing *
   ofstream ofs( trgPath.ustr(), ofstream::out | ofstream::trunc ) ;
   if ( ofs.is_open() )             // if input file open
   {
      //* First line is config-file ID tag and version number *
      gsOut.compose( L"%s %s", KeymapFileID, KeymapFileVersion ) ;
      ofs << gsOut.ustr() << endl ;

      //* Write the header section *
      ofs << KeymapCfgHdr << endl ;

      #if DEBUGckf != 0 && (0) // Static test population of user-defined fields
      for ( short i = ZERO ; i < KEYMAP_KEYS ; ++i )
         this->shm->keyMap.keydef[i].kuser = KeymapCfgDflt[i] ;
      populate = true ;
      #endif   // Static test population of user-defined fields

      //* Format the keymap records and write them to the file.*
      this->shm->keyMap.initialize( true, populate ) ;
      for ( short i = ZERO ; i < KEYMAP_KEYS ; ++i )
         ofs << this->shm->keyMap.keydef[i].ktext << '\n' ;
      ofs << endl ;

      ofs.close() ;                 // close the file
      status = csOK ;               // return the good news
   }
   return status ;

   #undef DEBUGckf
}  //* End CreateKeymapFile() *

//*************************
//*                       *
//*************************
//******************************************************************************
//*                                                                            *
//*                                                                            *
//*                                                                            *
//* Input  :                                                                   *
//*                                                                            *
//* Returns:                                                                   *
//******************************************************************************


