//********************************************************************************
//* File       : FileDlgUtil1.cpp                                                *
//* Author     : Mahlon R. Smith                                                 *
//*              Copyright (c) 2005-2025 Mahlon R. Smith, The Software Samurai   *
//*                  GNU GPL copyright notice located in FileMangler.hpp         *
//* Date       : 16-Jul-2025                                                     *
//* Version    : (see FileDlgVersion string in FileDlg.cpp)                      *
//*                                                                              *
//* Description:                                                                 *
//* This module of the FileDlg class contains an interactive implementation      *
//* for various command-line utilities.                                          *
//*                                                                              *
//* 1) Implementation of the 'FIND' command which scans the directory tree for   *
//*    filenames that match a specified pattern.                                 *
//*                                                                              *
//* 2) Implementation of the 'DIFF' ('Compare Files') command which gets the     *
//*    names of two files from the user and compares them.                       *
//*                                                                              *
//* 3) Implementation of the 'GREP' (substring search) command which prompts     *
//*    the user for a regexp substring and searches the specified                *
//*    (or 'selected') files for that substring.                                 *
//*    a) This includes a group of methods which manage the contents of          *
//*       OpenDocument and Office Open XML document files.                       *
//*                                                                              *
//*                                                                              *
//* Development Tools: see notes in FileMangler.cpp.                             *
//********************************************************************************
//* Version History (most recent first):                                         *
//*   See version history in FileDlg.cpp.                                        *
//********************************************************************************

#include "GlobalDef.hpp"
#include "FileDlg.hpp"

//******************************
//* Local definitions and data *
//******************************
//* Message text defined in FileDlgPrompt.cpp *
extern const char* fmtypeString[] ;
extern const char* fmtypeName[] ;
extern const char* readonlyFileSys[] ;
extern const char* notOntoSelf[] ;
extern const char* cannotOverrideWP[] ;
extern const char* noLinkOverwrite[] ;
extern const char* differentFileTypes[] ;
extern const char* cannotDeleteSpecial[] ;
extern const char* cannotOverwriteFifo[] ;
extern const char* cannotOverwriteDirectory[] ;
extern const char* warnSocketMod[] ;
extern const char* warnBdevCdevMod[] ;
extern const char* warnUnsuppMod[] ;
extern const char* fdNotImplemented[] ;


//* Dialog control definitions for Find-files Dialog *
enum ffControls : short { ffmatTB = ZERO, ffSE, ffcntTB, ffRetPB, ffSpin, ffCONTROLS } ;
//* Max matches displayed for Find-files. See notes in ffDynamicAllocation(). *
static const int maxMATCH = 200 ;
//* Message displayed if scan of directory tree returns no filenames *
//* which match search criteria.                                     *
static const char* noJOYMSG = " No files match the search criteria. " ;
static const short mattbWIDTH = 78 - 23 ;  // width of ffmatTB (Single-win max)

//* Delimiter characters for 'grep' methods.*
const wchar_t DQUOTE = L'"' ;
const wchar_t SQUOTE = L'\'' ;
const wchar_t BKSTROKE = '\\' ;     // prepend to 'escape' a single character

//* Max bytes in filename list (prevent grep buffer overflow) *
const short MAX_PAT = gsDFLTBYTES - 100 ;

#define DEBUG_FIND (0)
#define DEBUG_INODE (0)
#if (DEBUG_FIND != 0) || (DEBUG_INODE != 0)
ofstream ofs ;
#endif   //DEBUG_FIND || DEBUG_INODE

//********************
//* Local prototypes *
//********************



//******************************************************************************
//***                    'Find Files' Method Group                           ***
//******************************************************************************

//*************************
//*       FindFiles       *
//*************************
//********************************************************************************
//* Scan the directory tree beginning with the the current working directory     *
//* (CWD). Locate all files which match the user's search string.                *
//*                                                                              *
//* Allow user to optionally select a new CWD path to:                           *
//*   1) a directory whose filename matches the search criteria                  *
//*   2) a directory which contains a file matching the search criteria          *
//*                                                                              *
//* Input  : newCwd    : (by reference) receives user's path selection           *
//*                      (cleared if no selection made)                          *
//*          ul        : (by reference) upper left corner dialog position        *
//*          rows      : number of rows for dialog                               *
//*          cols      : number of columns for dialog                            *
//*                                                                              *
//* Returns: true if user has specified a new CWD ('newCwd' contains new path)   *
//*          else false                                                          *
//********************************************************************************
//* Notes:                                                                       *
//* ------                                                                       *
//* 1) In order to scan properly, we must have access to all directories and     *
//*    files _including_ hidden ones.                                            *
//*    a) If hidden files are not already visible, then we must rescan the       *
//*       CWD to synchronize the counters.                                       *
//*    b) If we do rescan, AND we are returning to the CWD, then we must scan    *
//*       again to remove the hidden files from view.                            *
//*    c) If hidden files ARE already visible, then we can skip these steps.     *
//*                                                                              *
//* 2) There is a logical hole in the mouse interface for this dialog.           *
//*    The mouse can select the Scrollext control ONLY if that control contains  *
//*    live data (not the placeholder). However, the initial test for live       *
//*    data occurs only after the first call to EditTextbox() returns.           *
//*    We compensate for this by initializing the Scrollext control to 'active'  *
//*    even though the Scrollext initially contains noJOYMSG. Then on return     *
//*    from EditTextbox(), if the Scrollext contains no live data we ensure      *
//*    that user cannot access it.                                               *
//********************************************************************************

bool FileDlg::FindFiles ( gString& newCwd, winPos& ul, short rows, short cols )
{
   gString origcwd, gstmp ;
   this->GetPath ( origcwd ) ;      // where we are now
   bool newCWD = false ;            // return value

   //* Reset caller's gString object *
   newCwd.clear() ;

   //* Enable scanning of hidden directories and files (see note above).*
   bool oldshowHidden = this->cfgOptions.showHidden ;
   if ( ! oldshowHidden )
   {
      if ( (this->fmPtr->ShowHiddenFiles ( true )) == OK )
      {
         this->cfgOptions.showHidden = true ;
         this->RefreshCurrDir ( false, true ) ;
      }
      else              // prevent the switch back at bottom of method
         oldshowHidden = true ;
   }

   //* Save the display for the parent dialog window *
   this->dPtr->SetDialogObscured () ;

   //* Open the Find-file Dialog *
   NcDialog* dp = this->ffOpenDialog ( ul, rows, cols ) ;

   //* Initialize the Scrollext data array.      *
   //* (initially one item: the 'empty' message) *
   ssetData sData ;
   this->ffDynamicAllocation ( sData ) ;
   dp->SetScrollextText ( ffSE, sData ) ;    // display the data


   uiInfo   Info ;                     // user interface data returned here
   short    icIndex = ZERO ;           // index of control with input focus
   bool done = (dp == NULL) ? true : false ;    // loop control

   //* Create the secondary thread to monitor contents of ffmatTB *
   thread monitorThread( &FileDlg::ffTextboxMonitor, this, dp, &sData, &done, (cols - 2) ) ;

   while ( ! done )
   {
      if ( icIndex == ffRetPB )
      {
         if ( Info.viaHotkey )
            Info.HotData2Primary () ;
         else
            icIndex = dp->EditPushbutton ( Info ) ;

         if ( Info.dataMod != false )
            done = true ;
      }

      else if ( icIndex == ffmatTB )
      {
         //* NOTE: The secondary thread is monitoring the contents of the      *
         //* Textbox and will update the matching-file list when data changes. *
         Info.viaHotkey = false ;
         icIndex = dp->EditTextbox ( Info ) ;

         //* Allow selection ONLY if there are file-match data *
         gstmp = sData.dispText[ZERO] ;
         if ( (gstmp.compare( noJOYMSG )) != ZERO )
            dp->ControlActive ( ffSE, true ) ;  // activate Scrollext control
         else
         {
            //* If the Scrollext control has focus, BUT contains no live data, *
            //* then move the focus to Pushbutton and (thence back to the      *
            //* Textbox). See note above.                                      *
            if ( icIndex == ffSE )
               icIndex = dp->NextControl () ;
            dp->ControlActive ( ffSE, false ) ; // deactivate Scrollext control
         }
      }
      
      else if ( icIndex == ffSE )
      {
         Info.viaHotkey = false ;      // ignore hotkey data
         icIndex = dp->EditScrollext ( Info ) ;

         //* If user has made a selection *
         // Programmer's Note: 'dataMod' is set if the highlight has moved 
         // OR if the Enter keys or the Space key have been pressed. However,
         // we don't care whether the highlight has moved, only where it IS.
         // So to protect ourselves, we directly test the key data.
         if ( (Info.wk.type == wktPRINT && Info.wk.key == nckSPACE) ||
              ((Info.wk.type == wktFUNKEY) && 
               (Info.wk.key == nckENTER || Info.wk.key == nckpENTER)) )
         {  //* Get the selected item and construct the target path *
            //* If display data have been sorted, send a copy of    *
            //* the selected display item text.                     *
            int spVal ; dp->GetSpinnerValue ( ffSpin, spVal ) ;
            if ( spVal > ZERO ) { newCwd = sData.dispText[Info.selMember] ; }
            newCWD = this->ffGetSelectedItem ( newCwd, sData, Info.selMember ) ;
            done = true ;
         }
      }

      else if ( icIndex == ffSpin )
      {
         Info.viaHotkey = false ;      // ignore hotkey data
         icIndex = dp->EditSpinner ( Info ) ;
      }

      //* Move focus to appropriate control *
      if ( ! done && ! Info.viaHotkey )
      {
         if ( Info.keyIn == nckSTAB )
            icIndex = dp->PrevControl () ; 
         else
            icIndex = dp->NextControl () ;
      }
   }     // while(!done)

   //* Signal the secondary thread to terminate *
   //* and wait for it to return.               *
   monitorThread.join() ;

   //* Close the Find-file dialog *
   if ( dp != NULL )
      delete ( dp ) ;

   //* Release our dynamic memory allocation *
   this->ffDynamicAllocation ( sData, true, true ) ;

   this->dPtr->RefreshWin () ;      // restore the parent dialog

   //* If "hidden" files were not visible on entry, return  *
   //* them to the "hidden" state, AND if user is returning *
   //* to original CWD, re-read the CWD.                    *
   if ( ! oldshowHidden )
   {
      if ( (this->fmPtr->ShowHiddenFiles ( false )) == OK )
      {
         this->cfgOptions.showHidden = false ;
         if ( ! newCWD )
            this->RefreshCurrDir ( true, true ) ;
      }
   }
   
   return newCWD ;

}  //* End FindFiles() *

//*************************
//*   ffTextboxMonitor    *
//*************************
//********************************************************************************
//* This method is the target of a secondary execution thread which monitors     *
//* changes in the contents of the ffmatTB Textbox control and changes to        *
//* the ffSpin Spinner control.                                                  *
//*                                                                              *
//* The thread is launched in FindFiles() just after the dialog opens, and       *
//* terminates when the the FindFiles dialog closes.                             *
//*                                                                              *
//* The secondary thread is necessary for dynamically updating the list of       *
//* matching files while the user is entering the search text.                   *
//*                                                                              *
//* Input  : dp     : pointer to caller's dialog window                          *
//*          sData  : pointer to object which receives the display data for      *
//*                   matching filenames                                         *
//*          retFlag: pointer to flag which caller sets to indicate that         *
//*                   the thread should return (terminate)                       *
//*          dCols  : number of display columns in Scrollext control             *
//*                   (used for formatting the display data)                     *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************
//* Notes:                                                                       *
//* -- We can't use a callback method to perform this task because callbacks     *
//*    are non-member methods, and this operation requires direct and            *
//*    significant access to FileDlg methods and data members.                   *
//* -- We have to wake the thread often enough to be responsive, but still       *
//*    not waste a lot of CPU cycles.                                            *
//* -- There is a finite possibility that we could get a resource conflict       *
//*    between the monitor thread and the user-interface thread in accessing     *
//*    the 'sData' object. It is unlikely, and after thousands of test runs,     *
//*    we have not encountered it -- but be aware.                               *
//  -- Possible Enhancement: - Create a flag to signal that user wants to set    *
//*    the left-shift to zero so timestamps are fully visible.                   *
//*    This should not be a problem in dual-win mode, but in single-win mode,    *
//*    the leftmost data of records is often obscured.                           *
//*                                                                              *
//********************************************************************************

void FileDlg::ffTextboxMonitor ( NcDialog* dp, ssetData* sData, 
                                 const bool* retFlag, short dCols )
{
   gString oldText,              // previous TBox contents
           substr,               // new TBox contents
           ifmt ;                // for integer formatting
   chrono::duration<short, std::milli>aMoment( 250 ) ;

   int   fMatch = ZERO,       // number of filenames that match the search string
         tsFormat,            // timestamp format
         oldtsFormat ;

   dp->GetTextboxText ( ffmatTB, oldText ) ;       // get original TBox contents
   dp->GetSpinnerValue ( ffSpin, oldtsFormat ) ;   // get original Spinner value

   while ( ! *retFlag )
   {  //* Wait a moment for user to enter text into the Textbox control *
      this_thread::sleep_for( aMoment ) ;

      //* Compare current contents with previous contents *
      dp->GetTextboxText ( ffmatTB, substr ) ;
      dp->GetSpinnerValue ( ffSpin, tsFormat ) ;
      if ( (substr != oldText) || (tsFormat != oldtsFormat) )
      {
         this->ffDynamicAllocation ( *sData ) ; // re-initialize display data arrays
         fMatch = this->ffScanTree ( substr, *sData, tsFormat ) ;

         //* Because ths method is often used to find all instances  *
         //* of the same filename within the directory tree, if the  *
         //* timestamp is active, sort the list by timestamp.        *
         if ( (fMatch > ZERO) && (tsFormat > ZERO) )
            this->ffSortRecords ( sData, fMatch, tsFormat ) ;

         //* If we have matching files, adjust the string offsets *
         //* to ensure that the data will fit in the display area.*
         if ( fMatch > ZERO )
            this->ffAdjustOutput ( *sData, dCols ) ;
         dp->SetScrollextText ( ffSE, *sData ) ;    // display the data

         ifmt.formatInt( fMatch, 5 ) ;             // update match count
         dp->SetTextboxText ( ffcntTB, ifmt ) ;
         oldText = substr ;                        // save the updated substring
         oldtsFormat = tsFormat ;                  // save the updated format code
      }
   }
}  //* End ffTextboxMonitor() *

//*************************
//*     ffSortRecords     *
//*************************
//********************************************************************************
//* Sort the display data records. This is useful for comparison of mod dates    *
//* on several copies of the same file in the search area.                       *
//*                                                                              *
//* This is a simple, high-to-low bubble sort.                                   *
//*                                                                              *
//* Input  : sData   : pointer to object which contains the display data for     *
//*                    matching filenames                                        *
//*          fMatch  : number of records in the array                            *
//*          tsFormat: for sorting records according to timestamp, this is       *
//*                    the value of the spinner in the dialog window.            *
//*                    0 == no timestamp, sort by filename                       *
//*                         (currently no filename sort is performed)            *
//*                    1 == timestamp format is date only                        *
//*                         yyyy-mm-dd                                           *
//*                    2 == timestamp is canonical timestamp                     *
//*                         yyyy-mm-ddThh-mm-ss                                  *
//*          ascend  : direction of sort: true==low-to-high, false==high-to-low  *
//*                    Programmer's Note: Both ascending and descending sort     *
//*                    are functional, but the 'ascend' flag is currently never  *
//*                    set by caller.                                            *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************
//* Programmer's Note: The data membes of the ssetData class are defined as      *
//* constants to avoid developer stupidity; however, for sorting the records we  *
//* override this constant data pointers with non-const pointers.                *
//* (We promise not to trash the data via overt stupidity.)                      *
//********************************************************************************

void FileDlg::ffSortRecords ( ssetData* sData, int fMatch, int tsFormat, bool ascend )
{
   //* Sort ONLY by timestamp. Filename sort would be meaningless in this context.*
   if ( (tsFormat < 1) || (tsFormat > 2) || (fMatch < 2) )
      return ;

   //* Count of displayed records is limited by maxMATCH. *
   //* See note in ffDynamicAllocation().                 *
   if ( fMatch > sData->dispItems ) { fMatch = sData->dispItems ; }

   const short tsDATE  = 10 ;    // tsFormat==1 : timestamp==10 characters
   const short tsCANON = 19 ;    // tsFormat==2 : timestamp==19 characters
   gString gsI, gsL, gsG ;       // temp buffers
   char** sDataText = (char**)(sData->dispText) ;  // see note above
   attr_t* sDataAttr = (attr_t*)(sData->dispColor) ;
   attr_t atrI, atrL, atrG ;     // temp attributes
   int lti, gti,                 // low and high limits search indices
       ini,                      // inner-loop index
       cmpVal ;                  // result of comparison
   short loadCnt = tsFormat == 2 ? tsCANON : tsDATE ;

   for ( lti = 0, gti = fMatch-1 ; (gti-lti) > ZERO ; ++lti, --gti )
   {
      for ( ini = lti+1 ; ini <= gti ; ++ini )
      {
         gsI = sData->dispText[ini] ;     // load record text for comparison
         gsL = sDataText[lti] ;
         cmpVal = gsI.compare( sDataText[lti], true, loadCnt ) ;

         if ( (ascend && (cmpVal < ZERO)) || (!ascend && (cmpVal > ZERO)) )
         {
            gsI.copy( sDataText[lti], gsDFLTBYTES ) ;
            gsL.copy( sDataText[ini], gsDFLTBYTES ) ;
            atrI = sDataAttr[ini] ;       // swap attributes
            atrL = sDataAttr[lti] ;
            sDataAttr[lti] = atrI ;
            sDataAttr[ini] = atrL ;
            gsI = gsL ;
            gsL = sDataText[lti] ;
         }

         if ( ini < gti )                 // prevent comparing a record to itself
         {
            gsG = sDataText[gti] ;
            cmpVal = gsG.compare( gsI.gstr(), true, loadCnt ) ;

            if ( (ascend && (cmpVal < ZERO)) || (!ascend && (cmpVal > ZERO)) )
            {
               gsI.copy( sDataText[gti], gsDFLTBYTES ) ; // swap text
               gsG.copy( sDataText[ini], gsDFLTBYTES ) ;
               atrG = sDataAttr[gti] ;    // swap attributes
               atrI = sDataAttr[ini] ;
               sDataAttr[gti] = atrI ;
               sDataAttr[ini] = atrG ;
               gsI = gsG ;                // refresh comparison data
            }

            cmpVal = gsI.compare( gsL.gstr() ) ;
            if ( (ascend && (cmpVal < ZERO)) || (!ascend && (cmpVal > ZERO)) )
            {
               gsI.copy( sDataText[lti], gsDFLTBYTES ) ; // swap text
               gsL.copy( sDataText[ini], gsDFLTBYTES ) ;
               atrL = sDataAttr[lti] ;    // swap attributes
               atrI = sDataAttr[ini] ;
               sDataAttr[lti] = atrI ;
               sDataAttr[ini] = atrL ;
            }
         }
      }
   }

}  //* End ffSortRecords

//*************************
//*     ffOpenDialog      *
//*************************
//********************************************************************************
//* Open the Find-file dialog window and return a pointer to it.                 *
//*                                                                              *
//*                                                                              *
//* Input  : ul    : (by reference) upper left corner dialog position            *
//*          rows  : number of rows for dialog                                   *
//*          cols  : number of columns for dialog                                *
//*          fi    : (optional, 'false' by default)                              *
//*                  'false' == set up the dialog for FindFiles()                *
//*                  'true'  == set up the dialog for FindInodes()               *
//*                                                                              *
//* Returns: pointer to open dialog window                                       *
//*          OR NULL pointer if dialog did not open                              *
//********************************************************************************

NcDialog* FileDlg::ffOpenDialog ( const winPos& ul, short rows, short cols, bool fi )
{
   const char* Labels[4] = 
   {
      "Enter search text:",
      "     Target INODE:",
      "  Matching Files:",
      "Hard Links Found:"
   } ;
   attr_t bColor = this->cs.bb,        // border color
          dColor = bColor,             // interior color
          hColor  = this->cs.em ;      // spinner color

   //* Data for the ffSpin spinner control.*
   //*      (output formatting code)       *
   dspinData dsData( 0, 2, 0, dspinINTEGER, hColor ) ;

   //** Define the dialog controls **
   InitCtrl ic[ffCONTROLS] = 
   {
   {  //* 'SEARCH TEXT' Textbox  - - - - - - - - - - - - - - - - - - - ffmatTB *
      dctTEXTBOX,                   // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      2,                            // ulY:       upper left corner in Y
      21,                           // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      mattbWIDTH,                   // cols:      control columns
      NULL,                         // dispText:  
      this->cs.tn,                  // nColor:    non-focus color
      this->cs.tf,                  // fColor:    focus color
      #if LINUX_SPECIAL_CHARS != 0
      tbFileLinux,                  // filter: valid filename chars (incl. Linux "special")
      #else    // BASIC FILENAME FILTER
      tbFileName,                   // filter:    valid filename characters
      #endif   // BASIC FILENAME FILTER
      Labels[fi ? 1 : 0],           // label:     
      ZERO,                         // labY:      
      -19,                          // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[ffSE]                     // nextCtrl:  link in next structure
   },
   {  //* 'TARGET SELECTION' Scrollext  - - - - - - - - - - - - - - - -   ffSE *
      dctSCROLLEXT,                 // type:      define a scrolling-data control
      rbtTYPES,                     // sbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(ic[ffmatTB].ulY + 9),   // ulY:       upper left corner in Y
      ZERO,                         // ulX:       upper left corner in X
      short(rows - 11),             // lines:     control lines
      cols,                         // cols:      control columns
      NULL,                         // dispText:  n/a
      bColor,                       // nColor:    non-focus border color
      this->cs.pf,                  // fColor:    focus border color
      tbPrint,                      // filter:    (n/a)
      "  Matching Files  ",         // label:     control label
      ZERO,                         // labY:      offset from control's ulY
      ZERO,                         // labX       offset from control's ulX
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[ffcntTB]                  // nextCtrl:  link in next structure
   },
   {  //* 'MATCH COUNT' Textbox (inactive) - - - - - - - - - - - - - - ffcntTB *
      dctTEXTBOX,                   // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(ic[ffmatTB].ulY + 2),   // ulY:       upper left corner in Y
      ic[ffmatTB].ulX,              // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      5,                            // cols:      control columns
      "    0",                      // dispText:  
      this->cs.tf,                  // nColor:    non-focus color
      this->cs.tf,                  // fColor:    focus color
      tbPrint,                      // filter:    all printing characters
      Labels[fi ? 3 : 2],           // label:     control label
      ZERO,                         // labY:      
      -18,                          // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      false,                        // active:    this control is view-only
      &ic[ffRetPB]                  // nextCtrl:  link in next structure
   },
   {  //* 'RETURN' pushbutton - - - - - - - - - - - - - - - - - - - -  ffRetPB *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      ic[ffcntTB].ulY,              // ulY:       upper left corner in Y
      short(ic[ffcntTB].ulX + ic[ffcntTB].cols + 4),   // ulX: upper left corner in X
      1,                            // lines:     (n/a)
      8,                            // cols:      control columns
      " RETURN ",                   // dispText:  
      this->cs.pn,                  // nColor:    non-focus color
      this->cs.pf,                  // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "",                           // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[ffSpin]                   // nextCtrl:  link in next structure
   },
   {  //* 'Timestamp' spinner       - - - - - - - - - - - - - - - - - - ffSpin *
      dctSPINNER,                   // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      ic[ffRetPB].ulY,              // ulY:       upper left corner in Y
      short(ic[ffRetPB].ulX + 14),  // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      3,                            // cols:      control columns
      NULL,                         // dispText:  (n/a)
      this->cs.pn,                  // nColor:    non-focus color
      this->cs.pf,                  // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "Timestamp format",           // label:
      ZERO,                         // labY:      
      4,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      &dsData,                      // spinData:  spinner init
      true,                         // active:    allow control to gain focus
      NULL                          // nextCtrl:  link in next structure
   },
   } ;

   //* If inode search, do not instantiate spinner control.*
   if ( fi )
      ic[ffRetPB].nextCtrl = NULL ;

   //* Initial parameters for dialog window *
   InitNcDialog dInit( rows,           // number of display lines
                       cols,           // number of display columns
                       ul.ypos,        // Y offset from upper-left of terminal 
                       ul.xpos,        // X offset from upper-left of terminal 
                       NULL,           // dialog title
                       ncltSINGLE,     // border line-style
                       bColor,         // border color attribute
                       bColor,         // interior color attribute
                       ic              // pointer to list of control definitions
                     ) ;

   //* Instantiate the dialog window *
   NcDialog* dp = new NcDialog ( dInit ) ;
   if ( (dp->OpenWindow()) == OK )
   {
      dp->SetDialogTitle ( ((fi == false) ?
                            "  Find Files in Directory Tree  " :
                            "  Find Hard Links (shared inodes) in Filesystem  "),
                            this->cs.em ) ;

      //* Enable audible alert for invalid input *
      dp->TextboxAlert ( ffmatTB, true ) ;

      //* Set and lock 'Insert Mode' *
      dp->SetTextboxInputMode ( false, true ) ;
      
      //* Draw the static data *
      dp->WriteChar ( 0, 0, wcsLTEE, bColor ) ;
      dp->WriteChar ( 0, (cols - 1), wcsRTEE, bColor ) ;

      gString gscwd( this->currDir ) ;              // copy of current pathspec
      winPos instrPos( (ic[ffRetPB].ulY + 2), 2 ) ; // y/x position for instructions

      if ( ! fi )          // for FindFiles()
      {
         //* Display path of CWD *
         this->TrimPathString ( gscwd, (ic[ffmatTB].cols - 5) ) ;
         gscwd.insert( "CWD: " ) ;

         dp->WriteParagraph ( instrPos,
         "1) Enter all or part of the filename(s) for which to search.\n"
         "   (enter valid filename characters only)\n"
         "2) When the desired filename is displayed, Tab to 'Matching Files' control,\n"
         "   highlight target filename and press Enter key to set target directory.\n"
         "Press the RETURN Pushbutton to return without setting the target directory.", dColor ) ;
      }
      else
      {
         //* Display Mountpoint label *
         this->TrimPathString ( gscwd, (ic[ffmatTB].cols - 13) ) ;
         gscwd.insert( "MOUNT POINT: " ) ;

         dp->WriteParagraph ( instrPos,
         "1) The highlighted file provides the target INODE on entry. \"Matching Files\"\n"
         "   lists all files of the current filesystem which share the specified INODE.\n"
         "2) To scan for a specific INODE, enter it manually and then press Enter key.\n"
         "3) Navigate to any Matching File: highlight the filename and press Enter.\n"
         "Press the RETURN Pushbutton to return without setting the target directory.", dColor ) ;
      }
      dp->WriteString ( ic[ffmatTB].ulY - 1, ic[ffmatTB].ulX, gscwd, dColor ) ;

      //* Make a visual connection for the dctSCROLLEXT control that *
      //* overlaps the dialog window's border.                       *
      cdConnect cdConn ;
      cdConn.ul2Left = true ;
      cdConn.ur2Right = true ;
      cdConn.connection = true ;
      dp->ConnectControl2Border ( ffSE, cdConn ) ;
      dp->RefreshWin () ;
   }
   return dp ;

}  //* End ffOpenDialog() *

//*************************
//*  ffDynamicAllocation  *
//*************************
//********************************************************************************
//* Private Method:                                                              *
//* ---------------                                                              *
//* Manage dynamic allocation and release of space for Scrollext display data.   *
//*                                                                              *
//* Input  : sData  : receives the display data for matching filenames           *
//*          noJoy  : (optional, 'true' by default)                              *
//*                   if 'true'  initialize the 'no-match' message               *
//*                   if 'false' do not modify existing display data             *
//*          release: (optional, 'false' by default)                             *
//*                   'false' :                                                  *
//*                        a) if no previous allocation, allocate data space.    *
//*                        b) initialize the 'no-match' message as the only      *
//*                           display item                                       *
//*                   'true'  : release the dynamic memory allocation            *
//*                                                                              *
//* Returns: 'true' if successful, else 'false'                                  *
//*          -- For allocation: The members of 'sData' will point to the         *
//*             dynamically-allocated data                                       *
//*          -- For release: The members of 'sData' will be set to NULL          *
//*             pointers.                                                        *
//********************************************************************************
//* Notes:                                                                       *
//* ------                                                                       *
//* 1) gsDFLTBYTES == MAX_PATH i.e. space for a full path/filename spec,         *
//*    so each display item is potentially 4KB in length, although of course     *
//*    the window lacks the display area to display that much data.              *
//* 2) The list of matching files should ideally be sized dynamically; however,  *
//*    dynamically allocating the data arrays each time the search criteria      *
//*    change is both slow and prone to memory leaks.                            *
//*    For this reason, we limit the number of matching items to be displayed    *
//*    to 'maxMATCH' (defined at the top of this module).                        *
//*    a) This is far more than any sane user would want to examine, so it's     *
//*       likely that the user will never know about the limit.                  *
//*    b) We report the actual number of filename matches to the user, NOT       *
//*       the number of matches displayed.                                       *
//*                                                                              *
//*                                                                              *
//********************************************************************************

bool FileDlg::ffDynamicAllocation ( ssetData& sData, bool noJoy, bool release )
{
   static char*   blkPtr  = NULL ;      // raw text data
   static char**  itemPtr = NULL ;      // item pointers
   static attr_t* attrPtr = NULL ;      // color attributes

   bool status = false ;

   //* Allocate (or re-intialize) the data space *
   if ( ! release )
   {
      if ( blkPtr == NULL )
      {
         for ( short i = 3 ; i > ZERO ; --i )
         {
            if ( (blkPtr = new (std::nothrow) char[gsDFLTBYTES * (maxMATCH + 1)]) != NULL )
               break ;
         }
      }
      if ( itemPtr == NULL )
      {
         for ( short i = 3 ; i > ZERO ; --i )
         {
            if ( (itemPtr = new (std::nothrow) char*[maxMATCH + 1]) != NULL )
               break ;
         }
      }
      if ( attrPtr == NULL )
      {
         for ( short i = 3 ; i > ZERO ; --i )
         {
            if ( (attrPtr = new (std::nothrow) attr_t[maxMATCH + 1]) != NULL )
               break ;
         }
      }

      if ( blkPtr != NULL && itemPtr != NULL && attrPtr != NULL )
         status = true ;

      //* Initialize the text pointer array and the color-attribute array *
      int cOffset = ZERO ;
      for ( int i = ZERO ; i < maxMATCH ; ++i )
      {
         itemPtr[i] = &blkPtr[cOffset] ;
         cOffset += gsDFLTBYTES ;
         attrPtr[i] = this->ftColor[fmREG_TYPE] ;
      }

      //* Initialize caller's data container *
      sData.dispText  = (const char**)itemPtr ;
      sData.dispColor = attrPtr ;
      sData.dispItems = 1 ;
      sData.hlIndex = ZERO ;
      sData.hlShow = false ;

      //* Set the 'no-match' message *
      if ( noJoy )
      {
         gString gsFmt( noJOYMSG ) ;
         gsFmt.copy( itemPtr[ZERO], gsDFLTBYTES ) ;
      }
   }

   //* Release the dynamic allocation *
   else
   {
      if ( blkPtr != NULL )   { delete [] blkPtr ;    blkPtr  = NULL ; }
      if ( itemPtr != NULL )  { delete [] itemPtr ;   itemPtr = NULL ; }
      if ( attrPtr != NULL )  { delete [] attrPtr ;   attrPtr = NULL ; }
      sData.dispText = NULL ;
      sData.dispColor = NULL ;
      sData.dispItems = sData.hlIndex = ZERO ;
      sData.hlShow = false ;
      status = true ;
   }
   return status ;

}  //* End ffDynamicAllocation() *

//*************************
//*      ffScanTree       *
//*************************
//********************************************************************************
//* Private Method:                                                              *
//* Scan all files in the directory tree at and below the CWD for filenames      *
//* that contain the specified substring.                                        *
//*                                                                              *
//* Input  : substr : (by reference) substring for which to search               *
//*          sData  : (by reference) receives the display data for               *
//*                    matching filenames                                        *
//*          tsFmt  : formatting code for timestamp to be displayed              *
//*                                                                              *
//* Returns: number of matching filenames                                        *
//*          (this is not necessarily the same as number of display items)       *
//********************************************************************************
//* Notes:                                                                       *
//* ------                                                                       *
//* -- Because the dispText and dispColor members of the ssetData class are      *
//*    are pointers to 'const', we play a trick to initialize the data arrays.   *
//*                                                                              *
//* -- The display string MAY BE wider than the display area. This means that    *
//*    the display output for that line must be shifted to bring the tail of     *
//*    the string into view. This is handled by caller.                          *
//*                                                                              *
//********************************************************************************

int FileDlg::ffScanTree ( const gString& substr, ssetData& sData, int tsFmt )
{
   gString  fName,         // file name (not path)
            nodePath ;     // relative path of subdirectory at this level
   int   fMatch = ZERO ;   // return value

   //* Reset display indices *
   sData.dispItems = sData.hlIndex = ZERO ;
   sData.hlShow = false ;

   //* Get a pointer to the base TreeNode object *
   const TreeNode* tnPtr = this->fmPtr->GetBaseNode () ;

   #if DEBUG_FIND != 0
   gString gsdbg( "$HOME/Documents/SoftwareDesign/FileMangler/1_TestData/temp/filog.txt" ) ;
   this->fmPtr->EnvExpansion ( gsdbg ) ;
   ofs.open( gsdbg.ustr(), ofstream::out | ofstream::app ) ;
   if ( ofs.is_open() )
   {
      ofs << "DEBUG ffScanTree : '" << tnPtr->dirStat.fName << "'\n" 
          << "      substr: " << substr.ustr() 
          << "\n=====================================" << endl ;
   }
   #endif   //DEBUG_FIND

   //* Scan the directory tree beginning with the top node i.e. the CWD *
   fMatch = this->ffScanNode ( tnPtr, substr, nodePath, sData, tsFmt ) ;

   //* If no matching items were found, restore the 'no-match' message *
   if ( fMatch == ZERO )
      sData.dispItems = 1 ;

   #if DEBUG_FIND != 0
   if ( ofs.is_open() )
      ofs.close() ;
   #endif   //DEBUG_FIND

   return fMatch ;

}  //* End ffScanTree()

//*************************
//*      ffScanNode       *
//*************************
//********************************************************************************
//* Private Method.                                                              *
//* ---------------                                                              *
//* Recursively scan the specified node of the directory tree searching for      *
//* filenames which match the specified substring.                               *
//*                                                                              *
//*                                                                              *
//* Input  : nodePtr : directory-tree node to scan (by reference)                *
//*          substr  : substring for which to search                             *
//*          tnPath  : relative path from CWD to current node of dir tree        *
//*          sData   : (by reference) display data for matching filenames is     *
//*                    appended to any existing data.                            *
//*          tsFmt   : formatting code for timestamp to be displayed             *
//*                                                                              *
//* Returns: number of matching filenames                                        *
//*          (this is not necessarily the number of display items added)         *
//********************************************************************************

int FileDlg::ffScanNode ( const TreeNode* nodePtr, const gString& substr, 
                          const gString& tnPath, ssetData& sData, int tsFmt )
{
   gString  fPath,                                 // relative file path
            fName,                                 // filename
            subPath,                               // path for recursive calls
            gsfmt ;                                // text formatting
   char**   ptrArray = (char**)sData.dispText ;    // text pointer
   attr_t*  atrArray = (attr_t*) sData.dispColor ; // color-attribute pointer
   int fMatch = ZERO ;                             // return value

   #if DEBUG_FIND != 0
   if ( ofs.is_open() )
      ofs << "\nffScanNode(" << nodePtr->dirStat.fName << ")" << endl ;
   #endif   //DEBUG_FIND

   //* Scan lower-level nodes (if any) *
   if ( nodePtr->nextLevel != NULL && nodePtr->dirFiles > ZERO )
   {
      for ( UINT i = ZERO ; i < nodePtr->dirFiles ; ++i )
      {
         #if DEBUG_FIND != 0
         if ( ofs.is_open() )
            ofs << "  dN: '" << nodePtr->nextLevel[i].dirStat.fName << "'" << endl ;
         #endif   //DEBUG_FIND

         fName = nodePtr->nextLevel[i].dirStat.fName ;
         if ( tnPath.gschars() > 1 )
            fPath.compose( "%S/%S", tnPath.gstr(), fName.gstr() ) ;
         else
            fPath = fName ;
         if ( (fName.find( substr )) >= ZERO )
         {  //* Substring match found. Add it to the list.*
            if ( sData.dispItems < maxMATCH )   // if freespace in the list
            {
               switch ( tsFmt )
               {
                  case 1:
                     gsfmt.compose( "%04hd-%02hd-%02hd | %S",
                                    &nodePtr->nextLevel[i].dirStat.modTime.year,
                                    &nodePtr->nextLevel[i].dirStat.modTime.month,
                                    &nodePtr->nextLevel[i].dirStat.modTime.date,
                                    fPath.gstr() ) ;
                     break ;
                  case 2:
                     gsfmt.compose( "%04hd-%02hd-%02hdT%02hd:%02hd:%02hd | %S",
                                    &nodePtr->nextLevel[i].dirStat.modTime.year,
                                    &nodePtr->nextLevel[i].dirStat.modTime.month,
                                    &nodePtr->nextLevel[i].dirStat.modTime.date,
                                    &nodePtr->nextLevel[i].dirStat.modTime.hours,
                                    &nodePtr->nextLevel[i].dirStat.modTime.minutes,
                                    &nodePtr->nextLevel[i].dirStat.modTime.seconds,
                                    fPath.gstr() ) ;
                     break ;
                  default:
                     gsfmt = fPath ;
                     break ;
               }
               gsfmt.copy( ptrArray[sData.dispItems], gsDFLTBYTES ) ;
               atrArray[sData.dispItems] = this->ftColor[nodePtr->nextLevel[i].dirStat.fType] ;
               ++sData.dispItems ;
            }
            ++fMatch ;
         }

         //* If this node contains directory names *
         if ( (nodePtr->nextLevel[i].nextLevel != NULL) && 
              (nodePtr->nextLevel[i].dirFiles > ZERO) )
         {
            #if DEBUG_FIND != 0
            if ( ofs.is_open() )
               ofs << "    nextLevel[" << nodePtr->nextLevel[i].dirFiles << "]"
                   << "'" << fPath.ustr() << "'" << endl ;
            #endif   //DEBUG_FIND
            fMatch += this->ffScanNode ( &nodePtr->nextLevel[i], 
                                         substr, fPath, sData, tsFmt ) ;
         }

         //* If no directory names, we didn't recurse, *
         //* so scan non-directory list (if any).      *
         else if ( nodePtr->nextLevel[i].tnFiles != NULL && 
                   nodePtr->nextLevel[i].tnFCount > ZERO )
         {
            fMatch += this->ffScanList ( nodePtr->nextLevel[i].tnFiles,
                                         nodePtr->nextLevel[i].tnFCount,
                                         substr, fPath, sData, tsFmt ) ;
         }
         #if DEBUG_FIND != 0
         else if ( ofs.is_open() )
            ofs << "  dn: '" << fPath.ustr() << "' (empty)" << endl ;
         #endif   //DEBUG_FIND
      }
   }

   //* Scan the non-directory filename list *
   fMatch += this->ffScanList ( nodePtr->tnFiles, nodePtr->tnFCount, 
                                substr, tnPath, sData, tsFmt ) ;

   return fMatch ;

}  //* End ffScanNode() *

//*************************
//*      ffScanList       *
//*************************
//******************************************************************************
//* Private Method.                                                            *
//* Scan a list of non-directory files searching for filenames which match     *
//* the specified substring.                                                   *
//*                                                                            *
//* Input  : listPtr : pointer to array of tnFName objects                     *
//*          listCnt : number of items in 'listPtr' array                      *
//*          substr  : substring for which to search                           *
//*          listPath: relative path of directory in which the files live      *
//*          sData   : (by reference) display data for matching filenames is   *
//*                    appended to any existing data.                          *
//*          tsFmt   : formatting code for timestamp to be displayed           *
//*                                                                            *
//* Returns: number of matching filenames                                      *
//*          (this is not necessarily the number of display items added)       *
//******************************************************************************

int FileDlg::ffScanList ( const tnFName* listPtr, UINT listCnt, const gString& substr, 
                          const gString& listPath, ssetData& sData, int tsFmt )
{
   gString fPath,             // relative file path
           fName,             // filename
           gsfmt ;            // text formatting
   char**  ptrArray = (char**)sData.dispText ;    // text pointer
   attr_t* atrArray = (attr_t*) sData.dispColor ; // color-attribute pointer
   int     fMatch = ZERO ;    // return value

   for ( UINT i = ZERO ; i < listCnt ; ++i )
   {
      #if DEBUG_FIND != 0
      if ( ofs.is_open() )
         ofs << "  fN: '" << listPtr[i].fName 
             << "'  fType:" << listPtr[i].fType << endl ;
      #endif   //DEBUG_FIND

      fName = listPtr[i].fName ;
      if ( (fName.find( substr )) >= ZERO )
      {  //* Substring match found. Add it to the list.*
         if ( sData.dispItems < maxMATCH )   // if freespace in the list
         {
            if ( listPath.gschars() > 1 )
               fPath.compose( "%S/%S", listPath.gstr(), fName.gstr() ) ;
            else
               fPath = fName ;
            switch ( tsFmt )
            {
               case 1:
                  gsfmt.compose( "%04hd-%02hd-%02hd | %S",
                                 &listPtr[i].modTime.year,
                                 &listPtr[i].modTime.month,
                                 &listPtr[i].modTime.date,
                                 fPath.gstr() ) ;
                  break ;
               case 2:
                  gsfmt.compose( "%04hd-%02hd-%02hdT%02hd:%02hd:%02hd | %S",
                                 &listPtr[i].modTime.year,
                                 &listPtr[i].modTime.month,
                                 &listPtr[i].modTime.date,
                                 &listPtr[i].modTime.hours,
                                 &listPtr[i].modTime.minutes,
                                 &listPtr[i].modTime.seconds,
                                 fPath.gstr() ) ;
                  break ;
               default:
                  gsfmt = fPath ;
                  break ;
            }
            gsfmt.copy( ptrArray[sData.dispItems], gsDFLTBYTES ) ;
            atrArray[sData.dispItems] = this->ftColor[listPtr[i].fType] ;
            ++sData.dispItems ;
         }
         ++fMatch ;
      }
   }        // for(;;)
   return fMatch ;

}  //* End ffScanList() *

//*************************
//*   ffGetSelectedItem   *
//*************************
//********************************************************************************
//* 1) Extract the specified (relative) path from the array of display data.     *
//* 2) Construct the full filespec.                                              *
//*    a) If filespec references a filename, then strip the filename.            *
//*    b) If filespec references a directory name, we're done.                   *
//*                                                                              *
//* NOTE: Reinitializes the array of display pointers.                           *
//*       This must be done because the pointers may have been adjusted to get   *
//*       the data to fit in the display area. Note that this operation trashes  *
//*       everything in the 'sData' object EXCEPT the actual text data, but we   *
//*       only need the target string, AND we assume that caller no longer       *
//*       needs the data.                                                        *
//*                                                                              *
//*                                                                              *
//* Input  : newCwd   : (by reference)                                           *
//*                     On entry:                                                *
//*                      if clear: 'iIndex' is correct as received               *
//*                      else, contains the display text for the selected        *
//*                      display item.                                           *
//*                     On return: receives the target path string               *
//*          sData    : contains the display data for matching filenames         *
//*          iIndex   : index of user-selected item (display item)               *
//*                                                                              *
//* Returns: 'true' if successful, else 'false' (currently, always successful)   *
//********************************************************************************

bool FileDlg::ffGetSelectedItem ( gString& newCwd, ssetData& sData, short iIndex )
{
//   int itemCnt = sData.dispItems ;        // save current item count
   bool status = true ;                   // return value

   //* Reinitialize the text-pointer array (do not modify the text data) *
   this->ffDynamicAllocation ( sData, false ) ;

   //* Offset into display string for filespec *
   gString gstmp = sData.dispText[iIndex] ;
   short fspecOff = (gstmp.after( L'|' )) + 1 ;

   newCwd.compose( "%s/%S", this->currDir, &gstmp.gstr()[fspecOff] ) ;
   fmFType ft ;
   this->TargetExists ( newCwd, ft ) ;    // get the filetype
   if ( ft != fmDIR_TYPE )
   {  //* Truncate the filespec by removing the terminating filename *
      short tIndex = newCwd.findlast( fSLASH ) ;
      newCwd.limitChars( tIndex ) ;
   }

   //* If the target is the CWD, then no change of directory is needed *
   if ( (newCwd.compare( this->currDir )) == ZERO )
      status = false ;

   return status ;

}  //* End ffGetSelecteditem() *

//*************************
//*    ffAdjustOutput     *
//*************************
//******************************************************************************
//* Test the width of the display strings. If a string is too wide for the     *
//* display area, adjust the string pointer so the tail of the string will be  *
//* visible in the control.                                                    *
//*                                                                            *
//* Input  : sData  : contains the display data for matching filenames         *
//*          dCols  : number of display columns in Scrollext control.          *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************
//* Programmer's Note: This method analyzes the incomming text data, and if    *
//* an item contains more columns than can be displayed in the target control, *
//* the pointer to the string is advanced by the number of CHARACTERS          *
//* equivalent to the excess COLUMNS.                                          *
//*                                                                            *
//* Of course this means that the pointer no longer references the head of     *
//* the string, so if the string is needed for something other than display,   *
//* then it will have to be retrieved from the raw-data array. Beware!         *
//*                                                                            *
//******************************************************************************

void FileDlg::ffAdjustOutput ( ssetData& sData, short dCols )
{
   gString gs ;                                 // for analysis
   char**  ptrArray = (char**)sData.dispText ;  // text pointer
   short   txtCols,                             // text columns in display string
           txtChars,                            // text characters in display string
           colShift ;                           // columns to be shifted out

   for ( int i = ZERO ; i < sData.dispItems ; ++i )
   {
      gs = ptrArray[i] ;
      txtChars = gs.gschars() ;
      if ( (txtCols = gs.gscols()) > dCols )
      {
         colShift = -(txtCols - dCols) ;
         gs.shiftCols( colShift ) ;
         ptrArray[i] += txtChars - (gs.gschars() - 1) ;
      }
   }
}  //* End ffAdjustOutput() *


//******************************************************************************
//***                   'Find Inodes' Method Group                           ***
//******************************************************************************

//*************************
//*      FindInodes       *
//*************************
//******************************************************************************
//* Scan the the fileystem which contains the current-working-directory (CWD). *
//* Locate all files (hard links) which share either:                          *
//*  a) the user-specified 'inode'                                             *
//*  b) the inode of the highlighted file                                      *
//*                                                                            *
//* Input  : newCwd    : (by reference) receives user's path selection         *
//*                      (cleared if returned to initial CWD)                  *
//*          ul        : (by reference) upper left corner dialog position      *
//*          rows      : number of rows for dialog                             *
//*          cols      : number of columns for dialog                          *
//*                                                                            *
//* Returns: true if user has specified a new CWD ('newCwd' contains new path) *
//*          else false                                                        *
//******************************************************************************
//* Notes:                                                                     *
//* ------                                                                     *
//* 1) In order to scan properly, we must have access to all directories and   *
//*    files of the filesystem _including_ hidden ones.                        *
//*    a) Therefore, hidden files are set as 'visible' before we read the      *
//*       contents of the full filesystem.                                     *
//*    b) When the search is complete, we could optionally return the hidden   *
//*       files to the 'invisible' state; however this would remove user's     *
//*       access to any hidden file which share the specified inode.           *
//*    c) If hidden files ARE already visible, then we can skip these steps.   *
//*                                                                            *
//* 2) Inode of the highlighted file in the original CWD provides the default  *
//*    search criterion, and would usually be the user's desired target;       *
//*    however, for user convenience, we must offer the option to manually     *
//*    enter an inode number.                                                  *
//*    a) If the number of hard links to the target file == 1, then the search *
//*       need not be executed, but can/should be, to compensate for any       *
//*       filesystem error.                                                    *
//*    b) If the user manually enters an inode number, then that number will   *
//*       be the search criterion.                                             *
//*    c) We _could_ also have an option to display all files in the CWD which *
//*       have more than one hard link.                                        *
//*                                                                            *
//* 3) The 'df' utility is used (indirectly) to obtain the mountpoint to which *
//*    the target file is attached. This becomes the base directory for our    *
//*    search.                                                                 *
//*                                                                            *
//* 4) We re-purpose some of the methods for the FindFiles() group to create   *
//*    and manage the sub-dialog window. For 'ffOpenDialog()', this is done by *
//*    setting the 'fi' flag for the call. The following additional 'FindFiles'*
//*    methods are also re-used:                                               *
//*    -- ffDynamicAllocation()                                                *
//*    -- ffAdjustOutput()                                                     *
//*                                                                            *
//* At the command line, we would use the 'find' command.                      *
//* find /home -xdev -samefile /home/sam/1_TestData/targetfile.txt             *
//*   cd /home                                                                 *
//*   find -inum 12345678                                                      *
//*                                                                            *
//* sudo find /usr -type f -links +1 -printf '%i %n %p\n' | grep 2634639       *
//* ...                                                                        *
//* 2634639 4 /usr/share/zoneinfo/NZ                                           *
//*                                                                            *
//* sudo find /usr -type f -links +1 -printf '%i %n %p\n' | grep 2634639       *
//* [sudo] password for sam:                                                   *
//* 2634639 4 /usr/share/zoneinfo/Pacific/Auckland                             *
//* 2634639 4 /usr/share/zoneinfo/Antarctica/McMurdo                           *
//* 2634639 4 /usr/share/zoneinfo/Antarctica/South_Pole                        *
//* 2634639 4 /usr/share/zoneinfo/NZ                                           *
//*                                                                            *
//* List multi-hardlink files in local tree:                                   *
//* stat /home/sam/* | grep -B2 'Links:'                                       *
//* Unfortunately, only the primary is found in the scan...                    *
//*                                                                            *
//* Useful explanation of hard links:  http://www.linfo.org/hard_link.html     *
//******************************************************************************

bool FileDlg::FindInodes ( gString& newCwd, winPos& ul, short rows, short cols )
{
   gString origcwd, gsMpt( "unknown" ), gsTrg, gstmp ;
   this->GetPath ( origcwd ) ;      // where we are now
   tnFName trgStats ;               // receives stats for highlighted file
   fileSystemStats fsStats ;        // filesystem stat info
   bool  newCWD = false ;           // return value

   //* Reset caller's gString object *
   newCwd.clear() ;

   //* Locate the mountpoint directory for the filesystem *
   //* containing the highlighted file.                   *
   this->GetStats ( trgStats ) ;    // get stats of highlighted file
   this->fmPtr->CatPathFname ( gsTrg, this->currDir, trgStats.fName ) ; // create filespec
   this->fmPtr->FilesystemID ( gsTrg, gsMpt, false ) ;

   //* Enable scanning of hidden directories and files (see note above).*
   bool old_sh = this->cfgOptions.showHidden ;
   if ( ! old_sh )
   {
      if ( (this->fmPtr->ShowHiddenFiles ( true )) == OK )
         this->cfgOptions.showHidden = true ;
      else              // prevent the switch back at bottom of method
         old_sh = true ;
   }

   //* Enable recursive scan from the root directory. The root     *
   //* directory is often the mountpoint for the target filesystem.*
   bool old_sfr = this->fmPtr->ScanFromRoot ( true, true ) ;
   if ( ! old_sfr )
      this->fmPtr->ScanFromRoot ( true ) ;

   //* Set the CWD to the mountpoint directory so *
   //* we can scan all files in the filesystem.   *
   this->SetDirectory ( gsMpt ) ;

   //* Save the display for the parent dialog window *
   this->dPtr->SetDialogObscured () ;

   //* Open the Find-file Dialog *
   NcDialog* dp = this->ffOpenDialog ( ul, rows, cols, true ) ;

   //* Initialize the Scrollext data array.      *
   //* (initially one item: the 'empty' message) *
   ssetData sData ;
   this->ffDynamicAllocation ( sData ) ;
   dp->SetScrollextText ( ffSE, sData ) ;    // display the data

   uiInfo   Info ;                            // user interface data returned here
   short    icIndex = dp->GetCurrControl () ; // index of control with input focus
   bool done = (dp == NULL) ? true : false ; // loop control

   //* Set contents of search textbox to INODE value.*
   while ( (icIndex = dp->NextControl ()) != ffRetPB ) ;
   gstmp.compose( "%llu    (%s)", &trgStats.rawStats.st_ino, trgStats.fName ) ;
   dp->SetTextboxText ( ffmatTB, gstmp ) ;

   //* Scan the filesystem for files which match the default INODE *
   //* and display the results in the Scrollext control.           *
   this->fiScanInodes ( dp, sData, origcwd, (cols - 2) ) ;

   dp->RefreshWin () ;        // make everything visible

   while ( ! done )
   {
      if ( icIndex == ffRetPB )
      {
         if ( Info.viaHotkey )
            Info.HotData2Primary () ;
         else
            icIndex = dp->EditPushbutton ( Info ) ;

         if ( Info.dataMod != false )
            done = true ;
      }

      else if ( icIndex == ffmatTB )
      {
         Info.viaHotkey = false ;
         icIndex = dp->EditTextbox ( Info ) ;

         //* If user has specified a new INODE, re-scan  *
         //* the filesystem and update the filename list.*
         if ( Info.dataMod )
         {
            //* Reformat Textbox contents (discard everything after the INODE).*
            dp->GetTextboxText ( ffmatTB, gstmp ) ;  // get textbox contents
            short i = gstmp.find( nckSPACE ) ;
            if ( i > ZERO  )
            {
               gstmp.limitChars( i ) ;
               icIndex = dp->NextControl () ;
               dp->SetTextboxText ( ffmatTB, gstmp ) ;
               icIndex = dp->PrevControl () ;
            }
            //* Re-scan the filesystem and update the display.*
            this->fiScanInodes ( dp, sData, origcwd, (cols - 2) ) ;
         }

         //* Allow item selection ONLY if there are file-match data *
         gstmp = sData.dispText[ZERO] ;
         if ( (gstmp.compare( noJOYMSG )) != ZERO )
            dp->ControlActive ( ffSE, true ) ;  // activate Scrollext control
         else
         {
            //* If the Scrollext control has focus, BUT contains no live data, *
            //* then move the focus to Pushbutton and (thence back to the      *
            //* Textbox). See note above.                                      *
            if ( icIndex == ffSE )
               icIndex = dp->NextControl () ;
            dp->ControlActive ( ffSE, false ) ; // deactivate Scrollext control
         }
      }

      else if ( icIndex == ffSE )
      {
         Info.viaHotkey = false ;      // ignore hotkey data
         icIndex = dp->EditScrollext ( Info ) ;

         //* If user has made a selection *
         // Programmer's Note: 'dataMod' is set if the highlight has moved 
         // OR if the Enter keys or the Space key have been pressed. However,
         // we don't care whether the highlight has moved, only where it IS.
         // So to protect ourselves, we directly test the key data.
         if ( (Info.wk.type == wktPRINT && Info.wk.key == nckSPACE) ||
              ((Info.wk.type == wktFUNKEY) && 
               (Info.wk.key == nckENTER || Info.wk.key == nckpENTER)) )
         {
            //* Reinitialize the text-pointer array (text data not modified).*
            this->ffDynamicAllocation ( sData, false ) ;
            //* Get the selected item, i.e. the target filespec. *
            //* Then seperate target directory name from target  *
            //* filename.                                        *
            gstmp = sData.dispText[Info.selMember] ;
            this->fmPtr->ExtractPathname ( newCwd, gstmp ) ;
            this->fmPtr->ExtractFilename ( gsTrg, gstmp ) ;
            newCWD = done = true ;
         }
      }

      //* Move focus to appropriate control *
      if ( ! done && ! Info.viaHotkey )
      {
         if ( Info.keyIn == nckSTAB )
            icIndex = dp->PrevControl () ; 
         else
            icIndex = dp->NextControl () ;
      }
   }     // while(!done)

   //* Close the Find-file dialog *
   if ( dp != NULL )
      delete ( dp ) ;

   //* Release our dynamic memory allocation *
   this->ffDynamicAllocation ( sData, true, true ) ;

   //* If "hidden" files were not visible on entry, *
   //* return them to the "hidden" state.           *
   if ( ! old_sh )
      ;  // (see note above)

   this->dPtr->RefreshWin () ;      // restore the parent dialog

   //* Set the CWD.                             *
   //* a) Navigate to user-specified directory. *
   //* b) Return to caller's initial directory. *
   this->SetDirectory ( (newCWD ? newCwd : origcwd) ) ;

   //* Highlight the new target file.*
   this->HilightItemByName ( gsTrg ) ;

   //* Disable recursive scan from the root directory. *
   if ( ! old_sfr )
      this->fmPtr->ScanFromRoot ( false ) ;

   return newCWD ;

}  //* End FindInodes() *

//*************************
//*     fiScanInodes      *
//*************************
//******************************************************************************
//* Private Method.                                                            *
//* Extract the reference INODE from the dialog's input Textbox, then scan     *
//* the entire filesystem for:                                                 *
//*  A) all filenames with the same INODE.                                     *
//*  B) all filenames associated with multiple hard links.                     *
//* Format the results for display.                                            *
//*                                                                            *
//* Input  : dp     : pointer to caller's dialog window                        *
//*          sData  : (by reference) receives the display data for             *
//*                   files with matching Inode(s)                             *
//*          origCwd: (by reference) CWD when user invoked Inode scan          *
//*                   references ONLY when scanning for multiple hard links    *
//*          dCols  : number of display columns in Scrollext control           *
//*                   (used for formatting the display data)                   *
//*                                                                            *
//* Returns: multi==false: count of hard links found _including_ the original  *
//*                        target file                                         *
//*          multi!=false: count of multi-hard-link files found                *
//******************************************************************************

int FileDlg::fiScanInodes ( NcDialog* dp, ssetData& sData, const gString& origCwd, short dCols )
{
   gString inodeText,                     // contents of dialog Textbox
           ifmt ;                         // for integer formatting
   UINT64   refInode = ZERO ;             // reference inode
   int      hardLinks = ZERO ;            // number of hard links

   this->ffDynamicAllocation ( sData ) ;  // re-initialize display data arrays

   dp->GetTextboxText ( ffmatTB, inodeText ) ;  // get textbox contents

   //* This test for the word "multi" is an undocumented option to scan *
   //* the tree for all non-directory filenames that report multiple    *
   //* hard links. Used primarily for development and debugging.        *
   if ( (inodeText.compare( "multi" )) == ZERO )
      hardLinks = this->fiScanTree ( origCwd, sData, dp ) ;
   else if ( (swscanf ( inodeText.gstr(), L"%llu", &refInode )) == 1 )
      hardLinks = this->fiScanTree ( refInode, sData ) ;

   //* If we have matching files, adjust the string offsets *
   //* to ensure that the data will fit in the display area.*
   if ( hardLinks > ZERO )
      this->ffAdjustOutput ( sData, dCols ) ;

   short oldIndex = dp->GetCurrControl () ;
   while ( (dp->NextControl ()) != ffRetPB ) ;
   dp->SetScrollextText ( ffSE, sData ) ;    // display the data
   ifmt.formatInt( hardLinks, 5 ) ;          // update match count
   dp->SetTextboxText ( ffcntTB, ifmt ) ;
   while ( (dp->NextControl ()) != oldIndex ) ;

   return hardLinks ;

}  //* End fiScanInodes() *

//*************************
//*      fiScanTree       *
//*************************
//******************************************************************************
//* Private Method.                                                            *
//* Scan the entire filesystem tree for files with the same INODE as the       *
//* reference INODE.                                                           *
//* Format the results for display.                                            *
//*                                                                            *
//* Input  : refInode : reference INODE against which to test target file      *
//*          sData    : (by reference) receives the display data for           *
//*                     files with matching Inode(s)                           *
//*                                                                            *
//* Returns: count of hard links found _including_ the original target file    *
//******************************************************************************
//* Notes:                                                                     *
//* ------                                                                     *
//* -- The display string MAY BE wider than the display area. This means that  *
//*    the display output for that line must be shifted to bring the tail of   *
//*    the string into view. This is handled by caller.                        *
//*                                                                            *
//******************************************************************************

int FileDlg::fiScanTree ( UINT64 refInode, ssetData& sData )
{
   gString  nodePath( this->currDir ) ;   // path of filesystem base node
   int      hardLinks = ZERO ;            // number of hard links

   //* Reset display indices *
   sData.dispItems = sData.hlIndex = ZERO ;
   sData.hlShow = false ;

   //* Get a pointer to the base TreeNode object *
   const TreeNode* tnPtr = this->fmPtr->GetBaseNode () ;

   #if DEBUG_INODE != 0
   gString gsdbg( "$HOME/Apps/FileMangler/Temp/filog.txt" ) ;
   this->fmPtr->EnvExpansion ( gsdbg ) ;
   ofs.open( gsdbg.ustr(), ofstream::out | ofstream::trunc ) ;
   if ( ofs.is_open() )
   {
      ofs << "DEBUG fiScanTree : '" << tnPtr->dirStat.fName 
          << "'      refInode: " << refInode 
          << "\n====================================================" << endl ;
   }
   else
      this->dPtr->DebugMsg ( "Logfile not opened.", 3, true ) ;
   #endif   //DEBUG_INODE

   //* Test the mountpoint directory.*
   if ( tnPtr->dirStat.rawStats.st_ino == refInode )
   {
      if ( sData.dispItems < maxMATCH )   // if freespace in the list
      {
         char**   ptrArray = (char**)sData.dispText ;    // text pointer (see note above)
         attr_t*  atrArray = (attr_t*) sData.dispColor ; // color-attribute pointer
         gString gsMpt( this->currDir ) ;                // CWD is mountpoint
         gsMpt.copy( ptrArray[sData.dispItems], gsDFLTBYTES ) ;    // set display text
         atrArray[sData.dispItems] = this->ftColor[tnPtr->dirStat.fType] ; // set display attribute
         ++sData.dispItems ;              // increment display-item count
      }
      ++hardLinks ;
   }

   //* Scan the directory tree beginning with the top node i.e. the CWD *
   hardLinks += this->fiScanNode ( nodePath, tnPtr, sData, refInode ) ;

   #if DEBUG_INODE != 0
   if ( ofs.is_open() )
      ofs.close() ;
   #endif   //DEBUG_INODE

   return hardLinks ;

}  //* End fiScanTree() *

//*************************
//*      fiScanTree       *
//*************************
//******************************************************************************
//* Private Method.                                                            *
//* Scan the entire filesystem tree for non-directory files with multiple      *
//* hard links.                                                                *
//* Format the results for display.                                            *
//*                                                                            *
//* Input  : origCwd  : (by reference) base directory for scan                 *
//*          sData    : (by reference) receives the display data for           *
//*                     files matching the criteria                            *
//*                                                                            *
//* Returns: count of files found with multiple hard links                     *
//******************************************************************************

//int FileDlg::fiScanTree ( const gString& origCwd, ssetData& sData )
int FileDlg::fiScanTree ( const gString& origCwd, ssetData& sData, NcDialog* dp )
{
   gString  nodePath( this->currDir ),    // path of filesystem base node
            gstmp, gsdirName ;
   UINT  nlindx = ZERO ;                  // index for TreeNode arrays
   int   hardLinks = ZERO ;               // return value
   short indx = 1 ;                       // index for scanning text data

   //* Reset display indices *
   sData.dispItems = sData.hlIndex = ZERO ;
   sData.hlShow = false ;

   //* Get a pointer to the base TreeNode object *
   //* and find the node which matches origCwd.  *
   const TreeNode* tnPtr = this->fmPtr->GetBaseNode () ;

   gstmp = origCwd ;
   while ( nodePath != origCwd )
   {
      indx = gstmp.find( fSLASH, indx ) ;
      if ( indx > ZERO )
      {
         //* Special case: root directory *
         if ( (nodePath.compare( ROOT_PATH)) == ZERO )
         {
            gstmp.shiftChars( -(1) ) ;
            nodePath.clear() ;
         }
         else
            gstmp.shiftChars( -(indx + 1) ) ;
      }

      //* Unable to find target node. Should not happen *
      //* if origCwd is at or below this->currDir.      *
      else
         break ;

      gsdirName = gstmp ;
      indx = gsdirName.find( fSLASH ) ;
      if ( indx > ZERO )               // if not the last name in the path
         gsdirName.limitChars( indx ) ;

      for ( nlindx = ZERO ; nlindx < tnPtr->dirFiles ; ++nlindx )
      {
         if ( (gsdirName.compare( tnPtr->nextLevel[nlindx].dirStat.fName, true )) == ZERO )
         {
            nodePath.append( "/%s", tnPtr->nextLevel[nlindx].dirStat.fName ) ;
            dp->ClearLine ( 6, nc.maR ) ; dp->WriteString ( 6, 1, nodePath, nc.maR, true ) ;
            tnPtr = &tnPtr->nextLevel[nlindx] ;
            break ;
         }
      }
      indx = ZERO ;
   }     // while()

   //* Scan the directory tree beginning at the calculated node.*
   if ( nodePath == origCwd )
      hardLinks = this->fiScanNode ( nodePath, tnPtr, sData ) ;

   return hardLinks ;

}  //* End fiScanTree() *

//*************************
//*      fiScanNode       *
//*************************
//******************************************************************************
//* Private Method.                                                            *
//* Recursively scan the specified node of the directory tree searching for    *
//* filenames which match the specified Inode number.                          *
//*                                                                            *
//*                                                                            *
//* Input  : tnPath  : (by reference) pathspec of directory-tree node to scan  *
//*          nodePtr : (by reference) directory-tree node to scan              *
//*          sData   : (by reference) display data for matching filenames is   *
//*                    appended to any existing data.                          *
//*          refInode : reference INODE against which to test target files     *
//*                                                                            *
//* Returns: number of matching hard links found                               *
//******************************************************************************
//* Programmer's Note: A hard link may not point to a directory name, but we   *
//* compare directory name Inode numbers to 'refInode' in case user specified  *
//* the Inode of a directory name. In this case we will find only the filename *
//* specified, but no harm done.                                               *
//******************************************************************************

int FileDlg::fiScanNode ( const gString& tnPath, const TreeNode* nodePtr, 
                          ssetData& sData, UINT64 refInode )
{
   gString  fPath ;        // path of file with matching INODE
   int hardLinks = ZERO ;  // return value

   #if DEBUG_INODE != 0
   if ( ofs.is_open() )
      ofs << "\nfiScanNode(" << tnPath.ustr() << ")" << endl ;
   #endif   //DEBUG_INODE

   //* Scan lower-level nodes (if any) *
   if ( nodePtr->nextLevel != NULL && nodePtr->dirFiles > ZERO )
   {
      for ( UINT i = ZERO ; i < nodePtr->dirFiles ; ++i )
      {

         //* Compare INODE of subdirectory name to reference INODE.*
         hardLinks += fiCompare ( tnPath, nodePtr->nextLevel[i].dirStat, 
                                  sData, refInode ) ;

         //* Construct path to next-level target directory.*
         if ( (tnPath.compare( ROOT_PATH )) != ZERO )
            fPath.compose( "%S/%s", tnPath.gstr(), nodePtr->nextLevel[i].dirStat.fName ) ;
         else     // special case for '/'
            fPath.compose( "%S%s", tnPath.gstr(), nodePtr->nextLevel[i].dirStat.fName ) ;

         //* If this node contains directory names, recurse into lower level.*
         if ( (nodePtr->nextLevel[i].nextLevel != NULL) && 
              (nodePtr->nextLevel[i].dirFiles > ZERO) )
         {
            hardLinks += this->fiScanNode ( fPath, &nodePtr->nextLevel[i], 
                                            sData, refInode ) ;
         }

         //* If no directory names, we didn't recurse, *
         //* so scan non-directory list (if any).      *
         else if ( nodePtr->nextLevel[i].tnFiles != NULL && 
                   nodePtr->nextLevel[i].tnFCount > ZERO )
         {
            hardLinks += this->fiScanList ( fPath, nodePtr->nextLevel[i].tnFiles,
                                            sData, nodePtr->nextLevel[i].tnFCount,
                                            refInode ) ;
         }
         #if DEBUG_INODE != 0
         else if ( ofs.is_open() )
               ofs << "      (empty)" << endl ;
         #endif   // DEBUG_INODE
      }
   }

   //* Scan the non-directory filename list *
   hardLinks += this->fiScanList ( tnPath, nodePtr->tnFiles, sData, 
                                   nodePtr->tnFCount, refInode ) ;
   return hardLinks ;

}  //* End fiScanNode() *

//*************************
//*      fiScanNode       *
//*************************
//******************************************************************************
//* Private Method.                                                            *
//* Recursively scan the specified node of the directory tree searching for    *
//* filenames which are associated with multiple hard links.                   *
//*                                                                            *
//* Input  : tnPath  : (by reference) pathspec of directory-tree node to scan  *
//*          nodePtr : (by reference) directory-tree node to scan              *
//*          sData   : (by reference) display data for matching filenames is   *
//*                    appended to any existing data.                          *
//*                                                                            *
//* Returns: number of matching items found                                    *
//******************************************************************************

int FileDlg::fiScanNode ( const gString& tnPath, const TreeNode* nodePtr, ssetData& sData )
{
   gString  fPath ;        // path of file with matching INODE
   int hardLinks = ZERO ;  // return value

   //* Scan lower-level nodes (if any) *
   if ( nodePtr->nextLevel != NULL && nodePtr->dirFiles > ZERO )
   {
      for ( UINT i = ZERO ; i < nodePtr->dirFiles ; ++i )
      {
         //* If this node contains directory names *
         if ( (nodePtr->nextLevel[i].nextLevel != NULL) && 
              (nodePtr->nextLevel[i].dirFiles > ZERO) )
         {
            if ( (tnPath.compare( ROOT_PATH )) != ZERO )
               fPath.compose( "%S/%s", tnPath.gstr(), nodePtr->nextLevel[i].dirStat.fName ) ;
            else     // special case for '/'
               fPath.compose( "%S%s", tnPath.gstr(), nodePtr->nextLevel[i].dirStat.fName ) ;

            hardLinks += this->fiScanNode ( fPath, &nodePtr->nextLevel[i], sData ) ;
         }

         //* If no directory names, we didn't recurse, *
         //* so scan non-directory list (if any).      *
         else if ( nodePtr->nextLevel[i].tnFiles != NULL && 
                   nodePtr->nextLevel[i].tnFCount > ZERO )
         {
            hardLinks += this->fiScanList ( tnPath, nodePtr->nextLevel[i].tnFiles,
                                            sData, nodePtr->nextLevel[i].tnFCount ) ;
         }
      }
   }

   //* Scan the non-directory filename list *
   hardLinks += this->fiScanList ( tnPath, nodePtr->tnFiles, sData, nodePtr->tnFCount ) ;

   return hardLinks ;

}  //* End fiScanNode() *

//*************************
//*      fiScanList       *
//*************************
//******************************************************************************
//* Private Method.                                                            *
//* Scan a list of non-directory files searching for filenames which match     *
//* the specified substring.                                                   *
//*                                                                            *
//* Input  : listPath : (by reference) pathspec of directory containing files  *
//*          listPtr  : pointer to array of tnFName objects                    *
//*          sData    : (by reference) display data for matching filenames is  *
//*                     appended to any existing data.                         *
//*          listCnt  : number of items in 'listPtr' array                     *
//*          refInode : reference INODE against which to test target files     *
//*                                                                            *
//* Returns: number of matching hard links found                               *
//******************************************************************************

int FileDlg::fiScanList ( const gString& listPath, const tnFName* listPtr, 
                          ssetData& sData, UINT listCnt, UINT64 refInode )
{
   gString  fPath ;        // path of file with matching INODE
   int hardLinks = ZERO ;  // return value

   #if DEBUG_INODE != 0
   if ( ofs.is_open() )
      ofs << "  fiScanList(" << listPath.ustr() << ")" << endl ;
   #endif   //DEBUG_INODE

   for ( UINT i = ZERO ; i < listCnt ; ++i )
   {
//      #if DEBUG_INODE != 0
//      if ( ofs.is_open() )
//         ofs << "  fiScanList(" << listPtr[i].fName 
//             << "'  fType:" << listPtr[i].fType 
//             << " Inode: " << listPtr[i].rawStats.st_ino << endl ;
//      #endif   //DEBUG_INODE

      //* Compare INODE of target file to reference INODE.*
      hardLinks += fiCompare ( listPath, listPtr[i], sData, refInode ) ;
   }
   return hardLinks ;

}  //* End fiScanList() *

//*************************
//*      fiScanList       *
//*************************
//******************************************************************************
//* Private Method.                                                            *
//* Scan a list of non-directory files searching for filenames which match     *
//* the specified substring.                                                   *
//*                                                                            *
//* Input  : listPath : (by reference) pathspec of directory containing files  *
//*          listPtr  : pointer to array of tnFName objects                    *
//*          sData    : (by reference) display data for matching filenames is  *
//*                     appended to any existing data.                         *
//*          listCnt  : number of items in 'listPtr' array                     *
//*                                                                            *
//* Returns: number of matching hard links found                               *
//******************************************************************************

int FileDlg::fiScanList ( const gString& listPath, const tnFName* listPtr, 
                          ssetData& sData, UINT listCnt )
{
   gString  fPath ;        // path of file with matching INODE
   int hardLinks = ZERO ;  // return value

   for ( UINT i = ZERO ; i < listCnt ; ++i )
   {
      if ( (listPtr[i].rawStats.st_nlink > 1) && (sData.dispItems < maxMATCH) )
      {
         //* Add the filespec to the display data *
         char**   ptrArray = (char**)sData.dispText ;    // text pointer (see note above)
         attr_t*  atrArray = (attr_t*) sData.dispColor ; // color-attribute pointer
         if ( (listPath.compare( ROOT_PATH )) != ZERO )  // if not root directory, full filespec
            fPath.compose( "%S/%s (%llu)", listPath.gstr(),
                           listPtr[i].fName, &listPtr[i].rawStats.st_nlink ) ;
         else                                            // special case for '/'
            fPath.compose( "%S%s (%llu)", listPath.gstr(), 
                           listPtr[i].fName, &listPtr[i].rawStats.st_nlink ) ;
         fPath.copy( ptrArray[sData.dispItems], gsDFLTBYTES ) ;    // set display text
         atrArray[sData.dispItems] = this->ftColor[listPtr[i].fType] ; // set display attribute
         ++sData.dispItems ;     // increment display-item count
         ++hardLinks ;
      }
   }
   return hardLinks ;

}  //* End fiScanList() *

//*************************
//*       fiCompare       *
//*************************
//******************************************************************************
//* Compare the INODE of the file under test to the reference INODE.           *
//* If a match, add a display record to 'sData'.                               *
//*                                                                            *
//* Input  : pathSpec : (by reference) path of directory containing file       *
//*          fStats   : (by reference) stat data for file under test           *
//*          sData    : (by reference) display data for matching filenames is  *
//*                     appended to any existing data.                         *
//*          refInode : reference INODE against which to test target file      *
//*                                                                            *
//* Returns: 1 if matching INODE                                               *
//*          0 if not a match                                                  *
//******************************************************************************
//* Notes:                                                                     *
//* ------                                                                     *
//* -- Because the dispText and dispColor members of the ssetData class are    *
//*    are pointers to 'const', we play a trick to initialize the data arrays. *
//*                                                                            *
//******************************************************************************

int FileDlg::fiCompare ( const gString& pathSpec, const tnFName& fStats, 
                         ssetData& sData, UINT64 refInode )
{
   int   matchingLink = ZERO ;

   if ( fStats.rawStats.st_ino == refInode )
   {
      gString fPath ;                     // full filespec
      if ( sData.dispItems < maxMATCH )   // if freespace in the list
      {
         char**   ptrArray = (char**)sData.dispText ;    // text pointer (see note above)
         attr_t*  atrArray = (attr_t*) sData.dispColor ; // color-attribute pointer
         if ( (pathSpec.compare( ROOT_PATH )) != ZERO )
            fPath.compose( "%S/%s", pathSpec.gstr(), fStats.fName ) ; // full filespec
         else     // special case for '/'
            fPath.compose( "%S%s", pathSpec.gstr(), fStats.fName ) ;
         fPath.copy( ptrArray[sData.dispItems], gsDFLTBYTES ) ;    // set display text
         atrArray[sData.dispItems] = this->ftColor[fStats.fType] ; // set display attribute
         ++sData.dispItems ;     // increment display-item count
      }
      matchingLink = 1 ;      // return value

      #if DEBUG_INODE != 0
      if ( ofs.is_open() )
      {
         ofs << "** MATCH: " << fPath.ustr() << "  INODE: " 
             << fStats.rawStats.st_ino << endl ;
      }
      #endif   // DEBUG_INODE
   }
   return matchingLink ;

}  //* End fiCompare() *


//******************************************************************************
//***                 'Compare Files' Method Group                           ***
//******************************************************************************

//*************************
//*     MultiCompare      *
//*************************
//******************************************************************************
//* User has requested a file comparison.                                      *
//* Scan the 'selected' data and highlight position to determine whether the   *
//* user has indicated a two-file comparison or a multi-file comparison.       *
//* If multi-file comparison, call it directly, but if a two-file comparison   *
//* (or user error) return to caller for processing.                           *
//*                                                                            *
//* Input  : extPath   : path of directory which _potentially_ contains files  *
//*                      to be compared with selected files                    *
//*                                                                            *
//* Returns: 'true'  if multiple-file comparison completed                     *
//*          'false' if user selected less than two(2) source files            *
//******************************************************************************

bool FileDlg::MultiCompare ( const gString& extPath )
{
   bool  multiComp = false ;

   //* If multiple 'selected' files OR if one(1) 'selected file *
   //* AND it is the highlighted file, the multi-file compare.  *
   if ( ((this->selInfo.Count) >= 2) ||
        ((this->selInfo.Count == 1) && (this->IsSelected())) )
   {
      this->CompareFiles ( extPath ) ;
      multiComp = true ;
   }

   return multiComp ;

}  //* End MultiCompare() *

//*************************
//*     CompareFiles      *
//*************************
//******************************************************************************
//* Compare two files for the following data:                                  *
//* 1) timestamps                                                              *
//* 2) file size                                                               *
//* 3) file type                                                               *
//* 4) permission bits                                                         *
//* 5) Contents:                                                               *
//*    The 'diff' utility has several options for scan and output.             *
//*    We implement user selection for the most useful features.               *
//*                                                                            *
//* Input  : extFSpec  : if secondary file is external to this class,          *
//*                      this is the path/filename of that file                *
//*          extFStat  : if secondary file is external to this class,          *
//*                      this is the lstat data for the file                   *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void FileDlg::CompareFiles ( const gString& extFSpec, const tnFName& extFStat )
{
   const UINT64 TEN_MEGABYTES = 10240000 ; // file-size limit for detailed scan

   //* Define the dialog window *
   const short dialogROWS = minfitROWS,
               dialogCOLS = minfitCOLS,
               dialogYPOS = this->fulY + 1,
               dialogXPOS = this->fulX + (this->fMaxX / 2) - (dialogCOLS / 2) ;
   const attr_t dColor  = this->cs.sd,
                hColor  = this->cs.em ;

   //*   Data for the ctxSP spinner control.    *
   //* (context lines for single-column output) *
   dspinData dsData( ZERO, 99, 3, dspinINTEGER, hColor ) ;

   //* List of controls in this dialog *
   // Programmer's Note: If enum changes, also update the copy in cfConstructCmd().
   enum cfControls : short
   {
      cmpPB = ZERO, cloPB, logPB,
      brRB, binRB, colRB, com2RB, com1RB, com0RB, 
      ctxSP, cfCONTROLS
   } ;

   //** Define the dialog controls **
   InitCtrl ic[cfCONTROLS] = 
   {
   {  //* 'COMPARE' pushbutton  - - - - - - - - - - - - - - - - - - -    cmpPB *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(dialogROWS - 2),        // ulY:       upper left corner in Y
      short(dialogCOLS / 2 - 19),   // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      11,                           // cols:      control columns
      "  ^COMPARE  ",               // dispText:  
      this->cs.pn,                  // nColor:    non-focus color
      this->cs.pf,                  // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "",                           // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[cloPB]                    // nextCtrl:  link in next structure
   },
   {  //* 'CLOSE' pushbutton  - - - - - - - - - - - - - - - - - - - - -  cloPB *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      ic[cmpPB].ulY,                // ulY:       upper left corner in Y
      short(ic[cmpPB].ulX + ic[cmpPB].cols + 3), // ulX: upper left corner in X
      1,                            // lines:     (n/a)
      11,                           // cols:      control columns
      "   CL^OSE   ",               // dispText:  
      this->cs.pn,                  // nColor:    non-focus color
      this->cs.pf,                  // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "",                           // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[logPB]                    // nextCtrl:  link in next structure
   },
   {  //* 'SAVE-LOG" pushbutton - - - - - - - - - - - - - - - - - - - -  logPB *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      ic[cloPB].ulY,                // ulY:       upper left corner in Y
      short(ic[cloPB].ulX + ic[cloPB].cols + 3), // ulX: upper left corner in X
      1,                            // lines:     (n/a)
      12,                           // cols:      control columns
      "  ^SAVE LOG  ",              // dispText:  
      this->cs.pn,                  // nColor:    non-focus color
      this->cs.pf,                  // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "",                           // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[brRB]                     // nextCtrl:  link in next structure
   },
   {  //* 'Brief' radio button          - - - - - - - - - - - - - - - -   brRB *
      dctRADIOBUTTON,               // type:      
      rbtS3s,                       // rbSubtype: standard, 3-wide
      false,                        // rbSelect:  initial value
      4,                            // ulY:       upper left corner in Y
      2,                            // ulX:
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a)
      this->cs.sd,                  // nColor:    non-focus color
      this->cs.pf,                  // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "^Brief (report identical or different)", // label:     
      ZERO,                         // labY:      
      4,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[binRB]                    // nextCtrl:  link in next structure
   },
   {  //* 'Binary-as-text' radio button - - - - - - - - - - - - - - - -  binRB *
      dctRADIOBUTTON,               // type:      
      rbtS3s,                       // rbSubtype: standard, 3-wide
      false,                        // rbSelect:  initial value
      short(ic[brRB].ulY + 1),      // ulY:       upper left corner in Y
      ic[brRB].ulX,                 // ulX:
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a)
      this->cs.sd,                  // nColor:    non-focus color
      this->cs.pf,                  // fColor:    focus color
      tbPrint,                      // fi lter:    (n/a)
      "^Process binary files as text", // label:     
      ZERO,                         // labY:      
      4,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[colRB]                    // nextCtrl:  link in next structure
   },
   {  //* 'Two-Column radio button      - - - - - - - - - - - - - - - -  colRB *
      dctRADIOBUTTON,               // type:      
      rbtS3s,                       // rbSubtype: standard, 3-wide
      true,                         // rbSelect:  initial value
      short(ic[binRB].ulY + 1),     // ulY:       upper left corner in Y
      ic[binRB].ulX,                // ulX:
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a)
      this->cs.sd,                  // nColor:    non-focus color
      this->cs.pf,                  // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "^Two-column output (deselect for single-column)", // label:     
      ZERO,                         // labY:      
      4,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[com2RB]                   // nextCtrl:  link in next structure
   },
   {  //* 'Common lines 2' radio button - - - - - - - - - - - - - - - - com2RB *
      dctRADIOBUTTON,               // type:      
      rbtS3s,                       // rbSubtype: standard, 3-wide
      false,                        // rbSelect:  initial value
      short(ic[colRB].ulY + 1),     // ulY:       upper left corner in Y
      short(ic[colRB].ulX + 4),     // ulX:
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a)
      this->cs.sd,                  // nColor:    non-focus color
      this->cs.pf,                  // fColor:    focus color
      tbPrint,                      // fi lter:    (n/a)
      "print common lines in both columns", // label:     
      ZERO,                         // labY:      
      4,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[com1RB]                   // nextCtrl:  link in next structure
   },
   {  //* 'Common lines 1' radio button - - - - - - - - - - - - - - - - com1RB *
      dctRADIOBUTTON,               // type:      
      rbtS3s,                       // rbSubtype: standard, 3-wide
      true,                         // rbSelect:  initial value
      short(ic[com2RB].ulY + 1),    // ulY:       upper left corner in Y
      ic[com2RB].ulX,               // ulX:
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a)
      this->cs.sd,                  // nColor:    non-focus color
      this->cs.pf,                  // fColor:    focus color
      tbPrint,                      // fi lter:    (n/a)
      "print common lines on left only", // label:     
      ZERO,                         // labY:      
      4,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[com0RB]                   // nextCtrl:  link in next structure
   },
   {  //* 'Common lines 0' radio button - - - - - - - - - - - - - - - - com0RB *
      dctRADIOBUTTON,               // type:      
      rbtS3s,                       // rbSubtype: standard, 3-wide
      false,                        // rbSelect:  initial value
      short(ic[com1RB].ulY + 1),    // ulY:       upper left corner in Y
      ic[com1RB].ulX,               // ulX:
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a)
      this->cs.sd,                  // nColor:    non-focus color
      this->cs.pf,                  // fColor:    focus color
      tbPrint,                      // fi lter:    (n/a)
      "print ^differences only",    // label:     
      ZERO,                         // labY:      
      4,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[ctxSP]                    // nextCtrl:  link in next structure
   },
   {  //* 'Context-lines' spinner       - - - - - - - - - - - - - - - -  ctxSP *
      dctSPINNER,                   // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(ic[com0RB].ulY + 1),    // ulY:       upper left corner in Y
      ic[colRB].ulX,                // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      3,                            // cols:      control columns
      NULL,                         // dispText:  (n/a)
      nc.gyR,                       // nColor:    non-focus color
      this->cs.pf,                  // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "^Number of context lines for single-column output", // label:
      ZERO,                         // labY:      
      4,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      &dsData,                      // spinData:  spinner init
      true,                         // active:    allow control to gain focus
      NULL                          // nextCtrl:  link in next structure
   },
   } ;

   //* Save the parent dialog's display data *
   this->dPtr->SetDialogObscured () ;

   //* Open the interface dialog *
   //* Initial parameters for dialog window *
   InitNcDialog dInit( dialogROWS,     // number of display lines
                       dialogCOLS,     // number of display columns
                       dialogYPOS,     // Y offset from upper-left of terminal 
                       dialogXPOS,     // X offset from upper-left of terminal 
                       NULL,           // dialog title
                       ncltSINGLE,     // border line-style
                       dColor,         // border color attribute
                       dColor,         // interior color attribute
                       ic              // pointer to list of control definitions
                     ) ;

   //* Instantiate the dialog window *
   NcDialog* dp = new NcDialog ( dInit ) ;

   //* Open the dialog window *
   if ( (dp->OpenWindow()) == OK )
   {
      //* Set the dialog title *
      dp->SetDialogTitle ( "  Compare Files  ", this->cs.em ) ;

      //* Draw a line below the file data *
      LineDef  hLine(ncltHORIZ, ncltSINGLE, 3, ZERO, dialogCOLS, dColor) ;
      dp->DrawLine ( hLine ) ;

      //* Create XOR Radiobutton group *
      short XorGroup[] = { com2RB, com1RB, com0RB, -1 } ;
      dp->GroupRadiobuttons ( XorGroup ) ;

      gString fSpecA,               // FileA full path/filename specification
              fNameA,               // FileA name
              fLinkA,               // FileA: if symlink, filename of link file
              fSpecB,               // FileB full path/filename specification
              fNameB,               // FileB name
              fLinkB,               // FileB: if symlink, filename of link file
              logStats,             // name of temp file to receive source stats
              logHead,              // column headings for two-column output
              logIdent,             // formatted results of file identity test
              logFile,              // name of temp file to receive output
              gsOut,                // output formatting
              pErrorMsg ;           // status message
      winPos  namePos( 1, 10 ),     // filename display position
              msgPos( 11, 6 ) ;     // status message position
      tnFName fStatA,               // File1 stats
              fStatB ;              // File2 stats
      short   termRows, termCols,   // terminal window size
              icIndex = ZERO ;      // index of control with input focus
      bool    pError = false ;      // true if parameter error

      //* Screen dimensions. Used to set output formatting width *
      dp->ScreenDimensions ( termRows, termCols ) ;

      //* If caller sent us FileB (may be discarded later) *
      if ( (extFSpec.gschars()) > 1 )
      {
         this->fmPtr->ExtractFilename ( fNameB, extFSpec ) ;
         fSpecB = extFSpec ;
         fStatB = extFStat ;
      }

      //* Initialize parameters and do pre-processing. *
      pError = this->cfSetCompFiles ( fSpecA, fNameA, fStatA, fLinkA,
                                      fSpecB, fNameB, fStatB, fLinkB, 
                                      logStats, logHead, pErrorMsg, termCols ) ;

      //* Display the filenames *
      dp->WriteParagraph ( namePos.ypos, 2, "File A:\nFile B:", dColor ) ;
      dp->WriteString ( namePos, fNameA, hColor ) ;
      dp->WriteString ( namePos.ypos + 1, namePos.xpos, fNameB, hColor ) ;

      //* If files successfully identified, do prelimary scan *
      if ( pError == false )
      {
         //* For very large files, calm the user with a message *
         if ( fStatA.fBytes > TEN_MEGABYTES && fStatB.fBytes > TEN_MEGABYTES )
         {
            dp->ClearLine ( msgPos.ypos ) ;
            dp->WriteString ( msgPos, 
               "Performing prelinary scan, please wait a moment...", hColor, true ) ;
         }

         //* Perform the scan *
         this->cfPretest ( fSpecA, fNameA, fSpecB, fNameB,
                           logIdent, termCols ) ;

         dp->FlushKeyInputStream () ;  // guard against over-caffinated user

         //* For very large files, we refuse to create detailed output *
         if ( fStatA.fBytes > TEN_MEGABYTES && fStatB.fBytes > TEN_MEGABYTES )
         {
            dp->ClearLine ( msgPos.ypos ) ;
            dp->WriteString ( msgPos, "   Very large files. Brief output only.", hColor ) ;
            dp->SetRadiobuttonState ( brRB, true ) ;
            dp->ControlActive ( brRB, false ) ;
         }

         //* Create a temporary log file to contain comparison output *
         this->fmPtr->CreateTempname ( logFile ) ;
      }
      //* If error, report it *
      else
      {
         //* Display error message and disable all controls except 'Close' *
         dp->ClearLine ( msgPos.ypos ) ;
         dp->WriteString ( msgPos, pErrorMsg, hColor ) ;
         while ( (icIndex = dp->NextControl ()) != cloPB ) ;
         dp->ControlActive ( cmpPB, false ) ;
         dp->ControlActive ( logPB, false ) ;
         dp->ControlActive ( brRB, false ) ;
         dp->ControlActive ( binRB, false ) ;
         dp->ControlActive ( colRB, false ) ;
         dp->ControlActive ( com2RB, false ) ;
         dp->ControlActive ( com1RB, false ) ;
         dp->ControlActive ( com0RB, false ) ;
         dp->ControlActive ( ctxSP, false ) ;
      }

      dp->RefreshWin () ;              // make everything visible


      //*******************
      //* User input loop *
      //*******************
      uiInfo   Info ;                     // user interface data returned here
      bool     done = false ;             // loop control
      while ( ! done )
      {
         if ( ic[icIndex].type == dctPUSHBUTTON )
         {
            if ( Info.viaHotkey )
               Info.HotData2Primary () ;
            else
               icIndex = dp->EditPushbutton ( Info ) ;

            if ( Info.dataMod != false )
            {
               if ( Info.ctrlIndex == cmpPB )
               {  //* Construct the command (see method header) *
                  char cmd[MAX_PATH * 4] ;
                  int   ocOption = ZERO ;       // context lines
                  short tcOption = com2RB ;     // two-column output option
                  bool brief, binfiles, twocol ;
                  dp->GetRadiobuttonState ( brRB, brief ) ;
                  dp->GetRadiobuttonState ( binRB, binfiles ) ;
                  dp->GetRadiobuttonState ( colRB, twocol ) ;
                  tcOption = dp->GetRbGroupSelection ( com2RB ) ; // two-column output
                  dp->GetSpinnerValue ( ctxSP, ocOption ) ;       // one-column output
                  //* Copy the log stats to the output file *
                  this->fmPtr->CopyFile ( logStats.ustr(), logFile.ustr() ) ;
                  this->cfConstructCmd ( cmd, tcOption, (short)ocOption, 
                                         termCols, fSpecA, fSpecB, logFile, 
                                         brief, binfiles, twocol ) ;

                  //*********************
                  //* Compare the files *
                  //*********************
                  this->fmPtr->Systemcall ( cmd ) ;

                  //* Display the results *
                  this->cfPostprocLog ( logFile, fNameA, fNameB, logHead, logIdent, 
                                        brief, twocol, (ocOption > ZERO), binfiles ) ;
                  this->cfDisplayLog ( logFile, dp ) ;
                  this->dPtr->RefreshWin () ; // refresh and re-save parent dialog
                  this->dPtr->SetDialogObscured () ;
                  dp->RefreshWin () ;
               }
               else if ( Info.ctrlIndex == cloPB )
               {
                  done = true ;
               }
               else if ( Info.ctrlIndex == logPB )
               {
                  if ( (this->cfSaveLog ( logFile, fNameA, gsOut )) == OK )
                  {
                     gsOut.insert( "Log Saved To: " ) ;
                     //* Update the CWD display data *
                     this->dPtr->RefreshWin () ;
                     this->RefreshCurrDir () ;
                     this->dPtr->SetDialogObscured () ;
                  }
                  else
                     gsOut.insert( "Unable to save: " ) ;
                  dp->RefreshWin () ;
                  dp->WriteString ( msgPos, gsOut, hColor, true ) ;
                  while ( (icIndex = dp->PrevControl ()) != cmpPB ) ;
               }
            }
         }

         else if ( ic[icIndex].type == dctRADIOBUTTON )
         {
            if ( Info.viaHotkey )
               Info.HotData2Primary () ;
            else
               icIndex = dp->EditRadiobutton ( Info ) ;
         }

         else if ( ic[icIndex].type == dctSPINNER )
         {
            Info.viaHotkey = false ;
            icIndex = dp->EditSpinner ( Info ) ;
         }

         //* Move focus to appropriate control *
         if ( ! done && ! Info.viaHotkey )
         {
            if ( Info.keyIn == nckSTAB )
               icIndex = dp->PrevControl () ; 
            else
               icIndex = dp->NextControl () ;
         }
      }     // while(!done)

      //* Delete the temporary files *
      if ( (logFile.gschars()) > 1 )
         this->DeleteFile ( logFile ) ;
      if ( (logStats.gschars()) > 1 )
         this->DeleteFile ( logStats ) ;
   }
   if ( dp != NULL )                      // close the window
      delete ( dp ) ;

   this->dPtr->RefreshWin () ;   // Restore parent dialog display

}  //* End CompareFiles() *

//*************************
//*     CompareFiles      *
//*************************
//******************************************************************************
//* Compare the files in the selected group with the corresponding files       *
//* (if they exist) in the specified directory.                                *
//* 1) timestamps                                                              *
//* 2) file size                                                               *
//* 3) file type                                                               *
//* 4) permission bits                                                         *
//* 5) Contents: report contents as "identical" or "different" only            *
//* Note that ONLY "regular" source files will be processed. Unlike the        *
//* two-file comparison (above), symbolic links WILL NOT be followed.          *
//*                                                                            *
//* Input  : extPath   : path of directory containing files to be compared     *
//*                      with selected files                                   *
//* Returns: nothing                                                           *
//******************************************************************************

void FileDlg::CompareFiles ( const gString& extPath )
{
   const char* noAccess = "Access Error!" ;
   const char* unkName  = "-----" ;
   const char* notReg   = "(not a \"regular\" file)" ;
   const char* notFound = "(not found)" ;
   const char* Summary[] =
   {
      "Total Selected Files:",
      "  Valid Source Files:",
      "   Identical Targets:",
      "   Different Targets:",
      "   Targets Not Found:"
   } ;

   //* Define the dialog window *
   const short dialogROWS = minfitROWS,
               dialogCOLS = minfitCOLS,
               dialogYPOS = this->fulY + 1,
               dialogXPOS = this->fulX + (this->fMaxX / 2) - (dialogCOLS / 2) ;
   const attr_t dColor  = this->cs.sd,
                hColor  = this->cs.em ;

   //*   Data for the ctxSP spinner control.    *
   //* (context lines for single-column output) *
   dspinData dsData( ZERO, 99, 3, dspinINTEGER, hColor ) ;

   //* List of controls in this dialog *
   // Programmer's Note: If enum changes, also update the copy in cfConstructCmd().
   enum cfControls : short
   {
      closePB = ZERO, viewPB, savePB, cfCONTROLS
   } ;

   //** Define the dialog controls **
   InitCtrl ic[cfCONTROLS] = 
   {
   {  //* 'CLOSE' pushbutton  - - - - - - - - - - - - - - - - - - - -  closePB *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(dialogROWS - 2),        // ulY:       upper left corner in Y
      short(dialogCOLS / 2 - 19),   // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      11,                           // cols:      control columns
      "   CL^OSE   ",               // dispText:  
      this->cs.pn,                  // nColor:    non-focus color
      this->cs.pf,                  // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "",                           // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[viewPB]                   // nextCtrl:  link in next structure
   },
   {  //* 'VIEW-LOG" pushbutton - - - - - - - - - - - - - - - - - - - - viewPB *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      ic[closePB].ulY,              // ulY:       upper left corner in Y
      short(ic[closePB].ulX + ic[closePB].cols + 3), // ulX: upper left corner in X
      1,                            // lines:     (n/a)
      12,                           // cols:      control columns
      "  ^VIEW LOG  ",              // dispText:  
      this->cs.pn,                  // nColor:    non-focus color
      this->cs.pf,                  // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "",                           // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[savePB]                   // nextCtrl:  link in next structure
   },
   {  //* 'SAVE-LOG" pushbutton - - - - - - - - - - - - - - - - - - - - savePB *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      ic[viewPB].ulY,               // ulY:       upper left corner in Y
      short(ic[viewPB].ulX + ic[viewPB].cols + 3), // ulX: upper left corner in X
      1,                            // lines:     (n/a)
      12,                           // cols:      control columns
      "  ^SAVE LOG  ",              // dispText:  
      this->cs.pn,                  // nColor:    non-focus color
      this->cs.pf,                  // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "",                           // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      NULL                          // nextCtrl:  link in next structure
   },
   } ;

   //* Save the parent dialog's display data *
   this->dPtr->SetDialogObscured () ;

   //* Open the interface dialog *
   //* Initial parameters for dialog window *
   InitNcDialog dInit( dialogROWS,     // number of display lines
                       dialogCOLS,     // number of display columns
                       dialogYPOS,     // Y offset from upper-left of terminal 
                       dialogXPOS,     // X offset from upper-left of terminal 
                       NULL,           // dialog title
                       ncltSINGLE,     // border line-style
                       dColor,         // border color attribute
                       dColor,         // interior color attribute
                       ic              // pointer to list of control definitions
                     ) ;

   //* Instantiate the dialog window *
   NcDialog* dp = new NcDialog ( dInit ) ;

   //* Open the dialog window *
   if ( (dp->OpenWindow()) == OK )
   {
      //* Set the dialog title *
      dp->SetDialogTitle ( "  Compare Files  ", this->cs.em ) ;

      //* Draw a line below the file data *
      LineDef  hLine(ncltHORIZ, ncltSINGLE, 3, ZERO, dialogCOLS, dColor) ;
      dp->DrawLine ( hLine ) ;

      gString fSpecA,               // FileA full path/filename specification
              fNameA,               // FileA name
              fSpecB,               // FileB full path/filename specification
              fNameB,               // FileB name
              logStats,             // name of temp file to receive source stats
              logFile,              // name of temp file to receive output
              gsOut,                // output formatting
              gsInt,                // integer formatting
              gsMod,                // timestamp formatting
              gstmp,                // general temp data
              errMsg ;              // status message
      winPos  pathPos( 1, 1 ),      // position of source/target paths
              dataPos( 4, 1 ),      // positioning of dialog data
              msgPos( 11, 6 ) ;     // status message position
      tnFName fStatA,               // File1 stats
              fStatB ;              // File2 stats
      ExpStats es ;                 // expanded (human readable) stats
      UINT    srcCount = ZERO,      // 'regular' source files
              trgCount = ZERO,      // processed target files
              itrgCount = ZERO,     // identical targets
              dtrgCount = ZERO,     // different targets
              xtrgCount = ZERO ;    // number of targets not found
      short   termRows, termCols,   // terminal window size
              icIndex = ZERO ;      // index of control with input focus
      bool    fA_Error = false,     // 'true' if error getting file stats
              fB_Error = false,
              fB_Found = false ;

      //* Screen dimensions. Used to set output formatting width *
      dp->ScreenDimensions ( termRows, termCols ) ;

      //* Create a temporary log file to contain comparison output *
      this->fmPtr->CreateTempname ( logFile ) ;
      ofstream ofs( logFile.ustr(), ofstream::out | ofstream::trunc ) ;
      if ( ofs.is_open() )
      {
         ofs << "Compare Two Groups Of Files\n===========================\n"
             << "Source Dir: " << this->currDir << "\n"
             << "Target Dir: " << extPath.ustr() << "\n" << endl ;
      }

      this->FirstSelectedItem () ;  // highlight the first selected file
      do
      {
         fA_Error = fB_Error = false ; // no errors
         fB_Found = false ;
         errMsg.clear() ;

         //* Get stats for source file *
         if ( (this->GetStats ( fStatA )) >= ZERO )
         {
            fNameA = fStatA.fName ;
            if ( fStatA.fType == fmREG_TYPE )
               ++srcCount ;
            else
            { errMsg = notReg ; fA_Error = true ; }
         }
         else  // (this should never happen)
         { fNameA = unkName ; fA_Error = true ; errMsg = noAccess ; }

         //* Construct filespec for target file and stat *
         //* the target file to determine if it exists.  *
         if ( ! fA_Error )
         {
            fNameB = fNameA ;    // target filename
            this->fmPtr->CatPathFname ( fSpecB, extPath.ustr(), fNameB.ustr() ) ;
            if ( (this->fmPtr->GetFileStats ( fStatB, fSpecB )) == OK )
            {
               fB_Found = true ;
               if ( fStatB.fType == fmREG_TYPE )
                  ++trgCount ;
               else
               { errMsg = notReg ; fB_Error = true ; }
            }
            else     // target does not exist (or no read access)
            { fNameB = notFound ; ++xtrgCount ; }
         }
         else
            fNameB = unkName ;

         //* Compare source and target files (identical/different test)*
         if ( fB_Found && ! fA_Error && ! fB_Error )
         {
            this->fmPtr->CatPathFname ( fSpecA, this->currDir, fNameA.ustr() ) ;
            this->cfPretest ( fSpecA, fNameA, fSpecB, fNameB, gstmp, termCols ) ;
            gstmp.strip() ;
            gstmp.limitChars( gstmp.gschars() - 2 ) ;
            short indx = gstmp.findlast( L' ' ) + 1 ;
            gstmp.shiftChars( -(indx) ) ;
            if ( gstmp.gstr()[ZERO] == L'i' )
               ++itrgCount ;
            else
               ++dtrgCount ;
         }

         //* Report results of comparison *
         if ( ofs.is_open() )
         {
            short fnwidth = (((fNameA.gscols()) > (fNameB.gscols()) ? 
                             (fNameA.gscols() + 1) : (fNameB.gscols() + 1))) ;
            fNameA.padCols( fnwidth ) ;
            fNameB.padCols( fnwidth ) ;
            ofs << "SRC: " << fNameA.ustr() ;
            if ( ! fA_Error )
            {
               this->fmPtr->ExpandStats ( &fStatA, es ) ;
               gsInt.formatInt( es.fileSize, 9 ),
               this->FormatDateString ( es.modTime, gsMod ) ;
               gsOut.compose( "%S  %S", gsInt.gstr(), gsMod.gstr() ) ;
            }
            else
               gsOut = errMsg ;
            ofs << gsOut.ustr() << endl ;
            ofs << "TRG: " << fNameB.ustr() ;
            if ( fB_Found && ! fA_Error && ! fB_Error )
            {
               this->fmPtr->ExpandStats ( &fStatB, es ) ;
               gsInt.formatInt( es.fileSize, 9 ),
               this->FormatDateString ( es.modTime, gsMod ) ;
               gsOut.compose( "%S  %S", gsInt.gstr(), gsMod.gstr() ) ;
               gsOut.append( " (%S)", gstmp.gstr() ) ;
            }
            else if ( fB_Error )
               gsOut = errMsg ;
            else
               gsOut.clear() ;
            ofs << gsOut.ustr() << "\n" << endl ;
         }
      }
      while ( (this->NextSelectedItem ()) >= ZERO ) ;

      //* Append the scan totals *
      gsOut.compose( "%s %u\n"
                     "%s %u\n"
                     "%s %u\n"
                     "%s %u\n"
                     "%s %u\n", 
                     Summary[0], &this->selInfo.Count, 
                     Summary[1], &srcCount, 
                     Summary[2], &itrgCount, 
                     Summary[3], &dtrgCount, 
                     Summary[4], &xtrgCount ) ;
      ofs << "Summary:\n-------------------------\n" << gsOut.ustr() << endl ;

      ofs.close() ;              // close the log file


      //* Write summary results to dialog window *
      dp->WriteParagraph ( pathPos, "Source Path: \n"
                                    "Target Path: ", dColor ) ;
      pathPos.xpos += 13 ;
      gsOut = this->currDir ;
      this->TrimPathString ( gsOut, dialogCOLS - 15 ) ;
      dp->WriteString ( pathPos, gsOut, hColor ) ;
      ++pathPos.ypos ;
      gsOut = extPath ;
      this->TrimPathString ( gsOut, dialogCOLS - 15 ) ;
      dp->WriteString ( pathPos, gsOut, hColor ) ;

      gsOut.compose( "%s\n%s\n%s\n%s\n%s\n",
                     Summary[0], Summary[1], Summary[2], Summary[3], Summary[4] ) ;
      dp->WriteParagraph ( dataPos, gsOut, dColor ) ;
      dataPos.xpos += 22 ;
      gsOut.compose( "%u\n"
                     "%u\n"
                     "%u\n"
                     "%u\n"
                     "%u\n",
                     &this->selInfo.Count, &srcCount, &itrgCount, &dtrgCount, &xtrgCount ) ;
      dp->WriteParagraph ( dataPos, gsOut, hColor ) ;
      dp->RefreshWin () ;              // make everything visible


      //*******************
      //* User input loop *
      //*******************
      uiInfo   Info ;                     // user interface data returned here
      bool     done = false ;             // loop control
      while ( ! done )
      {
         if ( ic[icIndex].type == dctPUSHBUTTON )
         {
            if ( Info.viaHotkey )
               Info.HotData2Primary () ;
            else
               icIndex = dp->EditPushbutton ( Info ) ;

            if ( Info.dataMod != false )
            {
               if ( Info.ctrlIndex == closePB )
               {
                  done = true ;
               }
               else if ( Info.ctrlIndex == viewPB )
               {
                  //* Display the log file *
                  this->cfDisplayLog ( logFile, dp ) ;
                  this->dPtr->RefreshWin () ; // refresh and re-save parent dialog
                  this->dPtr->SetDialogObscured () ;
                  dp->RefreshWin () ;
                  Info.keyIn = nckTAB ;      // move to closePB
                  icIndex = dp->NextControl () ;
               }
               else if ( Info.ctrlIndex == savePB )
               {
                  fNameA = "Fm_Group_Compare" ;
                  if ( (this->cfSaveLog ( logFile, fNameA, gsOut )) == OK )
                  {
                     gsOut.insert( "Log Saved To: " ) ;
                     //* Update the CWD display data *
                     this->dPtr->RefreshWin () ;
                     this->RefreshCurrDir () ;
                     this->dPtr->SetDialogObscured () ;
                  }
                  else
                     gsOut.insert( "Unable to save: " ) ;
                  dp->RefreshWin () ;
                  dp->WriteString ( msgPos, gsOut, hColor, true ) ;
               }
            }
         }

         //* Move focus to appropriate control *
         if ( ! done && ! Info.viaHotkey )
         {
            if ( Info.keyIn == nckSTAB )
               icIndex = dp->PrevControl () ; 
            else
               icIndex = dp->NextControl () ;
         }
      }     // while(!done)

      this->fmPtr->DeleteTempname ( logFile ) ; // delete the temp file
   }
   if ( dp != NULL )                      // close the window
      delete ( dp ) ;

   this->dPtr->RefreshWin () ;   // Restore parent dialog display

}  //* End CompareFiles() *

//*************************
//*    cfSetCompFiles     *
//*************************
//******************************************************************************
//* Decode user's selection of files to be compared.                           *
//* 1) Identify FileA and FileB. If no errors, continue to step 2.             *
//*    Else initialize 'pErrorMsg' and return 'pError' == true                 *
//* 2) Initialize fSpecA, fNameA, fStatA, fLinkA                               *
//*               fSpecB, fNameB, fStatB, fLinkB                               *
//* 3) Create a temporary file 'logStats' and write the static stat data       *
//*    to it. This is the header data for each compare performed.              *
//* 4) For 2-column output, create column headings in 'colHead'/               *
//*                                                                            *
//* 5) Do a pretest for same/different contents. Encode the results in         *
//*    'logIdent'.                                                             *
//*                                                                            *
//*                                                                            *
//* Input  : fSpecA    : (by reference) receives filespec for FileA            *
//*          fNameA    : (by reference) receives name of FileA                 *
//*          fStatA    : (by reference) receives stats for FileA               *
//*          fLinkA    : (by reference) if FileA is a symbolic, this is the    *
//*                      name of the link file, and above get link target info *
//*          fSpecB    : (by reference) receives filespec for FileB            *
//*          fNameB    : (by reference) receives name of FileB                 *
//*          fStatB    : (by reference) receives stats for FileB               *
//*          fLinkB    : (by reference) if FileB is a symbolic, this is the    *
//*                      name of the link file, and above get link target info *
//*          logStats  : (by reference) receives filespec for temporary file   *
//*                      which contains static header data.                    *
//*          logHead   : (by reference) receives column headings for two-column*
//*                      output                                                *
//*          errMsg    : (by reference) if an error is encountered, error      *
//*                      message returned here.                                *
//*          termCols  : number of display columns in terminal window          *
//*                                                                            *
//* Returns: 'false' if no error, both files are acceptable and all parameters *
//*                  except 'errMsg' will be initialized                       *
//*          'true;  if error, files cannot be compared                        *
//*                  a) 'fNameA' and 'fNameB' initialize                       *
//*                  b) 'errMsg' will contain the appropriate warning          *
//*                  c) other parameters undefined                             *
//******************************************************************************

bool FileDlg::cfSetCompFiles ( gString& fSpecA, gString& fNameA, 
                               tnFName& fStatA, gString& fLinkA, 
                               gString& fSpecB, gString& fNameB, 
                               tnFName& fStatB, gString& fLinkB, 
                               gString& logStats, gString& logHead, 
                               gString& errMsg, short termCols )
{     // Max message width: a------------------------------------------------z
   const char* goodLink  = "%s  (%S)" ;
   const char* badLink   = "Link target invalid :'%s'" ;
   const char* noName    = "(not specified)" ;
   const char* tooMany   = "(unable to determine)" ;
   const char* notTwo    = "Must specify two (2) files to be compared." ;
   const char* irRegular = "Only 'Regular' files may be compared." ;

   gString gstmp ;                  // temp buffer
   bool    fileError = false ;      // return value


   //* Analyze user selections *
   if ( this->selInfo.Count > ZERO )
   {
      //* If two selected files, they will be FileA and FileB *
      if ( this->selInfo.Count == 2 )
      {
         this->FirstSelectedItem () ;
         this->GetStats ( fStatA ) ;
         this->fmPtr->CatPathFname ( fSpecA, this->currDir, fStatA.fName ) ;
         fNameA = fStatA.fName ;
         this->NextSelectedItem () ;
         this->GetStats ( fStatB ) ;
         this->fmPtr->CatPathFname ( fSpecB, this->currDir, fStatB.fName ) ;
         fNameB = fStatB.fName ;
      }

      //* If one selected file *
      else if ( this->selInfo.Count == 1 )
      {
         //* If selected file is not highlighted,        *
         //* selected file will be FileA and highlighted *
         //* file will be FileB.                         *
         if ( !(this->IsSelected ()) )
         {
            this->GetStats ( fStatB ) ;
            this->fmPtr->CatPathFname ( fSpecB, this->currDir, fStatB.fName ) ;
            fNameB = fStatB.fName ;
            this->FirstSelectedItem () ;
            this->GetStats ( fStatA ) ;
            this->fmPtr->CatPathFname ( fSpecA, this->currDir, fStatA.fName ) ;
            fNameA = fStatA.fName ;
         }

         //* If the selected file IS highlighted, it     *
         //* will be FileA, and caller must have sent us *
         //* FileB.                                      *
         else
         {
            this->GetStats ( fStatA ) ;
            this->fmPtr->CatPathFname ( fSpecA, this->currDir, fStatA.fName ) ;
            fNameA = fStatA.fName ;
            //* If caller did not send us FileB, we can't continue *
            if ( (fSpecB.gschars()) == 1 )
            {
               fNameB = noName ;
               errMsg = notTwo ;
               fileError = true ;
            }
         }
      }

      //* Otherwise, too many files selected. User is confused. *
      else
      {
         fNameA = tooMany ;
         fNameB = tooMany ;
         errMsg = notTwo ;
         fileError = true ;
      }
   }

   //* No selected files. The highlighted file will be FileA *
   else
   {
      if ( this->fileCount > ZERO )
      {
         this->GetStats ( fStatA ) ;
         this->fmPtr->CatPathFname ( fSpecA, this->currDir, fStatA.fName ) ;
         fNameA = fStatA.fName ;
         //* If caller did not send us FileB, we can't continue *
         if ( (fSpecB.gschars()) == 1 )
         {
            fNameB = noName ;
            errMsg = notTwo ;
            fileError = true ;
         }
      }
      else
      {
         fNameA = noName ;
         fNameB = noName ;
         errMsg = notTwo ;
         fileError = true ;
      }
   }

   //* Follow symbolic links *
   if ( !fileError && (fStatA.fType == fmLINK_TYPE) )
   {
      fLinkA = fNameA ;    // save name of symlink
      if ( (this->fmPtr->GetLinkTargetStats ( fSpecA, fStatA, gstmp )) == OK )
      {
         this->fmPtr->CatPathFname ( fSpecA, gstmp.ustr(), fStatA.fName ) ;
         fNameA.compose( goodLink, fStatA.fName, fLinkA.gstr() ) ;
      }
      else
      {
         errMsg.compose( badLink, fStatA.fName ) ;
         fileError = true ;
      }
   }
   if ( !fileError && (fStatB.fType == fmLINK_TYPE) )
   {
      fLinkB = fNameB ;    // save name of symlink
      if ( (this->fmPtr->GetLinkTargetStats ( fSpecB, fStatB, gstmp )) == OK )
      {
         this->fmPtr->CatPathFname ( fSpecB, gstmp.ustr(), fStatB.fName ) ;
         fNameB.compose( goodLink, fStatB.fName, fLinkB.gstr() ) ;
      }
      else
      {
         errMsg.compose( badLink, fStatB.fName ) ;
         fileError = true ;
      }
   }

   //* Regular files only
   if ( !fileError && ((fStatA.fType != fmREG_TYPE) || fStatB.fType != fmREG_TYPE) )
   {
      errMsg = irRegular ;
      fileError = true ;
   }

   //* Create the static data used for all comparisons this session *
   if ( ! fileError )
   {
      const char* statHDR = "\n             TYPE    SIZE USR GRP OTH    INODE  MODIFIED" ;
      const char* fromSL  = "             FROM SYMLINK: '%S'" ;
      const char* statEnd = "\n----------------------------------"
                              "----------------------------------\n" ;
      const char* columnHdr = "----------" ;
      const char* columnSpc = "          " ;

      //* Create a temporary file to hold the source *
      //* data stats and save the stat data.         *
      this->fmPtr->CreateTempname ( logStats ) ;
      ofstream ofs( logStats.ustr(), ofstream::out | ofstream::trunc ) ;
      if ( ofs.is_open() )
      {
         ofs << "FileA Path : " << fSpecA.ustr() << endl ;
         if ( (fLinkA.gschars()) > 1 )
         {
            gstmp.compose( fromSL, fLinkA.gstr() ) ;
            ofs << gstmp.ustr() << endl ;
         }
         ofs << "FileB Path : " << fSpecB.ustr() << endl ;
         if ( (fLinkB.gschars()) > 1 )
         {
            gstmp.compose( fromSL, fLinkB.gstr() ) ;
            ofs << gstmp.ustr() << endl ;
         }
         ofs << statHDR << endl ;      // write stat-column header

         //* FileA info *
         ExpStats es ;
         this->fmPtr->ExpandStats ( &fStatA, es ) ;
         gString  gsInt( es.fileSize, 6 ),
                  gsMod ;
         this->FormatDateString ( es.modTime, gsMod ) ;
         gstmp.compose( "FileA Stats:  %s %S %c%c%c %c%c%c %c%c%c %8llu  %S",
                        fmtypeString[es.fileType], gsInt.gstr(),
                        &es.usrProt.read, &es.usrProt.write, &es.usrProt.exec,
                        &es.grpProt.read, &es.grpProt.write, &es.grpProt.exec,
                        &es.othProt.read, &es.othProt.write, &es.othProt.exec,
                        &es.rawStats.st_ino, gsMod.gstr() ) ;
         ofs << gstmp.ustr() << endl ;

         //* FileB info *
         this->fmPtr->ExpandStats ( &fStatB, es ) ;
         gsInt.formatInt( es.fileSize, 6 ) ;
         this->FormatDateString ( es.modTime, gsMod ) ;
         gstmp.compose( "FileB Stats:  %s %S %c%c%c %c%c%c %c%c%c %8llu  %S",
                        fmtypeString[es.fileType], gsInt.gstr(),
                        &es.usrProt.read, &es.usrProt.write, &es.usrProt.exec,
                        &es.grpProt.read, &es.grpProt.write, &es.grpProt.exec,
                        &es.othProt.read, &es.othProt.write, &es.othProt.exec,
                        &es.rawStats.st_ino, gsMod.gstr() ) ;
         ofs << gstmp.ustr() << statEnd << endl ;

         ofs.close() ;                 // close the log-stats file
      }

      //* Create column headers for two-column output *
      logHead.compose( "FileA: %S", fNameA.gstr() ) ;
      while ( (logHead.gscols()) < (termCols / 2 + 4) )
         logHead.append( columnSpc ) ;
      logHead.limitCols( termCols / 2 - 2 ) ;
      logHead.append( "FileB: %S\n", fNameB.gstr() ) ;
      gstmp.clear() ;
      while ( (gstmp.gscols()) < (termCols / 2 + 1) )
         gstmp.append( columnHdr ) ;
      gstmp.limitCols( termCols / 2 - 5 ) ;
      logHead.append( "%S   %S", gstmp.gstr(), gstmp.gstr() ) ;
   }
   return fileError ;

}  //* End cfSetCompFiles() *

//*************************
//*       cfPretest       *
//*************************
//******************************************************************************
//* Perform a comparison pretest on the source files.                          *
//* This will create an entry for the log file reporting whether the files     *
//* are the same or different.                                                 *
//*                                                                            *
//* Input  : fSpecA    : full filespec for FileA                               *
//*          fNameA    : filename of FileA                                     *
//*          fSpecB    : full filespec for FileB                               *
//*          fNameB    : filename of FileB                                     *
//*          logIdent  : (by reference) receives formatted pretest results     *
//*                        "Files A and B are identical." OR                   *
//*                        "Files A and B differ."                             *
//*          termCols  : number of display columns in terminal window          *
//*                                                                            *
//* Returns: 'false' if files are 'identical'                                  *
//*          'true'  if files 'differ'                                         *
//******************************************************************************
//* Programmer's Note: Both filespecs have been verified to exist AND that     *
//* they are Regular files, so the chance of error is low.                     *
//******************************************************************************

bool FileDlg::cfPretest ( const gString& fSpecA, const gString& fNameA,
                          const gString& fSpecB, const gString& fNameB, 
                          gString& logIdent, short termCols )
{
   const char* identTemplate = "Files %S and %S %s.\n" ;
   // Programmer's Note: This is a cheat. It is the index of a dialog control 
   // which we don't know and don't care about at this level.
   const short com2RB    = 6 ;
   bool  differ = false ;        // return value

   char pretestCmd[gsDFLTBYTES * 3] ;
   gString pretestPath ;
   this->fmPtr->CreateTempname ( pretestPath ) ;
   this->cfConstructCmd ( pretestCmd, com2RB, ZERO, termCols, fSpecA, fSpecB,
                          pretestPath, true, false, false ) ;
   this->fmPtr->Systemcall ( pretestCmd ) ;
   ifstream ifs( pretestPath.ustr(), ifstream::in ) ;
   if ( ifs.is_open() )
   {  //* This file should have only one line *
      ifs.getline( pretestCmd, (gsDFLTBYTES * 3 - 2), NEWLINE ) ;
      if ( ifs.good() || ifs.gcount() > ZERO )
      {
         short i = ZERO ;
         while ( pretestCmd[i] != NULLCHAR )
            ++i ;
         if ( pretestCmd[--i] == 'r' )    // "differ"
            differ = true ;
         logIdent.compose( identTemplate,
                           fNameA.gstr(), fNameB.gstr(),
                           (differ ? "differ" : "are identical") ) ;
      }
      ifs.close() ;
   }
   this->DeleteFile ( pretestPath ) ;

   return differ ;

}  //* End cfPretest() *

//*************************
//*    cfConstructCmd     *
//*************************
//******************************************************************************
//* Construct the command for invoking the 'diff' utility.                     *
//* Also does some pre-processing of the log file.                             *
//*                                                                            *
//* Input  : cmdBuff  : pointer to caller's buffer which will receive the      *
//*                     command string (assumed to be long enough to hold data)*
//*          tcOption : formatting option for two-column output                *
//*                     (this is the option's radiobutton control index)       *
//*                     (keep definitions in synch with caller's enum. )       *
//*          ocOption : number of context lines (for single-column output)     *
//*          termCols : number of columns in terminal window                   *
//*                     (used for formatting width of output)                  *
//*          fSpecA   : path/filename for FileA                                *
//*          fSpecB   : path/filename for FileB                                *
//*          logFile  : path/filename for target log file                      *
//*          brief    : 'true'  if brief output                                *
//*                     'false' if detailed output                             *
//*          binfiles : 'true'  if binary files processed as text              *
//*          twocol   : 'true'  if two-column output                           *
//*                     'false' if one-column output                           *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************
//* Useful 'diff' features:                                                    *
//*    a) --brief (-q) whether or not files are different                      *
//*        PLUS '-s' (--report-identical-files)                                *
//*    b) --text  (-a) treat all files as if they were text                    *
//*    c) --context=LINES (-C LINES) amount of context around differing groups *
//*    d) --side-by-side (-y) print output in two columns, one for each file   *
//*    e) --left-column  with side-by-side, print common lines only for the    *
//*         left column                                                        *
//*    f) --suppress-comon-lines with side-by-side, do not print the common    *
//*         lines, only the differing lines                                    *
//*    g) --tabsize=COLUMNS i.e. 1 to avoid expanding tab characters in output *
//*    h) --width=NUM (-W NUM) maximum number of columns in output             *
//*         default == 130 columns, BUT actual max == width of terminal        *
//*    i) --expand-tabs' ('-t'): replace tabs with equivalent number of spaces.*
//*         Note that this actually screws up the tabs by pushing right-column *
//*         data to the right, with the difference indicator in the right      *
//*         column rather than between columns as it should be.                *
//*    j) --initial-tab places a tab after the indicator (single-column output)*
//*         which pushes all data lines over by the width of a tab.            *
//******************************************************************************
//* Notes:                                                                     *
//* -- We would like to disable TAB expansion in the output, but diff's        *
//*    two-column output relies on some TAB calculations.                      *
//*                                                                            *
//* -- We are reasonably confident that the call to 'diff' will not produce    *
//*    an error; however, if an error message is sent to 'stderr', neither we  *
//*    nor the user will see it because we send 'stderr' to the bit bucket.    *
//*    If this becomes a problem, set the CAPTURE_STDERR definition (below) to *
//*    1. This will redirect BOTH 'stdout' and 'stderr' to the log file.       *
//*                                                                            *
//* -- Beautification of the output:                                           *
//*    'diff' is a typical utility in that the output is serviceable, but not  *
//*    elegant. We can do something about that, but much of it would require   *
//*    parsing the output which is frankly too much work and is error-prone.   *
//*    We do the easy beautification:                                          *
//*    -- For two-column, non-brief output, insert column headers.             *
//*    -- For two-column, non-brief output, calculate --width based on widest  *
//*       line of FileA. This is for easier line matching. [NOT IMPLEMENTED]   *
//*    See also: cfPostprocLog() method.                                       *
//*                                                                            *
//* -- There is an inconsistency (bug) in 'diff':                              *
//*    For single-column output, the difference indicator is '!' (exclamation),*
//*    while for two-column output, the difference indicator is '|' (vertical  *
//*    bar). This is idiotic, and its origin is probably lost in UNIX history. *
//*                                                                            *
//******************************************************************************

void FileDlg::cfConstructCmd ( char* cmdBuff, short tcOption, short ocOption, 
                               short termCols, const gString& fSpecA, 
                               const gString& fSpecB, const gString& logFile, 
                               bool brief, bool binfiles, bool twocol )
{
   #define DEBUG_CF_CMD (0)      // set to '1' ONLY for debugging
   #define CAPTURE_STDERR (0)    // capture both 'stderr' and 'stdout' to log file

   //* List of controls in caller's dialog *
   enum cfControls : short
   {
      cmpPB = ZERO, cloPB, logPB,
      brRB, binRB, colRB, com2RB, com1RB, com0RB, 
      ctxSP, cfCONTROLS
   } ;

   const char* cmdBasic = "diff --width=%hd " ;
   #if CAPTURE_STDERR == 0 // DO NOT INCLUDE 'stderr' IN THE OUTPUT
   const char* cmdRedirect = " 1>>\"%S\" 2>/dev/null" ;
   #else    // INCLUDE 'stderr' IN THE OUTPUT
   const char* cmdRedirect = " 1>>\"%S\" 2>>\"%S\"" ;
   #endif   // CAPTURE_STDERR

   //* Open the log file for possible beautification *
   ofstream ofs( logFile.ustr(), ofstream::out | ofstream::app ) ;

   //* Construct the command string *
   gString gsOut( cmdBasic, &termCols ) ;

   if ( brief != false )      // brief output with 'identical'/'differ' messages
      gsOut.append( "--brief -s " ) ;
   else                       // differences reported
   {
      if ( twocol != false )  // formatting two-column output
      {
         gsOut.append( "--side-by-side " ) ;
         if ( tcOption == com1RB )
            gsOut.append( "--left-column " ) ;
         else if ( tcOption == com0RB )
            gsOut.append( "--suppress-common-lines " ) ;
      }
      else                    // formatting single-column output
      {
         if ( ocOption > ZERO )
            gsOut.append( "--context=%hd ", &ocOption ) ;
      }
   }
   if ( binfiles != false )   // process binary data as text
      gsOut.append( "--text " ) ;

   #if DEBUG_CF_CMD != 0      // for debugging only
   if ( ofs.is_open() )
   {
      ofs << gsOut.ustr() << endl ;
      ofs << fSpecA.ustr() << endl ;
      ofs << fSpecB.ustr() << endl ;
   }
   #endif   // DEBUG_CF_CMD

   gsOut.copy( cmdBuff, gsDFLTBYTES ) ;               // basic call parameters
   short indxA = gsOut.utfbytes() - 1 ;

   gString eSpecA = fSpecA,      // escape any "special characters" in filenames
           eSpecB = fSpecB ;
   this->EscapeSpecialChars ( eSpecA, escMIN ) ;
   this->EscapeSpecialChars ( eSpecB, escMIN ) ;

   gsOut.compose( "\"%S\" ", eSpecA.gstr() ) ;          // FileA source
   gsOut.copy( &cmdBuff[indxA], gsDFLTBYTES ) ;
   short indxB = (gsOut.utfbytes()) + indxA - 1 ;

   gsOut.compose( "\"%S\"", eSpecB.gstr() ) ;           // FileB source
   gsOut.copy( &cmdBuff[indxB], gsDFLTBYTES ) ;
   short indxC = (gsOut.utfbytes()) + indxB - 1 ;

   gsOut.compose( cmdRedirect, logFile.gstr(), logFile.gstr() ) ; // (see template)
   gsOut.copy( &cmdBuff[indxC], gsDFLTBYTES ) ;

   #if DEBUG_CF_CMD != 0      // for debugging only
   if ( ofs.is_open() )
   {
      ofs << gsOut.ustr() << endl ;
      ofs << cmdBuff << endl ;
      ofs << '\n' << endl ;
   }
   #endif   // DEBUG_CF_CMD

   //* Close the log file *
   if ( ofs.is_open() )
      ofs.close() ;

   #undef CAPTURE_STDERR
   #undef DEBUG_CF_CMD

}  //* End cfConstructCmd() *

//*************************
//*     cfPostprocLog     *
//*************************
//********************************************************************************
//* Post-processing of the log file for beauty and clarity.                      *
//* 1) The parsing will fail if this method is called multiple times on the      *
//*    same file.                                                                *
//*                                                                              *
//* Input  : logFile  : path/filename for target log file                        *
//*          fNameA   : filename for FileA                                       *
//*          fNameB   : filename for FileB                                       *
//*          logHead  : header text for two-column output                        *
//*          logIdent : identical/different message                              *
//*          brief    : 'true'  if brief output                                  *
//*                     'false' if detailed output                               *
//*          twocol   : 'true'  if two-column output                             *
//*                     'false' if one-column output                             *
//*          context  : 'true'  if context specified for one-column output       *
//*                     'false' if no context lines for one-column output        *
//*          binfiles : 'true'  if binary files processed as text                *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************
//* Do some beautification:                                                      *
//* 1) Copy the stat header intact.                                              *
//* 2) Insert the 'identical'/'differ' message if needed for clarity.            *
//*    a) All 'Brief' comparisons.                                               *
//*    b) Two-column identical (all sub-options)                                 *
//*    c) One-column identical (with or without context)                         *
//*    d) Intentional binary compare (we can't know if it's really binary data)  *
//*    e) "accidental binary" is defined as:                                     *
//*       !brief && (we get the "differ" message anyway                          *
//*                  || we get no output at all [implicit "identical"])          *
//*       If next line of input file begins with                                 *
//*         "Binary files /", then accidental binaries differ                    *
//*         EOF (no data), then accidental binaries are identical                *
//*         Note: for accidental binary files: 'diff' outputs "Binary files..."  *
//*               only if files differ, and "Files..." if identical with the     *
//*               '--report-identical-files' option.                             *
//*               This is _another_ bug in 'diff'.                               *
//* 3) For two-column output, insert column headings IF there are actually       *
//*    data to follow the headings.                                              *
//* 4) Capture and format the legend. Legend is present for single-column        *
//*    output, but only if:                                                      *
//*    a) with context and differences                                           *
//*    b) for intentional binary output                                          *
//* 5) Copy remaining detail data: text or binary:                               *
//*    a) Text data: copy line-by-line.                                          *
//*    b) Binary data (specified or accidental): copy byte-by-byte.              *
//* 6) Append an empty line to the output.                                       *
//* 7) Whenever you're processing someone else's data, there is always a         *
//*    chance that your parser will encounter something it has never seen        *
//*    before. Although we try to control the output that will be generated      *
//*    by 'diff', we can't anticipate every possible alternate output glitch.    *
//*                                                                              *
//*  - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -   *
//* Possible future enhancements:                                                *
//* -- For non-brief binary, filter through hexdump. This would have to be       *
//*    optional to avoid gigantic temp files, i.e. comparing VOB files would     *
//*    produce about 10GB and take an hour.                                      *
//* -- ANSI color code the differences or 'identical'/'differ'.                  *
//* -- The output may contain TAB characters. Can we expand the TABs to          *
//*    SPACEs? Is it worth it?                                                   *
//*                                                                              *
//********************************************************************************

void FileDlg::cfPostprocLog ( const gString& logFile, const gString& fNameA,
                              const gString& fNameB, const gString& logHead, 
                              const gString& logIdent, 
                              bool brief, bool twocol, bool context, bool binfiles )
{
   const char* diMsg1a  = "Binary files /" ;
   const char* diMsg1b  = "Files /" ;
   const char* diMsg2   = " and /" ;
   const short statHDR  = 8 ;       // stat header lines

   //* Create a path/filename for a new temp file and copy the log to it *
   gString tempLog ;
   this->fmPtr->CreateTempname ( tempLog ) ;
   this->fmPtr->CopyFile ( logFile.ustr(), tempLog.ustr() ) ;

   //* Open the temporary file for reading     *
   //* and the log file for writing (truncate) *
   ifstream ifs( tempLog.ustr(), ifstream::in ) ;
   ofstream ofs( logFile.ustr(), ofstream::out | ofstream::trunc ) ;

   if ( (ifs.is_open()) && (ofs.is_open()) )
   {
      gString gstmp ;
      const short ldSIZE = (gsDFLTBYTES * 2) + 1 ;
      char  lineData[ldSIZE] ;      // raw UTF-8 input

      //* Copy our stat header without modification *
      for ( short i = ZERO ; i < statHDR ; ++i )
      {
         ifs.getline ( lineData, ldSIZE, NEWLINE ) ;
         if ( ifs.good() || ifs.gcount() > ZERO )
            ofs << lineData << endl ;
      }

      //* Insert the identical/different message obtained from the first pass. *
      //* If message is repeated in the second pass, it will be discarded.     *
      ofs << logIdent.ustr() << endl ;

      //* Next line of input file is the first line *
      //* of 'diff' output. Analyze it.             *
      ifs.getline ( lineData, ldSIZE, NEWLINE ) ;
      gstmp = lineData ;

           //* bytes obtained from most recent read *
      bool readCount      = ((ifs.gcount()) > ZERO),
           //* 'true' if diff identified file as binary (but user didn't) *
           accidentalBin  = false ;

      if ( !brief && ((ifs.good()) || ((ifs.gcount()) > ZERO) ) &&
           ((((gstmp.find(diMsg1a )) == ZERO) || 
            ((gstmp.find( diMsg1b )) == ZERO))
            && ((gstmp.find( diMsg2 )))) )
      {  //* 'identical/differ' message found in second pass, discard it *
         //* and declare that diff saw binary data in the source files.  *
         ifs.getline ( lineData, ldSIZE, NEWLINE ) ;
         readCount = ((ifs.gcount()) > ZERO) ;
         accidentalBin = true ;
      }
      else if ( !(ifs.good() || (ifs.gcount() > ZERO)) )
      {  //* End-of-data reached with no 'diff' output.                 *
         //* Indicates identical files, but brief output not specified. *
         readCount = ZERO ;
         accidentalBin = true ;
      }

      //* If we _should_ and _do_ have more data to report *
      if ( !brief && !accidentalBin && readCount )
      {
         if ( twocol )  // two-column output
         {
            //* If instructed to report details, AND if we  *
            //* actually have details, write column headers.*
            ofs << logHead.ustr() << endl ;  // write column headings
            ofs << lineData << endl ;        // insert the tested line
         }
         else           // single-column output
         {
            //* Legend is present when there is context.*
            //* Format and display it.                  *
            if ( context )
               this->cfpplLegend ( ifs, ofs, lineData ) ;
            else                             // no legend in the output
               ofs << lineData << endl ;     // insert the tested line
         }
      }

      //* Copy remainder of data without modification *
      this->cfpplCopyAll ( ifs, ofs, lineData, (binfiles || accidentalBin) ) ;

      //* Append an extra blank line *
      ofs << endl ;
   }
   //* Close both files *
   if ( ifs.is_open() )
      ifs.close() ;
   if ( ofs.is_open() )
      ofs.close() ;

   //* Delete the temporary file *
   this->DeleteFile ( tempLog ) ;

}  //* End cfPostprocLog() *

//**************************
//*      cfpplCopyAll      *
//**************************
//********************************************************************************
//* Copy all remaining data from the input stream to the output stream.          *
//*                                                                              *
//* Input  : ifs     : (by reference) input stream from raw data                 *
//*          ofs     : (by reference) output stream to log file                  *
//*          lineData: caller's input buffer (use it, don't abuse it)            *
//*          binary  ; (optional, 'false' by default)                            *
//*                    'false' if data are line oriented                         *
//*                    'true'  if data are byte oriented                         *
//*                                                                              *
//* Returns: 'true' if legend processed, else 'false'                            *
//********************************************************************************
//* Programmer's Note: We really dislike outputting undisciplined binary data    *
//* to what SHOULD BE a text file. For this reason, we have inserted an option   *
//* to convert non-ASCII binary characters to ASCII i.e. a full-stop character.  *
//* Although 'diff' does not handle binary data well, we don't feel obligated to *
//* spew meaningless crap all over the screen.                                   *
//* Note that this may potentially interfere with comparing UTF-8 data for       *
//* non-English languages; however, testing shows that it does not interfere.    *
//* This would indicate that 'diff' actually does understand UTF-8 now, but in   *
//* an earlier version, had a few flaws. We have left the patch in, just in case.*
//********************************************************************************

void FileDlg::cfpplCopyAll ( std::ifstream& ifs, std::ofstream& ofs, 
                             char* lineData, bool binary )
{
   bool done = false ;

   if ( binary != false )           // binary input stream
   {
      char chIn ;
      bool done = false ;
      while ( !done )
      {
         ifs.get ( chIn ) ;
         if ( ifs.gcount() > ZERO )
         {
            #if 1    // Copy binary stream, See note above.
            if ( ((chIn < ' ') || (chIn > '~')) 
                 && (chIn != '\n') && (chIn != '\t') )
               chIn = '.' ;
            #endif   // Copy binary stream.
            ofs << chIn ;
         }
         else
            done = true ;
      }
      ofs << endl ;     // flush the buffer
   }

   else                             // line-by-line input stream
   {
      while ( !done )
      {
         ifs.getline ( lineData, gsDFLTBYTES, NEWLINE ) ;
         if ( ifs.good() || ifs.gcount() > ZERO )
            ofs << lineData << endl ;
         else
            done = true ;
      }
   }

}  //* End cfpplCopyAll() *

//**************************
//*      cfpplLegend       *
//**************************
//******************************************************************************
//* If the raw comparison data SHOULD have a legend, find it, verify it and    *
//* simplify it (remove path and timestamp).                                   *
//*                                                                            *
//* Input  : ifs     : (by reference) input stream from raw data               *
//*          ofs     : (by reference) output stream to log file                *
//*          lineData: caller's input buffer                                   *
//*                    initially contains an unprocessed line of data          *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************
//* For single-column, non-brief output WITH context, the legend preceeds the  *
//* detail data and takes the form:                                            *
//*                                                                            *
//*     *** path/filenameA          2016-05-30 20:24:24.001903517 +0800        *
//*     --- path/filenameB          2016-05-30 20:24:24.001903517 +0800        *
//*                                                                            *
//* Formatting make the legend look like this:                                 *
//*                                                                            *
//*     *** filenameA                                                          *
//*     --- filenameB                                                          *
//*                                                                            *
//* This formatting reveals some logical difficulties.                         *
//* a) Timestamp is 35 characters in the U.S. English locale shown above.      *
//*    Other locales may have a different timestamp format, so we can't search *
//*    for a pattern.                                                          *
//* b) Filenames may not have TABs, but we can't rely on 'diff' to insert      *
//*    TABs into the legend (even though our testing shows that a TAB follows  *
//*    immediately after the filename).                                        *
//* c) Filenames may have spaces, although seldom if ever two or more spaces   *
//*    in sequence. Therefore, our solution is to scan through and past the    *
//*    filename until we find either a TAB or two sequential SPACE characters, *
//*    and truncate the line at that point. If a filename DOES have two        *
//*    sequential SPACE characters, we will have created ugliness, (sorry),    *
//*    but no loss of functionality.                                           *
//******************************************************************************

void FileDlg::cfpplLegend ( std::ifstream& ifs, std::ofstream& ofs, char* lineData )
{
   const char* legendAf   = "*** /" ;
   const char* legendBf   = "--- /" ;
   const char* legendA    = "*** " ;
   const char* legendB    = "--- " ;

   gString gstmp ;      // data analysis

   //* If caller sent us the FileA legend line *
   gstmp = lineData ;
   if ( (gstmp.find( legendAf )) == ZERO )
   {
      ofs << legendA ;        // write FileA legend header

      //* Strip the path and timestamp *
      short indx = (gstmp.findlast( fSLASH )) + 1 ;
      gstmp.shiftChars( -(indx) ) ;
      if ( (indx = gstmp.find( L'\t' )) < ZERO )
         indx = gstmp.find( "  " ) ;
      if ( indx > ZERO )
         gstmp.limitChars( indx ) ;
      ofs << gstmp.ustr() << endl ;

      //* This SHOULD BE the FileB legend line *
      ifs.getline ( lineData, gsDFLTBYTES, NEWLINE ) ;
      if ( ifs.good() || ifs.gcount() > ZERO )
      {
         gstmp = lineData ;
         if ( (gstmp.find( legendBf )) == ZERO )
         {
            ofs << legendB ;  // write FileB legend leader

            //* Strip the path and timestamp *
            indx = (gstmp.findlast( fSLASH )) + 1 ;
            gstmp.shiftChars( -(indx) ) ;
            if ( (indx = gstmp.find( L'\t' )) < ZERO )
               indx = gstmp.find( "  " ) ;
            if ( indx > ZERO )
               gstmp.limitChars( indx ) ;
            ofs << gstmp.ustr() << endl ;
         }
         else     // unexpected format, output as-is
            ofs << lineData << endl ;
      }
   }
   else           // not a legend line, output as-is (should not happen)
      ofs << lineData << endl ;

}  //* End cfpplLegend() *

//**************************
//*      cfDisplayLog      *
//**************************
//******************************************************************************
//* Temporarily put the application to sleep and use 'less' utility to display *
//* the specified file.                                                        *
//*                                                                            *
//* Input  : logFile: path/filename of log file to be displayed                *
//*          ddp    : pointer to caller's dialog window                        *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************
//* Programmer's Note: We logically violate the maximum length of a gString    *
//* by composing the command PLUS the filespec into the same gString object    *
//* which is by definition just large enough to hold a path/filename           *
//* specification. We can take this risk because we assume that the path to    *
//* the temp file directory is rather short.                                   *
//*    If this ever comes back to bite us on the ass, we will do pennance.     *
//******************************************************************************

void FileDlg::cfDisplayLog ( const gString& logFile, NcDialog* ddp )
{

   gString cmd( "less -c '%S'", logFile.gstr() ) ;
   ddp->ShellOut ( soX, cmd.ustr() ) ;

}  //* End cfDisplayLog() *

//**************************
//*       cfSaveLog        *
//**************************
//******************************************************************************
//* Copy the log file to CWD where user can access it.                         *
//* -- Filename of public log is based on the filename of FileA,               *
//*    with ".log" appended                                                    *
//*                                                                            *
//*                                                                            *
//* Input  : logFile: path/filename of log file                                *
//*          fNameA : filename for FileA which is used to create target name   *
//*          logName: (by reference) receives name of target file              *
//*                                                                            *
//* Returns: OK if successful, else failure errno value                        *
//******************************************************************************

short FileDlg::cfSaveLog ( const gString& logFile, const gString&fNameA, gString& logName )
{
   gString trgPath ;

   //* Construct target path/filename *
   logName.compose( "%S.log", fNameA.gstr() ) ;
   this->fmPtr->CatPathFname ( trgPath, this->currDir, logName.ustr() ) ;
   return ( (this->fmPtr->CopyFile ( logFile.ustr(), trgPath.ustr() )) ) ;

}  //* End cfSaveLog() *


//******************************************************************************
//***                    'Grep Files' Method Group                           ***
//******************************************************************************

//*************************
//*       GrepFiles       *
//*************************
//******************************************************************************
//* Find matching substring in specified files.                                *
//* 1) File(s) to be scanned:                                                  *
//*    a) Use the 'selected" files in the parent dialog's window.              *
//*    b) Prompt user for a filename selection pattern.                        *
//* 2) Prompt for a search pattern, either a text string or a regexp pattern.  *
//* 3) Scan for the search pattern in the specified file(s).                   *
//*                                                                            *
//* This method sends the user's search parameters to the 'grep' utility and   *
//* captures the grep output to a temporary file.                              *
//*                                                                            *
//*                                                                            *
//* Input  : ul        : (by reference) upper left corner dialog position      *
//*          rows      : number of rows for dialog                             *
//*          cols      : number of columns for dialog                          *
//*                                                                            *
//* Returns: 'true'  if grep log saved to user's directory                     *
//*          'false' if grep log discarded                                     *
//******************************************************************************
//* Notes:                                                                     *
//* ======                                                                     *
//*                                                                            *
//* 1) We would like to implement a recursive scan, but it seems impractical   *
//*    at this time.                                                           *
//* 2) Implement a count of matches in the target files ('-c' switch).         *
//* 3) Implement a context option with fixed or variable amount of context,    *
//*    plus "before" (-B), "after" (-A) and "context" (-C).                    *
//* 4) We also want to offer a refined search i.e. a search of the results.    *
//*    Specifically, we want to offer the '-v' switch to filter our results    *
//*    which DO NOT meet user criteria.                                        *
//* 5) Formatting for the extracted OpenDocument text could be enhanced by     *
//*    decoding more of the XML, specifically for tables and other columnar    *
//*    data.                                                                   *
//*                                                                            *
//*                                                                            *
//******************************************************************************

bool FileDlg::GrepFiles ( const winPos& ul, short rows, short cols )
{
   const char* labels[3] = 
   {  "Source Files (% 3hd) : ",
      "Text search pattern: ",
      "Filter: ",
   } ;
   const char* fishRadio = "◉" ;
   const char* const noFName = 
      "Warning! No filenames match the specified pattern, or specified filenames\n"
      "contain un-escaped \"regexp\" characters. Press any key to continue...." ;
   const char* const fnOverflow = 
      "Warning! Too many source files specified. Some filenames\n"
      "may not be included in the scan. Press any key to continue...." ;

   const attr_t bColor = this->cs.sd,  // border color
         dColor = this->cs.sb,         // interior color
         eColor = this->cs.em,         // emphasis color
         wColor = this->cs.wr ;        // warning color

   gString fnList,                     // names of files to be scaned (or filename pattern)
           odList,                     // list of OpenDocument filenames to be scanned (if any)
           fnCount,                    // label for filename Textbox (incl. file count)
           gsPat,                      // primary search pattern (regexp)
           gsFilter,                   // secondary search pattern (regexp)
           gsCmd,                      // constructed ShellOut() command string
           gsDelim,                    // for delimiting filenames
           gs ;                        // general text formatting
   short indx,                         // index into file record display list
         scanCount = ZERO ;            // number of files to be scanned
   bool status = false,                // return value
        good_fPat = true,              // 'true' if valid filename list/pattern
        good_sPat = false,             // 'true' if valid search regexp
        good_xPat = true,              // 'true' if valid filter regexp (or no filter)
        caseSen  = true,               // 'true' if case-sensitive radiobutton set
        lineNum  = false,              // 'true' if line-number radiobutton set
        matchCnt = false ;             // 'true' if match-count radiobutton set

   enum gfControls : short { gfNameTB = ZERO, gfRegxTB, gfScanPB, gfClosePB, 
                             gfCaseRB, gfLnumRB, gfCntmRB, gfFiltTB, gfCONTROLS } ;

   InitCtrl ic[gfCONTROLS] =
   {
   {  //* 'FILENAME' Textbox  - - - - - - - - - - - - - - - - - - - - gfNameTB *
      dctTEXTBOX,                   // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      1,                            // ulY:       upper left corner in Y
      22,                           // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      short(cols - 23),             // cols:      control columns
      NULL,                         // dispText:  
      this->cs.tn,                  // nColor:    non-focus color
      this->cs.tf,                  // fColor:    focus color
      tbPrint,                      // filter:    valid filename characters
      fnCount.ustr(),               // label:
      ZERO,                         // labY:      
      -22,                          // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[gfRegxTB]                 // nextCtrl:  link in next structure
   },
   {  //* 'REGEXP PATTERN' Textbox - - - - - - - - - - - - - - - - -  gfRegxTB *
      dctTEXTBOX,                   // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      3,                            // ulY:       upper left corner in Y
      22,                           // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      short(cols - 23),             // cols:      control columns
      NULL,                         // dispText:  
      this->cs.tn,                  // nColor:    non-focus color
      this->cs.tf,                  // fColor:    focus color
      tbPrint,                      // filter:    valid filename characters
      labels[1],                    // label:     
      ZERO,                         // labY:      
      -22,                          // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[gfScanPB]                 // nextCtrl:  link in next structure
   },
   {  //* 'SCAN' pushbutton  - - - - - - - - - - - - - - - - - - - -  gfScanPB *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(rows - 3),              // ulY:       upper left corner in Y
      4,                            // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      11,                           // cols:      control columns
      "   ^SCAN    ",                // dispText:  
      this->cs.pn,                  // nColor:    non-focus color
      this->cs.pf,                  // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "",                           // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[gfClosePB]                // nextCtrl:  link in next structure
   },
   {  //* 'CLOSE' pushbutton  - - - - - - - - - - - - - - - - - - -  gfClosePB *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      ic[gfScanPB].ulY,             // ulY:       upper left corner in Y
      short(ic[gfScanPB].ulX + ic[gfScanPB].cols + 3), // ulX: upper left corner in X
      1,                            // lines:     (n/a)
      11,                           // cols:      control columns
      "   ^CLOSE   ",               // dispText:  
      this->cs.pn,                  // nColor:    non-focus color
      this->cs.pf,                  // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "",                           // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[gfCaseRB]                 // nextCtrl:  link in next structure
   },
   {  //* 'Case Sensitivity' radio button   - - - - - - - - - - - - - gfCaseRB *
      dctRADIOBUTTON,               // type:      
      rbtC1,                        // rbSubtype: custom, 1-wide
      caseSen,                      // rbSelect:  initial value (set)
      ic[gfClosePB].ulY,            // ulY:       upper left corner in Y
      short(ic[gfClosePB].ulX + ic[gfClosePB].cols + 4), // ulX:
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      fishRadio,                    // dispText:  (n/a)
      dColor,                       // nColor:    non-focus color
      wColor,                       // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "C^ase Sensitive",            // label:     
      ZERO,                         // labY:      
      2,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[gfLnumRB]                 // nextCtrl:  link in next structure
   },
   {  //* 'Line Numbers' radio button   - - - - - - - - - - - - - - - gfLnumRB *
      dctRADIOBUTTON,               // type:      
      rbtC1,                        // rbSubtype: custom, 1-wide
      lineNum,                      // rbSelect:  initial value (set)
      ic[gfCaseRB].ulY,             // ulY:       upper left corner in Y
      short(ic[gfCaseRB].ulX + 22), // ulX:
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      fishRadio,                    // dispText:  (n/a)
      dColor,                       // nColor:    non-focus color
      wColor,                       // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "Show ^Line Numbers",         // label:     
      ZERO,                         // labY:      
      2,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[gfCntmRB]                 // nextCtrl:  link in next structure
   },
   {  //* 'Count Matches' radio button  - - - - - - - - - - - - - - - gfCntmRB *
      dctRADIOBUTTON,               // type:      
      rbtC1,                        // rbSubtype: custom, 1-wide
      false,                        // rbSelect:  initial value (set)
      short(ic[gfCaseRB].ulY + 1),  // ulY:       upper left corner in Y
      ic[gfCaseRB].ulX,             // ulX:
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      fishRadio,                    // dispText:  (n/a)
      dColor,                       // nColor:    non-focus color
      wColor,                       // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "^Report Match Count",        // label:     
      ZERO,                         // labY:      
      2,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[gfFiltTB],                // nextCtrl:  link in next structure
   },
   {  //* 'FILTER' Textbox   - - - - - - - - - - - - - - - - - - - -  gfFiltTB *
      dctTEXTBOX,                   // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(ic[gfScanPB].ulY + 1),  // ulY:       upper left corner in Y
      9,                            // ulX:       upper left corner in X
      1,                            // lines:     number of lines
      20,                           // cols:      control columns
      NULL,                         // dispText:  
      this->cs.tn,                  // nColor:    non-focus color
      this->cs.tf,                  // fColor:    focus color
      tbPrint,                      // filter:    valid filename characters
      labels[2],                    // label:     
      ZERO,                         // labY:      
      -8,                           // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      NULL                          // nextCtrl:  link in next structure
   },
   } ;

   //* Create a temporary file to hold captured output of various utilities. *
   gString tmpFile ;                // filespec of temporary file
   this->fmPtr->CreateTempname ( tmpFile ) ;

   //* If there are 'selected' _regular_ files in CWD, these are the scan list.*
   //* We must also consider potential buffer overflow here.                   *
   if ( this->selInfo.Count > ZERO )
   {
      gString fileSpec ;                  // target filespec
      bool    isRt ;                      // 'true' if regular, text file
      fnList.clear() ;                    // initialize the buffer
      indx = this->FirstSelectedItem () ; // set highlight on first selected item

      while ( indx != ERR )      // for all 'selected' files
      {
         if ( this->deList[indx]->fType == fmREG_TYPE )
         {
            //* Verify that file IS NOT a binary file, OR that *
            //* file is an OpenOffice/OpenXML document.        *
            // Programmer's Note: 'Regular', 'Text' files are appended to 'fnList',
            // while OpenDocument/OpenXML filenames are appended to 'odList'.
            // (ePub documents not supported at this time.)
            this->fmPtr->CatPathFname ( fileSpec, this->currDir, 
                                        this->deList[indx]->fName ) ;
            if ( (isRt = this->gfIsRegTextfile ( fileSpec, tmpFile )) ||
                 (this->odIsOpenDocFile ( fileSpec )) ||
                 (this->odIsOpenXmlFile ( fileSpec )) ||
                 (this->odIsBinDocFile ( fileSpec )) )
            {
               //* If the filename contains Linux "special characters", *
               //* or spaces (' ') escape those characters.             *
               gsDelim = this->deList[indx]->fName ;
               this->fmPtr->EscapeSpecialChars ( gsDelim, escLINUX ) ;
               this->fmPtr->EscapeSpecialChars ( gsDelim, escWCHAR, L' ' ) ;

               if ( isRt )
                  fnList.append( "%S ", gsDelim.gstr() ) ;
               else
                  odList.append( "%S ", gsDelim.gstr() ) ;
               ++scanCount ;

               //* Prevent buffer overflow *
               if ( (fnList.utfbytes() >= MAX_PAT) || (odList.utfbytes() >= MAX_PAT) )
               {
                  good_fPat = false ;
                  break ;
               }
            }
            else
               this->DeselectFile ( false, ZERO ) ; // de-select the binary-file entry
         }
         else
            this->DeselectFile ( false, ZERO ) ;  // de-select the non-regular-file entry
         indx = this->NextSelectedItem () ;       // index next 'selected' item
      }  // while()

      this->FirstSelectedItem () ;                // return highlight to first selected item
   }
   else           // no filenames specified, alert user
      good_fPat = false ;

   fnCount.compose( labels[ZERO], &scanCount ) ;   // initialize Textbox label

   this->dPtr->SetDialogObscured () ;  // save parent dialog display

   //* Initial parameters for dialog window *
   InitNcDialog dInit( rows,           // number of display lines
                       cols,           // number of display columns
                       ul.ypos,        // Y offset from upper-left of terminal 
                       ul.xpos,        // X offset from upper-left of terminal 
                       NULL,           // dialog title
                       ncltSINGLE,     // border line-style
                       bColor,         // border color attribute
                       dColor,         // interior color attribute
                       ic              // pointer to list of control definitions
                     ) ;

   //* Instantiate the dialog window *
   NcDialog* dp = new NcDialog ( dInit ) ;

   if ( (dp->OpenWindow()) == OK )
   {
      short icIndex = ZERO ;           // index of control with input focus

      //* Set dialog title *
      dp->SetDialogTitle ( "  GREP FILES  ", this->cs.em ) ;

      //* Display static info *
      winPos wpHead( 1, 1 ) ;       // base text position
      dp->WriteString ( wpHead.ypos++, wpHead.xpos, // update filename textbox label
                        fnCount, eColor, true ) ;

      LineDef ldef( ncltHORIZ, ncltSINGLE, wpHead.ypos++, ZERO, cols, dColor ) ; 
      dp->DrawLine ( ldef ) ;

      //* Update search pattern label *
      dp->WriteString ( wpHead.ypos++, wpHead.xpos, labels[1], eColor, true ) ;
      //* Update filter label *
      dp->WriteString ( ic[gfFiltTB].ulY, (ic[gfFiltTB].ulX + ic[gfFiltTB].labX), 
                        labels[2], eColor, true ) ;

      ldef.startY = wpHead.ypos++ ;
      dp->DrawLine ( ldef ) ;
      ldef.startY = ic[gfScanPB].ulY - 1 ;
      dp->DrawLine ( ldef ) ;

      //* Enable audible alert for invalid input *
      dp->TextboxAlert ( gfNameTB, true ) ;
      dp->TextboxAlert ( gfRegxTB, true ) ;
      dp->TextboxAlert ( gfFiltTB, true ) ;

      //* Ensure that we are in the displayed CWD.          *
      //* If actual CWD is different from displayed CWD, we *
      //* will restore it when the operation is complete.   *
      gString currCwd, prevCwd ;
      this->fmPtr->GetCurrDir ( currCwd ) ;
      this->fmPtr->SetCWD ( currCwd, &prevCwd ) ;

      if ( good_fPat || (fnList.gschars() > 1) || (odList.gschars() > 1) )
      {
         icIndex = dp->NextControl () ;            // move focus to Pattern textbox
         gs = fnList ;
         if ( (odList.gschars()) > 1 )
            gs.append( "%S", odList.gstr() ) ;
         dp->SetTextboxText ( gfNameTB, gs ) ;     // set Filename textbox contents
         dp->ControlActive ( gfNameTB, false ) ;   // deactivate the Filename textbox

         if ( ! good_fPat && (scanCount > ZERO) )
         {  //* If buffer is full, warn user that some filenames may have *
            //* been lost. This is unlikely because we have over 3,000    *
            //* bytes available in each buffer.                           *
            dp->WriteParagraph ( wpHead, fnOverflow, wColor, true ) ;
            nckPause() ;
            dp->ClearLine ( wpHead.ypos ) ;
            dp->ClearLine ( wpHead.ypos + 1 ) ;
            good_fPat = true ;
         }
      }

      dp->RefreshWin () ;           // make everything visible

      uiInfo   Info ;               // user interface data returned here
      bool done = false ;           // loop control

      //*******************
      //* User input loop *
      //*******************
      while ( ! done )
      {
         //* Clear the context-help area *
         dp->ClearArea ( wpHead.ypos, wpHead.xpos,
                         (rows - wpHead.ypos - 4), (cols - 2), dColor ) ;

         if ( ic[icIndex].type == dctTEXTBOX )
         {
            if ( icIndex == gfNameTB )
            {
               winPos wp = dp->WriteParagraph ( wpHead, 
                              "Enter source filename(s) or selection pattern:\n", 
                              eColor ) ;
               wp = dp->WriteParagraph ( wp,
                  "1) Filenames ARE case sensitive. 2) '*' matches any sequence of characters.\n"
                  "3) '.' matches any single character. 4) '\\.' matches a fullstop character.\n", 
                  dColor ) ;
               dp->WriteParagraph ( wp, 
                  "Examples: *\\.eml         all filenames with the extension \".eml\"\n"
                  "          *\\.htm*        all filenames whose extension begins with \".htm\"\n"
                  "          *\\.[hc]pp      all filenames with an extension of \".hpp\" OR \".cpp\"\n"
                  "          *tax*\\.od[tsp] all 'Open Document' filenames containing \"tax\"\n"
                  " NOTE: If individual filenames specified, seperate them with commas (,).\n"
                  "       Example: DecodeXml.cpp,Fall Schedule.eml, Accounts Payable.odt\n"
                  , dColor, true ) ;
            }
            else if ( icIndex == gfRegxTB )
            {
               winPos wp = dp->WriteParagraph ( wpHead, 
                  "Enter \"regular expression\" for substring search.\n"
                  "Examples: '^Four score * seven [ybt]ears ago'   'New Balance: \\$[1-9]0,000'\n",
                  eColor ) ;
               wp = dp->WriteParagraph ( wp,
                  "'*' matches any sequence of characters.   '.' matches any single character.\n"
                  "'^' anchors pattern to beginning of line. '$' anchor pattern to end of line.\n"
                  "'\\$' To use a special char as an ordinary char, escape it with backslash.\n" 
                  "[ ] square brackets enclose a character list (any char will be matched).\n"
                  "Note: pattern will automatically be delimited by single- or double-quotes.\n"
                  "               For more information, at the command line type:\n"
                  "                  info grep 'Regular Expressions' (ENTER)\n",
                  dColor, true ) ;
            }
            else if ( icIndex == gfFiltTB )
            {
               winPos wp = dp->WriteParagraph ( wpHead, 
                               "Apply Secondary Filter:\n"
                               "  ", eColor ) ;
               wp = dp->WriteParagraph ( wp, 
                        "Filter the results of the initial scan to EXCLUDE entries which also\n"
                        "contain the specified secondary text. (begin pattern with a minus '-')\n",
                        eColor ) ;
               wp = dp->WriteParagraph ( wp,
                        "Example: Search for the Green Day album: \"American Idiot\" but exclude\n"
                        "         references to the (awful) Kerrang! cover album:     ",
                        dColor ) ;
               wp = dp->WriteParagraph ( wp, "-Kerrang\n", wColor ) ;
               wp.xpos = 3 ;
               wp = dp->WriteParagraph ( wp, 
                        "OR:  Filter the results of the initial scan to INCLUDE\n"
                        "entries which also contain the specified secondary text.\n",
                        eColor ) ;
               wp = dp->WriteParagraph ( wp,
                        "Example: Search for entries which contain both:\n"
                        "         \"American Idiot\" and a reference to the cover album: ",
                        dColor ) ;
               wp = dp->WriteParagraph ( wp, "Kerrang\n", wColor, true ) ;
            }

            Info.viaHotkey = false ;
            icIndex = dp->EditTextbox ( Info ) ;

            //* If user has entered one or more filenames OR a filename pattern *
            if ( Info.dataMod != false )
            {
               if ( Info.ctrlIndex == gfNameTB )
               {
                  //* If buffer contains regexp data, expand the pattern.*
                  //* Note that this is not a definitive test. See note  *
                  //* in called method.                                  *
                  dp->GetTextboxText ( gfNameTB, fnList ) ; // get a copy of user input

                  // Programmer's Note: The gfIsRegexpPattern() method searches for 
                  // certain un-escaped regexp characters to determine whether the 
                  // user has entered a regexp expression OR a series of filenames.
                  // The user must escape those characters when used in ordinary 
                  // filenames. It's fully documented, but users are idiots.
                  if ( (this->gfIsRegexpPattern ( fnList )) )
                     good_fPat = this->gfConstructFilenameList ( fnList, odList, scanCount ) ;

                  //* Else, assume that fnList contains one or more filenames,*
                  //* separated by commas. Verify that each file exists, and  *
                  //* that it is a 'regular' 'text' file. OR an OpenOffice/   *
                  //* OpenXML file.                                           *
                  else
                     good_fPat = this->gfValidateFilenameList ( fnList, odList, scanCount ) ;

                  //* Update displayed filename count *
                  fnCount.compose( labels[ZERO], &scanCount ) ;
                  dp->WriteString ( 1, 1, fnCount, eColor, true ) ;

                  //* Update the displayed list, and if we have a non-zero *
                  //* filename count, deactivate the Filename Textbox.     *
                  // Programmer's Note: Sorry about the tortured (but necessary)
                  // sequence in setting the focus.
                  bool moveFocus = false ;
                  if ( icIndex == gfNameTB )
                  { icIndex = dp->PrevControl () ; moveFocus = true ; }
                  gs = fnList ;
                  if ( (odList.gschars()) > 1 )
                     gs.append( "%S", odList.gstr() ) ;
                  dp->SetTextboxText ( gfNameTB, gs ) ;

                  //* If no files match user's pattern, or *
                  //* if buffer overflow, alert user.      *
                  if ( ! good_fPat )
                  {
                     dp->ClearArea ( wpHead.ypos, wpHead.xpos,
                                     (rows - wpHead.ypos - 4), (cols - 2), dColor ) ;
                     if ( scanCount == ZERO )
                        dp->WriteParagraph ( wpHead, noFName, this->cs.wr, true ) ;
                     else
                     {
                        dp->WriteParagraph ( wpHead, fnOverflow, this->cs.wr, true ) ;
                        good_fPat = true ;
                     }
                     nckPause() ;
                     dp->ClearLine ( wpHead.ypos ) ;
                     dp->ClearLine ( wpHead.ypos + 1 ) ;
                  }

                  if ( scanCount > ZERO )
                  {
                     dp->ControlActive ( gfNameTB, false ) ;
                     good_fPat = true ;
                  }
                  if ( ! good_fPat || (moveFocus && (Info.keyIn == nckSTAB)) )
                     icIndex = dp->NextControl () ;
               }

               else if ( Info.ctrlIndex == gfRegxTB )
               {
                  //* If user has entered a search pattern, validate it.*
                  dp->GetTextboxText( gfRegxTB, gsPat ) ;
                  good_sPat = gfValidateSearchPattern ( gsPat ) ;
                  if ( good_fPat )
                  {
                     bool moveFocus = false ;
                     if ( icIndex == gfRegxTB )
                     { icIndex = dp->PrevControl () ; moveFocus = true ; }
                     dp->SetTextboxText ( gfRegxTB, gsPat ) ;
                     if ( moveFocus )
                        icIndex = dp->NextControl () ;
                  }
               }

               else if ( Info.ctrlIndex == gfFiltTB )
               {
                  //* If user has entered a secondary filter, validate it.*
                  dp->GetTextboxText( gfFiltTB, gsFilter ) ;
                  good_xPat = gfValidateSearchPattern ( gsFilter ) ;
                  if ( good_xPat && (gsFilter.gschars() > 1) )
                  {
                     bool moveFocus = false ;
                     if ( icIndex == gfFiltTB )
                     { icIndex = dp->PrevControl () ; moveFocus = true ; }
                     dp->SetTextboxText ( gfFiltTB, gsFilter ) ;
                     if ( moveFocus )
                        icIndex = dp->NextControl () ;
                  }
               }
            }
         }

         else if ( ic[icIndex].type == dctPUSHBUTTON )
         {
            if ( Info.viaHotkey )
               Info.HotData2Primary () ;
            else
               icIndex = dp->EditPushbutton ( Info ) ;

            if ( Info.dataMod != false )
            {
               if ( Info.ctrlIndex == gfScanPB )
               {
                  if ( good_fPat && good_sPat )
                  {  //* Get parameter flags *
                     dp->GetRadiobuttonState ( gfLnumRB, lineNum ) ;
                     dp->GetRadiobuttonState ( gfCntmRB, matchCnt ) ;
                     dp->GetRadiobuttonState ( gfCaseRB, caseSen ) ;

                     dp->SetDialogObscured () ; // protect our display data

                     //* Construct and execute the 'grep' command, *
                     //* capturing the output to a temporary file. *
                     this->gfScan ( fnList, odList, gsPat, gsFilter, tmpFile, 
                                    lineNum, caseSen, matchCnt ) ;

                     //* Shell out to command line and display the captured data.*
                     gsCmd.compose( "less -c -R \"%S\"", tmpFile.gstr() ) ;
                     dp->ShellOut ( soX, gsCmd.ustr() ) ;

                     //* Refresh and re-save parent dialog display.*
                     this->dPtr->RefreshWin () ;
                     this->dPtr->SetDialogObscured () ;
                     dp->RefreshWin() ;          // restore sub-dialog display
                  }

                  //* Else, warn user of error *
                  else
                  {
                     gs.compose( "%s has not been specified.\n"
                                 "   Press Any Key To Continue...",
                                 (!good_fPat ? "A filename" : "A search pattern") ) ;
                     dp->WriteParagraph ( wpHead, gs, wColor, true ) ;
                     nckPause();
                  }
               }
               else // if ( Info.ctrlIndex == gfClosePB )
                  done = true ;
            }
         }

         else if ( ic[icIndex].type == dctRADIOBUTTON )
         {
            if ( Info.viaHotkey )
               Info.HotData2Primary () ;
            else
               icIndex = dp->EditRadiobutton ( Info ) ;
         }

         //* Move focus to appropriate control *
         if ( ! done && ! Info.viaHotkey )
         {
            if ( Info.keyIn == nckSTAB )
               icIndex = dp->PrevControl () ; 
            else
               icIndex = dp->NextControl () ;
         }
      }  // while ( !done )

      if ( prevCwd != currCwd )           // restore original CWD
         this->fmPtr->SetCWD ( prevCwd ) ;
   }

   if ( dp != NULL )                   // close the sub-dialog
      delete ( dp ) ;

   this->dPtr->RefreshWin () ;         // restore parent dialog

   this->fmPtr->DeleteTempname ( tmpFile ) ;   // delete the temp file

   return status ;

}  //* End GrepFiles() *

//*************************
//*        gfScan         *
//*************************
//******************************************************************************
//* Construct a 'grep' command using the provided parameters.                  *
//* Execute the command, redirecting the output to a temporary file.           *
//*                                                                            *
//* If 'fnList' contains filenames of LibreOffice documents, separate them     *
//* from the non-document files. Scan the non-document files in the            *
//* first pass. Then for the second pass, extract the text file from           *
//* each document file and run the second pass on those sub-files,             *
//* appending the results to first-pass results.                               *
//*                                                                            *
//* Input  : fnList  : (by reference) delimited list of 'text' filenames       *
//*                      (if any) to be scanned                                *
//*          odList  : (by reference) delimited list of OpenDocument/OpenXML   *
//*                      filenames (if any) to be scanned                      *
//*          gsPat   : (by reference) primary regexp search pattern            *
//*                      (delimited and fully escaped)                         *
//*          gsFilt  : (by reference) secondary regexp search pattern          *
//*                      (delimited and fully escaped)                         *
//*                      If a minus ('-') at offset gsFilt.gstr()[1]           *
//*                      (discarded), it indicates an exclusion filter.        *
//*          gsCap   : (by reference) filespec of temporary file to capture    *
//*                      the output                                            *
//*          lineNum : if 'false' no line-number switch                        *
//*                    if 'true'  set line-number switch 'n'                   *
//*          caseSen : if 'false' set case-insensitive switch 'i'              *
//*                    if 'true'  do not set case-insensitive switch           *
//*          matchCnt: if 'false' report each item matching pattern            *
//*                    if 'true'  set 'count-only' switch                      *
//*                                                                            *
//* Returns: 'true'  if scan complete [currently always returns 'true']        *
//*          'false' if error executing scan                                   *
//******************************************************************************
//* Default options used in the command:                                       *
//* ------------------------------------                                       *
//*  -n   Include line numbers (if specified by caller).                       *
//*  -i   Perform case-insensitive scan (if specified by caller).              *
//*  -c   Count matches for each source file (if specified by caller)          *
//*  -v   (optional) If specified, exclude items which match 'gsFilt' pattern. *
//*                                                                            *
//*  -H   report filename of each file containing matches                      *
//*  --color=always add ANSI color to matching text                            *
//*       If displaying the results with 'less', invoke with:                  *
//*         'less -R' tmpFilename                                              *
//*       If saving output file to user space, do not include ANSI color.      *
//*       (Color is automagically stripped on redirect unless 'always' set.)   *
//*                                                                            *
//* 'grep 'Notes:                                                              *
//* =============                                                              *
//*                                                                            *
//* command-line options:                                                      *
//*  -i       ignore case                                                      *
//*  -v       invert match                                                     *
//*  -w       whole-word matches only                                          *
//*  -x       whole-line matches only                                          *
//*  -c       report count of matches                                          *
//*  -l       report filenames with matches                                    *
//*  -m NUM   stop reading file after NUM matches                              *
//*  -o       report only the matching parts (not useful)                      *
//*  -s       suppress error messages                                          *
//*  -H       report filename for each match (default if multiple files)       *
//*  -n       report line number for matches                                   *
//*  -a       process binary files as text                                     *
//*  -r       recurse into subdirectories                                      *
//*  --color=always  always insert ANSI color sequences (retained on redirect) *
//*                  Note: To display these colors when piping to 'less', use  *
//*                        grep -n --color=always 'version' *.hpp | less -R    *
//*         =auto    automatic ANSI determination (stripped before redirect)   *
//*         =never   no ANSI sequences                                         *
//*                                                                            *
//*  context:                                                                  *
//*   -A NUM    lines After                                                    *
//*   -B NUM    lines Before                                                   *
//*   -C NUM    lines Before _and_ After                                       *
//*                                                                            *
//* search pattern options:                                                    *
//*  1) case sensitive                                                         *
//*  2) case insensitive                                                       *
//*  3) plain text string                                                      *
//*  4) stringA AND stringB                                                    *
//*  5) text[range]text                                                        *
//*  6) [.] [*] [^] [$] (literal period, astrisk, caret, dollar)               *
//*  7) ^  anchor to beginning of line                                         *
//*  8) $  anchor to end of line                                               *
//*  9) backslash combinations:                                                *
//*     \b \B edges of words                                                   *
//*           '\blog\b' (whole word only)                                      *
//*           '\blog\B' (word beginning with)                                  *
//*           '\Blog\b' (word ending with)                                     *
//*           '\Blog\B' (interior of a word)                                   *
//*     \< \> empty string at beginning/end of word (overlap with \b, \B)      *
//*     \w    [_[:alnum:]]    (single alphanumeric character)                  *
//*     \W    [^_[:alnum:]]   (single non-alphanumeric character)              *
//*     \s    [[:space:]]     (single whitespace character)                    *
//*     \S    [^[:space:]]    (single non-whitespace character)                *
//*                                                                            *
//*     "word constituents" are letters, digits and the underscore ('_')       *
//*     "whitespace" is space, tab, newline                                    *
//* 10) bracket expressions                                                    *
//*     [xyz]                 matches 'x' or 'y' or 'z'         (locale depend)*
//*     [a-r]                 matches 'a' through 'r' inclusive (locale depend)*
//*                                                                            *
//* 11) repetition operators                                                   *
//*     .                                                                      *
//*     ?                                                                      *
//*     *                                                                      *
//*     +                                                                      *
//*     {N}                                                                    *
//*     {N,}                                                                   *
//*     {,M}                                                                   *
//*     {N,M}                                                                  *
//* 12) |  infix operator: matches  A | B                                      *
//*                                                                            *
//*                                                                            *
//* Notes on the XML files in LibreOffice documents.                           *
//* ================================================                           *
//* 1) A LibreOffice/OpenOffice document is composed of some number of         *
//*    individual files laid out in a specific tree format and packed into a   *
//*    .ZIP archive.                                                           *
//*    a) The file which contains the actual document text is "content.xml",   *
//*       and lives at the top level of the archive tree.                      *
//*    b) The remaining files and subdirectories in the archive provide various*
//*       forms of support for display and editing of the document by the      *
//*       LibreOffice suite of applications.                                   *
//*    c) Extract the "content.xml" file from the archive:                     *
//*        unzip -Co SOURCEFILE content.xml -dTRGPATH 1>/dev/null 2>/dev/null *
//*                                                                            *
//* 2) The "content.xml" sub-document begins with a file-type identifier,      *
//*    example:  <?xml version="1.0" encoding="UTF-8"?>                        *
//*    This is terminated by a newline character. This may be the ONLY newline *
//*    in the entire file.                                                     *
//*    a) Note that when an XML file is displayed in a browser, it appears to  *
//*       be formated line-by-line, but this is actually done by the browser   *
//*       itself according to some internal idea of parsing the angle-bracked  *
//*       commands.                                                            *
//*                                                                            *
//* 3) The actual document text is found only OUTSIDE of angle-bracket pairs.  *
//*    Examples:  (line formatting provided by Firefox)                        *
//*                                                                            *
//*     <text:p text:style-name="P26">San Diego, CA 92123</text:p>             *
//*                                                                            *
//*     <text:p text:style-name="P28">                                         *
//*       They sent a letter wherein the total due                             *
//*     </text:p>                                                              *
//*     <text:p text:style-name="P28"><text:tab/><text:tab/>                   *
//*       $7,567.27                                                            *
//*     </text:p><text:p text:style-name="P28">                                *
//*       will be reduced if payment in full is received by 15 Jun, 2017 to:   *
//*     </text:p><text:p text:style-name="P28"><text:tab/><text:tab/>          *
//*       $6,053.00                                                            *
//*     </text:p>                                                              *
//*                                                                            *
//*    a) Note that we assume the XML file contains no angle brackets used as  *
//*      text. Instead, it should use "&lt;" and "&gt". Although it is         *
//*      possible to use a CDATA sequence: <![CDATA[<]]> or <![CDATA[>]]>      *
//*      what kind of self-loathing cuck would do that?                        *
//*                                                                            *
//* 4) To grep the text in a meaningful way, it must be separated from the     *
//*    surrounding XML command sequences.                                      *
//*    a) To do this, we open the "content.xml" file (as text) and step over   *
//*       all data enclosed within angle brackets, saving the actual text data *
//*       to a temporary file. For instance, for the XML sequence:             *
//*          <text:p text:style-name="P26">San Diego, CA 92123</text:p>        *
//*       the data written to the temporary file will be:                      *
//*          San Diego, CA 92123                                               *
//*       terminated by a newline character (which is not part of the source   *
//*       document).                                                           *
//*    b) Note that the insertion of newline characters is arbitrary, in that  *
//*       we cannot easily determine where the XML generator intends for the   *
//*       natural linebreak to occur. Additional research may yield insight    *
//*       into this problem. It may be that actually parsing some subset of    *
//*       the XML commands could yield the position of the intended linebreak. *
//*                                                                            *
//* 5) The file containing the extracted text data is written to a temporary   *
//*    file in the application's temp directory. This temporary file will have *
//*    the same name as the original source document. This allows the 'grep'   *
//*    utility to report the correct filename in conjunction with text that    *
//*    matches the regexp expression (but see next item).                      *
//*                                                                            *
//* 6) Special formatting for OpenDocument scan results:                       *
//*    For search results from OpenDocument files, the individual filenames    *
//*    for the source files include the full path of the source file rather    *
//*    than the filename only as with the scan of text files.                  *
//*    This may seem to be a cosmetic issue, but it may seriously confuse the  *
//*    user because it is a filespec of a temporary file with the same name as *
//*    the OpenDocument file. For this reason we strip the path from the source*
//*    filename in the search results.                                         *
//*                                                                            *
//*    How we do this:                                                         *
//*    Each source line of the grep output begins with an ANSI color attribute *
//*    followed by the filespec, which is then followed by the remaining color *
//*    attributes and text data for that line.                                 *
//*    We remove only the path, leaving the ANSI color data and other text data*
//*    intact. The filename is extracted from the filespec and is written      *
//*    separately. See "Transfer the OpenDocument search results" below.       *
//*                                                                            *
//*                                                                            *
//* Notes on the XML files in OpenXML (MS-Office) documents                    *
//* =======================================================                    *
//*                                                                            *
//*                                                                            *
//*                                                                            *
//*                                                                            *
//*                                                                            *
//* Notes on extracting the text from older MS-Word documents                  *
//* =========================================================                  *
//*                                                                            *
//*                                                                            *
//******************************************************************************

bool FileDlg::gfScan ( const gString fnList, const gString odList, const gString& gsPat, 
                       const gString& gsFilt, const gString& gsCap, 
                       bool lineNum, bool caseSen, bool matchCnt )
{
   #define DB_WINDOW (0)   // for debugging only - for single-window mode only
   #if ENABLE_DEBUGGING_CODE != 0 && DB_WINDOW != 0
   gString gsdb ;
   winPos  wpdb( 1, 1 ) ;
   attr_t dbColor = nc.grR ;
   short dbRows, dbCols ;
   NcDialog* dbPtr = NULL ;
   #endif   // ENABLE_DEBUGGING_CODE != 0 && DB_WINDOW != 0

   const char* const switchString[] = 
   {
      " ",                 // no switches specified
      "c",                 // 'c' overrides both 'n' and 'i'
      "n",                 // 'n' only
      "i",                 // 'i' only
      "ni",                // 'n' + 'i'
   } ;
   const char *switchPtr = switchString[0] ; // assume no switch
   gString gsFilter( " " ) ;     // secondary filter (initially, no filter)
   gString gsCmd ;               // for constructing system commands

   //* Because the capture file is reused, be sure *
   //* its existing data have been truncated.      *
   ofstream capofs( gsCap.ustr(), ofstream::out | ofstream::trunc ) ;
   capofs.close() ;

   //* If a secondary filter specified *
   if ( (gsFilt.gschars()) > 1 )
   {
      gsFilter = gsFilt ;
      short offset = gsFilter.find( L'-' ) ;
      if ( offset == 1 )
         gsFilter.erase( L"-", offset ) ;
      gsFilter.insert( L"| grep --color=always " ) ;
      if ( offset == 1 )
         gsFilter.insert( L"-v ", 22 ) ;
   }

   //* Convert switch settings to switch string.*
   if ( matchCnt != false )
      switchPtr = switchString[1] ;    // count matches
   else if ( lineNum && caseSen )
      switchPtr = switchString[2] ;    // report line numbers (case sensitive)
   else if ( !lineNum && !caseSen )
      switchPtr = switchString[3] ;    // case insensitive scan
   else if ( lineNum && !caseSen )
      switchPtr = switchString[4] ;    // report line numbers + case insensitive scan

   //* If one or more regular-text filenames specified.*
   if ( fnList.gschars() > 1 )
   {
      const char* grTemplate = "grep --color=always -H%s %S %S %S 1>\"%S\" 2>/dev/null" ;
      //* Create the system command *
      gsCmd.compose( grTemplate, switchPtr, 
                     gsPat.gstr(), fnList.gstr(), gsFilter.gstr(), gsCap.gstr() ) ;

      //* Execute the command, redirecting the output to temp file *
      /*short exitCode = */this->fmPtr->Systemcall ( gsCmd.ustr() ) ;

      #if ENABLE_DEBUGGING_CODE != 0 && DB_WINDOW != 0
      if ( (dbPtr = this->Open_DBwindow ( dbColor, dbRows, dbCols )) != NULL )
      {
         gsdb.compose( "COMMAND:  %S", gsCmd.gstr() ) ;
         short offset = ZERO ;
         if ( (offset = (gsdb.after( gsPat )) + 1) > ZERO )
            gsdb.insert( L'\n', offset ) ;
         if ( ((gsFilter.gschars()) > 1) && ((offset = gsdb.find( gsFilter)) >= ZERO) )
            gsdb.insert( L'\n', offset ) ;
         if ( (offset = gsdb.find( "1>" )) >= ZERO )
            gsdb.insert( L'\n', offset ) ;
         gsdb.append( L"\n\n" ) ;
         wpdb = dbPtr->WriteParagraph ( wpdb, gsdb, dbColor, true ) ;
      }
      #endif   // ENABLE_DEBUGGING_CODE != 0 && DB_WINDOW != 0
   }

   //* If OpenDocument filename(s) specified *
   if ( odList.gschars() > 1 )
   {  //* See notes in method header which describe manipulation *
      //* of OpenDocument files.                                 *
      const char* grTemplate = "grep --color=always -H%s %S \"%S\" %S 1>\"%S\" 2>/dev/null" ;

      gString tmpPath,        // path of application temp dir
              tmpSpec_old,    // filespec of target temp file (old)
              tmpSpec_new,    // filespec of target temp file (new)
              srcSpec,        // filespec of source document
              srcName,        // name of source document
              contentSpec,    // filespec of 'content.xml'
              odCap,          // filespec of temp file to hold grep output
              gsEntity ;      // convert "predefined entity" characters
      ofstream ofs ;          // write to output file
      ifstream ifs ;          // read from input file
      char lineBuff[gsDFLTBYTES] ; // input-stream line buffer
      fmFType fType ;         // file type code

      //* Create a temporary file as the target of the OpenDocument grep.*
      this->fmPtr->CreateTempname ( odCap ) ;

      this->fmPtr->ExtractPathname ( tmpPath, gsCap ) ;  // get tempfile directory
      this->fmPtr->CreateTempname ( tmpSpec_old ) ;      // create a temp file

      #if ENABLE_DEBUGGING_CODE != 0 && DB_WINDOW != 0
      // Programmer's Note: The debugging output cannot interpret multi-byte 
      // characters in the input stream. This is not a critical issue because 
      // our test documents contain very few non-ASCII characters.
      if ( dbPtr == NULL )
         dbPtr = this->Open_DBwindow ( dbColor, dbRows, dbCols ) ;
      if ( dbPtr != NULL )
      {
         gsdb.compose( "tmpPath    : %S\n"
                       "tmpSpec_old: %S\n"
                       "odList     : %S\n\n", 
                       tmpPath.gstr(), tmpSpec_old.gstr(), odList.gstr() ) ;
         wpdb = dbPtr->WriteParagraph( wpdb, gsdb, dbColor, true  ) ;
      }
      #endif   // ENABLE_DEBUGGING_CODE != 0 && DB_WINDOW != 0

      //* For each filename in the odList buffer *
      arcType aType ;                        // archive type (s/b either atODOC or atOXML)
      short windx = ZERO,                    // index into odList.gstr()
            lindx = odList.gschars() - 1 ;   // index of nullchar
      while ( windx < lindx )
      {
         //* Get a source filename from the buffer.                  *
         //* Remove backstroke on all escaped characters because the *
         //* rename operation would see '\' as a filename character. *
         windx = this->gfExtractFname ( odList, windx, srcName ) ;
         this->fmPtr->RemoveCharEscapes ( srcName ) ;

         //* Create the source filespec and verify *
         //* that is a supported file format.      *
         this->fmPtr->CatPathFname ( srcSpec, this->currDir, srcName.ustr() ) ;
         aType = this->ArchiveTarget ( srcSpec ) ;

         //* Rename the temp file using name of source document.  *
         this->fmPtr->CatPathFname ( tmpSpec_new, tmpPath.gstr(), srcName.gstr() ) ;
         this->fmPtr->RenameFile ( tmpSpec_old.ustr(), tmpSpec_new.ustr() ) ;
         tmpSpec_old = tmpSpec_new ;   // remember the new filespec

         #if ENABLE_DEBUGGING_CODE != 0 && DB_WINDOW != 0
         gsdb.compose( "srcName    : %S\n"
                       "srcSpec    : %S\n" 
                       "tmpSpec_new: %S\n",
                       srcName.gstr(), &srcSpec.gstr()[22], tmpSpec_new.gstr() ) ; 
         wpdb = dbPtr->WriteParagraph( wpdb, gsdb, dbColor, true  ) ;
         #endif   // ENABLE_DEBUGGING_CODE != 0 && DB_WINDOW != 0

         //* If source is an MS-Word binary file *
         //*   (Word 1997-2000-2002-2003-2007*   *
         if ( (aType == atNONE) && (this->odIsBinDocFile ( srcSpec )) )
         {
            //* Extract text data from binary file   *
            //* and convert to human readable format.*
            this->vfcDecodeBinDoc ( srcSpec, tmpSpec_new ) ;
         }

         //* Extract the XML containing the OpenDocument/OpenXML text data *
         else if ( (this->odExtractOD_Content ( srcSpec, tmpPath, contentSpec, aType )) &&
                   (this->TargetExists ( tmpSpec_new, fType )) )
         {
            //* Scan the XML source and extract the text data.*
            #if ENABLE_DEBUGGING_CODE != 0 && DB_WINDOW != 0
            this->odExtractOD_Text ( srcSpec, contentSpec, tmpSpec_new, aType, dbPtr, &wpdb ) ;
            #else    // PRODUCTION
            this->odExtractOD_Text ( srcSpec, contentSpec, tmpSpec_new, aType ) ;
            #endif   // ENABLE_DEBUGGING_CODE && DB_WINDOW
         }

         //* Create the command to grep the data in the temp file,*
         //* and execute the command, appending the results to    *
         //* capture file.                                        *
         //* If the filename contains one of the characters which *
         //* would cause the shell to choke ( " \ ` $ ), then     *
         //* escape those characters.                             *
         this->EscapeSpecialChars ( tmpSpec_new, escMIN ) ;
         gsCmd.compose( grTemplate, switchPtr, 
                        gsPat.gstr(), tmpSpec_new.gstr(), gsFilter.gstr(), odCap.gstr() ) ;

         #if ENABLE_DEBUGGING_CODE != 0 && DB_WINDOW != 0
         gsdb.compose( "COMMAND:  %S\n\n", gsCmd.gstr() ) ;                
         short offset = ZERO ;
         if ( (offset = (gsdb.after( gsPat )) + 1) > ZERO )
            gsdb.insert( L'\n', offset ) ;
         if ( ((gsFilter.gschars()) > 1) && ((offset = gsdb.find( gsFilter)) >= ZERO) )
            gsdb.insert( L'\n', offset ) ;
         if ( (offset = gsdb.find( "1>" )) >= ZERO )
            gsdb.insert( L'\n', offset ) ;
         wpdb = dbPtr->WriteParagraph( wpdb, gsdb, dbColor, true ) ;
         #endif   // ENABLE_DEBUGGING_CODE != 0 && DB_WINDOW != 0

         /*short exitCode = */this->fmPtr->Systemcall ( gsCmd.ustr() ) ;

         //* Transfer the OpenDocument search results to the master           *
         //* search-results file, stripping the path from source filenames.   *
         tnFName tnFile ;
         this->fmPtr->GetFileStats ( tnFile, odCap ) ;
         if ( tnFile.fBytes != ZERO )
         {
            gString gsIn,
                    extended_tmpPath( "%S/", tmpPath.gstr() ) ;
            bool done = false ;

            //* Open odCap for reading *
            ifs.open( odCap.ustr(), ifstream::in ) ;

            //* Open gsCap for append  *
            capofs.open( gsCap.ustr(), (ofstream::out | ofstream::app) ) ;

            if ( (ifs.is_open()) && (capofs.is_open()) )
            {
               char nameBuff[gsDFLTBYTES] ;  // receives filespec
               short nbindx,                 // index into nameBuff
                     lbindx ;                // index into lineBuff
               while ( ! done )
               {
                  ifs.getline( lineBuff, gsDFLTBYTES, NEWLINE ) ;
                  if ( (ifs.good()) || ((ifs.gcount()) > ZERO) )
                  {
                     for ( lbindx = ZERO ; lbindx < ifs.gcount() ; ++lbindx )
                     {
                        if ( lineBuff[lbindx] != fSLASH )
                           capofs.put( lineBuff[lbindx] ) ;
                        else
                           break ;
                     }
                     for ( nbindx = ZERO ; lbindx < ifs.gcount() ; ++lbindx )
                     {
                        if ( lineBuff[lbindx] != ESC )
                           nameBuff[nbindx++] = lineBuff[lbindx] ;
                        else
                           break ;
                     }
                     nameBuff[nbindx] = NULLCHAR ;    // terminate filespec
                     gsIn = nameBuff ;
                     nbindx = gsIn.findlast( fSLASH ) + 1 ;
                     gsIn.shiftChars( -nbindx ) ;
                     capofs.write( gsIn.ustr(), (gsIn.utfbytes() - 1) ) ;
                     capofs.write( &lineBuff[lbindx], (ifs.gcount() - lbindx - 1) ) ;
                     capofs.put( NEWLINE ) ;
                  }
                  else        // end-of-file
                     done = true ;
               }
            }
         }
         ifs.close() ;                             // close the files
         capofs.close() ;
      }     // while()

      //* Delete the temp grep capture file.*
      this->fmPtr->DeleteTempname ( odCap ) ;

      //* Delete the old 'content.xml' file.*
      this->DeleteFile ( contentSpec ) ;

      //* Delete the temp conversion file. *
      this->fmPtr->DeleteTempname ( tmpSpec_new ) ;

   }  // if(odList)

   //* If no data in the output file, insert an informational message.*
   //* First, 'stat' the capture file to get its size (in bytes).     *
   tnFName tnFile ;
   this->fmPtr->GetFileStats ( tnFile, gsCap ) ;
   if ( tnFile.fBytes == ZERO )
   {  //* Insert a friendly informational message.*
      const char msg[49] = "No text matching the search pattern was found.\n\n" ;
      capofs.open( gsCap.ustr(), ofstream::out | ofstream::trunc ) ;
      capofs.write( msg, 48) ;
      capofs.close() ;
   }

   #if ENABLE_DEBUGGING_CODE != 0 && DB_WINDOW != 0
   if ( dbPtr != NULL )
   { nckPause(); delete dbPtr ; }
   #endif   // ENABLE_DEBUGGING_CODE != 0 && DB_WINDOW != 0

   return true ;

   #undef DB_WINDOW
}  //* End gfScan() *

//*************************
//*    gfExtractFname     *
//*************************
//******************************************************************************
//* Extract a filename from the space-delimited list of filenames.             *
//* The filename may, or may not be SQUOTE or DQUOTE delimited.                *
//* Delimiter characters within filenames will be escaped.                     *
//* Space characters within filenames may be escaped.                          *
//*                                                                            *
//* Input  : odList  : list of filenames                                       *
//*          indx    : index at which to begin the scan                        *
//*          srcName : (by reference) receives extracted filename              *
//*                    if no filename found, empty string                      *
//*          strip   : (optional, 'true' by default)                           *
//*                    if 'true', strip any existing delimiters from the       *
//*                               returned filename                            *
//*                    if 'false', retain any existing delimiters with         *
//*                                returned filename                           *
//*                                                                            *
//* Returns: index of NEXT filename or index of NULLCHAR                       *
//******************************************************************************

short FileDlg::gfExtractFname ( const gString& odList, short indx, 
                                gString& srcName, bool strip )
{
   wchar_t wBuff[gsALLOCDFLT] ;        // work buffer
   odList.copy( wBuff, gsALLOCDFLT ) ; // make a working copy
   wchar_t delimChar = wBuff[indx] ;   // delimiter character
   short bindx = indx,                 // index begining of filename
         windx = bindx + 1 ;           // scanning index

   if ( (delimChar == SQUOTE) || (delimChar == DQUOTE) ) // if a delimiter
   {
      if ( strip != false )
         ++bindx ;            // step forward to first filename character
   }
   else
      delimChar = nckSPACE ;  // no delimiter, scan to space character

   //* Scan to delimter character or to end of string.*
   while ( wBuff[windx] != NULLCHAR )
   {
      if ( wBuff[windx] == delimChar )
      {
         //* If this is an escaped SQUOTE or DQUOTE or SPACE, step over it.*
         if ( wBuff[windx - 1] == BKSTROKE )
            ++windx ;
         else                 // closing delimiter found
            break ;
      }
      else if ( wBuff[windx] != nckNULLCHAR )
         ++windx ;
   }

   if ( wBuff[windx] != NULLCHAR )     // if not already at end of string
   {
      //* If indexing a delimiter:                            *
      //* strip    : terminate the filename by overwriting    *
      //*            closing delimiter. Then if not indexing  *
      //*            the NULLCHAR, index the next filename.   *
      //* ! strip  : terminate the filename by overwriting    *
      //*            the SPACE character separating filenames.*
      if ( (wBuff[windx] == SQUOTE) || (wBuff[windx] == DQUOTE) )
      {
         if ( strip == false )
            ++windx ;
         if ( wBuff[windx] != nckNULLCHAR )
            wBuff[windx++] = nckNULLCHAR ;
      }
      else if ( wBuff[windx] != nckNULLCHAR )
         wBuff[windx++] = nckNULLCHAR ;

      while ( wBuff[windx] == nckSPACE )  // step over whitespace between filenames
         ++windx ;
   }
   srcName = &wBuff[bindx] ;

   return windx ;

}  //* End gfExtractFname() *

//***************************
//* gfConstructFilenameList *
//***************************
//******************************************************************************
//* Convert specified regexp expression into a list of source filenames.       *
//*                                                                            *
//* Input  : fnList    : (by reference) on entry, contains regexp for group    *
//*                        of filenames to be scanned                          *
//*                        on return, contains constructed list text-only      *
//*                        filenames                                           *
//*          odList    : (by reference) initial contents ignored               *
//*                        on return, contains constructed list of OpenDocument*
//*                        filenames.                                          *
//*          scanCount : (by reference) receives number of filenames validated *
//*                                                                            *
//* Returns: 'true'  if one or more files validated for scan                   *
//*                   AND if no filenames discarded due to buffer overflow     *
//*          'false' if no validated files OR buffer overflow (unlikely)       *
//******************************************************************************

bool FileDlg::gfConstructFilenameList ( gString& fnList, gString& odList, short& scanCount )
{
   gString fileSpec,       // path/filename of file to be tested
           tmpPath,        // filespec of temporary file for capture of 'ls' output
           tmpVerify,      // filespec of temporary file for file validation
           gsDelimit ;     // for filename delimiting
   bool    status = true ; // return value
   scanCount = ZERO ;      // initialize caller's filename count

   //* Create temporary files to hold the output of 'ls' and 'file' utilities.*
   this->fmPtr->CreateTempname ( tmpPath ) ;
   this->fmPtr->CreateTempname ( tmpVerify ) ;

   //* Create the 'ls' command. (We assume that CWD is correct.) *
   gString gsCmd( "ls -1 %S 1>\"%S\" 2>/dev/null", fnList.gstr(), tmpPath.gstr() ) ;

   //* Execute the command, redirecting the output to temp file *
   /*short exitCode = */this->fmPtr->Systemcall ( gsCmd.ustr() ) ;

   //* Extract filenames from the temp file and verify     *
   //* that the files are both 'regular' and 'text' files. *
   //* Write validated filename list to callers buffer.    *
   ifstream ifs( tmpPath.ustr(), ifstream::in ) ;
   if ( ifs.is_open() )
   {
      fnList.clear() ;              // reinitialize the target buffers
      odList.clear() ;

      char  lineData[gsDFLTBYTES] ; // raw UTF-8 input
      bool  isRt,                   // 'true' if regular, text file
            done = false ;          // loop control

      while ( ! done )
      {
         ifs.getline ( lineData, gsDFLTBYTES, NEWLINE ) ;
         if ( ifs.good() || ifs.gcount() > ZERO )
         {
            //* Test the file-type and MIME-type encoding *
            this->fmPtr->CatPathFname ( fileSpec, this->currDir, lineData ) ;
            if ( (isRt = this->gfIsRegTextfile ( fileSpec, tmpVerify, true )) ||
                 (this->odIsOpenDocFile ( fileSpec )) ||
                 (this->odIsOpenXmlFile ( fileSpec )) ||
                 (this->odIsBinDocFile ( fileSpec )) )
            {
               //* If the filename contains Linux "special characters",  *
               //* or spaces (' ') escape those characters.              *
               //* Note that if the filenames contain "special"          *
               //* characters, the 'ls' output will have delimited those *
               //* filenames, BUT the delimiters will have been stripped *
               //* by the shell during redirection.                      *
               // Programmer's Note: We can't perform the escaping before 
               // testing the filetype because the lstat() function does its own escaping, 
               // (but does it badly) in that it requires an un-escaped filespec.
               gsDelimit = lineData ;
               this->fmPtr->EscapeSpecialChars ( gsDelimit, escLINUX ) ;
               this->fmPtr->EscapeSpecialChars ( gsDelimit, escWCHAR, L' ' ) ;
               if ( isRt )
                  fnList.append( "%S ", gsDelimit.gstr() ) ;
               else
                  odList.append( "%S ", gsDelimit.gstr() ) ;
               ++scanCount ;
            }

            //* Prevent buffer overflow *
            if ( (fnList.utfbytes() >= MAX_PAT) || (odList.utfbytes() >= MAX_PAT) )
            {
               status = false ;
               done = true ;
            }
         }
         else
            done = true ;
      }
      ifs.close() ;                 // close the temp file
   }

   //* Delete the temp files *
   this->fmPtr->DeleteTempname ( tmpPath ) ;
   this->fmPtr->DeleteTempname ( tmpVerify ) ;

   if ( scanCount == ZERO )
      status = false ;
   return ( status ) ;

}  //* End gfConstructFilenameList() *

//***************************
//* gfValidateFilenameList  *
//***************************
//******************************************************************************
//* Parse the filename list and verify that the filenames reference 'regular'  *
//* files and that those files are either text-only files ('fnList') OR        *
//* OpenDocument/OpenXML files ('odList').                                     *
//*                                                                            *
//* Adjust formatting of the filenames as necessary to delimit filenames       *
//* and/or to escape special characters.                                       *
//*                                                                            *
//* This filename list was manually entered by the user, comma-separated.      *
//*                                                                            *
//* Input  : fnList    : (by reference) on entry, contains user's list of      *
//*                         manually-typed filenames, and on return contains   *
//*                         list of text-only filenames                        *
//*          odList    : (by reference) initial contents ignored               *
//*                         on return, contains list of OpenDocument/OpenXML   *
//*                         filenames                                          *
//*          scanCount : (by reference) receives number of filenames validated *
//*                                                                            *
//* Returns: 'true'  if one or more files validated for scan                   *
//*                   AND if no filenames discarded due to buffer overflow     *
//*          'false' if no validated files OR buffer overflow (unlikely)       *
//******************************************************************************

bool FileDlg::gfValidateFilenameList ( gString& fnList, gString& odList, short& scanCount )
{
   short nullIndex = fnList.gschars() - 1 ;  // index of NULLCHAR in fnList
   bool  status = true ;                     // return value

   scanCount = ZERO ;      // initialize caller's filename counter

   if ( nullIndex > ZERO )
   {
      gString fileSpec,       // path/filename of file to be tested
              tmpVerify,      // filespec of temporary file for file validation
              baseDir( this->currDir ),   // base target directory (CWD)
              gsDelimit ;     // for delimiting filenames
      wchar_t wfBuff[gsALLOCDFLT],  // temp file-list buffer
              wfName[gsALLOCDFLT] ; // temp filename buffer
      short   indx = ZERO,    // index into filename list
              wfindx ;        // index into wfName
      bool    isRt ;          // 'true' if regular, text file

      fnList.copy( wfBuff, gsALLOCDFLT ) ; // working copy of filename list
      fnList.clear() ;        // reinitialize the target buffers
      odList.clear() ;

      //* Create temporary file to hold the output of 'file' utility.*
      this->fmPtr->CreateTempname ( tmpVerify ) ;

      //* User must escape the following characters when used in ordinary *
      //* filenames:  '*' '|' '^' '$' '[' ']' '{' '}' '+' '\'             *
      //* However, we must REMOVE any escapes before proceeding.          *
      //* Programmer's Note: If the filename contains an unescaped '\' as *
      //* an ordinary character, then that filename will be corrupted and *
      //* not included in the filename list. (this is unlikely)           *
      gsDelimit = wfBuff ;
      if ( (this->fmPtr->RemoveCharEscapes ( gsDelimit )) > ZERO )
         gsDelimit.copy( wfBuff, gsDFLTBYTES ) ;

      while ( indx < nullIndex )
      {
         //* Step over leading whitespace and discard trailing whitespace.*
         while ( (indx < nullIndex) && (wfBuff[indx] == nckSPACE) )
            ++indx ;
         if ( indx >= nullIndex )
            break ;

         for ( wfindx = ZERO ; (indx < nullIndex) ; ++wfindx )
         {
            if ( wfBuff[indx] == COMMA )
            {
               ++indx ;
               break ;
            }
            wfName[wfindx] = wfBuff[indx++] ;
         }
         wfName[wfindx] = nckNULLCHAR ; // terminate the string

         //* Validate the file type and contents *
         this->fmPtr->CatPathFname ( fileSpec, baseDir.gstr(), wfName ) ;
         if ( (isRt = this->gfIsRegTextfile ( fileSpec, tmpVerify, true )) ||
              (this->odIsOpenDocFile ( fileSpec )) ||
              (this->odIsOpenXmlFile ( fileSpec )) ||
              (this->odIsBinDocFile ( fileSpec )) )
         {
            //* If the filename contains Linux "special characters", *
            //* or spaces (' ') escape those characters.             *
            gsDelimit = wfName ;
            this->fmPtr->EscapeSpecialChars ( gsDelimit, escLINUX ) ;
            this->fmPtr->EscapeSpecialChars ( gsDelimit, escWCHAR, L' ' ) ;

            //* Update the list by appending the filename.*
            if ( isRt )
               fnList.append( "%S ", gsDelimit.gstr() ) ;
            else
               odList.append( "%S ", gsDelimit.gstr() ) ;
            ++scanCount ;

            //* Prevent buffer overflow (unlikely) *
            if ( (fnList.utfbytes() >= MAX_PAT) || (odList.utfbytes() >= MAX_PAT) )
            {
               status = false ;
               break ;
            }
         }
      }     // while()

      this->fmPtr->DeleteTempname ( tmpVerify ) ;  // delete the temp file
   }

   if ( scanCount == ZERO )
      status = false ;
   return ( status ) ;

}  //* End gfValidateFilenameList() *

//*************************
//*    gfIsRegTextfile    *
//*************************
//******************************************************************************
//* Verify that the specified file's filetype is 'regular', AND that the file  *
//* contains only text.                                                        *
//*                                                                            *
//* Input  : fileSpec  : path/filename of file to be validated                 *
//*          tmpFile   : temporary file for capture of stdout data             *
//*          checkReg  : (optional, 'false' by default)                        *
//*                      if 'false': assume that caller has verified that the  *
//*                                  file is a 'regular' file                  *
//*                      if 'true' : stat the target file to verify that it is *
//*                                  a 'regular' file                          *
//*                                                                            *
//* Returns: 'true;  if source file validated                                  *
//*          'false' if source file is either not 'regular' or is not 'text'   *
//******************************************************************************
//* Notes:                                                                     *
//* ======                                                                     *
//*                                                                            *
//* Determining whether a file is binary:                                      *
//*    file --mime-encoding filename                                           *
//*        yields: "binary"  "utf-8"  "ascii"  etc.                            *
//*                                                                            *
//* Note: The 'file' system utility reads over 1K of bytes from the file and   *
//*       if no NULLCHAR (0x00) characters are found, assumes that the file    *
//*       contains only text i.e. that it IS NOT a binary file.                *
//*                                                                            *
//* Programmer's Note: If the 'fileSpec' text contains a double-quote          *
//* character, the system call would fail due to the un-escaped delimiter.     *
//* This is a remote possibility because only a moron would use double-quotes  *
//* in a filename, but unfortunately, there are a few morons in the world.     *
//* For this reason, we "escape" the critical filename characters which might  *
//* be misinterpreted by the shell.                                            *
//******************************************************************************

bool FileDlg::gfIsRegTextfile ( const gString& fileSpec, const gString& tmpFile, bool checkReg )
{
   const char* txtTemplate = "file --brief --mime-encoding \"%S\" 1>\"%S\" 2>/dev/null" ;
   gString gsCmd,             // constructed sys-call command
           gsCompare ;        // for comparison
   tnFName tnFile ;           // file stats
   bool    status = false ;   // return value

   if ( checkReg != false )
      this->fmPtr->GetFileStats ( tnFile, fileSpec ) ;
   else
      tnFile.fType = fmREG_TYPE ;

   if ( tnFile.fType == fmREG_TYPE )
   {
      //* Construct and execute the command *
      gString escPath = fileSpec ;
      this->EscapeSpecialChars ( escPath, escMIN ) ;
      gString gsCmd( txtTemplate, escPath.gstr(), tmpFile.gstr() ) ;
      this->fmPtr->Systemcall ( gsCmd.ustr() ) ;

      //* Read the only line in the output file *
      ifstream ifs( tmpFile.ustr(), ifstream::in ) ;
      if ( ifs.is_open() )
      {  //* This file should have only one line *
         char lineData[gsDFLTBYTES] ;
         ifs.getline( lineData, gsDFLTBYTES, NEWLINE ) ;
         if ( ifs.good() || ifs.gcount() > ZERO )
         {
            //* If utility reports that the target   *
            //* IS NOT a binary file, return success.*
            gsCompare = lineData ;
            if ( (gsCompare.find( "binary" )) < ZERO )
               status = true ;
         }
         ifs.close() ;
      }
   }
   return status ;

}  //* End gfIsRegTextfile() *

//*************************
//*   gfIsRegexpPattern   *
//*************************
//******************************************************************************
//* Scan the specified string for a potential regexp pattern.                  *
//*                                                                            *
//*                                                                            *
//* Input  : fnList    : (by reference) regexp for group of filenames          *
//*                                                                            *
//* Returns: 'true'  if string contains non-escaped regexp characters          *
//*          'false' if no regexp characters found                             *
//******************************************************************************
//* Filename Pattern:                                                          *
//* -----------------                                                          *
//* The user may 'select' files before calling this method. If so a list of    *
//* those files is used and the Filename Textbox is set inactive (read-only),  *
//* in which case, this method is not called.                                  *
//*                                                                            *
//* If no 'selected' filenames, then user may manually enter a list of         *
//* filenames, OR may enter a regexp filename pattern. This method must        *
//* determine which the user has entered.                                      *
//*                                                                            *
//* The basic single-character regexp characters (unless escaped) technically  *
//* should be seen as potential regexp; HOWEVER, these characters may ALSO be  *
//* valid filename characters.                                                 *
//* a) The '*' and '|' (vertical bar) characters would only be used as filename*
//*    characters by a moron, so we test for these first.                      *
//* b) Line anchors '^' and '$' are considered as regexp unless escaped.       *
//* c) The '+' '[' ']' '{' '}' characters carry varying degrees of confidence  *
//*    as regexp rather than filename characters. The author does not          *
//*    personally use these characters in filenames, so our prejudice is to    *
//*    assume that they are in fact regexp unless they are escaped.            *
//* d) The '?' (question mark) character is often seen in filenames such as    *
//*    in the names of songs and book titles. Although we disagree with this   *
//*    usage, it is difficult to arbitrarily penalize those who use it as such.*
//*    In addition, it is less often used in simple regexp searches such as    *
//*    filenames. For these reasons, (and because we can't trust the user to   *
//*    escape the thing) we have made the design decision to treat the '?' as  *
//*    a regular character and do not test for it in determining whether a     *
//*    string is regexp.                                                       *
//* e) The '.' (full-stop) character is just as likely to be a filename        *
//*    character as a regexp character, (and we can't trust the user to escape *
//*    it), so we have made the design decision to treat '.' as a regular      *
//*    character and do not test for it in determining whether a string is     *
//*    regexp.                                                                 *
//******************************************************************************

bool FileDlg::gfIsRegexpPattern ( const gString& fnList )
{
   bool isregexp = false ;          // return value

   if ( fnList.gschars() > 1 )      // check for empty string
   {
      short ix ;
      ix = fnList.find( L'*' ) ;    // most-likely suspects
      if ( ix < ZERO )
         ix = fnList.find( L'|' ) ;
      if ( (ix == ZERO) ||
           ((ix > ZERO) && (fnList.gstr()[ix - 1] != L'\\')) )
         isregexp = true ;
      else                          // anchor characters
      {
         ix = fnList.find( L'^' ) ;
         if ( ix < ZERO )
            ix = fnList.find( L'$' ) ;
         if ( (ix > ZERO) && (fnList.gstr()[ix - 1] != L'\\') )
            isregexp = true ;
         else                       // second-tier characters
         {
            ix = fnList.find( L'[' ) ;
            if ( (ix == ZERO) || 
                 ((ix > ZERO) && (fnList.gstr()[ix - 1] != L'\\')) )
               isregexp = true ;
            else
            {
               ix = fnList.find( L'{' ) ;
               if ( (ix == ZERO) || 
                    ((ix > ZERO) && (fnList.gstr()[ix - 1] != L'\\')) )
                  isregexp = true ;
               else
               {
                  ix = fnList.find( L'+' ) ;
                  if ( (ix == ZERO) || 
                       ((ix > ZERO) && (fnList.gstr()[ix - 1] != L'\\')) )
                     isregexp = true ;
               }
            }
         }
      }
   }
   return isregexp ;

}  //* End gfIsRegexpPattern() *

//***************************
//* gfValidateSearchPattern *
//***************************
//******************************************************************************
//* Validate the syntax of the regexp search pattern. If an error is found,    *
//* repair it if possible.                                                     *
//*  1) Verify that we have a non-empty string.                                *
//*  2) Be sure pattern is delimited. This is done as a convenience in creating*
//*     the grep command, but if the pattern contains spaces, then delimiters  *
//*     become necessary.                                                      *
//*  3) If pattern contains delimiter character(s) used as characters, escape  *
//*     them.                                                                  *
//*                                                                            *
//*  This method is not user-proof. User may _intend_ to delimit but do it     *
//*  wrong, OR may _intend_ a search pattern which includes a delimiter        *
//*  character but forget to escape it. In short, the user is an               *
//*  inexhaustible fountain of bone-headed, fumble-fingered nonsense; and      *
//*  we cannot compensate for all possible foolishness.                        *
//*                                                                            *
//*                                                                            *
//* Input  : gsPat   : (by reference) search pattern to be validated           *
//*                                                                            *
//* Returns: 'true'  if pattern validated                                      *
//*          'false' if syntax or other error                                  *
//******************************************************************************

bool FileDlg::gfValidateSearchPattern ( gString& gsPat )
{
   bool status = true ;             // return value

   if ( gsPat.gschars() > 1 )       // if a non-empty string
   {
      //* If first character is a delimiter, check for        *
      //* matching closing delimiter and supply it if missing.*
      wchar_t delimChar = gsPat.gstr()[ZERO] ;
      if ( (delimChar == SQUOTE) || (delimChar == DQUOTE) )
      {
         if ( (gsPat.gstr()[gsPat.gschars() - 2]) != delimChar )
            gsPat.append( delimChar ) ;
      }

      //* Else, no leading delimiter. Assume no trailing delimiter.*
      else
         ;

      //* Delimit the search pattern. We do this because it is assumed that *
      //* the user is a novice grepper, and that if, after 20 years, we are *
      //* still screwing up occasionally, then the novice surely will.      *
      this->gfDelimitText ( gsPat, true ) ;
   }
   else           // no search pattern
      status = false ;

   return status ;

}  //* End gfValidateSearchPattern() *

//*************************
//*     gfDelimitText     *
//*************************
//******************************************************************************
//* Scan the text for space characters and delimiter-characters-as-characters. *
//* Delimit the text as specified or necessary. Please see logic table below.  *
//*                                                                            *
//* Input  : gsSrctxt   : (by reference) source text to be scanned             *
//*          forceDelim : (optional, 'false' by default)                       *
//*                       if 'true', always delimit the string even if it does *
//*                                  contain spaces                            *
//*                                                                            *
//* Returns: 'true'  if text modified, else 'false'                            *
//******************************************************************************
//* Logic Table (numeric sequence)                                             *
//* ==============================                                             *
//* DELIM   SPACES   EMBED   FORCE   COMMENT                                   *
//* -----   ------   -----   -----  ------------------------------------------ *
//*                                 not modified                               *
//*                            X    delimit                                    *
//*                   X             escape embeds                              *
//*                   X        X    delimit, escape embeds                     *
//*           X                     delimit                                    *
//*           X                X    delimit                                    *
//*           X       X             delimit, escape embeds                     *
//*           X       X        X    delimit, escape embeds                     *
//* - - - - - - - - - - - - - - - -                                            *
//*   X                             not modified                               *
//*   X                        X    not modified                               *
//*   X                X            escape embeds                              *
//*   X                X       X    escape embeds                              *
//*   X       X                     not modified                               *
//*   X       X                X    not modified                               *
//*   X       X        X            escape embeds                              *
//*   X       X        X       X    escape embeds                              *
//*                                                                            *
//******************************************************************************

bool FileDlg::gfDelimitText ( gString& gsSrctxt, bool forceDelim )
{
   bool txtModified = false ;

   //* If caller sent us a non-empty string.*
   if ( (gsSrctxt.gschars()) > 1 )
   {
      wchar_t firstChar = gsSrctxt.gstr()[ZERO],
              lastChar  = gsSrctxt.gstr()[gsSrctxt.gschars() - 2] ;
      short indx ;

      //* Test whether the string is already delimited.*
      if ( (firstChar != lastChar) ||
           ((firstChar != SQUOTE) && (firstChar != DQUOTE)) )
      {
         //* String is not delimited. Scan for spaces and          *
         //* delimiters-used-as-characters. If either found, then  *
         //* delimit the string and escape any embedded delimiter  *
         //* characters of the same type as the actual delimiters. *
         //* Note that if there are no spaces, delimiting is not   *
         //* technically necessary, but does no harm.              *
         if ( forceDelim != false ||
              ((gsSrctxt.find( nckSPACE )) >= ZERO) ||
              ((gsSrctxt.find( SQUOTE )) >= ZERO) ||
              ((gsSrctxt.find( DQUOTE )) >= ZERO) )
         {
            if ( (gsSrctxt.find( SQUOTE )) < ZERO )
            {
               gsSrctxt.insert( SQUOTE ) ;
               gsSrctxt.append( SQUOTE ) ;
               txtModified = true ;
            }
            else if ( (gsSrctxt.find( DQUOTE )) < ZERO )
            {
               gsSrctxt.insert( DQUOTE ) ;
               gsSrctxt.append( DQUOTE ) ;
               txtModified = true ;
            }
            //* String contains both single- AND double-quotes, but as         *
            //* characters, not as delimiters, and here's where it gets tricky:*
            //* We need to escape all the delimiter characters of one kind in  *
            //* the pattern, and then enclose the pattern in delimiters of that*
            //* kind.                                                          *
            //* We arbitrarily choose to escape the single-quote characters on *
            //* the supposition that there will be fewer of them i.e.          *
            //* double-quotes are likely to come in pairs.                     *
            else
            {
               indx = 1 ;
               while ( (indx = gsSrctxt.find( SQUOTE, indx )) > ZERO )
               {
                  //* If not already escaped *
                  if ( (gsSrctxt.gstr()[indx - 1]) != BKSTROKE )
                  {
                     gsSrctxt.insert ( BKSTROKE, indx ) ;
                     txtModified = true ;
                     indx += 2 ;
                  }
                  else
                     ++indx ;
               }
               gsSrctxt.insert( SQUOTE ) ;
               gsSrctxt.append( SQUOTE ) ;
            }
         }
      }
   
      //* String is already delimited, verify that any embedded *
      //* delimiter characters have been escaped.               *
      else
      {
         indx = 1 ;
         while ( (indx = gsSrctxt.find( firstChar, indx )) > ZERO )
         {
            //* If not yet at closing delimiter *
            if ( indx < (gsSrctxt.gschars() - 2) )
            {
               //* If not already escaped *
               if ( (gsSrctxt.gstr()[indx - 1]) != BKSTROKE )
               {
                  gsSrctxt.insert ( BKSTROKE, indx ) ;
                  txtModified = true ;
                  indx += 2 ;
               }
               else
                  ++indx ;
            }
            else     // closing delimiter found
               break ;
         }
      }
   }

   return txtModified ;

}  //* End gfDelimitText() *

//*************************
//*                       *
//*************************
//******************************************************************************
//*                                                                            *
//*                                                                            *
//*                                                                            *
//* Input  :                                                                   *
//*                                                                            *
//* Returns:                                                                   *
//******************************************************************************

