//********************************************************************************
//* File       : FileDlgPaste.cpp                                                *
//* Author     : Mahlon R. Smith                                                 *
//*              Copyright (c) 2006-2025 Mahlon R. Smith, The Software Samurai   *
//*                  GNU GPL copyright notice located in NcDialog.hpp            *
//* Date       : 12-May-2025                                                     *
//* Version    : (see FileDlgVersion string in FileDlg.cpp)                      *
//*                                                                              *
//* Description:                                                                 *
//* This module contains methods related to the file paste-from-clipboard        *
//* operation of the FileDlg class.                                              *
//*                                                                              *
//*                                                                              *
//* Developed using GNU G++ (Gcc v: 4.4.2)                                       *
//*  under Fedora Release 12, Kernel Linux 2.6.31.5-127.fc12.i686.PAE and above  *
//********************************************************************************
//* Version History (most recent first):                                         *
//*   See version history in FileDlg.cpp.                                        *
//********************************************************************************
//* Programmer's Notes:                                                          *
//* Be aware that it is perfectly acceptable to COPY the files in the            *
//* clipboard list to a target directory which is a node or sub-node on the      *
//* clipboard list. An endless loop will not occur because the copied data are   *
//* not part of the clipboard list.                                              *
//* But of course it is not possible to successfully MOVE data to a              *
//* subdirectory that is a node or sub-node of the clipboard list because        *
//* deletion of one or more parent directories will fail (file-in-use).          *
//*                                                                              *
//********************************************************************************

#include "GlobalDef.hpp"
#include "FileDlg.hpp"

//******************************
//* Local definitions and data *
//******************************

#define enableDBBAR (0)    // set to non-zero to enable debug of progress bar

//* Message text defined in FileDlgPrompt.cpp *
extern const char* fmtypeString[] ;
extern const char* fmtypeName[] ;
extern const char* readonlyFileSys[] ;
extern const char* notOntoSelf[] ;
extern const char* cannotOverrideWP[] ;
extern const char* noLinkOverwrite[] ;
extern const char* differentFileTypes[] ;
extern const char* cannotDeleteSpecial[] ;
extern const char* cannotOverwriteFifo[] ;
extern const char* cannotOverwriteDirectory[] ;
extern const char* warnSocketMod[] ;
extern const char* warnBdevCdevMod[] ;
extern const char* warnUnsuppMod[] ;
extern const char* fdNotImplemented[] ;

//* Flow-control flags for the PasteClipboardList() *
//* and the PasteSpecialList() method groups        *
static slcOptions cbSymLinkOption ; //* Symbolic-link copy option
            //* Confirmation required for overwrite of all existing targets    *
static bool cbConfirmAllOverwrites, 
            //* Silently overwrite any remaining targets in the list whose     *
            //* mod dates are newer than source file.                          *
            cbOverwriteNewerTrgAllowed,
            //* Silently skip overwrite of any remaining targets in the list   *
            //* whose mod dates are newer than source file.                    *
            cbSkipAllOverwriteNewerTrg,
            //* Silently override write protection for any remaining           *
            //* write-protected targets in the list.                           *
            cbOverwriteProtectedTrgAllowed,
            //* Silently skip overwrite of any remaining write-protected       *
            //* targets in the list.                                           *
            cbSkipAllWriteProtectedTrg,
            //* Silently override write protection for any remaining write-    *
            //* protected source files in the list. (for delete-after-copy).   *
            cbOverrideProtectedSrcAllowed,
            //* For delete-after-copy (cut-and-paste) operations, silently     *
            //* skip processing for any remaining write-protected source files *
            //* in the list.                                                   *
            cbSkipAllWriteProtectedSrc ; 

#if ENABLE_DEBUGGING_CODE != 0
#define enableDBWIN (0)          // set to non-zero to enable debugging window
#if enableDBWIN != 0
// NOTE: Testing must be done while running in Single-Window mode with terminal 
//       size >= 132x40 to avoid unsightly dialog overlap.
static NcDialog*  dbWin = NULL ;          // pointer to debugging window
static short      dbLines = ZERO,         // dimensions of debugging window
                  dbCols  = ZERO ;
static attr_t     dbColor = ZERO ;        // color attribute of debugging window
static winPos     dbwp( 1, 1 ) ;          // cursor position in debugging window
static short      dbPOffset = 60 ;        // path-string offset (for beauty)
#endif   // enableDBWIN
#endif   // ENABLE_DEBUGGING_CODE


//******************************************************************************
//** Methods related to PasteClipboardList(), i.e. a 'normal' Paste operation **
//******************************************************************************

//***********************
//* PasteClipboardList  *
//***********************
//******************************************************************************
//* Copy all files in the clipboard tree list to target with top level of tree *
//* written to the currently-displayed directory.                              *
//*                                                                            *
//* If a cut-and-paste operation, delete source files after successful copy.   *
//*                                                                            *
//*                                                                            *
//* Input  : attCount (by reference, initial value ignored)                    *
//*           on return, contains number of copy/move operations attempted,    *
//*           (usually) the number of files on the clipboard at the time of    *
//*           the call                                                         *
//*          delCount (by reference, initial value ignored)                    *
//*           if a delete-after-copy operation is in progress, on return,      *
//*           contains number of source files successfully deleted after       *
//*           being successfully copied.                                       *
//*           If not a delete-after-copy operation, ZERO is returned.          *
//* Returns: number of files successfully copied or moved to target            *
//*            if clipboardFiles != return value, then one or more errors      *
//*            occurred or user aborted the operation                          *
//******************************************************************************
//* Programmer's Note:                                                         *
//*                                                                            *
//******************************************************************************

UINT FileDlg::PasteClipboardList ( UINT& attCount, UINT& delCount  )
{
   //* Total number of files to copy or move *
   attCount = clipBoard[BNODE].totFiles ;
   //* If delete-after-copy, total files deleted after paste (else, ignored) *
   delCount = ZERO ;
   //* Return value: number of files successfully written *
   UINT pasteCount = ZERO ;

   this->SetErrorCode ( ecNOERROR ) ;  // clear any previous error code

   //* Initialize the confirmation flags.                                      *
   //* (Any flag may be interactively set by user during confirmation prompt.) *
   //* Set the rule for copying symbolic link files.                           *
   cbSymLinkOption = slcUSER_PROMPT ;
   //* If configuration indicates that all target overwrites should be         *
   //* confirmed, set 'true', else 'false'.                                    *
   cbConfirmAllOverwrites = 
      (bool)(this->cfgOptions.overWrite == owCONFIRM_ALL ? true : false) ;
   //* If configuration indicates that silent overwrite of newer targets with  *
   //* older sourcefiles is Ok.                                                *
   cbOverwriteNewerTrgAllowed = 
      (bool)(this->cfgOptions.overWrite == owNO_CONFIRM ? true : false) ;
   //* Always confirm overwrite of write-protected target files AND            *
   //* overwrite of targets that are newer than the source AND                 *
   //* delete-after-copy of write-protected source files until user specifies  *
   //* to continue without additional confirmations.                           *
   cbOverwriteProtectedTrgAllowed = 
   cbSkipAllWriteProtectedTrg     = 
   cbSkipAllOverwriteNewerTrg     = 
   cbSkipAllWriteProtectedSrc     = false ;
   //* Is this a copy-and-paste or a cut-and-paste operation?                  *
   bool  cutPending = this->CutPending () ;

   //* We need read permisison to copy source to target, and if this is a      *
   //* cut-and-paste operation, we also need write permission on the source.   *
   UINT  wpCount = ZERO,   // number of write-protected files
         rpCount = ZERO ;  // number of read-protected files
   cbOverrideProtectedSrcAllowed = 
      this->SourcePermission ( wpCount, rpCount, (cutPending ? true : false), true ) ;

   //* Determine whether target directory == source directory.                 *
   bool tDirEQsDir = this->TargDirEqualsClipDir ( false ) ;

   //* Determine whether user has read/write permission on target directory.   *
   bool trgdirPerm = this->CurrDirPermission () ;

   //* Test for available space on target file system *
   bool spaceAvailable = this->CheckTargetSpace () ;

   //* The Paste-Clipboard-files operation requires that                       *
   //* 1) at least one file is on the clipboard,                               *
   //* 2) that user has read/write permission for the target directory         *
   //* 3) that user has read permission for all source files OR user's         *
   //*    permission to proceed.                                               *
   //* 4) If this is a cut-and-paste operation, user must also have write      *
   //*    permission for all source files OR user's permission to proceed.     *
   //* 5) there is either plenty of free space on target filesystem OR we have *
   //*    user's permission to try the paste anyway.                           *
   //* 6) If target directory == source directory, user will be prompted for   *
   //*    paste-with-rename for all top-level files and directories.           *
   if ( attCount > ZERO && trgdirPerm != false &&
        spaceAvailable != false && cbOverrideProtectedSrcAllowed != false )
   {
      //* Prepare a pbarInit-class object in case a progress bar is needed.*
      pbarInit pbi = this->ProgbarInit ( this->mMax, this->mulY, this->mulX, true ) ;

      //* Initialize tracking for work-in-progress *
      this->wip.initialize( clipBoard[BNODE].totFiles, 
                            clipBoard[BNODE].totBytes, PB_THRESHOLD ) ;

      #if ENABLE_DEBUGGING_CODE != 0 && enableDBBAR != 0
      this->wip.initialize( clipBoard[BNODE].totFiles, 
                            clipBoard[BNODE].totBytes, ZERO ) ;
      #endif   // ENABLE_DEBUGGING_CODE && enableDBBAR

      //* Is target an MTP/GVfs filesystem? *
      clipBoard_gvfsTrg = this->fmPtr->isGvfsPath () ;

      //* If source data >= threshold, OR if source or target is an MTP/GVfs  *
      //* filesystem, launch a thread to manage the visual progress indicator.*
      thread *progbarThread = NULL ;// Progbar thread pointer
      bool pbtAllocated = false,    // 'true' if Progbar thread is allocated
           pbtLaunched = false ;    // 'true' if Progbar thread is launched
      if ( (clipBoard_gvfsTrg != false) ||
           (clipBoard[BNODE].totBytes >= (this->wip.getThreshold())) )
      {
         chrono::duration<short, std::milli>aMoment( 10 ) ;
         for ( short i = 3 ; i > ZERO ; --i ) // three strikes, and you're out!
         {
            if ( ! pbtAllocated )
            {
               if ( (progbarThread = new (std::nothrow) std::thread) != NULL )
                  pbtAllocated = true ;
            }
            if ( pbtAllocated )
            {
               try
               {
                  *progbarThread = thread( &FileDlg::pbManager, this, &pbi, &pbtLaunched ) ;
                  this_thread::sleep_for( aMoment ) ;  // give the new thread time to respond
                  if ( pbtLaunched != false )
                     break ;
               }
               catch ( ... )        // monitor thread created but not launched
               {
                  if ( i == 1 )
                  {  // give up on launching the thread (should never happen)
                     delete progbarThread ;
                     progbarThread = NULL ;
                     pbtAllocated = pbtLaunched = false ;
                     this->ProcessCWD () ;   // display a simple message
                     break ;
                  }
                  this_thread::sleep_for( aMoment ) ;  // wait a moment and try again
               }
            }
         }
      }
      //* If source data < threshold, display a simple message.*
      else
         this->ProcessCWD () ;

      //*******************************
      //* Copy source files to target *
      //*******************************
      #if ENABLE_DEBUGGING_CODE != 0 && enableDBWIN != 0
      if ( dbWin == NULL )
         dbWin = this->Open_DBwindow ( dbColor, dbLines, dbCols ) ;
      dbwp = { 1, 1 } ;
      #endif   // ENABLE_DEBUGGING_CODE && enableDBWIN

      //* Data passed among lower-level methods *
      pclParm  pcl( clipBoard_srcPath, this->currDir, NULL, // top-level accumulator
                    psstSimpleCopy, cutPending ) ;
      pclParm pclr( clipBoard_srcPath, this->currDir, // for recursive operations
                    &clipBoard[BNODE].nextLevel[ZERO], 
                    psstSimpleCopy, cutPending ) ;

      for ( UINT nodeIndex = ZERO ; nodeIndex < clipBoard_dirTrees ; nodeIndex++ )
      {
         pclr.srcTree = &clipBoard[BNODE].nextLevel[nodeIndex] ;
         pclr.attCount = pclr.cpyCount = pclr.delCount = ZERO ;
         pasteCount += this->pclCopyTnList ( pclr, tDirEQsDir ) ;
         pcl.attCount += pclr.attCount ;
         pcl.cpyCount += pclr.cpyCount ;
         pcl.delCount += pclr.delCount ;
         if ( pclr.psOption == psstABORT )
         {
            pcl.psOption = psstABORT ;
            break ;
         }
      }

      //* If non-directory files at this (top) level, copy/move them *
      if ( (pcl.psOption != psstABORT) && (clipBoard[BNODE].tnFCount > ZERO) 
           && (clipBoard[BNODE].tnFiles != NULL) )
      {
         pcl.srcTree = &clipBoard[BNODE] ;
         pasteCount += this->pclCopyFileList ( pcl, tDirEQsDir ) ;
      }

      //* Values returned in caller's parameters: *
      attCount = pcl.attCount ;  // copy operations attempted
      delCount = pcl.delCount ;  // for delete-after-copy, successful deletions

      #if ENABLE_DEBUGGING_CODE != 0 && enableDBWIN != 0
      if ( dbWin != NULL )
      {
         gString gs ;
         gs.compose( L"nodeFiles:%02u pasteCount:%02u", 
                     &clipBoard[BNODE].totFiles, &pasteCount ) ;
         dbWin->ClearLine ( dbLines - 3, false ) ;
         dbWin->WriteString ( (dbLines-3), 2, gs, dbColor ) ;
         gs.compose( L"attCount :%02u delCount  :%02u", 
                     &pcl.attCount, &pcl.delCount ) ;
         dbWin->ClearLine ( dbLines - 2, false ) ;
         dbWin->WriteString ( (dbLines-2), 2, gs, dbColor ) ;
         dbWin->WriteString ( (dbLines-2), (dbCols-14), 
                              " Press A Key ", this->cs.pf, true ) ;
         nckPause();
         delete dbWin ;
         dbWin = NULL ;
      }
      #endif   // ENABLE_DEBUGGING_CODE && enableDBWIN

      //* If this was a delete-after-copy operation, *
      //* clipboard contents are now invalid.        *
      if ( cutPending )
         this->ClearClipboard () ;

      if ( pbtAllocated != false )  // if Progbar thread was allocated
      {
         if ( pbtLaunched )         // if Progbar thread was launched
         {
            this->wip.setAbort() ;  // signal thread to return (if it hasn't already)
            progbarThread->join() ; // wait for thread to terminate
         }
         delete progbarThread ;     // delete the secondary thread
      }

      //* Refresh the file list for the target directory *
      //* and display the results of the operation.      *
      if ( pasteCount > ZERO || tDirEQsDir )
         this->RefreshCurrDir () ;
      //* Else, just restore path and stats messages.    *
      else
         this->DisplayStatus () ;
   }

   //* User lacks read and/or write access to target directory *
   else if ( trgdirPerm == false )
   {
      this->DisplayStatus ( ecREADONLY_T, EACCES ) ;
   }

   return pasteCount ;

}  //* End PasteClipboardList() *

//***********************
//*    pclCopyTnList    *
//***********************
//******************************************************************************
//* Copy all files and directories in a TreeNode list to target directory.     *
//*                                                                            *
//* Caller has verified that user has write permission on the target directory.*
//*                                                                            *
//*                                                                            *
//* Input  : pclParm structure (by reference)                                  *
//*            pcl.srcDir  : source directory path                              *
//*            pcl.trgDir  : target directory path                               *
//*            pcl.srcTree : pointer to TreeNode object describing subdir tree  *
//*            pcl.psOption: always psstSimpleCopy                             *
//*            pcl.attCount: (initial value ignored)                           *
//*                          on return, contains number of paste operations    *
//*                          attempted                                         *
//*            pcl.delCount: (initial value ignored) set to ZERO; however if   *
//*                          pcl.deleteSrc != false, then on return, contains  *
//*                          number of source files successfully deleted       *
//*                          after copy                                        *
//*            pcl.deleteSrc: 'true' if delete-after-copy operation            *
//*          tDirEQsDir: (optional, 'false' by default)                        *
//*                      if 'true' target base directory == source base        *
//*                      directory, so prompt user for copy-with-rename        *
//*                                                                            *
//* Returns: number of files copied                                            *
//*           (if operation was aborted, pcl.psOption set to psstABORT)        *
//******************************************************************************

UINT FileDlg::pclCopyTnList ( pclParm& pcl, bool tDirEQsDir )
{
   #if ENABLE_DEBUGGING_CODE != 0 && enableDBWIN != 0
   const bool  inclTN = false ;
   NcDialog* tnWin = NULL ;
   if ( dbWin != NULL )
   {
      const short tnWinLines = 19 ;
      dbWin->ClearWin () ;
      dbwp = { 1, 1 } ;
      if ( pcl.srcTree != NULL && inclTN )
      {
         dbwp.xpos = 79 ;
         tnWin = this->Display_TreeNode ( dbwp, pcl.srcTree, true ) ;
         dbwp = { short(dbwp.ypos + tnWinLines), 1 } ;
         dbWin->WriteString ( dbwp.ypos++, dbwp.xpos, 
                              "pclCopyTnList Source Directory", dbColor ) ;
         dbWin->WriteString ( dbwp.ypos++, dbwp.xpos, 
                              &pcl.srcDir.gstr()[dbPOffset], dbColor ) ;
         dbWin->WriteString ( dbwp.ypos++, dbwp.xpos, 
                              "pclCopyTnList Target Directory", dbColor ) ;
         dbWin->WriteString ( dbwp.ypos++, dbwp.xpos, 
                              &pcl.trgDir.gstr()[dbPOffset], dbColor ) ;
      }
   }
   #endif   // ENABLE_DEBUGGING_CODE && enableDBWIN

   UINT  pasteCount = ZERO ;        // return value: number of files copied

   //* Target directory name - modified by user if target dir == source dir    *
   tnFName* srcStats = &pcl.srcTree->dirStat ;
   gString trgFName( srcStats->fName ) ;

   //* If target directory path == source directory path, user is attempting   *
   //* to copy a file (files) onto itself. Prompt for new target-file name.    *
   if ( tDirEQsDir )
   {
      if ( (this->pclGetNewFilename ( srcStats, trgFName, tDirEQsDir )) != corProcessOnce )
         pcl.psOption = psstABORT ;
   }

   //* Copy/Move contents of TreeNode list *
   if ( pcl.psOption != psstABORT )
   {
      //* Path/filename string for source file *
      gString srcPath ;
      this->fmPtr->CatPathFname ( srcPath, pcl.srcDir.ustr(), srcStats->fName ) ;
      
      //* Path/filename string for target file *
      gString trgPath ;
      this->fmPtr->CatPathFname ( trgPath, pcl.trgDir.gstr(), trgFName.gstr() ) ;

      //* Determine whether target already exists, is write protected, etc. *
      //* and prompt user for a decision if necessary.                      *
      corReturn coStatus = 
            this->pslTargetExists ( trgPath, srcPath, srcStats, pcl.psOption ) ;
      ++pcl.attCount ;              // operation attempted
      if ( coStatus == corAbort )   // if user aborted the operation
         pcl.psOption = psstABORT ;

      #if ENABLE_DEBUGGING_CODE != 0 && enableDBWIN != 0
      if ( dbWin != NULL )
      {
         dbwp = dbWin->WriteString ( dbwp.ypos, dbwp.xpos, L"src: ", dbColor ) ;
         dbWin->WriteString ( dbwp.ypos++, dbwp.xpos, 
                              &srcPath.gstr()[dbPOffset], dbColor ) ;
         dbwp.xpos = 1 ;
         dbwp = dbWin->WriteString ( dbwp.ypos, dbwp.xpos, L"trg: ", dbColor ) ;
         dbWin->WriteString ( dbwp.ypos++, dbwp.xpos, 
                              &trgPath.gstr()[dbPOffset], dbColor ) ;
         dbwp.xpos = 1 ;
         dbWin->WriteString ( (dbLines-2), (dbCols-14), 
                              " Press A Key ", this->cs.pf, true ) ;
         if ( inclTN && tnWin != NULL )
            tnWin->RefreshWin () ;
         nckPause();
         if ( inclTN && tnWin != NULL )
            delete tnWin ;
         dbWin->WriteString ( (dbLines-2), (dbCols-14), 
                              "             ", dbColor, true ) ;
         if ( pcl.srcTree != NULL && inclTN )
         {
            dbWin->ClearWin () ;
            dbwp = { 1, 1 } ;
         }
      }
      #endif   // ENABLE_DEBUGGING_CODE && enableDBWIN

      //* If target does not already exist OR if we have user's *
      //* permission to overwrite existing target, then proceed.*
      if ( (coStatus == corProcessOnce || coStatus == corProcessAll) &&
           ((this->CopyFile ( *srcStats, srcPath, trgPath )) == OK) )
      {
         ++pasteCount ;       // target directory created

         //* If source subdirectory contains additional *
         //* files or subdirectories, copy/move them.   *
         UINT nodeFiles = 1 ; // for validating successful copy-before-delete
         UINT64 nodeBytes = ZERO ;
         if ( pcl.srcTree->totFiles > ZERO )
         {
            if ( pcl.deleteSrc != false )
            {
               nodeFiles = ZERO ;
               this->fmPtr->TreeNodeSummary ( pcl.srcTree, nodeFiles, nodeBytes ) ;
            }
            pclParm pcl2( srcPath.ustr(), trgPath.ustr(), pcl.srcTree, 
                          pcl.psOption, pcl.deleteSrc ) ;
            pasteCount += this->pslCopyTnList ( pcl2 ) ;
            pcl.attCount += pcl2.attCount ;
            pcl.cpyCount += pcl2.cpyCount ;
            pcl.delCount += pcl2.delCount ;
         }

         //* If delete-after-paste operation AND all files in node  *
         //* copied/deleted successfully, delete the directory name.*
         if ( pcl.deleteSrc && 
              (pasteCount == nodeFiles) && (pcl.delCount == (nodeFiles-1)) )
         {
            if ( (this->DeleteSourceFile ( srcStats, srcPath )) == OK )
               ++pcl.delCount ;
         }
      }
   }
   return pasteCount ;

}  //* End pclCopyTnList() *

//***********************
//*   pclCopyFileList   *
//***********************
//******************************************************************************
//* Copy the list of non-directory files target directory.                     *
//*                                                                            *
//* Caller has verified that user has write permission on the target directory.*
//*                                                                            *
//* Input  : pclParm structure (by reference)                                  *
//*            pcl.srcDir : source directory path                              *
//*            pcl.trgDir: target directory path                               *
//*            pcl.srcTree->tnFiles : array of tnFName objects for source files*
//*            pcl.srcTree->thFCount: number of items in array                 *
//*            pcl.delCount:                                                   *
//*            - if delete-after-copy operation, then number of source files   *
//*              deleted is added to delCount                                  *
//*            - else value returns unchanged                                  *
//*            pcl.deleteSrc: 'true' if delete-after-copy operation            *
//*          tDirEQsDir: (optional, 'false' by default)                        *
//*                      if 'true' target base directory == source base        *
//*                      directory, so prompt user for copy-with-rename        *
//*                                                                            *
//* Returns: number of files copied                                            *
//*           (if operation was aborted, pcl.psOption set to psstABORT)        *
//******************************************************************************

UINT FileDlg::pclCopyFileList ( pclParm& pcl, bool tDirEQsDir )
{
   UINT  pasteCount = ZERO ;     // number of files written (return value)
   renamePattern* rPat = NULL ;  // for batch rename option
   bool  rPrompt = false ;       // for individual rename option

   //* If target directory path == source directory path, user is attempting   *
   //* to copy a file (files) onto itself. Prompt for target-file rename.      *
   if ( tDirEQsDir )
   {
      if ( clipBoard[BNODE].tnFCount > 1 )
      {
         const char* renMsg = 
         "You have specified that files are to be copied into source directory.\n"
         "To do this, target filenames must be different from source names." ;
         rPat = new renamePattern ;
         this->fmPtr->strnCpy ( rPat->inStr, L"(2)", 4 ) ;  // 'suggested' target name
         rPat->in = true ;
         renameOption rOpt = this->rsfGetRenamePattern ( rPat, true, renMsg ) ;
         switch ( rOpt )
         {
            case promptRENAME:
               rPrompt = true ;  // indicate prompt for each file
               delete rPat ;     // we don't need the pattern generator
               rPat = NULL ;
               break ;
            case batchRENAME:    // batch-rename pattern generator initialized
               break ;
            default:             // abort the operation 
               pcl.psOption = psstABORT ;
               break ;
         }
      }
      else        // only one file in list
      {
         rPrompt = true ;
      }
   }

   //* Copy the files *
   if ( pcl.psOption != psstABORT )
      pasteCount = this->pslCopyFileList ( pcl, rPrompt, tDirEQsDir, rPat ) ;

   //* Release dynamic allocation *
   if ( rPat != NULL )
      delete rPat ;

   return pasteCount ;

}  //* End pclCopyFileList() *

//***************************
//*    pclGetNewFilename    *
//***************************
//******************************************************************************
//* Prompt user for a new target filename for the paste operation.             *
//*                                                                            *
//* At all times, a file may not be copied onto itself, but in addition, if    *
//* target directory == source directory, new target filename must not be in   *
//* use by any top-level file/directory in the clipboard list.                 *
//*                                                                            *
//*                                                                            *
//* Input  : srcStats : pointer to tnFName object describing the source file   *
//*          trgFName ; (by reference, initial contents ignored)               *
//*                     if user enters a valid filename, i.e. corProcessOnce   *
//*                     is returned, new filename is written here,             *
//*                     else contents are unchanged.                           *
//*          tDirEQsDir: if 'true' target directory == source directory        *
//*                     (that is, the file being named exists in the top-level)*
//*                     (directory of the clipboard list.                     )*
//*                                                                            *
//* Returns: member of corReturn:                                              *
//*            corProcessOnce: valid filename entered by user                  *
//*            corAbort      : user has aborted the Paste operation in progress*
//******************************************************************************
//* Programmer's Note: This method is called only for source filenames and     *
//* directory names living in the main clipboard directory, clipBoard_srcPath. *
//* Files and directories below the top level are copied using the sourcefile  *
//* name.                                                                      *
//******************************************************************************

corReturn FileDlg::pclGetNewFilename ( const tnFName* srcStats, 
                                       gString& trgFName, bool tDirEQsDir )
{
   const char* renMsg1 = 
   //  123456789x123456789x123456789x123456789x123456789xZ
      "A file cannot be copied onto itself. Please enter\n"
      "  a target filename different from source name.  " ;
   const char* renMsg2 = 
      "Please enter a target filename." ;

   corReturn corOption = corAbort ; // return value

   //* Create a 'suggested filename' of the form:  'filename(2).ext' *
   gString tmpFName( srcStats->fName ), suggestedFName ;
   if ( tDirEQsDir )
      this->pclCreateSuggestedName ( suggestedFName, tmpFName ) ;

   bool  status ;          // status of user interaction (always distasteful, but necessary)
   bool  done = false ;    // loop control
   do
   {
      tmpFName = srcStats->fName ;  // source filename
      if ( tDirEQsDir )
         status = this->GetNewFilename ( tmpFName, renMsg1, &suggestedFName ) ;
      else
         status = this->GetNewFilename ( tmpFName, renMsg2 ) ;
      if ( status != false )
      {  //* User has entered a valid filename. Now, if target directory ==    *
         //* source directory, verify that the new target name is not in use   *
         //* by any member of the clipboard list at this level including the   *
         //* source file itself.                                               *
         if ( tDirEQsDir )
         {
            bool tryAgain = true ;
            if ( (this->pclFNameInUse ( tmpFName, tryAgain )) == false )
            {
               trgFName  = tmpFName ;
               corOption = corProcessOnce ;  // alert caller to the good news
               done = true ;
            }
            else if ( tryAgain == false )
            {
               done = true ;
            }
         }
         else
         {
            trgFName  = tmpFName ;
            corOption = corProcessOnce ;  // alert caller to the good news
            done = true ;
         }
      }  // if(valid)
      else
         done = true ;
   }
   while ( ! done ) ;
   return corOption ;

}  //* End pclGetNewFilename() *

//***********************
//*   pclFNameInUse     *
//***********************
//******************************************************************************
//* Scan the top-level clipboard node for a filename or directory name which   *
//* matches the filename provided.                                             *
//* This is done on a paste-with-rename to avoid overwriting a file in the     *
//* target directory which is ALSO a file on the clipboard. Failure to do this *
//* test could result in a file that is pasted from the clipboard being a      *
//* different file from the one the user originally selected.                  *
//*                                                                            *
//* Input  : trgName  : filename to compare                                    *
//*          prompt   : (by reference) if 'true' AND filename matches a file   *
//*                     on the clipboard, then ask user whether he/she/it      *
//*                     wants to enter a different name                        *
//*                                                                            *
//* Returns: 'true' if matching name found, else 'false'                       *
//*                   if 'prompt' was true on entry, AND if matching file was  *
//*                   found, then user's response is returned in prompt        *
//*                   else 'prompt' value is unchanged                         *
//******************************************************************************

bool FileDlg::pclFNameInUse ( const gString& trgName, bool& prompt )
{
   bool targetOnCB = false ;
   
   for ( UINT i = ZERO ; i < clipBoard_dirTrees && !targetOnCB ; i++ )
   {
      if ( (strncmp ( trgName.ustr(), 
               clipBoard[BNODE].nextLevel[i].dirStat.fName, 
               MAX_FNAME)) == ZERO )
         targetOnCB = true ;
            
   }
   for ( UINT i = ZERO ; i < clipBoard[BNODE].tnFCount && !targetOnCB ; i++ )
   {
      if ( (strncmp ( trgName.ustr(),
               clipBoard[BNODE].tnFiles[i].fName, MAX_FNAME )) == ZERO )
         targetOnCB = true ;
   }

   if ( targetOnCB && prompt != false )
   {
      const char* otraVez[] = 
      {
         " Conflicting Filename ",
         "The provided target name is in use by another file.",
         trgName.ustr(),
         " ",
         "      Do you want to enter a different name?",
         NULL 
      } ;
      attr_t msgColor[INFODLG_MAX_MSGS] = 
      {
         (this->cs.sd | ncbATTR), this->cs.sd, (this->cs.sd & ~ncrATTR), this->cs.sd, 
         this->cs.sd, this->cs.sd, this->cs.sd, this->cs.sd
      } ;
      prompt = this->DecisionDialog ( otraVez, (const attr_t*)&msgColor ) ;
   }
   return targetOnCB ;

}  //* End pclFNameInUse() *


//******************************************************************************
//** Methods related to PasteSpecialList(), i.e. a 'Paste-Special' operation  **
//******************************************************************************

//***********************
//*  PasteSpecialList   *
//***********************
//******************************************************************************
//* Paste all files on the clipboard in one of the 'special' ways.             *
//*                                                                            *
//* A 'Paste-Special' operation can be any one of the following:               *
//*   [For all of these operations, target file can optionally be renamed.]    *
//*   - Create a symbolic link (shortcut) to an existing non-sym-link file     *
//*   - Create a 'hard' link to an existing non-sym-link, non-directory file   *
//*   - Copy/move a symbolic link file                                         *
//*   - Copy a sym-link file's target file                                     *
//*     If the source file is a symbolic link to a directory, the sym-link     *
//*     target directory and all its contents are copied to target directory.  *
//*   - Copy/move source file(s) with individual rename                        *
//*     Only the top-level target files will be renamed. If a top-level file   *
//*     is a subdirectory name, the CONTENTS of that subdirectory will be      *
//*     copied/moved without rename.                                           *
//*   - Copy/move source files with batch rename                               *
//*     Only the top-level target files will be renamed. If a top-level file   *
//*     is a subdirectory name, the CONTENTS of that subdirectory will be      *
//*     copied/moved without rename.                                           *
//*   - Perform a 'standard' copy/move operation (in case user decides that    *
//*     the 'special' operations are not appropriate).                         *
//*                                                                            *
//* NOTE: There are separate operations for a single file or single directory  *
//*       tree AND for multiple files and/or directories. This is done to offer*
//*       more flexibility when operating on just one file or directory tree.  *
//*       With multiple files/directory trees, the options are more limited.   *
//*                                                                            *
//*                                                                            *
//* Input  : attCount (by reference, initial value ignored)                    *
//*           on return, contains number of copy/move operations attempted,    *
//*           (usually) the number of files on the clipboard at the time of    *
//*           the call                                                         *
//*          delCount (by reference, initial value ignored)                    *
//*           if a delete-after-copy operation is in progress, on return,      *
//*           contains number of source files successfully deleted after       *
//*           being successfully copied.                                       *
//*           If not a delete-after-copy operation, ZERO is returned.          *
//*                                                                            *
//* Returns: number of files successfully copied or moved to target            *
//*            if clipboardFiles != return value, then one or more errors      *
//*            occurred or user aborted the operation                          *
//******************************************************************************

UINT FileDlg::PasteSpecialList ( UINT& attCount, UINT& delCount  )
{
   //* Total number of files to copy or move *
   attCount = clipBoard[BNODE].totFiles ;
   //* If delete-after-copy, total files deleted after paste (else, ignored) *
   delCount = ZERO ;
   //* Return value: number of files successfully written *
   UINT pasteCount = ZERO ;

   this->SetErrorCode ( ecNOERROR ) ;  // clear any previous error code

   //* Initialize the confirmation flags.                                      *
   //* (Any flag may be interactively set by user during confirmation prompt.) *
   //* Set the rule for copying symbolic link files.                           *
   cbSymLinkOption = slcUSER_PROMPT ;
   //* If configuration indicates that all target overwrites should be         *
   //* confirmed, set 'true', else 'false'.                                    *
   cbConfirmAllOverwrites = 
      (bool)(this->cfgOptions.overWrite == owCONFIRM_ALL ? true : false) ;
   //* If configuration indicates that silent overwrite of newer targets with  *
   //* older sourcefiles is Ok.                                                *
   cbOverwriteNewerTrgAllowed = 
      (bool)(this->cfgOptions.overWrite == owNO_CONFIRM ? true : false) ;
   //* Always confirm overwrite of write-protected target files AND            *
   //* overwrite of targets that are newer than the source AND                 *
   //* delete-after-copy of write-protected source files until user specifies  *
   //* to continue without additional confirmations.                           *
   cbOverwriteProtectedTrgAllowed = 
   cbSkipAllWriteProtectedTrg     = 
   cbSkipAllOverwriteNewerTrg     = 
   cbSkipAllWriteProtectedSrc     = false ;
   //* Is this a copy-and-paste or a cut-and-paste operation?                  *
   bool  cutPending = this->CutPending () ;

   //* We need read permisison to copy source to target, and if this is a      *
   //* cut-and-paste operation, we also need write permission on the source.   *
   UINT  wpCount = ZERO,   // number of write-protected files
         rpCount = ZERO ;  // number of read-protected files
   cbOverrideProtectedSrcAllowed = 
      this->SourcePermission ( wpCount, rpCount, (cutPending ? true : false), true ) ;

   //* Determine whether target directory == source directory.                 *
   bool tDirEQsDir = this->TargDirEqualsClipDir ( false ) ;

   //* Determine whether source dir and target dir are on the same file system.*
   //* This info determines whether creation of 'hard' links is allowed.       *
   // Programmer's Note: There is a logical hole in this test wherein if both 
   // source and target are on LOGICAL (tmpfs, etc.) filesystems, there could be
   // a false match; however, logical filesystem are (almost always) read-only,
   // so this is unlikely.
   bool sameFilesys = true ;
   if ( tDirEQsDir == false )
   {
      gString srcPath( clipBoard_srcPath ),
              trgPath( this->currDir ),
              srcDev, trgDev ;
      this->fmPtr->FilesystemID ( srcPath, srcDev ) ;
      this->fmPtr->FilesystemID ( trgPath, trgDev ) ;
      if ( srcDev != trgDev )
         sameFilesys = false ;
   }

   //* Determine whether user has write permission on target directory *
   bool trgdirPerm = this->CurrDirPermission () ;

   //* Test for available space on target file system *
   bool spaceAvailable = this->CheckTargetSpace () ;

   //* The Paste-Special operation requires that                               *
   //* 1) at least one file is on the clipboard,                               *
   //* 2) that user has read/write permission for the target directory AND     *
   //*    that if the target directory is also the clipboard source directory, *
   //*    then certain operations are restricted.                              *
   //* 3) that user has read permission for all source files OR user's         *
   //*    permission to proceed.                                               *
   //* 4) If this is a cut-and-paste operation, user must also have write      *
   //*    permission for all source files OR user's permission to proceed.     *
   //* 5) there is either plenty of free space on target OR we have user's     *
   //*    permission to try the paste anyway.                                  *
   if ( attCount > ZERO && trgdirPerm != false && 
        spaceAvailable != false && cbOverrideProtectedSrcAllowed != false )
   {
      //*******************************
      //* Copy source files to target *
      //*******************************
      //* If single source file OR a single directory tree *
      if ( attCount == 1 || (clipBoard[BNODE].tnFCount == ZERO && clipBoard_dirTrees == 1) )
      {  //* The file is at the top level of the directory tree, and   *
         //* may be either an ordinary file OR a single directory tree.*
         //* Determine which flavor of 'specialness' the user has      *
         //* in mind, and perform the copy/move operation.             *
         tnFName* srcStats = NULL ; // pointer to source-file stats on clipboard
         //* Data passed among lower-level methods *
         pclParm  pcl( clipBoard_srcPath, this->currDir ) ;

         if (   (clipBoard[BNODE].tnFCount > ZERO) 
             && (clipBoard[BNODE].tnFiles != NULL) )
         {  //* Stats of non-directory file *
            srcStats = &clipBoard[BNODE].tnFiles[ZERO] ;
         }
         else if ( clipBoard[BNODE].nextLevel != NULL )
         {  //* Stats of directory-type file. If directory is not empty, also  *
            //* point to TreeNode object describing its contents.              *
            srcStats = &clipBoard[BNODE].nextLevel[ZERO].dirStat ;
            if ( clipBoard[BNODE].nextLevel[ZERO].totFiles > ZERO )
               pcl.srcTree = &clipBoard[BNODE].nextLevel[ZERO] ;
         }
         if ( srcStats != NULL )
         {  //* Copy the file or directory tree to target.          *
            //* If delete-after-copy, also delete the source files. *
            pasteCount = this->pslSinglePaste ( srcStats, pcl, tDirEQsDir, sameFilesys ) ;
         }
         else
         {  //* Getting to this point means that the clipboard data are *
            //* corrupt, so it's an application error (unlikely).       *
            this->SetErrorCode ( ecAPPERROR, EINVAL ) ;
         }
         //* Values returned in caller's parameters: *
         attCount = pcl.attCount ;  // copy operations attempted
         delCount = pcl.delCount ;  // for delete-after-copy, successful deletions
      }

      //* Process a list of files and directories *
      else
      {  //* Multiple top-level files and/or directory trees have been     *
         //* placed on the clipboard for this operation.                   *
         //* Determine which flavor of 'specialness' the user has in mind, *
         //* and perform the copy/move operation.                          *

         //* Data passed among lower-level methods *
         pclParm  pcl2( clipBoard_srcPath, this->currDir, &clipBoard[BNODE] ) ;
         pasteCount = pslMultiPaste ( pcl2, tDirEQsDir, sameFilesys ) ;
         //* Values returned in caller's parameters: *
         attCount = pcl2.attCount ; // copy operations attempted
         delCount = pcl2.delCount ; // for delete-after-copy, successful deletions
      }

      //* If this was a delete-after-copy operation, *
      //* clipboard contents are now invalid.        *
      if ( cutPending )
         this->ClearClipboard () ;

      //* Refresh the file list for the target directory *
      //* and display the results of the operation.      *
      if ( pasteCount > ZERO || tDirEQsDir )
         this->RefreshCurrDir () ;
      //* Else, just restore path and stats messages.    *
      else
         this->DisplayStatus () ;
   }   // if(clipBoard[BNODE].totFiles && dWritePerm && spaceAvailable)

   //* User lacks read and/or write access to target directory *
   else if ( trgdirPerm == false )
   {
      this->DisplayStatus ( ecREADONLY_T, EACCES ) ;
   }

   return pasteCount ;

}  //* End PasteSpecialList() *

//***********************
//*   pslSinglePaste    *
//***********************
//******************************************************************************
//* Prompt the user for the desired 'Paste-Special' option, then perform the   *
//* specified copy/move/create operation.                                      *
//* This method operates on a single file OR on a single directory tree.       *
//*                                                                            *
//* Input  : srcStats: pointer to source-file stats on clipboard               *
//*          pcl : (by reference)                                              *
//*                pcl.srcDir  == directory path of source file(s)             *
//*                pcl.trgDir  == directory path to paste target               *
//*                pcl.srcTree == NULL* by default, however,                   *
//*                 If srcStats describes a directory name AND if the          *
//*                 directory is not empty, then this is a pointer to the      *
//*                 data for a recursive copy of the directory tree.           *
//*                pcl.attCount: on return, contains number of copy/move       *
//*                 operations actually attempted, usually the number of       *
//*                 files on the clipboard                                     *
//*                pcl.delCount: initialized to ZERO, but if                   *
//*                 delete-after-copy operation, receives number of files      *
//*                 successfully deleted after copy,                           *
//*                pcl.psOption: set through user interaction                  *
//*          tDirEQsDir: if 'true', target base dir == source base dir         *
//*          sameFileSys: if 'true', source directory and target directory     *
//*                         are on the same file system                        *
//*                       if 'false', source and target are on different file  *
//*                         systems                                            *
//*                                                                            *
//* Returns: number of files successfully copied/created                       *
//******************************************************************************
//* If source is a non-empty directory, pcl.srcTree points to the contents     *
//* of the directory. However, for certain flavours of 'specialness', the      *
//* directory's contents must not be copied.                                   *
//* 1. psstSimpleCopy: If the operation is a simple copy of a directory-tree   *
//*    source to a directory-tree target, (with or without rename), copy the   *
//*    entire directory tree.                                                  *
//* 2. psstCopyLinkTarget: If the operation is to copy a sym-link's target AND *
//*    if the sym-link target is a directory, then the sym-link target         *
//*    directory's contents SHOULD BE COPIED just as if the source file had    *
//*    been that directory; however, the sym-link target's contents are        *
//*    probably not on the clipboard, so we cannot copy them, so set           *
//*    srcTree == NULL, just in case.                                          *
//* 3. If the operation is to create a sym-link to a source directory,         *
//*    clearly, the sym-link contains no files, so set srcTree == NULL.        *
//* 4. If the operation is to create a hard link to a source directory, note   *
//*    that a hard link to a directory is not possible, and should not have    *
//*    been offered to the user as an option, i.e. srcTree should be a NULL*   *
//*    already, but set srcTree == NULL, just in case.                         *
//******************************************************************************

UINT FileDlg::pslSinglePaste ( const tnFName* srcStats, pclParm& pcl, 
                               bool tDirEQsDir, bool sameFileSys )
{
   //* Determine which flavor of 'specialness' the user has in mind, *
   //* and perform the copy/move operation.                          *
   tnFName  ltargStats ;   // if source is a sym-link, this describes link target
   gString trgFName ;      // file name for target file
   pcl.psOption = this->pslSinglePaste_Options ( srcStats, ltargStats, trgFName, 
                                                 tDirEQsDir, sameFileSys ) ;

   //* Return value: number of target files successfully written *
   UINT  pasteCount = ZERO ;

   pcl.attCount = ZERO ;      // number of source files to process
   pcl.delCount = ZERO ;      // number of source files successfully deleted after copy
   pcl.deleteSrc = this->CutPending () ; // 'true' if delete-after-copy
   // Note: user prompt allows only psstSimpleCopy with a delete-after-paste 
   // operation, but better to test for it and be safe.
   if ( pcl.deleteSrc && (pcl.psOption != psstSimpleCopy) )
      pcl.deleteSrc = false ;

   //* If source is a non-empty directory, check whether the type of           *
   //* 'specialness' allows for copying the directory contents (see note above)*
   if ( srcStats->fType == fmDIR_TYPE && pcl.psOption != psstSimpleCopy )
   {
      pcl.srcTree = NULL ;
      pcl.attCount = 1 ;
   }

   if ( pcl.psOption != psstABORT )
   {
      //* Path/filename string for source file *
      gString srcPath ;
      this->fmPtr->CatPathFname ( srcPath, pcl.srcDir.ustr(), srcStats->fName ) ;
      
      //* Path/filename string for target file *
      gString trgPath ;
      this->fmPtr->CatPathFname ( trgPath, pcl.trgDir.gstr(), trgFName.gstr() ) ;

      //* Determine whether target already exists, is write protected, etc. *
      //* and prompt user for a decision if necessary.                      *
      corReturn coStatus = 
            this->pslTargetExists ( trgPath, srcPath, srcStats, pcl.psOption ) ;
      ++pcl.attCount ;              // operation attempted

      //* If target does not already exist OR if we have user's *
      //* permission to overwrite existing target, then proceed.*
      if ( coStatus == corProcessOnce || coStatus == corProcessAll )
      {
         //* Display a message that lets the user know we're working *
         this->ProcessCWD () ;

         short status = 
            this->pslPasteSpecial ( pcl.psOption, srcStats, srcPath, trgPath ) ;
         if ( status == OK )
         {
            ++pasteCount ;          // copy successful

            //* If source file is a non-empty directory, copy the contents *
            if ( srcStats->fType == fmDIR_TYPE && pcl.srcTree != NULL )
            {
               #if ENABLE_DEBUGGING_CODE != 0 && enableDBWIN != 0
               if ( dbWin == NULL )
                  dbWin = this->Open_DBwindow ( dbColor, dbLines, dbCols ) ;
               dbwp = { 1, 1 } ;
               #endif   // ENABLE_DEBUGGING_CODE && enableDBWIN
               UINT nodeFiles = 1 ; // for validating successful copy-before-delete
               UINT64 nodeBytes = ZERO ;
               #if enableDBWIN == 0
               if ( pcl.deleteSrc != false )
               #endif   // enableDBWIN == 0
               {
                  nodeFiles = ZERO ;
                  this->fmPtr->TreeNodeSummary ( pcl.srcTree, nodeFiles, nodeBytes ) ;
               }
               pcl.srcDir = srcPath ;
               pcl.trgDir = trgPath ;
               --pcl.attCount ;     // hide top-level operation from count
               pasteCount += this->pslCopyTnList ( pcl ) ;
               ++pcl.attCount ;     // restore top-level operation to count

               //* If delete-after-paste operation AND all files in node  *
               //* copied/deleted successfully, delete the directory name.*
               if ( pcl.deleteSrc && 
                    (pasteCount == nodeFiles) && (pcl.delCount == (nodeFiles-1)) )
               {
                  if ( (this->DeleteSourceFile ( srcStats, srcPath )) == OK )
                     ++pcl.delCount ;
               }
               #if ENABLE_DEBUGGING_CODE != 0 && enableDBWIN != 0
               if ( dbWin != NULL )
               {
                  gString gs ;
                  gs.compose( L"nodeFiles:%02u pasteCount:%02u", 
                              &nodeFiles, &pasteCount ) ;
                  dbWin->ClearLine ( dbLines - 3, false ) ;
                  dbWin->WriteString ( (dbLines-3), 2, gs, dbColor ) ;
                  gs.compose( L"attCount :%02u delCount  :%02u", 
                              &pcl.attCount, &pcl.delCount ) ;
                  dbWin->ClearLine ( dbLines - 2, false ) ;
                  dbWin->WriteString ( (dbLines-2), 2, gs, dbColor ) ;
                  dbWin->WriteString ( (dbLines-2), (dbCols-14), 
                                       " Press A Key ", this->cs.pf, true ) ;
                  nckPause();
                  delete dbWin ;
                  dbWin = NULL ;
               }
               #endif   // ENABLE_DEBUGGING_CODE && enableDBWIN
            }

            //* If delete-after-copy operation AND a single, non-directory     *
            //* file or empty directory was successfully copied, delete the    *
            //* source file.                                                   *
            else if ( pcl.deleteSrc )
            {
               if ( (this->DeleteSourceFile ( srcStats, srcPath )) == OK )
                  ++pcl.delCount ;
            }
         }
         else
         {  //* Report error copying the file *
            this->DisplayErrorCode () ;
         }
      }
   }
   return pasteCount ;

}  //* End pslSinglePaste() *

//***********************
//*    pslCopyTnList    *
//***********************
//******************************************************************************
//* Copy all files and directories in a TreeNode list to target directory.     *
//*                                                                            *
//* Caller has verified that target base directory (described by               *
//* pcl.srcTree->dirStat) exists and that user has write permission for that   *
//* directory.                                                                 *
//*                                                                            *
//* Input  : pclParm structure (by reference)                                  *
//*            pcl.srcDir : source directory path                              *
//*            pcl.trgDir : target directory path                              *
//*            pcl.psOption: flavour of 'specialness' in Paste-Special         *
//*            pcl.srcTree: pointer to TreeNode object describing subdir tree  *
//*             pcl.srcTree->nextLevel:array of TreeNode objects               *
//*             pcl.srcTree->dirFiles:number of items in nextLevel array       *
//*             pcl.srcTree->tnFiles :array of tnFName objects for source files*
//*             pcl.srcTree->thFCount:number of items in tnFiles array         *
//*            pcl.attCount:                                                   *
//*            - number of copy operations attempted is added to existing total*
//*            pcl.delCount:                                                   *
//*            - if delete-after-copy operation, then number of source files   *
//*              deleted is added to delCount                                  *
//*            - else value returns unchanged                                  *
//*                                                                            *
//* Returns: number of files copied                                            *
//*           (if operation was aborted, pcl.psOption set to psstABORT)        *
//******************************************************************************

UINT FileDlg::pslCopyTnList ( pclParm& pcl )
{
UINT  pasteCount = ZERO ;

   #if ENABLE_DEBUGGING_CODE != 0 && enableDBWIN != 0
   const bool  inclPV = false ;
   if ( dbWin != NULL )
   {
      if ( inclPV )
      {
         const bool  inclTN = false ;
         NcDialog* tnWin = NULL ;
         const short tnWinLines = 19 ;
         dbWin->ClearWin () ;
         dbwp = { 1, 1 } ;
         dbWin->WriteString ( dbwp.ypos++, dbwp.xpos, "Source Directory", dbColor ) ;
         dbWin->WriteString ( dbwp.ypos++, dbwp.xpos, 
                              &pcl.srcDir.gstr()[dbPOffset], dbColor ) ;
         dbWin->WriteString ( dbwp.ypos++, dbwp.xpos, "Target Directory", dbColor ) ;
         dbWin->WriteString ( dbwp.ypos++, dbwp.xpos, 
                              &pcl.trgDir.gstr()[dbPOffset], dbColor ) ;
         if ( pcl.srcTree != NULL )
         {
            if ( inclTN )
            {
               dbwp.xpos = 79 ;
               tnWin = this->Display_TreeNode ( dbwp, pcl.srcTree, true ) ;
               dbwp = { short(dbwp.ypos + tnWinLines), 1 } ;
            }
   
            dbWin->WriteString ( dbwp.ypos++, dbwp.xpos++, "Sub-nodes:", dbColor ) ;
            if ( pcl.srcTree->dirFiles > ZERO && pcl.srcTree->nextLevel != NULL )
            {
               gString gs ;
               for ( UINT i = ZERO ; i < pcl.srcTree->dirFiles ; i++ )
               {
                  this->fmPtr->CatPathFname ( gs, pcl.srcDir.ustr(), 
                                       pcl.srcTree->nextLevel[i].dirStat.fName ) ;
                  dbWin->WriteString ( dbwp.ypos++, dbwp.xpos, 
                                       &gs.gstr()[dbPOffset], dbColor ) ;
               }
            }
            else
               dbWin->WriteString ( dbwp.ypos++, dbwp.xpos, "none", dbColor ) ;
            --dbwp.xpos ;
            dbWin->WriteString ( dbwp.ypos++, dbwp.xpos++, "Non-dir Files:", dbColor ) ;
            if ( pcl.srcTree->tnFCount > ZERO && pcl.srcTree->tnFiles != NULL )
            {
               gString gs ;
               for ( UINT i = ZERO ; i < pcl.srcTree->tnFCount ; i++ )
               {
                  this->fmPtr->CatPathFname ( gs, pcl.srcDir.ustr(), 
                                              pcl.srcTree->tnFiles[i].fName ) ;
                  dbWin->WriteString ( dbwp.ypos++, dbwp.xpos, 
                                       &gs.gstr()[dbPOffset], dbColor ) ;
               }
            }
            else
               dbWin->WriteString ( dbwp.ypos++, dbwp.xpos, "none", dbColor ) ;
            dbWin->WriteString ( (dbLines-2), (dbCols-14), 
                                 " Press A Key ", this->cs.pf, true ) ;
            if ( inclTN && tnWin != NULL )
               tnWin->RefreshWin () ;
            nckPause();
            dbWin->WriteString ( (dbLines-2), (dbCols-14), 
                                 "             ", dbColor, true ) ;
            if ( inclTN && tnWin != NULL )
            {
               delete tnWin ;
               dbwp = { 6, 2 } ;
            }
         }
         dbWin->RefreshWin () ;
      }
   }
   #endif   // ENABLE_DEBUGGING_CODE && enableDBWIN

   //* If caller actually sent us some source files *
   if ( pcl.srcTree != NULL )
   {
      gString gsSrc, gsTrg ;  // for creating path/filename strings
      pclParm  pcl2 ;         // for making recursive calls
      if ( pcl.srcTree->dirFiles > ZERO && pcl.srcTree->nextLevel != NULL )
      {
         for ( UINT i = ZERO ; i < pcl.srcTree->dirFiles ; i++ )
         {  //* Create the target directories at this level, then              *
            //* Recursively call this method until we reach the lowest level   *
            //* of the TreeNode list, then process the files and directory     *
            //* contents in reverse order.                                     *
            this->fmPtr->CatPathFname ( gsSrc, pcl.srcDir.ustr(), 
                                        pcl.srcTree->nextLevel[i].dirStat.fName ) ;
            pcl2.srcDir = gsSrc ;
            this->fmPtr->CatPathFname ( gsTrg, pcl.trgDir.ustr(), 
                                        pcl.srcTree->nextLevel[i].dirStat.fName ) ;
            pcl2.trgDir = gsTrg ;
            pcl2.srcTree = &pcl.srcTree->nextLevel[i] ;
            pcl2.attCount = pcl2.delCount = ZERO ;
            pcl2.deleteSrc = pcl.deleteSrc ;

            //* Determine whether target already exists, is write protected,   *
            //* etc. and prompt user for a decision if necessary.              *
            corReturn coStatus = 
               this->pslTargetExists ( gsTrg, gsSrc, 
                            &pcl.srcTree->nextLevel[i].dirStat, pcl.psOption ) ;
            ++pcl.attCount ;        // operation attempted

            //* If target does not already exist OR if we have user's *
            //* permission to overwrite existing target, then proceed.*
            if ( (coStatus == corProcessOnce || coStatus == corProcessAll) &&
                 ((this->CopyFile ( pcl.srcTree->nextLevel[i].dirStat, 
                                    gsSrc, gsTrg )) == OK) )
            {
               #if ENABLE_DEBUGGING_CODE != 0 && enableDBWIN != 0
               winPos dbSD ;
               if ( ! inclPV && dbWin != NULL )
               {
                  dbwp = dbWin->WriteString ( dbwp.ypos, (dbwp.xpos=1), "DIR: ", dbColor ) ;
                  dbSD = dbWin->WriteString ( dbwp.ypos++, dbwp.xpos, 
                                              &gsTrg.gstr()[dbPOffset], dbColor ) ;
                  dbWin->WriteString ( (dbLines-2), (dbCols-14), " Press A Key ", 
                                       this->cs.pf, true ) ;
                  nckPause();
                  dbWin->WriteString ( (dbLines-2), (dbCols-14), "             ", 
                                       dbColor, true ) ;
               }
               #endif   // ENABLE_DEBUGGING_CODE && enableDBWIN

               ++pasteCount ;       // target directory created

               //* If source subdirectory contains additional *
               //* files or subdirectories, copy/move them.   *
               if ( pcl2.srcTree->totFiles > ZERO )
               {
                  UINT pCount = this->pslCopyTnList ( pcl2 ) ;
                  pasteCount += pCount ;           // update the accumulators
                  pcl.attCount += pcl2.attCount ;
                  pcl.delCount += pcl2.delCount ;
               }

               //* If a delete-after-copy operation AND all lower-level files  *
               //* and subdirs moved successfully, delete the source directory.*
               if ( pcl.deleteSrc &&
                    (pcl.attCount == pasteCount) && (pcl.delCount == (pasteCount - 1)) )
               {
                  if ( (this->DeleteSourceFile ( 
                              &pcl.srcTree->nextLevel[i].dirStat, gsSrc )) == OK )
                     ++pcl.delCount ;
                  #if ENABLE_DEBUGGING_CODE != 0 && enableDBWIN != 0
                  if ( ! inclPV && dbWin != NULL )
                  {
                     dbWin->WriteString ( dbSD.ypos, dbSD.xpos, " SD", dbColor ) ;
                  }
                  #endif   // ENABLE_DEBUGGING_CODE && enableDBWIN
               }
            }
            else
            {  //* Unable to create a subdirectory on the tree, so we can't    *
               //* continue to traverse the tree. If user has not aborted,     *
               //* move on to the next tree node (if any), although it will    *
               //* likely fail as well.                                        *
               if ( coStatus == corAbort )
                  pcl.psOption = psstABORT ;
            }
            this->wip.update( 1, pcl.srcTree->nextLevel[i].dirStat.fBytes ) ;

            #if ENABLE_DEBUGGING_CODE != 0 && enableDBBAR != 0
            //* Insert 150 mS delay to allow debug of Progress Bar *
            chrono::duration<short, std::milli>aMoment( 150 ) ;
            this_thread::sleep_for( aMoment ) ;
            #endif   // ENABLE_DEBUGGING_CODE && enableDBBAR
         }     // for(;;)
      }        // (pcl.srcTree->dirFiles>ZERO && pcl.srcTree->nextLevel!=NULL)
      //* If non-directory files at this level, copy/move them *
      pasteCount += this->pslCopyFileList ( pcl ) ;
   }

   return pasteCount ;

}  //* End pslCopyTnList() *

//***********************
//*   pslCopyFileList   *
//***********************
//******************************************************************************
//* Copy the list of non-directory files to target directory.                  *
//*                                                                            *
//* Caller has verified that user has write permission on the target directory.*
//*                                                                            *
//*                                                                            *
//* Input  : pclParm structure (by reference)                                  *
//*            pcl.srcDir  : source directory path                             *
//*            pcl.trgDir  : target directory path                             *
//*            pcl.psOption: flavour of 'specialness' in Paste-Special         *
//*                          (not referenced - 'specials' handled elsewhere)   *
//*            pcl.srcTree : pointer to TreeNode object describing subdir tree *
//*             pcl.srcTree->tnFiles :array of tnFName objects for source files*
//*             pcl.srcTree->tnFCount:number of items in tnFiles array         *
//*            pcl.delCount:                                                   *
//*            - if delete-after-copy operation, then number of source files   *
//*              deleted is added to delCount                                  *
//*            - else value returns unchanged                                  *
//*            pcl.deleteSrc: 'true' if delete-after-copy operation            *
//*          rPrompt: (optional, 'false' by default)                           *
//*                if 'true', prompt user to provide a name for each target    *
//*                file copied                                                 *
//*          tDirEQsDir: (optional, 'false' by default)                        *
//*                      if 'true' target base directory == source base        *
//*                      directory, so prompt user for copy-with-rename        *
//*          rPat: (optional, NULL* by default)                                *
//*                If != NULL, (and rPrompt==false) then name the target files *
//*                according to the specified pattern.                         *
//*                                                                            *
//* Returns: number of files copied                                            *
//*           (if operation was aborted, pcl.psOption set to psstABORT)        *
//******************************************************************************

UINT FileDlg::pslCopyFileList ( pclParm& pcl, bool rPrompt, bool tDirEQsDir, 
                                renamePattern* rPat )
{
   UINT  pasteCount = ZERO ;

   #if ENABLE_DEBUGGING_CODE != 0 && enableDBWIN != 0
   if ( dbWin != NULL )
   {
      gString dbgs ;
      dbgs.compose( L"Non-dir Files:%02u", &pcl.srcTree->tnFCount ) ;
      dbWin->WriteString ( dbwp.ypos++, (dbwp.xpos=1), dbgs, dbColor ) ;
   }
   #endif   // ENABLE_DEBUGGING_CODE && enableDBWIN

   if ( pcl.srcTree != NULL && 
        pcl.srcTree->tnFCount > ZERO && pcl.srcTree->tnFiles != NULL )
   {
      gString srcPath, trgPath ; // for creating path/filename strings
      gString newFName ;         // if target filename provided by user
      tnFName* srcStats ;        // pointer to source file stats
      corReturn coStatus ;       // status flag for processing individual files
//      short     opStatus ;       // status from copy/delete call
      for ( UINT i = ZERO ; i < pcl.srcTree->tnFCount && coStatus != corAbort ; i++ )
      {
         coStatus = corProcessOnce ;   // hope for the best

         //* Point to source file stat info and      *
         //* create source file path/filename string *
         srcStats = &pcl.srcTree->tnFiles[i] ;
         this->fmPtr->CatPathFname ( srcPath, pcl.srcDir.ustr(), srcStats->fName ) ;

         //* Create target file path/filename string *
         if ( rPrompt != false ) // prompt user for target filename
         {
            coStatus = this->pclGetNewFilename ( srcStats, newFName, tDirEQsDir ) ;
            if ( coStatus == corProcessOnce )
               this->fmPtr->CatPathFname ( trgPath, pcl.trgDir.gstr(), newFName.gstr() ) ;
            else
               pcl.psOption = psstABORT ;
         }
         else if ( rPat != NULL )// create target name from batch-rename pattern
         {
            rPat->formatFName ( newFName, srcStats->fName ) ;
            //* If target directory == source directory, check to be sure that *
            //* we are not going to overwrite a source file.                   *
            if ( tDirEQsDir )
            {
               bool tryAgain = false ;
               if ( (this->pclFNameInUse ( newFName, tryAgain )) != false )
                  this->pclCreateSuggestedName ( newFName, newFName ) ;
            }
            this->fmPtr->CatPathFname ( trgPath, pcl.trgDir.gstr(), newFName.gstr() ) ;
         }
         else                    // target file name == source file name
            this->fmPtr->CatPathFname ( trgPath, pcl.trgDir.ustr(), srcStats->fName ) ;
         #if ENABLE_DEBUGGING_CODE != 0 && enableDBWIN != 0
         if ( dbWin != NULL )
         {
            dbwp = dbWin->WriteString ( dbwp.ypos, (dbwp.xpos=2), "trg: ", dbColor ) ;
            dbwp = dbWin->WriteString ( dbwp.ypos, dbwp.xpos, &trgPath.gstr()[dbPOffset], dbColor ) ;
         }
         #endif   // ENABLE_DEBUGGING_CODE && enableDBWIN

         //* Determine whether target already exists, is write protected, etc. *
         //* and prompt user for a decision if necessary. If we are in         *
         //* interactive rename mode, this has already been checked.           *
         if ( ! tDirEQsDir )
            coStatus = this->pslTargetExists ( trgPath, srcPath, srcStats, psstSimpleCopy ) ;
         ++pcl.attCount ;           // operation attempted

         //* If target does not already exist OR if we have user's *
         //* permission to overwrite existing target, then proceed.*
         if ( coStatus == corProcessOnce || coStatus == corProcessAll )
         {
            //* Write the target file using the specified type of copy *
            //* (currently, pcl.psOption always == psstSimpleCopy)     *
            if ( (this->CopyFile ( *srcStats, srcPath, trgPath )) == OK )
            {  //* Source file successfully copied *
               ++pasteCount ;

               #if ENABLE_DEBUGGING_CODE != 0 && enableDBWIN != 0
               if ( dbWin != NULL )
                  dbwp = dbWin->WriteString ( dbwp.ypos, dbwp.xpos, " OK", dbColor ) ;
               #endif   // ENABLE_DEBUGGING_CODE && enableDBWIN

               //* If delete-after-copy operation, delete the source file *
               if ( pcl.deleteSrc )
               {
                  if ( (this->DeleteSourceFile ( srcStats, srcPath )) == OK )
                     ++pcl.delCount ;

                  #if ENABLE_DEBUGGING_CODE != 0 && enableDBWIN != 0
                  if ( dbWin != NULL )
                     dbWin->WriteString ( dbwp.ypos, dbwp.xpos, " SD", dbColor ) ;
                  #endif   // ENABLE_DEBUGGING_CODE && enableDBWIN
               }
            }
            #if ENABLE_DEBUGGING_CODE != 0 && enableDBWIN != 0
            ++dbwp.ypos ;
            #endif   // ENABLE_DEBUGGING_CODE && enableDBWIN
         }  // (coStatus==corProcessOnce || coStatus==corProcessAll)

         //* Update work-in-progress stats *
         this->wip.update( 1, srcStats->fBytes ) ;

         #if ENABLE_DEBUGGING_CODE != 0 && enableDBBAR != 0
         //* Insert 150 mS delay to allow debug of Progress Bar *
         chrono::duration<short, std::milli>aMoment( 150 ) ;
         this_thread::sleep_for( aMoment ) ;
         #endif   // ENABLE_DEBUGGING_CODE && enableDBBAR
      }     // for(;;)
   }
   #if ENABLE_DEBUGGING_CODE != 0 && enableDBWIN != 0
   if ( dbWin != NULL )
   {
      dbWin->WriteString ( (dbLines-2), (dbCols-14), " Press A Key ", 
                           this->cs.pf, true ) ;
      nckPause();
      dbWin->WriteString ( (dbLines-2), (dbCols-14), "             ", 
                           dbColor, true ) ;
   }
   #endif   // ENABLE_DEBUGGING_CODE && enableDBWIN

   return pasteCount ;

}  //* End pslCopyFileList() *

//***********************
//*   pslPasteSpecial   *
//***********************
//******************************************************************************
//* Perform the specified copy/move/create on the source file.                 *
//*                                                                            *
//*                                                                            *
//* Input  : pasteOption: Paste-Special processing option                      *
//*          srcStats   : pointer to stat info for source file                 *
//*          srcPath    : full path/filename specification for source file     *
//*          trgPath    : full path/filename specification for target file     *
//*                                                                            *
//* Returns: OK if successful                                                  *
//*          'errno' value if error. Also error status is saved and can be     *
//*             retrieved through GetErrorCode() or DisplayErrorCode().        *
//******************************************************************************

short FileDlg::pslPasteSpecial ( psSubType pasteOption, const tnFName* srcStats, 
                                 const gString& srcPath, const gString& trgPath )
{
short status = OK ;

   //* Create a symbolic link at trgPath, referencing file at srcPath *
   if ( pasteOption == psstCreateSymLink )
   {  //* If source file is not a symbolic link,     *
      //* create a symbolic link to the source file. *
      if ( srcStats->fType != fmLINK_TYPE )
      {
         status = this->fmPtr->CreateSymbolicLink ( srcPath.ustr(), trgPath.ustr() ) ;
         if ( status != OK )
            this->SetErrorCode ( ecLINK ) ;
      }
      else
      {  //* Source is a symbolic link *
         // Programmer's Note: The user prompt will not allow the option of 
         // creating a link to a link, so we should never get here. If we 
         // arrive here, it's an application-logic error.
         status = EINVAL ;
         this->SetErrorCode ( ecAPPERROR, EINVAL ) ;
      }
   }

   //* Copy the srcPath symbolic-link's target file to trgPath *
   else if ( pasteOption == psstCopyLinkTarget )
   {  //* The link target SHOULD exist because the user prompt would not allow *
      //* this option if the source file was a 'broken link'.                  *
      status = this->CopyFile ( *srcStats, srcPath, trgPath, true ) ;
   }

   //* Do a simple copy of srcPath to trgPath (this is not actually a *
   //* 'special' paste, but it's easier than arguing with the user.   *
   else if ( pasteOption == psstSimpleCopy )
   {
      status = this->CopyFile ( *srcStats, srcPath.ustr(), trgPath.ustr() ) ;
   }

   //* Create a hard link to srcPath, located at trgPath *
   else if ( pasteOption == psstCreateHardLink )
   {
      status = this->fmPtr->CreateHardLink ( srcPath.ustr(), trgPath.ustr() ) ;
      if ( status != OK )
         this->SetErrorCode ( ecHARDLINK ) ;
   }

   //* Else, caller has head stuck in a dark, moist, smelly orifice. *
   else
      status = EPERM ;

   return status ;

}  //* End pslPasteSpecial() *

//***********************
//*   pslTargetExists   *
//***********************
//******************************************************************************
//* For a copy/move operation, determine whether the target file already       *
//* exists.                                                                    *
//* 1. If the target does not exist, just say so.                              *
//* 2. If the target DOES exist, test in presumed order of likelihood:         *
//*    a. If a file is about to be copied onto itself, give user the news and  *
//*       do not allow overwrite.                                              *
//*    b. If target is a directory file: fmDEV_TYPE:                           *
//*       - If source is not a directory, then give user the news and do not   *
//*         allow overwrite.                                                   *
//*    c. If source is a symbolic link and existing target is not a symbolic   *
//*       link, give user the news and do not allow overwrite.                 *
//*    d. If target is a FIFO, allow it to be overwritten only by another FIFO.*
//*       If source is not a FIFO, give user the news and do not allow         *
//*       overwrite.                                                           *
//*    e. If target is a 'special' file: fmCHDEV_TYPE, fmBLKDEV_TYPE or        *
//*       fmSOCK_TYPE, warn user that they are about to trash their system,    *
//*       and do not allow overwrite.                                          *
//*    f. If existing target is of a different file type than the source,      *
//*       prompt user for a decision whether to overwrite the target.          *
//*       - Exception: If existing target is a sym-link, silently allow        *
//*         overwrite because overwriting a sym-link costs nothing.            *
//*    g. Otherwise, check the configuration option for overwrite of existing  *
//*       targets, also the process-control rules (flags) for the paste        *
//*       operation, as well as whether target is write protected, and if      *
//*       necessary, prompt user for a decision.                               *
//*                                                                            *
//*                                                                            *
//* Input  : trgPath : path/filename of target file                            *
//*          srcPath : path/filename of source file (for copy-onto-self test)  *
//*          srcStats: stats for file to be copied/moved                       *
//*          pOpt    : type of 'paste' to be performed (used only to test      *
//*                    whether operation is to create a symbolic link to       *
//*                    source)                                                 *
//*                                                                            *
//* Returns: member of enum corReturn                                          *
//******************************************************************************

corReturn FileDlg::pslTargetExists ( const gString& trgPath, const gString& srcPath,
                                     const tnFName* srcStats, psSubType pOpt )
{
   tnFName  trgStats ;     // stats for existing target file
   bool     targExists = this->TargetExists ( trgPath.ustr(), trgStats ) ;
   corReturn trgStatus = corProcessOnce ; // return value

   //* If the target physically exists *
   if ( targExists )
   {
      //* If source and target path/filename are identical, caller is confused.*
      if ( (wcscmp ( srcPath.gstr(), trgPath.gstr() )) == ZERO )
      {
         this->InfoDialog ( notOntoSelf ) ;
         trgStatus = corSkipOnce ;
      }
      //* Cannot overwrite a directory with a non-directory file *
      else if ( srcStats->fType != fmDIR_TYPE && trgStats.fType == fmDIR_TYPE )
      {
         this->InfoDialog ( cannotOverwriteDirectory ) ;
         trgStatus = corSkipOnce ;
      }
      //* Do not allow overwrite of a real file by a symbolic link *
      else if ( (srcStats->fType == fmLINK_TYPE || pOpt == psstCreateSymLink) &&
                trgStats.fType != fmLINK_TYPE )
      {  //* Tell user that he/she/it is being un-cool *
         this->SetErrorCode ( ecLINKOVER, ZERO ) ;
         this->InfoDialog ( noLinkOverwrite ) ;
         trgStatus = corSkipOnce ;
      }
      //* Cannot overwrite a FIFO with a non-FIFO file *
      else if ( trgStats.fType == fmFIFO_TYPE && srcStats->fType != fmFIFO_TYPE )
      {
         this->InfoDialog ( cannotOverwriteFifo ) ;
         trgStatus = corSkipOnce ;
      }
      //* Cannot overwrite a 'special' file *
      else if ( trgStats.fType == fmCHDEV_TYPE  || 
                trgStats.fType == fmBKDEV_TYPE || 
                trgStats.fType == fmSOCK_TYPE )
      {
         this->InfoDialog ( cannotDeleteSpecial ) ;
         trgStatus = corSkipOnce ;
      }
      //* Warn if existing target is of a different file type than source *
      else if ( srcStats->fType != trgStats.fType && trgStats.fType != fmLINK_TYPE )
      {
         if ( ! (this->DecisionDialog ( differentFileTypes )) )
            trgStatus = corSkipOnce ;
      }
      else
      {
         //* Compare source and target modification dates. This is to determine*
         //* the prompting level for overwriting existing target files.        *
         //* (mod date does not apply to fmDIR_TYPE files.)                    *
         bool  targModDateGE = false ;
         if ( trgStats.fType != fmDIR_TYPE )
            targModDateGE = (bool)(trgStats.rawStats.st_mtime >= srcStats->rawStats.st_mtime) ;

         if (   cbConfirmAllOverwrites 
             || (trgStats.writeAcc == false && 
                 !cbOverwriteProtectedTrgAllowed &&
                 !cbSkipAllWriteProtectedSrc) 
             || (targModDateGE && 
                 !cbOverwriteNewerTrgAllowed && !cbSkipAllOverwriteNewerTrg) ) 
         {
            trgStatus = this->ConfirmOverwrite ( srcStats, &trgStats ) ;

            //* Adjust our flow-control flags *
            switch ( trgStatus )
            {
               //* If user opted to override all rules for remaining files in  *
               //* list Provisionally comply. If file is write-protected, allow*
               //* silent overwrite of write-protected target files. If file's *
               //* mod date is newer than source mod date, allow silent        *
               //* overwrite of newer target files. Finally, reset the         *
               //* cbConfirmAllOverwrites flag.                                *
               case corProcessAll:
                  if ( trgStats.writeAcc == false && !cbOverwriteProtectedTrgAllowed )
                     cbOverwriteProtectedTrgAllowed = true ;
                  if ( targModDateGE && !cbOverwriteNewerTrgAllowed )
                     cbOverwriteNewerTrgAllowed = true ;
                  cbConfirmAllOverwrites = false ;
                  break ;

               //* If user has opted to skip processing for this source file,  *
               //* and silently obey the current processing rules for all      *
               //* remaining files in the list.                                *
               case corSkipAll:
                  if ( trgStats.writeAcc == false && !cbSkipAllWriteProtectedTrg )
                  {  //* If user request is to silently skip all remaining     *
                     //* write-protected target files                          *
                     cbOverwriteProtectedTrgAllowed = false ;
                     cbSkipAllWriteProtectedTrg = true ;
                  }
                  if ( targModDateGE && !cbOverwriteNewerTrgAllowed )
                  {  //* If user request is to silently skip all remaining     *
                     //* target-newer-than-source files                        *
                     cbSkipAllOverwriteNewerTrg = true ;
                     cbOverwriteNewerTrgAllowed = false ;
                  }
                  cbConfirmAllOverwrites = false ;
                  break ;

               //* If user opted to override write protection on the existing  *
               //* target file, leave the rules unchanged, but attempt to      *
               //* remove target's write-protection before alerting caller     *
               //* that it's ok to overwrite this target file.                 *
               case corProcessOnceWP:
                  if ( this->fmPtr->WritePermission ( trgStats, trgPath, true ) )
                     trgStatus = corProcessOnce ;
                  else
                  {
                     trgStatus = corNoAccess ;
                     this->SetErrorCode ( ecTARGPROTECT ) ;
                  }
                  break ;

               //* If user opted to override rules for this file only, leave   *
               //* the rules unchanged, but alert caller it's ok to overwrite  *
               //* this target file.                                           *
               case corProcessOnce: // (fall through)
               //* If user opted to skip processing for this source file, then *
               //* the rules are left unchanged, and existing target file will *
               //* remain unmodified.                                          *
               case corSkipOnce: // (fall through)
               //* If user has opted to abort the operation in progress.       *
               //* Do not process this file or any remaining files in the list.*
               case corAbort:    // (fall through)
               case corNoAccess: // this value is not returned by user prompt
               default:
                  break ;
            }
         }
         else if ( trgStats.writeAcc == false && 
                   !cbOverwriteProtectedTrgAllowed &&
                   cbSkipAllWriteProtectedSrc)
         {  //* If target is write-protected, but user has previously specified*
            //* that write-protected targets should be skipped, skip it.       *
            trgStatus = corSkipOnce ;
         }
         else if ( targModDateGE && !cbOverwriteNewerTrgAllowed && 
                   cbSkipAllOverwriteNewerTrg ) 
         {  //* If target is newer than source, but user has previously        *
            //* specified that newer targets should be skipped, skip it.       *
            trgStatus = corSkipOnce ;
         }
         else
         { /* No objections to overwrite */ }
      }
   }
   return trgStatus ;

}  //* End pslTargetExists() *

//***************************
//* pslSinglePaste_Options  *
//***************************
//******************************************************************************
//* Called by pslSinglePaste() if there is only one source file or one         *
//* directory tree to paste.                                                   *
//*                                                                            *
//* Ask user to select from available Paste-Special options.                   *
//*                                                                            *
//*                                                                            *
//* Input  : srcStats  : pointer to source file's data structure in clipboard  *
//*                       list                                                 *
//*          ltargStats: (by reference, initial contents ignored)              *
//*                       if srcStats references a symbolic link, on return,   *
//*                       this parameter is initialized with stats for         *
//*                       sym-link target. Else, contents undefined.           *
//*          trgFName  :(by reference, initial contents ignored)               *
//*                       receives the target filename                         *
//*          tDirEQsDir: if 'true' target base dir == source base dir          *
//*          sameFileSys: if 'true', source directory and target directory     *
//*                         are on the same file system                        *
//*                       if 'false', source and target are on different file  *
//*                         systems                                            *
//*                                                                            *
//* Returns: member of enum psSubType                                          *
//******************************************************************************
//* Programmer's Note: This dialog is a bit complex because some control       *
//* definitions are modified before instantiation. Instantiation depends upon  *
//* the type of source file AND the pending clipboard operation (copy or cut). *
//*                                                                            *
//*                                                                            *
//* Let's look at the logic of these selections:                               *
//* --------------------------------------------                               *
//* A. If source is one file only, the actual operation is easier than a       *
//*    multi-file operation, but a comprehensive solution requires more user   *
//*    options.                                                                *
//*    1. If 'copy' operation in progress:                                     *
//*       a. If source is a symbolic link (with a valid link target file):     *
//*          - copy the link                                                   *
//*            - with source name  ('normal' copy)                             *
//*            - with link-target name                                         *
//*            - with user-supplied name                                       *
//*          - copy the link target                                            *
//*            - with link name                                                *
//*            - with link-target name       (Default selection)               *
//*            - with user-supplied name                                       *
//*                                                                            *
//*       b. If source is a broken symbolic link (i.e. with no link target):   *
//*          - copy the link                                                   *
//*            - with source name            (Default selection)               *
//*            - with link-target name, if known (this option not offered)     *
//*            - with user-supplied name                                       *
//*                                                                            *
//*       c. If source is not a symbolic link:                                 *
//*          - copy the source file                                            *
//*            - with source name ('normal' copy)                              *
//*            - with user-supplied name                                       *
//*          - create symbolic link to source file                             *
//*            - with source name            (Default selection)               *
//*            - with user-supplied name                                       *
//*          - create hard link (note: cannot create hard link to a directory) *
//*            - with source name                                              *
//*            - with user-supplied name                                       *
//*                                                                            *
//*    2. If 'cut' operation in progress:                                      *
//*       a. If source is a symbolic link (with a valid link target file):     *
//*          - move the link                                                   *
//*            - with source name            (Default selection)               *
//*            - with link-target name                                         *
//*            - with user-supplied name                                       *
//*                                                                            *
//*       b. If source is a broken symbolic link (i.e. with no link target):   *
//*          - move the link                                                   *
//*            - with source name            (Default selection)               *
//*            - with link-target name, if known (this option not offered)     *
//*            - with user-supplied name                                       *
//*                                                                            *
//*       c. If source is not a symbolic link:                                 *
//*          - move the source file                                            *
//*            - with source name            (Default selection)               *
//*            - with user-supplied name                                       *
//*                                                                            *
//* B. For multiple files, see pslMultiPaste_Options().                        *
//*                                                                            *
//*                                                                            *
//* Configure control array parameters to match the type of source file.       *
//* Defaults selections for various scenarios.                                 *
//* 1) Default selection for regular files is to copy the source file with     *
//*    rename. (ic[] array is defined for this)                                *
//*                                                                            *
//*    Specify type for new file:                                              *
//*     < x > Copy the source file                                             *
//*     <   > Create a symbolic link to file                                   *
//*     <   > Create a 'hard' link to file                                     *
//*    Specify name for new file:                                              *
//*     <   > Use the source file name                                         *
//*     < x > Enter a new file name below                                      *
//*     .....                                                                  *
//*                            trgTB active and contains source name           *
//*                                                                            *
//* 2) Default selection for 'special' files is to create a symbolic link to   *
//*    it, using the source file's name.                                       *
//*                                                                            *
//*    Specify type for new file:                                              *
//*     <   > Copy the source file                                             *
//*     < x > Create a symbolic link to file                                   *
//*     <   > Create a 'hard' link to file                                     *
//*    Specify name for new file:                                              *
//*     < x > Use the source file name                                         *
//*     <   > Enter a new file name below                                      *
//*     .....                                                                  *
//*                            trgTB inactive and contains source name         *
//*                                                                            *
//* 3) Default selection for directories is to copy the directory tree with    *
//*    rename of top directory.                                                *
//*                                                                            *
//*    Specify type for new file:                                              *
//*     < x > Copy the directory tree                                          *
//*     <   > Create a symbolic link to directory                              *
//*     .....                                                                  *
//*    Specify name for new file:                                              *
//*     <   > Use the source directory name                                    *
//*     < x > Enter a new directory name below                                 *
//*     .....                                                                  *
//*                            trgTB active and contains source name           *
//*                                                                            *
//* 4) Default selection for symbolic links is to copy the link target (if     *
//*    any) using the link target's name. If no valid target, copy the link    *
//*    itself.                                                                 *
//*                                                                            *
//*    Specify type for new file:                                              *
//*     <   > Copy the link file             [dflt if no link target]          *
//*     < x > Copy the link target           [if it exists]                    *
//*     .....                                                                  *
//*    Specify name for new file:                                              *
//*     <   > Use the link file name         [dflt if no link target]          *
//*     <   > Enter a new file name below                                      *
//*     < x > Use the link-target name                                         *
//*                            trgTB inactive and contains target name if      *
//*                            available, else source name                     *
//*                                                                            *
//* About creating links:                                                      *
//*  1. Hard links may be created only within a file system.                   *
//*     If source and target directories are on the same file system AND if the*
//*     source file is not fmDIR_TYPE, then all is well.                       *
//*     If source is a fmDIR_TYPE (directory file) OR if source and destination*
//*     are on different file systems, then we do not offer the                *
//*     create-hard-link option.                                               *
//*                                                                            *
//*  2. Symbolic links mean nothing, so we can create them across file systems,*
//*     so long as the target file system supports sym-links.                  *
//*     If the link is broken because the target resides on an un-mounted      *
//*     file system, then it is not a disaster. Still, we should warn the user *
//*     when a link is being created on a different file system.               *
//*                                                                            *
//******************************************************************************

psSubType FileDlg::pslSinglePaste_Options ( const tnFName* srcStats, 
                                            tnFName& ltargStats, gString& trgFName, 
                                            bool tDirEQsDir, bool sameFileSys )
{
   const char* Labels[] = 
   {                                                              // INDEX
      " ",                                                        // (0)
      "Copy the source file",                      // psrcRB labels  (1)
      "Move the source file",                                     // (2)
      "Copy the directory tree",                                  // (3)
      "Move the directory tree",                                  // (4)
      "Copy the symbolic link",                                   // (5)
      "Move the symbolic link",                                   // (6)
   
      "Create a symbolic link to source file",     // pslnRB labels  (7)
      "Create a symbolic link to source directory",               // (8)
      "Copy the link target",                                     // (9)
   
      "Create a 'hard' link to source file",       // phlnRB labels  (10)
   
      "Use the source file name",                  // nsrcRB labels  (11)
      "Use the source directory name",                            // (12)
      "Use the sym-link name",                                    // (13)
   
      "Enter a new file name below",               // nnewRB labels  (14)
   
      "Use name of link target",                   // ntrgRB labels  (15)
   } ;
   const char* customButton = "....." ;
   const short dialogLines = 18,                // dialog lines
               dialogCols  = INFODLG_WIDTH ;    // dialog columns
   short    ctrY = this->fulY + this->fMaxY/2,
            ctrX = this->fulX + this->fMaxX/2,
            ulY  = ctrY - dialogLines / 2,
            ulX  = ctrX - dialogCols / 2 ;
   attr_t   dColor  = this->cs.sb,              // window background color
            pnColor = this->cs.pn,              // pushbutton non-focus color
            pfColor = this->cs.pf,              // pushbutton focus color
            rnColor = dColor,                   // radio button non-focus color
            rfColor = this->cs.tf,              // radio button focus color
            inactiveColor = this->cs.sd ;       // inactive radio button color
   enum spoControls : short 
   {  okPB = ZERO, canPB,        // pushbuttons
      psrcRB, pslnRB, phlnRB,    // source radio-button group
      nsrcRB, nnewRB, ntrgRB,    // name radio-button group
      srcTB, trgTB,              // name text 
      spoCONTROLS                // number of controls
   } ;
   short XorSrcGroup[4]  = { psrcRB, pslnRB, phlnRB, -1 } ;
   short XorNameGroup[4] = { nsrcRB, nnewRB, -1, -1 } ;
   psSubType pasteOption = psstABORT ;          // return value

   //* Set the 'inactive control' color attribute from extended color map *
   if ( nc.gybl != ZERO )
   {
      if      ( this->cs.sd == nc.blR )   inactiveColor = nc.gybl ;
      else if ( this->cs.sd == nc.bk  )   inactiveColor = nc.gybk ;
      else if ( this->cs.sd == nc.reR )   inactiveColor = nc.gyre ;
      else if ( this->cs.sd == nc.grR )   inactiveColor = nc.gygr ;
      else if ( this->cs.sd == nc.brR )   inactiveColor = nc.gybr ;
      else if ( this->cs.sd == nc.maR )   inactiveColor = nc.gyma ;
      else if ( this->cs.sd == nc.cyR )   inactiveColor = nc.gycy ;
      else if ( this->cs.sd == nc.gyR )   inactiveColor = nc.gycy ;
   }

InitCtrl ic[spoCONTROLS] =          // array of dialog control info
{
   {  //* 'OK' pushbutton - - - - - - - - - - - - - - - - - - - - - -     okPB *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(dialogLines - 2),       // ulY:       upper left corner in Y
      short(dialogCols / 3 - 4),    // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      8,                            // cols:      control columns
      "   ^OK   ",                  // dispText:  
      pnColor,                      // nColor:    non-focus color
      pfColor,                      // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "",                           // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[canPB],                   // nextCtrl:  link in next structure
   },
   {  //* 'Cancel' pushbutton - - - - - - - - - - - - - - - - - - --     canPB *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(dialogLines - 2),       // ulY:       upper left corner in Y
      short(dialogCols / 3 * 2 - 4),// ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      8,                            // cols:      control columns
      " ^CANCEL ",                  // dispText:  
      pnColor,                      // nColor:    non-focus color
      pfColor,                      // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "",                           // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[psrcRB],                  // nextCtrl:  link in next structure
   },
   {  //* 'Paste source file' radio button  - - - - - - - - - - - -    psrcRB *
      dctRADIOBUTTON,               // type:      
      rbtS5a,                       // rbSubtype: standard, 5-wide
      true,                         // rbSelect:  initially set
      5,                            // ulY:       upper left corner in Y
      3,                            // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a)
      rnColor,                      // nColor:    non-focus color
      rfColor,                      // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      Labels[1],                    // label:     
      ZERO,                         // labY:      
      6,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[pslnRB],                  // nextCtrl:  link in next structure
   },
   {  //* 'Paste-as-symlink' AKA 'Paste-link-target' - - - - - - -      pslnRB *
      dctRADIOBUTTON,               // type:      
      rbtS5a,                       // rbSubtype: standard, 5-wide
      false,                        // rbSelect:  initially reset
      short(ic[psrcRB].ulY + 1),    // ulY:       upper left corner in Y
      short(ic[psrcRB].ulX),        // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a)
      rnColor,                      // nColor:    non-focus color
      rfColor,                      // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      Labels[7],                    // label:     
      ZERO,                         // labY:      
      6,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[phlnRB],                  // nextCtrl:  link in next structure
   },
   {  //* 'Create hard link" radio button  - - - - - - - - - - - -      phlnRB *
      dctRADIOBUTTON,               // type:      
      rbtS5a,                       // rbSubtype: standard, 5-wide
      false,                        // rbSelect:  initially reset
      short(ic[pslnRB].ulY + 1),    // ulY:       upper left corner in Y
      short(ic[pslnRB].ulX),        // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a)
      rnColor,                      // nColor:    non-focus color
      rfColor,                      // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      Labels[10],                   // label:     
      ZERO,                         // labY:      
      6,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[nsrcRB],                  // nextCtrl:  link in next structure
   },
   {  //* 'Use source name" radio button- - - - - - - - - - - - - -     nsrcRB *
      dctRADIOBUTTON,               // type:      
      rbtS5a,                       // rbSubtype: standard, 5-wide
      false,                        // rbSelect:  initially reset
      short(ic[phlnRB].ulY + 2),    // ulY:       upper left corner in Y
      short(ic[phlnRB].ulX),        // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a)
      rnColor,                      // nColor:    non-focus color
      rfColor,                      // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      Labels[11],                   // label:     
      ZERO,                         // labY:      
      6,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[nnewRB],                  // nextCtrl:  link in next structure
   },
   {  //* 'Use new name" radio button - - - - - - - - - - - - - - -     nnewRB *
      dctRADIOBUTTON,               // type:      
      rbtS5a,                       // rbSubtype: standard, 5-wide
      true,                         // rbSelect:  initially set
      short(ic[nsrcRB].ulY + 1),    // ulY:       upper left corner in Y
      short(ic[nsrcRB].ulX),        // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a)
      rnColor,                      // nColor:    non-focus color
      rfColor,                      // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      Labels[14],                   // label:     
      ZERO,                         // labY:      
      6,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[ntrgRB],                  // nextCtrl:  link in next structure
   },
   {  //* 'Use link target name" radio button   - - - - - - - -   trgnameRADIO *
      dctRADIOBUTTON,               // type:      
      rbtC5,                        // rbSubtype: initially a custom button
      true,                         // rbSelect:  initially reset
      short(ic[nnewRB].ulY + 1),    // ulY:       upper left corner in Y
      short(ic[nnewRB].ulX),        // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      customButton,                 // dispText:  (n/a)
      inactiveColor,                // nColor:    non-focus color
      rfColor,                      // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      Labels[0],                    // label:     
      ZERO,                         // labY:      
      6,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      false,                        // active:    initially inactive
      &ic[srcTB],                   // nextCtrl:  link in next structure
   },
   {  //* "Source name" text box - - - - - - - - - - - - - - - - - - -   srcTB *
      dctTEXTBOX,                   // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(dialogLines - 5),       // ulY:       upper left corner in Y
      11,                           // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      29,                           // cols:      control columns
      srcStats->fName,              // dispText:  initially source name
      dColor,                       // nColor:    non-focus color
      dColor,                       // fColor:    focus color
      #if LINUX_SPECIAL_CHARS != 0
      tbFileLinux,                  // filter: valid filename chars (incl. Linux "special")
      #else    // BASIC FILENAME FILTER
      tbFileName,                   // filter:    valid filename characters
      #endif   // BASIC FILENAME FILTER
      "Copying:",                   // label:     
      ZERO,                         // labY:      
      -9,                           // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      false,                        // active:    display only, cannot gain focus
      &ic[trgTB],                   // nextCtrl:  link in next structure
   },
   {  //* 'Target name" text box - - - - - - - - - - - - - - - - - - -   trgTB *
      dctTEXTBOX,                   // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(dialogLines - 4),       // ulY:       upper left corner in Y
      11,                           // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      29,                           // cols:      control columns
      srcStats->fName,              // dispText:  initially source name
      this->cs.tn,                  // nColor:    non-focus color
      this->cs.tf,                  // fColor:    focus color
      #if LINUX_SPECIAL_CHARS != 0
      tbFileLinux,                  // filter: valid filename chars (incl. Linux "special")
      #else    // BASIC FILENAME FILTER
      tbFileName,                   // filter:    valid filename characters
      #endif   // BASIC FILENAME FILTER
      "To: ",                       // label:     
      ZERO,                         // labY:      
      -4,                           // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    initially active
      NULL,                         // nextCtrl:  link in next structure
   },
} ;

   //* Configure control array parameters to match the type of source file     *
   //* and the operations available for it. See notes in method header.        *
   fmFType fileType = srcStats->fType ;
   bool  brokenLink = false, // 'true' if source is sym-link AND link is broken
         cutPending = this->CutPending () ; // 'true' if delete-after-copy

   if ( fileType == fmREG_TYPE )
   {
      if ( cutPending )
      {
         ic[psrcRB].label = Labels[2] ;
         ic[pslnRB].active = false ;            // no access to pslnRB (but visible)
         ic[pslnRB].nColor = inactiveColor ;
         ic[phlnRB].active = false ;            // no access to phlnRB (but visible)
         ic[phlnRB].nColor = inactiveColor ;
         XorSrcGroup[1] = (-1) ;
      }
      else if ( ! sameFileSys )
      {
         ic[phlnRB].active = false ;            // no access to phlnRB (but visible)
         ic[phlnRB].nColor = inactiveColor ;
         XorSrcGroup[2] = (-1) ;
      }
   }
   else if ( fileType == fmDIR_TYPE )
   {
      ic[pslnRB].label = Labels[8] ;
      ic[phlnRB].active = false ;            // no access to phlnRB
      ic[phlnRB].nColor = inactiveColor ;
      ic[phlnRB].rbSubtype = rbtC5 ;
      ic[phlnRB].dispText = customButton ;
      ic[phlnRB].rbSelect = true ;
      ic[phlnRB].label = Labels[0] ;

      if ( cutPending )
      {
         ic[psrcRB].label = Labels[4] ;
         ic[pslnRB].active = false ;         // no access to pslnRB (but visible)
         ic[pslnRB].nColor = inactiveColor ;
         XorSrcGroup[1] = (-1) ;
      }
      else
      {
         ic[psrcRB].label = Labels[3] ;
         XorSrcGroup[2] = (-1) ;
      }
   }
   else if ( fileType == fmLINK_TYPE )
   {
      //* Retrieve link-target stat informaton *
      gString linkPath ;
      this->fmPtr->CatPathFname ( linkPath, clipBoard_srcPath, srcStats->fName ) ;
      short lTarget = this->fmPtr->GetFileStats ( ltargStats, linkPath, true ) ;
      if ( lTarget != OK && ltargStats.fType == fmTYPES )
         brokenLink = true ;

      ic[phlnRB].active = false ;            // no access to phlnRB
      ic[phlnRB].nColor = inactiveColor ;
      ic[phlnRB].rbSubtype = rbtC5 ;
      ic[phlnRB].dispText = customButton ;
      ic[phlnRB].rbSelect = true ;
      ic[phlnRB].label = Labels[0] ;
      XorSrcGroup[2] = (-1) ;

      ic[nsrcRB].label = Labels[13] ;

      if ( cutPending )
      {
         ic[psrcRB].label = Labels[6] ;
         ic[pslnRB].active = false ;         // no access to pslnRB
         ic[pslnRB].nColor = inactiveColor ;
         ic[pslnRB].rbSubtype = rbtC5 ;
         ic[pslnRB].dispText = customButton ;
         ic[pslnRB].rbSelect = true ;
         ic[pslnRB].label = Labels[0] ;
         XorSrcGroup[1] = (-1) ;
      }
      else
      {
         ic[psrcRB].label = Labels[5] ;
         ic[pslnRB].label = Labels[9] ;
         if ( brokenLink )
         {
            ic[pslnRB].active = false ;
            ic[pslnRB].nColor = inactiveColor ;
            XorSrcGroup[1] = (-1) ;
         }
         else
         {
            ic[psrcRB].rbSelect = false ;
            ic[pslnRB].rbSelect = true ;
            ic[nnewRB].rbSelect = false ;
            ic[ntrgRB].rbSelect = true ;
            ic[ntrgRB].active = true ;
            ic[ntrgRB].nColor = rnColor ;
            ic[ntrgRB].rbSubtype = rbtS5a ;
            ic[ntrgRB].label = Labels[15] ;
            XorNameGroup[2] = ntrgRB ;
            ic[srcTB].dispText = ltargStats.fName ;
            ic[trgTB].dispText = ltargStats.fName ;
            ic[trgTB].active = false ;
         }
      }
   }
   else if ( fileType != fmREG_TYPE )
   {  //* Special files, fmCHDEV_TYPE, fmBKDEV_TYPE, *
      //* fmSOCK_TYPE, fmFIFO_TYPE, fmUNKNOWN_TYPE   *
      if ( cutPending )
      {
         ic[psrcRB].label = Labels[2] ;
         ic[pslnRB].active = false ;            // no access to pslnRB (but visible)
         ic[pslnRB].nColor = inactiveColor ;
         ic[phlnRB].active = false ;            // no access to phlnRB (but visible)
         ic[phlnRB].nColor = inactiveColor ;
         XorSrcGroup[1] = (-1) ;
      }
      else if ( ! sameFileSys )
      {
         ic[phlnRB].active = false ;            // no access to phlnRB (but visible)
         ic[phlnRB].nColor = inactiveColor ;
         XorSrcGroup[2] = (-1) ;
      }
   }

   //* Save the display for the parent dialog window *
   this->dPtr->SetDialogObscured () ;

   //* Initial parameters for dialog window *
   InitNcDialog dInit( dialogLines,    // number of display lines
                       dialogCols,     // number of display columns
                       ulY,            // Y offset from upper-left of terminal 
                       ulX,            // X offset from upper-left of terminal 
                       "  Paste Special  ",// dialog title
                       ncltSINGLE,     // border line-style
                       dColor,         // border color attribute
                       dColor,         // interior color attribute
                       ic              // pointer to list of control definitions
                     ) ;

   //* Instantiate the dialog window *
   NcDialog* dp = new NcDialog ( dInit ) ;

   //* Open the dialog window *
   if ( (dp->OpenWindow()) == OK )
   {
      LineDef  lDefH( ncltHORIZ, ncltSINGLE, (dialogLines - 6), ZERO, 
                      dialogCols, dColor ) ;
      dp->DrawLine ( lDefH ) ;

      //* Write the static text *
      winPos wp( 1, 2 ) ;
      wp = dp->WriteString ( wp.ypos,  wp.xpos, "Source file: ", dColor ) ;
      gString dispSrc ;
      dispSrc.compose( L" %s ", srcStats->fName ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, dispSrc, dColor | ncuATTR ) ;
      wp = dp->WriteString ( wp.ypos, (wp.xpos=2), "  File type: ", dColor ) ;
      wp = dp->WriteString ( wp.ypos, wp.xpos, fmtypeName[srcStats->fType], dColor ) ;
      if ( brokenLink )
         dp->WriteString ( wp.ypos, wp.xpos, " (link target not found)", dColor ) ;
      dp->WriteString ( 4, 2, "Specify type for new file:", dColor ) ;
      dp->WriteString ( 8, 2, "Specify name for new file:", dColor ) ;

      //* Set radio buttons as an exclusive-OR group *
      //* (only one can be selected at any moment).  *
      if ( (dp->GroupRadiobuttons ( XorSrcGroup )) != OK )
         ;//this->dPtr->DebugMsg ( "XorSrcGroup failed...", 3, true ) ;
      if ( (dp->GroupRadiobuttons ( XorNameGroup )) != OK )
         ;//this->dPtr->DebugMsg ( "XorNameGroup failed...", 3, true ) ;

      //* Adjust color of labels for inactive radio buttons *
      if ( ic[pslnRB].active == false )
         dp->WriteString ( ic[pslnRB].ulY + ic[pslnRB].labY,
                           ic[pslnRB].ulX + ic[pslnRB].labX,
                           ic[pslnRB].label, inactiveColor ) ;
      if ( ic[phlnRB].active == false )
         dp->WriteString ( ic[phlnRB].ulY + ic[phlnRB].labY,
                           ic[phlnRB].ulX + ic[phlnRB].labX,
                           ic[phlnRB].label, inactiveColor ) ;

      //* Enable audible alert for invalid input *
      dp->TextboxAlert ( trgTB, true ) ;
      //* Set text box input mode to 'insert' *
      dp->SetTextboxInputMode ( false ) ;
      //* If 'Enter new name' is default, move focus to text box for user input*
      short    icIndex = ZERO ;     // index of control with input focus
      if ( ic[nnewRB].active && ic[nnewRB].rbSelect )
         icIndex = dp->PrevControl () ;


      dp->RefreshWin () ;           // make everything visible

      //************************
      //* Get user's selection *
      //************************
      uiInfo   Info ;               // user interface data returned here
      bool     done = false ;       // loop control
      while ( ! done )
      {
         if ( ic[icIndex].type == dctPUSHBUTTON )
         {
            if ( Info.viaHotkey == false )
               icIndex = dp->EditPushbutton ( Info ) ;
            else
               Info.HotData2Primary () ;
            if ( Info.dataMod != false )
            {
               //* User has selected 'paste special' options - huzzah! *
               if ( Info.ctrlIndex == okPB )
               {
                  //* Get the source-file radio-button-group's *
                  //* selection and set the copy/move option.  *
                  short src = dp->GetRbGroupSelection ( psrcRB ) ;
                  if ( srcStats->fType == fmLINK_TYPE )
                  {  //* Source is a symbolic link *
                     if ( src == psrcRB ) pasteOption = psstSimpleCopy ;
                     else                 pasteOption = psstCopyLinkTarget ;
                  }
                  else
                  {  //* Source is 'regular', 'directory', or 'special' *
                     if ( src == psrcRB )       pasteOption = psstSimpleCopy ;
                     else if ( src == pslnRB )  pasteOption = psstCreateSymLink ;
                     else                       pasteOption = psstCreateHardLink ;
                  }

                  //* Copy user's choice of target filename *
                  //* to caller's gString object.           *
                  dp->GetTextboxText ( trgTB, trgFName ) ;

                  //* If target directory == source directory, AND source      *
                  //* name == target name, then we have some variety of a file *
                  //* being copied onto itself, which is not possible.         *
                  if ( tDirEQsDir && 
                       !(strncmp ( trgFName.ustr(), srcStats->fName, MAX_FNAME )) )
                  {
                     wp = { short(dialogLines - 3), 2 } ; 
                     dp->WriteString ( wp.ypos, wp.xpos, 
                     " CANNOT COPY FILE ONTO ITSELF, MODIFY TARGET NAME. ", 
                     pfColor, true ) ;
                     chrono::duration<short>aWhile( 4 ) ;
                     this_thread::sleep_for( aWhile ) ;
                     dp->ClearLine ( wp.ypos, false ) ;
                     //* Set the enter-new-filename option and place focus on  *
                     //* target filename control.                              *
                     dp->SetRbGroupSelection ( nnewRB ) ;
                     dp->ControlActive ( trgTB, true ) ;
                     while ( (icIndex = dp->PrevControl ()) != trgTB ) ;
                     icIndex = dp->PrevControl () ;
                     pasteOption = psstABORT ;
                  }

                  #if LINUX_SPECIAL_CHARS != 0
                  else if ( !(this->gnfSpecialCharsQuery ( trgFName, dp, trgTB, tbFileName )) )
                  {  //* Place focus on target filename control.*
                     while ( (icIndex = dp->PrevControl ()) != trgTB ) ;
                     icIndex = dp->PrevControl () ;
                     pasteOption = psstABORT ;
                  }
                  #endif   // LINUX_SPECIAL_CHARS
                  else     // success!!
                     done = true ;
               }
               //* Cancel the operation *
               else if ( Info.ctrlIndex == canPB )
               { done = true ; }
            }
            else if ( Info.keyIn == nckESC )
            {  //* User panic, abort the operation *
               done = true ;
            }
         }
         else if ( ic[icIndex].type == dctRADIOBUTTON )
         {
            if ( Info.viaHotkey )
               Info.HotData2Primary () ;
            else
               icIndex = dp->EditRadiobutton ( Info ) ;

            if ( Info.dataMod )     // if there was a change
            {
               if ( Info.ctrlIndex <= phlnRB )  // source group
               {
                  //* Display the corresponding source-file name *
                  if ( Info.selMember == pslnRB
                       && srcStats->fType == fmLINK_TYPE && !brokenLink )
                     dp->SetTextboxText ( srcTB, ltargStats.fName ) ;
                  else
                     dp->SetTextboxText ( srcTB, srcStats->fName ) ;
               }
               else                             // name group
               {
                  //* Display the corresponding target-file name *
                  if ( Info.selMember == nnewRB )
                     dp->ControlActive ( trgTB, true ) ; // enable name editing
                  else
                  {
                     dp->ControlActive ( trgTB, false ) ;// disable name editing
                     if ( Info.selMember == ntrgRB )     // link-target name
                        dp->SetTextboxText ( trgTB, ltargStats.fName ) ;
                     else                                // source name
                        dp->SetTextboxText ( trgTB, srcStats->fName ) ;
                  }
               }
            }
         }

         else if ( ic[icIndex].type == dctTEXTBOX )
         {
            Info.viaHotkey = false ;      // ignore hotkey data
            icIndex = dp->EditTextbox ( Info ) ;
         }

         //* Move focus to appropriate control *
         if ( !done && !Info.viaHotkey )
         {
            if ( Info.keyIn == nckSTAB )
               icIndex = dp->PrevControl () ; 
            else
               icIndex = dp->NextControl () ;
         }
      }     // while(!done)
   }
   else           // dialog did not open (probably memory allocation error)
      this->DisplayStatus ( ecMEMALLOC ) ;

   if ( dp != NULL )                         // close the window
      delete ( dp ) ;

   //* Restore display of parent dialog window *
   this->dPtr->RefreshWin () ;

   return pasteOption ;

}  //* End pslSinglePaste_Options() *

//***********************
//*    pslMultiPaste    *
//***********************
//******************************************************************************
//* There are multiple files, and/or directories and/or directory trees on the *
//* clipboard, and user must select the desired 'Paste-Special' option before  *
//* we perform the copy/move operation.                                        *
//*                                                                            *
//* Multi-object Paste Special is limited to directory-trees and 'regular'     *
//* files. If 'special' files live at the top level of the clipboard list, we  *
//* refuse to process the list, and the user is warned to process 'special'    *
//* files individually. (Note that 'special' files may live within a lower     *
//* level of a selected directory tree; because all files of a directory tree  *
//* below the directory name itself are copied/moved as psstSimpleCopy.)       *
//*                                                                            *
//* In the case of a cut-and-paste operation where target directory == source  *
//* directory, we refuse to process the list because that is actually a        *
//* 'Rename' operation. NOTE: The code would actually support this operation,  *
//* but if offends me that the user would request it.                          *
//*                                                                            *
//* Input  : pcl : (by reference)                                              *
//*                pcl.srcDir  == directory path of source file(s)             *
//*                pcl.trgDir  == directory path to paste target               *
//*                pcl.srcTree == points to base node of clipboard tree        *
//*                pcl.attCount: on return, contains number of copy/move       *
//*                 operations actually attempted, usually the number of       *
//*                 files on the clipboard                                     *
//*                pcl.delCount: initialized to ZERO, but if                   *
//*                 delete-after-copy operation, receives number of files      *
//*                 successfully deleted after copy,                           *
//*          tDirEQsDir: if 'true' target base dir == source base dir          *
//*          sameFileSys: if 'true', source directory and target directory     *
//*                         are on the same file system                        *
//*                       if 'false', source and target are on different file  *
//*                         systems                                            *
//*                                                                            *
//* Returns: number of files copied/moved/created                              *
//*          if non-directory/non-regular files have been specified, OR        *
//*           if user aborts, then on return pcl.psOption == psstABORT         *
//******************************************************************************

UINT FileDlg::pslMultiPaste ( pclParm& pcl, bool tDirEQsDir, bool sameFileSys )
{
   UINT  pasteCount = ZERO ;  // number of files successfully pasted to target
   pcl.attCount = ZERO ;      // number of copy/move/create operations attempted
   pcl.delCount = ZERO ;      // number of source files successfully deleted
   pcl.deleteSrc = this->CutPending () ; // 'true' if delete-after-copy
   //* If target dir == source dir AND this is a cut-and-paste.(see note above)*
   bool renameOperation = ((tDirEQsDir != false) && (pcl.deleteSrc != false) ) ;

   //* Determine whether all selected top-level, non-directory *
   //* files (if any) are 'regular' files.                     *
   UINT ndfCount = clipBoard[BNODE].tnFCount ;  // non-dir file count
   tnFName* ndfPtr = clipBoard[BNODE].tnFiles ; // pointer to non-dir file array
   bool ndfAllReg = true ;    // true if all non-dir files are 'regular' files
   if ( ndfPtr != NULL && ndfCount > ZERO )
   {
      for ( UINT i = ZERO ; i < ndfCount ; i++ )
      {
         if ( ndfPtr[i].fType != fmREG_TYPE )
         {
            ndfAllReg = false ;
            break ;
         }
      }
   }
   if ( ndfAllReg != false && renameOperation == false )
   {
      renamePattern* rPat = NULL ;  // for batch rename option
      bool  rPrompt = false ;       // for individual rename option

      //* Ask user to select 'Paste-Special' options *
      renameOption rOpt ;           // renaming option
      pcl.psOption = pslMultiPaste_Options ( rOpt, ndfCount, clipBoard_dirTrees, 
                                             clipBoard[BNODE].totFiles, 
                                             clipBoard[BNODE].totBytes, 
                                             tDirEQsDir, sameFileSys ) ;
      if ( pcl.psOption != psstABORT )
      {  //* Indicate prompt for each filename *
         if ( rOpt == promptRENAME )
            rPrompt = true ;
         //* Initialize batch-rename pattern generator *
         else if ( rOpt == batchRENAME )
         {  // Programmer's Note: Size of the rsfConstructPattern() dialog is
            // fixed in rsfGetRenamePattern(). We are only 'borrowing' the
            // dialog from the rename module, so we just duplicate them here.
            // This is poor programming practice, but save a lot of code.
            const short dialogLines = 12, dialogCols = 74 ; 
            short ctrY = this->fulY + this->fMaxY/2,
                  ctrX = (this->dCols + this->dulX) / 2, // center across application dialog
                  ulY  = ctrY - dialogLines / 2,
                  ulX  = ctrX - dialogCols / 2 ;
            rPat = new renamePattern ;
            if ( tDirEQsDir ) // to avoid source overwrite, suggest a pattern
            {
               this->fmPtr->strnCpy ( rPat->inStr, L"(2)", 4 ) ;  // 'suggested' target name
               rPat->in = true ;
            }
            //* Save the display for the parent dialog window *
            this->dPtr->SetDialogObscured () ;

            //* Ask user to create the batch-rename pattern *
            if ( (rOpt = this->rsfConstructPattern ( 
                     rPat, dialogLines, dialogCols, ulY, ulX )) != batchRENAME )
               pcl.psOption = psstABORT ;

            //* Restore display of parent dialog window *
            this->dPtr->RefreshWin () ;
         }
      }
      if ( pcl.psOption != psstABORT )
      {
         //* Display a message that lets the user know we're working *
         this->ProcessCWD () ;

         gString  srcPath,          // source path/filename string
                  trgPath,          // target path/filename string
                  newFName ;        // new targetfile name
         tnFName* srcStats ;        // source-file information
         corReturn coStatus = corProcessOnce ; // status of user interactions

         //********************************
         //* Copy the directory sub-trees *
         //********************************
         for ( UINT dIndex = ZERO ; dIndex < clipBoard_dirTrees ; dIndex++ )
         {
            //* Point to stats for source file *
            srcStats = &pcl.srcTree->nextLevel[dIndex].dirStat ;

            //* Path/filename string for source file *
            this->fmPtr->CatPathFname ( srcPath, pcl.srcDir.ustr(), srcStats->fName ) ;
      
            //* Path/filename string for target file *
            if ( rOpt != noRENAME )
            {
               coStatus = this->pclGetNewFilename ( srcStats, newFName, tDirEQsDir ) ;
               if ( coStatus == corProcessOnce )
                  this->fmPtr->CatPathFname ( trgPath, 
                                              pcl.trgDir.gstr(), newFName.gstr() ) ;
            }
            else  // (renOption==batchRENAME || renOption==noRENAME)
               this->fmPtr->CatPathFname ( trgPath, 
                                           pcl.trgDir.ustr(), srcStats->fName ) ;

            //* Determine whether target already exists, is write protected,   *
            //* etc. and prompt user for a decision if necessary.              *
            if ( coStatus != corAbort )
            {
               coStatus = 
                  this->pslTargetExists ( trgPath, srcPath, srcStats, pcl.psOption ) ;
               ++pcl.attCount ;           // operation attempted

               //* If target does not already exist OR if we have user's *
               //* permission to overwrite existing target, then proceed.*
               if ( coStatus == corProcessOnce || coStatus == corProcessAll )
               {
                  short status = 
                     this->pslPasteSpecial ( pcl.psOption, srcStats, srcPath, trgPath ) ;
                  if ( status == OK )
                  {
                     ++pasteCount ;          // copy successful
                     bool ok2delete = true ; // provisional go-ahead for delete-after-copy

                     //* If directory is not empty, AND if the target is *
                     //* really a directory file, copy its contents.     *
                     if ( pcl.psOption != psstCreateSymLink )
                     {
                        UINT pCount = 1 ;    // successful pastes for this node
                        pclParm pcl2( srcPath.ustr(), trgPath.ustr(), 
                                      &pcl.srcTree->nextLevel[dIndex], 
                                      pcl.psOption, pcl.deleteSrc ) ;
                        if ( pcl2.srcTree->totFiles > ZERO )
                        {
                           pCount = this->pslCopyTnList ( pcl2 ) ;
                           pasteCount += pCount ;
                           pcl.attCount += pcl2.attCount ;
                           pcl.cpyCount += pcl2.cpyCount ;
                           pcl.delCount += pcl2.delCount ;
                           //* If delete-after-paste operation AND all files in   *
                           //* source node copied/deleted successfully, delete    *
                           //* the source directory name.                         *
                           if ( pcl.deleteSrc && 
                               !(pcl2.attCount == pCount) && (pcl2.delCount == pCount) )
                              ok2delete = false ;
                        }
                     }
                     if ( pcl.deleteSrc && ok2delete )
                     {
                        if ( (this->DeleteSourceFile ( srcStats, srcPath )) == OK )
                           ++pcl.delCount ;
                     }
                  }
                  else
                  {  //* Report error copying the file *
                     this->DisplayErrorCode () ;
                  }
               }
            }
            //* If user has aborted the process
            if ( coStatus == corAbort )
            {
               pcl.psOption = psstABORT ;
               break ;
            }
         }  // for(;nodeIndex < clipBoard_dirTrees)

         //************************************
         //* Copy the top-level non-dir files *
         //************************************
         if ( pcl.psOption != psstABORT )
         {
            for ( UINT fIndex = ZERO ; fIndex < ndfCount && coStatus != corAbort ; fIndex++ )
            {
               coStatus = corProcessOnce ;   // hope for the best

               //* Point to stats for source file *
               srcStats = &ndfPtr[fIndex] ;
   
               //* Path/filename string for source file *
               this->fmPtr->CatPathFname ( srcPath, 
                                           pcl.srcDir.ustr(), srcStats->fName ) ;
      
               //* Create target file path/filename string *
               if ( rPrompt != false ) // prompt user for target filename
               {
                  coStatus = this->pclGetNewFilename ( srcStats, newFName, tDirEQsDir ) ;
                  if ( coStatus == corProcessOnce )
                     this->fmPtr->CatPathFname ( trgPath, 
                                                 pcl.trgDir.gstr(), newFName.gstr() ) ;
                  else
                     coStatus = corSkipOnce ;
               }
               else if ( rPat != NULL )// create target name from batch-rename pattern
               {
                  rPat->formatFName ( newFName, srcStats->fName ) ;
                  //* If target directory == source directory, check to be     *
                  //* sure that we are not going to overwrite a source file.   *
                  if ( tDirEQsDir )
                  {  //* If the new target name IS in use by a file on the     *
                     //* clipboard, attempt to increment the sequence number   *
                     //* or otherwise modify the target name. If we fail,      *
                     //* abort, rather than wipe out a source file.            *
                     bool dummy = false ; // (not used here)
                     if ( (this->pclFNameInUse ( newFName, dummy )) != false )
                        this->pclCreateSuggestedName ( newFName, newFName ) ;
                     if ( (this->pclFNameInUse ( newFName, dummy )) != false )
                     {
                        this->SetErrorCode ( ecTARGEXISTS, EEXIST ) ;// set error code to report
                        coStatus = corAbort ;      // don't process this file
                        pcl.psOption = psstABORT ; // terminate the loop
                     }
                  }
                  this->fmPtr->CatPathFname ( trgPath, 
                                              pcl.trgDir.gstr(), newFName.gstr() ) ;
               }
               else                    // target file name == source file name
                  this->fmPtr->CatPathFname ( trgPath, 
                                              pcl.trgDir.ustr(), srcStats->fName ) ;

               //* Determine whether target already exists, is write protected,*
               //* etc. and prompt user for a decision if necessary.           *
               if ( coStatus == corProcessOnce )
               {
                  coStatus = this->pslTargetExists ( trgPath, srcPath, 
                                                     srcStats, pcl.psOption ) ;
               }
               ++pcl.attCount ;              // operation attempted

               //* If target does not already exist OR if we have user's *
               //* permission to overwrite existing target, then proceed.*
               if ( coStatus == corProcessOnce || coStatus == corProcessAll )
               {
                  short status = 
                     this->pslPasteSpecial ( pcl.psOption, srcStats, srcPath, trgPath ) ;
                  if ( status == OK )
                  {
                     ++pasteCount ;          // copy successful
         
                     //* If delete-after-copy operation, *
                     //* delete the source file.         *
                     if ( pcl.deleteSrc )
                     {
                        if ( (this->DeleteSourceFile ( srcStats, srcPath )) == OK )
                           ++pcl.delCount ;
                     }
                  }
                  else
                  {  //* Report error copying the file *
                     this->DisplayErrorCode () ;
                  }
               }
            }  // for(; fIndex < ndfCount && coStatus != corAbort ;)
         }  // if(pcl.psOption != psstABORT)
      }  // if(pcl.psOption != psstABORT)

      //* Release dynamic allocation *
      if ( rPat != NULL )
         delete rPat ;
   }  // if(ndfAllReg && !renameOperation )

   else if ( ! ndfAllReg )
   {
      const char* msg[] = 
      { //1234567890123456789012345678901234567890123456789012 - (max line length)
         "  WARNING!  WARNING!  ",
         "One or more of the files you have selected for",
         "processing is a 'special' file (SYM-LINK, FIFO,",
         "SOCKET, CHAR or BLOCK DEVICE, or is an unknown type.",
         " ",
         "Multi-file 'Paste Special' operates only on Regular",
         "files and directory trees.",
         "   Please process 'special' files individually.",
         NULL
      } ;
      this->InfoDialog ( msg ) ;
      pcl.psOption = psstABORT ;
   }
   else  // renameOperation: an unlikely but annoying scenario
   {
      const char* msg[] = 
      { //1234567890123456789012345678901234567890123456789012 - (max line length)
         "  WARNING!  WARNING!  ",
         " You have requested a cut-and-paste operation, but",
         " the target directory equals the source directory.",
         " ",
         "      Please use 'Rename' operation instead.",
         NULL
      } ;
      this->InfoDialog ( msg ) ;
      pcl.psOption = psstABORT ;
   }

   return pasteCount ;

}  //* End pslMultiPaste() *

//*****************************
//*   pslMultiPaste_Options   *
//*****************************
//******************************************************************************
//* Called by pslMultiPaste() if there are multiple source files and/or        *
//* multiple subdirectory trees to paste.                                      *
//*                                                                            *
//* Ask user to select from available Paste-Special options.                   *
//*                                                                            *
//*                                                                            *
//* Input  : renOption : (by reference, initial value ignored) receives member *
//*                        of enum renameOption for naming target files        *
//*          regCount  : number of 'regular' files in source list              *
//*          dirCount  ; number of directory trees in source list              *
//*          totCount  : total number of files to be processed                 *
//*          totBytes  : total byte count for files to be processed            *
//*          tDirEQsDir: if 'true', target base dir == source base dir         *
//*          sameFileSys: if 'true', source directory and target directory     *
//*                         are on the same file system                        *
//*                       if 'false', source and target are on different file  *
//*                         systems                                            *
//*                                                                            *
//* Returns: member of psSubType                                               *
//******************************************************************************
//* Paste Special Options for multi-file operations:                           *
//* Only directory trees and 'regular' top-level files may be selected for     *
//* this operation. 'Special' files are not handled in multi-file operations.  *
//* Special files must be processed individually because the available options *
//* are different for each file type. (user is warned, and operation aborts)   *
//*                                                                            *
//* A. Copy-and-Paste                                                          *
//*    1. Directory names: process each dir name individually (batch == n/a)   *
//*       a. create sym-link  (psstCreateSymLink)                              *
//*          - using source name (except when target dir == source dir)        *
//*          - with individual rename                                          *
//*       b. (note: cannot create a 'hard link' to a directory)                *
//*       c. copy the source directory (psstSimpleCopy)                        *
//*          - using source name (except when target dir == source dir)        *
//*          - with individual rename                                          *
//*          if directory is not empty, copy its contents to target with a     *
//*          simple copy using source file names                               *
//*    2. Regular files:                                                       *
//*       a. create sym-links  (psstCreateSynLink)                             *
//*          - using source name (except when target dir == source dir)        *
//*          - with individual rename                                          *
//*          - with batch rename                                               *
//*       b. create hard links (psstCreateHardLink)                            *
//*          - using source name (except when target dir == source dir)        *
//*          - with individual rename                                          *
//*          - with batch rename                                               *
//*       c. copy the source files (psstSimpleCopy)                            *
//*          - using source name (except when target dir == source dir)        *
//*          - with individual rename                                          *
//*          - with batch rename                                               *
//*                                                                            *
//* B. Cut-and-Paste  (target dir must be different from source dir)           *
//*    1. Directory names: process each dir name individually (batch == n/a)   *
//*       a. copy the source directory (psstSimpleCopy)                        *
//*          - using source name                                               *
//*          - with individual rename                                          *
//*          if directory is not empty, copy its contents to target with a     *
//*          simple copy using source file names                               *
//*    2. Regular files:                                                       *
//*       a. copy the source files (psstSimpleCopy)                            *
//*          - using source name                                               *
//*          - with individual rename                                          *
//*          - with batch rename                                               *
//*                                                                            *
//* C. For a single source file or single directory tree,                      *
//*    see pslSinglePaste_Options().                                           *
//*                                                                            *
//******************************************************************************

psSubType FileDlg::pslMultiPaste_Options ( renameOption& renOption, UINT regCount, 
                                           UINT dirCount, UINT totCount, UINT64 totBytes, 
                                           bool tDirEQsDir, bool sameFileSys )
{
const char* Labels[] = 
{                                                              // INDEX
   " ",                                                        // (0)
   "Copy the source files",                     // psrcRB labels  (1)
   "Move the source files",                                    // (2)

   "Create symbolic links to source files",     // pslnRB labels  (3)

   "Create 'hard' links to source files",       // phlnRB labels  (4)

   "Use the source file names",                 // nsrcRB labels  (5)

   "Each file to be renamed individually",      // nindRB labels  (6)

   "Batch rename files using pattern (non-dirs)"// nbatRB labels  (7)

} ;
const char* customButton = "....." ;
const short dialogROWS = 18,                 // dialog lines
            dialogCOLS  = INFODLG_WIDTH ;    // dialog columns
short    ctrY = this->fulY + this->fMaxY/2,
         ctrX = this->fulX + this->fMaxX/2,
         ulY  = ctrY - dialogROWS / 2,
         ulX  = ctrX - dialogCOLS / 2 ;
attr_t   dColor  = this->cs.sb,              // window background color
         pnColor = this->cs.pn,              // pushbutton non-focus color
         pfColor = this->cs.pf,              // pushbutton focus color
         rnColor = dColor,                   // radio button non-focus color
         rfColor = this->cs.tf,              // radio button focus color
         inactiveColor = this->cs.sd ;       // inactive radio button color
enum mpoControls : short 
{  okPB = ZERO, canPB,        // pushbuttons
   psrcRB, pslnRB, phlnRB,    // source radio-button group
   nsrcRB, nindRB, nbatRB,    // name radio-button group
   mpoCONTROLS                // number of controls
} ;
short XorSrcGroup[4]  = { psrcRB, pslnRB, phlnRB, -1 } ;
short XorNameGroup[4] = { nsrcRB, nindRB, nbatRB, -1 } ;
psSubType pasteOption = psstABORT ;          // return value
          renOption = abortRENAME ;          // plan for the worst

   //* Set the 'inactive control' color attribute from extended color map *
   if ( nc.gybl != ZERO )
   {
      if      ( this->cs.sd == nc.blR )   inactiveColor = nc.gybl ;
      else if ( this->cs.sd == nc.bk  )   inactiveColor = nc.gybk ;
      else if ( this->cs.sd == nc.reR )   inactiveColor = nc.gyre ;
      else if ( this->cs.sd == nc.grR )   inactiveColor = nc.gygr ;
      else if ( this->cs.sd == nc.brR )   inactiveColor = nc.gybr ;
      else if ( this->cs.sd == nc.maR )   inactiveColor = nc.gyma ;
      else if ( this->cs.sd == nc.cyR )   inactiveColor = nc.gycy ;
      else if ( this->cs.sd == nc.gyR )   inactiveColor = nc.gycy ;
   }

InitCtrl ic[mpoCONTROLS] =          // array of dialog control info
{
   {  //* 'OK' pushbutton - - - - - - - - - - - - - - - - - - - - - -     okPB *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(dialogROWS - 2),        // ulY:       upper left corner in Y
      short(dialogCOLS / 3 - 4),    // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      8,                            // cols:      control columns
      "   ^OK   ",                  // dispText:  
      pnColor,                      // nColor:    non-focus color
      pfColor,                      // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "",                           // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[canPB],                   // nextCtrl:  link in next structure
   },
   {  //* 'Cancel' pushbutton - - - - - - - - - - - - - - - - - - --     canPB *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(dialogROWS - 2),        // ulY:       upper left corner in Y
      short(dialogCOLS / 3 * 2 - 4),// ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      8,                            // cols:      control columns
      " ^CANCEL ",                  // dispText:  
      pnColor,                      // nColor:    non-focus color
      pfColor,                      // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "",                           // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[psrcRB],                  // nextCtrl:  link in next structure
   },
   {  //* 'Copy/Move source files' radio button  - - - - - - - - - -    psrcRB *
      dctRADIOBUTTON,               // type:      
      rbtS5a,                       // rbSubtype: standard, 5-wide
      true,                         // rbSelect:  initially set
      5,                            // ulY:       upper left corner in Y
      3,                            // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      Labels[1],                    // dispText:  (n/a)
      rnColor,                      // nColor:    non-focus color
      rfColor,                      // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      Labels[1],                    // label:     
      ZERO,                         // labY:      
      6,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[pslnRB],                  // nextCtrl:  link in next structure
   },
   {  //* 'Create-symlinks' radio button - - - - - - - - - - - - -      pslnRB *
      dctRADIOBUTTON,               // type:      
      rbtS5a,                       // rbSubtype: standard, 5-wide
      false,                        // rbSelect:  initially reset
      short(ic[psrcRB].ulY + 1),    // ulY:       upper left corner in Y
      short(ic[psrcRB].ulX),        // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a)
      rnColor,                      // nColor:    non-focus color
      rfColor,                      // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      Labels[3],                    // label:     
      ZERO,                         // labY:      
      6,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[phlnRB],                  // nextCtrl:  link in next structure
   },
   {  //* 'Create hard links' radio button - - - - - - - - - - - -      phlnRB *
      dctRADIOBUTTON,               // type:      
      rbtS5a,                       // rbSubtype: standard, 5-wide
      false,                        // rbSelect:  initially reset
      short(ic[pslnRB].ulY + 1),    // ulY:       upper left corner in Y
      short(ic[pslnRB].ulX),        // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a)
      rnColor,                      // nColor:    non-focus color
      rfColor,                      // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      Labels[4],                    // label:     
      ZERO,                         // labY:      
      6,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[nsrcRB],                  // nextCtrl:  link in next structure
   },
   {  //* 'Use source names' radio button - - - - - - - - - - - - -     nsrcRB *
      dctRADIOBUTTON,               // type:      
      rbtS5a,                       // rbSubtype: standard, 5-wide
      false,                        // rbSelect:  initially reset
      short(ic[phlnRB].ulY + 3),    // ulY:       upper left corner in Y
      short(ic[phlnRB].ulX),        // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a)
      rnColor,                      // nColor:    non-focus color
      rfColor,                      // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      Labels[5],                    // label:     
      ZERO,                         // labY:      
      6,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[nindRB],                  // nextCtrl:  link in next structure
   },
   {  //* 'Individual Rename' radio button  - - - - - - - - - - - -     nindRB *
      dctRADIOBUTTON,               // type:      
      rbtS5a,                       // rbSubtype: standard, 5-wide
      true,                         // rbSelect:  initially set
      short(ic[nsrcRB].ulY + 1),    // ulY:       upper left corner in Y
      short(ic[nsrcRB].ulX),        // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a)
      rnColor,                      // nColor:    non-focus color
      rfColor,                      // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      Labels[6],                    // label:     
      ZERO,                         // labY:      
      6,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[nbatRB],                  // nextCtrl:  link in next structure
   },
   {  //* 'Batch Rename' radio button - - - - - - - - - - - - - - - -   nbatRB *
      dctRADIOBUTTON,               // type:      
      rbtS5a,                       // rbSubtype: standard, 5-wide
      false,                        // rbSelect:  initially reset
      short(ic[nindRB].ulY + 1),    // ulY:       upper left corner in Y
      short(ic[nindRB].ulX),        // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a)
      rnColor,                      // nColor:    non-focus color
      rfColor,                      // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      Labels[7],                    // label:     
      ZERO,                         // labY:      
      6,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    initially inactive
      NULL,                         // nextCtrl:  link in next structure
   },
} ;

   //* Configure control array parameters to match the type of source files    *
   //* and the operations available for them. See notes in method header.      *
   if ( this->CutPending () )
   {  
      //* Change label from 'Copy' to 'Move' *
      ic[psrcRB].label = Labels[2] ;
      //* If a cut-and-paste operation is in progress, both     *
      //* the sym-link and hard-link options are disabled.      *
      ic[pslnRB].active = false ;
      ic[pslnRB].rbSubtype = rbtC5 ;
      ic[pslnRB].dispText = customButton ;
      ic[pslnRB].rbSelect = true ;
      ic[pslnRB].nColor = inactiveColor ;
      ic[phlnRB].active = false ;
      ic[phlnRB].rbSubtype = rbtC5 ;
      ic[phlnRB].dispText = customButton ;
      ic[phlnRB].rbSelect = true ;
      ic[phlnRB].nColor = inactiveColor ;
      XorSrcGroup[1] = XorSrcGroup[2] = (-1) ;
   }
   else if ( (dirCount > ZERO) || ! sameFileSys )
   {  //* Cannot create 'hard' links to directory names, *
      //* and/or cannot create hard link across          *
      //* filesystems, so disable the hard-link option.  *
      ic[phlnRB].active = false ;
      ic[phlnRB].rbSubtype = rbtC5 ;
      ic[phlnRB].dispText = customButton ;
      ic[phlnRB].rbSelect = true ;
      ic[phlnRB].nColor = inactiveColor ;
      XorSrcGroup[2] = (-1) ;
   }
   if ( tDirEQsDir != false )
   {  //* If source dir == target dir, then source-file names   *
      //* cannot be used, so disable the use-source-name option.*
      ic[nsrcRB].active = false ;
      ic[nsrcRB].rbSubtype = rbtC5 ;
      ic[nsrcRB].dispText = customButton ;
      ic[nsrcRB].rbSelect = true ;
      ic[nsrcRB].nColor = inactiveColor ;
      XorNameGroup[0] = nindRB ;
      XorNameGroup[1] = nbatRB ;
      XorNameGroup[2] = (-1) ;
   }
   if ( regCount < 2 )
   {  //* Only directory trees selected, OR only one non-directory *
      //* file selected, so batch rename does not apply.           *
      ic[nbatRB].active = false ;
      ic[nbatRB].rbSubtype = rbtC5 ;
      ic[nbatRB].dispText = customButton ;
      ic[nbatRB].rbSelect = true ;
      ic[nbatRB].nColor = inactiveColor ;
      XorNameGroup[tDirEQsDir ? 1 : 2] = (-1) ;
   }

   //* Save the display for the parent dialog window *
   this->dPtr->SetDialogObscured () ;

   //* Initial parameters for dialog window *
   InitNcDialog dInit( dialogROWS,     // number of display lines
                       dialogCOLS,     // number of display columns
                       ulY,            // Y offset from upper-left of terminal 
                       ulX,            // X offset from upper-left of terminal 
                       "  Paste Special  ",// dialog title
                       ncltSINGLE,     // border line-style
                       dColor,         // border color attribute
                       dColor,         // interior color attribute
                       ic              // pointer to list of control definitions
                     ) ;

   //* Instantiate the dialog window *
   NcDialog* dp = new NcDialog ( dInit ) ;

   //* Open the dialog window *
   if ( (dp->OpenWindow()) == OK )
   {
      LineDef  lDefH( ncltHORIZ, ncltSINGLE, (dialogROWS - 5), ZERO, 
                      dialogCOLS, dColor ) ;
      dp->DrawLine ( lDefH ) ;

      //* Write the static text *
      winPos wp( 1, 2 ) ;
      wp = dp->WriteString ( wp.ypos,  wp.xpos, "Source Items: ", dColor ) ;
      char rBuff[32], dBuff[32] ;
      gString dispSrc ;
      dispSrc.formatInt( (ULONG)regCount, 5, true ) ; dispSrc.copy( rBuff, 6 ) ;
      dispSrc.formatInt( (ULONG)dirCount, 5, true ) ; dispSrc.copy( dBuff, 6 ) ;
      dispSrc.compose( L" %s Reg Files and %s Dir Trees ", rBuff, dBuff ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, dispSrc, dColor | ncuATTR ) ;

      dispSrc.formatInt( (ULONG)totCount, 6, true ) ; dispSrc.copy( rBuff, 7 ) ;
      dispSrc.formatInt( (ULONG)totBytes, 6, true ) ; dispSrc.copy( dBuff, 7 ) ;
      wp = dp->WriteString ( wp.ypos, (wp.xpos=2), " Total Files: ", dColor ) ;
      dispSrc.compose( L"%s  (%s bytes)", rBuff, dBuff ) ;
      dp->WriteString ( wp.ypos, wp.xpos, dispSrc, dColor ) ;

      dp->WriteString ( 4, 2, "Specify type for new files:", dColor ) ;
      dp->WriteString ( 9, 2, "Specify name for new files:", dColor ) ;

      //* Set radio buttons as an exclusive-OR group *
      //* (only one can be selected at any moment).  *
      if ( (dp->GroupRadiobuttons ( XorSrcGroup )) != OK )
         ;//this->dPtr->DebugMsg ( "XorSrcGroup failed...", 3, true ) ;
      if ( (dp->GroupRadiobuttons ( XorNameGroup )) != OK )
         ;//this->dPtr->DebugMsg ( "XorNameGroup failed...", 3, true ) ;

      //* Adjust color of labels for inactive radio buttons *
      if ( ic[pslnRB].active == false )
         dp->WriteString ( (ic[pslnRB].ulY + ic[pslnRB].labY), 
                           (ic[pslnRB].ulX + ic[pslnRB].labX), 
                           ic[pslnRB].label, inactiveColor ) ;
      if ( ic[phlnRB].active == false )
         dp->WriteString ( (ic[phlnRB].ulY + ic[phlnRB].labY), 
                           (ic[phlnRB].ulX + ic[phlnRB].labX), 
                           ic[phlnRB].label, inactiveColor ) ;
      if ( ic[nsrcRB].active == false )
         dp->WriteString ( (ic[nsrcRB].ulY + ic[nsrcRB].labY), 
                           (ic[nsrcRB].ulX + ic[nsrcRB].labX), 
                           ic[nsrcRB].label, inactiveColor ) ;
      if ( ic[nbatRB].active == false )
         dp->WriteString ( (ic[nbatRB].ulY + ic[nbatRB].labY), 
                           (ic[nbatRB].ulX + ic[nbatRB].labX), 
                           ic[nbatRB].label, inactiveColor ) ;

      dp->RefreshWin () ;           // make everything visible

      //************************
      //* Get user's selection *
      //************************
      uiInfo   Info ;               // user interface data returned here
      short    icIndex = ZERO ;     // index of control with input focus
      bool     done = false ;       // loop control
      while ( ! done )
      {
         if ( ic[icIndex].type == dctPUSHBUTTON )
         {
            if ( Info.viaHotkey == false )
               icIndex = dp->EditPushbutton ( Info ) ;
            else
               Info.HotData2Primary () ;
            if ( Info.dataMod != false )
            {
               //* User has selected 'paste special' options - huzzah! *
               if ( Info.ctrlIndex == okPB )
               {
                  //* Get the source-file radio-button-group's selection *
                  short src = dp->GetRbGroupSelection ( XorSrcGroup[0] ) ;
                  if ( src == pslnRB )
                  {  //* If target files will be symbolic links *
                     pasteOption = psstCreateSymLink ;
                  }
                  else if ( src == psrcRB )
                  {  //* If targets will be copies of source files *
                     pasteOption = psstSimpleCopy ;
                  }
                  else
                  {  //* If targets will be hard links *
                     pasteOption = psstCreateHardLink ;
                  }
                  //* Get the target-filename radio-button-group's selection *
                  short trg = dp->GetRbGroupSelection ( XorNameGroup[0] ) ;
                  //* Targets will have the same names as the source files *
                  if ( trg == nsrcRB )
                     renOption = noRENAME ;
                  //* User will be asked to enter a rename pattern *
                  else if ( trg == nbatRB )
                     renOption = batchRENAME ;
                  //* User will be prompted for each target name *
                  else  // (trg == nindRB )
                     renOption = promptRENAME ;
               }
               //* Cancel the operation *
               else if ( Info.ctrlIndex == canPB )
                  ;     // nothing to do
               done = true ;
            }
            else if ( Info.keyIn == nckESC )
            {  //* User panic, abort the operation *
               done = true ;
            }
         }
         else if ( ic[icIndex].type == dctRADIOBUTTON )
         {
            if ( Info.viaHotkey == false )
               icIndex = dp->EditRadiobutton ( Info ) ;
            else
               Info.HotData2Primary () ;
         }

         //* Move focus to appropriate control *
         if ( !done && !Info.viaHotkey )
         {
            if ( Info.keyIn == nckSTAB )
               icIndex = dp->PrevControl () ; 
            else
               icIndex = dp->NextControl () ;
         }
      }     // while(!done)
   }
   else           // dialog did not open (probably memory allocation error)
      this->DisplayStatus ( ecMEMALLOC ) ;

   if ( dp != NULL )                         // close the window
      delete ( dp ) ;

   //* Restore display of parent dialog window *
   this->dPtr->RefreshWin () ;

   return pasteOption ;

}  //* End pslMultiPaste_Options() *

//******************************************************************************
//** General-purpose methods supporting paste-from-clipboard operations       **
//******************************************************************************

//***********************
//*     CopyFile        *
//***********************
//******************************************************************************
//* Copy a file from source to target.                                         *
//* This method calls the appropriate FMgr-class method for the specified      *
//* file type.                                                                 *
//*                                                                            *
//* Input  : sStats : sourcefile stats incl. file type and read/write access   *
//*          srcPath: full path/filename specification for source file         *
//*          trgPath: full path/filename specification for target file         *
//*          lnkTarget: (optional, false by default)                           *
//*                   if 'true', copy the sym-link target file instead of the  *
//*                   sym-link itself. (for consistency in error reporting,    *
//*                   caller should verify that link target exists)            *
//*                                                                            *
//* Returns: OK if successful                                                  *
//*          'errno' value if error. Also error status is saved and can be     *
//*             retrieved through GetErrorCode() or DisplayErrorCode().        *
//******************************************************************************
//* Programmer's Note: At this time, we do not support copy of                 *
//*       Character-device files                                               *
//*       Block-device files                                                   *
//*       Socket files                                                         *
//* For these types, we display an info dialog with the bad news.              *
//*                                                                            *
//* Note on read/write access:                                                 *
//* Write access on the source is required for moving, deleting or renaming    *
//* the file, and read access on the sources is required for copying           *
//* (or moving) the file. Thus, it is possible to delete a file that we are    *
//* not able to read, or copy a file we are not able to modify. While this is  *
//* logical, do not make assumptions about read/write access: Verify!          *
//******************************************************************************

short FileDlg::CopyFile ( const tnFName& sStats, const gString& srcPath, 
                          const gString& trgPath, bool lnkTarget )
{
   short status = OK ;
   //* User must have read access to source UNLESS copying a symbolic link *
   //* (which is always accessible because its permissions are ignored).   *
   if ( sStats.readAcc || (sStats.fType == fmLINK_TYPE && !lnkTarget) )
   {
      switch ( sStats.fType )
      {
         case fmREG_TYPE:
            if ( clipBoard_gvfsSrc || clipBoard_gvfsTrg )
               status = this->fmPtr->CopyFile_gvfs ( srcPath.ustr(), trgPath.ustr() ) ;
            else
               status = this->fmPtr->CopyFile ( srcPath.ustr(), trgPath.ustr() ) ;
            break ;
         case fmLINK_TYPE:
            //* If source is a symbolic link AND caller has specified *
            //* to copy the link target, do so if possible.           *
            //* (Note that we trust caller to know that the source)   *
            //* (file is actually a sym-link.                     )   *
            if ( lnkTarget != false )
            {
               gString  ltsrcPath ;    // path/filename of link-target file
               tnFName  ltsrcStats ;   // stats for link target
               status = this->fmPtr->GetLinkTargetStats ( srcPath, ltsrcStats, ltsrcPath ) ;
               //* Link target exists. Note that this is a recursive call *
               if ( status == OK )
               {
                  if ( clipBoard_gvfsSrc || clipBoard_gvfsTrg )
                     status = this->fmPtr->CopyFile_gvfs ( srcPath.ustr(), trgPath.ustr() ) ;
                  else
                     status = this->CopyFile ( ltsrcStats, ltsrcPath.ustr(), trgPath.ustr() ) ;
               }
               else
               { /* Link target does not exist */ }
            }
            else     // copy the symbolic-link source file itself
            {
               if ( clipBoard_gvfsSrc || clipBoard_gvfsTrg )
                  status = this->fmPtr->CopyFile_gvfs ( srcPath.ustr(), trgPath.ustr() ) ;
               else
                  status = this->fmPtr->CopyFile ( srcPath.ustr(), trgPath.ustr() ) ;
            }
            break ;
         case fmFIFO_TYPE:
            if ( ! clipBoard_gvfsSrc && ! clipBoard_gvfsTrg )
               status = this->fmPtr->CopyFifo ( srcPath.ustr(), trgPath.ustr() ) ;
            else     // FIFO copy not supported for GVfs filesystems
               status = EACCES ;
            break ;

         #if 0    // CURRENTLY NOT SUPPORTED
         case fmCHDEV_TYPE:
            if ( ! clipBoard_gvfsSrc && ! clipBoard_gvfsTrg )
               status = this->fmPtr->CopyCharDevice ( srcPath.ustr(), trgPath.ustr() ) ;
            else     // Char Device copy not supported for GVfs filesystems
               status = EACCES ;
            break ;
         case fmBKDEV_TYPE:
            if ( ! clipBoard_gvfsSrc && ! clipBoard_gvfsTrg )
               status = this->fmPtr->CopyBlockDevice ( srcPath.ustr(), trgPath.ustr() ) ;
            else     // Block Device copy not supported for GVfs filesystems
               status = EACCES ;
            break ;
         case fmSOCK_TYPE:
            if ( ! clipBoard_gvfsSrc && ! clipBoard_gvfsTrg )
               status = this->fmPtr->CopySocket ( srcPath.ustr(), trgPath.ustr() ) ;
            else     // Socket copy not supported for GVfs filesystems
               status = EACCES ;
            break ;
         #else    // UNSUPPORTED FILE-TYPE COPY: WARNING MESSAGE
         case fmCHDEV_TYPE:
         case fmBKDEV_TYPE:
         case fmSOCK_TYPE:
            {  // (scoping)
            gString sname ;
            this->fmPtr->ExtractFilename ( sname, srcPath ) ;
            status = this->CopyOrDeleteNotSupported( sStats.fType, sname.ustr() ) ;
            }
            break ;
         #endif   // WARNING MESSAGE ONLY

         case fmDIR_TYPE:
            if ( clipBoard_gvfsSrc || clipBoard_gvfsTrg )
               status = this->fmPtr->CopyDirName_gvfs ( srcPath.ustr(), trgPath.ustr() ) ;
            else
               status = this->fmPtr->CopyDirName ( srcPath.ustr(), trgPath.ustr() ) ;
            break ;
         case fmUNKNOWN_TYPE:
            if ( (this->DecisionDialog ( warnUnsuppMod )) != false )
               status = this->fmPtr->CopyUnsupportedType ( srcPath.ustr(), trgPath.ustr() ) ;
            else  // user aborted the copy operation
               status = ENOSYS ;
            break ;
         case fmTYPES:        // silence compiler warning
         default:             // getting here would be an logical error (gasp!)
            break ;
      }
      if ( status != OK )
         this->SetErrorCode ( ecPACCV ) ;
   }
   else                       // no read access for source file
   {
      status = EACCES ;
      this->SetErrorCode ( ecPERMISSION, status ) ;
   }
   return status ;

}  //* End CopyFile() *

//***********************
//*  DeleteSourceFile   *
//***********************
//******************************************************************************
//* Delete the specified file.                                                 *
//*                                                                            *
//* Note: It is assumed that srcStats->writeAcc has been correctly initialized,*
//*       and we WILL NOT attempt to override write protection here.           *
//*                                                                            *
//* Input  : srcStats: pointer to source file's data structure in clipboard    *
//*                    list                                                    *
//*          srcPath : full path/filename specification for source file        *
//*                                                                            *
//* Returns: OK if file successfully deleted                                   *
//*          else: FMgr error code if operation fails                          *
//*                ecSRCPROTECT    if file is write-protected                  *
//*                ecFTYPE         if incorrect file type for operation        *
//*                ecUNSUPP        if operation not supported for file type.   *
//*            Also error status is saved and can be retrieved through         *
//*            GetErrorCode() or DisplayErrorCode().                           *
//******************************************************************************

short FileDlg::DeleteSourceFile ( const tnFName* srcStats, const gString& srcPath )
{
   short status = OK ;

   //* User must have write access to the file *
   if ( srcStats->writeAcc != false )
   {
      switch ( srcStats->fType )
      {
         case fmREG_TYPE:     // regular file
         case fmLINK_TYPE:    // symbolic link file
         case fmFIFO_TYPE:    // FIFO file
            if ( clipBoard_gvfsSrc )
               status = this->fmPtr->DeleteFile_gvfs ( srcPath.ustr() ) ;
            else
               status = this->fmPtr->DeleteFile ( srcPath.ustr() ) ;
            break ;
         case fmDIR_TYPE:     // directory file
            //* Directory must be empty *
            if ( clipBoard_gvfsSrc )
            {
               if ( (this->fmPtr->isEmptyDir_gvfs ( srcPath.ustr() )) )
                  status = this->fmPtr->DeleteEmptyDir_gvfs ( srcPath.ustr() ) ;
               else  // this should never happen because caller should check
                  status = this->SetErrorCode ( ecPACCV, ENOTEMPTY ) ;
            }
            else
            {
               if ( (this->fmPtr->isEmptyDir ( srcPath.ustr() )) )
                  status = this->fmPtr->DeleteEmptyDir ( srcPath.ustr() ) ;
               else  // this should never happen because caller should check
                  status = this->SetErrorCode ( ecPACCV, ENOTEMPTY ) ;
            }
            break ;
         case fmCHDEV_TYPE:   // character device file
            // NOTE: srcPath is full path/filename specification, and this call 
            //       requires the path string only (but not currently used)
            status = this->dsfDeleteChdevFile ( *srcStats, srcPath ) ;
            this->SetErrorCode ( ecUNSUPP, EPERM ) ;
            break ;
         case fmBKDEV_TYPE:   // block device file
            // NOTE: srcPath is full path/filename specification, and this call 
            //       requires the path string only (but not currently used)
            status = this->dsfDeleteBkdevFile ( *srcStats, srcPath ) ;
            this->SetErrorCode ( ecUNSUPP, EPERM ) ;
            break ;
         case fmSOCK_TYPE:    // socket file
            // NOTE: srcPath is full path/filename specification, and this call 
            //       requires the path string only (but not currently used)
            status = this->dsfDeleteSocketFile ( *srcStats, srcPath ) ;
            this->SetErrorCode ( ecUNSUPP, EPERM ) ;
            break ;
         case fmUNKNOWN_TYPE: // unknown file type
            // NOTE: srcPath is full path/filename specification, and this call 
            //       requires the path string only (but not currently used)
            status = this->dsfDeleteUnsuppFile ( *srcStats, srcPath ) ;
            this->SetErrorCode ( ecUNSUPP, EPERM ) ;
            break ;
         default:             // getting here would be an logical error (gasp!)
            status = this->SetErrorCode ( ecUNSUPP, EACCES ) ;
            break ;
      }
   }
   else
   {
      status = this->SetErrorCode ( ecSRCPROTECT, EACCES ) ;
   }
   return status ;

}  //* End DeleteSourceFile() *

//****************************
//*    SourceWriteEnable     *
//****************************
//******************************************************************************
//* ** Private Method **                                                       *
//* Test write access for clipboard source files and directories.              *
//* Determine whether user has write access for each source file, which is     *
//* indicated by the 'writeAcc' flag for each file.                            *
//*                  (see also SourceWriteProtect())                           *
//*                                                                            *
//* Optionally, attempt to enable write-access on all protected files before   *
//* returning to caller.                                                       *
//*                                                                            *
//* Input  : writeEnable: (optional, false by default)                         *
//*            if 'true' attempt to set all write-enable flags for Owner       *
//*            and Group (NOT Others) for all protected files in the list.     *
//*                                                                            *
//* Returns: number of write-enabled files in the clipboard list               *
//*            If return value==number of files in clipboard list, then ALL    *
//*            files are write enabled.                                        *
//******************************************************************************
//* Programmer's Note: This solution is a bit inefficient in terms of          *
//* processing time, but it leverages existing code. :-)                       *
//******************************************************************************

UINT FileDlg::SourceWriteEnable ( bool writeEnable )
{
   UINT wpCount, rpCount ;
   this->SourcePermission ( wpCount, rpCount, writeEnable, false, true ) ;
   return ( clipBoard[BNODE].totFiles - wpCount ) ;

}  //* End SourceWriteEnable() *

//****************************
//*    SourceWriteProtect    *
//****************************
//******************************************************************************
//* Test write access for clipboard source files and directories.              *
//* Determine whether user has write access for each source file, indicating   *
//* the result through the writeAcc flag for each file.                        *
//* Returns the number of write-PROTECTED files found.                         *
//*                   (see also SourceWriteEnable())                           *
//*                                                                            *
//* Optionally, attempt to write-protect each file in the list before          *
//* returning to caller (see 'writeProtect' parameter).                        *
//*                                                                            *
//* Input  : writeProtect: (optional, false by default)                        *
//*            if 'true' attempt to reset all write-enable permission bits for *
//*            Owner, Group, and Others of all unprotected files in the list.  *
//*                                                                            *
//* Returns: number of write-protected files in the list.                      *
//*            If return value==number of files in clipboard list, then ALL    *
//*            files are write protected.                                      *
//******************************************************************************

UINT FileDlg::SourceWriteProtect ( bool writeProtect )
{
   UINT  wpCount = ZERO ;                  // number of write-protected files
   gString basePath( clipBoard_srcPath ),  // base path/filename string
           dirPath ;                       // subdirectory path/filename

   //* Test non-directory top-level files *
   wpCount += this->SourceWProtect ( clipBoard[BNODE].tnFiles, 
                                     clipBoard[BNODE].tnFCount, 
                                     basePath, writeProtect ) ;

   //* Test directory trees *
   for ( UINT i = ZERO ; i < clipBoard_dirTrees ; i++ )
   {
      this->fmPtr->CatPathFname ( dirPath, basePath.ustr(), 
                                  clipBoard[BNODE].nextLevel[i].dirStat.fName ) ;
      if ( clipBoard[BNODE].nextLevel[i].dirStat.writeAcc == false )
      {  //* Increment count of write-protected files  *
         //* AND count of protected files in this node *
         ++wpCount ;
         ++clipBoard[BNODE].nextLevel[i].wprotFiles ;
      }
      else if ( (writeProtect != false) &&
                ((this->FileWriteProtection ( clipBoard[BNODE].nextLevel[i].dirStat, 
                                              dirPath, true )) != false) )
      {  //* Increment count of write-protected files  *
         //* AND count of protected files in this node *
         ++wpCount ;
         ++clipBoard[BNODE].nextLevel[i].wprotFiles ;
      }

      //* Then traverse its subdirectory tree  *
      wpCount += this->SourceWProtect ( &clipBoard[BNODE].nextLevel[i], 
                                        dirPath, writeProtect ) ;
   }
   return wpCount ;

}  //* End SourceWriteProtect() *

//***********************
//*   SourceWProtect    *
//***********************
//******************************************************************************
//* Recursively travel the specified TreeNode tree and test whether each file  *
//* is write protected.                                                        *
//* Called by SourceWriteProtect().                                            *
//*                                                                            *
//* The writeAcc flag in the file's tnFname class object is set/reset by       *
//* call to FileWriteProtection() reflecting write-permission status.          *
//*                                                                            *
//* Input  : tnPtr  : pointer to top of TreeNode list to be tested             *
//*          dirPath: base directory path under which the specified directory  *
//*                   tree lives                                               *
//*          protect: (optional, false by default)                             *
//*                   if 'true' attempt to write-protect each unprotected      *
//*                   file in list                                             *
//*                                                                            *
//* Returns: number of write-protected files in list.                          *
//******************************************************************************

UINT FileDlg::SourceWProtect ( TreeNode* tnPtr, const gString& dirPath, bool protect )
{
   UINT wpCount = ZERO ;      // return value - write-protected files in this node
   tnPtr->wprotFiles = ZERO ; // write-protected files in this node

   //* Process non-directory files on this level *
   if ( tnPtr->tnFCount > ZERO && tnPtr->tnFiles != NULL )
   {
      wpCount = this->SourceWProtect ( tnPtr->tnFiles, tnPtr->tnFCount, 
                                       dirPath, protect ) ;
      tnPtr->wprotFiles = wpCount ;  // write-protected files for this node
   }

   //* If there are subdirectories below this one, process them *
   if ( tnPtr->dirFiles > ZERO && tnPtr->nextLevel != NULL )
   {
      gString sPath ;         // subdirectory path/filename string
      for ( UINT i = ZERO ; i < tnPtr->dirFiles ; i++ )
      {
         this->fmPtr->CatPathFname ( sPath, dirPath.ustr(), 
                                     tnPtr->nextLevel[i].dirStat.fName ) ;
         if ( tnPtr->nextLevel[i].dirStat.writeAcc == false )
         {
            ++wpCount ;
            ++tnPtr->wprotFiles ;
         }
         else if ( (protect != false) &&
                   ((this->FileWriteProtection ( tnPtr->nextLevel[i].dirStat, 
                                                 sPath, protect )) != false) )
         {
            ++wpCount ;
            ++tnPtr->wprotFiles ;
         }
         //* Test lower-level subdirectory contents *
         wpCount += this->SourceWProtect ( &tnPtr->nextLevel[i], sPath, protect ) ;
      }
   }
   return wpCount ;

}  //* End SourceWProtect() *

//***********************
//*   SourceWProtect    *
//***********************
//******************************************************************************
//* For each file in the tnFName array, test whether file is write protected.  *
//*                                                                            *
//* The writeAcc flag in the file's tnFname class object is set/reset by       *
//* call to FileWriteProtection() reflecting write-permission status.          *
//*                                                                            *
//* Input  : tnfList: pointer to an array of tnFName objects containing file   *
//*                   data.                                                    *
//*          tnfCount: number of files in the tnfList[] array                  *
//*          dirPath: base directory path for the files in this list.          *
//*          protect: (optional, false by default)                             *
//*            if 'true' attempt to write-protect each unprotected file in list*
//*                                                                            *
//* Returns: number of write-protected files in list.                          *
//******************************************************************************

UINT FileDlg::SourceWProtect ( tnFName* tnf, UINT tnfCount, 
                               const gString& dirPath, bool protect )
{
   UINT wpCount = ZERO ;
   gString filePath ;

   for ( UINT i = ZERO ; i < tnfCount ; i++ )
   {
      if ( tnf[i].fType != fmLINK_TYPE && tnf[i].writeAcc == false )
         ++wpCount ;
      else if ( protect != false )
      {
         this->fmPtr->CatPathFname ( filePath, dirPath.ustr(), tnf[i].fName ) ;
         if ( (this->FileWriteProtection ( tnf[i], filePath, true )) != false )
            ++wpCount ;
      }
   }
   return wpCount ;

}  //* End SourceWProtect() *

//***********************
//* FileWriteProtection *
//***********************
//******************************************************************************
//* Test the write permissions on the specified file and return 'true' if      *
//* user DOES NOT HAVE permission to modify/delete the file.                   *
//* Optionally, attempt to write-protect the file.                             *
//*                                                                            *
//*                                                                            *
//* Input  : fStats : tnFName class object (by reference) describing the file  *
//*                    On return, writeAcc flag is set/reset to reflect write- *
//*                    permission status. Other data members are unmodified.   *
//*          fPath  : path/filename of the file to be modified                 *
//*          protect : (optional, false by default)                            *
//*                     if true, disable 'owner' and 'group' and 'others'      *
//*                     write permission if possible                           *
//*                                                                            *
//* Returns: true if user DOES NOT HAVE write permission on the specified      *
//*          file; else, false                                                 *
//******************************************************************************
//* Programmer's Note: This method could be simplified IF it is only used on   *
//* files in the display or clipboard trees; however in future, we may want    *
//* to use it on an arbitrary file.                                            *
//******************************************************************************

bool FileDlg::FileWriteProtection ( tnFName& fStats, const gString& fPath, 
                                    bool writeProtect )
{
   bool isProtected = ((this->fmPtr->WritePermission ( fStats, fPath )) == false) ;

   //* If we are to attempt write protection. (fStats.writeAcc will be reset)  *
   if ( writeProtect != false && isProtected == false )
   {
      if ( (this->fmPtr->WriteProtect ( fPath, fStats )) == OK )
         isProtected = true ;
   }
   return isProtected ;

}  //* End FileWriteProtection() *

//**************************
//* pclCreateSuggestedName *
//**************************
//******************************************************************************
//* Create a 'suggested filename' of the form:  'filename(2).ext'.             *
//* If origName is already in this format, increase the sequence number.       *
//* Used for making a copy of a source file in the same directory.             *
//*                                                                            *
//* Input  : suggName : (by reference) receives filename string                *
//*          origName : (by reference) current filename                        *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void FileDlg::pclCreateSuggestedName ( gString& suggName, const gString& origName )
{
   wchar_t tname[MAX_FNAME] ;
   origName.copy( tname, MAX_FNAME ) ;
   long int val = 2 ;
   short t, tEnd ;
   for ( tEnd = ZERO ; tname[tEnd] != NULLCHAR ; tEnd++ ) ; // find the end of the string
   t = tEnd ;
   while ( tname[t] != PERIOD && t > ZERO )
      --t ;
   if ( t > ZERO )
   {
      short tExt = t ;
      tname[tExt++] = NULLCHAR ;
      if ( t >= 4 && tname[t-3] == '(' && tname[t-1] == ')'
           && (tname[t-2] >= '0' && tname[t-2] < '9') )
      {
         val = wcstol ( &tname[t-2], NULL, 10 ) ;
         ++val ;
         tname[t-3] = NULLCHAR ;
      }
      suggName.compose( L"%S(%ld).%S", tname, &val, &tname[tExt] ) ;
   }
   else                    // filename has no extension
   {
      if ( tEnd >= 4 && tname[tEnd-3] == '(' && tname[tEnd-1] == ')'
           && (tname[tEnd-2] >= '0' && tname[tEnd-2] < '9') )
      {
         val = wcstol ( &tname[tEnd-2], NULL, 10 ) ;
         ++val ;
         tname[tEnd-3] = NULLCHAR ;
      }
      suggName.compose( L"%S(%ld)", tname, &val ) ;
   }

}  //* End pclCreateSuggestedName() *

//***********************
//*    pclCTLerror      *
//***********************
//******************************************************************************
//* Report an error in pclCopyTnList that is serious enough to abort the       *
//* copy-and-paste or cut-and-paste operation.                                 *
//*                                                                            *
//* Input  : pointer to directory name for directory which caused the error    *
//*                                                                            *
//* Returns: 'true' i.e. ABORT                                                 *
//******************************************************************************

bool FileDlg::pclCTLerror ( const char* dirName, short errnoVal )
{
   gString oBuff ;
   oBuff.compose( L"  \"%s\"", dirName ) ;
   const char* dlgMsg[] = 
   { //1234567890123456789012345678901234567890123456789012 - (max line length)
      "  ALERT!!  ALERT!!  ",
      "Directory-tree copy operation has encountered",
      "unrecoverable errors, and had to be aborted. ",
      " ",
      oBuff.ustr(),
      "System reports:",
      (char*)this->fmPtr->GetErrnoMessage ( errnoVal ),
      "",
      NULL,
   } ;
   if ( errnoVal == EACCES )        // existing directory is write protected
   {
      dlgMsg[3] = "Existing target directory is write-protected:" ;
      dlgMsg[7] = "Please remove write protection before copying files." ;
   }
   else if ( errnoVal == ENOTDIR )  // existing file is not of type fnDIR_TYPE
   {
      dlgMsg[3] = "A filename on the target path is not a directory:" ;
   }
   else                             // unable to create the target directory
   {
      dlgMsg[3] = "Unable to create target sub-directory:" ;
   }
   this->InfoDialog ( (const char**)dlgMsg ) ;
   return true ;

}  //* End pclCTLerror() *

//***********************
//*  CheckTargetSpace   *
//***********************
//******************************************************************************
//* Test for available space on target file system in preparation for a        *
//* 'Paste-clipboard-list' operation.                                          *
//*                                                                            *
//* Also, existing files on the target may be overwritten during the operation,*
//* which would significantly reduce the amount of free space required. It     *
//* would be tedious and time consuming to test for this factor, however.      *
//*                                                                            *
//* If there is any question whether we will be successful, prompt for user's  *
//* permission to proceed.                                                     *
//*                                                                            *
//* Input Values : none                                                        *
//*                                                                            *
//* Return Value : 'true' if target space available > size of files on         *
//*                  clipboard, OR user has given permission to continue.      *
//*                'false' if insufficient space AND user aborts the operation.*
//******************************************************************************
//* Notes of sizing the target:                                                *
//* Note that the actual target space needed is a complex calculation involving*
//* the number of 'blocks' allocated for the source, the size of those blocks, *
//* and the block size on the target system. This seems like too much          *
//* calculation time, so we approximate the value with a reasonable comfort    *
//* margin.                                                                    *
//*                                                                            *
//* For a 'copy' operation, we could be targeting the same file system as the  *
//* source, or a different file system. Either way, we test the target file    *
//* system for free space of total size of source files (+10%).                *
//*                                                                            *
//* For a 'cut' operation, we have choices:                                    *
//* If the target file system is different from the source file system, we do  *
//* the same test as above (total source size +10%). If we are moving files    *
//* within a file system, however, we COULD get by on only enough free space   *
//* to copy the largest file; however, the PasteClipboardList() method is not  *
//* guaranteed to delete each source file immediately after the copy, we       *
//* still test for total source + 10% to be safe.                              *
//*                                                                            *
//*                                                                            *
//* Note on read-only filesystems:                                             *
//* For read-only targets, the Free Blocks, Available Blocks, Total Inodes,    *
//* and Available Inodes will all be ZERO. In addition the permissions on the  *
//* target will be set to read-only. Thus, it will be clear from multiple      *
//* perspectives that it is not possible to copy files to the target.          *
//*                                                                            *
//* The decision to be made , then is whether we declare the file system as    *
//* read-only or simply play dumb and say there insufficient space on the      *
//* target. The decision is that if FMgr reports ZERO in tFreeBytes, we will   *
//* declare it a read-only filesystem, even though it may just be filled.      *
//* This results in smaller, cleaner code.                                     *
//******************************************************************************

bool FileDlg::CheckTargetSpace ( void )
{
bool  spaceOk = false ;

   //* If nothing is on the clipboard, don't bother *
   if ( clipBoard[BNODE].totFiles > ZERO )
   {
      UINT64   tFileSys, tFreeBytes ;
      short    tStatus ;

      tStatus = this->fmPtr->TargetFilesystemFreespace ( this->currDir, 
                                                      tFileSys, tFreeBytes ) ;
      if ( tStatus == OK )
      {
         double needed = (double)clipBoard[BNODE].totBytes * 1.10 ;
         if ( needed > tFreeBytes )
         {
            //* Space on the target is tight, get user permission to proceed   *
            if ( tFreeBytes > ZERO )
            {
               UINT64 bNeed = (UINT64)needed ;
               spaceOk = this->ConfirmPasteToTarget ( bNeed, tFreeBytes ) ;
            }
            //* No space on target - probably read-only filesystem (see note)  *
            else
            {
               this->InfoDialog ( readonlyFileSys ) ;
            }
         }
         else
            spaceOk = true ;
      }
      else
      {
         //* Just because the filesystem stat failed, it is not cause to       *
         //* forbid the operation; however, we should warn user of potential   *
         //* problems.                                                         *
         spaceOk = this->ConfirmPasteToTarget ( ZERO, ZERO ) ;
      }
   }
   else
      spaceOk = true ;  // there is plenty of space for doing nothing

   return spaceOk ;

}  //* End CheckTargetSpace() *

//***********************
//*                     *
//***********************
//******************************************************************************
//*                                                                            *
//*                                                                            *
//*                                                                            *
//* Input  :                                                                   *
//*                                                                            *
//* Returns:                                                                   *
//******************************************************************************


