//********************************************************************************
//* File       : FileDlgMedia.cpp                                                *
//* Author     : Mahlon R. Smith                                                 *
//*              Copyright (c) 2016-2025 Mahlon R. Smith, The Software Samurai   *
//*                  GNU GPL copyright notice located in FileMangler.hpp         *
//* Date       : 12-May-2025                                                     *
//* Version    : (see FileDlgVersion string in FileDlg.cpp)                      *
//*                                                                              *
//* Description:                                                                 *
//* This module of the FileDlg class contains decoding algorithms for            *
//* extracting metadata from media files (audio/video).                          *
//*                                                                              *
//* Supported Formats:                                                           *
//* ------------------                                                           *
//*  MPEG-3 Audio (MP3)                                                          *
//*  OGG/Vorbis Audio (OGG, OGA)                                                 *
//*  MPEG-4 Audio (M4A)                                                          *
//*  Windows Media (audio and video, WMA/WMV)                                    *
//*                                                                              *
//* Development Tools: see notes in FileMangler.cpp.                             *
//********************************************************************************
//* Version History (most recent first):                                         *
//*   See version history in FileDlg.cpp.                                        *
//********************************************************************************
//* Programmer's Notes:                                                          *
//* Audio/video metadata can be extracted from the media file and displayed      *
//* as part of the View-file-contents sequence.                                  *
//*                                                                              *
//* Because the media data are binary, there is nothing to see there; however,   *
//* each media-file format contains a mechanism for storing human-readable       *
//* data for Artist, Composer, Producer, Copyright, etc. These data fields are   *
//* generically known as "tags."                                                 *
//*                                                                              *
//* While MP3 audio is by far the most popular, other audio media formats are    *
//* in general use. Specifically the open-standard OGG/Vorbis audio encoding.    *
//*                                                                              *
//* Please note that the formatting standards are not well implemented by        *
//* media players (VLC and others), so a certain tolerance for technically       *
//* invalid formatting is built into these methods.                              *
//*                                                                              *
//* Support for other media formats will be added as requested.                  *
//*                                                                              *
//* FileMangler does not support editing of the metadata because it would        *
//* stray from the basic mission of file management. Other open-source           *
//* applications are available for editing media metadata. If these turn out     *
//* to be clunky, may we recommend our own Taggit application for editing tag    *
//* data.                                                                        *
//*                                                                              *
//********************************************************************************

#include "GlobalDef.hpp"
#include "FileDlg.hpp"

//******************************
//* Local definitions and data *
//******************************

//******************************************************************************
//**               Definitions for MPEG-3 (MP3) audio files                   **
//******************************************************************************

//* Codes used to identify frames which contain text. *
const short TEXT_FRAMES = 39 ;      // number of text frame types
const char* const TextFrameID[TEXT_FRAMES] = 
{
"TALB",    // #TALB Album/Movie/Show title
"TBPM",    // #TBPM BPM (beats per minute)
"TCOM",    // #TCOM Composer
"TCON",    // #TCON Content type
"TCOP",    // #TCOP Copyright message
"TDAT",    // #TDAT Date
"TDLY",    // #TDLY Playlist delay
"TENC",    // #TENC Encoded by
"TEXT",    // #TEXT Lyricist/Text writer
"TFLT",    // #TFLT File type
"TIME",    // #TIME Time
"TIT1",    // #TIT1 Content group description
"TIT2",    // #TIT2 Title/songname/content description
"TIT3",    // #TIT3 Subtitle/Description refinement
"TKEY",    // #TKEY Initial key
"TLAN",    // #TLAN Language(s)
"TLEN",    // #TLEN Length
"TMED",    // #TMED Media type
"TOAL",    // #TOAL Original album/movie/show title
"TOFN",    // #TOFN Original filename
"TOLY",    // #TOLY Original lyricist(s)/text writer(s)
"TOPE",    // #TOPE Original artist(s)/performer(s)
"TORY",    // #TORY Original release year
"TOWN",    // #TOWN File owner/licensee
"TPE1",    // #TPE1 Lead performer(s)/Soloist(s)
"TPE2",    // #TPE2 Band/orchestra/accompaniment
"TPE3",    // #TPE3 Conductor/performer refinement
"TPE4",    // #TPE4 Interpreted, remixed, or otherwise modified by
"TPOS",    // #TPOS Part of a set
"TPUB",    // #TPUB Publisher
"TRCK",    // #TRCK Track number/Position in set
"TRDA",    // #TRDA Recording dates
"TRSN",    // #TRSN Internet radio station name
"TRSO",    // #TRSO Internet radio station owner
"TSIZ",    // #TSIZ Size
"TSRC",    // #TSRC ISRC (international standard recording code)
"TSSE",    // #TSEE Software/Hardware and settings used for encoding
"TYER",    // #TYER Year
"TXXX",    // #TXXX User defined text information frame
} ;
//* Human-friendly equivalents to the TextFrameID list above *
const char* const TextFrameDesc[TEXT_FRAMES] = 
{
"Album Title ------------------ ",
"BPM (beats per minute) ------- ",
"Composer --------------------- ",
"Content type ----------------- ",
"Copyright © ------------------ ",
"Date ------------------------- ",
"Playlist delay --------------- ",
"Encoded by ------------------- ",
"Lyricist/Text writer --------- ",
"File type -------------------- ",
"Time ------------------------- ",
"Content Group ---------------- ",
"Title ------------------------ ",
"Subtitle --------------------- ",
"Initial Key ------------------ ",
"Language --------------------- ",
"Length ----------------------- ",
"Media type ------------------- ",
"Original Title --------------- ",
"Original Filename ------------ ",
"Original lyricist ------------ ",
"Original artist -------------- ",
"Original Release ------------- ",
"File owner ------------------- ",
"Lead performer(s) ------------ ",
"Accompaniment ---------------- ",
"Conductor/performer ---------- ",
"Remixed by ------------------- ",
"Part of a set ---------------- ",
"Publisher -------------------- ",
"Track number/Position -------- ",
"Recording dates -------------- ",
"Internet radio station name -- ",
"Internet radio station owner - ",
"Size ------------------------- ",
"ISRC ------------------------- ",
"Software/Hardware ------------ ",
"Year ------------------------- ",
"User Defined ----------------- ",
} ;

//* "Picture Type" descriptions as defined for id3v2 standard.  *
//* (Used for both MP3 and OGG decoding.)                       *
const short PIC_TYPES = 21 ;     // number of descriptions per sub-array
const char* const pType[PIC_TYPES] = 
{  // English
  "Other",
  "32x32 pixels 'file icon' (PNG only)",
  "Other file icon",
  "Cover (front)",
  "Cover (back)",
  "Leaflet page",
  "Media (e.g. label side of CD)",
  "Lead artist/lead performer/soloist",
  "Artist/performer",
  "Conductor",
  "Band/Orchestra",
  "Composer",
  "Lyricist/text writer",
  "Recording Location",
  "During recording",
  "During performance",
  "Movie/video screen capture",
  "A bright coloured fish",
  "Illustration",
  "Band/artist logotype",
  "Publisher/Studio logotype",
} ;

//* According to Wikipedia, the following text-encoding markers are valid.     *
//* Note: As of Dec 2016, many media players cannot decode ID3v2.4.            *
enum TextEncode : char
{
   ENCODE_ASCII   = 0x00,  // ISO-8859-1 (0x20 - 0xFF plus 0x0A)
                           // (LATIN-1, Identical to ASCII for values smaller than 0x80).
   ENCODE_UTF16   = 0x01,  // UTF-16, (big-endian) encoded with BOM, ID3v2.2 and ID3v2.3.
                           // (Formerly UCS-2 (ISO/IEC 10646-1:1993, obsolete))
   ENCODE_UTF16BE = 0x02,  // UTF-16BE (big-endian) encoded without BOM, in ID3v2.4.
   ENCODE_UTF8    = 0x03,  // UTF-8 encoded in ID3v2.4.
   ENCODE_TXTERR           // Invalid text-frame encoding
} ;

//* Definitions for encoding and decoding UTF-16 data.*
const UINT usxMIN20  = 0x0010000 ;  // minimum value requiring 20-bit conversion
const UINT usxMAX20  = 0x010FFFF ;  // maximum value for 20-bit UTF-16
const UINT usxMSU16  = 0x000D800 ;  // MS unit mask value
                                    // and beginning of 16-bit reserved sequence
const UINT usxMSU16e = 0x000DBFF ;  // end of MS unit range
const UINT usxLSU16  = 0x000DC00 ;  // LS unit mask value
const UINT usxLSU16e = 0x000DFFF ;  // end of LS unit range
const UINT usxHIGH16 = 0x00E000 ;   // beginning of high-range 16-bit codepoints
const UINT usxMAX16  = 0x00FFFD ;   // maximum value for 16-bit codepoint
const UINT usxMASK10 = 0x0003FF ;   // 10-bit mask

//* If a text frame begins with ENCODE_UTF16, then the next two  *
//* bytes must be either "UTF_MSB UTF_LSB" or "UTF_LSB UTF_MSB". *
//* If not, it is not a valid Unicode-16 string.                 *
const char UTF_MSB = 0xFE ;
const char UTF_LSB = 0xFF ;

//* A valid Unicode-16 encoded string must be at least this long: *
//*             "01 FF FE xx xx" or "01 FE FF xx xx               *
//*             "02 xx xx" or "02 xx xx                           *
const int  UTF16_MIN = 5 ;
const int  UTF16BE_MIN = 3 ;

//* A valid ASCII or UTF-8 encoded string must be at least this long: *
//*             "00 xx"                                               *
const int  ASCII_MIN = 2 ;

//* Map of ISO8859-1, single-byte characters between 0x7F and 0xFF.   *
const short iso8859FIRST = 0x00A0,  // first character in array
            iso8859LAST  = 0x00FF,  // last character in array
            iso8859COUNT = 96 ;     // number of elements in array

const ULONG framehdrCNT = 10 ;  // size of frame header
class id3v2_framehdr
{
   public:

   id3v2_framehdr ( void )
   { this->reset () ; }
   ~id3v2_framehdr ( void ) {}
   void reset ( void )
   {
      this->frame_id[0] = this->frame_id[1] = this->frame_id[2] = 
      this->frame_id[3] = this->frame_id[4] = this->frame_id[5] = NULLCHAR ;
      this->frame_size = ZERO ;
      this->status_flags = this->encode_flags = ZERO ;
      this->flag_tag_pres = this->flag_file_pres = this->flag_readonly = 
      /*this->flag_d = this->flag_e = this->flag_f = this->flag_g = this->flag_h =*/ false ;
      this->flag_compress = this->flag_encrypt   = this->flag_grouped  = 
      /*this->flag_l = this->flag_m = this->flag_n = this->flag_o = this->flag_p =*/ false ;
      this->encoding = ENCODE_ASCII ;
      this->big_endian = false ;
      this->decomp = ZERO ;   // if compressed, Zlib compression is used
      this->crypto = ZERO ;   // if active, s/b a value greater than 0x80 (see ENCR frame)
      this->group_id = ZERO ; // if active, s/b a value greater than 0x80 (see GRID frame)
   }

   //* Convert a 4-byte sequence (MSB at offset 0) to an integer value. *
   //* Programmer's Note: This clunky construct avoids the C library's  *
   //* "helpful" automatic sign extension.                              *
   int intConv ( const UCHAR* ucp )
   {
      int i =    (UINT)ucp[3] 
              | ((UINT)ucp[2] << 8)
              | ((UINT)ucp[1] << 16)
              | ((UINT)ucp[0] << 24) ;
      return i ;
   }

   //* Convert a 32-bit integer into a big-endian bytes stream.   *
   //* (used for both 'frame-size' and 'decomp')                  *
   short intConv ( int ival, char* obuff )
   {
      const int bMASK = 0x000000FF ;
      short indx = ZERO ;     // return value

      obuff[indx++] = (char)((ival >> 24) & bMASK) ;
      obuff[indx++] = (char)((ival >> 16) & bMASK) ;
      obuff[indx++] = (char)((ival >>  8) & bMASK) ;
      obuff[indx++] = (char)(ival & bMASK) ;

      return indx ;
   }

   //* Convert raw byte data to gString (UTF-8) format.           *
   //* -- Assumes that 'frame_count' has been initialized.        *
   //* -- First byte of source indicates encoding:                *
   //*      ENCODE_ASCII or ENCODE_UTF16 (with BOM) or            *
   //*      ENCODE_UTF16BE (no BOM) or ENCODE_UTF8                *
   //* -- ENCODE_UTF16 requires frame_size >= 5 because"          *
   //*        "01 FF FE xx xx" or "01 FE FF xx xx"                *
   //*    is the shortest possible valid UTF-16 string.           *
   //* -- ENCODE_UTF16BE requires frame_size >= 3 because:        *
   //*        "02 xx xx"                                          *
   //*    is the shortest possible UTF16BE string.                *
   //* -- ENCODE_ASCII and ENCODE_UTF8 require frame_size >= 2    *
   //*       because: 0n xx is the shorted possible string        *
   //*                                                            *
   //* Note that if UTF-16 were written as a byte stream as it    *
   //* should have been, then there wouldn't be an issue with     *
   //* 'endian-ness'. However, we must accomodate the encoders    *
   //* who write the data as 16-bit integers on either big-endian *
   //* or little-endian hardware without regard to what they are  *
   //* actually doing.                                            *
   //* Be aware that when we _encode_, we always encode byte-wise *
   //* as big-endian.                                             *
   //*                                                            *
   //* Because the folks at the VLC project and elsewhere have    *
   //* little regard for the standard, we must do some defensive  *
   //* programming:                                               *
   //* a) Verify any claim that the source data are ASCII.        *
   //*    Note that we do not support the Latin-1 extensions      *
   //*    (ISO-8859-1). It is either pure ASCII or it is not.     *
   //* b) Verify that a UTF-16 string actually is.                *
   //* c) Verify the endian-ness of UTF-16 data.                  *
   //* d) For id3v2.4 (but not id3v2.3) text frames may consist   *
   //*    of multiple strings which are delimited by the null     *
   //*    character (00h(00h)). We believe that this is a huge    *
   //*    mistake and a bone-headed "enhancement." However, since *
   //*    they didn't consult us, we have to parse for it, and if *
   //*    found, we substitute a forward slash '/' to concatenate *
   //*    the strings for display.                                *
   //*         "This is a \0 concatenated string."                *
   //*    becomes:                                                *
   //*         "This is a / concatenated string."                 *
   //* e) If encoding error, then attempt to decode as UTF-8,     *
   //*    which has been routinely used in versions prior to      *
   //*    id3v2.4 even though it was not part of the standard.    *
   //* -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  - *
   //* Input  : rawPtr  : pointer to source byte stream           *
   //*          gsTrg   : (by reference) receives converted text  *
   //*                                                            *
   //* Return: number of wchar_t characters converted             *
   //*         or (-1) if conversion error                        *
   short txtConv ( const char* rawPtr, gString& gsTrg )
   {
      short convChars = ZERO ;         // return value

      this->encoding = rawPtr[0] ;     // text encoding

      if ( this->encoding == ENCODE_UTF16 || this->encoding == ENCODE_UTF16BE )
      {
         this->big_endian = true ;     // assume big-endian input
         if ( this->encoding == ENCODE_UTF16 )
         {
            //* Check the byte-order mark.        *
            //* If the BOM was stored backward,   *
            //* then it is little-endian encoding.*
            if ( rawPtr[1] == UTF_LSB )
               this->big_endian = false ;
         }

         if ( ((this->encoding == ENCODE_UTF16) && (this->frame_size >= UTF16_MIN)
                && (rawPtr[1] == UTF_MSB || rawPtr[1] == UTF_LSB)
                && (rawPtr[2] == UTF_MSB || rawPtr[2] == UTF_LSB))
              ||
              ((this->encoding == ENCODE_UTF16BE) && (this->frame_size >= UTF16BE_MIN)) )
         {
            short offset = this->encoding == ENCODE_UTF16 ? 3 : 1 ;
            convChars = this->utf16Conv ( &rawPtr[offset], gsTrg ) ;
         }
         else  // invalid UTF-16 encoding
         {
            convChars = this->utf8Conv ( &rawPtr[1], gsTrg ) ;
         }
      }

      else if ( this->encoding == ENCODE_ASCII )
         convChars = this->ascConv ( &rawPtr[1], gsTrg ) ;

      else     // ENCODE_UTF8 or unsupported format
         convChars = this->utf8Conv ( &rawPtr[1], gsTrg ) ;

      return convChars ;
   }  //* End txtConv() *

   //* Public Method.                                             *
   //* Encode the wchar_t (wide text) array into a form that MP3  *
   //* text frames can understand.                                *
   //*  Note that we always include the NULL terminator to the    *
   //*  text data. The standard neither demands nor forbids it.   *
   //*                                                            *
   //* Input  : enc   : inticates the type of encoding:           *
   //*                   a) ENCODE_ASCII                          *
   //*                   b) ENCODE_UTF8                           *
   //*                   c) ENCODE_UTF16 (big-endian with BOM)    *
   //*                   d) ENCODE_UTF16BE (big-endian, no BOM)   *
   //*          idIndx: index into the TextFrameID[] array (above)*
   //*          fData : receives the encoded data                 *
   //*          src   : wchar_t-encoded source text data          *
   //*                                                            *
   //* Returns: number of bytes in formatted output               *
   //*          or ZERO if encoding error                         *
   short txtEncode ( TextEncode enc, short idIndx, char* fData, const gString& src )
   {
      //* Initialize the data members *
      this->reset () ;

      //* Caller's choice of encoding (validate) *
      if ( (enc == ENCODE_ASCII)   || (enc == ENCODE_UTF16) || 
           (enc == ENCODE_UTF16BE) || (enc == ENCODE_UTF8) )
         this->encoding = enc ;
      else
         this->encoding = ENCODE_UTF16 ;
      this->big_endian = true ;           // 'true' but ignored

      //* Validate the frame ID *
      if ( (idIndx >= ZERO) && (idIndx < TEXT_FRAMES) )
      {
         gString gsid( TextFrameID[idIndx] ) ; // 4-byte Tnnn frame ID
         gsid.copy( this->frame_id, 6 ) ;

         //* Encode the text data.                                       *
         //* -- For ASCII and UTF-8, simply copy the source to target.   *
         //*    'frame_size' == number of text bytes + encoding byte     *
         //* -- For UTF-16 varients, convert from wchar_t to 16-bit data.*
         //*    'frame_size' == number of text bytes + encoding byte     *
         if ( (this->encoding == ENCODE_ASCII) || (this->encoding == ENCODE_UTF8) )
         {
            short fIndex = ZERO ;   // index into output array
            fData[fIndex++] = this->encoding ;
            src.copy( &fData[fIndex], src.utfbytes() ) ;
            this->frame_size = src.utfbytes() + 1 ;
         }
         else
            this->frame_size = this->utf16Encode ( enc, fData, src ) ;
      }
      return (short)this->frame_size ;

   }  //* End txtEncode() *

   #if 0    // FOR DEBUGGING ONLY
   //* Encode source data as UTF-16, little-endian.               *
   //* Note that for debugging purposes, we have implemented code *
   //* that can encode in a little-endian format, but this option *
   //* is not useful in production mode.                          *
   //* Note that only 'frame_size' member is initialized.         *
   short utf16leEncode ( char* fText, const gString& src )
   {
      short fIndex = ZERO ;   // index into output array

      fText[fIndex++] = ENCODE_UTF16 ;    // type of encoding
      fText[fIndex++] = (char)0x0FF ;     // LSB of Byte-Order-Mark
      fText[fIndex++] = (char)0x0FE ;     // MSB of Byte-Order-Mark

      //* Establish the source array of wchar_t (wide) characters *
      short wcnt ;   // number of source characters (including the null terminator)
      const wchar_t *wstr = src.gstr( wcnt ) ;
      UINT cx ;      // source character

      for ( short wIndex = ZERO ; wIndex < wcnt ; ++wIndex )
      {
         cx = wstr[wIndex] ;     // get a character from the input stream

         //* If the character can be encoded with a single, 16-bit value *
         if ( ((cx >= ZERO) && (cx < usxMSU16)) || 
              ((cx >= usxHIGH16) && (cx <= usxMAX16)) )
         {  //* Encode the bytes in little-endian order *
            fText[fIndex++] = cx & 0x000000FF ;
            fText[fIndex++] = (cx >> 8) & 0x000000FF ;
         }

         //* Else, character requires a pair of 16-bit values *
         else
         {
            UINT msUnit = ZERO, lsUnit = ZERO ;
            if ( (cx >= usxMIN20) && (cx <= usxMAX20) )
            {
               msUnit = ((cx - usxMIN20) >> 10) | usxMSU16 ;
               lsUnit = (cx & usxMASK10) | usxLSU16 ;
               fText[fIndex++] = msUnit & 0x000000FF ;
               fText[fIndex++] = (msUnit >> 8) & 0x000000FF ;
               fText[fIndex++] = lsUnit & 0x000000FF ;
               fText[fIndex++] = (lsUnit >> 8) & 0x000000FF ;
            }
            //* Character cannot be encoded as UTF-16. *
            //* Encode a question mark character.      *
            else
            {
               msUnit = L'?' ;
            }
         }
      }
      return ( this->frame_size = fIndex ) ;

   }  //* End utf16leEncode() *
   #endif   // FOR DEBUGGING ONLY

   //***************************************************************
   //* Methods that require controlled setup sequences are private.*
   //***************************************************************
   private:

   //* Private Method.                                            *
   //* Convert a raw ISO 8859-1 string to gString format.         *
   //* -- Assumes that 'frame_count' has been initialized.        *
   //* -- ISO 8859-1 is a group of single-byte characters.        *
   //* -- Note that the input strings are not always terminated.  *
   //* -- Note that Linux systems do not use the "Latin-1"        *
   //*    extensions of ISO8859-1 by default, so we must match    *
   //*    the Danish, German, Spanish etc. special alphabetical   *
   //*    characters (A0-FFh) explicitly.                         *
   short ascConv ( const char* rawPtr, gString& gsTrg )
   {
      gsTrg.clear() ;                        // initialize the target buffer
      int rawBytes = this->frame_size - 1 ;  // number of bytes in raw string
      wchar_t wc ;                           // 32-bit character
      for ( short ti = ZERO ; ti < rawBytes ; ++ti )
      {
         //* Printing characters of the ISO 8859-1 standard *
         if (   (((UCHAR)(rawPtr[ti]) >= 0x20) && (((UCHAR)rawPtr[ti]) <= 0x7E))
             || (rawPtr[ti] == '\n')
             || ((UCHAR)(rawPtr[ti]) >= iso8859FIRST && (UCHAR)(rawPtr[ti]) <= iso8859LAST)
            )
         {
            wc = (wchar_t)((UCHAR)(rawPtr[ti])) ;
            gsTrg.append( wc ) ;
         }
         //* If a null terminator is encountered BEFORE end-of-text,         *
         //* concatenate the separate strings in the field. (see note above) *
         else if ( (rawPtr[ti] == NULLCHAR) && (ti < (rawBytes - 1)) && (rawPtr[ti - 1] != ' ') )
         {
            gsTrg.append( fSLASH ) ;
         }
         //* Control character OR a character not defined in ISO 8859-1.*
         else if ( rawPtr[ti] != NULLCHAR ) { wc = L'?' ; gsTrg.append( wc ) ; }
      }
      return ( gsTrg.gschars() ) ;
   }  //* End ascConv() *

   //* Private Method.                                            *
   //* Convert an (assumed) UTF-8 string to gString format.       *
   //* Note that UTF-8 is not supported by id3v2.3 but is         *
   //* supported by id3.v2.4.                                     *
   //* -- Assumes that 'frame_count' has been initialized.        *
   //* -- Assumes that 'rawPtr' points to head of text (not       *
   //*    encoding byte)                                          *
   //* -- The string may or may not be terminated.                *
   //* -- Returns number of wchar_t (wide) characters created.    *

   short utf8Conv ( const char* rawPtr, gString& gsTrg )
   {
      char cbuff[this->frame_size] ;
      int ti,                                // source index
          rawBytes = this->frame_size - 1 ;  // number of bytes in raw string
      gsTrg.clear() ;                        // initialize the target buffer
      for ( ti = ZERO ; ti < rawBytes ; ++ti ) // replace non-terminal terminators
      {
         if ( rawPtr[ti] != '\0' )
            cbuff[ti] = rawPtr[ti] ;
         else           // null terminator encountered before end-of-text
         {              // concatenate the strings (see note above)
            if ( ti < (rawBytes - 1) )
               cbuff[ti] = '/' ;
            else        // end of text
               break ;
         }
      }
      cbuff[ti] = '\0' ;                     // be sure string is terminated
      gsTrg = cbuff ;
      return ( gsTrg.gschars() ) ;
   }  //* End utf8Conv() *

   //* Private Method:                                            *
   //* Convert a Unicode-16 string (big-endian or little-endian)  *
   //* to gString format.                                         *
   //* -- Assumes that 'frame_count' has been initialized.        *
   //* -- Note that the frame may contain multiple strings, but   *
   //*    if so, we concatenate them. It is assumed that 2nd and  *
   //*    subsequent strings have the same byte order as the 1st. *
   //* -- Unicode-16 strings are generally not terminated (except *
   //*    for empty strings), but if multiple strings, then       *
   //*    obviously, all but the last must be terminated.         *
   //* -- See additional information on decoding UTF-16 in the    *
   //*    mptDecodetextFrame() method header.                     *
   //* -- Note that because the wchar_t type is a signed integer, *
   //*    the compiler wants to sign-extend. Don't allow this.    *
   //* -- Returns number of wchar_t (wide) characters created.    *

   short utf16Conv ( const char* rawPtr, gString& gsTrg )
   {
      int ti = ZERO ;                        // source index
      gsTrg.clear() ;                        // initialize the target buffer
      int rawBytes = this->frame_size        // number of bytes in raw string
                     - ((this->encoding == ENCODE_UTF16) ? 3 : 1) ;
      UINT msUnit, lsUnit,                   // for converting unit pairs
           cx ;                              // undecoded 16-bit character
      while ( rawBytes > ZERO )
      {
         if ( this->big_endian )
         {
            cx = ((UINT(rawPtr[ti++]) << 8) & 0x0000FF00) ;
            cx |= (UINT(rawPtr[ti++]) & 0x000000FF) ;
         }
         else  // (little-endian)
         {
            cx = (UINT(rawPtr[ti++]) & 0x000000FF) ;
            cx |= ((UINT(rawPtr[ti++]) << 8) & 0x0000FF00) ;
         }
         rawBytes -= 2 ;
         //* If the character is fully represented by 16 bits *
         //* (most characters are)                            *
         if ( ((cx >= ZERO) && (cx < usxMSU16)) || 
              ((cx >= usxHIGH16) && (cx <= usxMAX16)) )
         {
            if ( cx != 0 )
               gsTrg.append( cx ) ;
            else if ( rawBytes > ZERO ) // if interim NULLCHAR found
            {
               gsTrg.append( fSLASH ) ; // string concatenation (see note above)
               //* Note that the following string may also have a *
               //* byte-order-mark (BOM). If so, step over it.    *
               if ( (rawBytes >= 2) &&
                    ((rawPtr[ti] == UTF_MSB && rawPtr[ti + 1] == UTF_LSB)
                     ||
                     (rawPtr[ti] == UTF_LSB && rawPtr[ti + 1] == UTF_MSB)) )
               {
                  rawBytes -= 2 ;
                  ti += 2 ;
               }
            }
         }

         //* Character is represented by 32 bits    *
         //* (20 bits actually used), 16 MSBs first.*
         else
         {
            msUnit = cx ;
            lsUnit = ZERO ;
            if ( this->big_endian )
            {
               lsUnit = ((UINT(rawPtr[ti++]) << 8) & 0x0000FF00) ;
               lsUnit |= (UINT(rawPtr[ti++]) & 0x000000FF) ;
            }
            else     // (little-endian)
            {
               lsUnit = (UINT(rawPtr[ti++]) & 0x000000FF) ;
               lsUnit |= ((UINT(rawPtr[ti++]) << 8) & 0x0000FF00) ;
            }
            rawBytes -= 2 ;

            //* Validate the range *
            if ( ((msUnit >= usxMSU16) && (msUnit <= usxMSU16e))
                 &&
                 ((lsUnit >= usxLSU16) && (lsUnit <= usxLSU16e)) )
            {
               cx = usxMIN20 + ((msUnit & usxMASK10) << 10) ;
               cx |= (lsUnit & usxMASK10) ;
               if ( (cx == 0) && rawBytes > ZERO )
               {
                  cx = fSLASH ;   // string concatenation (see note above)
                  //* Note that the following string may also have a *
                  //* byte-order-mark (BOM). If so, step over it.    *
                  if ( (rawBytes >= 2) &&
                       ((rawPtr[ti] == UTF_MSB && rawPtr[ti + 1] == UTF_LSB)
                        ||
                        (rawPtr[ti] == UTF_LSB && rawPtr[ti + 1] == UTF_MSB)) )
                  {
                     rawBytes -= 2 ;
                     ti += 2 ;
                  }
               }
            }
            else                 // invalid UTF-16 codepoint
               cx = L'?' ;
            gsTrg.append( cx ) ; // add the chararacter to output buffer
         }
      }
      return ( gsTrg.gschars() ) ;

   }  //* End utf16Conv() *

   //* Private Method.                                            *
   //* Convert the wchar_t string to a UTF-16-encoded             *
   //* (big-endian) byte stream.                                  *
   //*                                                            *
   //* Input  : enc   : inticates the type of encoding:           *
   //*                   a) ENCODE_UTF16 (big-endian with BOM)    *
   //*                   b) ENCODE_UTF16BE (big-endian, no BOM)   *
   //*          fText : receives the encoded data                 *
   //*          src   : wchar_t-encoded source text data          *
   //*                                                            *
   //* Returns: number of bytes in formatted output               *
   //*          or (-1) if encoding error                         *
   //* -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  - *
   //* Note: UTF-16 may be encoded on either big-endian or        *
   //* little-endian hardware; however, we encode the UTF-16 as   *
   //* a byte stream, so our output is always encoded as          *
   //* big-endian.                                                *
   short utf16Encode ( TextEncode enc, char* fText, const gString& src )
   {
      short fIndex = ZERO ;   // index into output array

      if ( enc == ENCODE_UTF16 )
      {
         fText[fIndex++] = ENCODE_UTF16 ;    // type of encoding
         fText[fIndex++] = (char)0x0FE ;     // MSB of Byte-Order-Mark
         fText[fIndex++] = (char)0x0FF ;     // LSB of Byte-Order-Mark
      }
      else
         fText[fIndex++] = ENCODE_UTF16BE ;  // type of encoding

      //* Establish the source array of wchar_t (wide) characters *
      int   wcnt ;   // number of source characters (including the null terminator)
      const wchar_t *wstr = src.gstr( wcnt ) ;
      UINT cx ;      // source character

      for ( short wIndex = ZERO ; wIndex < wcnt ; ++wIndex )
      {
         cx = wstr[wIndex] ;     // get a character from the input stream

         //* If the character can be encoded with a single, 16-bit value *
         if ( ((cx >= ZERO) && (cx < usxMSU16)) || 
              ((cx >= usxHIGH16) && (cx <= usxMAX16)) )
         {  //* Encode the bytes in big-endian order *
            fText[fIndex++] = (cx >> 8) & 0x000000FF ;
            fText[fIndex++] = cx & 0x000000FF ;
         }

         //* Else, character requires a pair of 16-bit values *
         else
         {
            UINT msUnit = ZERO, lsUnit = ZERO ;
            if ( (cx >= usxMIN20) && (cx <= usxMAX20) )
            {
               msUnit = ((cx - usxMIN20) >> 10) | usxMSU16 ;
               lsUnit = (cx & usxMASK10) | usxLSU16 ;
               fText[fIndex++] = (msUnit >> 8) & 0x000000FF ;
               fText[fIndex++] = msUnit & 0x000000FF ;
               fText[fIndex++] = (lsUnit >> 8) & 0x000000FF ;
               fText[fIndex++] = lsUnit & 0x000000FF ;
            }
            //* Character cannot be encoded as UTF-16. *
            //* Encode a question mark character.      *
            else
            {
               msUnit = L'?' ;
            }
         }
      }
      return ( this->frame_size = fIndex ) ;

   }  //* End utf16Encode() *


   //********************************
   //* All data members are public. *
   //********************************
   public:
   char  frame_id[6] ;     // S/B one of the 4-character names defined by the standard
   int   frame_size ;      // size of the frame in bytes
   char  status_flags ;    // status bits %abc0.0000
   char  encode_flags ;    // encoding flags %ijk0.0000
   bool  flag_tag_pres ;   // if 'true',  preserve frame if tag is modified
                           // if 'false', discard frame if tag is modified
   bool  flag_file_pres ;  // if 'true',  preserve frame if non-tag data modified
                           // if 'false', discard frame if non-tag data is modified
   bool  flag_readonly ;   // if 'true',  frame is read-only unless software know what it's doing
   bool  flag_compress ;   // if 'true',  frame is compressed
                           //    (4 bytes "decompressed size" added to frame header)
                           // if 'false', frame is not compressed
   bool  flag_encrypt ;    // if 'true',  frame is encrypted
                           //    (1 byte "encryption type"" added to frame header)
                           // if 'false', frame is not encrypted
   bool  flag_grouped ;    // if 'true',  frame is a member of a frame group
                           //    (1 byte "group identifier" added to frame header)
                           // if 'false', frame is not grouped
   char  encoding ;        // for text frames only: text encoding
                           // 00h == ISO8859-1 (ASCII, i.e. Latin-1: 0x20 - 0xFF)
                           // 01h == UTF-16 (with BOM)
                           // 02h == UTF-16 big-endian (no BOM) (id3v2.4 only)
                           // 03h == UTF-8 encoding (id3v2.4 only)
   bool  big_endian ;      // for UTF-16 text frames only
                           // if 'true', big-endian Unicode-16
                           // if 'false', little-endian Unicode-16
   int   decomp ;          // for compressed frames only: expanded size
   char  crypto ;          // for encrypted frames only: type of encryption
   char  group_id ;        // for grouped frames only: group ID

} ;   // id3v2_framehdr

//* Flag byte for id3v2_taghdr class *
//* Four bits are defined: Bits 7-4  *
//*  bit7 unsynch
//*  bit6 extended header follows
//*  bit5 experimental tag
//*  bit4 tag footer terminates the tag (defined in id3v2.4)
//*       (bits 6 and 4 are mutually exclusive)
//*  bits 3-0 currently undefined (always reset)
// Programmer's Note: We could have the same functionality 
// using a union with bit fields, but this is more fun.
class taghdrFlags
{
   public:
   ~taghdrFlags ( void ) {}         // destructor
   taghdrFlags ( void )             // default constructor
   { this->reset () ; }
   taghdrFlags ( UCHAR init )       // initialization constructor
   { this->reset () ; this->Flags = init ; }
   void reset ( void )
   { this->Flags = ZERO ; }
   UCHAR setflags ( UCHAR reinit )  // Initialize all flags (requires intelligence)
   { return ( (UCHAR)((this->Flags = UINT(reinit & 0x000000F0))) ) ; }
   UCHAR getflags ( void ) const    // Returns flag byte
   { return (UCHAR)(this->Flags) ; }

   bool unsynch ( void ) const      // Bit7 - unsynch
   { return bool(this->Flags & 0x080) ; }
   void unsynch ( bool set )
   { 
      if ( set ) this->Flags |= 0x080 ;
      else       this->Flags &= 0x07F ;
   }

   bool exthdr ( void ) const       // Bit6 - exthdr
   { return bool(this->Flags & 0x040) ; }
   void exthdr ( bool set )
   { 
      if ( set ) this->Flags |= 0x040 ;
      else       this->Flags &= 0x0BF ;
   }

   bool exper ( void ) const        // Bit5 - exper
   { return bool(this->Flags & 0x020) ; }
   void exper ( bool set )
   { 
      if ( set ) this->Flags |= 0x020 ;
      else       this->Flags &= 0x0DF ;
   }

   bool footer ( void ) const       // Bit4 - footer
   { return bool(this->Flags & 0x010) ; }
   void footer ( bool set )
   { 
      if ( set ) this->Flags |= 0x010 ;
      else       this->Flags &= 0x0EF ;
   }

   //* Data Members *
   UINT Flags ;
} ;

//* Decoded data from an ID3v2.x Tag Header.           *
//* The tag header lives at the top of every MP3 file. *
const UINT taghdrCNT = 10 ;   // size of tag header
const UINT exthdrCNT = 6 ;    // size of extended header (if present)
class id3v2_taghdr
{
   public:
   id3v2_taghdr ( void )
   { this->reset () ; }
   ~id3v2_taghdr ( void ) {}
   void reset ( void )
   {
      this->file_id[0] = this->file_id[1] = this->file_id[2] = this->file_id[3] = 
      this->major = this->rev = ZERO ;
      this->flags.reset() ;
      this->tag_size = ZERO ;
      this->exBytes = this->padBytes = this->tagCRC = ZERO ;
      this->exFlags = ZERO ;
      this->crcFlag = false ;
   }

   void setHeader ( void )
   {
      this->file_id[0] = 'I' ;
      this->file_id[1] = 'D' ;
      this->file_id[2] = '3' ;
      this->file_id[3] = NULLCHAR ;
      this->major      = 3 ;     // id3v2.3.0
      this->rev        = 0 ;
   }

   //* Convert a 4-byte sequence (MSB at offset 0) to an integer value. *
   //* Integer data are stored MSBit first in the bytes a,d MSByte first*
   //* in multi-byte integers, i.e. big-endian.                         *
   //* Programmer's Note: This clunky construct avoids the C library's  *
   //* "helpful" automatic sign extension.                              *
   int intConv ( const UCHAR* ucp ) const
   {
      int i =    (UINT)ucp[3] 
              | ((UINT)ucp[2] << 8)
              | ((UINT)ucp[1] << 16)
              | ((UINT)ucp[0] << 24) ;
      return i ;
   }

   //* Convert a 32-bit integer into an 4-byte, big-endian binary stream *
   int intConv ( int val, UCHAR* ucp ) const
   {
      int i = ZERO ;
      ucp[i++] = (UCHAR)((val >> 24) & 0x000000FF) ;
      ucp[i++] = (UCHAR)((val >> 16) & 0x000000FF) ;
      ucp[i++] = (UCHAR)((val >>  8) & 0x000000FF) ;
      ucp[i++] = (UCHAR)((val      ) & 0x000000FF) ;
      return i ;
   }

   //* Convert a 2-byte sequence (MSB at offset 0) to a short integer value. *
   short intConv16 ( const UCHAR* ucp ) const
   {
      short s =    (short)ucp[1]
                | ((short)ucp[0] << 8) ;
      return s ;
   }

   //* Convert a 16-bit integer into an 2-byte, big-endian binary stream *
   int intConv ( short val, UCHAR* ucp ) const
   {
      int i = ZERO ;
      ucp[i++] = (UCHAR)((val >>  8) & 0x00FF) ;
      ucp[i++] = (UCHAR)((val      ) & 0x00FF) ;
      return i ;
   }

   //* Decode the formatted tag size, a 28-bit value.         *
   //* Four(4) raw hex bytes with bit7 of each byte reset.    *
   //* Example: $00 00 02 01 == 257 byte (256 + 1) tag length.*
   int decodeTagSize ( const char* rawPtr )
   {
      this->tag_size = ZERO ;
      if ( !(rawPtr[0] & 0x80) && !(rawPtr[1] & 0x80) &&
           !(rawPtr[2] & 0x80) && !(rawPtr[3] & 0x80) )
      {
         this->tag_size =    int(rawPtr[3])
                          + (int(rawPtr[2]) << 7)
                          + (int(rawPtr[1]) << 14)
                          + (int(rawPtr[0]) << 21) ;
      }
      return this->tag_size ;
   }

   //* Convert the integer tag size (byte count) to a byte stream.*
   int encodeTagSize ( char* obuff ) const
   {
      const int bMASK = 0x0000007F ;
      short indx = ZERO ;
      obuff[indx++] = (this->tag_size >> 21) & bMASK ;
      obuff[indx++] = (this->tag_size >> 14) & bMASK ;
      obuff[indx++] = (this->tag_size >>  7) & bMASK ;
      obuff[indx++] = this->tag_size & bMASK ;
      return indx ;
   }  //* End encodeTagSize() *

   char  file_id[4] ;      // S/B "ID3", if not, then no metadata
   char  major ;           // ID3 major version (hex)
   char  rev ;             // ID3 revision (hex)
   taghdrFlags flags ;     // flag byte

   //* Size of the tag record EXCLUDING the 10-byte tag header *
   int   tag_size ;

   //* Extended Header (if present) *
   int   exBytes ;         // (32-bit int) (indicated header size excludes this value)
                           // currently: six(6) bytes without CRC or
                           //            ten(10) bytes with CRC 
   int   padBytes ;        // bytes of reserved tag space i.e. the padding
   int   tagCRC ;          // 32-bit CRC value (if CRC used)
                           // Note: CRC is calculated over the range from AFTER the 
                           // extended header to the beginning of the padding i.e.
                           // only the actual frame data (not headers and not padding)
   short exFlags ;         // (all except MSB currently unused)
   bool  crcFlag ;         // true if CRC is used
   bool  spare ;           // (unused)

} ;   // id3v2_taghdr

//* From the ID3v2.4 specification: section 3.4:                               *
//* "To speed up the process of locating an ID3v2 tag when searching from      *
//*  the end of a file, a footer can be added to the tag. It is REQUIRED       *
//*  to add a footer to an appended tag, i.e. a tag located after all          *
//*  audio data. The footer is a copy of the header, but with a different      *
//*  identifier."                                                              *
//*                  ID3v2 identifier           "3DI"                          *
//*                  ID3v2 version              $04 00                         *
//*                  ID3v2 flags                %abcd0000                      *
//*                  ID3v2 size             4 * %0xxxxxxx                      *
//*  a) The size of the footer is the same as the header: taghdrCNT (10 bytes).*
//*  a) A tag footer is the same as a tag header except with a different       *
//*     identifier ("3DI" vs. "ID3").                                          *
//*  b) The 'size' is the same 28-bit format as the header                     *
//*  c) There must be no tag padding if a footer is present.                   *
//*  d) If a "SEEK" frame is found in the prepended tag, then use it to scan   *
//*     for additional tag(s).                                                 *
//*  e) If the continuation flag is set in the extended header, it indicates   *
//*     a continuation of the tag data.                                        *
//*                                                                            *
//*                                                                            *
//*                                                                            *

class id3v2_tagfooter
{
   public:
   ~id3v2_tagfooter ( void ) {}
   id3v2_tagfooter ( void )
   {
      this->reset() ;
   }
   void reset ( void )
   {
      this->foot_id[0] = this->foot_id[1] = this->foot_id[2] = this->foot_id[3] = 
      this->major = this->rev = /*this->flags = */ZERO ;
      this->flags.reset() ;
      this->tag_size = ZERO ;
   }

   void setFootID ( void )
   {
      this->foot_id[0] = '3' ;
      this->foot_id[1] = 'D' ;
      this->foot_id[2] = 'I' ;
      this->foot_id[3] = NULLCHAR ;
   }

   //* Decode the formatted tag size. *
   int decodeTagSize ( const char* rawPtr )
   {
      this->tag_size = ZERO ;
      if ( !(rawPtr[0] & 0x80) && !(rawPtr[1] & 0x80) &&
           !(rawPtr[2] & 0x80) && !(rawPtr[3] & 0x80) )
      {
         this->tag_size =    int(rawPtr[3])
                          + (int(rawPtr[2]) << 7)
                          + (int(rawPtr[1]) << 14)
                          + (int(rawPtr[0]) << 21) ;
      }
      return this->tag_size ;
   }

   //* Convert the integer tag size (byte count) to a byte stream.*
   int encodeTagSize ( char* obuff ) const
   {
      const int bMASK = 0x0000007F ;
      short indx = ZERO ;
      obuff[indx++] = (this->tag_size >> 21) & bMASK ;
      obuff[indx++] = (this->tag_size >> 14) & bMASK ;
      obuff[indx++] = (this->tag_size >>  7) & bMASK ;
      obuff[indx++] = this->tag_size & bMASK ;
      return indx ;
   }  //* End encodeTagSize() *

   char  foot_id[4] ;      // S/B "3DI"
   char  major ;           // ID3 major version (hex)
   char  rev ;             // ID3 revision (hex)
   taghdrFlags flags ;     // flag byte

   //* Size of the tag record EXCLUDING the 10-byte tag header *
   //* -- four(4) raw hex bytes with bit7 of each byte ignored.*
   //*    Thus $00 00 02 01 == 257 byte (256 + 1) tag length.  *
   //* -- This value is the DECODED tag size.                  *
   int   tag_size ;
} ;   // id3v2_tagfooter

//* From the ID3v2.4 specification: section 4.15                               *
//* Data contained in an 'APIC' image frame.                                   *
//* <Header for 'Attached picture', ID: "APIC">                                *
//* Text encoding   $xx                                                        *
//* MIME type       <text string> $00                                          *
//* Picture type    $xx                                                        *
//* Description     <text string according to encoding> $00 (00)               *
//* Picture data    <binary data>                                              *
const short imgMAX_TYPE = 64 ;         // Max bytes in type strings
const short imgMAX_DESC = 256 ;        // Max bytes in description string

class id3v2_image
{
   public:
   ~id3v2_image ( void ) {}
   id3v2_image ( void )
   {
      this->reset() ;
   }
   void reset ( void )
   {
      //* NOTE: If 'picPath' points to a temp file, *
      //*       the reset orphans the file.         *
      this->picPath[0] = this->txtDesc[0] = 
      this->mimType[0] = this->picExpl[0] = NULLCHAR ;
      this->picSize = ZERO ;
      this->picType = 0x00 ;           // default to "Other"
      this->encoding = ENCODE_ASCII ;  // default to ASCII
   }

   //* Convert raw byte data to gString (UTF-8) format.           *
   //*                                                            *
   //* Programmer's Note: For the image description, we take a    *
   //* shortcut in decoding ISO8859-1 text. If the data contain   *
   //* "Latin-1" extension characters, we will incorrectly decode *
   //* them because we assume pure ASCII or UTF-8 encoding.       *
   //* For full decoding of ISO8859-1 characters, see the         *
   //* id3v2_framehdr class definition.                           *
   //*                                                            *
   //* Input  : rawPtr  : pointer to source byte stream           *
   //*                    Note that string _should be_ null       *
   //*                    terminated, but we verify.              *
   //*          enc     : contains the text encoding type.        *
   //*                                                            *
   //* Return: number of wchar_t characters converted             *
   //*         or (-1) if conversion error                        *
   short txtConv ( const char* rawPtr, TextEncode enc )
   {
      short convChars = ZERO ;         // return value

      //* Range check and set text encoding *
      if ( (enc == ENCODE_ASCII) || (enc == ENCODE_UTF16) || 
           (enc == ENCODE_UTF16BE) || (enc == ENCODE_UTF8) )
         this->encoding = enc ;
      else
         this->encoding = ENCODE_UTF8 ;

      //* If ASCII encoding was specified, verify that the *
      //* data are actually ASCII. If not, assume UTF-8.   *
      if ( this->encoding == ENCODE_ASCII )
      {
         gString gs( rawPtr, imgMAX_DESC ) ;
         if ( !(gs.isASCII()) )
            this->encoding = ENCODE_UTF8 ;
      }

      if ( (this->encoding == ENCODE_UTF16) || (this->encoding == ENCODE_UTF16BE) )
      {
         // Programmer's Note: If the text encoding byte is wrong or if the BOM
         // is missing or otherwise invalid, we may produce garbage output, but
         // we rely on the presence of the null terminator to stop the madness.
         int ti = ZERO ;               // source index
         gString gs ;                  // temp buffer
         UINT msUnit, lsUnit,          // for converting unit pairs
              cx ;                     // undecoded 16-bit character
         bool big_endian = true ;      // assume big-endian input
         if ( this->encoding == ENCODE_UTF16 )
         {
            //* Check the byte-order mark.        *
            //* If the BOM was stored backward,   *
            //* then it is little-endian encoding.*
            if ( rawPtr[0] == UTF_LSB )
               big_endian = false ;
            if ( (rawPtr[0] == UTF_MSB || rawPtr[0] == UTF_LSB) &&
                 (rawPtr[1] == UTF_MSB || rawPtr[1] == UTF_LSB) )
               ti += 2 ;
         }
         do
         {
            if ( big_endian )
            {
               cx = ((UINT(rawPtr[ti++]) << 8) & 0x0000FF00) ;
               cx |= (UINT(rawPtr[ti++]) & 0x000000FF) ;
            }
            else
            {
               cx = (UINT(rawPtr[ti++]) & 0x000000FF) ;
               cx |= ((UINT(rawPtr[ti++]) << 8) & 0x0000FF00) ;
            }

            //* If the character is fully represented by 16 bits *
            //* (most characters are)                            *
            if ( ((cx >= ZERO) && (cx < usxMSU16)) || 
                 ((cx >= usxHIGH16) && (cx <= usxMAX16)) )
            {
               if ( cx != ZERO )    // add the chararacter to output buffer
                  gs.append( cx ) ;
            }
            //* Character is represented by 32 bits    *
            //* (20 bits actually used), 16 MSBs first.*
            else
            {
               msUnit = cx ;
               lsUnit = ZERO ;
               if ( big_endian )
               {
                  lsUnit = ((UINT(rawPtr[ti++]) << 8) & 0x0000FF00) ;
                  lsUnit |= (UINT(rawPtr[ti++]) & 0x000000FF) ;
               }
               else     // (little-endian)
               {
                  lsUnit = (UINT(rawPtr[ti++]) & 0x000000FF) ;
                  lsUnit |= ((UINT(rawPtr[ti++]) << 8) & 0x0000FF00) ;
               }
   
               //* Validate the range *
               if ( ((msUnit >= usxMSU16) && (msUnit <= usxMSU16e))
                    &&
                    ((lsUnit >= usxLSU16) && (msUnit <= usxLSU16e)) )
               {
                  cx = usxMIN20 + ((msUnit & usxMASK10) << 10) ;
                  cx |= (lsUnit & usxMASK10) ;
               }
               else                 // invalid UTF-16 codepoint
                  cx = L'?' ;
               if ( cx != ZERO )    // add the chararacter to output buffer
                  gs.append( cx ) ;
            }
         }
         while ( cx != ZERO ) ;
         convChars = gs.gschars() ;
         gs.copy( this->txtDesc, imgMAX_DESC ) ;
      }

      else if ( (this->encoding == ENCODE_ASCII) || (this->encoding == ENCODE_UTF8) )
      {
         gString gs( rawPtr, imgMAX_DESC ) ;
         convChars = gs.gschars() ;
         gs.copy( this->txtDesc, imgMAX_DESC ) ;
      }
      return convChars ;
   }  //* End txtConv() *

   //* Public Method.                                             *
   //* Encode the wchar_t (wide text) array into a form that MP3  *
   //* text frames can understand.                                *
   //*  Note that we always include the NULL terminator to the    *
   //*  text data. The standard neither demands nor forbids it.   *
   //*                                                            *
   //* Input  : enc   : inticates the type of encoding:           *
   //*                   a) ENCODE_ASCII                          *
   //*                   b) ENCODE_UTF8                           *
   //*                   c) ENCODE_UTF16 (big-endian with BOM)    *
   //*                   d) ENCODE_UTF16BE (big-endian, no BOM)   *
   //*          fData : receives the encoded data                 *
   //*          src   : wchar_t-encoded source text data          *
   //*                                                            *
   //* Returns: number of bytes in formatted output               *
   //*          or ZERO if encoding error                         *
   short txtEncode ( TextEncode enc, char* fData, const gString& src )
   {
      gString srcx = src ;
      srcx.limitChars( imgMAX_DESC ) ; // standard limits size to 64 "characters"
      short fIndex = ZERO ;

      //* Caller's choice of encoding (validate) *
      if ( !(enc == ENCODE_ASCII)   && !(enc == ENCODE_UTF16) && 
           !(enc == ENCODE_UTF16BE) && !(enc == ENCODE_UTF8) )
         enc = ENCODE_UTF16 ;

      if ( (this->encoding == ENCODE_ASCII) || (this->encoding == ENCODE_UTF8) )
      {
         srcx.copy( &fData[fIndex], srcx.utfbytes() ) ;
         fIndex = srcx.utfbytes() ;
      }
      else
      {
         if ( enc == ENCODE_UTF16 )
         {
            fData[fIndex++] = (char)0x0FE ;     // MSB of Byte-Order-Mark
            fData[fIndex++] = (char)0x0FF ;     // LSB of Byte-Order-Mark
         }
         int   wcnt ;   // number of source chars (incl. null)
         const wchar_t *wstr = srcx.gstr( wcnt ) ;
         UINT cx ;      // source character

         for ( short wIndex = ZERO ; wIndex < wcnt ; ++wIndex )
         {
            cx = wstr[wIndex] ;     // get a character from the input stream

            //* If the character can be encoded with a single, 16-bit value *
            if ( ((cx >= ZERO) && (cx < usxMSU16)) || 
                 ((cx >= usxHIGH16) && (cx <= usxMAX16)) )
            {  //* Encode the bytes in big-endian order *
               fData[fIndex++] = (cx >> 8) & 0x000000FF ;
               fData[fIndex++] = cx & 0x000000FF ;
            }

            //* Else, character requires a pair of 16-bit values *
            else
            {
               UINT msUnit = ZERO, lsUnit = ZERO ;
               if ( (cx >= usxMIN20) && (cx <= usxMAX20) )
               {
                  msUnit = ((cx - usxMIN20) >> 10) | usxMSU16 ;
                  lsUnit = (cx & usxMASK10) | usxLSU16 ;
                  fData[fIndex++] = (msUnit >> 8) & 0x000000FF ;
                  fData[fIndex++] = msUnit & 0x000000FF ;
                  fData[fIndex++] = (lsUnit >> 8) & 0x000000FF ;
                  fData[fIndex++] = lsUnit & 0x000000FF ;
               }
               //* Character cannot be encoded as UTF-16. *
               //* Encode a question mark character.      *
               else
               {
                  cx = L'?' ;
                  fData[fIndex++] = (cx >> 8) & 0x000000FF ;
                  fData[fIndex++] = cx & 0x000000FF ;
               }
            }
         }     // for(;;)
      }
      return fIndex ;

   }  // txtEncode()

   //* Convert the image info to a byte stream.      *
   //* Note that caller will append the binary image *
   //* data at the returned insertion point.         *
   short encode ( char* obuff )
   {
      short oindx = ZERO ;
      gString gs( this->txtDesc ) ;
      this->encoding = (gs.isASCII()) ? ENCODE_ASCII : ENCODE_UTF16 ;
      obuff[oindx++] = this->encoding ;
      for ( short i = ZERO ; i < imgMAX_TYPE ; ++i )
      {
         obuff[oindx++] = this->mimType[i] ;
         if ( this->mimType[i] == NULLCHAR )
            break ;
      }
      obuff[oindx++] = this->picType ;
      oindx += this->txtEncode( this->encoding, &obuff[oindx], gs ) ;

      return oindx ;
   }

   //* Encode the image header data into a byte stream.*

   //***********************
   //* Public Data Members *
   //***********************
   char picPath[gsDFLTBYTES] ;      // path/filename of external image file
   char txtDesc[imgMAX_DESC] ;      // text description of image (optional)
   char mimType[imgMAX_TYPE] ;      // MIME type: generally "image/jpg" or 
                                    //   "image/png" or "image/" (default)
   char picExpl[imgMAX_DESC] ;      // explanation of 'picType'. Note: Array of
                                    // pre-defined description strings: pType[][x]
   int  picSize ;                   // image size in bytes
   UCHAR picType ;                  // picture type code
   TextEncode encoding ;            // text encoding (member of enum TextEncode)

} ;   // id3v2_image

const char* const POP_TAG = "POPM" ;   // Tag header code for a Popularimeter.
const char* const CNT_TAG = "PCNT" ;   // Tag header code for a Play Counter.

//* Tag data for the POPM 'Popularimeter' and PCNT 'Play Counter'
class popMeter
{
   public:
   popMeter ( void )
   {
      this->reset () ;
   }
   void reset ( void )
   {
      this->popEmail[0] = NULLCHAR ;
      this->popCount = this->playCount = ZERO ;
      this->popStar = ZERO ;
      this->popdata = this->cntdata = this->pcMod = false ;
   }

   //* Convert a 4-byte sequence (MSB at offset 0) to an integer value. *
   //* Programmer's Note: This clunky construct avoids the C library's  *
   //* "helpful" automatic sign extension.                              *
   UINT intConv ( const UCHAR* ucp ) const
   {
      UINT i =    (UINT)ucp[3] 
               | ((UINT)ucp[2] << 8)
               | ((UINT)ucp[1] << 16)
               | ((UINT)ucp[0] << 24) ;
      return i ;
   }

   //* Convert a 32-bit integer into a big-endian byte stream.    *
   //* (used for both 'frame-size' and 'decomp')                  *
   short intConv ( UINT uval, char* obuff ) const
   {
      const int bMASK = 0x000000FF ;
      short indx = ZERO ;     // return value

      obuff[indx++] = (char)((uval >> 24) & bMASK) ;
      obuff[indx++] = (char)((uval >> 16) & bMASK) ;
      obuff[indx++] = (char)((uval >>  8) & bMASK) ;
      obuff[indx++] = (char)(uval & bMASK) ;

      return indx ;
   }

   char  popEmail[gsDFLTBYTES] ; // User's email address, etc.
   UINT  popCount ;     // Popularimeter (personal) play counter
   UINT  playCount ;    // Play-count counter
   UCHAR popStar ;      // Popularimeter value (star rating)
   bool  popdata ;      // 'true' if POPM data read from source file OR entered by user
   bool  cntdata ;      // 'true' if PCNT data read from source file OR entered by user
   bool  pcMod ;        // 'true' if user has modified the data
} ;   // popMeter

//*****************************
//* Local, non-member methods *
//*****************************
static short mptDecodeFrameHeader ( ifstream& ifs, id3v2_framehdr& fhdr, 
                                    id3v2_tagfooter& tftr ) ;
static UINT  mptDecodeTextFrame ( ifstream& ifs, id3v2_framehdr& fhdr, 
                                  gString& gsOut, TextEncode& enc ) ;
static UINT  mptDecodeImageFrame ( ifstream& ifs, const id3v2_framehdr& fhdr, 
                                  id3v2_image& imgData ) ;
static UINT  mptDecodePopFrame ( ifstream& ifs, const id3v2_framehdr& fhdr, 
                                 popMeter& popData ) ;



//******************************************************************************
//**                Definitions for Ogg/Vorbis audio files                    **
//******************************************************************************
//* Notes on Ogg/Vorbis formatting:                                            *
//* 1) Page Header (https://xiph.org/vorbis/doc/framing.html)                  *
//*     00-03   "OggS"                                                         *
//*     04      stream structure version (always 00h)                          *
//*     05      header-type flags:                                             *
//*             0x01: unset = fresh packet                                     *
//*                     set = continued packet                                 *
//*             0x02: unset = not first page of logical bitstream              *
//*                     set = first page of logical bitstream (bos)            *
//*             0x04: unset = not last page of logical bitstream               *
//*                     set = last page of logical bitstream (eos)             *
//*             (remaining bits reserved)                                      *
//*     06-13   absolute granule position (little-endian, LSByte first)        *
//*     14-17   stream serial number (little-endian)                           *
//*     18-21   page sequence number (little-endian)                           *
//*     22-25   page checksum (little-endian)                                  *
//*     26      page segment count (0-255 segments)                            *
//*     27 ...  segment table, one byte for each segment, i.e. table is        *
//*             page-segment-count bytes long                                  *
//*     -- Page size is the size of the header plus the size of the segment    *
//*        table plus the size of the data segments described in the segment   *
//*        table.                                                              *
//*     -- Length of last segment is always < 255 bytes.                       *
//*                                                                            *
//*  2) The first page of an Ogg stream contains the Identification header.    *
//*       (https://xiph.org/vorbis/doc/Vorbis_I_spec.html#x1-620004.2.1)       *
//*     This page is exactly 58 bytes in length.                               *
//*     00-03   "OggS"                                                         *
//*     04      stream structure version (always 00h)                          *
//*     05      header type (always 02h i.e. begin stream)                     *
//*     06-13   granpos (always zero)                                          *
//*     14-17   stream serial number                                           *
//*     18-21   page sequence number (zero)                                    *
//*     22-25   page checksum                                                  *
//*     26      segment count (01h)                                            *
//*     27      segment table (number of bytes in ID header: 1Eh)              *
//*     28      packet type (01h for Identification header)                    *
//*     29-34   'v' 'o' 'r' 'b' 'i' 's'                                        *
//*     35-38   vorbis version (00h for Vorbis I)                              *
//*     39      audio channels                                                 *
//*     40-43   sample rate                                                    *
//*     44-47   bitrate maximum                                                *
//*     48-51   bitrate nominal                                                *
//*     52-55   bitrate minimum                                                *
//*     56      blocksize 0 (top 4 bits)                                       *
//*             blocksize 1 (bottom 4 bits)                                    *
//*     57      framing flag (always 80h)                                      *
//*                                                                            *
//*                                                                            *
//******************************************************************************

const char* Ogg_Tag = "OggS" ;      // Tag indicating an Ogg-format file
const int OGG_TAG_BYTES = 4 ;       // Length of Ogg_Tag
const int OGG_PAGE1_BYTES = 58 ;    // Length of first Ogg page
const int OGG_PAGEHDR_BYTES = 27 ;  // Minimum number of bytes in a page header
                                    // 0-255 'segments' follow
const char* const Ogg_PktHdr = "vorbis" ; // Packet Header ID
const int OGG_PKTHDR_BYTES = 7 ;    // Length of packet header (see Ogg_Packet)
const int OGG_SEGSIZE = 0x0FF ;     // Maximum byte count in segment table entry (255)
                                       // AND maximum number of segments
const UCHAR FRAMING_BIT = 0x01 ;    // The framing bit (byte) at the end of a pac
typedef u_int32_t UINT32 ;          // alias a 32-bit integer type
//* Embedded image comment name (APIC in MP3 world) *
const short opcBYTES = 22 ;   // bytes in oggPicComment string (not incl. NULLCHAR)
const char *oggPicComment = "METADATA_BLOCK_PICTURE", // standard comment name
           *oggPicCommentOld = "COVERART" ;           // ad-hoc and obsolete comment name

//* Each OGG page begins with a page header.                *
//* For reading the metadata, we don't care about the page  *
//* header contents except to verify that it is valid data. *
class Ogg_PageHdr
{
   public:

   Ogg_PageHdr ( void )
   { this->reset () ; }
   ~Ogg_PageHdr ( void ) {}
   void reset ( void )
   {
      *this->tag = NULLCHAR ;
      this->version = this->ptype = this->segcount = ZERO ;
      this->granpos = ZERO ;
      this->streamser = this->pageseq = this->cksum = this->segBytes = ZERO ;
   }

   //* Convert a 4-byte sequence (LSB at offset 0) to an integer value. *
   int intConv ( const UCHAR* ucp ) const
   {
      int i =    (UINT)ucp[0] 
              | ((UINT)ucp[1] << 8)
              | ((UINT)ucp[2] << 16)
              | ((UINT)ucp[3] << 24) ;
      return i ;
   }

   //* Convert a 32-bit integer into an 4-byte, little-endian binary stream *
   void intConv ( int val, UCHAR* ucp ) const
   {
      ucp[0] = (UCHAR)(val & 0x000000FF) ;
      ucp[1] = (UCHAR)((val >>  8) & 0x000000FF) ;
      ucp[2] = (UCHAR)((val >> 16) & 0x000000FF) ;
      ucp[3] = (UCHAR)((val >> 24) & 0x000000FF) ;
   }

   //* Convert an 8-byte sequence (LSB at offset 0) to an int64 value. *
   long long int intConv64 ( const UCHAR* ucp ) const
   {
      long long int i =
              (UINT64)ucp[0] 
            | ((UINT64)ucp[1] << 8)
            | ((UINT64)ucp[2] << 16)
            | ((UINT64)ucp[3] << 24)
            | ((UINT64)ucp[4] << 32)
            | ((UINT64)ucp[5] << 40)
            | ((UINT64)ucp[6] << 48)
            | ((UINT64)ucp[7] << 56) ;
      return i ;
   }

   //* Convert a 64-bit integer into an 8-byte, little-endian binary stream *
   void intConv64 ( long long int val64, UCHAR* ucp ) const
   {
      ucp[0] = (UCHAR)(val64 & 0x00000000000000FF) ;
      ucp[1] = (UCHAR)((val64 >>  8) & 0x00000000000000FF) ;
      ucp[2] = (UCHAR)((val64 >> 16) & 0x00000000000000FF) ;
      ucp[3] = (UCHAR)((val64 >> 24) & 0x00000000000000FF) ;
      ucp[4] = (UCHAR)((val64 >> 32) & 0x00000000000000FF) ;
      ucp[5] = (UCHAR)((val64 >> 40) & 0x00000000000000FF) ;
      ucp[6] = (UCHAR)((val64 >> 48) & 0x00000000000000FF) ;
      ucp[7] = (UCHAR)((val64 >> 56) & 0x00000000000000FF) ;
   }

   char   tag[5] ;         // should be 'OggS'
   UCHAR  version ;        // should be 0x00
   UCHAR  ptype ;          // may be 0x01, 0x02, 0x04 or an OR value from these
   UCHAR  segcount ;       // number of segments in page (range: 0-255)
   UINT64 granpos ;        // encoded data used by decoder (8 bytes)
   UINT   streamser ;      // uniquely identifies the stream (4 bytes)
   UINT   pageseq ;        // page sequence number (4 bytes)
   UINT   cksum ;          // CRC checksum (4 bytes)
   UINT   segBytes ;       // total bytes in page segments
} ;

//* Each of the three headers begins with the sequence:  *
//*             nn 'v' 'o' 'r' 'b' 'i' 's'               *
//* where 'nn' is the packet type:                       *
//*             01h==Identification header               *
//*             03h==Comment header                      *
//*             05h==Setup header                        *
class Ogg_Packet
{
   public:

   Ogg_Packet ( void )
   { this->reset () ; }
   ~Ogg_Packet ( void ) {}
   void reset ( void )
   {
      this->ptype = ZERO ;
      this->pid[0] = this->pid[1] = this->pid[2] = this->pid[3] = 
      this->pid[4] = this->pid[5] = this->pid[6] = NULLCHAR ;
   }
   void populate ( UINT page )
   {
      this->reset () ;
      this->ptype = page == 1 ? 1 : page == 2 ? 3 : 5 ;
      this->pid[0] = 'v' ; this->pid[1] = 'o' ; this->pid[2] = 'r' ; 
      this->pid[3] = 'b' ; this->pid[4] = 'i' ; this->pid[5] = 's' ; 
   }

   UCHAR ptype ;     // packet type
   char  pid[7] ;    // packet identifier
} ;

class Ogg_ID
{
   public:

   Ogg_ID ( void )
   { this->reset () ; }
   ~Ogg_ID ( void ) {}
   void reset ( void )
   {
      this->version = this->blksize0 = this->blksize1 = ZERO ;
      this->samprate = this->bitratemax = this->bitratenom = this->bitratemin = ZERO ;
      this->channels = ZERO ;
      this->frameflag = false ;
   }

   //* Convert a 4-byte sequence (LSB at offset 0) to an integer value. *
   //* Programmer's Note: This clunky construct avoids the C library's  *
   //* "helpful" automatic sign extension.                              *
   int intConv ( const UCHAR* ucp )
   {
      int i =    (UINT)ucp[0] 
              | ((UINT)ucp[1] << 8)
              | ((UINT)ucp[2] << 16)
              | ((UINT)ucp[3] << 24) ;
      return i ;
   }

   UINT  version ;      // vorbis I version (must be zero)
   int   samprate ;     // audio sample rate (must be > zero)
   int   bitratemax ;   // bitrate maximum
   int   bitratenom ;   // bitrate nominal
   int   bitratemin ;   // bitrate minimum
   UINT  blksize0 ;     // exponent for power of 2 (4 bits only)
   UINT  blksize1 ;     // exponent for power of 2 (4 bits only)
   UCHAR channels ;     // number of audio channels (must be > zero)
   bool  frameflag ;    // framing flag (true==GOOD, false==ERROR)
} ;
const int OGG_ID_BYTES = sizeof(Ogg_ID) ;

//* Individual comment *
class Ogg_Comment
{
   public:

   Ogg_Comment ( void )       // constructor
   { this->reset () ; }
   ~Ogg_Comment ( void ) {}   // destructor

   void reset ( void )
   {
      this->clen = ZERO ;
      for ( short i = ZERO ; i < gsDFLTBYTES ; ++i )
         this->ctxt[i] = NULLCHAR ;
   }

   //* Convert a 4-byte sequence (LSB at offset 0) to an integer value. *
   //* Programmer's Note: This clunky construct avoids the C library's  *
   //* "helpful" automatic sign extension.                              *
   int intConv ( const UCHAR* ucp )
   {
      int i =    (UINT)ucp[0] 
              | ((UINT)ucp[1] << 8)
              | ((UINT)ucp[2] << 16)
              | ((UINT)ucp[3] << 24) ;
      return i ;
   }

   //* Convert a 32-bit unsigned integer to a little-endian byte stream.      *
   //* NOTE: In a 32-bit or 64-bit little-endian system, 'u' and 'ucp'  will  *
   //*       be the same, but we can't make assumptions about user's hardware.*
   void intConv ( UINT u, UCHAR* ucp ) const
   {
      ucp[0] = (UCHAR)(u & 0x000000FF) ;
      ucp[1] = (UCHAR)((u >>  8) & 0x000000FF) ;
      ucp[2] = (UCHAR)((u >> 16) & 0x000000FF) ;
      ucp[3] = (UCHAR)((u >> 24) & 0x000000FF) ;
   }

   UINT  clen ;               // number of bytes in comment (excl. NULLCHAR)
   char  ctxt[gsDFLTBYTES] ;  // null terminated text comment
   // NOTE: The specification says that a comment may have a length of up to
   //       (2 to the 32nd) - 1. However it also suggests that a comment be
   //       no more than a short paragraph. For this reason, we discard any
   //       part of the comment beyond four(4) kbytes.
   //       The specification also allows comments to extend into the next 
   //       logical page (page 3); however we make the assumption that this
   //       will never happen because page 2 can hold 256-squared bytes. 
   //       If comments DO extend into page 3, then the comment output from
   //       page 3 will be corrupted.
   //       This will become a problem if embedded images are encoded into 
   //       the comment header. Stay tuned for an update from Taggit....
} ;

//*****************************
//* Local, non-member methods *
//*****************************
static bool  oggDecodeImageVector ( const char* ibuff, Ogg_Comment& ogc ) ;


//** - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -  **
//** - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -  **

//*************************
//*    MediafileTarget    *
//*************************
//******************************************************************************
//* Compare the filename extension of the specified file against the list of   *
//* supported media files from which we can extract the file's metadata.       *
//*                                                                            *
//*                                                                            *
//* Input  : trgPath  : filespec of file to be tested.                         *
//*                                                                            *
//* Returns: member of enum MediafileType                                      *
//******************************************************************************

MediafileType FileDlg::MediafileTarget ( const gString& trgPath )
{
   gString trgExt ;                    // filename extension
   this->fmPtr->ExtractExtension ( trgExt, trgPath ) ;
   MediafileType mftype = mftNONE ;    // return value

   if ( (trgExt.compare( L"mp3" )) == ZERO )
      mftype = mftMP3 ;
   else if (    ((trgExt.compare( L"oga" )) == ZERO) 
             || ((trgExt.compare( L"ogg" )) == ZERO) )
      mftype = mftOGG ;
   else if ( (trgExt.compare( L"m4a" )) == ZERO )
      mftype = mftM4A ;
   else if (    ((trgExt.compare( L"wma" )) == ZERO)
             || ((trgExt.compare( L"wmv" )) == ZERO) )
      mftype = mftASF ;
   else if ( (trgExt.compare( L"png", false )) == ZERO )
      mftype = mftPNG ;
   else if (    ((trgExt.compare( L"jpg",  false )) == ZERO)
             || ((trgExt.compare( L"jpeg", false )) == ZERO)
             || ((trgExt.compare( L"jfif", false )) == ZERO)
             || ((trgExt.compare( L"jpe",  false )) == ZERO)
             || ((trgExt.compare( L"jif",  false )) == ZERO) )
      mftype = mftJPG ;

   return mftype ;

}  //* End MediafileTarget() *

//*************************
//*  ExtractMetadata_MP3  *
//*************************
//******************************************************************************
//* Read the media file and write the metadata to the temp file.               *
//* Note: Certain data are not displayed:                                      *
//*       a) images                                                            *
//*       b) compressed data                                                   *
//*       c) encrypted data                                                    *
//*       d) non-text frames are displayed only in verbose mode                *
//*                                                                            *
//* Input  : ofs    : open output stream to temporary file                     *
//*          srcPath: filespec of media file to be read                        *
//*          vebose : (optional, 'false' by default)                           *
//*                   if 'false', display all text-frame records               *
//*                   if 'true', ALSO display extended technical data and      *
//*                              non-text frames (used primarily for debugging)*
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void FileDlg::ExtractMetadata_MP3 ( ofstream& ofs, const gString& srcPath, bool verbose )
{
   #define DEBUG_EMDATA (0)      // Set to non-zero for debugging only

   const char* ImgTemplate = "  Image: %S bytes, MIME:%s, Type:%s, Desc:%s" ;
   const char* PopTemplate = "  Popularity: %hhu of 255, (%u) Message:'%s' Play-count: %S" ;
   // Programmer's Note: There is no practical limit to the number of images 
   // that may be embedded in an audio file; however, we _arbitrariy_ limit
   // the number of images we actually report.
   // (Each instance of the id3v2_image class is about 4800 bytes of data.)
   const short MAX_IMAGES = 16 ;

   gString gsOut, gstmp ;        // text formatting
   id3v2_image* img = new id3v2_image[MAX_IMAGES] ; // image data (dynamic allocation)
   short imgCount = ZERO ;       // image counter
   popMeter popData ;            // Popularimeter and Play Counter data

   #if DEBUG_EMDATA != 0
   const char* tbrTemplate = "** Tag Bytes Remaining: %S\n" ;
   const char* padTemplate = "Uninitialized space (padding): %S\n" ;
   verbose = true ;              // force verbose output
   #endif   // DEBUG_EMDATA

   tnFName fn ;
   this->fmPtr->GetFileStats ( fn, srcPath, true ) ;
   gstmp.formatInt( fn.fBytes, 11, true ) ;

   short fni = srcPath.findlast( fSLASH ) + 1 ;
   ofs << "FILE NAME  : \"" << &srcPath.ustr()[fni] 
       << "\"  (" << gstmp.ustr() << " bytes)\n\n" ;

   //* Open the source file *
   ifstream ifs( srcPath.ustr(), ifstream::in ) ;
   if ( ifs.is_open() )
   {
      char ibuff[gsDFLTBYTES + 4] ; // input buffer
      UINT Tag_eod = ZERO ;         // unread bytes in tag
      id3v2_taghdr thdr ;

      //* Read the file header and verify that we have valid MP3 data.*
      UINT cnt ;                 // for validating header record
      bool valid_hdr = false ;

      //* Read header data from source file *
      ifs.read ( ibuff, taghdrCNT ) ;
      cnt = ifs.gcount() ;

      //* A valid ID3 tag header will be of the form: *
      //*    $49 44 43 yy yy xx zz zz zz zz           *
      //* where yy < FFh, xx==flags, zz < 80h         *
      if ( cnt == taghdrCNT )
      {
         thdr.file_id[0]   = ibuff[0] ;
         thdr.file_id[1]   = ibuff[1] ;
         thdr.file_id[2]   = ibuff[2] ;
         thdr.file_id[3]   = NULLCHAR ;
         thdr.major        = ibuff[3] ;
         thdr.rev          = ibuff[4] ;
         thdr.flags.setflags( ibuff[5] ) ;

         //* Tag size value is a 28-bit value encoded *
         //* in 32 bits: bytes 6, 7, 8, 9.            *
         thdr.decodeTagSize ( &ibuff[6] ) ;

         //* Validate the remainder of the header *
         gString gstmp( thdr.file_id ) ;
         if ( (gstmp.compare( "ID3" )) == ZERO &&
              (thdr.major != 0xFF && thdr.rev != 0xFF) )
            valid_hdr = true ;
      }

      if ( valid_hdr )
      {
         Tag_eod = thdr.tag_size ;     // tag data byte count
         short ttCount = ZERO ;        // number of text tags scanned

         #if DEBUG_EMDATA != 0
         gstmp.formatInt( thdr.tag_size, 11, true ) ;
         bool funsynch = thdr.flags.unsynch(),
              fexthdr  = thdr.flags.exthdr(),
              fexper   = thdr.flags.exper(),
              ffooter  = thdr.flags.footer() ;
         gsOut.compose( "ID3v2 Tag Header:\n"
                        " File ID: %s\n"
                        " Version: 2.%hhX.%hhX\n"
                        " Flags  : %hhd %hhd %hhd %hhd\n"
                        " TagSize: %S\n"
                        " Valid  : true\n",
                        thdr.file_id, &thdr.major, &thdr.rev, 
                        &funsynch, &fexthdr, &fexper, &ffooter, 
                        gstmp.gstr() ) ;
         ofs << gsOut.ustr() << endl ;
         #endif   // DEBUG_EMDATA

         //* If an "extended header" is defined *
         if ( thdr.flags.exthdr() )
         {
            //* MSB of flags set indicates presence of CRC error detection *
            const short CRC_MASK = 0x8000 ;
            short bytesRead = ZERO ;   // return value

            //* Get size of extended header *
            ifs.read ( ibuff, 4 ) ;
            bytesRead += ifs.gcount() ;
            thdr.exBytes = thdr.intConv( (UCHAR*)&ibuff[0] ) ;

            //* Read the extended header *
            ifs.read ( ibuff, thdr.exBytes ) ;
            bytesRead += ifs.gcount() ;

            // Decode the flags (16 bits, big-endian) *
            thdr.exFlags = thdr.intConv16( (UCHAR*)(&ibuff[0]) ) ;
            thdr.crcFlag = (bool)(thdr.exFlags & CRC_MASK ) ;

            //* Decode the size of the padding (32 bits, big-endian) *
            thdr.padBytes = thdr.intConv( (UCHAR*)&ibuff[2] ) ;

            //* Decode the CRC value if present (32 bits, big-endian) *
            if ( thdr.crcFlag )
               thdr.tagCRC = thdr.intConv( (UCHAR*)&ibuff[6] ) ;

            #if DEBUG_EMDATA != 0
            gstmp.formatInt( thdr.padBytes, 11, true ) ;
            gsOut.compose( L"Extended Header:\n"
                            "  exBytes: %d\n"
                            "  exFlags: %-#hb\n"
                            "  crcFlag: %hhd\n"
                            " padBytes: %S\n"
                            "   tagCRC: %08X\n",
                            &thdr.exBytes, &thdr.exFlags, &thdr.crcFlag, 
                            gstmp.gstr(), &thdr.tagCRC ) ;
            ofs << gsOut.ustr() << endl ;
            #endif   // DEBUG_EMDATA

            Tag_eod -= bytesRead ;
         }

         #if DEBUG_EMDATA != 0
         gstmp.formatInt( Tag_eod, 11, true ) ;
         gsOut.compose( tbrTemplate, gstmp.gstr() ) ;
         ofs << gsOut.ustr() << endl ;
         #endif   // DEBUG_EMDATA

         //* Read each frame and report its contents *
         id3v2_framehdr fhdr ;
         id3v2_tagfooter tftr ;
         TextEncode enc ;
         do
         {
            //* Get type of frame and its size *
            Tag_eod -= mptDecodeFrameHeader ( ifs, fhdr, tftr ) ;

            #if DEBUG_EMDATA != 0
            if ( verbose &&
                  ((*fhdr.frame_id >= 'A' && *fhdr.frame_id <= 'Z') ||
                   (*fhdr.frame_id >= '0' && *fhdr.frame_id <= '9')) )
            {
               gstmp.formatInt( fhdr.frame_size, 11, true ) ;
               gsOut.compose( L"Frame %s\n"
                               " size     : %S\n"
                               " flags    : %-#hhb %-#hhb\n",
                               fhdr.frame_id, gstmp.gstr(), 
                               &fhdr.status_flags, &fhdr.encode_flags ) ;
               //* Optional data appended to frame header *
               if ( fhdr.decomp > ZERO )     // decompressed frame-data size
               {
                  gstmp.formatInt( fhdr.decomp, 11, true ) ;
                  gsOut.append( " expanded : %S\n", gstmp.gstr() ) ;
               }
               if ( fhdr.crypto != ZERO )
                  gsOut.append( " crypto   : %hhX\n", &fhdr.crypto ) ;
               if ( fhdr.group_id != ZERO )
                  gsOut.append( " group id : %hhX\n", &fhdr.group_id ) ;

               ofs << gsOut.ustr() ;
            }
            #endif   // DEBUG_EMDATA

            if ( *fhdr.frame_id == 'T' )        // text frames
            {
               //* Get the text-frame text *
               Tag_eod -= mptDecodeTextFrame ( ifs, fhdr, gsOut, enc ) ;

               //* Determine which field the tag belongs to.  *
               //* Walk through the list of field identifiers *
               //* until a match is found, then write the text*
               //* to the corresponding tag field.            *
               bool goodID = false ;
               gstmp = fhdr.frame_id ;
               for ( short i = ZERO ; i < TEXT_FRAMES ; ++i )
               {
                  if ( (gstmp.compare( TextFrameID[i] )) == ZERO )
                  {
                     #if DEBUG_EMDATA != 0
                     if ( verbose )
                     {
                        gsOut.compose( " encoding : %02hhX (%s)", &enc,
                           (enc == ENCODE_ASCII   ? "ASCII"    : 
                            enc == ENCODE_UTF16   ? "UTF-16"   :
                            enc == ENCODE_UTF16BE ? "UTF-16BE" :
                            enc == ENCODE_UTF8    ? "UTF-8"    : "UNKNOWN" ) ) ;
                        ofs << gsOut.ustr() << endl ;
                     }
                     #endif   // DEBUG_EMDATA

                     ofs << TextFrameDesc[i] << gsOut.ustr() << '\n' ;

                     #if DEBUG_EMDATA != 0
                     gstmp.formatInt( Tag_eod, 11, true ) ;
                     gsOut.compose( tbrTemplate, gstmp.gstr() ) ;
                     ofs << gsOut.ustr() << endl ;
                     #endif   // DEBUG_EMDATA

                     ++ttCount ;
                     goodID = true ;
                     break ;
                  }
               }
               if ( ! goodID )
               {  //* Text-frame ID is not recognized.   *
                  //* Ignore the non-standard text frame.*
                  #if DEBUG_EMDATA != 0
                  gsOut.compose( "Warning: Non-standard text-frame ID: '%S'\n"
                                 "         (frame discarded)", gstmp.gstr() ) ;
                  ofs << gsOut.ustr() << endl ;
                  #endif   // DEBUG_EMDATA
               }
            }

            //* Collect image-frame description (Step over the image data.) *
            else if ( (fhdr.frame_id[0] == 'A') && (fhdr.frame_id[1] == 'P') &&
                      (fhdr.frame_id[2] == 'I') && (fhdr.frame_id[3] == 'C') )
            {
               if ( imgCount < MAX_IMAGES )
                  Tag_eod -= mptDecodeImageFrame ( ifs, fhdr, img[imgCount++] ) ;
               else  // scan and discard excess images (see note above)
               {
                  id3v2_image eximg ;
                  Tag_eod -= mptDecodeImageFrame ( ifs, fhdr, eximg ) ;
               }
            }

            //* Scan for POPM (Popularimeter) and PCNT (Play Counter) frames. *
            //* Because only one of each is allowed, we can report it _after_ *
            //* the text-frame and image-frame data.                          *
            else if ( (fhdr.frame_id[0] == 'P') &&
                      (((fhdr.frame_id[1] == 'O') && (fhdr.frame_id[2] == 'P') && 
                        (fhdr.frame_id[3] == 'M'))
                       ||
                       ((fhdr.frame_id[1] == 'C') && (fhdr.frame_id[2] == 'N') && 
                        (fhdr.frame_id[3] == 'T'))) )
            {
               Tag_eod -= mptDecodePopFrame ( ifs, fhdr, popData ) ;
               popData.pcMod = true ;
            }

            //* Non-text frames are not decoded.*
            else if (   (*fhdr.frame_id >= 'A' && *fhdr.frame_id <= 'Z')
                     || (*fhdr.frame_id >= '0' && *fhdr.frame_id <= '9') )
            {
               int loop = fhdr.frame_size / gsDFLTBYTES, 
                   remainder = fhdr.frame_size % gsDFLTBYTES ;

               //* Read the raw data *
               for ( int i = ZERO ; i < loop ; ++i )
               {
                  ifs.read( ibuff, gsDFLTBYTES ) ;
                  Tag_eod -= gsDFLTBYTES ;
               }
               if ( remainder > ZERO )
               {
                  ifs.read( ibuff, remainder ) ;
                  Tag_eod -= remainder ;
               }

               #if DEBUG_EMDATA != 0
               if ( verbose )
               {
                  gstmp.formatInt( Tag_eod, 11, true ) ;
                  gsOut.compose( tbrTemplate, gstmp.gstr() ) ;
                  ofs << " (non-text frame: '"
                      << fhdr.frame_id
                      << "' not decoded)\n"
                      << gsOut.ustr() << endl ;
               }
               #endif   // DEBUG_EMDATA
            }
            else
            {
               // Programmer's Note: In real life, padding is routinely added
               // to the tag without bothering to mention it (no extended header).
               // Although this is technically an error, there's not much we 
               // can do about it at this level.
               // Therefore, an invalid frame ID means we're done.
               #if DEBUG_EMDATA != 0
               int padCount = Tag_eod + framehdrCNT ;
               gstmp.formatInt( padCount, 11, true ) ;
               gsOut.compose( padTemplate, gstmp.gstr() ) ;
               ofs << gsOut.ustr() ;
               #endif   // DEBUG_EMDATA

               Tag_eod = ZERO ;     // discard remainder of tag
            }
         }
         while ( Tag_eod > ZERO ) ;

         //* If one or more images were captured *
         if ( imgCount > ZERO )
         {
            for ( short i = ZERO ; i < imgCount ; ++i )
            {
               gstmp.formatInt( img[i].picSize, 13, true ) ;
               gsOut.compose( ImgTemplate, gstmp.gstr(), img[i].mimType, 
                              &img[i].picExpl, img[i].txtDesc ) ;
               ofs << gsOut.ustr() << endl ;
               
            }
         }
         //* If Popularimeter and/or PlayCounter was captured *
         if ( popData.pcMod != false )
         {
            gstmp.formatInt( popData.playCount, 11, true ) ;
            gsOut.compose( PopTemplate, &popData.popStar, &popData.popCount,
                           popData.popEmail, gstmp.gstr() ) ;
            ofs << gsOut.ustr() << endl ;
         }

         //* If a tag footer is defined *
         if ( thdr.flags.footer() )
         {  // Programmer's Note: We do not decode this because it is of no 
            // interest to the user AND because it means we are at the end 
            // of the tag data.
            //mptDecodeTagFooter ( ifs, tftr ) ;
         }

         if ( ttCount == ZERO )
            ofs << "File contains no ID3v2 text frames.\n\n" ;
      }
      else
         ofs << "No ID3v2 metadata available.\n\n" ;
      ifs.close() ;           // close the source file
   }
   else
      ofs << "Sorry, unable to open file for reading.\n" ;
   ofs << endl ;

   if ( img != NULL )
   { delete [] img ; img = NULL ; }   // release the dynamic allocation

   #undef DEBUG_EMDATA

}  //* End ExtractMetadata_MP3() *

//***************************
//*  mptDecodeFrameHeader   *
//***************************
//******************************************************************************
//* Read and decode the MP3 frame header.                                      *
//* A frame header details the kind of data contained in the frame.            *
//*                                                                            *
//*                                                                            *
//* Input  : ifs     : handle for open input stream                            *
//*          fhdr    : (by reference) receives the decoded header record       *
//*          tftr    : (by reference) if a Tag Footer is identified INSTEAD OF *
//*                    a frame header, this object recieves its decoded data   *
//*                                                                            *
//* Returns: number of bytes read from the input stream                        *
//******************************************************************************
//* Notes:                                                                     *
//* 1) For id3v2.4 an optional "Tag Footer" is defined.                        *
//*    a) In order to avoid corrupting it if found, we do a special test of    *
//*       the frame_id.                                                        *
//*    b) If the tag-footer ID is found, then 'tftr' will receive the decoded  *
//*       data. For efficiency, we do not reset the 'tftr' members if input    *
//*       data are not a tag footer. It is the caller's responsibility to      *
//*       notice if the members have been initialized.                         *
//*    c) By lucky chance, framehdrCNT == taghdrCNT, so the input stream       *
//*       remains synchronized.                                                *
//******************************************************************************

static short mptDecodeFrameHeader ( ifstream& ifs, id3v2_framehdr& fhdr, 
                                    id3v2_tagfooter& tftr )
{
   const UCHAR usefulFLAGS = 0xE0 ;
   char ibuff[gsDFLTBYTES] ;  // input buffer
   short bytesRead = ZERO,    // return value
         i = ZERO ;           // input index

   fhdr.reset() ;       // clear previous data

   ifs.read ( ibuff, framehdrCNT ) ;   // read the frame header
   bytesRead += ifs.gcount() ;

   //* Test for Tag Footer (see note above) *
   if (    (ibuff[0] == '3')
        && ((ibuff[1] == 'D') || (ibuff[1] == 'd'))
        && ((ibuff[2] == 'I') || (ibuff[2] == 'i')) )
   {
      tftr.reset() ;
      tftr.foot_id[0] = ibuff[i++] ;
      tftr.foot_id[1] = ibuff[i++] ;
      tftr.foot_id[2] = ibuff[i++] ;
      tftr.major       = ibuff[i++] ;
      tftr.rev         = ibuff[i++] ;
      tftr.flags.setflags( ibuff[i++] ) ;
      tftr.decodeTagSize( &ibuff[i] ) ;
   }
   else        // frame header
   {
      fhdr.frame_id[0] = ibuff[i++] ;
      fhdr.frame_id[1] = ibuff[i++] ;
      fhdr.frame_id[2] = ibuff[i++] ;
      fhdr.frame_id[3] = ibuff[i++] ;
      fhdr.frame_size = fhdr.intConv( (UCHAR*)&ibuff[i] ) ;
      i += 4 ;    // step over the integer
      fhdr.status_flags = ibuff[i++] & usefulFLAGS ;
      fhdr.encode_flags = ibuff[i] & usefulFLAGS ;
      fhdr.flag_tag_pres  = (fhdr.status_flags & 0x80) ? true : false ;
      fhdr.flag_file_pres = (fhdr.status_flags & 0x40) ? true : false ;
      fhdr.flag_readonly  = (fhdr.status_flags & 0x20) ? true : false ;
      fhdr.flag_compress  = (fhdr.encode_flags & 0x80) ? true : false ;
      fhdr.flag_encrypt   = (fhdr.encode_flags & 0x40) ? true : false ;
      fhdr.flag_grouped   = (fhdr.encode_flags & 0x20) ? true : false ;

      //* Simple test to verify that frame_id is valid data *
      if ( ((*fhdr.frame_id >= 'A' && *fhdr.frame_id <= 'Z') ||
            (*fhdr.frame_id >= '0' && *fhdr.frame_id <= '9')) )
      {
         // Programmer's Note: The following optional additions are not 
         // included in frame header size, but are included in frame size.
         //* If 'compress' flag, 4 bytes added to header *
         //* indicating de-compressed size.              *
         if ( fhdr.flag_compress )
         {
            ifs.read( ibuff, 4 ) ;
            bytesRead += ifs.gcount() ;
            fhdr.decomp = fhdr.intConv( (UCHAR*)&ibuff[0] ) ;
         }
         //* If 'encrypt' flag,  1 byte added to header *
         //* indicating type of encryption.             *
         if ( fhdr.flag_encrypt )
         {
            ifs.read( ibuff, 1 ) ;
            bytesRead += ifs.gcount() ;
            fhdr.crypto = ibuff[0] ;
         }
         //* If 'grouped' flag,  1 byte added to header indicating group ID.*
         if ( fhdr.flag_grouped )
         {
            ifs.read( ibuff, 1 ) ;
            bytesRead += ifs.gcount() ;
            fhdr.group_id = ibuff[0] ;
         }
      }
   }
   return bytesRead ;

}  //* End mptDecodeFrameHeader() *

//***************************
//*   mptDecodeTextFrame    *
//***************************
//******************************************************************************
//* Read and decode an MP3 text frame.                                         *
//*   1) Read the raw data.                                                    *
//*   2) Convert the raw text to wchar_t (gString).                            *
//*                                                                            *
//*                                                                            *
//* Input  : ifs     : handle for open input stream                            *
//*          fhdr    : (by reference) receives the decoded header record       *
//*          gsOut   : (by reference) receives the decoded text string         *
//*          enc     : (by reference) receives text-encoding code              *
//*                                                                            *
//* Returns: number of bytes read from the input stream                        *
//******************************************************************************
//* Notes:                                                                     *
//* ======                                                                     *
//* A text frame is encoded as:   (see 'enum TextEncode')                      *
//*   a) ISO8859-1 (ASCII + Latin-1 extensions) or                             *
//*   b) UTF-16 with Byte-Order-Mark (BOM)  id3v2.2 and id3v2.3                *
//*      -- big-endian, read as   : 0xFE 0xFF, or                              *
//*      -- little-endian, read as: 0xFF 0xFE).                                *
//*   c) UTF-16BE (no BOM) id3v2.4                                             *
//*   d) UTF-8   id3v2.4                                                       *
//*                                                                            *
//******************************************************************************

static UINT mptDecodeTextFrame ( ifstream& ifs, id3v2_framehdr& fhdr, 
                                 gString& gsOut, TextEncode& enc )
{
   char ibuff[gsDFLTBYTES] ;  // input buffer
   UINT bytesRead = ZERO ;    // return value

   if ( fhdr.frame_size < gsDFLTBYTES )
   {
      //* Read the raw text *
      ifs.read( ibuff, fhdr.frame_size ) ;
      bytesRead += ifs.gcount() ;
      enc = (TextEncode)ibuff[0] ;
   }
   else
   {
      // Programmer's Note: If text frame is larger than our buffer,
      // (unlikely), we will display a truncated text string.
      // Also, declaring the data captured as less than a full buffer
      // prevents buffer overrun during Unicode conversion.
      ifs.read( ibuff, gsDFLTBYTES ) ;
      char ctmp[gsDFLTBYTES] ;
      int loop = (fhdr.frame_size - gsDFLTBYTES) / gsDFLTBYTES, 
          remainder = (fhdr.frame_size - gsDFLTBYTES) % gsDFLTBYTES ;
      for ( int i = ZERO ; i < loop ; ++i )
      {
         ifs.read( ctmp, gsDFLTBYTES ) ;
         if ( i == ZERO )
            enc = (TextEncode)ibuff[0] ;
      }
      if ( remainder > ZERO )
         ifs.read( ctmp, remainder ) ;
      bytesRead += fhdr.frame_size ;
      fhdr.frame_size = gsDFLTBYTES - 3 ;
   }

   //* Convert the raw byte stream to text *
   fhdr.txtConv( ibuff, gsOut ) ;

   return bytesRead ;

}  //* End mptDecodeTextFrame() *

//***************************
//*   mptDecodeImageFrame   *
//***************************
//******************************************************************************
//* Read and decode an MP3 image frame.                                        *
//*   1) Read the setup and description data and save it for later.            *
//*   2) Read the binary image data and,                                       *
//*      a) save it to the specified file, or                                  *
//*      b) discard it
//*                                                                            *
//* Input  : ifs     : handle for open input stream                            *
//*          fhdr    : (by reference) receives the decoded header record       *
//*          imgData : (by reference) receives the decoded image data          *
//*                                                                            *
//* Returns: number of bytes read from the input stream                        *
//******************************************************************************

static UINT mptDecodeImageFrame ( ifstream& ifs, const id3v2_framehdr& fhdr, 
                                  id3v2_image& imgData )
{
//   //* "Picture Type" descriptions as defined for id3v2 standard.  *
//   const short PIC_TYPES = 21 ;     // number of descriptions per sub-array
//   const char* const pType[PIC_TYPES] = 
//   {  // English
//     "Other",
//     "32x32 pixels 'file icon' (PNG only)",
//     "Other file icon",
//     "Cover (front)",
//     "Cover (back)",
//     "Leaflet page",
//     "Media (e.g. label side of CD)",
//     "Lead artist/lead performer/soloist",
//     "Artist/performer",
//     "Conductor",
//     "Band/Orchestra",
//     "Composer",
//     "Lyricist/text writer",
//     "Recording Location",
//     "During recording",
//     "During performance",
//     "Movie/video screen capture",
//     "A bright coloured fish",
//     "Illustration",
//     "Band/artist logotype",
//     "Publisher/Studio logotype",
//   } ;

   char ibuff[gsDFLTBYTES] ;  // input buffer
   gString gstmp ;            // data formatting
   ofstream ofs ;             // temp-file access
   bool toFile = (*imgData.picPath != NULLCHAR ? true : false) ;
   UINT bytesRead = ZERO ;    // return value

   //* Read and decode the text data *
   ifs.read( ibuff, 1 ) ;           // text-encoding indicator
   bytesRead += ifs.gcount() ;
   imgData.encoding = (TextEncode)(char)ibuff[0] ;

   for ( short i = ZERO ; i < imgMAX_TYPE ; ++i )   // MIME type
   {
      ifs.read( &ibuff[i], 1 ) ;
      bytesRead += ifs.gcount() ;
      if ( ibuff[i] == NULLCHAR )
         break ;
   }
   gstmp = ibuff ;
   gstmp.copy( imgData.mimType, imgMAX_TYPE ) ;

   ifs.read( ibuff, 1 ) ;           // picture-type indicator
   bytesRead += ifs.gcount() ;
   imgData.picType = ibuff[0] ;
   gstmp = pType[imgData.picType] ;
   gstmp.copy( imgData.picExpl, imgMAX_TYPE ) ;

   //* Read the description string according to text encoding *
   if ( (imgData.encoding == ENCODE_ASCII) || (imgData.encoding == ENCODE_UTF8) )
   {
      for ( short i = ZERO ; i < imgMAX_DESC ; ++i )
      {
         ifs.read( &ibuff[i], 1 ) ;
         bytesRead += ifs.gcount() ;
         if ( ibuff[i] == NULLCHAR )
            break ;
         ibuff[i + 1] = NULLCHAR ;     // ensure that string is terminated
      }
   }
   else     // one of the UTF-16 flavours
   {
      for ( short i = ZERO ; i < imgMAX_DESC ; i += 2 )
      {
         ifs.read( &ibuff[i], 2 ) ;
         bytesRead += ifs.gcount() ;
         if ( ibuff[i] == ZERO && ibuff[i+1] == ZERO )
            break ;
         ibuff[i + 2] = ibuff[i + 3] = NULLCHAR ; // ensure that string is terminated
      }
   }
   imgData.txtConv( ibuff, imgData.encoding ) ;

   imgData.picSize = fhdr.frame_size - bytesRead ; // bytes of image data
   int loop = imgData.picSize / gsDFLTBYTES, 
       remainder = imgData.picSize % gsDFLTBYTES ;

   if ( toFile )     // open the temp file
      ofs.open( imgData.picPath, (ofstream::out | ofstream::trunc) ) ;

   for ( int i = ZERO ; i < loop ; ++i )
   {
      ifs.read( ibuff, gsDFLTBYTES ) ;
      bytesRead += ifs.gcount() ;
      if ( toFile && (ofs.is_open()) )
         ofs.write( ibuff, ifs.gcount() ) ;
   }
   if ( remainder > ZERO )
   {
      ifs.read( ibuff, remainder ) ;
      bytesRead += ifs.gcount() ;
      if ( toFile && (ofs.is_open()) )
         ofs.write( ibuff, ifs.gcount() ) ;
   }
   if ( toFile )     // close the temp file
      ofs.close() ;

   return bytesRead ;

}  //* End mptDecodeImageFrame() *

//***************************
//*    mptDecodePopFrame    *
//***************************
//******************************************************************************
//* Read and decode an MP3 POPM or PCNT frame.                                 *
//*                                                                            *
//* a) Note that even though the standard allows for multiple popularimeters   *
//*    (POPM), we will only retain the one most recently read. This should not *
//*    be a problem since only a complete fool would encode more than one.     *
//* b) Only one play counter (PCNT) is allowed by the standard, but if         *
//*    multiple instances are encountered, only the most recent will be        *
//*    retained.                                                               *
//*                                                                            *
//* Input  : ifs     : handle for open input stream                            *
//*          fhdr    : (by reference) receives the decoded header record       *
//*          popData : (by reference) receives the decoded frame data          *
//*                                                                            *
//* Returns: number of bytes read from the input stream                        *
//******************************************************************************
//* From the id3v2.3.0 standard:                                               *
//* ============================                                               *
//* <Header for 'Play counter', ID: "PCNT">                                    *
//* Counter         $xx xx xx xx (xx ...)                                      *
//*                                                                            *
//* "This is simply a counter of the number of times a file has been played.   *
//*  The value is increased by one every time the file begins to play.         *
//*  There may only be one "PCNT" frame in each tag. When the counter reaches  *
//*  all one's, one byte is inserted in front of the counter thus making the   *
//*  counter eight bits bigger. The counter must be at least 32-bits long to   *
//*  begin with."                                                              *
//* This explanation _implies_ that the value is a big-endian integer.         *
//*                                                                            *
//* <Header for 'Popularimeter', ID: "POPM">                                   *
//* Email to user   <text string> $00                                          *
//* Rating          $xx                                                        *
//* Counter         $xx xx xx xx (xx ...)                                      *                                                          *
//*                                                                            *
//* "The purpose of this frame is to specify how good an audio file is.        *
//*  Many interesting applications could be found to this frame such as a      *
//*  playlist that features better audiofiles more often than others or it     *
//*  could be used to profile a person's taste and find other 'good' files     *
//*  by comparing people's profiles. The frame is very simple. It contains     *
//*  the email address to the user, one rating byte and a four byte play       *
//*  counter, intended to be increased with one for every time the file is     *
//*  played. The email is a terminated string.                                 *
//*  The rating is 1-255 where 1 is worst and 255 is best. 0 is unknown.       *
//*  If no personal counter is wanted it may be omitted. When the counter      *
//*  reaches all one's, one byte is inserted in front of the counter thus      *
//*  making the counter eight bits bigger in the same way as the play          *
//*  counter ("PCNT"). There may be more than one "POPM" frame in each tag,    *
//*  but only one with the same email address.                                 *
//* This explanation _implies_ that not only is the integer a big-endian value,*
//* but that the email address (if any) is an ASCII (or UTF-8) string.         *
//*                                                                            *
//******************************************************************************

static UINT mptDecodePopFrame ( ifstream& ifs, const id3v2_framehdr& fhdr, 
                                popMeter& popData )
{
   char ibuff[gsDFLTBYTES] ;           // input buffer
   gString gstmp( fhdr.frame_id ) ;    // data formatting
   UINT bytesRead = ZERO ;             // return value

   //* Read the frame data *
   ifs.read( ibuff, fhdr.frame_size ) ;
   bytesRead = ifs.gcount() ;

   if ( (gstmp.compare( POP_TAG )) == ZERO )
   {
      //* Reinitialize the target fields (possible previous tag field data) *
      popData.popEmail[0] = NULLCHAR ;
      popData.popStar = ZERO ;
      popData.popCount = ZERO ;

      //* Scan the email address(if any). If valid, it will be null terminated.*
      short i = ZERO ;
      while ( i < fhdr.frame_size )
      {
         popData.popEmail[i] = ibuff[i] ;
         if ( popData.popEmail[i++] == NULLCHAR )
            break ;
         else if ( i >= (MAX_FNAME - 2) )  // prevent buffer overrun
            break ;
      }
      popData.popEmail[i] = NULLCHAR ;    // be sure string is terminated

      if ( i < fhdr.frame_size )
      {
         popData.popStar = ibuff[i++] ;
         if ( i <= (fhdr.frame_size - 4) )
         {  //* This is a big-endian integer of at least 32 bits (4 bytes). *
            //* Data beyond 32 bits is silently discard.                    *
            popData.popCount = popData.intConv( (UCHAR*)&ibuff[fhdr.frame_size - 4] ) ;
         }
      }
      popData.popdata = true ;            // set data flag
   }
   else     // CNT_TAG
   {  //* This is a big-endian integer of at least 32 bits (4 bytes). *
      //* Data beyond 32 bits is silently discard.                    *
      popData.playCount = popData.intConv( (UCHAR*)&ibuff[fhdr.frame_size - 4] ) ;
      popData.cntdata = true ;            // set data flag
   }
   return bytesRead ;

}  //* End mptDecodePopFrame() *


//******************************************************************************
//*********************    OGG/Vorbis Audio Files    ***************************
//******************************************************************************

//*************************
//*  ExtractMetadata_OGG  *
//*************************
//******************************************************************************
//* Read the media file and write the metadata to the temp file.               *
//* The original '.ogg' and the audio-only '.oga' filename extensions indicate *
//* Ogg/Vorbis audio files.                                                    *
//*                                                                            *
//* Ogg/Vorbis 'I' files contain three metadata headers:                       *
//*  1) Identification - identifies the file as Ogg/Vorbis format.             *
//*     We decode this to verify that the file can be decoded.                 *
//*     This header is reported only in verbose mode OR if format is invalid.  *
//*  2) Comments - text fields describing the audio data                       *
//*     We decode and report the fields within the Comments header.            *
//*  3) Setup - codec setup and bitstream definitions                          *
//*     This is encoded binary data which is not decoded here.                 *
//*                                                                            *
//*                                                                            *
//* The fields of the Identification header:                                   *
//*     [vorbis_version] = read 32 bits as unsigned integer            (0)     *
//*     [audio_channels] = read 8 bit integer as unsigned              (>0)    *
//*     [audio_sample_rate] = read 32 bits as unsigned integer         (>0)    *
//*     [bitrate_maximum] = read 32 bits as signed integer                     *
//*     [bitrate_nominal] = read 32 bits as signed integer                     *
//*     [bitrate_minimum] = read 32 bits as signed integer                     *
//*       a) The fields are meaningful only when greater than zero.            *
//*          None set indicates the encoder does not care to speculate.        *
//*       b) All three fields set to the same value implies a fixed rate, or   *
//*          tightly bounded, nearly fixed-rate bitstream.                     *
//*       c) Only nominal set implies a VBR or ABR stream that averages the    *
//*          nominal bitrate.                                                  *
//*       d) Maximum and or minimum set implies a VBR bitstream that obeys the *
//*          bitrate limits.                                                   *
//*     [blocksize_0] = power-of-2 (read 4 bits as unsigned integer)   (<=bs_1)*
//*                     [64 | 128 | 256 | 512 | 1024 | 2048 | 4096 | 8192      *
//*     [blocksize_1] = power-of-2 (read 4 bits as unsigned integer)   (>bs_0) *
//*     [framing_flag] = read one bit (see note below on framing flag)         *
//*   Note that the integer fields are defined as little-endian (LSByte first).*
//*                                                                            *
//* The Comments header begins with:                                           *
//*     a) [vendor\_length] = unsigned 32-bit integer                          *
//*     b) [vendor\_string] = UTF-8 vector of [vendor\_length] bytes.          *
//*     c) [number of comment fields] = unsigned 32-bit integer                *
//*                  The number of fields is limited to 2 to the 32nd - 1.     *
//*     d) Zero or more comment fields as specified by (c)                     *
//*         i) [field length] = unsigned 32-bit integer (little-endian)        *
//*                         This (apparently) includes the FIELDNAME, 0x3D     *
//*                         and the UTF-8 text.                                *
//*        ii) [field contents] = FIELDNAME + '=' + UTF-8(unterminated) string *
//*     e) [framing\_bit] = read a single bit as boolean                       *
//*        If ( [framing\_bit] unset or end-of-packet ) then ERROR             *
//*        Note that as stated, this is completely insane because a single bit *
//*        cannot be read from a byte stream. This is the problem with         *
//*        letting morons write the documentation. In practice, this is the    *
//*        LSbit of the byte.                                                  *
//*     f) End of Comments header                                              *
//*                                                                            *
//* The individual fields contained the Comments header are of the form:       *
//*           nnnnFIELDNAME=<field contents>                                   *
//*     a) 'nnnn' is a 4-byte, little-endian integer indicating the length of  *
//*        the text record.                                                    *
//*     b) The 'FIELDNAME' may be any 7-bit ASCII sequence.                    *
//*        The field name is case-insensitive and may consist of ASCII 0x20    *
//*        through 0x7D, 0x3D (’=’) excluded. ASCII 0x41 through 0x5A inclusive*
//*        (characters A-Z) is to be considered equivalent to ASCII 0x61       *
//*        through 0x7A inclusive (characters a-z).                            *
//*                                                                            *
//*        The basic list includes:                                            *
//*         TITLE : Track/Work name                                            *
//*         VERSION : The version field may be used to differentiate multiple  *
//*                   versions of the same track title in a single collection. *
//*                  (e.g. remix info)                                         *
//*         ALBUM : The collection name to which this track belongs            *
//*         TRACKNUMBER : The track number of this piece if part of a specific *
//*                       larger collection or album.                          *
//*         ARTIST : The artist generally considered responsible for the work. *
//*                  In popular music this is usually the performing band or   *
//*                  singer. For classical music it would be the composer.     *
//*                  For an audio book it would be the author of the original  *
//*                  text.                                                     *
//*         PERFORMER : The artist(s) who performed the work. In classical     *
//*                     music this would be the conductor, orchestra, soloists.*
//*                     In an audio book it would be the actor who did the     *
//*                     reading. In popular music this is typically the same   *
//*                     as the ARTIST and is omitted.                          *
//*         COPYRIGHT : Copyright attribution, e.g., ’2001 Nobody’s Band’ or   *
//*                     ’1999 Jack Moffitt’                                    *
//*         LICENSE : License information, eg, ’All Rights Reserved’,          *
//*                   ’Any Use Permitted’, a URL to a license such as a        *
//*                   Creative Commons license                                 *
//*                   (”www.creativecommons.org/blahblah/license.html”) or the *
//*                   EFF Open Audio License (’distributed under the terms of  *
//*                   the Open Audio License.                                  *
//*                   see http://www.eff.org/IP/Open_licenses/eff_oal.html     *
//*                   for details’), etc.                                      *
//*         ORGANIZATION : Name of the organization producing the track (i.e.  *
//*                        the ’record label’)                                 *
//*         DESCRIPTION : A short text description of the contents             *
//*         GENRE : A short text indication of music genre                     *
//*         DATE : Date the track was recorded                                 *
//*         LOCATION : Location where track was recorded                       *
//*         CONTACT : Contact information for the creators or distributors of  *
//*                   the track. This could be a URL, an email address, the    *
//*                   physical address of the producing label.                 *
//*         ISRC : International Standard Recording Code for the track; see    *
//*                the ISRC intro page for more information on ISRC numbers.   *
//*     b) The '=' character (0x3D) terminates the 'FIELDNAME'.                *
//*     c) The 'field contents' are UTF-8 encoded text (NOT null terminated).  *
//*        Length is limited to 2 to the 32nd - 1 bytes; HOWEVER, the          *
//*        recommendation is a length of no more than a short paragraph (about *
//*        1,000 bytes).                                                       *
//*                                                                            *
//*                                                                            *
//* Input  : ofs    : open output stream to temporary file                     *
//*          srcPath: filespec of media file to be read                        *
//*          vebose : (optional, 'false' by default)                           *
//*                   if 'false', display all text-frame records               *
//*                   if 'true', ALSO display extended technical data and      *
//*                              non-text frames (used primarily for debugging)*
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void FileDlg::ExtractMetadata_OGG ( ofstream& ofs, const gString& srcPath, bool verbose )
{
   #define DEBUG_EMDATA (0)      // Set to non-zero for debugging only

   #if DEBUG_EMDATA != 0
   verbose = true ;              // force verbose output
   #endif   // DEBUG_EMDATA

   const char* BadOgg = "Error: Invalid Ogg/Vorbis format.\n\n" ;
   const char* noComment =  "No Ogg metadata available.\n\n" ;
   const short CNAME_COLS = 18 ; // output column alignment

   gString gsOut, gstmp ;     // output formatting
   tnFName fn ;               // source file stats
   this->fmPtr->GetFileStats ( fn, srcPath, true ) ;
   gstmp.formatInt( fn.fBytes, 11, true ) ;

   short fni = srcPath.findlast( fSLASH ) + 1 ;
   ofs << "FILE NAME  : \"" << &srcPath.ustr()[fni] 
       << "\"  (" << gstmp.ustr() << " bytes)\n\n" ;

   //* Open the source file *
   ifstream ifs( srcPath.ustr(), ifstream::in ) ;
   if ( ifs.is_open() )
   {
      char ibuff[gsDFLTBYTES] ;  // input buffer
      Ogg_ID oid ;               // Identification header data
      UINT cnt ;                 // number of bytes read from source file
      bool valid_hdr = true ;    // true if valid header record, else false

      //* Read the Ogg ID header *
      Ogg_PageHdr oph ;
      ifs.read ( ibuff, OGG_PAGE1_BYTES ) ;
      if ( (cnt = ifs.gcount()) == OGG_PAGE1_BYTES )
      {
         gstmp = ibuff ;   // (version number should also be the null terminator)
         gstmp.copy( oph.tag, OGG_TAG_BYTES ) ;
         oph.version = ibuff[4] ;
         oph.ptype   = ibuff[5] ;
                                       // granpos
                                       // streamser
                                       // pageseq
                                       // cksum
         oph.segcount = ibuff[26] ;

         //* Count the total segment bytes (s/b 30 bytes) *
         for ( UCHAR i = 27 ; i < (oph.segcount + 27) ; ++i )
            oph.segBytes += (UCHAR)(ibuff[i]) ;

         //* Read packet header *
         Ogg_Packet opkt ;
         opkt.ptype  = ibuff[28] ;
         opkt.pid[0] = ibuff[29] ;
         opkt.pid[1] = ibuff[30] ;
         opkt.pid[2] = ibuff[31] ;
         opkt.pid[3] = ibuff[32] ;
         opkt.pid[4] = ibuff[33] ;
         opkt.pid[5] = ibuff[34] ;

         //* Read the identification header *
         oid.version = oid.intConv( (UCHAR*)&ibuff[35] ) ;
         oid.channels = (UCHAR)ibuff[39] ;
         oid.samprate = oid.intConv( (UCHAR*)&ibuff[40] ) ;
         oid.bitratemax = oid.intConv( (UCHAR*)&ibuff[44] ) ;
         oid.bitratenom = oid.intConv( (UCHAR*)&ibuff[48] ) ;
         oid.bitratemin = oid.intConv( (UCHAR*)&ibuff[52] ) ;
         oid.blksize0 = (((UINT)ibuff[56]) & 0x0000000F) ;
         oid.blksize0 = ((UINT)0x00000001) << oid.blksize0 ;
         oid.blksize1 = (((UINT)ibuff[56]) & 0x000000F0) >> 4 ;
         oid.blksize1 = ((UINT)0x00000001) << oid.blksize1 ;
         oid.frameflag = (bool)(ibuff[57] & 0x01) ;

         //* Validate the page header data *
         if ( (gstmp.compare( Ogg_Tag ) == ZERO) &&
              (oph.version == ZERO) && (oph.ptype == 0x02) && 
              (oph.segcount &= 1) && (oph.segBytes == 30)
            )
         {
            //* Validate the ID header *
            if ( (oid.version != ZERO) ||
                 (oid.channels <= ZERO) ||
                 (oid.samprate <= ZERO) ||
                 !((oid.blksize0 == 64)   || (oid.blksize0 == 128) || 
                   (oid.blksize0 == 256)  || (oid.blksize0 == 512) || 
                   (oid.blksize0 == 1024) || (oid.blksize0 == 2048) || 
                   (oid.blksize0 == 4096) || (oid.blksize0 == 8192)) ||
                 !((oid.blksize1 == 64)   || (oid.blksize1 == 128) || 
                   (oid.blksize1 == 256)  || (oid.blksize1 == 512) || 
                   (oid.blksize1 == 1024) || (oid.blksize1 == 2048) || 
                   (oid.blksize1 == 4096) || (oid.blksize1 == 8192)) ||
                 (oid.frameflag == false) )
               valid_hdr = false ;
         }
         else
            valid_hdr = false ;

         #if DEBUG_EMDATA != 0
         gsOut.compose( "Packet Header : %02hhX '%s'\n"
                        "Ogg/Vorbis tag:'%S'\n"
                        "Version       : %02hhX\n"
                        "Page Type     : %02hhXh\n"
                        "Segment Count : %hhu\n"
                        "Segment Bytes : %u\n"
                        "Valid Page Hdr: %s\n\n",
                        &opkt.ptype, opkt.pid,
                        gstmp.gstr(), &oph.version, &oph.ptype, 
                        &oph.segcount, &oph.segBytes,
                        (char*)(valid_hdr ? "true" : "false") ) ;
         ofs << gsOut.ustr() ;
         #endif   // DEBUG_EMDATA

         if ( verbose )
         {
            gsOut.compose( "Ogg/Vorbis ID Header\n"
                           "--------------------\n"
                           "version   : %u\n"
                           "channels  : %hhu\n"
                           "samprate  : %d\n"
                           "bitratemax: %d\n"
                           "bitratenom: %d\n"
                           "bitratemin: %d\n"
                           "blksize0  : %u\n"
                           "blksize1  : %u\n"
                           "frameflag : %hhx\n\n",
                           &oid.version, &oid.channels, &oid.samprate,
                           &oid.bitratemax, &oid.bitratenom, &oid.bitratemin,
                           &oid.blksize0, &oid.blksize1, &oid.frameflag ) ;
            ofs << gsOut.ustr() ;
         }

         //* If ID header is valid, continue to Comment Header *
         //* Note that an error in the Comment header is       *
         //* defined as a non-fatal error.                     *
         if ( valid_hdr )
         {
            //* Read header of second page *
            char segTable[gsALLOCDFLT] ;
            oph.reset() ;
            ifs.read ( segTable, OGG_PAGEHDR_BYTES ) ;
            gstmp = segTable ;   // (version number should also be the null terminator)
            gstmp.copy( oph.tag, OGG_TAG_BYTES ) ;
            oph.version = segTable[4] ;
            oph.ptype   = segTable[5] ;
                                       // granpos
                                       // streamser
                                       // pageseq
            oph.segcount = segTable[26] ;
            //* Read the segment table (0-255 bytes) *
            //* and count the total segment bytes.   *
            ifs.read ( segTable, oph.segcount ) ;
            for ( UCHAR i = ZERO ; i < oph.segcount ; ++i )
               oph.segBytes += (UCHAR)(segTable[i]) ;

            //* Read packet header *
            UINT remBytes = oph.segBytes ; // unread bytes in Comment header
            opkt.reset() ;
            if ( remBytes >= OGG_PKTHDR_BYTES )
            {
               ifs.read ( ibuff, OGG_PKTHDR_BYTES ) ;
               remBytes -= OGG_PKTHDR_BYTES ;
               opkt.ptype  = ibuff[0] ;
               opkt.pid[0] = ibuff[1] ;
               opkt.pid[1] = ibuff[2] ;
               opkt.pid[2] = ibuff[3] ;
               opkt.pid[3] = ibuff[4] ;
               opkt.pid[4] = ibuff[5] ;
               opkt.pid[5] = ibuff[6] ;
            }

            #if DEBUG_EMDATA != 0
            gsOut.compose( "Header Packet : %02hhX '%s'\n"
                           "Ogg/Vorbis tag:'%S'\n"
                           "Version       : %02hhX\n"
                           "Page Type     : %02hhXh\n"
                           "Segment Count : %hhu\n"
                           "Segment Bytes : %u\n"
                           "Valid Page Hdr: %s\n\n", 
                           &opkt.ptype, opkt.pid,
                           gstmp.gstr(), &oph.version, &oph.ptype, 
                           &oph.segcount, &oph.segBytes,
                           (char*)(valid_hdr ? "true" : "false") ) ;
            ofs << gsOut.ustr() ;
            #endif   // DEBUG_EMDATA

            //* Read the Comment Header *
            if ( remBytes > ZERO )
            {
               Ogg_Comment ogc ;

               //* Read the watermark message - reported only in verbose mode *
               if ( remBytes >= 4 )
               {
                  ogc.reset() ;           // clear the buffer
                  ifs.read ( ibuff, 4 ) ; // read length of comment vector
                  remBytes -= 4 ;
                  ogc.clen = ogc.intConv( (UCHAR*)ibuff ) ;
                  // Programmer's Note: There is a logical error here. 
                  // If ogc.clen > gsDFLTBYTES, we will get an out-of-bounds error.
                  // The likelihood of this happening is remote.
                  ifs.read ( ibuff, ogc.clen ) ;   // read comment text
                  remBytes -= ogc.clen ;
                  if ( verbose )
                  {
                     ibuff[ogc.clen] = NULLCHAR ;
                     gsOut.compose( "%s\n", ibuff ) ;
                     ofs << gsOut.ustr() ;
                  }
               }
               //* Get the number of comment vectors *
               UINT cVectors = ZERO ;
               if ( remBytes >= 4 )
               {
                  ifs.read ( ibuff, 4 ) ; // read length of comment vector
                  remBytes -= 4 ;
                  cVectors = ogc.intConv( (UCHAR*)ibuff ) ;
                  if ( verbose )
                  {
                     gsOut.compose( "\nTotal Comments: %u\n\n", &cVectors ) ;
                     ofs << gsOut.ustr() ;
                  }
               }

               //* Read and report each vector *
               if ( cVectors > ZERO )
               {
                  short cIndex ;
                  for ( UINT v = ZERO ; v < cVectors ; ++v )
                  {
                     ogc.reset() ;           // clear the buffer
                     ifs.read ( ibuff, 4 ) ; // read length of comment vector
                     ogc.clen = ogc.intConv( (UCHAR*)ibuff ) ;
                     if ( ogc.clen < (gsDFLTBYTES - 2) )
                     {
                        ifs.read ( ibuff, ogc.clen ) ;   // read comment text

                        //* If an Image Vector, decode the header *
                        //* and discard the image itself.         *
                        if ( !(oggDecodeImageVector ( ibuff, ogc )) )
                        {
                           ibuff[ogc.clen] = NULLCHAR ;
                           gstmp = ibuff ;
                           gstmp.copy( ogc.ctxt, gsDFLTBYTES ) ;
                        }
                     }
                     else     // output limited to four(4) kbyte blocks
                     {
                        // Programmer's Note: We may split a multi-byte UTF-8
                        // character here, but the gString object will automagically
                        // discard the partial character.
                        ifs.read ( ibuff, (gsDFLTBYTES - 2) ) ;

                        //* If an Image Vector, decode the header *
                        //* and discard the image itself.         *
                        if ( !(oggDecodeImageVector ( ibuff, ogc )) )
                        {
                           ibuff[gsDFLTBYTES - 2] = NULLCHAR ;
                           gstmp = ibuff ;
                           gstmp.copy( ogc.ctxt, gsDFLTBYTES ) ;
                        }

                        remBytes = ogc.clen - (gsDFLTBYTES - 2) ;
                        while ( remBytes > gsDFLTBYTES ) // discard the remaining data
                        {
                           ifs.read ( ibuff, gsDFLTBYTES ) ;
                           remBytes -= gsDFLTBYTES ;
                        }
                        if ( remBytes > ZERO )
                           ifs.read ( ibuff, remBytes ) ;
                     }
                     // Programmer's Note: The specification states that the
                     // comment name is ASCII, so 1 byte == 1 character.
                     // The comment text however, is UTF-8.
                     gstmp = ogc.ctxt ;
                     if ( (cIndex = gstmp.find( L'=' )) >= ZERO ) // if valid comment format
                     {
                        if ( (gstmp.find( oggPicComment )) == ZERO )
                        {
                           gstmp.shiftChars( -(cIndex + 1) ) ;
                           ofs << gstmp.ustr() << '\n' ;
                        }
                        else
                        {
                           gstmp.limitChars( cIndex ) ;
                           gstmp.append( L' ' ) ;
                           while ( gstmp.gschars() < CNAME_COLS )
                              gstmp.append( L'-' ) ;
                           gstmp.append( L' ' ) ;
                           #if 0    // For Debugging only
                           gstmp.replace( L"|| ", L'\n', ZERO, false, true ) ;
                           #endif   // For Debugging only
                           ofs << gstmp.ustr() << &ogc.ctxt[cIndex + 1] << '\n' ;
                        }
                     }
                     else if ( verbose )
                        ofs << "Non-standard encoding: '" << ibuff << "'\n" ;
                  }
               }
               else
                  ofs << noComment ;
            }
            else
               ofs << noComment ;
         }
         else
            ofs << BadOgg ;
      }
      else
         ofs << BadOgg ;

      ifs.close() ;        // close the source file
   }
   else
      ofs << "Sorry, unable to open file for reading.\n" ;
   ofs << endl ;

   #undef DEBUG_EMDATA

}  //* End ExtractMetadata_OGG() *

//***************************
//*  oggDecodeImageVector   *
//***************************
//********************************************************************************
//* Read and decode an OGG image vector                                          *
//*   1) Read the setup and description data and save it for later.              *
//*   2) Read the binary image data and,                                         *
//*      a) save it to the specified file, or                                    *
//*      b) discard it                                                           *
//*                                                                              *
//* Input  : ibuff   : contains the first gsDFLTBYTES of the raw Image Vector    *
//*          ogc     : (by reference) receives a summary report of image header  *
//*                                                                              *
//* Returns: number of bytes read from the input stream                          *
//********************************************************************************

static bool oggDecodeImageVector ( const char* ibuff, Ogg_Comment& ogc )
{
   const char* ImgTemplate = "%s=  Image: %S bytes, MIME:%s, Type:%s, Desc:%s" ;

   bool isImage = false ;     // return value

   //* Do the easy test first *
   if ( (ibuff[0] == 'M') || (ibuff[0] == 'C') || (ibuff[0] == 'm') || (ibuff[0] == 'c') )
   {
      gString gstmp( ibuff, opcBYTES + 1 ) ;    // read the comment name + '='
      if ( ((gstmp.find( oggPicComment )) == ZERO) ||
           ((gstmp.find( oggPicCommentOld )) == ZERO) )
      {
         char mimType[gsDFLTBYTES], // MIME-type string
              txtDesc[gsDFLTBYTES] ;// description string
         popMeter popm ;            // borrowed for conversion of big-endian integers
         int32_t hdrIndex = ZERO ;  // index of image-vector header
         UINT32 mimLen = ZERO,      // bytes in MIME-type string
                descLen = ZERO,     // bytes in description string
                imgBytes = ZERO,    // bytes of ENCODED image data
                picType = ZERO ;    // picture type code
         //* 'true' if standard image-vector format, 'false' if obsolete format*
         bool   stdFmt = bool((ibuff[0] == 'M') || (ibuff[0] == 'm')) ;

         //* Extract header data *
         if ( (hdrIndex = gstmp.find( L'=' )) >= ZERO )
         {
            ++hdrIndex ;               // index first integer value
            isImage = true ;           // valid image vector name format
            if ( stdFmt )              // standard image format
            {
               picType = popm.intConv( (UCHAR*)&ibuff[hdrIndex] ) ;
               hdrIndex += 4 ;         // step over integer value
               mimLen = popm.intConv( (UCHAR*)&ibuff[hdrIndex] ) ;
               hdrIndex += 4 ;         // step over integer value
               gstmp.loadChars( &ibuff[hdrIndex], mimLen ) ;
               gstmp.copy( mimType, gsDFLTBYTES ) ;
               hdrIndex += mimLen ;    // step over MIME type
               descLen = popm.intConv( (UCHAR*)&ibuff[hdrIndex] ) ;
               hdrIndex += 4 ;         // step over integer value
               gstmp.loadChars( &ibuff[hdrIndex], descLen ) ;
               gstmp.copy( txtDesc, gsDFLTBYTES ) ;
               hdrIndex += descLen ;   // step over description
               hdrIndex += 4 * 4 ;     // step over image dimension data (4 integers)
               imgBytes = popm.intConv ( (UCHAR*)&ibuff[hdrIndex] ) ;
            }
            else                 // obsolete image format (i.e. no header info)
            {
               imgBytes = ogc.clen - hdrIndex ;
               gstmp = "none" ;
               gstmp.copy( mimType, gsDFLTBYTES ) ;
               gstmp = "(obsolete OGG image format)" ;
               gstmp.copy( txtDesc, gsDFLTBYTES ) ;
            }

            //* Create display text *
            gstmp.formatInt( imgBytes, 13, true ) ;
            gString gsOut( ImgTemplate, oggPicComment, gstmp.gstr(), 
                           mimType, pType[picType], txtDesc ) ;
            gsOut.copy( ogc.ctxt, gsDFLTBYTES ) ;
         }
         else
         { /* invalid comment-header format */ }
      }
   }
   return isImage ;

}  //* End mptDecodeImageVector() *


//******************************************************************************
//*****************  Definitions for MPEG-4 Audio Files (M4A)  *****************
//******************************************************************************

#define DEBUG_M4A (0)      // Set to non-zero for debugging only
#if DEBUG_M4A != 0
gString gsdbg1, gsdbg2 ;
#endif   // DEBUG_M4A

const uint32_t ubuffLEN = 0x001E8480 ; // size of input buffer (2.0 megabytes)
const uint32_t intLEN = 4 ;            // bytes in a 32-bit integer
const uint32_t idLEN = (intLEN * 2) ;  // size for initial header read


//*   ----  ----  ----  ----  ----  ----  ----  ----  ----  *
//* List of box types defined by ISO/IEC 14496-12:2005(E)   *
//* Not all types will be found in M4A (audio-only) files.  *
const uint32_t metaINDEX = ZERO, 
               metaGROUP = 17,
               freeINDEX = metaINDEX + metaGROUP,
               freeGROUP = 1,
               skipINDEX = freeINDEX + freeGROUP,
               skipGROUP = 1,
               udtaINDEX = skipINDEX + skipGROUP,
               udtaGROUP = 2,
               uuidINDEX = udtaINDEX + udtaGROUP,
               uuidGROUP = 1,
               ftypINDEX = uuidINDEX + uuidGROUP,
               ftypGROUP = 1,
               pdinINDEX = ftypINDEX + ftypGROUP,
               pdinGROUP = 1,
               moovINDEX = pdinINDEX + pdinGROUP,
               moovGROUP = 38,
               moofINDEX = moovINDEX + moovGROUP,
               moofGROUP = 8,
               mfraINDEX = moofINDEX + moofGROUP,
               mfraGROUP = 3,
               mdatINDEX = mfraINDEX + mfraGROUP,
               mdatGROUP = 1,
               MAX_BOXTYPES = mdatINDEX + mdatGROUP ;

//* Additional indices *
const uint32_t schiINDEX = metaINDEX + 11,
               xmlINDEX  = schiINDEX + 1,
               bxmlINDEX = xmlINDEX + 1 ;

const char* mpeg4_boxType[MAX_BOXTYPES] = 
{
   "meta",              // "meta" group
   "hdlr",                 // declared metadata handler typpe
   "dinf",                 // data-information-box container
   "dref",                 // declares source of metadata items
   "ipmc",                 // IPMP Control Box
   "iloc",                 // item location
   "ipro",                 // item protection (DRM)
   "sinf",                 // protection scheme information
   "frma",                 // original format
   "imif",                 // IMPM information
   "schm",                 // scheme type
   "schi",                 // scheme information
   "xml ",                 // XML container
   "bxml",                 // binary XML container
   "iinf",                 // item information
   "pitm",                 // primary item reference
   "ilst",                 // contains metadata (not part of MPEG-4 specification)
                           // This is an iTunes construct.

   "free",              // "free" space group

   "skip",              // "skip" group
   "udta",              // user data group
   "cprt",                 // copyright info

   "uuid",               // "uuid group" (user-defined type)

   "ftyp",              // "ftyp" group (file type and compatibility)

   "pdin",              // "pdin" group (progressive download information)

   "moov",              // "moov" group (container for all metadata)
   "mvhd",                 // (this box may contain data for user view)
   "trak",                 // 
   "tkhd",                 // (this box may contain data for user view 'track_ID')
   "tref",                 // 
   "edts",                 // 
   "elst",                 // 
   "mdia",                 // 
   "mdhd",                 // 
   "hdlr",                 // 
   "minf",                 // 
   "vmhd",                 // 
   "smhd",                 // 
   "hmhd",                 // 
   "mnhd",                 // 
   "dinf",                 // 
   "dref",                 // 
   "stbl",                 // 
   "stsd",                 // 
   "stts",                 // 
   "ctts",                 // 
   "stsc",                 // 
   "stsz",                 // 
   "stz2",                 // 
   "stco",                 // 
   "co64",                 // 
   "stss",                 // 
   "stsh",                 // 
   "padb",                 // 
   "stdp",                 // 
   "sdtp",                 // 
   "sbgp",                 // 
   "sgpd",                 // 
   "subs",                 // 
   "mvex",                 // 
   "mehd",                 // 
   "trex",                 // 
   "ipmc",                 // 

   "moof",              // "moof" group (movie fragment)
   "mfhd",                 // 
   "traf",                 // 
   "tfhd",                 // 
   "trun",                 // 
   "sdtp",                 // 
   "sbgp",                 // 
   "subs",                 // 

   "mfra",              // "mfra" group (movie fragment random access)
   "tfra",                 // 
   "mfro",                 //
   
   "mdat",              // "mdat" group (media data container)
} ;
// Programmer's Note: Certain box types were used in "pre-standard" files.
// clip, crgn, matt, kmat, pnot, ctab, load, imap;
// Track reference types: tmcd, chap, sync, scpt, ssrc.

//*   ----  ----  ----  ----  ----  ----  ----  ----  ----  ----  ----  *
//* Atom identifiers defined in the Apple(tm) iTunes database of tag    *
//* identifiers, along with a human-readable version of the identifier. *
//*   ----  ----  ----  ----  ----  ----  ----  ----  ----  ----  ----  *
const uint32_t looneyITEMS = 44 ;   // items in liTunes[] array
//* Item index for first integer-value record. *
//* (First two items in this group ('trkn' and 'disk' are two-value records)
const uint32_t looneyINTVALUE = 24 ;
//* Nominal column width for tag name display string.*
const short tagDescWIDTH = 23 ;
//* Special non-ASCII character used by the iTunes dictionary *
const uint8_t BYTE_A9 = 0x0A9 ;
//* Number of bytes in luniTunesDB::hName member. *
const uint32_t looneyLEN = 32 ;

class luniTunesDB
{
   public:
   luniTunesDB ( void )    // default constructor
   {
      this->reset () ;
   }
   //* Initialization constructor *
   luniTunesDB ( const uint8_t* a, const char* h )
   {
      uint32_t indx ;
      for ( indx = ZERO ; indx <= intLEN ; ++indx ) // copy atom name
      {
         if ( (this->aName[indx] = a[indx]) == '\0' )
            break ;
      }
      for ( indx = ZERO ; indx < looneyLEN ; ++indx ) // copy human-readable name
      {
         if ( (this->hName[indx] = h[indx]) == '\0' )
            break ;
      }
   }
   void reset ( void )
   {
      this->aName[0] = '\0' ;
      this->hName[0] = '\0' ;
   }
   uint8_t aName[intLEN + 1] ;         // atom name defined by iTunes
   char    hName[looneyLEN + 1] ;      // human-readable atom description
} ;

luniTunesDB liTunes[looneyITEMS] = 
{  //* : dataConst1 S/B 0x01
   { (const uint8_t*)"\xa9""nam", "Title------------------", },
   { (const uint8_t*)"\xa9""alb", "Album------------------", },
   { (const uint8_t*)"\xa9""ART", "Artist-----------------", },
   { (const uint8_t*)"aART",      "Album Artist-----------", },
   { (const uint8_t*)"\xa9""wrt", "Composer---------------", },
   { (const uint8_t*)"\xa9""day", "Year-------------------", },
   { (const uint8_t*)"\xa9""cmt", "Comment----------------", },
   { (const uint8_t*)"desc",      "Description------------", },
   { (const uint8_t*)"purd",      "Purchase Date----------", },
   { (const uint8_t*)"\xa9""grp", "Grouping---------------", },
   { (const uint8_t*)"\xa9""gen", "Genre------------------", },
   { (const uint8_t*)"\xa9""lyr", "Lyrics-----------------", },
   { (const uint8_t*)"purl",      "Podcast URL------------", },
   { (const uint8_t*)"egid",      "Podcast Episode GUID---", },
   { (const uint8_t*)"catg",      "Podcast Category-------", },
   { (const uint8_t*)"keyw",      "Podcast Keywords-------", },
   { (const uint8_t*)"\xa9""too", "Encoded By-------------", },
   { (const uint8_t*)"cprt",      "Copyright--------------", },
   { (const uint8_t*)"tvsh",      "Show Name--------------", },
   { (const uint8_t*)"\xa9""wrk", "Work-------------------", },
   { (const uint8_t*)"\xa9""mvn", "Movement---------------", },
   { (const uint8_t*)"cpil",      "Part of Compilation----", },
   { (const uint8_t*)"pgap",      "Part of Gapless Album--", },
   { (const uint8_t*)"apID",      "User ID----------------", },

   //* Obsolete Tags *
   { (const uint8_t*)"gnre",      "Genre------------------", }, // (see "\a9gen")

   //* Integer Group: dataConst1 S/B 0x00 *
   { (const uint8_t*)"trkn",      "Track Number-----------", }, // two-value (xx/yy)
   { (const uint8_t*)"disk",      "Disc Number------------", }, // two-value (xx/yy)
   { (const uint8_t*)"tmpo",      "Tempo/BPM--------------", },
   { (const uint8_t*)"\xa9""mvc", "Movement Count---------", },
   { (const uint8_t*)"\xa9""mvi", "Movement Index---------", },
   { (const uint8_t*)"shwm",      "Work/Movemenmt---------", },
   { (const uint8_t*)"stik",      "Media Kind-------------", },
   { (const uint8_t*)"hdvd",      "HD Video---------------", },
   { (const uint8_t*)"rtng",      "Content Rating---------", },
   { (const uint8_t*)"tves",      "TV Episode-------------", },
   { (const uint8_t*)"tvsn",      "TV Season--------------", },

   //* iTunes Internal Codes: dataConst1 S/B 0x15 *
   { (const uint8_t*)"plID",      "(iTunes internal)------", },
   { (const uint8_t*)"cnID",      "(iTunes internal)------", },
   { (const uint8_t*)"geID",      "(iTunes internal)------", },
   { (const uint8_t*)"atID",      "(iTunes internal)------", },
   { (const uint8_t*)"sfID",      "(iTunes internal)------", },
   { (const uint8_t*)"cmID",      "(iTunes internal)------", },
   { (const uint8_t*)"akID",      "(iTunes internal)------", },

   //* Cover Art Codes: dataConst1 S/B 0x0D *
   { (const uint8_t*)"covr",      "Cover Art--------------", },

   #if 0  //* Non-public tags, ignored *
   { (const uint8_t*)"soal",      "Album Sort Order-------", }, // non-public
   { (const uint8_t*)"soaa",      "Album Artist Sort Order", }, // non-public
   { (const uint8_t*)"soar",      "Artist Sort Order------", }, // non-public
   { (const uint8_t*)"sonm",      "Title Sort Order-------", }, // non-public
   { (const uint8_t*)"soco",      "Composer Sort Order----", }, // non-public
   { (const uint8_t*)"sosn",      "Show Sort Order--------", }, // non-public
   { (const uint8_t*)"pcst",      "Podcast (import only)--", }, // non-public
   #endif //* Non-public tags, ignored *
} ;

luniTunesDB genericTag = {(const uint8_t*)"data", "Generic Tag------------"} ;


//*   ----  ----  ----  ----  ----  ----  ----  ----  ----  ----  ----  ----   *
//*   Decode and store an iTunes atom dictionary entry.                        *
//    Programmer's Note: For integer values, our only clue about the size of 
//    the integer is the bytes remaining in the stream. Unfortunately, this is 
//    not always enough.
//    -- For for a single-value tags, we must rely on the 1, 2, or 4 bytes 
//       remaining and decode whatever remains as a single value.
//       Example: 'gnre' apparently takes a 16-bit value.
//    -- For the two-value tags 'trkn' and 'disk', it seems that two 32-bit 
//       integers are provided; HOWEVER, the second value should be 
//       interpreted as 16-bit and the final 16 bits ignored. 
//       This analysis is based on files encoded by iTunes, which for better 
//       or worse must be considered as the gold standard. Wouldn't it be 
//       nice if the specification actually specified the format?
//    Rant: Your author wrote the original Award Software BIOS specification,
//          and you may be assured that _everything_ was fully specified.
//          Unfortunately, our work at Apple was related to the II-GS and 
//          early Macs, and not the iTunes specification. (their loss :-)
//*   ----  ----  ----  ----  ----  ----  ----  ----  ----  ----  ----  ----   *

//* The 'dataConst1' member indicates the type of data stored in that atom.*
const uint32_t itaMETA_NUMERIC = (0x000) ;   // numeric (integer) data
const uint32_t itaMETA_TEXT    = (0x001) ;   // text data
const uint32_t itaMETA_ITUNES  = (0x015) ;   // internal iTunes codes
const uint32_t itaMETA_JFIF    = (0x00D) ;   // image data in JFIF format

const uint32_t itaMETA_MAX = gsDFLTBYTES ;   // size of data-capture buffer (4.0KB)
class itAtom
{
   public:

   //******************
   //* Public Methods *
   //******************
   ~itAtom ( void )              // destructor
   {
      if ( this->metaStr != NULL )
      { delete [] this->metaStr ; this->metaStr = NULL ; }
   }

   itAtom ( void )               // default constructor
   {
      this->reset () ;

      //* Allocate a data-capture buffer.*
      this->metaStr = new char[itaMETA_MAX] ;
   }

   //***********************************************************
   //* Decode the binary data and initialize all data members. *
   //***********************************************************
   // Programmer's Note: 'binData' SHOULD BE const; however, 
   // see kludgy fix below.
   bool initialize ( uint32_t s,  uint32_t a, const char* n, uint8_t* binData )
   {
      this->reset () ;
      this->atomSize = s ;                            // copy atom size
      this->aIndex   = a ;                            // copy liTunes-database index
      for ( uint8_t i = ZERO ; i <= intLEN ; ++i )    // copy atom name
         this->atomName[i] = n[i] ;

      this->dataSize = this->decodeInt32 ( binData ) ;
      uint32_t indx = intLEN ;                        // set index after data-size field
      bool dnOk = false ;
      for ( uint8_t i = ZERO ; i < intLEN ; ++i )     // copy data name
         this->dataName[i] = binData[indx++] ;
      this->dataName[intLEN] = '\0' ;  // terminate the string
      if ( (this->dataName[0] == genericTag.aName[0]) && 
           (this->dataName[1] == genericTag.aName[1]) &&
           (this->dataName[2] == genericTag.aName[2]) && 
           (this->dataName[1] == genericTag.aName[3]) )
         dnOk = true ;

      //* Decode the "data" constants *
      this->dataConst1 = this->decodeInt32 ( &binData[indx] ) ;
      indx += intLEN ;
      this->dataConst2 = this->decodeInt32 ( &binData[indx] ) ;
      indx += intLEN ;
      uint32_t eod = this->atomSize - indx + idLEN ; // bytes of actual metadata

      //* Validate the internal data structure.*
      if ( dnOk &&                                          // valid "data" atom name
           (this->dataSize == (this->atomSize - idLEN)) &&  // valid data-atom size
           (this->aIndex < looneyITEMS ) &&                 // valid database index
           (this->dataConst2 == ZERO)                       // valid constant value
         )
      {
         gString gstmp ;         // formatting

         //* If atom contains text data *
         if ( this->dataConst1 == itaMETA_TEXT )
         {
            #if 1    // KLUDGY FIX
            // Programmer's Note: At least one of our sample file places a 0xA9 
            // byte as the first byte character of the metadata in a "\xa9com" atom.
            // This is unbelievably stupid because it is not UTF-8.
            if ( binData[indx] == BYTE_A9 )
               binData[indx] = ' ' ;
            #endif   // KLUDGEY FIX

            //* Copy the text metadata *
            uint32_t trg = ZERO ;
            for ( ; (indx < eod) && (trg < (itaMETA_MAX - 1)) ; ++trg )
            {
               this->metaStr[trg] = binData[indx++] ;
               if ( this->metaStr[trg] == '\0' ) break ;
            }
            this->metaStr[trg] = '\0' ;   // ensure termination

            #if 1    // Enable Beautification
            //* Strip leading and trailing whitespace from *
            //* the data, for a more professional display. *
            gstmp = this->metaStr ;
            gstmp.strip() ;
            gstmp.copy( this->metaStr, itaMETA_MAX ) ;
            #endif   // Beautification

            this->valid = true ;
         }

         else if ( this->dataConst1 == itaMETA_NUMERIC )
         {
            //* Capture the first value               *
            //* (See note above about integer width.) *
            if ( (eod - indx) >= intLEN )
            {
               this->metaVal1 = this->decodeInt32 ( &binData[indx] ) ;
               indx += intLEN ;
            }
            else if ( (eod - indx) >= (intLEN / 2) )
            {
               this->metaVal1 = this->decodeInt16 ( &binData[indx] ) ;
               indx += (intLEN / 2) ;
            }
            else if ( (eod - indx) >= (intLEN / 4) )
            {
               this->metaVal1 = binData[indx] ;
               ++indx ;
            }
            else
               this->metaVal1 = ZERO ;
            gstmp.compose( "%u", &this->metaVal1 ) ;

            //* If this atom type defines a second value, it should *
            //* be only for the 'trkn' and 'disk' tags. For both of *
            //* these, it is assumed that the second value is       *
            //* encoded in 16 bits or less.                         *
            if ( indx < eod )
            {
               #if 0    // disable 32-bit capture
               if ( (eod - indx) >= intLEN )
                  this->metaVal2 = this->decodeInt32 ( &binData[indx] ) ;
               else
               #endif   // disable 32-bit capture
               if ( (eod - indx) >= (intLEN / 2) )
                  this->metaVal2 = this->decodeInt16 ( &binData[indx] ) ;
               else if ( (eod - indx) >= (intLEN / 4) )
                  this->metaVal2 = binData[indx] ;
               else
                  this->metaVal2 = ZERO ;
               indx += intLEN ;
               gstmp.append( "/%02u", &this->metaVal2 ) ;
            }
            gstmp.copy( this->metaStr, itaMETA_MAX ) ;

            this->valid = true ;
         }

         else if ( this->dataConst1 == itaMETA_JFIF )
         {
            //* A "JFIF" container holds a JPEG image. The format for the  *
            //* container is:
            //* 0xFFD8 0xFFE0 JFIF-APPOsegment [JFXX-APPOsegment][other markers]
            //* 0xFFDA .... 0xFFD9
            //* Legend:
            //*   0xFFD8      Start Of Image
            //*   0xFFE0      JFIF APPO marker
            //*      s1s2     16-bit big-endian: length of segment (excl. APPO marker)
            //*      JFIF\0   literal "JFIF" (null terminated)
            //*      MMmm     two bytes representing Major and minor version
            //*      du       8-bits Density Units: 00h, 01h, 02h
            //*      hpd      16-bit horizontal pixel density
            //*      vpd      16-bit vertical pixel density
            //*      hthumb   8-bit horizontal thumbnail pixel count (0 if none)
            //*      vthumb   8-bit vertical thumbnail pixel count (0 if none)
            //*      thdata   thumbnail data (24-bit RGB format)
            //*   [Optional JFXX APPO extension block, describes thumbnail]
            //*   0xFFDA      Start Of Scan
            //*   ....        JPEG compressed image data. Per spec. image is 
            //*                square and <= 3000x3000 pixels
            //*   0xFFD9      End Of Image

            //* Calculate and report image dimensions *
            // Programmer's Note: There is a potential compiler bug near here
            // regarding variable alignment. (g++ 10.1.1 20200507), Defining 
            // the 'char' array first masks the problem, but keep an eye on it.
            char     segName[64] ;
            char     imgSize[64] ;
            char     pixels[64] ;
            uint32_t soi,        // start-of-image (0xFFD8)
                     appo,       // JFIF APPO (0xFFE0)
                     appolen,    // segment size (excl. 'appo')
                     imgsize ;   // image size in bytes (approximate)
            uint32_t vmaj,       // major version
                     vmin,       // minor version
                     denunits,   // density units: 00h==no units
                                 //                01h==pixels-per-inch (2.54cm)
                                 //                02==pixels-per-centimeter
                     xden,       // horizontal pixel density
                     yden ;      // vertical pixel density
            soi = this->decodeInt16 ( &binData[indx] ) ;
            indx += (intLEN / 2) ;
            appo = this->decodeInt16 ( &binData[indx] ) ;
            indx += (intLEN / 2) ;
            appolen = this->decodeInt16 ( &binData[indx] ) ;
            indx += (intLEN / 2) ;
            for ( uint32_t trg = ZERO ; trg <= (intLEN + 1) ; ++trg )
               segName[trg] = binData[indx++] ;
            vmaj = binData[indx++] ;
            vmin = binData[indx++] ;
            denunits = binData[indx++] ;
            xden = this->decodeInt16 ( &binData[indx] ) ;
            indx += (intLEN / 2) ;
            yden = this->decodeInt16 ( &binData[indx] ) ;
            indx += (intLEN / 2) ;
            #if 0    // CAPTURE AND REPORTING OF OPTIONAL THUMBNAIL IMAGE NOT IMPLEMENTED
            //* Thumbnails supported only by iTunes specification v:1.2 and higher *
            if ( (vmax >= 1) && (vmin >= 2 )
            {
               uint32_t hthumb = binData[indx++],
                        vthumb = binData[indx++] ;
               //* If embedded thumbnail image *
               if ( (hthumb > ZERO) && (vthumb > ZERO) )
               {
               }
            }
            #endif   // THUMBNAIL IMAGE INFO
            imgsize = this->dataSize - appolen - 2 ;
            gstmp.formatInt( imgsize, 5, true, false, false, fiKb ) ;
            gstmp.copy( imgSize, 64 ) ;
            
            if ( denunits == 0x00 )
               gstmp.compose( "%ux%u pixels", &xden, &yden ) ;
            else if ( denunits == 0x01 )
               gstmp.compose( "%u pixels-per-inch", &xden ) ;
            else  // (denunits==0x02)
               gstmp.compose( "%u pixels-per-cm", &xden ) ;
            gstmp.copy( pixels, 64 ) ;

            #if DEBUG_M4A == 0   // PRODUCTION
            gstmp.compose( "%s, %s, Encoding:%s", imgSize, pixels, segName ) ;

            // Programer's Note: This is only to silence the compiler warning 
            // about unused variables when not in debugging mode.
            if ( (soi == 0xFFD8) || (appo == 0xFFE0) || (vmaj >= 1) || (vmin >= 1) )
            { /* do nothing */ }
            #else    // DEBUG ONLY
            gstmp.compose( "%s, %s, Encoding:%s\n"
                           "soi:%04X appo:%04X (%u x%04X bytes) "
                           "v:%02u.%02u dunits:%02X xden:%u yden:%u",
                           imgSize, pixels, segName,
                           &soi, &appo, &appolen, &appolen, &vmaj, 
                           &vmin, &denunits, &xden, &yden ) ;
            #endif   // DEBUG_M4A

            gstmp.copy( this->metaStr, itaMETA_MAX ) ;
            this->valid = true ;
         }

         #if DEBUG_M4A != 0
         //* Code for iTunes internal coded data. *
         //* Reported only in debugging mode.     *
         else if ( this->dataConst1 == itaMETA_ITUNES )
         {
            //* Capture the value *
            if ( (eod - indx) >= intLEN )
               this->metaVal1 = this->decodeInt32 ( &binData[indx] ) ;
            else if ( (eod - indx) >= (intLEN / 2) )
               this->decodeInt16 ( &binData[indx] ) ;
            else if ( (eod - indx) >= (intLEN / 4) )
               this->metaVal1 = binData[indx] ;
            else
               this->metaVal1 = ZERO ;
            gstmp.compose( "%04Xh", &this->metaVal1 ) ;
            gstmp.copy( this->metaStr, itaMETA_MAX ) ;
            
            this->valid = true ;
         }
         #endif   // DEBUG_M4A

         //* Unknown data format code. Currently ignored.*
         //else
         //{
         //}
      }

      return this->valid ;
   }

   //* Scan a 32-bit integer (4 bytes) from the binary data *
   uint32_t decodeInt32 ( const uint8_t* binData ) const
   {
      uint32_t uiVal = ZERO ;

      uiVal =    uint32_t(binData[3])
              + (uint32_t(binData[2]) << 8)
              + (uint32_t(binData[1]) << 16)
              + (uint32_t(binData[0]) << 24) ;

      return uiVal ;
   }

   //* Scan a 16-bit integer (2 bytes) from the binary data *
   uint32_t decodeInt16 ( const uint8_t* binData ) const
   {
      uint32_t uiVal = ZERO ;
      uiVal =    uint32_t(binData[1])
              + (uint32_t(binData[0]) << 8) ;

      return uiVal ;
   }

   void reset ( void )           // set all data members to default values
   {
      this->atomSize = this->dataSize = 
      this->dataConst1 = this->dataConst2 = ZERO ;
      this->atomName[0] = this->dataName[0] = '\0' ;
      this->valid = false ;
   }


   //***********************
   //* Public Data Members *
   //***********************
   uint32_t atomSize ;                 // atom size in bytes
   uint32_t dataSize ;                 // size of "data" atom in bytes
   uint32_t dataConst1 ;               // const value #1: itaMETA_ NUMERIC, 
                                       // or itaMETA_TEXT, or itaMETA_ITUNES
   uint32_t dataConst2 ;               // const value #2: always ZERO
   uint32_t aIndex ;                   // index of matching entry in liTunes[] database
   uint32_t metaVal1 ;                 // for numeric metadata, first value
   uint32_t metaVal2 ;                 // for numeric metadata, second value
   char*    metaStr ;                  // for text metadata, pointer to capture buffer
   char     atomName[intLEN + 1] ;     // atom name
   uint8_t  dataName[intLEN + 1] ;     // embedded name (usually "data")
   bool     valid ;                    // 'true' if valid data record captured

} ;   // End itAtom


//*   ----  ----  ----  ----  ----  ----  ----  ----  ----  ----  ----  ----   *
//*   Decode and store an (unencapsulated) "data" Atom.                        *
//*   ----  ----  ----  ----  ----  ----  ----  ----  ----  ----  ----  ----   *
//* Data records take the format:                                              *
//* "nnnndataval1val2meta"                                                     *
//* 'nnnn' size of the "data" atom                                             *
//* 'data' literal token                                                       *
//* 'val1' const value for "data" == 0x01 || 0x00 || 0x15                      *
//* 'val2' const value for "data" == 0x00                                      *
//* 'meta' the text or numeric metadata                                        *
//*        Text data are probably underminated UTF-8 data, but see hdlr.format.*
//*        Numeric data are assumed to be a single 32-bit, 16-bit or 8-bit     *
//*        integer value.                                                      *
//*                                                                            *
//* Note that "data" atoms are usually embedded within other atoms, but are    *
//* occasionally encountered wandering free. There may also be a "data" atom   *
//* within an unrecognized container atom, in which case we would skip over    *
//* the unrecognized container, but still identify the "data" atom it contains.*
//* In such circumstances, this class may be used to decode the atom.          *

class dataAtom
{
   public:

   //******************
   //* Public Methods *
   //******************
   ~dataAtom ( void )               // destructor
   {
      if ( this->metaStr != NULL )
      { delete [] this->metaStr ; this->metaStr = NULL ; }
   }

   dataAtom ( void )                // default constructor
   {
      this->reset () ;

      //* Allocate a data-capture buffer.*
      this->metaStr = new char[itaMETA_MAX] ;
   }

   //***********************************************************
   //* Decode the binary data and initialize all data members. *
   //***********************************************************
   bool initialize ( uint32_t s,  const char* n, const uint8_t* binData )
   {
      this->reset () ;
      this->atomSize = s ;                            // copy atom size
      for ( uint8_t i = ZERO ; i <= intLEN ; ++i )    // copy atom name
         this->atomName[i] = n[i] ;
      bool dkeyOk = (bool)((this->atomName[0] == genericTag.aName[0]) && 
                           (this->atomName[1] == genericTag.aName[1]) && 
                           (this->atomName[2] == genericTag.aName[2]) && 
                           (this->atomName[3] == genericTag.aName[3])) ;

      uint32_t indx = ZERO,               // scan index
               eod = atomSize - idLEN ;   // end-of-data index

      //* Decode the "data" constants *
      this->dataConst1 = this->decodeInt32 ( &binData[indx] ) ;
      indx += intLEN ;
      this->dataConst2 = this->decodeInt32 ( &binData[indx] ) ;
      indx += intLEN ;

      //* Validate the record format *
      if ( dkeyOk && (this->dataConst2 == 0x00) &&
           ((this->dataConst1 == itaMETA_TEXT) ||
            (this->dataConst1 == itaMETA_NUMERIC) || 
            (this->dataConst1 == itaMETA_ITUNES) ||
            (this->dataConst1 == itaMETA_JFIF)) )
      {
         gString gstmp ;         // formatting

         //* If atom contains text data *
         if ( this->dataConst1 == itaMETA_TEXT )
         {
            #if 0    // KLUDGY FIX - iTunes ONLY
            // Programmer's Note: At least one of our sample file places a 0xA9 
            // byte as the first byte character of the metadata in a "\xa9com" atom.
            // This is unbelievably stupid because it is not UTF-8.
            if ( binData[indx] == BYTE_A9 )
               binData[indx] = ' ' ;
            #endif   // KLUDGEY FIX

            //* Copy the text metadata *
            uint32_t trg = ZERO ;
            for ( ; (indx < eod) && (trg < (itaMETA_MAX - 1)) ; ++trg )
            {
               this->metaStr[trg] = binData[indx++] ;
               if ( this->metaStr[trg] == '\0' ) break ;
            }
            this->metaStr[trg] = '\0' ;   // ensure termination

            #if 1    // Enable Beautification
            //* Strip leading and trailing whitespace from *
            //* the data, for a more professional display. *
            gstmp = this->metaStr ;
            gstmp.strip() ;
            gstmp.copy( this->metaStr, itaMETA_MAX ) ;
            #endif   // Beautification

            this->valid = true ;
         }

         else if ( this->dataConst1 == itaMETA_NUMERIC )
         {
            //* Capture the first value               *
            //* (See note above about integer width.) *
            if ( (eod - indx) >= intLEN )
            {
               this->metaVal1 = this->decodeInt32 ( &binData[indx] ) ;
               indx += intLEN ;
            }
            else if ( (eod - indx) >= (intLEN / 2) )
            {
               this->metaVal1 = this->decodeInt16 ( &binData[indx] ) ;
               indx += (intLEN / 2) ;
            }
            else if ( (eod - indx) >= (intLEN / 4) )
            {
               this->metaVal1 = binData[indx] ;
               ++indx ;
            }
            else
               this->metaVal1 = ZERO ;
            gstmp.compose( "%u", &this->dataConst1 ) ;
            gstmp.copy( this->metaStr, itaMETA_MAX ) ;

            this->valid = true ;
         }

         else if ( this->dataConst1 == itaMETA_JFIF )
         {
            //* Calculate and report image dimensions *
            // Programmer's Note: There is a potential compiler bug near here
            // regarding variable alignment. (g++ 10.1.1 20200507), Defining 
            // the 'char' array first masks the problem, but keep an eye on it.
            char     segName[64] ;
            char     imgSize[64] ;
            char     pixels[64] ;
            uint32_t soi,        // start-of-image (0xFFD8)
                     appo,       // JFIF APPO (0xFFE0)
                     appolen,    // segment size (excl. 'appo')
                     imgsize,    // image size in bytes (approximate)
                     vmaj,       // major version
                     vmin,       // minor version
                     denunits,   // density units: 00h==no units
                                 //                01h==pixels-per-inch (2.54cm)
                                 //                02==pixels-per-centimeter
                     xden,       // horizontal pixel density
                     yden ;      // vertical pixel density

            soi = this->decodeInt16 ( &binData[indx] ) ;
            indx += (intLEN / 2) ;
            appo = this->decodeInt16 ( &binData[indx] ) ;
            indx += (intLEN / 2) ;
            appolen = this->decodeInt16 ( &binData[indx] ) ;
            indx += (intLEN / 2) ;
            for ( uint32_t trg = ZERO ; trg <= (intLEN + 1) ; ++trg )
               segName[trg] = binData[indx++] ;
            vmaj = binData[indx++] ;
            vmin = binData[indx++] ;
            denunits = binData[indx++] ;
            xden = this->decodeInt16 ( &binData[indx] ) ;
            indx += (intLEN / 2) ;
            yden = this->decodeInt16 ( &binData[indx] ) ;
            indx += (intLEN / 2) ;
            imgsize = this->atomSize - appolen - 2 ;
            gstmp.formatInt( imgsize, 5, true, false, false, fiKb ) ;
            gstmp.copy( imgSize, 64 ) ;
            
            if ( denunits == 0x00 )
               gstmp.compose( "%ux%u pixels", &xden, &yden ) ;
            else if ( denunits == 0x01 )
               gstmp.compose( "%u pixels-per-inch", &xden ) ;
            else  // (denunits==0x02)
               gstmp.compose( "%u pixels-per-cm", &xden ) ;
            gstmp.copy( pixels, 64 ) ;

            #if DEBUG_M4A == 0   // PRODUCTION
            gstmp.compose( "%s, %s, Encoding:%s", imgSize, pixels, segName ) ;

            // Programer's Note: This is only to silence the compiler warning 
            // about unused variables when not in debugging mode.
            if ( (soi == 0xFFD8) || (appo == 0xFFE0) || (vmaj >= 1) || (vmin >= 1) )
            { /* do nothing */ }
            #else    // DEBUG ONLY
            gstmp.compose( "%s, %s, Encoding:%s\n"
                           "soi:%04X appo:%04X (%u x%04X bytes) "
                           "v:%02u.%02u dunits:%02X xden:%u yden:%u",
                           imgSize, pixels, segName,
                           &soi, &appo, &appolen, &appolen, &vmaj, 
                           &vmin, &denunits, &xden, &yden ) ;
            #endif   // DEBUG_M4A

            gstmp.copy( this->metaStr, itaMETA_MAX ) ;
            this->valid = true ;
         }

         #if DEBUG_M4A != 0
         //* Code for iTunes internal coded data. *
         //* Reported only in debugging mode.     *
         else if ( this->dataConst1 == itaMETA_ITUNES )
         {
            //* Capture the value *
            if ( (eod - indx) >= intLEN )
               this->metaVal1 = this->decodeInt32 ( &binData[indx] ) ;
            else if ( (eod - indx) >= (intLEN / 2) )
               this->decodeInt16 ( &binData[indx] ) ;
            else if ( (eod - indx) >= (intLEN / 4) )
               this->metaVal1 = binData[indx] ;
            else
               this->metaVal1 = ZERO ;
            gstmp.compose( "%04Xh", &this->metaVal1 ) ;
            gstmp.copy( this->metaStr, itaMETA_MAX ) ;
            
            this->valid = true ;
         }
         #endif   // DEBUG_M4A

         //* Unknown data format code. Currently ignored.*
         //else
         //{
         //}
      }

      return this->valid ;
   }

   //* Scan a 32-bit integer (4 bytes) from the binary data *
   uint32_t decodeInt32 ( const uint8_t* binData ) const
   {
      uint32_t uiVal = ZERO ;

      uiVal =    uint32_t(binData[3])
              + (uint32_t(binData[2]) << 8)
              + (uint32_t(binData[1]) << 16)
              + (uint32_t(binData[0]) << 24) ;

      return uiVal ;
   }

   //* Scan a 16-bit integer (2 bytes) from the binary data *
   uint32_t decodeInt16 ( const uint8_t* binData ) const
   {
      uint32_t uiVal = ZERO ;

      uiVal =    uint32_t(binData[1])
              + (uint32_t(binData[0]) << 8) ;
              

      return uiVal ;
   }

   void reset ( void )              // set all data members to default values
   {
      this->atomSize = 
      this->dataConst1 = this->dataConst2 = this->metaVal1 = ZERO ;
      this->atomName[0] = '\0' ;
      this->valid = false ;
   }

   //***********************
   //* Public Data Members *
   //***********************
   uint32_t atomSize ;                 // atom size in bytes
   char     atomName[intLEN + 1] ;     // atom name
   uint32_t dataConst1 ;               // const value #1: always 0x01 || 0x00 || 0x15
   uint32_t dataConst2 ;               // const value #2: always 0x00
   uint32_t metaVal1 ;                 // for numeric metadata, first value
   char*    metaStr ;                  // pointer to metadata-text capture buffer
   bool     valid ;                    // 'true' if valid data record captured

} ;   // End dataAtom


//*   ----  ----  ----  ----  ----  ----  ----  ----  ----  ----  ----  ----   *
//*   User-defined tag atom.                                                   *
//*   ----  ----  ----  ----  ----  ----  ----  ----  ----  ----  ----  ----   *
//* Freeform records take the format:                                          *
//* "nnnn----meanzzzzaaaannnnnamezzzzNNNNnnnndatameta"                         *
//* 'nnnn' is the integer size of the record                                   *
//* '----' is the literal freeform type                                        *
//* 'mean' literal token                                                       *
//* 'zzzz' is a 32-bit integer value == 0                                      *
//* 'aaaa' this is usually "com.apple.iTunes"                                  *
//* 'nnnn' size of the "name" sub-atom                                         *
//* 'name' is the unique identifier for the record                             *
//* 'zzzz' is a 32-bit integer value == 0                                      *
//* 'NNNN' a unique tag description                                            *
//* 'nnnn' size of the "data" sub-atom                                         *
//* 'data' literal token                                                       *
//* 'val1' const value for "data" == 0x01 || 0x00 || 0x15                      *
//* 'val2' const value for "data" == 0x00                                      *
//* 'meta' the value associated with the custom name above                     *
//*        (probably underminated UTF-8 data, but see hdlr.format)             *
//*                                                                            *

luniTunesDB userdefTag = {(const uint8_t*)"----", "User-defined Tag-------"} ;
const uint8_t* userdefTag_mean = (uint8_t*)"mean" ;

class udefAtom
{
   public:

   //******************
   //* Public Methods *
   //******************
   ~udefAtom ( void )               // destructor
   {
      if ( this->metaStr != NULL )
      { delete [] this->metaStr ; this->metaStr = NULL ; }
   }

   udefAtom ( void )                // default constructor
   {
      this->reset () ;

      //* Allocate a data-capture buffer.*
      this->metaStr = new char[itaMETA_MAX] ;
   }

   //* Decode the binary data and initialize all data members.*
   bool initialize ( uint32_t s,  const char* n, uint8_t* binData )
   {
      this->reset () ;
      this->atomSize = s ;                            // copy atom size
      for ( uint8_t i = ZERO ; i <= intLEN ; ++i )    // copy atom name
         this->atomName[i] = n[i] ;

      uint32_t indx = ZERO,               // scan index
               trg,                       // target index
               eod = atomSize - idLEN ;   // end-of-data index
      bool     mkeyOk = false,            // true if "mean" key valid
               nkeyOk = false,            // true if "name" key valid
               dkeyOk = false ;           // true if "data" key valid

      //* Decode length of "mean" data *
      this->meanSize = this->decodeInt32 ( &binData[indx] ) ;
      indx += intLEN ;

      //* Decode the "mean" key *
      for ( trg = ZERO ; trg < intLEN ; ++trg )
         this->meanKey[trg] = binData[indx++] ;
      this->meanKey[trg] = '\0' ;
      mkeyOk = (bool)((this->meanKey[0] == 'm') && (this->meanKey[1] == 'e') && 
                      (this->meanKey[2] == 'a') && (this->meanKey[3] == 'n')) ;

      //* Decode the "mean" constant value (0x00) *
      this->meanConst = this->decodeInt32 ( &binData[indx] ) ;
      indx += intLEN ;

      //* Copy the text data of the "mean" atom *
      for ( trg = ZERO ; (trg < (this->meanSize - (intLEN * 3))) && (trg < 63) ; ++trg )
         this->meanDesc[trg] = binData[indx++] ;
      this->meanDesc[trg] = '\0' ;

      //* Decode length of "name" atom *
      this->nameSize = this->decodeInt32 ( &binData[indx] ) ;
      indx += intLEN ;

      //* Decode the "name" key *
      for ( trg = ZERO ; trg < intLEN ; ++trg )
         this->nameKey[trg] = binData[indx++] ;
      this->nameKey[trg] = '\0' ;
      nkeyOk = (bool)((this->nameKey[0] == 'n') && (this->nameKey[1] == 'a') && 
                      (this->nameKey[2] == 'm') && (this->nameKey[3] == 'e')) ;

      //* Decode the "name" constant value (0x00) *
      this->nameConst = this->decodeInt32 ( &binData[indx] ) ;
      indx += intLEN ;

      //* Copy the text data of the "name" atom *
      for ( trg = ZERO ; (trg < (this->nameSize - (intLEN * 3))) && (trg < 63) ; ++trg )
         this->nameDesc[trg] = binData[indx++] ;
      this->nameDesc[trg] = '\0' ;

      //* Decode length of "data" atom *
      this->dataSize = this->decodeInt32 ( &binData[indx] ) ;
      indx += intLEN ;

      //* Decode the "data" key *
      for ( trg = ZERO ; trg < intLEN ; ++trg )
         this->dataKey[trg] = binData[indx++] ;
      this->dataKey[trg] = '\0' ;
      dkeyOk = (bool)((this->dataKey[0] == 'd') && (this->dataKey[1] == 'a') && 
                      (this->dataKey[2] == 't') && (this->dataKey[3] == 'a')) ;

      //* Decode the "data" constants *
      this->dataConst1 = this->decodeInt32 ( &binData[indx] ) ;
      indx += intLEN ;
      this->dataConst2 = this->decodeInt32 ( &binData[indx] ) ;
      indx += intLEN ;

      //* Validate the record format *
      if ( mkeyOk && nkeyOk && dkeyOk && 
           (this->meanConst == 0x00) && (this->nameConst == 0x00) &&
           (this->dataConst2 == 0x00) &&
           #if 0     // Allow text metadata only
           (this->dataConst1 == itaMETA_TEXT) )
           #else     // Allow both text and numeric metadata
           ((this->dataConst1 == itaMETA_TEXT) ||
            (this->dataConst1 == itaMETA_NUMERIC) || 
            (this->dataConst1 == itaMETA_ITUNES)) )
           // Note: Numeric metadata is not supported at this time.
           #endif   // 
      {
         gString gstmp ;         // formatting

         //* Copy the user-defined text metadata to our data member *
         if ( this->dataConst1 == itaMETA_TEXT )
         {
            for ( trg = ZERO ; trg < (itaMETA_MAX - 1) ; ++trg )
            {
               this->metaStr[trg] = binData[indx++] ;
               if ( this->metaStr[trg] == '\0' ) break ;
               if ( indx >= eod ) break ;
               if ( trg >= (this->dataSize - (intLEN * 4)) ) break ;
            }
            this->metaStr[trg] = '\0' ;   // ensure termination

            #if 1    // Enable Beautification
            //* Strip leading and trailing whitespace from *
            //* the data, for a more professional display. *
            gstmp = this->metaStr ;
            gstmp.strip() ;
            gstmp.copy( this->metaStr, itaMETA_MAX ) ;
            #endif   // Beautification

            valid = true ;
         }

         //* Decode and store user-defined integer metadata.*
         else if ( this->dataConst1 == itaMETA_NUMERIC )
         {
            //* Capture the value *
            if ( (eod - indx) >= intLEN )
               this->metaVal1 = this->decodeInt32 ( &binData[indx] ) ;
            else if ( (eod - indx) >= (intLEN / 2) )
               this->metaVal1 = this->decodeInt16 ( &binData[indx] ) ;
            else if ( (eod - indx) >= (intLEN / 4) )
               this->metaVal1 = ZERO ;
            gstmp.compose( "%u", &this->metaVal1 ) ;
            gstmp.copy( this->metaStr, itaMETA_MAX ) ;

            this->valid = true ;
         }
      }

      return this->valid ;
   }

   //* Scan a 32-bit integer (4 bytes) from the binary data *
   uint32_t decodeInt32 ( const uint8_t* binData ) const
   {
      uint32_t uiVal = ZERO ;

      uiVal =    uint32_t(binData[3])
              + (uint32_t(binData[2]) << 8)
              + (uint32_t(binData[1]) << 16)
              + (uint32_t(binData[0]) << 24) ;

      return uiVal ;
   }

   //* Scan a 16-bit integer (2 bytes) from the binary data *
   uint32_t decodeInt16 ( const uint8_t* binData ) const
   {
      uint32_t uiVal = ZERO ;
      uiVal =    uint32_t(binData[1])
              + (uint32_t(binData[0]) << 8) ;

      return uiVal ;
   }

   void reset ( void )              // set all data members to default values
   {
      this->atomSize = 
      this->meanSize = this->meanConst = 
      this->nameSize = this->nameConst = 
      this->dataSize = this->dataConst1 = this->dataConst2 = this->metaVal1 = ZERO ;
      this->atomName[0] = 
      this->meanKey[0] = this->meanDesc[0] = 
      this->nameKey[0] = this->nameDesc[0] = 
      this->dataKey[0] = '\0' ;
      this->valid = false ;
   }

   //***********************
   //* Public Data Members *
   //***********************
   uint32_t atomSize ;                 // atom size in bytes
   char     atomName[intLEN + 1] ;     // atom name

   uint32_t meanSize ;                 // size of "mean" atom
   char     meanKey[intLEN + 1] ;      // "mean" keyword
   char     meanDesc[64] ;             // description (usually "com.apple.iTunes")
   uint32_t meanConst ;                // constant following "mean" (s/b zero)
   
   uint32_t nameSize ;                 // size of "name" atom
   char     nameKey[intLEN + 1] ;      // "name" keyword
   uint32_t nameConst ;                // constant following "name" (s/b zero)
   char     nameDesc[64] ;             // custom tag description

   uint32_t dataSize ;                 // size of "data" atom
   char     dataKey[intLEN + 1] ;      // "data" keyword
   uint32_t dataConst1 ;               // const value #1: always 0x01 || 0x00 || 0x15
   uint32_t dataConst2 ;               // const value #2: always 0x00
   uint32_t metaVal1 ;                 // for numeric metadata
   char*    metaStr ;                  // pointer to metadata-text capture buffer

   bool     valid ;                    // 'true' if valid data record captured

} ;   //* End udefAtom *


//*   ----  ----  ----  ----  ----  ----  ----  ----  ----  ----  ----  ----   *
//*   Metadata Box structure to contain M4A metadata.                          *
//*   ----  ----  ----  ----  ----  ----  ----  ----  ----  ----  ----  ----   *
class mdBox
{
   public:

   //******************
   //* Public Methods *
   //******************
   virtual ~mdBox ( void )             // Destructor
   {
      //* Release dynamic memory allocation.*
      if ( this->metadata != NULL )
      { delete [] this->metadata ; this->metadata = NULL ; }
   }

   mdBox ( void )                      // Default constructor
   {
      this->reset() ;
   }

   //* Initialize data members from the specified mdBox object.*
   //* The 'next' and 'prev' members are not modified.         *
   void operator= ( const mdBox* srcPtr )
   {
      this->size      = srcPtr->size ;
      this->size64    = srcPtr->size64 ;
      this->metasize  = srcPtr->metasize ;
      this->tindex    = srcPtr->tindex ;
      this->metadata  = srcPtr->metadata ;
      this->textmeta  = srcPtr->textmeta ;
      this->imagemeta = srcPtr->imagemeta ;
      this->valid     = srcPtr->valid ;
      for ( uint32_t indx = ZERO ; indx <= intLEN ; ++indx )
         this->type[indx] = srcPtr->type[indx] ;
   }

   //* Decode the object type (name) and length.                    *
   //* Returns: 'true' if successful. Initialized data members:     *
   //*                 'size', 'type', 'metasize', 'tindex'         *
   //*          'false' if name is not ASCII or if                  *
   //*                  name is not on the list of registered names.*
   //*                  or if length >= MAX_BOXLEN                  *
   bool decodeHeader ( const uint8_t* ubuff )
   {
      // Programmer's Note: We arbitrarily limit the size field on the 
      // assumption that no music file will be larger that this limit, and 
      // that if the size is larger, then it is not a valid size field.
      // ALSO note the special size values:
      //    1 == actual size is an embedded 64-bit value
      //    0 == box size not given (lazy) because box continues to end-of-file
      const uint32_t MAX_BOXLEN = 0x01312D00 ;     // 20 megabytes

      this->size = ZERO ;     // reinitialize target data members
      *this->type = '\0' ;
      this->valid = false ;   // (return value)

      //* If the type field contains only ASCII characters, *
      //* it MAY BE a box type identifier.                  *
      if ( (this->isAsciiPrint ( &ubuff[intLEN] )) )
      {
         //* Isolate the (potential) box length and *
         //* the (potential) box name fields.       *
         uint32_t sz ;
         if ( (sz = this->decodeInt32 ( ubuff )) < MAX_BOXLEN )
         {
            //* Copy the (potential) name to temp buffer.*
            char tmpType[intLEN + 1] ;
            for ( uint32_t trg = ZERO, src = intLEN ; trg < intLEN ; )
               tmpType[trg++] = ubuff[src++] ;
            tmpType[intLEN] = '\0' ;
            uint32_t tableIndex ;
            if ( (tableIndex = this->validateType ( tmpType )) < MAX_BOXTYPES )
            {
               //* Null-terminated 'type' (box name) *
               for ( uint32_t i = ZERO ; i <= intLEN ; ++i )
                  this->type[i] = tmpType[i] ;
               this->tindex = tableIndex ;            // index into 'type' table
               this->size = sz ;                      // object data bytes
               this->metasize = this->size - idLEN ;  // data bytes excl. header

               #if 0    // currently unused
               //* Decode 64-bit box size.                   *
               //* (This should not happen in a music file.) *
               if ( this->size == 1 )
                  this->size64 = this->decodeInt64 ( &ubuff[intLEN * 2] ) ;
               #endif   // U/C

               this->valid = true ;
            }
         }
      }
      return this->valid ;
   }

   //* Scan a 32-bit integer (4 bytes) from the raw data *
   uint32_t decodeInt32 ( const uint8_t* ubuff ) const
   {
      uint32_t uiVal = ZERO ;

      uiVal =    uint32_t(ubuff[3])
              + (uint32_t(ubuff[2]) << 8)
              + (uint32_t(ubuff[1]) << 16)
              + (uint32_t(ubuff[0]) << 24) ;

      return uiVal ;
   }

   //* Compare raw data to the list of MPEG-4 box types. *
   //* If a valid box type, initialize 'type' member.    *
   //*           (case-insentitive comparison)           *
   uint32_t validateType ( const char* ubuff )
   {
      const uint8_t UPPERCASE = 0x30 ;
      uint32_t src, tab, upp ;
      uint32_t indx = ZERO, offset ;
      for ( ; indx < MAX_BOXTYPES ; ++indx )
      {
         offset = ZERO ;
         for ( offset = ZERO ; offset < intLEN ; ++offset )
         {
            src = (uint32_t)ubuff[offset] ;
            tab = (uint32_t)mpeg4_boxType[indx][offset] ;   // lowercase
            upp = tab + UPPERCASE ;                         // uppercase
            if ( (src != tab) && (src != upp) )
               break ;
         }
         if ( offset == intLEN ) // if matching substring found
            break ;
      }
      return indx ;
   }

   //* Analyze four(4) bytes of raw data. If each of the four bytes *
   //* is in the range of ASCII printing characters, it MAY be the  *
   //* name of an MPEG-4 box-object name.                           *
   bool isAsciiPrint ( const uint8_t* ubuff )
   {
      bool status = false ;
   
      if ( (ubuff[0] >= ' ') && (ubuff[0] <= '~') &&
           (ubuff[1] >= ' ') && (ubuff[1] <= '~') &&
           (ubuff[2] >= ' ') && (ubuff[2] <= '~') &&
           (ubuff[3] >= ' ') && (ubuff[3] <= '~') )
         status = true ;
      return status ;
   }  //* End isAsciiPrint() *

   //* Reset all data members to default values.*
   void reset ( void )
   {
      this->size = this->metasize = this->handler = ZERO ;
      this->tindex = MAX_BOXTYPES ;
      this->size64 = ZERO ;
      this->type[0] = '\0' ;
      this->metadata = NULL ;
      this->textmeta = this->imagemeta = this->valid = false ;
   }


   //***********************
   //* Public Data Members *
   //***********************
   uint32_t size ;               // record size
   uint64_t size64 ;             // record size (if 64-bit)
   uint32_t metasize ;           // for metadata boxes only: size of metadata
   uint32_t tindex ;             // index of box type name in mpeg4_boxType[] array
   uint32_t handler ;            // data handler type code
   uint8_t  type[intLEN + 1] ;   // record type (name)
   uint8_t *metadata ;           // if 'textmeta', then contains UTF-8 text
                                 // if 'imagemeta' contains filespec of image file
   bool     textmeta ;           // 'true' if text metadata captured
   bool     imagemeta ;          // 'true' if image metadata captured
   bool     valid ;              // 'true' if both 'size' and 'type' fields initialized

} ;   //* End mdBox *


//*   ----  ----  ----  ----  ----  ----  ----  ----  ----  ----  ----  ----   *
//* A valid M4A File Header Box will be of the form:                           *
//* nnnnnnnn tttttttt MMMMMMMM mmmmmmmm (compatible brand info)                *
//* where nnnnnnnn == record length (32-bit BigEndian)                         *
//*       tttttttt == "ftyp"                                                   *
//*       MMMMMMMM == major brand   (32-bit)                                   *
//*       mmmmmmmm == minor version (32-bit)                                   *
//* A TYPICAL File Header Box will contain something similar to:               *
//*     00000018   == record length, in this example 18 hex (24 bytes decimal) *
//*       "ftyp"   == constant substring                                       *
//*       "M4A "   == major file-type identifier (brand)                       *
//*     00000200   == minor version (informational only)                       *
//*     "isomiso2" == where "isom" is optional and indicates the first version *
//*                   of the MP-4 specification, and "iso2" indicates          *
//*                   compliance with the ammended version of the specification*
//*                   Additional compatibility tokens which may be seen are    *
//*                   "M4A ", "mp42"                                           *
//*                                                                            *
//* NOTE: The spec does not instist that this be the first record, but should  *
//*       be "placed as early as possible". For instance, a Quicktime container*
//*       tag might be seen first; however, we look only at files with an      *
//*       ".m4a" extension, so the M4A header should be seen first.            *
//*                                                                            *
//* Programmer's Note: It would be a minor effort to include extraction of     *
//* metadata from .MP4 (video) files. Some valid 'ftype' records would be      *
//* "M4V ", "MP42", "isom". 'Compatibility' records have similar differences   *
//* from M4A audio files.                                                      *
//*                                                                            *
//*   ----  ----  ----  ----  ----  ----  ----  ----  ----  ----  ----  ----   *
const uint32_t MAX_COMPAT = (intLEN * 8) ;   // size of 'compat' buffer
class m4aFileHdr
{
   public:

   //******************
   //* Public Methods *
   //******************
   // Default constructor *
   m4aFileHdr ( void )
   {
      this->reset() ;
   }

   //* Initialization constructor *
   m4aFileHdr ( uint32_t boxSize, const char* binaryRecord )
   {
      this->reset() ;
      this->size = boxSize ;
      uint32_t src = ZERO,
               eod  = this->size - intLEN,
               trg ;
      for ( trg = ZERO ; trg < intLEN && src < eod ; ++trg )
         this->type[trg] = binaryRecord[src++] ;
      this->type[trg] = '\0' ;
   }

   //* initialize the data members with the provided raw information.*
   //* Input  : boxSize: total number of bytes of input data
   //*          boxType: box type string (s/b "ftyp")
   //*          binData: raw data to be formatted
   //* Returns: 'true' if valid M4A file header
   //*          'false' if validation failed
   bool initialize ( uint32_t boxSize, const uint8_t* boxType, 
                     const uint8_t* binData )
   {
      this->reset() ;                     // reset to default values
      this->size = boxSize ;              // record size

      uint32_t eod = this->size - idLEN,  // loop control
               src = ZERO, trg = ZERO ;   // buffer indices

      //* Record Type *
      for ( ; trg <= intLEN ; ++trg )
         this->type[trg] = boxType[trg] ;
      this->type[intLEN] = '\0' ;

      //* "major" version (s/b "MP4 ") *
      for ( trg = ZERO ; trg < intLEN && src < eod ; ++trg )
         this->major[trg] = binData[src++] ;
      this->major[trg] = '\0' ;

      //* "minor" version (s/b  ) *
      this->minor = this->decodeInt32 ( &binData[src] ) ;
      src += intLEN ;

      //* "Compatibility Information: MPEG-4/ISO specification version *
      for ( trg = ZERO ; (trg < MAX_COMPAT) && (src < eod) ; ++trg )
         this->compat[trg] = binData[src++] ;
      this->compat[trg] = '\0' ;

      return ( this->validate() ) ;
   }  // initialize() *

   //* Scan a 32-bit integer (4 bytes) from the binary data *
   uint32_t decodeInt32 ( const uint8_t* binaryRecord ) const
   {
      uint32_t uiVal = ZERO ;

      uiVal =    uint32_t(binaryRecord[3])
              + (uint32_t(binaryRecord[2]) << 8)
              + (uint32_t(binaryRecord[1]) << 16)
              + (uint32_t(binaryRecord[0]) << 24) ;

      return uiVal ;
   }

   //* Test for valid file-header data *
   bool validate ( void )
   {
      const char* recType = "ftyp" ;
      const char* recMaj  = "M4A " ;
      const char* verOne  = "isom" ;
      const char* verTwo  = "iso2" ;

      this->valid = false ;         // assume an invalid record
      gString gs( this->type ) ;    // comparisons NOT case-sensitive
      if ( (gs.compare( recType, false )) == ZERO )
      {
         gs = this->major ;
         if ( (gs.compare( recMaj, false )) == ZERO )
         {
            gs = this->compat ;
            if ( ((gs.find( verTwo )) >= ZERO) ||
                 ((gs.find( verOne )) >= ZERO) )
            {
               this->valid = true ;
            }
         }
      }
      return this->valid ;
   }

   //* Initialize all data members to default values *
   void reset ( void )
   {
      this->size = this->minor = ZERO ;
      this->type[0] = this->major[0] = this->compat[0] = '\0' ;
      this->valid = false ;
   }

   //***********************
   //* Public Data Members *
   //***********************
   uint32_t size ;               // record size
   char     type[intLEN + 1] ;   // record type ( "ftyp" )
   char     major[intLEN + 1] ;  // major version ( "M4A " )
   uint32_t minor ;              // minor version
   char     compat[MAX_COMPAT] ; // additional compatible types
   bool     valid ;              // 'true' if object contains a valid 'ftyp' record
} ;   // m4aFileHdr


//* Define a class for passing data among the M4A methods.*
class m4aParms
{
   public:
   virtual ~m4aParms ( void )          // destructor
   {
      if ( this->ofs.is_open() )       // if target file is open, close it
         this->ofs.close() ;
      if ( this->ubuff != NULL )       // release dynamic allocation
      { delete [] this->ubuff ; this->ubuff = NULL ; }
   }

   m4aParms ( void )                   // default constructor
   {
      this->reset () ;
   }

   //* Initialization constructor.                                            *
   //*                                                                        *
   //* Input:                                                                 *
   //*   bufflen  : number of bytes to allocate for input buffer              *
   //*   srcsize  : total bytes in source file                                *
   //*   trgspec  : filespec of existing temporary file to contain captured   *
   //*              and formatted metadata                                    *
   //*   verb     : if 'true' produce verbose output (currently unused)       *
   //*              if 'false' produce standard output                        *
   //*                                                                        *
   m4aParms ( uint32_t bufflen, uint32_t srcsize, const char* trgspec, bool verb )
   {
      this->reset () ;
      //* Allocate an input buffer *
      this->ubuff = new uint8_t[bufflen] ;
      //* Save size of source file *
      this->srcBytes = srcsize ;
      //* Open the target file (append) *
      this->ofs.open( trgspec, (ofstream::out | ofstream::app) ) ;
      //* Flag signals standard vs. verbose metadata reporting *
      this->verbose = verb ;
   }

   void reset ( void )                 // set data members to default values
   {
      this->ubuff = NULL ;
      this->mdbList = NULL ;
      this->mdbCount = this->fileBytes = this->srcBytes = ZERO ;
      this->eof = this->verbose = false ;
      #if DEBUG_M4A != 0
      this->srcoffset = ZERO ;
      #endif   // DEBUG_M4A
   }

   //** Data Members **
   std::ifstream ifs ;              // source file (input) stream
   std::ofstream ofs ;              // target file (output) stream
   uint8_t *ubuff ;                 // input stream buffer
   m4aFileHdr fileHdr ;             // file header info
   mdBox   *mdbList ;               // anchor for linked list of records
   mdBox    mdbMeta ;               // info for a "meta" box object
   mdBox    mdbHdlr ;               // info for a "meta.hdlr" box object
   uint32_t mdbCount ;              // number of metadata records captured to 'mdbList'
   uint32_t fileBytes ;             // index into source file (bytes read)
   uint32_t srcBytes ;              // source file total number of bytes
   bool     eof ;                   // 'true' if end-of-file reached
   bool     verbose ;               // verbose output (currently unused)
   #if DEBUG_M4A != 0
   uint32_t srcoffset ;             // for debugging only: report offset in source file
   #endif   // DEBUG_M4A
} ;   // m4aParms

//* Prototypes for non-member methods.*
static void m4aScan4Metadata ( m4aParms& p ) ;
static void m4aExtractMetadata ( m4aParms& p, mdBox& metabox ) ;
static void m4aExtractMD_ilst ( m4aParms& p, const mdBox& mdbCon ) ;
static void m4aExtractMD_xml ( m4aParms& p, const mdBox& mdbCon ) ;
static void m4aExtractMD_bxml ( m4aParms& p, const mdBox& mdbCon ) ;
static void m4aExtractMD_schi ( m4aParms& p, const mdBox& mdbCon ) ;
static void m4aExtractMD_free ( m4aParms& p, const mdBox& mdbCon ) ;
static void m4aExtractMD_skip ( m4aParms& p, const mdBox& mdbCon ) ;
static void m4aExtractMD_uuid ( m4aParms& p, const mdBox& mdbCon ) ;
static bool m4aReadFileHeader ( m4aParms& p ) ;
static uint32_t m4aReadBytes ( m4aParms& p, uint32_t byteCnt, uint32_t offset = ZERO ) ;
static uint32_t m4aReadHeader ( m4aParms& p ) ;
static uint32_t m4aShiftHeader ( m4aParms& p ) ;
static uint32_t m4aDiscardBytes ( m4aParms& p, uint32_t byteCnt ) ;
static uint32_t m4aIsLooneyTunes ( const uint8_t* atomName ) ;


//*************************
//*  ExtractMetadata_M4A  *
//*************************
//******************************************************************************
//* Read the media file and write the metadata to the temp file.               *
//*                                                                            *
//* Input  : ofs    : open output stream to temporary file                     *
//*          srcPath: filespec of media file to be read                        *
//*          trgPath: filespec of open temp file (see 'ofs')                   *
//*          vebose : (optional, 'false' by default)                           *
//*                   if 'false', display all text-frame records               *
//*                   if 'true', ALSO display extended technical data and      *
//*                              non-text frames (used primarily for debugging)*
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************
//* Notes:                                                                     *
//* ------                                                                     *
//* Tree Structure:                                                            *
//* Box structures may be, and often are nested to an arbitrary level.         *
//* This nested design provides flexibility, but mandates an incredibly slow   *
//* slow processing time when scanning for tag data.                           *
//*                                                                            *
//* Alignment:                                                                 *
//* The specification states that all box objects are to be aligned on an      *
//* 8-byte boundary. Unsurprisingly, this part of the specification is         *
//* completely ignored by coders.                                              *
//* For this reason, the most straight-forward scanning method is to read the  *
//* file in groups of eight(8) bytes (two 32-bit integers). When synchronized  *
//* with the data, this yields the basic information needed to identify the    *
//* object.                                                                    *
//*    Integer 0: is the size of the object in bytes. (32-bits big-endian)     *
//*               Two exceptions:                                              *
//*               a) A size value of zero (0x00000000) indicates that the      *
//*                  object continues to the end-of-file.                      *
//*               b) A size value of one (0x00000001) indicates that the       *
//*                  object contains more data that can be represented in      *
//*                  32 bits. In this case, the actual object size is contained*
//*                  in a 64-bit big-endian value located immediately after the*
//*                  object name. Because music files are not that large, we   *
//*                  make little or no effort to support objects which are     *
//*                  of 64-bit size.                                           *
//*    Integer 1: is four(4) ASCII-alpha bytes which constitute the name of    *
//*               the object. Two exceptions:                                  *
//*               a) The Apple(tm) iTunes dictionary of tag names uses the     *
//*                  idiotic value of 0xA9 as the first byte for several of    *
//*                  the defined tag names.                                    *
//*               b) The free-form tag name is the literal "----" i.e. four    *
//*                  consecutive hyphen characters.                            *
//* In order to remain synchronized with the data, it is necessary to carefully*
//* count the number of bytes read for each object, and if the length of the   *
//* object is not an even multiple of eight bytes, the raw data must be parsed *
//* to locate the start of the next object.                                    *
//*                                                                            *
//* File Header:                                                               *
//* ------------                                                               *
//* Every file must have an "ftyp" box object as close to the top of the file  *
//* as possible. If a file type or container type requires a signature at the  *
//* top of the file, then the "ftyp" object will follow it. The structure and  *
//* content of this object is discussed elsewhere in this module.              *
//*                                                                            *
//* Container Box Objects:                                                     *
//* ----------------------                                                     *
//* Metadata must be contained within another box object. The most common is   *
//* the "meta" box object. Not all metadata is human readable. This algorithm  *
//* identifies each object, but decodes only those objects which contain       *
//* text or image data.                                                        *
//*                                                                            *
//*                                                                            *
//*                                                                            *
//******************************************************************************

void FileDlg::ExtractMetadata_M4A ( ofstream& ofs, const gString& srcPath, 
                                    const gString& trgPath, bool verbose )
{
   gString gsOut, gstmp ;        // text formatting
   tnFName fn ;                  // file stats
   this->fmPtr->GetFileStats ( fn, srcPath, true ) ;
   gstmp.formatInt( fn.fBytes, 11, true ) ;

   //* Close the target file so we can open it with our own stream.*
   //* Will be re-opened with caller's object before return.       *
   ofs.close() ;

   //* Initialize parameter block *
   m4aParms p ( ubuffLEN, fn.fBytes, trgPath.ustr(), verbose ) ;

   //* Open the source file *
   p.ifs.open( srcPath.ustr(), ifstream::in ) ;
   if ( (p.ifs.is_open()) && (p.ofs.is_open()) )
   {
      short fni = srcPath.findlast( fSLASH ) + 1 ;
      p.ofs << "FILE NAME  : \"" << &srcPath.ustr()[fni] 
          << "\"  (" << gstmp.ustr() << " bytes)\n\n" ;

      //************************************************
      //* Read the file header and verify that we have *
      //* a valid M4A (MPEG-4 audio) file, then scan   *
      //* the remainder of the file. Extract, format   *
      //* and display text and image metadata.         *
      //************************************************
      if ( (m4aReadFileHeader ( p )) )
         m4aScan4Metadata ( p ) ;
      else
         p.ofs << "Error: not a valid MPEG-4 audio (.m4a) file.\n" ;

      p.ifs.close() ;           // close the source file

   }     // ifs.is_open()
   else
      p.ofs << "Sorry, unable to open file for reading.\n" ;
   p.ofs << endl ;

   //* Close our local output-stream handler.*
   p.ofs.close() ;
   //* Re-open caller's target file (append) *
   ofs.open( trgPath.ustr(), (ofstream::out | ofstream::app) ) ;

}  //* End ExtractMetadata_M4A() *

//*************************
//*   m4aScan4Metadata    *
//*************************
//******************************************************************************
//* Non-member Method:                                                         *
//* ------------------                                                         *
//* Caller has read the file header and has verified that the file contains    *
//* MPEG-4 audio data (M4A).                                                   *
//*                                                                            *
//* 1) Scan the source data for recognized box-object names.                   *
//* 2) For box objects which may contain human readable (text or image) data,  *
//* 3) Extract the human-readable data and format it for display.              *
//* 4) Write the formatted data to the temp file to be displayed to the user.  *
//*                                                                            *
//* Input  : p      : (by reference) M4A-capture parameters                    *
//*                                                                            *
//* Returns: pointer to a linked list of mdBox objects                         *
//*          If no text or image metadata found, then a NULL pointer will      *
//*          be returned.                                                      *
//******************************************************************************
//* 1) Data format:                                                            *
//*    A valid M4A "Box" header will be of the form:                           *
//*    nnnnnnnn tttttttt (exsize) vv ffffff                                    *
//*    where nnnnnnnn == record length (32-bit Big Endian)                     *
//*          tttttttt == record type (32-bit, usually 4 ASCII bytes)           *
//*          exsize   == expanded size (optional 64-bit field)                 *
//*                      If record length == 1, then the true record length is *
//*                      is this 64-bit value following the record type        *
//*          vv       == 8-bit version                                         *
//*          ffffff   == 24-bit flags field                                    *
//*                                                                            *
//*    Like most specifications in the audio/video world, this structure       *
//*    cannot be relied upon. Fortunately, the version and flags are currently *
//*    fixed at zero (0x00000000), so whether or not they are actually present,*
//*    it is unnecessary to decode them.                                       *
//*                                                                            *
//* 2) Logic:                                                                  *
//*    a) According to the specification, all box objects must be aligned on   *
//*       an 8-byte boundary. Unfortunately, the real-world sample files show  *
//*       that this is not always the case.                                    *
//*       When a misaligned box object is encountered, boxSize mod 8 != ZERO.  *
//*       It is necessary to re-synch the stream so we do not miss any box     *
//*       objects.                                                             *
//*    b) A box object has a minimum of eight(8) bytes, a 32-but integer size  *
//*       member and a 32-bit name (usually) consisting of four ASCII          *
//*       characters.                                                          *
//*    c) A non-empty object will have a minimum of sixteen(16) bytes.         *
//*    d) A box object may contain other box objects.                          *
//*       To contain another object the container box size must be at least    *
//*       sixteen(16) bytes. (at least 24 bytes to contain a non-empty object) *
//*                                                                            *
//* 3) Implementation:                                                         *
//*    a) On entry, the input stream is on an 8-byte boundry.                  *
//*    b) The input stream is read in increments of eight(8) bytes.            *
//*       Each increment is tested for a format that is potentially a box      *
//*       header (see item 2b).                                                *
//*    c) When a potential box header is identified, the name field is         *
//*       compared with the list of box-object names to verify that a box      *
//*       object has been located.                                             *
//*    d) If the object name is a recognized metadata box name, the object is  *
//*       decoded and the contents formatted for display.                      *
//*       Certain local classes are defined for object decoding, storage and   *
//*       formatting. See definition of the mdBox class.                       *
//*                                                                            *
//******************************************************************************

static void m4aScan4Metadata ( m4aParms& p )
{
   mdBox    mdbRef ;                // temp object for data analysis
   uint32_t idLen ;                 // bytes returned by source read

   #if DEBUG_M4A != 0
   #define DEBUG_M4A_VERBOSE (0)
   uint32_t modulo ;                // modulus of eight(8) bytes
   #endif   // DEBUG_M4A

   p.mdbCount = ZERO ;        // initialize caller's counter

   //* Load buffer with new test data.*
   idLen = m4aReadHeader ( p ) ;

   //* Read until end-of-file reached *
   while ( ! p.eof )
   {
      #if DEBUG_M4A != 0 && DEBUG_M4A_VERBOSE != 0
      gsdbg1.compose( "RAW: %02hhX %02hhX %02hhX %02hhX %02hhX %02hhX %02hhX %02hhX",
                      &p.ubuff[0], &p.ubuff[1], &p.ubuff[2], &p.ubuff[3], 
                      &p.ubuff[4], &p.ubuff[5], &p.ubuff[6], &p.ubuff[7] ) ;
      p.ofs << gsdbg1.ustr() << endl ;
      #endif   // DEBUG_M4A && DEBUG_M4A_VERBOSE

      //* If all four characters of the name field are ASCII printing *
      //* characters AND if name is a registered box type.            *
      if ( (mdbRef.decodeHeader ( p.ubuff )) )
      {
         #if DEBUG_M4A != 0
         modulo = mdbRef.size % mdbRef.size ;
         gsdbg2.formatInt( mdbRef.size, 11, true ) ;
         gsdbg1.compose( "BOX: %s (%s bytes at x%08X)", 
                        mdbRef.type, gsdbg2.ustr(), &p.srcoffset ) ;
         p.ofs << gsdbg1.ustr() ;
         if ( modulo != ZERO )
         {
            gsdbg1.compose( " mod:%u", &modulo ) ;
            p.ofs << gsdbg1.ustr() ;
         }
         p.ofs << endl ;
         // Programmer's Note: The reported position of the box object in the 
         // stream (p.srcoffset) is approximate only due to misalignment of 
         // preceeding objects. According to the specification, all box objects 
         // must be aligned on an 8-byte boundary, but in practice, they are not.
         #endif   // DEBUG_M4A

         //* If the current box object is a "meta" box, *
         //* it may contain text or image metadata.     *
         //* Extract and display the metadata.          *
         if ( mdbRef.tindex == metaINDEX )
         {
            m4aExtractMetadata ( p, mdbRef ) ;
         }
         //* If the current box object is a "free" box, *
         //* it may contain interesting data, but no    *
         //* embedded boxes.                            *
         else if ( mdbRef.tindex == freeINDEX )
         {
            m4aExtractMD_free ( p, mdbRef ) ;
         }
         //* If the current box object is a "skip" box, *
         //* it may contain interesting data, either    *
         //* directly or in an embedded "udta" or       *
         //* "cprt" box.                                *
         else if ( mdbRef.tindex == skipINDEX )
         {
            m4aExtractMD_skip ( p, mdbRef ) ;
         }
         //* If the current box object is a "udta" box, *
         //* it may contain interesting data, either a  *
         //* "cprt" box or another user-oriented box    *
         //* such as a "meta" box.                      *
         else if ( mdbRef.tindex == udtaINDEX )
         {
            /* Do nothing. Allow the main loop to identify  *
             * boxes/atoms contained within the 'udata' box.*/
         }
         //* If the current box object is a "uuid" box, *
         //* it may contain interesting user-defined    *
         //* data, but no embedded boxes.               *
         else if ( mdbRef.tindex == uuidINDEX )
         {
            m4aExtractMD_uuid ( p, mdbRef ) ;
         }
         //* If the current box object is a "schi" box, *
         //* it may contain the name of the person who  *
         //* downloaded the file from iTunes.           *
         if ( mdbRef.tindex == schiINDEX )
         {
            m4aExtractMD_schi ( p, mdbRef ) ;
         }
         //* Box objects which are "known" to contain   *
         //* neither interesting data nor embedded boxes*
         else if ( (mdbRef.tindex == pdinINDEX) ||
                   (mdbRef.tindex == mdatINDEX) )
         {
            m4aDiscardBytes ( p, mdbRef.metasize ) ;
         }
         //* Else, unhandled box type. Ignore it and    *
         //* scan its contents for embedded box objects.*
         else
         {
         }

         //* Load buffer with new test data.*
         if ( (idLen = m4aReadHeader ( p )) < idLEN )
         {  //* Fewer than idLEN bytes read. *
            //* Read to end-of-file.         *
            m4aReadBytes ( p, idLEN ) ;
         }
      }              // decodeHeader()
      else
      {  //* Invalid header. Shift, read and retest.*
         m4aShiftHeader ( p ) ;
      }
   }                    // while()

   #undef DEBUG_M4A_VERBOSE
}  //* End m4aScan4Metadata() *

//*************************
//*  m4aExtractMetadata   *
//*************************
//********************************************************************************
//* Non-member Method:                                                           *
//* ------------------                                                           *
//* Caller has read the header information for the "meta" box object             *
//* ('metabox' parameter). This is a container for other box objects, some of    *
//* which may contain text or image metadata.                                    *
//*                                                                              *
//* Read and decode each object in the container, capturing any human-readable   *
//* data.                                                                        *
//*                                                                              *
//*                                                                              *
//*                                                                              *
//* Input  : p      : (by reference) M4A-capture parameters                      *
//*          metabox: (by reference) contains basic information for "meta"       *
//*                    container: 'size', 'type', 'metasize', 'tindex', 'valid'  *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************
//* Notes:                                                                       *
//* ------                                                                       *
//* The "meta" container box may contain any of several different box types;     *
//* however, most are data-stream information.                                   *
//* We are interested primarily in the 'ilst' object which contains any text     *
//* metadata based on the Apple iTunes(tm) dictionary.                           *
//* Additional interesting types within a "meta" container are: 'xml ' and       *
//* 'bxml'.                                                                      *
//* Other contained objects such as the additional boxes listed under "meta"     *
//* in the specification (see below) as well as 'free', 'skip', 'udta', 'cprt',  *
//* 'uuid' may contain some user-display data, but currently we have no samples  *
//* of those for testing, so they are ignored.                                   *
//*                                                                              *
//* A) The 'meta' object contains various types of metadata.                     *
//*    uint32_t box size                                                         *
//*    uint32_t "meta"                                                           *
//*    uint32_t version == 0                                                     *
//*                                                                              *
//*    1) The 'hdlr' type (handler) object must be present. All other types      *
//*       are optional.                                                          *
//*       a) uint32_t version == 0 (box-type version)                            *
//*       b) uint32_t byte 3 == version == 0, bytes 2-0 predefined == 0x000000   *
//*       c) uint32_t handler type (format of meta box contents)                 *
//*          The MPEG-4 specification only says that this is: "an appropriate    *
//*          value to indicate the format of the meta box contents".             *
//*          Not very bloody helpful, but it is assumed that 0x00 == UTF-8.      *
//*          For Quicktime, this is "mdta".                                      *
//*          For non-metadata, this should be 'vide', 'soun', 'hint              *
//*       d) 3 x uint32_t reserved, all set to 0                                 *
//*          [NOTE: This may not be true for non-Quicktime files.]               *
//*          In the real world, this is two(2) integers set to zero, with the    *
//*          third set to "mdir" and the name "appl", OR two zero integers       *
//*          followed by the name "mdirappl". Dealer's choice....                *
//*       e) A null-terminated UTF-8 string: name of handler                     *
//*                                                                              *
//*    2) The 'ilst' sub-type (item list) contains the formatted metadata.       *
//*       Note that 'ilst' is NOT part of the MPEG-4 specification.              *
//*       The 'ilst' header is of the form: "nnnnilstxxxxyyyy", where            *
//*       'nnnn' is the size of the ilst box                                     *
//*       'ilst' is the box name                                                 *
//*       'xxxx' is an integer value indicating ??                               *
//*       'yyyy' is an integer value indicating ??                               *
//*                                                                              *
//*       a) Metadata chunks are defined by the iTunes database of tag names.    *
//*          See liTunes[] array.                                                *
//*          Format: "nnnnAAAAdatainfo"                                          *
//*          'nnnn' is the integer size of the record                            *
//*          'AAAA' is a 4-byte identifier, either four ASCII alpha bytes _OR_   *
//*                 the hex value 0xa9 followed by three ASCII alpha bytes       *
//*          'data' is a "data" atom record: four 32-bit integers + string data  *
//*                 (see "dataAtom" class and "genericTag" const above)          *
//*                                                                              *
//*                                                                              *
//*    3) The 'xml' type object contains XML string data, usually UTF-8 format,  *
//*       but potentially another format specified by the handler object.        *
//*                                                                              *
//*    4) The 'bxml' type object contains unsigned-byte (uint8_t) binary data,   *
//*       which fills the box.                                                   *
//*                                                                              *
//*    5) The 'iloc' type (location) object specifies the location of metadata   *
//*       either in the current file or an external file.                        *
//*       a) This object begins with three 4-bit integer values:                 *
//*          -- offset_size: bytes of the offset field                           *
//*          -- length_size: bytes of the length field                           *
//*          -- base_offset_size: bytes of the base_offset field                 *
//*       b) The values must be of the set {0,4,8}                               *
//*          -- An offset of 0 indicates beginning of file.                      *
//*          -- A length of 0 indicates the entire file.                         *
//*       c) The 4th integer (4-bit) is reserved.                                *
//*       d) The 5th integer (16-bit) is the item count                          *
//*       e) The items follow:                                                   *
//*          -- item_ID (16-bit)                                                 *
//*          -- data_reference_index (16-bit)                                    *
//*          -- base_offset (base_offset_size * 8)                               *
//*          -- extent_count (16-bit)                                            *
//*             -- extent_offset (offset_size * 8)                               *
//*             -- extent_length (length_size * 8)                               *
//*                                                                              *
//*    6) The 'pitm' type (primary item) object                                  *
//*                                                                              *
//*    7) The 'ipro' type (item protection) object                               *
//*                                                                              *
//*    8) The 'iinf' type (item information) object                              *
//*                                                                              *
//*    9) URLs may refer to other files OR to the current file.                  *
//*                                                                              *
//*   10) Static Metadata: ISO/IEC 14496-12, section 8.44.8                      *
//*       a) simple textual                                                      *
//*          -- The only defined user-data box is the copyright 'cprt'.          *
//*          -- Other registered box types                                       *
//*          -- The UUID escape type                                             *
//*          -- Registered tags from MPEG-7 ISO/IEC 15938                        *
//*             -- Text: handler == 'mp7t' Unicode                               *
//*                An 'xml' box contains the text data, OR a primary item box    *
//*                which identifies the item containing the MPEG-7 XML data.     *
//*             -- Binary: handler == 'mp7b' compressed binary in BIM format     *
//*                A 'bxml' box contains the configuration info, followed by     *
//*                the binarized 'xml' data.                                     *
//*                                                                              *
//********************************************************************************

static void m4aExtractMetadata ( m4aParms& p, mdBox& metabox )
{
   mdBox mdbRef, 
         mdbHdlr ;                        // "hdlr" box contents
   int64_t  metaSize = metabox.metasize ; // unread bytes in the "meta" container
   uint32_t gCnt ;                        // bytes read for each source read

   //* Read and decode the "meta.version" field *
   metaSize -= gCnt = m4aReadBytes ( p, intLEN ) ;

   #if DEBUG_M4A != 0
   uint32_t modulo ;                      // box size multiple of eight(8) bytes?
   gsdbg2.formatInt( metabox.size, 11, true ) ;
   gsdbg1.compose( "\n"
                   "** m4aExtractMetadata **\n"
                   " type    : %s\n"
                   " size    : %s (%Xh)\n",
                   metabox.type, gsdbg2.ustr(), &metabox.size ) ;
   p.ofs << gsdbg1.ustr() ;
   gsdbg2.formatInt( metabox.metasize, 11, true ) ;
   gsdbg1.compose( " metasize: %s (%Xh)\n"
                   " valid   : %s\n",
                   gsdbg2.ustr(), &metabox.metasize, 
                   (metabox.valid ? "yes" : "no") ) ;
   p.ofs << gsdbg1.ustr() << endl ;
   #endif   // DEBUG_M4A

   if ( gCnt == intLEN )
   {
      if ( (mdbRef.decodeInt32 ( p.ubuff )) == ZERO )
      {
         //* Read and decode the "hdlr" (handler) box.*
         gCnt = m4aReadHeader ( p ) ;
         metaSize -= gCnt ;
         if ( gCnt == idLEN )
         {
            if ( (mdbHdlr.decodeHeader ( p.ubuff )) )
            {
               #if DEBUG_M4A != 0
               modulo = mdbHdlr.size % idLEN ;
               gsdbg2.formatInt( mdbHdlr.size, 11, true ) ;
               gsdbg1.compose( " type    : %s\n"
                               " size    : %s (%X)",
                               mdbHdlr.type, gsdbg2.ustr(), &mdbHdlr.size ) ;
               if ( modulo != ZERO )
                  gsdbg1.append( " mod:%u", &modulo ) ;
               gsdbg2.formatInt( mdbHdlr.metasize, 11, true ) ;
               gsdbg1.append( "\n metasize: %s (%Xh)\n",
                              gsdbg2.ustr(), &mdbHdlr.metasize ) ; 
               p.ofs << gsdbg1.ustr() ;
               p.ofs.flush() ;
               #endif   // DEBUG_M4A

               //* Read to the end of the 'hdlr' box object. *
               //* It is assumed that it will fit the buffer.*
               gCnt = m4aReadBytes ( p, mdbHdlr.metasize ) ;
               metaSize -= gCnt ;
               if ( gCnt == mdbHdlr.metasize )
               {
                  //* Validate 'version' and 'handler-type' integers.*
                  //* Get the null-terminated handler-type string.   *
                  uint32_t hdlrVersion = mdbHdlr.decodeInt32( p.ubuff ) ;
                  mdbHdlr.handler = mdbHdlr.decodeInt32( &p.ubuff[intLEN] ) ;
                  mdbHdlr.metadata = new uint8_t[mdbHdlr.metasize - (intLEN * 2) + 1] ;
                  uint32_t srcindx = (intLEN * 2),  // index handler name
                           trgindx = ZERO ;         
                  while ( srcindx < mdbHdlr.metasize )
                     mdbHdlr.metadata[trgindx++] = p.ubuff[srcindx++] ;
                  mdbHdlr.metadata[trgindx] = NULLCHAR ; // be sure string is terminated

                  //* If handler data verified,     *
                  //* scan remainder of 'meta' box. *
                  if ( hdlrVersion == ZERO )
                     mdbHdlr.valid = true ;
                  else
                     mdbHdlr.valid = false ;

                  #if DEBUG_M4A != 0
                  gsdbg1.compose( " version : %02Xh\n"
                                  " hdlrtype: %02Xh\n"
                                  " hdlrname: %s\n"
                                  " valid   : %s\n", 
                                  &hdlrVersion, &mdbHdlr.handler, mdbHdlr.metadata, 
                                  (mdbHdlr.valid ? "yes" : "no") ) ;
                  p.ofs << gsdbg1.ustr() << endl ;
                  #endif   // DEBUG_M4A
               }
               #if DEBUG_M4A != 0
               else
                  p.ofs << "Error! \"meta.hdlr\" truncated." << endl ;
               #endif   // DEBUG_M4A
            }
            #if DEBUG_M4A != 0
            else
            {
               p.ofs << "Error! \"meta.hdlr\" out-of-synch." << endl ;
            }
            #endif   // DEBUG_M4A
         }
         #if DEBUG_M4A != 0
         else
            p.ofs << "Error! \"meta.hdlr\" incorrectly formated." << endl ;
         #endif   // DEBUG_M4A
      }
      #if DEBUG_M4A != 0
      else
         p.ofs << "Error! \"meta\" version != ZERO" << endl ;
      #endif   // DEBUG_M4A
   }
   #if DEBUG_M4A != 0
   else
      p.ofs << "Error! \"meta\" incorrectly formatted." << endl ;
   #endif   // DEBUG_M4A

   #if 0    // For Debugging Only - DUMP LiTunes[] ARRAY
   ofs << "\n** LiTunes[] **" << endl ;
   for ( uint32_t i = ZERO ; i < looneyITEMS ; ++i )
   {
      p.ofs << liTunes[i].hName << liTunes[i].aName << endl ;
   }
   p.ofs << endl ;
   #endif   // For Debugging Only

   //************************************************************
   //* If 'hdlr' box verified, scan the remainder of 'meta' box *
   //* and extract any text or image data.                      *
   //************************************************************
   if ( mdbHdlr.valid )
   {
      gString gsDisplay, gstmp ; // formatting
      uint32_t ltIndex ;         // index into liTunes[] array

      //* Read and decode the remainder of the "meta" box object.*
      gCnt = m4aReadHeader ( p ) ;
      metaSize -= gCnt ;

      if ( gCnt == idLEN )
      {
         do
         {
            mdbRef.reset() ;
            if ( (mdbRef.decodeHeader ( p.ubuff )) )
            {
               metaSize -= mdbRef.metasize ;    // bytes of "meta" box remaining

               #if DEBUG_M4A !=  0
               modulo = mdbRef.size % idLEN ;
               gsdbg2.formatInt( mdbRef.size, 11, true ) ;
               gsdbg1.compose( "BOX: %s (%s bytes at x%08X)", 
                              mdbRef.type, gsdbg2.ustr(), &p.srcoffset ) ;
               p.ofs << gsdbg1.ustr() ;
               if ( modulo != ZERO )
               {
                  gsdbg1.compose( " mod:%u", &modulo ) ;
                  p.ofs << gsdbg1.ustr() ;
               }
               p.ofs << endl ;
               #endif   // DEBUG_M4A

// DO NOT DISPLAY RECORDS WITH EMPTY STRINGS
// IMPLEMENT SUPPORT FOR 'XML' RECORDS.
// IMPLEMENT SUPPORT FOR 'BXML' RECORDS.
// IMPLEMENT SUPPORT FOR IMAGE FIELDS.
// IMPLEMENT SUPPORT FOR METADATA WHICH IS NOT IN THE 'META' BOX.
               //* Select metadata processing method.*
               gstmp = (char*)mdbRef.type ;
               if ( (gstmp.compare( "ilst" )) == ZERO )
               {
                  m4aExtractMD_ilst ( p, mdbRef ) ;
                  metaSize -= mdbRef.metasize ;
               }
               else if ( (gstmp.compare( "xml " )) == ZERO )
               {
                  m4aExtractMD_xml ( p, mdbRef ) ;
                  metaSize -= mdbRef.metasize ;
               }
               else if ( (gstmp.compare( "bxml" )) == ZERO )
               {
                  m4aExtractMD_bxml ( p, mdbRef ) ;
                  metaSize -= mdbRef.metasize ;
               }
               else if ( (gstmp.compare( "free" )) == ZERO )
               {
                  m4aExtractMD_free ( p, mdbRef ) ;
                  metaSize -= mdbRef.metasize ;
               }
               else if ( (gstmp.compare( "skip" )) == ZERO )
               {
                  m4aExtractMD_skip ( p, mdbRef ) ;
                  metaSize -= mdbRef.metasize ;
               }
               else if ( (ltIndex = m4aIsLooneyTunes ( mdbRef.type )) < looneyITEMS )
               {
                  itAtom itaRef ;

                  //* Read the remainder of the atom.*
                  metaSize -= m4aReadBytes ( p, (mdbRef.size - idLEN ) ) ;

                  if ( (itaRef.initialize ( mdbRef.size, ltIndex, 
                                            (char*)mdbRef.type, p.ubuff )) )
                  {
                     //* Strip leading and trailing whitespace from *
                     //* the data, for a more professional display. *
                     gstmp = itaRef.metaStr ;
                     gstmp.strip() ;
                     gstmp.copy( itaRef.metaStr, itaMETA_MAX ) ;

                     #if DEBUG_M4A == 0   // PRODUCTION
                     p.ofs << liTunes[ltIndex].hName << itaRef.metaStr << endl ;
                     #else                // DEBUG ONLY
                     //* Replace BYTE_A9 (0xA9) with '_' (0x5F).*
                     //*     (0xA9 is not valid UTF-8 data.)    *
                     if ( (uint8_t)itaRef.atomName[0] == BYTE_A9 )
                        itaRef.atomName[0] = '_' ;

                     uint32_t foffset = p.srcoffset - idLEN ;
                     gsdbg1.compose( "ATOM: %s (bytes: %u x%02X @ x%08X)"
                                     " constA:%08X || %s%s",
                                     itaRef.atomName, 
                                     &itaRef.atomSize, &itaRef.atomSize,
                                     &foffset, 
                                     &itaRef.dataConst1, 
                                     liTunes[itaRef.aIndex].hName, itaRef.metaStr ) ;
                     p.ofs << gsdbg1.ustr() << endl ;
                     #endif   // DEBUG_M4A
                  }
               }
               else  // unparsed/unreported metadata
               {
                  metaSize -= m4aDiscardBytes ( p, mdbRef.metasize ) ;
               }
            }

            //* Not a valid box header, shift left, append *
            //* a byte from the input stream and continue. *
            else
            {
               metaSize -= m4aShiftHeader ( p ) ;
            }
         }
         while ( metaSize > ZERO ) ;
         #if DEBUG_M4A != 0
         p.ofs << "** End m4aExtractMetadata **" << endl ;
         #endif   // DEBUG_M4A
      }
      else
      {
         #if DEBUG_M4A != 0
         p.ofs << "Read error before end of \"meta\" reached." << endl ;
         #endif   // DEBUG_M4A
      }
   }        // (mdbHdlr.valid)

}  //* End m4aExtractMetadata() *

//*************************
//*   m4aExtractMD_ilst   *
//*************************
//******************************************************************************
//* Non-member Method: called only by m4aExtractMetadata()                     *
//* ------------------                                                         *
//* Process the contents of an "ilst" box object.                              *
//*                                                                            *
//*                                                                            *
//* Input  : p      : (by reference) M4A-capture parameters                    *
//*          mdbCon : (by reference) header information for containing object  *
//*                                                                            *
//* Returns: number of bytes read from input stream                            *
//******************************************************************************

static void m4aExtractMD_ilst ( m4aParms& p, const mdBox& mdbCon )
{
   gString gsDisplay, gstmp ; // formatting and display
   mdBox    mdbRef ;          // temp for decoding and reference (MPEG-4 boxes)
   dataAtom dtaRef ;          // temp for decoding and reference ("data" atom)
   itAtom   itaRef ;          // temp for decoding and reference (iTunes atoms)
   udefAtom udfRef ;          // temp for decoding and reference (user-defined atom)

   uint32_t atomSize,         // decoded byte count for an atom
            trg = ZERO,       // index into 'atomName'
            ltIndex ;         // index into liTunes[] array
   int64_t  ilst_remain = mdbCon.metasize ; // unread bytes in the 'ilst' object
   char     atomName[intLEN + 1] ;

   while ( ilst_remain > ZERO )
   {
      //* Read 8 bytes from the input stream *
      if ( (m4aReadHeader ( p )) == idLEN )
      {
         ilst_remain -= idLEN ;

         //* Scan for ASCII-alpha atom name,         *
         //* plus "----" (freeform) atom name.       *
         //* plus 0xA9 used by the iTunes dictionary.*
         trg = ZERO ;
         for ( uint32_t indx = intLEN ; indx < idLEN ; ++ indx )
         {
            if ( ((p.ubuff[indx] >= 'a') && (p.ubuff[indx] <= 'z')) ||
                 ((p.ubuff[indx] >= 'A') && (p.ubuff[indx] <= 'Z')) ||
                 (p.ubuff[indx] == '-') || (p.ubuff[indx] == BYTE_A9) )
            {
               atomName[trg++] = p.ubuff[indx] ;
            }
         }

         //* If four consecutive potential atom-name characters captured *
         if ( trg == intLEN )
         {
            //* Compare atom identifier against the list of *
            //* Apple(tm) iTunes(tm) dictionary entries.    *
            //* These have a 4-byte fixed-length key.       *
            if ( (ltIndex = m4aIsLooneyTunes ( (uint8_t*)atomName )) < looneyITEMS )
            {
               //* Decode the number of bytes in the atom.*
               atomSize = mdbCon.decodeInt32( p.ubuff ) ;

               //* Read the remainder of the atom.*
               ilst_remain -= m4aReadBytes ( p, (atomSize - idLEN ) ) ;

               //* Decode and validate the atom.*
               if ( (itaRef.initialize ( atomSize, ltIndex, atomName, p.ubuff )) )
               {
                  //* Strip leading and trailing whitespace from *
                  //* the data, for a more professional display. *
                  gstmp = itaRef.metaStr ;
                  gstmp.strip() ;
                  gstmp.copy( itaRef.metaStr, itaMETA_MAX ) ;

                  #if DEBUG_M4A == 0   // PRODUCTION
                  p.ofs << liTunes[ltIndex].hName << itaRef.metaStr << endl ;
                  #else                // DEBUG ONLY
                  //* Replace BYTE_A9 (0xA9) with '_' (0x5F).*
                  //*     (0xA9 is not valid UTF-8 data.)    *
                  if ( (uint8_t)itaRef.atomName[0] == BYTE_A9 )
                     itaRef.atomName[0] = '_' ;

                  uint32_t foffset = p.srcoffset - idLEN ;
                  gsdbg1.compose( "ATOM: %s (bytes: %u x%02X @ x%08X)"
                                  " constA:%08X || %s%s",
                                  itaRef.atomName, 
                                  &itaRef.atomSize, &itaRef.atomSize,
                                  &foffset, 
                                  &itaRef.dataConst1, 
                                  liTunes[itaRef.aIndex].hName, itaRef.metaStr ) ;
                  p.ofs << gsdbg1.ustr() << endl ;
                  #endif   // DEBUG_M4A
               }
            }

            //*****************************************************
            //* If not looney-toons, test for MPEG-4 type name    *
            //*****************************************************
            else if ( (mdbRef.decodeHeader ( (uint8_t*)atomName )) )
            {
               //* If all four characters of the name field are ASCII printing *
               //* characters AND if name is a registered box type.            *
               #if DEBUG_M4A != 0
               uint32_t modulo = mdbRef.size % mdbRef.size ;
               gsdbg2.formatInt( mdbRef.size, 11, true ) ;
               gsdbg1.compose( "BOX : %s (%s bytes at x%08X)", 
                               mdbRef.type, gsdbg2.ustr(), &p.srcoffset ) ;
               p.ofs << gsdbg1.ustr() ;
               if ( modulo != ZERO )
               {
                  gsdbg1.compose( " mod:%u", &modulo ) ;
                  p.ofs << gsdbg1.ustr() ;
               }
               p.ofs << endl ;
               #endif   // DEBUG_M4A

               // There may never be an MPEG-4 box inside an 'islt' container. 
               // If such a box object is identified during testing, we will 
               // implement processing here.
            }

            //*****************************************************
            //* '----' freeform atom                              *
            //* This is a variable-length atom identifier.        *
            //*****************************************************
            else if ( (atomName[0] == '-') && (atomName[1] == '-') &&
                      (atomName[2] == '-') && (atomName[3] == '-') )
            {
               //* Decode the number of bytes in the atom.*
               atomSize = mdbCon.decodeInt32( p.ubuff ) ;

               //* Read the remainder of the atom.*
               ilst_remain -= m4aReadBytes ( p, (atomSize - idLEN ) ) ;

               //* Decode and validate the atom.*
               if ( (udfRef.initialize ( atomSize, atomName, p.ubuff )) )
               {
                  //* Strip leading and trailing whitespace from *
                  //* the data, for a more professional display. *
                  gstmp = udfRef.metaStr ;
                  gstmp.strip() ;
                  gstmp.copy( udfRef.metaStr, itaMETA_MAX ) ;

                  gstmp = udfRef.nameDesc ;
                  gstmp.padCols( tagDescWIDTH, L'-' ) ;
                  
                  #if DEBUG_M4A == 0   // PRODUCTION
                  p.ofs << gstmp.ustr() << udfRef.metaStr << endl ;
                  #else                // DEBUG ONLY
                  uint32_t foffset = p.srcoffset - idLEN ;
                  gsdbg1.compose( "ATOM: %s (bytes: %u x%02X @ x%08X)\n"
                                  "      %s (bytes: %u x%02X, const:%02X) '%s'\n"
                                  "      %s (bytes: %u x%02X const:%02X) '%s'\n"
                                  "      %s (bytes: %u x%02X const1:%02X const2:%02X)\n"
                                  "      %s%s",
                                  udfRef.atomName, 
                                  &udfRef.atomSize, &udfRef.atomSize, &foffset,
                                  udfRef.meanKey, &udfRef.meanSize, &udfRef.meanSize,
                                  &udfRef.meanConst, udfRef.meanDesc,
                                  udfRef.nameKey, &udfRef.nameSize, &udfRef.nameSize,
                                  &udfRef.nameConst, udfRef.nameDesc,
                                  udfRef.dataKey, &udfRef.dataSize, &udfRef.dataSize,
                                  &udfRef.dataConst1, &udfRef.dataConst2,
                                  gstmp.ustr(), udfRef.metaStr ) ;
                  p.ofs << gsdbg1.ustr() << endl ;
                  #endif   // DEBUG_M4A
               }
            }        // freeform atom

            //*****************************************************
            //* Else, test for an un-encapsulated "data" atom.    *
            //*****************************************************
            else if ( (atomName[0] == genericTag.aName[0]) && 
                      (atomName[1] == genericTag.aName[1]) &&
                      (atomName[2] == genericTag.aName[2]) && 
                      (atomName[3] == genericTag.aName[3]) )
            {
               //* Decode the number of bytes in the atom.*
               atomSize = mdbCon.decodeInt32( p.ubuff ) ;

               //* Read the remainder of the atom.*
               ilst_remain -= m4aReadBytes ( p, (atomSize - idLEN ) ) ;

               if ( (dtaRef.initialize( atomSize, atomName, p.ubuff )) )
               {

                  #if DEBUG_M4A == 0   // PRODUCTION
                  p.ofs << genericTag.hName << dtaRef.metaStr << endl ;
                  #else                // DEBUG ONLY
                  uint32_t foffset = p.srcoffset - idLEN ;
                  gsdbg1.compose( "ATOM: %s (bytes: %u x%02X @ x%08X)"
                                  " constA:%08X || %s%s",
                                  dtaRef.atomName, 
                                  &dtaRef.atomSize, &dtaRef.atomSize,
                                  &foffset, 
                                  &dtaRef.dataConst1, 
                                  genericTag.hName, dtaRef.metaStr ) ;
                  p.ofs << gsdbg1.ustr() << endl ;
                  #endif   // DEBUG_M4A
               }
            }        // "data" atom

            //* Unrecognized box/atom name *
            else
            {
               if ( (m4aShiftHeader ( p )) != 1 )
                  break ;  // end-of-file (unlikely)
               --ilst_remain ;
            }
         }  // (trg==intLEN)

         //* Else out-of-synch. Shift in another byte and continue.*
         else
         {
            if ( (m4aShiftHeader ( p )) != 1 )
               break ;  // end-of-file (unlikely)
            --ilst_remain ;
         }
      }  // m4aReadHeader()

      else  // end-of-file (unlikely)
         break ;

   }  // while()

}  //* End m4aExtractMD_ilst() *

//*************************
//*   m4aExtractMD_xml    *
//*************************
//******************************************************************************
//* Non-member Method:                                                         *
//* ------------------                                                         *
//*                                                                            *
//* Input  : p      : (by reference) M4A-capture parameters                    *
//*          mdbCon : (by reference) header information for containing object  *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

static void m4aExtractMD_xml ( m4aParms& p, const mdBox& mdbCon )
{

#if 1    // TEMP - m4aExtractMD_xml NOT IMPLEMENTED
gString gstmp( "m4aExtractMD_xml( %s %u bytes)", mdbCon.type, &mdbCon.size ) ;
p.ofs << gstmp.ustr() << endl ;
m4aDiscardBytes ( p, mdbCon.metasize ) ;
#endif   // U/C

}  //* End m4aExtractMD_xml() *

//*************************
//*   m4aExtractMD_bxml   *
//*************************
//******************************************************************************
//* Non-member Method:                                                         *
//* ------------------                                                         *
//*                                                                            *
//* Input  : p      : (by reference) M4A-capture parameters                    *
//*          mdbCon : (by reference) header information for containing object  *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

static void m4aExtractMD_bxml ( m4aParms& p, const mdBox& mdbCon )
{

#if 1    // TEMP - m4aExtractMD_bxml NOT IMPLEMENTED
gString gstmp( "m4aExtractMD_bxml( %s %u bytes)", mdbCon.type, &mdbCon.size ) ;
p.ofs << gstmp.ustr() << endl ;
m4aDiscardBytes ( p, mdbCon.metasize ) ;
#endif   // U/C

}  //* End m4aExtractMD_bxml() *

//*************************
//*   m4aExtractMD_schi   *
//*************************
//******************************************************************************
//* Non-member Method:                                                         *
//* ------------------                                                         *
//* The current box object is a "schi" box.                                    *
//* It may contain a wide variety information, but we are interested ONLY in   *
//* the name (or account name) of the person who downloaded the file from      *
//* iTunes if it was recorded here.                                            *
//*                                                                            *
//* Input  : p      : (by reference) M4A-capture parameters                    *
//*          mdbCon : (by reference) header information for containing object  *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

static void m4aExtractMD_schi ( m4aParms& p, const mdBox& mdbCon )
{
   luniTunesDB schiTag = {(const uint8_t*)"name", "Downloaded By----------"} ;

   //* Read in the remainder of the box object.*
   if ( (m4aReadBytes ( p, mdbCon.metasize )) == mdbCon.metasize )
   {
      gString  gstmp ;
      uint8_t  nameBuff[intLEN + 1] ;     // receives "name" const string
      uint32_t indx = ZERO,               // p.ubuff index
               nbi  = ZERO ;              // nameBuff index

      //* Scan for the const "name" *
      while ( indx < mdbCon.metasize )
      {
         if ( (p.ubuff[indx] == schiTag.aName[0]) && (nbi == 0) )
            nameBuff[nbi++] = p.ubuff[indx] ;
         else if ( (p.ubuff[indx] == schiTag.aName[1]) && (nbi == 1) )
            nameBuff[nbi++] = p.ubuff[indx] ;
         else if ( (p.ubuff[indx] == schiTag.aName[2]) && (nbi == 2) )
            nameBuff[nbi++] = p.ubuff[indx] ;
         else if ( (p.ubuff[indx] == schiTag.aName[3]) && (nbi == 3) )
         {
            nameBuff[nbi++] = p.ubuff[indx] ;
            nameBuff[nbi] = NULLCHAR ;
            gstmp = (char*)nameBuff ;
            break ;
         }
         else
            nbi = ZERO ;
         ++indx ;
      }

      if ( (gstmp.compare ( (char*)schiTag.aName )) == ZERO )
      {
         //* Capture and display the name *
         gstmp.compose( "%s%s", schiTag.hName, &p.ubuff[++indx] ) ;
         p.ofs << gstmp.ustr() << endl ;
      }
   }

}  //* End m4aExtractMD_schi() *

//*************************
//*   m4aExtractMD_free   *
//*************************
//******************************************************************************
//* Non-member Method:                                                         *
//* ------------------                                                         *
//* The current box object is a "free" box.                                    *
//* Scan the data for any human-readable information, and if found report it.  *
//*                                                                            *
//* Input  : p      : (by reference) M4A-capture parameters                    *
//*          mdbCon : (by reference) header information for containing object  *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

static void m4aExtractMD_free ( m4aParms& p, const mdBox& mdbCon )
{
   if ( (m4aReadBytes ( p, mdbCon.metasize )) == mdbCon.metasize )
   {
      #if DEBUG_M4A != 0
      gString gstmp( "Contents of \"free\" atom:" ) ;
      uint32_t baseCnt = gstmp.gschars(),
               indx = ZERO ;

      while ( indx < mdbCon.metasize )
      {
         if (   (((p.ubuff[indx] >= ' ') && (p.ubuff[indx] >= '~')) 
             || (p.ubuff[indx] == '\n')) && (gstmp.gschars() < gsALLOCDFLT) )
         {
            gstmp.append( (wchar_t)p.ubuff[indx] ) ;
         }
         ++indx ;
      }
      if ( (uint32_t)(gstmp.gschars()) > baseCnt )
      {
         gstmp.strip() ;
         p.ofs << gstmp.ustr() << endl ;
      }
      #endif   // DEBUG_M4A
   }

}  //* End m4aExtractMD_free() *

//*************************
//*   m4aExtractMD_skip   *
//*************************
//******************************************************************************
//* Non-member Method:                                                         *
//* ------------------                                                         *
//* The current box object is a "skip" box.                                    *
//* Scan the data for any human-readable information, and if found report it.  *
//*                                                                            *
//* Input  : p      : (by reference) M4A-capture parameters                    *
//*          mdbCon : (by reference) header information for containing object  *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

static void m4aExtractMD_skip ( m4aParms& p, const mdBox& mdbCon )
{
   if ( (m4aReadBytes ( p, mdbCon.metasize )) == mdbCon.metasize )
   {
      #if DEBUG_M4A != 0
      gString gstmp( "Contents of \"skip\" atom:" ) ;
      uint32_t baseCnt = gstmp.gschars(),
               indx = ZERO ;

      while ( indx < mdbCon.metasize )
      {
         if (   (((p.ubuff[indx] >= ' ') && (p.ubuff[indx] >= '~')) 
             || (p.ubuff[indx] == '\n')) && (gstmp.gschars() < gsALLOCDFLT) )
         {
            gstmp.append( (wchar_t)p.ubuff[indx] ) ;
         }
         ++indx ;
      }
      if ( (uint32_t)(gstmp.gschars()) > baseCnt )
      {
         gstmp.strip() ;
         p.ofs << gstmp.ustr() << endl ;
      }
      #endif   // DEBUG_M4A
   }

}  //* End m4aExtractMD_skip() *

#if 0    // CURRENTLY UNUSED
//*************************
//*   m4aExtractMD_udta   *
//*************************
//******************************************************************************
//* Non-member Method:                                                         *
//* ------------------                                                         *
//* The current box object is a "udta" box.                                    *
//* Scan the data for any human-readable information, and if found report it.  *
//*                                                                            *
//* Input  : p      : (by reference) M4A-capture parameters                    *
//*          mdbCon : (by reference) header information for containing object  *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

static void m4aExtractMD_udta ( m4aParms& p, const mdBox& mdbCon ) ;
static void m4aExtractMD_udta ( m4aParms& p, const mdBox& mdbCon )
{
   if ( (m4aReadBytes ( p, mdbCon.metasize )) == mdbCon.metasize )
   {
      #if DEBUG_M4A != 0
      #endif   // DEBUG_M4A
   }

}  //* End m4aExtractMD_udta() *
#endif   // CURRENTLY UNUSED

//*************************
//*   m4aExtractMD_uuid   *
//*************************
//******************************************************************************
//* Non-member Method:                                                         *
//* ------------------                                                         *
//* The current box object is a "uuid" box.                                    *
//* Scan the data for any human-readable information, and if found report it.  *
//*                                                                            *
//* Input  : p      : (by reference) M4A-capture parameters                    *
//*          mdbCon : (by reference) header information for containing object  *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

static void m4aExtractMD_uuid ( m4aParms& p, const mdBox& mdbCon )
{
   if ( (m4aReadBytes ( p, mdbCon.metasize )) == mdbCon.metasize )
   {
      #if DEBUG_M4A != 0
#if 1    // CZONE - REPORT "uuid" ATOM
#endif   // U/C
      #endif   // DEBUG_M4A
   }

}  //* End m4aExtractMD_uuid() *

//*************************
//*   m4aReadFileHeader   *
//*************************
//******************************************************************************
//* Non-member Method:                                                         *
//* ------------------                                                         *
//* Caller has opened the source file but has not yet read any data from it.   *
//* Scan the input stream until the file header is found.                      *
//*                                                                            *
//* The "ftyp" box is usually the first box defined; however, the specification*
//* states that it MUST be positioned as close to the beginning of the file as *
//* possible, and before any variable-size boxes.                              *
//*                                                                            *
//* When the file header has been located, parse the data and verify           *
//* formatting.                                                                *
//*                                                                            *
//* If the file header has not been identified within the first 2.0KB of the   *
//* file, it is assumed that the file is not an M4A file.                      *
//*                                                                            *
//* Input  : p      : (by reference) M4A-capture parameters                    *
//*                   p.fileHdr receives the file header data                  *
//*                                                                            *
//* Returns: 'true' if file header successfully read and format verified       *
//*          'false' if file header not found OR if invalid format             *
//******************************************************************************

static bool m4aReadFileHeader ( m4aParms& p )
{
   mdBox mdbRef ;          // temp storage
   gString gstmp ;         // data analysis
   uint32_t idLen ;        // bytes returned from m4aReadHeader()

   p.fileHdr.reset() ;     // re-initialize the target structure

   while ( ! p.fileHdr.valid && ! p.eof && (p.fileBytes < 0x0800) )
   {
      //* Read the 8-byte header record.*
      if ( (idLen = m4aReadHeader ( p )) == idLEN )
      {
         if ( (mdbRef.decodeHeader( p.ubuff )) )
         {
            //* If "ftyp" object identified *
            if ( mdbRef.tindex == ftypINDEX )
            {
               //* Read the remainder of the object.*
               if ( (m4aReadBytes ( p, mdbRef.metasize )) == mdbRef.metasize )
               {
                  //* Decode and format the raw data.         *
                  //* If successful, p.fileHdr is initialized.*
                  p.fileHdr.initialize( mdbRef.size, mdbRef.type, p.ubuff ) ;
                  
                  #if DEBUG_M4A != 0
                  gsdbg2.formatInt( p.fileHdr.size, 11, true ) ;
                  gsdbg1.compose( "M4A File Header:\n"
                                  " size  : %s (%Xh)\n"
                                  " type  : '%s'\n"
                                  " major : '%s'\n"
                                  " minor : 0x%08X\n"
                                  " compat: '%s'\n"
                                  " valid : %s\n",
                                  gsdbg2.ustr(), &p.fileHdr.size, p.fileHdr.type, 
                                  p.fileHdr.major, &p.fileHdr.minor, p.fileHdr.compat,
                                  (p.fileHdr.valid ? "yes" : "no") ) ;
                  p.ofs << gsdbg1.ustr() << endl ;
                  #endif   // DEBUG_M4A
               }
            }

            //* Else, not the file header. Discard the object.*
            else
            {
               m4aReadBytes ( p, mdbRef.metasize ) ;
            }
         }
      }
      else           // file read error, unable to continue
         break ;
   }
   return p.fileHdr.valid ;

}  //* End m4aReadFileHeader() *

//*************************
//*     m4aReadBytes      *
//*************************
//******************************************************************************
//* Non-member Method:                                                         *
//* ------------------                                                         *
//* Read 'byteCount' bytes from the input stream into the buffer.              *
//* This method consolidates all reads from the "xxx.m4a" source file.         *
//*                                                                            *
//* Input  : p      : (by reference) M4A-capture parameters                    *
//*          byteCnt: number of bytes to read                                  *
//*          offset : (optional, ZERO by default)                              *
//*                   if non-zero, write new data at specified p.ubuff offset  *
//*                   if zero, write new data at beginning of p.ubuff          *
//*                                                                            *
//* Returns: number of bytes read from input stream                            *
//*          p.ubuff      receives the data                                    *
//*          p.fileBytes  accumulator updated                                  *
//*          p.eof        set if bytes read < bytes requested                  *
//*          p.srcoffset  previous file offset (for debugging only)            *
//******************************************************************************

static uint32_t m4aReadBytes ( m4aParms& p, uint32_t byteCnt, uint32_t offset )
{
   if ( byteCnt > ubuffLEN )     // prevent buffer overflow
      byteCnt = ubuffLEN ;
   if ( (offset > ubuffLEN) || (byteCnt > (ubuffLEN - offset)) )
      offset = ZERO ;
   uint32_t gCnt = ZERO ;

   if ( (byteCnt > ZERO) && ! p.eof ) // if not yet at end-of-file
   {
      p.ifs.read( (char*)&p.ubuff[offset], byteCnt ) ;
      gCnt = p.ifs.gcount() ;
      if ( gCnt < byteCnt )
         p.eof = true ;

      #if DEBUG_M4A != 0
      p.srcoffset = p.fileBytes ;   // save previous file offset
      #endif   // DEBUG_M4A

      p.fileBytes += gCnt ;         // update current file offset
   }

   return ( gCnt ) ;

}  //* End m4aReadBytes() *

//*************************
//*     m4aReadHeader     *
//*************************
//******************************************************************************
//* Non-member Method:                                                         *
//* ------------------                                                         *
//* Read idLEN bytes from the input stream into the buffer.                    *
//*                                                                            *
//* For convenience, a null character '\0' is appended after bytes read.       *
//*                                                                            *
//* Input  : p      : (by reference) M4A-capture parameters                    *
//*                                                                            *
//* Returns: number of bytes read from input stream (idLEN if successful)      *
//******************************************************************************

static uint32_t m4aReadHeader ( m4aParms& p )
{

   uint32_t gCnt = m4aReadBytes ( p, idLEN ) ;
   p.ubuff[gCnt] = NULLCHAR ;

   return ( gCnt ) ;

}  //* End m4aReadHeader() *

//*************************
//*    m4aShiftHeader     *
//*************************
//******************************************************************************
//* Non-member Method:                                                         *
//* ------------------                                                         *
//* Shift the first idLEN bytes in the buffer left by one byte and append      *
//* one byte from the input stream.                                            *
//*                                                                            *
//* Programmer's Note: We use the 'offset' parameter of m4aReadBytes() to      *
//* bring in the new byte at buffer offset (idLEN - 1).                        *
//*                                                                            *
//* For convenience, a null character '\0' is appended to the shifted data.    *
//*                                                                            *
//* Input  : p      : (by reference) M4A-capture parameters                    *
//*                                                                            *
//* Returns: number of bytes read from input stream (1 if success, else 0)     *
//******************************************************************************

static uint32_t m4aShiftHeader ( m4aParms& p )
{
   #if DEBUG_M4A != 0
   #define DEBUG_M4A_VERBOSE (0)
   #if DEBUG_M4A_VERBOSE != 0
   gsdbg1.compose( "B: %02hhX %02hhX %02hhX %02hhX %02hhX %02hhX %02hhX %02hhX",
                   &p.ubuff[0], &p.ubuff[1], &p.ubuff[2], &p.ubuff[3], 
                   &p.ubuff[4], &p.ubuff[5], &p.ubuff[6], &p.ubuff[7] ) ;
   p.ofs << gsdbg1.ustr() << endl ;
   #endif   // DEBUG_M4A_VERBOSE
   #endif   // DEBUG_M4A

   #if DEBUG_M4A != 0
   p.srcoffset = p.fileBytes ;
   #endif   // DEBUG_M4A

   uint32_t src = 1 ;
   for ( uint32_t trg = ZERO ; src < idLEN ; ++trg, ++src )
      p.ubuff[trg] = p.ubuff[src] ;

   --src ;                          // position the index
   uint32_t gCnt = m4aReadBytes ( p, 1, src ) ;
   if ( gCnt == 1 )
      p.ubuff[idLEN] = NULLCHAR ;   // terminate the (potential) box name string
   else
      p.ubuff[idLEN - 1] = NULLCHAR ;

   #if DEBUG_M4A != 0 && DEBUG_M4A_VERBOSE != 0
   gsdbg1.compose( "A: %02hhX %02hhX %02hhX %02hhX %02hhX %02hhX %02hhX %02hhX\n",
                   &p.ubuff[0], &p.ubuff[1], &p.ubuff[2], &p.ubuff[3], 
                   &p.ubuff[4], &p.ubuff[5], &p.ubuff[6], &p.ubuff[7] ) ;
   p.ofs << gsdbg1.ustr() << endl ;
   #endif   // DEBUG_M4A && DEBUG_M4A_VERBOSE

   return ( p.ifs.gcount() ) ;

   #undef DEBUG_M4A_VERBOSE
}  //* End m4aShiftHeader() *

//*************************
//*    m4aDiscardBytes    *
//*************************
//******************************************************************************
//* Non-member Method:                                                         *
//* ------------------                                                         *
//* Read and discard the specified number of bytes from the input stream.      *
//* This method may be called to scan past the data of a box object which      *
//* contains neither human-readable metadata (text/images) nor other           *
//* interesting box objects.                                                   *
//*                                                                            *
//* Input  : p      : (by reference) M4A-capture parameters                    *
//*          byteCnt: number of bytes to read                                  *
//*                                                                            *
//* Returns: number of bytes read from input stream                            *
//*          p.ubuff      receives the data                                    *
//*          p.fileBytes  accumulator updated                                  *
//*          p.eof        set if bytes read < bytes requested                  *
//*          p.srcoffset  previous file offset (for debugging only)            *
//******************************************************************************

static uint32_t m4aDiscardBytes ( m4aParms& p, uint32_t byteCnt )
{
   uint32_t sub = ubuffLEN,      // stream divisor (efficiency w/o buffer overflow)
            bytesRead = ZERO,    // total bytes read (return value)
            gCnt ;               // bytes read each pass

   for ( int64_t remain = byteCnt ; remain > ZERO ; )
   {
      if ( remain < sub )
         sub = remain ;

      gCnt = m4aReadBytes ( p, sub ) ;
      remain -= gCnt ;
      bytesRead -= gCnt ;
   }
   return bytesRead ;

}  //* End m4aDiscardBytes() *

//*************************
//*   m4aIsLooneyTunes    *
//*************************
//******************************************************************************
//* Non-member Method:                                                         *
//* ------------------                                                         *
//* Compare the specified string to the elements of the liTunes[] array which  *
//* implements the Apple iTunes dictionary of tag identifiers.                 *
//*                                                                            *
//* Caller's data compared to the 'aName' element of the record.               *
//* If a match, then the 'hName' (human-readable name) can be used to format   *
//* and display the metadata record.                                           *
//*                                                                            *
//* Input  : atomName: name of atom to be tested                               *
//*                                                                            *
//* Returns: index of matching record in the dictionary                        *
//*          or looneyITEMS if no match found                                  *
//******************************************************************************

static uint32_t m4aIsLooneyTunes ( const uint8_t* atomName )
{
   uint32_t ltIndex ;

   for ( ltIndex = ZERO ; ltIndex < looneyITEMS ; ++ltIndex )
   {
      if ( (atomName[0] == liTunes[ltIndex].aName[0]) &&
           (atomName[1] == liTunes[ltIndex].aName[1]) &&
           (atomName[2] == liTunes[ltIndex].aName[2]) &&
           (atomName[3] == liTunes[ltIndex].aName[3]) &&
           (atomName[4] == NULLCHAR) )
         break ;
   }

   return ltIndex ;

}  //* End m4aIsLoneyTunes() *



//******************************************************************************
//******  Definitions for ASF container (WMA Audio and WMV Video Files)  *******
//******************************************************************************
//* Notes on ASF formatting:                                                   *
//* ------------------------                                                   *
//* ASF (Advanced Systems Format, formerly Advanced Streaming Format or Active *
//* Streaming Format) is a Microsoft proprietary container format. Any use of  *
//* this container format without licensing will likely result in lawsuits     *
//* from the evil troll in Redmmond, WA. You have been warned.                 *
//*                                                                            *
//* For our purposes, we simply read the human-readable metadata, which so far *
//* as we know, is not illegal. However, Microsloth is without humor or        *
//* humanity when it comes to their patents and copyrights.                    *
//* This author was formerly a beta-tester for early versions of Windows(tm),  *
//* and it is difficult to express precisely how much we loathe Microsoft and  *
//* the horse it rode in on.                                                   *
//*                                                                            *
//* This code is based on ASF version 1.2 (Dec. 2004).                         *
//*                                                                            *
//* Each ASF object takes the format:                                          *
//* GUID: 128 bits (16 bytes)    This is the unique identifier for the object. *
//* Size:  64 bits ( 8 bytes)    24 + size of data in bytes.                   *
//* Data:                        'Size' - 24 bytes of data >= ZERO bytes.      *
//*                                                                            *
//* Numeric Format:                                                            *
//* ---------------                                                            *
//* A) All values are unsigned values.                                         *
//* B) All GUID values are stored as a sequence of three little-endian values  *
//*    followed by two big-endian values. This is just freaky-stupid i.e.      *
//*    the Microsloth way.                                                     *
//*    Example  : 75B22630-668E-11CF-A6D9-00AA0062CE6C                         *
//*    Stored as: 3026B275 - 8E66 - CF11 - A6D9 - 00AA0062CE6C                 *
//* C) 64-bit values (e.g. the object-size value) is stored in                 *
//*                                                                            *
//* Text Format:                                                               *
//* ------------                                                               *
//* Unless otherwise specified, human-readable text within the file will be    *
//* encoded as UTF-16 (little-endian, no byte-order mark), null terminated     *
//* unless otherwise specified. This is the Windoze standard for text data     *
//* because up until mid-2019 they seemed to believe that UTF-8 was a          *
//* Commie plot. Actually UTF-16 is considered to be a serious risk to         *
//* computer security, and Microsloth is slowly transitioning to UTF-8.        *
//*                                                                            *
//* Overall Structure:                                                         *
//* ------------------                                                         *
//* A) Each file begins with a top-level Header Object.                        *
//*    Only this Header Object may contain other ASF objects.                  *
//*    "Bibliographic data" are contained in a Content Description object.     *
//* B) A top-level Data Object follows the header.                             *
//*    This object contains the audio/video or other binary data.              *
//*    These data are ignored in our context.                                  *
//* C) An _optional_ Index Object (if present) must be the last object in      *
//*    the file. These data are ignored in our context.                        *
//*                                                                            *
//* Header Object: (see asfHeader class)                                       *
//* ------------------------------------                                       *
//* 128 bits      GUID ASF_Header_Object                                       *
//*                    75B22630-668E-11CF-A6D9-00AA0062CE6C                    *
//*  64 bits      object size                                                  *
//*  32 bits      number of objects contained in header (not incl. this one)   *
//*   8 bits      reserved (always 0x01)                                       *
//*   8 bits      reserved (always 0x02)                                       *
//*                                                                            *
//* Content Description Object: (see asfConDesc class)                         *
//* --------------------------------------------------                         *
//* 128 bits      GUID ASF_Content_Description_Object                          *
//*                    75B22633-668E-11CF-A6D9-00AA0062CE6C                    *
//*  64 bits      object size (34 bytes minimum)                               *
//*  16 bits      Title length        (bytes)                                  *
//*  16 bits      Author length          "                                     *
//*  16 bits      Copyright length       "                                     *
//*  16 bits      Description length     "                                     *
//*  16 bits      Rating length          "                                     *
//*  varies       Title text          (UTF-16LE, no BOM)                       *
//*  varies       Author text            "                                     *
//*  varies       Copyright text         "                                     *
//*  varies       Description text       "                                     *
//*  varies       Rating text            "                                     *
//*                                                                            *
//*       (optional, only one allowed)                                         *
//*                                                                            *
//* Extended Content Description Object: (see asfConDesc class)                *
//* -----------------------------------------------------------                *
//* 128 bits      GUID ASF_Extended_Content_Description_Object                 *
//*                    D2D0A440-E307-11D2-97F0-00A0C95EA850                    *
//*  64 bits      object size (26 bytes minimum)                               *
//*  16 bits      number of attached descriptor objects (see below)            *
//*       (optional, only one allowed)                                         *
//*                                                                            *
//* Content Descriptor:                                                        *
//* -------------------                                                        *
//* 16 bits       Descriptor-name length                                       *
//* varies        Descriptor Name (UTF-16 little-endian, no BOM)               *
//* 16 bits       Descriptor-value data type (see enum asfDataType)            *
//* 16 bits       Descriptor-value length                                      *
//* varies        Descriptor Value (UTF-16 little-endian, no BOM)              *
//*                                                                            *
//* Content Branding Object                                                    *
//* -----------------------                                                    *
//* 128 bits      GUID ASF_Content_Branding_Object                             *
//*  64 bits      object size (40 bytes minimum)                               *
//*  32 bits      banner image type                                            *
//*               00 00 00 00 == none                                          *
//*               01 00 00 00 == bitmap                                        *
//*               02 00 00 00 == JPEG                                          *
//*               03 00 00 00 == GIF                                           *
//*  32 bits      banner image data size (bytes)                               *
//*   varies      image data                                                   *
//*  32 bits      banner image URL length                                      *
//*   varies      banner image URL (ASCII)                                     *
//*  32 bits      copyright URL length                                         *
//*   varies      copyright URL (ASCII)                                        *
//*                                                                            *
//* Metadata Object (ignored)                                                  *
//* -------------------------                                                  *
//*                                                                            *
//* Metadata Library Object (ignored)                                          *
//* ---------------------------------                                          *
//*                                                                            *
//******************************************************************************

//* Disable code that is unnecessary for this application.  *
//* This code is lifted directly from Taggit v:0.0.8 but    *
//* this application uses only the source parsing algorithm.*
#define ENABLE_WRITE_CODE (0)

//* Parse CBO, MO and MLO objects   (1) *
//* Discard CBO, MO and MLO objects (0) *
#define PARSE_EXTRA_METADATA (0)

const short asfASF_HDR_BYTES = 30 ; // number of bytes in ASF container header
const short asfGUID_BYTES = 16 ;    // number of bytes in a GUID field
const short asfGUID_ASCII = 37 ;    // ASCII-hex equivalent of asfGUID_BYTES (incl. null)
const short asfSIZE_BYTES = 8 ;     // number of bytes in an object size value (uint64_t)
const short asfHOBJ_BYTES = 4 ;     // number of bytes in header-object count value (uint32_t)
const uint8_t asfRES1     = 0x01 ;  // value of header reserved-byte #1
const uint8_t asfRES2     = 0x02 ;  // value of header reserved-byte #1
const short asfMIN_CDO    = 34 ;    // minimum number of bytes in CDO
const short asfMIN_ECDO   = 26 ;    // minimum number of bytes in ECDO
const short asfDESC_BYTES = 2 ;     // number of bytes in descriptor-count and descriptor sizes
const short asfECD_MAX_COUNT = 64 ; // maximum number of Extended Content Descriptors (array size)
//* For descriptors which contain images, the 'descValue' field contains a *
//* preamble before the image data. See notes in asfReadBinaryDescriptor().*
const uint16_t PREAMBLE_BYTES = 29 ;
const uint64_t KB64 = 0x010000 ; // Size of dynamically-allocated input buffer


//* Types of metadata an ASF Container may hold.       *
//* These are defined in the ASF specification,        *
//* Revision: 01.20.03 December 2004                   *
//* (these constants declared 'extern' in TagAsf.hpp.) *
//* ASF_Header_Object *
const char* HEADER_GUID = "75B22630-668E-11CF-A6D9-00AA0062CE6C" ;
//* ASF_Content_Description_Object *
const char* CDO_GUID    = "75B22633-668E-11CF-A6D9-00AA0062CE6C" ;
//* ASF_Extended_Content_Description_Object *
const char* ECDO_GUID   = "D2D0A440-E307-11D2-97F0-00A0C95EA850" ;
//* ASF_Content_Branding_Object *
const char* CBO_GUID    = "2211B3FA-BD23-11D2-B4B7-00A0C955FC6E" ;
//* ASF_Metadata_Object *
const char* MO_GUID     = "C5F8CBEA-5BAF-4877-8467-AA8C44FA4CCA" ;
//* ASF_Metadata_Library_Object *
const char* MLO_GUID    = "44231C94-9498-49D1-A141-1D134E457054" ;

//* Convert GUID from little-endian (mixed with lunacy) *
//* to the character order defined in the specification.*
const short canonicalOrder[asfGUID_ASCII] = 
{  6,  7,  4,  5,  2,  3,  0,  1, 0x002D, 10, 11,  8,  9, 0x002D, 14, 15, 
  12, 13, 0x002D, 16, 17, 18, 19, 0x002D, 20, 21, 22, 23, 24, 25, 26, 27, 
  28, 29, 30, 31, 32 } ;

//* Descriptor names that may indicate an image descriptor *
static const short inCOUNT = 4 ;
static const char* const imageNames[inCOUNT] = 
{
   "Picture",
   "Image",
   "Photo",
   "CoverArt",
} ;

//* Data types for Extended ContentDescriptionObject descriptor value fields. *
//* Note: non-text data converted to ASCII hex for local storage and display. *
enum asfDataType : short
{
   asfdtUNKNOWN = (-1),
   asfdtUTF16,             // UTF-16LE (no BOM)
   asfdtBYTE,              // byte array    (image, or other binary data)
   asfdtBOOL,              // boolean value (source: 32 bits) [not text]
   asfdtDWORD,             // "DWORD"       (source: 32 bits) [not text]
   asfdtQWORD,             // "QWORD"       (source: 64 bits) [not text]
   asfdtWORD,              // "WORD"        (source: 16 bits) [not text]
} ;

//* Return value from asfConDesc::validateObjectHeader().   *
//* These are the valid metadata objects specified by the   *
//* Advanced Systems Format (ASF) specification.            *
enum ObjHdrType : short
{
   ohtCDO,                 // Content Description Object
   ohtECDO,                // Extended Content Description Object
   ohtCBO,                 // Content Branding Object
   ohtMO,                  // Metadata Object
   ohtMLO,                 // Metadata Library Object
   ohtNONE,                // Invalid or unrecognized object structure
} ;

#if ENABLE_WRITE_CODE != 0
//* Binary-encoded sequences corresponding to the GUID above definitions.*
const uint8_t EncodedGUID[][asfGUID_BYTES] = 
{
   //* Content Description Object *
   { 0x33, 0x26, 0xB2, 0x75, 0x8E, 0x66, 0xCF, 0x11, 
     0xA6, 0xD9, 0x00, 0xAA, 0x00, 0x62, 0xCE, 0x6C },
   //* Extended Content Description Object *
   { 0x40, 0xA4, 0xD0, 0xD2, 0x07, 0xE3, 0xD2, 0x11, 
     0x97, 0xF0, 0x00, 0xA0, 0xC9, 0x5E, 0xA8, 0x50 },
   //* Content Branding Object *
   { 0xFA, 0xB3, 0x11, 0x22, 0x23, 0xBD, 0xD2, 0x11, 
     0xB7, 0xB4, 0x00, 0xA0, 0xC9, 0x55, 0xFC, 0x6E },
   //* Metadata Object *
   { 0xEA, 0xCB, 0xF8, 0xC5, 0xAF, 0x5B, 0x77, 0x48, 
     0x67, 0x84, 0xAA, 0x8C, 0x44, 0xFA, 0x4C, 0xCA },
   //* Metadata Library Object *
   { 0x94, 0x1C, 0x23, 0x44, 0x98, 0x94, 0xD1, 0x49,
     0x41, 0xA1, 0x1D, 0x13, 0x4E, 0x45, 0x70, 0x54 },
   //* Advanced Systems Format (ASF) Header *
   { 0x30, 0x26, 0xB2, 0x75, 0x8E, 0x66, 0xCF, 0x11, 
     0xA6, 0xD9, 0x00, 0xAA, 0x00, 0x62, 0xCE, 0x6C },
} ;

//* For display of source-data fields which cannot be mapped *
//* to a relevant display field, these characters are used   *
//* to delimit name/value pairs.                             *
const wchar_t DELIM_NAME  = L'名' ;
const wchar_t DELIM_VALUE = L'值' ;

//* Basic list of ASF Tag-field names *
const short AFMA_COUNT = 5 ;     // number of pre-defined field names (CDO)
const short AFMB_COUNT = 15 ;    // number of search substrings defined for
                                 // comparison with user-defined field names

//* Information for an image embedded in the metadata.*
class EmbeddedImage
{
   public:

   ~EmbeddedImage ( void ) {}       // destructor
   EmbeddedImage ( void )
   { this->reset() ; }

   void reset ( void )
   {
      this->mp3img.reset() ;
      this->next = this->prev = NULL ;
      this->imMod = this->inUse = false ;
   }

   //***********************
   //* Public Data Members *
   //***********************
   //* id3v2 image frame data definition                                       *
   id3v2_image mp3img ;

   //* 'true' if image or associated data have been modified, else false.      *
   bool imMod ;

   //* 'true' if 'mp3img' member contains data.                                *
   bool inUse ;

   EmbeddedImage* next ;            // pointers for a linked list of objects
   EmbeddedImage* prev ;

} ;

//* Temporary storage for extracted image data *
class picList
{
   public:
   picList ( void )
   { this->reset () ; }

   void reset ( void )
   {
      *this->picPath = '\0' ;
      *this->mimType = '\0' ;
      this->picSize = 0 ;
   }

   char  picPath[gsDFLTBYTES] ;     // filespec of image
   char  mimType[gsDFLTBYTES] ;     // MIME type of image
   short picSize ;                  // size of image in bytes (<=64Kib)
} ;

class AsfFieldMap
{
   public:
   const wchar_t* const fName ;  // ASF field name or partial name (substring)
   const short fLen ;            // number of characters to compare
   //const TagFields fIndex ;      // display-field index
} ;
#endif // ENABLE_WRITE_CODE

//***********************************
//* Classes for processing ASF data *
//***********************************
//* Top-level Header Object data.*
class asfHeader
{
   public:
   ~asfHeader ( void ) {}        // destructor
   asfHeader ( void )            // default constructor
   {
      this->reset() ;
   }

   void reset ( void )
   {
      this->guid[0] = '\0' ;
      this->size = 0 ;
      this->hdrobj = 0 ;
      this->res1 = this->res2 = 0 ;
   }

   //* Normalize, validate and store the GUID value.   *
   //* Decode and store the object size.               *
   //* Decode and store the number of embedded objects.*
   //* Verify and store the two reserved bytes.        *
   bool validateHeader ( const char* srcPtr )
   {
      uint8_t uBuff[asfGUID_ASCII+1] ;
      uint8_t rawByte, lowNybble, highNybble ;
      short   srcindx = ZERO, trgindx = ZERO ;
      bool valid = false ;

      //* Decode the GUID byte stream into ASCII hex *
      for ( ; srcindx < asfGUID_BYTES ; ++srcindx )
      {
         rawByte = srcPtr[srcindx] ;
         highNybble = (uint8_t)(rawByte >> 4) ;
         if ( highNybble <= 0x09 )  highNybble += 0x30 ;          // '0'
         else                       highNybble += 0x41 - 0x0A ;   // 'A'
         lowNybble = (uint8_t)(rawByte & 0x0F) ;
         if ( lowNybble <= 0x09 )   lowNybble += 0x30 ;           // '0'
         else                       lowNybble += 0x41 -0x0A ;     // 'A'
         uBuff[trgindx++] = highNybble ;
         uBuff[trgindx++] = lowNybble ;
      }
      uBuff[trgindx] = '\0' ;

      //* Place data in canonical order *
      for ( trgindx = ZERO ; trgindx <= asfGUID_ASCII ; ++trgindx )
      {
         if ( canonicalOrder[trgindx] == 0x002D )
            this->guid[trgindx] = '-' ;
         else
         {
            this->guid[trgindx] = uBuff[canonicalOrder[trgindx]] ;
            if ( this->guid[trgindx] == '\0' )
               break ;
         }
      }
      this->guid[trgindx] = '\0' ;  // be sure sequence is terminated

      //* Extract header size (uint64_t) *
      this->size  =  ((uint64_t)srcPtr[srcindx++] & 0x00FF) ;
      this->size |= (((uint64_t)srcPtr[srcindx++] & 0x00FF) << 8) ;
      this->size |= (((uint64_t)srcPtr[srcindx++] & 0x00FF) << 16) ;
      this->size |= (((uint64_t)srcPtr[srcindx++] & 0x00FF) << 24) ;
      this->size |= (((uint64_t)srcPtr[srcindx++] & 0x00FF) << 32) ;
      this->size |= (((uint64_t)srcPtr[srcindx++] & 0x00FF) << 40) ;
      this->size |= (((uint64_t)srcPtr[srcindx++] & 0x00FF) << 48) ;
      this->size |= (((uint64_t)srcPtr[srcindx++] & 0x00FF) << 56) ;

      //* Extract object count (uint32_t) *
      this->hdrobj  =  ((uint32_t)srcPtr[srcindx++] & 0x00FF) ;
      this->hdrobj |= (((uint32_t)srcPtr[srcindx++] & 0x00FF) << 8) ;
      this->hdrobj |= (((uint32_t)srcPtr[srcindx++] & 0x00FF) << 16) ;
      this->hdrobj |= (((uint32_t)srcPtr[srcindx++] & 0x00FF) << 24) ;

      //* Extract the reserved markers *
      this->res1 = srcPtr[srcindx++] ;
      this->res2 = srcPtr[srcindx] ;

      if ( ((strncmp ( (const char*)this->guid, HEADER_GUID, asfGUID_ASCII )) == 0) &&
           (this->res1 == asfRES1) && (this->res2 == asfRES2) )
      {
         valid = true ;
      }
      return valid ;
   }     // validateHeader()

#if ENABLE_WRITE_CODE != 0
   //* Encode a 64-bit QWORD into an eight-byte byte stream. *
   short encode64Bit ( uint64_t val64, char* trgBuff ) const
   {
      for ( short i = 0 ; i < asfSIZE_BYTES ; ++i )
      {
         trgBuff[i] = (val64 & 0x00FF) ;
         val64 >>= 8 ;
      }
      return asfSIZE_BYTES ;  // return size of encoded value
   }

   //* Encode a 32-bit DWORD into a four-byte byte stream. *
   short encode32Bit ( uint32_t val32, char* trgBuff ) const
   {
      for ( short i = 0 ; i < 4 ; ++i )
      {
         trgBuff[i] = (val32 & 0x00FF) ;
         val32 >>= 8 ;
      }
      return 4 ;     // return size of encoded value
   }

   //* Return the encoded GUID byte sequence.                  *
   //* Encoded sequences are defined in EncodedGUID[] array.   *
   void encodeGUID ( uint8_t *ubuff ) const
   {
      short indx = 5 ;           // index into EncodedGUID[] array

      for ( short i = 0 ; i < asfGUID_BYTES ; ++i )
         ubuff[i] = EncodedGUID[indx][i] ;

   }  // encodeGUID()
#endif // ENABLE_WRITE_CODE

   //* Public data members *
   uint8_t  guid[asfGUID_ASCII+1] ; // Global Unique Identifier
   uint64_t size ;         // Size of object 24 bytes + size of data (>= 30)
   uint32_t hdrobj ;       // Number of object contained in header (excl. this object)
   uint8_t  res1 ;         // Reserved (always 0x01)
   uint8_t  res2 ;         // Reserved (always 0x02)
} ;

//* Extended Content Description Object decoded data *
class asfExDesc
{
   public:
   ~asfExDesc ( void ) {}        // destructor
   asfExDesc ( void )            // default constructor
   {
      this->reset() ;
   }

   void reset ( void )
   {
      this->descName[0] = this->descValue[0] = '\0' ;
      this->nameSize = this->valueSize = 0 ;
      this->dataType = asfdtUNKNOWN ;
   }

   //* Public data members *
   char descName[gsDFLTBYTES] ;  // Name of descriptor  (UTF-8 text)
   char descValue[gsDFLTBYTES] ; // Value of descriptor (UTF-8 text)
   uint16_t nameSize ;           // bytes of encoded source for 'descName'
   uint16_t valueSize ;          // bytes of encoded source for 'descValue'
   asfDataType dataType ;        // type of source data for 'descValue' field
} ;

//* Content Description Object and Extended Content Description Object data.  *
//* Text data are stored as UTF-8 regardless of the native encoding used.     *
class asfConDesc
{
   public:
   ~asfConDesc ( void ) {}       // destructor
   asfConDesc ( void )           // default constructor
   {
      this->reset() ;
   }

   void reset ( void )
   {
      //* CDO data *
      this->guid[0] = this->title[0]  = this->author[0] = this->copyrt[0] = 
      this->desc[0] = this->rating[0] = 0 ;
      this->cdoSize =  0 ;
      this->titleSize = this->authorSize = this->copyrtSize = 
      this->descSize  = this->ratingSize = 0 ;
      this->cdoVerified = false ;
      //* ECDO data *
      this->eguid[0]  = '\0' ;   this->ecdoSize = 0 ; this->ecdoVerified = false ;
      this->descCount = 0 ;
      //* CBO data *
      this->cboguid[0] = '\0' ;  this->cboSize = 0 ;  this->cboVerified = false ;
      //* MO data
      this->moguid[0] = '\0' ;   this->moSize = 0 ;   this->moVerified = false ;
      //* MLO data *
      this->mloguid[0] = '\0' ;  this->mloSize = 0 ;  this->mloVerified = false ;
   }

   //* Decode the object GUID and object size.*
   //* If one of the following objects found, *
   //* store the data and return confirmation.*
   //* 1) Content Description Object          *
   //* 2) Extended Content Description Object *
   //* 3) Content Branding Object             *
   //* 4) Metadata Object                     *
   //* 5) Metadata Library Object             *
   ObjHdrType validateObjectHeader ( const char* srcPtr, uint64_t& objSize )
   {
      uint8_t uBuff[asfGUID_ASCII+1] ;          // temp storage (raw guid)
      uint8_t gtmp[asfGUID_ASCII+1] ;           // temp storage (formatted guid)
      uint8_t rawByte, lowNybble, highNybble ;  // integer conversion
      short   srcindx = ZERO, trgindx = ZERO ;
      ObjHdrType metaHdr = ohtNONE ;            // return value

      //* Decode the GUID byte stream into ASCII hex *
      for ( ; srcindx < asfGUID_BYTES ; ++srcindx )
      {
         rawByte = srcPtr[srcindx] ;
         highNybble = (uint8_t)(rawByte >> 4) ;
         if ( highNybble <= 0x09 )  highNybble += 0x30 ;          // '0'
         else                       highNybble += 0x41 - 0x0A ;   // 'A'
         lowNybble = (uint8_t)(rawByte & 0x0F) ;
         if ( lowNybble <= 0x09 )   lowNybble += 0x30 ;           // '0'
         else                       lowNybble += 0x41 -0x0A ;     // 'A'
         uBuff[trgindx++] = highNybble ;
         uBuff[trgindx++] = lowNybble ;
      }
      uBuff[trgindx] = '\0' ;

      //* Place data in canonical order *
      for ( trgindx = ZERO ; trgindx <= asfGUID_ASCII ; ++trgindx )
      {
         if ( canonicalOrder[trgindx] == 0x002D )
            gtmp[trgindx] = '-' ;
         else
         {
            gtmp[trgindx] = uBuff[canonicalOrder[trgindx]] ;
            if ( gtmp[trgindx] == '\0' )
               break ;
         }
      }
      gtmp[trgindx] = '\0' ;     // be sure sequence is terminated

      //* Decode the object size *
      objSize = this->decode64Bit ( &srcPtr[srcindx] ) ;

      //* Test the captured value against the five(5) valid metadata objects *
      //* If a match, save it to our data member.                            *
      // Programmer's Note: We can take some shortcuts with the test because 
      // the specification mandates that there be only one instance of each.
      if ( ((strncmp ( (char*)gtmp, CDO_GUID, asfGUID_ASCII )) == 0) )
      {
         strncpy ( (char*)this->guid, (char*)gtmp, asfGUID_ASCII ) ;
         this->cdoSize = objSize ;
         this->cdoVerified = true ;
         metaHdr = ohtCDO ;
      }
      else if ( ((strncmp ( (char*)gtmp, ECDO_GUID, asfGUID_ASCII )) == 0) )
      {
         strncpy ( (char*)this->eguid, (char*)gtmp, asfGUID_ASCII ) ;
         this->ecdoSize = objSize ;
         this->ecdoVerified = true ;
         metaHdr = ohtECDO ;
      }
      else if ( ((strncmp ( (char*)gtmp, CBO_GUID, asfGUID_ASCII )) == 0) )
      {
         strncpy ( (char*)this->cboguid, (char*)gtmp, asfGUID_ASCII ) ;
         this->cboSize = objSize ;
         this->cboVerified = true ;
         metaHdr = ohtCBO ;
      }
      else if ( ((strncmp ( (char*)gtmp, MO_GUID, asfGUID_ASCII )) == 0) )
      {
         strncpy ( (char*)this->moguid, (char*)gtmp, asfGUID_ASCII ) ;
         this->moSize = objSize ;
         this->moVerified = true ;
         metaHdr = ohtMO ;
      }
      else if ( ((strncmp ( (char*)gtmp, MLO_GUID, asfGUID_ASCII )) == 0) )
      {
         strncpy ( (char*)this->mloguid, (char*)gtmp, asfGUID_ASCII ) ;
         this->mloSize = objSize ;
         this->mloVerified = true ;
         metaHdr = ohtMLO ;
      }
      else  // save found-but-uninteresting GUID for debugging by caller
      {
         strncpy ( (char*)this->xguid, (char*)gtmp, asfGUID_ASCII ) ;
      }

      return metaHdr ;

   }  // validateObjectHeader()

   //* Extract the five(5) 16-bit tag-size values from the byte stream.*
   void decodeTagSizes ( const char* srcPtr )
   {
      short srcindx = ZERO ;
      this->titleSize = this->decode16Bit ( &srcPtr[srcindx] ) ;
      srcindx += asfDESC_BYTES ;
      this->authorSize = this->decode16Bit ( &srcPtr[srcindx] ) ;
      srcindx += asfDESC_BYTES ;
      this->copyrtSize = this->decode16Bit ( &srcPtr[srcindx] ) ;
      srcindx += asfDESC_BYTES ;
      this->descSize = this->decode16Bit ( &srcPtr[srcindx] ) ;
      srcindx += asfDESC_BYTES ;
      this->ratingSize = this->decode16Bit ( &srcPtr[srcindx] ) ;
   }

   //* Decode eight(8) bytes of the byte stream into a 64-bit QWORD *
   uint64_t decode64Bit ( const char* srcPtr )
   {
      uint64_t qword ;
      short srcindx = ZERO ;

      qword  =  ((uint64_t)srcPtr[srcindx++] & 0x00FF) ;
      qword |= (((uint64_t)srcPtr[srcindx++] & 0x00FF) << 8) ;
      qword |= (((uint64_t)srcPtr[srcindx++] & 0x00FF) << 16) ;
      qword |= (((uint64_t)srcPtr[srcindx++] & 0x00FF) << 24) ;
      qword |= (((uint64_t)srcPtr[srcindx++] & 0x00FF) << 32) ;
      qword |= (((uint64_t)srcPtr[srcindx++] & 0x00FF) << 40) ;
      qword |= (((uint64_t)srcPtr[srcindx++] & 0x00FF) << 48) ;
      qword |= (((uint64_t)srcPtr[srcindx++] & 0x00FF) << 56) ;
      return qword ;

   }  // decode64Bit()

#if ENABLE_WRITE_CODE != 0
   //* Encode a 64-bit QWORD into an eight-byte byte stream. *
   short encode64Bit ( uint64_t val64, char* trgBuff )
   {
      for ( short i = 0 ; i < asfSIZE_BYTES ; ++i )
      {
         trgBuff[i] = (val64 & 0x00FF) ;
         val64 >>= 8 ;
      }
      return asfSIZE_BYTES ;  // return size of encoded value
   }
#endif // ENABLE_WRITE_CODE

   //* Decode four(4) bytes of the byte stream into a 32-bit DWORD *
   uint32_t decode32Bit ( const char* srcPtr )
   {
      uint32_t dword ;
      short srcindx = ZERO ;

      dword  =  ((uint32_t)srcPtr[srcindx++] & 0x00FF) ;
      dword |= (((uint32_t)srcPtr[srcindx++] & 0x00FF) << 8) ;
      dword |= (((uint32_t)srcPtr[srcindx++] & 0x00FF) << 16) ;
      dword |= (((uint32_t)srcPtr[srcindx++] & 0x00FF) << 24) ;

      return dword ;

   }  // decode32Bit()

#if ENABLE_WRITE_CODE != 0
   //* Encode a 32-bit DWORD into a four-byte byte stream. *
   short encode32Bit ( uint32_t val32, char* trgBuff )
   {
      for ( short i = 0 ; i < 4 ; ++i )
      {
         trgBuff[i] = (val32 & 0x00FF) ;
         val32 >>= 8 ;
      }
      return 4 ;     // return size of encoded value
   }
#endif // ENABLE_WRITE_CODE

   //* Decode two(2) bytes of the byte stream into a 16-bit WORD *
   uint16_t decode16Bit ( const char* srcPtr )
   {
      uint16_t word ;
      short srcindx = ZERO ;

      word  =  ((uint16_t)srcPtr[srcindx++] & 0x00FF) ;
      word |= (((uint16_t)srcPtr[srcindx++] & 0x00FF) << 8) ;

      return word ;

   }  // decode16Bit()

#if ENABLE_WRITE_CODE == 0
   //* Encode a 16-bit WORD into a two-byte byte stream. *
   short encode16Bit ( uint16_t val16, char* trgBuff )
   {
      trgBuff[0] = (val16 & 0x00FF) ;
      trgBuff[1] = (val16 >> 8) ;
      return 2 ;     // return size of encoded value
   }
#endif // ENABLE_WRITE_CODE

   //* Convert a Unicode-16 string (little-endian, no byte-order mark)  *
   //* to gString format. Source may or may not be null terminated.     *
   short utf16Decode ( const char* rawPtr, short rawBytes, gString& gsTrg )
   {
      int ti = ZERO ;                        // source index
      gsTrg.clear() ;                        // initialize the target buffer
      uint32_t msUnit, lsUnit,               // for converting unit pairs
               cx ;                          // decoded 16-bit character
      while ( rawBytes > ZERO )
      {
         //* Little-endian conversion *
         cx = (uint32_t(rawPtr[ti++]) & 0x000000FF) ;
         cx |= ((uint32_t(rawPtr[ti++]) << 8) & 0x0000FF00) ;
         rawBytes -= 2 ;

         //* If the character is fully represented by 16 bits *
         //* (most characters are)                            *
         if ( ((cx >= ZERO) && (cx < usxMSU16)) || 
              ((cx >= usxHIGH16) && (cx <= usxMAX16)) )
         {
            gsTrg.append( cx ) ;
            if ( cx == '\0' )    // if premature newline, we're done
               break ;
         }

         //* Character is represented by 32 bits    *
         //* (20 bits actually used), 16 MSBs first.*
         else
         {
            msUnit = cx ;
            lsUnit = ZERO ;
            //* Little-endian conversion *
            lsUnit = (UINT(rawPtr[ti++]) & 0x000000FF) ;
            lsUnit |= ((UINT(rawPtr[ti++]) << 8) & 0x0000FF00) ;
            rawBytes -= 2 ;

            //* Validate the range *
            if ( ((msUnit >= usxMSU16) && (msUnit <= usxMSU16e))
                 &&
                 ((lsUnit >= usxLSU16) && (lsUnit <= usxLSU16e)) )
            {
               cx = usxMIN20 + ((msUnit & usxMASK10) << 10) ;
               cx |= (lsUnit & usxMASK10) ;
            }
            else                 // invalid UTF-16 codepoint
               cx = L'?' ;
            gsTrg.append( cx ) ; // add the chararacter to output buffer
         }
      }
      return ( gsTrg.gschars() ) ;
   }  //* End utf16Decode() *

#if ENABLE_WRITE_CODE != 0
   //* Convert a wchar_t string to UTF-16LE (little-endian, no BOM).*
   //* Returns number of bytes encoded data.                        *
   short utf16Encode ( char* trg, const gString& src )
   {
      short trgCnt = 0 ;      // index into output array (return value)

      //* Establish the source array of wchar_t (wide) characters *
      short wcnt ;   // number of source characters (including the null terminator)
      const wchar_t *wstr = src.gstr( wcnt ) ;
      uint32_t cx ;  // source character

      for ( short wIndex = ZERO ; wIndex < wcnt ; ++wIndex )
      {
         cx = wstr[wIndex] ;     // get a character from the input stream

         //* If the character can be encoded with a single, 16-bit value *
         if ( ((cx >= ZERO) && (cx < usxMSU16)) || 
              ((cx >= usxHIGH16) && (cx <= usxMAX16)) )
         {  //* Encode the bytes in little-endian order *
            trg[trgCnt++] = cx & 0x000000FF ;
            trg[trgCnt++] = (cx >> 8) & 0x000000FF ;
         }

         //* Else, character requires a pair of 16-bit values *
         else
         {
            uint32_t msUnit = ZERO, lsUnit = ZERO ;
            if ( (cx >= usxMIN20) && (cx <= usxMAX20) )
            {
               msUnit = ((cx - usxMIN20) >> 10) | usxMSU16 ;
               lsUnit = (cx & usxMASK10) | usxLSU16 ;
               trg[trgCnt++] = msUnit & 0x000000FF ;
               trg[trgCnt++] = (msUnit >> 8) & 0x000000FF ;
               trg[trgCnt++] = lsUnit & 0x000000FF ;
               trg[trgCnt++] = (lsUnit >> 8) & 0x000000FF ;
            }
            //* Character cannot be encoded as UTF-16. *
            //* Encode a question mark character.      *
            else
            {
               cx = L'?' ;
               trg[trgCnt++] = (cx >> 8) & 0x000000FF ;
               trg[trgCnt++] = cx & 0x000000FF ;
            }
         }
      }
      return trgCnt ;
   }  //* End utf16Encode() *

   //* Given an ASCII-hex GUID string, return the equivalent   *
   //* encoded GUID byte sequence.                             *
   //* Encoded sequences are defined in EncodedGUID[] array.   *
   //* Returns 'true' if recognized GUID string, else 'false'. *
   bool encodeGUID ( const gString& guid, uint8_t *ubuff )
   {
      short indx ;               // index into EncodedGUID[] array
      bool  encoded = true ;     // hope for the best
      if ( (guid.compare( CDO_GUID )) == ZERO )
         indx = ZERO ;
      else if ( (guid.compare( ECDO_GUID )) == ZERO )
         indx = 1 ;
      else if ( (guid.compare( CBO_GUID )) == ZERO )
         indx = 2 ;
      else if ( (guid.compare( MO_GUID )) == ZERO )
         indx = 3 ;
      else if ( (guid.compare( MLO_GUID )) == ZERO )
         indx = 4 ;
      else if ( (guid.compare( HEADER_GUID )) == ZERO )
         indx = 5 ;
      else
         encoded = false ;
      if ( encoded )
      {
         for ( short i = 0 ; i < asfGUID_BYTES ; ++i )
            ubuff[i] = EncodedGUID[indx][i] ;
      }
      return encoded ;
   }
#endif // ENABLE_WRITE_CODE


   //* Public data members *
   uint8_t  guid[asfGUID_ASCII+1] ;    // CDO Global Unique Identifier (normalized)
   uint8_t  eguid[asfGUID_ASCII+1] ;   // Extended CDO GUID (if present)
   uint8_t  cboguid[asfGUID_ASCII+1] ; // Content Branding GUID (if present)
   uint8_t  moguid[asfGUID_ASCII+1] ;  // Metadata GUID (if present)
   uint8_t  mloguid[asfGUID_ASCII+1] ; // Metadata Library GUID (if present)
   uint8_t  xguid[asfGUID_ASCII+1] ;   // GUID for non-metadata objects
                                       // (used primarily for debugging)
   char     title[gsDFLTBYTES] ;    // 'title' tag
   char     author[gsDFLTBYTES] ;   // 'author' tag
   char     copyrt[gsDFLTBYTES] ;   // 'copyright' tag
   char     desc[gsDFLTBYTES] ;     // 'description' tag
   char     rating[gsDFLTBYTES] ;   // 'rating' tag
   uint64_t cdoSize ;               // Size of CDO source object (bytes)
   uint64_t ecdoSize ;              // Size of Extended CDO source object (bytes)
   uint64_t cboSize ;               // Size of CBO source object (bytes)
   uint64_t moSize ;                // Size of MO source object (bytes)
   uint64_t mloSize ;               // Size of MLO source object (bytes)
   uint16_t descCount ;             // Number of ECDO descriptors
   uint16_t titleSize ;             // length of 'title' tag (bytes, UTF-16 string)
   uint16_t authorSize ;            // length of 'author' tag (bytes, UTF-16 string)
   uint16_t copyrtSize ;            // length of 'copyrt' tag (bytes, UTF-16 string)
   uint16_t descSize ;              // length of 'description' tag (bytes, UTF-16 string)
   uint16_t ratingSize ;            // length of 'rating' tag (bytes, UTF-16 string)
   bool     cdoVerified ;           // 'true' if CDO GUID and Size initialized
   bool     ecdoVerified ;          // 'true' if ECDO GUID and Size initialized
   bool     cboVerified ;           // 'true' if CBO GUID and Size initialized
   bool     moVerified ;            // 'true' if MO GUID and Size initialized
   bool     mloVerified ;           // 'true' if MLO GUID and Size initialized
   // Array of Descriptor Name / Descriptor Value records
   asfExDesc exDesc[asfECD_MAX_COUNT] ;

} ;   // asfConDesc

//**********************
//* Non-member methods *
//**********************
static bool asfReadBinaryDescriptor ( char* ibuff, asfConDesc& asfMeta, 
                                      short descIndx, ofstream& dbg ) ;
static uint32_t asfDiscardDescriptor ( char *ibuff, uint32_t objSize, ifstream& ifs ) ;


//*************************
//*  ExtractMetadata_ASF  *
//*************************
//******************************************************************************
//* Read the media file and write the metadata to the temp file.               *
//*                                                                            *
//* Input  : ofs    : open output stream to temporary file                     *
//*          srcPath: filespec of media file to be read                        *
//*          vebose : (optional, 'false' by default)                           *
//*                   if 'false', display all text-frame records               *
//*                   if 'true', ALSO display extended technical data and      *
//*                              non-text frames (used primarily for debugging)*
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void FileDlg::ExtractMetadata_ASF ( ofstream& ofs, const gString& srcPath, bool verbose )
{
   #define DEBUG_EMDATA (0)      // Set to non-zero for debugging only
   #if DEBUG_EMDATA != 0
   const char* BadAsf = "Error: Invalid ASF/WMA format.\n" ;
   uint64_t srcBytes = ZERO ;    // number of source bytes read
   #endif   // DEBUG_EMDATA

   const short CNAME_COLS = 28 ; // output column alignment

   char *ibuff = new char[KB64] ;// input buffer (64KiB dynamic allocation)
   short cnt ;                   // for validating header record
   uint64_t objSize ;            // size (bytes) of current object
   asfHeader   asfHdr ;          // ASF container header
   asfConDesc  asfMeta ;         // Content Desc. / Extended Content Desc. data
   gString gsOut, gstmp ;        // text formatting
   bool valid_hdr = false ;      // 'true' if file format verified

   //* Report basic information about the source file.*
   tnFName fn ;
   this->fmPtr->GetFileStats ( fn, srcPath, true ) ;
   gstmp.formatInt( fn.fBytes, 11, true ) ;
   short fni = srcPath.findlast( fSLASH ) + 1 ;
   ofs << "FILE NAME  : \"" << &srcPath.ustr()[fni] 
       << "\"  (" << gstmp.ustr() << " bytes)\n\n" ;

   //* Open the source file *
   ifstream ifs( srcPath.ustr(), ifstream::in ) ;
   if ( ifs.is_open() )
   {
      //* Read the ASF container header *
      ifs.read ( ibuff, asfASF_HDR_BYTES ) ;
      cnt = ifs.gcount() ;
      if ( cnt == asfASF_HDR_BYTES )
         valid_hdr = asfHdr.validateHeader ( ibuff ) ;

      #if DEBUG_EMDATA != 0
      //* Display the header information *
      gstmp.formatInt( asfHdr.size, 13, true ) ;
      gsOut.compose( "GUID: %s  (at 0x%06llX)\n"
                     "SIZE: %S\n"
                     "OBJS: %u\n"
                     "Res1: %hhd\n"
                     "Res2: %hhd   valid:%s\n",
                     asfHdr.guid, &srcBytes, gstmp.gstr(), &asfHdr.hdrobj,
                     &asfHdr.res1, &asfHdr.res2,
                     (char*)(valid_hdr ? "true" : "false")) ;
      ofs << gsOut.ustr() << endl ;
      srcBytes += cnt ;
      #endif   // DEBUG_EMDATA

      //* If source file format verified as ASF *
      if ( valid_hdr )
      {
         //* Validation of object header *
         ObjHdrType metaHdr ;
         //* Total number of objects in header (loop control) *
         uint64_t hdrObjs = asfHdr.hdrobj ;

         //* Scan the header objects and isolate    *
         //* 1) Content Description Object          *
         //* 2) Extended Content Description Object *
         while ( hdrObjs > ZERO )
         {
            ifs.read ( ibuff, (asfGUID_BYTES + asfSIZE_BYTES) ) ;

            #if DEBUG_EMDATA != 0
            srcBytes += ifs.gcount() ;
            #endif   // DEBUG_EMDATA

            if ( ifs.gcount() == (asfGUID_BYTES + asfSIZE_BYTES) )
            {
               if ( ((metaHdr = asfMeta.validateObjectHeader ( ibuff, objSize ))) != ohtNONE )
               {
                  //********************************************
                  //* If validated header is CDO, get tag data *
                  //********************************************
                  if ( metaHdr == ohtCDO )
                  {
                     //* There are five(5) 16-bit size value for the five     *
                     //* possible tag fields. Read and decode the size values.*
                     //*      (assumes that source bytes are available)       *
                     ifs.read( ibuff, (asfDESC_BYTES * 5) ) ;
                     #if DEBUG_EMDATA != 0
                     srcBytes += ifs.gcount() ;
                     #endif   // DEBUG_EMDATA
                     asfMeta.decodeTagSizes( ibuff ) ;

                     //* Read the tag fields *
                     if ( asfMeta.titleSize > ZERO )
                     {
                        ifs.read( ibuff, asfMeta.titleSize ) ;
                        #if DEBUG_EMDATA != 0
                        srcBytes += ifs.gcount() ;
                        #endif   // DEBUG_EMDATA
                        asfMeta.utf16Decode ( ibuff, asfMeta.titleSize, gstmp ) ;
                        gstmp.copy( asfMeta.title, gsDFLTBYTES ) ;
                     }
                     if ( asfMeta.authorSize > ZERO )
                     {
                        ifs.read( ibuff, asfMeta.authorSize ) ;
                        #if DEBUG_EMDATA != 0
                        srcBytes += ifs.gcount() ;
                        #endif   // DEBUG_EMDATA
                        asfMeta.utf16Decode ( ibuff, asfMeta.authorSize, gstmp ) ;
                        gstmp.copy( asfMeta.author, gsDFLTBYTES ) ;
                     }
                     if ( asfMeta.copyrtSize > ZERO )
                     {
                        ifs.read( ibuff, asfMeta.copyrtSize ) ;
                        #if DEBUG_EMDATA != 0
                        srcBytes += ifs.gcount() ;
                        #endif   // DEBUG_EMDATA
                        asfMeta.utf16Decode ( ibuff, asfMeta.copyrtSize, gstmp ) ;
                        gstmp.copy( asfMeta.copyrt, gsDFLTBYTES ) ;
                     }
                     if ( asfMeta.descSize > ZERO )
                     {
                        ifs.read( ibuff, asfMeta.descSize ) ;
                        #if DEBUG_EMDATA != 0
                        srcBytes += ifs.gcount() ;
                        #endif   // DEBUG_EMDATA
                        asfMeta.utf16Decode ( ibuff, asfMeta.descSize, gstmp ) ;
                        gstmp.copy( asfMeta.desc, gsDFLTBYTES ) ;
                     }
                     if ( asfMeta.ratingSize > ZERO )
                     {
                        ifs.read( ibuff, asfMeta.ratingSize ) ;
                        #if DEBUG_EMDATA != 0
                        srcBytes += ifs.gcount() ;
                        #endif   // DEBUG_EMDATA
                        asfMeta.utf16Decode ( ibuff, asfMeta.ratingSize, gstmp ) ;
                        gstmp.copy( asfMeta.rating, gsDFLTBYTES ) ;
                     }

                     #if DEBUG_EMDATA != 0
                     //* Display the CDO information *
                     uint64_t gpos = srcBytes - ((asfGUID_BYTES + asfSIZE_BYTES)
                        + (2 * 5) + asfMeta.titleSize + asfMeta.authorSize 
                        + asfMeta.copyrtSize + asfMeta.descSize 
                        + asfMeta.ratingSize ) ;
                     gstmp.formatInt( asfMeta.cdoSize, 13, true ) ;
                     gsOut.compose( "CDO  GUID: %s  (at 0x%06llX)\n"
                                    "     SIZE: %S (%llXh)\n"
                                    "    Title: %02hu '%s'\n"
                                    "   Author: %02hu '%s'\n"
                                    "   Copyrt: %02hu '%s'\n"
                                    "     Desc: %02hu '%s'\n"
                                    "   Rating: %02hu '%s'\n",
                                    asfMeta.guid, &gpos, gstmp.gstr(), &asfMeta.cdoSize,
                                    &asfMeta.titleSize,  asfMeta.title,
                                    &asfMeta.authorSize, asfMeta.author,
                                    &asfMeta.copyrtSize, asfMeta.copyrt,
                                    &asfMeta.descSize,   asfMeta.desc,
                                    &asfMeta.ratingSize, asfMeta.rating ) ;
                     ofs << gsOut.ustr() << endl ;
                     #endif   // DEBUG_EMDATA

                     //* Report the captured tag data *
                     if ( asfMeta.titleSize > ZERO )
                     {
                        gsOut = "Title " ;
                        while ( gsOut.gschars() < CNAME_COLS )
                           gsOut.append( L'-' ) ;
                        gsOut.append( L' ' ) ;
                        ofs << gsOut.ustr() << asfMeta.title << endl ;
                     }
                     if ( asfMeta.authorSize > ZERO )
                     {
                        gsOut = "Author " ;
                        while ( gsOut.gschars() < CNAME_COLS )
                           gsOut.append( L'-' ) ;
                        gsOut.append( L' ' ) ;
                        ofs << gsOut.ustr() << asfMeta.author << endl ;
                     }
                     if ( asfMeta.copyrtSize > ZERO )
                     {
                        gsOut = "Copyright " ;
                        while ( gsOut.gschars() < CNAME_COLS )
                           gsOut.append( L'-' ) ;
                        gsOut.append( L' ' ) ;
                        ofs << gsOut.ustr() << asfMeta.copyrt << endl ;
                     }
                     if ( asfMeta.descSize > ZERO )
                     {
                        gsOut = "Description " ;
                        while ( gsOut.gschars() < CNAME_COLS )
                           gsOut.append( L'-' ) ;
                        gsOut.append( L' ' ) ;
                        ofs << gsOut.ustr() << asfMeta.desc << endl ;
                     }
                     if ( asfMeta.ratingSize > ZERO )
                     {
                        gsOut = "Rating " ;
                        while ( gsOut.gschars() < CNAME_COLS )
                           gsOut.append( L'-' ) ;
                        gsOut.append( L' ' ) ;
                        ofs << gsOut.ustr() << asfMeta.rating << endl ;
                     }
                  }

                  //*****************************************************
                  //* If validated header is ECDO, get descriptor count *
                  //*     (assumes that source bytes are available)     *
                  //*****************************************************
                  else if ( metaHdr == ohtECDO )
                  {
                     ifs.read ( ibuff, asfDESC_BYTES ) ;
                     asfMeta.descCount = asfMeta.decode16Bit ( ibuff ) ;

                     #if DEBUG_EMDATA != 0
                     //* Display the CDO information *
                     srcBytes += ifs.gcount() ;
                     uint64_t gpos = srcBytes - ((asfGUID_BYTES + asfSIZE_BYTES)
                              + asfDESC_BYTES) ;
                     gstmp.formatInt( asfMeta.ecdoSize, 13, true ) ;
                     gsOut.compose( "ECDO GUID: %s  (at 0x%06llX)\n"
                                    "     SIZE: %S (%llXh)\n"
                                    "     OBJS: %hu\n",
                                    asfMeta.eguid, &gpos, gstmp.gstr(), 
                                    &asfMeta.ecdoSize, &asfMeta.descCount ) ;
                     ofs << "\n" << gsOut.ustr() << endl ;
                     #endif   // DEBUG_EMDATA

                     //* Read the tag data.                 *
                     //* (do not overrun our storage array) *
                     for ( short descIndx = ZERO ; 
                           ((descIndx < asfMeta.descCount) &&
                            (descIndx < asfECD_MAX_COUNT)) ; ++descIndx )
                     {
                        //* Decode descriptor name length *
                        ifs.read( ibuff, asfDESC_BYTES ) ;
                        #if DEBUG_EMDATA != 0
                        srcBytes += ifs.gcount() ;
                        #endif   // DEBUG_EMDATA
                        asfMeta.exDesc[descIndx].nameSize = asfMeta.decode16Bit ( ibuff ) ;

                        //* Decode descriptor name *
                        ifs.read( ibuff, asfMeta.exDesc[descIndx].nameSize ) ;
                        #if DEBUG_EMDATA != 0
                        srcBytes += ifs.gcount() ;
                        #endif   // DEBUG_EMDATA
                        asfMeta.utf16Decode ( ibuff, asfMeta.exDesc[descIndx].nameSize, 
                                              gstmp ) ;
                        #if 0    // (keep this for now)
                        //* For some reason, the encoder often prepends this  *
                        //* substring to the description name. Remove it.     *
                        if ( (gstmp.find( "WM/" )) == ZERO )
                           gstmp.erase( "WM/" ) ;
                        #endif   // U/C
                        gstmp.copy( asfMeta.exDesc[descIndx].descName, gsDFLTBYTES ) ;

                        //* Decode descriptor value data type *
                        ifs.read( ibuff, asfDESC_BYTES ) ;
                        #if DEBUG_EMDATA != 0
                        srcBytes += ifs.gcount() ;
                        #endif   // DEBUG_EMDATA
                        asfMeta.exDesc[descIndx].dataType = (asfDataType)asfMeta.decode16Bit ( ibuff ) ;
                        
                        //* Decode descriptor value length *
                        ifs.read( ibuff, asfDESC_BYTES ) ;
                        #if DEBUG_EMDATA != 0
                        srcBytes += ifs.gcount() ;
                        #endif   // DEBUG_EMDATA
                        asfMeta.exDesc[descIndx].valueSize = asfMeta.decode16Bit ( ibuff ) ;

                        //* Decode descriptor value.             *
                        //* Descriptors guaranteed to be < 64Kib.*
                        ifs.read( ibuff, asfMeta.exDesc[descIndx].valueSize ) ;
                        #if DEBUG_EMDATA != 0
                        srcBytes += ifs.gcount() ;
                        #endif   // DEBUG_EMDATA
                        if ( asfMeta.exDesc[descIndx].dataType == asfdtUTF16 )
                           asfMeta.utf16Decode ( ibuff, 
                                      asfMeta.exDesc[descIndx].valueSize, gstmp ) ;

                        //* 8-bit byte array - determine type of source data *
                        else if ( asfMeta.exDesc[descIndx].dataType == asfdtBYTE )
                        {
                           //* Extract and save the image data. An simple *
                           //* description is returned in 'descValue'.    *
                           asfReadBinaryDescriptor ( ibuff, asfMeta, descIndx, ofs ) ;
                           gstmp = asfMeta.exDesc[descIndx].descValue ;
                        }

                        //* Numeric values will be reported as ASCII decimal.  *
                        //* Common numeric values are for tracks, track number,*
                        //* runtime, etc.                                      *
                        else if ( (asfMeta.exDesc[descIndx].dataType == asfdtBOOL) ||
                                  (asfMeta.exDesc[descIndx].dataType == asfdtDWORD) )
                        {
                           uint32_t u32 = asfMeta.decode32Bit ( ibuff ) ;
                           gstmp.compose( "%u", &u32 ) ;
                        }
                        else if ( asfMeta.exDesc[descIndx].dataType == asfdtQWORD )
                        {
                           uint64_t u64 = asfMeta.decode64Bit ( ibuff ) ;
                           gstmp.compose( "%llu", &u64 ) ;
                        }
                        else if ( asfMeta.exDesc[descIndx].dataType == asfdtWORD )
                        {
                           uint16_t u16 = asfMeta.decode16Bit ( ibuff ) ;
                           gstmp.compose( "%hu", &u16 ) ;
                        }
                        gstmp.copy( asfMeta.exDesc[descIndx].descValue, gsDFLTBYTES ) ;

                        #if DEBUG_EMDATA != 0
                        gsOut.compose( "%02hd) (%02hu) %s --- (type:%02hXh, %02hu) %s",
                                       &descIndx, 
                                       &asfMeta.exDesc[descIndx].nameSize,
                                       asfMeta.exDesc[descIndx].descName,
                                       &asfMeta.exDesc[descIndx].dataType,
                                       &asfMeta.exDesc[descIndx].valueSize,
                                       asfMeta.exDesc[descIndx].descValue ) ;
                        ofs << gsOut.ustr() << endl ;
                        #endif   // DEBUG_EMDATA
                     }

                     //* If some descriptors were not decoded *
                     // Programmer's Note: We make the assumption that there are 
                     // few, if any WMA files which have more descriptors than 
                     // will fit in our descriptor array. If it happens, we
                     // simply discard the remaining descriptors.
                     if ( asfMeta.descCount > asfECD_MAX_COUNT )
                     {
                        #if DEBUG_EMDATA != 0
                        if ( ofs.is_open() )
                        {
                           ofs << "------------------------------------"
                                  "--------------------------\n"
                                  "CAUTION! asf.exDesc[] array filled. "
                                  "Some metadata may be lost." << endl ;
                        }
                        #endif   // DEBUG_EMDATA
                        uint16_t fieldsize ;
                        for ( short descIndx = asfECD_MAX_COUNT ;
                              descIndx < asfMeta.descCount ; ++descIndx )
                        {
                           ifs.read( ibuff, asfDESC_BYTES ) ;  // desc name
                           fieldsize = asfMeta.decode16Bit( ibuff ) ;
                           #if DEBUG_EMDATA != 0
                           if ( ofs.is_open() )
                              gstmp.compose( "%2hd) Name Bytes :%3hu  ", 
                                             &descIndx, &fieldsize ) ;
                           #endif   // DEBUG_EMDATA
                           ifs.read( ibuff, fieldsize ) ;
                           ifs.read( ibuff, asfDESC_BYTES ) ;  // desc value
                           ifs.read( ibuff, asfDESC_BYTES ) ;  // desc value
                           fieldsize = asfMeta.decode16Bit( ibuff ) ;
                           #if DEBUG_EMDATA != 0
                           ifs.read( ibuff, fieldsize ) ;
                           if ( ofs.is_open() )
                           {
                              gstmp.append( "Value Bytes:%4hu", &fieldsize ) ;
                              ofs << gstmp.ustr() << endl ;
                              if ( descIndx == (asfMeta.descCount - 1) )
                                 ofs << endl ;
                           }
                           #endif   // DEBUG_EMDATA
                        }
                        asfMeta.descCount = asfECD_MAX_COUNT ;
                     }

                     //* Report the captured tag data *
                     for ( short descIndx = ZERO ; 
                              descIndx < asfMeta.descCount ; ++descIndx )
                     {
                        gsOut = asfMeta.exDesc[descIndx].descName ;
                        while ( gsOut.gschars() < CNAME_COLS )
                           gsOut.append( L'-' ) ;
                        gsOut.append( L' ' ) ;
                        ofs << gsOut.ustr() 
                            << asfMeta.exDesc[descIndx].descValue << endl ;
                     }
                  }

                  // Programmer's Note: Currently we ignore the contents of 
                  // Content Branding Object, Metadata Object and 
                  // Metadata Library Object. If in future we find these may
                  // contain useful data, These hooks will get us started.
                  else if ( metaHdr == ohtCBO )
                  {
                     #if DEBUG_EMDATA != 0
                     //* Display the CBO information *
                     srcBytes += ifs.gcount() ;
                     uint64_t gpos = srcBytes - (asfGUID_BYTES + asfSIZE_BYTES) ;
                     gstmp.formatInt( asfMeta.cboSize, 13, true ) ;
                     gsOut.compose( "CBO  GUID: %s  (at 0x%06llX)\n"
                                    "     SIZE: %S (%llXh)\n",
                                    asfMeta.cboguid, &gpos, gstmp.gstr(), 
                                    &asfMeta.cboSize ) ;
                     ofs << gsOut.ustr() << endl ;
                     #endif   // DEBUG_EMDATA

                     objSize -= (asfGUID_BYTES + asfSIZE_BYTES) ;
                     uint32_t bytesRead = asfDiscardDescriptor ( ibuff, objSize, ifs ) ;

                     if ( bytesRead != objSize )   // unexpected EOF
                     {
                        #if DEBUG_EMDATA != 0
                        ofs << "\nERROR! Unexpected end-of-file (CBO object)" << endl ;
                        #endif   // DEBUG_EMDATA

                        break ;     // end of loop
                     }
                  }

                  else if ( metaHdr == ohtMO )
                  {
                     #if DEBUG_EMDATA != 0
                     //* Display the MO information *
                     srcBytes += ifs.gcount() ;
                     uint64_t gpos = srcBytes - (asfGUID_BYTES + asfSIZE_BYTES) ;
                     gstmp.formatInt( asfMeta.moSize, 13, true ) ;
                     gsOut.compose( "MO   GUID: %s  (at 0x%06llX)\n"
                                    "     SIZE: %S (%llXh)\n",
                                    asfMeta.moguid, &gpos, gstmp.gstr(), 
                                    &asfMeta.moSize ) ;
                     ofs << gsOut.ustr() << endl ;
                     #endif   // DEBUG_EMDATA

                     objSize -= (asfGUID_BYTES + asfSIZE_BYTES) ;
                     uint32_t bytesRead = asfDiscardDescriptor ( ibuff, objSize, ifs ) ;

                     if ( bytesRead != objSize )   // unexpected EOF
                     {
                        #if DEBUG_EMDATA != 0
                        ofs << "\nERROR! Unexpected end-of-file (MO object)" << endl ;
                        #endif   // DEBUG_EMDATA
   
                        break ;     // end of loop
                     }
                  }

                  else if ( metaHdr == ohtMLO )
                  {
                     #if DEBUG_EMDATA != 0
                     //* Display the MLO information *
                     srcBytes += ifs.gcount() ;
                     uint64_t gpos = srcBytes - (asfGUID_BYTES + asfSIZE_BYTES) ;
                     gstmp.formatInt( asfMeta.mloSize, 13, true ) ;
                     gsOut.compose( "MLO  GUID: %s  (at 0x%06llX)\n"
                                    "     SIZE: %S (%llXh)\n",
                                    asfMeta.mloguid, &gpos, gstmp.gstr(), 
                                    &asfMeta.mloSize ) ;
                     ofs << gsOut.ustr() << endl ;
                     #endif   // DEBUG_EMDATA

                     objSize -= (asfGUID_BYTES + asfSIZE_BYTES) ;
                     uint32_t bytesRead = asfDiscardDescriptor ( ibuff, objSize, ifs ) ;

                     if ( bytesRead != objSize )   // unexpected EOF
                     {
                        #if DEBUG_EMDATA != 0
                        ofs << "\nERROR! Unexpected end-of-file (MLO object)" << endl ;
                        #endif   // DEBUG_EMDATA
   
                        break ;     // end of loop
                     }
                  }
               }

               //* Discard contents of uninteresting object *
               else
               {
                  #if DEBUG_EMDATA != 0
                  uint64_t gpos = srcBytes - (asfGUID_BYTES + asfSIZE_BYTES) ;
                  gstmp.compose( "DISCARDED OBJECT: %s (size:%llX at 0x%06llX)", 
                                 asfMeta.xguid, &objSize, &gpos ) ;
                  ofs << gstmp.ustr() << endl ;
                  #endif   // DEBUG_EMDATA
                  objSize -= (asfGUID_BYTES + asfSIZE_BYTES) ;

                  uint32_t bytesRead = asfDiscardDescriptor ( ibuff, objSize, ifs ) ;
                  if ( bytesRead != objSize )   // unexpected EOF
                  {
                     #if DEBUG_EMDATA != 0
                     ofs << "\nERROR! Unexpected end-of-file (uninteresting object)" 
                         << endl ;
                     #endif   // DEBUG_EMDATA

                     break ;     // end of loop
                  }
               }
            }
            else     // unexpected EOF (unlikely)
            {
               ofs << "Unexpected End-Of-File! Metadata not found.\n" << endl ;
               break ;
            }
            --hdrObjs ;
         }     // while()
      }        // valid header
      #if DEBUG_EMDATA != 0
      else        // not a valid ASF container
         ofs << BadAsf << endl ;
      #endif   // DEBUG_EMDATA

      ifs.close() ;           // close the source file
   }

   delete [] ibuff ;             // release dynamic allocation
   ibuff = NULL ;

   #if DEBUG_EMDATA != 0
   ofs << endl ;
   #endif   // DEBUG_EMDATA

   #undef DEBUG_EMDATA

}  //* End ExtractMetadata_ASF() *

//**************************
//* asfReadImageDescriptor *
//**************************
//******************************************************************************
//* Called only by asfExtractMetadata().                                       *
//* Caller has determined that a descriptor within the ECDO object contains    *
//* binary (non-text) data.                                                    *
//* Determine whether the descriptor specifies an embedded image or non-image  *
//* data.                                                                      *
//*                                                                            *
//* Compare descriptor name with likely image descriptions.                    *
//* 1) If a match is found:                                                    *
//*    a) Extract the identifier byte, image byte count, reserved value and    *
//*       MIME type from input stream.                                         *
//*    b) Test the MIME type to determine whether it indicates an image.       *
//*    c) Index start of image data.                                           *
//*    d) Locate image-type identifier embedded within the image data.         *
//*       (not currently implemented).                                         *
//*    e) Create a node on the ePic list describing the image.                 *
//*    f) Write the image data to a temporary file.                            *
//*                                                                            *
//* 2) If not image data:                                                      *
//*    a) Copy an informational message into the 'descValue' field.            *
//*    b) Discard the binary data. (see notes below)                           *
//*                                                                            *
//*                                                                            *
//* Input  : si       : index of source file with focus                        *
//*          ibuff    : caller's input buffer containing any header data and   *
//*                     all of the descriptor's value data.                    *
//*          asfMeta  : (by reference) data already captured for descriptor    *
//*                                    and tools for decoding data             *
//*                     'exDesc' member :                                      *
//*                        descName  : descriptor name                         *
//*                        descValue : receives MIME type (for debugging only) *
//*                        valueLen  : number of bytes in 'ibuff'              *
//*          descIndx : index into asfMeta::exDesc array                       *
//*          dbg      : handle to open output file to hold debugging data      *
//*                                                                            *
//* Returns: 'true'  if image found and scanned                                *
//*          'false' if not image data                                         *
//*                  (informational message written to 'descValue')            *
//******************************************************************************
//* Notes:                                                                     *
//* ------                                                                     *
//* The structure for inserting an image into the ASF container is not well    *
//* documented. For this reason, we are forced to reverse engineer the         *
//* process from the sample data we have obtained.                             *
//*                                                                            *
//* The examples have 29 bytes of data following the 'dataType' field and      *
//* before the actual image data. These 29 bytes are consistent within our     *
//* test data. Because all example files use JPEG-encoded images, we do not    *
//* know if the preamble to GIF or Bitmap images is different.                 *
//*                                                                            *
//* a) offset 00: could be the code for image format: 03==JPEG                 *
//*               03 is also the id3v2 code for "cover (front)"                *
//* b) offset 01: 16-bit size of remaining data (29 bytes less than 'valueLen')*
//* c) offset 03: 16-bit reserved value (always 0x0000)                        *
//* d) offset 05: The remaining 24 bytes consist of a UTF-16LE(no BOM),        *
//*               null-terminated string that is the MIME type for the image.  *
//*               JPEG MIME type: "image/jpeg"                                 *
//*                                                                            *
//* This is followed by the image data, either JPEG or GIF, (or other format)  *
//* Note that this is based only example data. The specification seems too     *
//* quiet on the acceptable image formats.                                     *
//* Note that JPEG images begin with the sequence:                             *
//*    FF D8 FF E0 00 10 4A 46 49 46                                           *
//*                       J  F  I  F                                           *
//* Note that GIF images begin with the sequence:                              *
//*    47 49 46                                                                *
//*     G  I  F                                                                *
//* Note that BMP images begin with the sequence:                              *
//*    42 4D                                                                   *
//*     B  M                                                                   *
//*                                                                            *
//*   ----  ----  ----  ----  ----  ----  ----  ----  ----  ----  ----  ----   *
//*                                                                            *
//* Non-image binary descriptor:                                               *
//* ----------------------------                                               *
//* If the descriptor contains non-image binary data, there is no place to     *
//* display it, and no convenient way to save it for re-integration during     *
//* write to the target file.                                                  *
//*                                                                            *
//* The most common non-image binary descriptor is "MCDI". These data are      *
//* not actually binary, but UTF-16 encoded ASCII-hex numeric data.            *
//* This is presumably a code used by the Millennium Compact Disc Industries.  *
//* MCDI is a compact-disc manufacturer in Vancouver. The meaning of this      *
//* code is unclear; however, its presence or absence has no affect on a file  *
//* which is not actually on a CD.                                             *
//* Example:                                                                   *
//*  11+96+7053+982A+B337+D01C+E3A7+10258+11BA3+13674+14D9C+170D8+1CAE4+2319F+ *
//*  29974+311BB+37521+3AF79+3F7AF                                             *
//* At a guess, these are probably sector numbers.                             *
//*                                                                            *
//******************************************************************************

static bool asfReadBinaryDescriptor ( char* ibuff, asfConDesc& asfMeta, 
                                      short descIndx, ofstream& dbg )
{
   const char* NonImageBinary = "(non-image binary data)" ;

   gString  gs( asfMeta.exDesc[descIndx].descName ), // text formatting
            gstmp ;
   uint16_t imageBytes,          // actual size of image data
            reserved ;           // reserved word s/b 0x0000
   bool isImage = false ;        // return value

   //* Compare descriptor name with image-name options *
   for ( short i = ZERO ; i < inCOUNT ; ++i )
   {
      if ( (gs.find( imageNames[i] )) >= ZERO )
      { isImage = true ; break ; }
   }

   if ( isImage )
   {
      //* Extract identifier, byte count, reserved *
      //* value and MIME type (if any).            *
      imageBytes = asfMeta.decode16Bit ( &ibuff[1] ) ;
      reserved   = asfMeta.decode16Bit ( &ibuff[3] ) ;
      asfMeta.utf16Decode ( &ibuff[5], 24, gstmp ) ;


      if ( (imageBytes < asfMeta.exDesc[descIndx].valueSize) && 
           (reserved == ZERO) &&
           (((gstmp.find( "image" )) >= ZERO) ||
            ((gstmp.find( "jpeg" )) >= ZERO)  ||
            ((gstmp.find( "gif" )) >= ZERO))
         )
      {
         //* Append the image size (if available), and copy *
         //* the MIME type to 'descValue' field             *
         if ( imageBytes > ZERO )
         {
            gs.formatInt( imageBytes, 6, true ) ;
            gstmp.append( " (%S bytes)", gs.gstr() ) ;
         }
         gstmp.copy( asfMeta.exDesc[descIndx].descValue, gsDFLTBYTES ) ;
      }
      else
         isImage = false ;
   }

   //* Non-image binary data *
   if ( ! isImage )
   {
      gstmp = NonImageBinary ;
      gstmp.copy( asfMeta.exDesc[descIndx].descValue, gsDFLTBYTES ) ;
   }

   return isImage ;

}  //* End asfReadBinaryDescriptor() *

//*************************
//* asfDiscardDescriptor  *
//*************************
//******************************************************************************
//* Non-member Method                                                          *
//* -----------------                                                          *
//* Read specified number of bytes from the input stream. (data are not saved) *
//* Used to step over data from the input stream which will not be decoded.    *
//*                                                                            *
//* Input  : ibuff   : input buffer                                            *
//*          objSize : number of bytes to read                                 *
//*          ifs     : handle of open input stream                             *
//*                                                                            *
//* Returns: number of bytes actually read                                     *
//*          caller should compare this value with objSize                     *
//******************************************************************************

static uint32_t asfDiscardDescriptor ( char *ibuff, uint32_t objSize, ifstream& ifs )
{
   uint32_t readBlock,           // bytes to read each time through the loop
            bytesRead = ZERO ;   // total bytes read (return value)

   while ( objSize > ZERO )
   {
      if ( objSize <= KB64 )
      {
         readBlock = objSize ;
         objSize = ZERO ;
      }
      else
      {
         readBlock = KB64 ;
         objSize -= KB64 ;
      }
      ifs.read( ibuff, readBlock ) ;
      bytesRead += ifs.gcount() ;

      if ( ifs.gcount() < readBlock )  // unexpected end-of-file
         break ;
   }
   return bytesRead ;

}  //* End asfDiscardDescriptor() *


//******************************************************************************
//******               Definitions for PNG image files.                  *******
//******************************************************************************
//* Notes on PNG (Portable Network Graphics) format:                           *
//* ------------------------------------------------                           *
//*                                                                            *
//* The first eight bytes of a PNG file always contain the following values:   *
//*    (decimal)              137  80  78  71  13  10  26  10                  *
//*    (hexadecimal)           89  50  4e  47  0d  0a  1a  0a                  *
//*    (ASCII C notation)    \211   P   N   G  \r  \n \032 \n                  *
//*                                                                            *
//* - All integer values are stored MSB first i.e. B3 B2 B1 B0.                *
//* - Chunk length is limited to 31 bits: 0x7FFFFFFF                           *
//* - Chunk layout:                                                            *
//*   - 4 bytes chunk length                                                   *
//*   - 4-byte chunk type (name)                                               *
//*   - Between 0 and 2,147,483,647 bytes (31 bits) of chunk data.             *
//*     In practice, the image data are most often broken into chunks of 4Kb   *
//*     or 8Kb, although we have seen chunks approaching 64Kb. For transfer    *
//*     across the net, it would be idiotic to use chuncks larger than 64Kb,   *
//*     but it is allowed, so watch out.                                       *
//*   - 4-byte CRC value.                                                      *
//*     Note: The chunk-length value is not included in the CRC calculation.   *
//*                                                                            *
//* A "chunk" header contains:                                                 *
//*   Chunk length: 4 bytes (unsigned) [any length between the minimum  ]      *
//*                                    [(12 bytes) and maximum (31-bit) ]      *
//*                 Data contained in the chunk EXCLUDING:                     *
//*                  four(4) bytes for length,                                 *
//*                  four(4) bytes for chunk type (name), and                  *
//*                  four(4) bytes of CRC data.                                *
//*   Chunk type  : 4 ASCII text bytes                                         *
//*   Chunk data  : formatted for the specific chunk type                      *
//*   CRC         : 4-byte CRC (see notes on CRC algorithm below)              *
//*                 The chunk length _apparently_ does not include the four    *
//*                 bytes of the CRC.                                          *
//*                                                                            *
//* Chunks may appear in any order (with certain constraints) except for       *
//* "IHDR" and "IEND". Note that some decoders do not scan for text chunks     *
//* after the image data even though it is legal and often useful to place     *
//* large text chunks after the image data.                                    *
//*                                                                            *
//* Text "chunks" are identified by one of three(3) chunk IDs:                 *
//* tEXt : uncompressed ISO/IEC 8859-1 (Latin-1) character set                 *
//*        Note: All keywords must use the ISO 8859-1 character set (see below)*
//*        Note: It is likely that some encoders use ISO 8859-15 or            *
//*              Windows-1252 instead of the original 8859-1. These people     *
//*              will be disappointed in the results since characters and      *
//*              character order are different for these.                      *
//* zTXt : compressed ISO/IEC 8859-1 (Latin-1) character set                   *
//*        Note: Recommended for larger blocks of text.                        *
//* iTXt : uncompressed or compressed UTF-8 character set                      *
//*        Note: This is the modern way to store text data, but some platforms *
//*              and applications (e.g. Windoze) may not be able to process    *
//*              them, so they may be ignored.                                 *
//*                                                                            *
//* Registered text-chunk tags (spec v:1.2):                                   *
//* ----------------------------------------                                   *
//* Title            Short (one line) title or caption for image               *
//*                  One (short) line of ISO 8859-1 text.
//* Author           Name of image's creator                                   *
//* Description      Description of image (possibly long)                      *
//* Copyright        Copyright notice                                          *
//* CreationTime     Time of original image creation                           *
//*                  RFC 1123,section 5.2.14 is suggested, but not required    *
//* Software         Software used to create the image
//* Disclaimer       Legal disclaimer
//* Warning          Warning of nature of content                              *
//* Source           Device used to create the image                           *
//* Comment          Miscellaneous comment; conversion from GIF comment        *
//*                                                                            *
//*                                                                            *
//* A) Keywords:                                                               *
//*    a) length >= 1 && < 80 bytes                                            *
//*    b) ISO 8859-1 printable characters and spaces only.                     *
//*       0x20 through 0x7E, and 0xA1 through 0xFF only.                       *
//*       Control characters and the non-breaking space 0xA0 are not allowed.  *
//*    c) Leading/trailing spaces and consecutive spaces are forbidden.        *
//*    d) The keyword may not contain a null character.                        *
//*    e) Registered keywords are considered as case-sensitive.                *
//*    f) A keyword may be used multiple times.                                *
//*    g) Private (non-registered) keywords may be used, but will be ignored   *
//*       by applications which do not support them.                           *
//*       -- Guidelines for private keywords:                                  *
//* B) A null character separates the keyword and the text.                    *
//* C) Text data:                                                              *
//*    a) The text may not contain a null character.                           *
//*    b) Except for the 'Creation Time', the text message should be free-form.*
//*       Newlines are allowed, but other control characters are discouraged.  *
//*    c) There _should be_ no more that 79 characters per line of output,     *
//*       i.e. between newline characters, but this is not reliable.           *
//*    d) For tEXt and zTXt chunks, only ISO 8859-1 printable characters and   *
//*       spaces are allowed. 0x20 through 0x7E, and 0xA1 through 0xFF, and    *
//*       newline (0x0A) only. The non-breaking space 0xA0 is not allowed.     *
//*    e) Convert ISO 8859-1 characters in the range 0x7F-0xFF, 127-255 decimal*
//*       to UTF-8 before display to prevent possible security problems. Some  *
//*       systems (but probably not Linux) may interpret some characters in    *
//*       this range as control codes.                                         *
//*    f) Compressed text data was compressed using the "deflate" algorithn    *
//*       found in the 'zlib' library provided by libpng.org. The algorithm    *
//*       is complex, so an independent implementation would be both large and *
//*       prone to errors. Re-inflation of text data is NOT YET IMPLEMENTED.   *
//*       - It is _possible_ that we can extract the compressed text to a      *
//*         temporary file and call 'gunzip' to reinflate the data.            *
//*       - It is also possible to create a simple re-inflate algorithm using  *
//*         public information.                                                *
//*                                                                            *
//* Section 9.7. Text chunk processing                                         *
//* ----------------------------------                                         *
//* -- A nonempty keyword must be provided for each text chunk (iTXt, tEXt,    *
//*    or zTXt). The generic keyword "Comment" can be used if no better        *
//*    description of the text is available. If a user-supplied keyword is     *
//*    used, be sure to check that it meets the restrictions on keywords.      *
//* -- Text stored in tEXt or zTXt chunks is expected to use the Latin-1       *
//*    character set. Encoders should provide character code remapping if the  *
//*    local system's character set is not Latin-1. Encoders wishing to store  *
//*    characters not defined in Latin-1 should use the iTXt chunk.            *
//* -- Encoders should discourage the creation of single lines of text longer  *
//*    than 79 characters, in order to facilitate easy reading.                *
//* -- It is recommended that text items less than 1K (1024 bytes) in size     *
//*    should be output using uncompressed text chunks. In particular, it is   *
//*    recommended that the text associated with basic title and author        *
//*    keywords should always be output with uncompressed chunks. Lengthy      *
//*    disclaimers, on the other hand, are ideal candidates for compression.   *
//* -- Placing large text chunks after the image data (after IDAT) can speed   *
//*    up image display in some situations, since the decoder won't have to    *
//*    read over the text to get to the image data. But it is recommended that *
//*    small text chunks, such as the image title, appear before IDAT.         *
//*                                                                            *
//* 4.2.3.1. tEXt Textual data                                                 *
//* -- Textual information that the encoder wishes to record with the image    *
//*    can be stored in tEXt chunks. Each tEXt chunk contains a keyword        *
//*   (see above) and a text string, in the format:                            *
//*     Keyword:        1-79 bytes (character string)                          *
//*     Null separator: 1 byte                                                 *
//*     Text:           n bytes (character string)                             *
//* -- The keyword and text string are separated by a zero byte (null          *
//*    character). Neither the keyword nor the text string can contain a null  *
//*    character. Note that the text string is not null-terminated (the length *
//*    of the chunk is sufficient information to locate the ending). The text  *
//*    string can be of any length from zero bytes up to the maximum           *
//*    permissible chunk size less the length of the keyword and separator.    *
//* -- The text is interpreted according to the ISO/IEC 8859-1 (Latin-1)       *
//*    character set [ISO/IEC-8859-1]. The text string can contain any Latin-1 *
//*    character. Newlines in the text string should be represented by a       *
//*    single linefeed character (decimal 10); use of other control characters *
//*    in the text is discouraged.                                             *
//*                                                                            *
//* 4.2.3.2. zTXt Compressed textual data                                      *
//* -- The zTXt chunk contains textual data, just as tEXt does; however,       *
//*    zTXt takes advantage of compression. The zTXt and tEXt chunks are       *
//*    semantically equivalent, but zTXt is recommended for storing large      *
//*    blocks of text.                                                         *
//* -- A zTXt chunk contains:                                                  *
//*     Keyword:            1-79 bytes (character string)                      *
//*     Null separator:     1 byte                                             *
//*     Compression method: 1 byte                                             *
//*     Compressed text:    n bytes                                            *
//* -- The keyword and null separator are exactly the same as in the tEXt      *
//*    chunk. Note that the keyword is not compressed. The compression method  *
//*    byte identifies the compression method used in this zTXt chunk.         *
//*    The only value presently defined for it is 0 (deflate/inflate           *
//*    compression). The compression method byte is followed by a compressed   *
//*    datastream that makes up the remainder of the chunk. For compression    *
//*    method 0, this datastream adheres to the zlib datastream format         *
//*    (see Deflate/Inflate Compression). Decompression of this datastream     *
//*    yields Latin-1 text that is identical to the text that would be stored  *
//*    in an equivalent tEXt chunk.                                            *
//*                                                                            *
//* 4.2.3.3. iTXt International textual data                                   *
//* This chunk is semantically equivalent to the tEXt and zTXt chunks, but the *
//* textual data is in the UTF-8 encoding of the Unicode character set instead *
//* of Latin-1. This chunk contains:                                           *
//*   Keyword:             1-79 bytes (character string)                       *
//*   Null separator:      1 byte                                              *
//*   Compression flag:    1 byte                                              *
//*   Compression method:  1 byte                                              *
//*   Language tag:        0 or more bytes (character string)                  *
//*   Null separator:      1 byte                                              *
//*   Translated keyword:  0 or more bytes                                     *
//*   Null separator:      1 byte                                              *
//*   Text:                0 or more bytes                                     *
//* -- The keyword is described above.                                         *
//* -- The compression flag is 0 for uncompressed text, 1 for compressed text. *
//*    Only the text field may be compressed. The only value presently defined *
//*    for the compression method byte is 0, meaning zlib datastream with      *
//*    deflate compression. For uncompressed text, encoders should set the     *
//*    compression method to 0 and decoders should ignore it.                  *
//* -- The language tag [RFC-1766] indicates the human language used by the    *
//*    translated keyword and the text. Unlike the keyword, the language tag   *
//*    is case-insensitive. It is an ASCII [ISO-646] string consisting of      *
//*    hyphen-separated words of 1-8 letters each (for example: cn, en-uk,     *
//*    no-bok, x-klingon). If the first word is two letters long, it is an     *
//*    ISO language code [ISO-639]. If the language tag is empty, the language *
//*    is unspecified.                                                         *
//* -- The translated keyword and text both use the UTF-8 encoding of the      *
//*    Unicode character set [ISO/IEC-10646-1], and neither may contain a zero *
//*    byte (null character). The text, unlike the other strings, is not       *
//*    null-terminated; its length is implied by the chunk length.             *
//* -- Line breaks should not appear in the translated keyword. In the text,   *
//*    a newline should be represented by a single line feed character         *
//*    (decimal 10). The remaining control characters (1-9, 11-31, and 127-159)*
//*    are discouraged in both the translated keyword and the text. Note that  *
//*    in UTF-8 there is a difference between the characters 128-159 (which are*
//*    discouraged) and the bytes 128-159 (which are often necessary).         *
//*                                                                            *
//* Section 10.11. Text chunk processing                                       *
//* ------------------------------------                                       *
//* -- If practical, decoders should have a way to display to the user all     *
//*    text chunks found in the file. Even if the decoder does not recognize   *
//*    a particular text keyword, the user might be able to understand it.     *
//* -- Text in the tEXt and zTXt chunks is not supposed to contain any         *
//*    characters outside the ISO 8859-1 (Latin-1) character set (that is, no  *
//*    codes 0-31 or 127-159), except for the newline character (decimal 10).  *
//*    But decoders might encounter such characters anyway. Some of these      *
//*    characters can be safely displayed (e.g., TAB, FF, and CR, decimal 9,   *
//*    12, and 13, respectively), but others, especially the ESC character     *
//*    (decimal 27), could pose a security hazard because unexpected actions   *
//*    may be taken by display hardware or software. To prevent such hazards,  *
//*    decoders should not attempt to directly display any non-Latin-1         *
//*    characters (except for newline and perhaps TAB, FF, CR) encountered in  *
//*    a tEXt or zTXt chunk. Instead, ignore them or display them in a visible *
//*    notation such as "\nnn". See Security considerations.                   *
//* -- Even though encoders are supposed to represent newlines as LF, it is    *
//*    recommended that decoders not rely on this; it's best to recognize all  *
//*    the common newline combinations (CR, LF, and CR-LF) and display each as *
//*    a single newline. TAB can be expanded to the proper number of spaces    *
//*    needed to arrive at a column multiple of 8.                             *
//* -- Decoders running on systems with non-Latin-1 character set encoding     *
//*    should provide character code remapping so that Latin-1 characters are  *
//*    displayed correctly. Some systems may not provide all the characters    *
//*    defined in Latin-1. Mapping unavailable characters to a visible notation*
//*    such as "\nnn" is a good fallback. In particular, character codes       *
//*    127-255 should be displayed only if they are printable characters on    *
//*    the decoding system. Some systems may interpret such codes as control   *
//*    characters; for security, decoders running on such systems should not   *
//*    display such characters literally.                                      *
//* -- Decoders should be prepared to display text chunks that contain any     *
//*    number of printing characters between newline characters, even though   *
//*    encoders are encouraged to avoid creating lines in excess of 79         *
//*    characters.                                                             *
//*                                                                            *
//* Section 3.4. CRC algorithm                                                 *
//* --------------------------                                                 *
//* Chunk CRCs are calculated using standard CRC methods with pre and post     *
//* conditioning, as defined by ISO 3309 [ISO-3309] or ITU-T V.42 [ITU-T-V42]. *
//* The CRC polynomial employed is:                                            *
//*    x^32+x^26+x^23+x^22+x^16+x^12+x^11+x^10+x^8+x^7+x^5+x^4+x^2+x+1         *
//* The 32-bit CRC register is initialized to all 1's, and then the data from  *
//* each byte is processed from the least significant bit (1) to the most      *
//* significant bit (128). After all the data bytes are processed, the CRC     *
//* register is inverted (its ones complement is taken).                       *
//* This value is transmitted (stored in the file) MSB first. For the purpose  *
//* of separating into bytes and ordering, the least significant bit of the    *
//* 32-bit CRC is defined to be the coefficient of the x31 term.               *
//* Practical calculation of the CRC always employs a precalculated table to   *
//* greatly accelerate the computation. See Sample CRC Code.                   *
//*                                                                            *
//*                                                                            *
//*                                                                            *
//******************************************************************************

//***********************
//* PNG constant values *
//***********************
const uint32_t pngCHUNK_MAX = 0x7FFFFFFF ; // maximum chunk size
const uint32_t pngINBUFF    = 0x010000 ;   // size of input buffer (64Kbytes)
const short pngHDR_BYTES = 8 ;      // number of bytes in PNG file header
const short pngCHUNK_HDR = 8 ;      // chunk length and chunk type data bytes
const short pngCNAME_LEN = 5 ;      // chunk name length (4 chars + null terminator)
const short pngLEN_LEN   = 4 ;      // size of chunk-length integer (32 bits, 4 bytes)
const short pngKEY_LEN   = 80 ;     // maximum length of a keyword within a chunk
const short pngCRC_LEN   = 4 ;      // CRC is a 32-bit integer (4 bytes)
const short pngZHDR_LEN  = 1 ;      // zTXt header is one byte: compression type
const short pngNAMELEN = 3 ;        // length of ASCII sequence "PNG" in file header
const uint8_t pngNAME[pngNAMELEN+1] = "PNG" ; // ASCII string identifying PNG file
const uint8_t pngID     = 0x89 ;    // first byte of PNG file
const uint8_t pngCR     = 0x0D ;    // Carriage return
const uint8_t pngLF     = 0x0A ;    // linefeed
const uint8_t pngCTRL_Z = 0x1A ;    // Ctrl+Z

#define DEBUGpng (0)    // For debugging only, output intermediate results
#if DEBUGpng != 0
#define DEBUGpng_VERBOSE (0) // Verbose debugging output
#define DEBUGpng_CONVERT (0) // Test conversion ISO 8859-1 to UTF-8
#endif   // DEBUGpng

//* Defined text-chunk types *
const short pngCHUNKLEN = 5 ;     // Chunk names are 4 ASCII chars + '\0'
const char  tEXtChunk[pngCHUNKLEN] = "tEXt" ;   // uncompressed ISO 8859-1
const char  zTXtChunk[pngCHUNKLEN] = "zTXt" ;   // compressed ISO 8859-1
const char  iTXtChunk[pngCHUNKLEN] = "iTXt" ;   // UTF-8 (compression flag embeded in header)
const char  IEND_Chunk[pngCHUNKLEN] = "IEND" ;  // End-of-data chunk name

//* Interesting chunk types: Text chunks, end-of-data chunk,    *
//* generic non-text chunk and read-error code (unexpected EOF).*
enum chType : short { chtTEXT, chtZTXT, chtITXT, chtIEND, chtNONTEXT, chtEOF } ;



//***********************************
//* Classes for processing PNG data *
//***********************************

//*****************************
//* Top-level file validation *
//*****************************
class pngHeader
{
   public:
   ~pngHeader ( void ) {}        // destructor
   pngHeader ( void )            // default constructor
   {
      this->reset() ;
   }

   void reset ( void )           // initialize all data members
   {
      this->pngName[0] = this->pngName[1] = 
      this->pngName[2] = this->pngName[3] = NULLCHAR ;
      this->pngCode  = NULLCHAR ;
      this->pngValid = false ;
   }

   //* Normalize, validate and store the header information.    *
   //*  (hexadecimal)           89  50  4e  47  0d  0a  1a  0a  *
   //*  (ASCII)                      P   N   G  \r  \n  ^Z  \n  *
   //* The 0x89 is a non-ASCII value which along with the 'P'   *
   //* are indended to uniquely identify the file format as PNG.*
   //* The constant control codes must be present:              *
   //* The CR-LF sequence and the trailing LF code are a safety *
   //* measures for file transfers.                             *
   //* The Ctrl+Z is a hold-over from MS-DOS and is not used.   *
   //*                                                               *
   //* Input  : ifs : (by reference) an open file input stream       *
   //* Returns: 'true' if file is a valid PNG format, else 'false'   *
   bool validateHeader ( ifstream& ifs )
   {
      this->reset () ;                                // reset our data members
      ifs.read ( (char*)this->ibuff, pngHDR_BYTES ) ; // read the header
      if ( (ifs.gcount()) == pngHDR_BYTES )           // if a good read
      {
         bool valid = true ;        // hope for the best

         for ( short i = ZERO ; (i < pngHDR_BYTES) && (valid != false) ; ++i )
         {
            switch ( i )
            {
               case 0:
                  if ( this->ibuff[i] != pngID ) { valid = false ; }  break ;
               case 1:
                  if ( this->ibuff[i] != pngNAME[0] ) { valid = false ; } break ;
               case 2:
                  if ( this->ibuff[i] != pngNAME[1] ) { valid = false ; } break ;
               case 3:
                  if ( this->ibuff[i] != pngNAME[2] ) { valid = false ; } break ;
               case 4:
                  if ( this->ibuff[i] != pngCR ) { valid = false ; } break ;
               case 5:
               case 7:
                  if ( this->ibuff[i] != pngLF ) { valid = false ; } break ;
               case 6:
                  if ( this->ibuff[i] != pngCTRL_Z ) { valid = false ; } break ;
            } ;
         }  // for(;;)

         if ( valid != false )
         {
            this->pngCode    = this->ibuff[0] ;
            this->pngName[0] = this->ibuff[1] ;
            this->pngName[1] = this->ibuff[2] ;
            this->pngName[2] = this->ibuff[3] ;
            this->pngName[3] = NULLCHAR ;
            this->pngValid = true ;
         }
      }
      return this->pngValid ;

   }  //* End validateHeader() *

   //* Public data members *
   uint8_t ibuff[pngHDR_BYTES + 2] ; // input buffer
   uint8_t pngName[pngNAMELEN+1] ;   // ASCII name i.e. "PNG"
   int8_t  pngCode ;                 // First byte of file (s/b 0x89) indicating
                                     // that file is a PNG file
   bool    pngValid ;                // 'true' if file header has been validated

} ;   // pngHeader

//********************************************
//* Chunk validation and processing.         *
//* Includes text chunk decoding and storage.*
//********************************************
class pngChunk
{
   public:
   ~pngChunk ( void )            // destructor
   {
      if ( this->ibuff != NULL ) // release the dynamic allocation
      {
         delete [] this->ibuff ;
         this->ibuff = NULL ;
      }
   }
   pngChunk ( void )             // default constructor
   {
      this->ibuff = NULL ;       // be sure pointer is initialized
      this->reset() ;            // initialize all data members
      this->ibuff = new uint8_t[pngINBUFF] ; // dynamic allocation of input buffer
   }

   //* Set all data members to initial values *
   void reset ( void )
   {
      if ( this->ibuff != NULL )
         *this->ibuff = NULLCHAR ;
      *this->cText     = NULLCHAR ;
      *this->cKeyword  = NULLCHAR ;
      *this->cTransKw  = NULLCHAR ;
      *this->cLanguage = NULLCHAR ;
      *this->cName     = NULLCHAR ;
      this->cLength    = ZERO ;
      this->txtLength  = ZERO ;
      this->kwLength   = ZERO ;
      this->hdrLength  = ZERO ;
      this->cType      = chtNONTEXT ;
      this->compressed = false ;
   }

   //* Read a chunk header and determine whether it is a text chunk. *
   //* The first four(4) bytes of a chunk are the chunk length.      *
   //*  - Capture the chunk length as a 32-bit integer value.        *
   //*    This value is stored in the 'cLength' member.              *
   //* The next four(4) bytes should be the chunk type (chunk name). *
   //*  - Compare the specified chunk type with the established      *
   //*    types for chunks containing text. (see enum chType)        *
   //*    This string is stored in the 'cName' member.               *
   //*                                                               *
   //* Input  : ifs : (by reference) an open file input stream       *
   //*          dnt : if 'true',  read and Discard Non-Text chunk    *
   //*                if 'false', perform no additional processing   *
   //*                            on non-text chunks                 *
   //*          ofs : For debugging only, access to an open output   *
   //*                stream, 'ofs' which is used to report          *
   //*                intermediate results of the scan.              *
   //*                                                               *
   //* Returns: member of enum chType                                *
   //*          If a text chunk, return chtTEXT, chtZTXT or chtITXT. *
   //*          If good read of non-text chunk, return chtNONTEXT.   *
   //*          If error reading source file, return chtEOF.         *
   chType readChunkHeader ( ifstream& ifs, bool dnt, ofstream& ofs )
   {
      gString gs ;            // text formatting
      uint32_t textBytes,     // number of text bytes to read from input stream
               cbCap ;        // bytes of compressed text actually captured
      this->reset () ;        // reset our data members

      ifs.read ( (char*)this->ibuff, pngCHUNK_HDR ) ;
      if ( (ifs.gcount()) == pngCHUNK_HDR )  // if a good read
      {
         if ( (this->cLength = this->intConv ( this->ibuff )) <= pngCHUNK_MAX )
         {
            //* Decode the chunk name (chunk type) and      *
            //* store the chunk name in the 'cName' member. *
            //* The type code returned indicates whether    *
            //* this is a chunk which contains text.        *
            this->cType = this->isTextChunk ( &this->ibuff[pngLEN_LEN] ) ;

            #if DEBUGpng != (0)
            gs.compose ( "Chunk Name: \"%s\" %u bytes", this->cName, &this->cLength ) ;
            ofs << gs.ustr() << endl ;
            #endif   // DEBUGpng

            //* If one of the text-chunk type identified,   *
            //* decode the keyword and text data.           *
            if ( (this->cType == chtZTXT) || (this->cType == chtITXT) ||
                 (this->cType == chtTEXT) )
            {
               //* Read the keyword (characters through first null character) *
               //* If successful, keyword will be stored in 'cKeyword' and    *
               //* 'kwLength' (characters, NOT bytes) will be initialized.    *
               if ( (this->scanKeyword ( ifs )) )
               {
                  #if DEBUGpng != 0 && DEBUGpng_VERBOSE
                  ofs << " Keyword: \"" << this->cKeyword 
                      << "\" kwLength: " << this->kwLength << endl ;
                  #endif   // DEBUGpng && DEBUGpng_VERBOSE

                  //* Compressed 8859-1 text chunk *
                  if ( this->cType == chtZTXT )
                  {  //* There is currently only one compression method *
                     //* defined, so we just read and discard it.       *
                     ifs.read ( (char*)this->ibuff, pngZHDR_LEN ) ;
                     this->hdrLength = pngZHDR_LEN ;
                     this->compressed = true ;
                     textBytes = (this->cLength - this->kwLength 
                                                - this->hdrLength) ;

                     #if DEBUGpng != 0 && DEBUGpng_VERBOSE != ZERO
                     cbCap = this->decompressText ( ifs, textBytes, false, ofs ) ;
                     #else    // PRODUCTION
                     cbCap = this->decompressText ( ifs, textBytes, false ) ;
                     #endif   // DEBUGpng && DEBUGpng_VERBOSE

                     //* Read and discard the trailing CRC value *
                     this->scanInteger ( ifs ) ;

                     #if DEBUGpng != 0 && DEBUGpng_VERBOSE
                     gs.compose( " zTXt Header (%u byte)\n"
                                 "  compress : %hhu\n"
                                 "  zTXtBytes: %u\n",
                                 &this->hdrLength, &this->compressed, 
                                 &this->txtLength ) ;
                     if ( cbCap != textBytes )
                        gs.append( "  cbCap    : %u\n", &cbCap ) ;
                     gs.append( "  zTxtText : \"%s\"", this->cText ) ;
                     ofs << gs.ustr() << endl ;
                     #else
                     if ( cbCap > ZERO ) ;      // silence the compiler warning
                     #endif   // DEBUGpng && DEBUGpng_VERBOSE
                  }

                  //* UTF-8 text chunk *
                  else if ( this->cType == chtITXT )
                  {  //* Get the header information and *
                     //* store it in our data members.  *
                     this->scanUtfHeader ( ifs ) ;

                     textBytes = this->cLength - this->kwLength 
                                                - this->hdrLength ;
                     //* Read the compressed text data, re-inflate it *
                     //* and save it to the 'cText' member.           *
                     #if DEBUGpng != 0 && DEBUGpng_VERBOSE
                     if ( this->compressed )
                        cbCap = this->decompressText ( ifs, textBytes, true, ofs ) ;
                     else
                        cbCap = this->scanUtfData ( ifs, textBytes, ofs ) ;
                     #else    // PRODUCTION
                     if ( this->compressed )
                        cbCap = this->decompressText ( ifs, textBytes, true ) ;
                     else
                        cbCap = this->scanUtfData ( ifs, textBytes ) ;
                     #endif   // DEBUGpng && DEBUGpng_VERBOSE

                     //* Read and discard the trailing CRC value *
                     this->scanInteger ( ifs ) ;

                     #if DEBUGpng != 0 && DEBUGpng_VERBOSE
                     gs.compose( " iTXt Header (%u bytes):\n"
                                 "  Language : \"%s\"\n"
                                 "  Trans Kw : \"%s\"\n"
                                 "  compress : %hhd\n"
                                 "  iTXtBytes: %u\n",
                                 &this->hdrLength, this->cLanguage, this->cTransKw, 
                                 &this->compressed, &this->txtLength ) ;
                     if ( cbCap != textBytes )
                        gs.append( "  cbCap    : %u\n", &cbCap ) ;
                     gs.append( "  iTxtText : \"%s\"", this->cText ) ;
                     ofs << gs.ustr() << endl ;
                     #endif   // DEBUGpng && DEBUGpng_VERBOSE
                  }

                  //* Uncompressed 8859-1 text chunk has no additional *
                  //* header info. Convert the 8859-1 data to UTF-8.   *
                  else if ( this->cType == chtTEXT )
                  {
                     textBytes = this->cLength - this->kwLength ;
                     ifs.read ( (char*)this->ibuff, textBytes ) ;
                     if ( (ifs.gcount()) == textBytes )
                     {
                        this->ibuff[textBytes] = NULLCHAR ;
                        convert8859_to_UTF8 ( this->ibuff, gs ) ;
                        gs.copy( this->cText, gsDFLTBYTES ) ;
                        this->scanInteger ( ifs ) ;  // discard the CRC

                        #if DEBUGpng != 0
                        ofs << " tTXt bytes:" << textBytes 
                            << "\n tTXt text: \"" << gs.ustr() << "\"" << endl ;
                        #endif   // DEBUGpng
                     }
                     else     // unexpected EOF
                        ;
                  }
               }
               else     // unexpected EOF or keyword invalid (too long)
               {
                  this->reset () ;
                  this->cType = chtEOF ;
               }
            }

            //* Else, chunk does not contain text data. *
            //* If specified, discard non-text chunk.   *
            else if ( dnt != false )
            {
               #if DEBUGpng != 0 && DEBUGpng_VERBOSE != 0
               this->discardBytes ( ifs, (this->cLength + pngCRC_LEN), ofs ) ;
               #else    // PRODUCTION
               this->discardBytes ( ifs, (this->cLength + pngCRC_LEN) ) ;
               #endif   // DEBUGpng && DEBUGpng_VERBOSE
            }
         }
         else     // chunk size is out-of-range (bad file format)
         { this->reset () ; this->cType = chtEOF ; }
      }
      else        // unexpected EOF
      { this->reset () ; this->cType = chtEOF ; }

      return this->cType ;

   }  // readChunkHeader()

   //* Compare the chunk name with the defined text-chunk types.     *
   //* If not a text chunk, compare with the end-of-data chunk       *
   //* name "IEND".                                                  *
   //*                                                               *
   //* Input  : srcbuf : pointer to raw input data                   *
   //*                                                               *
   //* Returns: member of enum chType                                *
   chType isTextChunk ( uint8_t* srcbuf )
   {
      gString gs ;                        // text formatting
      chType chunkType = chtNONTEXT ;     // return value (assume non-text chunk)

      //* Capture four raw characters, convert to UTF-8 *
      //* and store the chunk name in 'cName' member.   *
      srcbuf[pngCNAME_LEN - 1] = NULLCHAR ;        // terminate the sequence
      this->convert8859_to_UTF8 ( srcbuf, gs ) ;   // convert the check
      gs.copy( this->cName, pngCNAME_LEN ) ;       // save the chunk name

      if ( (gs.compare( tEXtChunk )) == ZERO )
         chunkType = chtTEXT ;
      else if ( (gs.compare( zTXtChunk )) == ZERO )
         chunkType = chtZTXT ;
      else if ( (gs.compare( iTXtChunk )) == ZERO )
         chunkType = chtITXT ;
      else if ( (gs.compare( IEND_Chunk )) == ZERO )
         chunkType = chtIEND ;

      return chunkType ;

   }  //* End pngIsTextChunk() *

   //* Read the next 1-80 characters from the input stream.           *
   //* A null character should end the keyword. If the maximum length *
   //* is reached without encountering a null character, then the     *
   //* file format is corrupted. (or there's a bug in our code :)     *
   //*                                                                *
   //* Input  : ifs : (by reference) an open file input stream        *
   //*                                                                *
   //* Returns: 'true' if successful                                  *
   //*             cKeyword and kwLength will be initialized          *
   //*          'false' if unexpected EOF or keyword too long         *
   bool scanKeyword ( ifstream& ifs )
   {
      bool status = false ;

      *this->cKeyword = NULLCHAR ;  // initialize keyword-group members
      this->kwLength  = ZERO ;

      for ( short i = ZERO ; i < pngKEY_LEN ; ++i )
      {
         ifs.read ( (char*)&this->ibuff[i], 1 ) ;
         if ( (ifs.gcount()) == 1 )
         {
            if ( this->ibuff[i] == '\0' )    // end of keyword found
            {
               if ( i >= 1 ) // keyword must contain at least one non-null character
               {
                  //* Convert 8859-1 text to UTF-8 text *
                  gString gs ;
                  this->convert8859_to_UTF8 ( this->ibuff, gs ) ;
                  gs.copy( this->cKeyword, gsDFLTBYTES ) ;
                  this->kwLength = gs.gschars() ;
                  if ( (gs.gschars()) > 1 )
                     status = true ;
               }
               break ;
            }
         }
         else     // read error
            break ;
      }

      return status ;

   }  //* End scanKeyword() *

   //* Decode the header information for a UTF-8 text chunk.   *
   //* Byte                                                    *
   //*   0  compression flag: 0==uncompressed, 1==compressed   *
   //*   1  compression type: always 0                         *
   //*   2  optional language code string ISO 646 code, hyphen *
   //*      separated words of 1-8 chars each. (e.g. en-uk).   *
   //*      If first word is two(2) chars, it is an ISO639 code*
   //*  --  Language code if followed by mandatory '\0'.       *
   //*  --  optional translated keyword (UTF-8)                *
   //*  --  Translated keyword is followed by mandatory '\0'.  *
   //*                                                         *
   //* Input  : ifs : (by reference) an open file input stream *
   //*                                                         *
   //* Returns: 'true' if successful                           *
   //*             'cLanguage', 'hdrLength' and 'compressed'   *
   //*             members will be initialized                 *
   //*          'false' if unexpected EOF or corrupted stream  *
   bool scanUtfHeader ( ifstream& ifs )
   {
      gString gsTrg ;            // receives decoded sub-strings
      short nullChars = ZERO ;
      bool  status = false ;

      this->hdrLength = ZERO ;   // initialize header-group members
      *this->cLanguage = NULLCHAR ;
      this->compressed = false ;

      //* Read compression flag and compression type. *
      //* Store the compression flag.                 *
      ifs.read ( (char*)this->ibuff, 2 ) ;
      if ( (ifs.gcount()) == 2 )
      {
         this->hdrLength += 2 ;
         this->compressed = this->ibuff[0] == 0x00 ? false : true ;

         //* Scan for a language code *
         uint32_t maxBytes = this->cLength - this->kwLength ;
         for ( uint32_t i = ZERO ; i < maxBytes ; ++i )
         {
            ifs.read ( (char*)&this->ibuff[i], 1 ) ;
            if ( (ifs.gcount()) == 1 )
            {
               if ( this->ibuff[i] == NULLCHAR )   // end of language sub-string
               {
                  ++nullChars ;
                  this->convert8859_to_UTF8 ( this->ibuff, gsTrg ) ;
                  gsTrg.copy( this->cLanguage, gsDFLTBYTES ) ;
                  this->hdrLength += gsTrg.gschars() ;
                  break ;
               }
            }
            else        // unexpected EOF
               break ;
         }

         //* If language sub-string was captured successfully, *
         //* scan for keyword translation sub-string.          *
         if ( nullChars == 1 )
         {
            uint32_t maxBytes = this->cLength - this->kwLength - (gsTrg.gschars()) ;
            for ( uint32_t i = ZERO ; i < maxBytes ; ++i )
            {
               ifs.read ( (char*)&this->ibuff[i], 1 ) ;
               if ( (ifs.gcount()) == 1 )
               {
                  if ( this->ibuff[i] == NULLCHAR )   // end of translation sub-string
                  {
                     ++nullChars ;
                     this->convert8859_to_UTF8 ( this->ibuff, gsTrg ) ;
                     gsTrg.copy( this->cTransKw, gsDFLTBYTES ) ;
                     this->hdrLength += gsTrg.gschars() ;
                     break ;
                  }
               }
               else        // unexpected EOF
                  break ;
            }
         }

         //* If valid header format, declare success *
         if ( nullChars == 2 )
            status = true ;
      }
      else        // unexpected EOF
         ;

      return status ;

   }  //* End scanUtfHeader() *

   //* Read the text of an iTXt chunk. Because the data are not      *
   //* null terminated, caller must specify the number of bytes      *
   //* to read.                                                      *
   //* The captured data are stored in the 'cText' member.           *
   //*                                                               *
   //* Important Note: Although the number of bytes to input can     *
   //* approach 2^31, we retain only the first gsDFLTBYTES           *
   //* (4096 bytes) of data and discard the remainder of the         *
   //* input data. While it is unlikely that the source data are     *
   //* larger than the target buffer, if this happens, then byte     *
   //* count returned to caller will not match the specified         *
   //* source bytes to be read, indicating that some data were       *
   //* discarded. (We may want to revisit this design.)              *
   //*                                                               *
   //* Input  : ifs     : (by reference) an open file input stream   *
   //*          srcBytes: number of bytes to read from the input     *
   //*                    stream                                     *
   //*                                                               *
   //* Returns: total number of bytes CAPTURED from the input stream *
   //*          (see note above about potential discarded data)      *
   //*                                                               *
   //* On return, the 'cText' member will contain the captured text  *
   //* and the 'txtLength' member will be the number of UTF-8        *
   //* characters (NOT THE NUMBER OF BYTES) stored in 'cText'.       *
   #if DEBUGpng != 0 && DEBUGpng_VERBOSE != 0
   uint32_t scanUtfData ( ifstream& ifs, uint32_t srcBytes, ofstream& ofs )
   #else    // PRODUCTION
   uint32_t scanUtfData ( ifstream& ifs, uint32_t srcBytes )
   #endif   // DEBUGpng && DEBUGpng_VERBOSE
   {
      // Programmer's Note: We do some defensive programming here to prevent a
      // partial UTF-8 character from being truncated. While the UTF-8 specification 
      // states that a character may require up to six(6) bytes, in practice 
      // no character requires more than four(4) bytes.
      const short maxIN_BYTES = (gsDFLTBYTES - 4) ;
      gString gs ;                     // text formatting
      uint32_t overflowBytes = ZERO,   // bytes beyond buffer size
               bytesCaptured = ZERO ;  // return value

      *this->cText = NULLCHAR ;        // initialize the target data members
      this->txtLength = ZERO ;

      //* Test for potential overflow. *
      if ( srcBytes > maxIN_BYTES )
      {
         overflowBytes = srcBytes - maxIN_BYTES ;
         srcBytes = maxIN_BYTES ;
      }

      ifs.read ( (char*)this->ibuff, srcBytes ) ;
      if ( (bytesCaptured = (ifs.gcount())) == srcBytes )
      {
         this->ibuff[bytesCaptured] = NULLCHAR ;   // terminate the string
         gs = (char*)this->ibuff ;                 // store the captured text
         gs.copy( this->cText, gsDFLTBYTES ) ;
         this->txtLength = gs.gschars() ;          // number of characters stored

         //* Discard the un-captured text data. *
         // Programmer's Note: If a read error, the caller will not be notified.
         if ( overflowBytes > ZERO )
         {
            #if DEBUGpng != 0 && DEBUGpng_VERBOSE != 0
            this->discardBytes ( ifs, overflowBytes, ofs ) ;
            #else    // PRODUCTION
            this->discardBytes ( ifs, overflowBytes ) ;
            #endif   // DEBUGpng && DEBUGpng_VERBOSE
         }
      }
      else     // unexpected EOF - empty string in 'cText'
         ;

      return bytesCaptured ;

   }  //* End scanUtfData() *

   //* Convert the ISO 8859-1 text data to UTF-8 data.              *
   //* Programmer's Note: This algorithm has been manually verified *
   //* with static 8859-1 character data and visual inspection.     *
   //* See the 'test8859_Conversion() method, below.                *
   //*                                                              *
   //* Input  : srcbuf : null-terminated 8859-1 data                *
   //*          gsTrg  : (by reference) receives UTF-8 data         *
   //*                                                              *
   //* Returns: number of UTF-8 characters converted                *
   short convert8859_to_UTF8 ( const uint8_t* srcbuf, gString& gsTrg )
   {
      const uint8_t minASCII  = 0x20,     // min printing ASCII (' ')
                    maxASCII  = 0x7E,     // max printing ASCII ('~')
                    minLATIN  = 0xA1,     // high-range Latin minimum ('¡')
                    maxLATIN  = 0xFF,     // high-range Latin maximum ('ÿ')
                    latinLF   = 0x0A ;    // linefeed code (newline character)
      const wchar_t fillCHAR  = 0x5F ;    // '_' replaces invalid chars in stream
      //* The 8859-1 codes between 0x80 and 0x9F are invalid in this context.  *
      //* The non-breaking space (0xA0) is not allowed in this context.        *
      //* The ASCII control codes (except newline, 0x0A) and ASCII delete      *
      //* (0x7F) are also invalid in this context. These will be replaced by   *
      //* the 'fillCHAR' character.                                            *

      wchar_t wtmp ;       // conversion character

      gsTrg.clear() ;      // initialize caller's buffer

      for ( short i = ZERO ; i < pngKEY_LEN ; ++i )
      {
         //* Within lower block of character range? *
         if ( ((srcbuf[i] >= minASCII) && (srcbuf[i] <= maxASCII)) 
              || (srcbuf[i] == latinLF) )
         { wtmp = (wchar_t)srcbuf[i] & 0x007F ; }

         //* Within higher block of character range? *
         else if ( (srcbuf[i] >= minLATIN) && (srcbuf[i] <= maxLATIN) )
         { wtmp = (((wchar_t)srcbuf[i]) & 0x007F) + 0x0080 ; }

         //* End of keyword found *
         else if ( srcbuf[i] == '\0')
            break ;

         // character code is out-of-range, substitute the fill character
         else
            wtmp = fillCHAR ;

         //* Append the verified character to target *
         gsTrg.append( wtmp ) ;
      }
      return ( gsTrg.gschars() ) ;

   }  //* End convert8859_to_UTF8() *

   #if DEBUGpng != 0 && DEBUGpng_CONVERT != 0
   //* For debugging only: Test conversion of ISO 8859-1 data to UTF-8. *
   //* Create the full ISL 8859-1 character set, then convert the data  *
   //* in segments and write the converted data to the target file.     *
   //*                                                                  *
   //* Input  : ofs    : (by reference) an open output stream           *
   //*                                                                  *
   //* Returns: number of UTF-8 characters converted                    *
   short test8859_Conversion ( ostream& ofs )
   {
      uint8_t char8859[261] ;    // work buffer
      short cVal = 0x01,         // value to be written to array
            convIndex,           // index at which to begin conversion
            totalBytes = ZERO ;  // return value
      bool  dataOut = false ;    // if true, write the formatted output

      for ( short indx = ZERO ; (indx < 261) && (cVal < 0x0100) ; ++indx )
      {
         char8859[indx] = cVal++ ;     // save the ISO 8859-1 character code
         gString gs ;                  // text formatting

         //* Terminate and display the sub-group *
         if ( cVal == 0x20 )
         {
            char8859[++indx] = '\0' ;
            totalBytes += this->convert8859_to_UTF8 ( char8859, gs ) ;
            gs.insert( "0x01-0x1F: " ) ;
            dataOut = true ;
         }

         //* Terminate and display the sub-group *
         else if ( cVal == 0x5B )
         {
            char8859[++indx] = '\0' ;
            totalBytes += this->convert8859_to_UTF8 ( &char8859[convIndex], gs ) ;
            gs.insert( "0x20-0x5A: " ) ;
            dataOut = true ;
         }

         //* Terminate and display the sub-group *
         else if ( cVal == 0x80 )
         {
            char8859[++indx] = '\0' ;
            totalBytes += this->convert8859_to_UTF8 ( &char8859[convIndex], gs ) ;
            gs.insert( "0x5B-0x7F: " ) ;
            dataOut = true ;
         }

         //* Terminate and display the sub-group *
         else if ( cVal == 0xA0 )
         {
            char8859[++indx] = '\0' ;
            totalBytes += this->convert8859_to_UTF8 ( &char8859[convIndex], gs ) ;
            gs.insert( "0x80-0x9F: " ) ;
            dataOut = true ;
         }

         //* Terminate and display the sub-group *
         else if ( cVal == 0xD0 )
         {
            char8859[++indx] = '\0' ;
            totalBytes += this->convert8859_to_UTF8 ( &char8859[convIndex], gs ) ;
            gs.insert( "0xA0-0xCF: " ) ;
            dataOut = true ;
         }

         //* Full character set written.         *
         //* Terminate and display the sub-group,*
         //* then exit the loop.                 *
         else if ( cVal == 0x0100 )
         {
            char8859[++indx] = '\n' ;
            char8859[++indx] = '\0' ;
            totalBytes += this->convert8859_to_UTF8 ( &char8859[convIndex], gs ) ;
            gs.insert( "0xD0-0xFF: " ) ;
            dataOut = true ;
         }

         //* If we have a full output buffer, write to target.*
         if ( dataOut )
         {
            ofs << gs.ustr() << endl ;
            convIndex = indx + 1 ;
            dataOut = false ;
         }
      }
      return totalBytes ;

   }  //* End test8859_Conversion() *
   #endif   // DEBUGpng && DEBUGpng_CONVERT

   //* Read the specified bytes of compressed text data and re-inflate *
   //* the text. The uncompressed date will be stored in the 'cText'   *
   //* member, and the number of character (not byte) will be stored   *
   //* Input  : ifs     : (by reference) an open file input stream     *
   //*          srcBytes: number of source bytes to read from the      *
   //*                    input stream                                 *
   //*          utfSrc  : if 'true', source is compressed UTF-8 text   *
   //*                    if 'false', source is ISO 8859-1 text        *
   //*                                                                 *
   //* Returns: total number of bytes CAPTURED from the input stream   *
   //*          (see note above about potential discarded data)        *
   #if DEBUGpng != 0 && DEBUGpng_VERBOSE != ZERO
   uint32_t decompressText ( ifstream& ifs, uint32_t srcBytes, bool utfSrc, ofstream& ofs )
   #else    // PRODUCTION
   uint32_t decompressText ( ifstream& ifs, uint32_t srcBytes, bool utfSrc )
   #endif   // DEBUGpng && DEBUGpng_VERBOSE
   {
      uint32_t bytesCaptured = ZERO ;  // return value

#if 0    // CZONE - decompressText()
      if ( utfSrc )
      {
      }
      else
      {
      }
#else    // TEMP TEMP TEMP
#if DEBUGpng != 0 && DEBUGpng_VERBOSE != ZERO
this->discardBytes ( ifs, srcBytes, ofs ) ;
gString gs( "compressed %s data, not decoded.", (utfSrc ? "iTXt" : "zTXt") ) ;
#else
this->discardBytes ( ifs, srcBytes ) ;
gString gs( "(compressed data, not decoded)" ) ;
#endif   // DEBUGpng && DEBUGpng_VERBOSE
gs.copy( this->cText, gsDFLTBYTES ) ;
this->txtLength = gs.gschars () ;
#endif   // U/C

      return bytesCaptured ;

   }  //* End decompressText() *

   //* Read and discard the specified number of byte from the        *
   //* input stream.                                                 *
   //*                                                               *
   //* Input  : ifs     : (by reference) an open file input stream   *
   //*          byteCnt :                                            *
   //*          For debugging only, we have access to an open output *
   //*          stream, 'ofs' which is used to report intermediate   *
   //*          results of the scan.                                 *
   //*                                                               *
   //* Returns: number of bytes actually read and discarded          *
   #if DEBUGpng != 0 && DEBUGpng_VERBOSE != 0
   uint32_t discardBytes ( ifstream& ifs, uint32_t byteCnt, ofstream& ofs )
   #else    // PRODUCTION
   uint32_t discardBytes ( ifstream& ifs, uint32_t byteCnt )
   #endif   // DEBUGpng && DEBUGpng_VERBOSE
   {
      int32_t loop = byteCnt / pngINBUFF,
              remainder = byteCnt % pngINBUFF,
              bytesRead = ZERO, brTmp ;

      for ( int32_t i = ZERO ; i < loop ; ++i )
      {
         ifs.read( (char*)this->ibuff, pngINBUFF ) ;
         bytesRead += brTmp = (ifs.gcount()) ;
         if ( brTmp != pngINBUFF )
         { remainder = ZERO ; break ; }
      }
      if ( remainder > ZERO )
      {
         ifs.read( (char*)this->ibuff, remainder ) ;
         bytesRead += (ifs.gcount()) ;
      }

      #if DEBUGpng != 0 && DEBUGpng_VERBOSE != 0
      gString gs( "discardBytes: byteCnt:%u loop:%u remainder:%u bytesRead:%u", 
                  &byteCnt, &loop, &remainder, &bytesRead ) ;
      ofs << gs.ustr() << endl ;
      #endif   // DEBUGpng && DEBUGpng_VERBOSE

      return bytesRead ;

   }  //* End discardBytes

   //* Read four bytes (32 bits) from the input stream and     *
   //* convert it to a 32-bit integer value. Used primarily to *
   //* capture the CRC value, but valid for any integer.       *
   //*                                                         *
   //* Input  : ifs : (by reference) an open file input stream *
   //*                                                         *
   //* Returns: converted integer value                        *
   //*          if read error, returns zero                    *
   uint32_t scanInteger ( ifstream& ifs )
   {
      uint32_t iVal = ZERO ;

      ifs.read ( (char*)this->ibuff, pngCRC_LEN ) ;
      if ( (ifs.gcount()) == pngCRC_LEN )
         iVal = this->intConv ( this->ibuff ) ;

      return iVal ;

   }  //* End scanInteger() *

   //* Convert a 4-byte sequence (MSB at offset 0) to an integer value. *
   //* Programmer's Note: This clunky construct avoids the C library's  *
   //* "helpful" automatic sign extension.                              *
   uint32_t intConv ( const uint8_t* ucp )
   {
      uint32_t i =   (UINT)ucp[3] 
                   | ((UINT)ucp[2] << 8)
                   | ((UINT)ucp[1] << 16)
                   | ((UINT)ucp[0] << 24) ;
      return i ;
   }

   //* Public data members *
   uint8_t  *ibuff ;                // pointer to input buffer
   char     cText[gsDFLTBYTES] ;    // Normalized (UTF-8) text data
   char     cKeyword[gsDFLTBYTES] ; // keyword string (UTF-8 format)
   char     cTransKw[gsDFLTBYTES] ; // translated keyword string (UTF-8 format)
   char     cLanguage[gsDFLTBYTES] ;// language string (UTF-8 format)
   char     cName[pngCNAME_LEN+1] ; // chunk type (name) 4 ASCII chars, null-terminated
   uint32_t cLength ;               // length of chunk in bytes (does not include:
                                    // 4-byte length value, 4-byte chunk ID, 4-byte CRC)
   uint32_t txtLength ;             // characters (not bytes) in 'cText' incl.null terminator
   uint32_t kwLength ;              // characters (not bytes) in 'cKeyword' incl. null terminator
   uint32_t hdrLength ;             // number of bytes of header data following keyword
                                    // tEXt == 0, no header info
                                    // zTXt == 1, compression method code
                                    // iTXt == >= 4 bytes (see scanUtfHeader() for details)
   chType   cType ;                 // type of chunk most recently read
   bool     compressed ;            // true if text data compressed, else false

} ;   // pngChunk


//*************************
//*  ExtractMetadata_PNG  *
//*************************
//******************************************************************************
//* Read the media file and write the metadata to the temp file.               *
//*                                                                            *
//* Input  : ofs    : open output stream to temporary file                     *
//*          srcPath: filespec of media file to be read                        *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void FileDlg::ExtractMetadata_PNG ( ofstream& ofs, const gString& srcPath )
{
   char *ibuff = new char[KB64] ;// input buffer (64KiB dynamic allocation)
   gString gsOut, gstmp ;        // text formatting
   pngHeader pngHdr ;            // file header record
   uint32_t  chunkLen ;          // decoded chunk length in bytes
   pngChunk  pc ;                // chunk processing
   chType    cType ;             // chunk type
   bool autoDiscard = true ;     // read and discard non-text chunks

   //* Report basic information about the source file.*
   tnFName fn ;
   localTime lt ;
   this->fmPtr->GetFileStats ( fn, srcPath, true ) ;
   this->fmPtr->DecodeEpochTime ( fn.rawStats.st_mtime, lt ) ;
   gstmp.formatInt( fn.fBytes, 11, true ) ;
   short fni = srcPath.findlast( fSLASH ) + 1 ;
   gsOut.compose( "FILE NAME  : \"%s\"  (%s bytes)\n"
                  "FILE DATE  : %04hd-%02hd-%02hd %02hd:%02hd:%02hd\n",
                  &srcPath.ustr()[fni], gstmp.ustr(),
                  &lt.year, &lt.month, &lt.date, &lt.hours, &lt.minutes, &lt.seconds ) ;
   ofs << gsOut << endl ;

   #if DEBUGpng != 0 && DEBUGpng_CONVERT != 0
   pc.test8859_Conversion ( ofs ) ;
   #endif   // DEBUGpng && DEBUGpng_CONVERT

   //* Open the source file *
   ifstream ifs( srcPath.ustr(), ifstream::in ) ;
   if ( ifs.is_open() )
   {
      //* Read and validate the PNG file header *
      pngHdr.validateHeader ( ifs ) ;

      #if DEBUGpng != 0
      //* Display the file header information *
      gsOut.compose( "ID  : %02hhXh\n"
                     "NAME: \"%s\"\n"
                     "HDR : %s\n",
                     &pngHdr.pngCode, pngHdr.pngName,
                     (char*)(pngHdr.pngValid ? "valid" : "error")) ;
      ofs << gsOut.ustr() << endl ;
      #endif   // DEBUGpng

      //* If file header validated, scan for text chunks *
      if ( pngHdr.pngValid )
      {
         bool done = false ;
         do
         {
            //* Read a chunk header and determine whether *
            //* it is one of the text chunk types.        *
            //* If not, read and discard the chunk.       *
            //* Note: We pass the open ofstream to the    *
            //* pngChunk object; however, the object uses *
            //* it only if the 'DEBUGpng' flag is enabled.*
            cType = pc.readChunkHeader ( ifs, autoDiscard, ofs ) ;

            if ( (cType == chtTEXT) || (cType == chtZTXT) || (cType == chtITXT) )
            {
               if ( *pc.cTransKw == NULLCHAR )
                  gsOut.compose( "%s: %s", pc.cKeyword, pc.cText ) ;
               else       // if translated keyword is available
                  gsOut.compose( "%s (%s): %s", pc.cKeyword, pc.cTransKw, pc.cText ) ;
               ofs << gsOut.ustr() << endl ;
            }

            //* Not a text chunk. Read and discard it.*
            else if ( cType == chtNONTEXT )
            {
               #if 0    // For debugging only, report non-text chunk names
               ofs << pc.cName << endl ;
               #endif   // for debugging only
               if ( ! autoDiscard )
               {
                  chunkLen = pc.cLength + pngCRC_LEN ;
                  #if DEBUGpng != 0 && DEBUGpng_VERBOSE != 0
                  pc.discardBytes ( ifs, chunkLen, ofs ) ;
                  #else    // PRODUCTION
                  pc.discardBytes ( ifs, chunkLen ) ;
                  #endif   // DEBUGpng && DEBUGpng_VERBOSE
               }
            }

            //* If end-of-data marker found (chtIEND) or if    *
            //* unexpected end-of-file (chtEOF), exit the loop.*
            else
               done = true ;
         }
         while ( ! done ) ;
         ofs << endl ;        // clear the output buffer
      }  // (pngHdr.pngValid)
      #if DEBUGpng != 0
      else        // not a valid PNG file format
         ofs << "Error: Invalid PNG format.\n" << endl ;
      #endif   // DEBUGpng

      ifs.close() ;           // close the source file
   }

   delete [] ibuff ;             // release dynamic allocation
   ibuff = NULL ;

}  //* End ExtractMetadata_PNG() *

//******************************************************************************
//******               Definitions for JPG image files.                  *******
//******************************************************************************
//* Notes on JPG (Joint Photographic Experts Group) format:                    *
//* -------------------------------------------------------                    *
//* Editorial: Like many "standards" which are born in the wild, abandoned by  *
//* their mothers and are raised by apes, the JPEG/JFIF "standard" file        *
//* structure is poorly defined and seldom carefully followed by implementors. *
//* For this reason we are likely to see almost any crazy construct in random  *
//* files from the internet or from different digital camera manufacturers.    *
//* There is no way we can catch every bonkers construct. Sorry about that.    *
//*                   ----------------------------                             *
//*                                                                            *
//* The JPEG standard is codified under several ISO/IEC specifications.        *
//*     Part 5     ISO/IEC 10918-5:2013  JPEG File Interchange Format (JFIF)   *
//*                <https://www.iso.org/standard/54989.html>                   *
//*     Part 6     ISO/IEC 10918-6:2013  Application to printing systems       *
//*                <https://www.iso.org/standard/59634.html>                   *
//*     Part 7     ISO/IEC 10918-7:2019  Reference software for digital coding *
//*                <https://www.iso.org/standard/75845.html>                   *
//*                                                                            *
//* "JPEG or JPG is a commonly used method of lossy compression for digital    *
//* images, particularly for those images produced by digital photography.     *
//* The degree of compression can be adjusted, allowing a selectable trade-off *
//* between storage size and image quality. JPEG typically achieves 10:1       *
//* compression with little perceptible loss in image quality. Since its       *
//* introduction in 1992, JPEG has been the most widely used image compression *
//* standard in the world, and the most widely used digital image format, with *
//* several billion JPEG images produced every day as of 2015." -- Wikipedia   *
//*                                                                            *
//* Valid filename extensions: ".jpg" ".jpeg ".jpe" ".jfif" ".jif"             *
//*                                                                            *
//* The JFIF file format is structured as a byte stream using 16-bit,          *
//* big-endian values.                                                         *
//*                                                                            *
//* The file header for JPEG/JFIF files is shown. For JFIF-compliant files,    *
//* the APP0 marker follows the Start-Of-Image. Application-specific formats   *
//* are indicated by APP1, APP2, and so through APP15 (0xFFEF).                *
//* The most common application-specific marker is for "Exchange Image File    *
//* Format" (EXIF) used by most digital cameras, including mobile phones.      *
//* The layout of the first sixteen(16) bytes should be the same for all       *
//* application-specific headers.                                              *
//*                                                                            *
//* typedef struct _JFIFHeader                                                 *
//* {                    INDEX VALUE         DESCRIPTION                       *
//*   BYTE SOI[2];        00h  FF D8         Start of Image Marker             *
//*   BYTE APP0[2];       02h  FF E0         Application Use Marker            *
//*   BYTE Length[2];     04h  16+3*XTh*YTh  Length of APP0 Field              *
//*   BYTE Identifier[5]; 06h  JFIF\0        "JFIF" (zero-terminated) Id String*
//*   BYTE Version[2];    07h  01 02         JFIF Format Revision              *
//*   BYTE Units;         09h  00 | 01 | 02  Resolution Units (pix, dpi, dp-cm)*
//*   BYTE Xdensity[2];   0Ah  --            Horizontal Resolution             *
//*   BYTE Ydensity[2];   0Ch  --            Vertical Resolution               *
//*   BYTE XThumbnail;    0Eh  --            Horizontal Pixel Count            *
//*   BYTE YThumbnail;    0Fh  --            Vertical Pixel Count              *
//* } JFIFHEAD;                                                                *
//*                                                                            *
//* Following the JFIF Header, there may be one or more "JFIF extensions".     *
//* These extensions are used primarily as containers for thumbnail images.    *
//*                                                                            *
//* typedef struct _JFIFExtension                                              *
//* {                    INDEX VALUE         DESCRIPTION                       *
//* BYTE   APP0[2];       00h  FF E0         Application Use Marker            *
//* BYTE   Length[2];     02h  -- --         Length of APP0 Field              *
//* BYTE   Identifier[5]; 04h 4A 46 58 58 \0 "JFXX" (zero terminated) Id String*
//* BYTE   ExtensionCode; 09h Extension ID Code (thumbnail encoding)           *
//*                           10 == JPEG format, 11 == 1-byte-per-pixel,       *
//*                           13 == 3-bytes-per pixel (RGB)                    *
//* } JFIFExtension;                                                           *
//*                                                                            *
//*                                                                            *
//* The interesting tags are: <https://docs.fileformat.com/image/jpeg/>        *
//* ID    CONTENT  LENGTH   NAME, DESCRIPTION                                  *
//* ----- -------  -------  -------------------------------------------------- *
//* SOI   FF D8    0        Start Of Image                                     *
//* SOF0  FF C0    varies   Start Of Frame (baseline)                          *
//* SOF2  FF C2    varies   Start Of Frame (progressive)                       *
//* DHT   FF C4    varies   Define Huffman Tables                              *
//* DQT   FF DB    varies   Define Quantization Tables                         *
//* DNL   FF DC    4 bytes  Define number of lines                             *
//* DRI   FF DD    4 bytes  Define Restart Interval                            *
//* SOS   FF DA    varies   Start Of Scan                                      *
//* RSTn  FF Dn    0        Restart                                            *
//* APPn  FF En    varies   Application Specific:                              *
//*                         Example: APP0 == JFIF header  APP1 == EXIF header  *
//* COM   FF FE    varies   Comment                                            *
//* EOI   FF D9    0        End Of Image                                       *
//*                                                                            *
//* The "Comment" tag is 0xFFFE, followed by the 16-bit byte count, followed   *
//* by the null-terminated data.                                               *
//*                                                                            *
//* Note: If the 0xFF byte appears within the compressed data, it is followed  *
//*       by a 0x00 byte to prevent an apparent marker where none was intended.*
//* <https://en.wikibooks.org/wiki/JPEG_-_Idea_and_Practice/The_header_part>   *
//*                                                                            *
//* The Start-Of-Frame marker indicates the type of compression used.          *
//* The most common compression types are:                                     *
//* SOF0  FFC0              Baseline sequential                                *
//* SOF1  FFC1              Extended sequential                                *
//* SOF2  FFC2              Progressive                                        *
//* SOF3  FFC3              Lossless sequential                                *
//* Additional Start-Of-Frame markers are available:                           *
//* <https://www.disktuna.com/list-of-jpeg-markers/>                           *
//*    FFC5, FFC6, FFC7, FFC9, and so on, which indicate compression mode.     *
//* See notes in jpgThumb::decodeThumbnail() method for structure of the       *
//* Start-Of-Frame tag. <http://lad.dsc.ufcg.edu.br/multimidia/jpegmarker.pdf> *
//*                                                                            *
//* Library Of Congress, digital formats:                                      *
//* <https://www.loc.gov/preservation/digital/formats/fdd/fdd000018.shtml>     *
//*                                                                            *
//* EXIF (Exchange Image File Format)                                          *
//* ---------------------------------                                          *
//* The EXIF encapsulates the JFIF format for images as well as TIFF, and      *
//* various audio formats (WAV).                                               *
//* See: <https://www.media.mit.edu/pia/Research/deepview/exif.html>           *
//* <https://web.archive.org/web/20190624045241if_/                            *
//*       http://www.cipa.jp:80/std/documents/e/DC-008-Translation-2019-E.pdf> *
//* This is the internal format used by digital cameras, and provides specific *
//* information on the type of camera and its capabilities.                    *
//* This may be transparent to the end user because the file name uses the     *
//* ".jpg" filename extension; however, the header differs somewhat from the   *
//* JFIF standard:                                                             *
//* ff d8 ff e1 -- --  45 78 69 66 00 00 ...exif data...                       *
//* FF D8 FF D1 (size) E  X  i  f  \0 \0                                       *
//*                                                                            *
//* typedef struct exifExample                                                 *
//* {                    INDEX VALUE         DESCRIPTION                       *
//*   BYTE SOI[2];        00h  FF D8         Start of Image Marker             *
//*   BYTE APP1[2];       02h  FF E1         Application Use Marker            *
//*   BYTE Length[2];     04h  -- --         Length of APP1 Field              *
//*   BYTE Id[6];         06h  45 78 69 66 \0 \0  Id String "Exif\0\0"         *
//*   BYTE byteAlign[2];  0Ch  49 49 ("II") == Intel (little-endian data)      *
//*                            4D 4D {"MM") == Motorola (big-endian data)      *
//*   BYTE tiffConst[2];  0Eh  2A 00         Intel (little-endian value)       *
//*                            00 2A         Motorola (big-endian value)       *
//*   BYTE ifdOffset[4];  10h                offset to first ImageFileDirectory*
//*                            IFD _usually_ follows immediately after TIFF    *
//*                            header, so eight bytes:                         *
//*                            08 00 00 00   (little-endian value)             *
//*                            00 00 00 08   (big-endian value)                *
//*   ... Image File Directory sections (see below) ...                        *
//* }                                                                          *
//*                                                                            *
//* The FF E1 sequence is the EXIF APP1 marker, which is followed by the EXIF  *
//* data size and the constant string "EXif\0\0". This is followed by the      *
//* remainder of the EXIF data segment. The normal structures of the JPG/JFIF  *
//* file follow afterward.                                                     *
//* -- Note that the EXIF data-size value includes the size of the EXIF        *
//*    descriptor so take this into account when scanning past the data.       *
//* -- There are three(3) IFD arrays of interest:                              *
//*     1) TIFF IFD, Rev. 6.0 Attrribute Information (Table 4 of specification)*
//*     2) Exif IFD, Attribute Information (Tables 7 and 8 of specification)   *
//*     3) GPS IFD, Attribute Information (Table 15 of specification)          *
//*    The "decodeHeader()" method of the exifHeader class captures the raw    *
//*    data from these arrays. Not all data are useful for a non-technical     *
//*    user, so only certain record types are reported by the                  *
//*    ExtractMetadata_JPG() method, and the remainder are ignored.            *
//*    -- See the EXIF standard for the complete list of tag names.            *
//*       "Exchangeable Image File Format For Digital Still Cameras"           *
//*    -- Note: During development, we report additional record types using    *
//*       "verbose" output. The verbose output may be disabled for production. *
//*                                                                            *
//*    Exif Record Notes:                                                      *
//*    ------------------                                                      *
//*    -- Record types that are always reported if present are listed in the   *
//*       "exifTags" enumerated type.                                          *
//*    -- Record types that are reported only in "verbose" mode are listed     *
//*       in the "exifVTags" enumerated type.                                  *
//*    -- Records containing string data are specified to contain only ASCII   *
//*       data _except_ for 0x9286 "UserComment" which can contain multi-byte  *
//*       characters which we interpret to mean UTF-8, although UTF-16 data    *
//*       are possible. (After all, Microsloth stuck its nose in.)             *
//*    -- Tag 0x8298 Copyright info:                                           *
//*       Note: this field MAY contain two, null-terminated strings.           *
//*    -- "Rational" values are stored as integer fractions. Some are displayed*
//*       as fractions e.g. 1/200 while others are displayed as floating-point *
//*       data. Most fractions are unsigned; however, a few are signed         *
//*       fractions (as indicated in the display routine).                     *
//*    -- Tag 0x9209 "Flash" is a bit field with LSB indicating whether flash  *
//*       was used. Other bits describe flash control options (ignored).       *
//*    -- Tag 0x8769 is a pointer (offset) to the EXIF IFD record array        *
//*    -- Tag 0x8825 is a pointer (offset) to GPS IFD record array             *
//*    -- Some records use integer codes to indicate the value. These are      *
//*       listed below.                                                        *
//*                                                                            *
//* TAG   Record Name and defined values  TAG   Record Name and defined values *
//* ---   ------------------------------  ---   ------------------------------ *
//* 0112  Orientation (rotation)          9207  Metering mode                  *
//*       VALUE 0th Row 0th Col                   0   unknown                  *
//*       ----- ------- -------                   1   average                  *
//*         1   top     left                      2   center weighted average  *
//*         2   top     right                     3   spot                     *
//*         3   bottom  right                     4   multi-spot               *
//*         4   bottom  left                      5   pattern                  *
//*         5   left    top                       6   partial                  *
//*         6   right   top                       255 other                    *
//*         7   right   bottom            A403  White balance                  *
//*         8   left    bottom                    0   auto                     *
//* 0128  Resolution units                        1   manual                   *
//*         2   inches                    A408  Contrast                       *
//*         3   centimeters                       0   normal                   *
//* 8822  Exposure program                        1   soft                     *
//*         1   manual                            2   hard                     *
//*         2   normal                    A409  Saturation                     *
//*         3   aperture priority                 0   normal                   *
//*         4   shutter priority                  1   low saturation           *
//*         5   creative (prioritize              2   high saturation          *
//*                      depth of field)  A40A  Sharpness                      *
//*         6   action (prioritize speed)         0   normal                   *
//*         7   portrait mode                     1   soft                     *
//*         8   landscape mode                    2   hard                     *
//*                                                                            *
//*                                                                            *
//*    ----------------------------                                            *
//*    013E  White Point                                                       *
//*    013F  Primary Chromaticities                                            *
//*    0201  JPEG Interchange Format                                           *
//*    0202  JPEG Interchange Format Length                                    *
//*    0211  YCbCrCoefficients                                                 *
//*    0212  Subsampling ratio of Y to C                                       *
//*    0213  Y and C positioning                                               *
//*    0214  Reference Black/White                                             *
//*                                                                            *
//*  * 927C  Manfacturer Notes (string)                                        *
//*  * 9286  Exif Private tag UserComment (this can be used for UTF-8 data)    *
//*          See notes in the header of the ucEncoding() method, below.        *
//*    A005  Interoperability IFD                                              *
//*                                                                            *
//*                                                                            *
//* -- The last item in the EXIF area should be the thumbnail image            *
//*    (usually, but not necessarily a JPEG image) .                           *
//*    JPEG images are bracketed by the tags StartData (FFD8) ...data...       *
//*    EndData (FFD9). Because the FFD9 is inside the EXIF section, it will    *
//*    not be seen as the end-of-file.                                         *
//*    It is _possible_ that the thumbnail is not the last data in the header. *
//*    This needs to be tested.                                                *
//*                                                                            *
//*                                                                            *
//* Image File Directory (IFD) Records                                         *
//* ----------------------------------                                         *
//* Each record is twelve(12) bytes in length.                                 *
//* 00-01 Tag that identifies the field                                        *
//*       <https://exiftool.org/TagNames/EXIF.html>                            *
//* 02-03 Field-type code                                                      *
//*       1: uint8_t, 2: 7-bit ASCII string, 3:uint16_t, 4:uint32_t,           *
//*       5: unsigned fraction 32-bit numerator and 32-bit denominator         *
//*       7: one(1) 8-bit byte with meaning determined by the field definition *
//*       9: int32_t (2's complement notation)                                 *
//*      10: signed fraction 32-bit numerator and 32-bit denominator           *
//* 04-07 Number of items of the specified type in the data section            *
//* 08-11 File offset from the top of the TIFF header to the data section      *
//*       -- If the specified data type and number of values will fit into     *
//*          these 32 bits, the value itself is stored here, NOT the offset.   *
//*          At this time, we believe that only one 32-bit or 16-bit value may *
//*          be stored in the offset field. It is possible that two(2) 16-bit  *
//*          values may be stored in the offset field; however, we have not    *
//*          been able to find an example of this. Note that up to four(4)     *
//*          8-bit bytes may be stored in the offset field.                    *
//*       -- Within a TIFF file, this is the offset from top-of-file; however, *
//*          the start of the TIFF header in a JPEG file is offset four(4).    *
//*       -- The specified length of a string value includes the null          *
//*          terminator (if any).                                              *
//*                                                                            *
//* -----  -----  -----  -----  -----  -----  -----  -----  -----  -----  -----*
//* The "jhead" utility may be used to initialize test data, both for the EXIF *
//* block and for a subset of JPEG/JFIF file blocks.                           *
//* Add a comment tag:  jhead -cl "Comment" file.jpeg                          *
//*                     jhead -cs ?                                            *
//*                     jhead -ci ?                                            *
//* Note that comments are added to the tail of the header _after_ the         *
//* thumbnail image (if present).                                              *
//*                                                                            *
//* Jhead can also be used to rotate the image to right-side-up. This calls    *
//* the external "jpegtran" utility.
//* Jhead can also be used to adjust the EXIF timestamp or to copy the original*
//* EXIF section from another file.                                            *
//*                                                                            *
//******************************************************************************

//***********************
//* JPG constant values *
//***********************
const uint32_t jpgINBUFF = 0x010000 ;  // Size of input buffer (64Kbytes)
const uint16_t jpgSOI   = 0xFFD8 ;     // Start-Of-Image tag
const uint16_t jpgEOI   = 0xFFD9 ;     // End-Of-Image tag
const uint16_t jpgAPP0  = 0xFFE0 ;     // JFIF Application Use Marker
const uint16_t jpgAPP1  = 0xFFE1 ;     // EXIF Application Use Marker
const uint16_t jpgSOF0  = 0xFFC0 ;     // JPEG Start-Of-Frame (baseline)
const uint16_t jpgSOF2  = 0xFFC2 ;     // JPEG Start-Of-Frame (progressive)
const uint16_t jpgDHT   = 0xFFC4 ;     // JPEG Define Huffman Tables
const uint16_t jpgDQT   = 0xFFDB ;     // JPEG Define Quantization Tables
const uint16_t jpgDRI   = 0xFFDD ;     // JPEG Define Restart Interval
const uint16_t jpgSOS   = 0xFFDA ;     // JPEG Start-Of-Scan tag
const uint16_t jpgCOM   = 0xFFFE ;     // JPEG Comment tag
const short jpgINT16    = 2 ;          // Number of bytes in a short integer
const short jpgINT32    = 4 ;          // Number of bytes in a 32-bit integer
const short jpgIFDMAX   = 96 ;         // Records in IFD record array in exifHeader class
const short jpgTXTMAX   = 128 ;        // Length of text fields in JPEG classes
const short jpgCOMMAX   = 32 ;         // Max JFIF comments captured
const short jpgJFIF_LEN = 5 ;          // Length of "JFIF\0" string
const char* jpgJFIF  = "JFIF" ;        // Constant string (null terminated)
const short jpgEXIF_MARKER_LEN = 6 ;   // Length of EXIF marker string
const char* jpgEXIF_MARKER = "EXif\0" ;// EXIF marker string
enum jpgRU : short { ruPixels = 0x00, ruDpi = 0x01, ruDpcm = 0x02 } ; // Resolution Units

#define DEBUGjpg (0)    // For debugging only, output intermediate results

//***********************************
//* Classes for processing JPG data *
//***********************************

//* Image File Directory records are defined as part of     *
//* the EXIF data included within a JPEG image file. These  *
//* records may contain information interesting to the user.*
class ifdRecord
{
   public:
   ~ifdRecord ( void ) {}  // default destructor
   ifdRecord ( void )
   {
      this->reset () ;
   }

   void reset ( void )
   {  //* Initialize the text field *
      for ( short i = ZERO ; i < jpgTXTMAX ; ++i )
         this->str[i] = '\0' ;
      this->valCap = false ;  // reset the "value captured" flag
   }

   uint16_t tag ;          // IFD record tag
   uint16_t typ ;          // IFD record data type (member of enum ifdTypes)
   uint32_t cnt ;          // number of items of 'type'
   uint32_t off ;          // offset from top of TIFF header to data
                           // (or the actual data, if it fits)
   uint32_t val32 ;        // decoded 16-bit or 32-bit value
   uint32_t fracNum ;      // numerator of fraction
   uint32_t fracDen ;      // denominator of fraction
   uint8_t  val8 ;         // decoded 8-bit value (hex or character)
   char     str[jpgTXTMAX];// decoded text string (null-terminated)
   bool     valCap ;       // 'true' if value(s) fully captured
} ;   // ifdRecord

//* The Exif header may contain GPS (Global Positioning System) data.*
//* If specified, these data will be stored in an instance of this   *
//* class within the exifHeader class object.                        *
//* Note that many of the GPS data records are not decoded. The raw  *
//* data will be read into a temporary structure and discarded before*
//* the returning from the 'exifHeader::decodeHeader' method.        *
class exifGps
{
   public:
   ~exifGps ( void ) {}    // destructor
   exifGps ( void )        // default constructor
   {
      this->reset () ;
   }

   void reset ( void )     // initialize all data members
   {
      this->latDeg = this->latMin = this->latSec = 0.0 ;
      this->lonDeg = this->lonMin = this->lonSec = 0.0 ;
      *this->areaName = '\0' ;
      *this->tstamp   = '\0' ;
      *this->dstamp   = '\0' ;
      this->latRef    = ' ' ;
      this->lonRef    = ' ' ;
      this->altCap    = false ;
      this->validData = false ;
   }

   //*************************
   //** Public Data Members **
   //*************************
   double latDeg ;         // Latitude Degrees
   double latMin ;         // Latitude Minutes
   double latSec ;         // Latitude Seconds
   double lonDeg ;         // Longitude Degrees
   double lonMin ;         // Longitude Minutes
   double lonSec ;         // Longitude Seconds
   double altitude ;       // Altitude (meters)
   char   areaName[jpgTXTMAX] ; // Area name, Example: "Sichuan, China"
   char   tstamp[jpgTXTMAX] ;   // GPS timestamp (converted to string: HH:MM:SS)
   char   dstamp[jpgTXTMAX] ;   // GPS datestamp (converted to string: YYYY:MM:DD)
   char   latRef ;         // Latitude reference : 'N' == North, 'S' == South
   char   lonRef ;         // Longitude reference: 'E' == East,  'W' == West
   bool   altCap ;         // 'true' if altitude value captured
   bool   validData ;      // true if valid data captured
} ;   // exifGps

//* Information describing the thumbnail image (if present) *
class jpgThumb
{
   public:
      ~jpgThumb ( void ) {}   // destructor
      jpgThumb ( void )       // default constructor
      {
         this->reset () ;
      }

   void reset ( void )
   {
      this->offset = this->bytes = 0x00000000 ;
      this->width = this->height = 0x0000 ;
      this->sprec = this->hsamp = this->vsamp = this->cif = this->csid =
      this->qtable = 0x00 ;
      this->exists = false ;
   }  // reset()

   void operator = ( const jpgThumb *jtPtr )
   {
      this->offset = jtPtr->offset ;
      this->bytes  = jtPtr->bytes ;
      this->width  = jtPtr->width ;
      this->height = jtPtr->height ;
      this->sprec  = jtPtr->sprec ;
      this->hsamp  = jtPtr->hsamp ;
      this->vsamp  = jtPtr->vsamp ;
      this->cif    = jtPtr->cif ;
      this->csid   = jtPtr->csid ;
      this->qtable = jtPtr->qtable ;
      this->exists = jtPtr->exists ;
   }  // operator=()

   //* Scan the contents of the specified thumbnail image and extract*
   //* basic information. Captured information is stored in the      *
   //* member variables.                                             *
   //*                                                               *
   //* The interesting data are embedded in the Start-Of-Frame tag.  *
   //* SOF0 (baseline scan) or SOF2 (progressive scan).              *
   //* The Start-Of-Frame structure is of the form:                  *
   //*  SOFn                       16 bits                           *
   //*  Header Length              16 bits                           *
   //*  Sample Precision            8 bits (bits/color sample)       *
   //*                                     range 2-16 bits so:       *
   //*                                     RGB==3*precision==24      *
   //*                                     bits per pixel            *
   //*  Line Count                 16 bits                           *
   //*  Samples-per-line           16 bits                           *
   //*  Components In Frame         8 bits                           *
   //*  Frame Component Specifier  24 bits:                          *
   //*     8 bits  color-space identifier                            *
   //*             valid values: 1==Y,  luma                         *
   //*                           2==Cb, chroma, blue-difference      *
   //*                           3==Cr  chroma, red-difference       *
   //*     4 bits  horizontal sampling factor (upper 4 bits)         *
   //*             valid values: 1, 2, 3, 4                          *
   //*     4 bits  vertical sampling factor (lower 4 bits)           *
   //*             valid values: 1, 2, 3, 4                          *
   //*     8 bits  quantization table destination                    *
   //*             valid values: 0, 1, 2, 3                          *
   //*                                                               *
   //* Note that additional start-of-frame tags exist for specialized*
   //* scan formats with the same basic data layout for the first    *
   //* thirteen(13) bytes, but they are rarely used in consumer-     *
   //* oriented devices or in web images, so they are ignored here.  *
   //*                                                               *
   //* The size of the thumbnail image in bytes is calculated by     *
   //* counting the bytes from the start-of-image (SOI) marker and   *
   //* the end-of-image marker (including the markers themselves).   *
   //*                                                               *
   //* Input  : srcbuff : pointer to source data array               *
   //*          srcBytes: number of data bytes in 'src'              *
   //*          ofs     : For debugging only, access to an open      *
   //*                    output stream, 'ofs' which is used to      *
   //*                    report intermediate results of the scan.   *
   //*                                                               *
   //* Returns: 'true'  if data successfully captured                *
   //*          'false' if source data is not a valid JPEG image     *
// CZONE - RETURN INDEX OF BYTE FOLLOWING END OF THUMBNAIL? (this->bytes)
   bool decodeThumbnail ( const uint8_t *srcbuff, uint16_t srcBytes, ofstream& ofs )
   {
      #if DEBUGjpg != 0
      #define DEBUG_THUMB (0)    // for debugging only
      #if DEBUG_THUMB != 0
      #define DEBUG_TTAGS (0)
      #endif   // DEBUG_THUMB
      #endif   // DEBUGjpg

      uint16_t tagcode, taglen,  // tag name and tag length (bytes)
               tagdata,          // tag data bytes (excluding length value)
               indx = jpgINT16 ; // index into source-data array
      bool     status = true ;   // return value (hope for the best)

      //* Verify that source data references beginning of thumbnail *
      tagcode = this->shortConvBE ( srcbuff ) ;
      if ( tagcode != jpgSOI )   // do not perform the scan
         return ( false ) ;      // (note the early return)

      #if DEBUGjpg != 0 && DEBUG_THUMB != 0
      gString gsfmt( "Thumbnail Scan: %04hX  %s\n"
                     "-------------------------\n", 
                     &tagcode, (tagcode == jpgSOI ? "OK" : "ERR")  ) ;
      ofs << gsfmt.ustr() ;
      #endif   // DEBUGjpg && DEBUG_THUMB

      //* Locate the Start-Of-Frame, either jpgSOF0 or jpgSOF2. *
      while ( indx < srcBytes )
      {
         tagcode = this->shortConvBE ( &srcbuff[indx] ) ;
         indx += jpgINT16 ;
         taglen  = this->shortConvBE ( &srcbuff[indx] ) ;
            indx += jpgINT16 ;
         tagdata = taglen - 2 ;

         #if DEBUGjpg != 0 && DEBUG_THUMB != 0 && DEBUG_TTAGS != 0
         gsfmt.compose( " tagcode:%04hX taglen:%04hX (%hu)\n", &tagcode, &taglen, &taglen ) ; ofs << gsfmt.ustr() ;
         #endif   // DEBUGjpg && DEBUG_THUMB && DEBUG_TTAGS

         if ( (tagcode == jpgSOF0) || (tagcode == jpgSOF2) )
         {
            this->sprec = srcbuff[indx++] ;
            this->height = this->shortConvBE ( &srcbuff[indx] ) ;
            indx += jpgINT16 ;
            this->width = this->shortConvBE ( &srcbuff[indx] ) ;
            indx += jpgINT16 ;
            this->cif    = srcbuff[indx++] ;
            this->csid   = srcbuff[indx++] ;
            this->hsamp  = (uint8_t)((srcbuff[indx] & 0xF0) >> 4) ;
            this->vsamp  = (uint8_t)(srcbuff[indx++] & 0x0F) ;
            this->qtable = (uint8_t)srcbuff[indx] ;

            #if DEBUGjpg != 0 && DEBUG_THUMB != 0
            gsfmt.compose( "%02hhX         Sample Precision\n"
                           "%04hX (%3hu) Line Count\n"
                           "%04hX (%3hu) Samples Per Line\n"
                           "%02hhX         Components In Frame\n"
                           "%02hhX         Identifier\n"
                           "%02hhX         Horizontal sampling factor\n"
                           "%02hhX         Vertical sampling factor\n"
                           "%02hhX         Quantization table dest.\n",
                           &this->sprec, &this->height, &this->height, 
                           &this->width, &this->width, &this->cif,
                           &this->csid, &this->hsamp, &this->vsamp, &this->qtable ) ;
            ofs << gsfmt.ustr() ;
            #endif   // DEBUGjpg && DEBUG_THUMB

            //* Scan to the end of the thumbnail image *
            //* to obtain the size of the image.       *
            while ( indx < srcBytes )
            {
               //* Possible tag code *
               if ( srcbuff[indx] == 0xFF )
               {
                   if ( srcbuff[indx + 1] != 0x00 )
                   {
                      tagcode = this->shortConvBE ( &srcbuff[indx] ) ;

                      #if DEBUGjpg != 0 && DEBUG_THUMB != 0 && DEBUG_TTAGS != 0
                      gsfmt.compose( " tagcode: %04hX indx:%04hX\n", &tagcode, &indx ) ; ofs << gsfmt.ustr() ;
                      #endif   // DEBUGjpg && DEBUG_THUMB && DEBUG_TTAGS
                      indx += jpgINT16 ;

                      if ( tagcode == jpgEOI )
                      {
                         this->bytes = indx ;
                         indx = srcBytes ;   // terminate the outer loop

                         #if DEBUGjpg != 0 && DEBUG_THUMB != 0
                         gsfmt.compose( "%5hu      Thumbnail size\n", &this->bytes ) ;
                         ofs << gsfmt.ustr() ;
                         #endif   // DEBUGjpg && DEBUG_THUMB

                         break ;
                      }
                   }
                   else
                      ++indx ;
               }
               else
                  ++indx ;
            }
         }
         else     // step past the data of the curret tag
            indx += tagdata ;
      }
      return status ;

   }  // decodeThumbnail()

   //* Convert a 2-byte, big-endian sequence to a 16-bit short integer.*
   //* Note that under the JPEG standard, all integers are big-endian. *
   uint16_t shortConvBE ( const uint8_t* srcbuf )
   {
      uint16_t i =    (uint16_t)srcbuf[1] 
                   | ((uint16_t)srcbuf[0] << 8) ;
      return i ;
   }  // shortConv()


   //*************************
   //** Public Data Members **
   //*************************
   uint32_t offset ;          // Used internally during data scan:
                              // Offset from top of Exif header to thumbnail image
                              // _or_ offset from top of JFIF header to thumbnail image
   uint32_t bytes ;           // Length of thumbnail image in bytes
   uint16_t width ;           // Width of image in pixels
   uint16_t height ;          // Height of image in pixels
   uint8_t  sprec ;           // Sampling precision (see above)
   uint8_t  hsamp ;           // Horizontal sampling factor (see above)
   uint8_t  vsamp ;           // Vertical sampling factor (see above)
   uint8_t  cif ;             // Components-in-frame (see above)
   uint8_t  csid ;            // Color-space identifier (see above)
   uint8_t  qtable ;          // Quantization table destination (table index?)
   bool     exists ;          // 'true' if thumbnail image located
} ;   // jpgThumb

//* The Exif header is often present in JPEG files. *
//* It is used by digital cameras to record various *
//* data about the device and the image. It is      *
//* indicated by the "APP1" tag. See notes above.   *
class exifHeader
{
   public:
   ~exifHeader ( void ) {}       // destructor
   exifHeader ( void )           // default constructor
   {
      this->reset() ;
   }

   void reset ( void )           // initialize all data members
   {
      *this->exifID   = '\0' ;
      this->ifdOffset = 0x00000000 ;
      this->appMarker = this->tiffConst = this->hdrSize = 
      this->ifdEntries = 0x0000 ;
      this->leByteOrder = false ;
#if 0    // CZONE - RESETTING THE EMBEDDED OBJECTS IS NOT NECESSARY.
      this->gps.reset() ;
      this->thumb.reset() ;
      for ( short i = ZERO ; i < jpgIFDMAX ; ++i )
         this->ifd[i].reset() ;
#endif   // U/C
   }

   //* Decode the Exif header section.                                     *
   //* -- IFD records for the primary IFD and secondary IFD are returned   *
   //*    in the 'ifd' array.                                              *
   //* -- GPS data (if present) are decoded locally, so the GPS raw data   *
   //*    records are not returned to caller.                              *
   //* -- Note that for rational values, the "count" indicated the number  *
   //*    of numerator/denominator pairs, NOT the number of integers.      *
   //* -- The following two tag codes MAY indicate text data; however, the *
   //*    data type is "undefined" by the standard. If the data type is    *
   //*    one of the types defined by the standard (see enum ifdTypes),    *
   //*    the data will be captured, else data record will be ignored.     *
   //*     0x927C  Manufacturer's Notes                                    *
   //*     0x9286  User Comment (see also ucEncoding())                    *
   //*                                                                     *
   //* NOTE: The image-encoding software calculates values for the         *
   //* "offset" fields of each IFD record from the position following the  *
   //* "Exif\0\0" constant string. The 'tiffHead' variable indicates this  *
   //* position as an offset from the top of the 'srcbuff' source buffer.  *
   //* This value is then added to the "offset" field value for each tag   *
   //* record when extracting the record data.                             *
   //*                                                                     *
   //* Input  : srcbuff : pointer to source data                           *
   //*                    Important Note: The first two(2) bytes of source *
   //*                    data contain the length of the header data.      *
   //*          appMark : application marker code (captured by caller)     *
   //*          ofs     : For debugging only, access to an open output     *
   //*                    stream, 'ofs' which is used to report            *
   //*                    intermediate results of the scan.                *
   //* Returns: nothing                                                    *
   void decodeHeader ( const uint8_t* srcbuff, uint16_t appMark, ofstream& ofs )
   {
      const uint16_t versTag  = 0x9000 ;     // Exif version tag (special format)
      const uint16_t ucomTag  = 0x9286 ;     // "User Comment" tag (special format)
      const uint16_t ifd2Tag  = 0x8769 ;     // Offset to Exif IFD Attribute array
      const uint16_t ifd3Tag  = 0x8825 ;     // Offset to GPS IFD array
      const uint16_t tOffTag  = 0x0201 ;     // Offset in IFD header to thumbnail image
      const uint16_t tLenTag  = 0x0202 ;     // Number of bytes in thumbnail image

      //* Data types defined by the Exif standard *
      enum ifdTypes : uint16_t
      {
         ifdtBYTE      = 1,   // unsigned 8-bit integer
         ifdtASCII     = 2,   // null-terminated ASCII string
         ifdtSHORT     = 3,   // unsigned 16-bit integer
         ifdtLONG      = 4,   // unsigned 32-bit integer
         ifdtRATIONAL  = 5,   // a pair of unsigned 32-bit integers: numerator,denominator
         ifdtUNDEF     = 7,   // one or more 8-bit bytes as defined by the field
         ifdtSLONG     = 9,   // signed 32-bit integer (2's complement)
         ifdtSRATIONAL = 10   // a pair of signed 32-bit integers: numerator,denominator
      } ;

      uint16_t srcIndx  = ZERO,  // source index
               trgIndx  = ZERO,  // target index
               ifd2     = ZERO,  // offset to Exif-specific IFD (if specified)
               ifd3     = ZERO,  // offset to Exif GPS IFD (if specified)
               tiffHead = ZERO,  // "offset" field correction factor (see note above)
               stIndx ; // short-text index for text located in 'off' field (<= 4 bytes)
      gString gsfmt ;            // text formatting

      this->reset () ;           // reset our data members

      //* Save the header size and application marker, then decode the header *
      this->hdrSize   = this->shortConvBE ( srcbuff ) ;
      srcIndx += jpgINT16 ;
      this->appMarker = appMark ;

      //* Capture the identification string *
      while ( trgIndx < jpgEXIF_MARKER_LEN )
         this->exifID[trgIndx++] = srcbuff[srcIndx++] ;
      this->exifID[trgIndx - 1] = '\0' ;// be sure string is terminated
      //* Set the correction value for "offset" fields *
      tiffHead = srcIndx ;

      //* Determine whether numeric data are stored *
      //* big-endian ("II") or little-endian ("MM").*
      if ( (srcbuff[srcIndx] == 'I') && (srcbuff[srcIndx + 1] == 'I') )
         this->leByteOrder = true ;
      srcIndx += 2 ;
      //* Capture the TIFF constant value *
      this->tiffConst = this->shortConv ( srcbuff, srcIndx ) ;
      //* Capture the IFD offset *
      this->ifdOffset = this->intConv ( srcbuff, srcIndx ) ;
      //* Capture the number of IFD records in the array *
      this->ifdEntries = this->shortConv  ( srcbuff, srcIndx ) ;

      //* Scan the IFD structures for interesting data.    *
      for ( int16_t i = ZERO ; i < this->ifdEntries ; ++i )
      {
         //* Capture the record tag *
         this->ifd[i].tag = this->shortConv ( srcbuff, srcIndx ) ;
         //* Capture the record type *
         this->ifd[i].typ = this->shortConv ( srcbuff, srcIndx ) ;
         //* Capture the count *
         this->ifd[i].cnt = this->intConv ( srcbuff, srcIndx ) ;
         //* Capture the offset/value *
         stIndx = srcIndx ;   // save index for possible "short text"
         this->ifd[i].off = this->intConv ( srcbuff, srcIndx ) ;
#if 0    // TEMP TEMP TEMP
gsfmt.compose( "[i:%2hu] tag:%04hX typ:%04hX cnt:%04X off:%04X\n", 
               &i, &this->ifd[i].tag, &this->ifd[i].typ, &this->ifd[i].cnt, &this->ifd[i].off ) ;
ofs << gsfmt.ustr() ;
#endif   // TEMP TEMP TEMP

         //********************************************
         //* For values that are stored in 'off',     *
         //* extract the value(s).                    *
         //* For values that are stored at offset     *
         //* specified by 'off', capture the value(s) *
         //********************************************
         // Programmer's Note: No record may contain multiple 32-bit or 16-bit values.
         // However, multiple "rational" values are possible in the GPS data only.

         //* One, 32-bit value, (signed or unsigned) *
         if ( ((this->ifd[i].typ == ifdtLONG) || (this->ifd[i].typ == ifdtSLONG))
              && (this->ifd[i].cnt == 1) )
         {
            this->ifd[i].val32 = this->ifd[i].off ;
            if ( this->ifd[i].tag == ifd2Tag )      // IFD attribute array (Exif) exists
               ifd2 = this->ifd[i].off ;
            else if ( this->ifd[i].tag == ifd3Tag ) // IFD attribute array (GPS) exists
               ifd3 = this->ifd[i].off ;
            else if ( this->ifd[i].tag == tOffTag ) // offset to thumbnail IFD
            { this->thumb.offset = this->ifd[i].off ; this->thumb.exists = true ; }
            else if ( this->ifd[i].tag == tLenTag ) // length of thumbnail IFD
               this->thumb.bytes = this->ifd[i].off ;
            this->ifd[i].valCap = true ;
         }

         //* One, 16-bit value, (signed or unsigned) *
         else if ( (this->ifd[i].typ == ifdtSHORT) )
         {
            this->ifd[i].val32 = (uint32_t)(this->ifd[i].off & 0x0000FFFF) ;
            this->ifd[i].valCap = true ;
         }

         //* An 8-bit byte. May be text character or  *
         //* an 8-bit integer (unsigned or signed).   *
         else if ( (this->ifd[i].typ == ifdtBYTE) || 
                   (this->ifd[i].typ == ifdtUNDEF) )
         {
            this->ifd[i].val32 = (uint32_t)(this->ifd[i].off & 0x000000FF) ;
            this->ifd[i].valCap = true ;
         }

         //* Text string. (may or may not actually be ASCII data, *
         //* see 'ucomTag' 0x9286)                                *
         else if ( (this->ifd[i].typ == ifdtASCII) )
         {
            int16_t txtIndx = tiffHead + (int16_t)this->ifd[i].off ;
            if ( this->ifd[i].cnt <= 4 )  // text <= 4 bytes (incl. null char)
               txtIndx = stIndx ;
            trgIndx = ZERO ;

            //* Special case: user comment field includes an 8-byte *
            //* indicator about the text-encoding method.           *
            //* See notes in header of ucEncoding() method.         *
            if ( (this->ifd[i].tag == ucomTag) && (this->ifd[i].cnt > 8) )
            {
               uint16_t offset = this->ucEncoding ( &srcbuff[txtIndx] ) ;
               this->decodeTextField ( &srcbuff[txtIndx + offset], this->ifd[i].str, 
                                       (this->ifd[i].cnt - offset) ) ;
            }
            else
               this->decodeTextField ( &srcbuff[txtIndx], this->ifd[i].str, 
                                       this->ifd[i].cnt ) ;
            this->ifd[i].valCap = true ;
         }

         //* "Rational" value i.e. a fraction with *
         //* 32-bit numerator and denominator.     *
         //* Type 5 = unsigned, Type 10 == signed. *
         // Programmer's Note: For simplicity, we assume that the captured 
         // values will be positive and will never require more than 31 bits.
         else if ( (this->ifd[i].typ == ifdtRATIONAL) || 
                   (this->ifd[i].typ == ifdtSRATIONAL) )
         {
            uint16_t valIndx = tiffHead + (int16_t)this->ifd[i].off ;
            this->ifd[i].fracNum = this->intConv ( srcbuff, valIndx ) ;
            this->ifd[i].fracDen = this->intConv ( srcbuff, valIndx ) ;
            this->ifd[i].valCap = true ;
         }
      }  // for(;;)

      //* If an Exif-specific Attribute IFD  *
      //* was specified, decode its records. *
      if ( ifd2 != ZERO )
      {
         srcIndx = tiffHead + ifd2 ;
         uint16_t ifd2Cnt  = this->shortConv ( srcbuff, srcIndx ),
                  ifdTotal = this->ifdEntries + ifd2Cnt ;

         #if DEBUGjpg != 0    // For debugging only
         uint16_t foff = srcIndx + 4 ;
         gString gsdbg( "\n"
                        "Exif Attribute IFD\n"
                        "-----------------------------------------------------------------\n"
                        "ifd2:%hXh (%hu) ifd2Cnt:%hu ifdTotal:%hu srcIndx:%04hX (foff:0x%04hX)", 
                        &ifd2, &ifd2, &ifd2Cnt, &ifdTotal, &srcIndx, &foff ) ;
         if ( ifdTotal > jpgIFDMAX )
         {
            int16_t discarded = ifdTotal - jpgIFDMAX ;
            gsdbg.append( "\nNOTE: %hu IFD records were discarded.", &discarded ) ;
         }
         ofs << gsdbg.ustr() << endl ;
         #endif   // DEBUGjpg

         //* Prevent overrun of 'ifd' array.  *
         //* Excess records will be discarded.*
         //* (This is unlikely, but possible.)*
         if ( ifdTotal > jpgIFDMAX )
            ifdTotal = jpgIFDMAX ;

         for ( int16_t i = this->ifdEntries ; i < ifdTotal ; ++i )
         {
            //* Capture the record tag *
            this->ifd[i].tag = this->shortConv ( srcbuff, srcIndx ) ;
            //* Capture the record type *
            this->ifd[i].typ = this->shortConv ( srcbuff, srcIndx ) ;
            //* Capture the count *
            this->ifd[i].cnt = this->intConv ( srcbuff, srcIndx ) ;
            //* Capture the offset/value *
            stIndx = srcIndx ;   // save index for possible "short text"
            this->ifd[i].off = this->intConv ( srcbuff, srcIndx ) ;

            //********************************************
            //* For values that are stored in 'off',     *
            //* extract the value(s).                    *
            //* For values that are stored at offset     *
            //* specified by 'off', capture the value(s) *
            //********************************************
            // Programmer's Note: No record may contain multiple 32-bit or 16-bit values.
            // However, multiple "rational" values are possible in the GPS data only.

            //* One, 32-bit value, (signed or unsigned) *
            if ( (this->ifd[i].typ == ifdtLONG) && (this->ifd[i].cnt == 1) )
            {
               this->ifd[i].val32 = this->ifd[i].off ;
               this->ifd[i].valCap = true ;
            }

            //* One, 16-bit value, (signed or unsigned) *
            else if ( (this->ifd[i].typ == ifdtSHORT) )
            {
               this->ifd[i].val32 = (uint32_t)(this->ifd[i].off & 0x0000FFFF) ;
               this->ifd[i].valCap = true ;
            }

            //* An 8-bit byte. May be text character or  *
            //* an 8-bit integer (unsigned or signed).   *
            else if ( (this->ifd[i].typ == ifdtBYTE) || 
                      (this->ifd[i].typ == ifdtUNDEF) )
            {  //* A single byte value *
               // Programmer's Note: It is possible that the byte count will be
               // greater than 1, but if so, we would need to verify that data
               // are ASCII bytes before storing in the 'str' member.
               // In the special case of tag 'versTag' (0x9000) 
               // (or Flashpix version 0xA000 (ignored),we can assume four 
               // ASCII-numeric bytes according to the spec.
               if ( this->ifd[i].cnt == 1 )
                  this->ifd[i].val32 = (uint32_t)(this->ifd[i].off & 0x000000FF) ;

               else if ( (this->ifd[i].tag == versTag) || (this->ifd[i].cnt == 4) )
               {//* Special Case: Exif version, four(4) ASCII-numeric bytes *
                  gsfmt.compose( "%c.%c.%c.%c",
                                 (char*)(&this->ifd[i].off),
                                 (char*)((char*)&this->ifd[i].off + 1),
                                 (char*)((char*)&this->ifd[i].off + 2),
                                 (char*)((char*)&this->ifd[i].off + 3) ) ;
                  //* If little-endian, reverse character order *
                  if ( this->leByteOrder )
                     gsfmt.textReverse() ;
                  gsfmt.copy( this->ifd[i].str, jpgTXTMAX ) ;
               }
               else
               { /* CAPTURE OF 2, 3, 4 BYTES IS NOT IMPLEMENTED */ }

               this->ifd[i].valCap = true ;
            }

            //* Text string. (may or may not actually be ASCII data, see tag 0x9286) *
            else if ( (this->ifd[i].typ == ifdtASCII) )
            {
               int16_t txtIndx = tiffHead + (int16_t)this->ifd[i].off ;
               if ( this->ifd[i].cnt <= 4 )  // text <= 4 bytes (incl. null char)
                  txtIndx = stIndx ;
               trgIndx = ZERO ;

               //* Special case: user comment field includes an 8-byte *
               //* indicator about the text-encoding method.           *
               //* See notes in header of ucEncoding() method.         *
               if ( (this->ifd[i].tag == 0x9286) && (this->ifd[i].cnt > 8) )
               {
                  uint16_t offset = this->ucEncoding ( &srcbuff[txtIndx] ) ;
                  this->decodeTextField ( &srcbuff[txtIndx + offset], this->ifd[i].str, 
                                          (this->ifd[i].cnt - offset) ) ;
               }
               else
                  this->decodeTextField ( &srcbuff[txtIndx], this->ifd[i].str, 
                                          this->ifd[i].cnt ) ;
               this->ifd[i].valCap = true ;
            }

            //* "Rational" value i.e. a fraction with *
            //* 32-bit numerator and denominator.     *
            //* Type 5 = unsigned, Type 10 == signed. *
            // Programmer's Note: For simplicity, we assume above that the 
            // captured values will be positive and will never require more 
            // than 31 bits. However, for the supplementary data array this 
            // is not always true: "Brightness" and "ExposureBiasValue" at 
            // least, are SIGNED fractions.
            else if ( (this->ifd[i].typ == ifdtRATIONAL) || 
                      (this->ifd[i].typ == ifdtSRATIONAL) )
            {
               uint16_t valIndx = tiffHead + (int16_t)this->ifd[i].off ;
               this->ifd[i].fracNum = this->intConv ( srcbuff, valIndx ) ;
               this->ifd[i].fracDen = this->intConv ( srcbuff, valIndx ) ;
               this->ifd[i].valCap = true ;
            }

            ++this->ifdEntries ;
         }  // for(;;)
      }  // if(ifd2!=ZERO)

      //* If an Exif GPSc IFD was specified, decode*
      //* selected records and store them in the   *
      //* 'gps' member object.                     *
      //* Note that the raw data are not retained. *
      //* Notes on Latitude and Longitude:                          *
      //* a) encoded as three(3) rational values (fractions)        *
      //* b) "degrees" should be a whole number i.e. denominator==1 *
      //* c) "minutes" can be either whole minutes e.g. 109.00      *
      //*                       OR decimal minutes e.g. 109.192486  *
      //*    If decimal minutes, then seconds should be ZERO.       *
      //* d) "seconds" is likely a decimal value e.g. 19.2486       *
      //*    (or ZERO if decimal minutes).
      if ( ifd3 != ZERO )
      {
         enum exifGps : uint16_t
         {
            extgVersion    = 0x0000,   // GPS Info IFD version
            extgLatRef     = 0x0001,   // 'N'orth or 'S'outh latitude
            extgLatitude   = 0x0002,   // Latitude hh,mm,ss
            extgLonRef     = 0x0003,   // 'E'ast or 'W'est longitude
            extgLongitude  = 0x0004,   // Longitude hh,mm,ss
            extgAltRef     = 0x0005,   // Altitude reference:
                                       //   0 at or above sea-level (positive value)
                                       //   1 below sea-level (negative value)
            extgAltitude   = 0x0006,   // Altitude (sign depends on 'extgAltRef'
            extgTime       = 0x0007,   // Time: hours:minutes:seconds
            extgAreaInfo   = 0x001C,   // Area description (string MAY NOT be null terminated)
            extgDate       = 0x001D,   // Date: year:month:date
         } ;

         srcIndx = tiffHead + ifd3 ;   // source index
         uint16_t ifd3Cnt  = this->shortConv ( srcbuff, srcIndx ) ; // record count
         uint16_t valIndx ;            // record index
         ifdRecord ifdRec ;            // temp buffer
         double fpNum, fpDen ;         // floating-point conversions
         uint8_t altref ;              // altitude reference code
         if ( ifd3Cnt > ZERO )         // if data will be reported, set the flag
            this->gps.validData = true ;

         #if DEBUGjpg != 0    // For debugging only
         uint16_t foff = srcIndx + 4 ;
         gString gs( "offset:%hXh (%hu) ifd3Cnt:%hu srcIndx:%04hXh (foff:%04hXh)\n", 
                     &ifd3, &ifd3, &ifd3Cnt, &srcIndx, &foff ) ;
         ofs << "\nGPS DATA\n------------------------------------------------------\n"
             << gs.ustr()
             << "INDX TAG_  TYPE  COUNT     OFFSET\n"
             << "---- ----  ----  --------  --------\n" ;
         #endif   // DEBUGjpg

         for ( int16_t i = ZERO ; i < ifd3Cnt ; ++i )
         {
            ifdRec.tag = this->shortConv ( srcbuff, srcIndx ) ;   // record tag
            ifdRec.typ = this->shortConv ( srcbuff, srcIndx ) ;   // record type
            ifdRec.cnt = this->intConv ( srcbuff, srcIndx ) ;     // record item count
            stIndx = srcIndx ;   // possible text (or other byte data) in 'off' field
            ifdRec.off = this->intConv ( srcbuff, srcIndx ) ;     // record item offset
            valIndx = tiffHead + (int16_t)ifdRec.off ;

            #if DEBUGjpg != 0    // For debugging only
            gs.compose( "[% 2hd] %04hX  %04hX  %08X  %08X\n",
                        &i, &ifdRec.tag, &ifdRec.typ, &ifdRec.cnt, &ifdRec.off ) ;
            ofs << gs.ustr() ;
            #endif   // DEBUGjpg

            if ( ifdRec.tag == extgLatRef )           // latitude reference
            {
               this->gps.latRef = srcbuff[stIndx] ;
            }
            else if ( ifdRec.tag == extgLatitude )    // latitude
            {
               ifdRec.fracNum = this->intConv ( srcbuff, valIndx ) ;
               ifdRec.fracDen = this->intConv ( srcbuff, valIndx ) ;
               this->gps.latDeg = double(ifdRec.fracNum / ifdRec.fracDen) ;
               ifdRec.fracNum = this->intConv ( srcbuff, valIndx ) ;
               ifdRec.fracDen = this->intConv ( srcbuff, valIndx ) ;
               if ( ifdRec.fracDen == 1 )    // whole-number value
                  this->gps.latMin = double(ifdRec.fracNum / ifdRec.fracDen) ;
               else                          // decimal value
               {
                  fpNum = double(ifdRec.fracNum) ;
                  fpDen = double(ifdRec.fracDen) ;
                  this->gps.latMin = fpNum / fpDen ;
               }
               ifdRec.fracNum = this->intConv ( srcbuff, valIndx ) ;
               ifdRec.fracDen = this->intConv ( srcbuff, valIndx ) ;
               fpNum = double(ifdRec.fracNum) ;
               fpDen = double(ifdRec.fracDen) ;
               this->gps.latSec = fpNum / fpDen ;
            }
            else if ( ifdRec.tag == extgLonRef )      // longitude reference
            {
               this->gps.lonRef = srcbuff[stIndx] ;
            }
            else if ( ifdRec.tag == extgLongitude )   // longitude
            {
               ifdRec.fracNum = this->intConv ( srcbuff, valIndx ) ;
               ifdRec.fracDen = this->intConv ( srcbuff, valIndx ) ;
               this->gps.lonDeg = double(ifdRec.fracNum / ifdRec.fracDen) ;
               ifdRec.fracNum = this->intConv ( srcbuff, valIndx ) ;
               ifdRec.fracDen = this->intConv ( srcbuff, valIndx ) ;
               if ( ifdRec.fracDen == 1 )    // whole-number value
                  this->gps.lonMin = double(ifdRec.fracNum / ifdRec.fracDen) ;
               else                          // decimal value
               {
                  fpNum = double(ifdRec.fracNum) ;
                  fpDen = double(ifdRec.fracDen) ;
                  this->gps.lonMin = fpNum / fpDen ;
               }
               ifdRec.fracNum = this->intConv ( srcbuff, valIndx ) ;
               ifdRec.fracDen = this->intConv ( srcbuff, valIndx ) ;
               fpNum = double(ifdRec.fracNum) ;
               fpDen = double(ifdRec.fracDen) ;
               this->gps.lonSec = fpNum / fpDen ;
            }
            else if ( ifdRec.tag == extgAltRef )      // altitude reference
            {
               altref = srcbuff[valIndx] ;
            }
            else if ( ifdRec.tag == extgAltitude )    // altitude
            {
               ifdRec.fracNum = this->intConv ( srcbuff, valIndx ) ;
               ifdRec.fracDen = this->intConv ( srcbuff, valIndx ) ;
               this->gps.altitude = ifdRec.fracNum / ifdRec.fracDen ;
               if ( altref == 1 )      // below sea-level
                  this->gps.altitude = 0.0 - this->gps.altitude ;
            }
            else if ( (ifdRec.tag == extgTime) )      // GPS timestamp (UTC)
            {
               gsfmt.clear() ;
               uint16_t val ;
               ifdRec.fracNum = this->intConv ( srcbuff, valIndx ) ;
               ifdRec.fracDen = this->intConv ( srcbuff, valIndx ) ;
               val = uint16_t(ifdRec.fracNum / ifdRec.fracDen) ;
               gsfmt.append( "%02hu:", &val ) ; // Hour
               ifdRec.fracNum = this->intConv ( srcbuff, valIndx ) ;
               ifdRec.fracDen = this->intConv ( srcbuff, valIndx ) ;
               val = ifdRec.fracNum / ifdRec.fracDen ;
               gsfmt.append( "%02hu:", &val ) ; // Minutes
               ifdRec.fracNum = this->intConv ( srcbuff, valIndx ) ;
               ifdRec.fracDen = this->intConv ( srcbuff, valIndx ) ;
               val = ifdRec.fracNum / ifdRec.fracDen ;
               gsfmt.append( "%02hu", &val ) ;  // Seconds
               gsfmt.copy( this->gps.tstamp, jpgTXTMAX ) ;
            }
            else if ( (ifdRec.tag == extgDate) )      // GPS date stamp (UTC)
            {
               int16_t txtIndx = tiffHead + (int16_t)ifdRec.off ;
               if ( ifdRec.cnt <= 4 )     // text <= 4 bytes (incl. null char)
                  txtIndx = stIndx ;
               trgIndx = ZERO ;
               this->decodeTextField ( &srcbuff[txtIndx], this->gps.dstamp, 
                                       ifdRec.cnt ) ;
            }
            else if ( ifdRec.tag == extgAreaInfo )    // geographic area string
            {
               int16_t txtIndx = tiffHead + (int16_t)ifdRec.off ;
               if ( ifdRec.cnt <= 4 )     // text <= 4 bytes (incl. null char)
                  txtIndx = stIndx ;
               trgIndx = ZERO ;
               this->decodeTextField ( &srcbuff[txtIndx], this->gps.areaName, 
                                       ifdRec.cnt ) ;
            }

            //* At this time, all other GPS records are ignored *
         }  // for(;;)

         #if DEBUGjpg != 0    // For debugging only
         ofs << "\n" << endl ;
         #endif   // DEBUGjpg
      }  // if(ifd3!=ZERO)

      //* If thumbnail image tag was not found, scan the *
      //* remaining header data for a thumbnail image.   *
      if ( ! this->thumb.exists )
      {
         srcIndx &= 0xFFFE ;  // tags must begin on a word boundary
         uint16_t soi ;       // value to be tested

         while ( srcIndx < this->hdrSize )
         {  //* Get next 16-bit value *
            soi = this->shortConvBE ( &srcbuff[srcIndx] ) ;
            if ( soi == jpgSOI )
            {
               this->thumb.offset = srcIndx ;
               this->thumb.exists = true ;
               break ;
            }
            srcIndx += jpgINT16 ;   // index the next value
         }
      }

      //* If thumbnail image tag was found OR if thumbnail image SOI *
      //* was found during header scan, extract thumbnail info.      *
      //* 'thumb.offset' indexes start of thumbnail image.           *
      //* Scan to the end of the thumbnail to obtain its size.       *
      if ( this->thumb.exists )
      {
         this->thumb.decodeThumbnail ( &srcbuff[this->thumb.offset], 
                                       (this->hdrSize - this->thumb.offset), ofs ) ;

         //* If data follow the thumbnail image *
         if ( (this->thumb.offset + this->thumb.bytes) < this->hdrSize )
         {
// CZONE - ADDITIONAL DATA MAY FOLLOW THE THUMBNAIL.
            ofs << "Warning: Header data following thumbnail image was ignored.\n" ;
/* TEMP */ uint16_t tmp = this->thumb.offset + this->thumb.bytes ;
/* TEMP */ gsfmt.compose( "thumb.offset:%hu + thumb.bytes:%hu == %hu , hdrSize:%hu\n",
/* TEMP */                &this->thumb.offset, &this->thumb.bytes, &tmp, &this->hdrSize ) ;
/* TEMP */ ofs << gsfmt.ustr() ;
         }
      }

   }  //* End decodeHeader() *

   //* Convert a 2-byte, big-endian sequence to a 16-bit short integer.*
   uint16_t shortConvBE ( const uint8_t* srcbuf )
   {
      uint16_t i =    (uint16_t)srcbuf[1] 
                   | ((uint16_t)srcbuf[0] << 8) ;
      return i ;
   }  // shortConv()

   //* Convert a 2-byte, little-endian sequence to a 16-bit short integer.*
   uint16_t shortConvLE ( const uint8_t* srcbuf )
   {
      uint16_t i =    (uint16_t)srcbuf[0] 
                   | ((uint16_t)srcbuf[1] << 8) ;
      return i ;
   }  // shortConv()

   //* Convert a 4-byte, big-endian sequence to a 32-bit integer value. *
   uint32_t intConvBE ( const uint8_t* srcbuf )
   {
      uint32_t i =    (uint32_t)srcbuf[3] 
                   | ((uint32_t)srcbuf[2] << 8)
                   | ((uint32_t)srcbuf[1] << 16)
                   | ((uint32_t)srcbuf[0] << 24) ;
      return i ;
   }

   //* Convert a 4-byte little-endian sequence to a 32-bit integer value. *
   uint32_t intConvLE ( const uint8_t* srcbuf )
   {
      uint32_t i =    (uint32_t)srcbuf[0] 
                   | ((uint32_t)srcbuf[1] << 8)
                   | ((uint32_t)srcbuf[2] << 16)
                   | ((uint32_t)srcbuf[3] << 24) ;
      return i ;
   }

   //* These private methods assume that the 'leByteOrder' flag has  *
   //* been initialized to determine the integer-decoding methods.   *
   private:
   //* Capture a raw text stream to the specified target buffer.     *
   //* Input  : srcbuff : pointer to source data buffer              *
   //*                    Note: Data are not guaranteed to be        *
   //*                          null terminated                      *
   //*          trgbuff : pointer to target buffer                   *
   //*                    Note: Size of target buffer is assumed to  *
   //*                          be >= to jpgTXTMAX                   *
   //*          srcBytes: number of source bytes to be captured      *
   //* Returns: 'true'  if all source data captured                  *
   //*          'false' if data truncated                            *
   bool decodeTextField ( const uint8_t *srcbuff, char *trgbuff, uint16_t srcBytes )
   {
      uint16_t srcIndx = ZERO,
               trgIndx = ZERO ;
      bool status = true ;

      for ( ; (trgIndx <= srcBytes) && (trgIndx < (jpgTXTMAX - 1)) ; ++trgIndx )
      {
         trgbuff[trgIndx] = srcbuff[srcIndx++] ;

         //* Replace non-terminating null characters *
         if ( (trgbuff[trgIndx] == '\0') && 
              (trgIndx < (srcBytes - 1)) )
         { trgbuff[trgIndx] = '_' ; }
      }
      trgbuff[trgIndx] = '\0' ;  // be sure target string is null terminated

      return status ;

   }  // decodeTextField()

   //* Convert a 2-byte, sequence to a 16-bit integer.               *
   //* The caller's 'srcIndx' value is incremented by two.           *
   //* Input  : srcbuff : pointer to source data buffer              *
   //* Input  : srcIndx : (by reference) index into 'ibuff' member   *
   //* Returns: converted value                                      *
   uint16_t shortConv ( const uint8_t* srcbuff, uint16_t& srcIndx )
   {
      uint16_t sInt ;

      if ( this->leByteOrder )
         sInt = this->shortConvLE ( &srcbuff[srcIndx] ) ;
      else
         sInt = this->shortConvBE ( &srcbuff[srcIndx] ) ;
      srcIndx += jpgINT16 ;      // advance caller's index

      return sInt ;

   }  // shortConv()

   //* Convert a 4-byte, sequence to a 32-bit integer.               *
   //* Input  : srcbuff : pointer to source data buffer              *
   //* Input  : srcIndx : (by reference) index into 'ibuff' member   *
   //* Returns: converted value                                      *
   uint32_t intConv ( const uint8_t *srcbuff, uint16_t& srcIndx )
   {
      uint32_t iInt ;

      if ( this->leByteOrder )
         iInt = this->intConvLE ( &srcbuff[srcIndx] ) ;
      else
         iInt = this->intConvBE ( &srcbuff[srcIndx] ) ;
      srcIndx += jpgINT32 ;

      return iInt ;

   }  // intConv()

   //* Test the specified source data against the constant strings defined *
   //* for the "User Comment" field of the Exif header (tag: 0x9286) which *
   //* indicate the type of text encoding used.                            *
   //*                                                                     *
   //* According to the specification:                                     *
   //*  The character code used in the UserComment tag is identified based *
   //*  on an ID code in a fixed 8-byte area at the start of the tag data  *
   //*  area. The unused portion of the area shall be padded with NULL     *
   //*  ("00.H").                                                          *
   //*  ID codes are assigned by means of registration. The designation    *
   //*  method and references for each character code are given in         *
   //*  below. The value of Count N is determined based on the 8 bytes     *
   //*  in the character code area and the number of bytes in the user     *
   //*  comment part. Since the TYPE is not ASCII, NULL termination is     *
   //*  not necessary.                                                     *
   //*    "ASCII\0\0\0"       ITU-T T.50 IA5                               *
   //*    "JIS\0\0\0\0\0"     JIS X208-1990                                *
   //*    "Unicode\0"         Unicode Standard (whatever that means)       *
   //*    "\0\0\0\0\0\0\0\0"  Undefined (whatever that means)              *
   //*                                                                     *
   //* Programmer's Note: With several of the sample JPEG files we have    *
   //* tested, this encoding preamble to the User Comment field is not     *
   //* present. For these test files, the actual data begins at the        *
   //* beginning of the target-data area. This is reasonable since the     *
   //* field is poorly conceived and a waste of space.                     *
   //*                                                                     *
   //* Input  : srcbuff : pointer to source data                           *
   //*                                                                     *
   //* Returns: offset of first byte _after_ the encoding field (if any)   *
   int16_t ucEncoding ( const uint8_t *srcbuff )
   {
      const short ENCODE_FIELD_LEN = 8 ;     // length of encoding field
      int16_t offset = ZERO ;                // return value
      gString gs( (char*)srcbuff, ENCODE_FIELD_LEN ) ;
      if ( ((gs.compare( "ASCII" )) == ZERO) ||
           ((gs.compare( "JIS" )) == ZERO) ||
           ((gs.compare( "Unicode" )) == ZERO) || (srcbuff[0] == '\0') )
         offset = ENCODE_FIELD_LEN ;

      return offset ;
   }  // ucEncoding()

   //*************************
   //** Public Data Members **
   //*************************
   public:
   ifdRecord ifd[jpgIFDMAX] ;    // array of IFD records
   char exifID[jpgEXIF_MARKER_LEN + 1] ; // marker string "EXif\0\0"
   exifGps  gps ;                // Receives GPS data if provided
   jpgThumb thumb ;              // Receives thumbnail image info if provided
   uint32_t ifdOffset ;          // Offset from top of header to first IFD
   uint16_t appMarker ;          // APP1 : EXIF Application Marker (0xFFE1)
   uint16_t tiffConst ;          // Constant value 0x002A
   uint16_t hdrSize ;            // Length of EXIF header in bytes
   uint16_t ifdEntries ;         // Number of IFD records captured
   bool     leByteOrder ;        // 'true' if byte order little-endian, 
                                 // 'false' if byte order big-endian
} ;   // exifHeader

//* Although there is no formal header defined for the JFIF *
//* file format, except for the Start-of-Image tag, the     *
//* defacto heading will be found as in the notes above.    *
//* Note that "APP0" specifies a JFIF-compliant header, and *
//* "APP1" specifies an EXIF-compliant header.              *
class jpgHeader
{
   public:
   ~jpgHeader ( void )           // destructor
   {
      if ( this->ibuff != NULL ) // release the dynamic allocation
      { delete [] this->ibuff ; this->ibuff = NULL ; }
   }
   jpgHeader ( void )            // default constructor
   {
      this->ibuff = NULL ;       // be sure pointers are initialized
      this->reset() ;            // initialize all data members
      this->ibuff   = new uint8_t[jpgINBUFF] ; // dynamic allocation of input buffer
   }

   void reset ( void )           // initialize all data members
   {
      if ( this->ibuff != NULL )
         *this->ibuff = NULLCHAR ;
      *this->comment = NULLCHAR ;
      this->comIndx[0] = ZERO ;
      this->comCount = ZERO ;
      this->exif.reset() ;
      this->thumb.reset() ;
      *this->jfifID    = '\0' ;
      this->jfifSOI    = 0x0000 ;
      this->appMarker  = 0x0000 ;
      this->hdrSize    = 0x0000 ;
      this->fmtVersion = 0x0000 ;
      this->xDensity = this->yDensity = this->width = this->height = 0x0000 ;
      this->resUnits   = ruPixels ;
      this->jpgValid   = false ;
   }

   //* Scan and validate the JPEG file header.                         *
   //* If EXIF (camera-provided) data are available, decode it also.   *
   //*                                                                 *
   //* Input  : ifs : (by reference) an open file input stream         *
   //*          ofs     : For debugging only, access to an open output *
   //*                    stream, 'ofs' which is used to report        *
   //*                    intermediate results of the scan.            *
   //* Returns: 'true' if file is a valid JPG format, else 'false'     *
   bool validateHeader ( ifstream& ifs, ofstream& ofs )
   {
      this->reset () ;                                // reset our data members

#if 1    // CZONE - CONSOLIDATE THE FIRST FEW READS
      uint16_t soiTag = ZERO,       // start-of-image tag
               appTag = ZERO,       // application-specific tag
               hdrLen = ZERO ;      // number of bytes in file header

      //* Read 1) the file marker            *
      //*      2) the application marker     *
      //*      3) the length of the header   *
      ifs.read ( (char*)this->ibuff, (jpgINT16 * 3) ) ;
      if ( (ifs.gcount()) == (jpgINT16 * 3) )
      {
         //* Convert the raw (big-endian) input to a 16-bit integer *
         soiTag = this->shortConv( this->ibuff ),
         appTag = this->shortConv( &this->ibuff[jpgINT16] ) ;
         hdrLen = this->shortConv( &this->ibuff[jpgINT16 * 2] ) ;
      }

      //* If this is a valid JPEG image file, and if the application *
      //* tag is one of those supported by the application i.e.      *
      //* JPG/JFIF or JPG/EXIF, read and decode the remainder of the *
      //* file header.                                               *
      if ( (soiTag == jpgSOI) && ((appTag == jpgAPP0) || (appTag == jpgAPP1)) )
      {
         this->jfifSOI   = soiTag ; // valid Start-Of-Image tag
         this->appMarker = appTag ; // save the marker
         this->hdrSize   = hdrLen ; // save the header length

         #if DEBUGjpg != 0
         gString gsfmt ;
         gsfmt.compose( "jpgHeader::validateHeader\n"
                        "-------------------------\n"
                        "%04hXh Application Marker\n"
                        "%04hXh Header Length\n",
                        &this->appMarker, &this->hdrSize ) ;
         ofs << gsfmt.ustr() ;
         #endif   // DEBUGjpg

         //* Read the remainder of the header record.              *
         //* NOTE: The header length includes the length value     *
         //* itself,  so read two bytes less than the stated value.*
         //* Because the header length value is actually part of   *
         //* the header we place it at the top of the buffer and   *
         //* append the remaining data to it.                      *
         this->ibuff[0] = (uint8_t)(this->hdrSize >> 8) ;
         this->ibuff[1] = (uint8_t)(this->hdrSize & 0x00FF) ;
         ifs.read ( (char*)&this->ibuff[jpgINT16], (hdrLen - jpgINT16) ) ;
         if ( (ifs.gcount()) == (hdrLen - jpgINT16) )
         {
            uint16_t srcIndx = jpgINT16,  // source index
                     trgIndx = ZERO ;     // target index

            //* If file format is pure JPG/JFIF *
            if ( this->appMarker == jpgAPP0 )
            {
               //* Capture the identification string *
               while ( trgIndx < jpgJFIF_LEN )
                  this->jfifID[trgIndx++] = this->ibuff[srcIndx++] ;
               this->jfifID[trgIndx - 1] = '\0' ;// be sure string is terminated
               //* Capture the JFIF format revision number *
               this->fmtVersion = this->shortConv ( &this->ibuff[srcIndx] ) ;
               srcIndx += jpgINT16 ;
               //* Capture the image resolution units *
               this->resUnits = (jpgRU)this->ibuff[srcIndx++] ;
               //* Capture the horizontal and vertical resolution *
               this->xDensity = this->shortConv ( &this->ibuff[srcIndx] ) ;
               srcIndx += jpgINT16 ;
               this->yDensity = this->shortConv ( &this->ibuff[srcIndx] ) ;
               srcIndx += jpgINT16 ;
               //* Capture thumbnail image pixel count *
               this->thumb.width  = this->ibuff[srcIndx++] ;
               this->thumb.height = this->ibuff[srcIndx++] ;
               //* If thumbnail image lives in the first APP0 section, *
               //* find it and decode it. Image begins with jpgSOI tag.*
               if ( (this->thumb.width > ZERO) && (this->thumb.height > ZERO) )
               {
                  for ( uint16_t thIndx = srcIndx ; thIndx < this->hdrSize ; ++thIndx )
                  {
                     //* All tag codes begin with the 0xFF byte.   *
                     //* Second byte must not be 0x00.             *
                     //* Second byte of start-of-image tag == 0xD8.*
                     if ( (this->ibuff[thIndx] == 0xFF) &&
                          (this->ibuff[thIndx + 1] == 0xD8) )
                     {
                        this->thumb.offset = thIndx ;
                        this->thumb.exists = this->thumb.decodeThumbnail ( 
                                       &this->ibuff[thIndx],
                                       (this->hdrSize - thIndx), ofs ) ;
                     }
                  }
               }

               //* Scan remainder of header data from the currently indexed *
               //* position searching for additional tag codes.             *
// CZONE - MOVE THIS TO SEPARATE METHOD?
// CAPTURE INTERESTING TAG DATA (COMMENT, ETC.) TO THE 'comment[]' ARRAY
               while ( srcIndx < this->hdrSize )
               {
                  uint16_t tagcode, taglen ;
                  //* All tag codes begin with the 0xFF byte.   *
                  //* Second byte must not be 0x00.             *
                  //* Second byte of start-of-image tag == 0xD8.*
                  if ( (this->ibuff[srcIndx] == 0xFF) &&
                       (this->ibuff[srcIndx + 1] != 0x00) )
                  {
                     tagcode = this->shortConv ( &this->ibuff[srcIndx] ) ;
                     srcIndx += jpgINT16 ;
                     taglen  = this->shortConv ( &this->ibuff[srcIndx] ) ;
                     srcIndx += jpgINT16 ;

                     #if DEBUGjpg != 0
                     gsfmt.compose( "%04hX %04hX (%hu)\n", &tagcode, &taglen, &taglen ) ;
                     ofs << gsfmt.ustr() ;
                     #else
                     /* TEMP - SILENCE COMPILER WARNING */ if ( tagcode != taglen ) {}
                     #endif   // DEBUGjpg
                  }
                  else
                     ++srcIndx ;
               }

#if 0    // REFERENCE
//* Scan the remaining tags up to beginning of *
//* compressed data (0xFFDA).                  *
this->scanJfifTags ( ifs, ofs ) ;
#endif   // REFERENCE
// CZONE - THERE MAY BE ADDITIONAL (SUPPLIMENTARY) APP0 SECTIONS.
// ANY ONE OF WHICH MAY CONTAIN A THUMBNAIL IMAGE OR COMMENTS.

            }

            //* If file format is JPG with an EXIF header *
            else  // (this->appMarker == jpgAPP1)
            {
               this->exif.decodeHeader ( this->ibuff, this->appMarker, ofs ) ;
            }
            this->jpgValid = true ;    // return success
         }  // (JFIF or EXIF header read successfully)

         #if DEBUGjpg != 0
         ofs << endl ;
         #endif   // DEBUGjpg
      }
#else    // FUNCTIONAL
      ifs.read ( (char*)this->ibuff, jpgINT16 ) ;     // read the file marker
      if ( (ifs.gcount()) == jpgINT16 )               // if a good read
      {
         //* Convert the raw (big-endian) input to a 16-bit integer *
         uint16_t itmp = this->shortConv( this->ibuff ) ;

         //* Verify that this is a valid JPG/JFIF or JPG/EXIF file *
         if ( itmp == jpgSOI )
         {
            this->jfifSOI = itmp ;     // valid Start-Of-Image tag

            //* Read and decode the next marker.                  *
            //* For JPG/JFIF files this will be jpgAPP0 ( 0xFFE0).*
            //* For JPG/EXIF files this will be jpgAPP1 ( 0xFFE1).*
            //* Other proprietary application-specific markers are*
            //* possible, but we will not be able to decode them. *
            ifs.read ( (char*)this->ibuff, jpgINT16 ) ;  // read the marker
            if ( (ifs.gcount()) == jpgINT16 )            // if a good read
            {
               itmp = this->shortConv ( this->ibuff ) ;

               #if DEBUGjpg != 0
               gString gsfmt ;
               gsfmt.compose( "jpgHeader::validateHeader\n"
                              "-------------------------\n"
                              "%04hXh Application Marker\n", &itmp ) ;
               ofs << gsfmt.ustr() ;
               #endif   // DEBUGjpg

               if ( (itmp == jpgAPP0) || (itmp == jpgAPP1) )
               {
                  this->appMarker = itmp ;               // save the marker

                  //* Read the size of the header *
                  ifs.read ( (char*)this->ibuff, jpgINT16 ) ;
                  if ( (ifs.gcount()) == jpgINT16 )      // if a good read
                  {
                     //* The header length includes the length value itself, *
                     //* so read two bytes less than the stated value.       *
                     uint16_t hdrLen = this->shortConv ( this->ibuff ) ;

                     #if DEBUGjpg != 0
                     gsfmt.compose( "%04hXh Header Length\n", &hdrLen ) ;
                     ofs << gsfmt.ustr() ;
                     #endif   // DEBUGjpg

                     //* Read the remainder of the header record *
                     ifs.read ( (char*)this->ibuff, (hdrLen - jpgINT16) ) ;
                     if ( (ifs.gcount()) == (hdrLen - jpgINT16) )
                     {
                        this->hdrSize = hdrLen ;      // save the header length

                        //* Decode the header values *
                        uint16_t srcIndx = ZERO,
                                 trgIndx = ZERO ;

                        //* If file format is pure JPG/JFIF *
                        if ( this->appMarker == jpgAPP0 )
                        {
                           //* Capture the identification string *
                           while ( trgIndx < jpgJFIF_LEN )
                              this->jfifID[trgIndx++] = this->ibuff[srcIndx++] ;
                           this->jfifID[trgIndx - 1] = '\0' ;// be sure string is terminated
                           //* Capture the JFIF format revision number *
                           this->fmtVersion = this->shortConv ( &this->ibuff[srcIndx] ) ;
                           srcIndx += jpgINT16 ;
                           //* Capture the image resolution units *
                           this->resUnits = (jpgRU)this->ibuff[srcIndx++] ;
                           //* Capture the horizontal and vertical resolution *
                           this->xDensity = this->shortConv ( &this->ibuff[srcIndx] ) ;
                           srcIndx += jpgINT16 ;
                           this->yDensity = this->shortConv ( &this->ibuff[srcIndx] ) ;
                           srcIndx += jpgINT16 ;
                           //* Capture thumbnail image pixel count *
                           this->thumb.width  = this->ibuff[srcIndx++] ;
                           this->thumb.height = this->ibuff[srcIndx++] ;
                           if ( (this->thumb.width > ZERO) && (this->thumb.height > ZERO) )
                              this->thumb.exists = true ;
// CZONE - THERE MAY BE ADDITIONAL (SUPPLIMENTARY) APP0 SECTIONS.

#if 0    // CZONE - CAPTURE THUMBNAIL INFO
                           //* Read the thumbnail image data *

                           if ( this->thumb.exists )
                              this->thumb.decodeThumbnail () ;
#endif   // U/C

#if 1    // CZONE - MOVE THIS CALL TO AFTER APP0/APP1 ?
                           //* Scan the remaining tags up to beginning of *
                           //* compressed data (0xFFDA).                  *
                           this->scanJfifTags ( ifs, ofs ) ;
#endif   // U/C

                        }

                        //* If file format is JPG with an EXIF header *
                        else  // (this->appMarker == jpgAPP1)
                        {
                           this->exif.decodeHeader ( this->ibuff, this->hdrSize,
                                                     this->appMarker, ofs ) ;
                        }
                        this->jpgValid = true ;    // return success
                     }
                  }
               }
               #if DEBUGjpg != 0
               ofs << endl ;
               #endif   // DEBUGjpg
            }
         }
      }
#endif   // U/C - OBSOLETE CODE
      return this->jpgValid ;

   }  //* End validateHeader() *

   //* Scan the input stream from the current position to beginning of the   *
   //* main image data.                                                      *
   //* Input  : ifs  : pointer to the open input stream                      *
   //*          ofs  : For debugging only, access to an open output          *
   //*                 stream, 'ofs' which is used to report                 *
   //*                 intermediate results of the scan.                     *
   //*          full : if 'false', scan to beginning of image data (tag SOS) *
   //*                 if 'true',  scan to end of file (tag EOI)             *
   //*                                                                       *
   //* Returns: nothing                                                      *
   void scanJfifTags ( ifstream& ifs, ofstream& ofs, bool full )
   {
      #if DEBUGjpg != 0
      ofs << "TAG  LEN  OFFSET\n"
          << "---- ---- -------\n" ;
      #endif   // DEBUGjpg

      gString  gsfmt ;           // text formatting
      int      tagpos ;          // source-file offset
      uint16_t tagcode, taglen,  // tag name and tag length (bytes)
               tagdata,          // tag data bytes (excluding length value)
               tagi = ZERO,      // tag-string index
               lastTag = full ? jpgEOI : jpgSOS ;  // loop control (see above)

      //* Scan the remaining tags up to beginning of *
      //* compressed data Start-Of-Scan (0xFFDA).    *
      do
      {
         tagpos = ifs.tellg() ;
         ifs.read ( (char*)this->ibuff, jpgINT16 ) ;
         if ( (ifs.gcount()) == jpgINT16 )      // if a good read
         {
            tagcode = this->shortConv( this->ibuff ) ;
            if ( ((tagcode & 0xFF00) == 0xFF00) && (tagcode > 0xFF00) )
            {
               //* Read tag length *
               ifs.read ( (char*)this->ibuff, jpgINT16 ) ;
               taglen = this->shortConv( this->ibuff ) ;
               tagdata = taglen - 2 ;
               //* Read remainder of tag *
               ifs.read ( (char*)this->ibuff, (tagdata) ) ;
               #if DEBUGjpg != 0
               gsfmt.compose( "%04hX %04hX %06Xh\n", 
                              &tagcode, &taglen, &tagpos ) ;
               ofs << gsfmt.ustr() ;
               #else
               /* TEMP - SILENCE COMPILER WARNING */ if ( tagpos > ZERO ) {}
               #endif   // DEBUGjpg

               //* If this is a comment, store it now *
               if ( tagcode == jpgCOM )
               {
                  //* If there is space in the buffer *
                  if ( this->comCount < jpgCOMMAX )
                  {
                     this->ibuff[taglen - 2] = '\0' ;
                     gsfmt = (char*)this->ibuff ;
                     tagi = this->comIndx[this->comCount] ;
                     gsfmt.copy( &this->comment[tagi], 
                                 jpgTXTMAX ) ;
                     ++this->comCount ;
                     this->comIndx[this->comCount] = tagi + gsfmt.utfbytes() ;
                  }
               }

#if 0    // CZONE - SEE 'JpegEdit' VERSION OF THIS CODE SECTION.
               //* If a thumbnail image is identified, scan its contents *
               else if ( tagcode == jpgSOI )
               {
                  this->thumb.decodeThumbnail ( 
               }
#endif   // U/C
#if 1    // CZONE - MANUALLY EXTRACT START-OF-FRAME INFO
               //* Extract information from Start-Of-Frame.      *
               //* baseline == jpgSOF0 || progressive == jpgSOF2 *
               else if ( (tagcode == jpgSOF0) || (tagcode == jpgSOF2) )
               {
                  //* Data bytes s/b at least 9, (see notes above).*
                  if ( tagdata >= 9 )
                  {
//                     uint8_t  sp = (uint8_t)this->ibuff[0] ;
                     this->height = this->shortConv ( &this->ibuff[1] ) ;
                     this->width  = this->shortConv ( &this->ibuff[3] ) ;
//                     uint8_t  cf = (uint8_t)this->ibuff[5] ;
//                     uint8_t  id = (uint8_t)this->ibuff[6] ;
//                     uint8_t  hs = (uint8_t)((this->ibuff[7] & 0xF0) >> 4) ;
//                     uint8_t  vs = (uint8_t)(this->ibuff[7] & 0x0F) ;
//                     uint8_t  qu = (uint8_t)this->ibuff[8] ;

//                     #if DEBUGjpg != 0 // THIS WILL PROBABLY GO AWAY
//                     gsfmt.compose( "     %s, %hu bytes:\n"
//                                    "     %02hhX         Sample Precision\n"
//                                    "     %04hX (%3hu) Line Count\n"
//                                    "     %04hX (%3hu) Samples Per Line\n"
//                                    "     %02hhX         Components In Frame\n"
//                                    "     %02hhX         Identifier\n"
//                                    "     %02hhX         Horizontal sampling factor\n"
//                                    "     %02hhX         Vertical sampling factor\n"
//                                    "     %02hhX         Quantization table dest.\n",
//                                    (char*)(tagcode == jpgSOF0 ? "Baseline" : "Progressive"),
//                                    &tagdata, &sp, &this->height, &this->height, 
//                                    &this->width, &this->width, &cf,
//                                    &id, &hs, &vs, &qu ) ;
//                     ofs << gsfmt.ustr() ;
//                     #endif   // DEBUGjpg

// CZONE - See jpegquality.c for estimating "quality" via quantization tables.
                  }
               }
            }
#endif   // U/C
         }
         else
            break ;
      }
      while ( tagcode != lastTag ) ;
//OLD      while ( tagcode != jpgSOS ) ;

      #if DEBUGjpg != 0
      gsfmt.compose( "JFIF Comments: %hd byte total:%hd\n", 
                     &this->comCount, &this->comIndx[this->comCount] ) ;
      ofs << gsfmt.ustr() ;
      #endif   // DEBUGjpg
   }  //* End scanJfifTags() *

   //* Convert a 2-byte sequence (MSB at offset 0) to a 16-bit short integer.*
   //* Programmer's Note: This clunky construct avoids the C library's       *
   //* "helpful" automatic sign extension.                                   *
   uint16_t shortConv ( const uint8_t* srcbuf )
   {
      uint16_t i =    (uint16_t)srcbuf[1] 
                   | ((uint16_t)srcbuf[0] << 8) ;
      return i ;
   }  // shortConv()

   //** Public Data Members **
   exifHeader exif ;             // EXIF header information
   jpgThumb thumb ;              // Receives thumbnail image info if provided
   uint8_t *ibuff ;              // Pointer to input buffer
   char jfifID[jpgJFIF_LEN + 1] ;// JFIF Identifier string
   char comment[jpgCOMMAX * jpgTXTMAX] ; // Receives comment strings
   short comIndx[jpgCOMMAX] ;    // Indices for strings within 'comment' member
   short comCount ;              // Number of comment strings captured
   uint16_t jfifSOI ;            // JFIF File ID (Start-Of-Image: 0xFFD0)
   uint16_t appMarker ;          // APP0 : JFIF Application Marker (0xFFE0)
                                 // APP1 : EXIF Application Marker (0xFFE1)
   uint16_t hdrSize ;            // header length in bytes
   uint16_t fmtVersion ;         // JFIF version (usually 0x0102)
   uint16_t xDensity ;           // Horizontal resolution
   uint16_t yDensity ;           // Vertical resolution
   uint16_t width ;              // Image width (samples-per-line)
   uint16_t height ;             // Image height (line count)
   jpgRU    resUnits ;           // Resolution Units (0x00, 0x01, 0x02)
   bool     jpgValid ;           // 'true' if file header has been validated

} ;   // jpgHeader


//*************************
//*  ExtractMetadata_JPEG *
//*************************
//******************************************************************************
//* Read the media file and write the metadata to the temp file.               *
//*                                                                            *
//* Input  : ofs    : open output stream to temporary file                     *
//*          srcPath: filespec of media file to be read                        *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void FileDlg::ExtractMetadata_JPG ( ofstream& ofs, const gString& srcPath )
{
   enum exifTags : uint16_t         // Standard EXIF tag codes
   {
      extImageWidth      = 0x0100,  // Image width                   (integer)
      extImageHeight     = 0x0101,  // Image height                  (integer)
      extTitle           = 0x010E,  // Image description (title)     (string)
      extHardware        = 0x010F,  // Hardware manufacturer         (string)
      extModel           = 0x0110,  // Hardware Model                (string)
      extOrientation     = 0x0112,  // Orientation (rotation) (see list of values above)
      extResolutionX     = 0x011A,  // Resolution in X               (integer)
      extResolutionY     = 0x011B,  // Resolution in Y               (integer)
      extResUnits        = 0x0128,  // Resolution units (see list of values above)
      extSoftware        = 0x0131,  // Software used                 (string)
      extTimestamp       = 0x0132,  // Date-Time                     (string)
      extArtist          = 0x013B,  // Artist (photographer)         (string)
      extCopyright       = 0x8298,  // Copyright message             (string)
      extExifIfdOffset   = 0x8769,  // *Offset to Exif IFD Attribute array (integer)
      extGpsIfdOffset    = 0x8825,  // *Offset to GPS IFD array      (integer)
      extMfgNotes        = 0x927C,  // Manufacturer notes            (string)
      extUserComment     = 0x9286,  // Exif private tag: UserComment (UTF-8 string)
//      extJpegQuality     = 0x
   } ;
   const char *Dim = "Dimensions  : %u x %u pixels\n",
              *Tit = "Image Title : %s\n",
              *Man = "Manufacturer: %s\n",
              *Mod = "Model       : %s\n",
              *Sof = "Software    : %s\n",
              *Dat = "Timestamp   : %s\n",
              *Res = "Resolution  : X:%u/%u Y:%u/%u %s\n",
              *Ori = "Rotation    : %s degrees\n",
              *Art = "Artist      : %s\n",
              *Cop = "Copyright   : %s\n",
              *Mfg = "Mfg. Notes  : %s\n",
              *Usr = "Comment     : %s\n" ;

   //* Additional tags found in the EXIF header for JPEG files.*
   #define EXIF_VERBOSE (1)
   #if EXIF_VERBOSE != 0
   enum exifVTags : uint16_t
   {
      extvExposureTime  = 0x829A,   // Exposure time                 (rational)
      extvFnumber       = 0x829D,   // F (f-stop) number             (rational)
      extvExposureProg  = 0x8822,   // Exposure program (see list of values above)
      extvSensitivity   = 0x8827,   // Photographic sensitivity      (integer)
      extvISOspeed      = 0x8833,   // ISO 12232 speed index         (integer)
      extvVersion       = 0x9000,   // Exif version                  (4 bytes)
      extvShutterSpeed  = 0x9201,   // Shutter Speed                 (signed rational)
      extvAperture      = 0x9202,   // Lens Aperture (f/stop)        (rational)
      extvBrightness    = 0x9203,   // Brightness value              (signed rational)
      extvMaxAperture   = 0x9205,   // Max Aperture (smallest f/stop)(rational)
      extvMeteringMode  = 0x9207,   // Metering mode (see list of values above)
      extvFlash         = 0x9209,   // Flash used                    (integer bit field)
      extvFocalLength   = 0x920A,   // Focal length in millimeters   (rational)
      extvWhiteBalance  = 0xA403,   // White balance mode  (see list of values above)
      extvDigitalZoom   = 0xA404,   // Digital zoom ratio            (rational)
      extvFocalLen35mm  = 0xA405,   // Equivalent 35mm Focal length  (integer)
      extvContrast      = 0xA408,   // Contrast processing direction (see list of values above)
      extvSaturation    = 0xA409,   // Saturation processing direction (see list of values above)
      extvSharpness     = 0xA40A,   // Sharpness processing direction (see list of values above)
   } ;
   const char *vExptime = "ExposureTime: %1.4lf s  (%u/%u)\n",
              *vFnumber = "F-Number    : f/%1.1lf\n",
              *vExpProg = "ExposureProg: %s\n",
              *vPsensi  = "Sensitivity : %u\n",
              *vIsoEqu  = "ISO Speed   : %u\n",
              #if DEBUGjpg != 0
              *vVersion = "Exif Version: %s\n",
              #endif // DEBUGjpg
              *vShutter = "ShutterSpeed: %u/%u\n",
              *vAper    = "Aperture    : f/%1.1lf",
              *vBright  = "Brightness  : %+2.2lf\n",
              *vMMode   = "MeteringMode: %s\n",
              *vFocal   = "Focal Length: %2.1lfmm",
              *vWBal    = "WhiteBalance: %s\n",
              *vZoom    = "Digital Zoom: %2.3lfx\n",
              *vContr   = "Contrast    : %s\n",
              *vSatur   = "Saturation  : %s\n",
              *vSharp   = "Sharpness   : %s\n",
              *vFlash   = "Flash Used  : %s\n" ;
   #endif   // EXIF_VERBOSE

   char *ibuff = new char[KB64] ;// input buffer (64KiB dynamic allocation)
   gString gsOut, gstmp ;        // text formatting
   jpgHeader jpgHdr ;            // file header record

   //* Report basic information about the source file.*
   tnFName fn ;
   localTime lt ;
   this->fmPtr->GetFileStats ( fn, srcPath, true ) ;
   this->fmPtr->DecodeEpochTime ( fn.rawStats.st_mtime, lt ) ;
   gstmp.formatInt( fn.fBytes, 11, true ) ;
   short fni = srcPath.findlast( fSLASH ) + 1 ;
   gsOut.compose( "FILE NAME  : \"%s\"  (%s bytes)\n"
                  "FILE DATE  : %04hd-%02hd-%02hd %02hd:%02hd:%02hd\n",
                  &srcPath.ustr()[fni], gstmp.ustr(),
                  &lt.year, &lt.month, &lt.date, &lt.hours, &lt.minutes, &lt.seconds ) ;
   ofs << gsOut << endl ;

   //* Open the source file *
   ifstream ifs( srcPath.ustr(), ifstream::in ) ;
   if ( ifs.is_open() )
   {
      //* Read and validate the JPG file header *
      jpgHdr.validateHeader ( ifs, ofs ) ;

      //* If file header validated, scan for text and other interesting tags.*
      if ( jpgHdr.jpgValid )
      {
         if ( jpgHdr.appMarker == jpgAPP0 )
         {
            // Programmer's Note: The "resolution" in the JFIF header is sometimes 
            // used as an aspect ratio, e.g. "1x1 pixels", but "resolution" is more
            // accurately pixels-per-unit-measure, e.g. "96x96 dots-per-inch" or
            // simply reports width and height dimensions.
            #if DEBUGjpg == 0    // Production: display only data useful to user
            //------------------------------------------------------------------
            const char* jpgHdrTemplateA = "Resolution  : %hu x %hu %s\n" ;
            const char* jpgHdrTemplateB = "Thumbnail   : %hhu x %hhu pixels\n" ;

            gsOut.compose( jpgHdrTemplateA, &jpgHdr.xDensity, &jpgHdr.yDensity,
                           (char*)(jpgHdr.resUnits == ruPixels ? "pixels" :
                                   jpgHdr.resUnits == ruDpi ? "dots-per-inch" : 
                                   "dots-per-centimeter") ) ;
            //* If image contains a thumbnail, report it.*
            if ( (jpgHdr.width > ZERO) && (jpgHdr.height > ZERO) )
               gsOut.append( jpgHdrTemplateB, &jpgHdr.width, &jpgHdr.height ) ;
            ofs << gsOut.ustr() << endl ;
            #else                // For debugging only, spill your guts!
            const char* jpgHdrTemplate    = "JPEG File Marker  : %04hXh\n"
                                            "Application Marker: %04hXh\n"
                                            "Identifier String : %s\n"
                                            "Header Size       : %s\n"
                                            "JFIF Version      : %hhX.%hhX\n"
                                            "Image Resolution  : %hu x %hu %s\n"
                                            "Thumbnail Pixels  : %hhu x %hhu\n"
                                            "Valid Header      : %s\n"
                                            "Curr. Scan Offset : %06Xh\n" ;
            gstmp.formatInt( jpgHdr.hdrSize, 6, true ) ;
            int afterHdr = ifs.tellg() ;     // get current file position
            uint8_t maj = jpgHdr.fmtVersion >> 8,
                    min = jpgHdr.fmtVersion & 0x0F ;
            gsOut.compose( jpgHdrTemplate,
                           &jpgHdr.jfifSOI, &jpgHdr.appMarker, jpgHdr.jfifID, 
                           gstmp.ustr(), &maj, &min, 
                           &jpgHdr.xDensity, &jpgHdr.yDensity,
                           (char*)(jpgHdr.resUnits == ruPixels ? "pixels" :
                                   jpgHdr.resUnits == ruDpi ? "dots-per-inch" : 
                                   "dots-per-centimeter"),
                           &jpgHdr.thumb.width, &jpgHdr.thumb.height,
                           (char*)(jpgHdr.jpgValid ? "yes" : "no"),
                           &afterHdr ) ;
            ofs << gsOut.ustr() << endl ;
            #endif   // DEBUGjpg==0

            //****************************
            //* Report the captured data *
            //****************************

            //* Report the JPEG/JFIF comments *
            for ( short i = ZERO ; i < jpgHdr.comCount ; ++i )
            {
               gsOut.compose( Usr, &jpgHdr.comment[jpgHdr.comIndx[i]] ) ;
               ofs << gsOut.ustr() ;
            }
         }
         else if ( jpgHdr.appMarker == jpgAPP1 )
         {
            #if DEBUGjpg != 0    // For debugging only, display file header
            const char* exifHdrTemplate = 
                        "StartOfImage: %04hXh\n"
                        "App Marker  : %04hXh\n"
                        "Identifier  : %s\n"
                        "TIFF Const  : %04hXh\n"
                        "Byte Order  : %s\n"
                        "Header Size : %s\n"
                        "IFD Offset  : %u\n"
                        "Thumb Exists: %s\n" ;
            const char* exifThumbTemplate = 
                        "Thumb Offset: %04Xh\n"
                        "Thumb Length: %u bytes\n"
                        "Thumb Width : %hu pixels\n"
                        "Thumb Height: %hu pixels\n" ;
            const char* typeName[11] = { "(unused)",
                                         "byte value ",
                                         "string  ",
                                         "uint16_t ",
                                         "uint32_t ",
                                         "uint32_t ufraction ",
                                         "(unused)",
                                         "uint8_t  ",
                                         "(unused)",
                                         "int32_t  ",
                                         "int32_t  sfraction "
                                       } ;

            gstmp.formatInt( jpgHdr.exif.hdrSize, 6, true ) ;
            gsOut.compose( exifHdrTemplate,
                           &jpgHdr.jfifSOI, &jpgHdr.exif.appMarker,
                           jpgHdr.exif.exifID, &jpgHdr.exif.tiffConst, 
                           (jpgHdr.exif.leByteOrder ? "little-endian" : "big-endian"),
                           gstmp.ustr(), &jpgHdr.exif.ifdOffset,
                           (jpgHdr.exif.thumb.exists ? "yes" : "no")
                         ) ;
            if ( jpgHdr.exif.thumb.exists )
               gsOut.append( exifThumbTemplate,
                             &jpgHdr.exif.thumb.offset, &jpgHdr.exif.thumb.bytes,
                             &jpgHdr.exif.thumb.width, &jpgHdr.exif.thumb.height ) ;
            ofs << gsOut.ustr() << endl ;

            //* Display the raw data of the EXIF IFD array *
            gsOut = "INDX TAG_ TYPE COUNT    OFFSET\n"
                    "---- ---- ---- -------- --------" ;
            ofs << gsOut.ustr() << endl ;

            for ( int16_t i = ZERO ; i < jpgHdr.exif.ifdEntries ; ++i )
            {
               gsOut.compose( "[%2hu] %04hX %04hX %08X %08X %s", &i,
                              &jpgHdr.exif.ifd[i].tag, &jpgHdr.exif.ifd[i].typ,
                              &jpgHdr.exif.ifd[i].cnt, &jpgHdr.exif.ifd[i].off,
                              typeName[jpgHdr.exif.ifd[i].typ]
                           ) ;

               if ( jpgHdr.exif.ifd[i].typ == 2 )           // string
                  gsOut.append( " \"%s\"\n", jpgHdr.exif.ifd[i].str ) ;
               else if ( (jpgHdr.exif.ifd[i].typ == 5) ||
                         (jpgHdr.exif.ifd[i].typ == 10) )   // ratioal value
                  gsOut.append( "(%u / %u)\n", &jpgHdr.exif.ifd[i].fracNum, &jpgHdr.exif.ifd[i].fracDen ) ;
               else                                         // integer
                  gsOut.append( "(%u)\n", &jpgHdr.exif.ifd[i].val32 ) ;
               ofs << gsOut.ustr() ;
            }
            ofs << endl ;
            #endif   // DEBUGjpg

            //* Format and display the interesting data *
            //* of the EXIF IFD array.                  *
            const short MAX_UCOM = 32 ;   // maximum number of user comments reported
            double   focalLen = 0.0 ;
            uint32_t dimx = ZERO, dimy = ZERO, 
                     resxN = ZERO, resxD = ZERO, 
                     resyN = ZERO, resyD = ZERO, orient = ZERO, 
                     units = 2,     // default: 2==dots-per-inch
                     focalLen35 = ZERO ; // equivalent focal length for 35mm film camera
            const char *man  = NULL,
                       *mod  = NULL,
                       *dat  = NULL,
                       *tit  = NULL,
                       *sof  = NULL,
                       *art  = NULL,
                       *cop  = NULL,
                       *mfg  = NULL,
                       *usr[MAX_UCOM] ;
            short usrIndx = ZERO ;        // index of next free element of usr[] array
            for ( short i = ZERO ; i < MAX_UCOM ; ++i )  // initialize comment pointers
               usr[i] = NULL ;

            for ( int16_t i = ZERO ; i < jpgHdr.exif.ifdEntries ; ++i )
            {
               if ( jpgHdr.exif.ifd[i].tag == extImageWidth )
                  dimx = jpgHdr.exif.ifd[i].val32 ;
               else if ( jpgHdr.exif.ifd[i].tag == extImageHeight )
                  dimy = jpgHdr.exif.ifd[i].val32 ;
               else if ( jpgHdr.exif.ifd[i].tag == extResUnits )
                  units = jpgHdr.exif.ifd[i].val32 ;
               else if ( jpgHdr.exif.ifd[i].tag == extTitle )
                  tit = jpgHdr.exif.ifd[i].str ;
               else if ( jpgHdr.exif.ifd[i].tag == extHardware )
                  man  = jpgHdr.exif.ifd[i].str ;
               else if ( jpgHdr.exif.ifd[i].tag == extModel )
                  mod  = jpgHdr.exif.ifd[i].str ;
               else if ( jpgHdr.exif.ifd[i].tag == extSoftware )
                  sof  = jpgHdr.exif.ifd[i].str ;
               else if ( jpgHdr.exif.ifd[i].tag == extResolutionX )
               {
                  resxN = jpgHdr.exif.ifd[i].fracNum ;
                  resxD = jpgHdr.exif.ifd[i].fracDen ;
               }
               else if ( jpgHdr.exif.ifd[i].tag == extResolutionY )
               {
                  resyN = jpgHdr.exif.ifd[i].fracNum ;
                  resyD = jpgHdr.exif.ifd[i].fracDen ;
               }
               else if ( jpgHdr.exif.ifd[i].tag == extOrientation )
                  orient = jpgHdr.exif.ifd[i].val32 ;
               else if ( jpgHdr.exif.ifd[i].tag == extTimestamp )
                  dat  = jpgHdr.exif.ifd[i].str ;
               else if ( jpgHdr.exif.ifd[i].tag == extArtist )
                  art = jpgHdr.exif.ifd[i].str ;
               else if ( jpgHdr.exif.ifd[i].tag == extCopyright )
                  cop = jpgHdr.exif.ifd[i].str ;
               else if ( jpgHdr.exif.ifd[i].tag == extMfgNotes )
                  mfg = jpgHdr.exif.ifd[i].str ;
               else if ( jpgHdr.exif.ifd[i].tag == extUserComment )
               {
                  if ( usrIndx < MAX_UCOM )
                     usr[usrIndx++] = jpgHdr.exif.ifd[i].str ;
               }
            }  // for(;;)

            //* Report the captured data to the log file *
            if ( dimx != ZERO || dimy != ZERO )
            {
               gsOut.compose( Dim, &dimx, &dimy ) ;
               ofs << gsOut.ustr() ;
            }
            if ( tit != NULL )
            { gsOut.compose( Tit, tit ) ; ofs << gsOut.ustr() ; }
            if ( man != NULL )
            { gsOut.compose( Man, man ) ; ofs << gsOut.ustr() ; }
            if ( mod != NULL )
            { gsOut.compose( Mod, mod ) ; ofs << gsOut.ustr() ; }
            if ( sof != NULL )
            { gsOut.compose( Sof, sof ) ; ofs << gsOut.ustr() ; }
            if ( dat != NULL )
            { gsOut.compose( Dat, dat ) ; ofs << gsOut.ustr() ; }
            if ( resxN != ZERO || resyN != ZERO )
            {
               gsOut.compose( Res, &resxN, &resxD, &resyN, &resyD, 
                              (units == 3 ? "dp-cm" : "dpi") ) ;
               ofs << gsOut.ustr() ;
            }
            if ( orient != ZERO )
            {
               gsOut.compose( Ori, (orient == 1 || orient == 2) ? "0" :
                                   (orient == 5 || orient == 8) ? "-90" :
                                   (orient == 6 || orient == 7) ? "+90" : "+180" ) ;
               ofs << gsOut.ustr() ;
            }
            if ( art != NULL )
            { gsOut.compose( Art, art ) ; ofs << gsOut.ustr() ; }
            if ( cop != NULL )
            { gsOut.compose( Cop, cop ) ; ofs << gsOut.ustr() ; }
            if ( mfg != NULL )
            { gsOut.compose( Mfg, mfg ) ; ofs << gsOut.ustr() ; }
            for ( short i = ZERO ; (usr[i] != NULL) && (i < MAX_UCOM) ; ++i )
            {
               gsOut.compose( Usr, usr[i] ) ;
               ofs << gsOut.ustr() ;
            }

            #if EXIF_VERBOSE != 0
            double aperCurr = 0.0, aperMax = 0.0,  // aperture && max aperture
                   ratVal ;                        // generic rational value
            bool w ;                               // 'true' if record to be written
            for ( int16_t i = ZERO ; i < jpgHdr.exif.ifdEntries ; ++i )
            {
               w = true ;
               if ( jpgHdr.exif.ifd[i].tag == extvExposureTime )
               {
                  double ratVal = (double)jpgHdr.exif.ifd[i].fracNum / (double)jpgHdr.exif.ifd[i].fracDen ;
                  gsOut.compose( vExptime, &ratVal, &jpgHdr.exif.ifd[i].fracNum, 
                                 &jpgHdr.exif.ifd[i].fracDen ) ;
               }
               else if ( jpgHdr.exif.ifd[i].tag == extvFnumber )
               {
                  double fnum = (double)jpgHdr.exif.ifd[i].fracNum / (double)jpgHdr.exif.ifd[i].fracDen ;
                  gsOut.compose( vFnumber, &fnum ) ;
               }
               else if ( jpgHdr.exif.ifd[i].tag == extvExposureProg )
               {
                  const char* progName[8] = 
                  {
                     "manual", "normal", "aperture priority", "shutter priority",
                     "creative (depth)", "action (speed)", "portrait mode", 
                     "landscape mode"
                  } ;
                  uint16_t prog = (uint16_t)jpgHdr.exif.ifd[i].val32 ;
                  if ( prog < 1 || prog > 8 )
                     prog = 2 ;
                  --prog ;    // convert to index
                  gsOut.compose( vExpProg, progName[prog] ) ;
               }
               else if ( jpgHdr.exif.ifd[i].tag == extvSensitivity )
               { gsOut.compose( vPsensi, &jpgHdr.exif.ifd[i].val32 ) ; }
               else if ( jpgHdr.exif.ifd[i].tag == extvISOspeed )
               { gsOut.compose( vIsoEqu, &jpgHdr.exif.ifd[i].val32 ) ; }
               #if DEBUGjpg != 0
               else if ( jpgHdr.exif.ifd[i].tag == extvVersion )
               {  //* This is four(4) ASCII-NUMERIC BYTES formatted as a string *
                  gsOut.compose( vVersion, jpgHdr.exif.ifd[i].str ) ;
               }
               #endif // DEBUGjpg
               else if ( jpgHdr.exif.ifd[i].tag == extvShutterSpeed )
               {
                  gsOut.compose( vShutter, &jpgHdr.exif.ifd[i].fracNum,
                                 &jpgHdr.exif.ifd[i].fracDen ) ;
               }
               else if ( jpgHdr.exif.ifd[i].tag == extvAperture )
               {
                  aperCurr = (double)jpgHdr.exif.ifd[i].fracNum / (double)jpgHdr.exif.ifd[i].fracDen ;
                  w = false ;
               }
               else if ( jpgHdr.exif.ifd[i].tag == extvMaxAperture )
               {
                  aperMax = (double)jpgHdr.exif.ifd[i].fracNum / (double)jpgHdr.exif.ifd[i].fracDen ;
                  w = false ;
               }
               else if ( jpgHdr.exif.ifd[i].tag == extvBrightness )
               {  //* Note that brightness is a SIGNED rational value.*
                  int sNum = int(jpgHdr.exif.ifd[i].fracNum),
                      sDen = int(jpgHdr.exif.ifd[i].fracDen) ;
                  ratVal = (double)sNum / (double)sDen ;
                  gsOut.compose( vBright, &ratVal ) ;
               }
               else if ( jpgHdr.exif.ifd[i].tag == extvMeteringMode )
               {
                  const char* mMode[8] = 
                  {
                     "unknown", "Average", "Center Weight", "Spot",
                     "Multi-spot", "Pattern", "Partial", "other"
                  } ;
                  uint16_t mm = uint16_t(jpgHdr.exif.ifd[i].val32) ;
                  if ( mm >= 255 )  mm = 7 ;    // convert to index
                  gsOut.compose( vMMode, mMode[mm] ) ;
               }
               else if ( jpgHdr.exif.ifd[i].tag == extvFlash )
               {
                  bool flashFired = bool(jpgHdr.exif.ifd[i].val32 & 0x0001) ;
                  gsOut.compose( vFlash,  flashFired ? "yes" : "no" ) ;
               }
               else if ( jpgHdr.exif.ifd[i].tag == extvFocalLength )
               {
                  focalLen = (double)jpgHdr.exif.ifd[i].fracNum / 
                             (double)jpgHdr.exif.ifd[i].fracDen ;
                  w = false ;
               }
               else if ( jpgHdr.exif.ifd[i].tag == extvFocalLen35mm )
               {
                  focalLen35 = jpgHdr.exif.ifd[i].val32 ;
                  w = false ;
               }
               else if ( jpgHdr.exif.ifd[i].tag == extvWhiteBalance )
               {
                  gsOut.compose( vWBal, (jpgHdr.exif.ifd[i].val32 == 1 ?
                                 "Manual" : "Auto") ) ;
               }
               else if ( jpgHdr.exif.ifd[i].tag == extvDigitalZoom )
               {
                  ratVal = (double)jpgHdr.exif.ifd[i].fracNum / 
                           (double)jpgHdr.exif.ifd[i].fracDen ;
                  gsOut.compose( vZoom, &ratVal ) ;
               }
               else if ( jpgHdr.exif.ifd[i].tag == extvContrast )
               {
                  const char* contType[3] = { "normal", "soft", "hard" } ;
                  uint16_t contrast = uint16_t(jpgHdr.exif.ifd[i].val32) ;
                  if ( contrast > 2 )  contrast = ZERO ; // convert to index
                  gsOut.compose( vContr, contType[contrast] ) ;
               }
               else if ( jpgHdr.exif.ifd[i].tag == extvSaturation )
               {
                  const char* saturType[3] = { "normal", "low", "high" } ;
                  uint16_t saturation = uint16_t(jpgHdr.exif.ifd[i].val32) ;
                  if ( saturation > 2 )  saturation = ZERO ; // convert to index
                  gsOut.compose( vSatur, saturType[saturation] ) ;
               }
               else if ( jpgHdr.exif.ifd[i].tag == extvSharpness )
               { 
                  const char* sharpType[3] = { "normal", "soft", "hard" } ;
                  uint16_t sharpness = uint16_t(jpgHdr.exif.ifd[i].val32) ;
                  if ( sharpness > 2 )  sharpness = ZERO ; // convert to index
                  gsOut.compose( vSharp, sharpType[sharpness] ) ;
               }
               else
                  w = false ;

               if ( w )
                  ofs << gsOut.ustr() ;
            }
            if ( aperCurr > 0.0 )
            {
               gsOut.compose( vAper, &aperCurr ) ;
               if ( aperMax > 0.0 )
                  gsOut.append( "  (f/%1.1lf min)", &aperMax ) ;
               gsOut.append( L'\n' ) ;
               ofs << gsOut.ustr() ;
            }
            if ( focalLen > 0.0 )
            {
               gsOut.compose( vFocal, &focalLen ) ;
               if ( focalLen35 > ZERO )
                  gsOut.append( " (35mm equivalent: %umm)", &focalLen35 ) ;
               gsOut.append( L'\n' ) ;
               ofs << gsOut.ustr() ;
            }
            if ( ! jpgHdr.exif.gps.validData )
               ofs << endl ;
            #endif   // EXIF_VERBOSE

            //* If GPS data were captured, report it now.*
            if ( jpgHdr.exif.gps.validData )
            {
               // Programmer's Note: minutes may be either a whole number or a
               // decimal value. Format minutes and seconds accordingly.

               //* Report latitude (if not available, zeros will be reported) *
               gsOut.compose( "Latitude    : %3.0lf° ", &jpgHdr.exif.gps.latDeg ) ;
               if ( jpgHdr.exif.gps.latSec != 0.0 )
               {
                  gsOut.append( "%2.0lfm %2.4lfs %c\n",
                                &jpgHdr.exif.gps.latMin, &jpgHdr.exif.gps.latSec, 
                                &jpgHdr.exif.gps.latRef ) ;
               }
               else
               {
                  gsOut.append( "%2.4lfm %c\n",
                                &jpgHdr.exif.gps.latMin, &jpgHdr.exif.gps.latRef ) ;
               }
               ofs << gsOut.ustr() ;

               //* Report longitude (if not available, zeros will be reported) *
               gsOut.compose( "Longitude   : %3.0lf° ", &jpgHdr.exif.gps.lonDeg ) ;
               if ( jpgHdr.exif.gps.lonSec != 0.0 )
               {
                  gsOut.append( "%2.0lfm %2.4lfs %c\n",
                                &jpgHdr.exif.gps.lonMin, &jpgHdr.exif.gps.lonSec, 
                                &jpgHdr.exif.gps.lonRef ) ;
               }
               else
               {
                  gsOut.append( "%2.4lfm %c\n",
                                &jpgHdr.exif.gps.lonMin, &jpgHdr.exif.gps.lonRef ) ;
               }
               ofs << gsOut.ustr() ;

               if ( *jpgHdr.exif.gps.areaName != '\0' )
               {
                  gsOut.compose( "GPS Location: %s\n", jpgHdr.exif.gps.areaName ) ;
                  ofs << gsOut.ustr() ;
               }

               if ( jpgHdr.exif.gps.altCap )
               {
                  gsOut.compose( "GPS Altitude: %-3.2lf meters\n", &jpgHdr.exif.gps.altitude ) ;
                  ofs << gsOut.ustr() ;
               }

               //* Report datestamp/timestamp *
               if ( (*jpgHdr.exif.gps.dstamp != '\0') || (*jpgHdr.exif.gps.tstamp != '\0') )
               {
                  gsOut = "GPS time UTC: " ;
                  if ( *jpgHdr.exif.gps.dstamp != '\0' )
                     gsOut.append( "%s ", jpgHdr.exif.gps.dstamp ) ;
                  if ( *jpgHdr.exif.gps.tstamp != '\0' )
                     gsOut.append( "%s", jpgHdr.exif.gps.tstamp ) ;
                  ofs << gsOut.ustr() << '\n' ;
               }

               ofs << endl ;
            }
         }
         ofs << endl ;

         //* If a thumbnail image is present, report it *
         if ( jpgHdr.exif.thumb.exists || jpgHdr.thumb.exists )
         {
            if ( ! jpgHdr.thumb.exists && jpgHdr.exif.thumb.exists )
               jpgHdr.thumb = jpgHdr.exif.thumb ;
// CZONE - REPORT THUMBNAIL INFO.
// CZONE - jpgHdr.thumb.exists.
         }

         //* Scan the actual JPEG data for interesting data *
#if 1    // CZONE - ACTUAL JPEG-DATA SCAN



#endif   // U/C
      }  // (jpgHdr.jpgValid)
      #if DEBUGjpg != 0
      else        // not a valid PNG file format
         ofs << "Error: Invalid JPG format.\n" << endl ;
      #endif   // DEBUGjpg

      ifs.close() ;           // close the source file
   }

   delete [] ibuff ;             // release dynamic allocation
   ibuff = NULL ;

}  //* End ExtractMetadata_JPG() *

//*************************
//*                       *
//*************************
//********************************************************************************
//*                                                                              *
//*                                                                              *
//*                                                                              *
//* Input  :                                                                     *
//*                                                                              *
//* Returns:                                                                     *
//********************************************************************************

