//********************************************************************************
//* File       : FileDlgDelete.cpp                                               *
//* Author     : Mahlon R. Smith                                                 *
//*              Copyright (c) 2005-2024 Mahlon R. Smith, The Software Samurai   *
//*                  GNU GPL copyright notice located in FileMangler.hpp         *
//* Date       : 24-Jul-2020                                                     *
//* Version    : (see FileDlgVersion string in FileDlg.cpp)                      *
//*                                                                              *
//* Description:                                                                 *
//* This module contains methods related to the file delete operation of the     *
//* FileDlg class.                                                               *
//*                                                                              *
//*                                                                              *
//* Developed using GNU G++ (Gcc v: 4.4.2)                                       *
//*  under Fedora Release 12, Kernel Linux 2.6.31.5-127.fc12.i686.PAE and above  *
//********************************************************************************
//* Version History (most recent first):                                         *
//*   See version history in FileDlg.cpp.                                        *
//********************************************************************************
//* Programmer's Notes:                                                          *
//*                                                                              *
//*                                                                              *
//********************************************************************************

#include "GlobalDef.hpp"
#include "FileDlg.hpp"

//******************************
//* Local definitions and data *
//******************************

//* Warn user that the 'Delete' operation cannot be undone                     *
const char* deleteForever[] = 
{ //1234567890123456789012345678901234567890123456789012 - (max line length)
   "  WARNING!  WARNING!  ",
   " ",
   "          This operation cannot be undone!",
   "   These files WILL NOT be placed in the Trashcan,",
   "   and the deleted files will be lost forever.",
   " ",
   "      Are you sure that you want to continue?",
   NULL
} ;

//* Message text defined in FileDlgPrompt.cpp *
extern const char* fmtypeString[] ;
extern const char* fmtypeName[] ;
extern const char* readonlyFileSys[] ;
extern const char* notOntoSelf[] ;
extern const char* cannotOverrideWP[] ;
extern const char* noLinkOverwrite[] ;
extern const char* differentFileTypes[] ;
extern const char* cannotDeleteSpecial[] ;
extern const char* cannotOverwriteFifo[] ;
extern const char* cannotOverwriteDirectory[] ;
extern const char* warnSocketMod[] ;
extern const char* warnBdevCdevMod[] ;
extern const char* warnUnsuppMod[] ;
extern const char* fdNotImplemented[] ;


//***********************
//* DeleteSelectedFiles *
//***********************
//******************************************************************************
//* Delete all 'selected' files in the current window.                         *
//* This is the public-access method for deleting files.                       *
//* (See also TrashcanSelectedFiles().)                                        *
//*                                                                            *
//* Input  : selected: (by reference, initial value ignored)                   *
//*            On return, contains number of files in selection list           *
//*                                                                            *
//* Returns: number of files deleted                                           *
//*           If caller's 'selected' parameter == return value,                *
//*           then no errors                                                   *
//******************************************************************************

UINT FileDlg::DeleteSelectedFiles ( UINT& selected )
{
   UINT  deletedFiles = ZERO ;      // number of deleted file (return value)

   //* User must have write permission on source directory in order to         *
   //* modify the source files. This is verified during selection.             *
   //* Visually mark 'selected' files for processing.                          *
   //* Re-display file list, update stats control, and move highlight to       *
   //* first 'selected' file in list. Returns ZERO if no files selected.       *
   bool srcdirWPerm ;
   selected = this->MarkSelection ( this->deleteColor, srcdirWPerm ) ;

   //* Warn user that this operation cannot be undone *
   bool yesDelete = this->DecisionDialog ( deleteForever ) ;

   if ( (srcdirWPerm != false) && (selected > ZERO) && (yesDelete != false) )
   {
      //* Place selected files on clipboard *
      this->FillClipboard ( opDELETE ) ;

      //* Test whether user has write access on all the source files to be     *
      //* deleted. If the number of write-protected files > ZERO, then user is *
      //* asked whether we should proceed. If yes, attempt is made to enable   *
      //* write access for all protected files in the list. Value returned     *
      //* will then be number of files which REMAIN write-protected.           *
      UINT  wpCount = ZERO,   // number of write-protected files
            rpCount = ZERO ;  // number of read-protected files
      bool overrideSrcWProt = this->SourcePermission ( wpCount, rpCount, true ) ;
      if ( overrideSrcWProt )
      {  //* We now have write access to all source files OR *
         //* permission to do as much damage as possible.    *

         this->ProcessCWD () ;      // display the default processing message

         UINT  markedFiles, dirIndex = ZERO, fileIndex = ZERO ;
         for ( markedFiles = this->selInfo.Count ; markedFiles ; markedFiles-- )
         {
            //* For directory or directory tree *
            if ( this->deList[this->hIndex]->fType == fmDIR_TYPE )
            {
               //* (a little defensive programming never hurts) *
               if ( clipBoard[BNODE].dirFiles > dirIndex 
                    && clipBoard[BNODE].nextLevel != NULL )
               {
                  TreeNode* tnPtr = &clipBoard[BNODE].nextLevel[dirIndex] ;
                  if ( tnPtr != NULL )
                  {  // count of files NOT deleted (not used at this level)
                     UINT errCount = ZERO ;
                     deletedFiles += 
                        this->dsfDeleteDirTree ( tnPtr, clipBoard_srcPath, errCount ) ;
                  }
               }
               ++dirIndex ;
            }
            //* For non-directory files at top level *
            else
            {
               deletedFiles += 
                  this->dsfDeleteFileList ( &clipBoard[BNODE].tnFiles[fileIndex], 
                                            1, clipBoard_srcPath ) ;
               ++fileIndex ;
            }
            this->NextSelectedItem () ;   // track to next 'selected' item
         }  // for(;;)

         this->ProcessCWD ( true ) ;   // clear the processing message
      }
      else
      {  //* User has aborted the operation because one or more files are      *
         //* write-protected.                                                  *
         this->SetErrorCode ( ecSRCPROTECT, EACCES ) ;
      }
      //* Re-read directory contents (clipboard will be cleared) *
      this->RefreshCurrDir () ;
   }
   else                             // no files processed
   {
      this->DeselectFile ( true ) ; // deselect all files
      if ( ! srcdirWPerm )          // no write permission on this directory
      {
         this->DisplayStatus ( ecREADONLY_S, EACCES ) ;
      }
   }
   return deletedFiles ;

}  //* End DeleteSelectedFiles () *

//***********************
//*  dsfDeleteDirTree   *
//***********************
//******************************************************************************
//* Delete the specified directory tree.                                       *
//* Recursively walk through the TreeNode list, deleting all files.            *
//*           Private method. Called through DeleteSelectedFiles().            *
//*                                                                            *
//* First, for each subdirectory within the specified node:                    *
//*   1) delete contents of the subdirectory.                                  *
//*   2) Then, if the subdirectory is empty, delete the subdirectory itself.   *
//* Next, delete the non-directory files within the specified directory.       *
//* Last, if no subdirectories or files remain, delete the specified directory.*
//*                                                                            *
//* NOTE: No Directory-type files will be deleted unless the contents of that  *
//*       directory have already been successfully deleted.                    *
//*                                                                            *
//* Input  : tnPtr: pointer to a TreeNode class object which describes the     *
//*            directory file and its contents.                                *
//*          sPath: pointer to source path string                              *
//*          errCount: (by reference, initial value ignored)                   *
//*            on return, contains number of files NOT deleted                 *
//*            (if errCount==ZERO, then all files successfully deleted)        *
//*                                                                            *
//* Returns: number of files successfully deleted                              *
//******************************************************************************

UINT FileDlg::dsfDeleteDirTree ( TreeNode* tnPtr, const char* sPath, UINT& errCount )
{
   UINT  deletedFiles = ZERO, // return value: number of files deleted
      eCount = ZERO ;         // count of file-deletion errors at this level

   gString  sBuff ;
   this->fmPtr->CatPathFname ( sBuff, sPath, tnPtr->dirStat.fName ) ;

   //* If there are subdirectories within this directory *
   if ( tnPtr->dirFiles > ZERO && tnPtr->nextLevel != NULL )
   {
      //* For each subdirectory . . . *
      for ( UINT i = ZERO ; i < tnPtr->dirFiles ; i++ )
      {
         //* Delete contents of the subdirectory *
         deletedFiles += this->dsfDeleteDirTree ( &tnPtr->nextLevel[i], sBuff.ustr(), eCount ) ;
      }
   }

   //* Delete non-directory files at this level *
   if ( tnPtr->tnFCount > ZERO && tnPtr->tnFiles != NULL )
   {
      UINT  dFiles ;
      dFiles = this->dsfDeleteFileList ( tnPtr->tnFiles, tnPtr->tnFCount, sBuff.ustr() ) ;
      if ( dFiles != tnPtr->tnFCount )  // update accumulator for un-deleted files
         eCount += tnPtr->tnFCount - dFiles ;
      deletedFiles += dFiles ;
   }

   //* If all contents of the directory have been deleted, now delete directory*
   if ( eCount == ZERO )
   {
      if ( (this->dsfDeleteDirectory ( tnPtr->dirStat, sBuff )) == OK )
         ++deletedFiles ;
      else
         ++eCount ;
   }
   errCount += eCount ;    // update caller's count of un-deleted files
   return deletedFiles ;

}  //* End dsfDeleteDirTree() *

//***********************
//*  dsfDeleteFileList  *
//***********************
//******************************************************************************
//* Delete the list of (non-directory) files.                                  *
//*           Private method. Called through DeleteSelectedFiles()             *
//*                     or dsfDeleteDirTree().                                 *
//*                                                                            *
//* Input  : tnf   : pointer to an array of tnFName class objects              *
//*          fCount: number of elements in array                               *
//*          sPath : pointer to source path string                             *
//*                                                                            *
//* Returns: number of files successfully deleted                              *
//******************************************************************************

UINT FileDlg::dsfDeleteFileList ( const tnFName* tnf, UINT fileCount, const char* sPath )
{
   gString delPath ;
   UINT  deletedFiles = ZERO ;
   short status ;

   for ( UINT i = ZERO ; i < fileCount ; i++ )
   {
      this->fmPtr->CatPathFname ( delPath, sPath, tnf[i].fName ) ;
      switch ( tnf[i].fType )
      {
         case fmREG_TYPE:     // regular file
         case fmLINK_TYPE:    // symbolic link file
         case fmFIFO_TYPE:    // FIFO file
            status = this->dsfDeleteRegFile ( tnf[i], delPath ) ;
            break ;
         case fmCHDEV_TYPE:   // character device file
            status = this->dsfDeleteChdevFile ( tnf[i], delPath ) ;
            break ;
         case fmBKDEV_TYPE:   // block device file
            status = this->dsfDeleteBkdevFile ( tnf[i], delPath ) ;
            break ;
         case fmSOCK_TYPE:    // socket file
            status = this->dsfDeleteSocketFile ( tnf[i], delPath ) ;
            break ;
         case fmUNKNOWN_TYPE: // unknown file type
            status = this->dsfDeleteUnsuppFile ( tnf[i], delPath ) ;
         default:
            status = ecUNSUPP ;
            break ;
      }
      if ( status == OK )
         ++deletedFiles ;
   }
   return deletedFiles ;

}  //* End dsfDeleteFileList() *

//***********************
//*  dsfDeleteRegFile   *
//***********************
//******************************************************************************
//* Delete the specified file. Call this method to delete file of these types: *
//*   regular file (fmREG_TYPE)                                                *
//*   symbolic link file (fmLINK_TYPE)                                         *
//*     (note that link/shortcut file, NOT link target will be deleted)        *
//*   FIFO file (fmFIFO_TYPE)                                                  *
//*                                                                            *
//* Note: It is assumed that tnf->writeAcc has been correctly initialized,     *
//*       and we WILL NOT attempt to override write protection here.           *
//*                                                                            *
//* Input  : tnf   : (by reference) tnFName class object describing the file   *
//*          sPath : (by reference) source path/filename                       *
//*                                                                            *
//* Returns: OK if file successfully deleted                                   *
//*          else: FMgr error code if operation fails                          *
//*                ecFTYPE         if incorrect file type                      *
//*                ecTARGPROTECT   if file is write-protected                  *
//******************************************************************************

short FileDlg::dsfDeleteRegFile ( const tnFName& tnf, const gString& sPath )
{
   short status ;       // return value

   if ( tnf.writeAcc && 
        (tnf.fType == fmREG_TYPE || tnf.fType == fmLINK_TYPE || 
         (!clipBoard_gvfsSrc && tnf.fType == fmFIFO_TYPE)) )
   {
      if ( clipBoard_gvfsSrc )
         status = this->fmPtr->DeleteFile_gvfs ( sPath.ustr() ) ;
      else
         status = this->fmPtr->DeleteFile ( sPath.ustr() ) ;
   }
   else if ( ! tnf.writeAcc )
      status = ecTARGPROTECT ;
   else
      status = ecFTYPE ;

   return status ;

}  //* End dsfDeleteRegFile() *

//***********************
//* dsfDeleteDirectory  *
//***********************
//******************************************************************************
//* Delete the specified file.                                                 *
//* Call this method to delete files of 'directory' type, but ONLY if the      *
//*   specified directory is empty.                                            *
//*                                                                            *
//* Input  : tnf  : (by reference) tnFName class object describing the file    *
//*          sPath: (by reference) source path/filename                        *
//*                                                                            *
//* Returns: OK if file successfully deleted                                   *
//*          else: FMgr error code if operation fails                          *
//*                ecFTYPE         if incorrect file type                      *
//*                ecTARGPROTECT   if file is write-protected                  *
//*                ecINUSE         if directory in use by another process      *
//*                ecPACCV         if directory is not empty OR general error  *
//******************************************************************************

short FileDlg::dsfDeleteDirectory ( const tnFName& tnf, const gString& sPath )
{
   short status = ecFTYPE ;   // return value (initialize to "not-a-directory")

   if ( tnf.fType == fmDIR_TYPE )         // must be of directory type
   {
      if ( tnf.writeAcc != false )        // user must have write permission
      {
         if ( clipBoard_gvfsSrc )
         {
            if ( (this->fmPtr->isEmptyDir_gvfs ( sPath.ustr() )) )
               status = this->fmPtr->DeleteEmptyDir_gvfs ( sPath.ustr() ) ;
            else  // (this should never happen because caller should check for this)
               status = ecPACCV ;
         }
         else
         {
            if ( (this->fmPtr->isEmptyDir ( sPath.ustr() )) )
               status = this->fmPtr->DeleteEmptyDir ( sPath.ustr() ) ;
            else  // (this should never happen because caller should check for this)
               status = ecPACCV ;
         }

         if ( status != OK )
         {
            //* Error deleting directory file *
            if ( status == EBUSY )     // in use
            {
               this->DisplayStatus ( ecINUSE, EBUSY ) ;
               status = ecINUSE ;
            }
            else           // not empty after all, or general access error
            {
               this->DisplayStatus ( ecPACCV ) ;
               status = ecPACCV ;
            }
         }
      }
      else
         status = ecTARGPROTECT ;
   }
   return status ;

}  //* End dsfDeleteDirectory() *

//***********************
//* dsfDeleteChdevFile  *
//***********************
//******************************************************************************
//* Delete the specified file.                                                 *
//* Call this method to delete files of 'Character Device' type.               *
//*                                                                            *
//* Input  : tnf  : (by reference) tnFName class object describing the file    *
//*          sPath: (by reference) source path/filename                        *
//*                                                                            *
//* Returns: OK if file successfully deleted                                   *
//*          else: FMgr error code if operation fails                          *
//*                ecFTYPE         if incorrect file type                      *
//*                ecTARGPROTECT   if file is write-protected                  *
//*                ecUNSUPP        if operation not supported for file type.   *
//******************************************************************************

short FileDlg::dsfDeleteChdevFile ( const tnFName& tnf, const gString& sPath )
{
#if 0    // NOT CURRENTLY SUPPORTED
   return ecUNSUPP ;
#else    // USER WARNING
   return ( this->CopyOrDeleteNotSupported ( tnf.fType, tnf.fName ) ) ;
#endif   // USER WARNING

}  //* End dsfDeleteChdevFile() *

//***********************
//* dsfDeleteBkdevFile  *
//***********************
//******************************************************************************
//* Delete the specified file.                                                 *
//* Call this method to delete files of 'Block Device' type.                   *
//*                                                                            *
//* Input  : tnf  : (by reference) tnFName class object describing the file    *
//*          sPath: (by reference) source path/filename                        *
//*                                                                            *
//* Returns: OK if file successfully deleted                                   *
//*          else: FMgr error code if operation fails                          *
//*                ecFTYPE         if incorrect file type                      *
//*                ecTARGPROTECT   if file is write-protected                  *
//*                ecUNSUPP        if operation not supported for file type.   *
//******************************************************************************

short FileDlg::dsfDeleteBkdevFile ( const tnFName& tnf, const gString& sPath )
{
#if 0    // NOT CURRENTLY SUPPORTED
   return ecUNSUPP ;
#else    // USER WARNING
   return ( this->CopyOrDeleteNotSupported ( tnf.fType, tnf.fName ) ) ;
#endif   // USER WARNING

}  //* End dsfDeleteBkdevFile() *

//***********************
//* dsfDeleteSocketFile *
//***********************
//******************************************************************************
//* Delete the specified file.                                                 *
//* Call this method to delete files of 'Socket' type.                         *
//*                                                                            *
//* Input  : tnf  : (by reference) tnFName class object describing the file    *
//*          sPath: (by reference) source path/filename                        *
//*                                                                            *
//* Returns: OK if file successfully deleted                                   *
//*          else: FMgr error code if operation fails                          *
//*                ecFTYPE         if incorrect file type                      *
//*                ecTARGPROTECT   if file is write-protected                  *
//*                ecUNSUPP        if operation not supported for file type.   *
//******************************************************************************

short FileDlg::dsfDeleteSocketFile ( const tnFName& tnf, const gString& sPath )
{
#if 0    // NOT CURRENTLY SUPPORTED
   return ecUNSUPP ;
#else    // USER WARNING
   return ( this->CopyOrDeleteNotSupported ( tnf.fType, tnf.fName ) ) ;
#endif   // USER WARNING

}  //* End dsfDeleteSocketFile() *

//***********************
//* dsfDeleteUnsuppFile *
//***********************
//******************************************************************************
//* Delete the specified file.                                                 *
//* Call this method to delete files of 'Socket' type.                         *
//*                                                                            *
//* Input  : tnf  : (by reference) tnFName class object describing the file    *
//*          sPath: (by reference) source path/filename                        *
//*                                                                            *
//* Returns: OK if file successfully deleted                                   *
//*          else: FMgr error code if operation fails                          *
//*                ecFTYPE         if incorrect file type                      *
//*                ecTARGPROTECT   if file is write-protected                  *
//*                ecUNSUPP        if operation not supported for file type.   *
//******************************************************************************

short FileDlg::dsfDeleteUnsuppFile ( const tnFName& tnf, const gString& sPath )
{
short status = ecUNSUPP ;

   gString gs ;
   gs.compose( L" %s ", tnf.fName ) ;
   const char* formattedFName = gs.ustr() ;
   const char* msg[] = 
   {
      " Delete File of Unknown Type ",
      "The following file:",
      formattedFName,
      "is of an unknown or unsupported file type. Deleting",
      "files of this type is not currently supported.",
      NULL,
   } ;
   attr_t msgAttr[] = { (this->cs.sd | ncbATTR), this->cs.sd, (this->cs.sd & ~ncrATTR), 
                        this->cs.sd, this->cs.sd, this->cs.sd, } ;
   this->InfoDialog ( (const char**)msg, msgAttr ) ;

   return status ;

}  //* End dsfDeleteUnsuppFile() *

//***********************
//*                     *
//***********************
//******************************************************************************
//*                                                                            *
//*                                                                            *
//*                                                                            *
//* Input  :                                                                   *
//*                                                                            *
//* Returns:                                                                   *
//******************************************************************************


