//********************************************************************************
//* File       : FileDlg.hpp                                                     *
//* Author     : Mahlon R. Smith                                                 *
//*              Copyright (c) 2005-2025 Mahlon R. Smith, The Software Samurai   *
//*                  GNU GPL copyright notice located in FileMangler.hpp         *
//* Date       : 09-Jul-2025                                                     *
//* Version    : (see FileDlgVersion string in FileDlg.cpp)                      *
//*                                                                              *
//* Description: This class implements a file-browsing window using the          *
//* NcDialog class functionality.                                                *
//*                                                                              *
//* The File data are displayed through a dctSCROLLEXT control embedded in an    *
//* instance of the NcDialog class. This NcDialog class instance and the         *
//* control objects it contains have been instantiated at a higher               *
//* (application) level. The methods and data members of this NcDialog class     *
//* object are accessed through a pointer to the class instance which is         *
//* passed as a parameter to the FileDlg class constructor.                      *
//*                                                                              *
//* Note that the FileDlg class instantiates the FMgr class and controls all     *
//* access to its methods and data members. The application has no direct        *
//* access to the methods and data of the FMgr class.                            *
//*                                                                              *
//*                                                                              *
//* Developed using GNU G++ (Gcc v: 4.4.2)                                       *
//*  under Fedora Release 12, Kernel Linux 2.6.31.5-127.fc12.i686.PAE            *
//********************************************************************************
//* Version History (most recent first):                                         *
//*   See version history in FileDlg.cpp                                         *
//********************************************************************************
//* Programmer's Notes:                                                          *
//*                                                                              *
//********************************************************************************

#ifndef FILEDLG_INCLUDED
#define FILEDLG_INCLUDED
#include "FMgr.hpp"           // low-level file manipulation
#include "Progbar.hpp"        // Progbar class definition
#include "Pinwheel.hpp"       // Pinwheel class definition
#include "renamePattern.hpp"  // special code for batch rename


//* Define as non-zero to enable debugging methods *
#define ENABLE_DEBUGGING_CODE (0)

//* Enable/disable Linux special characters in filenames.    *
//* Linux/UNIX filesystems accept the "special" characters   *
//* but other filesystems (VFAT, NTFS) may not support them. *
//* See enum tbChars in the NcDialog API.                    *
//* File operations (copy, move, rename, delete, etc. which  *
//* access shell utilities must be aware of these characters.*
#define LINUX_SPECIAL_CHARS (1)
//* Enable/disable request for user verification when using  *
//* Linux "special" characters. (enabled by default)         *
#define LINUX_SPECIAL_QUERY (1)

//* Operating mode *
enum WinMode : short { wmSINGLE_WIN, wmDUAL_WIN, wmTERM_WIN } ;

//* Options for confirming overwrite of one file by another of the same name.*
//* (Directories are never overwritten. Write-protected files are overwritten*
//*  only if overwrite is confirmed.                                         *
enum owTypes : short
{
   owNO_CONFIRM,           //* Never confirm file overwrite (except when target write-protected)
   owCONFIRM_NEWER,        //* Confirm when overwriting a newer or same-date file (default) 
   owCONFIRM_ALL           //* Always confirm on file overwrite
} ;

//* Options for copying symbolic link files *
enum slcOptions : short
{
   slcCOPY_LINK,           //* Copy the symbolic link, itself
   slcCOPY_TARGET_L,       //* Copy the link target file using name of symbolic Link
   slcCOPY_TARGET_T,       //* Copy the link target file using name of link Target
   slcUSER_PROMPT          //* Prompt user for the type of copy to perform (default)
} ;

//* Status values returned by user interaction methods during rename operations*
enum renameOption : short
{
   promptRENAME,  //* User wants to be prompted for a name for each file processed
   batchRENAME,   //* User has entered a pattern for batch rename
   abortRENAME,   //* User has cancelled the rename operation
   noRENAME,      //* For copy/move operations, use sourcefile name as target name
} ;

//* Formatting options for ViewFileContents() method *
enum viewFormat : short 
{ vfPrompt, vfText, vfHexByte, vfHexWord, 
  vfDecimal, vfOctal, vfBinary, vfSymlink, vfArchive, vfMetadata
} ;

//* Supported archive types for view/create/update/expand *
//* operations.                                           *
//* - Note that the corresponding archive utility must be *
//*   correctly installed on the system.                  *
//* - See the ArchiveTarget() method, below.              *
//* - Supported archives use either 'tar' (with optional  *
//*   compression) or 'zip'.                              *
enum arcType : short
{
   atNONE = -1,      // not a recognized archive type
   atTAR,            // 'tar' archive (uncompressed)
   atGZIP,           // 'gzip' archive
   atBZIP2,          // 'bzip2' archive
   atZIP,            // 'zip' archive (zip/unzip utility, not tar)
   atEPUB,           // 'epub' archive (zip utility)
   atLZIP,           // 'lzip' archive
   atLZOP,           // 'lzop' archive
   atXZ,             // 'xz' archive
   atLZMA,           // 'lzma' archive (requires 'xz' utility)
   atODOC,           // OpenDocument (XML-based, zip archive) file (view/expand only)
   atOXML,           // OpenXML (OOXML - Microsloth Office) zip archive file (view/expand only)
} ;

//* For rcdScan(), identifies calling method. *
enum rcdsCaller : short { rcdsRefresh, rcdsChild, rcdsParent, rcdsSet } ;

//* List of filetypes supported for decoding of file metadata                  *
enum MediafileType : short { mftMP3, mftOGG, mftM4A, mftASF, mftPNG, mftJPG, 
                             mftNONE } ;

//* Interface language options. (see also alStrings[]) *
enum AppLang : short
{ 
  enLang = ZERO,  // English
  esLang,         // Español (Spanish)
  zhLang,         // Zhōngwén (中文) (Chinese, simplified)
//  viLang,         // Tiếng Việt (Vietnamese)
  locLang         // language taken from locale if supported, else default language
} ;
//* For LTR (left-to-right) languages, reset. *
//* For RTL (right-to-left) languages, set.   *
const bool AppDirection[] =
{
   false,         // English,   LTR
   false,         // Espanol,   LTR
   false,         // Zhongwen,  LTR
   false,         // TiengViet, LTR
} ;

//* Color Scheme. Provides standardized colors for various display objects, so *
//* user can select desired background color on startup, and the dialog's      *
//* foreground objects and sub-dialogs will display in complementary colors.   *
class ColorScheme
{
   public:
   NcBaseColors scheme ;   // color scheme indicator
   attr_t bb ;             // dialog backgrounds and borders
   attr_t sd ;             // sub-dialog color
   attr_t sb ;             // sub-dialog Bold color
   attr_t em ;             // sub-dialog Emphasis color
   attr_t wr ;             // sub-dialog Warning color
   attr_t mn ;             // menu color (non-focus)
   attr_t mf ;             // menu color (with focus)
   attr_t pn ;             // pushbutton color (non-focus)
   attr_t pf ;             // pushbutton color (with focus)
   attr_t tn ;             // text box color (non-focus)
   attr_t tf ;             // text box color (with focus)
} ;

//* Options for launching external programs using the "Open (execute)" item    *
//* of the ViewFile context menu. See ConfigOptions::lCode.                    *
enum LaunchCodes : short
{
   lcAutoLaunch = ZERO,    // automatically determine launch parameters if possible,
                           // else, prompt user for launch parameters
   lcAutoAudio,            // 'lcAutoLaunch' PLUS audio files
   lcSafeLaunch,           // always launch from a new terminal window (no prompt)
   lcManualLaunch,         // always prompt user for launch parameters
} ;

//* Configuration Options structure *
class ConfigOptions
{
   public:

   //* Default constructor *
   ConfigOptions ( void )
   {
      this->Reset ( true ) ;
   }

   void Reset ( bool all = false )
   {
      this->winMode        = wmTERM_WIN ;    // mode based on size of terminal window
      this->sortOption     = fmNAME_SORT ;   // sort by name (ascending)
      this->overWrite      = owCONFIRM_NEWER;// confirm if target date >= source date
      this->cScheme.scheme = ncbcCOLORS ;    // application default scheme
                                             // (individual colors initialized later)
      this->lCode          = lcAutoLaunch ;  // external program auto-launch (if possible)
      this->appLang        = enLang ;        // default to English interface
      this->showHidden     = false ;         // do not display 'hidden' files
      this->confirmDelete  = false ;         // do not confirm when moving files to trashcan
      this->caseSensitive  = true ;          // case-sensitive filename sort
      this->enableMouse    = false ;         // mouse support disabled
      if ( all )
      {
         *this->appPath    = NULLCHAR ;      // will be initialized during startup
         *this->tfPath     = NULLCHAR ;      // will be initialized during startup
         *this->trPath     = NULLCHAR ;      // will be initialized during startup
         *this->bkPath     = NULLCHAR ;      // will be initialized during startup
      }
   }

   //* Assignment operator: copy one instance to another *
   void operator = ( const ConfigOptions& coSrc )
   {
      this->winMode        = coSrc.winMode ;
      this->sortOption     = coSrc.sortOption ;
      this->overWrite      = coSrc.overWrite ;
      this->cScheme        = coSrc.cScheme ;
      this->showHidden     = coSrc.showHidden ;
      this->confirmDelete  = coSrc.confirmDelete ;
      this->caseSensitive  = coSrc.caseSensitive ;
      this->enableMouse    = coSrc.enableMouse ;
      this->lCode          = coSrc.lCode ;
      this->appLang        = coSrc.appLang ;
      for ( short i = ZERO ; i < MAX_PATH ; i++ )
      {
         this->appPath[i] = coSrc.appPath[i] ;
         if ( this->appPath[i] == NULLCHAR )
            break ;
      }
      for ( short i = ZERO ; i < MAX_PATH ; i++ )
      {
         this->tfPath[i] = coSrc.tfPath[i] ;
         if ( this->tfPath[i] == NULLCHAR )
            break ;
      }
      for ( short i = ZERO ; i < MAX_PATH ; i++ )
      {
         this->trPath[i] = coSrc.trPath[i] ;
         if ( this->trPath[i] == NULLCHAR )
            break ;
      }
      for ( short i = ZERO ; i < MAX_PATH ; i++ )
      {
         this->bkPath[i] = coSrc.bkPath[i] ;
         if ( this->bkPath[i] == NULLCHAR )
            break ;
      }
   }

   //* DATA MEMBERS (public) *
   WinMode     winMode  ;  //* Indicates Single- or Dual-Window operation mode
   char appPath[MAX_PATH] ;   //* Directory of application executable
                              //* config binary, default config file, and Help live here
   char tfPath[MAX_PATH] ;    //* Path specification for creating temporary files
   char trPath[MAX_PATH] ;    //* Path specification for desktop trashcan directory
   char bkPath[MAX_PATH] ;    //* Path specification for backup definition file
   fmSort      sortOption ;   //* Sort option for file display
   owTypes     overWrite ;    //* Level of confirmation when overwriting files
   ColorScheme cScheme ;      //* Dialog color scheme (dialog border color)
   LaunchCodes lCode ;        //* External-program launch option
   AppLang     appLang ;      //* Language used for application interface
   bool        showHidden ;   //* If true, display hidden files
   bool        confirmDelete ;//* If true, confirm file deletion (to trash)
   bool        caseSensitive ;//* If true, case sensitive name sort
   bool        enableMouse ;  //* If true, enable mouse support
} ;

const short minfitROWS = 14 ;          // minimum (inside) rows for the ScrollExt control
const short minfitCOLS = 57 ;          // minimum (inside) columns for the ScrollExt control
                                       // (see MIN_DUALWINDOW_WIDTH)
const short DATE_STRING_LEN = (21) ;   // length of formatted date/time string (incl. null)
const short INFODLG_WIDTH = (55) ;     // width of the InfoDialog window (see below)
const short INFODLG_MAX_MSGS = (8) ;   // max message lines for the InfoDialog window (see below)

//* Subdirectory names living within desktop Trashcan. *
//* Base directory specified in 'trashDir' member.     *
const char* const trashFiles = "files" ;
const char* const trashInfo  = "info" ;

//* For Paste Special command, the operation sub-type selected by user. *
enum psSubType : short 
{
   psstSimpleCopy,      // copy/move source file to target
   psstCreateSymLink,   // create a symbolic link to source file
   psstCreateHardLink,  // create a 'hard' link to source file
   psstCopyLinkTarget,  // copy the sym-link target file
   psstABORT            // abort the operation
} ;

//* Returned from user prompts to confirm override of file protection or *
//* to confirm overwrite of an existing target file.                     *
enum corReturn : short 
{ 
   corProcessOnce,   // override processing rules (if necessary) for this file only
   corProcessOnceWP, // same as corProcessOnce except that existing target's
                     // write protection must be removed before continuing
   corProcessAll,    // silently override processing rules for remaining 
                     // files in processing list
   corSkipOnce,      // do not process this file (processing rules remain in place)
   corSkipAll,       // silently skip remaining files in list if processing them 
                     // would violate the processing rules
   corNoAccess,      // cannot obtain write access to an existing target file
   corAbort          // abort - do not process any remaining files in processing list
} ;
   
//* Operation-pending flag for data held on clipboard *
enum OpPend : short { opNONE, opCUT, opCOPY, opDELETE, opRENAME, opTOUCH, opPROTECT } ;

//* FileDlg context-help codes. Indicates context of call to CallContextHelp().*
enum contextCode : short 
{
   cxtMAIN = ZERO,         // top menu of help file
   cxtMOD_STATS,           // 'Modify File Stats' sub-dialog
   cxtARCHIVE_C,           // 'Archive_Prompt' sub-dialog (create)
   cxtARCHIVE_U,           // 'Archive_Prompt' sub-dialog (update)
   cxtARCHIVE_E,           // 'Archive_Prompt' sub-dialog (expand)
   cxtTREE_VIEW,           // 'dftDisplayTree' sub-dialog
} ;

//* Supported terminal-emulator types for opening a new terminal window.   *
//* gnome-terminal is the default. Other term emulators may be added later.*
enum EmuType : int { gnomeType = ZERO, konsoleType, xtermType } ;

//* Parameters for defining a child terminal window  *
//* and the command(s) to be executed in that window.*
static const int enSIZE = 64 ; // buffer size (in bytes) for emulator name, etc.
class DC_Emulator
{
   public:
   ~DC_Emulator ( void ) {}               // destructor
   DC_Emulator ( void )                   // default constructor
   { this->reset() ; }

   //* Initialization constructor *
   DC_Emulator ( const gString& tfspec, const gString& pfspec, 
                 const gString& pfargs, bool autoClose )
   {
      this->reset() ;
      tfspec.copy( this->tmpFile, gsDFLTBYTES ) ;
      pfspec.copy( this->prgFile, gsDFLTBYTES ) ;
      pfargs.copy( this->argList, gsDFLTBYTES ) ;
      this->autoReturn = autoClose ;
   }

   void reset ( void )
   {
      *this->emuName = 
      *this->shlName = 
      *this->geoData = 
      *this->argList = 
      *this->prgFile = 
      *this->tmpFile = 
      *this->cmdData = NULLCHAR ;
      this->emuType  = gnomeType ;
      this->xpos  = this->ypos  = this->cols  = this->rows  = 
      this->pxoff = this->pyoff = this->pxsiz = this->pysiz = 
      this->pcols = this->prows = ZERO ;
      this->autoReturn = false ;
   }

   //* Determine the size and position for a new terminal-emulator window.*
   //* This method is located in FileDlgContext.cpp.                      *
   EmuType DefineChildEmulator ( short c = (-1), short r = (-1), 
                                 short x = (-1), short y = (-1) ) ;

   //* Spawn a new process and open previously-defined terminal window.   *
   //* This method is located in FileDlgContext.cpp.                      *
   pid_t OpenTermwin ( void ) ;

   //******************
   //** Data Members **
   //******************
   char  emuName[enSIZE] ;          // terminal emulator name
   char  shlName[enSIZE] ;          // shell program name
   char  geoData[enSIZE] ;          // geometry for new terminal window
   char  argList[gsDFLTBYTES] ;     // command-line arguments for program to be executed
   char  prgFile[gsDFLTBYTES] ;     // name (or path) of program to be executed
   char  tmpFile[gsDFLTBYTES] ;     // temporary filename for system data capture
   char  cmdData[gsDFLTBYTES * 2] ; // constructed command
   EmuType emuType ;                // terminal-emulator code
   int   xpos ;                     // child window position (in pixels)
   int   ypos ;
   int   cols ;                     // child window size (columns and rows)
   int   rows ;
   int   pxoff ;                    // parent window position (in pixels)
   int   pyoff ;
   int   pxsiz ;                    // parent window size (in pixels)
   int   pysiz ;
   int   pcols ;                    // parent window size (columns and rows)
   int   prows ;
   bool  autoReturn ;               // if 'true' close child window immediately
                                    // after executing the list of commands
} ;   // DC_Emulator

//* FileDlg class error codes. Error message strings corresponding to these *
//* codes are located in the 'GetErrorCode' method. Keep them synchronized. *
//*                 (see also errorCode class below)                        *
const short FIRST_ERRORCODE = 1000 ;   // Lowest value for FileDlg error code
enum fdErrCode : short
{ 
   ecNOERROR = 0,             // Error-free operation, Huzzah!
   ecMEMALLOC=FIRST_ERRORCODE,// Memory allocation error 
   ecPACCV,                   // Possible (non-specific) access violation 
   ecFINFO,                   // Error accessing file-system data 
   ecFILESELF,                // Cannot copy a file onto itself
   ecNOWRITE,                 // Can't write target,source unchanged
   ecPERMISSION,              // Insufficient permission for file
   ecDIRNOREAD,               // No read access for source directory
   ecFTYPE,                   // Incorrect file type for operation
   ecINUSE,                   // Directory or file is in use
   ecUNSUPP,                  // Operation unsupported for file type
   ecTARGPROTECT,             // Existing target file is write protected
   ecTARGEXISTS,              // File of target name already exists
   ecSRCPROTECT,              // Source write protected, not deleted
   ecLINK,                    // Unable to create symbolic link
   ecHARDLINK,                // Unable to create hard link
   ecREADONLY_S,              // Source directory set as read-only
   ecREADONLY_T,              // Target directory set as read-only
   ecTARGDIR,                 // Unable to read target directory
   ecLINKOVER,                // Cannot overwrite file with sym link
   ecRECURSECOPY,             // Recursive copy operation failed
   ecNOTRASHCAN,              // Move-to-trashcan operation disabled
   ecAPPERROR,                // Application error (should not happen)
   ecNOTIMPL,                 // Requested operation is not yet implemented
   ecUNKERROR,                // Unknown error
   ecERRCODES
} ;

//* Error code and description returned by GetErrorCode() *
// NOTE: Descriptions of the system error codes from 'errno' are taken from 
//       /usr/include/asm/errno.h (system dependent)
class errorCode
{
   public:
   errorCode ()
   {
      fdErrorCode  = ecNOERROR ;
      sysErrorCode = ZERO ;
      fdErrorDesc = sysErrorDesc = NULL ;
   }
   fdErrCode fdErrorCode ;          // FileDlg class error code
   const char*  fdErrorDesc ;       // Pointer to FileDLg error description string
   short sysErrorCode ;             // System error code from 'errno'
   const char*  sysErrorDesc ;      // Pointer to 'errno' description string
} ;

class FileDlg ;   //* Dummy declaration

//* Initialization structure passed to the FileDlg constructor *
class InitFileDlg
{
   public:
   NcDialog* dPtr ;        // Pointer to the NcDialog object which contains the
                           // controls described here
   InitCtrl statusInit ;   // Copy of data used to instantiate the dctTEXTBOX
                           // used to display status messages
   short    statusIndex ;  // Index number of dctTEXTBOX control for status msgs

   InitCtrl fdispInit ;    // Copy of data used to instantiate the dctSCROLLEXT 
                           // control. File data are displayed in this control
   short    fdispIndex ;   // Index number of dctSCROLLEXT control for file diaplay

   InitCtrl pathInit ;     // Copy of data used to instantiate the dctTEXTBOX
                           // used to display directory path
   short    pathIndex ;    // Index number of dctTEXTBOX control for dir path

   InitCtrl statsInit ;    // Copy of data used to instantiate the dctTEXTBOX
                           // used to display file-operation statistics messages
   short    statsIndex ;   // Index number of dctTEXTBOX control for stats msgs
   bool     rootScan ;     // If 'true', enable full scan from root directory
   bool     refresh ;      // If 'true', then read/display directory contents
   const char* startPath ; // Path of directory in which to begin
   ConfigOptions cfgInit ; // Caller's configuration options for this class
} ;

//* File Selection: description of 'selected' files in window *
class fileSelect
{
   public:
   UINT64   Size ;      //* Total byte size of 'selected' files - top level only
   UINT64   subSize ;   //* Total byte size in 'selected' subdirectories
   UINT     Count ;     //* Count of 'selected' files - top level only
   UINT     subCount ;  //* Count of files in 'selected' subdirectories
} ;

//* IMPORTANT NOTE: The global clipBoard' group of variables are declared in   *
//* FileDlg.cpp using:                                                         *
//* On first instantiation of the FileDlg class, the top-level (base node)     *
//* TreeNode class is also instantiated to hold the grand totals for all files *
//* on the clipboard:  clipBoard = new TreeNode[1] ;                           *
//* Access the summary-data-node thus:  clipBoard[BNODE].xxxx                  * 
//* This node is deleted when the FileDlg class is deleted. (It is assumed     *
//* that all instances of the FileDlg class will be deleted together. If this  *
//* changes in the future, it will be necessary to either track which instance *
//* of FileDlg created the base node OR call ClearClipboard() immediately after*
//* the delete of one of the FileDlg instances to re-allocate the base node.)  *
extern TreeNode*  clipBoard ;
extern char       clipBoard_srcPath[] ;
extern UINT       clipBoard_dirTrees ;
extern UINT       clipBoard_topFiles ;
extern OpPend     clipBoard_opPending ;
extern bool       clipBoard_gvfsSrc ;
extern bool       clipBoard_gvfsTrg ;


//* Construct for visually representing the directory tree *
//* and interacting with the user to traverse the tree.    *
const short dftITEM_LEN = 1024 ;    // width of a display item string
class dftDItem
{
   public:
   dftDItem ( void )
   { *this->n = NULLCHAR ;  this->p = this->l = ZERO ; }

   wchar_t n[dftITEM_LEN] ;   // name string
   UINT    p ;                // index of parent directory's entry
   UINT    l ;                // level of item in directory tree
} ;
class dftDisplay
{
   public:
   ~dftDisplay ( void )          // destructor
   { if ( this->data != NULL ) { delete [] this->data ; this->data = NULL ; } }

   //* Constructors *
   dftDisplay ( void ) {}
   dftDisplay ( UINT dItems, short dRows, short dCols, attr_t iCol, attr_t hCol )
   {
      this->total  = dItems ;
      this->rows   = dRows ;
      this->cols   = dCols ;              // (note: signed short int)
      this->iColor = iCol ;
      this->hColor = hCol ;
      this->data   = new dftDItem[this->total] ;
      this->reset() ;                     // all other members to default values
   }

   void realloc ( UINT dItems )
   {
      if ( this->data != NULL )
      {
         delete [] this->data ;
         this->data = NULL ;
         this->total = ZERO ;
      }
      if ( dItems > ZERO )
      {
         for ( short i = 3 ; i > ZERO ; --i )
         {
            if ( (this->data = new (std::nothrow) dftDItem[this->total = dItems]) != NULL )
               break ;
         }
         // Programmer's Note: If this allocation fails three times, the system 
         // has bigger problems than an application crash.
      }
      this->reset() ;                     // reset indices
   }

   void reset ( void )
   {
      this->first = this->curr = ZERO ;                  // (unsigned int)
      this->last  = (this->total < this->rows ? this->total : this->rows) ;
      if ( this->last > ZERO ) { --this->last ; }
      this->lshift = this->width = this->levels = ZERO ; // (signed short int)
   }

   dftDItem* data ;     // pointer to two-dimensional array of character data
   UINT first ;         // index of first displayed item
   UINT last ;          // index of last displayed item
   UINT curr ;          // index of highlighted item
   UINT total ;         // total items in display list
   UINT rows ;          // number of display rows
   short cols ;         // number of display columns
   short lshift ;       // number of columns data are left-shifted
   short width ;        // number of columns in widest display string
   short levels ;       // number of directory levels captured (0-based)
   attr_t iColor ;      // display-item color
   attr_t hColor ;      // highlighted-display-item color
} ;

//* Parameters passed among the private methods for cut/copy/paste operations  *
class pclParm
{
   public:
   //* Default constructor *
   pclParm (void )
   {
      srcTree = (TreeNode*)NULL ;
      psOption = psstSimpleCopy ;
      attCount = cpyCount = delCount = ZERO ;
      this->deleteSrc = false ;
   }
   //* Initialization constructor *
   pclParm ( const char* s,         // source base directory
             const char* t,         // target base directory
             TreeNode* n = NULL,    // anchor node for source data
             psSubType y = psstSimpleCopy, // paste-special sub-type
             bool c = false,        // 'cut' operation
             UINT a = ZERO,         // files attempted
             UINT p = ZERO,         // files copied
             UINT d = ZERO,         // files deleted after copy
             const char* tm = NULL )// timestamp (for move-to-trash only)
   {
      this->srcDir = s ; this->trgDir = t ; this->srcTree = n ;
      this->psOption = y ; this->deleteSrc = c ; this->attCount = a ; 
      this->cpyCount = p ; this->delCount = d ;
      this->tStamp = tm ; 
   }

   gString     srcDir ;       // source directory path
   gString     trgDir ;       // target directory path
   gString     trgIDir ;      // target info directory path (Trashcan log directory)
   gString     tStamp ;       // date-timestamp for Trashcan operation log
   TreeNode*   srcTree ;      // pointer to anchor node of source subdirectory tree
   psSubType   psOption ;     // only for 'Paste-Special': flavour of specialness
   UINT        attCount ;     // number of copy/move/create operations ATTempted
   UINT        cpyCount ;     // number of source files copied to target
   UINT        delCount ;     // number of source files DELeted after copy
   bool        deleteSrc ;    // 'true' if delete-after-copy operation in progress
} ;

//* Parameters passed among the private methods for Trashcan operations  *
class tcDataInfo
{
   public:
   char   srcPath[MAX_PATH] ; // path/filename of item in 'Trash/files' directory
   char   trgPath[MAX_PATH] ; // path/filename of item in its original location
   char   infoPath[MAX_PATH] ;// path/filename of '.trashinfo' file in 'Trash/info'
   localTime tDate ;          // datestamp for when item was trashed
   fmFType fType ;            // file type of source item (dir or non-dir)
   UINT   files ;             // if 'fType' == fmDIR_TYPE, number of files item
                              // contains (including the item name)
   UINT64 size ;              // size of item in 'Trash/files' directory
} ;

//* Record for Trashcan scan (summary) *
class tcSummaryInfo
{
   public:
   void reset ( void )
   {
      this->items = this->files = this->iFiles = ZERO ;
      this->bytes = this->tSpace = this->fSpace = ZERO ;
   }

   UINT   items ;          // number of top-level items
   UINT   files ;          // number of data files
   UINT   iFiles ;         // number of information records
   UINT64 bytes ;          // total size of data files
   UINT64 tSpace ;         // total combined size of data and info files on disk
   UINT64 fSpace ;         // free disk space (bytes)
} ;

//* Record for Trashcan scan (item detail)  *
class tcDetailInfo
{
   public:
   gString srcPath ;    // path/filename of item in 'Trash/files' directory
   gString trgPath ;    // path/filename of item in its original location
   gString infoPath ;   // path/filename of '.trashinfo' file in 'Trash/info'
   gString dRecord ;    // data record formatted for output (132 columns, optimal)
   #if 1    // for use with FileMangler
   tnFName  dfs ;
   #else    // for use with cTrash
   DFileStats dfs ;     // stat data for srcPath
   #endif   // for use with cTrash
   localTime tDate ;    // datestamp for when item was trashed
   fmFType fType ;      // file type of source item (dir or non-dir)
   UINT   files ;       // total number of files in the item (non-directory==1)
                        // if 'fType' == fmDIR_TYPE, number of files item
                        // contains (including the item name)
   UINT64 size ;        // total size of item in 'Trash/files' directory
   bool select ;        // 'true' if item is selected for restore or delete
} ;

//* Parameters for Backup and Synchronization operations.*
//* -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  - *
enum bsetType : short
{  //* These options are used for Backup and Synch operations *
   bsetBackup, bsetSynch, bsetArchive,
   //* These are sub-options for Archive operations only *
   bsetACreate, bsetAUpdate, bsetAExpand
} ;
//* Formatting code for Backup/Synch log file.*
//* - - - - - - - - - - - - - - - - - - - - - *
enum bsetLog : short
{
   blVERBOSE = ZERO, // display an entry for each source file scanned
   blUPDATE,         // display each attempted update (successes and errors)
   blERROR,          // display only update errors
   blSUMMARY,        // display only operational summary
   blTSTAMP,         // for debugging only, display timestamps and access flags
                     // for each update attempted
   blNONE            // do not create a log file (currently not used)
} ;
//* Exclusion record: identifies a filename extension or other grouping to be  *
//* excluded from the Backup operation. The string is compared to the end of   *
//* the source filespec (just before the NULL terminator). If a match, then    *
//* that source file is excluded. Examples: '.o'  '~'  '.bak'                  *
//* Does not apply to creating archive files or to Synchronization operations. *
class bsetExcl
{
   public:
   wchar_t  excl[MAX_PATH] ;  // string to be compared agains filespec
   short    len ;             // number of characters in string (incl. NULLCHAR)
} ;
//* Defines a backup operation. Used to format, verify and pass data among     *
//* the methods of the backup/archive/synch group.                             *
const short zipDEFAULT = 6 ;  // zip 'deflate' default compression level
class BSet 
{
   public:

   ~BSet ( void )             // destructor
   { if ( this->bex != NULL ) { delete [] this->bex ; this->bex = NULL ; } }
   BSet ( void ) { this->bex = NULL ; this->reset() ; }   // constructor
   void reset ( bool all = true )   // reset all (or nearly all) data members
   {
      if ( all )
      { this->bsrc.clear() ; this->btrg.clear() ; }
      this->barc.clear() ; blog.clear() ;
      this->bex_reset() ;
      this->btime.reset() ;
      this->ftotal = this->fupdated = 
      this->fskipped = this->ferrors = ZERO ;
      this->uaFiles = ZERO ;
      this->uaBytes = this->uaTarget = ZERO ;
      this->bhour = this->bmin = ZERO ;
      this->btype = bsetBackup ;
      this->blfmt = blVERBOSE ;
      this->zipLevel = zipDEFAULT ; // zip 'deflate' compression level
      this->slSupport = true ;      // assume target supports symlinks
      this->scanOnly = false ;      // operation fully enabled
      this->blflag = false ;        // no auto-save of log file
   }
   void bex_reset ( void )
   {
      if ( this->bex != NULL )
         delete [] this->bex ;
      this->bex = NULL ;
      this->bexCount = ZERO ;
   }
   gString     bsrc ;         // path/filename for source directory
   gString     btrg ;         // path/filename for target directory
   gString     barc ;         // filespec for target archive file
   gString     blog ;         // path/filename for operational logfile
   bsetExcl*   bex ;          // pointer to list of exclusions
   UINT        bexCount ;     // number of records in bex array
   UINT        ftotal ;       // total number of source files
   UINT        fupdated ;     // number of files updated
   UINT        fskipped ;     // number of files requiring no update
   UINT        ferrors ;      // number of files skipped due to access or other error
   UINT        uaFiles ;      // for archive update/expand:
                              // (original file count, or target files written)
   UINT64      uaBytes ;      // for archive update/expand:
                              // (original byte count, or target bytes written)
   UINT64      uaTarget ;     // for archive update/expand:
                              // (for update: original target size)
                              // (for expand: total bytes for expanded files)
   localTime   btime ;        // system-local time for operation to begin
   int         bhour ;        // transport for hour-spinner value
   int         bmin ;         // transport for minute-spinner value
   bsetType    btype ;        // type of backup operation
   bsetLog     blfmt ;        // format for operational log
   short       zipLevel ;     // zip archives only: 'deflate' level (0,3,6,9)
   bool        slSupport ;    // 'true' if target filesystem supports symbolic links
   bool        scanOnly ;     // 'true' if src/trg scanned, but updates not performed
   bool        blflag ;       // 'true' if logfile to be saved
} ;

//* Operations in progress will regularly update the members of this class.    *
//* The Progress Bar thread then monitors the progress to produce a visual     *
//* progress indicator. See the pbManager() method for more information.       *
//* NOTE: The methods of this class are implemented in FileDlgBackup.cpp.      *
static const UINT64 PB_THRESHOLD = 20000000 ;   // 20 megabytes)
//OLDstatic const UINT64 PB_THRESHOLD = 5000000 ;    // (5 megabytes)
class WorkInProgress
{
   public:
   virtual ~WorkInProgress ( void ) ;     // destructor
   WorkInProgress ( void ) ;              // default constructor
   WorkInProgress ( UINT sFiles, UINT64 sBytes,   // initialization constructor
                    UINT64 thresh = PB_THRESHOLD ) ;
   //* Initialize source values and optionally set progress-bar threshold *
   void initialize ( UINT sFiles, UINT64 sBytes, UINT64 thresh = PB_THRESHOLD ) ;
   //* Returns percentage complete (floating point value <= 1.00) *
   double progress ( UINT& fProc, UINT64& bProc ) ;
   //* Update the data completion accumulators *
   void update ( UINT tFiles, UINT64 tBytes ) ;
   //* Create and open (display) a Progbar object *
   bool createPBar ( const pbarInit& pbi ) ;
   //* Delete any attached Progbar objects *
   bool deletePBar ( void ) ;
   //* Increment the progress bar display *
   short pbarUpdate ( short inc ) ;
   //* Refresh progress bar display *
   void  pbarRefresh ( void ) ;
   //* Returns progress bar threshold for creating secondary execution thread. *
   UINT64 getThreshold ( void ) ;
   //* Set the 'wipAbort' flag *
   bool setAbort ( void ) ;
   //* Returns current state of 'wipAbort' flag *
   bool getAbort ( void ) ;

   private:
   void reset ( void ) ;   // reset all data members
   bool reset_pbar ( void ) ; // delete attached Progbar object (if any)

   mutex  data_Lock ;      // access protection for data members

   UINT   srcFiles ;       // number of source files (data on clipboard)
   UINT64 srcBytes ;       // total source bytes
   UINT   trgFiles ;       // files processed
   UINT64 trgBytes ;       // bytes processed
   UINT64 pbThresh ;       // byte threshold for instantiation of progress bar
   Progbar* pBar ;         // pointer to a Progbar object
   bool   wipAbort ;       // set 'true' by primary thread if user aborts
   bool   wipPause ;       // set 'true' by primary thread if user requests a pause

} ;   // WorkInProgress



//****************************
//* FileDlg class definition *
//****************************
class FileDlg
{
public:
   //*******************
   //* Public Data     *
   //*******************
   // none

   //*******************
   //* Public Methods  *
   //*******************
   virtual ~FileDlg () ;            //* Destructor
   
//* Constructor for FileDlg object.                                            *
//*                                                                            *
//* Instantiate the FileDlg class, and the FMgr class to access the file       *
//* system and to display the results.                                         *
//* Caller provides information for NcDialog controls to be used.              *
//* On return, the caller's specified Path, File, and Statistics controls have *
//* been updated.                                                              *
//*                                                                            *
//* Input : init : InitFileDlg structure (by reference)                        *
//*                See discription of the fields of the InitFileDlg            *
//*                structure.                                                  *
//* Returns: if successful, returns a reference to FileDlg object.             *
//*           (else will probably throw a system exception)                    *
FileDlg ( InitFileDlg& init ) ;

//* Scroll through the list of files in the dctSCROLLEXT control which         *
//* currently has the input focus.                                             *
//* Returns when a non-scrolling key is detected.                              *
//* Scrolling keys are: nckUP, nckDOWN, nckPGUP, nckPGDOWN, nckHOME, nckEND.   *
//* Note that nckLEFT and nckRIGHT ARE NOT handled by this method; they are    *
//* returned to caller for processing.                                         *
//*                                                                            *
//* Input  : info : uiInfo class object (by reference, initial values ignored) *
//*                 on return, contains data on user selection                 *
//* Returns: index of dialog control which currently has input focus           *
short ScrollThruFileList ( uiInfo& info ) ;
   
//* Scan from currently-highlighted filename forward until a filename matching *
//* the specified sub-string is found. Highlight the matching file.            *
//* If no match found, highlight is not moved.                                 *
//*                                                                            *
//* Input  : substr : (by reference) contains at least one character to be     *
//*                   compared with the beginning of the displayed filenames   *
//*          indx   : (by reference) receives index of currently-highlighted   *
//*                   item                                                     *
//*          sensi  : (optional, false by default)                             *
//*                   if 'true' do a case-sensitive comparison,                *
//*                   if 'false' do a case-INsensitive comparison              *
//* Returns: 'true' if substring match found                                   *
bool  Scroll2MatchingFile ( const gString& substr, short& indx, bool sensi = false ) ;

//* Read contents of the current directory and display them in the dialog's    *
//* file-display control.                                                      *
//* Any data on the clipboard MAY BE discarded, if it is from the currently-   *
//* displayed directory.                                                       *
//*                                                                            *
//* Input  : withHilight:    (optional, true by default)                       *
//*                          if true, display data with highlight on           *
//*                            current item                                    *
//*                          if false, display data with no highlight          *
//*        : forceClipClear: (optional, false by default)                      *
//*                          if 'true' force clipboard to be cleared           *
//*                            regardless of its source directory              *
//* Returns: OK if successful, else ERR                                        *
//*          (if ERR, access error number through GetErrorCode())              *
short RefreshCurrDir ( bool withHilight = true, bool forceClipClear = false ) ;

//* Redraw the current data set in the file-display control.                   *
//* Statistics control is also updated.                                        *
//* Any data on the clipboard are not affected.                                *
//*                                                                            *
//* Input  : withHilight:    (optional, true by default)                       *
//*                          if true, display data with highlight on           *
//*                            current item                                    *
//*                          if false, display data with no highlight          *
//*        : clearSelection: (optional, false by default)                      *
//*                          if 'true' clear file selections.                  *
//* Return : OK if successful, else ERR                                        *
//*          (if ERR, access error number through GetErrorCode())              *
short RedrawCurrDir ( bool withHilight = true, bool clearSelection = false ) ;

//* If the currently-highlighted filename is the name of a directory,          *
//* set it as the current directory and display its contents.                  *
//*                                                                            *
//* Input  : none                                                              *
//* Returns: OK if successful, else ERR                                        *
//*          (if ERR, access error number through GetErrorCode())              *
short ChildDirectory ( void ) ;

//* If the current directory is not the root directory, set the parent         *
//* directory as the current directory and display its contents.               *
//* Else, redisplay the root directory.                                        *
//*                                                                            *
//* Input  : none                                                              *
//* Returns: OK if successful, else ERR                                        *
//*          (if ERR, access error number through GetErrorCode())              *
short ParentDirectory ( void ) ;

//* Set the specified directory path as the new directory to be displayed      *
//* in the window.                                                             *
//*                                                                            *
//* Note: Always change directories using the automatic built-in               *
//* ChildDir() and ParentDir() methods when possible, and call this method     *
//* only when it is necessary to 'jump' to a different directory.              *
//*                                                                            *
//* Input  : dirPath : full path specification for new target directory        *
//*          recurse : (optional, 'true' by default)                           *
//*                    'true'  : scan the full target-directory tree           *
//*                    'false' : scan only the top-level of the target tree    *
//*                              (used ONLY when target is known to be the)    *
//*                              (mountpoint for a large external drive   )    *
//*                                                                            *
//* Returns: OK if successful, else ERR                                        *
//*          (if ERR, access error number through GetErrorCode())              *
short SetDirectory ( const gString& dirPath, bool recurse = true ) ;

//* 'Select' i.e. set the the color attribute for the highlighted file to      *
//* indicate that the file has been selected for processing.                   *
//* 'Selected' files will be operated on by the next Copy/Cut etc. operation   *
//*                                                                            *
//* Input  : all : (optional, false by default)                                *
//*                if true, select all files in directory                      *
//*                   (highlight position unchanged)                           *
//*                if false, select highlighted file and move highlight        *
//*                   in specified direction                                   *
//*          dir : (optional, 1 by default - referenced only if all==false)    *
//*                after highlighted file is selected:                         *
//*                if  1, then move highlight down by one (default)            *
//*                if  0, then do not move highlight                           *
//*                if -1, then move highlight up by one                        *
//* Returns: true if successful, else false                                    *
bool  SelectFile ( bool all = false, short dir = 1 ) ;

//* Reset the 'selected' state (color attribute) for the file under the        *
//* cursor indicating that it IS NOT "selected"                                *
//* Clipboard contents are not affected.                                       *
//*                                                                            *
//* Input  : all : (optional, false by default)                                *
//*                if true, de-select all files in directory                   *
//*                   (highlight position unchanged)                           *
//*                if false, de-select highlighted file and move highlight     *
//*                   in specified direction                                   *
//*          dir : (optional, 1 by default - referenced only if all==false)    *
//*                after highlighted file is de-selected:                      *
//*                if  1, then move highlight down by one (default)            *
//*                if  0, then do not move highlight                           *
//*                if -1, then move highlight up by one                        *
//*                                                                            *
//* Returns: true if successful, else false                                    *
bool  DeselectFile ( bool all = false, short dir = 1 ) ;

//* Returns 'true' if highlighted file has been 'selected'.                    *
//* Alternatively, if 'all' parameter is true, returns 'true' if ANY file in   *
//* the list is selected.                                                      *
//*                                                                            *
//* Input  : all (optional, false by default): if true, test whether           *
//*            ANY files in directory have been 'selected'.                    *
//*                                                                            *
//* Returns: true if currently-highlighted file is 'selected', else false      *
bool  IsSelected ( bool all = false ) ;

//* Returns 'true' if highlighted file has been 'selected'.                    *
//* This method provides the data necessary for caller to perform group file   *
//* selections. For all other needs, call the previous IsSelected() method.    *
//*                                                                            *
//* Input  : currItem  : (by reference) receives index of highlighted item     *
//*          totalItems: (by reference) receives number of items in list       *
//*          prevSel   : (by reference)                                        *
//*                      set to 'true' if currItem > ZERO AND item at          *
//*                      (currItem-1) is selected                              *
//*                      else set to 'false'                                   *
//*          nextSel   : (by reference)                                        *
//*                      set to 'true' if currItem < (totalItems-1) AND item   *
//*                      at (currItem+1) is selected                           *
//* Returns: true if currently-highlighted file is 'selected', else false      *
bool IsSelected ( short& currItem, short& totalItems, 
                  bool& prevSel, bool& nextSel ) ;

//* Set color attribute of selected file(s) to indicate that files are in      *
//* the "copy" state.                                                          *
//*                                                                            *
//* Copy the list of selected files to our clipboard using FillClipboard().    *
//*                                                                            *
//* The "copy" state means that selected file(s) have been identified as       *
//* candidates for copying from one directory to another using the copy-and-   *
//* paste sequence.                                                            *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: true if successful, false if no files tagged.                     *
bool  Copy2Clipboard ( void ) ;

//* Set color attribute of selected file(s) to indicate that files are in      *
//* the "cut" state.                                                           *
//*                                                                            *
//* Copy the list of selected files to our clipboard using FillClipboard().    *
//*                                                                            *
//* The "cut" state means that selected file(s) have been identified as        *
//* candidates for moving from one directory to another using the cut-and-     *
//* paste sequence.                                                            *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: true if successful, false if no files tagged.                     *
bool  Cut2Clipboard ( void ) ;
   
//* Returns number of files on clipboard.                                      *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: file count                                                        *
UINT  ClipboardFiles ( void ) ;

//* Returns total number of bytes represented by files on clipboard.           *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: total size of all files on the clipboard (in bytes)               *
//*          NOTE: value returned is an 'unsigned long long int' (64-bit value)*
UINT64 ClipboardSize ( void ) ;

//* Returns a copy of the path to base directory where clipboard data live.    *
//*                                                                            *
//* Input  : cbPath : (by reference) receives path to clipboard data           *
//*                                                                            *
//* Returns: OK  if clipboard contains data                                    *
//*          ERR if clipboard is empty (cbPath == empty string)                *
short ClipboardPath ( gString& cbPath ) ;

//* Returns 'true' if files on clipboard were placed there by a                *
//* cut-to-clipboard operation. See enum OpPend.                               *
//*                                                                            *
//* Input  :none                                                               *
//*                                                                            *
//* Returns: true if pending operation is opCUT                                *
//*           else 'false', operation is not a 'cut', or clipboard empty       *
bool  CutPending ( void ) ;

//* Copy all files in the clipboard tree list to target with top level of tree *
//* written to the currently-displayed directory.                              *
//* If a cut-and-paste operation, delete source files after successful copy.   *
//*                                                                            *
//* Input  : attCount (by reference, initial value ignored)                    *
//*           on return, contains number of copy/move operations attempted,    *
//*           (usually) the number of files on the clipboard at the time of    *
//*           the call                                                         *
//*          delCount (by reference, initial value ignored)                    *
//*           if a delete-after-copy operation is in progress, on return,      *
//*           contains number of source files successfully deleted after       *
//*           being successfully copied.                                       *
//*           If not a delete-after-copy operation, ZERO is returned.          *
//* Returns: number of files successfully copied or moved to target            *
//*            if clipboardFiles != return value, then one or more errors      *
//*            occurred or user aborted the operation                          *
UINT  PasteClipboardList ( UINT& attCount, UINT&delCount ) ;

//* Paste all files on the clipboard in one of the 'special' ways.             *
//* A 'Paste-Special' operation can be any one of the following:               *
//*   [For all of thesee operations, target file can optionally be renamed.]   *
//*   - Create a symbolic link (shortcut) to an existing non-sym-link file     *
//*   - Create a 'hard' link to an existing non-sym-link file                  *
//*   - Copy/move a symbolic link                                              *
//*   - Copy a sym-link file's target file                                     *
//*   - Copy/move source file(s) with individual rename                        *
//*   - Copy/move source files with batch rename                               *
//*   - Perform a 'standard' copy/move operation (in case user decides that    *
//*     the 'special' operations are not appropriate).                         *
//*                                                                            *
//* Input  : attCount (by reference, initial value ignored)                    *
//*           on return, contains number of copy/move operations attempted,    *
//*           (usually) the number of files on the clipboard at the time of    *
//*           the call                                                         *
//*          delCount (by reference, initial value ignored)                    *
//*           if a delete-after-copy operation is in progress, on return,      *
//*           contains number of source files successfully deleted after       *
//*           being successfully copied.                                       *
//*           If not a delete-after-copy operation, ZERO is returned.          *
//* Returns: number of files successfully copied or moved to target            *
//*            if clipboardFiles != return value, then one or more errors      *
//*            occurred or user aborted the operation                          *
UINT  PasteSpecialList ( UINT& attCount, UINT& delCount ) ;

//* Move all 'selected' files in current window to the desktop environment's   *
//* Trashcan.                                                                  *
//*                                                                            *
//* Input  : (by reference, initial value ignored)                             *
//*          On return, contains number of files in selection list             *
//* Returns: number of files moved to Trashcan                                 *
//*           If caller's 'selected' parameter == return value,                *
//*           then no errors                                                   *
UINT  TrashcanSelectedFiles ( UINT& selected ) ;

//* Delete all 'selected' files in the current window.                         *
//* (See also TrashcanSelectedFiles().)                                        *
//*                                                                            *
//* Input  : selected: (by reference, initial value ignored)                   *
//*            On return, contains number of files in selection list           *
//* Returns: number of files deleted                                           *
//*           If caller's 'selected' parameter == return value,                *
//*           then no errors                                                   *
UINT  DeleteSelectedFiles ( UINT& selected ) ;

//* Interactively rename all 'selected' files in the current window.           *
//* This is the public method for accessing the file-rename functionality.     *
//* All associated methods are private.                                        *
//* IMPORTANT NOTE: Batch rename may be used on multiple 'selected' files, and *
//*                 directory names at the top level; however, recursive       *
//*                 rename of all files in a directory tree is not supported   *
//*                 at this time.                                              *
//*                                                                            *
//* Input  : selected: (by reference, initial value ignored)                   *
//*            On return, contains number of files in selection list           *
//* Returns: number of files renamed                                           *
UINT  RenameSelectedFiles ( UINT& selected ) ;
   
//* Update the date/time stamp on selected file(s). By default the current     *
//* system timestamp is used, but user has the option of specifying any        *
//* valid timestamp.                                                           *
//*                                                                            *
//* Updates both 'modification' date and 'access' date to specified date/time  *
//* stamp.  Updates inode change time, 'ctime', to current system date/time.   *
//*                                                                            *
//* Input  : selected: receives number of updates attempted                    *
//* Returns: number of files updated                                           *
//*           if return value == 'selected' parm, all selected files were      *
//*             updated successfully                                           *
//*           else at least one selected file was not updated                  *
UINT  TouchSelectedFiles ( UINT& selected ) ;
   
//* Write-enable or write-protect selected file(s).                            *
//*   OR                                                                       *
//* Allow user to set custom permissions for selected file(s).                 *
//*                                                                            *
//* If a displayed directory name is selected, then all files in all           *
//* subdirectories it contains will ALSO be write-enabled/protected/modified.  *
//* NOTES:                                                                     *
//* 1) Write permission on the containing subdirectory IS NOT required in      *
//*    order to adjust permissions on the files it contains, BUT read          *
//*    permission on the containing subdirectory IS obvioulsy required;        *
//*    otherwise, we could not read the subdirectory contents.                 *
//* 2) If a read-protected subdirectory is in the list of selected files, then *
//*    its contents WILL NOT be on the clipboard (because we can't read them), *
//*    and thus, will not be processed during this operation.                  *
//*                                                                            *
//* Input  : selFiles : (by reference, initial value ignored)                  *
//*                     receives number of files selected for update           *
//*          modFiles : (by reference, initial value ignored)                  *
//*                     receives number of files successfully modified         *
//*          protect  : if 'true', write-protect selected files                *
//*                     if 'false', write-enable selected files                *
//* Returns: operation performed:                                              *
//*                     0 == write protect selected files                      *
//*                     1 == write enable selected files                       *
//*                     2 == custom permissions applied to selected files      *
//*                   ERR == if user aborts the operation                      *
//*                if selFiles == modFiles, then all selected files were       *
//*                   updated successfully                                     *
//*                else at least one selected file was not updated             *
short ProtectSelectedFiles ( UINT& selFiles, UINT& modFiles, bool protect ) ;

//* Perform a backup of all files and directory trees on the clipboard.        *
//* 1) If target file does not already exist, copy source to target.           *
//* 2) If target file already exists, then compare source and target           *
//*    modification dates.                                                     *
//*    a) If source is newer, overwrite old target file.                       *
//*    b) If target is newer or the same date (+/- one second), then do not    *
//*       copy source to target.                                               *
//*                                                                            *
//* Input  : bkSet : operational parameters (by reference)                     *
//*                  'bex' is an array of exclusion records                    *
//*                  'bexCount' is the number of records in 'bex' array        *
//*                  'blog' is filespec for operational log                    *
//*          pbForce: (optional, false by default)                             *
//*                  'false' programatically determine whether the progress    *
//*                          bar will be instantiated.                         *
//*                   'true' force the progress bar to be instantiated         *
//*                    NOTE: At this time, this flag is used _only_ during     *
//*                          directory synchronizaton.                         *
//*          passTwo: (optional, false by default, for Synch only)             *
//*                   If 'pbForce' != false, then:                             *
//*                      false == first pass                                   *
//*                      true  == second pass                                  *
//*                   If 'pbForce' == false, then 'passTwo' is ignored         *
//*                                                                            *
//* Returns: number of files successfully copied or updated                    *
//*          'ftotal'   receives total number of source files                  *
//*          'fupdated' receives number of target files updated                *
//*          'fskipped' receives number of target files requiring              *
//*                     no update (this includes 'excluded' files)             *
//*          'ferrors'  receives the number of files which were                *
//*                     not processed due to access errors or other errors     *
//*                     OR not processed because user aborted the operation    *
UINT  BackupClipboardList ( BSet& bkSet, bool pbForce = false, bool passTwo = false ) ;

//* Create an archive file containing all files and directory trees on the     *
//* clipoard. Called when target of a backup operation is an archive.          *
//*                                                                            *
//* Input  : bkSet : operational parameters                                    *
//*                  'blog' is filespec for operational log                    *
//*                                                                            *
//* Returns: OK if successful, or ERR if one or more errors detected           *
//*          bkSet members receive accumulated data:                           *
//*           'ftotal'   receives number of source files on clipboard          *
//*           'fupdated' receives number of files added to archive             *
//*           'ferrors'  receives number of errors during processing           *
short ArchiveClipboardList ( BSet& bkSet ) ;

//* Determine whether the specified file is one of the supported archive-file  *
//* types.                                                                     *
//*                                                                            *
//* Scan the filename for a recognized archive filename extension.             *
//* If a match found, return the archive type. Note that this does not         *
//* guarantee that the target is actually of the specified type, only that     *
//* the user named it that way.                                                *
//*                                                                            *
//* Supported archive types: tar (with or without compression), zip, or        *
//* or an OpenDocument/OpenXML file (zip archive).                             *
//*                                                                            *
//* IMPORTANT NOTE: The corresponding utility or utilities must be correctly   *
//*                 installed on the system.                                   *
//*                                                                            *
//* Input  : trgPath: path/filename of target file                             *
//*          trgExt : (optional, NULL pointer by default)                      *
//*                   pointer to gString object to receive filename extension  *
//* Returns: member of enum arcType                                            *
arcType ArchiveTarget ( const gString& trgPath, gString* trgExt = NULL ) ;

//* Scan the specified archive, counting the number of files it contains and   *
//* the approximate total size of the data when expanded.                      *
//*                                                                            *
//* Input  : trgPath: path/filename of target file                             *
//*          tfFile : (by reference) receives the target-file stats            *
//*          aFiles : (by reference) receives number of files in archive       *
//*          aBytes : (by reference) receives (approximate) expanded total size*
//* Returns: OK  if success                                                    *
//*          ERR if file not found or is not a recognized archive type         *
short ArchiveSummary ( const gString& trgPath, tnFName& tnFile, 
                       UINT& aFiles, UINT64& aBytes ) ;

//* Initialize parameters for creation, update or expansion of an archive file.*
//* User may specify:                                                          *
//*  1) source data to create a new archive.                                   *
//*  2) the name of an existing archive to be expanded                         *
//*  3) source data to be appended to a specified existing archive.            *
//*                                                                            *
//* Because a single command is used for 'create' and 'expand' AND 'update',   *
//* see the logic for selecting the operation to be performed.                 *
//*                                                                            *
//* Input  : none                                                              *
//* Returns: nothing                                                           *
void  Archive ( void ) ;

//* Get name for new sub-directory from user, and                              *
//* create the new sub-directory in the currently-displayed directory.         *
//*                                                                            *
//* Input  : none                                                              *
//* Returns:  OK if directory created successfully,                            *
//*           ERR if operation aborted                                         *
//*           or 'errno' value if operation failed                             *
short CreateDirectory ( void ) ;

//* Create a new sub-directory file.                                           *
//*                                                                            *
//* Input  : dirName: name of directory to be created (not filespec)           *
//* Returns: OK if successful, else ERR                                        *
short CreateDirectory ( const gString& dirName ) ;

//* Delete the specified file.                                                 *
//* NOTE: This method is for removing unneeded application temp files only.    *
//*       Normal application file deletion must use a more formal sequence.    *
//*                                                                            *
//* Input  : full path/filename of directory to be deleted                     *
//* Returns: OK if successful, else ERR                                        *
short DeleteFile ( const gString& trgPath ) ;

//* Return a copy of the path string for the currently-displayed directory.    *
//*                                                                            *
//* Input  : currPath: (by reference) gString object to receive path string    *
//* Returns: OK                                                                *
short GetPath ( gString& currPath ) ;

//* Return a copy of the path string for the currently-displayed directory.    *
//* Also returns read- and write-access flags.                                 *
//*                                                                            *
//* Input  : currPath: (by reference) gString object to receive path string    *
//*          rPerm   : (by reference) receives read-access flag                *
//*          wPerm   : (by reference) receives write-access flag               *
//* Returns: OK                                                                *
short GetPath ( gString& currPath, bool& rPerm, bool& wPerm ) ;

//* Determine whether the specified target file exists.                        *
//*                                                                            *
//* Input  : fPath   : full path/filename specification of file to be tested   *
//*          fType   : if target exists, initialized to target fileType        *
//*                    else fmUNKNOWN_TYPE                                     *
//* Returns: true if target file exists, else false                            *
bool  TargetExists ( const gString& fPath, fmFType& fType ) ;

//* Returns a copy of the tnFName class object containing data for the         *
//* currently highlighted file.                                                *
//*                                                                            *
//* Input  : fn   : (by reference, initial values ignored)                     *
//*                 receives a copy of file stats for highlighted file         *
//* Returns: index of highlighted filename, or ERR if empty directory          *
short GetStats ( tnFName& fn ) ;

//* Expand and format for human readability the 'stat' info for the file       *
//* specified by fs structure.                                                 *
//* Pass-through method to FMgr class method of the same name.                 *
//*                                                                            *
//* Input  : fs   : pointer to FileStats structure (stat64) returned by        *
//*                 the GNU C library stat64() and lstat64() functions.        *
//*                 (Note: for displayed files and files on the    )           *
//*                 (clipboard, this structure is the 'rawStats'   )           *
//*                 (member of the tnFName class object.           )           *
//*          es   : ExpStats structure for holding expanded statistics         *
//*                 a) caller (optionally) initializes es.filePath with        *
//*                    the path string for the file (excluding filename)       *
//*                    (This is needed to 'stat' a symbolic link target.)      *
//*                 b) if the specified file is a symbolic link,               *
//*                    caller (optionally) initializes es.symbTarg             *
//*                    referencing a second ExpStats structure to              *
//*                    hold the info for the symbolic link's target.           *
//*                 (other fields of es initialized on return)                 *
//* Return : OK if successful, else ERR                                        *
//*             ERR indicates a problem with the symbolic link target:         *
//*               a) invalid es.filePath, or                                   *
//*               b) specified file is a symbolic link, but es.symbTarg        *
//*                  does not point to an ExpStats structure to hold           *
//*                  target file's information, or                             *
//*               c) broken symbolic link                                      *
//*                  Note: if ERR returned, and cause of error was a           *
//*                   broken symbolic link, then                               *
//*                es.symbTarg->fileType == fmTYPES,                           *
//*                es.symbTarg->fileName == filename of missing target         *
//*                                         if available, else "unknown"       *
//*                es.symbTarg->filePath == path of missing target             *
//*                                         if available, else "unknown"       *
//*                   and all other fields of es.symbTarg are undefined.       *
short ExpandStats ( const FileStats* fs, ExpStats& es ) ;

//* Display the file statistics for the currently-highlighted file.            *
//* If the file is a symbolic link, then also display the statistics for the   *
//* the target file.                                                           *
//*                                                                            *
//* Input  : none                                                              *
//* Returns: OK     if stats successfully displayed                            *
//*          ERR    if error displaying or modifying stats                     *
//*          > ZERO if file stats were successfully modified by user           *
short DisplayFileStats ( void ) ;

//* Directory window's highlight is on a symbolic-link file, and user has      *
//* requested that we find the link target.                                    *
//*                                                                            *
//* Input  : none                                                              *
//* Returns: OK  if operation successful                                       *
//*          ERR if target not found or is inaccessible                        *
short FindLinkTarget ( void ) ;

//* View contents of the currently-highlighted file.                           *
//*                                                                            *
//* Output is first formated (if necessary) and then displayed using the       *
//* 'less' command-line utility.                                               *
//*                                                                            *
//* Input  : vfFmt : (optional, vfPrompt by default) -member of enum viewFormat*
//*                  format in which to view file contents                     *
//*                  [ CURRENTLY IGNORED ]                                     *
//* Returns: OK if successful,                                                 *
//*          ERR if file not found or file is not a 'regular' file             *
short ViewFileContents ( viewFormat vfFmt = vfPrompt ) ;

//* User has requested a file comparison.                                      *
//* Scan the 'selected' data and highlight position to determine whether the   *
//* user has indicate a two-file comparison or a multi-file comparison.        *
//* If multi-file comparison, call it directly, but if a two-file comparison   *
//* return to caller for processing.                                           *
//*                                                                            *
//* Input  : extPath   : path of directory which _potentially_ contains files  *
//*                      to be compared with selected files                    *
//*                                                                            *
//* Returns: 'true'  if multiple-file comparison completed                     *
//*          'false' if user selected less than two(2) source files            *
bool  MultiCompare ( const gString& extPath ) ;

//* Compare two files for the following data:                                  *
//* 1) timestamps                                                              *
//* 2) file size                                                               *
//* 3) file type                                                               *
//* 4) permission bits                                                         *
//* 5) Contents:                                                               *
//*    The 'diff' utility has several options for scan and output.             *
//*    We implement user selection for the most useful features.               *
//*                                                                            *
//* Input  : extFSpec  : if secondary file is external to this class,          *
//*                      this is the path/filename of that file                *
//*          extFStat  : if secondary file is external to this class,          *
//*                      this is the lstat data for the file                   *
//* Returns: nothing                                                           *
void  CompareFiles ( const gString& extFSpec, const tnFName& extFStat ) ;

//* Compare the files in the selected group with the corresponding files       *
//* (if they exist) in the specified directory.                                *
//* 1) timestamps                                                              *
//* 2) file size                                                               *
//* 3) file type                                                               *
//* 4) permission bits                                                         *
//* 5) Contents: report contents as "identical" or "different" only            *
//*                                                                            *
//* Input  : extPath   : path of directory containing files to be compared     *
//*                      with selected files                                   *
//* Returns: nothing                                                           *
void  CompareFiles ( const gString& extPath ) ;

//* Execute (run, not murder) the currently-highlighted file.                  *
//*  1. Determine whether the targeted file is 'executable'. It could be       *
//*     a compiled application, a shell script, a Python script, etc.          *
//*  2. Create a command string with optional user-supplied arguments.         *
//*  3. Temporarily put the current application to sleep.                      *
//*  4. Execute the command at the command line.                               *
//*  5. Return to this application.                                            *
//*                                                                            *
//* Invoke the system's default application for the source file-type.          *
//*  1. Media files: images, audio, video.                                     *
//*  2. Documents: pdf, odt, etc.                                              *
//*  This relies on the system script file 'xdg-open' to select the appropriate*
//*  external application. See LaunchDefaultApplication() method for details.  *
//*                                                                            *
//* Input  : cmdBuff: (optional, NULL* by default)                             *
//*                   if specified, return the command string to caller        *
//*                   instead of executing it locally. In this case, caller    *
//*                   should close the the NcDialog object, stop the ncurses   *
//*                   engine, fork the process and exit the application by     *
//*                   calling 'execv' or a similar exec function.              *
//* Returns: OK if successful,                                                 *
//*          ERR if file not found or file is not a executable file            *
short OpenExternalProgram ( char* cmdBuff = NULL ) ;
short oepUserPrompt ( const gString& fPath, gString& argList, 
                      bool exeFlag, bool binFlag ) ;

//* Launch the system default application for handling the specified file type.*
//*                                                                            *
//* Input  : targFile : filespec of file to be processed by the external       *
//*                     application                                            *
//*          redirect : (optional, 'true' by default)                          *
//*                     if 'true',  redirect stdout and stderr to temp files   *
//*                     if 'false', allow stdout and stderr to write to the    *
//*                                 terminal window                            *
//*                                                                            *
//* Returns: a) the child process ID                                           *
//*          b) -1 if child process not created                                *
int   LaunchDefaultApplication ( const char* targFile, bool redirect = true ) ;

//* Launch a specific external application along with desired arguments.       *
//* See notes below for a detailed discussion of the arguments.                *
//*                                                                            *
//* Input  : appSpec  : filespec (or filename) of application                  *
//*        : argList  : arguments sent to target application as argv[] array   *
//*        : redirect : (optional, 'true' by default)                          *
//*                     if 'true',  redirect stdout and stderr to temp files   *
//*                     if 'false', allow stdout and stderr to write to the    *
//*                                 terminal window                            *
//*                                                                            *
//* Returns: a) the child process ID                                           *
//*          b) -1 if child process not created                                *
int   LaunchExternalApplication ( const char* appSpec, 
                                  const gString& argList, bool redirect = true ) ;

//* Set file permissions for the specified file: USR rwx, GRP rwx, OTHER rwx,  *
//* sUID, sGID, and sticky bit.                                                *
//* (requires either superuser access or ownership of the file.)               *
//*                                                                            *
//* Input  : ExpStats structure with at least the following fields             *
//*          initialized: fileName, filePath, userID,                          *
//*          usrProt, grpProt, othProt, setUID, setGID, sticky                 *
//* Returns: OK if successful, else code from 'errno'.                         *
short SetFilePermissions ( ExpStats& newStats ) ;

//* Set file ownership (userID).                                               *
//* (requires superuser access)                                                *
//* Note that the setUID and setGID bits may be reset depending on Linux       *
//* kernel version.                                                            *
//*                                                                            *
//* Input  : ExpStats structure with at least the following fields initialized:*
//*            fileName, filePath, userID                                      *
//* Returns: OK if successful, else code from 'errno'.                         *
short SetFileOwner ( ExpStats& newStats ) ;

//* Set file group (grpID).                                                    *
//* (requires either superuser access or ownership of the file)                *
//* (if non-superuser, owner must be a member of the specified group)          *
//*                                                                            *
//* Note that the setUID and setGID bits may be reset depending on Linux       *
//* kernel version and on whether the user is superuser or file owner.         *
//*                                                                            *
//*                                                                            *
//* Input  : ExpStats structure with at least the following fields             *
//*          initialized: fileName, filePath, userID, grpID                    *
//*                                                                            *
//* Returns: OK if successful, else code from 'errno'.                         *
short SetFileGroup ( ExpStats& newStats ) ;

//* Set file Modify and/or Access date/timestamps.                             *
//* As a side effect, cTime will be set to current system time.                *
//* (requires file write permission)                                           *
//*                                                                            *
//*                                                                            *
//* Input  : pathFilename: full path/filename specification                    *
//*          newDate     : structure must be correctly initialized             *
//*                        (except for 'day' which is ignored)                 *
//*          updateMod   : (optional) if true, update modification time        *
//*          updateAcc   : (optional) if true, update access time              *
//*                                                                            *
//* Returns: OK if successful, else code from 'errno'                          *
short SetFileTimestamp ( const char* PathFilename, const localTime& newDate, 
                         bool updateMod = true, bool updateAcc = true ) ;

//* Set file Modify and/or Access date/timestamps.                             *
//* As a side effect, cTime will be set to current system time.                *
//* (requires file write permission)                                           *
//*                                                                            *
//* Input  : ExpStats structure with the following fields initialized:         *
//*            1. fileName == valid file name                                  *
//*            2. filePath == valid file path                                  *
//*            3. if updateMod != false, then modTime == new date/time data    *
//*                                      (except for 'day' which is ignored)   *
//*            4. if updateAcc != false, then accTime == new date/time data    *
//*                                      (except for 'day' which is ignored)   *
//*                updateMod : if true, update modification time               *
//*                updateAcc : if true, update access time                     *
//*                updateAll : (optional, false by default)                    *
//*                            if true, set BOTH Mod and Access dates using    *
//*                            data in modTime field                           *
//* Returns: OK if successful, else code from 'errno'                          *
short SetFileTimestamp ( const ExpStats& fileStats, bool updateMod, 
                         bool updateAcc, bool updateAll ) ;

//* Set all three file times to current system local time.                     *
//* PathFilename is the full filespec string for the target file               *
//* Returns OK if successful, else code from 'errno'.                          *
//short SetFileTimestamp ( const char* PathFilename ) ;// NOT YET IMPLEMENTED!!
   
//* Shorten the path string so it will fit into the available display space.   *
//* This is done by replacing one or more directory names with the ellipsis.   *
//*     Example: /home/mongoose/Applications/CoffeeCups   --becomes--          *
//*              /home/mongoose/.../CoffeeCups                                 *
//*                                                                            *
//* Input  : gsPath  : gString object (by reference) containing path string    *
//*          maxWidth: maximum number of columns in display string             *
//* Returns: OK if successful, else ERR                                        *
short TrimPathString ( gString& gsPath, short maxWidth ) ;

//* Determine whether the specified filespec refers to a mountpoint directory. *
//* This is a pass-through to the FMgr-class method of the same name.          *
//*                                                                            *
//* Input  : trgPath : target filespec                                         *
//* Returns: 'true'  if target is a mountpoint directory                       *
//*          'false' if target is not a mountpoint                             *
//*                  (or is not a directory, or does not exist)                *
bool  isMountpoint ( const char* trgPath ) ;

//* Get the system timecode and convert it to localTime format.                *
//*                                                                            *
//* Input  : ft  : (by reference, initial values ignored)                      *
//*                on return, contains decoded date/time                       *
//* Returns: true if successful, false if system call fails                    *
bool  GetLocalTime ( localTime& lt ) ;

//* Return the current system-local epoch time.                                *
//*                                                                            *
//* Input  : epoch_time : (by reference, initial value ignored)                *
//*                       on return, contains epoch timecode                   *
//*                       (or ZERO if system call fails)                       *
//*                                                                            *
//* Returns: true if successful, false if system call fails                    *
bool  GetLocalTime ( int64_t& epoch_time ) ;

//* Convert specified epoch time to human-readable time.                       *
//*                                                                            *
//* Input  : lt  : (by reference) 'epoch' member (seconds since the epoch)     *
//*                contains time to be decoded into human-readable form        *
//* Returns: nothing                                                           *
void  DecodeLocalTime ( localTime& lt ) ;

//* Convert specified human-readable time to epoch time.                       *
//*                                                                            *
//* Input  : lt  : (by reference) human-readable members are converted to      *
//*                'epoch' and 'sysepoch' integers (seconds since the epoch)   *
//* Returns: nothing                                                           *
void  EncodeLocalTime ( localTime& lt ) ;

//* Produce a formatted output string based on data in a localTime object.     *
//*   Format:  "dd Mmm yyyy hh:mm:ss"                                          *
//*                                                                            *
//* Input  : ft  : initialized localTime structure (by reference)              *
//*                Note: 'day', the day-of-the-week field is ignored.          *
//*          str : buffer to receive formatted data                            *
//* Returns: nothing                                                           *
void  FormatDateString ( const localTime& ft, gString& str ) ;

//* Scan the date/time string and initialize a localTime class object.         *
//* String must be of the format: "dd Mmm yyyy hh:mm:ss"                       *
//* Date/time string must be correctly formatted and the specified value       *
//* must be a valid date/time between:                                         *
//*          01 Jan 1970 00:00:00 and 31 Dec 2037 23:59:59.                    *
//*          (system's signed long int can hold dates to 18 Jan 2038 19:14:07) *
//* Note: 'day' (day-of-the-week) field is not initialized.                    *
//*                                                                            *
//* Input  : ft  : localTime class object gets validated data (by reference)   *
//*          str : buffer containing caller's date/time string of format:      *
//*                "dd Mmm yyyy hh:mm:ss"                                      *
//* Returns: true if successful, false if bad format or date out of range      *
bool  ValidateDateString ( localTime& ft, const gString& str ) ;

//* Test the specified character to determine whether it may be used as a      *
//* character in a filename or directory name.                                 *
//*                                                                            *
//* Input  : character to test (wchar_t)                                       *
//* Returns: true if acceptable, else false                                    *
bool  IsFNameChar ( wchar_t ch ) ;

//* Interactively modify file-sorting options.                                 *
//*         copt.sortOption: file sort order                                   *
//*      copt.caseSensitive: whether sort is case sensitive                    *
//*         copt.showHidden: whether 'hidden' files will be displayed          *
//*                                                                            *
//* Input  : copt  : (by reference, initial contents ignored)                  *
//*                  on return, above members will reflect the current settings*
//* Returns: 'true'  if options modified, else 'false'                         *
//*          Display list will be updated if options modified                  *
bool  ModifySortOptions ( ConfigOptions& copt ) ;

//* Set file-sorting options according to the following data-member values.    *
//*         copt.sortOption: file sort order                                   *
//*      copt.caseSensitive: whether sort is case sensitive                    *
//*         copt.showHidden: whether 'hidden' files will be displayed          *
//*                                                                            *
//* Input  : copt  : (by reference)                                            *
//* Returns: 'true'  if options changed, else 'false'                          *
//*          This method DOES NOT redraw the display.                          *
bool  UpdateSortOptions ( ConfigOptions& copt ) ;

//* Update the 'configOpt.mouseEnabled' flag indicating whether mouse is active*
//*                                                                            *
//* Input  : mEnabled : 'true' if mouse currently enabled, else 'false'        *
//* Returns: nothing                                                           *
void  UpdateMouse ( bool mEnabled ) ;

//* Compare the path of the target (currently displayed) directory with the    *
//* directory from which the clipboard was filled. This helps the mainline     *
//* code decide whether the clipboard data are stale (or are about to become   *
//* so).                                                                       *
//* For example, if the name/date/etc. of any file on the clipboard list has   *
//* been (or is about to be) modified, the clipboard data become invalid.      *
//*                                                                            *
//* Input  : treeMember: (optional, true by default)                           *
//*                  if 'true' test whether specified directory is             *
//*                    a node on the clipboard tree list                       *
//*                  if 'false' test only whether specified directory          *
//*                    is the top node in the clipboard tree list              *
//* Returns: 'true' if target directory == base clipboard directory            *
//*            OR target directory is a member node on the clipboard tree      *
//*          'false' if target is not a node on clipboard list or if           *
//*            clipboard is empty.                                             *
bool  TargDirEqualsClipDir ( bool treeMember = true ) ;

//* Returns a copy of the file system information for the file system          *
//* that includes the 'current' directory.                                     *
//* This is a pass-through method for FMgr class method of the same name.      *
//*                                                                            *
//* Input  : trgPath : filespec of file or directory to be scanned             *
//*          fsStats : (by reference, initial values ignored)                  *
//*                    receives filesystem information                         *
//* Returns: OK if successful                                                  *
//*           All members of 'fsStats' for which the target filesystem         *
//*           provides data have been initialized.                             *
//*           Note: Some filesystems do not provide data for all fields.       *
//*          if error, 'errno' is returned (some fsStats members unreliable)   *
short GetFilesystemStats ( const gString& trgPath, fileSystemStats& fsStats ) ;

//* Scan for devices attached to the Universal Serial Bus (USB).               *
//* Capture filesystem and support information for the devices.                *
//* This method is designed primarily for smartphones, tablets and other       *
//* virtual filesystems attached to the USB bus. Devices may, or may not be    *
//* mounted and visible to the user.                                           *
//*                                                                            *
//* Optionally, this method can report the basic bus, device, ID and           *
//* descriptive text information for each device attached to the USB bus.      *
//* This option is not used by the FileMangler application, but may be helpful *
//* in other applications.                                                     *
//*                                                                            *
//* This is a pass-through method for FMgr class method of the same name.      *
//*                                                                            *
//* Important Note: This method allocates dynamic memory. It is the caller's   *
//*                 responsibility to release this memory when it is no longer *
//*                 needed. Example: delete [] usbdPtr ;                       *
//*                                                                            *
//* Input  : usbdPtr  : (by reference, initial value ignored)                  *
//*                     receives pointer to a dynamically-allocated array of   *
//*                     usbDevice objects. Set to NULL pointer if no USB       *
//*                     devices matching the criteria are found.               *
//*          testmnt  : (optional, 'false' by default)                         *
//*                     if 'false', do not test for mounted status             *
//*                     if 'true',  for each MTP/GVFS device determine whether *
//*                                 it is currently mounted                    *
//*          all      : (optional, 'false' by default)                         *
//*                     if 'false', report only MTP/GVFS devices               *
//*                     if 'true',  summary report of all devices attached     *
//*                                 to USB bus                                 *
//*                                                                            *
//* Returns: count of USB devices captured (elements in usbDevice array)       *
short USB_DeviceStats ( usbDevice*& usbdPtr, bool testmnt = false, bool all = false ) ;

//* Determine the device-driver path associated with the specified URI.        *
//*                                                                            *
//* The system often increments the device index of attached USB devices.      *
//* This happens most often when a mount operation for a USB device fails.     *
//* When this happens, the the device path must be updated before the mount    *
//* can be retried.                                                            *
//*                                                                            *
//* Input  : uri   : (by reference) URI of target device                       *
//*          devid : (by reference) receives device-driver path                *
//*                                                                            *
//* Returns: 'true'  if device ID found                                        *
//*          'false' if no matching device found                               *
bool  USB_DeviceID ( const char* uri, gString& devid ) ;

//* Scan for drives attached to the local system.                              *
//* Capture filesystem and support information for the devices.                *
//*                                                                            *
//* This method is designed primarily for locating optical drives              *
//* (DVD,CD,BLU-RAY, etc). Data discs contained in the drive may, or may not   *
//* be mounted and visible to the user.                                        *
//*                                                                            *
//* Optionally, this method can report information for all attached drives.    *
//* This option is not used by the FileMangler application, but may be helpful *
//* in other applications.                                                     *
//*                                                                            *
//* This is a pass-through method for FMgr class method of the same name.      *
//*                                                                            *
//* Important Note: This method allocates dynamic memory. It is the caller's   *
//*                 responsibility to release this memory when it is no longer *
//*                 needed. Example: delete [] usbDev ;                        *
//*                                                                            *
//* Input  : usbdPtr  : (by reference, initial value ignored)                  *
//*                     receives pointer to a dynamically-allocated array of   *
//*                     usbDevice objects. Set to NULL pointer if no optical   *
//*                     devices matching the criteria are found.               *
//*          testmnt  : (optional, 'false' by default)                         *
//*                     if 'false', do not test for mounted status             *
//*                     if 'true',  for each optical drive identified,         *
//*                                 determine whether it is currently mounted  *
//*          all      : (optional, 'false' by default)                         *
//*                     if 'false', report only optical drive devices          *
//*                     if 'true',  summary report of all drives identified    *
//*                                                                            *
//* Returns: count of optical drives captured (elements in usbDevice array)    *
short DVD_DeviceStats ( usbDevice*& usbdPtr, bool testmnt = false, bool all = false ) ;

//* Displays the file system information for the file system that includes     *
//* the 'current' directory. OR optionally, display file system info for       *
//* specified path.                                                            *
//*                                                                            *
//* Input  : trgPath : (optional, NULL pointer by default)                     *
//*                    if specified, use the path as the target for reading    *
//*                    filesystem information                                  *
//* Returns: 'true' if information read successfully, else 'false'             *
bool  DisplayFilesystemInfo ( const char* trgPath = NULL ) ;

//* Returns the version string for the GnomeVirtualFileSystem (GVfs) tools.    *
//* This is the version of the 'gio' utility if installed.                     *
//*                                                                            *
//* Input  : gvfsVersion : (by reference) receives the GVfs utilities version  *
//*                        string, or empty string if 'gio' not installed      *
//* Returns: true if GVfs tools are accessible, else false                     *
bool  GetGVfsVersion ( gString& gvfsVersion ) ;

//* Returns a copy of user information: uid, gid, user name, etc. stored in    *
//* userInfo structure.                                                        *
//*                                                                            *
//* Input  : (by reference) UserInfo-class object to receive data              *
//* Returns: true if successful, false if data unavailable                     *
bool  GetUserInfo ( UserInfo& uInfo ) ;

//* Open a dialog and display user's information. Primarily for debugging,     *
//* but may provide mainline code with useful functionality.                   *
//* Note that if the supplimentary GID list is long, we don't display all of   *
//* it, only the first 15 GIDs. Also, because the actual password strings      *
//* are probably not returned by the system calls, we don't display the        *
//* dummy password strings (it would be too confusing to the user).            *
//*                                                                            *
//* Input  :none                                                               *
//* Returns: 'true' if successful,                                             *
//*          'false' if data unavailable or memory allocation error            *
bool  DisplayUserInfo ( void ) ;

//* Restore the item(s) most recently moved to the Trashcan.                   *
//* This is an 'undo' function for item(s) sent to trash.                      *
//*                                                                            *
//* Input  : overwrite : (optional, 'true' by default)                         *
//*                      if 'true'  silently overwrite any existing targets    *
//*                      if 'false confirm before overwrite [CURRENTLY IGNORED]*
//*          altTarget : (optional, NULL pointer by default)                   *
//*                      if specified, alternate target directory for restore  *
//*                      operation(s) [CURRENTLY IGNORED]                      *
//* Returns: number of files restored from Trashcan                            *
UINT  TrashcanUndo ( bool overwrite = true, const char* altTarget = NULL ) ;

//* Allow user to view the contents of the Trashcan, delete (unlink) items     *
//* and restore item(s) to to their original location.                         *
//*                                                                            *
//* Input  : restore  : (optional, 'false' by default)                         *
//*                     if 'false', user will interactively manage Trashcan    *
//*                     if 'true',  restore the most recent move-to-trash      *
//* Returns: nothing                                                           *
void  ManageTrashcan ( bool restore = false ) ;

//* Invoke the 'Info' system to display contextual help information.           *
//*                                                                            *
//* Input  : ccode  : member of contextCode indicating the context of call     *
//* Returns:  OK if successful, ERR if error                                   *
short CallContextHelp ( contextCode ccode ) ;

//* Generic information dialog. May be used to display any list of constant    *
//* strings, then waits for user to press Enter (or ESC).                      *
//*                                                                            *
//* Dialog minimum height is 4 (top line, space above OK control, line for     *
//* OK control, and bottom line) PLUS one message line.                        *
//* Dialog maximum height is MIN_LINES - 9. This value represents the minimum  *
//* interior height of the file-display control, and provides space for        *
//* INFODLG_MAX_MSGS message lines including dialog title.                     *
//* Dialog width is fixed at INFODLG_WIDTH. This value represents the minimum  *
//* interior width of the file-display control.                                *
//* For reasons of beauty, messages begin at offset 2 in X and SHOULD end at   *
//* dialog-width minus 2.                                                      *
//*                                                                            *
//* Input  : msgList: array of char pointers to messages                       *
//*                   The first message line is interpreted as the dialog title*
//*                   " " == empty line and NULL (null pointer) == end of list *
//*          msgAttr: (optional, NULL by default)                              *
//*                   if not specified, each message line is displayed in the  *
//*                     base interior color set during FileDlg instantiation.  *
//*                   if specified, this is a pointer to an array of color     *
//*                     attributes, one for each message line in msgList.      *
//* Returns: nothing                                                           *
void  InfoDialog ( const char** msgList, const attr_t* msgAttr = NULL ) ;

//* Generic decision-making dialog. Similar to InfoDialog() above, except      *
//* that user must choose between the "YES" and "NO" pushbuttons.              *
//*                                                                            *
//* For this reason, your message should be one that can be answered by        *
//* either a "yes" or "no" response.                                           *
//*                                                                            *
//* Input  : msgList: array of char pointers to messages                       *
//*                   The first message line is interpreted as the dialog title*
//*                   " " == empty line and NULL (null pointer) == end of list *
//*          msgAttr: (optional, NULL by default)                              *
//*                   if not specified, each message line is displayed in the  *
//*                     base interior color set during FileDlg instantiation.  *
//*                   if specified, this is a pointer to an array of color     *
//*                     attributes, one for each message line in msgList.      *
//*                                                                            *
//* Returns: 'true' if user selects "YES", else 'false'                        *
bool  DecisionDialog ( const char** msgList, const attr_t* msgAttr = NULL ) ;

//* Display the file-system tree structure beginning with the 'current'        *
//* directory and traversing all subdirectories below current directory.       *
//* User has the option of selecting a new current-working-directory' (CWD)    *
//* from the displayed data.                                                   *
//*                                                                            *
//* Input  : ul        : (by reference) upper left corner dialog position      *
//*          rows      : number of rows for dialog                             *
//*          cols      : number of columns for dialog                          *
//* Returns: true if current working directory (CWD) has changed               *
//*          else false                                                        *
bool  DisplayFileTree ( winPos ul, short rows, short cols ) ;

//* Scan the directory tree beginning with the the current working directory   *
//* (CWD). Locate all files which match the user's search string.              *
//*                                                                            *
//* Allow user to optionally select a new CWD path to:                         *
//*   1) a directory whose filename matches the search criteria                *
//*   2) a directory which contains a file matching the search criteria        *
//*                                                                            *
//* Input  : newCwd    : (by reference) receives user's path selection         *
//*                      (cleared if no selection made)                        *
//*          ul        : (by reference) upper left corner dialog position      *
//*          rows      : number of rows for dialog                             *
//*          cols      : number of columns for dialog                          *
//* Returns: true if user has specified a new CWD ('newCwd' contains new path) *
//*          else false                                                        *
bool  FindFiles ( gString& newCwd, winPos& ul, short rows, short cols ) ;

//* Scan the the fileystem which contains the current-working-directory (CWD). *
//* Locate all files (hard links) which share:                                 *
//*  a) the user-specified 'inode'                                             *
//*  b) the inode of the highlighted file                                      *
bool  FindInodes ( gString& newCwd, winPos& ul, short rows, short cols ) ;

//* Scan the contents of the 'selected' files (and optionally directory trees) *
//* searching for strings that match the user's pattern.                       *
//*                                                                            *
//* This method sends the user's search string to the 'grep' utility and       *
//* captures the grep output to a temporary file.                              *
//*                                                                            *
//* Input  : ul        : (by reference) upper left corner dialog position      *
//*          rows      : number of rows for dialog                             *
//*          cols      : number of columns for dialog                          *
//* Returns: 'true'  if grep log saved to user's directory                     *
//*          'false' if grep log discarded                                     *
bool  GrepFiles ( const winPos& ul, short rows, short cols  ) ;

//* Allow/disallow scan of full directory tree when CWD is the root directory. *
//* This is a pass-through method to the FMgr-class method of the same name.   *
//*                                                                            *
//* Input  : enable     : if 'true',  recursive tree scan from root is enabled *
//*                       if 'false', recursive tree scan from root is disabled*
//*          reportOnly : (optional, 'false' by default)                       *
//*                       if 'false', internal flag is set/reset as indicated  *
//*                                   by 'enable' parameter                    *
//*                       if 'true',  state of internal flag is is not         *
//*                                   modified, only reported                  *
//* Returns: current state of recursive-scan flag                              *
bool  ScanFromRoot ( bool enable, bool report = false ) ;

//* Scan the source string (typically a filespec). If the string contains any  *
//* characters of the specified character group (enum escGroup) 'escape' them  *
//* with a backslash character.                                                *
//*                                                                            *
//* Do not use this method for URL (Universal Resource Locator) specs.         *
//* URL specifications require a specific protocol which is not handled here.  *
//*                                                                            *
//* This is a pass-through to the FMgr-class method of the same name.          *
//*                                                                            *
//* Input  : pathSpec : (by reference) source text data                        *
//*                     on return the data has been 'escaped' according to the  *
//*                     specified criteria                                     *
//*          group    : (member of enum escGroup) specifies the group of       *
//*                     characters to be 'escaped'                             *
//*          wch      : (optional, NULLCHAR by default)                        *
//*                     if 'group' parameter == escWCHAR, then 'wch' specifies *
//*                     a single character to be 'escaped'                     *
//*                                                                            *
//* Returns: number of characters 'escaped'                                    *
short EscapeSpecialChars ( gString& pathSpec, escGroup group, wchar_t wch = NULLCHAR ) ;

//* Construct the full filespec for the specified filesystem based on the      *
//* Uniform Resource Identifier (URI).                                         *
//* 1) MTP/GVfs URI:                                                           *
//*    Example URI: mtp://Android_ce012345c012345d01/                          *
//*    Filespec   : /run/user/1000/gvfs/mtp:host=Android_ce012345c012345d01    *
//* 2) Audio Disc URI:                                                         *
//*    Example URI: cdda://sr0/                                                *
//*    Filespec   : /run/user/1000/gvfs/cdda:host=sr0                          *
//*                                                                            *
//* This is a pass-through to the FMgr-class method of the same name.          *
//*                                                                            *
//* Input  : pathSpec : (by reference) receives the target filespec            *
//*          uri      : URI for an MTP/GVFS filesystem                         *
//*                                                                            *
//* Returns: 'true' if valid filespec,       (pathSpec initialized)            *
//*          'false' if invalid or unrecognized 'uri' format (pathSpec cleared)*
bool  Uri2Filespec ( gString& pathSpec, const char* uri ) ;

//* Eject the removable media from the specified device, or if no device       *
//* specified, then eject the media from the first DVD/CD drive.               *
//*    Example: EjectMedia( "/dev/sr0" ) ;                                     *
//* Typical devices with ejectable media: CD-ROM, DVD-ROM, floppy disk, tape   *
//* drive, ZIP disk or USB disk.                                               *
//*                                                                            *
//* 1) Not all optical drives can be closed under software control.            *
//* 2) If target device is mounted and writable, be sure to close the output   *
//*    stream or unmount the device before ejecting the tray to avoid potential*
//*    loss of data.                                                           *
//* 3) If target device does not support removable media, the operation will   *
//*    silently fail.                                                          *
//* 4) The 'close' and 'toggle' flags are mutually exclusive. If 'toggle' is   *
//*    set, then 'close' is ignored.                                           *
//*                                                                            *
//* Input  : device : (optional, NULL pointer by default)                      *
//*                   If not specified, media will be ejected from the         *
//*                    system's default DVD/CD drive: "/dev/cdrom"             *
//*                   If specified, device whose media are to be ejected.      *
//*          close  : (optional, 'false' by default)                           *
//*                   If 'false', open the tray.                               *
//*                   If 'true',  close it tray.                               *
//*          toggle : (optional, 'false' by default)                           *
//*                   If 'false', ignore.                                      *
//*                   If 'true', toggle the state of the tray.                 *
//*                                                                            *
//* Returns: nothing                                                           *
void  EjectMedia ( const char* device = NULL, bool close = false, bool toggle = false ) ;

//* Query whether Move-to-Trashcan operations have been properly set up.       *
//* Input  : none                                                              *
//* Returns: 'true'  if Trashcan operations are available                      *
//*          'false' if Trashcan operations have been disabled                 *
bool  TrashcanConfigured ( void ) ;

//* Initialize a progress bar to keep user amused. Called by Backup, Synch,    *
//* Copy, Move and others.                                                     *
//* 1) 'dlgColor', 'barColor', 'cells' and 'horiz' are explicitly set by the   *
//*    instantiation.                                                          *
//* 2) 'steps', 'border', 'yOffset' and 'xOffset' are explicitly set within    *
//*    this method.                                                            *
//* 3) All other pbarInit members receive default values.                      *
//* The progInit object is created in anticipation of the need for a progress  *
//* bar. The progress bar itself will be instantiated only if data to be       *
//* processed >= monitor threshold AND if the operation extends beyond the     *
//* grace period.                                                              *
//* Input  : cells   : size of progress bar in cells (rows or columns)         *
//*          yOff    : offset in Y from upper-left of application dialog       *
//*          xOff    : offset in X from upper-left of application dialog       *
//*          horiz   : (optional, 'true' by default)                           *
//*                    if 'true'  initialize as a horizontal ProgBar           *
//*                    if 'false' initialize as a vertical ProgBar             *
//* Returns: partially-initialized pbarInit object                             *
pbarInit ProgbarInit ( short cells, short yOff, short xOff, bool horiz = true ) ;

//* Returns the code and description for the most recent FileDlg error or      *
//* file system (FMgr) error.                                                  *
//* Note that the error information returned is not necessarily for the most   *
//* recent activity, but for the most recent activity that generated an error. *
//*                                                                            *
//* Input  : errorCode class object (by reference, initial contents ignored)   *
//*           (all fields initialized on return)                               *
//*                                                                            *
//* Returns: OK                                                                *
short GetErrorCode ( errorCode& eCode ) ;

   //************************************************
   //* Public methods for development and debugging *
   //************************************************

   //* Important Note: If user clears the clipboard while browsing, and any    *
   //* files previously have been marked as 'selected', then on return, caller *
   //* must de-select them and redraw the controls' display data to show that  *
   //* there are no longer 'selected' files in any view.                       *
   //* Input  : none                                                           *
   //* Returns: 'true' if clipboard cleared and we need to redisplay the data  *
   //*           else 'false'                                                  *
   bool  BrowseClipboard ( void ) ;

   //* for testing dialog layouts and color scheme only *
   short InvokeDialog ( short tstnum ) ;

   #if ENABLE_DEBUGGING_CODE != 0
   //* Methods for debugging only! - Disabled in distribution build            *
   NcDialog* Display_TreeNode ( winPos ul, const TreeNode& tn, bool stayOpen = false ) ;
   NcDialog* Display_tnFName ( winPos ul, tnFName& tnf, bool stayOpen = false ) ;
   NcDialog* Open_DBwindow ( attr_t& dColor, short& dLines, short&dCols ) ;
   #endif   // ENABLE_DEBUGGING_CODE

   const char* Get_FileDlg_Version ( void ) ;
   const char* Get_FMgr_Version ( void ) ;      // pass-through method

private:
   //*******************
   //* Private Methods *
   //*******************

   //* Prompting user for information or decisions *
   bool  GetNewFilename ( tnFName* tnfPtr, gString& newName ) ;
   bool  GetNewFilename ( gString& newName, const char* msg = NULL, 
                          const gString* suggName = NULL ) ;
   bool  gnfSpecialCharsQuery ( const gString& newName, NcDialog *cDialog, 
                                short tbIndex, tbChars filter ) ;
   bool  GetNewTimestamp ( localTime& newDate ) ;
   corReturn ConfirmOverwrite ( const tnFName* srcStats, const tnFName* trgStats ) ;
   corReturn cowPrompt ( const tnFName* srcStats, const tnFName* trgStats, 
                         short srcNewer ) ;
   bool  ConfirmPasteToTarget ( UINT64 bytesNeeded, UINT64 bytesAvail ) ;
   bool  GenericDialog ( const char** msgList, bool decision, 
                         const attr_t* msgAttr = NULL ) ;

   //* General utility methods *
   bool  VerifyTrashcanConfig ( void ) ;
   short rcdScan ( rcdsCaller callID, const gString* dirPath = NULL, bool hilight = true ) ;
   bool  rcdLaunchMonitor ( DispData& dData, thread** monThread ) ;
   void  rcdProgmon ( DispData* ddPtr, bool* active ) ;
   void  ProcessCWD ( bool clear = false ) ;
   void  DisplayStatus ( fdErrCode fdErrnum = ecNOERROR, short sysErrnum = ZERO ) ;
   fdErrCode SetErrorCode ( fdErrCode eCode, short erCode = -1 ) ;
   short CopyOrDeleteNotSupported ( fmFType fType, const char* srcName ) ;
   void  DisplayErrorCode ( void ) ;
   void  DisplayDirPath ( void ) ;
   void  DisplayFiles ( bool withHilight = true ) ;
   void  FillClipboard ( OpPend operation ) ;
   void  ClearClipboard ( bool freeAll = false ) ;
   UINT  MarkSelection ( attr_t selColor, bool& srcdirWPerm ) ;
   UINT  MarkSelectionNR ( attr_t selColor, bool& srcdirWPerm ) ;
   bool  TossSelection ( void ) ;
   short ResetFileDisplayColors ( void ) ;
   bool  CheckTargetSpace ( void ) ;
   bool  CurrDirPermission ( void ) ;
   bool  IsGroupMember ( gid_t gID ) ;
   void  FirstDisplayItem ( void ) ;
   void  NextDisplayItem ( void ) ;
   void  PrevDisplayItem ( void ) ;
   short FirstSelectedItem ( void ) ;
   short NextSelectedItem ( void ) ;
   short Scroll2MatchingFile ( const char* substr, bool sensi = false ) ;
   short HilightItemByName ( const gString& fName ) ;
   short HilightItemByName ( const char* fName ) ;
   bool  TargetExists ( const char* fPath, tnFName& tnf ) ;
   bool  TargetExists ( const char* fPath, bool& writePerm ) ;
   bool  TargetExists ( const char* fPath, fmFType& fType, bool& writePerm ) ;
   void  dfsDisplayFilename ( winPos& wp, attr_t color, const char* fPath, 
                              const char* fName, short maxDisp ) ;
   void  dfsDisplayStatData ( winPos& wp, attr_t color, ExpStats& es ) ;
   short dfsModifyFileStats ( short ctrY, short ctrX, ExpStats& es ) ;

   //* Group of methods to support public method, DisplayFileContents()        *
   short vfcDecodeSymlink ( const gString& trgPath, gString& tmpPath ) ;
   short vfcDecodeBinary ( const gString& trgPath, gString& tmpPath ) ;
   short vfcDecodeArchive ( const gString& trgPath, gString& tmpPath ) ;
   short vfcDecodeOpenDoc ( const gString& srcPath, gString& tmpPath, arcType aType ) ;
   short vfcDecodeBinDoc ( const gString& srcPath, gString& tmpPath ) ;
   short vfcDecodeMetadata ( const gString& trgPath, gString& tmpPath,
                             MediafileType mfType ) ;
   short vfcDecodeEmail ( const gString& trgPath, gString& tmpPath ) ;
   bool  vfcFormatHTML_Source ( const gString& trgPath, const gString& tmpPath ) ;
   bool  vfcFormatXML_Source ( const gString& trgPath, const gString& tmpPath ) ;
   bool  vfcFormatPERL_Source ( const gString& srcPath, const gString& tmpPath ) ;
   bool  vfcIsTextfile ( const gString& gsTrg ) ;
   short vfcGetLine ( std::ifstream& ifs, gString& gsIn ) ;
   MediafileType MediafileTarget ( const gString& trgPath ) ;
   void  ExtractMetadata_MP3 ( ofstream& ofs, const gString& srcPath, bool verbose = false ) ;
   void  ExtractMetadata_OGG ( ofstream& ofs, const gString& srcPath, bool verbose = false ) ;
   void  ExtractMetadata_M4A ( ofstream& ofs, const gString& srcPath, 
                               const gString& trgPath, bool verbose = false ) ;
   void  ExtractMetadata_ASF ( ofstream& ofs, const gString& srcPath, bool verbose = false ) ;
   void  ExtractMetadata_PNG ( ofstream& ofs, const gString& srcPath ) ;
   void  ExtractMetadata_JPG ( ofstream& ofs, const gString& srcPath ) ;

   //* Group of methods to support public method, CompareFiles()               *
   bool  cfSetCompFiles ( gString& fSpecA, gString& fNameA, tnFName& fStatA, 
                          gString& fLinkA, gString& fSpecB, gString& fNameB, 
                          tnFName& fStatB, gString& fLinkB, gString& logStats, 
                          gString& logHead, gString& errMsg, 
                          short termCols ) ;
   bool  cfPretest ( const gString& fSpecA, const gString& fNameA, 
                     const gString& fSpecB, const gString& fNameB, 
                     gString& logIdent, short termCols ) ;
   void  cfConstructCmd ( char* cmdBuff, short tcOption, short ocOption, 
                          short termCols, const gString& fSpecA, 
                          const gString& fSpecB, const gString& logFile, 
                          bool brief, bool binfiles, bool twocol ) ;
   void  cfPostprocLog ( const gString& logFile, const gString& fNameA,
                         const gString& fNameB, const gString& logHead, 
                         const gString& logIdent, 
                         bool brief, bool twocol, bool context, bool binfiles ) ;
   void  cfpplCopyAll ( std::ifstream& ifs, std::ofstream& ofs, char* lineData, 
                        bool binary = false ) ;
   void  cfpplLegend ( std::ifstream& ifs, std::ofstream& ofs, char* lineData ) ;
   void  cfDisplayLog ( const gString& logFile, NcDialog* ddp ) ;
   short cfSaveLog ( const gString& logFile, const gString&fNameA, gString& logName ) ;

   //* Group of methods to support public method, PasteClipboardList()         *
   short CopyFile ( const tnFName& sStats, const gString& srcPath, 
                    const gString& trgPath, bool lnkTarget = false ) ;
   short DeleteSourceFile ( const tnFName* srcStats, const gString& srcPath ) ;
   UINT  pclCopyTnList ( pclParm& pcl, bool sDirEQtDir = false ) ;
   UINT  pclCopyFileList ( pclParm& pcl, bool sDirEQtDir = false ) ;
   UINT  pclDeleteTnList ( pclParm& pcl ) ;
   UINT  pclDeleteFileList ( pclParm& pcl ) ;
   corReturn pclGetNewFilename ( const tnFName* srcStats, gString& trgFName, bool sDirEQtDir ) ;
   bool  pclFNameInUse ( const gString& trgName, bool& prompt4newName ) ;
   void  pclCreateSuggestedName ( gString& suggName, const gString& origName ) ;
   bool  pclCTLerror ( const char* dirName, short errnoVal ) ;

   //* Group of methods to support public method, PasteSpecialList()           *
   UINT  pslSinglePaste ( const tnFName* srcStats, pclParm& pcl, 
                          bool tDirEQsDir, bool sameFileSys ) ;
   psSubType pslSinglePaste_Options ( const tnFName* srcStats, tnFName& ltargStats, 
                                      gString& trgFName, bool tDirEQsDir, bool sameFileSys ) ;
   UINT  pslMultiPaste ( pclParm& pcl, bool tDirEQsDir, bool sameFileSys ) ;
   psSubType pslMultiPaste_Options ( renameOption& renOption, UINT regCount, 
                                     UINT dirCount, UINT totCount, UINT64 totBytes,
                                     bool tDirEQsDir, bool sameFileSys ) ;
   short pslPasteSpecial ( psSubType opType, const tnFName* srcStats, 
                           const gString& srcPath, const gString& trgPath ) ;
   corReturn pslTargetExists ( const gString& trgPath, const gString& srcPath, 
                               const tnFName* srcStats, psSubType pOpt ) ;
   UINT  pslCopyTnList ( pclParm& pcl ) ;
   UINT  pslCopyFileList ( pclParm& pcl, bool rPrompt = false, 
                           bool sDirEQtDir = false, renamePattern* rPat = NULL ) ;

   //* 'Backup/Synchronize files' group of methods.                            *
   //* Called through public methods, BackupClipboardList() or                 *
   //* ArchiveClipboardList().                                                 *
   UINT  bkCopyTnList ( pclParm& pcl, const BSet& bkSet, bool parentCreated = false ) ;
   UINT  bkCopyFileList ( pclParm& pcl, const BSet& bkSet, bool parentCreated = false ) ;
//   UINT  bkScanTnList ( pclParm& pcl, const BSet& bkSet, bool parentCreated = false ) ;
   UINT  bkScanTnList ( pclParm& pcl, const BSet& bkSet ) ;
   UINT  bkScanFileList ( pclParm& pcl, const BSet& bkSet ) ;
   bool  bkNeedsUpdate ( const gString& srcPath, const tnFName* srcStats, 
                         const gString& trgPath, const BSet& bkSet, UINT* attPtr ) ;
   bool  bkVerifyTargetDir ( const gString& trgPath, bool& trgCreated, 
                             bool testOnly = false ) ;
   bool  bkOpenLogfile ( const BSet& bkSet ) ;
   bool  bkWriteLogfile ( const gString& logData, bsetLog filter = blVERBOSE, 
                          short rectype = ZERO, bool lf = true ) ;
   bool  bkWriteLogfile ( const char* logData, bsetLog filter = blVERBOSE, 
                          short rectype = ZERO, bool lf = true ) ;
   void  bkCloseLogfile ( void ) ;
   bool  AbortQuery ( void ) ;
   void  pbManager ( pbarInit* pbi, bool* active ) ;

   //* 'Archive' group of methods. Called through public Archive() method.     *
   short Archive_Prompt ( BSet& bkSet ) ;
   bsetType Archive_UpdateQuery ( const BSet& bkSet, const gString& hlPath,
                                  arcType hlaType, arcType cbaType, UINT cbFiles ) ;
   arcType ArchiveOnClipboard ( gString& arcPath ) ;
   bool  ScanClipboard ( const gString& srcSpec ) ;
   short Archive_Setup ( BSet& bkSet ) ;
   short Archive_Create ( BSet& bkSet ) ;
   short acCreateTarArchive ( BSet& bkSet, bool update = false ) ;
   short acCreateZipArchive ( BSet& bkSet, bool update = false ) ;
   short acVerifyArchive ( BSet& bkSet, bool update = false ) ;
   short Archive_Append ( BSet& bkSet ) ;
   short Archive_Expand ( BSet& bkSet ) ;
   short aeExpandTarArchive ( BSet& bkSet ) ;
   short aeExpandZipArchive ( BSet& bkSet ) ;
   void  Archive_Summary ( const BSet& bkSet, short opStatus ) ;
   short Archive_FormatLog ( const BSet& bkSet, const gString& trgLog, 
                             const gString& summary ) ;
   short Archive_SaveLog ( const gString& srcLog, const gString& trgLog ) ;
   short Archive_FileList ( const gString& fileList, bool zipWild = false ) ;

   //* 'Move-to-Trashcan' group and Empty/Restore trashed files group.         *
   //*  Called through public methods, TrashcanSelectedFiles(), and            *
   //*  ManageTrashcan().                                                      *
   bool  tsfTopFile ( const tnFName& srcInfo, pclParm& pcl, bool ignoreErrors ) ;
   bool  tsfMoveDirTree ( pclParm& pcl ) ;
   short tsfDeleteSource ( const tnFName& srcInfo, const gString& srcPath ) ;
   void  tsfConstructDeletionDate ( gString& timeStamp ) ;
   bool  tsfSourceEqTrashcan ( void ) ;
   bool  tsfAlert ( const char* srcName, short errType ) ;
   void  tcDialog ( bool restore = false ) ;
   int   tcScroll ( NcDialog* dp, uiInfo& info, 
                    ssetData& sData, tcDataInfo* tcdi, short& sOption ) ;
   bool  tcSummaryScan ( tcSummaryInfo& tcSummary ) ;
   UINT  tcDetailScan ( char*& blkPtr, ssetData& sData, 
                        tcDataInfo*& tcdi, short dialogCols, short sOption ) ;
   bool  tcDecodeLog  ( const gString& srcPath, gString& trgPath, 
                        gString& delDate, UINT& trgFiles, UINT64& trgSize, 
                        fmFType& trgType ) ;
   void  tcDecodeDate ( const gString& trashedDate, localTime& tDate ) ;
   UINT  tcRemoveConfirm ( NcDialog* dpp, const ssetData& sData, 
                           const tcDataInfo* tcdi ) ;
   UINT  tcDeleteSelectedFiles ( void ) ;
   UINT  tcRestoreConfirm ( NcDialog* dpp, ssetData& sData, tcDataInfo* tcdi, 
                            int rCount, const gString& userCwd, gString& statMsg,
                            bool* res2Cwd ) ;
   bool  tcrValidateTarget ( ssetData& sData, tcDataInfo* tcdi, 
                             const gString& altTrg, gString& errMsg ) ;
   bool  tcrTargetTest ( const tcDataInfo& tcItem, gString& errMsg ) ;
   UINT  tcRestoreItem ( const tcDataInfo& tcItem, UINT& delFiles ) ;
   UINT  tcMarkSelection ( const ssetData& sData, const tcDataInfo* tcdi, 
                           bool infoDir, OpPend opCode ) ;
   UINT  tcMarkSelection ( const tcDataInfo& tcItem, bool infoDir, OpPend opCode ) ;

   //* 'Delete-file' group of methods.                                         *
   //*  Called through public method, DeleteSelectedFiles().                   *
   UINT  dsfDeleteDirTree ( TreeNode* tnPtr, const char* sPath, UINT& errCount ) ;
   UINT  dsfDeleteFileList ( const tnFName* tnf, UINT fileCount, const char* sPath ) ;
   short dsfDeleteRegFile ( const tnFName& tnf, const gString& sPath ) ;
   short dsfDeleteDirectory ( const tnFName& tnf, const gString& sPath ) ;
   short dsfDeleteChdevFile ( const tnFName& tnf, const gString& sPath ) ;
   short dsfDeleteBkdevFile ( const tnFName& tnf, const gString& sPath ) ;
   short dsfDeleteSocketFile ( const tnFName& tnf, const gString& sPath ) ;
   short dsfDeleteUnsuppFile ( const tnFName& tnf, const gString& sPath ) ;

   //* 'Rename-file' group of methods.                                         *
   //*  Called through public method, RenameSelectedFiles().                   *
   short rsfRenameFile ( const tnFName* tnf, const char* sPath, const gString& newFName ) ;
   short rsfRenameChdevFile ( const tnFName* tnf, const char* sPath, const gString& newFName ) ;
   short rsfRenameBkdevFile ( const tnFName* tnf, const char* sPath, const gString& newFName ) ;
   short rsfRenameSocketFile ( const tnFName* tnf, const char* sPath, const gString& newFName ) ;
   short rsfRenameUnsuppFile ( const tnFName* tnf, const char* sPath, const gString& newFName ) ;
   renameOption rsfGetRenamePattern ( renamePattern* rPat, bool 
                                      sDirEQtDir = false, const char* msg = NULL ) ;
   renameOption rsfConstructPattern ( renamePattern* rPat, 
                               short dLines, short dCols, short dulY, short dulX ) ;
   void  rsfCP_Help ( short dLines, short dCols, short dulY, short dulX ) ;
   bool  rsfAlert ( const char* origName, const char* errMsg = NULL ) ;

   //* File read/write protection group of methods.*
   bool  SourcePermission ( UINT& wpCount, UINT& rpCount, bool wEnable = false, 
                            bool rEnable = false, bool silent = false ) ;
   void  spEnable ( UINT& weCount, UINT& reCount, tnFName* tnfPtr, UINT tnfCount, 
                    const gString& dirPath, bool wEnable = false, bool rEnable = false ) ;
   void  spEnable ( UINT& weCount, UINT& reCount, TreeNode* tnPtr,  
                    const gString& dirPath, bool wEnable = false, bool rEnable = false ) ;
   bool  spDecision ( UINT wpCount, UINT rpCount, enum OpPend opPending ) ;

   UINT  SourceWriteEnable ( bool writeEnable = false ) ;
   UINT  SourceWriteProtect ( bool writeProtect = false ) ;
   UINT  SourceWProtect ( TreeNode* tnPtr, 
                          const gString& dirPath, bool enable = false ) ;
   UINT  SourceWProtect ( tnFName* tnf, UINT tnfCount, 
                           const gString& dirPath, bool enable = false ) ;
   UINT  SourceSetPermission ( const ExpStats& newPerms, bool xwd ) ;
   UINT  sspSet ( const ExpStats& eStats, const tnFName* tnfPtr, 
                  UINT tnfCount, bool xwd ) ;
   UINT  sspSet ( const ExpStats& eStats, TreeNode* tnPtr, bool xwd ) ;
   short AdjustPermissionBits ( ExpStats& newMask, bool& xwd, bool protect ) ;
   bool  FileWriteProtection ( tnFName& fStats, const gString& fPath, 
                               bool writeProtect = false ) ;

   //* Methods for display and user-traversal of the file tree *
   NcDialog* dftOpenDialog ( const winPos& ul, short rows, short cols, attr_t iColor ) ;
   TreeNode* dftCaptureTree ( const gString& gsBase, NcDialog* dp, DispData* ddPtr ) ;
   void  dftProgmon ( DispData* ddPtr, bool* active, NcDialog* dp ) ;
   void  dftFormatTree ( dftDisplay& dftd, TreeNode* baseNode ) ;
   UINT  dftFormatNode ( dftDisplay& dftd, TreeNode* tnPtr, UINT di ) ;
   short dftDisplayTree ( NcDialog* dp, dftDisplay& dftd, gString& newcwd ) ;
   bool  dftNamePrompt ( const winPos& spPos, gString& ptrn ) ;

   //* Methods for finding files in the directory tree *
   NcDialog* ffOpenDialog ( const winPos& ul, short rows, short cols, bool fi = false ) ;
   bool  ffDynamicAllocation ( ssetData& sData, bool noJoy = true, bool release = false ) ;
   void  ffTextboxMonitor ( NcDialog* dp, ssetData* sData, const bool* retFlag, short dCols ) ;
   void  ffSortRecords ( ssetData* sData, int fMatch, int tsFormat, bool ascend = false ) ;
   int   ffScanTree ( const gString& substr, ssetData& sData, int tsFmt ) ;
   int   ffScanNode ( const TreeNode* nodePtr, const gString& substr, 
                      const gString& tnPath, ssetData& sData, int tsFmt ) ;
   int   ffScanList ( const tnFName* listPtr, UINT listCnt, const gString& substr, 
                      const gString& listPath, ssetData& sData, int tsFmt ) ;
   bool  ffGetSelectedItem ( gString& newCwd, ssetData& sData, short iIndex ) ;
   void  ffAdjustOutput ( ssetData& sData, short dCols ) ;

   //* Methods for finding files with a specific 'inode' *
   int   fiScanInodes ( NcDialog* dp, ssetData& sData, const gString& origCwd, short dCols ) ;
   int   fiScanTree ( UINT64 refInode, ssetData& sData ) ;
   int   fiScanTree ( const gString& origCwd, ssetData& sData, NcDialog* dp ) ;
   int   fiScanNode ( const gString& tnPath, const TreeNode* nodePtr, 
                      ssetData& sData, UINT64 refInode ) ;
   int   fiScanNode ( const gString& tnPath, const TreeNode* nodePtr, ssetData& sData ) ;
   int   fiScanList ( const gString& listPath, const tnFName* listPtr, 
                      ssetData& sData, UINT listCnt, UINT64 refInode ) ;
   int   fiScanList ( const gString& listPath, const tnFName* listPtr, 
                      ssetData& sData, UINT listCnt ) ;
   int   fiCompare ( const gString& pathSpec, const tnFName& fStats, 
                     ssetData& sData, UINT64 refInode ) ;


   //* Methods for grep of source files *
   bool  gfScan ( const gString fnList, const gString loList, const gString& gsPat, 
                  const gString& gsFilt, const gString& gsCap, 
                  bool lineNum, bool caseSen, bool matchCnt ) ;
   short gfExtractFname ( const gString& loList, short indx, 
                          gString& srcName, bool strip = true ) ;
   bool  gfConstructFilenameList ( gString& fnList, gString& odList, short& scanCount ) ;
   bool  gfValidateFilenameList ( gString& fnList, gString& odList, short& scanCount ) ;
   bool  gfIsRegTextfile ( const gString& fileSpec, 
                           const gString& tmpFile, bool checkReg = false ) ;
   bool  gfIsRegexpPattern ( const gString& fnList ) ;
   bool  gfValidateSearchPattern ( gString& gsPat ) ;
   bool  gfDelimitText ( gString& gsSrctxt, bool forceDelim = false ) ;

   //* Methods specific to OpenDocument and Office Open XML documents *
   bool  odIsOpenDocFile ( const gString& fileSpec, bool all = false ) ; 
   bool  odIsOpenDocBackup ( const gString& fileSpec ) ;
   bool  odIsOpenXmlFile ( const gString& fileSpec, bool all = false ) ; 
   bool  odIsBinDocFile  ( const gString& fileSpec, bool all = false ) ; 
   bool  odExtractOD_Content ( const gString& srcSpec, const gString& tmpPath, 
                               gString& conSpec, arcType aType ) ;
   bool  odExtractOD_Text ( const gString& srcSpec, const gString& conSpec, 
                            const gString& capSpec, arcType aType, 
                            NcDialog* dbPtr = NULL, winPos* dbwp = NULL ) ;
   void  odParseOX_xlsx ( const gString& srcSpec, std::ifstream& ifs, 
                          std::ofstream& ofs, short tCols ) ;
   void  odParseOX_pptx ( const gString& srcSpec, std::ofstream& ofs, short tCols ) ;

   //* Group of methods for displaying clipboard data *
   bool  BrowseCbDetail ( NcDialog* dp, char* blkPtr, ssetData& sData ) ;
   void  bcbStandardView ( char* blkPtr, ssetData& sData ) ;
   void  bcbDetailedView ( char* blkPtr, ssetData& sData ) ;
   void  bcbDetailedLevel ( TreeNode& tnPtr, char*& strPtr, ssetData& sData ) ;


   //*******************
   //* Private Data    *
   //*******************
   FMgr*       fmPtr ;        //* Pointer to instance of FMgr class
   NcDialog*   dPtr ;         //* Pointer to NcDialog object (from caller)
   tnFName**   deList ;       //* Pointer to an array of tnFName pointers
   char**      textData ;     //* Pointer to an array of character pointers
   attr_t*     colorData ;    //* Pointer to an array of color attributes
   short       dulY ;         //* Parent dialog's position in terminal window
   short       dulX ;         
   short       dRows ;        //* Parent dialog's dimensions
   short       dCols ;        
   short       fIndex ;       //* Index of file-display control
   short       fulY ;         //* Upper left Y of file display control
   short       fulX ;         //* Upper left X of file display control
   short       fMaxY ;        //* Display rows in file display control
   short       fMaxX ;        //* Display columns in file display control
   short       pIndex ;       //* Index of path-display control
   short       pMax ;         //* Max width of path display
   attr_t      pNcolor ;      //* Path string normal color
   attr_t      pFcolor ;      //* Path string special color
   short       sIndex ;       //* Index of statistics-display control
   short       sMax ;         //* Max length of statistics message
   attr_t      sNcolor ;      //* Stats message normal color
   attr_t      sFcolor ;      //* Stats message special color
   short       mIndex ;       //* Index of status-message control
   short       mulY ;         //* Upper left Y of status-message control
   short       mulX ;         //* Upper left X of status-message control
   short       mMax ;         //* Max width of status message
   attr_t      mNcolor ;      //* Status message normal color
   attr_t      mFcolor ;      //* Status message special color
   ColorScheme cs ;           //* Application 'Color Scheme' for subdialogs etc.
   UINT        hIndex ;       //* Index of hilighted display item

   //* Number of files in current directory (NOT incl. subdir contents)        *
   UINT        fileCount ;
   //* Total bytes for all files in current directory (NOT incl.subdir content)*
   UINT64      dirSize ;

   fileSelect  selInfo ;               //* File 'selection' data
   char        currDir[MAX_PATH] ;     //* Full path string for current directory
   char**      EmptyStr ;              //* Display string pointer for empty directory
   attr_t*     EmptyAttr ;             //* Display attribute pointer for empty directory
   attr_t      selectAttr ;            //* Color attribute bit mask for file-select
   attr_t      copyColor ;             //* Color and attribute for file(s) in "copy" state
   attr_t      cutColor ;              //* Color and attribute for file(s) in "cut" state
   attr_t      deleteColor ;           //* Color and attribute for file(s) in "delete" state
   attr_t      renameColor ;           //* Color and attribute for file(s) in "rename" state
   attr_t      touchColor ;            //* Color and attribute for file(s) in "touch" state
   fdErrCode   recentError ;           //* Error code for most recent FileDlg error
   short       recentErrno ;           //* Error code for most recent FMgr error (system's errno)
   bool        autoRecurse ;           //* 'true' _except_ when scanning a very large external disc
   bool        trashcanOK ;            //* 'true' if Trashcan directories are accessible
   char        trashDir[MAX_PATH] ;    //* Path to application's 'Trash' directory
   char        tempDir[MAX_PATH] ;     //* Path to application's 'Temp' directory
   UserInfo    userInfo ;              //* Information on application-user's account
   ofstream    ofsLog ;                //* pointer to open logfile for Backup/Archive/Synch operations
   WorkInProgress wip ;                //* Tracks operations in progress for providing visual feedback
   ConfigOptions cfgOptions ;          //* Copy of configuration options sent from mainline code


   //* Color attribute table for display of the eight different file types. *
   //* Indexed access through enum fmFType defined in FMgr.hpp.             *
   attr_t ftColor[fmTYPES] ;

} ;

#endif   // FILEDLG_INCLUDED


