//********************************************************************************
//* File       : FMgrDispData.hpp                                                *
//* Author     : Mahlon R. Smith                                                 *
//*              Copyright (c) 2005-2025 Mahlon R. Smith, The Software Samurai   *
//*                  GNU GPL copyright notice located in FileMangler.hpp         *
//* Date       : 08-Jun-2025                                                     *
//* Version    : (see FMgrVersion string in FMgr.cpp)                            *
//*                                                                              *
//* Description: This class implements a wrapper for the multi-threaded capture  *
//* of directory-tree data. Because the number of subdirectories and files       *
//* in a directory tree may number in the thousands, the FMgr:CaptureDirTree()   *
//* method creates a separate thread of execution for each branch of the tree.   *
//* Then, when the entire tree has been scanned, the sub-threads are deleted.    *
//*                                                                              *
//* IMPORTANT NOTE: This class requires a version of the C++ standard library    *
//*                 which fully supports the C++11 standard, specifically the    *
//*                 std:thread class, the std::mutex class and the std::chrono   *
//*                 class.                                                       *
//*                                                                              *
//* Developed using GNU G++ (Gcc v: 4.8.0 and libstdC++.so.6.0.18)               *
//*  under Fedora Release 16, Kernel Linux 3.6.11-4.fc16.i686.PAE                *
//********************************************************************************
//* Version History (most recent first):                                         *
//*   See version history in FMgr.cpp                                            *
//*                                                                              *
//********************************************************************************

//********************************************************************************
//* Data exchange for calling methods that read directory data:                  *
//* RefreshCurrDir(), CdChild(), CdParent(), CdPath(), etc.                      *
//*                                                                              *
//* During the capture process, certain members are protected by a mutex which   *
//* allows multiple threads to access the data in an orderly way. This means     *
//* that a thread that wants to read/write the data may block while waiting      *
//* for the requested resources.                                                 *
//*                                                                              *
//* Use of this class requires some cooperation on the part of the thread that   *
//* calls the constructor, in that it must not destroy the object until all      *
//* secondary threads have completed their assignments. Failure to abide by      *
//* this condition may result in a wait-forever condition OR a system            *
//* exception (crash-and-burn).                                                  *
//*                                                                              *
//********************************************************************************
class DispData
{
   public:
   //********************
   //** Public Methods **
   //********************

   //* Destructor. *
   ~DispData ( void )
   {  //* Wait for all executing sub-threads to finish *
      //* before allowing ourselves to be deleted.     *
      chrono::duration<short, std::milli>aMoment( 50 ) ;
      while ( (this->getActiveThreads ()) != ZERO )
         this_thread::sleep_for( aMoment ) ;
   }

   //* Constructor.                            *
   //* Width of display data must be specified.*
   //* All other data set to default values.   *
   DispData(USHORT strwidth) : strWidth(strwidth) 
   {
      this->reset() ;
   }

   //* Initialize caller's parameters *
   void getData ( tnFName**& list, char**& tData, attr_t*& cData,
                  UINT& fCount, UINT64& dSize )
   {
      this->getDataLock () ;        //* Get a lock on the data
      list     = this->deList ;
      tData    = this->textData ;
      cData    = this->colorData ;
      fCount   = this->listCount ;
      dSize    = this->listBytes ;
      this->freeDataLock () ;       //* Release the data lock
   }

   //* Returns a count of the threads still gathering data.   *
   //* When count has reached ZERO, the secondary threads are *
   //* 'join'ed with caller, i.e. the main application thread.*
   //* The scan-manager thread is also 'join'ed.              *
   USHORT getActiveThreads ( void )
   {
      this->getDataLock () ;        //* Get a lock on the data
      if ( this->activeThreads == ZERO && this->totalThreads > ZERO )
      {  //* Sub-threads have all exited, but their dynamic data still exist.*
         //* Call 'join' for each thread, and then delete the dynamic data.  *
         for ( USHORT stIndex = ZERO ; stIndex < this->totalThreads ; stIndex++ )
         {
            this->scanThread[stIndex].join() ;
         }
         delete [] this->scanThread ; this->scanThread = NULL ;
         delete [] this->scanThreadPath ; this->scanThreadPath = NULL ;
         this->totalThreads = ZERO ;

         //* Terminate and delete the manager thread.*
         // Programmer's Note: The manager thread will almost certainly 
         // finish before the sub-threads finish, so there should be no 
         // delay here (unless the debugging log is active).
         if ( this->mgrThread != NULL )
         {
            chrono::duration<short, std::milli>aMoment( 10 ) ;
            while ( this->mgrthreadActive )
               this_thread::sleep_for( aMoment ) ;
            this->mgrThread->join() ;
            delete [] this->mgrThread ; this->mgrThread = NULL ;
         }
      }

      USHORT ot = this->activeThreads ;   // return value
      this->freeDataLock () ;       //* Release the data lock
      return ot ;
   }

   //* Unlike the getActiveThreads() method above, simply *
   //* return the number of currently active threads.     *
   USHORT getActiveThreads_snapshot ( void )
   {
      this->getDataLock () ;        //* Get a lock on the data

      USHORT ot = this->activeThreads ;   // return value

      this->freeDataLock () ;       //* Release the data lock
      return ot ;
   }

   bool autoRecursion ( bool enable )
   {
      this->getDataLock () ;        //* Get a lock on the data
      bool ar = this->autoRecurse ;
      this->autoRecurse = enable ;
      this->freeDataLock () ;       //* Release the data lock
      return ar ;
   }

   //* Increment the exception counter. Called if     *
   //* exception thrown while launching sub-thread.   *
   //* Append the new message to existing messages.   *
   void setException ( const char* exmsg )
   {
      this->getDataLock () ;        //* Get a lock on the data
      ++this->sysExceptions ;
      //* Append the new message to existing messages.*
      if ( this->exmIndex > ZERO )  --this->exmIndex ;
      for ( short src = ZERO ; (this->exmIndex < (gsDFLTBYTES - 1)) ; ++src )
      {
         if ( exmsg[src] != NULLCHAR )
         {
            this->exMsg[this->exmIndex++] = exmsg[src] ;
         }
         else
         {
            this->exMsg[this->exmIndex++] = NEWLINE ;
            this->exMsg[this->exmIndex++] = NULLCHAR ;
            break ;
         }
      }
      this->exMsg[this->exmIndex] = NULLCHAR ;  // be sure string is terminated
      this->freeDataLock () ;       //* Release the data lock
   }

   //* Clear the 'exMsg' member and reset the 'exmIndex' member.*
   void resetException ( void )
   {
      this->getDataLock () ;        //* Get a lock on the data
      this->exMsg[ZERO] = NULLCHAR ;
      this->exmIndex = ZERO ;
      this->freeDataLock () ;       //* Release the data lock
   }

   //* Returns the number of system exceptions "caught" *
   //* during the scan of the directory tree.           *
   USHORT getExceptions ( const char*& msgPtr )
   {
      msgPtr = this->exMsg ;
      return this->sysExceptions ;
   }

   //* Each sub-thread launched successfully will call this method.*
   void goodSubthread ( void )
   {
      this->getDataLock () ;        //* Get a lock on the data
      ++this->goodSubs ;
      this->freeDataLock () ;       //* Release the data lock
   }

   //* Returns count of sub-threads successfully launched.*
   USHORT goodSubthreads ( void )
   {
      return this->goodSubs ;
   }

   //* Initialize our private data using caller's info *
   void setData ( tnFName** list, char** tData, attr_t* cData,
                  UINT lCount, UINT64 lBytes )
   {
      this->getDataLock () ;        //* Get a lock on the data
      this->deList    = list ;
      this->textData  = tData ;
      this->colorData = cData ;
      this->listCount = lCount ;
      this->listBytes = lBytes ;
      this->freeDataLock () ;       //* Release the data lock
   }

   //* Update (add to) our private data using caller's info *
   void updateData ( UINT fCount, UINT64 fBytes )
   {
      this->getDataLock () ;        //* Get a lock on the data
      this->treeCount += fCount ; 
      this->treeBytes += fBytes ;
      this->freeDataLock () ;       //* Release the data lock
   }

   //* Get only the cumulative file count and total  *
   //* tree size. These will continue to be updated  *
   //* until all sub-threads have finished.          *
   void getData ( UINT& fCount, UINT64& dSize )
   {
      this->getDataLock () ;        //* Get a lock on the data
      fCount   = this->treeCount ;
      dSize    = this->treeBytes ;
      this->freeDataLock () ;       //* Release the data lock
   }

   //* The user-interface (progress monitor) thread must wait a finite * 
   //* amount of time before starting to report the scan progress.     *
   //* This flag indicates that user interface may continue.           *
   void unblockUI ( void )
   {
      this->getDataLock () ;        //* Get a lock on the data
      this->uiEnable = true ;
      this->freeDataLock () ;       //* Release the data lock
   }

   //* Called by the user-interface layer to determine whether the *
   //* user-interface thread may begin to report scan progress.    *
   bool uiReport ( void )
   {
      return this->uiEnable ;
   }

   //* This flag is set to indicate that the recursive scan of *
   //* the directory was initiated from Tree-view mode, and    *
   //* indicates that only directory file types are to be fully*
   //* decoded. Non-directory entries are counted, but are not *
   //* fully decoded.                                          *
   bool setTreeViewFlag ( void )
   {
      return ( this->treeView = true ) ;
   }
   bool getTreeViewFlag ( void )
   {
      return this->treeView ;
   }

   //* Called by the user-interface layer (manager thread) to signal that  *
   //* all active threads should terminate immediately. Proper function    *
   //* requires that the sub-threads monitor the 'abortSignal' flag.       *
   //* This is done via a call to getAbortFlag().                          *
   bool setAbortFlag ( void )
   {
      return ( this->abortSignal = true ) ;
   }

   //* Returns the current state of the 'abortSignal' flag. *
   // Programmer's Note: The problem with signalling an abort is that 
   // the actual recursive scan loops do not have access to the DispData 
   // object. Therefore the abort signal is currently nonfunctional.
   bool getAbortFlag ( void )
   {
      return this->abortSignal ;
   }

   //* Create an array of sub-threads and storage for their associated paths.  *
   //* On return, caller will activate the threads individually.               *
   //* NOTE: We do not throw an exception in the case of an allocation error,  *
   //*       so it is the caller's responsibility to check the return value.   *
   USHORT createSubthreads ( USHORT threadCount )
   {
      this->getDataLock () ;        //* Get a lock on the data
      chrono::duration<short, std::milli>aMoment( 30 ) ;
      USHORT allocatedThreads = ZERO ;    // return value
      if ( this->totalThreads == ZERO )   // do not allow multiple allocations
      {
         for ( short i = 3 ; i > ZERO ; --i ) // three strikes, and you're out!
         {
            if ( ((this->scanThread = 
                  new (std::nothrow) std::thread[threadCount]) != NULL)
                 && 
                 ((this->scanThreadPath = 
                  new (std::nothrow) gString[threadCount]) != NULL) )
            {
               this->totalThreads = 
               this->activeThreads = allocatedThreads = threadCount ;
               break ;
            }
            else
            {
               if ( this->scanThread != NULL )
               { delete [] this->scanThread ; this->scanThread = NULL ; }
               if ( this->scanThreadPath != NULL )
               { delete [] this->scanThreadPath ; this->scanThreadPath = NULL ; }
               this_thread::sleep_for( aMoment ) ;  // give the system some time
            }
         }
      }
      if ( allocatedThreads == ZERO ) // if dynamic allocation failed
      {
         ++this->sysExceptions ;
         const char* errMsg = "Scan sub-thread(s) not allocated." ;
         ++this->sysExceptions ;
         short s = ZERO, t = -1 ;
         while ( (this->exMsg[++t] = errMsg[s++]) != NULLCHAR ) ;
      }
      this->freeDataLock () ;       //* Release the data lock
      return allocatedThreads ;
   }

   //* Create a manager thread to monitor the 'scanThread' array.   *
   //* On success 'mgrThread' != NULL && 'mgrthreadActive' != false.*
   void createMgrthread ( void )
   {
      this->getDataLock () ;        //* Get a lock on the data
      chrono::duration<short, std::milli>aMoment( 10 ) ;
      if ( this->mgrThread == NULL )   // do not allow multiple allocations
      {
         for ( short i = 3 ; i > ZERO ; --i ) // three strikes, and you're out!
         {
            if ( (this->mgrThread = new (std::nothrow) std::thread[1]) != NULL )
            {
               this->mgrthreadActive = true ;
               break ;
            }
            this_thread::sleep_for( aMoment ) ;  // give the system some time
         }
      }
      if ( this->mgrThread == NULL )
      {
         const char* errMsg = "Scan-manager thread not allocated." ;
         ++this->sysExceptions ;
         short s = ZERO, t = -1 ;
         while ( (this->exMsg[++t] = errMsg[s++]) != NULLCHAR ) ;
      }
      this->freeDataLock () ;       //* Release the data lock
   }

   //* Called by the manager thread when it exits.*
   // Programmer's Note: There is no access lock on this data member because 
   // if the debugging code is active, we would get into a race condition.
   // (See this->getActiveThreads().)
   void completedMgrthread ( void )
   {
      //this->getDataLock () ;        //* Get a lock on the data
      this->mgrthreadActive = false ;
      //this->freeDataLock () ;       //* Release the data lock
   }

   //* Called by secondary threads when they have finished. *
   //* The count of active threads is decremented, and the  *
   //* accumulators for files/bytes processed is updated.   *
   void completedSubthread ( void )
   {
      this->getDataLock () ;        //* Get a lock on the data
      if ( this->activeThreads > ZERO )   // (be safe)
         --this->activeThreads ;
      this->freeDataLock () ;       //* Release the data lock
   }

   //*************************
   //** Public Data Members **
   //*************************
   USHORT      strWidth ;     //* Number of display columns for string

   //*********************
   //** Private Methods **
   //*********************
   private:
   //* Reset our data members. Note that this is dangerous. If there are still *
   //* active threads, they will go off into the weeds. For this reason, this  *
   //* the method may be called externally only by the FMgr class and only at  *
   //* the beginning of a directory-tree data capture sequence.                *
   void reset ( void )
   {
      this->getDataLock () ;        //* Get a lock on the data
      this->deList          = NULL ;
      this->textData        = NULL ;
      this->colorData       = NULL ;
      this->listCount       = this->treeCount     = ZERO ;
      this->listBytes       = this->treeBytes     = ZERO ;
      this->totalThreads    = this->activeThreads = 
      this->goodSubs        = this->sysExceptions = ZERO ;
      this->exMsg[0]        = NULLCHAR ;
      this->exmIndex        = ZERO ;
      this->scanThread      = NULL ;
      this->scanThreadPath  = NULL ;
      this->mgrThread       = NULL ;
      this->mgrthreadActive = 
      this->uiEnable        = 
      this->treeView        = 
      this->abortSignal     = false ;
      this->autoRecurse     = true ;
      this->freeDataLock () ;       //* Release the data lock
   }

   //* Establish an exclusive lock on our private data members.*
   //* If another thread currently holds the lock, then the    *
   //* current thread will block (sleep).                      *
   void getDataLock ( void )
   {
      this->dataLock = unique_lock<mutex>( this->ddLock ) ;
   }

   //* Release previously-established exclusive lock on private data members.*
   void freeDataLock ( void )
   {
      this->dataLock.release() ;    // disassociate the dataLock from ddLock
      this->ddLock.unlock() ;       // unlock the ddLock mutex
   }

   //**************************
   //** Private Data Members **
   //**************************
   //* Lock controlling access to the following private data members *
   unique_lock<mutex> dataLock ;
   mutex ddLock ;
   tnFName**   deList ;       //* Pointer to an array of pointers to 
                              //* file-information data structures
   char**      textData ;     //* Pointer to an array of file-info 
                              //* display strings
   attr_t*     colorData ;    //* Pointer to an array of color attributes 
                              //* for the display strings
   UINT        listCount ;    //* Number of files in display list
   UINT64      listBytes ;    //* Combined size of all files in display list
   UINT        treeCount ;    //* Total number of files in directory tree
   UINT64      treeBytes ;    //* Total size of all files in directory tree
   USHORT      totalThreads ; //* Number of secondary threads created
   USHORT      goodSubs ;     //* Number of responsive secondary threads
   USHORT      activeThreads ;//* Number of secondary threads still active
   USHORT      sysExceptions ;//* Number of exceptions thrown and/or caught
   char        exMsg[gsDFLTBYTES] ; //* System-exception message(s)
   short       exmIndex ;     //* Index of next free byte of 'exMsg'
   bool        autoRecurse ;  //* 'true' for fully recursive scan of directory tree
   bool        uiEnable ;     //* 'true' signals user interface thread to continue
   bool        treeView ;     //* set to indicate Tree-view mode scan
   bool        abortSignal ;  //* set as a signal to terminate all active threads

   //* Dynamic allocation for top-level manager thread for sub-thread array.   *
   thread* mgrThread ;
   bool    mgrthreadActive ;  //* 'true' while mgrThread is active

   //* Dynamic data allocation for sub-threads and their associated paths.     *
   //* The dynamic objects are created in FMgr::CaptureDirTree() and are       *
   //* destroyed by our getActiveThreads() method, either through an           *
   //* explicit call from the main execution thread OR when the DispData       *
   //* instance is destroyed.                                                  *
   //* Note that these members ARE NOT under the control of ddLock because     *
   //* they are accessed only at the beginning and end of the capture sequence.*
   thread*  scanThread ;
   gString* scanThreadPath ;

   friend class FMgr ;        //* our dear and intimate friend
} ;

